"use strict";

const process = require('process');

const fs = require('fs');

const path = require('path');

const util = require('util');

const I18N = require('@ladjs/i18n');

const _ = require('lodash');

const consolidate = require('consolidate');

const debug = require('debug')('email-templates');

const getPaths = require('get-paths');

const {
  convert
} = require('html-to-text');

const juice = require('juice');

const nodemailer = require('nodemailer');

const previewEmail = require('preview-email'); // promise version of `juice.juiceResources`


const juiceResources = (html, options) => {
  return new Promise((resolve, reject) => {
    juice.juiceResources(html, options, (err, html) => {
      if (err) return reject(err);
      resolve(html);
    });
  });
};

const env = (process.env.NODE_ENV || 'development').toLowerCase();
const stat = util.promisify(fs.stat);
const readFile = util.promisify(fs.readFile);

class Email {
  constructor(config = {}) {
    debug('config passed %O', config); // 2.x backwards compatible support

    if (config.juiceOptions) {
      config.juiceResources = config.juiceOptions;
      delete config.juiceOptions;
    }

    if (config.disableJuice) {
      config.juice = false;
      delete config.disableJuice;
    }

    if (config.render) {
      config.customRender = true;
    }

    this.config = _.merge({
      views: {
        // directory where email templates reside
        root: path.resolve('emails'),
        options: {
          // default file extension for template
          extension: 'pug',
          map: {
            hbs: 'handlebars',
            njk: 'nunjucks'
          },
          engineSource: consolidate
        },
        // locals to pass to templates for rendering
        locals: {
          // turn on caching for non-development environments
          cache: !['development', 'test'].includes(env),
          // pretty is automatically set to `false` for subject/text
          pretty: true
        }
      },
      // <https://nodemailer.com/message/>
      message: {},
      send: !['development', 'test'].includes(env),
      preview: env === 'development',
      // <https://github.com/ladjs/i18n>
      // set to an object to configure and enable it
      i18n: false,
      // pass a custom render function if necessary
      render: this.render.bind(this),
      customRender: false,
      // force text-only rendering of template (disregards template folder)
      textOnly: false,
      // <https://github.com/html-to-text/node-html-to-text>
      htmlToText: {
        selectors: [{
          selector: 'img',
          format: 'skip'
        }]
      },
      subjectPrefix: false,
      // <https://github.com/Automattic/juice>
      juice: true,
      // Override juice global settings <https://github.com/Automattic/juice#juicecodeblockss>
      juiceSettings: {
        tableElements: ['TABLE']
      },
      juiceResources: {
        preserveImportant: true,
        webResources: {
          relativeTo: path.resolve('build'),
          images: false
        }
      },
      // pass a transport configuration object or a transport instance
      // (e.g. an instance is created via `nodemailer.createTransport`)
      // <https://nodemailer.com/transports/>
      transport: {},
      // last locale field name (also used by @ladjs/i18n)
      lastLocaleField: 'last_locale',

      getPath(type, template) {
        return path.join(template, type);
      }

    }, config); // override existing method

    this.render = this.config.render;
    if (!_.isFunction(this.config.transport.sendMail)) this.config.transport = nodemailer.createTransport(this.config.transport); // Override juice global settings https://github.com/Automattic/juice#juicecodeblocks

    if (_.isObject(this.config.juiceSettings)) {
      for (const [key, value] of Object.entries(this.config.juiceSettings)) {
        juice[key] = value;
      }
    }

    debug('transformed config %O', this.config);
    this.juiceResources = this.juiceResources.bind(this);
    this.getTemplatePath = this.getTemplatePath.bind(this);
    this.templateExists = this.templateExists.bind(this);
    this.checkAndRender = this.checkAndRender.bind(this);
    this.render = this.render.bind(this);
    this.renderAll = this.renderAll.bind(this);
    this.send = this.send.bind(this);
  } // shorthand use of `juiceResources` with the config
  // (mainly for custom renders like from a database)


  juiceResources(html, juiceRenderResources = {}) {
    const juiceR = _.merge(this.config.juiceResources, juiceRenderResources);

    return juiceResources(html, juiceR);
  } // a simple helper function that gets the actual file path for the template


  async getTemplatePath(template) {
    let juiceRenderResources = {};

    if (_.isObject(template)) {
      juiceRenderResources = template.juiceResources;
      template = template.path;
    }

    const [root, view] = path.isAbsolute(template) ? [path.dirname(template), path.basename(template)] : [this.config.views.root, template];
    const paths = await getPaths(root, view, this.config.views.options.extension);
    const filePath = path.resolve(root, paths.rel);
    return {
      filePath,
      paths,
      juiceRenderResources
    };
  } // returns true or false if a template exists
  // (uses same look-up approach as `render` function)


  async templateExists(view) {
    try {
      const {
        filePath
      } = await this.getTemplatePath(view);
      const stats = await stat(filePath);
      if (!stats.isFile()) throw new Error(`${filePath} was not a file`);
      return true;
    } catch (err) {
      debug('templateExists', err);
      return false;
    }
  }

  async checkAndRender(type, template, locals) {
    let juiceRenderResources = {};

    if (_.isObject(template)) {
      juiceRenderResources = template.juiceResources;
      template = template.path;
    }

    const string = this.config.getPath(type, template, locals);

    if (!this.config.customRender) {
      const exists = await this.templateExists(string);
      if (!exists) return;
    }

    return this.render(string, { ...locals,
      ...(type === 'html' ? {} : {
        pretty: false
      })
    }, juiceRenderResources);
  } // promise version of consolidate's render
  // inspired by koa-views and re-uses the same config
  // <https://github.com/queckezz/koa-views>


  async render(view, locals = {}) {
    const {
      map,
      engineSource
    } = this.config.views.options;
    const {
      filePath,
      paths,
      juiceRenderResources
    } = await this.getTemplatePath(view);

    if (paths.ext === 'html' && !map) {
      const res = await readFile(filePath, 'utf8');
      return res;
    }

    const engineName = map && map[paths.ext] ? map[paths.ext] : paths.ext;
    const renderFn = engineSource[engineName];
    if (!engineName || !renderFn) throw new Error(`Engine not found for the ".${paths.ext}" file extension`);

    if (_.isObject(this.config.i18n)) {
      if (this.config.i18n.lastLocaleField && this.config.lastLocaleField && this.config.i18n.lastLocaleField !== this.config.lastLocaleField) throw new Error(`The 'lastLocaleField' (String) option for @ladjs/i18n and email-templates do not match, i18n value was ${this.config.i18n.lastLocaleField} and email-templates value was ${this.config.lastLocaleField}`);
      const i18n = new I18N({ ...this.config.i18n,
        register: locals
      }); // support `locals.user.last_locale` (variable based name lastLocaleField)
      // (e.g. for <https://lad.js.org>)

      const locale = i18n.getLocale();
      if (_.isObject(locals.user) && _.isString(locals.user[this.config.lastLocaleField])) locals.locale = locals.user[this.config.lastLocaleField];else if (!_.isString(locals.locale)) locals.locale = locale;
      if (locale !== locals.locale) i18n.setLocale(locals.locale);
    }

    const res = await util.promisify(renderFn)(filePath, locals); // transform the html with juice using remote paths
    // google now supports media queries
    // https://developers.google.com/gmail/design/reference/supported_css

    if (!this.config.juice) return res;
    const html = await this.juiceResources(res, juiceRenderResources);
    return html;
  } // eslint-disable-next-line complexity


  async renderAll(template, locals = {}, nodemailerMessage = {}) {
    const message = { ...nodemailerMessage
    };

    if (template && (!message.subject || !message.html || !message.text)) {
      const [subject, html, text] = await Promise.all(['subject', 'html', 'text'].map(type => this.checkAndRender(type, template, locals)));
      if (subject && !message.subject) message.subject = subject;
      if (html && !message.html) message.html = html;
      if (text && !message.text) message.text = text;
    }

    if (message.subject && this.config.subjectPrefix) message.subject = this.config.subjectPrefix + message.subject; // trim subject

    if (message.subject) message.subject = message.subject.trim();
    if (this.config.htmlToText && message.html && !message.text) // we'd use nodemailer-html-to-text plugin
      // but we really don't need to support cid
      // <https://github.com/andris9/nodemailer-html-to-text>
      // (and it is also not updated with latest html-to-text)
      message.text = convert(message.html, this.config.htmlToText); // if we only want a text-based version of the email

    if (this.config.textOnly) delete message.html; // if no subject, html, or text content exists then we should
    // throw an error that says at least one must be found
    // otherwise the email would be blank (defeats purpose of email-templates)

    if ((!_.isString(message.subject) || _.isEmpty(_.trim(message.subject))) && (!_.isString(message.text) || _.isEmpty(_.trim(message.text))) && (!_.isString(message.html) || _.isEmpty(_.trim(message.html))) && _.isEmpty(message.attachments)) throw new Error(`No content was passed for subject, html, text, nor attachments message props. Check that the files for the template "${template}" exist.`);
    return message;
  }

  async send(options = {}) {
    options = {
      template: '',
      message: {},
      locals: {},
      ...options
    };
    let {
      template,
      message,
      locals
    } = options;
    const attachments = message.attachments || this.config.message.attachments || [];
    message = _.defaultsDeep({}, _.omit(message, 'attachments'), _.omit(this.config.message, 'attachments'));
    locals = _.defaultsDeep({}, this.config.views.locals, locals);
    if (attachments) message.attachments = attachments;
    debug('template %s', template);
    debug('message %O', message);
    debug('locals (keys only): %O', Object.keys(locals)); // get all available templates

    const object = await this.renderAll(template, locals, message); // assign the object variables over to the message

    Object.assign(message, object);

    if (this.config.preview) {
      debug('using `preview-email` to preview email');
      await (_.isObject(this.config.preview) ? previewEmail(message, this.config.preview) : previewEmail(message));
    }

    if (!this.config.send) {
      debug('send disabled so we are ensuring JSONTransport'); // <https://github.com/nodemailer/nodemailer/issues/798>
      // if (this.config.transport.name !== 'JSONTransport')

      this.config.transport = nodemailer.createTransport({
        jsonTransport: true
      });
    }

    const res = await this.config.transport.sendMail(message);
    debug('message sent');
    res.originalMessage = message;
    return res;
  }

}

module.exports = Email;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uL3NyYy9pbmRleC5qcyJdLCJuYW1lcyI6WyJwcm9jZXNzIiwicmVxdWlyZSIsImZzIiwicGF0aCIsInV0aWwiLCJJMThOIiwiXyIsImNvbnNvbGlkYXRlIiwiZGVidWciLCJnZXRQYXRocyIsImNvbnZlcnQiLCJqdWljZSIsIm5vZGVtYWlsZXIiLCJwcmV2aWV3RW1haWwiLCJqdWljZVJlc291cmNlcyIsImh0bWwiLCJvcHRpb25zIiwiUHJvbWlzZSIsInJlc29sdmUiLCJyZWplY3QiLCJlcnIiLCJlbnYiLCJOT0RFX0VOViIsInRvTG93ZXJDYXNlIiwic3RhdCIsInByb21pc2lmeSIsInJlYWRGaWxlIiwiRW1haWwiLCJjb25zdHJ1Y3RvciIsImNvbmZpZyIsImp1aWNlT3B0aW9ucyIsImRpc2FibGVKdWljZSIsInJlbmRlciIsImN1c3RvbVJlbmRlciIsIm1lcmdlIiwidmlld3MiLCJyb290IiwiZXh0ZW5zaW9uIiwibWFwIiwiaGJzIiwibmprIiwiZW5naW5lU291cmNlIiwibG9jYWxzIiwiY2FjaGUiLCJpbmNsdWRlcyIsInByZXR0eSIsIm1lc3NhZ2UiLCJzZW5kIiwicHJldmlldyIsImkxOG4iLCJiaW5kIiwidGV4dE9ubHkiLCJodG1sVG9UZXh0Iiwic2VsZWN0b3JzIiwic2VsZWN0b3IiLCJmb3JtYXQiLCJzdWJqZWN0UHJlZml4IiwianVpY2VTZXR0aW5ncyIsInRhYmxlRWxlbWVudHMiLCJwcmVzZXJ2ZUltcG9ydGFudCIsIndlYlJlc291cmNlcyIsInJlbGF0aXZlVG8iLCJpbWFnZXMiLCJ0cmFuc3BvcnQiLCJsYXN0TG9jYWxlRmllbGQiLCJnZXRQYXRoIiwidHlwZSIsInRlbXBsYXRlIiwiam9pbiIsImlzRnVuY3Rpb24iLCJzZW5kTWFpbCIsImNyZWF0ZVRyYW5zcG9ydCIsImlzT2JqZWN0Iiwia2V5IiwidmFsdWUiLCJPYmplY3QiLCJlbnRyaWVzIiwiZ2V0VGVtcGxhdGVQYXRoIiwidGVtcGxhdGVFeGlzdHMiLCJjaGVja0FuZFJlbmRlciIsInJlbmRlckFsbCIsImp1aWNlUmVuZGVyUmVzb3VyY2VzIiwianVpY2VSIiwidmlldyIsImlzQWJzb2x1dGUiLCJkaXJuYW1lIiwiYmFzZW5hbWUiLCJwYXRocyIsImZpbGVQYXRoIiwicmVsIiwic3RhdHMiLCJpc0ZpbGUiLCJFcnJvciIsInN0cmluZyIsImV4aXN0cyIsImV4dCIsInJlcyIsImVuZ2luZU5hbWUiLCJyZW5kZXJGbiIsInJlZ2lzdGVyIiwibG9jYWxlIiwiZ2V0TG9jYWxlIiwidXNlciIsImlzU3RyaW5nIiwic2V0TG9jYWxlIiwibm9kZW1haWxlck1lc3NhZ2UiLCJzdWJqZWN0IiwidGV4dCIsImFsbCIsInRyaW0iLCJpc0VtcHR5IiwiYXR0YWNobWVudHMiLCJkZWZhdWx0c0RlZXAiLCJvbWl0Iiwia2V5cyIsIm9iamVjdCIsImFzc2lnbiIsImpzb25UcmFuc3BvcnQiLCJvcmlnaW5hbE1lc3NhZ2UiLCJtb2R1bGUiLCJleHBvcnRzIl0sIm1hcHBpbmdzIjoiOztBQUFBLE1BQU1BLE9BQU8sR0FBR0MsT0FBTyxDQUFDLFNBQUQsQ0FBdkI7O0FBQ0EsTUFBTUMsRUFBRSxHQUFHRCxPQUFPLENBQUMsSUFBRCxDQUFsQjs7QUFDQSxNQUFNRSxJQUFJLEdBQUdGLE9BQU8sQ0FBQyxNQUFELENBQXBCOztBQUNBLE1BQU1HLElBQUksR0FBR0gsT0FBTyxDQUFDLE1BQUQsQ0FBcEI7O0FBRUEsTUFBTUksSUFBSSxHQUFHSixPQUFPLENBQUMsYUFBRCxDQUFwQjs7QUFDQSxNQUFNSyxDQUFDLEdBQUdMLE9BQU8sQ0FBQyxRQUFELENBQWpCOztBQUNBLE1BQU1NLFdBQVcsR0FBR04sT0FBTyxDQUFDLGFBQUQsQ0FBM0I7O0FBQ0EsTUFBTU8sS0FBSyxHQUFHUCxPQUFPLENBQUMsT0FBRCxDQUFQLENBQWlCLGlCQUFqQixDQUFkOztBQUNBLE1BQU1RLFFBQVEsR0FBR1IsT0FBTyxDQUFDLFdBQUQsQ0FBeEI7O0FBQ0EsTUFBTTtBQUFFUyxFQUFBQTtBQUFGLElBQWNULE9BQU8sQ0FBQyxjQUFELENBQTNCOztBQUNBLE1BQU1VLEtBQUssR0FBR1YsT0FBTyxDQUFDLE9BQUQsQ0FBckI7O0FBQ0EsTUFBTVcsVUFBVSxHQUFHWCxPQUFPLENBQUMsWUFBRCxDQUExQjs7QUFDQSxNQUFNWSxZQUFZLEdBQUdaLE9BQU8sQ0FBQyxlQUFELENBQTVCLEMsQ0FFQTs7O0FBQ0EsTUFBTWEsY0FBYyxHQUFHLENBQUNDLElBQUQsRUFBT0MsT0FBUCxLQUFtQjtBQUN4QyxTQUFPLElBQUlDLE9BQUosQ0FBWSxDQUFDQyxPQUFELEVBQVVDLE1BQVYsS0FBcUI7QUFDdENSLElBQUFBLEtBQUssQ0FBQ0csY0FBTixDQUFxQkMsSUFBckIsRUFBMkJDLE9BQTNCLEVBQW9DLENBQUNJLEdBQUQsRUFBTUwsSUFBTixLQUFlO0FBQ2pELFVBQUlLLEdBQUosRUFBUyxPQUFPRCxNQUFNLENBQUNDLEdBQUQsQ0FBYjtBQUNURixNQUFBQSxPQUFPLENBQUNILElBQUQsQ0FBUDtBQUNELEtBSEQ7QUFJRCxHQUxNLENBQVA7QUFNRCxDQVBEOztBQVNBLE1BQU1NLEdBQUcsR0FBRyxDQUFDckIsT0FBTyxDQUFDcUIsR0FBUixDQUFZQyxRQUFaLElBQXdCLGFBQXpCLEVBQXdDQyxXQUF4QyxFQUFaO0FBQ0EsTUFBTUMsSUFBSSxHQUFHcEIsSUFBSSxDQUFDcUIsU0FBTCxDQUFldkIsRUFBRSxDQUFDc0IsSUFBbEIsQ0FBYjtBQUNBLE1BQU1FLFFBQVEsR0FBR3RCLElBQUksQ0FBQ3FCLFNBQUwsQ0FBZXZCLEVBQUUsQ0FBQ3dCLFFBQWxCLENBQWpCOztBQUVBLE1BQU1DLEtBQU4sQ0FBWTtBQUNWQyxFQUFBQSxXQUFXLENBQUNDLE1BQU0sR0FBRyxFQUFWLEVBQWM7QUFDdkJyQixJQUFBQSxLQUFLLENBQUMsa0JBQUQsRUFBcUJxQixNQUFyQixDQUFMLENBRHVCLENBR3ZCOztBQUNBLFFBQUlBLE1BQU0sQ0FBQ0MsWUFBWCxFQUF5QjtBQUN2QkQsTUFBQUEsTUFBTSxDQUFDZixjQUFQLEdBQXdCZSxNQUFNLENBQUNDLFlBQS9CO0FBQ0EsYUFBT0QsTUFBTSxDQUFDQyxZQUFkO0FBQ0Q7O0FBRUQsUUFBSUQsTUFBTSxDQUFDRSxZQUFYLEVBQXlCO0FBQ3ZCRixNQUFBQSxNQUFNLENBQUNsQixLQUFQLEdBQWUsS0FBZjtBQUNBLGFBQU9rQixNQUFNLENBQUNFLFlBQWQ7QUFDRDs7QUFFRCxRQUFJRixNQUFNLENBQUNHLE1BQVgsRUFBbUI7QUFDakJILE1BQUFBLE1BQU0sQ0FBQ0ksWUFBUCxHQUFzQixJQUF0QjtBQUNEOztBQUVELFNBQUtKLE1BQUwsR0FBY3ZCLENBQUMsQ0FBQzRCLEtBQUYsQ0FDWjtBQUNFQyxNQUFBQSxLQUFLLEVBQUU7QUFDTDtBQUNBQyxRQUFBQSxJQUFJLEVBQUVqQyxJQUFJLENBQUNlLE9BQUwsQ0FBYSxRQUFiLENBRkQ7QUFHTEYsUUFBQUEsT0FBTyxFQUFFO0FBQ1A7QUFDQXFCLFVBQUFBLFNBQVMsRUFBRSxLQUZKO0FBR1BDLFVBQUFBLEdBQUcsRUFBRTtBQUNIQyxZQUFBQSxHQUFHLEVBQUUsWUFERjtBQUVIQyxZQUFBQSxHQUFHLEVBQUU7QUFGRixXQUhFO0FBT1BDLFVBQUFBLFlBQVksRUFBRWxDO0FBUFAsU0FISjtBQVlMO0FBQ0FtQyxRQUFBQSxNQUFNLEVBQUU7QUFDTjtBQUNBQyxVQUFBQSxLQUFLLEVBQUUsQ0FBQyxDQUFDLGFBQUQsRUFBZ0IsTUFBaEIsRUFBd0JDLFFBQXhCLENBQWlDdkIsR0FBakMsQ0FGRjtBQUdOO0FBQ0F3QixVQUFBQSxNQUFNLEVBQUU7QUFKRjtBQWJILE9BRFQ7QUFxQkU7QUFDQUMsTUFBQUEsT0FBTyxFQUFFLEVBdEJYO0FBdUJFQyxNQUFBQSxJQUFJLEVBQUUsQ0FBQyxDQUFDLGFBQUQsRUFBZ0IsTUFBaEIsRUFBd0JILFFBQXhCLENBQWlDdkIsR0FBakMsQ0F2QlQ7QUF3QkUyQixNQUFBQSxPQUFPLEVBQUUzQixHQUFHLEtBQUssYUF4Qm5CO0FBeUJFO0FBQ0E7QUFDQTRCLE1BQUFBLElBQUksRUFBRSxLQTNCUjtBQTRCRTtBQUNBakIsTUFBQUEsTUFBTSxFQUFFLEtBQUtBLE1BQUwsQ0FBWWtCLElBQVosQ0FBaUIsSUFBakIsQ0E3QlY7QUE4QkVqQixNQUFBQSxZQUFZLEVBQUUsS0E5QmhCO0FBK0JFO0FBQ0FrQixNQUFBQSxRQUFRLEVBQUUsS0FoQ1o7QUFpQ0U7QUFDQUMsTUFBQUEsVUFBVSxFQUFFO0FBQ1ZDLFFBQUFBLFNBQVMsRUFBRSxDQUFDO0FBQUVDLFVBQUFBLFFBQVEsRUFBRSxLQUFaO0FBQW1CQyxVQUFBQSxNQUFNLEVBQUU7QUFBM0IsU0FBRDtBQURELE9BbENkO0FBcUNFQyxNQUFBQSxhQUFhLEVBQUUsS0FyQ2pCO0FBc0NFO0FBQ0E3QyxNQUFBQSxLQUFLLEVBQUUsSUF2Q1Q7QUF3Q0U7QUFDQThDLE1BQUFBLGFBQWEsRUFBRTtBQUNiQyxRQUFBQSxhQUFhLEVBQUUsQ0FBQyxPQUFEO0FBREYsT0F6Q2pCO0FBNENFNUMsTUFBQUEsY0FBYyxFQUFFO0FBQ2Q2QyxRQUFBQSxpQkFBaUIsRUFBRSxJQURMO0FBRWRDLFFBQUFBLFlBQVksRUFBRTtBQUNaQyxVQUFBQSxVQUFVLEVBQUUxRCxJQUFJLENBQUNlLE9BQUwsQ0FBYSxPQUFiLENBREE7QUFFWjRDLFVBQUFBLE1BQU0sRUFBRTtBQUZJO0FBRkEsT0E1Q2xCO0FBbURFO0FBQ0E7QUFDQTtBQUNBQyxNQUFBQSxTQUFTLEVBQUUsRUF0RGI7QUF1REU7QUFDQUMsTUFBQUEsZUFBZSxFQUFFLGFBeERuQjs7QUF5REVDLE1BQUFBLE9BQU8sQ0FBQ0MsSUFBRCxFQUFPQyxRQUFQLEVBQWlCO0FBQ3RCLGVBQU9oRSxJQUFJLENBQUNpRSxJQUFMLENBQVVELFFBQVYsRUFBb0JELElBQXBCLENBQVA7QUFDRDs7QUEzREgsS0FEWSxFQThEWnJDLE1BOURZLENBQWQsQ0FsQnVCLENBbUZ2Qjs7QUFDQSxTQUFLRyxNQUFMLEdBQWMsS0FBS0gsTUFBTCxDQUFZRyxNQUExQjtBQUVBLFFBQUksQ0FBQzFCLENBQUMsQ0FBQytELFVBQUYsQ0FBYSxLQUFLeEMsTUFBTCxDQUFZa0MsU0FBWixDQUFzQk8sUUFBbkMsQ0FBTCxFQUNFLEtBQUt6QyxNQUFMLENBQVlrQyxTQUFaLEdBQXdCbkQsVUFBVSxDQUFDMkQsZUFBWCxDQUEyQixLQUFLMUMsTUFBTCxDQUFZa0MsU0FBdkMsQ0FBeEIsQ0F2RnFCLENBeUZ2Qjs7QUFDQSxRQUFJekQsQ0FBQyxDQUFDa0UsUUFBRixDQUFXLEtBQUszQyxNQUFMLENBQVk0QixhQUF2QixDQUFKLEVBQTJDO0FBQ3pDLFdBQUssTUFBTSxDQUFDZ0IsR0FBRCxFQUFNQyxLQUFOLENBQVgsSUFBMkJDLE1BQU0sQ0FBQ0MsT0FBUCxDQUFlLEtBQUsvQyxNQUFMLENBQVk0QixhQUEzQixDQUEzQixFQUFzRTtBQUNwRTlDLFFBQUFBLEtBQUssQ0FBQzhELEdBQUQsQ0FBTCxHQUFhQyxLQUFiO0FBQ0Q7QUFDRjs7QUFFRGxFLElBQUFBLEtBQUssQ0FBQyx1QkFBRCxFQUEwQixLQUFLcUIsTUFBL0IsQ0FBTDtBQUVBLFNBQUtmLGNBQUwsR0FBc0IsS0FBS0EsY0FBTCxDQUFvQm9DLElBQXBCLENBQXlCLElBQXpCLENBQXRCO0FBQ0EsU0FBSzJCLGVBQUwsR0FBdUIsS0FBS0EsZUFBTCxDQUFxQjNCLElBQXJCLENBQTBCLElBQTFCLENBQXZCO0FBQ0EsU0FBSzRCLGNBQUwsR0FBc0IsS0FBS0EsY0FBTCxDQUFvQjVCLElBQXBCLENBQXlCLElBQXpCLENBQXRCO0FBQ0EsU0FBSzZCLGNBQUwsR0FBc0IsS0FBS0EsY0FBTCxDQUFvQjdCLElBQXBCLENBQXlCLElBQXpCLENBQXRCO0FBQ0EsU0FBS2xCLE1BQUwsR0FBYyxLQUFLQSxNQUFMLENBQVlrQixJQUFaLENBQWlCLElBQWpCLENBQWQ7QUFDQSxTQUFLOEIsU0FBTCxHQUFpQixLQUFLQSxTQUFMLENBQWU5QixJQUFmLENBQW9CLElBQXBCLENBQWpCO0FBQ0EsU0FBS0gsSUFBTCxHQUFZLEtBQUtBLElBQUwsQ0FBVUcsSUFBVixDQUFlLElBQWYsQ0FBWjtBQUNELEdBMUdTLENBNEdWO0FBQ0E7OztBQUNBcEMsRUFBQUEsY0FBYyxDQUFDQyxJQUFELEVBQU9rRSxvQkFBb0IsR0FBRyxFQUE5QixFQUFrQztBQUM5QyxVQUFNQyxNQUFNLEdBQUc1RSxDQUFDLENBQUM0QixLQUFGLENBQVEsS0FBS0wsTUFBTCxDQUFZZixjQUFwQixFQUFvQ21FLG9CQUFwQyxDQUFmOztBQUNBLFdBQU9uRSxjQUFjLENBQUNDLElBQUQsRUFBT21FLE1BQVAsQ0FBckI7QUFDRCxHQWpIUyxDQW1IVjs7O0FBQ3FCLFFBQWZMLGVBQWUsQ0FBQ1YsUUFBRCxFQUFXO0FBQzlCLFFBQUljLG9CQUFvQixHQUFHLEVBQTNCOztBQUVBLFFBQUkzRSxDQUFDLENBQUNrRSxRQUFGLENBQVdMLFFBQVgsQ0FBSixFQUEwQjtBQUN4QmMsTUFBQUEsb0JBQW9CLEdBQUdkLFFBQVEsQ0FBQ3JELGNBQWhDO0FBQ0FxRCxNQUFBQSxRQUFRLEdBQUdBLFFBQVEsQ0FBQ2hFLElBQXBCO0FBQ0Q7O0FBRUQsVUFBTSxDQUFDaUMsSUFBRCxFQUFPK0MsSUFBUCxJQUFlaEYsSUFBSSxDQUFDaUYsVUFBTCxDQUFnQmpCLFFBQWhCLElBQ2pCLENBQUNoRSxJQUFJLENBQUNrRixPQUFMLENBQWFsQixRQUFiLENBQUQsRUFBeUJoRSxJQUFJLENBQUNtRixRQUFMLENBQWNuQixRQUFkLENBQXpCLENBRGlCLEdBRWpCLENBQUMsS0FBS3RDLE1BQUwsQ0FBWU0sS0FBWixDQUFrQkMsSUFBbkIsRUFBeUIrQixRQUF6QixDQUZKO0FBR0EsVUFBTW9CLEtBQUssR0FBRyxNQUFNOUUsUUFBUSxDQUMxQjJCLElBRDBCLEVBRTFCK0MsSUFGMEIsRUFHMUIsS0FBS3RELE1BQUwsQ0FBWU0sS0FBWixDQUFrQm5CLE9BQWxCLENBQTBCcUIsU0FIQSxDQUE1QjtBQUtBLFVBQU1tRCxRQUFRLEdBQUdyRixJQUFJLENBQUNlLE9BQUwsQ0FBYWtCLElBQWIsRUFBbUJtRCxLQUFLLENBQUNFLEdBQXpCLENBQWpCO0FBQ0EsV0FBTztBQUFFRCxNQUFBQSxRQUFGO0FBQVlELE1BQUFBLEtBQVo7QUFBbUJOLE1BQUFBO0FBQW5CLEtBQVA7QUFDRCxHQXRJUyxDQXdJVjtBQUNBOzs7QUFDb0IsUUFBZEgsY0FBYyxDQUFDSyxJQUFELEVBQU87QUFDekIsUUFBSTtBQUNGLFlBQU07QUFBRUssUUFBQUE7QUFBRixVQUFlLE1BQU0sS0FBS1gsZUFBTCxDQUFxQk0sSUFBckIsQ0FBM0I7QUFDQSxZQUFNTyxLQUFLLEdBQUcsTUFBTWxFLElBQUksQ0FBQ2dFLFFBQUQsQ0FBeEI7QUFDQSxVQUFJLENBQUNFLEtBQUssQ0FBQ0MsTUFBTixFQUFMLEVBQXFCLE1BQU0sSUFBSUMsS0FBSixDQUFXLEdBQUVKLFFBQVMsaUJBQXRCLENBQU47QUFDckIsYUFBTyxJQUFQO0FBQ0QsS0FMRCxDQUtFLE9BQU9wRSxHQUFQLEVBQVk7QUFDWlosTUFBQUEsS0FBSyxDQUFDLGdCQUFELEVBQW1CWSxHQUFuQixDQUFMO0FBQ0EsYUFBTyxLQUFQO0FBQ0Q7QUFDRjs7QUFFbUIsUUFBZDJELGNBQWMsQ0FBQ2IsSUFBRCxFQUFPQyxRQUFQLEVBQWlCekIsTUFBakIsRUFBeUI7QUFDM0MsUUFBSXVDLG9CQUFvQixHQUFHLEVBQTNCOztBQUVBLFFBQUkzRSxDQUFDLENBQUNrRSxRQUFGLENBQVdMLFFBQVgsQ0FBSixFQUEwQjtBQUN4QmMsTUFBQUEsb0JBQW9CLEdBQUdkLFFBQVEsQ0FBQ3JELGNBQWhDO0FBQ0FxRCxNQUFBQSxRQUFRLEdBQUdBLFFBQVEsQ0FBQ2hFLElBQXBCO0FBQ0Q7O0FBRUQsVUFBTTBGLE1BQU0sR0FBRyxLQUFLaEUsTUFBTCxDQUFZb0MsT0FBWixDQUFvQkMsSUFBcEIsRUFBMEJDLFFBQTFCLEVBQW9DekIsTUFBcEMsQ0FBZjs7QUFDQSxRQUFJLENBQUMsS0FBS2IsTUFBTCxDQUFZSSxZQUFqQixFQUErQjtBQUM3QixZQUFNNkQsTUFBTSxHQUFHLE1BQU0sS0FBS2hCLGNBQUwsQ0FBb0JlLE1BQXBCLENBQXJCO0FBQ0EsVUFBSSxDQUFDQyxNQUFMLEVBQWE7QUFDZDs7QUFFRCxXQUFPLEtBQUs5RCxNQUFMLENBQ0w2RCxNQURLLEVBRUwsRUFDRSxHQUFHbkQsTUFETDtBQUVFLFVBQUl3QixJQUFJLEtBQUssTUFBVCxHQUFrQixFQUFsQixHQUF1QjtBQUFFckIsUUFBQUEsTUFBTSxFQUFFO0FBQVYsT0FBM0I7QUFGRixLQUZLLEVBTUxvQyxvQkFOSyxDQUFQO0FBUUQsR0E1S1MsQ0E4S1Y7QUFDQTtBQUNBOzs7QUFDWSxRQUFOakQsTUFBTSxDQUFDbUQsSUFBRCxFQUFPekMsTUFBTSxHQUFHLEVBQWhCLEVBQW9CO0FBQzlCLFVBQU07QUFBRUosTUFBQUEsR0FBRjtBQUFPRyxNQUFBQTtBQUFQLFFBQXdCLEtBQUtaLE1BQUwsQ0FBWU0sS0FBWixDQUFrQm5CLE9BQWhEO0FBQ0EsVUFBTTtBQUFFd0UsTUFBQUEsUUFBRjtBQUFZRCxNQUFBQSxLQUFaO0FBQW1CTixNQUFBQTtBQUFuQixRQUNKLE1BQU0sS0FBS0osZUFBTCxDQUFxQk0sSUFBckIsQ0FEUjs7QUFFQSxRQUFJSSxLQUFLLENBQUNRLEdBQU4sS0FBYyxNQUFkLElBQXdCLENBQUN6RCxHQUE3QixFQUFrQztBQUNoQyxZQUFNMEQsR0FBRyxHQUFHLE1BQU10RSxRQUFRLENBQUM4RCxRQUFELEVBQVcsTUFBWCxDQUExQjtBQUNBLGFBQU9RLEdBQVA7QUFDRDs7QUFFRCxVQUFNQyxVQUFVLEdBQUczRCxHQUFHLElBQUlBLEdBQUcsQ0FBQ2lELEtBQUssQ0FBQ1EsR0FBUCxDQUFWLEdBQXdCekQsR0FBRyxDQUFDaUQsS0FBSyxDQUFDUSxHQUFQLENBQTNCLEdBQXlDUixLQUFLLENBQUNRLEdBQWxFO0FBQ0EsVUFBTUcsUUFBUSxHQUFHekQsWUFBWSxDQUFDd0QsVUFBRCxDQUE3QjtBQUNBLFFBQUksQ0FBQ0EsVUFBRCxJQUFlLENBQUNDLFFBQXBCLEVBQ0UsTUFBTSxJQUFJTixLQUFKLENBQ0gsOEJBQTZCTCxLQUFLLENBQUNRLEdBQUksa0JBRHBDLENBQU47O0FBSUYsUUFBSXpGLENBQUMsQ0FBQ2tFLFFBQUYsQ0FBVyxLQUFLM0MsTUFBTCxDQUFZb0IsSUFBdkIsQ0FBSixFQUFrQztBQUNoQyxVQUNFLEtBQUtwQixNQUFMLENBQVlvQixJQUFaLENBQWlCZSxlQUFqQixJQUNBLEtBQUtuQyxNQUFMLENBQVltQyxlQURaLElBRUEsS0FBS25DLE1BQUwsQ0FBWW9CLElBQVosQ0FBaUJlLGVBQWpCLEtBQXFDLEtBQUtuQyxNQUFMLENBQVltQyxlQUhuRCxFQUtFLE1BQU0sSUFBSTRCLEtBQUosQ0FDSCwwR0FBeUcsS0FBSy9ELE1BQUwsQ0FBWW9CLElBQVosQ0FBaUJlLGVBQWdCLGtDQUFpQyxLQUFLbkMsTUFBTCxDQUFZbUMsZUFBZ0IsRUFEcE0sQ0FBTjtBQUlGLFlBQU1mLElBQUksR0FBRyxJQUFJNUMsSUFBSixDQUFTLEVBQUUsR0FBRyxLQUFLd0IsTUFBTCxDQUFZb0IsSUFBakI7QUFBdUJrRCxRQUFBQSxRQUFRLEVBQUV6RDtBQUFqQyxPQUFULENBQWIsQ0FWZ0MsQ0FZaEM7QUFDQTs7QUFDQSxZQUFNMEQsTUFBTSxHQUFHbkQsSUFBSSxDQUFDb0QsU0FBTCxFQUFmO0FBQ0EsVUFDRS9GLENBQUMsQ0FBQ2tFLFFBQUYsQ0FBVzlCLE1BQU0sQ0FBQzRELElBQWxCLEtBQ0FoRyxDQUFDLENBQUNpRyxRQUFGLENBQVc3RCxNQUFNLENBQUM0RCxJQUFQLENBQVksS0FBS3pFLE1BQUwsQ0FBWW1DLGVBQXhCLENBQVgsQ0FGRixFQUlFdEIsTUFBTSxDQUFDMEQsTUFBUCxHQUFnQjFELE1BQU0sQ0FBQzRELElBQVAsQ0FBWSxLQUFLekUsTUFBTCxDQUFZbUMsZUFBeEIsQ0FBaEIsQ0FKRixLQUtLLElBQUksQ0FBQzFELENBQUMsQ0FBQ2lHLFFBQUYsQ0FBVzdELE1BQU0sQ0FBQzBELE1BQWxCLENBQUwsRUFBZ0MxRCxNQUFNLENBQUMwRCxNQUFQLEdBQWdCQSxNQUFoQjtBQUVyQyxVQUFJQSxNQUFNLEtBQUsxRCxNQUFNLENBQUMwRCxNQUF0QixFQUE4Qm5ELElBQUksQ0FBQ3VELFNBQUwsQ0FBZTlELE1BQU0sQ0FBQzBELE1BQXRCO0FBQy9COztBQUVELFVBQU1KLEdBQUcsR0FBRyxNQUFNNUYsSUFBSSxDQUFDcUIsU0FBTCxDQUFleUUsUUFBZixFQUF5QlYsUUFBekIsRUFBbUM5QyxNQUFuQyxDQUFsQixDQXpDOEIsQ0EwQzlCO0FBQ0E7QUFDQTs7QUFDQSxRQUFJLENBQUMsS0FBS2IsTUFBTCxDQUFZbEIsS0FBakIsRUFBd0IsT0FBT3FGLEdBQVA7QUFDeEIsVUFBTWpGLElBQUksR0FBRyxNQUFNLEtBQUtELGNBQUwsQ0FBb0JrRixHQUFwQixFQUF5QmYsb0JBQXpCLENBQW5CO0FBQ0EsV0FBT2xFLElBQVA7QUFDRCxHQWpPUyxDQW1PVjs7O0FBQ2UsUUFBVGlFLFNBQVMsQ0FBQ2IsUUFBRCxFQUFXekIsTUFBTSxHQUFHLEVBQXBCLEVBQXdCK0QsaUJBQWlCLEdBQUcsRUFBNUMsRUFBZ0Q7QUFDN0QsVUFBTTNELE9BQU8sR0FBRyxFQUFFLEdBQUcyRDtBQUFMLEtBQWhCOztBQUVBLFFBQUl0QyxRQUFRLEtBQUssQ0FBQ3JCLE9BQU8sQ0FBQzRELE9BQVQsSUFBb0IsQ0FBQzVELE9BQU8sQ0FBQy9CLElBQTdCLElBQXFDLENBQUMrQixPQUFPLENBQUM2RCxJQUFuRCxDQUFaLEVBQXNFO0FBQ3BFLFlBQU0sQ0FBQ0QsT0FBRCxFQUFVM0YsSUFBVixFQUFnQjRGLElBQWhCLElBQXdCLE1BQU0xRixPQUFPLENBQUMyRixHQUFSLENBQ2xDLENBQUMsU0FBRCxFQUFZLE1BQVosRUFBb0IsTUFBcEIsRUFBNEJ0RSxHQUE1QixDQUFpQzRCLElBQUQsSUFDOUIsS0FBS2EsY0FBTCxDQUFvQmIsSUFBcEIsRUFBMEJDLFFBQTFCLEVBQW9DekIsTUFBcEMsQ0FERixDQURrQyxDQUFwQztBQU1BLFVBQUlnRSxPQUFPLElBQUksQ0FBQzVELE9BQU8sQ0FBQzRELE9BQXhCLEVBQWlDNUQsT0FBTyxDQUFDNEQsT0FBUixHQUFrQkEsT0FBbEI7QUFDakMsVUFBSTNGLElBQUksSUFBSSxDQUFDK0IsT0FBTyxDQUFDL0IsSUFBckIsRUFBMkIrQixPQUFPLENBQUMvQixJQUFSLEdBQWVBLElBQWY7QUFDM0IsVUFBSTRGLElBQUksSUFBSSxDQUFDN0QsT0FBTyxDQUFDNkQsSUFBckIsRUFBMkI3RCxPQUFPLENBQUM2RCxJQUFSLEdBQWVBLElBQWY7QUFDNUI7O0FBRUQsUUFBSTdELE9BQU8sQ0FBQzRELE9BQVIsSUFBbUIsS0FBSzdFLE1BQUwsQ0FBWTJCLGFBQW5DLEVBQ0VWLE9BQU8sQ0FBQzRELE9BQVIsR0FBa0IsS0FBSzdFLE1BQUwsQ0FBWTJCLGFBQVosR0FBNEJWLE9BQU8sQ0FBQzRELE9BQXRELENBaEIyRCxDQWtCN0Q7O0FBQ0EsUUFBSTVELE9BQU8sQ0FBQzRELE9BQVosRUFBcUI1RCxPQUFPLENBQUM0RCxPQUFSLEdBQWtCNUQsT0FBTyxDQUFDNEQsT0FBUixDQUFnQkcsSUFBaEIsRUFBbEI7QUFFckIsUUFBSSxLQUFLaEYsTUFBTCxDQUFZdUIsVUFBWixJQUEwQk4sT0FBTyxDQUFDL0IsSUFBbEMsSUFBMEMsQ0FBQytCLE9BQU8sQ0FBQzZELElBQXZELEVBQ0U7QUFDQTtBQUNBO0FBQ0E7QUFDQTdELE1BQUFBLE9BQU8sQ0FBQzZELElBQVIsR0FBZWpHLE9BQU8sQ0FBQ29DLE9BQU8sQ0FBQy9CLElBQVQsRUFBZSxLQUFLYyxNQUFMLENBQVl1QixVQUEzQixDQUF0QixDQTFCMkQsQ0E0QjdEOztBQUNBLFFBQUksS0FBS3ZCLE1BQUwsQ0FBWXNCLFFBQWhCLEVBQTBCLE9BQU9MLE9BQU8sQ0FBQy9CLElBQWYsQ0E3Qm1DLENBK0I3RDtBQUNBO0FBQ0E7O0FBQ0EsUUFDRSxDQUFDLENBQUNULENBQUMsQ0FBQ2lHLFFBQUYsQ0FBV3pELE9BQU8sQ0FBQzRELE9BQW5CLENBQUQsSUFBZ0NwRyxDQUFDLENBQUN3RyxPQUFGLENBQVV4RyxDQUFDLENBQUN1RyxJQUFGLENBQU8vRCxPQUFPLENBQUM0RCxPQUFmLENBQVYsQ0FBakMsTUFDQyxDQUFDcEcsQ0FBQyxDQUFDaUcsUUFBRixDQUFXekQsT0FBTyxDQUFDNkQsSUFBbkIsQ0FBRCxJQUE2QnJHLENBQUMsQ0FBQ3dHLE9BQUYsQ0FBVXhHLENBQUMsQ0FBQ3VHLElBQUYsQ0FBTy9ELE9BQU8sQ0FBQzZELElBQWYsQ0FBVixDQUQ5QixNQUVDLENBQUNyRyxDQUFDLENBQUNpRyxRQUFGLENBQVd6RCxPQUFPLENBQUMvQixJQUFuQixDQUFELElBQTZCVCxDQUFDLENBQUN3RyxPQUFGLENBQVV4RyxDQUFDLENBQUN1RyxJQUFGLENBQU8vRCxPQUFPLENBQUMvQixJQUFmLENBQVYsQ0FGOUIsS0FHQVQsQ0FBQyxDQUFDd0csT0FBRixDQUFVaEUsT0FBTyxDQUFDaUUsV0FBbEIsQ0FKRixFQU1FLE1BQU0sSUFBSW5CLEtBQUosQ0FDSCx3SEFBdUh6QixRQUFTLFVBRDdILENBQU47QUFJRixXQUFPckIsT0FBUDtBQUNEOztBQUVTLFFBQUpDLElBQUksQ0FBQy9CLE9BQU8sR0FBRyxFQUFYLEVBQWU7QUFDdkJBLElBQUFBLE9BQU8sR0FBRztBQUNSbUQsTUFBQUEsUUFBUSxFQUFFLEVBREY7QUFFUnJCLE1BQUFBLE9BQU8sRUFBRSxFQUZEO0FBR1JKLE1BQUFBLE1BQU0sRUFBRSxFQUhBO0FBSVIsU0FBRzFCO0FBSkssS0FBVjtBQU9BLFFBQUk7QUFBRW1ELE1BQUFBLFFBQUY7QUFBWXJCLE1BQUFBLE9BQVo7QUFBcUJKLE1BQUFBO0FBQXJCLFFBQWdDMUIsT0FBcEM7QUFFQSxVQUFNK0YsV0FBVyxHQUNmakUsT0FBTyxDQUFDaUUsV0FBUixJQUF1QixLQUFLbEYsTUFBTCxDQUFZaUIsT0FBWixDQUFvQmlFLFdBQTNDLElBQTBELEVBRDVEO0FBR0FqRSxJQUFBQSxPQUFPLEdBQUd4QyxDQUFDLENBQUMwRyxZQUFGLENBQ1IsRUFEUSxFQUVSMUcsQ0FBQyxDQUFDMkcsSUFBRixDQUFPbkUsT0FBUCxFQUFnQixhQUFoQixDQUZRLEVBR1J4QyxDQUFDLENBQUMyRyxJQUFGLENBQU8sS0FBS3BGLE1BQUwsQ0FBWWlCLE9BQW5CLEVBQTRCLGFBQTVCLENBSFEsQ0FBVjtBQUtBSixJQUFBQSxNQUFNLEdBQUdwQyxDQUFDLENBQUMwRyxZQUFGLENBQWUsRUFBZixFQUFtQixLQUFLbkYsTUFBTCxDQUFZTSxLQUFaLENBQWtCTyxNQUFyQyxFQUE2Q0EsTUFBN0MsQ0FBVDtBQUVBLFFBQUlxRSxXQUFKLEVBQWlCakUsT0FBTyxDQUFDaUUsV0FBUixHQUFzQkEsV0FBdEI7QUFFakJ2RyxJQUFBQSxLQUFLLENBQUMsYUFBRCxFQUFnQjJELFFBQWhCLENBQUw7QUFDQTNELElBQUFBLEtBQUssQ0FBQyxZQUFELEVBQWVzQyxPQUFmLENBQUw7QUFDQXRDLElBQUFBLEtBQUssQ0FBQyx3QkFBRCxFQUEyQm1FLE1BQU0sQ0FBQ3VDLElBQVAsQ0FBWXhFLE1BQVosQ0FBM0IsQ0FBTCxDQXhCdUIsQ0EwQnZCOztBQUNBLFVBQU15RSxNQUFNLEdBQUcsTUFBTSxLQUFLbkMsU0FBTCxDQUFlYixRQUFmLEVBQXlCekIsTUFBekIsRUFBaUNJLE9BQWpDLENBQXJCLENBM0J1QixDQTZCdkI7O0FBQ0E2QixJQUFBQSxNQUFNLENBQUN5QyxNQUFQLENBQWN0RSxPQUFkLEVBQXVCcUUsTUFBdkI7O0FBRUEsUUFBSSxLQUFLdEYsTUFBTCxDQUFZbUIsT0FBaEIsRUFBeUI7QUFDdkJ4QyxNQUFBQSxLQUFLLENBQUMsd0NBQUQsQ0FBTDtBQUNBLGFBQU9GLENBQUMsQ0FBQ2tFLFFBQUYsQ0FBVyxLQUFLM0MsTUFBTCxDQUFZbUIsT0FBdkIsSUFDSG5DLFlBQVksQ0FBQ2lDLE9BQUQsRUFBVSxLQUFLakIsTUFBTCxDQUFZbUIsT0FBdEIsQ0FEVCxHQUVIbkMsWUFBWSxDQUFDaUMsT0FBRCxDQUZoQjtBQUdEOztBQUVELFFBQUksQ0FBQyxLQUFLakIsTUFBTCxDQUFZa0IsSUFBakIsRUFBdUI7QUFDckJ2QyxNQUFBQSxLQUFLLENBQUMsZ0RBQUQsQ0FBTCxDQURxQixDQUVyQjtBQUNBOztBQUNBLFdBQUtxQixNQUFMLENBQVlrQyxTQUFaLEdBQXdCbkQsVUFBVSxDQUFDMkQsZUFBWCxDQUEyQjtBQUNqRDhDLFFBQUFBLGFBQWEsRUFBRTtBQURrQyxPQUEzQixDQUF4QjtBQUdEOztBQUVELFVBQU1yQixHQUFHLEdBQUcsTUFBTSxLQUFLbkUsTUFBTCxDQUFZa0MsU0FBWixDQUFzQk8sUUFBdEIsQ0FBK0J4QixPQUEvQixDQUFsQjtBQUNBdEMsSUFBQUEsS0FBSyxDQUFDLGNBQUQsQ0FBTDtBQUNBd0YsSUFBQUEsR0FBRyxDQUFDc0IsZUFBSixHQUFzQnhFLE9BQXRCO0FBQ0EsV0FBT2tELEdBQVA7QUFDRDs7QUF2VVM7O0FBMFVadUIsTUFBTSxDQUFDQyxPQUFQLEdBQWlCN0YsS0FBakIiLCJzb3VyY2VzQ29udGVudCI6WyJjb25zdCBwcm9jZXNzID0gcmVxdWlyZSgncHJvY2VzcycpO1xuY29uc3QgZnMgPSByZXF1aXJlKCdmcycpO1xuY29uc3QgcGF0aCA9IHJlcXVpcmUoJ3BhdGgnKTtcbmNvbnN0IHV0aWwgPSByZXF1aXJlKCd1dGlsJyk7XG5cbmNvbnN0IEkxOE4gPSByZXF1aXJlKCdAbGFkanMvaTE4bicpO1xuY29uc3QgXyA9IHJlcXVpcmUoJ2xvZGFzaCcpO1xuY29uc3QgY29uc29saWRhdGUgPSByZXF1aXJlKCdjb25zb2xpZGF0ZScpO1xuY29uc3QgZGVidWcgPSByZXF1aXJlKCdkZWJ1ZycpKCdlbWFpbC10ZW1wbGF0ZXMnKTtcbmNvbnN0IGdldFBhdGhzID0gcmVxdWlyZSgnZ2V0LXBhdGhzJyk7XG5jb25zdCB7IGNvbnZlcnQgfSA9IHJlcXVpcmUoJ2h0bWwtdG8tdGV4dCcpO1xuY29uc3QganVpY2UgPSByZXF1aXJlKCdqdWljZScpO1xuY29uc3Qgbm9kZW1haWxlciA9IHJlcXVpcmUoJ25vZGVtYWlsZXInKTtcbmNvbnN0IHByZXZpZXdFbWFpbCA9IHJlcXVpcmUoJ3ByZXZpZXctZW1haWwnKTtcblxuLy8gcHJvbWlzZSB2ZXJzaW9uIG9mIGBqdWljZS5qdWljZVJlc291cmNlc2BcbmNvbnN0IGp1aWNlUmVzb3VyY2VzID0gKGh0bWwsIG9wdGlvbnMpID0+IHtcbiAgcmV0dXJuIG5ldyBQcm9taXNlKChyZXNvbHZlLCByZWplY3QpID0+IHtcbiAgICBqdWljZS5qdWljZVJlc291cmNlcyhodG1sLCBvcHRpb25zLCAoZXJyLCBodG1sKSA9PiB7XG4gICAgICBpZiAoZXJyKSByZXR1cm4gcmVqZWN0KGVycik7XG4gICAgICByZXNvbHZlKGh0bWwpO1xuICAgIH0pO1xuICB9KTtcbn07XG5cbmNvbnN0IGVudiA9IChwcm9jZXNzLmVudi5OT0RFX0VOViB8fCAnZGV2ZWxvcG1lbnQnKS50b0xvd2VyQ2FzZSgpO1xuY29uc3Qgc3RhdCA9IHV0aWwucHJvbWlzaWZ5KGZzLnN0YXQpO1xuY29uc3QgcmVhZEZpbGUgPSB1dGlsLnByb21pc2lmeShmcy5yZWFkRmlsZSk7XG5cbmNsYXNzIEVtYWlsIHtcbiAgY29uc3RydWN0b3IoY29uZmlnID0ge30pIHtcbiAgICBkZWJ1ZygnY29uZmlnIHBhc3NlZCAlTycsIGNvbmZpZyk7XG5cbiAgICAvLyAyLnggYmFja3dhcmRzIGNvbXBhdGlibGUgc3VwcG9ydFxuICAgIGlmIChjb25maWcuanVpY2VPcHRpb25zKSB7XG4gICAgICBjb25maWcuanVpY2VSZXNvdXJjZXMgPSBjb25maWcuanVpY2VPcHRpb25zO1xuICAgICAgZGVsZXRlIGNvbmZpZy5qdWljZU9wdGlvbnM7XG4gICAgfVxuXG4gICAgaWYgKGNvbmZpZy5kaXNhYmxlSnVpY2UpIHtcbiAgICAgIGNvbmZpZy5qdWljZSA9IGZhbHNlO1xuICAgICAgZGVsZXRlIGNvbmZpZy5kaXNhYmxlSnVpY2U7XG4gICAgfVxuXG4gICAgaWYgKGNvbmZpZy5yZW5kZXIpIHtcbiAgICAgIGNvbmZpZy5jdXN0b21SZW5kZXIgPSB0cnVlO1xuICAgIH1cblxuICAgIHRoaXMuY29uZmlnID0gXy5tZXJnZShcbiAgICAgIHtcbiAgICAgICAgdmlld3M6IHtcbiAgICAgICAgICAvLyBkaXJlY3Rvcnkgd2hlcmUgZW1haWwgdGVtcGxhdGVzIHJlc2lkZVxuICAgICAgICAgIHJvb3Q6IHBhdGgucmVzb2x2ZSgnZW1haWxzJyksXG4gICAgICAgICAgb3B0aW9uczoge1xuICAgICAgICAgICAgLy8gZGVmYXVsdCBmaWxlIGV4dGVuc2lvbiBmb3IgdGVtcGxhdGVcbiAgICAgICAgICAgIGV4dGVuc2lvbjogJ3B1ZycsXG4gICAgICAgICAgICBtYXA6IHtcbiAgICAgICAgICAgICAgaGJzOiAnaGFuZGxlYmFycycsXG4gICAgICAgICAgICAgIG5qazogJ251bmp1Y2tzJ1xuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIGVuZ2luZVNvdXJjZTogY29uc29saWRhdGVcbiAgICAgICAgICB9LFxuICAgICAgICAgIC8vIGxvY2FscyB0byBwYXNzIHRvIHRlbXBsYXRlcyBmb3IgcmVuZGVyaW5nXG4gICAgICAgICAgbG9jYWxzOiB7XG4gICAgICAgICAgICAvLyB0dXJuIG9uIGNhY2hpbmcgZm9yIG5vbi1kZXZlbG9wbWVudCBlbnZpcm9ubWVudHNcbiAgICAgICAgICAgIGNhY2hlOiAhWydkZXZlbG9wbWVudCcsICd0ZXN0J10uaW5jbHVkZXMoZW52KSxcbiAgICAgICAgICAgIC8vIHByZXR0eSBpcyBhdXRvbWF0aWNhbGx5IHNldCB0byBgZmFsc2VgIGZvciBzdWJqZWN0L3RleHRcbiAgICAgICAgICAgIHByZXR0eTogdHJ1ZVxuICAgICAgICAgIH1cbiAgICAgICAgfSxcbiAgICAgICAgLy8gPGh0dHBzOi8vbm9kZW1haWxlci5jb20vbWVzc2FnZS8+XG4gICAgICAgIG1lc3NhZ2U6IHt9LFxuICAgICAgICBzZW5kOiAhWydkZXZlbG9wbWVudCcsICd0ZXN0J10uaW5jbHVkZXMoZW52KSxcbiAgICAgICAgcHJldmlldzogZW52ID09PSAnZGV2ZWxvcG1lbnQnLFxuICAgICAgICAvLyA8aHR0cHM6Ly9naXRodWIuY29tL2xhZGpzL2kxOG4+XG4gICAgICAgIC8vIHNldCB0byBhbiBvYmplY3QgdG8gY29uZmlndXJlIGFuZCBlbmFibGUgaXRcbiAgICAgICAgaTE4bjogZmFsc2UsXG4gICAgICAgIC8vIHBhc3MgYSBjdXN0b20gcmVuZGVyIGZ1bmN0aW9uIGlmIG5lY2Vzc2FyeVxuICAgICAgICByZW5kZXI6IHRoaXMucmVuZGVyLmJpbmQodGhpcyksXG4gICAgICAgIGN1c3RvbVJlbmRlcjogZmFsc2UsXG4gICAgICAgIC8vIGZvcmNlIHRleHQtb25seSByZW5kZXJpbmcgb2YgdGVtcGxhdGUgKGRpc3JlZ2FyZHMgdGVtcGxhdGUgZm9sZGVyKVxuICAgICAgICB0ZXh0T25seTogZmFsc2UsXG4gICAgICAgIC8vIDxodHRwczovL2dpdGh1Yi5jb20vaHRtbC10by10ZXh0L25vZGUtaHRtbC10by10ZXh0PlxuICAgICAgICBodG1sVG9UZXh0OiB7XG4gICAgICAgICAgc2VsZWN0b3JzOiBbeyBzZWxlY3RvcjogJ2ltZycsIGZvcm1hdDogJ3NraXAnIH1dXG4gICAgICAgIH0sXG4gICAgICAgIHN1YmplY3RQcmVmaXg6IGZhbHNlLFxuICAgICAgICAvLyA8aHR0cHM6Ly9naXRodWIuY29tL0F1dG9tYXR0aWMvanVpY2U+XG4gICAgICAgIGp1aWNlOiB0cnVlLFxuICAgICAgICAvLyBPdmVycmlkZSBqdWljZSBnbG9iYWwgc2V0dGluZ3MgPGh0dHBzOi8vZ2l0aHViLmNvbS9BdXRvbWF0dGljL2p1aWNlI2p1aWNlY29kZWJsb2Nrc3M+XG4gICAgICAgIGp1aWNlU2V0dGluZ3M6IHtcbiAgICAgICAgICB0YWJsZUVsZW1lbnRzOiBbJ1RBQkxFJ11cbiAgICAgICAgfSxcbiAgICAgICAganVpY2VSZXNvdXJjZXM6IHtcbiAgICAgICAgICBwcmVzZXJ2ZUltcG9ydGFudDogdHJ1ZSxcbiAgICAgICAgICB3ZWJSZXNvdXJjZXM6IHtcbiAgICAgICAgICAgIHJlbGF0aXZlVG86IHBhdGgucmVzb2x2ZSgnYnVpbGQnKSxcbiAgICAgICAgICAgIGltYWdlczogZmFsc2VcbiAgICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICAgIC8vIHBhc3MgYSB0cmFuc3BvcnQgY29uZmlndXJhdGlvbiBvYmplY3Qgb3IgYSB0cmFuc3BvcnQgaW5zdGFuY2VcbiAgICAgICAgLy8gKGUuZy4gYW4gaW5zdGFuY2UgaXMgY3JlYXRlZCB2aWEgYG5vZGVtYWlsZXIuY3JlYXRlVHJhbnNwb3J0YClcbiAgICAgICAgLy8gPGh0dHBzOi8vbm9kZW1haWxlci5jb20vdHJhbnNwb3J0cy8+XG4gICAgICAgIHRyYW5zcG9ydDoge30sXG4gICAgICAgIC8vIGxhc3QgbG9jYWxlIGZpZWxkIG5hbWUgKGFsc28gdXNlZCBieSBAbGFkanMvaTE4bilcbiAgICAgICAgbGFzdExvY2FsZUZpZWxkOiAnbGFzdF9sb2NhbGUnLFxuICAgICAgICBnZXRQYXRoKHR5cGUsIHRlbXBsYXRlKSB7XG4gICAgICAgICAgcmV0dXJuIHBhdGguam9pbih0ZW1wbGF0ZSwgdHlwZSk7XG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBjb25maWdcbiAgICApO1xuXG4gICAgLy8gb3ZlcnJpZGUgZXhpc3RpbmcgbWV0aG9kXG4gICAgdGhpcy5yZW5kZXIgPSB0aGlzLmNvbmZpZy5yZW5kZXI7XG5cbiAgICBpZiAoIV8uaXNGdW5jdGlvbih0aGlzLmNvbmZpZy50cmFuc3BvcnQuc2VuZE1haWwpKVxuICAgICAgdGhpcy5jb25maWcudHJhbnNwb3J0ID0gbm9kZW1haWxlci5jcmVhdGVUcmFuc3BvcnQodGhpcy5jb25maWcudHJhbnNwb3J0KTtcblxuICAgIC8vIE92ZXJyaWRlIGp1aWNlIGdsb2JhbCBzZXR0aW5ncyBodHRwczovL2dpdGh1Yi5jb20vQXV0b21hdHRpYy9qdWljZSNqdWljZWNvZGVibG9ja3NcbiAgICBpZiAoXy5pc09iamVjdCh0aGlzLmNvbmZpZy5qdWljZVNldHRpbmdzKSkge1xuICAgICAgZm9yIChjb25zdCBba2V5LCB2YWx1ZV0gb2YgT2JqZWN0LmVudHJpZXModGhpcy5jb25maWcuanVpY2VTZXR0aW5ncykpIHtcbiAgICAgICAganVpY2Vba2V5XSA9IHZhbHVlO1xuICAgICAgfVxuICAgIH1cblxuICAgIGRlYnVnKCd0cmFuc2Zvcm1lZCBjb25maWcgJU8nLCB0aGlzLmNvbmZpZyk7XG5cbiAgICB0aGlzLmp1aWNlUmVzb3VyY2VzID0gdGhpcy5qdWljZVJlc291cmNlcy5iaW5kKHRoaXMpO1xuICAgIHRoaXMuZ2V0VGVtcGxhdGVQYXRoID0gdGhpcy5nZXRUZW1wbGF0ZVBhdGguYmluZCh0aGlzKTtcbiAgICB0aGlzLnRlbXBsYXRlRXhpc3RzID0gdGhpcy50ZW1wbGF0ZUV4aXN0cy5iaW5kKHRoaXMpO1xuICAgIHRoaXMuY2hlY2tBbmRSZW5kZXIgPSB0aGlzLmNoZWNrQW5kUmVuZGVyLmJpbmQodGhpcyk7XG4gICAgdGhpcy5yZW5kZXIgPSB0aGlzLnJlbmRlci5iaW5kKHRoaXMpO1xuICAgIHRoaXMucmVuZGVyQWxsID0gdGhpcy5yZW5kZXJBbGwuYmluZCh0aGlzKTtcbiAgICB0aGlzLnNlbmQgPSB0aGlzLnNlbmQuYmluZCh0aGlzKTtcbiAgfVxuXG4gIC8vIHNob3J0aGFuZCB1c2Ugb2YgYGp1aWNlUmVzb3VyY2VzYCB3aXRoIHRoZSBjb25maWdcbiAgLy8gKG1haW5seSBmb3IgY3VzdG9tIHJlbmRlcnMgbGlrZSBmcm9tIGEgZGF0YWJhc2UpXG4gIGp1aWNlUmVzb3VyY2VzKGh0bWwsIGp1aWNlUmVuZGVyUmVzb3VyY2VzID0ge30pIHtcbiAgICBjb25zdCBqdWljZVIgPSBfLm1lcmdlKHRoaXMuY29uZmlnLmp1aWNlUmVzb3VyY2VzLCBqdWljZVJlbmRlclJlc291cmNlcyk7XG4gICAgcmV0dXJuIGp1aWNlUmVzb3VyY2VzKGh0bWwsIGp1aWNlUik7XG4gIH1cblxuICAvLyBhIHNpbXBsZSBoZWxwZXIgZnVuY3Rpb24gdGhhdCBnZXRzIHRoZSBhY3R1YWwgZmlsZSBwYXRoIGZvciB0aGUgdGVtcGxhdGVcbiAgYXN5bmMgZ2V0VGVtcGxhdGVQYXRoKHRlbXBsYXRlKSB7XG4gICAgbGV0IGp1aWNlUmVuZGVyUmVzb3VyY2VzID0ge307XG5cbiAgICBpZiAoXy5pc09iamVjdCh0ZW1wbGF0ZSkpIHtcbiAgICAgIGp1aWNlUmVuZGVyUmVzb3VyY2VzID0gdGVtcGxhdGUuanVpY2VSZXNvdXJjZXM7XG4gICAgICB0ZW1wbGF0ZSA9IHRlbXBsYXRlLnBhdGg7XG4gICAgfVxuXG4gICAgY29uc3QgW3Jvb3QsIHZpZXddID0gcGF0aC5pc0Fic29sdXRlKHRlbXBsYXRlKVxuICAgICAgPyBbcGF0aC5kaXJuYW1lKHRlbXBsYXRlKSwgcGF0aC5iYXNlbmFtZSh0ZW1wbGF0ZSldXG4gICAgICA6IFt0aGlzLmNvbmZpZy52aWV3cy5yb290LCB0ZW1wbGF0ZV07XG4gICAgY29uc3QgcGF0aHMgPSBhd2FpdCBnZXRQYXRocyhcbiAgICAgIHJvb3QsXG4gICAgICB2aWV3LFxuICAgICAgdGhpcy5jb25maWcudmlld3Mub3B0aW9ucy5leHRlbnNpb25cbiAgICApO1xuICAgIGNvbnN0IGZpbGVQYXRoID0gcGF0aC5yZXNvbHZlKHJvb3QsIHBhdGhzLnJlbCk7XG4gICAgcmV0dXJuIHsgZmlsZVBhdGgsIHBhdGhzLCBqdWljZVJlbmRlclJlc291cmNlcyB9O1xuICB9XG5cbiAgLy8gcmV0dXJucyB0cnVlIG9yIGZhbHNlIGlmIGEgdGVtcGxhdGUgZXhpc3RzXG4gIC8vICh1c2VzIHNhbWUgbG9vay11cCBhcHByb2FjaCBhcyBgcmVuZGVyYCBmdW5jdGlvbilcbiAgYXN5bmMgdGVtcGxhdGVFeGlzdHModmlldykge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCB7IGZpbGVQYXRoIH0gPSBhd2FpdCB0aGlzLmdldFRlbXBsYXRlUGF0aCh2aWV3KTtcbiAgICAgIGNvbnN0IHN0YXRzID0gYXdhaXQgc3RhdChmaWxlUGF0aCk7XG4gICAgICBpZiAoIXN0YXRzLmlzRmlsZSgpKSB0aHJvdyBuZXcgRXJyb3IoYCR7ZmlsZVBhdGh9IHdhcyBub3QgYSBmaWxlYCk7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGRlYnVnKCd0ZW1wbGF0ZUV4aXN0cycsIGVycik7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuICB9XG5cbiAgYXN5bmMgY2hlY2tBbmRSZW5kZXIodHlwZSwgdGVtcGxhdGUsIGxvY2Fscykge1xuICAgIGxldCBqdWljZVJlbmRlclJlc291cmNlcyA9IHt9O1xuXG4gICAgaWYgKF8uaXNPYmplY3QodGVtcGxhdGUpKSB7XG4gICAgICBqdWljZVJlbmRlclJlc291cmNlcyA9IHRlbXBsYXRlLmp1aWNlUmVzb3VyY2VzO1xuICAgICAgdGVtcGxhdGUgPSB0ZW1wbGF0ZS5wYXRoO1xuICAgIH1cblxuICAgIGNvbnN0IHN0cmluZyA9IHRoaXMuY29uZmlnLmdldFBhdGgodHlwZSwgdGVtcGxhdGUsIGxvY2Fscyk7XG4gICAgaWYgKCF0aGlzLmNvbmZpZy5jdXN0b21SZW5kZXIpIHtcbiAgICAgIGNvbnN0IGV4aXN0cyA9IGF3YWl0IHRoaXMudGVtcGxhdGVFeGlzdHMoc3RyaW5nKTtcbiAgICAgIGlmICghZXhpc3RzKSByZXR1cm47XG4gICAgfVxuXG4gICAgcmV0dXJuIHRoaXMucmVuZGVyKFxuICAgICAgc3RyaW5nLFxuICAgICAge1xuICAgICAgICAuLi5sb2NhbHMsXG4gICAgICAgIC4uLih0eXBlID09PSAnaHRtbCcgPyB7fSA6IHsgcHJldHR5OiBmYWxzZSB9KVxuICAgICAgfSxcbiAgICAgIGp1aWNlUmVuZGVyUmVzb3VyY2VzXG4gICAgKTtcbiAgfVxuXG4gIC8vIHByb21pc2UgdmVyc2lvbiBvZiBjb25zb2xpZGF0ZSdzIHJlbmRlclxuICAvLyBpbnNwaXJlZCBieSBrb2Etdmlld3MgYW5kIHJlLXVzZXMgdGhlIHNhbWUgY29uZmlnXG4gIC8vIDxodHRwczovL2dpdGh1Yi5jb20vcXVlY2tlenova29hLXZpZXdzPlxuICBhc3luYyByZW5kZXIodmlldywgbG9jYWxzID0ge30pIHtcbiAgICBjb25zdCB7IG1hcCwgZW5naW5lU291cmNlIH0gPSB0aGlzLmNvbmZpZy52aWV3cy5vcHRpb25zO1xuICAgIGNvbnN0IHsgZmlsZVBhdGgsIHBhdGhzLCBqdWljZVJlbmRlclJlc291cmNlcyB9ID1cbiAgICAgIGF3YWl0IHRoaXMuZ2V0VGVtcGxhdGVQYXRoKHZpZXcpO1xuICAgIGlmIChwYXRocy5leHQgPT09ICdodG1sJyAmJiAhbWFwKSB7XG4gICAgICBjb25zdCByZXMgPSBhd2FpdCByZWFkRmlsZShmaWxlUGF0aCwgJ3V0ZjgnKTtcbiAgICAgIHJldHVybiByZXM7XG4gICAgfVxuXG4gICAgY29uc3QgZW5naW5lTmFtZSA9IG1hcCAmJiBtYXBbcGF0aHMuZXh0XSA/IG1hcFtwYXRocy5leHRdIDogcGF0aHMuZXh0O1xuICAgIGNvbnN0IHJlbmRlckZuID0gZW5naW5lU291cmNlW2VuZ2luZU5hbWVdO1xuICAgIGlmICghZW5naW5lTmFtZSB8fCAhcmVuZGVyRm4pXG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIGBFbmdpbmUgbm90IGZvdW5kIGZvciB0aGUgXCIuJHtwYXRocy5leHR9XCIgZmlsZSBleHRlbnNpb25gXG4gICAgICApO1xuXG4gICAgaWYgKF8uaXNPYmplY3QodGhpcy5jb25maWcuaTE4bikpIHtcbiAgICAgIGlmIChcbiAgICAgICAgdGhpcy5jb25maWcuaTE4bi5sYXN0TG9jYWxlRmllbGQgJiZcbiAgICAgICAgdGhpcy5jb25maWcubGFzdExvY2FsZUZpZWxkICYmXG4gICAgICAgIHRoaXMuY29uZmlnLmkxOG4ubGFzdExvY2FsZUZpZWxkICE9PSB0aGlzLmNvbmZpZy5sYXN0TG9jYWxlRmllbGRcbiAgICAgIClcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgIGBUaGUgJ2xhc3RMb2NhbGVGaWVsZCcgKFN0cmluZykgb3B0aW9uIGZvciBAbGFkanMvaTE4biBhbmQgZW1haWwtdGVtcGxhdGVzIGRvIG5vdCBtYXRjaCwgaTE4biB2YWx1ZSB3YXMgJHt0aGlzLmNvbmZpZy5pMThuLmxhc3RMb2NhbGVGaWVsZH0gYW5kIGVtYWlsLXRlbXBsYXRlcyB2YWx1ZSB3YXMgJHt0aGlzLmNvbmZpZy5sYXN0TG9jYWxlRmllbGR9YFxuICAgICAgICApO1xuXG4gICAgICBjb25zdCBpMThuID0gbmV3IEkxOE4oeyAuLi50aGlzLmNvbmZpZy5pMThuLCByZWdpc3RlcjogbG9jYWxzIH0pO1xuXG4gICAgICAvLyBzdXBwb3J0IGBsb2NhbHMudXNlci5sYXN0X2xvY2FsZWAgKHZhcmlhYmxlIGJhc2VkIG5hbWUgbGFzdExvY2FsZUZpZWxkKVxuICAgICAgLy8gKGUuZy4gZm9yIDxodHRwczovL2xhZC5qcy5vcmc+KVxuICAgICAgY29uc3QgbG9jYWxlID0gaTE4bi5nZXRMb2NhbGUoKTtcbiAgICAgIGlmIChcbiAgICAgICAgXy5pc09iamVjdChsb2NhbHMudXNlcikgJiZcbiAgICAgICAgXy5pc1N0cmluZyhsb2NhbHMudXNlclt0aGlzLmNvbmZpZy5sYXN0TG9jYWxlRmllbGRdKVxuICAgICAgKVxuICAgICAgICBsb2NhbHMubG9jYWxlID0gbG9jYWxzLnVzZXJbdGhpcy5jb25maWcubGFzdExvY2FsZUZpZWxkXTtcbiAgICAgIGVsc2UgaWYgKCFfLmlzU3RyaW5nKGxvY2Fscy5sb2NhbGUpKSBsb2NhbHMubG9jYWxlID0gbG9jYWxlO1xuXG4gICAgICBpZiAobG9jYWxlICE9PSBsb2NhbHMubG9jYWxlKSBpMThuLnNldExvY2FsZShsb2NhbHMubG9jYWxlKTtcbiAgICB9XG5cbiAgICBjb25zdCByZXMgPSBhd2FpdCB1dGlsLnByb21pc2lmeShyZW5kZXJGbikoZmlsZVBhdGgsIGxvY2Fscyk7XG4gICAgLy8gdHJhbnNmb3JtIHRoZSBodG1sIHdpdGgganVpY2UgdXNpbmcgcmVtb3RlIHBhdGhzXG4gICAgLy8gZ29vZ2xlIG5vdyBzdXBwb3J0cyBtZWRpYSBxdWVyaWVzXG4gICAgLy8gaHR0cHM6Ly9kZXZlbG9wZXJzLmdvb2dsZS5jb20vZ21haWwvZGVzaWduL3JlZmVyZW5jZS9zdXBwb3J0ZWRfY3NzXG4gICAgaWYgKCF0aGlzLmNvbmZpZy5qdWljZSkgcmV0dXJuIHJlcztcbiAgICBjb25zdCBodG1sID0gYXdhaXQgdGhpcy5qdWljZVJlc291cmNlcyhyZXMsIGp1aWNlUmVuZGVyUmVzb3VyY2VzKTtcbiAgICByZXR1cm4gaHRtbDtcbiAgfVxuXG4gIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBjb21wbGV4aXR5XG4gIGFzeW5jIHJlbmRlckFsbCh0ZW1wbGF0ZSwgbG9jYWxzID0ge30sIG5vZGVtYWlsZXJNZXNzYWdlID0ge30pIHtcbiAgICBjb25zdCBtZXNzYWdlID0geyAuLi5ub2RlbWFpbGVyTWVzc2FnZSB9O1xuXG4gICAgaWYgKHRlbXBsYXRlICYmICghbWVzc2FnZS5zdWJqZWN0IHx8ICFtZXNzYWdlLmh0bWwgfHwgIW1lc3NhZ2UudGV4dCkpIHtcbiAgICAgIGNvbnN0IFtzdWJqZWN0LCBodG1sLCB0ZXh0XSA9IGF3YWl0IFByb21pc2UuYWxsKFxuICAgICAgICBbJ3N1YmplY3QnLCAnaHRtbCcsICd0ZXh0J10ubWFwKCh0eXBlKSA9PlxuICAgICAgICAgIHRoaXMuY2hlY2tBbmRSZW5kZXIodHlwZSwgdGVtcGxhdGUsIGxvY2FscylcbiAgICAgICAgKVxuICAgICAgKTtcblxuICAgICAgaWYgKHN1YmplY3QgJiYgIW1lc3NhZ2Uuc3ViamVjdCkgbWVzc2FnZS5zdWJqZWN0ID0gc3ViamVjdDtcbiAgICAgIGlmIChodG1sICYmICFtZXNzYWdlLmh0bWwpIG1lc3NhZ2UuaHRtbCA9IGh0bWw7XG4gICAgICBpZiAodGV4dCAmJiAhbWVzc2FnZS50ZXh0KSBtZXNzYWdlLnRleHQgPSB0ZXh0O1xuICAgIH1cblxuICAgIGlmIChtZXNzYWdlLnN1YmplY3QgJiYgdGhpcy5jb25maWcuc3ViamVjdFByZWZpeClcbiAgICAgIG1lc3NhZ2Uuc3ViamVjdCA9IHRoaXMuY29uZmlnLnN1YmplY3RQcmVmaXggKyBtZXNzYWdlLnN1YmplY3Q7XG5cbiAgICAvLyB0cmltIHN1YmplY3RcbiAgICBpZiAobWVzc2FnZS5zdWJqZWN0KSBtZXNzYWdlLnN1YmplY3QgPSBtZXNzYWdlLnN1YmplY3QudHJpbSgpO1xuXG4gICAgaWYgKHRoaXMuY29uZmlnLmh0bWxUb1RleHQgJiYgbWVzc2FnZS5odG1sICYmICFtZXNzYWdlLnRleHQpXG4gICAgICAvLyB3ZSdkIHVzZSBub2RlbWFpbGVyLWh0bWwtdG8tdGV4dCBwbHVnaW5cbiAgICAgIC8vIGJ1dCB3ZSByZWFsbHkgZG9uJ3QgbmVlZCB0byBzdXBwb3J0IGNpZFxuICAgICAgLy8gPGh0dHBzOi8vZ2l0aHViLmNvbS9hbmRyaXM5L25vZGVtYWlsZXItaHRtbC10by10ZXh0PlxuICAgICAgLy8gKGFuZCBpdCBpcyBhbHNvIG5vdCB1cGRhdGVkIHdpdGggbGF0ZXN0IGh0bWwtdG8tdGV4dClcbiAgICAgIG1lc3NhZ2UudGV4dCA9IGNvbnZlcnQobWVzc2FnZS5odG1sLCB0aGlzLmNvbmZpZy5odG1sVG9UZXh0KTtcblxuICAgIC8vIGlmIHdlIG9ubHkgd2FudCBhIHRleHQtYmFzZWQgdmVyc2lvbiBvZiB0aGUgZW1haWxcbiAgICBpZiAodGhpcy5jb25maWcudGV4dE9ubHkpIGRlbGV0ZSBtZXNzYWdlLmh0bWw7XG5cbiAgICAvLyBpZiBubyBzdWJqZWN0LCBodG1sLCBvciB0ZXh0IGNvbnRlbnQgZXhpc3RzIHRoZW4gd2Ugc2hvdWxkXG4gICAgLy8gdGhyb3cgYW4gZXJyb3IgdGhhdCBzYXlzIGF0IGxlYXN0IG9uZSBtdXN0IGJlIGZvdW5kXG4gICAgLy8gb3RoZXJ3aXNlIHRoZSBlbWFpbCB3b3VsZCBiZSBibGFuayAoZGVmZWF0cyBwdXJwb3NlIG9mIGVtYWlsLXRlbXBsYXRlcylcbiAgICBpZiAoXG4gICAgICAoIV8uaXNTdHJpbmcobWVzc2FnZS5zdWJqZWN0KSB8fCBfLmlzRW1wdHkoXy50cmltKG1lc3NhZ2Uuc3ViamVjdCkpKSAmJlxuICAgICAgKCFfLmlzU3RyaW5nKG1lc3NhZ2UudGV4dCkgfHwgXy5pc0VtcHR5KF8udHJpbShtZXNzYWdlLnRleHQpKSkgJiZcbiAgICAgICghXy5pc1N0cmluZyhtZXNzYWdlLmh0bWwpIHx8IF8uaXNFbXB0eShfLnRyaW0obWVzc2FnZS5odG1sKSkpICYmXG4gICAgICBfLmlzRW1wdHkobWVzc2FnZS5hdHRhY2htZW50cylcbiAgICApXG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIGBObyBjb250ZW50IHdhcyBwYXNzZWQgZm9yIHN1YmplY3QsIGh0bWwsIHRleHQsIG5vciBhdHRhY2htZW50cyBtZXNzYWdlIHByb3BzLiBDaGVjayB0aGF0IHRoZSBmaWxlcyBmb3IgdGhlIHRlbXBsYXRlIFwiJHt0ZW1wbGF0ZX1cIiBleGlzdC5gXG4gICAgICApO1xuXG4gICAgcmV0dXJuIG1lc3NhZ2U7XG4gIH1cblxuICBhc3luYyBzZW5kKG9wdGlvbnMgPSB7fSkge1xuICAgIG9wdGlvbnMgPSB7XG4gICAgICB0ZW1wbGF0ZTogJycsXG4gICAgICBtZXNzYWdlOiB7fSxcbiAgICAgIGxvY2Fsczoge30sXG4gICAgICAuLi5vcHRpb25zXG4gICAgfTtcblxuICAgIGxldCB7IHRlbXBsYXRlLCBtZXNzYWdlLCBsb2NhbHMgfSA9IG9wdGlvbnM7XG5cbiAgICBjb25zdCBhdHRhY2htZW50cyA9XG4gICAgICBtZXNzYWdlLmF0dGFjaG1lbnRzIHx8IHRoaXMuY29uZmlnLm1lc3NhZ2UuYXR0YWNobWVudHMgfHwgW107XG5cbiAgICBtZXNzYWdlID0gXy5kZWZhdWx0c0RlZXAoXG4gICAgICB7fSxcbiAgICAgIF8ub21pdChtZXNzYWdlLCAnYXR0YWNobWVudHMnKSxcbiAgICAgIF8ub21pdCh0aGlzLmNvbmZpZy5tZXNzYWdlLCAnYXR0YWNobWVudHMnKVxuICAgICk7XG4gICAgbG9jYWxzID0gXy5kZWZhdWx0c0RlZXAoe30sIHRoaXMuY29uZmlnLnZpZXdzLmxvY2FscywgbG9jYWxzKTtcblxuICAgIGlmIChhdHRhY2htZW50cykgbWVzc2FnZS5hdHRhY2htZW50cyA9IGF0dGFjaG1lbnRzO1xuXG4gICAgZGVidWcoJ3RlbXBsYXRlICVzJywgdGVtcGxhdGUpO1xuICAgIGRlYnVnKCdtZXNzYWdlICVPJywgbWVzc2FnZSk7XG4gICAgZGVidWcoJ2xvY2FscyAoa2V5cyBvbmx5KTogJU8nLCBPYmplY3Qua2V5cyhsb2NhbHMpKTtcblxuICAgIC8vIGdldCBhbGwgYXZhaWxhYmxlIHRlbXBsYXRlc1xuICAgIGNvbnN0IG9iamVjdCA9IGF3YWl0IHRoaXMucmVuZGVyQWxsKHRlbXBsYXRlLCBsb2NhbHMsIG1lc3NhZ2UpO1xuXG4gICAgLy8gYXNzaWduIHRoZSBvYmplY3QgdmFyaWFibGVzIG92ZXIgdG8gdGhlIG1lc3NhZ2VcbiAgICBPYmplY3QuYXNzaWduKG1lc3NhZ2UsIG9iamVjdCk7XG5cbiAgICBpZiAodGhpcy5jb25maWcucHJldmlldykge1xuICAgICAgZGVidWcoJ3VzaW5nIGBwcmV2aWV3LWVtYWlsYCB0byBwcmV2aWV3IGVtYWlsJyk7XG4gICAgICBhd2FpdCAoXy5pc09iamVjdCh0aGlzLmNvbmZpZy5wcmV2aWV3KVxuICAgICAgICA/IHByZXZpZXdFbWFpbChtZXNzYWdlLCB0aGlzLmNvbmZpZy5wcmV2aWV3KVxuICAgICAgICA6IHByZXZpZXdFbWFpbChtZXNzYWdlKSk7XG4gICAgfVxuXG4gICAgaWYgKCF0aGlzLmNvbmZpZy5zZW5kKSB7XG4gICAgICBkZWJ1Zygnc2VuZCBkaXNhYmxlZCBzbyB3ZSBhcmUgZW5zdXJpbmcgSlNPTlRyYW5zcG9ydCcpO1xuICAgICAgLy8gPGh0dHBzOi8vZ2l0aHViLmNvbS9ub2RlbWFpbGVyL25vZGVtYWlsZXIvaXNzdWVzLzc5OD5cbiAgICAgIC8vIGlmICh0aGlzLmNvbmZpZy50cmFuc3BvcnQubmFtZSAhPT0gJ0pTT05UcmFuc3BvcnQnKVxuICAgICAgdGhpcy5jb25maWcudHJhbnNwb3J0ID0gbm9kZW1haWxlci5jcmVhdGVUcmFuc3BvcnQoe1xuICAgICAgICBqc29uVHJhbnNwb3J0OiB0cnVlXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCByZXMgPSBhd2FpdCB0aGlzLmNvbmZpZy50cmFuc3BvcnQuc2VuZE1haWwobWVzc2FnZSk7XG4gICAgZGVidWcoJ21lc3NhZ2Ugc2VudCcpO1xuICAgIHJlcy5vcmlnaW5hbE1lc3NhZ2UgPSBtZXNzYWdlO1xuICAgIHJldHVybiByZXM7XG4gIH1cbn1cblxubW9kdWxlLmV4cG9ydHMgPSBFbWFpbDtcbiJdfQ==