"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.canMakePermissionsChange = void 0;
const plextv_1 = __importDefault(require("../../api/plextv"));
const tautulli_1 = __importDefault(require("../../api/tautulli"));
const media_1 = require("../../constants/media");
const user_1 = require("../../constants/user");
const datasource_1 = require("../../datasource");
const Media_1 = __importDefault(require("../../entity/Media"));
const MediaRequest_1 = require("../../entity/MediaRequest");
const User_1 = require("../../entity/User");
const UserPushSubscription_1 = require("../../entity/UserPushSubscription");
const permissions_1 = require("../../lib/permissions");
const settings_1 = require("../../lib/settings");
const logger_1 = __importDefault(require("../../logger"));
const auth_1 = require("../../middleware/auth");
const express_1 = require("express");
const gravatar_url_1 = __importDefault(require("gravatar-url"));
const lodash_1 = require("lodash");
const typeorm_1 = require("typeorm");
const usersettings_1 = __importDefault(require("./usersettings"));
const router = (0, express_1.Router)();
router.get('/', async (req, res, next) => {
    try {
        const pageSize = req.query.take ? Number(req.query.take) : 10;
        const skip = req.query.skip ? Number(req.query.skip) : 0;
        let query = (0, datasource_1.getRepository)(User_1.User).createQueryBuilder('user');
        switch (req.query.sort) {
            case 'updated':
                query = query.orderBy('user.updatedAt', 'DESC');
                break;
            case 'displayname':
                query = query.orderBy("(CASE WHEN (user.username IS NULL OR user.username = '') THEN (CASE WHEN (user.plexUsername IS NULL OR user.plexUsername = '') THEN user.email ELSE LOWER(user.plexUsername) END) ELSE LOWER(user.username) END)", 'ASC');
                break;
            case 'requests':
                query = query
                    .addSelect((subQuery) => {
                    return subQuery
                        .select('COUNT(request.id)', 'requestCount')
                        .from(MediaRequest_1.MediaRequest, 'request')
                        .where('request.requestedBy.id = user.id');
                }, 'requestCount')
                    .orderBy('requestCount', 'DESC');
                break;
            default:
                query = query.orderBy('user.id', 'ASC');
                break;
        }
        const [users, userCount] = await query
            .take(pageSize)
            .skip(skip)
            .getManyAndCount();
        return res.status(200).json({
            pageInfo: {
                pages: Math.ceil(userCount / pageSize),
                pageSize,
                results: userCount,
                page: Math.ceil(skip / pageSize) + 1,
            },
            results: User_1.User.filterMany(users, req.user?.hasPermission(permissions_1.Permission.MANAGE_USERS)),
        });
    }
    catch (e) {
        next({ status: 500, message: e.message });
    }
});
router.post('/', (0, auth_1.isAuthenticated)(permissions_1.Permission.MANAGE_USERS), async (req, res, next) => {
    try {
        const settings = (0, settings_1.getSettings)();
        const body = req.body;
        const userRepository = (0, datasource_1.getRepository)(User_1.User);
        const existingUser = await userRepository
            .createQueryBuilder('user')
            .where('user.email = :email', {
            email: body.email.toLowerCase(),
        })
            .getOne();
        if (existingUser) {
            return next({
                status: 409,
                message: 'User already exists with submitted email.',
                errors: ['USER_EXISTS'],
            });
        }
        const passedExplicitPassword = body.password && body.password.length > 0;
        const avatar = (0, gravatar_url_1.default)(body.email, { default: 'mm', size: 200 });
        if (!passedExplicitPassword &&
            !settings.notifications.agents.email.enabled) {
            throw new Error('Email notifications must be enabled');
        }
        const user = new User_1.User({
            avatar: body.avatar ?? avatar,
            username: body.username,
            email: body.email,
            password: body.password,
            permissions: settings.main.defaultPermissions,
            plexToken: '',
            userType: user_1.UserType.LOCAL,
        });
        if (passedExplicitPassword) {
            await user?.setPassword(body.password);
        }
        else {
            await user?.generatePassword();
        }
        await userRepository.save(user);
        return res.status(201).json(user.filter());
    }
    catch (e) {
        next({ status: 500, message: e.message });
    }
});
router.post('/registerPushSubscription', async (req, res, next) => {
    try {
        const userPushSubRepository = (0, datasource_1.getRepository)(UserPushSubscription_1.UserPushSubscription);
        const existingSubs = await userPushSubRepository.find({
            relations: { user: true },
            where: { auth: req.body.auth, user: { id: req.user?.id } },
        });
        if (existingSubs.length > 0) {
            logger_1.default.debug('User push subscription already exists. Skipping registration.', { label: 'API' });
            return res.status(204).send();
        }
        const userPushSubscription = new UserPushSubscription_1.UserPushSubscription({
            auth: req.body.auth,
            endpoint: req.body.endpoint,
            p256dh: req.body.p256dh,
            userAgent: req.body.userAgent,
            user: req.user,
        });
        userPushSubRepository.save(userPushSubscription);
        return res.status(204).send();
    }
    catch (e) {
        logger_1.default.error('Failed to register user push subscription', {
            label: 'API',
        });
        next({ status: 500, message: 'Failed to register subscription.' });
    }
});
router.get('/:userId/pushSubscriptions', async (req, res, next) => {
    try {
        const userPushSubRepository = (0, datasource_1.getRepository)(UserPushSubscription_1.UserPushSubscription);
        const userPushSubs = await userPushSubRepository.find({
            relations: { user: true },
            where: { user: { id: req.params.userId } },
        });
        return res.status(200).json(userPushSubs);
    }
    catch (e) {
        next({ status: 404, message: 'User subscriptions not found.' });
    }
});
router.get('/:userId/pushSubscription/:endpoint', async (req, res, next) => {
    try {
        const userPushSubRepository = (0, datasource_1.getRepository)(UserPushSubscription_1.UserPushSubscription);
        const userPushSub = await userPushSubRepository.findOneOrFail({
            relations: {
                user: true,
            },
            where: {
                user: { id: req.params.userId },
                endpoint: req.params.endpoint,
            },
        });
        return res.status(200).json(userPushSub);
    }
    catch (e) {
        next({ status: 404, message: 'User subscription not found.' });
    }
});
router.delete('/:userId/pushSubscription/:endpoint', async (req, res, next) => {
    try {
        const userPushSubRepository = (0, datasource_1.getRepository)(UserPushSubscription_1.UserPushSubscription);
        const userPushSub = await userPushSubRepository.findOneOrFail({
            relations: {
                user: true,
            },
            where: {
                user: { id: req.params.userId },
                endpoint: req.params.endpoint,
            },
        });
        await userPushSubRepository.remove(userPushSub);
        return res.status(204).send();
    }
    catch (e) {
        logger_1.default.error('Something went wrong deleting the user push subcription', {
            label: 'API',
            endpoint: req.params.endpoint,
            errorMessage: e.message,
        });
        return next({
            status: 500,
            message: 'User push subcription not found',
        });
    }
});
router.get('/:id', async (req, res, next) => {
    try {
        const userRepository = (0, datasource_1.getRepository)(User_1.User);
        const user = await userRepository.findOneOrFail({
            where: { id: Number(req.params.id) },
        });
        return res
            .status(200)
            .json(user.filter(req.user?.hasPermission(permissions_1.Permission.MANAGE_USERS)));
    }
    catch (e) {
        next({ status: 404, message: 'User not found.' });
    }
});
router.use('/:id/settings', usersettings_1.default);
router.get('/:id/requests', async (req, res, next) => {
    const pageSize = req.query.take ? Number(req.query.take) : 20;
    const skip = req.query.skip ? Number(req.query.skip) : 0;
    try {
        const user = await (0, datasource_1.getRepository)(User_1.User).findOne({
            where: { id: Number(req.params.id) },
        });
        if (!user) {
            return next({ status: 404, message: 'User not found.' });
        }
        if (user.id !== req.user?.id &&
            !req.user?.hasPermission([permissions_1.Permission.MANAGE_REQUESTS, permissions_1.Permission.REQUEST_VIEW], { type: 'or' })) {
            return next({
                status: 403,
                message: "You do not have permission to view this user's requests.",
            });
        }
        const [requests, requestCount] = await (0, datasource_1.getRepository)(MediaRequest_1.MediaRequest)
            .createQueryBuilder('request')
            .leftJoinAndSelect('request.media', 'media')
            .leftJoinAndSelect('request.seasons', 'seasons')
            .leftJoinAndSelect('request.modifiedBy', 'modifiedBy')
            .leftJoinAndSelect('request.requestedBy', 'requestedBy')
            .andWhere('requestedBy.id = :id', {
            id: user.id,
        })
            .orderBy('request.id', 'DESC')
            .take(pageSize)
            .skip(skip)
            .getManyAndCount();
        return res.status(200).json({
            pageInfo: {
                pages: Math.ceil(requestCount / pageSize),
                pageSize,
                results: requestCount,
                page: Math.ceil(skip / pageSize) + 1,
            },
            results: requests,
        });
    }
    catch (e) {
        next({ status: 500, message: e.message });
    }
});
const canMakePermissionsChange = (permissions, user) => 
// Only let the owner grant admin privileges
!((0, permissions_1.hasPermission)(permissions_1.Permission.ADMIN, permissions) && user?.id !== 1);
exports.canMakePermissionsChange = canMakePermissionsChange;
router.put('/', (0, auth_1.isAuthenticated)(permissions_1.Permission.MANAGE_USERS), async (req, res, next) => {
    try {
        const isOwner = req.user?.id === 1;
        if (!(0, exports.canMakePermissionsChange)(req.body.permissions, req.user)) {
            return next({
                status: 403,
                message: 'You do not have permission to grant this level of access',
            });
        }
        const userRepository = (0, datasource_1.getRepository)(User_1.User);
        const users = await userRepository.find({
            where: {
                id: (0, typeorm_1.In)(isOwner ? req.body.ids : req.body.ids.filter((id) => Number(id) !== 1)),
            },
        });
        const updatedUsers = await Promise.all(users.map(async (user) => {
            return userRepository.save({
                ...user,
                ...{ permissions: req.body.permissions },
            });
        }));
        return res.status(200).json(updatedUsers);
    }
    catch (e) {
        next({ status: 500, message: e.message });
    }
});
router.put('/:id', (0, auth_1.isAuthenticated)(permissions_1.Permission.MANAGE_USERS), async (req, res, next) => {
    try {
        const userRepository = (0, datasource_1.getRepository)(User_1.User);
        const user = await userRepository.findOneOrFail({
            where: { id: Number(req.params.id) },
        });
        // Only let the owner user modify themselves
        if (user.id === 1 && req.user?.id !== 1) {
            return next({
                status: 403,
                message: 'You do not have permission to modify this user',
            });
        }
        if (!(0, exports.canMakePermissionsChange)(req.body.permissions, req.user)) {
            return next({
                status: 403,
                message: 'You do not have permission to grant this level of access',
            });
        }
        Object.assign(user, {
            username: req.body.username,
            permissions: req.body.permissions,
        });
        await userRepository.save(user);
        return res.status(200).json(user.filter());
    }
    catch (e) {
        next({ status: 404, message: 'User not found.' });
    }
});
router.delete('/:id', (0, auth_1.isAuthenticated)(permissions_1.Permission.MANAGE_USERS), async (req, res, next) => {
    try {
        const userRepository = (0, datasource_1.getRepository)(User_1.User);
        const user = await userRepository.findOne({
            where: { id: Number(req.params.id) },
            relations: { requests: true },
        });
        if (!user) {
            return next({ status: 404, message: 'User not found.' });
        }
        if (user.id === 1) {
            return next({
                status: 405,
                message: 'This account cannot be deleted.',
            });
        }
        if (user.hasPermission(permissions_1.Permission.ADMIN) && req.user?.id !== 1) {
            return next({
                status: 405,
                message: 'You cannot delete users with administrative privileges.',
            });
        }
        const requestRepository = (0, datasource_1.getRepository)(MediaRequest_1.MediaRequest);
        /**
         * Requests are usually deleted through a cascade constraint. Those however, do
         * not trigger the removal event so listeners to not run and the parent Media
         * will not be updated back to unknown for titles that were still pending. So
         * we manually remove all requests from the user here so the parent media's
         * properly reflect the change.
         */
        await requestRepository.remove(user.requests, {
            /**
             * Break-up into groups of 1000 requests to be removed at a time.
             * Necessary for users with >1000 requests, else an SQLite 'Expression tree is too large' error occurs.
             * https://typeorm.io/repository-api#additional-options
             */
            chunk: user.requests.length / 1000,
        });
        await userRepository.delete(user.id);
        return res.status(200).json(user.filter());
    }
    catch (e) {
        logger_1.default.error('Something went wrong deleting a user', {
            label: 'API',
            userId: req.params.id,
            errorMessage: e.message,
        });
        return next({
            status: 500,
            message: 'Something went wrong deleting the user',
        });
    }
});
router.post('/import-from-plex', (0, auth_1.isAuthenticated)(permissions_1.Permission.MANAGE_USERS), async (req, res, next) => {
    try {
        const settings = (0, settings_1.getSettings)();
        const userRepository = (0, datasource_1.getRepository)(User_1.User);
        const body = req.body;
        // taken from auth.ts
        const mainUser = await userRepository.findOneOrFail({
            select: { id: true, plexToken: true },
            where: { id: 1 },
        });
        const mainPlexTv = new plextv_1.default(mainUser.plexToken ?? '');
        const plexUsersResponse = await mainPlexTv.getUsers();
        const createdUsers = [];
        for (const rawUser of plexUsersResponse.MediaContainer.User) {
            const account = rawUser.$;
            if (account.email) {
                const user = await userRepository
                    .createQueryBuilder('user')
                    .where('user.plexId = :id', { id: account.id })
                    .orWhere('user.email = :email', {
                    email: account.email.toLowerCase(),
                })
                    .getOne();
                if (user) {
                    // Update the user's avatar with their Plex thumbnail, in case it changed
                    user.avatar = account.thumb;
                    user.email = account.email;
                    user.plexUsername = account.username;
                    // In case the user was previously a local account
                    if (user.userType === user_1.UserType.LOCAL) {
                        user.userType = user_1.UserType.PLEX;
                        user.plexId = parseInt(account.id);
                    }
                    await userRepository.save(user);
                }
                else if (!body || body.plexIds.includes(account.id)) {
                    if (await mainPlexTv.checkUserAccess(parseInt(account.id))) {
                        const newUser = new User_1.User({
                            plexUsername: account.username,
                            email: account.email,
                            permissions: settings.main.defaultPermissions,
                            plexId: parseInt(account.id),
                            plexToken: '',
                            avatar: account.thumb,
                            userType: user_1.UserType.PLEX,
                        });
                        await userRepository.save(newUser);
                        createdUsers.push(newUser);
                    }
                }
            }
        }
        return res.status(201).json(User_1.User.filterMany(createdUsers));
    }
    catch (e) {
        next({ status: 500, message: e.message });
    }
});
router.get('/:id/quota', async (req, res, next) => {
    try {
        const userRepository = (0, datasource_1.getRepository)(User_1.User);
        if (Number(req.params.id) !== req.user?.id &&
            !req.user?.hasPermission([permissions_1.Permission.MANAGE_USERS, permissions_1.Permission.MANAGE_REQUESTS], { type: 'and' })) {
            return next({
                status: 403,
                message: "You do not have permission to view this user's request limits.",
            });
        }
        const user = await userRepository.findOneOrFail({
            where: { id: Number(req.params.id) },
        });
        const quotas = await user.getQuota();
        return res.status(200).json(quotas);
    }
    catch (e) {
        next({ status: 404, message: e.message });
    }
});
router.get('/:id/watch_data', async (req, res, next) => {
    if (Number(req.params.id) !== req.user?.id &&
        !req.user?.hasPermission(permissions_1.Permission.ADMIN)) {
        return next({
            status: 403,
            message: "You do not have permission to view this user's recently watched media.",
        });
    }
    const settings = (0, settings_1.getSettings)().tautulli;
    if (!settings.hostname || !settings.port || !settings.apiKey) {
        return next({
            status: 404,
            message: 'Tautulli API not configured.',
        });
    }
    try {
        const user = await (0, datasource_1.getRepository)(User_1.User).findOneOrFail({
            where: { id: Number(req.params.id) },
            select: { id: true, plexId: true },
        });
        const tautulli = new tautulli_1.default(settings);
        const watchStats = await tautulli.getUserWatchStats(user);
        const watchHistory = await tautulli.getUserWatchHistory(user);
        const recentlyWatched = (0, lodash_1.sortBy)(await (0, datasource_1.getRepository)(Media_1.default).find({
            where: [
                {
                    mediaType: media_1.MediaType.MOVIE,
                    ratingKey: (0, typeorm_1.In)(watchHistory
                        .filter((record) => record.media_type === 'movie')
                        .map((record) => record.rating_key)),
                },
                {
                    mediaType: media_1.MediaType.MOVIE,
                    ratingKey4k: (0, typeorm_1.In)(watchHistory
                        .filter((record) => record.media_type === 'movie')
                        .map((record) => record.rating_key)),
                },
                {
                    mediaType: media_1.MediaType.TV,
                    ratingKey: (0, typeorm_1.In)(watchHistory
                        .filter((record) => record.media_type === 'episode')
                        .map((record) => record.grandparent_rating_key)),
                },
                {
                    mediaType: media_1.MediaType.TV,
                    ratingKey4k: (0, typeorm_1.In)(watchHistory
                        .filter((record) => record.media_type === 'episode')
                        .map((record) => record.grandparent_rating_key)),
                },
            ],
        }), [
            (media) => (0, lodash_1.findIndex)(watchHistory, (record) => (!!media.ratingKey &&
                parseInt(media.ratingKey) ===
                    (record.media_type === 'movie'
                        ? record.rating_key
                        : record.grandparent_rating_key)) ||
                (!!media.ratingKey4k &&
                    parseInt(media.ratingKey4k) ===
                        (record.media_type === 'movie'
                            ? record.rating_key
                            : record.grandparent_rating_key))),
        ]);
        return res.status(200).json({
            recentlyWatched,
            playCount: watchStats.total_plays,
        });
    }
    catch (e) {
        logger_1.default.error('Something went wrong fetching user watch data', {
            label: 'API',
            errorMessage: e.message,
            userId: req.params.id,
        });
        next({
            status: 500,
            message: 'Failed to fetch user watch data.',
        });
    }
});
router.get('/:id/watchlist', async (req, res, next) => {
    if (Number(req.params.id) !== req.user?.id &&
        !req.user?.hasPermission([permissions_1.Permission.MANAGE_REQUESTS, permissions_1.Permission.WATCHLIST_VIEW], {
            type: 'or',
        })) {
        return next({
            status: 403,
            message: "You do not have permission to view this user's Plex Watchlist.",
        });
    }
    const itemsPerPage = 20;
    const page = Number(req.query.page) ?? 1;
    const offset = (page - 1) * itemsPerPage;
    const user = await (0, datasource_1.getRepository)(User_1.User).findOneOrFail({
        where: { id: Number(req.params.id) },
        select: { id: true, plexToken: true },
    });
    if (!user?.plexToken) {
        // We will just return an empty array if the user has no Plex token
        return res.json({
            page: 1,
            totalPages: 1,
            totalResults: 0,
            results: [],
        });
    }
    const plexTV = new plextv_1.default(user.plexToken);
    const watchlist = await plexTV.getWatchlist({ offset });
    return res.json({
        page,
        totalPages: Math.ceil(watchlist.totalSize / itemsPerPage),
        totalResults: watchlist.totalSize,
        results: watchlist.items.map((item) => ({
            ratingKey: item.ratingKey,
            title: item.title,
            mediaType: item.type === 'show' ? 'tv' : 'movie',
            tmdbId: item.tmdbId,
        })),
    });
});
exports.default = router;
