"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const issue_1 = require("../../../constants/issue");
const media_1 = require("../../../constants/media");
const datasource_1 = require("../../../datasource");
const User_1 = require("../../../entity/User");
const email_1 = __importDefault(require("../../../lib/email"));
const settings_1 = require("../../../lib/settings");
const logger_1 = __importDefault(require("../../../logger"));
const path_1 = __importDefault(require("path"));
const __1 = require("..");
const agent_1 = require("./agent");
class EmailAgent extends agent_1.BaseAgent {
    getSettings() {
        if (this.settings) {
            return this.settings;
        }
        const settings = (0, settings_1.getSettings)();
        return settings.notifications.agents.email;
    }
    shouldSend() {
        const settings = this.getSettings();
        if (settings.enabled &&
            settings.options.emailFrom &&
            settings.options.smtpHost &&
            settings.options.smtpPort) {
            return true;
        }
        return false;
    }
    buildMessage(type, payload, recipientEmail, recipientName) {
        const { applicationUrl, applicationTitle } = (0, settings_1.getSettings)().main;
        if (type === __1.Notification.TEST_NOTIFICATION) {
            return {
                template: path_1.default.join(__dirname, '../../../templates/email/test-email'),
                message: {
                    to: recipientEmail,
                },
                locals: {
                    body: payload.message,
                    applicationUrl,
                    applicationTitle,
                    recipientName,
                    recipientEmail,
                },
            };
        }
        const mediaType = payload.media
            ? payload.media.mediaType === media_1.MediaType.MOVIE
                ? 'movie'
                : 'series'
            : undefined;
        const is4k = payload.request?.is4k;
        if (payload.request) {
            let body = '';
            switch (type) {
                case __1.Notification.MEDIA_PENDING:
                    body = `A new request for the following ${mediaType} ${is4k ? 'in 4K ' : ''}is pending approval:`;
                    break;
                case __1.Notification.MEDIA_AUTO_REQUESTED:
                    body = `A new request for the following ${mediaType} ${is4k ? 'in 4K ' : ''}was automatically submitted:`;
                    break;
                case __1.Notification.MEDIA_APPROVED:
                    body = `Your request for the following ${mediaType} ${is4k ? 'in 4K ' : ''}has been approved:`;
                    break;
                case __1.Notification.MEDIA_AUTO_APPROVED:
                    body = `A new request for the following ${mediaType} ${is4k ? 'in 4K ' : ''}has been automatically approved:`;
                    break;
                case __1.Notification.MEDIA_AVAILABLE:
                    body = `Your request for the following ${mediaType} ${is4k ? 'in 4K ' : ''}is now available:`;
                    break;
                case __1.Notification.MEDIA_DECLINED:
                    body = `Your request for the following ${mediaType} ${is4k ? 'in 4K ' : ''}was declined:`;
                    break;
                case __1.Notification.MEDIA_FAILED:
                    body = `A request for the following ${mediaType} ${is4k ? 'in 4K ' : ''}failed to be added to ${payload.media?.mediaType === media_1.MediaType.MOVIE ? 'Radarr' : 'Sonarr'}:`;
                    break;
            }
            return {
                template: path_1.default.join(__dirname, '../../../templates/email/media-request'),
                message: {
                    to: recipientEmail,
                },
                locals: {
                    event: payload.event,
                    body,
                    mediaName: payload.subject,
                    mediaExtra: payload.extra ?? [],
                    imageUrl: payload.image,
                    timestamp: new Date().toTimeString(),
                    requestedBy: payload.request.requestedBy.displayName,
                    actionUrl: applicationUrl
                        ? `${applicationUrl}/${payload.media?.mediaType}/${payload.media?.tmdbId}`
                        : undefined,
                    applicationUrl,
                    applicationTitle,
                    recipientName,
                    recipientEmail,
                },
            };
        }
        else if (payload.issue) {
            const issueType = payload.issue && payload.issue.issueType !== issue_1.IssueType.OTHER
                ? `${issue_1.IssueTypeName[payload.issue.issueType].toLowerCase()} issue`
                : 'issue';
            let body = '';
            switch (type) {
                case __1.Notification.ISSUE_CREATED:
                    body = `A new ${issueType} has been reported by ${payload.issue.createdBy.displayName} for the ${mediaType} ${payload.subject}:`;
                    break;
                case __1.Notification.ISSUE_COMMENT:
                    body = `${payload.comment?.user.displayName} commented on the ${issueType} for the ${mediaType} ${payload.subject}:`;
                    break;
                case __1.Notification.ISSUE_RESOLVED:
                    body = `The ${issueType} for the ${mediaType} ${payload.subject} was marked as resolved by ${payload.issue.modifiedBy?.displayName}!`;
                    break;
                case __1.Notification.ISSUE_REOPENED:
                    body = `The ${issueType} for the ${mediaType} ${payload.subject} was reopened by ${payload.issue.modifiedBy?.displayName}.`;
                    break;
            }
            return {
                template: path_1.default.join(__dirname, '../../../templates/email/media-issue'),
                message: {
                    to: recipientEmail,
                },
                locals: {
                    event: payload.event,
                    body,
                    issueDescription: payload.message,
                    issueComment: payload.comment?.message,
                    mediaName: payload.subject,
                    extra: payload.extra ?? [],
                    imageUrl: payload.image,
                    timestamp: new Date().toTimeString(),
                    actionUrl: applicationUrl
                        ? `${applicationUrl}/issues/${payload.issue.id}`
                        : undefined,
                    applicationUrl,
                    applicationTitle,
                    recipientName,
                    recipientEmail,
                },
            };
        }
        return undefined;
    }
    async send(type, payload) {
        if (payload.notifyUser) {
            if (!payload.notifyUser.settings ||
                // Check if user has email notifications enabled and fallback to true if undefined
                // since email should default to true
                (payload.notifyUser.settings.hasNotificationType(settings_1.NotificationAgentKey.EMAIL, type) ??
                    true)) {
                logger_1.default.debug('Sending email notification', {
                    label: 'Notifications',
                    recipient: payload.notifyUser.displayName,
                    type: __1.Notification[type],
                    subject: payload.subject,
                });
                try {
                    const email = new email_1.default(this.getSettings(), payload.notifyUser.settings?.pgpKey);
                    await email.send(this.buildMessage(type, payload, payload.notifyUser.email, payload.notifyUser.displayName));
                }
                catch (e) {
                    logger_1.default.error('Error sending email notification', {
                        label: 'Notifications',
                        recipient: payload.notifyUser.displayName,
                        type: __1.Notification[type],
                        subject: payload.subject,
                        errorMessage: e.message,
                    });
                    return false;
                }
            }
        }
        if (payload.notifyAdmin) {
            const userRepository = (0, datasource_1.getRepository)(User_1.User);
            const users = await userRepository.find();
            await Promise.all(users
                .filter((user) => (!user.settings ||
                // Check if user has email notifications enabled and fallback to true if undefined
                // since email should default to true
                (user.settings.hasNotificationType(settings_1.NotificationAgentKey.EMAIL, type) ??
                    true)) &&
                (0, __1.shouldSendAdminNotification)(type, user, payload))
                .map(async (user) => {
                logger_1.default.debug('Sending email notification', {
                    label: 'Notifications',
                    recipient: user.displayName,
                    type: __1.Notification[type],
                    subject: payload.subject,
                });
                try {
                    const email = new email_1.default(this.getSettings(), user.settings?.pgpKey);
                    await email.send(this.buildMessage(type, payload, user.email, user.displayName));
                }
                catch (e) {
                    logger_1.default.error('Error sending email notification', {
                        label: 'Notifications',
                        recipient: user.displayName,
                        type: __1.Notification[type],
                        subject: payload.subject,
                        errorMessage: e.message,
                    });
                    return false;
                }
            }));
        }
        return true;
    }
}
exports.default = EmailAgent;
