import { Strategy } from "remix-auth/strategy";
export declare class FormStrategy<User> extends Strategy<User, FormStrategy.VerifyOptions> {
    name: string;
    authenticate(request: Request): Promise<User>;
}
/**
 * Whenever you try to call `authenticator.authenticate("form", request)` and
 * the request's body was already read before, the FormStrategy will throw this
 * error.
 *
 * To fix it, ensure that you either move the logic that depends on the body to
 * inside the strategy's `verify` callback, or clone the request before reading.
 *
 * @example
 * let formData = await request.clone().formData()
 * // do something with formData
 * let user = await authenticator.authenticate("form", request);
 *
 * @example
 * authenticator.use(
 *   new FormStrategy(async ({ form }) => {
 *     // do something with form here
 *   }),
 *   "login",
 * );
 */
export declare class BodyUsedError extends Error {
    name: string;
    constructor();
}
export declare namespace FormStrategy {
    interface VerifyOptions {
        /**
         * A FormData object with the content of the form used to trigger the
         * authentication.
         *
         * Here you can read any input value using the FormData API.
         */
        form: FormData;
        /**
         * The request that triggered the authentication.
         */
        request: Request;
    }
}
