var _a, _b;
import { createRequestHandler } from "@react-router/express";
import express from "express";
import { createProxyMiddleware } from "http-proxy-middleware";
import WebSocket from "ws";
import { createCookieSessionStorage } from "react-router";
import crypto from "crypto";
class BackendClient {
  async isOnboarding() {
    const url = process.env.BACKEND_URL + "/api/is-onboarding";
    const response = await fetch(url, {
      method: "GET",
      headers: {
        "Content-Type": "application/json",
        "x-api-key": process.env.FRONTEND_BACKEND_API_KEY || ""
      }
    });
    if (!response.ok) {
      throw new Error(`Failed to fetch onboarding status: ${(await response.json()).error}`);
    }
    const data = await response.json();
    return data.isOnboarding;
  }
  async createAccount(username, password) {
    const url = process.env.BACKEND_URL + "/api/create-account";
    const response = await fetch(url, {
      method: "POST",
      headers: {
        "x-api-key": process.env.FRONTEND_BACKEND_API_KEY || ""
      },
      body: (() => {
        const form = new FormData();
        form.append("username", username);
        form.append("password", password);
        form.append("type", "admin");
        return form;
      })()
    });
    if (!response.ok) {
      throw new Error(`Failed to create account: ${(await response.json()).error}`);
    }
    const data = await response.json();
    return data.status;
  }
  async authenticate(username, password) {
    const url = process.env.BACKEND_URL + "/api/authenticate";
    const apiKey = process.env.FRONTEND_BACKEND_API_KEY || "";
    const response = await fetch(url, {
      method: "POST",
      headers: { "x-api-key": apiKey },
      body: (() => {
        const form = new FormData();
        form.append("username", username);
        form.append("password", password);
        form.append("type", "admin");
        return form;
      })()
    });
    if (!response.ok) {
      throw new Error(`Failed to authenticate: ${(await response.json()).error}`);
    }
    const data = await response.json();
    return data.authenticated;
  }
  async getQueue() {
    const url = process.env.BACKEND_URL + "/api?mode=queue";
    const apiKey = process.env.FRONTEND_BACKEND_API_KEY || "";
    const response = await fetch(url, { headers: { "x-api-key": apiKey } });
    if (!response.ok) {
      throw new Error(`Failed to get queue: ${(await response.json()).error}`);
    }
    const data = await response.json();
    return data.queue;
  }
  async getHistory() {
    const url = process.env.BACKEND_URL + "/api?mode=history";
    const apiKey = process.env.FRONTEND_BACKEND_API_KEY || "";
    const response = await fetch(url, { headers: { "x-api-key": apiKey } });
    if (!response.ok) {
      throw new Error(`Failed to get history: ${(await response.json()).error}`);
    }
    const data = await response.json();
    return data.history;
  }
  async addNzb(nzbFile) {
    const url = process.env.BACKEND_URL + "/api?mode=addfile&cat=uncategorized&priority=0&pp=0";
    const apiKey = process.env.FRONTEND_BACKEND_API_KEY || "";
    const response = await fetch(url, {
      method: "POST",
      headers: { "x-api-key": apiKey },
      body: (() => {
        const form = new FormData();
        form.append("nzbFile", nzbFile, nzbFile.name);
        return form;
      })()
    });
    if (!response.ok) {
      throw new Error(`Failed to add nzb file: ${(await response.json()).error}`);
    }
    const data = await response.json();
    if (!data.nzo_ids || data.nzo_ids.length != 1) {
      throw new Error(`Failed to add nzb file: unexpected response format`);
    }
    return data.nzo_ids[0];
  }
  async listWebdavDirectory(directory) {
    const url = process.env.BACKEND_URL + "/api/list-webdav-directory";
    const apiKey = process.env.FRONTEND_BACKEND_API_KEY || "";
    const response = await fetch(url, {
      method: "POST",
      headers: { "x-api-key": apiKey },
      body: (() => {
        const form = new FormData();
        form.append("directory", directory);
        return form;
      })()
    });
    if (!response.ok) {
      throw new Error(`Failed to list webdav directory: ${(await response.json()).error}`);
    }
    const data = await response.json();
    return data.items;
  }
  async getConfig(keys) {
    const url = process.env.BACKEND_URL + "/api/get-config";
    const apiKey = process.env.FRONTEND_BACKEND_API_KEY || "";
    const response = await fetch(url, {
      method: "POST",
      headers: { "x-api-key": apiKey },
      body: (() => {
        const form = new FormData();
        for (const key of keys) {
          form.append("config-keys", key);
        }
        return form;
      })()
    });
    if (!response.ok) {
      throw new Error(`Failed to get config items: ${(await response.json()).error}`);
    }
    const data = await response.json();
    return data.configItems || [];
  }
  async updateConfig(configItems) {
    const url = process.env.BACKEND_URL + "/api/update-config";
    const apiKey = process.env.FRONTEND_BACKEND_API_KEY || "";
    const response = await fetch(url, {
      method: "POST",
      headers: { "x-api-key": apiKey },
      body: (() => {
        const form = new FormData();
        for (const item of configItems) {
          form.append(item.configName, item.configValue);
        }
        return form;
      })()
    });
    if (!response.ok) {
      throw new Error(`Failed to update config items: ${(await response.json()).error}`);
    }
    const data = await response.json();
    return data.status;
  }
  async getHealthCheckQueue(pageSize) {
    let url = process.env.BACKEND_URL + "/api/get-health-check-queue";
    if (pageSize !== void 0) {
      url += `?pageSize=${pageSize}`;
    }
    const apiKey = process.env.FRONTEND_BACKEND_API_KEY || "";
    const response = await fetch(url, {
      method: "GET",
      headers: { "x-api-key": apiKey }
    });
    if (!response.ok) {
      throw new Error(`Failed to get health check queue: ${(await response.json()).error}`);
    }
    const data = await response.json();
    return data;
  }
  async getHealthCheckHistory(pageSize) {
    let url = process.env.BACKEND_URL + "/api/get-health-check-history";
    if (pageSize !== void 0) {
      url += `?pageSize=${pageSize}`;
    }
    const apiKey = process.env.FRONTEND_BACKEND_API_KEY || "";
    const response = await fetch(url, {
      method: "GET",
      headers: { "x-api-key": apiKey }
    });
    if (!response.ok) {
      throw new Error(`Failed to get health check history: ${(await response.json()).error}`);
    }
    const data = await response.json();
    return data;
  }
}
const backendClient = new BackendClient();
const oneYear = 60 * 60 * 24 * 365;
const sessionStorage = createCookieSessionStorage({
  cookie: {
    name: "__session",
    httpOnly: true,
    path: "/",
    sameSite: "strict",
    secrets: [((_a = process == null ? void 0 : process.env) == null ? void 0 : _a.SESSION_KEY) || crypto.randomBytes(64).toString("hex")],
    secure: ["true", "yes"].includes(((_b = process == null ? void 0 : process.env) == null ? void 0 : _b.SECURE_COOKIES) || ""),
    maxAge: oneYear
  }
});
async function authenticate(request) {
  var _a2, _b2;
  const formData = await request.formData();
  const username = (_a2 = formData.get("username")) == null ? void 0 : _a2.toString();
  const password = (_b2 = formData.get("password")) == null ? void 0 : _b2.toString();
  if (!username || !password) throw new Error("username and password required");
  if (await backendClient.authenticate(username, password)) return { username };
  throw new Error("Invalid credentials");
}
async function isAuthenticated(cookieHeader) {
  const session = await sessionStorage.getSession(cookieHeader);
  const user = session.get("user");
  return !!user;
}
function initializeWebsocketServer$1(wss) {
  const websockets = /* @__PURE__ */ new Map();
  const subscriptions = /* @__PURE__ */ new Map();
  const lastMessage = /* @__PURE__ */ new Map();
  initializeWebsocketClient(subscriptions, lastMessage);
  wss.on("connection", async (ws, request) => {
    const cookieHeader = request.headers.cookie;
    if (cookieHeader) {
      try {
        const session = await sessionStorage.getSession(cookieHeader);
        const user = session.get("user");
        if (!user) {
          ws.close(1008, "Unauthorized");
          return;
        }
        ws.onmessage = (event) => {
          try {
            var topics = JSON.parse(event.data.toString());
            websockets.set(ws, topics);
            for (const topic in topics) {
              var topicSubscriptions = subscriptions.get(topic);
              if (topicSubscriptions) topicSubscriptions.add(ws);
              else subscriptions.set(topic, /* @__PURE__ */ new Set([ws]));
              if (topics[topic] === "state") {
                var messageToSend = lastMessage.get(topic);
                if (messageToSend) ws.send(messageToSend);
              }
            }
          } catch {
            ws.close(1003, "Could not process topic subscription. If recently updated, try refreshing the page.");
          }
        };
        ws.onclose = () => {
          var topics = websockets.get(ws);
          if (topics) {
            websockets.delete(ws);
            for (const topic in topics) {
              var topicSubscriptions = subscriptions.get(topic);
              if (topicSubscriptions) topicSubscriptions.delete(ws);
            }
          }
        };
      } catch (error) {
        console.error("Error authenticating websocket session:", error);
        ws.close(1011, "Internal server error");
        return;
      }
    } else {
      ws.close(1008, "Unauthorized");
      return;
    }
  });
}
function initializeWebsocketClient(subscriptions, lastMessage) {
  let reconnectRetryDelay = 1e3;
  let reconnectTimeout = null;
  const url = getBackendWebsocketUrl();
  function connect() {
    const socket = new WebSocket(url);
    socket.onopen = () => {
      console.info("WebSocket connected");
      if (reconnectTimeout) {
        clearTimeout(reconnectTimeout);
        reconnectTimeout = null;
      }
      socket.send(Buffer.from(process.env.FRONTEND_BACKEND_API_KEY, "utf-8"), { binary: false });
    };
    socket.onmessage = (event) => {
      var rawMessage = event.data.toString();
      var topicMessage = JSON.parse(rawMessage);
      var [topic, message] = [topicMessage.Topic, topicMessage.Message];
      if (!topic || !message) return;
      lastMessage.set(topic, rawMessage);
      var subscribed = subscriptions.get(topic) || [];
      subscribed.forEach((client) => {
        if (client.readyState === client.OPEN) {
          client.send(rawMessage);
        }
      });
    };
    socket.onerror = (event) => {
      console.error("WebSocket error:", event.message);
    };
    socket.onclose = (event) => {
      console.info(`WebSocket closed (code: ${event.code}, reason: ${event.reason})`);
      scheduleReconnect();
    };
  }
  function scheduleReconnect() {
    if (reconnectTimeout) clearTimeout(reconnectTimeout);
    reconnectTimeout = setTimeout(() => {
      console.info(`WebSocket reconnecting...`);
      connect();
    }, reconnectRetryDelay);
  }
  connect();
}
function getBackendWebsocketUrl() {
  const host = process.env.BACKEND_URL;
  return `${host.replace(/\/$/, "")}/ws`.replace(/^http/, "ws");
}
const websocketServer = {
  initialize: initializeWebsocketServer$1
};
const app = express();
const initializeWebsocketServer = websocketServer.initialize;
const forwardToBackend = createProxyMiddleware({
  target: process.env.BACKEND_URL,
  changeOrigin: true
});
const setApiKeyForAuthenticatedRequests = async (req) => {
  if (!req.path.startsWith("/api")) return;
  var apikey = req.query.apikey || req.query.apiKey || req.headers["x-api-key"];
  var hasApiKey = apikey && typeof apikey === "string";
  if (hasApiKey) return;
  const authenticated = await isAuthenticated(req.headers.cookie);
  if (!authenticated) return;
  req.headers["x-api-key"] = process.env.FRONTEND_BACKEND_API_KEY || "";
};
app.use(async (req, res, next) => {
  if (req.method.toUpperCase() === "PROPFIND" || req.method.toUpperCase() === "OPTIONS" || req.path.startsWith("/api") || req.path.startsWith("/view") || req.path.startsWith("/.ids") || req.path.startsWith("/nzbs") || req.path.startsWith("/content") || req.path.startsWith("/completed-symlinks")) {
    await setApiKeyForAuthenticatedRequests(req);
    return forwardToBackend(req, res, next);
  }
  next();
});
app.use(
  createRequestHandler({
    build: () => import("./server-build-DpQM1QUm.js"),
    getLoadContext() {
      return {
        VALUE_FROM_EXPRESS: "Hello from Express"
      };
    }
  })
);
export {
  authenticate as a,
  backendClient as b,
  app as c,
  initializeWebsocketServer as i,
  sessionStorage as s
};
