"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const imageproxy_1 = __importDefault(require("../lib/imageproxy"));
const logger_1 = __importDefault(require("../logger"));
const express_1 = require("express");
const router = (0, express_1.Router)();
const tmdbImageProxy = new imageproxy_1.default('tmdb', 'https://image.tmdb.org', {
    rateLimitOptions: {
        maxRequests: 20,
        maxRPS: 50,
    },
});
const tvdbImageProxy = new imageproxy_1.default('tvdb', 'https://artworks.thetvdb.com', {
    rateLimitOptions: {
        maxRequests: 20,
        maxRPS: 50,
    },
});
router.get('/:type/*', async (req, res) => {
    const imagePath = req.path.replace(/^\/\w+/, '');
    try {
        let imageData;
        if (req.params.type === 'tmdb') {
            imageData = await tmdbImageProxy.getImage(imagePath);
        }
        else if (req.params.type === 'tvdb') {
            imageData = await tvdbImageProxy.getImage(imagePath);
        }
        else {
            logger_1.default.error('Unsupported image type', {
                imagePath,
                type: req.params.type,
            });
            res.status(400).send('Unsupported image type');
            return;
        }
        res.writeHead(200, {
            'Content-Type': `image/${imageData.meta.extension}`,
            'Content-Length': imageData.imageBuffer.length,
            'Cache-Control': `public, max-age=${imageData.meta.curRevalidate}`,
            'OS-Cache-Key': imageData.meta.cacheKey,
            'OS-Cache-Status': imageData.meta.cacheMiss ? 'MISS' : 'HIT',
        });
        res.end(imageData.imageBuffer);
    }
    catch (e) {
        logger_1.default.error('Failed to proxy image', {
            imagePath,
            errorMessage: e.message,
        });
        res.status(500).send();
    }
});
exports.default = router;
