"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createIdTokenSchema = exports.validateUserClaims = exports.tryGetUserInfoKey = exports.getOpenIdUserInfo = exports.fetchOpenIdTokenData = exports.getOpenIdRedirectUrl = exports.getOpenIdConfiguration = void 0;
const yup = __importStar(require("yup"));
/** Fetch the issuer configuration from the OpenID Connect Discovery endpoint */
async function getOpenIdConfiguration(domain) {
    // remove trailing slash from url if it exists and add /.well-known/openid-configuration path
    const wellKnownUrl = new URL(domain.replace(/\/$/, '') + '/.well-known/openid-configuration').toString();
    const wellKnownInfo = await fetch(wellKnownUrl, {
        headers: {
            'Content-Type': 'application/json',
        },
    }).then((r) => r.json());
    return wellKnownInfo;
}
exports.getOpenIdConfiguration = getOpenIdConfiguration;
function getOpenIdCallbackUrl(req, provider) {
    const callbackUrl = new URL(`/login`, `${req.protocol}://${req.headers.host}`);
    callbackUrl.searchParams.set('provider', provider.slug);
    callbackUrl.searchParams.set('callback', 'true');
    return callbackUrl.toString();
}
/** Generate authentication request url */
async function getOpenIdRedirectUrl(req, provider, state) {
    const wellKnownInfo = await getOpenIdConfiguration(provider.issuerUrl);
    const url = new URL(wellKnownInfo.authorization_endpoint);
    url.searchParams.set('response_type', 'code');
    url.searchParams.set('client_id', provider.clientId);
    url.searchParams.set('redirect_uri', getOpenIdCallbackUrl(req, provider));
    url.searchParams.set('scope', provider.scopes ?? 'openid profile email');
    url.searchParams.set('state', state);
    return url.toString();
}
exports.getOpenIdRedirectUrl = getOpenIdRedirectUrl;
/** Exchange authorization code for token data */
async function fetchOpenIdTokenData(req, provider, wellKnownInfo, code) {
    const formData = new URLSearchParams();
    formData.append('client_secret', provider.clientSecret);
    formData.append('grant_type', 'authorization_code');
    formData.append('redirect_uri', getOpenIdCallbackUrl(req, provider));
    formData.append('client_id', provider.clientId);
    formData.append('code', code);
    return await fetch(wellKnownInfo.token_endpoint, {
        method: 'POST',
        body: formData,
    }).then((r) => r.json());
}
exports.fetchOpenIdTokenData = fetchOpenIdTokenData;
async function getOpenIdUserInfo(wellKnownInfo, authToken) {
    return fetch(wellKnownInfo.userinfo_endpoint, {
        headers: {
            Authorization: `Bearer ${authToken}`,
            Accept: 'application/json',
        },
    }).then((r) => r.json());
}
exports.getOpenIdUserInfo = getOpenIdUserInfo;
class OidcAuthorizationError extends Error {
}
class OidcMissingKeyError extends OidcAuthorizationError {
    constructor(userInfo, key) {
        super(`Key ${key} was missing on OIDC userinfo but was expected.`);
        this.userInfo = userInfo;
        this.key = key;
    }
}
function tryGetUserInfoKey(userInfo, key, expectedType) {
    if (!Object.hasOwn(userInfo, key) || typeof userInfo[key] !== expectedType) {
        throw new OidcMissingKeyError(userInfo, key);
    }
    return userInfo[key];
}
exports.tryGetUserInfoKey = tryGetUserInfoKey;
function validateUserClaims(userInfo, requiredClaims) {
    requiredClaims.some((claim) => {
        const value = tryGetUserInfoKey(userInfo, claim, 'boolean');
        if (!value)
            throw new OidcAuthorizationError('User was missing a required claim.');
    });
}
exports.validateUserClaims = validateUserClaims;
/** Generates a schema to validate ID token JWT and userinfo claims */
const createIdTokenSchema = ({ oidcDomain, oidcClientId, requiredClaims, }) => {
    return yup.object().shape({
        iss: yup
            .string()
            .oneOf([oidcDomain, `${oidcDomain}/`], `The token iss value doesn't match the oidc_DOMAIN (${oidcDomain})`)
            .required("The token didn't come with an iss value."),
        aud: yup.lazy((val) => {
            // single audience
            if (typeof val === 'string')
                return yup
                    .string()
                    .oneOf([oidcClientId], `The token aud value doesn't match the oidc_CLIENT_ID (${oidcClientId})`)
                    .required("The token didn't come with an aud value.");
            // several audiences
            if (typeof val === 'object' && Array.isArray(val))
                return yup
                    .array()
                    .of(yup.string())
                    .test('contains-client-id', `The token aud value doesn't contain the oidc_CLIENT_ID (${oidcClientId})`, (value) => !!(value && value.includes(oidcClientId)));
            // invalid type
            return yup
                .mixed()
                .typeError('The token aud value is not a string or array.');
        }),
        exp: yup
            .number()
            .required()
            .test('is_before_date', 'Token exp value is before current time.', (value) => {
            if (!value)
                return false;
            if (value < Math.ceil(Date.now() / 1000))
                return false;
            return true;
        }),
        iat: yup
            .number()
            .required()
            .test('is_before_one_day', 'Token was issued before one day ago and is now invalid.', (value) => {
            if (!value)
                return false;
            const date = new Date();
            date.setDate(date.getDate() - 1);
            if (value < Math.ceil(Number(date) / 1000))
                return false;
            return true;
        }),
        // TODO: only require this for new user login
        email: yup.string().required(),
        // ensure all required claims are present and are booleans
        ...requiredClaims.reduce((a, v) => ({ ...a, [v]: yup.boolean().required() }), {}),
    });
};
exports.createIdTokenSchema = createIdTokenSchema;
