"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.jellyfinRecentScanner = exports.jellyfinFullScanner = void 0;
const jellyfin_1 = __importDefault(require("../../../api/jellyfin"));
const themoviedb_1 = __importDefault(require("../../../api/themoviedb"));
const media_1 = require("../../../constants/media");
const server_1 = require("../../../constants/server");
const datasource_1 = require("../../../datasource");
const Media_1 = __importDefault(require("../../../entity/Media"));
const Season_1 = __importDefault(require("../../../entity/Season"));
const User_1 = require("../../../entity/User");
const settings_1 = require("../../../lib/settings");
const logger_1 = __importDefault(require("../../../logger"));
const asyncLock_1 = __importDefault(require("../../../utils/asyncLock"));
const getHostname_1 = require("../../../utils/getHostname");
const crypto_1 = require("crypto");
const lodash_1 = require("lodash");
const BUNDLE_SIZE = 20;
const UPDATE_RATE = 4 * 1000;
class JellyfinScanner {
    constructor({ isRecentOnly } = {}) {
        this.items = [];
        this.progress = 0;
        this.running = false;
        this.isRecentOnly = false;
        this.enable4kMovie = false;
        this.enable4kShow = false;
        this.asyncLock = new asyncLock_1.default();
        this.tmdb = new themoviedb_1.default();
        this.isRecentOnly = isRecentOnly ?? false;
    }
    async getExisting(tmdbId, mediaType) {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        const existing = await mediaRepository.findOne({
            where: { tmdbId: tmdbId, mediaType },
        });
        return existing;
    }
    async processMovie(jellyfinitem) {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        try {
            const metadata = await this.jfClient.getItemData(jellyfinitem.Id);
            const newMedia = new Media_1.default();
            if (!metadata?.Id) {
                logger_1.default.debug('No Id metadata for this title. Skipping', {
                    label: 'Jellyfin Sync',
                    jellyfinItemId: jellyfinitem.Id,
                });
                return;
            }
            newMedia.tmdbId = Number(metadata.ProviderIds.Tmdb ?? null);
            newMedia.imdbId = metadata.ProviderIds.Imdb;
            if (newMedia.imdbId && !isNaN(newMedia.tmdbId)) {
                const tmdbMovie = await this.tmdb.getMediaByImdbId({
                    imdbId: newMedia.imdbId,
                });
                newMedia.tmdbId = tmdbMovie.id;
            }
            if (!newMedia.tmdbId) {
                throw new Error('Unable to find TMDb ID');
            }
            const has4k = metadata.MediaSources?.some((MediaSource) => {
                return MediaSource.MediaStreams.filter((MediaStream) => MediaStream.Type === 'Video').some((MediaStream) => {
                    return (MediaStream.Width ?? 0) > 2000;
                });
            });
            const hasOtherResolution = metadata.MediaSources?.some((MediaSource) => {
                return MediaSource.MediaStreams.filter((MediaStream) => MediaStream.Type === 'Video').some((MediaStream) => {
                    return (MediaStream.Width ?? 0) <= 2000;
                });
            });
            await this.asyncLock.dispatch(newMedia.tmdbId, async () => {
                const existing = await this.getExisting(newMedia.tmdbId, media_1.MediaType.MOVIE);
                if (existing) {
                    let changedExisting = false;
                    if ((hasOtherResolution || (!this.enable4kMovie && has4k)) &&
                        existing.status !== media_1.MediaStatus.AVAILABLE) {
                        existing.status = media_1.MediaStatus.AVAILABLE;
                        existing.mediaAddedAt = new Date(metadata.DateCreated ?? '');
                        changedExisting = true;
                    }
                    if (has4k &&
                        this.enable4kMovie &&
                        existing.status4k !== media_1.MediaStatus.AVAILABLE) {
                        existing.status4k = media_1.MediaStatus.AVAILABLE;
                        changedExisting = true;
                    }
                    if (!existing.mediaAddedAt && !changedExisting) {
                        existing.mediaAddedAt = new Date(metadata.DateCreated ?? '');
                        changedExisting = true;
                    }
                    if ((hasOtherResolution || (has4k && !this.enable4kMovie)) &&
                        existing.jellyfinMediaId !== metadata.Id) {
                        existing.jellyfinMediaId = metadata.Id;
                        changedExisting = true;
                    }
                    if (has4k &&
                        this.enable4kMovie &&
                        existing.jellyfinMediaId4k !== metadata.Id) {
                        existing.jellyfinMediaId4k = metadata.Id;
                        changedExisting = true;
                    }
                    if (changedExisting) {
                        await mediaRepository.save(existing);
                        this.log(`Request for ${metadata.Name} exists. New media types set to AVAILABLE`, 'info');
                    }
                    else {
                        this.log(`Title already exists and no new media types found ${metadata.Name}`);
                    }
                }
                else {
                    newMedia.status =
                        hasOtherResolution || (!this.enable4kMovie && has4k)
                            ? media_1.MediaStatus.AVAILABLE
                            : media_1.MediaStatus.UNKNOWN;
                    newMedia.status4k =
                        has4k && this.enable4kMovie
                            ? media_1.MediaStatus.AVAILABLE
                            : media_1.MediaStatus.UNKNOWN;
                    newMedia.mediaType = media_1.MediaType.MOVIE;
                    newMedia.mediaAddedAt = new Date(metadata.DateCreated ?? '');
                    newMedia.jellyfinMediaId =
                        hasOtherResolution || (!this.enable4kMovie && has4k)
                            ? metadata.Id
                            : null;
                    newMedia.jellyfinMediaId4k =
                        has4k && this.enable4kMovie ? metadata.Id : null;
                    await mediaRepository.save(newMedia);
                    this.log(`Saved ${metadata.Name}`);
                }
            });
        }
        catch (e) {
            this.log(`Failed to process Jellyfin item, id: ${jellyfinitem.Id}`, 'error', {
                errorMessage: e.message,
                jellyfinitem,
            });
        }
    }
    async processShow(jellyfinitem) {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        let tvShow = null;
        try {
            const Id = jellyfinitem.SeriesId ?? jellyfinitem.SeasonId ?? jellyfinitem.Id;
            const metadata = await this.jfClient.getItemData(Id);
            if (!metadata?.Id) {
                logger_1.default.debug('No Id metadata for this title. Skipping', {
                    label: 'Jellyfin Sync',
                    jellyfinItemId: jellyfinitem.Id,
                });
                return;
            }
            if (metadata.ProviderIds.Tmdb) {
                try {
                    tvShow = await this.tmdb.getTvShow({
                        tvId: Number(metadata.ProviderIds.Tmdb),
                    });
                }
                catch {
                    this.log('Unable to find TMDb ID for this title.', 'debug', {
                        jellyfinitem,
                    });
                }
            }
            if (!tvShow && metadata.ProviderIds.Tvdb) {
                try {
                    tvShow = await this.tmdb.getShowByTvdbId({
                        tvdbId: Number(metadata.ProviderIds.Tvdb),
                    });
                }
                catch {
                    this.log('Unable to find TVDb ID for this title.', 'debug', {
                        jellyfinitem,
                    });
                }
            }
            if (tvShow) {
                await this.asyncLock.dispatch(tvShow.id, async () => {
                    if (!tvShow) {
                        // this will never execute, but typescript thinks somebody could reset tvShow from
                        // outer scope back to null before this async gets called
                        return;
                    }
                    // Lets get the available seasons from Jellyfin
                    const seasons = tvShow.seasons;
                    const media = await this.getExisting(tvShow.id, media_1.MediaType.TV);
                    const newSeasons = [];
                    const currentStandardSeasonAvailable = (media?.seasons.filter((season) => season.status === media_1.MediaStatus.AVAILABLE) ?? []).length;
                    const current4kSeasonAvailable = (media?.seasons.filter((season) => season.status4k === media_1.MediaStatus.AVAILABLE) ?? []).length;
                    for (const season of seasons) {
                        const JellyfinSeasons = await this.jfClient.getSeasons(Id);
                        const matchedJellyfinSeason = JellyfinSeasons.find((md) => Number(md.IndexNumber) === season.season_number);
                        const existingSeason = media?.seasons.find((es) => es.seasonNumber === season.season_number);
                        // Check if we found the matching season and it has all the available episodes
                        if (matchedJellyfinSeason) {
                            // If we have a matched Jellyfin season, get its children metadata so we can check details
                            const episodes = await this.jfClient.getEpisodes(Id, matchedJellyfinSeason.Id);
                            //Get count of episodes that are HD and 4K
                            let totalStandard = 0;
                            let total4k = 0;
                            //use for loop to make sure this loop _completes_ in full
                            //before the next section
                            for (const episode of episodes) {
                                let episodeCount = 1;
                                // count number of combined episodes
                                if (episode.IndexNumber !== undefined &&
                                    episode.IndexNumberEnd !== undefined) {
                                    episodeCount =
                                        episode.IndexNumberEnd - episode.IndexNumber + 1;
                                }
                                if (!this.enable4kShow) {
                                    totalStandard += episodeCount;
                                }
                                else {
                                    const ExtendedEpisodeData = await this.jfClient.getItemData(episode.Id);
                                    ExtendedEpisodeData?.MediaSources?.some((MediaSource) => {
                                        return MediaSource.MediaStreams.some((MediaStream) => {
                                            if (MediaStream.Type === 'Video') {
                                                if ((MediaStream.Width ?? 0) >= 2000) {
                                                    total4k += episodeCount;
                                                }
                                                else {
                                                    totalStandard += episodeCount;
                                                }
                                            }
                                        });
                                    });
                                }
                            }
                            if (media &&
                                (totalStandard > 0 || (total4k > 0 && !this.enable4kShow)) &&
                                media.jellyfinMediaId !== Id) {
                                media.jellyfinMediaId = Id;
                            }
                            if (media &&
                                total4k > 0 &&
                                this.enable4kShow &&
                                media.jellyfinMediaId4k !== Id) {
                                media.jellyfinMediaId4k = Id;
                            }
                            if (existingSeason) {
                                // These ternary statements look super confusing, but they are simply
                                // setting the status to AVAILABLE if all of a type is there, partially if some,
                                // and then not modifying the status if there are 0 items
                                existingSeason.status =
                                    totalStandard >= season.episode_count ||
                                        existingSeason.status === media_1.MediaStatus.AVAILABLE
                                        ? media_1.MediaStatus.AVAILABLE
                                        : totalStandard > 0
                                            ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                            : existingSeason.status;
                                existingSeason.status4k =
                                    (this.enable4kShow && total4k >= season.episode_count) ||
                                        existingSeason.status4k === media_1.MediaStatus.AVAILABLE
                                        ? media_1.MediaStatus.AVAILABLE
                                        : this.enable4kShow && total4k > 0
                                            ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                            : existingSeason.status4k;
                            }
                            else {
                                newSeasons.push(new Season_1.default({
                                    seasonNumber: season.season_number,
                                    // This ternary is the same as the ones above, but it just falls back to "UNKNOWN"
                                    // if we dont have any items for the season
                                    status: totalStandard >= season.episode_count
                                        ? media_1.MediaStatus.AVAILABLE
                                        : totalStandard > 0
                                            ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                            : media_1.MediaStatus.UNKNOWN,
                                    status4k: this.enable4kShow && total4k >= season.episode_count
                                        ? media_1.MediaStatus.AVAILABLE
                                        : this.enable4kShow && total4k > 0
                                            ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                            : media_1.MediaStatus.UNKNOWN,
                                }));
                            }
                        }
                    }
                    // Remove extras season. We dont count it for determining availability
                    const filteredSeasons = tvShow.seasons.filter((season) => season.season_number !== 0);
                    const isAllStandardSeasons = newSeasons.filter((season) => season.status === media_1.MediaStatus.AVAILABLE).length +
                        (media?.seasons.filter((season) => season.status === media_1.MediaStatus.AVAILABLE).length ?? 0) >=
                        filteredSeasons.length;
                    const isAll4kSeasons = newSeasons.filter((season) => season.status4k === media_1.MediaStatus.AVAILABLE).length +
                        (media?.seasons.filter((season) => season.status4k === media_1.MediaStatus.AVAILABLE).length ?? 0) >=
                        filteredSeasons.length;
                    if (media) {
                        // Update existing
                        media.seasons = [...media.seasons, ...newSeasons];
                        const newStandardSeasonAvailable = (media.seasons.filter((season) => season.status === media_1.MediaStatus.AVAILABLE) ?? []).length;
                        const new4kSeasonAvailable = (media.seasons.filter((season) => season.status4k === media_1.MediaStatus.AVAILABLE) ?? []).length;
                        // If at least one new season has become available, update
                        // the lastSeasonChange field so we can trigger notifications
                        if (newStandardSeasonAvailable > currentStandardSeasonAvailable) {
                            this.log(`Detected ${newStandardSeasonAvailable - currentStandardSeasonAvailable} new standard season(s) for ${tvShow.name}`, 'debug');
                            media.lastSeasonChange = new Date();
                            media.mediaAddedAt = new Date(metadata.DateCreated ?? '');
                        }
                        if (new4kSeasonAvailable > current4kSeasonAvailable) {
                            this.log(`Detected ${new4kSeasonAvailable - current4kSeasonAvailable} new 4K season(s) for ${tvShow.name}`, 'debug');
                            media.lastSeasonChange = new Date();
                        }
                        if (!media.mediaAddedAt) {
                            media.mediaAddedAt = new Date(metadata.DateCreated ?? '');
                        }
                        // If the show is already available, and there are no new seasons, dont adjust
                        // the status
                        const shouldStayAvailable = media.status === media_1.MediaStatus.AVAILABLE &&
                            newSeasons.filter((season) => season.status !== media_1.MediaStatus.UNKNOWN).length === 0;
                        const shouldStayAvailable4k = media.status4k === media_1.MediaStatus.AVAILABLE &&
                            newSeasons.filter((season) => season.status4k !== media_1.MediaStatus.UNKNOWN).length === 0;
                        media.status =
                            isAllStandardSeasons || shouldStayAvailable
                                ? media_1.MediaStatus.AVAILABLE
                                : media.seasons.some((season) => season.status !== media_1.MediaStatus.UNKNOWN)
                                    ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                    : media_1.MediaStatus.UNKNOWN;
                        media.status4k =
                            (isAll4kSeasons || shouldStayAvailable4k) && this.enable4kShow
                                ? media_1.MediaStatus.AVAILABLE
                                : this.enable4kShow &&
                                    media.seasons.some((season) => season.status4k !== media_1.MediaStatus.UNKNOWN)
                                    ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                    : media_1.MediaStatus.UNKNOWN;
                        await mediaRepository.save(media);
                        this.log(`Updating existing title: ${tvShow.name}`);
                    }
                    else {
                        const newMedia = new Media_1.default({
                            mediaType: media_1.MediaType.TV,
                            seasons: newSeasons,
                            tmdbId: tvShow.id,
                            tvdbId: tvShow.external_ids.tvdb_id,
                            mediaAddedAt: new Date(metadata.DateCreated ?? ''),
                            jellyfinMediaId: isAllStandardSeasons ? Id : null,
                            jellyfinMediaId4k: isAll4kSeasons && this.enable4kShow ? Id : null,
                            status: isAllStandardSeasons
                                ? media_1.MediaStatus.AVAILABLE
                                : newSeasons.some((season) => season.status !== media_1.MediaStatus.UNKNOWN)
                                    ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                    : media_1.MediaStatus.UNKNOWN,
                            status4k: isAll4kSeasons && this.enable4kShow
                                ? media_1.MediaStatus.AVAILABLE
                                : this.enable4kShow &&
                                    newSeasons.some((season) => season.status4k !== media_1.MediaStatus.UNKNOWN)
                                    ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                    : media_1.MediaStatus.UNKNOWN,
                        });
                        await mediaRepository.save(newMedia);
                        this.log(`Saved ${tvShow.name}`);
                    }
                });
            }
            else {
                this.log(`No information found for the show: ${metadata.Name}`, 'debug', {
                    jellyfinitem,
                });
            }
        }
        catch (e) {
            this.log(`Failed to process Jellyfin item. Id: ${jellyfinitem.SeriesId ?? jellyfinitem.SeasonId ?? jellyfinitem.Id}`, 'error', {
                errorMessage: e.message,
                jellyfinitem,
            });
        }
    }
    async processItems(slicedItems) {
        await Promise.all(slicedItems.map(async (item) => {
            if (item.Type === 'Movie') {
                await this.processMovie(item);
            }
            else if (item.Type === 'Series') {
                await this.processShow(item);
            }
        }));
    }
    async loop({ start = 0, end = BUNDLE_SIZE, sessionId, } = {}) {
        const slicedItems = this.items.slice(start, end);
        if (!this.running) {
            throw new Error('Sync was aborted.');
        }
        if (this.sessionId !== sessionId) {
            throw new Error('New session was started. Old session aborted.');
        }
        if (start < this.items.length) {
            this.progress = start;
            await this.processItems(slicedItems);
            await new Promise((resolve, reject) => setTimeout(() => {
                this.loop({
                    start: start + BUNDLE_SIZE,
                    end: end + BUNDLE_SIZE,
                    sessionId,
                })
                    .then(() => resolve())
                    .catch((e) => reject(new Error(e.message)));
            }, UPDATE_RATE));
        }
    }
    log(message, level = 'debug', optional) {
        logger_1.default[level](message, { label: 'Jellyfin Sync', ...optional });
    }
    async run() {
        const settings = (0, settings_1.getSettings)();
        if (settings.main.mediaServerType != server_1.MediaServerType.JELLYFIN &&
            settings.main.mediaServerType != server_1.MediaServerType.EMBY) {
            return;
        }
        const sessionId = (0, crypto_1.randomUUID)();
        this.sessionId = sessionId;
        logger_1.default.info('Jellyfin Sync Starting', {
            sessionId,
            label: 'Jellyfin Sync',
        });
        try {
            this.running = true;
            const userRepository = (0, datasource_1.getRepository)(User_1.User);
            const admin = await userRepository.findOne({
                where: { id: 1 },
                select: ['id', 'jellyfinUserId', 'jellyfinDeviceId'],
                order: { id: 'ASC' },
            });
            if (!admin) {
                return this.log('No admin configured. Jellyfin sync skipped.', 'warn');
            }
            this.jfClient = new jellyfin_1.default((0, getHostname_1.getHostname)(), settings.jellyfin.apiKey, admin.jellyfinDeviceId);
            this.jfClient.setUserId(admin.jellyfinUserId ?? '');
            this.libraries = settings.jellyfin.libraries.filter((library) => library.enabled);
            this.enable4kMovie = settings.radarr.some((radarr) => radarr.is4k);
            if (this.enable4kMovie) {
                this.log('At least one 4K Radarr server was detected. 4K movie detection is now enabled', 'info');
            }
            this.enable4kShow = settings.sonarr.some((sonarr) => sonarr.is4k);
            if (this.enable4kShow) {
                this.log('At least one 4K Sonarr server was detected. 4K series detection is now enabled', 'info');
            }
            if (this.isRecentOnly) {
                for (const library of this.libraries) {
                    this.currentLibrary = library;
                    this.log(`Beginning to process recently added for library: ${library.name}`, 'info');
                    const libraryItems = await this.jfClient.getRecentlyAdded(library.id);
                    // Bundle items up by rating keys
                    this.items = (0, lodash_1.uniqWith)(libraryItems, (mediaA, mediaB) => {
                        if (mediaA.SeriesId && mediaB.SeriesId) {
                            return mediaA.SeriesId === mediaB.SeriesId;
                        }
                        if (mediaA.SeasonId && mediaB.SeasonId) {
                            return mediaA.SeasonId === mediaB.SeasonId;
                        }
                        return mediaA.Id === mediaB.Id;
                    });
                    await this.loop({ sessionId });
                }
            }
            else {
                for (const library of this.libraries) {
                    this.currentLibrary = library;
                    this.log(`Beginning to process library: ${library.name}`, 'info');
                    this.items = await this.jfClient.getLibraryContents(library.id);
                    await this.loop({ sessionId });
                }
            }
            this.log(this.isRecentOnly
                ? 'Recently Added Scan Complete'
                : 'Full Scan Complete', 'info');
        }
        catch (e) {
            logger_1.default.error('Sync interrupted', {
                label: 'Jellyfin Sync',
                errorMessage: e.message,
            });
        }
        finally {
            // If a new scanning session hasnt started, set running back to false
            if (this.sessionId === sessionId) {
                this.running = false;
            }
        }
    }
    status() {
        return {
            running: this.running,
            progress: this.progress,
            total: this.items.length,
            currentLibrary: this.currentLibrary,
            libraries: this.libraries,
        };
    }
    cancel() {
        this.running = false;
    }
}
exports.jellyfinFullScanner = new JellyfinScanner();
exports.jellyfinRecentScanner = new JellyfinScanner({
    isRecentOnly: true,
});
