"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const themoviedb_1 = require("../api/themoviedb");
const media_1 = require("../constants/media");
const datasource_1 = __importDefault(require("../datasource"));
const Blacklist_1 = require("../entity/Blacklist");
const Media_1 = __importDefault(require("../entity/Media"));
const settings_1 = require("../lib/settings");
const logger_1 = __importDefault(require("../logger"));
const discover_1 = require("../routes/discover");
const TMDB_API_DELAY_MS = 250;
class AbortTransaction extends Error {
}
class BlacklistedTagProcessor {
    constructor() {
        this.running = false;
        this.progress = 0;
        this.total = 0;
    }
    async run() {
        this.running = true;
        try {
            await datasource_1.default.transaction(async (em) => {
                await this.cleanBlacklist(em);
                await this.createBlacklistEntries(em);
            });
        }
        catch (err) {
            if (err instanceof AbortTransaction) {
                logger_1.default.info('Aborting job: Process Blacklisted Tags', {
                    label: 'Jobs',
                });
            }
            else {
                throw err;
            }
        }
        finally {
            this.reset();
        }
    }
    status() {
        return {
            running: this.running,
            progress: this.progress,
            total: this.total,
        };
    }
    cancel() {
        this.running = false;
        this.progress = 0;
        this.total = 0;
    }
    reset() {
        this.cancel();
    }
    async createBlacklistEntries(em) {
        const tmdb = (0, discover_1.createTmdbWithRegionLanguage)();
        const settings = (0, settings_1.getSettings)();
        const blacklistedTags = settings.main.blacklistedTags;
        const blacklistedTagsArr = blacklistedTags.split(',');
        const pageLimit = settings.main.blacklistedTagsLimit;
        if (blacklistedTags.length === 0) {
            return;
        }
        // The maximum number of queries we're expected to execute
        this.total =
            2 * blacklistedTagsArr.length * pageLimit * themoviedb_1.SortOptionsIterable.length;
        for (const type of [media_1.MediaType.MOVIE, media_1.MediaType.TV]) {
            const getDiscover = type === media_1.MediaType.MOVIE ? tmdb.getDiscoverMovies : tmdb.getDiscoverTv;
            // Iterate for each tag
            for (const tag of blacklistedTagsArr) {
                let queryMax = pageLimit * themoviedb_1.SortOptionsIterable.length;
                let fixedSortMode = false; // Set to true when the page limit allows for getting every page of tag
                for (let query = 0; query < queryMax; query++) {
                    const page = fixedSortMode
                        ? query + 1
                        : (query % pageLimit) + 1;
                    const sortBy = fixedSortMode
                        ? undefined
                        : themoviedb_1.SortOptionsIterable[query % themoviedb_1.SortOptionsIterable.length];
                    if (!this.running) {
                        throw new AbortTransaction();
                    }
                    const response = await getDiscover({
                        page,
                        sortBy,
                        keywords: tag,
                    });
                    await this.processResults(response, tag, type, em);
                    await new Promise((res) => setTimeout(res, TMDB_API_DELAY_MS));
                    this.progress++;
                    if (page === 1 && response.total_pages <= queryMax) {
                        // We will finish the tag with less queries than expected, move progress accordingly
                        this.progress += queryMax - response.total_pages;
                        fixedSortMode = true;
                        queryMax = response.total_pages;
                    }
                }
            }
        }
    }
    async processResults(response, keywordId, mediaType, em) {
        const blacklistRepository = em.getRepository(Blacklist_1.Blacklist);
        for (const entry of response.results) {
            const blacklistEntry = await blacklistRepository.findOne({
                where: { tmdbId: entry.id },
            });
            if (blacklistEntry) {
                // Don't mark manual blacklists with tags
                // If media wasn't previously blacklisted for this tag, add the tag to the media's blacklist
                if (blacklistEntry.blacklistedTags &&
                    !blacklistEntry.blacklistedTags.includes(`,${keywordId},`)) {
                    await blacklistRepository.update(blacklistEntry.id, {
                        blacklistedTags: `${blacklistEntry.blacklistedTags}${keywordId},`,
                    });
                }
            }
            else {
                // Media wasn't previously blacklisted, add it to the blacklist
                await Blacklist_1.Blacklist.addToBlacklist({
                    blacklistRequest: {
                        mediaType,
                        title: 'title' in entry ? entry.title : entry.name,
                        tmdbId: entry.id,
                        blacklistedTags: `,${keywordId},`,
                    },
                }, em);
            }
        }
    }
    async cleanBlacklist(em) {
        // Remove blacklist and media entries blacklisted by tags
        const mediaRepository = em.getRepository(Media_1.default);
        const mediaToRemove = await mediaRepository
            .createQueryBuilder('media')
            .innerJoinAndSelect(Blacklist_1.Blacklist, 'blist', 'blist.tmdbId = media.tmdbId')
            .where(`blist.blacklistedTags IS NOT NULL`)
            .getMany();
        // Batch removes so the query doesn't get too large
        for (let i = 0; i < mediaToRemove.length; i += 500) {
            await mediaRepository.remove(mediaToRemove.slice(i, i + 500)); // This also deletes the blacklist entries via cascading
        }
    }
}
const blacklistedTagsProcessor = new BlacklistedTagProcessor();
exports.default = blacklistedTagsProcessor;
