"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TimezoneOffsetTagnames = exports.defaultVideosToUTC = exports.UnsetZoneName = exports.UnsetZone = exports.UnsetZoneOffsetMinutes = void 0;
exports.isUTC = isUTC;
exports.isZoneValid = isZoneValid;
exports.normalizeZone = normalizeZone;
exports.zoneToShortOffset = zoneToShortOffset;
exports.validTzOffsetMinutes = validTzOffsetMinutes;
exports.offsetMinutesToZoneName = offsetMinutesToZoneName;
exports.extractZone = extractZone;
exports.incrementZone = incrementZone;
exports.extractTzOffsetFromTags = extractTzOffsetFromTags;
exports.extractTzOffsetFromDatestamps = extractTzOffsetFromDatestamps;
exports.extractTzOffsetFromTimeStamp = extractTzOffsetFromTimeStamp;
exports.inferLikelyOffsetMinutes = inferLikelyOffsetMinutes;
exports.extractTzOffsetFromUTCOffset = extractTzOffsetFromUTCOffset;
exports.equivalentZones = equivalentZones;
exports.getZoneName = getZoneName;
const luxon_1 = require("luxon");
const Array_1 = require("./Array");
const BinaryField_1 = require("./BinaryField");
const CapturedAtTagNames_1 = require("./CapturedAtTagNames");
const DefaultExifToolOptions_1 = require("./DefaultExifToolOptions");
const ExifDate_1 = require("./ExifDate");
const ExifDateTime_1 = require("./ExifDateTime");
const ExifTime_1 = require("./ExifTime");
const Lazy_1 = require("./Lazy");
const Maybe_1 = require("./Maybe");
const Number_1 = require("./Number");
const Pick_1 = require("./Pick");
const String_1 = require("./String");
// Unique values from
// https://en.wikipedia.org/wiki/List_of_tz_database_time_zones, excluding those
// that have not been used for at least 50 years.
const ValidTimezoneOffsets = [
    // "-12:00", // not used for any populated land
    "-11:00",
    // "-10:30", // used by Hawaii 1896-1947
    "-10:00",
    "-09:30",
    "-09:00",
    "-08:30",
    "-08:00",
    "-07:00",
    "-06:00",
    "-05:00",
    "-04:30", // used by Venezuela 1912-1965 and 2007-2016
    "-04:00",
    "-03:30",
    "-03:00",
    "-02:30",
    "-02:00",
    "-01:00",
    // "-00:44", // used by Liberia until 1972
    // "-00:25:21", // Ireland 1880-1916 https://en.wikipedia.org/wiki/UTC%E2%88%9200:25:21
    "+00:00",
    // "+00:20", // used by Netherlands until 1940
    // "+00:30", // used by Switzerland until 1936
    "+01:00",
    // "+01:24", // used by Warsaw until 1915
    // "+01:30", // used by some southern African countries until 1903
    "+02:00",
    // "+02:30", // archaic Moscow time
    "+03:00",
    "+03:30",
    "+04:00",
    "+04:30",
    // "+04:51", // used by Bombay until 1955 https://en.wikipedia.org/wiki/UTC%2B04:51
    "+05:00",
    "+05:30",
    // "+05:40", // used by Nepal until 1920
    "+05:45", // Nepal
    "+06:00",
    "+06:30",
    "+07:00",
    // "+07:20", // used by Singapore and Malaya until 1941
    "+07:30", // used by Mayasia until 1982
    "+08:00",
    "+08:30", // used by North Korea until 2018
    "+08:45", // used by Western Australia, but not in tz database
    "+09:00",
    "+09:30",
    "+09:45", // used by Western Australia, but not in tz database
    "+10:00",
    "+10:30",
    "+11:00",
    "+12:00",
    "+12:45", // New Zealand islands
    "+13:00", // New Zealand and Antarctica
    "+13:45", // New Zealand islands
    "+14:00",
];
function offsetToMinutes(offset) {
    const [h, m] = offset.split(":").map(Number);
    // we can't just return `h * 60 + m`: that doesn't work with negative
    // offsets (minutes will be positive but hours will be negative)
    const sign = h < 0 ? -1 : 1;
    return h * 60 + sign * m;
}
const ValidOffsetMinutes = (0, Lazy_1.lazy)(() => new Set(ValidTimezoneOffsets.map(offsetToMinutes)));
/**
 * Zone instances with this offset are a placeholder for being "unset".
 */
exports.UnsetZoneOffsetMinutes = -1;
/**
 * This is a placeholder for dates where the zone is unknown/unset, because
 * Luxon doesn't officially support "unset" zones.
 */
exports.UnsetZone = luxon_1.Info.normalizeZone(exports.UnsetZoneOffsetMinutes);
/**
 * Zone instances with this name are a placeholder for being "unset".
 */
exports.UnsetZoneName = exports.UnsetZone.name;
const Zulus = [
    luxon_1.FixedOffsetZone.utcInstance,
    0,
    -0,
    "UTC",
    "GMT",
    "Z",
    "+0",
    "+00:00",
    "UTC+0",
    "GMT+0",
    "UTC+00:00",
    // ...sigh, so much for "normalizeZone"...
];
function isUTC(zone) {
    var _a;
    const z = zone;
    return (zone != null && (Zulus.includes(z) || Zulus.includes((_a = z.zoneName) !== null && _a !== void 0 ? _a : z.fixed)));
}
function isZoneValid(zone) {
    return (zone != null && zone.isValid && Math.abs(zone.offset(Date.now())) < 14 * 60);
}
/**
 * If `tzSource` matches this value, the tags are from a video, and we had to
 * resort to assuming time fields are in UTC.
 * @see https://github.com/photostructure/exiftool-vendored.js/issues/113
 */
exports.defaultVideosToUTC = "defaultVideosToUTC";
// https://en.wikipedia.org/wiki/List_of_tz_database_time_zones -- not that
// "WET" and "W-SU" are full TZs (!!!), and "America/Indiana/Indianapolis" is
// also a thing.
const IanaFormatRE = /^\w{2,15}(?:\/\w{3,15}){0,2}$/;
// Luxon requires fixed-offset zones to look like "UTC+H", "UTC-H",
// "UTC+H:mm", "UTC-H:mm":
const FixedFormatRE = /^UTC[+-]\d{1,2}(?::\d\d)?$/;
/**
 * @param input must be either a number, which is the offset in minutes, or a
 * string in the format "UTC+H" or "UTC+HH:mm"
 */
function normalizeZone(input) {
    // wrapped in a try/catch as Luxon.settings.throwOnInvalid may be true:
    try {
        // Info.normalizeZone returns the system zone if the input is null or
        // blank (!!!), but we want to return undefined instead:
        if ((0, String_1.blank)(input))
            return undefined;
        if (input instanceof luxon_1.Zone) {
            return isZoneValid(input) ? input : undefined;
        }
        // This test and short-circuit may not be necessary, but it's cheap and
        // explicit:
        if (isUTC(input))
            return luxon_1.FixedOffsetZone.utcInstance;
        let z = input;
        if (typeof z === "string") {
            z = z.replace(/^(?:Zulu|Z|GMT)(?:\b|$)/, "UTC");
            // We also don't need to tease Info.normalizeZone with obviously
            // non-offset inputs:
            if ((0, String_1.blank)(z) || (!IanaFormatRE.test(z) && !FixedFormatRE.test(z))) {
                return;
            }
        }
        const result = luxon_1.Info.normalizeZone(z);
        return isZoneValid(result) && result.name !== exports.UnsetZoneName
            ? result
            : undefined;
    }
    catch {
        return;
    }
}
/**
 * @param ts must be provided if the zone is not a fixed offset
 * @return the zone offset (in "±HH:MM" format) for the given zone, or "" if
 * the zone is invalid
 */
function zoneToShortOffset(zone, ts) {
    var _a, _b;
    return (_b = (_a = normalizeZone(zone)) === null || _a === void 0 ? void 0 : _a.formatOffset(ts !== null && ts !== void 0 ? ts : Date.now(), "short")) !== null && _b !== void 0 ? _b : "";
}
function validTzOffsetMinutes(tzOffsetMinutes) {
    return (tzOffsetMinutes != null &&
        (0, Number_1.isNumber)(tzOffsetMinutes) &&
        tzOffsetMinutes !== exports.UnsetZoneOffsetMinutes &&
        ValidOffsetMinutes().has(tzOffsetMinutes));
}
/**
 * Returns a "zone name" (used by `luxon`) that encodes the given offset.
 */
function offsetMinutesToZoneName(offsetMinutes) {
    if (!validTzOffsetMinutes(offsetMinutes)) {
        return undefined;
    }
    if (offsetMinutes === 0)
        return "UTC";
    const sign = offsetMinutes < 0 ? "-" : "+";
    const absMinutes = Math.abs(offsetMinutes);
    const hours = Math.floor(absMinutes / 60);
    const minutes = Math.abs(absMinutes % 60);
    // luxon now renders simple hour offsets without padding:
    return `UTC${sign}` + hours + (minutes === 0 ? "" : `:${(0, String_1.pad2)(minutes)}`);
}
function tzHourToOffset(n) {
    return (0, Number_1.isNumber)(n) && validTzOffsetMinutes(n * 60)
        ? offsetMinutesToZoneName(n * 60)
        : undefined;
}
// Accept "Z", "UTC+2", "UTC+02", "UTC+2:00", "UTC+02:00", "+2", "+02", and
// "+02:00". Require the sign (+ or -) and a ":" separator if there are
// minutes.
const tzRe = /(?<Z>Z)|((UTC)?(?<sign>[+-])(?<hours>\d\d?)(?::(?<minutes>\d\d))?)$/;
function extractOffsetFromHours(hourOffset) {
    return (0, Number_1.isNumber)(hourOffset)
        ? (0, Maybe_1.map)(tzHourToOffset(hourOffset), (tz) => ({
            tz,
            src: "hourOffset",
        }))
        : Array.isArray(hourOffset)
            ? extractOffsetFromHours(hourOffset[0])
            : undefined;
}
/**
 * Parse a timezone offset and return the offset minutes
 *
 * @param opts.stripTZA If false, do not strip off the timezone abbreviation
 * (TZA) from the value. Defaults to true.
 *
 * @return undefined if the value cannot be parsed as a valid timezone offset
 */
function extractZone(value, opts) {
    var _a, _b;
    if (value == null ||
        typeof value === "boolean" ||
        value instanceof BinaryField_1.BinaryField ||
        value instanceof ExifDate_1.ExifDate) {
        return;
    }
    if (Array.isArray(value)) {
        // we only ever care about the first non-null value
        return extractZone(value.find((ea) => ea != null));
    }
    if (value instanceof ExifDateTime_1.ExifDateTime || value instanceof ExifTime_1.ExifTime) {
        return value.zone == null
            ? undefined
            : { tz: value.zone, src: value.constructor.name + ".zone" };
    }
    if ((0, Number_1.isNumber)(value)) {
        return extractOffsetFromHours(value);
    }
    if (typeof value !== "string" || (0, String_1.blank)(value)) {
        // don't accept ExifDate, boolean, BinaryField, ResourceEvent, Struct, or
        // Version instances:
        return;
    }
    {
        // If value is a proper timezone name, this may be easy!
        const z = normalizeZone(value);
        if (z != null) {
            return { tz: z.name, src: "normalizeZone" };
        }
    }
    let str = value.trim();
    // Some EXIF datetime will "over-specify" and include both the utc offset
    // *and* the "time zone abbreviation"/TZA, like "PST" or "PDT". TZAs are
    // between 2 (AT) and 5 (WEST) characters.
    if ((opts === null || opts === void 0 ? void 0 : opts.stripTZA) !== false &&
        // We only want to strip off the TZA if the input _doesn't_ end with "UTC"
        // or "Z"
        !/[.\d\s](?:UTC|Z)$/.test(str)) {
        str = str.replace(/\s[a-z]{2,5}$/i, "");
    }
    {
        if ((0, String_1.blank)(str))
            return;
        const z = normalizeZone(str);
        if (z != null) {
            return { tz: z.name, src: "normalizeZone" };
        }
    }
    const match = tzRe.exec(str);
    const capturedGroups = match === null || match === void 0 ? void 0 : match.groups;
    if (match != null && capturedGroups != null) {
        const leftovers = str.slice(0, match.index);
        if (capturedGroups.Z === "Z")
            return {
                tz: "UTC",
                src: "Z",
                leftovers,
            };
        const offsetMinutes = (capturedGroups.sign === "-" ? -1 : 1) *
            (parseInt((_a = capturedGroups.hours) !== null && _a !== void 0 ? _a : "0") * 60 +
                parseInt((_b = capturedGroups.minutes) !== null && _b !== void 0 ? _b : "0"));
        const tz = offsetMinutesToZoneName(offsetMinutes);
        if (tz != null) {
            return { tz, src: "offsetMinutesToZoneName", leftovers };
        }
    }
    return;
}
exports.TimezoneOffsetTagnames = [
    "TimeZone",
    "OffsetTime",
    // time zone for DateTimeOriginal, "-08:00"
    "OffsetTimeOriginal",
    // time zone for CreateDate, "-08:00"
    "OffsetTimeDigitized",
    // srsly who came up with these wholly inconsistent tag names? _why not just
    // prefix tag names with "Offset"?!11_ SADNESS AND WOE
    // 1 or 2 values: 1. The time zone offset of DateTimeOriginal from GMT in
    // hours, 2. If present, the time zone offset of ModifyDate (which we
    // ignore) @see https://www.exiftool.org/TagNames/EXIF.html
    "TimeZoneOffset", // number | number[] | string
    "GeolocationTimeZone",
];
function incrementZone(z, minutes) {
    const norm = normalizeZone(z);
    if (norm == null || true !== norm.isUniversal)
        return;
    const fixed = norm.offset(Date.now()); // < arg doesn't matter, it's universal
    return (0, Number_1.isNumber)(fixed) ? luxon_1.FixedOffsetZone.instance(fixed + minutes) : undefined;
}
function extractTzOffsetFromTags(t, opts) {
    var _a;
    const adjustFn = (_a = opts === null || opts === void 0 ? void 0 : opts.adjustTimeZoneIfDaylightSavings) !== null && _a !== void 0 ? _a : DefaultExifToolOptions_1.defaultAdjustTimeZoneIfDaylightSavings;
    for (const tagName of exports.TimezoneOffsetTagnames) {
        const offset = extractZone(t[tagName]);
        if (offset == null)
            continue;
        // UGH. See https://github.com/photostructure/exiftool-vendored.js/issues/215
        const minutes = adjustFn(t, offset.tz);
        if (minutes != null) {
            const adjustedZone = incrementZone(offset.tz, minutes);
            if (adjustedZone != null)
                return {
                    tz: adjustedZone.name,
                    src: tagName + " (adjusted for DaylightSavings)",
                };
        }
        // No fancy adjustments needed, just return the extracted zone:
        return { tz: offset.tz, src: tagName };
    }
    return;
}
function extractTzOffsetFromDatestamps(t, opts) {
    var _a;
    if ((opts === null || opts === void 0 ? void 0 : opts.inferTimezoneFromDatestamps) === true) {
        for (const tagName of (_a = opts.inferTimezoneFromDatestampTags) !== null && _a !== void 0 ? _a : []) {
            if (t[tagName] != null) {
                const offset = extractZone(t[tagName]);
                // Some applications (looking at you, Google Takeout) will add a
                // spurious "+00:00" timezone offset to random datestamp tags, so
                // ignore UTC offsets here.
                if (offset != null && !isUTC(offset.tz)) {
                    return { tz: offset.tz, src: tagName };
                }
            }
        }
    }
    return;
}
function extractTzOffsetFromTimeStamp(t, opts) {
    var _a;
    if ((opts === null || opts === void 0 ? void 0 : opts.inferTimezoneFromTimeStamp) !== true)
        return;
    const ts = ExifDateTime_1.ExifDateTime.from(t.TimeStamp);
    if (ts == null)
        return;
    for (const tagName of (_a = opts.inferTimezoneFromDatestampTags) !== null && _a !== void 0 ? _a : []) {
        const ea = ExifDateTime_1.ExifDateTime.from(t[tagName]);
        if (ea == null)
            continue;
        if (ea.zone != null) {
            return { tz: ea.zone, src: tagName };
        }
        const deltaMinutes = Math.floor((ea.toEpochSeconds("UTC") - ts.toEpochSeconds()) / 60);
        const likelyOffsetZone = inferLikelyOffsetMinutes(deltaMinutes);
        const tz = offsetMinutesToZoneName(likelyOffsetZone);
        if (tz != null) {
            return { tz, src: "offset between " + tagName + " and TimeStamp" };
        }
    }
    return;
}
// timezone offsets may be on a 15 minute boundary, but if GPS acquisition is
// old, this can be spurious. We get less mistakes with a larger multiple, so
// we're using 30 minutes instead of 15. See
// https://www.timeanddate.com/time/time-zones-interesting.html
const LikelyOffsetMinutes = ValidTimezoneOffsets.map(offsetToMinutes);
function inferLikelyOffsetMinutes(deltaMinutes) {
    const nearest = (0, Array_1.leastBy)(LikelyOffsetMinutes, (ea) => Math.abs(ea - deltaMinutes));
    // Reject timezone offsets more than 30 minutes away from the nearest:
    return Math.abs(nearest - deltaMinutes) < 30 ? nearest : undefined;
}
/**
 * Convert blank strings to undefined.
 */
function blankToNull(x) {
    return x == null || (typeof x === "string" && (0, String_1.blank)(x)) ? undefined : x;
}
function extractTzOffsetFromUTCOffset(t) {
    const utcSources = {
        ...(0, Pick_1.pick)(t, "GPSDateTime", "DateTimeUTC", "SonyDateTime2"),
        GPSDateTimeStamp: (0, Maybe_1.map2)(blankToNull(t.GPSDateStamp), // Example: "2022:04:13"
        blankToNull(t.GPSTimeStamp), // Example: "23:59:41.001"
        (a, b) => a + " " + b),
    };
    // We can always assume these are in UTC:
    const utc = (0, Maybe_1.first)([
        "GPSDateTime",
        "DateTimeUTC",
        "GPSDateTimeStamp",
        "SonyDateTime2",
    ], (tagName) => {
        var _a;
        const v = utcSources[tagName];
        const edt = v instanceof ExifDateTime_1.ExifDateTime ? v : ExifDateTime_1.ExifDateTime.fromExifStrict(v);
        const s = edt != null && (edt.zone == null || isUTC(edt.zone))
            ? (_a = edt.setZone("UTC", { keepLocalTime: true })) === null || _a === void 0 ? void 0 : _a.toEpochSeconds()
            : undefined;
        return s != null
            ? {
                tagName,
                s,
            }
            : undefined;
    });
    if (utc == null)
        return;
    // If we can find any of these without a zone, the timezone should be the
    // offset between this time and the GPS time.
    const dt = (0, Maybe_1.first)(CapturedAtTagNames_1.CapturedAtTagNames, (tagName) => {
        var _a;
        const edt = ExifDateTime_1.ExifDateTime.fromExifStrict(t[tagName]);
        const s = edt != null && edt.zone == null
            ? (_a = edt.setZone("UTC", { keepLocalTime: true })) === null || _a === void 0 ? void 0 : _a.toEpochSeconds()
            : undefined;
        return s != null
            ? {
                tagName,
                s,
            }
            : undefined;
    });
    if (dt == null)
        return;
    const diffSeconds = dt.s - utc.s;
    const offsetMinutes = inferLikelyOffsetMinutes(diffSeconds / 60);
    return (0, Maybe_1.map)(offsetMinutesToZoneName(offsetMinutes), (tz) => ({
        tz,
        src: `offset between ${dt.tagName} and ${utc.tagName}`,
    }));
}
function equivalentZones(a, b) {
    const az = normalizeZone(a);
    const bz = normalizeZone(b);
    return (az != null &&
        bz != null &&
        (az.equals(bz) || az.offset(Date.now()) === bz.offset(Date.now())));
}
function getZoneName(args = {}) {
    var _a, _b, _c, _d;
    const result = (_d = (_b = (_a = normalizeZone(args.zone)) === null || _a === void 0 ? void 0 : _a.name) !== null && _b !== void 0 ? _b : (_c = normalizeZone(args.zoneName)) === null || _c === void 0 ? void 0 : _c.name) !== null && _d !== void 0 ? _d : offsetMinutesToZoneName(args.tzoffsetMinutes);
    return (0, String_1.blank)(result) || result === exports.UnsetZoneName ? undefined : result;
}
//# sourceMappingURL=Timezones.js.map