import { ChildCommand, ParentCommand } from '../enums';
const sandbox = (processFile, childPool) => {
    return async function process(job, token) {
        const child = await childPool.retain(processFile);
        let msgHandler;
        let exitHandler;
        await child.send({
            cmd: ChildCommand.Start,
            job: job.asJSONSandbox(),
            token,
        });
        const done = new Promise((resolve, reject) => {
            msgHandler = async (msg) => {
                var _a, _b;
                switch (msg.cmd) {
                    case ParentCommand.Completed:
                        resolve(msg.value);
                        break;
                    case ParentCommand.Failed:
                    case ParentCommand.Error: {
                        const err = new Error();
                        Object.assign(err, msg.value);
                        reject(err);
                        break;
                    }
                    case ParentCommand.Progress:
                        await job.updateProgress(msg.value);
                        break;
                    case ParentCommand.Log:
                        await job.log(msg.value);
                        break;
                    case ParentCommand.MoveToDelayed:
                        await job.moveToDelayed((_a = msg.value) === null || _a === void 0 ? void 0 : _a.timestamp, (_b = msg.value) === null || _b === void 0 ? void 0 : _b.token);
                        break;
                    case ParentCommand.Update:
                        await job.updateData(msg.value);
                        break;
                }
            };
            exitHandler = (exitCode, signal) => {
                reject(new Error('Unexpected exit code: ' + exitCode + ' signal: ' + signal));
            };
            child.on('message', msgHandler);
            child.on('exit', exitHandler);
        });
        try {
            await done;
            return done;
        }
        finally {
            child.off('message', msgHandler);
            child.off('exit', exitHandler);
            if (child.exitCode !== null || /SIG.*/.test(`${child.signalCode}`)) {
                childPool.remove(child);
            }
            else {
                childPool.release(child);
            }
        }
    };
};
export default sandbox;
//# sourceMappingURL=sandbox.js.map