import { IoredisListener, QueueEventsOptions } from '../interfaces';
import { QueueBase } from './queue-base';
import { RedisConnection } from './redis-connection';
export interface QueueEventsListener extends IoredisListener {
    /**
     * Listen to 'active' event.
     *
     * This event is triggered when a job enters the 'active' state.
     */
    active: (args: {
        jobId: string;
        prev?: string;
    }, id: string) => void;
    /**
     * Listen to 'added' event.
     *
     * This event is triggered when a job is created.
     */
    added: (args: {
        jobId: string;
        name: string;
    }, id: string) => void;
    /**
     * Listen to 'cleaned' event.
     *
     * This event is triggered when a cleaned method is triggered.
     */
    cleaned: (args: {
        count: string;
    }, id: string) => void;
    /**
     * Listen to 'completed' event.
     *
     * This event is triggered when a job has successfully completed.
     */
    completed: (args: {
        jobId: string;
        returnvalue: string;
        prev?: string;
    }, id: string) => void;
    /**
     * Listen to 'delayed' event.
     *
     * This event is triggered when a job is delayed.
     */
    delayed: (args: {
        jobId: string;
        delay: number;
    }, id: string) => void;
    /**
     * Listen to 'drained' event.
     *
     * This event is triggered when the queue has drained the waiting list.
     * Note that there could still be delayed jobs waiting their timers to expire
     * and this event will still be triggered as long as the waiting list has emptied.
     */
    drained: (id: string) => void;
    /**
     * Listen to 'duplicated' event.
     *
     * This event is triggered when a job is not created because it already exist.
     */
    duplicated: (args: {
        jobId: string;
    }, id: string) => void;
    /**
     * Listen to 'error' event.
     *
     * This event is triggered when an exception is thrown.
     */
    error: (args: Error) => void;
    /**
     * Listen to 'failed' event.
     *
     * This event is triggered when a job has thrown an exception.
     */
    failed: (args: {
        jobId: string;
        failedReason: string;
        prev?: string;
    }, id: string) => void;
    /**
     * Listen to 'paused' event.
     *
     * This event is triggered when a queue is paused.
     */
    paused: (args: {}, id: string) => void;
    /**
     * Listen to 'progress' event.
     *
     * This event is triggered when a job updates it progress, i.e. the
     * Job##updateProgress() method is called. This is useful to notify
     * progress or any other data from within a processor to the rest of the
     * world.
     */
    progress: (args: {
        jobId: string;
        data: number | object;
    }, id: string) => void;
    /**
     * Listen to 'removed' event.
     *
     * This event is triggered when a job has been manually
     * removed from the queue.
     */
    removed: (args: {
        jobId: string;
        prev: string;
    }, id: string) => void;
    /**
     * Listen to 'resumed' event.
     *
     * This event is triggered when a queue is resumed.
     */
    resumed: (args: {}, id: string) => void;
    /**
     * Listen to 'retries-exhausted' event.
     *
     * This event is triggered when a job has retried the maximum attempts.
     */
    'retries-exhausted': (args: {
        jobId: string;
        attemptsMade: string;
    }, id: string) => void;
    /**
     * Listen to 'stalled' event.
     *
     * This event is triggered when a job has been moved from 'active' back
     * to 'waiting'/'failed' due to the processor not being able to renew
     * the lock on the said job.
     */
    stalled: (args: {
        jobId: string;
    }, id: string) => void;
    /**
     * Listen to 'waiting' event.
     *
     * This event is triggered when a job enters the 'waiting' state.
     */
    waiting: (args: {
        jobId: string;
        prev?: string;
    }, id: string) => void;
    /**
     * Listen to 'waiting-children' event.
     *
     * This event is triggered when a job enters the 'waiting-children' state.
     */
    'waiting-children': (args: {
        jobId: string;
    }, id: string) => void;
}
/**
 * The QueueEvents class is used for listening to the global events
 * emitted by a given queue.
 *
 * This class requires a dedicated redis connection.
 *
 */
export declare class QueueEvents extends QueueBase {
    private running;
    constructor(name: string, { connection, autorun, ...opts }?: QueueEventsOptions, Connection?: typeof RedisConnection);
    emit<U extends keyof QueueEventsListener>(event: U, ...args: Parameters<QueueEventsListener[U]>): boolean;
    off<U extends keyof QueueEventsListener>(eventName: U, listener: QueueEventsListener[U]): this;
    on<U extends keyof QueueEventsListener>(event: U, listener: QueueEventsListener[U]): this;
    once<U extends keyof QueueEventsListener>(event: U, listener: QueueEventsListener[U]): this;
    /**
     * Manually starts running the event consumming loop. This shall be used if you do not
     * use the default "autorun" option on the constructor.
     */
    run(): Promise<void>;
    private consumeEvents;
    /**
     * Stops consuming events and close the underlying Redis connection if necessary.
     *
     * @returns
     */
    close(): Promise<void>;
}
