import { ChainableCommander } from 'ioredis';
import { BulkJobOptions, DependenciesOpts, JobJson, JobJsonRaw, MinimalJob, MoveToWaitingChildrenOpts, ParentKeys, ParentOpts, RedisClient } from '../interfaces';
import { FinishedStatus, JobsOptions, JobState, JobJsonSandbox, MinimalQueue } from '../types';
import { Scripts } from './scripts';
import type { QueueEvents } from './queue-events';
export declare const PRIORITY_LIMIT: number;
/**
 * Job
 *
 * This class represents a Job in the queue. Normally job are implicitly created when
 * you add a job to the queue with methods such as Queue.addJob( ... )
 *
 * A Job instance is also passed to the Worker's process function.
 *
 * @class Job
 */
export declare class Job<DataType = any, ReturnType = any, NameType extends string = string> implements MinimalJob<DataType, ReturnType, NameType> {
    protected queue: MinimalQueue;
    /**
     * The name of the Job
     */
    name: NameType;
    /**
     * The payload for this job.
     */
    data: DataType;
    /**
     * The options object for this job.
     */
    opts: JobsOptions;
    id?: string;
    /**
     * It includes the prefix, the namespace separator :, and queue name.
     * @see https://www.gnu.org/software/gawk/manual/html_node/Qualified-Names.html
     */
    readonly queueQualifiedName: string;
    /**
     * The progress a job has performed so far.
     * @defaultValue 0
     */
    progress: number | object;
    /**
     * The value returned by the processor when processing this job.
     * @defaultValue null
     */
    returnvalue: ReturnType;
    /**
     * Stacktrace for the error (for failed jobs).
     * @defaultValue null
     */
    stacktrace: string[];
    /**
     * An amount of milliseconds to wait until this job can be processed.
     * @defaultValue 0
     */
    delay: number;
    /**
     * Timestamp when the job was created (unless overridden with job options).
     */
    timestamp: number;
    /**
     * Number of attempts after the job has failed.
     * @defaultValue 0
     */
    attemptsMade: number;
    /**
     * Reason for failing.
     */
    failedReason: string;
    /**
     * Timestamp for when the job finished (completed or failed).
     */
    finishedOn?: number;
    /**
     * Timestamp for when the job was processed.
     */
    processedOn?: number;
    /**
     * Fully qualified key (including the queue prefix) pointing to the parent of this job.
     */
    parentKey?: string;
    /**
     * Object that contains parentId (id) and parent queueKey.
     */
    parent?: ParentKeys;
    /**
     * Base repeat job key.
     */
    repeatJobKey?: string;
    /**
     * The token used for locking this job.
     */
    token?: string;
    protected toKey: (type: string) => string;
    protected discarded: boolean;
    protected scripts: Scripts;
    constructor(queue: MinimalQueue, 
    /**
     * The name of the Job
     */
    name: NameType, 
    /**
     * The payload for this job.
     */
    data: DataType, 
    /**
     * The options object for this job.
     */
    opts?: JobsOptions, id?: string);
    /**
     * Creates a new job and adds it to the queue.
     *
     * @param queue - the queue where to add the job.
     * @param name - the name of the job.
     * @param data - the payload of the job.
     * @param opts - the options bag for this job.
     * @returns
     */
    static create<T = any, R = any, N extends string = string>(queue: MinimalQueue, name: N, data: T, opts?: JobsOptions): Promise<Job<T, R, N>>;
    /**
     * Creates a bulk of jobs and adds them atomically to the given queue.
     *
     * @param queue -the queue were to add the jobs.
     * @param jobs - an array of jobs to be added to the queue.
     * @returns
     */
    static createBulk<T = any, R = any, N extends string = string>(queue: MinimalQueue, jobs: {
        name: N;
        data: T;
        opts?: BulkJobOptions;
    }[]): Promise<Job<T, R, N>[]>;
    /**
     * Instantiates a Job from a JobJsonRaw object (coming from a deserialized JSON object)
     *
     * @param queue - the queue where the job belongs to.
     * @param json - the plain object containing the job.
     * @param jobId - an optional job id (overrides the id coming from the JSON object)
     * @returns
     */
    static fromJSON<T = any, R = any, N extends string = string>(queue: MinimalQueue, json: JobJsonRaw, jobId?: string): Job<T, R, N>;
    private static optsFromJSON;
    /**
     * Fetches a Job from the queue given the passed job id.
     *
     * @param queue - the queue where the job belongs to.
     * @param jobId - the job id.
     * @returns
     */
    static fromId<T = any, R = any, N extends string = string>(queue: MinimalQueue, jobId: string): Promise<Job<T, R, N> | undefined>;
    /**
     * addJobLog
     *
     * @param queue Queue instance
     * @param jobId Job id
     * @param logRow Log row
     * @param keepLogs optional maximum number of logs to keep
     *
     * @returns The total number of log entries for this job so far.
     */
    static addJobLog(queue: MinimalQueue, jobId: string, logRow: string, keepLogs?: number): Promise<number>;
    toJSON(): Omit<this, "toJSON" | "scripts" | "changeDelay" | "changePriority" | "extendLock" | "getState" | "moveToDelayed" | "moveToWaitingChildren" | "promote" | "updateData" | "updateProgress" | "discard" | "addJob" | "prefix" | "queue" | "asJSON" | "asJSONSandbox" | "log" | "clearLogs" | "remove" | "moveToCompleted" | "moveToFailed" | "isCompleted" | "isFailed" | "isDelayed" | "isWaitingChildren" | "isActive" | "isWaiting" | "queueName" | "getChildrenValues" | "getDependencies" | "getDependenciesCount" | "waitUntilFinished" | "retry">;
    /**
     * Prepares a job to be serialized for storage in Redis.
     * @returns
     */
    asJSON(): JobJson;
    private optsAsJSON;
    /**
     * Prepares a job to be passed to Sandbox.
     * @returns
     */
    asJSONSandbox(): JobJsonSandbox;
    /**
     * Updates a job's data
     *
     * @param data - the data that will replace the current jobs data.
     */
    updateData(data: DataType): Promise<void>;
    /**
     * Updates a job's progress
     *
     * @param progress - number or object to be saved as progress.
     */
    updateProgress(progress: number | object): Promise<void>;
    /**
     * Logs one row of log data.
     *
     * @param logRow - string with log data to be logged.
     * @returns The total number of log entries for this job so far.
     */
    log(logRow: string): Promise<number>;
    /**
     * Clears job's logs
     *
     * @param keepLogs - the amount of log entries to preserve
     */
    clearLogs(keepLogs?: number): Promise<void>;
    /**
     * Completely remove the job from the queue.
     * Note, this call will throw an exception if the job
     * is being processed when the call is performed.
     *
     * @param opts - Options to remove a job
     */
    remove({ removeChildren }?: {
        removeChildren?: boolean;
    }): Promise<void>;
    /**
     * Extend the lock for this job.
     *
     * @param token - unique token for the lock
     * @param duration - lock duration in milliseconds
     */
    extendLock(token: string, duration: number): Promise<number>;
    /**
     * Moves a job to the completed queue.
     * Returned job to be used with Queue.prototype.nextJobFromJobData.
     *
     * @param returnValue - The jobs success message.
     * @param token - Worker token used to acquire completed job.
     * @param fetchNext - True when wanting to fetch the next job.
     * @returns Returns the jobData of the next job in the waiting queue.
     */
    moveToCompleted(returnValue: ReturnType, token: string, fetchNext?: boolean): Promise<any[]>;
    /**
     * Moves a job to the failed queue.
     *
     * @param err - the jobs error message.
     * @param token - token to check job is locked by current worker
     * @param fetchNext - true when wanting to fetch the next job
     * @returns void
     */
    moveToFailed<E extends Error>(err: E, token: string, fetchNext?: boolean): Promise<void>;
    /**
     * @returns true if the job has completed.
     */
    isCompleted(): Promise<boolean>;
    /**
     * @returns true if the job has failed.
     */
    isFailed(): Promise<boolean>;
    /**
     * @returns true if the job is delayed.
     */
    isDelayed(): Promise<boolean>;
    /**
     * @returns true if the job is waiting for children.
     */
    isWaitingChildren(): Promise<boolean>;
    /**
     * @returns true of the job is active.
     */
    isActive(): Promise<boolean>;
    /**
     * @returns true if the job is waiting.
     */
    isWaiting(): Promise<boolean>;
    /**
     * @returns the queue name this job belongs to.
     */
    get queueName(): string;
    /**
     * @returns the prefix that is used.
     */
    get prefix(): string;
    /**
     * Get current state.
     *
     * @returns Returns one of these values:
     * 'completed', 'failed', 'delayed', 'active', 'waiting', 'waiting-children', 'unknown'.
     */
    getState(): Promise<JobState | 'unknown'>;
    /**
     * Change delay of a delayed job.
     *
     * @param delay - milliseconds to be added to current time.
     * @returns void
     */
    changeDelay(delay: number): Promise<void>;
    /**
     * Change job priority.
     *
     * @returns void
     */
    changePriority(opts: {
        priority?: number;
        lifo?: boolean;
    }): Promise<void>;
    /**
     * Get this jobs children result values if any.
     *
     * @returns Object mapping children job keys with their values.
     */
    getChildrenValues<CT = any>(): Promise<{
        [jobKey: string]: CT;
    }>;
    /**
     * Get children job keys if this job is a parent and has children.
     * @remarks
     * Count options before Redis v7.2 works as expected with any quantity of entries
     * on processed/unprocessed dependencies, since v7.2 you must consider that count
     * won't have any effect until processed/unprocessed dependencies have a length
     * greater than 127
     * @see https://redis.io/docs/management/optimization/memory-optimization/#redis--72
     * @returns dependencies separated by processed and unprocessed.
     */
    getDependencies(opts?: DependenciesOpts): Promise<{
        nextProcessedCursor?: number;
        processed?: Record<string, any>;
        nextUnprocessedCursor?: number;
        unprocessed?: string[];
    }>;
    /**
     * Get children job counts if this job is a parent and has children.
     *
     * @returns dependencies count separated by processed and unprocessed.
     */
    getDependenciesCount(opts?: {
        processed?: boolean;
        unprocessed?: boolean;
    }): Promise<{
        processed?: number;
        unprocessed?: number;
    }>;
    /**
     * Returns a promise the resolves when the job has completed (containing the return value of the job),
     * or rejects when the job has failed (containing the failedReason).
     *
     * @param queueEvents - Instance of QueueEvents.
     * @param ttl - Time in milliseconds to wait for job to finish before timing out.
     */
    waitUntilFinished(queueEvents: QueueEvents, ttl?: number): Promise<ReturnType>;
    /**
     * Moves the job to the delay set.
     *
     * @param timestamp - timestamp where the job should be moved back to "wait"
     * @param token - token to check job is locked by current worker
     * @returns
     */
    moveToDelayed(timestamp: number, token?: string): Promise<void>;
    /**
     * Moves the job to the waiting-children set.
     *
     * @param token - Token to check job is locked by current worker
     * @param opts - The options bag for moving a job to waiting-children.
     * @returns true if the job was moved
     */
    moveToWaitingChildren(token: string, opts?: MoveToWaitingChildrenOpts): Promise<boolean>;
    /**
     * Promotes a delayed job so that it starts to be processed as soon as possible.
     */
    promote(): Promise<void>;
    /**
     * Attempts to retry the job. Only a job that has failed or completed can be retried.
     *
     * @param state - completed / failed
     * @returns If resolved and return code is 1, then the queue emits a waiting event
     * otherwise the operation was not a success and throw the corresponding error. If the promise
     * rejects, it indicates that the script failed to execute
     */
    retry(state?: FinishedStatus): Promise<void>;
    /**
     * Marks a job to not be retried if it fails (even if attempts has been configured)
     */
    discard(): void;
    private isInZSet;
    private isInList;
    /**
     * Adds the job to Redis.
     *
     * @param client -
     * @param parentOpts -
     * @returns
     */
    addJob(client: RedisClient, parentOpts?: ParentOpts): Promise<string>;
    protected validateOptions(jobData: JobJson): void;
    protected saveStacktrace(multi: ChainableCommander, err: Error): void;
}
