"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.HostMetrics = void 0;
const BaseMetrics_1 = require("./BaseMetrics");
const enum_1 = require("./enum");
const common_1 = require("./stats/common");
const si_1 = require("./stats/si");
/**
 * Metrics Collector - collects metrics for CPU, Memory, Network
 */
class HostMetrics extends BaseMetrics_1.BaseMetrics {
    _batchUpdateCpuUsages(observableResult, cpuUsages) {
        const stateAttr = enum_1.ATTRIBUTE_NAMES.SYSTEM_CPU_STATE;
        const cpuAttr = enum_1.ATTRIBUTE_NAMES.SYSTEM_CPU_LOGICAL_NUMBER;
        for (let i = 0, j = cpuUsages.length; i < j; i++) {
            const cpuUsage = cpuUsages[i];
            observableResult.observe(this._cpuTime, cpuUsage.user, {
                [stateAttr]: enum_1.CPU_LABELS.USER,
                [cpuAttr]: cpuUsage.cpuNumber,
            });
            observableResult.observe(this._cpuTime, cpuUsage.system, {
                [stateAttr]: enum_1.CPU_LABELS.SYSTEM,
                [cpuAttr]: cpuUsage.cpuNumber,
            });
            observableResult.observe(this._cpuTime, cpuUsage.idle, {
                [stateAttr]: enum_1.CPU_LABELS.IDLE,
                [cpuAttr]: cpuUsage.cpuNumber,
            });
            observableResult.observe(this._cpuTime, cpuUsage.interrupt, {
                [stateAttr]: enum_1.CPU_LABELS.INTERRUPT,
                [cpuAttr]: cpuUsage.cpuNumber,
            });
            observableResult.observe(this._cpuTime, cpuUsage.nice, {
                [stateAttr]: enum_1.CPU_LABELS.NICE,
                [cpuAttr]: cpuUsage.cpuNumber,
            });
            observableResult.observe(this._cpuUtilization, cpuUsage.userP, {
                [stateAttr]: enum_1.CPU_LABELS.USER,
                [cpuAttr]: cpuUsage.cpuNumber,
            });
            observableResult.observe(this._cpuUtilization, cpuUsage.systemP, {
                [stateAttr]: enum_1.CPU_LABELS.SYSTEM,
                [cpuAttr]: cpuUsage.cpuNumber,
            });
            observableResult.observe(this._cpuUtilization, cpuUsage.idleP, {
                [stateAttr]: enum_1.CPU_LABELS.IDLE,
                [cpuAttr]: cpuUsage.cpuNumber,
            });
            observableResult.observe(this._cpuUtilization, cpuUsage.interruptP, {
                [stateAttr]: enum_1.CPU_LABELS.INTERRUPT,
                [cpuAttr]: cpuUsage.cpuNumber,
            });
            observableResult.observe(this._cpuUtilization, cpuUsage.niceP, {
                [stateAttr]: enum_1.CPU_LABELS.NICE,
                [cpuAttr]: cpuUsage.cpuNumber,
            });
        }
    }
    _batchUpdateProcessCpuUsages(observableResult, processCpuUsage) {
        const stateAttr = enum_1.ATTRIBUTE_NAMES.PROCESS_CPU_STATE;
        observableResult.observe(this._processCpuTime, processCpuUsage.user, {
            [stateAttr]: enum_1.CPU_LABELS.USER,
        });
        observableResult.observe(this._processCpuTime, processCpuUsage.system, {
            [stateAttr]: enum_1.CPU_LABELS.SYSTEM,
        });
        observableResult.observe(this._processCpuUtilization, processCpuUsage.userP, {
            [stateAttr]: enum_1.CPU_LABELS.USER,
        });
        observableResult.observe(this._processCpuUtilization, processCpuUsage.systemP, {
            [stateAttr]: enum_1.CPU_LABELS.SYSTEM,
        });
    }
    _batchUpdateMemUsages(observableResult, memUsage) {
        const stateAttr = enum_1.ATTRIBUTE_NAMES.SYSTEM_MEMORY_STATE;
        observableResult.observe(this._memoryUsage, memUsage.used, {
            [stateAttr]: enum_1.MEMORY_LABELS.USED,
        });
        observableResult.observe(this._memoryUsage, memUsage.free, {
            [stateAttr]: enum_1.MEMORY_LABELS.FREE,
        });
        observableResult.observe(this._memoryUtilization, memUsage.usedP, {
            [stateAttr]: enum_1.MEMORY_LABELS.USED,
        });
        observableResult.observe(this._memoryUtilization, memUsage.freeP, {
            [stateAttr]: enum_1.MEMORY_LABELS.FREE,
        });
    }
    _batchUpdateProcessMemUsage(observableResult, memoryUsage) {
        observableResult.observe(this._processMemoryUsage, memoryUsage);
    }
    _batchUpdateNetworkData(observableResult, networkUsages) {
        const deviceAttr = enum_1.ATTRIBUTE_NAMES.SYSTEM_DEVICE;
        const directionAttr = enum_1.ATTRIBUTE_NAMES.NETWORK_IO_DIRECTION;
        for (let i = 0, j = networkUsages.length; i < j; i++) {
            const networkUsage = networkUsages[i];
            observableResult.observe(this._networkDropped, networkUsage.rx_dropped, {
                [deviceAttr]: networkUsage.iface,
                [directionAttr]: enum_1.NETWORK_LABELS.RECEIVE,
            });
            observableResult.observe(this._networkDropped, networkUsage.tx_dropped, {
                [deviceAttr]: networkUsage.iface,
                [directionAttr]: enum_1.NETWORK_LABELS.TRANSMIT,
            });
            observableResult.observe(this._networkErrors, networkUsage.rx_errors, {
                [deviceAttr]: networkUsage.iface,
                [directionAttr]: enum_1.NETWORK_LABELS.RECEIVE,
            });
            observableResult.observe(this._networkErrors, networkUsage.tx_errors, {
                [deviceAttr]: networkUsage.iface,
                [directionAttr]: enum_1.NETWORK_LABELS.TRANSMIT,
            });
            observableResult.observe(this._networkIo, networkUsage.rx_bytes, {
                [deviceAttr]: networkUsage.iface,
                [directionAttr]: enum_1.NETWORK_LABELS.RECEIVE,
            });
            observableResult.observe(this._networkIo, networkUsage.tx_bytes, {
                [deviceAttr]: networkUsage.iface,
                [directionAttr]: enum_1.NETWORK_LABELS.TRANSMIT,
            });
        }
    }
    /**
     * Creates metrics
     */
    _createMetrics() {
        this._cpuTime = this._meter.createObservableCounter(enum_1.METRIC_NAMES.CPU_TIME, {
            description: 'Cpu time in seconds',
            unit: 's',
        });
        this._cpuUtilization = this._meter.createObservableGauge(enum_1.METRIC_NAMES.CPU_UTILIZATION, {
            description: 'Cpu usage time 0-1',
        });
        this._memoryUsage = this._meter.createObservableGauge(enum_1.METRIC_NAMES.MEMORY_USAGE, {
            description: 'Memory usage in bytes',
        });
        this._memoryUtilization = this._meter.createObservableGauge(enum_1.METRIC_NAMES.MEMORY_UTILIZATION, {
            description: 'Memory usage 0-1',
        });
        this._networkDropped = this._meter.createObservableCounter(enum_1.METRIC_NAMES.NETWORK_DROPPED, {
            description: 'Network dropped packets',
        });
        this._networkErrors = this._meter.createObservableCounter(enum_1.METRIC_NAMES.NETWORK_ERRORS, {
            description: 'Network errors counter',
        });
        this._networkIo = this._meter.createObservableCounter(enum_1.METRIC_NAMES.NETWORK_IO, {
            description: 'Network transmit and received bytes',
        });
        this._processCpuTime = this._meter.createObservableCounter(enum_1.METRIC_NAMES.PROCESS_CPU_TIME, {
            description: 'Process Cpu time in seconds',
            unit: 's',
        });
        this._processCpuUtilization = this._meter.createObservableGauge(enum_1.METRIC_NAMES.PROCESS_CPU_UTILIZATION, {
            description: 'Process Cpu usage time 0-1',
        });
        this._processMemoryUsage = this._meter.createObservableGauge(enum_1.METRIC_NAMES.PROCESS_MEMORY_USAGE, {
            description: 'Process Memory usage in bytes',
        });
        this._meter.addBatchObservableCallback(async (observableResult) => {
            const cpuUsages = (0, common_1.getCpuUsageData)();
            const memoryUsages = (0, common_1.getMemoryData)();
            const processCpuUsages = (0, common_1.getProcessCpuUsageData)();
            const processMemoryUsages = (0, common_1.getProcessMemoryData)();
            const networkData = await (0, si_1.getNetworkData)();
            this._batchUpdateCpuUsages(observableResult, cpuUsages);
            this._batchUpdateMemUsages(observableResult, memoryUsages);
            this._batchUpdateProcessCpuUsages(observableResult, processCpuUsages);
            this._batchUpdateProcessMemUsage(observableResult, processMemoryUsages);
            this._batchUpdateNetworkData(observableResult, networkData);
        }, [
            this._cpuTime,
            this._cpuUtilization,
            this._memoryUsage,
            this._memoryUtilization,
            this._processCpuTime,
            this._processCpuUtilization,
            this._processMemoryUsage,
            this._networkDropped,
            this._networkErrors,
            this._networkIo,
        ]);
    }
    /**
     * Starts collecting metrics
     */
    start() {
        this._createMetrics();
    }
}
exports.HostMetrics = HostMetrics;
//# sourceMappingURL=metric.js.map