#!/usr/bin/env node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const node_fs_1 = require("node:fs");
const node_path_1 = require("node:path");
const semver_1 = require("semver");
const constants_1 = require("../constants");
const outputPath = (0, node_path_1.resolve)(process.cwd(), '../open-api/immich-openapi-specs.json');
const spec = JSON.parse((0, node_fs_1.readFileSync)(outputPath).toString());
const metadata = {};
const trackVersion = (version) => {
    if (!metadata[version]) {
        metadata[version] = {
            oldEndpoints: [],
            newEndpoints: [],
            oldProperties: [],
            newProperties: [],
        };
    }
    return metadata[version];
};
for (const [url, methods] of Object.entries(spec.paths)) {
    for (const [method, endpoint] of Object.entries(methods)) {
        const deprecatedAt = endpoint[constants_1.LIFECYCLE_EXTENSION]?.deprecatedAt;
        if (deprecatedAt) {
            trackVersion(deprecatedAt).oldEndpoints.push({ url, method, endpoint });
        }
        const addedAt = endpoint[constants_1.LIFECYCLE_EXTENSION]?.addedAt;
        if (addedAt) {
            trackVersion(addedAt).newEndpoints.push({ url, method, endpoint });
        }
    }
}
for (const [schemaName, schema] of Object.entries(spec.components?.schemas || {})) {
    for (const [propertyName, property] of Object.entries(schema.properties || {})) {
        const propertySchema = property;
        if (propertySchema.description?.startsWith(constants_1.DEPRECATED_IN_PREFIX)) {
            const deprecatedAt = propertySchema.description.replace(constants_1.DEPRECATED_IN_PREFIX, '').trim();
            trackVersion(deprecatedAt).oldProperties.push({ schema: schemaName, property: propertyName });
        }
        if (propertySchema.description?.startsWith(constants_1.ADDED_IN_PREFIX)) {
            const addedAt = propertySchema.description.replace(constants_1.ADDED_IN_PREFIX, '').trim();
            trackVersion(addedAt).newProperties.push({ schema: schemaName, property: propertyName });
        }
    }
}
const sortedVersions = Object.keys(metadata).sort((a, b) => {
    if (a === constants_1.NEXT_RELEASE) {
        return -1;
    }
    if (b === constants_1.NEXT_RELEASE) {
        return 1;
    }
    return new semver_1.SemVer(b).compare(new semver_1.SemVer(a));
});
for (const version of sortedVersions) {
    const { oldEndpoints, newEndpoints, oldProperties, newProperties } = metadata[version];
    console.log(`\nChanges in ${version}`);
    console.log('---------------------');
    for (const { url, method, endpoint } of oldEndpoints) {
        console.log(`- Deprecated ${method.toUpperCase()} ${url} (${endpoint.operationId})`);
    }
    for (const { url, method, endpoint } of newEndpoints) {
        console.log(`- Added      ${method.toUpperCase()} ${url} (${endpoint.operationId})`);
    }
    for (const { schema, property } of oldProperties) {
        console.log(`- Deprecated ${schema}.${property}`);
    }
    for (const { schema, property } of newProperties) {
        console.log(`- Added      ${schema}.${property}`);
    }
}
//# sourceMappingURL=lifecycle.js.map