"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.asStreamableFile = exports.sendFile = exports.ImmichFileResponse = void 0;
exports.getFileNameWithoutExtension = getFileNameWithoutExtension;
exports.getFilenameExtension = getFilenameExtension;
exports.getLivePhotoMotionFilename = getLivePhotoMotionFilename;
const common_1 = require("@nestjs/common");
const promises_1 = require("node:fs/promises");
const node_path_1 = require("node:path");
const node_util_1 = require("node:util");
const enum_1 = require("../enum");
const misc_1 = require("./misc");
function getFileNameWithoutExtension(path) {
    return (0, node_path_1.basename)(path, (0, node_path_1.extname)(path));
}
function getFilenameExtension(path) {
    return (0, node_path_1.extname)(path);
}
function getLivePhotoMotionFilename(stillName, motionName) {
    return getFileNameWithoutExtension(stillName) + (0, node_path_1.extname)(motionName);
}
class ImmichFileResponse {
    path;
    contentType;
    cacheControl;
    fileName;
    constructor(response) {
        Object.assign(this, response);
    }
}
exports.ImmichFileResponse = ImmichFileResponse;
const cacheControlHeaders = {
    [enum_1.CacheControl.PRIVATE_WITH_CACHE]: 'private, max-age=86400, no-transform',
    [enum_1.CacheControl.PRIVATE_WITHOUT_CACHE]: 'private, no-cache, no-transform',
    [enum_1.CacheControl.NONE]: null,
};
const sendFile = async (res, next, handler, logger) => {
    const _sendFile = (path, options) => (0, node_util_1.promisify)(res.sendFile).bind(res)(path, options);
    try {
        const file = await handler();
        const cacheControlHeader = cacheControlHeaders[file.cacheControl];
        if (cacheControlHeader) {
            res.set('Cache-Control', cacheControlHeader);
        }
        res.header('Content-Type', file.contentType);
        if (file.fileName) {
            res.header('Content-Disposition', `inline; filename*=UTF-8''${encodeURIComponent(file.fileName)}`);
        }
        const options = { dotfiles: 'allow' };
        if (!(0, node_path_1.isAbsolute)(file.path)) {
            options.root = process.cwd();
        }
        await (0, promises_1.access)(file.path, promises_1.constants.R_OK);
        return await _sendFile(file.path, options);
    }
    catch (error) {
        if ((0, misc_1.isConnectionAborted)(error) || res.headersSent) {
            return;
        }
        if (error instanceof common_1.HttpException === false) {
            logger.error(`Unable to send file: ${error.name}`, error.stack);
        }
        res.header('Cache-Control', 'none');
        next(error);
    }
};
exports.sendFile = sendFile;
const asStreamableFile = ({ stream, type, length }) => {
    return new common_1.StreamableFile(stream, { type, length });
};
exports.asStreamableFile = asStreamableFile;
//# sourceMappingURL=file.js.map