"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ASSET_CHECKSUM_CONSTRAINT = exports.removeUndefinedKeys = exports.unnest = exports.asVector = exports.anyUuid = exports.asUuid = exports.getKyselyConfig = exports.asPostgresConnectionConfig = void 0;
exports.toJson = toJson;
exports.withExif = withExif;
exports.withExifInner = withExifInner;
exports.withSmartSearch = withSmartSearch;
exports.withFaces = withFaces;
exports.withFiles = withFiles;
exports.withFacesAndPeople = withFacesAndPeople;
exports.hasPeople = hasPeople;
exports.hasTags = hasTags;
exports.withOwner = withOwner;
exports.withLibrary = withLibrary;
exports.withTags = withTags;
exports.truncatedDate = truncatedDate;
exports.withTagId = withTagId;
exports.searchAssetBuilder = searchAssetBuilder;
const kysely_1 = require("kysely");
const kysely_postgres_js_1 = require("kysely-postgres-js");
const postgres_1 = require("kysely/helpers/postgres");
const pg_connection_string_1 = require("pg-connection-string");
const postgres_2 = __importDefault(require("postgres"));
const database_1 = require("../database");
const isValidSsl = (ssl) => typeof ssl !== 'string' || ssl === 'require' || ssl === 'allow' || ssl === 'prefer' || ssl === 'verify-full';
const asPostgresConnectionConfig = (params) => {
    if (params.connectionType === 'parts') {
        return {
            host: params.host,
            port: params.port,
            username: params.username,
            password: params.password,
            database: params.database,
            ssl: undefined,
        };
    }
    const { host, port, user, password, database, ...rest } = (0, pg_connection_string_1.parse)(params.url);
    let ssl;
    if (rest.ssl) {
        if (!isValidSsl(rest.ssl)) {
            throw new Error(`Invalid ssl option: ${rest.ssl}`);
        }
        ssl = rest.ssl;
    }
    return {
        host: host ?? undefined,
        port: port ? Number(port) : undefined,
        username: user,
        password,
        database: database ?? undefined,
        ssl,
    };
};
exports.asPostgresConnectionConfig = asPostgresConnectionConfig;
const getKyselyConfig = (params, options = {}) => {
    const config = (0, exports.asPostgresConnectionConfig)(params);
    return {
        dialect: new kysely_postgres_js_1.PostgresJSDialect({
            postgres: (0, postgres_2.default)({
                onnotice: (notice) => {
                    if (notice['severity'] !== 'NOTICE') {
                        console.warn('Postgres notice:', notice);
                    }
                },
                max: 10,
                types: {
                    date: {
                        to: 1184,
                        from: [1082, 1114, 1184],
                        serialize: (x) => (x instanceof Date ? x.toISOString() : x),
                        parse: (x) => new Date(x),
                    },
                    bigint: {
                        to: 20,
                        from: [20, 1700],
                        parse: (value) => Number.parseInt(value),
                        serialize: (value) => value.toString(),
                    },
                },
                connection: {
                    TimeZone: 'UTC',
                },
                host: config.host,
                port: config.port,
                username: config.username,
                password: config.password,
                database: config.database,
                ssl: config.ssl,
                ...options,
            }),
        }),
        log(event) {
            if (event.level === 'error') {
                console.error('Query failed :', {
                    durationMs: event.queryDurationMillis,
                    error: event.error,
                    sql: event.query.sql,
                    params: event.query.parameters,
                });
            }
        },
    };
};
exports.getKyselyConfig = getKyselyConfig;
const asUuid = (id) => (0, kysely_1.sql) `${id}::uuid`;
exports.asUuid = asUuid;
const anyUuid = (ids) => (0, kysely_1.sql) `any(${`{${ids}}`}::uuid[])`;
exports.anyUuid = anyUuid;
const asVector = (embedding) => (0, kysely_1.sql) `${`[${embedding}]`}::vector`;
exports.asVector = asVector;
const unnest = (array) => (0, kysely_1.sql) `unnest(array[${kysely_1.sql.join(array)}]::text[])`;
exports.unnest = unnest;
const removeUndefinedKeys = (update, template) => {
    for (const key in update) {
        if (template[key] === undefined) {
            delete update[key];
        }
    }
    return update;
};
exports.removeUndefinedKeys = removeUndefinedKeys;
function toJson(eb, table) {
    return eb.fn.toJson(table);
}
exports.ASSET_CHECKSUM_CONSTRAINT = 'UQ_assets_owner_checksum';
function withExif(qb) {
    return qb
        .leftJoin('exif', 'assets.id', 'exif.assetId')
        .select((eb) => eb.fn.toJson(eb.table('exif')).$castTo().as('exifInfo'));
}
function withExifInner(qb) {
    return qb
        .innerJoin('exif', 'assets.id', 'exif.assetId')
        .select((eb) => eb.fn.toJson(eb.table('exif')).$castTo().as('exifInfo'));
}
function withSmartSearch(qb) {
    return qb
        .leftJoin('smart_search', 'assets.id', 'smart_search.assetId')
        .select((eb) => toJson(eb, 'smart_search').as('smartSearch'));
}
function withFaces(eb, withDeletedFace) {
    return (0, postgres_1.jsonArrayFrom)(eb
        .selectFrom('asset_faces')
        .selectAll('asset_faces')
        .whereRef('asset_faces.assetId', '=', 'assets.id')
        .$if(!withDeletedFace, (qb) => qb.where('asset_faces.deletedAt', 'is', null))).as('faces');
}
function withFiles(eb, type) {
    return (0, postgres_1.jsonArrayFrom)(eb
        .selectFrom('asset_files')
        .select(database_1.columns.assetFiles)
        .whereRef('asset_files.assetId', '=', 'assets.id')
        .$if(!!type, (qb) => qb.where('asset_files.type', '=', type))).as('files');
}
function withFacesAndPeople(eb, withDeletedFace) {
    return (0, postgres_1.jsonArrayFrom)(eb
        .selectFrom('asset_faces')
        .leftJoinLateral((eb) => eb.selectFrom('person').selectAll('person').whereRef('asset_faces.personId', '=', 'person.id').as('person'), (join) => join.onTrue())
        .selectAll('asset_faces')
        .select((eb) => eb.table('person').$castTo().as('person'))
        .whereRef('asset_faces.assetId', '=', 'assets.id')
        .$if(!withDeletedFace, (qb) => qb.where('asset_faces.deletedAt', 'is', null))).as('faces');
}
function hasPeople(qb, personIds) {
    return qb.innerJoin((eb) => eb
        .selectFrom('asset_faces')
        .select('assetId')
        .where('personId', '=', (0, exports.anyUuid)(personIds))
        .where('deletedAt', 'is', null)
        .groupBy('assetId')
        .having((eb) => eb.fn.count('personId').distinct(), '=', personIds.length)
        .as('has_people'), (join) => join.onRef('has_people.assetId', '=', 'assets.id'));
}
function hasTags(qb, tagIds) {
    return qb.innerJoin((eb) => eb
        .selectFrom('tag_asset')
        .select('assetsId')
        .innerJoin('tags_closure', 'tag_asset.tagsId', 'tags_closure.id_descendant')
        .where('tags_closure.id_ancestor', '=', (0, exports.anyUuid)(tagIds))
        .groupBy('assetsId')
        .having((eb) => eb.fn.count('tags_closure.id_ancestor').distinct(), '>=', tagIds.length)
        .as('has_tags'), (join) => join.onRef('has_tags.assetsId', '=', 'assets.id'));
}
function withOwner(eb) {
    return (0, postgres_1.jsonObjectFrom)(eb.selectFrom('users').select(database_1.columns.user).whereRef('users.id', '=', 'assets.ownerId')).as('owner');
}
function withLibrary(eb) {
    return (0, postgres_1.jsonObjectFrom)(eb.selectFrom('libraries').selectAll('libraries').whereRef('libraries.id', '=', 'assets.libraryId')).as('library');
}
function withTags(eb) {
    return (0, postgres_1.jsonArrayFrom)(eb
        .selectFrom('tags')
        .select(database_1.columns.tag)
        .innerJoin('tag_asset', 'tags.id', 'tag_asset.tagsId')
        .whereRef('assets.id', '=', 'tag_asset.assetsId')).as('tags');
}
function truncatedDate(size) {
    return (0, kysely_1.sql) `date_trunc(${size}, "localDateTime" at time zone 'UTC') at time zone 'UTC'`;
}
function withTagId(qb, tagId) {
    return qb.where((eb) => eb.exists(eb
        .selectFrom('tags_closure')
        .innerJoin('tag_asset', 'tag_asset.tagsId', 'tags_closure.id_descendant')
        .whereRef('tag_asset.assetsId', '=', 'assets.id')
        .where('tags_closure.id_ancestor', '=', tagId)));
}
const joinDeduplicationPlugin = new kysely_1.DeduplicateJoinsPlugin();
function searchAssetBuilder(kysely, options) {
    options.isArchived ??= options.withArchived ? undefined : false;
    options.withDeleted ||= !!(options.trashedAfter || options.trashedBefore || options.isOffline);
    return kysely
        .withPlugin(joinDeduplicationPlugin)
        .selectFrom('assets')
        .selectAll('assets')
        .$if(!!options.tagIds && options.tagIds.length > 0, (qb) => hasTags(qb, options.tagIds))
        .$if(!!options.personIds && options.personIds.length > 0, (qb) => hasPeople(qb, options.personIds))
        .$if(!!options.createdBefore, (qb) => qb.where('assets.createdAt', '<=', options.createdBefore))
        .$if(!!options.createdAfter, (qb) => qb.where('assets.createdAt', '>=', options.createdAfter))
        .$if(!!options.updatedBefore, (qb) => qb.where('assets.updatedAt', '<=', options.updatedBefore))
        .$if(!!options.updatedAfter, (qb) => qb.where('assets.updatedAt', '>=', options.updatedAfter))
        .$if(!!options.trashedBefore, (qb) => qb.where('assets.deletedAt', '<=', options.trashedBefore))
        .$if(!!options.trashedAfter, (qb) => qb.where('assets.deletedAt', '>=', options.trashedAfter))
        .$if(!!options.takenBefore, (qb) => qb.where('assets.fileCreatedAt', '<=', options.takenBefore))
        .$if(!!options.takenAfter, (qb) => qb.where('assets.fileCreatedAt', '>=', options.takenAfter))
        .$if(options.city !== undefined, (qb) => qb
        .innerJoin('exif', 'assets.id', 'exif.assetId')
        .where('exif.city', options.city === null ? 'is' : '=', options.city))
        .$if(options.state !== undefined, (qb) => qb
        .innerJoin('exif', 'assets.id', 'exif.assetId')
        .where('exif.state', options.state === null ? 'is' : '=', options.state))
        .$if(options.country !== undefined, (qb) => qb
        .innerJoin('exif', 'assets.id', 'exif.assetId')
        .where('exif.country', options.country === null ? 'is' : '=', options.country))
        .$if(options.make !== undefined, (qb) => qb
        .innerJoin('exif', 'assets.id', 'exif.assetId')
        .where('exif.make', options.make === null ? 'is' : '=', options.make))
        .$if(options.model !== undefined, (qb) => qb
        .innerJoin('exif', 'assets.id', 'exif.assetId')
        .where('exif.model', options.model === null ? 'is' : '=', options.model))
        .$if(options.lensModel !== undefined, (qb) => qb
        .innerJoin('exif', 'assets.id', 'exif.assetId')
        .where('exif.lensModel', options.lensModel === null ? 'is' : '=', options.lensModel))
        .$if(options.rating !== undefined, (qb) => qb
        .innerJoin('exif', 'assets.id', 'exif.assetId')
        .where('exif.rating', options.rating === null ? 'is' : '=', options.rating))
        .$if(!!options.checksum, (qb) => qb.where('assets.checksum', '=', options.checksum))
        .$if(!!options.deviceAssetId, (qb) => qb.where('assets.deviceAssetId', '=', options.deviceAssetId))
        .$if(!!options.deviceId, (qb) => qb.where('assets.deviceId', '=', options.deviceId))
        .$if(!!options.id, (qb) => qb.where('assets.id', '=', (0, exports.asUuid)(options.id)))
        .$if(!!options.libraryId, (qb) => qb.where('assets.libraryId', '=', (0, exports.asUuid)(options.libraryId)))
        .$if(!!options.userIds, (qb) => qb.where('assets.ownerId', '=', (0, exports.anyUuid)(options.userIds)))
        .$if(!!options.encodedVideoPath, (qb) => qb.where('assets.encodedVideoPath', '=', options.encodedVideoPath))
        .$if(!!options.originalPath, (qb) => qb.where((0, kysely_1.sql) `f_unaccent(assets."originalPath")`, 'ilike', (0, kysely_1.sql) `'%' || f_unaccent(${options.originalPath}) || '%'`))
        .$if(!!options.originalFileName, (qb) => qb.where((0, kysely_1.sql) `f_unaccent(assets."originalFileName")`, 'ilike', (0, kysely_1.sql) `'%' || f_unaccent(${options.originalFileName}) || '%'`))
        .$if(!!options.description, (qb) => qb
        .innerJoin('exif', 'assets.id', 'exif.assetId')
        .where((0, kysely_1.sql) `f_unaccent(exif.description)`, 'ilike', (0, kysely_1.sql) `'%' || f_unaccent(${options.description}) || '%'`))
        .$if(!!options.type, (qb) => qb.where('assets.type', '=', options.type))
        .$if(options.isFavorite !== undefined, (qb) => qb.where('assets.isFavorite', '=', options.isFavorite))
        .$if(options.isOffline !== undefined, (qb) => qb.where('assets.isOffline', '=', options.isOffline))
        .$if(options.isVisible !== undefined, (qb) => qb.where('assets.isVisible', '=', options.isVisible))
        .$if(options.isArchived !== undefined, (qb) => qb.where('assets.isArchived', '=', options.isArchived))
        .$if(options.isEncoded !== undefined, (qb) => qb.where('assets.encodedVideoPath', options.isEncoded ? 'is not' : 'is', null))
        .$if(options.isMotion !== undefined, (qb) => qb.where('assets.livePhotoVideoId', options.isMotion ? 'is not' : 'is', null))
        .$if(!!options.isNotInAlbum, (qb) => qb.where((eb) => eb.not(eb.exists((eb) => eb.selectFrom('albums_assets_assets').whereRef('assetsId', '=', 'assets.id')))))
        .$if(!!options.withExif, withExifInner)
        .$if(!!(options.withFaces || options.withPeople || options.personIds), (qb) => qb.select(withFacesAndPeople))
        .$if(!options.withDeleted, (qb) => qb.where('assets.deletedAt', 'is', null));
}
//# sourceMappingURL=database.js.map