"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.asRequest = exports.onAfterUnlink = exports.onBeforeUnlink = exports.onBeforeLink = exports.getMyPartnerIds = exports.removeAssets = exports.addAssets = exports.getAssetFiles = exports.getAssetFile = void 0;
exports.mapToUploadFile = mapToUploadFile;
const common_1 = require("@nestjs/common");
const storage_core_1 = require("../cores/storage.core");
const asset_ids_response_dto_1 = require("../dtos/asset-ids.response.dto");
const enum_1 = require("../enum");
const access_1 = require("./access");
const getAssetFile = (files, type) => {
    return files.find((file) => file.type === type);
};
exports.getAssetFile = getAssetFile;
const getAssetFiles = (files) => ({
    fullsizeFile: (0, exports.getAssetFile)(files, enum_1.AssetFileType.FULLSIZE),
    previewFile: (0, exports.getAssetFile)(files, enum_1.AssetFileType.PREVIEW),
    thumbnailFile: (0, exports.getAssetFile)(files, enum_1.AssetFileType.THUMBNAIL),
});
exports.getAssetFiles = getAssetFiles;
const addAssets = async (auth, repositories, dto) => {
    const { access, bulk } = repositories;
    const existingAssetIds = await bulk.getAssetIds(dto.parentId, dto.assetIds);
    const notPresentAssetIds = dto.assetIds.filter((id) => !existingAssetIds.has(id));
    const allowedAssetIds = await (0, access_1.checkAccess)(access, {
        auth,
        permission: enum_1.Permission.ASSET_SHARE,
        ids: notPresentAssetIds,
    });
    const results = [];
    for (const assetId of dto.assetIds) {
        const hasAsset = existingAssetIds.has(assetId);
        if (hasAsset) {
            results.push({ id: assetId, success: false, error: asset_ids_response_dto_1.BulkIdErrorReason.DUPLICATE });
            continue;
        }
        const hasAccess = allowedAssetIds.has(assetId);
        if (!hasAccess) {
            results.push({ id: assetId, success: false, error: asset_ids_response_dto_1.BulkIdErrorReason.NO_PERMISSION });
            continue;
        }
        existingAssetIds.add(assetId);
        results.push({ id: assetId, success: true });
    }
    const newAssetIds = results.filter(({ success }) => success).map(({ id }) => id);
    if (newAssetIds.length > 0) {
        await bulk.addAssetIds(dto.parentId, newAssetIds);
    }
    return results;
};
exports.addAssets = addAssets;
const removeAssets = async (auth, repositories, dto) => {
    const { access, bulk } = repositories;
    const canAlwaysRemove = await (0, access_1.checkAccess)(access, { auth, permission: dto.canAlwaysRemove, ids: [dto.parentId] });
    const existingAssetIds = await bulk.getAssetIds(dto.parentId, dto.assetIds);
    const allowedAssetIds = canAlwaysRemove.has(dto.parentId)
        ? existingAssetIds
        : await (0, access_1.checkAccess)(access, { auth, permission: enum_1.Permission.ASSET_SHARE, ids: existingAssetIds });
    const results = [];
    for (const assetId of dto.assetIds) {
        const hasAsset = existingAssetIds.has(assetId);
        if (!hasAsset) {
            results.push({ id: assetId, success: false, error: asset_ids_response_dto_1.BulkIdErrorReason.NOT_FOUND });
            continue;
        }
        const hasAccess = allowedAssetIds.has(assetId);
        if (!hasAccess) {
            results.push({ id: assetId, success: false, error: asset_ids_response_dto_1.BulkIdErrorReason.NO_PERMISSION });
            continue;
        }
        existingAssetIds.delete(assetId);
        results.push({ id: assetId, success: true });
    }
    const removedIds = results.filter(({ success }) => success).map(({ id }) => id);
    if (removedIds.length > 0) {
        await bulk.removeAssetIds(dto.parentId, removedIds);
    }
    return results;
};
exports.removeAssets = removeAssets;
const getMyPartnerIds = async ({ userId, repository, timelineEnabled }) => {
    const partnerIds = new Set();
    const partners = await repository.getAll(userId);
    for (const partner of partners) {
        if (!partner.sharedBy || !partner.sharedWith) {
            continue;
        }
        if (partner.sharedWithId !== userId) {
            continue;
        }
        if (timelineEnabled && !partner.inTimeline) {
            continue;
        }
        partnerIds.add(partner.sharedById);
    }
    return [...partnerIds];
};
exports.getMyPartnerIds = getMyPartnerIds;
const onBeforeLink = async ({ asset: assetRepository, event: eventRepository }, { userId, livePhotoVideoId }) => {
    const motionAsset = await assetRepository.getById(livePhotoVideoId);
    if (!motionAsset) {
        throw new common_1.BadRequestException('Live photo video not found');
    }
    if (motionAsset.type !== enum_1.AssetType.VIDEO) {
        throw new common_1.BadRequestException('Live photo video must be a video');
    }
    if (motionAsset.ownerId !== userId) {
        throw new common_1.BadRequestException('Live photo video does not belong to the user');
    }
    if (motionAsset?.isVisible) {
        await assetRepository.update({ id: livePhotoVideoId, isVisible: false });
        await eventRepository.emit('asset.hide', { assetId: motionAsset.id, userId });
    }
};
exports.onBeforeLink = onBeforeLink;
const onBeforeUnlink = async ({ asset: assetRepository }, { livePhotoVideoId }) => {
    const motion = await assetRepository.getById(livePhotoVideoId);
    if (!motion) {
        return null;
    }
    if (storage_core_1.StorageCore.isAndroidMotionPath(motion.originalPath)) {
        throw new common_1.BadRequestException('Cannot unlink Android motion photos');
    }
    return motion;
};
exports.onBeforeUnlink = onBeforeUnlink;
const onAfterUnlink = async ({ asset: assetRepository, event: eventRepository }, { userId, livePhotoVideoId }) => {
    await assetRepository.update({ id: livePhotoVideoId, isVisible: true });
    await eventRepository.emit('asset.show', { assetId: livePhotoVideoId, userId });
};
exports.onAfterUnlink = onAfterUnlink;
function mapToUploadFile(file) {
    return {
        uuid: file.uuid,
        checksum: file.checksum,
        originalPath: file.path,
        originalName: Buffer.from(file.originalname, 'latin1').toString('utf8'),
        size: file.size,
    };
}
const asRequest = (request, file) => {
    return {
        auth: request.user || null,
        fieldName: file.fieldname,
        file: mapToUploadFile(file),
    };
};
exports.asRequest = asRequest;
//# sourceMappingURL=asset.util.js.map