"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkAccess = exports.requireAccess = exports.requireUploadAccess = exports.isGranted = void 0;
const common_1 = require("@nestjs/common");
const enum_1 = require("../enum");
const set_1 = require("./set");
const isGranted = ({ requested, current }) => {
    if (current.includes(enum_1.Permission.ALL)) {
        return true;
    }
    return (0, set_1.setIsSuperset)(new Set(current), new Set(requested));
};
exports.isGranted = isGranted;
const requireUploadAccess = (auth) => {
    if (!auth || (auth.sharedLink && !auth.sharedLink.allowUpload)) {
        throw new common_1.UnauthorizedException();
    }
    return auth;
};
exports.requireUploadAccess = requireUploadAccess;
const requireAccess = async (access, request) => {
    const allowedIds = await (0, exports.checkAccess)(access, request);
    if (!(0, set_1.setIsEqual)(new Set(request.ids), allowedIds)) {
        throw new common_1.BadRequestException(`Not found or no ${request.permission} access`);
    }
};
exports.requireAccess = requireAccess;
const checkAccess = async (access, { ids, auth, permission }) => {
    const idSet = Array.isArray(ids) ? new Set(ids) : ids;
    if (idSet.size === 0) {
        return new Set();
    }
    return auth.sharedLink
        ? checkSharedLinkAccess(access, { sharedLink: auth.sharedLink, permission, ids: idSet })
        : checkOtherAccess(access, { auth, permission, ids: idSet });
};
exports.checkAccess = checkAccess;
const checkSharedLinkAccess = async (access, request) => {
    const { sharedLink, permission, ids } = request;
    const sharedLinkId = sharedLink.id;
    switch (permission) {
        case enum_1.Permission.ASSET_READ: {
            return await access.asset.checkSharedLinkAccess(sharedLinkId, ids);
        }
        case enum_1.Permission.ASSET_VIEW: {
            return await access.asset.checkSharedLinkAccess(sharedLinkId, ids);
        }
        case enum_1.Permission.ASSET_DOWNLOAD: {
            return sharedLink.allowDownload ? await access.asset.checkSharedLinkAccess(sharedLinkId, ids) : new Set();
        }
        case enum_1.Permission.ASSET_UPLOAD: {
            return sharedLink.allowUpload ? ids : new Set();
        }
        case enum_1.Permission.ASSET_SHARE: {
            return await access.asset.checkOwnerAccess(sharedLink.userId, ids);
        }
        case enum_1.Permission.ALBUM_READ: {
            return await access.album.checkSharedLinkAccess(sharedLinkId, ids);
        }
        case enum_1.Permission.ALBUM_DOWNLOAD: {
            return sharedLink.allowDownload ? await access.album.checkSharedLinkAccess(sharedLinkId, ids) : new Set();
        }
        case enum_1.Permission.ALBUM_ADD_ASSET: {
            return sharedLink.allowUpload ? await access.album.checkSharedLinkAccess(sharedLinkId, ids) : new Set();
        }
        default: {
            return new Set();
        }
    }
};
const checkOtherAccess = async (access, request) => {
    const { auth, permission, ids } = request;
    switch (permission) {
        case enum_1.Permission.ACTIVITY_CREATE: {
            return await access.activity.checkCreateAccess(auth.user.id, ids);
        }
        case enum_1.Permission.ACTIVITY_DELETE: {
            const isOwner = await access.activity.checkOwnerAccess(auth.user.id, ids);
            const isAlbumOwner = await access.activity.checkAlbumOwnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            return (0, set_1.setUnion)(isOwner, isAlbumOwner);
        }
        case enum_1.Permission.ASSET_READ: {
            const isOwner = await access.asset.checkOwnerAccess(auth.user.id, ids);
            const isAlbum = await access.asset.checkAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            const isPartner = await access.asset.checkPartnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner, isAlbum));
            return (0, set_1.setUnion)(isOwner, isAlbum, isPartner);
        }
        case enum_1.Permission.ASSET_SHARE: {
            const isOwner = await access.asset.checkOwnerAccess(auth.user.id, ids);
            const isPartner = await access.asset.checkPartnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            return (0, set_1.setUnion)(isOwner, isPartner);
        }
        case enum_1.Permission.ASSET_VIEW: {
            const isOwner = await access.asset.checkOwnerAccess(auth.user.id, ids);
            const isAlbum = await access.asset.checkAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            const isPartner = await access.asset.checkPartnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner, isAlbum));
            return (0, set_1.setUnion)(isOwner, isAlbum, isPartner);
        }
        case enum_1.Permission.ASSET_DOWNLOAD: {
            const isOwner = await access.asset.checkOwnerAccess(auth.user.id, ids);
            const isAlbum = await access.asset.checkAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            const isPartner = await access.asset.checkPartnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner, isAlbum));
            return (0, set_1.setUnion)(isOwner, isAlbum, isPartner);
        }
        case enum_1.Permission.ASSET_UPDATE: {
            return await access.asset.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.ASSET_DELETE: {
            return await access.asset.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.ALBUM_READ: {
            const isOwner = await access.album.checkOwnerAccess(auth.user.id, ids);
            const isShared = await access.album.checkSharedAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner), enum_1.AlbumUserRole.VIEWER);
            return (0, set_1.setUnion)(isOwner, isShared);
        }
        case enum_1.Permission.ALBUM_ADD_ASSET: {
            const isOwner = await access.album.checkOwnerAccess(auth.user.id, ids);
            const isShared = await access.album.checkSharedAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner), enum_1.AlbumUserRole.EDITOR);
            return (0, set_1.setUnion)(isOwner, isShared);
        }
        case enum_1.Permission.ALBUM_UPDATE: {
            return await access.album.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.ALBUM_DELETE: {
            return await access.album.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.ALBUM_SHARE: {
            return await access.album.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.ALBUM_DOWNLOAD: {
            const isOwner = await access.album.checkOwnerAccess(auth.user.id, ids);
            const isShared = await access.album.checkSharedAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner), enum_1.AlbumUserRole.VIEWER);
            return (0, set_1.setUnion)(isOwner, isShared);
        }
        case enum_1.Permission.ALBUM_REMOVE_ASSET: {
            const isOwner = await access.album.checkOwnerAccess(auth.user.id, ids);
            const isShared = await access.album.checkSharedAlbumAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner), enum_1.AlbumUserRole.EDITOR);
            return (0, set_1.setUnion)(isOwner, isShared);
        }
        case enum_1.Permission.ASSET_UPLOAD: {
            return ids.has(auth.user.id) ? new Set([auth.user.id]) : new Set();
        }
        case enum_1.Permission.ARCHIVE_READ: {
            return ids.has(auth.user.id) ? new Set([auth.user.id]) : new Set();
        }
        case enum_1.Permission.AUTH_DEVICE_DELETE: {
            return await access.authDevice.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.FACE_DELETE: {
            return access.person.checkFaceOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.TAG_ASSET:
        case enum_1.Permission.TAG_READ:
        case enum_1.Permission.TAG_UPDATE:
        case enum_1.Permission.TAG_DELETE: {
            return await access.tag.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.TIMELINE_READ: {
            const isOwner = ids.has(auth.user.id) ? new Set([auth.user.id]) : new Set();
            const isPartner = await access.timeline.checkPartnerAccess(auth.user.id, (0, set_1.setDifference)(ids, isOwner));
            return (0, set_1.setUnion)(isOwner, isPartner);
        }
        case enum_1.Permission.TIMELINE_DOWNLOAD: {
            return ids.has(auth.user.id) ? new Set([auth.user.id]) : new Set();
        }
        case enum_1.Permission.MEMORY_READ: {
            return access.memory.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.MEMORY_UPDATE: {
            return access.memory.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.MEMORY_DELETE: {
            return access.memory.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.PERSON_READ: {
            return await access.person.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.PERSON_UPDATE: {
            return await access.person.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.PERSON_MERGE: {
            return await access.person.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.PERSON_CREATE: {
            return access.person.checkFaceOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.PERSON_REASSIGN: {
            return access.person.checkFaceOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.PARTNER_UPDATE: {
            return await access.partner.checkUpdateAccess(auth.user.id, ids);
        }
        case enum_1.Permission.STACK_READ: {
            return access.stack.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.STACK_UPDATE: {
            return access.stack.checkOwnerAccess(auth.user.id, ids);
        }
        case enum_1.Permission.STACK_DELETE: {
            return access.stack.checkOwnerAccess(auth.user.id, ids);
        }
        default: {
            return new Set();
        }
    }
};
//# sourceMappingURL=access.js.map