"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.asColumnList = exports.asColumnComment = exports.getColumnModifiers = exports.getColumnType = exports.isDefaultEqual = exports.compare = exports.haveEqualColumns = exports.setIsEqual = exports.fromColumnValue = exports.parseTriggerType = exports.hasMask = exports.sha1 = exports.asOptions = exports.asKey = exports.asSnakeCase = exports.asMetadataKey = void 0;
const node_crypto_1 = require("node:crypto");
const asMetadataKey = (name) => `sql-tools:${name}`;
exports.asMetadataKey = asMetadataKey;
const asSnakeCase = (name) => name.replaceAll(/([a-z])([A-Z])/g, '$1_$2').toLowerCase();
exports.asSnakeCase = asSnakeCase;
const asKey = (prefix, tableName, values) => (prefix + (0, exports.sha1)(`${tableName}_${values.toSorted().join('_')}`)).slice(0, 30);
exports.asKey = asKey;
const asOptions = (options) => {
    if (typeof options === 'string') {
        return { name: options };
    }
    return options;
};
exports.asOptions = asOptions;
const sha1 = (value) => (0, node_crypto_1.createHash)('sha1').update(value).digest('hex');
exports.sha1 = sha1;
const hasMask = (input, mask) => (input & mask) === mask;
exports.hasMask = hasMask;
const parseTriggerType = (type) => {
    const scope = (0, exports.hasMask)(type, 1 << 0) ? 'row' : 'statement';
    let timing = 'after';
    const timingMasks = [
        { mask: 1 << 1, value: 'before' },
        { mask: 1 << 6, value: 'instead of' },
    ];
    for (const { mask, value } of timingMasks) {
        if ((0, exports.hasMask)(type, mask)) {
            timing = value;
            break;
        }
    }
    const actions = [];
    const actionMasks = [
        { mask: 1 << 2, value: 'insert' },
        { mask: 1 << 3, value: 'delete' },
        { mask: 1 << 4, value: 'update' },
        { mask: 1 << 5, value: 'truncate' },
    ];
    for (const { mask, value } of actionMasks) {
        if ((0, exports.hasMask)(type, mask)) {
            actions.push(value);
            break;
        }
    }
    if (actions.length === 0) {
        throw new Error(`Unable to parse trigger type ${type}`);
    }
    return { actions, timing, scope };
};
exports.parseTriggerType = parseTriggerType;
const fromColumnValue = (columnValue) => {
    if (columnValue === undefined) {
        return;
    }
    if (typeof columnValue === 'function') {
        return columnValue();
    }
    const value = columnValue;
    if (value === null) {
        return value;
    }
    if (typeof value === 'number') {
        return String(value);
    }
    if (typeof value === 'boolean') {
        return value ? 'true' : 'false';
    }
    if (value instanceof Date) {
        return `'${value.toISOString()}'`;
    }
    return `'${String(value)}'`;
};
exports.fromColumnValue = fromColumnValue;
const setIsEqual = (source, target) => source.size === target.size && [...source].every((x) => target.has(x));
exports.setIsEqual = setIsEqual;
const haveEqualColumns = (sourceColumns, targetColumns) => {
    return (0, exports.setIsEqual)(new Set(sourceColumns ?? []), new Set(targetColumns ?? []));
};
exports.haveEqualColumns = haveEqualColumns;
const compare = (sources, targets, options, comparer) => {
    options = options || {};
    const sourceMap = Object.fromEntries(sources.map((table) => [table.name, table]));
    const targetMap = Object.fromEntries(targets.map((table) => [table.name, table]));
    const items = [];
    const keys = new Set([...Object.keys(sourceMap), ...Object.keys(targetMap)]);
    for (const key of keys) {
        const source = sourceMap[key];
        const target = targetMap[key];
        if (isIgnored(source, target, options)) {
            continue;
        }
        if (isSynchronizeDisabled(source, target)) {
            continue;
        }
        if (source && !target) {
            items.push(...comparer.onMissing(source));
        }
        else if (!source && target) {
            items.push(...comparer.onExtra(target));
        }
        else {
            items.push(...comparer.onCompare(source, target));
        }
    }
    return items;
};
exports.compare = compare;
const isIgnored = (source, target, options) => {
    return (options.ignoreExtra && !source) || (options.ignoreMissing && !target);
};
const isSynchronizeDisabled = (source, target) => {
    return source?.synchronize === false || target?.synchronize === false;
};
const isDefaultEqual = (source, target) => {
    if (source.default === target.default) {
        return true;
    }
    if (source.default === undefined || target.default === undefined) {
        return false;
    }
    if (withTypeCast(source.default, (0, exports.getColumnType)(source)) === target.default ||
        source.default === withTypeCast(target.default, (0, exports.getColumnType)(target))) {
        return true;
    }
    return false;
};
exports.isDefaultEqual = isDefaultEqual;
const getColumnType = (column) => {
    let type = column.enumName || column.type;
    if (column.isArray) {
        type += `[${column.length ?? ''}]`;
    }
    else if (column.length !== undefined) {
        type += `(${column.length})`;
    }
    return type;
};
exports.getColumnType = getColumnType;
const withTypeCast = (value, type) => {
    if (!value.startsWith(`'`)) {
        value = `'${value}'`;
    }
    return `${value}::${type}`;
};
const getColumnModifiers = (column) => {
    const modifiers = [];
    if (!column.nullable) {
        modifiers.push('NOT NULL');
    }
    if (column.default) {
        modifiers.push(`DEFAULT ${column.default}`);
    }
    if (column.identity) {
        modifiers.push(`GENERATED ALWAYS AS IDENTITY`);
    }
    return modifiers.length === 0 ? '' : ' ' + modifiers.join(' ');
};
exports.getColumnModifiers = getColumnModifiers;
const asColumnComment = (tableName, columnName, comment) => {
    return `COMMENT ON COLUMN "${tableName}"."${columnName}" IS '${comment}';`;
};
exports.asColumnComment = asColumnComment;
const asColumnList = (columns) => columns.map((column) => `"${column}"`).join(', ');
exports.asColumnList = asColumnList;
//# sourceMappingURL=helpers.js.map