"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.schemaFromDatabase = void 0;
const kysely_1 = require("kysely");
const kysely_postgres_js_1 = require("kysely-postgres-js");
const postgres_1 = require("kysely/helpers/postgres");
const helpers_1 = require("../helpers");
const types_1 = require("../types");
const schemaFromDatabase = async (postgres, options = {}) => {
    const db = createDatabaseClient(postgres);
    const warnings = [];
    const warn = (message) => {
        warnings.push(message);
    };
    const schemaName = options.schemaName || 'public';
    const tablesMap = {};
    const [databaseName, tables, columns, indexes, constraints, enums, routines, extensions, triggers, parameters, comments,] = await Promise.all([
        getDatabaseName(db),
        getTables(db, schemaName),
        getTableColumns(db, schemaName),
        getTableIndexes(db, schemaName),
        getTableConstraints(db, schemaName),
        getUserDefinedEnums(db, schemaName),
        getRoutines(db, schemaName),
        getExtensions(db),
        getTriggers(db, schemaName),
        getParameters(db),
        getObjectComments(db),
    ]);
    const schemaEnums = [];
    const schemaFunctions = [];
    const schemaExtensions = [];
    const schemaParameters = [];
    const enumMap = Object.fromEntries(enums.map((e) => [e.name, e.values]));
    for (const { name } of extensions) {
        schemaExtensions.push({ name, synchronize: true });
    }
    for (const { name, values } of enums) {
        schemaEnums.push({ name, values, synchronize: true });
    }
    for (const parameter of parameters) {
        schemaParameters.push({
            name: parameter.name,
            value: parameter.value,
            databaseName,
            scope: parameter.scope,
            synchronize: true,
        });
    }
    for (const { name, expression } of routines) {
        schemaFunctions.push({
            name,
            expression,
            synchronize: true,
        });
    }
    for (const table of tables) {
        const tableName = table.table_name;
        if (tablesMap[tableName]) {
            continue;
        }
        tablesMap[table.table_name] = {
            name: table.table_name,
            columns: [],
            indexes: [],
            triggers: [],
            constraints: [],
            synchronize: true,
        };
    }
    for (const column of columns) {
        const table = tablesMap[column.table_name];
        if (!table) {
            continue;
        }
        const columnName = column.column_name;
        const item = {
            type: column.data_type,
            name: columnName,
            tableName: column.table_name,
            nullable: column.is_nullable === 'YES',
            isArray: column.array_type !== null,
            numericPrecision: column.numeric_precision ?? undefined,
            numericScale: column.numeric_scale ?? undefined,
            length: column.character_maximum_length ?? undefined,
            default: column.column_default ?? undefined,
            synchronize: true,
        };
        const columnLabel = `${table.name}.${columnName}`;
        switch (column.data_type) {
            case 'ARRAY': {
                if (!column.array_type) {
                    warn(`Unable to find type for ${columnLabel} (ARRAY)`);
                    continue;
                }
                item.type = column.array_type;
                break;
            }
            case 'USER-DEFINED': {
                if (!enumMap[column.udt_name]) {
                    warn(`Unable to find type for ${columnLabel} (ENUM)`);
                    continue;
                }
                item.type = 'enum';
                item.enumName = column.udt_name;
                break;
            }
        }
        table.columns.push(item);
    }
    for (const index of indexes) {
        const table = tablesMap[index.table_name];
        if (!table) {
            continue;
        }
        const indexName = index.index_name;
        table.indexes.push({
            name: indexName,
            tableName: index.table_name,
            columnNames: index.column_names ?? undefined,
            expression: index.expression ?? undefined,
            using: index.using,
            where: index.where ?? undefined,
            unique: index.unique,
            synchronize: true,
        });
    }
    for (const constraint of constraints) {
        const table = tablesMap[constraint.table_name];
        if (!table) {
            continue;
        }
        const constraintName = constraint.constraint_name;
        switch (constraint.constraint_type) {
            case 'p': {
                if (!constraint.column_names) {
                    warn(`Skipping CONSTRAINT "${constraintName}", no columns found`);
                    continue;
                }
                table.constraints.push({
                    type: types_1.DatabaseConstraintType.PRIMARY_KEY,
                    name: constraintName,
                    tableName: constraint.table_name,
                    columnNames: constraint.column_names,
                    synchronize: true,
                });
                break;
            }
            case 'f': {
                if (!constraint.column_names || !constraint.reference_table_name || !constraint.reference_column_names) {
                    warn(`Skipping CONSTRAINT "${constraintName}", missing either columns, referenced table, or referenced columns,`);
                    continue;
                }
                table.constraints.push({
                    type: types_1.DatabaseConstraintType.FOREIGN_KEY,
                    name: constraintName,
                    tableName: constraint.table_name,
                    columnNames: constraint.column_names,
                    referenceTableName: constraint.reference_table_name,
                    referenceColumnNames: constraint.reference_column_names,
                    onUpdate: asDatabaseAction(constraint.update_action),
                    onDelete: asDatabaseAction(constraint.delete_action),
                    synchronize: true,
                });
                break;
            }
            case 'u': {
                table.constraints.push({
                    type: types_1.DatabaseConstraintType.UNIQUE,
                    name: constraintName,
                    tableName: constraint.table_name,
                    columnNames: constraint.column_names,
                    synchronize: true,
                });
                break;
            }
            case 'c': {
                table.constraints.push({
                    type: types_1.DatabaseConstraintType.CHECK,
                    name: constraint.constraint_name,
                    tableName: constraint.table_name,
                    expression: constraint.expression.replace('CHECK ', ''),
                    synchronize: true,
                });
                break;
            }
        }
    }
    for (const trigger of triggers) {
        const table = tablesMap[trigger.table_name];
        if (!table) {
            continue;
        }
        table.triggers.push({
            name: trigger.name,
            tableName: trigger.table_name,
            functionName: trigger.function_name,
            referencingNewTableAs: trigger.referencing_new_table_as ?? undefined,
            referencingOldTableAs: trigger.referencing_old_table_as ?? undefined,
            when: trigger.when_expression,
            synchronize: true,
            ...(0, helpers_1.parseTriggerType)(trigger.type),
        });
    }
    for (const comment of comments) {
        if (comment.object_type === 'r') {
            const table = tablesMap[comment.object_name];
            if (!table) {
                continue;
            }
            if (comment.column_name) {
                const column = table.columns.find(({ name }) => name === comment.column_name);
                if (column) {
                    column.comment = comment.value;
                }
            }
        }
    }
    await db.destroy();
    return {
        name: databaseName,
        schemaName,
        parameters: schemaParameters,
        functions: schemaFunctions,
        enums: schemaEnums,
        extensions: schemaExtensions,
        tables: Object.values(tablesMap),
        warnings,
    };
};
exports.schemaFromDatabase = schemaFromDatabase;
const createDatabaseClient = (postgres) => new kysely_1.Kysely({ dialect: new kysely_postgres_js_1.PostgresJSDialect({ postgres }) });
const asDatabaseAction = (action) => {
    switch (action) {
        case 'a': {
            return types_1.DatabaseActionType.NO_ACTION;
        }
        case 'c': {
            return types_1.DatabaseActionType.CASCADE;
        }
        case 'r': {
            return types_1.DatabaseActionType.RESTRICT;
        }
        case 'n': {
            return types_1.DatabaseActionType.SET_NULL;
        }
        case 'd': {
            return types_1.DatabaseActionType.SET_DEFAULT;
        }
        default: {
            return types_1.DatabaseActionType.NO_ACTION;
        }
    }
};
const getDatabaseName = async (db) => {
    const result = (await (0, kysely_1.sql) `SELECT current_database() as name`.execute(db));
    return result.rows[0].name;
};
const getTables = (db, schemaName) => {
    return db
        .selectFrom('information_schema.tables')
        .where('table_schema', '=', schemaName)
        .where('table_type', '=', kysely_1.sql.lit('BASE TABLE'))
        .selectAll()
        .execute();
};
const getTableColumns = (db, schemaName) => {
    return db
        .selectFrom('information_schema.columns as c')
        .leftJoin('information_schema.element_types as o', (join) => join
        .onRef('c.table_catalog', '=', 'o.object_catalog')
        .onRef('c.table_schema', '=', 'o.object_schema')
        .onRef('c.table_name', '=', 'o.object_name')
        .on('o.object_type', '=', kysely_1.sql.lit('TABLE'))
        .onRef('c.dtd_identifier', '=', 'o.collection_type_identifier'))
        .leftJoin('pg_type as t', (join) => join.onRef('t.typname', '=', 'c.udt_name').on('c.data_type', '=', kysely_1.sql.lit('USER-DEFINED')))
        .leftJoin('pg_enum as e', (join) => join.onRef('e.enumtypid', '=', 't.oid'))
        .select([
        'c.table_name',
        'c.column_name',
        'c.data_type',
        'c.column_default',
        'c.is_nullable',
        'c.character_maximum_length',
        'c.numeric_precision',
        'c.numeric_scale',
        'c.datetime_precision',
        'c.udt_catalog',
        'c.udt_schema',
        'c.udt_name',
        'o.data_type as array_type',
    ])
        .where('table_schema', '=', schemaName)
        .execute();
};
const getTableIndexes = (db, schemaName) => {
    return (db
        .selectFrom('pg_index as ix')
        .innerJoin('pg_class as i', 'ix.indexrelid', 'i.oid')
        .innerJoin('pg_am as a', 'i.relam', 'a.oid')
        .innerJoin('pg_class as t', 'ix.indrelid', 't.oid')
        .innerJoin('pg_namespace', 'pg_namespace.oid', 'i.relnamespace')
        .leftJoin('pg_constraint', (join) => join
        .onRef('pg_constraint.conindid', '=', 'i.oid')
        .on('pg_constraint.contype', 'in', [kysely_1.sql.lit('p'), kysely_1.sql.lit('u')]))
        .where('pg_constraint.oid', 'is', null)
        .select((eb) => [
        'i.relname as index_name',
        't.relname as table_name',
        'ix.indisunique as unique',
        'a.amname as using',
        eb.fn('pg_get_expr', ['ix.indexprs', 'ix.indrelid']).as('expression'),
        eb.fn('pg_get_expr', ['ix.indpred', 'ix.indrelid']).as('where'),
        eb
            .selectFrom('pg_attribute as a')
            .where('t.relkind', '=', kysely_1.sql.lit('r'))
            .whereRef('a.attrelid', '=', 't.oid')
            .whereRef('a.attnum', '=', (0, kysely_1.sql) `any("ix"."indkey")`)
            .select((eb) => eb.fn('json_agg', ['a.attname']).as('column_name'))
            .as('column_names'),
    ])
        .where('pg_namespace.nspname', '=', schemaName)
        .where('ix.indisprimary', '=', kysely_1.sql.lit(false))
        .execute());
};
const getTableConstraints = (db, schemaName) => {
    return db
        .selectFrom('pg_constraint')
        .innerJoin('pg_namespace', 'pg_namespace.oid', 'pg_constraint.connamespace')
        .innerJoin('pg_class as source_table', (join) => join.onRef('source_table.oid', '=', 'pg_constraint.conrelid').on('source_table.relkind', 'in', [
        kysely_1.sql.lit('r'),
        kysely_1.sql.lit('p'),
        kysely_1.sql.lit('f'),
    ]))
        .leftJoin('pg_class as reference_table', 'reference_table.oid', 'pg_constraint.confrelid')
        .select((eb) => [
        'pg_constraint.contype as constraint_type',
        'pg_constraint.conname as constraint_name',
        'source_table.relname as table_name',
        'reference_table.relname as reference_table_name',
        'pg_constraint.confupdtype as update_action',
        'pg_constraint.confdeltype as delete_action',
        eb
            .selectFrom('pg_attribute')
            .whereRef('pg_attribute.attrelid', '=', 'pg_constraint.conrelid')
            .whereRef('pg_attribute.attnum', '=', (0, kysely_1.sql) `any("pg_constraint"."conkey")`)
            .select((eb) => eb.fn('json_agg', ['pg_attribute.attname']).as('column_name'))
            .as('column_names'),
        eb
            .selectFrom('pg_attribute')
            .whereRef('pg_attribute.attrelid', '=', 'pg_constraint.confrelid')
            .whereRef('pg_attribute.attnum', '=', (0, kysely_1.sql) `any("pg_constraint"."confkey")`)
            .select((eb) => eb.fn('json_agg', ['pg_attribute.attname']).as('column_name'))
            .as('reference_column_names'),
        eb.fn('pg_get_constraintdef', ['pg_constraint.oid']).as('expression'),
    ])
        .where('pg_namespace.nspname', '=', schemaName)
        .execute();
};
const getUserDefinedEnums = async (db, schemaName) => {
    const items = await db
        .selectFrom('pg_type')
        .innerJoin('pg_namespace', (join) => join.onRef('pg_namespace.oid', '=', 'pg_type.typnamespace').on('pg_namespace.nspname', '=', schemaName))
        .where('typtype', '=', kysely_1.sql.lit('e'))
        .select((eb) => [
        'pg_type.typname as name',
        (0, postgres_1.jsonArrayFrom)(eb.selectFrom('pg_enum as e').select(['e.enumlabel as value']).whereRef('e.enumtypid', '=', 'pg_type.oid')).as('values'),
    ])
        .execute();
    return items.map((item) => ({
        name: item.name,
        values: item.values.map(({ value }) => value),
    }));
};
const getRoutines = async (db, schemaName) => {
    return db
        .selectFrom('pg_proc as p')
        .innerJoin('pg_namespace', 'pg_namespace.oid', 'p.pronamespace')
        .leftJoin('pg_depend as d', (join) => join.onRef('d.objid', '=', 'p.oid').on('d.deptype', '=', kysely_1.sql.lit('e')))
        .where('d.objid', 'is', kysely_1.sql.lit(null))
        .where('p.prokind', '=', kysely_1.sql.lit('f'))
        .where('pg_namespace.nspname', '=', schemaName)
        .select((eb) => [
        'p.proname as name',
        eb.fn('pg_get_function_identity_arguments', ['p.oid']).as('arguments'),
        eb.fn('pg_get_functiondef', ['p.oid']).as('expression'),
    ])
        .execute();
};
const getExtensions = async (db) => {
    return (db
        .selectFrom('pg_catalog.pg_extension')
        .select(['extname as name', 'extversion as version'])
        .execute());
};
const getTriggers = async (db, schemaName) => {
    return db
        .selectFrom('pg_trigger as t')
        .innerJoin('pg_proc as p', 't.tgfoid', 'p.oid')
        .innerJoin('pg_namespace as n', 'p.pronamespace', 'n.oid')
        .innerJoin('pg_class as c', 't.tgrelid', 'c.oid')
        .select((eb) => [
        't.tgname as name',
        't.tgenabled as enabled',
        't.tgtype as type',
        't.tgconstraint as _constraint',
        't.tgdeferrable as is_deferrable',
        't.tginitdeferred as is_initially_deferred',
        't.tgargs as arguments',
        't.tgoldtable as referencing_old_table_as',
        't.tgnewtable as referencing_new_table_as',
        eb.fn('pg_get_expr', ['t.tgqual', 't.tgrelid']).as('when_expression'),
        'p.proname as function_name',
        'c.relname as table_name',
    ])
        .where('t.tgisinternal', '=', false)
        .where('n.nspname', '=', schemaName)
        .execute();
};
const getParameters = async (db) => {
    return db
        .selectFrom('pg_settings')
        .where('source', 'in', [kysely_1.sql.lit('database'), kysely_1.sql.lit('user')])
        .select(['name', 'setting as value', 'source as scope'])
        .execute();
};
const getObjectComments = async (db) => {
    return db
        .selectFrom('pg_description as d')
        .innerJoin('pg_class as c', 'd.objoid', 'c.oid')
        .leftJoin('pg_attribute as a', (join) => join.onRef('a.attrelid', '=', 'c.oid').onRef('a.attnum', '=', 'd.objsubid'))
        .select([
        'c.relname as object_name',
        'c.relkind as object_type',
        'd.description as value',
        'a.attname as column_name',
    ])
        .where('d.description', 'is not', null)
        .orderBy('object_type')
        .orderBy('object_name')
        .execute();
};
//# sourceMappingURL=index.js.map