"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.processForeignKeyConstraints = void 0;
const column_processor_1 = require("./column.processor");
const table_processor_1 = require("./table.processor");
const helpers_1 = require("../../helpers");
const types_1 = require("../../types");
const processForeignKeyConstraints = (builder, items) => {
    for (const { item: { object, propertyName, options, target }, } of items.filter((item) => item.type === 'foreignKeyColumn')) {
        const { table, column } = (0, column_processor_1.resolveColumn)(builder, object, propertyName);
        if (!table) {
            (0, table_processor_1.onMissingTable)(builder, '@ForeignKeyColumn', object);
            continue;
        }
        if (!column) {
            (0, column_processor_1.onMissingColumn)(builder, '@ForeignKeyColumn', object, propertyName);
            continue;
        }
        const referenceTable = (0, table_processor_1.resolveTable)(builder, target());
        if (!referenceTable) {
            (0, table_processor_1.onMissingTable)(builder, '@ForeignKeyColumn', object, propertyName);
            continue;
        }
        const columnNames = [column.name];
        const referenceColumns = referenceTable.columns.filter((column) => column.primary);
        if (referenceColumns.length === 1) {
            column.type = referenceColumns[0].type;
        }
        table.constraints.push({
            name: options.constraintName || asForeignKeyConstraintName(table.name, columnNames),
            tableName: table.name,
            columnNames,
            type: types_1.DatabaseConstraintType.FOREIGN_KEY,
            referenceTableName: referenceTable.name,
            referenceColumnNames: referenceColumns.map((column) => column.name),
            onUpdate: options.onUpdate,
            onDelete: options.onDelete,
            synchronize: options.synchronize ?? true,
        });
        if (options.unique || options.uniqueConstraintName) {
            table.constraints.push({
                name: options.uniqueConstraintName || asRelationKeyConstraintName(table.name, columnNames),
                tableName: table.name,
                columnNames,
                type: types_1.DatabaseConstraintType.UNIQUE,
                synchronize: options.synchronize ?? true,
            });
        }
    }
};
exports.processForeignKeyConstraints = processForeignKeyConstraints;
const asForeignKeyConstraintName = (table, columns) => (0, helpers_1.asKey)('FK_', table, columns);
const asRelationKeyConstraintName = (table, columns) => (0, helpers_1.asKey)('REL_', table, columns);
//# sourceMappingURL=foreign-key-constriant.processor.js.map