"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.onMissingColumn = exports.resolveColumn = exports.processColumns = void 0;
const table_processor_1 = require("./table.processor");
const helpers_1 = require("../../helpers");
const processColumns = (builder, items) => {
    for (const { type, item: { object, propertyName, options }, } of items.filter((item) => item.type === 'column' || item.type === 'foreignKeyColumn')) {
        const table = (0, table_processor_1.resolveTable)(builder, object.constructor);
        if (!table) {
            (0, table_processor_1.onMissingTable)(builder, type === 'column' ? '@Column' : '@ForeignKeyColumn', object, propertyName);
            continue;
        }
        const columnName = options.name ?? String(propertyName);
        const existingColumn = table.columns.find((column) => column.name === columnName);
        if (existingColumn) {
            continue;
        }
        const tableName = table.name;
        let defaultValue = (0, helpers_1.fromColumnValue)(options.default);
        let nullable = options.nullable ?? false;
        if (defaultValue === null) {
            nullable = true;
            defaultValue = undefined;
        }
        const isEnum = !!options.enum;
        const column = {
            name: columnName,
            tableName,
            primary: options.primary ?? false,
            default: defaultValue,
            nullable,
            isArray: options.array ?? false,
            length: options.length,
            type: isEnum ? 'enum' : options.type || 'character varying',
            enumName: isEnum ? options.enum.name : undefined,
            comment: options.comment,
            storage: options.storage,
            identity: options.identity,
            synchronize: options.synchronize ?? true,
        };
        writeMetadata(object, propertyName, { name: column.name, options });
        table.columns.push(column);
    }
};
exports.processColumns = processColumns;
const resolveColumn = (builder, object, propertyName) => {
    const table = (0, table_processor_1.resolveTable)(builder, object.constructor);
    if (!table) {
        return {};
    }
    const metadata = readMetadata(object, propertyName);
    if (!metadata) {
        return { table };
    }
    const column = table.columns.find((column) => column.name === metadata.name);
    return { table, column };
};
exports.resolveColumn = resolveColumn;
const onMissingColumn = (builder, context, object, propertyName) => {
    const label = object.constructor.name + (propertyName ? '.' + String(propertyName) : '');
    builder.warnings.push(`[${context}] Unable to find column (${label})`);
};
exports.onMissingColumn = onMissingColumn;
const METADATA_KEY = (0, helpers_1.asMetadataKey)('table-metadata');
const writeMetadata = (object, propertyName, metadata) => Reflect.defineMetadata(METADATA_KEY, metadata, object, propertyName);
const readMetadata = (object, propertyName) => Reflect.getMetadata(METADATA_KEY, object, propertyName);
//# sourceMappingURL=column.processor.js.map