"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.compareConstraints = void 0;
const helpers_1 = require("../../helpers");
const types_1 = require("../../types");
exports.compareConstraints = {
    onMissing: (source) => [
        {
            type: 'constraint.add',
            constraint: source,
            reason: types_1.Reason.MissingInTarget,
        },
    ],
    onExtra: (target) => [
        {
            type: 'constraint.drop',
            tableName: target.tableName,
            constraintName: target.name,
            reason: types_1.Reason.MissingInSource,
        },
    ],
    onCompare: (source, target) => {
        switch (source.type) {
            case types_1.DatabaseConstraintType.PRIMARY_KEY: {
                return comparePrimaryKeyConstraint(source, target);
            }
            case types_1.DatabaseConstraintType.FOREIGN_KEY: {
                return compareForeignKeyConstraint(source, target);
            }
            case types_1.DatabaseConstraintType.UNIQUE: {
                return compareUniqueConstraint(source, target);
            }
            case types_1.DatabaseConstraintType.CHECK: {
                return compareCheckConstraint(source, target);
            }
            default: {
                return [];
            }
        }
    },
};
const comparePrimaryKeyConstraint = (source, target) => {
    if (!(0, helpers_1.haveEqualColumns)(source.columnNames, target.columnNames)) {
        return dropAndRecreateConstraint(source, target, `Primary key columns are different: (${source.columnNames} vs ${target.columnNames})`);
    }
    return [];
};
const compareForeignKeyConstraint = (source, target) => {
    let reason = '';
    const sourceDeleteAction = source.onDelete ?? 'NO ACTION';
    const targetDeleteAction = target.onDelete ?? 'NO ACTION';
    const sourceUpdateAction = source.onUpdate ?? 'NO ACTION';
    const targetUpdateAction = target.onUpdate ?? 'NO ACTION';
    if (!(0, helpers_1.haveEqualColumns)(source.columnNames, target.columnNames)) {
        reason = `columns are different (${source.columnNames} vs ${target.columnNames})`;
    }
    else if (!(0, helpers_1.haveEqualColumns)(source.referenceColumnNames, target.referenceColumnNames)) {
        reason = `reference columns are different (${source.referenceColumnNames} vs ${target.referenceColumnNames})`;
    }
    else if (source.referenceTableName !== target.referenceTableName) {
        reason = `reference table is different (${source.referenceTableName} vs ${target.referenceTableName})`;
    }
    else if (sourceDeleteAction !== targetDeleteAction) {
        reason = `ON DELETE action is different (${sourceDeleteAction} vs ${targetDeleteAction})`;
    }
    else if (sourceUpdateAction !== targetUpdateAction) {
        reason = `ON UPDATE action is different (${sourceUpdateAction} vs ${targetUpdateAction})`;
    }
    if (reason) {
        return dropAndRecreateConstraint(source, target, reason);
    }
    return [];
};
const compareUniqueConstraint = (source, target) => {
    let reason = '';
    if (!(0, helpers_1.haveEqualColumns)(source.columnNames, target.columnNames)) {
        reason = `columns are different (${source.columnNames} vs ${target.columnNames})`;
    }
    if (reason) {
        return dropAndRecreateConstraint(source, target, reason);
    }
    return [];
};
const compareCheckConstraint = (source, target) => {
    if (source.expression !== target.expression) {
    }
    return [];
};
const dropAndRecreateConstraint = (source, target, reason) => {
    return [
        {
            type: 'constraint.drop',
            tableName: target.tableName,
            constraintName: target.name,
            reason,
        },
        { type: 'constraint.add', constraint: source, reason },
    ];
};
//# sourceMappingURL=constraint.comparer.js.map