"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SyncService = exports.SYNC_TYPES_ORDER = void 0;
const common_1 = require("@nestjs/common");
const luxon_1 = require("luxon");
const constants_1 = require("../constants");
const asset_response_dto_1 = require("../dtos/asset-response.dto");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const asset_util_1 = require("../utils/asset.util");
const set_1 = require("../utils/set");
const sync_1 = require("../utils/sync");
const FULL_SYNC = { needsFullSync: true, deleted: [], upserted: [] };
exports.SYNC_TYPES_ORDER = [
    enum_1.SyncRequestType.UsersV1,
    enum_1.SyncRequestType.PartnersV1,
    enum_1.SyncRequestType.AssetsV1,
    enum_1.SyncRequestType.AssetExifsV1,
    enum_1.SyncRequestType.PartnerAssetsV1,
    enum_1.SyncRequestType.PartnerAssetExifsV1,
];
const throwSessionRequired = () => {
    throw new common_1.ForbiddenException('Sync endpoints cannot be used with API keys');
};
let SyncService = class SyncService extends base_service_1.BaseService {
    getAcks(auth) {
        const sessionId = auth.session?.id;
        if (!sessionId) {
            return throwSessionRequired();
        }
        return this.syncRepository.getCheckpoints(sessionId);
    }
    async setAcks(auth, dto) {
        const sessionId = auth.session?.id;
        if (!sessionId) {
            return throwSessionRequired();
        }
        const checkpoints = {};
        for (const ack of dto.acks) {
            const { type } = (0, sync_1.fromAck)(ack);
            if (!Object.values(enum_1.SyncEntityType).includes(type)) {
                throw new common_1.BadRequestException(`Invalid ack type: ${type}`);
            }
            if (checkpoints[type]) {
                throw new common_1.BadRequestException('Only one ack per type is allowed');
            }
            checkpoints[type] = { sessionId, type, ack };
        }
        await this.syncRepository.upsertCheckpoints(Object.values(checkpoints));
    }
    async deleteAcks(auth, dto) {
        const sessionId = auth.session?.id;
        if (!sessionId) {
            return throwSessionRequired();
        }
        await this.syncRepository.deleteCheckpoints(sessionId, dto.types);
    }
    async stream(auth, response, dto) {
        const sessionId = auth.session?.id;
        if (!sessionId) {
            return throwSessionRequired();
        }
        const checkpoints = await this.syncRepository.getCheckpoints(sessionId);
        const checkpointMap = Object.fromEntries(checkpoints.map(({ type, ack }) => [type, (0, sync_1.fromAck)(ack)]));
        for (const type of exports.SYNC_TYPES_ORDER.filter((type) => dto.types.includes(type))) {
            switch (type) {
                case enum_1.SyncRequestType.UsersV1: {
                    const deletes = this.syncRepository.getUserDeletes(checkpointMap[enum_1.SyncEntityType.UserDeleteV1]);
                    for await (const { id, ...data } of deletes) {
                        response.write((0, sync_1.serialize)({ type: enum_1.SyncEntityType.UserDeleteV1, updateId: id, data }));
                    }
                    const upserts = this.syncRepository.getUserUpserts(checkpointMap[enum_1.SyncEntityType.UserV1]);
                    for await (const { updateId, ...data } of upserts) {
                        response.write((0, sync_1.serialize)({ type: enum_1.SyncEntityType.UserV1, updateId, data }));
                    }
                    break;
                }
                case enum_1.SyncRequestType.PartnersV1: {
                    const deletes = this.syncRepository.getPartnerDeletes(auth.user.id, checkpointMap[enum_1.SyncEntityType.PartnerDeleteV1]);
                    for await (const { id, ...data } of deletes) {
                        response.write((0, sync_1.serialize)({ type: enum_1.SyncEntityType.PartnerDeleteV1, updateId: id, data }));
                    }
                    const upserts = this.syncRepository.getPartnerUpserts(auth.user.id, checkpointMap[enum_1.SyncEntityType.PartnerV1]);
                    for await (const { updateId, ...data } of upserts) {
                        response.write((0, sync_1.serialize)({ type: enum_1.SyncEntityType.PartnerV1, updateId, data }));
                    }
                    break;
                }
                case enum_1.SyncRequestType.AssetsV1: {
                    const deletes = this.syncRepository.getAssetDeletes(auth.user.id, checkpointMap[enum_1.SyncEntityType.AssetDeleteV1]);
                    for await (const { id, ...data } of deletes) {
                        response.write((0, sync_1.serialize)({ type: enum_1.SyncEntityType.AssetDeleteV1, updateId: id, data }));
                    }
                    const upserts = this.syncRepository.getAssetUpserts(auth.user.id, checkpointMap[enum_1.SyncEntityType.AssetV1]);
                    for await (const { updateId, checksum, thumbhash, ...data } of upserts) {
                        response.write((0, sync_1.serialize)({
                            type: enum_1.SyncEntityType.AssetV1,
                            updateId,
                            data: {
                                ...data,
                                checksum: (0, asset_response_dto_1.hexOrBufferToBase64)(checksum),
                                thumbhash: thumbhash ? (0, asset_response_dto_1.hexOrBufferToBase64)(thumbhash) : null,
                            },
                        }));
                    }
                    break;
                }
                case enum_1.SyncRequestType.PartnerAssetsV1: {
                    const deletes = this.syncRepository.getPartnerAssetDeletes(auth.user.id, checkpointMap[enum_1.SyncEntityType.PartnerAssetDeleteV1]);
                    for await (const { id, ...data } of deletes) {
                        response.write((0, sync_1.serialize)({ type: enum_1.SyncEntityType.PartnerAssetDeleteV1, updateId: id, data }));
                    }
                    const upserts = this.syncRepository.getPartnerAssetsUpserts(auth.user.id, checkpointMap[enum_1.SyncEntityType.PartnerAssetV1]);
                    for await (const { updateId, checksum, thumbhash, ...data } of upserts) {
                        response.write((0, sync_1.serialize)({
                            type: enum_1.SyncEntityType.PartnerAssetV1,
                            updateId,
                            data: {
                                ...data,
                                checksum: (0, asset_response_dto_1.hexOrBufferToBase64)(checksum),
                                thumbhash: thumbhash ? (0, asset_response_dto_1.hexOrBufferToBase64)(thumbhash) : null,
                            },
                        }));
                    }
                    break;
                }
                case enum_1.SyncRequestType.AssetExifsV1: {
                    const upserts = this.syncRepository.getAssetExifsUpserts(auth.user.id, checkpointMap[enum_1.SyncEntityType.AssetExifV1]);
                    for await (const { updateId, ...data } of upserts) {
                        response.write((0, sync_1.serialize)({ type: enum_1.SyncEntityType.AssetExifV1, updateId, data }));
                    }
                    break;
                }
                case enum_1.SyncRequestType.PartnerAssetExifsV1: {
                    const upserts = this.syncRepository.getPartnerAssetExifsUpserts(auth.user.id, checkpointMap[enum_1.SyncEntityType.PartnerAssetExifV1]);
                    for await (const { updateId, ...data } of upserts) {
                        response.write((0, sync_1.serialize)({ type: enum_1.SyncEntityType.PartnerAssetExifV1, updateId, data }));
                    }
                    break;
                }
                default: {
                    this.logger.warn(`Unsupported sync type: ${type}`);
                    break;
                }
            }
        }
        response.end();
    }
    async getFullSync(auth, dto) {
        const userId = dto.userId || auth.user.id;
        await this.requireAccess({ auth, permission: enum_1.Permission.TIMELINE_READ, ids: [userId] });
        const assets = await this.assetRepository.getAllForUserFullSync({
            ownerId: userId,
            updatedUntil: dto.updatedUntil,
            lastId: dto.lastId,
            limit: dto.limit,
        });
        return assets.map((a) => (0, asset_response_dto_1.mapAsset)(a, { auth, stripMetadata: false, withStack: true }));
    }
    async getDeltaSync(auth, dto) {
        const duration = luxon_1.DateTime.now().diff(luxon_1.DateTime.fromJSDate(dto.updatedAfter));
        if (duration > constants_1.AUDIT_LOG_MAX_DURATION) {
            return FULL_SYNC;
        }
        const partnerIds = await (0, asset_util_1.getMyPartnerIds)({ userId: auth.user.id, repository: this.partnerRepository });
        const userIds = [auth.user.id, ...partnerIds];
        if (!(0, set_1.setIsEqual)(new Set(userIds), new Set(dto.userIds))) {
            return FULL_SYNC;
        }
        await this.requireAccess({ auth, permission: enum_1.Permission.TIMELINE_READ, ids: dto.userIds });
        const limit = 10_000;
        const upserted = await this.assetRepository.getChangedDeltaSync({ limit, updatedAfter: dto.updatedAfter, userIds });
        if (upserted.length === limit) {
            return FULL_SYNC;
        }
        const deleted = await this.auditRepository.getAfter(dto.updatedAfter, {
            userIds,
            entityType: enum_1.EntityType.ASSET,
            action: enum_1.DatabaseAction.DELETE,
        });
        const result = {
            needsFullSync: false,
            upserted: upserted
                .filter((a) => a.ownerId === auth.user.id || (a.ownerId !== auth.user.id && !a.isArchived))
                .map((a) => (0, asset_response_dto_1.mapAsset)(a, {
                auth,
                stripMetadata: false,
                withStack: a.ownerId === auth.user.id,
            })),
            deleted,
        };
        return result;
    }
};
exports.SyncService = SyncService;
exports.SyncService = SyncService = __decorate([
    (0, common_1.Injectable)()
], SyncService);
//# sourceMappingURL=sync.service.js.map