"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetadataService = void 0;
const common_1 = require("@nestjs/common");
const FirstDateTime_1 = require("exiftool-vendored/dist/FirstDateTime");
const lodash_1 = __importDefault(require("lodash"));
const luxon_1 = require("luxon");
const promises_1 = require("node:fs/promises");
const node_path_1 = __importDefault(require("node:path"));
const constants_1 = require("../constants");
const storage_core_1 = require("../cores/storage.core");
const decorators_1 = require("../decorators");
const enum_1 = require("../enum");
const asset_repository_1 = require("../repositories/asset.repository");
const base_service_1 = require("./base.service");
const misc_1 = require("../utils/misc");
const pagination_1 = require("../utils/pagination");
const tag_1 = require("../utils/tag");
const EXIF_DATE_TAGS = [
    'SubSecDateTimeOriginal',
    'DateTimeOriginal',
    'SubSecCreateDate',
    'CreationDate',
    'CreateDate',
    'SubSecMediaCreateDate',
    'MediaCreateDate',
    'DateTimeCreated',
    'SourceImageCreateTime',
];
const validate = (value) => {
    if (Array.isArray(value)) {
        value = value[0];
    }
    if (typeof value === 'string') {
        return null;
    }
    if (typeof value === 'number' && (Number.isNaN(value) || !Number.isFinite(value))) {
        return null;
    }
    return value ?? null;
};
const validateRange = (value, min, max) => {
    const val = validate(value);
    if (val == null || val < min || val > max) {
        return null;
    }
    return val;
};
const getLensModel = (exifTags) => {
    const lensModel = String(exifTags.LensID ?? exifTags.LensType ?? exifTags.LensSpec ?? exifTags.LensModel ?? '').trim();
    if (lensModel === '----') {
        return null;
    }
    if (lensModel.startsWith('Unknown')) {
        return null;
    }
    return lensModel || null;
};
let MetadataService = class MetadataService extends base_service_1.BaseService {
    async onBootstrap() {
        this.logger.log('Bootstrapping metadata service');
        await this.init();
    }
    async onShutdown() {
        await this.metadataRepository.teardown();
    }
    onConfigInit({ newConfig }) {
        this.metadataRepository.setMaxConcurrency(newConfig.job.metadataExtraction.concurrency);
    }
    onConfigUpdate({ newConfig }) {
        this.metadataRepository.setMaxConcurrency(newConfig.job.metadataExtraction.concurrency);
    }
    async init() {
        this.logger.log('Initializing metadata service');
        try {
            await this.jobRepository.pause(enum_1.QueueName.METADATA_EXTRACTION);
            await this.databaseRepository.withLock(enum_1.DatabaseLock.GeodataImport, () => this.mapRepository.init());
            await this.jobRepository.resume(enum_1.QueueName.METADATA_EXTRACTION);
            this.logger.log(`Initialized local reverse geocoder`);
        }
        catch (error) {
            this.logger.error(`Unable to initialize reverse geocoding: ${error}`, error?.stack);
            throw new Error(`Metadata service init failed`);
        }
    }
    async linkLivePhotos(asset, exifInfo) {
        if (!exifInfo.livePhotoCID) {
            return;
        }
        const otherType = asset.type === enum_1.AssetType.VIDEO ? enum_1.AssetType.IMAGE : enum_1.AssetType.VIDEO;
        const match = await this.assetRepository.findLivePhotoMatch({
            livePhotoCID: exifInfo.livePhotoCID,
            ownerId: asset.ownerId,
            libraryId: asset.libraryId,
            otherAssetId: asset.id,
            type: otherType,
        });
        if (!match) {
            return;
        }
        const [photoAsset, motionAsset] = asset.type === enum_1.AssetType.IMAGE ? [asset, match] : [match, asset];
        await Promise.all([
            this.assetRepository.update({ id: photoAsset.id, livePhotoVideoId: motionAsset.id }),
            this.assetRepository.update({ id: motionAsset.id, isVisible: false }),
            this.albumRepository.removeAsset(motionAsset.id),
        ]);
        await this.eventRepository.emit('asset.hide', { assetId: motionAsset.id, userId: motionAsset.ownerId });
    }
    async handleQueueMetadataExtraction(job) {
        const { force } = job;
        const assetPagination = (0, pagination_1.usePagination)(constants_1.JOBS_ASSET_PAGINATION_SIZE, (pagination) => {
            return force
                ? this.assetRepository.getAll(pagination)
                : this.assetRepository.getWithout(pagination, asset_repository_1.WithoutProperty.EXIF);
        });
        for await (const assets of assetPagination) {
            await this.jobRepository.queueAll(assets.map((asset) => ({ name: enum_1.JobName.METADATA_EXTRACTION, data: { id: asset.id } })));
        }
        return enum_1.JobStatus.SUCCESS;
    }
    async handleMetadataExtraction(data) {
        const [{ metadata, reverseGeocoding }, asset] = await Promise.all([
            this.getConfig({ withCache: true }),
            this.assetJobRepository.getForMetadataExtraction(data.id),
        ]);
        if (!asset) {
            return enum_1.JobStatus.FAILED;
        }
        const [exifTags, stats] = await Promise.all([
            this.getExifTags(asset),
            this.storageRepository.stat(asset.originalPath),
        ]);
        this.logger.verbose('Exif Tags', exifTags);
        const dates = this.getDates(asset, exifTags, stats);
        const { width, height } = this.getImageDimensions(exifTags);
        let geo = { country: null, state: null, city: null }, latitude = null, longitude = null;
        if (this.hasGeo(exifTags)) {
            latitude = exifTags.GPSLatitude;
            longitude = exifTags.GPSLongitude;
            if (reverseGeocoding.enabled) {
                geo = await this.mapRepository.reverseGeocode({ latitude, longitude });
            }
        }
        const exifData = {
            assetId: asset.id,
            dateTimeOriginal: dates.dateTimeOriginal,
            modifyDate: stats.mtime,
            timeZone: dates.timeZone,
            latitude,
            longitude,
            country: geo.country,
            state: geo.state,
            city: geo.city,
            fileSizeInByte: stats.size,
            exifImageHeight: validate(height),
            exifImageWidth: validate(width),
            orientation: validate(exifTags.Orientation)?.toString() ?? null,
            projectionType: exifTags.ProjectionType ? String(exifTags.ProjectionType).toUpperCase() : null,
            bitsPerSample: this.getBitsPerSample(exifTags),
            colorspace: exifTags.ColorSpace ?? null,
            make: exifTags.Make ?? exifTags?.Device?.Manufacturer ?? exifTags.AndroidMake ?? null,
            model: exifTags.Model ?? exifTags?.Device?.ModelName ?? exifTags.AndroidModel ?? null,
            fps: validate(Number.parseFloat(exifTags.VideoFrameRate)),
            iso: validate(exifTags.ISO),
            exposureTime: exifTags.ExposureTime ?? null,
            lensModel: getLensModel(exifTags),
            fNumber: validate(exifTags.FNumber),
            focalLength: validate(exifTags.FocalLength),
            description: String(exifTags.ImageDescription || exifTags.Description || '').trim(),
            profileDescription: exifTags.ProfileDescription || null,
            rating: validateRange(exifTags.Rating, -1, 5),
            livePhotoCID: (exifTags.ContentIdentifier || exifTags.MediaGroupUUID) ?? null,
            autoStackId: this.getAutoStackId(exifTags),
        };
        const promises = [
            this.assetRepository.upsertExif(exifData),
            this.assetRepository.update({
                id: asset.id,
                duration: exifTags.Duration?.toString() ?? null,
                localDateTime: dates.localDateTime,
                fileCreatedAt: dates.dateTimeOriginal ?? undefined,
                fileModifiedAt: stats.mtime,
            }),
            this.applyTagList(asset, exifTags),
        ];
        if (this.isMotionPhoto(asset, exifTags)) {
            promises.push(this.applyMotionPhotos(asset, exifTags, dates, stats));
        }
        if ((0, misc_1.isFaceImportEnabled)(metadata) && this.hasTaggedFaces(exifTags)) {
            promises.push(this.applyTaggedFaces(asset, exifTags));
        }
        await Promise.all(promises);
        if (exifData.livePhotoCID) {
            await this.linkLivePhotos(asset, exifData);
        }
        await this.assetRepository.upsertJobStatus({ assetId: asset.id, metadataExtractedAt: new Date() });
        return enum_1.JobStatus.SUCCESS;
    }
    async handleQueueSidecar(job) {
        const { force } = job;
        const assetPagination = (0, pagination_1.usePagination)(constants_1.JOBS_ASSET_PAGINATION_SIZE, (pagination) => {
            return force
                ? this.assetRepository.getAll(pagination)
                : this.assetRepository.getWithout(pagination, asset_repository_1.WithoutProperty.SIDECAR);
        });
        for await (const assets of assetPagination) {
            await this.jobRepository.queueAll(assets.map((asset) => ({
                name: force ? enum_1.JobName.SIDECAR_SYNC : enum_1.JobName.SIDECAR_DISCOVERY,
                data: { id: asset.id },
            })));
        }
        return enum_1.JobStatus.SUCCESS;
    }
    handleSidecarSync({ id }) {
        return this.processSidecar(id, true);
    }
    handleSidecarDiscovery({ id }) {
        return this.processSidecar(id, false);
    }
    async handleTagAsset({ assetId }) {
        await this.jobRepository.queue({ name: enum_1.JobName.SIDECAR_WRITE, data: { id: assetId, tags: true } });
    }
    async handleUntagAsset({ assetId }) {
        await this.jobRepository.queue({ name: enum_1.JobName.SIDECAR_WRITE, data: { id: assetId, tags: true } });
    }
    async handleSidecarWrite(job) {
        const { id, description, dateTimeOriginal, latitude, longitude, rating, tags } = job;
        const asset = await this.assetJobRepository.getForSidecarWriteJob(id);
        if (!asset) {
            return enum_1.JobStatus.FAILED;
        }
        const tagsList = (asset.tags || []).map((tag) => tag.value);
        const sidecarPath = asset.sidecarPath || `${asset.originalPath}.xmp`;
        const exif = lodash_1.default.omitBy({
            Description: description,
            ImageDescription: description,
            DateTimeOriginal: dateTimeOriginal,
            GPSLatitude: latitude,
            GPSLongitude: longitude,
            Rating: rating,
            TagsList: tags ? tagsList : undefined,
        }, lodash_1.default.isUndefined);
        if (Object.keys(exif).length === 0) {
            return enum_1.JobStatus.SKIPPED;
        }
        await this.metadataRepository.writeTags(sidecarPath, exif);
        if (!asset.sidecarPath) {
            await this.assetRepository.update({ id, sidecarPath });
        }
        return enum_1.JobStatus.SUCCESS;
    }
    getImageDimensions(exifTags) {
        let [width, height] = exifTags.ImageSize?.split('x').map((dim) => Number.parseInt(dim) || undefined) || [];
        if (!width || !height) {
            [width, height] = [exifTags.ImageWidth, exifTags.ImageHeight];
        }
        return { width, height };
    }
    getExifTags(asset) {
        if (!asset.sidecarPath && asset.type === enum_1.AssetType.IMAGE) {
            return this.metadataRepository.readTags(asset.originalPath);
        }
        return this.mergeExifTags(asset);
    }
    async mergeExifTags(asset) {
        const [mediaTags, sidecarTags, videoTags] = await Promise.all([
            this.metadataRepository.readTags(asset.originalPath),
            asset.sidecarPath ? this.metadataRepository.readTags(asset.sidecarPath) : null,
            asset.type === enum_1.AssetType.VIDEO ? this.getVideoTags(asset.originalPath) : null,
        ]);
        if (sidecarTags) {
            const sidecarDate = (0, FirstDateTime_1.firstDateTime)(sidecarTags, EXIF_DATE_TAGS);
            if (sidecarDate) {
                for (const tag of EXIF_DATE_TAGS) {
                    delete mediaTags[tag];
                }
            }
        }
        delete mediaTags.Duration;
        delete sidecarTags?.Duration;
        return { ...mediaTags, ...videoTags, ...sidecarTags };
    }
    getTagList(exifTags) {
        let tags;
        if (exifTags.TagsList) {
            tags = exifTags.TagsList.map(String);
        }
        else if (exifTags.HierarchicalSubject) {
            tags = exifTags.HierarchicalSubject.map((tag) => typeof tag === 'number'
                ? String(tag)
                : tag
                    .split('|')
                    .map((tag) => tag.replaceAll('/', '|'))
                    .join('/'));
        }
        else if (exifTags.Keywords) {
            let keywords = exifTags.Keywords;
            if (!Array.isArray(keywords)) {
                keywords = [keywords];
            }
            tags = keywords.map(String);
        }
        else {
            tags = [];
        }
        return tags;
    }
    async applyTagList(asset, exifTags) {
        const tags = this.getTagList(exifTags);
        const results = await (0, tag_1.upsertTags)(this.tagRepository, { userId: asset.ownerId, tags });
        await this.tagRepository.replaceAssetTags(asset.id, results.map((tag) => tag.id));
    }
    isMotionPhoto(asset, tags) {
        return asset.type === enum_1.AssetType.IMAGE && !!(tags.MotionPhoto || tags.MicroVideo);
    }
    async applyMotionPhotos(asset, tags, dates, stats) {
        const isMotionPhoto = tags.MotionPhoto;
        const isMicroVideo = tags.MicroVideo;
        const videoOffset = tags.MicroVideoOffset;
        const hasMotionPhotoVideo = tags.MotionPhotoVideo;
        const hasEmbeddedVideoFile = tags.EmbeddedVideoType === 'MotionPhoto_Data' && tags.EmbeddedVideoFile;
        const directory = Array.isArray(tags.ContainerDirectory)
            ? tags.ContainerDirectory
            : null;
        let length = 0;
        let padding = 0;
        if (isMotionPhoto && directory) {
            for (const entry of directory) {
                if (entry?.Item?.Semantic === 'MotionPhoto') {
                    length = entry.Item.Length ?? 0;
                    padding = entry.Item.Padding ?? 0;
                    break;
                }
            }
        }
        if (isMicroVideo && typeof videoOffset === 'number') {
            length = videoOffset;
        }
        if (!length && !hasEmbeddedVideoFile && !hasMotionPhotoVideo) {
            return;
        }
        this.logger.debug(`Starting motion photo video extraction for asset ${asset.id}: ${asset.originalPath}`);
        try {
            const position = stats.size - length - padding;
            let video;
            if (hasMotionPhotoVideo) {
                video = await this.metadataRepository.extractBinaryTag(asset.originalPath, 'MotionPhotoVideo');
            }
            else if (hasEmbeddedVideoFile) {
                video = await this.metadataRepository.extractBinaryTag(asset.originalPath, 'EmbeddedVideoFile');
            }
            else {
                video = await this.storageRepository.readFile(asset.originalPath, {
                    buffer: Buffer.alloc(length),
                    position,
                    length,
                });
            }
            const checksum = this.cryptoRepository.hashSha1(video);
            let motionAsset = await this.assetRepository.getByChecksum({
                ownerId: asset.ownerId,
                libraryId: asset.libraryId ?? undefined,
                checksum,
            });
            if (motionAsset) {
                this.logger.debugFn(() => {
                    const base64Checksum = checksum.toString('base64');
                    return `Motion asset with checksum ${base64Checksum} already exists for asset ${asset.id}: ${asset.originalPath}`;
                });
                if (motionAsset.isVisible) {
                    await this.assetRepository.update({ id: motionAsset.id, isVisible: false });
                    this.logger.log(`Hid unlinked motion photo video asset (${motionAsset.id})`);
                }
            }
            else {
                const motionAssetId = this.cryptoRepository.randomUUID();
                motionAsset = await this.assetRepository.create({
                    id: motionAssetId,
                    libraryId: asset.libraryId,
                    type: enum_1.AssetType.VIDEO,
                    fileCreatedAt: dates.dateTimeOriginal,
                    fileModifiedAt: stats.mtime,
                    localDateTime: dates.localDateTime,
                    checksum,
                    ownerId: asset.ownerId,
                    originalPath: storage_core_1.StorageCore.getAndroidMotionPath(asset, motionAssetId),
                    originalFileName: `${node_path_1.default.parse(asset.originalFileName).name}.mp4`,
                    isVisible: false,
                    deviceAssetId: 'NONE',
                    deviceId: 'NONE',
                });
                if (!asset.isExternal) {
                    await this.userRepository.updateUsage(asset.ownerId, video.byteLength);
                }
            }
            if (asset.livePhotoVideoId !== motionAsset.id) {
                await this.assetRepository.update({ id: asset.id, livePhotoVideoId: motionAsset.id });
                if (asset.livePhotoVideoId) {
                    await this.jobRepository.queue({
                        name: enum_1.JobName.ASSET_DELETION,
                        data: { id: asset.livePhotoVideoId, deleteOnDisk: true },
                    });
                    this.logger.log(`Removed old motion photo video asset (${asset.livePhotoVideoId})`);
                }
            }
            const existsOnDisk = await this.storageRepository.checkFileExists(motionAsset.originalPath);
            if (!existsOnDisk) {
                this.storageCore.ensureFolders(motionAsset.originalPath);
                await this.storageRepository.createFile(motionAsset.originalPath, video);
                this.logger.log(`Wrote motion photo video to ${motionAsset.originalPath}`);
                await this.handleMetadataExtraction({ id: motionAsset.id });
                await this.jobRepository.queue({ name: enum_1.JobName.VIDEO_CONVERSION, data: { id: motionAsset.id } });
            }
            this.logger.debug(`Finished motion photo video extraction for asset ${asset.id}: ${asset.originalPath}`);
        }
        catch (error) {
            this.logger.error(`Failed to extract motion video for ${asset.id}: ${asset.originalPath}: ${error}`, error?.stack);
        }
    }
    hasTaggedFaces(tags) {
        return (tags.RegionInfo !== undefined && tags.RegionInfo.AppliedToDimensions && tags.RegionInfo.RegionList.length > 0);
    }
    async applyTaggedFaces(asset, tags) {
        if (!tags.RegionInfo?.AppliedToDimensions || tags.RegionInfo.RegionList.length === 0) {
            return;
        }
        const facesToAdd = [];
        const existingNames = await this.personRepository.getDistinctNames(asset.ownerId, { withHidden: true });
        const existingNameMap = new Map(existingNames.map(({ id, name }) => [name.toLowerCase(), id]));
        const missing = [];
        const missingWithFaceAsset = [];
        for (const region of tags.RegionInfo.RegionList) {
            if (!region.Name) {
                continue;
            }
            const imageWidth = tags.RegionInfo.AppliedToDimensions.W;
            const imageHeight = tags.RegionInfo.AppliedToDimensions.H;
            const loweredName = region.Name.toLowerCase();
            const personId = existingNameMap.get(loweredName) || this.cryptoRepository.randomUUID();
            const face = {
                id: this.cryptoRepository.randomUUID(),
                personId,
                assetId: asset.id,
                imageWidth,
                imageHeight,
                boundingBoxX1: Math.floor((region.Area.X - region.Area.W / 2) * imageWidth),
                boundingBoxY1: Math.floor((region.Area.Y - region.Area.H / 2) * imageHeight),
                boundingBoxX2: Math.floor((region.Area.X + region.Area.W / 2) * imageWidth),
                boundingBoxY2: Math.floor((region.Area.Y + region.Area.H / 2) * imageHeight),
                sourceType: enum_1.SourceType.EXIF,
            };
            facesToAdd.push(face);
            if (!existingNameMap.has(loweredName)) {
                missing.push({ id: personId, ownerId: asset.ownerId, name: region.Name });
                missingWithFaceAsset.push({ id: personId, ownerId: asset.ownerId, faceAssetId: face.id });
            }
        }
        if (missing.length > 0) {
            this.logger.debugFn(() => `Creating missing persons: ${missing.map((p) => `${p.name}/${p.id}`)}`);
            const newPersonIds = await this.personRepository.createAll(missing);
            const jobs = newPersonIds.map((id) => ({ name: enum_1.JobName.GENERATE_PERSON_THUMBNAIL, data: { id } }));
            await this.jobRepository.queueAll(jobs);
        }
        const facesToRemove = asset.faces.filter((face) => face.sourceType === enum_1.SourceType.EXIF).map((face) => face.id);
        if (facesToRemove.length > 0) {
            this.logger.debug(`Removing ${facesToRemove.length} faces for asset ${asset.id}: ${asset.originalPath}`);
        }
        if (facesToAdd.length > 0) {
            this.logger.debug(`Creating ${facesToAdd.length} faces from metadata for asset ${asset.id}: ${asset.originalPath}`);
        }
        if (facesToRemove.length > 0 || facesToAdd.length > 0) {
            await this.personRepository.refreshFaces(facesToAdd, facesToRemove);
        }
        if (missingWithFaceAsset.length > 0) {
            await this.personRepository.updateAll(missingWithFaceAsset);
        }
    }
    getDates(asset, exifTags, stats) {
        const dateTime = (0, FirstDateTime_1.firstDateTime)(exifTags, EXIF_DATE_TAGS);
        this.logger.verbose(`Date and time is ${dateTime} for asset ${asset.id}: ${asset.originalPath}`);
        let timeZone = exifTags.tz ?? null;
        if (timeZone == null && dateTime?.rawValue?.endsWith('+00:00')) {
            timeZone = 'UTC+0';
        }
        if (timeZone) {
            this.logger.verbose(`Found timezone ${timeZone} via ${exifTags.tzSource} for asset ${asset.id}: ${asset.originalPath}`);
        }
        else {
            this.logger.debug(`No timezone information found for asset ${asset.id}: ${asset.originalPath}`);
        }
        let dateTimeOriginal = dateTime?.toDate();
        let localDateTime = dateTime?.toDateTime().setZone('UTC', { keepLocalTime: true }).toJSDate();
        if (!localDateTime || !dateTimeOriginal) {
            const earliestDate = stats.birthtimeMs ? new Date(Math.min(stats.mtimeMs, stats.birthtimeMs)) : stats.mtime;
            this.logger.debug(`No exif date time found, falling back on ${earliestDate.toISOString()}, earliest of file creation and modification for asset ${asset.id}: ${asset.originalPath}`);
            dateTimeOriginal = localDateTime = earliestDate;
        }
        this.logger.verbose(`Found local date time ${localDateTime.toISOString()} for asset ${asset.id}: ${asset.originalPath}`);
        return {
            dateTimeOriginal,
            timeZone,
            localDateTime,
        };
    }
    hasGeo(tags) {
        return (tags.GPSLatitude !== undefined &&
            tags.GPSLongitude !== undefined &&
            (tags.GPSLatitude !== 0 || tags.GPSLatitude !== 0));
    }
    getAutoStackId(tags) {
        if (!tags) {
            return null;
        }
        return tags.BurstID ?? tags.BurstUUID ?? tags.CameraBurstID ?? tags.MediaUniqueID ?? null;
    }
    getBitsPerSample(tags) {
        const bitDepthTags = [
            tags.BitsPerSample,
            tags.ComponentBitDepth,
            tags.ImagePixelDepth,
            tags.BitDepth,
            tags.ColorBitDepth,
        ].map((tag) => (typeof tag === 'string' ? Number.parseInt(tag) : tag));
        let bitsPerSample = bitDepthTags.find((tag) => typeof tag === 'number' && !Number.isNaN(tag)) ?? null;
        if (bitsPerSample && bitsPerSample >= 24 && bitsPerSample % 3 === 0) {
            bitsPerSample /= 3;
        }
        return bitsPerSample;
    }
    async getVideoTags(originalPath) {
        const { videoStreams, format } = await this.mediaRepository.probe(originalPath);
        const tags = {};
        if (videoStreams[0]) {
            switch (videoStreams[0].rotation) {
                case -90: {
                    tags.Orientation = enum_1.ExifOrientation.Rotate90CW;
                    break;
                }
                case 0: {
                    tags.Orientation = enum_1.ExifOrientation.Horizontal;
                    break;
                }
                case 90: {
                    tags.Orientation = enum_1.ExifOrientation.Rotate270CW;
                    break;
                }
                case 180: {
                    tags.Orientation = enum_1.ExifOrientation.Rotate180;
                    break;
                }
            }
        }
        if (format.duration) {
            tags.Duration = luxon_1.Duration.fromObject({ seconds: format.duration }).toFormat('hh:mm:ss.SSS');
        }
        return tags;
    }
    async processSidecar(id, isSync) {
        const [asset] = await this.assetRepository.getByIds([id]);
        if (!asset) {
            return enum_1.JobStatus.FAILED;
        }
        if (isSync && !asset.sidecarPath) {
            return enum_1.JobStatus.FAILED;
        }
        if (!isSync && (!asset.isVisible || asset.sidecarPath) && !asset.isExternal) {
            return enum_1.JobStatus.FAILED;
        }
        const assetPath = node_path_1.default.parse(asset.originalPath);
        const assetPathWithoutExt = node_path_1.default.join(assetPath.dir, assetPath.name);
        const sidecarPathWithoutExt = `${assetPathWithoutExt}.xmp`;
        const sidecarPathWithExt = `${asset.originalPath}.xmp`;
        const [sidecarPathWithExtExists, sidecarPathWithoutExtExists] = await Promise.all([
            this.storageRepository.checkFileExists(sidecarPathWithExt, promises_1.constants.R_OK),
            this.storageRepository.checkFileExists(sidecarPathWithoutExt, promises_1.constants.R_OK),
        ]);
        let sidecarPath = null;
        if (sidecarPathWithExtExists) {
            sidecarPath = sidecarPathWithExt;
        }
        else if (sidecarPathWithoutExtExists) {
            sidecarPath = sidecarPathWithoutExt;
        }
        if (asset.isExternal) {
            if (sidecarPath !== asset.sidecarPath) {
                await this.assetRepository.update({ id: asset.id, sidecarPath });
            }
            return enum_1.JobStatus.SUCCESS;
        }
        if (sidecarPath) {
            this.logger.debug(`Detected sidecar at '${sidecarPath}' for asset ${asset.id}: ${asset.originalPath}`);
            await this.assetRepository.update({ id: asset.id, sidecarPath });
            return enum_1.JobStatus.SUCCESS;
        }
        if (!isSync) {
            return enum_1.JobStatus.FAILED;
        }
        this.logger.debug(`No sidecar found for asset ${asset.id}: ${asset.originalPath}`);
        await this.assetRepository.update({ id: asset.id, sidecarPath: null });
        return enum_1.JobStatus.SUCCESS;
    }
};
exports.MetadataService = MetadataService;
__decorate([
    (0, decorators_1.OnEvent)({ name: 'app.bootstrap', workers: [enum_1.ImmichWorker.MICROSERVICES] }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "onBootstrap", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'app.shutdown' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "onShutdown", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'config.init', workers: [enum_1.ImmichWorker.MICROSERVICES] }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], MetadataService.prototype, "onConfigInit", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'config.update', workers: [enum_1.ImmichWorker.MICROSERVICES], server: true }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], MetadataService.prototype, "onConfigUpdate", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.QUEUE_METADATA_EXTRACTION, queue: enum_1.QueueName.METADATA_EXTRACTION }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleQueueMetadataExtraction", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.METADATA_EXTRACTION, queue: enum_1.QueueName.METADATA_EXTRACTION }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleMetadataExtraction", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.QUEUE_SIDECAR, queue: enum_1.QueueName.SIDECAR }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleQueueSidecar", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.SIDECAR_SYNC, queue: enum_1.QueueName.SIDECAR }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleSidecarSync", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.SIDECAR_DISCOVERY, queue: enum_1.QueueName.SIDECAR }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleSidecarDiscovery", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'asset.tag' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleTagAsset", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'asset.untag' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleUntagAsset", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.SIDECAR_WRITE, queue: enum_1.QueueName.SIDECAR }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MetadataService.prototype, "handleSidecarWrite", null);
exports.MetadataService = MetadataService = __decorate([
    (0, common_1.Injectable)()
], MetadataService);
//# sourceMappingURL=metadata.service.js.map