"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MediaService = void 0;
const common_1 = require("@nestjs/common");
const constants_1 = require("../constants");
const storage_core_1 = require("../cores/storage.core");
const decorators_1 = require("../decorators");
const enum_1 = require("../enum");
const asset_repository_1 = require("../repositories/asset.repository");
const base_service_1 = require("./base.service");
const asset_util_1 = require("../utils/asset.util");
const media_1 = require("../utils/media");
const mime_types_1 = require("../utils/mime-types");
const pagination_1 = require("../utils/pagination");
let MediaService = class MediaService extends base_service_1.BaseService {
    videoInterfaces = { dri: [], mali: false };
    async onBootstrap() {
        const [dri, mali] = await Promise.all([this.getDevices(), this.hasMaliOpenCL()]);
        this.videoInterfaces = { dri, mali };
    }
    async handleQueueGenerateThumbnails({ force }) {
        const thumbJobs = [];
        for await (const asset of this.assetJobRepository.streamForThumbnailJob(!!force)) {
            const { previewFile, thumbnailFile } = (0, asset_util_1.getAssetFiles)(asset.files);
            if (!previewFile || !thumbnailFile || !asset.thumbhash || force) {
                thumbJobs.push({ name: enum_1.JobName.GENERATE_THUMBNAILS, data: { id: asset.id } });
                continue;
            }
        }
        await this.jobRepository.queueAll(thumbJobs);
        const jobs = [];
        const people = this.personRepository.getAll(force ? undefined : { thumbnailPath: '' });
        for await (const person of people) {
            if (!person.faceAssetId) {
                const face = await this.personRepository.getRandomFace(person.id);
                if (!face) {
                    continue;
                }
                await this.personRepository.update({ id: person.id, faceAssetId: face.id });
            }
            jobs.push({ name: enum_1.JobName.GENERATE_PERSON_THUMBNAIL, data: { id: person.id } });
        }
        await this.jobRepository.queueAll(jobs);
        return enum_1.JobStatus.SUCCESS;
    }
    async handleQueueMigration() {
        const assetPagination = (0, pagination_1.usePagination)(constants_1.JOBS_ASSET_PAGINATION_SIZE, (pagination) => this.assetRepository.getAll(pagination));
        const { active, waiting } = await this.jobRepository.getJobCounts(enum_1.QueueName.MIGRATION);
        if (active === 1 && waiting === 0) {
            await this.storageCore.removeEmptyDirs(enum_1.StorageFolder.THUMBNAILS);
            await this.storageCore.removeEmptyDirs(enum_1.StorageFolder.ENCODED_VIDEO);
        }
        for await (const assets of assetPagination) {
            await this.jobRepository.queueAll(assets.map((asset) => ({ name: enum_1.JobName.MIGRATE_ASSET, data: { id: asset.id } })));
        }
        let jobs = [];
        for await (const person of this.personRepository.getAll()) {
            jobs.push({ name: enum_1.JobName.MIGRATE_PERSON, data: { id: person.id } });
            if (jobs.length === constants_1.JOBS_ASSET_PAGINATION_SIZE) {
                await this.jobRepository.queueAll(jobs);
                jobs = [];
            }
        }
        await this.jobRepository.queueAll(jobs);
        return enum_1.JobStatus.SUCCESS;
    }
    async handleAssetMigration({ id }) {
        const { image } = await this.getConfig({ withCache: true });
        const asset = await this.assetJobRepository.getForMigrationJob(id);
        if (!asset) {
            return enum_1.JobStatus.FAILED;
        }
        await this.storageCore.moveAssetImage(asset, enum_1.AssetPathType.FULLSIZE, image.fullsize.format);
        await this.storageCore.moveAssetImage(asset, enum_1.AssetPathType.PREVIEW, image.preview.format);
        await this.storageCore.moveAssetImage(asset, enum_1.AssetPathType.THUMBNAIL, image.thumbnail.format);
        await this.storageCore.moveAssetVideo(asset);
        return enum_1.JobStatus.SUCCESS;
    }
    async handleGenerateThumbnails({ id }) {
        const asset = await this.assetJobRepository.getForGenerateThumbnailJob(id);
        if (!asset) {
            this.logger.warn(`Thumbnail generation failed for asset ${id}: not found`);
            return enum_1.JobStatus.FAILED;
        }
        if (!asset.isVisible) {
            this.logger.verbose(`Thumbnail generation skipped for asset ${id}: not visible`);
            return enum_1.JobStatus.SKIPPED;
        }
        let generated;
        if (asset.type === enum_1.AssetType.VIDEO || asset.originalFileName.toLowerCase().endsWith('.gif')) {
            generated = await this.generateVideoThumbnails(asset);
        }
        else if (asset.type === enum_1.AssetType.IMAGE) {
            generated = await this.generateImageThumbnails(asset);
        }
        else {
            this.logger.warn(`Skipping thumbnail generation for asset ${id}: ${asset.type} is not an image or video`);
            return enum_1.JobStatus.SKIPPED;
        }
        const { previewFile, thumbnailFile, fullsizeFile } = (0, asset_util_1.getAssetFiles)(asset.files);
        const toUpsert = [];
        if (previewFile?.path !== generated.previewPath) {
            toUpsert.push({ assetId: asset.id, path: generated.previewPath, type: enum_1.AssetFileType.PREVIEW });
        }
        if (thumbnailFile?.path !== generated.thumbnailPath) {
            toUpsert.push({ assetId: asset.id, path: generated.thumbnailPath, type: enum_1.AssetFileType.THUMBNAIL });
        }
        if (generated.fullsizePath && fullsizeFile?.path !== generated.fullsizePath) {
            toUpsert.push({ assetId: asset.id, path: generated.fullsizePath, type: enum_1.AssetFileType.FULLSIZE });
        }
        if (toUpsert.length > 0) {
            await this.assetRepository.upsertFiles(toUpsert);
        }
        const pathsToDelete = [];
        if (previewFile && previewFile.path !== generated.previewPath) {
            this.logger.debug(`Deleting old preview for asset ${asset.id}`);
            pathsToDelete.push(previewFile.path);
        }
        if (thumbnailFile && thumbnailFile.path !== generated.thumbnailPath) {
            this.logger.debug(`Deleting old thumbnail for asset ${asset.id}`);
            pathsToDelete.push(thumbnailFile.path);
        }
        if (fullsizeFile && fullsizeFile.path !== generated.fullsizePath) {
            this.logger.debug(`Deleting old fullsize preview image for asset ${asset.id}`);
            pathsToDelete.push(fullsizeFile.path);
            if (!generated.fullsizePath) {
                await this.assetRepository.deleteFiles([fullsizeFile]);
            }
        }
        if (pathsToDelete.length > 0) {
            await Promise.all(pathsToDelete.map((path) => this.storageRepository.unlink(path)));
        }
        if (!asset.thumbhash || Buffer.compare(asset.thumbhash, generated.thumbhash) !== 0) {
            await this.assetRepository.update({ id: asset.id, thumbhash: generated.thumbhash });
        }
        await this.assetRepository.upsertJobStatus({ assetId: asset.id, previewAt: new Date(), thumbnailAt: new Date() });
        return enum_1.JobStatus.SUCCESS;
    }
    async generateImageThumbnails(asset) {
        const { image } = await this.getConfig({ withCache: true });
        const previewPath = storage_core_1.StorageCore.getImagePath(asset, enum_1.AssetPathType.PREVIEW, image.preview.format);
        const thumbnailPath = storage_core_1.StorageCore.getImagePath(asset, enum_1.AssetPathType.THUMBNAIL, image.thumbnail.format);
        this.storageCore.ensureFolders(previewPath);
        const processInvalidImages = process.env.IMMICH_PROCESS_INVALID_IMAGES === 'true';
        const colorspace = this.isSRGB(asset) ? enum_1.Colorspace.SRGB : image.colorspace;
        const { enabled: imageFullsizeEnabled, ...imageFullsizeConfig } = image.fullsize;
        const shouldConvertFullsize = imageFullsizeEnabled && !mime_types_1.mimeTypes.isWebSupportedImage(asset.originalFileName);
        const shouldExtractEmbedded = image.extractEmbedded && mime_types_1.mimeTypes.isRaw(asset.originalFileName);
        const decodeOptions = { colorspace, processInvalidImages, size: image.preview.size };
        let useExtracted = false;
        let decodeInputPath = asset.originalPath;
        let fullsizePath;
        if (shouldConvertFullsize) {
            decodeOptions.size = undefined;
            fullsizePath = storage_core_1.StorageCore.getImagePath(asset, enum_1.AssetPathType.FULLSIZE, image.fullsize.format);
        }
        if (shouldExtractEmbedded) {
            const extractedPath = storage_core_1.StorageCore.getImagePath(asset, enum_1.AssetPathType.FULLSIZE, enum_1.ImageFormat.JPEG);
            const didExtract = await this.mediaRepository.extract(asset.originalPath, extractedPath);
            useExtracted = didExtract && (await this.shouldUseExtractedImage(extractedPath, image.preview.size));
            if (useExtracted) {
                if (shouldConvertFullsize) {
                    fullsizePath = extractedPath;
                }
                decodeInputPath = extractedPath;
                if (asset.exifInfo) {
                    const exif = { orientation: asset.exifInfo.orientation, colorspace: asset.exifInfo.colorspace };
                    await this.mediaRepository.writeExif(exif, extractedPath);
                }
            }
        }
        const { info, data } = await this.mediaRepository.decodeImage(decodeInputPath, decodeOptions);
        const thumbnailOptions = { colorspace, processInvalidImages, raw: info };
        const promises = [
            this.mediaRepository.generateThumbhash(data, thumbnailOptions),
            this.mediaRepository.generateThumbnail(data, { ...image.thumbnail, ...thumbnailOptions }, thumbnailPath),
            this.mediaRepository.generateThumbnail(data, { ...image.preview, ...thumbnailOptions }, previewPath),
        ];
        if (fullsizePath && !useExtracted) {
            const fullsizeOptions = {
                ...imageFullsizeConfig,
                ...thumbnailOptions,
                size: undefined,
            };
            promises.push(this.mediaRepository.generateThumbnail(data, fullsizeOptions, fullsizePath));
        }
        const outputs = await Promise.all(promises);
        return { previewPath, thumbnailPath, fullsizePath, thumbhash: outputs[0] };
    }
    async generateVideoThumbnails(asset) {
        const { image, ffmpeg } = await this.getConfig({ withCache: true });
        const previewPath = storage_core_1.StorageCore.getImagePath(asset, enum_1.AssetPathType.PREVIEW, image.preview.format);
        const thumbnailPath = storage_core_1.StorageCore.getImagePath(asset, enum_1.AssetPathType.THUMBNAIL, image.thumbnail.format);
        this.storageCore.ensureFolders(previewPath);
        const { format, audioStreams, videoStreams } = await this.mediaRepository.probe(asset.originalPath);
        const mainVideoStream = this.getMainStream(videoStreams);
        if (!mainVideoStream) {
            throw new Error(`No video streams found for asset ${asset.id}`);
        }
        const mainAudioStream = this.getMainStream(audioStreams);
        const previewConfig = media_1.ThumbnailConfig.create({ ...ffmpeg, targetResolution: image.preview.size.toString() });
        const thumbnailConfig = media_1.ThumbnailConfig.create({ ...ffmpeg, targetResolution: image.thumbnail.size.toString() });
        const previewOptions = previewConfig.getCommand(enum_1.TranscodeTarget.VIDEO, mainVideoStream, mainAudioStream, format);
        const thumbnailOptions = thumbnailConfig.getCommand(enum_1.TranscodeTarget.VIDEO, mainVideoStream, mainAudioStream, format);
        await this.mediaRepository.transcode(asset.originalPath, previewPath, previewOptions);
        await this.mediaRepository.transcode(asset.originalPath, thumbnailPath, thumbnailOptions);
        const thumbhash = await this.mediaRepository.generateThumbhash(previewPath, {
            colorspace: image.colorspace,
            processInvalidImages: process.env.IMMICH_PROCESS_INVALID_IMAGES === 'true',
        });
        return { previewPath, thumbnailPath, thumbhash };
    }
    async handleQueueVideoConversion(job) {
        const { force } = job;
        const assetPagination = (0, pagination_1.usePagination)(constants_1.JOBS_ASSET_PAGINATION_SIZE, (pagination) => {
            return force
                ? this.assetRepository.getAll(pagination, { type: enum_1.AssetType.VIDEO })
                : this.assetRepository.getWithout(pagination, asset_repository_1.WithoutProperty.ENCODED_VIDEO);
        });
        for await (const assets of assetPagination) {
            await this.jobRepository.queueAll(assets.map((asset) => ({ name: enum_1.JobName.VIDEO_CONVERSION, data: { id: asset.id } })));
        }
        return enum_1.JobStatus.SUCCESS;
    }
    async handleVideoConversion({ id }) {
        const [asset] = await this.assetRepository.getByIds([id]);
        if (!asset || asset.type !== enum_1.AssetType.VIDEO) {
            return enum_1.JobStatus.FAILED;
        }
        const input = asset.originalPath;
        const output = storage_core_1.StorageCore.getEncodedVideoPath(asset);
        this.storageCore.ensureFolders(output);
        const { videoStreams, audioStreams, format } = await this.mediaRepository.probe(input, {
            countFrames: this.logger.isLevelEnabled(enum_1.LogLevel.DEBUG),
        });
        const videoStream = this.getMainStream(videoStreams);
        const audioStream = this.getMainStream(audioStreams);
        if (!videoStream || !format.formatName) {
            return enum_1.JobStatus.FAILED;
        }
        if (!videoStream.height || !videoStream.width) {
            this.logger.warn(`Skipped transcoding for asset ${asset.id}: no video streams found`);
            return enum_1.JobStatus.FAILED;
        }
        let { ffmpeg } = await this.getConfig({ withCache: true });
        const target = this.getTranscodeTarget(ffmpeg, videoStream, audioStream);
        if (target === enum_1.TranscodeTarget.NONE && !this.isRemuxRequired(ffmpeg, format)) {
            if (asset.encodedVideoPath) {
                this.logger.log(`Transcoded video exists for asset ${asset.id}, but is no longer required. Deleting...`);
                await this.jobRepository.queue({ name: enum_1.JobName.DELETE_FILES, data: { files: [asset.encodedVideoPath] } });
                await this.assetRepository.update({ id: asset.id, encodedVideoPath: null });
            }
            else {
                this.logger.verbose(`Asset ${asset.id} does not require transcoding based on current policy, skipping`);
            }
            return enum_1.JobStatus.SKIPPED;
        }
        const command = media_1.BaseConfig.create(ffmpeg, this.videoInterfaces).getCommand(target, videoStream, audioStream);
        if (ffmpeg.accel === enum_1.TranscodeHWAccel.DISABLED) {
            this.logger.log(`Transcoding video ${asset.id} without hardware acceleration`);
        }
        else {
            this.logger.log(`Transcoding video ${asset.id} with ${ffmpeg.accel.toUpperCase()}-accelerated encoding and${ffmpeg.accelDecode ? '' : ' software'} decoding`);
        }
        try {
            await this.mediaRepository.transcode(input, output, command);
        }
        catch (error) {
            this.logger.error(`Error occurred during transcoding: ${error.message}`);
            if (ffmpeg.accel === enum_1.TranscodeHWAccel.DISABLED) {
                return enum_1.JobStatus.FAILED;
            }
            let partialFallbackSuccess = false;
            if (ffmpeg.accelDecode) {
                try {
                    this.logger.error(`Retrying with ${ffmpeg.accel.toUpperCase()}-accelerated encoding and software decoding`);
                    ffmpeg = { ...ffmpeg, accelDecode: false };
                    const command = media_1.BaseConfig.create(ffmpeg, this.videoInterfaces).getCommand(target, videoStream, audioStream);
                    await this.mediaRepository.transcode(input, output, command);
                    partialFallbackSuccess = true;
                }
                catch (error) {
                    this.logger.error(`Error occurred during transcoding: ${error.message}`);
                }
            }
            if (!partialFallbackSuccess) {
                this.logger.error(`Retrying with ${ffmpeg.accel.toUpperCase()} acceleration disabled`);
                ffmpeg = { ...ffmpeg, accel: enum_1.TranscodeHWAccel.DISABLED };
                const command = media_1.BaseConfig.create(ffmpeg, this.videoInterfaces).getCommand(target, videoStream, audioStream);
                await this.mediaRepository.transcode(input, output, command);
            }
        }
        this.logger.log(`Successfully encoded ${asset.id}`);
        await this.assetRepository.update({ id: asset.id, encodedVideoPath: output });
        return enum_1.JobStatus.SUCCESS;
    }
    getMainStream(streams) {
        return streams
            .filter((stream) => stream.codecName !== 'unknown')
            .sort((stream1, stream2) => stream2.frameCount - stream1.frameCount)[0];
    }
    getTranscodeTarget(config, videoStream, audioStream) {
        const isAudioTranscodeRequired = this.isAudioTranscodeRequired(config, audioStream);
        const isVideoTranscodeRequired = this.isVideoTranscodeRequired(config, videoStream);
        if (isAudioTranscodeRequired && isVideoTranscodeRequired) {
            return enum_1.TranscodeTarget.ALL;
        }
        if (isAudioTranscodeRequired) {
            return enum_1.TranscodeTarget.AUDIO;
        }
        if (isVideoTranscodeRequired) {
            return enum_1.TranscodeTarget.VIDEO;
        }
        return enum_1.TranscodeTarget.NONE;
    }
    isAudioTranscodeRequired(ffmpegConfig, stream) {
        if (!stream) {
            return false;
        }
        switch (ffmpegConfig.transcode) {
            case enum_1.TranscodePolicy.DISABLED: {
                return false;
            }
            case enum_1.TranscodePolicy.ALL: {
                return true;
            }
            case enum_1.TranscodePolicy.REQUIRED:
            case enum_1.TranscodePolicy.OPTIMAL:
            case enum_1.TranscodePolicy.BITRATE: {
                return !ffmpegConfig.acceptedAudioCodecs.includes(stream.codecName);
            }
            default: {
                throw new Error(`Unsupported transcode policy: ${ffmpegConfig.transcode}`);
            }
        }
    }
    isVideoTranscodeRequired(ffmpegConfig, stream) {
        const scalingEnabled = ffmpegConfig.targetResolution !== 'original';
        const targetRes = Number.parseInt(ffmpegConfig.targetResolution);
        const isLargerThanTargetRes = scalingEnabled && Math.min(stream.height, stream.width) > targetRes;
        const isLargerThanTargetBitrate = stream.bitrate > this.parseBitrateToBps(ffmpegConfig.maxBitrate);
        const isTargetVideoCodec = ffmpegConfig.acceptedVideoCodecs.includes(stream.codecName);
        const isRequired = !isTargetVideoCodec || !stream.pixelFormat.endsWith('420p');
        switch (ffmpegConfig.transcode) {
            case enum_1.TranscodePolicy.DISABLED: {
                return false;
            }
            case enum_1.TranscodePolicy.ALL: {
                return true;
            }
            case enum_1.TranscodePolicy.REQUIRED: {
                return isRequired;
            }
            case enum_1.TranscodePolicy.OPTIMAL: {
                return isRequired || isLargerThanTargetRes;
            }
            case enum_1.TranscodePolicy.BITRATE: {
                return isRequired || isLargerThanTargetBitrate;
            }
            default: {
                throw new Error(`Unsupported transcode policy: ${ffmpegConfig.transcode}`);
            }
        }
    }
    isRemuxRequired(ffmpegConfig, { formatName, formatLongName }) {
        if (ffmpegConfig.transcode === enum_1.TranscodePolicy.DISABLED) {
            return false;
        }
        const name = formatLongName === 'QuickTime / MOV' ? enum_1.VideoContainer.MOV : formatName;
        return name !== enum_1.VideoContainer.MP4 && !ffmpegConfig.acceptedContainers.includes(name);
    }
    isSRGB(asset) {
        const { colorspace, profileDescription, bitsPerSample } = asset.exifInfo;
        if (colorspace || profileDescription) {
            return [colorspace, profileDescription].some((s) => s?.toLowerCase().includes('srgb'));
        }
        else if (bitsPerSample) {
            return bitsPerSample === 8;
        }
        else {
            return true;
        }
    }
    parseBitrateToBps(bitrateString) {
        const bitrateValue = Number.parseInt(bitrateString);
        if (Number.isNaN(bitrateValue)) {
            return 0;
        }
        if (bitrateString.toLowerCase().endsWith('k')) {
            return bitrateValue * 1000;
        }
        else if (bitrateString.toLowerCase().endsWith('m')) {
            return bitrateValue * 1_000_000;
        }
        else {
            return bitrateValue;
        }
    }
    async shouldUseExtractedImage(extractedPath, targetSize) {
        const { width, height } = await this.mediaRepository.getImageDimensions(extractedPath);
        const extractedSize = Math.min(width, height);
        return extractedSize >= targetSize;
    }
    async getDevices() {
        try {
            return await this.storageRepository.readdir('/dev/dri');
        }
        catch {
            this.logger.debug('No devices found in /dev/dri.');
            return [];
        }
    }
    async hasMaliOpenCL() {
        try {
            const [maliIcdStat, maliDeviceStat] = await Promise.all([
                this.storageRepository.stat('/etc/OpenCL/vendors/mali.icd'),
                this.storageRepository.stat('/dev/mali0'),
            ]);
            return maliIcdStat.isFile() && maliDeviceStat.isCharacterDevice();
        }
        catch {
            this.logger.debug('OpenCL not available for transcoding, so RKMPP acceleration will use CPU tonemapping');
            return false;
        }
    }
};
exports.MediaService = MediaService;
__decorate([
    (0, decorators_1.OnEvent)({ name: 'app.bootstrap' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "onBootstrap", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.QUEUE_GENERATE_THUMBNAILS, queue: enum_1.QueueName.THUMBNAIL_GENERATION }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleQueueGenerateThumbnails", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.QUEUE_MIGRATION, queue: enum_1.QueueName.MIGRATION }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleQueueMigration", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.MIGRATE_ASSET, queue: enum_1.QueueName.MIGRATION }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleAssetMigration", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.GENERATE_THUMBNAILS, queue: enum_1.QueueName.THUMBNAIL_GENERATION }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleGenerateThumbnails", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.QUEUE_VIDEO_CONVERSION, queue: enum_1.QueueName.VIDEO_CONVERSION }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleQueueVideoConversion", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.VIDEO_CONVERSION, queue: enum_1.QueueName.VIDEO_CONVERSION }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaService.prototype, "handleVideoConversion", null);
exports.MediaService = MediaService = __decorate([
    (0, common_1.Injectable)()
], MediaService);
//# sourceMappingURL=media.service.js.map