"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobService = void 0;
const common_1 = require("@nestjs/common");
const lodash_1 = require("lodash");
const decorators_1 = require("../decorators");
const asset_response_dto_1 = require("../dtos/asset-response.dto");
const job_dto_1 = require("../dtos/job.dto");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const asJobItem = (dto) => {
    switch (dto.name) {
        case enum_1.ManualJobName.TAG_CLEANUP: {
            return { name: enum_1.JobName.TAG_CLEANUP };
        }
        case enum_1.ManualJobName.PERSON_CLEANUP: {
            return { name: enum_1.JobName.PERSON_CLEANUP };
        }
        case enum_1.ManualJobName.USER_CLEANUP: {
            return { name: enum_1.JobName.USER_DELETE_CHECK };
        }
        case enum_1.ManualJobName.MEMORY_CLEANUP: {
            return { name: enum_1.JobName.MEMORIES_CLEANUP };
        }
        case enum_1.ManualJobName.MEMORY_CREATE: {
            return { name: enum_1.JobName.MEMORIES_CREATE };
        }
        case enum_1.ManualJobName.BACKUP_DATABASE: {
            return { name: enum_1.JobName.BACKUP_DATABASE };
        }
        default: {
            throw new common_1.BadRequestException('Invalid job name');
        }
    }
};
let JobService = class JobService extends base_service_1.BaseService {
    services = [];
    onConfigInit({ newConfig: config }) {
        this.logger.debug(`Updating queue concurrency settings`);
        for (const queueName of Object.values(enum_1.QueueName)) {
            let concurrency = 1;
            if (this.isConcurrentQueue(queueName)) {
                concurrency = config.job[queueName].concurrency;
            }
            this.logger.debug(`Setting ${queueName} concurrency to ${concurrency}`);
            this.jobRepository.setConcurrency(queueName, concurrency);
        }
    }
    onConfigUpdate({ newConfig: config }) {
        this.onConfigInit({ newConfig: config });
    }
    onBootstrap() {
        this.jobRepository.setup(this.services);
        if (this.worker === enum_1.ImmichWorker.MICROSERVICES) {
            this.jobRepository.startWorkers();
        }
    }
    setServices(services) {
        this.services = services;
    }
    async create(dto) {
        await this.jobRepository.queue(asJobItem(dto));
    }
    async handleCommand(queueName, dto) {
        this.logger.debug(`Handling command: queue=${queueName},command=${dto.command},force=${dto.force}`);
        switch (dto.command) {
            case enum_1.JobCommand.START: {
                await this.start(queueName, dto);
                break;
            }
            case enum_1.JobCommand.PAUSE: {
                await this.jobRepository.pause(queueName);
                break;
            }
            case enum_1.JobCommand.RESUME: {
                await this.jobRepository.resume(queueName);
                break;
            }
            case enum_1.JobCommand.EMPTY: {
                await this.jobRepository.empty(queueName);
                break;
            }
            case enum_1.JobCommand.CLEAR_FAILED: {
                const failedJobs = await this.jobRepository.clear(queueName, enum_1.QueueCleanType.FAILED);
                this.logger.debug(`Cleared failed jobs: ${failedJobs}`);
                break;
            }
        }
        return this.getJobStatus(queueName);
    }
    async getJobStatus(queueName) {
        const [jobCounts, queueStatus] = await Promise.all([
            this.jobRepository.getJobCounts(queueName),
            this.jobRepository.getQueueStatus(queueName),
        ]);
        return { jobCounts, queueStatus };
    }
    async getAllJobsStatus() {
        const response = new job_dto_1.AllJobStatusResponseDto();
        for (const queueName of Object.values(enum_1.QueueName)) {
            response[queueName] = await this.getJobStatus(queueName);
        }
        return response;
    }
    async start(name, { force }) {
        const { isActive } = await this.jobRepository.getQueueStatus(name);
        if (isActive) {
            throw new common_1.BadRequestException(`Job is already running`);
        }
        this.telemetryRepository.jobs.addToCounter(`immich.queues.${(0, lodash_1.snakeCase)(name)}.started`, 1);
        switch (name) {
            case enum_1.QueueName.VIDEO_CONVERSION: {
                return this.jobRepository.queue({ name: enum_1.JobName.QUEUE_VIDEO_CONVERSION, data: { force } });
            }
            case enum_1.QueueName.STORAGE_TEMPLATE_MIGRATION: {
                return this.jobRepository.queue({ name: enum_1.JobName.STORAGE_TEMPLATE_MIGRATION });
            }
            case enum_1.QueueName.MIGRATION: {
                return this.jobRepository.queue({ name: enum_1.JobName.QUEUE_MIGRATION });
            }
            case enum_1.QueueName.SMART_SEARCH: {
                return this.jobRepository.queue({ name: enum_1.JobName.QUEUE_SMART_SEARCH, data: { force } });
            }
            case enum_1.QueueName.DUPLICATE_DETECTION: {
                return this.jobRepository.queue({ name: enum_1.JobName.QUEUE_DUPLICATE_DETECTION, data: { force } });
            }
            case enum_1.QueueName.METADATA_EXTRACTION: {
                return this.jobRepository.queue({ name: enum_1.JobName.QUEUE_METADATA_EXTRACTION, data: { force } });
            }
            case enum_1.QueueName.SIDECAR: {
                return this.jobRepository.queue({ name: enum_1.JobName.QUEUE_SIDECAR, data: { force } });
            }
            case enum_1.QueueName.THUMBNAIL_GENERATION: {
                return this.jobRepository.queue({ name: enum_1.JobName.QUEUE_GENERATE_THUMBNAILS, data: { force } });
            }
            case enum_1.QueueName.FACE_DETECTION: {
                return this.jobRepository.queue({ name: enum_1.JobName.QUEUE_FACE_DETECTION, data: { force } });
            }
            case enum_1.QueueName.FACIAL_RECOGNITION: {
                return this.jobRepository.queue({ name: enum_1.JobName.QUEUE_FACIAL_RECOGNITION, data: { force } });
            }
            case enum_1.QueueName.LIBRARY: {
                return this.jobRepository.queue({ name: enum_1.JobName.LIBRARY_QUEUE_SCAN_ALL, data: { force } });
            }
            case enum_1.QueueName.BACKUP_DATABASE: {
                return this.jobRepository.queue({ name: enum_1.JobName.BACKUP_DATABASE, data: { force } });
            }
            default: {
                throw new common_1.BadRequestException(`Invalid job name: ${name}`);
            }
        }
    }
    async onJobStart(...[queueName, job]) {
        const queueMetric = `immich.queues.${(0, lodash_1.snakeCase)(queueName)}.active`;
        this.telemetryRepository.jobs.addToGauge(queueMetric, 1);
        try {
            const status = await this.jobRepository.run(job);
            const jobMetric = `immich.jobs.${job.name.replaceAll('-', '_')}.${status}`;
            this.telemetryRepository.jobs.addToCounter(jobMetric, 1);
            if (status === enum_1.JobStatus.SUCCESS || status == enum_1.JobStatus.SKIPPED) {
                await this.onDone(job);
            }
        }
        catch (error) {
            this.logger.error(`Unable to run job handler (${queueName}/${job.name}): ${error}`, error?.stack, JSON.stringify(job.data));
        }
        finally {
            this.telemetryRepository.jobs.addToGauge(queueMetric, -1);
        }
    }
    isConcurrentQueue(name) {
        return ![
            enum_1.QueueName.FACIAL_RECOGNITION,
            enum_1.QueueName.STORAGE_TEMPLATE_MIGRATION,
            enum_1.QueueName.DUPLICATE_DETECTION,
            enum_1.QueueName.BACKUP_DATABASE,
        ].includes(name);
    }
    async handleNightlyJobs() {
        await this.jobRepository.queueAll([
            { name: enum_1.JobName.ASSET_DELETION_CHECK },
            { name: enum_1.JobName.USER_DELETE_CHECK },
            { name: enum_1.JobName.PERSON_CLEANUP },
            { name: enum_1.JobName.MEMORIES_CLEANUP },
            { name: enum_1.JobName.MEMORIES_CREATE },
            { name: enum_1.JobName.QUEUE_GENERATE_THUMBNAILS, data: { force: false } },
            { name: enum_1.JobName.CLEAN_OLD_AUDIT_LOGS },
            { name: enum_1.JobName.USER_SYNC_USAGE },
            { name: enum_1.JobName.QUEUE_FACIAL_RECOGNITION, data: { force: false, nightly: true } },
            { name: enum_1.JobName.CLEAN_OLD_SESSION_TOKENS },
        ]);
    }
    async onDone(item) {
        switch (item.name) {
            case enum_1.JobName.SIDECAR_SYNC:
            case enum_1.JobName.SIDECAR_DISCOVERY: {
                await this.jobRepository.queue({ name: enum_1.JobName.METADATA_EXTRACTION, data: item.data });
                break;
            }
            case enum_1.JobName.SIDECAR_WRITE: {
                await this.jobRepository.queue({
                    name: enum_1.JobName.METADATA_EXTRACTION,
                    data: { id: item.data.id, source: 'sidecar-write' },
                });
                break;
            }
            case enum_1.JobName.METADATA_EXTRACTION: {
                if (item.data.source === 'sidecar-write') {
                    const [asset] = await this.assetRepository.getByIdsWithAllRelationsButStacks([item.data.id]);
                    if (asset) {
                        this.eventRepository.clientSend('on_asset_update', asset.ownerId, (0, asset_response_dto_1.mapAsset)(asset));
                    }
                }
                await this.jobRepository.queue({ name: enum_1.JobName.STORAGE_TEMPLATE_MIGRATION_SINGLE, data: item.data });
                break;
            }
            case enum_1.JobName.STORAGE_TEMPLATE_MIGRATION_SINGLE: {
                if (item.data.source === 'upload' || item.data.source === 'copy') {
                    await this.jobRepository.queue({ name: enum_1.JobName.GENERATE_THUMBNAILS, data: item.data });
                }
                break;
            }
            case enum_1.JobName.GENERATE_PERSON_THUMBNAIL: {
                const { id } = item.data;
                const person = await this.personRepository.getById(id);
                if (person) {
                    this.eventRepository.clientSend('on_person_thumbnail', person.ownerId, person.id);
                }
                break;
            }
            case enum_1.JobName.GENERATE_THUMBNAILS: {
                if (!item.data.notify && item.data.source !== 'upload') {
                    break;
                }
                const [asset] = await this.assetRepository.getByIdsWithAllRelationsButStacks([item.data.id]);
                if (!asset) {
                    this.logger.warn(`Could not find asset ${item.data.id} after generating thumbnails`);
                    break;
                }
                const jobs = [
                    { name: enum_1.JobName.SMART_SEARCH, data: item.data },
                    { name: enum_1.JobName.FACE_DETECTION, data: item.data },
                ];
                if (asset.type === enum_1.AssetType.VIDEO) {
                    jobs.push({ name: enum_1.JobName.VIDEO_CONVERSION, data: item.data });
                }
                await this.jobRepository.queueAll(jobs);
                if (asset.isVisible) {
                    this.eventRepository.clientSend('on_upload_success', asset.ownerId, (0, asset_response_dto_1.mapAsset)(asset));
                }
                break;
            }
            case enum_1.JobName.SMART_SEARCH: {
                if (item.data.source === 'upload') {
                    await this.jobRepository.queue({ name: enum_1.JobName.DUPLICATE_DETECTION, data: item.data });
                }
                break;
            }
            case enum_1.JobName.USER_DELETION: {
                this.eventRepository.clientBroadcast('on_user_delete', item.data.id);
                break;
            }
        }
    }
};
exports.JobService = JobService;
__decorate([
    (0, decorators_1.OnEvent)({ name: 'config.init', workers: [enum_1.ImmichWorker.MICROSERVICES] }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], JobService.prototype, "onConfigInit", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'config.update', server: true, workers: [enum_1.ImmichWorker.MICROSERVICES] }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], JobService.prototype, "onConfigUpdate", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'app.bootstrap', priority: enum_1.BootstrapEventPriority.JobService }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], JobService.prototype, "onBootstrap", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'job.start' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], JobService.prototype, "onJobStart", null);
exports.JobService = JobService = __decorate([
    (0, common_1.Injectable)()
], JobService);
//# sourceMappingURL=job.service.js.map