"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseService = void 0;
const common_1 = require("@nestjs/common");
const semver_1 = __importDefault(require("semver"));
const constants_1 = require("../constants");
const decorators_1 = require("../decorators");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const messages = {
    notInstalled: (name) => `The ${name} extension is not available in this Postgres instance.
    If using a container image, ensure the image has the extension installed.`,
    nightlyVersion: ({ name, extension, version }) => `
    The ${name} extension version is ${version}, which means it is a nightly release.

    Please run 'DROP EXTENSION IF EXISTS ${extension}' and switch to a release version.
    See https://immich.app/docs/guides/database-queries for how to query the database.`,
    outOfRange: ({ name, version, range }) => `The ${name} extension version is ${version}, but Immich only supports ${range}.
    Please change ${name} to a compatible version in the Postgres instance.`,
    createFailed: ({ name, extension, otherName }) => `Failed to activate ${name} extension.
    Please ensure the Postgres instance has ${name} installed.

    If the Postgres instance already has ${name} installed, Immich may not have the necessary permissions to activate it.
    In this case, please run 'CREATE EXTENSION IF NOT EXISTS ${extension}' manually as a superuser.
    See https://immich.app/docs/guides/database-queries for how to query the database.

    Alternatively, if your Postgres instance has ${otherName}, you may use this instead by setting the environment variable 'DB_VECTOR_EXTENSION=${otherName}'.
    Note that switching between the two extensions after a successful startup is not supported.
    The exception is if your version of Immich prior to upgrading was 1.90.2 or earlier.
    In this case, you may set either extension now, but you will not be able to switch to the other extension following a successful startup.`,
    updateFailed: ({ name, extension, availableVersion }) => `The ${name} extension can be updated to ${availableVersion}.
    Immich attempted to update the extension, but failed to do so.
    This may be because Immich does not have the necessary permissions to update the extension.

    Please run 'ALTER EXTENSION ${extension} UPDATE' manually as a superuser.
    See https://immich.app/docs/guides/database-queries for how to query the database.`,
    restartRequired: ({ name, availableVersion }) => `The ${name} extension has been updated to ${availableVersion}.
    Please restart the Postgres instance to complete the update.`,
    invalidDowngrade: ({ name, installedVersion, availableVersion }) => `The database currently has ${name} ${installedVersion} activated, but the Postgres instance only has ${availableVersion} available.
    This most likely means the extension was downgraded.
    If ${name} ${installedVersion} is compatible with Immich, please ensure the Postgres instance has this available.`,
};
let DatabaseService = class DatabaseService extends base_service_1.BaseService {
    async onBootstrap() {
        const version = await this.databaseRepository.getPostgresVersion();
        const current = semver_1.default.coerce(version);
        const postgresRange = this.databaseRepository.getPostgresVersionRange();
        if (!current || !semver_1.default.satisfies(current, postgresRange)) {
            throw new Error(`Invalid PostgreSQL version. Found ${version}, but needed ${postgresRange}. Please use a supported version.`);
        }
        await this.databaseRepository.withLock(enum_1.DatabaseLock.Migrations, async () => {
            const envData = this.configRepository.getEnv();
            const extension = envData.database.vectorExtension;
            const name = constants_1.EXTENSION_NAMES[extension];
            const extensionRange = this.databaseRepository.getExtensionVersionRange(extension);
            const { availableVersion, installedVersion } = await this.databaseRepository.getExtensionVersion(extension);
            if (!availableVersion) {
                throw new Error(messages.notInstalled(name));
            }
            if ([availableVersion, installedVersion].some((version) => version && semver_1.default.eq(version, '0.0.0'))) {
                throw new Error(messages.nightlyVersion({ name, extension, version: '0.0.0' }));
            }
            if (!semver_1.default.satisfies(availableVersion, extensionRange)) {
                throw new Error(messages.outOfRange({ name, extension, version: availableVersion, range: extensionRange }));
            }
            if (!installedVersion) {
                await this.createExtension(extension);
            }
            if (installedVersion && semver_1.default.gt(availableVersion, installedVersion)) {
                await this.updateExtension(extension, availableVersion);
            }
            else if (installedVersion && !semver_1.default.satisfies(installedVersion, extensionRange)) {
                throw new Error(messages.outOfRange({ name, extension, version: installedVersion, range: extensionRange }));
            }
            else if (installedVersion && semver_1.default.lt(availableVersion, installedVersion)) {
                throw new Error(messages.invalidDowngrade({ name, extension, availableVersion, installedVersion }));
            }
            await this.checkReindexing();
            const { database } = this.configRepository.getEnv();
            if (!database.skipMigrations) {
                await this.databaseRepository.runMigrations();
            }
        });
    }
    async createExtension(extension) {
        try {
            await this.databaseRepository.createExtension(extension);
        }
        catch (error) {
            const otherExtension = extension === enum_1.DatabaseExtension.VECTORS ? enum_1.DatabaseExtension.VECTOR : enum_1.DatabaseExtension.VECTORS;
            const name = constants_1.EXTENSION_NAMES[extension];
            this.logger.fatal(messages.createFailed({ name, extension, otherName: constants_1.EXTENSION_NAMES[otherExtension] }));
            throw error;
        }
    }
    async updateExtension(extension, availableVersion) {
        this.logger.log(`Updating ${constants_1.EXTENSION_NAMES[extension]} extension to ${availableVersion}`);
        try {
            const { restartRequired } = await this.databaseRepository.updateVectorExtension(extension, availableVersion);
            if (restartRequired) {
                this.logger.warn(messages.restartRequired({ name: constants_1.EXTENSION_NAMES[extension], availableVersion }));
            }
        }
        catch (error) {
            this.logger.warn(messages.updateFailed({ name: constants_1.EXTENSION_NAMES[extension], extension, availableVersion }));
            throw error;
        }
    }
    async checkReindexing() {
        try {
            if (await this.databaseRepository.shouldReindex(enum_1.VectorIndex.CLIP)) {
                await this.databaseRepository.reindex(enum_1.VectorIndex.CLIP);
            }
            if (await this.databaseRepository.shouldReindex(enum_1.VectorIndex.FACE)) {
                await this.databaseRepository.reindex(enum_1.VectorIndex.FACE);
            }
        }
        catch (error) {
            this.logger.warn('Could not run vector reindexing checks. If the extension was updated, please restart the Postgres instance.');
            throw error;
        }
    }
};
exports.DatabaseService = DatabaseService;
__decorate([
    (0, decorators_1.OnEvent)({ name: 'app.bootstrap', priority: enum_1.BootstrapEventPriority.DatabaseService }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], DatabaseService.prototype, "onBootstrap", null);
exports.DatabaseService = DatabaseService = __decorate([
    (0, common_1.Injectable)()
], DatabaseService);
//# sourceMappingURL=database.service.js.map