"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuditService = void 0;
const common_1 = require("@nestjs/common");
const luxon_1 = require("luxon");
const node_path_1 = require("node:path");
const constants_1 = require("../constants");
const storage_core_1 = require("../cores/storage.core");
const decorators_1 = require("../decorators");
const audit_dto_1 = require("../dtos/audit.dto");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const asset_util_1 = require("../utils/asset.util");
const pagination_1 = require("../utils/pagination");
let AuditService = class AuditService extends base_service_1.BaseService {
    async handleCleanup() {
        await this.auditRepository.removeBefore(luxon_1.DateTime.now().minus(constants_1.AUDIT_LOG_MAX_DURATION).toJSDate());
        return enum_1.JobStatus.SUCCESS;
    }
    async getChecksums(dto) {
        const results = [];
        for (const filename of dto.filenames) {
            if (!storage_core_1.StorageCore.isImmichPath(filename)) {
                throw new common_1.BadRequestException(`Could not get the checksum of ${filename} because the file isn't accessible by Immich`);
            }
            const checksum = await this.cryptoRepository.hashFile(filename);
            results.push({ filename, checksum: checksum.toString('base64') });
        }
        return results;
    }
    async fixItems(items) {
        for (const { entityId: id, pathType, pathValue } of items) {
            if (!storage_core_1.StorageCore.isImmichPath(pathValue)) {
                throw new common_1.BadRequestException(`Could not fix item ${id} with path ${pathValue} because the file isn't accessible by Immich`);
            }
            switch (pathType) {
                case enum_1.AssetPathType.ENCODED_VIDEO: {
                    await this.assetRepository.update({ id, encodedVideoPath: pathValue });
                    break;
                }
                case enum_1.AssetPathType.PREVIEW: {
                    await this.assetRepository.upsertFile({ assetId: id, type: enum_1.AssetFileType.PREVIEW, path: pathValue });
                    break;
                }
                case enum_1.AssetPathType.THUMBNAIL: {
                    await this.assetRepository.upsertFile({ assetId: id, type: enum_1.AssetFileType.THUMBNAIL, path: pathValue });
                    break;
                }
                case enum_1.AssetPathType.ORIGINAL: {
                    await this.assetRepository.update({ id, originalPath: pathValue });
                    break;
                }
                case enum_1.AssetPathType.SIDECAR: {
                    await this.assetRepository.update({ id, sidecarPath: pathValue });
                    break;
                }
                case enum_1.PersonPathType.FACE: {
                    await this.personRepository.update({ id, thumbnailPath: pathValue });
                    break;
                }
                case enum_1.UserPathType.PROFILE: {
                    await this.userRepository.update(id, { profileImagePath: pathValue });
                    break;
                }
            }
        }
    }
    fullPath(filename) {
        return (0, node_path_1.resolve)(filename);
    }
    async getFileReport() {
        const hasFile = (items, filename) => items.has(filename) || items.has(this.fullPath(filename));
        const crawl = async (folder) => new Set(await this.storageRepository.crawl({
            includeHidden: true,
            pathsToCrawl: [storage_core_1.StorageCore.getBaseFolder(folder)],
        }));
        const uploadFiles = await crawl(enum_1.StorageFolder.UPLOAD);
        const libraryFiles = await crawl(enum_1.StorageFolder.LIBRARY);
        const thumbFiles = await crawl(enum_1.StorageFolder.THUMBNAILS);
        const videoFiles = await crawl(enum_1.StorageFolder.ENCODED_VIDEO);
        const profileFiles = await crawl(enum_1.StorageFolder.PROFILE);
        const allFiles = new Set();
        for (const list of [libraryFiles, thumbFiles, videoFiles, profileFiles, uploadFiles]) {
            for (const item of list) {
                allFiles.add(item);
            }
        }
        const track = (filename) => {
            if (!filename) {
                return;
            }
            allFiles.delete(filename);
            allFiles.delete(this.fullPath(filename));
        };
        this.logger.log(`Found ${libraryFiles.size} original files, ${thumbFiles.size} thumbnails, ${videoFiles.size} encoded videos, ${profileFiles.size} profile files`);
        const pagination = (0, pagination_1.usePagination)(constants_1.JOBS_ASSET_PAGINATION_SIZE, (options) => this.assetRepository.getAll(options, { withDeleted: true, withArchived: true }));
        let assetCount = 0;
        const orphans = [];
        for await (const assets of pagination) {
            assetCount += assets.length;
            for (const { id, files, originalPath, encodedVideoPath, isExternal, checksum } of assets) {
                const { fullsizeFile, previewFile, thumbnailFile } = (0, asset_util_1.getAssetFiles)(files);
                for (const file of [
                    originalPath,
                    fullsizeFile?.path,
                    previewFile?.path,
                    encodedVideoPath,
                    thumbnailFile?.path,
                ]) {
                    track(file);
                }
                const entity = { entityId: id, entityType: audit_dto_1.PathEntityType.ASSET, checksum: checksum.toString('base64') };
                if (originalPath &&
                    !hasFile(libraryFiles, originalPath) &&
                    !hasFile(uploadFiles, originalPath) &&
                    !hasFile(videoFiles, originalPath) &&
                    !isExternal) {
                    orphans.push({ ...entity, pathType: enum_1.AssetPathType.ORIGINAL, pathValue: originalPath });
                }
                if (previewFile && !hasFile(thumbFiles, previewFile.path)) {
                    orphans.push({ ...entity, pathType: enum_1.AssetPathType.PREVIEW, pathValue: previewFile.path });
                }
                if (thumbnailFile && !hasFile(thumbFiles, thumbnailFile.path)) {
                    orphans.push({ ...entity, pathType: enum_1.AssetPathType.THUMBNAIL, pathValue: thumbnailFile.path });
                }
                if (encodedVideoPath && !hasFile(videoFiles, encodedVideoPath)) {
                    orphans.push({ ...entity, pathType: enum_1.AssetPathType.THUMBNAIL, pathValue: encodedVideoPath });
                }
            }
        }
        const users = await this.userRepository.getList();
        for (const { id, profileImagePath } of users) {
            track(profileImagePath);
            const entity = { entityId: id, entityType: audit_dto_1.PathEntityType.USER };
            if (profileImagePath && !hasFile(profileFiles, profileImagePath)) {
                orphans.push({ ...entity, pathType: enum_1.UserPathType.PROFILE, pathValue: profileImagePath });
            }
        }
        let peopleCount = 0;
        for await (const { id, thumbnailPath } of this.personRepository.getAll()) {
            track(thumbnailPath);
            const entity = { entityId: id, entityType: audit_dto_1.PathEntityType.PERSON };
            if (thumbnailPath && !hasFile(thumbFiles, thumbnailPath)) {
                orphans.push({ ...entity, pathType: enum_1.PersonPathType.FACE, pathValue: thumbnailPath });
            }
            if (peopleCount === constants_1.JOBS_ASSET_PAGINATION_SIZE) {
                this.logger.log(`Found ${assetCount} assets, ${users.length} users, ${peopleCount} people`);
                peopleCount = 0;
            }
        }
        this.logger.log(`Found ${assetCount} assets, ${users.length} users, ${peopleCount} people`);
        const extras = [];
        for (const file of allFiles) {
            extras.push(file);
        }
        for (const orphan of orphans) {
            orphan.pathValue = this.fullPath(orphan.pathValue);
        }
        return { orphans, extras };
    }
};
exports.AuditService = AuditService;
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.CLEAN_OLD_AUDIT_LOGS, queue: enum_1.QueueName.BACKGROUND_TASK }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], AuditService.prototype, "handleCleanup", null);
exports.AuditService = AuditService = __decorate([
    (0, common_1.Injectable)()
], AuditService);
//# sourceMappingURL=audit.service.js.map