"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetService = void 0;
const common_1 = require("@nestjs/common");
const lodash_1 = __importDefault(require("lodash"));
const luxon_1 = require("luxon");
const constants_1 = require("../constants");
const decorators_1 = require("../decorators");
const asset_response_dto_1 = require("../dtos/asset-response.dto");
const asset_dto_1 = require("../dtos/asset.dto");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const asset_util_1 = require("../utils/asset.util");
let AssetService = class AssetService extends base_service_1.BaseService {
    async getMemoryLane(auth, dto) {
        const partnerIds = await (0, asset_util_1.getMyPartnerIds)({
            userId: auth.user.id,
            repository: this.partnerRepository,
            timelineEnabled: true,
        });
        const userIds = [auth.user.id, ...partnerIds];
        const groups = await this.assetRepository.getByDayOfYear(userIds, dto);
        return groups.map(({ year, assets }) => {
            const yearsAgo = luxon_1.DateTime.utc().year - year;
            return {
                yearsAgo,
                title: `${yearsAgo} year${yearsAgo > 1 ? 's' : ''} ago`,
                assets: assets.map((asset) => (0, asset_response_dto_1.mapAsset)(asset, { auth })),
            };
        });
    }
    async getStatistics(auth, dto) {
        const stats = await this.assetRepository.getStatistics(auth.user.id, dto);
        return (0, asset_dto_1.mapStats)(stats);
    }
    async getRandom(auth, count) {
        const partnerIds = await (0, asset_util_1.getMyPartnerIds)({
            userId: auth.user.id,
            repository: this.partnerRepository,
            timelineEnabled: true,
        });
        const assets = await this.assetRepository.getRandom([auth.user.id, ...partnerIds], count);
        return assets.map((a) => (0, asset_response_dto_1.mapAsset)(a, { auth }));
    }
    async getUserAssetsByDeviceId(auth, deviceId) {
        return this.assetRepository.getAllByDeviceId(auth.user.id, deviceId);
    }
    async get(auth, id) {
        await this.requireAccess({ auth, permission: enum_1.Permission.ASSET_READ, ids: [id] });
        const asset = await this.assetRepository.getById(id, {
            exifInfo: true,
            owner: true,
            faces: { person: true },
            stack: { assets: true },
            tags: true,
        });
        if (!asset) {
            throw new common_1.BadRequestException('Asset not found');
        }
        if (auth.sharedLink && !auth.sharedLink.showExif) {
            return (0, asset_response_dto_1.mapAsset)(asset, { stripMetadata: true, withStack: true, auth });
        }
        const data = (0, asset_response_dto_1.mapAsset)(asset, { withStack: true, auth });
        if (auth.sharedLink) {
            delete data.owner;
        }
        if (data.ownerId !== auth.user.id || auth.sharedLink) {
            data.people = [];
        }
        return data;
    }
    async update(auth, id, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.ASSET_UPDATE, ids: [id] });
        const { description, dateTimeOriginal, latitude, longitude, rating, ...rest } = dto;
        const repos = { asset: this.assetRepository, event: this.eventRepository };
        let previousMotion = null;
        if (rest.livePhotoVideoId) {
            await (0, asset_util_1.onBeforeLink)(repos, { userId: auth.user.id, livePhotoVideoId: rest.livePhotoVideoId });
        }
        else if (rest.livePhotoVideoId === null) {
            const asset = await this.findOrFail(id);
            if (asset.livePhotoVideoId) {
                previousMotion = await (0, asset_util_1.onBeforeUnlink)(repos, { livePhotoVideoId: asset.livePhotoVideoId });
            }
        }
        await this.updateMetadata({ id, description, dateTimeOriginal, latitude, longitude, rating });
        const asset = await this.assetRepository.update({ id, ...rest });
        if (previousMotion) {
            await (0, asset_util_1.onAfterUnlink)(repos, { userId: auth.user.id, livePhotoVideoId: previousMotion.id });
        }
        if (!asset) {
            throw new common_1.BadRequestException('Asset not found');
        }
        return (0, asset_response_dto_1.mapAsset)(asset, { auth });
    }
    async updateAll(auth, dto) {
        const { ids, dateTimeOriginal, latitude, longitude, ...options } = dto;
        await this.requireAccess({ auth, permission: enum_1.Permission.ASSET_UPDATE, ids });
        if (dateTimeOriginal !== undefined || latitude !== undefined || longitude !== undefined) {
            await this.assetRepository.updateAllExif(ids, { dateTimeOriginal, latitude, longitude });
            await this.jobRepository.queueAll(ids.map((id) => ({ name: enum_1.JobName.SIDECAR_WRITE, data: { id, dateTimeOriginal, latitude, longitude } })));
        }
        if (options.isArchived != undefined ||
            options.isFavorite != undefined ||
            options.duplicateId != undefined ||
            options.rating != undefined) {
            await this.assetRepository.updateAll(ids, options);
        }
    }
    async handleAssetDeletionCheck() {
        const config = await this.getConfig({ withCache: false });
        const trashedDays = config.trash.enabled ? config.trash.days : 0;
        const trashedBefore = luxon_1.DateTime.now()
            .minus(luxon_1.Duration.fromObject({ days: trashedDays }))
            .toJSDate();
        let chunk = [];
        const queueChunk = async () => {
            if (chunk.length > 0) {
                await this.jobRepository.queueAll(chunk.map(({ id, isOffline }) => ({
                    name: enum_1.JobName.ASSET_DELETION,
                    data: { id, deleteOnDisk: !isOffline },
                })));
                chunk = [];
            }
        };
        const assets = this.assetJobRepository.streamForDeletedJob(trashedBefore);
        for await (const asset of assets) {
            chunk.push(asset);
            if (chunk.length >= constants_1.JOBS_ASSET_PAGINATION_SIZE) {
                await queueChunk();
            }
        }
        await queueChunk();
        return enum_1.JobStatus.SUCCESS;
    }
    async handleAssetDeletion(job) {
        const { id, deleteOnDisk } = job;
        const asset = await this.assetRepository.getById(id, {
            faces: { person: true },
            library: true,
            stack: { assets: true },
            exifInfo: true,
            files: true,
        });
        if (!asset) {
            return enum_1.JobStatus.FAILED;
        }
        if (asset.stack?.primaryAssetId === id) {
            const stackAssetIds = asset.stack?.assets.map((a) => a.id) ?? [];
            if (stackAssetIds.length > 2) {
                const newPrimaryAssetId = stackAssetIds.find((a) => a !== id);
                await this.stackRepository.update(asset.stack.id, {
                    id: asset.stack.id,
                    primaryAssetId: newPrimaryAssetId,
                });
            }
            else {
                await this.stackRepository.delete(asset.stack.id);
            }
        }
        await this.assetRepository.remove(asset);
        if (!asset.libraryId) {
            await this.userRepository.updateUsage(asset.ownerId, -(asset.exifInfo?.fileSizeInByte || 0));
        }
        await this.eventRepository.emit('asset.delete', { assetId: id, userId: asset.ownerId });
        if (asset.livePhotoVideoId) {
            const count = await this.assetRepository.getLivePhotoCount(asset.livePhotoVideoId);
            if (count === 0) {
                await this.jobRepository.queue({
                    name: enum_1.JobName.ASSET_DELETION,
                    data: { id: asset.livePhotoVideoId, deleteOnDisk },
                });
            }
        }
        const { fullsizeFile, previewFile, thumbnailFile } = (0, asset_util_1.getAssetFiles)(asset.files ?? []);
        const files = [thumbnailFile?.path, previewFile?.path, fullsizeFile?.path, asset.encodedVideoPath];
        if (deleteOnDisk) {
            files.push(asset.sidecarPath, asset.originalPath);
        }
        await this.jobRepository.queue({ name: enum_1.JobName.DELETE_FILES, data: { files } });
        return enum_1.JobStatus.SUCCESS;
    }
    async deleteAll(auth, dto) {
        const { ids, force } = dto;
        await this.requireAccess({ auth, permission: enum_1.Permission.ASSET_DELETE, ids });
        await this.assetRepository.updateAll(ids, {
            deletedAt: new Date(),
            status: force ? enum_1.AssetStatus.DELETED : enum_1.AssetStatus.TRASHED,
        });
        await this.eventRepository.emit(force ? 'assets.delete' : 'assets.trash', { assetIds: ids, userId: auth.user.id });
    }
    async run(auth, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.ASSET_UPDATE, ids: dto.assetIds });
        const jobs = [];
        for (const id of dto.assetIds) {
            switch (dto.name) {
                case asset_dto_1.AssetJobName.REFRESH_FACES: {
                    jobs.push({ name: enum_1.JobName.FACE_DETECTION, data: { id } });
                    break;
                }
                case asset_dto_1.AssetJobName.REFRESH_METADATA: {
                    jobs.push({ name: enum_1.JobName.METADATA_EXTRACTION, data: { id } });
                    break;
                }
                case asset_dto_1.AssetJobName.REGENERATE_THUMBNAIL: {
                    jobs.push({ name: enum_1.JobName.GENERATE_THUMBNAILS, data: { id } });
                    break;
                }
                case asset_dto_1.AssetJobName.TRANSCODE_VIDEO: {
                    jobs.push({ name: enum_1.JobName.VIDEO_CONVERSION, data: { id } });
                    break;
                }
            }
        }
        await this.jobRepository.queueAll(jobs);
    }
    async findOrFail(id) {
        const asset = await this.assetRepository.getById(id);
        if (!asset) {
            throw new common_1.BadRequestException('Asset not found');
        }
        return asset;
    }
    async updateMetadata(dto) {
        const { id, description, dateTimeOriginal, latitude, longitude, rating } = dto;
        const writes = lodash_1.default.omitBy({ description, dateTimeOriginal, latitude, longitude, rating }, lodash_1.default.isUndefined);
        if (Object.keys(writes).length > 0) {
            await this.assetRepository.upsertExif({ assetId: id, ...writes });
            await this.jobRepository.queue({ name: enum_1.JobName.SIDECAR_WRITE, data: { id, ...writes } });
        }
    }
};
exports.AssetService = AssetService;
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.ASSET_DELETION_CHECK, queue: enum_1.QueueName.BACKGROUND_TASK }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], AssetService.prototype, "handleAssetDeletionCheck", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.ASSET_DELETION, queue: enum_1.QueueName.BACKGROUND_TASK }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], AssetService.prototype, "handleAssetDeletion", null);
exports.AssetService = AssetService = __decorate([
    (0, common_1.Injectable)()
], AssetService);
//# sourceMappingURL=asset.service.js.map