"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetMediaService = void 0;
const common_1 = require("@nestjs/common");
const node_path_1 = require("node:path");
const sanitize_filename_1 = __importDefault(require("sanitize-filename"));
const storage_core_1 = require("../cores/storage.core");
const asset_media_response_dto_1 = require("../dtos/asset-media-response.dto");
const asset_media_dto_1 = require("../dtos/asset-media.dto");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const access_1 = require("../utils/access");
const asset_util_1 = require("../utils/asset.util");
const database_1 = require("../utils/database");
const file_1 = require("../utils/file");
const mime_types_1 = require("../utils/mime-types");
const request_1 = require("../utils/request");
let AssetMediaService = class AssetMediaService extends base_service_1.BaseService {
    async getUploadAssetIdByChecksum(auth, checksum) {
        if (!checksum) {
            return;
        }
        const assetId = await this.assetRepository.getUploadAssetIdByChecksum(auth.user.id, (0, request_1.fromChecksum)(checksum));
        if (!assetId) {
            return;
        }
        return { id: assetId, status: asset_media_response_dto_1.AssetMediaStatus.DUPLICATE };
    }
    canUploadFile({ auth, fieldName, file }) {
        (0, access_1.requireUploadAccess)(auth);
        const filename = file.originalName;
        switch (fieldName) {
            case asset_media_dto_1.UploadFieldName.ASSET_DATA: {
                if (mime_types_1.mimeTypes.isAsset(filename)) {
                    return true;
                }
                break;
            }
            case asset_media_dto_1.UploadFieldName.SIDECAR_DATA: {
                if (mime_types_1.mimeTypes.isSidecar(filename)) {
                    return true;
                }
                break;
            }
            case asset_media_dto_1.UploadFieldName.PROFILE_DATA: {
                if (mime_types_1.mimeTypes.isProfile(filename)) {
                    return true;
                }
                break;
            }
        }
        this.logger.error(`Unsupported file type ${filename}`);
        throw new common_1.BadRequestException(`Unsupported file type ${filename}`);
    }
    getUploadFilename({ auth, fieldName, file }) {
        (0, access_1.requireUploadAccess)(auth);
        const originalExtension = (0, node_path_1.extname)(file.originalName);
        const lookup = {
            [asset_media_dto_1.UploadFieldName.ASSET_DATA]: originalExtension,
            [asset_media_dto_1.UploadFieldName.SIDECAR_DATA]: '.xmp',
            [asset_media_dto_1.UploadFieldName.PROFILE_DATA]: originalExtension,
        };
        return (0, sanitize_filename_1.default)(`${file.uuid}${lookup[fieldName]}`);
    }
    getUploadFolder({ auth, fieldName, file }) {
        auth = (0, access_1.requireUploadAccess)(auth);
        let folder = storage_core_1.StorageCore.getNestedFolder(enum_1.StorageFolder.UPLOAD, auth.user.id, file.uuid);
        if (fieldName === asset_media_dto_1.UploadFieldName.PROFILE_DATA) {
            folder = storage_core_1.StorageCore.getFolderLocation(enum_1.StorageFolder.PROFILE, auth.user.id);
        }
        this.storageRepository.mkdirSync(folder);
        return folder;
    }
    async onUploadError(request, file) {
        const uploadFilename = this.getUploadFilename((0, asset_util_1.asRequest)(request, file));
        const uploadFolder = this.getUploadFolder((0, asset_util_1.asRequest)(request, file));
        const uploadPath = `${uploadFolder}/${uploadFilename}`;
        await this.jobRepository.queue({ name: enum_1.JobName.DELETE_FILES, data: { files: [uploadPath] } });
    }
    async uploadAsset(auth, dto, file, sidecarFile) {
        try {
            await this.requireAccess({
                auth,
                permission: enum_1.Permission.ASSET_UPLOAD,
                ids: [auth.user.id],
            });
            this.requireQuota(auth, file.size);
            if (dto.livePhotoVideoId) {
                await (0, asset_util_1.onBeforeLink)({ asset: this.assetRepository, event: this.eventRepository }, { userId: auth.user.id, livePhotoVideoId: dto.livePhotoVideoId });
            }
            const asset = await this.create(auth.user.id, dto, file, sidecarFile);
            await this.userRepository.updateUsage(auth.user.id, file.size);
            return { id: asset.id, status: asset_media_response_dto_1.AssetMediaStatus.CREATED };
        }
        catch (error) {
            return this.handleUploadError(error, auth, file, sidecarFile);
        }
    }
    async replaceAsset(auth, id, dto, file, sidecarFile) {
        try {
            await this.requireAccess({ auth, permission: enum_1.Permission.ASSET_UPDATE, ids: [id] });
            const asset = await this.assetRepository.getById(id);
            if (!asset) {
                throw new Error('Asset not found');
            }
            this.requireQuota(auth, file.size);
            await this.replaceFileData(asset.id, dto, file, sidecarFile?.originalPath);
            const copiedPhoto = await this.createCopy(asset);
            await this.assetRepository.updateAll([copiedPhoto.id], { deletedAt: new Date(), status: enum_1.AssetStatus.TRASHED });
            await this.eventRepository.emit('asset.trash', { assetId: copiedPhoto.id, userId: auth.user.id });
            await this.userRepository.updateUsage(auth.user.id, file.size);
            return { status: asset_media_response_dto_1.AssetMediaStatus.REPLACED, id: copiedPhoto.id };
        }
        catch (error) {
            return this.handleUploadError(error, auth, file, sidecarFile);
        }
    }
    async downloadOriginal(auth, id) {
        await this.requireAccess({ auth, permission: enum_1.Permission.ASSET_DOWNLOAD, ids: [id] });
        const asset = await this.findOrFail(id);
        return new file_1.ImmichFileResponse({
            path: asset.originalPath,
            contentType: mime_types_1.mimeTypes.lookup(asset.originalPath),
            cacheControl: enum_1.CacheControl.PRIVATE_WITH_CACHE,
        });
    }
    async viewThumbnail(auth, id, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.ASSET_VIEW, ids: [id] });
        const asset = await this.findOrFail(id);
        const size = dto.size ?? asset_media_dto_1.AssetMediaSize.THUMBNAIL;
        const { thumbnailFile, previewFile, fullsizeFile } = (0, asset_util_1.getAssetFiles)(asset.files ?? []);
        let filepath = previewFile?.path;
        if (size === asset_media_dto_1.AssetMediaSize.THUMBNAIL && thumbnailFile) {
            filepath = thumbnailFile.path;
        }
        else if (size === asset_media_dto_1.AssetMediaSize.FULLSIZE) {
            if (mime_types_1.mimeTypes.isWebSupportedImage(asset.originalPath)) {
                return { targetSize: 'original' };
            }
            if (!fullsizeFile) {
                return { targetSize: asset_media_dto_1.AssetMediaSize.PREVIEW };
            }
            filepath = fullsizeFile.path;
        }
        if (!filepath) {
            throw new common_1.NotFoundException('Asset media not found');
        }
        let fileName = (0, file_1.getFileNameWithoutExtension)(asset.originalFileName);
        fileName += `_${size}`;
        fileName += (0, file_1.getFilenameExtension)(filepath);
        return new file_1.ImmichFileResponse({
            fileName,
            path: filepath,
            contentType: mime_types_1.mimeTypes.lookup(filepath),
            cacheControl: enum_1.CacheControl.PRIVATE_WITH_CACHE,
        });
    }
    async playbackVideo(auth, id) {
        await this.requireAccess({ auth, permission: enum_1.Permission.ASSET_VIEW, ids: [id] });
        const asset = await this.findOrFail(id);
        if (asset.type !== enum_1.AssetType.VIDEO) {
            throw new common_1.BadRequestException('Asset is not a video');
        }
        const filepath = asset.encodedVideoPath || asset.originalPath;
        return new file_1.ImmichFileResponse({
            path: filepath,
            contentType: mime_types_1.mimeTypes.lookup(filepath),
            cacheControl: enum_1.CacheControl.PRIVATE_WITH_CACHE,
        });
    }
    async checkExistingAssets(auth, checkExistingAssetsDto) {
        const existingIds = await this.assetRepository.getByDeviceIds(auth.user.id, checkExistingAssetsDto.deviceId, checkExistingAssetsDto.deviceAssetIds);
        return { existingIds };
    }
    async bulkUploadCheck(auth, dto) {
        const checksums = dto.assets.map((asset) => (0, request_1.fromChecksum)(asset.checksum));
        const results = await this.assetRepository.getByChecksums(auth.user.id, checksums);
        const checksumMap = {};
        for (const { id, deletedAt, checksum } of results) {
            checksumMap[checksum.toString('hex')] = { id, isTrashed: !!deletedAt };
        }
        return {
            results: dto.assets.map(({ id, checksum }) => {
                const duplicate = checksumMap[(0, request_1.fromChecksum)(checksum).toString('hex')];
                if (duplicate) {
                    return {
                        id,
                        action: asset_media_response_dto_1.AssetUploadAction.REJECT,
                        reason: asset_media_response_dto_1.AssetRejectReason.DUPLICATE,
                        assetId: duplicate.id,
                        isTrashed: duplicate.isTrashed,
                    };
                }
                return {
                    id,
                    action: asset_media_response_dto_1.AssetUploadAction.ACCEPT,
                };
            }),
        };
    }
    async handleUploadError(error, auth, file, sidecarFile) {
        await this.jobRepository.queue({
            name: enum_1.JobName.DELETE_FILES,
            data: { files: [file.originalPath, sidecarFile?.originalPath] },
        });
        if (error.constraint_name === database_1.ASSET_CHECKSUM_CONSTRAINT) {
            const duplicateId = await this.assetRepository.getUploadAssetIdByChecksum(auth.user.id, file.checksum);
            if (!duplicateId) {
                this.logger.error(`Error locating duplicate for checksum constraint`);
                throw new common_1.InternalServerErrorException();
            }
            return { status: asset_media_response_dto_1.AssetMediaStatus.DUPLICATE, id: duplicateId };
        }
        this.logger.error(`Error uploading file ${error}`, error?.stack);
        throw error;
    }
    async replaceFileData(assetId, dto, file, sidecarPath) {
        await this.assetRepository.update({
            id: assetId,
            checksum: file.checksum,
            originalPath: file.originalPath,
            type: mime_types_1.mimeTypes.assetType(file.originalPath),
            originalFileName: file.originalName,
            deviceAssetId: dto.deviceAssetId,
            deviceId: dto.deviceId,
            fileCreatedAt: dto.fileCreatedAt,
            fileModifiedAt: dto.fileModifiedAt,
            localDateTime: dto.fileCreatedAt,
            duration: dto.duration || null,
            livePhotoVideoId: null,
            sidecarPath: sidecarPath || null,
        });
        await this.storageRepository.utimes(file.originalPath, new Date(), new Date(dto.fileModifiedAt));
        await this.assetRepository.upsertExif({ assetId, fileSizeInByte: file.size });
        await this.jobRepository.queue({
            name: enum_1.JobName.METADATA_EXTRACTION,
            data: { id: assetId, source: 'upload' },
        });
    }
    async createCopy(asset) {
        const created = await this.assetRepository.create({
            ownerId: asset.ownerId,
            originalPath: asset.originalPath,
            originalFileName: asset.originalFileName,
            libraryId: asset.libraryId,
            deviceAssetId: asset.deviceAssetId,
            deviceId: asset.deviceId,
            type: asset.type,
            checksum: asset.checksum,
            fileCreatedAt: asset.fileCreatedAt,
            localDateTime: asset.localDateTime,
            fileModifiedAt: asset.fileModifiedAt,
            livePhotoVideoId: asset.livePhotoVideoId,
            sidecarPath: asset.sidecarPath,
        });
        const { size } = await this.storageRepository.stat(created.originalPath);
        await this.assetRepository.upsertExif({ assetId: created.id, fileSizeInByte: size });
        await this.jobRepository.queue({ name: enum_1.JobName.METADATA_EXTRACTION, data: { id: created.id, source: 'copy' } });
        return created;
    }
    async create(ownerId, dto, file, sidecarFile) {
        const asset = await this.assetRepository.create({
            ownerId,
            libraryId: null,
            checksum: file.checksum,
            originalPath: file.originalPath,
            deviceAssetId: dto.deviceAssetId,
            deviceId: dto.deviceId,
            fileCreatedAt: dto.fileCreatedAt,
            fileModifiedAt: dto.fileModifiedAt,
            localDateTime: dto.fileCreatedAt,
            type: mime_types_1.mimeTypes.assetType(file.originalPath),
            isFavorite: dto.isFavorite,
            isArchived: dto.isArchived ?? false,
            duration: dto.duration || null,
            isVisible: dto.isVisible ?? true,
            livePhotoVideoId: dto.livePhotoVideoId,
            originalFileName: file.originalName,
            sidecarPath: sidecarFile?.originalPath,
        });
        if (sidecarFile) {
            await this.storageRepository.utimes(sidecarFile.originalPath, new Date(), new Date(dto.fileModifiedAt));
        }
        await this.storageRepository.utimes(file.originalPath, new Date(), new Date(dto.fileModifiedAt));
        await this.assetRepository.upsertExif({ assetId: asset.id, fileSizeInByte: file.size });
        await this.jobRepository.queue({ name: enum_1.JobName.METADATA_EXTRACTION, data: { id: asset.id, source: 'upload' } });
        return asset;
    }
    requireQuota(auth, size) {
        if (auth.user.quotaSizeInBytes !== null && auth.user.quotaSizeInBytes < auth.user.quotaUsageInBytes + size) {
            throw new common_1.BadRequestException('Quota has been exceeded!');
        }
    }
    async findOrFail(id) {
        const asset = await this.assetRepository.getById(id, { files: true });
        if (!asset) {
            throw new common_1.NotFoundException('Asset not found');
        }
        return asset;
    }
};
exports.AssetMediaService = AssetMediaService;
exports.AssetMediaService = AssetMediaService = __decorate([
    (0, common_1.Injectable)()
], AssetMediaService);
//# sourceMappingURL=asset-media.service.js.map