"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var DatabaseRepository_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseRepository = void 0;
const common_1 = require("@nestjs/common");
const async_lock_1 = __importDefault(require("async-lock"));
const kysely_1 = require("kysely");
const nestjs_kysely_1 = require("nestjs-kysely");
const promises_1 = require("node:fs/promises");
const node_path_1 = require("node:path");
const semver_1 = __importDefault(require("semver"));
const constants_1 = require("../constants");
const decorators_1 = require("../decorators");
const enum_1 = require("../enum");
const config_repository_1 = require("./config.repository");
const logging_repository_1 = require("./logging.repository");
const validation_1 = require("../validation");
const typeorm_1 = require("typeorm");
let DatabaseRepository = DatabaseRepository_1 = class DatabaseRepository {
    db;
    logger;
    configRepository;
    vectorExtension;
    asyncLock = new async_lock_1.default();
    constructor(db, logger, configRepository) {
        this.db = db;
        this.logger = logger;
        this.configRepository = configRepository;
        this.vectorExtension = configRepository.getEnv().database.vectorExtension;
        this.logger.setContext(DatabaseRepository_1.name);
    }
    async shutdown() {
        await this.db.destroy();
    }
    async getExtensionVersion(extension) {
        const { rows } = await (0, kysely_1.sql) `
      SELECT default_version as "availableVersion", installed_version as "installedVersion"
      FROM pg_available_extensions
      WHERE name = ${extension}
    `.execute(this.db);
        return rows[0] ?? { availableVersion: null, installedVersion: null };
    }
    getExtensionVersionRange(extension) {
        return extension === enum_1.DatabaseExtension.VECTORS ? constants_1.VECTORS_VERSION_RANGE : constants_1.VECTOR_VERSION_RANGE;
    }
    async getPostgresVersion() {
        const { rows } = await (0, kysely_1.sql) `SHOW server_version`.execute(this.db);
        return rows[0].server_version;
    }
    getPostgresVersionRange() {
        return constants_1.POSTGRES_VERSION_RANGE;
    }
    async createExtension(extension) {
        await (0, kysely_1.sql) `CREATE EXTENSION IF NOT EXISTS ${kysely_1.sql.raw(extension)}`.execute(this.db);
    }
    async updateVectorExtension(extension, targetVersion) {
        const { availableVersion, installedVersion } = await this.getExtensionVersion(extension);
        if (!installedVersion) {
            throw new Error(`${constants_1.EXTENSION_NAMES[extension]} extension is not installed`);
        }
        if (!availableVersion) {
            throw new Error(`No available version for ${constants_1.EXTENSION_NAMES[extension]} extension`);
        }
        targetVersion ??= availableVersion;
        const isVectors = extension === enum_1.DatabaseExtension.VECTORS;
        let restartRequired = false;
        await this.db.transaction().execute(async (tx) => {
            await this.setSearchPath(tx);
            if (isVectors && installedVersion === '0.1.1') {
                await this.setExtVersion(tx, enum_1.DatabaseExtension.VECTORS, '0.1.11');
            }
            const isSchemaUpgrade = semver_1.default.satisfies(installedVersion, '0.1.1 || 0.1.11');
            if (isSchemaUpgrade && isVectors) {
                await this.updateVectorsSchema(tx);
            }
            await (0, kysely_1.sql) `ALTER EXTENSION ${kysely_1.sql.raw(extension)} UPDATE TO ${kysely_1.sql.lit(targetVersion)}`.execute(tx);
            const diff = semver_1.default.diff(installedVersion, targetVersion);
            if (isVectors && diff && ['minor', 'major'].includes(diff)) {
                await (0, kysely_1.sql) `SELECT pgvectors_upgrade()`.execute(tx);
                restartRequired = true;
            }
            else {
                await this.reindex(enum_1.VectorIndex.CLIP);
                await this.reindex(enum_1.VectorIndex.FACE);
            }
        });
        return { restartRequired };
    }
    async reindex(index) {
        try {
            await (0, kysely_1.sql) `REINDEX INDEX ${kysely_1.sql.raw(index)}`.execute(this.db);
        }
        catch (error) {
            if (this.vectorExtension !== enum_1.DatabaseExtension.VECTORS) {
                throw error;
            }
            this.logger.warn(`Could not reindex index ${index}. Attempting to auto-fix.`);
            const table = await this.getIndexTable(index);
            const dimSize = await this.getDimSize(table);
            await this.db.transaction().execute(async (tx) => {
                await this.setSearchPath(tx);
                await (0, kysely_1.sql) `DROP INDEX IF EXISTS ${kysely_1.sql.raw(index)}`.execute(tx);
                await (0, kysely_1.sql) `ALTER TABLE ${kysely_1.sql.raw(table)} ALTER COLUMN embedding SET DATA TYPE real[]`.execute(tx);
                await (0, kysely_1.sql) `ALTER TABLE ${kysely_1.sql.raw(table)} ALTER COLUMN embedding SET DATA TYPE vector(${kysely_1.sql.raw(String(dimSize))})`.execute(tx);
                await (0, kysely_1.sql) `SET vectors.pgvector_compatibility=on`.execute(tx);
                await (0, kysely_1.sql) `
          CREATE INDEX IF NOT EXISTS ${kysely_1.sql.raw(index)} ON ${kysely_1.sql.raw(table)}
          USING hnsw (embedding vector_cosine_ops)
          WITH (ef_construction = 300, m = 16)
        `.execute(tx);
            });
        }
    }
    async shouldReindex(name) {
        if (this.vectorExtension !== enum_1.DatabaseExtension.VECTORS) {
            return false;
        }
        try {
            const { rows } = await (0, kysely_1.sql) `SELECT idx_status FROM pg_vector_index_stat WHERE indexname = ${name}`.execute(this.db);
            return rows[0]?.idx_status === 'UPGRADE';
        }
        catch (error) {
            const message = error.message;
            if (message.includes('index is not existing')) {
                return true;
            }
            else if (message.includes('relation "pg_vector_index_stat" does not exist')) {
                return false;
            }
            throw error;
        }
    }
    async setSearchPath(tx) {
        await (0, kysely_1.sql) `SET search_path TO "$user", public, vectors`.execute(tx);
    }
    async setExtVersion(tx, extName, version) {
        await (0, kysely_1.sql) `UPDATE pg_catalog.pg_extension SET extversion = ${version} WHERE extname = ${extName}`.execute(tx);
    }
    async getIndexTable(index) {
        const { rows } = await (0, kysely_1.sql) `SELECT relname FROM pg_stat_all_indexes WHERE indexrelname = ${index}`.execute(this.db);
        const table = rows[0]?.relname;
        if (!table) {
            throw new Error(`Could not find table for index ${index}`);
        }
        return table;
    }
    async updateVectorsSchema(tx) {
        const extension = enum_1.DatabaseExtension.VECTORS;
        await (0, kysely_1.sql) `CREATE SCHEMA IF NOT EXISTS ${extension}`.execute(tx);
        await (0, kysely_1.sql) `UPDATE pg_catalog.pg_extension SET extrelocatable = true WHERE extname = ${extension}`.execute(tx);
        await (0, kysely_1.sql) `ALTER EXTENSION vectors SET SCHEMA vectors`.execute(tx);
        await (0, kysely_1.sql) `UPDATE pg_catalog.pg_extension SET extrelocatable = false WHERE extname = ${extension}`.execute(tx);
    }
    async getDimSize(table, column = 'embedding') {
        const { rows } = await (0, kysely_1.sql) `
      SELECT atttypmod as dimsize
      FROM pg_attribute f
        JOIN pg_class c ON c.oid = f.attrelid
      WHERE c.relkind = 'r'::char
        AND f.attnum > 0
        AND c.relname = ${table}
        AND f.attname = '${column}'
    `.execute(this.db);
        const dimSize = rows[0]?.dimsize;
        if (!(0, validation_1.isValidInteger)(dimSize, { min: 1, max: 2 ** 16 })) {
            throw new Error(`Could not retrieve dimension size`);
        }
        return dimSize;
    }
    async runMigrations(options) {
        const { database } = this.configRepository.getEnv();
        this.logger.log('Running migrations, this may take a while');
        const tableExists = (0, kysely_1.sql) `select to_regclass('migrations') as "result"`;
        const { rows } = await tableExists.execute(this.db);
        const hasTypeOrmMigrations = !!rows[0]?.result;
        if (hasTypeOrmMigrations) {
            const dist = (0, node_path_1.resolve)(`${__dirname}/..`);
            this.logger.debug('Running typeorm migrations');
            const dataSource = new typeorm_1.DataSource({
                type: 'postgres',
                entities: [],
                subscribers: [],
                migrations: [`${dist}/migrations` + '/*.{js,ts}'],
                migrationsRun: false,
                synchronize: false,
                connectTimeoutMS: 10_000,
                parseInt8: true,
                ...(database.config.connectionType === 'url'
                    ? { url: database.config.url }
                    : {
                        host: database.config.host,
                        port: database.config.port,
                        username: database.config.username,
                        password: database.config.password,
                        database: database.config.database,
                    }),
            });
            await dataSource.initialize();
            await dataSource.runMigrations(options);
            await dataSource.destroy();
            this.logger.debug('Finished running typeorm migrations');
        }
        this.logger.debug('Running kysely migrations');
        const migrator = new kysely_1.Migrator({
            db: this.db,
            migrationLockTableName: 'kysely_migrations_lock',
            migrationTableName: 'kysely_migrations',
            provider: new kysely_1.FileMigrationProvider({
                fs: { readdir: promises_1.readdir },
                path: { join: node_path_1.join },
                migrationFolder: (0, node_path_1.join)(__dirname, '..', 'schema/migrations'),
            }),
        });
        const { error, results } = await migrator.migrateToLatest();
        for (const result of results ?? []) {
            if (result.status === 'Success') {
                this.logger.log(`Migration "${result.migrationName}" succeeded`);
            }
            if (result.status === 'Error') {
                this.logger.warn(`Migration "${result.migrationName}" failed`);
            }
        }
        if (error) {
            this.logger.error(`Kysely migrations failed: ${error}`);
            throw error;
        }
        this.logger.debug('Finished running kysely migrations');
    }
    async withLock(lock, callback) {
        let res;
        await this.asyncLock.acquire(enum_1.DatabaseLock[lock], async () => {
            await this.db.connection().execute(async (connection) => {
                try {
                    await this.acquireLock(lock, connection);
                    res = await callback();
                }
                finally {
                    await this.releaseLock(lock, connection);
                }
            });
        });
        return res;
    }
    tryLock(lock) {
        return this.db.connection().execute(async (connection) => this.acquireTryLock(lock, connection));
    }
    isBusy(lock) {
        return this.asyncLock.isBusy(enum_1.DatabaseLock[lock]);
    }
    async wait(lock) {
        await this.asyncLock.acquire(enum_1.DatabaseLock[lock], () => { });
    }
    async acquireLock(lock, connection) {
        await (0, kysely_1.sql) `SELECT pg_advisory_lock(${lock})`.execute(connection);
    }
    async acquireTryLock(lock, connection) {
        const { rows } = await (0, kysely_1.sql) `SELECT pg_try_advisory_lock(${lock})`.execute(connection);
        return rows[0].pg_try_advisory_lock;
    }
    async releaseLock(lock, connection) {
        await (0, kysely_1.sql) `SELECT pg_advisory_unlock(${lock})`.execute(connection);
    }
};
exports.DatabaseRepository = DatabaseRepository;
__decorate([
    (0, decorators_1.GenerateSql)({ params: [enum_1.DatabaseExtension.VECTORS] }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], DatabaseRepository.prototype, "getExtensionVersion", null);
__decorate([
    (0, decorators_1.GenerateSql)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], DatabaseRepository.prototype, "getPostgresVersion", null);
__decorate([
    (0, decorators_1.GenerateSql)({ params: [enum_1.VectorIndex.CLIP] }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], DatabaseRepository.prototype, "shouldReindex", null);
exports.DatabaseRepository = DatabaseRepository = DatabaseRepository_1 = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, nestjs_kysely_1.InjectKysely)()),
    __metadata("design:paramtypes", [kysely_1.Kysely,
        logging_repository_1.LoggingRepository,
        config_repository_1.ConfigRepository])
], DatabaseRepository);
//# sourceMappingURL=database.repository.js.map