# DMS Quickshell Interface

QML-based desktop shell interface for DankMaterialShell providing panels, widgets, and overlays.

**See [root README](../README.md) for project overview and installation.**

## Architecture

**Modular QML Structure**
- `Modules/` - UI components (panels, widgets, overlays)
- `Services/` - System integration singletons (audio, network, bluetooth)
- `Widgets/` - Reusable UI controls
- `Common/` - Shared resources and themes

**Technology Stack**
- [Quickshell](https://quickshell.org/) - QML-based shell framework
- Qt/QtQuick - UI rendering and controls
- Material Design 3 - Design system and theming

## Development

**Run the shell:**
```bash
quickshell -p quickshell/
```

**Code formatting:**
```bash
qmlfmt -t 4 -i 4 -b 250 -w path/to/file.qml
qmllint **/*.qml
```

## Components

**Panels & Bars**
- `Modules/TopBar/` - Multi-monitor status bars with workspace switching
- `Modules/DankBar/` - Customizable widget bar with plugin support
- `Modules/Dock/` - Application dock with window management

**System Controls**
- `Modules/ControlCenter/` - WiFi, Bluetooth, audio, display settings
- `Modules/Notifications/` - Notification center with popups
- `Modules/Greetd/` - Login greeter interface

**Overlays**
- `Modules/Spotlight/` - Application and file launcher
- `Modules/Overview/` - Workspace overview
- `Modules/Lock/` - Screen lock system

**Utilities**
- `Modules/ProcessList/` - System monitoring and process management
- `Modules/Calendar/` - Calendar widget with event sync
- `Modules/Weather/` - Weather display

## Services

Singletons providing system integration:

**Media & Audio**
- `AudioService` - PipeWire/PulseAudio volume and device control
- `MprisController` - Media player integration

**Network**
- `NetworkService` - NetworkManager WiFi control
- `BluetoothService` - BlueZ Bluetooth management

**Display**
- `DisplayService` - Brightness control and night mode
- `WallpaperService` - Wallpaper management and effects

**System**
- `BatteryService` - Battery status and power profiles
- `IdleService` - Idle detection and inhibit locks
- `ClipboardService` - Clipboard history with images
- `DgopService` - System metrics (CPU, RAM, GPU)

**Integration**
- `NiriService` - Niri workspace integration
- `HyprlandService` - Hyprland workspace integration
- `PluginService` - Plugin discovery and lifecycle

## Widgets

Reusable Material Design 3 components in `Widgets/`:

- `DankIcon` - Icon component with Material font
- `DankSlider` - Enhanced slider with animations
- `DankToggle` - Toggle switch component
- `DankTabBar` - Tab bar implementation
- `DankGridView` - Grid layout with adaptive columns
- `DankListView` - Scrollable list view
- `DankTextField` - Text input with validation
- `DankDropdown` - Dropdown selection
- `DankPopout` - Base for overlay components
- `StateLayer` - Material interaction states

## Theming

**Dynamic Color Schemes**

Wallpaper-based theming using [matugen](https://github.com/InioX/matugen):

```qml
import qs.Common

Rectangle {
    color: Theme.container
    border.color: Theme.outline
}
```

Theme singleton provides Material Design 3 color system, spacing, fonts, and elevation.

**Application Themes**

Templates in `scripts/templates/` generate themes for:
- GTK 3/4
- Qt5/Qt6
- Alacritty, Kitty, Ghostty, Foot, Wezterm terminals
- VSCode/VSCodium
- Firefox

## Multi-Monitor Support

Per-monitor panel instances using Quickshell `Variants`:

```qml
Variants {
    model: Quickshell.screens
    PanelWindow {
        screen: modelData
        // Per-screen configuration
    }
}
```

Workspace switchers adapt to compositor (Niri/Hyprland).

## Plugin System

External plugins in `~/.config/DankMaterialShell/plugins/`:

**Widget plugins** - UI components in DankBar
**Daemon plugins** - Background processes without UI

Plugin manifest (`plugin.json`):
```json
{
    "id": "pluginId",
    "name": "Plugin Name",
    "version": "1.0.0",
    "type": "widget",
    "component": "./Widget.qml",
    "settings": "./Settings.qml",
    "permissions": ["settings_read", "settings_write"]
}
```

Plugins access `pluginService` for persistent data:
```qml
pluginService.savePluginData("pluginId", "key", value)
pluginService.loadPluginData("pluginId", "key", defaultValue)
```

## IPC Integration

Backend IPC socket communication:

```qml
import Quickshell.Io

Process {
    command: ["dms", "ipc", "call", "spotlight", "toggle"]
    running: true
}
```

Common IPC commands exposed through services for reactive property bindings.

## Code Conventions

**Component Structure:**
```qml
import QtQuick
import Quickshell
import qs.Common
import qs.Services

Item {
    id: root

    property type name: value

    signal customSignal(type param)

    Component { /* children */ }
}
```

**Services (Singletons):**
```qml
import QtQuick
import Quickshell
pragma Singleton
pragma ComponentBehavior: Bound

Singleton {
    id: root

    property bool featureAvailable: false
    property type currentValue: defaultValue

    function performAction(param) { /* implementation */ }
}
```

**Guidelines:**
- Use `Theme.propertyName` for consistent styling
- Bind directly to service properties for reactivity
- Use `DankIcon` for all icons
- Implement feature detection and graceful degradation
- 4-space indentation, no unnecessary comments

## Translation

Internationalization using POEditor:

```bash
export POEDITOR_API_TOKEN="token"
export POEDITOR_PROJECT_ID="id"
python3 scripts/i18nsync.py sync
```

Pre-commit hook checks translation sync status.

## License

MIT License - See [LICENSE](../LICENSE) for details.
