"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Manifest = exports.VERSION_MISMATCH = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const jsonschema = require("jsonschema");
const semver = require("semver");
const assembly = require("./cloud-assembly");
/* eslint-disable @typescript-eslint/no-var-requires */
/* eslint-disable @typescript-eslint/no-require-imports */
// this prefix is used by the CLI to identify this specific error.
// in which case we want to instruct the user to upgrade his CLI.
// see exec.ts#createAssembly
exports.VERSION_MISMATCH = 'Cloud assembly schema version mismatch';
/**
 * CLI version is created at build and release time
 *
 * It needs to be .gitignore'd, otherwise the projen 'no uncommitted
 * changes' self-check will fail, which means it needs to be generated
 * at build time if it doesn't already exist.
 */
const CLI_VERSION = require("../cli-version.json");
const ASSETS_SCHEMA = require("../schema/assets.schema.json");
const ASSEMBLY_SCHEMA = require("../schema/cloud-assembly.schema.json");
const INTEG_SCHEMA = require("../schema/integ.schema.json");
/**
 * Version is shared for both manifests
 */
const SCHEMA_VERSION = require("../schema/version.json");
/**
 * Protocol utility class.
 */
class Manifest {
    /**
     * Validates and saves the cloud assembly manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     */
    static saveAssemblyManifest(manifest, filePath) {
        Manifest.saveManifest(manifest, filePath, ASSEMBLY_SCHEMA, Manifest.patchStackTagsOnWrite);
    }
    /**
     * Load and validates the cloud assembly manifest from file.
     *
     * @param filePath - path to the manifest file.
     */
    static loadAssemblyManifest(filePath, options) {
        return Manifest.loadManifest(filePath, ASSEMBLY_SCHEMA, Manifest.patchStackTagsOnRead, options);
    }
    /**
     * Validates and saves the asset manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     */
    static saveAssetManifest(manifest, filePath) {
        Manifest.saveManifest(manifest, filePath, ASSETS_SCHEMA, Manifest.patchStackTagsOnRead);
    }
    /**
     * Load and validates the asset manifest from file.
     *
     * @param filePath - path to the manifest file.
     */
    static loadAssetManifest(filePath) {
        return this.loadManifest(filePath, ASSETS_SCHEMA);
    }
    /**
     * Validates and saves the integ manifest to file.
     *
     * @param manifest - manifest.
     * @param filePath - output file path.
     */
    static saveIntegManifest(manifest, filePath) {
        Manifest.saveManifest(manifest, filePath, INTEG_SCHEMA);
    }
    /**
     * Load and validates the integ manifest from file.
     *
     * @param filePath - path to the manifest file.
     */
    static loadIntegManifest(filePath) {
        const manifest = this.loadManifest(filePath, INTEG_SCHEMA);
        // Adding typing to `validate()` led to `loadManifest()` to properly infer
        // its return type, which indicated that the return type of this
        // function may be a lie. I could change the schema to make `testCases`
        // optional, but that will bump the major version of this package and I
        // don't want to do that. So instead, just make sure `testCases` is always there.
        return {
            ...manifest,
            testCases: manifest.testCases ?? [],
        };
    }
    /**
     * Fetch the current schema version number.
     */
    static version() {
        return `${SCHEMA_VERSION.revision}.0.0`;
    }
    /**
     * Return the CLI version that supports this Cloud Assembly Schema version
     */
    static cliVersion() {
        const version = CLI_VERSION.version;
        return version ? version : undefined;
    }
    /**
     * Deprecated
     * @deprecated use `saveAssemblyManifest()`
     */
    static save(manifest, filePath) {
        return this.saveAssemblyManifest(manifest, filePath);
    }
    /**
     * Deprecated
     * @deprecated use `loadAssemblyManifest()`
     */
    static load(filePath) {
        return this.loadAssemblyManifest(filePath);
    }
    static validate(manifest, schema, options) {
        function parseVersion(version) {
            const ver = semver.valid(version);
            if (!ver) {
                throw new Error(`Invalid semver string: "${version}"`);
            }
            return ver;
        }
        const maxSupported = semver.major(parseVersion(Manifest.version()));
        const actual = parseVersion(manifest.version);
        // first validate the version should be accepted. all versions within the same minor version are fine
        if (maxSupported < semver.major(actual) && !options?.skipVersionCheck) {
            // If we have a more specific error to throw than the generic one below, make sure to add that info.
            const cliVersion = manifest.minimumCliVersion;
            let cliWarning = '';
            if (cliVersion) {
                cliWarning = `. You need at least CLI version ${cliVersion} to read this manifest.`;
            }
            // we use a well known error prefix so that the CLI can identify this specific error
            // and print some more context to the user.
            throw new Error(`${exports.VERSION_MISMATCH}: Maximum schema version supported is ${maxSupported}.x.x, but found ${actual}${cliWarning}`);
        }
        // now validate the format is good.
        const validator = new jsonschema.Validator();
        const result = validator.validate(manifest, schema, {
            // does exist but is not in the TypeScript definitions
            nestedErrors: true,
            allowUnknownAttributes: false,
            preValidateProperty: Manifest.validateAssumeRoleAdditionalOptions,
        });
        let errors = result.errors;
        if (options?.skipEnumCheck) {
            // Enum validations aren't useful when
            errors = stripEnumErrors(errors);
        }
        if (errors.length > 0) {
            throw new Error(`Invalid assembly manifest:\n${errors.map((e) => e.stack).join('\n')}`);
        }
    }
    static saveManifest(manifest, filePath, schema, preprocess) {
        let withVersion = {
            ...manifest,
            version: Manifest.version(),
            minimumCliVersion: Manifest.cliVersion(),
        };
        Manifest.validate(withVersion, schema);
        if (preprocess) {
            withVersion = preprocess(withVersion);
        }
        fs.writeFileSync(filePath, JSON.stringify(withVersion, undefined, 2));
    }
    static loadManifest(filePath, schema, preprocess, options) {
        const contents = fs.readFileSync(filePath, { encoding: 'utf-8' });
        let obj;
        try {
            obj = JSON.parse(contents);
        }
        catch (e) {
            throw new Error(`${e.message}, while parsing ${JSON.stringify(contents)}`);
        }
        if (preprocess) {
            obj = preprocess(obj);
        }
        Manifest.validate(obj, schema, options);
        return obj;
    }
    /**
     * This requires some explaining...
     *
     * We previously used `{ Key, Value }` for the object that represents a stack tag. (Notice the casing)
     * @link https://github.com/aws/aws-cdk/blob/v1.27.0/packages/aws-cdk/lib/api/cxapp/stacks.ts#L427.
     *
     * When that object moved to this package, it had to be JSII compliant, which meant the property
     * names must be `camelCased`, and not `PascalCased`. This meant it no longer matches the structure in the `manifest.json` file.
     * In order to support current manifest files, we have to translate the `PascalCased` representation to the new `camelCased` one.
     *
     * Note that the serialization itself still writes `PascalCased` because it relates to how CloudFormation expects it.
     *
     * Ideally, we would start writing the `camelCased` and translate to how CloudFormation expects it when needed. But this requires nasty
     * backwards-compatibility code and it just doesn't seem to be worth the effort.
     */
    static patchStackTagsOnRead(manifest) {
        return Manifest.replaceStackTags(manifest, (tags) => tags.map((diskTag) => ({
            key: diskTag.Key,
            value: diskTag.Value,
        })));
    }
    /**
     * Validates that `assumeRoleAdditionalOptions` doesn't contain nor `ExternalId` neither `RoleArn`, as they
     * should have dedicated properties preceding this (e.g `assumeRoleArn` and `assumeRoleExternalId`).
     */
    static validateAssumeRoleAdditionalOptions(instance, key, _schema, _options, _ctx) {
        if (key !== 'assumeRoleAdditionalOptions') {
            // note that this means that if we happen to have a property named like this, but that
            // does want to allow 'RoleArn' or 'ExternalId', this code will have to change to consider the full schema path.
            // I decided to make this less granular for now on purpose because it fits our needs and avoids having messy
            // validation logic due to various schema paths.
            return;
        }
        const assumeRoleOptions = instance[key];
        if (assumeRoleOptions?.RoleArn) {
            throw new Error(`RoleArn is not allowed inside '${key}'`);
        }
        if (assumeRoleOptions?.ExternalId) {
            throw new Error(`ExternalId is not allowed inside '${key}'`);
        }
    }
    /**
     * See explanation on `patchStackTagsOnRead`
     *
     * Translate stack tags metadata if it has the "right" casing.
     */
    static patchStackTagsOnWrite(manifest) {
        return Manifest.replaceStackTags(manifest, (tags) => tags.map((memTag) => 
        // Might already be uppercased (because stack synthesis generates it in final form yet)
        ('Key' in memTag ? memTag : { Key: memTag.key, Value: memTag.value })));
    }
    /**
     * Recursively replace stack tags in the stack metadata
     */
    static replaceStackTags(manifest, fn) {
        // Need to add in the `noUndefined`s because otherwise jest snapshot tests are going to freak out
        // about the keys with values that are `undefined` (even though they would never be JSON.stringified)
        return noUndefined({
            ...manifest,
            artifacts: mapValues(manifest.artifacts, (artifact) => {
                if (artifact.type !== assembly.ArtifactType.AWS_CLOUDFORMATION_STACK) {
                    return artifact;
                }
                return noUndefined({
                    ...artifact,
                    metadata: mapValues(artifact.metadata, (metadataEntries) => metadataEntries.map((metadataEntry) => {
                        if (metadataEntry.type !== assembly.ArtifactMetadataEntryType.STACK_TAGS ||
                            !metadataEntry.data) {
                            return metadataEntry;
                        }
                        return {
                            ...metadataEntry,
                            data: fn(metadataEntry.data),
                        };
                    })),
                });
            }),
        });
    }
    constructor() {
    }
}
exports.Manifest = Manifest;
_a = JSII_RTTI_SYMBOL_1;
Manifest[_a] = { fqn: "@aws-cdk/cloud-assembly-schema.Manifest", version: "41.2.0" };
function mapValues(xs, fn) {
    if (!xs) {
        return undefined;
    }
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        ret[k] = fn(v);
    }
    return ret;
}
function noUndefined(xs) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function stripEnumErrors(errors) {
    return errors.filter((e) => typeof e.schema === 'string' || !('enum' in e.schema));
}
//# sourceMappingURL=data:application/json;base64,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