r'''
# Tasks for AWS Step Functions

[AWS Step Functions](https://docs.aws.amazon.com/step-functions/latest/dg/welcome.html) is a web service that enables you to coordinate the
components of distributed applications and microservices using visual workflows.
You build applications from individual components that each perform a discrete
function, or task, allowing you to scale and change applications quickly.

A [Task](https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-task-state.html) state represents a single unit of work performed by a state machine.
All work in your state machine is performed by tasks.  This module contains a collection of classes that allow you to call various AWS services
from your Step Functions state machine.

Be sure to familiarize yourself with the [`aws-stepfunctions` module documentation](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions-readme.html) first.

This module is part of the [AWS Cloud Development Kit](https://github.com/aws/aws-cdk) project.

## Table Of Contents

* [Tasks for AWS Step Functions](#tasks-for-aws-step-functions)

  * [Table Of Contents](#table-of-contents)
  * [Paths](#paths)
  * [Evaluate Expression](#evaluate-expression)
  * [API Gateway](#api-gateway)

    * [Call REST API Endpoint](#call-rest-api-endpoint)
    * [Call HTTP API Endpoint](#call-http-api-endpoint)
  * [AWS SDK](#aws-sdk)

    * [Cross-region AWS API call](#cross-region-aws-api-call)
  * [Athena](#athena)

    * [StartQueryExecution](#startqueryexecution)
    * [GetQueryExecution](#getqueryexecution)
    * [GetQueryResults](#getqueryresults)
    * [StopQueryExecution](#stopqueryexecution)
  * [Batch](#batch)

    * [SubmitJob](#submitjob)
  * [Bedrock](#bedrock)

    * [InvokeModel](#invokemodel)
  * [CodeBuild](#codebuild)

    * [StartBuild](#startbuild)
  * [DynamoDB](#dynamodb)

    * [GetItem](#getitem)
    * [PutItem](#putitem)
    * [DeleteItem](#deleteitem)
    * [UpdateItem](#updateitem)
  * [ECS](#ecs)

    * [RunTask](#runtask)

      * [EC2](#ec2)
      * [Fargate](#fargate)
  * [EMR](#emr)

    * [Create Cluster](#create-cluster)
    * [Termination Protection](#termination-protection)
    * [Terminate Cluster](#terminate-cluster)
    * [Add Step](#add-step)
    * [Cancel Step](#cancel-step)
    * [Modify Instance Fleet](#modify-instance-fleet)
    * [Modify Instance Group](#modify-instance-group)
  * [EMR on EKS](#emr-on-eks)

    * [Create Virtual Cluster](#create-virtual-cluster)
    * [Delete Virtual Cluster](#delete-virtual-cluster)
    * [Start Job Run](#start-job-run)
  * [EKS](#eks)

    * [Call](#call)
  * [EventBridge](#eventbridge)

    * [Put Events](#put-events)
  * [Glue](#glue)

    * [Start Job Run](#start-job-run)
    * [Start Crawler Run](#startcrawlerrun)
  * [Glue DataBrew](#glue-databrew)

    * [Start Job Run](#start-job-run-1)
  * [Lambda](#lambda)

    * [Invoke](#invoke)
  * [MediaConvert](#mediaconvert)

    * [Create Job](#create-job)
  * [SageMaker](#sagemaker)

    * [Create Training Job](#create-training-job)
    * [Create Transform Job](#create-transform-job)
    * [Create Endpoint](#create-endpoint)
    * [Create Endpoint Config](#create-endpoint-config)
    * [Create Model](#create-model)
    * [Update Endpoint](#update-endpoint)
  * [SNS](#sns)

    * [Publish](#publish)
  * [Step Functions](#step-functions)

    * [Start Execution](#start-execution)
    * [Invoke Activity](#invoke-activity)
  * [SQS](#sqs)

    * [Send Message](#send-message)

## Paths

Learn more about input and output processing in Step Functions [here](https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html)

## Evaluate Expression

Use the `EvaluateExpression` to perform simple operations referencing state paths. The
`expression` referenced in the task will be evaluated in a Lambda function
(`eval()`). This allows you to not have to write Lambda code for simple operations.

Example: convert a wait time from milliseconds to seconds, concat this in a message and wait:

```python
convert_to_seconds = tasks.EvaluateExpression(self, "Convert to seconds",
    expression="$.waitMilliseconds / 1000",
    result_path="$.waitSeconds"
)

create_message = tasks.EvaluateExpression(self, "Create message",
    # Note: this is a string inside a string.
    expression="`Now waiting ${$.waitSeconds} seconds...`",
    runtime=lambda_.Runtime.NODEJS_LATEST,
    result_path="$.message"
)

publish_message = tasks.SnsPublish(self, "Publish message",
    topic=sns.Topic(self, "cool-topic"),
    message=sfn.TaskInput.from_json_path_at("$.message"),
    result_path="$.sns"
)

wait = sfn.Wait(self, "Wait",
    time=sfn.WaitTime.seconds_path("$.waitSeconds")
)

sfn.StateMachine(self, "StateMachine",
    definition=convert_to_seconds.next(create_message).next(publish_message).next(wait)
)
```

The `EvaluateExpression` supports a `runtime` prop to specify the Lambda
runtime to use to evaluate the expression. Currently, only runtimes
of the Node.js family are supported.

## API Gateway

Step Functions supports [API Gateway](https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html) through the service integration pattern.

HTTP APIs are designed for low-latency, cost-effective integrations with AWS services, including AWS Lambda, and HTTP endpoints.
HTTP APIs support OIDC and OAuth 2.0 authorization, and come with built-in support for CORS and automatic deployments.
Previous-generation REST APIs currently offer more features. More details can be found [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-vs-rest.html).

### Call REST API Endpoint

The `CallApiGatewayRestApiEndpoint` calls the REST API endpoint.

```python
import aws_cdk.aws_apigateway as apigateway

rest_api = apigateway.RestApi(self, "MyRestApi")

invoke_task = tasks.CallApiGatewayRestApiEndpoint(self, "Call REST API",
    api=rest_api,
    stage_name="prod",
    method=tasks.HttpMethod.GET
)
```

By default, the API endpoint URI will be constructed using the AWS region of
the stack in which the provided `api` is created.

To construct the endpoint with a different region, use the `region` parameter:

```python
import aws_cdk.aws_apigateway as apigateway

rest_api = apigateway.RestApi(self, "MyRestApi")
invoke_task = tasks.CallApiGatewayRestApiEndpoint(self, "Call REST API",
    api=rest_api,
    stage_name="prod",
    method=tasks.HttpMethod.GET,
    region="us-west-2"
)
```

Be aware that the header values must be arrays. When passing the Task Token
in the headers field `WAIT_FOR_TASK_TOKEN` integration, use
`JsonPath.array()` to wrap the token in an array:

```python
import aws_cdk.aws_apigateway as apigateway
# api: apigateway.RestApi


tasks.CallApiGatewayRestApiEndpoint(self, "Endpoint",
    api=api,
    stage_name="Stage",
    method=tasks.HttpMethod.PUT,
    integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
    headers=sfn.TaskInput.from_object({
        "TaskToken": sfn.JsonPath.array(sfn.JsonPath.task_token)
    })
)
```

### Call HTTP API Endpoint

The `CallApiGatewayHttpApiEndpoint` calls the HTTP API endpoint.

```python
import aws_cdk.aws_apigatewayv2 as apigatewayv2

http_api = apigatewayv2.HttpApi(self, "MyHttpApi")

invoke_task = tasks.CallApiGatewayHttpApiEndpoint(self, "Call HTTP API",
    api_id=http_api.api_id,
    api_stack=Stack.of(http_api),
    method=tasks.HttpMethod.GET
)
```

## AWS SDK

Step Functions supports calling [AWS service's API actions](https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html)
through the service integration pattern.

You can use Step Functions' AWS SDK integrations to call any of the over two hundred AWS services
directly from your state machine, giving you access to over nine thousand API actions.

```python
# my_bucket: s3.Bucket

get_object = tasks.CallAwsService(self, "GetObject",
    service="s3",
    action="getObject",
    parameters={
        "Bucket": my_bucket.bucket_name,
        "Key": sfn.JsonPath.string_at("$.key")
    },
    iam_resources=[my_bucket.arn_for_objects("*")]
)
```

Use camelCase for actions and PascalCase for parameter names.

The task automatically adds an IAM statement to the state machine role's policy based on the
service and action called. The resources for this statement must be specified in `iamResources`.

Use the `iamAction` prop to manually specify the IAM action name in the case where the IAM
action name does not match with the API service/action name:

```python
list_buckets = tasks.CallAwsService(self, "ListBuckets",
    service="s3",
    action="listBuckets",
    iam_resources=["*"],
    iam_action="s3:ListAllMyBuckets"
)
```

Use the `additionalIamStatements` prop to pass additional IAM statements that will be added to the
state machine role's policy. Use it in the case where the call requires more than a single statement
to be executed:

```python
detect_labels = tasks.CallAwsService(self, "DetectLabels",
    service="rekognition",
    action="detectLabels",
    iam_resources=["*"],
    additional_iam_statements=[
        iam.PolicyStatement(
            actions=["s3:getObject"],
            resources=["arn:aws:s3:::amzn-s3-demo-bucket/*"]
        )
    ]
)
```

### Cross-region AWS API call

You can call AWS API in a different region from your state machine by using the `CallAwsServiceCrossRegion` construct. In addition to the properties for `CallAwsService` construct, you have to set `region` property to call the API.

```python
# my_bucket: s3.Bucket

get_object = tasks.CallAwsServiceCrossRegion(self, "GetObject",
    region="ap-northeast-1",
    service="s3",
    action="getObject",
    parameters={
        "Bucket": my_bucket.bucket_name,
        "Key": sfn.JsonPath.string_at("$.key")
    },
    iam_resources=[my_bucket.arn_for_objects("*")]
)
```

Other properties such as `additionalIamStatements` can be used in the same way as the `CallAwsService` task.

Note that when you use `integrationPattern.WAIT_FOR_TASK_TOKEN`, the output path changes under `Payload` property.

## Athena

Step Functions supports [Athena](https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html) through the service integration pattern.

### StartQueryExecution

The [StartQueryExecution](https://docs.aws.amazon.com/athena/latest/APIReference/API_StartQueryExecution.html) API runs the SQL query statement.

```python
start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Start Athena Query",
    query_string=sfn.JsonPath.string_at("$.queryString"),
    query_execution_context=tasks.QueryExecutionContext(
        database_name="mydatabase"
    ),
    result_configuration=tasks.ResultConfiguration(
        encryption_configuration=tasks.EncryptionConfiguration(
            encryption_option=tasks.EncryptionOption.S3_MANAGED
        ),
        output_location=s3.Location(
            bucket_name="amzn-s3-demo-bucket",
            object_key="folder"
        )
    ),
    execution_parameters=["param1", "param2"]
)
```

You can reuse the query results by setting the `resultReuseConfigurationMaxAge` property.

```python
start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Start Athena Query",
    query_string=sfn.JsonPath.string_at("$.queryString"),
    query_execution_context=tasks.QueryExecutionContext(
        database_name="mydatabase"
    ),
    result_configuration=tasks.ResultConfiguration(
        encryption_configuration=tasks.EncryptionConfiguration(
            encryption_option=tasks.EncryptionOption.S3_MANAGED
        ),
        output_location=s3.Location(
            bucket_name="query-results-bucket",
            object_key="folder"
        )
    ),
    execution_parameters=["param1", "param2"],
    result_reuse_configuration_max_age=Duration.minutes(100)
)
```

### GetQueryExecution

The [GetQueryExecution](https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryExecution.html) API gets information about a single execution of a query.

```python
get_query_execution_job = tasks.AthenaGetQueryExecution(self, "Get Query Execution",
    query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
)
```

### GetQueryResults

The [GetQueryResults](https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryResults.html) API that streams the results of a single query execution specified by QueryExecutionId from S3.

```python
get_query_results_job = tasks.AthenaGetQueryResults(self, "Get Query Results",
    query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
)
```

### StopQueryExecution

The [StopQueryExecution](https://docs.aws.amazon.com/athena/latest/APIReference/API_StopQueryExecution.html) API that stops a query execution.

```python
stop_query_execution_job = tasks.AthenaStopQueryExecution(self, "Stop Query Execution",
    query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
)
```

## Batch

Step Functions supports [Batch](https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html) through the service integration pattern.

### SubmitJob

The [SubmitJob](https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html) API submits an AWS Batch job from a job definition.

```python
import aws_cdk.aws_batch as batch
# batch_job_definition: batch.EcsJobDefinition
# batch_queue: batch.JobQueue


task = tasks.BatchSubmitJob(self, "Submit Job",
    job_definition_arn=batch_job_definition.job_definition_arn,
    job_name="MyJob",
    job_queue_arn=batch_queue.job_queue_arn
)
```

## Bedrock

Step Functions supports [Bedrock](https://docs.aws.amazon.com/step-functions/latest/dg/connect-bedrock.html) through the service integration pattern.

### InvokeModel

The [InvokeModel](https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html) API
invokes the specified Bedrock model to run inference using the input provided.
The format of the input body and the response body depend on the model selected.

```python
import aws_cdk.aws_bedrock as bedrock


model = bedrock.FoundationModel.from_foundation_model_id(self, "Model", bedrock.FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1)

task = tasks.BedrockInvokeModel(self, "Prompt Model",
    model=model,
    body=sfn.TaskInput.from_object({
        "input_text": "Generate a list of five first names.",
        "text_generation_config": {
            "max_token_count": 100,
            "temperature": 1
        }
    }),
    result_selector={
        "names": sfn.JsonPath.string_at("$.Body.results[0].outputText")
    }
)
```

### Using Input Path for S3 URI

Provide S3 URI as an input or output path to invoke a model

To specify the S3 URI as JSON path to your input or output fields, use props `s3InputUri` and `s3OutputUri` under BedrockInvokeModelProps and set
feature flag `@aws-cdk/aws-stepfunctions-tasks:useNewS3UriParametersForBedrockInvokeModelTask` to true.

If this flag is not enabled, the code will populate the S3Uri using `InputPath` and `OutputPath` fields, which is not recommended.

```python
import aws_cdk.aws_bedrock as bedrock


model = bedrock.FoundationModel.from_foundation_model_id(self, "Model", bedrock.FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1)

task = tasks.BedrockInvokeModel(self, "Prompt Model",
    model=model,
    input=tasks.BedrockInvokeModelInputProps(s3_input_uri=sfn.JsonPath.string_at("$.prompt")),
    output=tasks.BedrockInvokeModelOutputProps(s3_output_uri=sfn.JsonPath.string_at("$.prompt"))
)
```

### Using Input Path

Provide S3 URI as an input or output path to invoke a model

Currently, input and output Path provided in the BedrockInvokeModelProps input is defined as S3URI field under task definition of state machine.
To modify the existing behaviour, set `@aws-cdk/aws-stepfunctions-tasks:useNewS3UriParametersForBedrockInvokeModelTask` to true.

If this feature flag is enabled, S3URI fields will be generated from other Props(`s3InputUri` and `s3OutputUri`), and the given inputPath, OutputPath will be rendered as
it is in the JSON task definition.

If the feature flag is set to `false`, the behavior will be to populate the S3Uri using the `InputPath` and `OutputPath` fields, which is not recommended.

```python
import aws_cdk.aws_bedrock as bedrock


model = bedrock.FoundationModel.from_foundation_model_id(self, "Model", bedrock.FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1)

task = tasks.BedrockInvokeModel(self, "Prompt Model",
    model=model,
    input_path=sfn.JsonPath.string_at("$.prompt"),
    output_path=sfn.JsonPath.string_at("$.prompt")
)
```

You can apply a guardrail to the invocation by setting `guardrail`.

```python
import aws_cdk.aws_bedrock as bedrock


model = bedrock.FoundationModel.from_foundation_model_id(self, "Model", bedrock.FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1)

task = tasks.BedrockInvokeModel(self, "Prompt Model with guardrail",
    model=model,
    body=sfn.TaskInput.from_object({
        "input_text": "Generate a list of five first names.",
        "text_generation_config": {
            "max_token_count": 100,
            "temperature": 1
        }
    }),
    guardrail=tasks.Guardrail.enable("guardrailId", 1),
    result_selector={
        "names": sfn.JsonPath.string_at("$.Body.results[0].outputText")
    }
)
```

## CodeBuild

Step Functions supports [CodeBuild](https://docs.aws.amazon.com/step-functions/latest/dg/connect-codebuild.html) through the service integration pattern.

### StartBuild

[StartBuild](https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html) starts a CodeBuild Project by Project Name.

```python
import aws_cdk.aws_codebuild as codebuild


codebuild_project = codebuild.Project(self, "Project",
    project_name="MyTestProject",
    build_spec=codebuild.BuildSpec.from_object({
        "version": "0.2",
        "phases": {
            "build": {
                "commands": ["echo \"Hello, CodeBuild!\""
                ]
            }
        }
    })
)

task = tasks.CodeBuildStartBuild(self, "Task",
    project=codebuild_project,
    integration_pattern=sfn.IntegrationPattern.RUN_JOB,
    environment_variables_override={
        "ZONE": codebuild.BuildEnvironmentVariable(
            type=codebuild.BuildEnvironmentVariableType.PLAINTEXT,
            value=sfn.JsonPath.string_at("$.envVariables.zone")
        )
    }
)
```

### StartBuildBatch

[StartBuildBatch](https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuildBatch.html) starts a batch CodeBuild for a project by project name.
It is necessary to enable the batch build feature in the CodeBuild project.

```python
import aws_cdk.aws_codebuild as codebuild


project = codebuild.Project(self, "Project",
    project_name="MyTestProject",
    build_spec=codebuild.BuildSpec.from_object_to_yaml({
        "version": 0.2,
        "batch": {
            "build-list": [{
                "identifier": "id",
                "buildspec": "version: 0.2\nphases:\n  build:\n    commands:\n      - echo \"Hello, from small!\""
            }
            ]
        }
    })
)
project.enable_batch_builds()

task = tasks.CodeBuildStartBuildBatch(self, "buildBatchTask",
    project=project,
    integration_pattern=sfn.IntegrationPattern.REQUEST_RESPONSE,
    environment_variables_override={
        "test": codebuild.BuildEnvironmentVariable(
            type=codebuild.BuildEnvironmentVariableType.PLAINTEXT,
            value="testValue"
        )
    }
)
```

**Note**: `enableBatchBuilds()` will do nothing for imported projects.
If you are using an imported project, you must ensure that the project is already configured for batch builds.

## DynamoDB

You can call DynamoDB APIs from a `Task` state.
Read more about calling DynamoDB APIs [here](https://docs.aws.amazon.com/step-functions/latest/dg/connect-ddb.html)

### GetItem

The [GetItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html) operation returns a set of attributes for the item with the given primary key.

```python
# my_table: dynamodb.Table

tasks.DynamoGetItem(self, "Get Item",
    key={"message_id": tasks.DynamoAttributeValue.from_string("message-007")},
    table=my_table
)
```

### PutItem

The [PutItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html) operation creates a new item, or replaces an old item with a new item.

```python
# my_table: dynamodb.Table

tasks.DynamoPutItem(self, "PutItem",
    item={
        "MessageId": tasks.DynamoAttributeValue.from_string("message-007"),
        "Text": tasks.DynamoAttributeValue.from_string(sfn.JsonPath.string_at("$.bar")),
        "TotalCount": tasks.DynamoAttributeValue.from_number(10)
    },
    table=my_table
)
```

### DeleteItem

The [DeleteItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html) operation deletes a single item in a table by primary key.

```python
# my_table: dynamodb.Table

tasks.DynamoDeleteItem(self, "DeleteItem",
    key={"MessageId": tasks.DynamoAttributeValue.from_string("message-007")},
    table=my_table,
    result_path=sfn.JsonPath.DISCARD
)
```

### UpdateItem

The [UpdateItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html) operation edits an existing item's attributes, or adds a new item
to the table if it does not already exist.

```python
# my_table: dynamodb.Table

tasks.DynamoUpdateItem(self, "UpdateItem",
    key={
        "MessageId": tasks.DynamoAttributeValue.from_string("message-007")
    },
    table=my_table,
    expression_attribute_values={
        ":val": tasks.DynamoAttributeValue.number_from_string(sfn.JsonPath.string_at("$.Item.TotalCount.N")),
        ":rand": tasks.DynamoAttributeValue.from_number(20)
    },
    update_expression="SET TotalCount = :val + :rand"
)
```

## ECS

Step Functions supports [ECS/Fargate](https://docs.aws.amazon.com/step-functions/latest/dg/connect-ecs.html) through the service integration pattern.

### RunTask

[RunTask](https://docs.aws.amazon.com/step-functions/latest/dg/connect-ecs.html) starts a new task using the specified task definition.

#### EC2

The EC2 launch type allows you to run your containerized applications on a cluster
of Amazon EC2 instances that you manage.

When a task that uses the EC2 launch type is launched, Amazon ECS must determine where
to place the task based on the requirements specified in the task definition, such as
CPU and memory. Similarly, when you scale down the task count, Amazon ECS must determine
which tasks to terminate. You can apply task placement strategies and constraints to
customize how Amazon ECS places and terminates tasks. Learn more about [task placement](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement.html)

The latest ACTIVE revision of the passed task definition is used for running the task.

The following example runs a job from a task definition on EC2

```python
vpc = ec2.Vpc.from_lookup(self, "Vpc",
    is_default=True
)

cluster = ecs.Cluster(self, "Ec2Cluster", vpc=vpc)
cluster.add_capacity("DefaultAutoScalingGroup",
    instance_type=ec2.InstanceType("t2.micro"),
    vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
)

task_definition = ecs.TaskDefinition(self, "TD",
    compatibility=ecs.Compatibility.EC2
)

task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("foo/bar"),
    memory_limit_mi_b=256
)

run_task = tasks.EcsRunTask(self, "Run",
    integration_pattern=sfn.IntegrationPattern.RUN_JOB,
    cluster=cluster,
    task_definition=task_definition,
    launch_target=tasks.EcsEc2LaunchTarget(
        placement_strategies=[
            ecs.PlacementStrategy.spread_across_instances(),
            ecs.PlacementStrategy.packed_by_cpu(),
            ecs.PlacementStrategy.randomly()
        ],
        placement_constraints=[
            ecs.PlacementConstraint.member_of("blieptuut")
        ]
    ),
    propagated_tag_source=ecs.PropagatedTagSource.TASK_DEFINITION
)
```

#### Fargate

AWS Fargate is a serverless compute engine for containers that works with Amazon
Elastic Container Service (ECS). Fargate makes it easy for you to focus on building
your applications. Fargate removes the need to provision and manage servers, lets you
specify and pay for resources per application, and improves security through application
isolation by design. Learn more about [Fargate](https://aws.amazon.com/fargate/)

The Fargate launch type allows you to run your containerized applications without the need
to provision and manage the backend infrastructure. Just register your task definition and
Fargate launches the container for you. The latest ACTIVE revision of the passed
task definition is used for running the task. Learn more about
[Fargate Versioning](https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskDefinition.html)

The following example runs a job from a task definition on Fargate

```python
vpc = ec2.Vpc.from_lookup(self, "Vpc",
    is_default=True
)

cluster = ecs.Cluster(self, "FargateCluster", vpc=vpc)

task_definition = ecs.TaskDefinition(self, "TD",
    memory_mi_b="512",
    cpu="256",
    compatibility=ecs.Compatibility.FARGATE
)

container_definition = task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("foo/bar"),
    memory_limit_mi_b=256
)

run_task = tasks.EcsRunTask(self, "RunFargate",
    integration_pattern=sfn.IntegrationPattern.RUN_JOB,
    cluster=cluster,
    task_definition=task_definition,
    assign_public_ip=True,
    container_overrides=[tasks.ContainerOverride(
        container_definition=container_definition,
        environment=[tasks.TaskEnvironmentVariable(name="SOME_KEY", value=sfn.JsonPath.string_at("$.SomeKey"))]
    )],
    launch_target=tasks.EcsFargateLaunchTarget(),
    propagated_tag_source=ecs.PropagatedTagSource.TASK_DEFINITION
)
```

#### Override CPU and Memory Parameter

By setting the property cpu or memoryMiB, you can override the Fargate or EC2 task instance size at runtime.

see: https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TaskOverride.html

```python
vpc = ec2.Vpc.from_lookup(self, "Vpc",
    is_default=True
)
cluster = ecs.Cluster(self, "ECSCluster", vpc=vpc)

task_definition = ecs.TaskDefinition(self, "TD",
    compatibility=ecs.Compatibility.FARGATE,
    cpu="256",
    memory_mi_b="512"
)

task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("foo/bar")
)

run_task = tasks.EcsRunTask(self, "Run",
    integration_pattern=sfn.IntegrationPattern.RUN_JOB,
    cluster=cluster,
    task_definition=task_definition,
    launch_target=tasks.EcsFargateLaunchTarget(),
    cpu="1024",
    memory_mi_b="1048"
)
```

#### ECS enable Exec

By setting the property [`enableExecuteCommand`](https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html#ECS-RunTask-request-enableExecuteCommand) to `true`, you can enable the [ECS Exec feature](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-exec.html) for the task for either Fargate or EC2 launch types.

```python
vpc = ec2.Vpc.from_lookup(self, "Vpc",
    is_default=True
)
cluster = ecs.Cluster(self, "ECSCluster", vpc=vpc)

task_definition = ecs.TaskDefinition(self, "TD",
    compatibility=ecs.Compatibility.EC2
)

task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("foo/bar"),
    memory_limit_mi_b=256
)

run_task = tasks.EcsRunTask(self, "Run",
    integration_pattern=sfn.IntegrationPattern.RUN_JOB,
    cluster=cluster,
    task_definition=task_definition,
    launch_target=tasks.EcsEc2LaunchTarget(),
    enable_execute_command=True
)
```

## EMR

Step Functions supports Amazon EMR through the service integration pattern.
The service integration APIs correspond to Amazon EMR APIs but differ in the
parameters that are used.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr.html) about the differences when using these service integrations.

### Create Cluster

Creates and starts running a cluster (job flow).
Corresponds to the [`runJobFlow`](https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html) API in EMR.

```python
cluster_role = iam.Role(self, "ClusterRole",
    assumed_by=iam.ServicePrincipal("ec2.amazonaws.com")
)

service_role = iam.Role(self, "ServiceRole",
    assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
)

auto_scaling_role = iam.Role(self, "AutoScalingRole",
    assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
)

auto_scaling_role.assume_role_policy.add_statements(
    iam.PolicyStatement(
        effect=iam.Effect.ALLOW,
        principals=[
            iam.ServicePrincipal("application-autoscaling.amazonaws.com")
        ],
        actions=["sts:AssumeRole"
        ]
    ))

tasks.EmrCreateCluster(self, "Create Cluster",
    instances=tasks.EmrCreateCluster.InstancesConfigProperty(),
    cluster_role=cluster_role,
    name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
    service_role=service_role,
    auto_scaling_role=auto_scaling_role
)
```

You can use the launch specification for On-Demand and Spot instances in the fleet.

```python
tasks.EmrCreateCluster(self, "OnDemandSpecification",
    instances=tasks.EmrCreateCluster.InstancesConfigProperty(
        instance_fleets=[tasks.EmrCreateCluster.InstanceFleetConfigProperty(
            instance_fleet_type=tasks.EmrCreateCluster.InstanceRoleType.MASTER,
            launch_specifications=tasks.EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty(
                on_demand_specification=tasks.EmrCreateCluster.OnDemandProvisioningSpecificationProperty(
                    allocation_strategy=tasks.EmrCreateCluster.OnDemandAllocationStrategy.LOWEST_PRICE
                )
            )
        )]
    ),
    name="OnDemandCluster",
    integration_pattern=sfn.IntegrationPattern.RUN_JOB
)

tasks.EmrCreateCluster(self, "SpotSpecification",
    instances=tasks.EmrCreateCluster.InstancesConfigProperty(
        instance_fleets=[tasks.EmrCreateCluster.InstanceFleetConfigProperty(
            instance_fleet_type=tasks.EmrCreateCluster.InstanceRoleType.MASTER,
            launch_specifications=tasks.EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty(
                spot_specification=tasks.EmrCreateCluster.SpotProvisioningSpecificationProperty(
                    allocation_strategy=tasks.EmrCreateCluster.SpotAllocationStrategy.CAPACITY_OPTIMIZED,
                    timeout_action=tasks.EmrCreateCluster.SpotTimeoutAction.TERMINATE_CLUSTER,
                    timeout=Duration.minutes(5)
                )
            )
        )]
    ),
    name="SpotCluster",
    integration_pattern=sfn.IntegrationPattern.RUN_JOB
)
```

If you want to run multiple steps in [parallel](https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-concurrent-steps.html),
you can specify the `stepConcurrencyLevel` property. The concurrency range is between 1
and 256 inclusive, where the default concurrency of 1 means no step concurrency is allowed.
`stepConcurrencyLevel` requires the EMR release label to be 5.28.0 or above.

```python
tasks.EmrCreateCluster(self, "Create Cluster",
    instances=tasks.EmrCreateCluster.InstancesConfigProperty(),
    name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
    step_concurrency_level=10
)
```

If you want to use an [auto-termination policy](https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-auto-termination-policy.html),
you can specify the `autoTerminationPolicyIdleTimeout` property.
Specifies the amount of idle time after which the cluster automatically terminates. You can specify a minimum of 60 seconds and a maximum of 604800 seconds (seven days).

```python
tasks.EmrCreateCluster(self, "Create Cluster",
    instances=tasks.EmrCreateCluster.InstancesConfigProperty(),
    name="ClusterName",
    auto_termination_policy_idle_timeout=Duration.seconds(100)
)
```

### Termination Protection

Locks a cluster (job flow) so the EC2 instances in the cluster cannot be
terminated by user intervention, an API call, or a job-flow error.

Corresponds to the [`setTerminationProtection`](https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr.html) API in EMR.

```python
tasks.EmrSetClusterTerminationProtection(self, "Task",
    cluster_id="ClusterId",
    termination_protected=False
)
```

### Terminate Cluster

Shuts down a cluster (job flow).
Corresponds to the [`terminateJobFlows`](https://docs.aws.amazon.com/emr/latest/APIReference/API_TerminateJobFlows.html) API in EMR.

```python
tasks.EmrTerminateCluster(self, "Task",
    cluster_id="ClusterId"
)
```

### Add Step

Adds a new step to a running cluster.
Corresponds to the [`addJobFlowSteps`](https://docs.aws.amazon.com/emr/latest/APIReference/API_AddJobFlowSteps.html) API in EMR.

```python
tasks.EmrAddStep(self, "Task",
    cluster_id="ClusterId",
    name="StepName",
    jar="Jar",
    action_on_failure=tasks.ActionOnFailure.CONTINUE
)
```

To specify a custom runtime role use the `executionRoleArn` property.

**Note:** The EMR cluster must be created with a security configuration and the runtime role must have a specific trust policy.
See this [blog post](https://aws.amazon.com/blogs/big-data/introducing-runtime-roles-for-amazon-emr-steps-use-iam-roles-and-aws-lake-formation-for-access-control-with-amazon-emr/) for more details.

```python
import aws_cdk.aws_emr as emr


cfn_security_configuration = emr.CfnSecurityConfiguration(self, "EmrSecurityConfiguration",
    name="AddStepRuntimeRoleSecConfig",
    security_configuration=JSON.parse("""
            {
              "AuthorizationConfiguration": {
                  "IAMConfiguration": {
                      "EnableApplicationScopedIAMRole": true,
                      "ApplicationScopedIAMRoleConfiguration":
                          {
                              "PropagateSourceIdentity": true
                          }
                  },
                  "LakeFormationConfiguration": {
                      "AuthorizedSessionTagValue": "Amazon EMR"
                  }
              }
            }""")
)

task = tasks.EmrCreateCluster(self, "Create Cluster",
    instances=tasks.EmrCreateCluster.InstancesConfigProperty(),
    name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
    security_configuration=cfn_security_configuration.name
)

execution_role = iam.Role(self, "Role",
    assumed_by=iam.ArnPrincipal(task.cluster_role.role_arn)
)

execution_role.assume_role_policy.add_statements(
    iam.PolicyStatement(
        effect=iam.Effect.ALLOW,
        principals=[task.cluster_role
        ],
        actions=["sts:SetSourceIdentity"
        ]
    ),
    iam.PolicyStatement(
        effect=iam.Effect.ALLOW,
        principals=[task.cluster_role
        ],
        actions=["sts:TagSession"
        ],
        conditions={
            "StringEquals": {
                "aws:RequestTag/LakeFormationAuthorizedCaller": "Amazon EMR"
            }
        }
    ))

tasks.EmrAddStep(self, "Task",
    cluster_id="ClusterId",
    execution_role_arn=execution_role.role_arn,
    name="StepName",
    jar="Jar",
    action_on_failure=tasks.ActionOnFailure.CONTINUE
)
```

### Cancel Step

Cancels a pending step in a running cluster.
Corresponds to the [`cancelSteps`](https://docs.aws.amazon.com/emr/latest/APIReference/API_CancelSteps.html) API in EMR.

```python
tasks.EmrCancelStep(self, "Task",
    cluster_id="ClusterId",
    step_id="StepId"
)
```

### Modify Instance Fleet

Modifies the target On-Demand and target Spot capacities for the instance
fleet with the specified InstanceFleetName.

Corresponds to the [`modifyInstanceFleet`](https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceFleet.html) API in EMR.

```python
tasks.EmrModifyInstanceFleetByName(self, "Task",
    cluster_id="ClusterId",
    instance_fleet_name="InstanceFleetName",
    target_on_demand_capacity=2,
    target_spot_capacity=0
)
```

### Modify Instance Group

Modifies the number of nodes and configuration settings of an instance group.

Corresponds to the [`modifyInstanceGroups`](https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceGroups.html) API in EMR.

```python
tasks.EmrModifyInstanceGroupByName(self, "Task",
    cluster_id="ClusterId",
    instance_group_name=sfn.JsonPath.string_at("$.InstanceGroupName"),
    instance_group=tasks.EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty(
        instance_count=1
    )
)
```

## EMR on EKS

Step Functions supports Amazon EMR on EKS through the service integration pattern.
The service integration APIs correspond to Amazon EMR on EKS APIs, but differ in the parameters that are used.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html) about the differences when using these service integrations.

[Setting up](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up.html) the EKS cluster is required.

### Create Virtual Cluster

The [CreateVirtualCluster](https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_CreateVirtualCluster.html) API creates a single virtual cluster that's mapped to a single Kubernetes namespace.

The EKS cluster containing the Kubernetes namespace where the virtual cluster will be mapped can be passed in from the task input.

```python
tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
    eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId"))
)
```

The EKS cluster can also be passed in directly.

```python
import aws_cdk.aws_eks as eks

# eks_cluster: eks.Cluster


tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
    eks_cluster=tasks.EksClusterInput.from_cluster(eks_cluster)
)
```

By default, the Kubernetes namespace that a virtual cluster maps to is "default", but a specific namespace within an EKS cluster can be selected.

```python
tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
    eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId")),
    eks_namespace="specified-namespace"
)
```

### Delete Virtual Cluster

The [DeleteVirtualCluster](https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_DeleteVirtualCluster.html) API deletes a virtual cluster.

```python
tasks.EmrContainersDeleteVirtualCluster(self, "Delete a Virtual Cluster",
    virtual_cluster_id=sfn.TaskInput.from_json_path_at("$.virtualCluster")
)
```

### Start Job Run

The [StartJobRun](https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_StartJobRun.html) API starts a job run. A job is a unit of work that you submit to Amazon EMR on EKS for execution. The work performed by the job can be defined by a Spark jar, PySpark script, or SparkSQL query. A job run is an execution of the job on the virtual cluster.

Required setup:

* If not done already, follow the [steps](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up.html) to setup EMR on EKS and [create an EKS Cluster](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-eks-readme.html#quick-start).
* Enable [Cluster access](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-cluster-access.html)
* Enable [IAM Role access](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-enable-IAM.html)

The following actions must be performed if the virtual cluster ID is supplied from the task input. Otherwise, if it is supplied statically in the state machine definition, these actions will be done automatically.

* Create an [IAM role](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-iam.Role.html)
* Update the [Role Trust Policy](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-trust-policy.html) of the Job Execution Role.

The job can be configured with spark submit parameters:

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_driver=tasks.JobDriver(
        spark_submit_job_driver=tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    )
)
```

Configuring the job can also be done via application configuration:

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_name="EMR-Containers-Job",
    job_driver=tasks.JobDriver(
        spark_submit_job_driver=tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
        )
    ),
    application_config=[tasks.ApplicationConfiguration(
        classification=tasks.Classification.SPARK_DEFAULTS,
        properties={
            "spark.executor.instances": "1",
            "spark.executor.memory": "512M"
        }
    )]
)
```

Job monitoring can be enabled if `monitoring.logging` is set true. This automatically generates an S3 bucket and CloudWatch logs.

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_driver=tasks.JobDriver(
        spark_submit_job_driver=tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    ),
    monitoring=tasks.Monitoring(
        logging=True
    )
)
```

Otherwise, providing monitoring for jobs with existing log groups and log buckets is also available.

```python
import aws_cdk.aws_logs as logs


log_group = logs.LogGroup(self, "Log Group")
log_bucket = s3.Bucket(self, "S3 Bucket")

tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_driver=tasks.JobDriver(
        spark_submit_job_driver=tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    ),
    monitoring=tasks.Monitoring(
        log_group=log_group,
        log_bucket=log_bucket
    )
)
```

Users can provide their own existing Job Execution Role.

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_task_input(sfn.TaskInput.from_json_path_at("$.VirtualClusterId")),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_name="EMR-Containers-Job",
    execution_role=iam.Role.from_role_arn(self, "Job-Execution-Role", "arn:aws:iam::xxxxxxxxxxxx:role/JobExecutionRole"),
    job_driver=tasks.JobDriver(
        spark_submit_job_driver=tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    )
)
```

## EKS

Step Functions supports Amazon EKS through the service integration pattern.
The service integration APIs correspond to Amazon EKS APIs.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html) about the differences when using these service integrations.

### Call

Read and write Kubernetes resource objects via a Kubernetes API endpoint.
Corresponds to the [`call`](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html) API in Step Functions Connector.

The following code snippet includes a Task state that uses eks:call to list the pods.

```python
import aws_cdk.aws_eks as eks
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


my_eks_cluster = eks.Cluster(self, "my sample cluster",
    version=eks.KubernetesVersion.V1_32,
    cluster_name="myEksCluster",
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)

tasks.EksCall(self, "Call a EKS Endpoint",
    cluster=my_eks_cluster,
    http_method=tasks.HttpMethods.GET,
    http_path="/api/v1/namespaces/default/pods"
)
```

## EventBridge

Step Functions supports Amazon EventBridge through the service integration pattern.
The service integration APIs correspond to Amazon EventBridge APIs.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eventbridge.html) about the differences when using these service integrations.

### Put Events

Send events to an EventBridge bus.
Corresponds to the [`put-events`](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eventbridge.html) API in Step Functions Connector.

The following code snippet includes a Task state that uses events:putevents to send an event to the default bus.

```python
import aws_cdk.aws_events as events


my_event_bus = events.EventBus(self, "EventBus",
    event_bus_name="MyEventBus1"
)

tasks.EventBridgePutEvents(self, "Send an event to EventBridge",
    entries=[tasks.EventBridgePutEventsEntry(
        detail=sfn.TaskInput.from_object({
            "Message": "Hello from Step Functions!"
        }),
        event_bus=my_event_bus,
        detail_type="MessageFromStepFunctions",
        source="step.functions"
    )]
)
```

## EventBridge Scheduler

You can call EventBridge Scheduler APIs from a `Task` state.
Read more about calling Scheduler APIs [here](https://docs.aws.amazon.com/scheduler/latest/APIReference/API_Operations.html)

### Create Scheduler

The [CreateSchedule](https://docs.aws.amazon.com/scheduler/latest/APIReference/API_CreateSchedule.html) API creates a new schedule.

Here is an example of how to create a schedule that puts an event to SQS queue every 5 minutes:

```python
import aws_cdk.aws_scheduler as scheduler
import aws_cdk.aws_kms as kms

# key: kms.Key
# schedule_group: scheduler.CfnScheduleGroup
# target_queue: sqs.Queue
# dead_letter_queue: sqs.Queue


scheduler_role = iam.Role(self, "SchedulerRole",
    assumed_by=iam.ServicePrincipal("scheduler.amazonaws.com")
)
# To send the message to the queue
# This policy changes depending on the type of target.
scheduler_role.add_to_principal_policy(iam.PolicyStatement(
    actions=["sqs:SendMessage"],
    resources=[target_queue.queue_arn]
))

create_schedule_task1 = tasks.EventBridgeSchedulerCreateScheduleTask(self, "createSchedule",
    schedule_name="TestSchedule",
    action_after_completion=tasks.ActionAfterCompletion.NONE,
    client_token="testToken",
    description="TestDescription",
    start_date=Date(),
    end_date=Date(Date().get_time() + 1000 * 60 * 60),
    flexible_time_window=Duration.minutes(5),
    group_name=schedule_group.ref,
    kms_key=key,
    schedule=tasks.Schedule.rate(Duration.minutes(5)),
    timezone="UTC",
    enabled=True,
    target=tasks.EventBridgeSchedulerTarget(
        arn=target_queue.queue_arn,
        role=scheduler_role,
        retry_policy=tasks.RetryPolicy(
            maximum_retry_attempts=2,
            maximum_event_age=Duration.minutes(5)
        ),
        dead_letter_queue=dead_letter_queue
    )
)
```

## Glue

Step Functions supports [AWS Glue](https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html) through the service integration pattern.

### StartJobRun

You can call the [`StartJobRun`](https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-StartJobRun) API from a `Task` state.

```python
tasks.GlueStartJobRun(self, "Task",
    glue_job_name="my-glue-job",
    arguments=sfn.TaskInput.from_object({
        "key": "value"
    }),
    task_timeout=sfn.Timeout.duration(Duration.minutes(30)),
    notify_delay_after=Duration.minutes(5)
)
```

You can configure workers by setting the `workerTypeV2` and `numberOfWorkers` properties.
`workerType` is deprecated and no longer recommended. Use `workerTypeV2` which is
a ENUM-like class for more powerful worker configuration around using pre-defined values or
dynamic values.

```python
tasks.GlueStartJobRun(self, "Task",
    glue_job_name="my-glue-job",
    worker_configuration=tasks.WorkerConfigurationProperty(
        worker_type_v2=tasks.WorkerTypeV2.G_1X,  # Worker type
        number_of_workers=2
    )
)
```

To configure the worker type or number of workers dynamically from StateMachine's input,
you can configure it using JSON Path values using `workerTypeV2` like this:

```python
tasks.GlueStartJobRun(self, "Glue Job Task",
    glue_job_name="my-glue-job",
    worker_configuration=tasks.WorkerConfigurationProperty(
        worker_type_v2=tasks.WorkerTypeV2.of(sfn.JsonPath.string_at("$.glue_jobs_configs.executor_type")),
        number_of_workers=sfn.JsonPath.number_at("$.glue_jobs_configs.max_number_workers")
    )
)
```

You can choose the execution class by setting the `executionClass` property.

```python
tasks.GlueStartJobRun(self, "Task",
    glue_job_name="my-glue-job",
    execution_class=tasks.ExecutionClass.FLEX
)
```

### StartCrawlerRun

You can call the [`StartCrawler`](https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StartCrawler) API from a `Task` state through AWS SDK service integrations.

```python
import aws_cdk.aws_glue as glue

# my_crawler: glue.CfnCrawler


# You can get the crawler name from `crawler.ref`
tasks.GlueStartCrawlerRun(self, "Task1",
    crawler_name=my_crawler.ref
)

# Of course, you can also specify the crawler name directly.
tasks.GlueStartCrawlerRun(self, "Task2",
    crawler_name="my-crawler-job"
)
```

## Glue DataBrew

Step Functions supports [AWS Glue DataBrew](https://docs.aws.amazon.com/step-functions/latest/dg/connect-databrew.html) through the service integration pattern.

### Start Job Run

You can call the [`StartJobRun`](https://docs.aws.amazon.com/databrew/latest/dg/API_StartJobRun.html) API from a `Task` state.

```python
tasks.GlueDataBrewStartJobRun(self, "Task",
    name="databrew-job"
)
```

## Invoke HTTP API

Step Functions supports [calling third-party APIs](https://docs.aws.amazon.com/step-functions/latest/dg/connect-third-party-apis.html) with credentials managed by Amazon EventBridge [Connections](https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_Connection.html).

The following snippet creates a new API destination connection, and uses it to make a POST request to the specified URL. The endpoint response is available at the `$.ResponseBody` path.

```python
import aws_cdk.aws_events as events


connection = events.Connection(self, "Connection",
    authorization=events.Authorization.basic("username", SecretValue.unsafe_plain_text("password"))
)

tasks.HttpInvoke(self, "Invoke HTTP API",
    api_root="https://api.example.com",
    api_endpoint=sfn.TaskInput.from_text("path/to/resource"),
    body=sfn.TaskInput.from_object({"foo": "bar"}),
    connection=connection,
    headers=sfn.TaskInput.from_object({"Content-Type": "application/json"}),
    method=sfn.TaskInput.from_text("POST"),
    query_string_parameters=sfn.TaskInput.from_object({"id": "123"}),
    url_encoding_format=tasks.URLEncodingFormat.BRACKETS
)
```

## Lambda

Step Functions supports [AWS Lambda](https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html) through the service integration pattern.

### Invoke

[Invoke](https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html) a Lambda function.

You can specify the input to your Lambda function through the `payload` attribute.
By default, Step Functions invokes Lambda function with the state input (JSON path '$')
as the input.

The following snippet invokes a Lambda Function with the state input as the payload
by referencing the `$` path.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke with state input",
    lambda_function=fn
)
```

When a function is invoked, the Lambda service sends  [these response
elements](https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html#API_Invoke_ResponseElements)
back.

⚠️ The response from the Lambda function is in an attribute called `Payload`

The following snippet invokes a Lambda Function by referencing the `$.Payload` path
to reference the output of a Lambda executed before it.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke with empty object as payload",
    lambda_function=fn,
    payload=sfn.TaskInput.from_object({})
)

# use the output of fn as input
tasks.LambdaInvoke(self, "Invoke with payload field in the state input",
    lambda_function=fn,
    payload=sfn.TaskInput.from_json_path_at("$.Payload")
)
```

The following snippet invokes a Lambda and sets the task output to only include
the Lambda function response.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke and set function response as task output",
    lambda_function=fn,
    output_path="$.Payload"
)
```

If you want to combine the input and the Lambda function response you can use
the `payloadResponseOnly` property and specify the `resultPath`. This will put the
Lambda function ARN directly in the "Resource" string, but it conflicts with the
integrationPattern, invocationType, clientContext, and qualifier properties.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke and combine function response with task input",
    lambda_function=fn,
    payload_response_only=True,
    result_path="$.fn"
)
```

You can have Step Functions pause a task, and wait for an external process to
return a task token. Read more about the [callback pattern](https://docs.aws.amazon.com/step-functions/latest/dg/callback-task-sample-sqs.html#call-back-lambda-example)

To use the callback pattern, set the `token` property on the task. Call the Step
Functions `SendTaskSuccess` or `SendTaskFailure` APIs with the token to
indicate that the task has completed and the state machine should resume execution.

The following snippet invokes a Lambda with the task token as part of the input
to the Lambda.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke with callback",
    lambda_function=fn,
    integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
    payload=sfn.TaskInput.from_object({
        "token": sfn.JsonPath.task_token,
        "input": sfn.JsonPath.string_at("$.someField")
    })
)
```

⚠️ The task will pause until it receives that task token back with a `SendTaskSuccess` or `SendTaskFailure`
call. Learn more about [Callback with the Task
Token](https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token).

AWS Lambda can occasionally experience transient service errors. In this case, invoking Lambda
results in a 500 error, such as `ClientExecutionTimeoutException`, `ServiceException`, `AWSLambdaException`, or `SdkClientException`.
As a best practice, the `LambdaInvoke` task will retry on those errors with an interval of 2 seconds,
a back-off rate of 2 and 6 maximum attempts. Set the `retryOnServiceExceptions` prop to `false` to
disable this behavior.

## MediaConvert

Step Functions supports [AWS MediaConvert](https://docs.aws.amazon.com/step-functions/latest/dg/connect-mediaconvert.html) through the Optimized integration pattern.

### CreateJob

The [CreateJob](https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html#jobspost) API creates a new transcoding job.
For information about jobs and job settings, see the User Guide at http://docs.aws.amazon.com/mediaconvert/latest/ug/what-is.html

You can call the `CreateJob` API from a `Task` state. Optionally you can specify the `integrationPattern`.

Make sure you update the required fields - [Role](https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html#jobs-prop-createjobrequest-role) &
[Settings](https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html#jobs-prop-createjobrequest-settings) and refer
[CreateJobRequest](https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html#jobs-model-createjobrequest) for all other optional parameters.

```python
tasks.MediaConvertCreateJob(self, "CreateJob",
    create_job_request={
        "Role": "arn:aws:iam::123456789012:role/MediaConvertRole",
        "Settings": {
            "OutputGroups": [{
                "Outputs": [{
                    "ContainerSettings": {
                        "Container": "MP4"
                    },
                    "VideoDescription": {
                        "CodecSettings": {
                            "Codec": "H_264",
                            "H264Settings": {
                                "MaxBitrate": 1000,
                                "RateControlMode": "QVBR",
                                "SceneChangeDetect": "TRANSITION_DETECTION"
                            }
                        }
                    },
                    "AudioDescriptions": [{
                        "CodecSettings": {
                            "Codec": "AAC",
                            "AacSettings": {
                                "Bitrate": 96000,
                                "CodingMode": "CODING_MODE_2_0",
                                "SampleRate": 48000
                            }
                        }
                    }
                    ]
                }
                ],
                "OutputGroupSettings": {
                    "Type": "FILE_GROUP_SETTINGS",
                    "FileGroupSettings": {
                        "Destination": "s3://EXAMPLE-DESTINATION-BUCKET/"
                    }
                }
            }
            ],
            "Inputs": [{
                "AudioSelectors": {
                    "Audio Selector 1": {
                        "DefaultSelection": "DEFAULT"
                    }
                },
                "FileInput": "s3://EXAMPLE-SOURCE-BUCKET/EXAMPLE-SOURCE_FILE"
            }
            ]
        }
    },
    integration_pattern=sfn.IntegrationPattern.RUN_JOB
)
```

## SageMaker

Step Functions supports [AWS SageMaker](https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html) through the service integration pattern.

If your training job or model uses resources from AWS Marketplace,
[network isolation is required](https://docs.aws.amazon.com/sagemaker/latest/dg/mkt-algo-model-internet-free.html).
To do so, set the `enableNetworkIsolation` property to `true` for `SageMakerCreateModel` or `SageMakerCreateTrainingJob`.

To set environment variables for the Docker container use the `environment` property.

### Create Training Job

You can call the [`CreateTrainingJob`](https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrainingJob.html) API from a `Task` state.

```python
tasks.SageMakerCreateTrainingJob(self, "TrainSagemaker",
    training_job_name=sfn.JsonPath.string_at("$.JobName"),
    algorithm_specification=tasks.AlgorithmSpecification(
        algorithm_name="BlazingText",
        training_input_mode=tasks.InputMode.FILE
    ),
    input_data_config=[tasks.Channel(
        channel_name="train",
        data_source=tasks.DataSource(
            s3_data_source=tasks.S3DataSource(
                s3_data_type=tasks.S3DataType.S3_PREFIX,
                s3_location=tasks.S3Location.from_json_expression("$.S3Bucket")
            )
        )
    )],
    output_data_config=tasks.OutputDataConfig(
        s3_output_location=tasks.S3Location.from_bucket(s3.Bucket.from_bucket_name(self, "Bucket", "amzn-s3-demo-bucket"), "myoutputpath")
    ),
    resource_config=tasks.ResourceConfig(
        instance_count=1,
        instance_type=ec2.InstanceType(sfn.JsonPath.string_at("$.InstanceType")),
        volume_size=Size.gibibytes(50)
    ),  # optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
    stopping_condition=tasks.StoppingCondition(
        max_runtime=Duration.hours(2)
    )
)
```

You can specify [TrainingInputMode](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_AlgorithmSpecification.html#API_AlgorithmSpecification_Contents) via the trainingInputMode property.

* To download the data from Amazon Simple Storage Service (Amazon S3) to the provisioned ML storage volume, and mount the directory to a Docker volume, choose `InputMode.FILE` if an algorithm supports it.
* To stream data directly from Amazon S3 to the container, choose `InputMode.PIPE` if an algorithm supports it.
* To stream data directly from Amazon S3 to the container with no code changes and to provide file system access to the data, choose `InputMode.FAST_FILE` if an algorithm supports it.

### Create Transform Job

You can call the [`CreateTransformJob`](https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTransformJob.html) API from a `Task` state.

```python
tasks.SageMakerCreateTransformJob(self, "Batch Inference",
    transform_job_name="MyTransformJob",
    model_name="MyModelName",
    model_client_options=tasks.ModelClientOptions(
        invocations_max_retries=3,  # default is 0
        invocations_timeout=Duration.minutes(5)
    ),
    transform_input=tasks.TransformInput(
        transform_data_source=tasks.TransformDataSource(
            s3_data_source=tasks.TransformS3DataSource(
                s3_uri="s3://inputbucket/train",
                s3_data_type=tasks.S3DataType.S3_PREFIX
            )
        )
    ),
    transform_output=tasks.TransformOutput(
        s3_output_path="s3://outputbucket/TransformJobOutputPath"
    ),
    transform_resources=tasks.TransformResources(
        instance_count=1,
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE)
    )
)
```

### Create Endpoint

You can call the [`CreateEndpoint`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpoint.html) API from a `Task` state.

```python
tasks.SageMakerCreateEndpoint(self, "SagemakerEndpoint",
    endpoint_name=sfn.JsonPath.string_at("$.EndpointName"),
    endpoint_config_name=sfn.JsonPath.string_at("$.EndpointConfigName")
)
```

### Create Endpoint Config

You can call the [`CreateEndpointConfig`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpointConfig.html) API from a `Task` state.

```python
tasks.SageMakerCreateEndpointConfig(self, "SagemakerEndpointConfig",
    endpoint_config_name="MyEndpointConfig",
    production_variants=[tasks.ProductionVariant(
        initial_instance_count=2,
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.XLARGE),
        model_name="MyModel",
        variant_name="awesome-variant"
    )]
)
```

### Create Model

You can call the [`CreateModel`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModel.html) API from a `Task` state.

```python
tasks.SageMakerCreateModel(self, "Sagemaker",
    model_name="MyModel",
    primary_container=tasks.ContainerDefinition(
        image=tasks.DockerImage.from_json_expression(sfn.JsonPath.string_at("$.Model.imageName")),
        mode=tasks.Mode.SINGLE_MODEL,
        model_s3_location=tasks.S3Location.from_json_expression("$.TrainingJob.ModelArtifacts.S3ModelArtifacts")
    )
)
```

### Update Endpoint

You can call the [`UpdateEndpoint`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateEndpoint.html) API from a `Task` state.

```python
tasks.SageMakerUpdateEndpoint(self, "SagemakerEndpoint",
    endpoint_name=sfn.JsonPath.string_at("$.Endpoint.Name"),
    endpoint_config_name=sfn.JsonPath.string_at("$.Endpoint.EndpointConfig")
)
```

## SNS

Step Functions supports [Amazon SNS](https://docs.aws.amazon.com/step-functions/latest/dg/connect-sns.html) through the service integration pattern.

### Publish

You can call the [`Publish`](https://docs.aws.amazon.com/sns/latest/api/API_Publish.html) API from a `Task` state to publish to an SNS topic.

```python
topic = sns.Topic(self, "Topic")

# Use a field from the execution data as message.
task1 = tasks.SnsPublish(self, "Publish1",
    topic=topic,
    integration_pattern=sfn.IntegrationPattern.REQUEST_RESPONSE,
    message=sfn.TaskInput.from_data_at("$.state.message"),
    message_attributes={
        "place": tasks.MessageAttribute(
            value=sfn.JsonPath.string_at("$.place")
        ),
        "pic": tasks.MessageAttribute(
            # BINARY must be explicitly set
            data_type=tasks.MessageAttributeDataType.BINARY,
            value=sfn.JsonPath.string_at("$.pic")
        ),
        "people": tasks.MessageAttribute(
            value=4
        ),
        "handles": tasks.MessageAttribute(
            value=["@kslater", "@jjf", null, "@mfanning"]
        )
    }
)

# Combine a field from the execution data with
# a literal object.
task2 = tasks.SnsPublish(self, "Publish2",
    topic=topic,
    message=sfn.TaskInput.from_object({
        "field1": "somedata",
        "field2": sfn.JsonPath.string_at("$.field2")
    })
)
```

## Step Functions

Step Functions supports [AWS Step Functions](https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html) through the service integration pattern.

### Start Execution

You can manage [AWS Step Functions](https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html) executions.

AWS Step Functions supports it's own [`StartExecution`](https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartExecution.html) API as a service integration.

```python
# Define a state machine with one Pass state
child = sfn.StateMachine(self, "ChildStateMachine",
    definition=sfn.Chain.start(sfn.Pass(self, "PassState"))
)

# Include the state machine in a Task state with callback pattern
task = tasks.StepFunctionsStartExecution(self, "ChildTask",
    state_machine=child,
    integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
    input=sfn.TaskInput.from_object({
        "token": sfn.JsonPath.task_token,
        "foo": "bar"
    }),
    name="MyExecutionName"
)

# Define a second state machine with the Task state above
sfn.StateMachine(self, "ParentStateMachine",
    definition=task
)
```

You can utilize [Associate Workflow Executions](https://docs.aws.amazon.com/step-functions/latest/dg/concepts-nested-workflows.html#nested-execution-startid)
via the `associateWithParent` property. This allows the Step Functions UI to link child
executions from parent executions, making it easier to trace execution flow across state machines.

```python
# child: sfn.StateMachine

task = tasks.StepFunctionsStartExecution(self, "ChildTask",
    state_machine=child,
    associate_with_parent=True
)
```

This will add the payload `AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID.$: $$.Execution.Id` to the
`input`property for you, which will pass the execution ID from the context object to the
execution input. It requires `input` to be an object or not be set at all.

### Invoke Activity

You can invoke a [Step Functions Activity](https://docs.aws.amazon.com/step-functions/latest/dg/concepts-activities.html) which enables you to have
a task in your state machine where the work is performed by a *worker* that can
be hosted on Amazon EC2, Amazon ECS, AWS Lambda, basically anywhere. Activities
are a way to associate code running somewhere (known as an activity worker) with
a specific task in a state machine.

When Step Functions reaches an activity task state, the workflow waits for an
activity worker to poll for a task. An activity worker polls Step Functions by
using GetActivityTask, and sending the ARN for the related activity.

After the activity worker completes its work, it can provide a report of its
success or failure by using `SendTaskSuccess` or `SendTaskFailure`. These two
calls use the taskToken provided by GetActivityTask to associate the result
with that task.

The following example creates an activity and creates a task that invokes the activity.

```python
submit_job_activity = sfn.Activity(self, "SubmitJob")

tasks.StepFunctionsInvokeActivity(self, "Submit Job",
    activity=submit_job_activity
)
```

Use the [Parameters](https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-parameters) field to create a collection of key-value pairs that are passed as input.
The values of each can either be static values that you include in your state machine definition, or selected from either the input or the context object with a path.

```python
submit_job_activity = sfn.Activity(self, "SubmitJob")

tasks.StepFunctionsInvokeActivity(self, "Submit Job",
    activity=submit_job_activity,
    parameters={
        "comment": "Selecting what I care about.",
        "MyDetails": {
            "size": sfn.JsonPath.string_at("$.product.details.size"),
            "exists": sfn.JsonPath.string_at("$.product.availability"),
            "StaticValue": "foo"
        }
    }
)
```

## SQS

Step Functions supports [Amazon SQS](https://docs.aws.amazon.com/step-functions/latest/dg/connect-sqs.html)

### Send Message

You can call the [`SendMessage`](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessage.html) API from a `Task` state
to send a message to an SQS queue.

```python
queue = sqs.Queue(self, "Queue")

# Use a field from the execution data as message.
task1 = tasks.SqsSendMessage(self, "Send1",
    queue=queue,
    message_body=sfn.TaskInput.from_json_path_at("$.message")
)

# Combine a field from the execution data with
# a literal object.
task2 = tasks.SqsSendMessage(self, "Send2",
    queue=queue,
    message_body=sfn.TaskInput.from_object({
        "field1": "somedata",
        "field2": sfn.JsonPath.string_at("$.field2")
    })
)
```
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    Duration as _Duration_4839e8c3,
    IgnoreMode as _IgnoreMode_655a98e8,
    Size as _Size_7b441c34,
    Stack as _Stack_2866e57f,
    SymlinkFollowMode as _SymlinkFollowMode_047ec1f6,
)
from ..aws_apigateway import IRestApi as _IRestApi_1f02523d
from ..aws_bedrock import IModel as _IModel_b88b328a
from ..aws_codebuild import (
    BuildEnvironmentVariable as _BuildEnvironmentVariable_7df4fa0c,
    IProject as _IProject_aafae30a,
)
from ..aws_dynamodb import ITable as _ITable_504fd401
from ..aws_ec2 import (
    Connections as _Connections_0f31fce8,
    IConnectable as _IConnectable_10015a05,
    ISecurityGroup as _ISecurityGroup_acf8a799,
    IVpc as _IVpc_f30d5663,
    InstanceSize as _InstanceSize_233b2620,
    InstanceType as _InstanceType_f64915b9,
    SubnetSelection as _SubnetSelection_e57d76df,
)
from ..aws_ecr import IRepository as _IRepository_e6004aa6
from ..aws_ecr_assets import (
    DockerCacheOption as _DockerCacheOption_58ef18ca,
    DockerImageAssetInvalidationOptions as _DockerImageAssetInvalidationOptions_4deb8d45,
    DockerImageAssetProps as _DockerImageAssetProps_6897287d,
    NetworkMode as _NetworkMode_897e5081,
    Platform as _Platform_d16f3cf1,
)
from ..aws_ecs import (
    ContainerDefinition as _ContainerDefinition_8f3b54dc,
    FargatePlatformVersion as _FargatePlatformVersion_55d8be5c,
    ICluster as _ICluster_16cddd09,
    ITaskDefinition as _ITaskDefinition_889ba4d8,
    PlacementConstraint as _PlacementConstraint_11d82a52,
    PlacementStrategy as _PlacementStrategy_2bb6c232,
    PropagatedTagSource as _PropagatedTagSource_ad4e874a,
    TaskDefinition as _TaskDefinition_a541a103,
)
from ..aws_eks import ICluster as _ICluster_6b2b80df
from ..aws_events import (
    IConnection as _IConnection_afaa0769, IEventBus as _IEventBus_88d13111
)
from ..aws_iam import (
    IGrantable as _IGrantable_71c4f5de,
    IPrincipal as _IPrincipal_539bb2fd,
    IRole as _IRole_235f5d8e,
    PolicyStatement as _PolicyStatement_0fe33853,
)
from ..aws_kms import IKey as _IKey_5f11635f
from ..aws_lambda import IFunction as _IFunction_6adb0ab8, Runtime as _Runtime_b4eaa844
from ..aws_logs import ILogGroup as _ILogGroup_3c4fa718
from ..aws_s3 import IBucket as _IBucket_42e086fd, Location as _Location_0948fa7f
from ..aws_sns import ITopic as _ITopic_9eca4852
from ..aws_sqs import IQueue as _IQueue_7ed6f679
from ..aws_stepfunctions import (
    Credentials as _Credentials_2cd64c6b,
    IActivity as _IActivity_3cf6efa2,
    IStateMachine as _IStateMachine_73e8d2b0,
    IntegrationPattern as _IntegrationPattern_949291bc,
    JsonataStateOptions as _JsonataStateOptions_ef099c3b,
    QueryLanguage as _QueryLanguage_be8414a8,
    ServiceIntegrationPattern as _ServiceIntegrationPattern_39230c49,
    TaskInput as _TaskInput_91b91b91,
    TaskMetricsConfig as _TaskMetricsConfig_32ea9403,
    TaskStateBase as _TaskStateBase_b5c0a816,
    TaskStateBaseProps as _TaskStateBaseProps_3a62b6d0,
    TaskStateJsonPathBaseProps as _TaskStateJsonPathBaseProps_57dc2260,
    TaskStateJsonataBaseProps as _TaskStateJsonataBaseProps_0e91775c,
    Timeout as _Timeout_d7c10551,
)


class AcceleratorClass(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AcceleratorClass",
):
    '''The generation of Elastic Inference (EI) instance.

    :see: https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        accelerator_class = stepfunctions_tasks.AcceleratorClass.of("version")
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, version: builtins.str) -> "AcceleratorClass":
        '''Custom AcceleratorType.

        :param version: - Elastic Inference accelerator generation.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__819c31c0c99426607244dc0c57ea1d116326b5a750cff637f19b64f1a598385c)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        return typing.cast("AcceleratorClass", jsii.sinvoke(cls, "of", [version]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EIA1")
    def EIA1(cls) -> "AcceleratorClass":
        '''Elastic Inference accelerator 1st generation.'''
        return typing.cast("AcceleratorClass", jsii.sget(cls, "EIA1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EIA2")
    def EIA2(cls) -> "AcceleratorClass":
        '''Elastic Inference accelerator 2nd generation.'''
        return typing.cast("AcceleratorClass", jsii.sget(cls, "EIA2"))

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> builtins.str:
        '''- Elastic Inference accelerator generation.'''
        return typing.cast(builtins.str, jsii.get(self, "version"))


class AcceleratorType(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AcceleratorType",
):
    '''The size of the Elastic Inference (EI) instance to use for the production variant.

    EI instances provide on-demand GPU computing for inference

    :see: https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        accelerator_type = stepfunctions_tasks.AcceleratorType("instanceTypeIdentifier")
    '''

    def __init__(self, instance_type_identifier: builtins.str) -> None:
        '''
        :param instance_type_identifier: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35c8fa35baadd94a7e0608eafaea8bc1776ff404188c0e8e931fe7a326a7bb88)
            check_type(argname="argument instance_type_identifier", value=instance_type_identifier, expected_type=type_hints["instance_type_identifier"])
        jsii.create(self.__class__, self, [instance_type_identifier])

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        accelerator_class: AcceleratorClass,
        instance_size: _InstanceSize_233b2620,
    ) -> "AcceleratorType":
        '''AcceleratorType.

        This class takes a combination of a class and size.

        :param accelerator_class: -
        :param instance_size: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7e8b0a3bd440b1d4ddc7b3420d17fb7fbb5662db801beeda1ce3cd59388af374)
            check_type(argname="argument accelerator_class", value=accelerator_class, expected_type=type_hints["accelerator_class"])
            check_type(argname="argument instance_size", value=instance_size, expected_type=type_hints["instance_size"])
        return typing.cast("AcceleratorType", jsii.sinvoke(cls, "of", [accelerator_class, instance_size]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Return the accelerator type as a dotted string.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.ActionAfterCompletion")
class ActionAfterCompletion(enum.Enum):
    '''The action that EventBridge Scheduler applies to the schedule after the schedule completes invoking the target.

    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_scheduler as scheduler
        import aws_cdk.aws_kms as kms
        
        # key: kms.Key
        # schedule_group: scheduler.CfnScheduleGroup
        # target_queue: sqs.Queue
        # dead_letter_queue: sqs.Queue
        
        
        scheduler_role = iam.Role(self, "SchedulerRole",
            assumed_by=iam.ServicePrincipal("scheduler.amazonaws.com")
        )
        # To send the message to the queue
        # This policy changes depending on the type of target.
        scheduler_role.add_to_principal_policy(iam.PolicyStatement(
            actions=["sqs:SendMessage"],
            resources=[target_queue.queue_arn]
        ))
        
        create_schedule_task1 = tasks.EventBridgeSchedulerCreateScheduleTask(self, "createSchedule",
            schedule_name="TestSchedule",
            action_after_completion=tasks.ActionAfterCompletion.NONE,
            client_token="testToken",
            description="TestDescription",
            start_date=Date(),
            end_date=Date(Date().get_time() + 1000 * 60 * 60),
            flexible_time_window=Duration.minutes(5),
            group_name=schedule_group.ref,
            kms_key=key,
            schedule=tasks.Schedule.rate(Duration.minutes(5)),
            timezone="UTC",
            enabled=True,
            target=tasks.EventBridgeSchedulerTarget(
                arn=target_queue.queue_arn,
                role=scheduler_role,
                retry_policy=tasks.RetryPolicy(
                    maximum_retry_attempts=2,
                    maximum_event_age=Duration.minutes(5)
                ),
                dead_letter_queue=dead_letter_queue
            )
        )
    '''

    NONE = "NONE"
    '''Takes no action.'''
    DELETE = "DELETE"
    '''Deletes the schedule.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.ActionOnFailure")
class ActionOnFailure(enum.Enum):
    '''The action to take when the cluster step fails.

    :default: CONTINUE

    :see:

    https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html

    Here, they are named as TERMINATE_JOB_FLOW, TERMINATE_CLUSTER, CANCEL_AND_WAIT, and CONTINUE respectively.
    :exampleMetadata: infused

    Example::

        tasks.EmrAddStep(self, "Task",
            cluster_id="ClusterId",
            name="StepName",
            jar="Jar",
            action_on_failure=tasks.ActionOnFailure.CONTINUE
        )
    '''

    TERMINATE_CLUSTER = "TERMINATE_CLUSTER"
    '''Terminate the Cluster on Step Failure.'''
    CANCEL_AND_WAIT = "CANCEL_AND_WAIT"
    '''Cancel Step execution and enter WAITING state.'''
    CONTINUE = "CONTINUE"
    '''Continue to the next Step.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AlgorithmSpecification",
    jsii_struct_bases=[],
    name_mapping={
        "algorithm_name": "algorithmName",
        "metric_definitions": "metricDefinitions",
        "training_image": "trainingImage",
        "training_input_mode": "trainingInputMode",
    },
)
class AlgorithmSpecification:
    def __init__(
        self,
        *,
        algorithm_name: typing.Optional[builtins.str] = None,
        metric_definitions: typing.Optional[typing.Sequence[typing.Union["MetricDefinition", typing.Dict[builtins.str, typing.Any]]]] = None,
        training_image: typing.Optional["DockerImage"] = None,
        training_input_mode: typing.Optional["InputMode"] = None,
    ) -> None:
        '''Specify the training algorithm and algorithm-specific metadata.

        :param algorithm_name: Name of the algorithm resource to use for the training job. This must be an algorithm resource that you created or subscribe to on AWS Marketplace. If you specify a value for this parameter, you can't specify a value for TrainingImage. Default: - No algorithm is specified
        :param metric_definitions: List of metric definition objects. Each object specifies the metric name and regular expressions used to parse algorithm logs. Default: - No metrics
        :param training_image: Registry path of the Docker image that contains the training algorithm. Default: - No Docker image is specified
        :param training_input_mode: Input mode that the algorithm supports. Default: 'File' mode

        :exampleMetadata: infused

        Example::

            tasks.SageMakerCreateTrainingJob(self, "TrainSagemaker",
                training_job_name=sfn.JsonPath.string_at("$.JobName"),
                algorithm_specification=tasks.AlgorithmSpecification(
                    algorithm_name="BlazingText",
                    training_input_mode=tasks.InputMode.FILE
                ),
                input_data_config=[tasks.Channel(
                    channel_name="train",
                    data_source=tasks.DataSource(
                        s3_data_source=tasks.S3DataSource(
                            s3_data_type=tasks.S3DataType.S3_PREFIX,
                            s3_location=tasks.S3Location.from_json_expression("$.S3Bucket")
                        )
                    )
                )],
                output_data_config=tasks.OutputDataConfig(
                    s3_output_location=tasks.S3Location.from_bucket(s3.Bucket.from_bucket_name(self, "Bucket", "amzn-s3-demo-bucket"), "myoutputpath")
                ),
                resource_config=tasks.ResourceConfig(
                    instance_count=1,
                    instance_type=ec2.InstanceType(sfn.JsonPath.string_at("$.InstanceType")),
                    volume_size=Size.gibibytes(50)
                ),  # optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
                stopping_condition=tasks.StoppingCondition(
                    max_runtime=Duration.hours(2)
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e11c280b4cbc931a6ec0d0b6e960ccb7f67456a5fcb77cb1fe1a027cb3c9be4d)
            check_type(argname="argument algorithm_name", value=algorithm_name, expected_type=type_hints["algorithm_name"])
            check_type(argname="argument metric_definitions", value=metric_definitions, expected_type=type_hints["metric_definitions"])
            check_type(argname="argument training_image", value=training_image, expected_type=type_hints["training_image"])
            check_type(argname="argument training_input_mode", value=training_input_mode, expected_type=type_hints["training_input_mode"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if algorithm_name is not None:
            self._values["algorithm_name"] = algorithm_name
        if metric_definitions is not None:
            self._values["metric_definitions"] = metric_definitions
        if training_image is not None:
            self._values["training_image"] = training_image
        if training_input_mode is not None:
            self._values["training_input_mode"] = training_input_mode

    @builtins.property
    def algorithm_name(self) -> typing.Optional[builtins.str]:
        '''Name of the algorithm resource to use for the training job.

        This must be an algorithm resource that you created or subscribe to on AWS Marketplace.
        If you specify a value for this parameter, you can't specify a value for TrainingImage.

        :default: - No algorithm is specified
        '''
        result = self._values.get("algorithm_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def metric_definitions(self) -> typing.Optional[typing.List["MetricDefinition"]]:
        '''List of metric definition objects.

        Each object specifies the metric name and regular expressions used to parse algorithm logs.

        :default: - No metrics
        '''
        result = self._values.get("metric_definitions")
        return typing.cast(typing.Optional[typing.List["MetricDefinition"]], result)

    @builtins.property
    def training_image(self) -> typing.Optional["DockerImage"]:
        '''Registry path of the Docker image that contains the training algorithm.

        :default: - No Docker image is specified
        '''
        result = self._values.get("training_image")
        return typing.cast(typing.Optional["DockerImage"], result)

    @builtins.property
    def training_input_mode(self) -> typing.Optional["InputMode"]:
        '''Input mode that the algorithm supports.

        :default: 'File' mode
        '''
        result = self._values.get("training_input_mode")
        return typing.cast(typing.Optional["InputMode"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AlgorithmSpecification(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.ApplicationConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "classification": "classification",
        "nested_config": "nestedConfig",
        "properties": "properties",
    },
)
class ApplicationConfiguration:
    def __init__(
        self,
        *,
        classification: "Classification",
        nested_config: typing.Optional[typing.Sequence[typing.Union["ApplicationConfiguration", typing.Dict[builtins.str, typing.Any]]]] = None,
        properties: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''A configuration specification to be used when provisioning virtual clusters, which can include configurations for applications and software bundled with Amazon EMR on EKS.

        A configuration consists of a classification, properties, and optional nested configurations.
        A classification refers to an application-specific configuration file.
        Properties are the settings you want to change in that file.

        :param classification: The classification within a configuration. Length Constraints: Minimum length of 1. Maximum length of 1024.
        :param nested_config: A list of additional configurations to apply within a configuration object. Array Members: Maximum number of 100 items. Default: - No other configurations
        :param properties: A set of properties specified within a configuration classification. Map Entries: Maximum number of 100 items. Default: - No properties

        :see: https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-configure-apps.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # application_configuration_: stepfunctions_tasks.ApplicationConfiguration
            # classification: stepfunctions_tasks.Classification
            
            application_configuration = stepfunctions_tasks.ApplicationConfiguration(
                classification=classification,
            
                # the properties below are optional
                nested_config=[application_configuration_],
                properties={
                    "properties_key": "properties"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2838fe25c75eb2d6ec4a40b92699edb9f4ed7d7743cd75d5155954a10402d192)
            check_type(argname="argument classification", value=classification, expected_type=type_hints["classification"])
            check_type(argname="argument nested_config", value=nested_config, expected_type=type_hints["nested_config"])
            check_type(argname="argument properties", value=properties, expected_type=type_hints["properties"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "classification": classification,
        }
        if nested_config is not None:
            self._values["nested_config"] = nested_config
        if properties is not None:
            self._values["properties"] = properties

    @builtins.property
    def classification(self) -> "Classification":
        '''The classification within a configuration.

        Length Constraints: Minimum length of 1. Maximum length of 1024.
        '''
        result = self._values.get("classification")
        assert result is not None, "Required property 'classification' is missing"
        return typing.cast("Classification", result)

    @builtins.property
    def nested_config(self) -> typing.Optional[typing.List["ApplicationConfiguration"]]:
        '''A list of additional configurations to apply within a configuration object.

        Array Members: Maximum number of 100 items.

        :default: - No other configurations
        '''
        result = self._values.get("nested_config")
        return typing.cast(typing.Optional[typing.List["ApplicationConfiguration"]], result)

    @builtins.property
    def properties(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''A set of properties specified within a configuration classification.

        Map Entries: Maximum number of 100 items.

        :default: - No properties
        '''
        result = self._values.get("properties")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApplicationConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AssembleWith")
class AssembleWith(enum.Enum):
    '''How to assemble the results of the transform job as a single S3 object.'''

    NONE = "NONE"
    '''Concatenate the results in binary format.'''
    LINE = "LINE"
    '''Add a newline character at the end of every transformed record.'''


class AthenaGetQueryExecution(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryExecution",
):
    '''Get an Athena Query Execution as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :exampleMetadata: infused

    Example::

        get_query_execution_job = tasks.AthenaGetQueryExecution(self, "Get Query Execution",
            query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f874d4b3d8cca8a1aa5d7809bc14fb2078e4f8cbdcf8a7ade80f98ea33352417)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaGetQueryExecutionProps(
            query_execution_id=query_execution_id,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
    ) -> "AthenaGetQueryExecution":
        '''Get an Athena Query Execution as a Task that using JSONata.

        :param scope: -
        :param id: -
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e4bb2129df6049d2eb9660152886d1b0a4e92831cdade43be61fd6e17e2773aa)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaGetQueryExecutionJsonataProps(
            query_execution_id=query_execution_id,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            outputs=outputs,
        )

        return typing.cast("AthenaGetQueryExecution", jsii.sinvoke(cls, "jsonata", [scope, id, props]))

    @jsii.member(jsii_name="jsonPath")
    @builtins.classmethod
    def json_path(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> "AthenaGetQueryExecution":
        '''Get an Athena Query Execution as a Task that using JSONPath.

        :param scope: -
        :param id: -
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e78a30035ec43e731ee3d24092dc3d20f4d5c5d2a8efb5f8d8137a7ab449eaf3)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaGetQueryExecutionJsonPathProps(
            query_execution_id=query_execution_id,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
        )

        return typing.cast("AthenaGetQueryExecution", jsii.sinvoke(cls, "jsonPath", [scope, id, props]))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryExecutionJsonPathProps",
    jsii_struct_bases=[_TaskStateJsonPathBaseProps_57dc2260],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "query_execution_id": "queryExecutionId",
    },
)
class AthenaGetQueryExecutionJsonPathProps(_TaskStateJsonPathBaseProps_57dc2260):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        query_execution_id: builtins.str,
    ) -> None:
        '''Properties for getting a Query Execution using JSONPath.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # result_selector: Any
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            athena_get_query_execution_json_path_props = stepfunctions_tasks.AthenaGetQueryExecutionJsonPathProps(
                query_execution_id="queryExecutionId",
            
                # the properties below are optional
                assign={
                    "assign_key": assign
                },
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output_path="outputPath",
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                result_path="resultPath",
                result_selector={
                    "result_selector_key": result_selector
                },
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2e7223ee7f40243c8be914d8206f4748c58ab6f82ac01df7ec2a3305cead289b)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be retrieved.

        Example value: ``adfsaf-23trf23-f23rt23``
        '''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaGetQueryExecutionJsonPathProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryExecutionJsonataProps",
    jsii_struct_bases=[_TaskStateJsonataBaseProps_0e91775c],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "outputs": "outputs",
        "query_execution_id": "queryExecutionId",
    },
)
class AthenaGetQueryExecutionJsonataProps(_TaskStateJsonataBaseProps_0e91775c):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
        query_execution_id: builtins.str,
    ) -> None:
        '''Properties for getting a Query Execution using JSONata.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # outputs: Any
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            athena_get_query_execution_jsonata_props = stepfunctions_tasks.AthenaGetQueryExecutionJsonataProps(
                query_execution_id="queryExecutionId",
            
                # the properties below are optional
                assign={
                    "assign_key": assign
                },
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                outputs=outputs,
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__55a1fcfcb931a65a4856a1d905ead7f9267b952f1e611de9fe9f893726f8d77f)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if outputs is not None:
            self._values["outputs"] = outputs

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be retrieved.

        Example value: ``adfsaf-23trf23-f23rt23``
        '''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaGetQueryExecutionJsonataProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryExecutionProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "outputs": "outputs",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "query_execution_id": "queryExecutionId",
    },
)
class AthenaGetQueryExecutionProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        query_execution_id: builtins.str,
    ) -> None:
        '''Properties for getting a Query Execution.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``

        :exampleMetadata: infused

        Example::

            get_query_execution_job = tasks.AthenaGetQueryExecution(self, "Get Query Execution",
                query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__426b683b3fff9459413103d38b8f467a1eef7b493511c26ddef6ccaf20d191b5)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if outputs is not None:
            self._values["outputs"] = outputs
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be retrieved.

        Example value: ``adfsaf-23trf23-f23rt23``
        '''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaGetQueryExecutionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AthenaGetQueryResults(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryResults",
):
    '''Get an Athena Query Results as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :exampleMetadata: infused

    Example::

        get_query_results_job = tasks.AthenaGetQueryResults(self, "Get Query Results",
            query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        max_results: typing.Optional[jsii.Number] = None,
        next_token: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param max_results: Max number of results. Default: 1000
        :param next_token: Pagination token. Default: - No next token
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4fbcf740a1e3e37ea0b4976ce01dbba60ab814c75d0b05f5ea68b86ab3392764)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaGetQueryResultsProps(
            query_execution_id=query_execution_id,
            max_results=max_results,
            next_token=next_token,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        max_results: typing.Optional[jsii.Number] = None,
        next_token: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
    ) -> "AthenaGetQueryResults":
        '''Get an Athena Query Results as a Task that using JSONata.

        :param scope: -
        :param id: -
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param max_results: Max number of results. Default: 1000
        :param next_token: Pagination token. Default: - No next token
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__001bc459fdd95a57a16c610596108126cb360c74107d16b168fc34de2fe48d77)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaGetQueryResultsJsonataProps(
            query_execution_id=query_execution_id,
            max_results=max_results,
            next_token=next_token,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            outputs=outputs,
        )

        return typing.cast("AthenaGetQueryResults", jsii.sinvoke(cls, "jsonata", [scope, id, props]))

    @jsii.member(jsii_name="jsonPath")
    @builtins.classmethod
    def json_path(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        max_results: typing.Optional[jsii.Number] = None,
        next_token: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> "AthenaGetQueryResults":
        '''Get an Athena Query Results as a Task that using JSONPath.

        :param scope: -
        :param id: -
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param max_results: Max number of results. Default: 1000
        :param next_token: Pagination token. Default: - No next token
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0b41d92701be8dc82f1782e6fcee3f48f6f4cc6815192db587d750905722ea16)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaGetQueryResultsJsonPathProps(
            query_execution_id=query_execution_id,
            max_results=max_results,
            next_token=next_token,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
        )

        return typing.cast("AthenaGetQueryResults", jsii.sinvoke(cls, "jsonPath", [scope, id, props]))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryResultsJsonPathProps",
    jsii_struct_bases=[_TaskStateJsonPathBaseProps_57dc2260],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "query_execution_id": "queryExecutionId",
        "max_results": "maxResults",
        "next_token": "nextToken",
    },
)
class AthenaGetQueryResultsJsonPathProps(_TaskStateJsonPathBaseProps_57dc2260):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        query_execution_id: builtins.str,
        max_results: typing.Optional[jsii.Number] = None,
        next_token: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for getting a Query Results using JSONPath.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param max_results: Max number of results. Default: 1000
        :param next_token: Pagination token. Default: - No next token

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # result_selector: Any
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            athena_get_query_results_json_path_props = stepfunctions_tasks.AthenaGetQueryResultsJsonPathProps(
                query_execution_id="queryExecutionId",
            
                # the properties below are optional
                assign={
                    "assign_key": assign
                },
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                max_results=123,
                next_token="nextToken",
                output_path="outputPath",
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                result_path="resultPath",
                result_selector={
                    "result_selector_key": result_selector
                },
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__18855f188eae28d98b146a498124216105af34edc7e4abcfac4bdab981dbb0c4)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
            check_type(argname="argument max_results", value=max_results, expected_type=type_hints["max_results"])
            check_type(argname="argument next_token", value=next_token, expected_type=type_hints["next_token"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if max_results is not None:
            self._values["max_results"] = max_results
        if next_token is not None:
            self._values["next_token"] = next_token

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be retrieved.

        Example value: ``adfsaf-23trf23-f23rt23``
        '''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def max_results(self) -> typing.Optional[jsii.Number]:
        '''Max number of results.

        :default: 1000
        '''
        result = self._values.get("max_results")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def next_token(self) -> typing.Optional[builtins.str]:
        '''Pagination token.

        :default: - No next token
        '''
        result = self._values.get("next_token")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaGetQueryResultsJsonPathProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryResultsJsonataProps",
    jsii_struct_bases=[_TaskStateJsonataBaseProps_0e91775c],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "outputs": "outputs",
        "query_execution_id": "queryExecutionId",
        "max_results": "maxResults",
        "next_token": "nextToken",
    },
)
class AthenaGetQueryResultsJsonataProps(_TaskStateJsonataBaseProps_0e91775c):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
        query_execution_id: builtins.str,
        max_results: typing.Optional[jsii.Number] = None,
        next_token: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for getting a Query Results using JSONata.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param max_results: Max number of results. Default: 1000
        :param next_token: Pagination token. Default: - No next token

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # outputs: Any
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            athena_get_query_results_jsonata_props = stepfunctions_tasks.AthenaGetQueryResultsJsonataProps(
                query_execution_id="queryExecutionId",
            
                # the properties below are optional
                assign={
                    "assign_key": assign
                },
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                max_results=123,
                next_token="nextToken",
                outputs=outputs,
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__277ad530996a22bd73c250e70addcf13a5a8c4667835cd4ae2d39baa6f3b5554)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
            check_type(argname="argument max_results", value=max_results, expected_type=type_hints["max_results"])
            check_type(argname="argument next_token", value=next_token, expected_type=type_hints["next_token"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if outputs is not None:
            self._values["outputs"] = outputs
        if max_results is not None:
            self._values["max_results"] = max_results
        if next_token is not None:
            self._values["next_token"] = next_token

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be retrieved.

        Example value: ``adfsaf-23trf23-f23rt23``
        '''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def max_results(self) -> typing.Optional[jsii.Number]:
        '''Max number of results.

        :default: 1000
        '''
        result = self._values.get("max_results")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def next_token(self) -> typing.Optional[builtins.str]:
        '''Pagination token.

        :default: - No next token
        '''
        result = self._values.get("next_token")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaGetQueryResultsJsonataProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaGetQueryResultsProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "outputs": "outputs",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "query_execution_id": "queryExecutionId",
        "max_results": "maxResults",
        "next_token": "nextToken",
    },
)
class AthenaGetQueryResultsProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        query_execution_id: builtins.str,
        max_results: typing.Optional[jsii.Number] = None,
        next_token: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for getting a Query Results.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param query_execution_id: Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param max_results: Max number of results. Default: 1000
        :param next_token: Pagination token. Default: - No next token

        :exampleMetadata: infused

        Example::

            get_query_results_job = tasks.AthenaGetQueryResults(self, "Get Query Results",
                query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e62d5ebfcc36e5adfee6cea02c96ccdc0cba73b7dc4169d47fcd62e6b0b398db)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
            check_type(argname="argument max_results", value=max_results, expected_type=type_hints["max_results"])
            check_type(argname="argument next_token", value=next_token, expected_type=type_hints["next_token"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if outputs is not None:
            self._values["outputs"] = outputs
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if max_results is not None:
            self._values["max_results"] = max_results
        if next_token is not None:
            self._values["next_token"] = next_token

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be retrieved.

        Example value: ``adfsaf-23trf23-f23rt23``
        '''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def max_results(self) -> typing.Optional[jsii.Number]:
        '''Max number of results.

        :default: 1000
        '''
        result = self._values.get("max_results")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def next_token(self) -> typing.Optional[builtins.str]:
        '''Pagination token.

        :default: - No next token
        '''
        result = self._values.get("next_token")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaGetQueryResultsProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AthenaStartQueryExecution(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStartQueryExecution",
):
    '''Start an Athena Query as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :exampleMetadata: infused

    Example::

        start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Start Athena Query",
            query_string=sfn.JsonPath.string_at("$.queryString"),
            query_execution_context=tasks.QueryExecutionContext(
                database_name="mydatabase"
            ),
            result_configuration=tasks.ResultConfiguration(
                encryption_configuration=tasks.EncryptionConfiguration(
                    encryption_option=tasks.EncryptionOption.S3_MANAGED
                ),
                output_location=s3.Location(
                    bucket_name="amzn-s3-demo-bucket",
                    object_key="folder"
                )
            ),
            execution_parameters=["param1", "param2"]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_string: builtins.str,
        client_request_token: typing.Optional[builtins.str] = None,
        execution_parameters: typing.Optional[typing.Sequence[builtins.str]] = None,
        query_execution_context: typing.Optional[typing.Union["QueryExecutionContext", typing.Dict[builtins.str, typing.Any]]] = None,
        result_configuration: typing.Optional[typing.Union["ResultConfiguration", typing.Dict[builtins.str, typing.Any]]] = None,
        result_reuse_configuration_max_age: typing.Optional[_Duration_4839e8c3] = None,
        work_group: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param query_string: Query that will be started.
        :param client_request_token: Unique string string to ensure idempotence. Default: - No client request token
        :param execution_parameters: A list of values for the parameters in a query. The values are applied sequentially to the parameters in the query in the order in which the parameters occur. Default: - No parameters
        :param query_execution_context: Database within which query executes. Default: - No query execution context
        :param result_configuration: Configuration on how and where to save query. Default: - No result configuration
        :param result_reuse_configuration_max_age: Specifies, in minutes, the maximum age of a previous query result that Athena should consider for reuse. Default: - Query results are not reused
        :param work_group: Configuration on how and where to save query. Default: - No work group
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__74ee21cd0bad6760da4cd18a6c4cd846a24d69d25b7bac8eb004edf6403103e8)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaStartQueryExecutionProps(
            query_string=query_string,
            client_request_token=client_request_token,
            execution_parameters=execution_parameters,
            query_execution_context=query_execution_context,
            result_configuration=result_configuration,
            result_reuse_configuration_max_age=result_reuse_configuration_max_age,
            work_group=work_group,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_string: builtins.str,
        client_request_token: typing.Optional[builtins.str] = None,
        execution_parameters: typing.Optional[typing.Sequence[builtins.str]] = None,
        query_execution_context: typing.Optional[typing.Union["QueryExecutionContext", typing.Dict[builtins.str, typing.Any]]] = None,
        result_configuration: typing.Optional[typing.Union["ResultConfiguration", typing.Dict[builtins.str, typing.Any]]] = None,
        result_reuse_configuration_max_age: typing.Optional[_Duration_4839e8c3] = None,
        work_group: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
    ) -> "AthenaStartQueryExecution":
        '''Start an Athena Query as a Task using JSONata.

        :param scope: -
        :param id: -
        :param query_string: Query that will be started.
        :param client_request_token: Unique string string to ensure idempotence. Default: - No client request token
        :param execution_parameters: A list of values for the parameters in a query. The values are applied sequentially to the parameters in the query in the order in which the parameters occur. Default: - No parameters
        :param query_execution_context: Database within which query executes. Default: - No query execution context
        :param result_configuration: Configuration on how and where to save query. Default: - No result configuration
        :param result_reuse_configuration_max_age: Specifies, in minutes, the maximum age of a previous query result that Athena should consider for reuse. Default: - Query results are not reused
        :param work_group: Configuration on how and where to save query. Default: - No work group
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8aa308732891a887435f78ac4c5fab6b8a95de36330a693882db1e7b07d2d1bc)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaStartQueryExecutionJsonataProps(
            query_string=query_string,
            client_request_token=client_request_token,
            execution_parameters=execution_parameters,
            query_execution_context=query_execution_context,
            result_configuration=result_configuration,
            result_reuse_configuration_max_age=result_reuse_configuration_max_age,
            work_group=work_group,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            outputs=outputs,
        )

        return typing.cast("AthenaStartQueryExecution", jsii.sinvoke(cls, "jsonata", [scope, id, props]))

    @jsii.member(jsii_name="jsonPath")
    @builtins.classmethod
    def json_path(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_string: builtins.str,
        client_request_token: typing.Optional[builtins.str] = None,
        execution_parameters: typing.Optional[typing.Sequence[builtins.str]] = None,
        query_execution_context: typing.Optional[typing.Union["QueryExecutionContext", typing.Dict[builtins.str, typing.Any]]] = None,
        result_configuration: typing.Optional[typing.Union["ResultConfiguration", typing.Dict[builtins.str, typing.Any]]] = None,
        result_reuse_configuration_max_age: typing.Optional[_Duration_4839e8c3] = None,
        work_group: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> "AthenaStartQueryExecution":
        '''Start an Athena Query as a Task using JSONPath.

        :param scope: -
        :param id: -
        :param query_string: Query that will be started.
        :param client_request_token: Unique string string to ensure idempotence. Default: - No client request token
        :param execution_parameters: A list of values for the parameters in a query. The values are applied sequentially to the parameters in the query in the order in which the parameters occur. Default: - No parameters
        :param query_execution_context: Database within which query executes. Default: - No query execution context
        :param result_configuration: Configuration on how and where to save query. Default: - No result configuration
        :param result_reuse_configuration_max_age: Specifies, in minutes, the maximum age of a previous query result that Athena should consider for reuse. Default: - Query results are not reused
        :param work_group: Configuration on how and where to save query. Default: - No work group
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2657a4bdaa73685fd889d64bc3960a2199852e66f8594cddac36002fa86d267b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaStartQueryExecutionJsonPathProps(
            query_string=query_string,
            client_request_token=client_request_token,
            execution_parameters=execution_parameters,
            query_execution_context=query_execution_context,
            result_configuration=result_configuration,
            result_reuse_configuration_max_age=result_reuse_configuration_max_age,
            work_group=work_group,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
        )

        return typing.cast("AthenaStartQueryExecution", jsii.sinvoke(cls, "jsonPath", [scope, id, props]))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStartQueryExecutionJsonPathProps",
    jsii_struct_bases=[_TaskStateJsonPathBaseProps_57dc2260],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "query_string": "queryString",
        "client_request_token": "clientRequestToken",
        "execution_parameters": "executionParameters",
        "query_execution_context": "queryExecutionContext",
        "result_configuration": "resultConfiguration",
        "result_reuse_configuration_max_age": "resultReuseConfigurationMaxAge",
        "work_group": "workGroup",
    },
)
class AthenaStartQueryExecutionJsonPathProps(_TaskStateJsonPathBaseProps_57dc2260):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        query_string: builtins.str,
        client_request_token: typing.Optional[builtins.str] = None,
        execution_parameters: typing.Optional[typing.Sequence[builtins.str]] = None,
        query_execution_context: typing.Optional[typing.Union["QueryExecutionContext", typing.Dict[builtins.str, typing.Any]]] = None,
        result_configuration: typing.Optional[typing.Union["ResultConfiguration", typing.Dict[builtins.str, typing.Any]]] = None,
        result_reuse_configuration_max_age: typing.Optional[_Duration_4839e8c3] = None,
        work_group: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for starting a Query Execution using JSONPath.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param query_string: Query that will be started.
        :param client_request_token: Unique string string to ensure idempotence. Default: - No client request token
        :param execution_parameters: A list of values for the parameters in a query. The values are applied sequentially to the parameters in the query in the order in which the parameters occur. Default: - No parameters
        :param query_execution_context: Database within which query executes. Default: - No query execution context
        :param result_configuration: Configuration on how and where to save query. Default: - No result configuration
        :param result_reuse_configuration_max_age: Specifies, in minutes, the maximum age of a previous query result that Athena should consider for reuse. Default: - Query results are not reused
        :param work_group: Configuration on how and where to save query. Default: - No work group

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_kms as kms
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # key: kms.Key
            # result_selector: Any
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            athena_start_query_execution_json_path_props = stepfunctions_tasks.AthenaStartQueryExecutionJsonPathProps(
                query_string="queryString",
            
                # the properties below are optional
                assign={
                    "assign_key": assign
                },
                client_request_token="clientRequestToken",
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                execution_parameters=["executionParameters"],
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output_path="outputPath",
                query_execution_context=stepfunctions_tasks.QueryExecutionContext(
                    catalog_name="catalogName",
                    database_name="databaseName"
                ),
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                result_configuration=stepfunctions_tasks.ResultConfiguration(
                    encryption_configuration=stepfunctions_tasks.EncryptionConfiguration(
                        encryption_option=stepfunctions_tasks.EncryptionOption.S3_MANAGED,
            
                        # the properties below are optional
                        encryption_key=key
                    ),
                    output_location=cdk.aws_s3.Location(
                        bucket_name="bucketName",
                        object_key="objectKey",
            
                        # the properties below are optional
                        object_version="objectVersion"
                    )
                ),
                result_path="resultPath",
                result_reuse_configuration_max_age=cdk.Duration.minutes(30),
                result_selector={
                    "result_selector_key": result_selector
                },
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30),
                work_group="workGroup"
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(query_execution_context, dict):
            query_execution_context = QueryExecutionContext(**query_execution_context)
        if isinstance(result_configuration, dict):
            result_configuration = ResultConfiguration(**result_configuration)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d8babbf84a89d95d7ea0b7eefcd23e72b802b426cd9ecd54331b72fb8a3b04f0)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument query_string", value=query_string, expected_type=type_hints["query_string"])
            check_type(argname="argument client_request_token", value=client_request_token, expected_type=type_hints["client_request_token"])
            check_type(argname="argument execution_parameters", value=execution_parameters, expected_type=type_hints["execution_parameters"])
            check_type(argname="argument query_execution_context", value=query_execution_context, expected_type=type_hints["query_execution_context"])
            check_type(argname="argument result_configuration", value=result_configuration, expected_type=type_hints["result_configuration"])
            check_type(argname="argument result_reuse_configuration_max_age", value=result_reuse_configuration_max_age, expected_type=type_hints["result_reuse_configuration_max_age"])
            check_type(argname="argument work_group", value=work_group, expected_type=type_hints["work_group"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_string": query_string,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if client_request_token is not None:
            self._values["client_request_token"] = client_request_token
        if execution_parameters is not None:
            self._values["execution_parameters"] = execution_parameters
        if query_execution_context is not None:
            self._values["query_execution_context"] = query_execution_context
        if result_configuration is not None:
            self._values["result_configuration"] = result_configuration
        if result_reuse_configuration_max_age is not None:
            self._values["result_reuse_configuration_max_age"] = result_reuse_configuration_max_age
        if work_group is not None:
            self._values["work_group"] = work_group

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def query_string(self) -> builtins.str:
        '''Query that will be started.'''
        result = self._values.get("query_string")
        assert result is not None, "Required property 'query_string' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def client_request_token(self) -> typing.Optional[builtins.str]:
        '''Unique string string to ensure idempotence.

        :default: - No client request token
        '''
        result = self._values.get("client_request_token")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def execution_parameters(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of values for the parameters in a query.

        The values are applied sequentially to the parameters in the query in the order
        in which the parameters occur.

        :default: - No parameters
        '''
        result = self._values.get("execution_parameters")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def query_execution_context(self) -> typing.Optional["QueryExecutionContext"]:
        '''Database within which query executes.

        :default: - No query execution context
        '''
        result = self._values.get("query_execution_context")
        return typing.cast(typing.Optional["QueryExecutionContext"], result)

    @builtins.property
    def result_configuration(self) -> typing.Optional["ResultConfiguration"]:
        '''Configuration on how and where to save query.

        :default: - No result configuration
        '''
        result = self._values.get("result_configuration")
        return typing.cast(typing.Optional["ResultConfiguration"], result)

    @builtins.property
    def result_reuse_configuration_max_age(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Specifies, in minutes, the maximum age of a previous query result that Athena should consider for reuse.

        :default: - Query results are not reused
        '''
        result = self._values.get("result_reuse_configuration_max_age")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def work_group(self) -> typing.Optional[builtins.str]:
        '''Configuration on how and where to save query.

        :default: - No work group
        '''
        result = self._values.get("work_group")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaStartQueryExecutionJsonPathProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStartQueryExecutionJsonataProps",
    jsii_struct_bases=[_TaskStateJsonataBaseProps_0e91775c],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "outputs": "outputs",
        "query_string": "queryString",
        "client_request_token": "clientRequestToken",
        "execution_parameters": "executionParameters",
        "query_execution_context": "queryExecutionContext",
        "result_configuration": "resultConfiguration",
        "result_reuse_configuration_max_age": "resultReuseConfigurationMaxAge",
        "work_group": "workGroup",
    },
)
class AthenaStartQueryExecutionJsonataProps(_TaskStateJsonataBaseProps_0e91775c):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
        query_string: builtins.str,
        client_request_token: typing.Optional[builtins.str] = None,
        execution_parameters: typing.Optional[typing.Sequence[builtins.str]] = None,
        query_execution_context: typing.Optional[typing.Union["QueryExecutionContext", typing.Dict[builtins.str, typing.Any]]] = None,
        result_configuration: typing.Optional[typing.Union["ResultConfiguration", typing.Dict[builtins.str, typing.Any]]] = None,
        result_reuse_configuration_max_age: typing.Optional[_Duration_4839e8c3] = None,
        work_group: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for starting a Query Execution using JSONata.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param query_string: Query that will be started.
        :param client_request_token: Unique string string to ensure idempotence. Default: - No client request token
        :param execution_parameters: A list of values for the parameters in a query. The values are applied sequentially to the parameters in the query in the order in which the parameters occur. Default: - No parameters
        :param query_execution_context: Database within which query executes. Default: - No query execution context
        :param result_configuration: Configuration on how and where to save query. Default: - No result configuration
        :param result_reuse_configuration_max_age: Specifies, in minutes, the maximum age of a previous query result that Athena should consider for reuse. Default: - Query results are not reused
        :param work_group: Configuration on how and where to save query. Default: - No work group

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_kms as kms
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # key: kms.Key
            # outputs: Any
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            athena_start_query_execution_jsonata_props = stepfunctions_tasks.AthenaStartQueryExecutionJsonataProps(
                query_string="queryString",
            
                # the properties below are optional
                assign={
                    "assign_key": assign
                },
                client_request_token="clientRequestToken",
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                execution_parameters=["executionParameters"],
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                outputs=outputs,
                query_execution_context=stepfunctions_tasks.QueryExecutionContext(
                    catalog_name="catalogName",
                    database_name="databaseName"
                ),
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                result_configuration=stepfunctions_tasks.ResultConfiguration(
                    encryption_configuration=stepfunctions_tasks.EncryptionConfiguration(
                        encryption_option=stepfunctions_tasks.EncryptionOption.S3_MANAGED,
            
                        # the properties below are optional
                        encryption_key=key
                    ),
                    output_location=cdk.aws_s3.Location(
                        bucket_name="bucketName",
                        object_key="objectKey",
            
                        # the properties below are optional
                        object_version="objectVersion"
                    )
                ),
                result_reuse_configuration_max_age=cdk.Duration.minutes(30),
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30),
                work_group="workGroup"
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(query_execution_context, dict):
            query_execution_context = QueryExecutionContext(**query_execution_context)
        if isinstance(result_configuration, dict):
            result_configuration = ResultConfiguration(**result_configuration)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76349415078d6a22293bcce030a19501c34a6eab72fff7d5ef177c82d9120267)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument query_string", value=query_string, expected_type=type_hints["query_string"])
            check_type(argname="argument client_request_token", value=client_request_token, expected_type=type_hints["client_request_token"])
            check_type(argname="argument execution_parameters", value=execution_parameters, expected_type=type_hints["execution_parameters"])
            check_type(argname="argument query_execution_context", value=query_execution_context, expected_type=type_hints["query_execution_context"])
            check_type(argname="argument result_configuration", value=result_configuration, expected_type=type_hints["result_configuration"])
            check_type(argname="argument result_reuse_configuration_max_age", value=result_reuse_configuration_max_age, expected_type=type_hints["result_reuse_configuration_max_age"])
            check_type(argname="argument work_group", value=work_group, expected_type=type_hints["work_group"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_string": query_string,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if outputs is not None:
            self._values["outputs"] = outputs
        if client_request_token is not None:
            self._values["client_request_token"] = client_request_token
        if execution_parameters is not None:
            self._values["execution_parameters"] = execution_parameters
        if query_execution_context is not None:
            self._values["query_execution_context"] = query_execution_context
        if result_configuration is not None:
            self._values["result_configuration"] = result_configuration
        if result_reuse_configuration_max_age is not None:
            self._values["result_reuse_configuration_max_age"] = result_reuse_configuration_max_age
        if work_group is not None:
            self._values["work_group"] = work_group

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def query_string(self) -> builtins.str:
        '''Query that will be started.'''
        result = self._values.get("query_string")
        assert result is not None, "Required property 'query_string' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def client_request_token(self) -> typing.Optional[builtins.str]:
        '''Unique string string to ensure idempotence.

        :default: - No client request token
        '''
        result = self._values.get("client_request_token")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def execution_parameters(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of values for the parameters in a query.

        The values are applied sequentially to the parameters in the query in the order
        in which the parameters occur.

        :default: - No parameters
        '''
        result = self._values.get("execution_parameters")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def query_execution_context(self) -> typing.Optional["QueryExecutionContext"]:
        '''Database within which query executes.

        :default: - No query execution context
        '''
        result = self._values.get("query_execution_context")
        return typing.cast(typing.Optional["QueryExecutionContext"], result)

    @builtins.property
    def result_configuration(self) -> typing.Optional["ResultConfiguration"]:
        '''Configuration on how and where to save query.

        :default: - No result configuration
        '''
        result = self._values.get("result_configuration")
        return typing.cast(typing.Optional["ResultConfiguration"], result)

    @builtins.property
    def result_reuse_configuration_max_age(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Specifies, in minutes, the maximum age of a previous query result that Athena should consider for reuse.

        :default: - Query results are not reused
        '''
        result = self._values.get("result_reuse_configuration_max_age")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def work_group(self) -> typing.Optional[builtins.str]:
        '''Configuration on how and where to save query.

        :default: - No work group
        '''
        result = self._values.get("work_group")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaStartQueryExecutionJsonataProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStartQueryExecutionProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "outputs": "outputs",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "query_string": "queryString",
        "client_request_token": "clientRequestToken",
        "execution_parameters": "executionParameters",
        "query_execution_context": "queryExecutionContext",
        "result_configuration": "resultConfiguration",
        "result_reuse_configuration_max_age": "resultReuseConfigurationMaxAge",
        "work_group": "workGroup",
    },
)
class AthenaStartQueryExecutionProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        query_string: builtins.str,
        client_request_token: typing.Optional[builtins.str] = None,
        execution_parameters: typing.Optional[typing.Sequence[builtins.str]] = None,
        query_execution_context: typing.Optional[typing.Union["QueryExecutionContext", typing.Dict[builtins.str, typing.Any]]] = None,
        result_configuration: typing.Optional[typing.Union["ResultConfiguration", typing.Dict[builtins.str, typing.Any]]] = None,
        result_reuse_configuration_max_age: typing.Optional[_Duration_4839e8c3] = None,
        work_group: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for starting a Query Execution.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param query_string: Query that will be started.
        :param client_request_token: Unique string string to ensure idempotence. Default: - No client request token
        :param execution_parameters: A list of values for the parameters in a query. The values are applied sequentially to the parameters in the query in the order in which the parameters occur. Default: - No parameters
        :param query_execution_context: Database within which query executes. Default: - No query execution context
        :param result_configuration: Configuration on how and where to save query. Default: - No result configuration
        :param result_reuse_configuration_max_age: Specifies, in minutes, the maximum age of a previous query result that Athena should consider for reuse. Default: - Query results are not reused
        :param work_group: Configuration on how and where to save query. Default: - No work group

        :exampleMetadata: infused

        Example::

            start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Start Athena Query",
                query_string=sfn.JsonPath.string_at("$.queryString"),
                query_execution_context=tasks.QueryExecutionContext(
                    database_name="mydatabase"
                ),
                result_configuration=tasks.ResultConfiguration(
                    encryption_configuration=tasks.EncryptionConfiguration(
                        encryption_option=tasks.EncryptionOption.S3_MANAGED
                    ),
                    output_location=s3.Location(
                        bucket_name="amzn-s3-demo-bucket",
                        object_key="folder"
                    )
                ),
                execution_parameters=["param1", "param2"]
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(query_execution_context, dict):
            query_execution_context = QueryExecutionContext(**query_execution_context)
        if isinstance(result_configuration, dict):
            result_configuration = ResultConfiguration(**result_configuration)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d2b6c9868cc3485b4ed4a4ef1f89308086ff873bc4d86413aa4b420c9cf24e55)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument query_string", value=query_string, expected_type=type_hints["query_string"])
            check_type(argname="argument client_request_token", value=client_request_token, expected_type=type_hints["client_request_token"])
            check_type(argname="argument execution_parameters", value=execution_parameters, expected_type=type_hints["execution_parameters"])
            check_type(argname="argument query_execution_context", value=query_execution_context, expected_type=type_hints["query_execution_context"])
            check_type(argname="argument result_configuration", value=result_configuration, expected_type=type_hints["result_configuration"])
            check_type(argname="argument result_reuse_configuration_max_age", value=result_reuse_configuration_max_age, expected_type=type_hints["result_reuse_configuration_max_age"])
            check_type(argname="argument work_group", value=work_group, expected_type=type_hints["work_group"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_string": query_string,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if outputs is not None:
            self._values["outputs"] = outputs
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if client_request_token is not None:
            self._values["client_request_token"] = client_request_token
        if execution_parameters is not None:
            self._values["execution_parameters"] = execution_parameters
        if query_execution_context is not None:
            self._values["query_execution_context"] = query_execution_context
        if result_configuration is not None:
            self._values["result_configuration"] = result_configuration
        if result_reuse_configuration_max_age is not None:
            self._values["result_reuse_configuration_max_age"] = result_reuse_configuration_max_age
        if work_group is not None:
            self._values["work_group"] = work_group

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def query_string(self) -> builtins.str:
        '''Query that will be started.'''
        result = self._values.get("query_string")
        assert result is not None, "Required property 'query_string' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def client_request_token(self) -> typing.Optional[builtins.str]:
        '''Unique string string to ensure idempotence.

        :default: - No client request token
        '''
        result = self._values.get("client_request_token")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def execution_parameters(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of values for the parameters in a query.

        The values are applied sequentially to the parameters in the query in the order
        in which the parameters occur.

        :default: - No parameters
        '''
        result = self._values.get("execution_parameters")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def query_execution_context(self) -> typing.Optional["QueryExecutionContext"]:
        '''Database within which query executes.

        :default: - No query execution context
        '''
        result = self._values.get("query_execution_context")
        return typing.cast(typing.Optional["QueryExecutionContext"], result)

    @builtins.property
    def result_configuration(self) -> typing.Optional["ResultConfiguration"]:
        '''Configuration on how and where to save query.

        :default: - No result configuration
        '''
        result = self._values.get("result_configuration")
        return typing.cast(typing.Optional["ResultConfiguration"], result)

    @builtins.property
    def result_reuse_configuration_max_age(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Specifies, in minutes, the maximum age of a previous query result that Athena should consider for reuse.

        :default: - Query results are not reused
        '''
        result = self._values.get("result_reuse_configuration_max_age")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def work_group(self) -> typing.Optional[builtins.str]:
        '''Configuration on how and where to save query.

        :default: - No work group
        '''
        result = self._values.get("work_group")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaStartQueryExecutionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AthenaStopQueryExecution(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStopQueryExecution",
):
    '''Stop an Athena Query Execution as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :exampleMetadata: infused

    Example::

        stop_query_execution_job = tasks.AthenaStopQueryExecution(self, "Stop Query Execution",
            query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param query_execution_id: Query that will be stopped.
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6d67b0a74228051a1266e72262ec4fc6e739a35f07a82a3e5faf57fbe904ab7b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaStopQueryExecutionProps(
            query_execution_id=query_execution_id,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
    ) -> "AthenaStopQueryExecution":
        '''Stop an Athena Query Execution as a Task using JSONata.

        :param scope: -
        :param id: -
        :param query_execution_id: Query that will be stopped.
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4301b10f0c422a7fd643bac05ef3aaa84c0914acf235153bf430c0edd5e4090a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaStopQueryExecutionJsonataProps(
            query_execution_id=query_execution_id,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            outputs=outputs,
        )

        return typing.cast("AthenaStopQueryExecution", jsii.sinvoke(cls, "jsonata", [scope, id, props]))

    @jsii.member(jsii_name="jsonPath")
    @builtins.classmethod
    def json_path(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> "AthenaStopQueryExecution":
        '''Stop an Athena Query Execution as a Task using JSONPath.

        :param scope: -
        :param id: -
        :param query_execution_id: Query that will be stopped.
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e669d6fd229ae91e23de94ffbff48c29f049c59113fc504beb171a15150bb985)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AthenaStopQueryExecutionJsonPathProps(
            query_execution_id=query_execution_id,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
        )

        return typing.cast("AthenaStopQueryExecution", jsii.sinvoke(cls, "jsonPath", [scope, id, props]))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStopQueryExecutionJsonPathProps",
    jsii_struct_bases=[_TaskStateJsonPathBaseProps_57dc2260],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "query_execution_id": "queryExecutionId",
    },
)
class AthenaStopQueryExecutionJsonPathProps(_TaskStateJsonPathBaseProps_57dc2260):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        query_execution_id: builtins.str,
    ) -> None:
        '''Properties for stopping a Query Execution using JSONPath.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param query_execution_id: Query that will be stopped.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # result_selector: Any
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            athena_stop_query_execution_json_path_props = stepfunctions_tasks.AthenaStopQueryExecutionJsonPathProps(
                query_execution_id="queryExecutionId",
            
                # the properties below are optional
                assign={
                    "assign_key": assign
                },
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output_path="outputPath",
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                result_path="resultPath",
                result_selector={
                    "result_selector_key": result_selector
                },
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__afb8ff79b65767c1002a508c9dbf7ed917e24202bacd663bab7b273278cb1b9b)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be stopped.'''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaStopQueryExecutionJsonPathProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStopQueryExecutionJsonataProps",
    jsii_struct_bases=[_TaskStateJsonataBaseProps_0e91775c],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "outputs": "outputs",
        "query_execution_id": "queryExecutionId",
    },
)
class AthenaStopQueryExecutionJsonataProps(_TaskStateJsonataBaseProps_0e91775c):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
        query_execution_id: builtins.str,
    ) -> None:
        '''Properties for stopping a Query Execution using JSONata.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param query_execution_id: Query that will be stopped.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # outputs: Any
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            athena_stop_query_execution_jsonata_props = stepfunctions_tasks.AthenaStopQueryExecutionJsonataProps(
                query_execution_id="queryExecutionId",
            
                # the properties below are optional
                assign={
                    "assign_key": assign
                },
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                outputs=outputs,
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__788f94ebafc647c960fed7fbdc7761e9d97144152a4c883ebd127b1af1c3209f)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if outputs is not None:
            self._values["outputs"] = outputs

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be stopped.'''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaStopQueryExecutionJsonataProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AthenaStopQueryExecutionProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "outputs": "outputs",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "query_execution_id": "queryExecutionId",
    },
)
class AthenaStopQueryExecutionProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        query_execution_id: builtins.str,
    ) -> None:
        '''Properties for stopping a Query Execution.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param query_execution_id: Query that will be stopped.

        :exampleMetadata: infused

        Example::

            stop_query_execution_job = tasks.AthenaStopQueryExecution(self, "Stop Query Execution",
                query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__95e47c48916d5954411b46d2ffee40460efc3e552562676549f5de7fe2db36ef)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument query_execution_id", value=query_execution_id, expected_type=type_hints["query_execution_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if outputs is not None:
            self._values["outputs"] = outputs
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''Query that will be stopped.'''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaStopQueryExecutionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.AuthType")
class AuthType(enum.Enum):
    '''The authentication method used to call the endpoint.'''

    NO_AUTH = "NO_AUTH"
    '''Call the API direclty with no authorization method.'''
    IAM_ROLE = "IAM_ROLE"
    '''Use the IAM role associated with the current state machine for authorization.'''
    RESOURCE_POLICY = "RESOURCE_POLICY"
    '''Use the resource policy of the API for authorization.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchContainerOverrides",
    jsii_struct_bases=[],
    name_mapping={
        "command": "command",
        "environment": "environment",
        "gpu_count": "gpuCount",
        "instance_type": "instanceType",
        "memory": "memory",
        "vcpus": "vcpus",
    },
)
class BatchContainerOverrides:
    def __init__(
        self,
        *,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        instance_type: typing.Optional[_InstanceType_f64915b9] = None,
        memory: typing.Optional[_Size_7b441c34] = None,
        vcpus: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''The overrides that should be sent to a container.

        :param command: The command to send to the container that overrides the default command from the Docker image or the job definition. Default: - No command overrides
        :param environment: The environment variables to send to the container. You can add new environment variables, which are added to the container at launch, or you can override the existing environment variables from the Docker image or the job definition. Default: - No environment overrides
        :param gpu_count: The number of physical GPUs to reserve for the container. The number of GPUs reserved for all containers in a job should not exceed the number of available GPUs on the compute resource that the job is launched on. Default: - No GPU reservation
        :param instance_type: The instance type to use for a multi-node parallel job. This parameter is not valid for single-node container jobs. Default: - No instance type overrides
        :param memory: Memory reserved for the job. Default: - No memory overrides. The memory supplied in the job definition will be used.
        :param vcpus: The number of vCPUs to reserve for the container. This value overrides the value set in the job definition. Default: - No vCPUs overrides

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # instance_type: ec2.InstanceType
            # size: cdk.Size
            
            batch_container_overrides = stepfunctions_tasks.BatchContainerOverrides(
                command=["command"],
                environment={
                    "environment_key": "environment"
                },
                gpu_count=123,
                instance_type=instance_type,
                memory=size,
                vcpus=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d006b47111b6db7647cd90008f27c393312ee16152c31394c4e4f623d7fcc2ed)
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument gpu_count", value=gpu_count, expected_type=type_hints["gpu_count"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument memory", value=memory, expected_type=type_hints["memory"])
            check_type(argname="argument vcpus", value=vcpus, expected_type=type_hints["vcpus"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if command is not None:
            self._values["command"] = command
        if environment is not None:
            self._values["environment"] = environment
        if gpu_count is not None:
            self._values["gpu_count"] = gpu_count
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if memory is not None:
            self._values["memory"] = memory
        if vcpus is not None:
            self._values["vcpus"] = vcpus

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The command to send to the container that overrides the default command from the Docker image or the job definition.

        :default: - No command overrides
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to send to the container.

        You can add new environment variables, which are added to the container
        at launch, or you can override the existing environment variables from
        the Docker image or the job definition.

        :default: - No environment overrides
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def gpu_count(self) -> typing.Optional[jsii.Number]:
        '''The number of physical GPUs to reserve for the container.

        The number of GPUs reserved for all containers in a job
        should not exceed the number of available GPUs on the compute
        resource that the job is launched on.

        :default: - No GPU reservation
        '''
        result = self._values.get("gpu_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def instance_type(self) -> typing.Optional[_InstanceType_f64915b9]:
        '''The instance type to use for a multi-node parallel job.

        This parameter is not valid for single-node container jobs.

        :default: - No instance type overrides
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional[_InstanceType_f64915b9], result)

    @builtins.property
    def memory(self) -> typing.Optional[_Size_7b441c34]:
        '''Memory reserved for the job.

        :default: - No memory overrides. The memory supplied in the job definition will be used.
        '''
        result = self._values.get("memory")
        return typing.cast(typing.Optional[_Size_7b441c34], result)

    @builtins.property
    def vcpus(self) -> typing.Optional[jsii.Number]:
        '''The number of vCPUs to reserve for the container.

        This value overrides the value set in the job definition.

        :default: - No vCPUs overrides
        '''
        result = self._values.get("vcpus")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BatchContainerOverrides(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchJobDependency",
    jsii_struct_bases=[],
    name_mapping={"job_id": "jobId", "type": "type"},
)
class BatchJobDependency:
    def __init__(
        self,
        *,
        job_id: typing.Optional[builtins.str] = None,
        type: typing.Optional[builtins.str] = None,
    ) -> None:
        '''An object representing an AWS Batch job dependency.

        :param job_id: The job ID of the AWS Batch job associated with this dependency. Default: - No jobId
        :param type: The type of the job dependency. Default: - No type

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            batch_job_dependency = stepfunctions_tasks.BatchJobDependency(
                job_id="jobId",
                type="type"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f18f114b4b57e22d6d5ccd29edff4424be5b1132c504c1b3ac2ecad9d9cab599)
            check_type(argname="argument job_id", value=job_id, expected_type=type_hints["job_id"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if job_id is not None:
            self._values["job_id"] = job_id
        if type is not None:
            self._values["type"] = type

    @builtins.property
    def job_id(self) -> typing.Optional[builtins.str]:
        '''The job ID of the AWS Batch job associated with this dependency.

        :default: - No jobId
        '''
        result = self._values.get("job_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type(self) -> typing.Optional[builtins.str]:
        '''The type of the job dependency.

        :default: - No type
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BatchJobDependency(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchStrategy")
class BatchStrategy(enum.Enum):
    '''Specifies the number of records to include in a mini-batch for an HTTP inference request.'''

    MULTI_RECORD = "MULTI_RECORD"
    '''Fits multiple records in a mini-batch.'''
    SINGLE_RECORD = "SINGLE_RECORD"
    '''Use a single record when making an invocation request.'''


class BatchSubmitJob(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchSubmitJob",
):
    '''Task to submits an AWS Batch job from a job definition.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_batch as batch
        # batch_job_definition: batch.EcsJobDefinition
        # batch_queue: batch.JobQueue
        
        
        task = tasks.BatchSubmitJob(self, "Submit Job",
            job_definition_arn=batch_job_definition.job_definition_arn,
            job_name="MyJob",
            job_queue_arn=batch_queue.job_queue_arn
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        job_definition_arn: builtins.str,
        job_name: builtins.str,
        job_queue_arn: builtins.str,
        array_size: typing.Optional[jsii.Number] = None,
        attempts: typing.Optional[jsii.Number] = None,
        container_overrides: typing.Optional[typing.Union[BatchContainerOverrides, typing.Dict[builtins.str, typing.Any]]] = None,
        depends_on: typing.Optional[typing.Sequence[typing.Union[BatchJobDependency, typing.Dict[builtins.str, typing.Any]]]] = None,
        payload: typing.Optional[_TaskInput_91b91b91] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param job_definition_arn: The arn of the job definition used by this job.
        :param job_name: The name of the job. The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.
        :param job_queue_arn: The arn of the job queue into which the job is submitted.
        :param array_size: The array size can be between 2 and 10,000. If you specify array properties for a job, it becomes an array job. For more information, see Array Jobs in the AWS Batch User Guide. Default: - No array size
        :param attempts: The number of times to move a job to the RUNNABLE status. You may specify between 1 and 10 attempts. If the value of attempts is greater than one, the job is retried on failure the same number of attempts as the value. Default: 1
        :param container_overrides: A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive. Default: - No container overrides
        :param depends_on: A list of dependencies for the job. A job can depend upon a maximum of 20 jobs. Default: - No dependencies
        :param payload: The payload to be passed as parameters to the batch job. Default: - No parameters are passed
        :param tags: The tags applied to the job request. Default: {} - no tags
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe5d08d3f91a8bb221c70885cc736bf4f22481309f41eda6cfe2f18298a12423)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BatchSubmitJobProps(
            job_definition_arn=job_definition_arn,
            job_name=job_name,
            job_queue_arn=job_queue_arn,
            array_size=array_size,
            attempts=attempts,
            container_overrides=container_overrides,
            depends_on=depends_on,
            payload=payload,
            tags=tags,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        job_definition_arn: builtins.str,
        job_name: builtins.str,
        job_queue_arn: builtins.str,
        array_size: typing.Optional[jsii.Number] = None,
        attempts: typing.Optional[jsii.Number] = None,
        container_overrides: typing.Optional[typing.Union[BatchContainerOverrides, typing.Dict[builtins.str, typing.Any]]] = None,
        depends_on: typing.Optional[typing.Sequence[typing.Union[BatchJobDependency, typing.Dict[builtins.str, typing.Any]]]] = None,
        payload: typing.Optional[_TaskInput_91b91b91] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
    ) -> "BatchSubmitJob":
        '''Task to submits an AWS Batch job from a job definition using JSONata.

        :param scope: -
        :param id: -
        :param job_definition_arn: The arn of the job definition used by this job.
        :param job_name: The name of the job. The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.
        :param job_queue_arn: The arn of the job queue into which the job is submitted.
        :param array_size: The array size can be between 2 and 10,000. If you specify array properties for a job, it becomes an array job. For more information, see Array Jobs in the AWS Batch User Guide. Default: - No array size
        :param attempts: The number of times to move a job to the RUNNABLE status. You may specify between 1 and 10 attempts. If the value of attempts is greater than one, the job is retried on failure the same number of attempts as the value. Default: 1
        :param container_overrides: A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive. Default: - No container overrides
        :param depends_on: A list of dependencies for the job. A job can depend upon a maximum of 20 jobs. Default: - No dependencies
        :param payload: The payload to be passed as parameters to the batch job. Default: - No parameters are passed
        :param tags: The tags applied to the job request. Default: {} - no tags
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5fbf5ab082b5d9bc0dca8e80911bf0d3ea894b7d4a10c9c7d2f64b6463a73830)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BatchSubmitJobJsonataProps(
            job_definition_arn=job_definition_arn,
            job_name=job_name,
            job_queue_arn=job_queue_arn,
            array_size=array_size,
            attempts=attempts,
            container_overrides=container_overrides,
            depends_on=depends_on,
            payload=payload,
            tags=tags,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            outputs=outputs,
        )

        return typing.cast("BatchSubmitJob", jsii.sinvoke(cls, "jsonata", [scope, id, props]))

    @jsii.member(jsii_name="jsonPath")
    @builtins.classmethod
    def json_path(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        job_definition_arn: builtins.str,
        job_name: builtins.str,
        job_queue_arn: builtins.str,
        array_size: typing.Optional[jsii.Number] = None,
        attempts: typing.Optional[jsii.Number] = None,
        container_overrides: typing.Optional[typing.Union[BatchContainerOverrides, typing.Dict[builtins.str, typing.Any]]] = None,
        depends_on: typing.Optional[typing.Sequence[typing.Union[BatchJobDependency, typing.Dict[builtins.str, typing.Any]]]] = None,
        payload: typing.Optional[_TaskInput_91b91b91] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> "BatchSubmitJob":
        '''Task to submits an AWS Batch job from a job definition using JSONPath.

        :param scope: -
        :param id: -
        :param job_definition_arn: The arn of the job definition used by this job.
        :param job_name: The name of the job. The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.
        :param job_queue_arn: The arn of the job queue into which the job is submitted.
        :param array_size: The array size can be between 2 and 10,000. If you specify array properties for a job, it becomes an array job. For more information, see Array Jobs in the AWS Batch User Guide. Default: - No array size
        :param attempts: The number of times to move a job to the RUNNABLE status. You may specify between 1 and 10 attempts. If the value of attempts is greater than one, the job is retried on failure the same number of attempts as the value. Default: 1
        :param container_overrides: A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive. Default: - No container overrides
        :param depends_on: A list of dependencies for the job. A job can depend upon a maximum of 20 jobs. Default: - No dependencies
        :param payload: The payload to be passed as parameters to the batch job. Default: - No parameters are passed
        :param tags: The tags applied to the job request. Default: {} - no tags
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d44af0d016142645633b9b099a1fdd734b79a3b58e9d22ea1ede9439be4fc181)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BatchSubmitJobJsonPathProps(
            job_definition_arn=job_definition_arn,
            job_name=job_name,
            job_queue_arn=job_queue_arn,
            array_size=array_size,
            attempts=attempts,
            container_overrides=container_overrides,
            depends_on=depends_on,
            payload=payload,
            tags=tags,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
        )

        return typing.cast("BatchSubmitJob", jsii.sinvoke(cls, "jsonPath", [scope, id, props]))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchSubmitJobJsonPathProps",
    jsii_struct_bases=[_TaskStateJsonPathBaseProps_57dc2260],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "job_definition_arn": "jobDefinitionArn",
        "job_name": "jobName",
        "job_queue_arn": "jobQueueArn",
        "array_size": "arraySize",
        "attempts": "attempts",
        "container_overrides": "containerOverrides",
        "depends_on": "dependsOn",
        "payload": "payload",
        "tags": "tags",
    },
)
class BatchSubmitJobJsonPathProps(_TaskStateJsonPathBaseProps_57dc2260):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        job_definition_arn: builtins.str,
        job_name: builtins.str,
        job_queue_arn: builtins.str,
        array_size: typing.Optional[jsii.Number] = None,
        attempts: typing.Optional[jsii.Number] = None,
        container_overrides: typing.Optional[typing.Union[BatchContainerOverrides, typing.Dict[builtins.str, typing.Any]]] = None,
        depends_on: typing.Optional[typing.Sequence[typing.Union[BatchJobDependency, typing.Dict[builtins.str, typing.Any]]]] = None,
        payload: typing.Optional[_TaskInput_91b91b91] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Properties for BatchSubmitJob using JSONPath.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param job_definition_arn: The arn of the job definition used by this job.
        :param job_name: The name of the job. The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.
        :param job_queue_arn: The arn of the job queue into which the job is submitted.
        :param array_size: The array size can be between 2 and 10,000. If you specify array properties for a job, it becomes an array job. For more information, see Array Jobs in the AWS Batch User Guide. Default: - No array size
        :param attempts: The number of times to move a job to the RUNNABLE status. You may specify between 1 and 10 attempts. If the value of attempts is greater than one, the job is retried on failure the same number of attempts as the value. Default: 1
        :param container_overrides: A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive. Default: - No container overrides
        :param depends_on: A list of dependencies for the job. A job can depend upon a maximum of 20 jobs. Default: - No dependencies
        :param payload: The payload to be passed as parameters to the batch job. Default: - No parameters are passed
        :param tags: The tags applied to the job request. Default: {} - no tags

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # instance_type: ec2.InstanceType
            # result_selector: Any
            # size: cdk.Size
            # task_input: stepfunctions.TaskInput
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            batch_submit_job_json_path_props = stepfunctions_tasks.BatchSubmitJobJsonPathProps(
                job_definition_arn="jobDefinitionArn",
                job_name="jobName",
                job_queue_arn="jobQueueArn",
            
                # the properties below are optional
                array_size=123,
                assign={
                    "assign_key": assign
                },
                attempts=123,
                comment="comment",
                container_overrides=stepfunctions_tasks.BatchContainerOverrides(
                    command=["command"],
                    environment={
                        "environment_key": "environment"
                    },
                    gpu_count=123,
                    instance_type=instance_type,
                    memory=size,
                    vcpus=123
                ),
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                depends_on=[stepfunctions_tasks.BatchJobDependency(
                    job_id="jobId",
                    type="type"
                )],
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output_path="outputPath",
                payload=task_input,
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                result_path="resultPath",
                result_selector={
                    "result_selector_key": result_selector
                },
                state_name="stateName",
                tags={
                    "tags_key": "tags"
                },
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(container_overrides, dict):
            container_overrides = BatchContainerOverrides(**container_overrides)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d004eb893e6aa585787580d15f1a37a773b8bf8a9edd39dcdfad3cdfb1cfa481)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument job_definition_arn", value=job_definition_arn, expected_type=type_hints["job_definition_arn"])
            check_type(argname="argument job_name", value=job_name, expected_type=type_hints["job_name"])
            check_type(argname="argument job_queue_arn", value=job_queue_arn, expected_type=type_hints["job_queue_arn"])
            check_type(argname="argument array_size", value=array_size, expected_type=type_hints["array_size"])
            check_type(argname="argument attempts", value=attempts, expected_type=type_hints["attempts"])
            check_type(argname="argument container_overrides", value=container_overrides, expected_type=type_hints["container_overrides"])
            check_type(argname="argument depends_on", value=depends_on, expected_type=type_hints["depends_on"])
            check_type(argname="argument payload", value=payload, expected_type=type_hints["payload"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "job_definition_arn": job_definition_arn,
            "job_name": job_name,
            "job_queue_arn": job_queue_arn,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if array_size is not None:
            self._values["array_size"] = array_size
        if attempts is not None:
            self._values["attempts"] = attempts
        if container_overrides is not None:
            self._values["container_overrides"] = container_overrides
        if depends_on is not None:
            self._values["depends_on"] = depends_on
        if payload is not None:
            self._values["payload"] = payload
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def job_definition_arn(self) -> builtins.str:
        '''The arn of the job definition used by this job.'''
        result = self._values.get("job_definition_arn")
        assert result is not None, "Required property 'job_definition_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def job_name(self) -> builtins.str:
        '''The name of the job.

        The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase),
        numbers, hyphens, and underscores are allowed.
        '''
        result = self._values.get("job_name")
        assert result is not None, "Required property 'job_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def job_queue_arn(self) -> builtins.str:
        '''The arn of the job queue into which the job is submitted.'''
        result = self._values.get("job_queue_arn")
        assert result is not None, "Required property 'job_queue_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def array_size(self) -> typing.Optional[jsii.Number]:
        '''The array size can be between 2 and 10,000.

        If you specify array properties for a job, it becomes an array job.
        For more information, see Array Jobs in the AWS Batch User Guide.

        :default: - No array size
        '''
        result = self._values.get("array_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def attempts(self) -> typing.Optional[jsii.Number]:
        '''The number of times to move a job to the RUNNABLE status.

        You may specify between 1 and 10 attempts.
        If the value of attempts is greater than one,
        the job is retried on failure the same number of attempts as the value.

        :default: 1
        '''
        result = self._values.get("attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def container_overrides(self) -> typing.Optional[BatchContainerOverrides]:
        '''A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive.

        :default: - No container overrides

        :see: https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-containerOverrides
        '''
        result = self._values.get("container_overrides")
        return typing.cast(typing.Optional[BatchContainerOverrides], result)

    @builtins.property
    def depends_on(self) -> typing.Optional[typing.List[BatchJobDependency]]:
        '''A list of dependencies for the job.

        A job can depend upon a maximum of 20 jobs.

        :default: - No dependencies

        :see: https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-dependsOn
        '''
        result = self._values.get("depends_on")
        return typing.cast(typing.Optional[typing.List[BatchJobDependency]], result)

    @builtins.property
    def payload(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''The payload to be passed as parameters to the batch job.

        :default: - No parameters are passed
        '''
        result = self._values.get("payload")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags applied to the job request.

        :default: {} - no tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BatchSubmitJobJsonPathProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchSubmitJobJsonataProps",
    jsii_struct_bases=[_TaskStateJsonataBaseProps_0e91775c],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "outputs": "outputs",
        "job_definition_arn": "jobDefinitionArn",
        "job_name": "jobName",
        "job_queue_arn": "jobQueueArn",
        "array_size": "arraySize",
        "attempts": "attempts",
        "container_overrides": "containerOverrides",
        "depends_on": "dependsOn",
        "payload": "payload",
        "tags": "tags",
    },
)
class BatchSubmitJobJsonataProps(_TaskStateJsonataBaseProps_0e91775c):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
        job_definition_arn: builtins.str,
        job_name: builtins.str,
        job_queue_arn: builtins.str,
        array_size: typing.Optional[jsii.Number] = None,
        attempts: typing.Optional[jsii.Number] = None,
        container_overrides: typing.Optional[typing.Union[BatchContainerOverrides, typing.Dict[builtins.str, typing.Any]]] = None,
        depends_on: typing.Optional[typing.Sequence[typing.Union[BatchJobDependency, typing.Dict[builtins.str, typing.Any]]]] = None,
        payload: typing.Optional[_TaskInput_91b91b91] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Properties for BatchSubmitJob using JSONata.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param job_definition_arn: The arn of the job definition used by this job.
        :param job_name: The name of the job. The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.
        :param job_queue_arn: The arn of the job queue into which the job is submitted.
        :param array_size: The array size can be between 2 and 10,000. If you specify array properties for a job, it becomes an array job. For more information, see Array Jobs in the AWS Batch User Guide. Default: - No array size
        :param attempts: The number of times to move a job to the RUNNABLE status. You may specify between 1 and 10 attempts. If the value of attempts is greater than one, the job is retried on failure the same number of attempts as the value. Default: 1
        :param container_overrides: A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive. Default: - No container overrides
        :param depends_on: A list of dependencies for the job. A job can depend upon a maximum of 20 jobs. Default: - No dependencies
        :param payload: The payload to be passed as parameters to the batch job. Default: - No parameters are passed
        :param tags: The tags applied to the job request. Default: {} - no tags

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # instance_type: ec2.InstanceType
            # outputs: Any
            # size: cdk.Size
            # task_input: stepfunctions.TaskInput
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            batch_submit_job_jsonata_props = stepfunctions_tasks.BatchSubmitJobJsonataProps(
                job_definition_arn="jobDefinitionArn",
                job_name="jobName",
                job_queue_arn="jobQueueArn",
            
                # the properties below are optional
                array_size=123,
                assign={
                    "assign_key": assign
                },
                attempts=123,
                comment="comment",
                container_overrides=stepfunctions_tasks.BatchContainerOverrides(
                    command=["command"],
                    environment={
                        "environment_key": "environment"
                    },
                    gpu_count=123,
                    instance_type=instance_type,
                    memory=size,
                    vcpus=123
                ),
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                depends_on=[stepfunctions_tasks.BatchJobDependency(
                    job_id="jobId",
                    type="type"
                )],
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                outputs=outputs,
                payload=task_input,
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                state_name="stateName",
                tags={
                    "tags_key": "tags"
                },
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(container_overrides, dict):
            container_overrides = BatchContainerOverrides(**container_overrides)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76545897f23202861bdbcf6e79bb5f460d750c7e3697eb14c11206c9008236dc)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument job_definition_arn", value=job_definition_arn, expected_type=type_hints["job_definition_arn"])
            check_type(argname="argument job_name", value=job_name, expected_type=type_hints["job_name"])
            check_type(argname="argument job_queue_arn", value=job_queue_arn, expected_type=type_hints["job_queue_arn"])
            check_type(argname="argument array_size", value=array_size, expected_type=type_hints["array_size"])
            check_type(argname="argument attempts", value=attempts, expected_type=type_hints["attempts"])
            check_type(argname="argument container_overrides", value=container_overrides, expected_type=type_hints["container_overrides"])
            check_type(argname="argument depends_on", value=depends_on, expected_type=type_hints["depends_on"])
            check_type(argname="argument payload", value=payload, expected_type=type_hints["payload"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "job_definition_arn": job_definition_arn,
            "job_name": job_name,
            "job_queue_arn": job_queue_arn,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if outputs is not None:
            self._values["outputs"] = outputs
        if array_size is not None:
            self._values["array_size"] = array_size
        if attempts is not None:
            self._values["attempts"] = attempts
        if container_overrides is not None:
            self._values["container_overrides"] = container_overrides
        if depends_on is not None:
            self._values["depends_on"] = depends_on
        if payload is not None:
            self._values["payload"] = payload
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def job_definition_arn(self) -> builtins.str:
        '''The arn of the job definition used by this job.'''
        result = self._values.get("job_definition_arn")
        assert result is not None, "Required property 'job_definition_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def job_name(self) -> builtins.str:
        '''The name of the job.

        The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase),
        numbers, hyphens, and underscores are allowed.
        '''
        result = self._values.get("job_name")
        assert result is not None, "Required property 'job_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def job_queue_arn(self) -> builtins.str:
        '''The arn of the job queue into which the job is submitted.'''
        result = self._values.get("job_queue_arn")
        assert result is not None, "Required property 'job_queue_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def array_size(self) -> typing.Optional[jsii.Number]:
        '''The array size can be between 2 and 10,000.

        If you specify array properties for a job, it becomes an array job.
        For more information, see Array Jobs in the AWS Batch User Guide.

        :default: - No array size
        '''
        result = self._values.get("array_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def attempts(self) -> typing.Optional[jsii.Number]:
        '''The number of times to move a job to the RUNNABLE status.

        You may specify between 1 and 10 attempts.
        If the value of attempts is greater than one,
        the job is retried on failure the same number of attempts as the value.

        :default: 1
        '''
        result = self._values.get("attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def container_overrides(self) -> typing.Optional[BatchContainerOverrides]:
        '''A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive.

        :default: - No container overrides

        :see: https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-containerOverrides
        '''
        result = self._values.get("container_overrides")
        return typing.cast(typing.Optional[BatchContainerOverrides], result)

    @builtins.property
    def depends_on(self) -> typing.Optional[typing.List[BatchJobDependency]]:
        '''A list of dependencies for the job.

        A job can depend upon a maximum of 20 jobs.

        :default: - No dependencies

        :see: https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-dependsOn
        '''
        result = self._values.get("depends_on")
        return typing.cast(typing.Optional[typing.List[BatchJobDependency]], result)

    @builtins.property
    def payload(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''The payload to be passed as parameters to the batch job.

        :default: - No parameters are passed
        '''
        result = self._values.get("payload")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags applied to the job request.

        :default: {} - no tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BatchSubmitJobJsonataProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BatchSubmitJobProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "outputs": "outputs",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "job_definition_arn": "jobDefinitionArn",
        "job_name": "jobName",
        "job_queue_arn": "jobQueueArn",
        "array_size": "arraySize",
        "attempts": "attempts",
        "container_overrides": "containerOverrides",
        "depends_on": "dependsOn",
        "payload": "payload",
        "tags": "tags",
    },
)
class BatchSubmitJobProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        job_definition_arn: builtins.str,
        job_name: builtins.str,
        job_queue_arn: builtins.str,
        array_size: typing.Optional[jsii.Number] = None,
        attempts: typing.Optional[jsii.Number] = None,
        container_overrides: typing.Optional[typing.Union[BatchContainerOverrides, typing.Dict[builtins.str, typing.Any]]] = None,
        depends_on: typing.Optional[typing.Sequence[typing.Union[BatchJobDependency, typing.Dict[builtins.str, typing.Any]]]] = None,
        payload: typing.Optional[_TaskInput_91b91b91] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Properties for BatchSubmitJob.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param job_definition_arn: The arn of the job definition used by this job.
        :param job_name: The name of the job. The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.
        :param job_queue_arn: The arn of the job queue into which the job is submitted.
        :param array_size: The array size can be between 2 and 10,000. If you specify array properties for a job, it becomes an array job. For more information, see Array Jobs in the AWS Batch User Guide. Default: - No array size
        :param attempts: The number of times to move a job to the RUNNABLE status. You may specify between 1 and 10 attempts. If the value of attempts is greater than one, the job is retried on failure the same number of attempts as the value. Default: 1
        :param container_overrides: A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive. Default: - No container overrides
        :param depends_on: A list of dependencies for the job. A job can depend upon a maximum of 20 jobs. Default: - No dependencies
        :param payload: The payload to be passed as parameters to the batch job. Default: - No parameters are passed
        :param tags: The tags applied to the job request. Default: {} - no tags

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_batch as batch
            # batch_job_definition: batch.EcsJobDefinition
            # batch_queue: batch.JobQueue
            
            
            task = tasks.BatchSubmitJob(self, "Submit Job",
                job_definition_arn=batch_job_definition.job_definition_arn,
                job_name="MyJob",
                job_queue_arn=batch_queue.job_queue_arn
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(container_overrides, dict):
            container_overrides = BatchContainerOverrides(**container_overrides)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c3197d21dad4ba55e29e09cf770b47b7859167ae6b1ae5ef674cca1bedf86a92)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument job_definition_arn", value=job_definition_arn, expected_type=type_hints["job_definition_arn"])
            check_type(argname="argument job_name", value=job_name, expected_type=type_hints["job_name"])
            check_type(argname="argument job_queue_arn", value=job_queue_arn, expected_type=type_hints["job_queue_arn"])
            check_type(argname="argument array_size", value=array_size, expected_type=type_hints["array_size"])
            check_type(argname="argument attempts", value=attempts, expected_type=type_hints["attempts"])
            check_type(argname="argument container_overrides", value=container_overrides, expected_type=type_hints["container_overrides"])
            check_type(argname="argument depends_on", value=depends_on, expected_type=type_hints["depends_on"])
            check_type(argname="argument payload", value=payload, expected_type=type_hints["payload"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "job_definition_arn": job_definition_arn,
            "job_name": job_name,
            "job_queue_arn": job_queue_arn,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if outputs is not None:
            self._values["outputs"] = outputs
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if array_size is not None:
            self._values["array_size"] = array_size
        if attempts is not None:
            self._values["attempts"] = attempts
        if container_overrides is not None:
            self._values["container_overrides"] = container_overrides
        if depends_on is not None:
            self._values["depends_on"] = depends_on
        if payload is not None:
            self._values["payload"] = payload
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def job_definition_arn(self) -> builtins.str:
        '''The arn of the job definition used by this job.'''
        result = self._values.get("job_definition_arn")
        assert result is not None, "Required property 'job_definition_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def job_name(self) -> builtins.str:
        '''The name of the job.

        The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase),
        numbers, hyphens, and underscores are allowed.
        '''
        result = self._values.get("job_name")
        assert result is not None, "Required property 'job_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def job_queue_arn(self) -> builtins.str:
        '''The arn of the job queue into which the job is submitted.'''
        result = self._values.get("job_queue_arn")
        assert result is not None, "Required property 'job_queue_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def array_size(self) -> typing.Optional[jsii.Number]:
        '''The array size can be between 2 and 10,000.

        If you specify array properties for a job, it becomes an array job.
        For more information, see Array Jobs in the AWS Batch User Guide.

        :default: - No array size
        '''
        result = self._values.get("array_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def attempts(self) -> typing.Optional[jsii.Number]:
        '''The number of times to move a job to the RUNNABLE status.

        You may specify between 1 and 10 attempts.
        If the value of attempts is greater than one,
        the job is retried on failure the same number of attempts as the value.

        :default: 1
        '''
        result = self._values.get("attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def container_overrides(self) -> typing.Optional[BatchContainerOverrides]:
        '''A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive.

        :default: - No container overrides

        :see: https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-containerOverrides
        '''
        result = self._values.get("container_overrides")
        return typing.cast(typing.Optional[BatchContainerOverrides], result)

    @builtins.property
    def depends_on(self) -> typing.Optional[typing.List[BatchJobDependency]]:
        '''A list of dependencies for the job.

        A job can depend upon a maximum of 20 jobs.

        :default: - No dependencies

        :see: https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-dependsOn
        '''
        result = self._values.get("depends_on")
        return typing.cast(typing.Optional[typing.List[BatchJobDependency]], result)

    @builtins.property
    def payload(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''The payload to be passed as parameters to the batch job.

        :default: - No parameters are passed
        '''
        result = self._values.get("payload")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags applied to the job request.

        :default: {} - no tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BatchSubmitJobProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class BedrockInvokeModel(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BedrockInvokeModel",
):
    '''A Step Functions Task to invoke a model in Bedrock.

    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_bedrock as bedrock
        
        
        model = bedrock.FoundationModel.from_foundation_model_id(self, "Model", bedrock.FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1)
        
        task = tasks.BedrockInvokeModel(self, "Prompt Model",
            model=model,
            body=sfn.TaskInput.from_object({
                "input_text": "Generate a list of five first names.",
                "text_generation_config": {
                    "max_token_count": 100,
                    "temperature": 1
                }
            }),
            result_selector={
                "names": sfn.JsonPath.string_at("$.Body.results[0].outputText")
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        model: _IModel_b88b328a,
        accept: typing.Optional[builtins.str] = None,
        body: typing.Optional[_TaskInput_91b91b91] = None,
        content_type: typing.Optional[builtins.str] = None,
        guardrail: typing.Optional["Guardrail"] = None,
        input: typing.Optional[typing.Union["BedrockInvokeModelInputProps", typing.Dict[builtins.str, typing.Any]]] = None,
        output: typing.Optional[typing.Union["BedrockInvokeModelOutputProps", typing.Dict[builtins.str, typing.Any]]] = None,
        trace_enabled: typing.Optional[builtins.bool] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param model: The Bedrock model that the task will invoke.
        :param accept: The desired MIME type of the inference body in the response. Default: 'application/json'
        :param body: The input data for the Bedrock model invocation. The inference parameters contained in the body depend on the Bedrock model being used. The body must be in the format specified in the ``contentType`` field. For example, if the content type is ``application/json``, the body must be JSON formatted. The body must be up to 256 KB in size. For input data that exceeds 256 KB, use ``input`` instead to retrieve the input data from S3. You must specify either the ``body`` or the ``input`` field, but not both. Default: - Input data is retrieved from the location specified in the ``input`` field
        :param content_type: (deprecated) The MIME type of the input data in the request. Default: 'application/json'
        :param guardrail: The guardrail is applied to the invocation. Default: - No guardrail is applied to the invocation.
        :param input: The source location to retrieve the input data from. Default: - Input data is retrieved from the ``body`` field
        :param output: The destination location where the API response is written. If you specify this field, the API response body is replaced with a reference to the output location. Default: - The API response body is returned in the result.
        :param trace_enabled: Specifies whether to enable or disable the Bedrock trace. Default: - Trace is not enabled for the invocation.
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3b20c515efa874adacf78d373dbda10fc59e519a3bd1b280ecbf56409aea2544)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BedrockInvokeModelProps(
            model=model,
            accept=accept,
            body=body,
            content_type=content_type,
            guardrail=guardrail,
            input=input,
            output=output,
            trace_enabled=trace_enabled,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        model: _IModel_b88b328a,
        accept: typing.Optional[builtins.str] = None,
        body: typing.Optional[_TaskInput_91b91b91] = None,
        content_type: typing.Optional[builtins.str] = None,
        guardrail: typing.Optional["Guardrail"] = None,
        input: typing.Optional[typing.Union["BedrockInvokeModelInputProps", typing.Dict[builtins.str, typing.Any]]] = None,
        output: typing.Optional[typing.Union["BedrockInvokeModelOutputProps", typing.Dict[builtins.str, typing.Any]]] = None,
        trace_enabled: typing.Optional[builtins.bool] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
    ) -> "BedrockInvokeModel":
        '''A Step Functions Task using JSONata to invoke a model in Bedrock.

        :param scope: -
        :param id: -
        :param model: The Bedrock model that the task will invoke.
        :param accept: The desired MIME type of the inference body in the response. Default: 'application/json'
        :param body: The input data for the Bedrock model invocation. The inference parameters contained in the body depend on the Bedrock model being used. The body must be in the format specified in the ``contentType`` field. For example, if the content type is ``application/json``, the body must be JSON formatted. The body must be up to 256 KB in size. For input data that exceeds 256 KB, use ``input`` instead to retrieve the input data from S3. You must specify either the ``body`` or the ``input`` field, but not both. Default: - Input data is retrieved from the location specified in the ``input`` field
        :param content_type: (deprecated) The MIME type of the input data in the request. Default: 'application/json'
        :param guardrail: The guardrail is applied to the invocation. Default: - No guardrail is applied to the invocation.
        :param input: The source location to retrieve the input data from. Default: - Input data is retrieved from the ``body`` field
        :param output: The destination location where the API response is written. If you specify this field, the API response body is replaced with a reference to the output location. Default: - The API response body is returned in the result.
        :param trace_enabled: Specifies whether to enable or disable the Bedrock trace. Default: - Trace is not enabled for the invocation.
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a36fd9a6583c2a2dc951e6de166cc97cb7fb3c20f03eb93e29f20e9fcd30e935)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BedrockInvokeModelJsonataProps(
            model=model,
            accept=accept,
            body=body,
            content_type=content_type,
            guardrail=guardrail,
            input=input,
            output=output,
            trace_enabled=trace_enabled,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            outputs=outputs,
        )

        return typing.cast("BedrockInvokeModel", jsii.sinvoke(cls, "jsonata", [scope, id, props]))

    @jsii.member(jsii_name="jsonPath")
    @builtins.classmethod
    def json_path(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        model: _IModel_b88b328a,
        accept: typing.Optional[builtins.str] = None,
        body: typing.Optional[_TaskInput_91b91b91] = None,
        content_type: typing.Optional[builtins.str] = None,
        guardrail: typing.Optional["Guardrail"] = None,
        input: typing.Optional[typing.Union["BedrockInvokeModelInputProps", typing.Dict[builtins.str, typing.Any]]] = None,
        output: typing.Optional[typing.Union["BedrockInvokeModelOutputProps", typing.Dict[builtins.str, typing.Any]]] = None,
        trace_enabled: typing.Optional[builtins.bool] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> "BedrockInvokeModel":
        '''A Step Functions Task using JSONPath to invoke a model in Bedrock.

        :param scope: -
        :param id: -
        :param model: The Bedrock model that the task will invoke.
        :param accept: The desired MIME type of the inference body in the response. Default: 'application/json'
        :param body: The input data for the Bedrock model invocation. The inference parameters contained in the body depend on the Bedrock model being used. The body must be in the format specified in the ``contentType`` field. For example, if the content type is ``application/json``, the body must be JSON formatted. The body must be up to 256 KB in size. For input data that exceeds 256 KB, use ``input`` instead to retrieve the input data from S3. You must specify either the ``body`` or the ``input`` field, but not both. Default: - Input data is retrieved from the location specified in the ``input`` field
        :param content_type: (deprecated) The MIME type of the input data in the request. Default: 'application/json'
        :param guardrail: The guardrail is applied to the invocation. Default: - No guardrail is applied to the invocation.
        :param input: The source location to retrieve the input data from. Default: - Input data is retrieved from the ``body`` field
        :param output: The destination location where the API response is written. If you specify this field, the API response body is replaced with a reference to the output location. Default: - The API response body is returned in the result.
        :param trace_enabled: Specifies whether to enable or disable the Bedrock trace. Default: - Trace is not enabled for the invocation.
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6dc72155af8219156eb49b8fb7d96443c8757489eb43ee1d5e4d8de71c3df899)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BedrockInvokeModelJsonPathProps(
            model=model,
            accept=accept,
            body=body,
            content_type=content_type,
            guardrail=guardrail,
            input=input,
            output=output,
            trace_enabled=trace_enabled,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
        )

        return typing.cast("BedrockInvokeModel", jsii.sinvoke(cls, "jsonPath", [scope, id, props]))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BedrockInvokeModelInputProps",
    jsii_struct_bases=[],
    name_mapping={"s3_input_uri": "s3InputUri", "s3_location": "s3Location"},
)
class BedrockInvokeModelInputProps:
    def __init__(
        self,
        *,
        s3_input_uri: typing.Optional[builtins.str] = None,
        s3_location: typing.Optional[typing.Union[_Location_0948fa7f, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Location to retrieve the input data, prior to calling Bedrock InvokeModel.

        :param s3_input_uri: The source location where the API response is written. This field can be used to specify s3 URI in the form of token Default: - The API response body is returned in the result.
        :param s3_location: S3 object to retrieve the input data from. If the S3 location is not set, then the Body must be set. Default: - Input data is retrieved from the ``body`` field

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-bedrock.html
        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_bedrock as bedrock
            
            
            model = bedrock.FoundationModel.from_foundation_model_id(self, "Model", bedrock.FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1)
            
            task = tasks.BedrockInvokeModel(self, "Prompt Model",
                model=model,
                input=tasks.BedrockInvokeModelInputProps(s3_input_uri=sfn.JsonPath.string_at("$.prompt")),
                output=tasks.BedrockInvokeModelOutputProps(s3_output_uri=sfn.JsonPath.string_at("$.prompt"))
            )
        '''
        if isinstance(s3_location, dict):
            s3_location = _Location_0948fa7f(**s3_location)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5832c6287635fafcd3211864b6568285ae981df509ea5c6b8dd8458d0254c232)
            check_type(argname="argument s3_input_uri", value=s3_input_uri, expected_type=type_hints["s3_input_uri"])
            check_type(argname="argument s3_location", value=s3_location, expected_type=type_hints["s3_location"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if s3_input_uri is not None:
            self._values["s3_input_uri"] = s3_input_uri
        if s3_location is not None:
            self._values["s3_location"] = s3_location

    @builtins.property
    def s3_input_uri(self) -> typing.Optional[builtins.str]:
        '''The source location where the API response is written.

        This field can be used to specify s3 URI in the form of token

        :default: - The API response body is returned in the result.
        '''
        result = self._values.get("s3_input_uri")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def s3_location(self) -> typing.Optional[_Location_0948fa7f]:
        '''S3 object to retrieve the input data from.

        If the S3 location is not set, then the Body must be set.

        :default: - Input data is retrieved from the ``body`` field
        '''
        result = self._values.get("s3_location")
        return typing.cast(typing.Optional[_Location_0948fa7f], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BedrockInvokeModelInputProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BedrockInvokeModelJsonPathProps",
    jsii_struct_bases=[_TaskStateJsonPathBaseProps_57dc2260],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "model": "model",
        "accept": "accept",
        "body": "body",
        "content_type": "contentType",
        "guardrail": "guardrail",
        "input": "input",
        "output": "output",
        "trace_enabled": "traceEnabled",
    },
)
class BedrockInvokeModelJsonPathProps(_TaskStateJsonPathBaseProps_57dc2260):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        model: _IModel_b88b328a,
        accept: typing.Optional[builtins.str] = None,
        body: typing.Optional[_TaskInput_91b91b91] = None,
        content_type: typing.Optional[builtins.str] = None,
        guardrail: typing.Optional["Guardrail"] = None,
        input: typing.Optional[typing.Union[BedrockInvokeModelInputProps, typing.Dict[builtins.str, typing.Any]]] = None,
        output: typing.Optional[typing.Union["BedrockInvokeModelOutputProps", typing.Dict[builtins.str, typing.Any]]] = None,
        trace_enabled: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Properties for invoking a Bedrock Model.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param model: The Bedrock model that the task will invoke.
        :param accept: The desired MIME type of the inference body in the response. Default: 'application/json'
        :param body: The input data for the Bedrock model invocation. The inference parameters contained in the body depend on the Bedrock model being used. The body must be in the format specified in the ``contentType`` field. For example, if the content type is ``application/json``, the body must be JSON formatted. The body must be up to 256 KB in size. For input data that exceeds 256 KB, use ``input`` instead to retrieve the input data from S3. You must specify either the ``body`` or the ``input`` field, but not both. Default: - Input data is retrieved from the location specified in the ``input`` field
        :param content_type: (deprecated) The MIME type of the input data in the request. Default: 'application/json'
        :param guardrail: The guardrail is applied to the invocation. Default: - No guardrail is applied to the invocation.
        :param input: The source location to retrieve the input data from. Default: - Input data is retrieved from the ``body`` field
        :param output: The destination location where the API response is written. If you specify this field, the API response body is replaced with a reference to the output location. Default: - The API response body is returned in the result.
        :param trace_enabled: Specifies whether to enable or disable the Bedrock trace. Default: - Trace is not enabled for the invocation.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_bedrock as bedrock
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # guardrail: stepfunctions_tasks.Guardrail
            # model: bedrock.IModel
            # result_selector: Any
            # task_input: stepfunctions.TaskInput
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            bedrock_invoke_model_json_path_props = stepfunctions_tasks.BedrockInvokeModelJsonPathProps(
                model=model,
            
                # the properties below are optional
                accept="accept",
                assign={
                    "assign_key": assign
                },
                body=task_input,
                comment="comment",
                content_type="contentType",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                guardrail=guardrail,
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                input=stepfunctions_tasks.BedrockInvokeModelInputProps(
                    s3_input_uri="s3InputUri",
                    s3_location=cdk.aws_s3.Location(
                        bucket_name="bucketName",
                        object_key="objectKey",
            
                        # the properties below are optional
                        object_version="objectVersion"
                    )
                ),
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output=stepfunctions_tasks.BedrockInvokeModelOutputProps(
                    s3_location=cdk.aws_s3.Location(
                        bucket_name="bucketName",
                        object_key="objectKey",
            
                        # the properties below are optional
                        object_version="objectVersion"
                    ),
                    s3_output_uri="s3OutputUri"
                ),
                output_path="outputPath",
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                result_path="resultPath",
                result_selector={
                    "result_selector_key": result_selector
                },
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30),
                trace_enabled=False
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(input, dict):
            input = BedrockInvokeModelInputProps(**input)
        if isinstance(output, dict):
            output = BedrockInvokeModelOutputProps(**output)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__87e326b14299aea3fcd18915f91ceeb40b9507bd0c92d0594cc269b8d548ed7f)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument model", value=model, expected_type=type_hints["model"])
            check_type(argname="argument accept", value=accept, expected_type=type_hints["accept"])
            check_type(argname="argument body", value=body, expected_type=type_hints["body"])
            check_type(argname="argument content_type", value=content_type, expected_type=type_hints["content_type"])
            check_type(argname="argument guardrail", value=guardrail, expected_type=type_hints["guardrail"])
            check_type(argname="argument input", value=input, expected_type=type_hints["input"])
            check_type(argname="argument output", value=output, expected_type=type_hints["output"])
            check_type(argname="argument trace_enabled", value=trace_enabled, expected_type=type_hints["trace_enabled"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "model": model,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if accept is not None:
            self._values["accept"] = accept
        if body is not None:
            self._values["body"] = body
        if content_type is not None:
            self._values["content_type"] = content_type
        if guardrail is not None:
            self._values["guardrail"] = guardrail
        if input is not None:
            self._values["input"] = input
        if output is not None:
            self._values["output"] = output
        if trace_enabled is not None:
            self._values["trace_enabled"] = trace_enabled

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def model(self) -> _IModel_b88b328a:
        '''The Bedrock model that the task will invoke.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/api-methods-run.html
        '''
        result = self._values.get("model")
        assert result is not None, "Required property 'model' is missing"
        return typing.cast(_IModel_b88b328a, result)

    @builtins.property
    def accept(self) -> typing.Optional[builtins.str]:
        '''The desired MIME type of the inference body in the response.

        :default: 'application/json'

        :see: https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html
        '''
        result = self._values.get("accept")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''The input data for the Bedrock model invocation.

        The inference parameters contained in the body depend on the Bedrock model being used.

        The body must be in the format specified in the ``contentType`` field.
        For example, if the content type is ``application/json``, the body must be
        JSON formatted.

        The body must be up to 256 KB in size. For input data that exceeds 256 KB,
        use ``input`` instead to retrieve the input data from S3.

        You must specify either the ``body`` or the ``input`` field, but not both.

        :default: - Input data is retrieved from the location specified in the ``input`` field

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html
        '''
        result = self._values.get("body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def content_type(self) -> typing.Optional[builtins.str]:
        '''(deprecated) The MIME type of the input data in the request.

        :default: 'application/json'

        :deprecated: This property does not require configuration because the only acceptable value is 'application/json'.

        :see: https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html
        :stability: deprecated
        '''
        result = self._values.get("content_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def guardrail(self) -> typing.Optional["Guardrail"]:
        '''The guardrail is applied to the invocation.

        :default: - No guardrail is applied to the invocation.
        '''
        result = self._values.get("guardrail")
        return typing.cast(typing.Optional["Guardrail"], result)

    @builtins.property
    def input(self) -> typing.Optional[BedrockInvokeModelInputProps]:
        '''The source location to retrieve the input data from.

        :default: - Input data is retrieved from the ``body`` field
        '''
        result = self._values.get("input")
        return typing.cast(typing.Optional[BedrockInvokeModelInputProps], result)

    @builtins.property
    def output(self) -> typing.Optional["BedrockInvokeModelOutputProps"]:
        '''The destination location where the API response is written.

        If you specify this field, the API response body is replaced with a reference to the
        output location.

        :default: - The API response body is returned in the result.
        '''
        result = self._values.get("output")
        return typing.cast(typing.Optional["BedrockInvokeModelOutputProps"], result)

    @builtins.property
    def trace_enabled(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether to enable or disable the Bedrock trace.

        :default: - Trace is not enabled for the invocation.
        '''
        result = self._values.get("trace_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BedrockInvokeModelJsonPathProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BedrockInvokeModelJsonataProps",
    jsii_struct_bases=[_TaskStateJsonataBaseProps_0e91775c],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "outputs": "outputs",
        "model": "model",
        "accept": "accept",
        "body": "body",
        "content_type": "contentType",
        "guardrail": "guardrail",
        "input": "input",
        "output": "output",
        "trace_enabled": "traceEnabled",
    },
)
class BedrockInvokeModelJsonataProps(_TaskStateJsonataBaseProps_0e91775c):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
        model: _IModel_b88b328a,
        accept: typing.Optional[builtins.str] = None,
        body: typing.Optional[_TaskInput_91b91b91] = None,
        content_type: typing.Optional[builtins.str] = None,
        guardrail: typing.Optional["Guardrail"] = None,
        input: typing.Optional[typing.Union[BedrockInvokeModelInputProps, typing.Dict[builtins.str, typing.Any]]] = None,
        output: typing.Optional[typing.Union["BedrockInvokeModelOutputProps", typing.Dict[builtins.str, typing.Any]]] = None,
        trace_enabled: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Properties for invoking a Bedrock Model.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param model: The Bedrock model that the task will invoke.
        :param accept: The desired MIME type of the inference body in the response. Default: 'application/json'
        :param body: The input data for the Bedrock model invocation. The inference parameters contained in the body depend on the Bedrock model being used. The body must be in the format specified in the ``contentType`` field. For example, if the content type is ``application/json``, the body must be JSON formatted. The body must be up to 256 KB in size. For input data that exceeds 256 KB, use ``input`` instead to retrieve the input data from S3. You must specify either the ``body`` or the ``input`` field, but not both. Default: - Input data is retrieved from the location specified in the ``input`` field
        :param content_type: (deprecated) The MIME type of the input data in the request. Default: 'application/json'
        :param guardrail: The guardrail is applied to the invocation. Default: - No guardrail is applied to the invocation.
        :param input: The source location to retrieve the input data from. Default: - Input data is retrieved from the ``body`` field
        :param output: The destination location where the API response is written. If you specify this field, the API response body is replaced with a reference to the output location. Default: - The API response body is returned in the result.
        :param trace_enabled: Specifies whether to enable or disable the Bedrock trace. Default: - Trace is not enabled for the invocation.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_bedrock as bedrock
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # guardrail: stepfunctions_tasks.Guardrail
            # model: bedrock.IModel
            # outputs: Any
            # task_input: stepfunctions.TaskInput
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            bedrock_invoke_model_jsonata_props = stepfunctions_tasks.BedrockInvokeModelJsonataProps(
                model=model,
            
                # the properties below are optional
                accept="accept",
                assign={
                    "assign_key": assign
                },
                body=task_input,
                comment="comment",
                content_type="contentType",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                guardrail=guardrail,
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                input=stepfunctions_tasks.BedrockInvokeModelInputProps(
                    s3_input_uri="s3InputUri",
                    s3_location=cdk.aws_s3.Location(
                        bucket_name="bucketName",
                        object_key="objectKey",
            
                        # the properties below are optional
                        object_version="objectVersion"
                    )
                ),
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output=stepfunctions_tasks.BedrockInvokeModelOutputProps(
                    s3_location=cdk.aws_s3.Location(
                        bucket_name="bucketName",
                        object_key="objectKey",
            
                        # the properties below are optional
                        object_version="objectVersion"
                    ),
                    s3_output_uri="s3OutputUri"
                ),
                outputs=outputs,
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30),
                trace_enabled=False
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(input, dict):
            input = BedrockInvokeModelInputProps(**input)
        if isinstance(output, dict):
            output = BedrockInvokeModelOutputProps(**output)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d371beb47e5e85e66d96a67d5e03ae1b9264f265b59b2043542b09096d4eb0ce)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument model", value=model, expected_type=type_hints["model"])
            check_type(argname="argument accept", value=accept, expected_type=type_hints["accept"])
            check_type(argname="argument body", value=body, expected_type=type_hints["body"])
            check_type(argname="argument content_type", value=content_type, expected_type=type_hints["content_type"])
            check_type(argname="argument guardrail", value=guardrail, expected_type=type_hints["guardrail"])
            check_type(argname="argument input", value=input, expected_type=type_hints["input"])
            check_type(argname="argument output", value=output, expected_type=type_hints["output"])
            check_type(argname="argument trace_enabled", value=trace_enabled, expected_type=type_hints["trace_enabled"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "model": model,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if outputs is not None:
            self._values["outputs"] = outputs
        if accept is not None:
            self._values["accept"] = accept
        if body is not None:
            self._values["body"] = body
        if content_type is not None:
            self._values["content_type"] = content_type
        if guardrail is not None:
            self._values["guardrail"] = guardrail
        if input is not None:
            self._values["input"] = input
        if output is not None:
            self._values["output"] = output
        if trace_enabled is not None:
            self._values["trace_enabled"] = trace_enabled

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def model(self) -> _IModel_b88b328a:
        '''The Bedrock model that the task will invoke.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/api-methods-run.html
        '''
        result = self._values.get("model")
        assert result is not None, "Required property 'model' is missing"
        return typing.cast(_IModel_b88b328a, result)

    @builtins.property
    def accept(self) -> typing.Optional[builtins.str]:
        '''The desired MIME type of the inference body in the response.

        :default: 'application/json'

        :see: https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html
        '''
        result = self._values.get("accept")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''The input data for the Bedrock model invocation.

        The inference parameters contained in the body depend on the Bedrock model being used.

        The body must be in the format specified in the ``contentType`` field.
        For example, if the content type is ``application/json``, the body must be
        JSON formatted.

        The body must be up to 256 KB in size. For input data that exceeds 256 KB,
        use ``input`` instead to retrieve the input data from S3.

        You must specify either the ``body`` or the ``input`` field, but not both.

        :default: - Input data is retrieved from the location specified in the ``input`` field

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html
        '''
        result = self._values.get("body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def content_type(self) -> typing.Optional[builtins.str]:
        '''(deprecated) The MIME type of the input data in the request.

        :default: 'application/json'

        :deprecated: This property does not require configuration because the only acceptable value is 'application/json'.

        :see: https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html
        :stability: deprecated
        '''
        result = self._values.get("content_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def guardrail(self) -> typing.Optional["Guardrail"]:
        '''The guardrail is applied to the invocation.

        :default: - No guardrail is applied to the invocation.
        '''
        result = self._values.get("guardrail")
        return typing.cast(typing.Optional["Guardrail"], result)

    @builtins.property
    def input(self) -> typing.Optional[BedrockInvokeModelInputProps]:
        '''The source location to retrieve the input data from.

        :default: - Input data is retrieved from the ``body`` field
        '''
        result = self._values.get("input")
        return typing.cast(typing.Optional[BedrockInvokeModelInputProps], result)

    @builtins.property
    def output(self) -> typing.Optional["BedrockInvokeModelOutputProps"]:
        '''The destination location where the API response is written.

        If you specify this field, the API response body is replaced with a reference to the
        output location.

        :default: - The API response body is returned in the result.
        '''
        result = self._values.get("output")
        return typing.cast(typing.Optional["BedrockInvokeModelOutputProps"], result)

    @builtins.property
    def trace_enabled(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether to enable or disable the Bedrock trace.

        :default: - Trace is not enabled for the invocation.
        '''
        result = self._values.get("trace_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BedrockInvokeModelJsonataProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BedrockInvokeModelOutputProps",
    jsii_struct_bases=[],
    name_mapping={"s3_location": "s3Location", "s3_output_uri": "s3OutputUri"},
)
class BedrockInvokeModelOutputProps:
    def __init__(
        self,
        *,
        s3_location: typing.Optional[typing.Union[_Location_0948fa7f, typing.Dict[builtins.str, typing.Any]]] = None,
        s3_output_uri: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Location where the Bedrock InvokeModel API response is written.

        :param s3_location: S3 object where the Bedrock InvokeModel API response is written. If you specify this field, the API response body is replaced with a reference to the Amazon S3 location of the original output. Default: - Response body is returned in the task result
        :param s3_output_uri: The destination location where the API response is written. This field can be used to specify s3 URI in the form of token Default: - The API response body is returned in the result.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-bedrock.html
        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_bedrock as bedrock
            
            
            model = bedrock.FoundationModel.from_foundation_model_id(self, "Model", bedrock.FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1)
            
            task = tasks.BedrockInvokeModel(self, "Prompt Model",
                model=model,
                input=tasks.BedrockInvokeModelInputProps(s3_input_uri=sfn.JsonPath.string_at("$.prompt")),
                output=tasks.BedrockInvokeModelOutputProps(s3_output_uri=sfn.JsonPath.string_at("$.prompt"))
            )
        '''
        if isinstance(s3_location, dict):
            s3_location = _Location_0948fa7f(**s3_location)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1ae5c9f4810ca6d7749a0e5f6df435956d3660251fb9427f7b8aea21bc5a7fb1)
            check_type(argname="argument s3_location", value=s3_location, expected_type=type_hints["s3_location"])
            check_type(argname="argument s3_output_uri", value=s3_output_uri, expected_type=type_hints["s3_output_uri"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if s3_location is not None:
            self._values["s3_location"] = s3_location
        if s3_output_uri is not None:
            self._values["s3_output_uri"] = s3_output_uri

    @builtins.property
    def s3_location(self) -> typing.Optional[_Location_0948fa7f]:
        '''S3 object where the Bedrock InvokeModel API response is written.

        If you specify this field, the API response body is replaced with
        a reference to the Amazon S3 location of the original output.

        :default: - Response body is returned in the task result
        '''
        result = self._values.get("s3_location")
        return typing.cast(typing.Optional[_Location_0948fa7f], result)

    @builtins.property
    def s3_output_uri(self) -> typing.Optional[builtins.str]:
        '''The destination location where the API response is written.

        This field can be used to specify s3 URI in the form of token

        :default: - The API response body is returned in the result.
        '''
        result = self._values.get("s3_output_uri")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BedrockInvokeModelOutputProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.BedrockInvokeModelProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "outputs": "outputs",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "model": "model",
        "accept": "accept",
        "body": "body",
        "content_type": "contentType",
        "guardrail": "guardrail",
        "input": "input",
        "output": "output",
        "trace_enabled": "traceEnabled",
    },
)
class BedrockInvokeModelProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        model: _IModel_b88b328a,
        accept: typing.Optional[builtins.str] = None,
        body: typing.Optional[_TaskInput_91b91b91] = None,
        content_type: typing.Optional[builtins.str] = None,
        guardrail: typing.Optional["Guardrail"] = None,
        input: typing.Optional[typing.Union[BedrockInvokeModelInputProps, typing.Dict[builtins.str, typing.Any]]] = None,
        output: typing.Optional[typing.Union[BedrockInvokeModelOutputProps, typing.Dict[builtins.str, typing.Any]]] = None,
        trace_enabled: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Properties for invoking a Bedrock Model.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param model: The Bedrock model that the task will invoke.
        :param accept: The desired MIME type of the inference body in the response. Default: 'application/json'
        :param body: The input data for the Bedrock model invocation. The inference parameters contained in the body depend on the Bedrock model being used. The body must be in the format specified in the ``contentType`` field. For example, if the content type is ``application/json``, the body must be JSON formatted. The body must be up to 256 KB in size. For input data that exceeds 256 KB, use ``input`` instead to retrieve the input data from S3. You must specify either the ``body`` or the ``input`` field, but not both. Default: - Input data is retrieved from the location specified in the ``input`` field
        :param content_type: (deprecated) The MIME type of the input data in the request. Default: 'application/json'
        :param guardrail: The guardrail is applied to the invocation. Default: - No guardrail is applied to the invocation.
        :param input: The source location to retrieve the input data from. Default: - Input data is retrieved from the ``body`` field
        :param output: The destination location where the API response is written. If you specify this field, the API response body is replaced with a reference to the output location. Default: - The API response body is returned in the result.
        :param trace_enabled: Specifies whether to enable or disable the Bedrock trace. Default: - Trace is not enabled for the invocation.

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_bedrock as bedrock
            
            
            model = bedrock.FoundationModel.from_foundation_model_id(self, "Model", bedrock.FoundationModelIdentifier.AMAZON_TITAN_TEXT_G1_EXPRESS_V1)
            
            task = tasks.BedrockInvokeModel(self, "Prompt Model",
                model=model,
                body=sfn.TaskInput.from_object({
                    "input_text": "Generate a list of five first names.",
                    "text_generation_config": {
                        "max_token_count": 100,
                        "temperature": 1
                    }
                }),
                result_selector={
                    "names": sfn.JsonPath.string_at("$.Body.results[0].outputText")
                }
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if isinstance(input, dict):
            input = BedrockInvokeModelInputProps(**input)
        if isinstance(output, dict):
            output = BedrockInvokeModelOutputProps(**output)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a9bf54cbc3850dd1a65ada3b96e97b5e68b7027badd90592dbcc79a35d9b216e)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument model", value=model, expected_type=type_hints["model"])
            check_type(argname="argument accept", value=accept, expected_type=type_hints["accept"])
            check_type(argname="argument body", value=body, expected_type=type_hints["body"])
            check_type(argname="argument content_type", value=content_type, expected_type=type_hints["content_type"])
            check_type(argname="argument guardrail", value=guardrail, expected_type=type_hints["guardrail"])
            check_type(argname="argument input", value=input, expected_type=type_hints["input"])
            check_type(argname="argument output", value=output, expected_type=type_hints["output"])
            check_type(argname="argument trace_enabled", value=trace_enabled, expected_type=type_hints["trace_enabled"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "model": model,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if outputs is not None:
            self._values["outputs"] = outputs
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if accept is not None:
            self._values["accept"] = accept
        if body is not None:
            self._values["body"] = body
        if content_type is not None:
            self._values["content_type"] = content_type
        if guardrail is not None:
            self._values["guardrail"] = guardrail
        if input is not None:
            self._values["input"] = input
        if output is not None:
            self._values["output"] = output
        if trace_enabled is not None:
            self._values["trace_enabled"] = trace_enabled

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def model(self) -> _IModel_b88b328a:
        '''The Bedrock model that the task will invoke.

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/api-methods-run.html
        '''
        result = self._values.get("model")
        assert result is not None, "Required property 'model' is missing"
        return typing.cast(_IModel_b88b328a, result)

    @builtins.property
    def accept(self) -> typing.Optional[builtins.str]:
        '''The desired MIME type of the inference body in the response.

        :default: 'application/json'

        :see: https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html
        '''
        result = self._values.get("accept")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''The input data for the Bedrock model invocation.

        The inference parameters contained in the body depend on the Bedrock model being used.

        The body must be in the format specified in the ``contentType`` field.
        For example, if the content type is ``application/json``, the body must be
        JSON formatted.

        The body must be up to 256 KB in size. For input data that exceeds 256 KB,
        use ``input`` instead to retrieve the input data from S3.

        You must specify either the ``body`` or the ``input`` field, but not both.

        :default: - Input data is retrieved from the location specified in the ``input`` field

        :see: https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html
        '''
        result = self._values.get("body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def content_type(self) -> typing.Optional[builtins.str]:
        '''(deprecated) The MIME type of the input data in the request.

        :default: 'application/json'

        :deprecated: This property does not require configuration because the only acceptable value is 'application/json'.

        :see: https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_InvokeModel.html
        :stability: deprecated
        '''
        result = self._values.get("content_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def guardrail(self) -> typing.Optional["Guardrail"]:
        '''The guardrail is applied to the invocation.

        :default: - No guardrail is applied to the invocation.
        '''
        result = self._values.get("guardrail")
        return typing.cast(typing.Optional["Guardrail"], result)

    @builtins.property
    def input(self) -> typing.Optional[BedrockInvokeModelInputProps]:
        '''The source location to retrieve the input data from.

        :default: - Input data is retrieved from the ``body`` field
        '''
        result = self._values.get("input")
        return typing.cast(typing.Optional[BedrockInvokeModelInputProps], result)

    @builtins.property
    def output(self) -> typing.Optional[BedrockInvokeModelOutputProps]:
        '''The destination location where the API response is written.

        If you specify this field, the API response body is replaced with a reference to the
        output location.

        :default: - The API response body is returned in the result.
        '''
        result = self._values.get("output")
        return typing.cast(typing.Optional[BedrockInvokeModelOutputProps], result)

    @builtins.property
    def trace_enabled(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether to enable or disable the Bedrock trace.

        :default: - Trace is not enabled for the invocation.
        '''
        result = self._values.get("trace_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BedrockInvokeModelProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayEndpointBaseOptions",
    jsii_struct_bases=[],
    name_mapping={
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
    },
)
class CallApiGatewayEndpointBaseOptions:
    def __init__(
        self,
        *,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
    ) -> None:
        '''Base CallApiGatewayEdnpoint Task Props.

        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # task_input: stepfunctions.TaskInput
            
            call_api_gateway_endpoint_base_options = stepfunctions_tasks.CallApiGatewayEndpointBaseOptions(
                method=stepfunctions_tasks.HttpMethod.GET,
            
                # the properties below are optional
                api_path="apiPath",
                auth_type=stepfunctions_tasks.AuthType.NO_AUTH,
                headers=task_input,
                query_parameters=task_input,
                request_body=task_input
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4107848c47aba9a7dbc4e9b2361d44557c7a6b88dafbfdc6e2cb60d31f37739a)
            check_type(argname="argument method", value=method, expected_type=type_hints["method"])
            check_type(argname="argument api_path", value=api_path, expected_type=type_hints["api_path"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
            check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            check_type(argname="argument request_body", value=request_body, expected_type=type_hints["request_body"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "method": method,
        }
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body

    @builtins.property
    def method(self) -> "HttpMethod":
        '''Http method for the API.'''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''Path parameters appended after API endpoint.

        :default: - No path
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''Authentication methods.

        :default: AuthType.NO_AUTH
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP request information that does not relate to contents of the request.

        :default: - No headers
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''Query strings attatched to end of request.

        :default: - No query parameters
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP Request body.

        :default: - No request body
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayEndpointBaseOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayEndpointBaseProps",
    jsii_struct_bases=[
        _TaskStateBaseProps_3a62b6d0, CallApiGatewayEndpointBaseOptions
    ],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "outputs": "outputs",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
    },
)
class CallApiGatewayEndpointBaseProps(
    _TaskStateBaseProps_3a62b6d0,
    CallApiGatewayEndpointBaseOptions,
):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
    ) -> None:
        '''Base CallApiGatewayEndpoint Task Props.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # outputs: Any
            # result_selector: Any
            # task_input: stepfunctions.TaskInput
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            call_api_gateway_endpoint_base_props = stepfunctions_tasks.CallApiGatewayEndpointBaseProps(
                method=stepfunctions_tasks.HttpMethod.GET,
            
                # the properties below are optional
                api_path="apiPath",
                assign={
                    "assign_key": assign
                },
                auth_type=stepfunctions_tasks.AuthType.NO_AUTH,
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                headers=task_input,
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output_path="outputPath",
                outputs=outputs,
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                query_parameters=task_input,
                request_body=task_input,
                result_path="resultPath",
                result_selector={
                    "result_selector_key": result_selector
                },
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e9dbc58cb4f9a28fbab12032a3c39bc929dcf76e35816fe9ba25aab36499c005)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument method", value=method, expected_type=type_hints["method"])
            check_type(argname="argument api_path", value=api_path, expected_type=type_hints["api_path"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
            check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            check_type(argname="argument request_body", value=request_body, expected_type=type_hints["request_body"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "method": method,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if outputs is not None:
            self._values["outputs"] = outputs
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def method(self) -> "HttpMethod":
        '''Http method for the API.'''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''Path parameters appended after API endpoint.

        :default: - No path
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''Authentication methods.

        :default: AuthType.NO_AUTH
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP request information that does not relate to contents of the request.

        :default: - No headers
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''Query strings attatched to end of request.

        :default: - No query parameters
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP Request body.

        :default: - No request body
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayEndpointBaseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayEndpointJsonPathBaseProps",
    jsii_struct_bases=[
        _TaskStateJsonPathBaseProps_57dc2260, CallApiGatewayEndpointBaseOptions
    ],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
    },
)
class CallApiGatewayEndpointJsonPathBaseProps(
    _TaskStateJsonPathBaseProps_57dc2260,
    CallApiGatewayEndpointBaseOptions,
):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
    ) -> None:
        '''Base CallApiGatewayEndpoint Task Props.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # result_selector: Any
            # task_input: stepfunctions.TaskInput
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            call_api_gateway_endpoint_json_path_base_props = stepfunctions_tasks.CallApiGatewayEndpointJsonPathBaseProps(
                method=stepfunctions_tasks.HttpMethod.GET,
            
                # the properties below are optional
                api_path="apiPath",
                assign={
                    "assign_key": assign
                },
                auth_type=stepfunctions_tasks.AuthType.NO_AUTH,
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                headers=task_input,
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output_path="outputPath",
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                query_parameters=task_input,
                request_body=task_input,
                result_path="resultPath",
                result_selector={
                    "result_selector_key": result_selector
                },
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__179d2325071ab6b6aa0bc2b9cc4422c61170ff2cd60598bbfe08fd660cf6da7f)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument method", value=method, expected_type=type_hints["method"])
            check_type(argname="argument api_path", value=api_path, expected_type=type_hints["api_path"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
            check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            check_type(argname="argument request_body", value=request_body, expected_type=type_hints["request_body"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "method": method,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def method(self) -> "HttpMethod":
        '''Http method for the API.'''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''Path parameters appended after API endpoint.

        :default: - No path
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''Authentication methods.

        :default: AuthType.NO_AUTH
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP request information that does not relate to contents of the request.

        :default: - No headers
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''Query strings attatched to end of request.

        :default: - No query parameters
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP Request body.

        :default: - No request body
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayEndpointJsonPathBaseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayEndpointJsonataBaseProps",
    jsii_struct_bases=[
        _TaskStateJsonataBaseProps_0e91775c, CallApiGatewayEndpointBaseOptions
    ],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "outputs": "outputs",
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
    },
)
class CallApiGatewayEndpointJsonataBaseProps(
    _TaskStateJsonataBaseProps_0e91775c,
    CallApiGatewayEndpointBaseOptions,
):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
    ) -> None:
        '''Base CallApiGatewayEndpoint Task Props.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # outputs: Any
            # task_input: stepfunctions.TaskInput
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            call_api_gateway_endpoint_jsonata_base_props = stepfunctions_tasks.CallApiGatewayEndpointJsonataBaseProps(
                method=stepfunctions_tasks.HttpMethod.GET,
            
                # the properties below are optional
                api_path="apiPath",
                assign={
                    "assign_key": assign
                },
                auth_type=stepfunctions_tasks.AuthType.NO_AUTH,
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                headers=task_input,
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                outputs=outputs,
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                query_parameters=task_input,
                request_body=task_input,
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b02e7875aeb3f5d4e016a1a2a7837171c32dfd482bc4676d6f4b2356f7d8d8c)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument method", value=method, expected_type=type_hints["method"])
            check_type(argname="argument api_path", value=api_path, expected_type=type_hints["api_path"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
            check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            check_type(argname="argument request_body", value=request_body, expected_type=type_hints["request_body"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "method": method,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if outputs is not None:
            self._values["outputs"] = outputs
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def method(self) -> "HttpMethod":
        '''Http method for the API.'''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''Path parameters appended after API endpoint.

        :default: - No path
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''Authentication methods.

        :default: AuthType.NO_AUTH
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP request information that does not relate to contents of the request.

        :default: - No headers
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''Query strings attatched to end of request.

        :default: - No query parameters
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP Request body.

        :default: - No request body
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayEndpointJsonataBaseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CallApiGatewayHttpApiEndpoint(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayHttpApiEndpoint",
):
    '''Call HTTP API endpoint as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_apigatewayv2 as apigatewayv2
        
        http_api = apigatewayv2.HttpApi(self, "MyHttpApi")
        
        invoke_task = tasks.CallApiGatewayHttpApiEndpoint(self, "Call HTTP API",
            api_id=http_api.api_id,
            api_stack=Stack.of(http_api),
            method=tasks.HttpMethod.GET
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        api_id: builtins.str,
        api_stack: _Stack_2866e57f,
        stage_name: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param api_id: The Id of the API to call.
        :param api_stack: The Stack in which the API is defined.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway. Default: '$default'
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e727eac1466f73756eed51efa40866442f05dae5d955ec7e6f15390221c1c6f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallApiGatewayHttpApiEndpointProps(
            api_id=api_id,
            api_stack=api_stack,
            stage_name=stage_name,
            result_path=result_path,
            result_selector=result_selector,
            method=method,
            api_path=api_path,
            auth_type=auth_type,
            headers=headers,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        api_id: builtins.str,
        api_stack: _Stack_2866e57f,
        stage_name: typing.Optional[builtins.str] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
    ) -> "CallApiGatewayHttpApiEndpoint":
        '''Call HTTP API endpoint as a Task using JSONata.

        :param scope: -
        :param id: -
        :param api_id: The Id of the API to call.
        :param api_stack: The Stack in which the API is defined.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway. Default: '$default'
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f91f56468356abd62afbce9e455619ba192e4cb3321d5da161170768f8fe8171)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallApiGatewayHttpApiEndpointJsonataProps(
            api_id=api_id,
            api_stack=api_stack,
            stage_name=stage_name,
            method=method,
            api_path=api_path,
            auth_type=auth_type,
            headers=headers,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            outputs=outputs,
        )

        return typing.cast("CallApiGatewayHttpApiEndpoint", jsii.sinvoke(cls, "jsonata", [scope, id, props]))

    @jsii.member(jsii_name="jsonPath")
    @builtins.classmethod
    def json_path(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        api_id: builtins.str,
        api_stack: _Stack_2866e57f,
        stage_name: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> "CallApiGatewayHttpApiEndpoint":
        '''Call HTTP API endpoint as a Task using JSONPath.

        :param scope: -
        :param id: -
        :param api_id: The Id of the API to call.
        :param api_stack: The Stack in which the API is defined.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway. Default: '$default'
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__989a0e41ca9ca0117e0a4554567eb7a2caa9572194b13179b7f4a25ead144304)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallApiGatewayHttpApiEndpointJsonPathProps(
            api_id=api_id,
            api_stack=api_stack,
            stage_name=stage_name,
            result_path=result_path,
            result_selector=result_selector,
            method=method,
            api_path=api_path,
            auth_type=auth_type,
            headers=headers,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
        )

        return typing.cast("CallApiGatewayHttpApiEndpoint", jsii.sinvoke(cls, "jsonPath", [scope, id, props]))

    @jsii.member(jsii_name="createPolicyStatements")
    def _create_policy_statements(self) -> typing.List[_PolicyStatement_0fe33853]:
        return typing.cast(typing.List[_PolicyStatement_0fe33853], jsii.invoke(self, "createPolicyStatements", []))

    @builtins.property
    @jsii.member(jsii_name="apiEndpoint")
    def _api_endpoint(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "apiEndpoint"))

    @builtins.property
    @jsii.member(jsii_name="arnForExecuteApi")
    def _arn_for_execute_api(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "arnForExecuteApi"))

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def _stage_name(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayHttpApiEndpointOptions",
    jsii_struct_bases=[],
    name_mapping={
        "api_id": "apiId",
        "api_stack": "apiStack",
        "stage_name": "stageName",
    },
)
class CallApiGatewayHttpApiEndpointOptions:
    def __init__(
        self,
        *,
        api_id: builtins.str,
        api_stack: _Stack_2866e57f,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Base properties for calling an HTTP API Endpoint.

        :param api_id: The Id of the API to call.
        :param api_stack: The Stack in which the API is defined.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway. Default: '$default'

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # stack: cdk.Stack
            
            call_api_gateway_http_api_endpoint_options = stepfunctions_tasks.CallApiGatewayHttpApiEndpointOptions(
                api_id="apiId",
                api_stack=stack,
            
                # the properties below are optional
                stage_name="stageName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d51e752bf5e1236aa7ae79bd260c9705ef8b03437c05e6d561db45d4623a495)
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument api_stack", value=api_stack, expected_type=type_hints["api_stack"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api_id": api_id,
            "api_stack": api_stack,
        }
        if stage_name is not None:
            self._values["stage_name"] = stage_name

    @builtins.property
    def api_id(self) -> builtins.str:
        '''The Id of the API to call.'''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def api_stack(self) -> _Stack_2866e57f:
        '''The Stack in which the API is defined.'''
        result = self._values.get("api_stack")
        assert result is not None, "Required property 'api_stack' is missing"
        return typing.cast(_Stack_2866e57f, result)

    @builtins.property
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''Name of the stage where the API is deployed to in API Gateway.

        :default: '$default'
        '''
        result = self._values.get("stage_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayHttpApiEndpointOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayHttpApiEndpointProps",
    jsii_struct_bases=[
        CallApiGatewayEndpointBaseProps, CallApiGatewayHttpApiEndpointOptions
    ],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "outputs": "outputs",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
        "api_id": "apiId",
        "api_stack": "apiStack",
        "stage_name": "stageName",
    },
)
class CallApiGatewayHttpApiEndpointProps(
    CallApiGatewayEndpointBaseProps,
    CallApiGatewayHttpApiEndpointOptions,
):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        api_id: builtins.str,
        api_stack: _Stack_2866e57f,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for calling an HTTP API Endpoint.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param api_id: The Id of the API to call.
        :param api_stack: The Stack in which the API is defined.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway. Default: '$default'

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_apigatewayv2 as apigatewayv2
            
            http_api = apigatewayv2.HttpApi(self, "MyHttpApi")
            
            invoke_task = tasks.CallApiGatewayHttpApiEndpoint(self, "Call HTTP API",
                api_id=http_api.api_id,
                api_stack=Stack.of(http_api),
                method=tasks.HttpMethod.GET
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a41fc59378bfacae33639438a7a9b775728e418d0a325893bbe9defe089edfd)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument method", value=method, expected_type=type_hints["method"])
            check_type(argname="argument api_path", value=api_path, expected_type=type_hints["api_path"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
            check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            check_type(argname="argument request_body", value=request_body, expected_type=type_hints["request_body"])
            check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
            check_type(argname="argument api_stack", value=api_stack, expected_type=type_hints["api_stack"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "method": method,
            "api_id": api_id,
            "api_stack": api_stack,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if outputs is not None:
            self._values["outputs"] = outputs
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body
        if stage_name is not None:
            self._values["stage_name"] = stage_name

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def method(self) -> "HttpMethod":
        '''Http method for the API.'''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''Path parameters appended after API endpoint.

        :default: - No path
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''Authentication methods.

        :default: AuthType.NO_AUTH
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP request information that does not relate to contents of the request.

        :default: - No headers
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''Query strings attatched to end of request.

        :default: - No query parameters
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP Request body.

        :default: - No request body
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def api_id(self) -> builtins.str:
        '''The Id of the API to call.'''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def api_stack(self) -> _Stack_2866e57f:
        '''The Stack in which the API is defined.'''
        result = self._values.get("api_stack")
        assert result is not None, "Required property 'api_stack' is missing"
        return typing.cast(_Stack_2866e57f, result)

    @builtins.property
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''Name of the stage where the API is deployed to in API Gateway.

        :default: '$default'
        '''
        result = self._values.get("stage_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayHttpApiEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CallApiGatewayRestApiEndpoint(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayRestApiEndpoint",
):
    '''Call REST API endpoint as a Task.

    Be aware that the header values must be arrays. When passing the Task Token
    in the headers field ``WAIT_FOR_TASK_TOKEN`` integration, use
    ``JsonPath.array()`` to wrap the token in an array::

       import aws_cdk.aws_apigateway as apigateway
       # api: apigateway.RestApi


       tasks.CallApiGatewayRestApiEndpoint(self, "Endpoint",
           api=api,
           stage_name="Stage",
           method=tasks.HttpMethod.PUT,
           integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
           headers=sfn.TaskInput.from_object({
               "TaskToken": sfn.JsonPath.array(sfn.JsonPath.task_token)
           })
       )

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_apigateway as apigateway
        # api: apigateway.RestApi
        
        
        tasks.CallApiGatewayRestApiEndpoint.jsonata(self, "Endpoint",
            api=api,
            stage_name="Stage",
            method=tasks.HttpMethod.PUT,
            integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            headers=sfn.TaskInput.from_object({
                "TaskToken": "{% States.Array($states.context.taskToken) %}"
            })
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        api: _IRestApi_1f02523d,
        stage_name: builtins.str,
        region: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param api: API to call.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway.
        :param region: Specify a custom Region where the API is deployed, e.g. 'us-east-1'. Default: - Uses the Region of the stack containing the ``api``.
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a76886b18651b0cbd9c7ad80e6a462a8540904f212eeb56575f9a2c9ab18e143)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallApiGatewayRestApiEndpointProps(
            api=api,
            stage_name=stage_name,
            region=region,
            result_path=result_path,
            result_selector=result_selector,
            method=method,
            api_path=api_path,
            auth_type=auth_type,
            headers=headers,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        api: _IRestApi_1f02523d,
        stage_name: builtins.str,
        region: typing.Optional[builtins.str] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
    ) -> "CallApiGatewayRestApiEndpoint":
        '''Call REST API endpoint as a Task using JSONata.

        Be aware that the header values must be arrays. When passing the Task Token
        in the headers field ``WAIT_FOR_TASK_TOKEN`` integration, use
        ``JsonPath.array()`` to wrap the token in an array::

           import aws_cdk.aws_apigateway as apigateway
           # api: apigateway.RestApi


           tasks.CallApiGatewayRestApiEndpoint.jsonata(self, "Endpoint",
               api=api,
               stage_name="Stage",
               method=tasks.HttpMethod.PUT,
               integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
               headers=sfn.TaskInput.from_object({
                   "TaskToken": "{% States.Array($states.context.taskToken) %}"
               })
           )

        :param scope: -
        :param id: -
        :param api: API to call.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway.
        :param region: Specify a custom Region where the API is deployed, e.g. 'us-east-1'. Default: - Uses the Region of the stack containing the ``api``.
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d2442fdcffc01527df3494d29d520e172a5fc4b5567d13ab1172c04a41e1592a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallApiGatewayRestApiEndpointJsonataProps(
            api=api,
            stage_name=stage_name,
            region=region,
            method=method,
            api_path=api_path,
            auth_type=auth_type,
            headers=headers,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            outputs=outputs,
        )

        return typing.cast("CallApiGatewayRestApiEndpoint", jsii.sinvoke(cls, "jsonata", [scope, id, props]))

    @jsii.member(jsii_name="jsonPath")
    @builtins.classmethod
    def json_path(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        api: _IRestApi_1f02523d,
        stage_name: builtins.str,
        region: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> "CallApiGatewayRestApiEndpoint":
        '''Call REST API endpoint as a Task  using JSONPath.

        Be aware that the header values must be arrays. When passing the Task Token
        in the headers field ``WAIT_FOR_TASK_TOKEN`` integration, use
        ``JsonPath.array()`` to wrap the token in an array::

           import aws_cdk.aws_apigateway as apigateway
           # api: apigateway.RestApi


           tasks.CallApiGatewayRestApiEndpoint.json_path(self, "Endpoint",
               api=api,
               stage_name="Stage",
               method=tasks.HttpMethod.PUT,
               integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
               headers=sfn.TaskInput.from_object({
                   "TaskToken": sfn.JsonPath.array(sfn.JsonPath.task_token)
               })
           )

        :param scope: -
        :param id: -
        :param api: API to call.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway.
        :param region: Specify a custom Region where the API is deployed, e.g. 'us-east-1'. Default: - Uses the Region of the stack containing the ``api``.
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e4dc06a85acae89b8512aa5463430dbf35b4fed27323d6928f986b34f2f857a0)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallApiGatewayRestApiEndpointJsonPathProps(
            api=api,
            stage_name=stage_name,
            region=region,
            result_path=result_path,
            result_selector=result_selector,
            method=method,
            api_path=api_path,
            auth_type=auth_type,
            headers=headers,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
        )

        return typing.cast("CallApiGatewayRestApiEndpoint", jsii.sinvoke(cls, "jsonPath", [scope, id, props]))

    @jsii.member(jsii_name="createPolicyStatements")
    def _create_policy_statements(self) -> typing.List[_PolicyStatement_0fe33853]:
        return typing.cast(typing.List[_PolicyStatement_0fe33853], jsii.invoke(self, "createPolicyStatements", []))

    @builtins.property
    @jsii.member(jsii_name="apiEndpoint")
    def _api_endpoint(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "apiEndpoint"))

    @builtins.property
    @jsii.member(jsii_name="arnForExecuteApi")
    def _arn_for_execute_api(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "arnForExecuteApi"))

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def _stage_name(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayRestApiEndpointOptions",
    jsii_struct_bases=[],
    name_mapping={"api": "api", "stage_name": "stageName", "region": "region"},
)
class CallApiGatewayRestApiEndpointOptions:
    def __init__(
        self,
        *,
        api: _IRestApi_1f02523d,
        stage_name: builtins.str,
        region: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Base properties for calling an REST API Endpoint.

        :param api: API to call.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway.
        :param region: Specify a custom Region where the API is deployed, e.g. 'us-east-1'. Default: - Uses the Region of the stack containing the ``api``.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # rest_api: apigateway.RestApi
            
            call_api_gateway_rest_api_endpoint_options = stepfunctions_tasks.CallApiGatewayRestApiEndpointOptions(
                api=rest_api,
                stage_name="stageName",
            
                # the properties below are optional
                region="region"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f3e7028403a9fb243cf40cd7590bda960bfb12daacd7648c71359997146e9dba)
            check_type(argname="argument api", value=api, expected_type=type_hints["api"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api": api,
            "stage_name": stage_name,
        }
        if region is not None:
            self._values["region"] = region

    @builtins.property
    def api(self) -> _IRestApi_1f02523d:
        '''API to call.'''
        result = self._values.get("api")
        assert result is not None, "Required property 'api' is missing"
        return typing.cast(_IRestApi_1f02523d, result)

    @builtins.property
    def stage_name(self) -> builtins.str:
        '''Name of the stage where the API is deployed to in API Gateway.'''
        result = self._values.get("stage_name")
        assert result is not None, "Required property 'stage_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def region(self) -> typing.Optional[builtins.str]:
        '''Specify a custom Region where the API is deployed, e.g. 'us-east-1'.

        :default: - Uses the Region of the stack containing the ``api``.
        '''
        result = self._values.get("region")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayRestApiEndpointOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallApiGatewayRestApiEndpointProps",
    jsii_struct_bases=[
        CallApiGatewayEndpointBaseProps, CallApiGatewayRestApiEndpointOptions
    ],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "outputs": "outputs",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
        "api": "api",
        "stage_name": "stageName",
        "region": "region",
    },
)
class CallApiGatewayRestApiEndpointProps(
    CallApiGatewayEndpointBaseProps,
    CallApiGatewayRestApiEndpointOptions,
):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_91b91b91] = None,
        query_parameters: typing.Optional[_TaskInput_91b91b91] = None,
        request_body: typing.Optional[_TaskInput_91b91b91] = None,
        api: _IRestApi_1f02523d,
        stage_name: builtins.str,
        region: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for calling an REST API Endpoint.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param method: Http method for the API.
        :param api_path: Path parameters appended after API endpoint. Default: - No path
        :param auth_type: Authentication methods. Default: AuthType.NO_AUTH
        :param headers: HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: Query strings attatched to end of request. Default: - No query parameters
        :param request_body: HTTP Request body. Default: - No request body
        :param api: API to call.
        :param stage_name: Name of the stage where the API is deployed to in API Gateway.
        :param region: Specify a custom Region where the API is deployed, e.g. 'us-east-1'. Default: - Uses the Region of the stack containing the ``api``.

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_apigateway as apigateway
            
            rest_api = apigateway.RestApi(self, "MyRestApi")
            invoke_task = tasks.CallApiGatewayRestApiEndpoint(self, "Call REST API",
                api=rest_api,
                stage_name="prod",
                method=tasks.HttpMethod.GET,
                region="us-west-2"
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f5e083701921cac85bdeb9778c84e51785062fe16c2575a0df424f5a5a1dfa22)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument method", value=method, expected_type=type_hints["method"])
            check_type(argname="argument api_path", value=api_path, expected_type=type_hints["api_path"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
            check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            check_type(argname="argument request_body", value=request_body, expected_type=type_hints["request_body"])
            check_type(argname="argument api", value=api, expected_type=type_hints["api"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "method": method,
            "api": api,
            "stage_name": stage_name,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if outputs is not None:
            self._values["outputs"] = outputs
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body
        if region is not None:
            self._values["region"] = region

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def method(self) -> "HttpMethod":
        '''Http method for the API.'''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''Path parameters appended after API endpoint.

        :default: - No path
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''Authentication methods.

        :default: AuthType.NO_AUTH
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP request information that does not relate to contents of the request.

        :default: - No headers
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''Query strings attatched to end of request.

        :default: - No query parameters
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_91b91b91]:
        '''HTTP Request body.

        :default: - No request body
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_91b91b91], result)

    @builtins.property
    def api(self) -> _IRestApi_1f02523d:
        '''API to call.'''
        result = self._values.get("api")
        assert result is not None, "Required property 'api' is missing"
        return typing.cast(_IRestApi_1f02523d, result)

    @builtins.property
    def stage_name(self) -> builtins.str:
        '''Name of the stage where the API is deployed to in API Gateway.'''
        result = self._values.get("stage_name")
        assert result is not None, "Required property 'stage_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def region(self) -> typing.Optional[builtins.str]:
        '''Specify a custom Region where the API is deployed, e.g. 'us-east-1'.

        :default: - Uses the Region of the stack containing the ``api``.
        '''
        result = self._values.get("region")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayRestApiEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CallAwsService(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallAwsService",
):
    '''A StepFunctions task to call an AWS service API.

    :exampleMetadata: infused

    Example::

        # my_bucket: s3.Bucket
        
        get_object = tasks.CallAwsService(self, "GetObject",
            service="s3",
            action="getObject",
            parameters={
                "Bucket": my_bucket.bucket_name,
                "Key": sfn.JsonPath.string_at("$.key")
            },
            iam_resources=[my_bucket.arn_for_objects("*")]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``.
        :param service: The AWS service to call.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param iam_action: The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call. Use PascalCase for the parameter names. Default: - no parameters
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e4534f8232f0840d6ef9dc3920130b2d24934f22c4483b9bd14a19e8152f0b6f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallAwsServiceProps(
            action=action,
            iam_resources=iam_resources,
            service=service,
            additional_iam_statements=additional_iam_statements,
            iam_action=iam_action,
            parameters=parameters,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
    ) -> "CallAwsService":
        '''A StepFunctions task using JSONata to call an AWS service API.

        :param scope: -
        :param id: -
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``.
        :param service: The AWS service to call.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param iam_action: The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call. Use PascalCase for the parameter names. Default: - no parameters
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ffce2a7c897638fb58b3e958eabda4cce51903f9f384d7ea55e68fba5a00654)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallAwsServiceJsonataProps(
            action=action,
            iam_resources=iam_resources,
            service=service,
            additional_iam_statements=additional_iam_statements,
            iam_action=iam_action,
            parameters=parameters,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            outputs=outputs,
        )

        return typing.cast("CallAwsService", jsii.sinvoke(cls, "jsonata", [scope, id, props]))

    @jsii.member(jsii_name="jsonPath")
    @builtins.classmethod
    def json_path(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> "CallAwsService":
        '''A StepFunctions task using JSONPath to call an AWS service API.

        :param scope: -
        :param id: -
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``.
        :param service: The AWS service to call.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param iam_action: The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call. Use PascalCase for the parameter names. Default: - no parameters
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fc043e415ee9179faf3e938b9c550288a291b4a3c6b51e3e732c8b516db78e4c)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallAwsServiceJsonPathProps(
            action=action,
            iam_resources=iam_resources,
            service=service,
            additional_iam_statements=additional_iam_statements,
            iam_action=iam_action,
            parameters=parameters,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
        )

        return typing.cast("CallAwsService", jsii.sinvoke(cls, "jsonPath", [scope, id, props]))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


class CallAwsServiceCrossRegion(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallAwsServiceCrossRegion",
):
    '''A Step Functions task to call an AWS service API across regions.

    This task creates a Lambda function to call cross-region AWS API and invokes it.

    :exampleMetadata: infused

    Example::

        # my_bucket: s3.Bucket
        
        get_object = tasks.CallAwsServiceCrossRegion(self, "GetObject",
            region="ap-northeast-1",
            service="s3",
            action="getObject",
            parameters={
                "Bucket": my_bucket.bucket_name,
                "Key": sfn.JsonPath.string_at("$.key")
            },
            iam_resources=[my_bucket.arn_for_objects("*")]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        region: builtins.str,
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        endpoint: typing.Optional[builtins.str] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        retry_on_service_exceptions: typing.Optional[builtins.bool] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.
        :param region: The AWS region to call this AWS API for.
        :param service: The AWS service to call in AWS SDK for JavaScript v3 format.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param endpoint: The AWS API endpoint. Default: Do not override API endpoint.
        :param iam_action: The action for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call in AWS SDK for JavaScript v3 format. Default: - no parameters
        :param retry_on_service_exceptions: Whether to retry on the backend Lambda service exceptions. This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException``, ``Lambda.SdkClientException``, and ``Lambda.ClientExecutionTimeoutException`` with an interval of 2 seconds, a back-off rate of 2 and 6 maximum attempts. Default: true
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__788d4ab0ebc83fedbcaa927c3a9c9a8e1a7299ac43edcd72cc49a907a5b183b1)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallAwsServiceCrossRegionProps(
            action=action,
            iam_resources=iam_resources,
            region=region,
            service=service,
            additional_iam_statements=additional_iam_statements,
            endpoint=endpoint,
            iam_action=iam_action,
            parameters=parameters,
            retry_on_service_exceptions=retry_on_service_exceptions,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        region: builtins.str,
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        endpoint: typing.Optional[builtins.str] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        retry_on_service_exceptions: typing.Optional[builtins.bool] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
    ) -> "CallAwsServiceCrossRegion":
        '''A Step Functions task using JSONata to call an AWS service API across regions.

        This task creates a Lambda function to call cross-region AWS API and invokes it.

        :param scope: -
        :param id: -
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.
        :param region: The AWS region to call this AWS API for.
        :param service: The AWS service to call in AWS SDK for JavaScript v3 format.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param endpoint: The AWS API endpoint. Default: Do not override API endpoint.
        :param iam_action: The action for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call in AWS SDK for JavaScript v3 format. Default: - no parameters
        :param retry_on_service_exceptions: Whether to retry on the backend Lambda service exceptions. This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException``, ``Lambda.SdkClientException``, and ``Lambda.ClientExecutionTimeoutException`` with an interval of 2 seconds, a back-off rate of 2 and 6 maximum attempts. Default: true
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__189a727c13bda96d236c874d36e7906322a6093008e28b2a1285aca85ddac008)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallAwsServiceCrossRegionJsonataProps(
            action=action,
            iam_resources=iam_resources,
            region=region,
            service=service,
            additional_iam_statements=additional_iam_statements,
            endpoint=endpoint,
            iam_action=iam_action,
            parameters=parameters,
            retry_on_service_exceptions=retry_on_service_exceptions,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            outputs=outputs,
        )

        return typing.cast("CallAwsServiceCrossRegion", jsii.sinvoke(cls, "jsonata", [scope, id, props]))

    @jsii.member(jsii_name="jsonPath")
    @builtins.classmethod
    def json_path(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        region: builtins.str,
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        endpoint: typing.Optional[builtins.str] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        retry_on_service_exceptions: typing.Optional[builtins.bool] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> "CallAwsServiceCrossRegion":
        '''A Step Functions task using JSONPath to call an AWS service API across regions.

        This task creates a Lambda function to call cross-region AWS API and invokes it.

        :param scope: -
        :param id: -
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.
        :param region: The AWS region to call this AWS API for.
        :param service: The AWS service to call in AWS SDK for JavaScript v3 format.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param endpoint: The AWS API endpoint. Default: Do not override API endpoint.
        :param iam_action: The action for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call in AWS SDK for JavaScript v3 format. Default: - no parameters
        :param retry_on_service_exceptions: Whether to retry on the backend Lambda service exceptions. This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException``, ``Lambda.SdkClientException``, and ``Lambda.ClientExecutionTimeoutException`` with an interval of 2 seconds, a back-off rate of 2 and 6 maximum attempts. Default: true
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ba03b02fba8439dd7145ae60187c001f7936820326e9bc5a6eff0b5ae0ce0edb)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CallAwsServiceCrossRegionJsonPathProps(
            action=action,
            iam_resources=iam_resources,
            region=region,
            service=service,
            additional_iam_statements=additional_iam_statements,
            endpoint=endpoint,
            iam_action=iam_action,
            parameters=parameters,
            retry_on_service_exceptions=retry_on_service_exceptions,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
        )

        return typing.cast("CallAwsServiceCrossRegion", jsii.sinvoke(cls, "jsonPath", [scope, id, props]))

    @builtins.property
    @jsii.member(jsii_name="lambdaFunction")
    def _lambda_function(self) -> _IFunction_6adb0ab8:
        return typing.cast(_IFunction_6adb0ab8, jsii.get(self, "lambdaFunction"))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallAwsServiceCrossRegionJsonPathProps",
    jsii_struct_bases=[_TaskStateJsonPathBaseProps_57dc2260],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "action": "action",
        "iam_resources": "iamResources",
        "region": "region",
        "service": "service",
        "additional_iam_statements": "additionalIamStatements",
        "endpoint": "endpoint",
        "iam_action": "iamAction",
        "parameters": "parameters",
        "retry_on_service_exceptions": "retryOnServiceExceptions",
    },
)
class CallAwsServiceCrossRegionJsonPathProps(_TaskStateJsonPathBaseProps_57dc2260):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        region: builtins.str,
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        endpoint: typing.Optional[builtins.str] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        retry_on_service_exceptions: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Properties for calling an AWS service's API action using JSONPath from your state machine across regions.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.
        :param region: The AWS region to call this AWS API for.
        :param service: The AWS service to call in AWS SDK for JavaScript v3 format.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param endpoint: The AWS API endpoint. Default: Do not override API endpoint.
        :param iam_action: The action for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call in AWS SDK for JavaScript v3 format. Default: - no parameters
        :param retry_on_service_exceptions: Whether to retry on the backend Lambda service exceptions. This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException``, ``Lambda.SdkClientException``, and ``Lambda.ClientExecutionTimeoutException`` with an interval of 2 seconds, a back-off rate of 2 and 6 maximum attempts. Default: true

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_iam as iam
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # parameters: Any
            # policy_statement: iam.PolicyStatement
            # result_selector: Any
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            call_aws_service_cross_region_json_path_props = stepfunctions_tasks.CallAwsServiceCrossRegionJsonPathProps(
                action="action",
                iam_resources=["iamResources"],
                region="region",
                service="service",
            
                # the properties below are optional
                additional_iam_statements=[policy_statement],
                assign={
                    "assign_key": assign
                },
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                endpoint="endpoint",
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                iam_action="iamAction",
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output_path="outputPath",
                parameters={
                    "parameters_key": parameters
                },
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                result_path="resultPath",
                result_selector={
                    "result_selector_key": result_selector
                },
                retry_on_service_exceptions=False,
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4e34621ad7f857ebf21cf485550f86fa2585edba2ad64e6abb18ee79936d516d)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument iam_resources", value=iam_resources, expected_type=type_hints["iam_resources"])
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument additional_iam_statements", value=additional_iam_statements, expected_type=type_hints["additional_iam_statements"])
            check_type(argname="argument endpoint", value=endpoint, expected_type=type_hints["endpoint"])
            check_type(argname="argument iam_action", value=iam_action, expected_type=type_hints["iam_action"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument retry_on_service_exceptions", value=retry_on_service_exceptions, expected_type=type_hints["retry_on_service_exceptions"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "action": action,
            "iam_resources": iam_resources,
            "region": region,
            "service": service,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if additional_iam_statements is not None:
            self._values["additional_iam_statements"] = additional_iam_statements
        if endpoint is not None:
            self._values["endpoint"] = endpoint
        if iam_action is not None:
            self._values["iam_action"] = iam_action
        if parameters is not None:
            self._values["parameters"] = parameters
        if retry_on_service_exceptions is not None:
            self._values["retry_on_service_exceptions"] = retry_on_service_exceptions

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def action(self) -> builtins.str:
        '''The API action to call.

        Use camelCase.
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def iam_resources(self) -> typing.List[builtins.str]:
        '''The resources for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.'''
        result = self._values.get("iam_resources")
        assert result is not None, "Required property 'iam_resources' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def region(self) -> builtins.str:
        '''The AWS region to call this AWS API for.

        Example::

            "us-east-1"
        '''
        result = self._values.get("region")
        assert result is not None, "Required property 'region' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''The AWS service to call in AWS SDK for JavaScript v3 format.

        :see: https://docs.aws.amazon.com/AWSJavaScriptSDK/v3/latest/

        Example::

            "s3"
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def additional_iam_statements(
        self,
    ) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        '''Additional IAM statements that will be added to the state machine role's policy.

        Use in the case where the call requires more than a single statement to
        be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions
        to read the object on which it must act.

        :default: - no additional statements are added
        '''
        result = self._values.get("additional_iam_statements")
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], result)

    @builtins.property
    def endpoint(self) -> typing.Optional[builtins.str]:
        '''The AWS API endpoint.

        :default: Do not override API endpoint.
        '''
        result = self._values.get("endpoint")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iam_action(self) -> typing.Optional[builtins.str]:
        '''The action for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.

        By default the action for this IAM statement will be ``service:action``.

        Use in the case where the IAM action name does not match with the
        API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``.

        :default: - service:action
        '''
        result = self._values.get("iam_action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Parameters for the API action call in AWS SDK for JavaScript v3 format.

        :default: - no parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def retry_on_service_exceptions(self) -> typing.Optional[builtins.bool]:
        '''Whether to retry on the backend Lambda service exceptions.

        This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException``,
        ``Lambda.SdkClientException``, and ``Lambda.ClientExecutionTimeoutException``
        with an interval of 2 seconds, a back-off rate
        of 2 and 6 maximum attempts.

        :default: true

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/bp-lambda-serviceexception.html
        '''
        result = self._values.get("retry_on_service_exceptions")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallAwsServiceCrossRegionJsonPathProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallAwsServiceCrossRegionJsonataProps",
    jsii_struct_bases=[_TaskStateJsonataBaseProps_0e91775c],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "outputs": "outputs",
        "action": "action",
        "iam_resources": "iamResources",
        "region": "region",
        "service": "service",
        "additional_iam_statements": "additionalIamStatements",
        "endpoint": "endpoint",
        "iam_action": "iamAction",
        "parameters": "parameters",
        "retry_on_service_exceptions": "retryOnServiceExceptions",
    },
)
class CallAwsServiceCrossRegionJsonataProps(_TaskStateJsonataBaseProps_0e91775c):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        region: builtins.str,
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        endpoint: typing.Optional[builtins.str] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        retry_on_service_exceptions: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Properties for calling an AWS service's API action using JSONata from your state machine across regions.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.
        :param region: The AWS region to call this AWS API for.
        :param service: The AWS service to call in AWS SDK for JavaScript v3 format.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param endpoint: The AWS API endpoint. Default: Do not override API endpoint.
        :param iam_action: The action for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call in AWS SDK for JavaScript v3 format. Default: - no parameters
        :param retry_on_service_exceptions: Whether to retry on the backend Lambda service exceptions. This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException``, ``Lambda.SdkClientException``, and ``Lambda.ClientExecutionTimeoutException`` with an interval of 2 seconds, a back-off rate of 2 and 6 maximum attempts. Default: true

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_iam as iam
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # outputs: Any
            # parameters: Any
            # policy_statement: iam.PolicyStatement
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            call_aws_service_cross_region_jsonata_props = stepfunctions_tasks.CallAwsServiceCrossRegionJsonataProps(
                action="action",
                iam_resources=["iamResources"],
                region="region",
                service="service",
            
                # the properties below are optional
                additional_iam_statements=[policy_statement],
                assign={
                    "assign_key": assign
                },
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                endpoint="endpoint",
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                iam_action="iamAction",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                outputs=outputs,
                parameters={
                    "parameters_key": parameters
                },
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                retry_on_service_exceptions=False,
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8d4b9b7e83537d6af3d6d427b9caf775959ef86631f837ddd868fdc8cd29829c)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument iam_resources", value=iam_resources, expected_type=type_hints["iam_resources"])
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument additional_iam_statements", value=additional_iam_statements, expected_type=type_hints["additional_iam_statements"])
            check_type(argname="argument endpoint", value=endpoint, expected_type=type_hints["endpoint"])
            check_type(argname="argument iam_action", value=iam_action, expected_type=type_hints["iam_action"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument retry_on_service_exceptions", value=retry_on_service_exceptions, expected_type=type_hints["retry_on_service_exceptions"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "action": action,
            "iam_resources": iam_resources,
            "region": region,
            "service": service,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if outputs is not None:
            self._values["outputs"] = outputs
        if additional_iam_statements is not None:
            self._values["additional_iam_statements"] = additional_iam_statements
        if endpoint is not None:
            self._values["endpoint"] = endpoint
        if iam_action is not None:
            self._values["iam_action"] = iam_action
        if parameters is not None:
            self._values["parameters"] = parameters
        if retry_on_service_exceptions is not None:
            self._values["retry_on_service_exceptions"] = retry_on_service_exceptions

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def action(self) -> builtins.str:
        '''The API action to call.

        Use camelCase.
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def iam_resources(self) -> typing.List[builtins.str]:
        '''The resources for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.'''
        result = self._values.get("iam_resources")
        assert result is not None, "Required property 'iam_resources' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def region(self) -> builtins.str:
        '''The AWS region to call this AWS API for.

        Example::

            "us-east-1"
        '''
        result = self._values.get("region")
        assert result is not None, "Required property 'region' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''The AWS service to call in AWS SDK for JavaScript v3 format.

        :see: https://docs.aws.amazon.com/AWSJavaScriptSDK/v3/latest/

        Example::

            "s3"
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def additional_iam_statements(
        self,
    ) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        '''Additional IAM statements that will be added to the state machine role's policy.

        Use in the case where the call requires more than a single statement to
        be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions
        to read the object on which it must act.

        :default: - no additional statements are added
        '''
        result = self._values.get("additional_iam_statements")
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], result)

    @builtins.property
    def endpoint(self) -> typing.Optional[builtins.str]:
        '''The AWS API endpoint.

        :default: Do not override API endpoint.
        '''
        result = self._values.get("endpoint")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iam_action(self) -> typing.Optional[builtins.str]:
        '''The action for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.

        By default the action for this IAM statement will be ``service:action``.

        Use in the case where the IAM action name does not match with the
        API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``.

        :default: - service:action
        '''
        result = self._values.get("iam_action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Parameters for the API action call in AWS SDK for JavaScript v3 format.

        :default: - no parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def retry_on_service_exceptions(self) -> typing.Optional[builtins.bool]:
        '''Whether to retry on the backend Lambda service exceptions.

        This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException``,
        ``Lambda.SdkClientException``, and ``Lambda.ClientExecutionTimeoutException``
        with an interval of 2 seconds, a back-off rate
        of 2 and 6 maximum attempts.

        :default: true

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/bp-lambda-serviceexception.html
        '''
        result = self._values.get("retry_on_service_exceptions")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallAwsServiceCrossRegionJsonataProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallAwsServiceCrossRegionProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "outputs": "outputs",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "action": "action",
        "iam_resources": "iamResources",
        "region": "region",
        "service": "service",
        "additional_iam_statements": "additionalIamStatements",
        "endpoint": "endpoint",
        "iam_action": "iamAction",
        "parameters": "parameters",
        "retry_on_service_exceptions": "retryOnServiceExceptions",
    },
)
class CallAwsServiceCrossRegionProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        region: builtins.str,
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        endpoint: typing.Optional[builtins.str] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        retry_on_service_exceptions: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Properties for calling an AWS service's API action from your state machine across regions.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.
        :param region: The AWS region to call this AWS API for.
        :param service: The AWS service to call in AWS SDK for JavaScript v3 format.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param endpoint: The AWS API endpoint. Default: Do not override API endpoint.
        :param iam_action: The action for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call in AWS SDK for JavaScript v3 format. Default: - no parameters
        :param retry_on_service_exceptions: Whether to retry on the backend Lambda service exceptions. This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException``, ``Lambda.SdkClientException``, and ``Lambda.ClientExecutionTimeoutException`` with an interval of 2 seconds, a back-off rate of 2 and 6 maximum attempts. Default: true

        :exampleMetadata: infused

        Example::

            # my_bucket: s3.Bucket
            
            get_object = tasks.CallAwsServiceCrossRegion(self, "GetObject",
                region="ap-northeast-1",
                service="s3",
                action="getObject",
                parameters={
                    "Bucket": my_bucket.bucket_name,
                    "Key": sfn.JsonPath.string_at("$.key")
                },
                iam_resources=[my_bucket.arn_for_objects("*")]
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__992363fbcea98bcb2c2657a6b897b7392fc6dc4d41155241ad7622bdc9d61923)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument iam_resources", value=iam_resources, expected_type=type_hints["iam_resources"])
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument additional_iam_statements", value=additional_iam_statements, expected_type=type_hints["additional_iam_statements"])
            check_type(argname="argument endpoint", value=endpoint, expected_type=type_hints["endpoint"])
            check_type(argname="argument iam_action", value=iam_action, expected_type=type_hints["iam_action"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument retry_on_service_exceptions", value=retry_on_service_exceptions, expected_type=type_hints["retry_on_service_exceptions"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "action": action,
            "iam_resources": iam_resources,
            "region": region,
            "service": service,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if outputs is not None:
            self._values["outputs"] = outputs
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if additional_iam_statements is not None:
            self._values["additional_iam_statements"] = additional_iam_statements
        if endpoint is not None:
            self._values["endpoint"] = endpoint
        if iam_action is not None:
            self._values["iam_action"] = iam_action
        if parameters is not None:
            self._values["parameters"] = parameters
        if retry_on_service_exceptions is not None:
            self._values["retry_on_service_exceptions"] = retry_on_service_exceptions

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def action(self) -> builtins.str:
        '''The API action to call.

        Use camelCase.
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def iam_resources(self) -> typing.List[builtins.str]:
        '''The resources for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.'''
        result = self._values.get("iam_resources")
        assert result is not None, "Required property 'iam_resources' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def region(self) -> builtins.str:
        '''The AWS region to call this AWS API for.

        Example::

            "us-east-1"
        '''
        result = self._values.get("region")
        assert result is not None, "Required property 'region' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''The AWS service to call in AWS SDK for JavaScript v3 format.

        :see: https://docs.aws.amazon.com/AWSJavaScriptSDK/v3/latest/

        Example::

            "s3"
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def additional_iam_statements(
        self,
    ) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        '''Additional IAM statements that will be added to the state machine role's policy.

        Use in the case where the call requires more than a single statement to
        be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions
        to read the object on which it must act.

        :default: - no additional statements are added
        '''
        result = self._values.get("additional_iam_statements")
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], result)

    @builtins.property
    def endpoint(self) -> typing.Optional[builtins.str]:
        '''The AWS API endpoint.

        :default: Do not override API endpoint.
        '''
        result = self._values.get("endpoint")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iam_action(self) -> typing.Optional[builtins.str]:
        '''The action for the IAM statement that will be added to the Lambda function role's policy to allow the state machine to make the API call.

        By default the action for this IAM statement will be ``service:action``.

        Use in the case where the IAM action name does not match with the
        API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``.

        :default: - service:action
        '''
        result = self._values.get("iam_action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Parameters for the API action call in AWS SDK for JavaScript v3 format.

        :default: - no parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def retry_on_service_exceptions(self) -> typing.Optional[builtins.bool]:
        '''Whether to retry on the backend Lambda service exceptions.

        This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException``,
        ``Lambda.SdkClientException``, and ``Lambda.ClientExecutionTimeoutException``
        with an interval of 2 seconds, a back-off rate
        of 2 and 6 maximum attempts.

        :default: true

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/bp-lambda-serviceexception.html
        '''
        result = self._values.get("retry_on_service_exceptions")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallAwsServiceCrossRegionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallAwsServiceJsonPathProps",
    jsii_struct_bases=[_TaskStateJsonPathBaseProps_57dc2260],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "action": "action",
        "iam_resources": "iamResources",
        "service": "service",
        "additional_iam_statements": "additionalIamStatements",
        "iam_action": "iamAction",
        "parameters": "parameters",
    },
)
class CallAwsServiceJsonPathProps(_TaskStateJsonPathBaseProps_57dc2260):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''Properties for calling an AWS service's API action using JSONPath from your state machine.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``.
        :param service: The AWS service to call.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param iam_action: The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call. Use PascalCase for the parameter names. Default: - no parameters

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_iam as iam
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # parameters: Any
            # policy_statement: iam.PolicyStatement
            # result_selector: Any
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            call_aws_service_json_path_props = stepfunctions_tasks.CallAwsServiceJsonPathProps(
                action="action",
                iam_resources=["iamResources"],
                service="service",
            
                # the properties below are optional
                additional_iam_statements=[policy_statement],
                assign={
                    "assign_key": assign
                },
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                iam_action="iamAction",
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output_path="outputPath",
                parameters={
                    "parameters_key": parameters
                },
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                result_path="resultPath",
                result_selector={
                    "result_selector_key": result_selector
                },
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3be9744b4f22c18ce3ad5036bb359e831ae4599acdc0d4971da04bb05fd3ec3d)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument iam_resources", value=iam_resources, expected_type=type_hints["iam_resources"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument additional_iam_statements", value=additional_iam_statements, expected_type=type_hints["additional_iam_statements"])
            check_type(argname="argument iam_action", value=iam_action, expected_type=type_hints["iam_action"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "action": action,
            "iam_resources": iam_resources,
            "service": service,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if additional_iam_statements is not None:
            self._values["additional_iam_statements"] = additional_iam_statements
        if iam_action is not None:
            self._values["iam_action"] = iam_action
        if parameters is not None:
            self._values["parameters"] = parameters

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def action(self) -> builtins.str:
        '''The API action to call.

        Use camelCase.
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def iam_resources(self) -> typing.List[builtins.str]:
        '''The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call.

        By default the action for this IAM statement will be ``service:action``.
        '''
        result = self._values.get("iam_resources")
        assert result is not None, "Required property 'iam_resources' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def service(self) -> builtins.str:
        '''The AWS service to call.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def additional_iam_statements(
        self,
    ) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        '''Additional IAM statements that will be added to the state machine role's policy.

        Use in the case where the call requires more than a single statement to
        be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions
        to read the object on which it must act.

        :default: - no additional statements are added
        '''
        result = self._values.get("additional_iam_statements")
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], result)

    @builtins.property
    def iam_action(self) -> typing.Optional[builtins.str]:
        '''The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call.

        Use in the case where the IAM action name does not match with the
        API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``.

        :default: - service:action
        '''
        result = self._values.get("iam_action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Parameters for the API action call.

        Use PascalCase for the parameter names.

        :default: - no parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallAwsServiceJsonPathProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallAwsServiceJsonataProps",
    jsii_struct_bases=[_TaskStateJsonataBaseProps_0e91775c],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "outputs": "outputs",
        "action": "action",
        "iam_resources": "iamResources",
        "service": "service",
        "additional_iam_statements": "additionalIamStatements",
        "iam_action": "iamAction",
        "parameters": "parameters",
    },
)
class CallAwsServiceJsonataProps(_TaskStateJsonataBaseProps_0e91775c):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''Properties for calling an AWS service's API action using JSONata from your state machine.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``.
        :param service: The AWS service to call.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param iam_action: The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call. Use PascalCase for the parameter names. Default: - no parameters

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_iam as iam
            from aws_cdk import aws_stepfunctions as stepfunctions
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # assign: Any
            # outputs: Any
            # parameters: Any
            # policy_statement: iam.PolicyStatement
            # task_role: stepfunctions.TaskRole
            # timeout: stepfunctions.Timeout
            
            call_aws_service_jsonata_props = stepfunctions_tasks.CallAwsServiceJsonataProps(
                action="action",
                iam_resources=["iamResources"],
                service="service",
            
                # the properties below are optional
                additional_iam_statements=[policy_statement],
                assign={
                    "assign_key": assign
                },
                comment="comment",
                credentials=stepfunctions.Credentials(
                    role=task_role
                ),
                heartbeat=cdk.Duration.minutes(30),
                heartbeat_timeout=timeout,
                iam_action="iamAction",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                outputs=outputs,
                parameters={
                    "parameters_key": parameters
                },
                query_language=stepfunctions.QueryLanguage.JSON_PATH,
                state_name="stateName",
                task_timeout=timeout,
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__47fb3cba2445f058ea8721a16841b05acab1c99a6ac195660d69cc2513f6e66d)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument iam_resources", value=iam_resources, expected_type=type_hints["iam_resources"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument additional_iam_statements", value=additional_iam_statements, expected_type=type_hints["additional_iam_statements"])
            check_type(argname="argument iam_action", value=iam_action, expected_type=type_hints["iam_action"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "action": action,
            "iam_resources": iam_resources,
            "service": service,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if outputs is not None:
            self._values["outputs"] = outputs
        if additional_iam_statements is not None:
            self._values["additional_iam_statements"] = additional_iam_statements
        if iam_action is not None:
            self._values["iam_action"] = iam_action
        if parameters is not None:
            self._values["parameters"] = parameters

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def action(self) -> builtins.str:
        '''The API action to call.

        Use camelCase.
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def iam_resources(self) -> typing.List[builtins.str]:
        '''The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call.

        By default the action for this IAM statement will be ``service:action``.
        '''
        result = self._values.get("iam_resources")
        assert result is not None, "Required property 'iam_resources' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def service(self) -> builtins.str:
        '''The AWS service to call.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def additional_iam_statements(
        self,
    ) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        '''Additional IAM statements that will be added to the state machine role's policy.

        Use in the case where the call requires more than a single statement to
        be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions
        to read the object on which it must act.

        :default: - no additional statements are added
        '''
        result = self._values.get("additional_iam_statements")
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], result)

    @builtins.property
    def iam_action(self) -> typing.Optional[builtins.str]:
        '''The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call.

        Use in the case where the IAM action name does not match with the
        API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``.

        :default: - service:action
        '''
        result = self._values.get("iam_action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Parameters for the API action call.

        Use PascalCase for the parameter names.

        :default: - no parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallAwsServiceJsonataProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CallAwsServiceProps",
    jsii_struct_bases=[_TaskStateBaseProps_3a62b6d0],
    name_mapping={
        "comment": "comment",
        "query_language": "queryLanguage",
        "state_name": "stateName",
        "credentials": "credentials",
        "heartbeat": "heartbeat",
        "heartbeat_timeout": "heartbeatTimeout",
        "integration_pattern": "integrationPattern",
        "task_timeout": "taskTimeout",
        "timeout": "timeout",
        "assign": "assign",
        "input_path": "inputPath",
        "output_path": "outputPath",
        "outputs": "outputs",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "action": "action",
        "iam_resources": "iamResources",
        "service": "service",
        "additional_iam_statements": "additionalIamStatements",
        "iam_action": "iamAction",
        "parameters": "parameters",
    },
)
class CallAwsServiceProps(_TaskStateBaseProps_3a62b6d0):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        service: builtins.str,
        additional_iam_statements: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''Properties for calling an AWS service's API action from your state machine.

        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param action: The API action to call. Use camelCase.
        :param iam_resources: The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``.
        :param service: The AWS service to call.
        :param additional_iam_statements: Additional IAM statements that will be added to the state machine role's policy. Use in the case where the call requires more than a single statement to be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions to read the object on which it must act. Default: - no additional statements are added
        :param iam_action: The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: Parameters for the API action call. Use PascalCase for the parameter names. Default: - no parameters

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html
        :exampleMetadata: infused

        Example::

            # my_bucket: s3.Bucket
            
            get_object = tasks.CallAwsService(self, "GetObject",
                service="s3",
                action="getObject",
                parameters={
                    "Bucket": my_bucket.bucket_name,
                    "Key": sfn.JsonPath.string_at("$.key")
                },
                iam_resources=[my_bucket.arn_for_objects("*")]
            )
        '''
        if isinstance(credentials, dict):
            credentials = _Credentials_2cd64c6b(**credentials)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__63c577047b5ec17a682dc869c299af67611178c845f52baceb7e79927eee3db3)
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument query_language", value=query_language, expected_type=type_hints["query_language"])
            check_type(argname="argument state_name", value=state_name, expected_type=type_hints["state_name"])
            check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
            check_type(argname="argument heartbeat", value=heartbeat, expected_type=type_hints["heartbeat"])
            check_type(argname="argument heartbeat_timeout", value=heartbeat_timeout, expected_type=type_hints["heartbeat_timeout"])
            check_type(argname="argument integration_pattern", value=integration_pattern, expected_type=type_hints["integration_pattern"])
            check_type(argname="argument task_timeout", value=task_timeout, expected_type=type_hints["task_timeout"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument assign", value=assign, expected_type=type_hints["assign"])
            check_type(argname="argument input_path", value=input_path, expected_type=type_hints["input_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument result_path", value=result_path, expected_type=type_hints["result_path"])
            check_type(argname="argument result_selector", value=result_selector, expected_type=type_hints["result_selector"])
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument iam_resources", value=iam_resources, expected_type=type_hints["iam_resources"])
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument additional_iam_statements", value=additional_iam_statements, expected_type=type_hints["additional_iam_statements"])
            check_type(argname="argument iam_action", value=iam_action, expected_type=type_hints["iam_action"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "action": action,
            "iam_resources": iam_resources,
            "service": service,
        }
        if comment is not None:
            self._values["comment"] = comment
        if query_language is not None:
            self._values["query_language"] = query_language
        if state_name is not None:
            self._values["state_name"] = state_name
        if credentials is not None:
            self._values["credentials"] = credentials
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if heartbeat_timeout is not None:
            self._values["heartbeat_timeout"] = heartbeat_timeout
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if task_timeout is not None:
            self._values["task_timeout"] = task_timeout
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign is not None:
            self._values["assign"] = assign
        if input_path is not None:
            self._values["input_path"] = input_path
        if output_path is not None:
            self._values["output_path"] = output_path
        if outputs is not None:
            self._values["outputs"] = outputs
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if additional_iam_statements is not None:
            self._values["additional_iam_statements"] = additional_iam_statements
        if iam_action is not None:
            self._values["iam_action"] = iam_action
        if parameters is not None:
            self._values["parameters"] = parameters

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment describing this state.

        :default: No comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_language(self) -> typing.Optional[_QueryLanguage_be8414a8]:
        '''The name of the query language used by the state.

        If the state does not contain a ``queryLanguage`` field,
        then it will use the query language specified in the top-level ``queryLanguage`` field.

        :default: - JSONPath
        '''
        result = self._values.get("query_language")
        return typing.cast(typing.Optional[_QueryLanguage_be8414a8], result)

    @builtins.property
    def state_name(self) -> typing.Optional[builtins.str]:
        '''Optional name for this state.

        :default: - The construct ID will be used as state name
        '''
        result = self._values.get("state_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def credentials(self) -> typing.Optional[_Credentials_2cd64c6b]:
        '''Credentials for an IAM Role that the State Machine assumes for executing the task.

        This enables cross-account resource invocations.

        :default: - None (Task is executed using the State Machine's execution role)

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-access-cross-acct-resources.html
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_Credentials_2cd64c6b], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the heartbeat.

        :default: - None

        :deprecated: use ``heartbeatTimeout``

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def heartbeat_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the heartbeat.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("heartbeat_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_949291bc]:
        '''AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns.

        Depending on the AWS Service, the Service Integration Pattern availability will vary.

        :default:

        - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks.
        ``IntegrationPattern.RUN_JOB`` for the following exceptions:
        ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-supported-services.html
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_949291bc], result)

    @builtins.property
    def task_timeout(self) -> typing.Optional[_Timeout_d7c10551]:
        '''Timeout for the task.

        [disable-awslint:duration-prop-type] is needed because all props interface in
        aws-stepfunctions-tasks extend this interface

        :default: - None
        '''
        result = self._values.get("task_timeout")
        return typing.cast(typing.Optional[_Timeout_d7c10551], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''(deprecated) Timeout for the task.

        :default: - None

        :deprecated: use ``taskTimeout``

        :stability: deprecated
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def assign(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Workflow variables to store in this step.

        Using workflow variables, you can store data in a step and retrieve that data in future steps.

        :default: - Not assign variables

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/workflow-variables.html
        '''
        result = self._values.get("assign")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: $
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to select part of the state to be the output to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default: $
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Any:
        '''Used to specify and transform output from the state.

        When specified, the value overrides the state output default.
        The output field accepts any JSON value (object, array, string, number, boolean, null).
        Any string value, including those inside objects or arrays,
        will be evaluated as JSONata if surrounded by {% %} characters.
        Output also accepts a JSONata expression directly.

        :default: - $states.result or $states.errorOutput

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Any, result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: $
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def action(self) -> builtins.str:
        '''The API action to call.

        Use camelCase.
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def iam_resources(self) -> typing.List[builtins.str]:
        '''The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call.

        By default the action for this IAM statement will be ``service:action``.
        '''
        result = self._values.get("iam_resources")
        assert result is not None, "Required property 'iam_resources' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def service(self) -> builtins.str:
        '''The AWS service to call.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def additional_iam_statements(
        self,
    ) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        '''Additional IAM statements that will be added to the state machine role's policy.

        Use in the case where the call requires more than a single statement to
        be executed, e.g. ``rekognition:detectLabels`` requires also S3 permissions
        to read the object on which it must act.

        :default: - no additional statements are added
        '''
        result = self._values.get("additional_iam_statements")
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], result)

    @builtins.property
    def iam_action(self) -> typing.Optional[builtins.str]:
        '''The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call.

        Use in the case where the IAM action name does not match with the
        API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``.

        :default: - service:action
        '''
        result = self._values.get("iam_action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''Parameters for the API action call.

        Use PascalCase for the parameter names.

        :default: - no parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallAwsServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.Channel",
    jsii_struct_bases=[],
    name_mapping={
        "channel_name": "channelName",
        "data_source": "dataSource",
        "compression_type": "compressionType",
        "content_type": "contentType",
        "input_mode": "inputMode",
        "record_wrapper_type": "recordWrapperType",
        "shuffle_config": "shuffleConfig",
    },
)
class Channel:
    def __init__(
        self,
        *,
        channel_name: builtins.str,
        data_source: typing.Union["DataSource", typing.Dict[builtins.str, typing.Any]],
        compression_type: typing.Optional["CompressionType"] = None,
        content_type: typing.Optional[builtins.str] = None,
        input_mode: typing.Optional["InputMode"] = None,
        record_wrapper_type: typing.Optional["RecordWrapperType"] = None,
        shuffle_config: typing.Optional[typing.Union["ShuffleConfig", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Describes the training, validation or test dataset and the Amazon S3 location where it is stored.

        :param channel_name: Name of the channel.
        :param data_source: Location of the channel data.
        :param compression_type: Compression type if training data is compressed. Default: - None
        :param content_type: The MIME type of the data. Default: - None
        :param input_mode: Input mode to use for the data channel in a training job. Default: - None
        :param record_wrapper_type: Specify RecordIO as the value when input data is in raw format but the training algorithm requires the RecordIO format. In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record. If the input data is already in RecordIO format, you don't need to set this attribute. Default: - None
        :param shuffle_config: Shuffle config option for input data in a channel. Default: - None

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # s3_location: stepfunctions_tasks.S3Location
            
            channel = stepfunctions_tasks.Channel(
                channel_name="channelName",
                data_source=stepfunctions_tasks.DataSource(
                    s3_data_source=stepfunctions_tasks.S3DataSource(
                        s3_location=s3_location,
            
                        # the properties below are optional
                        attribute_names=["attributeNames"],
                        s3_data_distribution_type=stepfunctions_tasks.S3DataDistributionType.FULLY_REPLICATED,
                        s3_data_type=stepfunctions_tasks.S3DataType.MANIFEST_FILE
                    )
                ),
            
                # the properties below are optional
                compression_type=stepfunctions_tasks.CompressionType.NONE,
                content_type="contentType",
                input_mode=stepfunctions_tasks.InputMode.PIPE,
                record_wrapper_type=stepfunctions_tasks.RecordWrapperType.NONE,
                shuffle_config=stepfunctions_tasks.ShuffleConfig(
                    seed=123
                )
            )
        '''
        if isinstance(data_source, dict):
            data_source = DataSource(**data_source)
        if isinstance(shuffle_config, dict):
            shuffle_config = ShuffleConfig(**shuffle_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fde65c1648641b68064c5fde3aa4a36cb79bd09822197ea0306c1b558328822e)
            check_type(argname="argument channel_name", value=channel_name, expected_type=type_hints["channel_name"])
            check_type(argname="argument data_source", value=data_source, expected_type=type_hints["data_source"])
            check_type(argname="argument compression_type", value=compression_type, expected_type=type_hints["compression_type"])
            check_type(argname="argument content_type", value=content_type, expected_type=type_hints["content_type"])
            check_type(argname="argument input_mode", value=input_mode, expected_type=type_hints["input_mode"])
            check_type(argname="argument record_wrapper_type", value=record_wrapper_type, expected_type=type_hints["record_wrapper_type"])
            check_type(argname="argument shuffle_config", value=shuffle_config, expected_type=type_hints["shuffle_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "channel_name": channel_name,
            "data_source": data_source,
        }
        if compression_type is not None:
            self._values["compression_type"] = compression_type
        if content_type is not None:
            self._values["content_type"] = content_type
        if input_mode is not None:
            self._values["input_mode"] = input_mode
        if record_wrapper_type is not None:
            self._values["record_wrapper_type"] = record_wrapper_type
        if shuffle_config is not None:
            self._values["shuffle_config"] = shuffle_config

    @builtins.property
    def channel_name(self) -> builtins.str:
        '''Name of the channel.'''
        result = self._values.get("channel_name")
        assert result is not None, "Required property 'channel_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def data_source(self) -> "DataSource":
        '''Location of the channel data.'''
        result = self._values.get("data_source")
        assert result is not None, "Required property 'data_source' is missing"
        return typing.cast("DataSource", result)

    @builtins.property
    def compression_type(self) -> typing.Optional["CompressionType"]:
        '''Compression type if training data is compressed.

        :default: - None
        '''
        result = self._values.get("compression_type")
        return typing.cast(typing.Optional["CompressionType"], result)

    @builtins.property
    def content_type(self) -> typing.Optional[builtins.str]:
        '''The MIME type of the data.

        :default: - None
        '''
        result = self._values.get("content_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_mode(self) -> typing.Optional["InputMode"]:
        '''Input mode to use for the data channel in a training job.

        :default: - None
        '''
        result = self._values.get("input_mode")
        return typing.cast(typing.Optional["InputMode"], result)

    @builtins.property
    def record_wrapper_type(self) -> typing.Optional["RecordWrapperType"]:
        '''Specify RecordIO as the value when input data is in raw format but the training algorithm requires the RecordIO format.

        In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record.
        If the input data is already in RecordIO format, you don't need to set this attribute.

        :default: - None
        '''
        result = self._values.get("record_wrapper_type")
        return typing.cast(typing.Optional["RecordWrapperType"], result)

    @builtins.property
    def shuffle_config(self) -> typing.Optional["ShuffleConfig"]:
        '''Shuffle config option for input data in a channel.

        :default: - None
        '''
        result = self._values.get("shuffle_config")
        return typing.cast(typing.Optional["ShuffleConfig"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Channel(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Classification(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.Classification",
):
    '''The classification within a EMR Containers application configuration.

    Class can be extended to add other classifications.
    For example, new Classification('xxx-yyy');

    :exampleMetadata: infused

    Example::

        tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
            virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
            release_label=tasks.ReleaseLabel.EMR_6_2_0,
            job_name="EMR-Containers-Job",
            job_driver=tasks.JobDriver(
                spark_submit_job_driver=tasks.SparkSubmitJobDriver(
                    entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
                )
            ),
            application_config=[tasks.ApplicationConfiguration(
                classification=tasks.Classification.SPARK_DEFAULTS,
                properties={
                    "spark.executor.instances": "1",
                    "spark.executor.memory": "512M"
                }
            )]
        )
    '''

    def __init__(self, classification_statement: builtins.str) -> None:
        '''Creates a new Classification.

        :param classification_statement: A literal string in case a new EMR classification is released, if not already defined.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4aa3125a418732ec7b4985c5191ef4461b33928aff66ff8ba99daa7508d3b9c1)
            check_type(argname="argument classification_statement", value=classification_statement, expected_type=type_hints["classification_statement"])
        jsii.create(self.__class__, self, [classification_statement])

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK")
    def SPARK(cls) -> "Classification":
        '''Sets the maximizeResourceAllocation property to true or false.

        When true, Amazon EMR automatically configures spark-defaults properties based on cluster hardware configuration.

        For more info:

        :see: https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-spark-configure.html#emr-spark-maximizeresourceallocation
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK_DEFAULTS")
    def SPARK_DEFAULTS(cls) -> "Classification":
        '''Sets values in the spark-defaults.conf file.

        For more info:

        :see: https://spark.apache.org/docs/latest/configuration.html
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_DEFAULTS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK_ENV")
    def SPARK_ENV(cls) -> "Classification":
        '''Sets values in the spark-env.sh file.

        For more info:

        :see: https://spark.apache.org/docs/latest/configuration.html#environment-variables
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_ENV"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK_HIVE_SITE")
    def SPARK_HIVE_SITE(cls) -> "Classification":
        '''Sets values in the hive-site.xml for Spark.'''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_HIVE_SITE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK_LOG4J")
    def SPARK_LOG4_J(cls) -> "Classification":
        '''Sets values in the log4j.properties file.

        For more settings and info:

        :see: https://github.com/apache/spark/blob/master/conf/log4j.properties.template
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_LOG4J"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="SPARK_METRICS")
    def SPARK_METRICS(cls) -> "Classification":
        '''Sets values in the metrics.properties file.

        For more settings and info:

        :see: https://github.com/apache/spark/blob/master/conf/metrics.properties.template
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_METRICS"))

    @builtins.property
    @jsii.member(jsii_name="classificationStatement")
    def classification_statement(self) -> builtins.str:
        '''A literal string in case a new EMR classification is released, if not already defined.'''
        return typing.cast(builtins.str, jsii.get(self, "classificationStatement"))


class CodeBuildStartBuild(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CodeBuildStartBuild",
):
    '''Start a CodeBuild Build as a task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-codebuild.html
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_codebuild as codebuild
        
        
        codebuild_project = codebuild.Project(self, "Project",
            project_name="MyTestProject",
            build_spec=codebuild.BuildSpec.from_object({
                "version": "0.2",
                "phases": {
                    "build": {
                        "commands": ["echo \"Hello, CodeBuild!\""
                        ]
                    }
                }
            })
        )
        
        task = tasks.CodeBuildStartBuild(self, "Task",
            project=codebuild_project,
            integration_pattern=sfn.IntegrationPattern.RUN_JOB,
            environment_variables_override={
                "ZONE": codebuild.BuildEnvironmentVariable(
                    type=codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value=sfn.JsonPath.string_at("$.envVariables.zone")
                )
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        project: _IProject_aafae30a,
        environment_variables_override: typing.Optional[typing.Mapping[builtins.str, typing.Union[_BuildEnvironmentVariable_7df4fa0c, typing.Dict[builtins.str, typing.Any]]]] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param project: CodeBuild project to start.
        :param environment_variables_override: A set of environment variables to be used for this build only. Default: - the latest environment variables already defined in the build project.
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c7420ddc3c5f914d45be1248775c7909c2f866fec2613b9197535be328a08761)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CodeBuildStartBuildProps(
            project=project,
            environment_variables_override=environment_variables_override,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        project: _IProject_aafae30a,
        environment_variables_override: typing.Optional[typing.Mapping[builtins.str, typing.Union[_BuildEnvironmentVariable_7df4fa0c, typing.Dict[builtins.str, typing.Any]]]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        outputs: typing.Any = None,
    ) -> "CodeBuildStartBuild":
        '''Start a CodeBuild Build as a task using JSONata.

        :param scope: -
        :param id: -
        :param project: CodeBuild project to start.
        :param environment_variables_override: A set of environment variables to be used for this build only. Default: - the latest environment variables already defined in the build project.
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__67c314959994800b5b80769f85e9c4a26ce8a9bad722a9daf2ca43be08faf2ef)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CodeBuildStartBuildJsonataProps(
            project=project,
            environment_variables_override=environment_variables_override,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            outputs=outputs,
        )

        return typing.cast("CodeBuildStartBuild", jsii.sinvoke(cls, "jsonata", [scope, id, props]))

    @jsii.member(jsii_name="jsonPath")
    @builtins.classmethod
    def json_path(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        project: _IProject_aafae30a,
        environment_variables_override: typing.Optional[typing.Mapping[builtins.str, typing.Union[_BuildEnvironmentVariable_7df4fa0c, typing.Dict[builtins.str, typing.Any]]]] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> "CodeBuildStartBuild":
        '''Start a CodeBuild Build as a task using JSONPath.

        :param scope: -
        :param id: -
        :param project: CodeBuild project to start.
        :param environment_variables_override: A set of environment variables to be used for this build only. Default: - the latest environment variables already defined in the build project.
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__218c1eecb18b0d886ba11e9c23837121839a1240154060b7f528b811ef975ee9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CodeBuildStartBuildJsonPathProps(
            project=project,
            environment_variables_override=environment_variables_override,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
        )

        return typing.cast("CodeBuildStartBuild", jsii.sinvoke(cls, "jsonPath", [scope, id, props]))

    @builtins.property
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_32ea9403]:
        return typing.cast(typing.Optional[_TaskMetricsConfig_32ea9403], jsii.get(self, "taskMetrics"))

    @builtins.property
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], jsii.get(self, "taskPolicies"))


class CodeBuildStartBuildBatch(
    _TaskStateBase_b5c0a816,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_stepfunctions_tasks.CodeBuildStartBuildBatch",
):
    '''Start a CodeBuild BatchBuild as a task.

    :see: https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuildBatch.html
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_codebuild as codebuild
        
        
        project = codebuild.Project(self, "Project",
            project_name="MyTestProject",
            build_spec=codebuild.BuildSpec.from_object_to_yaml({
                "version": 0.2,
                "batch": {
                    "build-list": [{
                        "identifier": "id",
                        "buildspec": "version: 0.2\nphases:\n  build:\n    commands:\n      - echo \"Hello, from small!\""
                    }
                    ]
                }
            })
        )
        project.enable_batch_builds()
        
        task = tasks.CodeBuildStartBuildBatch(self, "buildBatchTask",
            project=project,
            integration_pattern=sfn.IntegrationPattern.REQUEST_RESPONSE,
            environment_variables_override={
                "test": codebuild.BuildEnvironmentVariable(
                    type=codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value="testValue"
                )
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        project: _IProject_aafae30a,
        environment_variables_override: typing.Optional[typing.Mapping[builtins.str, typing.Union[_BuildEnvironmentVariable_7df4fa0c, typing.Dict[builtins.str, typing.Any]]]] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_949291bc] = None,
        task_timeout: typing.Optional[_Timeout_d7c10551] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        assign: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        input_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        outputs: typing.Any = None,
    ) -> None:
        '''
        :param scope: -
        :param id: Descriptive identifier for this chainable.
        :param project: CodeBuild project to start.
        :param environment_variables_override: A set of environment variables to be used for this build only. Default: - the latest environment variables already defined in the build project.
        :param result_path: JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: $
        :param result_selector: The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param comment: A comment describing this state. Default: No comment
        :param query_language: The name of the query language used by the state. If the state does not contain a ``queryLanguage`` field, then it will use the query language specified in the top-level ``queryLanguage`` field. Default: - JSONPath
        :param state_name: Optional name for this state. Default: - The construct ID will be used as state name
        :param credentials: Credentials for an IAM Role that the State Machine assumes for executing the task. This enables cross-account resource invocations. Default: - None (Task is executed using the State Machine's execution role)
        :param heartbeat: (deprecated) Timeout for the heartbeat. Default: - None
        :param heartbeat_timeout: Timeout for the heartbeat. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param integration_pattern: AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns. Depending on the AWS Service, the Service Integration Pattern availability will vary. Default: - ``IntegrationPattern.REQUEST_RESPONSE`` for most tasks. ``IntegrationPattern.RUN_JOB`` for the following exceptions: ``BatchSubmitJob``, ``EmrAddStep``, ``EmrCreateCluster``, ``EmrTerminationCluster``, and ``EmrContainersStartJobRun``.
        :param task_timeout: Timeout for the task. [disable-awslint:duration-prop-type] is needed because all props interface in aws-stepfunctions-tasks extend this interface Default: - None
        :param timeout: (deprecated) Timeout for the task. Default: - None
        :param assign: Workflow variables to store in this step. Using workflow variables, you can store data in a step and retrieve that data in future steps. Default: - Not assign variables
        :param input_path: JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: $
        :param output_path: JSONPath expression to select part of the state to be the output to this state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: $
        :param outputs: Used to specify and transform output from the state. When specified, the value overrides the state output default. The output field accepts any JSON value (object, array, string, number, boolean, null). Any string value, including those inside objects or arrays, will be evaluated as JSONata if surrounded by {% %} characters. Output also accepts a JSONata expression directly. Default: - $states.result or $states.errorOutput
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b14d418319c1b8a8fec4d9dc15efe8cd53bf6f18ce163c7aba19e3a7b934dd8f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CodeBuildStartBuildBatchProps(
            project=project,
            environment_variables_override=environment_variables_override,
            result_path=result_path,
            result_selector=result_selector,
            comment=comment,
            query_language=query_language,
            state_name=state_name,
            credentials=credentials,
            heartbeat=heartbeat,
            heartbeat_timeout=heartbeat_timeout,
            integration_pattern=integration_pattern,
            task_timeout=task_timeout,
            timeout=timeout,
            assign=assign,
            input_path=input_path,
            output_path=output_path,
            outputs=outputs,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="jsonata")
    @builtins.classmethod
    def jsonata(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        project: _IProject_aafae30a,
        environment_variables_override: typing.Optional[typing.Mapping[builtins.str, typing.Union[_BuildEnvironmentVariable_7df4fa0c, typing.Dict[builtins.str, typing.Any]]]] = None,
        comment: typing.Optional[builtins.str] = None,
        query_language: typing.Optional[_QueryLanguage_be8414a8] = None,
        state_name: typing.Optional[builtins.str] = None,
        credentials: typing.Optional[typing.Union[_Credentials_2cd64c6b, typing.Dict[builtins.str, typing.Any]]] = None,
        heartbeat: typing.Optional[_Duration_4839e8c3] = None,
        heartbeat_timeout: typing.Optional[_Timeout_d7c10551] = None,
        integration_pattern: typ