r'''
# AWS Lambda Construct Library

This construct library allows you to define AWS Lambda Functions.

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)
```

When deployed, this construct creates or updates an existing
`AWS::Lambda::Function` resource. When updating, AWS CloudFormation calls the
[UpdateFunctionConfiguration](https://docs.aws.amazon.com/lambda/latest/api/API_UpdateFunctionConfiguration.html)
and [UpdateFunctionCode](https://docs.aws.amazon.com/lambda/latest/api/API_UpdateFunctionCode.html)
Lambda APIs under the hood. Because these calls happen sequentially, and
invocations can happen between these calls, your function may encounter errors
in the time between the calls. For example, if you update an existing Lambda
function by removing an environment variable and the code that references that
environment variable in the same CDK deployment, you may see invocation errors
related to a missing environment variable. To work around this, you can invoke
your function against a version or alias by default, rather than the `$LATEST`
version.

To further mitigate these issues, you can ensure consistency between your function code and infrastructure configuration by defining environment variables as a single source of truth in your CDK stack. You can define them in a separate `env.ts` file and reference them in both your handler and CDK configuration. This approach allows you to catch errors at compile time, benefit from improved IDE support, minimize the risk of mismatched configurations, and enhance maintainability.

## Handler Code

The `lambda.Code` class includes static convenience methods for various types of
runtime code.

* `lambda.Code.fromBucket(bucket, key, objectVersion)` - specify an S3 object
  that contains the archive of your runtime code.
* `lambda.Code.fromBucketV2(bucket, key, {objectVersion: version, sourceKMSKey: key})` - specify an S3 object
  that contains the archive of your runtime code.

```python
from aws_cdk.aws_kms import Key
import aws_cdk.aws_s3 as s3
# key: Key


bucket = s3.Bucket(self, "Bucket")

options = {
    "source_kMSKey": key
}
fn_bucket = lambda_.Function(self, "myFunction2",
    runtime=lambda_.Runtime.NODEJS_LATEST,
    handler="index.handler",
    code=lambda_.Code.from_bucket_v2(bucket, "python-lambda-handler.zip", options)
)
```

* `lambda.Code.fromInline(code)` - inline the handle code as a string. This is
  limited to supported runtimes.
* `lambda.Code.fromAsset(path)` - specify a directory or a .zip file in the local
  filesystem which will be zipped and uploaded to S3 before deployment. See also
  [bundling asset code](#bundling-asset-code).
* `lambda.Code.fromDockerBuild(path, options)` - use the result of a Docker
  build as code. The runtime code is expected to be located at `/asset` in the
  image and will be zipped and uploaded to S3 as an asset.
* `lambda.Code.fromCustomCommand(output, command, customCommandOptions)` -
  supply a command that is invoked during cdk synth. That command is meant to direct
  the generated code to output (a zip file or a directory), which is then used as the
  code for the created AWS Lambda.

The following example shows how to define a Python function and deploy the code
from the local directory `my-lambda-handler` to it:

```python
lambda_.Function(self, "MyLambda",
    code=lambda_.Code.from_asset(path.join(__dirname, "my-lambda-handler")),
    handler="index.main",
    runtime=lambda_.Runtime.PYTHON_3_9
)
```

When deploying a stack that contains this code, the directory will be zip
archived and then uploaded to an S3 bucket, then the exact location of the S3
objects will be passed when the stack is deployed.

During synthesis, the CDK expects to find a directory on disk at the asset
directory specified. Note that we are referencing the asset directory relatively
to our CDK project directory. This is especially important when we want to share
this construct through a library. Different programming languages will have
different techniques for bundling resources into libraries.

## Docker Images

Lambda functions allow specifying their handlers within docker images. The docker
image can be an image from ECR or a local asset that the CDK will package and load
into ECR.

The following `DockerImageFunction` construct uses a local folder with a
Dockerfile as the asset that will be used as the function handler.

```python
lambda_.DockerImageFunction(self, "AssetFunction",
    code=lambda_.DockerImageCode.from_image_asset(path.join(__dirname, "docker-handler"))
)
```

You can also specify an image that already exists in ECR as the function handler.

```python
import aws_cdk.aws_ecr as ecr

repo = ecr.Repository(self, "Repository")

lambda_.DockerImageFunction(self, "ECRFunction",
    code=lambda_.DockerImageCode.from_ecr(repo)
)
```

The props for these docker image resources allow overriding the image's `CMD`, `ENTRYPOINT`, and `WORKDIR`
configurations as well as choosing a specific tag or digest. See their docs for more information.

To deploy a `DockerImageFunction` on Lambda `arm64` architecture, specify `Architecture.ARM_64` in `architecture`.
This will bundle docker image assets for `arm64` architecture with `--platform linux/arm64` even if build within an `x86_64` host.

With that being said, if you are bundling `DockerImageFunction` for Lambda `amd64` architecture from a `arm64` machine like a Macbook with `arm64` CPU, you would
need to specify `architecture: lambda.Architecture.X86_64` as well. This ensures the `--platform` argument is passed to the image assets
bundling process so you can bundle up `X86_64` images from the `arm64` machine.

```python
lambda_.DockerImageFunction(self, "AssetFunction",
    code=lambda_.DockerImageCode.from_image_asset(path.join(__dirname, "docker-arm64-handler")),
    architecture=lambda_.Architecture.ARM_64
)
```

## Execution Role

Lambda functions assume an IAM role during execution. In CDK by default, Lambda
functions will use an autogenerated Role if one is not provided.

The autogenerated Role is automatically given permissions to execute the Lambda
function. To reference the autogenerated Role:

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)

role = fn.role
```

You can also provide your own IAM role. Provided IAM roles will not automatically
be given permissions to execute the Lambda function. To provide a role and grant
it appropriate permissions:

```python
my_role = iam.Role(self, "My Role",
    assumed_by=iam.ServicePrincipal("lambda.amazonaws.com")
)

fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    role=my_role
)

my_role.add_managed_policy(iam.ManagedPolicy.from_aws_managed_policy_name("service-role/AWSLambdaBasicExecutionRole"))
my_role.add_managed_policy(iam.ManagedPolicy.from_aws_managed_policy_name("service-role/AWSLambdaVPCAccessExecutionRole"))
```

## Function Timeout

AWS Lambda functions have a default timeout of 3 seconds, but this can be increased
up to 15 minutes. The timeout is available as a property of `Function` so that
you can reference it elsewhere in your stack. For instance, you could use it to create
a CloudWatch alarm to report when your function timed out:

```python
import aws_cdk as cdk
import aws_cdk.aws_cloudwatch as cloudwatch


fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    timeout=Duration.minutes(5)
)

if fn.timeout:
    cloudwatch.Alarm(self, "MyAlarm",
        metric=fn.metric_duration().with(
            statistic="Maximum"
        ),
        evaluation_periods=1,
        datapoints_to_alarm=1,
        threshold=fn.timeout.to_milliseconds(),
        treat_missing_data=cloudwatch.TreatMissingData.IGNORE,
        alarm_name="My Lambda Timeout"
    )
```

## Advanced Logging

You can have more control over your function logs, by specifying the log format
(Json or plain text), the system log level, the application log level, as well
as choosing the log group:

```python
from aws_cdk.aws_logs import ILogGroup

# log_group: ILogGroup


lambda_.Function(self, "Lambda",
    code=lambda_.InlineCode("foo"),
    handler="index.handler",
    runtime=lambda_.Runtime.NODEJS_18_X,
    logging_format=lambda_.LoggingFormat.JSON,
    system_log_level_v2=lambda_.SystemLogLevel.INFO,
    application_log_level_v2=lambda_.ApplicationLogLevel.INFO,
    log_group=log_group
)
```

To use `applicationLogLevelV2` and/or `systemLogLevelV2` you must set `loggingFormat` to `LoggingFormat.JSON`.

## Resource-based Policies

AWS Lambda supports resource-based policies for controlling access to Lambda
functions and layers on a per-resource basis. In particular, this allows you to
give permission to AWS services, AWS Organizations, or other AWS accounts to
modify and invoke your functions.

### Grant function access to AWS services

```python
# Grant permissions to a service
# fn: lambda.Function

principal = iam.ServicePrincipal("my-service")

fn.grant_invoke(principal)

# Equivalent to:
fn.add_permission("my-service Invocation",
    principal=principal
)
```

You can also restrict permissions given to AWS services by providing
a source account or ARN (representing the account and identifier of the resource
that accesses the function or layer).

**Important**:

> By default `fn.grantInvoke()` grants permission to the principal to invoke any version of the function, including all past ones. If you only want the principal to be granted permission to invoke the latest version or the unqualified Lambda ARN, use `grantInvokeLatestVersion(grantee)`.

```python
# fn: lambda.Function

principal = iam.ServicePrincipal("my-service")
# Grant invoke only to latest version and unqualified lambda arn
fn.grant_invoke_latest_version(principal)
```

If you want to grant access for invoking a specific version of Lambda function, you can use `fn.grantInvokeVersion(grantee, version)`

```python
# fn: lambda.Function
# version: lambda.IVersion

principal = iam.ServicePrincipal("my-service")
# Grant invoke only to the specific version
fn.grant_invoke_version(principal, version)
```

For more information, see
[Granting function access to AWS services](https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html#permissions-resource-serviceinvoke)
in the AWS Lambda Developer Guide.

### Grant function access to an AWS Organization

```python
# Grant permissions to an entire AWS organization
# fn: lambda.Function

org = iam.OrganizationPrincipal("o-xxxxxxxxxx")

fn.grant_invoke(org)
```

In the above example, the `principal` will be `*` and all users in the
organization `o-xxxxxxxxxx` will get function invocation permissions.

You can restrict permissions given to the organization by specifying an
AWS account or role as the `principal`:

```python
# Grant permission to an account ONLY IF they are part of the organization
# fn: lambda.Function

account = iam.AccountPrincipal("123456789012")

fn.grant_invoke(account.in_organization("o-xxxxxxxxxx"))
```

For more information, see
[Granting function access to an organization](https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html#permissions-resource-xorginvoke)
in the AWS Lambda Developer Guide.

### Grant function access to other AWS accounts

```python
# Grant permission to other AWS account
# fn: lambda.Function

account = iam.AccountPrincipal("123456789012")

fn.grant_invoke(account)
```

For more information, see
[Granting function access to other accounts](https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html#permissions-resource-xaccountinvoke)
in the AWS Lambda Developer Guide.

### Grant function access to unowned principals

Providing an unowned principal (such as account principals, generic ARN
principals, service principals, and principals in other accounts) to a call to
`fn.grantInvoke` will result in a resource-based policy being created. If the
principal in question has conditions limiting the source account or ARN of the
operation (see above), these conditions will be automatically added to the
resource policy.

```python
# fn: lambda.Function

service_principal = iam.ServicePrincipal("my-service")
source_arn = "arn:aws:s3:::amzn-s3-demo-bucket"
source_account = "111122223333"
service_principal_with_conditions = service_principal.with_conditions({
    "ArnLike": {
        "aws:SourceArn": source_arn
    },
    "StringEquals": {
        "aws:SourceAccount": source_account
    }
})

fn.grant_invoke(service_principal_with_conditions)
```

### Grant function access to a CompositePrincipal

To grant invoke permissions to a `CompositePrincipal` use the `grantInvokeCompositePrincipal` method:

```python
# fn: lambda.Function

composite_principal = iam.CompositePrincipal(
    iam.OrganizationPrincipal("o-zzzzzzzzzz"),
    iam.ServicePrincipal("apigateway.amazonaws.com"))

fn.grant_invoke_composite_principal(composite_principal)
```

## Versions

You can use
[versions](https://docs.aws.amazon.com/lambda/latest/dg/configuration-versions.html)
to manage the deployment of your AWS Lambda functions. For example, you can
publish a new version of a function for beta testing without affecting users of
the stable production version.

The function version includes the following information:

* The function code and all associated dependencies.
* The Lambda runtime that executes the function.
* All of the function settings, including the environment variables.
* A unique Amazon Resource Name (ARN) to identify this version of the function.

You could create a version to your lambda function using the `Version` construct.

```python
# fn: lambda.Function

version = lambda_.Version(self, "MyVersion",
    lambda_=fn
)
```

The major caveat to know here is that a function version must always point to a
specific 'version' of the function. When the function is modified, the version
will continue to point to the 'then version' of the function.

One way to ensure that the `lambda.Version` always points to the latest version
of your `lambda.Function` is to set an environment variable which changes at
least as often as your code does. This makes sure the function always has the
latest code. For instance -

```python
code_version = "stringOrMethodToGetCodeVersion"
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    environment={
        "CodeVersionString": code_version
    }
)
```

The `fn.latestVersion` property returns a `lambda.IVersion` which represents
the `$LATEST` pseudo-version.

However, most AWS services require a specific AWS Lambda version,
and won't allow you to use `$LATEST`. Therefore, you would normally want
to use `lambda.currentVersion`.

The `fn.currentVersion` property can be used to obtain a `lambda.Version`
resource that represents the AWS Lambda function defined in your application.
Any change to your function's code or configuration will result in the creation
of a new version resource. You can specify options for this version through the
`currentVersionOptions` property.

NOTE: The `currentVersion` property is only supported when your AWS Lambda function
uses either `lambda.Code.fromAsset` or `lambda.Code.fromInline`. Other types
of code providers (such as `lambda.Code.fromBucket`) require that you define a
`lambda.Version` resource directly since the CDK is unable to determine if
their contents had changed.

### `currentVersion`: Updated hashing logic

To produce a new lambda version each time the lambda function is modified, the
`currentVersion` property under the hood, computes a new logical id based on the
properties of the function. This informs CloudFormation that a new
`AWS::Lambda::Version` resource should be created pointing to the updated Lambda
function.

However, a bug was introduced in this calculation that caused the logical id to
change when it was not required (ex: when the Function's `Tags` property, or
when the `DependsOn` clause was modified). This caused the deployment to fail
since the Lambda service does not allow creating duplicate versions.

This has been fixed in the AWS CDK but *existing* users need to opt-in via a
[feature flag](https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html). Users who have run `cdk init` since this fix will be opted in,
by default.

Otherwise, you will need to enable the [feature flag](https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html)
`@aws-cdk/aws-lambda:recognizeVersionProps`. Since CloudFormation does not
allow duplicate versions, you will also need to make some modification to
your function so that a new version can be created. To efficiently and trivially
modify all your lambda functions at once, you can attach the
`FunctionVersionUpgrade` aspect to the stack, which slightly alters the
function description. This aspect is intended for one-time use to upgrade the
version of all your functions at the same time, and can safely be removed after
deploying once.

```python
stack = Stack()
Aspects.of(stack).add(lambda_.FunctionVersionUpgrade(LAMBDA_RECOGNIZE_VERSION_PROPS))
```

When the new logic is in effect, you may rarely come across the following error:
`The following properties are not recognized as version properties`. This will
occur, typically when [property overrides](https://docs.aws.amazon.com/cdk/latest/guide/cfn_layer.html#cfn_layer_raw) are used, when a new property
introduced in `AWS::Lambda::Function` is used that CDK is still unaware of.

To overcome this error, use the API `Function.classifyVersionProperty()` to
record whether a new version should be generated when this property is changed.
This can be typically determined by checking whether the property can be
modified using the *[UpdateFunctionConfiguration](https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html)* API or not.

### `currentVersion`: Updated hashing logic for layer versions

An additional update to the hashing logic fixes two issues surrounding layers.
Prior to this change, updating the lambda layer version would have no effect on
the function version. Also, the order of lambda layers provided to the function
was unnecessarily baked into the hash.

This has been fixed in the AWS CDK starting with version 2.27. If you ran
`cdk init` with an earlier version, you will need to opt-in via a [feature flag](https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html).
If you run `cdk init` with v2.27 or later, this fix will be opted in, by default.

Existing users will need to enable the [feature flag](https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html)
`@aws-cdk/aws-lambda:recognizeLayerVersion`. Since CloudFormation does not
allow duplicate versions, they will also need to make some modification to
their function so that a new version can be created. To efficiently and trivially
modify all your lambda functions at once, users can attach the
`FunctionVersionUpgrade` aspect to the stack, which slightly alters the
function description. This aspect is intended for one-time use to upgrade the
version of all your functions at the same time, and can safely be removed after
deploying once.

```python
stack = Stack()
Aspects.of(stack).add(lambda_.FunctionVersionUpgrade(LAMBDA_RECOGNIZE_LAYER_VERSION))
```

## Aliases

You can define one or more
[aliases](https://docs.aws.amazon.com/lambda/latest/dg/configuration-aliases.html)
for your AWS Lambda function. A Lambda alias is like a pointer to a specific
Lambda function version. Users can access the function version using the alias
ARN.

The `version.addAlias()` method can be used to define an AWS Lambda alias that
points to a specific version.

The following example defines an alias named `live` which will always point to a
version that represents the function as defined in your CDK app. When you change
your lambda code or configuration, a new resource will be created. You can
specify options for the current version through the `currentVersionOptions`
property.

```python
fn = lambda_.Function(self, "MyFunction",
    current_version_options=lambda.VersionOptions(
        removal_policy=RemovalPolicy.RETAIN,  # retain old versions
        retry_attempts=1
    ),
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)

fn.add_alias("live")
```

## Function URL

A function URL is a dedicated HTTP(S) endpoint for your Lambda function. When you create a function URL, Lambda automatically generates a unique URL endpoint for you. Function URLs can be created for the latest version Lambda Functions, or Function Aliases (but not for Versions).

Function URLs are dual stack-enabled, supporting IPv4 and IPv6, and cross-origin resource sharing (CORS) configuration. After you configure a function URL for your function, you can invoke your function through its HTTP(S) endpoint via a web browser, curl, Postman, or any HTTP client. To invoke a function using IAM authentication your HTTP client must support SigV4 signing.

See the [Invoking Function URLs](https://docs.aws.amazon.com/lambda/latest/dg/urls-invocation.html) section of the AWS Lambda Developer Guide
for more information on the input and output payloads of Functions invoked in this way.

### IAM-authenticated Function URLs

To create a Function URL which can be called by an IAM identity, call `addFunctionUrl()`, followed by `grantInvokeFunctionUrl()`:

```python
# Can be a Function or an Alias
# fn: lambda.Function
# my_role: iam.Role


fn_url = fn.add_function_url()
fn_url.grant_invoke_url(my_role)

CfnOutput(self, "TheUrl",
    # The .url attributes will return the unique Function URL
    value=fn_url.url
)
```

Calls to this URL need to be signed with SigV4.

### Anonymous Function URLs

To create a Function URL which can be called anonymously, pass `authType: FunctionUrlAuthType.NONE` to `addFunctionUrl()`:

```python
# Can be a Function or an Alias
# fn: lambda.Function


fn_url = fn.add_function_url(
    auth_type=lambda_.FunctionUrlAuthType.NONE
)

CfnOutput(self, "TheUrl",
    value=fn_url.url
)
```

### CORS configuration for Function URLs

If you want your Function URLs to be invokable from a web page in browser, you
will need to configure cross-origin resource sharing to allow the call (if you do
not do this, your browser will refuse to make the call):

```python
# fn: lambda.Function


fn.add_function_url(
    auth_type=lambda_.FunctionUrlAuthType.NONE,
    cors=lambda.FunctionUrlCorsOptions(
        # Allow this to be called from websites on https://example.com.
        # Can also be ['*'] to allow all domain.
        allowed_origins=["https://example.com"]
    )
)
```

### Invoke Mode for Function URLs

Invoke mode determines how AWS Lambda invokes your function. You can configure the invoke mode when creating a Function URL using the invokeMode property

```python
# fn: lambda.Function


fn.add_function_url(
    auth_type=lambda_.FunctionUrlAuthType.NONE,
    invoke_mode=lambda_.InvokeMode.RESPONSE_STREAM
)
```

If the invokeMode property is not specified, the default BUFFERED mode will be used.

## Layers

The `lambda.LayerVersion` class can be used to define Lambda layers and manage
granting permissions to other AWS accounts or organizations.

```python
layer = lambda_.LayerVersion(stack, "MyLayer",
    code=lambda_.Code.from_asset(path.join(__dirname, "layer-code")),
    compatible_runtimes=[lambda_.Runtime.NODEJS_LATEST],
    license="Apache-2.0",
    description="A layer to test the L2 construct"
)

# To grant usage by other AWS accounts
layer.add_permission("remote-account-grant", account_id=aws_account_id)

# To grant usage to all accounts in some AWS Ogranization
# layer.grantUsage({ accountId: '*', organizationId });

lambda_.Function(stack, "MyLayeredLambda",
    code=lambda_.InlineCode("foo"),
    handler="index.handler",
    runtime=lambda_.Runtime.NODEJS_LATEST,
    layers=[layer]
)
```

By default, updating a layer creates a new layer version, and CloudFormation will delete the old version as part of the stack update.

Alternatively, a removal policy can be used to retain the old version:

```python
lambda_.LayerVersion(self, "MyLayer",
    removal_policy=RemovalPolicy.RETAIN,
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)
```

## Architecture

Lambda functions, by default, run on compute systems that have the 64 bit x86 architecture.

The AWS Lambda service also runs compute on the ARM architecture, which can reduce cost
for some workloads.

A lambda function can be configured to be run on one of these platforms:

```python
lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    architecture=lambda_.Architecture.ARM_64
)
```

Similarly, lambda layer versions can also be tagged with architectures it is compatible with.

```python
lambda_.LayerVersion(self, "MyLayer",
    removal_policy=RemovalPolicy.RETAIN,
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    compatible_architectures=[lambda_.Architecture.X86_64, lambda_.Architecture.ARM_64]
)
```

## Lambda Insights

Lambda functions can be configured to use CloudWatch [Lambda Insights](https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights.html)
which provides low-level runtime metrics for a Lambda functions.

```python
lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    insights_version=lambda_.LambdaInsightsVersion.VERSION_1_0_98_0
)
```

If the version of insights is not yet available in the CDK, you can also provide the ARN directly as so -

```python
layer_arn = "arn:aws:lambda:us-east-1:580247275435:layer:LambdaInsightsExtension:14"
lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    insights_version=lambda_.LambdaInsightsVersion.from_insight_version_arn(layer_arn)
)
```

If you are deploying an ARM_64 Lambda Function, you must specify a
Lambda Insights Version >= `1_0_119_0`.

```python
lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    architecture=lambda_.Architecture.ARM_64,
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    insights_version=lambda_.LambdaInsightsVersion.VERSION_1_0_119_0
)
```

### Parameters and Secrets Extension

Lambda functions can be configured to use the Parameters and Secrets Extension. The Parameters and Secrets Extension can be used to retrieve and cache [secrets](https://docs.aws.amazon.com/secretsmanager/latest/userguide/retrieving-secrets_lambda.html) from Secrets Manager or [parameters](https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-integration-lambda-extensions.html) from Parameter Store in Lambda functions without using an SDK.

```python
import aws_cdk.aws_secretsmanager as sm
import aws_cdk.aws_ssm as ssm


secret = sm.Secret(self, "Secret")
parameter = ssm.StringParameter(self, "Parameter",
    parameter_name="mySsmParameterName",
    string_value="mySsmParameterValue"
)

params_and_secrets = lambda_.ParamsAndSecretsLayerVersion.from_version(lambda_.ParamsAndSecretsVersions.V1_0_103,
    cache_size=500,
    log_level=lambda_.ParamsAndSecretsLogLevel.DEBUG
)

lambda_function = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    architecture=lambda_.Architecture.ARM_64,
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    params_and_secrets=params_and_secrets
)

secret.grant_read(lambda_function)
parameter.grant_read(lambda_function)
```

If the version of Parameters and Secrets Extension is not yet available in the CDK, you can also provide the ARN directly as so:

```python
import aws_cdk.aws_secretsmanager as sm
import aws_cdk.aws_ssm as ssm


secret = sm.Secret(self, "Secret")
parameter = ssm.StringParameter(self, "Parameter",
    parameter_name="mySsmParameterName",
    string_value="mySsmParameterValue"
)

layer_arn = "arn:aws:lambda:us-east-1:177933569100:layer:AWS-Parameters-and-Secrets-Lambda-Extension:4"
params_and_secrets = lambda_.ParamsAndSecretsLayerVersion.from_version_arn(layer_arn,
    cache_size=500
)

lambda_function = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    architecture=lambda_.Architecture.ARM_64,
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    params_and_secrets=params_and_secrets
)

secret.grant_read(lambda_function)
parameter.grant_read(lambda_function)
```

## Event Rule Target

You can use an AWS Lambda function as a target for an Amazon CloudWatch event
rule:

```python
import aws_cdk.aws_events as events
import aws_cdk.aws_events_targets as targets

# fn: lambda.Function

rule = events.Rule(self, "Schedule Rule",
    schedule=events.Schedule.cron(minute="0", hour="4")
)
rule.add_target(targets.LambdaFunction(fn))
```

## Event Sources

AWS Lambda supports a [variety of event sources](https://docs.aws.amazon.com/lambda/latest/dg/invoking-lambda-function.html).

In most cases, it is possible to trigger a function as a result of an event by
using one of the `add<Event>Notification` methods on the source construct. For
example, the `s3.Bucket` construct has an `onEvent` method which can be used to
trigger a Lambda when an event, such as PutObject occurs on an S3 bucket.

An alternative way to add event sources to a function is to use `function.addEventSource(source)`.
This method accepts an `IEventSource` object. The module **@aws-cdk/aws-lambda-event-sources**
includes classes for the various event sources supported by AWS Lambda.

For example, the following code adds an SQS queue as an event source for a function:

```python
import aws_cdk.aws_lambda_event_sources as eventsources
import aws_cdk.aws_sqs as sqs

# fn: lambda.Function

queue = sqs.Queue(self, "Queue")
fn.add_event_source(eventsources.SqsEventSource(queue))
```

The following code adds an S3 bucket notification as an event source:

```python
import aws_cdk.aws_lambda_event_sources as eventsources
import aws_cdk.aws_s3 as s3

# fn: lambda.Function

bucket = s3.Bucket(self, "Bucket")
fn.add_event_source(eventsources.S3EventSource(bucket,
    events=[s3.EventType.OBJECT_CREATED, s3.EventType.OBJECT_REMOVED],
    filters=[s3.NotificationKeyFilter(prefix="subdir/")]
))
```

The following code adds an DynamoDB notification as an event source filtering insert events:

```python
import aws_cdk.aws_lambda_event_sources as eventsources
import aws_cdk.aws_dynamodb as dynamodb

# fn: lambda.Function

table = dynamodb.Table(self, "Table",
    partition_key=dynamodb.Attribute(
        name="id",
        type=dynamodb.AttributeType.STRING
    ),
    stream=dynamodb.StreamViewType.NEW_IMAGE
)
fn.add_event_source(eventsources.DynamoEventSource(table,
    starting_position=lambda_.StartingPosition.LATEST,
    filters=[lambda_.FilterCriteria.filter({"event_name": lambda_.FilterRule.is_equal("INSERT")})]
))
```

By default, Lambda will encrypt Filter Criteria using AWS managed keys. But if you want to use a self managed KMS key to encrypt the filters, You can specify the self managed key using the `filterEncryption` property.

```python
import aws_cdk.aws_lambda_event_sources as eventsources
import aws_cdk.aws_dynamodb as dynamodb
from aws_cdk.aws_kms import Key

# fn: lambda.Function

table = dynamodb.Table(self, "Table",
    partition_key=dynamodb.Attribute(
        name="id",
        type=dynamodb.AttributeType.STRING
    ),
    stream=dynamodb.StreamViewType.NEW_IMAGE
)
# Your self managed KMS key
my_key = Key.from_key_arn(self, "SourceBucketEncryptionKey", "arn:aws:kms:us-east-1:123456789012:key/<key-id>")

fn.add_event_source(eventsources.DynamoEventSource(table,
    starting_position=lambda_.StartingPosition.LATEST,
    filters=[lambda_.FilterCriteria.filter({"event_name": lambda_.FilterRule.is_equal("INSERT")})],
    filter_encryption=my_key
))
```

> Lambda requires allow `kms:Decrypt` on Lambda principal `lambda.amazonaws.com` to use the key for Filter Criteria Encryption. If you create the KMS key in the stack, CDK will automatically add this permission to the Key when you creates eventSourceMapping. However, if you import the key using function like `Key.fromKeyArn` then you need to add the following permission to the KMS key before using it to encrypt Filter Criteria

```json
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Principal": {
                "Service": "lambda.amazonaws.com"
            },
            "Action": "kms:Decrypt",
            "Resource": "*"
        }
    ]
}
```

### Observability

Customers can now opt-in to get enhanced metrics for their event source mapping that capture each stage of processing using the `MetricsConfig` property.

The following code shows how to opt in for the enhanced metrics.

```python
import aws_cdk.aws_lambda_event_sources as eventsources
import aws_cdk.aws_dynamodb as dynamodb

# fn: lambda.Function

table = dynamodb.Table(self, "Table",
    partition_key=dynamodb.Attribute(
        name="id",
        type=dynamodb.AttributeType.STRING
    ),
    stream=dynamodb.StreamViewType.NEW_IMAGE
)

fn.add_event_source(eventsources.DynamoEventSource(table,
    starting_position=lambda_.StartingPosition.LATEST,
    metrics_config=lambda.MetricsConfig(
        metrics=[lambda_.MetricType.EVENT_COUNT]
    )
))
```

See the documentation for the **@aws-cdk/aws-lambda-event-sources** module for more details.

## Imported Lambdas

When referencing an imported lambda in the CDK, use `fromFunctionArn()` for most use cases:

```python
fn = lambda_.Function.from_function_arn(self, "Function", "arn:aws:lambda:us-east-1:123456789012:function:MyFn")
```

The `fromFunctionAttributes()` API is available for more specific use cases:

```python
fn = lambda_.Function.from_function_attributes(self, "Function",
    function_arn="arn:aws:lambda:us-east-1:123456789012:function:MyFn",
    # The following are optional properties for specific use cases and should be used with caution:

    # Use Case: imported function is in the same account as the stack. This tells the CDK that it
    # can modify the function's permissions.
    same_environment=True,

    # Use Case: imported function is in a different account and user commits to ensuring that the
    # imported function has the correct permissions outside the CDK.
    skip_permissions=True
)
```

`Function.fromFunctionArn()` and `Function.fromFunctionAttributes()` will attempt to parse the Function's Region and Account ID from the ARN. `addPermissions` will only work on the `Function` object if the Region and Account ID are deterministically the same as the scope of the Stack the referenced `Function` object is created in.
If the containing Stack is environment-agnostic or the Function ARN is a Token, this comparison will fail, and calls to `Function.addPermission` will do nothing.
If you know Function permissions can safely be added, you can use `Function.fromFunctionName()` instead, or pass `sameEnvironment: true` to `Function.fromFunctionAttributes()`.

```python
fn = lambda_.Function.from_function_name(self, "Function", "MyFn")
```

## Lambda with DLQ

A dead-letter queue can be automatically created for a Lambda function by
setting the `deadLetterQueueEnabled: true` configuration. In such case CDK creates
a `sqs.Queue` as `deadLetterQueue`.

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
    dead_letter_queue_enabled=True
)
```

It is also possible to provide a dead-letter queue instead of getting a new queue created:

```python
import aws_cdk.aws_sqs as sqs


dlq = sqs.Queue(self, "DLQ")
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
    dead_letter_queue=dlq
)
```

You can also use a `sns.Topic` instead of an `sqs.Queue` as dead-letter queue:

```python
import aws_cdk.aws_sns as sns


dlt = sns.Topic(self, "DLQ")
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_inline("// your code here"),
    dead_letter_topic=dlt
)
```

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/dlq.html)
to learn more about AWS Lambdas and DLQs.

## Lambda with X-Ray Tracing

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
    tracing=lambda_.Tracing.ACTIVE
)
```

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/lambda-x-ray.html)
to learn more about AWS Lambda's X-Ray support.

## Lambda with AWS Distro for OpenTelemetry layer

To have automatic integration with XRay without having to add dependencies or change your code, you can use the
[AWS Distro for OpenTelemetry Lambda (ADOT) layer](https://aws-otel.github.io/docs/getting-started/lambda).
Consuming the latest ADOT layer can be done with the following snippet:

```python
from aws_cdk.aws_lambda import AdotLambdaExecWrapper, AdotLayerVersion, AdotLambdaLayerJavaScriptSdkVersion


fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
    adot_instrumentation=lambda.AdotInstrumentationConfig(
        layer_version=AdotLayerVersion.from_java_script_sdk_layer_version(AdotLambdaLayerJavaScriptSdkVersion.LATEST),
        exec_wrapper=AdotLambdaExecWrapper.REGULAR_HANDLER
    )
)
```

To use a different layer version, use one of the following helper functions for the `layerVersion` prop:

* `AdotLayerVersion.fromJavaScriptSdkLayerVersion`
* `AdotLayerVersion.fromPythonSdkLayerVersion`
* `AdotLayerVersion.fromJavaSdkLayerVersion`
* `AdotLayerVersion.fromJavaAutoInstrumentationSdkLayerVersion`
* `AdotLayerVersion.fromGenericSdkLayerVersion`

Each helper function expects a version value from a corresponding enum-like class as below:

* `AdotLambdaLayerJavaScriptSdkVersion`
* `AdotLambdaLayerPythonSdkVersion`
* `AdotLambdaLayerJavaSdkVersion`
* `AdotLambdaLayerJavaAutoInstrumentationSdkVersion`
* `AdotLambdaLayerGenericSdkVersion`

For more examples, see our [the integration test](test/integ.lambda-adot.ts).

If you want to retrieve the ARN of the ADOT Lambda layer without enabling ADOT in a Lambda function:

```python
# fn: lambda.Function

layer_arn = lambda_.AdotLambdaLayerJavaSdkVersion.V1_19_0.layer_arn(fn.stack, fn.architecture)
```

When using the `AdotLambdaLayerPythonSdkVersion` the `AdotLambdaExecWrapper` needs to be `AdotLambdaExecWrapper.INSTRUMENT_HANDLER` as per [AWS Distro for OpenTelemetry Lambda Support For Python](https://aws-otel.github.io/docs/getting-started/lambda/lambda-python)

## Lambda with Profiling

The following code configures the lambda function with CodeGuru profiling. By default, this creates a new CodeGuru
profiling group -

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.PYTHON_3_9,
    handler="index.handler",
    code=lambda_.Code.from_asset("lambda-handler"),
    profiling=True
)
```

The `profilingGroup` property can be used to configure an existing CodeGuru profiler group.

CodeGuru profiling is supported for all Java runtimes and Python3.6+ runtimes.

See [the AWS documentation](https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html)
to learn more about AWS Lambda's Profiling support.

## Lambda with Reserved Concurrent Executions

```python
fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
    reserved_concurrent_executions=100
)
```

https://docs.aws.amazon.com/lambda/latest/dg/invocation-recursion.html

## Lambda with SnapStart

SnapStart is currently supported on Python 3.12, Python 3.13, .NET 8, and Java 11 and later [Java managed runtimes](https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html). SnapStart does not support provisioned concurrency, Amazon Elastic File System (Amazon EFS), or ephemeral storage greater than 512 MB. After you enable Lambda SnapStart for a particular Lambda function, publishing a new version of the function will trigger an optimization process.

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html) to learn more about AWS Lambda SnapStart

```python
fn = lambda_.Function(self, "MyFunction",
    code=lambda_.Code.from_asset(path.join(__dirname, "handler.zip")),
    runtime=lambda_.Runtime.JAVA_11,
    handler="example.Handler::handleRequest",
    snap_start=lambda_.SnapStartConf.ON_PUBLISHED_VERSIONS
)

version = fn.current_version
```

## AutoScaling

You can use Application AutoScaling to automatically configure the provisioned concurrency for your functions. AutoScaling can be set to track utilization or be based on a schedule. To configure AutoScaling on a function alias:

```python
import aws_cdk.aws_autoscaling as autoscaling

# fn: lambda.Function

alias = fn.add_alias("prod")

# Create AutoScaling target
as = alias.add_auto_scaling(max_capacity=50)

# Configure Target Tracking
as.scale_on_utilization(
    utilization_target=0.5
)

# Configure Scheduled Scaling
as.scale_on_schedule("ScaleUpInTheMorning",
    schedule=autoscaling.Schedule.cron(hour="8", minute="0"),
    min_capacity=20
)
```

```python
import aws_cdk.aws_applicationautoscaling as appscaling
import aws_cdk as cdk
from ......cx_api import LAMBDA_RECOGNIZE_LAYER_VERSION
import aws_cdk as lambda_

#
# Stack verification steps:
# aws application-autoscaling describe-scalable-targets --service-namespace lambda --resource-ids function:<function name>:prod
# has a minCapacity of 3 and maxCapacity of 50
#
class TestStack(cdk.Stack):
    def __init__(self, scope, id):
        super().__init__(scope, id)

        fn = lambda_.Function(self, "MyLambda",
            code=lambda_.InlineCode("exports.handler = async () => { console.log('hello world'); };"),
            handler="index.handler",
            runtime=lambda_.Runtime.NODEJS_LATEST
        )

        version = fn.current_version

        alias = lambda_.Alias(self, "Alias",
            alias_name="prod",
            version=version
        )

        scaling_target = alias.add_auto_scaling(min_capacity=3, max_capacity=50)

        scaling_target.scale_on_utilization(
            utilization_target=0.5
        )

        scaling_target.scale_on_schedule("ScaleUpInTheMorning",
            schedule=appscaling.Schedule.cron(hour="8", minute="0"),
            min_capacity=20
        )

        scaling_target.scale_on_schedule("ScaleDownAtNight",
            schedule=appscaling.Schedule.cron(hour="20", minute="0"),
            max_capacity=20
        )

        cdk.CfnOutput(self, "FunctionName",
            value=fn.function_name
        )

app = cdk.App()

stack = TestStack(app, "aws-lambda-autoscaling")

# Changes the function description when the feature flag is present
# to validate the changed function hash.
cdk.Aspects.of(stack).add(lambda_.FunctionVersionUpgrade(LAMBDA_RECOGNIZE_LAYER_VERSION))

app.synth()
```

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/invocation-scaling.html) on autoscaling lambda functions.

## Log Group

By default, Lambda functions automatically create a log group with the name `/aws/lambda/<function-name>` upon first execution with
log data set to never expire.
This is convenient, but prevents you from changing any of the properties of this auto-created log group using the AWS CDK.
For example you cannot set log retention or assign a data protection policy.

To fully customize the logging behavior of your Lambda function, create a `logs.LogGroup` ahead of time and use the `logGroup` property to instruct the Lambda function to send logs to it.
This way you can use the full features set supported by Amazon CloudWatch Logs.

```python
from aws_cdk.aws_logs import LogGroup


my_log_group = LogGroup(self, "MyLogGroupWithLogGroupName",
    log_group_name="customLogGroup"
)

lambda_.Function(self, "Lambda",
    code=lambda_.InlineCode("foo"),
    handler="index.handler",
    runtime=lambda_.Runtime.NODEJS_18_X,
    log_group=my_log_group
)
```

Providing a user-controlled log group was rolled out to commercial regions on 2023-11-16.
If you are deploying to another type of region, please check regional availability first.

## Lambda with Recursive Loop protection

Recursive loop protection is to stop unintended loops. The customers are opted in by default for Lambda to detect and terminate unintended loops between Lambda and other AWS Services.
The property can be assigned two values here, "Allow" and "Terminate".

The default value is set to "Terminate", which lets the Lambda to detect and terminate the recursive loops.

When the value is set to "Allow", the customers opt out of recursive loop detection and Lambda does not terminate recursive loops if any.

See [the AWS documentation](https://docs.aws.amazon.com/lambda/latest/dg/invocation-recursion.html) to learn more about AWS Lambda Recusrive Loop Detection

```python
fn = lambda_.Function(self, "MyFunction",
    code=lambda_.Code.from_asset(path.join(__dirname, "handler.zip")),
    runtime=lambda_.Runtime.JAVA_11,
    handler="example.Handler::handleRequest",
    recursive_loop=lambda_.RecursiveLoop.TERMINATE
)
```

### Legacy Log Retention

As an alternative to providing a custom, user controlled log group, the legacy `logRetention` property can be used to set a different expiration period.
This feature uses a Custom Resource to change the log retention of the automatically created log group.

By default, CDK uses the AWS SDK retry options when creating a log group. The `logRetentionRetryOptions` property
allows you to customize the maximum number of retries and base backoff duration.

*Note* that a [CloudFormation custom
resource](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cfn-customresource.html) is added
to the stack that pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the
correct log retention period (never expire, by default). This Custom Resource will also create a log group to log events of the custom resource. The log retention period for this addtional log group is hard-coded to 1 day.

*Further note* that, if the log group already exists and the `logRetention` is not set, the custom resource will reset
the log retention to never expire even if it was configured with a different value.

## FileSystem Access

You can configure a function to mount an Amazon Elastic File System (Amazon EFS) to a
directory in your runtime environment with the `filesystem` property. To access Amazon EFS
from lambda function, the Amazon EFS access point will be required.

The following sample allows the lambda function to mount the Amazon EFS access point to `/mnt/msg` in the runtime environment and access the filesystem with the POSIX identity defined in `posixUser`.

```python
import aws_cdk.aws_ec2 as ec2
import aws_cdk.aws_efs as efs


# create a new VPC
vpc = ec2.Vpc(self, "VPC")

# create a new Amazon EFS filesystem
file_system = efs.FileSystem(self, "Efs", vpc=vpc)

# create a new access point from the filesystem
access_point = file_system.add_access_point("AccessPoint",
    # set /export/lambda as the root of the access point
    path="/export/lambda",
    # as /export/lambda does not exist in a new efs filesystem, the efs will create the directory with the following createAcl
    create_acl=efs.Acl(
        owner_uid="1001",
        owner_gid="1001",
        permissions="750"
    ),
    # enforce the POSIX identity so lambda function will access with this identity
    posix_user=efs.PosixUser(
        uid="1001",
        gid="1001"
    )
)

fn = lambda_.Function(self, "MyLambda",
    # mount the access point to /mnt/msg in the lambda runtime environment
    filesystem=lambda_.FileSystem.from_efs_access_point(access_point, "/mnt/msg"),
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    vpc=vpc
)
```

## IPv6 support

You can configure IPv6 connectivity for lambda function by setting `Ipv6AllowedForDualStack` to true.
It allows Lambda functions to specify whether the IPv6 traffic should be allowed when using dual-stack VPCs.
To access IPv6 network using Lambda, Dual-stack VPC is required. Using dual-stack VPC a function communicates with subnet over either of IPv4 or IPv6.

```python
import aws_cdk.aws_ec2 as ec2


nat_provider = ec2.NatProvider.gateway()

# create dual-stack VPC
vpc = ec2.Vpc(self, "DualStackVpc",
    ip_protocol=ec2.IpProtocol.DUAL_STACK,
    subnet_configuration=[ec2.SubnetConfiguration(
        name="Ipv6Public1",
        subnet_type=ec2.SubnetType.PUBLIC
    ), ec2.SubnetConfiguration(
        name="Ipv6Public2",
        subnet_type=ec2.SubnetType.PUBLIC
    ), ec2.SubnetConfiguration(
        name="Ipv6Private1",
        subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS
    )
    ],
    nat_gateway_provider=nat_provider
)

nat_gateway_id = nat_provider.configured_gateways[0].gateway_id
(vpc.private_subnets[0]).add_ipv6_nat64_route(nat_gateway_id)

fn = lambda_.Function(self, "Lambda_with_IPv6_VPC",
    code=lambda_.InlineCode("def main(event, context): pass"),
    handler="index.main",
    runtime=lambda_.Runtime.PYTHON_3_9,
    vpc=vpc,
    ipv6_allowed_for_dual_stack=True
)
```

## Outbound traffic

By default, when creating a Lambda function, it would add a security group outbound rule to allow sending all network traffic (except IPv6). This is controlled by `allowAllOutbound` in function properties, which has a default value of `true`.

To allow outbound IPv6 traffic by default, explicitly set `allowAllIpv6Outbound` to `true` in function properties as shown below (the default value for `allowAllIpv6Outbound` is `false`):

```python
import aws_cdk.aws_ec2 as ec2


vpc = ec2.Vpc(self, "Vpc")

fn = lambda_.Function(self, "LambdaWithIpv6Outbound",
    code=lambda_.InlineCode("def main(event, context): pass"),
    handler="index.main",
    runtime=lambda_.Runtime.PYTHON_3_9,
    vpc=vpc,
    allow_all_ipv6_outbound=True
)
```

Do not specify `allowAllOutbound` or `allowAllIpv6Outbound` property if the `securityGroups` or `securityGroup` property is set. Instead, configure these properties directly on the security group.

## Ephemeral Storage

You can configure ephemeral storage on a function to control the amount of storage it gets for reading
or writing data, allowing you to use AWS Lambda for ETL jobs, ML inference, or other data-intensive workloads.
The ephemeral storage will be accessible in the functions' `/tmp` directory.

```python
from aws_cdk import Size


fn = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
    ephemeral_storage_size=Size.mebibytes(1024)
)
```

Read more about using this feature in [this AWS blog post](https://aws.amazon.com/blogs/aws/aws-lambda-now-supports-up-to-10-gb-ephemeral-storage/).

## Singleton Function

The `SingletonFunction` construct is a way to guarantee that a lambda function will be guaranteed to be part of the stack,
once and only once, irrespective of how many times the construct is declared to be part of the stack. This is guaranteed
as long as the `uuid` property and the optional `lambdaPurpose` property stay the same whenever they're declared into the
stack.

A typical use case of this function is when a higher level construct needs to declare a Lambda function as part of it but
needs to guarantee that the function is declared once. However, a user of this higher level construct can declare it any
number of times and with different properties. Using `SingletonFunction` here with a fixed `uuid` will guarantee this.

For example, the `AwsCustomResource` construct requires only one single lambda function for all api calls that are made.

## Bundling Asset Code

When using `lambda.Code.fromAsset(path)` it is possible to bundle the code by running a
command in a Docker container. The asset path will be mounted at `/asset-input`. The
Docker container is responsible for putting content at `/asset-output`. The content at
`/asset-output` will be zipped and used as Lambda code.

Example with Python:

```python
lambda_.Function(self, "Function",
    code=lambda_.Code.from_asset(path.join(__dirname, "my-python-handler"),
        bundling=BundlingOptions(
            image=lambda_.Runtime.PYTHON_3_9.bundling_image,
            command=["bash", "-c", "pip install -r requirements.txt -t /asset-output && cp -au . /asset-output"
            ]
        )
    ),
    runtime=lambda_.Runtime.PYTHON_3_9,
    handler="index.handler"
)
```

Runtimes expose a `bundlingImage` property that points to the [AWS SAM](https://github.com/awslabs/aws-sam-cli) build image.

Use `cdk.DockerImage.fromRegistry(image)` to use an existing image or
`cdk.DockerImage.fromBuild(path)` to build a specific image:

```python
lambda_.Function(self, "Function",
    code=lambda_.Code.from_asset("/path/to/handler",
        bundling=BundlingOptions(
            image=DockerImage.from_build("/path/to/dir/with/DockerFile",
                build_args={
                    "ARG1": "value1"
                }
            ),
            command=["my", "cool", "command"]
        )
    ),
    runtime=lambda_.Runtime.PYTHON_3_9,
    handler="index.handler"
)
```

## Language-specific APIs

Language-specific higher level constructs are provided in separate modules:

* `aws-cdk-lib/aws-lambda-nodejs`: [Github](https://github.com/aws/aws-cdk/tree/main/packages/aws-cdk-lib/aws-lambda-nodejs) & [CDK Docs](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_lambda_nodejs-readme.html)
* `@aws-cdk/aws-lambda-python-alpha`: [Github](https://github.com/aws/aws-cdk/tree/main/packages/%40aws-cdk/aws-lambda-python-alpha) & [CDK Docs](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-lambda-python-alpha-readme.html)

## Code Signing

Code signing for AWS Lambda helps to ensure that only trusted code runs in your Lambda functions.
When enabled, AWS Lambda checks every code deployment and verifies that the code package is signed by a trusted source.
For more information, see [Configuring code signing for AWS Lambda](https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html).
The following code configures a function with code signing.

Please note the code will not be automatically signed before deployment. To ensure your code is properly signed, you'll need to conduct the code signing process either through the AWS CLI (Command Line Interface) [start-signing-job](https://awscli.amazonaws.com/v2/documentation/api/latest/reference/signer/start-signing-job.html) or by accessing the AWS Signer console.

```python
import aws_cdk.aws_signer as signer


signing_profile = signer.SigningProfile(self, "SigningProfile",
    platform=signer.Platform.AWS_LAMBDA_SHA384_ECDSA
)

code_signing_config = lambda_.CodeSigningConfig(self, "CodeSigningConfig",
    signing_profiles=[signing_profile]
)

lambda_.Function(self, "Function",
    code_signing_config=code_signing_config,
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)
```

## Runtime updates

Lambda runtime management controls help reduce the risk of impact to your workloads in the rare event of a runtime version incompatibility.
For more information, see [Runtime management controls](https://docs.aws.amazon.com/lambda/latest/dg/runtimes-update.html#runtime-management-controls)

```python
lambda_.Function(self, "Lambda",
    runtime_management_mode=lambda_.RuntimeManagementMode.AUTO,
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)
```

If you want to set the "Manual" setting, using the ARN of the runtime version as the argument.

```python
lambda_.Function(self, "Lambda",
    runtime_management_mode=lambda_.RuntimeManagementMode.manual("runtimeVersion-arn"),
    runtime=lambda_.Runtime.NODEJS_18_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)
```

## Exclude Patterns for Assets

When using `lambda.Code.fromAsset(path)` an `exclude` property allows you to ignore particular files for assets by providing patterns for file paths to exclude. Note that this has no effect on `Assets` bundled using the `bundling` property.

The `ignoreMode` property can be used with the `exclude` property to specify the file paths to ignore based on the [.gitignore specification](https://git-scm.com/docs/gitignore) or the [.dockerignore specification](https://docs.docker.com/engine/reference/builder/#dockerignore-file). The default behavior is to ignore file paths based on simple glob patterns.

```python
lambda_.Function(self, "Function",
    code=lambda_.Code.from_asset(path.join(__dirname, "my-python-handler"),
        exclude=["*.ignore"],
        ignore_mode=IgnoreMode.DOCKER
    ),
    runtime=lambda_.Runtime.PYTHON_3_9,
    handler="index.handler"
)
```

You can also write to include only certain files by using a negation.

```python
lambda_.Function(self, "Function",
    code=lambda_.Code.from_asset(path.join(__dirname, "my-python-handler"),
        exclude=["*", "!index.py"]
    ),
    runtime=lambda_.Runtime.PYTHON_3_9,
    handler="index.handler"
)
```
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    AssetHashType as _AssetHashType_05b67f2d,
    BundlingOptions as _BundlingOptions_588cc936,
    CfnParameter as _CfnParameter_48fc1866,
    CfnResource as _CfnResource_9df397a6,
    CfnTag as _CfnTag_f6864754,
    DockerBuildOptions as _DockerBuildOptions_8e28be8a,
    DockerCacheOption as _DockerCacheOption_63aeecf2,
    DockerImage as _DockerImage_f97a0c12,
    Duration as _Duration_4839e8c3,
    IAspect as _IAspect_118c810a,
    IInspectable as _IInspectable_c2943556,
    IResolvable as _IResolvable_da3f097b,
    IResource as _IResource_c80c4260,
    ITaggable as _ITaggable_36806126,
    ITaggableV2 as _ITaggableV2_4e6798f8,
    IgnoreMode as _IgnoreMode_655a98e8,
    RemovalPolicy as _RemovalPolicy_9f93c814,
    Resource as _Resource_45bc6135,
    ResourceProps as _ResourceProps_15a65b4e,
    Size as _Size_7b441c34,
    SymlinkFollowMode as _SymlinkFollowMode_047ec1f6,
    TagManager as _TagManager_0a598cb3,
    TimeZone as _TimeZone_cdd72ac9,
    TreeInspector as _TreeInspector_488e0dd5,
)
from ..aws_applicationautoscaling import (
    BaseTargetTrackingProps as _BaseTargetTrackingProps_540ba713,
    ScalingSchedule as _ScalingSchedule_9604f271,
    Schedule as _Schedule_e93ba733,
)
from ..aws_cloudwatch import (
    Metric as _Metric_e396a4dc,
    MetricOptions as _MetricOptions_1788b62f,
    Unit as _Unit_61bc6f70,
)
from ..aws_codeguruprofiler import IProfilingGroup as _IProfilingGroup_0bba72c4
from ..aws_ec2 import (
    Connections as _Connections_0f31fce8,
    IClientVpnConnectionHandler as _IClientVpnConnectionHandler_715171c2,
    IConnectable as _IConnectable_10015a05,
    ISecurityGroup as _ISecurityGroup_acf8a799,
    IVpc as _IVpc_f30d5663,
    SubnetSelection as _SubnetSelection_e57d76df,
)
from ..aws_ecr import IRepository as _IRepository_e6004aa6
from ..aws_ecr_assets import (
    DockerCacheOption as _DockerCacheOption_58ef18ca,
    DockerImageAssetInvalidationOptions as _DockerImageAssetInvalidationOptions_4deb8d45,
    DockerImageAssetOptions as _DockerImageAssetOptions_9580cd76,
    NetworkMode as _NetworkMode_897e5081,
    Platform as _Platform_d16f3cf1,
)
from ..aws_efs import IAccessPoint as _IAccessPoint_ce87b375
from ..aws_iam import (
    CompositePrincipal as _CompositePrincipal_c2fe180a,
    Grant as _Grant_a7ae64f8,
    IGrantable as _IGrantable_71c4f5de,
    IPrincipal as _IPrincipal_539bb2fd,
    IRole as _IRole_235f5d8e,
    PolicyStatement as _PolicyStatement_0fe33853,
)
from ..aws_kms import IKey as _IKey_5f11635f
from ..aws_logs import (
    ILogGroup as _ILogGroup_3c4fa718,
    LogRetentionRetryOptions as _LogRetentionRetryOptions_62d80a14,
    RetentionDays as _RetentionDays_070f99f0,
)
from ..aws_s3 import IBucket as _IBucket_42e086fd, Location as _Location_0948fa7f
from ..aws_s3_assets import AssetOptions as _AssetOptions_2aa69621
from ..aws_signer import ISigningProfile as _ISigningProfile_5140a756
from ..aws_sns import ITopic as _ITopic_9eca4852
from ..aws_sqs import IQueue as _IQueue_7ed6f679


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.AdotInstrumentationConfig",
    jsii_struct_bases=[],
    name_mapping={"exec_wrapper": "execWrapper", "layer_version": "layerVersion"},
)
class AdotInstrumentationConfig:
    def __init__(
        self,
        *,
        exec_wrapper: "AdotLambdaExecWrapper",
        layer_version: "AdotLayerVersion",
    ) -> None:
        '''Properties for an ADOT instrumentation in Lambda.

        :param exec_wrapper: The startup script to run, see ADOT documentation to pick the right script for your use case: https://aws-otel.github.io/docs/getting-started/lambda.
        :param layer_version: The ADOT Lambda layer.

        :exampleMetadata: infused

        Example::

            from aws_cdk.aws_lambda import AdotLambdaExecWrapper, AdotLayerVersion, AdotLambdaLayerJavaScriptSdkVersion
            
            
            fn = lambda_.Function(self, "MyFunction",
                runtime=lambda_.Runtime.NODEJS_18_X,
                handler="index.handler",
                code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
                adot_instrumentation=lambda.AdotInstrumentationConfig(
                    layer_version=AdotLayerVersion.from_java_script_sdk_layer_version(AdotLambdaLayerJavaScriptSdkVersion.LATEST),
                    exec_wrapper=AdotLambdaExecWrapper.REGULAR_HANDLER
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2183d370fe0d0be64cb9e536b8e1ba5875c1db63d7705d98b3989f17d82d3bff)
            check_type(argname="argument exec_wrapper", value=exec_wrapper, expected_type=type_hints["exec_wrapper"])
            check_type(argname="argument layer_version", value=layer_version, expected_type=type_hints["layer_version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "exec_wrapper": exec_wrapper,
            "layer_version": layer_version,
        }

    @builtins.property
    def exec_wrapper(self) -> "AdotLambdaExecWrapper":
        '''The startup script to run, see ADOT documentation to pick the right script for your use case: https://aws-otel.github.io/docs/getting-started/lambda.'''
        result = self._values.get("exec_wrapper")
        assert result is not None, "Required property 'exec_wrapper' is missing"
        return typing.cast("AdotLambdaExecWrapper", result)

    @builtins.property
    def layer_version(self) -> "AdotLayerVersion":
        '''The ADOT Lambda layer.'''
        result = self._values.get("layer_version")
        assert result is not None, "Required property 'layer_version' is missing"
        return typing.cast("AdotLayerVersion", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AdotInstrumentationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_lambda.AdotLambdaExecWrapper")
class AdotLambdaExecWrapper(enum.Enum):
    '''The wrapper script to be used for the Lambda function in order to enable auto instrumentation with ADOT.

    :exampleMetadata: infused

    Example::

        from aws_cdk.aws_lambda import AdotLambdaExecWrapper, AdotLayerVersion, AdotLambdaLayerJavaScriptSdkVersion
        
        
        fn = lambda_.Function(self, "MyFunction",
            runtime=lambda_.Runtime.NODEJS_18_X,
            handler="index.handler",
            code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
            adot_instrumentation=lambda.AdotInstrumentationConfig(
                layer_version=AdotLayerVersion.from_java_script_sdk_layer_version(AdotLambdaLayerJavaScriptSdkVersion.LATEST),
                exec_wrapper=AdotLambdaExecWrapper.REGULAR_HANDLER
            )
        )
    '''

    REGULAR_HANDLER = "REGULAR_HANDLER"
    '''Wrapping regular Lambda handlers.'''
    PROXY_HANDLER = "PROXY_HANDLER"
    '''Wrapping regular handlers (implementing RequestHandler) proxied through API Gateway, enabling HTTP context propagation.'''
    STREAM_HANDLER = "STREAM_HANDLER"
    '''Wrapping streaming handlers (implementing RequestStreamHandler), enabling HTTP context propagation for HTTP requests.'''
    INSTRUMENT_HANDLER = "INSTRUMENT_HANDLER"
    '''Wrapping python lambda handlers see https://aws-otel.github.io/docs/getting-started/lambda/lambda-python.'''
    SQS_HANDLER = "SQS_HANDLER"
    '''Wrapping SQS-triggered function handlers (implementing RequestHandler).'''


class AdotLambdaLayerGenericVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.AdotLambdaLayerGenericVersion",
):
    '''The collection of versions of the ADOT Lambda Layer for generic purpose.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        adot_lambda_layer_generic_version = lambda_.AdotLambdaLayerGenericVersion.LATEST
    '''

    @jsii.member(jsii_name="layerArn")
    def layer_arn(
        self,
        scope: _constructs_77d1e7e8.IConstruct,
        architecture: "Architecture",
    ) -> builtins.str:
        '''The ARN of the Lambda layer.

        :param scope: The binding scope. Usually this is the stack where the Lambda layer is bound to
        :param architecture: The architecture of the Lambda layer (either X86_64 or ARM_64).
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bd486210c5e34bb4a915e0138637181c2370c211712e6591d46aa8eed39701ac)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
        return typing.cast(builtins.str, jsii.invoke(self, "layerArn", [scope, architecture]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LATEST")
    def LATEST(cls) -> "AdotLambdaLayerGenericVersion":
        '''The latest layer version available in this CDK version.

        New versions could
        introduce incompatible changes. Make sure to test them before deploying to production.
        '''
        return typing.cast("AdotLambdaLayerGenericVersion", jsii.sget(cls, "LATEST"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V0_102_1")
    def V0_102_1(cls) -> "AdotLambdaLayerGenericVersion":
        '''Version 0.102.1.'''
        return typing.cast("AdotLambdaLayerGenericVersion", jsii.sget(cls, "V0_102_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V0_115_0")
    def V0_115_0(cls) -> "AdotLambdaLayerGenericVersion":
        '''Version 0.115.0.'''
        return typing.cast("AdotLambdaLayerGenericVersion", jsii.sget(cls, "V0_115_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V0_62_1")
    def V0_62_1(cls) -> "AdotLambdaLayerGenericVersion":
        '''Version 0.62.1.'''
        return typing.cast("AdotLambdaLayerGenericVersion", jsii.sget(cls, "V0_62_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V0_82_0")
    def V0_82_0(cls) -> "AdotLambdaLayerGenericVersion":
        '''Version 0.82.0.'''
        return typing.cast("AdotLambdaLayerGenericVersion", jsii.sget(cls, "V0_82_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V0_84_0")
    def V0_84_0(cls) -> "AdotLambdaLayerGenericVersion":
        '''Version 0.84.0.'''
        return typing.cast("AdotLambdaLayerGenericVersion", jsii.sget(cls, "V0_84_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V0_88_0")
    def V0_88_0(cls) -> "AdotLambdaLayerGenericVersion":
        '''Version 0.88.0.'''
        return typing.cast("AdotLambdaLayerGenericVersion", jsii.sget(cls, "V0_88_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V0_90_1")
    def V0_90_1(cls) -> "AdotLambdaLayerGenericVersion":
        '''Version 0.90.1.'''
        return typing.cast("AdotLambdaLayerGenericVersion", jsii.sget(cls, "V0_90_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V0_98_0")
    def V0_98_0(cls) -> "AdotLambdaLayerGenericVersion":
        '''Version 0.98.0.'''
        return typing.cast("AdotLambdaLayerGenericVersion", jsii.sget(cls, "V0_98_0"))

    @builtins.property
    @jsii.member(jsii_name="layerVersion")
    def _layer_version(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "layerVersion"))


class AdotLambdaLayerJavaAutoInstrumentationVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.AdotLambdaLayerJavaAutoInstrumentationVersion",
):
    '''The collection of versions of the ADOT Lambda Layer for Java auto-instrumentation.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        adot_lambda_layer_java_auto_instrumentation_version = lambda_.AdotLambdaLayerJavaAutoInstrumentationVersion.LATEST
    '''

    @jsii.member(jsii_name="layerArn")
    def layer_arn(
        self,
        scope: _constructs_77d1e7e8.IConstruct,
        architecture: "Architecture",
    ) -> builtins.str:
        '''The ARN of the Lambda layer.

        :param scope: The binding scope. Usually this is the stack where the Lambda layer is bound to
        :param architecture: The architecture of the Lambda layer (either X86_64 or ARM_64).
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__693075328f0034af6059c14d85177ff932eac6ac3aae7ce66b41521c3d5b9d4a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
        return typing.cast(builtins.str, jsii.invoke(self, "layerArn", [scope, architecture]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LATEST")
    def LATEST(cls) -> "AdotLambdaLayerJavaAutoInstrumentationVersion":
        '''The latest layer version available in this CDK version.

        New versions could
        introduce incompatible changes. Make sure to test them before deploying to production.
        '''
        return typing.cast("AdotLambdaLayerJavaAutoInstrumentationVersion", jsii.sget(cls, "LATEST"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_19_2")
    def V1_19_2(cls) -> "AdotLambdaLayerJavaAutoInstrumentationVersion":
        '''Version 1.19.2.'''
        return typing.cast("AdotLambdaLayerJavaAutoInstrumentationVersion", jsii.sget(cls, "V1_19_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_28_1")
    def V1_28_1(cls) -> "AdotLambdaLayerJavaAutoInstrumentationVersion":
        '''Version 1.28.1.'''
        return typing.cast("AdotLambdaLayerJavaAutoInstrumentationVersion", jsii.sget(cls, "V1_28_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_30_0")
    def V1_30_0(cls) -> "AdotLambdaLayerJavaAutoInstrumentationVersion":
        '''Version 1.30.0.'''
        return typing.cast("AdotLambdaLayerJavaAutoInstrumentationVersion", jsii.sget(cls, "V1_30_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_31_0")
    def V1_31_0(cls) -> "AdotLambdaLayerJavaAutoInstrumentationVersion":
        '''Version 1.31.0.'''
        return typing.cast("AdotLambdaLayerJavaAutoInstrumentationVersion", jsii.sget(cls, "V1_31_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_32_0")
    def V1_32_0(cls) -> "AdotLambdaLayerJavaAutoInstrumentationVersion":
        '''Version 1.32.0.'''
        return typing.cast("AdotLambdaLayerJavaAutoInstrumentationVersion", jsii.sget(cls, "V1_32_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_32_0_1")
    def V1_32_0_1(cls) -> "AdotLambdaLayerJavaAutoInstrumentationVersion":
        '''Version 1.32.0.'''
        return typing.cast("AdotLambdaLayerJavaAutoInstrumentationVersion", jsii.sget(cls, "V1_32_0_1"))

    @builtins.property
    @jsii.member(jsii_name="layerVersion")
    def _layer_version(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "layerVersion"))


class AdotLambdaLayerJavaScriptSdkVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.AdotLambdaLayerJavaScriptSdkVersion",
):
    '''The collection of versions of the ADOT Lambda Layer for JavaScript SDK.

    :exampleMetadata: infused

    Example::

        from aws_cdk.aws_lambda import AdotLambdaExecWrapper, AdotLayerVersion, AdotLambdaLayerJavaScriptSdkVersion
        
        
        fn = lambda_.Function(self, "MyFunction",
            runtime=lambda_.Runtime.NODEJS_18_X,
            handler="index.handler",
            code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
            adot_instrumentation=lambda.AdotInstrumentationConfig(
                layer_version=AdotLayerVersion.from_java_script_sdk_layer_version(AdotLambdaLayerJavaScriptSdkVersion.LATEST),
                exec_wrapper=AdotLambdaExecWrapper.REGULAR_HANDLER
            )
        )
    '''

    @jsii.member(jsii_name="layerArn")
    def layer_arn(
        self,
        scope: _constructs_77d1e7e8.IConstruct,
        architecture: "Architecture",
    ) -> builtins.str:
        '''The ARN of the Lambda layer.

        :param scope: The binding scope. Usually this is the stack where the Lambda layer is bound to
        :param architecture: The architecture of the Lambda layer (either X86_64 or ARM_64).
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e40047738b35dabb920e369f87eca05f3eaa69616ec842d8a65dfb9471089537)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
        return typing.cast(builtins.str, jsii.invoke(self, "layerArn", [scope, architecture]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LATEST")
    def LATEST(cls) -> "AdotLambdaLayerJavaScriptSdkVersion":
        '''The latest layer version available in this CDK version.

        New versions could
        introduce incompatible changes. Make sure to test them before deploying to production.
        '''
        return typing.cast("AdotLambdaLayerJavaScriptSdkVersion", jsii.sget(cls, "LATEST"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_15_0_1")
    def V1_15_0_1(cls) -> "AdotLambdaLayerJavaScriptSdkVersion":
        '''Version 1.15.0.'''
        return typing.cast("AdotLambdaLayerJavaScriptSdkVersion", jsii.sget(cls, "V1_15_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_16_0")
    def V1_16_0(cls) -> "AdotLambdaLayerJavaScriptSdkVersion":
        '''Version 1.16.0.'''
        return typing.cast("AdotLambdaLayerJavaScriptSdkVersion", jsii.sget(cls, "V1_16_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_17_1")
    def V1_17_1(cls) -> "AdotLambdaLayerJavaScriptSdkVersion":
        '''Version 1.17.1.'''
        return typing.cast("AdotLambdaLayerJavaScriptSdkVersion", jsii.sget(cls, "V1_17_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_18_1")
    def V1_18_1(cls) -> "AdotLambdaLayerJavaScriptSdkVersion":
        '''Version 1.18.1.'''
        return typing.cast("AdotLambdaLayerJavaScriptSdkVersion", jsii.sget(cls, "V1_18_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_30_0")
    def V1_30_0(cls) -> "AdotLambdaLayerJavaScriptSdkVersion":
        '''Version 1.30.0.'''
        return typing.cast("AdotLambdaLayerJavaScriptSdkVersion", jsii.sget(cls, "V1_30_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_7_0")
    def V1_7_0(cls) -> "AdotLambdaLayerJavaScriptSdkVersion":
        '''Version 1.7.0.'''
        return typing.cast("AdotLambdaLayerJavaScriptSdkVersion", jsii.sget(cls, "V1_7_0"))

    @builtins.property
    @jsii.member(jsii_name="layerVersion")
    def _layer_version(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "layerVersion"))


class AdotLambdaLayerJavaSdkVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.AdotLambdaLayerJavaSdkVersion",
):
    '''The collection of versions of the ADOT Lambda Layer for Java SDK.

    :exampleMetadata: infused

    Example::

        # fn: lambda.Function
        
        layer_arn = lambda_.AdotLambdaLayerJavaSdkVersion.V1_19_0.layer_arn(fn.stack, fn.architecture)
    '''

    @jsii.member(jsii_name="layerArn")
    def layer_arn(
        self,
        scope: _constructs_77d1e7e8.IConstruct,
        architecture: "Architecture",
    ) -> builtins.str:
        '''The ARN of the Lambda layer.

        :param scope: The binding scope. Usually this is the stack where the Lambda layer is bound to
        :param architecture: The architecture of the Lambda layer (either X86_64 or ARM_64).
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d119168598fa9513c113749fb91a178b877ed845e9452a7e7fd8e8a56ea07824)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
        return typing.cast(builtins.str, jsii.invoke(self, "layerArn", [scope, architecture]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LATEST")
    def LATEST(cls) -> "AdotLambdaLayerJavaSdkVersion":
        '''The latest layer version available in this CDK version.

        New versions could
        introduce incompatible changes. Make sure to test them before deploying to production.
        '''
        return typing.cast("AdotLambdaLayerJavaSdkVersion", jsii.sget(cls, "LATEST"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_19_0")
    def V1_19_0(cls) -> "AdotLambdaLayerJavaSdkVersion":
        '''Version 1.19.0.'''
        return typing.cast("AdotLambdaLayerJavaSdkVersion", jsii.sget(cls, "V1_19_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_28_1")
    def V1_28_1(cls) -> "AdotLambdaLayerJavaSdkVersion":
        '''Version 1.28.1.'''
        return typing.cast("AdotLambdaLayerJavaSdkVersion", jsii.sget(cls, "V1_28_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_30_0")
    def V1_30_0(cls) -> "AdotLambdaLayerJavaSdkVersion":
        '''Version 1.30.0.'''
        return typing.cast("AdotLambdaLayerJavaSdkVersion", jsii.sget(cls, "V1_30_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_31_0")
    def V1_31_0(cls) -> "AdotLambdaLayerJavaSdkVersion":
        '''Version 1.31.0.'''
        return typing.cast("AdotLambdaLayerJavaSdkVersion", jsii.sget(cls, "V1_31_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_32_0")
    def V1_32_0(cls) -> "AdotLambdaLayerJavaSdkVersion":
        '''Version 1.32.0.'''
        return typing.cast("AdotLambdaLayerJavaSdkVersion", jsii.sget(cls, "V1_32_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_32_0_1")
    def V1_32_0_1(cls) -> "AdotLambdaLayerJavaSdkVersion":
        '''Version 1.32.0.'''
        return typing.cast("AdotLambdaLayerJavaSdkVersion", jsii.sget(cls, "V1_32_0_1"))

    @builtins.property
    @jsii.member(jsii_name="layerVersion")
    def _layer_version(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "layerVersion"))


class AdotLambdaLayerPythonSdkVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.AdotLambdaLayerPythonSdkVersion",
):
    '''The collection of versions of the ADOT Lambda Layer for Python SDK.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        adot_lambda_layer_python_sdk_version = lambda_.AdotLambdaLayerPythonSdkVersion.LATEST
    '''

    @jsii.member(jsii_name="layerArn")
    def layer_arn(
        self,
        scope: _constructs_77d1e7e8.IConstruct,
        architecture: "Architecture",
    ) -> builtins.str:
        '''The ARN of the Lambda layer.

        :param scope: The binding scope. Usually this is the stack where the Lambda layer is bound to
        :param architecture: The architecture of the Lambda layer (either X86_64 or ARM_64).
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__db562fe97fe7d4b3d224b7c9b4186de9fdd07e2d7ddf08d7fc51a581423089f4)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
        return typing.cast(builtins.str, jsii.invoke(self, "layerArn", [scope, architecture]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="LATEST")
    def LATEST(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''The latest layer version available in this CDK version.

        New versions could
        introduce incompatible changes. Make sure to test them before deploying to production.
        '''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "LATEST"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_13_0")
    def V1_13_0(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.13.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_13_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_14_0")
    def V1_14_0(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.14.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_14_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_15_0")
    def V1_15_0(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.15.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_15_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_16_0")
    def V1_16_0(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.16.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_16_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_17_0")
    def V1_17_0(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.17.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_17_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_18_0")
    def V1_18_0(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.18.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_18_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_19_0_1")
    def V1_19_0_1(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.19.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_19_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_20_0")
    def V1_20_0(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.20.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_20_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_20_0_1")
    def V1_20_0_1(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.20.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_20_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_21_0")
    def V1_21_0(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.21.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_21_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_24_0")
    def V1_24_0(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.24.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_24_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_25_0")
    def V1_25_0(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.25.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_25_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_29_0")
    def V1_29_0(cls) -> "AdotLambdaLayerPythonSdkVersion":
        '''Version 1.29.0.'''
        return typing.cast("AdotLambdaLayerPythonSdkVersion", jsii.sget(cls, "V1_29_0"))

    @builtins.property
    @jsii.member(jsii_name="layerVersion")
    def _layer_version(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "layerVersion"))


class AdotLayerVersion(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_lambda.AdotLayerVersion",
):
    '''An ADOT Lambda layer version that's specific to a lambda layer type and an architecture.

    :exampleMetadata: infused

    Example::

        from aws_cdk.aws_lambda import AdotLambdaExecWrapper, AdotLayerVersion, AdotLambdaLayerJavaScriptSdkVersion
        
        
        fn = lambda_.Function(self, "MyFunction",
            runtime=lambda_.Runtime.NODEJS_18_X,
            handler="index.handler",
            code=lambda_.Code.from_inline("exports.handler = function(event, ctx, cb) { return cb(null, \"hi\"); }"),
            adot_instrumentation=lambda.AdotInstrumentationConfig(
                layer_version=AdotLayerVersion.from_java_script_sdk_layer_version(AdotLambdaLayerJavaScriptSdkVersion.LATEST),
                exec_wrapper=AdotLambdaExecWrapper.REGULAR_HANDLER
            )
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromGenericLayerVersion")
    @builtins.classmethod
    def from_generic_layer_version(
        cls,
        version: AdotLambdaLayerGenericVersion,
    ) -> "AdotLayerVersion":
        '''The ADOT Lambda layer for generic use cases.

        :param version: The version of the Lambda layer to use.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b21d606ea4b9273fdfb1a2ec2e77276ab2f02bee2a0d9ed2f36e2ceb9c14ac6a)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        return typing.cast("AdotLayerVersion", jsii.sinvoke(cls, "fromGenericLayerVersion", [version]))

    @jsii.member(jsii_name="fromJavaAutoInstrumentationLayerVersion")
    @builtins.classmethod
    def from_java_auto_instrumentation_layer_version(
        cls,
        version: AdotLambdaLayerJavaAutoInstrumentationVersion,
    ) -> "AdotLayerVersion":
        '''The ADOT Lambda layer for Java auto instrumentation.

        :param version: The version of the Lambda layer to use.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd7144bdaaa47642946a3219733952b7667e7d0e7709ae1e24a1e67f11313182)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        return typing.cast("AdotLayerVersion", jsii.sinvoke(cls, "fromJavaAutoInstrumentationLayerVersion", [version]))

    @jsii.member(jsii_name="fromJavaScriptSdkLayerVersion")
    @builtins.classmethod
    def from_java_script_sdk_layer_version(
        cls,
        version: AdotLambdaLayerJavaScriptSdkVersion,
    ) -> "AdotLayerVersion":
        '''The ADOT Lambda layer for JavaScript SDK.

        :param version: The version of the Lambda layer to use.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1dabb45e8273fd1eba3e2fdeefc9cfe2607e7b5b0a33636f7dfb07e525d63e5d)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        return typing.cast("AdotLayerVersion", jsii.sinvoke(cls, "fromJavaScriptSdkLayerVersion", [version]))

    @jsii.member(jsii_name="fromJavaSdkLayerVersion")
    @builtins.classmethod
    def from_java_sdk_layer_version(
        cls,
        version: AdotLambdaLayerJavaSdkVersion,
    ) -> "AdotLayerVersion":
        '''The ADOT Lambda layer for Java SDK.

        :param version: The version of the Lambda layer to use.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cc888360b9d1867a4121f84478cb6e55d5c98d39b05b1787d140cb02f43daa46)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        return typing.cast("AdotLayerVersion", jsii.sinvoke(cls, "fromJavaSdkLayerVersion", [version]))

    @jsii.member(jsii_name="fromPythonSdkLayerVersion")
    @builtins.classmethod
    def from_python_sdk_layer_version(
        cls,
        version: AdotLambdaLayerPythonSdkVersion,
    ) -> "AdotLayerVersion":
        '''The ADOT Lambda layer for Python SDK.

        :param version: The version of the Lambda layer to use.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__14b0848168a6a70760882bc85e6c4e46042dec0b3b442eb24e880cd3c693e960)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        return typing.cast("AdotLayerVersion", jsii.sinvoke(cls, "fromPythonSdkLayerVersion", [version]))


class _AdotLayerVersionProxy(AdotLayerVersion):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AdotLayerVersion).__jsii_proxy_class__ = lambda : _AdotLayerVersionProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.AliasAttributes",
    jsii_struct_bases=[],
    name_mapping={"alias_name": "aliasName", "alias_version": "aliasVersion"},
)
class AliasAttributes:
    def __init__(self, *, alias_name: builtins.str, alias_version: "IVersion") -> None:
        '''
        :param alias_name: 
        :param alias_version: 

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            # version: lambda.Version
            
            alias_attributes = lambda.AliasAttributes(
                alias_name="aliasName",
                alias_version=version
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6cdbbf7a3bdd430d15235f9032173a36fa88f7e2593da8845460383e00559664)
            check_type(argname="argument alias_name", value=alias_name, expected_type=type_hints["alias_name"])
            check_type(argname="argument alias_version", value=alias_version, expected_type=type_hints["alias_version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "alias_name": alias_name,
            "alias_version": alias_version,
        }

    @builtins.property
    def alias_name(self) -> builtins.str:
        result = self._values.get("alias_name")
        assert result is not None, "Required property 'alias_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def alias_version(self) -> "IVersion":
        result = self._values.get("alias_version")
        assert result is not None, "Required property 'alias_version' is missing"
        return typing.cast("IVersion", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AliasAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_lambda.ApplicationLogLevel")
class ApplicationLogLevel(enum.Enum):
    '''Lambda service automatically captures logs generated by the function code (known as application logs) and sends these logs to a default CloudWatch log group named after the Lambda function.

    :exampleMetadata: infused

    Example::

        from aws_cdk.aws_logs import ILogGroup
        
        # log_group: ILogGroup
        
        
        lambda_.Function(self, "Lambda",
            code=lambda_.InlineCode("foo"),
            handler="index.handler",
            runtime=lambda_.Runtime.NODEJS_18_X,
            logging_format=lambda_.LoggingFormat.JSON,
            system_log_level_v2=lambda_.SystemLogLevel.INFO,
            application_log_level_v2=lambda_.ApplicationLogLevel.INFO,
            log_group=log_group
        )
    '''

    INFO = "INFO"
    '''Lambda will capture only logs at info level.'''
    DEBUG = "DEBUG"
    '''Lambda will capture only logs at debug level.'''
    WARN = "WARN"
    '''Lambda will capture only logs at warn level.'''
    TRACE = "TRACE"
    '''Lambda will capture only logs at trace level.'''
    ERROR = "ERROR"
    '''Lambda will capture only logs at error level.'''
    FATAL = "FATAL"
    '''Lambda will capture only logs at fatal level.'''


class Architecture(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.Architecture",
):
    '''Architectures supported by AWS Lambda.

    :exampleMetadata: infused

    Example::

        lambda_.Function(self, "MyFunction",
            runtime=lambda_.Runtime.NODEJS_18_X,
            handler="index.handler",
            architecture=lambda_.Architecture.ARM_64,
            code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
            insights_version=lambda_.LambdaInsightsVersion.VERSION_1_0_119_0
        )
    '''

    @jsii.member(jsii_name="custom")
    @builtins.classmethod
    def custom(
        cls,
        name: builtins.str,
        docker_platform: typing.Optional[builtins.str] = None,
    ) -> "Architecture":
        '''Used to specify a custom architecture name.

        Use this if the architecture name is not yet supported by the CDK.

        :param name: the architecture name as recognized by AWS Lambda.
        :param docker_platform: the platform to use for this architecture when building with Docker.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9f1ebe6bea035d7784b74a20153e5c0208119e4b6bf9ac0272ce6133e54ca58e)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument docker_platform", value=docker_platform, expected_type=type_hints["docker_platform"])
        return typing.cast("Architecture", jsii.sinvoke(cls, "custom", [name, docker_platform]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Returns a string representation of the architecture using the name.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ARM_64")
    def ARM_64(cls) -> "Architecture":
        '''64 bit architecture with the ARM instruction set.'''
        return typing.cast("Architecture", jsii.sget(cls, "ARM_64"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="X86_64")
    def X86_64(cls) -> "Architecture":
        '''64 bit architecture with x86 instruction set.'''
        return typing.cast("Architecture", jsii.sget(cls, "X86_64"))

    @builtins.property
    @jsii.member(jsii_name="dockerPlatform")
    def docker_platform(self) -> builtins.str:
        '''The platform to use for this architecture when building with Docker.'''
        return typing.cast(builtins.str, jsii.get(self, "dockerPlatform"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the architecture as recognized by the AWS Lambda service APIs.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.AssetImageCodeProps",
    jsii_struct_bases=[_DockerImageAssetOptions_9580cd76],
    name_mapping={
        "exclude": "exclude",
        "follow_symlinks": "followSymlinks",
        "ignore_mode": "ignoreMode",
        "extra_hash": "extraHash",
        "asset_name": "assetName",
        "build_args": "buildArgs",
        "build_secrets": "buildSecrets",
        "build_ssh": "buildSsh",
        "cache_disabled": "cacheDisabled",
        "cache_from": "cacheFrom",
        "cache_to": "cacheTo",
        "display_name": "displayName",
        "file": "file",
        "invalidation": "invalidation",
        "network_mode": "networkMode",
        "outputs": "outputs",
        "platform": "platform",
        "target": "target",
        "cmd": "cmd",
        "entrypoint": "entrypoint",
        "working_directory": "workingDirectory",
    },
)
class AssetImageCodeProps(_DockerImageAssetOptions_9580cd76):
    def __init__(
        self,
        *,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_047ec1f6] = None,
        ignore_mode: typing.Optional[_IgnoreMode_655a98e8] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        asset_name: typing.Optional[builtins.str] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        build_secrets: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        build_ssh: typing.Optional[builtins.str] = None,
        cache_disabled: typing.Optional[builtins.bool] = None,
        cache_from: typing.Optional[typing.Sequence[typing.Union[_DockerCacheOption_58ef18ca, typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_to: typing.Optional[typing.Union[_DockerCacheOption_58ef18ca, typing.Dict[builtins.str, typing.Any]]] = None,
        display_name: typing.Optional[builtins.str] = None,
        file: typing.Optional[builtins.str] = None,
        invalidation: typing.Optional[typing.Union[_DockerImageAssetInvalidationOptions_4deb8d45, typing.Dict[builtins.str, typing.Any]]] = None,
        network_mode: typing.Optional[_NetworkMode_897e5081] = None,
        outputs: typing.Optional[typing.Sequence[builtins.str]] = None,
        platform: typing.Optional[_Platform_d16f3cf1] = None,
        target: typing.Optional[builtins.str] = None,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties to initialize a new AssetImage.

        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        :param extra_hash: Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param asset_name: Unique identifier of the docker image asset and its potential revisions. Required if using AppScopedStagingSynthesizer. Default: - no asset name
        :param build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param build_secrets: Build secrets. Docker BuildKit must be enabled to use build secrets. Default: - no build secrets
        :param build_ssh: SSH agent socket or keys to pass to the ``docker build`` command. Docker BuildKit must be enabled to use the ssh flag Default: - no --ssh flag
        :param cache_disabled: Disable the cache and pass ``--no-cache`` to the ``docker build`` command. Default: - cache is used
        :param cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from options are passed to the build command
        :param cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to options are passed to the build command
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. If the same asset is added multiple times, the display name of the first occurrence is used. If ``assetName`` is given, it will also be used as the default ``displayName``. Otherwise, the default is the construct path of the ImageAsset construct, with respect to the enclosing stack. If the asset is produced by a construct helper function (such as ``lambda.Code.fromAssetImage()``), this will look like ``MyFunction/AssetImage``. We use the stack-relative construct path so that in the common case where you have multiple stacks with the same asset, we won't show something like ``/MyBetaStack/MyFunction/Code`` when you are actually deploying to production. Default: - Stack-relative construct path
        :param file: Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param invalidation: Options to control which parameters are used to invalidate the asset hash. Default: - hash all parameters
        :param network_mode: Networking mode for the RUN commands during build. Support docker API 1.25+. Default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        :param outputs: Outputs to pass to the ``docker build`` command. Default: - no outputs are passed to the build command (default outputs are used)
        :param platform: Platform to build for. *Requires Docker Buildx*. Default: - no platform specified (the current machine architecture will be used)
        :param target: Docker target to build to. Default: - no target
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_ecr_assets as ecr_assets
            from aws_cdk import aws_lambda as lambda_
            
            # network_mode: ecr_assets.NetworkMode
            # platform: ecr_assets.Platform
            
            asset_image_code_props = lambda.AssetImageCodeProps(
                asset_name="assetName",
                build_args={
                    "build_args_key": "buildArgs"
                },
                build_secrets={
                    "build_secrets_key": "buildSecrets"
                },
                build_ssh="buildSsh",
                cache_disabled=False,
                cache_from=[ecr_assets.DockerCacheOption(
                    type="type",
            
                    # the properties below are optional
                    params={
                        "params_key": "params"
                    }
                )],
                cache_to=ecr_assets.DockerCacheOption(
                    type="type",
            
                    # the properties below are optional
                    params={
                        "params_key": "params"
                    }
                ),
                cmd=["cmd"],
                display_name="displayName",
                entrypoint=["entrypoint"],
                exclude=["exclude"],
                extra_hash="extraHash",
                file="file",
                follow_symlinks=cdk.SymlinkFollowMode.NEVER,
                ignore_mode=cdk.IgnoreMode.GLOB,
                invalidation=ecr_assets.DockerImageAssetInvalidationOptions(
                    build_args=False,
                    build_secrets=False,
                    build_ssh=False,
                    extra_hash=False,
                    file=False,
                    network_mode=False,
                    outputs=False,
                    platform=False,
                    repository_name=False,
                    target=False
                ),
                network_mode=network_mode,
                outputs=["outputs"],
                platform=platform,
                target="target",
                working_directory="workingDirectory"
            )
        '''
        if isinstance(cache_to, dict):
            cache_to = _DockerCacheOption_58ef18ca(**cache_to)
        if isinstance(invalidation, dict):
            invalidation = _DockerImageAssetInvalidationOptions_4deb8d45(**invalidation)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ec4c3fb249340fe6042f51bdb6d1686308ab9ec697f2f453cba22fd221d8ad42)
            check_type(argname="argument exclude", value=exclude, expected_type=type_hints["exclude"])
            check_type(argname="argument follow_symlinks", value=follow_symlinks, expected_type=type_hints["follow_symlinks"])
            check_type(argname="argument ignore_mode", value=ignore_mode, expected_type=type_hints["ignore_mode"])
            check_type(argname="argument extra_hash", value=extra_hash, expected_type=type_hints["extra_hash"])
            check_type(argname="argument asset_name", value=asset_name, expected_type=type_hints["asset_name"])
            check_type(argname="argument build_args", value=build_args, expected_type=type_hints["build_args"])
            check_type(argname="argument build_secrets", value=build_secrets, expected_type=type_hints["build_secrets"])
            check_type(argname="argument build_ssh", value=build_ssh, expected_type=type_hints["build_ssh"])
            check_type(argname="argument cache_disabled", value=cache_disabled, expected_type=type_hints["cache_disabled"])
            check_type(argname="argument cache_from", value=cache_from, expected_type=type_hints["cache_from"])
            check_type(argname="argument cache_to", value=cache_to, expected_type=type_hints["cache_to"])
            check_type(argname="argument display_name", value=display_name, expected_type=type_hints["display_name"])
            check_type(argname="argument file", value=file, expected_type=type_hints["file"])
            check_type(argname="argument invalidation", value=invalidation, expected_type=type_hints["invalidation"])
            check_type(argname="argument network_mode", value=network_mode, expected_type=type_hints["network_mode"])
            check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            check_type(argname="argument platform", value=platform, expected_type=type_hints["platform"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
            check_type(argname="argument cmd", value=cmd, expected_type=type_hints["cmd"])
            check_type(argname="argument entrypoint", value=entrypoint, expected_type=type_hints["entrypoint"])
            check_type(argname="argument working_directory", value=working_directory, expected_type=type_hints["working_directory"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if exclude is not None:
            self._values["exclude"] = exclude
        if follow_symlinks is not None:
            self._values["follow_symlinks"] = follow_symlinks
        if ignore_mode is not None:
            self._values["ignore_mode"] = ignore_mode
        if extra_hash is not None:
            self._values["extra_hash"] = extra_hash
        if asset_name is not None:
            self._values["asset_name"] = asset_name
        if build_args is not None:
            self._values["build_args"] = build_args
        if build_secrets is not None:
            self._values["build_secrets"] = build_secrets
        if build_ssh is not None:
            self._values["build_ssh"] = build_ssh
        if cache_disabled is not None:
            self._values["cache_disabled"] = cache_disabled
        if cache_from is not None:
            self._values["cache_from"] = cache_from
        if cache_to is not None:
            self._values["cache_to"] = cache_to
        if display_name is not None:
            self._values["display_name"] = display_name
        if file is not None:
            self._values["file"] = file
        if invalidation is not None:
            self._values["invalidation"] = invalidation
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if outputs is not None:
            self._values["outputs"] = outputs
        if platform is not None:
            self._values["platform"] = platform
        if target is not None:
            self._values["target"] = target
        if cmd is not None:
            self._values["cmd"] = cmd
        if entrypoint is not None:
            self._values["entrypoint"] = entrypoint
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def exclude(self) -> typing.Optional[typing.List[builtins.str]]:
        '''File paths matching the patterns will be excluded.

        See ``ignoreMode`` to set the matching behavior.
        Has no effect on Assets bundled using the ``bundling`` property.

        :default: - nothing is excluded
        '''
        result = self._values.get("exclude")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def follow_symlinks(self) -> typing.Optional[_SymlinkFollowMode_047ec1f6]:
        '''A strategy for how to handle symlinks.

        :default: SymlinkFollowMode.NEVER
        '''
        result = self._values.get("follow_symlinks")
        return typing.cast(typing.Optional[_SymlinkFollowMode_047ec1f6], result)

    @builtins.property
    def ignore_mode(self) -> typing.Optional[_IgnoreMode_655a98e8]:
        '''The ignore behavior to use for ``exclude`` patterns.

        :default: IgnoreMode.GLOB
        '''
        result = self._values.get("ignore_mode")
        return typing.cast(typing.Optional[_IgnoreMode_655a98e8], result)

    @builtins.property
    def extra_hash(self) -> typing.Optional[builtins.str]:
        '''Extra information to encode into the fingerprint (e.g. build instructions and other inputs).

        :default: - hash is only based on source content
        '''
        result = self._values.get("extra_hash")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def asset_name(self) -> typing.Optional[builtins.str]:
        '''Unique identifier of the docker image asset and its potential revisions.

        Required if using AppScopedStagingSynthesizer.

        :default: - no asset name
        '''
        result = self._values.get("asset_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def build_args(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Build args to pass to the ``docker build`` command.

        Since Docker build arguments are resolved before deployment, keys and
        values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or
        ``queue.queueUrl``).

        :default: - no build args are passed
        '''
        result = self._values.get("build_args")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def build_secrets(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Build secrets.

        Docker BuildKit must be enabled to use build secrets.

        :default: - no build secrets

        :see: https://docs.docker.com/build/buildkit/

        Example::

            from aws_cdk import DockerBuildSecret
            
            
            build_secrets = {
                "MY_SECRET": DockerBuildSecret.from_src("file.txt")
            }
        '''
        result = self._values.get("build_secrets")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def build_ssh(self) -> typing.Optional[builtins.str]:
        '''SSH agent socket or keys to pass to the ``docker build`` command.

        Docker BuildKit must be enabled to use the ssh flag

        :default: - no --ssh flag

        :see: https://docs.docker.com/build/buildkit/
        '''
        result = self._values.get("build_ssh")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cache_disabled(self) -> typing.Optional[builtins.bool]:
        '''Disable the cache and pass ``--no-cache`` to the ``docker build`` command.

        :default: - cache is used
        '''
        result = self._values.get("cache_disabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cache_from(self) -> typing.Optional[typing.List[_DockerCacheOption_58ef18ca]]:
        '''Cache from options to pass to the ``docker build`` command.

        :default: - no cache from options are passed to the build command

        :see: https://docs.docker.com/build/cache/backends/
        '''
        result = self._values.get("cache_from")
        return typing.cast(typing.Optional[typing.List[_DockerCacheOption_58ef18ca]], result)

    @builtins.property
    def cache_to(self) -> typing.Optional[_DockerCacheOption_58ef18ca]:
        '''Cache to options to pass to the ``docker build`` command.

        :default: - no cache to options are passed to the build command

        :see: https://docs.docker.com/build/cache/backends/
        '''
        result = self._values.get("cache_to")
        return typing.cast(typing.Optional[_DockerCacheOption_58ef18ca], result)

    @builtins.property
    def display_name(self) -> typing.Optional[builtins.str]:
        '''A display name for this asset.

        If supplied, the display name will be used in locations where the asset
        identifier is printed, like in the CLI progress information. If the same
        asset is added multiple times, the display name of the first occurrence is
        used.

        If ``assetName`` is given, it will also be used as the default ``displayName``.
        Otherwise, the default is the construct path of the ImageAsset construct,
        with respect to the enclosing stack. If the asset is produced by a
        construct helper function (such as ``lambda.Code.fromAssetImage()``), this
        will look like ``MyFunction/AssetImage``.

        We use the stack-relative construct path so that in the common case where
        you have multiple stacks with the same asset, we won't show something like
        ``/MyBetaStack/MyFunction/Code`` when you are actually deploying to
        production.

        :default: - Stack-relative construct path
        '''
        result = self._values.get("display_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def file(self) -> typing.Optional[builtins.str]:
        '''Path to the Dockerfile (relative to the directory).

        :default: 'Dockerfile'
        '''
        result = self._values.get("file")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def invalidation(
        self,
    ) -> typing.Optional[_DockerImageAssetInvalidationOptions_4deb8d45]:
        '''Options to control which parameters are used to invalidate the asset hash.

        :default: - hash all parameters
        '''
        result = self._values.get("invalidation")
        return typing.cast(typing.Optional[_DockerImageAssetInvalidationOptions_4deb8d45], result)

    @builtins.property
    def network_mode(self) -> typing.Optional[_NetworkMode_897e5081]:
        '''Networking mode for the RUN commands during build.

        Support docker API 1.25+.

        :default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional[_NetworkMode_897e5081], result)

    @builtins.property
    def outputs(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Outputs to pass to the ``docker build`` command.

        :default: - no outputs are passed to the build command (default outputs are used)

        :see: https://docs.docker.com/engine/reference/commandline/build/#custom-build-outputs
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def platform(self) -> typing.Optional[_Platform_d16f3cf1]:
        '''Platform to build for.

        *Requires Docker Buildx*.

        :default: - no platform specified (the current machine architecture will be used)
        '''
        result = self._values.get("platform")
        return typing.cast(typing.Optional[_Platform_d16f3cf1], result)

    @builtins.property
    def target(self) -> typing.Optional[builtins.str]:
        '''Docker target to build to.

        :default: - no target
        '''
        result = self._values.get("target")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cmd(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the CMD on the specified Docker image or Dockerfile.

        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the CMD specified in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#cmd
        '''
        result = self._values.get("cmd")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entrypoint(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile.

        An ENTRYPOINT allows you to configure a container that will run as an executable.
        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the ENTRYPOINT in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        '''
        result = self._values.get("entrypoint")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''Specify or override the WORKDIR on the specified Docker image or Dockerfile.

        A WORKDIR allows you to configure the working directory the container will use.

        :default: - use the WORKDIR in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#workdir
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssetImageCodeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.AutoScalingOptions",
    jsii_struct_bases=[],
    name_mapping={"max_capacity": "maxCapacity", "min_capacity": "minCapacity"},
)
class AutoScalingOptions:
    def __init__(
        self,
        *,
        max_capacity: jsii.Number,
        min_capacity: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for enabling Lambda autoscaling.

        :param max_capacity: Maximum capacity to scale to.
        :param min_capacity: Minimum capacity to scale to. Default: 1

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_autoscaling as autoscaling
            
            # fn: lambda.Function
            
            alias = fn.add_alias("prod")
            
            # Create AutoScaling target
            as = alias.add_auto_scaling(max_capacity=50)
            
            # Configure Target Tracking
            as.scale_on_utilization(
                utilization_target=0.5
            )
            
            # Configure Scheduled Scaling
            as.scale_on_schedule("ScaleUpInTheMorning",
                schedule=autoscaling.Schedule.cron(hour="8", minute="0"),
                min_capacity=20
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d87a0ce22b000498263273a478e075d6808ca7f5931890c7a99744eb48762489)
            check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
            check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "max_capacity": max_capacity,
        }
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity

    @builtins.property
    def max_capacity(self) -> jsii.Number:
        '''Maximum capacity to scale to.'''
        result = self._values.get("max_capacity")
        assert result is not None, "Required property 'max_capacity' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def min_capacity(self) -> typing.Optional[jsii.Number]:
        '''Minimum capacity to scale to.

        :default: 1
        '''
        result = self._values.get("min_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AutoScalingOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.BucketOptions",
    jsii_struct_bases=[],
    name_mapping={"object_version": "objectVersion", "source_kms_key": "sourceKMSKey"},
)
class BucketOptions:
    def __init__(
        self,
        *,
        object_version: typing.Optional[builtins.str] = None,
        source_kms_key: typing.Optional[_IKey_5f11635f] = None,
    ) -> None:
        '''Optional parameters for creating code using bucket.

        :param object_version: Optional S3 object version.
        :param source_kms_key: The ARN of the KMS key used to encrypt the handler code. Default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.

        :exampleMetadata: infused

        Example::

            from aws_cdk.aws_kms import Key
            import aws_cdk.aws_s3 as s3
            # key: Key
            
            
            bucket = s3.Bucket(self, "Bucket")
            
            options = {
                "source_kMSKey": key
            }
            fn_bucket = lambda_.Function(self, "myFunction2",
                runtime=lambda_.Runtime.NODEJS_LATEST,
                handler="index.handler",
                code=lambda_.Code.from_bucket_v2(bucket, "python-lambda-handler.zip", options)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea5994c9827298565c305f3f7f771ab57a19a60665d41006e56da4741c2d0d56)
            check_type(argname="argument object_version", value=object_version, expected_type=type_hints["object_version"])
            check_type(argname="argument source_kms_key", value=source_kms_key, expected_type=type_hints["source_kms_key"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if object_version is not None:
            self._values["object_version"] = object_version
        if source_kms_key is not None:
            self._values["source_kms_key"] = source_kms_key

    @builtins.property
    def object_version(self) -> typing.Optional[builtins.str]:
        '''Optional S3 object version.'''
        result = self._values.get("object_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_kms_key(self) -> typing.Optional[_IKey_5f11635f]:
        '''The ARN of the KMS key used to encrypt the handler code.

        :default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.
        '''
        result = self._values.get("source_kms_key")
        return typing.cast(typing.Optional[_IKey_5f11635f], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BucketOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnAlias(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.CfnAlias",
):
    '''The ``AWS::Lambda::Alias`` resource creates an `alias <https://docs.aws.amazon.com/lambda/latest/dg/configuration-aliases.html>`_ for a Lambda function version. Use aliases to provide clients with a function identifier that you can update to invoke a different version.

    You can also map an alias to split invocation requests between two versions. Use the ``RoutingConfig`` parameter to specify a second version and the percentage of invocation requests that it receives.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html
    :cloudformationResource: AWS::Lambda::Alias
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        cfn_alias = lambda_.CfnAlias(self, "MyCfnAlias",
            function_name="functionName",
            function_version="functionVersion",
            name="name",
        
            # the properties below are optional
            description="description",
            provisioned_concurrency_config=lambda.CfnAlias.ProvisionedConcurrencyConfigurationProperty(
                provisioned_concurrent_executions=123
            ),
            routing_config=lambda.CfnAlias.AliasRoutingConfigurationProperty(
                additional_version_weights=[lambda.CfnAlias.VersionWeightProperty(
                    function_version="functionVersion",
                    function_weight=123
                )]
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        function_name: builtins.str,
        function_version: builtins.str,
        name: builtins.str,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrency_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAlias.ProvisionedConcurrencyConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        routing_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAlias.AliasRoutingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param function_name: The name or ARN of the Lambda function. **Name formats** - *Function name* - ``MyFunction`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Partial ARN* - ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param function_version: The function version that the alias invokes.
        :param name: The name of the alias.
        :param description: A description of the alias.
        :param provisioned_concurrency_config: Specifies a `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/configuration-concurrency.html>`_ configuration for a function's alias.
        :param routing_config: The `routing configuration <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ of the alias.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__681471c67952a7e725f76572ad9bf09e1c634a81914690dff68e934c039fd2f9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAliasProps(
            function_name=function_name,
            function_version=function_version,
            name=name,
            description=description,
            provisioned_concurrency_config=provisioned_concurrency_config,
            routing_config=routing_config,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9899ddb1059c2ccb62cea6384916f2265c3afd08fe74e95deae2f78e81be791d)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0ad15d3a05afb91167199517fea0b67063ed4ea8695420cd25afb222f58a3457)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrAliasArn")
    def attr_alias_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the alias.

        :cloudformationAttribute: AliasArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAliasArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name or ARN of the Lambda function.'''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f067ace8c43eedf007910f5f14492d64963599dbdd0ffec0108edfa512e589b8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="functionVersion")
    def function_version(self) -> builtins.str:
        '''The function version that the alias invokes.'''
        return typing.cast(builtins.str, jsii.get(self, "functionVersion"))

    @function_version.setter
    def function_version(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6b2ddeca57766bd54f58a827a23c3fbf1bd440bb65695c5a2a7d12b0f6f67301)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionVersion", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the alias.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a9ccb417a1dffcbc53b65cf21c87acfcd6f0963b3703129d5885a2ca11a9ddd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the alias.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__33cbeeca7420fade9e2c53654443f9ea6c3e1e129b2edb1ff6438a6f240660f9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="provisionedConcurrencyConfig")
    def provisioned_concurrency_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAlias.ProvisionedConcurrencyConfigurationProperty"]]:
        '''Specifies a `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/configuration-concurrency.html>`_ configuration for a function's alias.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAlias.ProvisionedConcurrencyConfigurationProperty"]], jsii.get(self, "provisionedConcurrencyConfig"))

    @provisioned_concurrency_config.setter
    def provisioned_concurrency_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAlias.ProvisionedConcurrencyConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a958bc2b3267e229d2253296cdbb66337d157b0c6e2ec1711eb372278993a948)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "provisionedConcurrencyConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="routingConfig")
    def routing_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAlias.AliasRoutingConfigurationProperty"]]:
        '''The `routing configuration <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ of the alias.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAlias.AliasRoutingConfigurationProperty"]], jsii.get(self, "routingConfig"))

    @routing_config.setter
    def routing_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAlias.AliasRoutingConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7525467ed4a8de7249cf79d24091ae59273412647fd30558b48ead6b615e5f51)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routingConfig", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnAlias.AliasRoutingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"additional_version_weights": "additionalVersionWeights"},
    )
    class AliasRoutingConfigurationProperty:
        def __init__(
            self,
            *,
            additional_version_weights: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAlias.VersionWeightProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The `traffic-shifting <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ configuration of a Lambda function alias.

            :param additional_version_weights: The second version, and the percentage of traffic that's routed to it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-aliasroutingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                alias_routing_configuration_property = lambda.CfnAlias.AliasRoutingConfigurationProperty(
                    additional_version_weights=[lambda.CfnAlias.VersionWeightProperty(
                        function_version="functionVersion",
                        function_weight=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6cdf547e432e98f71c10bd59a276cb1a69fa82898a8f92c15fff9c69bf506ce0)
                check_type(argname="argument additional_version_weights", value=additional_version_weights, expected_type=type_hints["additional_version_weights"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if additional_version_weights is not None:
                self._values["additional_version_weights"] = additional_version_weights

        @builtins.property
        def additional_version_weights(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAlias.VersionWeightProperty"]]]]:
            '''The second version, and the percentage of traffic that's routed to it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-aliasroutingconfiguration.html#cfn-lambda-alias-aliasroutingconfiguration-additionalversionweights
            '''
            result = self._values.get("additional_version_weights")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAlias.VersionWeightProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AliasRoutingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnAlias.ProvisionedConcurrencyConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "provisioned_concurrent_executions": "provisionedConcurrentExecutions",
        },
    )
    class ProvisionedConcurrencyConfigurationProperty:
        def __init__(self, *, provisioned_concurrent_executions: jsii.Number) -> None:
            '''A provisioned concurrency configuration for a function's alias.

            :param provisioned_concurrent_executions: The amount of provisioned concurrency to allocate for the alias.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-provisionedconcurrencyconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                provisioned_concurrency_configuration_property = lambda.CfnAlias.ProvisionedConcurrencyConfigurationProperty(
                    provisioned_concurrent_executions=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b8bfbe857cdca645fbdf0c710f027d5583ced240386c86b80c669d63937de71d)
                check_type(argname="argument provisioned_concurrent_executions", value=provisioned_concurrent_executions, expected_type=type_hints["provisioned_concurrent_executions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "provisioned_concurrent_executions": provisioned_concurrent_executions,
            }

        @builtins.property
        def provisioned_concurrent_executions(self) -> jsii.Number:
            '''The amount of provisioned concurrency to allocate for the alias.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-provisionedconcurrencyconfiguration.html#cfn-lambda-alias-provisionedconcurrencyconfiguration-provisionedconcurrentexecutions
            '''
            result = self._values.get("provisioned_concurrent_executions")
            assert result is not None, "Required property 'provisioned_concurrent_executions' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProvisionedConcurrencyConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnAlias.VersionWeightProperty",
        jsii_struct_bases=[],
        name_mapping={
            "function_version": "functionVersion",
            "function_weight": "functionWeight",
        },
    )
    class VersionWeightProperty:
        def __init__(
            self,
            *,
            function_version: builtins.str,
            function_weight: jsii.Number,
        ) -> None:
            '''The `traffic-shifting <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ configuration of a Lambda function alias.

            :param function_version: The qualifier of the second version.
            :param function_weight: The percentage of traffic that the alias routes to the second version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-versionweight.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                version_weight_property = lambda.CfnAlias.VersionWeightProperty(
                    function_version="functionVersion",
                    function_weight=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__51ab1050cc208f8091f983384e5eac3f313d485d82781cb4d5345b0e59bd56ff)
                check_type(argname="argument function_version", value=function_version, expected_type=type_hints["function_version"])
                check_type(argname="argument function_weight", value=function_weight, expected_type=type_hints["function_weight"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "function_version": function_version,
                "function_weight": function_weight,
            }

        @builtins.property
        def function_version(self) -> builtins.str:
            '''The qualifier of the second version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-versionweight.html#cfn-lambda-alias-versionweight-functionversion
            '''
            result = self._values.get("function_version")
            assert result is not None, "Required property 'function_version' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def function_weight(self) -> jsii.Number:
            '''The percentage of traffic that the alias routes to the second version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-alias-versionweight.html#cfn-lambda-alias-versionweight-functionweight
            '''
            result = self._values.get("function_weight")
            assert result is not None, "Required property 'function_weight' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VersionWeightProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CfnAliasProps",
    jsii_struct_bases=[],
    name_mapping={
        "function_name": "functionName",
        "function_version": "functionVersion",
        "name": "name",
        "description": "description",
        "provisioned_concurrency_config": "provisionedConcurrencyConfig",
        "routing_config": "routingConfig",
    },
)
class CfnAliasProps:
    def __init__(
        self,
        *,
        function_name: builtins.str,
        function_version: builtins.str,
        name: builtins.str,
        description: typing.Optional[builtins.str] = None,
        provisioned_concurrency_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAlias.ProvisionedConcurrencyConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        routing_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAlias.AliasRoutingConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnAlias``.

        :param function_name: The name or ARN of the Lambda function. **Name formats** - *Function name* - ``MyFunction`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Partial ARN* - ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param function_version: The function version that the alias invokes.
        :param name: The name of the alias.
        :param description: A description of the alias.
        :param provisioned_concurrency_config: Specifies a `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/configuration-concurrency.html>`_ configuration for a function's alias.
        :param routing_config: The `routing configuration <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ of the alias.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            cfn_alias_props = lambda.CfnAliasProps(
                function_name="functionName",
                function_version="functionVersion",
                name="name",
            
                # the properties below are optional
                description="description",
                provisioned_concurrency_config=lambda.CfnAlias.ProvisionedConcurrencyConfigurationProperty(
                    provisioned_concurrent_executions=123
                ),
                routing_config=lambda.CfnAlias.AliasRoutingConfigurationProperty(
                    additional_version_weights=[lambda.CfnAlias.VersionWeightProperty(
                        function_version="functionVersion",
                        function_weight=123
                    )]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c720b06c69637819f49e584891d37db068ae9dfc03bd6ad67a979d3f6fb19bc8)
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument function_version", value=function_version, expected_type=type_hints["function_version"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument provisioned_concurrency_config", value=provisioned_concurrency_config, expected_type=type_hints["provisioned_concurrency_config"])
            check_type(argname="argument routing_config", value=routing_config, expected_type=type_hints["routing_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_name": function_name,
            "function_version": function_version,
            "name": name,
        }
        if description is not None:
            self._values["description"] = description
        if provisioned_concurrency_config is not None:
            self._values["provisioned_concurrency_config"] = provisioned_concurrency_config
        if routing_config is not None:
            self._values["routing_config"] = routing_config

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The name or ARN of the Lambda function.

        **Name formats** - *Function name* - ``MyFunction`` .

        - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` .
        - *Partial ARN* - ``123456789012:function:MyFunction`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-functionname
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def function_version(self) -> builtins.str:
        '''The function version that the alias invokes.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-functionversion
        '''
        result = self._values.get("function_version")
        assert result is not None, "Required property 'function_version' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of the alias.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the alias.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def provisioned_concurrency_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAlias.ProvisionedConcurrencyConfigurationProperty]]:
        '''Specifies a `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/configuration-concurrency.html>`_ configuration for a function's alias.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-provisionedconcurrencyconfig
        '''
        result = self._values.get("provisioned_concurrency_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAlias.ProvisionedConcurrencyConfigurationProperty]], result)

    @builtins.property
    def routing_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAlias.AliasRoutingConfigurationProperty]]:
        '''The `routing configuration <https://docs.aws.amazon.com/lambda/latest/dg/lambda-traffic-shifting-using-aliases.html>`_ of the alias.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-alias.html#cfn-lambda-alias-routingconfig
        '''
        result = self._values.get("routing_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAlias.AliasRoutingConfigurationProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAliasProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnCodeSigningConfig(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.CfnCodeSigningConfig",
):
    '''Details about a `Code signing configuration <https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html>`_ .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html
    :cloudformationResource: AWS::Lambda::CodeSigningConfig
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        cfn_code_signing_config = lambda_.CfnCodeSigningConfig(self, "MyCfnCodeSigningConfig",
            allowed_publishers=lambda.CfnCodeSigningConfig.AllowedPublishersProperty(
                signing_profile_version_arns=["signingProfileVersionArns"]
            ),
        
            # the properties below are optional
            code_signing_policies=lambda.CfnCodeSigningConfig.CodeSigningPoliciesProperty(
                untrusted_artifact_on_deployment="untrustedArtifactOnDeployment"
            ),
            description="description",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        allowed_publishers: typing.Union[_IResolvable_da3f097b, typing.Union["CfnCodeSigningConfig.AllowedPublishersProperty", typing.Dict[builtins.str, typing.Any]]],
        code_signing_policies: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCodeSigningConfig.CodeSigningPoliciesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param allowed_publishers: List of allowed publishers.
        :param code_signing_policies: The code signing policy controls the validation failure action for signature mismatch or expiry.
        :param description: Code signing configuration description.
        :param tags: A list of tags to add to the code signing configuration. .. epigraph:: You must have the ``lambda:TagResource`` , ``lambda:UntagResource`` , and ``lambda:ListTags`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__df94ded3fb87e8ca56187dcab5a6bf12d335e2671120df0386f527f736b58b76)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnCodeSigningConfigProps(
            allowed_publishers=allowed_publishers,
            code_signing_policies=code_signing_policies,
            description=description,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2899ed65ab9f2fa74efa2f2afb7573cefa2bf40192b38cf9f93e3ef003342555)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d22d460465e5d835af0767ef182f11f6f1050c97ea008366ba17d7b0c2f0a526)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCodeSigningConfigArn")
    def attr_code_signing_config_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the code signing configuration.

        :cloudformationAttribute: CodeSigningConfigArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCodeSigningConfigArn"))

    @builtins.property
    @jsii.member(jsii_name="attrCodeSigningConfigId")
    def attr_code_signing_config_id(self) -> builtins.str:
        '''The code signing configuration ID.

        :cloudformationAttribute: CodeSigningConfigId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCodeSigningConfigId"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="allowedPublishers")
    def allowed_publishers(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnCodeSigningConfig.AllowedPublishersProperty"]:
        '''List of allowed publishers.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnCodeSigningConfig.AllowedPublishersProperty"], jsii.get(self, "allowedPublishers"))

    @allowed_publishers.setter
    def allowed_publishers(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnCodeSigningConfig.AllowedPublishersProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9bdbcfbe2ec024e11d4aa4a7f377a3ccf0e4dee3dbe6d9fe92735973e52807e0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allowedPublishers", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="codeSigningPolicies")
    def code_signing_policies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCodeSigningConfig.CodeSigningPoliciesProperty"]]:
        '''The code signing policy controls the validation failure action for signature mismatch or expiry.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCodeSigningConfig.CodeSigningPoliciesProperty"]], jsii.get(self, "codeSigningPolicies"))

    @code_signing_policies.setter
    def code_signing_policies(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCodeSigningConfig.CodeSigningPoliciesProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a765b217f65bf891367822f789aa6352de94fbe3b47883e28a1b13ebfcc06fc8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "codeSigningPolicies", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''Code signing configuration description.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2f4fd9a03a33998a08340df7c2ba638ae623513bc416c7f5fd4c5a48e5e2df47)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''A list of tags to add to the code signing configuration.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tags"))

    @tags.setter
    def tags(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__81744cfe981e606b04aa7e4811cb5be2a56198c9076feeda37ccfd2c7437ac1d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnCodeSigningConfig.AllowedPublishersProperty",
        jsii_struct_bases=[],
        name_mapping={"signing_profile_version_arns": "signingProfileVersionArns"},
    )
    class AllowedPublishersProperty:
        def __init__(
            self,
            *,
            signing_profile_version_arns: typing.Sequence[builtins.str],
        ) -> None:
            '''List of signing profiles that can sign a code package.

            :param signing_profile_version_arns: The Amazon Resource Name (ARN) for each of the signing profiles. A signing profile defines a trusted user who can sign a code package.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-codesigningconfig-allowedpublishers.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                allowed_publishers_property = lambda.CfnCodeSigningConfig.AllowedPublishersProperty(
                    signing_profile_version_arns=["signingProfileVersionArns"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__56249a2cdbb0f30384793ee6b6c0129e84d917ef7262e92e841c86b46dad7059)
                check_type(argname="argument signing_profile_version_arns", value=signing_profile_version_arns, expected_type=type_hints["signing_profile_version_arns"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "signing_profile_version_arns": signing_profile_version_arns,
            }

        @builtins.property
        def signing_profile_version_arns(self) -> typing.List[builtins.str]:
            '''The Amazon Resource Name (ARN) for each of the signing profiles.

            A signing profile defines a trusted user who can sign a code package.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-codesigningconfig-allowedpublishers.html#cfn-lambda-codesigningconfig-allowedpublishers-signingprofileversionarns
            '''
            result = self._values.get("signing_profile_version_arns")
            assert result is not None, "Required property 'signing_profile_version_arns' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AllowedPublishersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnCodeSigningConfig.CodeSigningPoliciesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "untrusted_artifact_on_deployment": "untrustedArtifactOnDeployment",
        },
    )
    class CodeSigningPoliciesProperty:
        def __init__(self, *, untrusted_artifact_on_deployment: builtins.str) -> None:
            '''Code signing configuration `policies <https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html#config-codesigning-policies>`_ specify the validation failure action for signature mismatch or expiry.

            :param untrusted_artifact_on_deployment: Code signing configuration policy for deployment validation failure. If you set the policy to ``Enforce`` , Lambda blocks the deployment request if signature validation checks fail. If you set the policy to ``Warn`` , Lambda allows the deployment and creates a CloudWatch log. Default value: ``Warn`` Default: - "Warn"

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-codesigningconfig-codesigningpolicies.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                code_signing_policies_property = lambda.CfnCodeSigningConfig.CodeSigningPoliciesProperty(
                    untrusted_artifact_on_deployment="untrustedArtifactOnDeployment"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dbe1caa9af541ad4af29ef8063007a5015689e44bde28713968e56b9d88cc846)
                check_type(argname="argument untrusted_artifact_on_deployment", value=untrusted_artifact_on_deployment, expected_type=type_hints["untrusted_artifact_on_deployment"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "untrusted_artifact_on_deployment": untrusted_artifact_on_deployment,
            }

        @builtins.property
        def untrusted_artifact_on_deployment(self) -> builtins.str:
            '''Code signing configuration policy for deployment validation failure.

            If you set the policy to ``Enforce`` , Lambda blocks the deployment request if signature validation checks fail. If you set the policy to ``Warn`` , Lambda allows the deployment and creates a CloudWatch log.

            Default value: ``Warn``

            :default: - "Warn"

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-codesigningconfig-codesigningpolicies.html#cfn-lambda-codesigningconfig-codesigningpolicies-untrustedartifactondeployment
            '''
            result = self._values.get("untrusted_artifact_on_deployment")
            assert result is not None, "Required property 'untrusted_artifact_on_deployment' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CodeSigningPoliciesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CfnCodeSigningConfigProps",
    jsii_struct_bases=[],
    name_mapping={
        "allowed_publishers": "allowedPublishers",
        "code_signing_policies": "codeSigningPolicies",
        "description": "description",
        "tags": "tags",
    },
)
class CfnCodeSigningConfigProps:
    def __init__(
        self,
        *,
        allowed_publishers: typing.Union[_IResolvable_da3f097b, typing.Union[CfnCodeSigningConfig.AllowedPublishersProperty, typing.Dict[builtins.str, typing.Any]]],
        code_signing_policies: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCodeSigningConfig.CodeSigningPoliciesProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnCodeSigningConfig``.

        :param allowed_publishers: List of allowed publishers.
        :param code_signing_policies: The code signing policy controls the validation failure action for signature mismatch or expiry.
        :param description: Code signing configuration description.
        :param tags: A list of tags to add to the code signing configuration. .. epigraph:: You must have the ``lambda:TagResource`` , ``lambda:UntagResource`` , and ``lambda:ListTags`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            cfn_code_signing_config_props = lambda.CfnCodeSigningConfigProps(
                allowed_publishers=lambda.CfnCodeSigningConfig.AllowedPublishersProperty(
                    signing_profile_version_arns=["signingProfileVersionArns"]
                ),
            
                # the properties below are optional
                code_signing_policies=lambda.CfnCodeSigningConfig.CodeSigningPoliciesProperty(
                    untrusted_artifact_on_deployment="untrustedArtifactOnDeployment"
                ),
                description="description",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__037c1ad6081170e8f74c6c1909b3fc62bca06393b1bf19efbc4ac600a4274a66)
            check_type(argname="argument allowed_publishers", value=allowed_publishers, expected_type=type_hints["allowed_publishers"])
            check_type(argname="argument code_signing_policies", value=code_signing_policies, expected_type=type_hints["code_signing_policies"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "allowed_publishers": allowed_publishers,
        }
        if code_signing_policies is not None:
            self._values["code_signing_policies"] = code_signing_policies
        if description is not None:
            self._values["description"] = description
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def allowed_publishers(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnCodeSigningConfig.AllowedPublishersProperty]:
        '''List of allowed publishers.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html#cfn-lambda-codesigningconfig-allowedpublishers
        '''
        result = self._values.get("allowed_publishers")
        assert result is not None, "Required property 'allowed_publishers' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnCodeSigningConfig.AllowedPublishersProperty], result)

    @builtins.property
    def code_signing_policies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCodeSigningConfig.CodeSigningPoliciesProperty]]:
        '''The code signing policy controls the validation failure action for signature mismatch or expiry.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html#cfn-lambda-codesigningconfig-codesigningpolicies
        '''
        result = self._values.get("code_signing_policies")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCodeSigningConfig.CodeSigningPoliciesProperty]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''Code signing configuration description.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html#cfn-lambda-codesigningconfig-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''A list of tags to add to the code signing configuration.

        .. epigraph::

           You must have the ``lambda:TagResource`` , ``lambda:UntagResource`` , and ``lambda:ListTags`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-codesigningconfig.html#cfn-lambda-codesigningconfig-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCodeSigningConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnEventInvokeConfig(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.CfnEventInvokeConfig",
):
    '''The ``AWS::Lambda::EventInvokeConfig`` resource configures options for `asynchronous invocation <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html>`_ on a version or an alias.

    By default, Lambda retries an asynchronous invocation twice if the function returns an error. It retains events in a queue for up to six hours. When an event fails all processing attempts or stays in the asynchronous invocation queue for too long, Lambda discards it.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html
    :cloudformationResource: AWS::Lambda::EventInvokeConfig
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        cfn_event_invoke_config = lambda_.CfnEventInvokeConfig(self, "MyCfnEventInvokeConfig",
            function_name="functionName",
            qualifier="qualifier",
        
            # the properties below are optional
            destination_config=lambda.CfnEventInvokeConfig.DestinationConfigProperty(
                on_failure=lambda.CfnEventInvokeConfig.OnFailureProperty(
                    destination="destination"
                ),
                on_success=lambda.CfnEventInvokeConfig.OnSuccessProperty(
                    destination="destination"
                )
            ),
            maximum_event_age_in_seconds=123,
            maximum_retry_attempts=123
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        function_name: builtins.str,
        qualifier: builtins.str,
        destination_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventInvokeConfig.DestinationConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        maximum_event_age_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param function_name: The name of the Lambda function. *Minimum* : ``1`` *Maximum* : ``64`` *Pattern* : ``([a-zA-Z0-9-_]+)``
        :param qualifier: The identifier of a version or alias. - *Version* - A version number. - *Alias* - An alias name. - *Latest* - To specify the unpublished version, use ``$LATEST`` .
        :param destination_config: A destination for events after they have been sent to a function for processing. **Destinations** - *Function* - The Amazon Resource Name (ARN) of a Lambda function. - *Queue* - The ARN of a standard SQS queue. - *Bucket* - The ARN of an Amazon S3 bucket. - *Topic* - The ARN of a standard SNS topic. - *Event Bus* - The ARN of an Amazon EventBridge event bus. .. epigraph:: S3 buckets are supported only for on-failure destinations. To retain records of successful invocations, use another destination type.
        :param maximum_event_age_in_seconds: The maximum age of a request that Lambda sends to a function for processing.
        :param maximum_retry_attempts: The maximum number of times to retry when the function returns an error.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6bd7732654f4625d1267d5f7861f25ea037a2874bbbd321167126c1bbd679189)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnEventInvokeConfigProps(
            function_name=function_name,
            qualifier=qualifier,
            destination_config=destination_config,
            maximum_event_age_in_seconds=maximum_event_age_in_seconds,
            maximum_retry_attempts=maximum_retry_attempts,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__68772001efdb2b23c5142cb8c286127a2a33c7f0f43c8f127a05d07d278f186e)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__78e959868943571bac7c64e3ec5a75b839194ba36d9b2ac39c5da5adccca334e)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function.'''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d970b6f2d883552778b4e43cb6b1092944e8bd2a1ffb21fd476f528028d7eb54)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="qualifier")
    def qualifier(self) -> builtins.str:
        '''The identifier of a version or alias.'''
        return typing.cast(builtins.str, jsii.get(self, "qualifier"))

    @qualifier.setter
    def qualifier(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2ce1d3a4eaa0093ffe4460407606714aac7cbce8d436264b0c50accaff3dda56)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "qualifier", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="destinationConfig")
    def destination_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventInvokeConfig.DestinationConfigProperty"]]:
        '''A destination for events after they have been sent to a function for processing.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventInvokeConfig.DestinationConfigProperty"]], jsii.get(self, "destinationConfig"))

    @destination_config.setter
    def destination_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventInvokeConfig.DestinationConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__07a740f6e4de90b7e4fc9ea661a30309dea4d28202f3a3214fa5ece3946e15b2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "destinationConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="maximumEventAgeInSeconds")
    def maximum_event_age_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The maximum age of a request that Lambda sends to a function for processing.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maximumEventAgeInSeconds"))

    @maximum_event_age_in_seconds.setter
    def maximum_event_age_in_seconds(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f91c293249696b2c6fbb6213764ab137fe93fdc710542158aa261b1c0bed577b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maximumEventAgeInSeconds", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="maximumRetryAttempts")
    def maximum_retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maximumRetryAttempts"))

    @maximum_retry_attempts.setter
    def maximum_retry_attempts(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7f4209ac8600c2a8a1bdc6dc5883ae254a78ad77d828e62cd4984fc618672b76)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maximumRetryAttempts", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventInvokeConfig.DestinationConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"on_failure": "onFailure", "on_success": "onSuccess"},
    )
    class DestinationConfigProperty:
        def __init__(
            self,
            *,
            on_failure: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventInvokeConfig.OnFailureProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            on_success: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventInvokeConfig.OnSuccessProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''A configuration object that specifies the destination of an event after Lambda processes it.

            :param on_failure: The destination configuration for failed invocations. .. epigraph:: When using an Amazon SQS queue as a destination, FIFO queues cannot be used.
            :param on_success: The destination configuration for successful invocations. .. epigraph:: When using an Amazon SQS queue as a destination, FIFO queues cannot be used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-destinationconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                destination_config_property = lambda.CfnEventInvokeConfig.DestinationConfigProperty(
                    on_failure=lambda.CfnEventInvokeConfig.OnFailureProperty(
                        destination="destination"
                    ),
                    on_success=lambda.CfnEventInvokeConfig.OnSuccessProperty(
                        destination="destination"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4f2e54c13e85a13b0eb705c6dc710b9c906a3d436f7593b3dddc05085143a0d0)
                check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
                check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if on_failure is not None:
                self._values["on_failure"] = on_failure
            if on_success is not None:
                self._values["on_success"] = on_success

        @builtins.property
        def on_failure(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventInvokeConfig.OnFailureProperty"]]:
            '''The destination configuration for failed invocations.

            .. epigraph::

               When using an Amazon SQS queue as a destination, FIFO queues cannot be used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-destinationconfig.html#cfn-lambda-eventinvokeconfig-destinationconfig-onfailure
            '''
            result = self._values.get("on_failure")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventInvokeConfig.OnFailureProperty"]], result)

        @builtins.property
        def on_success(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventInvokeConfig.OnSuccessProperty"]]:
            '''The destination configuration for successful invocations.

            .. epigraph::

               When using an Amazon SQS queue as a destination, FIFO queues cannot be used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-destinationconfig.html#cfn-lambda-eventinvokeconfig-destinationconfig-onsuccess
            '''
            result = self._values.get("on_success")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventInvokeConfig.OnSuccessProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DestinationConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventInvokeConfig.OnFailureProperty",
        jsii_struct_bases=[],
        name_mapping={"destination": "destination"},
    )
    class OnFailureProperty:
        def __init__(self, *, destination: builtins.str) -> None:
            '''A destination for events that failed processing.

            :param destination: The Amazon Resource Name (ARN) of the destination resource. To retain records of unsuccessful `asynchronous invocations <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-async-destinations>`_ , you can configure an Amazon SNS topic, Amazon SQS queue, Amazon S3 bucket, Lambda function, or Amazon EventBridge event bus as the destination. To retain records of failed invocations from `Kinesis <https://docs.aws.amazon.com/lambda/latest/dg/with-kinesis.html>`_ , `DynamoDB <https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html>`_ , `self-managed Kafka <https://docs.aws.amazon.com/lambda/latest/dg/with-kafka.html#services-smaa-onfailure-destination>`_ or `Amazon MSK <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-onfailure-destination>`_ , you can configure an Amazon SNS topic, Amazon SQS queue, or Amazon S3 bucket as the destination.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-onfailure.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                on_failure_property = lambda.CfnEventInvokeConfig.OnFailureProperty(
                    destination="destination"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6fecf46f3ba114d6cb3bafc6970878059a09be004b66dbacbb6a2348a4681bc2)
                check_type(argname="argument destination", value=destination, expected_type=type_hints["destination"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "destination": destination,
            }

        @builtins.property
        def destination(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the destination resource.

            To retain records of unsuccessful `asynchronous invocations <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-async-destinations>`_ , you can configure an Amazon SNS topic, Amazon SQS queue, Amazon S3 bucket, Lambda function, or Amazon EventBridge event bus as the destination.

            To retain records of failed invocations from `Kinesis <https://docs.aws.amazon.com/lambda/latest/dg/with-kinesis.html>`_ , `DynamoDB <https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html>`_ , `self-managed Kafka <https://docs.aws.amazon.com/lambda/latest/dg/with-kafka.html#services-smaa-onfailure-destination>`_ or `Amazon MSK <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-onfailure-destination>`_ , you can configure an Amazon SNS topic, Amazon SQS queue, or Amazon S3 bucket as the destination.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-onfailure.html#cfn-lambda-eventinvokeconfig-onfailure-destination
            '''
            result = self._values.get("destination")
            assert result is not None, "Required property 'destination' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OnFailureProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventInvokeConfig.OnSuccessProperty",
        jsii_struct_bases=[],
        name_mapping={"destination": "destination"},
    )
    class OnSuccessProperty:
        def __init__(self, *, destination: builtins.str) -> None:
            '''A destination for events that were processed successfully.

            To retain records of successful `asynchronous invocations <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-async-destinations>`_ , you can configure an Amazon SNS topic, Amazon SQS queue, Lambda function, or Amazon EventBridge event bus as the destination.

            :param destination: The Amazon Resource Name (ARN) of the destination resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-onsuccess.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                on_success_property = lambda.CfnEventInvokeConfig.OnSuccessProperty(
                    destination="destination"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dc7615bb8bd11fce4f56a0c5d830f988a0be9aa6ad1f43e72e3492c576c9b127)
                check_type(argname="argument destination", value=destination, expected_type=type_hints["destination"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "destination": destination,
            }

        @builtins.property
        def destination(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the destination resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventinvokeconfig-onsuccess.html#cfn-lambda-eventinvokeconfig-onsuccess-destination
            '''
            result = self._values.get("destination")
            assert result is not None, "Required property 'destination' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OnSuccessProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CfnEventInvokeConfigProps",
    jsii_struct_bases=[],
    name_mapping={
        "function_name": "functionName",
        "qualifier": "qualifier",
        "destination_config": "destinationConfig",
        "maximum_event_age_in_seconds": "maximumEventAgeInSeconds",
        "maximum_retry_attempts": "maximumRetryAttempts",
    },
)
class CfnEventInvokeConfigProps:
    def __init__(
        self,
        *,
        function_name: builtins.str,
        qualifier: builtins.str,
        destination_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEventInvokeConfig.DestinationConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        maximum_event_age_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnEventInvokeConfig``.

        :param function_name: The name of the Lambda function. *Minimum* : ``1`` *Maximum* : ``64`` *Pattern* : ``([a-zA-Z0-9-_]+)``
        :param qualifier: The identifier of a version or alias. - *Version* - A version number. - *Alias* - An alias name. - *Latest* - To specify the unpublished version, use ``$LATEST`` .
        :param destination_config: A destination for events after they have been sent to a function for processing. **Destinations** - *Function* - The Amazon Resource Name (ARN) of a Lambda function. - *Queue* - The ARN of a standard SQS queue. - *Bucket* - The ARN of an Amazon S3 bucket. - *Topic* - The ARN of a standard SNS topic. - *Event Bus* - The ARN of an Amazon EventBridge event bus. .. epigraph:: S3 buckets are supported only for on-failure destinations. To retain records of successful invocations, use another destination type.
        :param maximum_event_age_in_seconds: The maximum age of a request that Lambda sends to a function for processing.
        :param maximum_retry_attempts: The maximum number of times to retry when the function returns an error.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            cfn_event_invoke_config_props = lambda.CfnEventInvokeConfigProps(
                function_name="functionName",
                qualifier="qualifier",
            
                # the properties below are optional
                destination_config=lambda.CfnEventInvokeConfig.DestinationConfigProperty(
                    on_failure=lambda.CfnEventInvokeConfig.OnFailureProperty(
                        destination="destination"
                    ),
                    on_success=lambda.CfnEventInvokeConfig.OnSuccessProperty(
                        destination="destination"
                    )
                ),
                maximum_event_age_in_seconds=123,
                maximum_retry_attempts=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ac54c49f9269d96bcac2603a2bed06e61e0cc3f8f741fd5d43a4ecce77ab5480)
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument qualifier", value=qualifier, expected_type=type_hints["qualifier"])
            check_type(argname="argument destination_config", value=destination_config, expected_type=type_hints["destination_config"])
            check_type(argname="argument maximum_event_age_in_seconds", value=maximum_event_age_in_seconds, expected_type=type_hints["maximum_event_age_in_seconds"])
            check_type(argname="argument maximum_retry_attempts", value=maximum_retry_attempts, expected_type=type_hints["maximum_retry_attempts"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_name": function_name,
            "qualifier": qualifier,
        }
        if destination_config is not None:
            self._values["destination_config"] = destination_config
        if maximum_event_age_in_seconds is not None:
            self._values["maximum_event_age_in_seconds"] = maximum_event_age_in_seconds
        if maximum_retry_attempts is not None:
            self._values["maximum_retry_attempts"] = maximum_retry_attempts

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The name of the Lambda function.

        *Minimum* : ``1``

        *Maximum* : ``64``

        *Pattern* : ``([a-zA-Z0-9-_]+)``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-functionname
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def qualifier(self) -> builtins.str:
        '''The identifier of a version or alias.

        - *Version* - A version number.
        - *Alias* - An alias name.
        - *Latest* - To specify the unpublished version, use ``$LATEST`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-qualifier
        '''
        result = self._values.get("qualifier")
        assert result is not None, "Required property 'qualifier' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def destination_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventInvokeConfig.DestinationConfigProperty]]:
        '''A destination for events after they have been sent to a function for processing.

        **Destinations** - *Function* - The Amazon Resource Name (ARN) of a Lambda function.

        - *Queue* - The ARN of a standard SQS queue.
        - *Bucket* - The ARN of an Amazon S3 bucket.
        - *Topic* - The ARN of a standard SNS topic.
        - *Event Bus* - The ARN of an Amazon EventBridge event bus.

        .. epigraph::

           S3 buckets are supported only for on-failure destinations. To retain records of successful invocations, use another destination type.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-destinationconfig
        '''
        result = self._values.get("destination_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventInvokeConfig.DestinationConfigProperty]], result)

    @builtins.property
    def maximum_event_age_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-maximumeventageinseconds
        '''
        result = self._values.get("maximum_event_age_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def maximum_retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventinvokeconfig.html#cfn-lambda-eventinvokeconfig-maximumretryattempts
        '''
        result = self._values.get("maximum_retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEventInvokeConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnEventSourceMapping(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping",
):
    '''The ``AWS::Lambda::EventSourceMapping`` resource creates a mapping between an event source and an AWS Lambda function.

    Lambda reads items from the event source and triggers the function.

    For details about each event source type, see the following topics. In particular, each of the topics describes the required and optional parameters for the specific event source.

    - `Configuring a Dynamo DB stream as an event source <https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html#services-dynamodb-eventsourcemapping>`_
    - `Configuring a Kinesis stream as an event source <https://docs.aws.amazon.com/lambda/latest/dg/with-kinesis.html#services-kinesis-eventsourcemapping>`_
    - `Configuring an SQS queue as an event source <https://docs.aws.amazon.com/lambda/latest/dg/with-sqs.html#events-sqs-eventsource>`_
    - `Configuring an MQ broker as an event source <https://docs.aws.amazon.com/lambda/latest/dg/with-mq.html#services-mq-eventsourcemapping>`_
    - `Configuring MSK as an event source <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html>`_
    - `Configuring Self-Managed Apache Kafka as an event source <https://docs.aws.amazon.com/lambda/latest/dg/kafka-smaa.html>`_
    - `Configuring Amazon DocumentDB as an event source <https://docs.aws.amazon.com/lambda/latest/dg/with-documentdb.html>`_

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html
    :cloudformationResource: AWS::Lambda::EventSourceMapping
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        cfn_event_source_mapping = lambda_.CfnEventSourceMapping(self, "MyCfnEventSourceMapping",
            function_name="functionName",
        
            # the properties below are optional
            amazon_managed_kafka_event_source_config=lambda.CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty(
                consumer_group_id="consumerGroupId"
            ),
            batch_size=123,
            bisect_batch_on_function_error=False,
            destination_config=lambda.CfnEventSourceMapping.DestinationConfigProperty(
                on_failure=lambda.CfnEventSourceMapping.OnFailureProperty(
                    destination="destination"
                )
            ),
            document_db_event_source_config=lambda.CfnEventSourceMapping.DocumentDBEventSourceConfigProperty(
                collection_name="collectionName",
                database_name="databaseName",
                full_document="fullDocument"
            ),
            enabled=False,
            event_source_arn="eventSourceArn",
            filter_criteria=lambda.CfnEventSourceMapping.FilterCriteriaProperty(
                filters=[lambda.CfnEventSourceMapping.FilterProperty(
                    pattern="pattern"
                )]
            ),
            function_response_types=["functionResponseTypes"],
            kms_key_arn="kmsKeyArn",
            maximum_batching_window_in_seconds=123,
            maximum_record_age_in_seconds=123,
            maximum_retry_attempts=123,
            metrics_config=lambda.CfnEventSourceMapping.MetricsConfigProperty(
                metrics=["metrics"]
            ),
            parallelization_factor=123,
            provisioned_poller_config=lambda.CfnEventSourceMapping.ProvisionedPollerConfigProperty(
                maximum_pollers=123,
                minimum_pollers=123
            ),
            queues=["queues"],
            scaling_config=lambda.CfnEventSourceMapping.ScalingConfigProperty(
                maximum_concurrency=123
            ),
            self_managed_event_source=lambda.CfnEventSourceMapping.SelfManagedEventSourceProperty(
                endpoints=lambda.CfnEventSourceMapping.EndpointsProperty(
                    kafka_bootstrap_servers=["kafkaBootstrapServers"]
                )
            ),
            self_managed_kafka_event_source_config=lambda.CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty(
                consumer_group_id="consumerGroupId"
            ),
            source_access_configurations=[lambda.CfnEventSourceMapping.SourceAccessConfigurationProperty(
                type="type",
                uri="uri"
            )],
            starting_position="startingPosition",
            starting_position_timestamp=123,
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            topics=["topics"],
            tumbling_window_in_seconds=123
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        function_name: builtins.str,
        amazon_managed_kafka_event_source_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        batch_size: typing.Optional[jsii.Number] = None,
        bisect_batch_on_function_error: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        destination_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.DestinationConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        document_db_event_source_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.DocumentDBEventSourceConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        event_source_arn: typing.Optional[builtins.str] = None,
        filter_criteria: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.FilterCriteriaProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        function_response_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        kms_key_arn: typing.Optional[builtins.str] = None,
        maximum_batching_window_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_record_age_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_retry_attempts: typing.Optional[jsii.Number] = None,
        metrics_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.MetricsConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        parallelization_factor: typing.Optional[jsii.Number] = None,
        provisioned_poller_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.ProvisionedPollerConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        queues: typing.Optional[typing.Sequence[builtins.str]] = None,
        scaling_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.ScalingConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        self_managed_event_source: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.SelfManagedEventSourceProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        self_managed_kafka_event_source_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        source_access_configurations: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.SourceAccessConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        starting_position: typing.Optional[builtins.str] = None,
        starting_position_timestamp: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        topics: typing.Optional[typing.Sequence[builtins.str]] = None,
        tumbling_window_in_seconds: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param function_name: The name or ARN of the Lambda function. **Name formats** - *Function name* – ``MyFunction`` . - *Function ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Version or Alias ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction:PROD`` . - *Partial ARN* – ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it's limited to 64 characters in length.
        :param amazon_managed_kafka_event_source_config: Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.
        :param batch_size: The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to your function. Lambda passes all of the records in the batch to the function in a single call, up to the payload limit for synchronous invocation (6 MB). - *Amazon Kinesis* – Default 100. Max 10,000. - *Amazon DynamoDB Streams* – Default 100. Max 10,000. - *Amazon Simple Queue Service* – Default 10. For standard queues the max is 10,000. For FIFO queues the max is 10. - *Amazon Managed Streaming for Apache Kafka* – Default 100. Max 10,000. - *Self-managed Apache Kafka* – Default 100. Max 10,000. - *Amazon MQ (ActiveMQ and RabbitMQ)* – Default 100. Max 10,000. - *DocumentDB* – Default 100. Max 10,000.
        :param bisect_batch_on_function_error: (Kinesis and DynamoDB Streams only) If the function returns an error, split the batch in two and retry. The default value is false. .. epigraph:: When using ``BisectBatchOnFunctionError`` , check the ``BatchSize`` parameter in the ``OnFailure`` destination message's metadata. The ``BatchSize`` could be greater than 1 since Lambda consolidates failed messages metadata when writing to the ``OnFailure`` destination.
        :param destination_config: (Kinesis, DynamoDB Streams, Amazon MSK, and self-managed Apache Kafka event sources only) A configuration object that specifies the destination of an event after Lambda processes it.
        :param document_db_event_source_config: Specific configuration settings for a DocumentDB event source.
        :param enabled: When true, the event source mapping is active. When false, Lambda pauses polling and invocation. Default: True
        :param event_source_arn: The Amazon Resource Name (ARN) of the event source. - *Amazon Kinesis* – The ARN of the data stream or a stream consumer. - *Amazon DynamoDB Streams* – The ARN of the stream. - *Amazon Simple Queue Service* – The ARN of the queue. - *Amazon Managed Streaming for Apache Kafka* – The ARN of the cluster or the ARN of the VPC connection (for `cross-account event source mappings <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#msk-multi-vpc>`_ ). - *Amazon MQ* – The ARN of the broker. - *Amazon DocumentDB* – The ARN of the DocumentDB change stream.
        :param filter_criteria: An object that defines the filter criteria that determine whether Lambda should process an event. For more information, see `Lambda event filtering <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html>`_ .
        :param function_response_types: (Kinesis, DynamoDB Streams, and SQS) A list of current response type enums applied to the event source mapping. Valid Values: ``ReportBatchItemFailures``
        :param kms_key_arn: The ARN of the AWS Key Management Service ( AWS KMS ) customer managed key that Lambda uses to encrypt your function's `filter criteria <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html#filtering-basics>`_ .
        :param maximum_batching_window_in_seconds: The maximum amount of time, in seconds, that Lambda spends gathering records before invoking the function. *Default ( Kinesis , DynamoDB , Amazon SQS event sources)* : 0 *Default ( Amazon MSK , Kafka, Amazon MQ , Amazon DocumentDB event sources)* : 500 ms *Related setting:* For Amazon SQS event sources, when you set ``BatchSize`` to a value greater than 10, you must set ``MaximumBatchingWindowInSeconds`` to at least 1.
        :param maximum_record_age_in_seconds: (Kinesis and DynamoDB Streams only) Discard records older than the specified age. The default value is -1, which sets the maximum age to infinite. When the value is set to infinite, Lambda never discards old records. .. epigraph:: The minimum valid value for maximum record age is 60s. Although values less than 60 and greater than -1 fall within the parameter's absolute range, they are not allowed
        :param maximum_retry_attempts: (Kinesis and DynamoDB Streams only) Discard records after the specified number of retries. The default value is -1, which sets the maximum number of retries to infinite. When MaximumRetryAttempts is infinite, Lambda retries failed records until the record expires in the event source.
        :param metrics_config: The metrics configuration for your event source. For more information, see `Event source mapping metrics <https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics-types.html#event-source-mapping-metrics>`_ .
        :param parallelization_factor: (Kinesis and DynamoDB Streams only) The number of batches to process concurrently from each shard. The default value is 1.
        :param provisioned_poller_config: (Amazon MSK and self-managed Apache Kafka only) The provisioned mode configuration for the event source. For more information, see `provisioned mode <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventsourcemapping.html#invocation-eventsourcemapping-provisioned-mode>`_ .
        :param queues: (Amazon MQ) The name of the Amazon MQ broker destination queue to consume.
        :param scaling_config: (Amazon SQS only) The scaling configuration for the event source. For more information, see `Configuring maximum concurrency for Amazon SQS event sources <https://docs.aws.amazon.com/lambda/latest/dg/with-sqs.html#events-sqs-max-concurrency>`_ .
        :param self_managed_event_source: The self-managed Apache Kafka cluster for your event source.
        :param self_managed_kafka_event_source_config: Specific configuration settings for a self-managed Apache Kafka event source.
        :param source_access_configurations: An array of the authentication protocol, VPC components, or virtual host to secure and define your event source.
        :param starting_position: The position in a stream from which to start reading. Required for Amazon Kinesis and Amazon DynamoDB. - *LATEST* - Read only new records. - *TRIM_HORIZON* - Process all available records. - *AT_TIMESTAMP* - Specify a time from which to start reading records.
        :param starting_position_timestamp: With ``StartingPosition`` set to ``AT_TIMESTAMP`` , the time from which to start reading, in Unix time seconds. ``StartingPositionTimestamp`` cannot be in the future.
        :param tags: A list of tags to add to the event source mapping. .. epigraph:: You must have the ``lambda:TagResource`` , ``lambda:UntagResource`` , and ``lambda:ListTags`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.
        :param topics: The name of the Kafka topic.
        :param tumbling_window_in_seconds: (Kinesis and DynamoDB Streams only) The duration in seconds of a processing window for DynamoDB and Kinesis Streams event sources. A value of 0 seconds indicates no tumbling window.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2fc9432254acf5a7dbe3c68dcedbda61de1f0e804a81d20ae79e04857b83d419)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnEventSourceMappingProps(
            function_name=function_name,
            amazon_managed_kafka_event_source_config=amazon_managed_kafka_event_source_config,
            batch_size=batch_size,
            bisect_batch_on_function_error=bisect_batch_on_function_error,
            destination_config=destination_config,
            document_db_event_source_config=document_db_event_source_config,
            enabled=enabled,
            event_source_arn=event_source_arn,
            filter_criteria=filter_criteria,
            function_response_types=function_response_types,
            kms_key_arn=kms_key_arn,
            maximum_batching_window_in_seconds=maximum_batching_window_in_seconds,
            maximum_record_age_in_seconds=maximum_record_age_in_seconds,
            maximum_retry_attempts=maximum_retry_attempts,
            metrics_config=metrics_config,
            parallelization_factor=parallelization_factor,
            provisioned_poller_config=provisioned_poller_config,
            queues=queues,
            scaling_config=scaling_config,
            self_managed_event_source=self_managed_event_source,
            self_managed_kafka_event_source_config=self_managed_kafka_event_source_config,
            source_access_configurations=source_access_configurations,
            starting_position=starting_position,
            starting_position_timestamp=starting_position_timestamp,
            tags=tags,
            topics=topics,
            tumbling_window_in_seconds=tumbling_window_in_seconds,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__293725459b4b666915686961a6f9a4780f982bf85833cb07c717feec28594b71)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b292896de9dfac80938ff4083e3968c3796e23452cd6974d4bb583d5130bf8e0)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrEventSourceMappingArn")
    def attr_event_source_mapping_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the event source mapping.

        :cloudformationAttribute: EventSourceMappingArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEventSourceMappingArn"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The event source mapping's ID.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name or ARN of the Lambda function.'''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__62cda4d270a4340245d02bee55679c5d220663a0bda750ffc6b4672dec3a6f5b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="amazonManagedKafkaEventSourceConfig")
    def amazon_managed_kafka_event_source_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty"]]:
        '''Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty"]], jsii.get(self, "amazonManagedKafkaEventSourceConfig"))

    @amazon_managed_kafka_event_source_config.setter
    def amazon_managed_kafka_event_source_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__233d764d5ec4392f85e17319f53d7f6cc4067ccb274960b08fca3eafb9c77754)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "amazonManagedKafkaEventSourceConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="batchSize")
    def batch_size(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to your function.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "batchSize"))

    @batch_size.setter
    def batch_size(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__508a7787e5c30c8dd1e159cdbe8b8a12a49828aa6930f2a138ae3ecdfa6cb9bf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "batchSize", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="bisectBatchOnFunctionError")
    def bisect_batch_on_function_error(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''(Kinesis and DynamoDB Streams only) If the function returns an error, split the batch in two and retry.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "bisectBatchOnFunctionError"))

    @bisect_batch_on_function_error.setter
    def bisect_batch_on_function_error(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__66e2f20929a797c8075830f4ba3b2b69a933a7156ee29e476bb0e6ab51bd7773)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "bisectBatchOnFunctionError", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="destinationConfig")
    def destination_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.DestinationConfigProperty"]]:
        '''(Kinesis, DynamoDB Streams, Amazon MSK, and self-managed Apache Kafka event sources only) A configuration object that specifies the destination of an event after Lambda processes it.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.DestinationConfigProperty"]], jsii.get(self, "destinationConfig"))

    @destination_config.setter
    def destination_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.DestinationConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4ef86f269722ebd1d0b07f5394393f897ab44755b8f3c66d64b760c718d22522)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "destinationConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="documentDbEventSourceConfig")
    def document_db_event_source_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.DocumentDBEventSourceConfigProperty"]]:
        '''Specific configuration settings for a DocumentDB event source.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.DocumentDBEventSourceConfigProperty"]], jsii.get(self, "documentDbEventSourceConfig"))

    @document_db_event_source_config.setter
    def document_db_event_source_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.DocumentDBEventSourceConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__882060edb289a46d9c4b6ceb5996d087250387c92842a4cd33fc60bd9fe03b6a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "documentDbEventSourceConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="enabled")
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''When true, the event source mapping is active.

        When false, Lambda pauses polling and invocation.
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enabled"))

    @enabled.setter
    def enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__335aaf4ac58e917092578e94eea7b9439ca7a0365c948efd390845cdd23ba119)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enabled", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="eventSourceArn")
    def event_source_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the event source.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "eventSourceArn"))

    @event_source_arn.setter
    def event_source_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__80b34f80eeedc94f1aee7ad64b8a70c75cfc60660449680c9f3576ad3c3cb527)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "eventSourceArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="filterCriteria")
    def filter_criteria(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.FilterCriteriaProperty"]]:
        '''An object that defines the filter criteria that determine whether Lambda should process an event.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.FilterCriteriaProperty"]], jsii.get(self, "filterCriteria"))

    @filter_criteria.setter
    def filter_criteria(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.FilterCriteriaProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c72364a986837059e36a836853d19e31b439e0fdf4870137207ee5b4529f4096)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "filterCriteria", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="functionResponseTypes")
    def function_response_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(Kinesis, DynamoDB Streams, and SQS) A list of current response type enums applied to the event source mapping.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "functionResponseTypes"))

    @function_response_types.setter
    def function_response_types(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d464545d76bb56469faf58591d3fb0044c464a1cdb5b122bf47e0b3ea6f06d6b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionResponseTypes", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="kmsKeyArn")
    def kms_key_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the AWS Key Management Service ( AWS KMS ) customer managed key that Lambda uses to encrypt your function's `filter criteria <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html#filtering-basics>`_ .'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kmsKeyArn"))

    @kms_key_arn.setter
    def kms_key_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5110117b05ec57a413615f5bf30afd8bbafb2be839685cf0e158a1b4de420fbc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsKeyArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="maximumBatchingWindowInSeconds")
    def maximum_batching_window_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The maximum amount of time, in seconds, that Lambda spends gathering records before invoking the function.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maximumBatchingWindowInSeconds"))

    @maximum_batching_window_in_seconds.setter
    def maximum_batching_window_in_seconds(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e0ab317a438123c62fee46e89709c7bb879475c514910f3928a4fa4a397f556a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maximumBatchingWindowInSeconds", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="maximumRecordAgeInSeconds")
    def maximum_record_age_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Kinesis and DynamoDB Streams only) Discard records older than the specified age.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maximumRecordAgeInSeconds"))

    @maximum_record_age_in_seconds.setter
    def maximum_record_age_in_seconds(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dfbdec0cb3a2cf560ac8c17fad9deb0c3a42555935890c3122898a84f9cb3388)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maximumRecordAgeInSeconds", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="maximumRetryAttempts")
    def maximum_retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Kinesis and DynamoDB Streams only) Discard records after the specified number of retries.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maximumRetryAttempts"))

    @maximum_retry_attempts.setter
    def maximum_retry_attempts(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a60193f9ed8d7dcd9bc683d028f84f3d0c6fb607f147841953590623707286cd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maximumRetryAttempts", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="metricsConfig")
    def metrics_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.MetricsConfigProperty"]]:
        '''The metrics configuration for your event source.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.MetricsConfigProperty"]], jsii.get(self, "metricsConfig"))

    @metrics_config.setter
    def metrics_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.MetricsConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f5cd631413afcb74b02a0787cd61ff6f1062e9e8d6f6d0f366555a10ce55240b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "metricsConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="parallelizationFactor")
    def parallelization_factor(self) -> typing.Optional[jsii.Number]:
        '''(Kinesis and DynamoDB Streams only) The number of batches to process concurrently from each shard.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "parallelizationFactor"))

    @parallelization_factor.setter
    def parallelization_factor(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__17acebc79ed7e47bc94cbde42ca0c3c5cd8e8a67cc8a4a4de03ec16a17dc2809)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "parallelizationFactor", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="provisionedPollerConfig")
    def provisioned_poller_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.ProvisionedPollerConfigProperty"]]:
        '''(Amazon MSK and self-managed Apache Kafka only) The provisioned mode configuration for the event source.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.ProvisionedPollerConfigProperty"]], jsii.get(self, "provisionedPollerConfig"))

    @provisioned_poller_config.setter
    def provisioned_poller_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.ProvisionedPollerConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0fc82a3bfe1479a23e85d77c5f26cb1797656cf4b8a7e4285b0054919d70b267)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "provisionedPollerConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="queues")
    def queues(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(Amazon MQ) The name of the Amazon MQ broker destination queue to consume.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "queues"))

    @queues.setter
    def queues(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bbc5553ea55233b7a90d42308957e6684e5ee0076ede814be650457e0b146701)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "queues", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="scalingConfig")
    def scaling_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.ScalingConfigProperty"]]:
        '''(Amazon SQS only) The scaling configuration for the event source.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.ScalingConfigProperty"]], jsii.get(self, "scalingConfig"))

    @scaling_config.setter
    def scaling_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.ScalingConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1bd15b2a9fc8690809ee4226401692bd7666f069a0af6a98c56ccac182d01bbd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "scalingConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="selfManagedEventSource")
    def self_managed_event_source(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.SelfManagedEventSourceProperty"]]:
        '''The self-managed Apache Kafka cluster for your event source.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.SelfManagedEventSourceProperty"]], jsii.get(self, "selfManagedEventSource"))

    @self_managed_event_source.setter
    def self_managed_event_source(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.SelfManagedEventSourceProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__15f5a09c4e7c4882e5dd5f88d18a0698fbb755bbebbd097310ad730d1aacb102)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "selfManagedEventSource", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="selfManagedKafkaEventSourceConfig")
    def self_managed_kafka_event_source_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty"]]:
        '''Specific configuration settings for a self-managed Apache Kafka event source.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty"]], jsii.get(self, "selfManagedKafkaEventSourceConfig"))

    @self_managed_kafka_event_source_config.setter
    def self_managed_kafka_event_source_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__17bcf58d3b47186861370700f8542dc2f8d90ca869836ac1190e0846d32ee199)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "selfManagedKafkaEventSourceConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="sourceAccessConfigurations")
    def source_access_configurations(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.SourceAccessConfigurationProperty"]]]]:
        '''An array of the authentication protocol, VPC components, or virtual host to secure and define your event source.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.SourceAccessConfigurationProperty"]]]], jsii.get(self, "sourceAccessConfigurations"))

    @source_access_configurations.setter
    def source_access_configurations(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.SourceAccessConfigurationProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__154d952f087be022c948a51daee6d069392148a886ae7ab449b53c66e12c44d3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceAccessConfigurations", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="startingPosition")
    def starting_position(self) -> typing.Optional[builtins.str]:
        '''The position in a stream from which to start reading.

        Required for Amazon Kinesis and Amazon DynamoDB.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "startingPosition"))

    @starting_position.setter
    def starting_position(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e598fcc2f5e7192268523ee29301060bfc5b2736e861d6244dce5b8094c1a272)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "startingPosition", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="startingPositionTimestamp")
    def starting_position_timestamp(self) -> typing.Optional[jsii.Number]:
        '''With ``StartingPosition`` set to ``AT_TIMESTAMP`` , the time from which to start reading, in Unix time seconds.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "startingPositionTimestamp"))

    @starting_position_timestamp.setter
    def starting_position_timestamp(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__10f6f0aa6f7059677d942d821b3f8eaff9518ebb8fa2fe5425168602895326df)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "startingPositionTimestamp", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''A list of tags to add to the event source mapping.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tags"))

    @tags.setter
    def tags(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9bb6ad1b3bce01f266ad309d35f842ead20e44f25952e0d033834fd0e9d063c4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="topics")
    def topics(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The name of the Kafka topic.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "topics"))

    @topics.setter
    def topics(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d2153d23f0d0e6a947499030bcda617b9a68e9e84bccc6a23d9d65558f37bcac)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "topics", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tumblingWindowInSeconds")
    def tumbling_window_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Kinesis and DynamoDB Streams only) The duration in seconds of a processing window for DynamoDB and Kinesis Streams event sources.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "tumblingWindowInSeconds"))

    @tumbling_window_in_seconds.setter
    def tumbling_window_in_seconds(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ba4e20a14a70ac72313ba360ce912878e7f45dfe5ecfef9725f227fcd8f1d3e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tumblingWindowInSeconds", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"consumer_group_id": "consumerGroupId"},
    )
    class AmazonManagedKafkaEventSourceConfigProperty:
        def __init__(
            self,
            *,
            consumer_group_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.

            :param consumer_group_id: The identifier for the Kafka consumer group to join. The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-amazonmanagedkafkaeventsourceconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                amazon_managed_kafka_event_source_config_property = lambda.CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty(
                    consumer_group_id="consumerGroupId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__aac6572409154e0ba1b8c514b7863acbda031f98f5a42007db7c94d4afe31b72)
                check_type(argname="argument consumer_group_id", value=consumer_group_id, expected_type=type_hints["consumer_group_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if consumer_group_id is not None:
                self._values["consumer_group_id"] = consumer_group_id

        @builtins.property
        def consumer_group_id(self) -> typing.Optional[builtins.str]:
            '''The identifier for the Kafka consumer group to join.

            The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-amazonmanagedkafkaeventsourceconfig.html#cfn-lambda-eventsourcemapping-amazonmanagedkafkaeventsourceconfig-consumergroupid
            '''
            result = self._values.get("consumer_group_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AmazonManagedKafkaEventSourceConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.DestinationConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"on_failure": "onFailure"},
    )
    class DestinationConfigProperty:
        def __init__(
            self,
            *,
            on_failure: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.OnFailureProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''A configuration object that specifies the destination of an event after Lambda processes it.

            :param on_failure: The destination configuration for failed invocations.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-destinationconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                destination_config_property = lambda.CfnEventSourceMapping.DestinationConfigProperty(
                    on_failure=lambda.CfnEventSourceMapping.OnFailureProperty(
                        destination="destination"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f4b5968a55f3dd33118d231946be632a1c80d0c6eb6012302681fa8f95c0e8c5)
                check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if on_failure is not None:
                self._values["on_failure"] = on_failure

        @builtins.property
        def on_failure(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.OnFailureProperty"]]:
            '''The destination configuration for failed invocations.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-destinationconfig.html#cfn-lambda-eventsourcemapping-destinationconfig-onfailure
            '''
            result = self._values.get("on_failure")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.OnFailureProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DestinationConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.DocumentDBEventSourceConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "collection_name": "collectionName",
            "database_name": "databaseName",
            "full_document": "fullDocument",
        },
    )
    class DocumentDBEventSourceConfigProperty:
        def __init__(
            self,
            *,
            collection_name: typing.Optional[builtins.str] = None,
            database_name: typing.Optional[builtins.str] = None,
            full_document: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specific configuration settings for a DocumentDB event source.

            :param collection_name: The name of the collection to consume within the database. If you do not specify a collection, Lambda consumes all collections.
            :param database_name: The name of the database to consume within the DocumentDB cluster.
            :param full_document: Determines what DocumentDB sends to your event stream during document update operations. If set to UpdateLookup, DocumentDB sends a delta describing the changes, along with a copy of the entire document. Otherwise, DocumentDB sends only a partial document that contains the changes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-documentdbeventsourceconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                document_dBEvent_source_config_property = lambda.CfnEventSourceMapping.DocumentDBEventSourceConfigProperty(
                    collection_name="collectionName",
                    database_name="databaseName",
                    full_document="fullDocument"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5fe9f484b9e81d40f94147c3ffeca5a88af497c3a01e6d013b19a6249fb893d5)
                check_type(argname="argument collection_name", value=collection_name, expected_type=type_hints["collection_name"])
                check_type(argname="argument database_name", value=database_name, expected_type=type_hints["database_name"])
                check_type(argname="argument full_document", value=full_document, expected_type=type_hints["full_document"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if collection_name is not None:
                self._values["collection_name"] = collection_name
            if database_name is not None:
                self._values["database_name"] = database_name
            if full_document is not None:
                self._values["full_document"] = full_document

        @builtins.property
        def collection_name(self) -> typing.Optional[builtins.str]:
            '''The name of the collection to consume within the database.

            If you do not specify a collection, Lambda consumes all collections.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-documentdbeventsourceconfig.html#cfn-lambda-eventsourcemapping-documentdbeventsourceconfig-collectionname
            '''
            result = self._values.get("collection_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def database_name(self) -> typing.Optional[builtins.str]:
            '''The name of the database to consume within the DocumentDB cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-documentdbeventsourceconfig.html#cfn-lambda-eventsourcemapping-documentdbeventsourceconfig-databasename
            '''
            result = self._values.get("database_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def full_document(self) -> typing.Optional[builtins.str]:
            '''Determines what DocumentDB sends to your event stream during document update operations.

            If set to UpdateLookup, DocumentDB sends a delta describing the changes, along with a copy of the entire document. Otherwise, DocumentDB sends only a partial document that contains the changes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-documentdbeventsourceconfig.html#cfn-lambda-eventsourcemapping-documentdbeventsourceconfig-fulldocument
            '''
            result = self._values.get("full_document")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DocumentDBEventSourceConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.EndpointsProperty",
        jsii_struct_bases=[],
        name_mapping={"kafka_bootstrap_servers": "kafkaBootstrapServers"},
    )
    class EndpointsProperty:
        def __init__(
            self,
            *,
            kafka_bootstrap_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The list of bootstrap servers for your Kafka brokers in the following format: ``"KafkaBootstrapServers": ["abc.xyz.com:xxxx","abc2.xyz.com:xxxx"]`` .

            :param kafka_bootstrap_servers: The list of bootstrap servers for your Kafka brokers in the following format: ``"KafkaBootstrapServers": ["abc.xyz.com:xxxx","abc2.xyz.com:xxxx"]`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-endpoints.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                endpoints_property = lambda.CfnEventSourceMapping.EndpointsProperty(
                    kafka_bootstrap_servers=["kafkaBootstrapServers"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7884301d4f32edd6f6da28afc10d14681a96f53a3f1e0557171be89a97ca4544)
                check_type(argname="argument kafka_bootstrap_servers", value=kafka_bootstrap_servers, expected_type=type_hints["kafka_bootstrap_servers"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if kafka_bootstrap_servers is not None:
                self._values["kafka_bootstrap_servers"] = kafka_bootstrap_servers

        @builtins.property
        def kafka_bootstrap_servers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The list of bootstrap servers for your Kafka brokers in the following format: ``"KafkaBootstrapServers": ["abc.xyz.com:xxxx","abc2.xyz.com:xxxx"]`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-endpoints.html#cfn-lambda-eventsourcemapping-endpoints-kafkabootstrapservers
            '''
            result = self._values.get("kafka_bootstrap_servers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndpointsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.FilterCriteriaProperty",
        jsii_struct_bases=[],
        name_mapping={"filters": "filters"},
    )
    class FilterCriteriaProperty:
        def __init__(
            self,
            *,
            filters: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.FilterProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''An object that contains the filters for an event source.

            :param filters: A list of filters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-filtercriteria.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                filter_criteria_property = lambda.CfnEventSourceMapping.FilterCriteriaProperty(
                    filters=[lambda.CfnEventSourceMapping.FilterProperty(
                        pattern="pattern"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5f59ea4a33a39a96fc43dba5b09b8849ea478dcd4696d347c4c214fdb26d2c77)
                check_type(argname="argument filters", value=filters, expected_type=type_hints["filters"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if filters is not None:
                self._values["filters"] = filters

        @builtins.property
        def filters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.FilterProperty"]]]]:
            '''A list of filters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-filtercriteria.html#cfn-lambda-eventsourcemapping-filtercriteria-filters
            '''
            result = self._values.get("filters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.FilterProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FilterCriteriaProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.FilterProperty",
        jsii_struct_bases=[],
        name_mapping={"pattern": "pattern"},
    )
    class FilterProperty:
        def __init__(self, *, pattern: typing.Optional[builtins.str] = None) -> None:
            '''A structure within a ``FilterCriteria`` object that defines an event filtering pattern.

            :param pattern: A filter pattern. For more information on the syntax of a filter pattern, see `Filter rule syntax <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html#filtering-syntax>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-filter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                filter_property = lambda.CfnEventSourceMapping.FilterProperty(
                    pattern="pattern"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__acb927ec413c738ab6535754146b1867f086be8268300a02f7729708bac79c95)
                check_type(argname="argument pattern", value=pattern, expected_type=type_hints["pattern"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if pattern is not None:
                self._values["pattern"] = pattern

        @builtins.property
        def pattern(self) -> typing.Optional[builtins.str]:
            '''A filter pattern.

            For more information on the syntax of a filter pattern, see `Filter rule syntax <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html#filtering-syntax>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-filter.html#cfn-lambda-eventsourcemapping-filter-pattern
            '''
            result = self._values.get("pattern")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FilterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.MetricsConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"metrics": "metrics"},
    )
    class MetricsConfigProperty:
        def __init__(
            self,
            *,
            metrics: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The metrics configuration for your event source.

            Use this configuration object to define which metrics you want your event source mapping to produce.

            :param metrics: The metrics you want your event source mapping to produce. Include ``EventCount`` to receive event source mapping metrics related to the number of events processed by your event source mapping. For more information about these metrics, see `Event source mapping metrics <https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics-types.html#event-source-mapping-metrics>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-metricsconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                metrics_config_property = lambda.CfnEventSourceMapping.MetricsConfigProperty(
                    metrics=["metrics"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2794f7591f6e9d936a343631f5bf6dacc111c669ab59b5a405ceaa9be22e3bc3)
                check_type(argname="argument metrics", value=metrics, expected_type=type_hints["metrics"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if metrics is not None:
                self._values["metrics"] = metrics

        @builtins.property
        def metrics(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The metrics you want your event source mapping to produce.

            Include ``EventCount`` to receive event source mapping metrics related to the number of events processed by your event source mapping. For more information about these metrics, see `Event source mapping metrics <https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics-types.html#event-source-mapping-metrics>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-metricsconfig.html#cfn-lambda-eventsourcemapping-metricsconfig-metrics
            '''
            result = self._values.get("metrics")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MetricsConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.OnFailureProperty",
        jsii_struct_bases=[],
        name_mapping={"destination": "destination"},
    )
    class OnFailureProperty:
        def __init__(
            self,
            *,
            destination: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A destination for events that failed processing.

            :param destination: The Amazon Resource Name (ARN) of the destination resource. To retain records of unsuccessful `asynchronous invocations <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-async-destinations>`_ , you can configure an Amazon SNS topic, Amazon SQS queue, Amazon S3 bucket, Lambda function, or Amazon EventBridge event bus as the destination. To retain records of failed invocations from `Kinesis <https://docs.aws.amazon.com/lambda/latest/dg/with-kinesis.html>`_ , `DynamoDB <https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html>`_ , `self-managed Kafka <https://docs.aws.amazon.com/lambda/latest/dg/with-kafka.html#services-smaa-onfailure-destination>`_ or `Amazon MSK <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-onfailure-destination>`_ , you can configure an Amazon SNS topic, Amazon SQS queue, or Amazon S3 bucket as the destination.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-onfailure.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                on_failure_property = lambda.CfnEventSourceMapping.OnFailureProperty(
                    destination="destination"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b572101597157b23843e2ec6d0f49304df840ad72df4d555bbdad0c19583fe8b)
                check_type(argname="argument destination", value=destination, expected_type=type_hints["destination"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if destination is not None:
                self._values["destination"] = destination

        @builtins.property
        def destination(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the destination resource.

            To retain records of unsuccessful `asynchronous invocations <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-async-destinations>`_ , you can configure an Amazon SNS topic, Amazon SQS queue, Amazon S3 bucket, Lambda function, or Amazon EventBridge event bus as the destination.

            To retain records of failed invocations from `Kinesis <https://docs.aws.amazon.com/lambda/latest/dg/with-kinesis.html>`_ , `DynamoDB <https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html>`_ , `self-managed Kafka <https://docs.aws.amazon.com/lambda/latest/dg/with-kafka.html#services-smaa-onfailure-destination>`_ or `Amazon MSK <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-onfailure-destination>`_ , you can configure an Amazon SNS topic, Amazon SQS queue, or Amazon S3 bucket as the destination.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-onfailure.html#cfn-lambda-eventsourcemapping-onfailure-destination
            '''
            result = self._values.get("destination")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OnFailureProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.ProvisionedPollerConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "maximum_pollers": "maximumPollers",
            "minimum_pollers": "minimumPollers",
        },
    )
    class ProvisionedPollerConfigProperty:
        def __init__(
            self,
            *,
            maximum_pollers: typing.Optional[jsii.Number] = None,
            minimum_pollers: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The `provisioned mode <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventsourcemapping.html#invocation-eventsourcemapping-provisioned-mode>`_ configuration for the event source. Use provisioned mode to customize the minimum and maximum number of event pollers for your event source.

            :param maximum_pollers: The maximum number of event pollers this event source can scale up to.
            :param minimum_pollers: The minimum number of event pollers this event source can scale down to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-provisionedpollerconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                provisioned_poller_config_property = lambda.CfnEventSourceMapping.ProvisionedPollerConfigProperty(
                    maximum_pollers=123,
                    minimum_pollers=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a9766af813d9de7c91becd9b74bcc63c9c769afaf9eb6a30dc155ee230decc1b)
                check_type(argname="argument maximum_pollers", value=maximum_pollers, expected_type=type_hints["maximum_pollers"])
                check_type(argname="argument minimum_pollers", value=minimum_pollers, expected_type=type_hints["minimum_pollers"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if maximum_pollers is not None:
                self._values["maximum_pollers"] = maximum_pollers
            if minimum_pollers is not None:
                self._values["minimum_pollers"] = minimum_pollers

        @builtins.property
        def maximum_pollers(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of event pollers this event source can scale up to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-provisionedpollerconfig.html#cfn-lambda-eventsourcemapping-provisionedpollerconfig-maximumpollers
            '''
            result = self._values.get("maximum_pollers")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def minimum_pollers(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of event pollers this event source can scale down to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-provisionedpollerconfig.html#cfn-lambda-eventsourcemapping-provisionedpollerconfig-minimumpollers
            '''
            result = self._values.get("minimum_pollers")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProvisionedPollerConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.ScalingConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"maximum_concurrency": "maximumConcurrency"},
    )
    class ScalingConfigProperty:
        def __init__(
            self,
            *,
            maximum_concurrency: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''(Amazon SQS only) The scaling configuration for the event source.

            To remove the configuration, pass an empty value.

            :param maximum_concurrency: Limits the number of concurrent instances that the Amazon SQS event source can invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-scalingconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                scaling_config_property = lambda.CfnEventSourceMapping.ScalingConfigProperty(
                    maximum_concurrency=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cb04d1a40c8f33d48d5c4188609d038cd334985a6f5970b3c6b8dd1938b59cc8)
                check_type(argname="argument maximum_concurrency", value=maximum_concurrency, expected_type=type_hints["maximum_concurrency"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if maximum_concurrency is not None:
                self._values["maximum_concurrency"] = maximum_concurrency

        @builtins.property
        def maximum_concurrency(self) -> typing.Optional[jsii.Number]:
            '''Limits the number of concurrent instances that the Amazon SQS event source can invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-scalingconfig.html#cfn-lambda-eventsourcemapping-scalingconfig-maximumconcurrency
            '''
            result = self._values.get("maximum_concurrency")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.SelfManagedEventSourceProperty",
        jsii_struct_bases=[],
        name_mapping={"endpoints": "endpoints"},
    )
    class SelfManagedEventSourceProperty:
        def __init__(
            self,
            *,
            endpoints: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEventSourceMapping.EndpointsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The self-managed Apache Kafka cluster for your event source.

            :param endpoints: The list of bootstrap servers for your Kafka brokers in the following format: ``"KafkaBootstrapServers": ["abc.xyz.com:xxxx","abc2.xyz.com:xxxx"]`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-selfmanagedeventsource.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                self_managed_event_source_property = lambda.CfnEventSourceMapping.SelfManagedEventSourceProperty(
                    endpoints=lambda.CfnEventSourceMapping.EndpointsProperty(
                        kafka_bootstrap_servers=["kafkaBootstrapServers"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a93b51262658dc9dfd71e073cc42d4549a6e56e4c3b47edd31d26c3a46ea8929)
                check_type(argname="argument endpoints", value=endpoints, expected_type=type_hints["endpoints"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if endpoints is not None:
                self._values["endpoints"] = endpoints

        @builtins.property
        def endpoints(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.EndpointsProperty"]]:
            '''The list of bootstrap servers for your Kafka brokers in the following format: ``"KafkaBootstrapServers": ["abc.xyz.com:xxxx","abc2.xyz.com:xxxx"]`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-selfmanagedeventsource.html#cfn-lambda-eventsourcemapping-selfmanagedeventsource-endpoints
            '''
            result = self._values.get("endpoints")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEventSourceMapping.EndpointsProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SelfManagedEventSourceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"consumer_group_id": "consumerGroupId"},
    )
    class SelfManagedKafkaEventSourceConfigProperty:
        def __init__(
            self,
            *,
            consumer_group_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specific configuration settings for a self-managed Apache Kafka event source.

            :param consumer_group_id: The identifier for the Kafka consumer group to join. The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-selfmanagedkafkaeventsourceconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                self_managed_kafka_event_source_config_property = lambda.CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty(
                    consumer_group_id="consumerGroupId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__da32c0d6c5c0b8e4a97d90195ff7f689e87587b466dc192a0b2972c1b6740d3d)
                check_type(argname="argument consumer_group_id", value=consumer_group_id, expected_type=type_hints["consumer_group_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if consumer_group_id is not None:
                self._values["consumer_group_id"] = consumer_group_id

        @builtins.property
        def consumer_group_id(self) -> typing.Optional[builtins.str]:
            '''The identifier for the Kafka consumer group to join.

            The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-selfmanagedkafkaeventsourceconfig.html#cfn-lambda-eventsourcemapping-selfmanagedkafkaeventsourceconfig-consumergroupid
            '''
            result = self._values.get("consumer_group_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SelfManagedKafkaEventSourceConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMapping.SourceAccessConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "uri": "uri"},
    )
    class SourceAccessConfigurationProperty:
        def __init__(
            self,
            *,
            type: typing.Optional[builtins.str] = None,
            uri: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An array of the authentication protocol, VPC components, or virtual host to secure and define your event source.

            :param type: The type of authentication protocol, VPC components, or virtual host for your event source. For example: ``"Type":"SASL_SCRAM_512_AUTH"`` . - ``BASIC_AUTH`` – (Amazon MQ) The AWS Secrets Manager secret that stores your broker credentials. - ``BASIC_AUTH`` – (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL/PLAIN authentication of your Apache Kafka brokers. - ``VPC_SUBNET`` – (Self-managed Apache Kafka) The subnets associated with your VPC. Lambda connects to these subnets to fetch data from your self-managed Apache Kafka cluster. - ``VPC_SECURITY_GROUP`` – (Self-managed Apache Kafka) The VPC security group used to manage access to your self-managed Apache Kafka brokers. - ``SASL_SCRAM_256_AUTH`` – (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL SCRAM-256 authentication of your self-managed Apache Kafka brokers. - ``SASL_SCRAM_512_AUTH`` – (Amazon MSK, Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL SCRAM-512 authentication of your self-managed Apache Kafka brokers. - ``VIRTUAL_HOST`` –- (RabbitMQ) The name of the virtual host in your RabbitMQ broker. Lambda uses this RabbitMQ host as the event source. This property cannot be specified in an UpdateEventSourceMapping API call. - ``CLIENT_CERTIFICATE_TLS_AUTH`` – (Amazon MSK, self-managed Apache Kafka) The Secrets Manager ARN of your secret key containing the certificate chain (X.509 PEM), private key (PKCS#8 PEM), and private key password (optional) used for mutual TLS authentication of your MSK/Apache Kafka brokers. - ``SERVER_ROOT_CA_CERTIFICATE`` – (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key containing the root CA certificate (X.509 PEM) used for TLS encryption of your Apache Kafka brokers.
            :param uri: The value for your chosen configuration in ``Type`` . For example: ``"URI": "arn:aws:secretsmanager:us-east-1:01234567890:secret:MyBrokerSecretName"`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-sourceaccessconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                source_access_configuration_property = lambda.CfnEventSourceMapping.SourceAccessConfigurationProperty(
                    type="type",
                    uri="uri"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5697e31d4997b590460e48d14a70e090315e713aced0fbad6af21c3764a68ebd)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument uri", value=uri, expected_type=type_hints["uri"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if type is not None:
                self._values["type"] = type
            if uri is not None:
                self._values["uri"] = uri

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The type of authentication protocol, VPC components, or virtual host for your event source. For example: ``"Type":"SASL_SCRAM_512_AUTH"`` .

            - ``BASIC_AUTH`` – (Amazon MQ) The AWS Secrets Manager secret that stores your broker credentials.
            - ``BASIC_AUTH`` – (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL/PLAIN authentication of your Apache Kafka brokers.
            - ``VPC_SUBNET`` – (Self-managed Apache Kafka) The subnets associated with your VPC. Lambda connects to these subnets to fetch data from your self-managed Apache Kafka cluster.
            - ``VPC_SECURITY_GROUP`` – (Self-managed Apache Kafka) The VPC security group used to manage access to your self-managed Apache Kafka brokers.
            - ``SASL_SCRAM_256_AUTH`` – (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL SCRAM-256 authentication of your self-managed Apache Kafka brokers.
            - ``SASL_SCRAM_512_AUTH`` – (Amazon MSK, Self-managed Apache Kafka) The Secrets Manager ARN of your secret key used for SASL SCRAM-512 authentication of your self-managed Apache Kafka brokers.
            - ``VIRTUAL_HOST`` –- (RabbitMQ) The name of the virtual host in your RabbitMQ broker. Lambda uses this RabbitMQ host as the event source. This property cannot be specified in an UpdateEventSourceMapping API call.
            - ``CLIENT_CERTIFICATE_TLS_AUTH`` – (Amazon MSK, self-managed Apache Kafka) The Secrets Manager ARN of your secret key containing the certificate chain (X.509 PEM), private key (PKCS#8 PEM), and private key password (optional) used for mutual TLS authentication of your MSK/Apache Kafka brokers.
            - ``SERVER_ROOT_CA_CERTIFICATE`` – (Self-managed Apache Kafka) The Secrets Manager ARN of your secret key containing the root CA certificate (X.509 PEM) used for TLS encryption of your Apache Kafka brokers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-sourceaccessconfiguration.html#cfn-lambda-eventsourcemapping-sourceaccessconfiguration-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def uri(self) -> typing.Optional[builtins.str]:
            '''The value for your chosen configuration in ``Type`` .

            For example: ``"URI": "arn:aws:secretsmanager:us-east-1:01234567890:secret:MyBrokerSecretName"`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-sourceaccessconfiguration.html#cfn-lambda-eventsourcemapping-sourceaccessconfiguration-uri
            '''
            result = self._values.get("uri")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SourceAccessConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CfnEventSourceMappingProps",
    jsii_struct_bases=[],
    name_mapping={
        "function_name": "functionName",
        "amazon_managed_kafka_event_source_config": "amazonManagedKafkaEventSourceConfig",
        "batch_size": "batchSize",
        "bisect_batch_on_function_error": "bisectBatchOnFunctionError",
        "destination_config": "destinationConfig",
        "document_db_event_source_config": "documentDbEventSourceConfig",
        "enabled": "enabled",
        "event_source_arn": "eventSourceArn",
        "filter_criteria": "filterCriteria",
        "function_response_types": "functionResponseTypes",
        "kms_key_arn": "kmsKeyArn",
        "maximum_batching_window_in_seconds": "maximumBatchingWindowInSeconds",
        "maximum_record_age_in_seconds": "maximumRecordAgeInSeconds",
        "maximum_retry_attempts": "maximumRetryAttempts",
        "metrics_config": "metricsConfig",
        "parallelization_factor": "parallelizationFactor",
        "provisioned_poller_config": "provisionedPollerConfig",
        "queues": "queues",
        "scaling_config": "scalingConfig",
        "self_managed_event_source": "selfManagedEventSource",
        "self_managed_kafka_event_source_config": "selfManagedKafkaEventSourceConfig",
        "source_access_configurations": "sourceAccessConfigurations",
        "starting_position": "startingPosition",
        "starting_position_timestamp": "startingPositionTimestamp",
        "tags": "tags",
        "topics": "topics",
        "tumbling_window_in_seconds": "tumblingWindowInSeconds",
    },
)
class CfnEventSourceMappingProps:
    def __init__(
        self,
        *,
        function_name: builtins.str,
        amazon_managed_kafka_event_source_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        batch_size: typing.Optional[jsii.Number] = None,
        bisect_batch_on_function_error: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        destination_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEventSourceMapping.DestinationConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        document_db_event_source_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEventSourceMapping.DocumentDBEventSourceConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        event_source_arn: typing.Optional[builtins.str] = None,
        filter_criteria: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEventSourceMapping.FilterCriteriaProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        function_response_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        kms_key_arn: typing.Optional[builtins.str] = None,
        maximum_batching_window_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_record_age_in_seconds: typing.Optional[jsii.Number] = None,
        maximum_retry_attempts: typing.Optional[jsii.Number] = None,
        metrics_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEventSourceMapping.MetricsConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        parallelization_factor: typing.Optional[jsii.Number] = None,
        provisioned_poller_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEventSourceMapping.ProvisionedPollerConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        queues: typing.Optional[typing.Sequence[builtins.str]] = None,
        scaling_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEventSourceMapping.ScalingConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        self_managed_event_source: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEventSourceMapping.SelfManagedEventSourceProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        self_managed_kafka_event_source_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        source_access_configurations: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEventSourceMapping.SourceAccessConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        starting_position: typing.Optional[builtins.str] = None,
        starting_position_timestamp: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        topics: typing.Optional[typing.Sequence[builtins.str]] = None,
        tumbling_window_in_seconds: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnEventSourceMapping``.

        :param function_name: The name or ARN of the Lambda function. **Name formats** - *Function name* – ``MyFunction`` . - *Function ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Version or Alias ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction:PROD`` . - *Partial ARN* – ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it's limited to 64 characters in length.
        :param amazon_managed_kafka_event_source_config: Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.
        :param batch_size: The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to your function. Lambda passes all of the records in the batch to the function in a single call, up to the payload limit for synchronous invocation (6 MB). - *Amazon Kinesis* – Default 100. Max 10,000. - *Amazon DynamoDB Streams* – Default 100. Max 10,000. - *Amazon Simple Queue Service* – Default 10. For standard queues the max is 10,000. For FIFO queues the max is 10. - *Amazon Managed Streaming for Apache Kafka* – Default 100. Max 10,000. - *Self-managed Apache Kafka* – Default 100. Max 10,000. - *Amazon MQ (ActiveMQ and RabbitMQ)* – Default 100. Max 10,000. - *DocumentDB* – Default 100. Max 10,000.
        :param bisect_batch_on_function_error: (Kinesis and DynamoDB Streams only) If the function returns an error, split the batch in two and retry. The default value is false. .. epigraph:: When using ``BisectBatchOnFunctionError`` , check the ``BatchSize`` parameter in the ``OnFailure`` destination message's metadata. The ``BatchSize`` could be greater than 1 since Lambda consolidates failed messages metadata when writing to the ``OnFailure`` destination.
        :param destination_config: (Kinesis, DynamoDB Streams, Amazon MSK, and self-managed Apache Kafka event sources only) A configuration object that specifies the destination of an event after Lambda processes it.
        :param document_db_event_source_config: Specific configuration settings for a DocumentDB event source.
        :param enabled: When true, the event source mapping is active. When false, Lambda pauses polling and invocation. Default: True
        :param event_source_arn: The Amazon Resource Name (ARN) of the event source. - *Amazon Kinesis* – The ARN of the data stream or a stream consumer. - *Amazon DynamoDB Streams* – The ARN of the stream. - *Amazon Simple Queue Service* – The ARN of the queue. - *Amazon Managed Streaming for Apache Kafka* – The ARN of the cluster or the ARN of the VPC connection (for `cross-account event source mappings <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#msk-multi-vpc>`_ ). - *Amazon MQ* – The ARN of the broker. - *Amazon DocumentDB* – The ARN of the DocumentDB change stream.
        :param filter_criteria: An object that defines the filter criteria that determine whether Lambda should process an event. For more information, see `Lambda event filtering <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html>`_ .
        :param function_response_types: (Kinesis, DynamoDB Streams, and SQS) A list of current response type enums applied to the event source mapping. Valid Values: ``ReportBatchItemFailures``
        :param kms_key_arn: The ARN of the AWS Key Management Service ( AWS KMS ) customer managed key that Lambda uses to encrypt your function's `filter criteria <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html#filtering-basics>`_ .
        :param maximum_batching_window_in_seconds: The maximum amount of time, in seconds, that Lambda spends gathering records before invoking the function. *Default ( Kinesis , DynamoDB , Amazon SQS event sources)* : 0 *Default ( Amazon MSK , Kafka, Amazon MQ , Amazon DocumentDB event sources)* : 500 ms *Related setting:* For Amazon SQS event sources, when you set ``BatchSize`` to a value greater than 10, you must set ``MaximumBatchingWindowInSeconds`` to at least 1.
        :param maximum_record_age_in_seconds: (Kinesis and DynamoDB Streams only) Discard records older than the specified age. The default value is -1, which sets the maximum age to infinite. When the value is set to infinite, Lambda never discards old records. .. epigraph:: The minimum valid value for maximum record age is 60s. Although values less than 60 and greater than -1 fall within the parameter's absolute range, they are not allowed
        :param maximum_retry_attempts: (Kinesis and DynamoDB Streams only) Discard records after the specified number of retries. The default value is -1, which sets the maximum number of retries to infinite. When MaximumRetryAttempts is infinite, Lambda retries failed records until the record expires in the event source.
        :param metrics_config: The metrics configuration for your event source. For more information, see `Event source mapping metrics <https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics-types.html#event-source-mapping-metrics>`_ .
        :param parallelization_factor: (Kinesis and DynamoDB Streams only) The number of batches to process concurrently from each shard. The default value is 1.
        :param provisioned_poller_config: (Amazon MSK and self-managed Apache Kafka only) The provisioned mode configuration for the event source. For more information, see `provisioned mode <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventsourcemapping.html#invocation-eventsourcemapping-provisioned-mode>`_ .
        :param queues: (Amazon MQ) The name of the Amazon MQ broker destination queue to consume.
        :param scaling_config: (Amazon SQS only) The scaling configuration for the event source. For more information, see `Configuring maximum concurrency for Amazon SQS event sources <https://docs.aws.amazon.com/lambda/latest/dg/with-sqs.html#events-sqs-max-concurrency>`_ .
        :param self_managed_event_source: The self-managed Apache Kafka cluster for your event source.
        :param self_managed_kafka_event_source_config: Specific configuration settings for a self-managed Apache Kafka event source.
        :param source_access_configurations: An array of the authentication protocol, VPC components, or virtual host to secure and define your event source.
        :param starting_position: The position in a stream from which to start reading. Required for Amazon Kinesis and Amazon DynamoDB. - *LATEST* - Read only new records. - *TRIM_HORIZON* - Process all available records. - *AT_TIMESTAMP* - Specify a time from which to start reading records.
        :param starting_position_timestamp: With ``StartingPosition`` set to ``AT_TIMESTAMP`` , the time from which to start reading, in Unix time seconds. ``StartingPositionTimestamp`` cannot be in the future.
        :param tags: A list of tags to add to the event source mapping. .. epigraph:: You must have the ``lambda:TagResource`` , ``lambda:UntagResource`` , and ``lambda:ListTags`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.
        :param topics: The name of the Kafka topic.
        :param tumbling_window_in_seconds: (Kinesis and DynamoDB Streams only) The duration in seconds of a processing window for DynamoDB and Kinesis Streams event sources. A value of 0 seconds indicates no tumbling window.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            cfn_event_source_mapping_props = lambda.CfnEventSourceMappingProps(
                function_name="functionName",
            
                # the properties below are optional
                amazon_managed_kafka_event_source_config=lambda.CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty(
                    consumer_group_id="consumerGroupId"
                ),
                batch_size=123,
                bisect_batch_on_function_error=False,
                destination_config=lambda.CfnEventSourceMapping.DestinationConfigProperty(
                    on_failure=lambda.CfnEventSourceMapping.OnFailureProperty(
                        destination="destination"
                    )
                ),
                document_db_event_source_config=lambda.CfnEventSourceMapping.DocumentDBEventSourceConfigProperty(
                    collection_name="collectionName",
                    database_name="databaseName",
                    full_document="fullDocument"
                ),
                enabled=False,
                event_source_arn="eventSourceArn",
                filter_criteria=lambda.CfnEventSourceMapping.FilterCriteriaProperty(
                    filters=[lambda.CfnEventSourceMapping.FilterProperty(
                        pattern="pattern"
                    )]
                ),
                function_response_types=["functionResponseTypes"],
                kms_key_arn="kmsKeyArn",
                maximum_batching_window_in_seconds=123,
                maximum_record_age_in_seconds=123,
                maximum_retry_attempts=123,
                metrics_config=lambda.CfnEventSourceMapping.MetricsConfigProperty(
                    metrics=["metrics"]
                ),
                parallelization_factor=123,
                provisioned_poller_config=lambda.CfnEventSourceMapping.ProvisionedPollerConfigProperty(
                    maximum_pollers=123,
                    minimum_pollers=123
                ),
                queues=["queues"],
                scaling_config=lambda.CfnEventSourceMapping.ScalingConfigProperty(
                    maximum_concurrency=123
                ),
                self_managed_event_source=lambda.CfnEventSourceMapping.SelfManagedEventSourceProperty(
                    endpoints=lambda.CfnEventSourceMapping.EndpointsProperty(
                        kafka_bootstrap_servers=["kafkaBootstrapServers"]
                    )
                ),
                self_managed_kafka_event_source_config=lambda.CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty(
                    consumer_group_id="consumerGroupId"
                ),
                source_access_configurations=[lambda.CfnEventSourceMapping.SourceAccessConfigurationProperty(
                    type="type",
                    uri="uri"
                )],
                starting_position="startingPosition",
                starting_position_timestamp=123,
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                topics=["topics"],
                tumbling_window_in_seconds=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__28f15573fb2525439f034f01287568e3c4a4d28fda953b3059f294655b5624e6)
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument amazon_managed_kafka_event_source_config", value=amazon_managed_kafka_event_source_config, expected_type=type_hints["amazon_managed_kafka_event_source_config"])
            check_type(argname="argument batch_size", value=batch_size, expected_type=type_hints["batch_size"])
            check_type(argname="argument bisect_batch_on_function_error", value=bisect_batch_on_function_error, expected_type=type_hints["bisect_batch_on_function_error"])
            check_type(argname="argument destination_config", value=destination_config, expected_type=type_hints["destination_config"])
            check_type(argname="argument document_db_event_source_config", value=document_db_event_source_config, expected_type=type_hints["document_db_event_source_config"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument event_source_arn", value=event_source_arn, expected_type=type_hints["event_source_arn"])
            check_type(argname="argument filter_criteria", value=filter_criteria, expected_type=type_hints["filter_criteria"])
            check_type(argname="argument function_response_types", value=function_response_types, expected_type=type_hints["function_response_types"])
            check_type(argname="argument kms_key_arn", value=kms_key_arn, expected_type=type_hints["kms_key_arn"])
            check_type(argname="argument maximum_batching_window_in_seconds", value=maximum_batching_window_in_seconds, expected_type=type_hints["maximum_batching_window_in_seconds"])
            check_type(argname="argument maximum_record_age_in_seconds", value=maximum_record_age_in_seconds, expected_type=type_hints["maximum_record_age_in_seconds"])
            check_type(argname="argument maximum_retry_attempts", value=maximum_retry_attempts, expected_type=type_hints["maximum_retry_attempts"])
            check_type(argname="argument metrics_config", value=metrics_config, expected_type=type_hints["metrics_config"])
            check_type(argname="argument parallelization_factor", value=parallelization_factor, expected_type=type_hints["parallelization_factor"])
            check_type(argname="argument provisioned_poller_config", value=provisioned_poller_config, expected_type=type_hints["provisioned_poller_config"])
            check_type(argname="argument queues", value=queues, expected_type=type_hints["queues"])
            check_type(argname="argument scaling_config", value=scaling_config, expected_type=type_hints["scaling_config"])
            check_type(argname="argument self_managed_event_source", value=self_managed_event_source, expected_type=type_hints["self_managed_event_source"])
            check_type(argname="argument self_managed_kafka_event_source_config", value=self_managed_kafka_event_source_config, expected_type=type_hints["self_managed_kafka_event_source_config"])
            check_type(argname="argument source_access_configurations", value=source_access_configurations, expected_type=type_hints["source_access_configurations"])
            check_type(argname="argument starting_position", value=starting_position, expected_type=type_hints["starting_position"])
            check_type(argname="argument starting_position_timestamp", value=starting_position_timestamp, expected_type=type_hints["starting_position_timestamp"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument topics", value=topics, expected_type=type_hints["topics"])
            check_type(argname="argument tumbling_window_in_seconds", value=tumbling_window_in_seconds, expected_type=type_hints["tumbling_window_in_seconds"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_name": function_name,
        }
        if amazon_managed_kafka_event_source_config is not None:
            self._values["amazon_managed_kafka_event_source_config"] = amazon_managed_kafka_event_source_config
        if batch_size is not None:
            self._values["batch_size"] = batch_size
        if bisect_batch_on_function_error is not None:
            self._values["bisect_batch_on_function_error"] = bisect_batch_on_function_error
        if destination_config is not None:
            self._values["destination_config"] = destination_config
        if document_db_event_source_config is not None:
            self._values["document_db_event_source_config"] = document_db_event_source_config
        if enabled is not None:
            self._values["enabled"] = enabled
        if event_source_arn is not None:
            self._values["event_source_arn"] = event_source_arn
        if filter_criteria is not None:
            self._values["filter_criteria"] = filter_criteria
        if function_response_types is not None:
            self._values["function_response_types"] = function_response_types
        if kms_key_arn is not None:
            self._values["kms_key_arn"] = kms_key_arn
        if maximum_batching_window_in_seconds is not None:
            self._values["maximum_batching_window_in_seconds"] = maximum_batching_window_in_seconds
        if maximum_record_age_in_seconds is not None:
            self._values["maximum_record_age_in_seconds"] = maximum_record_age_in_seconds
        if maximum_retry_attempts is not None:
            self._values["maximum_retry_attempts"] = maximum_retry_attempts
        if metrics_config is not None:
            self._values["metrics_config"] = metrics_config
        if parallelization_factor is not None:
            self._values["parallelization_factor"] = parallelization_factor
        if provisioned_poller_config is not None:
            self._values["provisioned_poller_config"] = provisioned_poller_config
        if queues is not None:
            self._values["queues"] = queues
        if scaling_config is not None:
            self._values["scaling_config"] = scaling_config
        if self_managed_event_source is not None:
            self._values["self_managed_event_source"] = self_managed_event_source
        if self_managed_kafka_event_source_config is not None:
            self._values["self_managed_kafka_event_source_config"] = self_managed_kafka_event_source_config
        if source_access_configurations is not None:
            self._values["source_access_configurations"] = source_access_configurations
        if starting_position is not None:
            self._values["starting_position"] = starting_position
        if starting_position_timestamp is not None:
            self._values["starting_position_timestamp"] = starting_position_timestamp
        if tags is not None:
            self._values["tags"] = tags
        if topics is not None:
            self._values["topics"] = topics
        if tumbling_window_in_seconds is not None:
            self._values["tumbling_window_in_seconds"] = tumbling_window_in_seconds

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The name or ARN of the Lambda function.

        **Name formats** - *Function name* – ``MyFunction`` .

        - *Function ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` .
        - *Version or Alias ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction:PROD`` .
        - *Partial ARN* – ``123456789012:function:MyFunction`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it's limited to 64 characters in length.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-functionname
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def amazon_managed_kafka_event_source_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty]]:
        '''Specific configuration settings for an Amazon Managed Streaming for Apache Kafka (Amazon MSK) event source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-amazonmanagedkafkaeventsourceconfig
        '''
        result = self._values.get("amazon_managed_kafka_event_source_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.AmazonManagedKafkaEventSourceConfigProperty]], result)

    @builtins.property
    def batch_size(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of records in each batch that Lambda pulls from your stream or queue and sends to your function.

        Lambda passes all of the records in the batch to the function in a single call, up to the payload limit for synchronous invocation (6 MB).

        - *Amazon Kinesis* – Default 100. Max 10,000.
        - *Amazon DynamoDB Streams* – Default 100. Max 10,000.
        - *Amazon Simple Queue Service* – Default 10. For standard queues the max is 10,000. For FIFO queues the max is 10.
        - *Amazon Managed Streaming for Apache Kafka* – Default 100. Max 10,000.
        - *Self-managed Apache Kafka* – Default 100. Max 10,000.
        - *Amazon MQ (ActiveMQ and RabbitMQ)* – Default 100. Max 10,000.
        - *DocumentDB* – Default 100. Max 10,000.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-batchsize
        '''
        result = self._values.get("batch_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def bisect_batch_on_function_error(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''(Kinesis and DynamoDB Streams only) If the function returns an error, split the batch in two and retry.

        The default value is false.
        .. epigraph::

           When using ``BisectBatchOnFunctionError`` , check the ``BatchSize`` parameter in the ``OnFailure`` destination message's metadata. The ``BatchSize`` could be greater than 1 since Lambda consolidates failed messages metadata when writing to the ``OnFailure`` destination.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-bisectbatchonfunctionerror
        '''
        result = self._values.get("bisect_batch_on_function_error")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def destination_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.DestinationConfigProperty]]:
        '''(Kinesis, DynamoDB Streams, Amazon MSK, and self-managed Apache Kafka event sources only) A configuration object that specifies the destination of an event after Lambda processes it.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-destinationconfig
        '''
        result = self._values.get("destination_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.DestinationConfigProperty]], result)

    @builtins.property
    def document_db_event_source_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.DocumentDBEventSourceConfigProperty]]:
        '''Specific configuration settings for a DocumentDB event source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-documentdbeventsourceconfig
        '''
        result = self._values.get("document_db_event_source_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.DocumentDBEventSourceConfigProperty]], result)

    @builtins.property
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''When true, the event source mapping is active. When false, Lambda pauses polling and invocation.

        Default: True

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-enabled
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def event_source_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the event source.

        - *Amazon Kinesis* – The ARN of the data stream or a stream consumer.
        - *Amazon DynamoDB Streams* – The ARN of the stream.
        - *Amazon Simple Queue Service* – The ARN of the queue.
        - *Amazon Managed Streaming for Apache Kafka* – The ARN of the cluster or the ARN of the VPC connection (for `cross-account event source mappings <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#msk-multi-vpc>`_ ).
        - *Amazon MQ* – The ARN of the broker.
        - *Amazon DocumentDB* – The ARN of the DocumentDB change stream.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-eventsourcearn
        '''
        result = self._values.get("event_source_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def filter_criteria(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.FilterCriteriaProperty]]:
        '''An object that defines the filter criteria that determine whether Lambda should process an event.

        For more information, see `Lambda event filtering <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-filtercriteria
        '''
        result = self._values.get("filter_criteria")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.FilterCriteriaProperty]], result)

    @builtins.property
    def function_response_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(Kinesis, DynamoDB Streams, and SQS) A list of current response type enums applied to the event source mapping.

        Valid Values: ``ReportBatchItemFailures``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-functionresponsetypes
        '''
        result = self._values.get("function_response_types")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def kms_key_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the AWS Key Management Service ( AWS KMS ) customer managed key that Lambda uses to encrypt your function's `filter criteria <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html#filtering-basics>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-kmskeyarn
        '''
        result = self._values.get("kms_key_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def maximum_batching_window_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The maximum amount of time, in seconds, that Lambda spends gathering records before invoking the function.

        *Default ( Kinesis , DynamoDB , Amazon SQS event sources)* : 0

        *Default ( Amazon MSK , Kafka, Amazon MQ , Amazon DocumentDB event sources)* : 500 ms

        *Related setting:* For Amazon SQS event sources, when you set ``BatchSize`` to a value greater than 10, you must set ``MaximumBatchingWindowInSeconds`` to at least 1.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-maximumbatchingwindowinseconds
        '''
        result = self._values.get("maximum_batching_window_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def maximum_record_age_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Kinesis and DynamoDB Streams only) Discard records older than the specified age.

        The default value is -1,
        which sets the maximum age to infinite. When the value is set to infinite, Lambda never discards old records.
        .. epigraph::

           The minimum valid value for maximum record age is 60s. Although values less than 60 and greater than -1 fall within the parameter's absolute range, they are not allowed

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-maximumrecordageinseconds
        '''
        result = self._values.get("maximum_record_age_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def maximum_retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Kinesis and DynamoDB Streams only) Discard records after the specified number of retries.

        The default value is -1,
        which sets the maximum number of retries to infinite. When MaximumRetryAttempts is infinite, Lambda retries failed records until the record expires in the event source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-maximumretryattempts
        '''
        result = self._values.get("maximum_retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def metrics_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.MetricsConfigProperty]]:
        '''The metrics configuration for your event source.

        For more information, see `Event source mapping metrics <https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics-types.html#event-source-mapping-metrics>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-metricsconfig
        '''
        result = self._values.get("metrics_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.MetricsConfigProperty]], result)

    @builtins.property
    def parallelization_factor(self) -> typing.Optional[jsii.Number]:
        '''(Kinesis and DynamoDB Streams only) The number of batches to process concurrently from each shard.

        The default value is 1.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-parallelizationfactor
        '''
        result = self._values.get("parallelization_factor")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def provisioned_poller_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.ProvisionedPollerConfigProperty]]:
        '''(Amazon MSK and self-managed Apache Kafka only) The provisioned mode configuration for the event source.

        For more information, see `provisioned mode <https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventsourcemapping.html#invocation-eventsourcemapping-provisioned-mode>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-provisionedpollerconfig
        '''
        result = self._values.get("provisioned_poller_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.ProvisionedPollerConfigProperty]], result)

    @builtins.property
    def queues(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(Amazon MQ) The name of the Amazon MQ broker destination queue to consume.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-queues
        '''
        result = self._values.get("queues")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def scaling_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.ScalingConfigProperty]]:
        '''(Amazon SQS only) The scaling configuration for the event source.

        For more information, see `Configuring maximum concurrency for Amazon SQS event sources <https://docs.aws.amazon.com/lambda/latest/dg/with-sqs.html#events-sqs-max-concurrency>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-scalingconfig
        '''
        result = self._values.get("scaling_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.ScalingConfigProperty]], result)

    @builtins.property
    def self_managed_event_source(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.SelfManagedEventSourceProperty]]:
        '''The self-managed Apache Kafka cluster for your event source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-selfmanagedeventsource
        '''
        result = self._values.get("self_managed_event_source")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.SelfManagedEventSourceProperty]], result)

    @builtins.property
    def self_managed_kafka_event_source_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty]]:
        '''Specific configuration settings for a self-managed Apache Kafka event source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-selfmanagedkafkaeventsourceconfig
        '''
        result = self._values.get("self_managed_kafka_event_source_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.SelfManagedKafkaEventSourceConfigProperty]], result)

    @builtins.property
    def source_access_configurations(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.SourceAccessConfigurationProperty]]]]:
        '''An array of the authentication protocol, VPC components, or virtual host to secure and define your event source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-sourceaccessconfigurations
        '''
        result = self._values.get("source_access_configurations")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnEventSourceMapping.SourceAccessConfigurationProperty]]]], result)

    @builtins.property
    def starting_position(self) -> typing.Optional[builtins.str]:
        '''The position in a stream from which to start reading. Required for Amazon Kinesis and Amazon DynamoDB.

        - *LATEST* - Read only new records.
        - *TRIM_HORIZON* - Process all available records.
        - *AT_TIMESTAMP* - Specify a time from which to start reading records.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-startingposition
        '''
        result = self._values.get("starting_position")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def starting_position_timestamp(self) -> typing.Optional[jsii.Number]:
        '''With ``StartingPosition`` set to ``AT_TIMESTAMP`` , the time from which to start reading, in Unix time seconds.

        ``StartingPositionTimestamp`` cannot be in the future.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-startingpositiontimestamp
        '''
        result = self._values.get("starting_position_timestamp")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''A list of tags to add to the event source mapping.

        .. epigraph::

           You must have the ``lambda:TagResource`` , ``lambda:UntagResource`` , and ``lambda:ListTags`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def topics(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The name of the Kafka topic.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-topics
        '''
        result = self._values.get("topics")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tumbling_window_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Kinesis and DynamoDB Streams only) The duration in seconds of a processing window for DynamoDB and Kinesis Streams event sources.

        A value of 0 seconds indicates no tumbling window.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-eventsourcemapping.html#cfn-lambda-eventsourcemapping-tumblingwindowinseconds
        '''
        result = self._values.get("tumbling_window_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEventSourceMappingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnFunction(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.CfnFunction",
):
    '''The ``AWS::Lambda::Function`` resource creates a Lambda function.

    To create a function, you need a `deployment package <https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-package.html>`_ and an `execution role <https://docs.aws.amazon.com/lambda/latest/dg/lambda-intro-execution-role.html>`_ . The deployment package is a .zip file archive or container image that contains your function code. The execution role grants the function permission to use AWS services, such as Amazon CloudWatch Logs for log streaming and AWS X-Ray for request tracing.

    You set the package type to ``Image`` if the deployment package is a `container image <https://docs.aws.amazon.com/lambda/latest/dg/lambda-images.html>`_ . For these functions, include the URI of the container image in the Amazon ECR registry in the ```ImageUri`` property of the ``Code`` property <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-imageuri>`_ . You do not need to specify the handler and runtime properties.

    You set the package type to ``Zip`` if the deployment package is a `.zip file archive <https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-package.html#gettingstarted-package-zip>`_ . For these functions, specify the Amazon S3 location of your .zip file in the ``Code`` property. Alternatively, for Node.js and Python functions, you can define your function inline in the ```ZipFile`` property of the ``Code`` property <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-zipfile>`_ . In both cases, you must also specify the handler and runtime properties.

    You can use `code signing <https://docs.aws.amazon.com/lambda/latest/dg/configuration-codesigning.html>`_ if your deployment package is a .zip file archive. To enable code signing for this function, specify the ARN of a code-signing configuration. When a user attempts to deploy a code package with ``UpdateFunctionCode`` , Lambda checks that the code package has a valid signature from a trusted publisher. The code-signing configuration includes a set of signing profiles, which define the trusted publishers for this function.

    When you update a ``AWS::Lambda::Function`` resource, CloudFormation calls the `UpdateFunctionConfiguration <https://docs.aws.amazon.com/lambda/latest/api/API_UpdateFunctionConfiguration.html>`_ and `UpdateFunctionCode <https://docs.aws.amazon.com/lambda/latest/api/API_UpdateFunctionCode.html>`_ Lambda APIs under the hood. Because these calls happen sequentially, and invocations can happen between these calls, your function may encounter errors in the time between the calls. For example, if you remove an environment variable, and the code that references that environment variable in the same CloudFormation update, you may see invocation errors related to a missing environment variable. To work around this, you can invoke your function against a version or alias by default, rather than the ``$LATEST`` version.

    Note that you configure `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/provisioned-concurrency.html>`_ on a ``AWS::Lambda::Version`` or a ``AWS::Lambda::Alias`` .

    For a complete introduction to Lambda functions, see `What is Lambda? <https://docs.aws.amazon.com/lambda/latest/dg/lambda-welcome.html>`_ in the *Lambda developer guide.*

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html
    :cloudformationResource: AWS::Lambda::Function
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        cfn_function = lambda_.CfnFunction(self, "MyCfnFunction",
            code=lambda.CfnFunction.CodeProperty(
                image_uri="imageUri",
                s3_bucket="s3Bucket",
                s3_key="s3Key",
                s3_object_version="s3ObjectVersion",
                source_kms_key_arn="sourceKmsKeyArn",
                zip_file="zipFile"
            ),
            role="role",
        
            # the properties below are optional
            architectures=["architectures"],
            code_signing_config_arn="codeSigningConfigArn",
            dead_letter_config=lambda.CfnFunction.DeadLetterConfigProperty(
                target_arn="targetArn"
            ),
            description="description",
            environment=lambda.CfnFunction.EnvironmentProperty(
                variables={
                    "variables_key": "variables"
                }
            ),
            ephemeral_storage=lambda.CfnFunction.EphemeralStorageProperty(
                size=123
            ),
            file_system_configs=[lambda.CfnFunction.FileSystemConfigProperty(
                arn="arn",
                local_mount_path="localMountPath"
            )],
            function_name="functionName",
            handler="handler",
            image_config=lambda.CfnFunction.ImageConfigProperty(
                command=["command"],
                entry_point=["entryPoint"],
                working_directory="workingDirectory"
            ),
            kms_key_arn="kmsKeyArn",
            layers=["layers"],
            logging_config=lambda.CfnFunction.LoggingConfigProperty(
                application_log_level="applicationLogLevel",
                log_format="logFormat",
                log_group="logGroup",
                system_log_level="systemLogLevel"
            ),
            memory_size=123,
            package_type="packageType",
            recursive_loop="recursiveLoop",
            reserved_concurrent_executions=123,
            runtime="runtime",
            runtime_management_config=lambda.CfnFunction.RuntimeManagementConfigProperty(
                update_runtime_on="updateRuntimeOn",
        
                # the properties below are optional
                runtime_version_arn="runtimeVersionArn"
            ),
            snap_start=lambda.CfnFunction.SnapStartProperty(
                apply_on="applyOn"
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            timeout=123,
            tracing_config=lambda.CfnFunction.TracingConfigProperty(
                mode="mode"
            ),
            vpc_config=lambda.CfnFunction.VpcConfigProperty(
                ipv6_allowed_for_dual_stack=False,
                security_group_ids=["securityGroupIds"],
                subnet_ids=["subnetIds"]
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        code: typing.Union[_IResolvable_da3f097b, typing.Union["CfnFunction.CodeProperty", typing.Dict[builtins.str, typing.Any]]],
        role: builtins.str,
        architectures: typing.Optional[typing.Sequence[builtins.str]] = None,
        code_signing_config_arn: typing.Optional[builtins.str] = None,
        dead_letter_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFunction.DeadLetterConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFunction.EnvironmentProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ephemeral_storage: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFunction.EphemeralStorageProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        file_system_configs: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFunction.FileSystemConfigProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        function_name: typing.Optional[builtins.str] = None,
        handler: typing.Optional[builtins.str] = None,
        image_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFunction.ImageConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        kms_key_arn: typing.Optional[builtins.str] = None,
        layers: typing.Optional[typing.Sequence[builtins.str]] = None,
        logging_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFunction.LoggingConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        memory_size: typing.Optional[jsii.Number] = None,
        package_type: typing.Optional[builtins.str] = None,
        recursive_loop: typing.Optional[builtins.str] = None,
        reserved_concurrent_executions: typing.Optional[jsii.Number] = None,
        runtime: typing.Optional[builtins.str] = None,
        runtime_management_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFunction.RuntimeManagementConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        snap_start: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFunction.SnapStartProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        timeout: typing.Optional[jsii.Number] = None,
        tracing_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFunction.TracingConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        vpc_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFunction.VpcConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param code: The code for the function. You can define your function code in multiple ways:. - For .zip deployment packages, you can specify the Amazon S3 location of the .zip file in the ``S3Bucket`` , ``S3Key`` , and ``S3ObjectVersion`` properties. - For .zip deployment packages, you can alternatively define the function code inline in the ``ZipFile`` property. This method works only for Node.js and Python functions. - For container images, specify the URI of your container image in the Amazon ECR registry in the ``ImageUri`` property.
        :param role: The Amazon Resource Name (ARN) of the function's execution role.
        :param architectures: The instruction set architecture that the function supports. Enter a string array with one of the valid values (arm64 or x86_64). The default value is ``x86_64`` .
        :param code_signing_config_arn: To enable code signing for this function, specify the ARN of a code-signing configuration. A code-signing configuration includes a set of signing profiles, which define the trusted publishers for this function.
        :param dead_letter_config: A dead-letter queue configuration that specifies the queue or topic where Lambda sends asynchronous events when they fail processing. For more information, see `Dead-letter queues <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-dlq>`_ .
        :param description: A description of the function.
        :param environment: Environment variables that are accessible from function code during execution.
        :param ephemeral_storage: The size of the function's ``/tmp`` directory in MB. The default value is 512, but it can be any whole number between 512 and 10,240 MB.
        :param file_system_configs: Connection settings for an Amazon EFS file system. To connect a function to a file system, a mount target must be available in every Availability Zone that your function connects to. If your template contains an `AWS::EFS::MountTarget <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-mounttarget.html>`_ resource, you must also specify a ``DependsOn`` attribute to ensure that the mount target is created or updated before the function. For more information about using the ``DependsOn`` attribute, see `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ .
        :param function_name: The name of the Lambda function, up to 64 characters in length. If you don't specify a name, AWS CloudFormation generates one. If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param handler: The name of the method within your code that Lambda calls to run your function. Handler is required if the deployment package is a .zip file archive. The format includes the file name. It can also include namespaces and other qualifiers, depending on the runtime. For more information, see `Lambda programming model <https://docs.aws.amazon.com/lambda/latest/dg/foundation-progmodel.html>`_ .
        :param image_config: Configuration values that override the container image Dockerfile settings. For more information, see `Container image settings <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-parms>`_ .
        :param kms_key_arn: The ARN of the AWS Key Management Service ( AWS KMS ) customer managed key that's used to encrypt the following resources:. - The function's `environment variables <https://docs.aws.amazon.com/lambda/latest/dg/configuration-envvars.html#configuration-envvars-encryption>`_ . - The function's `Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart-security.html>`_ snapshots. - When used with ``SourceKMSKeyArn`` , the unzipped version of the .zip deployment package that's used for function invocations. For more information, see `Specifying a customer managed key for Lambda <https://docs.aws.amazon.com/lambda/latest/dg/encrypt-zip-package.html#enable-zip-custom-encryption>`_ . - The optimized version of the container image that's used for function invocations. Note that this is not the same key that's used to protect your container image in the Amazon Elastic Container Registry (Amazon ECR). For more information, see `Function lifecycle <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-lifecycle>`_ . If you don't provide a customer managed key, Lambda uses an `AWS owned key <https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-owned-cmk>`_ or an `AWS managed key <https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-managed-cmk>`_ .
        :param layers: A list of `function layers <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ to add to the function's execution environment. Specify each layer by its ARN, including the version.
        :param logging_config: The function's Amazon CloudWatch Logs configuration settings.
        :param memory_size: The amount of `memory available to the function <https://docs.aws.amazon.com/lambda/latest/dg/configuration-function-common.html#configuration-memory-console>`_ at runtime. Increasing the function memory also increases its CPU allocation. The default value is 128 MB. The value can be any multiple of 1 MB. Note that new AWS accounts have reduced concurrency and memory quotas. AWS raises these quotas automatically based on your usage. You can also request a quota increase.
        :param package_type: The type of deployment package. Set to ``Image`` for container image and set ``Zip`` for .zip file archive.
        :param recursive_loop: The status of your function's recursive loop detection configuration. When this value is set to ``Allow`` and Lambda detects your function being invoked as part of a recursive loop, it doesn't take any action. When this value is set to ``Terminate`` and Lambda detects your function being invoked as part of a recursive loop, it stops your function being invoked and notifies you.
        :param reserved_concurrent_executions: The number of simultaneous executions to reserve for the function.
        :param runtime: The identifier of the function's `runtime <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Runtime is required if the deployment package is a .zip file archive. Specifying a runtime results in an error if you're deploying a function using a container image. The following list includes deprecated runtimes. Lambda blocks creating new functions and updating existing functions shortly after each runtime is deprecated. For more information, see `Runtime use after deprecation <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html#runtime-deprecation-levels>`_ . For a list of all currently supported runtimes, see `Supported runtimes <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html#runtimes-supported>`_ .
        :param runtime_management_config: Sets the runtime management configuration for a function's version. For more information, see `Runtime updates <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-update.html>`_ .
        :param snap_start: The function's `AWS Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.
        :param tags: A list of `tags <https://docs.aws.amazon.com/lambda/latest/dg/tagging.html>`_ to apply to the function. .. epigraph:: You must have the ``lambda:TagResource`` , ``lambda:UntagResource`` , and ``lambda:ListTags`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.
        :param timeout: The amount of time (in seconds) that Lambda allows a function to run before stopping it. The default is 3 seconds. The maximum allowed value is 900 seconds. For more information, see `Lambda execution environment <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-context.html>`_ .
        :param tracing_config: Set ``Mode`` to ``Active`` to sample and trace a subset of incoming requests with `X-Ray <https://docs.aws.amazon.com/lambda/latest/dg/services-xray.html>`_ .
        :param vpc_config: For network connectivity to AWS resources in a VPC, specify a list of security groups and subnets in the VPC. When you connect a function to a VPC, it can access resources and the internet only through that VPC. For more information, see `Configuring a Lambda function to access resources in a VPC <https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html>`_ .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d971f3872acf20816e6da364ff9e6bec83fe2e68bbb9a7debc845b40036ee553)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnFunctionProps(
            code=code,
            role=role,
            architectures=architectures,
            code_signing_config_arn=code_signing_config_arn,
            dead_letter_config=dead_letter_config,
            description=description,
            environment=environment,
            ephemeral_storage=ephemeral_storage,
            file_system_configs=file_system_configs,
            function_name=function_name,
            handler=handler,
            image_config=image_config,
            kms_key_arn=kms_key_arn,
            layers=layers,
            logging_config=logging_config,
            memory_size=memory_size,
            package_type=package_type,
            recursive_loop=recursive_loop,
            reserved_concurrent_executions=reserved_concurrent_executions,
            runtime=runtime,
            runtime_management_config=runtime_management_config,
            snap_start=snap_start,
            tags=tags,
            timeout=timeout,
            tracing_config=tracing_config,
            vpc_config=vpc_config,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__23f7f329458448a99a6be54e2f842c699a159be1c8aa208602a46d495a891364)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fa93f21bdc0c15ec3cc2b011fd1e46b83cf5a5d84449fa05e4ed632209a45e98)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the function.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrSnapStartResponse")
    def attr_snap_start_response(self) -> _IResolvable_da3f097b:
        '''The function's `SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.

        :cloudformationAttribute: SnapStartResponse
        '''
        return typing.cast(_IResolvable_da3f097b, jsii.get(self, "attrSnapStartResponse"))

    @builtins.property
    @jsii.member(jsii_name="attrSnapStartResponseApplyOn")
    def attr_snap_start_response_apply_on(self) -> builtins.str:
        '''When set to ``PublishedVersions``, Lambda creates a snapshot of the execution environment when you publish a function version.

        :cloudformationAttribute: SnapStartResponse.ApplyOn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrSnapStartResponseApplyOn"))

    @builtins.property
    @jsii.member(jsii_name="attrSnapStartResponseOptimizationStatus")
    def attr_snap_start_response_optimization_status(self) -> builtins.str:
        '''When you provide a `qualified Amazon Resource Name (ARN) <https://docs.aws.amazon.com/lambda/latest/dg/configuration-versions.html#versioning-versions-using>`_, this response element indicates whether SnapStart is activated for the specified function version.

        :cloudformationAttribute: SnapStartResponse.OptimizationStatus
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrSnapStartResponseOptimizationStatus"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="code")
    def code(self) -> typing.Union[_IResolvable_da3f097b, "CfnFunction.CodeProperty"]:
        '''The code for the function.

        You can define your function code in multiple ways:.
        '''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnFunction.CodeProperty"], jsii.get(self, "code"))

    @code.setter
    def code(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnFunction.CodeProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__58971d313bbb31e4a176049232e48d257e2707fa93bae32f75eda0c6327c3121)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "code", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the function's execution role.'''
        return typing.cast(builtins.str, jsii.get(self, "role"))

    @role.setter
    def role(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0087a2968c8a22b8dd1ff3bf410b2d9939dd4741222544ab3d1519ae865e7631)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "role", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="architectures")
    def architectures(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The instruction set architecture that the function supports.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "architectures"))

    @architectures.setter
    def architectures(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__939e9547234560a62ae5c4e5194b2aba7beab9431fc0692025911c49ac554075)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "architectures", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="codeSigningConfigArn")
    def code_signing_config_arn(self) -> typing.Optional[builtins.str]:
        '''To enable code signing for this function, specify the ARN of a code-signing configuration.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "codeSigningConfigArn"))

    @code_signing_config_arn.setter
    def code_signing_config_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__97ad6c3509dbd27b0115d978338b5c8b85fccf3544961894ef6c8d5c51bda760)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "codeSigningConfigArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="deadLetterConfig")
    def dead_letter_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.DeadLetterConfigProperty"]]:
        '''A dead-letter queue configuration that specifies the queue or topic where Lambda sends asynchronous events when they fail processing.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.DeadLetterConfigProperty"]], jsii.get(self, "deadLetterConfig"))

    @dead_letter_config.setter
    def dead_letter_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.DeadLetterConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8e7260bd37534ff38acbffbc9ae7d7473b79520cd7320ebd30eb30dda7fd6aa1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deadLetterConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the function.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9adaa36c21a67de531bddc847ecff2947d2a1b03b4f9d8d078da3e6849878bae)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="environment")
    def environment(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.EnvironmentProperty"]]:
        '''Environment variables that are accessible from function code during execution.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.EnvironmentProperty"]], jsii.get(self, "environment"))

    @environment.setter
    def environment(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.EnvironmentProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__317039d2b9e835e52b1f6d855c9ef90ff3e3707ae46ff7505464e0c0ba080e32)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "environment", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="ephemeralStorage")
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.EphemeralStorageProperty"]]:
        '''The size of the function's ``/tmp`` directory in MB.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.EphemeralStorageProperty"]], jsii.get(self, "ephemeralStorage"))

    @ephemeral_storage.setter
    def ephemeral_storage(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.EphemeralStorageProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__19d417a7ef6f6dbe162ed6a9f33a83f606d97e53766d0a551f162f1a623bfce6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ephemeralStorage", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="fileSystemConfigs")
    def file_system_configs(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFunction.FileSystemConfigProperty"]]]]:
        '''Connection settings for an Amazon EFS file system.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFunction.FileSystemConfigProperty"]]]], jsii.get(self, "fileSystemConfigs"))

    @file_system_configs.setter
    def file_system_configs(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFunction.FileSystemConfigProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42281c8b42403c865af9a471e854ee48ac9c9efd9ed8eb48e102fae4dca64865)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "fileSystemConfigs", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Lambda function, up to 64 characters in length.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9b755bdfff9c92d112f48fabc138610cb95c629791eb94011add8f30bb1a8b8c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="handler")
    def handler(self) -> typing.Optional[builtins.str]:
        '''The name of the method within your code that Lambda calls to run your function.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "handler"))

    @handler.setter
    def handler(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__baa7eecceb76515056bd0313e9ea4d285ed7f4fb588cfa29fbf02e30807dc972)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "handler", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="imageConfig")
    def image_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.ImageConfigProperty"]]:
        '''Configuration values that override the container image Dockerfile settings.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.ImageConfigProperty"]], jsii.get(self, "imageConfig"))

    @image_config.setter
    def image_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.ImageConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fc473fff3eab69974e86b6ab728c48e766dacc39517c01dfd62e30519976a9de)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "imageConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="kmsKeyArn")
    def kms_key_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the AWS Key Management Service ( AWS KMS ) customer managed key that's used to encrypt the following resources:.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kmsKeyArn"))

    @kms_key_arn.setter
    def kms_key_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__51919b7dbfb08288b0214a1e2e4192a9dedfca251449ba8cae053cc5c0940523)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsKeyArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="layers")
    def layers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of `function layers <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ to add to the function's execution environment. Specify each layer by its ARN, including the version.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "layers"))

    @layers.setter
    def layers(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c1d2d3db6c352a7fe83bbdeaaf29d76df26e3ede060d08c593d9ef7006236b0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "layers", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="loggingConfig")
    def logging_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.LoggingConfigProperty"]]:
        '''The function's Amazon CloudWatch Logs configuration settings.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.LoggingConfigProperty"]], jsii.get(self, "loggingConfig"))

    @logging_config.setter
    def logging_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.LoggingConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3ce38cf1c7b3713eca9725f79f526f3b663f05dec93fd7dff426e717e1dc9d7c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "loggingConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="memorySize")
    def memory_size(self) -> typing.Optional[jsii.Number]:
        '''The amount of `memory available to the function <https://docs.aws.amazon.com/lambda/latest/dg/configuration-function-common.html#configuration-memory-console>`_ at runtime. Increasing the function memory also increases its CPU allocation. The default value is 128 MB. The value can be any multiple of 1 MB. Note that new AWS accounts have reduced concurrency and memory quotas. AWS raises these quotas automatically based on your usage. You can also request a quota increase.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "memorySize"))

    @memory_size.setter
    def memory_size(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1cad27a1fc937a1ee790969066fef17df69ed3826b2b85f10db980930d32b0fd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "memorySize", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="packageType")
    def package_type(self) -> typing.Optional[builtins.str]:
        '''The type of deployment package.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "packageType"))

    @package_type.setter
    def package_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7f9bb5c748b58d456d8f996b07509c9991044629122c76de58eaabc34056846e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "packageType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="recursiveLoop")
    def recursive_loop(self) -> typing.Optional[builtins.str]:
        '''The status of your function's recursive loop detection configuration.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "recursiveLoop"))

    @recursive_loop.setter
    def recursive_loop(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__021080e0643735be55c0ce187b6c3482fd8d23af0a29c95ad94660f1cf184a90)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "recursiveLoop", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="reservedConcurrentExecutions")
    def reserved_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''The number of simultaneous executions to reserve for the function.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "reservedConcurrentExecutions"))

    @reserved_concurrent_executions.setter
    def reserved_concurrent_executions(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__776a8ed6e24e90be835cb39b621ff66305005b357a6debbf3a62f69d83289278)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "reservedConcurrentExecutions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="runtime")
    def runtime(self) -> typing.Optional[builtins.str]:
        '''The identifier of the function's `runtime <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Runtime is required if the deployment package is a .zip file archive. Specifying a runtime results in an error if you're deploying a function using a container image.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "runtime"))

    @runtime.setter
    def runtime(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b21ecd1fe965145c47a3a79e27c6cd009ded1d8446934f7912383ff1e67a2b03)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "runtime", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="runtimeManagementConfig")
    def runtime_management_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.RuntimeManagementConfigProperty"]]:
        '''Sets the runtime management configuration for a function's version.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.RuntimeManagementConfigProperty"]], jsii.get(self, "runtimeManagementConfig"))

    @runtime_management_config.setter
    def runtime_management_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.RuntimeManagementConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0eb811f391544dcb8176a15cfd354317bce582d6de91f3a92abe633f8871c3f5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "runtimeManagementConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="snapStart")
    def snap_start(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.SnapStartProperty"]]:
        '''The function's `AWS Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.SnapStartProperty"]], jsii.get(self, "snapStart"))

    @snap_start.setter
    def snap_start(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.SnapStartProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3da00ae175903856cd5e3dffd1383604e1e7cca41ca0995aaf7f3ff2e12ae143)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "snapStart", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''A list of `tags <https://docs.aws.amazon.com/lambda/latest/dg/tagging.html>`_ to apply to the function.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__174edd947716c328063162b1193e4d16e0c3ac15949326027a5bfae9fe02dbc1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="timeout")
    def timeout(self) -> typing.Optional[jsii.Number]:
        '''The amount of time (in seconds) that Lambda allows a function to run before stopping it.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "timeout"))

    @timeout.setter
    def timeout(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__75ee5de813fc2d2189124f8a0417b55380d01a0a385d8d51fa9f785efab2b32e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "timeout", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tracingConfig")
    def tracing_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.TracingConfigProperty"]]:
        '''Set ``Mode`` to ``Active`` to sample and trace a subset of incoming requests with `X-Ray <https://docs.aws.amazon.com/lambda/latest/dg/services-xray.html>`_ .'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.TracingConfigProperty"]], jsii.get(self, "tracingConfig"))

    @tracing_config.setter
    def tracing_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.TracingConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2e2ae2e6d2b74fc490159641853a6b239db977126bd780dc9b5d3931c1e47577)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tracingConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="vpcConfig")
    def vpc_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.VpcConfigProperty"]]:
        '''For network connectivity to AWS resources in a VPC, specify a list of security groups and subnets in the VPC.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.VpcConfigProperty"]], jsii.get(self, "vpcConfig"))

    @vpc_config.setter
    def vpc_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFunction.VpcConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__033cfa97016c411e045aae5002e1ed6a505aefcc7ef38007d08100468c8903b9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcConfig", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.CodeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "image_uri": "imageUri",
            "s3_bucket": "s3Bucket",
            "s3_key": "s3Key",
            "s3_object_version": "s3ObjectVersion",
            "source_kms_key_arn": "sourceKmsKeyArn",
            "zip_file": "zipFile",
        },
    )
    class CodeProperty:
        def __init__(
            self,
            *,
            image_uri: typing.Optional[builtins.str] = None,
            s3_bucket: typing.Optional[builtins.str] = None,
            s3_key: typing.Optional[builtins.str] = None,
            s3_object_version: typing.Optional[builtins.str] = None,
            source_kms_key_arn: typing.Optional[builtins.str] = None,
            zip_file: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The `deployment package <https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-package.html>`_ for a Lambda function. To deploy a function defined as a container image, you specify the location of a container image in the Amazon ECR registry. For a .zip file deployment package, you can specify the location of an object in Amazon S3. For Node.js and Python functions, you can specify the function code inline in the template.

            .. epigraph::

               When you specify source code inline for a Node.js function, the ``index`` file that AWS CloudFormation creates uses the extension ``.js`` . This means that Lambda treats the file as a CommonJS module. ES modules aren't supported for inline functions.

            Changes to a deployment package in Amazon S3 or a container image in ECR are not detected automatically during stack updates. To update the function code, change the object key or version in the template.

            :param image_uri: URI of a `container image <https://docs.aws.amazon.com/lambda/latest/dg/lambda-images.html>`_ in the Amazon ECR registry.
            :param s3_bucket: An Amazon S3 bucket in the same AWS Region as your function. The bucket can be in a different AWS account .
            :param s3_key: The Amazon S3 key of the deployment package.
            :param s3_object_version: For versioned objects, the version of the deployment package object to use.
            :param source_kms_key_arn: The ARN of the AWS Key Management Service ( AWS KMS ) customer managed key that's used to encrypt your function's .zip deployment package. If you don't provide a customer managed key, Lambda uses an `AWS owned key <https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-owned-cmk>`_ .
            :param zip_file: (Node.js and Python) The source code of your Lambda function. If you include your function source inline with this parameter, AWS CloudFormation places it in a file named ``index`` and zips it to create a `deployment package <https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-package.html>`_ . This zip file cannot exceed 4MB. For the ``Handler`` property, the first part of the handler identifier must be ``index`` . For example, ``index.handler`` . .. epigraph:: When you specify source code inline for a Node.js function, the ``index`` file that AWS CloudFormation creates uses the extension ``.js`` . This means that Lambda treats the file as a CommonJS module. ES modules aren't supported for inline functions. For JSON, you must escape quotes and special characters such as newline ( ``\\n`` ) with a backslash. If you specify a function that interacts with an AWS CloudFormation custom resource, you don't have to write your own functions to send responses to the custom resource that invoked the function. AWS CloudFormation provides a response module ( `cfn-response <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-lambda-function-code-cfnresponsemodule.html>`_ ) that simplifies sending responses. See `Using AWS Lambda with AWS CloudFormation <https://docs.aws.amazon.com/lambda/latest/dg/services-cloudformation.html>`_ for details.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                code_property = lambda.CfnFunction.CodeProperty(
                    image_uri="imageUri",
                    s3_bucket="s3Bucket",
                    s3_key="s3Key",
                    s3_object_version="s3ObjectVersion",
                    source_kms_key_arn="sourceKmsKeyArn",
                    zip_file="zipFile"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7102a6215772d5cf5b9392746e4c0cd11ba84424f7dbcd39e9d78ba0840d4e5c)
                check_type(argname="argument image_uri", value=image_uri, expected_type=type_hints["image_uri"])
                check_type(argname="argument s3_bucket", value=s3_bucket, expected_type=type_hints["s3_bucket"])
                check_type(argname="argument s3_key", value=s3_key, expected_type=type_hints["s3_key"])
                check_type(argname="argument s3_object_version", value=s3_object_version, expected_type=type_hints["s3_object_version"])
                check_type(argname="argument source_kms_key_arn", value=source_kms_key_arn, expected_type=type_hints["source_kms_key_arn"])
                check_type(argname="argument zip_file", value=zip_file, expected_type=type_hints["zip_file"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if image_uri is not None:
                self._values["image_uri"] = image_uri
            if s3_bucket is not None:
                self._values["s3_bucket"] = s3_bucket
            if s3_key is not None:
                self._values["s3_key"] = s3_key
            if s3_object_version is not None:
                self._values["s3_object_version"] = s3_object_version
            if source_kms_key_arn is not None:
                self._values["source_kms_key_arn"] = source_kms_key_arn
            if zip_file is not None:
                self._values["zip_file"] = zip_file

        @builtins.property
        def image_uri(self) -> typing.Optional[builtins.str]:
            '''URI of a `container image <https://docs.aws.amazon.com/lambda/latest/dg/lambda-images.html>`_ in the Amazon ECR registry.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-imageuri
            '''
            result = self._values.get("image_uri")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_bucket(self) -> typing.Optional[builtins.str]:
            '''An Amazon S3 bucket in the same AWS Region as your function.

            The bucket can be in a different AWS account .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-s3bucket
            '''
            result = self._values.get("s3_bucket")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_key(self) -> typing.Optional[builtins.str]:
            '''The Amazon S3 key of the deployment package.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-s3key
            '''
            result = self._values.get("s3_key")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_object_version(self) -> typing.Optional[builtins.str]:
            '''For versioned objects, the version of the deployment package object to use.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-s3objectversion
            '''
            result = self._values.get("s3_object_version")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def source_kms_key_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of the AWS Key Management Service ( AWS KMS ) customer managed key that's used to encrypt your function's .zip deployment package. If you don't provide a customer managed key, Lambda uses an `AWS owned key <https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-owned-cmk>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-sourcekmskeyarn
            '''
            result = self._values.get("source_kms_key_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def zip_file(self) -> typing.Optional[builtins.str]:
            '''(Node.js and Python) The source code of your Lambda function. If you include your function source inline with this parameter, AWS CloudFormation places it in a file named ``index`` and zips it to create a `deployment package <https://docs.aws.amazon.com/lambda/latest/dg/gettingstarted-package.html>`_ . This zip file cannot exceed 4MB. For the ``Handler`` property, the first part of the handler identifier must be ``index`` . For example, ``index.handler`` .

            .. epigraph::

               When you specify source code inline for a Node.js function, the ``index`` file that AWS CloudFormation creates uses the extension ``.js`` . This means that Lambda treats the file as a CommonJS module. ES modules aren't supported for inline functions.

            For JSON, you must escape quotes and special characters such as newline ( ``\\n`` ) with a backslash.

            If you specify a function that interacts with an AWS CloudFormation custom resource, you don't have to write your own functions to send responses to the custom resource that invoked the function. AWS CloudFormation provides a response module ( `cfn-response <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-lambda-function-code-cfnresponsemodule.html>`_ ) that simplifies sending responses. See `Using AWS Lambda with AWS CloudFormation <https://docs.aws.amazon.com/lambda/latest/dg/services-cloudformation.html>`_ for details.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-zipfile
            '''
            result = self._values.get("zip_file")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CodeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.DeadLetterConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"target_arn": "targetArn"},
    )
    class DeadLetterConfigProperty:
        def __init__(self, *, target_arn: typing.Optional[builtins.str] = None) -> None:
            '''The `dead-letter queue <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#dlq>`_ for failed asynchronous invocations.

            :param target_arn: The Amazon Resource Name (ARN) of an Amazon SQS queue or Amazon SNS topic.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-deadletterconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                dead_letter_config_property = lambda.CfnFunction.DeadLetterConfigProperty(
                    target_arn="targetArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a812f89a4720ebfd342e2bae3205898204a71a6d7c463a024623d8c6bab023ff)
                check_type(argname="argument target_arn", value=target_arn, expected_type=type_hints["target_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if target_arn is not None:
                self._values["target_arn"] = target_arn

        @builtins.property
        def target_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of an Amazon SQS queue or Amazon SNS topic.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-deadletterconfig.html#cfn-lambda-function-deadletterconfig-targetarn
            '''
            result = self._values.get("target_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeadLetterConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.EnvironmentProperty",
        jsii_struct_bases=[],
        name_mapping={"variables": "variables"},
    )
    class EnvironmentProperty:
        def __init__(
            self,
            *,
            variables: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        ) -> None:
            '''A function's environment variable settings.

            You can use environment variables to adjust your function's behavior without updating code. An environment variable is a pair of strings that are stored in a function's version-specific configuration.

            :param variables: Environment variable key-value pairs. For more information, see `Using Lambda environment variables <https://docs.aws.amazon.com/lambda/latest/dg/configuration-envvars.html>`_ . If the value of the environment variable is a time or a duration, enclose the value in quotes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-environment.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                environment_property = lambda.CfnFunction.EnvironmentProperty(
                    variables={
                        "variables_key": "variables"
                    }
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1e6acbe5ce51d5a0f4abaff5a4e2373a76927c34b3ec39525f796932f708bde1)
                check_type(argname="argument variables", value=variables, expected_type=type_hints["variables"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if variables is not None:
                self._values["variables"] = variables

        @builtins.property
        def variables(
            self,
        ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
            '''Environment variable key-value pairs. For more information, see `Using Lambda environment variables <https://docs.aws.amazon.com/lambda/latest/dg/configuration-envvars.html>`_ .

            If the value of the environment variable is a time or a duration, enclose the value in quotes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-environment.html#cfn-lambda-function-environment-variables
            '''
            result = self._values.get("variables")
            return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EnvironmentProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.EphemeralStorageProperty",
        jsii_struct_bases=[],
        name_mapping={"size": "size"},
    )
    class EphemeralStorageProperty:
        def __init__(self, *, size: jsii.Number) -> None:
            '''The size of the function's ``/tmp`` directory in MB.

            The default value is 512, but it can be any whole number between 512 and 10,240 MB.

            :param size: The size of the function's ``/tmp`` directory.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-ephemeralstorage.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                ephemeral_storage_property = lambda.CfnFunction.EphemeralStorageProperty(
                    size=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6469046a2e3d6c9357d76baad38a3aea7c877ed959043d696ea793f983ea5c0e)
                check_type(argname="argument size", value=size, expected_type=type_hints["size"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "size": size,
            }

        @builtins.property
        def size(self) -> jsii.Number:
            '''The size of the function's ``/tmp`` directory.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-ephemeralstorage.html#cfn-lambda-function-ephemeralstorage-size
            '''
            result = self._values.get("size")
            assert result is not None, "Required property 'size' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EphemeralStorageProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.FileSystemConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"arn": "arn", "local_mount_path": "localMountPath"},
    )
    class FileSystemConfigProperty:
        def __init__(
            self,
            *,
            arn: builtins.str,
            local_mount_path: builtins.str,
        ) -> None:
            '''Details about the connection between a Lambda function and an `Amazon EFS file system <https://docs.aws.amazon.com/lambda/latest/dg/configuration-filesystem.html>`_ .

            :param arn: The Amazon Resource Name (ARN) of the Amazon EFS access point that provides access to the file system.
            :param local_mount_path: The path where the function can access the file system, starting with ``/mnt/`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-filesystemconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                file_system_config_property = lambda.CfnFunction.FileSystemConfigProperty(
                    arn="arn",
                    local_mount_path="localMountPath"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c0060fc6e723ccc0f68bebab137926b678e758f90348279d34ce1f7ff1522bc6)
                check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
                check_type(argname="argument local_mount_path", value=local_mount_path, expected_type=type_hints["local_mount_path"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "arn": arn,
                "local_mount_path": local_mount_path,
            }

        @builtins.property
        def arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the Amazon EFS access point that provides access to the file system.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-filesystemconfig.html#cfn-lambda-function-filesystemconfig-arn
            '''
            result = self._values.get("arn")
            assert result is not None, "Required property 'arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def local_mount_path(self) -> builtins.str:
            '''The path where the function can access the file system, starting with ``/mnt/`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-filesystemconfig.html#cfn-lambda-function-filesystemconfig-localmountpath
            '''
            result = self._values.get("local_mount_path")
            assert result is not None, "Required property 'local_mount_path' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FileSystemConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.ImageConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "command": "command",
            "entry_point": "entryPoint",
            "working_directory": "workingDirectory",
        },
    )
    class ImageConfigProperty:
        def __init__(
            self,
            *,
            command: typing.Optional[typing.Sequence[builtins.str]] = None,
            entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
            working_directory: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Configuration values that override the container image Dockerfile settings.

            For more information, see `Container image settings <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-parms>`_ .

            :param command: Specifies parameters that you want to pass in with ENTRYPOINT. You can specify a maximum of 1,500 parameters in the list.
            :param entry_point: Specifies the entry point to their application, which is typically the location of the runtime executable. You can specify a maximum of 1,500 string entries in the list.
            :param working_directory: Specifies the working directory. The length of the directory string cannot exceed 1,000 characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-imageconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                image_config_property = lambda.CfnFunction.ImageConfigProperty(
                    command=["command"],
                    entry_point=["entryPoint"],
                    working_directory="workingDirectory"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__58e6ee2d8441c4bd7e4483bde9f8b942dfb9f61120db1c1e4d4fb7c17a110b59)
                check_type(argname="argument command", value=command, expected_type=type_hints["command"])
                check_type(argname="argument entry_point", value=entry_point, expected_type=type_hints["entry_point"])
                check_type(argname="argument working_directory", value=working_directory, expected_type=type_hints["working_directory"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if command is not None:
                self._values["command"] = command
            if entry_point is not None:
                self._values["entry_point"] = entry_point
            if working_directory is not None:
                self._values["working_directory"] = working_directory

        @builtins.property
        def command(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specifies parameters that you want to pass in with ENTRYPOINT.

            You can specify a maximum of 1,500 parameters in the list.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-imageconfig.html#cfn-lambda-function-imageconfig-command
            '''
            result = self._values.get("command")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def entry_point(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specifies the entry point to their application, which is typically the location of the runtime executable.

            You can specify a maximum of 1,500 string entries in the list.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-imageconfig.html#cfn-lambda-function-imageconfig-entrypoint
            '''
            result = self._values.get("entry_point")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def working_directory(self) -> typing.Optional[builtins.str]:
            '''Specifies the working directory.

            The length of the directory string cannot exceed 1,000 characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-imageconfig.html#cfn-lambda-function-imageconfig-workingdirectory
            '''
            result = self._values.get("working_directory")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ImageConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.LoggingConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "application_log_level": "applicationLogLevel",
            "log_format": "logFormat",
            "log_group": "logGroup",
            "system_log_level": "systemLogLevel",
        },
    )
    class LoggingConfigProperty:
        def __init__(
            self,
            *,
            application_log_level: typing.Optional[builtins.str] = None,
            log_format: typing.Optional[builtins.str] = None,
            log_group: typing.Optional[builtins.str] = None,
            system_log_level: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The function's Amazon CloudWatch Logs configuration settings.

            :param application_log_level: Set this property to filter the application logs for your function that Lambda sends to CloudWatch. Lambda only sends application logs at the selected level of detail and lower, where ``TRACE`` is the highest level and ``FATAL`` is the lowest.
            :param log_format: The format in which Lambda sends your function's application and system logs to CloudWatch. Select between plain text and structured JSON.
            :param log_group: The name of the Amazon CloudWatch log group the function sends logs to. By default, Lambda functions send logs to a default log group named ``/aws/lambda/<function name>`` . To use a different log group, enter an existing log group or enter a new log group name.
            :param system_log_level: Set this property to filter the system logs for your function that Lambda sends to CloudWatch. Lambda only sends system logs at the selected level of detail and lower, where ``DEBUG`` is the highest level and ``WARN`` is the lowest.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-loggingconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                logging_config_property = lambda.CfnFunction.LoggingConfigProperty(
                    application_log_level="applicationLogLevel",
                    log_format="logFormat",
                    log_group="logGroup",
                    system_log_level="systemLogLevel"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e08ade62ca6578ed0ba00e3acd7b2efb522123c93a764991846c882e7039e16f)
                check_type(argname="argument application_log_level", value=application_log_level, expected_type=type_hints["application_log_level"])
                check_type(argname="argument log_format", value=log_format, expected_type=type_hints["log_format"])
                check_type(argname="argument log_group", value=log_group, expected_type=type_hints["log_group"])
                check_type(argname="argument system_log_level", value=system_log_level, expected_type=type_hints["system_log_level"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if application_log_level is not None:
                self._values["application_log_level"] = application_log_level
            if log_format is not None:
                self._values["log_format"] = log_format
            if log_group is not None:
                self._values["log_group"] = log_group
            if system_log_level is not None:
                self._values["system_log_level"] = system_log_level

        @builtins.property
        def application_log_level(self) -> typing.Optional[builtins.str]:
            '''Set this property to filter the application logs for your function that Lambda sends to CloudWatch.

            Lambda only sends application logs at the selected level of detail and lower, where ``TRACE`` is the highest level and ``FATAL`` is the lowest.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-loggingconfig.html#cfn-lambda-function-loggingconfig-applicationloglevel
            '''
            result = self._values.get("application_log_level")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def log_format(self) -> typing.Optional[builtins.str]:
            '''The format in which Lambda sends your function's application and system logs to CloudWatch.

            Select between plain text and structured JSON.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-loggingconfig.html#cfn-lambda-function-loggingconfig-logformat
            '''
            result = self._values.get("log_format")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def log_group(self) -> typing.Optional[builtins.str]:
            '''The name of the Amazon CloudWatch log group the function sends logs to.

            By default, Lambda functions send logs to a default log group named ``/aws/lambda/<function name>`` . To use a different log group, enter an existing log group or enter a new log group name.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-loggingconfig.html#cfn-lambda-function-loggingconfig-loggroup
            '''
            result = self._values.get("log_group")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def system_log_level(self) -> typing.Optional[builtins.str]:
            '''Set this property to filter the system logs for your function that Lambda sends to CloudWatch.

            Lambda only sends system logs at the selected level of detail and lower, where ``DEBUG`` is the highest level and ``WARN`` is the lowest.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-loggingconfig.html#cfn-lambda-function-loggingconfig-systemloglevel
            '''
            result = self._values.get("system_log_level")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.RuntimeManagementConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "update_runtime_on": "updateRuntimeOn",
            "runtime_version_arn": "runtimeVersionArn",
        },
    )
    class RuntimeManagementConfigProperty:
        def __init__(
            self,
            *,
            update_runtime_on: builtins.str,
            runtime_version_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Sets the runtime management configuration for a function's version.

            For more information, see `Runtime updates <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-update.html>`_ .

            :param update_runtime_on: Specify the runtime update mode. - *Auto (default)* - Automatically update to the most recent and secure runtime version using a `Two-phase runtime version rollout <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-update.html#runtime-management-two-phase>`_ . This is the best choice for most customers to ensure they always benefit from runtime updates. - *FunctionUpdate* - Lambda updates the runtime of you function to the most recent and secure runtime version when you update your function. This approach synchronizes runtime updates with function deployments, giving you control over when runtime updates are applied and allowing you to detect and mitigate rare runtime update incompatibilities early. When using this setting, you need to regularly update your functions to keep their runtime up-to-date. - *Manual* - You specify a runtime version in your function configuration. The function will use this runtime version indefinitely. In the rare case where a new runtime version is incompatible with an existing function, this allows you to roll back your function to an earlier runtime version. For more information, see `Roll back a runtime version <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-update.html#runtime-management-rollback>`_ . *Valid Values* : ``Auto`` | ``FunctionUpdate`` | ``Manual``
            :param runtime_version_arn: The ARN of the runtime version you want the function to use. .. epigraph:: This is only required if you're using the *Manual* runtime update mode.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-runtimemanagementconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                runtime_management_config_property = lambda.CfnFunction.RuntimeManagementConfigProperty(
                    update_runtime_on="updateRuntimeOn",
                
                    # the properties below are optional
                    runtime_version_arn="runtimeVersionArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__30a323208760254dc626db9a6b7cd7d96fe15c4516882e470f8d0fcbc382ba92)
                check_type(argname="argument update_runtime_on", value=update_runtime_on, expected_type=type_hints["update_runtime_on"])
                check_type(argname="argument runtime_version_arn", value=runtime_version_arn, expected_type=type_hints["runtime_version_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "update_runtime_on": update_runtime_on,
            }
            if runtime_version_arn is not None:
                self._values["runtime_version_arn"] = runtime_version_arn

        @builtins.property
        def update_runtime_on(self) -> builtins.str:
            '''Specify the runtime update mode.

            - *Auto (default)* - Automatically update to the most recent and secure runtime version using a `Two-phase runtime version rollout <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-update.html#runtime-management-two-phase>`_ . This is the best choice for most customers to ensure they always benefit from runtime updates.
            - *FunctionUpdate* - Lambda updates the runtime of you function to the most recent and secure runtime version when you update your function. This approach synchronizes runtime updates with function deployments, giving you control over when runtime updates are applied and allowing you to detect and mitigate rare runtime update incompatibilities early. When using this setting, you need to regularly update your functions to keep their runtime up-to-date.
            - *Manual* - You specify a runtime version in your function configuration. The function will use this runtime version indefinitely. In the rare case where a new runtime version is incompatible with an existing function, this allows you to roll back your function to an earlier runtime version. For more information, see `Roll back a runtime version <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-update.html#runtime-management-rollback>`_ .

            *Valid Values* : ``Auto`` | ``FunctionUpdate`` | ``Manual``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-runtimemanagementconfig.html#cfn-lambda-function-runtimemanagementconfig-updateruntimeon
            '''
            result = self._values.get("update_runtime_on")
            assert result is not None, "Required property 'update_runtime_on' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def runtime_version_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of the runtime version you want the function to use.

            .. epigraph::

               This is only required if you're using the *Manual* runtime update mode.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-runtimemanagementconfig.html#cfn-lambda-function-runtimemanagementconfig-runtimeversionarn
            '''
            result = self._values.get("runtime_version_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RuntimeManagementConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.SnapStartProperty",
        jsii_struct_bases=[],
        name_mapping={"apply_on": "applyOn"},
    )
    class SnapStartProperty:
        def __init__(self, *, apply_on: builtins.str) -> None:
            '''The function's `AWS Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.

            :param apply_on: Set ``ApplyOn`` to ``PublishedVersions`` to create a snapshot of the initialized execution environment when you publish a function version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-snapstart.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                snap_start_property = lambda.CfnFunction.SnapStartProperty(
                    apply_on="applyOn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__44d5e1d2513874c8e59e8a09c7ff8b50cbb6ea7fc2365c62d8a8d5eb668d6aab)
                check_type(argname="argument apply_on", value=apply_on, expected_type=type_hints["apply_on"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "apply_on": apply_on,
            }

        @builtins.property
        def apply_on(self) -> builtins.str:
            '''Set ``ApplyOn`` to ``PublishedVersions`` to create a snapshot of the initialized execution environment when you publish a function version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-snapstart.html#cfn-lambda-function-snapstart-applyon
            '''
            result = self._values.get("apply_on")
            assert result is not None, "Required property 'apply_on' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SnapStartProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.SnapStartResponseProperty",
        jsii_struct_bases=[],
        name_mapping={
            "apply_on": "applyOn",
            "optimization_status": "optimizationStatus",
        },
    )
    class SnapStartResponseProperty:
        def __init__(
            self,
            *,
            apply_on: typing.Optional[builtins.str] = None,
            optimization_status: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The function's `SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.

            :param apply_on: When set to ``PublishedVersions`` , Lambda creates a snapshot of the execution environment when you publish a function version.
            :param optimization_status: When you provide a `qualified Amazon Resource Name (ARN) <https://docs.aws.amazon.com/lambda/latest/dg/configuration-versions.html#versioning-versions-using>`_ , this response element indicates whether SnapStart is activated for the specified function version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-snapstartresponse.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                snap_start_response_property = lambda.CfnFunction.SnapStartResponseProperty(
                    apply_on="applyOn",
                    optimization_status="optimizationStatus"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a8c30fc32979de48dec265fdc148bad358e17833f41e94ce229570c6025c472d)
                check_type(argname="argument apply_on", value=apply_on, expected_type=type_hints["apply_on"])
                check_type(argname="argument optimization_status", value=optimization_status, expected_type=type_hints["optimization_status"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if apply_on is not None:
                self._values["apply_on"] = apply_on
            if optimization_status is not None:
                self._values["optimization_status"] = optimization_status

        @builtins.property
        def apply_on(self) -> typing.Optional[builtins.str]:
            '''When set to ``PublishedVersions`` , Lambda creates a snapshot of the execution environment when you publish a function version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-snapstartresponse.html#cfn-lambda-function-snapstartresponse-applyon
            '''
            result = self._values.get("apply_on")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def optimization_status(self) -> typing.Optional[builtins.str]:
            '''When you provide a `qualified Amazon Resource Name (ARN) <https://docs.aws.amazon.com/lambda/latest/dg/configuration-versions.html#versioning-versions-using>`_ , this response element indicates whether SnapStart is activated for the specified function version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-snapstartresponse.html#cfn-lambda-function-snapstartresponse-optimizationstatus
            '''
            result = self._values.get("optimization_status")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SnapStartResponseProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.TracingConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"mode": "mode"},
    )
    class TracingConfigProperty:
        def __init__(self, *, mode: typing.Optional[builtins.str] = None) -> None:
            '''The function's `AWS X-Ray <https://docs.aws.amazon.com/lambda/latest/dg/services-xray.html>`_ tracing configuration. To sample and record incoming requests, set ``Mode`` to ``Active`` .

            :param mode: The tracing mode.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-tracingconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                tracing_config_property = lambda.CfnFunction.TracingConfigProperty(
                    mode="mode"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__65e495ff14e5431c99933211c1c9d543a50b4aefcbe3cefdac26535ec6a0582a)
                check_type(argname="argument mode", value=mode, expected_type=type_hints["mode"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if mode is not None:
                self._values["mode"] = mode

        @builtins.property
        def mode(self) -> typing.Optional[builtins.str]:
            '''The tracing mode.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-tracingconfig.html#cfn-lambda-function-tracingconfig-mode
            '''
            result = self._values.get("mode")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TracingConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnFunction.VpcConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "ipv6_allowed_for_dual_stack": "ipv6AllowedForDualStack",
            "security_group_ids": "securityGroupIds",
            "subnet_ids": "subnetIds",
        },
    )
    class VpcConfigProperty:
        def __init__(
            self,
            *,
            ipv6_allowed_for_dual_stack: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
            subnet_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The VPC security groups and subnets that are attached to a Lambda function.

            When you connect a function to a VPC, Lambda creates an elastic network interface for each combination of security group and subnet in the function's VPC configuration. The function can only access resources and the internet through that VPC. For more information, see `VPC Settings <https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html>`_ .
            .. epigraph::

               When you delete a function, AWS CloudFormation monitors the state of its network interfaces and waits for Lambda to delete them before proceeding. If the VPC is defined in the same stack, the network interfaces need to be deleted by Lambda before AWS CloudFormation can delete the VPC's resources.

               To monitor network interfaces, AWS CloudFormation needs the ``ec2:DescribeNetworkInterfaces`` permission. It obtains this from the user or role that modifies the stack. If you don't provide this permission, AWS CloudFormation does not wait for network interfaces to be deleted.

            :param ipv6_allowed_for_dual_stack: Allows outbound IPv6 traffic on VPC functions that are connected to dual-stack subnets.
            :param security_group_ids: A list of VPC security group IDs.
            :param subnet_ids: A list of VPC subnet IDs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-vpcconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                vpc_config_property = lambda.CfnFunction.VpcConfigProperty(
                    ipv6_allowed_for_dual_stack=False,
                    security_group_ids=["securityGroupIds"],
                    subnet_ids=["subnetIds"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f84311d147ce21fce755207e6e5de59a158ee8d2d1ad51b2f80c05f54cf8365d)
                check_type(argname="argument ipv6_allowed_for_dual_stack", value=ipv6_allowed_for_dual_stack, expected_type=type_hints["ipv6_allowed_for_dual_stack"])
                check_type(argname="argument security_group_ids", value=security_group_ids, expected_type=type_hints["security_group_ids"])
                check_type(argname="argument subnet_ids", value=subnet_ids, expected_type=type_hints["subnet_ids"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if ipv6_allowed_for_dual_stack is not None:
                self._values["ipv6_allowed_for_dual_stack"] = ipv6_allowed_for_dual_stack
            if security_group_ids is not None:
                self._values["security_group_ids"] = security_group_ids
            if subnet_ids is not None:
                self._values["subnet_ids"] = subnet_ids

        @builtins.property
        def ipv6_allowed_for_dual_stack(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Allows outbound IPv6 traffic on VPC functions that are connected to dual-stack subnets.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-vpcconfig.html#cfn-lambda-function-vpcconfig-ipv6allowedfordualstack
            '''
            result = self._values.get("ipv6_allowed_for_dual_stack")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of VPC security group IDs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-vpcconfig.html#cfn-lambda-function-vpcconfig-securitygroupids
            '''
            result = self._values.get("security_group_ids")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def subnet_ids(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of VPC subnet IDs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-vpcconfig.html#cfn-lambda-function-vpcconfig-subnetids
            '''
            result = self._values.get("subnet_ids")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VpcConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CfnFunctionProps",
    jsii_struct_bases=[],
    name_mapping={
        "code": "code",
        "role": "role",
        "architectures": "architectures",
        "code_signing_config_arn": "codeSigningConfigArn",
        "dead_letter_config": "deadLetterConfig",
        "description": "description",
        "environment": "environment",
        "ephemeral_storage": "ephemeralStorage",
        "file_system_configs": "fileSystemConfigs",
        "function_name": "functionName",
        "handler": "handler",
        "image_config": "imageConfig",
        "kms_key_arn": "kmsKeyArn",
        "layers": "layers",
        "logging_config": "loggingConfig",
        "memory_size": "memorySize",
        "package_type": "packageType",
        "recursive_loop": "recursiveLoop",
        "reserved_concurrent_executions": "reservedConcurrentExecutions",
        "runtime": "runtime",
        "runtime_management_config": "runtimeManagementConfig",
        "snap_start": "snapStart",
        "tags": "tags",
        "timeout": "timeout",
        "tracing_config": "tracingConfig",
        "vpc_config": "vpcConfig",
    },
)
class CfnFunctionProps:
    def __init__(
        self,
        *,
        code: typing.Union[_IResolvable_da3f097b, typing.Union[CfnFunction.CodeProperty, typing.Dict[builtins.str, typing.Any]]],
        role: builtins.str,
        architectures: typing.Optional[typing.Sequence[builtins.str]] = None,
        code_signing_config_arn: typing.Optional[builtins.str] = None,
        dead_letter_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFunction.DeadLetterConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFunction.EnvironmentProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        ephemeral_storage: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFunction.EphemeralStorageProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        file_system_configs: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFunction.FileSystemConfigProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        function_name: typing.Optional[builtins.str] = None,
        handler: typing.Optional[builtins.str] = None,
        image_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFunction.ImageConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        kms_key_arn: typing.Optional[builtins.str] = None,
        layers: typing.Optional[typing.Sequence[builtins.str]] = None,
        logging_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFunction.LoggingConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        memory_size: typing.Optional[jsii.Number] = None,
        package_type: typing.Optional[builtins.str] = None,
        recursive_loop: typing.Optional[builtins.str] = None,
        reserved_concurrent_executions: typing.Optional[jsii.Number] = None,
        runtime: typing.Optional[builtins.str] = None,
        runtime_management_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFunction.RuntimeManagementConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        snap_start: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFunction.SnapStartProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        timeout: typing.Optional[jsii.Number] = None,
        tracing_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFunction.TracingConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        vpc_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFunction.VpcConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnFunction``.

        :param code: The code for the function. You can define your function code in multiple ways:. - For .zip deployment packages, you can specify the Amazon S3 location of the .zip file in the ``S3Bucket`` , ``S3Key`` , and ``S3ObjectVersion`` properties. - For .zip deployment packages, you can alternatively define the function code inline in the ``ZipFile`` property. This method works only for Node.js and Python functions. - For container images, specify the URI of your container image in the Amazon ECR registry in the ``ImageUri`` property.
        :param role: The Amazon Resource Name (ARN) of the function's execution role.
        :param architectures: The instruction set architecture that the function supports. Enter a string array with one of the valid values (arm64 or x86_64). The default value is ``x86_64`` .
        :param code_signing_config_arn: To enable code signing for this function, specify the ARN of a code-signing configuration. A code-signing configuration includes a set of signing profiles, which define the trusted publishers for this function.
        :param dead_letter_config: A dead-letter queue configuration that specifies the queue or topic where Lambda sends asynchronous events when they fail processing. For more information, see `Dead-letter queues <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-dlq>`_ .
        :param description: A description of the function.
        :param environment: Environment variables that are accessible from function code during execution.
        :param ephemeral_storage: The size of the function's ``/tmp`` directory in MB. The default value is 512, but it can be any whole number between 512 and 10,240 MB.
        :param file_system_configs: Connection settings for an Amazon EFS file system. To connect a function to a file system, a mount target must be available in every Availability Zone that your function connects to. If your template contains an `AWS::EFS::MountTarget <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-mounttarget.html>`_ resource, you must also specify a ``DependsOn`` attribute to ensure that the mount target is created or updated before the function. For more information about using the ``DependsOn`` attribute, see `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ .
        :param function_name: The name of the Lambda function, up to 64 characters in length. If you don't specify a name, AWS CloudFormation generates one. If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param handler: The name of the method within your code that Lambda calls to run your function. Handler is required if the deployment package is a .zip file archive. The format includes the file name. It can also include namespaces and other qualifiers, depending on the runtime. For more information, see `Lambda programming model <https://docs.aws.amazon.com/lambda/latest/dg/foundation-progmodel.html>`_ .
        :param image_config: Configuration values that override the container image Dockerfile settings. For more information, see `Container image settings <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-parms>`_ .
        :param kms_key_arn: The ARN of the AWS Key Management Service ( AWS KMS ) customer managed key that's used to encrypt the following resources:. - The function's `environment variables <https://docs.aws.amazon.com/lambda/latest/dg/configuration-envvars.html#configuration-envvars-encryption>`_ . - The function's `Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart-security.html>`_ snapshots. - When used with ``SourceKMSKeyArn`` , the unzipped version of the .zip deployment package that's used for function invocations. For more information, see `Specifying a customer managed key for Lambda <https://docs.aws.amazon.com/lambda/latest/dg/encrypt-zip-package.html#enable-zip-custom-encryption>`_ . - The optimized version of the container image that's used for function invocations. Note that this is not the same key that's used to protect your container image in the Amazon Elastic Container Registry (Amazon ECR). For more information, see `Function lifecycle <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-lifecycle>`_ . If you don't provide a customer managed key, Lambda uses an `AWS owned key <https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-owned-cmk>`_ or an `AWS managed key <https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-managed-cmk>`_ .
        :param layers: A list of `function layers <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ to add to the function's execution environment. Specify each layer by its ARN, including the version.
        :param logging_config: The function's Amazon CloudWatch Logs configuration settings.
        :param memory_size: The amount of `memory available to the function <https://docs.aws.amazon.com/lambda/latest/dg/configuration-function-common.html#configuration-memory-console>`_ at runtime. Increasing the function memory also increases its CPU allocation. The default value is 128 MB. The value can be any multiple of 1 MB. Note that new AWS accounts have reduced concurrency and memory quotas. AWS raises these quotas automatically based on your usage. You can also request a quota increase.
        :param package_type: The type of deployment package. Set to ``Image`` for container image and set ``Zip`` for .zip file archive.
        :param recursive_loop: The status of your function's recursive loop detection configuration. When this value is set to ``Allow`` and Lambda detects your function being invoked as part of a recursive loop, it doesn't take any action. When this value is set to ``Terminate`` and Lambda detects your function being invoked as part of a recursive loop, it stops your function being invoked and notifies you.
        :param reserved_concurrent_executions: The number of simultaneous executions to reserve for the function.
        :param runtime: The identifier of the function's `runtime <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Runtime is required if the deployment package is a .zip file archive. Specifying a runtime results in an error if you're deploying a function using a container image. The following list includes deprecated runtimes. Lambda blocks creating new functions and updating existing functions shortly after each runtime is deprecated. For more information, see `Runtime use after deprecation <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html#runtime-deprecation-levels>`_ . For a list of all currently supported runtimes, see `Supported runtimes <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html#runtimes-supported>`_ .
        :param runtime_management_config: Sets the runtime management configuration for a function's version. For more information, see `Runtime updates <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-update.html>`_ .
        :param snap_start: The function's `AWS Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.
        :param tags: A list of `tags <https://docs.aws.amazon.com/lambda/latest/dg/tagging.html>`_ to apply to the function. .. epigraph:: You must have the ``lambda:TagResource`` , ``lambda:UntagResource`` , and ``lambda:ListTags`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.
        :param timeout: The amount of time (in seconds) that Lambda allows a function to run before stopping it. The default is 3 seconds. The maximum allowed value is 900 seconds. For more information, see `Lambda execution environment <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-context.html>`_ .
        :param tracing_config: Set ``Mode`` to ``Active`` to sample and trace a subset of incoming requests with `X-Ray <https://docs.aws.amazon.com/lambda/latest/dg/services-xray.html>`_ .
        :param vpc_config: For network connectivity to AWS resources in a VPC, specify a list of security groups and subnets in the VPC. When you connect a function to a VPC, it can access resources and the internet only through that VPC. For more information, see `Configuring a Lambda function to access resources in a VPC <https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            cfn_function_props = lambda.CfnFunctionProps(
                code=lambda.CfnFunction.CodeProperty(
                    image_uri="imageUri",
                    s3_bucket="s3Bucket",
                    s3_key="s3Key",
                    s3_object_version="s3ObjectVersion",
                    source_kms_key_arn="sourceKmsKeyArn",
                    zip_file="zipFile"
                ),
                role="role",
            
                # the properties below are optional
                architectures=["architectures"],
                code_signing_config_arn="codeSigningConfigArn",
                dead_letter_config=lambda.CfnFunction.DeadLetterConfigProperty(
                    target_arn="targetArn"
                ),
                description="description",
                environment=lambda.CfnFunction.EnvironmentProperty(
                    variables={
                        "variables_key": "variables"
                    }
                ),
                ephemeral_storage=lambda.CfnFunction.EphemeralStorageProperty(
                    size=123
                ),
                file_system_configs=[lambda.CfnFunction.FileSystemConfigProperty(
                    arn="arn",
                    local_mount_path="localMountPath"
                )],
                function_name="functionName",
                handler="handler",
                image_config=lambda.CfnFunction.ImageConfigProperty(
                    command=["command"],
                    entry_point=["entryPoint"],
                    working_directory="workingDirectory"
                ),
                kms_key_arn="kmsKeyArn",
                layers=["layers"],
                logging_config=lambda.CfnFunction.LoggingConfigProperty(
                    application_log_level="applicationLogLevel",
                    log_format="logFormat",
                    log_group="logGroup",
                    system_log_level="systemLogLevel"
                ),
                memory_size=123,
                package_type="packageType",
                recursive_loop="recursiveLoop",
                reserved_concurrent_executions=123,
                runtime="runtime",
                runtime_management_config=lambda.CfnFunction.RuntimeManagementConfigProperty(
                    update_runtime_on="updateRuntimeOn",
            
                    # the properties below are optional
                    runtime_version_arn="runtimeVersionArn"
                ),
                snap_start=lambda.CfnFunction.SnapStartProperty(
                    apply_on="applyOn"
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                timeout=123,
                tracing_config=lambda.CfnFunction.TracingConfigProperty(
                    mode="mode"
                ),
                vpc_config=lambda.CfnFunction.VpcConfigProperty(
                    ipv6_allowed_for_dual_stack=False,
                    security_group_ids=["securityGroupIds"],
                    subnet_ids=["subnetIds"]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__06b7f494e25475a49ebed0d7ed6d0fca9653b5fa5e00ded0cba4fc40aebc3532)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument architectures", value=architectures, expected_type=type_hints["architectures"])
            check_type(argname="argument code_signing_config_arn", value=code_signing_config_arn, expected_type=type_hints["code_signing_config_arn"])
            check_type(argname="argument dead_letter_config", value=dead_letter_config, expected_type=type_hints["dead_letter_config"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument ephemeral_storage", value=ephemeral_storage, expected_type=type_hints["ephemeral_storage"])
            check_type(argname="argument file_system_configs", value=file_system_configs, expected_type=type_hints["file_system_configs"])
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument handler", value=handler, expected_type=type_hints["handler"])
            check_type(argname="argument image_config", value=image_config, expected_type=type_hints["image_config"])
            check_type(argname="argument kms_key_arn", value=kms_key_arn, expected_type=type_hints["kms_key_arn"])
            check_type(argname="argument layers", value=layers, expected_type=type_hints["layers"])
            check_type(argname="argument logging_config", value=logging_config, expected_type=type_hints["logging_config"])
            check_type(argname="argument memory_size", value=memory_size, expected_type=type_hints["memory_size"])
            check_type(argname="argument package_type", value=package_type, expected_type=type_hints["package_type"])
            check_type(argname="argument recursive_loop", value=recursive_loop, expected_type=type_hints["recursive_loop"])
            check_type(argname="argument reserved_concurrent_executions", value=reserved_concurrent_executions, expected_type=type_hints["reserved_concurrent_executions"])
            check_type(argname="argument runtime", value=runtime, expected_type=type_hints["runtime"])
            check_type(argname="argument runtime_management_config", value=runtime_management_config, expected_type=type_hints["runtime_management_config"])
            check_type(argname="argument snap_start", value=snap_start, expected_type=type_hints["snap_start"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument tracing_config", value=tracing_config, expected_type=type_hints["tracing_config"])
            check_type(argname="argument vpc_config", value=vpc_config, expected_type=type_hints["vpc_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "code": code,
            "role": role,
        }
        if architectures is not None:
            self._values["architectures"] = architectures
        if code_signing_config_arn is not None:
            self._values["code_signing_config_arn"] = code_signing_config_arn
        if dead_letter_config is not None:
            self._values["dead_letter_config"] = dead_letter_config
        if description is not None:
            self._values["description"] = description
        if environment is not None:
            self._values["environment"] = environment
        if ephemeral_storage is not None:
            self._values["ephemeral_storage"] = ephemeral_storage
        if file_system_configs is not None:
            self._values["file_system_configs"] = file_system_configs
        if function_name is not None:
            self._values["function_name"] = function_name
        if handler is not None:
            self._values["handler"] = handler
        if image_config is not None:
            self._values["image_config"] = image_config
        if kms_key_arn is not None:
            self._values["kms_key_arn"] = kms_key_arn
        if layers is not None:
            self._values["layers"] = layers
        if logging_config is not None:
            self._values["logging_config"] = logging_config
        if memory_size is not None:
            self._values["memory_size"] = memory_size
        if package_type is not None:
            self._values["package_type"] = package_type
        if recursive_loop is not None:
            self._values["recursive_loop"] = recursive_loop
        if reserved_concurrent_executions is not None:
            self._values["reserved_concurrent_executions"] = reserved_concurrent_executions
        if runtime is not None:
            self._values["runtime"] = runtime
        if runtime_management_config is not None:
            self._values["runtime_management_config"] = runtime_management_config
        if snap_start is not None:
            self._values["snap_start"] = snap_start
        if tags is not None:
            self._values["tags"] = tags
        if timeout is not None:
            self._values["timeout"] = timeout
        if tracing_config is not None:
            self._values["tracing_config"] = tracing_config
        if vpc_config is not None:
            self._values["vpc_config"] = vpc_config

    @builtins.property
    def code(self) -> typing.Union[_IResolvable_da3f097b, CfnFunction.CodeProperty]:
        '''The code for the function. You can define your function code in multiple ways:.

        - For .zip deployment packages, you can specify the Amazon S3 location of the .zip file in the ``S3Bucket`` , ``S3Key`` , and ``S3ObjectVersion`` properties.
        - For .zip deployment packages, you can alternatively define the function code inline in the ``ZipFile`` property. This method works only for Node.js and Python functions.
        - For container images, specify the URI of your container image in the Amazon ECR registry in the ``ImageUri`` property.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-code
        '''
        result = self._values.get("code")
        assert result is not None, "Required property 'code' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnFunction.CodeProperty], result)

    @builtins.property
    def role(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the function's execution role.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-role
        '''
        result = self._values.get("role")
        assert result is not None, "Required property 'role' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def architectures(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The instruction set architecture that the function supports.

        Enter a string array with one of the valid values (arm64 or x86_64). The default value is ``x86_64`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-architectures
        '''
        result = self._values.get("architectures")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def code_signing_config_arn(self) -> typing.Optional[builtins.str]:
        '''To enable code signing for this function, specify the ARN of a code-signing configuration.

        A code-signing configuration
        includes a set of signing profiles, which define the trusted publishers for this function.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-codesigningconfigarn
        '''
        result = self._values.get("code_signing_config_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def dead_letter_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.DeadLetterConfigProperty]]:
        '''A dead-letter queue configuration that specifies the queue or topic where Lambda sends asynchronous events when they fail processing.

        For more information, see `Dead-letter queues <https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-dlq>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-deadletterconfig
        '''
        result = self._values.get("dead_letter_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.DeadLetterConfigProperty]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the function.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.EnvironmentProperty]]:
        '''Environment variables that are accessible from function code during execution.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-environment
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.EnvironmentProperty]], result)

    @builtins.property
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.EphemeralStorageProperty]]:
        '''The size of the function's ``/tmp`` directory in MB.

        The default value is 512, but it can be any whole number between 512 and 10,240 MB.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-ephemeralstorage
        '''
        result = self._values.get("ephemeral_storage")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.EphemeralStorageProperty]], result)

    @builtins.property
    def file_system_configs(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnFunction.FileSystemConfigProperty]]]]:
        '''Connection settings for an Amazon EFS file system.

        To connect a function to a file system, a mount target must be available in every Availability Zone that your function connects to. If your template contains an `AWS::EFS::MountTarget <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-mounttarget.html>`_ resource, you must also specify a ``DependsOn`` attribute to ensure that the mount target is created or updated before the function.

        For more information about using the ``DependsOn`` attribute, see `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-filesystemconfigs
        '''
        result = self._values.get("file_system_configs")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnFunction.FileSystemConfigProperty]]]], result)

    @builtins.property
    def function_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Lambda function, up to 64 characters in length.

        If you don't specify a name, AWS CloudFormation generates one.

        If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-functionname
        '''
        result = self._values.get("function_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def handler(self) -> typing.Optional[builtins.str]:
        '''The name of the method within your code that Lambda calls to run your function.

        Handler is required if the deployment package is a .zip file archive. The format includes the file name. It can also include namespaces and other qualifiers, depending on the runtime. For more information, see `Lambda programming model <https://docs.aws.amazon.com/lambda/latest/dg/foundation-progmodel.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-handler
        '''
        result = self._values.get("handler")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def image_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.ImageConfigProperty]]:
        '''Configuration values that override the container image Dockerfile settings.

        For more information, see `Container image settings <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-parms>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-imageconfig
        '''
        result = self._values.get("image_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.ImageConfigProperty]], result)

    @builtins.property
    def kms_key_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the AWS Key Management Service ( AWS KMS ) customer managed key that's used to encrypt the following resources:.

        - The function's `environment variables <https://docs.aws.amazon.com/lambda/latest/dg/configuration-envvars.html#configuration-envvars-encryption>`_ .
        - The function's `Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart-security.html>`_ snapshots.
        - When used with ``SourceKMSKeyArn`` , the unzipped version of the .zip deployment package that's used for function invocations. For more information, see `Specifying a customer managed key for Lambda <https://docs.aws.amazon.com/lambda/latest/dg/encrypt-zip-package.html#enable-zip-custom-encryption>`_ .
        - The optimized version of the container image that's used for function invocations. Note that this is not the same key that's used to protect your container image in the Amazon Elastic Container Registry (Amazon ECR). For more information, see `Function lifecycle <https://docs.aws.amazon.com/lambda/latest/dg/images-create.html#images-lifecycle>`_ .

        If you don't provide a customer managed key, Lambda uses an `AWS owned key <https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-owned-cmk>`_ or an `AWS managed key <https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-managed-cmk>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-kmskeyarn
        '''
        result = self._values.get("kms_key_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def layers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of `function layers <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ to add to the function's execution environment. Specify each layer by its ARN, including the version.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-layers
        '''
        result = self._values.get("layers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def logging_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.LoggingConfigProperty]]:
        '''The function's Amazon CloudWatch Logs configuration settings.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-loggingconfig
        '''
        result = self._values.get("logging_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.LoggingConfigProperty]], result)

    @builtins.property
    def memory_size(self) -> typing.Optional[jsii.Number]:
        '''The amount of `memory available to the function <https://docs.aws.amazon.com/lambda/latest/dg/configuration-function-common.html#configuration-memory-console>`_ at runtime. Increasing the function memory also increases its CPU allocation. The default value is 128 MB. The value can be any multiple of 1 MB. Note that new AWS accounts have reduced concurrency and memory quotas. AWS raises these quotas automatically based on your usage. You can also request a quota increase.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-memorysize
        '''
        result = self._values.get("memory_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def package_type(self) -> typing.Optional[builtins.str]:
        '''The type of deployment package.

        Set to ``Image`` for container image and set ``Zip`` for .zip file archive.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-packagetype
        '''
        result = self._values.get("package_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def recursive_loop(self) -> typing.Optional[builtins.str]:
        '''The status of your function's recursive loop detection configuration.

        When this value is set to ``Allow`` and Lambda detects your function being invoked as part of a recursive loop, it doesn't take any action.

        When this value is set to ``Terminate`` and Lambda detects your function being invoked as part of a recursive loop, it stops your function being invoked and notifies you.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-recursiveloop
        '''
        result = self._values.get("recursive_loop")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def reserved_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''The number of simultaneous executions to reserve for the function.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-reservedconcurrentexecutions
        '''
        result = self._values.get("reserved_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def runtime(self) -> typing.Optional[builtins.str]:
        '''The identifier of the function's `runtime <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Runtime is required if the deployment package is a .zip file archive. Specifying a runtime results in an error if you're deploying a function using a container image.

        The following list includes deprecated runtimes. Lambda blocks creating new functions and updating existing functions shortly after each runtime is deprecated. For more information, see `Runtime use after deprecation <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html#runtime-deprecation-levels>`_ .

        For a list of all currently supported runtimes, see `Supported runtimes <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html#runtimes-supported>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-runtime
        '''
        result = self._values.get("runtime")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def runtime_management_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.RuntimeManagementConfigProperty]]:
        '''Sets the runtime management configuration for a function's version.

        For more information, see `Runtime updates <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-update.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-runtimemanagementconfig
        '''
        result = self._values.get("runtime_management_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.RuntimeManagementConfigProperty]], result)

    @builtins.property
    def snap_start(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.SnapStartProperty]]:
        '''The function's `AWS Lambda SnapStart <https://docs.aws.amazon.com/lambda/latest/dg/snapstart.html>`_ setting.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-snapstart
        '''
        result = self._values.get("snap_start")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.SnapStartProperty]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''A list of `tags <https://docs.aws.amazon.com/lambda/latest/dg/tagging.html>`_ to apply to the function.

        .. epigraph::

           You must have the ``lambda:TagResource`` , ``lambda:UntagResource`` , and ``lambda:ListTags`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[jsii.Number]:
        '''The amount of time (in seconds) that Lambda allows a function to run before stopping it.

        The default is 3 seconds. The maximum allowed value is 900 seconds. For more information, see `Lambda execution environment <https://docs.aws.amazon.com/lambda/latest/dg/runtimes-context.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-timeout
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tracing_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.TracingConfigProperty]]:
        '''Set ``Mode`` to ``Active`` to sample and trace a subset of incoming requests with `X-Ray <https://docs.aws.amazon.com/lambda/latest/dg/services-xray.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-tracingconfig
        '''
        result = self._values.get("tracing_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.TracingConfigProperty]], result)

    @builtins.property
    def vpc_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.VpcConfigProperty]]:
        '''For network connectivity to AWS resources in a VPC, specify a list of security groups and subnets in the VPC.

        When you connect a function to a VPC, it can access resources and the internet only through that VPC. For more information, see `Configuring a Lambda function to access resources in a VPC <https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-function.html#cfn-lambda-function-vpcconfig
        '''
        result = self._values.get("vpc_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFunction.VpcConfigProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnFunctionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnLayerVersion(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.CfnLayerVersion",
):
    '''The ``AWS::Lambda::LayerVersion`` resource creates a `Lambda layer <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ from a ZIP archive.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html
    :cloudformationResource: AWS::Lambda::LayerVersion
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        cfn_layer_version = lambda_.CfnLayerVersion(self, "MyCfnLayerVersion",
            content=lambda.CfnLayerVersion.ContentProperty(
                s3_bucket="s3Bucket",
                s3_key="s3Key",
        
                # the properties below are optional
                s3_object_version="s3ObjectVersion"
            ),
        
            # the properties below are optional
            compatible_architectures=["compatibleArchitectures"],
            compatible_runtimes=["compatibleRuntimes"],
            description="description",
            layer_name="layerName",
            license_info="licenseInfo"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        content: typing.Union[_IResolvable_da3f097b, typing.Union["CfnLayerVersion.ContentProperty", typing.Dict[builtins.str, typing.Any]]],
        compatible_architectures: typing.Optional[typing.Sequence[builtins.str]] = None,
        compatible_runtimes: typing.Optional[typing.Sequence[builtins.str]] = None,
        description: typing.Optional[builtins.str] = None,
        layer_name: typing.Optional[builtins.str] = None,
        license_info: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param content: The function layer archive.
        :param compatible_architectures: A list of compatible `instruction set architectures <https://docs.aws.amazon.com/lambda/latest/dg/foundation-arch.html>`_ .
        :param compatible_runtimes: A list of compatible `function runtimes <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Used for filtering with `ListLayers <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html>`_ and `ListLayerVersions <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html>`_ .
        :param description: The description of the version.
        :param layer_name: The name or Amazon Resource Name (ARN) of the layer.
        :param license_info: The layer's software license. It can be any of the following:. - An `SPDX license identifier <https://docs.aws.amazon.com/https://spdx.org/licenses/>`_ . For example, ``MIT`` . - The URL of a license hosted on the internet. For example, ``https://opensource.org/licenses/MIT`` . - The full text of the license.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__429b31c977f42a0ad4faddf9465b3e17e6ec6694dad3dbc572a06eef3ba7bb82)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnLayerVersionProps(
            content=content,
            compatible_architectures=compatible_architectures,
            compatible_runtimes=compatible_runtimes,
            description=description,
            layer_name=layer_name,
            license_info=license_info,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__14e5b636b604538f1d3b2380c9a062bf54dd58a7750f0114522f9032e57ec033)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ec62d0523f22b8c8b1e22d47cffef8679dd0951562552f4ace5fe7f1ae86e603)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrLayerVersionArn")
    def attr_layer_version_arn(self) -> builtins.str:
        '''The ARN of the layer version.

        :cloudformationAttribute: LayerVersionArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLayerVersionArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="content")
    def content(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnLayerVersion.ContentProperty"]:
        '''The function layer archive.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnLayerVersion.ContentProperty"], jsii.get(self, "content"))

    @content.setter
    def content(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnLayerVersion.ContentProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__16fdce2009fa44459d262b1473f3f14ba14cc73bd0ed3916d4bd0edcf3e43b4d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "content", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="compatibleArchitectures")
    def compatible_architectures(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of compatible `instruction set architectures <https://docs.aws.amazon.com/lambda/latest/dg/foundation-arch.html>`_ .'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "compatibleArchitectures"))

    @compatible_architectures.setter
    def compatible_architectures(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__08839a09e116d09d437639b549a9f46251026b97c49c02e63d22b976b1705dce)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "compatibleArchitectures", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="compatibleRuntimes")
    def compatible_runtimes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of compatible `function runtimes <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Used for filtering with `ListLayers <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html>`_ and `ListLayerVersions <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html>`_ .'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "compatibleRuntimes"))

    @compatible_runtimes.setter
    def compatible_runtimes(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__43473c5b3bfb35dd80a0a3acca8e7e0c5fc897741acf287f2ca8c7ea72a554f9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "compatibleRuntimes", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the version.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9de7bf47b62135fe00acab97e2e852614fdc2347e66707de5b86a002b25e1379)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="layerName")
    def layer_name(self) -> typing.Optional[builtins.str]:
        '''The name or Amazon Resource Name (ARN) of the layer.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "layerName"))

    @layer_name.setter
    def layer_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e993c2ac2a3c971df686f5d3e84116d1aa3ddc96939d07988c83cd8183f40191)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "layerName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="licenseInfo")
    def license_info(self) -> typing.Optional[builtins.str]:
        '''The layer's software license.

        It can be any of the following:.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "licenseInfo"))

    @license_info.setter
    def license_info(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__901b954aebd75eb5e434b423ce636efcd7b6175bcb5075a916935082cf58517d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "licenseInfo", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnLayerVersion.ContentProperty",
        jsii_struct_bases=[],
        name_mapping={
            "s3_bucket": "s3Bucket",
            "s3_key": "s3Key",
            "s3_object_version": "s3ObjectVersion",
        },
    )
    class ContentProperty:
        def __init__(
            self,
            *,
            s3_bucket: builtins.str,
            s3_key: builtins.str,
            s3_object_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A ZIP archive that contains the contents of an `Lambda layer <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ .

            :param s3_bucket: The Amazon S3 bucket of the layer archive.
            :param s3_key: The Amazon S3 key of the layer archive.
            :param s3_object_version: For versioned objects, the version of the layer archive object to use.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-layerversion-content.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                content_property = lambda.CfnLayerVersion.ContentProperty(
                    s3_bucket="s3Bucket",
                    s3_key="s3Key",
                
                    # the properties below are optional
                    s3_object_version="s3ObjectVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__33591b9aaf2c8e26d04a24fc6ce5bb8493831eb29a3c0a1fafded14881bc2eb1)
                check_type(argname="argument s3_bucket", value=s3_bucket, expected_type=type_hints["s3_bucket"])
                check_type(argname="argument s3_key", value=s3_key, expected_type=type_hints["s3_key"])
                check_type(argname="argument s3_object_version", value=s3_object_version, expected_type=type_hints["s3_object_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "s3_bucket": s3_bucket,
                "s3_key": s3_key,
            }
            if s3_object_version is not None:
                self._values["s3_object_version"] = s3_object_version

        @builtins.property
        def s3_bucket(self) -> builtins.str:
            '''The Amazon S3 bucket of the layer archive.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-layerversion-content.html#cfn-lambda-layerversion-content-s3bucket
            '''
            result = self._values.get("s3_bucket")
            assert result is not None, "Required property 's3_bucket' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def s3_key(self) -> builtins.str:
            '''The Amazon S3 key of the layer archive.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-layerversion-content.html#cfn-lambda-layerversion-content-s3key
            '''
            result = self._values.get("s3_key")
            assert result is not None, "Required property 's3_key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def s3_object_version(self) -> typing.Optional[builtins.str]:
            '''For versioned objects, the version of the layer archive object to use.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-layerversion-content.html#cfn-lambda-layerversion-content-s3objectversion
            '''
            result = self._values.get("s3_object_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContentProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556)
class CfnLayerVersionPermission(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.CfnLayerVersionPermission",
):
    '''The ``AWS::Lambda::LayerVersionPermission`` resource adds permissions to the resource-based policy of a version of an `Lambda layer <https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html>`_ . Use this action to grant layer usage permission to other accounts. You can grant permission to a single account, all AWS accounts, or all accounts in an organization.

    .. epigraph::

       Since the release of the `UpdateReplacePolicy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatereplacepolicy.html>`_ both ``UpdateReplacePolicy`` and ``DeletionPolicy`` are required to protect your Resources/LayerPermissions from deletion.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html
    :cloudformationResource: AWS::Lambda::LayerVersionPermission
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        cfn_layer_version_permission = lambda_.CfnLayerVersionPermission(self, "MyCfnLayerVersionPermission",
            action="action",
            layer_version_arn="layerVersionArn",
            principal="principal",
        
            # the properties below are optional
            organization_id="organizationId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        layer_version_arn: builtins.str,
        principal: builtins.str,
        organization_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param action: The API action that grants access to the layer. For example, ``lambda:GetLayerVersion`` .
        :param layer_version_arn: The name or Amazon Resource Name (ARN) of the layer.
        :param principal: An account ID, or ``*`` to grant layer usage permission to all accounts in an organization, or all AWS accounts (if ``organizationId`` is not specified). For the last case, make sure that you really do want all AWS accounts to have usage permission to this layer.
        :param organization_id: With the principal set to ``*`` , grant permission to all accounts in the specified organization.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a34577503cda5332d6d532dd850e19bd6607e7fdf611f9085d56969de31d2f8f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnLayerVersionPermissionProps(
            action=action,
            layer_version_arn=layer_version_arn,
            principal=principal,
            organization_id=organization_id,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1a296d6424b3fb5033365005032559d267c3b3f754c7bd9c6640c41b4437e873)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a4799c31a8e027dd6a988b6a4c393f9817fe36efa60fd72c1dd655dc2a7166a4)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''ID generated by service.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        '''The API action that grants access to the layer.'''
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @action.setter
    def action(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a8acb7cf08f86ac198f71bbdc5d1b990c075e39f8b0af699b561835f6d90d323)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "action", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="layerVersionArn")
    def layer_version_arn(self) -> builtins.str:
        '''The name or Amazon Resource Name (ARN) of the layer.'''
        return typing.cast(builtins.str, jsii.get(self, "layerVersionArn"))

    @layer_version_arn.setter
    def layer_version_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__06dce1e31e6399ecebdda1a7b070d5eb8239cbb3bb5ad1b06b9e4ba565584021)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "layerVersionArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="principal")
    def principal(self) -> builtins.str:
        '''An account ID, or ``*`` to grant layer usage permission to all accounts in an organization, or all AWS accounts (if ``organizationId`` is not specified).'''
        return typing.cast(builtins.str, jsii.get(self, "principal"))

    @principal.setter
    def principal(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c9a1279a904647e3b5f593df647c106c101d17fc942ffea4b4c867787632e62d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "principal", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="organizationId")
    def organization_id(self) -> typing.Optional[builtins.str]:
        '''With the principal set to ``*`` , grant permission to all accounts in the specified organization.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "organizationId"))

    @organization_id.setter
    def organization_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__45c67e7ddd4fde4ad3243685942dee1cdc90213c5a305f4ccea252422bd56b9d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "organizationId", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CfnLayerVersionPermissionProps",
    jsii_struct_bases=[],
    name_mapping={
        "action": "action",
        "layer_version_arn": "layerVersionArn",
        "principal": "principal",
        "organization_id": "organizationId",
    },
)
class CfnLayerVersionPermissionProps:
    def __init__(
        self,
        *,
        action: builtins.str,
        layer_version_arn: builtins.str,
        principal: builtins.str,
        organization_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnLayerVersionPermission``.

        :param action: The API action that grants access to the layer. For example, ``lambda:GetLayerVersion`` .
        :param layer_version_arn: The name or Amazon Resource Name (ARN) of the layer.
        :param principal: An account ID, or ``*`` to grant layer usage permission to all accounts in an organization, or all AWS accounts (if ``organizationId`` is not specified). For the last case, make sure that you really do want all AWS accounts to have usage permission to this layer.
        :param organization_id: With the principal set to ``*`` , grant permission to all accounts in the specified organization.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            cfn_layer_version_permission_props = lambda.CfnLayerVersionPermissionProps(
                action="action",
                layer_version_arn="layerVersionArn",
                principal="principal",
            
                # the properties below are optional
                organization_id="organizationId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__963d0e6aa91417f1b389ca86ebb0939b6f8784c8421540668eb489f936cf4428)
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument layer_version_arn", value=layer_version_arn, expected_type=type_hints["layer_version_arn"])
            check_type(argname="argument principal", value=principal, expected_type=type_hints["principal"])
            check_type(argname="argument organization_id", value=organization_id, expected_type=type_hints["organization_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "action": action,
            "layer_version_arn": layer_version_arn,
            "principal": principal,
        }
        if organization_id is not None:
            self._values["organization_id"] = organization_id

    @builtins.property
    def action(self) -> builtins.str:
        '''The API action that grants access to the layer.

        For example, ``lambda:GetLayerVersion`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-action
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def layer_version_arn(self) -> builtins.str:
        '''The name or Amazon Resource Name (ARN) of the layer.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-layerversionarn
        '''
        result = self._values.get("layer_version_arn")
        assert result is not None, "Required property 'layer_version_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def principal(self) -> builtins.str:
        '''An account ID, or ``*`` to grant layer usage permission to all accounts in an organization, or all AWS accounts (if ``organizationId`` is not specified).

        For the last case, make sure that you really do want all AWS accounts to have usage permission to this layer.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-principal
        '''
        result = self._values.get("principal")
        assert result is not None, "Required property 'principal' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def organization_id(self) -> typing.Optional[builtins.str]:
        '''With the principal set to ``*`` , grant permission to all accounts in the specified organization.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversionpermission.html#cfn-lambda-layerversionpermission-organizationid
        '''
        result = self._values.get("organization_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnLayerVersionPermissionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CfnLayerVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "content": "content",
        "compatible_architectures": "compatibleArchitectures",
        "compatible_runtimes": "compatibleRuntimes",
        "description": "description",
        "layer_name": "layerName",
        "license_info": "licenseInfo",
    },
)
class CfnLayerVersionProps:
    def __init__(
        self,
        *,
        content: typing.Union[_IResolvable_da3f097b, typing.Union[CfnLayerVersion.ContentProperty, typing.Dict[builtins.str, typing.Any]]],
        compatible_architectures: typing.Optional[typing.Sequence[builtins.str]] = None,
        compatible_runtimes: typing.Optional[typing.Sequence[builtins.str]] = None,
        description: typing.Optional[builtins.str] = None,
        layer_name: typing.Optional[builtins.str] = None,
        license_info: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnLayerVersion``.

        :param content: The function layer archive.
        :param compatible_architectures: A list of compatible `instruction set architectures <https://docs.aws.amazon.com/lambda/latest/dg/foundation-arch.html>`_ .
        :param compatible_runtimes: A list of compatible `function runtimes <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Used for filtering with `ListLayers <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html>`_ and `ListLayerVersions <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html>`_ .
        :param description: The description of the version.
        :param layer_name: The name or Amazon Resource Name (ARN) of the layer.
        :param license_info: The layer's software license. It can be any of the following:. - An `SPDX license identifier <https://docs.aws.amazon.com/https://spdx.org/licenses/>`_ . For example, ``MIT`` . - The URL of a license hosted on the internet. For example, ``https://opensource.org/licenses/MIT`` . - The full text of the license.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            cfn_layer_version_props = lambda.CfnLayerVersionProps(
                content=lambda.CfnLayerVersion.ContentProperty(
                    s3_bucket="s3Bucket",
                    s3_key="s3Key",
            
                    # the properties below are optional
                    s3_object_version="s3ObjectVersion"
                ),
            
                # the properties below are optional
                compatible_architectures=["compatibleArchitectures"],
                compatible_runtimes=["compatibleRuntimes"],
                description="description",
                layer_name="layerName",
                license_info="licenseInfo"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d1c808c735498f09b3f095efa70316f156f6c9dea123c1056d187add7104c0b9)
            check_type(argname="argument content", value=content, expected_type=type_hints["content"])
            check_type(argname="argument compatible_architectures", value=compatible_architectures, expected_type=type_hints["compatible_architectures"])
            check_type(argname="argument compatible_runtimes", value=compatible_runtimes, expected_type=type_hints["compatible_runtimes"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument layer_name", value=layer_name, expected_type=type_hints["layer_name"])
            check_type(argname="argument license_info", value=license_info, expected_type=type_hints["license_info"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "content": content,
        }
        if compatible_architectures is not None:
            self._values["compatible_architectures"] = compatible_architectures
        if compatible_runtimes is not None:
            self._values["compatible_runtimes"] = compatible_runtimes
        if description is not None:
            self._values["description"] = description
        if layer_name is not None:
            self._values["layer_name"] = layer_name
        if license_info is not None:
            self._values["license_info"] = license_info

    @builtins.property
    def content(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnLayerVersion.ContentProperty]:
        '''The function layer archive.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-content
        '''
        result = self._values.get("content")
        assert result is not None, "Required property 'content' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnLayerVersion.ContentProperty], result)

    @builtins.property
    def compatible_architectures(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of compatible `instruction set architectures <https://docs.aws.amazon.com/lambda/latest/dg/foundation-arch.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-compatiblearchitectures
        '''
        result = self._values.get("compatible_architectures")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def compatible_runtimes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of compatible `function runtimes <https://docs.aws.amazon.com/lambda/latest/dg/lambda-runtimes.html>`_ . Used for filtering with `ListLayers <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayers.html>`_ and `ListLayerVersions <https://docs.aws.amazon.com/lambda/latest/dg/API_ListLayerVersions.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-compatibleruntimes
        '''
        result = self._values.get("compatible_runtimes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the version.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def layer_name(self) -> typing.Optional[builtins.str]:
        '''The name or Amazon Resource Name (ARN) of the layer.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-layername
        '''
        result = self._values.get("layer_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def license_info(self) -> typing.Optional[builtins.str]:
        '''The layer's software license. It can be any of the following:.

        - An `SPDX license identifier <https://docs.aws.amazon.com/https://spdx.org/licenses/>`_ . For example, ``MIT`` .
        - The URL of a license hosted on the internet. For example, ``https://opensource.org/licenses/MIT`` .
        - The full text of the license.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-layerversion.html#cfn-lambda-layerversion-licenseinfo
        '''
        result = self._values.get("license_info")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnLayerVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CfnParametersCodeProps",
    jsii_struct_bases=[],
    name_mapping={
        "bucket_name_param": "bucketNameParam",
        "object_key_param": "objectKeyParam",
        "source_kms_key": "sourceKMSKey",
    },
)
class CfnParametersCodeProps:
    def __init__(
        self,
        *,
        bucket_name_param: typing.Optional[_CfnParameter_48fc1866] = None,
        object_key_param: typing.Optional[_CfnParameter_48fc1866] = None,
        source_kms_key: typing.Optional[_IKey_5f11635f] = None,
    ) -> None:
        '''Construction properties for ``CfnParametersCode``.

        :param bucket_name_param: The CloudFormation parameter that represents the name of the S3 Bucket where the Lambda code will be located in. Must be of type 'String'. Default: a new parameter will be created
        :param object_key_param: The CloudFormation parameter that represents the path inside the S3 Bucket where the Lambda code will be located at. Must be of type 'String'. Default: a new parameter will be created
        :param source_kms_key: The ARN of the KMS key used to encrypt the handler code. Default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_kms as kms
            from aws_cdk import aws_lambda as lambda_
            
            # cfn_parameter: cdk.CfnParameter
            # key: kms.Key
            
            cfn_parameters_code_props = lambda.CfnParametersCodeProps(
                bucket_name_param=cfn_parameter,
                object_key_param=cfn_parameter,
                source_kMSKey=key
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__45ce02257c922b893446f407552a2416c3356585f4b95a19a9069a0bb7e9115f)
            check_type(argname="argument bucket_name_param", value=bucket_name_param, expected_type=type_hints["bucket_name_param"])
            check_type(argname="argument object_key_param", value=object_key_param, expected_type=type_hints["object_key_param"])
            check_type(argname="argument source_kms_key", value=source_kms_key, expected_type=type_hints["source_kms_key"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if bucket_name_param is not None:
            self._values["bucket_name_param"] = bucket_name_param
        if object_key_param is not None:
            self._values["object_key_param"] = object_key_param
        if source_kms_key is not None:
            self._values["source_kms_key"] = source_kms_key

    @builtins.property
    def bucket_name_param(self) -> typing.Optional[_CfnParameter_48fc1866]:
        '''The CloudFormation parameter that represents the name of the S3 Bucket where the Lambda code will be located in.

        Must be of type 'String'.

        :default: a new parameter will be created
        '''
        result = self._values.get("bucket_name_param")
        return typing.cast(typing.Optional[_CfnParameter_48fc1866], result)

    @builtins.property
    def object_key_param(self) -> typing.Optional[_CfnParameter_48fc1866]:
        '''The CloudFormation parameter that represents the path inside the S3 Bucket where the Lambda code will be located at.

        Must be of type 'String'.

        :default: a new parameter will be created
        '''
        result = self._values.get("object_key_param")
        return typing.cast(typing.Optional[_CfnParameter_48fc1866], result)

    @builtins.property
    def source_kms_key(self) -> typing.Optional[_IKey_5f11635f]:
        '''The ARN of the KMS key used to encrypt the handler code.

        :default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.
        '''
        result = self._values.get("source_kms_key")
        return typing.cast(typing.Optional[_IKey_5f11635f], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnParametersCodeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnPermission(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.CfnPermission",
):
    '''The ``AWS::Lambda::Permission`` resource grants an AWS service or another account permission to use a function.

    You can apply the policy at the function level, or specify a qualifier to restrict access to a single version or alias. If you use a qualifier, the invoker must use the full Amazon Resource Name (ARN) of that version or alias to invoke the function.

    To grant permission to another account, specify the account ID as the ``Principal`` . To grant permission to an organization defined in AWS Organizations , specify the organization ID as the ``PrincipalOrgID`` . For AWS services, the principal is a domain-style identifier defined by the service, like ``s3.amazonaws.com`` or ``sns.amazonaws.com`` . For AWS services, you can also specify the ARN of the associated resource as the ``SourceArn`` . If you grant permission to a service principal without specifying the source, other accounts could potentially configure resources in their account to invoke your Lambda function.

    If your function has a function URL, you can specify the ``FunctionUrlAuthType`` parameter. This adds a condition to your permission that only applies when your function URL's ``AuthType`` matches the specified ``FunctionUrlAuthType`` . For more information about the ``AuthType`` parameter, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .

    This resource adds a statement to a resource-based permission policy for the function. For more information about function policies, see `Lambda Function Policies <https://docs.aws.amazon.com/lambda/latest/dg/access-control-resource-based.html>`_ .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html
    :cloudformationResource: AWS::Lambda::Permission
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        cfn_permission = lambda_.CfnPermission(self, "MyCfnPermission",
            action="action",
            function_name="functionName",
            principal="principal",
        
            # the properties below are optional
            event_source_token="eventSourceToken",
            function_url_auth_type="functionUrlAuthType",
            principal_org_id="principalOrgId",
            source_account="sourceAccount",
            source_arn="sourceArn"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        function_name: builtins.str,
        principal: builtins.str,
        event_source_token: typing.Optional[builtins.str] = None,
        function_url_auth_type: typing.Optional[builtins.str] = None,
        principal_org_id: typing.Optional[builtins.str] = None,
        source_account: typing.Optional[builtins.str] = None,
        source_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param action: The action that the principal can use on the function. For example, ``lambda:InvokeFunction`` or ``lambda:GetFunction`` .
        :param function_name: The name or ARN of the Lambda function, version, or alias. **Name formats** - *Function name* – ``my-function`` (name-only), ``my-function:v1`` (with alias). - *Function ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:my-function`` . - *Partial ARN* – ``123456789012:function:my-function`` . You can append a version number or alias to any of the formats. The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param principal: The AWS service , AWS account , IAM user, or IAM role that invokes the function. If you specify a service, use ``SourceArn`` or ``SourceAccount`` to limit who can invoke the function through that service.
        :param event_source_token: For Alexa Smart Home functions, a token that the invoker must supply.
        :param function_url_auth_type: The type of authentication that your function URL uses. Set to ``AWS_IAM`` if you want to restrict access to authenticated users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .
        :param principal_org_id: The identifier for your organization in AWS Organizations . Use this to grant permissions to all the AWS accounts under this organization.
        :param source_account: For AWS service , the ID of the AWS account that owns the resource. Use this together with ``SourceArn`` to ensure that the specified account owns the resource. It is possible for an Amazon S3 bucket to be deleted by its owner and recreated by another account.
        :param source_arn: For AWS services , the ARN of the AWS resource that invokes the function. For example, an Amazon S3 bucket or Amazon SNS topic. Note that Lambda configures the comparison using the ``StringLike`` operator.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c457a277b84dbba5bd94a2c0135335b8d7dbb3d409b1fa988b4f5a219def8833)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnPermissionProps(
            action=action,
            function_name=function_name,
            principal=principal,
            event_source_token=event_source_token,
            function_url_auth_type=function_url_auth_type,
            principal_org_id=principal_org_id,
            source_account=source_account,
            source_arn=source_arn,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9b5ba7ce4b7f278f54ce9df1c0cde9961d9509ae5738b07350bd88acec30b034)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42ec27677dc521210efe54fef5174d8d15cf01f05a5968a178a2821864726994)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        '''The action that the principal can use on the function.'''
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @action.setter
    def action(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__df115766595b459e6facc31968c7c21d3891662bf7de9b37367e71ea6d4114f0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "action", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name or ARN of the Lambda function, version, or alias.'''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6b78b627aafc24e5049b4f124fa9ed461ff5de5656d8eb50a73c2129a84e7dbb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="principal")
    def principal(self) -> builtins.str:
        '''The AWS service , AWS account , IAM user, or IAM role that invokes the function.'''
        return typing.cast(builtins.str, jsii.get(self, "principal"))

    @principal.setter
    def principal(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a8b1d5ebdfbdd1f6dd0ae27ea787a03534f8840c5afd7e97345329a87e64c9a3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "principal", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="eventSourceToken")
    def event_source_token(self) -> typing.Optional[builtins.str]:
        '''For Alexa Smart Home functions, a token that the invoker must supply.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "eventSourceToken"))

    @event_source_token.setter
    def event_source_token(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7584a173c7ddfb61695a8eaeb94df07bc5ed6da7dacb871ce3f736ac59ec2b62)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "eventSourceToken", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="functionUrlAuthType")
    def function_url_auth_type(self) -> typing.Optional[builtins.str]:
        '''The type of authentication that your function URL uses.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "functionUrlAuthType"))

    @function_url_auth_type.setter
    def function_url_auth_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c85fd263b0dc67595e83c426e4f8a064ea18d154ec013099a2ff44bb12a40473)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionUrlAuthType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="principalOrgId")
    def principal_org_id(self) -> typing.Optional[builtins.str]:
        '''The identifier for your organization in AWS Organizations .'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "principalOrgId"))

    @principal_org_id.setter
    def principal_org_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__15afcc0405054080f46df62ed10831279de0524bf06aeea4268901393a2625c3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "principalOrgId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="sourceAccount")
    def source_account(self) -> typing.Optional[builtins.str]:
        '''For AWS service , the ID of the AWS account that owns the resource.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceAccount"))

    @source_account.setter
    def source_account(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7daf8201bed0090388652951678b300be0c44422471d3bd64c0d0f26e4996f38)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceAccount", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="sourceArn")
    def source_arn(self) -> typing.Optional[builtins.str]:
        '''For AWS services , the ARN of the AWS resource that invokes the function.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceArn"))

    @source_arn.setter
    def source_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5e353afa8c93203f738da3be5d082cb1816a03108506ae44264327d28f36e502)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceArn", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CfnPermissionProps",
    jsii_struct_bases=[],
    name_mapping={
        "action": "action",
        "function_name": "functionName",
        "principal": "principal",
        "event_source_token": "eventSourceToken",
        "function_url_auth_type": "functionUrlAuthType",
        "principal_org_id": "principalOrgId",
        "source_account": "sourceAccount",
        "source_arn": "sourceArn",
    },
)
class CfnPermissionProps:
    def __init__(
        self,
        *,
        action: builtins.str,
        function_name: builtins.str,
        principal: builtins.str,
        event_source_token: typing.Optional[builtins.str] = None,
        function_url_auth_type: typing.Optional[builtins.str] = None,
        principal_org_id: typing.Optional[builtins.str] = None,
        source_account: typing.Optional[builtins.str] = None,
        source_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnPermission``.

        :param action: The action that the principal can use on the function. For example, ``lambda:InvokeFunction`` or ``lambda:GetFunction`` .
        :param function_name: The name or ARN of the Lambda function, version, or alias. **Name formats** - *Function name* – ``my-function`` (name-only), ``my-function:v1`` (with alias). - *Function ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:my-function`` . - *Partial ARN* – ``123456789012:function:my-function`` . You can append a version number or alias to any of the formats. The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param principal: The AWS service , AWS account , IAM user, or IAM role that invokes the function. If you specify a service, use ``SourceArn`` or ``SourceAccount`` to limit who can invoke the function through that service.
        :param event_source_token: For Alexa Smart Home functions, a token that the invoker must supply.
        :param function_url_auth_type: The type of authentication that your function URL uses. Set to ``AWS_IAM`` if you want to restrict access to authenticated users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .
        :param principal_org_id: The identifier for your organization in AWS Organizations . Use this to grant permissions to all the AWS accounts under this organization.
        :param source_account: For AWS service , the ID of the AWS account that owns the resource. Use this together with ``SourceArn`` to ensure that the specified account owns the resource. It is possible for an Amazon S3 bucket to be deleted by its owner and recreated by another account.
        :param source_arn: For AWS services , the ARN of the AWS resource that invokes the function. For example, an Amazon S3 bucket or Amazon SNS topic. Note that Lambda configures the comparison using the ``StringLike`` operator.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            cfn_permission_props = lambda.CfnPermissionProps(
                action="action",
                function_name="functionName",
                principal="principal",
            
                # the properties below are optional
                event_source_token="eventSourceToken",
                function_url_auth_type="functionUrlAuthType",
                principal_org_id="principalOrgId",
                source_account="sourceAccount",
                source_arn="sourceArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b0c90e5a512dc08c54978bc1f6bf13992ad2d1d5c793f2b05fc82eef380f8c66)
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument principal", value=principal, expected_type=type_hints["principal"])
            check_type(argname="argument event_source_token", value=event_source_token, expected_type=type_hints["event_source_token"])
            check_type(argname="argument function_url_auth_type", value=function_url_auth_type, expected_type=type_hints["function_url_auth_type"])
            check_type(argname="argument principal_org_id", value=principal_org_id, expected_type=type_hints["principal_org_id"])
            check_type(argname="argument source_account", value=source_account, expected_type=type_hints["source_account"])
            check_type(argname="argument source_arn", value=source_arn, expected_type=type_hints["source_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "action": action,
            "function_name": function_name,
            "principal": principal,
        }
        if event_source_token is not None:
            self._values["event_source_token"] = event_source_token
        if function_url_auth_type is not None:
            self._values["function_url_auth_type"] = function_url_auth_type
        if principal_org_id is not None:
            self._values["principal_org_id"] = principal_org_id
        if source_account is not None:
            self._values["source_account"] = source_account
        if source_arn is not None:
            self._values["source_arn"] = source_arn

    @builtins.property
    def action(self) -> builtins.str:
        '''The action that the principal can use on the function.

        For example, ``lambda:InvokeFunction`` or ``lambda:GetFunction`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-action
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The name or ARN of the Lambda function, version, or alias.

        **Name formats** - *Function name* – ``my-function`` (name-only), ``my-function:v1`` (with alias).

        - *Function ARN* – ``arn:aws:lambda:us-west-2:123456789012:function:my-function`` .
        - *Partial ARN* – ``123456789012:function:my-function`` .

        You can append a version number or alias to any of the formats. The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-functionname
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def principal(self) -> builtins.str:
        '''The AWS service , AWS account , IAM user, or IAM role that invokes the function.

        If you specify a service, use ``SourceArn`` or ``SourceAccount`` to limit who can invoke the function through that service.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-principal
        '''
        result = self._values.get("principal")
        assert result is not None, "Required property 'principal' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def event_source_token(self) -> typing.Optional[builtins.str]:
        '''For Alexa Smart Home functions, a token that the invoker must supply.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-eventsourcetoken
        '''
        result = self._values.get("event_source_token")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def function_url_auth_type(self) -> typing.Optional[builtins.str]:
        '''The type of authentication that your function URL uses.

        Set to ``AWS_IAM`` if you want to restrict access to authenticated users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-functionurlauthtype
        '''
        result = self._values.get("function_url_auth_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def principal_org_id(self) -> typing.Optional[builtins.str]:
        '''The identifier for your organization in AWS Organizations .

        Use this to grant permissions to all the AWS accounts under this organization.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-principalorgid
        '''
        result = self._values.get("principal_org_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_account(self) -> typing.Optional[builtins.str]:
        '''For AWS service , the ID of the AWS account that owns the resource.

        Use this together with ``SourceArn`` to ensure that the specified account owns the resource. It is possible for an Amazon S3 bucket to be deleted by its owner and recreated by another account.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-sourceaccount
        '''
        result = self._values.get("source_account")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_arn(self) -> typing.Optional[builtins.str]:
        '''For AWS services , the ARN of the AWS resource that invokes the function.

        For example, an Amazon S3 bucket or Amazon SNS topic.

        Note that Lambda configures the comparison using the ``StringLike`` operator.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-permission.html#cfn-lambda-permission-sourcearn
        '''
        result = self._values.get("source_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnPermissionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnUrl(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.CfnUrl",
):
    '''The ``AWS::Lambda::Url`` resource creates a function URL with the specified configuration parameters.

    A `function URL <https://docs.aws.amazon.com/lambda/latest/dg/lambda-urls.html>`_ is a dedicated HTTP(S) endpoint that you can use to invoke your function.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html
    :cloudformationResource: AWS::Lambda::Url
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        cfn_url = lambda_.CfnUrl(self, "MyCfnUrl",
            auth_type="authType",
            target_function_arn="targetFunctionArn",
        
            # the properties below are optional
            cors=lambda.CfnUrl.CorsProperty(
                allow_credentials=False,
                allow_headers=["allowHeaders"],
                allow_methods=["allowMethods"],
                allow_origins=["allowOrigins"],
                expose_headers=["exposeHeaders"],
                max_age=123
            ),
            invoke_mode="invokeMode",
            qualifier="qualifier"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        auth_type: builtins.str,
        target_function_arn: builtins.str,
        cors: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUrl.CorsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        invoke_mode: typing.Optional[builtins.str] = None,
        qualifier: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param auth_type: The type of authentication that your function URL uses. Set to ``AWS_IAM`` if you want to restrict access to authenticated users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .
        :param target_function_arn: The name of the Lambda function. **Name formats** - *Function name* - ``my-function`` . - *Function ARN* - ``lambda: : :function:my-function`` . - *Partial ARN* - ``:function:my-function`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param cors: The `Cross-Origin Resource Sharing (CORS) <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS>`_ settings for your function URL.
        :param invoke_mode: Use one of the following options:. - ``BUFFERED`` – This is the default option. Lambda invokes your function using the ``Invoke`` API operation. Invocation results are available when the payload is complete. The maximum payload size is 6 MB. - ``RESPONSE_STREAM`` – Your function streams payload results as they become available. Lambda invokes your function using the ``InvokeWithResponseStream`` API operation. The maximum response payload size is 20 MB, however, you can `request a quota increase <https://docs.aws.amazon.com/servicequotas/latest/userguide/request-quota-increase.html>`_ .
        :param qualifier: The alias name.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__850b2a3a2e0bfd3ea79643487b3e93ff15d7e3bd7ad17ee73f8cd8e694754343)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUrlProps(
            auth_type=auth_type,
            target_function_arn=target_function_arn,
            cors=cors,
            invoke_mode=invoke_mode,
            qualifier=qualifier,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a06c94f0eef5ba959a451758db2ef1e6b6ede9418c7b6df63cfb027f8128d85f)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__225a6eb3f1fda248f2fc2bed0b65c19c2c4afbfa852413b2d3c884ca1b60eb56)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrFunctionArn")
    def attr_function_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the function.

        :cloudformationAttribute: FunctionArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFunctionArn"))

    @builtins.property
    @jsii.member(jsii_name="attrFunctionUrl")
    def attr_function_url(self) -> builtins.str:
        '''The HTTP URL endpoint for your function.

        :cloudformationAttribute: FunctionUrl
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFunctionUrl"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="authType")
    def auth_type(self) -> builtins.str:
        '''The type of authentication that your function URL uses.'''
        return typing.cast(builtins.str, jsii.get(self, "authType"))

    @auth_type.setter
    def auth_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3825ac9d2b066b8684c19c73c49481a98ec726da763f3c47d81eecf450d60f11)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="targetFunctionArn")
    def target_function_arn(self) -> builtins.str:
        '''The name of the Lambda function.'''
        return typing.cast(builtins.str, jsii.get(self, "targetFunctionArn"))

    @target_function_arn.setter
    def target_function_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5fedf78e7a48e0b177ef81d5fbc67f89d31b5c9d7578d78803f563dcaf2c39a6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetFunctionArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="cors")
    def cors(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUrl.CorsProperty"]]:
        '''The `Cross-Origin Resource Sharing (CORS) <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS>`_ settings for your function URL.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUrl.CorsProperty"]], jsii.get(self, "cors"))

    @cors.setter
    def cors(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUrl.CorsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d41b12f6e3c274633a1ab237c9d2c8dae2757548abbc8e5bde3fe4176ff93b35)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cors", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="invokeMode")
    def invoke_mode(self) -> typing.Optional[builtins.str]:
        '''Use one of the following options:.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "invokeMode"))

    @invoke_mode.setter
    def invoke_mode(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__de96f9d091517b1e5b2c30ee19b262b1bc639d57f811b38dc952094468738887)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "invokeMode", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="qualifier")
    def qualifier(self) -> typing.Optional[builtins.str]:
        '''The alias name.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "qualifier"))

    @qualifier.setter
    def qualifier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__be10204dad3e3ae396b62dbe6a430db367f44e796f51c5039cef395b94e2e908)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "qualifier", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnUrl.CorsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "allow_credentials": "allowCredentials",
            "allow_headers": "allowHeaders",
            "allow_methods": "allowMethods",
            "allow_origins": "allowOrigins",
            "expose_headers": "exposeHeaders",
            "max_age": "maxAge",
        },
    )
    class CorsProperty:
        def __init__(
            self,
            *,
            allow_credentials: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            allow_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
            allow_methods: typing.Optional[typing.Sequence[builtins.str]] = None,
            allow_origins: typing.Optional[typing.Sequence[builtins.str]] = None,
            expose_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
            max_age: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The `Cross-Origin Resource Sharing (CORS) <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS>`_ settings for your function URL. Use CORS to grant access to your function URL from any origin. You can also use CORS to control access for specific HTTP headers and methods in requests to your function URL.

            :param allow_credentials: Whether you want to allow cookies or other credentials in requests to your function URL. The default is ``false`` .
            :param allow_headers: The HTTP headers that origins can include in requests to your function URL. For example: ``Date`` , ``Keep-Alive`` , ``X-Custom-Header`` .
            :param allow_methods: The HTTP methods that are allowed when calling your function URL. For example: ``GET`` , ``POST`` , ``DELETE`` , or the wildcard character ( ``*`` ).
            :param allow_origins: The origins that can access your function URL. You can list any number of specific origins, separated by a comma. For example: ``https://www.example.com`` , ``http://localhost:60905`` . Alternatively, you can grant access to all origins with the wildcard character ( ``*`` ).
            :param expose_headers: The HTTP headers in your function response that you want to expose to origins that call your function URL. For example: ``Date`` , ``Keep-Alive`` , ``X-Custom-Header`` .
            :param max_age: The maximum amount of time, in seconds, that browsers can cache results of a preflight request. By default, this is set to ``0`` , which means the browser will not cache results.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                cors_property = lambda.CfnUrl.CorsProperty(
                    allow_credentials=False,
                    allow_headers=["allowHeaders"],
                    allow_methods=["allowMethods"],
                    allow_origins=["allowOrigins"],
                    expose_headers=["exposeHeaders"],
                    max_age=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3069061575a0c1c37c172de2f11fad3cb36fe8a659e96295949f0ba09e4426ba)
                check_type(argname="argument allow_credentials", value=allow_credentials, expected_type=type_hints["allow_credentials"])
                check_type(argname="argument allow_headers", value=allow_headers, expected_type=type_hints["allow_headers"])
                check_type(argname="argument allow_methods", value=allow_methods, expected_type=type_hints["allow_methods"])
                check_type(argname="argument allow_origins", value=allow_origins, expected_type=type_hints["allow_origins"])
                check_type(argname="argument expose_headers", value=expose_headers, expected_type=type_hints["expose_headers"])
                check_type(argname="argument max_age", value=max_age, expected_type=type_hints["max_age"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if allow_credentials is not None:
                self._values["allow_credentials"] = allow_credentials
            if allow_headers is not None:
                self._values["allow_headers"] = allow_headers
            if allow_methods is not None:
                self._values["allow_methods"] = allow_methods
            if allow_origins is not None:
                self._values["allow_origins"] = allow_origins
            if expose_headers is not None:
                self._values["expose_headers"] = expose_headers
            if max_age is not None:
                self._values["max_age"] = max_age

        @builtins.property
        def allow_credentials(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Whether you want to allow cookies or other credentials in requests to your function URL.

            The default is ``false`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-allowcredentials
            '''
            result = self._values.get("allow_credentials")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def allow_headers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The HTTP headers that origins can include in requests to your function URL.

            For example: ``Date`` , ``Keep-Alive`` , ``X-Custom-Header`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-allowheaders
            '''
            result = self._values.get("allow_headers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def allow_methods(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The HTTP methods that are allowed when calling your function URL.

            For example: ``GET`` , ``POST`` , ``DELETE`` , or the wildcard character ( ``*`` ).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-allowmethods
            '''
            result = self._values.get("allow_methods")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def allow_origins(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The origins that can access your function URL.

            You can list any number of specific origins, separated by a comma. For example: ``https://www.example.com`` , ``http://localhost:60905`` .

            Alternatively, you can grant access to all origins with the wildcard character ( ``*`` ).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-alloworigins
            '''
            result = self._values.get("allow_origins")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def expose_headers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The HTTP headers in your function response that you want to expose to origins that call your function URL.

            For example: ``Date`` , ``Keep-Alive`` , ``X-Custom-Header`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-exposeheaders
            '''
            result = self._values.get("expose_headers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def max_age(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of time, in seconds, that browsers can cache results of a preflight request.

            By default, this is set to ``0`` , which means the browser will not cache results.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-url-cors.html#cfn-lambda-url-cors-maxage
            '''
            result = self._values.get("max_age")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CorsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CfnUrlProps",
    jsii_struct_bases=[],
    name_mapping={
        "auth_type": "authType",
        "target_function_arn": "targetFunctionArn",
        "cors": "cors",
        "invoke_mode": "invokeMode",
        "qualifier": "qualifier",
    },
)
class CfnUrlProps:
    def __init__(
        self,
        *,
        auth_type: builtins.str,
        target_function_arn: builtins.str,
        cors: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUrl.CorsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        invoke_mode: typing.Optional[builtins.str] = None,
        qualifier: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnUrl``.

        :param auth_type: The type of authentication that your function URL uses. Set to ``AWS_IAM`` if you want to restrict access to authenticated users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .
        :param target_function_arn: The name of the Lambda function. **Name formats** - *Function name* - ``my-function`` . - *Function ARN* - ``lambda: : :function:my-function`` . - *Partial ARN* - ``:function:my-function`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param cors: The `Cross-Origin Resource Sharing (CORS) <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS>`_ settings for your function URL.
        :param invoke_mode: Use one of the following options:. - ``BUFFERED`` – This is the default option. Lambda invokes your function using the ``Invoke`` API operation. Invocation results are available when the payload is complete. The maximum payload size is 6 MB. - ``RESPONSE_STREAM`` – Your function streams payload results as they become available. Lambda invokes your function using the ``InvokeWithResponseStream`` API operation. The maximum response payload size is 20 MB, however, you can `request a quota increase <https://docs.aws.amazon.com/servicequotas/latest/userguide/request-quota-increase.html>`_ .
        :param qualifier: The alias name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            cfn_url_props = lambda.CfnUrlProps(
                auth_type="authType",
                target_function_arn="targetFunctionArn",
            
                # the properties below are optional
                cors=lambda.CfnUrl.CorsProperty(
                    allow_credentials=False,
                    allow_headers=["allowHeaders"],
                    allow_methods=["allowMethods"],
                    allow_origins=["allowOrigins"],
                    expose_headers=["exposeHeaders"],
                    max_age=123
                ),
                invoke_mode="invokeMode",
                qualifier="qualifier"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4634895f8e4f2a27a10f8fe128328e17466d0a557507896b4af8bb0780b94865)
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument target_function_arn", value=target_function_arn, expected_type=type_hints["target_function_arn"])
            check_type(argname="argument cors", value=cors, expected_type=type_hints["cors"])
            check_type(argname="argument invoke_mode", value=invoke_mode, expected_type=type_hints["invoke_mode"])
            check_type(argname="argument qualifier", value=qualifier, expected_type=type_hints["qualifier"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "auth_type": auth_type,
            "target_function_arn": target_function_arn,
        }
        if cors is not None:
            self._values["cors"] = cors
        if invoke_mode is not None:
            self._values["invoke_mode"] = invoke_mode
        if qualifier is not None:
            self._values["qualifier"] = qualifier

    @builtins.property
    def auth_type(self) -> builtins.str:
        '''The type of authentication that your function URL uses.

        Set to ``AWS_IAM`` if you want to restrict access to authenticated users only. Set to ``NONE`` if you want to bypass IAM authentication to create a public endpoint. For more information, see `Security and auth model for Lambda function URLs <https://docs.aws.amazon.com/lambda/latest/dg/urls-auth.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-authtype
        '''
        result = self._values.get("auth_type")
        assert result is not None, "Required property 'auth_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_function_arn(self) -> builtins.str:
        '''The name of the Lambda function.

        **Name formats** - *Function name* - ``my-function`` .

        - *Function ARN* - ``lambda:  :  :function:my-function`` .
        - *Partial ARN* - ``:function:my-function`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-targetfunctionarn
        '''
        result = self._values.get("target_function_arn")
        assert result is not None, "Required property 'target_function_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cors(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUrl.CorsProperty]]:
        '''The `Cross-Origin Resource Sharing (CORS) <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS>`_ settings for your function URL.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-cors
        '''
        result = self._values.get("cors")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUrl.CorsProperty]], result)

    @builtins.property
    def invoke_mode(self) -> typing.Optional[builtins.str]:
        '''Use one of the following options:.

        - ``BUFFERED`` – This is the default option. Lambda invokes your function using the ``Invoke`` API operation. Invocation results are available when the payload is complete. The maximum payload size is 6 MB.
        - ``RESPONSE_STREAM`` – Your function streams payload results as they become available. Lambda invokes your function using the ``InvokeWithResponseStream`` API operation. The maximum response payload size is 20 MB, however, you can `request a quota increase <https://docs.aws.amazon.com/servicequotas/latest/userguide/request-quota-increase.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-invokemode
        '''
        result = self._values.get("invoke_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def qualifier(self) -> typing.Optional[builtins.str]:
        '''The alias name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-url.html#cfn-lambda-url-qualifier
        '''
        result = self._values.get("qualifier")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUrlProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnVersion(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.CfnVersion",
):
    '''The ``AWS::Lambda::Version`` resource creates a `version <https://docs.aws.amazon.com/lambda/latest/dg/versioning-aliases.html>`_ from the current code and configuration of a function. Use versions to create a snapshot of your function code and configuration that doesn't change.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html
    :cloudformationResource: AWS::Lambda::Version
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_lambda as lambda_
        
        # policy: Any
        
        cfn_version = lambda_.CfnVersion(self, "MyCfnVersion",
            function_name="functionName",
        
            # the properties below are optional
            code_sha256="codeSha256",
            description="description",
            policy=policy,
            provisioned_concurrency_config=lambda.CfnVersion.ProvisionedConcurrencyConfigurationProperty(
                provisioned_concurrent_executions=123
            ),
            runtime_policy=lambda.CfnVersion.RuntimePolicyProperty(
                update_runtime_on="updateRuntimeOn",
        
                # the properties below are optional
                runtime_version_arn="runtimeVersionArn"
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        function_name: builtins.str,
        code_sha256: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        policy: typing.Any = None,
        provisioned_concurrency_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVersion.ProvisionedConcurrencyConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        runtime_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVersion.RuntimePolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param function_name: The name or ARN of the Lambda function. **Name formats** - *Function name* - ``MyFunction`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Partial ARN* - ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param code_sha256: Only publish a version if the hash value matches the value that's specified. Use this option to avoid publishing a version if the function code has changed since you last updated it. Updates are not supported for this property.
        :param description: A description for the version to override the description in the function configuration. Updates are not supported for this property.
        :param policy: The resource policy of your function.
        :param provisioned_concurrency_config: Specifies a provisioned concurrency configuration for a function's version. Updates are not supported for this property.
        :param runtime_policy: Runtime Management Config of a function.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1d4b3bf8a38fd246db911713fe99ad93f55dc635dbdaae114631921a17a9520c)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVersionProps(
            function_name=function_name,
            code_sha256=code_sha256,
            description=description,
            policy=policy,
            provisioned_concurrency_config=provisioned_concurrency_config,
            runtime_policy=runtime_policy,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__10084f0972104b355cf751cb519cb0fb7c9afcc8153c0b9c362c1274bec3e441)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8bfd5dea5c31cc46884bcf16d5e2829693fc6d7b55ae9336136c9ef9b16135e8)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrFunctionArn")
    def attr_function_arn(self) -> builtins.str:
        '''The ARN of the function.

        :cloudformationAttribute: FunctionArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFunctionArn"))

    @builtins.property
    @jsii.member(jsii_name="attrVersion")
    def attr_version(self) -> builtins.str:
        '''The version number.

        :cloudformationAttribute: Version
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVersion"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name or ARN of the Lambda function.'''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @function_name.setter
    def function_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__517ec870f1330321d13eea9fbc5944c14ea20218b8aa28c35494f72f5a6a1be3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="codeSha256")
    def code_sha256(self) -> typing.Optional[builtins.str]:
        '''Only publish a version if the hash value matches the value that's specified.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "codeSha256"))

    @code_sha256.setter
    def code_sha256(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__71885c0338af377c06eae5f6eef8ce2cdc338fb7b10c9e98ffaa14c9a1e85064)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "codeSha256", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description for the version to override the description in the function configuration.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f00e141771e43b8312d6a22399ba2d956886a5861db2296d1a2f16f8bba28b35)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="policy")
    def policy(self) -> typing.Any:
        '''The resource policy of your function.'''
        return typing.cast(typing.Any, jsii.get(self, "policy"))

    @policy.setter
    def policy(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ca802d0cfdb44c8f362bdb1d342a752f1a5ebd921d11a7b2fd0806cca5d9200a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "policy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="provisionedConcurrencyConfig")
    def provisioned_concurrency_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVersion.ProvisionedConcurrencyConfigurationProperty"]]:
        '''Specifies a provisioned concurrency configuration for a function's version.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVersion.ProvisionedConcurrencyConfigurationProperty"]], jsii.get(self, "provisionedConcurrencyConfig"))

    @provisioned_concurrency_config.setter
    def provisioned_concurrency_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVersion.ProvisionedConcurrencyConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e46534e1d83b58a93edc70a851ca891701ea58cfe9a3fc7dc106915c1a475fa6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "provisionedConcurrencyConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="runtimePolicy")
    def runtime_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVersion.RuntimePolicyProperty"]]:
        '''Runtime Management Config of a function.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVersion.RuntimePolicyProperty"]], jsii.get(self, "runtimePolicy"))

    @runtime_policy.setter
    def runtime_policy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVersion.RuntimePolicyProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__584bd4a88ccc7115243718679495b0e532174830ccb955a353592cc13c8ac084)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "runtimePolicy", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnVersion.ProvisionedConcurrencyConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "provisioned_concurrent_executions": "provisionedConcurrentExecutions",
        },
    )
    class ProvisionedConcurrencyConfigurationProperty:
        def __init__(self, *, provisioned_concurrent_executions: jsii.Number) -> None:
            '''A `provisioned concurrency <https://docs.aws.amazon.com/lambda/latest/dg/configuration-concurrency.html>`_ configuration for a function's version.

            :param provisioned_concurrent_executions: The amount of provisioned concurrency to allocate for the version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-version-provisionedconcurrencyconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                provisioned_concurrency_configuration_property = lambda.CfnVersion.ProvisionedConcurrencyConfigurationProperty(
                    provisioned_concurrent_executions=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__854d45fdc4890be298f65711c16c5fd275ec662f68a7e3284d88b2f89110a205)
                check_type(argname="argument provisioned_concurrent_executions", value=provisioned_concurrent_executions, expected_type=type_hints["provisioned_concurrent_executions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "provisioned_concurrent_executions": provisioned_concurrent_executions,
            }

        @builtins.property
        def provisioned_concurrent_executions(self) -> jsii.Number:
            '''The amount of provisioned concurrency to allocate for the version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-version-provisionedconcurrencyconfiguration.html#cfn-lambda-version-provisionedconcurrencyconfiguration-provisionedconcurrentexecutions
            '''
            result = self._values.get("provisioned_concurrent_executions")
            assert result is not None, "Required property 'provisioned_concurrent_executions' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProvisionedConcurrencyConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_lambda.CfnVersion.RuntimePolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "update_runtime_on": "updateRuntimeOn",
            "runtime_version_arn": "runtimeVersionArn",
        },
    )
    class RuntimePolicyProperty:
        def __init__(
            self,
            *,
            update_runtime_on: builtins.str,
            runtime_version_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Runtime Management Config of a function.

            :param update_runtime_on: The runtime update mode.
            :param runtime_version_arn: The ARN of the runtime the function is configured to use. If the runtime update mode is manual, the ARN is returned, otherwise null is returned.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-version-runtimepolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_lambda as lambda_
                
                runtime_policy_property = lambda.CfnVersion.RuntimePolicyProperty(
                    update_runtime_on="updateRuntimeOn",
                
                    # the properties below are optional
                    runtime_version_arn="runtimeVersionArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5271b1ea8263a6edb2968858109cfe102982debe5174ae7adf47b240b1c69f27)
                check_type(argname="argument update_runtime_on", value=update_runtime_on, expected_type=type_hints["update_runtime_on"])
                check_type(argname="argument runtime_version_arn", value=runtime_version_arn, expected_type=type_hints["runtime_version_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "update_runtime_on": update_runtime_on,
            }
            if runtime_version_arn is not None:
                self._values["runtime_version_arn"] = runtime_version_arn

        @builtins.property
        def update_runtime_on(self) -> builtins.str:
            '''The runtime update mode.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-version-runtimepolicy.html#cfn-lambda-version-runtimepolicy-updateruntimeon
            '''
            result = self._values.get("update_runtime_on")
            assert result is not None, "Required property 'update_runtime_on' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def runtime_version_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of the runtime the function is configured to use.

            If the runtime update mode is manual, the ARN is returned, otherwise null is returned.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-version-runtimepolicy.html#cfn-lambda-version-runtimepolicy-runtimeversionarn
            '''
            result = self._values.get("runtime_version_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RuntimePolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CfnVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "function_name": "functionName",
        "code_sha256": "codeSha256",
        "description": "description",
        "policy": "policy",
        "provisioned_concurrency_config": "provisionedConcurrencyConfig",
        "runtime_policy": "runtimePolicy",
    },
)
class CfnVersionProps:
    def __init__(
        self,
        *,
        function_name: builtins.str,
        code_sha256: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        policy: typing.Any = None,
        provisioned_concurrency_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnVersion.ProvisionedConcurrencyConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        runtime_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnVersion.RuntimePolicyProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnVersion``.

        :param function_name: The name or ARN of the Lambda function. **Name formats** - *Function name* - ``MyFunction`` . - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` . - *Partial ARN* - ``123456789012:function:MyFunction`` . The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.
        :param code_sha256: Only publish a version if the hash value matches the value that's specified. Use this option to avoid publishing a version if the function code has changed since you last updated it. Updates are not supported for this property.
        :param description: A description for the version to override the description in the function configuration. Updates are not supported for this property.
        :param policy: The resource policy of your function.
        :param provisioned_concurrency_config: Specifies a provisioned concurrency configuration for a function's version. Updates are not supported for this property.
        :param runtime_policy: Runtime Management Config of a function.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            # policy: Any
            
            cfn_version_props = lambda.CfnVersionProps(
                function_name="functionName",
            
                # the properties below are optional
                code_sha256="codeSha256",
                description="description",
                policy=policy,
                provisioned_concurrency_config=lambda.CfnVersion.ProvisionedConcurrencyConfigurationProperty(
                    provisioned_concurrent_executions=123
                ),
                runtime_policy=lambda.CfnVersion.RuntimePolicyProperty(
                    update_runtime_on="updateRuntimeOn",
            
                    # the properties below are optional
                    runtime_version_arn="runtimeVersionArn"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__63ba63c43bc52bb365203cebb308fd393d4c03a8aee52a0336a1396964d3cf9e)
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument code_sha256", value=code_sha256, expected_type=type_hints["code_sha256"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
            check_type(argname="argument provisioned_concurrency_config", value=provisioned_concurrency_config, expected_type=type_hints["provisioned_concurrency_config"])
            check_type(argname="argument runtime_policy", value=runtime_policy, expected_type=type_hints["runtime_policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_name": function_name,
        }
        if code_sha256 is not None:
            self._values["code_sha256"] = code_sha256
        if description is not None:
            self._values["description"] = description
        if policy is not None:
            self._values["policy"] = policy
        if provisioned_concurrency_config is not None:
            self._values["provisioned_concurrency_config"] = provisioned_concurrency_config
        if runtime_policy is not None:
            self._values["runtime_policy"] = runtime_policy

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The name or ARN of the Lambda function.

        **Name formats** - *Function name* - ``MyFunction`` .

        - *Function ARN* - ``arn:aws:lambda:us-west-2:123456789012:function:MyFunction`` .
        - *Partial ARN* - ``123456789012:function:MyFunction`` .

        The length constraint applies only to the full ARN. If you specify only the function name, it is limited to 64 characters in length.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-functionname
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def code_sha256(self) -> typing.Optional[builtins.str]:
        '''Only publish a version if the hash value matches the value that's specified.

        Use this option to avoid publishing a version if the function code has changed since you last updated it. Updates are not supported for this property.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-codesha256
        '''
        result = self._values.get("code_sha256")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description for the version to override the description in the function configuration.

        Updates are not supported for this property.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def policy(self) -> typing.Any:
        '''The resource policy of your function.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-policy
        '''
        result = self._values.get("policy")
        return typing.cast(typing.Any, result)

    @builtins.property
    def provisioned_concurrency_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnVersion.ProvisionedConcurrencyConfigurationProperty]]:
        '''Specifies a provisioned concurrency configuration for a function's version.

        Updates are not supported for this property.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-provisionedconcurrencyconfig
        '''
        result = self._values.get("provisioned_concurrency_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnVersion.ProvisionedConcurrencyConfigurationProperty]], result)

    @builtins.property
    def runtime_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnVersion.RuntimePolicyProperty]]:
        '''Runtime Management Config of a function.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-lambda-version.html#cfn-lambda-version-runtimepolicy
        '''
        result = self._values.get("runtime_policy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnVersion.RuntimePolicyProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Code(metaclass=jsii.JSIIAbstractClass, jsii_type="aws-cdk-lib.aws_lambda.Code"):
    '''Represents the Lambda Handler Code.

    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_signer as signer
        
        
        signing_profile = signer.SigningProfile(self, "SigningProfile",
            platform=signer.Platform.AWS_LAMBDA_SHA384_ECDSA
        )
        
        code_signing_config = lambda_.CodeSigningConfig(self, "CodeSigningConfig",
            signing_profiles=[signing_profile]
        )
        
        lambda_.Function(self, "Function",
            code_signing_config=code_signing_config,
            runtime=lambda_.Runtime.NODEJS_18_X,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromAsset")
    @builtins.classmethod
    def from_asset(
        cls,
        path: builtins.str,
        *,
        deploy_time: typing.Optional[builtins.bool] = None,
        display_name: typing.Optional[builtins.str] = None,
        readers: typing.Optional[typing.Sequence[_IGrantable_71c4f5de]] = None,
        source_kms_key: typing.Optional[_IKey_5f11635f] = None,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[_AssetHashType_05b67f2d] = None,
        bundling: typing.Optional[typing.Union[_BundlingOptions_588cc936, typing.Dict[builtins.str, typing.Any]]] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_047ec1f6] = None,
        ignore_mode: typing.Optional[_IgnoreMode_655a98e8] = None,
    ) -> "AssetCode":
        '''Loads the function code from a local disk path.

        :param path: Either a directory with the Lambda code bundle or a .zip file.
        :param deploy_time: Whether or not the asset needs to exist beyond deployment time; i.e. are copied over to a different location and not needed afterwards. Setting this property to true has an impact on the lifecycle of the asset, because we will assume that it is safe to delete after the CloudFormation deployment succeeds. For example, Lambda Function assets are copied over to Lambda during deployment. Therefore, it is not necessary to store the asset in S3, so we consider those deployTime assets. Default: false
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. If the same asset is added multiple times, the display name of the first occurrence is used. The default is the construct path of the Asset construct, with respect to the enclosing stack. If the asset is produced by a construct helper function (such as ``lambda.Code.fromAsset()``), this will look like ``MyFunction/Code``. We use the stack-relative construct path so that in the common case where you have multiple stacks with the same asset, we won't show something like ``/MyBetaStack/MyFunction/Code`` when you are actually deploying to production. Default: - Stack-relative construct path
        :param readers: A list of principals that should be able to read this asset from S3. You can use ``asset.grantRead(principal)`` to grant read permissions later. Default: - No principals that can read file asset.
        :param source_kms_key: The ARN of the KMS key used to encrypt the handler code. Default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.
        :param asset_hash: Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: Bundle the asset by executing a command in a Docker container or a custom bundling provider. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f040a1ba6e87fe9c9a6496be4b8fbf73f2646b80291bfc4d04979d6efb7216bf)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        options = _AssetOptions_2aa69621(
            deploy_time=deploy_time,
            display_name=display_name,
            readers=readers,
            source_kms_key=source_kms_key,
            asset_hash=asset_hash,
            asset_hash_type=asset_hash_type,
            bundling=bundling,
            exclude=exclude,
            follow_symlinks=follow_symlinks,
            ignore_mode=ignore_mode,
        )

        return typing.cast("AssetCode", jsii.sinvoke(cls, "fromAsset", [path, options]))

    @jsii.member(jsii_name="fromAssetImage")
    @builtins.classmethod
    def from_asset_image(
        cls,
        directory: builtins.str,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
        asset_name: typing.Optional[builtins.str] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        build_secrets: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        build_ssh: typing.Optional[builtins.str] = None,
        cache_disabled: typing.Optional[builtins.bool] = None,
        cache_from: typing.Optional[typing.Sequence[typing.Union[_DockerCacheOption_58ef18ca, typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_to: typing.Optional[typing.Union[_DockerCacheOption_58ef18ca, typing.Dict[builtins.str, typing.Any]]] = None,
        display_name: typing.Optional[builtins.str] = None,
        file: typing.Optional[builtins.str] = None,
        invalidation: typing.Optional[typing.Union[_DockerImageAssetInvalidationOptions_4deb8d45, typing.Dict[builtins.str, typing.Any]]] = None,
        network_mode: typing.Optional[_NetworkMode_897e5081] = None,
        outputs: typing.Optional[typing.Sequence[builtins.str]] = None,
        platform: typing.Optional[_Platform_d16f3cf1] = None,
        target: typing.Optional[builtins.str] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_047ec1f6] = None,
        ignore_mode: typing.Optional[_IgnoreMode_655a98e8] = None,
    ) -> "AssetImageCode":
        '''Create an ECR image from the specified asset and bind it as the Lambda code.

        :param directory: the directory from which the asset must be created.
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.
        :param asset_name: Unique identifier of the docker image asset and its potential revisions. Required if using AppScopedStagingSynthesizer. Default: - no asset name
        :param build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param build_secrets: Build secrets. Docker BuildKit must be enabled to use build secrets. Default: - no build secrets
        :param build_ssh: SSH agent socket or keys to pass to the ``docker build`` command. Docker BuildKit must be enabled to use the ssh flag Default: - no --ssh flag
        :param cache_disabled: Disable the cache and pass ``--no-cache`` to the ``docker build`` command. Default: - cache is used
        :param cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from options are passed to the build command
        :param cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to options are passed to the build command
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. If the same asset is added multiple times, the display name of the first occurrence is used. If ``assetName`` is given, it will also be used as the default ``displayName``. Otherwise, the default is the construct path of the ImageAsset construct, with respect to the enclosing stack. If the asset is produced by a construct helper function (such as ``lambda.Code.fromAssetImage()``), this will look like ``MyFunction/AssetImage``. We use the stack-relative construct path so that in the common case where you have multiple stacks with the same asset, we won't show something like ``/MyBetaStack/MyFunction/Code`` when you are actually deploying to production. Default: - Stack-relative construct path
        :param file: Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param invalidation: Options to control which parameters are used to invalidate the asset hash. Default: - hash all parameters
        :param network_mode: Networking mode for the RUN commands during build. Support docker API 1.25+. Default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        :param outputs: Outputs to pass to the ``docker build`` command. Default: - no outputs are passed to the build command (default outputs are used)
        :param platform: Platform to build for. *Requires Docker Buildx*. Default: - no platform specified (the current machine architecture will be used)
        :param target: Docker target to build to. Default: - no target
        :param extra_hash: Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__29667a5c639f33aff38b1cbeda0e1db7f1f0c3d3bdb2b53ad013906752e3793b)
            check_type(argname="argument directory", value=directory, expected_type=type_hints["directory"])
        props = AssetImageCodeProps(
            cmd=cmd,
            entrypoint=entrypoint,
            working_directory=working_directory,
            asset_name=asset_name,
            build_args=build_args,
            build_secrets=build_secrets,
            build_ssh=build_ssh,
            cache_disabled=cache_disabled,
            cache_from=cache_from,
            cache_to=cache_to,
            display_name=display_name,
            file=file,
            invalidation=invalidation,
            network_mode=network_mode,
            outputs=outputs,
            platform=platform,
            target=target,
            extra_hash=extra_hash,
            exclude=exclude,
            follow_symlinks=follow_symlinks,
            ignore_mode=ignore_mode,
        )

        return typing.cast("AssetImageCode", jsii.sinvoke(cls, "fromAssetImage", [directory, props]))

    @jsii.member(jsii_name="fromBucket")
    @builtins.classmethod
    def from_bucket(
        cls,
        bucket: _IBucket_42e086fd,
        key: builtins.str,
        object_version: typing.Optional[builtins.str] = None,
    ) -> "S3Code":
        '''Lambda handler code as an S3 object.

        :param bucket: The S3 bucket.
        :param key: The object key.
        :param object_version: Optional S3 object version.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cf2f362d90d470e1ea550c48af2d201151dbe9e28567f1f024ec091a2871d95c)
            check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument object_version", value=object_version, expected_type=type_hints["object_version"])
        return typing.cast("S3Code", jsii.sinvoke(cls, "fromBucket", [bucket, key, object_version]))

    @jsii.member(jsii_name="fromBucketV2")
    @builtins.classmethod
    def from_bucket_v2(
        cls,
        bucket: _IBucket_42e086fd,
        key: builtins.str,
        *,
        object_version: typing.Optional[builtins.str] = None,
        source_kms_key: typing.Optional[_IKey_5f11635f] = None,
    ) -> "S3CodeV2":
        '''Lambda handler code as an S3 object.

        :param bucket: The S3 bucket.
        :param key: The object key.
        :param object_version: Optional S3 object version.
        :param source_kms_key: The ARN of the KMS key used to encrypt the handler code. Default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3435b4d9a286e912d3934fc747c05554a23d64416b95f4dd12b911cd5bce166)
            check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
        options = BucketOptions(
            object_version=object_version, source_kms_key=source_kms_key
        )

        return typing.cast("S3CodeV2", jsii.sinvoke(cls, "fromBucketV2", [bucket, key, options]))

    @jsii.member(jsii_name="fromCfnParameters")
    @builtins.classmethod
    def from_cfn_parameters(
        cls,
        *,
        bucket_name_param: typing.Optional[_CfnParameter_48fc1866] = None,
        object_key_param: typing.Optional[_CfnParameter_48fc1866] = None,
        source_kms_key: typing.Optional[_IKey_5f11635f] = None,
    ) -> "CfnParametersCode":
        '''Creates a new Lambda source defined using CloudFormation parameters.

        :param bucket_name_param: The CloudFormation parameter that represents the name of the S3 Bucket where the Lambda code will be located in. Must be of type 'String'. Default: a new parameter will be created
        :param object_key_param: The CloudFormation parameter that represents the path inside the S3 Bucket where the Lambda code will be located at. Must be of type 'String'. Default: a new parameter will be created
        :param source_kms_key: The ARN of the KMS key used to encrypt the handler code. Default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.

        :return: a new instance of ``CfnParametersCode``
        '''
        props = CfnParametersCodeProps(
            bucket_name_param=bucket_name_param,
            object_key_param=object_key_param,
            source_kms_key=source_kms_key,
        )

        return typing.cast("CfnParametersCode", jsii.sinvoke(cls, "fromCfnParameters", [props]))

    @jsii.member(jsii_name="fromCustomCommand")
    @builtins.classmethod
    def from_custom_command(
        cls,
        output: builtins.str,
        command: typing.Sequence[builtins.str],
        *,
        command_options: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        deploy_time: typing.Optional[builtins.bool] = None,
        display_name: typing.Optional[builtins.str] = None,
        readers: typing.Optional[typing.Sequence[_IGrantable_71c4f5de]] = None,
        source_kms_key: typing.Optional[_IKey_5f11635f] = None,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[_AssetHashType_05b67f2d] = None,
        bundling: typing.Optional[typing.Union[_BundlingOptions_588cc936, typing.Dict[builtins.str, typing.Any]]] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_047ec1f6] = None,
        ignore_mode: typing.Optional[_IgnoreMode_655a98e8] = None,
    ) -> "AssetCode":
        '''Runs a command to build the code asset that will be used.

        :param output: Where the output of the command will be directed, either a directory or a .zip file with the output Lambda code bundle * For example, if you use the command to run a build script (e.g., [ 'node', 'bundle_code.js' ]), and the build script generates a directory ``/my/lambda/code`` containing code that should be ran in a Lambda function, then output should be set to ``/my/lambda/code``.
        :param command: The command which will be executed to generate the output, for example, [ 'node', 'bundle_code.js' ].
        :param command_options: options that are passed to the spawned process, which determine the characteristics of the spawned process. Default: : see ``child_process.SpawnSyncOptions`` (https://nodejs.org/api/child_process.html#child_processspawnsynccommand-args-options).
        :param deploy_time: Whether or not the asset needs to exist beyond deployment time; i.e. are copied over to a different location and not needed afterwards. Setting this property to true has an impact on the lifecycle of the asset, because we will assume that it is safe to delete after the CloudFormation deployment succeeds. For example, Lambda Function assets are copied over to Lambda during deployment. Therefore, it is not necessary to store the asset in S3, so we consider those deployTime assets. Default: false
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. If the same asset is added multiple times, the display name of the first occurrence is used. The default is the construct path of the Asset construct, with respect to the enclosing stack. If the asset is produced by a construct helper function (such as ``lambda.Code.fromAsset()``), this will look like ``MyFunction/Code``. We use the stack-relative construct path so that in the common case where you have multiple stacks with the same asset, we won't show something like ``/MyBetaStack/MyFunction/Code`` when you are actually deploying to production. Default: - Stack-relative construct path
        :param readers: A list of principals that should be able to read this asset from S3. You can use ``asset.grantRead(principal)`` to grant read permissions later. Default: - No principals that can read file asset.
        :param source_kms_key: The ARN of the KMS key used to encrypt the handler code. Default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.
        :param asset_hash: Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: Bundle the asset by executing a command in a Docker container or a custom bundling provider. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f107aedaa96b9385600e34088d5cda9d8035f15776c846b0f0b4fbbe35d118df)
            check_type(argname="argument output", value=output, expected_type=type_hints["output"])
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
        options = CustomCommandOptions(
            command_options=command_options,
            deploy_time=deploy_time,
            display_name=display_name,
            readers=readers,
            source_kms_key=source_kms_key,
            asset_hash=asset_hash,
            asset_hash_type=asset_hash_type,
            bundling=bundling,
            exclude=exclude,
            follow_symlinks=follow_symlinks,
            ignore_mode=ignore_mode,
        )

        return typing.cast("AssetCode", jsii.sinvoke(cls, "fromCustomCommand", [output, command, options]))

    @jsii.member(jsii_name="fromDockerBuild")
    @builtins.classmethod
    def from_docker_build(
        cls,
        path: builtins.str,
        *,
        image_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        cache_disabled: typing.Optional[builtins.bool] = None,
        cache_from: typing.Optional[typing.Sequence[typing.Union[_DockerCacheOption_63aeecf2, typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_to: typing.Optional[typing.Union[_DockerCacheOption_63aeecf2, typing.Dict[builtins.str, typing.Any]]] = None,
        file: typing.Optional[builtins.str] = None,
        platform: typing.Optional[builtins.str] = None,
        target_stage: typing.Optional[builtins.str] = None,
    ) -> "AssetCode":
        '''Loads the function code from an asset created by a Docker build.

        By default, the asset is expected to be located at ``/asset`` in the
        image.

        :param path: The path to the directory containing the Docker file.
        :param image_path: The path in the Docker image where the asset is located after the build operation. Default: /asset
        :param output_path: The path on the local filesystem where the asset will be copied using ``docker cp``. Default: - a unique temporary directory in the system temp directory
        :param build_args: Build args. Default: - no build args
        :param cache_disabled: Disable the cache and pass ``--no-cache`` to the ``docker build`` command. Default: - cache is used
        :param cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from args are passed
        :param cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to args are passed
        :param file: Name of the Dockerfile, must relative to the docker build path. Default: ``Dockerfile``
        :param platform: Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*. Example value: ``linux/amd64`` Default: - no platform specified
        :param target_stage: Set build target for multi-stage container builds. Any stage defined afterwards will be ignored. Example value: ``build-env`` Default: - Build all stages defined in the Dockerfile
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d398ddff6fb1e58c2dafd59b4cd46de157e95f7c1faf544bd0e4cee3a6c9765)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        options = DockerBuildAssetOptions(
            image_path=image_path,
            output_path=output_path,
            build_args=build_args,
            cache_disabled=cache_disabled,
            cache_from=cache_from,
            cache_to=cache_to,
            file=file,
            platform=platform,
            target_stage=target_stage,
        )

        return typing.cast("AssetCode", jsii.sinvoke(cls, "fromDockerBuild", [path, options]))

    @jsii.member(jsii_name="fromEcrImage")
    @builtins.classmethod
    def from_ecr_image(
        cls,
        repository: _IRepository_e6004aa6,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        tag_or_digest: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> "EcrImageCode":
        '''Use an existing ECR image as the Lambda code.

        :param repository: the ECR repository that the image is in.
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param tag: (deprecated) The image tag to use when pulling the image from ECR. Default: 'latest'
        :param tag_or_digest: The image tag or digest to use when pulling the image from ECR (digests must start with ``sha256:``). Default: 'latest'
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b0cbf2b2fc1b9f587403b386e60550ab4b7016b5b442adcf077b6a77a83821d6)
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
        props = EcrImageCodeProps(
            cmd=cmd,
            entrypoint=entrypoint,
            tag=tag,
            tag_or_digest=tag_or_digest,
            working_directory=working_directory,
        )

        return typing.cast("EcrImageCode", jsii.sinvoke(cls, "fromEcrImage", [repository, props]))

    @jsii.member(jsii_name="fromInline")
    @builtins.classmethod
    def from_inline(cls, code: builtins.str) -> "InlineCode":
        '''Inline code for Lambda handler.

        :param code: The actual handler code (the resulting zip file cannot exceed 4MB).

        :return: ``LambdaInlineCode`` with inline code.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ad6d089865a75ad5d735297fff6fefd5609177b3e3ddcc122fb0a34c9081b3a4)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
        return typing.cast("InlineCode", jsii.sinvoke(cls, "fromInline", [code]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, scope: _constructs_77d1e7e8.Construct) -> "CodeConfig":
        '''Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.

        :param scope: The binding scope. Don't be smart about trying to down-cast or assume it's initialized. You may just use it as a construct scope.
        '''
        ...

    @jsii.member(jsii_name="bindToResource")
    def bind_to_resource(
        self,
        _resource: _CfnResource_9df397a6,
        *,
        resource_property: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Called after the CFN function resource has been created to allow the code class to bind to it.

        Specifically it's required to allow assets to add
        metadata for tooling like SAM CLI to be able to find their origins.

        :param _resource: -
        :param resource_property: The name of the CloudFormation property to annotate with asset metadata. Default: Code
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7d72d00c4ec0a638df8324832083731239aea97c6cdfae3924abcb51f464f525)
            check_type(argname="argument _resource", value=_resource, expected_type=type_hints["_resource"])
        _options = ResourceBindOptions(resource_property=resource_property)

        return typing.cast(None, jsii.invoke(self, "bindToResource", [_resource, _options]))


class _CodeProxy(Code):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _constructs_77d1e7e8.Construct) -> "CodeConfig":
        '''Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.

        :param scope: The binding scope. Don't be smart about trying to down-cast or assume it's initialized. You may just use it as a construct scope.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c426bb3ada8bbd044f56c38788717e2922c82efd2ff55a6f8e9dc05258d58e0b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("CodeConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Code).__jsii_proxy_class__ = lambda : _CodeProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CodeConfig",
    jsii_struct_bases=[],
    name_mapping={
        "image": "image",
        "inline_code": "inlineCode",
        "s3_location": "s3Location",
        "source_kms_key_arn": "sourceKMSKeyArn",
    },
)
class CodeConfig:
    def __init__(
        self,
        *,
        image: typing.Optional[typing.Union["CodeImageConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        inline_code: typing.Optional[builtins.str] = None,
        s3_location: typing.Optional[typing.Union[_Location_0948fa7f, typing.Dict[builtins.str, typing.Any]]] = None,
        source_kms_key_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Result of binding ``Code`` into a ``Function``.

        :param image: Docker image configuration (mutually exclusive with ``s3Location`` and ``inlineCode``). Default: - code is not an ECR container image
        :param inline_code: Inline code (mutually exclusive with ``s3Location`` and ``image``). Default: - code is not inline code
        :param s3_location: The location of the code in S3 (mutually exclusive with ``inlineCode`` and ``image``). Default: - code is not an s3 location
        :param source_kms_key_arn: The ARN of the KMS key used to encrypt the handler code. Default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            code_config = lambda.CodeConfig(
                image=lambda.CodeImageConfig(
                    image_uri="imageUri",
            
                    # the properties below are optional
                    cmd=["cmd"],
                    entrypoint=["entrypoint"],
                    working_directory="workingDirectory"
                ),
                inline_code="inlineCode",
                s3_location=Location(
                    bucket_name="bucketName",
                    object_key="objectKey",
            
                    # the properties below are optional
                    object_version="objectVersion"
                ),
                source_kMSKey_arn="sourceKMSKeyArn"
            )
        '''
        if isinstance(image, dict):
            image = CodeImageConfig(**image)
        if isinstance(s3_location, dict):
            s3_location = _Location_0948fa7f(**s3_location)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9f94faae4fd35a06e3f67763f77da934d65414f84781e7e17169eece98c58bef)
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
            check_type(argname="argument inline_code", value=inline_code, expected_type=type_hints["inline_code"])
            check_type(argname="argument s3_location", value=s3_location, expected_type=type_hints["s3_location"])
            check_type(argname="argument source_kms_key_arn", value=source_kms_key_arn, expected_type=type_hints["source_kms_key_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if image is not None:
            self._values["image"] = image
        if inline_code is not None:
            self._values["inline_code"] = inline_code
        if s3_location is not None:
            self._values["s3_location"] = s3_location
        if source_kms_key_arn is not None:
            self._values["source_kms_key_arn"] = source_kms_key_arn

    @builtins.property
    def image(self) -> typing.Optional["CodeImageConfig"]:
        '''Docker image configuration (mutually exclusive with ``s3Location`` and ``inlineCode``).

        :default: - code is not an ECR container image
        '''
        result = self._values.get("image")
        return typing.cast(typing.Optional["CodeImageConfig"], result)

    @builtins.property
    def inline_code(self) -> typing.Optional[builtins.str]:
        '''Inline code (mutually exclusive with ``s3Location`` and ``image``).

        :default: - code is not inline code
        '''
        result = self._values.get("inline_code")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def s3_location(self) -> typing.Optional[_Location_0948fa7f]:
        '''The location of the code in S3 (mutually exclusive with ``inlineCode`` and ``image``).

        :default: - code is not an s3 location
        '''
        result = self._values.get("s3_location")
        return typing.cast(typing.Optional[_Location_0948fa7f], result)

    @builtins.property
    def source_kms_key_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the KMS key used to encrypt the handler code.

        :default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.
        '''
        result = self._values.get("source_kms_key_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CodeConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CodeImageConfig",
    jsii_struct_bases=[],
    name_mapping={
        "image_uri": "imageUri",
        "cmd": "cmd",
        "entrypoint": "entrypoint",
        "working_directory": "workingDirectory",
    },
)
class CodeImageConfig:
    def __init__(
        self,
        *,
        image_uri: builtins.str,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Result of the bind when an ECR image is used.

        :param image_uri: URI to the Docker image.
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            code_image_config = lambda.CodeImageConfig(
                image_uri="imageUri",
            
                # the properties below are optional
                cmd=["cmd"],
                entrypoint=["entrypoint"],
                working_directory="workingDirectory"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a29631153db2616d88184dc9187bc6cffe21cf9d82b1ada0446fa60004ee7de1)
            check_type(argname="argument image_uri", value=image_uri, expected_type=type_hints["image_uri"])
            check_type(argname="argument cmd", value=cmd, expected_type=type_hints["cmd"])
            check_type(argname="argument entrypoint", value=entrypoint, expected_type=type_hints["entrypoint"])
            check_type(argname="argument working_directory", value=working_directory, expected_type=type_hints["working_directory"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "image_uri": image_uri,
        }
        if cmd is not None:
            self._values["cmd"] = cmd
        if entrypoint is not None:
            self._values["entrypoint"] = entrypoint
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def image_uri(self) -> builtins.str:
        '''URI to the Docker image.'''
        result = self._values.get("image_uri")
        assert result is not None, "Required property 'image_uri' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cmd(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the CMD on the specified Docker image or Dockerfile.

        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the CMD specified in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#cmd
        '''
        result = self._values.get("cmd")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entrypoint(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile.

        An ENTRYPOINT allows you to configure a container that will run as an executable.
        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the ENTRYPOINT in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        '''
        result = self._values.get("entrypoint")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''Specify or override the WORKDIR on the specified Docker image or Dockerfile.

        A WORKDIR allows you to configure the working directory the container will use.

        :default: - use the WORKDIR in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#workdir
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CodeImageConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CodeSigningConfigProps",
    jsii_struct_bases=[],
    name_mapping={
        "signing_profiles": "signingProfiles",
        "description": "description",
        "untrusted_artifact_on_deployment": "untrustedArtifactOnDeployment",
    },
)
class CodeSigningConfigProps:
    def __init__(
        self,
        *,
        signing_profiles: typing.Sequence[_ISigningProfile_5140a756],
        description: typing.Optional[builtins.str] = None,
        untrusted_artifact_on_deployment: typing.Optional["UntrustedArtifactOnDeployment"] = None,
    ) -> None:
        '''Construction properties for a Code Signing Config object.

        :param signing_profiles: List of signing profiles that defines a trusted user who can sign a code package.
        :param description: Code signing configuration description. Default: - No description.
        :param untrusted_artifact_on_deployment: Code signing configuration policy for deployment validation failure. If you set the policy to Enforce, Lambda blocks the deployment request if signature validation checks fail. If you set the policy to Warn, Lambda allows the deployment and creates a CloudWatch log. Default: UntrustedArtifactOnDeployment.WARN

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_signer as signer
            
            
            signing_profile = signer.SigningProfile(self, "SigningProfile",
                platform=signer.Platform.AWS_LAMBDA_SHA384_ECDSA
            )
            
            code_signing_config = lambda_.CodeSigningConfig(self, "CodeSigningConfig",
                signing_profiles=[signing_profile]
            )
            
            lambda_.Function(self, "Function",
                code_signing_config=code_signing_config,
                runtime=lambda_.Runtime.NODEJS_18_X,
                handler="index.handler",
                code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aefe636ff4357a418ff1336047558f42e531611ca56525b6b5997177b43a9ad3)
            check_type(argname="argument signing_profiles", value=signing_profiles, expected_type=type_hints["signing_profiles"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument untrusted_artifact_on_deployment", value=untrusted_artifact_on_deployment, expected_type=type_hints["untrusted_artifact_on_deployment"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "signing_profiles": signing_profiles,
        }
        if description is not None:
            self._values["description"] = description
        if untrusted_artifact_on_deployment is not None:
            self._values["untrusted_artifact_on_deployment"] = untrusted_artifact_on_deployment

    @builtins.property
    def signing_profiles(self) -> typing.List[_ISigningProfile_5140a756]:
        '''List of signing profiles that defines a trusted user who can sign a code package.'''
        result = self._values.get("signing_profiles")
        assert result is not None, "Required property 'signing_profiles' is missing"
        return typing.cast(typing.List[_ISigningProfile_5140a756], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''Code signing configuration description.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def untrusted_artifact_on_deployment(
        self,
    ) -> typing.Optional["UntrustedArtifactOnDeployment"]:
        '''Code signing configuration policy for deployment validation failure.

        If you set the policy to Enforce, Lambda blocks the deployment request
        if signature validation checks fail.
        If you set the policy to Warn, Lambda allows the deployment and
        creates a CloudWatch log.

        :default: UntrustedArtifactOnDeployment.WARN
        '''
        result = self._values.get("untrusted_artifact_on_deployment")
        return typing.cast(typing.Optional["UntrustedArtifactOnDeployment"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CodeSigningConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.CustomCommandOptions",
    jsii_struct_bases=[_AssetOptions_2aa69621],
    name_mapping={
        "asset_hash": "assetHash",
        "asset_hash_type": "assetHashType",
        "bundling": "bundling",
        "exclude": "exclude",
        "follow_symlinks": "followSymlinks",
        "ignore_mode": "ignoreMode",
        "deploy_time": "deployTime",
        "display_name": "displayName",
        "readers": "readers",
        "source_kms_key": "sourceKMSKey",
        "command_options": "commandOptions",
    },
)
class CustomCommandOptions(_AssetOptions_2aa69621):
    def __init__(
        self,
        *,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[_AssetHashType_05b67f2d] = None,
        bundling: typing.Optional[typing.Union[_BundlingOptions_588cc936, typing.Dict[builtins.str, typing.Any]]] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_047ec1f6] = None,
        ignore_mode: typing.Optional[_IgnoreMode_655a98e8] = None,
        deploy_time: typing.Optional[builtins.bool] = None,
        display_name: typing.Optional[builtins.str] = None,
        readers: typing.Optional[typing.Sequence[_IGrantable_71c4f5de]] = None,
        source_kms_key: typing.Optional[_IKey_5f11635f] = None,
        command_options: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''Options for creating ``AssetCode`` with a custom command, such as running a buildfile.

        :param asset_hash: Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: Bundle the asset by executing a command in a Docker container or a custom bundling provider. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        :param deploy_time: Whether or not the asset needs to exist beyond deployment time; i.e. are copied over to a different location and not needed afterwards. Setting this property to true has an impact on the lifecycle of the asset, because we will assume that it is safe to delete after the CloudFormation deployment succeeds. For example, Lambda Function assets are copied over to Lambda during deployment. Therefore, it is not necessary to store the asset in S3, so we consider those deployTime assets. Default: false
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. If the same asset is added multiple times, the display name of the first occurrence is used. The default is the construct path of the Asset construct, with respect to the enclosing stack. If the asset is produced by a construct helper function (such as ``lambda.Code.fromAsset()``), this will look like ``MyFunction/Code``. We use the stack-relative construct path so that in the common case where you have multiple stacks with the same asset, we won't show something like ``/MyBetaStack/MyFunction/Code`` when you are actually deploying to production. Default: - Stack-relative construct path
        :param readers: A list of principals that should be able to read this asset from S3. You can use ``asset.grantRead(principal)`` to grant read permissions later. Default: - No principals that can read file asset.
        :param source_kms_key: The ARN of the KMS key used to encrypt the handler code. Default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.
        :param command_options: options that are passed to the spawned process, which determine the characteristics of the spawned process. Default: : see ``child_process.SpawnSyncOptions`` (https://nodejs.org/api/child_process.html#child_processspawnsynccommand-args-options).

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_iam as iam
            from aws_cdk import aws_kms as kms
            from aws_cdk import aws_lambda as lambda_
            
            # command_options: Any
            # docker_image: cdk.DockerImage
            # grantable: iam.IGrantable
            # key: kms.Key
            # local_bundling: cdk.ILocalBundling
            
            custom_command_options = lambda.CustomCommandOptions(
                asset_hash="assetHash",
                asset_hash_type=cdk.AssetHashType.SOURCE,
                bundling=cdk.BundlingOptions(
                    image=docker_image,
            
                    # the properties below are optional
                    bundling_file_access=cdk.BundlingFileAccess.VOLUME_COPY,
                    command=["command"],
                    entrypoint=["entrypoint"],
                    environment={
                        "environment_key": "environment"
                    },
                    local=local_bundling,
                    network="network",
                    output_type=cdk.BundlingOutput.ARCHIVED,
                    platform="platform",
                    security_opt="securityOpt",
                    user="user",
                    volumes=[cdk.DockerVolume(
                        container_path="containerPath",
                        host_path="hostPath",
            
                        # the properties below are optional
                        consistency=cdk.DockerVolumeConsistency.CONSISTENT
                    )],
                    volumes_from=["volumesFrom"],
                    working_directory="workingDirectory"
                ),
                command_options={
                    "command_options_key": command_options
                },
                deploy_time=False,
                display_name="displayName",
                exclude=["exclude"],
                follow_symlinks=cdk.SymlinkFollowMode.NEVER,
                ignore_mode=cdk.IgnoreMode.GLOB,
                readers=[grantable],
                source_kMSKey=key
            )
        '''
        if isinstance(bundling, dict):
            bundling = _BundlingOptions_588cc936(**bundling)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__69255a578358e8a47662200dda7ce2e0b1f2ee573c1469268f14060b00a7863a)
            check_type(argname="argument asset_hash", value=asset_hash, expected_type=type_hints["asset_hash"])
            check_type(argname="argument asset_hash_type", value=asset_hash_type, expected_type=type_hints["asset_hash_type"])
            check_type(argname="argument bundling", value=bundling, expected_type=type_hints["bundling"])
            check_type(argname="argument exclude", value=exclude, expected_type=type_hints["exclude"])
            check_type(argname="argument follow_symlinks", value=follow_symlinks, expected_type=type_hints["follow_symlinks"])
            check_type(argname="argument ignore_mode", value=ignore_mode, expected_type=type_hints["ignore_mode"])
            check_type(argname="argument deploy_time", value=deploy_time, expected_type=type_hints["deploy_time"])
            check_type(argname="argument display_name", value=display_name, expected_type=type_hints["display_name"])
            check_type(argname="argument readers", value=readers, expected_type=type_hints["readers"])
            check_type(argname="argument source_kms_key", value=source_kms_key, expected_type=type_hints["source_kms_key"])
            check_type(argname="argument command_options", value=command_options, expected_type=type_hints["command_options"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if asset_hash is not None:
            self._values["asset_hash"] = asset_hash
        if asset_hash_type is not None:
            self._values["asset_hash_type"] = asset_hash_type
        if bundling is not None:
            self._values["bundling"] = bundling
        if exclude is not None:
            self._values["exclude"] = exclude
        if follow_symlinks is not None:
            self._values["follow_symlinks"] = follow_symlinks
        if ignore_mode is not None:
            self._values["ignore_mode"] = ignore_mode
        if deploy_time is not None:
            self._values["deploy_time"] = deploy_time
        if display_name is not None:
            self._values["display_name"] = display_name
        if readers is not None:
            self._values["readers"] = readers
        if source_kms_key is not None:
            self._values["source_kms_key"] = source_kms_key
        if command_options is not None:
            self._values["command_options"] = command_options

    @builtins.property
    def asset_hash(self) -> typing.Optional[builtins.str]:
        '''Specify a custom hash for this asset.

        If ``assetHashType`` is set it must
        be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will
        be SHA256 hashed and encoded as hex. The resulting hash will be the asset
        hash.

        NOTE: the hash is used in order to identify a specific revision of the asset, and
        used for optimizing and caching deployment activities related to this asset such as
        packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will
        need to make sure it is updated every time the asset changes, or otherwise it is
        possible that some deployments will not be invalidated.

        :default: - based on ``assetHashType``
        '''
        result = self._values.get("asset_hash")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def asset_hash_type(self) -> typing.Optional[_AssetHashType_05b67f2d]:
        '''Specifies the type of hash to calculate for this asset.

        If ``assetHash`` is configured, this option must be ``undefined`` or
        ``AssetHashType.CUSTOM``.

        :default:

        - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is
        explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        '''
        result = self._values.get("asset_hash_type")
        return typing.cast(typing.Optional[_AssetHashType_05b67f2d], result)

    @builtins.property
    def bundling(self) -> typing.Optional[_BundlingOptions_588cc936]:
        '''Bundle the asset by executing a command in a Docker container or a custom bundling provider.

        The asset path will be mounted at ``/asset-input``. The Docker
        container is responsible for putting content at ``/asset-output``.
        The content at ``/asset-output`` will be zipped and used as the
        final asset.

        :default:

        - uploaded as-is to S3 if the asset is a regular file or a .zip file,
        archived into a .zip file and uploaded to S3 otherwise
        '''
        result = self._values.get("bundling")
        return typing.cast(typing.Optional[_BundlingOptions_588cc936], result)

    @builtins.property
    def exclude(self) -> typing.Optional[typing.List[builtins.str]]:
        '''File paths matching the patterns will be excluded.

        See ``ignoreMode`` to set the matching behavior.
        Has no effect on Assets bundled using the ``bundling`` property.

        :default: - nothing is excluded
        '''
        result = self._values.get("exclude")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def follow_symlinks(self) -> typing.Optional[_SymlinkFollowMode_047ec1f6]:
        '''A strategy for how to handle symlinks.

        :default: SymlinkFollowMode.NEVER
        '''
        result = self._values.get("follow_symlinks")
        return typing.cast(typing.Optional[_SymlinkFollowMode_047ec1f6], result)

    @builtins.property
    def ignore_mode(self) -> typing.Optional[_IgnoreMode_655a98e8]:
        '''The ignore behavior to use for ``exclude`` patterns.

        :default: IgnoreMode.GLOB
        '''
        result = self._values.get("ignore_mode")
        return typing.cast(typing.Optional[_IgnoreMode_655a98e8], result)

    @builtins.property
    def deploy_time(self) -> typing.Optional[builtins.bool]:
        '''Whether or not the asset needs to exist beyond deployment time;

        i.e.
        are copied over to a different location and not needed afterwards.
        Setting this property to true has an impact on the lifecycle of the asset,
        because we will assume that it is safe to delete after the CloudFormation
        deployment succeeds.

        For example, Lambda Function assets are copied over to Lambda during
        deployment. Therefore, it is not necessary to store the asset in S3, so
        we consider those deployTime assets.

        :default: false
        '''
        result = self._values.get("deploy_time")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def display_name(self) -> typing.Optional[builtins.str]:
        '''A display name for this asset.

        If supplied, the display name will be used in locations where the asset
        identifier is printed, like in the CLI progress information. If the same
        asset is added multiple times, the display name of the first occurrence is
        used.

        The default is the construct path of the Asset construct, with respect to
        the enclosing stack. If the asset is produced by a construct helper
        function (such as ``lambda.Code.fromAsset()``), this will look like
        ``MyFunction/Code``.

        We use the stack-relative construct path so that in the common case where
        you have multiple stacks with the same asset, we won't show something like
        ``/MyBetaStack/MyFunction/Code`` when you are actually deploying to
        production.

        :default: - Stack-relative construct path
        '''
        result = self._values.get("display_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def readers(self) -> typing.Optional[typing.List[_IGrantable_71c4f5de]]:
        '''A list of principals that should be able to read this asset from S3.

        You can use ``asset.grantRead(principal)`` to grant read permissions later.

        :default: - No principals that can read file asset.
        '''
        result = self._values.get("readers")
        return typing.cast(typing.Optional[typing.List[_IGrantable_71c4f5de]], result)

    @builtins.property
    def source_kms_key(self) -> typing.Optional[_IKey_5f11635f]:
        '''The ARN of the KMS key used to encrypt the handler code.

        :default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.
        '''
        result = self._values.get("source_kms_key")
        return typing.cast(typing.Optional[_IKey_5f11635f], result)

    @builtins.property
    def command_options(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''options that are passed to the spawned process, which determine the characteristics of the spawned process.

        :default: : see ``child_process.SpawnSyncOptions`` (https://nodejs.org/api/child_process.html#child_processspawnsynccommand-args-options).
        '''
        result = self._values.get("command_options")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CustomCommandOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.DestinationConfig",
    jsii_struct_bases=[],
    name_mapping={"destination": "destination"},
)
class DestinationConfig:
    def __init__(self, *, destination: builtins.str) -> None:
        '''A destination configuration.

        :param destination: The Amazon Resource Name (ARN) of the destination resource.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            destination_config = lambda.DestinationConfig(
                destination="destination"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7b17dd9a780787e4acd806a7d2f2738f8873c3cee8af497862be683cbad3bd8a)
            check_type(argname="argument destination", value=destination, expected_type=type_hints["destination"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "destination": destination,
        }

    @builtins.property
    def destination(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the destination resource.'''
        result = self._values.get("destination")
        assert result is not None, "Required property 'destination' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DestinationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.DestinationOptions",
    jsii_struct_bases=[],
    name_mapping={"type": "type"},
)
class DestinationOptions:
    def __init__(self, *, type: "DestinationType") -> None:
        '''Options when binding a destination to a function.

        :param type: The destination type.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            destination_options = lambda.DestinationOptions(
                type=lambda_.DestinationType.FAILURE
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__41a2f579cde815bb4d51953e99d89f497d631429035e443b4e0c861700d84a9a)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }

    @builtins.property
    def type(self) -> "DestinationType":
        '''The destination type.'''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("DestinationType", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DestinationOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_lambda.DestinationType")
class DestinationType(enum.Enum):
    '''The type of destination.'''

    FAILURE = "FAILURE"
    '''Failure.'''
    SUCCESS = "SUCCESS"
    '''Success.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.DlqDestinationConfig",
    jsii_struct_bases=[],
    name_mapping={"destination": "destination"},
)
class DlqDestinationConfig:
    def __init__(self, *, destination: builtins.str) -> None:
        '''A destination configuration.

        :param destination: The Amazon Resource Name (ARN) of the destination resource.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            dlq_destination_config = lambda.DlqDestinationConfig(
                destination="destination"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2ac0cd879a29ceed59bd2456c1e1ffddc2206dcdfd5aa4e94b3add2e47fcf5c0)
            check_type(argname="argument destination", value=destination, expected_type=type_hints["destination"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "destination": destination,
        }

    @builtins.property
    def destination(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the destination resource.'''
        result = self._values.get("destination")
        assert result is not None, "Required property 'destination' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DlqDestinationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.DockerBuildAssetOptions",
    jsii_struct_bases=[_DockerBuildOptions_8e28be8a],
    name_mapping={
        "build_args": "buildArgs",
        "cache_disabled": "cacheDisabled",
        "cache_from": "cacheFrom",
        "cache_to": "cacheTo",
        "file": "file",
        "platform": "platform",
        "target_stage": "targetStage",
        "image_path": "imagePath",
        "output_path": "outputPath",
    },
)
class DockerBuildAssetOptions(_DockerBuildOptions_8e28be8a):
    def __init__(
        self,
        *,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        cache_disabled: typing.Optional[builtins.bool] = None,
        cache_from: typing.Optional[typing.Sequence[typing.Union[_DockerCacheOption_63aeecf2, typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_to: typing.Optional[typing.Union[_DockerCacheOption_63aeecf2, typing.Dict[builtins.str, typing.Any]]] = None,
        file: typing.Optional[builtins.str] = None,
        platform: typing.Optional[builtins.str] = None,
        target_stage: typing.Optional[builtins.str] = None,
        image_path: typing.Optional[builtins.str] = None,
        output_path: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options when creating an asset from a Docker build.

        :param build_args: Build args. Default: - no build args
        :param cache_disabled: Disable the cache and pass ``--no-cache`` to the ``docker build`` command. Default: - cache is used
        :param cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from args are passed
        :param cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to args are passed
        :param file: Name of the Dockerfile, must relative to the docker build path. Default: ``Dockerfile``
        :param platform: Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*. Example value: ``linux/amd64`` Default: - no platform specified
        :param target_stage: Set build target for multi-stage container builds. Any stage defined afterwards will be ignored. Example value: ``build-env`` Default: - Build all stages defined in the Dockerfile
        :param image_path: The path in the Docker image where the asset is located after the build operation. Default: /asset
        :param output_path: The path on the local filesystem where the asset will be copied using ``docker cp``. Default: - a unique temporary directory in the system temp directory

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            docker_build_asset_options = lambda.DockerBuildAssetOptions(
                build_args={
                    "build_args_key": "buildArgs"
                },
                cache_disabled=False,
                cache_from=[DockerCacheOption(
                    type="type",
            
                    # the properties below are optional
                    params={
                        "params_key": "params"
                    }
                )],
                cache_to=DockerCacheOption(
                    type="type",
            
                    # the properties below are optional
                    params={
                        "params_key": "params"
                    }
                ),
                file="file",
                image_path="imagePath",
                output_path="outputPath",
                platform="platform",
                target_stage="targetStage"
            )
        '''
        if isinstance(cache_to, dict):
            cache_to = _DockerCacheOption_63aeecf2(**cache_to)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cbc49d837c4da70ff80dadf559d6f8c32fce03c67b454d4d81e625f4f39d8f3f)
            check_type(argname="argument build_args", value=build_args, expected_type=type_hints["build_args"])
            check_type(argname="argument cache_disabled", value=cache_disabled, expected_type=type_hints["cache_disabled"])
            check_type(argname="argument cache_from", value=cache_from, expected_type=type_hints["cache_from"])
            check_type(argname="argument cache_to", value=cache_to, expected_type=type_hints["cache_to"])
            check_type(argname="argument file", value=file, expected_type=type_hints["file"])
            check_type(argname="argument platform", value=platform, expected_type=type_hints["platform"])
            check_type(argname="argument target_stage", value=target_stage, expected_type=type_hints["target_stage"])
            check_type(argname="argument image_path", value=image_path, expected_type=type_hints["image_path"])
            check_type(argname="argument output_path", value=output_path, expected_type=type_hints["output_path"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if build_args is not None:
            self._values["build_args"] = build_args
        if cache_disabled is not None:
            self._values["cache_disabled"] = cache_disabled
        if cache_from is not None:
            self._values["cache_from"] = cache_from
        if cache_to is not None:
            self._values["cache_to"] = cache_to
        if file is not None:
            self._values["file"] = file
        if platform is not None:
            self._values["platform"] = platform
        if target_stage is not None:
            self._values["target_stage"] = target_stage
        if image_path is not None:
            self._values["image_path"] = image_path
        if output_path is not None:
            self._values["output_path"] = output_path

    @builtins.property
    def build_args(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Build args.

        :default: - no build args
        '''
        result = self._values.get("build_args")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def cache_disabled(self) -> typing.Optional[builtins.bool]:
        '''Disable the cache and pass ``--no-cache`` to the ``docker build`` command.

        :default: - cache is used
        '''
        result = self._values.get("cache_disabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cache_from(self) -> typing.Optional[typing.List[_DockerCacheOption_63aeecf2]]:
        '''Cache from options to pass to the ``docker build`` command.

        :default: - no cache from args are passed
        '''
        result = self._values.get("cache_from")
        return typing.cast(typing.Optional[typing.List[_DockerCacheOption_63aeecf2]], result)

    @builtins.property
    def cache_to(self) -> typing.Optional[_DockerCacheOption_63aeecf2]:
        '''Cache to options to pass to the ``docker build`` command.

        :default: - no cache to args are passed
        '''
        result = self._values.get("cache_to")
        return typing.cast(typing.Optional[_DockerCacheOption_63aeecf2], result)

    @builtins.property
    def file(self) -> typing.Optional[builtins.str]:
        '''Name of the Dockerfile, must relative to the docker build path.

        :default: ``Dockerfile``
        '''
        result = self._values.get("file")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def platform(self) -> typing.Optional[builtins.str]:
        '''Set platform if server is multi-platform capable. *Requires Docker Engine API v1.38+*.

        Example value: ``linux/amd64``

        :default: - no platform specified
        '''
        result = self._values.get("platform")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def target_stage(self) -> typing.Optional[builtins.str]:
        '''Set build target for multi-stage container builds. Any stage defined afterwards will be ignored.

        Example value: ``build-env``

        :default: - Build all stages defined in the Dockerfile
        '''
        result = self._values.get("target_stage")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def image_path(self) -> typing.Optional[builtins.str]:
        '''The path in the Docker image where the asset is located after the build operation.

        :default: /asset
        '''
        result = self._values.get("image_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''The path on the local filesystem where the asset will be copied using ``docker cp``.

        :default: - a unique temporary directory in the system temp directory
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerBuildAssetOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DockerImageCode(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_lambda.DockerImageCode",
):
    '''Code property for the DockerImageFunction construct.

    :exampleMetadata: infused

    Example::

        lambda_.DockerImageFunction(self, "AssetFunction",
            code=lambda_.DockerImageCode.from_image_asset(path.join(__dirname, "docker-handler"))
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromEcr")
    @builtins.classmethod
    def from_ecr(
        cls,
        repository: _IRepository_e6004aa6,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        tag_or_digest: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> "DockerImageCode":
        '''Use an existing ECR image as the Lambda code.

        :param repository: the ECR repository that the image is in.
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param tag: (deprecated) The image tag to use when pulling the image from ECR. Default: 'latest'
        :param tag_or_digest: The image tag or digest to use when pulling the image from ECR (digests must start with ``sha256:``). Default: 'latest'
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__49d2ed2c7107e57f257881a35f5acf33e118b0e47d994d7896c1db4615f68437)
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
        props = EcrImageCodeProps(
            cmd=cmd,
            entrypoint=entrypoint,
            tag=tag,
            tag_or_digest=tag_or_digest,
            working_directory=working_directory,
        )

        return typing.cast("DockerImageCode", jsii.sinvoke(cls, "fromEcr", [repository, props]))

    @jsii.member(jsii_name="fromImageAsset")
    @builtins.classmethod
    def from_image_asset(
        cls,
        directory: builtins.str,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        working_directory: typing.Optional[builtins.str] = None,
        asset_name: typing.Optional[builtins.str] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        build_secrets: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        build_ssh: typing.Optional[builtins.str] = None,
        cache_disabled: typing.Optional[builtins.bool] = None,
        cache_from: typing.Optional[typing.Sequence[typing.Union[_DockerCacheOption_58ef18ca, typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_to: typing.Optional[typing.Union[_DockerCacheOption_58ef18ca, typing.Dict[builtins.str, typing.Any]]] = None,
        display_name: typing.Optional[builtins.str] = None,
        file: typing.Optional[builtins.str] = None,
        invalidation: typing.Optional[typing.Union[_DockerImageAssetInvalidationOptions_4deb8d45, typing.Dict[builtins.str, typing.Any]]] = None,
        network_mode: typing.Optional[_NetworkMode_897e5081] = None,
        outputs: typing.Optional[typing.Sequence[builtins.str]] = None,
        platform: typing.Optional[_Platform_d16f3cf1] = None,
        target: typing.Optional[builtins.str] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_047ec1f6] = None,
        ignore_mode: typing.Optional[_IgnoreMode_655a98e8] = None,
    ) -> "DockerImageCode":
        '''Create an ECR image from the specified asset and bind it as the Lambda code.

        :param directory: the directory from which the asset must be created.
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.
        :param asset_name: Unique identifier of the docker image asset and its potential revisions. Required if using AppScopedStagingSynthesizer. Default: - no asset name
        :param build_args: Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param build_secrets: Build secrets. Docker BuildKit must be enabled to use build secrets. Default: - no build secrets
        :param build_ssh: SSH agent socket or keys to pass to the ``docker build`` command. Docker BuildKit must be enabled to use the ssh flag Default: - no --ssh flag
        :param cache_disabled: Disable the cache and pass ``--no-cache`` to the ``docker build`` command. Default: - cache is used
        :param cache_from: Cache from options to pass to the ``docker build`` command. Default: - no cache from options are passed to the build command
        :param cache_to: Cache to options to pass to the ``docker build`` command. Default: - no cache to options are passed to the build command
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. If the same asset is added multiple times, the display name of the first occurrence is used. If ``assetName`` is given, it will also be used as the default ``displayName``. Otherwise, the default is the construct path of the ImageAsset construct, with respect to the enclosing stack. If the asset is produced by a construct helper function (such as ``lambda.Code.fromAssetImage()``), this will look like ``MyFunction/AssetImage``. We use the stack-relative construct path so that in the common case where you have multiple stacks with the same asset, we won't show something like ``/MyBetaStack/MyFunction/Code`` when you are actually deploying to production. Default: - Stack-relative construct path
        :param file: Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param invalidation: Options to control which parameters are used to invalidate the asset hash. Default: - hash all parameters
        :param network_mode: Networking mode for the RUN commands during build. Support docker API 1.25+. Default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        :param outputs: Outputs to pass to the ``docker build`` command. Default: - no outputs are passed to the build command (default outputs are used)
        :param platform: Platform to build for. *Requires Docker Buildx*. Default: - no platform specified (the current machine architecture will be used)
        :param target: Docker target to build to. Default: - no target
        :param extra_hash: Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5da1bfa983f51013181a622617f984d1361ab84bd63c101a85256915abb15b63)
            check_type(argname="argument directory", value=directory, expected_type=type_hints["directory"])
        props = AssetImageCodeProps(
            cmd=cmd,
            entrypoint=entrypoint,
            working_directory=working_directory,
            asset_name=asset_name,
            build_args=build_args,
            build_secrets=build_secrets,
            build_ssh=build_ssh,
            cache_disabled=cache_disabled,
            cache_from=cache_from,
            cache_to=cache_to,
            display_name=display_name,
            file=file,
            invalidation=invalidation,
            network_mode=network_mode,
            outputs=outputs,
            platform=platform,
            target=target,
            extra_hash=extra_hash,
            exclude=exclude,
            follow_symlinks=follow_symlinks,
            ignore_mode=ignore_mode,
        )

        return typing.cast("DockerImageCode", jsii.sinvoke(cls, "fromImageAsset", [directory, props]))


class _DockerImageCodeProxy(DockerImageCode):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, DockerImageCode).__jsii_proxy_class__ = lambda : _DockerImageCodeProxy


class EcrImageCode(
    Code,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.EcrImageCode",
):
    '''Represents a Docker image in ECR that can be bound as Lambda Code.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ecr as ecr
        from aws_cdk import aws_lambda as lambda_
        
        # repository: ecr.Repository
        
        ecr_image_code = lambda_.EcrImageCode(repository,
            cmd=["cmd"],
            entrypoint=["entrypoint"],
            tag="tag",
            tag_or_digest="tagOrDigest",
            working_directory="workingDirectory"
        )
    '''

    def __init__(
        self,
        repository: _IRepository_e6004aa6,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        tag_or_digest: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param repository: -
        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param tag: (deprecated) The image tag to use when pulling the image from ECR. Default: 'latest'
        :param tag_or_digest: The image tag or digest to use when pulling the image from ECR (digests must start with ``sha256:``). Default: 'latest'
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__691e176891c7595692ae5e2b0e727d9f4d5fe9a231772407da646e0bd81ecb1e)
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
        props = EcrImageCodeProps(
            cmd=cmd,
            entrypoint=entrypoint,
            tag=tag,
            tag_or_digest=tag_or_digest,
            working_directory=working_directory,
        )

        jsii.create(self.__class__, self, [repository, props])

    @jsii.member(jsii_name="bind")
    def bind(self, _scope: _constructs_77d1e7e8.Construct) -> CodeConfig:
        '''Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.

        :param _scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b5869e46cce31df5131a9eeeb1bfdc90c539d66df725b2c91d1710fa524b0c2f)
            check_type(argname="argument _scope", value=_scope, expected_type=type_hints["_scope"])
        return typing.cast(CodeConfig, jsii.invoke(self, "bind", [_scope]))

    @builtins.property
    @jsii.member(jsii_name="isInline")
    def is_inline(self) -> builtins.bool:
        '''Determines whether this Code is inline code or not.'''
        return typing.cast(builtins.bool, jsii.get(self, "isInline"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.EcrImageCodeProps",
    jsii_struct_bases=[],
    name_mapping={
        "cmd": "cmd",
        "entrypoint": "entrypoint",
        "tag": "tag",
        "tag_or_digest": "tagOrDigest",
        "working_directory": "workingDirectory",
    },
)
class EcrImageCodeProps:
    def __init__(
        self,
        *,
        cmd: typing.Optional[typing.Sequence[builtins.str]] = None,
        entrypoint: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        tag_or_digest: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties to initialize a new EcrImageCode.

        :param cmd: Specify or override the CMD on the specified Docker image or Dockerfile. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the CMD specified in the docker image or Dockerfile.
        :param entrypoint: Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile. An ENTRYPOINT allows you to configure a container that will run as an executable. This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``. Default: - use the ENTRYPOINT in the docker image or Dockerfile.
        :param tag: (deprecated) The image tag to use when pulling the image from ECR. Default: 'latest'
        :param tag_or_digest: The image tag or digest to use when pulling the image from ECR (digests must start with ``sha256:``). Default: 'latest'
        :param working_directory: Specify or override the WORKDIR on the specified Docker image or Dockerfile. A WORKDIR allows you to configure the working directory the container will use. Default: - use the WORKDIR in the docker image or Dockerfile.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            ecr_image_code_props = lambda.EcrImageCodeProps(
                cmd=["cmd"],
                entrypoint=["entrypoint"],
                tag="tag",
                tag_or_digest="tagOrDigest",
                working_directory="workingDirectory"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a6faedf61560761732112d2df63efe5f1c0ad7caa19ea2c4ffa2f9ada810ab71)
            check_type(argname="argument cmd", value=cmd, expected_type=type_hints["cmd"])
            check_type(argname="argument entrypoint", value=entrypoint, expected_type=type_hints["entrypoint"])
            check_type(argname="argument tag", value=tag, expected_type=type_hints["tag"])
            check_type(argname="argument tag_or_digest", value=tag_or_digest, expected_type=type_hints["tag_or_digest"])
            check_type(argname="argument working_directory", value=working_directory, expected_type=type_hints["working_directory"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cmd is not None:
            self._values["cmd"] = cmd
        if entrypoint is not None:
            self._values["entrypoint"] = entrypoint
        if tag is not None:
            self._values["tag"] = tag
        if tag_or_digest is not None:
            self._values["tag_or_digest"] = tag_or_digest
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def cmd(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the CMD on the specified Docker image or Dockerfile.

        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the CMD specified in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#cmd
        '''
        result = self._values.get("cmd")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entrypoint(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify or override the ENTRYPOINT on the specified Docker image or Dockerfile.

        An ENTRYPOINT allows you to configure a container that will run as an executable.
        This needs to be in the 'exec form', viz., ``[ 'executable', 'param1', 'param2' ]``.

        :default: - use the ENTRYPOINT in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        '''
        result = self._values.get("entrypoint")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tag(self) -> typing.Optional[builtins.str]:
        '''(deprecated) The image tag to use when pulling the image from ECR.

        :default: 'latest'

        :deprecated: use ``tagOrDigest``

        :stability: deprecated
        '''
        result = self._values.get("tag")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tag_or_digest(self) -> typing.Optional[builtins.str]:
        '''The image tag or digest to use when pulling the image from ECR (digests must start with ``sha256:``).

        :default: 'latest'
        '''
        result = self._values.get("tag_or_digest")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''Specify or override the WORKDIR on the specified Docker image or Dockerfile.

        A WORKDIR allows you to configure the working directory the container will use.

        :default: - use the WORKDIR in the docker image or Dockerfile.

        :see: https://docs.docker.com/engine/reference/builder/#workdir
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcrImageCodeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.EnvironmentOptions",
    jsii_struct_bases=[],
    name_mapping={"remove_in_edge": "removeInEdge"},
)
class EnvironmentOptions:
    def __init__(
        self,
        *,
        remove_in_edge: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Environment variables options.

        :param remove_in_edge: When used in Lambda@Edge via edgeArn() API, these environment variables will be removed. If not set, an error will be thrown. Default: false - using the function in Lambda@Edge will throw

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_lambda as lambda_
            
            environment_options = lambda.EnvironmentOptions(
                remove_in_edge=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__80ff3b5e3e4ebb34b3e749702628127b8b9d9d03cde362da9805a14d5e5c3c42)
            check_type(argname="argument remove_in_edge", value=remove_in_edge, expected_type=type_hints["remove_in_edge"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if remove_in_edge is not None:
            self._values["remove_in_edge"] = remove_in_edge

    @builtins.property
    def remove_in_edge(self) -> typing.Optional[builtins.bool]:
        '''When used in Lambda@Edge via edgeArn() API, these environment variables will be removed.

        If not set, an error will be thrown.

        :default: false - using the function in Lambda@Edge will throw

        :see: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-requirements-limits.html#lambda-requirements-lambda-function-configuration
        '''
        result = self._values.get("remove_in_edge")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EnvironmentOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EventInvokeConfig(
    _Resource_45bc6135,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.EventInvokeConfig",
):
    '''Configure options for asynchronous invocation on a version or an alias.

    By default, Lambda retries an asynchronous invocation twice if the function
    returns an error. It retains events in a queue for up to six hours. When an
    event fails all processing attempts or stays in the asynchronous invocation
    queue for too long, Lambda discards it.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        from aws_cdk import aws_lambda as lambda_
        
        # destination: lambda.IDestination
        # function_: lambda.Function
        
        event_invoke_config = lambda_.EventInvokeConfig(self, "MyEventInvokeConfig",
            function=function_,
        
            # the properties below are optional
            max_event_age=cdk.Duration.minutes(30),
            on_failure=destination,
            on_success=destination,
            qualifier="qualifier",
            retry_attempts=123
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        function: "IFunction",
        qualifier: typing.Optional[builtins.str] = None,
        max_event_age: typing.Optional[_Duration_4839e8c3] = None,
        on_failure: typing.Optional["IDestination"] = None,
        on_success: typing.Optional["IDestination"] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param function: The Lambda function.
        :param qualifier: The qualifier. Default: - latest version
        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e8bc902c1cc378370c0245054b299aa1405e2965c0bf089fd0f11e6a024a95b6)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = EventInvokeConfigProps(
            function=function,
            qualifier=qualifier,
            max_event_age=max_event_age,
            on_failure=on_failure,
            on_success=on_success,
            retry_attempts=retry_attempts,
        )

        jsii.create(self.__class__, self, [scope, id, props])


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.EventInvokeConfigOptions",
    jsii_struct_bases=[],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
    },
)
class EventInvokeConfigOptions:
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_Duration_4839e8c3] = None,
        on_failure: typing.Optional["IDestination"] = None,
        on_success: typing.Optional["IDestination"] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Options to add an EventInvokeConfig to a function.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_lambda as lambda_
            
            # destination: lambda.IDestination
            
            event_invoke_config_options = lambda.EventInvokeConfigOptions(
                max_event_age=cdk.Duration.minutes(30),
                on_failure=destination,
                on_success=destination,
                retry_attempts=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__292fde1c6ea1265f93e98915c7beb8aeb5dd1149de2b603c377b2018f123b4a0)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts

    @builtins.property
    def max_event_age(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IDestination"]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def on_success(self) -> typing.Optional["IDestination"]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EventInvokeConfigOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.EventInvokeConfigProps",
    jsii_struct_bases=[EventInvokeConfigOptions],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
        "function": "function",
        "qualifier": "qualifier",
    },
)
class EventInvokeConfigProps(EventInvokeConfigOptions):
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_Duration_4839e8c3] = None,
        on_failure: typing.Optional["IDestination"] = None,
        on_success: typing.Optional["IDestination"] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        function: "IFunction",
        qualifier: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for an EventInvokeConfig.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param function: The Lambda function.
        :param qualifier: The qualifier. Default: - latest version

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_lambda as lambda_
            
            # destination: lambda.IDestination
            # function_: lambda.Function
            
            event_invoke_config_props = lambda.EventInvokeConfigProps(
                function=function_,
            
                # the properties below are optional
                max_event_age=cdk.Duration.minutes(30),
                on_failure=destination,
                on_success=destination,
                qualifier="qualifier",
                retry_attempts=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__39c1b3a3185624e50b391ccba8992ee288e54789a9b063a9fbffc320735abc4b)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument function", value=function, expected_type=type_hints["function"])
            check_type(argname="argument qualifier", value=qualifier, expected_type=type_hints["qualifier"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function": function,
        }
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if qualifier is not None:
            self._values["qualifier"] = qualifier

    @builtins.property
    def max_event_age(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IDestination"]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def on_success(self) -> typing.Optional["IDestination"]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def function(self) -> "IFunction":
        '''The Lambda function.'''
        result = self._values.get("function")
        assert result is not None, "Required property 'function' is missing"
        return typing.cast("IFunction", result)

    @builtins.property
    def qualifier(self) -> typing.Optional[builtins.str]:
        '''The qualifier.

        :default: - latest version
        '''
        result = self._values.get("qualifier")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EventInvokeConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.EventSourceMappingOptions",
    jsii_struct_bases=[],
    name_mapping={
        "batch_size": "batchSize",
        "bisect_batch_on_error": "bisectBatchOnError",
        "enabled": "enabled",
        "event_source_arn": "eventSourceArn",
        "filter_encryption": "filterEncryption",
        "filters": "filters",
        "kafka_bootstrap_servers": "kafkaBootstrapServers",
        "kafka_consumer_group_id": "kafkaConsumerGroupId",
        "kafka_topic": "kafkaTopic",
        "max_batching_window": "maxBatchingWindow",
        "max_concurrency": "maxConcurrency",
        "max_record_age": "maxRecordAge",
        "metrics_config": "metricsConfig",
        "on_failure": "onFailure",
        "parallelization_factor": "parallelizationFactor",
        "provisioned_poller_config": "provisionedPollerConfig",
        "report_batch_item_failures": "reportBatchItemFailures",
        "retry_attempts": "retryAttempts",
        "source_access_configurations": "sourceAccessConfigurations",
        "starting_position": "startingPosition",
        "starting_position_timestamp": "startingPositionTimestamp",
        "support_s3_on_failure_destination": "supportS3OnFailureDestination",
        "tumbling_window": "tumblingWindow",
    },
)
class EventSourceMappingOptions:
    def __init__(
        self,
        *,
        batch_size: typing.Optional[jsii.Number] = None,
        bisect_batch_on_error: typing.Optional[builtins.bool] = None,
        enabled: typing.Optional[builtins.bool] = None,
        event_source_arn: typing.Optional[builtins.str] = None,
        filter_encryption: typing.Optional[_IKey_5f11635f] = None,
        filters: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        kafka_bootstrap_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        kafka_consumer_group_id: typing.Optional[builtins.str] = None,
        kafka_topic: typing.Optional[builtins.str] = None,
        max_batching_window: typing.Optional[_Duration_4839e8c3] = None,
        max_concurrency: typing.Optional[jsii.Number] = None,
        max_record_age: typing.Optional[_Duration_4839e8c3] = None,
        metrics_config: typing.Optional[typing.Union["MetricsConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        on_failure: typing.Optional["IEventSourceDlq"] = None,
        parallelization_factor: typing.Optional[jsii.Number] = None,
        provisioned_poller_config: typing.Optional[typing.Union["ProvisionedPollerConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        report_batch_item_failures: typing.Optional[builtins.bool] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        source_access_configurations: typing.Optional[typing.Sequence[typing.Union["SourceAccessConfiguration", typing.Dict[builtins.str, typing.Any]]]] = None,
        starting_position: typing.Optional["StartingPosition"] = None,
        starting_position_timestamp: typing.Optional[jsii.Number] = None,
        support_s3_on_failure_destination: typing.Optional[builtins.bool] = None,
        tumbling_window: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param batch_size: The largest number of records that AWS Lambda will retrieve from your event source at the time of invoking your function. Your function receives an event with all the retrieved records. Valid Range: Minimum value of 1. Maximum value of 10000. Default: - Amazon Kinesis, Amazon DynamoDB, and Amazon MSK is 100 records. The default for Amazon SQS is 10 messages. For standard SQS queues, the maximum is 10,000. For FIFO SQS queues, the maximum is 10.
        :param bisect_batch_on_error: If the function returns an error, split the batch in two and retry. Default: false
        :param enabled: Set to false to disable the event source upon creation. Default: true
        :param event_source_arn: The Amazon Resource Name (ARN) of the event source. Any record added to this stream can invoke the Lambda function. Default: - not set if using a self managed Kafka cluster, throws an error otherwise
        :param filter_encryption: Add Customer managed KMS key to encrypt Filter Criteria. Default: - none
        :param filters: Add filter criteria to Event Source. Default: - none
        :param kafka_bootstrap_servers: A list of host and port pairs that are the addresses of the Kafka brokers in a self managed "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself. They are in the format ``abc.example.com:9096``. Default: - none
        :param kafka_consumer_group_id: The identifier for the Kafka consumer group to join. The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. The value must have a length between 1 and 200 and full the pattern '[a-zA-Z0-9-/*:_+=.@-]*'. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_. Default: - none
        :param kafka_topic: The name of the Kafka topic. Default: - no topic
        :param max_batching_window: The maximum amount of time to gather records before invoking the function. Maximum of Duration.minutes(5) Default: Duration.seconds(0)
        :param max_concurrency: The maximum concurrency setting limits the number of concurrent instances of the function that an Amazon SQS event source can invoke. Default: - No specific limit.
        :param max_record_age: The maximum age of a record that Lambda sends to a function for processing. Valid Range: - Minimum value of 60 seconds - Maximum value of 7 days Default: - infinite or until the record expires.
        :param metrics_config: Configuration for enhanced monitoring metrics collection When specified, enables collection of additional metrics for the stream event source. Default: - Enhanced monitoring is disabled
        :param on_failure: An Amazon S3, Amazon SQS queue or Amazon SNS topic destination for discarded records. Default: discarded records are ignored
        :param parallelization_factor: The number of batches to process from each shard concurrently. Valid Range: - Minimum value of 1 - Maximum value of 10 Default: 1
        :param provisioned_poller_config: Configuration for provisioned pollers that read from the event source. When specified, allows control over the minimum and maximum number of pollers that can be provisioned to process events from the source. Default: - no provisioned pollers
        :param report_batch_item_failures: Allow functions to return partially successful responses for a batch of records. Default: false
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Set to ``undefined`` if you want lambda to keep retrying infinitely or until the record expires. Valid Range: - Minimum value of 0 - Maximum value of 10000 Default: - infinite or until the record expires.
        :param source_access_configurations: Specific settings like the authentication protocol or the VPC components to secure access to your event source. Default: - none
        :param starting_position: The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading. Default: - no starting position
        :param starting_position_timestamp: The time from which to start reading, in Unix time seconds. Default: - no timestamp
        :param support_s3_on_failure_destination: Check if support S3 onfailure destination(OFD). Kinesis, DynamoDB, MSK and self managed kafka event support S3 OFD Default: false
        :param tumbling_window: The size of the tumbling windows to group records sent to DynamoDB or Kinesis. Default: - None

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_kms as kms
            from aws_cdk import aws_lambda as lambda_
            
            # event_source_dlq: lambda.IEventSourceDlq
            # filters: Any
            # key: kms.Key
            # source_access_configuration_type: lambda.SourceAccessConfigurationType
            
            event_source_mapping_options = lambda.EventSourceMappingOptions(
                batch_size=123,
                bisect_batch_on_error=False,
                enabled=False,
                event_source_arn="eventSourceArn",
                filter_encryption=key,
                filters=[{
                    "filters_key": filters
                }],
                kafka_bootstrap_servers=["kafkaBootstrapServers"],
                kafka_consumer_group_id="kafkaConsumerGroupId",
                kafka_topic="kafkaTopic",
                max_batching_window=cdk.Duration.minutes(30),
                max_concurrency=123,
                max_record_age=cdk.Duration.minutes(30),
                metrics_config=lambda.MetricsConfig(
                    metrics=[lambda_.MetricType.EVENT_COUNT]
                ),
                on_failure=event_source_dlq,
                parallelization_factor=123,
                provisioned_poller_config=lambda.ProvisionedPollerConfig(
                    maximum_pollers=123,
                    minimum_pollers=123
                ),
                report_batch_item_failures=False,
                retry_attempts=123,
                source_access_configurations=[lambda.SourceAccessConfiguration(
                    type=source_access_configuration_type,
                    uri="uri"
                )],
                starting_position=lambda_.StartingPosition.TRIM_HORIZON,
                starting_position_timestamp=123,
                support_s3_on_failure_destination=False,
                tumbling_window=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(metrics_config, dict):
            metrics_config = MetricsConfig(**metrics_config)
        if isinstance(provisioned_poller_config, dict):
            provisioned_poller_config = ProvisionedPollerConfig(**provisioned_poller_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7442d2bd60e56a826eab54e95fa6a6ebc8961285a26558c7189840a124a0a2e0)
            check_type(argname="argument batch_size", value=batch_size, expected_type=type_hints["batch_size"])
            check_type(argname="argument bisect_batch_on_error", value=bisect_batch_on_error, expected_type=type_hints["bisect_batch_on_error"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument event_source_arn", value=event_source_arn, expected_type=type_hints["event_source_arn"])
            check_type(argname="argument filter_encryption", value=filter_encryption, expected_type=type_hints["filter_encryption"])
            check_type(argname="argument filters", value=filters, expected_type=type_hints["filters"])
            check_type(argname="argument kafka_bootstrap_servers", value=kafka_bootstrap_servers, expected_type=type_hints["kafka_bootstrap_servers"])
            check_type(argname="argument kafka_consumer_group_id", value=kafka_consumer_group_id, expected_type=type_hints["kafka_consumer_group_id"])
            check_type(argname="argument kafka_topic", value=kafka_topic, expected_type=type_hints["kafka_topic"])
            check_type(argname="argument max_batching_window", value=max_batching_window, expected_type=type_hints["max_batching_window"])
            check_type(argname="argument max_concurrency", value=max_concurrency, expected_type=type_hints["max_concurrency"])
            check_type(argname="argument max_record_age", value=max_record_age, expected_type=type_hints["max_record_age"])
            check_type(argname="argument metrics_config", value=metrics_config, expected_type=type_hints["metrics_config"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument parallelization_factor", value=parallelization_factor, expected_type=type_hints["parallelization_factor"])
            check_type(argname="argument provisioned_poller_config", value=provisioned_poller_config, expected_type=type_hints["provisioned_poller_config"])
            check_type(argname="argument report_batch_item_failures", value=report_batch_item_failures, expected_type=type_hints["report_batch_item_failures"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument source_access_configurations", value=source_access_configurations, expected_type=type_hints["source_access_configurations"])
            check_type(argname="argument starting_position", value=starting_position, expected_type=type_hints["starting_position"])
            check_type(argname="argument starting_position_timestamp", value=starting_position_timestamp, expected_type=type_hints["starting_position_timestamp"])
            check_type(argname="argument support_s3_on_failure_destination", value=support_s3_on_failure_destination, expected_type=type_hints["support_s3_on_failure_destination"])
            check_type(argname="argument tumbling_window", value=tumbling_window, expected_type=type_hints["tumbling_window"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if batch_size is not None:
            self._values["batch_size"] = batch_size
        if bisect_batch_on_error is not None:
            self._values["bisect_batch_on_error"] = bisect_batch_on_error
        if enabled is not None:
            self._values["enabled"] = enabled
        if event_source_arn is not None:
            self._values["event_source_arn"] = event_source_arn
        if filter_encryption is not None:
            self._values["filter_encryption"] = filter_encryption
        if filters is not None:
            self._values["filters"] = filters
        if kafka_bootstrap_servers is not None:
            self._values["kafka_bootstrap_servers"] = kafka_bootstrap_servers
        if kafka_consumer_group_id is not None:
            self._values["kafka_consumer_group_id"] = kafka_consumer_group_id
        if kafka_topic is not None:
            self._values["kafka_topic"] = kafka_topic
        if max_batching_window is not None:
            self._values["max_batching_window"] = max_batching_window
        if max_concurrency is not None:
            self._values["max_concurrency"] = max_concurrency
        if max_record_age is not None:
            self._values["max_record_age"] = max_record_age
        if metrics_config is not None:
            self._values["metrics_config"] = metrics_config
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if parallelization_factor is not None:
            self._values["parallelization_factor"] = parallelization_factor
        if provisioned_poller_config is not None:
            self._values["provisioned_poller_config"] = provisioned_poller_config
        if report_batch_item_failures is not None:
            self._values["report_batch_item_failures"] = report_batch_item_failures
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if source_access_configurations is not None:
            self._values["source_access_configurations"] = source_access_configurations
        if starting_position is not None:
            self._values["starting_position"] = starting_position
        if starting_position_timestamp is not None:
            self._values["starting_position_timestamp"] = starting_position_timestamp
        if support_s3_on_failure_destination is not None:
            self._values["support_s3_on_failure_destination"] = support_s3_on_failure_destination
        if tumbling_window is not None:
            self._values["tumbling_window"] = tumbling_window

    @builtins.property
    def batch_size(self) -> typing.Optional[jsii.Number]:
        '''The largest number of records that AWS Lambda will retrieve from your event source at the time of invoking your function.

        Your function receives an
        event with all the retrieved records.

        Valid Range: Minimum value of 1. Maximum value of 10000.

        :default:

        - Amazon Kinesis, Amazon DynamoDB, and Amazon MSK is 100 records.
        The default for Amazon SQS is 10 messages. For standard SQS queues, the maximum is 10,000. For FIFO SQS queues, the maximum is 10.
        '''
        result = self._values.get("batch_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def bisect_batch_on_error(self) -> typing.Optional[builtins.bool]:
        '''If the function returns an error, split the batch in two and retry.

        :default: false
        '''
        result = self._values.get("bisect_batch_on_error")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enabled(self) -> typing.Optional[builtins.bool]:
        '''Set to false to disable the event source upon creation.

        :default: true
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def event_source_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the event source.

        Any record added to
        this stream can invoke the Lambda function.

        :default: - not set if using a self managed Kafka cluster, throws an error otherwise
        '''
        result = self._values.get("event_source_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def filter_encryption(self) -> typing.Optional[_IKey_5f11635f]:
        '''Add Customer managed KMS key to encrypt Filter Criteria.

        :default: - none

        :see: https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-managed-cmk
        '''
        result = self._values.get("filter_encryption")
        return typing.cast(typing.Optional[_IKey_5f11635f], result)

    @builtins.property
    def filters(
        self,
    ) -> typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]]:
        '''Add filter criteria to Event Source.

        :default: - none

        :see: https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html
        '''
        result = self._values.get("filters")
        return typing.cast(typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]], result)

    @builtins.property
    def kafka_bootstrap_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of host and port pairs that are the addresses of the Kafka brokers in a self managed "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself.

        They are in the format ``abc.example.com:9096``.

        :default: - none
        '''
        result = self._values.get("kafka_bootstrap_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def kafka_consumer_group_id(self) -> typing.Optional[builtins.str]:
        '''The identifier for the Kafka consumer group to join.

        The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. The value must have a length between 1 and 200 and full the pattern '[a-zA-Z0-9-/*:_+=.@-]*'. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_.

        :default: - none

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-selfmanagedkafkaeventsourceconfig.html
        '''
        result = self._values.get("kafka_consumer_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def kafka_topic(self) -> typing.Optional[builtins.str]:
        '''The name of the Kafka topic.

        :default: - no topic
        '''
        result = self._values.get("kafka_topic")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_batching_window(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum amount of time to gather records before invoking the function.

        Maximum of Duration.minutes(5)

        :default: Duration.seconds(0)
        '''
        result = self._values.get("max_batching_window")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def max_concurrency(self) -> typing.Optional[jsii.Number]:
        '''The maximum concurrency setting limits the number of concurrent instances of the function that an Amazon SQS event source can invoke.

        :default: - No specific limit.

        :see:

        https://docs.aws.amazon.com/lambda/latest/dg/with-sqs.html#events-sqs-max-concurrency

        Valid Range: Minimum value of 2. Maximum value of 1000.
        '''
        result = self._values.get("max_concurrency")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_record_age(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum age of a record that Lambda sends to a function for processing.

        Valid Range:

        - Minimum value of 60 seconds
        - Maximum value of 7 days

        :default: - infinite or until the record expires.
        '''
        result = self._values.get("max_record_age")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def metrics_config(self) -> typing.Optional["MetricsConfig"]:
        '''Configuration for enhanced monitoring metrics collection When specified, enables collection of additional metrics for the stream event source.

        :default: - Enhanced monitoring is disabled
        '''
        result = self._values.get("metrics_config")
        return typing.cast(typing.Optional["MetricsConfig"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IEventSourceDlq"]:
        '''An Amazon S3, Amazon SQS queue or Amazon SNS topic destination for discarded records.

        :default: discarded records are ignored
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IEventSourceDlq"], result)

    @builtins.property
    def parallelization_factor(self) -> typing.Optional[jsii.Number]:
        '''The number of batches to process from each shard concurrently.

        Valid Range:

        - Minimum value of 1
        - Maximum value of 10

        :default: 1
        '''
        result = self._values.get("parallelization_factor")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def provisioned_poller_config(self) -> typing.Optional["ProvisionedPollerConfig"]:
        '''Configuration for provisioned pollers that read from the event source.

        When specified, allows control over the minimum and maximum number of pollers
        that can be provisioned to process events from the source.

        :default: - no provisioned pollers
        '''
        result = self._values.get("provisioned_poller_config")
        return typing.cast(typing.Optional["ProvisionedPollerConfig"], result)

    @builtins.property
    def report_batch_item_failures(self) -> typing.Optional[builtins.bool]:
        '''Allow functions to return partially successful responses for a batch of records.

        :default: false

        :see: https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html#services-ddb-batchfailurereporting
        '''
        result = self._values.get("report_batch_item_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Set to ``undefined`` if you want lambda to keep retrying infinitely or until
        the record expires.

        Valid Range:

        - Minimum value of 0
        - Maximum value of 10000

        :default: - infinite or until the record expires.
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def source_access_configurations(
        self,
    ) -> typing.Optional[typing.List["SourceAccessConfiguration"]]:
        '''Specific settings like the authentication protocol or the VPC components to secure access to your event source.

        :default: - none

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-sourceaccessconfiguration.html
        '''
        result = self._values.get("source_access_configurations")
        return typing.cast(typing.Optional[typing.List["SourceAccessConfiguration"]], result)

    @builtins.property
    def starting_position(self) -> typing.Optional["StartingPosition"]:
        '''The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading.

        :default: - no starting position

        :see: https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetShardIterator.html#Kinesis-GetShardIterator-request-ShardIteratorType
        '''
        result = self._values.get("starting_position")
        return typing.cast(typing.Optional["StartingPosition"], result)

    @builtins.property
    def starting_position_timestamp(self) -> typing.Optional[jsii.Number]:
        '''The time from which to start reading, in Unix time seconds.

        :default: - no timestamp
        '''
        result = self._values.get("starting_position_timestamp")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def support_s3_on_failure_destination(self) -> typing.Optional[builtins.bool]:
        '''Check if support S3 onfailure destination(OFD).

        Kinesis, DynamoDB, MSK and self managed kafka event support S3 OFD

        :default: false
        '''
        result = self._values.get("support_s3_on_failure_destination")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def tumbling_window(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The size of the tumbling windows to group records sent to DynamoDB or Kinesis.

        :default: - None

        :see:

        https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html#services-ddb-windows

        Valid Range: 0 - 15 minutes
        '''
        result = self._values.get("tumbling_window")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EventSourceMappingOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.EventSourceMappingProps",
    jsii_struct_bases=[EventSourceMappingOptions],
    name_mapping={
        "batch_size": "batchSize",
        "bisect_batch_on_error": "bisectBatchOnError",
        "enabled": "enabled",
        "event_source_arn": "eventSourceArn",
        "filter_encryption": "filterEncryption",
        "filters": "filters",
        "kafka_bootstrap_servers": "kafkaBootstrapServers",
        "kafka_consumer_group_id": "kafkaConsumerGroupId",
        "kafka_topic": "kafkaTopic",
        "max_batching_window": "maxBatchingWindow",
        "max_concurrency": "maxConcurrency",
        "max_record_age": "maxRecordAge",
        "metrics_config": "metricsConfig",
        "on_failure": "onFailure",
        "parallelization_factor": "parallelizationFactor",
        "provisioned_poller_config": "provisionedPollerConfig",
        "report_batch_item_failures": "reportBatchItemFailures",
        "retry_attempts": "retryAttempts",
        "source_access_configurations": "sourceAccessConfigurations",
        "starting_position": "startingPosition",
        "starting_position_timestamp": "startingPositionTimestamp",
        "support_s3_on_failure_destination": "supportS3OnFailureDestination",
        "tumbling_window": "tumblingWindow",
        "target": "target",
    },
)
class EventSourceMappingProps(EventSourceMappingOptions):
    def __init__(
        self,
        *,
        batch_size: typing.Optional[jsii.Number] = None,
        bisect_batch_on_error: typing.Optional[builtins.bool] = None,
        enabled: typing.Optional[builtins.bool] = None,
        event_source_arn: typing.Optional[builtins.str] = None,
        filter_encryption: typing.Optional[_IKey_5f11635f] = None,
        filters: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        kafka_bootstrap_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        kafka_consumer_group_id: typing.Optional[builtins.str] = None,
        kafka_topic: typing.Optional[builtins.str] = None,
        max_batching_window: typing.Optional[_Duration_4839e8c3] = None,
        max_concurrency: typing.Optional[jsii.Number] = None,
        max_record_age: typing.Optional[_Duration_4839e8c3] = None,
        metrics_config: typing.Optional[typing.Union["MetricsConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        on_failure: typing.Optional["IEventSourceDlq"] = None,
        parallelization_factor: typing.Optional[jsii.Number] = None,
        provisioned_poller_config: typing.Optional[typing.Union["ProvisionedPollerConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        report_batch_item_failures: typing.Optional[builtins.bool] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        source_access_configurations: typing.Optional[typing.Sequence[typing.Union["SourceAccessConfiguration", typing.Dict[builtins.str, typing.Any]]]] = None,
        starting_position: typing.Optional["StartingPosition"] = None,
        starting_position_timestamp: typing.Optional[jsii.Number] = None,
        support_s3_on_failure_destination: typing.Optional[builtins.bool] = None,
        tumbling_window: typing.Optional[_Duration_4839e8c3] = None,
        target: "IFunction",
    ) -> None:
        '''Properties for declaring a new event source mapping.

        :param batch_size: The largest number of records that AWS Lambda will retrieve from your event source at the time of invoking your function. Your function receives an event with all the retrieved records. Valid Range: Minimum value of 1. Maximum value of 10000. Default: - Amazon Kinesis, Amazon DynamoDB, and Amazon MSK is 100 records. The default for Amazon SQS is 10 messages. For standard SQS queues, the maximum is 10,000. For FIFO SQS queues, the maximum is 10.
        :param bisect_batch_on_error: If the function returns an error, split the batch in two and retry. Default: false
        :param enabled: Set to false to disable the event source upon creation. Default: true
        :param event_source_arn: The Amazon Resource Name (ARN) of the event source. Any record added to this stream can invoke the Lambda function. Default: - not set if using a self managed Kafka cluster, throws an error otherwise
        :param filter_encryption: Add Customer managed KMS key to encrypt Filter Criteria. Default: - none
        :param filters: Add filter criteria to Event Source. Default: - none
        :param kafka_bootstrap_servers: A list of host and port pairs that are the addresses of the Kafka brokers in a self managed "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself. They are in the format ``abc.example.com:9096``. Default: - none
        :param kafka_consumer_group_id: The identifier for the Kafka consumer group to join. The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. The value must have a length between 1 and 200 and full the pattern '[a-zA-Z0-9-/*:_+=.@-]*'. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_. Default: - none
        :param kafka_topic: The name of the Kafka topic. Default: - no topic
        :param max_batching_window: The maximum amount of time to gather records before invoking the function. Maximum of Duration.minutes(5) Default: Duration.seconds(0)
        :param max_concurrency: The maximum concurrency setting limits the number of concurrent instances of the function that an Amazon SQS event source can invoke. Default: - No specific limit.
        :param max_record_age: The maximum age of a record that Lambda sends to a function for processing. Valid Range: - Minimum value of 60 seconds - Maximum value of 7 days Default: - infinite or until the record expires.
        :param metrics_config: Configuration for enhanced monitoring metrics collection When specified, enables collection of additional metrics for the stream event source. Default: - Enhanced monitoring is disabled
        :param on_failure: An Amazon S3, Amazon SQS queue or Amazon SNS topic destination for discarded records. Default: discarded records are ignored
        :param parallelization_factor: The number of batches to process from each shard concurrently. Valid Range: - Minimum value of 1 - Maximum value of 10 Default: 1
        :param provisioned_poller_config: Configuration for provisioned pollers that read from the event source. When specified, allows control over the minimum and maximum number of pollers that can be provisioned to process events from the source. Default: - no provisioned pollers
        :param report_batch_item_failures: Allow functions to return partially successful responses for a batch of records. Default: false
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Set to ``undefined`` if you want lambda to keep retrying infinitely or until the record expires. Valid Range: - Minimum value of 0 - Maximum value of 10000 Default: - infinite or until the record expires.
        :param source_access_configurations: Specific settings like the authentication protocol or the VPC components to secure access to your event source. Default: - none
        :param starting_position: The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading. Default: - no starting position
        :param starting_position_timestamp: The time from which to start reading, in Unix time seconds. Default: - no timestamp
        :param support_s3_on_failure_destination: Check if support S3 onfailure destination(OFD). Kinesis, DynamoDB, MSK and self managed kafka event support S3 OFD Default: false
        :param tumbling_window: The size of the tumbling windows to group records sent to DynamoDB or Kinesis. Default: - None
        :param target: The target AWS Lambda function.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_kms as kms
            from aws_cdk import aws_lambda as lambda_
            
            # event_source_dlq: lambda.IEventSourceDlq
            # filters: Any
            # function_: lambda.Function
            # key: kms.Key
            # source_access_configuration_type: lambda.SourceAccessConfigurationType
            
            event_source_mapping_props = lambda.EventSourceMappingProps(
                target=function_,
            
                # the properties below are optional
                batch_size=123,
                bisect_batch_on_error=False,
                enabled=False,
                event_source_arn="eventSourceArn",
                filter_encryption=key,
                filters=[{
                    "filters_key": filters
                }],
                kafka_bootstrap_servers=["kafkaBootstrapServers"],
                kafka_consumer_group_id="kafkaConsumerGroupId",
                kafka_topic="kafkaTopic",
                max_batching_window=cdk.Duration.minutes(30),
                max_concurrency=123,
                max_record_age=cdk.Duration.minutes(30),
                metrics_config=lambda.MetricsConfig(
                    metrics=[lambda_.MetricType.EVENT_COUNT]
                ),
                on_failure=event_source_dlq,
                parallelization_factor=123,
                provisioned_poller_config=lambda.ProvisionedPollerConfig(
                    maximum_pollers=123,
                    minimum_pollers=123
                ),
                report_batch_item_failures=False,
                retry_attempts=123,
                source_access_configurations=[lambda.SourceAccessConfiguration(
                    type=source_access_configuration_type,
                    uri="uri"
                )],
                starting_position=lambda_.StartingPosition.TRIM_HORIZON,
                starting_position_timestamp=123,
                support_s3_on_failure_destination=False,
                tumbling_window=cdk.Duration.minutes(30)
            )
        '''
        if isinstance(metrics_config, dict):
            metrics_config = MetricsConfig(**metrics_config)
        if isinstance(provisioned_poller_config, dict):
            provisioned_poller_config = ProvisionedPollerConfig(**provisioned_poller_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e74d0bc5516fc715f7302bdf199df23dddf769e98771f0bac2ff026a4e386fec)
            check_type(argname="argument batch_size", value=batch_size, expected_type=type_hints["batch_size"])
            check_type(argname="argument bisect_batch_on_error", value=bisect_batch_on_error, expected_type=type_hints["bisect_batch_on_error"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument event_source_arn", value=event_source_arn, expected_type=type_hints["event_source_arn"])
            check_type(argname="argument filter_encryption", value=filter_encryption, expected_type=type_hints["filter_encryption"])
            check_type(argname="argument filters", value=filters, expected_type=type_hints["filters"])
            check_type(argname="argument kafka_bootstrap_servers", value=kafka_bootstrap_servers, expected_type=type_hints["kafka_bootstrap_servers"])
            check_type(argname="argument kafka_consumer_group_id", value=kafka_consumer_group_id, expected_type=type_hints["kafka_consumer_group_id"])
            check_type(argname="argument kafka_topic", value=kafka_topic, expected_type=type_hints["kafka_topic"])
            check_type(argname="argument max_batching_window", value=max_batching_window, expected_type=type_hints["max_batching_window"])
            check_type(argname="argument max_concurrency", value=max_concurrency, expected_type=type_hints["max_concurrency"])
            check_type(argname="argument max_record_age", value=max_record_age, expected_type=type_hints["max_record_age"])
            check_type(argname="argument metrics_config", value=metrics_config, expected_type=type_hints["metrics_config"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument parallelization_factor", value=parallelization_factor, expected_type=type_hints["parallelization_factor"])
            check_type(argname="argument provisioned_poller_config", value=provisioned_poller_config, expected_type=type_hints["provisioned_poller_config"])
            check_type(argname="argument report_batch_item_failures", value=report_batch_item_failures, expected_type=type_hints["report_batch_item_failures"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument source_access_configurations", value=source_access_configurations, expected_type=type_hints["source_access_configurations"])
            check_type(argname="argument starting_position", value=starting_position, expected_type=type_hints["starting_position"])
            check_type(argname="argument starting_position_timestamp", value=starting_position_timestamp, expected_type=type_hints["starting_position_timestamp"])
            check_type(argname="argument support_s3_on_failure_destination", value=support_s3_on_failure_destination, expected_type=type_hints["support_s3_on_failure_destination"])
            check_type(argname="argument tumbling_window", value=tumbling_window, expected_type=type_hints["tumbling_window"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "target": target,
        }
        if batch_size is not None:
            self._values["batch_size"] = batch_size
        if bisect_batch_on_error is not None:
            self._values["bisect_batch_on_error"] = bisect_batch_on_error
        if enabled is not None:
            self._values["enabled"] = enabled
        if event_source_arn is not None:
            self._values["event_source_arn"] = event_source_arn
        if filter_encryption is not None:
            self._values["filter_encryption"] = filter_encryption
        if filters is not None:
            self._values["filters"] = filters
        if kafka_bootstrap_servers is not None:
            self._values["kafka_bootstrap_servers"] = kafka_bootstrap_servers
        if kafka_consumer_group_id is not None:
            self._values["kafka_consumer_group_id"] = kafka_consumer_group_id
        if kafka_topic is not None:
            self._values["kafka_topic"] = kafka_topic
        if max_batching_window is not None:
            self._values["max_batching_window"] = max_batching_window
        if max_concurrency is not None:
            self._values["max_concurrency"] = max_concurrency
        if max_record_age is not None:
            self._values["max_record_age"] = max_record_age
        if metrics_config is not None:
            self._values["metrics_config"] = metrics_config
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if parallelization_factor is not None:
            self._values["parallelization_factor"] = parallelization_factor
        if provisioned_poller_config is not None:
            self._values["provisioned_poller_config"] = provisioned_poller_config
        if report_batch_item_failures is not None:
            self._values["report_batch_item_failures"] = report_batch_item_failures
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if source_access_configurations is not None:
            self._values["source_access_configurations"] = source_access_configurations
        if starting_position is not None:
            self._values["starting_position"] = starting_position
        if starting_position_timestamp is not None:
            self._values["starting_position_timestamp"] = starting_position_timestamp
        if support_s3_on_failure_destination is not None:
            self._values["support_s3_on_failure_destination"] = support_s3_on_failure_destination
        if tumbling_window is not None:
            self._values["tumbling_window"] = tumbling_window

    @builtins.property
    def batch_size(self) -> typing.Optional[jsii.Number]:
        '''The largest number of records that AWS Lambda will retrieve from your event source at the time of invoking your function.

        Your function receives an
        event with all the retrieved records.

        Valid Range: Minimum value of 1. Maximum value of 10000.

        :default:

        - Amazon Kinesis, Amazon DynamoDB, and Amazon MSK is 100 records.
        The default for Amazon SQS is 10 messages. For standard SQS queues, the maximum is 10,000. For FIFO SQS queues, the maximum is 10.
        '''
        result = self._values.get("batch_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def bisect_batch_on_error(self) -> typing.Optional[builtins.bool]:
        '''If the function returns an error, split the batch in two and retry.

        :default: false
        '''
        result = self._values.get("bisect_batch_on_error")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enabled(self) -> typing.Optional[builtins.bool]:
        '''Set to false to disable the event source upon creation.

        :default: true
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def event_source_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the event source.

        Any record added to
        this stream can invoke the Lambda function.

        :default: - not set if using a self managed Kafka cluster, throws an error otherwise
        '''
        result = self._values.get("event_source_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def filter_encryption(self) -> typing.Optional[_IKey_5f11635f]:
        '''Add Customer managed KMS key to encrypt Filter Criteria.

        :default: - none

        :see: https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-managed-cmk
        '''
        result = self._values.get("filter_encryption")
        return typing.cast(typing.Optional[_IKey_5f11635f], result)

    @builtins.property
    def filters(
        self,
    ) -> typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]]:
        '''Add filter criteria to Event Source.

        :default: - none

        :see: https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventfiltering.html
        '''
        result = self._values.get("filters")
        return typing.cast(typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]], result)

    @builtins.property
    def kafka_bootstrap_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of host and port pairs that are the addresses of the Kafka brokers in a self managed "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself.

        They are in the format ``abc.example.com:9096``.

        :default: - none
        '''
        result = self._values.get("kafka_bootstrap_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def kafka_consumer_group_id(self) -> typing.Optional[builtins.str]:
        '''The identifier for the Kafka consumer group to join.

        The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. The value must have a length between 1 and 200 and full the pattern '[a-zA-Z0-9-/*:_+=.@-]*'. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_.

        :default: - none

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-selfmanagedkafkaeventsourceconfig.html
        '''
        result = self._values.get("kafka_consumer_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def kafka_topic(self) -> typing.Optional[builtins.str]:
        '''The name of the Kafka topic.

        :default: - no topic
        '''
        result = self._values.get("kafka_topic")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_batching_window(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum amount of time to gather records before invoking the function.

        Maximum of Duration.minutes(5)

        :default: Duration.seconds(0)
        '''
        result = self._values.get("max_batching_window")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def max_concurrency(self) -> typing.Optional[jsii.Number]:
        '''The maximum concurrency setting limits the number of concurrent instances of the function that an Amazon SQS event source can invoke.

        :default: - No specific limit.

        :see:

        https://docs.aws.amazon.com/lambda/latest/dg/with-sqs.html#events-sqs-max-concurrency

        Valid Range: Minimum value of 2. Maximum value of 1000.
        '''
        result = self._values.get("max_concurrency")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_record_age(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum age of a record that Lambda sends to a function for processing.

        Valid Range:

        - Minimum value of 60 seconds
        - Maximum value of 7 days

        :default: - infinite or until the record expires.
        '''
        result = self._values.get("max_record_age")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def metrics_config(self) -> typing.Optional["MetricsConfig"]:
        '''Configuration for enhanced monitoring metrics collection When specified, enables collection of additional metrics for the stream event source.

        :default: - Enhanced monitoring is disabled
        '''
        result = self._values.get("metrics_config")
        return typing.cast(typing.Optional["MetricsConfig"], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IEventSourceDlq"]:
        '''An Amazon S3, Amazon SQS queue or Amazon SNS topic destination for discarded records.

        :default: discarded records are ignored
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IEventSourceDlq"], result)

    @builtins.property
    def parallelization_factor(self) -> typing.Optional[jsii.Number]:
        '''The number of batches to process from each shard concurrently.

        Valid Range:

        - Minimum value of 1
        - Maximum value of 10

        :default: 1
        '''
        result = self._values.get("parallelization_factor")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def provisioned_poller_config(self) -> typing.Optional["ProvisionedPollerConfig"]:
        '''Configuration for provisioned pollers that read from the event source.

        When specified, allows control over the minimum and maximum number of pollers
        that can be provisioned to process events from the source.

        :default: - no provisioned pollers
        '''
        result = self._values.get("provisioned_poller_config")
        return typing.cast(typing.Optional["ProvisionedPollerConfig"], result)

    @builtins.property
    def report_batch_item_failures(self) -> typing.Optional[builtins.bool]:
        '''Allow functions to return partially successful responses for a batch of records.

        :default: false

        :see: https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html#services-ddb-batchfailurereporting
        '''
        result = self._values.get("report_batch_item_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Set to ``undefined`` if you want lambda to keep retrying infinitely or until
        the record expires.

        Valid Range:

        - Minimum value of 0
        - Maximum value of 10000

        :default: - infinite or until the record expires.
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def source_access_configurations(
        self,
    ) -> typing.Optional[typing.List["SourceAccessConfiguration"]]:
        '''Specific settings like the authentication protocol or the VPC components to secure access to your event source.

        :default: - none

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-eventsourcemapping-sourceaccessconfiguration.html
        '''
        result = self._values.get("source_access_configurations")
        return typing.cast(typing.Optional[typing.List["SourceAccessConfiguration"]], result)

    @builtins.property
    def starting_position(self) -> typing.Optional["StartingPosition"]:
        '''The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading.

        :default: - no starting position

        :see: https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetShardIterator.html#Kinesis-GetShardIterator-request-ShardIteratorType
        '''
        result = self._values.get("starting_position")
        return typing.cast(typing.Optional["StartingPosition"], result)

    @builtins.property
    def starting_position_timestamp(self) -> typing.Optional[jsii.Number]:
        '''The time from which to start reading, in Unix time seconds.

        :default: - no timestamp
        '''
        result = self._values.get("starting_position_timestamp")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def support_s3_on_failure_destination(self) -> typing.Optional[builtins.bool]:
        '''Check if support S3 onfailure destination(OFD).

        Kinesis, DynamoDB, MSK and self managed kafka event support S3 OFD

        :default: false
        '''
        result = self._values.get("support_s3_on_failure_destination")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def tumbling_window(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The size of the tumbling windows to group records sent to DynamoDB or Kinesis.

        :default: - None

        :see:

        https://docs.aws.amazon.com/lambda/latest/dg/with-ddb.html#services-ddb-windows

        Valid Range: 0 - 15 minutes
        '''
        result = self._values.get("tumbling_window")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def target(self) -> "IFunction":
        '''The target AWS Lambda function.'''
        result = self._values.get("target")
        assert result is not None, "Required property 'target' is missing"
        return typing.cast("IFunction", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EventSourceMappingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class FileSystem(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.FileSystem",
):
    '''Represents the filesystem for the Lambda function.

    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_ec2 as ec2
        import aws_cdk.aws_efs as efs
        
        
        # create a new VPC
        vpc = ec2.Vpc(self, "VPC")
        
        # create a new Amazon EFS filesystem
        file_system = efs.FileSystem(self, "Efs", vpc=vpc)
        
        # create a new access point from the filesystem
        access_point = file_system.add_access_point("AccessPoint",
            # set /export/lambda as the root of the access point
            path="/export/lambda",
            # as /export/lambda does not exist in a new efs filesystem, the efs will create the directory with the following createAcl
            create_acl=efs.Acl(
                owner_uid="1001",
                owner_gid="1001",
                permissions="750"
            ),
            # enforce the POSIX identity so lambda function will access with this identity
            posix_user=efs.PosixUser(
                uid="1001",
                gid="1001"
            )
        )
        
        fn = lambda_.Function(self, "MyLambda",
            # mount the access point to /mnt/msg in the lambda runtime environment
            filesystem=lambda_.FileSystem.from_efs_access_point(access_point, "/mnt/msg"),
            runtime=lambda_.Runtime.NODEJS_18_X,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler")),
            vpc=vpc
        )
    '''

    def __init__(
        self,
        *,
        arn: builtins.str,
        local_mount_path: builtins.str,
        connections: typing.Optional[_Connections_0f31fce8] = None,
        dependency: typing.Optional[typing.Sequence[_constructs_77d1e7e8.IDependable]] = None,
        policies: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
    ) -> None:
        '''
        :param arn: ARN of the access point.
        :param local_mount_path: mount path in the lambda runtime environment.
        :param connections: connections object used to allow ingress traffic from lambda function. Default: - no connections required to add extra ingress rules for Lambda function
        :param dependency: array of IDependable that lambda function depends on. Default: - no dependency
        :param policies: additional IAM policies required for the lambda function. Default: - no additional policies required
        '''
        config = FileSystemConfig(
            arn=arn,
            local_mount_path=local_mount_path,
            connections=connections,
            dependency=dependency,
            policies=policies,
        )

        jsii.create(self.__class__, self, [config])

    @jsii.member(jsii_name="fromEfsAccessPoint")
    @builtins.classmethod
    def from_efs_access_point(
        cls,
        ap: _IAccessPoint_ce87b375,
        mount_path: builtins.str,
    ) -> "FileSystem":
        '''mount the filesystem from Amazon EFS.

        :param ap: the Amazon EFS access point.
        :param mount_path: the target path in the lambda runtime environment.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7962f6f4b398747cd2f496a8d711eb02f71c477dc2809a700efdcc9fa29e98af)
            check_type(argname="argument ap", value=ap, expected_type=type_hints["ap"])
            check_type(argname="argument mount_path", value=mount_path, expected_type=type_hints["mount_path"])
        return typing.cast("FileSystem", jsii.sinvoke(cls, "fromEfsAccessPoint", [ap, mount_path]))

    @builtins.property
    @jsii.member(jsii_name="config")
    def config(self) -> "FileSystemConfig":
        '''the FileSystem configurations for the Lambda function.'''
        return typing.cast("FileSystemConfig", jsii.get(self, "config"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.FileSystemConfig",
    jsii_struct_bases=[],
    name_mapping={
        "arn": "arn",
        "local_mount_path": "localMountPath",
        "connections": "connections",
        "dependency": "dependency",
        "policies": "policies",
    },
)
class FileSystemConfig:
    def __init__(
        self,
        *,
        arn: builtins.str,
        local_mount_path: builtins.str,
        connections: typing.Optional[_Connections_0f31fce8] = None,
        dependency: typing.Optional[typing.Sequence[_constructs_77d1e7e8.IDependable]] = None,
        policies: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
    ) -> None:
        '''FileSystem configurations for the Lambda function.

        :param arn: ARN of the access point.
        :param local_mount_path: mount path in the lambda runtime environment.
        :param connections: connections object used to allow ingress traffic from lambda function. Default: - no connections required to add extra ingress rules for Lambda function
        :param dependency: array of IDependable that lambda function depends on. Default: - no dependency
        :param policies: additional IAM policies required for the lambda function. Default: - no additional policies required

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_iam as iam
            from aws_cdk import aws_lambda as lambda_
            import constructs as constructs
            
            # connections: ec2.Connections
            # dependable: constructs.IDependable
            # policy_statement: iam.PolicyStatement
            
            file_system_config = lambda.FileSystemConfig(
                arn="arn",
                local_mount_path="localMountPath",
            
                # the properties below are optional
                connections=connections,
                dependency=[dependable],
                policies=[policy_statement]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f815a35cc45b35e0b2d8e3e27a1c080f57c8e3ec46cbfffc202f2790f84cfd1e)
            check_type(argname="argument arn", value=arn, expected_type=type_hints["arn"])
            check_type(argname="argument local_mount_path", value=local_mount_path, expected_type=type_hints["local_mount_path"])
            check_type(argname="argument connections", value=connections, expected_type=type_hints["connections"])
            check_type(argname="argument dependency", value=dependency, expected_type=type_hints["dependency"])
            check_type(argname="argument policies", value=policies, expected_type=type_hints["policies"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "arn": arn,
            "local_mount_path": local_mount_path,
        }
        if connections is not None:
            self._values["connections"] = connections
        if dependency is not None:
            self._values["dependency"] = dependency
        if policies is not None:
            self._values["policies"] = policies

    @builtins.property
    def arn(self) -> builtins.str:
        '''ARN of the access point.'''
        result = self._values.get("arn")
        assert result is not None, "Required property 'arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def local_mount_path(self) -> builtins.str:
        '''mount path in the lambda runtime environment.'''
        result = self._values.get("local_mount_path")
        assert result is not None, "Required property 'local_mount_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def connections(self) -> typing.Optional[_Connections_0f31fce8]:
        '''connections object used to allow ingress traffic from lambda function.

        :default: - no connections required to add extra ingress rules for Lambda function
        '''
        result = self._values.get("connections")
        return typing.cast(typing.Optional[_Connections_0f31fce8], result)

    @builtins.property
    def dependency(
        self,
    ) -> typing.Optional[typing.List[_constructs_77d1e7e8.IDependable]]:
        '''array of IDependable that lambda function depends on.

        :default: - no dependency
        '''
        result = self._values.get("dependency")
        return typing.cast(typing.Optional[typing.List[_constructs_77d1e7e8.IDependable]], result)

    @builtins.property
    def policies(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        '''additional IAM policies required for the lambda function.

        :default: - no additional policies required
        '''
        result = self._values.get("policies")
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FileSystemConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class FilterCriteria(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.FilterCriteria",
):
    '''Filter criteria for Lambda event filtering.

    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_dynamodb as dynamodb
        from aws_cdk.aws_lambda_event_sources import DynamoEventSource
        
        # table: dynamodb.Table
        
        # fn: lambda.Function
        
        fn.add_event_source(DynamoEventSource(table,
            starting_position=lambda_.StartingPosition.LATEST,
            filters=[
                lambda_.FilterCriteria.filter({
                    "event_name": lambda_.FilterRule.is_equal("INSERT"),
                    "dynamodb": {
                        "NewImage": {
                            "id": {"BOOL": lambda_.FilterRule.is_equal(True)}
                        }
                    }
                })
            ]
        ))
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="filter")
    @builtins.classmethod
    def filter(
        cls,
        filter: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''Filter for event source.

        :param filter: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2b818376fc240eac0f66f2fa2242db91912a7b53453f5035939625fc7aee74d6)
            check_type(argname="argument filter", value=filter, expected_type=type_hints["filter"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.sinvoke(cls, "filter", [filter]))


class FilterRule(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.FilterRule",
):
    '''Filter rules for Lambda event filtering.

    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_dynamodb as dynamodb
        from aws_cdk.aws_lambda_event_sources import DynamoEventSource
        
        # table: dynamodb.Table
        
        # fn: lambda.Function
        
        fn.add_event_source(DynamoEventSource(table,
            starting_position=lambda_.StartingPosition.LATEST,
            filters=[
                lambda_.FilterCriteria.filter({
                    "event_name": lambda_.FilterRule.is_equal("INSERT"),
                    "dynamodb": {
                        "NewImage": {
                            "id": {"BOOL": lambda_.FilterRule.is_equal(True)}
                        }
                    }
                })
            ]
        ))
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="beginsWith")
    @builtins.classmethod
    def begins_with(
        cls,
        elem: builtins.str,
    ) -> typing.List[typing.Mapping[builtins.str, builtins.str]]:
        '''Begins with comparison operator.

        :param elem: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e54c5c45849dc553b21bad8c311a5239b4658191778bbe62aca5fdf26b210360)
            check_type(argname="argument elem", value=elem, expected_type=type_hints["elem"])
        return typing.cast(typing.List[typing.Mapping[builtins.str, builtins.str]], jsii.sinvoke(cls, "beginsWith", [elem]))

    @jsii.member(jsii_name="between")
    @builtins.classmethod
    def between(
        cls,
        first: jsii.Number,
        second: jsii.Number,
    ) -> typing.List[typing.Mapping[builtins.str, typing.List[typing.Any]]]:
        '''Numeric range comparison operator.

        :param first: -
        :param second: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e532ccec0d2d2a11fd00b0da70142d367dca7a52eb785533bf0fe354fb1712ba)
            check_type(argname="argument first", value=first, expected_type=type_hints["first"])
            check_type(argname="argument second", value=second, expected_type=type_hints["second"])
        return typing.cast(typing.List[typing.Mapping[builtins.str, typing.List[typing.Any]]], jsii.sinvoke(cls, "between", [first, second]))

    @jsii.member(jsii_name="empty")
    @builtins.classmethod
    def empty(cls) -> typing.List[builtins.str]:
        '''Empty comparison operator.'''
        return typing.cast(typing.List[builtins.str], jsii.sinvoke(cls, "empty", []))

    @jsii.member(jsii_name="exists")
    @builtins.classmethod
    def exists(cls) -> typing.List[typing.Mapping[builtins.str, builtins.bool]]:
        '''Exists comparison operator.'''
        return typing.cast(typing.List[typing.Mapping[builtins.str, builtins.bool]], jsii.sinvoke(cls, "exists", []))

    @jsii.member(jsii_name="isEqual")
    @builtins.classmethod
    def is_equal(
        cls,
        item: typing.Union[builtins.str, jsii.Number, builtins.bool],
    ) -> typing.Any:
        '''Equals comparison operator.

        :param item: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6ffacf0ca301bfbc6a30c7fc32201cecba4e3530eb53e51d1b21c3349a4e5ba5)
            check_type(argname="argument item", value=item, expected_type=type_hints["item"])
        return typing.cast(typing.Any, jsii.sinvoke(cls, "isEqual", [item]))

    @jsii.member(jsii_name="notEquals")
    @builtins.classmethod
    def not_equals(
        cls,
        elem: builtins.str,
    ) -> typing.List[typing.Mapping[builtins.str, typing.List[builtins.str]]]:
        '''Not equals comparison operator.

        :param elem: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d06dc978dcba3d4d47c7e3a11eca8dfe2593851da932e3e0c1dcd0e5394ac817)
            check_type(argname="argument elem", value=elem, expected_type=type_hints["elem"])
        return typing.cast(typing.List[typing.Mapping[builtins.str, typing.List[builtins.str]]], jsii.sinvoke(cls, "notEquals", [elem]))

    @jsii.member(jsii_name="notExists")
    @builtins.classmethod
    def not_exists(cls) -> typing.List[typing.Mapping[builtins.str, builtins.bool]]:
        '''Not exists comparison operator.'''
        return typing.cast(typing.List[typing.Mapping[builtins.str, builtins.bool]], jsii.sinvoke(cls, "notExists", []))

    @jsii.member(jsii_name="null")
    @builtins.classmethod
    def null(cls) -> typing.Any:
        '''Null comparison operator.'''
        return typing.cast(typing.Any, jsii.sinvoke(cls, "null", []))

    @jsii.member(jsii_name="or")
    @builtins.classmethod
    def or_(cls, *elem: builtins.str) -> typing.List[builtins.str]:
        '''Or comparison operator.

        :param elem: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fec7655c4fb7536e55673dbcd07eddf9950bd4240435494d952707ad96585077)
            check_type(argname="argument elem", value=elem, expected_type=typing.Tuple[type_hints["elem"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast(typing.List[builtins.str], jsii.sinvoke(cls, "or", [*elem]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.FunctionAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "function_arn": "functionArn",
        "architecture": "architecture",
        "role": "role",
        "same_environment": "sameEnvironment",
        "security_group": "securityGroup",
        "skip_permissions": "skipPermissions",
    },
)
class FunctionAttributes:
    def __init__(
        self,
        *,
        function_arn: builtins.str,
        architecture: typing.Optional[Architecture] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
        same_environment: typing.Optional[builtins.bool] = None,
        security_group: typing.Optional[_ISecurityGroup_acf8a799] = None,
        skip_permissions: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Represents a Lambda function defined outside of this stack.

        :param function_arn: The ARN of the Lambda function. Format: arn::lambda:::function:
        :param architecture: The architecture of this Lambda Function (this is an optional attribute and defaults to X86_64). Default: - Architecture.X86_64
        :param role: The IAM execution role associated with this function. If the role is not specified, any role-related operations will no-op.
        :param same_environment: Setting this property informs the CDK that the imported function is in the same environment as the stack. This affects certain behaviours such as, whether this function's permission can be modified. When not configured, the CDK attempts to auto-determine this. For environment agnostic stacks, i.e., stacks where the account is not specified with the ``env`` property, this is determined to be false. Set this to property *ONLY IF* the imported function is in the same account as the stack it's imported in. Default: - depends: true, if the Stack is configured with an explicit ``env`` (account and region) and the account is the same as this function. For environment-agnostic stacks this will default to ``false``.
        :param security_group: The security group of this Lambda, if in a VPC. This needs to be given in order to support allowing connections to this Lambda.
        :param skip_permissions: Setting this property informs the CDK that the imported function ALREADY HAS the necessary permissions for what you are trying to do. When not configured, the CDK attempts to auto-determine whether or not additional permissions are necessary on the function when grant APIs are used. If the CDK tried to add permissions on an imported lambda, it will fail. Set this property *ONLY IF* you are committing to manage the imported function's permissions outside of CDK. You are acknowledging that your CDK code alone will have insufficient permissions to access the imported function. Default: false

        :exampleMetadata: infused

        Example::

            fn = lambda_.Function.from_function_attributes(self, "Function",
                function_arn="arn:aws:lambda:us-east-1:123456789012:function:MyFn",
                # The following are optional properties for specific use cases and should be used with caution:
            
                # Use Case: imported function is in the same account as the stack. This tells the CDK that it
                # can modify the function's permissions.
                same_environment=True,
            
                # Use Case: imported function is in a different account and user commits to ensuring that the
                # imported function has the correct permissions outside the CDK.
                skip_permissions=True
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__21b8745bfbc448de18ae3288cd649f0f58b46805fdde32ce01f134b33fa4e565)
            check_type(argname="argument function_arn", value=function_arn, expected_type=type_hints["function_arn"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument same_environment", value=same_environment, expected_type=type_hints["same_environment"])
            check_type(argname="argument security_group", value=security_group, expected_type=type_hints["security_group"])
            check_type(argname="argument skip_permissions", value=skip_permissions, expected_type=type_hints["skip_permissions"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_arn": function_arn,
        }
        if architecture is not None:
            self._values["architecture"] = architecture
        if role is not None:
            self._values["role"] = role
        if same_environment is not None:
            self._values["same_environment"] = same_environment
        if security_group is not None:
            self._values["security_group"] = security_group
        if skip_permissions is not None:
            self._values["skip_permissions"] = skip_permissions

    @builtins.property
    def function_arn(self) -> builtins.str:
        '''The ARN of the Lambda function.

        Format: arn::lambda:::function:
        '''
        result = self._values.get("function_arn")
        assert result is not None, "Required property 'function_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def architecture(self) -> typing.Optional[Architecture]:
        '''The architecture of this Lambda Function (this is an optional attribute and defaults to X86_64).

        :default: - Architecture.X86_64
        '''
        result = self._values.get("architecture")
        return typing.cast(typing.Optional[Architecture], result)

    @builtins.property
    def role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The IAM execution role associated with this function.

        If the role is not specified, any role-related operations will no-op.
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def same_environment(self) -> typing.Optional[builtins.bool]:
        '''Setting this property informs the CDK that the imported function is in the same environment as the stack.

        This affects certain behaviours such as, whether this function's permission can be modified.
        When not configured, the CDK attempts to auto-determine this. For environment agnostic stacks, i.e., stacks
        where the account is not specified with the ``env`` property, this is determined to be false.

        Set this to property *ONLY IF* the imported function is in the same account as the stack
        it's imported in.

        :default:

        - depends: true, if the Stack is configured with an explicit ``env`` (account and region) and the account is the same as this function.
        For environment-agnostic stacks this will default to ``false``.
        '''
        result = self._values.get("same_environment")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def security_group(self) -> typing.Optional[_ISecurityGroup_acf8a799]:
        '''The security group of this Lambda, if in a VPC.

        This needs to be given in order to support allowing connections
        to this Lambda.
        '''
        result = self._values.get("security_group")
        return typing.cast(typing.Optional[_ISecurityGroup_acf8a799], result)

    @builtins.property
    def skip_permissions(self) -> typing.Optional[builtins.bool]:
        '''Setting this property informs the CDK that the imported function ALREADY HAS the necessary permissions for what you are trying to do.

        When not configured, the CDK attempts to auto-determine whether or not
        additional permissions are necessary on the function when grant APIs are used. If the CDK tried to add
        permissions on an imported lambda, it will fail.

        Set this property *ONLY IF* you are committing to manage the imported function's permissions outside of
        CDK. You are acknowledging that your CDK code alone will have insufficient permissions to access the
        imported function.

        :default: false
        '''
        result = self._values.get("skip_permissions")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.FunctionOptions",
    jsii_struct_bases=[EventInvokeConfigOptions],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
        "adot_instrumentation": "adotInstrumentation",
        "allow_all_ipv6_outbound": "allowAllIpv6Outbound",
        "allow_all_outbound": "allowAllOutbound",
        "allow_public_subnet": "allowPublicSubnet",
        "application_log_level": "applicationLogLevel",
        "application_log_level_v2": "applicationLogLevelV2",
        "architecture": "architecture",
        "code_signing_config": "codeSigningConfig",
        "current_version_options": "currentVersionOptions",
        "dead_letter_queue": "deadLetterQueue",
        "dead_letter_queue_enabled": "deadLetterQueueEnabled",
        "dead_letter_topic": "deadLetterTopic",
        "description": "description",
        "environment": "environment",
        "environment_encryption": "environmentEncryption",
        "ephemeral_storage_size": "ephemeralStorageSize",
        "events": "events",
        "filesystem": "filesystem",
        "function_name": "functionName",
        "initial_policy": "initialPolicy",
        "insights_version": "insightsVersion",
        "ipv6_allowed_for_dual_stack": "ipv6AllowedForDualStack",
        "layers": "layers",
        "log_format": "logFormat",
        "logging_format": "loggingFormat",
        "log_group": "logGroup",
        "log_retention": "logRetention",
        "log_retention_retry_options": "logRetentionRetryOptions",
        "log_retention_role": "logRetentionRole",
        "memory_size": "memorySize",
        "params_and_secrets": "paramsAndSecrets",
        "profiling": "profiling",
        "profiling_group": "profilingGroup",
        "recursive_loop": "recursiveLoop",
        "reserved_concurrent_executions": "reservedConcurrentExecutions",
        "role": "role",
        "runtime_management_mode": "runtimeManagementMode",
        "security_groups": "securityGroups",
        "snap_start": "snapStart",
        "system_log_level": "systemLogLevel",
        "system_log_level_v2": "systemLogLevelV2",
        "timeout": "timeout",
        "tracing": "tracing",
        "vpc": "vpc",
        "vpc_subnets": "vpcSubnets",
    },
)
class FunctionOptions(EventInvokeConfigOptions):
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_Duration_4839e8c3] = None,
        on_failure: typing.Optional["IDestination"] = None,
        on_success: typing.Optional["IDestination"] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        adot_instrumentation: typing.Optional[typing.Union[AdotInstrumentationConfig, typing.Dict[builtins.str, typing.Any]]] = None,
        allow_all_ipv6_outbound: typing.Optional[builtins.bool] = None,
        allow_all_outbound: typing.Optional[builtins.bool] = None,
        allow_public_subnet: typing.Optional[builtins.bool] = None,
        application_log_level: typing.Optional[builtins.str] = None,
        application_log_level_v2: typing.Optional[ApplicationLogLevel] = None,
        architecture: typing.Optional[Architecture] = None,
        code_signing_config: typing.Optional["ICodeSigningConfig"] = None,
        current_version_options: typing.Optional[typing.Union["VersionOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        dead_letter_queue: typing.Optional[_IQueue_7ed6f679] = None,
        dead_letter_queue_enabled: typing.Optional[builtins.bool] = None,
        dead_letter_topic: typing.Optional[_ITopic_9eca4852] = None,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_encryption: typing.Optional[_IKey_5f11635f] = None,
        ephemeral_storage_size: typing.Optional[_Size_7b441c34] = None,
        events: typing.Optional[typing.Sequence["IEventSource"]] = None,
        filesystem: typing.Optional[FileSystem] = None,
        function_name: typing.Optional[builtins.str] = None,
        initial_policy: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        insights_version: typing.Optional["LambdaInsightsVersion"] = None,
        ipv6_allowed_for_dual_stack: typing.Optional[builtins.bool] = None,
        layers: typing.Optional[typing.Sequence["ILayerVersion"]] = None,
        log_format: typing.Optional[builtins.str] = None,
        logging_format: typing.Optional["LoggingFormat"] = None,
        log_group: typing.Optional[_ILogGroup_3c4fa718] = None,
        log_retention: typing.Optional[_RetentionDays_070f99f0] = None,
        log_retention_retry_options: typing.Optional[typing.Union["LogRetentionRetryOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        log_retention_role: typing.Optional[_IRole_235f5d8e] = None,
        memory_size: typing.Optional[jsii.Number] = None,
        params_and_secrets: typing.Optional["ParamsAndSecretsLayerVersion"] = None,
        profiling: typing.Optional[builtins.bool] = None,
        profiling_group: typing.Optional[_IProfilingGroup_0bba72c4] = None,
        recursive_loop: typing.Optional["RecursiveLoop"] = None,
        reserved_concurrent_executions: typing.Optional[jsii.Number] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
        runtime_management_mode: typing.Optional["RuntimeManagementMode"] = None,
        security_groups: typing.Optional[typing.Sequence[_ISecurityGroup_acf8a799]] = None,
        snap_start: typing.Optional["SnapStartConf"] = None,
        system_log_level: typing.Optional[builtins.str] = None,
        system_log_level_v2: typing.Optional["SystemLogLevel"] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        tracing: typing.Optional["Tracing"] = None,
        vpc: typing.Optional[_IVpc_f30d5663] = None,
        vpc_subnets: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Non runtime options.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param adot_instrumentation: Specify the configuration of AWS Distro for OpenTelemetry (ADOT) instrumentation. Default: - No ADOT instrumentation
        :param allow_all_ipv6_outbound: Whether to allow the Lambda to send all ipv6 network traffic. If set to true, there will only be a single egress rule which allows all outbound ipv6 traffic. If set to false, you must individually add traffic rules to allow the Lambda to connect to network targets using ipv6. Do not specify this property if the ``securityGroups`` or ``securityGroup`` property is set. Instead, configure ``allowAllIpv6Outbound`` directly on the security group. Default: false
        :param allow_all_outbound: Whether to allow the Lambda to send all network traffic (except ipv6). If set to false, you must individually add traffic rules to allow the Lambda to connect to network targets. Do not specify this property if the ``securityGroups`` or ``securityGroup`` property is set. Instead, configure ``allowAllOutbound`` directly on the security group. Default: true
        :param allow_public_subnet: Lambda Functions in a public subnet can NOT access the internet. Use this property to acknowledge this limitation and still place the function in a public subnet. Default: false
        :param application_log_level: (deprecated) Sets the application log level for the function. Default: "INFO"
        :param application_log_level_v2: Sets the application log level for the function. Default: ApplicationLogLevel.INFO
        :param architecture: The system architectures compatible with this lambda function. Default: Architecture.X86_64
        :param code_signing_config: Code signing config associated with this function. Default: - Not Sign the Code
        :param current_version_options: Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method. Default: - default options as described in ``VersionOptions``
        :param dead_letter_queue: The SQS queue to use if DLQ is enabled. If SNS topic is desired, specify ``deadLetterTopic`` property instead. Default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        :param dead_letter_queue_enabled: Enabled DLQ. If ``deadLetterQueue`` is undefined, an SQS queue with default options will be defined for your Function. Default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        :param dead_letter_topic: The SNS topic to use as a DLQ. Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly. Default: - no SNS topic
        :param description: A description of the function. Default: - No description.
        :param environment: Key-value pairs that Lambda caches and makes available for your Lambda functions. Use environment variables to apply configuration changes, such as test and production environment configurations, without changing your Lambda function source code. Default: - No environment variables.
        :param environment_encryption: The AWS KMS key that's used to encrypt your function's environment variables. Default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        :param ephemeral_storage_size: The size of the function’s /tmp directory in MiB. Default: 512 MiB
        :param events: Event sources for this function. You can also add event sources using ``addEventSource``. Default: - No event sources.
        :param filesystem: The filesystem configuration for the lambda function. Default: - will not mount any filesystem
        :param function_name: A name for the function. Default: - AWS CloudFormation generates a unique physical ID and uses that ID for the function's name. For more information, see Name Type.
        :param initial_policy: Initial policy statements to add to the created Lambda Role. You can call ``addToRolePolicy`` to the created lambda to add statements post creation. Default: - No policy statements are added to the created Lambda role.
        :param insights_version: Specify the version of CloudWatch Lambda insights to use for monitoring. Default: - No Lambda Insights
        :param ipv6_allowed_for_dual_stack: Allows outbound IPv6 traffic on VPC functions that are connected to dual-stack subnets. Only used if 'vpc' is supplied. Default: false
        :param layers: A list of layers to add to the function's execution environment. You can configure your Lambda function to pull in additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies that can be used by multiple functions. Default: - No layers.
        :param log_format: (deprecated) Sets the logFormat for the function. Default: "Text"
        :param logging_format: Sets the loggingFormat for the function. Default: LoggingFormat.TEXT
        :param log_group: The log group the function sends logs to. By default, Lambda functions send logs to an automatically created default log group named /aws/lambda/<function name>. However you cannot change the properties of this auto-created log group using the AWS CDK, e.g. you cannot set a different log retention. Use the ``logGroup`` property to create a fully customizable LogGroup ahead of time, and instruct the Lambda function to send logs to it. Providing a user-controlled log group was rolled out to commercial regions on 2023-11-16. If you are deploying to another type of region, please check regional availability first. Default: ``/aws/lambda/${this.functionName}`` - default log group created by Lambda
        :param log_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``INFINITE``. This is a legacy API and we strongly recommend you move away from it if you can. Instead create a fully customizable log group with ``logs.LogGroup`` and use the ``logGroup`` property to instruct the Lambda function to send logs to it. Migrating from ``logRetention`` to ``logGroup`` will cause the name of the log group to change. Users and code and referencing the name verbatim will have to adjust. In AWS CDK code, you can access the log group name directly from the LogGroup construct:: import * as logs from 'aws-cdk-lib/aws-logs'; declare const myLogGroup: logs.LogGroup; myLogGroup.logGroupName; Default: logs.RetentionDays.INFINITE
        :param log_retention_retry_options: When log retention is specified, a custom resource attempts to create the CloudWatch log group. These options control the retry policy when interacting with CloudWatch APIs. This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can. ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it. Default: - Default AWS SDK retry options.
        :param log_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can. ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it. Default: - A new role is created.
        :param memory_size: The amount of memory, in MB, that is allocated to your Lambda function. Lambda uses this value to proportionally allocate the amount of CPU power. For more information, see Resource Model in the AWS Lambda Developer Guide. Default: 128
        :param params_and_secrets: Specify the configuration of Parameters and Secrets Extension. Default: - No Parameters and Secrets Extension
        :param profiling: Enable profiling. Default: - No profiling.
        :param profiling_group: Profiling Group. Default: - A new profiling group will be created if ``profiling`` is set.
        :param recursive_loop: Sets the Recursive Loop Protection for Lambda Function. It lets Lambda detect and terminate unintended recursive loops. Default: RecursiveLoop.Terminate
        :param reserved_concurrent_executions: The maximum of concurrent executions you want to reserve for the function. Default: - No specific limit - account limit.
        :param role: Lambda execution role. This is the role that will be assumed by the function upon execution. It controls the permissions that the function will have. The Role must be assumable by the 'lambda.amazonaws.com' service principal. The default Role automatically has permissions granted for Lambda execution. If you provide a Role, you must add the relevant AWS managed policies yourself. The relevant managed policies are "service-role/AWSLambdaBasicExecutionRole" and "service-role/AWSLambdaVPCAccessExecutionRole". Default: - A unique role will be generated for this lambda function. Both supplied and generated roles can always be changed by calling ``addToRolePolicy``.
        :param runtime_management_mode: Sets the runtime management configuration for a function's version. Default: Auto
        :param security_groups: The list of security groups to associate with the Lambda's network interfaces. Only used if 'vpc' is supplied. Default: - If the function is placed within a VPC and a security group is not specified, either by this or securityGroup prop, a dedicated security group will be created for this function.
        :param snap_start: Enable SnapStart for Lambda Function. SnapStart is currently supported for Java 11, Java 17, Python 3.12, Python 3.13, and .NET 8 runtime Default: - No snapstart
        :param system_log_level: (deprecated) Sets the system log level for the function. Default: "INFO"
        :param system_log_level_v2: Sets the system log level for the function. Default: SystemLogLevel.INFO
        :param timeout: The function execution time (in seconds) after which Lambda terminates the function. Because the execution time affects cost, set this value based on the function's expected execution time. Default: Duration.seconds(3)
        :param tracing: Enable AWS X-Ray Tracing for Lambda Function. Default: Tracing.Disabled
        :param vpc: VPC network to place Lambda network interfaces. Specify this if the Lambda function needs to access resources in a VPC. This is required when ``vpcSubnets`` is specified. Default: - Function is not placed within a VPC.
        :param vpc_subnets: Where to place the network interfaces within the VPC. This requires ``vpc`` to be specified in order for interfaces to actually be placed in the subnets. If ``vpc`` is not specify, this will raise an error. Note: Internet access for Lambda Functions requires a NAT Gateway, so picking public subnets is not allowed (unless ``allowPublicSubnet`` is set to ``true``). Default: - the Vpc default strategy if not specified

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_codeguruprofiler as codeguruprofiler
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_iam as iam
            from aws_cdk import aws_kms as kms
            from aws_cdk import aws_lambda as lambda_
            from aws_cdk import aws_logs as logs
            from aws_cdk import aws_sns as sns
            from aws_cdk import aws_sqs as sqs
            
            # adot_layer_version: lambda.AdotLayerVersion
            # architecture: lambda.Architecture
            # code_signing_config: lambda.CodeSigningConfig
            # destination: lambda.IDestination
            # event_source: lambda.IEventSource
            # file_system: lambda.FileSystem
            # key: kms.Key
            # lambda_insights_version: lambda.LambdaInsightsVersion
            # layer_version: lambda.LayerVersion
            # log_group: logs.LogGroup
            # params_and_secrets_layer_version: lambda.ParamsAndSecretsLayerVersion
            # policy_statement: iam.PolicyStatement
            # profiling_group: codeguruprofiler.ProfilingGroup
            # queue: sqs.Queue
            # role: iam.Role
            # runtime_management_mode: lambda.RuntimeManagementMode
            # security_group: ec2.SecurityGroup
            # size: cdk.Size
            # snap_start_conf: lambda.SnapStartConf
            # subnet: ec2.Subnet
            # subnet_filter: ec2.SubnetFilter
            # topic: sns.Topic
            # vpc: ec2.Vpc
            
            function_options = lambda.FunctionOptions(
                adot_instrumentation=lambda.AdotInstrumentationConfig(
                    exec_wrapper=lambda_.AdotLambdaExecWrapper.REGULAR_HANDLER,
                    layer_version=adot_layer_version
                ),
                allow_all_ipv6_outbound=False,
                allow_all_outbound=False,
                allow_public_subnet=False,
                application_log_level="applicationLogLevel",
                application_log_level_v2=lambda_.ApplicationLogLevel.INFO,
                architecture=architecture,
                code_signing_config=code_signing_config,
                current_version_options=lambda.VersionOptions(
                    code_sha256="codeSha256",
                    description="description",
                    max_event_age=cdk.Duration.minutes(30),
                    on_failure=destination,
                    on_success=destination,
                    provisioned_concurrent_executions=123,
                    removal_policy=cdk.RemovalPolicy.DESTROY,
                    retry_attempts=123
                ),
                dead_letter_queue=queue,
                dead_letter_queue_enabled=False,
                dead_letter_topic=topic,
                description="description",
                environment={
                    "environment_key": "environment"
                },
                environment_encryption=key,
                ephemeral_storage_size=size,
                events=[event_source],
                filesystem=file_system,
                function_name="functionName",
                initial_policy=[policy_statement],
                insights_version=lambda_insights_version,
                ipv6_allowed_for_dual_stack=False,
                layers=[layer_version],
                log_format="logFormat",
                logging_format=lambda_.LoggingFormat.TEXT,
                log_group=log_group,
                log_retention=logs.RetentionDays.ONE_DAY,
                log_retention_retry_options=lambda.LogRetentionRetryOptions(
                    base=cdk.Duration.minutes(30),
                    max_retries=123
                ),
                log_retention_role=role,
                max_event_age=cdk.Duration.minutes(30),
                memory_size=123,
                on_failure=destination,
                on_success=destination,
                params_and_secrets=params_and_secrets_layer_version,
                profiling=False,
                profiling_group=profiling_group,
                recursive_loop=lambda_.RecursiveLoop.ALLOW,
                reserved_concurrent_executions=123,
                retry_attempts=123,
                role=role,
                runtime_management_mode=runtime_management_mode,
                security_groups=[security_group],
                snap_start=snap_start_conf,
                system_log_level="systemLogLevel",
                system_log_level_v2=lambda_.SystemLogLevel.INFO,
                timeout=cdk.Duration.minutes(30),
                tracing=lambda_.Tracing.ACTIVE,
                vpc=vpc,
                vpc_subnets=ec2.SubnetSelection(
                    availability_zones=["availabilityZones"],
                    one_per_az=False,
                    subnet_filters=[subnet_filter],
                    subnet_group_name="subnetGroupName",
                    subnets=[subnet],
                    subnet_type=ec2.SubnetType.PRIVATE_ISOLATED
                )
            )
        '''
        if isinstance(adot_instrumentation, dict):
            adot_instrumentation = AdotInstrumentationConfig(**adot_instrumentation)
        if isinstance(current_version_options, dict):
            current_version_options = VersionOptions(**current_version_options)
        if isinstance(log_retention_retry_options, dict):
            log_retention_retry_options = LogRetentionRetryOptions(**log_retention_retry_options)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _SubnetSelection_e57d76df(**vpc_subnets)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__59918bb957d892739733c7a5849db990615fe5329709ad7ba703e0ee45e54dbf)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument adot_instrumentation", value=adot_instrumentation, expected_type=type_hints["adot_instrumentation"])
            check_type(argname="argument allow_all_ipv6_outbound", value=allow_all_ipv6_outbound, expected_type=type_hints["allow_all_ipv6_outbound"])
            check_type(argname="argument allow_all_outbound", value=allow_all_outbound, expected_type=type_hints["allow_all_outbound"])
            check_type(argname="argument allow_public_subnet", value=allow_public_subnet, expected_type=type_hints["allow_public_subnet"])
            check_type(argname="argument application_log_level", value=application_log_level, expected_type=type_hints["application_log_level"])
            check_type(argname="argument application_log_level_v2", value=application_log_level_v2, expected_type=type_hints["application_log_level_v2"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
            check_type(argname="argument code_signing_config", value=code_signing_config, expected_type=type_hints["code_signing_config"])
            check_type(argname="argument current_version_options", value=current_version_options, expected_type=type_hints["current_version_options"])
            check_type(argname="argument dead_letter_queue", value=dead_letter_queue, expected_type=type_hints["dead_letter_queue"])
            check_type(argname="argument dead_letter_queue_enabled", value=dead_letter_queue_enabled, expected_type=type_hints["dead_letter_queue_enabled"])
            check_type(argname="argument dead_letter_topic", value=dead_letter_topic, expected_type=type_hints["dead_letter_topic"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument environment_encryption", value=environment_encryption, expected_type=type_hints["environment_encryption"])
            check_type(argname="argument ephemeral_storage_size", value=ephemeral_storage_size, expected_type=type_hints["ephemeral_storage_size"])
            check_type(argname="argument events", value=events, expected_type=type_hints["events"])
            check_type(argname="argument filesystem", value=filesystem, expected_type=type_hints["filesystem"])
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument initial_policy", value=initial_policy, expected_type=type_hints["initial_policy"])
            check_type(argname="argument insights_version", value=insights_version, expected_type=type_hints["insights_version"])
            check_type(argname="argument ipv6_allowed_for_dual_stack", value=ipv6_allowed_for_dual_stack, expected_type=type_hints["ipv6_allowed_for_dual_stack"])
            check_type(argname="argument layers", value=layers, expected_type=type_hints["layers"])
            check_type(argname="argument log_format", value=log_format, expected_type=type_hints["log_format"])
            check_type(argname="argument logging_format", value=logging_format, expected_type=type_hints["logging_format"])
            check_type(argname="argument log_group", value=log_group, expected_type=type_hints["log_group"])
            check_type(argname="argument log_retention", value=log_retention, expected_type=type_hints["log_retention"])
            check_type(argname="argument log_retention_retry_options", value=log_retention_retry_options, expected_type=type_hints["log_retention_retry_options"])
            check_type(argname="argument log_retention_role", value=log_retention_role, expected_type=type_hints["log_retention_role"])
            check_type(argname="argument memory_size", value=memory_size, expected_type=type_hints["memory_size"])
            check_type(argname="argument params_and_secrets", value=params_and_secrets, expected_type=type_hints["params_and_secrets"])
            check_type(argname="argument profiling", value=profiling, expected_type=type_hints["profiling"])
            check_type(argname="argument profiling_group", value=profiling_group, expected_type=type_hints["profiling_group"])
            check_type(argname="argument recursive_loop", value=recursive_loop, expected_type=type_hints["recursive_loop"])
            check_type(argname="argument reserved_concurrent_executions", value=reserved_concurrent_executions, expected_type=type_hints["reserved_concurrent_executions"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument runtime_management_mode", value=runtime_management_mode, expected_type=type_hints["runtime_management_mode"])
            check_type(argname="argument security_groups", value=security_groups, expected_type=type_hints["security_groups"])
            check_type(argname="argument snap_start", value=snap_start, expected_type=type_hints["snap_start"])
            check_type(argname="argument system_log_level", value=system_log_level, expected_type=type_hints["system_log_level"])
            check_type(argname="argument system_log_level_v2", value=system_log_level_v2, expected_type=type_hints["system_log_level_v2"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument tracing", value=tracing, expected_type=type_hints["tracing"])
            check_type(argname="argument vpc", value=vpc, expected_type=type_hints["vpc"])
            check_type(argname="argument vpc_subnets", value=vpc_subnets, expected_type=type_hints["vpc_subnets"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if adot_instrumentation is not None:
            self._values["adot_instrumentation"] = adot_instrumentation
        if allow_all_ipv6_outbound is not None:
            self._values["allow_all_ipv6_outbound"] = allow_all_ipv6_outbound
        if allow_all_outbound is not None:
            self._values["allow_all_outbound"] = allow_all_outbound
        if allow_public_subnet is not None:
            self._values["allow_public_subnet"] = allow_public_subnet
        if application_log_level is not None:
            self._values["application_log_level"] = application_log_level
        if application_log_level_v2 is not None:
            self._values["application_log_level_v2"] = application_log_level_v2
        if architecture is not None:
            self._values["architecture"] = architecture
        if code_signing_config is not None:
            self._values["code_signing_config"] = code_signing_config
        if current_version_options is not None:
            self._values["current_version_options"] = current_version_options
        if dead_letter_queue is not None:
            self._values["dead_letter_queue"] = dead_letter_queue
        if dead_letter_queue_enabled is not None:
            self._values["dead_letter_queue_enabled"] = dead_letter_queue_enabled
        if dead_letter_topic is not None:
            self._values["dead_letter_topic"] = dead_letter_topic
        if description is not None:
            self._values["description"] = description
        if environment is not None:
            self._values["environment"] = environment
        if environment_encryption is not None:
            self._values["environment_encryption"] = environment_encryption
        if ephemeral_storage_size is not None:
            self._values["ephemeral_storage_size"] = ephemeral_storage_size
        if events is not None:
            self._values["events"] = events
        if filesystem is not None:
            self._values["filesystem"] = filesystem
        if function_name is not None:
            self._values["function_name"] = function_name
        if initial_policy is not None:
            self._values["initial_policy"] = initial_policy
        if insights_version is not None:
            self._values["insights_version"] = insights_version
        if ipv6_allowed_for_dual_stack is not None:
            self._values["ipv6_allowed_for_dual_stack"] = ipv6_allowed_for_dual_stack
        if layers is not None:
            self._values["layers"] = layers
        if log_format is not None:
            self._values["log_format"] = log_format
        if logging_format is not None:
            self._values["logging_format"] = logging_format
        if log_group is not None:
            self._values["log_group"] = log_group
        if log_retention is not None:
            self._values["log_retention"] = log_retention
        if log_retention_retry_options is not None:
            self._values["log_retention_retry_options"] = log_retention_retry_options
        if log_retention_role is not None:
            self._values["log_retention_role"] = log_retention_role
        if memory_size is not None:
            self._values["memory_size"] = memory_size
        if params_and_secrets is not None:
            self._values["params_and_secrets"] = params_and_secrets
        if profiling is not None:
            self._values["profiling"] = profiling
        if profiling_group is not None:
            self._values["profiling_group"] = profiling_group
        if recursive_loop is not None:
            self._values["recursive_loop"] = recursive_loop
        if reserved_concurrent_executions is not None:
            self._values["reserved_concurrent_executions"] = reserved_concurrent_executions
        if role is not None:
            self._values["role"] = role
        if runtime_management_mode is not None:
            self._values["runtime_management_mode"] = runtime_management_mode
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if snap_start is not None:
            self._values["snap_start"] = snap_start
        if system_log_level is not None:
            self._values["system_log_level"] = system_log_level
        if system_log_level_v2 is not None:
            self._values["system_log_level_v2"] = system_log_level_v2
        if timeout is not None:
            self._values["timeout"] = timeout
        if tracing is not None:
            self._values["tracing"] = tracing
        if vpc is not None:
            self._values["vpc"] = vpc
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def max_event_age(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IDestination"]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def on_success(self) -> typing.Optional["IDestination"]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def adot_instrumentation(self) -> typing.Optional[AdotInstrumentationConfig]:
        '''Specify the configuration of AWS Distro for OpenTelemetry (ADOT) instrumentation.

        :default: - No ADOT instrumentation

        :see: https://aws-otel.github.io/docs/getting-started/lambda
        '''
        result = self._values.get("adot_instrumentation")
        return typing.cast(typing.Optional[AdotInstrumentationConfig], result)

    @builtins.property
    def allow_all_ipv6_outbound(self) -> typing.Optional[builtins.bool]:
        '''Whether to allow the Lambda to send all ipv6 network traffic.

        If set to true, there will only be a single egress rule which allows all
        outbound ipv6 traffic. If set to false, you must individually add traffic rules to allow the
        Lambda to connect to network targets using ipv6.

        Do not specify this property if the ``securityGroups`` or ``securityGroup`` property is set.
        Instead, configure ``allowAllIpv6Outbound`` directly on the security group.

        :default: false
        '''
        result = self._values.get("allow_all_ipv6_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def allow_all_outbound(self) -> typing.Optional[builtins.bool]:
        '''Whether to allow the Lambda to send all network traffic (except ipv6).

        If set to false, you must individually add traffic rules to allow the
        Lambda to connect to network targets.

        Do not specify this property if the ``securityGroups`` or ``securityGroup`` property is set.
        Instead, configure ``allowAllOutbound`` directly on the security group.

        :default: true
        '''
        result = self._values.get("allow_all_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def allow_public_subnet(self) -> typing.Optional[builtins.bool]:
        '''Lambda Functions in a public subnet can NOT access the internet.

        Use this property to acknowledge this limitation and still place the function in a public subnet.

        :default: false

        :see: https://stackoverflow.com/questions/52992085/why-cant-an-aws-lambda-function-inside-a-public-subnet-in-a-vpc-connect-to-the/52994841#52994841
        '''
        result = self._values.get("allow_public_subnet")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def application_log_level(self) -> typing.Optional[builtins.str]:
        '''(deprecated) Sets the application log level for the function.

        :default: "INFO"

        :deprecated: Use ``applicationLogLevelV2`` as a property instead.

        :stability: deprecated
        '''
        result = self._values.get("application_log_level")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def application_log_level_v2(self) -> typing.Optional[ApplicationLogLevel]:
        '''Sets the application log level for the function.

        :default: ApplicationLogLevel.INFO
        '''
        result = self._values.get("application_log_level_v2")
        return typing.cast(typing.Optional[ApplicationLogLevel], result)

    @builtins.property
    def architecture(self) -> typing.Optional[Architecture]:
        '''The system architectures compatible with this lambda function.

        :default: Architecture.X86_64
        '''
        result = self._values.get("architecture")
        return typing.cast(typing.Optional[Architecture], result)

    @builtins.property
    def code_signing_config(self) -> typing.Optional["ICodeSigningConfig"]:
        '''Code signing config associated with this function.

        :default: - Not Sign the Code
        '''
        result = self._values.get("code_signing_config")
        return typing.cast(typing.Optional["ICodeSigningConfig"], result)

    @builtins.property
    def current_version_options(self) -> typing.Optional["VersionOptions"]:
        '''Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method.

        :default: - default options as described in ``VersionOptions``
        '''
        result = self._values.get("current_version_options")
        return typing.cast(typing.Optional["VersionOptions"], result)

    @builtins.property
    def dead_letter_queue(self) -> typing.Optional[_IQueue_7ed6f679]:
        '''The SQS queue to use if DLQ is enabled.

        If SNS topic is desired, specify ``deadLetterTopic`` property instead.

        :default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        '''
        result = self._values.get("dead_letter_queue")
        return typing.cast(typing.Optional[_IQueue_7ed6f679], result)

    @builtins.property
    def dead_letter_queue_enabled(self) -> typing.Optional[builtins.bool]:
        '''Enabled DLQ.

        If ``deadLetterQueue`` is undefined,
        an SQS queue with default options will be defined for your Function.

        :default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        '''
        result = self._values.get("dead_letter_queue_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def dead_letter_topic(self) -> typing.Optional[_ITopic_9eca4852]:
        '''The SNS topic to use as a DLQ.

        Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created
        rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly.

        :default: - no SNS topic
        '''
        result = self._values.get("dead_letter_topic")
        return typing.cast(typing.Optional[_ITopic_9eca4852], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the function.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Key-value pairs that Lambda caches and makes available for your Lambda functions.

        Use environment variables to apply configuration changes, such
        as test and production environment configurations, without changing your
        Lambda function source code.

        :default: - No environment variables.
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def environment_encryption(self) -> typing.Optional[_IKey_5f11635f]:
        '''The AWS KMS key that's used to encrypt your function's environment variables.

        :default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        '''
        result = self._values.get("environment_encryption")
        return typing.cast(typing.Optional[_IKey_5f11635f], result)

    @builtins.property
    def ephemeral_storage_size(self) -> typing.Optional[_Size_7b441c34]:
        '''The size of the function’s /tmp directory in MiB.

        :default: 512 MiB
        '''
        result = self._values.get("ephemeral_storage_size")
        return typing.cast(typing.Optional[_Size_7b441c34], result)

    @builtins.property
    def events(self) -> typing.Optional[typing.List["IEventSource"]]:
        '''Event sources for this function.

        You can also add event sources using ``addEventSource``.

        :default: - No event sources.
        '''
        result = self._values.get("events")
        return typing.cast(typing.Optional[typing.List["IEventSource"]], result)

    @builtins.property
    def filesystem(self) -> typing.Optional[FileSystem]:
        '''The filesystem configuration for the lambda function.

        :default: - will not mount any filesystem
        '''
        result = self._values.get("filesystem")
        return typing.cast(typing.Optional[FileSystem], result)

    @builtins.property
    def function_name(self) -> typing.Optional[builtins.str]:
        '''A name for the function.

        :default:

        - AWS CloudFormation generates a unique physical ID and uses that
        ID for the function's name. For more information, see Name Type.
        '''
        result = self._values.get("function_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def initial_policy(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        '''Initial policy statements to add to the created Lambda Role.

        You can call ``addToRolePolicy`` to the created lambda to add statements post creation.

        :default: - No policy statements are added to the created Lambda role.
        '''
        result = self._values.get("initial_policy")
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], result)

    @builtins.property
    def insights_version(self) -> typing.Optional["LambdaInsightsVersion"]:
        '''Specify the version of CloudWatch Lambda insights to use for monitoring.

        :default: - No Lambda Insights

        :see: https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-Getting-Started-docker.html
        '''
        result = self._values.get("insights_version")
        return typing.cast(typing.Optional["LambdaInsightsVersion"], result)

    @builtins.property
    def ipv6_allowed_for_dual_stack(self) -> typing.Optional[builtins.bool]:
        '''Allows outbound IPv6 traffic on VPC functions that are connected to dual-stack subnets.

        Only used if 'vpc' is supplied.

        :default: false
        '''
        result = self._values.get("ipv6_allowed_for_dual_stack")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def layers(self) -> typing.Optional[typing.List["ILayerVersion"]]:
        '''A list of layers to add to the function's execution environment.

        You can configure your Lambda function to pull in
        additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies
        that can be used by multiple functions.

        :default: - No layers.
        '''
        result = self._values.get("layers")
        return typing.cast(typing.Optional[typing.List["ILayerVersion"]], result)

    @builtins.property
    def log_format(self) -> typing.Optional[builtins.str]:
        '''(deprecated) Sets the logFormat for the function.

        :default: "Text"

        :deprecated: Use ``loggingFormat`` as a property instead.

        :stability: deprecated
        '''
        result = self._values.get("log_format")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def logging_format(self) -> typing.Optional["LoggingFormat"]:
        '''Sets the loggingFormat for the function.

        :default: LoggingFormat.TEXT
        '''
        result = self._values.get("logging_format")
        return typing.cast(typing.Optional["LoggingFormat"], result)

    @builtins.property
    def log_group(self) -> typing.Optional[_ILogGroup_3c4fa718]:
        '''The log group the function sends logs to.

        By default, Lambda functions send logs to an automatically created default log group named /aws/lambda/.
        However you cannot change the properties of this auto-created log group using the AWS CDK, e.g. you cannot set a different log retention.

        Use the ``logGroup`` property to create a fully customizable LogGroup ahead of time, and instruct the Lambda function to send logs to it.

        Providing a user-controlled log group was rolled out to commercial regions on 2023-11-16.
        If you are deploying to another type of region, please check regional availability first.

        :default: ``/aws/lambda/${this.functionName}`` - default log group created by Lambda
        '''
        result = self._values.get("log_group")
        return typing.cast(typing.Optional[_ILogGroup_3c4fa718], result)

    @builtins.property
    def log_retention(self) -> typing.Optional[_RetentionDays_070f99f0]:
        '''The number of days log events are kept in CloudWatch Logs.

        When updating
        this property, unsetting it doesn't remove the log retention policy. To
        remove the retention policy, set the value to ``INFINITE``.

        This is a legacy API and we strongly recommend you move away from it if you can.
        Instead create a fully customizable log group with ``logs.LogGroup`` and use the ``logGroup`` property
        to instruct the Lambda function to send logs to it.
        Migrating from ``logRetention`` to ``logGroup`` will cause the name of the log group to change.
        Users and code and referencing the name verbatim will have to adjust.

        In AWS CDK code, you can access the log group name directly from the LogGroup construct::

           import aws_cdk.aws_logs as logs

           # my_log_group: logs.LogGroup

           my_log_group.log_group_name

        :default: logs.RetentionDays.INFINITE
        '''
        result = self._values.get("log_retention")
        return typing.cast(typing.Optional[_RetentionDays_070f99f0], result)

    @builtins.property
    def log_retention_retry_options(
        self,
    ) -> typing.Optional["LogRetentionRetryOptions"]:
        '''When log retention is specified, a custom resource attempts to create the CloudWatch log group.

        These options control the retry policy when interacting with CloudWatch APIs.

        This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can.
        ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it.

        :default: - Default AWS SDK retry options.
        '''
        result = self._values.get("log_retention_retry_options")
        return typing.cast(typing.Optional["LogRetentionRetryOptions"], result)

    @builtins.property
    def log_retention_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The IAM role for the Lambda function associated with the custom resource that sets the retention policy.

        This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can.
        ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it.

        :default: - A new role is created.
        '''
        result = self._values.get("log_retention_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def memory_size(self) -> typing.Optional[jsii.Number]:
        '''The amount of memory, in MB, that is allocated to your Lambda function.

        Lambda uses this value to proportionally allocate the amount of CPU
        power. For more information, see Resource Model in the AWS Lambda
        Developer Guide.

        :default: 128
        '''
        result = self._values.get("memory_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def params_and_secrets(self) -> typing.Optional["ParamsAndSecretsLayerVersion"]:
        '''Specify the configuration of Parameters and Secrets Extension.

        :default: - No Parameters and Secrets Extension

        :see: https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-integration-lambda-extensions.html
        '''
        result = self._values.get("params_and_secrets")
        return typing.cast(typing.Optional["ParamsAndSecretsLayerVersion"], result)

    @builtins.property
    def profiling(self) -> typing.Optional[builtins.bool]:
        '''Enable profiling.

        :default: - No profiling.

        :see: https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
        '''
        result = self._values.get("profiling")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def profiling_group(self) -> typing.Optional[_IProfilingGroup_0bba72c4]:
        '''Profiling Group.

        :default: - A new profiling group will be created if ``profiling`` is set.

        :see: https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
        '''
        result = self._values.get("profiling_group")
        return typing.cast(typing.Optional[_IProfilingGroup_0bba72c4], result)

    @builtins.property
    def recursive_loop(self) -> typing.Optional["RecursiveLoop"]:
        '''Sets the Recursive Loop Protection for Lambda Function.

        It lets Lambda detect and terminate unintended recursive loops.

        :default: RecursiveLoop.Terminate
        '''
        result = self._values.get("recursive_loop")
        return typing.cast(typing.Optional["RecursiveLoop"], result)

    @builtins.property
    def reserved_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''The maximum of concurrent executions you want to reserve for the function.

        :default: - No specific limit - account limit.

        :see: https://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html
        '''
        result = self._values.get("reserved_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''Lambda execution role.

        This is the role that will be assumed by the function upon execution.
        It controls the permissions that the function will have. The Role must
        be assumable by the 'lambda.amazonaws.com' service principal.

        The default Role automatically has permissions granted for Lambda execution. If you
        provide a Role, you must add the relevant AWS managed policies yourself.

        The relevant managed policies are "service-role/AWSLambdaBasicExecutionRole" and
        "service-role/AWSLambdaVPCAccessExecutionRole".

        :default:

        - A unique role will be generated for this lambda function.
        Both supplied and generated roles can always be changed by calling ``addToRolePolicy``.
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def runtime_management_mode(self) -> typing.Optional["RuntimeManagementMode"]:
        '''Sets the runtime management configuration for a function's version.

        :default: Auto
        '''
        result = self._values.get("runtime_management_mode")
        return typing.cast(typing.Optional["RuntimeManagementMode"], result)

    @builtins.property
    def security_groups(self) -> typing.Optional[typing.List[_ISecurityGroup_acf8a799]]:
        '''The list of security groups to associate with the Lambda's network interfaces.

        Only used if 'vpc' is supplied.

        :default:

        - If the function is placed within a VPC and a security group is
        not specified, either by this or securityGroup prop, a dedicated security
        group will be created for this function.
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[_ISecurityGroup_acf8a799]], result)

    @builtins.property
    def snap_start(self) -> typing.Optional["SnapStartConf"]:
        '''Enable SnapStart for Lambda Function.

        SnapStart is currently supported for Java 11, Java 17, Python 3.12, Python 3.13, and .NET 8 runtime

        :default: - No snapstart
        '''
        result = self._values.get("snap_start")
        return typing.cast(typing.Optional["SnapStartConf"], result)

    @builtins.property
    def system_log_level(self) -> typing.Optional[builtins.str]:
        '''(deprecated) Sets the system log level for the function.

        :default: "INFO"

        :deprecated: Use ``systemLogLevelV2`` as a property instead.

        :stability: deprecated
        '''
        result = self._values.get("system_log_level")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def system_log_level_v2(self) -> typing.Optional["SystemLogLevel"]:
        '''Sets the system log level for the function.

        :default: SystemLogLevel.INFO
        '''
        result = self._values.get("system_log_level_v2")
        return typing.cast(typing.Optional["SystemLogLevel"], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The function execution time (in seconds) after which Lambda terminates the function.

        Because the execution time affects cost, set this value
        based on the function's expected execution time.

        :default: Duration.seconds(3)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def tracing(self) -> typing.Optional["Tracing"]:
        '''Enable AWS X-Ray Tracing for Lambda Function.

        :default: Tracing.Disabled
        '''
        result = self._values.get("tracing")
        return typing.cast(typing.Optional["Tracing"], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_IVpc_f30d5663]:
        '''VPC network to place Lambda network interfaces.

        Specify this if the Lambda function needs to access resources in a VPC.
        This is required when ``vpcSubnets`` is specified.

        :default: - Function is not placed within a VPC.
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_IVpc_f30d5663], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_SubnetSelection_e57d76df]:
        '''Where to place the network interfaces within the VPC.

        This requires ``vpc`` to be specified in order for interfaces to actually be
        placed in the subnets. If ``vpc`` is not specify, this will raise an error.

        Note: Internet access for Lambda Functions requires a NAT Gateway, so picking
        public subnets is not allowed (unless ``allowPublicSubnet`` is set to ``true``).

        :default: - the Vpc default strategy if not specified
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_SubnetSelection_e57d76df], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.FunctionProps",
    jsii_struct_bases=[FunctionOptions],
    name_mapping={
        "max_event_age": "maxEventAge",
        "on_failure": "onFailure",
        "on_success": "onSuccess",
        "retry_attempts": "retryAttempts",
        "adot_instrumentation": "adotInstrumentation",
        "allow_all_ipv6_outbound": "allowAllIpv6Outbound",
        "allow_all_outbound": "allowAllOutbound",
        "allow_public_subnet": "allowPublicSubnet",
        "application_log_level": "applicationLogLevel",
        "application_log_level_v2": "applicationLogLevelV2",
        "architecture": "architecture",
        "code_signing_config": "codeSigningConfig",
        "current_version_options": "currentVersionOptions",
        "dead_letter_queue": "deadLetterQueue",
        "dead_letter_queue_enabled": "deadLetterQueueEnabled",
        "dead_letter_topic": "deadLetterTopic",
        "description": "description",
        "environment": "environment",
        "environment_encryption": "environmentEncryption",
        "ephemeral_storage_size": "ephemeralStorageSize",
        "events": "events",
        "filesystem": "filesystem",
        "function_name": "functionName",
        "initial_policy": "initialPolicy",
        "insights_version": "insightsVersion",
        "ipv6_allowed_for_dual_stack": "ipv6AllowedForDualStack",
        "layers": "layers",
        "log_format": "logFormat",
        "logging_format": "loggingFormat",
        "log_group": "logGroup",
        "log_retention": "logRetention",
        "log_retention_retry_options": "logRetentionRetryOptions",
        "log_retention_role": "logRetentionRole",
        "memory_size": "memorySize",
        "params_and_secrets": "paramsAndSecrets",
        "profiling": "profiling",
        "profiling_group": "profilingGroup",
        "recursive_loop": "recursiveLoop",
        "reserved_concurrent_executions": "reservedConcurrentExecutions",
        "role": "role",
        "runtime_management_mode": "runtimeManagementMode",
        "security_groups": "securityGroups",
        "snap_start": "snapStart",
        "system_log_level": "systemLogLevel",
        "system_log_level_v2": "systemLogLevelV2",
        "timeout": "timeout",
        "tracing": "tracing",
        "vpc": "vpc",
        "vpc_subnets": "vpcSubnets",
        "code": "code",
        "handler": "handler",
        "runtime": "runtime",
    },
)
class FunctionProps(FunctionOptions):
    def __init__(
        self,
        *,
        max_event_age: typing.Optional[_Duration_4839e8c3] = None,
        on_failure: typing.Optional["IDestination"] = None,
        on_success: typing.Optional["IDestination"] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        adot_instrumentation: typing.Optional[typing.Union[AdotInstrumentationConfig, typing.Dict[builtins.str, typing.Any]]] = None,
        allow_all_ipv6_outbound: typing.Optional[builtins.bool] = None,
        allow_all_outbound: typing.Optional[builtins.bool] = None,
        allow_public_subnet: typing.Optional[builtins.bool] = None,
        application_log_level: typing.Optional[builtins.str] = None,
        application_log_level_v2: typing.Optional[ApplicationLogLevel] = None,
        architecture: typing.Optional[Architecture] = None,
        code_signing_config: typing.Optional["ICodeSigningConfig"] = None,
        current_version_options: typing.Optional[typing.Union["VersionOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        dead_letter_queue: typing.Optional[_IQueue_7ed6f679] = None,
        dead_letter_queue_enabled: typing.Optional[builtins.bool] = None,
        dead_letter_topic: typing.Optional[_ITopic_9eca4852] = None,
        description: typing.Optional[builtins.str] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_encryption: typing.Optional[_IKey_5f11635f] = None,
        ephemeral_storage_size: typing.Optional[_Size_7b441c34] = None,
        events: typing.Optional[typing.Sequence["IEventSource"]] = None,
        filesystem: typing.Optional[FileSystem] = None,
        function_name: typing.Optional[builtins.str] = None,
        initial_policy: typing.Optional[typing.Sequence[_PolicyStatement_0fe33853]] = None,
        insights_version: typing.Optional["LambdaInsightsVersion"] = None,
        ipv6_allowed_for_dual_stack: typing.Optional[builtins.bool] = None,
        layers: typing.Optional[typing.Sequence["ILayerVersion"]] = None,
        log_format: typing.Optional[builtins.str] = None,
        logging_format: typing.Optional["LoggingFormat"] = None,
        log_group: typing.Optional[_ILogGroup_3c4fa718] = None,
        log_retention: typing.Optional[_RetentionDays_070f99f0] = None,
        log_retention_retry_options: typing.Optional[typing.Union["LogRetentionRetryOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        log_retention_role: typing.Optional[_IRole_235f5d8e] = None,
        memory_size: typing.Optional[jsii.Number] = None,
        params_and_secrets: typing.Optional["ParamsAndSecretsLayerVersion"] = None,
        profiling: typing.Optional[builtins.bool] = None,
        profiling_group: typing.Optional[_IProfilingGroup_0bba72c4] = None,
        recursive_loop: typing.Optional["RecursiveLoop"] = None,
        reserved_concurrent_executions: typing.Optional[jsii.Number] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
        runtime_management_mode: typing.Optional["RuntimeManagementMode"] = None,
        security_groups: typing.Optional[typing.Sequence[_ISecurityGroup_acf8a799]] = None,
        snap_start: typing.Optional["SnapStartConf"] = None,
        system_log_level: typing.Optional[builtins.str] = None,
        system_log_level_v2: typing.Optional["SystemLogLevel"] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        tracing: typing.Optional["Tracing"] = None,
        vpc: typing.Optional[_IVpc_f30d5663] = None,
        vpc_subnets: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
        code: Code,
        handler: builtins.str,
        runtime: "Runtime",
    ) -> None:
        '''
        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        :param adot_instrumentation: Specify the configuration of AWS Distro for OpenTelemetry (ADOT) instrumentation. Default: - No ADOT instrumentation
        :param allow_all_ipv6_outbound: Whether to allow the Lambda to send all ipv6 network traffic. If set to true, there will only be a single egress rule which allows all outbound ipv6 traffic. If set to false, you must individually add traffic rules to allow the Lambda to connect to network targets using ipv6. Do not specify this property if the ``securityGroups`` or ``securityGroup`` property is set. Instead, configure ``allowAllIpv6Outbound`` directly on the security group. Default: false
        :param allow_all_outbound: Whether to allow the Lambda to send all network traffic (except ipv6). If set to false, you must individually add traffic rules to allow the Lambda to connect to network targets. Do not specify this property if the ``securityGroups`` or ``securityGroup`` property is set. Instead, configure ``allowAllOutbound`` directly on the security group. Default: true
        :param allow_public_subnet: Lambda Functions in a public subnet can NOT access the internet. Use this property to acknowledge this limitation and still place the function in a public subnet. Default: false
        :param application_log_level: (deprecated) Sets the application log level for the function. Default: "INFO"
        :param application_log_level_v2: Sets the application log level for the function. Default: ApplicationLogLevel.INFO
        :param architecture: The system architectures compatible with this lambda function. Default: Architecture.X86_64
        :param code_signing_config: Code signing config associated with this function. Default: - Not Sign the Code
        :param current_version_options: Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method. Default: - default options as described in ``VersionOptions``
        :param dead_letter_queue: The SQS queue to use if DLQ is enabled. If SNS topic is desired, specify ``deadLetterTopic`` property instead. Default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        :param dead_letter_queue_enabled: Enabled DLQ. If ``deadLetterQueue`` is undefined, an SQS queue with default options will be defined for your Function. Default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        :param dead_letter_topic: The SNS topic to use as a DLQ. Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly. Default: - no SNS topic
        :param description: A description of the function. Default: - No description.
        :param environment: Key-value pairs that Lambda caches and makes available for your Lambda functions. Use environment variables to apply configuration changes, such as test and production environment configurations, without changing your Lambda function source code. Default: - No environment variables.
        :param environment_encryption: The AWS KMS key that's used to encrypt your function's environment variables. Default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        :param ephemeral_storage_size: The size of the function’s /tmp directory in MiB. Default: 512 MiB
        :param events: Event sources for this function. You can also add event sources using ``addEventSource``. Default: - No event sources.
        :param filesystem: The filesystem configuration for the lambda function. Default: - will not mount any filesystem
        :param function_name: A name for the function. Default: - AWS CloudFormation generates a unique physical ID and uses that ID for the function's name. For more information, see Name Type.
        :param initial_policy: Initial policy statements to add to the created Lambda Role. You can call ``addToRolePolicy`` to the created lambda to add statements post creation. Default: - No policy statements are added to the created Lambda role.
        :param insights_version: Specify the version of CloudWatch Lambda insights to use for monitoring. Default: - No Lambda Insights
        :param ipv6_allowed_for_dual_stack: Allows outbound IPv6 traffic on VPC functions that are connected to dual-stack subnets. Only used if 'vpc' is supplied. Default: false
        :param layers: A list of layers to add to the function's execution environment. You can configure your Lambda function to pull in additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies that can be used by multiple functions. Default: - No layers.
        :param log_format: (deprecated) Sets the logFormat for the function. Default: "Text"
        :param logging_format: Sets the loggingFormat for the function. Default: LoggingFormat.TEXT
        :param log_group: The log group the function sends logs to. By default, Lambda functions send logs to an automatically created default log group named /aws/lambda/<function name>. However you cannot change the properties of this auto-created log group using the AWS CDK, e.g. you cannot set a different log retention. Use the ``logGroup`` property to create a fully customizable LogGroup ahead of time, and instruct the Lambda function to send logs to it. Providing a user-controlled log group was rolled out to commercial regions on 2023-11-16. If you are deploying to another type of region, please check regional availability first. Default: ``/aws/lambda/${this.functionName}`` - default log group created by Lambda
        :param log_retention: The number of days log events are kept in CloudWatch Logs. When updating this property, unsetting it doesn't remove the log retention policy. To remove the retention policy, set the value to ``INFINITE``. This is a legacy API and we strongly recommend you move away from it if you can. Instead create a fully customizable log group with ``logs.LogGroup`` and use the ``logGroup`` property to instruct the Lambda function to send logs to it. Migrating from ``logRetention`` to ``logGroup`` will cause the name of the log group to change. Users and code and referencing the name verbatim will have to adjust. In AWS CDK code, you can access the log group name directly from the LogGroup construct:: import * as logs from 'aws-cdk-lib/aws-logs'; declare const myLogGroup: logs.LogGroup; myLogGroup.logGroupName; Default: logs.RetentionDays.INFINITE
        :param log_retention_retry_options: When log retention is specified, a custom resource attempts to create the CloudWatch log group. These options control the retry policy when interacting with CloudWatch APIs. This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can. ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it. Default: - Default AWS SDK retry options.
        :param log_retention_role: The IAM role for the Lambda function associated with the custom resource that sets the retention policy. This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can. ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it. Default: - A new role is created.
        :param memory_size: The amount of memory, in MB, that is allocated to your Lambda function. Lambda uses this value to proportionally allocate the amount of CPU power. For more information, see Resource Model in the AWS Lambda Developer Guide. Default: 128
        :param params_and_secrets: Specify the configuration of Parameters and Secrets Extension. Default: - No Parameters and Secrets Extension
        :param profiling: Enable profiling. Default: - No profiling.
        :param profiling_group: Profiling Group. Default: - A new profiling group will be created if ``profiling`` is set.
        :param recursive_loop: Sets the Recursive Loop Protection for Lambda Function. It lets Lambda detect and terminate unintended recursive loops. Default: RecursiveLoop.Terminate
        :param reserved_concurrent_executions: The maximum of concurrent executions you want to reserve for the function. Default: - No specific limit - account limit.
        :param role: Lambda execution role. This is the role that will be assumed by the function upon execution. It controls the permissions that the function will have. The Role must be assumable by the 'lambda.amazonaws.com' service principal. The default Role automatically has permissions granted for Lambda execution. If you provide a Role, you must add the relevant AWS managed policies yourself. The relevant managed policies are "service-role/AWSLambdaBasicExecutionRole" and "service-role/AWSLambdaVPCAccessExecutionRole". Default: - A unique role will be generated for this lambda function. Both supplied and generated roles can always be changed by calling ``addToRolePolicy``.
        :param runtime_management_mode: Sets the runtime management configuration for a function's version. Default: Auto
        :param security_groups: The list of security groups to associate with the Lambda's network interfaces. Only used if 'vpc' is supplied. Default: - If the function is placed within a VPC and a security group is not specified, either by this or securityGroup prop, a dedicated security group will be created for this function.
        :param snap_start: Enable SnapStart for Lambda Function. SnapStart is currently supported for Java 11, Java 17, Python 3.12, Python 3.13, and .NET 8 runtime Default: - No snapstart
        :param system_log_level: (deprecated) Sets the system log level for the function. Default: "INFO"
        :param system_log_level_v2: Sets the system log level for the function. Default: SystemLogLevel.INFO
        :param timeout: The function execution time (in seconds) after which Lambda terminates the function. Because the execution time affects cost, set this value based on the function's expected execution time. Default: Duration.seconds(3)
        :param tracing: Enable AWS X-Ray Tracing for Lambda Function. Default: Tracing.Disabled
        :param vpc: VPC network to place Lambda network interfaces. Specify this if the Lambda function needs to access resources in a VPC. This is required when ``vpcSubnets`` is specified. Default: - Function is not placed within a VPC.
        :param vpc_subnets: Where to place the network interfaces within the VPC. This requires ``vpc`` to be specified in order for interfaces to actually be placed in the subnets. If ``vpc`` is not specify, this will raise an error. Note: Internet access for Lambda Functions requires a NAT Gateway, so picking public subnets is not allowed (unless ``allowPublicSubnet`` is set to ``true``). Default: - the Vpc default strategy if not specified
        :param code: The source code of your Lambda function. You can point to a file in an Amazon Simple Storage Service (Amazon S3) bucket or specify your source code as inline text.
        :param handler: The name of the method within your code that Lambda calls to execute your function. The format includes the file name. It can also include namespaces and other qualifiers, depending on the runtime. For more information, see https://docs.aws.amazon.com/lambda/latest/dg/foundation-progmodel.html. Use ``Handler.FROM_IMAGE`` when defining a function from a Docker image. NOTE: If you specify your source code as inline text by specifying the ZipFile property within the Code property, specify index.function_name as the handler.
        :param runtime: The runtime environment for the Lambda function that you are uploading. For valid values, see the Runtime property in the AWS Lambda Developer Guide. Use ``Runtime.FROM_IMAGE`` when defining a function from a Docker image.

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_lambda as lambda_
            
            
            fn = lambda_.Function(self, "MyFunc",
                runtime=lambda_.Runtime.NODEJS_LATEST,
                handler="index.handler",
                code=lambda_.Code.from_inline("exports.handler = handler.toString()")
            )
            
            rule = events.Rule(self, "rule",
                event_pattern=events.EventPattern(
                    source=["aws.ec2"]
                )
            )
            
            queue = sqs.Queue(self, "Queue")
            
            rule.add_target(targets.LambdaFunction(fn,
                dead_letter_queue=queue,  # Optional: add a dead letter queue
                max_event_age=Duration.hours(2),  # Optional: set the maxEventAge retry policy
                retry_attempts=2
            ))
        '''
        if isinstance(adot_instrumentation, dict):
            adot_instrumentation = AdotInstrumentationConfig(**adot_instrumentation)
        if isinstance(current_version_options, dict):
            current_version_options = VersionOptions(**current_version_options)
        if isinstance(log_retention_retry_options, dict):
            log_retention_retry_options = LogRetentionRetryOptions(**log_retention_retry_options)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _SubnetSelection_e57d76df(**vpc_subnets)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__94e70d11aa3c53737d418dbb9983973dfc06dbdef5c8cc30613cc3c6d0f4f3e4)
            check_type(argname="argument max_event_age", value=max_event_age, expected_type=type_hints["max_event_age"])
            check_type(argname="argument on_failure", value=on_failure, expected_type=type_hints["on_failure"])
            check_type(argname="argument on_success", value=on_success, expected_type=type_hints["on_success"])
            check_type(argname="argument retry_attempts", value=retry_attempts, expected_type=type_hints["retry_attempts"])
            check_type(argname="argument adot_instrumentation", value=adot_instrumentation, expected_type=type_hints["adot_instrumentation"])
            check_type(argname="argument allow_all_ipv6_outbound", value=allow_all_ipv6_outbound, expected_type=type_hints["allow_all_ipv6_outbound"])
            check_type(argname="argument allow_all_outbound", value=allow_all_outbound, expected_type=type_hints["allow_all_outbound"])
            check_type(argname="argument allow_public_subnet", value=allow_public_subnet, expected_type=type_hints["allow_public_subnet"])
            check_type(argname="argument application_log_level", value=application_log_level, expected_type=type_hints["application_log_level"])
            check_type(argname="argument application_log_level_v2", value=application_log_level_v2, expected_type=type_hints["application_log_level_v2"])
            check_type(argname="argument architecture", value=architecture, expected_type=type_hints["architecture"])
            check_type(argname="argument code_signing_config", value=code_signing_config, expected_type=type_hints["code_signing_config"])
            check_type(argname="argument current_version_options", value=current_version_options, expected_type=type_hints["current_version_options"])
            check_type(argname="argument dead_letter_queue", value=dead_letter_queue, expected_type=type_hints["dead_letter_queue"])
            check_type(argname="argument dead_letter_queue_enabled", value=dead_letter_queue_enabled, expected_type=type_hints["dead_letter_queue_enabled"])
            check_type(argname="argument dead_letter_topic", value=dead_letter_topic, expected_type=type_hints["dead_letter_topic"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument environment_encryption", value=environment_encryption, expected_type=type_hints["environment_encryption"])
            check_type(argname="argument ephemeral_storage_size", value=ephemeral_storage_size, expected_type=type_hints["ephemeral_storage_size"])
            check_type(argname="argument events", value=events, expected_type=type_hints["events"])
            check_type(argname="argument filesystem", value=filesystem, expected_type=type_hints["filesystem"])
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
            check_type(argname="argument initial_policy", value=initial_policy, expected_type=type_hints["initial_policy"])
            check_type(argname="argument insights_version", value=insights_version, expected_type=type_hints["insights_version"])
            check_type(argname="argument ipv6_allowed_for_dual_stack", value=ipv6_allowed_for_dual_stack, expected_type=type_hints["ipv6_allowed_for_dual_stack"])
            check_type(argname="argument layers", value=layers, expected_type=type_hints["layers"])
            check_type(argname="argument log_format", value=log_format, expected_type=type_hints["log_format"])
            check_type(argname="argument logging_format", value=logging_format, expected_type=type_hints["logging_format"])
            check_type(argname="argument log_group", value=log_group, expected_type=type_hints["log_group"])
            check_type(argname="argument log_retention", value=log_retention, expected_type=type_hints["log_retention"])
            check_type(argname="argument log_retention_retry_options", value=log_retention_retry_options, expected_type=type_hints["log_retention_retry_options"])
            check_type(argname="argument log_retention_role", value=log_retention_role, expected_type=type_hints["log_retention_role"])
            check_type(argname="argument memory_size", value=memory_size, expected_type=type_hints["memory_size"])
            check_type(argname="argument params_and_secrets", value=params_and_secrets, expected_type=type_hints["params_and_secrets"])
            check_type(argname="argument profiling", value=profiling, expected_type=type_hints["profiling"])
            check_type(argname="argument profiling_group", value=profiling_group, expected_type=type_hints["profiling_group"])
            check_type(argname="argument recursive_loop", value=recursive_loop, expected_type=type_hints["recursive_loop"])
            check_type(argname="argument reserved_concurrent_executions", value=reserved_concurrent_executions, expected_type=type_hints["reserved_concurrent_executions"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument runtime_management_mode", value=runtime_management_mode, expected_type=type_hints["runtime_management_mode"])
            check_type(argname="argument security_groups", value=security_groups, expected_type=type_hints["security_groups"])
            check_type(argname="argument snap_start", value=snap_start, expected_type=type_hints["snap_start"])
            check_type(argname="argument system_log_level", value=system_log_level, expected_type=type_hints["system_log_level"])
            check_type(argname="argument system_log_level_v2", value=system_log_level_v2, expected_type=type_hints["system_log_level_v2"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument tracing", value=tracing, expected_type=type_hints["tracing"])
            check_type(argname="argument vpc", value=vpc, expected_type=type_hints["vpc"])
            check_type(argname="argument vpc_subnets", value=vpc_subnets, expected_type=type_hints["vpc_subnets"])
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
            check_type(argname="argument handler", value=handler, expected_type=type_hints["handler"])
            check_type(argname="argument runtime", value=runtime, expected_type=type_hints["runtime"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "code": code,
            "handler": handler,
            "runtime": runtime,
        }
        if max_event_age is not None:
            self._values["max_event_age"] = max_event_age
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if on_success is not None:
            self._values["on_success"] = on_success
        if retry_attempts is not None:
            self._values["retry_attempts"] = retry_attempts
        if adot_instrumentation is not None:
            self._values["adot_instrumentation"] = adot_instrumentation
        if allow_all_ipv6_outbound is not None:
            self._values["allow_all_ipv6_outbound"] = allow_all_ipv6_outbound
        if allow_all_outbound is not None:
            self._values["allow_all_outbound"] = allow_all_outbound
        if allow_public_subnet is not None:
            self._values["allow_public_subnet"] = allow_public_subnet
        if application_log_level is not None:
            self._values["application_log_level"] = application_log_level
        if application_log_level_v2 is not None:
            self._values["application_log_level_v2"] = application_log_level_v2
        if architecture is not None:
            self._values["architecture"] = architecture
        if code_signing_config is not None:
            self._values["code_signing_config"] = code_signing_config
        if current_version_options is not None:
            self._values["current_version_options"] = current_version_options
        if dead_letter_queue is not None:
            self._values["dead_letter_queue"] = dead_letter_queue
        if dead_letter_queue_enabled is not None:
            self._values["dead_letter_queue_enabled"] = dead_letter_queue_enabled
        if dead_letter_topic is not None:
            self._values["dead_letter_topic"] = dead_letter_topic
        if description is not None:
            self._values["description"] = description
        if environment is not None:
            self._values["environment"] = environment
        if environment_encryption is not None:
            self._values["environment_encryption"] = environment_encryption
        if ephemeral_storage_size is not None:
            self._values["ephemeral_storage_size"] = ephemeral_storage_size
        if events is not None:
            self._values["events"] = events
        if filesystem is not None:
            self._values["filesystem"] = filesystem
        if function_name is not None:
            self._values["function_name"] = function_name
        if initial_policy is not None:
            self._values["initial_policy"] = initial_policy
        if insights_version is not None:
            self._values["insights_version"] = insights_version
        if ipv6_allowed_for_dual_stack is not None:
            self._values["ipv6_allowed_for_dual_stack"] = ipv6_allowed_for_dual_stack
        if layers is not None:
            self._values["layers"] = layers
        if log_format is not None:
            self._values["log_format"] = log_format
        if logging_format is not None:
            self._values["logging_format"] = logging_format
        if log_group is not None:
            self._values["log_group"] = log_group
        if log_retention is not None:
            self._values["log_retention"] = log_retention
        if log_retention_retry_options is not None:
            self._values["log_retention_retry_options"] = log_retention_retry_options
        if log_retention_role is not None:
            self._values["log_retention_role"] = log_retention_role
        if memory_size is not None:
            self._values["memory_size"] = memory_size
        if params_and_secrets is not None:
            self._values["params_and_secrets"] = params_and_secrets
        if profiling is not None:
            self._values["profiling"] = profiling
        if profiling_group is not None:
            self._values["profiling_group"] = profiling_group
        if recursive_loop is not None:
            self._values["recursive_loop"] = recursive_loop
        if reserved_concurrent_executions is not None:
            self._values["reserved_concurrent_executions"] = reserved_concurrent_executions
        if role is not None:
            self._values["role"] = role
        if runtime_management_mode is not None:
            self._values["runtime_management_mode"] = runtime_management_mode
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if snap_start is not None:
            self._values["snap_start"] = snap_start
        if system_log_level is not None:
            self._values["system_log_level"] = system_log_level
        if system_log_level_v2 is not None:
            self._values["system_log_level_v2"] = system_log_level_v2
        if timeout is not None:
            self._values["timeout"] = timeout
        if tracing is not None:
            self._values["tracing"] = tracing
        if vpc is not None:
            self._values["vpc"] = vpc
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def max_event_age(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum age of a request that Lambda sends to a function for processing.

        Minimum: 60 seconds
        Maximum: 6 hours

        :default: Duration.hours(6)
        '''
        result = self._values.get("max_event_age")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def on_failure(self) -> typing.Optional["IDestination"]:
        '''The destination for failed invocations.

        :default: - no destination
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def on_success(self) -> typing.Optional["IDestination"]:
        '''The destination for successful invocations.

        :default: - no destination
        '''
        result = self._values.get("on_success")
        return typing.cast(typing.Optional["IDestination"], result)

    @builtins.property
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of times to retry when the function returns an error.

        Minimum: 0
        Maximum: 2

        :default: 2
        '''
        result = self._values.get("retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def adot_instrumentation(self) -> typing.Optional[AdotInstrumentationConfig]:
        '''Specify the configuration of AWS Distro for OpenTelemetry (ADOT) instrumentation.

        :default: - No ADOT instrumentation

        :see: https://aws-otel.github.io/docs/getting-started/lambda
        '''
        result = self._values.get("adot_instrumentation")
        return typing.cast(typing.Optional[AdotInstrumentationConfig], result)

    @builtins.property
    def allow_all_ipv6_outbound(self) -> typing.Optional[builtins.bool]:
        '''Whether to allow the Lambda to send all ipv6 network traffic.

        If set to true, there will only be a single egress rule which allows all
        outbound ipv6 traffic. If set to false, you must individually add traffic rules to allow the
        Lambda to connect to network targets using ipv6.

        Do not specify this property if the ``securityGroups`` or ``securityGroup`` property is set.
        Instead, configure ``allowAllIpv6Outbound`` directly on the security group.

        :default: false
        '''
        result = self._values.get("allow_all_ipv6_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def allow_all_outbound(self) -> typing.Optional[builtins.bool]:
        '''Whether to allow the Lambda to send all network traffic (except ipv6).

        If set to false, you must individually add traffic rules to allow the
        Lambda to connect to network targets.

        Do not specify this property if the ``securityGroups`` or ``securityGroup`` property is set.
        Instead, configure ``allowAllOutbound`` directly on the security group.

        :default: true
        '''
        result = self._values.get("allow_all_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def allow_public_subnet(self) -> typing.Optional[builtins.bool]:
        '''Lambda Functions in a public subnet can NOT access the internet.

        Use this property to acknowledge this limitation and still place the function in a public subnet.

        :default: false

        :see: https://stackoverflow.com/questions/52992085/why-cant-an-aws-lambda-function-inside-a-public-subnet-in-a-vpc-connect-to-the/52994841#52994841
        '''
        result = self._values.get("allow_public_subnet")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def application_log_level(self) -> typing.Optional[builtins.str]:
        '''(deprecated) Sets the application log level for the function.

        :default: "INFO"

        :deprecated: Use ``applicationLogLevelV2`` as a property instead.

        :stability: deprecated
        '''
        result = self._values.get("application_log_level")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def application_log_level_v2(self) -> typing.Optional[ApplicationLogLevel]:
        '''Sets the application log level for the function.

        :default: ApplicationLogLevel.INFO
        '''
        result = self._values.get("application_log_level_v2")
        return typing.cast(typing.Optional[ApplicationLogLevel], result)

    @builtins.property
    def architecture(self) -> typing.Optional[Architecture]:
        '''The system architectures compatible with this lambda function.

        :default: Architecture.X86_64
        '''
        result = self._values.get("architecture")
        return typing.cast(typing.Optional[Architecture], result)

    @builtins.property
    def code_signing_config(self) -> typing.Optional["ICodeSigningConfig"]:
        '''Code signing config associated with this function.

        :default: - Not Sign the Code
        '''
        result = self._values.get("code_signing_config")
        return typing.cast(typing.Optional["ICodeSigningConfig"], result)

    @builtins.property
    def current_version_options(self) -> typing.Optional["VersionOptions"]:
        '''Options for the ``lambda.Version`` resource automatically created by the ``fn.currentVersion`` method.

        :default: - default options as described in ``VersionOptions``
        '''
        result = self._values.get("current_version_options")
        return typing.cast(typing.Optional["VersionOptions"], result)

    @builtins.property
    def dead_letter_queue(self) -> typing.Optional[_IQueue_7ed6f679]:
        '''The SQS queue to use if DLQ is enabled.

        If SNS topic is desired, specify ``deadLetterTopic`` property instead.

        :default: - SQS queue with 14 day retention period if ``deadLetterQueueEnabled`` is ``true``
        '''
        result = self._values.get("dead_letter_queue")
        return typing.cast(typing.Optional[_IQueue_7ed6f679], result)

    @builtins.property
    def dead_letter_queue_enabled(self) -> typing.Optional[builtins.bool]:
        '''Enabled DLQ.

        If ``deadLetterQueue`` is undefined,
        an SQS queue with default options will be defined for your Function.

        :default: - false unless ``deadLetterQueue`` is set, which implies DLQ is enabled.
        '''
        result = self._values.get("dead_letter_queue_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def dead_letter_topic(self) -> typing.Optional[_ITopic_9eca4852]:
        '''The SNS topic to use as a DLQ.

        Note that if ``deadLetterQueueEnabled`` is set to ``true``, an SQS queue will be created
        rather than an SNS topic. Using an SNS topic as a DLQ requires this property to be set explicitly.

        :default: - no SNS topic
        '''
        result = self._values.get("dead_letter_topic")
        return typing.cast(typing.Optional[_ITopic_9eca4852], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the function.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Key-value pairs that Lambda caches and makes available for your Lambda functions.

        Use environment variables to apply configuration changes, such
        as test and production environment configurations, without changing your
        Lambda function source code.

        :default: - No environment variables.
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def environment_encryption(self) -> typing.Optional[_IKey_5f11635f]:
        '''The AWS KMS key that's used to encrypt your function's environment variables.

        :default: - AWS Lambda creates and uses an AWS managed customer master key (CMK).
        '''
        result = self._values.get("environment_encryption")
        return typing.cast(typing.Optional[_IKey_5f11635f], result)

    @builtins.property
    def ephemeral_storage_size(self) -> typing.Optional[_Size_7b441c34]:
        '''The size of the function’s /tmp directory in MiB.

        :default: 512 MiB
        '''
        result = self._values.get("ephemeral_storage_size")
        return typing.cast(typing.Optional[_Size_7b441c34], result)

    @builtins.property
    def events(self) -> typing.Optional[typing.List["IEventSource"]]:
        '''Event sources for this function.

        You can also add event sources using ``addEventSource``.

        :default: - No event sources.
        '''
        result = self._values.get("events")
        return typing.cast(typing.Optional[typing.List["IEventSource"]], result)

    @builtins.property
    def filesystem(self) -> typing.Optional[FileSystem]:
        '''The filesystem configuration for the lambda function.

        :default: - will not mount any filesystem
        '''
        result = self._values.get("filesystem")
        return typing.cast(typing.Optional[FileSystem], result)

    @builtins.property
    def function_name(self) -> typing.Optional[builtins.str]:
        '''A name for the function.

        :default:

        - AWS CloudFormation generates a unique physical ID and uses that
        ID for the function's name. For more information, see Name Type.
        '''
        result = self._values.get("function_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def initial_policy(self) -> typing.Optional[typing.List[_PolicyStatement_0fe33853]]:
        '''Initial policy statements to add to the created Lambda Role.

        You can call ``addToRolePolicy`` to the created lambda to add statements post creation.

        :default: - No policy statements are added to the created Lambda role.
        '''
        result = self._values.get("initial_policy")
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_0fe33853]], result)

    @builtins.property
    def insights_version(self) -> typing.Optional["LambdaInsightsVersion"]:
        '''Specify the version of CloudWatch Lambda insights to use for monitoring.

        :default: - No Lambda Insights

        :see: https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-Getting-Started-docker.html
        '''
        result = self._values.get("insights_version")
        return typing.cast(typing.Optional["LambdaInsightsVersion"], result)

    @builtins.property
    def ipv6_allowed_for_dual_stack(self) -> typing.Optional[builtins.bool]:
        '''Allows outbound IPv6 traffic on VPC functions that are connected to dual-stack subnets.

        Only used if 'vpc' is supplied.

        :default: false
        '''
        result = self._values.get("ipv6_allowed_for_dual_stack")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def layers(self) -> typing.Optional[typing.List["ILayerVersion"]]:
        '''A list of layers to add to the function's execution environment.

        You can configure your Lambda function to pull in
        additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies
        that can be used by multiple functions.

        :default: - No layers.
        '''
        result = self._values.get("layers")
        return typing.cast(typing.Optional[typing.List["ILayerVersion"]], result)

    @builtins.property
    def log_format(self) -> typing.Optional[builtins.str]:
        '''(deprecated) Sets the logFormat for the function.

        :default: "Text"

        :deprecated: Use ``loggingFormat`` as a property instead.

        :stability: deprecated
        '''
        result = self._values.get("log_format")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def logging_format(self) -> typing.Optional["LoggingFormat"]:
        '''Sets the loggingFormat for the function.

        :default: LoggingFormat.TEXT
        '''
        result = self._values.get("logging_format")
        return typing.cast(typing.Optional["LoggingFormat"], result)

    @builtins.property
    def log_group(self) -> typing.Optional[_ILogGroup_3c4fa718]:
        '''The log group the function sends logs to.

        By default, Lambda functions send logs to an automatically created default log group named /aws/lambda/.
        However you cannot change the properties of this auto-created log group using the AWS CDK, e.g. you cannot set a different log retention.

        Use the ``logGroup`` property to create a fully customizable LogGroup ahead of time, and instruct the Lambda function to send logs to it.

        Providing a user-controlled log group was rolled out to commercial regions on 2023-11-16.
        If you are deploying to another type of region, please check regional availability first.

        :default: ``/aws/lambda/${this.functionName}`` - default log group created by Lambda
        '''
        result = self._values.get("log_group")
        return typing.cast(typing.Optional[_ILogGroup_3c4fa718], result)

    @builtins.property
    def log_retention(self) -> typing.Optional[_RetentionDays_070f99f0]:
        '''The number of days log events are kept in CloudWatch Logs.

        When updating
        this property, unsetting it doesn't remove the log retention policy. To
        remove the retention policy, set the value to ``INFINITE``.

        This is a legacy API and we strongly recommend you move away from it if you can.
        Instead create a fully customizable log group with ``logs.LogGroup`` and use the ``logGroup`` property
        to instruct the Lambda function to send logs to it.
        Migrating from ``logRetention`` to ``logGroup`` will cause the name of the log group to change.
        Users and code and referencing the name verbatim will have to adjust.

        In AWS CDK code, you can access the log group name directly from the LogGroup construct::

           import aws_cdk.aws_logs as logs

           # my_log_group: logs.LogGroup

           my_log_group.log_group_name

        :default: logs.RetentionDays.INFINITE
        '''
        result = self._values.get("log_retention")
        return typing.cast(typing.Optional[_RetentionDays_070f99f0], result)

    @builtins.property
    def log_retention_retry_options(
        self,
    ) -> typing.Optional["LogRetentionRetryOptions"]:
        '''When log retention is specified, a custom resource attempts to create the CloudWatch log group.

        These options control the retry policy when interacting with CloudWatch APIs.

        This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can.
        ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it.

        :default: - Default AWS SDK retry options.
        '''
        result = self._values.get("log_retention_retry_options")
        return typing.cast(typing.Optional["LogRetentionRetryOptions"], result)

    @builtins.property
    def log_retention_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The IAM role for the Lambda function associated with the custom resource that sets the retention policy.

        This is a legacy API and we strongly recommend you migrate to ``logGroup`` if you can.
        ``logGroup`` allows you to create a fully customizable log group and instruct the Lambda function to send logs to it.

        :default: - A new role is created.
        '''
        result = self._values.get("log_retention_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def memory_size(self) -> typing.Optional[jsii.Number]:
        '''The amount of memory, in MB, that is allocated to your Lambda function.

        Lambda uses this value to proportionally allocate the amount of CPU
        power. For more information, see Resource Model in the AWS Lambda
        Developer Guide.

        :default: 128
        '''
        result = self._values.get("memory_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def params_and_secrets(self) -> typing.Optional["ParamsAndSecretsLayerVersion"]:
        '''Specify the configuration of Parameters and Secrets Extension.

        :default: - No Parameters and Secrets Extension

        :see: https://docs.aws.amazon.com/systems-manager/latest/userguide/ps-integration-lambda-extensions.html
        '''
        result = self._values.get("params_and_secrets")
        return typing.cast(typing.Optional["ParamsAndSecretsLayerVersion"], result)

    @builtins.property
    def profiling(self) -> typing.Optional[builtins.bool]:
        '''Enable profiling.

        :default: - No profiling.

        :see: https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
        '''
        result = self._values.get("profiling")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def profiling_group(self) -> typing.Optional[_IProfilingGroup_0bba72c4]:
        '''Profiling Group.

        :default: - A new profiling group will be created if ``profiling`` is set.

        :see: https://docs.aws.amazon.com/codeguru/latest/profiler-ug/setting-up-lambda.html
        '''
        result = self._values.get("profiling_group")
        return typing.cast(typing.Optional[_IProfilingGroup_0bba72c4], result)

    @builtins.property
    def recursive_loop(self) -> typing.Optional["RecursiveLoop"]:
        '''Sets the Recursive Loop Protection for Lambda Function.

        It lets Lambda detect and terminate unintended recursive loops.

        :default: RecursiveLoop.Terminate
        '''
        result = self._values.get("recursive_loop")
        return typing.cast(typing.Optional["RecursiveLoop"], result)

    @builtins.property
    def reserved_concurrent_executions(self) -> typing.Optional[jsii.Number]:
        '''The maximum of concurrent executions you want to reserve for the function.

        :default: - No specific limit - account limit.

        :see: https://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html
        '''
        result = self._values.get("reserved_concurrent_executions")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''Lambda execution role.

        This is the role that will be assumed by the function upon execution.
        It controls the permissions that the function will have. The Role must
        be assumable by the 'lambda.amazonaws.com' service principal.

        The default Role automatically has permissions granted for Lambda execution. If you
        provide a Role, you must add the relevant AWS managed policies yourself.

        The relevant managed policies are "service-role/AWSLambdaBasicExecutionRole" and
        "service-role/AWSLambdaVPCAccessExecutionRole".

        :default:

        - A unique role will be generated for this lambda function.
        Both supplied and generated roles can always be changed by calling ``addToRolePolicy``.
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def runtime_management_mode(self) -> typing.Optional["RuntimeManagementMode"]:
        '''Sets the runtime management configuration for a function's version.

        :default: Auto
        '''
        result = self._values.get("runtime_management_mode")
        return typing.cast(typing.Optional["RuntimeManagementMode"], result)

    @builtins.property
    def security_groups(self) -> typing.Optional[typing.List[_ISecurityGroup_acf8a799]]:
        '''The list of security groups to associate with the Lambda's network interfaces.

        Only used if 'vpc' is supplied.

        :default:

        - If the function is placed within a VPC and a security group is
        not specified, either by this or securityGroup prop, a dedicated security
        group will be created for this function.
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[_ISecurityGroup_acf8a799]], result)

    @builtins.property
    def snap_start(self) -> typing.Optional["SnapStartConf"]:
        '''Enable SnapStart for Lambda Function.

        SnapStart is currently supported for Java 11, Java 17, Python 3.12, Python 3.13, and .NET 8 runtime

        :default: - No snapstart
        '''
        result = self._values.get("snap_start")
        return typing.cast(typing.Optional["SnapStartConf"], result)

    @builtins.property
    def system_log_level(self) -> typing.Optional[builtins.str]:
        '''(deprecated) Sets the system log level for the function.

        :default: "INFO"

        :deprecated: Use ``systemLogLevelV2`` as a property instead.

        :stability: deprecated
        '''
        result = self._values.get("system_log_level")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def system_log_level_v2(self) -> typing.Optional["SystemLogLevel"]:
        '''Sets the system log level for the function.

        :default: SystemLogLevel.INFO
        '''
        result = self._values.get("system_log_level_v2")
        return typing.cast(typing.Optional["SystemLogLevel"], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The function execution time (in seconds) after which Lambda terminates the function.

        Because the execution time affects cost, set this value
        based on the function's expected execution time.

        :default: Duration.seconds(3)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def tracing(self) -> typing.Optional["Tracing"]:
        '''Enable AWS X-Ray Tracing for Lambda Function.

        :default: Tracing.Disabled
        '''
        result = self._values.get("tracing")
        return typing.cast(typing.Optional["Tracing"], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_IVpc_f30d5663]:
        '''VPC network to place Lambda network interfaces.

        Specify this if the Lambda function needs to access resources in a VPC.
        This is required when ``vpcSubnets`` is specified.

        :default: - Function is not placed within a VPC.
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_IVpc_f30d5663], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_SubnetSelection_e57d76df]:
        '''Where to place the network interfaces within the VPC.

        This requires ``vpc`` to be specified in order for interfaces to actually be
        placed in the subnets. If ``vpc`` is not specify, this will raise an error.

        Note: Internet access for Lambda Functions requires a NAT Gateway, so picking
        public subnets is not allowed (unless ``allowPublicSubnet`` is set to ``true``).

        :default: - the Vpc default strategy if not specified
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_SubnetSelection_e57d76df], result)

    @builtins.property
    def code(self) -> Code:
        '''The source code of your Lambda function.

        You can point to a file in an
        Amazon Simple Storage Service (Amazon S3) bucket or specify your source
        code as inline text.
        '''
        result = self._values.get("code")
        assert result is not None, "Required property 'code' is missing"
        return typing.cast(Code, result)

    @builtins.property
    def handler(self) -> builtins.str:
        '''The name of the method within your code that Lambda calls to execute your function.

        The format includes the file name. It can also include
        namespaces and other qualifiers, depending on the runtime.
        For more information, see https://docs.aws.amazon.com/lambda/latest/dg/foundation-progmodel.html.

        Use ``Handler.FROM_IMAGE`` when defining a function from a Docker image.

        NOTE: If you specify your source code as inline text by specifying the
        ZipFile property within the Code property, specify index.function_name as
        the handler.
        '''
        result = self._values.get("handler")
        assert result is not None, "Required property 'handler' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def runtime(self) -> "Runtime":
        '''The runtime environment for the Lambda function that you are uploading.

        For valid values, see the Runtime property in the AWS Lambda Developer
        Guide.

        Use ``Runtime.FROM_IMAGE`` when defining a function from a Docker image.
        '''
        result = self._values.get("runtime")
        assert result is not None, "Required property 'runtime' is missing"
        return typing.cast("Runtime", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_lambda.FunctionUrlAuthType")
class FunctionUrlAuthType(enum.Enum):
    '''The auth types for a function url.

    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_lambda as lambda_
        
        # fn: lambda.Function
        
        fn_url = fn.add_function_url(auth_type=lambda_.FunctionUrlAuthType.NONE)
        
        cloudfront.Distribution(self, "Distribution",
            default_behavior=cloudfront.BehaviorOptions(origin=origins.FunctionUrlOrigin(fn_url))
        )
    '''

    AWS_IAM = "AWS_IAM"
    '''Restrict access to authenticated IAM users only.'''
    NONE = "NONE"
    '''Bypass IAM authentication to create a public endpoint.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.FunctionUrlCorsOptions",
    jsii_struct_bases=[],
    name_mapping={
        "allow_credentials": "allowCredentials",
        "allowed_headers": "allowedHeaders",
        "allowed_methods": "allowedMethods",
        "allowed_origins": "allowedOrigins",
        "exposed_headers": "exposedHeaders",
        "max_age": "maxAge",
    },
)
class FunctionUrlCorsOptions:
    def __init__(
        self,
        *,
        allow_credentials: typing.Optional[builtins.bool] = None,
        allowed_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        allowed_methods: typing.Optional[typing.Sequence["HttpMethod"]] = None,
        allowed_origins: typing.Optional[typing.Sequence[builtins.str]] = None,
        exposed_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        max_age: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''Specifies a cross-origin access property for a function URL.

        :param allow_credentials: Whether to allow cookies or other credentials in requests to your function URL. Default: false
        :param allowed_headers: Headers that are specified in the Access-Control-Request-Headers header. Default: - No headers allowed.
        :param allowed_methods: An HTTP method that you allow the origin to execute. Default: - [HttpMethod.ALL]
        :param allowed_origins: One or more origins you want customers to be able to access the bucket from. Default: - No origins allowed.
        :param exposed_headers: One or more headers in the response that you want customers to be able to access from their applications. Default: - No headers exposed.
        :param max_age: The time in seconds that your browser is to cache the preflight response for the specified resource. Default: - Browser default of 5 seconds.

        :exampleMetadata: infused

        Example::

            # fn: lambda.Function
            
            
            fn.add_function_url(
                auth_type=lambda_.FunctionUrlAuthType.NONE,
                cors=lambda.FunctionUrlCorsOptions(
                    # Allow this to be called from websites on https://example.com.
                    # Can also be ['*'] to allow all domain.
                    allowed_origins=["https://example.com"]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0fbb41369d8abd77525f211bdc5c68d060bbaa2a32a2179b9d85fece80edf38a)
            check_type(argname="argument allow_credentials", value=allow_credentials, expected_type=type_hints["allow_credentials"])
            check_type(argname="argument allowed_headers", value=allowed_headers, expected_type=type_hints["allowed_headers"])
            check_type(argname="argument allowed_methods", value=allowed_methods, expected_type=type_hints["allowed_methods"])
            check_type(argname="argument allowed_origins", value=allowed_origins, expected_type=type_hints["allowed_origins"])
            check_type(argname="argument exposed_headers", value=exposed_headers, expected_type=type_hints["exposed_headers"])
            check_type(argname="argument max_age", value=max_age, expected_type=type_hints["max_age"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allow_credentials is not None:
            self._values["allow_credentials"] = allow_credentials
        if allowed_headers is not None:
            self._values["allowed_headers"] = allowed_headers
        if allowed_methods is not None:
            self._values["allowed_methods"] = allowed_methods
        if allowed_origins is not None:
            self._values["allowed_origins"] = allowed_origins
        if exposed_headers is not None:
            self._values["exposed_headers"] = exposed_headers
        if max_age is not None:
            self._values["max_age"] = max_age

    @builtins.property
    def allow_credentials(self) -> typing.Optional[builtins.bool]:
        '''Whether to allow cookies or other credentials in requests to your function URL.

        :default: false
        '''
        result = self._values.get("allow_credentials")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def allowed_headers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Headers that are specified in the Access-Control-Request-Headers header.

        :default: - No headers allowed.
        '''
        result = self._values.get("allowed_headers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def allowed_methods(self) -> typing.Optional[typing.List["HttpMethod"]]:
        '''An HTTP method that you allow the origin to execute.

        :default: - [HttpMethod.ALL]
        '''
        result = self._values.get("allowed_methods")
        return typing.cast(typing.Optional[typing.List["HttpMethod"]], result)

    @builtins.property
    def allowed_origins(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more origins you want customers to be able to access the bucket from.

        :default: - No origins allowed.
        '''
        result = self._values.get("allowed_origins")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def exposed_headers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''One or more headers in the response that you want customers to be able to access from their applications.

        :default: - No headers exposed.
        '''
        result = self._values.get("exposed_headers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def max_age(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The time in seconds that your browser is to cache the preflight response for the specified resource.

        :default: - Browser default of 5 seconds.
        '''
        result = self._values.get("max_age")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionUrlCorsOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.FunctionUrlOptions",
    jsii_struct_bases=[],
    name_mapping={
        "auth_type": "authType",
        "cors": "cors",
        "invoke_mode": "invokeMode",
    },
)
class FunctionUrlOptions:
    def __init__(
        self,
        *,
        auth_type: typing.Optional[FunctionUrlAuthType] = None,
        cors: typing.Optional[typing.Union[FunctionUrlCorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        invoke_mode: typing.Optional["InvokeMode"] = None,
    ) -> None:
        '''Options to add a url to a Lambda function.

        :param auth_type: The type of authentication that your function URL uses. Default: FunctionUrlAuthType.AWS_IAM
        :param cors: The cross-origin resource sharing (CORS) settings for your function URL. Default: - No CORS configuration.
        :param invoke_mode: The type of invocation mode that your Lambda function uses. Default: InvokeMode.BUFFERED

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_lambda as lambda_
            
            # fn: lambda.Function
            
            fn_url = fn.add_function_url(auth_type=lambda_.FunctionUrlAuthType.NONE)
            
            cloudfront.Distribution(self, "Distribution",
                default_behavior=cloudfront.BehaviorOptions(origin=origins.FunctionUrlOrigin(fn_url))
            )
        '''
        if isinstance(cors, dict):
            cors = FunctionUrlCorsOptions(**cors)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3a2aa6d01661bddcd688528df01856d4a75c7edf3cea37fef49083f1084593b7)
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument cors", value=cors, expected_type=type_hints["cors"])
            check_type(argname="argument invoke_mode", value=invoke_mode, expected_type=type_hints["invoke_mode"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if cors is not None:
            self._values["cors"] = cors
        if invoke_mode is not None:
            self._values["invoke_mode"] = invoke_mode

    @builtins.property
    def auth_type(self) -> typing.Optional[FunctionUrlAuthType]:
        '''The type of authentication that your function URL uses.

        :default: FunctionUrlAuthType.AWS_IAM
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[FunctionUrlAuthType], result)

    @builtins.property
    def cors(self) -> typing.Optional[FunctionUrlCorsOptions]:
        '''The cross-origin resource sharing (CORS) settings for your function URL.

        :default: - No CORS configuration.
        '''
        result = self._values.get("cors")
        return typing.cast(typing.Optional[FunctionUrlCorsOptions], result)

    @builtins.property
    def invoke_mode(self) -> typing.Optional["InvokeMode"]:
        '''The type of invocation mode that your Lambda function uses.

        :default: InvokeMode.BUFFERED
        '''
        result = self._values.get("invoke_mode")
        return typing.cast(typing.Optional["InvokeMode"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionUrlOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_lambda.FunctionUrlProps",
    jsii_struct_bases=[FunctionUrlOptions],
    name_mapping={
        "auth_type": "authType",
        "cors": "cors",
        "invoke_mode": "invokeMode",
        "function": "function",
    },
)
class FunctionUrlProps(FunctionUrlOptions):
    def __init__(
        self,
        *,
        auth_type: typing.Optional[FunctionUrlAuthType] = None,
        cors: typing.Optional[typing.Union[FunctionUrlCorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        invoke_mode: typing.Optional["InvokeMode"] = None,
        function: "IFunction",
    ) -> None:
        '''Properties for a FunctionUrl.

        :param auth_type: The type of authentication that your function URL uses. Default: FunctionUrlAuthType.AWS_IAM
        :param cors: The cross-origin resource sharing (CORS) settings for your function URL. Default: - No CORS configuration.
        :param invoke_mode: The type of invocation mode that your Lambda function uses. Default: InvokeMode.BUFFERED
        :param function: The function to which this url refers. It can also be an ``Alias`` but not a ``Version``.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_lambda as lambda_
            
            # function_: lambda.Function
            
            function_url_props = lambda.FunctionUrlProps(
                function=function_,
            
                # the properties below are optional
                auth_type=lambda_.FunctionUrlAuthType.AWS_IAM,
                cors=lambda.FunctionUrlCorsOptions(
                    allow_credentials=False,
                    allowed_headers=["allowedHeaders"],
                    allowed_methods=[lambda_.HttpMethod.GET],
                    allowed_origins=["allowedOrigins"],
                    exposed_headers=["exposedHeaders"],
                    max_age=cdk.Duration.minutes(30)
                ),
                invoke_mode=lambda_.InvokeMode.BUFFERED
            )
        '''
        if isinstance(cors, dict):
            cors = FunctionUrlCorsOptions(**cors)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__47778dc8c2df10c3a70bab19ed5dc69858b33e0f625dbf96a1c234d7d0353276)
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument cors", value=cors, expected_type=type_hints["cors"])
            check_type(argname="argument invoke_mode", value=invoke_mode, expected_type=type_hints["invoke_mode"])
            check_type(argname="argument function", value=function, expected_type=type_hints["function"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function": function,
        }
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if cors is not None:
            self._values["cors"] = cors
        if invoke_mode is not None:
            self._values["invoke_mode"] = invoke_mode

    @builtins.property
    def auth_type(self) -> typing.Optional[FunctionUrlAuthType]:
        '''The type of authentication that your function URL uses.

        :default: FunctionUrlAuthType.AWS_IAM
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[FunctionUrlAuthType], result)

    @builtins.property
    def cors(self) -> typing.Optional[FunctionUrlCorsOptions]:
        '''The cross-origin resource sharing (CORS) settings for your function URL.

        :default: - No CORS configuration.
        '''
        result = self._values.get("cors")
        return typing.cast(typing.Optional[FunctionUrlCorsOptions], result)

    @builtins.property
    def invoke_mode(self) -> typing.Optional["InvokeMode"]:
        '''The type of invocation mode that your Lambda function uses.

        :default: InvokeMode.BUFFERED
        '''
        result = self._values.get("invoke_mode")
        return typing.cast(typing.Optional["InvokeMode"], result)

    @builtins.property
    def function(self) -> "IFunction":
        '''The function to which this url refers.

        It can also be an ``Alias`` but not a ``Version``.
        '''
        result = self._values.get("function")
        assert result is not None, "Required property 'function' is missing"
        return typing.cast("IFunction", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionUrlProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IAspect_118c810a)
class FunctionVersionUpgrade(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_lambda.FunctionVersionUpgrade",
):
    '''Aspect for upgrading function versions when the provided feature flag is enabled.

    This can be necessary when the feature flag
    changes the function hash, as such changes must be associated with a new
    version. This aspect will change the function description in these cases,
    which "validates" the new function hash.

    :exampleMetadata: infused

    Example::

        stack = Stack()
        Aspects.of(stack).add(lambda_.FunctionVersionUpgrade(LAMBDA_RECOGNIZE_VERSION_PROPS))
    '''

    def __init__(
        self,
        feature_flag: builtins.str,
        enabled: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param feature_flag: -
        :param enabled: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a253018a976dd453cfa6b5cc5f6a6558c50144cec0682b3e38c91fcb99a513be)
            check_type(argname="argument feature_flag", value=feature_flag, expected_type=type_hints["feature_flag"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
        jsii.create(self.__class__, self, [feature_flag, enabled])

    @jsii.member(jsii_name="visit")
    def visit(self, node: _constructs_77d1e7e8.IConstruct) -> None:
        '''All aspects can visit an IConstruct.

        :param node: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0977713870c8acc93c3134ab9f62502cba706eb2aa67dfc882407ea66f61eb72)
            check_type(argname="argument node", value=node, expected_type=type_hints["node"])
        return typing.cast(None, jsii.invoke(self, "visit", [node]))


class Handler(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.aws_lambda.Handler"):
    '''Lambda function handler.'''

    @jsii.python.classproperty
    @jsii.member(jsii_name="FROM_IMAGE")
    def FROM_IMAGE(cls) -> builtins.str:
        '''A special handler when the function handler is part of a Docker image.'''
        return typing.cast(builtins.str, jsii.sget(cls, "FROM_IMAGE"))


@jsii.enum(jsii_type="aws-cdk-lib.aws_lambda.HttpMethod")
class HttpMethod(enum.Enum):
    '''All http request methods.'''

    GET = "GET"
    '''The GET method requests a representation of the specified resource.'''
    PUT = "PUT"
    '''The PUT method replaces all current representations of the target resource with the request payload.'''
    HEAD = "HEAD"
    '''The HEAD method asks for a response identical to that of a GET request, but without the response body.'''
    POST = "POST"
    '''The POST method is used to submit an entity to the specified resource, often causing a change in state or side effects on the server.'''
    DELETE = "DELETE"
    '''The DELETE method deletes the specified resource.'''
    PATCH = "PATCH"
    '''The PATCH method applies partial modifications to a resource.'''
    OPTIONS = "OPTIONS"
    '''The OPTIONS method describes the communication options for the target resource.'''
    ALL = "ALL"
    '''The wildcard entry to allow all methods.'''


@jsii.interface(jsii_type="aws-cdk-lib.aws_lambda.ICodeSigningConfig")
class ICodeSigningConfig(_IResource_c80c4260, typing_extensions.Protocol):
    '''A Code Signing Config.'''

    @builtins.property
    @jsii.member(jsii_name="codeSigningConfigArn")
    def code_signing_config_arn(self) -> builtins.str:
        '''The ARN of Code Signing Config.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="codeSigningConfigId")
    def code_signing_config_id(self) -> builtins.str:
        '''The id of Code Signing Config.

        :attribute: true
        '''
        ...


class _ICodeSigningConfigProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''A Code Signing Config.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_lambda.ICodeSigningConfig"

    @builtins.property
    @jsii.member(jsii_name="codeSigningConfigArn")
    def code_signing_config_arn(self) -> builtins.str:
        '''The ARN of Code Signing Config.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "codeSigningConfigArn"))

    @builtins.property
    @jsii.member(jsii_name="codeSigningConfigId")
    def code_signing_config_id(self) -> builtins.str:
        '''The id of Code Signing Config.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "codeSigningConfigId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ICodeSigningConfig).__jsii_proxy_class__ = lambda : _ICodeSigningConfigProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_lambda.IDestination")
class IDestination(typing_extensions.Protocol):
    '''A Lambda destination.'''

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _constructs_77d1e7e8.Construct,
        fn: "IFunction",
        *,
        type: DestinationType,
    ) -> DestinationConfig:
        '''Binds this destination to the Lambda function.

        :param scope: -
        :param fn: -
        :param type: The destination type.
        '''
        ...


class _IDestinationProxy:
    '''A Lambda destination.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_lambda.IDestination"

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _constructs_77d1e7e8.Construct,
        fn: "IFunction",
        *,
        type: DestinationType,
    ) -> DestinationConfig:
        '''Binds this destination to the Lambda function.

        :param scope: -
        :param fn: -
        :param type: The destination type.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__616c8be191a72476ac6a26f4d9a6a7a4d902eba991c038bc7103501ce9812693)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument fn", value=fn, expected_type=type_hints["fn"])
        options = DestinationOptions(type=type)

        return typing.cast(DestinationConfig, jsii.invoke(self, "bind", [scope, fn, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IDestination).__jsii_proxy_class__ = lambda : _IDestinationProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_lambda.IEventSource")
class IEventSource(typing_extensions.Protocol):
    '''An abstract class which represents an AWS Lambda event source.'''

    @jsii.member(jsii_name="bind")
    def bind(self, target: "IFunction") -> None:
        '''Called by ``lambda.addEventSource`` to allow the event source to bind to this function.

        :param target: That lambda function to bind to.
        '''
        ...


class _IEventSourceProxy:
    '''An abstract class which represents an AWS Lambda event source.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_lambda.IEventSource"

    @jsii.member(jsii_name="bind")
    def bind(self, target: "IFunction") -> None:
        '''Called by ``lambda.addEventSource`` to allow the event source to bind to this function.

        :param target: That lambda function to bind to.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7e218810764d8a7de205f86f0ac5d969d1a7fbea24b5ed951f1967b7ef18a8bd)
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        return typing.cast(None, jsii.invoke(self, "bind", [target]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEventSource).__jsii_proxy_class__ = lambda : _IEventSourceProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_lambda.IEventSourceDlq")
class IEventSourceDlq(typing_extensions.Protocol):
    '''A DLQ for an event source.'''

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        target: "IEventSourceMapping",
        target_handler: "IFunction",
    ) -> DlqDestinationConfig:
        '''Returns the DLQ destination config of the DLQ.

        :param target: -
        :param target_handler: -
        '''
        ...


class _IEventSourceDlqProxy:
    '''A DLQ for an event source.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_lambda.IEventSourceDlq"

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        target: "IEventSourceMapping",
        target_handler: "IFunction",
    ) -> DlqDestinationConfig:
        '''Returns the DLQ destination config of the DLQ.

        :param target: -
        :param target_handler: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6606d66e2e03f7c17fbcbb3ef40d22586d15236fc21bc250dc4534d120f7ad4b)
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
            check_type(argname="argument target_handler", value=target_handler, expected_type=type_hints["target_handler"])
        return typing.cast(DlqDestinationConfig, jsii.invoke(self, "bind", [target, target_handler]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEventSourceDlq).__jsii_proxy_class__ = lambda : _IEventSourceDlqProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_lambda.IEventSourceMapping")
class IEventSourceMapping(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents an event source mapping for a lambda function.

    :see: https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventsourcemapping.html
    '''

    @builtins.property
    @jsii.member(jsii_name="eventSourceMappingArn")
    def event_source_mapping_arn(self) -> builtins.str:
        '''The ARN of the event source mapping (i.e. arn:aws:lambda:region:account-id:event-source-mapping/event-source-mapping-id).'''
        ...

    @builtins.property
    @jsii.member(jsii_name="eventSourceMappingId")
    def event_source_mapping_id(self) -> builtins.str:
        '''The identifier for this EventSourceMapping.

        :attribute: true
        '''
        ...


class _IEventSourceMappingProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents an event source mapping for a lambda function.

    :see: https://docs.aws.amazon.com/lambda/latest/dg/invocation-eventsourcemapping.html
    '''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_lambda.IEventSourceMapping"

    @builtins.property
    @jsii.member(jsii_name="eventSourceMappingArn")
    def event_source_mapping_arn(self) -> builtins.str:
        '''The ARN of the event source mapping (i.e. arn:aws:lambda:region:account-id:event-source-mapping/event-source-mapping-id).'''
        return typing.cast(builtins.str, jsii.get(self, "eventSourceMappingArn"))

    @builtins.property
    @jsii.member(jsii_name="eventSourceMappingId")
    def event_source_mapping_id(self) -> builtins.str:
        '''The identifier for this EventSourceMapping.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "eventSourceMappingId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEventSourceMapping).__jsii_proxy_class__ = lambda : _IEventSourceMappingProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_lambda.IFunction")
class IFunction(
    _IResource_c80c4260,
    _IConnectable_10015a05,
    _IGrantable_71c4f5de,
    typing_extensions.Protocol,
):
    @builtins.property
    @jsii.member(jsii_name="architecture")
    def architecture(self) -> Architecture:
        '''The system architectures compatible with this lambda function.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="functionArn")
    def function_arn(self) -> builtins.str:
        '''The ARN of the function.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the function.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="isBoundToVpc")
    def is_bound_to_vpc(self) -> builtins.bool:
        '''Whether or not this Lambda function was bound to a VPC.

        If this is is ``false``, trying to access the ``connections`` object will fail.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="latestVersion")
    def latest_version(self) -> "IVersion":
        '''The ``$LATEST`` version of this function.

        Note that this is reference to a non-specific AWS Lambda version, which
        means the function this version refers to can return different results in
        different invocations.

        To obtain a reference to an explicit version which references the current
        function configuration, use ``lambdaFunction.currentVersion`` instead.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="permissionsNode")
    def permissions_node(self) -> _constructs_77d1e7e8.Node:
        '''The construct node where permissions are attached.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="resourceArnsForGrantInvoke")
    def resource_arns_for_grant_invoke(self) -> typing.List[builtins.str]:
        '''The ARN(s) to put into the resource field of the generated IAM policy for grantInvoke().

        This property is for cdk modules to consume only. You should not need to use this property.
        Instead, use grantInvoke() directly.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The IAM role associated with this function.'''
        ...

    @jsii.member(jsii_name="addEventSource")
    def add_event_source(self, source: IEventSource) -> None:
        '''Adds an event source to this function.

        Event sources are implemented in the aws-cdk-lib/aws-lambda-event-sources module.

        The following example adds an SQS Queue as an event source::

           import { SqsEventSource } from 'aws-cdk-lib/aws-lambda-event-sources';
           myFunction.addEventSource(new SqsEventSource(myQueue));

        :param source: -
        '''
        ...

    @jsii.member(jsii_name="addEventSourceMapping")
    def add_event_source_mapping(
        self,
        id: builtins.str,
        *,
        batch_size: typing.Optional[jsii.Number] = None,
        bisect_batch_on_error: typing.Optional[builtins.bool] = None,
        enabled: typing.Optional[builtins.bool] = None,
        event_source_arn: typing.Optional[builtins.str] = None,
        filter_encryption: typing.Optional[_IKey_5f11635f] = None,
        filters: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        kafka_bootstrap_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        kafka_consumer_group_id: typing.Optional[builtins.str] = None,
        kafka_topic: typing.Optional[builtins.str] = None,
        max_batching_window: typing.Optional[_Duration_4839e8c3] = None,
        max_concurrency: typing.Optional[jsii.Number] = None,
        max_record_age: typing.Optional[_Duration_4839e8c3] = None,
        metrics_config: typing.Optional[typing.Union["MetricsConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        on_failure: typing.Optional[IEventSourceDlq] = None,
        parallelization_factor: typing.Optional[jsii.Number] = None,
        provisioned_poller_config: typing.Optional[typing.Union["ProvisionedPollerConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        report_batch_item_failures: typing.Optional[builtins.bool] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
        source_access_configurations: typing.Optional[typing.Sequence[typing.Union["SourceAccessConfiguration", typing.Dict[builtins.str, typing.Any]]]] = None,
        starting_position: typing.Optional["StartingPosition"] = None,
        starting_position_timestamp: typing.Optional[jsii.Number] = None,
        support_s3_on_failure_destination: typing.Optional[builtins.bool] = None,
        tumbling_window: typing.Optional[_Duration_4839e8c3] = None,
    ) -> "EventSourceMapping":
        '''Adds an event source that maps to this AWS Lambda function.

        :param id: construct ID.
        :param batch_size: The largest number of records that AWS Lambda will retrieve from your event source at the time of invoking your function. Your function receives an event with all the retrieved records. Valid Range: Minimum value of 1. Maximum value of 10000. Default: - Amazon Kinesis, Amazon DynamoDB, and Amazon MSK is 100 records. The default for Amazon SQS is 10 messages. For standard SQS queues, the maximum is 10,000. For FIFO SQS queues, the maximum is 10.
        :param bisect_batch_on_error: If the function returns an error, split the batch in two and retry. Default: false
        :param enabled: Set to false to disable the event source upon creation. Default: true
        :param event_source_arn: The Amazon Resource Name (ARN) of the event source. Any record added to this stream can invoke the Lambda function. Default: - not set if using a self managed Kafka cluster, throws an error otherwise
        :param filter_encryption: Add Customer managed KMS key to encrypt Filter Criteria. Default: - none
        :param filters: Add filter criteria to Event Source. Default: - none
        :param kafka_bootstrap_servers: A list of host and port pairs that are the addresses of the Kafka brokers in a self managed "bootstrap" Kafka cluster that a Kafka client connects to initially to bootstrap itself. They are in the format ``abc.example.com:9096``. Default: - none
        :param kafka_consumer_group_id: The identifier for the Kafka consumer group to join. The consumer group ID must be unique among all your Kafka event sources. After creating a Kafka event source mapping with the consumer group ID specified, you cannot update this value. The value must have a length between 1 and 200 and full the pattern '[a-zA-Z0-9-/*:_+=.@-]*'. For more information, see `Customizable consumer group ID <https://docs.aws.amazon.com/lambda/latest/dg/with-msk.html#services-msk-consumer-group-id>`_. Default: - none
        :param kafka_topic: The name of the Kafka topic. Default: - no topic
        :param max_batching_window: The maximum amount of time to gather records before invoking the function. Maximum of Duration.minutes(5) Default: Duration.seconds(0)
        :param max_concurrency: The maximum concurrency setting limits the number of concurrent instances of the function that an Amazon SQS event source can invoke. Default: - No specific limit.
        :param max_record_age: The maximum age of a record that Lambda sends to a function for processing. Valid Range: - Minimum value of 60 seconds - Maximum value of 7 days Default: - infinite or until the record expires.
        :param metrics_config: Configuration for enhanced monitoring metrics collection When specified, enables collection of additional metrics for the stream event source. Default: - Enhanced monitoring is disabled
        :param on_failure: An Amazon S3, Amazon SQS queue or Amazon SNS topic destination for discarded records. Default: discarded records are ignored
        :param parallelization_factor: The number of batches to process from each shard concurrently. Valid Range: - Minimum value of 1 - Maximum value of 10 Default: 1
        :param provisioned_poller_config: Configuration for provisioned pollers that read from the event source. When specified, allows control over the minimum and maximum number of pollers that can be provisioned to process events from the source. Default: - no provisioned pollers
        :param report_batch_item_failures: Allow functions to return partially successful responses for a batch of records. Default: false
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Set to ``undefined`` if you want lambda to keep retrying infinitely or until the record expires. Valid Range: - Minimum value of 0 - Maximum value of 10000 Default: - infinite or until the record expires.
        :param source_access_configurations: Specific settings like the authentication protocol or the VPC components to secure access to your event source. Default: - none
        :param starting_position: The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start reading. Default: - no starting position
        :param starting_position_timestamp: The time from which to start reading, in Unix time seconds. Default: - no timestamp
        :param support_s3_on_failure_destination: Check if support S3 onfailure destination(OFD). Kinesis, DynamoDB, MSK and self managed kafka event support S3 OFD Default: false
        :param tumbling_window: The size of the tumbling windows to group records sent to DynamoDB or Kinesis. Default: - None
        '''
        ...

    @jsii.member(jsii_name="addFunctionUrl")
    def add_function_url(
        self,
        *,
        auth_type: typing.Optional[FunctionUrlAuthType] = None,
        cors: typing.Optional[typing.Union[FunctionUrlCorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        invoke_mode: typing.Optional["InvokeMode"] = None,
    ) -> "FunctionUrl":
        '''Adds a url to this lambda function.

        :param auth_type: The type of authentication that your function URL uses. Default: FunctionUrlAuthType.AWS_IAM
        :param cors: The cross-origin resource sharing (CORS) settings for your function URL. Default: - No CORS configuration.
        :param invoke_mode: The type of invocation mode that your Lambda function uses. Default: InvokeMode.BUFFERED
        '''
        ...

    @jsii.member(jsii_name="addPermission")
    def add_permission(
        self,
        id: builtins.str,
        *,
        principal: _IPrincipal_539bb2fd,
        action: typing.Optional[builtins.str] = None,
        event_source_token: typing.Optional[builtins.str] = None,
        function_url_auth_type: typing.Optional[FunctionUrlAuthType] = None,
        organization_id: typing.Optional[builtins.str] = None,
        scope: typing.Optional[_constructs_77d1e7e8.Construct] = None,
        source_account: typing.Optional[builtins.str] = None,
        source_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Adds a permission to the Lambda resource policy.

        :param id: The id for the permission construct.
        :param principal: The entity for which you are granting permission to invoke the Lambda function. This entity can be any of the following: - a valid AWS service principal, such as ``s3.amazonaws.com`` or ``sns.amazonaws.com`` - an AWS account ID for cross-account permissions. For example, you might want to allow a custom application in another AWS account to push events to Lambda by invoking your function. - an AWS organization principal to grant permissions to an entire organization. The principal can be an AccountPrincipal, an ArnPrincipal, a ServicePrincipal, or an OrganizationPrincipal.
        :param action: The Lambda actions that you want to allow in this statement. For example, you can specify lambda:CreateFunction to specify a certain action, or use a wildcard (``lambda:*``) to grant permission to all Lambda actions. For a list of actions, see Actions and Condition Context Keys for AWS Lambda in the IAM User Guide. Default: 'lambda:InvokeFunction'
        :param event_source_token: A unique token that must be supplied by the principal invoking the function. Default: - The caller would not need to present a token.
        :param function_url_auth_type: The authType for the function URL that you are granting permissions for. Default: - No functionUrlAuthType
        :param organization_id: The organization you want to grant permissions to. Use this ONLY if you need to grant permissions to a subset of the organization. If you want to grant permissions to the entire organization, sending the organization principal through the ``principal`` property will suffice. You can use this property to ensure that all source principals are owned by a specific organization. Default: - No organizationId
        :param scope: The scope to which the permission constructs be attached. The default is the Lambda function construct itself, but this would need to be different in cases such as cross-stack references where the Permissions would need to sit closer to the consumer of this permission (i.e., the caller). Default: - The instance of lambda.IFunction
        :param source_account: The AWS account ID (without hyphens) of the source owner. For example, if you specify an S3 bucket in the SourceArn property, this value is the bucket owner's account ID. You can use this property to ensure that all source principals are owned by a specific account.
        :param source_arn: The ARN of a resource that is invoking your function. When granting Amazon Simple Storage Service (Amazon S3) permission to invoke your function, specify this property with the bucket ARN as its value. This ensures that events generated only from the specified bucket, not just any bucket from any AWS account that creates a mapping to your function, can invoke the function.

        :see: Permission for details.
        '''
        ...

    @jsii.member(jsii_name="addToRolePolicy")
    def add_to_role_policy(self, statement: _PolicyStatement_0fe33853) -> None:
        '''Adds a statement to the IAM role assumed by the instance.

        :param statement: -
        '''
        ...

    @jsii.member(jsii_name="configureAsyncInvoke")
    def configure_async_invoke(
        self,
        *,
        max_event_age: typing.Optional[_Duration_4839e8c3] = None,
        on_failure: typing.Optional[IDestination] = None,
        on_success: typing.Optional[IDestination] = None,
        retry_attempts: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Configures options for asynchronous invocation.

        :param max_event_age: The maximum age of a request that Lambda sends to a function for processing. Minimum: 60 seconds Maximum: 6 hours Default: Duration.hours(6)
        :param on_failure: The destination for failed invocations. Default: - no destination
        :param on_success: The destination for successful invocations. Default: - no destination
        :param retry_attempts: The maximum number of times to retry when the function returns an error. Minimum: 0 Maximum: 2 Default: 2
        '''
        ...

    @jsii.member(jsii_name="grantInvoke")
    def grant_invoke(self, identity: _IGrantable_71c4f5de) -> _Grant_a7ae64f8:
        '''Grant the given identity permissions to invoke this Lambda.

        :param identity: -
        '''
        ...

    @jsii.member(jsii_name="grantInvokeCompositePrincipal")
    def grant_invoke_composite_principal(
        self,
        composite_principal: _CompositePrincipal_c2fe180a,
    ) -> typing.List[_Grant_a7ae64f8]:
        '''Grant multiple principals the ability to invoke this Lambda via CompositePrincipal.

        :param composite_principal: -
        '''
        ...

    @jsii.member(jsii_name="grantInvokeLatestVersion")
    def grant_invoke_latest_version(
        self,
        identity: _IGrantable_71c4f5de,
    ) -> _Grant_a7ae64f8:
        '''Grant the given identity permissions to invoke the $LATEST version or unqualified version of this Lambda.

        :param identity: -
        '''
        ...

    @jsii.member(jsii_name="grantInvokeUrl")
    def grant_invoke_url(self, identity: _IGrantable_71c4f5de) -> _Grant_a7ae64f8:
        '''Grant the given identity permissions to invoke this Lambda Function URL.

        :param identity: -
        '''
        ...

    @jsii.member(jsii_name="grantInvokeVersion")
    def grant_invoke_version(
        self,
        identity: _IGrantable_71c4f5de,
        version: "IVersion",
    ) -> _Grant_a7ae64f8:
        '''Grant the given identity permissions to invoke the given version of this Lambda.

        :param identity: -
        :param version: -
        '''
        ...

    @jsii.member(jsii_name="metric")
    def metric(
        self,
        metric_name: builtins.str,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_Duration_4839e8c3] = None,
        region: typing.Optional[builtins.str] = None,
        stack_account: typing.Optional[builtins.str] = None,
        stack_region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_Unit_61bc6f70] = None,
    ) -> _Metric_e396a4dc:
        '''Return the given named metric for this Lambda Return the given named metric for this Function.

        :param metric_name: -
        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param stack_account: Account of the stack this metric is attached to. Default: - Deployment account.
        :param stack_region: Region of the stack this metric is attached to. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Use the ``aws_cloudwatch.Stats`` helper class to construct valid input strings. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" - "tmNN.NN" | "tm(NN.NN%:NN.NN%)" - "iqm" - "wmNN.NN" | "wm(NN.NN%:NN.NN%)" - "tcNN.NN" | "tc(NN.NN%:NN.NN%)" - "tsNN.NN" | "ts(NN.NN%:NN.NN%)" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        '''
        ...

    @jsii.member(jsii_name="metricDuration")
    def metric_duration(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_Duration_4839e8c3] = None,
        region: typing.Optional[builtins.str] = None,
        stack_account: typing.Optional[builtins.str] = None,
        stack_region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_Unit_61bc6f70] = None,
    ) -> _Metric_e396a4dc:
        '''Metric for the Duration of this Lambda How long execution of this Lambda takes.

        Average over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param stack_account: Account of the stack this metric is attached to. Default: - Deployment account.
        :param stack_region: Region of the stack this metric is attached to. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Use the ``aws_cloudwatch.Stats`` helper class to construct valid input strings. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" - "tmNN.NN" | "tm(NN.NN%:NN.NN%)" - "iqm" - "wmNN.NN" | "wm(NN.NN%:NN.NN%)" - "tcNN.NN" | "tc(NN.NN%:NN.NN%)" - "tsNN.NN" | "ts(NN.NN%:NN.NN%)" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream

        :default: average over 5 minutes
        '''
        ...

    @jsii.member(jsii_name="metricErrors")
    def metric_errors(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_Duration_4839e8c3] = None,
        region: typing.Optional[builtins.str] = None,
        stack_account: typing.Optional[builtins.str] = None,
        stack_region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_Unit_61bc6f70] = None,
    ) -> _Metric_e396a4dc:
        '''How many invocations of this Lambda fail.

        Sum over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param stack_account: Account of the stack this metric is attached to. Default: - Deployment account.
        :param stack_region: Region of the stack this metric is attached to. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Use the ``aws_cloudwatch.Stats`` helper class to construct valid input strings. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" - "tmNN.NN" | "tm(NN.NN%:NN.NN%)" - "iqm" - "wmNN.NN" | "wm(NN.NN%:NN.NN%)" - "tcNN.NN" | "tc(NN.NN%:NN.NN%)" - "tsNN.NN" | "ts(NN.NN%:NN.NN%)" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream
        '''
        ...

    @jsii.member(jsii_name="metricInvocations")
    def metric_invocations(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_Duration_4839e8c3] = None,
        region: typing.Optional[builtins.str] = None,
        stack_account: typing.Optional[builtins.str] = None,
        stack_region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_Unit_61bc6f70] = None,
    ) -> _Metric_e396a4dc:
        '''Metric for the number of invocations of this Lambda How often this Lambda is invoked.

        Sum over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param stack_account: Account of the stack this metric is attached to. Default: - Deployment account.
        :param stack_region: Region of the stack this metric is attached to. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Use the ``aws_cloudwatch.Stats`` helper class to construct valid input strings. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" - "tmNN.NN" | "tm(NN.NN%:NN.NN%)" - "iqm" - "wmNN.NN" | "wm(NN.NN%:NN.NN%)" - "tcNN.NN" | "tc(NN.NN%:NN.NN%)" - "tsNN.NN" | "ts(NN.NN%:NN.NN%)" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream

        :default: sum over 5 minutes
        '''
        ...

    @jsii.member(jsii_name="metricThrottles")
    def metric_throttles(
        self,
        *,
        account: typing.Optional[builtins.str] = None,
        color: typing.Optional[builtins.str] = None,
        dimensions_map: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        label: typing.Optional[builtins.str] = None,
        period: typing.Optional[_Duration_4839e8c3] = None,
        region: typing.Optional[builtins.str] = None,
        stack_account: typing.Optional[builtins.str] = None,
        stack_region: typing.Optional[builtins.str] = None,
        statistic: typing.Optional[builtins.str] = None,
        unit: typing.Optional[_Unit_61bc6f70] = None,
    ) -> _Metric_e396a4dc:
        '''Metric for the number of throttled invocations of this Lambda How often this Lambda is throttled.

        Sum over 5 minutes

        :param account: Account which this metric comes from. Default: - Deployment account.
        :param color: The hex color code, prefixed with '#' (e.g. '#00ff00'), to use when this metric is rendered on a graph. The ``Color`` class has a set of standard colors that can be used here. Default: - Automatic color
        :param dimensions_map: Dimensions of the metric. Default: - No dimensions.
        :param label: Label for this metric when added to a Graph in a Dashboard. You can use `dynamic labels <https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/graph-dynamic-labels.html>`_ to show summary information about the entire displayed time series in the legend. For example, if you use:: [max: ${MAX}] MyMetric As the metric label, the maximum value in the visible range will be shown next to the time series name in the graph's legend. Default: - No label
        :param period: The period over which the specified statistic is applied. Default: Duration.minutes(5)
        :param region: Region which this metric comes from. Default: - Deployment region.
        :param stack_account: Account of the stack this metric is attached to. Default: - Deployment account.
        :param stack_region: Region of the stack this metric is attached to. Default: - Deployment region.
        :param statistic: What function to use for aggregating. Use the ``aws_cloudwatch.Stats`` helper class to construct valid input strings. Can be one of the following: - "Minimum" | "min" - "Maximum" | "max" - "Average" | "avg" - "Sum" | "sum" - "SampleCount | "n" - "pNN.NN" - "tmNN.NN" | "tm(NN.NN%:NN.NN%)" - "iqm" - "wmNN.NN" | "wm(NN.NN%:NN.NN%)" - "tcNN.NN" | "tc(NN.NN%:NN.NN%)" - "tsNN.NN" | "ts(NN.NN%:NN.NN%)" Default: Average
        :param unit: Unit used to filter the metric stream. Only refer to datums emitted to the metric stream with the given unit and ignore all others. Only useful when datums are being emitted to the same metric stream under different units. The default is to use all matric datums in the stream, regardless of unit, which is recommended in nearly all cases. CloudWatch does not honor this property for graphs. Default: - All metric datums in the given metric stream

        :default: sum over 5 minutes
        '''
        ...


class _IFunctionProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
    jsii.proxy_for(_IConnectable_10015a05), # type: ignore[misc]
    jsii.proxy_for(_IGrantable_71c4f5de), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_lambda.IFunction"

    @builtins.property
    @jsii.member(jsii_name="architecture")
    def architecture(self) -> Architecture:
        '''The system architectures compatible with this lambda function.'''
        return typing.cast(Architecture, jsii.get(self, "architecture"))

    @builtins.property
    @jsii.member(jsii_name="functionArn")
    def function_arn(self) -> builtins.str:
        '''The ARN of the function.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionArn"))

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the function.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

    @builtins.property
    @jsii.member(jsii_name="isBoundToVpc")
    def is_bound_to_vpc(self) -> builtins.bool:
        '''Whether or not this Lambda function was bound to a VPC.

        If this is is ``false``, trying to access the ``connections`` object will fail.
        '''
        return typing.cast(builtins.bool, jsii.get(self, "isBoundToVpc"))

    @builtins.property
    @jsii.member(jsii_name="latestVersion")
    def latest_version(self) -> "IVersion":
        '''The ``$LATEST`` version of this function.

        Note that this is reference to a non-specific AWS Lambda version, which
        means the function this version refers to can return different results in
        different invocations.

        To obtain a reference to an explicit version which references the current
        function configuration, use ``lambdaFunction.currentVersion`` instead.
        '''
        return typing.cast("IVersion", jsii.get(self, "latestVersion"))

    @builtins.property
    @jsii.member(jsii_name="permissionsNode")
    def permissions_node(self) -> _constructs_77d1e7e8.Node:
        '''The construct node where permissions are attached.'''
        return typing.cast(_constructs_77d1e7e8.Node, jsii.get(self, "permissionsNode"))

    @builtins.property
    @jsii.member(jsii_name="resourceArnsForGrantInvoke")
    def resource_arns_for_grant_invoke(self) -> typing.List[builtins.str]:
        '''The ARN(s) to put into the resource field of the generated IAM policy for grantInvoke().

        This property is for cdk modules to consume only. You should not need to use this property.
        Instead, use grantInvoke() directly.
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "resourceArnsForGrantInvoke"))

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The IAM role associated with this function.'''
        return typing.cast(typing.Optional[_IRole_235f5d8e], jsii.get(self, "role"))

    @jsii.member(jsii_name="addEventSource")
    def add_event_source(self, source: IEventSource) -> None:
        '''Adds an event source to this function.

        Event sources are implemented in the aws-cdk-lib/aws-lambda-event-sources module.

        The following example adds an SQS Queue as an event source::

           import { SqsEventSource } from 'aws-cdk-lib/aws-lambda-event-sources';
           myFunc