r'''
# Amazon EKS Construct Library

This construct library allows you to define [Amazon Elastic Container Service for Kubernetes (EKS)](https://aws.amazon.com/eks/) clusters.
In addition, the library also supports defining Kubernetes resource manifests within EKS clusters.

## Table Of Contents

* [Amazon EKS Construct Library](#amazon-eks-construct-library)

  * [Table Of Contents](#table-of-contents)
  * [Quick Start](#quick-start)
  * [Architectural Overview](#architectural-overview)
  * [Provisioning clusters](#provisioning-clusters)

    * [Managed node groups](#managed-node-groups)

      * [Node Groups with IPv6 Support](#node-groups-with-ipv6-support)
      * [Spot Instances Support](#spot-instances-support)
      * [Launch Template Support](#launch-template-support)
    * [Update clusters](#update-clusters)
    * [Fargate profiles](#fargate-profiles)
    * [Self-managed nodes](#self-managed-nodes)

      * [Spot Instances](#spot-instances)
      * [Bottlerocket](#bottlerocket)
    * [Endpoint Access](#endpoint-access)
    * [Alb Controller](#alb-controller)
    * [VPC Support](#vpc-support)

      * [Kubectl Handler](#kubectl-handler)
      * [Cluster Handler](#cluster-handler)
    * [IPv6 Support](#ipv6-support)
    * [Kubectl Support](#kubectl-support)

      * [Environment](#environment)
      * [Runtime](#runtime)
      * [Memory](#memory)
    * [ARM64 Support](#arm64-support)
    * [Masters Role](#masters-role)
    * [Encryption](#encryption)
    * [Hybrid nodes](#hybrid-nodes)
  * [Permissions and Security](#permissions-and-security)

    * [AWS IAM Mapping](#aws-iam-mapping)
    * [Access Config](#access-config)
    * [Access Entry](#access-mapping)
    * [Cluster Security Group](#cluster-security-group)
    * [Node SSH Access](#node-ssh-access)
    * [Service Accounts](#service-accounts)
    * [Pod Identities](#pod-identities)
  * [Applying Kubernetes Resources](#applying-kubernetes-resources)

    * [Kubernetes Manifests](#kubernetes-manifests)

      * [ALB Controller Integration](#alb-controller-integration)
      * [Adding resources from a URL](#adding-resources-from-a-url)
      * [Dependencies](#dependencies)
      * [Resource Pruning](#resource-pruning)
      * [Manifests Validation](#manifests-validation)
    * [Helm Charts](#helm-charts)
    * [OCI Charts](#oci-charts)
    * [CDK8s Charts](#cdk8s-charts)

      * [Custom CDK8s Constructs](#custom-cdk8s-constructs)
      * [Manually importing k8s specs and CRD's](#manually-importing-k8s-specs-and-crds)
  * [Patching Kubernetes Resources](#patching-kubernetes-resources)
  * [Querying Kubernetes Resources](#querying-kubernetes-resources)
  * [Add-ons](#add-ons)
  * [Using existing clusters](#using-existing-clusters)
  * [Logging](#logging)
  * [Known Issues and Limitations](#known-issues-and-limitations)

## Quick Start

This example defines an Amazon EKS cluster with the following configuration:

* Dedicated VPC with default configuration (Implicitly created using [ec2.Vpc](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#vpc))
* A Kubernetes pod with a container based on the [paulbouwer/hello-kubernetes](https://github.com/paulbouwer/hello-kubernetes) image.

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


# provisioning a cluster
cluster = eks.Cluster(self, "hello-eks",
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)

# apply a kubernetes manifest to the cluster
cluster.add_manifest("mypod", {
    "api_version": "v1",
    "kind": "Pod",
    "metadata": {"name": "mypod"},
    "spec": {
        "containers": [{
            "name": "hello",
            "image": "paulbouwer/hello-kubernetes:1.5",
            "ports": [{"container_port": 8080}]
        }
        ]
    }
})
```

## Architectural Overview

The following is a qualitative diagram of the various possible components involved in the cluster deployment.

```text
 +-----------------------------------------------+               +-----------------+
 | EKS Cluster | kubectl |  |
 | ----------- |<-------------+| Kubectl Handler |
 |                                               |               |                 |
 |                                               |               +-----------------+
 | +--------------------+    +-----------------+ |
 | |                    |    |                 | |
 | | Managed Node Group |    | Fargate Profile | |               +-----------------+
 | |                    |    |                 | |               |                 |
 | +--------------------+    +-----------------+ |               | Cluster Handler |
 |                                               |               |                 |
 +-----------------------------------------------+               +-----------------+
    ^                                   ^                          +
    |                                   |                          |
    | connect self managed capacity     |                          | aws-sdk
    |                                   | create/update/delete     |
    +                                   |                          v
 +--------------------+                 +              +-------------------+
 |                    |                 --------------+| eks.amazonaws.com |
 | Auto Scaling Group |                                +-------------------+
 |                    |
 +--------------------+
```

In a nutshell:

* `EKS Cluster` - The cluster endpoint created by EKS.
* `Managed Node Group` - EC2 worker nodes managed by EKS.
* `Fargate Profile` - Fargate worker nodes managed by EKS.
* `Auto Scaling Group` - EC2 worker nodes managed by the user.
* `KubectlHandler` - Lambda function for invoking `kubectl` commands on the cluster - created by CDK.
* `ClusterHandler` - Lambda function for interacting with EKS API to manage the cluster lifecycle - created by CDK.

A more detailed breakdown of each is provided further down this README.

## Provisioning clusters

Creating a new cluster is done using the `Cluster` or `FargateCluster` constructs. The only required properties are the kubernetes `version` and `kubectlLayer`.

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


eks.Cluster(self, "HelloEKS",
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

You can also use `FargateCluster` to provision a cluster that uses only fargate workers.

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


eks.FargateCluster(self, "HelloEKS",
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

> **NOTE: Only 1 cluster per stack is supported.** If you have a use-case for multiple clusters per stack, or would like to understand more about this limitation, see [https://github.com/aws/aws-cdk/issues/10073](https://github.com/aws/aws-cdk/issues/10073).

Below you'll find a few important cluster configuration options. First of which is Capacity.
Capacity is the amount and the type of worker nodes that are available to the cluster for deploying resources. Amazon EKS offers 3 ways of configuring capacity, which you can combine as you like:

### Managed node groups

Amazon EKS managed node groups automate the provisioning and lifecycle management of nodes (Amazon EC2 instances) for Amazon EKS Kubernetes clusters.
With Amazon EKS managed node groups, you don't need to separately provision or register the Amazon EC2 instances that provide compute capacity to run your Kubernetes applications. You can create, update, or terminate nodes for your cluster with a single operation. Nodes run using the latest Amazon EKS optimized AMIs in your AWS account while node updates and terminations gracefully drain nodes to ensure that your applications stay available.

> For more details visit [Amazon EKS Managed Node Groups](https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html).

**Managed Node Groups are the recommended way to allocate cluster capacity.**

By default, this library will allocate a managed node group with 2 *m5.large* instances (this instance type suits most common use-cases, and is good value for money).

At cluster instantiation time, you can customize the number of instances and their type:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


eks.Cluster(self, "HelloEKS",
    version=eks.KubernetesVersion.V1_32,
    default_capacity=5,
    default_capacity_instance=ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.SMALL),
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

To access the node group that was created on your behalf, you can use `cluster.defaultNodegroup`.

Additional customizations are available post instantiation. To apply them, set the default capacity to 0, and use the `cluster.addNodegroupCapacity` method:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


cluster = eks.Cluster(self, "HelloEKS",
    version=eks.KubernetesVersion.V1_32,
    default_capacity=0,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)

cluster.add_nodegroup_capacity("custom-node-group",
    instance_types=[ec2.InstanceType("m5.large")],
    min_size=4,
    disk_size=100
)
```

To set node taints, you can set `taints` option.

```python
# cluster: eks.Cluster

cluster.add_nodegroup_capacity("custom-node-group",
    instance_types=[ec2.InstanceType("m5.large")],
    taints=[eks.TaintSpec(
        effect=eks.TaintEffect.NO_SCHEDULE,
        key="foo",
        value="bar"
    )
    ]
)
```

To define the type of the AMI for the node group, you may explicitly define `amiType` according to your requirements, supported amiType could be found [HERE](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_eks.NodegroupAmiType.html).

```python
# cluster: eks.Cluster


# X86_64 based AMI managed node group
cluster.add_nodegroup_capacity("custom-node-group",
    instance_types=[ec2.InstanceType("m5.large")],  # NOTE: if amiType is x86_64-based image, the instance types here must be x86_64-based.
    ami_type=eks.NodegroupAmiType.AL2023_X86_64_STANDARD
)

# ARM_64 based AMI managed node group
cluster.add_nodegroup_capacity("custom-node-group",
    instance_types=[ec2.InstanceType("m6g.medium")],  # NOTE: if amiType is ARM-based image, the instance types here must be ARM-based.
    ami_type=eks.NodegroupAmiType.AL2023_ARM_64_STANDARD
)
```

To define the maximum number of instances which can be simultaneously replaced in a node group during a version update you can set `maxUnavailable` or `maxUnavailablePercentage` options.

> For more details visit [Updating a managed node group](https://docs.aws.amazon.com/eks/latest/userguide/update-managed-node-group.html)

```python
# cluster: eks.Cluster

cluster.add_nodegroup_capacity("custom-node-group",
    instance_types=[ec2.InstanceType("m5.large")],
    max_size=5,
    max_unavailable=2
)
```

```python
# cluster: eks.Cluster

cluster.add_nodegroup_capacity("custom-node-group",
    instance_types=[ec2.InstanceType("m5.large")],
    max_unavailable_percentage=33
)
```

> **NOTE:** If you add instances with the inferentia class (`inf1` or `inf2`) or trainium class (`trn1` or `trn1n`)
> the [neuron plugin](https://awsdocs-neuron.readthedocs-hosted.com/en/latest/containers/dlc-then-eks-devflow.html)
> will be automatically installed in the kubernetes cluster.

#### Node Groups with IPv6 Support

Node groups are available with IPv6 configured networks.  For custom roles assigned to node groups additional permissions are necessary in order for pods to obtain an IPv6 address.  The default node role will include these permissions.

> For more details visit [Configuring the Amazon VPC CNI plugin for Kubernetes to use IAM roles for service accounts](https://docs.aws.amazon.com/eks/latest/userguide/cni-iam-role.html#cni-iam-role-create-role)

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


ipv6_management = iam.PolicyDocument(
    statements=[iam.PolicyStatement(
        resources=["arn:aws:ec2:*:*:network-interface/*"],
        actions=["ec2:AssignIpv6Addresses", "ec2:UnassignIpv6Addresses"
        ]
    )]
)

eks_cluster_node_group_role = iam.Role(self, "eksClusterNodeGroupRole",
    role_name="eksClusterNodeGroupRole",
    assumed_by=iam.ServicePrincipal("ec2.amazonaws.com"),
    managed_policies=[
        iam.ManagedPolicy.from_aws_managed_policy_name("AmazonEKSWorkerNodePolicy"),
        iam.ManagedPolicy.from_aws_managed_policy_name("AmazonEC2ContainerRegistryReadOnly"),
        iam.ManagedPolicy.from_aws_managed_policy_name("AmazonEKS_CNI_Policy")
    ],
    inline_policies={
        "ipv6_management": ipv6_management
    }
)

cluster = eks.Cluster(self, "HelloEKS",
    version=eks.KubernetesVersion.V1_32,
    default_capacity=0,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)

cluster.add_nodegroup_capacity("custom-node-group",
    instance_types=[ec2.InstanceType("m5.large")],
    min_size=2,
    disk_size=100,
    node_role=eks_cluster_node_group_role
)
```

#### Spot Instances Support

Use `capacityType` to create managed node groups comprised of spot instances. To maximize the availability of your applications while using
Spot Instances, we recommend that you configure a Spot managed node group to use multiple instance types with the `instanceTypes` property.

> For more details visit [Managed node group capacity types](https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html#managed-node-group-capacity-types).

```python
# cluster: eks.Cluster

cluster.add_nodegroup_capacity("extra-ng-spot",
    instance_types=[
        ec2.InstanceType("c5.large"),
        ec2.InstanceType("c5a.large"),
        ec2.InstanceType("c5d.large")
    ],
    min_size=3,
    capacity_type=eks.CapacityType.SPOT
)
```

#### Launch Template Support

You can specify a launch template that the node group will use. For example, this can be useful if you want to use
a custom AMI or add custom user data.

When supplying a custom user data script, it must be encoded in the MIME multi-part archive format, since Amazon EKS merges with its own user data. Visit the [Launch Template Docs](https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html#launch-template-user-data)
for mode details.

```python
# cluster: eks.Cluster


user_data = """MIME-Version: 1.0
Content-Type: multipart/mixed; boundary="==MYBOUNDARY=="

--==MYBOUNDARY==
Content-Type: text/x-shellscript; charset="us-ascii"

#!/bin/bash
echo "Running custom user data script"

--==MYBOUNDARY==--\\
"""
lt = ec2.CfnLaunchTemplate(self, "LaunchTemplate",
    launch_template_data=ec2.CfnLaunchTemplate.LaunchTemplateDataProperty(
        instance_type="t3.small",
        user_data=Fn.base64(user_data)
    )
)

cluster.add_nodegroup_capacity("extra-ng",
    launch_template_spec=eks.LaunchTemplateSpec(
        id=lt.ref,
        version=lt.attr_latest_version_number
    )
)
```

Note that when using a custom AMI, Amazon EKS doesn't merge any user data. Which means you do not need the multi-part encoding. and are responsible for supplying the required bootstrap commands for nodes to join the cluster.
In the following example, `/ect/eks/bootstrap.sh` from the AMI will be used to bootstrap the node.

```python
# cluster: eks.Cluster

user_data = ec2.UserData.for_linux()
user_data.add_commands("set -o xtrace", f"/etc/eks/bootstrap.sh {cluster.clusterName}")
lt = ec2.CfnLaunchTemplate(self, "LaunchTemplate",
    launch_template_data=ec2.CfnLaunchTemplate.LaunchTemplateDataProperty(
        image_id="some-ami-id",  # custom AMI
        instance_type="t3.small",
        user_data=Fn.base64(user_data.render())
    )
)
cluster.add_nodegroup_capacity("extra-ng",
    launch_template_spec=eks.LaunchTemplateSpec(
        id=lt.ref,
        version=lt.attr_latest_version_number
    )
)
```

You may specify one `instanceType` in the launch template or multiple `instanceTypes` in the node group, **but not both**.

> For more details visit [Launch Template Support](https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html).

Graviton 2 instance types are supported including `c6g`, `m6g`, `r6g` and `t4g`.
Graviton 3 instance types are supported including `c7g`.

### Update clusters

When you rename the cluster name and redeploy the stack, the cluster replacement will be triggered and
the existing one will be deleted after the new one is provisioned. As the cluster resource ARN has been changed,
the cluster resource handler would not be able to delete the old one as the resource ARN in the IAM policy
has been changed. As a workaround, you need to add a temporary policy to the cluster admin role for
successful replacement. Consider this example if you are renaming the cluster from `foo` to `bar`:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


cluster = eks.Cluster(self, "cluster-to-rename",
    cluster_name="foo",  # rename this to 'bar'
    kubectl_layer=KubectlV32Layer(self, "kubectl"),
    version=eks.KubernetesVersion.V1_32
)

# allow the cluster admin role to delete the cluster 'foo'
cluster.admin_role.add_to_policy(iam.PolicyStatement(
    actions=["eks:DeleteCluster", "eks:DescribeCluster"
    ],
    resources=[
        Stack.of(self).format_arn(service="eks", resource="cluster", resource_name="foo")
    ]
))
```

### Fargate profiles

AWS Fargate is a technology that provides on-demand, right-sized compute
capacity for containers. With AWS Fargate, you no longer have to provision,
configure, or scale groups of virtual machines to run containers. This removes
the need to choose server types, decide when to scale your node groups, or
optimize cluster packing.

You can control which pods start on Fargate and how they run with Fargate
Profiles, which are defined as part of your Amazon EKS cluster.

See [Fargate Considerations](https://docs.aws.amazon.com/eks/latest/userguide/fargate.html#fargate-considerations) in the AWS EKS User Guide.

You can add Fargate Profiles to any EKS cluster defined in your CDK app
through the `addFargateProfile()` method. The following example adds a profile
that will match all pods from the "default" namespace:

```python
# cluster: eks.Cluster

cluster.add_fargate_profile("MyProfile",
    selectors=[eks.Selector(namespace="default")]
)
```

You can also directly use the `FargateProfile` construct to create profiles under different scopes:

```python
# cluster: eks.Cluster

eks.FargateProfile(self, "MyProfile",
    cluster=cluster,
    selectors=[eks.Selector(namespace="default")]
)
```

To create an EKS cluster that **only** uses Fargate capacity, you can use `FargateCluster`.
The following code defines an Amazon EKS cluster with a default Fargate Profile that matches all pods from the "kube-system" and "default" namespaces. It is also configured to [run CoreDNS on Fargate](https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns).

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


cluster = eks.FargateCluster(self, "MyCluster",
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

`FargateCluster` will create a default `FargateProfile` which can be accessed via the cluster's `defaultProfile` property. The created profile can also be customized by passing options as with `addFargateProfile`.

**NOTE**: Classic Load Balancers and Network Load Balancers are not supported on
pods running on Fargate. For ingress, we recommend that you use the [ALB Ingress
Controller](https://docs.aws.amazon.com/eks/latest/userguide/alb-ingress.html)
on Amazon EKS (minimum version v1.1.4).

### Self-managed nodes

Another way of allocating capacity to an EKS cluster is by using self-managed nodes.
EC2 instances that are part of the auto-scaling group will serve as worker nodes for the cluster.
This type of capacity is also commonly referred to as *EC2 Capacity** or *EC2 Nodes*.

For a detailed overview please visit [Self Managed Nodes](https://docs.aws.amazon.com/eks/latest/userguide/worker.html).

Creating an auto-scaling group and connecting it to the cluster is done using the `cluster.addAutoScalingGroupCapacity` method:

```python
# cluster: eks.Cluster

cluster.add_auto_scaling_group_capacity("frontend-nodes",
    instance_type=ec2.InstanceType("t2.medium"),
    min_capacity=3,
    vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
)
```

To connect an already initialized auto-scaling group, use the `cluster.connectAutoScalingGroupCapacity()` method:

```python
# cluster: eks.Cluster
# asg: autoscaling.AutoScalingGroup

cluster.connect_auto_scaling_group_capacity(asg)
```

To connect a self-managed node group to an imported cluster, use the `cluster.connectAutoScalingGroupCapacity()` method:

```python
# cluster: eks.Cluster
# asg: autoscaling.AutoScalingGroup

imported_cluster = eks.Cluster.from_cluster_attributes(self, "ImportedCluster",
    cluster_name=cluster.cluster_name,
    cluster_security_group_id=cluster.cluster_security_group_id
)

imported_cluster.connect_auto_scaling_group_capacity(asg)
```

In both cases, the [cluster security group](https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html#cluster-sg) will be automatically attached to
the auto-scaling group, allowing for traffic to flow freely between managed and self-managed nodes.

> **Note:** The default `updateType` for auto-scaling groups does not replace existing nodes. Since security groups are determined at launch time, self-managed nodes that were provisioned with version `1.78.0` or lower, will not be updated.
> To apply the new configuration on all your self-managed nodes, you'll need to replace the nodes using the `UpdateType.REPLACING_UPDATE` policy for the [`updateType`](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-autoscaling.AutoScalingGroup.html#updatetypespan-classapi-icon-api-icon-deprecated-titlethis-api-element-is-deprecated-its-use-is-not-recommended%EF%B8%8Fspan) property.

You can customize the [/etc/eks/boostrap.sh](https://github.com/awslabs/amazon-eks-ami/blob/master/files/bootstrap.sh) script, which is responsible
for bootstrapping the node to the EKS cluster. For example, you can use `kubeletExtraArgs` to add custom node labels or taints.

```python
# cluster: eks.Cluster

cluster.add_auto_scaling_group_capacity("spot",
    instance_type=ec2.InstanceType("t3.large"),
    min_capacity=2,
    bootstrap_options=eks.BootstrapOptions(
        kubelet_extra_args="--node-labels foo=bar,goo=far",
        aws_api_retry_attempts=5
    )
)
```

To disable bootstrapping altogether (i.e. to fully customize user-data), set `bootstrapEnabled` to `false`.
You can also configure the cluster to use an auto-scaling group as the default capacity:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


cluster = eks.Cluster(self, "HelloEKS",
    version=eks.KubernetesVersion.V1_32,
    default_capacity_type=eks.DefaultCapacityType.EC2,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

This will allocate an auto-scaling group with 2 *m5.large* instances (this instance type suits most common use-cases, and is good value for money).
To access the `AutoScalingGroup` that was created on your behalf, you can use `cluster.defaultCapacity`.
You can also independently create an `AutoScalingGroup` and connect it to the cluster using the `cluster.connectAutoScalingGroupCapacity` method:

```python
# cluster: eks.Cluster
# asg: autoscaling.AutoScalingGroup

cluster.connect_auto_scaling_group_capacity(asg)
```

This will add the necessary user-data to access the apiserver and configure all connections, roles, and tags needed for the instances in the auto-scaling group to properly join the cluster.

#### Spot Instances

When using self-managed nodes, you can configure the capacity to use spot instances, greatly reducing capacity cost.
To enable spot capacity, use the `spotPrice` property:

```python
# cluster: eks.Cluster

cluster.add_auto_scaling_group_capacity("spot",
    spot_price="0.1094",
    instance_type=ec2.InstanceType("t3.large"),
    max_capacity=10
)
```

> Spot instance nodes will be labeled with `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.

The [AWS Node Termination Handler](https://github.com/aws/aws-node-termination-handler) `DaemonSet` will be
installed from [Amazon EKS Helm chart repository](https://github.com/aws/eks-charts/tree/master/stable/aws-node-termination-handler) on these nodes.
The termination handler ensures that the Kubernetes control plane responds appropriately to events that
can cause your EC2 instance to become unavailable, such as [EC2 maintenance events](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/monitoring-instances-status-check_sched.html)
and [EC2 Spot interruptions](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-interruptions.html) and helps gracefully stop all pods running on spot nodes that are about to be
terminated.

> Handler Version: [1.7.0](https://github.com/aws/aws-node-termination-handler/releases/tag/v1.7.0)
>
> Chart Version: [0.9.5](https://github.com/aws/eks-charts/blob/v0.0.28/stable/aws-node-termination-handler/Chart.yaml)

To disable the installation of the termination handler, set the `spotInterruptHandler` property to `false`. This applies both to `addAutoScalingGroupCapacity` and `connectAutoScalingGroupCapacity`.

#### Bottlerocket

[Bottlerocket](https://aws.amazon.com/bottlerocket/) is a Linux-based open-source operating system that is purpose-built by Amazon Web Services for running containers on virtual machines or bare metal hosts.

`Bottlerocket` is supported when using managed nodegroups or self-managed auto-scaling groups.

To create a Bottlerocket managed nodegroup:

```python
# cluster: eks.Cluster

cluster.add_nodegroup_capacity("BottlerocketNG",
    ami_type=eks.NodegroupAmiType.BOTTLEROCKET_X86_64
)
```

The following example will create an auto-scaling group of 2 `t3.small` Linux instances running with the `Bottlerocket` AMI.

```python
# cluster: eks.Cluster

cluster.add_auto_scaling_group_capacity("BottlerocketNodes",
    instance_type=ec2.InstanceType("t3.small"),
    min_capacity=2,
    machine_image_type=eks.MachineImageType.BOTTLEROCKET
)
```

The specific Bottlerocket AMI variant will be auto selected according to the k8s version for the `x86_64` architecture.
For example, if the Amazon EKS cluster version is `1.17`, the Bottlerocket AMI variant will be auto selected as
`aws-k8s-1.17` behind the scene.

> See [Variants](https://github.com/bottlerocket-os/bottlerocket/blob/develop/README.md#variants) for more details.

Please note Bottlerocket does not allow to customize bootstrap options and `bootstrapOptions` properties is not supported when you create the `Bottlerocket` capacity.

To create a Bottlerocket managed nodegroup with Nvidia-based EC2 instance types use the `BOTTLEROCKET_X86_64_NVIDIA` or
`BOTTLEROCKET_ARM_64_NVIDIA` AMIs:

```python
# cluster: eks.Cluster

cluster.add_nodegroup_capacity("BottlerocketNvidiaNG",
    ami_type=eks.NodegroupAmiType.BOTTLEROCKET_X86_64_NVIDIA,
    instance_types=[ec2.InstanceType("g4dn.xlarge")]
)
```

For more details about Bottlerocket, see [Bottlerocket FAQs](https://aws.amazon.com/bottlerocket/faqs/) and [Bottlerocket Open Source Blog](https://aws.amazon.com/blogs/opensource/announcing-the-general-availability-of-bottlerocket-an-open-source-linux-distribution-purpose-built-to-run-containers/).

### Endpoint Access

When you create a new cluster, Amazon EKS creates an endpoint for the managed Kubernetes API server that you use to communicate with your cluster (using Kubernetes management tools such as `kubectl`)

By default, this API server endpoint is public to the internet, and access to the API server is secured using a combination of
AWS Identity and Access Management (IAM) and native Kubernetes [Role Based Access Control](https://kubernetes.io/docs/reference/access-authn-authz/rbac/) (RBAC).

You can configure the [cluster endpoint access](https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html) by using the `endpointAccess` property:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


cluster = eks.Cluster(self, "hello-eks",
    version=eks.KubernetesVersion.V1_32,
    endpoint_access=eks.EndpointAccess.PRIVATE,  # No access outside of your VPC.
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

The default value is `eks.EndpointAccess.PUBLIC_AND_PRIVATE`. Which means the cluster endpoint is accessible from outside of your VPC, but worker node traffic and `kubectl` commands issued by this library stay within your VPC.

### Alb Controller

Some Kubernetes resources are commonly implemented on AWS with the help of the [ALB Controller](https://kubernetes-sigs.github.io/aws-load-balancer-controller/latest/).

From the docs:

> AWS Load Balancer Controller is a controller to help manage Elastic Load Balancers for a Kubernetes cluster.
>
> * It satisfies Kubernetes Ingress resources by provisioning Application Load Balancers.
> * It satisfies Kubernetes Service resources by provisioning Network Load Balancers.

To deploy the controller on your EKS cluster, configure the `albController` property:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


eks.Cluster(self, "HelloEKS",
    version=eks.KubernetesVersion.V1_32,
    alb_controller=eks.AlbControllerOptions(
        version=eks.AlbControllerVersion.V2_8_2
    ),
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

The `albController` requires `defaultCapacity` or at least one nodegroup. If there's no `defaultCapacity` or available
nodegroup for the cluster, the `albController` deployment would fail.

Querying the controller pods should look something like this:

```console
❯ kubectl get pods -n kube-system
NAME                                            READY   STATUS    RESTARTS   AGE
aws-load-balancer-controller-76bd6c7586-d929p   1/1     Running   0          109m
aws-load-balancer-controller-76bd6c7586-fqxph   1/1     Running   0          109m
...
...
```

Every Kubernetes manifest that utilizes the ALB Controller is effectively dependant on the controller.
If the controller is deleted before the manifest, it might result in dangling ELB/ALB resources.
Currently, the EKS construct library does not detect such dependencies, and they should be done explicitly.

For example:

```python
# cluster: eks.Cluster

manifest = cluster.add_manifest("manifest", {})
if cluster.alb_controller:
    manifest.node.add_dependency(cluster.alb_controller)
```

### VPC Support

You can specify the VPC of the cluster using the `vpc` and `vpcSubnets` properties:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer

# vpc: ec2.Vpc


eks.Cluster(self, "HelloEKS",
    version=eks.KubernetesVersion.V1_32,
    vpc=vpc,
    vpc_subnets=[ec2.SubnetSelection(subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS)],
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

> Note: Isolated VPCs (i.e with no internet access) are not fully supported. See https://github.com/aws/aws-cdk/issues/12171. Check out [this aws-cdk-example](https://github.com/aws-samples/aws-cdk-examples/tree/master/java/eks/private-cluster) for reference.

If you do not specify a VPC, one will be created on your behalf, which you can then access via `cluster.vpc`. The cluster VPC will be associated to any EKS managed capacity (i.e Managed Node Groups and Fargate Profiles).

Please note that the `vpcSubnets` property defines the subnets where EKS will place the *control plane* ENIs. To choose
the subnets where EKS will place the worker nodes, please refer to the **Provisioning clusters** section above.

If you allocate self managed capacity, you can specify which subnets should the auto-scaling group use:

```python
# vpc: ec2.Vpc
# cluster: eks.Cluster

cluster.add_auto_scaling_group_capacity("nodes",
    vpc_subnets=ec2.SubnetSelection(subnets=vpc.private_subnets),
    instance_type=ec2.InstanceType("t2.medium")
)
```

There are two additional components you might want to provision within the VPC.

#### Kubectl Handler

The `KubectlHandler` is a Lambda function responsible to issuing `kubectl` and `helm` commands against the cluster when you add resource manifests to the cluster.

The handler association to the VPC is derived from the `endpointAccess` configuration. The rule of thumb is: *If the cluster VPC can be associated, it will be*.

Breaking this down, it means that if the endpoint exposes private access (via `EndpointAccess.PRIVATE` or `EndpointAccess.PUBLIC_AND_PRIVATE`), and the VPC contains **private** subnets, the Lambda function will be provisioned inside the VPC and use the private subnets to interact with the cluster. This is the common use-case.

If the endpoint does not expose private access (via `EndpointAccess.PUBLIC`) **or** the VPC does not contain private subnets, the function will not be provisioned within the VPC.

If your use-case requires control over the IAM role that the KubeCtl Handler assumes, a custom role can be passed through the ClusterProps (as `kubectlLambdaRole`) of the EKS Cluster construct.

#### Cluster Handler

The `ClusterHandler` is a set of Lambda functions (`onEventHandler`, `isCompleteHandler`) responsible for interacting with the EKS API in order to control the cluster lifecycle. To provision these functions inside the VPC, set the `placeClusterHandlerInVpc` property to `true`. This will place the functions inside the private subnets of the VPC based on the selection strategy specified in the [`vpcSubnets`](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-eks.Cluster.html#vpcsubnetsspan-classapi-icon-api-icon-experimental-titlethis-api-element-is-experimental-it-may-change-without-noticespan) property.

You can configure the environment of the Cluster Handler functions by specifying it at cluster instantiation. For example, this can be useful in order to configure an http proxy:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer

# proxy_instance_security_group: ec2.SecurityGroup

cluster = eks.Cluster(self, "hello-eks",
    version=eks.KubernetesVersion.V1_32,
    cluster_handler_environment={
        "https_proxy": "http://proxy.myproxy.com"
    },
    #
    # If the proxy is not open publicly, you can pass a security group to the
    # Cluster Handler Lambdas so that it can reach the proxy.
    #
    cluster_handler_security_group=proxy_instance_security_group,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

### IPv6 Support

You can optionally choose to configure your cluster to use IPv6 using the [`ipFamily`](https://docs.aws.amazon.com/eks/latest/APIReference/API_KubernetesNetworkConfigRequest.html#AmazonEKS-Type-KubernetesNetworkConfigRequest-ipFamily) definition for your cluster.  Note that this will require the underlying subnets to have an associated IPv6 CIDR.

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
# vpc: ec2.Vpc


def associate_subnet_with_v6_cidr(self, vpc, count, subnet):
    cfn_subnet = subnet.node.default_child
    cfn_subnet.ipv6_cidr_block = Fn.select(count, Fn.cidr(Fn.select(0, vpc.vpc_ipv6_cidr_blocks), 256, (128 - 64).to_string()))
    cfn_subnet.assign_ipv6_address_on_creation = True

# make an ipv6 cidr
ipv6cidr = ec2.CfnVPCCidrBlock(self, "CIDR6",
    vpc_id=vpc.vpc_id,
    amazon_provided_ipv6_cidr_block=True
)

# connect the ipv6 cidr to all vpc subnets
subnetcount = 0
subnets = vpc.public_subnets.concat(vpc.private_subnets)
for subnet in subnets:
    # Wait for the ipv6 cidr to complete
    subnet.node.add_dependency(ipv6cidr)
    associate_subnet_with_v6_cidr(vpc, subnetcount, subnet)
    subnetcount = subnetcount + 1

cluster = eks.Cluster(self, "hello-eks",
    version=eks.KubernetesVersion.V1_32,
    vpc=vpc,
    ip_family=eks.IpFamily.IP_V6,
    vpc_subnets=[ec2.SubnetSelection(subnets=vpc.public_subnets)],
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

### Kubectl Support

The resources are created in the cluster by running `kubectl apply` from a python lambda function.

By default, CDK will create a new python lambda function to apply your k8s manifests. If you want to use an existing kubectl provider function, for example with tight trusted entities on your IAM Roles - you can import the existing provider and then use the imported provider when importing the cluster:

```python
handler_role = iam.Role.from_role_arn(self, "HandlerRole", "arn:aws:iam::123456789012:role/lambda-role")
# get the serviceToken from the custom resource provider
function_arn = lambda_.Function.from_function_name(self, "ProviderOnEventFunc", "ProviderframeworkonEvent-XXX").function_arn
kubectl_provider = eks.KubectlProvider.from_kubectl_provider_attributes(self, "KubectlProvider",
    function_arn=function_arn,
    kubectl_role_arn="arn:aws:iam::123456789012:role/kubectl-role",
    handler_role=handler_role
)

cluster = eks.Cluster.from_cluster_attributes(self, "Cluster",
    cluster_name="cluster",
    kubectl_provider=kubectl_provider
)
```

#### Environment

You can configure the environment of this function by specifying it at cluster instantiation. For example, this can be useful in order to configure an http proxy:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


cluster = eks.Cluster(self, "hello-eks",
    version=eks.KubernetesVersion.V1_32,
    kubectl_environment={
        "http_proxy": "http://proxy.myproxy.com"
    },
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

#### Runtime

The kubectl handler uses `kubectl`, `helm` and the `aws` CLI in order to
interact with the cluster. These are bundled into AWS Lambda layers included in
the `@aws-cdk/lambda-layer-awscli` and `@aws-cdk/lambda-layer-kubectl` modules.

The version of kubectl used must be compatible with the Kubernetes version of the
cluster. kubectl is supported within one minor version (older or newer) of Kubernetes
(see [Kubernetes version skew policy](https://kubernetes.io/releases/version-skew-policy/#kubectl)).
Depending on which version of kubernetes you're targeting, you will need to use one of
the `@aws-cdk/lambda-layer-kubectl-vXY` packages.

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


cluster = eks.Cluster(self, "hello-eks",
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

You can also specify a custom `lambda.LayerVersion` if you wish to use a
different version of these tools, or a version not available in any of the
`@aws-cdk/lambda-layer-kubectl-vXY` packages. The handler expects the layer to
include the following two executables:

```text
helm/helm
kubectl/kubectl
```

See more information in the
[Dockerfile](https://github.com/aws/aws-cdk/tree/main/packages/%40aws-cdk/lambda-layer-awscli/layer) for @aws-cdk/lambda-layer-awscli
and the
[Dockerfile](https://github.com/aws/aws-cdk/tree/main/packages/%40aws-cdk/lambda-layer-kubectl/layer) for @aws-cdk/lambda-layer-kubectl.

```python
layer = lambda_.LayerVersion(self, "KubectlLayer",
    code=lambda_.Code.from_asset("layer.zip")
)
```

Now specify when the cluster is defined:

```python
# layer: lambda.LayerVersion
# vpc: ec2.Vpc


cluster1 = eks.Cluster(self, "MyCluster",
    kubectl_layer=layer,
    vpc=vpc,
    cluster_name="cluster-name",
    version=eks.KubernetesVersion.V1_32
)

# or
cluster2 = eks.Cluster.from_cluster_attributes(self, "MyCluster",
    kubectl_layer=layer,
    vpc=vpc,
    cluster_name="cluster-name"
)
```

#### Memory

By default, the kubectl provider is configured with 1024MiB of memory. You can use the `kubectlMemory` option to specify the memory size for the AWS Lambda function:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer

# or
# vpc: ec2.Vpc


eks.Cluster(self, "MyCluster",
    kubectl_memory=Size.gibibytes(4),
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
eks.Cluster.from_cluster_attributes(self, "MyCluster",
    kubectl_memory=Size.gibibytes(4),
    vpc=vpc,
    cluster_name="cluster-name"
)
```

### ARM64 Support

Instance types with `ARM64` architecture are supported in both managed nodegroup and self-managed capacity. Simply specify an ARM64 `instanceType` (such as `m6g.medium`), and the latest
Amazon Linux 2 AMI for ARM64 will be automatically selected.

```python
# cluster: eks.Cluster

# add a managed ARM64 nodegroup
cluster.add_nodegroup_capacity("extra-ng-arm",
    instance_types=[ec2.InstanceType("m6g.medium")],
    min_size=2
)

# add a self-managed ARM64 nodegroup
cluster.add_auto_scaling_group_capacity("self-ng-arm",
    instance_type=ec2.InstanceType("m6g.medium"),
    min_capacity=2
)
```

### Masters Role

When you create a cluster, you can specify a `mastersRole`. The `Cluster` construct will associate this role with the `system:masters` [RBAC](https://kubernetes.io/docs/reference/access-authn-authz/rbac/) group, giving it super-user access to the cluster.

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer

# role: iam.Role

eks.Cluster(self, "HelloEKS",
    version=eks.KubernetesVersion.V1_32,
    masters_role=role,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

In order to interact with your cluster through `kubectl`, you can use the `aws eks update-kubeconfig` [AWS CLI command](https://docs.aws.amazon.com/cli/latest/reference/eks/update-kubeconfig.html)
to configure your local kubeconfig. The EKS module will define a CloudFormation output in your stack which contains the command to run. For example:

```plaintext
Outputs:
ClusterConfigCommand43AAE40F = aws eks update-kubeconfig --name cluster-xxxxx --role-arn arn:aws:iam::112233445566:role/yyyyy
```

Execute the `aws eks update-kubeconfig ...` command in your terminal to create or update a local kubeconfig context:

```console
$ aws eks update-kubeconfig --name cluster-xxxxx --role-arn arn:aws:iam::112233445566:role/yyyyy
Added new context arn:aws:eks:rrrrr:112233445566:cluster/cluster-xxxxx to /home/boom/.kube/config
```

And now you can simply use `kubectl`:

```console
$ kubectl get all -n kube-system
NAME                           READY   STATUS    RESTARTS   AGE
pod/aws-node-fpmwv             1/1     Running   0          21m
pod/aws-node-m9htf             1/1     Running   0          21m
pod/coredns-5cb4fb54c7-q222j   1/1     Running   0          23m
pod/coredns-5cb4fb54c7-v9nxx   1/1     Running   0          23m
...
```

If you do not specify it, you won't have access to the cluster from outside of the CDK application.

> Note that `cluster.addManifest` and `new KubernetesManifest` will still work.

### Encryption

When you create an Amazon EKS cluster, envelope encryption of Kubernetes secrets using the AWS Key Management Service (AWS KMS) can be enabled.
The documentation on [creating a cluster](https://docs.aws.amazon.com/eks/latest/userguide/create-cluster.html)
can provide more details about the customer master key (CMK) that can be used for the encryption.

You can use the `secretsEncryptionKey` to configure which key the cluster will use to encrypt Kubernetes secrets. By default, an AWS Managed key will be used.

> This setting can only be specified when the cluster is created and cannot be updated.

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


secrets_key = kms.Key(self, "SecretsKey")
cluster = eks.Cluster(self, "MyCluster",
    secrets_encryption_key=secrets_key,
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

You can also use a similar configuration for running a cluster built using the FargateCluster construct.

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


secrets_key = kms.Key(self, "SecretsKey")
cluster = eks.FargateCluster(self, "MyFargateCluster",
    secrets_encryption_key=secrets_key,
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

The Amazon Resource Name (ARN) for that CMK can be retrieved.

```python
# cluster: eks.Cluster

cluster_encryption_config_key_arn = cluster.cluster_encryption_config_key_arn
```

### Hybrid Nodes

When you create an Amazon EKS cluster, you can configure it to leverage the [EKS Hybrid Nodes](https://aws.amazon.com/eks/hybrid-nodes/) feature, allowing you to use your on-premises and edge infrastructure as nodes in your EKS cluster. Refer to the Hyrid Nodes [networking documentation](https://docs.aws.amazon.com/eks/latest/userguide/hybrid-nodes-networking.html) to configure your on-premises network, node and pod CIDRs, access control, etc before creating your EKS Cluster.

Once you have identified the on-premises node and pod (optional) CIDRs you will use for your hybrid nodes and the workloads running on them, you can specify them during cluster creation using the `remoteNodeNetworks` and `remotePodNetworks` (optional) properties:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


eks.Cluster(self, "Cluster",
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "KubectlLayer"),
    remote_node_networks=[eks.RemoteNodeNetwork(
        cidrs=["10.0.0.0/16"]
    )
    ],
    remote_pod_networks=[eks.RemotePodNetwork(
        cidrs=["192.168.0.0/16"]
    )
    ]
)
```

## Permissions and Security

Amazon EKS provides several mechanism of securing the cluster and granting permissions to specific IAM users and roles.

### AWS IAM Mapping

As described in the [Amazon EKS User Guide](https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html), you can map AWS IAM users and roles to [Kubernetes Role-based access control (RBAC)](https://kubernetes.io/docs/reference/access-authn-authz/rbac).

The Amazon EKS construct manages the *aws-auth* `ConfigMap` Kubernetes resource on your behalf and exposes an API through the `cluster.awsAuth` for mapping
users, roles and accounts.

Furthermore, when auto-scaling group capacity is added to the cluster, the IAM instance role of the auto-scaling group will be automatically mapped to RBAC so nodes can connect to the cluster. No manual mapping is required.

For example, let's say you want to grant an IAM user administrative privileges on your cluster:

```python
# cluster: eks.Cluster

admin_user = iam.User(self, "Admin")
cluster.aws_auth.add_user_mapping(admin_user, groups=["system:masters"])
```

A convenience method for mapping a role to the `system:masters` group is also available:

```python
# cluster: eks.Cluster
# role: iam.Role

cluster.aws_auth.add_masters_role(role)
```

To access the Kubernetes resources from the console, make sure your viewing principal is defined
in the `aws-auth` ConfigMap. Some options to consider:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
# cluster: eks.Cluster
# your_current_role: iam.Role
# vpc: ec2.Vpc


# Option 1: Add your current assumed IAM role to system:masters. Make sure to add relevant policies.
cluster.aws_auth.add_masters_role(your_current_role)

your_current_role.add_to_policy(iam.PolicyStatement(
    actions=["eks:AccessKubernetesApi", "eks:Describe*", "eks:List*"
    ],
    resources=[cluster.cluster_arn]
))
```

```python
# Option 2: create your custom mastersRole with scoped assumeBy arn as the Cluster prop. Switch to this role from the AWS console.
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
# vpc: ec2.Vpc


masters_role = iam.Role(self, "MastersRole",
    assumed_by=iam.ArnPrincipal("arn_for_trusted_principal")
)

cluster = eks.Cluster(self, "EksCluster",
    vpc=vpc,
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "KubectlLayer"),
    masters_role=masters_role
)

masters_role.add_to_policy(iam.PolicyStatement(
    actions=["eks:AccessKubernetesApi", "eks:Describe*", "eks:List*"
    ],
    resources=[cluster.cluster_arn]
))
```

```python
# Option 3: Create a new role that allows the account root principal to assume. Add this role in the `system:masters` and witch to this role from the AWS console.
# cluster: eks.Cluster


console_read_only_role = iam.Role(self, "ConsoleReadOnlyRole",
    assumed_by=iam.ArnPrincipal("arn_for_trusted_principal")
)
console_read_only_role.add_to_policy(iam.PolicyStatement(
    actions=["eks:AccessKubernetesApi", "eks:Describe*", "eks:List*"
    ],
    resources=[cluster.cluster_arn]
))

# Add this role to system:masters RBAC group
cluster.aws_auth.add_masters_role(console_read_only_role)
```

### Access Config

Amazon EKS supports three modes of authentication: `CONFIG_MAP`, `API_AND_CONFIG_MAP`, and `API`. You can enable cluster
to use access entry APIs by using authenticationMode `API` or `API_AND_CONFIG_MAP`. Use authenticationMode `CONFIG_MAP`
to continue using aws-auth configMap exclusively. When `API_AND_CONFIG_MAP` is enabled, the cluster will source authenticated
AWS IAM principals from both Amazon EKS access entry APIs and the aws-auth configMap, with priority given to the access entry API.

To specify the `authenticationMode`:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
# vpc: ec2.Vpc


eks.Cluster(self, "Cluster",
    vpc=vpc,
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "KubectlLayer"),
    authentication_mode=eks.AuthenticationMode.API_AND_CONFIG_MAP
)
```

> **Note** - Switching authentication modes on an existing cluster is a one-way operation. You can switch from
> `CONFIG_MAP` to `API_AND_CONFIG_MAP`. You can then switch from `API_AND_CONFIG_MAP` to `API`.
> You cannot revert these operations in the opposite direction. Meaning you cannot switch back to
> `CONFIG_MAP` or `API_AND_CONFIG_MAP` from `API`. And you cannot switch back to `CONFIG_MAP` from `API_AND_CONFIG_MAP`.

Read [A deep dive into simplified Amazon EKS access management controls
](https://aws.amazon.com/blogs/containers/a-deep-dive-into-simplified-amazon-eks-access-management-controls/) for more details.

You can disable granting the cluster admin permissions to the cluster creator role on bootstrapping by setting
`bootstrapClusterCreatorAdminPermissions` to false.

> **Note** - Switching `bootstrapClusterCreatorAdminPermissions` on an existing cluster would cause cluster replacement and should be avoided in production.

### Access Entry

An access entry is a cluster identity—directly linked to an AWS IAM principal user or role that is used to authenticate to
an Amazon EKS cluster. An Amazon EKS access policy authorizes an access entry to perform specific cluster actions.

Access policies are Amazon EKS-specific policies that assign Kubernetes permissions to access entries. Amazon EKS supports
only predefined and AWS managed policies. Access policies are not AWS IAM entities and are defined and managed by Amazon EKS.
Amazon EKS access policies include permission sets that support common use cases of administration, editing, or read-only access
to Kubernetes resources. See [Access Policy Permissions](https://docs.aws.amazon.com/eks/latest/userguide/access-policies.html#access-policy-permissions) for more details.

Use `AccessPolicy` to include predefined AWS managed policies:

```python
# AmazonEKSClusterAdminPolicy with `cluster` scope
eks.AccessPolicy.from_access_policy_name("AmazonEKSClusterAdminPolicy",
    access_scope_type=eks.AccessScopeType.CLUSTER
)
# AmazonEKSAdminPolicy with `namespace` scope
eks.AccessPolicy.from_access_policy_name("AmazonEKSAdminPolicy",
    access_scope_type=eks.AccessScopeType.NAMESPACE,
    namespaces=["foo", "bar"]
)
```

Use `grantAccess()` to grant the AccessPolicy to an IAM principal:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
# vpc: ec2.Vpc


cluster_admin_role = iam.Role(self, "ClusterAdminRole",
    assumed_by=iam.ArnPrincipal("arn_for_trusted_principal")
)

eks_admin_role = iam.Role(self, "EKSAdminRole",
    assumed_by=iam.ArnPrincipal("arn_for_trusted_principal")
)

eks_admin_view_role = iam.Role(self, "EKSAdminViewRole",
    assumed_by=iam.ArnPrincipal("arn_for_trusted_principal")
)

cluster = eks.Cluster(self, "Cluster",
    vpc=vpc,
    masters_role=cluster_admin_role,
    version=eks.KubernetesVersion.V1_32,
    kubectl_layer=KubectlV32Layer(self, "KubectlLayer"),
    authentication_mode=eks.AuthenticationMode.API_AND_CONFIG_MAP
)

# Cluster Admin role for this cluster
cluster.grant_access("clusterAdminAccess", cluster_admin_role.role_arn, [
    eks.AccessPolicy.from_access_policy_name("AmazonEKSClusterAdminPolicy",
        access_scope_type=eks.AccessScopeType.CLUSTER
    )
])

# EKS Admin role for specified namespaces of this cluster
cluster.grant_access("eksAdminRoleAccess", eks_admin_role.role_arn, [
    eks.AccessPolicy.from_access_policy_name("AmazonEKSAdminPolicy",
        access_scope_type=eks.AccessScopeType.NAMESPACE,
        namespaces=["foo", "bar"]
    )
])

# EKS Admin Viewer role for specified namespaces of this cluster
cluster.grant_access("eksAdminViewRoleAccess", eks_admin_view_role.role_arn, [
    eks.AccessPolicy.from_access_policy_name("AmazonEKSAdminViewPolicy",
        access_scope_type=eks.AccessScopeType.NAMESPACE,
        namespaces=["foo", "bar"]
    )
])
```

### Migrating from ConfigMap to Access Entry

If the cluster is created with the `authenticationMode` property left undefined,
it will default to `CONFIG_MAP`.

The update path is:

`undefined`(`CONFIG_MAP`) -> `API_AND_CONFIG_MAP` -> `API`

If you have explicitly declared `AwsAuth` resources and then try to switch to the `API` mode, which no longer supports the
`ConfigMap`, AWS CDK will throw an error as a protective measure to prevent you from losing all the access entries in the `ConfigMap`. In this case, you will need to remove all the declared `AwsAuth` resources explicitly and define the access entries before you are allowed to transition to the `API` mode.

> **Note** - This is a one-way transition. Once you switch to the `API` mode,
> you will not be able to switch back. Therefore, it is crucial to ensure that you have defined all the necessary access entries before making the switch to the `API` mode.

### Cluster Security Group

When you create an Amazon EKS cluster, a [cluster security group](https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html)
is automatically created as well. This security group is designed to allow all traffic from the control plane and managed node groups to flow freely
between each other.

The ID for that security group can be retrieved after creating the cluster.

```python
# cluster: eks.Cluster

cluster_security_group_id = cluster.cluster_security_group_id
```

### Node SSH Access

If you want to be able to SSH into your worker nodes, you must already have an SSH key in the region you're connecting to and pass it when
you add capacity to the cluster. You must also be able to connect to the hosts (meaning they must have a public IP and you
should be allowed to connect to them on port 22):

See [SSH into nodes](test/example.ssh-into-nodes.lit.ts) for a code example.

If you want to SSH into nodes in a private subnet, you should set up a bastion host in a public subnet. That setup is recommended, but is
unfortunately beyond the scope of this documentation.

### Service Accounts

With services account you can provide Kubernetes Pods access to AWS resources.

```python
# cluster: eks.Cluster

# add service account
service_account = cluster.add_service_account("MyServiceAccount")

bucket = s3.Bucket(self, "Bucket")
bucket.grant_read_write(service_account)

mypod = cluster.add_manifest("mypod", {
    "api_version": "v1",
    "kind": "Pod",
    "metadata": {"name": "mypod"},
    "spec": {
        "service_account_name": service_account.service_account_name,
        "containers": [{
            "name": "hello",
            "image": "paulbouwer/hello-kubernetes:1.5",
            "ports": [{"container_port": 8080}]
        }
        ]
    }
})

# create the resource after the service account.
mypod.node.add_dependency(service_account)

# print the IAM role arn for this service account
CfnOutput(self, "ServiceAccountIamRole", value=service_account.role.role_arn)
```

Note that using `serviceAccount.serviceAccountName` above **does not** translate into a resource dependency.
This is why an explicit dependency is needed. See [https://github.com/aws/aws-cdk/issues/9910](https://github.com/aws/aws-cdk/issues/9910) for more details.

It is possible to pass annotations and labels to the service account.

```python
# cluster: eks.Cluster

# add service account with annotations and labels
service_account = cluster.add_service_account("MyServiceAccount",
    annotations={
        "eks.amazonaws.com/sts-regional-endpoints": "false"
    },
    labels={
        "some-label": "with-some-value"
    }
)
```

You can also add service accounts to existing clusters.
To do so, pass the `openIdConnectProvider` property when you import the cluster into the application.

```python
# or create a new one using an existing issuer url
# issuer_url: str
# you can import an existing provider
provider = eks.OpenIdConnectProvider.from_open_id_connect_provider_arn(self, "Provider", "arn:aws:iam::123456:oidc-provider/oidc.eks.eu-west-1.amazonaws.com/id/AB123456ABC")
provider2 = eks.OpenIdConnectProvider(self, "Provider",
    url=issuer_url
)

cluster = eks.Cluster.from_cluster_attributes(self, "MyCluster",
    cluster_name="Cluster",
    open_id_connect_provider=provider,
    kubectl_role_arn="arn:aws:iam::123456:role/service-role/k8sservicerole"
)

service_account = cluster.add_service_account("MyServiceAccount")

bucket = s3.Bucket(self, "Bucket")
bucket.grant_read_write(service_account)
```

Note that adding service accounts requires running `kubectl` commands against the cluster.
This means you must also pass the `kubectlRoleArn` when importing the cluster.
See [Using existing Clusters](https://github.com/aws/aws-cdk/tree/main/packages/aws-cdk-lib/aws-eks#using-existing-clusters).

### Pod Identities

[Amazon EKS Pod Identities](https://docs.aws.amazon.com/eks/latest/userguide/pod-identities.html) is a feature that simplifies how
Kubernetes applications running on Amazon EKS can obtain AWS IAM credentials. It provides a way to associate an IAM role with a
Kubernetes service account, allowing pods to retrieve temporary AWS credentials without the need
to manage IAM roles and policies directly.

By default, `ServiceAccount` creates an `OpenIdConnectProvider` for
[IRSA(IAM roles for service accounts)](https://docs.aws.amazon.com/eks/latest/userguide/iam-roles-for-service-accounts.html) if
`identityType` is `undefined` or `IdentityType.IRSA`.

You may opt in Amaozn EKS Pod Identities as below:

```python
# cluster: eks.Cluster


eks.ServiceAccount(self, "ServiceAccount",
    cluster=cluster,
    name="test-sa",
    namespace="default",
    identity_type=eks.IdentityType.POD_IDENTITY
)
```

When you create the ServiceAccount with the `identityType` set to `POD_IDENTITY`,
`ServiceAccount` contruct will perform the following actions behind the scenes:

1. It will create an IAM role with the necessary trust policy to allow the "pods.eks.amazonaws.com" principal to assume the role.
   This trust policy grants the EKS service the permission to retrieve temporary AWS credentials on behalf of the pods using this service account.
2. It will enable the "Amazon EKS Pod Identity Agent" add-on on the EKS cluster. This add-on is responsible for managing the temporary
   AWS credentials and making them available to the pods.
3. It will create an association between the IAM role and the Kubernetes service account. This association allows the pods using this
   service account to obtain the temporary AWS credentials from the associated IAM role.

This simplifies the process of configuring IAM permissions for your Kubernetes applications running on Amazon EKS. It handles the creation of the IAM role,
the installation of the Pod Identity Agent add-on, and the association between the role and the service account, making it easier to manage AWS credentials
for your applications.

## Applying Kubernetes Resources

The library supports several popular resource deployment mechanisms, among which are:

### Kubernetes Manifests

The `KubernetesManifest` construct or `cluster.addManifest` method can be used
to apply Kubernetes resource manifests to this cluster.

> When using `cluster.addManifest`, the manifest construct is defined within the cluster's stack scope. If the manifest contains
> attributes from a different stack which depend on the cluster stack, a circular dependency will be created and you will get a synth time error.
> To avoid this, directly use `new KubernetesManifest` to create the manifest in the scope of the other stack.

The following examples will deploy the [paulbouwer/hello-kubernetes](https://github.com/paulbouwer/hello-kubernetes)
service on the cluster:

```python
# cluster: eks.Cluster

app_label = {"app": "hello-kubernetes"}

deployment = {
    "api_version": "apps/v1",
    "kind": "Deployment",
    "metadata": {"name": "hello-kubernetes"},
    "spec": {
        "replicas": 3,
        "selector": {"match_labels": app_label},
        "template": {
            "metadata": {"labels": app_label},
            "spec": {
                "containers": [{
                    "name": "hello-kubernetes",
                    "image": "paulbouwer/hello-kubernetes:1.5",
                    "ports": [{"container_port": 8080}]
                }
                ]
            }
        }
    }
}

service = {
    "api_version": "v1",
    "kind": "Service",
    "metadata": {"name": "hello-kubernetes"},
    "spec": {
        "type": "LoadBalancer",
        "ports": [{"port": 80, "target_port": 8080}],
        "selector": app_label
    }
}

# option 1: use a construct
eks.KubernetesManifest(self, "hello-kub",
    cluster=cluster,
    manifest=[deployment, service]
)

# or, option2: use `addManifest`
cluster.add_manifest("hello-kub", service, deployment)
```

#### ALB Controller Integration

The `KubernetesManifest` construct can detect ingress resources inside your manifest and automatically add the necessary annotations
so they are picked up by the ALB Controller.

> See [Alb Controller](#alb-controller)

To that end, it offers the following properties:

* `ingressAlb` - Signal that the ingress detection should be done.
* `ingressAlbScheme` - Which ALB scheme should be applied. Defaults to `internal`.

#### Adding resources from a URL

The following example will deploy the resource manifest hosting on remote server:

```text
// This example is only available in TypeScript

import * as yaml from 'js-yaml';
import * as request from 'sync-request';

declare const cluster: eks.Cluster;
const manifestUrl = 'https://url/of/manifest.yaml';
const manifest = yaml.safeLoadAll(request('GET', manifestUrl).getBody());
cluster.addManifest('my-resource', manifest);
```

#### Dependencies

There are cases where Kubernetes resources must be deployed in a specific order.
For example, you cannot define a resource in a Kubernetes namespace before the
namespace was created.

You can represent dependencies between `KubernetesManifest`s using
`resource.node.addDependency()`:

```python
# cluster: eks.Cluster

namespace = cluster.add_manifest("my-namespace", {
    "api_version": "v1",
    "kind": "Namespace",
    "metadata": {"name": "my-app"}
})

service = cluster.add_manifest("my-service", {
    "metadata": {
        "name": "myservice",
        "namespace": "my-app"
    },
    "spec": {}
})

service.node.add_dependency(namespace)
```

**NOTE:** when a `KubernetesManifest` includes multiple resources (either directly
or through `cluster.addManifest()`) (e.g. `cluster.addManifest('foo', r1, r2, r3,...)`), these resources will be applied as a single manifest via `kubectl`
and will be applied sequentially (the standard behavior in `kubectl`).

---


Since Kubernetes manifests are implemented as CloudFormation resources in the
CDK. This means that if the manifest is deleted from your code (or the stack is
deleted), the next `cdk deploy` will issue a `kubectl delete` command and the
Kubernetes resources in that manifest will be deleted.

#### Resource Pruning

When a resource is deleted from a Kubernetes manifest, the EKS module will
automatically delete these resources by injecting a *prune label* to all
manifest resources. This label is then passed to [`kubectl apply --prune`](https://kubernetes.io/docs/tasks/manage-kubernetes-objects/declarative-config/#alternative-kubectl-apply-f-directory-prune-l-your-label).

Pruning is enabled by default but can be disabled through the `prune` option
when a cluster is defined:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


eks.Cluster(self, "MyCluster",
    version=eks.KubernetesVersion.V1_32,
    prune=False,
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

#### Manifests Validation

The `kubectl` CLI supports applying a manifest by skipping the validation.
This can be accomplished by setting the `skipValidation` flag to `true` in the `KubernetesManifest` props.

```python
# cluster: eks.Cluster

eks.KubernetesManifest(self, "HelloAppWithoutValidation",
    cluster=cluster,
    manifest=[{"foo": "bar"}],
    skip_validation=True
)
```

### Helm Charts

The `HelmChart` construct or `cluster.addHelmChart` method can be used
to add Kubernetes resources to this cluster using Helm.

> When using `cluster.addHelmChart`, the manifest construct is defined within the cluster's stack scope. If the manifest contains
> attributes from a different stack which depend on the cluster stack, a circular dependency will be created and you will get a synth time error.
> To avoid this, directly use `new HelmChart` to create the chart in the scope of the other stack.

The following example will install the [NGINX Ingress Controller](https://kubernetes.github.io/ingress-nginx/) to your cluster using Helm.

```python
# cluster: eks.Cluster

# option 1: use a construct
eks.HelmChart(self, "NginxIngress",
    cluster=cluster,
    chart="nginx-ingress",
    repository="https://helm.nginx.com/stable",
    namespace="kube-system"
)

# or, option2: use `addHelmChart`
cluster.add_helm_chart("NginxIngress",
    chart="nginx-ingress",
    repository="https://helm.nginx.com/stable",
    namespace="kube-system"
)
```

Helm charts will be installed and updated using `helm upgrade --install`, where a few parameters
are being passed down (such as `repo`, `values`, `version`, `namespace`, `wait`, `timeout`, etc).
This means that if the chart is added to CDK with the same release name, it will try to update
the chart in the cluster.

Additionally, the `chartAsset` property can be an `aws-s3-assets.Asset`. This allows the use of local, private helm charts.

```python
import aws_cdk.aws_s3_assets as s3_assets

# cluster: eks.Cluster

chart_asset = s3_assets.Asset(self, "ChartAsset",
    path="/path/to/asset"
)

cluster.add_helm_chart("test-chart",
    chart_asset=chart_asset
)
```

Nested values passed to the `values` parameter should be provided as a nested dictionary:

```python
# cluster: eks.Cluster


cluster.add_helm_chart("ExternalSecretsOperator",
    chart="external-secrets",
    release="external-secrets",
    repository="https://charts.external-secrets.io",
    namespace="external-secrets",
    values={
        "install_cRDs": True,
        "webhook": {
            "port": 9443
        }
    }
)
```

Helm chart can come with Custom Resource Definitions (CRDs) defined that by default will be installed by helm as well. However in special cases it might be needed to skip the installation of CRDs, for that the property `skipCrds` can be used.

```python
# cluster: eks.Cluster

# option 1: use a construct
eks.HelmChart(self, "NginxIngress",
    cluster=cluster,
    chart="nginx-ingress",
    repository="https://helm.nginx.com/stable",
    namespace="kube-system",
    skip_crds=True
)
```

### OCI Charts

OCI charts are also supported.
Also replace the `${VARS}` with appropriate values.

```python
# cluster: eks.Cluster

# option 1: use a construct
eks.HelmChart(self, "MyOCIChart",
    cluster=cluster,
    chart="some-chart",
    repository="oci://${ACCOUNT_ID}.dkr.ecr.${ACCOUNT_REGION}.amazonaws.com/${REPO_NAME}",
    namespace="oci",
    version="0.0.1"
)
```

Helm charts are implemented as CloudFormation resources in CDK.
This means that if the chart is deleted from your code (or the stack is
deleted), the next `cdk deploy` will issue a `helm uninstall` command and the
Helm chart will be deleted.

When there is no `release` defined, a unique ID will be allocated for the release based
on the construct path.

By default, all Helm charts will be installed concurrently. In some cases, this
could cause race conditions where two Helm charts attempt to deploy the same
resource or if Helm charts depend on each other. You can use
`chart.node.addDependency()` in order to declare a dependency order between
charts:

```python
# cluster: eks.Cluster

chart1 = cluster.add_helm_chart("MyChart",
    chart="foo"
)
chart2 = cluster.add_helm_chart("MyChart",
    chart="bar"
)

chart2.node.add_dependency(chart1)
```

### CDK8s Charts

[CDK8s](https://cdk8s.io/) is an open-source library that enables Kubernetes manifest authoring using familiar programming languages. It is founded on the same technologies as the AWS CDK, such as [`constructs`](https://github.com/aws/constructs) and [`jsii`](https://github.com/aws/jsii).

> To learn more about cdk8s, visit the [Getting Started](https://cdk8s.io/docs/latest/getting-started/) tutorials.

The EKS module natively integrates with cdk8s and allows you to apply cdk8s charts on AWS EKS clusters via the `cluster.addCdk8sChart` method.

In addition to `cdk8s`, you can also use [`cdk8s+`](https://cdk8s.io/docs/latest/plus/), which provides higher level abstraction for the core kubernetes api objects.
You can think of it like the `L2` constructs for Kubernetes. Any other `cdk8s` based libraries are also supported, for example [`cdk8s-debore`](https://github.com/toricls/cdk8s-debore).

To get started, add the following dependencies to your `package.json` file:

```json
"dependencies": {
  "cdk8s": "^2.0.0",
  "cdk8s-plus-25": "^2.0.0",
  "constructs": "^10.0.0"
}
```

Note that here we are using `cdk8s-plus-25` as we are targeting Kubernetes version 1.25.0. If you operate a different kubernetes version, you should
use the corresponding `cdk8s-plus-XX` library.
See [Select the appropriate cdk8s+ library](https://cdk8s.io/docs/latest/plus/#i-operate-kubernetes-version-1xx-which-cdk8s-library-should-i-be-using)
for more details.

Similarly to how you would create a stack by extending `aws-cdk-lib.Stack`, we recommend you create a chart of your own that extends `cdk8s.Chart`,
and add your kubernetes resources to it. You can use `aws-cdk` construct attributes and properties inside your `cdk8s` construct freely.

In this example we create a chart that accepts an `s3.Bucket` and passes its name to a kubernetes pod as an environment variable.

`+ my-chart.ts`

```python
from aws_cdk import aws_s3 as s3
import constructs as constructs
import cdk8s as cdk8s
import cdk8s_plus_25 as kplus

class MyChart(cdk8s.Chart):
    def __init__(self, scope, id, *, bucket):
        super().__init__(scope, id)

        kplus.Pod(self, "Pod",
            containers=[kplus.ContainerProps(
                image="my-image",
                env_variables={
                    "BUCKET_NAME": kplus.EnvValue.from_value(bucket.bucket_name)
                }
            )
            ]
        )
```

Then, in your AWS CDK app:

```python
# cluster: eks.Cluster


# some bucket..
bucket = s3.Bucket(self, "Bucket")

# create a cdk8s chart and use `cdk8s.App` as the scope.
my_chart = MyChart(cdk8s.App(), "MyChart", bucket=bucket)

# add the cdk8s chart to the cluster
cluster.add_cdk8s_chart("my-chart", my_chart)
```

#### Custom CDK8s Constructs

You can also compose a few stock `cdk8s+` constructs into your own custom construct. However, since mixing scopes between `aws-cdk` and `cdk8s` is currently not supported, the `Construct` class
you'll need to use is the one from the [`constructs`](https://github.com/aws/constructs) module, and not from `aws-cdk-lib` like you normally would.
This is why we used `new cdk8s.App()` as the scope of the chart above.

```python
import constructs as constructs
import cdk8s as cdk8s
import cdk8s_plus_25 as kplus

app = cdk8s.App()
chart = cdk8s.Chart(app, "my-chart")

class LoadBalancedWebService(constructs.Construct):
    def __init__(self, scope, id, props):
        super().__init__(scope, id)

        deployment = kplus.Deployment(chart, "Deployment",
            replicas=props.replicas,
            containers=[kplus.Container(image=props.image)]
        )

        deployment.expose_via_service(
            ports=[kplus.ServicePort(port=props.port)
            ],
            service_type=kplus.ServiceType.LOAD_BALANCER
        )
```

#### Manually importing k8s specs and CRD's

If you find yourself unable to use `cdk8s+`, or just like to directly use the `k8s` native objects or CRD's, you can do so by manually importing them using the `cdk8s-cli`.

See [Importing kubernetes objects](https://cdk8s.io/docs/latest/cli/import/) for detailed instructions.

## Patching Kubernetes Resources

The `KubernetesPatch` construct can be used to update existing kubernetes
resources. The following example can be used to patch the `hello-kubernetes`
deployment from the example above with 5 replicas.

```python
# cluster: eks.Cluster

eks.KubernetesPatch(self, "hello-kub-deployment-label",
    cluster=cluster,
    resource_name="deployment/hello-kubernetes",
    apply_patch={"spec": {"replicas": 5}},
    restore_patch={"spec": {"replicas": 3}}
)
```

## Querying Kubernetes Resources

The `KubernetesObjectValue` construct can be used to query for information about kubernetes objects,
and use that as part of your CDK application.

For example, you can fetch the address of a [`LoadBalancer`](https://kubernetes.io/docs/concepts/services-networking/service/#loadbalancer) type service:

```python
# cluster: eks.Cluster

# query the load balancer address
my_service_address = eks.KubernetesObjectValue(self, "LoadBalancerAttribute",
    cluster=cluster,
    object_type="service",
    object_name="my-service",
    json_path=".status.loadBalancer.ingress[0].hostname"
)

# pass the address to a lambda function
proxy_function = lambda_.Function(self, "ProxyFunction",
    handler="index.handler",
    code=lambda_.Code.from_inline("my-code"),
    runtime=lambda_.Runtime.NODEJS_LATEST,
    environment={
        "my_service_address": my_service_address.value
    }
)
```

Specifically, since the above use-case is quite common, there is an easier way to access that information:

```python
# cluster: eks.Cluster

load_balancer_address = cluster.get_service_load_balancer_address("my-service")
```

## Add-ons

[Add-ons](https://docs.aws.amazon.com/eks/latest/userguide/eks-add-ons.html) is a software that provides supporting operational capabilities to Kubernetes applications. The EKS module supports adding add-ons to your cluster using the `eks.Addon` class.

```python
# cluster: eks.Cluster


eks.Addon(self, "Addon",
    cluster=cluster,
    addon_name="aws-guardduty-agent",
    addon_version="v1.6.1",
    # whether to preserve the add-on software on your cluster but Amazon EKS stops managing any settings for the add-on.
    preserve_on_delete=False
)
```

## Using existing clusters

The Amazon EKS library allows defining Kubernetes resources such as [Kubernetes
manifests](#kubernetes-resources) and [Helm charts](#helm-charts) on clusters
that are not defined as part of your CDK app.

First, you'll need to "import" a cluster to your CDK app. To do that, use the
`eks.Cluster.fromClusterAttributes()` static method:

```python
cluster = eks.Cluster.from_cluster_attributes(self, "MyCluster",
    cluster_name="my-cluster-name",
    kubectl_role_arn="arn:aws:iam::1111111:role/iam-role-that-has-masters-access"
)
```

Then, you can use `addManifest` or `addHelmChart` to define resources inside
your Kubernetes cluster. For example:

```python
# cluster: eks.Cluster

cluster.add_manifest("Test", {
    "api_version": "v1",
    "kind": "ConfigMap",
    "metadata": {
        "name": "myconfigmap"
    },
    "data": {
        "Key": "value",
        "Another": "123454"
    }
})
```

At the minimum, when importing clusters for `kubectl` management, you will need
to specify:

* `clusterName` - the name of the cluster.
* `kubectlRoleArn` - the ARN of an IAM role mapped to the `system:masters` RBAC
  role. If the cluster you are importing was created using the AWS CDK, the
  CloudFormation stack has an output that includes an IAM role that can be used.
  Otherwise, you can create an IAM role and map it to `system:masters` manually.
  The trust policy of this role should include the the
  `arn:aws::iam::${accountId}:root` principal in order to allow the execution
  role of the kubectl resource to assume it.

If the cluster is configured with private-only or private and restricted public
Kubernetes [endpoint access](#endpoint-access), you must also specify:

* `kubectlSecurityGroupId` - the ID of an EC2 security group that is allowed
  connections to the cluster's control security group. For example, the EKS managed [cluster security group](#cluster-security-group).
* `kubectlPrivateSubnetIds` - a list of private VPC subnets IDs that will be used
  to access the Kubernetes endpoint.

## Logging

EKS supports cluster logging for 5 different types of events:

* API requests to the cluster.
* Cluster access via the Kubernetes API.
* Authentication requests into the cluster.
* State of cluster controllers.
* Scheduling decisions.

You can enable logging for each one separately using the `clusterLogging`
property. For example:

```python
from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer


cluster = eks.Cluster(self, "Cluster",
    # ...
    version=eks.KubernetesVersion.V1_32,
    cluster_logging=[eks.ClusterLoggingTypes.API, eks.ClusterLoggingTypes.AUTHENTICATOR, eks.ClusterLoggingTypes.SCHEDULER
    ],
    kubectl_layer=KubectlV32Layer(self, "kubectl")
)
```

## NodeGroup Repair Config

You can enable Managed Node Group [auto-repair config](https://docs.aws.amazon.com/eks/latest/userguide/node-health.html#node-auto-repair) using `enableNodeAutoRepair`
property. For example:

```python
# cluster: eks.Cluster


cluster.add_nodegroup_capacity("NodeGroup",
    enable_node_auto_repair=True
)
```

## Known Issues and Limitations

* [One cluster per stack](https://github.com/aws/aws-cdk/issues/10073)
* [Service Account dependencies](https://github.com/aws/aws-cdk/issues/9910)
* [Support isolated VPCs](https://github.com/aws/aws-cdk/issues/12171)
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    CfnResource as _CfnResource_9df397a6,
    CfnTag as _CfnTag_f6864754,
    Duration as _Duration_4839e8c3,
    IInspectable as _IInspectable_c2943556,
    IResolvable as _IResolvable_da3f097b,
    IResource as _IResource_c80c4260,
    ITaggable as _ITaggable_36806126,
    ITaggableV2 as _ITaggableV2_4e6798f8,
    NestedStack as _NestedStack_dd393a45,
    Resource as _Resource_45bc6135,
    Size as _Size_7b441c34,
    TagManager as _TagManager_0a598cb3,
    TreeInspector as _TreeInspector_488e0dd5,
)
from ..aws_autoscaling import (
    AutoScalingGroup as _AutoScalingGroup_c547a7b9,
    BlockDevice as _BlockDevice_0cfc0568,
    CapacityDistributionStrategy as _CapacityDistributionStrategy_2393ccfe,
    CommonAutoScalingGroupProps as _CommonAutoScalingGroupProps_808bbf2d,
    GroupMetrics as _GroupMetrics_7cdf729b,
    HealthCheck as _HealthCheck_03a4bd5a,
    HealthChecks as _HealthChecks_b8757873,
    Monitoring as _Monitoring_50020f91,
    NotificationConfiguration as _NotificationConfiguration_d5911670,
    Signals as _Signals_69fbeb6e,
    TerminationPolicy as _TerminationPolicy_89633c56,
    UpdatePolicy as _UpdatePolicy_6dffc7ca,
)
from ..aws_ec2 import (
    Connections as _Connections_0f31fce8,
    IConnectable as _IConnectable_10015a05,
    IKeyPair as _IKeyPair_bc344eda,
    IMachineImage as _IMachineImage_0e8bd50b,
    ISecurityGroup as _ISecurityGroup_acf8a799,
    ISubnet as _ISubnet_d57d1229,
    IVpc as _IVpc_f30d5663,
    InstanceType as _InstanceType_f64915b9,
    MachineImageConfig as _MachineImageConfig_187edaee,
    SubnetSelection as _SubnetSelection_e57d76df,
)
from ..aws_iam import (
    AddToPrincipalPolicyResult as _AddToPrincipalPolicyResult_946c9561,
    IOpenIdConnectProvider as _IOpenIdConnectProvider_203f0793,
    IPrincipal as _IPrincipal_539bb2fd,
    IRole as _IRole_235f5d8e,
    IUser as _IUser_c32311f7,
    OpenIdConnectProvider as _OpenIdConnectProvider_5cb7bc9f,
    PolicyStatement as _PolicyStatement_0fe33853,
    PrincipalPolicyFragment as _PrincipalPolicyFragment_6a855d11,
    Role as _Role_e8c6e11f,
)
from ..aws_kms import IKey as _IKey_5f11635f
from ..aws_lambda import ILayerVersion as _ILayerVersion_5ac127c8
from ..aws_s3_assets import Asset as _Asset_ac2a7e61


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AccessEntryAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "access_entry_arn": "accessEntryArn",
        "access_entry_name": "accessEntryName",
    },
)
class AccessEntryAttributes:
    def __init__(
        self,
        *,
        access_entry_arn: builtins.str,
        access_entry_name: builtins.str,
    ) -> None:
        '''Represents the attributes of an access entry.

        :param access_entry_arn: The Amazon Resource Name (ARN) of the access entry.
        :param access_entry_name: The name of the access entry.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            access_entry_attributes = eks.AccessEntryAttributes(
                access_entry_arn="accessEntryArn",
                access_entry_name="accessEntryName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea57d074f938dd093d38b977c20869681c2abd3bacc2931046328147dc4d8d72)
            check_type(argname="argument access_entry_arn", value=access_entry_arn, expected_type=type_hints["access_entry_arn"])
            check_type(argname="argument access_entry_name", value=access_entry_name, expected_type=type_hints["access_entry_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "access_entry_arn": access_entry_arn,
            "access_entry_name": access_entry_name,
        }

    @builtins.property
    def access_entry_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the access entry.'''
        result = self._values.get("access_entry_arn")
        assert result is not None, "Required property 'access_entry_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_entry_name(self) -> builtins.str:
        '''The name of the access entry.'''
        result = self._values.get("access_entry_name")
        assert result is not None, "Required property 'access_entry_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AccessEntryAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AccessEntryProps",
    jsii_struct_bases=[],
    name_mapping={
        "access_policies": "accessPolicies",
        "cluster": "cluster",
        "principal": "principal",
        "access_entry_name": "accessEntryName",
        "access_entry_type": "accessEntryType",
    },
)
class AccessEntryProps:
    def __init__(
        self,
        *,
        access_policies: typing.Sequence["IAccessPolicy"],
        cluster: "ICluster",
        principal: builtins.str,
        access_entry_name: typing.Optional[builtins.str] = None,
        access_entry_type: typing.Optional["AccessEntryType"] = None,
    ) -> None:
        '''Represents the properties required to create an Amazon EKS access entry.

        :param access_policies: The access policies that define the permissions and scope for the access entry.
        :param cluster: The Amazon EKS cluster to which the access entry applies.
        :param principal: The Amazon Resource Name (ARN) of the principal (user or role) to associate the access entry with.
        :param access_entry_name: The name of the AccessEntry. Default: - No access entry name is provided
        :param access_entry_type: The type of the AccessEntry. Default: STANDARD

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            # access_policy: eks.AccessPolicy
            # cluster: eks.Cluster
            
            access_entry_props = eks.AccessEntryProps(
                access_policies=[access_policy],
                cluster=cluster,
                principal="principal",
            
                # the properties below are optional
                access_entry_name="accessEntryName",
                access_entry_type=eks.AccessEntryType.STANDARD
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0cc467f068aa2e977dd81e9c0227cfe45f7381ea6d31cea9c6f7f80e6d83e048)
            check_type(argname="argument access_policies", value=access_policies, expected_type=type_hints["access_policies"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument principal", value=principal, expected_type=type_hints["principal"])
            check_type(argname="argument access_entry_name", value=access_entry_name, expected_type=type_hints["access_entry_name"])
            check_type(argname="argument access_entry_type", value=access_entry_type, expected_type=type_hints["access_entry_type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "access_policies": access_policies,
            "cluster": cluster,
            "principal": principal,
        }
        if access_entry_name is not None:
            self._values["access_entry_name"] = access_entry_name
        if access_entry_type is not None:
            self._values["access_entry_type"] = access_entry_type

    @builtins.property
    def access_policies(self) -> typing.List["IAccessPolicy"]:
        '''The access policies that define the permissions and scope for the access entry.'''
        result = self._values.get("access_policies")
        assert result is not None, "Required property 'access_policies' is missing"
        return typing.cast(typing.List["IAccessPolicy"], result)

    @builtins.property
    def cluster(self) -> "ICluster":
        '''The Amazon EKS cluster to which the access entry applies.'''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def principal(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the principal (user or role) to associate the access entry with.'''
        result = self._values.get("principal")
        assert result is not None, "Required property 'principal' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_entry_name(self) -> typing.Optional[builtins.str]:
        '''The name of the AccessEntry.

        :default: - No access entry name is provided
        '''
        result = self._values.get("access_entry_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def access_entry_type(self) -> typing.Optional["AccessEntryType"]:
        '''The type of the AccessEntry.

        :default: STANDARD
        '''
        result = self._values.get("access_entry_type")
        return typing.cast(typing.Optional["AccessEntryType"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AccessEntryProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.AccessEntryType")
class AccessEntryType(enum.Enum):
    '''Represents the different types of access entries that can be used in an Amazon EKS cluster.

    :enum: true
    '''

    STANDARD = "STANDARD"
    '''Represents a standard access entry.'''
    FARGATE_LINUX = "FARGATE_LINUX"
    '''Represents a Fargate Linux access entry.'''
    EC2_LINUX = "EC2_LINUX"
    '''Represents an EC2 Linux access entry.'''
    EC2_WINDOWS = "EC2_WINDOWS"
    '''Represents an EC2 Windows access entry.'''


class AccessPolicyArn(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.AccessPolicyArn",
):
    '''Represents an Amazon EKS Access Policy ARN.

    Amazon EKS Access Policies are used to control access to Amazon EKS clusters.

    :see: https://docs.aws.amazon.com/eks/latest/userguide/access-policies.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        access_policy_arn = eks.AccessPolicyArn.AMAZON_EKS_ADMIN_POLICY
    '''

    def __init__(self, policy_name: builtins.str) -> None:
        '''Constructs a new instance of the ``AccessEntry`` class.

        :param policy_name: - The name of the Amazon EKS access policy. This is used to construct the policy ARN.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0ac946189967c669f9d1c47c0772f99ed1ce20197e6c76e4496836bbe19d2a72)
            check_type(argname="argument policy_name", value=policy_name, expected_type=type_hints["policy_name"])
        jsii.create(self.__class__, self, [policy_name])

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, policy_name: builtins.str) -> "AccessPolicyArn":
        '''Creates a new instance of the AccessPolicy class with the specified policy name.

        :param policy_name: The name of the access policy.

        :return: A new instance of the AccessPolicy class.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__23236f8d1dae1650ef58623c900288d55afe1fd4ead26b7b1aff290d073de854)
            check_type(argname="argument policy_name", value=policy_name, expected_type=type_hints["policy_name"])
        return typing.cast("AccessPolicyArn", jsii.sinvoke(cls, "of", [policy_name]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_EKS_ADMIN_POLICY")
    def AMAZON_EKS_ADMIN_POLICY(cls) -> "AccessPolicyArn":
        '''The Amazon EKS Admin Policy.

        This access policy includes permissions that grant an IAM principal
        most permissions to resources. When associated to an access entry, its access scope is typically
        one or more Kubernetes namespaces.
        '''
        return typing.cast("AccessPolicyArn", jsii.sget(cls, "AMAZON_EKS_ADMIN_POLICY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_EKS_ADMIN_VIEW_POLICY")
    def AMAZON_EKS_ADMIN_VIEW_POLICY(cls) -> "AccessPolicyArn":
        '''The Amazon EKS Admin View Policy.

        This access policy includes permissions that grant an IAM principal
        access to list/view all resources in a cluster.
        '''
        return typing.cast("AccessPolicyArn", jsii.sget(cls, "AMAZON_EKS_ADMIN_VIEW_POLICY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_EKS_CLUSTER_ADMIN_POLICY")
    def AMAZON_EKS_CLUSTER_ADMIN_POLICY(cls) -> "AccessPolicyArn":
        '''The Amazon EKS Cluster Admin Policy.

        This access policy includes permissions that grant an IAM
        principal administrator access to a cluster. When associated to an access entry, its access scope
        is typically the cluster, rather than a Kubernetes namespace.
        '''
        return typing.cast("AccessPolicyArn", jsii.sget(cls, "AMAZON_EKS_CLUSTER_ADMIN_POLICY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_EKS_EDIT_POLICY")
    def AMAZON_EKS_EDIT_POLICY(cls) -> "AccessPolicyArn":
        '''The Amazon EKS Edit Policy.

        This access policy includes permissions that allow an IAM principal
        to edit most Kubernetes resources.
        '''
        return typing.cast("AccessPolicyArn", jsii.sget(cls, "AMAZON_EKS_EDIT_POLICY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_EKS_VIEW_POLICY")
    def AMAZON_EKS_VIEW_POLICY(cls) -> "AccessPolicyArn":
        '''The Amazon EKS View Policy.

        This access policy includes permissions that grant an IAM principal
        access to list/view all resources in a cluster.
        '''
        return typing.cast("AccessPolicyArn", jsii.sget(cls, "AMAZON_EKS_VIEW_POLICY"))

    @builtins.property
    @jsii.member(jsii_name="policyArn")
    def policy_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the access policy.'''
        return typing.cast(builtins.str, jsii.get(self, "policyArn"))

    @builtins.property
    @jsii.member(jsii_name="policyName")
    def policy_name(self) -> builtins.str:
        '''- The name of the Amazon EKS access policy.

        This is used to construct the policy ARN.
        '''
        return typing.cast(builtins.str, jsii.get(self, "policyName"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AccessPolicyNameOptions",
    jsii_struct_bases=[],
    name_mapping={"access_scope_type": "accessScopeType", "namespaces": "namespaces"},
)
class AccessPolicyNameOptions:
    def __init__(
        self,
        *,
        access_scope_type: "AccessScopeType",
        namespaces: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Represents the options required to create an Amazon EKS Access Policy using the ``fromAccessPolicyName()`` method.

        :param access_scope_type: The scope of the access policy. This determines the level of access granted by the policy.
        :param namespaces: An optional array of Kubernetes namespaces to which the access policy applies. Default: - no specific namespaces for this scope

        :exampleMetadata: infused

        Example::

            # AmazonEKSClusterAdminPolicy with `cluster` scope
            eks.AccessPolicy.from_access_policy_name("AmazonEKSClusterAdminPolicy",
                access_scope_type=eks.AccessScopeType.CLUSTER
            )
            # AmazonEKSAdminPolicy with `namespace` scope
            eks.AccessPolicy.from_access_policy_name("AmazonEKSAdminPolicy",
                access_scope_type=eks.AccessScopeType.NAMESPACE,
                namespaces=["foo", "bar"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__249ef277acd24f14314e76608ae6685b8ec176bb0872ba8327c446714e5afaee)
            check_type(argname="argument access_scope_type", value=access_scope_type, expected_type=type_hints["access_scope_type"])
            check_type(argname="argument namespaces", value=namespaces, expected_type=type_hints["namespaces"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "access_scope_type": access_scope_type,
        }
        if namespaces is not None:
            self._values["namespaces"] = namespaces

    @builtins.property
    def access_scope_type(self) -> "AccessScopeType":
        '''The scope of the access policy.

        This determines the level of access granted by the policy.
        '''
        result = self._values.get("access_scope_type")
        assert result is not None, "Required property 'access_scope_type' is missing"
        return typing.cast("AccessScopeType", result)

    @builtins.property
    def namespaces(self) -> typing.Optional[typing.List[builtins.str]]:
        '''An optional array of Kubernetes namespaces to which the access policy applies.

        :default: - no specific namespaces for this scope
        '''
        result = self._values.get("namespaces")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AccessPolicyNameOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AccessPolicyProps",
    jsii_struct_bases=[],
    name_mapping={"access_scope": "accessScope", "policy": "policy"},
)
class AccessPolicyProps:
    def __init__(
        self,
        *,
        access_scope: typing.Union["AccessScope", typing.Dict[builtins.str, typing.Any]],
        policy: AccessPolicyArn,
    ) -> None:
        '''Properties for configuring an Amazon EKS Access Policy.

        :param access_scope: The scope of the access policy, which determines the level of access granted.
        :param policy: The access policy itself, which defines the specific permissions.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            # access_policy_arn: eks.AccessPolicyArn
            
            access_policy_props = eks.AccessPolicyProps(
                access_scope=eks.AccessScope(
                    type=eks.AccessScopeType.NAMESPACE,
            
                    # the properties below are optional
                    namespaces=["namespaces"]
                ),
                policy=access_policy_arn
            )
        '''
        if isinstance(access_scope, dict):
            access_scope = AccessScope(**access_scope)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d76ffc136ce61df3ec4331aefef6219d2ab1e0d4a6c6da1cd604f5d3a29a1c20)
            check_type(argname="argument access_scope", value=access_scope, expected_type=type_hints["access_scope"])
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "access_scope": access_scope,
            "policy": policy,
        }

    @builtins.property
    def access_scope(self) -> "AccessScope":
        '''The scope of the access policy, which determines the level of access granted.'''
        result = self._values.get("access_scope")
        assert result is not None, "Required property 'access_scope' is missing"
        return typing.cast("AccessScope", result)

    @builtins.property
    def policy(self) -> AccessPolicyArn:
        '''The access policy itself, which defines the specific permissions.'''
        result = self._values.get("policy")
        assert result is not None, "Required property 'policy' is missing"
        return typing.cast(AccessPolicyArn, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AccessPolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AccessScope",
    jsii_struct_bases=[],
    name_mapping={"type": "type", "namespaces": "namespaces"},
)
class AccessScope:
    def __init__(
        self,
        *,
        type: "AccessScopeType",
        namespaces: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Represents the scope of an access policy.

        The scope defines the namespaces or cluster-level access granted by the policy.

        :param type: The scope type of the policy, either 'namespace' or 'cluster'.
        :param namespaces: A Kubernetes namespace that an access policy is scoped to. A value is required if you specified namespace for Type. Default: - no specific namespaces for this scope.

        :interface: AccessScope
        :property: {AccessScopeType} type - The scope type of the policy, either 'namespace' or 'cluster'.
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            access_scope = eks.AccessScope(
                type=eks.AccessScopeType.NAMESPACE,
            
                # the properties below are optional
                namespaces=["namespaces"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5f979c2154a9a6bd2f77cf7ff51d6f83944d3c1290fcb70cdab0b403b6a0a8f8)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument namespaces", value=namespaces, expected_type=type_hints["namespaces"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }
        if namespaces is not None:
            self._values["namespaces"] = namespaces

    @builtins.property
    def type(self) -> "AccessScopeType":
        '''The scope type of the policy, either 'namespace' or 'cluster'.'''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("AccessScopeType", result)

    @builtins.property
    def namespaces(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A Kubernetes namespace that an access policy is scoped to.

        A value is required if you specified
        namespace for Type.

        :default: - no specific namespaces for this scope.
        '''
        result = self._values.get("namespaces")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AccessScope(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.AccessScopeType")
class AccessScopeType(enum.Enum):
    '''Represents the scope type of an access policy.

    The scope type determines the level of access granted by the policy.

    :enum: true
    :export: true
    :exampleMetadata: infused

    Example::

        # AmazonEKSClusterAdminPolicy with `cluster` scope
        eks.AccessPolicy.from_access_policy_name("AmazonEKSClusterAdminPolicy",
            access_scope_type=eks.AccessScopeType.CLUSTER
        )
        # AmazonEKSAdminPolicy with `namespace` scope
        eks.AccessPolicy.from_access_policy_name("AmazonEKSAdminPolicy",
            access_scope_type=eks.AccessScopeType.NAMESPACE,
            namespaces=["foo", "bar"]
        )
    '''

    NAMESPACE = "NAMESPACE"
    '''The policy applies to a specific namespace within the cluster.'''
    CLUSTER = "CLUSTER"
    '''The policy applies to the entire cluster.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AddonAttributes",
    jsii_struct_bases=[],
    name_mapping={"addon_name": "addonName", "cluster_name": "clusterName"},
)
class AddonAttributes:
    def __init__(self, *, addon_name: builtins.str, cluster_name: builtins.str) -> None:
        '''Represents the attributes of an addon for an Amazon EKS cluster.

        :param addon_name: The name of the addon.
        :param cluster_name: The name of the Amazon EKS cluster the addon is associated with.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            addon_attributes = eks.AddonAttributes(
                addon_name="addonName",
                cluster_name="clusterName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__77d0746018f1cbe3ed090d492996344d98293f7a76446705a3ac043408c02cfe)
            check_type(argname="argument addon_name", value=addon_name, expected_type=type_hints["addon_name"])
            check_type(argname="argument cluster_name", value=cluster_name, expected_type=type_hints["cluster_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "addon_name": addon_name,
            "cluster_name": cluster_name,
        }

    @builtins.property
    def addon_name(self) -> builtins.str:
        '''The name of the addon.'''
        result = self._values.get("addon_name")
        assert result is not None, "Required property 'addon_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cluster_name(self) -> builtins.str:
        '''The name of the Amazon EKS cluster the addon is associated with.'''
        result = self._values.get("cluster_name")
        assert result is not None, "Required property 'cluster_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddonAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AddonProps",
    jsii_struct_bases=[],
    name_mapping={
        "addon_name": "addonName",
        "cluster": "cluster",
        "addon_version": "addonVersion",
        "preserve_on_delete": "preserveOnDelete",
    },
)
class AddonProps:
    def __init__(
        self,
        *,
        addon_name: builtins.str,
        cluster: "ICluster",
        addon_version: typing.Optional[builtins.str] = None,
        preserve_on_delete: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Properties for creating an Amazon EKS Add-On.

        :param addon_name: Name of the Add-On.
        :param cluster: The EKS cluster the Add-On is associated with.
        :param addon_version: Version of the Add-On. You can check all available versions with describe-addon-versions. For example, this lists all available versions for the ``eks-pod-identity-agent`` addon: $ aws eks describe-addon-versions --addon-name eks-pod-identity-agent --query 'addons[*].addonVersions[*].addonVersion' Default: the latest version.
        :param preserve_on_delete: Specifying this option preserves the add-on software on your cluster but Amazon EKS stops managing any settings for the add-on. If an IAM account is associated with the add-on, it isn't removed. Default: true

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            
            eks.Addon(self, "Addon",
                cluster=cluster,
                addon_name="aws-guardduty-agent",
                addon_version="v1.6.1",
                # whether to preserve the add-on software on your cluster but Amazon EKS stops managing any settings for the add-on.
                preserve_on_delete=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__febc9f6cb4243d885b1b1838be38d633e7c5fc6534eaaf731f00a24653ee7591)
            check_type(argname="argument addon_name", value=addon_name, expected_type=type_hints["addon_name"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument addon_version", value=addon_version, expected_type=type_hints["addon_version"])
            check_type(argname="argument preserve_on_delete", value=preserve_on_delete, expected_type=type_hints["preserve_on_delete"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "addon_name": addon_name,
            "cluster": cluster,
        }
        if addon_version is not None:
            self._values["addon_version"] = addon_version
        if preserve_on_delete is not None:
            self._values["preserve_on_delete"] = preserve_on_delete

    @builtins.property
    def addon_name(self) -> builtins.str:
        '''Name of the Add-On.'''
        result = self._values.get("addon_name")
        assert result is not None, "Required property 'addon_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cluster(self) -> "ICluster":
        '''The EKS cluster the Add-On is associated with.'''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def addon_version(self) -> typing.Optional[builtins.str]:
        '''Version of the Add-On.

        You can check all available versions with describe-addon-versions.
        For example, this lists all available versions for the ``eks-pod-identity-agent`` addon:
        $ aws eks describe-addon-versions --addon-name eks-pod-identity-agent
        --query 'addons[*].addonVersions[*].addonVersion'

        :default: the latest version.
        '''
        result = self._values.get("addon_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def preserve_on_delete(self) -> typing.Optional[builtins.bool]:
        '''Specifying this option preserves the add-on software on your cluster but Amazon EKS stops managing any settings for the add-on.

        If an IAM account is associated with the add-on, it isn't removed.

        :default: true
        '''
        result = self._values.get("preserve_on_delete")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddonProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AlbController(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.AlbController",
):
    '''Construct for installing the AWS ALB Contoller on EKS clusters.

    Use the factory functions ``get`` and ``getOrCreate`` to obtain/create instances of this controller.

    :see: https://kubernetes-sigs.github.io/aws-load-balancer-controller
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        # alb_controller_version: eks.AlbControllerVersion
        # cluster: eks.Cluster
        # policy: Any
        
        alb_controller = eks.AlbController(self, "MyAlbController",
            cluster=cluster,
            version=alb_controller_version,
        
            # the properties below are optional
            policy=policy,
            repository="repository"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: "Cluster",
        version: "AlbControllerVersion",
        policy: typing.Any = None,
        repository: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster: [disable-awslint:ref-via-interface] Cluster to install the controller onto.
        :param version: Version of the controller.
        :param policy: The IAM policy to apply to the service account. If you're using one of the built-in versions, this is not required since CDK ships with the appropriate policies for those versions. However, if you are using a custom version, this is required (and validated). Default: - Corresponds to the predefined version.
        :param repository: The repository to pull the controller image from. Note that the default repository works for most regions, but not all. If the repository is not applicable to your region, use a custom repository according to the information here: https://github.com/kubernetes-sigs/aws-load-balancer-controller/releases. Default: '602401143452.dkr.ecr.us-west-2.amazonaws.com/amazon/aws-load-balancer-controller'
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5e2ca421e3f17c3114d53057ba096ab3f90bd3b8ed6c2e0f75f61c88dd5aed4b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AlbControllerProps(
            cluster=cluster, version=version, policy=policy, repository=repository
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="create")
    @builtins.classmethod
    def create(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        *,
        cluster: "Cluster",
        version: "AlbControllerVersion",
        policy: typing.Any = None,
        repository: typing.Optional[builtins.str] = None,
    ) -> "AlbController":
        '''Create the controller construct associated with this cluster and scope.

        Singleton per stack/cluster.

        :param scope: -
        :param cluster: [disable-awslint:ref-via-interface] Cluster to install the controller onto.
        :param version: Version of the controller.
        :param policy: The IAM policy to apply to the service account. If you're using one of the built-in versions, this is not required since CDK ships with the appropriate policies for those versions. However, if you are using a custom version, this is required (and validated). Default: - Corresponds to the predefined version.
        :param repository: The repository to pull the controller image from. Note that the default repository works for most regions, but not all. If the repository is not applicable to your region, use a custom repository according to the information here: https://github.com/kubernetes-sigs/aws-load-balancer-controller/releases. Default: '602401143452.dkr.ecr.us-west-2.amazonaws.com/amazon/aws-load-balancer-controller'
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1b3813db11381f0166360b7dc6066bdeadc4a52043da6eba56f9a55a4bfd6157)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        props = AlbControllerProps(
            cluster=cluster, version=version, policy=policy, repository=repository
        )

        return typing.cast("AlbController", jsii.sinvoke(cls, "create", [scope, props]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AlbControllerOptions",
    jsii_struct_bases=[],
    name_mapping={
        "version": "version",
        "policy": "policy",
        "repository": "repository",
    },
)
class AlbControllerOptions:
    def __init__(
        self,
        *,
        version: "AlbControllerVersion",
        policy: typing.Any = None,
        repository: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options for ``AlbController``.

        :param version: Version of the controller.
        :param policy: The IAM policy to apply to the service account. If you're using one of the built-in versions, this is not required since CDK ships with the appropriate policies for those versions. However, if you are using a custom version, this is required (and validated). Default: - Corresponds to the predefined version.
        :param repository: The repository to pull the controller image from. Note that the default repository works for most regions, but not all. If the repository is not applicable to your region, use a custom repository according to the information here: https://github.com/kubernetes-sigs/aws-load-balancer-controller/releases. Default: '602401143452.dkr.ecr.us-west-2.amazonaws.com/amazon/aws-load-balancer-controller'

        :exampleMetadata: infused

        Example::

            from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
            
            
            eks.Cluster(self, "HelloEKS",
                version=eks.KubernetesVersion.V1_32,
                alb_controller=eks.AlbControllerOptions(
                    version=eks.AlbControllerVersion.V2_8_2
                ),
                kubectl_layer=KubectlV32Layer(self, "kubectl")
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b22ec5f19b5d1b4d655cc304c12c33352da257e2109041355aa01fc993ec3ef9)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "version": version,
        }
        if policy is not None:
            self._values["policy"] = policy
        if repository is not None:
            self._values["repository"] = repository

    @builtins.property
    def version(self) -> "AlbControllerVersion":
        '''Version of the controller.'''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return typing.cast("AlbControllerVersion", result)

    @builtins.property
    def policy(self) -> typing.Any:
        '''The IAM policy to apply to the service account.

        If you're using one of the built-in versions, this is not required since
        CDK ships with the appropriate policies for those versions.

        However, if you are using a custom version, this is required (and validated).

        :default: - Corresponds to the predefined version.
        '''
        result = self._values.get("policy")
        return typing.cast(typing.Any, result)

    @builtins.property
    def repository(self) -> typing.Optional[builtins.str]:
        '''The repository to pull the controller image from.

        Note that the default repository works for most regions, but not all.
        If the repository is not applicable to your region, use a custom repository
        according to the information here: https://github.com/kubernetes-sigs/aws-load-balancer-controller/releases.

        :default: '602401143452.dkr.ecr.us-west-2.amazonaws.com/amazon/aws-load-balancer-controller'
        '''
        result = self._values.get("repository")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AlbControllerOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AlbControllerProps",
    jsii_struct_bases=[AlbControllerOptions],
    name_mapping={
        "version": "version",
        "policy": "policy",
        "repository": "repository",
        "cluster": "cluster",
    },
)
class AlbControllerProps(AlbControllerOptions):
    def __init__(
        self,
        *,
        version: "AlbControllerVersion",
        policy: typing.Any = None,
        repository: typing.Optional[builtins.str] = None,
        cluster: "Cluster",
    ) -> None:
        '''Properties for ``AlbController``.

        :param version: Version of the controller.
        :param policy: The IAM policy to apply to the service account. If you're using one of the built-in versions, this is not required since CDK ships with the appropriate policies for those versions. However, if you are using a custom version, this is required (and validated). Default: - Corresponds to the predefined version.
        :param repository: The repository to pull the controller image from. Note that the default repository works for most regions, but not all. If the repository is not applicable to your region, use a custom repository according to the information here: https://github.com/kubernetes-sigs/aws-load-balancer-controller/releases. Default: '602401143452.dkr.ecr.us-west-2.amazonaws.com/amazon/aws-load-balancer-controller'
        :param cluster: [disable-awslint:ref-via-interface] Cluster to install the controller onto.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            # alb_controller_version: eks.AlbControllerVersion
            # cluster: eks.Cluster
            # policy: Any
            
            alb_controller_props = eks.AlbControllerProps(
                cluster=cluster,
                version=alb_controller_version,
            
                # the properties below are optional
                policy=policy,
                repository="repository"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9f52254abb63608be11e6e9e1ec6c94ebb428a9ab274e1bda653dd78d26cd509)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "version": version,
            "cluster": cluster,
        }
        if policy is not None:
            self._values["policy"] = policy
        if repository is not None:
            self._values["repository"] = repository

    @builtins.property
    def version(self) -> "AlbControllerVersion":
        '''Version of the controller.'''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return typing.cast("AlbControllerVersion", result)

    @builtins.property
    def policy(self) -> typing.Any:
        '''The IAM policy to apply to the service account.

        If you're using one of the built-in versions, this is not required since
        CDK ships with the appropriate policies for those versions.

        However, if you are using a custom version, this is required (and validated).

        :default: - Corresponds to the predefined version.
        '''
        result = self._values.get("policy")
        return typing.cast(typing.Any, result)

    @builtins.property
    def repository(self) -> typing.Optional[builtins.str]:
        '''The repository to pull the controller image from.

        Note that the default repository works for most regions, but not all.
        If the repository is not applicable to your region, use a custom repository
        according to the information here: https://github.com/kubernetes-sigs/aws-load-balancer-controller/releases.

        :default: '602401143452.dkr.ecr.us-west-2.amazonaws.com/amazon/aws-load-balancer-controller'
        '''
        result = self._values.get("repository")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cluster(self) -> "Cluster":
        '''[disable-awslint:ref-via-interface] Cluster to install the controller onto.'''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("Cluster", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AlbControllerProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AlbControllerVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.AlbControllerVersion",
):
    '''Controller version.

    Corresponds to the image tag of 'amazon/aws-load-balancer-controller' image.

    :exampleMetadata: infused

    Example::

        from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
        
        
        eks.Cluster(self, "HelloEKS",
            version=eks.KubernetesVersion.V1_32,
            alb_controller=eks.AlbControllerOptions(
                version=eks.AlbControllerVersion.V2_8_2
            ),
            kubectl_layer=KubectlV32Layer(self, "kubectl")
        )
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(
        cls,
        version: builtins.str,
        helm_chart_version: typing.Optional[builtins.str] = None,
    ) -> "AlbControllerVersion":
        '''Specify a custom version and an associated helm chart version.

        Use this if the version you need is not available in one of the predefined versions.
        Note that in this case, you will also need to provide an IAM policy in the controller options.

        ALB controller version and helm chart version compatibility information can be found
        here: https://github.com/aws/eks-charts/blob/v0.0.133/stable/aws-load-balancer-controller/Chart.yaml

        :param version: The version number.
        :param helm_chart_version: The version of the helm chart. Version 1.4.1 is the default version to support legacy users.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35908bd3fc3babeb7b853915f1020066b438c04104691b1f434fbdfd052a7b51)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            check_type(argname="argument helm_chart_version", value=helm_chart_version, expected_type=type_hints["helm_chart_version"])
        return typing.cast("AlbControllerVersion", jsii.sinvoke(cls, "of", [version, helm_chart_version]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_0_0")
    def V2_0_0(cls) -> "AlbControllerVersion":
        '''v2.0.0.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_0_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_0_1")
    def V2_0_1(cls) -> "AlbControllerVersion":
        '''v2.0.1.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_0_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_1_0")
    def V2_1_0(cls) -> "AlbControllerVersion":
        '''v2.1.0.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_1_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_1_1")
    def V2_1_1(cls) -> "AlbControllerVersion":
        '''v2.1.1.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_1_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_1_2")
    def V2_1_2(cls) -> "AlbControllerVersion":
        '''v2.1.2.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_1_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_1_3")
    def V2_1_3(cls) -> "AlbControllerVersion":
        '''v2.1.3.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_1_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_2_0")
    def V2_2_0(cls) -> "AlbControllerVersion":
        '''v2.0.0.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_2_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_2_1")
    def V2_2_1(cls) -> "AlbControllerVersion":
        '''v2.2.1.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_2_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_2_2")
    def V2_2_2(cls) -> "AlbControllerVersion":
        '''v2.2.2.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_2_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_2_3")
    def V2_2_3(cls) -> "AlbControllerVersion":
        '''v2.2.3.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_2_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_2_4")
    def V2_2_4(cls) -> "AlbControllerVersion":
        '''v2.2.4.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_2_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_3_0")
    def V2_3_0(cls) -> "AlbControllerVersion":
        '''v2.3.0.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_3_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_3_1")
    def V2_3_1(cls) -> "AlbControllerVersion":
        '''v2.3.1.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_3_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_4_1")
    def V2_4_1(cls) -> "AlbControllerVersion":
        '''v2.4.1.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_4_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_4_2")
    def V2_4_2(cls) -> "AlbControllerVersion":
        '''v2.4.2.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_4_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_4_3")
    def V2_4_3(cls) -> "AlbControllerVersion":
        '''v2.4.3.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_4_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_4_4")
    def V2_4_4(cls) -> "AlbControllerVersion":
        '''v2.4.4.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_4_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_4_5")
    def V2_4_5(cls) -> "AlbControllerVersion":
        '''v2.4.5.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_4_5"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_4_6")
    def V2_4_6(cls) -> "AlbControllerVersion":
        '''v2.4.6.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_4_6"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_4_7")
    def V2_4_7(cls) -> "AlbControllerVersion":
        '''v2.4.7.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_4_7"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_5_0")
    def V2_5_0(cls) -> "AlbControllerVersion":
        '''v2.5.0.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_5_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_5_1")
    def V2_5_1(cls) -> "AlbControllerVersion":
        '''v2.5.1.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_5_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_5_2")
    def V2_5_2(cls) -> "AlbControllerVersion":
        '''v2.5.2.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_5_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_5_3")
    def V2_5_3(cls) -> "AlbControllerVersion":
        '''v2.5.3.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_5_3"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_5_4")
    def V2_5_4(cls) -> "AlbControllerVersion":
        '''v2.5.4.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_5_4"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_6_0")
    def V2_6_0(cls) -> "AlbControllerVersion":
        '''v2.6.0.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_6_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_6_1")
    def V2_6_1(cls) -> "AlbControllerVersion":
        '''v2.6.1.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_6_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_6_2")
    def V2_6_2(cls) -> "AlbControllerVersion":
        '''v2.6.2.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_6_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_7_0")
    def V2_7_0(cls) -> "AlbControllerVersion":
        '''v2.7.0.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_7_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_7_1")
    def V2_7_1(cls) -> "AlbControllerVersion":
        '''v2.7.1.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_7_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_7_2")
    def V2_7_2(cls) -> "AlbControllerVersion":
        '''v2.7.2.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_7_2"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_8_0")
    def V2_8_0(cls) -> "AlbControllerVersion":
        '''v2.8.0.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_8_0"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_8_1")
    def V2_8_1(cls) -> "AlbControllerVersion":
        '''v2.8.1.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_8_1"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V2_8_2")
    def V2_8_2(cls) -> "AlbControllerVersion":
        '''v2.8.2.'''
        return typing.cast("AlbControllerVersion", jsii.sget(cls, "V2_8_2"))

    @builtins.property
    @jsii.member(jsii_name="custom")
    def custom(self) -> builtins.bool:
        '''Whether or not its a custom version.'''
        return typing.cast(builtins.bool, jsii.get(self, "custom"))

    @builtins.property
    @jsii.member(jsii_name="helmChartVersion")
    def helm_chart_version(self) -> builtins.str:
        '''The version of the helm chart to use.'''
        return typing.cast(builtins.str, jsii.get(self, "helmChartVersion"))

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> builtins.str:
        '''The version string.'''
        return typing.cast(builtins.str, jsii.get(self, "version"))


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.AlbScheme")
class AlbScheme(enum.Enum):
    '''ALB Scheme.

    :see: https://kubernetes-sigs.github.io/aws-load-balancer-controller/v2.3/guide/ingress/annotations/#scheme
    '''

    INTERNAL = "INTERNAL"
    '''The nodes of an internal load balancer have only private IP addresses.

    The DNS name of an internal load balancer is publicly resolvable to the private IP addresses of the nodes.
    Therefore, internal load balancers can only route requests from clients with access to the VPC for the load balancer.
    '''
    INTERNET_FACING = "INTERNET_FACING"
    '''An internet-facing load balancer has a publicly resolvable DNS name, so it can route requests from clients over the internet to the EC2 instances that are registered with the load balancer.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.AuthenticationMode")
class AuthenticationMode(enum.Enum):
    '''Represents the authentication mode for an Amazon EKS cluster.

    :exampleMetadata: infused

    Example::

        from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
        # vpc: ec2.Vpc
        
        
        eks.Cluster(self, "Cluster",
            vpc=vpc,
            version=eks.KubernetesVersion.V1_32,
            kubectl_layer=KubectlV32Layer(self, "KubectlLayer"),
            authentication_mode=eks.AuthenticationMode.API_AND_CONFIG_MAP
        )
    '''

    CONFIG_MAP = "CONFIG_MAP"
    '''Authenticates using a Kubernetes ConfigMap.'''
    API_AND_CONFIG_MAP = "API_AND_CONFIG_MAP"
    '''Authenticates using both the Kubernetes API server and a ConfigMap.'''
    API = "API"
    '''Authenticates using the Kubernetes API server.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AutoScalingGroupCapacityOptions",
    jsii_struct_bases=[_CommonAutoScalingGroupProps_808bbf2d],
    name_mapping={
        "allow_all_outbound": "allowAllOutbound",
        "associate_public_ip_address": "associatePublicIpAddress",
        "auto_scaling_group_name": "autoScalingGroupName",
        "az_capacity_distribution_strategy": "azCapacityDistributionStrategy",
        "block_devices": "blockDevices",
        "capacity_rebalance": "capacityRebalance",
        "cooldown": "cooldown",
        "default_instance_warmup": "defaultInstanceWarmup",
        "desired_capacity": "desiredCapacity",
        "group_metrics": "groupMetrics",
        "health_check": "healthCheck",
        "health_checks": "healthChecks",
        "ignore_unmodified_size_properties": "ignoreUnmodifiedSizeProperties",
        "instance_monitoring": "instanceMonitoring",
        "key_name": "keyName",
        "key_pair": "keyPair",
        "max_capacity": "maxCapacity",
        "max_instance_lifetime": "maxInstanceLifetime",
        "min_capacity": "minCapacity",
        "new_instances_protected_from_scale_in": "newInstancesProtectedFromScaleIn",
        "notifications": "notifications",
        "signals": "signals",
        "spot_price": "spotPrice",
        "ssm_session_permissions": "ssmSessionPermissions",
        "termination_policies": "terminationPolicies",
        "termination_policy_custom_lambda_function_arn": "terminationPolicyCustomLambdaFunctionArn",
        "update_policy": "updatePolicy",
        "vpc_subnets": "vpcSubnets",
        "instance_type": "instanceType",
        "bootstrap_enabled": "bootstrapEnabled",
        "bootstrap_options": "bootstrapOptions",
        "machine_image_type": "machineImageType",
        "map_role": "mapRole",
        "spot_interrupt_handler": "spotInterruptHandler",
    },
)
class AutoScalingGroupCapacityOptions(_CommonAutoScalingGroupProps_808bbf2d):
    def __init__(
        self,
        *,
        allow_all_outbound: typing.Optional[builtins.bool] = None,
        associate_public_ip_address: typing.Optional[builtins.bool] = None,
        auto_scaling_group_name: typing.Optional[builtins.str] = None,
        az_capacity_distribution_strategy: typing.Optional[_CapacityDistributionStrategy_2393ccfe] = None,
        block_devices: typing.Optional[typing.Sequence[typing.Union[_BlockDevice_0cfc0568, typing.Dict[builtins.str, typing.Any]]]] = None,
        capacity_rebalance: typing.Optional[builtins.bool] = None,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        default_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        desired_capacity: typing.Optional[jsii.Number] = None,
        group_metrics: typing.Optional[typing.Sequence[_GroupMetrics_7cdf729b]] = None,
        health_check: typing.Optional[_HealthCheck_03a4bd5a] = None,
        health_checks: typing.Optional[_HealthChecks_b8757873] = None,
        ignore_unmodified_size_properties: typing.Optional[builtins.bool] = None,
        instance_monitoring: typing.Optional[_Monitoring_50020f91] = None,
        key_name: typing.Optional[builtins.str] = None,
        key_pair: typing.Optional[_IKeyPair_bc344eda] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        max_instance_lifetime: typing.Optional[_Duration_4839e8c3] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        new_instances_protected_from_scale_in: typing.Optional[builtins.bool] = None,
        notifications: typing.Optional[typing.Sequence[typing.Union[_NotificationConfiguration_d5911670, typing.Dict[builtins.str, typing.Any]]]] = None,
        signals: typing.Optional[_Signals_69fbeb6e] = None,
        spot_price: typing.Optional[builtins.str] = None,
        ssm_session_permissions: typing.Optional[builtins.bool] = None,
        termination_policies: typing.Optional[typing.Sequence[_TerminationPolicy_89633c56]] = None,
        termination_policy_custom_lambda_function_arn: typing.Optional[builtins.str] = None,
        update_policy: typing.Optional[_UpdatePolicy_6dffc7ca] = None,
        vpc_subnets: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
        instance_type: _InstanceType_f64915b9,
        bootstrap_enabled: typing.Optional[builtins.bool] = None,
        bootstrap_options: typing.Optional[typing.Union["BootstrapOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        map_role: typing.Optional[builtins.bool] = None,
        spot_interrupt_handler: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for adding worker nodes.

        :param allow_all_outbound: Whether the instances can initiate connections to anywhere by default. Default: true
        :param associate_public_ip_address: Whether instances in the Auto Scaling Group should have public IP addresses associated with them. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Use subnet setting.
        :param auto_scaling_group_name: The name of the Auto Scaling group. This name must be unique per Region per account. Default: - Auto generated by CloudFormation
        :param az_capacity_distribution_strategy: The strategy for distributing instances across Availability Zones. Default: None
        :param block_devices: Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes. Each instance that is launched has an associated root device volume, either an Amazon EBS volume or an instance store volume. You can use block device mappings to specify additional EBS volumes or instance store volumes to attach to an instance when it is launched. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Uses the block device mapping of the AMI
        :param capacity_rebalance: Indicates whether Capacity Rebalancing is enabled. When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of interruption. After launching a new instance, it then terminates an old instance. Default: false
        :param cooldown: Default scaling cooldown for this AutoScalingGroup. Default: Duration.minutes(5)
        :param default_instance_warmup: The amount of time, in seconds, until a newly launched instance can contribute to the Amazon CloudWatch metrics. This delay lets an instance finish initializing before Amazon EC2 Auto Scaling aggregates instance metrics, resulting in more reliable usage data. Set this value equal to the amount of time that it takes for resource consumption to become stable after an instance reaches the InService state. To optimize the performance of scaling policies that scale continuously, such as target tracking and step scaling policies, we strongly recommend that you enable the default instance warmup, even if its value is set to 0 seconds Default instance warmup will not be added if no value is specified Default: None
        :param desired_capacity: Initial amount of instances in the fleet. If this is set to a number, every deployment will reset the amount of instances to this number. It is recommended to leave this value blank. Default: minCapacity, and leave unchanged during deployment
        :param group_metrics: Enable monitoring for group metrics, these metrics describe the group rather than any of its instances. To report all group metrics use ``GroupMetrics.all()`` Group metrics are reported in a granularity of 1 minute at no additional charge. Default: - no group metrics will be reported
        :param health_check: (deprecated) Configuration for health checks. Default: - HealthCheck.ec2 with no grace period
        :param health_checks: Configuration for EC2 or additional health checks. Even when using ``HealthChecks.withAdditionalChecks()``, the EC2 type is implicitly included. Default: - EC2 type with no grace period
        :param ignore_unmodified_size_properties: If the ASG has scheduled actions, don't reset unchanged group sizes. Only used if the ASG has scheduled actions (which may scale your ASG up or down regardless of cdk deployments). If true, the size of the group will only be reset if it has been changed in the CDK app. If false, the sizes will always be changed back to what they were in the CDK app on deployment. Default: true
        :param instance_monitoring: Controls whether instances in this group are launched with detailed or basic monitoring. When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Monitoring.DETAILED
        :param key_name: (deprecated) Name of SSH keypair to grant access to instances. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified You can either specify ``keyPair`` or ``keyName``, not both. Default: - No SSH access will be possible.
        :param key_pair: The SSH keypair to grant access to the instance. Feature flag ``AUTOSCALING_GENERATE_LAUNCH_TEMPLATE`` must be enabled to use this property. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified. You can either specify ``keyPair`` or ``keyName``, not both. Default: - No SSH access will be possible.
        :param max_capacity: Maximum number of instances in the fleet. Default: desiredCapacity
        :param max_instance_lifetime: The maximum amount of time that an instance can be in service. The maximum duration applies to all current and future instances in the group. As an instance approaches its maximum duration, it is terminated and replaced, and cannot be used again. You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value, leave this property undefined. Default: none
        :param min_capacity: Minimum number of instances in the fleet. Default: 1
        :param new_instances_protected_from_scale_in: Whether newly-launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in. By default, Auto Scaling can terminate an instance at any time after launch when scaling in an Auto Scaling Group, subject to the group's termination policy. However, you may wish to protect newly-launched instances from being scaled in if they are going to run critical applications that should not be prematurely terminated. This flag must be enabled if the Auto Scaling Group will be associated with an ECS Capacity Provider with managed termination protection. Default: false
        :param notifications: Configure autoscaling group to send notifications about fleet changes to an SNS topic(s). Default: - No fleet change notifications will be sent.
        :param signals: Configure waiting for signals during deployment. Use this to pause the CloudFormation deployment to wait for the instances in the AutoScalingGroup to report successful startup during creation and updates. The UserData script needs to invoke ``cfn-signal`` with a success or failure code after it is done setting up the instance. Without waiting for signals, the CloudFormation deployment will proceed as soon as the AutoScalingGroup has been created or updated but before the instances in the group have been started. For example, to have instances wait for an Elastic Load Balancing health check before they signal success, add a health-check verification by using the cfn-init helper script. For an example, see the verify_instance_health command in the Auto Scaling rolling updates sample template: https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/services/AutoScaling/AutoScalingRollingUpdates.yaml Default: - Do not wait for signals
        :param spot_price: The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request. Spot Instances are launched when the price you specify exceeds the current Spot market price. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: none
        :param ssm_session_permissions: Add SSM session permissions to the instance role. Setting this to ``true`` adds the necessary permissions to connect to the instance using SSM Session Manager. You can do this from the AWS Console. NOTE: Setting this flag to ``true`` may not be enough by itself. You must also use an AMI that comes with the SSM Agent, or install the SSM Agent yourself. See `Working with SSM Agent <https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-agent.html>`_ in the SSM Developer Guide. Default: false
        :param termination_policies: A policy or a list of policies that are used to select the instances to terminate. The policies are executed in the order that you list them. Default: - ``TerminationPolicy.DEFAULT``
        :param termination_policy_custom_lambda_function_arn: A lambda function Arn that can be used as a custom termination policy to select the instances to terminate. This property must be specified if the TerminationPolicy.CUSTOM_LAMBDA_FUNCTION is used. Default: - No lambda function Arn will be supplied
        :param update_policy: What to do when an AutoScalingGroup's instance configuration is changed. This is applied when any of the settings on the ASG are changed that affect how the instances should be created (VPC, instance type, startup scripts, etc.). It indicates how the existing instances should be replaced with new instances matching the new config. By default, nothing is done and only new instances are launched with the new config. Default: - ``UpdatePolicy.rollingUpdate()`` if using ``init``, ``UpdatePolicy.none()`` otherwise
        :param vpc_subnets: Where to place instances within the VPC. Default: - All Private subnets.
        :param instance_type: Instance type of the instances to start.
        :param bootstrap_enabled: Configures the EC2 user-data script for instances in this autoscaling group to bootstrap the node (invoke ``/etc/eks/bootstrap.sh``) and associate it with the EKS cluster. If you wish to provide a custom user data script, set this to ``false`` and manually invoke ``autoscalingGroup.addUserData()``. Default: true
        :param bootstrap_options: EKS node bootstrapping options. Default: - none
        :param machine_image_type: Machine image type. Default: MachineImageType.AMAZON_LINUX_2
        :param map_role: Will automatically update the aws-auth ConfigMap to map the IAM instance role to RBAC. This cannot be explicitly set to ``true`` if the cluster has kubectl disabled. Default: - true if the cluster has kubectl enabled (which is the default).
        :param spot_interrupt_handler: Installs the AWS spot instance interrupt handler on the cluster if it's not already added. Only relevant if ``spotPrice`` is used. Default: true

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            cluster.add_auto_scaling_group_capacity("BottlerocketNodes",
                instance_type=ec2.InstanceType("t3.small"),
                min_capacity=2,
                machine_image_type=eks.MachineImageType.BOTTLEROCKET
            )
        '''
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _SubnetSelection_e57d76df(**vpc_subnets)
        if isinstance(bootstrap_options, dict):
            bootstrap_options = BootstrapOptions(**bootstrap_options)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ac94eb5cd9569dcf4122cf20026c6f104b737f68ecd3395b237320bd8862be7)
            check_type(argname="argument allow_all_outbound", value=allow_all_outbound, expected_type=type_hints["allow_all_outbound"])
            check_type(argname="argument associate_public_ip_address", value=associate_public_ip_address, expected_type=type_hints["associate_public_ip_address"])
            check_type(argname="argument auto_scaling_group_name", value=auto_scaling_group_name, expected_type=type_hints["auto_scaling_group_name"])
            check_type(argname="argument az_capacity_distribution_strategy", value=az_capacity_distribution_strategy, expected_type=type_hints["az_capacity_distribution_strategy"])
            check_type(argname="argument block_devices", value=block_devices, expected_type=type_hints["block_devices"])
            check_type(argname="argument capacity_rebalance", value=capacity_rebalance, expected_type=type_hints["capacity_rebalance"])
            check_type(argname="argument cooldown", value=cooldown, expected_type=type_hints["cooldown"])
            check_type(argname="argument default_instance_warmup", value=default_instance_warmup, expected_type=type_hints["default_instance_warmup"])
            check_type(argname="argument desired_capacity", value=desired_capacity, expected_type=type_hints["desired_capacity"])
            check_type(argname="argument group_metrics", value=group_metrics, expected_type=type_hints["group_metrics"])
            check_type(argname="argument health_check", value=health_check, expected_type=type_hints["health_check"])
            check_type(argname="argument health_checks", value=health_checks, expected_type=type_hints["health_checks"])
            check_type(argname="argument ignore_unmodified_size_properties", value=ignore_unmodified_size_properties, expected_type=type_hints["ignore_unmodified_size_properties"])
            check_type(argname="argument instance_monitoring", value=instance_monitoring, expected_type=type_hints["instance_monitoring"])
            check_type(argname="argument key_name", value=key_name, expected_type=type_hints["key_name"])
            check_type(argname="argument key_pair", value=key_pair, expected_type=type_hints["key_pair"])
            check_type(argname="argument max_capacity", value=max_capacity, expected_type=type_hints["max_capacity"])
            check_type(argname="argument max_instance_lifetime", value=max_instance_lifetime, expected_type=type_hints["max_instance_lifetime"])
            check_type(argname="argument min_capacity", value=min_capacity, expected_type=type_hints["min_capacity"])
            check_type(argname="argument new_instances_protected_from_scale_in", value=new_instances_protected_from_scale_in, expected_type=type_hints["new_instances_protected_from_scale_in"])
            check_type(argname="argument notifications", value=notifications, expected_type=type_hints["notifications"])
            check_type(argname="argument signals", value=signals, expected_type=type_hints["signals"])
            check_type(argname="argument spot_price", value=spot_price, expected_type=type_hints["spot_price"])
            check_type(argname="argument ssm_session_permissions", value=ssm_session_permissions, expected_type=type_hints["ssm_session_permissions"])
            check_type(argname="argument termination_policies", value=termination_policies, expected_type=type_hints["termination_policies"])
            check_type(argname="argument termination_policy_custom_lambda_function_arn", value=termination_policy_custom_lambda_function_arn, expected_type=type_hints["termination_policy_custom_lambda_function_arn"])
            check_type(argname="argument update_policy", value=update_policy, expected_type=type_hints["update_policy"])
            check_type(argname="argument vpc_subnets", value=vpc_subnets, expected_type=type_hints["vpc_subnets"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument bootstrap_enabled", value=bootstrap_enabled, expected_type=type_hints["bootstrap_enabled"])
            check_type(argname="argument bootstrap_options", value=bootstrap_options, expected_type=type_hints["bootstrap_options"])
            check_type(argname="argument machine_image_type", value=machine_image_type, expected_type=type_hints["machine_image_type"])
            check_type(argname="argument map_role", value=map_role, expected_type=type_hints["map_role"])
            check_type(argname="argument spot_interrupt_handler", value=spot_interrupt_handler, expected_type=type_hints["spot_interrupt_handler"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "instance_type": instance_type,
        }
        if allow_all_outbound is not None:
            self._values["allow_all_outbound"] = allow_all_outbound
        if associate_public_ip_address is not None:
            self._values["associate_public_ip_address"] = associate_public_ip_address
        if auto_scaling_group_name is not None:
            self._values["auto_scaling_group_name"] = auto_scaling_group_name
        if az_capacity_distribution_strategy is not None:
            self._values["az_capacity_distribution_strategy"] = az_capacity_distribution_strategy
        if block_devices is not None:
            self._values["block_devices"] = block_devices
        if capacity_rebalance is not None:
            self._values["capacity_rebalance"] = capacity_rebalance
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if default_instance_warmup is not None:
            self._values["default_instance_warmup"] = default_instance_warmup
        if desired_capacity is not None:
            self._values["desired_capacity"] = desired_capacity
        if group_metrics is not None:
            self._values["group_metrics"] = group_metrics
        if health_check is not None:
            self._values["health_check"] = health_check
        if health_checks is not None:
            self._values["health_checks"] = health_checks
        if ignore_unmodified_size_properties is not None:
            self._values["ignore_unmodified_size_properties"] = ignore_unmodified_size_properties
        if instance_monitoring is not None:
            self._values["instance_monitoring"] = instance_monitoring
        if key_name is not None:
            self._values["key_name"] = key_name
        if key_pair is not None:
            self._values["key_pair"] = key_pair
        if max_capacity is not None:
            self._values["max_capacity"] = max_capacity
        if max_instance_lifetime is not None:
            self._values["max_instance_lifetime"] = max_instance_lifetime
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity
        if new_instances_protected_from_scale_in is not None:
            self._values["new_instances_protected_from_scale_in"] = new_instances_protected_from_scale_in
        if notifications is not None:
            self._values["notifications"] = notifications
        if signals is not None:
            self._values["signals"] = signals
        if spot_price is not None:
            self._values["spot_price"] = spot_price
        if ssm_session_permissions is not None:
            self._values["ssm_session_permissions"] = ssm_session_permissions
        if termination_policies is not None:
            self._values["termination_policies"] = termination_policies
        if termination_policy_custom_lambda_function_arn is not None:
            self._values["termination_policy_custom_lambda_function_arn"] = termination_policy_custom_lambda_function_arn
        if update_policy is not None:
            self._values["update_policy"] = update_policy
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets
        if bootstrap_enabled is not None:
            self._values["bootstrap_enabled"] = bootstrap_enabled
        if bootstrap_options is not None:
            self._values["bootstrap_options"] = bootstrap_options
        if machine_image_type is not None:
            self._values["machine_image_type"] = machine_image_type
        if map_role is not None:
            self._values["map_role"] = map_role
        if spot_interrupt_handler is not None:
            self._values["spot_interrupt_handler"] = spot_interrupt_handler

    @builtins.property
    def allow_all_outbound(self) -> typing.Optional[builtins.bool]:
        '''Whether the instances can initiate connections to anywhere by default.

        :default: true
        '''
        result = self._values.get("allow_all_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def associate_public_ip_address(self) -> typing.Optional[builtins.bool]:
        '''Whether instances in the Auto Scaling Group should have public IP addresses associated with them.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: - Use subnet setting.
        '''
        result = self._values.get("associate_public_ip_address")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def auto_scaling_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Auto Scaling group.

        This name must be unique per Region per account.

        :default: - Auto generated by CloudFormation
        '''
        result = self._values.get("auto_scaling_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def az_capacity_distribution_strategy(
        self,
    ) -> typing.Optional[_CapacityDistributionStrategy_2393ccfe]:
        '''The strategy for distributing instances across Availability Zones.

        :default: None
        '''
        result = self._values.get("az_capacity_distribution_strategy")
        return typing.cast(typing.Optional[_CapacityDistributionStrategy_2393ccfe], result)

    @builtins.property
    def block_devices(self) -> typing.Optional[typing.List[_BlockDevice_0cfc0568]]:
        '''Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.

        Each instance that is launched has an associated root device volume,
        either an Amazon EBS volume or an instance store volume.
        You can use block device mappings to specify additional EBS volumes or
        instance store volumes to attach to an instance when it is launched.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: - Uses the block device mapping of the AMI

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html
        '''
        result = self._values.get("block_devices")
        return typing.cast(typing.Optional[typing.List[_BlockDevice_0cfc0568]], result)

    @builtins.property
    def capacity_rebalance(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether Capacity Rebalancing is enabled.

        When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling
        attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of
        interruption. After launching a new instance, it then terminates an old instance.

        :default: false

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-capacityrebalance
        '''
        result = self._values.get("capacity_rebalance")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Default scaling cooldown for this AutoScalingGroup.

        :default: Duration.minutes(5)
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def default_instance_warmup(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The amount of time, in seconds, until a newly launched instance can contribute to the Amazon CloudWatch metrics.

        This delay lets an instance finish initializing before Amazon EC2 Auto Scaling aggregates instance metrics,
        resulting in more reliable usage data. Set this value equal to the amount of time that it takes for resource
        consumption to become stable after an instance reaches the InService state.

        To optimize the performance of scaling policies that scale continuously, such as target tracking and
        step scaling policies, we strongly recommend that you enable the default instance warmup, even if its value is set to 0 seconds

        Default instance warmup will not be added if no value is specified

        :default: None

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-default-instance-warmup.html
        '''
        result = self._values.get("default_instance_warmup")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def desired_capacity(self) -> typing.Optional[jsii.Number]:
        '''Initial amount of instances in the fleet.

        If this is set to a number, every deployment will reset the amount of
        instances to this number. It is recommended to leave this value blank.

        :default: minCapacity, and leave unchanged during deployment

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-desiredcapacity
        '''
        result = self._values.get("desired_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def group_metrics(self) -> typing.Optional[typing.List[_GroupMetrics_7cdf729b]]:
        '''Enable monitoring for group metrics, these metrics describe the group rather than any of its instances.

        To report all group metrics use ``GroupMetrics.all()``
        Group metrics are reported in a granularity of 1 minute at no additional charge.

        :default: - no group metrics will be reported
        '''
        result = self._values.get("group_metrics")
        return typing.cast(typing.Optional[typing.List[_GroupMetrics_7cdf729b]], result)

    @builtins.property
    def health_check(self) -> typing.Optional[_HealthCheck_03a4bd5a]:
        '''(deprecated) Configuration for health checks.

        :default: - HealthCheck.ec2 with no grace period

        :deprecated: Use ``healthChecks`` instead

        :stability: deprecated
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional[_HealthCheck_03a4bd5a], result)

    @builtins.property
    def health_checks(self) -> typing.Optional[_HealthChecks_b8757873]:
        '''Configuration for EC2 or additional health checks.

        Even when using ``HealthChecks.withAdditionalChecks()``, the EC2 type is implicitly included.

        :default: - EC2 type with no grace period

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/ec2-auto-scaling-health-checks.html
        '''
        result = self._values.get("health_checks")
        return typing.cast(typing.Optional[_HealthChecks_b8757873], result)

    @builtins.property
    def ignore_unmodified_size_properties(self) -> typing.Optional[builtins.bool]:
        '''If the ASG has scheduled actions, don't reset unchanged group sizes.

        Only used if the ASG has scheduled actions (which may scale your ASG up
        or down regardless of cdk deployments). If true, the size of the group
        will only be reset if it has been changed in the CDK app. If false, the
        sizes will always be changed back to what they were in the CDK app
        on deployment.

        :default: true
        '''
        result = self._values.get("ignore_unmodified_size_properties")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def instance_monitoring(self) -> typing.Optional[_Monitoring_50020f91]:
        '''Controls whether instances in this group are launched with detailed or basic monitoring.

        When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account
        is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: - Monitoring.DETAILED

        :see: https://docs.aws.amazon.com/autoscaling/latest/userguide/as-instance-monitoring.html#enable-as-instance-metrics
        '''
        result = self._values.get("instance_monitoring")
        return typing.cast(typing.Optional[_Monitoring_50020f91], result)

    @builtins.property
    def key_name(self) -> typing.Optional[builtins.str]:
        '''(deprecated) Name of SSH keypair to grant access to instances.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        You can either specify ``keyPair`` or ``keyName``, not both.

        :default: - No SSH access will be possible.

        :deprecated: - Use ``keyPair`` instead - https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2-readme.html#using-an-existing-ec2-key-pair

        :stability: deprecated
        '''
        result = self._values.get("key_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def key_pair(self) -> typing.Optional[_IKeyPair_bc344eda]:
        '''The SSH keypair to grant access to the instance.

        Feature flag ``AUTOSCALING_GENERATE_LAUNCH_TEMPLATE`` must be enabled to use this property.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified.

        You can either specify ``keyPair`` or ``keyName``, not both.

        :default: - No SSH access will be possible.
        '''
        result = self._values.get("key_pair")
        return typing.cast(typing.Optional[_IKeyPair_bc344eda], result)

    @builtins.property
    def max_capacity(self) -> typing.Optional[jsii.Number]:
        '''Maximum number of instances in the fleet.

        :default: desiredCapacity
        '''
        result = self._values.get("max_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_instance_lifetime(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum amount of time that an instance can be in service.

        The maximum duration applies
        to all current and future instances in the group. As an instance approaches its maximum duration,
        it is terminated and replaced, and cannot be used again.

        You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value,
        leave this property undefined.

        :default: none

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html
        '''
        result = self._values.get("max_instance_lifetime")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def min_capacity(self) -> typing.Optional[jsii.Number]:
        '''Minimum number of instances in the fleet.

        :default: 1
        '''
        result = self._values.get("min_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def new_instances_protected_from_scale_in(self) -> typing.Optional[builtins.bool]:
        '''Whether newly-launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in.

        By default, Auto Scaling can terminate an instance at any time after launch
        when scaling in an Auto Scaling Group, subject to the group's termination
        policy. However, you may wish to protect newly-launched instances from
        being scaled in if they are going to run critical applications that should
        not be prematurely terminated.

        This flag must be enabled if the Auto Scaling Group will be associated with
        an ECS Capacity Provider with managed termination protection.

        :default: false
        '''
        result = self._values.get("new_instances_protected_from_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def notifications(
        self,
    ) -> typing.Optional[typing.List[_NotificationConfiguration_d5911670]]:
        '''Configure autoscaling group to send notifications about fleet changes to an SNS topic(s).

        :default: - No fleet change notifications will be sent.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-notificationconfigurations
        '''
        result = self._values.get("notifications")
        return typing.cast(typing.Optional[typing.List[_NotificationConfiguration_d5911670]], result)

    @builtins.property
    def signals(self) -> typing.Optional[_Signals_69fbeb6e]:
        '''Configure waiting for signals during deployment.

        Use this to pause the CloudFormation deployment to wait for the instances
        in the AutoScalingGroup to report successful startup during
        creation and updates. The UserData script needs to invoke ``cfn-signal``
        with a success or failure code after it is done setting up the instance.

        Without waiting for signals, the CloudFormation deployment will proceed as
        soon as the AutoScalingGroup has been created or updated but before the
        instances in the group have been started.

        For example, to have instances wait for an Elastic Load Balancing health check before
        they signal success, add a health-check verification by using the
        cfn-init helper script. For an example, see the verify_instance_health
        command in the Auto Scaling rolling updates sample template:

        https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/services/AutoScaling/AutoScalingRollingUpdates.yaml

        :default: - Do not wait for signals
        '''
        result = self._values.get("signals")
        return typing.cast(typing.Optional[_Signals_69fbeb6e], result)

    @builtins.property
    def spot_price(self) -> typing.Optional[builtins.str]:
        '''The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request.

        Spot Instances are
        launched when the price you specify exceeds the current Spot market price.

        ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified

        :default: none
        '''
        result = self._values.get("spot_price")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ssm_session_permissions(self) -> typing.Optional[builtins.bool]:
        '''Add SSM session permissions to the instance role.

        Setting this to ``true`` adds the necessary permissions to connect
        to the instance using SSM Session Manager. You can do this
        from the AWS Console.

        NOTE: Setting this flag to ``true`` may not be enough by itself.
        You must also use an AMI that comes with the SSM Agent, or install
        the SSM Agent yourself. See
        `Working with SSM Agent <https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-agent.html>`_
        in the SSM Developer Guide.

        :default: false
        '''
        result = self._values.get("ssm_session_permissions")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def termination_policies(
        self,
    ) -> typing.Optional[typing.List[_TerminationPolicy_89633c56]]:
        '''A policy or a list of policies that are used to select the instances to terminate.

        The policies are executed in the order that you list them.

        :default: - ``TerminationPolicy.DEFAULT``

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html
        '''
        result = self._values.get("termination_policies")
        return typing.cast(typing.Optional[typing.List[_TerminationPolicy_89633c56]], result)

    @builtins.property
    def termination_policy_custom_lambda_function_arn(
        self,
    ) -> typing.Optional[builtins.str]:
        '''A lambda function Arn that can be used as a custom termination policy to select the instances to terminate.

        This property must be specified if the TerminationPolicy.CUSTOM_LAMBDA_FUNCTION
        is used.

        :default: - No lambda function Arn will be supplied

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/lambda-custom-termination-policy.html
        '''
        result = self._values.get("termination_policy_custom_lambda_function_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def update_policy(self) -> typing.Optional[_UpdatePolicy_6dffc7ca]:
        '''What to do when an AutoScalingGroup's instance configuration is changed.

        This is applied when any of the settings on the ASG are changed that
        affect how the instances should be created (VPC, instance type, startup
        scripts, etc.). It indicates how the existing instances should be
        replaced with new instances matching the new config. By default, nothing
        is done and only new instances are launched with the new config.

        :default: - ``UpdatePolicy.rollingUpdate()`` if using ``init``, ``UpdatePolicy.none()`` otherwise
        '''
        result = self._values.get("update_policy")
        return typing.cast(typing.Optional[_UpdatePolicy_6dffc7ca], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_SubnetSelection_e57d76df]:
        '''Where to place instances within the VPC.

        :default: - All Private subnets.
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_SubnetSelection_e57d76df], result)

    @builtins.property
    def instance_type(self) -> _InstanceType_f64915b9:
        '''Instance type of the instances to start.'''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(_InstanceType_f64915b9, result)

    @builtins.property
    def bootstrap_enabled(self) -> typing.Optional[builtins.bool]:
        '''Configures the EC2 user-data script for instances in this autoscaling group to bootstrap the node (invoke ``/etc/eks/bootstrap.sh``) and associate it with the EKS cluster.

        If you wish to provide a custom user data script, set this to ``false`` and
        manually invoke ``autoscalingGroup.addUserData()``.

        :default: true
        '''
        result = self._values.get("bootstrap_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def bootstrap_options(self) -> typing.Optional["BootstrapOptions"]:
        '''EKS node bootstrapping options.

        :default: - none
        '''
        result = self._values.get("bootstrap_options")
        return typing.cast(typing.Optional["BootstrapOptions"], result)

    @builtins.property
    def machine_image_type(self) -> typing.Optional["MachineImageType"]:
        '''Machine image type.

        :default: MachineImageType.AMAZON_LINUX_2
        '''
        result = self._values.get("machine_image_type")
        return typing.cast(typing.Optional["MachineImageType"], result)

    @builtins.property
    def map_role(self) -> typing.Optional[builtins.bool]:
        '''Will automatically update the aws-auth ConfigMap to map the IAM instance role to RBAC.

        This cannot be explicitly set to ``true`` if the cluster has kubectl disabled.

        :default: - true if the cluster has kubectl enabled (which is the default).
        '''
        result = self._values.get("map_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def spot_interrupt_handler(self) -> typing.Optional[builtins.bool]:
        '''Installs the AWS spot instance interrupt handler on the cluster if it's not already added.

        Only relevant if ``spotPrice`` is used.

        :default: true
        '''
        result = self._values.get("spot_interrupt_handler")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AutoScalingGroupCapacityOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AutoScalingGroupOptions",
    jsii_struct_bases=[],
    name_mapping={
        "bootstrap_enabled": "bootstrapEnabled",
        "bootstrap_options": "bootstrapOptions",
        "machine_image_type": "machineImageType",
        "map_role": "mapRole",
        "spot_interrupt_handler": "spotInterruptHandler",
    },
)
class AutoScalingGroupOptions:
    def __init__(
        self,
        *,
        bootstrap_enabled: typing.Optional[builtins.bool] = None,
        bootstrap_options: typing.Optional[typing.Union["BootstrapOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        map_role: typing.Optional[builtins.bool] = None,
        spot_interrupt_handler: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for adding an AutoScalingGroup as capacity.

        :param bootstrap_enabled: Configures the EC2 user-data script for instances in this autoscaling group to bootstrap the node (invoke ``/etc/eks/bootstrap.sh``) and associate it with the EKS cluster. If you wish to provide a custom user data script, set this to ``false`` and manually invoke ``autoscalingGroup.addUserData()``. Default: true
        :param bootstrap_options: Allows options for node bootstrapping through EC2 user data. Default: - default options
        :param machine_image_type: Allow options to specify different machine image type. Default: MachineImageType.AMAZON_LINUX_2
        :param map_role: Will automatically update the aws-auth ConfigMap to map the IAM instance role to RBAC. This cannot be explicitly set to ``true`` if the cluster has kubectl disabled. Default: - true if the cluster has kubectl enabled (which is the default).
        :param spot_interrupt_handler: Installs the AWS spot instance interrupt handler on the cluster if it's not already added. Only relevant if ``spotPrice`` is configured on the auto-scaling group. Default: true

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            # asg: autoscaling.AutoScalingGroup
            
            cluster.connect_auto_scaling_group_capacity(asg)
        '''
        if isinstance(bootstrap_options, dict):
            bootstrap_options = BootstrapOptions(**bootstrap_options)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3f231fb31d38ad5098df181b5a14255f260c8ad94579c736d49bc069dd611cdb)
            check_type(argname="argument bootstrap_enabled", value=bootstrap_enabled, expected_type=type_hints["bootstrap_enabled"])
            check_type(argname="argument bootstrap_options", value=bootstrap_options, expected_type=type_hints["bootstrap_options"])
            check_type(argname="argument machine_image_type", value=machine_image_type, expected_type=type_hints["machine_image_type"])
            check_type(argname="argument map_role", value=map_role, expected_type=type_hints["map_role"])
            check_type(argname="argument spot_interrupt_handler", value=spot_interrupt_handler, expected_type=type_hints["spot_interrupt_handler"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if bootstrap_enabled is not None:
            self._values["bootstrap_enabled"] = bootstrap_enabled
        if bootstrap_options is not None:
            self._values["bootstrap_options"] = bootstrap_options
        if machine_image_type is not None:
            self._values["machine_image_type"] = machine_image_type
        if map_role is not None:
            self._values["map_role"] = map_role
        if spot_interrupt_handler is not None:
            self._values["spot_interrupt_handler"] = spot_interrupt_handler

    @builtins.property
    def bootstrap_enabled(self) -> typing.Optional[builtins.bool]:
        '''Configures the EC2 user-data script for instances in this autoscaling group to bootstrap the node (invoke ``/etc/eks/bootstrap.sh``) and associate it with the EKS cluster.

        If you wish to provide a custom user data script, set this to ``false`` and
        manually invoke ``autoscalingGroup.addUserData()``.

        :default: true
        '''
        result = self._values.get("bootstrap_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def bootstrap_options(self) -> typing.Optional["BootstrapOptions"]:
        '''Allows options for node bootstrapping through EC2 user data.

        :default: - default options
        '''
        result = self._values.get("bootstrap_options")
        return typing.cast(typing.Optional["BootstrapOptions"], result)

    @builtins.property
    def machine_image_type(self) -> typing.Optional["MachineImageType"]:
        '''Allow options to specify different machine image type.

        :default: MachineImageType.AMAZON_LINUX_2
        '''
        result = self._values.get("machine_image_type")
        return typing.cast(typing.Optional["MachineImageType"], result)

    @builtins.property
    def map_role(self) -> typing.Optional[builtins.bool]:
        '''Will automatically update the aws-auth ConfigMap to map the IAM instance role to RBAC.

        This cannot be explicitly set to ``true`` if the cluster has kubectl disabled.

        :default: - true if the cluster has kubectl enabled (which is the default).
        '''
        result = self._values.get("map_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def spot_interrupt_handler(self) -> typing.Optional[builtins.bool]:
        '''Installs the AWS spot instance interrupt handler on the cluster if it's not already added.

        Only relevant if ``spotPrice`` is configured on the auto-scaling group.

        :default: true
        '''
        result = self._values.get("spot_interrupt_handler")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AutoScalingGroupOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AwsAuth(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.AwsAuth",
):
    '''Manages mapping between IAM users and roles to Kubernetes RBAC configuration.

    :see: https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        # cluster: eks.Cluster
        
        aws_auth = eks.AwsAuth(self, "MyAwsAuth",
            cluster=cluster
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: "Cluster",
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster: The EKS cluster to apply this configuration to. [disable-awslint:ref-via-interface]
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ccdf644e45d1cfe0e898914ed144f93474506ac792f3dccd00de4bd19fff264a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AwsAuthProps(cluster=cluster)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addAccount")
    def add_account(self, account_id: builtins.str) -> None:
        '''Additional AWS account to add to the aws-auth configmap.

        :param account_id: account number.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d79f752ab45aba315c94de1359997835319b87f0f34c50a60a039951a075fa92)
            check_type(argname="argument account_id", value=account_id, expected_type=type_hints["account_id"])
        return typing.cast(None, jsii.invoke(self, "addAccount", [account_id]))

    @jsii.member(jsii_name="addMastersRole")
    def add_masters_role(
        self,
        role: _IRole_235f5d8e,
        username: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Adds the specified IAM role to the ``system:masters`` RBAC group, which means that anyone that can assume it will be able to administer this Kubernetes system.

        :param role: The IAM role to add.
        :param username: Optional user (defaults to the role ARN).
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e032cff9510517b495b5e794499275fa8d73e98093fbfd95e1a6b56ed5b947f0)
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument username", value=username, expected_type=type_hints["username"])
        return typing.cast(None, jsii.invoke(self, "addMastersRole", [role, username]))

    @jsii.member(jsii_name="addRoleMapping")
    def add_role_mapping(
        self,
        role: _IRole_235f5d8e,
        *,
        groups: typing.Sequence[builtins.str],
        username: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Adds a mapping between an IAM role to a Kubernetes user and groups.

        :param role: The IAM role to map.
        :param groups: A list of groups within Kubernetes to which the role is mapped.
        :param username: The user name within Kubernetes to map to the IAM role. Default: - By default, the user name is the ARN of the IAM role.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__33d9f55ab6a57bdec369eabfd8196dc29cb774dd7757c2cf74a2137c01aa1149)
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
        mapping = AwsAuthMapping(groups=groups, username=username)

        return typing.cast(None, jsii.invoke(self, "addRoleMapping", [role, mapping]))

    @jsii.member(jsii_name="addUserMapping")
    def add_user_mapping(
        self,
        user: _IUser_c32311f7,
        *,
        groups: typing.Sequence[builtins.str],
        username: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Adds a mapping between an IAM user to a Kubernetes user and groups.

        :param user: The IAM user to map.
        :param groups: A list of groups within Kubernetes to which the role is mapped.
        :param username: The user name within Kubernetes to map to the IAM role. Default: - By default, the user name is the ARN of the IAM role.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0b64e8778c21e9b806e837c16293b8f92038b6c71b896bec7d07e7483f238c3d)
            check_type(argname="argument user", value=user, expected_type=type_hints["user"])
        mapping = AwsAuthMapping(groups=groups, username=username)

        return typing.cast(None, jsii.invoke(self, "addUserMapping", [user, mapping]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AwsAuthMapping",
    jsii_struct_bases=[],
    name_mapping={"groups": "groups", "username": "username"},
)
class AwsAuthMapping:
    def __init__(
        self,
        *,
        groups: typing.Sequence[builtins.str],
        username: typing.Optional[builtins.str] = None,
    ) -> None:
        '''AwsAuth mapping.

        :param groups: A list of groups within Kubernetes to which the role is mapped.
        :param username: The user name within Kubernetes to map to the IAM role. Default: - By default, the user name is the ARN of the IAM role.

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            admin_user = iam.User(self, "Admin")
            cluster.aws_auth.add_user_mapping(admin_user, groups=["system:masters"])
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d6587bb2a0cbd5dbbee00f28def3f2a4a7e5d3e7a8979d7c438ea88c01ba3685)
            check_type(argname="argument groups", value=groups, expected_type=type_hints["groups"])
            check_type(argname="argument username", value=username, expected_type=type_hints["username"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "groups": groups,
        }
        if username is not None:
            self._values["username"] = username

    @builtins.property
    def groups(self) -> typing.List[builtins.str]:
        '''A list of groups within Kubernetes to which the role is mapped.

        :see: https://kubernetes.io/docs/reference/access-authn-authz/rbac/#default-roles-and-role-bindings
        '''
        result = self._values.get("groups")
        assert result is not None, "Required property 'groups' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def username(self) -> typing.Optional[builtins.str]:
        '''The user name within Kubernetes to map to the IAM role.

        :default: - By default, the user name is the ARN of the IAM role.
        '''
        result = self._values.get("username")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsAuthMapping(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.AwsAuthProps",
    jsii_struct_bases=[],
    name_mapping={"cluster": "cluster"},
)
class AwsAuthProps:
    def __init__(self, *, cluster: "Cluster") -> None:
        '''Configuration props for the AwsAuth construct.

        :param cluster: The EKS cluster to apply this configuration to. [disable-awslint:ref-via-interface]

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            # cluster: eks.Cluster
            
            aws_auth_props = eks.AwsAuthProps(
                cluster=cluster
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b0039255e0d9fd09f738a75b17acb78d30b32450d91c427c942ca1e2e512e341)
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
        }

    @builtins.property
    def cluster(self) -> "Cluster":
        '''The EKS cluster to apply this configuration to.

        [disable-awslint:ref-via-interface]
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("Cluster", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsAuthProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.BootstrapOptions",
    jsii_struct_bases=[],
    name_mapping={
        "additional_args": "additionalArgs",
        "aws_api_retry_attempts": "awsApiRetryAttempts",
        "dns_cluster_ip": "dnsClusterIp",
        "docker_config_json": "dockerConfigJson",
        "enable_docker_bridge": "enableDockerBridge",
        "kubelet_extra_args": "kubeletExtraArgs",
        "use_max_pods": "useMaxPods",
    },
)
class BootstrapOptions:
    def __init__(
        self,
        *,
        additional_args: typing.Optional[builtins.str] = None,
        aws_api_retry_attempts: typing.Optional[jsii.Number] = None,
        dns_cluster_ip: typing.Optional[builtins.str] = None,
        docker_config_json: typing.Optional[builtins.str] = None,
        enable_docker_bridge: typing.Optional[builtins.bool] = None,
        kubelet_extra_args: typing.Optional[builtins.str] = None,
        use_max_pods: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''EKS node bootstrapping options.

        :param additional_args: Additional command line arguments to pass to the ``/etc/eks/bootstrap.sh`` command. Default: - none
        :param aws_api_retry_attempts: Number of retry attempts for AWS API call (DescribeCluster). Default: 3
        :param dns_cluster_ip: Overrides the IP address to use for DNS queries within the cluster. Default: - 10.100.0.10 or 172.20.0.10 based on the IP address of the primary interface.
        :param docker_config_json: The contents of the ``/etc/docker/daemon.json`` file. Useful if you want a custom config differing from the default one in the EKS AMI. Default: - none
        :param enable_docker_bridge: Restores the docker default bridge network. Default: false
        :param kubelet_extra_args: Extra arguments to add to the kubelet. Useful for adding labels or taints. For example, ``--node-labels foo=bar,goo=far``. Default: - none
        :param use_max_pods: Sets ``--max-pods`` for the kubelet based on the capacity of the EC2 instance. Default: true

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            cluster.add_auto_scaling_group_capacity("spot",
                instance_type=ec2.InstanceType("t3.large"),
                min_capacity=2,
                bootstrap_options=eks.BootstrapOptions(
                    kubelet_extra_args="--node-labels foo=bar,goo=far",
                    aws_api_retry_attempts=5
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__514fd8a17aeef0a56af459f9ba6654b4157e4327bc6ac0e214f66e0fdeca17b7)
            check_type(argname="argument additional_args", value=additional_args, expected_type=type_hints["additional_args"])
            check_type(argname="argument aws_api_retry_attempts", value=aws_api_retry_attempts, expected_type=type_hints["aws_api_retry_attempts"])
            check_type(argname="argument dns_cluster_ip", value=dns_cluster_ip, expected_type=type_hints["dns_cluster_ip"])
            check_type(argname="argument docker_config_json", value=docker_config_json, expected_type=type_hints["docker_config_json"])
            check_type(argname="argument enable_docker_bridge", value=enable_docker_bridge, expected_type=type_hints["enable_docker_bridge"])
            check_type(argname="argument kubelet_extra_args", value=kubelet_extra_args, expected_type=type_hints["kubelet_extra_args"])
            check_type(argname="argument use_max_pods", value=use_max_pods, expected_type=type_hints["use_max_pods"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if additional_args is not None:
            self._values["additional_args"] = additional_args
        if aws_api_retry_attempts is not None:
            self._values["aws_api_retry_attempts"] = aws_api_retry_attempts
        if dns_cluster_ip is not None:
            self._values["dns_cluster_ip"] = dns_cluster_ip
        if docker_config_json is not None:
            self._values["docker_config_json"] = docker_config_json
        if enable_docker_bridge is not None:
            self._values["enable_docker_bridge"] = enable_docker_bridge
        if kubelet_extra_args is not None:
            self._values["kubelet_extra_args"] = kubelet_extra_args
        if use_max_pods is not None:
            self._values["use_max_pods"] = use_max_pods

    @builtins.property
    def additional_args(self) -> typing.Optional[builtins.str]:
        '''Additional command line arguments to pass to the ``/etc/eks/bootstrap.sh`` command.

        :default: - none

        :see: https://github.com/awslabs/amazon-eks-ami/blob/master/files/bootstrap.sh
        '''
        result = self._values.get("additional_args")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def aws_api_retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''Number of retry attempts for AWS API call (DescribeCluster).

        :default: 3
        '''
        result = self._values.get("aws_api_retry_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def dns_cluster_ip(self) -> typing.Optional[builtins.str]:
        '''Overrides the IP address to use for DNS queries within the cluster.

        :default:

        - 10.100.0.10 or 172.20.0.10 based on the IP
        address of the primary interface.
        '''
        result = self._values.get("dns_cluster_ip")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def docker_config_json(self) -> typing.Optional[builtins.str]:
        '''The contents of the ``/etc/docker/daemon.json`` file. Useful if you want a custom config differing from the default one in the EKS AMI.

        :default: - none
        '''
        result = self._values.get("docker_config_json")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enable_docker_bridge(self) -> typing.Optional[builtins.bool]:
        '''Restores the docker default bridge network.

        :default: false
        '''
        result = self._values.get("enable_docker_bridge")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def kubelet_extra_args(self) -> typing.Optional[builtins.str]:
        '''Extra arguments to add to the kubelet. Useful for adding labels or taints.

        For example, ``--node-labels foo=bar,goo=far``.

        :default: - none
        '''
        result = self._values.get("kubelet_extra_args")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def use_max_pods(self) -> typing.Optional[builtins.bool]:
        '''Sets ``--max-pods`` for the kubelet based on the capacity of the EC2 instance.

        :default: true
        '''
        result = self._values.get("use_max_pods")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BootstrapOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.CapacityType")
class CapacityType(enum.Enum):
    '''Capacity type of the managed node group.

    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        cluster.add_nodegroup_capacity("extra-ng-spot",
            instance_types=[
                ec2.InstanceType("c5.large"),
                ec2.InstanceType("c5a.large"),
                ec2.InstanceType("c5d.large")
            ],
            min_size=3,
            capacity_type=eks.CapacityType.SPOT
        )
    '''

    SPOT = "SPOT"
    '''spot instances.'''
    ON_DEMAND = "ON_DEMAND"
    '''on-demand instances.'''
    CAPACITY_BLOCK = "CAPACITY_BLOCK"
    '''capacity block instances.'''


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnAccessEntry(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.CfnAccessEntry",
):
    '''Creates an access entry.

    An access entry allows an IAM principal to access your cluster. Access entries can replace the need to maintain entries in the ``aws-auth`` ``ConfigMap`` for authentication. You have the following options for authorizing an IAM principal to access Kubernetes objects on your cluster: Kubernetes role-based access control (RBAC), Amazon EKS, or both. Kubernetes RBAC authorization requires you to create and manage Kubernetes ``Role`` , ``ClusterRole`` , ``RoleBinding`` , and ``ClusterRoleBinding`` objects, in addition to managing access entries. If you use Amazon EKS authorization exclusively, you don't need to create and manage Kubernetes ``Role`` , ``ClusterRole`` , ``RoleBinding`` , and ``ClusterRoleBinding`` objects.

    For more information about access entries, see `Access entries <https://docs.aws.amazon.com/eks/latest/userguide/access-entries.html>`_ in the *Amazon EKS User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-accessentry.html
    :cloudformationResource: AWS::EKS::AccessEntry
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        cfn_access_entry = eks.CfnAccessEntry(self, "MyCfnAccessEntry",
            cluster_name="clusterName",
            principal_arn="principalArn",
        
            # the properties below are optional
            access_policies=[eks.CfnAccessEntry.AccessPolicyProperty(
                access_scope=eks.CfnAccessEntry.AccessScopeProperty(
                    type="type",
        
                    # the properties below are optional
                    namespaces=["namespaces"]
                ),
                policy_arn="policyArn"
            )],
            kubernetes_groups=["kubernetesGroups"],
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            type="type",
            username="username"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_name: builtins.str,
        principal_arn: builtins.str,
        access_policies: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAccessEntry.AccessPolicyProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        kubernetes_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        type: typing.Optional[builtins.str] = None,
        username: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param cluster_name: The name of your cluster.
        :param principal_arn: The ARN of the IAM principal for the ``AccessEntry`` . You can specify one ARN for each access entry. You can't specify the same ARN in more than one access entry. This value can't be changed after access entry creation. The valid principals differ depending on the type of the access entry in the ``type`` field. For ``STANDARD`` access entries, you can use every IAM principal type. For nodes ( ``EC2`` (for EKS Auto Mode), ``EC2_LINUX`` , ``EC2_WINDOWS`` , ``FARGATE_LINUX`` , and ``HYBRID_LINUX`` ), the only valid ARN is IAM roles. You can't use the STS session principal type with access entries because this is a temporary principal for each session and not a permanent identity that can be assigned permissions. `IAM best practices <https://docs.aws.amazon.com/IAM/latest/UserGuide/best-practices.html#bp-users-federation-idp>`_ recommend using IAM roles with temporary credentials, rather than IAM users with long-term credentials.
        :param access_policies: The access policies to associate to the access entry.
        :param kubernetes_groups: The value for ``name`` that you've specified for ``kind: Group`` as a ``subject`` in a Kubernetes ``RoleBinding`` or ``ClusterRoleBinding`` object. Amazon EKS doesn't confirm that the value for ``name`` exists in any bindings on your cluster. You can specify one or more names. Kubernetes authorizes the ``principalArn`` of the access entry to access any cluster objects that you've specified in a Kubernetes ``Role`` or ``ClusterRole`` object that is also specified in a binding's ``roleRef`` . For more information about creating Kubernetes ``RoleBinding`` , ``ClusterRoleBinding`` , ``Role`` , or ``ClusterRole`` objects, see `Using RBAC Authorization in the Kubernetes documentation <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/access-authn-authz/rbac/>`_ . If you want Amazon EKS to authorize the ``principalArn`` (instead of, or in addition to Kubernetes authorizing the ``principalArn`` ), you can associate one or more access policies to the access entry using ``AssociateAccessPolicy`` . If you associate any access policies, the ``principalARN`` has all permissions assigned in the associated access policies and all permissions in any Kubernetes ``Role`` or ``ClusterRole`` objects that the group names are bound to.
        :param tags: Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.
        :param type: The type of the new access entry. Valid values are ``STANDARD`` , ``FARGATE_LINUX`` , ``EC2_LINUX`` , ``EC2_WINDOWS`` , ``EC2`` (for EKS Auto Mode), ``HYBRID_LINUX`` , and ``HYPERPOD_LINUX`` . If the ``principalArn`` is for an IAM role that's used for self-managed Amazon EC2 nodes, specify ``EC2_LINUX`` or ``EC2_WINDOWS`` . Amazon EKS grants the necessary permissions to the node for you. If the ``principalArn`` is for any other purpose, specify ``STANDARD`` . If you don't specify a value, Amazon EKS sets the value to ``STANDARD`` . If you have the access mode of the cluster set to ``API_AND_CONFIG_MAP`` , it's unnecessary to create access entries for IAM roles used with Fargate profiles or managed Amazon EC2 nodes, because Amazon EKS creates entries in the ``aws-auth`` ``ConfigMap`` for the roles. You can't change this value once you've created the access entry. If you set the value to ``EC2_LINUX`` or ``EC2_WINDOWS`` , you can't specify values for ``kubernetesGroups`` , or associate an ``AccessPolicy`` to the access entry.
        :param username: The username to authenticate to Kubernetes with. We recommend not specifying a username and letting Amazon EKS specify it for you. For more information about the value Amazon EKS specifies for you, or constraints before specifying your own username, see `Creating access entries <https://docs.aws.amazon.com/eks/latest/userguide/access-entries.html#creating-access-entries>`_ in the *Amazon EKS User Guide* .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a2f5b1ef69d6e68b0a3a090b2b57166d8868d0487e670242c9187309335baaeb)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAccessEntryProps(
            cluster_name=cluster_name,
            principal_arn=principal_arn,
            access_policies=access_policies,
            kubernetes_groups=kubernetes_groups,
            tags=tags,
            type=type,
            username=username,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0d2a19d3fcdc61c683ec8a1d3fbbec66a1f1ec722b4eb7cc1ecb1b5779af27e5)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a0c13d72a7a5316ef8a586966947202d07a63867b7ec76c9501f50c21e5ab89)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrAccessEntryArn")
    def attr_access_entry_arn(self) -> builtins.str:
        '''The ARN of the access entry.

        :cloudformationAttribute: AccessEntryArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAccessEntryArn"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''The name of your cluster.'''
        return typing.cast(builtins.str, jsii.get(self, "clusterName"))

    @cluster_name.setter
    def cluster_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f39f11be83c62cd1ef0b79bbec37cd76332b93db644a6f5f897a2fd1c6e4d15a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clusterName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="principalArn")
    def principal_arn(self) -> builtins.str:
        '''The ARN of the IAM principal for the ``AccessEntry`` .'''
        return typing.cast(builtins.str, jsii.get(self, "principalArn"))

    @principal_arn.setter
    def principal_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__53028d8402e60bf71e6a724645feb7096873c998ac8df5343a3472840aba20b3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "principalArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="accessPolicies")
    def access_policies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAccessEntry.AccessPolicyProperty"]]]]:
        '''The access policies to associate to the access entry.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAccessEntry.AccessPolicyProperty"]]]], jsii.get(self, "accessPolicies"))

    @access_policies.setter
    def access_policies(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAccessEntry.AccessPolicyProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__72525fb57b071712551a1a70513a19122fa4b07addc9565813f0c1ad6c531aa6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "accessPolicies", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="kubernetesGroups")
    def kubernetes_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The value for ``name`` that you've specified for ``kind: Group`` as a ``subject`` in a Kubernetes ``RoleBinding`` or ``ClusterRoleBinding`` object.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "kubernetesGroups"))

    @kubernetes_groups.setter
    def kubernetes_groups(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eeca928ef55890afb86c2a4b773e5b085ea41bbda1031d5514018ff03ea79670)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kubernetesGroups", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Metadata that assists with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tags"))

    @tags.setter
    def tags(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__343de9e1ac76cbf6a67d73d349089c21fca695294c67398109df2c6ad597bba3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> typing.Optional[builtins.str]:
        '''The type of the new access entry.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "type"))

    @type.setter
    def type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d75f74439880fd1ba12bf85f7549ef8ddce92bd3683d6d8b7818c245923ce357)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "type", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="username")
    def username(self) -> typing.Optional[builtins.str]:
        '''The username to authenticate to Kubernetes with.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "username"))

    @username.setter
    def username(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__de59b674f92cd0ad5088412238c3b9a96b647a258d8a75b03ffc744e43723761)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "username", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnAccessEntry.AccessPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"access_scope": "accessScope", "policy_arn": "policyArn"},
    )
    class AccessPolicyProperty:
        def __init__(
            self,
            *,
            access_scope: typing.Union[_IResolvable_da3f097b, typing.Union["CfnAccessEntry.AccessScopeProperty", typing.Dict[builtins.str, typing.Any]]],
            policy_arn: builtins.str,
        ) -> None:
            '''An access policy includes permissions that allow Amazon EKS to authorize an IAM principal to work with Kubernetes objects on your cluster.

            The policies are managed by Amazon EKS, but they're not IAM policies. You can't view the permissions in the policies using the API. The permissions for many of the policies are similar to the Kubernetes ``cluster-admin`` , ``admin`` , ``edit`` , and ``view`` cluster roles. For more information about these cluster roles, see `User-facing roles <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/access-authn-authz/rbac/#user-facing-roles>`_ in the Kubernetes documentation. To view the contents of the policies, see `Access policy permissions <https://docs.aws.amazon.com/eks/latest/userguide/access-policies.html#access-policy-permissions>`_ in the *Amazon EKS User Guide* .

            :param access_scope: The scope of an ``AccessPolicy`` that's associated to an ``AccessEntry`` .
            :param policy_arn: The ARN of the access policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-accessentry-accesspolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                access_policy_property = eks.CfnAccessEntry.AccessPolicyProperty(
                    access_scope=eks.CfnAccessEntry.AccessScopeProperty(
                        type="type",
                
                        # the properties below are optional
                        namespaces=["namespaces"]
                    ),
                    policy_arn="policyArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cd63c671a92f33564702cfd54129ecf86dbf6c06f9defe37cd7fdf97ce8d85cd)
                check_type(argname="argument access_scope", value=access_scope, expected_type=type_hints["access_scope"])
                check_type(argname="argument policy_arn", value=policy_arn, expected_type=type_hints["policy_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "access_scope": access_scope,
                "policy_arn": policy_arn,
            }

        @builtins.property
        def access_scope(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnAccessEntry.AccessScopeProperty"]:
            '''The scope of an ``AccessPolicy`` that's associated to an ``AccessEntry`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-accessentry-accesspolicy.html#cfn-eks-accessentry-accesspolicy-accessscope
            '''
            result = self._values.get("access_scope")
            assert result is not None, "Required property 'access_scope' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnAccessEntry.AccessScopeProperty"], result)

        @builtins.property
        def policy_arn(self) -> builtins.str:
            '''The ARN of the access policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-accessentry-accesspolicy.html#cfn-eks-accessentry-accesspolicy-policyarn
            '''
            result = self._values.get("policy_arn")
            assert result is not None, "Required property 'policy_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnAccessEntry.AccessScopeProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "namespaces": "namespaces"},
    )
    class AccessScopeProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            namespaces: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The scope of an ``AccessPolicy`` that's associated to an ``AccessEntry`` .

            :param type: The scope type of an access policy.
            :param namespaces: A Kubernetes ``namespace`` that an access policy is scoped to. A value is required if you specified ``namespace`` for ``Type`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-accessentry-accessscope.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                access_scope_property = eks.CfnAccessEntry.AccessScopeProperty(
                    type="type",
                
                    # the properties below are optional
                    namespaces=["namespaces"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__40ff081851bbe198376dafc0e508f336b124e3296c59c57a15bb9f68d4feb3b0)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument namespaces", value=namespaces, expected_type=type_hints["namespaces"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if namespaces is not None:
                self._values["namespaces"] = namespaces

        @builtins.property
        def type(self) -> builtins.str:
            '''The scope type of an access policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-accessentry-accessscope.html#cfn-eks-accessentry-accessscope-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def namespaces(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A Kubernetes ``namespace`` that an access policy is scoped to.

            A value is required if you specified ``namespace`` for ``Type`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-accessentry-accessscope.html#cfn-eks-accessentry-accessscope-namespaces
            '''
            result = self._values.get("namespaces")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessScopeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.CfnAccessEntryProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster_name": "clusterName",
        "principal_arn": "principalArn",
        "access_policies": "accessPolicies",
        "kubernetes_groups": "kubernetesGroups",
        "tags": "tags",
        "type": "type",
        "username": "username",
    },
)
class CfnAccessEntryProps:
    def __init__(
        self,
        *,
        cluster_name: builtins.str,
        principal_arn: builtins.str,
        access_policies: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAccessEntry.AccessPolicyProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        kubernetes_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        type: typing.Optional[builtins.str] = None,
        username: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnAccessEntry``.

        :param cluster_name: The name of your cluster.
        :param principal_arn: The ARN of the IAM principal for the ``AccessEntry`` . You can specify one ARN for each access entry. You can't specify the same ARN in more than one access entry. This value can't be changed after access entry creation. The valid principals differ depending on the type of the access entry in the ``type`` field. For ``STANDARD`` access entries, you can use every IAM principal type. For nodes ( ``EC2`` (for EKS Auto Mode), ``EC2_LINUX`` , ``EC2_WINDOWS`` , ``FARGATE_LINUX`` , and ``HYBRID_LINUX`` ), the only valid ARN is IAM roles. You can't use the STS session principal type with access entries because this is a temporary principal for each session and not a permanent identity that can be assigned permissions. `IAM best practices <https://docs.aws.amazon.com/IAM/latest/UserGuide/best-practices.html#bp-users-federation-idp>`_ recommend using IAM roles with temporary credentials, rather than IAM users with long-term credentials.
        :param access_policies: The access policies to associate to the access entry.
        :param kubernetes_groups: The value for ``name`` that you've specified for ``kind: Group`` as a ``subject`` in a Kubernetes ``RoleBinding`` or ``ClusterRoleBinding`` object. Amazon EKS doesn't confirm that the value for ``name`` exists in any bindings on your cluster. You can specify one or more names. Kubernetes authorizes the ``principalArn`` of the access entry to access any cluster objects that you've specified in a Kubernetes ``Role`` or ``ClusterRole`` object that is also specified in a binding's ``roleRef`` . For more information about creating Kubernetes ``RoleBinding`` , ``ClusterRoleBinding`` , ``Role`` , or ``ClusterRole`` objects, see `Using RBAC Authorization in the Kubernetes documentation <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/access-authn-authz/rbac/>`_ . If you want Amazon EKS to authorize the ``principalArn`` (instead of, or in addition to Kubernetes authorizing the ``principalArn`` ), you can associate one or more access policies to the access entry using ``AssociateAccessPolicy`` . If you associate any access policies, the ``principalARN`` has all permissions assigned in the associated access policies and all permissions in any Kubernetes ``Role`` or ``ClusterRole`` objects that the group names are bound to.
        :param tags: Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.
        :param type: The type of the new access entry. Valid values are ``STANDARD`` , ``FARGATE_LINUX`` , ``EC2_LINUX`` , ``EC2_WINDOWS`` , ``EC2`` (for EKS Auto Mode), ``HYBRID_LINUX`` , and ``HYPERPOD_LINUX`` . If the ``principalArn`` is for an IAM role that's used for self-managed Amazon EC2 nodes, specify ``EC2_LINUX`` or ``EC2_WINDOWS`` . Amazon EKS grants the necessary permissions to the node for you. If the ``principalArn`` is for any other purpose, specify ``STANDARD`` . If you don't specify a value, Amazon EKS sets the value to ``STANDARD`` . If you have the access mode of the cluster set to ``API_AND_CONFIG_MAP`` , it's unnecessary to create access entries for IAM roles used with Fargate profiles or managed Amazon EC2 nodes, because Amazon EKS creates entries in the ``aws-auth`` ``ConfigMap`` for the roles. You can't change this value once you've created the access entry. If you set the value to ``EC2_LINUX`` or ``EC2_WINDOWS`` , you can't specify values for ``kubernetesGroups`` , or associate an ``AccessPolicy`` to the access entry.
        :param username: The username to authenticate to Kubernetes with. We recommend not specifying a username and letting Amazon EKS specify it for you. For more information about the value Amazon EKS specifies for you, or constraints before specifying your own username, see `Creating access entries <https://docs.aws.amazon.com/eks/latest/userguide/access-entries.html#creating-access-entries>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-accessentry.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            cfn_access_entry_props = eks.CfnAccessEntryProps(
                cluster_name="clusterName",
                principal_arn="principalArn",
            
                # the properties below are optional
                access_policies=[eks.CfnAccessEntry.AccessPolicyProperty(
                    access_scope=eks.CfnAccessEntry.AccessScopeProperty(
                        type="type",
            
                        # the properties below are optional
                        namespaces=["namespaces"]
                    ),
                    policy_arn="policyArn"
                )],
                kubernetes_groups=["kubernetesGroups"],
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                type="type",
                username="username"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__de3a1caab0efd683002e5238c8a564c2c01affc00c6d18e8fceab9a9ef2ac1f2)
            check_type(argname="argument cluster_name", value=cluster_name, expected_type=type_hints["cluster_name"])
            check_type(argname="argument principal_arn", value=principal_arn, expected_type=type_hints["principal_arn"])
            check_type(argname="argument access_policies", value=access_policies, expected_type=type_hints["access_policies"])
            check_type(argname="argument kubernetes_groups", value=kubernetes_groups, expected_type=type_hints["kubernetes_groups"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument username", value=username, expected_type=type_hints["username"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_name": cluster_name,
            "principal_arn": principal_arn,
        }
        if access_policies is not None:
            self._values["access_policies"] = access_policies
        if kubernetes_groups is not None:
            self._values["kubernetes_groups"] = kubernetes_groups
        if tags is not None:
            self._values["tags"] = tags
        if type is not None:
            self._values["type"] = type
        if username is not None:
            self._values["username"] = username

    @builtins.property
    def cluster_name(self) -> builtins.str:
        '''The name of your cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-accessentry.html#cfn-eks-accessentry-clustername
        '''
        result = self._values.get("cluster_name")
        assert result is not None, "Required property 'cluster_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def principal_arn(self) -> builtins.str:
        '''The ARN of the IAM principal for the ``AccessEntry`` .

        You can specify one ARN for each access entry. You can't specify the same ARN in more than one access entry. This value can't be changed after access entry creation.

        The valid principals differ depending on the type of the access entry in the ``type`` field. For ``STANDARD`` access entries, you can use every IAM principal type. For nodes ( ``EC2`` (for EKS Auto Mode), ``EC2_LINUX`` , ``EC2_WINDOWS`` , ``FARGATE_LINUX`` , and ``HYBRID_LINUX`` ), the only valid ARN is IAM roles. You can't use the STS session principal type with access entries because this is a temporary principal for each session and not a permanent identity that can be assigned permissions.

        `IAM best practices <https://docs.aws.amazon.com/IAM/latest/UserGuide/best-practices.html#bp-users-federation-idp>`_ recommend using IAM roles with temporary credentials, rather than IAM users with long-term credentials.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-accessentry.html#cfn-eks-accessentry-principalarn
        '''
        result = self._values.get("principal_arn")
        assert result is not None, "Required property 'principal_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_policies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAccessEntry.AccessPolicyProperty]]]]:
        '''The access policies to associate to the access entry.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-accessentry.html#cfn-eks-accessentry-accesspolicies
        '''
        result = self._values.get("access_policies")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAccessEntry.AccessPolicyProperty]]]], result)

    @builtins.property
    def kubernetes_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The value for ``name`` that you've specified for ``kind: Group`` as a ``subject`` in a Kubernetes ``RoleBinding`` or ``ClusterRoleBinding`` object.

        Amazon EKS doesn't confirm that the value for ``name`` exists in any bindings on your cluster. You can specify one or more names.

        Kubernetes authorizes the ``principalArn`` of the access entry to access any cluster objects that you've specified in a Kubernetes ``Role`` or ``ClusterRole`` object that is also specified in a binding's ``roleRef`` . For more information about creating Kubernetes ``RoleBinding`` , ``ClusterRoleBinding`` , ``Role`` , or ``ClusterRole`` objects, see `Using RBAC Authorization in the Kubernetes documentation <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/access-authn-authz/rbac/>`_ .

        If you want Amazon EKS to authorize the ``principalArn`` (instead of, or in addition to Kubernetes authorizing the ``principalArn`` ), you can associate one or more access policies to the access entry using ``AssociateAccessPolicy`` . If you associate any access policies, the ``principalARN`` has all permissions assigned in the associated access policies and all permissions in any Kubernetes ``Role`` or ``ClusterRole`` objects that the group names are bound to.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-accessentry.html#cfn-eks-accessentry-kubernetesgroups
        '''
        result = self._values.get("kubernetes_groups")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Metadata that assists with categorization and organization.

        Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-accessentry.html#cfn-eks-accessentry-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def type(self) -> typing.Optional[builtins.str]:
        '''The type of the new access entry.

        Valid values are ``STANDARD`` , ``FARGATE_LINUX`` , ``EC2_LINUX`` , ``EC2_WINDOWS`` , ``EC2`` (for EKS Auto Mode), ``HYBRID_LINUX`` , and ``HYPERPOD_LINUX`` .

        If the ``principalArn`` is for an IAM role that's used for self-managed Amazon EC2 nodes, specify ``EC2_LINUX`` or ``EC2_WINDOWS`` . Amazon EKS grants the necessary permissions to the node for you. If the ``principalArn`` is for any other purpose, specify ``STANDARD`` . If you don't specify a value, Amazon EKS sets the value to ``STANDARD`` . If you have the access mode of the cluster set to ``API_AND_CONFIG_MAP`` , it's unnecessary to create access entries for IAM roles used with Fargate profiles or managed Amazon EC2 nodes, because Amazon EKS creates entries in the ``aws-auth`` ``ConfigMap`` for the roles. You can't change this value once you've created the access entry.

        If you set the value to ``EC2_LINUX`` or ``EC2_WINDOWS`` , you can't specify values for ``kubernetesGroups`` , or associate an ``AccessPolicy`` to the access entry.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-accessentry.html#cfn-eks-accessentry-type
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def username(self) -> typing.Optional[builtins.str]:
        '''The username to authenticate to Kubernetes with.

        We recommend not specifying a username and letting Amazon EKS specify it for you. For more information about the value Amazon EKS specifies for you, or constraints before specifying your own username, see `Creating access entries <https://docs.aws.amazon.com/eks/latest/userguide/access-entries.html#creating-access-entries>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-accessentry.html#cfn-eks-accessentry-username
        '''
        result = self._values.get("username")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAccessEntryProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnAddon(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.CfnAddon",
):
    '''Creates an Amazon EKS add-on.

    Amazon EKS add-ons help to automate the provisioning and lifecycle management of common operational software for Amazon EKS clusters. For more information, see `Amazon EKS add-ons <https://docs.aws.amazon.com/eks/latest/userguide/eks-add-ons.html>`_ in the *Amazon EKS User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-addon.html
    :cloudformationResource: AWS::EKS::Addon
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        cfn_addon = eks.CfnAddon(self, "MyCfnAddon",
            addon_name="addonName",
            cluster_name="clusterName",
        
            # the properties below are optional
            addon_version="addonVersion",
            configuration_values="configurationValues",
            pod_identity_associations=[eks.CfnAddon.PodIdentityAssociationProperty(
                role_arn="roleArn",
                service_account="serviceAccount"
            )],
            preserve_on_delete=False,
            resolve_conflicts="resolveConflicts",
            service_account_role_arn="serviceAccountRoleArn",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        addon_name: builtins.str,
        cluster_name: builtins.str,
        addon_version: typing.Optional[builtins.str] = None,
        configuration_values: typing.Optional[builtins.str] = None,
        pod_identity_associations: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAddon.PodIdentityAssociationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        preserve_on_delete: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        resolve_conflicts: typing.Optional[builtins.str] = None,
        service_account_role_arn: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param addon_name: The name of the add-on.
        :param cluster_name: The name of your cluster.
        :param addon_version: The version of the add-on.
        :param configuration_values: The configuration values that you provided.
        :param pod_identity_associations: An array of Pod Identity Assocations owned by the Addon. Each EKS Pod Identity association maps a role to a service account in a namespace in the cluster. For more information, see `Attach an IAM Role to an Amazon EKS add-on using Pod Identity <https://docs.aws.amazon.com/eks/latest/userguide/add-ons-iam.html>`_ in the *Amazon EKS User Guide* .
        :param preserve_on_delete: Specifying this option preserves the add-on software on your cluster but Amazon EKS stops managing any settings for the add-on. If an IAM account is associated with the add-on, it isn't removed.
        :param resolve_conflicts: How to resolve field value conflicts for an Amazon EKS add-on. Conflicts are handled based on the value you choose: - *None* – If the self-managed version of the add-on is installed on your cluster, Amazon EKS doesn't change the value. Creation of the add-on might fail. - *Overwrite* – If the self-managed version of the add-on is installed on your cluster and the Amazon EKS default value is different than the existing value, Amazon EKS changes the value to the Amazon EKS default value. - *Preserve* – This is similar to the NONE option. If the self-managed version of the add-on is installed on your cluster Amazon EKS doesn't change the add-on resource properties. Creation of the add-on might fail if conflicts are detected. This option works differently during the update operation. For more information, see ```UpdateAddon`` <https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html>`_ . If you don't currently have the self-managed version of the add-on installed on your cluster, the Amazon EKS add-on is installed. Amazon EKS sets all values to default values, regardless of the option that you specify.
        :param service_account_role_arn: The Amazon Resource Name (ARN) of an existing IAM role to bind to the add-on's service account. The role must be assigned the IAM permissions required by the add-on. If you don't specify an existing IAM role, then the add-on uses the permissions assigned to the node IAM role. For more information, see `Amazon EKS node IAM role <https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html>`_ in the *Amazon EKS User Guide* . .. epigraph:: To specify an existing IAM role, you must have an IAM OpenID Connect (OIDC) provider created for your cluster. For more information, see `Enabling IAM roles for service accounts on your cluster <https://docs.aws.amazon.com/eks/latest/userguide/enable-iam-roles-for-service-accounts.html>`_ in the *Amazon EKS User Guide* .
        :param tags: The metadata that you apply to the add-on to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Add-on tags do not propagate to any other resources associated with the cluster.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__45ff0728c7d6fc5f47c97aa791c327f70a32e19bdf463d94d9351053fc197fcb)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAddonProps(
            addon_name=addon_name,
            cluster_name=cluster_name,
            addon_version=addon_version,
            configuration_values=configuration_values,
            pod_identity_associations=pod_identity_associations,
            preserve_on_delete=preserve_on_delete,
            resolve_conflicts=resolve_conflicts,
            service_account_role_arn=service_account_role_arn,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fb634c71637a3029d784eb1cc23e12801a6d069d381448751935635b986d50bc)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1b1271b2ed8638cd1b5a3d17691c2d2954ae570c28642744175994be98826319)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The ARN of the add-on, such as ``arn:aws:eks:us-west-2:111122223333:addon/1-19/vpc-cni/xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx`` .

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="addonName")
    def addon_name(self) -> builtins.str:
        '''The name of the add-on.'''
        return typing.cast(builtins.str, jsii.get(self, "addonName"))

    @addon_name.setter
    def addon_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aa2f59cc2f15ceecb25c97a05918b5204a973e21e99d20f1a8d5ccaf93d479dc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "addonName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''The name of your cluster.'''
        return typing.cast(builtins.str, jsii.get(self, "clusterName"))

    @cluster_name.setter
    def cluster_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__32e5d7f68b583ad28c101848b4b9915ac4edfdb52f37771404157d760548a2f0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clusterName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="addonVersion")
    def addon_version(self) -> typing.Optional[builtins.str]:
        '''The version of the add-on.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "addonVersion"))

    @addon_version.setter
    def addon_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eb85a2b34531ebb86af42e5229b0ed17a08e49a0d033880c119269f998089248)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "addonVersion", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="configurationValues")
    def configuration_values(self) -> typing.Optional[builtins.str]:
        '''The configuration values that you provided.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "configurationValues"))

    @configuration_values.setter
    def configuration_values(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f2b158aed78a78d2962c2650df64f6c3880ccb508ebd6b281bda6c1a1961620a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "configurationValues", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="podIdentityAssociations")
    def pod_identity_associations(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAddon.PodIdentityAssociationProperty"]]]]:
        '''An array of Pod Identity Assocations owned by the Addon.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAddon.PodIdentityAssociationProperty"]]]], jsii.get(self, "podIdentityAssociations"))

    @pod_identity_associations.setter
    def pod_identity_associations(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAddon.PodIdentityAssociationProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__04a430658e28600fba10a8c3e5edab2978904829dda6f2c70e9cca8560f7e400)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "podIdentityAssociations", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="preserveOnDelete")
    def preserve_on_delete(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifying this option preserves the add-on software on your cluster but Amazon EKS stops managing any settings for the add-on.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "preserveOnDelete"))

    @preserve_on_delete.setter
    def preserve_on_delete(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c38bc0bc32707ba96b79f5dda73d99054ea5b77577796b10a0f95ff6fe51b133)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "preserveOnDelete", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="resolveConflicts")
    def resolve_conflicts(self) -> typing.Optional[builtins.str]:
        '''How to resolve field value conflicts for an Amazon EKS add-on.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "resolveConflicts"))

    @resolve_conflicts.setter
    def resolve_conflicts(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5ae5d99ca6a1d34e61263b73b34a1a3a9b8ec0abb558a77c4507f0d9ea472046)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "resolveConflicts", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="serviceAccountRoleArn")
    def service_account_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of an existing IAM role to bind to the add-on's service account.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "serviceAccountRoleArn"))

    @service_account_role_arn.setter
    def service_account_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__533ac32ef2e5bc8d5ce0313c18089cdf16b45b01e51842a29288fbe7ffec2431)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serviceAccountRoleArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The metadata that you apply to the add-on to assist with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__61cfcc2cd9aba81e02df7f2a5c976044dc5e5cbf6c05b880c4944cb357ebd775)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnAddon.PodIdentityAssociationProperty",
        jsii_struct_bases=[],
        name_mapping={"role_arn": "roleArn", "service_account": "serviceAccount"},
    )
    class PodIdentityAssociationProperty:
        def __init__(
            self,
            *,
            role_arn: builtins.str,
            service_account: builtins.str,
        ) -> None:
            '''Amazon EKS Pod Identity associations provide the ability to manage credentials for your applications, similar to the way that Amazon EC2 instance profiles provide credentials to Amazon EC2 instances.

            :param role_arn: The Amazon Resource Name (ARN) of the IAM role to associate with the service account. The EKS Pod Identity agent manages credentials to assume this role for applications in the containers in the pods that use this service account.
            :param service_account: The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-addon-podidentityassociation.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                pod_identity_association_property = eks.CfnAddon.PodIdentityAssociationProperty(
                    role_arn="roleArn",
                    service_account="serviceAccount"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3925c850dd0d0ad3b9faeea87aafbe69220a7bf33d95af5527715674625c9891)
                check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
                check_type(argname="argument service_account", value=service_account, expected_type=type_hints["service_account"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "role_arn": role_arn,
                "service_account": service_account,
            }

        @builtins.property
        def role_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the IAM role to associate with the service account.

            The EKS Pod Identity agent manages credentials to assume this role for applications in the containers in the pods that use this service account.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-addon-podidentityassociation.html#cfn-eks-addon-podidentityassociation-rolearn
            '''
            result = self._values.get("role_arn")
            assert result is not None, "Required property 'role_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def service_account(self) -> builtins.str:
            '''The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-addon-podidentityassociation.html#cfn-eks-addon-podidentityassociation-serviceaccount
            '''
            result = self._values.get("service_account")
            assert result is not None, "Required property 'service_account' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PodIdentityAssociationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.CfnAddonProps",
    jsii_struct_bases=[],
    name_mapping={
        "addon_name": "addonName",
        "cluster_name": "clusterName",
        "addon_version": "addonVersion",
        "configuration_values": "configurationValues",
        "pod_identity_associations": "podIdentityAssociations",
        "preserve_on_delete": "preserveOnDelete",
        "resolve_conflicts": "resolveConflicts",
        "service_account_role_arn": "serviceAccountRoleArn",
        "tags": "tags",
    },
)
class CfnAddonProps:
    def __init__(
        self,
        *,
        addon_name: builtins.str,
        cluster_name: builtins.str,
        addon_version: typing.Optional[builtins.str] = None,
        configuration_values: typing.Optional[builtins.str] = None,
        pod_identity_associations: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAddon.PodIdentityAssociationProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        preserve_on_delete: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        resolve_conflicts: typing.Optional[builtins.str] = None,
        service_account_role_arn: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnAddon``.

        :param addon_name: The name of the add-on.
        :param cluster_name: The name of your cluster.
        :param addon_version: The version of the add-on.
        :param configuration_values: The configuration values that you provided.
        :param pod_identity_associations: An array of Pod Identity Assocations owned by the Addon. Each EKS Pod Identity association maps a role to a service account in a namespace in the cluster. For more information, see `Attach an IAM Role to an Amazon EKS add-on using Pod Identity <https://docs.aws.amazon.com/eks/latest/userguide/add-ons-iam.html>`_ in the *Amazon EKS User Guide* .
        :param preserve_on_delete: Specifying this option preserves the add-on software on your cluster but Amazon EKS stops managing any settings for the add-on. If an IAM account is associated with the add-on, it isn't removed.
        :param resolve_conflicts: How to resolve field value conflicts for an Amazon EKS add-on. Conflicts are handled based on the value you choose: - *None* – If the self-managed version of the add-on is installed on your cluster, Amazon EKS doesn't change the value. Creation of the add-on might fail. - *Overwrite* – If the self-managed version of the add-on is installed on your cluster and the Amazon EKS default value is different than the existing value, Amazon EKS changes the value to the Amazon EKS default value. - *Preserve* – This is similar to the NONE option. If the self-managed version of the add-on is installed on your cluster Amazon EKS doesn't change the add-on resource properties. Creation of the add-on might fail if conflicts are detected. This option works differently during the update operation. For more information, see ```UpdateAddon`` <https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html>`_ . If you don't currently have the self-managed version of the add-on installed on your cluster, the Amazon EKS add-on is installed. Amazon EKS sets all values to default values, regardless of the option that you specify.
        :param service_account_role_arn: The Amazon Resource Name (ARN) of an existing IAM role to bind to the add-on's service account. The role must be assigned the IAM permissions required by the add-on. If you don't specify an existing IAM role, then the add-on uses the permissions assigned to the node IAM role. For more information, see `Amazon EKS node IAM role <https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html>`_ in the *Amazon EKS User Guide* . .. epigraph:: To specify an existing IAM role, you must have an IAM OpenID Connect (OIDC) provider created for your cluster. For more information, see `Enabling IAM roles for service accounts on your cluster <https://docs.aws.amazon.com/eks/latest/userguide/enable-iam-roles-for-service-accounts.html>`_ in the *Amazon EKS User Guide* .
        :param tags: The metadata that you apply to the add-on to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Add-on tags do not propagate to any other resources associated with the cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-addon.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            cfn_addon_props = eks.CfnAddonProps(
                addon_name="addonName",
                cluster_name="clusterName",
            
                # the properties below are optional
                addon_version="addonVersion",
                configuration_values="configurationValues",
                pod_identity_associations=[eks.CfnAddon.PodIdentityAssociationProperty(
                    role_arn="roleArn",
                    service_account="serviceAccount"
                )],
                preserve_on_delete=False,
                resolve_conflicts="resolveConflicts",
                service_account_role_arn="serviceAccountRoleArn",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__484b2779e40e4780cb0940ac7bc9daaf91fa04347613d732138d3be3d3f5a2d9)
            check_type(argname="argument addon_name", value=addon_name, expected_type=type_hints["addon_name"])
            check_type(argname="argument cluster_name", value=cluster_name, expected_type=type_hints["cluster_name"])
            check_type(argname="argument addon_version", value=addon_version, expected_type=type_hints["addon_version"])
            check_type(argname="argument configuration_values", value=configuration_values, expected_type=type_hints["configuration_values"])
            check_type(argname="argument pod_identity_associations", value=pod_identity_associations, expected_type=type_hints["pod_identity_associations"])
            check_type(argname="argument preserve_on_delete", value=preserve_on_delete, expected_type=type_hints["preserve_on_delete"])
            check_type(argname="argument resolve_conflicts", value=resolve_conflicts, expected_type=type_hints["resolve_conflicts"])
            check_type(argname="argument service_account_role_arn", value=service_account_role_arn, expected_type=type_hints["service_account_role_arn"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "addon_name": addon_name,
            "cluster_name": cluster_name,
        }
        if addon_version is not None:
            self._values["addon_version"] = addon_version
        if configuration_values is not None:
            self._values["configuration_values"] = configuration_values
        if pod_identity_associations is not None:
            self._values["pod_identity_associations"] = pod_identity_associations
        if preserve_on_delete is not None:
            self._values["preserve_on_delete"] = preserve_on_delete
        if resolve_conflicts is not None:
            self._values["resolve_conflicts"] = resolve_conflicts
        if service_account_role_arn is not None:
            self._values["service_account_role_arn"] = service_account_role_arn
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def addon_name(self) -> builtins.str:
        '''The name of the add-on.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-addon.html#cfn-eks-addon-addonname
        '''
        result = self._values.get("addon_name")
        assert result is not None, "Required property 'addon_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cluster_name(self) -> builtins.str:
        '''The name of your cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-addon.html#cfn-eks-addon-clustername
        '''
        result = self._values.get("cluster_name")
        assert result is not None, "Required property 'cluster_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def addon_version(self) -> typing.Optional[builtins.str]:
        '''The version of the add-on.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-addon.html#cfn-eks-addon-addonversion
        '''
        result = self._values.get("addon_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def configuration_values(self) -> typing.Optional[builtins.str]:
        '''The configuration values that you provided.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-addon.html#cfn-eks-addon-configurationvalues
        '''
        result = self._values.get("configuration_values")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def pod_identity_associations(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAddon.PodIdentityAssociationProperty]]]]:
        '''An array of Pod Identity Assocations owned by the Addon.

        Each EKS Pod Identity association maps a role to a service account in a namespace in the cluster.

        For more information, see `Attach an IAM Role to an Amazon EKS add-on using Pod Identity <https://docs.aws.amazon.com/eks/latest/userguide/add-ons-iam.html>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-addon.html#cfn-eks-addon-podidentityassociations
        '''
        result = self._values.get("pod_identity_associations")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAddon.PodIdentityAssociationProperty]]]], result)

    @builtins.property
    def preserve_on_delete(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifying this option preserves the add-on software on your cluster but Amazon EKS stops managing any settings for the add-on.

        If an IAM account is associated with the add-on, it isn't removed.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-addon.html#cfn-eks-addon-preserveondelete
        '''
        result = self._values.get("preserve_on_delete")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def resolve_conflicts(self) -> typing.Optional[builtins.str]:
        '''How to resolve field value conflicts for an Amazon EKS add-on.

        Conflicts are handled based on the value you choose:

        - *None* – If the self-managed version of the add-on is installed on your cluster, Amazon EKS doesn't change the value. Creation of the add-on might fail.
        - *Overwrite* – If the self-managed version of the add-on is installed on your cluster and the Amazon EKS default value is different than the existing value, Amazon EKS changes the value to the Amazon EKS default value.
        - *Preserve* – This is similar to the NONE option. If the self-managed version of the add-on is installed on your cluster Amazon EKS doesn't change the add-on resource properties. Creation of the add-on might fail if conflicts are detected. This option works differently during the update operation. For more information, see ```UpdateAddon`` <https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html>`_ .

        If you don't currently have the self-managed version of the add-on installed on your cluster, the Amazon EKS add-on is installed. Amazon EKS sets all values to default values, regardless of the option that you specify.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-addon.html#cfn-eks-addon-resolveconflicts
        '''
        result = self._values.get("resolve_conflicts")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_account_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of an existing IAM role to bind to the add-on's service account.

        The role must be assigned the IAM permissions required by the add-on. If you don't specify an existing IAM role, then the add-on uses the permissions assigned to the node IAM role. For more information, see `Amazon EKS node IAM role <https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html>`_ in the *Amazon EKS User Guide* .
        .. epigraph::

           To specify an existing IAM role, you must have an IAM OpenID Connect (OIDC) provider created for your cluster. For more information, see `Enabling IAM roles for service accounts on your cluster <https://docs.aws.amazon.com/eks/latest/userguide/enable-iam-roles-for-service-accounts.html>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-addon.html#cfn-eks-addon-serviceaccountrolearn
        '''
        result = self._values.get("service_account_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The metadata that you apply to the add-on to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Add-on tags do not propagate to any other resources associated with the cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-addon.html#cfn-eks-addon-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAddonProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnCluster(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.CfnCluster",
):
    '''Creates an Amazon EKS control plane.

    The Amazon EKS control plane consists of control plane instances that run the Kubernetes software, such as ``etcd`` and the API server. The control plane runs in an account managed by AWS , and the Kubernetes API is exposed by the Amazon EKS API server endpoint. Each Amazon EKS cluster control plane is single tenant and unique. It runs on its own set of Amazon EC2 instances.

    The cluster control plane is provisioned across multiple Availability Zones and fronted by an Elastic Load Balancing Network Load Balancer. Amazon EKS also provisions elastic network interfaces in your VPC subnets to provide connectivity from the control plane instances to the nodes (for example, to support ``kubectl exec`` , ``logs`` , and ``proxy`` data flows).

    Amazon EKS nodes run in your AWS account and connect to your cluster's control plane over the Kubernetes API server endpoint and a certificate file that is created for your cluster.

    You can use the ``endpointPublicAccess`` and ``endpointPrivateAccess`` parameters to enable or disable public and private access to your cluster's Kubernetes API server endpoint. By default, public access is enabled, and private access is disabled. For more information, see `Amazon EKS Cluster Endpoint Access Control <https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html>`_ in the **Amazon EKS User Guide** .

    You can use the ``logging`` parameter to enable or disable exporting the Kubernetes control plane logs for your cluster to CloudWatch Logs. By default, cluster control plane logs aren't exported to CloudWatch Logs. For more information, see `Amazon EKS Cluster Control Plane Logs <https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html>`_ in the **Amazon EKS User Guide** .
    .. epigraph::

       CloudWatch Logs ingestion, archive storage, and data scanning rates apply to exported control plane logs. For more information, see `CloudWatch Pricing <https://docs.aws.amazon.com/cloudwatch/pricing/>`_ .

    In most cases, it takes several minutes to create a cluster. After you create an Amazon EKS cluster, you must configure your Kubernetes tooling to communicate with the API server and launch nodes into your cluster. For more information, see `Allowing users to access your cluster <https://docs.aws.amazon.com/eks/latest/userguide/cluster-auth.html>`_ and `Launching Amazon EKS nodes <https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html>`_ in the *Amazon EKS User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html
    :cloudformationResource: AWS::EKS::Cluster
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        cfn_cluster = eks.CfnCluster(self, "MyCfnCluster",
            resources_vpc_config=eks.CfnCluster.ResourcesVpcConfigProperty(
                subnet_ids=["subnetIds"],
        
                # the properties below are optional
                endpoint_private_access=False,
                endpoint_public_access=False,
                public_access_cidrs=["publicAccessCidrs"],
                security_group_ids=["securityGroupIds"]
            ),
            role_arn="roleArn",
        
            # the properties below are optional
            access_config=eks.CfnCluster.AccessConfigProperty(
                authentication_mode="authenticationMode",
                bootstrap_cluster_creator_admin_permissions=False
            ),
            bootstrap_self_managed_addons=False,
            compute_config=eks.CfnCluster.ComputeConfigProperty(
                enabled=False,
                node_pools=["nodePools"],
                node_role_arn="nodeRoleArn"
            ),
            encryption_config=[eks.CfnCluster.EncryptionConfigProperty(
                provider=eks.CfnCluster.ProviderProperty(
                    key_arn="keyArn"
                ),
                resources=["resources"]
            )],
            force=False,
            kubernetes_network_config=eks.CfnCluster.KubernetesNetworkConfigProperty(
                elastic_load_balancing=eks.CfnCluster.ElasticLoadBalancingProperty(
                    enabled=False
                ),
                ip_family="ipFamily",
                service_ipv4_cidr="serviceIpv4Cidr",
                service_ipv6_cidr="serviceIpv6Cidr"
            ),
            logging=eks.CfnCluster.LoggingProperty(
                cluster_logging=eks.CfnCluster.ClusterLoggingProperty(
                    enabled_types=[eks.CfnCluster.LoggingTypeConfigProperty(
                        type="type"
                    )]
                )
            ),
            name="name",
            outpost_config=eks.CfnCluster.OutpostConfigProperty(
                control_plane_instance_type="controlPlaneInstanceType",
                outpost_arns=["outpostArns"],
        
                # the properties below are optional
                control_plane_placement=eks.CfnCluster.ControlPlanePlacementProperty(
                    group_name="groupName"
                )
            ),
            remote_network_config=eks.CfnCluster.RemoteNetworkConfigProperty(
                remote_node_networks=[eks.CfnCluster.RemoteNodeNetworkProperty(
                    cidrs=["cidrs"]
                )],
        
                # the properties below are optional
                remote_pod_networks=[eks.CfnCluster.RemotePodNetworkProperty(
                    cidrs=["cidrs"]
                )]
            ),
            storage_config=eks.CfnCluster.StorageConfigProperty(
                block_storage=eks.CfnCluster.BlockStorageProperty(
                    enabled=False
                )
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            upgrade_policy=eks.CfnCluster.UpgradePolicyProperty(
                support_type="supportType"
            ),
            version="version",
            zonal_shift_config=eks.CfnCluster.ZonalShiftConfigProperty(
                enabled=False
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        resources_vpc_config: typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ResourcesVpcConfigProperty", typing.Dict[builtins.str, typing.Any]]],
        role_arn: builtins.str,
        access_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.AccessConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        bootstrap_self_managed_addons: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        compute_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ComputeConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        encryption_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.EncryptionConfigProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        force: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        kubernetes_network_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.KubernetesNetworkConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        logging: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.LoggingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        name: typing.Optional[builtins.str] = None,
        outpost_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.OutpostConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        remote_network_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.RemoteNetworkConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        storage_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.StorageConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        upgrade_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.UpgradePolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        version: typing.Optional[builtins.str] = None,
        zonal_shift_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ZonalShiftConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param resources_vpc_config: The VPC configuration that's used by the cluster control plane. Amazon EKS VPC resources have specific requirements to work properly with Kubernetes. For more information, see `Cluster VPC Considerations <https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html>`_ and `Cluster Security Group Considerations <https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html>`_ in the *Amazon EKS User Guide* . You must specify at least two subnets. You can specify up to five security groups, but we recommend that you use a dedicated security group for your cluster control plane.
        :param role_arn: The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf. For more information, see `Amazon EKS Service IAM Role <https://docs.aws.amazon.com/eks/latest/userguide/service_IAM_role.html>`_ in the **Amazon EKS User Guide** .
        :param access_config: The access configuration for the cluster.
        :param bootstrap_self_managed_addons: If you set this value to ``False`` when creating a cluster, the default networking add-ons will not be installed. The default networking addons include vpc-cni, coredns, and kube-proxy. Use this option when you plan to install third-party alternative add-ons or self-manage the default networking add-ons.
        :param compute_config: Indicates the current configuration of the compute capability on your EKS Auto Mode cluster. For example, if the capability is enabled or disabled. If the compute capability is enabled, EKS Auto Mode will create and delete EC2 Managed Instances in your AWS account. For more information, see EKS Auto Mode compute capability in the *Amazon EKS User Guide* .
        :param encryption_config: The encryption configuration for the cluster.
        :param force: Set this value to ``true`` to override upgrade-blocking readiness checks when updating a cluster. Default: - false
        :param kubernetes_network_config: The Kubernetes network configuration for the cluster.
        :param logging: The logging configuration for your cluster.
        :param name: The unique name to give to your cluster. The name can contain only alphanumeric characters (case-sensitive) and hyphens. It must start with an alphanumeric character and can't be longer than 100 characters. The name must be unique within the AWS Region and AWS account that you're creating the cluster in. Note that underscores can't be used in AWS CloudFormation .
        :param outpost_config: An object representing the configuration of your local Amazon EKS cluster on an AWS Outpost. This object isn't available for clusters on the AWS cloud.
        :param remote_network_config: The configuration in the cluster for EKS Hybrid Nodes. You can add, change, or remove this configuration after the cluster is created.
        :param storage_config: Indicates the current configuration of the block storage capability on your EKS Auto Mode cluster. For example, if the capability is enabled or disabled. If the block storage capability is enabled, EKS Auto Mode will create and delete EBS volumes in your AWS account. For more information, see EKS Auto Mode block storage capability in the *Amazon EKS User Guide* .
        :param tags: The metadata that you apply to the cluster to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Cluster tags don't propagate to any other resources associated with the cluster. .. epigraph:: You must have the ``eks:TagResource`` and ``eks:UntagResource`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.
        :param upgrade_policy: This value indicates if extended support is enabled or disabled for the cluster. `Learn more about EKS Extended Support in the *Amazon EKS User Guide* . <https://docs.aws.amazon.com/eks/latest/userguide/extended-support-control.html>`_
        :param version: The desired Kubernetes version for your cluster. If you don't specify a value here, the default version available in Amazon EKS is used. .. epigraph:: The default version might not be the latest version available.
        :param zonal_shift_config: The configuration for zonal shift for the cluster.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d3e62a858014f3867f3039d1328d57223fb0d16e3fb6d1e2d79279938756eb35)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnClusterProps(
            resources_vpc_config=resources_vpc_config,
            role_arn=role_arn,
            access_config=access_config,
            bootstrap_self_managed_addons=bootstrap_self_managed_addons,
            compute_config=compute_config,
            encryption_config=encryption_config,
            force=force,
            kubernetes_network_config=kubernetes_network_config,
            logging=logging,
            name=name,
            outpost_config=outpost_config,
            remote_network_config=remote_network_config,
            storage_config=storage_config,
            tags=tags,
            upgrade_policy=upgrade_policy,
            version=version,
            zonal_shift_config=zonal_shift_config,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4ce8181eaff5e47deffee284e9005fc3985d7f0cc2ae10f69530ae44c00c9022)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b16f0b8fe893f26effc2cc7e839c081e4b5ed345b77ecb63e1f80d40f74890b9)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The ARN of the cluster, such as ``arn:aws:eks:us-west-2:666666666666:cluster/prod`` .

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrCertificateAuthorityData")
    def attr_certificate_authority_data(self) -> builtins.str:
        '''The ``certificate-authority-data`` for your cluster.

        :cloudformationAttribute: CertificateAuthorityData
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCertificateAuthorityData"))

    @builtins.property
    @jsii.member(jsii_name="attrClusterSecurityGroupId")
    def attr_cluster_security_group_id(self) -> builtins.str:
        '''The cluster security group that was created by Amazon EKS for the cluster.

        Managed node groups use this security group for control plane to data plane communication.

        This parameter is only returned by Amazon EKS clusters that support managed node groups. For more information, see `Managed node groups <https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html>`_ in the *Amazon EKS User Guide* .

        :cloudformationAttribute: ClusterSecurityGroupId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrClusterSecurityGroupId"))

    @builtins.property
    @jsii.member(jsii_name="attrEncryptionConfigKeyArn")
    def attr_encryption_config_key_arn(self) -> builtins.str:
        '''Amazon Resource Name (ARN) or alias of the customer master key (CMK).

        :cloudformationAttribute: EncryptionConfigKeyArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEncryptionConfigKeyArn"))

    @builtins.property
    @jsii.member(jsii_name="attrEndpoint")
    def attr_endpoint(self) -> builtins.str:
        '''The endpoint for your Kubernetes API server, such as ``https://5E1D0CEXAMPLEA591B746AFC5AB30262.yl4.us-west-2.eks.amazonaws.com`` .

        :cloudformationAttribute: Endpoint
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEndpoint"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID of your local Amazon EKS cluster on an AWS Outpost.

        This property isn't available for an Amazon EKS cluster on the AWS cloud.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrKubernetesNetworkConfigServiceIpv6Cidr")
    def attr_kubernetes_network_config_service_ipv6_cidr(self) -> builtins.str:
        '''The CIDR block that Kubernetes Service IP addresses are assigned from if you created a ``1.21`` or later cluster with version ``>1.10.1`` or later of the Amazon VPC CNI add-on and specified ``ipv6`` for *ipFamily* when you created the cluster. Kubernetes assigns Service addresses from the unique local address range ( ``fc00::/7`` ) because you can't specify a custom ``IPv6`` CIDR block when you create the cluster.

        :cloudformationAttribute: KubernetesNetworkConfig.ServiceIpv6Cidr
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrKubernetesNetworkConfigServiceIpv6Cidr"))

    @builtins.property
    @jsii.member(jsii_name="attrOpenIdConnectIssuerUrl")
    def attr_open_id_connect_issuer_url(self) -> builtins.str:
        '''The issuer URL for the OIDC identity provider of the cluster, such as ``https://oidc.eks.us-west-2.amazonaws.com/id/EXAMPLED539D4633E53DE1B716D3041E`` . If you need to remove ``https://`` from this output value, you can include the following code in your template.

        ``!Select [1, !Split ["//", !GetAtt EKSCluster.OpenIdConnectIssuerUrl]]``

        :cloudformationAttribute: OpenIdConnectIssuerUrl
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrOpenIdConnectIssuerUrl"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="resourcesVpcConfig")
    def resources_vpc_config(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnCluster.ResourcesVpcConfigProperty"]:
        '''The VPC configuration that's used by the cluster control plane.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnCluster.ResourcesVpcConfigProperty"], jsii.get(self, "resourcesVpcConfig"))

    @resources_vpc_config.setter
    def resources_vpc_config(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnCluster.ResourcesVpcConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7faae717d69e9ef08dfb05330d411b9baaf6babf8cf2b4715f8a02a4bd949f0c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "resourcesVpcConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf.'''
        return typing.cast(builtins.str, jsii.get(self, "roleArn"))

    @role_arn.setter
    def role_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__206ca712455daef1d5df9029fc8d95ff1ce9b54bd3c64781c94f6e97961f0df6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "roleArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="accessConfig")
    def access_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.AccessConfigProperty"]]:
        '''The access configuration for the cluster.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.AccessConfigProperty"]], jsii.get(self, "accessConfig"))

    @access_config.setter
    def access_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.AccessConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__249431e71bfb6d15cdb94aea4df14d4b3371f709cb261f00cb7dc77e72be7680)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "accessConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="bootstrapSelfManagedAddons")
    def bootstrap_self_managed_addons(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''If you set this value to ``False`` when creating a cluster, the default networking add-ons will not be installed.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "bootstrapSelfManagedAddons"))

    @bootstrap_self_managed_addons.setter
    def bootstrap_self_managed_addons(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1b3725246139251af199def1d548b17a13e8ddd4df825377563ea01cdea555c4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "bootstrapSelfManagedAddons", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="computeConfig")
    def compute_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ComputeConfigProperty"]]:
        '''Indicates the current configuration of the compute capability on your EKS Auto Mode cluster.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ComputeConfigProperty"]], jsii.get(self, "computeConfig"))

    @compute_config.setter
    def compute_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ComputeConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d35f88b28db161e1414d604c41ffc1d10fcf76351a0503d110f81158a8e15ca)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "computeConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="encryptionConfig")
    def encryption_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.EncryptionConfigProperty"]]]]:
        '''The encryption configuration for the cluster.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.EncryptionConfigProperty"]]]], jsii.get(self, "encryptionConfig"))

    @encryption_config.setter
    def encryption_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.EncryptionConfigProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b161fda542258d1cd8a20fecd3943cacecb658f19ab16b918baf49908459644c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "encryptionConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="force")
    def force(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Set this value to ``true`` to override upgrade-blocking readiness checks when updating a cluster.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "force"))

    @force.setter
    def force(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ec11778764d939bb12cbec68a418a76be768d7a638339c5189ae6cfe44059ebd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "force", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="kubernetesNetworkConfig")
    def kubernetes_network_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.KubernetesNetworkConfigProperty"]]:
        '''The Kubernetes network configuration for the cluster.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.KubernetesNetworkConfigProperty"]], jsii.get(self, "kubernetesNetworkConfig"))

    @kubernetes_network_config.setter
    def kubernetes_network_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.KubernetesNetworkConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1a14e543582631e80cb6fc2093270dba17f568b8779b381e3bc7398bfafb6699)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kubernetesNetworkConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="logging")
    def logging(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.LoggingProperty"]]:
        '''The logging configuration for your cluster.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.LoggingProperty"]], jsii.get(self, "logging"))

    @logging.setter
    def logging(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.LoggingProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1b894c6ae9e0b1d84fb3b436c969bb8c93262be35c47c6439769089b1b511372)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "logging", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''The unique name to give to your cluster.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2abdb0223ce1a286014d3194e6bdb56dedf9d45dcf81c88f7982910040414c06)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="outpostConfig")
    def outpost_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.OutpostConfigProperty"]]:
        '''An object representing the configuration of your local Amazon EKS cluster on an AWS Outpost.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.OutpostConfigProperty"]], jsii.get(self, "outpostConfig"))

    @outpost_config.setter
    def outpost_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.OutpostConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f8fa649bbaac6b12d9fb99e325d6fa46e4aea0d9d4d376abd8c7ac60009fc3d6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "outpostConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="remoteNetworkConfig")
    def remote_network_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.RemoteNetworkConfigProperty"]]:
        '''The configuration in the cluster for EKS Hybrid Nodes.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.RemoteNetworkConfigProperty"]], jsii.get(self, "remoteNetworkConfig"))

    @remote_network_config.setter
    def remote_network_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.RemoteNetworkConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cbca5e74535463c8d56fe42358d9cdf712c85039267d6580c6b1dbd855c5f255)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "remoteNetworkConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="storageConfig")
    def storage_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.StorageConfigProperty"]]:
        '''Indicates the current configuration of the block storage capability on your EKS Auto Mode cluster.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.StorageConfigProperty"]], jsii.get(self, "storageConfig"))

    @storage_config.setter
    def storage_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.StorageConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d43c0cb2280b293bbb9f09aa10579d43dbb7df4786771157ba149230c817ecfa)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "storageConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The metadata that you apply to the cluster to assist with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__201bfcb16f7ebd474e1c66a732f38f2234c0211f4a0ed45e7caebec7a8eb318a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="upgradePolicy")
    def upgrade_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.UpgradePolicyProperty"]]:
        '''This value indicates if extended support is enabled or disabled for the cluster.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.UpgradePolicyProperty"]], jsii.get(self, "upgradePolicy"))

    @upgrade_policy.setter
    def upgrade_policy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.UpgradePolicyProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__877948a2bf4549e042b17570af8e5dd5d247428b7baca13d83a600f308601333)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "upgradePolicy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[builtins.str]:
        '''The desired Kubernetes version for your cluster.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "version"))

    @version.setter
    def version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b02d8ac6f71535d635ac9b6a00c5e6c427f81f7fd71264c11d7353d212085e7a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "version", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="zonalShiftConfig")
    def zonal_shift_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ZonalShiftConfigProperty"]]:
        '''The configuration for zonal shift for the cluster.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ZonalShiftConfigProperty"]], jsii.get(self, "zonalShiftConfig"))

    @zonal_shift_config.setter
    def zonal_shift_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ZonalShiftConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42ffc6381adc5fa8c427a16d5fff74a44f4b6ebca1899d06adc2bd5949d0cfca)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "zonalShiftConfig", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.AccessConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "authentication_mode": "authenticationMode",
            "bootstrap_cluster_creator_admin_permissions": "bootstrapClusterCreatorAdminPermissions",
        },
    )
    class AccessConfigProperty:
        def __init__(
            self,
            *,
            authentication_mode: typing.Optional[builtins.str] = None,
            bootstrap_cluster_creator_admin_permissions: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''The access configuration for the cluster.

            :param authentication_mode: The desired authentication mode for the cluster. If you create a cluster by using the EKS API, AWS SDKs, or AWS CloudFormation , the default is ``CONFIG_MAP`` . If you create the cluster by using the AWS Management Console , the default value is ``API_AND_CONFIG_MAP`` .
            :param bootstrap_cluster_creator_admin_permissions: Specifies whether or not the cluster creator IAM principal was set as a cluster admin access entry during cluster creation time. The default value is ``true`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-accessconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                access_config_property = eks.CfnCluster.AccessConfigProperty(
                    authentication_mode="authenticationMode",
                    bootstrap_cluster_creator_admin_permissions=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4bfbf6adbd6203efb1ecf28834fc96b1030344f6fe766203b105462b96d8301f)
                check_type(argname="argument authentication_mode", value=authentication_mode, expected_type=type_hints["authentication_mode"])
                check_type(argname="argument bootstrap_cluster_creator_admin_permissions", value=bootstrap_cluster_creator_admin_permissions, expected_type=type_hints["bootstrap_cluster_creator_admin_permissions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if authentication_mode is not None:
                self._values["authentication_mode"] = authentication_mode
            if bootstrap_cluster_creator_admin_permissions is not None:
                self._values["bootstrap_cluster_creator_admin_permissions"] = bootstrap_cluster_creator_admin_permissions

        @builtins.property
        def authentication_mode(self) -> typing.Optional[builtins.str]:
            '''The desired authentication mode for the cluster.

            If you create a cluster by using the EKS API, AWS SDKs, or AWS CloudFormation , the default is ``CONFIG_MAP`` . If you create the cluster by using the AWS Management Console , the default value is ``API_AND_CONFIG_MAP`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-accessconfig.html#cfn-eks-cluster-accessconfig-authenticationmode
            '''
            result = self._values.get("authentication_mode")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def bootstrap_cluster_creator_admin_permissions(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether or not the cluster creator IAM principal was set as a cluster admin access entry during cluster creation time.

            The default value is ``true`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-accessconfig.html#cfn-eks-cluster-accessconfig-bootstrapclustercreatoradminpermissions
            '''
            result = self._values.get("bootstrap_cluster_creator_admin_permissions")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.BlockStorageProperty",
        jsii_struct_bases=[],
        name_mapping={"enabled": "enabled"},
    )
    class BlockStorageProperty:
        def __init__(
            self,
            *,
            enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''Indicates the current configuration of the block storage capability on your EKS Auto Mode cluster.

            For example, if the capability is enabled or disabled. If the block storage capability is enabled, EKS Auto Mode will create and delete EBS volumes in your AWS account. For more information, see EKS Auto Mode block storage capability in the *Amazon EKS User Guide* .

            :param enabled: Indicates if the block storage capability is enabled on your EKS Auto Mode cluster. If the block storage capability is enabled, EKS Auto Mode will create and delete EBS volumes in your AWS account.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-blockstorage.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                block_storage_property = eks.CfnCluster.BlockStorageProperty(
                    enabled=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__11cdae187ae128dcf0eca6a85c8d94b7c2ef69a0d54a9bd1917a1e27f48a5125)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if enabled is not None:
                self._values["enabled"] = enabled

        @builtins.property
        def enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates if the block storage capability is enabled on your EKS Auto Mode cluster.

            If the block storage capability is enabled, EKS Auto Mode will create and delete EBS volumes in your AWS account.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-blockstorage.html#cfn-eks-cluster-blockstorage-enabled
            '''
            result = self._values.get("enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BlockStorageProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.ClusterLoggingProperty",
        jsii_struct_bases=[],
        name_mapping={"enabled_types": "enabledTypes"},
    )
    class ClusterLoggingProperty:
        def __init__(
            self,
            *,
            enabled_types: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.LoggingTypeConfigProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The cluster control plane logging configuration for your cluster.

            .. epigraph::

               When updating a resource, you must include this ``ClusterLogging`` property if the previous CloudFormation template of the resource had it.

            :param enabled_types: The enabled control plane logs for your cluster. All log types are disabled if the array is empty. .. epigraph:: When updating a resource, you must include this ``EnabledTypes`` property if the previous CloudFormation template of the resource had it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-clusterlogging.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                cluster_logging_property = eks.CfnCluster.ClusterLoggingProperty(
                    enabled_types=[eks.CfnCluster.LoggingTypeConfigProperty(
                        type="type"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bf130126df3bc60611c7a3c0cab41949e35646410234341f59bc4dd2629f5996)
                check_type(argname="argument enabled_types", value=enabled_types, expected_type=type_hints["enabled_types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if enabled_types is not None:
                self._values["enabled_types"] = enabled_types

        @builtins.property
        def enabled_types(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.LoggingTypeConfigProperty"]]]]:
            '''The enabled control plane logs for your cluster. All log types are disabled if the array is empty.

            .. epigraph::

               When updating a resource, you must include this ``EnabledTypes`` property if the previous CloudFormation template of the resource had it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-clusterlogging.html#cfn-eks-cluster-clusterlogging-enabledtypes
            '''
            result = self._values.get("enabled_types")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.LoggingTypeConfigProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClusterLoggingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.ComputeConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled": "enabled",
            "node_pools": "nodePools",
            "node_role_arn": "nodeRoleArn",
        },
    )
    class ComputeConfigProperty:
        def __init__(
            self,
            *,
            enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            node_pools: typing.Optional[typing.Sequence[builtins.str]] = None,
            node_role_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Indicates the current configuration of the compute capability on your EKS Auto Mode cluster.

            For example, if the capability is enabled or disabled. If the compute capability is enabled, EKS Auto Mode will create and delete EC2 Managed Instances in your AWS account. For more information, see EKS Auto Mode compute capability in the *Amazon EKS User Guide* .

            :param enabled: Request to enable or disable the compute capability on your EKS Auto Mode cluster. If the compute capability is enabled, EKS Auto Mode will create and delete EC2 Managed Instances in your AWS account.
            :param node_pools: Configuration for node pools that defines the compute resources for your EKS Auto Mode cluster. For more information, see EKS Auto Mode Node Pools in the *Amazon EKS User Guide* .
            :param node_role_arn: The ARN of the IAM Role EKS will assign to EC2 Managed Instances in your EKS Auto Mode cluster. This value cannot be changed after the compute capability of EKS Auto Mode is enabled. For more information, see the IAM Reference in the *Amazon EKS User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-computeconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                compute_config_property = eks.CfnCluster.ComputeConfigProperty(
                    enabled=False,
                    node_pools=["nodePools"],
                    node_role_arn="nodeRoleArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__340acb2fac13d871050a5222d012205ae59e347bfba164c5431785621f6661b6)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument node_pools", value=node_pools, expected_type=type_hints["node_pools"])
                check_type(argname="argument node_role_arn", value=node_role_arn, expected_type=type_hints["node_role_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if enabled is not None:
                self._values["enabled"] = enabled
            if node_pools is not None:
                self._values["node_pools"] = node_pools
            if node_role_arn is not None:
                self._values["node_role_arn"] = node_role_arn

        @builtins.property
        def enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Request to enable or disable the compute capability on your EKS Auto Mode cluster.

            If the compute capability is enabled, EKS Auto Mode will create and delete EC2 Managed Instances in your AWS account.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-computeconfig.html#cfn-eks-cluster-computeconfig-enabled
            '''
            result = self._values.get("enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def node_pools(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Configuration for node pools that defines the compute resources for your EKS Auto Mode cluster.

            For more information, see EKS Auto Mode Node Pools in the *Amazon EKS User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-computeconfig.html#cfn-eks-cluster-computeconfig-nodepools
            '''
            result = self._values.get("node_pools")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def node_role_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of the IAM Role EKS will assign to EC2 Managed Instances in your EKS Auto Mode cluster.

            This value cannot be changed after the compute capability of EKS Auto Mode is enabled. For more information, see the IAM Reference in the *Amazon EKS User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-computeconfig.html#cfn-eks-cluster-computeconfig-noderolearn
            '''
            result = self._values.get("node_role_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ComputeConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.ControlPlanePlacementProperty",
        jsii_struct_bases=[],
        name_mapping={"group_name": "groupName"},
    )
    class ControlPlanePlacementProperty:
        def __init__(self, *, group_name: typing.Optional[builtins.str] = None) -> None:
            '''The placement configuration for all the control plane instances of your local Amazon EKS cluster on an AWS Outpost.

            For more information, see `Capacity considerations <https://docs.aws.amazon.com/eks/latest/userguide/eks-outposts-capacity-considerations.html>`_ in the *Amazon EKS User Guide* .

            :param group_name: The name of the placement group for the Kubernetes control plane instances. This property is only used for a local cluster on an AWS Outpost.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-controlplaneplacement.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                control_plane_placement_property = eks.CfnCluster.ControlPlanePlacementProperty(
                    group_name="groupName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d660e36a8e8c346e58bf586d09e62d3f36938b9cb7b40e9842c007af8090a1d3)
                check_type(argname="argument group_name", value=group_name, expected_type=type_hints["group_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if group_name is not None:
                self._values["group_name"] = group_name

        @builtins.property
        def group_name(self) -> typing.Optional[builtins.str]:
            '''The name of the placement group for the Kubernetes control plane instances.

            This property is only used for a local cluster on an AWS Outpost.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-controlplaneplacement.html#cfn-eks-cluster-controlplaneplacement-groupname
            '''
            result = self._values.get("group_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ControlPlanePlacementProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.ElasticLoadBalancingProperty",
        jsii_struct_bases=[],
        name_mapping={"enabled": "enabled"},
    )
    class ElasticLoadBalancingProperty:
        def __init__(
            self,
            *,
            enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''Indicates the current configuration of the load balancing capability on your EKS Auto Mode cluster.

            For example, if the capability is enabled or disabled. For more information, see EKS Auto Mode load balancing capability in the *Amazon EKS User Guide* .

            :param enabled: Indicates if the load balancing capability is enabled on your EKS Auto Mode cluster. If the load balancing capability is enabled, EKS Auto Mode will create and delete load balancers in your AWS account.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-elasticloadbalancing.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                elastic_load_balancing_property = eks.CfnCluster.ElasticLoadBalancingProperty(
                    enabled=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__41c8959626c6cd4c3ff5a15cfb2de0bea1a0b654c1bd2e0a62c4841ab19277cc)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if enabled is not None:
                self._values["enabled"] = enabled

        @builtins.property
        def enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates if the load balancing capability is enabled on your EKS Auto Mode cluster.

            If the load balancing capability is enabled, EKS Auto Mode will create and delete load balancers in your AWS account.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-elasticloadbalancing.html#cfn-eks-cluster-elasticloadbalancing-enabled
            '''
            result = self._values.get("enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ElasticLoadBalancingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.EncryptionConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"provider": "provider", "resources": "resources"},
    )
    class EncryptionConfigProperty:
        def __init__(
            self,
            *,
            provider: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ProviderProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            resources: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The encryption configuration for the cluster.

            :param provider: The encryption provider for the cluster.
            :param resources: Specifies the resources to be encrypted. The only supported value is ``secrets`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-encryptionconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                encryption_config_property = eks.CfnCluster.EncryptionConfigProperty(
                    provider=eks.CfnCluster.ProviderProperty(
                        key_arn="keyArn"
                    ),
                    resources=["resources"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f49095254b4d4121f694873b18c2ef5026ef5539eb56b045195b577601e55f41)
                check_type(argname="argument provider", value=provider, expected_type=type_hints["provider"])
                check_type(argname="argument resources", value=resources, expected_type=type_hints["resources"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if provider is not None:
                self._values["provider"] = provider
            if resources is not None:
                self._values["resources"] = resources

        @builtins.property
        def provider(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ProviderProperty"]]:
            '''The encryption provider for the cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-encryptionconfig.html#cfn-eks-cluster-encryptionconfig-provider
            '''
            result = self._values.get("provider")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ProviderProperty"]], result)

        @builtins.property
        def resources(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specifies the resources to be encrypted.

            The only supported value is ``secrets`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-encryptionconfig.html#cfn-eks-cluster-encryptionconfig-resources
            '''
            result = self._values.get("resources")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EncryptionConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.KubernetesNetworkConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "elastic_load_balancing": "elasticLoadBalancing",
            "ip_family": "ipFamily",
            "service_ipv4_cidr": "serviceIpv4Cidr",
            "service_ipv6_cidr": "serviceIpv6Cidr",
        },
    )
    class KubernetesNetworkConfigProperty:
        def __init__(
            self,
            *,
            elastic_load_balancing: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ElasticLoadBalancingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            ip_family: typing.Optional[builtins.str] = None,
            service_ipv4_cidr: typing.Optional[builtins.str] = None,
            service_ipv6_cidr: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The Kubernetes network configuration for the cluster.

            :param elastic_load_balancing: Request to enable or disable the load balancing capability on your EKS Auto Mode cluster. For more information, see EKS Auto Mode load balancing capability in the *Amazon EKS User Guide* .
            :param ip_family: Specify which IP family is used to assign Kubernetes pod and service IP addresses. If you don't specify a value, ``ipv4`` is used by default. You can only specify an IP family when you create a cluster and can't change this value once the cluster is created. If you specify ``ipv6`` , the VPC and subnets that you specify for cluster creation must have both ``IPv4`` and ``IPv6`` CIDR blocks assigned to them. You can't specify ``ipv6`` for clusters in China Regions. You can only specify ``ipv6`` for ``1.21`` and later clusters that use version ``1.10.1`` or later of the Amazon VPC CNI add-on. If you specify ``ipv6`` , then ensure that your VPC meets the requirements listed in the considerations listed in `Assigning IPv6 addresses to pods and services <https://docs.aws.amazon.com/eks/latest/userguide/cni-ipv6.html>`_ in the *Amazon EKS User Guide* . Kubernetes assigns services ``IPv6`` addresses from the unique local address range ``(fc00::/7)`` . You can't specify a custom ``IPv6`` CIDR block. Pod addresses are assigned from the subnet's ``IPv6`` CIDR.
            :param service_ipv4_cidr: Don't specify a value if you select ``ipv6`` for *ipFamily* . The CIDR block to assign Kubernetes service IP addresses from. If you don't specify a block, Kubernetes assigns addresses from either the ``10.100.0.0/16`` or ``172.20.0.0/16`` CIDR blocks. We recommend that you specify a block that does not overlap with resources in other networks that are peered or connected to your VPC. The block must meet the following requirements: - Within one of the following private IP address blocks: ``10.0.0.0/8`` , ``172.16.0.0/12`` , or ``192.168.0.0/16`` . - Doesn't overlap with any CIDR block assigned to the VPC that you selected for VPC. - Between ``/24`` and ``/12`` . .. epigraph:: You can only specify a custom CIDR block when you create a cluster. You can't change this value after the cluster is created.
            :param service_ipv6_cidr: The CIDR block that Kubernetes pod and service IP addresses are assigned from if you created a 1.21 or later cluster with version 1.10.1 or later of the Amazon VPC CNI add-on and specified ``ipv6`` for *ipFamily* when you created the cluster. Kubernetes assigns service addresses from the unique local address range ( ``fc00::/7`` ) because you can't specify a custom IPv6 CIDR block when you create the cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-kubernetesnetworkconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                kubernetes_network_config_property = eks.CfnCluster.KubernetesNetworkConfigProperty(
                    elastic_load_balancing=eks.CfnCluster.ElasticLoadBalancingProperty(
                        enabled=False
                    ),
                    ip_family="ipFamily",
                    service_ipv4_cidr="serviceIpv4Cidr",
                    service_ipv6_cidr="serviceIpv6Cidr"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0a92b4e070a67c74edb8714df451b4d215084e3116055e205e49b92778b8704f)
                check_type(argname="argument elastic_load_balancing", value=elastic_load_balancing, expected_type=type_hints["elastic_load_balancing"])
                check_type(argname="argument ip_family", value=ip_family, expected_type=type_hints["ip_family"])
                check_type(argname="argument service_ipv4_cidr", value=service_ipv4_cidr, expected_type=type_hints["service_ipv4_cidr"])
                check_type(argname="argument service_ipv6_cidr", value=service_ipv6_cidr, expected_type=type_hints["service_ipv6_cidr"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if elastic_load_balancing is not None:
                self._values["elastic_load_balancing"] = elastic_load_balancing
            if ip_family is not None:
                self._values["ip_family"] = ip_family
            if service_ipv4_cidr is not None:
                self._values["service_ipv4_cidr"] = service_ipv4_cidr
            if service_ipv6_cidr is not None:
                self._values["service_ipv6_cidr"] = service_ipv6_cidr

        @builtins.property
        def elastic_load_balancing(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ElasticLoadBalancingProperty"]]:
            '''Request to enable or disable the load balancing capability on your EKS Auto Mode cluster.

            For more information, see EKS Auto Mode load balancing capability in the *Amazon EKS User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-kubernetesnetworkconfig.html#cfn-eks-cluster-kubernetesnetworkconfig-elasticloadbalancing
            '''
            result = self._values.get("elastic_load_balancing")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ElasticLoadBalancingProperty"]], result)

        @builtins.property
        def ip_family(self) -> typing.Optional[builtins.str]:
            '''Specify which IP family is used to assign Kubernetes pod and service IP addresses.

            If you don't specify a value, ``ipv4`` is used by default. You can only specify an IP family when you create a cluster and can't change this value once the cluster is created. If you specify ``ipv6`` , the VPC and subnets that you specify for cluster creation must have both ``IPv4`` and ``IPv6`` CIDR blocks assigned to them. You can't specify ``ipv6`` for clusters in China Regions.

            You can only specify ``ipv6`` for ``1.21`` and later clusters that use version ``1.10.1`` or later of the Amazon VPC CNI add-on. If you specify ``ipv6`` , then ensure that your VPC meets the requirements listed in the considerations listed in `Assigning IPv6 addresses to pods and services <https://docs.aws.amazon.com/eks/latest/userguide/cni-ipv6.html>`_ in the *Amazon EKS User Guide* . Kubernetes assigns services ``IPv6`` addresses from the unique local address range ``(fc00::/7)`` . You can't specify a custom ``IPv6`` CIDR block. Pod addresses are assigned from the subnet's ``IPv6`` CIDR.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-kubernetesnetworkconfig.html#cfn-eks-cluster-kubernetesnetworkconfig-ipfamily
            '''
            result = self._values.get("ip_family")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def service_ipv4_cidr(self) -> typing.Optional[builtins.str]:
            '''Don't specify a value if you select ``ipv6`` for *ipFamily* .

            The CIDR block to assign Kubernetes service IP addresses from. If you don't specify a block, Kubernetes assigns addresses from either the ``10.100.0.0/16`` or ``172.20.0.0/16`` CIDR blocks. We recommend that you specify a block that does not overlap with resources in other networks that are peered or connected to your VPC. The block must meet the following requirements:

            - Within one of the following private IP address blocks: ``10.0.0.0/8`` , ``172.16.0.0/12`` , or ``192.168.0.0/16`` .
            - Doesn't overlap with any CIDR block assigned to the VPC that you selected for VPC.
            - Between ``/24`` and ``/12`` .

            .. epigraph::

               You can only specify a custom CIDR block when you create a cluster. You can't change this value after the cluster is created.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-kubernetesnetworkconfig.html#cfn-eks-cluster-kubernetesnetworkconfig-serviceipv4cidr
            '''
            result = self._values.get("service_ipv4_cidr")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def service_ipv6_cidr(self) -> typing.Optional[builtins.str]:
            '''The CIDR block that Kubernetes pod and service IP addresses are assigned from if you created a 1.21 or later cluster with version 1.10.1 or later of the Amazon VPC CNI add-on and specified ``ipv6`` for *ipFamily* when you created the cluster. Kubernetes assigns service addresses from the unique local address range ( ``fc00::/7`` ) because you can't specify a custom IPv6 CIDR block when you create the cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-kubernetesnetworkconfig.html#cfn-eks-cluster-kubernetesnetworkconfig-serviceipv6cidr
            '''
            result = self._values.get("service_ipv6_cidr")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KubernetesNetworkConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.LoggingProperty",
        jsii_struct_bases=[],
        name_mapping={"cluster_logging": "clusterLogging"},
    )
    class LoggingProperty:
        def __init__(
            self,
            *,
            cluster_logging: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ClusterLoggingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Enable or disable exporting the Kubernetes control plane logs for your cluster to CloudWatch Logs.

            By default, cluster control plane logs aren't exported to CloudWatch Logs. For more information, see `Amazon EKS Cluster control plane logs <https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html>`_ in the **Amazon EKS User Guide** .
            .. epigraph::

               When updating a resource, you must include this ``Logging`` property if the previous CloudFormation template of the resource had it. > CloudWatch Logs ingestion, archive storage, and data scanning rates apply to exported control plane logs. For more information, see `CloudWatch Pricing <https://docs.aws.amazon.com/cloudwatch/pricing/>`_ .

            :param cluster_logging: The cluster control plane logging configuration for your cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-logging.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                logging_property = eks.CfnCluster.LoggingProperty(
                    cluster_logging=eks.CfnCluster.ClusterLoggingProperty(
                        enabled_types=[eks.CfnCluster.LoggingTypeConfigProperty(
                            type="type"
                        )]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9fa9915cb49fabeb00a3235507f20066b62866e18750f198da96f824197d5226)
                check_type(argname="argument cluster_logging", value=cluster_logging, expected_type=type_hints["cluster_logging"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cluster_logging is not None:
                self._values["cluster_logging"] = cluster_logging

        @builtins.property
        def cluster_logging(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ClusterLoggingProperty"]]:
            '''The cluster control plane logging configuration for your cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-logging.html#cfn-eks-cluster-logging-clusterlogging
            '''
            result = self._values.get("cluster_logging")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ClusterLoggingProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.LoggingTypeConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type"},
    )
    class LoggingTypeConfigProperty:
        def __init__(self, *, type: typing.Optional[builtins.str] = None) -> None:
            '''The enabled logging type.

            For a list of the valid logging types, see the ```types`` property of ``LogSetup`` <https://docs.aws.amazon.com/eks/latest/APIReference/API_LogSetup.html#AmazonEKS-Type-LogSetup-types>`_ in the *Amazon EKS API Reference* .

            :param type: The name of the log type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-loggingtypeconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                logging_type_config_property = eks.CfnCluster.LoggingTypeConfigProperty(
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2416911a4b34f3fd91c8c2b50f74aa6d73ad710b43f2975cbddb4f6858a274f1)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The name of the log type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-loggingtypeconfig.html#cfn-eks-cluster-loggingtypeconfig-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingTypeConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.OutpostConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "control_plane_instance_type": "controlPlaneInstanceType",
            "outpost_arns": "outpostArns",
            "control_plane_placement": "controlPlanePlacement",
        },
    )
    class OutpostConfigProperty:
        def __init__(
            self,
            *,
            control_plane_instance_type: builtins.str,
            outpost_arns: typing.Sequence[builtins.str],
            control_plane_placement: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.ControlPlanePlacementProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration of your local Amazon EKS cluster on an AWS Outpost.

            Before creating a cluster on an Outpost, review `Creating a local cluster on an Outpost <https://docs.aws.amazon.com/eks/latest/userguide/eks-outposts-local-cluster-create.html>`_ in the *Amazon EKS User Guide* . This API isn't available for Amazon EKS clusters on the AWS cloud.

            :param control_plane_instance_type: The Amazon EC2 instance type that you want to use for your local Amazon EKS cluster on Outposts. Choose an instance type based on the number of nodes that your cluster will have. For more information, see `Capacity considerations <https://docs.aws.amazon.com/eks/latest/userguide/eks-outposts-capacity-considerations.html>`_ in the *Amazon EKS User Guide* . The instance type that you specify is used for all Kubernetes control plane instances. The instance type can't be changed after cluster creation. The control plane is not automatically scaled by Amazon EKS.
            :param outpost_arns: The ARN of the Outpost that you want to use for your local Amazon EKS cluster on Outposts. Only a single Outpost ARN is supported.
            :param control_plane_placement: An object representing the placement configuration for all the control plane instances of your local Amazon EKS cluster on an AWS Outpost. For more information, see `Capacity considerations <https://docs.aws.amazon.com/eks/latest/userguide/eks-outposts-capacity-considerations.html>`_ in the *Amazon EKS User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-outpostconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                outpost_config_property = eks.CfnCluster.OutpostConfigProperty(
                    control_plane_instance_type="controlPlaneInstanceType",
                    outpost_arns=["outpostArns"],
                
                    # the properties below are optional
                    control_plane_placement=eks.CfnCluster.ControlPlanePlacementProperty(
                        group_name="groupName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__db660b2ea60bc92d49db95b07a452b00775f8630f8510e0a8b731bd1e01399a3)
                check_type(argname="argument control_plane_instance_type", value=control_plane_instance_type, expected_type=type_hints["control_plane_instance_type"])
                check_type(argname="argument outpost_arns", value=outpost_arns, expected_type=type_hints["outpost_arns"])
                check_type(argname="argument control_plane_placement", value=control_plane_placement, expected_type=type_hints["control_plane_placement"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "control_plane_instance_type": control_plane_instance_type,
                "outpost_arns": outpost_arns,
            }
            if control_plane_placement is not None:
                self._values["control_plane_placement"] = control_plane_placement

        @builtins.property
        def control_plane_instance_type(self) -> builtins.str:
            '''The Amazon EC2 instance type that you want to use for your local Amazon EKS cluster on Outposts.

            Choose an instance type based on the number of nodes that your cluster will have. For more information, see `Capacity considerations <https://docs.aws.amazon.com/eks/latest/userguide/eks-outposts-capacity-considerations.html>`_ in the *Amazon EKS User Guide* .

            The instance type that you specify is used for all Kubernetes control plane instances. The instance type can't be changed after cluster creation. The control plane is not automatically scaled by Amazon EKS.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-outpostconfig.html#cfn-eks-cluster-outpostconfig-controlplaneinstancetype
            '''
            result = self._values.get("control_plane_instance_type")
            assert result is not None, "Required property 'control_plane_instance_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def outpost_arns(self) -> typing.List[builtins.str]:
            '''The ARN of the Outpost that you want to use for your local Amazon EKS cluster on Outposts.

            Only a single Outpost ARN is supported.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-outpostconfig.html#cfn-eks-cluster-outpostconfig-outpostarns
            '''
            result = self._values.get("outpost_arns")
            assert result is not None, "Required property 'outpost_arns' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def control_plane_placement(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ControlPlanePlacementProperty"]]:
            '''An object representing the placement configuration for all the control plane instances of your local Amazon EKS cluster on an AWS Outpost.

            For more information, see `Capacity considerations <https://docs.aws.amazon.com/eks/latest/userguide/eks-outposts-capacity-considerations.html>`_ in the *Amazon EKS User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-outpostconfig.html#cfn-eks-cluster-outpostconfig-controlplaneplacement
            '''
            result = self._values.get("control_plane_placement")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.ControlPlanePlacementProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OutpostConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.ProviderProperty",
        jsii_struct_bases=[],
        name_mapping={"key_arn": "keyArn"},
    )
    class ProviderProperty:
        def __init__(self, *, key_arn: typing.Optional[builtins.str] = None) -> None:
            '''Identifies the AWS Key Management Service ( AWS KMS ) key used to encrypt the secrets.

            :param key_arn: Amazon Resource Name (ARN) or alias of the KMS key. The KMS key must be symmetric and created in the same AWS Region as the cluster. If the KMS key was created in a different account, the `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ must have access to the KMS key. For more information, see `Allowing users in other accounts to use a KMS key <https://docs.aws.amazon.com/kms/latest/developerguide/key-policy-modifying-external-accounts.html>`_ in the *AWS Key Management Service Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-provider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                provider_property = eks.CfnCluster.ProviderProperty(
                    key_arn="keyArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0335c5b37409b58aeeda75dd12fbc1f97e1731a3d1f059652c77cea883faebe8)
                check_type(argname="argument key_arn", value=key_arn, expected_type=type_hints["key_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if key_arn is not None:
                self._values["key_arn"] = key_arn

        @builtins.property
        def key_arn(self) -> typing.Optional[builtins.str]:
            '''Amazon Resource Name (ARN) or alias of the KMS key.

            The KMS key must be symmetric and created in the same AWS Region as the cluster. If the KMS key was created in a different account, the `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ must have access to the KMS key. For more information, see `Allowing users in other accounts to use a KMS key <https://docs.aws.amazon.com/kms/latest/developerguide/key-policy-modifying-external-accounts.html>`_ in the *AWS Key Management Service Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-provider.html#cfn-eks-cluster-provider-keyarn
            '''
            result = self._values.get("key_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.RemoteNetworkConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "remote_node_networks": "remoteNodeNetworks",
            "remote_pod_networks": "remotePodNetworks",
        },
    )
    class RemoteNetworkConfigProperty:
        def __init__(
            self,
            *,
            remote_node_networks: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.RemoteNodeNetworkProperty", typing.Dict[builtins.str, typing.Any]]]]],
            remote_pod_networks: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.RemotePodNetworkProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The configuration in the cluster for EKS Hybrid Nodes.

            You can add, change, or remove this configuration after the cluster is created.

            :param remote_node_networks: The list of network CIDRs that can contain hybrid nodes. These CIDR blocks define the expected IP address range of the hybrid nodes that join the cluster. These blocks are typically determined by your network administrator. Enter one or more IPv4 CIDR blocks in decimal dotted-quad notation (for example, ``10.2.0.0/16`` ). It must satisfy the following requirements: - Each block must be within an ``IPv4`` RFC-1918 network range. Minimum allowed size is /24, maximum allowed size is /8. Publicly-routable addresses aren't supported. - Each block cannot overlap with the range of the VPC CIDR blocks for your EKS resources, or the block of the Kubernetes service IP range. - Each block must have a route to the VPC that uses the VPC CIDR blocks, not public IPs or Elastic IPs. There are many options including AWS Transit Gateway , AWS Site-to-Site VPN , or AWS Direct Connect . - Each host must allow outbound connection to the EKS cluster control plane on TCP ports ``443`` and ``10250`` . - Each host must allow inbound connection from the EKS cluster control plane on TCP port 10250 for logs, exec and port-forward operations. - Each host must allow TCP and UDP network connectivity to and from other hosts that are running ``CoreDNS`` on UDP port ``53`` for service and pod DNS names.
            :param remote_pod_networks: The list of network CIDRs that can contain pods that run Kubernetes webhooks on hybrid nodes. These CIDR blocks are determined by configuring your Container Network Interface (CNI) plugin. We recommend the Calico CNI or Cilium CNI. Note that the Amazon VPC CNI plugin for Kubernetes isn't available for on-premises and edge locations. Enter one or more IPv4 CIDR blocks in decimal dotted-quad notation (for example, ``10.2.0.0/16`` ). It must satisfy the following requirements: - Each block must be within an ``IPv4`` RFC-1918 network range. Minimum allowed size is /24, maximum allowed size is /8. Publicly-routable addresses aren't supported. - Each block cannot overlap with the range of the VPC CIDR blocks for your EKS resources, or the block of the Kubernetes service IP range.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-remotenetworkconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                remote_network_config_property = eks.CfnCluster.RemoteNetworkConfigProperty(
                    remote_node_networks=[eks.CfnCluster.RemoteNodeNetworkProperty(
                        cidrs=["cidrs"]
                    )],
                
                    # the properties below are optional
                    remote_pod_networks=[eks.CfnCluster.RemotePodNetworkProperty(
                        cidrs=["cidrs"]
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__29b132ee7bcaaae9fba08ec6dcbfb1b776b9043d57574e18c26975e2109c5a02)
                check_type(argname="argument remote_node_networks", value=remote_node_networks, expected_type=type_hints["remote_node_networks"])
                check_type(argname="argument remote_pod_networks", value=remote_pod_networks, expected_type=type_hints["remote_pod_networks"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "remote_node_networks": remote_node_networks,
            }
            if remote_pod_networks is not None:
                self._values["remote_pod_networks"] = remote_pod_networks

        @builtins.property
        def remote_node_networks(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.RemoteNodeNetworkProperty"]]]:
            '''The list of network CIDRs that can contain hybrid nodes.

            These CIDR blocks define the expected IP address range of the hybrid nodes that join the cluster. These blocks are typically determined by your network administrator.

            Enter one or more IPv4 CIDR blocks in decimal dotted-quad notation (for example, ``10.2.0.0/16`` ).

            It must satisfy the following requirements:

            - Each block must be within an ``IPv4`` RFC-1918 network range. Minimum allowed size is /24, maximum allowed size is /8. Publicly-routable addresses aren't supported.
            - Each block cannot overlap with the range of the VPC CIDR blocks for your EKS resources, or the block of the Kubernetes service IP range.
            - Each block must have a route to the VPC that uses the VPC CIDR blocks, not public IPs or Elastic IPs. There are many options including AWS Transit Gateway , AWS Site-to-Site VPN , or AWS Direct Connect .
            - Each host must allow outbound connection to the EKS cluster control plane on TCP ports ``443`` and ``10250`` .
            - Each host must allow inbound connection from the EKS cluster control plane on TCP port 10250 for logs, exec and port-forward operations.
            - Each host must allow TCP and UDP network connectivity to and from other hosts that are running ``CoreDNS`` on UDP port ``53`` for service and pod DNS names.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-remotenetworkconfig.html#cfn-eks-cluster-remotenetworkconfig-remotenodenetworks
            '''
            result = self._values.get("remote_node_networks")
            assert result is not None, "Required property 'remote_node_networks' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.RemoteNodeNetworkProperty"]]], result)

        @builtins.property
        def remote_pod_networks(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.RemotePodNetworkProperty"]]]]:
            '''The list of network CIDRs that can contain pods that run Kubernetes webhooks on hybrid nodes.

            These CIDR blocks are determined by configuring your Container Network Interface (CNI) plugin. We recommend the Calico CNI or Cilium CNI. Note that the Amazon VPC CNI plugin for Kubernetes isn't available for on-premises and edge locations.

            Enter one or more IPv4 CIDR blocks in decimal dotted-quad notation (for example, ``10.2.0.0/16`` ).

            It must satisfy the following requirements:

            - Each block must be within an ``IPv4`` RFC-1918 network range. Minimum allowed size is /24, maximum allowed size is /8. Publicly-routable addresses aren't supported.
            - Each block cannot overlap with the range of the VPC CIDR blocks for your EKS resources, or the block of the Kubernetes service IP range.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-remotenetworkconfig.html#cfn-eks-cluster-remotenetworkconfig-remotepodnetworks
            '''
            result = self._values.get("remote_pod_networks")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCluster.RemotePodNetworkProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RemoteNetworkConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.RemoteNodeNetworkProperty",
        jsii_struct_bases=[],
        name_mapping={"cidrs": "cidrs"},
    )
    class RemoteNodeNetworkProperty:
        def __init__(self, *, cidrs: typing.Sequence[builtins.str]) -> None:
            '''A network CIDR that can contain hybrid nodes.

            These CIDR blocks define the expected IP address range of the hybrid nodes that join the cluster. These blocks are typically determined by your network administrator.

            Enter one or more IPv4 CIDR blocks in decimal dotted-quad notation (for example, ``10.2.0.0/16`` ).

            It must satisfy the following requirements:

            - Each block must be within an ``IPv4`` RFC-1918 network range. Minimum allowed size is /24, maximum allowed size is /8. Publicly-routable addresses aren't supported.
            - Each block cannot overlap with the range of the VPC CIDR blocks for your EKS resources, or the block of the Kubernetes service IP range.
            - Each block must have a route to the VPC that uses the VPC CIDR blocks, not public IPs or Elastic IPs. There are many options including AWS Transit Gateway , AWS Site-to-Site VPN , or AWS Direct Connect .
            - Each host must allow outbound connection to the EKS cluster control plane on TCP ports ``443`` and ``10250`` .
            - Each host must allow inbound connection from the EKS cluster control plane on TCP port 10250 for logs, exec and port-forward operations.
            - Each host must allow TCP and UDP network connectivity to and from other hosts that are running ``CoreDNS`` on UDP port ``53`` for service and pod DNS names.

            :param cidrs: A network CIDR that can contain hybrid nodes. These CIDR blocks define the expected IP address range of the hybrid nodes that join the cluster. These blocks are typically determined by your network administrator. Enter one or more IPv4 CIDR blocks in decimal dotted-quad notation (for example, ``10.2.0.0/16`` ). It must satisfy the following requirements: - Each block must be within an ``IPv4`` RFC-1918 network range. Minimum allowed size is /24, maximum allowed size is /8. Publicly-routable addresses aren't supported. - Each block cannot overlap with the range of the VPC CIDR blocks for your EKS resources, or the block of the Kubernetes service IP range. - Each block must have a route to the VPC that uses the VPC CIDR blocks, not public IPs or Elastic IPs. There are many options including AWS Transit Gateway , AWS Site-to-Site VPN , or AWS Direct Connect . - Each host must allow outbound connection to the EKS cluster control plane on TCP ports ``443`` and ``10250`` . - Each host must allow inbound connection from the EKS cluster control plane on TCP port 10250 for logs, exec and port-forward operations. - Each host must allow TCP and UDP network connectivity to and from other hosts that are running ``CoreDNS`` on UDP port ``53`` for service and pod DNS names.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-remotenodenetwork.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                remote_node_network_property = eks.CfnCluster.RemoteNodeNetworkProperty(
                    cidrs=["cidrs"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3c788baedbcb7f230114fd8b1ee014478977a6bcf1e2ac104dbaa8f48eda3f53)
                check_type(argname="argument cidrs", value=cidrs, expected_type=type_hints["cidrs"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "cidrs": cidrs,
            }

        @builtins.property
        def cidrs(self) -> typing.List[builtins.str]:
            '''A network CIDR that can contain hybrid nodes.

            These CIDR blocks define the expected IP address range of the hybrid nodes that join the cluster. These blocks are typically determined by your network administrator.

            Enter one or more IPv4 CIDR blocks in decimal dotted-quad notation (for example, ``10.2.0.0/16`` ).

            It must satisfy the following requirements:

            - Each block must be within an ``IPv4`` RFC-1918 network range. Minimum allowed size is /24, maximum allowed size is /8. Publicly-routable addresses aren't supported.
            - Each block cannot overlap with the range of the VPC CIDR blocks for your EKS resources, or the block of the Kubernetes service IP range.
            - Each block must have a route to the VPC that uses the VPC CIDR blocks, not public IPs or Elastic IPs. There are many options including AWS Transit Gateway , AWS Site-to-Site VPN , or AWS Direct Connect .
            - Each host must allow outbound connection to the EKS cluster control plane on TCP ports ``443`` and ``10250`` .
            - Each host must allow inbound connection from the EKS cluster control plane on TCP port 10250 for logs, exec and port-forward operations.
            - Each host must allow TCP and UDP network connectivity to and from other hosts that are running ``CoreDNS`` on UDP port ``53`` for service and pod DNS names.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-remotenodenetwork.html#cfn-eks-cluster-remotenodenetwork-cidrs
            '''
            result = self._values.get("cidrs")
            assert result is not None, "Required property 'cidrs' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RemoteNodeNetworkProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.RemotePodNetworkProperty",
        jsii_struct_bases=[],
        name_mapping={"cidrs": "cidrs"},
    )
    class RemotePodNetworkProperty:
        def __init__(self, *, cidrs: typing.Sequence[builtins.str]) -> None:
            '''A network CIDR that can contain pods that run Kubernetes webhooks on hybrid nodes.

            These CIDR blocks are determined by configuring your Container Network Interface (CNI) plugin. We recommend the Calico CNI or Cilium CNI. Note that the Amazon VPC CNI plugin for Kubernetes isn't available for on-premises and edge locations.

            Enter one or more IPv4 CIDR blocks in decimal dotted-quad notation (for example, ``10.2.0.0/16`` ).

            It must satisfy the following requirements:

            - Each block must be within an ``IPv4`` RFC-1918 network range. Minimum allowed size is /24, maximum allowed size is /8. Publicly-routable addresses aren't supported.
            - Each block cannot overlap with the range of the VPC CIDR blocks for your EKS resources, or the block of the Kubernetes service IP range.

            :param cidrs: A network CIDR that can contain pods that run Kubernetes webhooks on hybrid nodes. These CIDR blocks are determined by configuring your Container Network Interface (CNI) plugin. We recommend the Calico CNI or Cilium CNI. Note that the Amazon VPC CNI plugin for Kubernetes isn't available for on-premises and edge locations. Enter one or more IPv4 CIDR blocks in decimal dotted-quad notation (for example, ``10.2.0.0/16`` ). It must satisfy the following requirements: - Each block must be within an ``IPv4`` RFC-1918 network range. Minimum allowed size is /24, maximum allowed size is /8. Publicly-routable addresses aren't supported. - Each block cannot overlap with the range of the VPC CIDR blocks for your EKS resources, or the block of the Kubernetes service IP range.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-remotepodnetwork.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                remote_pod_network_property = eks.CfnCluster.RemotePodNetworkProperty(
                    cidrs=["cidrs"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__51b845e43ba054d464150b1f0fffb60f026af086df16857e5625284cc8b922c6)
                check_type(argname="argument cidrs", value=cidrs, expected_type=type_hints["cidrs"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "cidrs": cidrs,
            }

        @builtins.property
        def cidrs(self) -> typing.List[builtins.str]:
            '''A network CIDR that can contain pods that run Kubernetes webhooks on hybrid nodes.

            These CIDR blocks are determined by configuring your Container Network Interface (CNI) plugin. We recommend the Calico CNI or Cilium CNI. Note that the Amazon VPC CNI plugin for Kubernetes isn't available for on-premises and edge locations.

            Enter one or more IPv4 CIDR blocks in decimal dotted-quad notation (for example, ``10.2.0.0/16`` ).

            It must satisfy the following requirements:

            - Each block must be within an ``IPv4`` RFC-1918 network range. Minimum allowed size is /24, maximum allowed size is /8. Publicly-routable addresses aren't supported.
            - Each block cannot overlap with the range of the VPC CIDR blocks for your EKS resources, or the block of the Kubernetes service IP range.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-remotepodnetwork.html#cfn-eks-cluster-remotepodnetwork-cidrs
            '''
            result = self._values.get("cidrs")
            assert result is not None, "Required property 'cidrs' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RemotePodNetworkProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.ResourcesVpcConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "subnet_ids": "subnetIds",
            "endpoint_private_access": "endpointPrivateAccess",
            "endpoint_public_access": "endpointPublicAccess",
            "public_access_cidrs": "publicAccessCidrs",
            "security_group_ids": "securityGroupIds",
        },
    )
    class ResourcesVpcConfigProperty:
        def __init__(
            self,
            *,
            subnet_ids: typing.Sequence[builtins.str],
            endpoint_private_access: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            endpoint_public_access: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            public_access_cidrs: typing.Optional[typing.Sequence[builtins.str]] = None,
            security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object representing the VPC configuration to use for an Amazon EKS cluster.

            .. epigraph::

               When updating a resource, you must include these properties if the previous CloudFormation template of the resource had them:

               - ``EndpointPublicAccess``
               - ``EndpointPrivateAccess``
               - ``PublicAccessCidrs``

            :param subnet_ids: Specify subnets for your Amazon EKS nodes. Amazon EKS creates cross-account elastic network interfaces in these subnets to allow communication between your nodes and the Kubernetes control plane.
            :param endpoint_private_access: Set this value to ``true`` to enable private access for your cluster's Kubernetes API server endpoint. If you enable private access, Kubernetes API requests from within your cluster's VPC use the private VPC endpoint. The default value for this parameter is ``false`` , which disables private access for your Kubernetes API server. If you disable private access and you have nodes or AWS Fargate pods in the cluster, then ensure that ``publicAccessCidrs`` includes the necessary CIDR blocks for communication with the nodes or Fargate pods. For more information, see `Amazon EKS cluster endpoint access control <https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html>`_ in the **Amazon EKS User Guide** .
            :param endpoint_public_access: Set this value to ``false`` to disable public access to your cluster's Kubernetes API server endpoint. If you disable public access, your cluster's Kubernetes API server can only receive requests from within the cluster VPC. The default value for this parameter is ``true`` , which enables public access for your Kubernetes API server. For more information, see `Amazon EKS cluster endpoint access control <https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html>`_ in the **Amazon EKS User Guide** .
            :param public_access_cidrs: The CIDR blocks that are allowed access to your cluster's public Kubernetes API server endpoint. Communication to the endpoint from addresses outside of the CIDR blocks that you specify is denied. The default value is ``0.0.0.0/0`` . If you've disabled private endpoint access, make sure that you specify the necessary CIDR blocks for every node and AWS Fargate ``Pod`` in the cluster. For more information, see `Amazon EKS cluster endpoint access control <https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html>`_ in the **Amazon EKS User Guide** .
            :param security_group_ids: Specify one or more security groups for the cross-account elastic network interfaces that Amazon EKS creates to use that allow communication between your nodes and the Kubernetes control plane. If you don't specify any security groups, then familiarize yourself with the difference between Amazon EKS defaults for clusters deployed with Kubernetes. For more information, see `Amazon EKS security group considerations <https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html>`_ in the **Amazon EKS User Guide** .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-resourcesvpcconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                resources_vpc_config_property = eks.CfnCluster.ResourcesVpcConfigProperty(
                    subnet_ids=["subnetIds"],
                
                    # the properties below are optional
                    endpoint_private_access=False,
                    endpoint_public_access=False,
                    public_access_cidrs=["publicAccessCidrs"],
                    security_group_ids=["securityGroupIds"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__986289b8a80017f390950fa94e8370d9961848f3cf42f347c78c0c91f0d06148)
                check_type(argname="argument subnet_ids", value=subnet_ids, expected_type=type_hints["subnet_ids"])
                check_type(argname="argument endpoint_private_access", value=endpoint_private_access, expected_type=type_hints["endpoint_private_access"])
                check_type(argname="argument endpoint_public_access", value=endpoint_public_access, expected_type=type_hints["endpoint_public_access"])
                check_type(argname="argument public_access_cidrs", value=public_access_cidrs, expected_type=type_hints["public_access_cidrs"])
                check_type(argname="argument security_group_ids", value=security_group_ids, expected_type=type_hints["security_group_ids"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "subnet_ids": subnet_ids,
            }
            if endpoint_private_access is not None:
                self._values["endpoint_private_access"] = endpoint_private_access
            if endpoint_public_access is not None:
                self._values["endpoint_public_access"] = endpoint_public_access
            if public_access_cidrs is not None:
                self._values["public_access_cidrs"] = public_access_cidrs
            if security_group_ids is not None:
                self._values["security_group_ids"] = security_group_ids

        @builtins.property
        def subnet_ids(self) -> typing.List[builtins.str]:
            '''Specify subnets for your Amazon EKS nodes.

            Amazon EKS creates cross-account elastic network interfaces in these subnets to allow communication between your nodes and the Kubernetes control plane.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-resourcesvpcconfig.html#cfn-eks-cluster-resourcesvpcconfig-subnetids
            '''
            result = self._values.get("subnet_ids")
            assert result is not None, "Required property 'subnet_ids' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def endpoint_private_access(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Set this value to ``true`` to enable private access for your cluster's Kubernetes API server endpoint.

            If you enable private access, Kubernetes API requests from within your cluster's VPC use the private VPC endpoint. The default value for this parameter is ``false`` , which disables private access for your Kubernetes API server. If you disable private access and you have nodes or AWS Fargate pods in the cluster, then ensure that ``publicAccessCidrs`` includes the necessary CIDR blocks for communication with the nodes or Fargate pods. For more information, see `Amazon EKS cluster endpoint access control <https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html>`_ in the **Amazon EKS User Guide** .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-resourcesvpcconfig.html#cfn-eks-cluster-resourcesvpcconfig-endpointprivateaccess
            '''
            result = self._values.get("endpoint_private_access")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def endpoint_public_access(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Set this value to ``false`` to disable public access to your cluster's Kubernetes API server endpoint.

            If you disable public access, your cluster's Kubernetes API server can only receive requests from within the cluster VPC. The default value for this parameter is ``true`` , which enables public access for your Kubernetes API server. For more information, see `Amazon EKS cluster endpoint access control <https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html>`_ in the **Amazon EKS User Guide** .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-resourcesvpcconfig.html#cfn-eks-cluster-resourcesvpcconfig-endpointpublicaccess
            '''
            result = self._values.get("endpoint_public_access")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def public_access_cidrs(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The CIDR blocks that are allowed access to your cluster's public Kubernetes API server endpoint.

            Communication to the endpoint from addresses outside of the CIDR blocks that you specify is denied. The default value is ``0.0.0.0/0`` . If you've disabled private endpoint access, make sure that you specify the necessary CIDR blocks for every node and AWS Fargate ``Pod`` in the cluster. For more information, see `Amazon EKS cluster endpoint access control <https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html>`_ in the **Amazon EKS User Guide** .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-resourcesvpcconfig.html#cfn-eks-cluster-resourcesvpcconfig-publicaccesscidrs
            '''
            result = self._values.get("public_access_cidrs")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify one or more security groups for the cross-account elastic network interfaces that Amazon EKS creates to use that allow communication between your nodes and the Kubernetes control plane.

            If you don't specify any security groups, then familiarize yourself with the difference between Amazon EKS defaults for clusters deployed with Kubernetes. For more information, see `Amazon EKS security group considerations <https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html>`_ in the **Amazon EKS User Guide** .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-resourcesvpcconfig.html#cfn-eks-cluster-resourcesvpcconfig-securitygroupids
            '''
            result = self._values.get("security_group_ids")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ResourcesVpcConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.StorageConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"block_storage": "blockStorage"},
    )
    class StorageConfigProperty:
        def __init__(
            self,
            *,
            block_storage: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCluster.BlockStorageProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Request to update the configuration of the storage capability of your EKS Auto Mode cluster.

            For example, enable the capability. For more information, see EKS Auto Mode block storage capability in the *Amazon EKS User Guide* .

            :param block_storage: Request to configure EBS Block Storage settings for your EKS Auto Mode cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-storageconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                storage_config_property = eks.CfnCluster.StorageConfigProperty(
                    block_storage=eks.CfnCluster.BlockStorageProperty(
                        enabled=False
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7c0cf4e49e1451505acf9a439adfa8e025be8470793be5a9598964122b667fa8)
                check_type(argname="argument block_storage", value=block_storage, expected_type=type_hints["block_storage"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if block_storage is not None:
                self._values["block_storage"] = block_storage

        @builtins.property
        def block_storage(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.BlockStorageProperty"]]:
            '''Request to configure EBS Block Storage settings for your EKS Auto Mode cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-storageconfig.html#cfn-eks-cluster-storageconfig-blockstorage
            '''
            result = self._values.get("block_storage")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnCluster.BlockStorageProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StorageConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.UpgradePolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"support_type": "supportType"},
    )
    class UpgradePolicyProperty:
        def __init__(
            self,
            *,
            support_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The support policy to use for the cluster.

            Extended support allows you to remain on specific Kubernetes versions for longer. Clusters in extended support have higher costs. The default value is ``EXTENDED`` . Use ``STANDARD`` to disable extended support.

            `Learn more about EKS Extended Support in the *Amazon EKS User Guide* . <https://docs.aws.amazon.com/eks/latest/userguide/extended-support-control.html>`_

            :param support_type: If the cluster is set to ``EXTENDED`` , it will enter extended support at the end of standard support. If the cluster is set to ``STANDARD`` , it will be automatically upgraded at the end of standard support. `Learn more about EKS Extended Support in the *Amazon EKS User Guide* . <https://docs.aws.amazon.com/eks/latest/userguide/extended-support-control.html>`_

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-upgradepolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                upgrade_policy_property = eks.CfnCluster.UpgradePolicyProperty(
                    support_type="supportType"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__86073c3bbbda3151a59b8c04b7d831693a2bba55bb8a817197ab0618c69197b6)
                check_type(argname="argument support_type", value=support_type, expected_type=type_hints["support_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if support_type is not None:
                self._values["support_type"] = support_type

        @builtins.property
        def support_type(self) -> typing.Optional[builtins.str]:
            '''If the cluster is set to ``EXTENDED`` , it will enter extended support at the end of standard support.

            If the cluster is set to ``STANDARD`` , it will be automatically upgraded at the end of standard support.

            `Learn more about EKS Extended Support in the *Amazon EKS User Guide* . <https://docs.aws.amazon.com/eks/latest/userguide/extended-support-control.html>`_

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-upgradepolicy.html#cfn-eks-cluster-upgradepolicy-supporttype
            '''
            result = self._values.get("support_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "UpgradePolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnCluster.ZonalShiftConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"enabled": "enabled"},
    )
    class ZonalShiftConfigProperty:
        def __init__(
            self,
            *,
            enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''The configuration for zonal shift for the cluster.

            :param enabled: If zonal shift is enabled, AWS configures zonal autoshift for the cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-zonalshiftconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                zonal_shift_config_property = eks.CfnCluster.ZonalShiftConfigProperty(
                    enabled=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__12178fb427a2491053da380c7def573901cbee28defad0982fa8c86d78adcec0)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if enabled is not None:
                self._values["enabled"] = enabled

        @builtins.property
        def enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''If zonal shift is enabled, AWS configures zonal autoshift for the cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-zonalshiftconfig.html#cfn-eks-cluster-zonalshiftconfig-enabled
            '''
            result = self._values.get("enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ZonalShiftConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.CfnClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "resources_vpc_config": "resourcesVpcConfig",
        "role_arn": "roleArn",
        "access_config": "accessConfig",
        "bootstrap_self_managed_addons": "bootstrapSelfManagedAddons",
        "compute_config": "computeConfig",
        "encryption_config": "encryptionConfig",
        "force": "force",
        "kubernetes_network_config": "kubernetesNetworkConfig",
        "logging": "logging",
        "name": "name",
        "outpost_config": "outpostConfig",
        "remote_network_config": "remoteNetworkConfig",
        "storage_config": "storageConfig",
        "tags": "tags",
        "upgrade_policy": "upgradePolicy",
        "version": "version",
        "zonal_shift_config": "zonalShiftConfig",
    },
)
class CfnClusterProps:
    def __init__(
        self,
        *,
        resources_vpc_config: typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.ResourcesVpcConfigProperty, typing.Dict[builtins.str, typing.Any]]],
        role_arn: builtins.str,
        access_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.AccessConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        bootstrap_self_managed_addons: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        compute_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.ComputeConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        encryption_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.EncryptionConfigProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        force: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        kubernetes_network_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.KubernetesNetworkConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        logging: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.LoggingProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        name: typing.Optional[builtins.str] = None,
        outpost_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.OutpostConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        remote_network_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.RemoteNetworkConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        storage_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.StorageConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        upgrade_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.UpgradePolicyProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        version: typing.Optional[builtins.str] = None,
        zonal_shift_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCluster.ZonalShiftConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnCluster``.

        :param resources_vpc_config: The VPC configuration that's used by the cluster control plane. Amazon EKS VPC resources have specific requirements to work properly with Kubernetes. For more information, see `Cluster VPC Considerations <https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html>`_ and `Cluster Security Group Considerations <https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html>`_ in the *Amazon EKS User Guide* . You must specify at least two subnets. You can specify up to five security groups, but we recommend that you use a dedicated security group for your cluster control plane.
        :param role_arn: The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf. For more information, see `Amazon EKS Service IAM Role <https://docs.aws.amazon.com/eks/latest/userguide/service_IAM_role.html>`_ in the **Amazon EKS User Guide** .
        :param access_config: The access configuration for the cluster.
        :param bootstrap_self_managed_addons: If you set this value to ``False`` when creating a cluster, the default networking add-ons will not be installed. The default networking addons include vpc-cni, coredns, and kube-proxy. Use this option when you plan to install third-party alternative add-ons or self-manage the default networking add-ons.
        :param compute_config: Indicates the current configuration of the compute capability on your EKS Auto Mode cluster. For example, if the capability is enabled or disabled. If the compute capability is enabled, EKS Auto Mode will create and delete EC2 Managed Instances in your AWS account. For more information, see EKS Auto Mode compute capability in the *Amazon EKS User Guide* .
        :param encryption_config: The encryption configuration for the cluster.
        :param force: Set this value to ``true`` to override upgrade-blocking readiness checks when updating a cluster. Default: - false
        :param kubernetes_network_config: The Kubernetes network configuration for the cluster.
        :param logging: The logging configuration for your cluster.
        :param name: The unique name to give to your cluster. The name can contain only alphanumeric characters (case-sensitive) and hyphens. It must start with an alphanumeric character and can't be longer than 100 characters. The name must be unique within the AWS Region and AWS account that you're creating the cluster in. Note that underscores can't be used in AWS CloudFormation .
        :param outpost_config: An object representing the configuration of your local Amazon EKS cluster on an AWS Outpost. This object isn't available for clusters on the AWS cloud.
        :param remote_network_config: The configuration in the cluster for EKS Hybrid Nodes. You can add, change, or remove this configuration after the cluster is created.
        :param storage_config: Indicates the current configuration of the block storage capability on your EKS Auto Mode cluster. For example, if the capability is enabled or disabled. If the block storage capability is enabled, EKS Auto Mode will create and delete EBS volumes in your AWS account. For more information, see EKS Auto Mode block storage capability in the *Amazon EKS User Guide* .
        :param tags: The metadata that you apply to the cluster to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Cluster tags don't propagate to any other resources associated with the cluster. .. epigraph:: You must have the ``eks:TagResource`` and ``eks:UntagResource`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.
        :param upgrade_policy: This value indicates if extended support is enabled or disabled for the cluster. `Learn more about EKS Extended Support in the *Amazon EKS User Guide* . <https://docs.aws.amazon.com/eks/latest/userguide/extended-support-control.html>`_
        :param version: The desired Kubernetes version for your cluster. If you don't specify a value here, the default version available in Amazon EKS is used. .. epigraph:: The default version might not be the latest version available.
        :param zonal_shift_config: The configuration for zonal shift for the cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            cfn_cluster_props = eks.CfnClusterProps(
                resources_vpc_config=eks.CfnCluster.ResourcesVpcConfigProperty(
                    subnet_ids=["subnetIds"],
            
                    # the properties below are optional
                    endpoint_private_access=False,
                    endpoint_public_access=False,
                    public_access_cidrs=["publicAccessCidrs"],
                    security_group_ids=["securityGroupIds"]
                ),
                role_arn="roleArn",
            
                # the properties below are optional
                access_config=eks.CfnCluster.AccessConfigProperty(
                    authentication_mode="authenticationMode",
                    bootstrap_cluster_creator_admin_permissions=False
                ),
                bootstrap_self_managed_addons=False,
                compute_config=eks.CfnCluster.ComputeConfigProperty(
                    enabled=False,
                    node_pools=["nodePools"],
                    node_role_arn="nodeRoleArn"
                ),
                encryption_config=[eks.CfnCluster.EncryptionConfigProperty(
                    provider=eks.CfnCluster.ProviderProperty(
                        key_arn="keyArn"
                    ),
                    resources=["resources"]
                )],
                force=False,
                kubernetes_network_config=eks.CfnCluster.KubernetesNetworkConfigProperty(
                    elastic_load_balancing=eks.CfnCluster.ElasticLoadBalancingProperty(
                        enabled=False
                    ),
                    ip_family="ipFamily",
                    service_ipv4_cidr="serviceIpv4Cidr",
                    service_ipv6_cidr="serviceIpv6Cidr"
                ),
                logging=eks.CfnCluster.LoggingProperty(
                    cluster_logging=eks.CfnCluster.ClusterLoggingProperty(
                        enabled_types=[eks.CfnCluster.LoggingTypeConfigProperty(
                            type="type"
                        )]
                    )
                ),
                name="name",
                outpost_config=eks.CfnCluster.OutpostConfigProperty(
                    control_plane_instance_type="controlPlaneInstanceType",
                    outpost_arns=["outpostArns"],
            
                    # the properties below are optional
                    control_plane_placement=eks.CfnCluster.ControlPlanePlacementProperty(
                        group_name="groupName"
                    )
                ),
                remote_network_config=eks.CfnCluster.RemoteNetworkConfigProperty(
                    remote_node_networks=[eks.CfnCluster.RemoteNodeNetworkProperty(
                        cidrs=["cidrs"]
                    )],
            
                    # the properties below are optional
                    remote_pod_networks=[eks.CfnCluster.RemotePodNetworkProperty(
                        cidrs=["cidrs"]
                    )]
                ),
                storage_config=eks.CfnCluster.StorageConfigProperty(
                    block_storage=eks.CfnCluster.BlockStorageProperty(
                        enabled=False
                    )
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                upgrade_policy=eks.CfnCluster.UpgradePolicyProperty(
                    support_type="supportType"
                ),
                version="version",
                zonal_shift_config=eks.CfnCluster.ZonalShiftConfigProperty(
                    enabled=False
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__270f142a59c249328ab174c5b0484cfdae6e3110ab52578dbe783d6f8a898e92)
            check_type(argname="argument resources_vpc_config", value=resources_vpc_config, expected_type=type_hints["resources_vpc_config"])
            check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
            check_type(argname="argument access_config", value=access_config, expected_type=type_hints["access_config"])
            check_type(argname="argument bootstrap_self_managed_addons", value=bootstrap_self_managed_addons, expected_type=type_hints["bootstrap_self_managed_addons"])
            check_type(argname="argument compute_config", value=compute_config, expected_type=type_hints["compute_config"])
            check_type(argname="argument encryption_config", value=encryption_config, expected_type=type_hints["encryption_config"])
            check_type(argname="argument force", value=force, expected_type=type_hints["force"])
            check_type(argname="argument kubernetes_network_config", value=kubernetes_network_config, expected_type=type_hints["kubernetes_network_config"])
            check_type(argname="argument logging", value=logging, expected_type=type_hints["logging"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument outpost_config", value=outpost_config, expected_type=type_hints["outpost_config"])
            check_type(argname="argument remote_network_config", value=remote_network_config, expected_type=type_hints["remote_network_config"])
            check_type(argname="argument storage_config", value=storage_config, expected_type=type_hints["storage_config"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument upgrade_policy", value=upgrade_policy, expected_type=type_hints["upgrade_policy"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            check_type(argname="argument zonal_shift_config", value=zonal_shift_config, expected_type=type_hints["zonal_shift_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "resources_vpc_config": resources_vpc_config,
            "role_arn": role_arn,
        }
        if access_config is not None:
            self._values["access_config"] = access_config
        if bootstrap_self_managed_addons is not None:
            self._values["bootstrap_self_managed_addons"] = bootstrap_self_managed_addons
        if compute_config is not None:
            self._values["compute_config"] = compute_config
        if encryption_config is not None:
            self._values["encryption_config"] = encryption_config
        if force is not None:
            self._values["force"] = force
        if kubernetes_network_config is not None:
            self._values["kubernetes_network_config"] = kubernetes_network_config
        if logging is not None:
            self._values["logging"] = logging
        if name is not None:
            self._values["name"] = name
        if outpost_config is not None:
            self._values["outpost_config"] = outpost_config
        if remote_network_config is not None:
            self._values["remote_network_config"] = remote_network_config
        if storage_config is not None:
            self._values["storage_config"] = storage_config
        if tags is not None:
            self._values["tags"] = tags
        if upgrade_policy is not None:
            self._values["upgrade_policy"] = upgrade_policy
        if version is not None:
            self._values["version"] = version
        if zonal_shift_config is not None:
            self._values["zonal_shift_config"] = zonal_shift_config

    @builtins.property
    def resources_vpc_config(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnCluster.ResourcesVpcConfigProperty]:
        '''The VPC configuration that's used by the cluster control plane.

        Amazon EKS VPC resources have specific requirements to work properly with Kubernetes. For more information, see `Cluster VPC Considerations <https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html>`_ and `Cluster Security Group Considerations <https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html>`_ in the *Amazon EKS User Guide* . You must specify at least two subnets. You can specify up to five security groups, but we recommend that you use a dedicated security group for your cluster control plane.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-resourcesvpcconfig
        '''
        result = self._values.get("resources_vpc_config")
        assert result is not None, "Required property 'resources_vpc_config' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnCluster.ResourcesVpcConfigProperty], result)

    @builtins.property
    def role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf.

        For more information, see `Amazon EKS Service IAM Role <https://docs.aws.amazon.com/eks/latest/userguide/service_IAM_role.html>`_ in the **Amazon EKS User Guide** .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-rolearn
        '''
        result = self._values.get("role_arn")
        assert result is not None, "Required property 'role_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.AccessConfigProperty]]:
        '''The access configuration for the cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-accessconfig
        '''
        result = self._values.get("access_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.AccessConfigProperty]], result)

    @builtins.property
    def bootstrap_self_managed_addons(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''If you set this value to ``False`` when creating a cluster, the default networking add-ons will not be installed.

        The default networking addons include vpc-cni, coredns, and kube-proxy.

        Use this option when you plan to install third-party alternative add-ons or self-manage the default networking add-ons.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-bootstrapselfmanagedaddons
        '''
        result = self._values.get("bootstrap_self_managed_addons")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def compute_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.ComputeConfigProperty]]:
        '''Indicates the current configuration of the compute capability on your EKS Auto Mode cluster.

        For example, if the capability is enabled or disabled. If the compute capability is enabled, EKS Auto Mode will create and delete EC2 Managed Instances in your AWS account. For more information, see EKS Auto Mode compute capability in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-computeconfig
        '''
        result = self._values.get("compute_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.ComputeConfigProperty]], result)

    @builtins.property
    def encryption_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnCluster.EncryptionConfigProperty]]]]:
        '''The encryption configuration for the cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-encryptionconfig
        '''
        result = self._values.get("encryption_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnCluster.EncryptionConfigProperty]]]], result)

    @builtins.property
    def force(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Set this value to ``true`` to override upgrade-blocking readiness checks when updating a cluster.

        :default: - false

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-force
        '''
        result = self._values.get("force")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def kubernetes_network_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.KubernetesNetworkConfigProperty]]:
        '''The Kubernetes network configuration for the cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-kubernetesnetworkconfig
        '''
        result = self._values.get("kubernetes_network_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.KubernetesNetworkConfigProperty]], result)

    @builtins.property
    def logging(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.LoggingProperty]]:
        '''The logging configuration for your cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-logging
        '''
        result = self._values.get("logging")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.LoggingProperty]], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''The unique name to give to your cluster.

        The name can contain only alphanumeric characters (case-sensitive) and hyphens. It must start with an alphanumeric character and can't be longer than 100 characters. The name must be unique within the AWS Region and AWS account that you're creating the cluster in. Note that underscores can't be used in AWS CloudFormation .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outpost_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.OutpostConfigProperty]]:
        '''An object representing the configuration of your local Amazon EKS cluster on an AWS Outpost.

        This object isn't available for clusters on the AWS cloud.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-outpostconfig
        '''
        result = self._values.get("outpost_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.OutpostConfigProperty]], result)

    @builtins.property
    def remote_network_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.RemoteNetworkConfigProperty]]:
        '''The configuration in the cluster for EKS Hybrid Nodes.

        You can add, change, or remove this configuration after the cluster is created.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-remotenetworkconfig
        '''
        result = self._values.get("remote_network_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.RemoteNetworkConfigProperty]], result)

    @builtins.property
    def storage_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.StorageConfigProperty]]:
        '''Indicates the current configuration of the block storage capability on your EKS Auto Mode cluster.

        For example, if the capability is enabled or disabled. If the block storage capability is enabled, EKS Auto Mode will create and delete EBS volumes in your AWS account. For more information, see EKS Auto Mode block storage capability in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-storageconfig
        '''
        result = self._values.get("storage_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.StorageConfigProperty]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The metadata that you apply to the cluster to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Cluster tags don't propagate to any other resources associated with the cluster.
        .. epigraph::

           You must have the ``eks:TagResource`` and ``eks:UntagResource`` permissions for your `IAM principal <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html>`_ to manage the AWS CloudFormation stack. If you don't have these permissions, there might be unexpected behavior with stack-level tags propagating to the resource during resource creation and update.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def upgrade_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.UpgradePolicyProperty]]:
        '''This value indicates if extended support is enabled or disabled for the cluster.

        `Learn more about EKS Extended Support in the *Amazon EKS User Guide* . <https://docs.aws.amazon.com/eks/latest/userguide/extended-support-control.html>`_

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-upgradepolicy
        '''
        result = self._values.get("upgrade_policy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.UpgradePolicyProperty]], result)

    @builtins.property
    def version(self) -> typing.Optional[builtins.str]:
        '''The desired Kubernetes version for your cluster.

        If you don't specify a value here, the default version available in Amazon EKS is used.
        .. epigraph::

           The default version might not be the latest version available.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-version
        '''
        result = self._values.get("version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def zonal_shift_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.ZonalShiftConfigProperty]]:
        '''The configuration for zonal shift for the cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-cluster.html#cfn-eks-cluster-zonalshiftconfig
        '''
        result = self._values.get("zonal_shift_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnCluster.ZonalShiftConfigProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnFargateProfile(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.CfnFargateProfile",
):
    '''Creates an AWS Fargate profile for your Amazon EKS cluster.

    You must have at least one Fargate profile in a cluster to be able to run pods on Fargate.

    The Fargate profile allows an administrator to declare which pods run on Fargate and specify which pods run on which Fargate profile. This declaration is done through the profile's selectors. Each profile can have up to five selectors that contain a namespace and labels. A namespace is required for every selector. The label field consists of multiple optional key-value pairs. Pods that match the selectors are scheduled on Fargate. If a to-be-scheduled pod matches any of the selectors in the Fargate profile, then that pod is run on Fargate.

    When you create a Fargate profile, you must specify a pod execution role to use with the pods that are scheduled with the profile. This role is added to the cluster's Kubernetes `Role Based Access Control <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/access-authn-authz/rbac/>`_ (RBAC) for authorization so that the ``kubelet`` that is running on the Fargate infrastructure can register with your Amazon EKS cluster so that it can appear in your cluster as a node. The pod execution role also provides IAM permissions to the Fargate infrastructure to allow read access to Amazon ECR image repositories. For more information, see `Pod Execution Role <https://docs.aws.amazon.com/eks/latest/userguide/pod-execution-role.html>`_ in the *Amazon EKS User Guide* .

    Fargate profiles are immutable. However, you can create a new updated profile to replace an existing profile and then delete the original after the updated profile has finished creating.

    If any Fargate profiles in a cluster are in the ``DELETING`` status, you must wait for that Fargate profile to finish deleting before you can create any other profiles in that cluster.

    For more information, see `AWS Fargate profile <https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html>`_ in the *Amazon EKS User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-fargateprofile.html
    :cloudformationResource: AWS::EKS::FargateProfile
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        cfn_fargate_profile = eks.CfnFargateProfile(self, "MyCfnFargateProfile",
            cluster_name="clusterName",
            pod_execution_role_arn="podExecutionRoleArn",
            selectors=[eks.CfnFargateProfile.SelectorProperty(
                namespace="namespace",
        
                # the properties below are optional
                labels=[eks.CfnFargateProfile.LabelProperty(
                    key="key",
                    value="value"
                )]
            )],
        
            # the properties below are optional
            fargate_profile_name="fargateProfileName",
            subnets=["subnets"],
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_name: builtins.str,
        pod_execution_role_arn: builtins.str,
        selectors: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFargateProfile.SelectorProperty", typing.Dict[builtins.str, typing.Any]]]]],
        fargate_profile_name: typing.Optional[builtins.str] = None,
        subnets: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param cluster_name: The name of your cluster.
        :param pod_execution_role_arn: The Amazon Resource Name (ARN) of the ``Pod`` execution role to use for a ``Pod`` that matches the selectors in the Fargate profile. The ``Pod`` execution role allows Fargate infrastructure to register with your cluster as a node, and it provides read access to Amazon ECR image repositories. For more information, see ```Pod`` execution role <https://docs.aws.amazon.com/eks/latest/userguide/pod-execution-role.html>`_ in the *Amazon EKS User Guide* .
        :param selectors: The selectors to match for a ``Pod`` to use this Fargate profile. Each selector must have an associated Kubernetes ``namespace`` . Optionally, you can also specify ``labels`` for a ``namespace`` . You may specify up to five selectors in a Fargate profile.
        :param fargate_profile_name: The name of the Fargate profile.
        :param subnets: The IDs of subnets to launch a ``Pod`` into. A ``Pod`` running on Fargate isn't assigned a public IP address, so only private subnets (with no direct route to an Internet Gateway) are accepted for this parameter.
        :param tags: Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d74e3378581d3898ad60f22c46414b34a44ce82bd54d09c56466afda96989a2f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnFargateProfileProps(
            cluster_name=cluster_name,
            pod_execution_role_arn=pod_execution_role_arn,
            selectors=selectors,
            fargate_profile_name=fargate_profile_name,
            subnets=subnets,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__43ebc50d6c5b13856e87922886b34c19885bddbafafc65bf42a5e790bc129b85)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5a4354626a4d07c7dcafd70560f4d3e9b370a6b56025c13cc1f40a27c30d0b59)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The ARN of the cluster, such as ``arn:aws:eks:us-west-2:666666666666:fargateprofile/myCluster/myFargateProfile/1cb1a11a-1dc1-1d11-cf11-1111f11fa111`` .

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''The name of your cluster.'''
        return typing.cast(builtins.str, jsii.get(self, "clusterName"))

    @cluster_name.setter
    def cluster_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__67c0af53421929d46be84728da3b0c110c22a4c1524b69d145f4b40578d660a1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clusterName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="podExecutionRoleArn")
    def pod_execution_role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the ``Pod`` execution role to use for a ``Pod`` that matches the selectors in the Fargate profile.'''
        return typing.cast(builtins.str, jsii.get(self, "podExecutionRoleArn"))

    @pod_execution_role_arn.setter
    def pod_execution_role_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0898914ea4216cb0612ddbbd8ae321af052b066ed58f2de99ea8cb26e78b3564)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "podExecutionRoleArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="selectors")
    def selectors(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFargateProfile.SelectorProperty"]]]:
        '''The selectors to match for a ``Pod`` to use this Fargate profile.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFargateProfile.SelectorProperty"]]], jsii.get(self, "selectors"))

    @selectors.setter
    def selectors(
        self,
        value: typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFargateProfile.SelectorProperty"]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3467e37d13629d1721d8e96f13c9b469fdaf5eb821e204838ae461e856021d34)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "selectors", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="fargateProfileName")
    def fargate_profile_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Fargate profile.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "fargateProfileName"))

    @fargate_profile_name.setter
    def fargate_profile_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__52c64967dbf24e58f449238fa9259bd8892dadc64bcf87d5d998efb9efbf9596)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "fargateProfileName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="subnets")
    def subnets(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IDs of subnets to launch a ``Pod`` into.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "subnets"))

    @subnets.setter
    def subnets(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__03bcf21600a97aafc8e6d8376599c773b37b4c36c562644eaaa8ec11bee5dab6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "subnets", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Metadata that assists with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__93fe5db83ef1635270c7973406a7644b19576f909eda3840d06be03d3182d9aa)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnFargateProfile.LabelProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class LabelProperty:
        def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
            '''A key-value pair.

            :param key: Enter a key.
            :param value: Enter a value.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-fargateprofile-label.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                label_property = eks.CfnFargateProfile.LabelProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__77b2ddf382f74aae66c244100dec19bdba0ae078778fe1aef1ed30e8af4b93cc)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''Enter a key.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-fargateprofile-label.html#cfn-eks-fargateprofile-label-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''Enter a value.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-fargateprofile-label.html#cfn-eks-fargateprofile-label-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LabelProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnFargateProfile.SelectorProperty",
        jsii_struct_bases=[],
        name_mapping={"namespace": "namespace", "labels": "labels"},
    )
    class SelectorProperty:
        def __init__(
            self,
            *,
            namespace: builtins.str,
            labels: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFargateProfile.LabelProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''An object representing an AWS Fargate profile selector.

            :param namespace: The Kubernetes ``namespace`` that the selector should match.
            :param labels: The Kubernetes labels that the selector should match. A pod must contain all of the labels that are specified in the selector for it to be considered a match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-fargateprofile-selector.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                selector_property = eks.CfnFargateProfile.SelectorProperty(
                    namespace="namespace",
                
                    # the properties below are optional
                    labels=[eks.CfnFargateProfile.LabelProperty(
                        key="key",
                        value="value"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c3e5f35d3f8424d880666a42b64297747e45b40a9eb095d38dd451d997ff0150)
                check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
                check_type(argname="argument labels", value=labels, expected_type=type_hints["labels"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "namespace": namespace,
            }
            if labels is not None:
                self._values["labels"] = labels

        @builtins.property
        def namespace(self) -> builtins.str:
            '''The Kubernetes ``namespace`` that the selector should match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-fargateprofile-selector.html#cfn-eks-fargateprofile-selector-namespace
            '''
            result = self._values.get("namespace")
            assert result is not None, "Required property 'namespace' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def labels(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFargateProfile.LabelProperty"]]]]:
            '''The Kubernetes labels that the selector should match.

            A pod must contain all of the labels that are specified in the selector for it to be considered a match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-fargateprofile-selector.html#cfn-eks-fargateprofile-selector-labels
            '''
            result = self._values.get("labels")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFargateProfile.LabelProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SelectorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.CfnFargateProfileProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster_name": "clusterName",
        "pod_execution_role_arn": "podExecutionRoleArn",
        "selectors": "selectors",
        "fargate_profile_name": "fargateProfileName",
        "subnets": "subnets",
        "tags": "tags",
    },
)
class CfnFargateProfileProps:
    def __init__(
        self,
        *,
        cluster_name: builtins.str,
        pod_execution_role_arn: builtins.str,
        selectors: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFargateProfile.SelectorProperty, typing.Dict[builtins.str, typing.Any]]]]],
        fargate_profile_name: typing.Optional[builtins.str] = None,
        subnets: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnFargateProfile``.

        :param cluster_name: The name of your cluster.
        :param pod_execution_role_arn: The Amazon Resource Name (ARN) of the ``Pod`` execution role to use for a ``Pod`` that matches the selectors in the Fargate profile. The ``Pod`` execution role allows Fargate infrastructure to register with your cluster as a node, and it provides read access to Amazon ECR image repositories. For more information, see ```Pod`` execution role <https://docs.aws.amazon.com/eks/latest/userguide/pod-execution-role.html>`_ in the *Amazon EKS User Guide* .
        :param selectors: The selectors to match for a ``Pod`` to use this Fargate profile. Each selector must have an associated Kubernetes ``namespace`` . Optionally, you can also specify ``labels`` for a ``namespace`` . You may specify up to five selectors in a Fargate profile.
        :param fargate_profile_name: The name of the Fargate profile.
        :param subnets: The IDs of subnets to launch a ``Pod`` into. A ``Pod`` running on Fargate isn't assigned a public IP address, so only private subnets (with no direct route to an Internet Gateway) are accepted for this parameter.
        :param tags: Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-fargateprofile.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            cfn_fargate_profile_props = eks.CfnFargateProfileProps(
                cluster_name="clusterName",
                pod_execution_role_arn="podExecutionRoleArn",
                selectors=[eks.CfnFargateProfile.SelectorProperty(
                    namespace="namespace",
            
                    # the properties below are optional
                    labels=[eks.CfnFargateProfile.LabelProperty(
                        key="key",
                        value="value"
                    )]
                )],
            
                # the properties below are optional
                fargate_profile_name="fargateProfileName",
                subnets=["subnets"],
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3cc35ff12592fd31c30b9207ed088e39d5bfa0a1d226cce36d8a65809e6ce576)
            check_type(argname="argument cluster_name", value=cluster_name, expected_type=type_hints["cluster_name"])
            check_type(argname="argument pod_execution_role_arn", value=pod_execution_role_arn, expected_type=type_hints["pod_execution_role_arn"])
            check_type(argname="argument selectors", value=selectors, expected_type=type_hints["selectors"])
            check_type(argname="argument fargate_profile_name", value=fargate_profile_name, expected_type=type_hints["fargate_profile_name"])
            check_type(argname="argument subnets", value=subnets, expected_type=type_hints["subnets"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_name": cluster_name,
            "pod_execution_role_arn": pod_execution_role_arn,
            "selectors": selectors,
        }
        if fargate_profile_name is not None:
            self._values["fargate_profile_name"] = fargate_profile_name
        if subnets is not None:
            self._values["subnets"] = subnets
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def cluster_name(self) -> builtins.str:
        '''The name of your cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-fargateprofile.html#cfn-eks-fargateprofile-clustername
        '''
        result = self._values.get("cluster_name")
        assert result is not None, "Required property 'cluster_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def pod_execution_role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the ``Pod`` execution role to use for a ``Pod`` that matches the selectors in the Fargate profile.

        The ``Pod`` execution role allows Fargate infrastructure to register with your cluster as a node, and it provides read access to Amazon ECR image repositories. For more information, see ```Pod`` execution role <https://docs.aws.amazon.com/eks/latest/userguide/pod-execution-role.html>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-fargateprofile.html#cfn-eks-fargateprofile-podexecutionrolearn
        '''
        result = self._values.get("pod_execution_role_arn")
        assert result is not None, "Required property 'pod_execution_role_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def selectors(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnFargateProfile.SelectorProperty]]]:
        '''The selectors to match for a ``Pod`` to use this Fargate profile.

        Each selector must have an associated Kubernetes ``namespace`` . Optionally, you can also specify ``labels`` for a ``namespace`` . You may specify up to five selectors in a Fargate profile.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-fargateprofile.html#cfn-eks-fargateprofile-selectors
        '''
        result = self._values.get("selectors")
        assert result is not None, "Required property 'selectors' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnFargateProfile.SelectorProperty]]], result)

    @builtins.property
    def fargate_profile_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Fargate profile.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-fargateprofile.html#cfn-eks-fargateprofile-fargateprofilename
        '''
        result = self._values.get("fargate_profile_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def subnets(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IDs of subnets to launch a ``Pod`` into.

        A ``Pod`` running on Fargate isn't assigned a public IP address, so only private subnets (with no direct route to an Internet Gateway) are accepted for this parameter.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-fargateprofile.html#cfn-eks-fargateprofile-subnets
        '''
        result = self._values.get("subnets")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Metadata that assists with categorization and organization.

        Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-fargateprofile.html#cfn-eks-fargateprofile-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnFargateProfileProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnIdentityProviderConfig(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.CfnIdentityProviderConfig",
):
    '''Associates an identity provider configuration to a cluster.

    If you want to authenticate identities using an identity provider, you can create an identity provider configuration and associate it to your cluster. After configuring authentication to your cluster you can create Kubernetes ``Role`` and ``ClusterRole`` objects, assign permissions to them, and then bind them to the identities using Kubernetes ``RoleBinding`` and ``ClusterRoleBinding`` objects. For more information see `Using RBAC Authorization <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/access-authn-authz/rbac/>`_ in the Kubernetes documentation.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-identityproviderconfig.html
    :cloudformationResource: AWS::EKS::IdentityProviderConfig
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        cfn_identity_provider_config = eks.CfnIdentityProviderConfig(self, "MyCfnIdentityProviderConfig",
            cluster_name="clusterName",
            type="type",
        
            # the properties below are optional
            identity_provider_config_name="identityProviderConfigName",
            oidc=eks.CfnIdentityProviderConfig.OidcIdentityProviderConfigProperty(
                client_id="clientId",
                issuer_url="issuerUrl",
        
                # the properties below are optional
                groups_claim="groupsClaim",
                groups_prefix="groupsPrefix",
                required_claims=[eks.CfnIdentityProviderConfig.RequiredClaimProperty(
                    key="key",
                    value="value"
                )],
                username_claim="usernameClaim",
                username_prefix="usernamePrefix"
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_name: builtins.str,
        type: builtins.str,
        identity_provider_config_name: typing.Optional[builtins.str] = None,
        oidc: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnIdentityProviderConfig.OidcIdentityProviderConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param cluster_name: The name of your cluster.
        :param type: The type of the identity provider configuration. The only type available is ``oidc`` .
        :param identity_provider_config_name: The name of the configuration.
        :param oidc: An object representing an OpenID Connect (OIDC) identity provider configuration.
        :param tags: Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5aefb3bc5a12905f1fee3f10f2a7ba558d09704a51fff92ba5d790fc1bf90471)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnIdentityProviderConfigProps(
            cluster_name=cluster_name,
            type=type,
            identity_provider_config_name=identity_provider_config_name,
            oidc=oidc,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aca6289219889a58359f4ccacfd4a7e605c15ea354401b67c76587c445a36e88)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0b7ddfc422f59e66988481abb9f8011d3ac476532dacce903d2406e4ca758c04)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrIdentityProviderConfigArn")
    def attr_identity_provider_config_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) associated with the identity provider config.

        :cloudformationAttribute: IdentityProviderConfigArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIdentityProviderConfigArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''The name of your cluster.'''
        return typing.cast(builtins.str, jsii.get(self, "clusterName"))

    @cluster_name.setter
    def cluster_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6f1edfdd1a019d81346000b97e8efa1ff85d5a87672d59b6da96ffe29a83bebf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clusterName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''The type of the identity provider configuration.'''
        return typing.cast(builtins.str, jsii.get(self, "type"))

    @type.setter
    def type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ee09b440e3a44fe78f165402158b981b9ea03938b33dcb754948e432f6231789)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "type", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="identityProviderConfigName")
    def identity_provider_config_name(self) -> typing.Optional[builtins.str]:
        '''The name of the configuration.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "identityProviderConfigName"))

    @identity_provider_config_name.setter
    def identity_provider_config_name(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__61ed3819d1ad5446bf8deef67e4587b9fb5317ad3756afebd93604eb9b6beb96)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identityProviderConfigName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="oidc")
    def oidc(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIdentityProviderConfig.OidcIdentityProviderConfigProperty"]]:
        '''An object representing an OpenID Connect (OIDC) identity provider configuration.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIdentityProviderConfig.OidcIdentityProviderConfigProperty"]], jsii.get(self, "oidc"))

    @oidc.setter
    def oidc(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIdentityProviderConfig.OidcIdentityProviderConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3cbe654a8630cea4fa1389f38c348cf1618c501d058aa09878d216f3bc8d816)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "oidc", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Metadata that assists with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__67379d77a7e93ae2068934f518398b760115c19c314ddb0ce7aff1f3c0dfdb24)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnIdentityProviderConfig.OidcIdentityProviderConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "client_id": "clientId",
            "issuer_url": "issuerUrl",
            "groups_claim": "groupsClaim",
            "groups_prefix": "groupsPrefix",
            "required_claims": "requiredClaims",
            "username_claim": "usernameClaim",
            "username_prefix": "usernamePrefix",
        },
    )
    class OidcIdentityProviderConfigProperty:
        def __init__(
            self,
            *,
            client_id: builtins.str,
            issuer_url: builtins.str,
            groups_claim: typing.Optional[builtins.str] = None,
            groups_prefix: typing.Optional[builtins.str] = None,
            required_claims: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnIdentityProviderConfig.RequiredClaimProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            username_claim: typing.Optional[builtins.str] = None,
            username_prefix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the configuration for an OpenID Connect (OIDC) identity provider.

            :param client_id: This is also known as *audience* . The ID of the client application that makes authentication requests to the OIDC identity provider.
            :param issuer_url: The URL of the OIDC identity provider that allows the API server to discover public signing keys for verifying tokens.
            :param groups_claim: The JSON web token (JWT) claim that the provider uses to return your groups.
            :param groups_prefix: The prefix that is prepended to group claims to prevent clashes with existing names (such as ``system:`` groups). For example, the value ``oidc:`` creates group names like ``oidc:engineering`` and ``oidc:infra`` . The prefix can't contain ``system:``
            :param required_claims: The key-value pairs that describe required claims in the identity token. If set, each claim is verified to be present in the token with a matching value.
            :param username_claim: The JSON Web token (JWT) claim that is used as the username.
            :param username_prefix: The prefix that is prepended to username claims to prevent clashes with existing names. The prefix can't contain ``system:``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-identityproviderconfig-oidcidentityproviderconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                oidc_identity_provider_config_property = eks.CfnIdentityProviderConfig.OidcIdentityProviderConfigProperty(
                    client_id="clientId",
                    issuer_url="issuerUrl",
                
                    # the properties below are optional
                    groups_claim="groupsClaim",
                    groups_prefix="groupsPrefix",
                    required_claims=[eks.CfnIdentityProviderConfig.RequiredClaimProperty(
                        key="key",
                        value="value"
                    )],
                    username_claim="usernameClaim",
                    username_prefix="usernamePrefix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2bc29678a953d129714a2e096b4d03235397c76d8e042af2d7d2f6566901e4ea)
                check_type(argname="argument client_id", value=client_id, expected_type=type_hints["client_id"])
                check_type(argname="argument issuer_url", value=issuer_url, expected_type=type_hints["issuer_url"])
                check_type(argname="argument groups_claim", value=groups_claim, expected_type=type_hints["groups_claim"])
                check_type(argname="argument groups_prefix", value=groups_prefix, expected_type=type_hints["groups_prefix"])
                check_type(argname="argument required_claims", value=required_claims, expected_type=type_hints["required_claims"])
                check_type(argname="argument username_claim", value=username_claim, expected_type=type_hints["username_claim"])
                check_type(argname="argument username_prefix", value=username_prefix, expected_type=type_hints["username_prefix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "client_id": client_id,
                "issuer_url": issuer_url,
            }
            if groups_claim is not None:
                self._values["groups_claim"] = groups_claim
            if groups_prefix is not None:
                self._values["groups_prefix"] = groups_prefix
            if required_claims is not None:
                self._values["required_claims"] = required_claims
            if username_claim is not None:
                self._values["username_claim"] = username_claim
            if username_prefix is not None:
                self._values["username_prefix"] = username_prefix

        @builtins.property
        def client_id(self) -> builtins.str:
            '''This is also known as *audience* .

            The ID of the client application that makes authentication requests to the OIDC identity provider.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-identityproviderconfig-oidcidentityproviderconfig.html#cfn-eks-identityproviderconfig-oidcidentityproviderconfig-clientid
            '''
            result = self._values.get("client_id")
            assert result is not None, "Required property 'client_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def issuer_url(self) -> builtins.str:
            '''The URL of the OIDC identity provider that allows the API server to discover public signing keys for verifying tokens.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-identityproviderconfig-oidcidentityproviderconfig.html#cfn-eks-identityproviderconfig-oidcidentityproviderconfig-issuerurl
            '''
            result = self._values.get("issuer_url")
            assert result is not None, "Required property 'issuer_url' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def groups_claim(self) -> typing.Optional[builtins.str]:
            '''The JSON web token (JWT) claim that the provider uses to return your groups.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-identityproviderconfig-oidcidentityproviderconfig.html#cfn-eks-identityproviderconfig-oidcidentityproviderconfig-groupsclaim
            '''
            result = self._values.get("groups_claim")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def groups_prefix(self) -> typing.Optional[builtins.str]:
            '''The prefix that is prepended to group claims to prevent clashes with existing names (such as ``system:`` groups).

            For example, the value ``oidc:`` creates group names like ``oidc:engineering`` and ``oidc:infra`` . The prefix can't contain ``system:``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-identityproviderconfig-oidcidentityproviderconfig.html#cfn-eks-identityproviderconfig-oidcidentityproviderconfig-groupsprefix
            '''
            result = self._values.get("groups_prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def required_claims(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIdentityProviderConfig.RequiredClaimProperty"]]]]:
            '''The key-value pairs that describe required claims in the identity token.

            If set, each claim is verified to be present in the token with a matching value.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-identityproviderconfig-oidcidentityproviderconfig.html#cfn-eks-identityproviderconfig-oidcidentityproviderconfig-requiredclaims
            '''
            result = self._values.get("required_claims")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIdentityProviderConfig.RequiredClaimProperty"]]]], result)

        @builtins.property
        def username_claim(self) -> typing.Optional[builtins.str]:
            '''The JSON Web token (JWT) claim that is used as the username.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-identityproviderconfig-oidcidentityproviderconfig.html#cfn-eks-identityproviderconfig-oidcidentityproviderconfig-usernameclaim
            '''
            result = self._values.get("username_claim")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def username_prefix(self) -> typing.Optional[builtins.str]:
            '''The prefix that is prepended to username claims to prevent clashes with existing names.

            The prefix can't contain ``system:``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-identityproviderconfig-oidcidentityproviderconfig.html#cfn-eks-identityproviderconfig-oidcidentityproviderconfig-usernameprefix
            '''
            result = self._values.get("username_prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OidcIdentityProviderConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnIdentityProviderConfig.RequiredClaimProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class RequiredClaimProperty:
        def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
            '''A key-value pair that describes a required claim in the identity token.

            If set, each claim is verified to be present in the token with a matching value.

            :param key: The key to match from the token.
            :param value: The value for the key from the token.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-identityproviderconfig-requiredclaim.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                required_claim_property = eks.CfnIdentityProviderConfig.RequiredClaimProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f273337cc0201e68cefdd07b19f02c7126d79147068de4f5c2163d43a9266227)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''The key to match from the token.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-identityproviderconfig-requiredclaim.html#cfn-eks-identityproviderconfig-requiredclaim-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The value for the key from the token.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-identityproviderconfig-requiredclaim.html#cfn-eks-identityproviderconfig-requiredclaim-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RequiredClaimProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.CfnIdentityProviderConfigProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster_name": "clusterName",
        "type": "type",
        "identity_provider_config_name": "identityProviderConfigName",
        "oidc": "oidc",
        "tags": "tags",
    },
)
class CfnIdentityProviderConfigProps:
    def __init__(
        self,
        *,
        cluster_name: builtins.str,
        type: builtins.str,
        identity_provider_config_name: typing.Optional[builtins.str] = None,
        oidc: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnIdentityProviderConfig.OidcIdentityProviderConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnIdentityProviderConfig``.

        :param cluster_name: The name of your cluster.
        :param type: The type of the identity provider configuration. The only type available is ``oidc`` .
        :param identity_provider_config_name: The name of the configuration.
        :param oidc: An object representing an OpenID Connect (OIDC) identity provider configuration.
        :param tags: Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-identityproviderconfig.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            cfn_identity_provider_config_props = eks.CfnIdentityProviderConfigProps(
                cluster_name="clusterName",
                type="type",
            
                # the properties below are optional
                identity_provider_config_name="identityProviderConfigName",
                oidc=eks.CfnIdentityProviderConfig.OidcIdentityProviderConfigProperty(
                    client_id="clientId",
                    issuer_url="issuerUrl",
            
                    # the properties below are optional
                    groups_claim="groupsClaim",
                    groups_prefix="groupsPrefix",
                    required_claims=[eks.CfnIdentityProviderConfig.RequiredClaimProperty(
                        key="key",
                        value="value"
                    )],
                    username_claim="usernameClaim",
                    username_prefix="usernamePrefix"
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__28f8995e79be097e3130f01d37037f0936541b862af435d696534c628204a03f)
            check_type(argname="argument cluster_name", value=cluster_name, expected_type=type_hints["cluster_name"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument identity_provider_config_name", value=identity_provider_config_name, expected_type=type_hints["identity_provider_config_name"])
            check_type(argname="argument oidc", value=oidc, expected_type=type_hints["oidc"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_name": cluster_name,
            "type": type,
        }
        if identity_provider_config_name is not None:
            self._values["identity_provider_config_name"] = identity_provider_config_name
        if oidc is not None:
            self._values["oidc"] = oidc
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def cluster_name(self) -> builtins.str:
        '''The name of your cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-identityproviderconfig.html#cfn-eks-identityproviderconfig-clustername
        '''
        result = self._values.get("cluster_name")
        assert result is not None, "Required property 'cluster_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type(self) -> builtins.str:
        '''The type of the identity provider configuration.

        The only type available is ``oidc`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-identityproviderconfig.html#cfn-eks-identityproviderconfig-type
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def identity_provider_config_name(self) -> typing.Optional[builtins.str]:
        '''The name of the configuration.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-identityproviderconfig.html#cfn-eks-identityproviderconfig-identityproviderconfigname
        '''
        result = self._values.get("identity_provider_config_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def oidc(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnIdentityProviderConfig.OidcIdentityProviderConfigProperty]]:
        '''An object representing an OpenID Connect (OIDC) identity provider configuration.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-identityproviderconfig.html#cfn-eks-identityproviderconfig-oidc
        '''
        result = self._values.get("oidc")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnIdentityProviderConfig.OidcIdentityProviderConfigProperty]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Metadata that assists with categorization and organization.

        Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-identityproviderconfig.html#cfn-eks-identityproviderconfig-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnIdentityProviderConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnNodegroup(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.CfnNodegroup",
):
    '''Creates a managed node group for an Amazon EKS cluster.

    You can only create a node group for your cluster that is equal to the current Kubernetes version for the cluster. All node groups are created with the latest AMI release version for the respective minor Kubernetes version of the cluster, unless you deploy a custom AMI using a launch template.

    For later updates, you will only be able to update a node group using a launch template only if it was originally deployed with a launch template. Additionally, the launch template ID or name must match what was used when the node group was created. You can update the launch template version with necessary changes. For more information about using launch templates, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ .

    An Amazon EKS managed node group is an Amazon EC2 Auto Scaling group and associated Amazon EC2 instances that are managed by AWS for an Amazon EKS cluster. For more information, see `Managed node groups <https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html>`_ in the *Amazon EKS User Guide* .
    .. epigraph::

       Windows AMI types are only supported for commercial AWS Regions that support Windows on Amazon EKS.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html
    :cloudformationResource: AWS::EKS::Nodegroup
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        cfn_nodegroup = eks.CfnNodegroup(self, "MyCfnNodegroup",
            cluster_name="clusterName",
            node_role="nodeRole",
            subnets=["subnets"],
        
            # the properties below are optional
            ami_type="amiType",
            capacity_type="capacityType",
            disk_size=123,
            force_update_enabled=False,
            instance_types=["instanceTypes"],
            labels={
                "labels_key": "labels"
            },
            launch_template=eks.CfnNodegroup.LaunchTemplateSpecificationProperty(
                id="id",
                name="name",
                version="version"
            ),
            nodegroup_name="nodegroupName",
            node_repair_config=eks.CfnNodegroup.NodeRepairConfigProperty(
                enabled=False
            ),
            release_version="releaseVersion",
            remote_access=eks.CfnNodegroup.RemoteAccessProperty(
                ec2_ssh_key="ec2SshKey",
        
                # the properties below are optional
                source_security_groups=["sourceSecurityGroups"]
            ),
            scaling_config=eks.CfnNodegroup.ScalingConfigProperty(
                desired_size=123,
                max_size=123,
                min_size=123
            ),
            tags={
                "tags_key": "tags"
            },
            taints=[eks.CfnNodegroup.TaintProperty(
                effect="effect",
                key="key",
                value="value"
            )],
            update_config=eks.CfnNodegroup.UpdateConfigProperty(
                max_unavailable=123,
                max_unavailable_percentage=123,
                update_strategy="updateStrategy"
            ),
            version="version"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_name: builtins.str,
        node_role: builtins.str,
        subnets: typing.Sequence[builtins.str],
        ami_type: typing.Optional[builtins.str] = None,
        capacity_type: typing.Optional[builtins.str] = None,
        disk_size: typing.Optional[jsii.Number] = None,
        force_update_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        instance_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        labels: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        launch_template: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnNodegroup.LaunchTemplateSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        nodegroup_name: typing.Optional[builtins.str] = None,
        node_repair_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnNodegroup.NodeRepairConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        release_version: typing.Optional[builtins.str] = None,
        remote_access: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnNodegroup.RemoteAccessProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        scaling_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnNodegroup.ScalingConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        taints: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnNodegroup.TaintProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        update_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnNodegroup.UpdateConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        version: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param cluster_name: The name of your cluster.
        :param node_role: The Amazon Resource Name (ARN) of the IAM role to associate with your node group. The Amazon EKS worker node ``kubelet`` daemon makes calls to AWS APIs on your behalf. Nodes receive permissions for these API calls through an IAM instance profile and associated policies. Before you can launch nodes and register them into a cluster, you must create an IAM role for those nodes to use when they are launched. For more information, see `Amazon EKS node IAM role <https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html>`_ in the **Amazon EKS User Guide** . If you specify ``launchTemplate`` , then don't specify ``[IamInstanceProfile](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_IamInstanceProfile.html)`` in your launch template, or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param subnets: The subnets to use for the Auto Scaling group that is created for your node group. If you specify ``launchTemplate`` , then don't specify ``[SubnetId](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html)`` in your launch template, or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param ami_type: The AMI type for your node group. If you specify ``launchTemplate`` , and your launch template uses a custom AMI, then don't specify ``amiType`` , or the node group deployment will fail. If your launch template uses a Windows custom AMI, then add ``eks:kube-proxy-windows`` to your Windows nodes ``rolearn`` in the ``aws-auth`` ``ConfigMap`` . For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param capacity_type: The capacity type of your managed node group.
        :param disk_size: The root device disk size (in GiB) for your node group instances. The default disk size is 20 GiB for Linux and Bottlerocket. The default disk size is 50 GiB for Windows. If you specify ``launchTemplate`` , then don't specify ``diskSize`` , or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param force_update_enabled: Force the update if any ``Pod`` on the existing node group can't be drained due to a ``Pod`` disruption budget issue. If an update fails because all Pods can't be drained, you can force the update after it fails to terminate the old node whether or not any ``Pod`` is running on the node. Default: - false
        :param instance_types: Specify the instance types for a node group. If you specify a GPU instance type, make sure to also specify an applicable GPU AMI type with the ``amiType`` parameter. If you specify ``launchTemplate`` , then you can specify zero or one instance type in your launch template *or* you can specify 0-20 instance types for ``instanceTypes`` . If however, you specify an instance type in your launch template *and* specify any ``instanceTypes`` , the node group deployment will fail. If you don't specify an instance type in a launch template or for ``instanceTypes`` , then ``t3.medium`` is used, by default. If you specify ``Spot`` for ``capacityType`` , then we recommend specifying multiple values for ``instanceTypes`` . For more information, see `Managed node group capacity types <https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html#managed-node-group-capacity-types>`_ and `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param labels: The Kubernetes ``labels`` applied to the nodes in the node group. .. epigraph:: Only ``labels`` that are applied with the Amazon EKS API are shown here. There may be other Kubernetes ``labels`` applied to the nodes in this group.
        :param launch_template: An object representing a node group's launch template specification. When using this object, don't directly specify ``instanceTypes`` , ``diskSize`` , or ``remoteAccess`` . You cannot later specify a different launch template ID or name than what was used to create the node group. Make sure that the launch template meets the requirements in ``launchTemplateSpecification`` . Also refer to `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param nodegroup_name: The unique name to give your node group.
        :param node_repair_config: The node auto repair configuration for the node group.
        :param release_version: The AMI version of the Amazon EKS optimized AMI to use with your node group (for example, ``1.14.7- *YYYYMMDD*`` ). By default, the latest available AMI version for the node group's current Kubernetes version is used. For more information, see `Amazon EKS optimized Linux AMI Versions <https://docs.aws.amazon.com/eks/latest/userguide/eks-linux-ami-versions.html>`_ in the *Amazon EKS User Guide* . .. epigraph:: Changing this value triggers an update of the node group if one is available. You can't update other properties at the same time as updating ``Release Version`` .
        :param remote_access: The remote access configuration to use with your node group. For Linux, the protocol is SSH. For Windows, the protocol is RDP. If you specify ``launchTemplate`` , then don't specify ``remoteAccess`` , or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param scaling_config: The scaling configuration details for the Auto Scaling group that is created for your node group.
        :param tags: Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.
        :param taints: The Kubernetes taints to be applied to the nodes in the node group when they are created. Effect is one of ``No_Schedule`` , ``Prefer_No_Schedule`` , or ``No_Execute`` . Kubernetes taints can be used together with tolerations to control how workloads are scheduled to your nodes. For more information, see `Node taints on managed node groups <https://docs.aws.amazon.com/eks/latest/userguide/node-taints-managed-node-groups.html>`_ .
        :param update_config: The node group update configuration.
        :param version: The Kubernetes version to use for your managed nodes. By default, the Kubernetes version of the cluster is used, and this is the only accepted specified value. If you specify ``launchTemplate`` , and your launch template uses a custom AMI, then don't specify ``version`` , or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Launch template support <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* . .. epigraph:: You can't update other properties at the same time as updating ``Version`` .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__27ebd660a66f96284eec036f7614b1586f77d9990c9dd345fe73522c7eac4866)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnNodegroupProps(
            cluster_name=cluster_name,
            node_role=node_role,
            subnets=subnets,
            ami_type=ami_type,
            capacity_type=capacity_type,
            disk_size=disk_size,
            force_update_enabled=force_update_enabled,
            instance_types=instance_types,
            labels=labels,
            launch_template=launch_template,
            nodegroup_name=nodegroup_name,
            node_repair_config=node_repair_config,
            release_version=release_version,
            remote_access=remote_access,
            scaling_config=scaling_config,
            tags=tags,
            taints=taints,
            update_config=update_config,
            version=version,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__213f52373701671efa519b6eac011932f18609528cdff921d2e7509412eb174e)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a8b41163a6b6a86e481761d59ae06c31182ed7794318b2d79180938999660d15)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) associated with the managed node group.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrClusterName")
    def attr_cluster_name(self) -> builtins.str:
        '''The name of your cluster.

        :cloudformationAttribute: ClusterName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrClusterName"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrNodegroupName")
    def attr_nodegroup_name(self) -> builtins.str:
        '''The name associated with an Amazon EKS managed node group.

        :cloudformationAttribute: NodegroupName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrNodegroupName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''The name of your cluster.'''
        return typing.cast(builtins.str, jsii.get(self, "clusterName"))

    @cluster_name.setter
    def cluster_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__329bcf82464d6a9c568249464ff57d3b274cb5ad910619278bdffcc807095f6e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clusterName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="nodeRole")
    def node_role(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the IAM role to associate with your node group.'''
        return typing.cast(builtins.str, jsii.get(self, "nodeRole"))

    @node_role.setter
    def node_role(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__467ca0d8c05cf0da6447c7ff6f0041c9de68fa584c110ab5415ef4445f50989b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "nodeRole", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="subnets")
    def subnets(self) -> typing.List[builtins.str]:
        '''The subnets to use for the Auto Scaling group that is created for your node group.'''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "subnets"))

    @subnets.setter
    def subnets(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a1044111401adec5b1ee9395d5c43c39358192850ffc1a6609ff114e5f131dff)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "subnets", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="amiType")
    def ami_type(self) -> typing.Optional[builtins.str]:
        '''The AMI type for your node group.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "amiType"))

    @ami_type.setter
    def ami_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f36a5e9b2e583fba8ca22f2ea52e76686b1f69526eb390cab382cf44c0ba644b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "amiType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="capacityType")
    def capacity_type(self) -> typing.Optional[builtins.str]:
        '''The capacity type of your managed node group.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "capacityType"))

    @capacity_type.setter
    def capacity_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c6c720be1e59b98ec3e40b47c8f3cc04d3280bef9ba638e855a6888c777fe157)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "capacityType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="diskSize")
    def disk_size(self) -> typing.Optional[jsii.Number]:
        '''The root device disk size (in GiB) for your node group instances.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "diskSize"))

    @disk_size.setter
    def disk_size(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3516f16c81673ae72899d2c96f98e95ae2e5abd57afd40b83d7a1435ed459d27)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "diskSize", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="forceUpdateEnabled")
    def force_update_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Force the update if any ``Pod`` on the existing node group can't be drained due to a ``Pod`` disruption budget issue.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "forceUpdateEnabled"))

    @force_update_enabled.setter
    def force_update_enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7cce0ad9b5b51aaeaa6d678d457aa515ac721bcdd6f2676cca34a3fb307a858b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "forceUpdateEnabled", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="instanceTypes")
    def instance_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify the instance types for a node group.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "instanceTypes"))

    @instance_types.setter
    def instance_types(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d7945a05c2376d70b30fc28a96959abf6f7e23d90fd6e700764137f4b551604d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceTypes", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="labels")
    def labels(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''The Kubernetes ``labels`` applied to the nodes in the node group.'''
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], jsii.get(self, "labels"))

    @labels.setter
    def labels(
        self,
        value: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bfdcdc90e2da680bcf24a101a162039fcfa5c8ddbf5eae9a97a45451dc10a0e1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "labels", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="launchTemplate")
    def launch_template(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.LaunchTemplateSpecificationProperty"]]:
        '''An object representing a node group's launch template specification.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.LaunchTemplateSpecificationProperty"]], jsii.get(self, "launchTemplate"))

    @launch_template.setter
    def launch_template(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.LaunchTemplateSpecificationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f986a2a8b814558b87814aecfa28259e5d1b52be1fde18a8f0252c4f20b20b44)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "launchTemplate", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="nodegroupName")
    def nodegroup_name(self) -> typing.Optional[builtins.str]:
        '''The unique name to give your node group.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "nodegroupName"))

    @nodegroup_name.setter
    def nodegroup_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4f424c6e7f54a85ddc7d4750a4a59bfb7f9e1faed7e8263996db88acd70c2546)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "nodegroupName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="nodeRepairConfig")
    def node_repair_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.NodeRepairConfigProperty"]]:
        '''The node auto repair configuration for the node group.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.NodeRepairConfigProperty"]], jsii.get(self, "nodeRepairConfig"))

    @node_repair_config.setter
    def node_repair_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.NodeRepairConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__32d9a193c1fbf4f744119faa14907eda38fe01339dc08c10a27310a3e1360bee)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "nodeRepairConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="releaseVersion")
    def release_version(self) -> typing.Optional[builtins.str]:
        '''The AMI version of the Amazon EKS optimized AMI to use with your node group (for example, ``1.14.7- *YYYYMMDD*`` ). By default, the latest available AMI version for the node group's current Kubernetes version is used. For more information, see `Amazon EKS optimized Linux AMI Versions <https://docs.aws.amazon.com/eks/latest/userguide/eks-linux-ami-versions.html>`_ in the *Amazon EKS User Guide* .'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "releaseVersion"))

    @release_version.setter
    def release_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8d0a201b4d5113f2762fa5568ac8994d40e377b014ef0043d75172492c9679fa)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "releaseVersion", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="remoteAccess")
    def remote_access(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.RemoteAccessProperty"]]:
        '''The remote access configuration to use with your node group.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.RemoteAccessProperty"]], jsii.get(self, "remoteAccess"))

    @remote_access.setter
    def remote_access(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.RemoteAccessProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2306ea427981e78be59e04448bcf0d6eacd69ff104d5df09fcb847b77a2903c8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "remoteAccess", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="scalingConfig")
    def scaling_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.ScalingConfigProperty"]]:
        '''The scaling configuration details for the Auto Scaling group that is created for your node group.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.ScalingConfigProperty"]], jsii.get(self, "scalingConfig"))

    @scaling_config.setter
    def scaling_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.ScalingConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a5e379ed1faa0fad6a31a3083279e1b70d6e65ab0aeb2918f33039ee9868e7c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "scalingConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Metadata that assists with categorization and organization.'''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__94ec64dc986623100a074c1a0c77742b6dd8af77e4a7b1b62cafec394cf8e03e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="taints")
    def taints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.TaintProperty"]]]]:
        '''The Kubernetes taints to be applied to the nodes in the node group when they are created.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.TaintProperty"]]]], jsii.get(self, "taints"))

    @taints.setter
    def taints(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.TaintProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a536373a8cd2abd0a608e8b4018137f75247b8cfdc1d5dd505b555e1f8679d4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "taints", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="updateConfig")
    def update_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.UpdateConfigProperty"]]:
        '''The node group update configuration.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.UpdateConfigProperty"]], jsii.get(self, "updateConfig"))

    @update_config.setter
    def update_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnNodegroup.UpdateConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a91d7f81c1a3c53f8f75a4727dbdc4bdd123a517aa6a6faa4540c448309ab792)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "updateConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[builtins.str]:
        '''The Kubernetes version to use for your managed nodes.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "version"))

    @version.setter
    def version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7c7b5dd411ee27c14eb459522efe494a08bd3f2592d384283dbbcc7a5c3e8834)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "version", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnNodegroup.LaunchTemplateSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"id": "id", "name": "name", "version": "version"},
    )
    class LaunchTemplateSpecificationProperty:
        def __init__(
            self,
            *,
            id: typing.Optional[builtins.str] = None,
            name: typing.Optional[builtins.str] = None,
            version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing a node group launch template specification.

            The launch template can't include ```SubnetId`` <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html>`_ , ```IamInstanceProfile`` <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_IamInstanceProfile.html>`_ , ```RequestSpotInstances`` <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html>`_ , ```HibernationOptions`` <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_HibernationOptionsRequest.html>`_ , or ```TerminateInstances`` <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateInstances.html>`_ , or the node group deployment or update will fail. For more information about launch templates, see ```CreateLaunchTemplate`` <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplate.html>`_ in the Amazon EC2 API Reference. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .

            You must specify either the launch template ID or the launch template name in the request, but not both.

            :param id: The ID of the launch template. You must specify either the launch template ID or the launch template name in the request, but not both. After node group creation, you cannot use a different ID.
            :param name: The name of the launch template. You must specify either the launch template name or the launch template ID in the request, but not both. After node group creation, you cannot use a different name.
            :param version: The version number of the launch template to use. If no version is specified, then the template's default version is used. You can use a different version for node group updates.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-launchtemplatespecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                launch_template_specification_property = eks.CfnNodegroup.LaunchTemplateSpecificationProperty(
                    id="id",
                    name="name",
                    version="version"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e659212680af90c8732b5ec096030b6902f35121f1ca1a82a513ebaa5a9cabb4)
                check_type(argname="argument id", value=id, expected_type=type_hints["id"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if id is not None:
                self._values["id"] = id
            if name is not None:
                self._values["name"] = name
            if version is not None:
                self._values["version"] = version

        @builtins.property
        def id(self) -> typing.Optional[builtins.str]:
            '''The ID of the launch template.

            You must specify either the launch template ID or the launch template name in the request, but not both. After node group creation, you cannot use a different ID.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-launchtemplatespecification.html#cfn-eks-nodegroup-launchtemplatespecification-id
            '''
            result = self._values.get("id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the launch template.

            You must specify either the launch template name or the launch template ID in the request, but not both. After node group creation, you cannot use a different name.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-launchtemplatespecification.html#cfn-eks-nodegroup-launchtemplatespecification-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def version(self) -> typing.Optional[builtins.str]:
            '''The version number of the launch template to use.

            If no version is specified, then the template's default version is used. You can use a different version for node group updates.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-launchtemplatespecification.html#cfn-eks-nodegroup-launchtemplatespecification-version
            '''
            result = self._values.get("version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnNodegroup.NodeRepairConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"enabled": "enabled"},
    )
    class NodeRepairConfigProperty:
        def __init__(
            self,
            *,
            enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''The node auto repair configuration for the node group.

            :param enabled: Specifies whether to enable node auto repair for the node group. Node auto repair is disabled by default.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-noderepairconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                node_repair_config_property = eks.CfnNodegroup.NodeRepairConfigProperty(
                    enabled=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__29dbda55ee07f00e62bcfcbc392973b5c2850e347abc3e6692b5d82704d445f0)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if enabled is not None:
                self._values["enabled"] = enabled

        @builtins.property
        def enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether to enable node auto repair for the node group.

            Node auto repair is disabled by default.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-noderepairconfig.html#cfn-eks-nodegroup-noderepairconfig-enabled
            '''
            result = self._values.get("enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NodeRepairConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnNodegroup.RemoteAccessProperty",
        jsii_struct_bases=[],
        name_mapping={
            "ec2_ssh_key": "ec2SshKey",
            "source_security_groups": "sourceSecurityGroups",
        },
    )
    class RemoteAccessProperty:
        def __init__(
            self,
            *,
            ec2_ssh_key: builtins.str,
            source_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object representing the remote access configuration for the managed node group.

            :param ec2_ssh_key: The Amazon EC2 SSH key name that provides access for SSH communication with the nodes in the managed node group. For more information, see `Amazon EC2 key pairs and Linux instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html>`_ in the *Amazon Elastic Compute Cloud User Guide for Linux Instances* . For Windows, an Amazon EC2 SSH key is used to obtain the RDP password. For more information, see `Amazon EC2 key pairs and Windows instances <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-key-pairs.html>`_ in the *Amazon Elastic Compute Cloud User Guide for Windows Instances* .
            :param source_security_groups: The security group IDs that are allowed SSH access (port 22) to the nodes. For Windows, the port is 3389. If you specify an Amazon EC2 SSH key but don't specify a source security group when you create a managed node group, then the port on the nodes is opened to the internet ( ``0.0.0.0/0`` ). For more information, see `Security Groups for Your VPC <https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html>`_ in the *Amazon Virtual Private Cloud User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-remoteaccess.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                remote_access_property = eks.CfnNodegroup.RemoteAccessProperty(
                    ec2_ssh_key="ec2SshKey",
                
                    # the properties below are optional
                    source_security_groups=["sourceSecurityGroups"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__20ad8c5a88d11d94fe39192936f148394a3c64c19310bb51fdcbb909cfab8bb7)
                check_type(argname="argument ec2_ssh_key", value=ec2_ssh_key, expected_type=type_hints["ec2_ssh_key"])
                check_type(argname="argument source_security_groups", value=source_security_groups, expected_type=type_hints["source_security_groups"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "ec2_ssh_key": ec2_ssh_key,
            }
            if source_security_groups is not None:
                self._values["source_security_groups"] = source_security_groups

        @builtins.property
        def ec2_ssh_key(self) -> builtins.str:
            '''The Amazon EC2 SSH key name that provides access for SSH communication with the nodes in the managed node group.

            For more information, see `Amazon EC2 key pairs and Linux instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html>`_ in the *Amazon Elastic Compute Cloud User Guide for Linux Instances* . For Windows, an Amazon EC2 SSH key is used to obtain the RDP password. For more information, see `Amazon EC2 key pairs and Windows instances <https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-key-pairs.html>`_ in the *Amazon Elastic Compute Cloud User Guide for Windows Instances* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-remoteaccess.html#cfn-eks-nodegroup-remoteaccess-ec2sshkey
            '''
            result = self._values.get("ec2_ssh_key")
            assert result is not None, "Required property 'ec2_ssh_key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def source_security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The security group IDs that are allowed SSH access (port 22) to the nodes.

            For Windows, the port is 3389. If you specify an Amazon EC2 SSH key but don't specify a source security group when you create a managed node group, then the port on the nodes is opened to the internet ( ``0.0.0.0/0`` ). For more information, see `Security Groups for Your VPC <https://docs.aws.amazon.com/vpc/latest/userguide/VPC_SecurityGroups.html>`_ in the *Amazon Virtual Private Cloud User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-remoteaccess.html#cfn-eks-nodegroup-remoteaccess-sourcesecuritygroups
            '''
            result = self._values.get("source_security_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RemoteAccessProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnNodegroup.ScalingConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "desired_size": "desiredSize",
            "max_size": "maxSize",
            "min_size": "minSize",
        },
    )
    class ScalingConfigProperty:
        def __init__(
            self,
            *,
            desired_size: typing.Optional[jsii.Number] = None,
            max_size: typing.Optional[jsii.Number] = None,
            min_size: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object representing the scaling configuration details for the Auto Scaling group that is associated with your node group.

            When creating a node group, you must specify all or none of the properties. When updating a node group, you can specify any or none of the properties.

            :param desired_size: The current number of nodes that the managed node group should maintain. .. epigraph:: If you use the Kubernetes `Cluster Autoscaler <https://docs.aws.amazon.com/https://github.com/kubernetes/autoscaler#kubernetes-autoscaler>`_ , you shouldn't change the ``desiredSize`` value directly, as this can cause the Cluster Autoscaler to suddenly scale up or scale down. Whenever this parameter changes, the number of worker nodes in the node group is updated to the specified size. If this parameter is given a value that is smaller than the current number of running worker nodes, the necessary number of worker nodes are terminated to match the given value. When using CloudFormation, no action occurs if you remove this parameter from your CFN template. This parameter can be different from ``minSize`` in some cases, such as when starting with extra hosts for testing. This parameter can also be different when you want to start with an estimated number of needed hosts, but let the Cluster Autoscaler reduce the number if there are too many. When the Cluster Autoscaler is used, the ``desiredSize`` parameter is altered by the Cluster Autoscaler (but can be out-of-date for short periods of time). the Cluster Autoscaler doesn't scale a managed node group lower than ``minSize`` or higher than ``maxSize`` .
            :param max_size: The maximum number of nodes that the managed node group can scale out to. For information about the maximum number that you can specify, see `Amazon EKS service quotas <https://docs.aws.amazon.com/eks/latest/userguide/service-quotas.html>`_ in the *Amazon EKS User Guide* .
            :param min_size: The minimum number of nodes that the managed node group can scale in to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-scalingconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                scaling_config_property = eks.CfnNodegroup.ScalingConfigProperty(
                    desired_size=123,
                    max_size=123,
                    min_size=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0464e44c4b12d9975542fbf5d31e8159afcbfc5a1b0c0c41795b3305d31460e5)
                check_type(argname="argument desired_size", value=desired_size, expected_type=type_hints["desired_size"])
                check_type(argname="argument max_size", value=max_size, expected_type=type_hints["max_size"])
                check_type(argname="argument min_size", value=min_size, expected_type=type_hints["min_size"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if desired_size is not None:
                self._values["desired_size"] = desired_size
            if max_size is not None:
                self._values["max_size"] = max_size
            if min_size is not None:
                self._values["min_size"] = min_size

        @builtins.property
        def desired_size(self) -> typing.Optional[jsii.Number]:
            '''The current number of nodes that the managed node group should maintain.

            .. epigraph::

               If you use the Kubernetes `Cluster Autoscaler <https://docs.aws.amazon.com/https://github.com/kubernetes/autoscaler#kubernetes-autoscaler>`_ , you shouldn't change the ``desiredSize`` value directly, as this can cause the Cluster Autoscaler to suddenly scale up or scale down.

            Whenever this parameter changes, the number of worker nodes in the node group is updated to the specified size. If this parameter is given a value that is smaller than the current number of running worker nodes, the necessary number of worker nodes are terminated to match the given value. When using CloudFormation, no action occurs if you remove this parameter from your CFN template.

            This parameter can be different from ``minSize`` in some cases, such as when starting with extra hosts for testing. This parameter can also be different when you want to start with an estimated number of needed hosts, but let the Cluster Autoscaler reduce the number if there are too many. When the Cluster Autoscaler is used, the ``desiredSize`` parameter is altered by the Cluster Autoscaler (but can be out-of-date for short periods of time). the Cluster Autoscaler doesn't scale a managed node group lower than ``minSize`` or higher than ``maxSize`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-scalingconfig.html#cfn-eks-nodegroup-scalingconfig-desiredsize
            '''
            result = self._values.get("desired_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def max_size(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of nodes that the managed node group can scale out to.

            For information about the maximum number that you can specify, see `Amazon EKS service quotas <https://docs.aws.amazon.com/eks/latest/userguide/service-quotas.html>`_ in the *Amazon EKS User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-scalingconfig.html#cfn-eks-nodegroup-scalingconfig-maxsize
            '''
            result = self._values.get("max_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min_size(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of nodes that the managed node group can scale in to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-scalingconfig.html#cfn-eks-nodegroup-scalingconfig-minsize
            '''
            result = self._values.get("min_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnNodegroup.TaintProperty",
        jsii_struct_bases=[],
        name_mapping={"effect": "effect", "key": "key", "value": "value"},
    )
    class TaintProperty:
        def __init__(
            self,
            *,
            effect: typing.Optional[builtins.str] = None,
            key: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A property that allows a node to repel a ``Pod`` .

            For more information, see `Node taints on managed node groups <https://docs.aws.amazon.com/eks/latest/userguide/node-taints-managed-node-groups.html>`_ in the *Amazon EKS User Guide* .

            :param effect: The effect of the taint.
            :param key: The key of the taint.
            :param value: The value of the taint.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-taint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                taint_property = eks.CfnNodegroup.TaintProperty(
                    effect="effect",
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d6fd0504aab0911cb5bf46ea45177d42f1452898b7e3e8edeef996366df3afaa)
                check_type(argname="argument effect", value=effect, expected_type=type_hints["effect"])
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if effect is not None:
                self._values["effect"] = effect
            if key is not None:
                self._values["key"] = key
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def effect(self) -> typing.Optional[builtins.str]:
            '''The effect of the taint.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-taint.html#cfn-eks-nodegroup-taint-effect
            '''
            result = self._values.get("effect")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def key(self) -> typing.Optional[builtins.str]:
            '''The key of the taint.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-taint.html#cfn-eks-nodegroup-taint-key
            '''
            result = self._values.get("key")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value of the taint.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-taint.html#cfn-eks-nodegroup-taint-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TaintProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_eks.CfnNodegroup.UpdateConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_unavailable": "maxUnavailable",
            "max_unavailable_percentage": "maxUnavailablePercentage",
            "update_strategy": "updateStrategy",
        },
    )
    class UpdateConfigProperty:
        def __init__(
            self,
            *,
            max_unavailable: typing.Optional[jsii.Number] = None,
            max_unavailable_percentage: typing.Optional[jsii.Number] = None,
            update_strategy: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The update configuration for the node group.

            :param max_unavailable: The maximum number of nodes unavailable at once during a version update. Nodes are updated in parallel. This value or ``maxUnavailablePercentage`` is required to have a value.The maximum number is 100.
            :param max_unavailable_percentage: The maximum percentage of nodes unavailable during a version update. This percentage of nodes are updated in parallel, up to 100 nodes at once. This value or ``maxUnavailable`` is required to have a value.
            :param update_strategy: The configuration for the behavior to follow during a node group version update of this managed node group. You choose between two possible strategies for replacing nodes during an ```UpdateNodegroupVersion`` <https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateNodegroupVersion.html>`_ action. An Amazon EKS managed node group updates by replacing nodes with new nodes of newer AMI versions in parallel. The *update strategy* changes the managed node update behavior of the managed node group for each quantity. The *default* strategy has guardrails to protect you from misconfiguration and launches the new instances first, before terminating the old instances. The *minimal* strategy removes the guardrails and terminates the old instances before launching the new instances. This minimal strategy is useful in scenarios where you are constrained to resources or costs (for example, with hardware accelerators such as GPUs).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-updateconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_eks as eks
                
                update_config_property = eks.CfnNodegroup.UpdateConfigProperty(
                    max_unavailable=123,
                    max_unavailable_percentage=123,
                    update_strategy="updateStrategy"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__883c5208b02930e4808e078b3ecb98b51411fa92d248a8031301fe5434326e12)
                check_type(argname="argument max_unavailable", value=max_unavailable, expected_type=type_hints["max_unavailable"])
                check_type(argname="argument max_unavailable_percentage", value=max_unavailable_percentage, expected_type=type_hints["max_unavailable_percentage"])
                check_type(argname="argument update_strategy", value=update_strategy, expected_type=type_hints["update_strategy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max_unavailable is not None:
                self._values["max_unavailable"] = max_unavailable
            if max_unavailable_percentage is not None:
                self._values["max_unavailable_percentage"] = max_unavailable_percentage
            if update_strategy is not None:
                self._values["update_strategy"] = update_strategy

        @builtins.property
        def max_unavailable(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of nodes unavailable at once during a version update.

            Nodes are updated in parallel. This value or ``maxUnavailablePercentage`` is required to have a value.The maximum number is 100.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-updateconfig.html#cfn-eks-nodegroup-updateconfig-maxunavailable
            '''
            result = self._values.get("max_unavailable")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def max_unavailable_percentage(self) -> typing.Optional[jsii.Number]:
            '''The maximum percentage of nodes unavailable during a version update.

            This percentage of nodes are updated in parallel, up to 100 nodes at once. This value or ``maxUnavailable`` is required to have a value.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-updateconfig.html#cfn-eks-nodegroup-updateconfig-maxunavailablepercentage
            '''
            result = self._values.get("max_unavailable_percentage")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def update_strategy(self) -> typing.Optional[builtins.str]:
            '''The configuration for the behavior to follow during a node group version update of this managed node group.

            You choose between two possible strategies for replacing nodes during an ```UpdateNodegroupVersion`` <https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateNodegroupVersion.html>`_ action.

            An Amazon EKS managed node group updates by replacing nodes with new nodes of newer AMI versions in parallel. The *update strategy* changes the managed node update behavior of the managed node group for each quantity. The *default* strategy has guardrails to protect you from misconfiguration and launches the new instances first, before terminating the old instances. The *minimal* strategy removes the guardrails and terminates the old instances before launching the new instances. This minimal strategy is useful in scenarios where you are constrained to resources or costs (for example, with hardware accelerators such as GPUs).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-updateconfig.html#cfn-eks-nodegroup-updateconfig-updatestrategy
            '''
            result = self._values.get("update_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "UpdateConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.CfnNodegroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster_name": "clusterName",
        "node_role": "nodeRole",
        "subnets": "subnets",
        "ami_type": "amiType",
        "capacity_type": "capacityType",
        "disk_size": "diskSize",
        "force_update_enabled": "forceUpdateEnabled",
        "instance_types": "instanceTypes",
        "labels": "labels",
        "launch_template": "launchTemplate",
        "nodegroup_name": "nodegroupName",
        "node_repair_config": "nodeRepairConfig",
        "release_version": "releaseVersion",
        "remote_access": "remoteAccess",
        "scaling_config": "scalingConfig",
        "tags": "tags",
        "taints": "taints",
        "update_config": "updateConfig",
        "version": "version",
    },
)
class CfnNodegroupProps:
    def __init__(
        self,
        *,
        cluster_name: builtins.str,
        node_role: builtins.str,
        subnets: typing.Sequence[builtins.str],
        ami_type: typing.Optional[builtins.str] = None,
        capacity_type: typing.Optional[builtins.str] = None,
        disk_size: typing.Optional[jsii.Number] = None,
        force_update_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        instance_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        labels: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        launch_template: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnNodegroup.LaunchTemplateSpecificationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        nodegroup_name: typing.Optional[builtins.str] = None,
        node_repair_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnNodegroup.NodeRepairConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        release_version: typing.Optional[builtins.str] = None,
        remote_access: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnNodegroup.RemoteAccessProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        scaling_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnNodegroup.ScalingConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        taints: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnNodegroup.TaintProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        update_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnNodegroup.UpdateConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        version: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnNodegroup``.

        :param cluster_name: The name of your cluster.
        :param node_role: The Amazon Resource Name (ARN) of the IAM role to associate with your node group. The Amazon EKS worker node ``kubelet`` daemon makes calls to AWS APIs on your behalf. Nodes receive permissions for these API calls through an IAM instance profile and associated policies. Before you can launch nodes and register them into a cluster, you must create an IAM role for those nodes to use when they are launched. For more information, see `Amazon EKS node IAM role <https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html>`_ in the **Amazon EKS User Guide** . If you specify ``launchTemplate`` , then don't specify ``[IamInstanceProfile](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_IamInstanceProfile.html)`` in your launch template, or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param subnets: The subnets to use for the Auto Scaling group that is created for your node group. If you specify ``launchTemplate`` , then don't specify ``[SubnetId](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html)`` in your launch template, or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param ami_type: The AMI type for your node group. If you specify ``launchTemplate`` , and your launch template uses a custom AMI, then don't specify ``amiType`` , or the node group deployment will fail. If your launch template uses a Windows custom AMI, then add ``eks:kube-proxy-windows`` to your Windows nodes ``rolearn`` in the ``aws-auth`` ``ConfigMap`` . For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param capacity_type: The capacity type of your managed node group.
        :param disk_size: The root device disk size (in GiB) for your node group instances. The default disk size is 20 GiB for Linux and Bottlerocket. The default disk size is 50 GiB for Windows. If you specify ``launchTemplate`` , then don't specify ``diskSize`` , or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param force_update_enabled: Force the update if any ``Pod`` on the existing node group can't be drained due to a ``Pod`` disruption budget issue. If an update fails because all Pods can't be drained, you can force the update after it fails to terminate the old node whether or not any ``Pod`` is running on the node. Default: - false
        :param instance_types: Specify the instance types for a node group. If you specify a GPU instance type, make sure to also specify an applicable GPU AMI type with the ``amiType`` parameter. If you specify ``launchTemplate`` , then you can specify zero or one instance type in your launch template *or* you can specify 0-20 instance types for ``instanceTypes`` . If however, you specify an instance type in your launch template *and* specify any ``instanceTypes`` , the node group deployment will fail. If you don't specify an instance type in a launch template or for ``instanceTypes`` , then ``t3.medium`` is used, by default. If you specify ``Spot`` for ``capacityType`` , then we recommend specifying multiple values for ``instanceTypes`` . For more information, see `Managed node group capacity types <https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html#managed-node-group-capacity-types>`_ and `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param labels: The Kubernetes ``labels`` applied to the nodes in the node group. .. epigraph:: Only ``labels`` that are applied with the Amazon EKS API are shown here. There may be other Kubernetes ``labels`` applied to the nodes in this group.
        :param launch_template: An object representing a node group's launch template specification. When using this object, don't directly specify ``instanceTypes`` , ``diskSize`` , or ``remoteAccess`` . You cannot later specify a different launch template ID or name than what was used to create the node group. Make sure that the launch template meets the requirements in ``launchTemplateSpecification`` . Also refer to `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param nodegroup_name: The unique name to give your node group.
        :param node_repair_config: The node auto repair configuration for the node group.
        :param release_version: The AMI version of the Amazon EKS optimized AMI to use with your node group (for example, ``1.14.7- *YYYYMMDD*`` ). By default, the latest available AMI version for the node group's current Kubernetes version is used. For more information, see `Amazon EKS optimized Linux AMI Versions <https://docs.aws.amazon.com/eks/latest/userguide/eks-linux-ami-versions.html>`_ in the *Amazon EKS User Guide* . .. epigraph:: Changing this value triggers an update of the node group if one is available. You can't update other properties at the same time as updating ``Release Version`` .
        :param remote_access: The remote access configuration to use with your node group. For Linux, the protocol is SSH. For Windows, the protocol is RDP. If you specify ``launchTemplate`` , then don't specify ``remoteAccess`` , or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        :param scaling_config: The scaling configuration details for the Auto Scaling group that is created for your node group.
        :param tags: Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.
        :param taints: The Kubernetes taints to be applied to the nodes in the node group when they are created. Effect is one of ``No_Schedule`` , ``Prefer_No_Schedule`` , or ``No_Execute`` . Kubernetes taints can be used together with tolerations to control how workloads are scheduled to your nodes. For more information, see `Node taints on managed node groups <https://docs.aws.amazon.com/eks/latest/userguide/node-taints-managed-node-groups.html>`_ .
        :param update_config: The node group update configuration.
        :param version: The Kubernetes version to use for your managed nodes. By default, the Kubernetes version of the cluster is used, and this is the only accepted specified value. If you specify ``launchTemplate`` , and your launch template uses a custom AMI, then don't specify ``version`` , or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Launch template support <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* . .. epigraph:: You can't update other properties at the same time as updating ``Version`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            cfn_nodegroup_props = eks.CfnNodegroupProps(
                cluster_name="clusterName",
                node_role="nodeRole",
                subnets=["subnets"],
            
                # the properties below are optional
                ami_type="amiType",
                capacity_type="capacityType",
                disk_size=123,
                force_update_enabled=False,
                instance_types=["instanceTypes"],
                labels={
                    "labels_key": "labels"
                },
                launch_template=eks.CfnNodegroup.LaunchTemplateSpecificationProperty(
                    id="id",
                    name="name",
                    version="version"
                ),
                nodegroup_name="nodegroupName",
                node_repair_config=eks.CfnNodegroup.NodeRepairConfigProperty(
                    enabled=False
                ),
                release_version="releaseVersion",
                remote_access=eks.CfnNodegroup.RemoteAccessProperty(
                    ec2_ssh_key="ec2SshKey",
            
                    # the properties below are optional
                    source_security_groups=["sourceSecurityGroups"]
                ),
                scaling_config=eks.CfnNodegroup.ScalingConfigProperty(
                    desired_size=123,
                    max_size=123,
                    min_size=123
                ),
                tags={
                    "tags_key": "tags"
                },
                taints=[eks.CfnNodegroup.TaintProperty(
                    effect="effect",
                    key="key",
                    value="value"
                )],
                update_config=eks.CfnNodegroup.UpdateConfigProperty(
                    max_unavailable=123,
                    max_unavailable_percentage=123,
                    update_strategy="updateStrategy"
                ),
                version="version"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__61a7b4277678abead400083fb1974a4f71ee28a78b5e79235fc3a458144e88c3)
            check_type(argname="argument cluster_name", value=cluster_name, expected_type=type_hints["cluster_name"])
            check_type(argname="argument node_role", value=node_role, expected_type=type_hints["node_role"])
            check_type(argname="argument subnets", value=subnets, expected_type=type_hints["subnets"])
            check_type(argname="argument ami_type", value=ami_type, expected_type=type_hints["ami_type"])
            check_type(argname="argument capacity_type", value=capacity_type, expected_type=type_hints["capacity_type"])
            check_type(argname="argument disk_size", value=disk_size, expected_type=type_hints["disk_size"])
            check_type(argname="argument force_update_enabled", value=force_update_enabled, expected_type=type_hints["force_update_enabled"])
            check_type(argname="argument instance_types", value=instance_types, expected_type=type_hints["instance_types"])
            check_type(argname="argument labels", value=labels, expected_type=type_hints["labels"])
            check_type(argname="argument launch_template", value=launch_template, expected_type=type_hints["launch_template"])
            check_type(argname="argument nodegroup_name", value=nodegroup_name, expected_type=type_hints["nodegroup_name"])
            check_type(argname="argument node_repair_config", value=node_repair_config, expected_type=type_hints["node_repair_config"])
            check_type(argname="argument release_version", value=release_version, expected_type=type_hints["release_version"])
            check_type(argname="argument remote_access", value=remote_access, expected_type=type_hints["remote_access"])
            check_type(argname="argument scaling_config", value=scaling_config, expected_type=type_hints["scaling_config"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument taints", value=taints, expected_type=type_hints["taints"])
            check_type(argname="argument update_config", value=update_config, expected_type=type_hints["update_config"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_name": cluster_name,
            "node_role": node_role,
            "subnets": subnets,
        }
        if ami_type is not None:
            self._values["ami_type"] = ami_type
        if capacity_type is not None:
            self._values["capacity_type"] = capacity_type
        if disk_size is not None:
            self._values["disk_size"] = disk_size
        if force_update_enabled is not None:
            self._values["force_update_enabled"] = force_update_enabled
        if instance_types is not None:
            self._values["instance_types"] = instance_types
        if labels is not None:
            self._values["labels"] = labels
        if launch_template is not None:
            self._values["launch_template"] = launch_template
        if nodegroup_name is not None:
            self._values["nodegroup_name"] = nodegroup_name
        if node_repair_config is not None:
            self._values["node_repair_config"] = node_repair_config
        if release_version is not None:
            self._values["release_version"] = release_version
        if remote_access is not None:
            self._values["remote_access"] = remote_access
        if scaling_config is not None:
            self._values["scaling_config"] = scaling_config
        if tags is not None:
            self._values["tags"] = tags
        if taints is not None:
            self._values["taints"] = taints
        if update_config is not None:
            self._values["update_config"] = update_config
        if version is not None:
            self._values["version"] = version

    @builtins.property
    def cluster_name(self) -> builtins.str:
        '''The name of your cluster.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-clustername
        '''
        result = self._values.get("cluster_name")
        assert result is not None, "Required property 'cluster_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def node_role(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the IAM role to associate with your node group.

        The Amazon EKS worker node ``kubelet`` daemon makes calls to AWS APIs on your behalf. Nodes receive permissions for these API calls through an IAM instance profile and associated policies. Before you can launch nodes and register them into a cluster, you must create an IAM role for those nodes to use when they are launched. For more information, see `Amazon EKS node IAM role <https://docs.aws.amazon.com/eks/latest/userguide/create-node-role.html>`_ in the **Amazon EKS User Guide** . If you specify ``launchTemplate`` , then don't specify ``[IamInstanceProfile](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_IamInstanceProfile.html)`` in your launch template, or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-noderole
        '''
        result = self._values.get("node_role")
        assert result is not None, "Required property 'node_role' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def subnets(self) -> typing.List[builtins.str]:
        '''The subnets to use for the Auto Scaling group that is created for your node group.

        If you specify ``launchTemplate`` , then don't specify ``[SubnetId](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html)`` in your launch template, or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-subnets
        '''
        result = self._values.get("subnets")
        assert result is not None, "Required property 'subnets' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def ami_type(self) -> typing.Optional[builtins.str]:
        '''The AMI type for your node group.

        If you specify ``launchTemplate`` , and your launch template uses a custom AMI, then don't specify ``amiType`` , or the node group deployment will fail. If your launch template uses a Windows custom AMI, then add ``eks:kube-proxy-windows`` to your Windows nodes ``rolearn`` in the ``aws-auth`` ``ConfigMap`` . For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-amitype
        '''
        result = self._values.get("ami_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def capacity_type(self) -> typing.Optional[builtins.str]:
        '''The capacity type of your managed node group.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-capacitytype
        '''
        result = self._values.get("capacity_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def disk_size(self) -> typing.Optional[jsii.Number]:
        '''The root device disk size (in GiB) for your node group instances.

        The default disk size is 20 GiB for Linux and Bottlerocket. The default disk size is 50 GiB for Windows. If you specify ``launchTemplate`` , then don't specify ``diskSize`` , or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-disksize
        '''
        result = self._values.get("disk_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def force_update_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Force the update if any ``Pod`` on the existing node group can't be drained due to a ``Pod`` disruption budget issue.

        If an update fails because all Pods can't be drained, you can force the update after it fails to terminate the old node whether or not any ``Pod`` is running on the node.

        :default: - false

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-forceupdateenabled
        '''
        result = self._values.get("force_update_enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def instance_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify the instance types for a node group.

        If you specify a GPU instance type, make sure to also specify an applicable GPU AMI type with the ``amiType`` parameter. If you specify ``launchTemplate`` , then you can specify zero or one instance type in your launch template *or* you can specify 0-20 instance types for ``instanceTypes`` . If however, you specify an instance type in your launch template *and* specify any ``instanceTypes`` , the node group deployment will fail. If you don't specify an instance type in a launch template or for ``instanceTypes`` , then ``t3.medium`` is used, by default. If you specify ``Spot`` for ``capacityType`` , then we recommend specifying multiple values for ``instanceTypes`` . For more information, see `Managed node group capacity types <https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html#managed-node-group-capacity-types>`_ and `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-instancetypes
        '''
        result = self._values.get("instance_types")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def labels(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''The Kubernetes ``labels`` applied to the nodes in the node group.

        .. epigraph::

           Only ``labels`` that are applied with the Amazon EKS API are shown here. There may be other Kubernetes ``labels`` applied to the nodes in this group.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-labels
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

    @builtins.property
    def launch_template(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnNodegroup.LaunchTemplateSpecificationProperty]]:
        '''An object representing a node group's launch template specification.

        When using this object, don't directly specify ``instanceTypes`` , ``diskSize`` , or ``remoteAccess`` . You cannot later specify a different launch template ID or name than what was used to create the node group.

        Make sure that the launch template meets the requirements in ``launchTemplateSpecification`` . Also refer to `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-launchtemplate
        '''
        result = self._values.get("launch_template")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnNodegroup.LaunchTemplateSpecificationProperty]], result)

    @builtins.property
    def nodegroup_name(self) -> typing.Optional[builtins.str]:
        '''The unique name to give your node group.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-nodegroupname
        '''
        result = self._values.get("nodegroup_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def node_repair_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnNodegroup.NodeRepairConfigProperty]]:
        '''The node auto repair configuration for the node group.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-noderepairconfig
        '''
        result = self._values.get("node_repair_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnNodegroup.NodeRepairConfigProperty]], result)

    @builtins.property
    def release_version(self) -> typing.Optional[builtins.str]:
        '''The AMI version of the Amazon EKS optimized AMI to use with your node group (for example, ``1.14.7- *YYYYMMDD*`` ). By default, the latest available AMI version for the node group's current Kubernetes version is used. For more information, see `Amazon EKS optimized Linux AMI Versions <https://docs.aws.amazon.com/eks/latest/userguide/eks-linux-ami-versions.html>`_ in the *Amazon EKS User Guide* .

        .. epigraph::

           Changing this value triggers an update of the node group if one is available. You can't update other properties at the same time as updating ``Release Version`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-releaseversion
        '''
        result = self._values.get("release_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def remote_access(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnNodegroup.RemoteAccessProperty]]:
        '''The remote access configuration to use with your node group.

        For Linux, the protocol is SSH. For Windows, the protocol is RDP. If you specify ``launchTemplate`` , then don't specify ``remoteAccess`` , or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Customizing managed nodes with launch templates <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-remoteaccess
        '''
        result = self._values.get("remote_access")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnNodegroup.RemoteAccessProperty]], result)

    @builtins.property
    def scaling_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnNodegroup.ScalingConfigProperty]]:
        '''The scaling configuration details for the Auto Scaling group that is created for your node group.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-scalingconfig
        '''
        result = self._values.get("scaling_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnNodegroup.ScalingConfigProperty]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Metadata that assists with categorization and organization.

        Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def taints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnNodegroup.TaintProperty]]]]:
        '''The Kubernetes taints to be applied to the nodes in the node group when they are created.

        Effect is one of ``No_Schedule`` , ``Prefer_No_Schedule`` , or ``No_Execute`` . Kubernetes taints can be used together with tolerations to control how workloads are scheduled to your nodes. For more information, see `Node taints on managed node groups <https://docs.aws.amazon.com/eks/latest/userguide/node-taints-managed-node-groups.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-taints
        '''
        result = self._values.get("taints")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnNodegroup.TaintProperty]]]], result)

    @builtins.property
    def update_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnNodegroup.UpdateConfigProperty]]:
        '''The node group update configuration.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-updateconfig
        '''
        result = self._values.get("update_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnNodegroup.UpdateConfigProperty]], result)

    @builtins.property
    def version(self) -> typing.Optional[builtins.str]:
        '''The Kubernetes version to use for your managed nodes.

        By default, the Kubernetes version of the cluster is used, and this is the only accepted specified value. If you specify ``launchTemplate`` , and your launch template uses a custom AMI, then don't specify ``version`` , or the node group deployment will fail. For more information about using launch templates with Amazon EKS, see `Launch template support <https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html>`_ in the *Amazon EKS User Guide* .
        .. epigraph::

           You can't update other properties at the same time as updating ``Version`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-version
        '''
        result = self._values.get("version")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnNodegroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnPodIdentityAssociation(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.CfnPodIdentityAssociation",
):
    '''Amazon EKS Pod Identity associations provide the ability to manage credentials for your applications, similar to the way that Amazon EC2 instance profiles provide credentials to Amazon EC2 instances.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-podidentityassociation.html
    :cloudformationResource: AWS::EKS::PodIdentityAssociation
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        cfn_pod_identity_association = eks.CfnPodIdentityAssociation(self, "MyCfnPodIdentityAssociation",
            cluster_name="clusterName",
            namespace="namespace",
            role_arn="roleArn",
            service_account="serviceAccount",
        
            # the properties below are optional
            disable_session_tags=False,
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            target_role_arn="targetRoleArn"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_name: builtins.str,
        namespace: builtins.str,
        role_arn: builtins.str,
        service_account: builtins.str,
        disable_session_tags: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        target_role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param cluster_name: The name of the cluster that the association is in.
        :param namespace: The name of the Kubernetes namespace inside the cluster to create the association in. The service account and the pods that use the service account must be in this namespace.
        :param role_arn: The Amazon Resource Name (ARN) of the IAM role to associate with the service account. The EKS Pod Identity agent manages credentials to assume this role for applications in the containers in the pods that use this service account.
        :param service_account: The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.
        :param disable_session_tags: The Disable Session Tags of the pod identity association.
        :param tags: Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources. The following basic restrictions apply to tags: - Maximum number of tags per resource – 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length – 128 Unicode characters in UTF-8 - Maximum value length – 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : /
        :param target_role_arn: The Target Role Arn of the pod identity association.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__be8311b6089cea26f85c63a586f0c5b063230a1b4a96ffcd4c6c983a331d8652)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnPodIdentityAssociationProps(
            cluster_name=cluster_name,
            namespace=namespace,
            role_arn=role_arn,
            service_account=service_account,
            disable_session_tags=disable_session_tags,
            tags=tags,
            target_role_arn=target_role_arn,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__26b7c71efd1b005f002a41b4ba8f08c91a1b0c6903db3395e3e4dbc78202dc97)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e70edd7c10909d184b262375eab581d57950253501cc6fa6b59fc7e777369f0d)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrAssociationArn")
    def attr_association_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the association.

        :cloudformationAttribute: AssociationArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAssociationArn"))

    @builtins.property
    @jsii.member(jsii_name="attrAssociationId")
    def attr_association_id(self) -> builtins.str:
        '''The ID of the association.

        :cloudformationAttribute: AssociationId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAssociationId"))

    @builtins.property
    @jsii.member(jsii_name="attrExternalId")
    def attr_external_id(self) -> builtins.str:
        '''The External Id of the pod identity association.

        :cloudformationAttribute: ExternalId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrExternalId"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''The name of the cluster that the association is in.'''
        return typing.cast(builtins.str, jsii.get(self, "clusterName"))

    @cluster_name.setter
    def cluster_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1b2143669c4f5de45fe1227ae8c525fd924072b1aeaecb67e1334b0f293ce9bb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clusterName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="namespace")
    def namespace(self) -> builtins.str:
        '''The name of the Kubernetes namespace inside the cluster to create the association in.'''
        return typing.cast(builtins.str, jsii.get(self, "namespace"))

    @namespace.setter
    def namespace(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d926eb0433042c883c2612bd63954350831d0e2bc0b93accd3707ab8375f1288)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "namespace", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the IAM role to associate with the service account.'''
        return typing.cast(builtins.str, jsii.get(self, "roleArn"))

    @role_arn.setter
    def role_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1bb4ff18d6b0174e9d12bd2098a77eaabd9d96392e92da7e0fe1521109814e7d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "roleArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="serviceAccount")
    def service_account(self) -> builtins.str:
        '''The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.'''
        return typing.cast(builtins.str, jsii.get(self, "serviceAccount"))

    @service_account.setter
    def service_account(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea3bb34348aff57e29a5352e7460510bda8dd51720dbf7d275297137fb59e6c1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serviceAccount", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="disableSessionTags")
    def disable_session_tags(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''The Disable Session Tags of the pod identity association.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "disableSessionTags"))

    @disable_session_tags.setter
    def disable_session_tags(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cb3dbe4cc3b44e9265bbfe13e41235db909b0c1dc0e052b3bdda07fd4b228e8b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "disableSessionTags", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Metadata that assists with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tags"))

    @tags.setter
    def tags(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b0e0a0551adefc10761733af04b8c51e7dad6b483be9252882ecff10539c7dcc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="targetRoleArn")
    def target_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Target Role Arn of the pod identity association.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "targetRoleArn"))

    @target_role_arn.setter
    def target_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cb6220c6db8cf93a8a307b1ba0630d6bc64b4a09325e7cfe5854228aa75ff833)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetRoleArn", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.CfnPodIdentityAssociationProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster_name": "clusterName",
        "namespace": "namespace",
        "role_arn": "roleArn",
        "service_account": "serviceAccount",
        "disable_session_tags": "disableSessionTags",
        "tags": "tags",
        "target_role_arn": "targetRoleArn",
    },
)
class CfnPodIdentityAssociationProps:
    def __init__(
        self,
        *,
        cluster_name: builtins.str,
        namespace: builtins.str,
        role_arn: builtins.str,
        service_account: builtins.str,
        disable_session_tags: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        target_role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnPodIdentityAssociation``.

        :param cluster_name: The name of the cluster that the association is in.
        :param namespace: The name of the Kubernetes namespace inside the cluster to create the association in. The service account and the pods that use the service account must be in this namespace.
        :param role_arn: The Amazon Resource Name (ARN) of the IAM role to associate with the service account. The EKS Pod Identity agent manages credentials to assume this role for applications in the containers in the pods that use this service account.
        :param service_account: The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.
        :param disable_session_tags: The Disable Session Tags of the pod identity association.
        :param tags: Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources. The following basic restrictions apply to tags: - Maximum number of tags per resource – 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length – 128 Unicode characters in UTF-8 - Maximum value length – 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : /
        :param target_role_arn: The Target Role Arn of the pod identity association.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-podidentityassociation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            cfn_pod_identity_association_props = eks.CfnPodIdentityAssociationProps(
                cluster_name="clusterName",
                namespace="namespace",
                role_arn="roleArn",
                service_account="serviceAccount",
            
                # the properties below are optional
                disable_session_tags=False,
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                target_role_arn="targetRoleArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__40e8da56b529234cdbb596fa46af952a935adf744e907347861dfc232b89038b)
            check_type(argname="argument cluster_name", value=cluster_name, expected_type=type_hints["cluster_name"])
            check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
            check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
            check_type(argname="argument service_account", value=service_account, expected_type=type_hints["service_account"])
            check_type(argname="argument disable_session_tags", value=disable_session_tags, expected_type=type_hints["disable_session_tags"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument target_role_arn", value=target_role_arn, expected_type=type_hints["target_role_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_name": cluster_name,
            "namespace": namespace,
            "role_arn": role_arn,
            "service_account": service_account,
        }
        if disable_session_tags is not None:
            self._values["disable_session_tags"] = disable_session_tags
        if tags is not None:
            self._values["tags"] = tags
        if target_role_arn is not None:
            self._values["target_role_arn"] = target_role_arn

    @builtins.property
    def cluster_name(self) -> builtins.str:
        '''The name of the cluster that the association is in.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-podidentityassociation.html#cfn-eks-podidentityassociation-clustername
        '''
        result = self._values.get("cluster_name")
        assert result is not None, "Required property 'cluster_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def namespace(self) -> builtins.str:
        '''The name of the Kubernetes namespace inside the cluster to create the association in.

        The service account and the pods that use the service account must be in this namespace.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-podidentityassociation.html#cfn-eks-podidentityassociation-namespace
        '''
        result = self._values.get("namespace")
        assert result is not None, "Required property 'namespace' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the IAM role to associate with the service account.

        The EKS Pod Identity agent manages credentials to assume this role for applications in the containers in the pods that use this service account.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-podidentityassociation.html#cfn-eks-podidentityassociation-rolearn
        '''
        result = self._values.get("role_arn")
        assert result is not None, "Required property 'role_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service_account(self) -> builtins.str:
        '''The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-podidentityassociation.html#cfn-eks-podidentityassociation-serviceaccount
        '''
        result = self._values.get("service_account")
        assert result is not None, "Required property 'service_account' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def disable_session_tags(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''The Disable Session Tags of the pod identity association.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-podidentityassociation.html#cfn-eks-podidentityassociation-disablesessiontags
        '''
        result = self._values.get("disable_session_tags")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Metadata that assists with categorization and organization.

        Each tag consists of a key and an optional value. You define both. Tags don't propagate to any other cluster or AWS resources.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource – 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length – 128 Unicode characters in UTF-8
        - Maximum value length – 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : /

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-podidentityassociation.html#cfn-eks-podidentityassociation-tags
        ::

        .

        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def target_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Target Role Arn of the pod identity association.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-podidentityassociation.html#cfn-eks-podidentityassociation-targetrolearn
        '''
        result = self._values.get("target_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnPodIdentityAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.ClusterAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "cluster_name": "clusterName",
        "awscli_layer": "awscliLayer",
        "cluster_certificate_authority_data": "clusterCertificateAuthorityData",
        "cluster_encryption_config_key_arn": "clusterEncryptionConfigKeyArn",
        "cluster_endpoint": "clusterEndpoint",
        "cluster_handler_security_group_id": "clusterHandlerSecurityGroupId",
        "cluster_security_group_id": "clusterSecurityGroupId",
        "ip_family": "ipFamily",
        "kubectl_environment": "kubectlEnvironment",
        "kubectl_lambda_role": "kubectlLambdaRole",
        "kubectl_layer": "kubectlLayer",
        "kubectl_memory": "kubectlMemory",
        "kubectl_private_subnet_ids": "kubectlPrivateSubnetIds",
        "kubectl_provider": "kubectlProvider",
        "kubectl_role_arn": "kubectlRoleArn",
        "kubectl_security_group_id": "kubectlSecurityGroupId",
        "on_event_layer": "onEventLayer",
        "open_id_connect_provider": "openIdConnectProvider",
        "prune": "prune",
        "security_group_ids": "securityGroupIds",
        "vpc": "vpc",
    },
)
class ClusterAttributes:
    def __init__(
        self,
        *,
        cluster_name: builtins.str,
        awscli_layer: typing.Optional[_ILayerVersion_5ac127c8] = None,
        cluster_certificate_authority_data: typing.Optional[builtins.str] = None,
        cluster_encryption_config_key_arn: typing.Optional[builtins.str] = None,
        cluster_endpoint: typing.Optional[builtins.str] = None,
        cluster_handler_security_group_id: typing.Optional[builtins.str] = None,
        cluster_security_group_id: typing.Optional[builtins.str] = None,
        ip_family: typing.Optional["IpFamily"] = None,
        kubectl_environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        kubectl_lambda_role: typing.Optional[_IRole_235f5d8e] = None,
        kubectl_layer: typing.Optional[_ILayerVersion_5ac127c8] = None,
        kubectl_memory: typing.Optional[_Size_7b441c34] = None,
        kubectl_private_subnet_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        kubectl_provider: typing.Optional["IKubectlProvider"] = None,
        kubectl_role_arn: typing.Optional[builtins.str] = None,
        kubectl_security_group_id: typing.Optional[builtins.str] = None,
        on_event_layer: typing.Optional[_ILayerVersion_5ac127c8] = None,
        open_id_connect_provider: typing.Optional[_IOpenIdConnectProvider_203f0793] = None,
        prune: typing.Optional[builtins.bool] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        vpc: typing.Optional[_IVpc_f30d5663] = None,
    ) -> None:
        '''Attributes for EKS clusters.

        :param cluster_name: The physical name of the Cluster.
        :param awscli_layer: An AWS Lambda layer that contains the ``aws`` CLI. The handler expects the layer to include the following executables:: /opt/awscli/aws Default: - a default layer with the AWS CLI 1.x
        :param cluster_certificate_authority_data: The certificate-authority-data for your cluster. Default: - if not specified ``cluster.clusterCertificateAuthorityData`` will throw an error
        :param cluster_encryption_config_key_arn: Amazon Resource Name (ARN) or alias of the customer master key (CMK). Default: - if not specified ``cluster.clusterEncryptionConfigKeyArn`` will throw an error
        :param cluster_endpoint: The API Server endpoint URL. Default: - if not specified ``cluster.clusterEndpoint`` will throw an error.
        :param cluster_handler_security_group_id: A security group id to associate with the Cluster Handler's Lambdas. The Cluster Handler's Lambdas are responsible for calling AWS's EKS API. Default: - No security group.
        :param cluster_security_group_id: The cluster security group that was created by Amazon EKS for the cluster. Default: - if not specified ``cluster.clusterSecurityGroupId`` will throw an error
        :param ip_family: Specify which IP family is used to assign Kubernetes pod and service IP addresses. Default: - IpFamily.IP_V4
        :param kubectl_environment: Environment variables to use when running ``kubectl`` against this cluster. Default: - no additional variables
        :param kubectl_lambda_role: An IAM role that can perform kubectl operations against this cluster. The role should be mapped to the ``system:masters`` Kubernetes RBAC role. This role is directly passed to the lambda handler that sends Kube Ctl commands to the cluster. Default: - if not specified, the default role created by a lambda function will be used.
        :param kubectl_layer: An AWS Lambda Layer which includes ``kubectl`` and Helm. This layer is used by the kubectl handler to apply manifests and install helm charts. You must pick an appropriate releases of one of the ``@aws-cdk/layer-kubectl-vXX`` packages, that works with the version of Kubernetes you have chosen. The handler expects the layer to include the following executables:: /opt/helm/helm /opt/kubectl/kubectl Default: - No default layer will be provided
        :param kubectl_memory: Amount of memory to allocate to the provider's lambda function. Default: Size.gibibytes(1)
        :param kubectl_private_subnet_ids: Subnets to host the ``kubectl`` compute resources. If not specified, the k8s endpoint is expected to be accessible publicly. Default: - k8s endpoint is expected to be accessible publicly
        :param kubectl_provider: KubectlProvider for issuing kubectl commands. Default: - Default CDK provider
        :param kubectl_role_arn: An IAM role with cluster administrator and "system:masters" permissions. Default: - if not specified, it not be possible to issue ``kubectl`` commands against an imported cluster.
        :param kubectl_security_group_id: A security group to use for ``kubectl`` execution. If not specified, the k8s endpoint is expected to be accessible publicly. Default: - k8s endpoint is expected to be accessible publicly
        :param on_event_layer: An AWS Lambda Layer which includes the NPM dependency ``proxy-agent``. This layer is used by the onEvent handler to route AWS SDK requests through a proxy. The handler expects the layer to include the following node_modules: proxy-agent Default: - a layer bundled with this module.
        :param open_id_connect_provider: An Open ID Connect provider for this cluster that can be used to configure service accounts. You can either import an existing provider using ``iam.OpenIdConnectProvider.fromProviderArn``, or create a new provider using ``new eks.OpenIdConnectProvider`` Default: - if not specified ``cluster.openIdConnectProvider`` and ``cluster.addServiceAccount`` will throw an error.
        :param prune: Indicates whether Kubernetes resources added through ``addManifest()`` can be automatically pruned. When this is enabled (default), prune labels will be allocated and injected to each resource. These labels will then be used when issuing the ``kubectl apply`` operation with the ``--prune`` switch. Default: true
        :param security_group_ids: Additional security groups associated with this cluster. Default: - if not specified, no additional security groups will be considered in ``cluster.connections``.
        :param vpc: The VPC in which this Cluster was created. Default: - if not specified ``cluster.vpc`` will throw an error

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            # asg: autoscaling.AutoScalingGroup
            
            imported_cluster = eks.Cluster.from_cluster_attributes(self, "ImportedCluster",
                cluster_name=cluster.cluster_name,
                cluster_security_group_id=cluster.cluster_security_group_id
            )
            
            imported_cluster.connect_auto_scaling_group_capacity(asg)
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__82438aafc3b8617d6addf9a67750499050d5a25c5c0ded527e516cc3b3091a34)
            check_type(argname="argument cluster_name", value=cluster_name, expected_type=type_hints["cluster_name"])
            check_type(argname="argument awscli_layer", value=awscli_layer, expected_type=type_hints["awscli_layer"])
            check_type(argname="argument cluster_certificate_authority_data", value=cluster_certificate_authority_data, expected_type=type_hints["cluster_certificate_authority_data"])
            check_type(argname="argument cluster_encryption_config_key_arn", value=cluster_encryption_config_key_arn, expected_type=type_hints["cluster_encryption_config_key_arn"])
            check_type(argname="argument cluster_endpoint", value=cluster_endpoint, expected_type=type_hints["cluster_endpoint"])
            check_type(argname="argument cluster_handler_security_group_id", value=cluster_handler_security_group_id, expected_type=type_hints["cluster_handler_security_group_id"])
            check_type(argname="argument cluster_security_group_id", value=cluster_security_group_id, expected_type=type_hints["cluster_security_group_id"])
            check_type(argname="argument ip_family", value=ip_family, expected_type=type_hints["ip_family"])
            check_type(argname="argument kubectl_environment", value=kubectl_environment, expected_type=type_hints["kubectl_environment"])
            check_type(argname="argument kubectl_lambda_role", value=kubectl_lambda_role, expected_type=type_hints["kubectl_lambda_role"])
            check_type(argname="argument kubectl_layer", value=kubectl_layer, expected_type=type_hints["kubectl_layer"])
            check_type(argname="argument kubectl_memory", value=kubectl_memory, expected_type=type_hints["kubectl_memory"])
            check_type(argname="argument kubectl_private_subnet_ids", value=kubectl_private_subnet_ids, expected_type=type_hints["kubectl_private_subnet_ids"])
            check_type(argname="argument kubectl_provider", value=kubectl_provider, expected_type=type_hints["kubectl_provider"])
            check_type(argname="argument kubectl_role_arn", value=kubectl_role_arn, expected_type=type_hints["kubectl_role_arn"])
            check_type(argname="argument kubectl_security_group_id", value=kubectl_security_group_id, expected_type=type_hints["kubectl_security_group_id"])
            check_type(argname="argument on_event_layer", value=on_event_layer, expected_type=type_hints["on_event_layer"])
            check_type(argname="argument open_id_connect_provider", value=open_id_connect_provider, expected_type=type_hints["open_id_connect_provider"])
            check_type(argname="argument prune", value=prune, expected_type=type_hints["prune"])
            check_type(argname="argument security_group_ids", value=security_group_ids, expected_type=type_hints["security_group_ids"])
            check_type(argname="argument vpc", value=vpc, expected_type=type_hints["vpc"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster_name": cluster_name,
        }
        if awscli_layer is not None:
            self._values["awscli_layer"] = awscli_layer
        if cluster_certificate_authority_data is not None:
            self._values["cluster_certificate_authority_data"] = cluster_certificate_authority_data
        if cluster_encryption_config_key_arn is not None:
            self._values["cluster_encryption_config_key_arn"] = cluster_encryption_config_key_arn
        if cluster_endpoint is not None:
            self._values["cluster_endpoint"] = cluster_endpoint
        if cluster_handler_security_group_id is not None:
            self._values["cluster_handler_security_group_id"] = cluster_handler_security_group_id
        if cluster_security_group_id is not None:
            self._values["cluster_security_group_id"] = cluster_security_group_id
        if ip_family is not None:
            self._values["ip_family"] = ip_family
        if kubectl_environment is not None:
            self._values["kubectl_environment"] = kubectl_environment
        if kubectl_lambda_role is not None:
            self._values["kubectl_lambda_role"] = kubectl_lambda_role
        if kubectl_layer is not None:
            self._values["kubectl_layer"] = kubectl_layer
        if kubectl_memory is not None:
            self._values["kubectl_memory"] = kubectl_memory
        if kubectl_private_subnet_ids is not None:
            self._values["kubectl_private_subnet_ids"] = kubectl_private_subnet_ids
        if kubectl_provider is not None:
            self._values["kubectl_provider"] = kubectl_provider
        if kubectl_role_arn is not None:
            self._values["kubectl_role_arn"] = kubectl_role_arn
        if kubectl_security_group_id is not None:
            self._values["kubectl_security_group_id"] = kubectl_security_group_id
        if on_event_layer is not None:
            self._values["on_event_layer"] = on_event_layer
        if open_id_connect_provider is not None:
            self._values["open_id_connect_provider"] = open_id_connect_provider
        if prune is not None:
            self._values["prune"] = prune
        if security_group_ids is not None:
            self._values["security_group_ids"] = security_group_ids
        if vpc is not None:
            self._values["vpc"] = vpc

    @builtins.property
    def cluster_name(self) -> builtins.str:
        '''The physical name of the Cluster.'''
        result = self._values.get("cluster_name")
        assert result is not None, "Required property 'cluster_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def awscli_layer(self) -> typing.Optional[_ILayerVersion_5ac127c8]:
        '''An AWS Lambda layer that contains the ``aws`` CLI.

        The handler expects the layer to include the following executables::

           /opt/awscli/aws

        :default: - a default layer with the AWS CLI 1.x
        '''
        result = self._values.get("awscli_layer")
        return typing.cast(typing.Optional[_ILayerVersion_5ac127c8], result)

    @builtins.property
    def cluster_certificate_authority_data(self) -> typing.Optional[builtins.str]:
        '''The certificate-authority-data for your cluster.

        :default:

        - if not specified ``cluster.clusterCertificateAuthorityData`` will
        throw an error
        '''
        result = self._values.get("cluster_certificate_authority_data")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cluster_encryption_config_key_arn(self) -> typing.Optional[builtins.str]:
        '''Amazon Resource Name (ARN) or alias of the customer master key (CMK).

        :default:

        - if not specified ``cluster.clusterEncryptionConfigKeyArn`` will
        throw an error
        '''
        result = self._values.get("cluster_encryption_config_key_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cluster_endpoint(self) -> typing.Optional[builtins.str]:
        '''The API Server endpoint URL.

        :default: - if not specified ``cluster.clusterEndpoint`` will throw an error.
        '''
        result = self._values.get("cluster_endpoint")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cluster_handler_security_group_id(self) -> typing.Optional[builtins.str]:
        '''A security group id to associate with the Cluster Handler's Lambdas.

        The Cluster Handler's Lambdas are responsible for calling AWS's EKS API.

        :default: - No security group.
        '''
        result = self._values.get("cluster_handler_security_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cluster_security_group_id(self) -> typing.Optional[builtins.str]:
        '''The cluster security group that was created by Amazon EKS for the cluster.

        :default:

        - if not specified ``cluster.clusterSecurityGroupId`` will throw an
        error
        '''
        result = self._values.get("cluster_security_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ip_family(self) -> typing.Optional["IpFamily"]:
        '''Specify which IP family is used to assign Kubernetes pod and service IP addresses.

        :default: - IpFamily.IP_V4

        :see: https://docs.aws.amazon.com/eks/latest/APIReference/API_KubernetesNetworkConfigRequest.html#AmazonEKS-Type-KubernetesNetworkConfigRequest-ipFamily
        '''
        result = self._values.get("ip_family")
        return typing.cast(typing.Optional["IpFamily"], result)

    @builtins.property
    def kubectl_environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Environment variables to use when running ``kubectl`` against this cluster.

        :default: - no additional variables
        '''
        result = self._values.get("kubectl_environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def kubectl_lambda_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''An IAM role that can perform kubectl operations against this cluster.

        The role should be mapped to the ``system:masters`` Kubernetes RBAC role.

        This role is directly passed to the lambda handler that sends Kube Ctl commands
        to the cluster.

        :default:

        - if not specified, the default role created by a lambda function will
        be used.
        '''
        result = self._values.get("kubectl_lambda_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def kubectl_layer(self) -> typing.Optional[_ILayerVersion_5ac127c8]:
        '''An AWS Lambda Layer which includes ``kubectl`` and Helm.

        This layer is used by the kubectl handler to apply manifests and install
        helm charts. You must pick an appropriate releases of one of the
        ``@aws-cdk/layer-kubectl-vXX`` packages, that works with the version of
        Kubernetes you have chosen.

        The handler expects the layer to include the following executables::

           /opt/helm/helm
           /opt/kubectl/kubectl

        :default: - No default layer will be provided
        '''
        result = self._values.get("kubectl_layer")
        return typing.cast(typing.Optional[_ILayerVersion_5ac127c8], result)

    @builtins.property
    def kubectl_memory(self) -> typing.Optional[_Size_7b441c34]:
        '''Amount of memory to allocate to the provider's lambda function.

        :default: Size.gibibytes(1)
        '''
        result = self._values.get("kubectl_memory")
        return typing.cast(typing.Optional[_Size_7b441c34], result)

    @builtins.property
    def kubectl_private_subnet_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Subnets to host the ``kubectl`` compute resources.

        If not specified, the k8s
        endpoint is expected to be accessible publicly.

        :default: - k8s endpoint is expected to be accessible publicly
        '''
        result = self._values.get("kubectl_private_subnet_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def kubectl_provider(self) -> typing.Optional["IKubectlProvider"]:
        '''KubectlProvider for issuing kubectl commands.

        :default: - Default CDK provider
        '''
        result = self._values.get("kubectl_provider")
        return typing.cast(typing.Optional["IKubectlProvider"], result)

    @builtins.property
    def kubectl_role_arn(self) -> typing.Optional[builtins.str]:
        '''An IAM role with cluster administrator and "system:masters" permissions.

        :default:

        - if not specified, it not be possible to issue ``kubectl`` commands
        against an imported cluster.
        '''
        result = self._values.get("kubectl_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def kubectl_security_group_id(self) -> typing.Optional[builtins.str]:
        '''A security group to use for ``kubectl`` execution.

        If not specified, the k8s
        endpoint is expected to be accessible publicly.

        :default: - k8s endpoint is expected to be accessible publicly
        '''
        result = self._values.get("kubectl_security_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def on_event_layer(self) -> typing.Optional[_ILayerVersion_5ac127c8]:
        '''An AWS Lambda Layer which includes the NPM dependency ``proxy-agent``.

        This layer
        is used by the onEvent handler to route AWS SDK requests through a proxy.

        The handler expects the layer to include the following node_modules:

        proxy-agent

        :default: - a layer bundled with this module.
        '''
        result = self._values.get("on_event_layer")
        return typing.cast(typing.Optional[_ILayerVersion_5ac127c8], result)

    @builtins.property
    def open_id_connect_provider(
        self,
    ) -> typing.Optional[_IOpenIdConnectProvider_203f0793]:
        '''An Open ID Connect provider for this cluster that can be used to configure service accounts.

        You can either import an existing provider using ``iam.OpenIdConnectProvider.fromProviderArn``,
        or create a new provider using ``new eks.OpenIdConnectProvider``

        :default: - if not specified ``cluster.openIdConnectProvider`` and ``cluster.addServiceAccount`` will throw an error.
        '''
        result = self._values.get("open_id_connect_provider")
        return typing.cast(typing.Optional[_IOpenIdConnectProvider_203f0793], result)

    @builtins.property
    def prune(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether Kubernetes resources added through ``addManifest()`` can be automatically pruned.

        When this is enabled (default), prune labels will be
        allocated and injected to each resource. These labels will then be used
        when issuing the ``kubectl apply`` operation with the ``--prune`` switch.

        :default: true
        '''
        result = self._values.get("prune")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Additional security groups associated with this cluster.

        :default:

        - if not specified, no additional security groups will be
        considered in ``cluster.connections``.
        '''
        result = self._values.get("security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_IVpc_f30d5663]:
        '''The VPC in which this Cluster was created.

        :default: - if not specified ``cluster.vpc`` will throw an error
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_IVpc_f30d5663], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ClusterAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.ClusterLoggingTypes")
class ClusterLoggingTypes(enum.Enum):
    '''EKS cluster logging types.

    :exampleMetadata: infused

    Example::

        from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
        
        
        cluster = eks.Cluster(self, "Cluster",
            # ...
            version=eks.KubernetesVersion.V1_32,
            cluster_logging=[eks.ClusterLoggingTypes.API, eks.ClusterLoggingTypes.AUTHENTICATOR, eks.ClusterLoggingTypes.SCHEDULER
            ],
            kubectl_layer=KubectlV32Layer(self, "kubectl")
        )
    '''

    API = "API"
    '''Logs pertaining to API requests to the cluster.'''
    AUDIT = "AUDIT"
    '''Logs pertaining to cluster access via the Kubernetes API.'''
    AUTHENTICATOR = "AUTHENTICATOR"
    '''Logs pertaining to authentication requests into the cluster.'''
    CONTROLLER_MANAGER = "CONTROLLER_MANAGER"
    '''Logs pertaining to state of cluster controllers.'''
    SCHEDULER = "SCHEDULER"
    '''Logs pertaining to scheduling decisions.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.CommonClusterOptions",
    jsii_struct_bases=[],
    name_mapping={
        "version": "version",
        "cluster_name": "clusterName",
        "output_cluster_name": "outputClusterName",
        "output_config_command": "outputConfigCommand",
        "role": "role",
        "security_group": "securityGroup",
        "vpc": "vpc",
        "vpc_subnets": "vpcSubnets",
    },
)
class CommonClusterOptions:
    def __init__(
        self,
        *,
        version: "KubernetesVersion",
        cluster_name: typing.Optional[builtins.str] = None,
        output_cluster_name: typing.Optional[builtins.bool] = None,
        output_config_command: typing.Optional[builtins.bool] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
        security_group: typing.Optional[_ISecurityGroup_acf8a799] = None,
        vpc: typing.Optional[_IVpc_f30d5663] = None,
        vpc_subnets: typing.Optional[typing.Sequence[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Options for configuring an EKS cluster.

        :param version: The Kubernetes version to run in the cluster.
        :param cluster_name: Name for the cluster. Default: - Automatically generated name
        :param output_cluster_name: Determines whether a CloudFormation output with the name of the cluster will be synthesized. Default: false
        :param output_config_command: Determines whether a CloudFormation output with the ``aws eks update-kubeconfig`` command will be synthesized. This command will include the cluster name and, if applicable, the ARN of the masters IAM role. Default: true
        :param role: Role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf. Default: - A role is automatically created for you
        :param security_group: Security Group to use for Control Plane ENIs. Default: - A security group is automatically created
        :param vpc: The VPC in which to create the Cluster. Default: - a VPC with default configuration will be created and can be accessed through ``cluster.vpc``.
        :param vpc_subnets: Where to place EKS Control Plane ENIs. For example, to only select private subnets, supply the following: ``vpcSubnets: [{ subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS }]`` Default: - All public and private subnets

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_eks as eks
            from aws_cdk import aws_iam as iam
            
            # kubernetes_version: eks.KubernetesVersion
            # role: iam.Role
            # security_group: ec2.SecurityGroup
            # subnet: ec2.Subnet
            # subnet_filter: ec2.SubnetFilter
            # vpc: ec2.Vpc
            
            common_cluster_options = eks.CommonClusterOptions(
                version=kubernetes_version,
            
                # the properties below are optional
                cluster_name="clusterName",
                output_cluster_name=False,
                output_config_command=False,
                role=role,
                security_group=security_group,
                vpc=vpc,
                vpc_subnets=[ec2.SubnetSelection(
                    availability_zones=["availabilityZones"],
                    one_per_az=False,
                    subnet_filters=[subnet_filter],
                    subnet_group_name="subnetGroupName",
                    subnets=[subnet],
                    subnet_type=ec2.SubnetType.PRIVATE_ISOLATED
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1f8c7de9c27316f4e34258a0f6fcae4b96da91fa4bd5e531f1ed83df5844b4bd)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            check_type(argname="argument cluster_name", value=cluster_name, expected_type=type_hints["cluster_name"])
            check_type(argname="argument output_cluster_name", value=output_cluster_name, expected_type=type_hints["output_cluster_name"])
            check_type(argname="argument output_config_command", value=output_config_command, expected_type=type_hints["output_config_command"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument security_group", value=security_group, expected_type=type_hints["security_group"])
            check_type(argname="argument vpc", value=vpc, expected_type=type_hints["vpc"])
            check_type(argname="argument vpc_subnets", value=vpc_subnets, expected_type=type_hints["vpc_subnets"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "version": version,
        }
        if cluster_name is not None:
            self._values["cluster_name"] = cluster_name
        if output_cluster_name is not None:
            self._values["output_cluster_name"] = output_cluster_name
        if output_config_command is not None:
            self._values["output_config_command"] = output_config_command
        if role is not None:
            self._values["role"] = role
        if security_group is not None:
            self._values["security_group"] = security_group
        if vpc is not None:
            self._values["vpc"] = vpc
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def version(self) -> "KubernetesVersion":
        '''The Kubernetes version to run in the cluster.'''
        result = self._values.get("version")
        assert result is not None, "Required property 'version' is missing"
        return typing.cast("KubernetesVersion", result)

    @builtins.property
    def cluster_name(self) -> typing.Optional[builtins.str]:
        '''Name for the cluster.

        :default: - Automatically generated name
        '''
        result = self._values.get("cluster_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_cluster_name(self) -> typing.Optional[builtins.bool]:
        '''Determines whether a CloudFormation output with the name of the cluster will be synthesized.

        :default: false
        '''
        result = self._values.get("output_cluster_name")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def output_config_command(self) -> typing.Optional[builtins.bool]:
        '''Determines whether a CloudFormation output with the ``aws eks update-kubeconfig`` command will be synthesized.

        This command will include
        the cluster name and, if applicable, the ARN of the masters IAM role.

        :default: true
        '''
        result = self._values.get("output_config_command")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''Role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf.

        :default: - A role is automatically created for you
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def security_group(self) -> typing.Optional[_ISecurityGroup_acf8a799]:
        '''Security Group to use for Control Plane ENIs.

        :default: - A security group is automatically created
        '''
        result = self._values.get("security_group")
        return typing.cast(typing.Optional[_ISecurityGroup_acf8a799], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_IVpc_f30d5663]:
        '''The VPC in which to create the Cluster.

        :default: - a VPC with default configuration will be created and can be accessed through ``cluster.vpc``.
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_IVpc_f30d5663], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[typing.List[_SubnetSelection_e57d76df]]:
        '''Where to place EKS Control Plane ENIs.

        For example, to only select private subnets, supply the following:

        ``vpcSubnets: [{ subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS }]``

        :default: - All public and private subnets
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[typing.List[_SubnetSelection_e57d76df]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonClusterOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.CoreDnsComputeType")
class CoreDnsComputeType(enum.Enum):
    '''The type of compute resources to use for CoreDNS.'''

    EC2 = "EC2"
    '''Deploy CoreDNS on EC2 instances.'''
    FARGATE = "FARGATE"
    '''Deploy CoreDNS on Fargate-managed instances.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.CpuArch")
class CpuArch(enum.Enum):
    '''CPU architecture.'''

    ARM_64 = "ARM_64"
    '''arm64 CPU type.'''
    X86_64 = "X86_64"
    '''x86_64 CPU type.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.DefaultCapacityType")
class DefaultCapacityType(enum.Enum):
    '''The default capacity type for the cluster.

    :exampleMetadata: infused

    Example::

        from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
        
        
        cluster = eks.Cluster(self, "HelloEKS",
            version=eks.KubernetesVersion.V1_32,
            default_capacity_type=eks.DefaultCapacityType.EC2,
            kubectl_layer=KubectlV32Layer(self, "kubectl")
        )
    '''

    NODEGROUP = "NODEGROUP"
    '''managed node group.'''
    EC2 = "EC2"
    '''EC2 autoscaling group.'''


@jsii.implements(_IMachineImage_0e8bd50b)
class EksOptimizedImage(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.EksOptimizedImage",
):
    '''Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        eks_optimized_image = eks.EksOptimizedImage(
            cpu_arch=eks.CpuArch.ARM_64,
            kubernetes_version="kubernetesVersion",
            node_type=eks.NodeType.STANDARD
        )
    '''

    def __init__(
        self,
        *,
        cpu_arch: typing.Optional[CpuArch] = None,
        kubernetes_version: typing.Optional[builtins.str] = None,
        node_type: typing.Optional["NodeType"] = None,
    ) -> None:
        '''Constructs a new instance of the EcsOptimizedAmi class.

        :param cpu_arch: What cpu architecture to retrieve the image for (arm64 or x86_64). Default: CpuArch.X86_64
        :param kubernetes_version: The Kubernetes version to use. Default: - The latest version
        :param node_type: What instance type to retrieve the image for (standard or GPU-optimized). Default: NodeType.STANDARD
        '''
        props = EksOptimizedImageProps(
            cpu_arch=cpu_arch,
            kubernetes_version=kubernetes_version,
            node_type=node_type,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="getImage")
    def get_image(
        self,
        scope: _constructs_77d1e7e8.Construct,
    ) -> _MachineImageConfig_187edaee:
        '''Return the correct image.

        :param scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__629e4d4811fb64af8e661195cc881959015c1512e23da98c5c9484740a0b770d)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast(_MachineImageConfig_187edaee, jsii.invoke(self, "getImage", [scope]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.EksOptimizedImageProps",
    jsii_struct_bases=[],
    name_mapping={
        "cpu_arch": "cpuArch",
        "kubernetes_version": "kubernetesVersion",
        "node_type": "nodeType",
    },
)
class EksOptimizedImageProps:
    def __init__(
        self,
        *,
        cpu_arch: typing.Optional[CpuArch] = None,
        kubernetes_version: typing.Optional[builtins.str] = None,
        node_type: typing.Optional["NodeType"] = None,
    ) -> None:
        '''Properties for EksOptimizedImage.

        :param cpu_arch: What cpu architecture to retrieve the image for (arm64 or x86_64). Default: CpuArch.X86_64
        :param kubernetes_version: The Kubernetes version to use. Default: - The latest version
        :param node_type: What instance type to retrieve the image for (standard or GPU-optimized). Default: NodeType.STANDARD

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            eks_optimized_image_props = eks.EksOptimizedImageProps(
                cpu_arch=eks.CpuArch.ARM_64,
                kubernetes_version="kubernetesVersion",
                node_type=eks.NodeType.STANDARD
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8ee78088a323b1b3c0cd8695e784c0dfbeddac5953ad981ebf97d150e4d4bc9f)
            check_type(argname="argument cpu_arch", value=cpu_arch, expected_type=type_hints["cpu_arch"])
            check_type(argname="argument kubernetes_version", value=kubernetes_version, expected_type=type_hints["kubernetes_version"])
            check_type(argname="argument node_type", value=node_type, expected_type=type_hints["node_type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cpu_arch is not None:
            self._values["cpu_arch"] = cpu_arch
        if kubernetes_version is not None:
            self._values["kubernetes_version"] = kubernetes_version
        if node_type is not None:
            self._values["node_type"] = node_type

    @builtins.property
    def cpu_arch(self) -> typing.Optional[CpuArch]:
        '''What cpu architecture to retrieve the image for (arm64 or x86_64).

        :default: CpuArch.X86_64
        '''
        result = self._values.get("cpu_arch")
        return typing.cast(typing.Optional[CpuArch], result)

    @builtins.property
    def kubernetes_version(self) -> typing.Optional[builtins.str]:
        '''The Kubernetes version to use.

        :default: - The latest version
        '''
        result = self._values.get("kubernetes_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def node_type(self) -> typing.Optional["NodeType"]:
        '''What instance type to retrieve the image for (standard or GPU-optimized).

        :default: NodeType.STANDARD
        '''
        result = self._values.get("node_type")
        return typing.cast(typing.Optional["NodeType"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EksOptimizedImageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EndpointAccess(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.EndpointAccess",
):
    '''Endpoint access characteristics.

    :exampleMetadata: infused

    Example::

        from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
        
        
        cluster = eks.Cluster(self, "hello-eks",
            version=eks.KubernetesVersion.V1_32,
            endpoint_access=eks.EndpointAccess.PRIVATE,  # No access outside of your VPC.
            kubectl_layer=KubectlV32Layer(self, "kubectl")
        )
    '''

    @jsii.member(jsii_name="onlyFrom")
    def only_from(self, *cidr: builtins.str) -> "EndpointAccess":
        '''Restrict public access to specific CIDR blocks.

        If public access is disabled, this method will result in an error.

        :param cidr: CIDR blocks.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b28399c52747fcad8e5cbba677f3627fb748e2cb4b111ffc04d23ccb91348d3d)
            check_type(argname="argument cidr", value=cidr, expected_type=typing.Tuple[type_hints["cidr"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("EndpointAccess", jsii.invoke(self, "onlyFrom", [*cidr]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PRIVATE")
    def PRIVATE(cls) -> "EndpointAccess":
        '''The cluster endpoint is only accessible through your VPC.

        Worker node traffic to the endpoint will stay within your VPC.
        '''
        return typing.cast("EndpointAccess", jsii.sget(cls, "PRIVATE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PUBLIC")
    def PUBLIC(cls) -> "EndpointAccess":
        '''The cluster endpoint is accessible from outside of your VPC.

        Worker node traffic will leave your VPC to connect to the endpoint.

        By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the ``PUBLIC.onlyFrom`` method.
        If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
        specify include the addresses that worker nodes and Fargate pods (if you use them)
        access the public endpoint from.
        '''
        return typing.cast("EndpointAccess", jsii.sget(cls, "PUBLIC"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PUBLIC_AND_PRIVATE")
    def PUBLIC_AND_PRIVATE(cls) -> "EndpointAccess":
        '''The cluster endpoint is accessible from outside of your VPC.

        Worker node traffic to the endpoint will stay within your VPC.

        By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the ``PUBLIC_AND_PRIVATE.onlyFrom`` method.
        If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
        specify include the addresses that worker nodes and Fargate pods (if you use them)
        access the public endpoint from.
        '''
        return typing.cast("EndpointAccess", jsii.sget(cls, "PUBLIC_AND_PRIVATE"))


@jsii.implements(_ITaggable_36806126)
class FargateProfile(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.FargateProfile",
):
    '''Fargate profiles allows an administrator to declare which pods run on Fargate.

    This declaration is done through the profile’s selectors. Each
    profile can have up to five selectors that contain a namespace and optional
    labels. You must define a namespace for every selector. The label field
    consists of multiple optional key-value pairs. Pods that match a selector (by
    matching a namespace for the selector and all of the labels specified in the
    selector) are scheduled on Fargate. If a namespace selector is defined
    without any labels, Amazon EKS will attempt to schedule all pods that run in
    that namespace onto Fargate using the profile. If a to-be-scheduled pod
    matches any of the selectors in the Fargate profile, then that pod is
    scheduled on Fargate.

    If a pod matches multiple Fargate profiles, Amazon EKS picks one of the
    matches at random. In this case, you can specify which profile a pod should
    use by adding the following Kubernetes label to the pod specification:
    eks.amazonaws.com/fargate-profile: profile_name. However, the pod must still
    match a selector in that profile in order to be scheduled onto Fargate.

    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        eks.FargateProfile(self, "MyProfile",
            cluster=cluster,
            selectors=[eks.Selector(namespace="default")]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: "Cluster",
        selectors: typing.Sequence[typing.Union["Selector", typing.Dict[builtins.str, typing.Any]]],
        fargate_profile_name: typing.Optional[builtins.str] = None,
        pod_execution_role: typing.Optional[_IRole_235f5d8e] = None,
        subnet_selection: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
        vpc: typing.Optional[_IVpc_f30d5663] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster: The EKS cluster to apply the Fargate profile to. [disable-awslint:ref-via-interface]
        :param selectors: The selectors to match for pods to use this Fargate profile. Each selector must have an associated namespace. Optionally, you can also specify labels for a namespace. At least one selector is required and you may specify up to five selectors.
        :param fargate_profile_name: The name of the Fargate profile. Default: - generated
        :param pod_execution_role: The pod execution role to use for pods that match the selectors in the Fargate profile. The pod execution role allows Fargate infrastructure to register with your cluster as a node, and it provides read access to Amazon ECR image repositories. Default: - a role will be automatically created
        :param subnet_selection: Select which subnets to launch your pods into. At this time, pods running on Fargate are not assigned public IP addresses, so only private subnets (with no direct route to an Internet Gateway) are allowed. You must specify the VPC to customize the subnet selection Default: - all private subnets of the VPC are selected.
        :param vpc: The VPC from which to select subnets to launch your pods into. By default, all private subnets are selected. You can customize this using ``subnetSelection``. Default: - all private subnets used by the EKS cluster
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__715eb0a6f3424ca3c03a6b90c35bd9430b647b9a0c3882fa5276fc51a5b94216)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = FargateProfileProps(
            cluster=cluster,
            selectors=selectors,
            fargate_profile_name=fargate_profile_name,
            pod_execution_role=pod_execution_role,
            subnet_selection=subnet_selection,
            vpc=vpc,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property
    @jsii.member(jsii_name="fargateProfileArn")
    def fargate_profile_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) of the Fargate profile.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "fargateProfileArn"))

    @builtins.property
    @jsii.member(jsii_name="fargateProfileName")
    def fargate_profile_name(self) -> builtins.str:
        '''The name of the Fargate profile.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "fargateProfileName"))

    @builtins.property
    @jsii.member(jsii_name="podExecutionRole")
    def pod_execution_role(self) -> _IRole_235f5d8e:
        '''The pod execution role to use for pods that match the selectors in the Fargate profile.

        The pod execution role allows Fargate infrastructure to
        register with your cluster as a node, and it provides read access to Amazon
        ECR image repositories.
        '''
        return typing.cast(_IRole_235f5d8e, jsii.get(self, "podExecutionRole"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Resource tags.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.FargateProfileOptions",
    jsii_struct_bases=[],
    name_mapping={
        "selectors": "selectors",
        "fargate_profile_name": "fargateProfileName",
        "pod_execution_role": "podExecutionRole",
        "subnet_selection": "subnetSelection",
        "vpc": "vpc",
    },
)
class FargateProfileOptions:
    def __init__(
        self,
        *,
        selectors: typing.Sequence[typing.Union["Selector", typing.Dict[builtins.str, typing.Any]]],
        fargate_profile_name: typing.Optional[builtins.str] = None,
        pod_execution_role: typing.Optional[_IRole_235f5d8e] = None,
        subnet_selection: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
        vpc: typing.Optional[_IVpc_f30d5663] = None,
    ) -> None:
        '''Options for defining EKS Fargate Profiles.

        :param selectors: The selectors to match for pods to use this Fargate profile. Each selector must have an associated namespace. Optionally, you can also specify labels for a namespace. At least one selector is required and you may specify up to five selectors.
        :param fargate_profile_name: The name of the Fargate profile. Default: - generated
        :param pod_execution_role: The pod execution role to use for pods that match the selectors in the Fargate profile. The pod execution role allows Fargate infrastructure to register with your cluster as a node, and it provides read access to Amazon ECR image repositories. Default: - a role will be automatically created
        :param subnet_selection: Select which subnets to launch your pods into. At this time, pods running on Fargate are not assigned public IP addresses, so only private subnets (with no direct route to an Internet Gateway) are allowed. You must specify the VPC to customize the subnet selection Default: - all private subnets of the VPC are selected.
        :param vpc: The VPC from which to select subnets to launch your pods into. By default, all private subnets are selected. You can customize this using ``subnetSelection``. Default: - all private subnets used by the EKS cluster

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            cluster.add_fargate_profile("MyProfile",
                selectors=[eks.Selector(namespace="default")]
            )
        '''
        if isinstance(subnet_selection, dict):
            subnet_selection = _SubnetSelection_e57d76df(**subnet_selection)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0e8594b197fd7e55b0136bb091bd0b4603a55eaed0b794264a1fce4763848498)
            check_type(argname="argument selectors", value=selectors, expected_type=type_hints["selectors"])
            check_type(argname="argument fargate_profile_name", value=fargate_profile_name, expected_type=type_hints["fargate_profile_name"])
            check_type(argname="argument pod_execution_role", value=pod_execution_role, expected_type=type_hints["pod_execution_role"])
            check_type(argname="argument subnet_selection", value=subnet_selection, expected_type=type_hints["subnet_selection"])
            check_type(argname="argument vpc", value=vpc, expected_type=type_hints["vpc"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "selectors": selectors,
        }
        if fargate_profile_name is not None:
            self._values["fargate_profile_name"] = fargate_profile_name
        if pod_execution_role is not None:
            self._values["pod_execution_role"] = pod_execution_role
        if subnet_selection is not None:
            self._values["subnet_selection"] = subnet_selection
        if vpc is not None:
            self._values["vpc"] = vpc

    @builtins.property
    def selectors(self) -> typing.List["Selector"]:
        '''The selectors to match for pods to use this Fargate profile.

        Each selector
        must have an associated namespace. Optionally, you can also specify labels
        for a namespace.

        At least one selector is required and you may specify up to five selectors.
        '''
        result = self._values.get("selectors")
        assert result is not None, "Required property 'selectors' is missing"
        return typing.cast(typing.List["Selector"], result)

    @builtins.property
    def fargate_profile_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Fargate profile.

        :default: - generated
        '''
        result = self._values.get("fargate_profile_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def pod_execution_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The pod execution role to use for pods that match the selectors in the Fargate profile.

        The pod execution role allows Fargate infrastructure to
        register with your cluster as a node, and it provides read access to Amazon
        ECR image repositories.

        :default: - a role will be automatically created

        :see: https://docs.aws.amazon.com/eks/latest/userguide/pod-execution-role.html
        '''
        result = self._values.get("pod_execution_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def subnet_selection(self) -> typing.Optional[_SubnetSelection_e57d76df]:
        '''Select which subnets to launch your pods into.

        At this time, pods running
        on Fargate are not assigned public IP addresses, so only private subnets
        (with no direct route to an Internet Gateway) are allowed.

        You must specify the VPC to customize the subnet selection

        :default: - all private subnets of the VPC are selected.
        '''
        result = self._values.get("subnet_selection")
        return typing.cast(typing.Optional[_SubnetSelection_e57d76df], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_IVpc_f30d5663]:
        '''The VPC from which to select subnets to launch your pods into.

        By default, all private subnets are selected. You can customize this using
        ``subnetSelection``.

        :default: - all private subnets used by the EKS cluster
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_IVpc_f30d5663], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FargateProfileOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.FargateProfileProps",
    jsii_struct_bases=[FargateProfileOptions],
    name_mapping={
        "selectors": "selectors",
        "fargate_profile_name": "fargateProfileName",
        "pod_execution_role": "podExecutionRole",
        "subnet_selection": "subnetSelection",
        "vpc": "vpc",
        "cluster": "cluster",
    },
)
class FargateProfileProps(FargateProfileOptions):
    def __init__(
        self,
        *,
        selectors: typing.Sequence[typing.Union["Selector", typing.Dict[builtins.str, typing.Any]]],
        fargate_profile_name: typing.Optional[builtins.str] = None,
        pod_execution_role: typing.Optional[_IRole_235f5d8e] = None,
        subnet_selection: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
        vpc: typing.Optional[_IVpc_f30d5663] = None,
        cluster: "Cluster",
    ) -> None:
        '''Configuration props for EKS Fargate Profiles.

        :param selectors: The selectors to match for pods to use this Fargate profile. Each selector must have an associated namespace. Optionally, you can also specify labels for a namespace. At least one selector is required and you may specify up to five selectors.
        :param fargate_profile_name: The name of the Fargate profile. Default: - generated
        :param pod_execution_role: The pod execution role to use for pods that match the selectors in the Fargate profile. The pod execution role allows Fargate infrastructure to register with your cluster as a node, and it provides read access to Amazon ECR image repositories. Default: - a role will be automatically created
        :param subnet_selection: Select which subnets to launch your pods into. At this time, pods running on Fargate are not assigned public IP addresses, so only private subnets (with no direct route to an Internet Gateway) are allowed. You must specify the VPC to customize the subnet selection Default: - all private subnets of the VPC are selected.
        :param vpc: The VPC from which to select subnets to launch your pods into. By default, all private subnets are selected. You can customize this using ``subnetSelection``. Default: - all private subnets used by the EKS cluster
        :param cluster: The EKS cluster to apply the Fargate profile to. [disable-awslint:ref-via-interface]

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            eks.FargateProfile(self, "MyProfile",
                cluster=cluster,
                selectors=[eks.Selector(namespace="default")]
            )
        '''
        if isinstance(subnet_selection, dict):
            subnet_selection = _SubnetSelection_e57d76df(**subnet_selection)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1a8f283cd286303b5ae6cacf59d824959f30c127ab7d08994b28cb93caa74d7d)
            check_type(argname="argument selectors", value=selectors, expected_type=type_hints["selectors"])
            check_type(argname="argument fargate_profile_name", value=fargate_profile_name, expected_type=type_hints["fargate_profile_name"])
            check_type(argname="argument pod_execution_role", value=pod_execution_role, expected_type=type_hints["pod_execution_role"])
            check_type(argname="argument subnet_selection", value=subnet_selection, expected_type=type_hints["subnet_selection"])
            check_type(argname="argument vpc", value=vpc, expected_type=type_hints["vpc"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "selectors": selectors,
            "cluster": cluster,
        }
        if fargate_profile_name is not None:
            self._values["fargate_profile_name"] = fargate_profile_name
        if pod_execution_role is not None:
            self._values["pod_execution_role"] = pod_execution_role
        if subnet_selection is not None:
            self._values["subnet_selection"] = subnet_selection
        if vpc is not None:
            self._values["vpc"] = vpc

    @builtins.property
    def selectors(self) -> typing.List["Selector"]:
        '''The selectors to match for pods to use this Fargate profile.

        Each selector
        must have an associated namespace. Optionally, you can also specify labels
        for a namespace.

        At least one selector is required and you may specify up to five selectors.
        '''
        result = self._values.get("selectors")
        assert result is not None, "Required property 'selectors' is missing"
        return typing.cast(typing.List["Selector"], result)

    @builtins.property
    def fargate_profile_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Fargate profile.

        :default: - generated
        '''
        result = self._values.get("fargate_profile_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def pod_execution_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The pod execution role to use for pods that match the selectors in the Fargate profile.

        The pod execution role allows Fargate infrastructure to
        register with your cluster as a node, and it provides read access to Amazon
        ECR image repositories.

        :default: - a role will be automatically created

        :see: https://docs.aws.amazon.com/eks/latest/userguide/pod-execution-role.html
        '''
        result = self._values.get("pod_execution_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def subnet_selection(self) -> typing.Optional[_SubnetSelection_e57d76df]:
        '''Select which subnets to launch your pods into.

        At this time, pods running
        on Fargate are not assigned public IP addresses, so only private subnets
        (with no direct route to an Internet Gateway) are allowed.

        You must specify the VPC to customize the subnet selection

        :default: - all private subnets of the VPC are selected.
        '''
        result = self._values.get("subnet_selection")
        return typing.cast(typing.Optional[_SubnetSelection_e57d76df], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_IVpc_f30d5663]:
        '''The VPC from which to select subnets to launch your pods into.

        By default, all private subnets are selected. You can customize this using
        ``subnetSelection``.

        :default: - all private subnets used by the EKS cluster
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_IVpc_f30d5663], result)

    @builtins.property
    def cluster(self) -> "Cluster":
        '''The EKS cluster to apply the Fargate profile to.

        [disable-awslint:ref-via-interface]
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("Cluster", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FargateProfileProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class HelmChart(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.HelmChart",
):
    '''Represents a helm chart within the Kubernetes system.

    Applies/deletes the resources using ``kubectl`` in sync with the resource.

    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        # option 1: use a construct
        eks.HelmChart(self, "MyOCIChart",
            cluster=cluster,
            chart="some-chart",
            repository="oci://${ACCOUNT_ID}.dkr.ecr.${ACCOUNT_REGION}.amazonaws.com/${REPO_NAME}",
            namespace="oci",
            version="0.0.1"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: "ICluster",
        atomic: typing.Optional[builtins.bool] = None,
        chart: typing.Optional[builtins.str] = None,
        chart_asset: typing.Optional[_Asset_ac2a7e61] = None,
        create_namespace: typing.Optional[builtins.bool] = None,
        namespace: typing.Optional[builtins.str] = None,
        release: typing.Optional[builtins.str] = None,
        repository: typing.Optional[builtins.str] = None,
        skip_crds: typing.Optional[builtins.bool] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        values: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        version: typing.Optional[builtins.str] = None,
        wait: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster: The EKS cluster to apply this configuration to. [disable-awslint:ref-via-interface]
        :param atomic: Whether or not Helm should treat this operation as atomic; if set, upgrade process rolls back changes made in case of failed upgrade. The --wait flag will be set automatically if --atomic is used. Default: false
        :param chart: The name of the chart. Either this or ``chartAsset`` must be specified. Default: - No chart name. Implies ``chartAsset`` is used.
        :param chart_asset: The chart in the form of an asset. Either this or ``chart`` must be specified. Default: - No chart asset. Implies ``chart`` is used.
        :param create_namespace: create namespace if not exist. Default: true
        :param namespace: The Kubernetes namespace scope of the requests. Default: default
        :param release: The name of the release. Default: - If no release name is given, it will use the last 53 characters of the node's unique id.
        :param repository: The repository which contains the chart. For example: https://charts.helm.sh/stable/ Default: - No repository will be used, which means that the chart needs to be an absolute URL.
        :param skip_crds: if set, no CRDs will be installed. Default: - CRDs are installed if not already present
        :param timeout: Amount of time to wait for any individual Kubernetes operation. Maximum 15 minutes. Default: Duration.minutes(5)
        :param values: The values to be used by the chart. For nested values use a nested dictionary. For example: values: { installationCRDs: true, webhook: { port: 9443 } } Default: - No values are provided to the chart.
        :param version: The chart version to install. Default: - If this is not specified, the latest version is installed
        :param wait: Whether or not Helm should wait until all Pods, PVCs, Services, and minimum number of Pods of a Deployment, StatefulSet, or ReplicaSet are in a ready state before marking the release as successful. Default: - Helm will not wait before marking release as successful
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__52726bf866c12e59b4b3f451037674503c3d95da4a298075df121981ea0750fc)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = HelmChartProps(
            cluster=cluster,
            atomic=atomic,
            chart=chart,
            chart_asset=chart_asset,
            create_namespace=create_namespace,
            namespace=namespace,
            release=release,
            repository=repository,
            skip_crds=skip_crds,
            timeout=timeout,
            values=values,
            version=version,
            wait=wait,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.python.classproperty
    @jsii.member(jsii_name="RESOURCE_TYPE")
    def RESOURCE_TYPE(cls) -> builtins.str:
        '''The CloudFormation resource type.'''
        return typing.cast(builtins.str, jsii.sget(cls, "RESOURCE_TYPE"))

    @builtins.property
    @jsii.member(jsii_name="atomic")
    def atomic(self) -> typing.Optional[builtins.bool]:
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "atomic"))

    @builtins.property
    @jsii.member(jsii_name="chart")
    def chart(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "chart"))

    @builtins.property
    @jsii.member(jsii_name="chartAsset")
    def chart_asset(self) -> typing.Optional[_Asset_ac2a7e61]:
        return typing.cast(typing.Optional[_Asset_ac2a7e61], jsii.get(self, "chartAsset"))

    @builtins.property
    @jsii.member(jsii_name="repository")
    def repository(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "repository"))

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "version"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.HelmChartOptions",
    jsii_struct_bases=[],
    name_mapping={
        "atomic": "atomic",
        "chart": "chart",
        "chart_asset": "chartAsset",
        "create_namespace": "createNamespace",
        "namespace": "namespace",
        "release": "release",
        "repository": "repository",
        "skip_crds": "skipCrds",
        "timeout": "timeout",
        "values": "values",
        "version": "version",
        "wait": "wait",
    },
)
class HelmChartOptions:
    def __init__(
        self,
        *,
        atomic: typing.Optional[builtins.bool] = None,
        chart: typing.Optional[builtins.str] = None,
        chart_asset: typing.Optional[_Asset_ac2a7e61] = None,
        create_namespace: typing.Optional[builtins.bool] = None,
        namespace: typing.Optional[builtins.str] = None,
        release: typing.Optional[builtins.str] = None,
        repository: typing.Optional[builtins.str] = None,
        skip_crds: typing.Optional[builtins.bool] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        values: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        version: typing.Optional[builtins.str] = None,
        wait: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Helm Chart options.

        :param atomic: Whether or not Helm should treat this operation as atomic; if set, upgrade process rolls back changes made in case of failed upgrade. The --wait flag will be set automatically if --atomic is used. Default: false
        :param chart: The name of the chart. Either this or ``chartAsset`` must be specified. Default: - No chart name. Implies ``chartAsset`` is used.
        :param chart_asset: The chart in the form of an asset. Either this or ``chart`` must be specified. Default: - No chart asset. Implies ``chart`` is used.
        :param create_namespace: create namespace if not exist. Default: true
        :param namespace: The Kubernetes namespace scope of the requests. Default: default
        :param release: The name of the release. Default: - If no release name is given, it will use the last 53 characters of the node's unique id.
        :param repository: The repository which contains the chart. For example: https://charts.helm.sh/stable/ Default: - No repository will be used, which means that the chart needs to be an absolute URL.
        :param skip_crds: if set, no CRDs will be installed. Default: - CRDs are installed if not already present
        :param timeout: Amount of time to wait for any individual Kubernetes operation. Maximum 15 minutes. Default: Duration.minutes(5)
        :param values: The values to be used by the chart. For nested values use a nested dictionary. For example: values: { installationCRDs: true, webhook: { port: 9443 } } Default: - No values are provided to the chart.
        :param version: The chart version to install. Default: - If this is not specified, the latest version is installed
        :param wait: Whether or not Helm should wait until all Pods, PVCs, Services, and minimum number of Pods of a Deployment, StatefulSet, or ReplicaSet are in a ready state before marking the release as successful. Default: - Helm will not wait before marking release as successful

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_s3_assets as s3_assets
            
            # cluster: eks.Cluster
            
            chart_asset = s3_assets.Asset(self, "ChartAsset",
                path="/path/to/asset"
            )
            
            cluster.add_helm_chart("test-chart",
                chart_asset=chart_asset
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f6972a55b9cddfcfdae45e3907a04dc773d9fa367223106572ea64bf22eafe3d)
            check_type(argname="argument atomic", value=atomic, expected_type=type_hints["atomic"])
            check_type(argname="argument chart", value=chart, expected_type=type_hints["chart"])
            check_type(argname="argument chart_asset", value=chart_asset, expected_type=type_hints["chart_asset"])
            check_type(argname="argument create_namespace", value=create_namespace, expected_type=type_hints["create_namespace"])
            check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
            check_type(argname="argument release", value=release, expected_type=type_hints["release"])
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
            check_type(argname="argument skip_crds", value=skip_crds, expected_type=type_hints["skip_crds"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument values", value=values, expected_type=type_hints["values"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            check_type(argname="argument wait", value=wait, expected_type=type_hints["wait"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if atomic is not None:
            self._values["atomic"] = atomic
        if chart is not None:
            self._values["chart"] = chart
        if chart_asset is not None:
            self._values["chart_asset"] = chart_asset
        if create_namespace is not None:
            self._values["create_namespace"] = create_namespace
        if namespace is not None:
            self._values["namespace"] = namespace
        if release is not None:
            self._values["release"] = release
        if repository is not None:
            self._values["repository"] = repository
        if skip_crds is not None:
            self._values["skip_crds"] = skip_crds
        if timeout is not None:
            self._values["timeout"] = timeout
        if values is not None:
            self._values["values"] = values
        if version is not None:
            self._values["version"] = version
        if wait is not None:
            self._values["wait"] = wait

    @builtins.property
    def atomic(self) -> typing.Optional[builtins.bool]:
        '''Whether or not Helm should treat this operation as atomic;

        if set, upgrade process rolls back changes
        made in case of failed upgrade. The --wait flag will be set automatically if --atomic is used.

        :default: false
        '''
        result = self._values.get("atomic")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def chart(self) -> typing.Optional[builtins.str]:
        '''The name of the chart.

        Either this or ``chartAsset`` must be specified.

        :default: - No chart name. Implies ``chartAsset`` is used.
        '''
        result = self._values.get("chart")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def chart_asset(self) -> typing.Optional[_Asset_ac2a7e61]:
        '''The chart in the form of an asset.

        Either this or ``chart`` must be specified.

        :default: - No chart asset. Implies ``chart`` is used.
        '''
        result = self._values.get("chart_asset")
        return typing.cast(typing.Optional[_Asset_ac2a7e61], result)

    @builtins.property
    def create_namespace(self) -> typing.Optional[builtins.bool]:
        '''create namespace if not exist.

        :default: true
        '''
        result = self._values.get("create_namespace")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def namespace(self) -> typing.Optional[builtins.str]:
        '''The Kubernetes namespace scope of the requests.

        :default: default
        '''
        result = self._values.get("namespace")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def release(self) -> typing.Optional[builtins.str]:
        '''The name of the release.

        :default: - If no release name is given, it will use the last 53 characters of the node's unique id.
        '''
        result = self._values.get("release")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def repository(self) -> typing.Optional[builtins.str]:
        '''The repository which contains the chart.

        For example: https://charts.helm.sh/stable/

        :default: - No repository will be used, which means that the chart needs to be an absolute URL.
        '''
        result = self._values.get("repository")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def skip_crds(self) -> typing.Optional[builtins.bool]:
        '''if set, no CRDs will be installed.

        :default: - CRDs are installed if not already present
        '''
        result = self._values.get("skip_crds")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Amount of time to wait for any individual Kubernetes operation.

        Maximum 15 minutes.

        :default: Duration.minutes(5)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def values(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The values to be used by the chart.

        For nested values use a nested dictionary. For example:
        values: {
        installationCRDs: true,
        webhook: { port: 9443 }
        }

        :default: - No values are provided to the chart.
        '''
        result = self._values.get("values")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def version(self) -> typing.Optional[builtins.str]:
        '''The chart version to install.

        :default: - If this is not specified, the latest version is installed
        '''
        result = self._values.get("version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def wait(self) -> typing.Optional[builtins.bool]:
        '''Whether or not Helm should wait until all Pods, PVCs, Services, and minimum number of Pods of a Deployment, StatefulSet, or ReplicaSet are in a ready state before marking the release as successful.

        :default: - Helm will not wait before marking release as successful
        '''
        result = self._values.get("wait")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HelmChartOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.HelmChartProps",
    jsii_struct_bases=[HelmChartOptions],
    name_mapping={
        "atomic": "atomic",
        "chart": "chart",
        "chart_asset": "chartAsset",
        "create_namespace": "createNamespace",
        "namespace": "namespace",
        "release": "release",
        "repository": "repository",
        "skip_crds": "skipCrds",
        "timeout": "timeout",
        "values": "values",
        "version": "version",
        "wait": "wait",
        "cluster": "cluster",
    },
)
class HelmChartProps(HelmChartOptions):
    def __init__(
        self,
        *,
        atomic: typing.Optional[builtins.bool] = None,
        chart: typing.Optional[builtins.str] = None,
        chart_asset: typing.Optional[_Asset_ac2a7e61] = None,
        create_namespace: typing.Optional[builtins.bool] = None,
        namespace: typing.Optional[builtins.str] = None,
        release: typing.Optional[builtins.str] = None,
        repository: typing.Optional[builtins.str] = None,
        skip_crds: typing.Optional[builtins.bool] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        values: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        version: typing.Optional[builtins.str] = None,
        wait: typing.Optional[builtins.bool] = None,
        cluster: "ICluster",
    ) -> None:
        '''Helm Chart properties.

        :param atomic: Whether or not Helm should treat this operation as atomic; if set, upgrade process rolls back changes made in case of failed upgrade. The --wait flag will be set automatically if --atomic is used. Default: false
        :param chart: The name of the chart. Either this or ``chartAsset`` must be specified. Default: - No chart name. Implies ``chartAsset`` is used.
        :param chart_asset: The chart in the form of an asset. Either this or ``chart`` must be specified. Default: - No chart asset. Implies ``chart`` is used.
        :param create_namespace: create namespace if not exist. Default: true
        :param namespace: The Kubernetes namespace scope of the requests. Default: default
        :param release: The name of the release. Default: - If no release name is given, it will use the last 53 characters of the node's unique id.
        :param repository: The repository which contains the chart. For example: https://charts.helm.sh/stable/ Default: - No repository will be used, which means that the chart needs to be an absolute URL.
        :param skip_crds: if set, no CRDs will be installed. Default: - CRDs are installed if not already present
        :param timeout: Amount of time to wait for any individual Kubernetes operation. Maximum 15 minutes. Default: Duration.minutes(5)
        :param values: The values to be used by the chart. For nested values use a nested dictionary. For example: values: { installationCRDs: true, webhook: { port: 9443 } } Default: - No values are provided to the chart.
        :param version: The chart version to install. Default: - If this is not specified, the latest version is installed
        :param wait: Whether or not Helm should wait until all Pods, PVCs, Services, and minimum number of Pods of a Deployment, StatefulSet, or ReplicaSet are in a ready state before marking the release as successful. Default: - Helm will not wait before marking release as successful
        :param cluster: The EKS cluster to apply this configuration to. [disable-awslint:ref-via-interface]

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            # option 1: use a construct
            eks.HelmChart(self, "MyOCIChart",
                cluster=cluster,
                chart="some-chart",
                repository="oci://${ACCOUNT_ID}.dkr.ecr.${ACCOUNT_REGION}.amazonaws.com/${REPO_NAME}",
                namespace="oci",
                version="0.0.1"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3cca1be87809b190e4c5e436f984009d3dce2043a4fb17be5ea74ad9b58debef)
            check_type(argname="argument atomic", value=atomic, expected_type=type_hints["atomic"])
            check_type(argname="argument chart", value=chart, expected_type=type_hints["chart"])
            check_type(argname="argument chart_asset", value=chart_asset, expected_type=type_hints["chart_asset"])
            check_type(argname="argument create_namespace", value=create_namespace, expected_type=type_hints["create_namespace"])
            check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
            check_type(argname="argument release", value=release, expected_type=type_hints["release"])
            check_type(argname="argument repository", value=repository, expected_type=type_hints["repository"])
            check_type(argname="argument skip_crds", value=skip_crds, expected_type=type_hints["skip_crds"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument values", value=values, expected_type=type_hints["values"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            check_type(argname="argument wait", value=wait, expected_type=type_hints["wait"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
        }
        if atomic is not None:
            self._values["atomic"] = atomic
        if chart is not None:
            self._values["chart"] = chart
        if chart_asset is not None:
            self._values["chart_asset"] = chart_asset
        if create_namespace is not None:
            self._values["create_namespace"] = create_namespace
        if namespace is not None:
            self._values["namespace"] = namespace
        if release is not None:
            self._values["release"] = release
        if repository is not None:
            self._values["repository"] = repository
        if skip_crds is not None:
            self._values["skip_crds"] = skip_crds
        if timeout is not None:
            self._values["timeout"] = timeout
        if values is not None:
            self._values["values"] = values
        if version is not None:
            self._values["version"] = version
        if wait is not None:
            self._values["wait"] = wait

    @builtins.property
    def atomic(self) -> typing.Optional[builtins.bool]:
        '''Whether or not Helm should treat this operation as atomic;

        if set, upgrade process rolls back changes
        made in case of failed upgrade. The --wait flag will be set automatically if --atomic is used.

        :default: false
        '''
        result = self._values.get("atomic")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def chart(self) -> typing.Optional[builtins.str]:
        '''The name of the chart.

        Either this or ``chartAsset`` must be specified.

        :default: - No chart name. Implies ``chartAsset`` is used.
        '''
        result = self._values.get("chart")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def chart_asset(self) -> typing.Optional[_Asset_ac2a7e61]:
        '''The chart in the form of an asset.

        Either this or ``chart`` must be specified.

        :default: - No chart asset. Implies ``chart`` is used.
        '''
        result = self._values.get("chart_asset")
        return typing.cast(typing.Optional[_Asset_ac2a7e61], result)

    @builtins.property
    def create_namespace(self) -> typing.Optional[builtins.bool]:
        '''create namespace if not exist.

        :default: true
        '''
        result = self._values.get("create_namespace")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def namespace(self) -> typing.Optional[builtins.str]:
        '''The Kubernetes namespace scope of the requests.

        :default: default
        '''
        result = self._values.get("namespace")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def release(self) -> typing.Optional[builtins.str]:
        '''The name of the release.

        :default: - If no release name is given, it will use the last 53 characters of the node's unique id.
        '''
        result = self._values.get("release")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def repository(self) -> typing.Optional[builtins.str]:
        '''The repository which contains the chart.

        For example: https://charts.helm.sh/stable/

        :default: - No repository will be used, which means that the chart needs to be an absolute URL.
        '''
        result = self._values.get("repository")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def skip_crds(self) -> typing.Optional[builtins.bool]:
        '''if set, no CRDs will be installed.

        :default: - CRDs are installed if not already present
        '''
        result = self._values.get("skip_crds")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Amount of time to wait for any individual Kubernetes operation.

        Maximum 15 minutes.

        :default: Duration.minutes(5)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def values(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The values to be used by the chart.

        For nested values use a nested dictionary. For example:
        values: {
        installationCRDs: true,
        webhook: { port: 9443 }
        }

        :default: - No values are provided to the chart.
        '''
        result = self._values.get("values")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def version(self) -> typing.Optional[builtins.str]:
        '''The chart version to install.

        :default: - If this is not specified, the latest version is installed
        '''
        result = self._values.get("version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def wait(self) -> typing.Optional[builtins.bool]:
        '''Whether or not Helm should wait until all Pods, PVCs, Services, and minimum number of Pods of a Deployment, StatefulSet, or ReplicaSet are in a ready state before marking the release as successful.

        :default: - Helm will not wait before marking release as successful
        '''
        result = self._values.get("wait")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cluster(self) -> "ICluster":
        '''The EKS cluster to apply this configuration to.

        [disable-awslint:ref-via-interface]
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HelmChartProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="aws-cdk-lib.aws_eks.IAccessEntry")
class IAccessEntry(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents an access entry in an Amazon EKS cluster.

    An access entry defines the permissions and scope for a user or role to access an Amazon EKS cluster.

    :extends: IResource *
    :interface: IAccessEntry
    :property: {string} accessEntryArn - The Amazon Resource Name (ARN) of the access entry.
    '''

    @builtins.property
    @jsii.member(jsii_name="accessEntryArn")
    def access_entry_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the access entry.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="accessEntryName")
    def access_entry_name(self) -> builtins.str:
        '''The name of the access entry.

        :attribute: true
        '''
        ...


class _IAccessEntryProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents an access entry in an Amazon EKS cluster.

    An access entry defines the permissions and scope for a user or role to access an Amazon EKS cluster.

    :extends: IResource *
    :interface: IAccessEntry
    :property: {string} accessEntryArn - The Amazon Resource Name (ARN) of the access entry.
    '''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_eks.IAccessEntry"

    @builtins.property
    @jsii.member(jsii_name="accessEntryArn")
    def access_entry_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the access entry.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "accessEntryArn"))

    @builtins.property
    @jsii.member(jsii_name="accessEntryName")
    def access_entry_name(self) -> builtins.str:
        '''The name of the access entry.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "accessEntryName"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAccessEntry).__jsii_proxy_class__ = lambda : _IAccessEntryProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_eks.IAccessPolicy")
class IAccessPolicy(typing_extensions.Protocol):
    '''Represents an access policy that defines the permissions and scope for a user or role to access an Amazon EKS cluster.

    :interface: IAccessPolicy
    '''

    @builtins.property
    @jsii.member(jsii_name="accessScope")
    def access_scope(self) -> AccessScope:
        '''The scope of the access policy, which determines the level of access granted.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="policy")
    def policy(self) -> builtins.str:
        '''The access policy itself, which defines the specific permissions.'''
        ...


class _IAccessPolicyProxy:
    '''Represents an access policy that defines the permissions and scope for a user or role to access an Amazon EKS cluster.

    :interface: IAccessPolicy
    '''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_eks.IAccessPolicy"

    @builtins.property
    @jsii.member(jsii_name="accessScope")
    def access_scope(self) -> AccessScope:
        '''The scope of the access policy, which determines the level of access granted.'''
        return typing.cast(AccessScope, jsii.get(self, "accessScope"))

    @builtins.property
    @jsii.member(jsii_name="policy")
    def policy(self) -> builtins.str:
        '''The access policy itself, which defines the specific permissions.'''
        return typing.cast(builtins.str, jsii.get(self, "policy"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAccessPolicy).__jsii_proxy_class__ = lambda : _IAccessPolicyProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_eks.IAddon")
class IAddon(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents an Amazon EKS Add-On.'''

    @builtins.property
    @jsii.member(jsii_name="addonArn")
    def addon_arn(self) -> builtins.str:
        '''ARN of the Add-On.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="addonName")
    def addon_name(self) -> builtins.str:
        '''Name of the Add-On.

        :attribute: true
        '''
        ...


class _IAddonProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents an Amazon EKS Add-On.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_eks.IAddon"

    @builtins.property
    @jsii.member(jsii_name="addonArn")
    def addon_arn(self) -> builtins.str:
        '''ARN of the Add-On.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "addonArn"))

    @builtins.property
    @jsii.member(jsii_name="addonName")
    def addon_name(self) -> builtins.str:
        '''Name of the Add-On.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "addonName"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAddon).__jsii_proxy_class__ = lambda : _IAddonProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_eks.ICluster")
class ICluster(_IResource_c80c4260, _IConnectable_10015a05, typing_extensions.Protocol):
    '''An EKS cluster.'''

    @builtins.property
    @jsii.member(jsii_name="clusterArn")
    def cluster_arn(self) -> builtins.str:
        '''The unique ARN assigned to the service by AWS in the form of arn:aws:eks:.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="clusterCertificateAuthorityData")
    def cluster_certificate_authority_data(self) -> builtins.str:
        '''The certificate-authority-data for your cluster.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="clusterEncryptionConfigKeyArn")
    def cluster_encryption_config_key_arn(self) -> builtins.str:
        '''Amazon Resource Name (ARN) or alias of the customer master key (CMK).

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="clusterEndpoint")
    def cluster_endpoint(self) -> builtins.str:
        '''The API Server endpoint URL.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''The physical name of the Cluster.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="clusterSecurityGroup")
    def cluster_security_group(self) -> _ISecurityGroup_acf8a799:
        '''The cluster security group that was created by Amazon EKS for the cluster.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="clusterSecurityGroupId")
    def cluster_security_group_id(self) -> builtins.str:
        '''The id of the cluster security group that was created by Amazon EKS for the cluster.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="openIdConnectProvider")
    def open_id_connect_provider(self) -> _IOpenIdConnectProvider_203f0793:
        '''The Open ID Connect Provider of the cluster used to configure Service Accounts.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="prune")
    def prune(self) -> builtins.bool:
        '''Indicates whether Kubernetes resources can be automatically pruned.

        When
        this is enabled (default), prune labels will be allocated and injected to
        each resource. These labels will then be used when issuing the ``kubectl apply`` operation with the ``--prune`` switch.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="vpc")
    def vpc(self) -> _IVpc_f30d5663:
        '''The VPC in which this Cluster was created.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="authenticationMode")
    def authentication_mode(self) -> typing.Optional[AuthenticationMode]:
        '''The authentication mode for the cluster.

        :default: AuthenticationMode.CONFIG_MAP
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="awscliLayer")
    def awscli_layer(self) -> typing.Optional[_ILayerVersion_5ac127c8]:
        '''An AWS Lambda layer that contains the ``aws`` CLI.

        If not defined, a default layer will be used containing the AWS CLI 1.x.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="clusterHandlerSecurityGroup")
    def cluster_handler_security_group(
        self,
    ) -> typing.Optional[_ISecurityGroup_acf8a799]:
        '''A security group to associate with the Cluster Handler's Lambdas.

        The Cluster Handler's Lambdas are responsible for calling AWS's EKS API.

        Requires ``placeClusterHandlerInVpc`` to be set to true.

        :default: - No security group.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="eksPodIdentityAgent")
    def eks_pod_identity_agent(self) -> typing.Optional[IAddon]:
        '''The EKS Pod Identity Agent addon for the EKS cluster.

        The EKS Pod Identity Agent is responsible for managing the temporary credentials
        used by pods in the cluster to access AWS resources. It runs as a DaemonSet on
        each node and provides the necessary credentials to the pods based on their
        associated service account.

        This property returns the ``CfnAddon`` resource representing the EKS Pod Identity
        Agent addon. If the addon has not been created yet, it will be created and
        returned.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="ipFamily")
    def ip_family(self) -> typing.Optional["IpFamily"]:
        '''Specify which IP family is used to assign Kubernetes pod and service IP addresses.

        :default: - IpFamily.IP_V4

        :see: https://docs.aws.amazon.com/eks/latest/APIReference/API_KubernetesNetworkConfigRequest.html#AmazonEKS-Type-KubernetesNetworkConfigRequest-ipFamily
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="kubectlEnvironment")
    def kubectl_environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Custom environment variables when running ``kubectl`` against this cluster.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="kubectlLambdaRole")
    def kubectl_lambda_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''An IAM role that can perform kubectl operations against this cluster.

        The role should be mapped to the ``system:masters`` Kubernetes RBAC role.

        This role is directly passed to the lambda handler that sends Kube Ctl commands to the cluster.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="kubectlLayer")
    def kubectl_layer(self) -> typing.Optional[_ILayerVersion_5ac127c8]:
        '''An AWS Lambda layer that includes ``kubectl`` and ``helm``.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="kubectlMemory")
    def kubectl_memory(self) -> typing.Optional[_Size_7b441c34]:
        '''Amount of memory to allocate to the provider's lambda function.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="kubectlPrivateSubnets")
    def kubectl_private_subnets(
        self,
    ) -> typing.Optional[typing.List[_ISubnet_d57d1229]]:
        '''Subnets to host the ``kubectl`` compute resources.

        If this is undefined, the k8s endpoint is expected to be accessible
        publicly.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="kubectlProvider")
    def kubectl_provider(self) -> typing.Optional["IKubectlProvider"]:
        '''Kubectl Provider for issuing kubectl commands against it.

        If not defined, a default provider will be used
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="kubectlRole")
    def kubectl_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''An IAM role that can perform kubectl operations against this cluster.

        The role should be mapped to the ``system:masters`` Kubernetes RBAC role.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="kubectlSecurityGroup")
    def kubectl_security_group(self) -> typing.Optional[_ISecurityGroup_acf8a799]:
        '''A security group to use for ``kubectl`` execution.

        If this is undefined, the k8s endpoint is expected to be accessible
        publicly.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="onEventLayer")
    def on_event_layer(self) -> typing.Optional[_ILayerVersion_5ac127c8]:
        '''An AWS Lambda layer that includes the NPM dependency ``proxy-agent``.

        If not defined, a default layer will be used.
        '''
        ...

    @jsii.member(jsii_name="addCdk8sChart")
    def add_cdk8s_chart(
        self,
        id: builtins.str,
        chart: _constructs_77d1e7e8.Construct,
        *,
        ingress_alb: typing.Optional[builtins.bool] = None,
        ingress_alb_scheme: typing.Optional[AlbScheme] = None,
        prune: typing.Optional[builtins.bool] = None,
        skip_validation: typing.Optional[builtins.bool] = None,
    ) -> "KubernetesManifest":
        '''Defines a CDK8s chart in this cluster.

        :param id: logical id of this chart.
        :param chart: the cdk8s chart.
        :param ingress_alb: Automatically detect ``Ingress`` resources in the manifest and annotate them so they are picked up by an ALB Ingress Controller. Default: false
        :param ingress_alb_scheme: Specify the ALB scheme that should be applied to ``Ingress`` resources. Only applicable if ``ingressAlb`` is set to ``true``. Default: AlbScheme.INTERNAL
        :param prune: When a resource is removed from a Kubernetes manifest, it no longer appears in the manifest, and there is no way to know that this resource needs to be deleted. To address this, ``kubectl apply`` has a ``--prune`` option which will query the cluster for all resources with a specific label and will remove all the labeld resources that are not part of the applied manifest. If this option is disabled and a resource is removed, it will become "orphaned" and will not be deleted from the cluster. When this option is enabled (default), the construct will inject a label to all Kubernetes resources included in this manifest which will be used to prune resources when the manifest changes via ``kubectl apply --prune``. The label name will be ``aws.cdk.eks/prune-<ADDR>`` where ``<ADDR>`` is the 42-char unique address of this construct in the construct tree. Value is empty. Default: - based on the prune option of the cluster, which is ``true`` unless otherwise specified.
        :param skip_validation: A flag to signify if the manifest validation should be skipped. Default: false

        :return: a ``KubernetesManifest`` construct representing the chart.
        '''
        ...

    @jsii.member(jsii_name="addHelmChart")
    def add_helm_chart(
        self,
        id: builtins.str,
        *,
        atomic: typing.Optional[builtins.bool] = None,
        chart: typing.Optional[builtins.str] = None,
        chart_asset: typing.Optional[_Asset_ac2a7e61] = None,
        create_namespace: typing.Optional[builtins.bool] = None,
        namespace: typing.Optional[builtins.str] = None,
        release: typing.Optional[builtins.str] = None,
        repository: typing.Optional[builtins.str] = None,
        skip_crds: typing.Optional[builtins.bool] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        values: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        version: typing.Optional[builtins.str] = None,
        wait: typing.Optional[builtins.bool] = None,
    ) -> HelmChart:
        '''Defines a Helm chart in this cluster.

        :param id: logical id of this chart.
        :param atomic: Whether or not Helm should treat this operation as atomic; if set, upgrade process rolls back changes made in case of failed upgrade. The --wait flag will be set automatically if --atomic is used. Default: false
        :param chart: The name of the chart. Either this or ``chartAsset`` must be specified. Default: - No chart name. Implies ``chartAsset`` is used.
        :param chart_asset: The chart in the form of an asset. Either this or ``chart`` must be specified. Default: - No chart asset. Implies ``chart`` is used.
        :param create_namespace: create namespace if not exist. Default: true
        :param namespace: The Kubernetes namespace scope of the requests. Default: default
        :param release: The name of the release. Default: - If no release name is given, it will use the last 53 characters of the node's unique id.
        :param repository: The repository which contains the chart. For example: https://charts.helm.sh/stable/ Default: - No repository will be used, which means that the chart needs to be an absolute URL.
        :param skip_crds: if set, no CRDs will be installed. Default: - CRDs are installed if not already present
        :param timeout: Amount of time to wait for any individual Kubernetes operation. Maximum 15 minutes. Default: Duration.minutes(5)
        :param values: The values to be used by the chart. For nested values use a nested dictionary. For example: values: { installationCRDs: true, webhook: { port: 9443 } } Default: - No values are provided to the chart.
        :param version: The chart version to install. Default: - If this is not specified, the latest version is installed
        :param wait: Whether or not Helm should wait until all Pods, PVCs, Services, and minimum number of Pods of a Deployment, StatefulSet, or ReplicaSet are in a ready state before marking the release as successful. Default: - Helm will not wait before marking release as successful

        :return: a ``HelmChart`` construct
        '''
        ...

    @jsii.member(jsii_name="addManifest")
    def add_manifest(
        self,
        id: builtins.str,
        *manifest: typing.Mapping[builtins.str, typing.Any],
    ) -> "KubernetesManifest":
        '''Defines a Kubernetes resource in this cluster.

        The manifest will be applied/deleted using kubectl as needed.

        :param id: logical id of this manifest.
        :param manifest: a list of Kubernetes resource specifications.

        :return: a ``KubernetesManifest`` object.
        '''
        ...

    @jsii.member(jsii_name="addServiceAccount")
    def add_service_account(
        self,
        id: builtins.str,
        *,
        annotations: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        identity_type: typing.Optional["IdentityType"] = None,
        labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        name: typing.Optional[builtins.str] = None,
        namespace: typing.Optional[builtins.str] = None,
    ) -> "ServiceAccount":
        '''Creates a new service account with corresponding IAM Role (IRSA).

        :param id: logical id of service account.
        :param annotations: Additional annotations of the service account. Default: - no additional annotations
        :param identity_type: The identity type to use for the service account. Default: IdentityType.IRSA
        :param labels: Additional labels of the service account. Default: - no additional labels
        :param name: The name of the service account. The name of a ServiceAccount object must be a valid DNS subdomain name. https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/ Default: - If no name is given, it will use the id of the resource.
        :param namespace: The namespace of the service account. All namespace names must be valid RFC 1123 DNS labels. https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/#namespaces-and-dns Default: "default"
        '''
        ...

    @jsii.member(jsii_name="connectAutoScalingGroupCapacity")
    def connect_auto_scaling_group_capacity(
        self,
        auto_scaling_group: _AutoScalingGroup_c547a7b9,
        *,
        bootstrap_enabled: typing.Optional[builtins.bool] = None,
        bootstrap_options: typing.Optional[typing.Union[BootstrapOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        map_role: typing.Optional[builtins.bool] = None,
        spot_interrupt_handler: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Connect capacity in the form of an existing AutoScalingGroup to the EKS cluster.

        The AutoScalingGroup must be running an EKS-optimized AMI containing the
        /etc/eks/bootstrap.sh script. This method will configure Security Groups,
        add the right policies to the instance role, apply the right tags, and add
        the required user data to the instance's launch configuration.

        Spot instances will be labeled ``lifecycle=Ec2Spot`` and tainted with ``PreferNoSchedule``.
        If kubectl is enabled, the
        `spot interrupt handler <https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler>`_
        daemon will be installed on all spot instances to handle
        `EC2 Spot Instance Termination Notices <https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/>`_.

        Prefer to use ``addAutoScalingGroupCapacity`` if possible.

        :param auto_scaling_group: [disable-awslint:ref-via-interface].
        :param bootstrap_enabled: Configures the EC2 user-data script for instances in this autoscaling group to bootstrap the node (invoke ``/etc/eks/bootstrap.sh``) and associate it with the EKS cluster. If you wish to provide a custom user data script, set this to ``false`` and manually invoke ``autoscalingGroup.addUserData()``. Default: true
        :param bootstrap_options: Allows options for node bootstrapping through EC2 user data. Default: - default options
        :param machine_image_type: Allow options to specify different machine image type. Default: MachineImageType.AMAZON_LINUX_2
        :param map_role: Will automatically update the aws-auth ConfigMap to map the IAM instance role to RBAC. This cannot be explicitly set to ``true`` if the cluster has kubectl disabled. Default: - true if the cluster has kubectl enabled (which is the default).
        :param spot_interrupt_handler: Installs the AWS spot instance interrupt handler on the cluster if it's not already added. Only relevant if ``spotPrice`` is configured on the auto-scaling group. Default: true

        :see: https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
        '''
        ...


class _IClusterProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
    jsii.proxy_for(_IConnectable_10015a05), # type: ignore[misc]
):
    '''An EKS cluster.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_eks.ICluster"

    @builtins.property
    @jsii.member(jsii_name="clusterArn")
    def cluster_arn(self) -> builtins.str:
        '''The unique ARN assigned to the service by AWS in the form of arn:aws:eks:.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "clusterArn"))

    @builtins.property
    @jsii.member(jsii_name="clusterCertificateAuthorityData")
    def cluster_certificate_authority_data(self) -> builtins.str:
        '''The certificate-authority-data for your cluster.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "clusterCertificateAuthorityData"))

    @builtins.property
    @jsii.member(jsii_name="clusterEncryptionConfigKeyArn")
    def cluster_encryption_config_key_arn(self) -> builtins.str:
        '''Amazon Resource Name (ARN) or alias of the customer master key (CMK).

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "clusterEncryptionConfigKeyArn"))

    @builtins.property
    @jsii.member(jsii_name="clusterEndpoint")
    def cluster_endpoint(self) -> builtins.str:
        '''The API Server endpoint URL.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "clusterEndpoint"))

    @builtins.property
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''The physical name of the Cluster.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "clusterName"))

    @builtins.property
    @jsii.member(jsii_name="clusterSecurityGroup")
    def cluster_security_group(self) -> _ISecurityGroup_acf8a799:
        '''The cluster security group that was created by Amazon EKS for the cluster.

        :attribute: true
        '''
        return typing.cast(_ISecurityGroup_acf8a799, jsii.get(self, "clusterSecurityGroup"))

    @builtins.property
    @jsii.member(jsii_name="clusterSecurityGroupId")
    def cluster_security_group_id(self) -> builtins.str:
        '''The id of the cluster security group that was created by Amazon EKS for the cluster.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "clusterSecurityGroupId"))

    @builtins.property
    @jsii.member(jsii_name="openIdConnectProvider")
    def open_id_connect_provider(self) -> _IOpenIdConnectProvider_203f0793:
        '''The Open ID Connect Provider of the cluster used to configure Service Accounts.'''
        return typing.cast(_IOpenIdConnectProvider_203f0793, jsii.get(self, "openIdConnectProvider"))

    @builtins.property
    @jsii.member(jsii_name="prune")
    def prune(self) -> builtins.bool:
        '''Indicates whether Kubernetes resources can be automatically pruned.

        When
        this is enabled (default), prune labels will be allocated and injected to
        each resource. These labels will then be used when issuing the ``kubectl apply`` operation with the ``--prune`` switch.
        '''
        return typing.cast(builtins.bool, jsii.get(self, "prune"))

    @builtins.property
    @jsii.member(jsii_name="vpc")
    def vpc(self) -> _IVpc_f30d5663:
        '''The VPC in which this Cluster was created.'''
        return typing.cast(_IVpc_f30d5663, jsii.get(self, "vpc"))

    @builtins.property
    @jsii.member(jsii_name="authenticationMode")
    def authentication_mode(self) -> typing.Optional[AuthenticationMode]:
        '''The authentication mode for the cluster.

        :default: AuthenticationMode.CONFIG_MAP
        '''
        return typing.cast(typing.Optional[AuthenticationMode], jsii.get(self, "authenticationMode"))

    @builtins.property
    @jsii.member(jsii_name="awscliLayer")
    def awscli_layer(self) -> typing.Optional[_ILayerVersion_5ac127c8]:
        '''An AWS Lambda layer that contains the ``aws`` CLI.

        If not defined, a default layer will be used containing the AWS CLI 1.x.
        '''
        return typing.cast(typing.Optional[_ILayerVersion_5ac127c8], jsii.get(self, "awscliLayer"))

    @builtins.property
    @jsii.member(jsii_name="clusterHandlerSecurityGroup")
    def cluster_handler_security_group(
        self,
    ) -> typing.Optional[_ISecurityGroup_acf8a799]:
        '''A security group to associate with the Cluster Handler's Lambdas.

        The Cluster Handler's Lambdas are responsible for calling AWS's EKS API.

        Requires ``placeClusterHandlerInVpc`` to be set to true.

        :default: - No security group.

        :attribute: true
        '''
        return typing.cast(typing.Optional[_ISecurityGroup_acf8a799], jsii.get(self, "clusterHandlerSecurityGroup"))

    @builtins.property
    @jsii.member(jsii_name="eksPodIdentityAgent")
    def eks_pod_identity_agent(self) -> typing.Optional[IAddon]:
        '''The EKS Pod Identity Agent addon for the EKS cluster.

        The EKS Pod Identity Agent is responsible for managing the temporary credentials
        used by pods in the cluster to access AWS resources. It runs as a DaemonSet on
        each node and provides the necessary credentials to the pods based on their
        associated service account.

        This property returns the ``CfnAddon`` resource representing the EKS Pod Identity
        Agent addon. If the addon has not been created yet, it will be created and
        returned.
        '''
        return typing.cast(typing.Optional[IAddon], jsii.get(self, "eksPodIdentityAgent"))

    @builtins.property
    @jsii.member(jsii_name="ipFamily")
    def ip_family(self) -> typing.Optional["IpFamily"]:
        '''Specify which IP family is used to assign Kubernetes pod and service IP addresses.

        :default: - IpFamily.IP_V4

        :see: https://docs.aws.amazon.com/eks/latest/APIReference/API_KubernetesNetworkConfigRequest.html#AmazonEKS-Type-KubernetesNetworkConfigRequest-ipFamily
        '''
        return typing.cast(typing.Optional["IpFamily"], jsii.get(self, "ipFamily"))

    @builtins.property
    @jsii.member(jsii_name="kubectlEnvironment")
    def kubectl_environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Custom environment variables when running ``kubectl`` against this cluster.'''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "kubectlEnvironment"))

    @builtins.property
    @jsii.member(jsii_name="kubectlLambdaRole")
    def kubectl_lambda_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''An IAM role that can perform kubectl operations against this cluster.

        The role should be mapped to the ``system:masters`` Kubernetes RBAC role.

        This role is directly passed to the lambda handler that sends Kube Ctl commands to the cluster.
        '''
        return typing.cast(typing.Optional[_IRole_235f5d8e], jsii.get(self, "kubectlLambdaRole"))

    @builtins.property
    @jsii.member(jsii_name="kubectlLayer")
    def kubectl_layer(self) -> typing.Optional[_ILayerVersion_5ac127c8]:
        '''An AWS Lambda layer that includes ``kubectl`` and ``helm``.'''
        return typing.cast(typing.Optional[_ILayerVersion_5ac127c8], jsii.get(self, "kubectlLayer"))

    @builtins.property
    @jsii.member(jsii_name="kubectlMemory")
    def kubectl_memory(self) -> typing.Optional[_Size_7b441c34]:
        '''Amount of memory to allocate to the provider's lambda function.'''
        return typing.cast(typing.Optional[_Size_7b441c34], jsii.get(self, "kubectlMemory"))

    @builtins.property
    @jsii.member(jsii_name="kubectlPrivateSubnets")
    def kubectl_private_subnets(
        self,
    ) -> typing.Optional[typing.List[_ISubnet_d57d1229]]:
        '''Subnets to host the ``kubectl`` compute resources.

        If this is undefined, the k8s endpoint is expected to be accessible
        publicly.
        '''
        return typing.cast(typing.Optional[typing.List[_ISubnet_d57d1229]], jsii.get(self, "kubectlPrivateSubnets"))

    @builtins.property
    @jsii.member(jsii_name="kubectlProvider")
    def kubectl_provider(self) -> typing.Optional["IKubectlProvider"]:
        '''Kubectl Provider for issuing kubectl commands against it.

        If not defined, a default provider will be used
        '''
        return typing.cast(typing.Optional["IKubectlProvider"], jsii.get(self, "kubectlProvider"))

    @builtins.property
    @jsii.member(jsii_name="kubectlRole")
    def kubectl_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''An IAM role that can perform kubectl operations against this cluster.

        The role should be mapped to the ``system:masters`` Kubernetes RBAC role.
        '''
        return typing.cast(typing.Optional[_IRole_235f5d8e], jsii.get(self, "kubectlRole"))

    @builtins.property
    @jsii.member(jsii_name="kubectlSecurityGroup")
    def kubectl_security_group(self) -> typing.Optional[_ISecurityGroup_acf8a799]:
        '''A security group to use for ``kubectl`` execution.

        If this is undefined, the k8s endpoint is expected to be accessible
        publicly.
        '''
        return typing.cast(typing.Optional[_ISecurityGroup_acf8a799], jsii.get(self, "kubectlSecurityGroup"))

    @builtins.property
    @jsii.member(jsii_name="onEventLayer")
    def on_event_layer(self) -> typing.Optional[_ILayerVersion_5ac127c8]:
        '''An AWS Lambda layer that includes the NPM dependency ``proxy-agent``.

        If not defined, a default layer will be used.
        '''
        return typing.cast(typing.Optional[_ILayerVersion_5ac127c8], jsii.get(self, "onEventLayer"))

    @jsii.member(jsii_name="addCdk8sChart")
    def add_cdk8s_chart(
        self,
        id: builtins.str,
        chart: _constructs_77d1e7e8.Construct,
        *,
        ingress_alb: typing.Optional[builtins.bool] = None,
        ingress_alb_scheme: typing.Optional[AlbScheme] = None,
        prune: typing.Optional[builtins.bool] = None,
        skip_validation: typing.Optional[builtins.bool] = None,
    ) -> "KubernetesManifest":
        '''Defines a CDK8s chart in this cluster.

        :param id: logical id of this chart.
        :param chart: the cdk8s chart.
        :param ingress_alb: Automatically detect ``Ingress`` resources in the manifest and annotate them so they are picked up by an ALB Ingress Controller. Default: false
        :param ingress_alb_scheme: Specify the ALB scheme that should be applied to ``Ingress`` resources. Only applicable if ``ingressAlb`` is set to ``true``. Default: AlbScheme.INTERNAL
        :param prune: When a resource is removed from a Kubernetes manifest, it no longer appears in the manifest, and there is no way to know that this resource needs to be deleted. To address this, ``kubectl apply`` has a ``--prune`` option which will query the cluster for all resources with a specific label and will remove all the labeld resources that are not part of the applied manifest. If this option is disabled and a resource is removed, it will become "orphaned" and will not be deleted from the cluster. When this option is enabled (default), the construct will inject a label to all Kubernetes resources included in this manifest which will be used to prune resources when the manifest changes via ``kubectl apply --prune``. The label name will be ``aws.cdk.eks/prune-<ADDR>`` where ``<ADDR>`` is the 42-char unique address of this construct in the construct tree. Value is empty. Default: - based on the prune option of the cluster, which is ``true`` unless otherwise specified.
        :param skip_validation: A flag to signify if the manifest validation should be skipped. Default: false

        :return: a ``KubernetesManifest`` construct representing the chart.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__027a1f8770d590b40eccba041afe2250c0b1eab8a8d7e20ca1928a01a693e731)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument chart", value=chart, expected_type=type_hints["chart"])
        options = KubernetesManifestOptions(
            ingress_alb=ingress_alb,
            ingress_alb_scheme=ingress_alb_scheme,
            prune=prune,
            skip_validation=skip_validation,
        )

        return typing.cast("KubernetesManifest", jsii.invoke(self, "addCdk8sChart", [id, chart, options]))

    @jsii.member(jsii_name="addHelmChart")
    def add_helm_chart(
        self,
        id: builtins.str,
        *,
        atomic: typing.Optional[builtins.bool] = None,
        chart: typing.Optional[builtins.str] = None,
        chart_asset: typing.Optional[_Asset_ac2a7e61] = None,
        create_namespace: typing.Optional[builtins.bool] = None,
        namespace: typing.Optional[builtins.str] = None,
        release: typing.Optional[builtins.str] = None,
        repository: typing.Optional[builtins.str] = None,
        skip_crds: typing.Optional[builtins.bool] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        values: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        version: typing.Optional[builtins.str] = None,
        wait: typing.Optional[builtins.bool] = None,
    ) -> HelmChart:
        '''Defines a Helm chart in this cluster.

        :param id: logical id of this chart.
        :param atomic: Whether or not Helm should treat this operation as atomic; if set, upgrade process rolls back changes made in case of failed upgrade. The --wait flag will be set automatically if --atomic is used. Default: false
        :param chart: The name of the chart. Either this or ``chartAsset`` must be specified. Default: - No chart name. Implies ``chartAsset`` is used.
        :param chart_asset: The chart in the form of an asset. Either this or ``chart`` must be specified. Default: - No chart asset. Implies ``chart`` is used.
        :param create_namespace: create namespace if not exist. Default: true
        :param namespace: The Kubernetes namespace scope of the requests. Default: default
        :param release: The name of the release. Default: - If no release name is given, it will use the last 53 characters of the node's unique id.
        :param repository: The repository which contains the chart. For example: https://charts.helm.sh/stable/ Default: - No repository will be used, which means that the chart needs to be an absolute URL.
        :param skip_crds: if set, no CRDs will be installed. Default: - CRDs are installed if not already present
        :param timeout: Amount of time to wait for any individual Kubernetes operation. Maximum 15 minutes. Default: Duration.minutes(5)
        :param values: The values to be used by the chart. For nested values use a nested dictionary. For example: values: { installationCRDs: true, webhook: { port: 9443 } } Default: - No values are provided to the chart.
        :param version: The chart version to install. Default: - If this is not specified, the latest version is installed
        :param wait: Whether or not Helm should wait until all Pods, PVCs, Services, and minimum number of Pods of a Deployment, StatefulSet, or ReplicaSet are in a ready state before marking the release as successful. Default: - Helm will not wait before marking release as successful

        :return: a ``HelmChart`` construct
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__08729ac4099dd779679b753fc203b6492daf687a1f8a07194d707cc6e3650d37)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = HelmChartOptions(
            atomic=atomic,
            chart=chart,
            chart_asset=chart_asset,
            create_namespace=create_namespace,
            namespace=namespace,
            release=release,
            repository=repository,
            skip_crds=skip_crds,
            timeout=timeout,
            values=values,
            version=version,
            wait=wait,
        )

        return typing.cast(HelmChart, jsii.invoke(self, "addHelmChart", [id, options]))

    @jsii.member(jsii_name="addManifest")
    def add_manifest(
        self,
        id: builtins.str,
        *manifest: typing.Mapping[builtins.str, typing.Any],
    ) -> "KubernetesManifest":
        '''Defines a Kubernetes resource in this cluster.

        The manifest will be applied/deleted using kubectl as needed.

        :param id: logical id of this manifest.
        :param manifest: a list of Kubernetes resource specifications.

        :return: a ``KubernetesManifest`` object.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__72628a5f5b08c75e6d63d4b55d2b010edad747e9e6fdcc7dd098d93d854b5e32)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument manifest", value=manifest, expected_type=typing.Tuple[type_hints["manifest"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("KubernetesManifest", jsii.invoke(self, "addManifest", [id, *manifest]))

    @jsii.member(jsii_name="addServiceAccount")
    def add_service_account(
        self,
        id: builtins.str,
        *,
        annotations: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        identity_type: typing.Optional["IdentityType"] = None,
        labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        name: typing.Optional[builtins.str] = None,
        namespace: typing.Optional[builtins.str] = None,
    ) -> "ServiceAccount":
        '''Creates a new service account with corresponding IAM Role (IRSA).

        :param id: logical id of service account.
        :param annotations: Additional annotations of the service account. Default: - no additional annotations
        :param identity_type: The identity type to use for the service account. Default: IdentityType.IRSA
        :param labels: Additional labels of the service account. Default: - no additional labels
        :param name: The name of the service account. The name of a ServiceAccount object must be a valid DNS subdomain name. https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/ Default: - If no name is given, it will use the id of the resource.
        :param namespace: The namespace of the service account. All namespace names must be valid RFC 1123 DNS labels. https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/#namespaces-and-dns Default: "default"
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e1ebfaeb10359620b55323126554d3e31b14090625de1618808646a519d578de)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = ServiceAccountOptions(
            annotations=annotations,
            identity_type=identity_type,
            labels=labels,
            name=name,
            namespace=namespace,
        )

        return typing.cast("ServiceAccount", jsii.invoke(self, "addServiceAccount", [id, options]))

    @jsii.member(jsii_name="connectAutoScalingGroupCapacity")
    def connect_auto_scaling_group_capacity(
        self,
        auto_scaling_group: _AutoScalingGroup_c547a7b9,
        *,
        bootstrap_enabled: typing.Optional[builtins.bool] = None,
        bootstrap_options: typing.Optional[typing.Union[BootstrapOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        map_role: typing.Optional[builtins.bool] = None,
        spot_interrupt_handler: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Connect capacity in the form of an existing AutoScalingGroup to the EKS cluster.

        The AutoScalingGroup must be running an EKS-optimized AMI containing the
        /etc/eks/bootstrap.sh script. This method will configure Security Groups,
        add the right policies to the instance role, apply the right tags, and add
        the required user data to the instance's launch configuration.

        Spot instances will be labeled ``lifecycle=Ec2Spot`` and tainted with ``PreferNoSchedule``.
        If kubectl is enabled, the
        `spot interrupt handler <https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler>`_
        daemon will be installed on all spot instances to handle
        `EC2 Spot Instance Termination Notices <https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/>`_.

        Prefer to use ``addAutoScalingGroupCapacity`` if possible.

        :param auto_scaling_group: [disable-awslint:ref-via-interface].
        :param bootstrap_enabled: Configures the EC2 user-data script for instances in this autoscaling group to bootstrap the node (invoke ``/etc/eks/bootstrap.sh``) and associate it with the EKS cluster. If you wish to provide a custom user data script, set this to ``false`` and manually invoke ``autoscalingGroup.addUserData()``. Default: true
        :param bootstrap_options: Allows options for node bootstrapping through EC2 user data. Default: - default options
        :param machine_image_type: Allow options to specify different machine image type. Default: MachineImageType.AMAZON_LINUX_2
        :param map_role: Will automatically update the aws-auth ConfigMap to map the IAM instance role to RBAC. This cannot be explicitly set to ``true`` if the cluster has kubectl disabled. Default: - true if the cluster has kubectl enabled (which is the default).
        :param spot_interrupt_handler: Installs the AWS spot instance interrupt handler on the cluster if it's not already added. Only relevant if ``spotPrice`` is configured on the auto-scaling group. Default: true

        :see: https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0276008d48460dbfc38953b2433e1ae377a5b249ca8b272dbea133cc69a44e84)
            check_type(argname="argument auto_scaling_group", value=auto_scaling_group, expected_type=type_hints["auto_scaling_group"])
        options = AutoScalingGroupOptions(
            bootstrap_enabled=bootstrap_enabled,
            bootstrap_options=bootstrap_options,
            machine_image_type=machine_image_type,
            map_role=map_role,
            spot_interrupt_handler=spot_interrupt_handler,
        )

        return typing.cast(None, jsii.invoke(self, "connectAutoScalingGroupCapacity", [auto_scaling_group, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ICluster).__jsii_proxy_class__ = lambda : _IClusterProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_eks.IKubectlProvider")
class IKubectlProvider(_constructs_77d1e7e8.IConstruct, typing_extensions.Protocol):
    '''Imported KubectlProvider that can be used in place of the default one created by CDK.'''

    @builtins.property
    @jsii.member(jsii_name="handlerRole")
    def handler_role(self) -> _IRole_235f5d8e:
        '''The IAM execution role of the handler.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        '''The IAM role to assume in order to perform kubectl operations against this cluster.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="serviceToken")
    def service_token(self) -> builtins.str:
        '''The custom resource provider's service token.'''
        ...


class _IKubectlProviderProxy(
    jsii.proxy_for(_constructs_77d1e7e8.IConstruct), # type: ignore[misc]
):
    '''Imported KubectlProvider that can be used in place of the default one created by CDK.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_eks.IKubectlProvider"

    @builtins.property
    @jsii.member(jsii_name="handlerRole")
    def handler_role(self) -> _IRole_235f5d8e:
        '''The IAM execution role of the handler.'''
        return typing.cast(_IRole_235f5d8e, jsii.get(self, "handlerRole"))

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        '''The IAM role to assume in order to perform kubectl operations against this cluster.'''
        return typing.cast(builtins.str, jsii.get(self, "roleArn"))

    @builtins.property
    @jsii.member(jsii_name="serviceToken")
    def service_token(self) -> builtins.str:
        '''The custom resource provider's service token.'''
        return typing.cast(builtins.str, jsii.get(self, "serviceToken"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IKubectlProvider).__jsii_proxy_class__ = lambda : _IKubectlProviderProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_eks.INodegroup")
class INodegroup(_IResource_c80c4260, typing_extensions.Protocol):
    '''NodeGroup interface.'''

    @builtins.property
    @jsii.member(jsii_name="nodegroupName")
    def nodegroup_name(self) -> builtins.str:
        '''Name of the nodegroup.

        :attribute: true
        '''
        ...


class _INodegroupProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''NodeGroup interface.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_eks.INodegroup"

    @builtins.property
    @jsii.member(jsii_name="nodegroupName")
    def nodegroup_name(self) -> builtins.str:
        '''Name of the nodegroup.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "nodegroupName"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, INodegroup).__jsii_proxy_class__ = lambda : _INodegroupProxy


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.IdentityType")
class IdentityType(enum.Enum):
    '''Enum representing the different identity types that can be used for a Kubernetes service account.

    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        
        eks.ServiceAccount(self, "ServiceAccount",
            cluster=cluster,
            name="test-sa",
            namespace="default",
            identity_type=eks.IdentityType.POD_IDENTITY
        )
    '''

    IRSA = "IRSA"
    '''Use the IAM Roles for Service Accounts (IRSA) identity type.

    IRSA allows you to associate an IAM role with a Kubernetes service account.
    This provides a way to grant permissions to Kubernetes pods by associating an IAM role with a Kubernetes service account.
    The IAM role can then be used to provide AWS credentials to the pods, allowing them to access other AWS resources.

    When enabled, the openIdConnectProvider of the cluster would be created when you create the ServiceAccount.

    :see: https://docs.aws.amazon.com/eks/latest/userguide/iam-roles-for-service-accounts.html
    '''
    POD_IDENTITY = "POD_IDENTITY"
    '''Use the EKS Pod Identities identity type.

    EKS Pod Identities provide the ability to manage credentials for your applications, similar to the way that Amazon EC2 instance profiles
    provide credentials to Amazon EC2 instances. Instead of creating and distributing your AWS credentials to the containers or using the
    Amazon EC2 instance's role, you associate an IAM role with a Kubernetes service account and configure your Pods to use the service account.

    When enabled, the Pod Identity Agent AddOn of the cluster would be created when you create the ServiceAccount.

    :see: https://docs.aws.amazon.com/eks/latest/userguide/pod-identities.html
    '''


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.IpFamily")
class IpFamily(enum.Enum):
    '''EKS cluster IP family.

    :exampleMetadata: infused

    Example::

        from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
        # vpc: ec2.Vpc
        
        
        def associate_subnet_with_v6_cidr(self, vpc, count, subnet):
            cfn_subnet = subnet.node.default_child
            cfn_subnet.ipv6_cidr_block = Fn.select(count, Fn.cidr(Fn.select(0, vpc.vpc_ipv6_cidr_blocks), 256, (128 - 64).to_string()))
            cfn_subnet.assign_ipv6_address_on_creation = True
        
        # make an ipv6 cidr
        ipv6cidr = ec2.CfnVPCCidrBlock(self, "CIDR6",
            vpc_id=vpc.vpc_id,
            amazon_provided_ipv6_cidr_block=True
        )
        
        # connect the ipv6 cidr to all vpc subnets
        subnetcount = 0
        subnets = vpc.public_subnets.concat(vpc.private_subnets)
        for subnet in subnets:
            # Wait for the ipv6 cidr to complete
            subnet.node.add_dependency(ipv6cidr)
            associate_subnet_with_v6_cidr(vpc, subnetcount, subnet)
            subnetcount = subnetcount + 1
        
        cluster = eks.Cluster(self, "hello-eks",
            version=eks.KubernetesVersion.V1_32,
            vpc=vpc,
            ip_family=eks.IpFamily.IP_V6,
            vpc_subnets=[ec2.SubnetSelection(subnets=vpc.public_subnets)],
            kubectl_layer=KubectlV32Layer(self, "kubectl")
        )
    '''

    IP_V4 = "IP_V4"
    '''Use IPv4 for pods and services in your cluster.'''
    IP_V6 = "IP_V6"
    '''Use IPv6 for pods and services in your cluster.'''


@jsii.implements(IKubectlProvider)
class KubectlProvider(
    _NestedStack_dd393a45,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.KubectlProvider",
):
    '''Implementation of Kubectl Lambda.

    :exampleMetadata: infused

    Example::

        handler_role = iam.Role.from_role_arn(self, "HandlerRole", "arn:aws:iam::123456789012:role/lambda-role")
        # get the serviceToken from the custom resource provider
        function_arn = lambda_.Function.from_function_name(self, "ProviderOnEventFunc", "ProviderframeworkonEvent-XXX").function_arn
        kubectl_provider = eks.KubectlProvider.from_kubectl_provider_attributes(self, "KubectlProvider",
            function_arn=function_arn,
            kubectl_role_arn="arn:aws:iam::123456789012:role/kubectl-role",
            handler_role=handler_role
        )
        
        cluster = eks.Cluster.from_cluster_attributes(self, "Cluster",
            cluster_name="cluster",
            kubectl_provider=kubectl_provider
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: ICluster,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster: The cluster to control.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__24dada33beb6c104bc8b7b823dcac3b05870723d95a8d91a6a6cf2c6809248e3)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = KubectlProviderProps(cluster=cluster)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="fromKubectlProviderAttributes")
    @builtins.classmethod
    def from_kubectl_provider_attributes(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        function_arn: builtins.str,
        handler_role: _IRole_235f5d8e,
        kubectl_role_arn: builtins.str,
    ) -> IKubectlProvider:
        '''Import an existing provider.

        :param scope: Construct.
        :param id: an id of resource.
        :param function_arn: The custom resource provider's service token.
        :param handler_role: The IAM execution role of the handler. This role must be able to assume kubectlRoleArn
        :param kubectl_role_arn: The IAM role to assume in order to perform kubectl operations against this cluster.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3dc34a0a76a27458531e2bd85fdb3a0e12cbd5c3ed0b306da869f7b6fb06f600)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        attrs = KubectlProviderAttributes(
            function_arn=function_arn,
            handler_role=handler_role,
            kubectl_role_arn=kubectl_role_arn,
        )

        return typing.cast(IKubectlProvider, jsii.sinvoke(cls, "fromKubectlProviderAttributes", [scope, id, attrs]))

    @jsii.member(jsii_name="getOrCreate")
    @builtins.classmethod
    def get_or_create(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        cluster: ICluster,
    ) -> IKubectlProvider:
        '''Take existing provider or create new based on cluster.

        :param scope: Construct.
        :param cluster: k8s cluster.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a1be6786d017d5c8ecdac27431e97d30ee84a7aa05b526b912aea67bfd5bb3c7)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
        return typing.cast(IKubectlProvider, jsii.sinvoke(cls, "getOrCreate", [scope, cluster]))

    @builtins.property
    @jsii.member(jsii_name="handlerRole")
    def handler_role(self) -> _IRole_235f5d8e:
        '''The IAM execution role of the handler.'''
        return typing.cast(_IRole_235f5d8e, jsii.get(self, "handlerRole"))

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        '''The IAM role to assume in order to perform kubectl operations against this cluster.'''
        return typing.cast(builtins.str, jsii.get(self, "roleArn"))

    @builtins.property
    @jsii.member(jsii_name="serviceToken")
    def service_token(self) -> builtins.str:
        '''The custom resource provider's service token.'''
        return typing.cast(builtins.str, jsii.get(self, "serviceToken"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.KubectlProviderAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "function_arn": "functionArn",
        "handler_role": "handlerRole",
        "kubectl_role_arn": "kubectlRoleArn",
    },
)
class KubectlProviderAttributes:
    def __init__(
        self,
        *,
        function_arn: builtins.str,
        handler_role: _IRole_235f5d8e,
        kubectl_role_arn: builtins.str,
    ) -> None:
        '''Kubectl Provider Attributes.

        :param function_arn: The custom resource provider's service token.
        :param handler_role: The IAM execution role of the handler. This role must be able to assume kubectlRoleArn
        :param kubectl_role_arn: The IAM role to assume in order to perform kubectl operations against this cluster.

        :exampleMetadata: infused

        Example::

            handler_role = iam.Role.from_role_arn(self, "HandlerRole", "arn:aws:iam::123456789012:role/lambda-role")
            # get the serviceToken from the custom resource provider
            function_arn = lambda_.Function.from_function_name(self, "ProviderOnEventFunc", "ProviderframeworkonEvent-XXX").function_arn
            kubectl_provider = eks.KubectlProvider.from_kubectl_provider_attributes(self, "KubectlProvider",
                function_arn=function_arn,
                kubectl_role_arn="arn:aws:iam::123456789012:role/kubectl-role",
                handler_role=handler_role
            )
            
            cluster = eks.Cluster.from_cluster_attributes(self, "Cluster",
                cluster_name="cluster",
                kubectl_provider=kubectl_provider
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__847baf0df9252f2b3dffe68261a08333d86298da8bcca05e12d417d4d2d4942a)
            check_type(argname="argument function_arn", value=function_arn, expected_type=type_hints["function_arn"])
            check_type(argname="argument handler_role", value=handler_role, expected_type=type_hints["handler_role"])
            check_type(argname="argument kubectl_role_arn", value=kubectl_role_arn, expected_type=type_hints["kubectl_role_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_arn": function_arn,
            "handler_role": handler_role,
            "kubectl_role_arn": kubectl_role_arn,
        }

    @builtins.property
    def function_arn(self) -> builtins.str:
        '''The custom resource provider's service token.'''
        result = self._values.get("function_arn")
        assert result is not None, "Required property 'function_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def handler_role(self) -> _IRole_235f5d8e:
        '''The IAM execution role of the handler.

        This role must be able to assume kubectlRoleArn
        '''
        result = self._values.get("handler_role")
        assert result is not None, "Required property 'handler_role' is missing"
        return typing.cast(_IRole_235f5d8e, result)

    @builtins.property
    def kubectl_role_arn(self) -> builtins.str:
        '''The IAM role to assume in order to perform kubectl operations against this cluster.'''
        result = self._values.get("kubectl_role_arn")
        assert result is not None, "Required property 'kubectl_role_arn' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "KubectlProviderAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.KubectlProviderProps",
    jsii_struct_bases=[],
    name_mapping={"cluster": "cluster"},
)
class KubectlProviderProps:
    def __init__(self, *, cluster: ICluster) -> None:
        '''Properties for a KubectlProvider.

        :param cluster: The cluster to control.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            # cluster: eks.Cluster
            
            kubectl_provider_props = eks.KubectlProviderProps(
                cluster=cluster
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eee4a407d1851bcb733f863ecb7adf5b7b6146cfb2b71cd124c91bf6001272ab)
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
        }

    @builtins.property
    def cluster(self) -> ICluster:
        '''The cluster to control.'''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(ICluster, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "KubectlProviderProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class KubernetesManifest(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.KubernetesManifest",
):
    '''Represents a manifest within the Kubernetes system.

    Alternatively, you can use ``cluster.addManifest(resource[, resource, ...])``
    to define resources on this cluster.

    Applies/deletes the manifest using ``kubectl``.

    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        namespace = cluster.add_manifest("my-namespace", {
            "api_version": "v1",
            "kind": "Namespace",
            "metadata": {"name": "my-app"}
        })
        
        service = cluster.add_manifest("my-service", {
            "metadata": {
                "name": "myservice",
                "namespace": "my-app"
            },
            "spec": {}
        })
        
        service.node.add_dependency(namespace)
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: ICluster,
        manifest: typing.Sequence[typing.Mapping[builtins.str, typing.Any]],
        overwrite: typing.Optional[builtins.bool] = None,
        ingress_alb: typing.Optional[builtins.bool] = None,
        ingress_alb_scheme: typing.Optional[AlbScheme] = None,
        prune: typing.Optional[builtins.bool] = None,
        skip_validation: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster: The EKS cluster to apply this manifest to. [disable-awslint:ref-via-interface]
        :param manifest: The manifest to apply. Consists of any number of child resources. When the resources are created/updated, this manifest will be applied to the cluster through ``kubectl apply`` and when the resources or the stack is deleted, the resources in the manifest will be deleted through ``kubectl delete``.
        :param overwrite: Overwrite any existing resources. If this is set, we will use ``kubectl apply`` instead of ``kubectl create`` when the resource is created. Otherwise, if there is already a resource in the cluster with the same name, the operation will fail. Default: false
        :param ingress_alb: Automatically detect ``Ingress`` resources in the manifest and annotate them so they are picked up by an ALB Ingress Controller. Default: false
        :param ingress_alb_scheme: Specify the ALB scheme that should be applied to ``Ingress`` resources. Only applicable if ``ingressAlb`` is set to ``true``. Default: AlbScheme.INTERNAL
        :param prune: When a resource is removed from a Kubernetes manifest, it no longer appears in the manifest, and there is no way to know that this resource needs to be deleted. To address this, ``kubectl apply`` has a ``--prune`` option which will query the cluster for all resources with a specific label and will remove all the labeld resources that are not part of the applied manifest. If this option is disabled and a resource is removed, it will become "orphaned" and will not be deleted from the cluster. When this option is enabled (default), the construct will inject a label to all Kubernetes resources included in this manifest which will be used to prune resources when the manifest changes via ``kubectl apply --prune``. The label name will be ``aws.cdk.eks/prune-<ADDR>`` where ``<ADDR>`` is the 42-char unique address of this construct in the construct tree. Value is empty. Default: - based on the prune option of the cluster, which is ``true`` unless otherwise specified.
        :param skip_validation: A flag to signify if the manifest validation should be skipped. Default: false
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5ab7ee50af05789a0a98caa55010e1acd1c6a48de4b074e4c0aa31ef40d7ef16)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = KubernetesManifestProps(
            cluster=cluster,
            manifest=manifest,
            overwrite=overwrite,
            ingress_alb=ingress_alb,
            ingress_alb_scheme=ingress_alb_scheme,
            prune=prune,
            skip_validation=skip_validation,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.python.classproperty
    @jsii.member(jsii_name="RESOURCE_TYPE")
    def RESOURCE_TYPE(cls) -> builtins.str:
        '''The CloudFormation resource type.'''
        return typing.cast(builtins.str, jsii.sget(cls, "RESOURCE_TYPE"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.KubernetesManifestOptions",
    jsii_struct_bases=[],
    name_mapping={
        "ingress_alb": "ingressAlb",
        "ingress_alb_scheme": "ingressAlbScheme",
        "prune": "prune",
        "skip_validation": "skipValidation",
    },
)
class KubernetesManifestOptions:
    def __init__(
        self,
        *,
        ingress_alb: typing.Optional[builtins.bool] = None,
        ingress_alb_scheme: typing.Optional[AlbScheme] = None,
        prune: typing.Optional[builtins.bool] = None,
        skip_validation: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for ``KubernetesManifest``.

        :param ingress_alb: Automatically detect ``Ingress`` resources in the manifest and annotate them so they are picked up by an ALB Ingress Controller. Default: false
        :param ingress_alb_scheme: Specify the ALB scheme that should be applied to ``Ingress`` resources. Only applicable if ``ingressAlb`` is set to ``true``. Default: AlbScheme.INTERNAL
        :param prune: When a resource is removed from a Kubernetes manifest, it no longer appears in the manifest, and there is no way to know that this resource needs to be deleted. To address this, ``kubectl apply`` has a ``--prune`` option which will query the cluster for all resources with a specific label and will remove all the labeld resources that are not part of the applied manifest. If this option is disabled and a resource is removed, it will become "orphaned" and will not be deleted from the cluster. When this option is enabled (default), the construct will inject a label to all Kubernetes resources included in this manifest which will be used to prune resources when the manifest changes via ``kubectl apply --prune``. The label name will be ``aws.cdk.eks/prune-<ADDR>`` where ``<ADDR>`` is the 42-char unique address of this construct in the construct tree. Value is empty. Default: - based on the prune option of the cluster, which is ``true`` unless otherwise specified.
        :param skip_validation: A flag to signify if the manifest validation should be skipped. Default: false

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            kubernetes_manifest_options = eks.KubernetesManifestOptions(
                ingress_alb=False,
                ingress_alb_scheme=eks.AlbScheme.INTERNAL,
                prune=False,
                skip_validation=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0108fc2a4c68c3bf78313f8a2848721ac1dffbae3c8a2f17b1cab3111a244680)
            check_type(argname="argument ingress_alb", value=ingress_alb, expected_type=type_hints["ingress_alb"])
            check_type(argname="argument ingress_alb_scheme", value=ingress_alb_scheme, expected_type=type_hints["ingress_alb_scheme"])
            check_type(argname="argument prune", value=prune, expected_type=type_hints["prune"])
            check_type(argname="argument skip_validation", value=skip_validation, expected_type=type_hints["skip_validation"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if ingress_alb is not None:
            self._values["ingress_alb"] = ingress_alb
        if ingress_alb_scheme is not None:
            self._values["ingress_alb_scheme"] = ingress_alb_scheme
        if prune is not None:
            self._values["prune"] = prune
        if skip_validation is not None:
            self._values["skip_validation"] = skip_validation

    @builtins.property
    def ingress_alb(self) -> typing.Optional[builtins.bool]:
        '''Automatically detect ``Ingress`` resources in the manifest and annotate them so they are picked up by an ALB Ingress Controller.

        :default: false
        '''
        result = self._values.get("ingress_alb")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def ingress_alb_scheme(self) -> typing.Optional[AlbScheme]:
        '''Specify the ALB scheme that should be applied to ``Ingress`` resources.

        Only applicable if ``ingressAlb`` is set to ``true``.

        :default: AlbScheme.INTERNAL
        '''
        result = self._values.get("ingress_alb_scheme")
        return typing.cast(typing.Optional[AlbScheme], result)

    @builtins.property
    def prune(self) -> typing.Optional[builtins.bool]:
        '''When a resource is removed from a Kubernetes manifest, it no longer appears in the manifest, and there is no way to know that this resource needs to be deleted.

        To address this, ``kubectl apply`` has a ``--prune`` option which will
        query the cluster for all resources with a specific label and will remove
        all the labeld resources that are not part of the applied manifest. If this
        option is disabled and a resource is removed, it will become "orphaned" and
        will not be deleted from the cluster.

        When this option is enabled (default), the construct will inject a label to
        all Kubernetes resources included in this manifest which will be used to
        prune resources when the manifest changes via ``kubectl apply --prune``.

        The label name will be ``aws.cdk.eks/prune-<ADDR>`` where ``<ADDR>`` is the
        42-char unique address of this construct in the construct tree. Value is
        empty.

        :default:

        - based on the prune option of the cluster, which is ``true`` unless
        otherwise specified.

        :see: https://kubernetes.io/docs/tasks/manage-kubernetes-objects/declarative-config/#alternative-kubectl-apply-f-directory-prune-l-your-label
        '''
        result = self._values.get("prune")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def skip_validation(self) -> typing.Optional[builtins.bool]:
        '''A flag to signify if the manifest validation should be skipped.

        :default: false
        '''
        result = self._values.get("skip_validation")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "KubernetesManifestOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.KubernetesManifestProps",
    jsii_struct_bases=[KubernetesManifestOptions],
    name_mapping={
        "ingress_alb": "ingressAlb",
        "ingress_alb_scheme": "ingressAlbScheme",
        "prune": "prune",
        "skip_validation": "skipValidation",
        "cluster": "cluster",
        "manifest": "manifest",
        "overwrite": "overwrite",
    },
)
class KubernetesManifestProps(KubernetesManifestOptions):
    def __init__(
        self,
        *,
        ingress_alb: typing.Optional[builtins.bool] = None,
        ingress_alb_scheme: typing.Optional[AlbScheme] = None,
        prune: typing.Optional[builtins.bool] = None,
        skip_validation: typing.Optional[builtins.bool] = None,
        cluster: ICluster,
        manifest: typing.Sequence[typing.Mapping[builtins.str, typing.Any]],
        overwrite: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Properties for KubernetesManifest.

        :param ingress_alb: Automatically detect ``Ingress`` resources in the manifest and annotate them so they are picked up by an ALB Ingress Controller. Default: false
        :param ingress_alb_scheme: Specify the ALB scheme that should be applied to ``Ingress`` resources. Only applicable if ``ingressAlb`` is set to ``true``. Default: AlbScheme.INTERNAL
        :param prune: When a resource is removed from a Kubernetes manifest, it no longer appears in the manifest, and there is no way to know that this resource needs to be deleted. To address this, ``kubectl apply`` has a ``--prune`` option which will query the cluster for all resources with a specific label and will remove all the labeld resources that are not part of the applied manifest. If this option is disabled and a resource is removed, it will become "orphaned" and will not be deleted from the cluster. When this option is enabled (default), the construct will inject a label to all Kubernetes resources included in this manifest which will be used to prune resources when the manifest changes via ``kubectl apply --prune``. The label name will be ``aws.cdk.eks/prune-<ADDR>`` where ``<ADDR>`` is the 42-char unique address of this construct in the construct tree. Value is empty. Default: - based on the prune option of the cluster, which is ``true`` unless otherwise specified.
        :param skip_validation: A flag to signify if the manifest validation should be skipped. Default: false
        :param cluster: The EKS cluster to apply this manifest to. [disable-awslint:ref-via-interface]
        :param manifest: The manifest to apply. Consists of any number of child resources. When the resources are created/updated, this manifest will be applied to the cluster through ``kubectl apply`` and when the resources or the stack is deleted, the resources in the manifest will be deleted through ``kubectl delete``.
        :param overwrite: Overwrite any existing resources. If this is set, we will use ``kubectl apply`` instead of ``kubectl create`` when the resource is created. Otherwise, if there is already a resource in the cluster with the same name, the operation will fail. Default: false

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            app_label = {"app": "hello-kubernetes"}
            
            deployment = {
                "api_version": "apps/v1",
                "kind": "Deployment",
                "metadata": {"name": "hello-kubernetes"},
                "spec": {
                    "replicas": 3,
                    "selector": {"match_labels": app_label},
                    "template": {
                        "metadata": {"labels": app_label},
                        "spec": {
                            "containers": [{
                                "name": "hello-kubernetes",
                                "image": "paulbouwer/hello-kubernetes:1.5",
                                "ports": [{"container_port": 8080}]
                            }
                            ]
                        }
                    }
                }
            }
            
            service = {
                "api_version": "v1",
                "kind": "Service",
                "metadata": {"name": "hello-kubernetes"},
                "spec": {
                    "type": "LoadBalancer",
                    "ports": [{"port": 80, "target_port": 8080}],
                    "selector": app_label
                }
            }
            
            # option 1: use a construct
            eks.KubernetesManifest(self, "hello-kub",
                cluster=cluster,
                manifest=[deployment, service]
            )
            
            # or, option2: use `addManifest`
            cluster.add_manifest("hello-kub", service, deployment)
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a5bb00898157a38a0e39d6984245e816d0c01e9016a0d4412db0764b823c6104)
            check_type(argname="argument ingress_alb", value=ingress_alb, expected_type=type_hints["ingress_alb"])
            check_type(argname="argument ingress_alb_scheme", value=ingress_alb_scheme, expected_type=type_hints["ingress_alb_scheme"])
            check_type(argname="argument prune", value=prune, expected_type=type_hints["prune"])
            check_type(argname="argument skip_validation", value=skip_validation, expected_type=type_hints["skip_validation"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument manifest", value=manifest, expected_type=type_hints["manifest"])
            check_type(argname="argument overwrite", value=overwrite, expected_type=type_hints["overwrite"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
            "manifest": manifest,
        }
        if ingress_alb is not None:
            self._values["ingress_alb"] = ingress_alb
        if ingress_alb_scheme is not None:
            self._values["ingress_alb_scheme"] = ingress_alb_scheme
        if prune is not None:
            self._values["prune"] = prune
        if skip_validation is not None:
            self._values["skip_validation"] = skip_validation
        if overwrite is not None:
            self._values["overwrite"] = overwrite

    @builtins.property
    def ingress_alb(self) -> typing.Optional[builtins.bool]:
        '''Automatically detect ``Ingress`` resources in the manifest and annotate them so they are picked up by an ALB Ingress Controller.

        :default: false
        '''
        result = self._values.get("ingress_alb")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def ingress_alb_scheme(self) -> typing.Optional[AlbScheme]:
        '''Specify the ALB scheme that should be applied to ``Ingress`` resources.

        Only applicable if ``ingressAlb`` is set to ``true``.

        :default: AlbScheme.INTERNAL
        '''
        result = self._values.get("ingress_alb_scheme")
        return typing.cast(typing.Optional[AlbScheme], result)

    @builtins.property
    def prune(self) -> typing.Optional[builtins.bool]:
        '''When a resource is removed from a Kubernetes manifest, it no longer appears in the manifest, and there is no way to know that this resource needs to be deleted.

        To address this, ``kubectl apply`` has a ``--prune`` option which will
        query the cluster for all resources with a specific label and will remove
        all the labeld resources that are not part of the applied manifest. If this
        option is disabled and a resource is removed, it will become "orphaned" and
        will not be deleted from the cluster.

        When this option is enabled (default), the construct will inject a label to
        all Kubernetes resources included in this manifest which will be used to
        prune resources when the manifest changes via ``kubectl apply --prune``.

        The label name will be ``aws.cdk.eks/prune-<ADDR>`` where ``<ADDR>`` is the
        42-char unique address of this construct in the construct tree. Value is
        empty.

        :default:

        - based on the prune option of the cluster, which is ``true`` unless
        otherwise specified.

        :see: https://kubernetes.io/docs/tasks/manage-kubernetes-objects/declarative-config/#alternative-kubectl-apply-f-directory-prune-l-your-label
        '''
        result = self._values.get("prune")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def skip_validation(self) -> typing.Optional[builtins.bool]:
        '''A flag to signify if the manifest validation should be skipped.

        :default: false
        '''
        result = self._values.get("skip_validation")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cluster(self) -> ICluster:
        '''The EKS cluster to apply this manifest to.

        [disable-awslint:ref-via-interface]
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(ICluster, result)

    @builtins.property
    def manifest(self) -> typing.List[typing.Mapping[builtins.str, typing.Any]]:
        '''The manifest to apply.

        Consists of any number of child resources.

        When the resources are created/updated, this manifest will be applied to the
        cluster through ``kubectl apply`` and when the resources or the stack is
        deleted, the resources in the manifest will be deleted through ``kubectl delete``.

        Example::

            [{
                "api_version": "v1",
                "kind": "Pod",
                "metadata": {"name": "mypod"},
                "spec": {
                    "containers": [{"name": "hello", "image": "paulbouwer/hello-kubernetes:1.5", "ports": [{"container_port": 8080}]}]
                }
            }]
        '''
        result = self._values.get("manifest")
        assert result is not None, "Required property 'manifest' is missing"
        return typing.cast(typing.List[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def overwrite(self) -> typing.Optional[builtins.bool]:
        '''Overwrite any existing resources.

        If this is set, we will use ``kubectl apply`` instead of ``kubectl create``
        when the resource is created. Otherwise, if there is already a resource
        in the cluster with the same name, the operation will fail.

        :default: false
        '''
        result = self._values.get("overwrite")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "KubernetesManifestProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class KubernetesObjectValue(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.KubernetesObjectValue",
):
    '''Represents a value of a specific object deployed in the cluster.

    Use this to fetch any information available by the ``kubectl get`` command.

    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        # query the load balancer address
        my_service_address = eks.KubernetesObjectValue(self, "LoadBalancerAttribute",
            cluster=cluster,
            object_type="service",
            object_name="my-service",
            json_path=".status.loadBalancer.ingress[0].hostname"
        )
        
        # pass the address to a lambda function
        proxy_function = lambda_.Function(self, "ProxyFunction",
            handler="index.handler",
            code=lambda_.Code.from_inline("my-code"),
            runtime=lambda_.Runtime.NODEJS_LATEST,
            environment={
                "my_service_address": my_service_address.value
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: ICluster,
        json_path: builtins.str,
        object_name: builtins.str,
        object_type: builtins.str,
        object_namespace: typing.Optional[builtins.str] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster: The EKS cluster to fetch attributes from. [disable-awslint:ref-via-interface]
        :param json_path: JSONPath to the specific value.
        :param object_name: The name of the object to query.
        :param object_type: The object type to query. (e.g 'service', 'pod'...)
        :param object_namespace: The namespace the object belongs to. Default: 'default'
        :param timeout: Timeout for waiting on a value. Default: Duration.minutes(5)
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b43125517e8558a2f4aa3e5574e0ff3e4db0d48b2397cf0b105b5f1efda669e8)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = KubernetesObjectValueProps(
            cluster=cluster,
            json_path=json_path,
            object_name=object_name,
            object_type=object_type,
            object_namespace=object_namespace,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.python.classproperty
    @jsii.member(jsii_name="RESOURCE_TYPE")
    def RESOURCE_TYPE(cls) -> builtins.str:
        '''The CloudFormation resource type.'''
        return typing.cast(builtins.str, jsii.sget(cls, "RESOURCE_TYPE"))

    @builtins.property
    @jsii.member(jsii_name="value")
    def value(self) -> builtins.str:
        '''The value as a string token.'''
        return typing.cast(builtins.str, jsii.get(self, "value"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.KubernetesObjectValueProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "json_path": "jsonPath",
        "object_name": "objectName",
        "object_type": "objectType",
        "object_namespace": "objectNamespace",
        "timeout": "timeout",
    },
)
class KubernetesObjectValueProps:
    def __init__(
        self,
        *,
        cluster: ICluster,
        json_path: builtins.str,
        object_name: builtins.str,
        object_type: builtins.str,
        object_namespace: typing.Optional[builtins.str] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''Properties for KubernetesObjectValue.

        :param cluster: The EKS cluster to fetch attributes from. [disable-awslint:ref-via-interface]
        :param json_path: JSONPath to the specific value.
        :param object_name: The name of the object to query.
        :param object_type: The object type to query. (e.g 'service', 'pod'...)
        :param object_namespace: The namespace the object belongs to. Default: 'default'
        :param timeout: Timeout for waiting on a value. Default: Duration.minutes(5)

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            # query the load balancer address
            my_service_address = eks.KubernetesObjectValue(self, "LoadBalancerAttribute",
                cluster=cluster,
                object_type="service",
                object_name="my-service",
                json_path=".status.loadBalancer.ingress[0].hostname"
            )
            
            # pass the address to a lambda function
            proxy_function = lambda_.Function(self, "ProxyFunction",
                handler="index.handler",
                code=lambda_.Code.from_inline("my-code"),
                runtime=lambda_.Runtime.NODEJS_LATEST,
                environment={
                    "my_service_address": my_service_address.value
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1c51b5fedde07e77790a96098d170fb8ee0ab33d54b6b958e324191a05908740)
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument json_path", value=json_path, expected_type=type_hints["json_path"])
            check_type(argname="argument object_name", value=object_name, expected_type=type_hints["object_name"])
            check_type(argname="argument object_type", value=object_type, expected_type=type_hints["object_type"])
            check_type(argname="argument object_namespace", value=object_namespace, expected_type=type_hints["object_namespace"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
            "json_path": json_path,
            "object_name": object_name,
            "object_type": object_type,
        }
        if object_namespace is not None:
            self._values["object_namespace"] = object_namespace
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def cluster(self) -> ICluster:
        '''The EKS cluster to fetch attributes from.

        [disable-awslint:ref-via-interface]
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(ICluster, result)

    @builtins.property
    def json_path(self) -> builtins.str:
        '''JSONPath to the specific value.

        :see: https://kubernetes.io/docs/reference/kubectl/jsonpath/
        '''
        result = self._values.get("json_path")
        assert result is not None, "Required property 'json_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def object_name(self) -> builtins.str:
        '''The name of the object to query.'''
        result = self._values.get("object_name")
        assert result is not None, "Required property 'object_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def object_type(self) -> builtins.str:
        '''The object type to query.

        (e.g 'service', 'pod'...)
        '''
        result = self._values.get("object_type")
        assert result is not None, "Required property 'object_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def object_namespace(self) -> typing.Optional[builtins.str]:
        '''The namespace the object belongs to.

        :default: 'default'
        '''
        result = self._values.get("object_namespace")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Timeout for waiting on a value.

        :default: Duration.minutes(5)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "KubernetesObjectValueProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class KubernetesPatch(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.KubernetesPatch",
):
    '''A CloudFormation resource which applies/restores a JSON patch into a Kubernetes resource.

    :see: https://kubernetes.io/docs/tasks/run-application/update-api-object-kubectl-patch/
    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        eks.KubernetesPatch(self, "hello-kub-deployment-label",
            cluster=cluster,
            resource_name="deployment/hello-kubernetes",
            apply_patch={"spec": {"replicas": 5}},
            restore_patch={"spec": {"replicas": 3}}
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        apply_patch: typing.Mapping[builtins.str, typing.Any],
        cluster: ICluster,
        resource_name: builtins.str,
        restore_patch: typing.Mapping[builtins.str, typing.Any],
        patch_type: typing.Optional["PatchType"] = None,
        resource_namespace: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param apply_patch: The JSON object to pass to ``kubectl patch`` when the resource is created/updated.
        :param cluster: The cluster to apply the patch to. [disable-awslint:ref-via-interface]
        :param resource_name: The full name of the resource to patch (e.g. ``deployment/coredns``).
        :param restore_patch: The JSON object to pass to ``kubectl patch`` when the resource is removed.
        :param patch_type: The patch type to pass to ``kubectl patch``. The default type used by ``kubectl patch`` is "strategic". Default: PatchType.STRATEGIC
        :param resource_namespace: The kubernetes API namespace. Default: "default"
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e534a9f53f33b6a1a523b5f3d458dd93cf0155fa305556b69e031307b17691a9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = KubernetesPatchProps(
            apply_patch=apply_patch,
            cluster=cluster,
            resource_name=resource_name,
            restore_patch=restore_patch,
            patch_type=patch_type,
            resource_namespace=resource_namespace,
        )

        jsii.create(self.__class__, self, [scope, id, props])


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.KubernetesPatchProps",
    jsii_struct_bases=[],
    name_mapping={
        "apply_patch": "applyPatch",
        "cluster": "cluster",
        "resource_name": "resourceName",
        "restore_patch": "restorePatch",
        "patch_type": "patchType",
        "resource_namespace": "resourceNamespace",
    },
)
class KubernetesPatchProps:
    def __init__(
        self,
        *,
        apply_patch: typing.Mapping[builtins.str, typing.Any],
        cluster: ICluster,
        resource_name: builtins.str,
        restore_patch: typing.Mapping[builtins.str, typing.Any],
        patch_type: typing.Optional["PatchType"] = None,
        resource_namespace: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for KubernetesPatch.

        :param apply_patch: The JSON object to pass to ``kubectl patch`` when the resource is created/updated.
        :param cluster: The cluster to apply the patch to. [disable-awslint:ref-via-interface]
        :param resource_name: The full name of the resource to patch (e.g. ``deployment/coredns``).
        :param restore_patch: The JSON object to pass to ``kubectl patch`` when the resource is removed.
        :param patch_type: The patch type to pass to ``kubectl patch``. The default type used by ``kubectl patch`` is "strategic". Default: PatchType.STRATEGIC
        :param resource_namespace: The kubernetes API namespace. Default: "default"

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            eks.KubernetesPatch(self, "hello-kub-deployment-label",
                cluster=cluster,
                resource_name="deployment/hello-kubernetes",
                apply_patch={"spec": {"replicas": 5}},
                restore_patch={"spec": {"replicas": 3}}
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3902bca2a2f14fd821e806fc36e44b81045b0f22d504ee922f2271fdb8327c9f)
            check_type(argname="argument apply_patch", value=apply_patch, expected_type=type_hints["apply_patch"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
            check_type(argname="argument resource_name", value=resource_name, expected_type=type_hints["resource_name"])
            check_type(argname="argument restore_patch", value=restore_patch, expected_type=type_hints["restore_patch"])
            check_type(argname="argument patch_type", value=patch_type, expected_type=type_hints["patch_type"])
            check_type(argname="argument resource_namespace", value=resource_namespace, expected_type=type_hints["resource_namespace"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "apply_patch": apply_patch,
            "cluster": cluster,
            "resource_name": resource_name,
            "restore_patch": restore_patch,
        }
        if patch_type is not None:
            self._values["patch_type"] = patch_type
        if resource_namespace is not None:
            self._values["resource_namespace"] = resource_namespace

    @builtins.property
    def apply_patch(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''The JSON object to pass to ``kubectl patch`` when the resource is created/updated.'''
        result = self._values.get("apply_patch")
        assert result is not None, "Required property 'apply_patch' is missing"
        return typing.cast(typing.Mapping[builtins.str, typing.Any], result)

    @builtins.property
    def cluster(self) -> ICluster:
        '''The cluster to apply the patch to.

        [disable-awslint:ref-via-interface]
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(ICluster, result)

    @builtins.property
    def resource_name(self) -> builtins.str:
        '''The full name of the resource to patch (e.g. ``deployment/coredns``).'''
        result = self._values.get("resource_name")
        assert result is not None, "Required property 'resource_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def restore_patch(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''The JSON object to pass to ``kubectl patch`` when the resource is removed.'''
        result = self._values.get("restore_patch")
        assert result is not None, "Required property 'restore_patch' is missing"
        return typing.cast(typing.Mapping[builtins.str, typing.Any], result)

    @builtins.property
    def patch_type(self) -> typing.Optional["PatchType"]:
        '''The patch type to pass to ``kubectl patch``.

        The default type used by ``kubectl patch`` is "strategic".

        :default: PatchType.STRATEGIC
        '''
        result = self._values.get("patch_type")
        return typing.cast(typing.Optional["PatchType"], result)

    @builtins.property
    def resource_namespace(self) -> typing.Optional[builtins.str]:
        '''The kubernetes API namespace.

        :default: "default"
        '''
        result = self._values.get("resource_namespace")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "KubernetesPatchProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class KubernetesVersion(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.KubernetesVersion",
):
    '''Kubernetes cluster version.

    :see: https://docs.aws.amazon.com/eks/latest/userguide/kubernetes-versions.html#kubernetes-release-calendar
    :exampleMetadata: infused

    Example::

        from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
        
        # or
        # vpc: ec2.Vpc
        
        
        eks.Cluster(self, "MyCluster",
            kubectl_memory=Size.gibibytes(4),
            version=eks.KubernetesVersion.V1_32,
            kubectl_layer=KubectlV32Layer(self, "kubectl")
        )
        eks.Cluster.from_cluster_attributes(self, "MyCluster",
            kubectl_memory=Size.gibibytes(4),
            vpc=vpc,
            cluster_name="cluster-name"
        )
    '''

    @jsii.member(jsii_name="of")
    @builtins.classmethod
    def of(cls, version: builtins.str) -> "KubernetesVersion":
        '''Custom cluster version.

        :param version: custom version number.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7337c1059ea59a0eff38ee38d6366f2ba11bccf54be23296baeb1818880280cf)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        return typing.cast("KubernetesVersion", jsii.sinvoke(cls, "of", [version]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_14")
    def V1_14(cls) -> "KubernetesVersion":
        '''(deprecated) Kubernetes version 1.14.

        :deprecated: Use newer version of EKS

        :stability: deprecated
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_14"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_15")
    def V1_15(cls) -> "KubernetesVersion":
        '''(deprecated) Kubernetes version 1.15.

        :deprecated: Use newer version of EKS

        :stability: deprecated
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_15"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_16")
    def V1_16(cls) -> "KubernetesVersion":
        '''(deprecated) Kubernetes version 1.16.

        :deprecated: Use newer version of EKS

        :stability: deprecated
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_16"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_17")
    def V1_17(cls) -> "KubernetesVersion":
        '''(deprecated) Kubernetes version 1.17.

        :deprecated: Use newer version of EKS

        :stability: deprecated
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_17"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_18")
    def V1_18(cls) -> "KubernetesVersion":
        '''(deprecated) Kubernetes version 1.18.

        :deprecated: Use newer version of EKS

        :stability: deprecated
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_18"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_19")
    def V1_19(cls) -> "KubernetesVersion":
        '''(deprecated) Kubernetes version 1.19.

        :deprecated: Use newer version of EKS

        :stability: deprecated
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_19"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_20")
    def V1_20(cls) -> "KubernetesVersion":
        '''(deprecated) Kubernetes version 1.20.

        :deprecated: Use newer version of EKS

        :stability: deprecated
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_20"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_21")
    def V1_21(cls) -> "KubernetesVersion":
        '''(deprecated) Kubernetes version 1.21.

        :deprecated: Use newer version of EKS

        :stability: deprecated
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_21"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_22")
    def V1_22(cls) -> "KubernetesVersion":
        '''(deprecated) Kubernetes version 1.22.

        :deprecated: Use newer version of EKS

        :stability: deprecated
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_22"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_23")
    def V1_23(cls) -> "KubernetesVersion":
        '''Kubernetes version 1.23.

        When creating a ``Cluster`` with this version, you need to also specify the
        ``kubectlLayer`` property with a ``KubectlV23Layer`` from
        ``@aws-cdk/lambda-layer-kubectl-v23``.
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_23"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_24")
    def V1_24(cls) -> "KubernetesVersion":
        '''Kubernetes version 1.24.

        When creating a ``Cluster`` with this version, you need to also specify the
        ``kubectlLayer`` property with a ``KubectlV24Layer`` from
        ``@aws-cdk/lambda-layer-kubectl-v24``.
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_24"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_25")
    def V1_25(cls) -> "KubernetesVersion":
        '''Kubernetes version 1.25.

        When creating a ``Cluster`` with this version, you need to also specify the
        ``kubectlLayer`` property with a ``KubectlV25Layer`` from
        ``@aws-cdk/lambda-layer-kubectl-v25``.
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_25"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_26")
    def V1_26(cls) -> "KubernetesVersion":
        '''Kubernetes version 1.26.

        When creating a ``Cluster`` with this version, you need to also specify the
        ``kubectlLayer`` property with a ``KubectlV26Layer`` from
        ``@aws-cdk/lambda-layer-kubectl-v26``.
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_26"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_27")
    def V1_27(cls) -> "KubernetesVersion":
        '''Kubernetes version 1.27.

        When creating a ``Cluster`` with this version, you need to also specify the
        ``kubectlLayer`` property with a ``KubectlV27Layer`` from
        ``@aws-cdk/lambda-layer-kubectl-v27``.
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_27"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_28")
    def V1_28(cls) -> "KubernetesVersion":
        '''Kubernetes version 1.28.

        When creating a ``Cluster`` with this version, you need to also specify the
        ``kubectlLayer`` property with a ``KubectlV28Layer`` from
        ``@aws-cdk/lambda-layer-kubectl-v28``.
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_28"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_29")
    def V1_29(cls) -> "KubernetesVersion":
        '''Kubernetes version 1.29.

        When creating a ``Cluster`` with this version, you need to also specify the
        ``kubectlLayer`` property with a ``KubectlV29Layer`` from
        ``@aws-cdk/lambda-layer-kubectl-v29``.
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_29"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_30")
    def V1_30(cls) -> "KubernetesVersion":
        '''Kubernetes version 1.30.

        When creating a ``Cluster`` with this version, you need to also specify the
        ``kubectlLayer`` property with a ``KubectlV30Layer`` from
        ``@aws-cdk/lambda-layer-kubectl-v30``.
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_30"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_31")
    def V1_31(cls) -> "KubernetesVersion":
        '''Kubernetes version 1.31.

        When creating a ``Cluster`` with this version, you need to also specify the
        ``kubectlLayer`` property with a ``KubectlV31Layer`` from
        ``@aws-cdk/lambda-layer-kubectl-v31``.
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_31"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="V1_32")
    def V1_32(cls) -> "KubernetesVersion":
        '''Kubernetes version 1.32.

        When creating a ``Cluster`` with this version, you need to also specify the
        ``kubectlLayer`` property with a ``KubectlV32Layer`` from
        ``@aws-cdk/lambda-layer-kubectl-v32``.
        '''
        return typing.cast("KubernetesVersion", jsii.sget(cls, "V1_32"))

    @builtins.property
    @jsii.member(jsii_name="version")
    def version(self) -> builtins.str:
        '''cluster version number.'''
        return typing.cast(builtins.str, jsii.get(self, "version"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.LaunchTemplateSpec",
    jsii_struct_bases=[],
    name_mapping={"id": "id", "version": "version"},
)
class LaunchTemplateSpec:
    def __init__(
        self,
        *,
        id: builtins.str,
        version: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Launch template property specification.

        :param id: The Launch template ID.
        :param version: The launch template version to be used (optional). Default: - the default version of the launch template

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            
            user_data = """MIME-Version: 1.0
            Content-Type: multipart/mixed; boundary="==MYBOUNDARY=="
            
            --==MYBOUNDARY==
            Content-Type: text/x-shellscript; charset="us-ascii"
            
            #!/bin/bash
            echo "Running custom user data script"
            
            --==MYBOUNDARY==--\\
            """
            lt = ec2.CfnLaunchTemplate(self, "LaunchTemplate",
                launch_template_data=ec2.CfnLaunchTemplate.LaunchTemplateDataProperty(
                    instance_type="t3.small",
                    user_data=Fn.base64(user_data)
                )
            )
            
            cluster.add_nodegroup_capacity("extra-ng",
                launch_template_spec=eks.LaunchTemplateSpec(
                    id=lt.ref,
                    version=lt.attr_latest_version_number
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7a5df2cfdbe729211fa59821b75c6dffbf879cfb3beebf3cf51225c00e81c22c)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "id": id,
        }
        if version is not None:
            self._values["version"] = version

    @builtins.property
    def id(self) -> builtins.str:
        '''The Launch template ID.'''
        result = self._values.get("id")
        assert result is not None, "Required property 'id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def version(self) -> typing.Optional[builtins.str]:
        '''The launch template version to be used (optional).

        :default: - the default version of the launch template
        '''
        result = self._values.get("version")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LaunchTemplateSpec(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.MachineImageType")
class MachineImageType(enum.Enum):
    '''The machine image type.

    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        cluster.add_auto_scaling_group_capacity("BottlerocketNodes",
            instance_type=ec2.InstanceType("t3.small"),
            min_capacity=2,
            machine_image_type=eks.MachineImageType.BOTTLEROCKET
        )
    '''

    AMAZON_LINUX_2 = "AMAZON_LINUX_2"
    '''Amazon EKS-optimized Linux AMI.'''
    BOTTLEROCKET = "BOTTLEROCKET"
    '''Bottlerocket AMI.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.NodeType")
class NodeType(enum.Enum):
    '''Whether the worker nodes should support GPU or just standard instances.'''

    STANDARD = "STANDARD"
    '''Standard instances.'''
    GPU = "GPU"
    '''GPU instances.'''
    INFERENTIA = "INFERENTIA"
    '''Inferentia instances.'''
    TRAINIUM = "TRAINIUM"
    '''Trainium instances.'''


@jsii.implements(INodegroup)
class Nodegroup(
    _Resource_45bc6135,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.Nodegroup",
):
    '''The Nodegroup resource class.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        from aws_cdk import aws_eks as eks
        from aws_cdk import aws_iam as iam
        
        # cluster: eks.Cluster
        # instance_type: ec2.InstanceType
        # role: iam.Role
        # security_group: ec2.SecurityGroup
        # subnet: ec2.Subnet
        # subnet_filter: ec2.SubnetFilter
        
        nodegroup = eks.Nodegroup(self, "MyNodegroup",
            cluster=cluster,
        
            # the properties below are optional
            ami_type=eks.NodegroupAmiType.AL2_X86_64,
            capacity_type=eks.CapacityType.SPOT,
            desired_size=123,
            disk_size=123,
            enable_node_auto_repair=False,
            force_update=False,
            instance_types=[instance_type],
            labels={
                "labels_key": "labels"
            },
            launch_template_spec=eks.LaunchTemplateSpec(
                id="id",
        
                # the properties below are optional
                version="version"
            ),
            max_size=123,
            max_unavailable=123,
            max_unavailable_percentage=123,
            min_size=123,
            nodegroup_name="nodegroupName",
            node_role=role,
            release_version="releaseVersion",
            remote_access=eks.NodegroupRemoteAccess(
                ssh_key_name="sshKeyName",
        
                # the properties below are optional
                source_security_groups=[security_group]
            ),
            subnets=ec2.SubnetSelection(
                availability_zones=["availabilityZones"],
                one_per_az=False,
                subnet_filters=[subnet_filter],
                subnet_group_name="subnetGroupName",
                subnets=[subnet],
                subnet_type=ec2.SubnetType.PRIVATE_ISOLATED
            ),
            tags={
                "tags_key": "tags"
            },
            taints=[eks.TaintSpec(
                effect=eks.TaintEffect.NO_SCHEDULE,
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: ICluster,
        ami_type: typing.Optional["NodegroupAmiType"] = None,
        capacity_type: typing.Optional[CapacityType] = None,
        desired_size: typing.Optional[jsii.Number] = None,
        disk_size: typing.Optional[jsii.Number] = None,
        enable_node_auto_repair: typing.Optional[builtins.bool] = None,
        force_update: typing.Optional[builtins.bool] = None,
        instance_types: typing.Optional[typing.Sequence[_InstanceType_f64915b9]] = None,
        labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        launch_template_spec: typing.Optional[typing.Union[LaunchTemplateSpec, typing.Dict[builtins.str, typing.Any]]] = None,
        max_size: typing.Optional[jsii.Number] = None,
        max_unavailable: typing.Optional[jsii.Number] = None,
        max_unavailable_percentage: typing.Optional[jsii.Number] = None,
        min_size: typing.Optional[jsii.Number] = None,
        nodegroup_name: typing.Optional[builtins.str] = None,
        node_role: typing.Optional[_IRole_235f5d8e] = None,
        release_version: typing.Optional[builtins.str] = None,
        remote_access: typing.Optional[typing.Union["NodegroupRemoteAccess", typing.Dict[builtins.str, typing.Any]]] = None,
        subnets: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        taints: typing.Optional[typing.Sequence[typing.Union["TaintSpec", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster: Cluster resource.
        :param ami_type: The AMI type for your node group. If you explicitly specify the launchTemplate with custom AMI, do not specify this property, or the node group deployment will fail. In other cases, you will need to specify correct amiType for the nodegroup. Default: - auto-determined from the instanceTypes property when launchTemplateSpec property is not specified
        :param capacity_type: The capacity type of the nodegroup. Default: - ON_DEMAND
        :param desired_size: The current number of worker nodes that the managed node group should maintain. If not specified, the nodewgroup will initially create ``minSize`` instances. Default: 2
        :param disk_size: The root device disk size (in GiB) for your node group instances. Default: 20
        :param enable_node_auto_repair: Specifies whether to enable node auto repair for the node group. Node auto repair is disabled by default. Default: - disabled
        :param force_update: Force the update if the existing node group's pods are unable to be drained due to a pod disruption budget issue. If an update fails because pods could not be drained, you can force the update after it fails to terminate the old node whether or not any pods are running on the node. Default: true
        :param instance_types: The instance types to use for your node group. Default: t3.medium will be used according to the cloudformation document.
        :param labels: The Kubernetes labels to be applied to the nodes in the node group when they are created. Default: - None
        :param launch_template_spec: Launch template specification used for the nodegroup. Default: - no launch template
        :param max_size: The maximum number of worker nodes that the managed node group can scale out to. Managed node groups can support up to 100 nodes by default. Default: - desiredSize
        :param max_unavailable: The maximum number of nodes unavailable at once during a version update. Nodes will be updated in parallel. The maximum number is 100. This value or ``maxUnavailablePercentage`` is required to have a value for custom update configurations to be applied. Default: 1
        :param max_unavailable_percentage: The maximum percentage of nodes unavailable during a version update. This percentage of nodes will be updated in parallel, up to 100 nodes at once. This value or ``maxUnavailable`` is required to have a value for custom update configurations to be applied. Default: undefined - node groups will update instances one at a time
        :param min_size: The minimum number of worker nodes that the managed node group can scale in to. This number must be greater than or equal to zero. Default: 1
        :param nodegroup_name: Name of the Nodegroup. Default: - resource ID
        :param node_role: The IAM role to associate with your node group. The Amazon EKS worker node kubelet daemon makes calls to AWS APIs on your behalf. Worker nodes receive permissions for these API calls through an IAM instance profile and associated policies. Before you can launch worker nodes and register them into a cluster, you must create an IAM role for those worker nodes to use when they are launched. Default: - None. Auto-generated if not specified.
        :param release_version: The AMI version of the Amazon EKS-optimized AMI to use with your node group (for example, ``1.14.7-YYYYMMDD``). Default: - The latest available AMI version for the node group's current Kubernetes version is used.
        :param remote_access: The remote access (SSH) configuration to use with your node group. Disabled by default, however, if you specify an Amazon EC2 SSH key but do not specify a source security group when you create a managed node group, then port 22 on the worker nodes is opened to the internet (0.0.0.0/0) Default: - disabled
        :param subnets: The subnets to use for the Auto Scaling group that is created for your node group. By specifying the SubnetSelection, the selected subnets will automatically apply required tags i.e. ``kubernetes.io/cluster/CLUSTER_NAME`` with a value of ``shared``, where ``CLUSTER_NAME`` is replaced with the name of your cluster. Default: - private subnets
        :param tags: The metadata to apply to the node group to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Node group tags do not propagate to any other resources associated with the node group, such as the Amazon EC2 instances or subnets. Default: - None
        :param taints: The Kubernetes taints to be applied to the nodes in the node group when they are created. Default: - None
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__786fdb6d4c92794281e2a7ea4e6305e651a12884bcdc063cf6fe5a64cd52c68d)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = NodegroupProps(
            cluster=cluster,
            ami_type=ami_type,
            capacity_type=capacity_type,
            desired_size=desired_size,
            disk_size=disk_size,
            enable_node_auto_repair=enable_node_auto_repair,
            force_update=force_update,
            instance_types=instance_types,
            labels=labels,
            launch_template_spec=launch_template_spec,
            max_size=max_size,
            max_unavailable=max_unavailable,
            max_unavailable_percentage=max_unavailable_percentage,
            min_size=min_size,
            nodegroup_name=nodegroup_name,
            node_role=node_role,
            release_version=release_version,
            remote_access=remote_access,
            subnets=subnets,
            tags=tags,
            taints=taints,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="fromNodegroupName")
    @builtins.classmethod
    def from_nodegroup_name(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        nodegroup_name: builtins.str,
    ) -> INodegroup:
        '''Import the Nodegroup from attributes.

        :param scope: -
        :param id: -
        :param nodegroup_name: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__853fd7be23ae253444f7875436479f89054ff8c6552c1b486b1c86e24e94b6be)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument nodegroup_name", value=nodegroup_name, expected_type=type_hints["nodegroup_name"])
        return typing.cast(INodegroup, jsii.sinvoke(cls, "fromNodegroupName", [scope, id, nodegroup_name]))

    @builtins.property
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> ICluster:
        '''the Amazon EKS cluster resource.

        :attribute: ClusterName
        '''
        return typing.cast(ICluster, jsii.get(self, "cluster"))

    @builtins.property
    @jsii.member(jsii_name="nodegroupArn")
    def nodegroup_arn(self) -> builtins.str:
        '''ARN of the nodegroup.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "nodegroupArn"))

    @builtins.property
    @jsii.member(jsii_name="nodegroupName")
    def nodegroup_name(self) -> builtins.str:
        '''Nodegroup name.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "nodegroupName"))

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> _IRole_235f5d8e:
        '''IAM role of the instance profile for the nodegroup.'''
        return typing.cast(_IRole_235f5d8e, jsii.get(self, "role"))


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.NodegroupAmiType")
class NodegroupAmiType(enum.Enum):
    '''The AMI type for your node group.

    GPU instance types should use the ``AL2_x86_64_GPU`` AMI type, which uses the
    Amazon EKS-optimized Linux AMI with GPU support or the ``BOTTLEROCKET_ARM_64_NVIDIA`` or ``BOTTLEROCKET_X86_64_NVIDIA``
    AMI types, which uses the Amazon EKS-optimized Linux AMI with Nvidia-GPU support.

    Non-GPU instances should use the ``AL2_x86_64`` AMI type, which uses the Amazon EKS-optimized Linux AMI.

    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        cluster.add_nodegroup_capacity("BottlerocketNvidiaNG",
            ami_type=eks.NodegroupAmiType.BOTTLEROCKET_X86_64_NVIDIA,
            instance_types=[ec2.InstanceType("g4dn.xlarge")]
        )
    '''

    AL2_X86_64 = "AL2_X86_64"
    '''Amazon Linux 2 (x86-64).'''
    AL2_X86_64_GPU = "AL2_X86_64_GPU"
    '''Amazon Linux 2 with GPU support.'''
    AL2_ARM_64 = "AL2_ARM_64"
    '''Amazon Linux 2 (ARM-64).'''
    BOTTLEROCKET_ARM_64 = "BOTTLEROCKET_ARM_64"
    '''Bottlerocket Linux (ARM-64).'''
    BOTTLEROCKET_X86_64 = "BOTTLEROCKET_X86_64"
    '''Bottlerocket (x86-64).'''
    BOTTLEROCKET_ARM_64_NVIDIA = "BOTTLEROCKET_ARM_64_NVIDIA"
    '''Bottlerocket Linux with Nvidia-GPU support (ARM-64).'''
    BOTTLEROCKET_X86_64_NVIDIA = "BOTTLEROCKET_X86_64_NVIDIA"
    '''Bottlerocket with Nvidia-GPU support (x86-64).'''
    BOTTLEROCKET_ARM_64_FIPS = "BOTTLEROCKET_ARM_64_FIPS"
    '''Bottlerocket Linux (ARM-64) with FIPS enabled.'''
    BOTTLEROCKET_X86_64_FIPS = "BOTTLEROCKET_X86_64_FIPS"
    '''Bottlerocket (x86-64) with FIPS enabled.'''
    WINDOWS_CORE_2019_X86_64 = "WINDOWS_CORE_2019_X86_64"
    '''Windows Core 2019 (x86-64).'''
    WINDOWS_CORE_2022_X86_64 = "WINDOWS_CORE_2022_X86_64"
    '''Windows Core 2022 (x86-64).'''
    WINDOWS_FULL_2019_X86_64 = "WINDOWS_FULL_2019_X86_64"
    '''Windows Full 2019 (x86-64).'''
    WINDOWS_FULL_2022_X86_64 = "WINDOWS_FULL_2022_X86_64"
    '''Windows Full 2022 (x86-64).'''
    AL2023_X86_64_STANDARD = "AL2023_X86_64_STANDARD"
    '''Amazon Linux 2023 (x86-64).'''
    AL2023_X86_64_NEURON = "AL2023_X86_64_NEURON"
    '''Amazon Linux 2023 with AWS Neuron drivers (x86-64).'''
    AL2023_X86_64_NVIDIA = "AL2023_X86_64_NVIDIA"
    '''Amazon Linux 2023 with NVIDIA drivers (x86-64).'''
    AL2023_ARM_64_STANDARD = "AL2023_ARM_64_STANDARD"
    '''Amazon Linux 2023 (ARM-64).'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.NodegroupOptions",
    jsii_struct_bases=[],
    name_mapping={
        "ami_type": "amiType",
        "capacity_type": "capacityType",
        "desired_size": "desiredSize",
        "disk_size": "diskSize",
        "enable_node_auto_repair": "enableNodeAutoRepair",
        "force_update": "forceUpdate",
        "instance_types": "instanceTypes",
        "labels": "labels",
        "launch_template_spec": "launchTemplateSpec",
        "max_size": "maxSize",
        "max_unavailable": "maxUnavailable",
        "max_unavailable_percentage": "maxUnavailablePercentage",
        "min_size": "minSize",
        "nodegroup_name": "nodegroupName",
        "node_role": "nodeRole",
        "release_version": "releaseVersion",
        "remote_access": "remoteAccess",
        "subnets": "subnets",
        "tags": "tags",
        "taints": "taints",
    },
)
class NodegroupOptions:
    def __init__(
        self,
        *,
        ami_type: typing.Optional[NodegroupAmiType] = None,
        capacity_type: typing.Optional[CapacityType] = None,
        desired_size: typing.Optional[jsii.Number] = None,
        disk_size: typing.Optional[jsii.Number] = None,
        enable_node_auto_repair: typing.Optional[builtins.bool] = None,
        force_update: typing.Optional[builtins.bool] = None,
        instance_types: typing.Optional[typing.Sequence[_InstanceType_f64915b9]] = None,
        labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        launch_template_spec: typing.Optional[typing.Union[LaunchTemplateSpec, typing.Dict[builtins.str, typing.Any]]] = None,
        max_size: typing.Optional[jsii.Number] = None,
        max_unavailable: typing.Optional[jsii.Number] = None,
        max_unavailable_percentage: typing.Optional[jsii.Number] = None,
        min_size: typing.Optional[jsii.Number] = None,
        nodegroup_name: typing.Optional[builtins.str] = None,
        node_role: typing.Optional[_IRole_235f5d8e] = None,
        release_version: typing.Optional[builtins.str] = None,
        remote_access: typing.Optional[typing.Union["NodegroupRemoteAccess", typing.Dict[builtins.str, typing.Any]]] = None,
        subnets: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        taints: typing.Optional[typing.Sequence[typing.Union["TaintSpec", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''The Nodegroup Options for addNodeGroup() method.

        :param ami_type: The AMI type for your node group. If you explicitly specify the launchTemplate with custom AMI, do not specify this property, or the node group deployment will fail. In other cases, you will need to specify correct amiType for the nodegroup. Default: - auto-determined from the instanceTypes property when launchTemplateSpec property is not specified
        :param capacity_type: The capacity type of the nodegroup. Default: - ON_DEMAND
        :param desired_size: The current number of worker nodes that the managed node group should maintain. If not specified, the nodewgroup will initially create ``minSize`` instances. Default: 2
        :param disk_size: The root device disk size (in GiB) for your node group instances. Default: 20
        :param enable_node_auto_repair: Specifies whether to enable node auto repair for the node group. Node auto repair is disabled by default. Default: - disabled
        :param force_update: Force the update if the existing node group's pods are unable to be drained due to a pod disruption budget issue. If an update fails because pods could not be drained, you can force the update after it fails to terminate the old node whether or not any pods are running on the node. Default: true
        :param instance_types: The instance types to use for your node group. Default: t3.medium will be used according to the cloudformation document.
        :param labels: The Kubernetes labels to be applied to the nodes in the node group when they are created. Default: - None
        :param launch_template_spec: Launch template specification used for the nodegroup. Default: - no launch template
        :param max_size: The maximum number of worker nodes that the managed node group can scale out to. Managed node groups can support up to 100 nodes by default. Default: - desiredSize
        :param max_unavailable: The maximum number of nodes unavailable at once during a version update. Nodes will be updated in parallel. The maximum number is 100. This value or ``maxUnavailablePercentage`` is required to have a value for custom update configurations to be applied. Default: 1
        :param max_unavailable_percentage: The maximum percentage of nodes unavailable during a version update. This percentage of nodes will be updated in parallel, up to 100 nodes at once. This value or ``maxUnavailable`` is required to have a value for custom update configurations to be applied. Default: undefined - node groups will update instances one at a time
        :param min_size: The minimum number of worker nodes that the managed node group can scale in to. This number must be greater than or equal to zero. Default: 1
        :param nodegroup_name: Name of the Nodegroup. Default: - resource ID
        :param node_role: The IAM role to associate with your node group. The Amazon EKS worker node kubelet daemon makes calls to AWS APIs on your behalf. Worker nodes receive permissions for these API calls through an IAM instance profile and associated policies. Before you can launch worker nodes and register them into a cluster, you must create an IAM role for those worker nodes to use when they are launched. Default: - None. Auto-generated if not specified.
        :param release_version: The AMI version of the Amazon EKS-optimized AMI to use with your node group (for example, ``1.14.7-YYYYMMDD``). Default: - The latest available AMI version for the node group's current Kubernetes version is used.
        :param remote_access: The remote access (SSH) configuration to use with your node group. Disabled by default, however, if you specify an Amazon EC2 SSH key but do not specify a source security group when you create a managed node group, then port 22 on the worker nodes is opened to the internet (0.0.0.0/0) Default: - disabled
        :param subnets: The subnets to use for the Auto Scaling group that is created for your node group. By specifying the SubnetSelection, the selected subnets will automatically apply required tags i.e. ``kubernetes.io/cluster/CLUSTER_NAME`` with a value of ``shared``, where ``CLUSTER_NAME`` is replaced with the name of your cluster. Default: - private subnets
        :param tags: The metadata to apply to the node group to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Node group tags do not propagate to any other resources associated with the node group, such as the Amazon EC2 instances or subnets. Default: - None
        :param taints: The Kubernetes taints to be applied to the nodes in the node group when they are created. Default: - None

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            cluster.add_nodegroup_capacity("extra-ng-spot",
                instance_types=[
                    ec2.InstanceType("c5.large"),
                    ec2.InstanceType("c5a.large"),
                    ec2.InstanceType("c5d.large")
                ],
                min_size=3,
                capacity_type=eks.CapacityType.SPOT
            )
        '''
        if isinstance(launch_template_spec, dict):
            launch_template_spec = LaunchTemplateSpec(**launch_template_spec)
        if isinstance(remote_access, dict):
            remote_access = NodegroupRemoteAccess(**remote_access)
        if isinstance(subnets, dict):
            subnets = _SubnetSelection_e57d76df(**subnets)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__019154b49c40dcdcdd533db7688b958baf443f513473d96721969a0d5f3d2ffa)
            check_type(argname="argument ami_type", value=ami_type, expected_type=type_hints["ami_type"])
            check_type(argname="argument capacity_type", value=capacity_type, expected_type=type_hints["capacity_type"])
            check_type(argname="argument desired_size", value=desired_size, expected_type=type_hints["desired_size"])
            check_type(argname="argument disk_size", value=disk_size, expected_type=type_hints["disk_size"])
            check_type(argname="argument enable_node_auto_repair", value=enable_node_auto_repair, expected_type=type_hints["enable_node_auto_repair"])
            check_type(argname="argument force_update", value=force_update, expected_type=type_hints["force_update"])
            check_type(argname="argument instance_types", value=instance_types, expected_type=type_hints["instance_types"])
            check_type(argname="argument labels", value=labels, expected_type=type_hints["labels"])
            check_type(argname="argument launch_template_spec", value=launch_template_spec, expected_type=type_hints["launch_template_spec"])
            check_type(argname="argument max_size", value=max_size, expected_type=type_hints["max_size"])
            check_type(argname="argument max_unavailable", value=max_unavailable, expected_type=type_hints["max_unavailable"])
            check_type(argname="argument max_unavailable_percentage", value=max_unavailable_percentage, expected_type=type_hints["max_unavailable_percentage"])
            check_type(argname="argument min_size", value=min_size, expected_type=type_hints["min_size"])
            check_type(argname="argument nodegroup_name", value=nodegroup_name, expected_type=type_hints["nodegroup_name"])
            check_type(argname="argument node_role", value=node_role, expected_type=type_hints["node_role"])
            check_type(argname="argument release_version", value=release_version, expected_type=type_hints["release_version"])
            check_type(argname="argument remote_access", value=remote_access, expected_type=type_hints["remote_access"])
            check_type(argname="argument subnets", value=subnets, expected_type=type_hints["subnets"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument taints", value=taints, expected_type=type_hints["taints"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if ami_type is not None:
            self._values["ami_type"] = ami_type
        if capacity_type is not None:
            self._values["capacity_type"] = capacity_type
        if desired_size is not None:
            self._values["desired_size"] = desired_size
        if disk_size is not None:
            self._values["disk_size"] = disk_size
        if enable_node_auto_repair is not None:
            self._values["enable_node_auto_repair"] = enable_node_auto_repair
        if force_update is not None:
            self._values["force_update"] = force_update
        if instance_types is not None:
            self._values["instance_types"] = instance_types
        if labels is not None:
            self._values["labels"] = labels
        if launch_template_spec is not None:
            self._values["launch_template_spec"] = launch_template_spec
        if max_size is not None:
            self._values["max_size"] = max_size
        if max_unavailable is not None:
            self._values["max_unavailable"] = max_unavailable
        if max_unavailable_percentage is not None:
            self._values["max_unavailable_percentage"] = max_unavailable_percentage
        if min_size is not None:
            self._values["min_size"] = min_size
        if nodegroup_name is not None:
            self._values["nodegroup_name"] = nodegroup_name
        if node_role is not None:
            self._values["node_role"] = node_role
        if release_version is not None:
            self._values["release_version"] = release_version
        if remote_access is not None:
            self._values["remote_access"] = remote_access
        if subnets is not None:
            self._values["subnets"] = subnets
        if tags is not None:
            self._values["tags"] = tags
        if taints is not None:
            self._values["taints"] = taints

    @builtins.property
    def ami_type(self) -> typing.Optional[NodegroupAmiType]:
        '''The AMI type for your node group.

        If you explicitly specify the launchTemplate with custom AMI, do not specify this property, or
        the node group deployment will fail. In other cases, you will need to specify correct amiType for the nodegroup.

        :default: - auto-determined from the instanceTypes property when launchTemplateSpec property is not specified
        '''
        result = self._values.get("ami_type")
        return typing.cast(typing.Optional[NodegroupAmiType], result)

    @builtins.property
    def capacity_type(self) -> typing.Optional[CapacityType]:
        '''The capacity type of the nodegroup.

        :default: - ON_DEMAND
        '''
        result = self._values.get("capacity_type")
        return typing.cast(typing.Optional[CapacityType], result)

    @builtins.property
    def desired_size(self) -> typing.Optional[jsii.Number]:
        '''The current number of worker nodes that the managed node group should maintain.

        If not specified,
        the nodewgroup will initially create ``minSize`` instances.

        :default: 2
        '''
        result = self._values.get("desired_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def disk_size(self) -> typing.Optional[jsii.Number]:
        '''The root device disk size (in GiB) for your node group instances.

        :default: 20
        '''
        result = self._values.get("disk_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_node_auto_repair(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether to enable node auto repair for the node group.

        Node auto repair is disabled by default.

        :default: - disabled

        :see: https://docs.aws.amazon.com/eks/latest/userguide/node-health.html#node-auto-repair
        '''
        result = self._values.get("enable_node_auto_repair")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def force_update(self) -> typing.Optional[builtins.bool]:
        '''Force the update if the existing node group's pods are unable to be drained due to a pod disruption budget issue.

        If an update fails because pods could not be drained, you can force the update after it fails to terminate the old
        node whether or not any pods are
        running on the node.

        :default: true
        '''
        result = self._values.get("force_update")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def instance_types(self) -> typing.Optional[typing.List[_InstanceType_f64915b9]]:
        '''The instance types to use for your node group.

        :default: t3.medium will be used according to the cloudformation document.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-instancetypes
        '''
        result = self._values.get("instance_types")
        return typing.cast(typing.Optional[typing.List[_InstanceType_f64915b9]], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The Kubernetes labels to be applied to the nodes in the node group when they are created.

        :default: - None
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def launch_template_spec(self) -> typing.Optional[LaunchTemplateSpec]:
        '''Launch template specification used for the nodegroup.

        :default: - no launch template

        :see: https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html
        '''
        result = self._values.get("launch_template_spec")
        return typing.cast(typing.Optional[LaunchTemplateSpec], result)

    @builtins.property
    def max_size(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of worker nodes that the managed node group can scale out to.

        Managed node groups can support up to 100 nodes by default.

        :default: - desiredSize
        '''
        result = self._values.get("max_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_unavailable(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of nodes unavailable at once during a version update.

        Nodes will be updated in parallel. The maximum number is 100.

        This value or ``maxUnavailablePercentage`` is required to have a value for custom update configurations to be applied.

        :default: 1

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-updateconfig.html#cfn-eks-nodegroup-updateconfig-maxunavailable
        '''
        result = self._values.get("max_unavailable")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_unavailable_percentage(self) -> typing.Optional[jsii.Number]:
        '''The maximum percentage of nodes unavailable during a version update.

        This percentage of nodes will be updated in parallel, up to 100 nodes at once.

        This value or ``maxUnavailable`` is required to have a value for custom update configurations to be applied.

        :default: undefined - node groups will update instances one at a time

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-updateconfig.html#cfn-eks-nodegroup-updateconfig-maxunavailablepercentage
        '''
        result = self._values.get("max_unavailable_percentage")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_size(self) -> typing.Optional[jsii.Number]:
        '''The minimum number of worker nodes that the managed node group can scale in to.

        This number must be greater than or equal to zero.

        :default: 1
        '''
        result = self._values.get("min_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def nodegroup_name(self) -> typing.Optional[builtins.str]:
        '''Name of the Nodegroup.

        :default: - resource ID
        '''
        result = self._values.get("nodegroup_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def node_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The IAM role to associate with your node group.

        The Amazon EKS worker node kubelet daemon
        makes calls to AWS APIs on your behalf. Worker nodes receive permissions for these API calls through
        an IAM instance profile and associated policies. Before you can launch worker nodes and register them
        into a cluster, you must create an IAM role for those worker nodes to use when they are launched.

        :default: - None. Auto-generated if not specified.
        '''
        result = self._values.get("node_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def release_version(self) -> typing.Optional[builtins.str]:
        '''The AMI version of the Amazon EKS-optimized AMI to use with your node group (for example, ``1.14.7-YYYYMMDD``).

        :default: - The latest available AMI version for the node group's current Kubernetes version is used.
        '''
        result = self._values.get("release_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def remote_access(self) -> typing.Optional["NodegroupRemoteAccess"]:
        '''The remote access (SSH) configuration to use with your node group.

        Disabled by default, however, if you
        specify an Amazon EC2 SSH key but do not specify a source security group when you create a managed node group,
        then port 22 on the worker nodes is opened to the internet (0.0.0.0/0)

        :default: - disabled
        '''
        result = self._values.get("remote_access")
        return typing.cast(typing.Optional["NodegroupRemoteAccess"], result)

    @builtins.property
    def subnets(self) -> typing.Optional[_SubnetSelection_e57d76df]:
        '''The subnets to use for the Auto Scaling group that is created for your node group.

        By specifying the
        SubnetSelection, the selected subnets will automatically apply required tags i.e.
        ``kubernetes.io/cluster/CLUSTER_NAME`` with a value of ``shared``, where ``CLUSTER_NAME`` is replaced with
        the name of your cluster.

        :default: - private subnets
        '''
        result = self._values.get("subnets")
        return typing.cast(typing.Optional[_SubnetSelection_e57d76df], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The metadata to apply to the node group to assist with categorization and organization.

        Each tag consists of
        a key and an optional value, both of which you define. Node group tags do not propagate to any other resources
        associated with the node group, such as the Amazon EC2 instances or subnets.

        :default: - None
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def taints(self) -> typing.Optional[typing.List["TaintSpec"]]:
        '''The Kubernetes taints to be applied to the nodes in the node group when they are created.

        :default: - None
        '''
        result = self._values.get("taints")
        return typing.cast(typing.Optional[typing.List["TaintSpec"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "NodegroupOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.NodegroupProps",
    jsii_struct_bases=[NodegroupOptions],
    name_mapping={
        "ami_type": "amiType",
        "capacity_type": "capacityType",
        "desired_size": "desiredSize",
        "disk_size": "diskSize",
        "enable_node_auto_repair": "enableNodeAutoRepair",
        "force_update": "forceUpdate",
        "instance_types": "instanceTypes",
        "labels": "labels",
        "launch_template_spec": "launchTemplateSpec",
        "max_size": "maxSize",
        "max_unavailable": "maxUnavailable",
        "max_unavailable_percentage": "maxUnavailablePercentage",
        "min_size": "minSize",
        "nodegroup_name": "nodegroupName",
        "node_role": "nodeRole",
        "release_version": "releaseVersion",
        "remote_access": "remoteAccess",
        "subnets": "subnets",
        "tags": "tags",
        "taints": "taints",
        "cluster": "cluster",
    },
)
class NodegroupProps(NodegroupOptions):
    def __init__(
        self,
        *,
        ami_type: typing.Optional[NodegroupAmiType] = None,
        capacity_type: typing.Optional[CapacityType] = None,
        desired_size: typing.Optional[jsii.Number] = None,
        disk_size: typing.Optional[jsii.Number] = None,
        enable_node_auto_repair: typing.Optional[builtins.bool] = None,
        force_update: typing.Optional[builtins.bool] = None,
        instance_types: typing.Optional[typing.Sequence[_InstanceType_f64915b9]] = None,
        labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        launch_template_spec: typing.Optional[typing.Union[LaunchTemplateSpec, typing.Dict[builtins.str, typing.Any]]] = None,
        max_size: typing.Optional[jsii.Number] = None,
        max_unavailable: typing.Optional[jsii.Number] = None,
        max_unavailable_percentage: typing.Optional[jsii.Number] = None,
        min_size: typing.Optional[jsii.Number] = None,
        nodegroup_name: typing.Optional[builtins.str] = None,
        node_role: typing.Optional[_IRole_235f5d8e] = None,
        release_version: typing.Optional[builtins.str] = None,
        remote_access: typing.Optional[typing.Union["NodegroupRemoteAccess", typing.Dict[builtins.str, typing.Any]]] = None,
        subnets: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        taints: typing.Optional[typing.Sequence[typing.Union["TaintSpec", typing.Dict[builtins.str, typing.Any]]]] = None,
        cluster: ICluster,
    ) -> None:
        '''NodeGroup properties interface.

        :param ami_type: The AMI type for your node group. If you explicitly specify the launchTemplate with custom AMI, do not specify this property, or the node group deployment will fail. In other cases, you will need to specify correct amiType for the nodegroup. Default: - auto-determined from the instanceTypes property when launchTemplateSpec property is not specified
        :param capacity_type: The capacity type of the nodegroup. Default: - ON_DEMAND
        :param desired_size: The current number of worker nodes that the managed node group should maintain. If not specified, the nodewgroup will initially create ``minSize`` instances. Default: 2
        :param disk_size: The root device disk size (in GiB) for your node group instances. Default: 20
        :param enable_node_auto_repair: Specifies whether to enable node auto repair for the node group. Node auto repair is disabled by default. Default: - disabled
        :param force_update: Force the update if the existing node group's pods are unable to be drained due to a pod disruption budget issue. If an update fails because pods could not be drained, you can force the update after it fails to terminate the old node whether or not any pods are running on the node. Default: true
        :param instance_types: The instance types to use for your node group. Default: t3.medium will be used according to the cloudformation document.
        :param labels: The Kubernetes labels to be applied to the nodes in the node group when they are created. Default: - None
        :param launch_template_spec: Launch template specification used for the nodegroup. Default: - no launch template
        :param max_size: The maximum number of worker nodes that the managed node group can scale out to. Managed node groups can support up to 100 nodes by default. Default: - desiredSize
        :param max_unavailable: The maximum number of nodes unavailable at once during a version update. Nodes will be updated in parallel. The maximum number is 100. This value or ``maxUnavailablePercentage`` is required to have a value for custom update configurations to be applied. Default: 1
        :param max_unavailable_percentage: The maximum percentage of nodes unavailable during a version update. This percentage of nodes will be updated in parallel, up to 100 nodes at once. This value or ``maxUnavailable`` is required to have a value for custom update configurations to be applied. Default: undefined - node groups will update instances one at a time
        :param min_size: The minimum number of worker nodes that the managed node group can scale in to. This number must be greater than or equal to zero. Default: 1
        :param nodegroup_name: Name of the Nodegroup. Default: - resource ID
        :param node_role: The IAM role to associate with your node group. The Amazon EKS worker node kubelet daemon makes calls to AWS APIs on your behalf. Worker nodes receive permissions for these API calls through an IAM instance profile and associated policies. Before you can launch worker nodes and register them into a cluster, you must create an IAM role for those worker nodes to use when they are launched. Default: - None. Auto-generated if not specified.
        :param release_version: The AMI version of the Amazon EKS-optimized AMI to use with your node group (for example, ``1.14.7-YYYYMMDD``). Default: - The latest available AMI version for the node group's current Kubernetes version is used.
        :param remote_access: The remote access (SSH) configuration to use with your node group. Disabled by default, however, if you specify an Amazon EC2 SSH key but do not specify a source security group when you create a managed node group, then port 22 on the worker nodes is opened to the internet (0.0.0.0/0) Default: - disabled
        :param subnets: The subnets to use for the Auto Scaling group that is created for your node group. By specifying the SubnetSelection, the selected subnets will automatically apply required tags i.e. ``kubernetes.io/cluster/CLUSTER_NAME`` with a value of ``shared``, where ``CLUSTER_NAME`` is replaced with the name of your cluster. Default: - private subnets
        :param tags: The metadata to apply to the node group to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Node group tags do not propagate to any other resources associated with the node group, such as the Amazon EC2 instances or subnets. Default: - None
        :param taints: The Kubernetes taints to be applied to the nodes in the node group when they are created. Default: - None
        :param cluster: Cluster resource.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_eks as eks
            from aws_cdk import aws_iam as iam
            
            # cluster: eks.Cluster
            # instance_type: ec2.InstanceType
            # role: iam.Role
            # security_group: ec2.SecurityGroup
            # subnet: ec2.Subnet
            # subnet_filter: ec2.SubnetFilter
            
            nodegroup_props = eks.NodegroupProps(
                cluster=cluster,
            
                # the properties below are optional
                ami_type=eks.NodegroupAmiType.AL2_X86_64,
                capacity_type=eks.CapacityType.SPOT,
                desired_size=123,
                disk_size=123,
                enable_node_auto_repair=False,
                force_update=False,
                instance_types=[instance_type],
                labels={
                    "labels_key": "labels"
                },
                launch_template_spec=eks.LaunchTemplateSpec(
                    id="id",
            
                    # the properties below are optional
                    version="version"
                ),
                max_size=123,
                max_unavailable=123,
                max_unavailable_percentage=123,
                min_size=123,
                nodegroup_name="nodegroupName",
                node_role=role,
                release_version="releaseVersion",
                remote_access=eks.NodegroupRemoteAccess(
                    ssh_key_name="sshKeyName",
            
                    # the properties below are optional
                    source_security_groups=[security_group]
                ),
                subnets=ec2.SubnetSelection(
                    availability_zones=["availabilityZones"],
                    one_per_az=False,
                    subnet_filters=[subnet_filter],
                    subnet_group_name="subnetGroupName",
                    subnets=[subnet],
                    subnet_type=ec2.SubnetType.PRIVATE_ISOLATED
                ),
                tags={
                    "tags_key": "tags"
                },
                taints=[eks.TaintSpec(
                    effect=eks.TaintEffect.NO_SCHEDULE,
                    key="key",
                    value="value"
                )]
            )
        '''
        if isinstance(launch_template_spec, dict):
            launch_template_spec = LaunchTemplateSpec(**launch_template_spec)
        if isinstance(remote_access, dict):
            remote_access = NodegroupRemoteAccess(**remote_access)
        if isinstance(subnets, dict):
            subnets = _SubnetSelection_e57d76df(**subnets)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c522b5bc1207d683d19377a7c1fae066520518733e5dc4e1b4952f88cfec5456)
            check_type(argname="argument ami_type", value=ami_type, expected_type=type_hints["ami_type"])
            check_type(argname="argument capacity_type", value=capacity_type, expected_type=type_hints["capacity_type"])
            check_type(argname="argument desired_size", value=desired_size, expected_type=type_hints["desired_size"])
            check_type(argname="argument disk_size", value=disk_size, expected_type=type_hints["disk_size"])
            check_type(argname="argument enable_node_auto_repair", value=enable_node_auto_repair, expected_type=type_hints["enable_node_auto_repair"])
            check_type(argname="argument force_update", value=force_update, expected_type=type_hints["force_update"])
            check_type(argname="argument instance_types", value=instance_types, expected_type=type_hints["instance_types"])
            check_type(argname="argument labels", value=labels, expected_type=type_hints["labels"])
            check_type(argname="argument launch_template_spec", value=launch_template_spec, expected_type=type_hints["launch_template_spec"])
            check_type(argname="argument max_size", value=max_size, expected_type=type_hints["max_size"])
            check_type(argname="argument max_unavailable", value=max_unavailable, expected_type=type_hints["max_unavailable"])
            check_type(argname="argument max_unavailable_percentage", value=max_unavailable_percentage, expected_type=type_hints["max_unavailable_percentage"])
            check_type(argname="argument min_size", value=min_size, expected_type=type_hints["min_size"])
            check_type(argname="argument nodegroup_name", value=nodegroup_name, expected_type=type_hints["nodegroup_name"])
            check_type(argname="argument node_role", value=node_role, expected_type=type_hints["node_role"])
            check_type(argname="argument release_version", value=release_version, expected_type=type_hints["release_version"])
            check_type(argname="argument remote_access", value=remote_access, expected_type=type_hints["remote_access"])
            check_type(argname="argument subnets", value=subnets, expected_type=type_hints["subnets"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument taints", value=taints, expected_type=type_hints["taints"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
        }
        if ami_type is not None:
            self._values["ami_type"] = ami_type
        if capacity_type is not None:
            self._values["capacity_type"] = capacity_type
        if desired_size is not None:
            self._values["desired_size"] = desired_size
        if disk_size is not None:
            self._values["disk_size"] = disk_size
        if enable_node_auto_repair is not None:
            self._values["enable_node_auto_repair"] = enable_node_auto_repair
        if force_update is not None:
            self._values["force_update"] = force_update
        if instance_types is not None:
            self._values["instance_types"] = instance_types
        if labels is not None:
            self._values["labels"] = labels
        if launch_template_spec is not None:
            self._values["launch_template_spec"] = launch_template_spec
        if max_size is not None:
            self._values["max_size"] = max_size
        if max_unavailable is not None:
            self._values["max_unavailable"] = max_unavailable
        if max_unavailable_percentage is not None:
            self._values["max_unavailable_percentage"] = max_unavailable_percentage
        if min_size is not None:
            self._values["min_size"] = min_size
        if nodegroup_name is not None:
            self._values["nodegroup_name"] = nodegroup_name
        if node_role is not None:
            self._values["node_role"] = node_role
        if release_version is not None:
            self._values["release_version"] = release_version
        if remote_access is not None:
            self._values["remote_access"] = remote_access
        if subnets is not None:
            self._values["subnets"] = subnets
        if tags is not None:
            self._values["tags"] = tags
        if taints is not None:
            self._values["taints"] = taints

    @builtins.property
    def ami_type(self) -> typing.Optional[NodegroupAmiType]:
        '''The AMI type for your node group.

        If you explicitly specify the launchTemplate with custom AMI, do not specify this property, or
        the node group deployment will fail. In other cases, you will need to specify correct amiType for the nodegroup.

        :default: - auto-determined from the instanceTypes property when launchTemplateSpec property is not specified
        '''
        result = self._values.get("ami_type")
        return typing.cast(typing.Optional[NodegroupAmiType], result)

    @builtins.property
    def capacity_type(self) -> typing.Optional[CapacityType]:
        '''The capacity type of the nodegroup.

        :default: - ON_DEMAND
        '''
        result = self._values.get("capacity_type")
        return typing.cast(typing.Optional[CapacityType], result)

    @builtins.property
    def desired_size(self) -> typing.Optional[jsii.Number]:
        '''The current number of worker nodes that the managed node group should maintain.

        If not specified,
        the nodewgroup will initially create ``minSize`` instances.

        :default: 2
        '''
        result = self._values.get("desired_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def disk_size(self) -> typing.Optional[jsii.Number]:
        '''The root device disk size (in GiB) for your node group instances.

        :default: 20
        '''
        result = self._values.get("disk_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_node_auto_repair(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether to enable node auto repair for the node group.

        Node auto repair is disabled by default.

        :default: - disabled

        :see: https://docs.aws.amazon.com/eks/latest/userguide/node-health.html#node-auto-repair
        '''
        result = self._values.get("enable_node_auto_repair")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def force_update(self) -> typing.Optional[builtins.bool]:
        '''Force the update if the existing node group's pods are unable to be drained due to a pod disruption budget issue.

        If an update fails because pods could not be drained, you can force the update after it fails to terminate the old
        node whether or not any pods are
        running on the node.

        :default: true
        '''
        result = self._values.get("force_update")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def instance_types(self) -> typing.Optional[typing.List[_InstanceType_f64915b9]]:
        '''The instance types to use for your node group.

        :default: t3.medium will be used according to the cloudformation document.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-instancetypes
        '''
        result = self._values.get("instance_types")
        return typing.cast(typing.Optional[typing.List[_InstanceType_f64915b9]], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The Kubernetes labels to be applied to the nodes in the node group when they are created.

        :default: - None
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def launch_template_spec(self) -> typing.Optional[LaunchTemplateSpec]:
        '''Launch template specification used for the nodegroup.

        :default: - no launch template

        :see: https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html
        '''
        result = self._values.get("launch_template_spec")
        return typing.cast(typing.Optional[LaunchTemplateSpec], result)

    @builtins.property
    def max_size(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of worker nodes that the managed node group can scale out to.

        Managed node groups can support up to 100 nodes by default.

        :default: - desiredSize
        '''
        result = self._values.get("max_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_unavailable(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of nodes unavailable at once during a version update.

        Nodes will be updated in parallel. The maximum number is 100.

        This value or ``maxUnavailablePercentage`` is required to have a value for custom update configurations to be applied.

        :default: 1

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-updateconfig.html#cfn-eks-nodegroup-updateconfig-maxunavailable
        '''
        result = self._values.get("max_unavailable")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_unavailable_percentage(self) -> typing.Optional[jsii.Number]:
        '''The maximum percentage of nodes unavailable during a version update.

        This percentage of nodes will be updated in parallel, up to 100 nodes at once.

        This value or ``maxUnavailable`` is required to have a value for custom update configurations to be applied.

        :default: undefined - node groups will update instances one at a time

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-updateconfig.html#cfn-eks-nodegroup-updateconfig-maxunavailablepercentage
        '''
        result = self._values.get("max_unavailable_percentage")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_size(self) -> typing.Optional[jsii.Number]:
        '''The minimum number of worker nodes that the managed node group can scale in to.

        This number must be greater than or equal to zero.

        :default: 1
        '''
        result = self._values.get("min_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def nodegroup_name(self) -> typing.Optional[builtins.str]:
        '''Name of the Nodegroup.

        :default: - resource ID
        '''
        result = self._values.get("nodegroup_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def node_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The IAM role to associate with your node group.

        The Amazon EKS worker node kubelet daemon
        makes calls to AWS APIs on your behalf. Worker nodes receive permissions for these API calls through
        an IAM instance profile and associated policies. Before you can launch worker nodes and register them
        into a cluster, you must create an IAM role for those worker nodes to use when they are launched.

        :default: - None. Auto-generated if not specified.
        '''
        result = self._values.get("node_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def release_version(self) -> typing.Optional[builtins.str]:
        '''The AMI version of the Amazon EKS-optimized AMI to use with your node group (for example, ``1.14.7-YYYYMMDD``).

        :default: - The latest available AMI version for the node group's current Kubernetes version is used.
        '''
        result = self._values.get("release_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def remote_access(self) -> typing.Optional["NodegroupRemoteAccess"]:
        '''The remote access (SSH) configuration to use with your node group.

        Disabled by default, however, if you
        specify an Amazon EC2 SSH key but do not specify a source security group when you create a managed node group,
        then port 22 on the worker nodes is opened to the internet (0.0.0.0/0)

        :default: - disabled
        '''
        result = self._values.get("remote_access")
        return typing.cast(typing.Optional["NodegroupRemoteAccess"], result)

    @builtins.property
    def subnets(self) -> typing.Optional[_SubnetSelection_e57d76df]:
        '''The subnets to use for the Auto Scaling group that is created for your node group.

        By specifying the
        SubnetSelection, the selected subnets will automatically apply required tags i.e.
        ``kubernetes.io/cluster/CLUSTER_NAME`` with a value of ``shared``, where ``CLUSTER_NAME`` is replaced with
        the name of your cluster.

        :default: - private subnets
        '''
        result = self._values.get("subnets")
        return typing.cast(typing.Optional[_SubnetSelection_e57d76df], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The metadata to apply to the node group to assist with categorization and organization.

        Each tag consists of
        a key and an optional value, both of which you define. Node group tags do not propagate to any other resources
        associated with the node group, such as the Amazon EC2 instances or subnets.

        :default: - None
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def taints(self) -> typing.Optional[typing.List["TaintSpec"]]:
        '''The Kubernetes taints to be applied to the nodes in the node group when they are created.

        :default: - None
        '''
        result = self._values.get("taints")
        return typing.cast(typing.Optional[typing.List["TaintSpec"]], result)

    @builtins.property
    def cluster(self) -> ICluster:
        '''Cluster resource.'''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(ICluster, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "NodegroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.NodegroupRemoteAccess",
    jsii_struct_bases=[],
    name_mapping={
        "ssh_key_name": "sshKeyName",
        "source_security_groups": "sourceSecurityGroups",
    },
)
class NodegroupRemoteAccess:
    def __init__(
        self,
        *,
        ssh_key_name: builtins.str,
        source_security_groups: typing.Optional[typing.Sequence[_ISecurityGroup_acf8a799]] = None,
    ) -> None:
        '''The remote access (SSH) configuration to use with your node group.

        :param ssh_key_name: The Amazon EC2 SSH key that provides access for SSH communication with the worker nodes in the managed node group.
        :param source_security_groups: The security groups that are allowed SSH access (port 22) to the worker nodes. If you specify an Amazon EC2 SSH key but do not specify a source security group when you create a managed node group, then port 22 on the worker nodes is opened to the internet (0.0.0.0/0). Default: - port 22 on the worker nodes is opened to the internet (0.0.0.0/0)

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-nodegroup-remoteaccess.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_eks as eks
            
            # security_group: ec2.SecurityGroup
            
            nodegroup_remote_access = eks.NodegroupRemoteAccess(
                ssh_key_name="sshKeyName",
            
                # the properties below are optional
                source_security_groups=[security_group]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d2ac0f8076733623c4989d2cd59ea2c4091215dc03c2749533886d1c5e3c6843)
            check_type(argname="argument ssh_key_name", value=ssh_key_name, expected_type=type_hints["ssh_key_name"])
            check_type(argname="argument source_security_groups", value=source_security_groups, expected_type=type_hints["source_security_groups"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "ssh_key_name": ssh_key_name,
        }
        if source_security_groups is not None:
            self._values["source_security_groups"] = source_security_groups

    @builtins.property
    def ssh_key_name(self) -> builtins.str:
        '''The Amazon EC2 SSH key that provides access for SSH communication with the worker nodes in the managed node group.'''
        result = self._values.get("ssh_key_name")
        assert result is not None, "Required property 'ssh_key_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def source_security_groups(
        self,
    ) -> typing.Optional[typing.List[_ISecurityGroup_acf8a799]]:
        '''The security groups that are allowed SSH access (port 22) to the worker nodes.

        If you specify an Amazon EC2 SSH
        key but do not specify a source security group when you create a managed node group, then port 22 on the worker
        nodes is opened to the internet (0.0.0.0/0).

        :default: - port 22 on the worker nodes is opened to the internet (0.0.0.0/0)
        '''
        result = self._values.get("source_security_groups")
        return typing.cast(typing.Optional[typing.List[_ISecurityGroup_acf8a799]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "NodegroupRemoteAccess(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class OpenIdConnectProvider(
    _OpenIdConnectProvider_5cb7bc9f,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.OpenIdConnectProvider",
):
    '''IAM OIDC identity providers are entities in IAM that describe an external identity provider (IdP) service that supports the OpenID Connect (OIDC) standard, such as Google or Salesforce.

    You use an IAM OIDC identity provider
    when you want to establish trust between an OIDC-compatible IdP and your AWS
    account.

    This implementation has default values for thumbprints and clientIds props
    that will be compatible with the eks cluster

    :see: https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc.html
    :resource: AWS::CloudFormation::CustomResource
    :exampleMetadata: infused

    Example::

        # or create a new one using an existing issuer url
        # issuer_url: str
        # you can import an existing provider
        provider = eks.OpenIdConnectProvider.from_open_id_connect_provider_arn(self, "Provider", "arn:aws:iam::123456:oidc-provider/oidc.eks.eu-west-1.amazonaws.com/id/AB123456ABC")
        provider2 = eks.OpenIdConnectProvider(self, "Provider",
            url=issuer_url
        )
        
        cluster = eks.Cluster.from_cluster_attributes(self, "MyCluster",
            cluster_name="Cluster",
            open_id_connect_provider=provider,
            kubectl_role_arn="arn:aws:iam::123456:role/service-role/k8sservicerole"
        )
        
        service_account = cluster.add_service_account("MyServiceAccount")
        
        bucket = s3.Bucket(self, "Bucket")
        bucket.grant_read_write(service_account)
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        url: builtins.str,
    ) -> None:
        '''Defines an OpenID Connect provider.

        :param scope: The definition scope.
        :param id: Construct ID.
        :param url: The URL of the identity provider. The URL must begin with https:// and should correspond to the iss claim in the provider's OpenID Connect ID tokens. Per the OIDC standard, path components are allowed but query parameters are not. Typically the URL consists of only a hostname, like https://server.example.org or https://example.com. You can find your OIDC Issuer URL by: aws eks describe-cluster --name %cluster_name% --query "cluster.identity.oidc.issuer" --output text
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__71eee7d03a26aea18beb68698a39be2bf4e6d9b2f46d16ba179993326ea5a086)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = OpenIdConnectProviderProps(url=url)

        jsii.create(self.__class__, self, [scope, id, props])


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.OpenIdConnectProviderProps",
    jsii_struct_bases=[],
    name_mapping={"url": "url"},
)
class OpenIdConnectProviderProps:
    def __init__(self, *, url: builtins.str) -> None:
        '''Initialization properties for ``OpenIdConnectProvider``.

        :param url: The URL of the identity provider. The URL must begin with https:// and should correspond to the iss claim in the provider's OpenID Connect ID tokens. Per the OIDC standard, path components are allowed but query parameters are not. Typically the URL consists of only a hostname, like https://server.example.org or https://example.com. You can find your OIDC Issuer URL by: aws eks describe-cluster --name %cluster_name% --query "cluster.identity.oidc.issuer" --output text

        :exampleMetadata: infused

        Example::

            # or create a new one using an existing issuer url
            # issuer_url: str
            # you can import an existing provider
            provider = eks.OpenIdConnectProvider.from_open_id_connect_provider_arn(self, "Provider", "arn:aws:iam::123456:oidc-provider/oidc.eks.eu-west-1.amazonaws.com/id/AB123456ABC")
            provider2 = eks.OpenIdConnectProvider(self, "Provider",
                url=issuer_url
            )
            
            cluster = eks.Cluster.from_cluster_attributes(self, "MyCluster",
                cluster_name="Cluster",
                open_id_connect_provider=provider,
                kubectl_role_arn="arn:aws:iam::123456:role/service-role/k8sservicerole"
            )
            
            service_account = cluster.add_service_account("MyServiceAccount")
            
            bucket = s3.Bucket(self, "Bucket")
            bucket.grant_read_write(service_account)
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c02764139ca6306efb78e2db6695149f8ddc6b3e8adb63a11131864ce9246c30)
            check_type(argname="argument url", value=url, expected_type=type_hints["url"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "url": url,
        }

    @builtins.property
    def url(self) -> builtins.str:
        '''The URL of the identity provider.

        The URL must begin with https:// and
        should correspond to the iss claim in the provider's OpenID Connect ID
        tokens. Per the OIDC standard, path components are allowed but query
        parameters are not. Typically the URL consists of only a hostname, like
        https://server.example.org or https://example.com.

        You can find your OIDC Issuer URL by:
        aws eks describe-cluster --name %cluster_name% --query "cluster.identity.oidc.issuer" --output text
        '''
        result = self._values.get("url")
        assert result is not None, "Required property 'url' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "OpenIdConnectProviderProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.PatchType")
class PatchType(enum.Enum):
    '''Values for ``kubectl patch`` --type argument.'''

    JSON = "JSON"
    '''JSON Patch, RFC 6902.'''
    MERGE = "MERGE"
    '''JSON Merge patch.'''
    STRATEGIC = "STRATEGIC"
    '''Strategic merge patch.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.RemoteNodeNetwork",
    jsii_struct_bases=[],
    name_mapping={"cidrs": "cidrs"},
)
class RemoteNodeNetwork:
    def __init__(self, *, cidrs: typing.Sequence[builtins.str]) -> None:
        '''Network configuration of nodes run on-premises with EKS Hybrid Nodes.

        :param cidrs: Specifies the list of remote node CIDRs.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            remote_node_network = eks.RemoteNodeNetwork(
                cidrs=["cidrs"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__600789f5d1adc105e950fc1e01201ea975b89bb797b63227b757a633425a0f09)
            check_type(argname="argument cidrs", value=cidrs, expected_type=type_hints["cidrs"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cidrs": cidrs,
        }

    @builtins.property
    def cidrs(self) -> typing.List[builtins.str]:
        '''Specifies the list of remote node CIDRs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-remotenodenetwork.html#cfn-eks-cluster-remotenodenetwork-cidrs
        '''
        result = self._values.get("cidrs")
        assert result is not None, "Required property 'cidrs' is missing"
        return typing.cast(typing.List[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RemoteNodeNetwork(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.RemotePodNetwork",
    jsii_struct_bases=[],
    name_mapping={"cidrs": "cidrs"},
)
class RemotePodNetwork:
    def __init__(self, *, cidrs: typing.Sequence[builtins.str]) -> None:
        '''Network configuration of pods run on-premises with EKS Hybrid Nodes.

        :param cidrs: Specifies the list of remote pod CIDRs.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            remote_pod_network = eks.RemotePodNetwork(
                cidrs=["cidrs"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f9878a6e6680b6c2c6cb0db908c65c1de65fe68965909386c87176ba98e30705)
            check_type(argname="argument cidrs", value=cidrs, expected_type=type_hints["cidrs"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cidrs": cidrs,
        }

    @builtins.property
    def cidrs(self) -> typing.List[builtins.str]:
        '''Specifies the list of remote pod CIDRs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-eks-cluster-remotepodnetwork.html#cfn-eks-cluster-remotepodnetwork-cidrs
        '''
        result = self._values.get("cidrs")
        assert result is not None, "Required property 'cidrs' is missing"
        return typing.cast(typing.List[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RemotePodNetwork(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.Selector",
    jsii_struct_bases=[],
    name_mapping={"namespace": "namespace", "labels": "labels"},
)
class Selector:
    def __init__(
        self,
        *,
        namespace: builtins.str,
        labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Fargate profile selector.

        :param namespace: The Kubernetes namespace that the selector should match. You must specify a namespace for a selector. The selector only matches pods that are created in this namespace, but you can create multiple selectors to target multiple namespaces.
        :param labels: The Kubernetes labels that the selector should match. A pod must contain all of the labels that are specified in the selector for it to be considered a match. Default: - all pods within the namespace will be selected.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            selector = eks.Selector(
                namespace="namespace",
            
                # the properties below are optional
                labels={
                    "labels_key": "labels"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__594b3f5a610588bf33bb1a98e98b19b5ddfb0609f59e93022c2cec8d2a17f411)
            check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
            check_type(argname="argument labels", value=labels, expected_type=type_hints["labels"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "namespace": namespace,
        }
        if labels is not None:
            self._values["labels"] = labels

    @builtins.property
    def namespace(self) -> builtins.str:
        '''The Kubernetes namespace that the selector should match.

        You must specify a namespace for a selector. The selector only matches pods
        that are created in this namespace, but you can create multiple selectors
        to target multiple namespaces.
        '''
        result = self._values.get("namespace")
        assert result is not None, "Required property 'namespace' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The Kubernetes labels that the selector should match.

        A pod must contain
        all of the labels that are specified in the selector for it to be
        considered a match.

        :default: - all pods within the namespace will be selected.
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Selector(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IPrincipal_539bb2fd)
class ServiceAccount(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.ServiceAccount",
):
    '''Service Account.

    :exampleMetadata: infused

    Example::

        # or create a new one using an existing issuer url
        # issuer_url: str
        # you can import an existing provider
        provider = eks.OpenIdConnectProvider.from_open_id_connect_provider_arn(self, "Provider", "arn:aws:iam::123456:oidc-provider/oidc.eks.eu-west-1.amazonaws.com/id/AB123456ABC")
        provider2 = eks.OpenIdConnectProvider(self, "Provider",
            url=issuer_url
        )
        
        cluster = eks.Cluster.from_cluster_attributes(self, "MyCluster",
            cluster_name="Cluster",
            open_id_connect_provider=provider,
            kubectl_role_arn="arn:aws:iam::123456:role/service-role/k8sservicerole"
        )
        
        service_account = cluster.add_service_account("MyServiceAccount")
        
        bucket = s3.Bucket(self, "Bucket")
        bucket.grant_read_write(service_account)
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster: ICluster,
        annotations: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        identity_type: typing.Optional[IdentityType] = None,
        labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        name: typing.Optional[builtins.str] = None,
        namespace: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster: The cluster to apply the patch to.
        :param annotations: Additional annotations of the service account. Default: - no additional annotations
        :param identity_type: The identity type to use for the service account. Default: IdentityType.IRSA
        :param labels: Additional labels of the service account. Default: - no additional labels
        :param name: The name of the service account. The name of a ServiceAccount object must be a valid DNS subdomain name. https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/ Default: - If no name is given, it will use the id of the resource.
        :param namespace: The namespace of the service account. All namespace names must be valid RFC 1123 DNS labels. https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/#namespaces-and-dns Default: "default"
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c59483a03e00366cbc5eed954b787cea3e7b09f1579c5c9badd84776ccd54cc1)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ServiceAccountProps(
            cluster=cluster,
            annotations=annotations,
            identity_type=identity_type,
            labels=labels,
            name=name,
            namespace=namespace,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addToPrincipalPolicy")
    def add_to_principal_policy(
        self,
        statement: _PolicyStatement_0fe33853,
    ) -> _AddToPrincipalPolicyResult_946c9561:
        '''Add to the policy of this principal.

        :param statement: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__644b8e999f78647ce72c2476e43febe6c5bec18a337cfb7b041c87773cdc07cf)
            check_type(argname="argument statement", value=statement, expected_type=type_hints["statement"])
        return typing.cast(_AddToPrincipalPolicyResult_946c9561, jsii.invoke(self, "addToPrincipalPolicy", [statement]))

    @builtins.property
    @jsii.member(jsii_name="assumeRoleAction")
    def assume_role_action(self) -> builtins.str:
        '''When this Principal is used in an AssumeRole policy, the action to use.'''
        return typing.cast(builtins.str, jsii.get(self, "assumeRoleAction"))

    @builtins.property
    @jsii.member(jsii_name="grantPrincipal")
    def grant_principal(self) -> _IPrincipal_539bb2fd:
        '''The principal to grant permissions to.'''
        return typing.cast(_IPrincipal_539bb2fd, jsii.get(self, "grantPrincipal"))

    @builtins.property
    @jsii.member(jsii_name="policyFragment")
    def policy_fragment(self) -> _PrincipalPolicyFragment_6a855d11:
        '''Return the policy fragment that identifies this principal in a Policy.'''
        return typing.cast(_PrincipalPolicyFragment_6a855d11, jsii.get(self, "policyFragment"))

    @builtins.property
    @jsii.member(jsii_name="role")
    def role(self) -> _IRole_235f5d8e:
        '''The role which is linked to the service account.'''
        return typing.cast(_IRole_235f5d8e, jsii.get(self, "role"))

    @builtins.property
    @jsii.member(jsii_name="serviceAccountName")
    def service_account_name(self) -> builtins.str:
        '''The name of the service account.'''
        return typing.cast(builtins.str, jsii.get(self, "serviceAccountName"))

    @builtins.property
    @jsii.member(jsii_name="serviceAccountNamespace")
    def service_account_namespace(self) -> builtins.str:
        '''The namespace where the service account is located in.'''
        return typing.cast(builtins.str, jsii.get(self, "serviceAccountNamespace"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.ServiceAccountOptions",
    jsii_struct_bases=[],
    name_mapping={
        "annotations": "annotations",
        "identity_type": "identityType",
        "labels": "labels",
        "name": "name",
        "namespace": "namespace",
    },
)
class ServiceAccountOptions:
    def __init__(
        self,
        *,
        annotations: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        identity_type: typing.Optional[IdentityType] = None,
        labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        name: typing.Optional[builtins.str] = None,
        namespace: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options for ``ServiceAccount``.

        :param annotations: Additional annotations of the service account. Default: - no additional annotations
        :param identity_type: The identity type to use for the service account. Default: IdentityType.IRSA
        :param labels: Additional labels of the service account. Default: - no additional labels
        :param name: The name of the service account. The name of a ServiceAccount object must be a valid DNS subdomain name. https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/ Default: - If no name is given, it will use the id of the resource.
        :param namespace: The namespace of the service account. All namespace names must be valid RFC 1123 DNS labels. https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/#namespaces-and-dns Default: "default"

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            # add service account with annotations and labels
            service_account = cluster.add_service_account("MyServiceAccount",
                annotations={
                    "eks.amazonaws.com/sts-regional-endpoints": "false"
                },
                labels={
                    "some-label": "with-some-value"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c16813f7f34b0f551b6879a204a04016f3eb45d120b546a7afd47fee08551d86)
            check_type(argname="argument annotations", value=annotations, expected_type=type_hints["annotations"])
            check_type(argname="argument identity_type", value=identity_type, expected_type=type_hints["identity_type"])
            check_type(argname="argument labels", value=labels, expected_type=type_hints["labels"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if annotations is not None:
            self._values["annotations"] = annotations
        if identity_type is not None:
            self._values["identity_type"] = identity_type
        if labels is not None:
            self._values["labels"] = labels
        if name is not None:
            self._values["name"] = name
        if namespace is not None:
            self._values["namespace"] = namespace

    @builtins.property
    def annotations(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Additional annotations of the service account.

        :default: - no additional annotations
        '''
        result = self._values.get("annotations")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def identity_type(self) -> typing.Optional[IdentityType]:
        '''The identity type to use for the service account.

        :default: IdentityType.IRSA
        '''
        result = self._values.get("identity_type")
        return typing.cast(typing.Optional[IdentityType], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Additional labels of the service account.

        :default: - no additional labels
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of the service account.

        The name of a ServiceAccount object must be a valid DNS subdomain name.
        https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/

        :default: - If no name is given, it will use the id of the resource.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def namespace(self) -> typing.Optional[builtins.str]:
        '''The namespace of the service account.

        All namespace names must be valid RFC 1123 DNS labels.
        https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/#namespaces-and-dns

        :default: "default"
        '''
        result = self._values.get("namespace")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ServiceAccountOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.ServiceAccountProps",
    jsii_struct_bases=[ServiceAccountOptions],
    name_mapping={
        "annotations": "annotations",
        "identity_type": "identityType",
        "labels": "labels",
        "name": "name",
        "namespace": "namespace",
        "cluster": "cluster",
    },
)
class ServiceAccountProps(ServiceAccountOptions):
    def __init__(
        self,
        *,
        annotations: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        identity_type: typing.Optional[IdentityType] = None,
        labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        name: typing.Optional[builtins.str] = None,
        namespace: typing.Optional[builtins.str] = None,
        cluster: ICluster,
    ) -> None:
        '''Properties for defining service accounts.

        :param annotations: Additional annotations of the service account. Default: - no additional annotations
        :param identity_type: The identity type to use for the service account. Default: IdentityType.IRSA
        :param labels: Additional labels of the service account. Default: - no additional labels
        :param name: The name of the service account. The name of a ServiceAccount object must be a valid DNS subdomain name. https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/ Default: - If no name is given, it will use the id of the resource.
        :param namespace: The namespace of the service account. All namespace names must be valid RFC 1123 DNS labels. https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/#namespaces-and-dns Default: "default"
        :param cluster: The cluster to apply the patch to.

        :exampleMetadata: infused

        Example::

            # cluster: eks.Cluster
            
            
            eks.ServiceAccount(self, "ServiceAccount",
                cluster=cluster,
                name="test-sa",
                namespace="default",
                identity_type=eks.IdentityType.POD_IDENTITY
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f409e147cd54788bf9d9542d66a6b0445436e408deb553426c2dca2bd73b6d76)
            check_type(argname="argument annotations", value=annotations, expected_type=type_hints["annotations"])
            check_type(argname="argument identity_type", value=identity_type, expected_type=type_hints["identity_type"])
            check_type(argname="argument labels", value=labels, expected_type=type_hints["labels"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
            check_type(argname="argument cluster", value=cluster, expected_type=type_hints["cluster"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cluster": cluster,
        }
        if annotations is not None:
            self._values["annotations"] = annotations
        if identity_type is not None:
            self._values["identity_type"] = identity_type
        if labels is not None:
            self._values["labels"] = labels
        if name is not None:
            self._values["name"] = name
        if namespace is not None:
            self._values["namespace"] = namespace

    @builtins.property
    def annotations(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Additional annotations of the service account.

        :default: - no additional annotations
        '''
        result = self._values.get("annotations")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def identity_type(self) -> typing.Optional[IdentityType]:
        '''The identity type to use for the service account.

        :default: IdentityType.IRSA
        '''
        result = self._values.get("identity_type")
        return typing.cast(typing.Optional[IdentityType], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Additional labels of the service account.

        :default: - no additional labels
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of the service account.

        The name of a ServiceAccount object must be a valid DNS subdomain name.
        https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/

        :default: - If no name is given, it will use the id of the resource.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def namespace(self) -> typing.Optional[builtins.str]:
        '''The namespace of the service account.

        All namespace names must be valid RFC 1123 DNS labels.
        https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/#namespaces-and-dns

        :default: "default"
        '''
        result = self._values.get("namespace")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cluster(self) -> ICluster:
        '''The cluster to apply the patch to.'''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(ICluster, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ServiceAccountProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.ServiceLoadBalancerAddressOptions",
    jsii_struct_bases=[],
    name_mapping={"namespace": "namespace", "timeout": "timeout"},
)
class ServiceLoadBalancerAddressOptions:
    def __init__(
        self,
        *,
        namespace: typing.Optional[builtins.str] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''Options for fetching a ServiceLoadBalancerAddress.

        :param namespace: The namespace the service belongs to. Default: 'default'
        :param timeout: Timeout for waiting on the load balancer address. Default: Duration.minutes(5)

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_eks as eks
            
            service_load_balancer_address_options = eks.ServiceLoadBalancerAddressOptions(
                namespace="namespace",
                timeout=cdk.Duration.minutes(30)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__928497e107c4d882d88858dab4d9814ffc4c4d8d70a5381c6f68a142f1984c07)
            check_type(argname="argument namespace", value=namespace, expected_type=type_hints["namespace"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if namespace is not None:
            self._values["namespace"] = namespace
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def namespace(self) -> typing.Optional[builtins.str]:
        '''The namespace the service belongs to.

        :default: 'default'
        '''
        result = self._values.get("namespace")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Timeout for waiting on the load balancer address.

        :default: Duration.minutes(5)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ServiceLoadBalancerAddressOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_eks.TaintEffect")
class TaintEffect(enum.Enum):
    '''Effect types of kubernetes node taint.

    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        cluster.add_nodegroup_capacity("custom-node-group",
            instance_types=[ec2.InstanceType("m5.large")],
            taints=[eks.TaintSpec(
                effect=eks.TaintEffect.NO_SCHEDULE,
                key="foo",
                value="bar"
            )
            ]
        )
    '''

    NO_SCHEDULE = "NO_SCHEDULE"
    '''NoSchedule.'''
    PREFER_NO_SCHEDULE = "PREFER_NO_SCHEDULE"
    '''PreferNoSchedule.'''
    NO_EXECUTE = "NO_EXECUTE"
    '''NoExecute.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_eks.TaintSpec",
    jsii_struct_bases=[],
    name_mapping={"effect": "effect", "key": "key", "value": "value"},
)
class TaintSpec:
    def __init__(
        self,
        *,
        effect: typing.Optional[TaintEffect] = None,
        key: typing.Optional[builtins.str] = None,
        value: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Taint interface.

        :param effect: Effect type. Default: - None
        :param key: Taint key. Default: - None
        :param value: Taint value. Default: - None

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_eks as eks
            
            taint_spec = eks.TaintSpec(
                effect=eks.TaintEffect.NO_SCHEDULE,
                key="key",
                value="value"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__be119d20277d81d5cacb542dcf0ff7927e8c934305e8be443e95ce3218b6e24f)
            check_type(argname="argument effect", value=effect, expected_type=type_hints["effect"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if effect is not None:
            self._values["effect"] = effect
        if key is not None:
            self._values["key"] = key
        if value is not None:
            self._values["value"] = value

    @builtins.property
    def effect(self) -> typing.Optional[TaintEffect]:
        '''Effect type.

        :default: - None
        '''
        result = self._values.get("effect")
        return typing.cast(typing.Optional[TaintEffect], result)

    @builtins.property
    def key(self) -> typing.Optional[builtins.str]:
        '''Taint key.

        :default: - None
        '''
        result = self._values.get("key")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def value(self) -> typing.Optional[builtins.str]:
        '''Taint value.

        :default: - None
        '''
        result = self._values.get("value")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "TaintSpec(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IAccessEntry)
class AccessEntry(
    _Resource_45bc6135,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.AccessEntry",
):
    '''Represents an access entry in an Amazon EKS cluster.

    An access entry defines the permissions and scope for a user or role to access an Amazon EKS cluster.

    :implements: IAccessEntry
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_eks as eks
        
        # access_policy: eks.AccessPolicy
        # cluster: eks.Cluster
        
        access_entry = eks.AccessEntry(self, "MyAccessEntry",
            access_policies=[access_policy],
            cluster=cluster,
            principal="principal",
        
            # the properties below are optional
            access_entry_name="accessEntryName",
            access_entry_type=eks.AccessEntryType.STANDARD
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        access_policies: typing.Sequence[IAccessPolicy],
        cluster: ICluster,
        principal: builtins.str,
        access_entry_name: typing.Optional[builtins.str] = None,
        access_entry_type: typing.Optional[AccessEntryType] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param access_policies: The access policies that define the permissions and scope for the access entry.
        :param cluster: The Amazon EKS cluster to which the access entry applies.
        :param principal: The Amazon Resource Name (ARN) of the principal (user or role) to associate the access entry with.
        :param access_entry_name: The name of the AccessEntry. Default: - No access entry name is provided
        :param access_entry_type: The type of the AccessEntry. Default: STANDARD
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76acfe0dd4f79a87279c3d9cbf3bd8c7327733233aec66908901483da7f17d5b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AccessEntryProps(
            access_policies=access_policies,
            cluster=cluster,
            principal=principal,
            access_entry_name=access_entry_name,
            access_entry_type=access_entry_type,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="fromAccessEntryAttributes")
    @builtins.classmethod
    def from_access_entry_attributes(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        access_entry_arn: builtins.str,
        access_entry_name: builtins.str,
    ) -> IAccessEntry:
        '''Imports an ``AccessEntry`` from its attributes.

        :param scope: - The parent construct.
        :param id: - The ID of the imported construct.
        :param access_entry_arn: The Amazon Resource Name (ARN) of the access entry.
        :param access_entry_name: The name of the access entry.

        :return: The imported access entry.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f179e9982369f73f3bb7a13ff87f073fda0da2cd11932479d54f6d69d8849141)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        attrs = AccessEntryAttributes(
            access_entry_arn=access_entry_arn, access_entry_name=access_entry_name
        )

        return typing.cast(IAccessEntry, jsii.sinvoke(cls, "fromAccessEntryAttributes", [scope, id, attrs]))

    @jsii.member(jsii_name="addAccessPolicies")
    def add_access_policies(
        self,
        new_access_policies: typing.Sequence[IAccessPolicy],
    ) -> None:
        '''Add the access policies for this entry.

        :param new_access_policies: - The new access policies to add.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e1007432e51c3db7b596578b73c0068b372fc6af638d2adb2faa12db70799372)
            check_type(argname="argument new_access_policies", value=new_access_policies, expected_type=type_hints["new_access_policies"])
        return typing.cast(None, jsii.invoke(self, "addAccessPolicies", [new_access_policies]))

    @builtins.property
    @jsii.member(jsii_name="accessEntryArn")
    def access_entry_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the access entry.'''
        return typing.cast(builtins.str, jsii.get(self, "accessEntryArn"))

    @builtins.property
    @jsii.member(jsii_name="accessEntryName")
    def access_entry_name(self) -> builtins.str:
        '''The name of the access entry.'''
        return typing.cast(builtins.str, jsii.get(self, "accessEntryName"))


@jsii.implements(IAccessPolicy)
class AccessPolicy(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.AccessPolicy",
):
    '''Represents an Amazon EKS Access Policy that implements the IAccessPolicy interface.

    :implements: IAccessPolicy
    :exampleMetadata: infused

    Example::

        # AmazonEKSClusterAdminPolicy with `cluster` scope
        eks.AccessPolicy.from_access_policy_name("AmazonEKSClusterAdminPolicy",
            access_scope_type=eks.AccessScopeType.CLUSTER
        )
        # AmazonEKSAdminPolicy with `namespace` scope
        eks.AccessPolicy.from_access_policy_name("AmazonEKSAdminPolicy",
            access_scope_type=eks.AccessScopeType.NAMESPACE,
            namespaces=["foo", "bar"]
        )
    '''

    def __init__(
        self,
        *,
        access_scope: typing.Union[AccessScope, typing.Dict[builtins.str, typing.Any]],
        policy: AccessPolicyArn,
    ) -> None:
        '''Constructs a new instance of the AccessPolicy class.

        :param access_scope: The scope of the access policy, which determines the level of access granted.
        :param policy: The access policy itself, which defines the specific permissions.
        '''
        props = AccessPolicyProps(access_scope=access_scope, policy=policy)

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="fromAccessPolicyName")
    @builtins.classmethod
    def from_access_policy_name(
        cls,
        policy_name: builtins.str,
        *,
        access_scope_type: AccessScopeType,
        namespaces: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> IAccessPolicy:
        '''Import AccessPolicy by name.

        :param policy_name: -
        :param access_scope_type: The scope of the access policy. This determines the level of access granted by the policy.
        :param namespaces: An optional array of Kubernetes namespaces to which the access policy applies. Default: - no specific namespaces for this scope
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a928f26a921cd1d01ec556e4421fe3bf4a1ac17a9b598a554215bfae5af842aa)
            check_type(argname="argument policy_name", value=policy_name, expected_type=type_hints["policy_name"])
        options = AccessPolicyNameOptions(
            access_scope_type=access_scope_type, namespaces=namespaces
        )

        return typing.cast(IAccessPolicy, jsii.sinvoke(cls, "fromAccessPolicyName", [policy_name, options]))

    @builtins.property
    @jsii.member(jsii_name="accessScope")
    def access_scope(self) -> AccessScope:
        '''The scope of the access policy, which determines the level of access granted.'''
        return typing.cast(AccessScope, jsii.get(self, "accessScope"))

    @builtins.property
    @jsii.member(jsii_name="policy")
    def policy(self) -> builtins.str:
        '''The access policy itself, which defines the specific permissions.'''
        return typing.cast(builtins.str, jsii.get(self, "policy"))


@jsii.implements(IAddon)
class Addon(
    _Resource_45bc6135,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.Addon",
):
    '''Represents an Amazon EKS Add-On.

    :exampleMetadata: infused

    Example::

        # cluster: eks.Cluster
        
        
        eks.Addon(self, "Addon",
            cluster=cluster,
            addon_name="aws-guardduty-agent",
            addon_version="v1.6.1",
            # whether to preserve the add-on software on your cluster but Amazon EKS stops managing any settings for the add-on.
            preserve_on_delete=False
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        addon_name: builtins.str,
        cluster: ICluster,
        addon_version: typing.Optional[builtins.str] = None,
        preserve_on_delete: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Creates a new Amazon EKS Add-On.

        :param scope: The parent construct.
        :param id: The construct ID.
        :param addon_name: Name of the Add-On.
        :param cluster: The EKS cluster the Add-On is associated with.
        :param addon_version: Version of the Add-On. You can check all available versions with describe-addon-versions. For example, this lists all available versions for the ``eks-pod-identity-agent`` addon: $ aws eks describe-addon-versions --addon-name eks-pod-identity-agent --query 'addons[*].addonVersions[*].addonVersion' Default: the latest version.
        :param preserve_on_delete: Specifying this option preserves the add-on software on your cluster but Amazon EKS stops managing any settings for the add-on. If an IAM account is associated with the add-on, it isn't removed. Default: true
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a8342124e215d4789acf852df764143c4809251dbcaa86f6b4a11860e46f830d)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AddonProps(
            addon_name=addon_name,
            cluster=cluster,
            addon_version=addon_version,
            preserve_on_delete=preserve_on_delete,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="fromAddonArn")
    @builtins.classmethod
    def from_addon_arn(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        addon_arn: builtins.str,
    ) -> IAddon:
        '''Creates an ``IAddon`` from an existing addon ARN.

        :param scope: - The parent construct.
        :param id: - The ID of the construct.
        :param addon_arn: - The ARN of the addon.

        :return: An ``IAddon`` implementation.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d4c2296edfe5b5c8603ac11e589a95341ba550799d11a043d66684ed98365879)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument addon_arn", value=addon_arn, expected_type=type_hints["addon_arn"])
        return typing.cast(IAddon, jsii.sinvoke(cls, "fromAddonArn", [scope, id, addon_arn]))

    @jsii.member(jsii_name="fromAddonAttributes")
    @builtins.classmethod
    def from_addon_attributes(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        addon_name: builtins.str,
        cluster_name: builtins.str,
    ) -> IAddon:
        '''Creates an ``IAddon`` instance from the given addon attributes.

        :param scope: - The parent construct.
        :param id: - The construct ID.
        :param addon_name: The name of the addon.
        :param cluster_name: The name of the Amazon EKS cluster the addon is associated with.

        :return: An ``IAddon`` instance.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a990f16d21217e79d80780ad054930c23d18b30b674b483763b39dcdb7fdac7)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        attrs = AddonAttributes(addon_name=addon_name, cluster_name=cluster_name)

        return typing.cast(IAddon, jsii.sinvoke(cls, "fromAddonAttributes", [scope, id, attrs]))

    @builtins.property
    @jsii.member(jsii_name="addonArn")
    def addon_arn(self) -> builtins.str:
        '''Arn of the addon.'''
        return typing.cast(builtins.str, jsii.get(self, "addonArn"))

    @builtins.property
    @jsii.member(jsii_name="addonName")
    def addon_name(self) -> builtins.str:
        '''Name of the addon.'''
        return typing.cast(builtins.str, jsii.get(self, "addonName"))


@jsii.implements(ICluster)
class Cluster(
    _Resource_45bc6135,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_eks.Cluster",
):
    '''A Cluster represents a managed Kubernetes Service (EKS).

    This is a fully managed cluster of API Servers (control-plane)
    The user is still required to create the worker nodes.

    :exampleMetadata: infused

    Example::

        from aws_cdk.lambda_layer_kubectl_v32 import KubectlV32Layer
        
        # or
        # vpc: ec2.Vpc
        
        
        eks.Cluster(self, "MyCluster",
            kubectl_memory=Size.gibibytes(4),
            version=eks.KubernetesVersion.V1_32,
            kubectl_layer=KubectlV32Layer(self, "kubectl")
        )
        eks.Cluster.from_cluster_attributes(self, "MyCluster",
            kubectl_memory=Size.gibibytes(4),
            vpc=vpc,
            cluster_name="cluster-name"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        bootstrap_cluster_creator_admin_permissions: typing.Optional[builtins.bool] = None,
        default_capacity: typing.Optional[jsii.Number] = None,
        default_capacity_instance: typing.Optional[_InstanceType_f64915b9] = None,
        default_capacity_type: typing.Optional[DefaultCapacityType] = None,
        kubectl_lambda_role: typing.Optional[_IRole_235f5d8e] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        kubectl_layer: _ILayerVersion_5ac127c8,
        alb_controller: typing.Optional[typing.Union[AlbControllerOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        authentication_mode: typing.Optional[AuthenticationMode] = None,
        awscli_layer: typing.Optional[_ILayerVersion_5ac127c8] = None,
        cluster_handler_environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        cluster_handler_security_group: typing.Optional[_ISecurityGroup_acf8a799] = None,
        cluster_logging: typing.Optional[typing.Sequence[ClusterLoggingTypes]] = None,
        core_dns_compute_type: typing.Optional[CoreDnsComputeType] = None,
        endpoint_access: typing.Optional[EndpointAccess] = None,
        ip_family: typing.Optional[IpFamily] = None,
        kubectl_environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        kubectl_memory: typing.Optional[_Size_7b441c34] = None,
        masters_role: typing.Optional[_IRole_235f5d8e] = None,
        on_event_layer: typing.Optional[_ILayerVersion_5ac127c8] = None,
        output_masters_role_arn: typing.Optional[builtins.bool] = None,
        place_cluster_handler_in_vpc: typing.Optional[builtins.bool] = None,
        prune: typing.Optional[builtins.bool] = None,
        remote_node_networks: typing.Optional[typing.Sequence[typing.Union[RemoteNodeNetwork, typing.Dict[builtins.str, typing.Any]]]] = None,
        remote_pod_networks: typing.Optional[typing.Sequence[typing.Union[RemotePodNetwork, typing.Dict[builtins.str, typing.Any]]]] = None,
        secrets_encryption_key: typing.Optional[_IKey_5f11635f] = None,
        service_ipv4_cidr: typing.Optional[builtins.str] = None,
        version: KubernetesVersion,
        cluster_name: typing.Optional[builtins.str] = None,
        output_cluster_name: typing.Optional[builtins.bool] = None,
        output_config_command: typing.Optional[builtins.bool] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
        security_group: typing.Optional[_ISecurityGroup_acf8a799] = None,
        vpc: typing.Optional[_IVpc_f30d5663] = None,
        vpc_subnets: typing.Optional[typing.Sequence[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Initiates an EKS Cluster with the supplied arguments.

        :param scope: a Construct, most likely a cdk.Stack created.
        :param id: the id of the Construct to create.
        :param bootstrap_cluster_creator_admin_permissions: Whether or not IAM principal of the cluster creator was set as a cluster admin access entry during cluster creation time. Changing this value after the cluster has been created will result in the cluster being replaced. Default: true
        :param default_capacity: Number of instances to allocate as an initial capacity for this cluster. Instance type can be configured through ``defaultCapacityInstanceType``, which defaults to ``m5.large``. Use ``cluster.addAutoScalingGroupCapacity`` to add additional customized capacity. Set this to ``0`` is you wish to avoid the initial capacity allocation. Default: 2
        :param default_capacity_instance: The instance type to use for the default capacity. This will only be taken into account if ``defaultCapacity`` is > 0. Default: m5.large
        :param default_capacity_type: The default capacity type for the cluster. Default: NODEGROUP
        :param kubectl_lambda_role: The IAM role to pass to the Kubectl Lambda Handler. Default: - Default Lambda IAM Execution Role
        :param tags: The tags assigned to the EKS cluster. Default: - none
        :param kubectl_layer: An AWS Lambda Layer which includes ``kubectl`` and Helm. This layer is used by the kubectl handler to apply manifests and install helm charts. You must pick an appropriate releases of one of the ``@aws-cdk/layer-kubectl-vXX`` packages, that works with the version of Kubernetes you have chosen. The handler expects the layer to include the following executables:: /opt/helm/helm /opt/kubectl/kubectl
        :param alb_controller: Install the AWS Load Balancer Controller onto the cluster. Default: - The controller is not installed.
        :param authentication_mode: The desired authentication mode for the cluster. Default: AuthenticationMode.CONFIG_MAP
        :param awscli_layer: An AWS Lambda layer that contains the ``aws`` CLI. The handler expects the layer to include the following executables:: /opt/awscli/aws Default: - a default layer with the AWS CLI 1.x
        :param cluster_handler_environment: Custom environment variables when interacting with the EKS endpoint to manage the cluster lifecycle. Default: - No environment variables.
        :param cluster_handler_security_group: A security group to associate with the Cluster Handler's Lambdas. The Cluster Handler's Lambdas are responsible for calling AWS's EKS API. Requires ``placeClusterHandlerInVpc`` to be set to true. Default: - No security group.
        :param cluster_logging: The cluster log types which you want to enable. Default: - none
        :param core_dns_compute_type: Controls the "eks.amazonaws.com/compute-type" annotation in the CoreDNS configuration on your cluster to determine which compute type to use for CoreDNS. Default: CoreDnsComputeType.EC2 (for ``FargateCluster`` the default is FARGATE)
        :param endpoint_access: Configure access to the Kubernetes API server endpoint.. Default: EndpointAccess.PUBLIC_AND_PRIVATE
        :param ip_family: Specify which IP family is used to assign Kubernetes pod and service IP addresses. Default: - IpFamily.IP_V4
        :param kubectl_environment: Environment variables for the kubectl execution. Only relevant for kubectl enabled clusters. Default: - No environment variables.
        :param kubectl_memory: Amount of memory to allocate to the provider's lambda function. Default: Size.gibibytes(1)
        :param masters_role: An IAM role that will be added to the ``system:masters`` Kubernetes RBAC group. Default: - no masters role.
        :param on_event_layer: An AWS Lambda Layer which includes the NPM dependency ``proxy-agent``. This layer is used by the onEvent handler to route AWS SDK requests through a proxy. By default, the provider will use the layer included in the "aws-lambda-layer-node-proxy-agent" SAR application which is available in all commercial regions. To deploy the layer locally define it in your app as follows:: const layer = new lambda.LayerVersion(this, 'proxy-agent-layer', { code: lambda.Code.fromAsset(`${__dirname}/layer.zip`), compatibleRuntimes: [lambda.Runtime.NODEJS_LATEST], }); Default: - a layer bundled with this module.
        :param output_masters_role_arn: Determines whether a CloudFormation output with the ARN of the "masters" IAM role will be synthesized (if ``mastersRole`` is specified). Default: false
        :param place_cluster_handler_in_vpc: If set to true, the cluster handler functions will be placed in the private subnets of the cluster vpc, subject to the ``vpcSubnets`` selection strategy. Default: false
        :param prune: Indicates whether Kubernetes resources added through ``addManifest()`` can be automatically pruned. When this is enabled (default), prune labels will be allocated and injected to each resource. These labels will then be used when issuing the ``kubectl apply`` operation with the ``--prune`` switch. Default: true
        :param remote_node_networks: IPv4 CIDR blocks defining the expected address range of hybrid nodes that will join the cluster. Default: - none
        :param remote_pod_networks: IPv4 CIDR blocks for Pods running Kubernetes webhooks on hybrid nodes. Default: - none
        :param secrets_encryption_key: KMS secret for envelope encryption for Kubernetes secrets. Default: - By default, Kubernetes stores all secret object data within etcd and all etcd volumes used by Amazon EKS are encrypted at the disk-level using AWS-Managed encryption keys.
        :param service_ipv4_cidr: The CIDR block to assign Kubernetes service IP addresses from. Default: - Kubernetes assigns addresses from either the 10.100.0.0/16 or 172.20.0.0/16 CIDR blocks
        :param version: The Kubernetes version to run in the cluster.
        :param cluster_name: Name for the cluster. Default: - Automatically generated name
        :param output_cluster_name: Determines whether a CloudFormation output with the name of the cluster will be synthesized. Default: false
        :param output_config_command: Determines whether a CloudFormation output with the ``aws eks update-kubeconfig`` command will be synthesized. This command will include the cluster name and, if applicable, the ARN of the masters IAM role. Default: true
        :param role: Role that provides permissions for the Kubernetes control plane to make calls to AWS API operations on your behalf. Default: - A role is automatically created for you
        :param security_group: Security Group to use for Control Plane ENIs. Default: - A security group is automatically created
        :param vpc: The VPC in which to create the Cluster. Default: - a VPC with default configuration will be created and can be accessed through ``cluster.vpc``.
        :param vpc_subnets: Where to place EKS Control Plane ENIs. For example, to only select private subnets, supply the following: ``vpcSubnets: [{ subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS }]`` Default: - All public and private subnets
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__786576ad54eacdb9ab8e92277c0fd07f813bc56d4243937f3b5a85c0c575cac9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ClusterProps(
            bootstrap_cluster_creator_admin_permissions=bootstrap_cluster_creator_admin_permissions,
            default_capacity=default_capacity,
            default_capacity_instance=default_capacity_instance,
            default_capacity_type=default_capacity_type,
            kubectl_lambda_role=kubectl_lambda_role,
            tags=tags,
            kubectl_layer=kubectl_layer,
            alb_controller=alb_controller,
            authentication_mode=authentication_mode,
            awscli_layer=awscli_layer,
            cluster_handler_environment=cluster_handler_environment,
            cluster_handler_security_group=cluster_handler_security_group,
            cluster_logging=cluster_logging,
            core_dns_compute_type=core_dns_compute_type,
            endpoint_access=endpoint_access,
            ip_family=ip_family,
            kubectl_environment=kubectl_environment,
            kubectl_memory=kubectl_memory,
            masters_role=masters_role,
            on_event_layer=on_event_layer,
            output_masters_role_arn=output_masters_role_arn,
            place_cluster_handler_in_vpc=place_cluster_handler_in_vpc,
            prune=prune,
            remote_node_networks=remote_node_networks,
            remote_pod_networks=remote_pod_networks,
            secrets_encryption_key=secrets_encryption_key,
            service_ipv4_cidr=service_ipv4_cidr,
            version=version,
            cluster_name=cluster_name,
            output_cluster_name=output_cluster_name,
            output_config_command=output_config_command,
            role=role,
            security_group=security_group,
            vpc=vpc,
            vpc_subnets=vpc_subnets,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="fromClusterAttributes")
    @builtins.classmethod
    def from_cluster_attributes(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cluster_name: builtins.str,
        awscli_layer: typing.Optional[_ILayerVersion_5ac127c8] = None,
        cluster_certificate_authority_data: typing.Optional[builtins.str] = None,
        cluster_encryption_config_key_arn: typing.Optional[builtins.str] = None,
        cluster_endpoint: typing.Optional[builtins.str] = None,
        cluster_handler_security_group_id: typing.Optional[builtins.str] = None,
        cluster_security_group_id: typing.Optional[builtins.str] = None,
        ip_family: typing.Optional[IpFamily] = None,
        kubectl_environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        kubectl_lambda_role: typing.Optional[_IRole_235f5d8e] = None,
        kubectl_layer: typing.Optional[_ILayerVersion_5ac127c8] = None,
        kubectl_memory: typing.Optional[_Size_7b441c34] = None,
        kubectl_private_subnet_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        kubectl_provider: typing.Optional[IKubectlProvider] = None,
        kubectl_role_arn: typing.Optional[builtins.str] = None,
        kubectl_security_group_id: typing.Optional[builtins.str] = None,
        on_event_layer: typing.Optional[_ILayerVersion_5ac127c8] = None,
        open_id_connect_provider: typing.Optional[_IOpenIdConnectProvider_203f0793] = None,
        prune: typing.Optional[builtins.bool] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        vpc: typing.Optional[_IVpc_f30d5663] = None,
    ) -> ICluster:
        '''Import an existing cluster.

        :param scope: the construct scope, in most cases 'this'.
        :param id: the id or name to import as.
        :param cluster_name: The physical name of the Cluster.
        :param awscli_layer: An AWS Lambda layer that contains the ``aws`` CLI. The handler expects the layer to include the following executables:: /opt/awscli/aws Default: - a default layer with the AWS CLI 1.x
        :param cluster_certificate_authority_data: The certificate-authority-data for your cluster. Default: - if not specified ``cluster.clusterCertificateAuthorityData`` will throw an error
        :param cluster_encryption_config_key_arn: Amazon Resource Name (ARN) or alias of the customer master key (CMK). Default: - if not specified ``cluster.clusterEncryptionConfigKeyArn`` will throw an error
        :param cluster_endpoint: The API Server endpoint URL. Default: - if not specified ``cluster.clusterEndpoint`` will throw an error.
        :param cluster_handler_security_group_id: A security group id to associate with the Cluster Handler's Lambdas. The Cluster Handler's Lambdas are responsible for calling AWS's EKS API. Default: - No security group.
        :param cluster_security_group_id: The cluster security group that was created by Amazon EKS for the cluster. Default: - if not specified ``cluster.clusterSecurityGroupId`` will throw an error
        :param ip_family: Specify which IP family is used to assign Kubernetes pod and service IP addresses. Default: - IpFamily.IP_V4
        :param kubectl_environment: Environment variables to use when running ``kubectl`` against this cluster. Default: - no additional variables
        :param kubectl_lambda_role: An IAM role that can perform kubectl operations against this cluster. The role should be mapped to the ``system:masters`` Kubernetes RBAC role. This role is directly passed to the lambda handler that sends Kube Ctl commands to the cluster. Default: - if not specified, the default role created by a lambda function will be used.
        :param kubectl_layer: An AWS Lambda Layer which includes ``kubectl`` and Helm. This layer is used by the kubectl handler to apply manifests and install helm charts. You must pick an appropriate releases of one of the ``@aws-cdk/layer-kubectl-vXX`` packages, that works with the version of Kubernetes you have chosen. The handler expects the layer to include the following executables:: /opt/helm/helm /opt/kubectl/kubectl Default: - No default layer will be provided
        :param kubectl_memory: Amount of memory to allocate to the provider's lambda function. Default: Size.gibibytes(1)
        :param kubectl_private_subnet_ids: Subnets to host the ``kubectl`` compute resources. If not specified, the k8s endpoint is expected to be accessible publicly. Default: - k8s endpoint is expected to be accessible publicly
        :param kubectl_provider: KubectlProvider for issuing kubectl commands. Default: - Default CDK provider
        :param kubectl_role_arn: An IAM role with cluster administrator and "system:masters" permissions. Default: - if not specified, it not be possible to issue ``kubectl`` commands against an imported cluster.
        :param kubectl_security_group_id: A security group to use for ``kubectl`` execution. If not specified, the k8s endpoint is expected to be accessible publicly. Default: - k8s endpoint is expected to be accessible publicly
        :param on_event_layer: An AWS Lambda Layer which includes the NPM dependency ``proxy-agent``. This layer is used by the onEvent handler to route AWS SDK requests through a proxy. The handler expects the layer to include the following node_modules: proxy-agent Default: - a layer bundled with this module.
        :param open_id_connect_provider: An Open ID Connect provider for this cluster that can be used to configure service accounts. You can either import an existing provider using ``iam.OpenIdConnectProvider.fromProviderArn``, or create a new provider using ``new eks.OpenIdConnectProvider`` Default: - if not specified ``cluster.openIdConnectProvider`` and ``cluster.addServiceAccount`` will throw an error.
        :param prune: Indicates whether Kubernetes resources added through ``addManifest()`` can be automatically pruned. When this is enabled (default), prune labels will be allocated and injected to each resource. These labels will then be used when issuing the ``kubectl apply`` operation with the ``--prune`` switch. Default: true
        :param security_group_ids: Additional security groups associated with this cluster. Default: - if not specified, no additional security groups will be considered in ``cluster.connections``.
        :param vpc: The VPC in which this Cluster was created. Default: - if not specified ``cluster.vpc`` will throw an error
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c15d6f78a63887fd1847d93bb1b9ec971cf6424645a0049c1fca30cb9341c321)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        attrs = ClusterAttributes(
            cluster_name=cluster_name,
            awscli_layer=awscli_layer,
            cluster_certificate_authority_data=cluster_certificate_authority_data,
            cluster_encryption_config_key_arn=cluster_encryption_config_key_arn,
            cluster_endpoint=cluster_endpoint,
            cluster_handler_security_group_id=cluster_handler_security_group_id,
            cluster_security_group_id=cluster_security_group_id,
            ip_family=ip_family,
            kubectl_environment=kubectl_environment,
            kubectl_lambda_role=kubectl_lambda_role,
            kubectl_layer=kubectl_layer,
            kubectl_memory=kubectl_memory,
            kubectl_private_subnet_ids=kubectl_private_subnet_ids,
            kubectl_provider=kubectl_provider,
            kubectl_role_arn=kubectl_role_arn,
            kubectl_security_group_id=kubectl_security_group_id,
            on_event_layer=on_event_layer,
            open_id_connect_provider=open_id_connect_provider,
            prune=prune,
            security_group_ids=security_group_ids,
            vpc=vpc,
        )

        return typing.cast(ICluster, jsii.sinvoke(cls, "fromClusterAttributes", [scope, id, attrs]))

    @jsii.member(jsii_name="addAutoScalingGroupCapacity")
    def add_auto_scaling_group_capacity(
        self,
        id: builtins.str,
        *,
        instance_type: _InstanceType_f64915b9,
        bootstrap_enabled: typing.Optional[builtins.bool] = None,
        bootstrap_options: typing.Optional[typing.Union[BootstrapOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        machine_image_type: typing.Optional[MachineImageType] = None,
        map_role: typing.Optional[builtins.bool] = None,
        spot_interrupt_handler: typing.Optional[builtins.bool] = None,
        allow_all_outbound: typing.Optional[builtins.bool] = None,
        associate_public_ip_address: typing.Optional[builtins.bool] = None,
        auto_scaling_group_name: typing.Optional[builtins.str] = None,
        az_capacity_distribution_strategy: typing.Optional[_CapacityDistributionStrategy_2393ccfe] = None,
        block_devices: typing.Optional[typing.Sequence[typing.Union[_BlockDevice_0cfc0568, typing.Dict[builtins.str, typing.Any]]]] = None,
        capacity_rebalance: typing.Optional[builtins.bool] = None,
        cooldown: typing.Optional[_Duration_4839e8c3] = None,
        default_instance_warmup: typing.Optional[_Duration_4839e8c3] = None,
        desired_capacity: typing.Optional[jsii.Number] = None,
        group_metrics: typing.Optional[typing.Sequence[_GroupMetrics_7cdf729b]] = None,
        health_check: typing.Optional[_HealthCheck_03a4bd5a] = None,
        health_checks: typing.Optional[_HealthChecks_b8757873] = None,
        ignore_unmodified_size_properties: typing.Optional[builtins.bool] = None,
        instance_monitoring: typing.Optional[_Monitoring_50020f91] = None,
        key_name: typing.Optional[builtins.str] = None,
        key_pair: typing.Optional[_IKeyPair_bc344eda] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        max_instance_lifetime: typing.Optional[_Duration_4839e8c3] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        new_instances_protected_from_scale_in: typing.Optional[builtins.bool] = None,
        notifications: typing.Optional[typing.Sequence[typing.Union[_NotificationConfiguration_d5911670, typing.Dict[builtins.str, typing.Any]]]] = None,
        signals: typing.Optional[_Signals_69fbeb6e] = None,
        spot_price: typing.Optional[builtins.str] = None,
        ssm_session_permissions: typing.Optional[builtins.bool] = None,
        termination_policies: typing.Optional[typing.Sequence[_TerminationPolicy_89633c56]] = None,
        termination_policy_custom_lambda_function_arn: typing.Optional[builtins.str] = None,
        update_policy: typing.Optional[_UpdatePolicy_6dffc7ca] = None,
        vpc_subnets: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> _AutoScalingGroup_c547a7b9:
        '''Add nodes to this EKS cluster.

        The nodes will automatically be configured with the right VPC and AMI
        for the instance type and Kubernetes version.

        Note that if you specify ``updateType: RollingUpdate`` or ``updateType: ReplacingUpdate``, your nodes might be replaced at deploy
        time without notice in case the recommended AMI for your machine image type has been updated by AWS.
        The default behavior for ``updateType`` is ``None``, which means only new instances will be launched using the new AMI.

        Spot instances will be labeled ``lifecycle=Ec2Spot`` and tainted with ``PreferNoSchedule``.
        In addition, the `spot interrupt handler <https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler>`_
        daemon will be installed on all spot instances to handle
        `EC2 Spot Instance Termination Notices <https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/>`_.

        :param id: -
        :param instance_type: Instance type of the instances to start.
        :param bootstrap_enabled: Configures the EC2 user-data script for instances in this autoscaling group to bootstrap the node (invoke ``/etc/eks/bootstrap.sh``) and associate it with the EKS cluster. If you wish to provide a custom user data script, set this to ``false`` and manually invoke ``autoscalingGroup.addUserData()``. Default: true
        :param bootstrap_options: EKS node bootstrapping options. Default: - none
        :param machine_image_type: Machine image type. Default: MachineImageType.AMAZON_LINUX_2
        :param map_role: Will automatically update the aws-auth ConfigMap to map the IAM instance role to RBAC. This cannot be explicitly set to ``true`` if the cluster has kubectl disabled. Default: - true if the cluster has kubectl enabled (which is the default).
        :param spot_interrupt_handler: Installs the AWS spot instance interrupt handler on the cluster if it's not already added. Only relevant if ``spotPrice`` is used. Default: true
        :param allow_all_outbound: Whether the instances can initiate connections to anywhere by default. Default: true
        :param associate_public_ip_address: Whether instances in the Auto Scaling Group should have public IP addresses associated with them. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Use subnet setting.
        :param auto_scaling_group_name: The name of the Auto Scaling group. This name must be unique per Region per account. Default: - Auto generated by CloudFormation
        :param az_capacity_distribution_strategy: The strategy for distributing instances across Availability Zones. Default: None
        :param block_devices: Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes. Each instance that is launched has an associated root device volume, either an Amazon EBS volume or an instance store volume. You can use block device mappings to specify additional EBS volumes or instance store volumes to attach to an instance when it is launched. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Uses the block device mapping of the AMI
        :param capacity_rebalance: Indicates whether Capacity Rebalancing is enabled. When you turn on Capacity Rebalancing, Amazon EC2 Auto Scaling attempts to launch a Spot Instance whenever Amazon EC2 notifies that a Spot Instance is at an elevated risk of interruption. After launching a new instance, it then terminates an old instance. Default: false
        :param cooldown: Default scaling cooldown for this AutoScalingGroup. Default: Duration.minutes(5)
        :param default_instance_warmup: The amount of time, in seconds, until a newly launched instance can contribute to the Amazon CloudWatch metrics. This delay lets an instance finish initializing before Amazon EC2 Auto Scaling aggregates instance metrics, resulting in more reliable usage data. Set this value equal to the amount of time that it takes for resource consumption to become stable after an instance reaches the InService state. To optimize the performance of scaling policies that scale continuously, such as target tracking and step scaling policies, we strongly recommend that you enable the default instance warmup, even if its value is set to 0 seconds Default instance warmup will not be added if no value is specified Default: None
        :param desired_capacity: Initial amount of instances in the fleet. If this is set to a number, every deployment will reset the amount of instances to this number. It is recommended to leave this value blank. Default: minCapacity, and leave unchanged during deployment
        :param group_metrics: Enable monitoring for group metrics, these metrics describe the group rather than any of its instances. To report all group metrics use ``GroupMetrics.all()`` Group metrics are reported in a granularity of 1 minute at no additional charge. Default: - no group metrics will be reported
        :param health_check: (deprecated) Configuration for health checks. Default: - HealthCheck.ec2 with no grace period
        :param health_checks: Configuration for EC2 or additional health checks. Even when using ``HealthChecks.withAdditionalChecks()``, the EC2 type is implicitly included. Default: - EC2 type with no grace period
        :param ignore_unmodified_size_properties: If the ASG has scheduled actions, don't reset unchanged group sizes. Only used if the ASG has scheduled actions (which may scale your ASG up or down regardless of cdk deployments). If true, the size of the group will only be reset if it has been changed in the CDK app. If false, the sizes will always be changed back to what they were in the CDK app on deployment. Default: true
        :param instance_monitoring: Controls whether instances in this group are launched with detailed or basic monitoring. When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: - Monitoring.DETAILED
        :param key_name: (deprecated) Name of SSH keypair to grant access to instances. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified You can either specify ``keyPair`` or ``keyName``, not both. Default: - No SSH access will be possible.
        :param key_pair: The SSH keypair to grant access to the instance. Feature flag ``AUTOSCALING_GENERATE_LAUNCH_TEMPLATE`` must be enabled to use this property. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified. You can either specify ``keyPair`` or ``keyName``, not both. Default: - No SSH access will be possible.
        :param max_capacity: Maximum number of instances in the fleet. Default: desiredCapacity
        :param max_instance_lifetime: The maximum amount of time that an instance can be in service. The maximum duration applies to all current and future instances in the group. As an instance approaches its maximum duration, it is terminated and replaced, and cannot be used again. You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value, leave this property undefined. Default: none
        :param min_capacity: Minimum number of instances in the fleet. Default: 1
        :param new_instances_protected_from_scale_in: Whether newly-launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in. By default, Auto Scaling can terminate an instance at any time after launch when scaling in an Auto Scaling Group, subject to the group's termination policy. However, you may wish to protect newly-launched instances from being scaled in if they are going to run critical applications that should not be prematurely terminated. This flag must be enabled if the Auto Scaling Group will be associated with an ECS Capacity Provider with managed termination protection. Default: false
        :param notifications: Configure autoscaling group to send notifications about fleet changes to an SNS topic(s). Default: - No fleet change notifications will be sent.
        :param signals: Configure waiting for signals during deployment. Use this to pause the CloudFormation deployment to wait for the instances in the AutoScalingGroup to report successful startup during creation and updates. The UserData script needs to invoke ``cfn-signal`` with a success or failure code after it is done setting up the instance. Without waiting for signals, the CloudFormation deployment will proceed as soon as the AutoScalingGroup has been created or updated but before the instances in the group have been started. For example, to have instances wait for an Elastic Load Balancing health check before they signal success, add a health-check verification by using the cfn-init helper script. For an example, see the verify_instance_health command in the Auto Scaling rolling updates sample template: https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/services/AutoScaling/AutoScalingRollingUpdates.yaml Default: - Do not wait for signals
        :param spot_price: The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request. Spot Instances are launched when the price you specify exceeds the current Spot market price. ``launchTemplate`` and ``mixedInstancesPolicy`` must not be specified when this property is specified Default: none
        :param ssm_session_permissions: Add SSM session permissions to the instance role. Setting this to ``true`` adds the necessary permissions to connect to the instance using SSM Session Manager. You can do this from the AWS Console. NOTE: Setting this flag to ``true`` may not be enough by itself. You must also use an AMI that comes with the SSM Agent, or install the SSM Agent yourself. See `Working with SSM Agent <https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-agent.html>`_ in the SSM Developer Guide. Default: false
        :param termination_policies: A policy or a list of policies that are used to select the instances to terminate. The policies are executed in the order that you list them. Default: - ``TerminationPolicy.DEFAULT``
        :param termination_policy_custom_lambda_function_arn: A lambda function Arn that can be used as a custom termination policy to select the instances to terminate. This property must be specified if the TerminationPolicy.CUSTOM_LAMBDA_FUNCTION is used. Default: - No lambda function Arn will be supplied
        :param update_policy: What to do when an AutoScalingGroup's instance configuration is changed. This is applied when any of the settings on the ASG are changed that affect how the instances should be created (VPC, instance type, startup scripts, etc.). It indicates how the existing instances should be replaced with new instances matching the new config. By default, nothing is done and only new instances are launched with the new config. Default: - ``UpdatePolicy.rollingUpdate()`` if using ``init``, ``UpdatePolicy.none()`` otherwise
        :param vpc_subnets: Where to place instances within the VPC. Default: - All Private subnets.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e9e81d821b1c1d14225d1c9cc695af8e71b96a7489dcd36bd237c9363a7c77e6)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = AutoScalingGroupCapacityOptions(
            instance_type=instance_type,
            bootstrap_enabled=bootstrap_enabled,
            bootstrap_options=bootstrap_options,
            machine_image_type=machine_image_type,
            map_role=map_role,
            spot_interrupt_handler=spot_interrupt_handler,
            allow_all_outbound=allow_all_outbound,
            associate_public_ip_address=associate_public_ip_address,
            auto_scaling_group_name=auto_scaling_group_name,
            az_capacity_distribution_strategy=az_capacity_distribution_strategy,
            block_devices=block_devices,
            capacity_rebalance=capacity_rebalance,
            cooldown=cooldown,
            default_instance_warmup=default_instance_warmup,
            desired_capacity=desired_capacity,
            group_metrics=group_metrics,
            health_check=health_check,
            health_checks=health_checks,
            ignore_unmodified_size_properties=ignore_unmodified_size_properties,
            instance_monitoring=instance_monitoring,
            key_name=key_name,
            key_pair=key_pair,
            max_capacity=max_capacity,
            max_instance_lifetime=max_instance_lifetime,
            min_capacity=min_capacity,
            new_instances_protected_from_scale_in=new_instances_protected_from_scale_in,
            notifications=notifications,
            signals=signals,
            spot_price=spot_price,
            ssm_session_permissions=ssm_session_permissions,
            termination_policies=termination_policies,
            termination_policy_custom_lambda_function_arn=termination_policy_custom_lambda_function_arn,
            update_policy=update_policy,
            vpc_subnets=vpc_subnets,
        )

        return typing.cast(_AutoScalingGroup_c547a7b9, jsii.invoke(self, "addAutoScalingGroupCapacity", [id, options]))

    @jsii.member(jsii_name="addCdk8sChart")
    def add_cdk8s_chart(
        self,
        id: builtins.str,
        chart: _constructs_77d1e7e8.Construct,
        *,
        ingress_alb: typing.Optional[builtins.bool] = None,
        ingress_alb_scheme: typing.Optional[AlbScheme] = None,
        prune: typing.Optional[builtins.bool] = None,
        skip_validation: typing.Optional[builtins.bool] = None,
    ) -> KubernetesManifest:
        '''Defines a CDK8s chart in this cluster.

        :param id: logical id of this chart.
        :param chart: the cdk8s chart.
        :param ingress_alb: Automatically detect ``Ingress`` resources in the manifest and annotate them so they are picked up by an ALB Ingress Controller. Default: false
        :param ingress_alb_scheme: Specify the ALB scheme that should be applied to ``Ingress`` resources. Only applicable if ``ingressAlb`` is set to ``true``. Default: AlbScheme.INTERNAL
        :param prune: When a resource is removed from a Kubernetes manifest, it no longer appears in the manifest, and there is no way to know that this resource needs to be deleted. To address this, ``kubectl apply`` has a ``--prune`` option which will query the cluster for all resources with a specific label and will remove all the labeld resources that are not part of the applied manifest. If this option is disabled and a resource is removed, it will become "orphaned" and will not be deleted from the cluster. When this option is enabled (default), the construct will inject a label to all Kubernetes resources included in this manifest which will be used to prune resources when the manifest changes via ``kubectl apply --prune``. The label name will be ``aws.cdk.eks/prune-<ADDR>`` where ``<ADDR>`` is the 42-char unique address of this construct in the construct tree. Value is empty. Default: - based on the prune option of the cluster, which is ``true`` unless otherwise specified.
        :param skip_validation: A flag to signify if the manifest validation should be skipped. Default: false

        :return: a ``KubernetesManifest`` construct representing the chart.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c0b87c41553cee5becfcb7d4f0a1e65b0d2f83e2b04c50e5031eef8599636915)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument chart", value=chart, expected_type=type_hints["chart"])
        options = KubernetesManifestOptions(
            ingress_alb=ingress_alb,
            ingress_alb_scheme=ingress_alb_scheme,
            prune=prune,
            skip_validation=skip_validation,
        )

        return typing.cast(KubernetesManifest, jsii.invoke(self, "addCdk8sChart", [id, chart, options]))

    @jsii.member(jsii_name="addFargateProfile")
    def add_fargate_profile(
        self,
        id: builtins.str,
        *,
        selectors: typing.Sequence[typing.Union[Selector, typing.Dict[builtins.str, typing.Any]]],
        fargate_profile_name: typing.Optional[builtins.str] = None,
        pod_execution_role: typing.Optional[_IRole_235f5d8e] = None,
        subnet_selection: typing.Optional[typing.Union[_SubnetSelection_e57d76df, typing.Dict[builtins.str, typing.Any]]] = None,
        vpc: typing.Optional[_IVpc_f30d5663] = None,
    ) -> FargateProfile:
        '''Adds a Fargate profile to this cluster.

        :param id: the id of this profile.
        :param selectors: The selectors to match for pods to use this Fargate profile. Each selector must have an associated namespace. Optionally, you can also specify labels for a namespace. At least one selector is required and you may specify up to five selectors.
        :param fargate_profile_name: The name of the Fargate profile. Default: - generated
        :param pod_execution_role: The pod execution role to use for pods that match the selectors in the Fargate profile. The pod execution role allows Fargate infrastructure to register with your cluster as a node, and it provides read access to Amazon ECR image repositories. Default: - a role will be automatically created
        :param subnet_selection: Select which subnets to launch your pods into. At this time, pods running on Fargate are not assigned public IP addresses, so only private subnets (with no direct route to an Internet Gateway) are allowed. You must specify the VPC to customize the subnet selection Default: - all private subnets of the VPC are selected.
        :param vpc: The VPC from which to select subnets to launch your pods into. By default, all private subnets are selected. You can customize this using ``subnetSelection``. Default: - all private subnets used by the EKS cluster

        :see: https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a6aa2e382a694744786e7d6854ca13bd607970f73536c975ae7475325e830142)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = FargateProfileOptions(
            selectors=selectors,
            fargate_profile_name=fargate_profile_name,
            pod_execution_role=pod_execution_role,
            subnet_selection=subnet_selection,
            vpc=vpc,
        )

        return typing.cast(FargateProfile, jsii.invoke(self, "addFargateProfile", [id, options]))

    @jsii.member(jsii_name="addHelmChart")
    def add_helm_chart(
        self,
        id: builtins.str,
        *,
        atomic: typing.Optional[builtins.bool] = None,
        chart: typing.Optional[builtins.str] = None,
        chart_asset: typing.Optional[_Asset_ac2a7e61] = None,
        create_namespace: typing.Optional[builtins.bool] = None,
        namespace: typing.Optional[builtins.str] = None,
        release: typing.Optional[builtins.str] = None,
        repository: typing.Optional[builtins.str] = None,
        skip_crds: typing.Optional[builtins.bool] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        values: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        version: typing.Optional[builtins.str] = None,
        wait: typing.Optional[builtins.bool] = None,
    ) -> HelmChart:
        '''Defines a Helm chart in this cluster.

        :param id: logical id of this chart.
        :param atomic: Whether or not Helm should treat this operation as atomic; if set, upgrade process rolls back changes made in case of failed upgrade. The --wait flag will be set automatically if --atomic is used. Default: false
        :param chart: The name of the chart. Either this or ``chartAsset`` must be specified. Default: - No chart name. Implies ``chartAsset`` is used.
        :param chart_asset: The chart in the form of an asset. Either this or ``chart`` must be specified. Default: - No chart asset. Implies ``chart`` is used.
        :param create_namespace: create namespace if not exist. Default: true
        :param namespace: The Kubernetes namespace scope of the requests. Default: default
        :param release: The name of the release. Default: - If no release name is given, it will use the last 53 characters of the node's unique id.
        :param repository: The repository which contains the chart. For example: https://charts.helm.sh/stable/ Default: - No repository will be used, which means that the chart needs to be an absolute URL.
        :param skip_crds: if set, no CRDs will be installed. Default: - CRDs are installed if not already present
        :param timeout: Amount of time to wait for any individual Kubernetes operation. Maximum 15 minutes. Default: Duration.minutes(5)
        :param values: The values to be used by the chart. For nested values use a nested dictionary. For example: values: { installationCRDs: true, webhook: { port: 9443 } } Default: - No values are provided to the chart.
        :param version: The chart version to install. Default: - If this is not specified, the latest version is installed
        :param wait: Whether or not Helm should wait until all Pods, PVCs, Services, and minimum number of Pods of a Deployment, StatefulSet, or ReplicaSet are in a ready state before marking the release as successful. Default: - Helm will not wait before marking release as successful

        :return: a ``HelmChart`` construct
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ff2724c18a4ac2eb51d401c5ea14857dd81b51b7f44f46a894e6cac37edd774f)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = HelmChartOptions(
            atomic=atomic,
            chart=chart,
            chart_asset=chart_asset,
            create_namespace=create_namespace,
            namespace=namespace,
            release=release,
            repository=repository,
            skip_crds=skip_crds,
            timeout=timeout,
            values=values,
            version=version,
            wait=wait,
        )

        return typing.cast(HelmChart, jsii.invoke(self, "addHelmChart", [id, options]))

    @jsii.member(jsii_name="addManifest")
    def add_manifest(
        self,
        id: builtins.str,
        *manifest: typing.Mapping[builtins.str, typing.Any],
    ) -> KubernetesManifest:
        '''Defines a Kubernetes resource in this cluster.

        The manifest will be applied/deleted using kubectl as needed.

        :param id: logical id of this manifest.
        :param manifest: a list of Kubernetes resource specifications.

        :return: a ``KubernetesResource`` object.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8e86f333dc31590babdc3e5ef2e86a2ab43f7ef63e489882759a3079a37c9874)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument manifest", value=manifest, expected_type=typing.Tuple[type_hints["manifest"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast(KubernetesManifest, jsii.invoke(self, "addManifest", [id, *