r'''
# Amazon EC2 Construct Library

The `aws-cdk-lib/aws-ec2` package contains primitives for setting up networking and
instances.

```python
import aws_cdk.aws_ec2 as ec2
```

## VPC

Most projects need a Virtual Private Cloud to provide security by means of
network partitioning. This is achieved by creating an instance of
`Vpc`:

```python
vpc = ec2.Vpc(self, "VPC")
```

All default constructs require EC2 instances to be launched inside a VPC, so
you should generally start by defining a VPC whenever you need to launch
instances for your project.

### Subnet Types

A VPC consists of one or more subnets that instances can be placed into. CDK
distinguishes three different subnet types:

* **Public (`SubnetType.PUBLIC`)** - public subnets connect directly to the Internet using an
  Internet Gateway. If you want your instances to have a public IP address
  and be directly reachable from the Internet, you must place them in a
  public subnet.
* **Private with Internet Access (`SubnetType.PRIVATE_WITH_EGRESS`)** - instances in private subnets are not directly routable from the
  Internet, and you must provide a way to connect out to the Internet.
  By default, a NAT gateway is created in every public subnet for maximum availability. Be
  aware that you will be charged for NAT gateways.
  Alternatively you can set `natGateways:0` and provide your own egress configuration (i.e through Transit Gateway)
* **Isolated (`SubnetType.PRIVATE_ISOLATED`)** - isolated subnets do not route from or to the Internet, and
  as such do not require NAT gateways. They can only connect to or be
  connected to from other instances in the same VPC. A default VPC configuration
  will not include isolated subnets,

A default VPC configuration will create public and **private** subnets. However, if
`natGateways:0` **and** `subnetConfiguration` is undefined, default VPC configuration
will create public and **isolated** subnets. See [*Advanced Subnet Configuration*](#advanced-subnet-configuration)
below for information on how to change the default subnet configuration.

Constructs using the VPC will "launch instances" (or more accurately, create
Elastic Network Interfaces) into one or more of the subnets. They all accept
a property called `subnetSelection` (sometimes called `vpcSubnets`) to allow
you to select in what subnet to place the ENIs, usually defaulting to
*private* subnets if the property is omitted.

If you would like to save on the cost of NAT gateways, you can use
*isolated* subnets instead of *private* subnets (as described in Advanced
*Subnet Configuration*). If you need private instances to have
internet connectivity, another option is to reduce the number of NAT gateways
created by setting the `natGateways` property to a lower value (the default
is one NAT gateway per availability zone). Be aware that this may have
availability implications for your application.

[Read more about
subnets](https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Subnets.html).

### Control over availability zones

By default, a VPC will spread over at most 3 Availability Zones available to
it. To change the number of Availability Zones that the VPC will spread over,
specify the `maxAzs` property when defining it.

The number of Availability Zones that are available depends on the *region*
and *account* of the Stack containing the VPC. If the [region and account are
specified](https://docs.aws.amazon.com/cdk/latest/guide/environments.html) on
the Stack, the CLI will [look up the existing Availability
Zones](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones.html#using-regions-availability-zones-describe)
and get an accurate count. The result of this operation will be written to a file
called `cdk.context.json`. You must commit this file to source control so
that the lookup values are available in non-privileged environments such
as CI build steps, and to ensure your template builds are repeatable.

If region and account are not specified, the stack
could be deployed anywhere and it will have to make a safe choice, limiting
itself to 2 Availability Zones.

Therefore, to get the VPC to spread over 3 or more availability zones, you
must specify the environment where the stack will be deployed.

You can gain full control over the availability zones selection strategy by overriding the Stack's [`get availabilityZones()`](https://github.com/aws/aws-cdk/blob/main/packages/@aws-cdk/core/lib/stack.ts) method:

```text
// This example is only available in TypeScript

class MyStack extends Stack {

  constructor(scope: Construct, id: string, props?: StackProps) {
    super(scope, id, props);

    // ...
  }

  get availabilityZones(): string[] {
    return ['us-west-2a', 'us-west-2b'];
  }

}
```

Note that overriding the `get availabilityZones()` method will override the default behavior for all constructs defined within the Stack.

### Choosing subnets for resources

When creating resources that create Elastic Network Interfaces (such as
databases or instances), there is an option to choose which subnets to place
them in. For example, a VPC endpoint by default is placed into a subnet in
every availability zone, but you can override which subnets to use. The property
is typically called one of `subnets`, `vpcSubnets` or `subnetSelection`.

The example below will place the endpoint into two AZs (`us-east-1a` and `us-east-1c`),
in Isolated subnets:

```python
# vpc: ec2.Vpc


ec2.InterfaceVpcEndpoint(self, "VPC Endpoint",
    vpc=vpc,
    service=ec2.InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    subnets=ec2.SubnetSelection(
        subnet_type=ec2.SubnetType.PRIVATE_ISOLATED,
        availability_zones=["us-east-1a", "us-east-1c"]
    )
)
```

You can also specify specific subnet objects for granular control:

```python
# vpc: ec2.Vpc
# subnet1: ec2.Subnet
# subnet2: ec2.Subnet


ec2.InterfaceVpcEndpoint(self, "VPC Endpoint",
    vpc=vpc,
    service=ec2.InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    subnets=ec2.SubnetSelection(
        subnets=[subnet1, subnet2]
    )
)
```

Which subnets are selected is evaluated as follows:

* `subnets`: if specific subnet objects are supplied, these are selected, and no other
  logic is used.
* `subnetType`/`subnetGroupName`: otherwise, a set of subnets is selected by
  supplying either type or name:

  * `subnetType` will select all subnets of the given type.
  * `subnetGroupName` should be used to distinguish between multiple groups of subnets of
    the same type (for example, you may want to separate your application instances and your
    RDS instances into two distinct groups of Isolated subnets).
  * If neither are given, the first available subnet group of a given type that
    exists in the VPC will be used, in this order: Private, then Isolated, then Public.
    In short: by default ENIs will preferentially be placed in subnets not connected to
    the Internet.
* `availabilityZones`/`onePerAz`: finally, some availability-zone based filtering may be done.
  This filtering by availability zones will only be possible if the VPC has been created or
  looked up in a non-environment agnostic stack (so account and region have been set and
  availability zones have been looked up).

  * `availabilityZones`: only the specific subnets from the selected subnet groups that are
    in the given availability zones will be returned.
  * `onePerAz`: per availability zone, a maximum of one subnet will be returned (Useful for resource
    types that do not allow creating two ENIs in the same availability zone).
* `subnetFilters`: additional filtering on subnets using any number of user-provided filters which
  extend `SubnetFilter`.  The following methods on the `SubnetFilter` class can be used to create
  a filter:

  * `byIds`: chooses subnets from a list of ids
  * `availabilityZones`: chooses subnets in the provided list of availability zones
  * `onePerAz`: chooses at most one subnet per availability zone
  * `containsIpAddresses`: chooses a subnet which contains *any* of the listed ip addresses
  * `byCidrMask`: chooses subnets that have the provided CIDR netmask
  * `byCidrRanges`: chooses subnets which are inside any of the specified CIDR ranges

### Using NAT instances

By default, the `Vpc` construct will create NAT *gateways* for you, which
are managed by AWS. If you would prefer to use your own managed NAT
*instances* instead, specify a different value for the `natGatewayProvider`
property, as follows:

The construct will automatically selects the latest version of Amazon Linux 2023.
If you prefer to use a custom AMI, use `machineImage: MachineImage.genericLinux({ ... })` and configure the right AMI ID for the
regions you want to deploy to.

> **Warning**
> The NAT instances created using this method will be **unmonitored**.
> They are not part of an Auto Scaling Group,
> and if they become unavailable or are terminated for any reason,
> will not be restarted or replaced.

By default, the NAT instances will route all traffic. To control what traffic
gets routed, pass a custom value for `defaultAllowedTraffic` and access the
`NatInstanceProvider.connections` member after having passed the NAT provider to
the VPC:

```python
# instance_type: ec2.InstanceType


provider = ec2.NatProvider.instance_v2(
    instance_type=instance_type,
    default_allowed_traffic=ec2.NatTrafficDirection.OUTBOUND_ONLY
)
ec2.Vpc(self, "TheVPC",
    nat_gateway_provider=provider
)
provider.connections.allow_from(ec2.Peer.ipv4("1.2.3.4/8"), ec2.Port.HTTP)
```

You can also customize the characteristics of your NAT instances, including their security group,
as well as their initialization scripts:

```python
# bucket: s3.Bucket


user_data = ec2.UserData.for_linux()
user_data.add_commands(
    (SpreadElement ...ec2.NatInstanceProviderV2.DEFAULT_USER_DATA_COMMANDS
      ec2.NatInstanceProviderV2.DEFAULT_USER_DATA_COMMANDS), "echo \"hello world!\" > hello.txt", f"aws s3 cp hello.txt s3://{bucket.bucketName}")

provider = ec2.NatProvider.instance_v2(
    instance_type=ec2.InstanceType("t3.small"),
    credit_specification=ec2.CpuCredits.UNLIMITED,
    default_allowed_traffic=ec2.NatTrafficDirection.NONE
)

vpc = ec2.Vpc(self, "TheVPC",
    nat_gateway_provider=provider,
    nat_gateways=2
)

security_group = ec2.SecurityGroup(self, "SecurityGroup", vpc=vpc)
security_group.add_egress_rule(ec2.Peer.any_ipv4(), ec2.Port.tcp(443))
for gateway in provider.gateway_instances:
    bucket.grant_write(gateway)
    gateway.add_security_group(security_group)
```

```python
# Configure the `natGatewayProvider` when defining a Vpc
nat_gateway_provider = ec2.NatProvider.instance(
    instance_type=ec2.InstanceType("t3.small")
)

vpc = ec2.Vpc(self, "MyVpc",
    nat_gateway_provider=nat_gateway_provider,

    # The 'natGateways' parameter now controls the number of NAT instances
    nat_gateways=2
)
```

The V1 `NatProvider.instance` construct will use the AWS official NAT instance AMI, which has already
reached EOL on Dec 31, 2023. For more information, see the following blog post:
[Amazon Linux AMI end of life](https://aws.amazon.com/blogs/aws/update-on-amazon-linux-ami-end-of-life/).

```python
# instance_type: ec2.InstanceType


provider = ec2.NatProvider.instance(
    instance_type=instance_type,
    default_allowed_traffic=ec2.NatTrafficDirection.OUTBOUND_ONLY
)
ec2.Vpc(self, "TheVPC",
    nat_gateway_provider=provider
)
provider.connections.allow_from(ec2.Peer.ipv4("1.2.3.4/8"), ec2.Port.HTTP)
```

### Associate Public IP Address to NAT Instance

You can choose to associate public IP address to a NAT instance V2 by specifying `associatePublicIpAddress`
like the following:

```python
nat_gateway_provider = ec2.NatProvider.instance_v2(
    instance_type=ec2.InstanceType("t3.small"),
    associate_public_ip_address=True
)
```

In certain scenarios where the public subnet has set `mapPublicIpOnLaunch` to `false`, NAT instances does not
get public IP addresses assigned which would result in non-working NAT instance as NAT instance requires a public
IP address to enable outbound internet connectivity. Users can specify `associatePublicIpAddress` to `true` to
solve this problem.

### Ip Address Management

The VPC spans a supernet IP range, which contains the non-overlapping IPs of its contained subnets. Possible sources for this IP range are:

* You specify an IP range directly by specifying a CIDR
* You allocate an IP range of a given size automatically from AWS IPAM

By default the Vpc will allocate the `10.0.0.0/16` address range which will be exhaustively spread across all subnets in the subnet configuration. This behavior can be changed by passing an object that implements `IIpAddresses` to the `ipAddress` property of a Vpc. See the subsequent sections for the options.

Be aware that if you don't explicitly reserve subnet groups in `subnetConfiguration`, the address space will be fully allocated! If you predict you may need to add more subnet groups later, add them early on and set `reserved: true` (see the "Advanced Subnet Configuration" section for more information).

#### Specifying a CIDR directly

Use `IpAddresses.cidr` to define a Cidr range for your Vpc directly in code:

```python
from aws_cdk.aws_ec2 import IpAddresses


ec2.Vpc(self, "TheVPC",
    ip_addresses=IpAddresses.cidr("10.0.1.0/20")
)
```

Space will be allocated to subnets in the following order:

* First, spaces is allocated for all subnets groups that explicitly have a `cidrMask` set as part of their configuration (including reserved subnets).
* Afterwards, any remaining space is divided evenly between the rest of the subnets (if any).

The argument to `IpAddresses.cidr` may not be a token, and concrete Cidr values are generated in the synthesized CloudFormation template.

#### Allocating an IP range from AWS IPAM

Amazon VPC IP Address Manager (IPAM) manages a large IP space, from which chunks can be allocated for use in the Vpc. For information on Amazon VPC IP Address Manager please see the [official documentation](https://docs.aws.amazon.com/vpc/latest/ipam/what-it-is-ipam.html). An example of allocating from AWS IPAM looks like this:

```python
from aws_cdk.aws_ec2 import IpAddresses

# pool: ec2.CfnIPAMPool


ec2.Vpc(self, "TheVPC",
    ip_addresses=IpAddresses.aws_ipam_allocation(
        ipv4_ipam_pool_id=pool.ref,
        ipv4_netmask_length=18,
        default_subnet_ipv4_netmask_length=24
    )
)
```

`IpAddresses.awsIpamAllocation` requires the following:

* `ipv4IpamPoolId`, the id of an IPAM Pool from which the VPC range should be allocated.
* `ipv4NetmaskLength`, the size of the IP range that will be requested from the Pool at deploy time.
* `defaultSubnetIpv4NetmaskLength`, the size of subnets in groups that don't have `cidrMask` set.

With this method of IP address management, no attempt is made to guess at subnet group sizes or to exhaustively allocate the IP range. All subnet groups must have an explicit `cidrMask` set as part of their subnet configuration, or `defaultSubnetIpv4NetmaskLength` must be set for a default size. If not, synthesis will fail and you must provide one or the other.

### Dual Stack configuration

To allocate both IPv4 and IPv6 addresses in your VPC, you can configure your VPC to have a dual stack protocol.

```python
ec2.Vpc(self, "DualStackVpc",
    ip_protocol=ec2.IpProtocol.DUAL_STACK
)
```

By default, a dual stack VPC will create an Amazon provided IPv6 /56 CIDR block associated to the VPC. It will then assign /64 portions of the block to each subnet. For each subnet, auto-assigning an IPv6 address will be enabled, and auto-asigning a public IPv4 address will be disabled. An egress only internet gateway will be created for `PRIVATE_WITH_EGRESS` subnets, and IPv6 routes will be added for IGWs and EIGWs.

Disabling the auto-assigning of a public IPv4 address by default can avoid the cost of public IPv4 addresses starting 2/1/2024. For use cases that need an IPv4 address, the `mapPublicIpOnLaunch` property in `subnetConfiguration` can be set to auto-assign the IPv4 address. Note that private IPv4 address allocation will not be changed.

See [Advanced Subnet Configuration](#advanced-subnet-configuration) for all IPv6 specific properties.

### Reserving availability zones

There are situations where the IP space for availability zones will
need to be reserved. This is useful in situations where availability
zones would need to be added after the vpc is originally deployed,
without causing IP renumbering for availability zones subnets. The IP
space for reserving `n` availability zones can be done by setting the
`reservedAzs` to `n` in vpc props, as shown below:

```python
vpc = ec2.Vpc(self, "TheVPC",
    cidr="10.0.0.0/21",
    max_azs=3,
    reserved_azs=1
)
```

In the example above, the subnets for reserved availability zones is not
actually provisioned but its IP space is still reserved. If, in the future,
new availability zones needs to be provisioned, then we would decrement
the value of `reservedAzs` and increment the `maxAzs` or `availabilityZones`
accordingly. This action would not cause the IP address of subnets to get
renumbered, but rather the IP space that was previously reserved will be
used for the new availability zones subnets.

### Advanced Subnet Configuration

If the default VPC configuration (public and private subnets spanning the
size of the VPC) don't suffice for you, you can configure what subnets to
create by specifying the `subnetConfiguration` property. It allows you
to configure the number and size of all subnets. Specifying an advanced
subnet configuration could look like this:

```python
vpc = ec2.Vpc(self, "TheVPC",
    # 'IpAddresses' configures the IP range and size of the entire VPC.
    # The IP space will be divided based on configuration for the subnets.
    ip_addresses=ec2.IpAddresses.cidr("10.0.0.0/21"),

    # 'maxAzs' configures the maximum number of availability zones to use.
    # If you want to specify the exact availability zones you want the VPC
    # to use, use `availabilityZones` instead.
    max_azs=3,

    # 'subnetConfiguration' specifies the "subnet groups" to create.
    # Every subnet group will have a subnet for each AZ, so this
    # configuration will create `3 groups × 3 AZs = 9` subnets.
    subnet_configuration=[ec2.SubnetConfiguration(
        # 'subnetType' controls Internet access, as described above.
        subnet_type=ec2.SubnetType.PUBLIC,

        # 'name' is used to name this particular subnet group. You will have to
        # use the name for subnet selection if you have more than one subnet
        # group of the same type.
        name="Ingress",

        # 'cidrMask' specifies the IP addresses in the range of of individual
        # subnets in the group. Each of the subnets in this group will contain
        # `2^(32 address bits - 24 subnet bits) - 2 reserved addresses = 254`
        # usable IP addresses.
        #
        # If 'cidrMask' is left out the available address space is evenly
        # divided across the remaining subnet groups.
        cidr_mask=24
    ), ec2.SubnetConfiguration(
        cidr_mask=24,
        name="Application",
        subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS
    ), ec2.SubnetConfiguration(
        cidr_mask=28,
        name="Database",
        subnet_type=ec2.SubnetType.PRIVATE_ISOLATED,

        # 'reserved' can be used to reserve IP address space. No resources will
        # be created for this subnet, but the IP range will be kept available for
        # future creation of this subnet, or even for future subdivision.
        reserved=True
    )
    ]
)
```

The example above is one possible configuration, but the user can use the
constructs above to implement many other network configurations.

The `Vpc` from the above configuration in a Region with three
availability zones will be the following:

Subnet Name       |Type      |IP Block      |AZ|Features
------------------|----------|--------------|--|--------
IngressSubnet1    |`PUBLIC`  |`10.0.0.0/24` |#1|NAT Gateway
IngressSubnet2    |`PUBLIC`  |`10.0.1.0/24` |#2|NAT Gateway
IngressSubnet3    |`PUBLIC`  |`10.0.2.0/24` |#3|NAT Gateway
ApplicationSubnet1|`PRIVATE` |`10.0.3.0/24` |#1|Route to NAT in IngressSubnet1
ApplicationSubnet2|`PRIVATE` |`10.0.4.0/24` |#2|Route to NAT in IngressSubnet2
ApplicationSubnet3|`PRIVATE` |`10.0.5.0/24` |#3|Route to NAT in IngressSubnet3
DatabaseSubnet1   |`ISOLATED`|`10.0.6.0/28` |#1|Only routes within the VPC
DatabaseSubnet2   |`ISOLATED`|`10.0.6.16/28`|#2|Only routes within the VPC
DatabaseSubnet3   |`ISOLATED`|`10.0.6.32/28`|#3|Only routes within the VPC

#### Dual Stack Configurations

Here is a break down of IPv4 and IPv6 specific `subnetConfiguration` properties in a dual stack VPC:

```python
vpc = ec2.Vpc(self, "TheVPC",
    ip_protocol=ec2.IpProtocol.DUAL_STACK,

    subnet_configuration=[ec2.SubnetConfiguration(
        # general properties
        name="Public",
        subnet_type=ec2.SubnetType.PUBLIC,
        reserved=False,

        # IPv4 specific properties
        map_public_ip_on_launch=True,
        cidr_mask=24,

        # new IPv6 specific property
        ipv6_assign_address_on_creation=True
    )
    ]
)
```

The property `mapPublicIpOnLaunch` controls if a public IPv4 address will be assigned. This defaults to `false` for dual stack VPCs to avoid inadvertant costs of having the public address. However, a public IP must be enabled (or otherwise configured with BYOIP or IPAM) in order for services that rely on the address to function.

The `ipv6AssignAddressOnCreation` property controls the same behavior for the IPv6 address. It defaults to true.

Using IPv6 specific properties in an IPv4 only VPC will result in errors.

### Accessing the Internet Gateway

If you need access to the internet gateway, you can get its ID like so:

```python
# vpc: ec2.Vpc


igw_id = vpc.internet_gateway_id
```

For a VPC with only `ISOLATED` subnets, this value will be undefined.

This is only supported for VPCs created in the stack - currently you're
unable to get the ID for imported VPCs. To do that you'd have to specifically
look up the Internet Gateway by name, which would require knowing the name
beforehand.

This can be useful for configuring routing using a combination of gateways:
for more information see [Routing](#routing) below.

### Disabling the creation of the default internet gateway

If you need to control the creation of the internet gateway explicitly,
you can disable the creation of the default one using the `createInternetGateway`
property:

```python
vpc = ec2.Vpc(self, "VPC",
    create_internet_gateway=False,
    subnet_configuration=[ec2.SubnetConfiguration(
        subnet_type=ec2.SubnetType.PUBLIC,
        name="Public"
    )]
)
```

#### Routing

It's possible to add routes to any subnets using the `addRoute()` method. If for
example you want an isolated subnet to have a static route via the default
Internet Gateway created for the public subnet - perhaps for routing a VPN
connection - you can do so like this:

```python
vpc = ec2.Vpc(self, "VPC",
    subnet_configuration=[ec2.SubnetConfiguration(
        subnet_type=ec2.SubnetType.PUBLIC,
        name="Public"
    ), ec2.SubnetConfiguration(
        subnet_type=ec2.SubnetType.PRIVATE_ISOLATED,
        name="Isolated"
    )]
)

(vpc.isolated_subnets[0]).add_route("StaticRoute",
    router_id=vpc.internet_gateway_id,
    router_type=ec2.RouterType.GATEWAY,
    destination_cidr_block="8.8.8.8/32"
)
```

*Note that we cast to `Subnet` here because the list of subnets only returns an
`ISubnet`.*

### Reserving subnet IP space

There are situations where the IP space for a subnet or number of subnets
will need to be reserved. This is useful in situations where subnets would
need to be added after the vpc is originally deployed, without causing IP
renumbering for existing subnets. The IP space for a subnet may be reserved
by setting the `reserved` subnetConfiguration property to true, as shown
below:

```python
vpc = ec2.Vpc(self, "TheVPC",
    nat_gateways=1,
    subnet_configuration=[ec2.SubnetConfiguration(
        cidr_mask=26,
        name="Public",
        subnet_type=ec2.SubnetType.PUBLIC
    ), ec2.SubnetConfiguration(
        cidr_mask=26,
        name="Application1",
        subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS
    ), ec2.SubnetConfiguration(
        cidr_mask=26,
        name="Application2",
        subnet_type=ec2.SubnetType.PRIVATE_WITH_EGRESS,
        reserved=True
    ), ec2.SubnetConfiguration(
        cidr_mask=27,
        name="Database",
        subnet_type=ec2.SubnetType.PRIVATE_ISOLATED
    )
    ]
)
```

In the example above, the subnet for Application2 is not actually provisioned
but its IP space is still reserved. If in the future this subnet needs to be
provisioned, then the `reserved: true` property should be removed. Reserving
parts of the IP space prevents the other subnets from getting renumbered.

### Sharing VPCs between stacks

If you are creating multiple `Stack`s inside the same CDK application, you
can reuse a VPC defined in one Stack in another by simply passing the VPC
instance around:

```python
#
# Stack1 creates the VPC
#
class Stack1(cdk.Stack):

    def __init__(self, scope, id, *, description=None, env=None, stackName=None, tags=None, notificationArns=None, synthesizer=None, terminationProtection=None, analyticsReporting=None, crossRegionReferences=None, permissionsBoundary=None, suppressTemplateIndentation=None):
        super().__init__(scope, id, description=description, env=env, stackName=stackName, tags=tags, notificationArns=notificationArns, synthesizer=synthesizer, terminationProtection=terminationProtection, analyticsReporting=analyticsReporting, crossRegionReferences=crossRegionReferences, permissionsBoundary=permissionsBoundary, suppressTemplateIndentation=suppressTemplateIndentation)

        self.vpc = ec2.Vpc(self, "VPC")

#
# Stack2 consumes the VPC
#
class Stack2(cdk.Stack):
    def __init__(self, scope, id, *, vpc, description=None, env=None, stackName=None, tags=None, notificationArns=None, synthesizer=None, terminationProtection=None, analyticsReporting=None, crossRegionReferences=None, permissionsBoundary=None, suppressTemplateIndentation=None):
        super().__init__(scope, id, vpc=vpc, description=description, env=env, stackName=stackName, tags=tags, notificationArns=notificationArns, synthesizer=synthesizer, terminationProtection=terminationProtection, analyticsReporting=analyticsReporting, crossRegionReferences=crossRegionReferences, permissionsBoundary=permissionsBoundary, suppressTemplateIndentation=suppressTemplateIndentation)

        # Pass the VPC to a construct that needs it
        ConstructThatTakesAVpc(self, "Construct",
            vpc=vpc
        )

stack1 = Stack1(app, "Stack1")
stack2 = Stack2(app, "Stack2",
    vpc=stack1.vpc
)
```

### Importing an existing VPC

If your VPC is created outside your CDK app, you can use `Vpc.fromLookup()`.
The CDK CLI will search for the specified VPC in the the stack's region and
account, and import the subnet configuration. Looking up can be done by VPC
ID, but more flexibly by searching for a specific tag on the VPC.

Subnet types will be determined from the `aws-cdk:subnet-type` tag on the
subnet if it exists, or the presence of a route to an Internet Gateway
otherwise. Subnet names will be determined from the `aws-cdk:subnet-name` tag
on the subnet if it exists, or will mirror the subnet type otherwise (i.e.
a public subnet will have the name `"Public"`).

The result of the `Vpc.fromLookup()` operation will be written to a file
called `cdk.context.json`. You must commit this file to source control so
that the lookup values are available in non-privileged environments such
as CI build steps, and to ensure your template builds are repeatable.

Here's how `Vpc.fromLookup()` can be used:

```python
vpc = ec2.Vpc.from_lookup(stack, "VPC",
    # This imports the default VPC but you can also
    # specify a 'vpcName' or 'tags'.
    is_default=True
)
```

`Vpc.fromLookup` is the recommended way to import VPCs. If for whatever
reason you do not want to use the context mechanism to look up a VPC at
synthesis time, you can also use `Vpc.fromVpcAttributes`. This has the
following limitations:

* Every subnet group in the VPC must have a subnet in each availability zone
  (for example, each AZ must have both a public and private subnet). Asymmetric
  VPCs are not supported.
* All VpcId, SubnetId, RouteTableId, ... parameters must either be known at
  synthesis time, or they must come from deploy-time list parameters whose
  deploy-time lengths are known at synthesis time.

Using `Vpc.fromVpcAttributes()` looks like this:

```python
vpc = ec2.Vpc.from_vpc_attributes(self, "VPC",
    vpc_id="vpc-1234",
    availability_zones=["us-east-1a", "us-east-1b"],

    # Either pass literals for all IDs
    public_subnet_ids=["s-12345", "s-67890"],

    # OR: import a list of known length
    private_subnet_ids=Fn.import_list_value("PrivateSubnetIds", 2),

    # OR: split an imported string to a list of known length
    isolated_subnet_ids=Fn.split(",", ssm.StringParameter.value_for_string_parameter(self, "MyParameter"), 2)
)
```

For each subnet group the import function accepts optional parameters for subnet
names, route table ids and IPv4 CIDR blocks. When supplied, the length of these
lists are required to match the length of the list of subnet ids, allowing the
lists to be zipped together to form `ISubnet` instances.

Public subnet group example (for private or isolated subnet groups, use the properties with the respective prefix):

```python
vpc = ec2.Vpc.from_vpc_attributes(self, "VPC",
    vpc_id="vpc-1234",
    availability_zones=["us-east-1a", "us-east-1b", "us-east-1c"],
    public_subnet_ids=["s-12345", "s-34567", "s-56789"],
    public_subnet_names=["Subnet A", "Subnet B", "Subnet C"],
    public_subnet_route_table_ids=["rt-12345", "rt-34567", "rt-56789"],
    public_subnet_ipv4_cidr_blocks=["10.0.0.0/24", "10.0.1.0/24", "10.0.2.0/24"]
)
```

The above example will create an `IVpc` instance with three public subnets:

| Subnet id | Availability zone | Subnet name | Route table id | IPv4 CIDR   |
| --------- | ----------------- | ----------- | -------------- | ----------- |
| s-12345   | us-east-1a        | Subnet A    | rt-12345       | 10.0.0.0/24 |
| s-34567   | us-east-1b        | Subnet B    | rt-34567       | 10.0.1.0/24 |
| s-56789   | us-east-1c        | Subnet B    | rt-56789       | 10.0.2.0/24 |

### Restricting access to the VPC default security group

AWS Security best practices recommend that the [VPC default security group should
not allow inbound and outbound
traffic](https://docs.aws.amazon.com/securityhub/latest/userguide/ec2-controls.html#ec2-2).
When the `@aws-cdk/aws-ec2:restrictDefaultSecurityGroup` feature flag is set to
`true` (default for new projects) this will be enabled by default. If you do not
have this feature flag set you can either set the feature flag *or* you can set
the `restrictDefaultSecurityGroup` property to `true`.

```python
ec2.Vpc(self, "VPC",
    restrict_default_security_group=True
)
```

If you set this property to `true` and then later remove it or set it to `false`
the default ingress/egress will be restored on the default security group.

## Allowing Connections

In AWS, all network traffic in and out of **Elastic Network Interfaces** (ENIs)
is controlled by **Security Groups**. You can think of Security Groups as a
firewall with a set of rules. By default, Security Groups allow no incoming
(ingress) traffic and all outgoing (egress) traffic. You can add ingress rules
to them to allow incoming traffic streams. To exert fine-grained control over
egress traffic, set `allowAllOutbound: false` on the `SecurityGroup`, after
which you can add egress traffic rules.

You can manipulate Security Groups directly:

```python
my_security_group = ec2.SecurityGroup(self, "SecurityGroup",
    vpc=vpc,
    description="Allow ssh access to ec2 instances",
    allow_all_outbound=True
)
my_security_group.add_ingress_rule(ec2.Peer.any_ipv4(), ec2.Port.tcp(22), "allow ssh access from the world")
```

All constructs that create ENIs on your behalf (typically constructs that create
EC2 instances or other VPC-connected resources) will all have security groups
automatically assigned. Those constructs have an attribute called
**connections**, which is an object that makes it convenient to update the
security groups. If you want to allow connections between two constructs that
have security groups, you have to add an **Egress** rule to one Security Group,
and an **Ingress** rule to the other. The connections object will automatically
take care of this for you:

```python
# load_balancer: elbv2.ApplicationLoadBalancer
# app_fleet: autoscaling.AutoScalingGroup
# db_fleet: autoscaling.AutoScalingGroup


# Allow connections from anywhere
load_balancer.connections.allow_from_any_ipv4(ec2.Port.HTTPS, "Allow inbound HTTPS")

# The same, but an explicit IP address
load_balancer.connections.allow_from(ec2.Peer.ipv4("1.2.3.4/32"), ec2.Port.HTTPS, "Allow inbound HTTPS")

# Allow connection between AutoScalingGroups
app_fleet.connections.allow_to(db_fleet, ec2.Port.HTTPS, "App can call database")
```

### Connection Peers

There are various classes that implement the connection peer part:

```python
# app_fleet: autoscaling.AutoScalingGroup
# db_fleet: autoscaling.AutoScalingGroup


# Simple connection peers
peer = ec2.Peer.ipv4("10.0.0.0/16")
peer = ec2.Peer.any_ipv4()
peer = ec2.Peer.ipv6("::0/0")
peer = ec2.Peer.any_ipv6()
peer = ec2.Peer.prefix_list("pl-12345")
app_fleet.connections.allow_to(peer, ec2.Port.HTTPS, "Allow outbound HTTPS")
```

Any object that has a security group can itself be used as a connection peer:

```python
# fleet1: autoscaling.AutoScalingGroup
# fleet2: autoscaling.AutoScalingGroup
# app_fleet: autoscaling.AutoScalingGroup


# These automatically create appropriate ingress and egress rules in both security groups
fleet1.connections.allow_to(fleet2, ec2.Port.HTTP, "Allow between fleets")

app_fleet.connections.allow_from_any_ipv4(ec2.Port.HTTP, "Allow from load balancer")
```

### Port Ranges

The connections that are allowed are specified by port ranges. A number of classes provide
the connection specifier:

```python
ec2.Port.tcp(80)
ec2.Port.HTTPS
ec2.Port.tcp_range(60000, 65535)
ec2.Port.all_tcp()
ec2.Port.all_icmp()
ec2.Port.all_icmp_v6()
ec2.Port.all_traffic()
```

> NOTE: Not all protocols have corresponding helper methods. In the absence of a helper method,
> you can instantiate `Port` yourself with your own settings. You are also welcome to contribute
> new helper methods.

### Default Ports

Some Constructs have default ports associated with them. For example, the
listener of a load balancer does (it's the public port), or instances of an
RDS database (it's the port the database is accepting connections on).

If the object you're calling the peering method on has a default port associated with it, you can call
`allowDefaultPortFrom()` and omit the port specifier. If the argument has an associated default port, call
`allowDefaultPortTo()`.

For example:

```python
# listener: elbv2.ApplicationListener
# app_fleet: autoscaling.AutoScalingGroup
# rds_database: rds.DatabaseCluster


# Port implicit in listener
listener.connections.allow_default_port_from_any_ipv4("Allow public")

# Port implicit in peer
app_fleet.connections.allow_default_port_to(rds_database, "Fleet can access database")
```

### Security group rules

By default, security group wills be added inline to the security group in the output cloud formation
template, if applicable.  This includes any static rules by ip address and port range.  This
optimization helps to minimize the size of the template.

In some environments this is not desirable, for example if your security group access is controlled
via tags. You can disable inline rules per security group or globally via the context key
`@aws-cdk/aws-ec2.securityGroupDisableInlineRules`.

```python
my_security_group_without_inline_rules = ec2.SecurityGroup(self, "SecurityGroup",
    vpc=vpc,
    description="Allow ssh access to ec2 instances",
    allow_all_outbound=True,
    disable_inline_rules=True
)
# This will add the rule as an external cloud formation construct
my_security_group_without_inline_rules.add_ingress_rule(ec2.Peer.any_ipv4(), ec2.Port.SSH, "allow ssh access from the world")
```

### Importing an existing security group

If you know the ID and the configuration of the security group to import, you can use `SecurityGroup.fromSecurityGroupId`:

```python
sg = ec2.SecurityGroup.from_security_group_id(self, "SecurityGroupImport", "sg-1234",
    allow_all_outbound=True
)
```

Alternatively, use lookup methods to import security groups if you do not know the ID or the configuration details. Method `SecurityGroup.fromLookupByName` looks up a security group if the security group ID is unknown.

```python
sg = ec2.SecurityGroup.from_lookup_by_name(self, "SecurityGroupLookup", "security-group-name", vpc)
```

If the security group ID is known and configuration details are unknown, use method `SecurityGroup.fromLookupById` instead. This method will lookup property `allowAllOutbound` from the current configuration of the security group.

```python
sg = ec2.SecurityGroup.from_lookup_by_id(self, "SecurityGroupLookup", "sg-1234")
```

The result of `SecurityGroup.fromLookupByName` and `SecurityGroup.fromLookupById` operations will be written to a file called `cdk.context.json`. You must commit this file to source control so that the lookup values are available in non-privileged environments such as CI build steps, and to ensure your template builds are repeatable.

### Cross Stack Connections

If you are attempting to add a connection from a peer in one stack to a peer in a different stack, sometimes it is necessary to ensure that you are making the connection in
a specific stack in order to avoid a cyclic reference. If there are no other dependencies between stacks then it will not matter in which stack you make
the connection, but if there are existing dependencies (i.e. stack1 already depends on stack2), then it is important to make the connection in the dependent stack (i.e. stack1).

Whenever you make a `connections` function call, the ingress and egress security group rules will be added to the stack that the calling object exists in.
So if you are doing something like `peer1.connections.allowFrom(peer2)`, then the security group rules (both ingress and egress) will be created in `peer1`'s Stack.

As an example, if we wanted to allow a connection from a security group in one stack (egress) to a security group in a different stack (ingress),
we would make the connection like:

**If Stack1 depends on Stack2**

```python
# Stack 1
# stack1: Stack
# stack2: Stack


sg1 = ec2.SecurityGroup(stack1, "SG1",
    allow_all_outbound=False,  # if this is `true` then no egress rule will be created
    vpc=vpc
)

# Stack 2
sg2 = ec2.SecurityGroup(stack2, "SG2",
    allow_all_outbound=False,  # if this is `true` then no egress rule will be created
    vpc=vpc
)

# `connections.allowTo` on `sg1` since we want the
# rules to be created in Stack1
sg1.connections.allow_to(sg2, ec2.Port.tcp(3333))
```

In this case both the Ingress Rule for `sg2` and the Egress Rule for `sg1` will both be created
in `Stack 1` which avoids the cyclic reference.

**If Stack2 depends on Stack1**

```python
# Stack 1
# stack1: Stack
# stack2: Stack


sg1 = ec2.SecurityGroup(stack1, "SG1",
    allow_all_outbound=False,  # if this is `true` then no egress rule will be created
    vpc=vpc
)

# Stack 2
sg2 = ec2.SecurityGroup(stack2, "SG2",
    allow_all_outbound=False,  # if this is `true` then no egress rule will be created
    vpc=vpc
)

# `connections.allowFrom` on `sg2` since we want the
# rules to be created in Stack2
sg2.connections.allow_from(sg1, ec2.Port.tcp(3333))
```

In this case both the Ingress Rule for `sg2` and the Egress Rule for `sg1` will both be created
in `Stack 2` which avoids the cyclic reference.

## Machine Images (AMIs)

AMIs control the OS that gets launched when you start your EC2 instance. The EC2
library contains constructs to select the AMI you want to use.

Depending on the type of AMI, you select it a different way. Here are some
examples of images you might want to use:

```python
# Pick the right Amazon Linux edition. All arguments shown are optional
# and will default to these values when omitted.
amzn_linux = ec2.MachineImage.latest_amazon_linux(
    generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
    edition=ec2.AmazonLinuxEdition.STANDARD,
    virtualization=ec2.AmazonLinuxVirt.HVM,
    storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
    cpu_type=ec2.AmazonLinuxCpuType.X86_64
)

# Pick a Windows edition to use
windows = ec2.MachineImage.latest_windows(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)

# Read AMI id from SSM parameter store
ssm = ec2.MachineImage.from_ssm_parameter("/my/ami", os=ec2.OperatingSystemType.LINUX)

# Look up the most recent image matching a set of AMI filters.
# In this case, look up the NAT instance AMI, by using a wildcard
# in the 'name' field:
nat_ami = ec2.MachineImage.lookup(
    name="amzn-ami-vpc-nat-*",
    owners=["amazon"]
)

# For other custom (Linux) images, instantiate a `GenericLinuxImage` with
# a map giving the AMI to in for each region:
linux = ec2.MachineImage.generic_linux({
    "us-east-1": "ami-97785bed",
    "eu-west-1": "ami-12345678"
})

# For other custom (Windows) images, instantiate a `GenericWindowsImage` with
# a map giving the AMI to in for each region:
generic_windows = ec2.MachineImage.generic_windows({
    "us-east-1": "ami-97785bed",
    "eu-west-1": "ami-12345678"
})
```

> NOTE: The AMIs selected by `MachineImage.lookup()` will be cached in
> `cdk.context.json`, so that your AutoScalingGroup instances aren't replaced while
> you are making unrelated changes to your CDK app.
>
> To query for the latest AMI again, remove the relevant cache entry from
> `cdk.context.json`, or use the `cdk context` command. For more information, see
> [Runtime Context](https://docs.aws.amazon.com/cdk/latest/guide/context.html) in the CDK
> developer guide.
>
> `MachineImage.genericLinux()`, `MachineImage.genericWindows()` will use `CfnMapping` in
> an agnostic stack.

## Special VPC configurations

### VPN connections to a VPC

Create your VPC with VPN connections by specifying the `vpnConnections` props (keys are construct `id`s):

```python
from aws_cdk import SecretValue


vpc = ec2.Vpc(self, "MyVpc",
    vpn_connections={
        "dynamic": ec2.VpnConnectionOptions( # Dynamic routing (BGP)
            ip="1.2.3.4",
            tunnel_options=[ec2.VpnTunnelOption(
                pre_shared_key_secret=SecretValue.unsafe_plain_text("secretkey1234")
            ), ec2.VpnTunnelOption(
                pre_shared_key_secret=SecretValue.unsafe_plain_text("secretkey5678")
            )
            ]),
        "static": ec2.VpnConnectionOptions( # Static routing
            ip="4.5.6.7",
            static_routes=["192.168.10.0/24", "192.168.20.0/24"
            ])
    }
)
```

To create a VPC that can accept VPN connections, set `vpnGateway` to `true`:

```python
vpc = ec2.Vpc(self, "MyVpc",
    vpn_gateway=True
)
```

VPN connections can then be added:

```python
vpc.add_vpn_connection("Dynamic",
    ip="1.2.3.4"
)
```

By default, routes will be propagated on the route tables associated with the private subnets. If no
private subnets exist, isolated subnets are used. If no isolated subnets exist, public subnets are
used. Use the `Vpc` property `vpnRoutePropagation` to customize this behavior.

VPN connections expose [metrics (cloudwatch.Metric)](https://github.com/aws/aws-cdk/blob/main/packages/aws-cdk-lib/aws-cloudwatch/README.md) across all tunnels in the account/region and per connection:

```python
# Across all tunnels in the account/region
all_data_out = ec2.VpnConnection.metric_all_tunnel_data_out()

# For a specific vpn connection
vpn_connection = vpc.add_vpn_connection("Dynamic",
    ip="1.2.3.4"
)
state = vpn_connection.metric_tunnel_state()
```

### VPC endpoints

A VPC endpoint enables you to privately connect your VPC to supported AWS services and VPC endpoint services powered by PrivateLink without requiring an internet gateway, NAT device, VPN connection, or AWS Direct Connect connection. Instances in your VPC do not require public IP addresses to communicate with resources in the service. Traffic between your VPC and the other service does not leave the Amazon network.

Endpoints are virtual devices. They are horizontally scaled, redundant, and highly available VPC components that allow communication between instances in your VPC and services without imposing availability risks or bandwidth constraints on your network traffic.

```python
# Add gateway endpoints when creating the VPC
vpc = ec2.Vpc(self, "MyVpc",
    gateway_endpoints={
        "S3": cdk.aws_ec2.GatewayVpcEndpointOptions(
            service=ec2.GatewayVpcEndpointAwsService.S3
        )
    }
)

# Alternatively gateway endpoints can be added on the VPC
dynamo_db_endpoint = vpc.add_gateway_endpoint("DynamoDbEndpoint",
    service=ec2.GatewayVpcEndpointAwsService.DYNAMODB
)

# This allows to customize the endpoint policy
dynamo_db_endpoint.add_to_policy(
    iam.PolicyStatement( # Restrict to listing and describing tables
        principals=[iam.AnyPrincipal()],
        actions=["dynamodb:DescribeTable", "dynamodb:ListTables"],
        resources=["*"]))

# Add an interface endpoint
vpc.add_interface_endpoint("EcrDockerEndpoint",
    service=ec2.InterfaceVpcEndpointAwsService.ECR_DOCKER
)
```

By default, CDK will place a VPC endpoint in one subnet per AZ. If you wish to override the AZs CDK places the VPC endpoint in,
use the `subnets` parameter as follows:

```python
# vpc: ec2.Vpc


ec2.InterfaceVpcEndpoint(self, "VPC Endpoint",
    vpc=vpc,
    service=ec2.InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    # Choose which availability zones to place the VPC endpoint in, based on
    # available AZs
    subnets=ec2.SubnetSelection(
        availability_zones=["us-east-1a", "us-east-1c"]
    )
)
```

Per the [AWS documentation](https://aws.amazon.com/premiumsupport/knowledge-center/interface-endpoint-availability-zone/), not all
VPC endpoint services are available in all AZs. If you specify the parameter `lookupSupportedAzs`, CDK attempts to discover which
AZs an endpoint service is available in, and will ensure the VPC endpoint is not placed in a subnet that doesn't match those AZs.
These AZs will be stored in cdk.context.json.

```python
# vpc: ec2.Vpc


ec2.InterfaceVpcEndpoint(self, "VPC Endpoint",
    vpc=vpc,
    service=ec2.InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    # Choose which availability zones to place the VPC endpoint in, based on
    # available AZs
    lookup_supported_azs=True
)
```

Pre-defined AWS services are defined in the [InterfaceVpcEndpointAwsService](lib/vpc-endpoint.ts) class, and can be used to
create VPC endpoints without having to configure name, ports, etc. For example, a Keyspaces endpoint can be created for
use in your VPC:

```python
# vpc: ec2.Vpc


ec2.InterfaceVpcEndpoint(self, "VPC Endpoint",
    vpc=vpc,
    service=ec2.InterfaceVpcEndpointAwsService.KEYSPACES
)
```

#### Security groups for interface VPC endpoints

By default, interface VPC endpoints create a new security group and all traffic to the endpoint from within the VPC will be automatically allowed.

Use the `connections` object to allow other traffic to flow to the endpoint:

```python
# my_endpoint: ec2.InterfaceVpcEndpoint


my_endpoint.connections.allow_default_port_from_any_ipv4()
```

Alternatively, existing security groups can be used by specifying the `securityGroups` prop.

### VPC endpoint services

A VPC endpoint service enables you to expose a Network Load Balancer(s) as a provider service to consumers, who connect to your service over a VPC endpoint. You can restrict access to your service via allowed principals (anything that extends ArnPrincipal), and require that new connections be manually accepted. You can also enable Contributor Insight rules.

```python
# network_load_balancer1: elbv2.NetworkLoadBalancer
# network_load_balancer2: elbv2.NetworkLoadBalancer


ec2.VpcEndpointService(self, "EndpointService",
    vpc_endpoint_service_load_balancers=[network_load_balancer1, network_load_balancer2],
    acceptance_required=True,
    allowed_principals=[iam.ArnPrincipal("arn:aws:iam::123456789012:root")],
    contributor_insights=True
)
```

You can also include a service principal in the `allowedPrincipals` property by specifying it as a parameter to the  `ArnPrincipal` constructor.
The resulting VPC endpoint will have an allowlisted principal of type `Service`, instead of `Arn` for that item in the list.

```python
# network_load_balancer: elbv2.NetworkLoadBalancer


ec2.VpcEndpointService(self, "EndpointService",
    vpc_endpoint_service_load_balancers=[network_load_balancer],
    allowed_principals=[iam.ArnPrincipal("ec2.amazonaws.com")]
)
```

You can specify which IP address types (IPv4, IPv6, or both) are supported for your VPC endpoint service:

```python
# network_load_balancer: elbv2.NetworkLoadBalancer


ec2.VpcEndpointService(self, "EndpointService",
    vpc_endpoint_service_load_balancers=[network_load_balancer],
    # Support both IPv4 and IPv6 connections to the endpoint service
    supported_ip_address_types=[ec2.IpAddressType.IPV4, ec2.IpAddressType.IPV6
    ]
)
```

You can restrict access to your endpoint service to specific AWS regions:

```python
# network_load_balancer: elbv2.NetworkLoadBalancer


ec2.VpcEndpointService(self, "EndpointService",
    vpc_endpoint_service_load_balancers=[network_load_balancer],
    # Allow service consumers from these regions only
    allowed_regions=["us-east-1", "eu-west-1"]
)
```

Endpoint services support private DNS, which makes it easier for clients to connect to your service by automatically setting up DNS in their VPC.
You can enable private DNS on an endpoint service like so:

```python
from aws_cdk.aws_route53 import PublicHostedZone, VpcEndpointServiceDomainName
# zone: PublicHostedZone
# vpces: ec2.VpcEndpointService


VpcEndpointServiceDomainName(self, "EndpointDomain",
    endpoint_service=vpces,
    domain_name="my-stuff.aws-cdk.dev",
    public_hosted_zone=zone
)
```

Note: The domain name must be owned (registered through Route53) by the account the endpoint service is in, or delegated to the account.
The VpcEndpointServiceDomainName will handle the AWS side of domain verification, the process for which can be found
[here](https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-dns-validation.html)

### Client VPN endpoint

AWS Client VPN is a managed client-based VPN service that enables you to securely access your AWS
resources and resources in your on-premises network. With Client VPN, you can access your resources
from any location using an OpenVPN-based VPN client.

Use the `addClientVpnEndpoint()` method to add a client VPN endpoint to a VPC:

```python
vpc.add_client_vpn_endpoint("Endpoint",
    cidr="10.100.0.0/16",
    server_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id",
    # Mutual authentication
    client_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/client-certificate-id",
    # User-based authentication
    user_based_authentication=ec2.ClientVpnUserBasedAuthentication.federated(saml_provider)
)
```

The endpoint must use at least one [authentication method](https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/client-authentication.html):

* Mutual authentication with a client certificate
* User-based authentication (directory or federated)

If user-based authentication is used, the [self-service portal URL](https://docs.aws.amazon.com/vpn/latest/clientvpn-user/self-service-portal.html)
is made available via a CloudFormation output.

By default, a new security group is created, and logging is enabled. Moreover, a rule to
authorize all users to the VPC CIDR is created.

To customize authorization rules, set the `authorizeAllUsersToVpcCidr` prop to `false`
and use `addAuthorizationRule()`:

```python
endpoint = vpc.add_client_vpn_endpoint("Endpoint",
    cidr="10.100.0.0/16",
    server_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id",
    user_based_authentication=ec2.ClientVpnUserBasedAuthentication.federated(saml_provider),
    authorize_all_users_to_vpc_cidr=False
)

endpoint.add_authorization_rule("Rule",
    cidr="10.0.10.0/32",
    group_id="group-id"
)
```

Use `addRoute()` to configure network routes:

```python
endpoint = vpc.add_client_vpn_endpoint("Endpoint",
    cidr="10.100.0.0/16",
    server_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id",
    user_based_authentication=ec2.ClientVpnUserBasedAuthentication.federated(saml_provider)
)

# Client-to-client access
endpoint.add_route("Route",
    cidr="10.100.0.0/16",
    target=ec2.ClientVpnRouteTarget.local()
)
```

Use the `connections` object of the endpoint to allow traffic to other security groups.

## Instances

You can use the `Instance` class to start up a single EC2 instance. For production setups, we recommend
you use an `AutoScalingGroup` from the `aws-autoscaling` module instead, as AutoScalingGroups will take
care of restarting your instance if it ever fails.

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType


# Amazon Linux 2
ec2.Instance(self, "Instance2",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=ec2.MachineImage.latest_amazon_linux2()
)

# Amazon Linux 2 with kernel 5.x
ec2.Instance(self, "Instance3",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=ec2.MachineImage.latest_amazon_linux2(
        kernel=ec2.AmazonLinux2Kernel.KERNEL_5_10
    )
)

# Amazon Linux 2023
ec2.Instance(self, "Instance4",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=ec2.MachineImage.latest_amazon_linux2023()
)

# Graviton 3 Processor
ec2.Instance(self, "Instance5",
    vpc=vpc,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.C7G, ec2.InstanceSize.LARGE),
    machine_image=ec2.MachineImage.latest_amazon_linux2023(
        cpu_type=ec2.AmazonLinuxCpuType.ARM_64
    )
)
```

### Latest Amazon Linux Images

Rather than specifying a specific AMI ID to use, it is possible to specify a SSM
Parameter that contains the AMI ID. AWS publishes a set of [public parameters](https://docs.aws.amazon.com/systems-manager/latest/userguide/parameter-store-public-parameters-ami.html)
that contain the latest Amazon Linux AMIs. To make it easier to query a
particular image parameter, the CDK provides a couple of constructs `AmazonLinux2ImageSsmParameter`,
`AmazonLinux2022ImageSsmParameter`, & `AmazonLinux2023SsmParameter`. For example
to use the latest `al2023` image:

```python
# vpc: ec2.Vpc


ec2.Instance(self, "LatestAl2023",
    vpc=vpc,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.C7G, ec2.InstanceSize.LARGE),
    machine_image=ec2.MachineImage.latest_amazon_linux2023()
)
```

> **Warning**
> Since this retrieves the value from an SSM parameter at deployment time, the
> value will be resolved each time the stack is deployed. This means that if
> the parameter contains a different value on your next deployment, the instance
> will be replaced.

It is also possible to perform the lookup once at synthesis time and then cache
the value in CDK context. This way the value will not change on future
deployments unless you manually refresh the context.

```python
# vpc: ec2.Vpc


ec2.Instance(self, "LatestAl2023",
    vpc=vpc,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.C7G, ec2.InstanceSize.LARGE),
    machine_image=ec2.MachineImage.latest_amazon_linux2023(
        cached_in_context=True
    )
)

# or
ec2.Instance(self, "LatestAl2023",
    vpc=vpc,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.C7G, ec2.InstanceSize.LARGE),
    # context cache is turned on by default
    machine_image=ec2.AmazonLinux2023ImageSsmParameter()
)
```

#### Kernel Versions

Each Amazon Linux AMI uses a specific kernel version. Most Amazon Linux
generations come with an AMI using the "default" kernel and then 1 or more
AMIs using a specific kernel version, which may or may not be different from the
default kernel version.

For example, Amazon Linux 2 has two different AMIs available from the SSM
parameters.

* `/aws/service/ami-amazon-linux-latest/amzn2-ami-hvm-x86_64-ebs`

  * This is the "default" kernel which uses `kernel-4.14`
* `/aws/service/ami-amazon-linux-latest/amzn2-ami-kernel-5.10-hvm-x86_64-ebs`

If a new Amazon Linux generation AMI is published with a new kernel version,
then a new SSM parameter will be created with the new version
(e.g. `/aws/service/ami-amazon-linux-latest/amzn2-ami-kernel-5.15-hvm-x86_64-ebs`),
but the "default" AMI may or may not be updated.

If you would like to make sure you always have the latest kernel version, then
either specify the specific latest kernel version or opt-in to using the CDK
latest kernel version.

```python
# vpc: ec2.Vpc


ec2.Instance(self, "LatestAl2023",
    vpc=vpc,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.C7G, ec2.InstanceSize.LARGE),
    # context cache is turned on by default
    machine_image=ec2.AmazonLinux2023ImageSsmParameter(
        kernel=ec2.AmazonLinux2023Kernel.KERNEL_6_1
    )
)
```

*CDK managed latest*

```python
# vpc: ec2.Vpc


ec2.Instance(self, "LatestAl2023",
    vpc=vpc,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.C7G, ec2.InstanceSize.LARGE),
    # context cache is turned on by default
    machine_image=ec2.AmazonLinux2023ImageSsmParameter(
        kernel=ec2.AmazonLinux2023Kernel.CDK_LATEST
    )
)

# or

ec2.Instance(self, "LatestAl2023",
    vpc=vpc,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.C7G, ec2.InstanceSize.LARGE),
    machine_image=ec2.MachineImage.latest_amazon_linux2023()
)
```

When using the CDK managed latest version, when a new kernel version is made
available the `LATEST` will be updated to point to the new kernel version. You
then would be required to update the newest CDK version for it to take effect.

### Configuring Instances using CloudFormation Init (cfn-init)

CloudFormation Init allows you to configure your instances by writing files to them, installing software
packages, starting services and running arbitrary commands. By default, if any of the instance setup
commands throw an error; the deployment will fail and roll back to the previously known good state.
The following documentation also applies to `AutoScalingGroup`s.

For the full set of capabilities of this system, see the documentation for
[`AWS::CloudFormation::Init`](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html).
Here is an example of applying some configuration to an instance:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # Showing the most complex setup, if you have simpler requirements
    # you can use `CloudFormationInit.fromElements()`.
    init=ec2.CloudFormationInit.from_config_sets(
        config_sets={
            # Applies the configs below in this order
            "default": ["yumPreinstall", "config"]
        },
        configs={
            "yum_preinstall": ec2.InitConfig([
                # Install an Amazon Linux package using yum
                ec2.InitPackage.yum("git")
            ]),
            "config": ec2.InitConfig([
                # Create a JSON file from tokens (can also create other files)
                ec2.InitFile.from_object("/etc/stack.json", {
                    "stack_id": Stack.of(self).stack_id,
                    "stack_name": Stack.of(self).stack_name,
                    "region": Stack.of(self).region
                }),

                # Create a group and user
                ec2.InitGroup.from_name("my-group"),
                ec2.InitUser.from_name("my-user"),

                # Install an RPM from the internet
                ec2.InitPackage.rpm("http://mirrors.ukfast.co.uk/sites/dl.fedoraproject.org/pub/epel/8/Everything/x86_64/Packages/r/rubygem-git-1.5.0-2.el8.noarch.rpm")
            ])
        }
    ),
    init_options=ec2.ApplyCloudFormationInitOptions(
        # Optional, which configsets to activate (['default'] by default)
        config_sets=["default"],

        # Optional, how long the installation is expected to take (5 minutes by default)
        timeout=Duration.minutes(30),

        # Optional, whether to include the --url argument when running cfn-init and cfn-signal commands (false by default)
        include_url=True,

        # Optional, whether to include the --role argument when running cfn-init and cfn-signal commands (false by default)
        include_role=True
    )
)
```

`InitCommand` can not be used to start long-running processes. At deploy time,
`cfn-init` will always wait for the process to exit before continuing, causing
the CloudFormation deployment to fail because the signal hasn't been received
within the expected timeout.

Instead, you should install a service configuration file onto your machine `InitFile`,
and then use `InitService` to start it.

If your Linux OS is using SystemD (like Amazon Linux 2 or higher), the CDK has
helpers to create a long-running service using CFN Init. You can create a
SystemD-compatible config file using `InitService.systemdConfigFile()`, and
start it immediately. The following examples shows how to start a trivial Python
3 web server:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType


ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=ec2.MachineImage.latest_amazon_linux2023(),

    init=ec2.CloudFormationInit.from_elements(
        # Create a simple config file that runs a Python web server
        ec2.InitService.systemd_config_file("simpleserver",
            command="/usr/bin/python3 -m http.server 8080",
            cwd="/var/www/html"
        ),
        # Start the server using SystemD
        ec2.InitService.enable("simpleserver",
            service_manager=ec2.ServiceManager.SYSTEMD
        ),
        # Drop an example file to show the web server working
        ec2.InitFile.from_string("/var/www/html/index.html", "Hello! It's working!"))
)
```

You can have services restarted after the init process has made changes to the system.
To do that, instantiate an `InitServiceRestartHandle` and pass it to the config elements
that need to trigger the restart and the service itself. For example, the following
config writes a config file for nginx, extracts an archive to the root directory, and then
restarts nginx so that it picks up the new config and files:

```python
# my_bucket: s3.Bucket


handle = ec2.InitServiceRestartHandle()

ec2.CloudFormationInit.from_elements(
    ec2.InitFile.from_string("/etc/nginx/nginx.conf", "...", service_restart_handles=[handle]),
    ec2.InitSource.from_s3_object("/var/www/html", my_bucket, "html.zip", service_restart_handles=[handle]),
    ec2.InitService.enable("nginx",
        service_restart_handle=handle
    ))
```

You can use the `environmentVariables` or `environmentFiles` parameters to specify environment variables
for your services:

```python
ec2.InitConfig([
    ec2.InitFile.from_string("/myvars.env", "VAR_FROM_FILE=\"VAR_FROM_FILE\""),
    ec2.InitService.systemd_config_file("myapp",
        command="/usr/bin/python3 -m http.server 8080",
        cwd="/var/www/html",
        environment_variables={
            "MY_VAR": "MY_VAR"
        },
        environment_files=["/myvars.env"]
    )
])
```

### Bastion Hosts

A bastion host functions as an instance used to access servers and resources in a VPC without open up the complete VPC on a network level.
You can use bastion hosts using a standard SSH connection targeting port 22 on the host. As an alternative, you can connect the SSH connection
feature of AWS Systems Manager Session Manager, which does not need an opened security group. (https://aws.amazon.com/about-aws/whats-new/2019/07/session-manager-launches-tunneling-support-for-ssh-and-scp/)

A default bastion host for use via SSM can be configured like:

```python
host = ec2.BastionHostLinux(self, "BastionHost", vpc=vpc)
```

If you want to connect from the internet using SSH, you need to place the host into a public subnet. You can then configure allowed source hosts.

```python
host = ec2.BastionHostLinux(self, "BastionHost",
    vpc=vpc,
    subnet_selection=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
)
host.allow_ssh_access_from(ec2.Peer.ipv4("1.2.3.4/32"))
```

As there are no SSH public keys deployed on this machine, you need to use [EC2 Instance Connect](https://aws.amazon.com/de/blogs/compute/new-using-amazon-ec2-instance-connect-for-ssh-access-to-your-ec2-instances/)
with the command `aws ec2-instance-connect send-ssh-public-key` to provide your SSH public key.

EBS volume for the bastion host can be encrypted like:

```python
host = ec2.BastionHostLinux(self, "BastionHost",
    vpc=vpc,
    block_devices=[ec2.BlockDevice(
        device_name="/dev/sdh",
        volume=ec2.BlockDeviceVolume.ebs(10,
            encrypted=True
        )
    )]
)
```

It's recommended to set the `@aws-cdk/aws-ec2:bastionHostUseAmazonLinux2023ByDefault`
[feature flag](https://docs.aws.amazon.com/cdk/v2/guide/featureflags.html) to `true` to use Amazon Linux 2023 as the
bastion host AMI. Without this flag set, the bastion host will default to Amazon Linux 2, which will be unsupported in
June 2025.

```json
{
  "context": {
    "@aws-cdk/aws-ec2:bastionHostUseAmazonLinux2023ByDefault": true
  }
}
```

### Placement Group

Specify `placementGroup` to enable the placement group support:

```python
# instance_type: ec2.InstanceType


pg = ec2.PlacementGroup(self, "test-pg",
    strategy=ec2.PlacementGroupStrategy.SPREAD
)

ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=ec2.MachineImage.latest_amazon_linux2023(),
    placement_group=pg
)
```

### Block Devices

To add EBS block device mappings, specify the `blockDevices` property. The following example sets the EBS-backed
root device (`/dev/sda1`) size to 50 GiB, and adds another EBS-backed device mapped to `/dev/sdm` that is 100 GiB in
size:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    block_devices=[ec2.BlockDevice(
        device_name="/dev/sda1",
        volume=ec2.BlockDeviceVolume.ebs(50)
    ), ec2.BlockDevice(
        device_name="/dev/sdm",
        volume=ec2.BlockDeviceVolume.ebs(100)
    )
    ]
)
```

It is also possible to encrypt the block devices. In this example we will create an customer managed key encrypted EBS-backed root device:

```python
from aws_cdk.aws_kms import Key

# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


kms_key = Key(self, "KmsKey")

ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    block_devices=[ec2.BlockDevice(
        device_name="/dev/sda1",
        volume=ec2.BlockDeviceVolume.ebs(50,
            encrypted=True,
            kms_key=kms_key
        )
    )
    ]
)
```

To specify the throughput value for `gp3` volumes, use the `throughput` property:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    block_devices=[ec2.BlockDevice(
        device_name="/dev/sda1",
        volume=ec2.BlockDeviceVolume.ebs(100,
            volume_type=ec2.EbsDeviceVolumeType.GP3,
            throughput=250
        )
    )
    ]
)
```

#### EBS Optimized Instances

An Amazon EBS–optimized instance uses an optimized configuration stack and provides additional, dedicated capacity for Amazon EBS I/O. This optimization provides the best performance for your EBS volumes by minimizing contention between Amazon EBS I/O and other traffic from your instance.

Depending on the instance type, this features is enabled by default while others require explicit activation. Please refer to the [documentation](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html) for details.

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,
    ebs_optimized=True,
    block_devices=[ec2.BlockDevice(
        device_name="/dev/xvda",
        volume=ec2.BlockDeviceVolume.ebs(8)
    )]
)
```

### Volumes

Whereas a `BlockDeviceVolume` is an EBS volume that is created and destroyed as part of the creation and destruction of a specific instance. A `Volume` is for when you want an EBS volume separate from any particular instance. A `Volume` is an EBS block device that can be attached to, or detached from, any instance at any time. Some types of `Volume`s can also be attached to multiple instances at the same time to allow you to have shared storage between those instances.

A notable restriction is that a Volume can only be attached to instances in the same availability zone as the Volume itself.

The following demonstrates how to create a 500 GiB encrypted Volume in the `us-west-2a` availability zone, and give a role the ability to attach that Volume to a specific instance:

```python
# instance: ec2.Instance
# role: iam.Role


volume = ec2.Volume(self, "Volume",
    availability_zone="us-west-2a",
    size=Size.gibibytes(500),
    encrypted=True
)

volume.grant_attach_volume(role, [instance])
```

#### Instances Attaching Volumes to Themselves

If you need to grant an instance the ability to attach/detach an EBS volume to/from itself, then using `grantAttachVolume` and `grantDetachVolume` as outlined above
will lead to an unresolvable circular reference between the instance role and the instance. In this case, use `grantAttachVolumeByResourceTag` and `grantDetachVolumeByResourceTag` as follows:

```python
# instance: ec2.Instance
# volume: ec2.Volume


attach_grant = volume.grant_attach_volume_by_resource_tag(instance.grant_principal, [instance])
detach_grant = volume.grant_detach_volume_by_resource_tag(instance.grant_principal, [instance])
```

#### Attaching Volumes

The Amazon EC2 documentation for
[Linux Instances](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AmazonEBS.html) and
[Windows Instances](https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ebs-volumes.html) contains information on how
to attach and detach your Volumes to/from instances, and how to format them for use.

The following is a sample skeleton of EC2 UserData that can be used to attach a Volume to the Linux instance that it is running on:

```python
# instance: ec2.Instance
# volume: ec2.Volume


volume.grant_attach_volume_by_resource_tag(instance.grant_principal, [instance])
target_device = "/dev/xvdz"
instance.user_data.add_commands("TOKEN=$(curl -SsfX PUT \"http://169.254.169.254/latest/api/token\" -H \"X-aws-ec2-metadata-token-ttl-seconds: 21600\")", "INSTANCE_ID=$(curl -SsfH \"X-aws-ec2-metadata-token: $TOKEN\" http://169.254.169.254/latest/meta-data/instance-id)", f"aws --region {Stack.of(this).region} ec2 attach-volume --volume-id {volume.volumeId} --instance-id $INSTANCE_ID --device {targetDevice}", f"while ! test -e {targetDevice}; do sleep 1; done")
```

#### Tagging Volumes

You can configure [tag propagation on volume creation](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-propagatetagstovolumeoncreation).

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


ec2.Instance(self, "Instance",
    vpc=vpc,
    machine_image=machine_image,
    instance_type=instance_type,
    propagate_tags_to_volume_on_creation=True
)
```

#### Throughput on GP3 Volumes

You can specify the `throughput` of a GP3 volume from 125 (default) to 1000.

```python
ec2.Volume(self, "Volume",
    availability_zone="us-east-1a",
    size=Size.gibibytes(125),
    volume_type=ec2.EbsDeviceVolumeType.GP3,
    throughput=125
)
```

### Configuring Instance Metadata Service (IMDS)

#### Toggling IMDSv1

You can configure [EC2 Instance Metadata Service](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html) options to either
allow both IMDSv1 and IMDSv2 or enforce IMDSv2 when interacting with the IMDS.

To do this for a single `Instance`, you can use the `requireImdsv2` property.
The example below demonstrates IMDSv2 being required on a single `Instance`:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType
# machine_image: ec2.IMachineImage


ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=machine_image,

    # ...

    require_imdsv2=True
)
```

You can also use the either the `InstanceRequireImdsv2Aspect` for EC2 instances or the `LaunchTemplateRequireImdsv2Aspect` for EC2 launch templates
to apply the operation to multiple instances or launch templates, respectively.

The following example demonstrates how to use the `InstanceRequireImdsv2Aspect` to require IMDSv2 for all EC2 instances in a stack:

```python
aspect = ec2.InstanceRequireImdsv2Aspect()
Aspects.of(self).add(aspect)
```

### Associating a Public IP Address with an Instance

All subnets have an attribute that determines whether instances launched into that subnet are assigned a public IPv4 address. This attribute is set to true by default for default public subnets. Thus, an EC2 instance launched into a default public subnet will be assigned a public IPv4 address. Nondefault public subnets have this attribute set to false by default and any EC2 instance launched into a nondefault public subnet will not be assigned a public IPv4 address automatically. To automatically assign a public IPv4 address to an instance launched into a nondefault public subnet, you can set the `associatePublicIpAddress` property on the `Instance` construct to true. Alternatively, to not automatically assign a public IPv4 address to an instance launched into a default public subnet, you can set `associatePublicIpAddress` to false. Including this property, removing this property, or updating the value of this property on an existing instance will result in replacement of the instance.

```python
vpc = ec2.Vpc(self, "VPC",
    cidr="10.0.0.0/16",
    nat_gateways=0,
    max_azs=3,
    subnet_configuration=[ec2.SubnetConfiguration(
        name="public-subnet-1",
        subnet_type=ec2.SubnetType.PUBLIC,
        cidr_mask=24
    )
    ]
)

instance = ec2.Instance(self, "Instance",
    vpc=vpc,
    vpc_subnets=ec2.SubnetSelection(subnet_group_name="public-subnet-1"),
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.NANO),
    machine_image=ec2.AmazonLinuxImage(generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX_2),
    detailed_monitoring=True,
    associate_public_ip_address=True
)
```

### Specifying a key pair

To allow SSH access to an EC2 instance by default, a Key Pair must be specified. Key pairs can
be provided with the `keyPair` property to instances and launch templates. You can create a
key pair for an instance like this:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType


key_pair = ec2.KeyPair(self, "KeyPair",
    type=ec2.KeyPairType.ED25519,
    format=ec2.KeyPairFormat.PEM
)
instance = ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=ec2.MachineImage.latest_amazon_linux2023(),
    # Use the custom key pair
    key_pair=key_pair
)
```

When a new EC2 Key Pair is created (without imported material), the private key material is
automatically stored in Systems Manager Parameter Store. This can be retrieved from the key pair
construct:

```python
key_pair = ec2.KeyPair(self, "KeyPair")
private_key = key_pair.private_key
```

If you already have an SSH key that you wish to use in EC2, that can be provided when constructing the
`KeyPair`. If public key material is provided, the key pair is considered "imported" and there
will not be any data automatically stored in Systems Manager Parameter Store and the `type` property
cannot be specified for the key pair.

```python
key_pair = ec2.KeyPair(self, "KeyPair",
    public_key_material="ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAIB7jpNzG+YG0s+xIGWbxrxIZiiozHOEuzIJacvASP0mq"
)
```

#### Using an existing EC2 Key Pair

If you already have an EC2 Key Pair created outside of the CDK, you can import that key to
your CDK stack.

You can import it purely by name:

```python
key_pair = ec2.KeyPair.from_key_pair_name(self, "KeyPair", "the-keypair-name")
```

Or by specifying additional attributes:

```python
key_pair = ec2.KeyPair.from_key_pair_attributes(self, "KeyPair",
    key_pair_name="the-keypair-name",
    type=ec2.KeyPairType.RSA
)
```

### Using IPv6 IPs

Instances can be given IPv6 IPs by launching them into a subnet of a dual stack VPC.

```python
vpc = ec2.Vpc(self, "Ip6VpcDualStack",
    ip_protocol=ec2.IpProtocol.DUAL_STACK,
    subnet_configuration=[ec2.SubnetConfiguration(
        name="Public",
        subnet_type=ec2.SubnetType.PUBLIC,
        map_public_ip_on_launch=True
    ), ec2.SubnetConfiguration(
        name="Private",
        subnet_type=ec2.SubnetType.PRIVATE_ISOLATED
    )
    ]
)

instance = ec2.Instance(self, "MyInstance",
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
    machine_image=ec2.MachineImage.latest_amazon_linux2(),
    vpc=vpc,
    vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC),
    allow_all_ipv6_outbound=True
)

instance.connections.allow_from(ec2.Peer.any_ipv6(), ec2.Port.all_icmp_v6(), "allow ICMPv6")
```

Note to set `mapPublicIpOnLaunch` to true in the `subnetConfiguration`.

Additionally, IPv6 support varies by instance type. Most instance types have IPv6 support with exception of m1-m3, c1, g2, and t1.micro. A full list can be found here: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html#AvailableIpPerENI.

#### Specifying the IPv6 Address

If you want to specify [the number of IPv6 addresses](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/MultipleIP.html#assign-multiple-ipv6) to assign to the instance, you can use the `ipv6AddresseCount` property:

```python
# dual stack VPC
# vpc: ec2.Vpc


instance = ec2.Instance(self, "MyInstance",
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE),
    machine_image=ec2.MachineImage.latest_amazon_linux2(),
    vpc=vpc,
    vpc_subnets=ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC),
    # Assign 2 IPv6 addresses to the instance
    ipv6_address_count=2
)
```

### Credit configuration modes for burstable instances

You can set the [credit configuration mode](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-credits-baseline-concepts.html) for burstable instances (T2, T3, T3a and T4g instance types):

```python
# vpc: ec2.Vpc


instance = ec2.Instance(self, "Instance",
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.MICRO),
    machine_image=ec2.MachineImage.latest_amazon_linux2(),
    vpc=vpc,
    credit_specification=ec2.CpuCredits.STANDARD
)
```

It is also possible to set the credit configuration mode for NAT instances.

```python
nat_instance_provider = ec2.NatProvider.instance(
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.T4G, ec2.InstanceSize.LARGE),
    machine_image=ec2.AmazonLinuxImage(),
    credit_specification=ec2.CpuCredits.UNLIMITED
)
ec2.Vpc(self, "VPC",
    nat_gateway_provider=nat_instance_provider
)
```

**Note**: `CpuCredits.UNLIMITED` mode is not supported for T3 instances that are launched on a Dedicated Host.

### Shutdown behavior

You can specify the behavior of the instance when you initiate shutdown from the instance (using the operating system command for system shutdown).

```python
# vpc: ec2.Vpc


ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.NANO),
    machine_image=ec2.AmazonLinuxImage(generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX_2),
    instance_initiated_shutdown_behavior=ec2.InstanceInitiatedShutdownBehavior.TERMINATE
)
```

### Enabling Nitro Enclaves

You can enable [AWS Nitro Enclaves](https://docs.aws.amazon.com/enclaves/latest/user/nitro-enclave.html) for
your EC2 instances by setting the `enclaveEnabled` property to `true`. Nitro Enclaves is a feature of
AWS Nitro System that enables creating isolated and highly constrained CPU environments known as enclaves.

```python
# vpc: ec2.Vpc


instance = ec2.Instance(self, "Instance",
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.XLARGE),
    machine_image=ec2.AmazonLinuxImage(),
    vpc=vpc,
    enclave_enabled=True
)
```

> NOTE: You must use an instance type and operating system that support Nitro Enclaves.
> For more information, see [Requirements](https://docs.aws.amazon.com/enclaves/latest/user/nitro-enclave.html#nitro-enclave-reqs).

### Enabling Termination Protection

You can enable [Termination Protection](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_ChangingDisableAPITermination.html) for
your EC2 instances by setting the `disableApiTermination` property to `true`. Termination Protection controls whether the instance can be terminated using the AWS Management Console, AWS Command Line Interface (AWS CLI), or API.

```python
# vpc: ec2.Vpc


instance = ec2.Instance(self, "Instance",
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.XLARGE),
    machine_image=ec2.AmazonLinuxImage(),
    vpc=vpc,
    disable_api_termination=True
)
```

### Enabling Instance Hibernation

You can enable [Instance Hibernation](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Hibernate.html) for
your EC2 instances by setting the `hibernationEnabled` property to `true`. Instance Hibernation saves the
instance's in-memory (RAM) state when an instance is stopped, and restores that state when the instance is started.

```python
# vpc: ec2.Vpc


instance = ec2.Instance(self, "Instance",
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.XLARGE),
    machine_image=ec2.AmazonLinuxImage(),
    vpc=vpc,
    hibernation_enabled=True,
    block_devices=[ec2.BlockDevice(
        device_name="/dev/xvda",
        volume=ec2.BlockDeviceVolume.ebs(30,
            volume_type=ec2.EbsDeviceVolumeType.GP3,
            encrypted=True,
            delete_on_termination=True
        )
    )]
)
```

> NOTE: You must use an instance and a volume that meet the requirements for hibernation.
> For more information, see [Prerequisites for Amazon EC2 instance hibernation](https://docs.aws.amazon.com/enclaves/latest/user/nitro-enclave.html#nitro-enclave-reqs).

## VPC Flow Logs

VPC Flow Logs is a feature that enables you to capture information about the IP traffic going to and from network interfaces in your VPC. Flow log data can be published to Amazon CloudWatch Logs and Amazon S3. After you've created a flow log, you can retrieve and view its data in the chosen destination. ([https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html](https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html)).

By default, a flow log will be created with CloudWatch Logs as the destination.

You can create a flow log like this:

```python
# vpc: ec2.Vpc


ec2.FlowLog(self, "FlowLog",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc)
)
```

Or you can add a Flow Log to a VPC by using the addFlowLog method like this:

```python
vpc = ec2.Vpc(self, "Vpc")

vpc.add_flow_log("FlowLog")
```

You can also add multiple flow logs with different destinations.

```python
vpc = ec2.Vpc(self, "Vpc")

vpc.add_flow_log("FlowLogS3",
    destination=ec2.FlowLogDestination.to_s3()
)

# Only reject traffic and interval every minute.
vpc.add_flow_log("FlowLogCloudWatch",
    traffic_type=ec2.FlowLogTrafficType.REJECT,
    max_aggregation_interval=ec2.FlowLogMaxAggregationInterval.ONE_MINUTE
)
```

To create a Transit Gateway flow log, you can use the `fromTransitGatewayId` method:

```python
# tgw: ec2.CfnTransitGateway


ec2.FlowLog(self, "TransitGatewayFlowLog",
    resource_type=ec2.FlowLogResourceType.from_transit_gateway_id(tgw.ref)
)
```

To create a Transit Gateway Attachment flow log, you can use the `fromTransitGatewayAttachmentId` method:

```python
# tgw_attachment: ec2.CfnTransitGatewayAttachment


ec2.FlowLog(self, "TransitGatewayAttachmentFlowLog",
    resource_type=ec2.FlowLogResourceType.from_transit_gateway_attachment_id(tgw_attachment.ref)
)
```

For flow logs targeting TransitGateway and TransitGatewayAttachment, specifying the `trafficType` is not possible.

### Custom Formatting

You can also custom format flow logs.

```python
vpc = ec2.Vpc(self, "Vpc")

vpc.add_flow_log("FlowLog",
    log_format=[ec2.LogFormat.DST_PORT, ec2.LogFormat.SRC_PORT
    ]
)

# If you just want to add a field to the default field
vpc.add_flow_log("FlowLog",
    log_format=[ec2.LogFormat.VERSION, ec2.LogFormat.ALL_DEFAULT_FIELDS
    ]
)

# If AWS CDK does not support the new fields
vpc.add_flow_log("FlowLog",
    log_format=[ec2.LogFormat.SRC_PORT,
        ec2.LogFormat.custom("${new-field}")
    ]
)
```

By default, the CDK will create the necessary resources for the destination. For the CloudWatch Logs destination
it will create a CloudWatch Logs Log Group as well as the IAM role with the necessary permissions to publish to
the log group. In the case of an S3 destination, it will create the S3 bucket.

If you want to customize any of the destination resources you can provide your own as part of the `destination`.

*CloudWatch Logs*

```python
# vpc: ec2.Vpc


log_group = logs.LogGroup(self, "MyCustomLogGroup")

role = iam.Role(self, "MyCustomRole",
    assumed_by=iam.ServicePrincipal("vpc-flow-logs.amazonaws.com")
)

ec2.FlowLog(self, "FlowLog",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc),
    destination=ec2.FlowLogDestination.to_cloud_watch_logs(log_group, role)
)
```

*S3*

```python
# vpc: ec2.Vpc


bucket = s3.Bucket(self, "MyCustomBucket")

ec2.FlowLog(self, "FlowLog",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc),
    destination=ec2.FlowLogDestination.to_s3(bucket)
)

ec2.FlowLog(self, "FlowLogWithKeyPrefix",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc),
    destination=ec2.FlowLogDestination.to_s3(bucket, "prefix/")
)
```

*Amazon Data Firehose*

```python
import aws_cdk.aws_kinesisfirehose as firehose

# vpc: ec2.Vpc
# delivery_stream: firehose.CfnDeliveryStream


vpc.add_flow_log("FlowLogsKinesisDataFirehose",
    destination=ec2.FlowLogDestination.to_kinesis_data_firehose_destination(delivery_stream.attr_arn)
)
```

When the S3 destination is configured, AWS will automatically create an S3 bucket policy
that allows the service to write logs to the bucket. This makes it impossible to later update
that bucket policy. To have CDK create the bucket policy so that future updates can be made,
the `@aws-cdk/aws-s3:createDefaultLoggingPolicy` [feature flag](https://docs.aws.amazon.com/cdk/v2/guide/featureflags.html) can be used. This can be set
in the `cdk.json` file.

```json
{
  "context": {
    "@aws-cdk/aws-s3:createDefaultLoggingPolicy": true
  }
}
```

## User Data

User data enables you to run a script when your instances start up.  In order to configure these scripts you can add commands directly to the script
or you can use the UserData's convenience functions to aid in the creation of your script.

A user data could be configured to run a script found in an asset through the following:

```python
from aws_cdk.aws_s3_assets import Asset

# instance: ec2.Instance


asset = Asset(self, "Asset",
    path="./configure.sh"
)

local_path = instance.user_data.add_s3_download_command(
    bucket=asset.bucket,
    bucket_key=asset.s3_object_key,
    region="us-east-1"
)
instance.user_data.add_execute_file_command(
    file_path=local_path,
    arguments="--verbose -y"
)
asset.grant_read(instance.role)
```

### Persisting user data

By default, EC2 UserData is run once on only the first time that an instance is started. It is possible to make the
user data script run on every start of the instance.

When creating a Windows UserData you can use the `persist` option to set whether or not to add
`<persist>true</persist>` [to the user data script](https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ec2-windows-user-data.html#user-data-scripts). it can be used as follows:

```python
windows_user_data = ec2.UserData.for_windows(persist=True)
```

For a Linux instance, this can be accomplished by using a Multipart user data to configure cloud-config as detailed
in: https://aws.amazon.com/premiumsupport/knowledge-center/execute-user-data-ec2/

### Multipart user data

In addition, to above the `MultipartUserData` can be used to change instance startup behavior. Multipart user data are composed
from separate parts forming archive. The most common parts are scripts executed during instance set-up. However, there are other
kinds, too.

The advantage of multipart archive is in flexibility when it's needed to add additional parts or to use specialized parts to
fine tune instance startup. Some services (like AWS Batch) support only `MultipartUserData`.

The parts can be executed at different moment of instance start-up and can serve a different purpose. This is controlled by `contentType` property.
For common scripts, `text/x-shellscript; charset="utf-8"` can be used as content type.

In order to create archive the `MultipartUserData` has to be instantiated. Than, user can add parts to multipart archive using `addPart`. The `MultipartBody` contains methods supporting creation of body parts.

If the very custom part is required, it can be created using `MultipartUserData.fromRawBody`, in this case full control over content type,
transfer encoding, and body properties is given to the user.

Below is an example for creating multipart user data with single body part responsible for installing `awscli` and configuring maximum size
of storage used by Docker containers:

```python
boot_hook_conf = ec2.UserData.for_linux()
boot_hook_conf.add_commands("cloud-init-per once docker_options echo 'OPTIONS=\"${OPTIONS} --storage-opt dm.basesize=40G\"' >> /etc/sysconfig/docker")

setup_commands = ec2.UserData.for_linux()
setup_commands.add_commands("sudo yum install awscli && echo Packages installed らと > /var/tmp/setup")

multipart_user_data = ec2.MultipartUserData()
# The docker has to be configured at early stage, so content type is overridden to boothook
multipart_user_data.add_part(ec2.MultipartBody.from_user_data(boot_hook_conf, "text/cloud-boothook; charset=\"us-ascii\""))
# Execute the rest of setup
multipart_user_data.add_part(ec2.MultipartBody.from_user_data(setup_commands))

ec2.LaunchTemplate(self, "",
    user_data=multipart_user_data,
    block_devices=[]
)
```

For more information see
[Specifying Multiple User Data Blocks Using a MIME Multi Part Archive](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#multi-part_user_data)

#### Using add*Command on MultipartUserData

To use the `add*Command` methods, that are inherited from the `UserData` interface, on `MultipartUserData` you must add a part
to the `MultipartUserData` and designate it as the receiver for these methods. This is accomplished by using the `addUserDataPart()`
method on `MultipartUserData` with the `makeDefault` argument set to `true`:

```python
multipart_user_data = ec2.MultipartUserData()
commands_user_data = ec2.UserData.for_linux()
multipart_user_data.add_user_data_part(commands_user_data, ec2.MultipartBody.SHELL_SCRIPT, True)

# Adding commands to the multipartUserData adds them to commandsUserData, and vice-versa.
multipart_user_data.add_commands("touch /root/multi.txt")
commands_user_data.add_commands("touch /root/userdata.txt")
```

When used on an EC2 instance, the above `multipartUserData` will create both `multi.txt` and `userdata.txt` in `/root`.

## Importing existing subnet

To import an existing Subnet, call `Subnet.fromSubnetAttributes()` or
`Subnet.fromSubnetId()`. Only if you supply the subnet's Availability Zone
and Route Table Ids when calling `Subnet.fromSubnetAttributes()` will you be
able to use the CDK features that use these values (such as selecting one
subnet per AZ).

Importing an existing subnet looks like this:

```python
# Supply all properties
subnet1 = ec2.Subnet.from_subnet_attributes(self, "SubnetFromAttributes",
    subnet_id="s-1234",
    availability_zone="pub-az-4465",
    route_table_id="rt-145"
)

# Supply only subnet id
subnet2 = ec2.Subnet.from_subnet_id(self, "SubnetFromId", "s-1234")
```

## Launch Templates

A Launch Template is a standardized template that contains the configuration information to launch an instance.
They can be used when launching instances on their own, through Amazon EC2 Auto Scaling, EC2 Fleet, and Spot Fleet.
Launch templates enable you to store launch parameters so that you do not have to specify them every time you launch
an instance. For information on Launch Templates please see the
[official documentation](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html).

The following demonstrates how to create a launch template with an Amazon Machine Image, security group, and an instance profile.

```python
# vpc: ec2.Vpc


role = iam.Role(self, "Role",
    assumed_by=iam.ServicePrincipal("ec2.amazonaws.com")
)
instance_profile = iam.InstanceProfile(self, "InstanceProfile",
    role=role
)

template = ec2.LaunchTemplate(self, "LaunchTemplate",
    launch_template_name="MyTemplateV1",
    version_description="This is my v1 template",
    machine_image=ec2.MachineImage.latest_amazon_linux2023(),
    security_group=ec2.SecurityGroup(self, "LaunchTemplateSG",
        vpc=vpc
    ),
    instance_profile=instance_profile
)
```

And the following demonstrates how to enable metadata options support.

```python
ec2.LaunchTemplate(self, "LaunchTemplate",
    http_endpoint=True,
    http_protocol_ipv6=True,
    http_put_response_hop_limit=1,
    http_tokens=ec2.LaunchTemplateHttpTokens.REQUIRED,
    instance_metadata_tags=True
)
```

And the following demonstrates how to add one or more security groups to launch template.

```python
# vpc: ec2.Vpc


sg1 = ec2.SecurityGroup(self, "sg1",
    vpc=vpc
)
sg2 = ec2.SecurityGroup(self, "sg2",
    vpc=vpc
)

launch_template = ec2.LaunchTemplate(self, "LaunchTemplate",
    machine_image=ec2.MachineImage.latest_amazon_linux2023(),
    security_group=sg1
)

launch_template.add_security_group(sg2)
```

To use [AWS Systems Manager parameters instead of AMI IDs](https://docs.aws.amazon.com/autoscaling/ec2/userguide/using-systems-manager-parameters.html) in launch templates and resolve the AMI IDs at instance launch time:

```python
launch_template = ec2.LaunchTemplate(self, "LaunchTemplate",
    machine_image=ec2.MachineImage.resolve_ssm_parameter_at_launch("parameterName")
)
```

### Placement Group

Specify `placementGroup` to enable the placement group support:

```python
# instance_type: ec2.InstanceType


pg = ec2.PlacementGroup(self, "test-pg",
    strategy=ec2.PlacementGroupStrategy.SPREAD
)

ec2.LaunchTemplate(self, "LaunchTemplate",
    instance_type=instance_type,
    machine_image=ec2.MachineImage.latest_amazon_linux2023(),
    placement_group=pg
)
```

Please note this feature does not support Launch Configurations.

## Detailed Monitoring

The following demonstrates how to enable [Detailed Monitoring](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-cloudwatch-new.html) for an EC2 instance. Keep in mind that Detailed Monitoring results in [additional charges](http://aws.amazon.com/cloudwatch/pricing/).

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType


ec2.Instance(self, "Instance1",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=ec2.MachineImage.latest_amazon_linux2023(),
    detailed_monitoring=True
)
```

## Connecting to your instances using SSM Session Manager

SSM Session Manager makes it possible to connect to your instances from the
AWS Console, without preparing SSH keys.

To do so, you need to:

* Use an image with [SSM agent](https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-agent.html) installed
  and configured. [Many images come with SSM Agent
  preinstalled](https://docs.aws.amazon.com/systems-manager/latest/userguide/ami-preinstalled-agent.html), otherwise you
  may need to manually put instructions to [install SSM
  Agent](https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-manual-agent-install.html) into your
  instance's UserData or use EC2 Init).
* Create the instance with `ssmSessionPermissions: true`.

If these conditions are met, you can connect to the instance from the EC2 Console. Example:

```python
# vpc: ec2.Vpc
# instance_type: ec2.InstanceType


ec2.Instance(self, "Instance1",
    vpc=vpc,
    instance_type=instance_type,

    # Amazon Linux 2023 comes with SSM Agent by default
    machine_image=ec2.MachineImage.latest_amazon_linux2023(),

    # Turn on SSM
    ssm_session_permissions=True
)
```

## Managed Prefix Lists

Create and manage customer-managed prefix lists. If you don't specify anything in this construct, it will manage IPv4 addresses.

You can also create an empty Prefix List with only the maximum number of entries specified, as shown in the following code. If nothing is specified, maxEntries=1.

```python
ec2.PrefixList(self, "EmptyPrefixList",
    max_entries=100
)
```

`maxEntries` can also be omitted as follows. In this case `maxEntries: 2`, will be set.

```python
ec2.PrefixList(self, "PrefixList",
    entries=[ec2.CfnPrefixList.EntryProperty(cidr="10.0.0.1/32"), ec2.CfnPrefixList.EntryProperty(cidr="10.0.0.2/32", description="sample1")
    ]
)
```

To import AWS-managed prefix list, you can use `PrefixList.fromLookup()`.

```python
ec2.PrefixList.from_lookup(self, "PrefixListFromName",
    prefix_list_name="com.amazonaws.global.cloudfront.origin-facing"
)
```

For more information see [Work with customer-managed prefix lists](https://docs.aws.amazon.com/vpc/latest/userguide/working-with-managed-prefix-lists.html).

### IAM instance profile

Use `instanceProfile` to apply specific IAM Instance Profile. Cannot be used with role

```python
# instance_type: ec2.InstanceType
# vpc: ec2.Vpc


role = iam.Role(self, "Role",
    assumed_by=iam.ServicePrincipal("ec2.amazonaws.com")
)
instance_profile = iam.InstanceProfile(self, "InstanceProfile",
    role=role
)

ec2.Instance(self, "Instance",
    vpc=vpc,
    instance_type=instance_type,
    machine_image=ec2.MachineImage.latest_amazon_linux2023(),
    instance_profile=instance_profile
)
```
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    AssetHashType as _AssetHashType_05b67f2d,
    BundlingOptions as _BundlingOptions_588cc936,
    CfnResource as _CfnResource_9df397a6,
    CfnTag as _CfnTag_f6864754,
    Duration as _Duration_4839e8c3,
    Expiration as _Expiration_059d47d0,
    IAspect as _IAspect_118c810a,
    IInspectable as _IInspectable_c2943556,
    IResolvable as _IResolvable_da3f097b,
    IResource as _IResource_c80c4260,
    ITaggable as _ITaggable_36806126,
    ITaggableV2 as _ITaggableV2_4e6798f8,
    IgnoreMode as _IgnoreMode_655a98e8,
    RemovalPolicy as _RemovalPolicy_9f93c814,
    Resource as _Resource_45bc6135,
    ResourceProps as _ResourceProps_15a65b4e,
    SecretValue as _SecretValue_3dd0ddae,
    Size as _Size_7b441c34,
    Stack as _Stack_2866e57f,
    SymlinkFollowMode as _SymlinkFollowMode_047ec1f6,
    TagManager as _TagManager_0a598cb3,
    TreeInspector as _TreeInspector_488e0dd5,
)
from ..aws_cloudwatch import (
    Metric as _Metric_e396a4dc,
    MetricOptions as _MetricOptions_1788b62f,
    Unit as _Unit_61bc6f70,
)
from ..aws_iam import (
    ArnPrincipal as _ArnPrincipal_d31ca6bc,
    Grant as _Grant_a7ae64f8,
    IGrantable as _IGrantable_71c4f5de,
    IInstanceProfile as _IInstanceProfile_10d5ce2c,
    IPrincipal as _IPrincipal_539bb2fd,
    IRole as _IRole_235f5d8e,
    ISamlProvider as _ISamlProvider_63f03582,
    PolicyDocument as _PolicyDocument_3ac34393,
    PolicyStatement as _PolicyStatement_0fe33853,
)
from ..aws_kms import IKey as _IKey_5f11635f
from ..aws_logs import (
    ILogGroup as _ILogGroup_3c4fa718, ILogStream as _ILogStream_dcfca8c2
)
from ..aws_s3 import IBucket as _IBucket_42e086fd
from ..aws_s3_assets import (
    Asset as _Asset_ac2a7e61, AssetOptions as _AssetOptions_2aa69621
)
from ..aws_ssm import IStringParameter as _IStringParameter_f2b707f9


class AclCidr(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_ec2.AclCidr",
):
    '''Either an IPv4 or an IPv6 CIDR.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        acl_cidr = ec2.AclCidr.any_ipv4()
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="anyIpv4")
    @builtins.classmethod
    def any_ipv4(cls) -> "AclCidr":
        '''The CIDR containing all IPv4 addresses (i.e., 0.0.0.0/0).'''
        return typing.cast("AclCidr", jsii.sinvoke(cls, "anyIpv4", []))

    @jsii.member(jsii_name="anyIpv6")
    @builtins.classmethod
    def any_ipv6(cls) -> "AclCidr":
        '''The CIDR containing all IPv6 addresses (i.e., ::/0).'''
        return typing.cast("AclCidr", jsii.sinvoke(cls, "anyIpv6", []))

    @jsii.member(jsii_name="ipv4")
    @builtins.classmethod
    def ipv4(cls, ipv4_cidr: builtins.str) -> "AclCidr":
        '''An IP network range in CIDR notation (for example, 172.16.0.0/24).

        :param ipv4_cidr: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f3ad2a5e4919c72fd0ec671df247f101b5b82e588009ce91566c9f396c19964d)
            check_type(argname="argument ipv4_cidr", value=ipv4_cidr, expected_type=type_hints["ipv4_cidr"])
        return typing.cast("AclCidr", jsii.sinvoke(cls, "ipv4", [ipv4_cidr]))

    @jsii.member(jsii_name="ipv6")
    @builtins.classmethod
    def ipv6(cls, ipv6_cidr: builtins.str) -> "AclCidr":
        '''An IPv6 network range in CIDR notation (for example, 2001:db8::/48).

        :param ipv6_cidr: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ab6b9e0c88e64ab22e19f5a242a9ffd751298689bdc4a3606320998b2b7e339d)
            check_type(argname="argument ipv6_cidr", value=ipv6_cidr, expected_type=type_hints["ipv6_cidr"])
        return typing.cast("AclCidr", jsii.sinvoke(cls, "ipv6", [ipv6_cidr]))

    @jsii.member(jsii_name="toCidrConfig")
    @abc.abstractmethod
    def to_cidr_config(self) -> "AclCidrConfig":
        ...


class _AclCidrProxy(AclCidr):
    @jsii.member(jsii_name="toCidrConfig")
    def to_cidr_config(self) -> "AclCidrConfig":
        return typing.cast("AclCidrConfig", jsii.invoke(self, "toCidrConfig", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AclCidr).__jsii_proxy_class__ = lambda : _AclCidrProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AclCidrConfig",
    jsii_struct_bases=[],
    name_mapping={"cidr_block": "cidrBlock", "ipv6_cidr_block": "ipv6CidrBlock"},
)
class AclCidrConfig:
    def __init__(
        self,
        *,
        cidr_block: typing.Optional[builtins.str] = None,
        ipv6_cidr_block: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Acl Configuration for CIDR.

        :param cidr_block: Ipv4 CIDR.
        :param ipv6_cidr_block: Ipv6 CIDR.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            acl_cidr_config = ec2.AclCidrConfig(
                cidr_block="cidrBlock",
                ipv6_cidr_block="ipv6CidrBlock"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a528042c63b989d3b0c11f7e3e2d7d5c63ef7f27ec1567d962210008609ac034)
            check_type(argname="argument cidr_block", value=cidr_block, expected_type=type_hints["cidr_block"])
            check_type(argname="argument ipv6_cidr_block", value=ipv6_cidr_block, expected_type=type_hints["ipv6_cidr_block"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cidr_block is not None:
            self._values["cidr_block"] = cidr_block
        if ipv6_cidr_block is not None:
            self._values["ipv6_cidr_block"] = ipv6_cidr_block

    @builtins.property
    def cidr_block(self) -> typing.Optional[builtins.str]:
        '''Ipv4 CIDR.'''
        result = self._values.get("cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ipv6_cidr_block(self) -> typing.Optional[builtins.str]:
        '''Ipv6 CIDR.'''
        result = self._values.get("ipv6_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclCidrConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AclIcmp",
    jsii_struct_bases=[],
    name_mapping={"code": "code", "type": "type"},
)
class AclIcmp:
    def __init__(
        self,
        *,
        code: typing.Optional[jsii.Number] = None,
        type: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties to create Icmp.

        :param code: The Internet Control Message Protocol (ICMP) code. You can use -1 to specify all ICMP codes for the given ICMP type. Requirement is conditional: Required if you specify 1 (ICMP) for the protocol parameter.
        :param type: The Internet Control Message Protocol (ICMP) type. You can use -1 to specify all ICMP types. Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            acl_icmp = ec2.AclIcmp(
                code=123,
                type=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eb17a149fd86e4bb35aa154a67bb9dd9d7ab9c56f326c8456deb5341e61d7f00)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if code is not None:
            self._values["code"] = code
        if type is not None:
            self._values["type"] = type

    @builtins.property
    def code(self) -> typing.Optional[jsii.Number]:
        '''The Internet Control Message Protocol (ICMP) code.

        You can use -1 to specify all ICMP
        codes for the given ICMP type. Requirement is conditional: Required if you
        specify 1 (ICMP) for the protocol parameter.
        '''
        result = self._values.get("code")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def type(self) -> typing.Optional[jsii.Number]:
        '''The Internet Control Message Protocol (ICMP) type.

        You can use -1 to specify all ICMP types.
        Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclIcmp(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AclPortRange",
    jsii_struct_bases=[],
    name_mapping={"from_": "from", "to": "to"},
)
class AclPortRange:
    def __init__(
        self,
        *,
        from_: typing.Optional[jsii.Number] = None,
        to: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties to create PortRange.

        :param from_: The first port in the range. Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.
        :param to: The last port in the range. Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            acl_port_range = ec2.AclPortRange(
                from=123,
                to=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1c33fb537fef4ba8402530ec5630aee8c8410497a0863ede5f99ca517f82f449)
            check_type(argname="argument from_", value=from_, expected_type=type_hints["from_"])
            check_type(argname="argument to", value=to, expected_type=type_hints["to"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if from_ is not None:
            self._values["from_"] = from_
        if to is not None:
            self._values["to"] = to

    @builtins.property
    def from_(self) -> typing.Optional[jsii.Number]:
        '''The first port in the range.

        Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.
        '''
        result = self._values.get("from_")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def to(self) -> typing.Optional[jsii.Number]:
        '''The last port in the range.

        Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.
        '''
        result = self._values.get("to")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclPortRange(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AclTraffic(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_ec2.AclTraffic",
):
    '''The traffic that is configured using a Network ACL entry.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        acl_traffic = ec2.AclTraffic.all_traffic()
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="allTraffic")
    @builtins.classmethod
    def all_traffic(cls) -> "AclTraffic":
        '''Apply the ACL entry to all traffic.'''
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "allTraffic", []))

    @jsii.member(jsii_name="icmp")
    @builtins.classmethod
    def icmp(
        cls,
        *,
        code: typing.Optional[jsii.Number] = None,
        type: typing.Optional[jsii.Number] = None,
    ) -> "AclTraffic":
        '''Apply the ACL entry to ICMP traffic of given type and code.

        :param code: The Internet Control Message Protocol (ICMP) code. You can use -1 to specify all ICMP codes for the given ICMP type. Requirement is conditional: Required if you specify 1 (ICMP) for the protocol parameter.
        :param type: The Internet Control Message Protocol (ICMP) type. You can use -1 to specify all ICMP types. Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.
        '''
        props = AclIcmp(code=code, type=type)

        return typing.cast("AclTraffic", jsii.sinvoke(cls, "icmp", [props]))

    @jsii.member(jsii_name="icmpv6")
    @builtins.classmethod
    def icmpv6(
        cls,
        *,
        code: typing.Optional[jsii.Number] = None,
        type: typing.Optional[jsii.Number] = None,
    ) -> "AclTraffic":
        '''Apply the ACL entry to ICMPv6 traffic of given type and code.

        Requires an IPv6 CIDR block.

        :param code: The Internet Control Message Protocol (ICMP) code. You can use -1 to specify all ICMP codes for the given ICMP type. Requirement is conditional: Required if you specify 1 (ICMP) for the protocol parameter.
        :param type: The Internet Control Message Protocol (ICMP) type. You can use -1 to specify all ICMP types. Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.
        '''
        props = AclIcmp(code=code, type=type)

        return typing.cast("AclTraffic", jsii.sinvoke(cls, "icmpv6", [props]))

    @jsii.member(jsii_name="tcpPort")
    @builtins.classmethod
    def tcp_port(cls, port: jsii.Number) -> "AclTraffic":
        '''Apply the ACL entry to TCP traffic on a given port.

        :param port: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9e173d91f1c1835ccfdf6134dedc027e620c0f44039934fb8a281ef312e37d65)
            check_type(argname="argument port", value=port, expected_type=type_hints["port"])
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "tcpPort", [port]))

    @jsii.member(jsii_name="tcpPortRange")
    @builtins.classmethod
    def tcp_port_range(
        cls,
        start_port: jsii.Number,
        end_port: jsii.Number,
    ) -> "AclTraffic":
        '''Apply the ACL entry to TCP traffic on a given port range.

        :param start_port: -
        :param end_port: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6e335ec8df7c430d1e714bfe44c78e0cbeee9db3e25113497f4dd34eaf73bbf2)
            check_type(argname="argument start_port", value=start_port, expected_type=type_hints["start_port"])
            check_type(argname="argument end_port", value=end_port, expected_type=type_hints["end_port"])
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "tcpPortRange", [start_port, end_port]))

    @jsii.member(jsii_name="udpPort")
    @builtins.classmethod
    def udp_port(cls, port: jsii.Number) -> "AclTraffic":
        '''Apply the ACL entry to UDP traffic on a given port.

        :param port: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bdfed3d1e83596c6d4a6c551d18618296a9bc3f9cb56f39949877474341e83d3)
            check_type(argname="argument port", value=port, expected_type=type_hints["port"])
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "udpPort", [port]))

    @jsii.member(jsii_name="udpPortRange")
    @builtins.classmethod
    def udp_port_range(
        cls,
        start_port: jsii.Number,
        end_port: jsii.Number,
    ) -> "AclTraffic":
        '''Apply the ACL entry to UDP traffic on a given port range.

        :param start_port: -
        :param end_port: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c14e6bbd7559b33e3aa0635a4f52616c7f7431ee9711dcae11b1b5b3960e432d)
            check_type(argname="argument start_port", value=start_port, expected_type=type_hints["start_port"])
            check_type(argname="argument end_port", value=end_port, expected_type=type_hints["end_port"])
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "udpPortRange", [start_port, end_port]))

    @jsii.member(jsii_name="toTrafficConfig")
    @abc.abstractmethod
    def to_traffic_config(self) -> "AclTrafficConfig":
        ...


class _AclTrafficProxy(AclTraffic):
    @jsii.member(jsii_name="toTrafficConfig")
    def to_traffic_config(self) -> "AclTrafficConfig":
        return typing.cast("AclTrafficConfig", jsii.invoke(self, "toTrafficConfig", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AclTraffic).__jsii_proxy_class__ = lambda : _AclTrafficProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AclTrafficConfig",
    jsii_struct_bases=[],
    name_mapping={"protocol": "protocol", "icmp": "icmp", "port_range": "portRange"},
)
class AclTrafficConfig:
    def __init__(
        self,
        *,
        protocol: jsii.Number,
        icmp: typing.Optional[typing.Union[AclIcmp, typing.Dict[builtins.str, typing.Any]]] = None,
        port_range: typing.Optional[typing.Union[AclPortRange, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Acl Configuration for traffic.

        :param protocol: The protocol number. A value of "-1" means all protocols. If you specify "-1" or a protocol number other than "6" (TCP), "17" (UDP), or "1" (ICMP), traffic on all ports is allowed, regardless of any ports or ICMP types or codes that you specify. If you specify protocol "58" (ICMPv6) and specify an IPv4 CIDR block, traffic for all ICMP types and codes allowed, regardless of any that you specify. If you specify protocol "58" (ICMPv6) and specify an IPv6 CIDR block, you must specify an ICMP type and code. Default: 17
        :param icmp: The Internet Control Message Protocol (ICMP) code and type. Default: - Required if specifying 1 (ICMP) for the protocol parameter.
        :param port_range: The range of port numbers for the UDP/TCP protocol. Default: - Required if specifying 6 (TCP) or 17 (UDP) for the protocol parameter

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            acl_traffic_config = ec2.AclTrafficConfig(
                protocol=123,
            
                # the properties below are optional
                icmp=ec2.AclIcmp(
                    code=123,
                    type=123
                ),
                port_range=ec2.AclPortRange(
                    from=123,
                    to=123
                )
            )
        '''
        if isinstance(icmp, dict):
            icmp = AclIcmp(**icmp)
        if isinstance(port_range, dict):
            port_range = AclPortRange(**port_range)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d163c256d269f1d2f5078983924b9ad072aef3816da73da84929680d06b8d0bb)
            check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
            check_type(argname="argument icmp", value=icmp, expected_type=type_hints["icmp"])
            check_type(argname="argument port_range", value=port_range, expected_type=type_hints["port_range"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "protocol": protocol,
        }
        if icmp is not None:
            self._values["icmp"] = icmp
        if port_range is not None:
            self._values["port_range"] = port_range

    @builtins.property
    def protocol(self) -> jsii.Number:
        '''The protocol number.

        A value of "-1" means all protocols.

        If you specify "-1" or a protocol number other than "6" (TCP), "17" (UDP),
        or "1" (ICMP), traffic on all ports is allowed, regardless of any ports or
        ICMP types or codes that you specify.

        If you specify protocol "58" (ICMPv6) and specify an IPv4 CIDR
        block, traffic for all ICMP types and codes allowed, regardless of any that
        you specify. If you specify protocol "58" (ICMPv6) and specify an IPv6 CIDR
        block, you must specify an ICMP type and code.

        :default: 17
        '''
        result = self._values.get("protocol")
        assert result is not None, "Required property 'protocol' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def icmp(self) -> typing.Optional[AclIcmp]:
        '''The Internet Control Message Protocol (ICMP) code and type.

        :default: - Required if specifying 1 (ICMP) for the protocol parameter.
        '''
        result = self._values.get("icmp")
        return typing.cast(typing.Optional[AclIcmp], result)

    @builtins.property
    def port_range(self) -> typing.Optional[AclPortRange]:
        '''The range of port numbers for the UDP/TCP protocol.

        :default: - Required if specifying 6 (TCP) or 17 (UDP) for the protocol parameter
        '''
        result = self._values.get("port_range")
        return typing.cast(typing.Optional[AclPortRange], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclTrafficConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_ec2.Action")
class Action(enum.Enum):
    '''What action to apply to traffic matching the ACL.'''

    ALLOW = "ALLOW"
    '''Allow the traffic.'''
    DENY = "DENY"
    '''Deny the traffic.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AddRouteOptions",
    jsii_struct_bases=[],
    name_mapping={
        "router_id": "routerId",
        "router_type": "routerType",
        "destination_cidr_block": "destinationCidrBlock",
        "destination_ipv6_cidr_block": "destinationIpv6CidrBlock",
        "enables_internet_connectivity": "enablesInternetConnectivity",
    },
)
class AddRouteOptions:
    def __init__(
        self,
        *,
        router_id: builtins.str,
        router_type: "RouterType",
        destination_cidr_block: typing.Optional[builtins.str] = None,
        destination_ipv6_cidr_block: typing.Optional[builtins.str] = None,
        enables_internet_connectivity: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for adding a new route to a subnet.

        :param router_id: The ID of the router. Can be an instance ID, gateway ID, etc, depending on the router type.
        :param router_type: What type of router to route this traffic to.
        :param destination_cidr_block: IPv4 range this route applies to. Default: '0.0.0.0/0'
        :param destination_ipv6_cidr_block: IPv6 range this route applies to. Default: - Uses IPv6
        :param enables_internet_connectivity: Whether this route will enable internet connectivity. If true, this route will be added before any AWS resources that depend on internet connectivity in the VPC will be created. Default: false

        :exampleMetadata: infused

        Example::

            vpc = ec2.Vpc(self, "VPC",
                subnet_configuration=[ec2.SubnetConfiguration(
                    subnet_type=ec2.SubnetType.PUBLIC,
                    name="Public"
                ), ec2.SubnetConfiguration(
                    subnet_type=ec2.SubnetType.PRIVATE_ISOLATED,
                    name="Isolated"
                )]
            )
            
            (vpc.isolated_subnets[0]).add_route("StaticRoute",
                router_id=vpc.internet_gateway_id,
                router_type=ec2.RouterType.GATEWAY,
                destination_cidr_block="8.8.8.8/32"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f59a164a1424d8008d54093d179bfdbc68c5da0cf45fe708c6f2aa4ec7240d43)
            check_type(argname="argument router_id", value=router_id, expected_type=type_hints["router_id"])
            check_type(argname="argument router_type", value=router_type, expected_type=type_hints["router_type"])
            check_type(argname="argument destination_cidr_block", value=destination_cidr_block, expected_type=type_hints["destination_cidr_block"])
            check_type(argname="argument destination_ipv6_cidr_block", value=destination_ipv6_cidr_block, expected_type=type_hints["destination_ipv6_cidr_block"])
            check_type(argname="argument enables_internet_connectivity", value=enables_internet_connectivity, expected_type=type_hints["enables_internet_connectivity"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "router_id": router_id,
            "router_type": router_type,
        }
        if destination_cidr_block is not None:
            self._values["destination_cidr_block"] = destination_cidr_block
        if destination_ipv6_cidr_block is not None:
            self._values["destination_ipv6_cidr_block"] = destination_ipv6_cidr_block
        if enables_internet_connectivity is not None:
            self._values["enables_internet_connectivity"] = enables_internet_connectivity

    @builtins.property
    def router_id(self) -> builtins.str:
        '''The ID of the router.

        Can be an instance ID, gateway ID, etc, depending on the router type.
        '''
        result = self._values.get("router_id")
        assert result is not None, "Required property 'router_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def router_type(self) -> "RouterType":
        '''What type of router to route this traffic to.'''
        result = self._values.get("router_type")
        assert result is not None, "Required property 'router_type' is missing"
        return typing.cast("RouterType", result)

    @builtins.property
    def destination_cidr_block(self) -> typing.Optional[builtins.str]:
        '''IPv4 range this route applies to.

        :default: '0.0.0.0/0'
        '''
        result = self._values.get("destination_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def destination_ipv6_cidr_block(self) -> typing.Optional[builtins.str]:
        '''IPv6 range this route applies to.

        :default: - Uses IPv6
        '''
        result = self._values.get("destination_ipv6_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enables_internet_connectivity(self) -> typing.Optional[builtins.bool]:
        '''Whether this route will enable internet connectivity.

        If true, this route will be added before any AWS resources that depend
        on internet connectivity in the VPC will be created.

        :default: false
        '''
        result = self._values.get("enables_internet_connectivity")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddRouteOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_ec2.AddressFamily")
class AddressFamily(enum.Enum):
    '''The IP address type.'''

    IP_V4 = "IP_V4"
    IP_V6 = "IP_V6"


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AllocateCidrRequest",
    jsii_struct_bases=[],
    name_mapping={"requested_subnets": "requestedSubnets", "vpc_cidr": "vpcCidr"},
)
class AllocateCidrRequest:
    def __init__(
        self,
        *,
        requested_subnets: typing.Sequence[typing.Union["RequestedSubnet", typing.Dict[builtins.str, typing.Any]]],
        vpc_cidr: builtins.str,
    ) -> None:
        '''Request for subnets CIDR to be allocated for a Vpc.

        :param requested_subnets: The Subnets to be allocated.
        :param vpc_cidr: The IPv4 CIDR block for this Vpc.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            allocate_cidr_request = ec2.AllocateCidrRequest(
                requested_subnets=[ec2.RequestedSubnet(
                    availability_zone="availabilityZone",
                    configuration=ec2.SubnetConfiguration(
                        name="name",
                        subnet_type=ec2.SubnetType.PRIVATE_ISOLATED,
            
                        # the properties below are optional
                        cidr_mask=123,
                        ipv6_assign_address_on_creation=False,
                        map_public_ip_on_launch=False,
                        reserved=False
                    ),
                    subnet_construct_id="subnetConstructId"
                )],
                vpc_cidr="vpcCidr"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1df4b2c29495f98d544593b7df7715ec829f51c17796f9e361e366395a4a9b0a)
            check_type(argname="argument requested_subnets", value=requested_subnets, expected_type=type_hints["requested_subnets"])
            check_type(argname="argument vpc_cidr", value=vpc_cidr, expected_type=type_hints["vpc_cidr"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "requested_subnets": requested_subnets,
            "vpc_cidr": vpc_cidr,
        }

    @builtins.property
    def requested_subnets(self) -> typing.List["RequestedSubnet"]:
        '''The Subnets to be allocated.'''
        result = self._values.get("requested_subnets")
        assert result is not None, "Required property 'requested_subnets' is missing"
        return typing.cast(typing.List["RequestedSubnet"], result)

    @builtins.property
    def vpc_cidr(self) -> builtins.str:
        '''The IPv4 CIDR block for this Vpc.'''
        result = self._values.get("vpc_cidr")
        assert result is not None, "Required property 'vpc_cidr' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AllocateCidrRequest(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AllocateIpv6CidrRequest",
    jsii_struct_bases=[],
    name_mapping={"allocated_subnets": "allocatedSubnets", "ipv6_cidrs": "ipv6Cidrs"},
)
class AllocateIpv6CidrRequest:
    def __init__(
        self,
        *,
        allocated_subnets: typing.Sequence[typing.Union["AllocatedSubnet", typing.Dict[builtins.str, typing.Any]]],
        ipv6_cidrs: typing.Sequence[builtins.str],
    ) -> None:
        '''Request for subnet IPv6 CIDRs to be allocated for a VPC.

        :param allocated_subnets: List of subnets allocated with IPv4 CIDRs.
        :param ipv6_cidrs: The IPv6 CIDRs to be allocated to the subnets.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            allocate_ipv6_cidr_request = ec2.AllocateIpv6CidrRequest(
                allocated_subnets=[ec2.AllocatedSubnet(
                    cidr="cidr",
            
                    # the properties below are optional
                    ipv6_cidr="ipv6Cidr"
                )],
                ipv6_cidrs=["ipv6Cidrs"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0d2d15673b9e53df657dc88df46f1098eea51256e5f4addb1db32dc0e2690f00)
            check_type(argname="argument allocated_subnets", value=allocated_subnets, expected_type=type_hints["allocated_subnets"])
            check_type(argname="argument ipv6_cidrs", value=ipv6_cidrs, expected_type=type_hints["ipv6_cidrs"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "allocated_subnets": allocated_subnets,
            "ipv6_cidrs": ipv6_cidrs,
        }

    @builtins.property
    def allocated_subnets(self) -> typing.List["AllocatedSubnet"]:
        '''List of subnets allocated with IPv4 CIDRs.'''
        result = self._values.get("allocated_subnets")
        assert result is not None, "Required property 'allocated_subnets' is missing"
        return typing.cast(typing.List["AllocatedSubnet"], result)

    @builtins.property
    def ipv6_cidrs(self) -> typing.List[builtins.str]:
        '''The IPv6 CIDRs to be allocated to the subnets.'''
        result = self._values.get("ipv6_cidrs")
        assert result is not None, "Required property 'ipv6_cidrs' is missing"
        return typing.cast(typing.List[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AllocateIpv6CidrRequest(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AllocateVpcIpv6CidrRequest",
    jsii_struct_bases=[],
    name_mapping={"scope": "scope", "vpc_id": "vpcId"},
)
class AllocateVpcIpv6CidrRequest:
    def __init__(
        self,
        *,
        scope: _constructs_77d1e7e8.Construct,
        vpc_id: builtins.str,
    ) -> None:
        '''Request for allocation of the VPC IPv6 CIDR.

        :param scope: The VPC construct to attach to.
        :param vpc_id: The id of the VPC.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            import constructs as constructs
            
            # construct: constructs.Construct
            
            allocate_vpc_ipv6_cidr_request = ec2.AllocateVpcIpv6CidrRequest(
                scope=construct,
                vpc_id="vpcId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__201ab4d786fad817892547309e43b9b3ecac389ee3325f1ab6705b11dbd09c0c)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument vpc_id", value=vpc_id, expected_type=type_hints["vpc_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "scope": scope,
            "vpc_id": vpc_id,
        }

    @builtins.property
    def scope(self) -> _constructs_77d1e7e8.Construct:
        '''The VPC construct to attach to.'''
        result = self._values.get("scope")
        assert result is not None, "Required property 'scope' is missing"
        return typing.cast(_constructs_77d1e7e8.Construct, result)

    @builtins.property
    def vpc_id(self) -> builtins.str:
        '''The id of the VPC.'''
        result = self._values.get("vpc_id")
        assert result is not None, "Required property 'vpc_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AllocateVpcIpv6CidrRequest(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AllocatedSubnet",
    jsii_struct_bases=[],
    name_mapping={"cidr": "cidr", "ipv6_cidr": "ipv6Cidr"},
)
class AllocatedSubnet:
    def __init__(
        self,
        *,
        cidr: builtins.str,
        ipv6_cidr: typing.Optional[builtins.str] = None,
    ) -> None:
        '''CIDR Allocated Subnet.

        :param cidr: IPv4 CIDR Allocations for a Subnet. Note this is specific to the IPv4 CIDR.
        :param ipv6_cidr: IPv6 CIDR Allocations for a Subnet. Note this is specific to the IPv6 CIDR. Default: - no IPV6 CIDR

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            allocated_subnet = ec2.AllocatedSubnet(
                cidr="cidr",
            
                # the properties below are optional
                ipv6_cidr="ipv6Cidr"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cca8efe6fa73141fbec2a2cf91f8f6bf2e0f69d49252bd2d45b21fcb2dce3566)
            check_type(argname="argument cidr", value=cidr, expected_type=type_hints["cidr"])
            check_type(argname="argument ipv6_cidr", value=ipv6_cidr, expected_type=type_hints["ipv6_cidr"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cidr": cidr,
        }
        if ipv6_cidr is not None:
            self._values["ipv6_cidr"] = ipv6_cidr

    @builtins.property
    def cidr(self) -> builtins.str:
        '''IPv4 CIDR Allocations for a Subnet.

        Note this is specific to the IPv4 CIDR.
        '''
        result = self._values.get("cidr")
        assert result is not None, "Required property 'cidr' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def ipv6_cidr(self) -> typing.Optional[builtins.str]:
        '''IPv6 CIDR Allocations for a Subnet.

        Note this is specific to the IPv6 CIDR.

        :default: - no IPV6 CIDR
        '''
        result = self._values.get("ipv6_cidr")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AllocatedSubnet(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AmazonLinux2022Kernel(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.AmazonLinux2022Kernel",
):
    '''Amazon Linux 2022 kernel versions.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        amazon_linux2022_kernel = ec2.AmazonLinux2022Kernel.CDK_LATEST
    '''

    def __init__(self, version: builtins.str) -> None:
        '''
        :param version: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__67731f51c6dad8d404e1a6ed177e5a2605da4ce360058d786cf253f72d1aee3d)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        jsii.create(self.__class__, self, [version])

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Generate a string representation of the kernel.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CDK_LATEST")
    def CDK_LATEST(cls) -> "AmazonLinux2022Kernel":
        '''The latest kernel version currently available in a published AMI.

        When a new kernel version is available for an al2022 AMI this will be
        updated to contain the latest kernel version and will cause your instances
        to be replaced. Do not store stateful information on the instance if you are
        using this version.
        '''
        return typing.cast("AmazonLinux2022Kernel", jsii.sget(cls, "CDK_LATEST"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DEFAULT")
    def DEFAULT(cls) -> "AmazonLinux2022Kernel":
        '''The default kernel version for Amazon Linux 2022 is 5.15 and the SSM parameter does not include it in the name (i.e. /aws/service/ami-amazon-linux-latest/amzn2022-ami-kernel-default-x86_64).'''
        return typing.cast("AmazonLinux2022Kernel", jsii.sget(cls, "DEFAULT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KERNEL_5_15")
    def KERNEL_5_15(cls) -> "AmazonLinux2022Kernel":
        '''Kernel version 5.15.'''
        return typing.cast("AmazonLinux2022Kernel", jsii.sget(cls, "KERNEL_5_15"))


class AmazonLinux2023Kernel(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.AmazonLinux2023Kernel",
):
    '''Amazon Linux 2023 kernel versions.

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        
        
        ec2.Instance(self, "LatestAl2023",
            vpc=vpc,
            instance_type=ec2.InstanceType.of(ec2.InstanceClass.C7G, ec2.InstanceSize.LARGE),
            # context cache is turned on by default
            machine_image=ec2.AmazonLinux2023ImageSsmParameter(
                kernel=ec2.AmazonLinux2023Kernel.KERNEL_6_1
            )
        )
    '''

    def __init__(self, version: builtins.str) -> None:
        '''
        :param version: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0ab727190b8ad0e28c2d1ddc702af61630c845ff5aedd9578b75429a06aba8f5)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        jsii.create(self.__class__, self, [version])

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Generate a string representation of the kernel.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CDK_LATEST")
    def CDK_LATEST(cls) -> "AmazonLinux2023Kernel":
        '''The latest kernel version currently available in a published AMI.

        When a new kernel version is available for an al2023 AMI this will be
        updated to contain the latest kernel version and will cause your instances
        to be replaced. Do not store stateful information on the instance if you are
        using this version.
        '''
        return typing.cast("AmazonLinux2023Kernel", jsii.sget(cls, "CDK_LATEST"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DEFAULT")
    def DEFAULT(cls) -> "AmazonLinux2023Kernel":
        '''The default kernel version for Amazon Linux 2023 is 6.1 and the SSM parameter does not include it in the name (i.e. /aws/service/ami-amazon-linux-latest/amzn2023-ami-kernel-default-x86_64).'''
        return typing.cast("AmazonLinux2023Kernel", jsii.sget(cls, "DEFAULT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KERNEL_6_1")
    def KERNEL_6_1(cls) -> "AmazonLinux2023Kernel":
        '''Kernel version 6.1.'''
        return typing.cast("AmazonLinux2023Kernel", jsii.sget(cls, "KERNEL_6_1"))


class AmazonLinux2Kernel(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.AmazonLinux2Kernel",
):
    '''Amazon Linux 2 kernel versions.

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        # instance_type: ec2.InstanceType
        
        
        # Amazon Linux 2
        ec2.Instance(self, "Instance2",
            vpc=vpc,
            instance_type=instance_type,
            machine_image=ec2.MachineImage.latest_amazon_linux2()
        )
        
        # Amazon Linux 2 with kernel 5.x
        ec2.Instance(self, "Instance3",
            vpc=vpc,
            instance_type=instance_type,
            machine_image=ec2.MachineImage.latest_amazon_linux2(
                kernel=ec2.AmazonLinux2Kernel.KERNEL_5_10
            )
        )
        
        # Amazon Linux 2023
        ec2.Instance(self, "Instance4",
            vpc=vpc,
            instance_type=instance_type,
            machine_image=ec2.MachineImage.latest_amazon_linux2023()
        )
        
        # Graviton 3 Processor
        ec2.Instance(self, "Instance5",
            vpc=vpc,
            instance_type=ec2.InstanceType.of(ec2.InstanceClass.C7G, ec2.InstanceSize.LARGE),
            machine_image=ec2.MachineImage.latest_amazon_linux2023(
                cpu_type=ec2.AmazonLinuxCpuType.ARM_64
            )
        )
    '''

    def __init__(self, version: builtins.str) -> None:
        '''
        :param version: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3d9d80aebc17990a8f3419f9444ec1fdd3e5bc1c150c3db0d42eda409806c7a8)
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        jsii.create(self.__class__, self, [version])

    @jsii.member(jsii_name="toString")
    def to_string(self) -> typing.Optional[builtins.str]:
        '''Generate a string representation of the kernel.'''
        return typing.cast(typing.Optional[builtins.str], jsii.invoke(self, "toString", []))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CDK_LATEST")
    def CDK_LATEST(cls) -> "AmazonLinux2Kernel":
        '''The latest kernel version currently available in a published AMI.

        When a new kernel version is available for an amzn2 AMI this will be
        updated to contain the latest kernel version and will cause your instances
        to be replaced. Do not store stateful information on the instance if you are
        using this version.
        '''
        return typing.cast("AmazonLinux2Kernel", jsii.sget(cls, "CDK_LATEST"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DEFAULT")
    def DEFAULT(cls) -> "AmazonLinux2Kernel":
        '''The default kernel version for Amazon Linux 2 is 4.14 and Linux 4.14.311-233.529.amzn2.x86_64 the SSM parameter does not include it in the name (i.e. /aws/service/ami-amazon-linux-latest/amzn2-ami-hvm-x86_64-gp2).'''
        return typing.cast("AmazonLinux2Kernel", jsii.sget(cls, "DEFAULT"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="KERNEL_5_10")
    def KERNEL_5_10(cls) -> "AmazonLinux2Kernel":
        '''Kernel version 5.10.'''
        return typing.cast("AmazonLinux2Kernel", jsii.sget(cls, "KERNEL_5_10"))


@jsii.enum(jsii_type="aws-cdk-lib.aws_ec2.AmazonLinuxCpuType")
class AmazonLinuxCpuType(enum.Enum):
    '''CPU type.

    :exampleMetadata: lit=aws-ec2/test/example.images.lit.ts infused

    Example::

        # Pick the right Amazon Linux edition. All arguments shown are optional
        # and will default to these values when omitted.
        amzn_linux = ec2.MachineImage.latest_amazon_linux(
            generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
            edition=ec2.AmazonLinuxEdition.STANDARD,
            virtualization=ec2.AmazonLinuxVirt.HVM,
            storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
            cpu_type=ec2.AmazonLinuxCpuType.X86_64
        )
        
        # Pick a Windows edition to use
        windows = ec2.MachineImage.latest_windows(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)
        
        # Read AMI id from SSM parameter store
        ssm = ec2.MachineImage.from_ssm_parameter("/my/ami", os=ec2.OperatingSystemType.LINUX)
        
        # Look up the most recent image matching a set of AMI filters.
        # In this case, look up the NAT instance AMI, by using a wildcard
        # in the 'name' field:
        nat_ami = ec2.MachineImage.lookup(
            name="amzn-ami-vpc-nat-*",
            owners=["amazon"]
        )
        
        # For other custom (Linux) images, instantiate a `GenericLinuxImage` with
        # a map giving the AMI to in for each region:
        linux = ec2.MachineImage.generic_linux({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
        
        # For other custom (Windows) images, instantiate a `GenericWindowsImage` with
        # a map giving the AMI to in for each region:
        generic_windows = ec2.MachineImage.generic_windows({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
    '''

    ARM_64 = "ARM_64"
    '''arm64 CPU type.'''
    X86_64 = "X86_64"
    '''x86_64 CPU type.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_ec2.AmazonLinuxEdition")
class AmazonLinuxEdition(enum.Enum):
    '''Amazon Linux edition.

    :exampleMetadata: lit=aws-autoscaling/test/example.images.lit.ts infused

    Example::

        # Pick a Windows edition to use
        windows = ec2.WindowsImage(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)
        
        # Pick the right Amazon Linux edition. All arguments shown are optional
        # and will default to these values when omitted.
        amzn_linux = ec2.AmazonLinuxImage(
            generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
            edition=ec2.AmazonLinuxEdition.STANDARD,
            virtualization=ec2.AmazonLinuxVirt.HVM,
            storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE
        )
        
        # For other custom (Linux) images, instantiate a `GenericLinuxImage` with
        # a map giving the AMI to in for each region:
        
        linux = ec2.GenericLinuxImage({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
    '''

    STANDARD = "STANDARD"
    '''Standard edition.'''
    MINIMAL = "MINIMAL"
    '''Minimal edition.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_ec2.AmazonLinuxGeneration")
class AmazonLinuxGeneration(enum.Enum):
    '''What generation of Amazon Linux to use.

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        
        
        ec2.Instance(self, "Instance",
            vpc=vpc,
            instance_type=ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.NANO),
            machine_image=ec2.AmazonLinuxImage(generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX_2),
            instance_initiated_shutdown_behavior=ec2.InstanceInitiatedShutdownBehavior.TERMINATE
        )
    '''

    AMAZON_LINUX = "AMAZON_LINUX"
    '''Amazon Linux.'''
    AMAZON_LINUX_2 = "AMAZON_LINUX_2"
    '''Amazon Linux 2.'''
    AMAZON_LINUX_2022 = "AMAZON_LINUX_2022"
    '''Amazon Linux 2022.'''
    AMAZON_LINUX_2023 = "AMAZON_LINUX_2023"
    '''Amazon Linux 2023.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AmazonLinuxImageProps",
    jsii_struct_bases=[],
    name_mapping={
        "cached_in_context": "cachedInContext",
        "cpu_type": "cpuType",
        "edition": "edition",
        "generation": "generation",
        "kernel": "kernel",
        "storage": "storage",
        "user_data": "userData",
        "virtualization": "virtualization",
    },
)
class AmazonLinuxImageProps:
    def __init__(
        self,
        *,
        cached_in_context: typing.Optional[builtins.bool] = None,
        cpu_type: typing.Optional[AmazonLinuxCpuType] = None,
        edition: typing.Optional[AmazonLinuxEdition] = None,
        generation: typing.Optional[AmazonLinuxGeneration] = None,
        kernel: typing.Optional["AmazonLinuxKernel"] = None,
        storage: typing.Optional["AmazonLinuxStorage"] = None,
        user_data: typing.Optional["UserData"] = None,
        virtualization: typing.Optional["AmazonLinuxVirt"] = None,
    ) -> None:
        '''Amazon Linux image properties.

        :param cached_in_context: Whether the AMI ID is cached to be stable between deployments. By default, the newest image is used on each deployment. This will cause instances to be replaced whenever a new version is released, and may cause downtime if there aren't enough running instances in the AutoScalingGroup to reschedule the tasks on. If set to true, the AMI ID will be cached in ``cdk.context.json`` and the same value will be used on future runs. Your instances will not be replaced but your AMI version will grow old over time. To refresh the AMI lookup, you will have to evict the value from the cache using the ``cdk context`` command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information. Can not be set to ``true`` in environment-agnostic stacks. Default: false
        :param cpu_type: CPU Type. Default: X86_64
        :param edition: What edition of Amazon Linux to use. Default: Standard
        :param generation: What generation of Amazon Linux to use. Default: AmazonLinux
        :param kernel: What kernel version of Amazon Linux to use. Default: -
        :param storage: What storage backed image to use. Default: GeneralPurpose
        :param user_data: Initial user data. Default: - Empty UserData for Linux machines
        :param virtualization: Virtualization type. Default: HVM

        :exampleMetadata: infused

        Example::

            # vpc: ec2.IVpc
            
            lb = elb.LoadBalancer(self, "LB",
                vpc=vpc,
                internet_facing=True
            )
            
            # instance to add as the target for load balancer.
            instance = ec2.Instance(self, "targetInstance",
                vpc=vpc,
                instance_type=ec2.InstanceType.of(ec2.InstanceClass.BURSTABLE2, ec2.InstanceSize.MICRO),
                machine_image=ec2.AmazonLinuxImage(generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX_2)
            )
            lb.add_target(elb.InstanceTarget(instance))
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dd95a518c06563a07debc9bf1ca51713b747cf64bded6f7cc81a28680d7ea9fc)
            check_type(argname="argument cached_in_context", value=cached_in_context, expected_type=type_hints["cached_in_context"])
            check_type(argname="argument cpu_type", value=cpu_type, expected_type=type_hints["cpu_type"])
            check_type(argname="argument edition", value=edition, expected_type=type_hints["edition"])
            check_type(argname="argument generation", value=generation, expected_type=type_hints["generation"])
            check_type(argname="argument kernel", value=kernel, expected_type=type_hints["kernel"])
            check_type(argname="argument storage", value=storage, expected_type=type_hints["storage"])
            check_type(argname="argument user_data", value=user_data, expected_type=type_hints["user_data"])
            check_type(argname="argument virtualization", value=virtualization, expected_type=type_hints["virtualization"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cached_in_context is not None:
            self._values["cached_in_context"] = cached_in_context
        if cpu_type is not None:
            self._values["cpu_type"] = cpu_type
        if edition is not None:
            self._values["edition"] = edition
        if generation is not None:
            self._values["generation"] = generation
        if kernel is not None:
            self._values["kernel"] = kernel
        if storage is not None:
            self._values["storage"] = storage
        if user_data is not None:
            self._values["user_data"] = user_data
        if virtualization is not None:
            self._values["virtualization"] = virtualization

    @builtins.property
    def cached_in_context(self) -> typing.Optional[builtins.bool]:
        '''Whether the AMI ID is cached to be stable between deployments.

        By default, the newest image is used on each deployment. This will cause
        instances to be replaced whenever a new version is released, and may cause
        downtime if there aren't enough running instances in the AutoScalingGroup
        to reschedule the tasks on.

        If set to true, the AMI ID will be cached in ``cdk.context.json`` and the
        same value will be used on future runs. Your instances will not be replaced
        but your AMI version will grow old over time. To refresh the AMI lookup,
        you will have to evict the value from the cache using the ``cdk context``
        command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for
        more information.

        Can not be set to ``true`` in environment-agnostic stacks.

        :default: false
        '''
        result = self._values.get("cached_in_context")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cpu_type(self) -> typing.Optional[AmazonLinuxCpuType]:
        '''CPU Type.

        :default: X86_64
        '''
        result = self._values.get("cpu_type")
        return typing.cast(typing.Optional[AmazonLinuxCpuType], result)

    @builtins.property
    def edition(self) -> typing.Optional[AmazonLinuxEdition]:
        '''What edition of Amazon Linux to use.

        :default: Standard
        '''
        result = self._values.get("edition")
        return typing.cast(typing.Optional[AmazonLinuxEdition], result)

    @builtins.property
    def generation(self) -> typing.Optional[AmazonLinuxGeneration]:
        '''What generation of Amazon Linux to use.

        :default: AmazonLinux
        '''
        result = self._values.get("generation")
        return typing.cast(typing.Optional[AmazonLinuxGeneration], result)

    @builtins.property
    def kernel(self) -> typing.Optional["AmazonLinuxKernel"]:
        '''What kernel version of Amazon Linux to use.

        :default: -
        '''
        result = self._values.get("kernel")
        return typing.cast(typing.Optional["AmazonLinuxKernel"], result)

    @builtins.property
    def storage(self) -> typing.Optional["AmazonLinuxStorage"]:
        '''What storage backed image to use.

        :default: GeneralPurpose
        '''
        result = self._values.get("storage")
        return typing.cast(typing.Optional["AmazonLinuxStorage"], result)

    @builtins.property
    def user_data(self) -> typing.Optional["UserData"]:
        '''Initial user data.

        :default: - Empty UserData for Linux machines
        '''
        result = self._values.get("user_data")
        return typing.cast(typing.Optional["UserData"], result)

    @builtins.property
    def virtualization(self) -> typing.Optional["AmazonLinuxVirt"]:
        '''Virtualization type.

        :default: HVM
        '''
        result = self._values.get("virtualization")
        return typing.cast(typing.Optional["AmazonLinuxVirt"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AmazonLinuxImageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AmazonLinuxImageSsmParameterBaseOptions",
    jsii_struct_bases=[],
    name_mapping={"cached_in_context": "cachedInContext", "user_data": "userData"},
)
class AmazonLinuxImageSsmParameterBaseOptions:
    def __init__(
        self,
        *,
        cached_in_context: typing.Optional[builtins.bool] = None,
        user_data: typing.Optional["UserData"] = None,
    ) -> None:
        '''Base options for amazon linux ssm parameters.

        :param cached_in_context: Whether the AMI ID is cached to be stable between deployments. By default, the newest image is used on each deployment. This will cause instances to be replaced whenever a new version is released, and may cause downtime if there aren't enough running instances in the AutoScalingGroup to reschedule the tasks on. If set to true, the AMI ID will be cached in ``cdk.context.json`` and the same value will be used on future runs. Your instances will not be replaced but your AMI version will grow old over time. To refresh the AMI lookup, you will have to evict the value from the cache using the ``cdk context`` command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information. Can not be set to ``true`` in environment-agnostic stacks. Default: false
        :param user_data: Initial user data. Default: - Empty UserData for Linux machines

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            # user_data: ec2.UserData
            
            amazon_linux_image_ssm_parameter_base_options = ec2.AmazonLinuxImageSsmParameterBaseOptions(
                cached_in_context=False,
                user_data=user_data
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__08eb71b09afb1d361b3f043a5c2d7d4be91f84b114302afefc8a69963ee6333f)
            check_type(argname="argument cached_in_context", value=cached_in_context, expected_type=type_hints["cached_in_context"])
            check_type(argname="argument user_data", value=user_data, expected_type=type_hints["user_data"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cached_in_context is not None:
            self._values["cached_in_context"] = cached_in_context
        if user_data is not None:
            self._values["user_data"] = user_data

    @builtins.property
    def cached_in_context(self) -> typing.Optional[builtins.bool]:
        '''Whether the AMI ID is cached to be stable between deployments.

        By default, the newest image is used on each deployment. This will cause
        instances to be replaced whenever a new version is released, and may cause
        downtime if there aren't enough running instances in the AutoScalingGroup
        to reschedule the tasks on.

        If set to true, the AMI ID will be cached in ``cdk.context.json`` and the
        same value will be used on future runs. Your instances will not be replaced
        but your AMI version will grow old over time. To refresh the AMI lookup,
        you will have to evict the value from the cache using the ``cdk context``
        command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for
        more information.

        Can not be set to ``true`` in environment-agnostic stacks.

        :default: false
        '''
        result = self._values.get("cached_in_context")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def user_data(self) -> typing.Optional["UserData"]:
        '''Initial user data.

        :default: - Empty UserData for Linux machines
        '''
        result = self._values.get("user_data")
        return typing.cast(typing.Optional["UserData"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AmazonLinuxImageSsmParameterBaseOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AmazonLinuxImageSsmParameterBaseProps",
    jsii_struct_bases=[AmazonLinuxImageSsmParameterBaseOptions],
    name_mapping={
        "cached_in_context": "cachedInContext",
        "user_data": "userData",
        "parameter_name": "parameterName",
    },
)
class AmazonLinuxImageSsmParameterBaseProps(AmazonLinuxImageSsmParameterBaseOptions):
    def __init__(
        self,
        *,
        cached_in_context: typing.Optional[builtins.bool] = None,
        user_data: typing.Optional["UserData"] = None,
        parameter_name: builtins.str,
    ) -> None:
        '''Base properties for an Amazon Linux SSM Parameter.

        :param cached_in_context: Whether the AMI ID is cached to be stable between deployments. By default, the newest image is used on each deployment. This will cause instances to be replaced whenever a new version is released, and may cause downtime if there aren't enough running instances in the AutoScalingGroup to reschedule the tasks on. If set to true, the AMI ID will be cached in ``cdk.context.json`` and the same value will be used on future runs. Your instances will not be replaced but your AMI version will grow old over time. To refresh the AMI lookup, you will have to evict the value from the cache using the ``cdk context`` command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information. Can not be set to ``true`` in environment-agnostic stacks. Default: false
        :param user_data: Initial user data. Default: - Empty UserData for Linux machines
        :param parameter_name: The name of the SSM parameter that contains the AMI value.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            # user_data: ec2.UserData
            
            amazon_linux_image_ssm_parameter_base_props = ec2.AmazonLinuxImageSsmParameterBaseProps(
                parameter_name="parameterName",
            
                # the properties below are optional
                cached_in_context=False,
                user_data=user_data
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__78a32cfc14171e780d6e44d3e6a99faa6da50655ac09560f7b229b0c0c7383f1)
            check_type(argname="argument cached_in_context", value=cached_in_context, expected_type=type_hints["cached_in_context"])
            check_type(argname="argument user_data", value=user_data, expected_type=type_hints["user_data"])
            check_type(argname="argument parameter_name", value=parameter_name, expected_type=type_hints["parameter_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "parameter_name": parameter_name,
        }
        if cached_in_context is not None:
            self._values["cached_in_context"] = cached_in_context
        if user_data is not None:
            self._values["user_data"] = user_data

    @builtins.property
    def cached_in_context(self) -> typing.Optional[builtins.bool]:
        '''Whether the AMI ID is cached to be stable between deployments.

        By default, the newest image is used on each deployment. This will cause
        instances to be replaced whenever a new version is released, and may cause
        downtime if there aren't enough running instances in the AutoScalingGroup
        to reschedule the tasks on.

        If set to true, the AMI ID will be cached in ``cdk.context.json`` and the
        same value will be used on future runs. Your instances will not be replaced
        but your AMI version will grow old over time. To refresh the AMI lookup,
        you will have to evict the value from the cache using the ``cdk context``
        command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for
        more information.

        Can not be set to ``true`` in environment-agnostic stacks.

        :default: false
        '''
        result = self._values.get("cached_in_context")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def user_data(self) -> typing.Optional["UserData"]:
        '''Initial user data.

        :default: - Empty UserData for Linux machines
        '''
        result = self._values.get("user_data")
        return typing.cast(typing.Optional["UserData"], result)

    @builtins.property
    def parameter_name(self) -> builtins.str:
        '''The name of the SSM parameter that contains the AMI value.'''
        result = self._values.get("parameter_name")
        assert result is not None, "Required property 'parameter_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AmazonLinuxImageSsmParameterBaseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AmazonLinuxImageSsmParameterCommonOptions",
    jsii_struct_bases=[AmazonLinuxImageSsmParameterBaseOptions],
    name_mapping={
        "cached_in_context": "cachedInContext",
        "user_data": "userData",
        "cpu_type": "cpuType",
        "edition": "edition",
    },
)
class AmazonLinuxImageSsmParameterCommonOptions(
    AmazonLinuxImageSsmParameterBaseOptions,
):
    def __init__(
        self,
        *,
        cached_in_context: typing.Optional[builtins.bool] = None,
        user_data: typing.Optional["UserData"] = None,
        cpu_type: typing.Optional[AmazonLinuxCpuType] = None,
        edition: typing.Optional[AmazonLinuxEdition] = None,
    ) -> None:
        '''Common options across all generations.

        :param cached_in_context: Whether the AMI ID is cached to be stable between deployments. By default, the newest image is used on each deployment. This will cause instances to be replaced whenever a new version is released, and may cause downtime if there aren't enough running instances in the AutoScalingGroup to reschedule the tasks on. If set to true, the AMI ID will be cached in ``cdk.context.json`` and the same value will be used on future runs. Your instances will not be replaced but your AMI version will grow old over time. To refresh the AMI lookup, you will have to evict the value from the cache using the ``cdk context`` command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information. Can not be set to ``true`` in environment-agnostic stacks. Default: false
        :param user_data: Initial user data. Default: - Empty UserData for Linux machines
        :param cpu_type: CPU Type. Default: AmazonLinuxCpuType.X86_64
        :param edition: What edition of Amazon Linux to use. Default: AmazonLinuxEdition.Standard

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            # user_data: ec2.UserData
            
            amazon_linux_image_ssm_parameter_common_options = ec2.AmazonLinuxImageSsmParameterCommonOptions(
                cached_in_context=False,
                cpu_type=ec2.AmazonLinuxCpuType.ARM_64,
                edition=ec2.AmazonLinuxEdition.STANDARD,
                user_data=user_data
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8fcdd01a6ab45c77447fba339b5a9358c3ab40c4833c1f86c68009a6a3bb1422)
            check_type(argname="argument cached_in_context", value=cached_in_context, expected_type=type_hints["cached_in_context"])
            check_type(argname="argument user_data", value=user_data, expected_type=type_hints["user_data"])
            check_type(argname="argument cpu_type", value=cpu_type, expected_type=type_hints["cpu_type"])
            check_type(argname="argument edition", value=edition, expected_type=type_hints["edition"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cached_in_context is not None:
            self._values["cached_in_context"] = cached_in_context
        if user_data is not None:
            self._values["user_data"] = user_data
        if cpu_type is not None:
            self._values["cpu_type"] = cpu_type
        if edition is not None:
            self._values["edition"] = edition

    @builtins.property
    def cached_in_context(self) -> typing.Optional[builtins.bool]:
        '''Whether the AMI ID is cached to be stable between deployments.

        By default, the newest image is used on each deployment. This will cause
        instances to be replaced whenever a new version is released, and may cause
        downtime if there aren't enough running instances in the AutoScalingGroup
        to reschedule the tasks on.

        If set to true, the AMI ID will be cached in ``cdk.context.json`` and the
        same value will be used on future runs. Your instances will not be replaced
        but your AMI version will grow old over time. To refresh the AMI lookup,
        you will have to evict the value from the cache using the ``cdk context``
        command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for
        more information.

        Can not be set to ``true`` in environment-agnostic stacks.

        :default: false
        '''
        result = self._values.get("cached_in_context")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def user_data(self) -> typing.Optional["UserData"]:
        '''Initial user data.

        :default: - Empty UserData for Linux machines
        '''
        result = self._values.get("user_data")
        return typing.cast(typing.Optional["UserData"], result)

    @builtins.property
    def cpu_type(self) -> typing.Optional[AmazonLinuxCpuType]:
        '''CPU Type.

        :default: AmazonLinuxCpuType.X86_64
        '''
        result = self._values.get("cpu_type")
        return typing.cast(typing.Optional[AmazonLinuxCpuType], result)

    @builtins.property
    def edition(self) -> typing.Optional[AmazonLinuxEdition]:
        '''What edition of Amazon Linux to use.

        :default: AmazonLinuxEdition.Standard
        '''
        result = self._values.get("edition")
        return typing.cast(typing.Optional[AmazonLinuxEdition], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AmazonLinuxImageSsmParameterCommonOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_ec2.AmazonLinuxKernel")
class AmazonLinuxKernel(enum.Enum):
    '''Amazon Linux Kernel.'''

    KERNEL5_X = "KERNEL5_X"
    '''Kernel version 5.10.'''
    KERNEL6_1 = "KERNEL6_1"
    '''Kernel version 6.1.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_ec2.AmazonLinuxStorage")
class AmazonLinuxStorage(enum.Enum):
    '''Available storage options for Amazon Linux images Only applies to Amazon Linux & Amazon Linux 2.

    :exampleMetadata: lit=aws-autoscaling/test/example.images.lit.ts infused

    Example::

        # Pick a Windows edition to use
        windows = ec2.WindowsImage(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)
        
        # Pick the right Amazon Linux edition. All arguments shown are optional
        # and will default to these values when omitted.
        amzn_linux = ec2.AmazonLinuxImage(
            generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
            edition=ec2.AmazonLinuxEdition.STANDARD,
            virtualization=ec2.AmazonLinuxVirt.HVM,
            storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE
        )
        
        # For other custom (Linux) images, instantiate a `GenericLinuxImage` with
        # a map giving the AMI to in for each region:
        
        linux = ec2.GenericLinuxImage({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
    '''

    EBS = "EBS"
    '''EBS-backed storage.'''
    S3 = "S3"
    '''S3-backed storage.'''
    GENERAL_PURPOSE = "GENERAL_PURPOSE"
    '''General Purpose-based storage (recommended).'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_ec2.AmazonLinuxVirt")
class AmazonLinuxVirt(enum.Enum):
    '''Virtualization type for Amazon Linux.

    :exampleMetadata: lit=aws-autoscaling/test/example.images.lit.ts infused

    Example::

        # Pick a Windows edition to use
        windows = ec2.WindowsImage(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)
        
        # Pick the right Amazon Linux edition. All arguments shown are optional
        # and will default to these values when omitted.
        amzn_linux = ec2.AmazonLinuxImage(
            generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
            edition=ec2.AmazonLinuxEdition.STANDARD,
            virtualization=ec2.AmazonLinuxVirt.HVM,
            storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE
        )
        
        # For other custom (Linux) images, instantiate a `GenericLinuxImage` with
        # a map giving the AMI to in for each region:
        
        linux = ec2.GenericLinuxImage({
            "us-east-1": "ami-97785bed",
            "eu-west-1": "ami-12345678"
        })
    '''

    HVM = "HVM"
    '''HVM virtualization (recommended).'''
    PV = "PV"
    '''PV virtualization.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.ApplyCloudFormationInitOptions",
    jsii_struct_bases=[],
    name_mapping={
        "config_sets": "configSets",
        "embed_fingerprint": "embedFingerprint",
        "ignore_failures": "ignoreFailures",
        "include_role": "includeRole",
        "include_url": "includeUrl",
        "print_log": "printLog",
        "timeout": "timeout",
    },
)
class ApplyCloudFormationInitOptions:
    def __init__(
        self,
        *,
        config_sets: typing.Optional[typing.Sequence[builtins.str]] = None,
        embed_fingerprint: typing.Optional[builtins.bool] = None,
        ignore_failures: typing.Optional[builtins.bool] = None,
        include_role: typing.Optional[builtins.bool] = None,
        include_url: typing.Optional[builtins.bool] = None,
        print_log: typing.Optional[builtins.bool] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''Options for applying CloudFormation init to an instance or instance group.

        :param config_sets: ConfigSet to activate. Default: ['default']
        :param embed_fingerprint: Force instance replacement by embedding a config fingerprint. If ``true`` (the default), a hash of the config will be embedded into the UserData, so that if the config changes, the UserData changes. - If the EC2 instance is instance-store backed or ``userDataCausesReplacement`` is set, this will cause the instance to be replaced and the new configuration to be applied. - If the instance is EBS-backed and ``userDataCausesReplacement`` is not set, the change of UserData will make the instance restart but not be replaced, and the configuration will not be applied automatically. If ``false``, no hash will be embedded, and if the CloudFormation Init config changes nothing will happen to the running instance. If a config update introduces errors, you will not notice until after the CloudFormation deployment successfully finishes and the next instance fails to launch. Default: true
        :param ignore_failures: Don't fail the instance creation when cfn-init fails. You can use this to prevent CloudFormation from rolling back when instances fail to start up, to help in debugging. Default: false
        :param include_role: Include --role argument when running cfn-init and cfn-signal commands. This will be the IAM instance profile attached to the EC2 instance Default: false
        :param include_url: Include --url argument when running cfn-init and cfn-signal commands. This will be the cloudformation endpoint in the deployed region e.g. https://cloudformation.us-east-1.amazonaws.com Default: false
        :param print_log: Print the results of running cfn-init to the Instance System Log. By default, the output of running cfn-init is written to a log file on the instance. Set this to ``true`` to print it to the System Log (visible from the EC2 Console), ``false`` to not print it. (Be aware that the system log is refreshed at certain points in time of the instance life cycle, and successful execution may not always show up). Default: true
        :param timeout: Timeout waiting for the configuration to be applied. Default: Duration.minutes(5)

        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            # instance_type: ec2.InstanceType
            # machine_image: ec2.IMachineImage
            
            
            ec2.Instance(self, "Instance",
                vpc=vpc,
                instance_type=instance_type,
                machine_image=machine_image,
            
                # Showing the most complex setup, if you have simpler requirements
                # you can use `CloudFormationInit.fromElements()`.
                init=ec2.CloudFormationInit.from_config_sets(
                    config_sets={
                        # Applies the configs below in this order
                        "default": ["yumPreinstall", "config"]
                    },
                    configs={
                        "yum_preinstall": ec2.InitConfig([
                            # Install an Amazon Linux package using yum
                            ec2.InitPackage.yum("git")
                        ]),
                        "config": ec2.InitConfig([
                            # Create a JSON file from tokens (can also create other files)
                            ec2.InitFile.from_object("/etc/stack.json", {
                                "stack_id": Stack.of(self).stack_id,
                                "stack_name": Stack.of(self).stack_name,
                                "region": Stack.of(self).region
                            }),
            
                            # Create a group and user
                            ec2.InitGroup.from_name("my-group"),
                            ec2.InitUser.from_name("my-user"),
            
                            # Install an RPM from the internet
                            ec2.InitPackage.rpm("http://mirrors.ukfast.co.uk/sites/dl.fedoraproject.org/pub/epel/8/Everything/x86_64/Packages/r/rubygem-git-1.5.0-2.el8.noarch.rpm")
                        ])
                    }
                ),
                init_options=ec2.ApplyCloudFormationInitOptions(
                    # Optional, which configsets to activate (['default'] by default)
                    config_sets=["default"],
            
                    # Optional, how long the installation is expected to take (5 minutes by default)
                    timeout=Duration.minutes(30),
            
                    # Optional, whether to include the --url argument when running cfn-init and cfn-signal commands (false by default)
                    include_url=True,
            
                    # Optional, whether to include the --role argument when running cfn-init and cfn-signal commands (false by default)
                    include_role=True
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e7f632381407c0e83d84ca5b16928a38d4d8f6de29513094b92a876e870ddb63)
            check_type(argname="argument config_sets", value=config_sets, expected_type=type_hints["config_sets"])
            check_type(argname="argument embed_fingerprint", value=embed_fingerprint, expected_type=type_hints["embed_fingerprint"])
            check_type(argname="argument ignore_failures", value=ignore_failures, expected_type=type_hints["ignore_failures"])
            check_type(argname="argument include_role", value=include_role, expected_type=type_hints["include_role"])
            check_type(argname="argument include_url", value=include_url, expected_type=type_hints["include_url"])
            check_type(argname="argument print_log", value=print_log, expected_type=type_hints["print_log"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if config_sets is not None:
            self._values["config_sets"] = config_sets
        if embed_fingerprint is not None:
            self._values["embed_fingerprint"] = embed_fingerprint
        if ignore_failures is not None:
            self._values["ignore_failures"] = ignore_failures
        if include_role is not None:
            self._values["include_role"] = include_role
        if include_url is not None:
            self._values["include_url"] = include_url
        if print_log is not None:
            self._values["print_log"] = print_log
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def config_sets(self) -> typing.Optional[typing.List[builtins.str]]:
        '''ConfigSet to activate.

        :default: ['default']
        '''
        result = self._values.get("config_sets")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def embed_fingerprint(self) -> typing.Optional[builtins.bool]:
        '''Force instance replacement by embedding a config fingerprint.

        If ``true`` (the default), a hash of the config will be embedded into the
        UserData, so that if the config changes, the UserData changes.

        - If the EC2 instance is instance-store backed or
          ``userDataCausesReplacement`` is set, this will cause the instance to be
          replaced and the new configuration to be applied.
        - If the instance is EBS-backed and ``userDataCausesReplacement`` is not
          set, the change of UserData will make the instance restart but not be
          replaced, and the configuration will not be applied automatically.

        If ``false``, no hash will be embedded, and if the CloudFormation Init
        config changes nothing will happen to the running instance. If a
        config update introduces errors, you will not notice until after the
        CloudFormation deployment successfully finishes and the next instance
        fails to launch.

        :default: true
        '''
        result = self._values.get("embed_fingerprint")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def ignore_failures(self) -> typing.Optional[builtins.bool]:
        '''Don't fail the instance creation when cfn-init fails.

        You can use this to prevent CloudFormation from rolling back when
        instances fail to start up, to help in debugging.

        :default: false
        '''
        result = self._values.get("ignore_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def include_role(self) -> typing.Optional[builtins.bool]:
        '''Include --role argument when running cfn-init and cfn-signal commands.

        This will be the IAM instance profile attached to the EC2 instance

        :default: false
        '''
        result = self._values.get("include_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def include_url(self) -> typing.Optional[builtins.bool]:
        '''Include --url argument when running cfn-init and cfn-signal commands.

        This will be the cloudformation endpoint in the deployed region
        e.g. https://cloudformation.us-east-1.amazonaws.com

        :default: false
        '''
        result = self._values.get("include_url")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def print_log(self) -> typing.Optional[builtins.bool]:
        '''Print the results of running cfn-init to the Instance System Log.

        By default, the output of running cfn-init is written to a log file
        on the instance. Set this to ``true`` to print it to the System Log
        (visible from the EC2 Console), ``false`` to not print it.

        (Be aware that the system log is refreshed at certain points in
        time of the instance life cycle, and successful execution may
        not always show up).

        :default: true
        '''
        result = self._values.get("print_log")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Timeout waiting for the configuration to be applied.

        :default: Duration.minutes(5)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApplyCloudFormationInitOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AttachInitOptions",
    jsii_struct_bases=[],
    name_mapping={
        "instance_role": "instanceRole",
        "platform": "platform",
        "user_data": "userData",
        "config_sets": "configSets",
        "embed_fingerprint": "embedFingerprint",
        "ignore_failures": "ignoreFailures",
        "include_role": "includeRole",
        "include_url": "includeUrl",
        "print_log": "printLog",
        "signal_resource": "signalResource",
    },
)
class AttachInitOptions:
    def __init__(
        self,
        *,
        instance_role: _IRole_235f5d8e,
        platform: "OperatingSystemType",
        user_data: "UserData",
        config_sets: typing.Optional[typing.Sequence[builtins.str]] = None,
        embed_fingerprint: typing.Optional[builtins.bool] = None,
        ignore_failures: typing.Optional[builtins.bool] = None,
        include_role: typing.Optional[builtins.bool] = None,
        include_url: typing.Optional[builtins.bool] = None,
        print_log: typing.Optional[builtins.bool] = None,
        signal_resource: typing.Optional[_CfnResource_9df397a6] = None,
    ) -> None:
        '''Options for attaching a CloudFormationInit to a resource.

        :param instance_role: Instance role of the consuming instance or fleet.
        :param platform: OS Platform the init config will be used for.
        :param user_data: UserData to add commands to.
        :param config_sets: ConfigSet to activate. Default: ['default']
        :param embed_fingerprint: Whether to embed a hash into the userData. If ``true`` (the default), a hash of the config will be embedded into the UserData, so that if the config changes, the UserData changes and the instance will be replaced. If ``false``, no such hash will be embedded, and if the CloudFormation Init config changes nothing will happen to the running instance. Default: true
        :param ignore_failures: Don't fail the instance creation when cfn-init fails. You can use this to prevent CloudFormation from rolling back when instances fail to start up, to help in debugging. Default: false
        :param include_role: Include --role argument when running cfn-init and cfn-signal commands. This will be the IAM instance profile attached to the EC2 instance Default: false
        :param include_url: Include --url argument when running cfn-init and cfn-signal commands. This will be the cloudformation endpoint in the deployed region e.g. https://cloudformation.us-east-1.amazonaws.com Default: false
        :param print_log: Print the results of running cfn-init to the Instance System Log. By default, the output of running cfn-init is written to a log file on the instance. Set this to ``true`` to print it to the System Log (visible from the EC2 Console), ``false`` to not print it. (Be aware that the system log is refreshed at certain points in time of the instance life cycle, and successful execution may not always show up). Default: true
        :param signal_resource: When provided, signals this resource instead of the attached resource. You can use this to support signaling LaunchTemplate while attaching AutoScalingGroup Default: - if this property is undefined cfn-signal signals the attached resource

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_ec2 as ec2
            from aws_cdk import aws_iam as iam
            
            # cfn_resource: cdk.CfnResource
            # role: iam.Role
            # user_data: ec2.UserData
            
            attach_init_options = ec2.AttachInitOptions(
                instance_role=role,
                platform=ec2.OperatingSystemType.LINUX,
                user_data=user_data,
            
                # the properties below are optional
                config_sets=["configSets"],
                embed_fingerprint=False,
                ignore_failures=False,
                include_role=False,
                include_url=False,
                print_log=False,
                signal_resource=cfn_resource
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d40dbbec96cabb7e34633204b5a62ce85e126032034ff03db0d8c739abfb5a2d)
            check_type(argname="argument instance_role", value=instance_role, expected_type=type_hints["instance_role"])
            check_type(argname="argument platform", value=platform, expected_type=type_hints["platform"])
            check_type(argname="argument user_data", value=user_data, expected_type=type_hints["user_data"])
            check_type(argname="argument config_sets", value=config_sets, expected_type=type_hints["config_sets"])
            check_type(argname="argument embed_fingerprint", value=embed_fingerprint, expected_type=type_hints["embed_fingerprint"])
            check_type(argname="argument ignore_failures", value=ignore_failures, expected_type=type_hints["ignore_failures"])
            check_type(argname="argument include_role", value=include_role, expected_type=type_hints["include_role"])
            check_type(argname="argument include_url", value=include_url, expected_type=type_hints["include_url"])
            check_type(argname="argument print_log", value=print_log, expected_type=type_hints["print_log"])
            check_type(argname="argument signal_resource", value=signal_resource, expected_type=type_hints["signal_resource"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "instance_role": instance_role,
            "platform": platform,
            "user_data": user_data,
        }
        if config_sets is not None:
            self._values["config_sets"] = config_sets
        if embed_fingerprint is not None:
            self._values["embed_fingerprint"] = embed_fingerprint
        if ignore_failures is not None:
            self._values["ignore_failures"] = ignore_failures
        if include_role is not None:
            self._values["include_role"] = include_role
        if include_url is not None:
            self._values["include_url"] = include_url
        if print_log is not None:
            self._values["print_log"] = print_log
        if signal_resource is not None:
            self._values["signal_resource"] = signal_resource

    @builtins.property
    def instance_role(self) -> _IRole_235f5d8e:
        '''Instance role of the consuming instance or fleet.'''
        result = self._values.get("instance_role")
        assert result is not None, "Required property 'instance_role' is missing"
        return typing.cast(_IRole_235f5d8e, result)

    @builtins.property
    def platform(self) -> "OperatingSystemType":
        '''OS Platform the init config will be used for.'''
        result = self._values.get("platform")
        assert result is not None, "Required property 'platform' is missing"
        return typing.cast("OperatingSystemType", result)

    @builtins.property
    def user_data(self) -> "UserData":
        '''UserData to add commands to.'''
        result = self._values.get("user_data")
        assert result is not None, "Required property 'user_data' is missing"
        return typing.cast("UserData", result)

    @builtins.property
    def config_sets(self) -> typing.Optional[typing.List[builtins.str]]:
        '''ConfigSet to activate.

        :default: ['default']
        '''
        result = self._values.get("config_sets")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def embed_fingerprint(self) -> typing.Optional[builtins.bool]:
        '''Whether to embed a hash into the userData.

        If ``true`` (the default), a hash of the config will be embedded into the
        UserData, so that if the config changes, the UserData changes and
        the instance will be replaced.

        If ``false``, no such hash will be embedded, and if the CloudFormation Init
        config changes nothing will happen to the running instance.

        :default: true
        '''
        result = self._values.get("embed_fingerprint")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def ignore_failures(self) -> typing.Optional[builtins.bool]:
        '''Don't fail the instance creation when cfn-init fails.

        You can use this to prevent CloudFormation from rolling back when
        instances fail to start up, to help in debugging.

        :default: false
        '''
        result = self._values.get("ignore_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def include_role(self) -> typing.Optional[builtins.bool]:
        '''Include --role argument when running cfn-init and cfn-signal commands.

        This will be the IAM instance profile attached to the EC2 instance

        :default: false
        '''
        result = self._values.get("include_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def include_url(self) -> typing.Optional[builtins.bool]:
        '''Include --url argument when running cfn-init and cfn-signal commands.

        This will be the cloudformation endpoint in the deployed region
        e.g. https://cloudformation.us-east-1.amazonaws.com

        :default: false
        '''
        result = self._values.get("include_url")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def print_log(self) -> typing.Optional[builtins.bool]:
        '''Print the results of running cfn-init to the Instance System Log.

        By default, the output of running cfn-init is written to a log file
        on the instance. Set this to ``true`` to print it to the System Log
        (visible from the EC2 Console), ``false`` to not print it.

        (Be aware that the system log is refreshed at certain points in
        time of the instance life cycle, and successful execution may
        not always show up).

        :default: true
        '''
        result = self._values.get("print_log")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def signal_resource(self) -> typing.Optional[_CfnResource_9df397a6]:
        '''When provided, signals this resource instead of the attached resource.

        You can use this to support signaling LaunchTemplate while attaching AutoScalingGroup

        :default: - if this property is undefined cfn-signal signals the attached resource
        '''
        result = self._values.get("signal_resource")
        return typing.cast(typing.Optional[_CfnResource_9df397a6], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AttachInitOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.AwsIpamProps",
    jsii_struct_bases=[],
    name_mapping={
        "ipv4_ipam_pool_id": "ipv4IpamPoolId",
        "ipv4_netmask_length": "ipv4NetmaskLength",
        "default_subnet_ipv4_netmask_length": "defaultSubnetIpv4NetmaskLength",
    },
)
class AwsIpamProps:
    def __init__(
        self,
        *,
        ipv4_ipam_pool_id: builtins.str,
        ipv4_netmask_length: jsii.Number,
        default_subnet_ipv4_netmask_length: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Configuration for AwsIpam.

        :param ipv4_ipam_pool_id: Ipam Pool Id for ipv4 allocation.
        :param ipv4_netmask_length: Netmask length for Vpc.
        :param default_subnet_ipv4_netmask_length: Default length for Subnet ipv4 Network mask. Specify this option only if you do not specify all Subnets using SubnetConfiguration with a cidrMask Default: - Default ipv4 Subnet Mask for subnets in Vpc

        :exampleMetadata: infused

        Example::

            from aws_cdk.aws_ec2 import IpAddresses
            
            # pool: ec2.CfnIPAMPool
            
            
            ec2.Vpc(self, "TheVPC",
                ip_addresses=IpAddresses.aws_ipam_allocation(
                    ipv4_ipam_pool_id=pool.ref,
                    ipv4_netmask_length=18,
                    default_subnet_ipv4_netmask_length=24
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f03eeb114f6c99cb527203711f14fcc87021cebcc68936ea2dc38a933caa1694)
            check_type(argname="argument ipv4_ipam_pool_id", value=ipv4_ipam_pool_id, expected_type=type_hints["ipv4_ipam_pool_id"])
            check_type(argname="argument ipv4_netmask_length", value=ipv4_netmask_length, expected_type=type_hints["ipv4_netmask_length"])
            check_type(argname="argument default_subnet_ipv4_netmask_length", value=default_subnet_ipv4_netmask_length, expected_type=type_hints["default_subnet_ipv4_netmask_length"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "ipv4_ipam_pool_id": ipv4_ipam_pool_id,
            "ipv4_netmask_length": ipv4_netmask_length,
        }
        if default_subnet_ipv4_netmask_length is not None:
            self._values["default_subnet_ipv4_netmask_length"] = default_subnet_ipv4_netmask_length

    @builtins.property
    def ipv4_ipam_pool_id(self) -> builtins.str:
        '''Ipam Pool Id for ipv4 allocation.'''
        result = self._values.get("ipv4_ipam_pool_id")
        assert result is not None, "Required property 'ipv4_ipam_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def ipv4_netmask_length(self) -> jsii.Number:
        '''Netmask length for Vpc.'''
        result = self._values.get("ipv4_netmask_length")
        assert result is not None, "Required property 'ipv4_netmask_length' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def default_subnet_ipv4_netmask_length(self) -> typing.Optional[jsii.Number]:
        '''Default length for Subnet ipv4 Network mask.

        Specify this option only if you do not specify all Subnets using SubnetConfiguration with a cidrMask

        :default: - Default ipv4 Subnet Mask for subnets in Vpc
        '''
        result = self._values.get("default_subnet_ipv4_netmask_length")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsIpamProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.BastionHostLinuxProps",
    jsii_struct_bases=[],
    name_mapping={
        "vpc": "vpc",
        "availability_zone": "availabilityZone",
        "block_devices": "blockDevices",
        "init": "init",
        "init_options": "initOptions",
        "instance_name": "instanceName",
        "instance_type": "instanceType",
        "machine_image": "machineImage",
        "require_imdsv2": "requireImdsv2",
        "security_group": "securityGroup",
        "subnet_selection": "subnetSelection",
        "user_data_causes_replacement": "userDataCausesReplacement",
    },
)
class BastionHostLinuxProps:
    def __init__(
        self,
        *,
        vpc: "IVpc",
        availability_zone: typing.Optional[builtins.str] = None,
        block_devices: typing.Optional[typing.Sequence[typing.Union["BlockDevice", typing.Dict[builtins.str, typing.Any]]]] = None,
        init: typing.Optional["CloudFormationInit"] = None,
        init_options: typing.Optional[typing.Union[ApplyCloudFormationInitOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        instance_name: typing.Optional[builtins.str] = None,
        instance_type: typing.Optional["InstanceType"] = None,
        machine_image: typing.Optional["IMachineImage"] = None,
        require_imdsv2: typing.Optional[builtins.bool] = None,
        security_group: typing.Optional["ISecurityGroup"] = None,
        subnet_selection: typing.Optional[typing.Union["SubnetSelection", typing.Dict[builtins.str, typing.Any]]] = None,
        user_data_causes_replacement: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Properties of the bastion host.

        :param vpc: VPC to launch the instance in.
        :param availability_zone: In which AZ to place the instance within the VPC. Default: - Random zone.
        :param block_devices: Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes. Each instance that is launched has an associated root device volume, either an Amazon EBS volume or an instance store volume. You can use block device mappings to specify additional EBS volumes or instance store volumes to attach to an instance when it is launched. Default: - Uses the block device mapping of the AMI
        :param init: Apply the given CloudFormation Init configuration to the instance at startup. Default: - no CloudFormation init
        :param init_options: Use the given options for applying CloudFormation Init. Describes the configsets to use and the timeout to wait Default: - default options
        :param instance_name: The name of the instance. Default: 'BastionHost'
        :param instance_type: Type of instance to launch. Default: 't3.nano'
        :param machine_image: The machine image to use, assumed to have SSM Agent preinstalled. Default: - An Amazon Linux 2023 image if the ``@aws-cdk/aws-ec2:bastionHostUseAmazonLinux2023ByDefault`` feature flag is enabled. Otherwise, an Amazon Linux 2 image. In both cases, the image is kept up-to-date automatically (the instance may be replaced on every deployment) and already has SSM Agent installed.
        :param require_imdsv2: Whether IMDSv2 should be required on this instance. Default: - false
        :param security_group: Security Group to assign to this instance. Default: - create new security group with no inbound and all outbound traffic allowed
        :param subnet_selection: Select the subnets to run the bastion host in. Set this to PUBLIC if you need to connect to this instance via the internet and cannot use SSM. You have to allow port 22 manually by using the connections field Default: - private subnets of the supplied VPC
        :param user_data_causes_replacement: Determines whether changes to the UserData will force instance replacement. Depending on the EC2 instance type, modifying the UserData may either restart or replace the instance: - Instance store-backed instances are replaced. - EBS-backed instances are restarted. Note that by default, restarting does not execute the updated UserData, so an alternative mechanism is needed to ensure the instance re-executes the UserData. When set to ``true``, the instance's Logical ID will depend on the UserData, causing CloudFormation to replace the instance if the UserData changes. Default: - ``true`` if ``initOptions`` is specified, otherwise ``false``.

        :exampleMetadata: fixture=with-vpc infused

        Example::

            host = ec2.BastionHostLinux(self, "BastionHost",
                vpc=vpc,
                block_devices=[ec2.BlockDevice(
                    device_name="/dev/sdh",
                    volume=ec2.BlockDeviceVolume.ebs(10,
                        encrypted=True
                    )
                )]
            )
        '''
        if isinstance(init_options, dict):
            init_options = ApplyCloudFormationInitOptions(**init_options)
        if isinstance(subnet_selection, dict):
            subnet_selection = SubnetSelection(**subnet_selection)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2647a77163fdd79c5b81f9523b8e35e195386f549d272d3474261e5251a24e52)
            check_type(argname="argument vpc", value=vpc, expected_type=type_hints["vpc"])
            check_type(argname="argument availability_zone", value=availability_zone, expected_type=type_hints["availability_zone"])
            check_type(argname="argument block_devices", value=block_devices, expected_type=type_hints["block_devices"])
            check_type(argname="argument init", value=init, expected_type=type_hints["init"])
            check_type(argname="argument init_options", value=init_options, expected_type=type_hints["init_options"])
            check_type(argname="argument instance_name", value=instance_name, expected_type=type_hints["instance_name"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument machine_image", value=machine_image, expected_type=type_hints["machine_image"])
            check_type(argname="argument require_imdsv2", value=require_imdsv2, expected_type=type_hints["require_imdsv2"])
            check_type(argname="argument security_group", value=security_group, expected_type=type_hints["security_group"])
            check_type(argname="argument subnet_selection", value=subnet_selection, expected_type=type_hints["subnet_selection"])
            check_type(argname="argument user_data_causes_replacement", value=user_data_causes_replacement, expected_type=type_hints["user_data_causes_replacement"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "vpc": vpc,
        }
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if block_devices is not None:
            self._values["block_devices"] = block_devices
        if init is not None:
            self._values["init"] = init
        if init_options is not None:
            self._values["init_options"] = init_options
        if instance_name is not None:
            self._values["instance_name"] = instance_name
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if machine_image is not None:
            self._values["machine_image"] = machine_image
        if require_imdsv2 is not None:
            self._values["require_imdsv2"] = require_imdsv2
        if security_group is not None:
            self._values["security_group"] = security_group
        if subnet_selection is not None:
            self._values["subnet_selection"] = subnet_selection
        if user_data_causes_replacement is not None:
            self._values["user_data_causes_replacement"] = user_data_causes_replacement

    @builtins.property
    def vpc(self) -> "IVpc":
        '''VPC to launch the instance in.'''
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return typing.cast("IVpc", result)

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''In which AZ to place the instance within the VPC.

        :default: - Random zone.
        '''
        result = self._values.get("availability_zone")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def block_devices(self) -> typing.Optional[typing.List["BlockDevice"]]:
        '''Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.

        Each instance that is launched has an associated root device volume,
        either an Amazon EBS volume or an instance store volume.
        You can use block device mappings to specify additional EBS volumes or
        instance store volumes to attach to an instance when it is launched.

        :default: - Uses the block device mapping of the AMI

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html
        '''
        result = self._values.get("block_devices")
        return typing.cast(typing.Optional[typing.List["BlockDevice"]], result)

    @builtins.property
    def init(self) -> typing.Optional["CloudFormationInit"]:
        '''Apply the given CloudFormation Init configuration to the instance at startup.

        :default: - no CloudFormation init
        '''
        result = self._values.get("init")
        return typing.cast(typing.Optional["CloudFormationInit"], result)

    @builtins.property
    def init_options(self) -> typing.Optional[ApplyCloudFormationInitOptions]:
        '''Use the given options for applying CloudFormation Init.

        Describes the configsets to use and the timeout to wait

        :default: - default options
        '''
        result = self._values.get("init_options")
        return typing.cast(typing.Optional[ApplyCloudFormationInitOptions], result)

    @builtins.property
    def instance_name(self) -> typing.Optional[builtins.str]:
        '''The name of the instance.

        :default: 'BastionHost'
        '''
        result = self._values.get("instance_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_type(self) -> typing.Optional["InstanceType"]:
        '''Type of instance to launch.

        :default: 't3.nano'
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional["InstanceType"], result)

    @builtins.property
    def machine_image(self) -> typing.Optional["IMachineImage"]:
        '''The machine image to use, assumed to have SSM Agent preinstalled.

        :default:

        - An Amazon Linux 2023 image if the ``@aws-cdk/aws-ec2:bastionHostUseAmazonLinux2023ByDefault`` feature flag is enabled. Otherwise, an Amazon Linux 2 image. In both cases, the image is kept up-to-date automatically (the instance
        may be replaced on every deployment) and already has SSM Agent installed.
        '''
        result = self._values.get("machine_image")
        return typing.cast(typing.Optional["IMachineImage"], result)

    @builtins.property
    def require_imdsv2(self) -> typing.Optional[builtins.bool]:
        '''Whether IMDSv2 should be required on this instance.

        :default: - false
        '''
        result = self._values.get("require_imdsv2")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def security_group(self) -> typing.Optional["ISecurityGroup"]:
        '''Security Group to assign to this instance.

        :default: - create new security group with no inbound and all outbound traffic allowed
        '''
        result = self._values.get("security_group")
        return typing.cast(typing.Optional["ISecurityGroup"], result)

    @builtins.property
    def subnet_selection(self) -> typing.Optional["SubnetSelection"]:
        '''Select the subnets to run the bastion host in.

        Set this to PUBLIC if you need to connect to this instance via the internet and cannot use SSM.
        You have to allow port 22 manually by using the connections field

        :default: - private subnets of the supplied VPC
        '''
        result = self._values.get("subnet_selection")
        return typing.cast(typing.Optional["SubnetSelection"], result)

    @builtins.property
    def user_data_causes_replacement(self) -> typing.Optional[builtins.bool]:
        '''Determines whether changes to the UserData will force instance replacement.

        Depending on the EC2 instance type, modifying the UserData may either restart
        or replace the instance:

        - Instance store-backed instances are replaced.
        - EBS-backed instances are restarted.

        Note that by default, restarting does not execute the updated UserData, so an alternative
        mechanism is needed to ensure the instance re-executes the UserData.

        When set to ``true``, the instance's Logical ID will depend on the UserData, causing
        CloudFormation to replace the instance if the UserData changes.

        :default: - ``true`` if ``initOptions`` is specified, otherwise ``false``.
        '''
        result = self._values.get("user_data_causes_replacement")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BastionHostLinuxProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.BlockDevice",
    jsii_struct_bases=[],
    name_mapping={
        "device_name": "deviceName",
        "volume": "volume",
        "mapping_enabled": "mappingEnabled",
    },
)
class BlockDevice:
    def __init__(
        self,
        *,
        device_name: builtins.str,
        volume: "BlockDeviceVolume",
        mapping_enabled: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Block device.

        :param device_name: The device name exposed to the EC2 instance. For example, a value like ``/dev/sdh``, ``xvdh``.
        :param volume: Defines the block device volume, to be either an Amazon EBS volume or an ephemeral instance store volume. For example, a value like ``BlockDeviceVolume.ebs(15)``, ``BlockDeviceVolume.ephemeral(0)``.
        :param mapping_enabled: If false, the device mapping will be suppressed. If set to false for the root device, the instance might fail the Amazon EC2 health check. Amazon EC2 Auto Scaling launches a replacement instance if the instance fails the health check. Default: true - device mapping is left untouched

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            # block_device_volume: ec2.BlockDeviceVolume
            
            block_device = ec2.BlockDevice(
                device_name="deviceName",
                volume=block_device_volume,
            
                # the properties below are optional
                mapping_enabled=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d6658aae654252e134d66f13fd022e06de0b6f39cf994c929751371e10c0774d)
            check_type(argname="argument device_name", value=device_name, expected_type=type_hints["device_name"])
            check_type(argname="argument volume", value=volume, expected_type=type_hints["volume"])
            check_type(argname="argument mapping_enabled", value=mapping_enabled, expected_type=type_hints["mapping_enabled"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "device_name": device_name,
            "volume": volume,
        }
        if mapping_enabled is not None:
            self._values["mapping_enabled"] = mapping_enabled

    @builtins.property
    def device_name(self) -> builtins.str:
        '''The device name exposed to the EC2 instance.

        For example, a value like ``/dev/sdh``, ``xvdh``.

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/device_naming.html
        '''
        result = self._values.get("device_name")
        assert result is not None, "Required property 'device_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def volume(self) -> "BlockDeviceVolume":
        '''Defines the block device volume, to be either an Amazon EBS volume or an ephemeral instance store volume.

        For example, a value like ``BlockDeviceVolume.ebs(15)``, ``BlockDeviceVolume.ephemeral(0)``.
        '''
        result = self._values.get("volume")
        assert result is not None, "Required property 'volume' is missing"
        return typing.cast("BlockDeviceVolume", result)

    @builtins.property
    def mapping_enabled(self) -> typing.Optional[builtins.bool]:
        '''If false, the device mapping will be suppressed.

        If set to false for the root device, the instance might fail the Amazon EC2 health check.
        Amazon EC2 Auto Scaling launches a replacement instance if the instance fails the health check.

        :default: true - device mapping is left untouched
        '''
        result = self._values.get("mapping_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BlockDevice(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class BlockDeviceVolume(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.BlockDeviceVolume",
):
    '''Describes a block device mapping for an EC2 instance or Auto Scaling group.

    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        # instance_type: ec2.InstanceType
        # machine_image: ec2.IMachineImage
        
        
        ec2.Instance(self, "Instance",
            vpc=vpc,
            instance_type=instance_type,
            machine_image=machine_image,
        
            # ...
        
            block_devices=[ec2.BlockDevice(
                device_name="/dev/sda1",
                volume=ec2.BlockDeviceVolume.ebs(100,
                    volume_type=ec2.EbsDeviceVolumeType.GP3,
                    throughput=250
                )
            )
            ]
        )
    '''

    def __init__(
        self,
        ebs_device: typing.Optional[typing.Union["EbsDeviceProps", typing.Dict[builtins.str, typing.Any]]] = None,
        virtual_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param ebs_device: EBS device info.
        :param virtual_name: Virtual device name.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8805c499a7a0c5dd9580d5a1c8b85b4f543fea2c6e746e9d2815091e05d36c52)
            check_type(argname="argument ebs_device", value=ebs_device, expected_type=type_hints["ebs_device"])
            check_type(argname="argument virtual_name", value=virtual_name, expected_type=type_hints["virtual_name"])
        jsii.create(self.__class__, self, [ebs_device, virtual_name])

    @jsii.member(jsii_name="ebs")
    @builtins.classmethod
    def ebs(
        cls,
        volume_size: jsii.Number,
        *,
        encrypted: typing.Optional[builtins.bool] = None,
        kms_key: typing.Optional[_IKey_5f11635f] = None,
        delete_on_termination: typing.Optional[builtins.bool] = None,
        iops: typing.Optional[jsii.Number] = None,
        throughput: typing.Optional[jsii.Number] = None,
        volume_type: typing.Optional["EbsDeviceVolumeType"] = None,
    ) -> "BlockDeviceVolume":
        '''Creates a new Elastic Block Storage device.

        :param volume_size: The volume size, in Gibibytes (GiB).
        :param encrypted: Specifies whether the EBS volume is encrypted. Encrypted EBS volumes can only be attached to instances that support Amazon EBS encryption Default: false
        :param kms_key: The ARN of the AWS Key Management Service (AWS KMS) CMK used for encryption. You have to ensure that the KMS CMK has the correct permissions to be used by the service launching the ec2 instances. Default: - If encrypted is true, the default aws/ebs KMS key will be used.
        :param delete_on_termination: Indicates whether to delete the volume when the instance is terminated. Default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        :param iops: The number of I/O operations per second (IOPS) to provision for the volume. Must only be set for ``volumeType``: ``EbsDeviceVolumeType.IO1`` The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS, you need at least 100 GiB storage on the volume. Default: - none, required for ``EbsDeviceVolumeType.IO1``
        :param throughput: The throughput to provision for a ``gp3`` volume. Valid Range: Minimum value of 125. Maximum value of 1000. ``gp3`` volumes deliver a consistent baseline throughput performance of 125 MiB/s. You can provision additional throughput for an additional cost at a ratio of 0.25 MiB/s per provisioned IOPS. Default: - 125 MiB/s.
        :param volume_type: The EBS volume type. Default: ``EbsDeviceVolumeType.GENERAL_PURPOSE_SSD`` or ``EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3`` if ``@aws-cdk/aws-ec2:ebsDefaultGp3Volume`` is enabled.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e34634be08d573bd7d4eca032a2a66e619c889f2b51e62608777ae0db2639e6a)
            check_type(argname="argument volume_size", value=volume_size, expected_type=type_hints["volume_size"])
        options = EbsDeviceOptions(
            encrypted=encrypted,
            kms_key=kms_key,
            delete_on_termination=delete_on_termination,
            iops=iops,
            throughput=throughput,
            volume_type=volume_type,
        )

        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ebs", [volume_size, options]))

    @jsii.member(jsii_name="ebsFromSnapshot")
    @builtins.classmethod
    def ebs_from_snapshot(
        cls,
        snapshot_id: builtins.str,
        *,
        volume_size: typing.Optional[jsii.Number] = None,
        delete_on_termination: typing.Optional[builtins.bool] = None,
        iops: typing.Optional[jsii.Number] = None,
        throughput: typing.Optional[jsii.Number] = None,
        volume_type: typing.Optional["EbsDeviceVolumeType"] = None,
    ) -> "BlockDeviceVolume":
        '''Creates a new Elastic Block Storage device from an existing snapshot.

        :param snapshot_id: The snapshot ID of the volume to use.
        :param volume_size: The volume size, in Gibibytes (GiB). If you specify volumeSize, it must be equal or greater than the size of the snapshot. Default: - The snapshot size
        :param delete_on_termination: Indicates whether to delete the volume when the instance is terminated. Default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        :param iops: The number of I/O operations per second (IOPS) to provision for the volume. Must only be set for ``volumeType``: ``EbsDeviceVolumeType.IO1`` The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS, you need at least 100 GiB storage on the volume. Default: - none, required for ``EbsDeviceVolumeType.IO1``
        :param throughput: The throughput to provision for a ``gp3`` volume. Valid Range: Minimum value of 125. Maximum value of 1000. ``gp3`` volumes deliver a consistent baseline throughput performance of 125 MiB/s. You can provision additional throughput for an additional cost at a ratio of 0.25 MiB/s per provisioned IOPS. Default: - 125 MiB/s.
        :param volume_type: The EBS volume type. Default: ``EbsDeviceVolumeType.GENERAL_PURPOSE_SSD`` or ``EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3`` if ``@aws-cdk/aws-ec2:ebsDefaultGp3Volume`` is enabled.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ce9e60bc1da1c6f1d133c0cfaad0c5984d2f7a428c12f02640d8f31c6eab041e)
            check_type(argname="argument snapshot_id", value=snapshot_id, expected_type=type_hints["snapshot_id"])
        options = EbsDeviceSnapshotOptions(
            volume_size=volume_size,
            delete_on_termination=delete_on_termination,
            iops=iops,
            throughput=throughput,
            volume_type=volume_type,
        )

        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ebsFromSnapshot", [snapshot_id, options]))

    @jsii.member(jsii_name="ephemeral")
    @builtins.classmethod
    def ephemeral(cls, volume_index: jsii.Number) -> "BlockDeviceVolume":
        '''Creates a virtual, ephemeral device.

        The name will be in the form ephemeral{volumeIndex}.

        :param volume_index: the volume index. Must be equal or greater than 0
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3bf1e1313f470f70824bf7224f1fd91d59ad3cef90451484660764b691d8481)
            check_type(argname="argument volume_index", value=volume_index, expected_type=type_hints["volume_index"])
        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ephemeral", [volume_index]))

    @builtins.property
    @jsii.member(jsii_name="ebsDevice")
    def ebs_device(self) -> typing.Optional["EbsDeviceProps"]:
        '''EBS device info.'''
        return typing.cast(typing.Optional["EbsDeviceProps"], jsii.get(self, "ebsDevice"))

    @builtins.property
    @jsii.member(jsii_name="virtualName")
    def virtual_name(self) -> typing.Optional[builtins.str]:
        '''Virtual device name.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualName"))


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnCapacityReservation(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnCapacityReservation",
):
    '''Creates a new Capacity Reservation with the specified attributes.

    For more information, see `Capacity Reservations <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html>`_ in the *Amazon EC2 User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html
    :cloudformationResource: AWS::EC2::CapacityReservation
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_capacity_reservation = ec2.CfnCapacityReservation(self, "MyCfnCapacityReservation",
            instance_count=123,
            instance_platform="instancePlatform",
            instance_type="instanceType",
        
            # the properties below are optional
            availability_zone="availabilityZone",
            availability_zone_id="availabilityZoneId",
            ebs_optimized=False,
            end_date="endDate",
            end_date_type="endDateType",
            ephemeral_storage=False,
            instance_match_criteria="instanceMatchCriteria",
            out_post_arn="outPostArn",
            placement_group_arn="placementGroupArn",
            tag_specifications=[ec2.CfnCapacityReservation.TagSpecificationProperty(
                resource_type="resourceType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )],
            tenancy="tenancy",
            unused_reservation_billing_owner_id="unusedReservationBillingOwnerId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        instance_count: jsii.Number,
        instance_platform: builtins.str,
        instance_type: builtins.str,
        availability_zone: typing.Optional[builtins.str] = None,
        availability_zone_id: typing.Optional[builtins.str] = None,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        end_date: typing.Optional[builtins.str] = None,
        end_date_type: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        instance_match_criteria: typing.Optional[builtins.str] = None,
        out_post_arn: typing.Optional[builtins.str] = None,
        placement_group_arn: typing.Optional[builtins.str] = None,
        tag_specifications: typing.Optional[typing.Sequence[typing.Union["CfnCapacityReservation.TagSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tenancy: typing.Optional[builtins.str] = None,
        unused_reservation_billing_owner_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param instance_count: The number of instances for which to reserve capacity. .. epigraph:: You can request future-dated Capacity Reservations for an instance count with a minimum of 100 vCPUs. For example, if you request a future-dated Capacity Reservation for ``m5.xlarge`` instances, you must request at least 25 instances ( *25 * m5.xlarge = 100 vCPUs* ). Valid range: 1 - 1000
        :param instance_platform: The type of operating system for which to reserve capacity.
        :param instance_type: The instance type for which to reserve capacity. .. epigraph:: You can request future-dated Capacity Reservations for instance types in the C, M, R, I, and T instance families only. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* .
        :param availability_zone: The Availability Zone in which to create the Capacity Reservation.
        :param availability_zone_id: The Availability Zone ID of the Capacity Reservation.
        :param ebs_optimized: Indicates whether the Capacity Reservation supports EBS-optimized instances. This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using an EBS- optimized instance.
        :param end_date: The date and time at which the Capacity Reservation expires. When a Capacity Reservation expires, the reserved capacity is released and you can no longer launch instances into it. The Capacity Reservation's state changes to ``expired`` when it reaches its end date and time. You must provide an ``EndDate`` value if ``EndDateType`` is ``limited`` . Omit ``EndDate`` if ``EndDateType`` is ``unlimited`` . If the ``EndDateType`` is ``limited`` , the Capacity Reservation is cancelled within an hour from the specified time. For example, if you specify 5/31/2019, 13:30:55, the Capacity Reservation is guaranteed to end between 13:30:55 and 14:30:55 on 5/31/2019. If you are requesting a future-dated Capacity Reservation, you can't specify an end date and time that is within the commitment duration.
        :param end_date_type: Indicates the way in which the Capacity Reservation ends. A Capacity Reservation can have one of the following end types: - ``unlimited`` - The Capacity Reservation remains active until you explicitly cancel it. Do not provide an ``EndDate`` if the ``EndDateType`` is ``unlimited`` . - ``limited`` - The Capacity Reservation expires automatically at a specified date and time. You must provide an ``EndDate`` value if the ``EndDateType`` value is ``limited`` .
        :param ephemeral_storage: *Deprecated.*.
        :param instance_match_criteria: Indicates the type of instance launches that the Capacity Reservation accepts. The options include:. - ``open`` - The Capacity Reservation automatically matches all instances that have matching attributes (instance type, platform, and Availability Zone). Instances that have matching attributes run in the Capacity Reservation automatically without specifying any additional parameters. - ``targeted`` - The Capacity Reservation only accepts instances that have matching attributes (instance type, platform, and Availability Zone), and explicitly target the Capacity Reservation. This ensures that only permitted instances can use the reserved capacity. .. epigraph:: If you are requesting a future-dated Capacity Reservation, you must specify ``targeted`` . Default: ``open``
        :param out_post_arn: .. epigraph:: Not supported for future-dated Capacity Reservations. The Amazon Resource Name (ARN) of the Outpost on which to create the Capacity Reservation.
        :param placement_group_arn: .. epigraph:: Not supported for future-dated Capacity Reservations. The Amazon Resource Name (ARN) of the cluster placement group in which to create the Capacity Reservation. For more information, see `Capacity Reservations for cluster placement groups <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/cr-cpg.html>`_ in the *Amazon EC2 User Guide* .
        :param tag_specifications: The tags to apply to the Capacity Reservation during launch.
        :param tenancy: Indicates the tenancy of the Capacity Reservation. A Capacity Reservation can have one of the following tenancy settings:. - ``default`` - The Capacity Reservation is created on hardware that is shared with other AWS accounts . - ``dedicated`` - The Capacity Reservation is created on single-tenant hardware that is dedicated to a single AWS account .
        :param unused_reservation_billing_owner_id: The ID of the AWS account to which to assign billing of the unused capacity of the Capacity Reservation. A request will be sent to the specified account. That account must accept the request for the billing to be assigned to their account. For more information, see `Billing assignment for shared Amazon EC2 Capacity Reservations <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/assign-billing.html>`_ . You can assign billing only for shared Capacity Reservations. To share a Capacity Reservation, you must add it to a resource share. For more information, see `AWS::RAM::ResourceShare <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ram-resourceshare.html>`_ .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__96fb3bc559aaa9df971e86ea7cdd3cdc3de550019a2d3bf247d3fb169b5e9f7e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnCapacityReservationProps(
            instance_count=instance_count,
            instance_platform=instance_platform,
            instance_type=instance_type,
            availability_zone=availability_zone,
            availability_zone_id=availability_zone_id,
            ebs_optimized=ebs_optimized,
            end_date=end_date,
            end_date_type=end_date_type,
            ephemeral_storage=ephemeral_storage,
            instance_match_criteria=instance_match_criteria,
            out_post_arn=out_post_arn,
            placement_group_arn=placement_group_arn,
            tag_specifications=tag_specifications,
            tenancy=tenancy,
            unused_reservation_billing_owner_id=unused_reservation_billing_owner_id,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ac561edf867860b2e6780f7d06fe916bfb499cbbdab9e47874e784f8ee13259f)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4b837e3858811127cec4a1a49ef5fbc9c4920b9ca228950a0e8797ab41d9a406)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrAvailabilityZone")
    def attr_availability_zone(self) -> builtins.str:
        '''Returns the Availability Zone in which the capacity is reserved.

        For example: ``us-east-1a`` .

        :cloudformationAttribute: AvailabilityZone
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAvailabilityZone"))

    @builtins.property
    @jsii.member(jsii_name="attrAvailableInstanceCount")
    def attr_available_instance_count(self) -> jsii.Number:
        '''Returns the remaining capacity, which indicates the number of instances that can be launched in the Capacity Reservation.

        For example: ``9`` .

        :cloudformationAttribute: AvailableInstanceCount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrAvailableInstanceCount"))

    @builtins.property
    @jsii.member(jsii_name="attrCapacityAllocationSet")
    def attr_capacity_allocation_set(self) -> _IResolvable_da3f097b:
        '''
        :cloudformationAttribute: CapacityAllocationSet
        '''
        return typing.cast(_IResolvable_da3f097b, jsii.get(self, "attrCapacityAllocationSet"))

    @builtins.property
    @jsii.member(jsii_name="attrCapacityReservationArn")
    def attr_capacity_reservation_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the Capacity Reservation.

        :cloudformationAttribute: CapacityReservationArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCapacityReservationArn"))

    @builtins.property
    @jsii.member(jsii_name="attrCapacityReservationFleetId")
    def attr_capacity_reservation_fleet_id(self) -> builtins.str:
        '''The ID of the Capacity Reservation Fleet to which the Capacity Reservation belongs.

        Only valid for Capacity Reservations that were created by a Capacity Reservation Fleet.

        :cloudformationAttribute: CapacityReservationFleetId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCapacityReservationFleetId"))

    @builtins.property
    @jsii.member(jsii_name="attrCommitmentInfo")
    def attr_commitment_info(self) -> _IResolvable_da3f097b:
        '''
        :cloudformationAttribute: CommitmentInfo
        '''
        return typing.cast(_IResolvable_da3f097b, jsii.get(self, "attrCommitmentInfo"))

    @builtins.property
    @jsii.member(jsii_name="attrCreateDate")
    def attr_create_date(self) -> builtins.str:
        '''The date and time at which the Capacity Reservation was created.

        :cloudformationAttribute: CreateDate
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreateDate"))

    @builtins.property
    @jsii.member(jsii_name="attrDeliveryPreference")
    def attr_delivery_preference(self) -> builtins.str:
        '''The delivery method for a future-dated Capacity Reservation.

        ``incremental`` indicates that the requested capacity is delivered in addition to any running instances and reserved capacity that you have in your account at the requested date and time.

        :cloudformationAttribute: DeliveryPreference
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDeliveryPreference"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID of the Capacity Reservation.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrInstanceType")
    def attr_instance_type(self) -> builtins.str:
        '''Returns the type of instance for which the capacity is reserved.

        For example: ``m4.large`` .

        :cloudformationAttribute: InstanceType
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrInstanceType"))

    @builtins.property
    @jsii.member(jsii_name="attrOwnerId")
    def attr_owner_id(self) -> builtins.str:
        '''The ID of the AWS account that owns the Capacity Reservation.

        :cloudformationAttribute: OwnerId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrOwnerId"))

    @builtins.property
    @jsii.member(jsii_name="attrReservationType")
    def attr_reservation_type(self) -> builtins.str:
        '''The type of Capacity Reservation.

        :cloudformationAttribute: ReservationType
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrReservationType"))

    @builtins.property
    @jsii.member(jsii_name="attrStartDate")
    def attr_start_date(self) -> builtins.str:
        '''The date and time at which the Capacity Reservation was started.

        :cloudformationAttribute: StartDate
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrStartDate"))

    @builtins.property
    @jsii.member(jsii_name="attrState")
    def attr_state(self) -> builtins.str:
        '''The current state of the Capacity Reservation. A Capacity Reservation can be in one of the following states:.

        - ``active`` - The capacity is available for use.
        - ``expired`` - The Capacity Reservation expired automatically at the date and time specified in your reservation request. The reserved capacity is no longer available for your use.
        - ``cancelled`` - The Capacity Reservation was canceled. The reserved capacity is no longer available for your use.
        - ``pending`` - The Capacity Reservation request was successful but the capacity provisioning is still pending.
        - ``failed`` - The Capacity Reservation request has failed. A request can fail due to request parameters that are not valid, capacity constraints, or instance limit constraints. You can view a failed request for 60 minutes.
        - ``scheduled`` - ( *Future-dated Capacity Reservations* ) The future-dated Capacity Reservation request was approved and the Capacity Reservation is scheduled for delivery on the requested start date.
        - ``payment-pending`` - ( *Capacity Blocks* ) The upfront payment has not been processed yet.
        - ``payment-failed`` - ( *Capacity Blocks* ) The upfront payment was not processed in the 12-hour time frame. Your Capacity Block was released.
        - ``assessing`` - ( *Future-dated Capacity Reservations* ) Amazon EC2 is assessing your request for a future-dated Capacity Reservation.
        - ``delayed`` - ( *Future-dated Capacity Reservations* ) Amazon EC2 encountered a delay in provisioning the requested future-dated Capacity Reservation. Amazon EC2 is unable to deliver the requested capacity by the requested start date and time.
        - ``unsupported`` - ( *Future-dated Capacity Reservations* ) Amazon EC2 can't support the future-dated Capacity Reservation request due to capacity constraints. You can view unsupported requests for 30 days. The Capacity Reservation will not be delivered.

        :cloudformationAttribute: State
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrState"))

    @builtins.property
    @jsii.member(jsii_name="attrTenancy")
    def attr_tenancy(self) -> builtins.str:
        '''Returns the tenancy of the Capacity Reservation.

        For example: ``dedicated`` .

        :cloudformationAttribute: Tenancy
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrTenancy"))

    @builtins.property
    @jsii.member(jsii_name="attrTotalInstanceCount")
    def attr_total_instance_count(self) -> jsii.Number:
        '''Returns the total number of instances for which the Capacity Reservation reserves capacity.

        For example: ``15`` .

        :cloudformationAttribute: TotalInstanceCount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrTotalInstanceCount"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="instanceCount")
    def instance_count(self) -> jsii.Number:
        '''The number of instances for which to reserve capacity.'''
        return typing.cast(jsii.Number, jsii.get(self, "instanceCount"))

    @instance_count.setter
    def instance_count(self, value: jsii.Number) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__22b368a04915e99dd5744c69aee4e5f18ce694f374f86d57efdd11669b182094)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceCount", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="instancePlatform")
    def instance_platform(self) -> builtins.str:
        '''The type of operating system for which to reserve capacity.'''
        return typing.cast(builtins.str, jsii.get(self, "instancePlatform"))

    @instance_platform.setter
    def instance_platform(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__61255c59c261f39ea55478e744de40ef81c899601741a479966b55964910425d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instancePlatform", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> builtins.str:
        '''The instance type for which to reserve capacity.'''
        return typing.cast(builtins.str, jsii.get(self, "instanceType"))

    @instance_type.setter
    def instance_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__77123d0c19c0832f7509cd76b26aa9299b7596606999f89281b470a4f13db290)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="availabilityZone")
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''The Availability Zone in which to create the Capacity Reservation.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "availabilityZone"))

    @availability_zone.setter
    def availability_zone(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c78e4680a157e3466009de77e8aa625d8cf7fee8e6f0094ed98d312434e17d0f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "availabilityZone", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="availabilityZoneId")
    def availability_zone_id(self) -> typing.Optional[builtins.str]:
        '''The Availability Zone ID of the Capacity Reservation.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "availabilityZoneId"))

    @availability_zone_id.setter
    def availability_zone_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3491534eeb43f169595d5744c952b4e6e91b59d668a8f0213de96d7e46d772f0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "availabilityZoneId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="ebsOptimized")
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether the Capacity Reservation supports EBS-optimized instances.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "ebsOptimized"))

    @ebs_optimized.setter
    def ebs_optimized(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__678dbc8559016cf13cc067fc0b1a2efbca67b43c58a19ad6cb861f9f92eaffde)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ebsOptimized", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="endDate")
    def end_date(self) -> typing.Optional[builtins.str]:
        '''The date and time at which the Capacity Reservation expires.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "endDate"))

    @end_date.setter
    def end_date(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d4cb6531edd4f2f2d84580cab98baebb9524704a86fe4420ac1fedab59c3f11a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endDate", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="endDateType")
    def end_date_type(self) -> typing.Optional[builtins.str]:
        '''Indicates the way in which the Capacity Reservation ends.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "endDateType"))

    @end_date_type.setter
    def end_date_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5ab7a3bc671d2cc1cbe9b8533b6cf3f81fbcdd7c629787d6fbcf162a2f7991b9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endDateType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="ephemeralStorage")
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''*Deprecated.*.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "ephemeralStorage"))

    @ephemeral_storage.setter
    def ephemeral_storage(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e71563d063a54b01a1641ea15d2304c47a9442638bf103f23bb2279e72a805b0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ephemeralStorage", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="instanceMatchCriteria")
    def instance_match_criteria(self) -> typing.Optional[builtins.str]:
        '''Indicates the type of instance launches that the Capacity Reservation accepts.

        The options include:.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceMatchCriteria"))

    @instance_match_criteria.setter
    def instance_match_criteria(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f474d13d66628cd099f8b6e63a960fb1e11e349a6e30dd4bcea58da40fde501a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceMatchCriteria", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="outPostArn")
    def out_post_arn(self) -> typing.Optional[builtins.str]:
        '''.. epigraph::

   Not supported for future-dated Capacity Reservations.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "outPostArn"))

    @out_post_arn.setter
    def out_post_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0a8dbcf49c490eda04d32bf6b351f510a5d071b5754a19fa2c2b95c6ce0c22f0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "outPostArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="placementGroupArn")
    def placement_group_arn(self) -> typing.Optional[builtins.str]:
        '''.. epigraph::

   Not supported for future-dated Capacity Reservations.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "placementGroupArn"))

    @placement_group_arn.setter
    def placement_group_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__16fa52eb55faf78ebc17e0e8d27a241715110e3c4255fe91c5b59c36ac83d329)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "placementGroupArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.List["CfnCapacityReservation.TagSpecificationProperty"]]:
        '''The tags to apply to the Capacity Reservation during launch.'''
        return typing.cast(typing.Optional[typing.List["CfnCapacityReservation.TagSpecificationProperty"]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.List["CfnCapacityReservation.TagSpecificationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d2d4ed674b30189cbf7b11b79e68ea556b2b7c9921a4d61f7610b58d25fed8ca)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagSpecifications", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tenancy")
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''Indicates the tenancy of the Capacity Reservation.

        A Capacity Reservation can have one of the following tenancy settings:.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "tenancy"))

    @tenancy.setter
    def tenancy(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__26e29c48a6cb47934fcf7b54e3d3eed16da0c88c8d717089ac043c03ebac0b5b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tenancy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="unusedReservationBillingOwnerId")
    def unused_reservation_billing_owner_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the AWS account to which to assign billing of the unused capacity of the Capacity Reservation.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "unusedReservationBillingOwnerId"))

    @unused_reservation_billing_owner_id.setter
    def unused_reservation_billing_owner_id(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a09cfe18a64a35ca3513da8b832d14a3961e5101708c3d59880377b4beea919)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "unusedReservationBillingOwnerId", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnCapacityReservation.CapacityAllocationProperty",
        jsii_struct_bases=[],
        name_mapping={"allocation_type": "allocationType", "count": "count"},
    )
    class CapacityAllocationProperty:
        def __init__(
            self,
            *,
            allocation_type: typing.Optional[builtins.str] = None,
            count: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Information about instance capacity usage for a Capacity Reservation.

            :param allocation_type: The usage type. ``used`` indicates that the instance capacity is in use by instances that are running in the Capacity Reservation.
            :param count: The amount of instance capacity associated with the usage. For example a value of ``4`` indicates that instance capacity for 4 instances is currently in use.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-capacityallocation.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                capacity_allocation_property = ec2.CfnCapacityReservation.CapacityAllocationProperty(
                    allocation_type="allocationType",
                    count=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bb73541f409f83260eb045bf4e0e3879b1c1c509522736f018397f8ac1cf955c)
                check_type(argname="argument allocation_type", value=allocation_type, expected_type=type_hints["allocation_type"])
                check_type(argname="argument count", value=count, expected_type=type_hints["count"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if allocation_type is not None:
                self._values["allocation_type"] = allocation_type
            if count is not None:
                self._values["count"] = count

        @builtins.property
        def allocation_type(self) -> typing.Optional[builtins.str]:
            '''The usage type.

            ``used`` indicates that the instance capacity is in use by instances that are running in the Capacity Reservation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-capacityallocation.html#cfn-ec2-capacityreservation-capacityallocation-allocationtype
            '''
            result = self._values.get("allocation_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def count(self) -> typing.Optional[jsii.Number]:
            '''The amount of instance capacity associated with the usage.

            For example a value of ``4`` indicates that instance capacity for 4 instances is currently in use.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-capacityallocation.html#cfn-ec2-capacityreservation-capacityallocation-count
            '''
            result = self._values.get("count")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityAllocationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnCapacityReservation.CommitmentInfoProperty",
        jsii_struct_bases=[],
        name_mapping={
            "commitment_end_date": "commitmentEndDate",
            "committed_instance_count": "committedInstanceCount",
        },
    )
    class CommitmentInfoProperty:
        def __init__(
            self,
            *,
            commitment_end_date: typing.Optional[builtins.str] = None,
            committed_instance_count: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Information about your commitment for a future-dated Capacity Reservation.

            :param commitment_end_date: The date and time at which the commitment duration expires, in the ISO8601 format in the UTC time zone ( ``YYYY-MM-DDThh:mm:ss.sssZ`` ). You can't decrease the instance count or cancel the Capacity Reservation before this date and time.
            :param committed_instance_count: The instance capacity that you committed to when you requested the future-dated Capacity Reservation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-commitmentinfo.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                commitment_info_property = ec2.CfnCapacityReservation.CommitmentInfoProperty(
                    commitment_end_date="commitmentEndDate",
                    committed_instance_count=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3c56a4c8d14f2a52b13c3cf61a95ca1ca01e35305d2313a0778cf6fad31ac4ef)
                check_type(argname="argument commitment_end_date", value=commitment_end_date, expected_type=type_hints["commitment_end_date"])
                check_type(argname="argument committed_instance_count", value=committed_instance_count, expected_type=type_hints["committed_instance_count"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if commitment_end_date is not None:
                self._values["commitment_end_date"] = commitment_end_date
            if committed_instance_count is not None:
                self._values["committed_instance_count"] = committed_instance_count

        @builtins.property
        def commitment_end_date(self) -> typing.Optional[builtins.str]:
            '''The date and time at which the commitment duration expires, in the ISO8601 format in the UTC time zone ( ``YYYY-MM-DDThh:mm:ss.sssZ`` ). You can't decrease the instance count or cancel the Capacity Reservation before this date and time.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-commitmentinfo.html#cfn-ec2-capacityreservation-commitmentinfo-commitmentenddate
            '''
            result = self._values.get("commitment_end_date")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def committed_instance_count(self) -> typing.Optional[jsii.Number]:
            '''The instance capacity that you committed to when you requested the future-dated Capacity Reservation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-commitmentinfo.html#cfn-ec2-capacityreservation-commitmentinfo-committedinstancecount
            '''
            result = self._values.get("committed_instance_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CommitmentInfoProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnCapacityReservation.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: typing.Optional[builtins.str] = None,
            tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An array of key-value pairs to apply to this resource.

            For more information, see `Tag <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-resource-tags.html>`_ .

            :param resource_type: The type of resource to tag. Specify ``capacity-reservation`` .
            :param tags: The tags to apply to the resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-tagspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                tag_specification_property = ec2.CfnCapacityReservation.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__578daf872c6424406c4ac67bfb16e1a373fb40f41078950b64a62c991d0be846)
                check_type(argname="argument resource_type", value=resource_type, expected_type=type_hints["resource_type"])
                check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if resource_type is not None:
                self._values["resource_type"] = resource_type
            if tags is not None:
                self._values["tags"] = tags

        @builtins.property
        def resource_type(self) -> typing.Optional[builtins.str]:
            '''The type of resource to tag.

            Specify ``capacity-reservation`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-tagspecification.html#cfn-ec2-capacityreservation-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
            '''The tags to apply to the resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-tagspecification.html#cfn-ec2-capacityreservation-tagspecification-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556)
class CfnCapacityReservationFleet(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnCapacityReservationFleet",
):
    '''Creates a new Capacity Reservation Fleet with the specified attributes.

    For more information, see `Capacity Reservation Fleets <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/cr-fleets.html>`_ in the *Amazon EC2 User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html
    :cloudformationResource: AWS::EC2::CapacityReservationFleet
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_capacity_reservation_fleet = ec2.CfnCapacityReservationFleet(self, "MyCfnCapacityReservationFleet",
            allocation_strategy="allocationStrategy",
            end_date="endDate",
            instance_match_criteria="instanceMatchCriteria",
            instance_type_specifications=[ec2.CfnCapacityReservationFleet.InstanceTypeSpecificationProperty(
                availability_zone="availabilityZone",
                availability_zone_id="availabilityZoneId",
                ebs_optimized=False,
                instance_platform="instancePlatform",
                instance_type="instanceType",
                priority=123,
                weight=123
            )],
            no_remove_end_date=False,
            remove_end_date=False,
            tag_specifications=[ec2.CfnCapacityReservationFleet.TagSpecificationProperty(
                resource_type="resourceType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )],
            tenancy="tenancy",
            total_target_capacity=123
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        allocation_strategy: typing.Optional[builtins.str] = None,
        end_date: typing.Optional[builtins.str] = None,
        instance_match_criteria: typing.Optional[builtins.str] = None,
        instance_type_specifications: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCapacityReservationFleet.InstanceTypeSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        no_remove_end_date: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        remove_end_date: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        tag_specifications: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnCapacityReservationFleet.TagSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        tenancy: typing.Optional[builtins.str] = None,
        total_target_capacity: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param allocation_strategy: The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to use. Currently, only the ``prioritized`` allocation strategy is supported. For more information, see `Allocation strategy <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy>`_ in the *Amazon EC2 User Guide* . Valid values: ``prioritized``
        :param end_date: The date and time at which the Capacity Reservation Fleet expires. When the Capacity Reservation Fleet expires, its state changes to ``expired`` and all of the Capacity Reservations in the Fleet expire. The Capacity Reservation Fleet expires within an hour after the specified time. For example, if you specify ``5/31/2019`` , ``13:30:55`` , the Capacity Reservation Fleet is guaranteed to expire between ``13:30:55`` and ``14:30:55`` on ``5/31/2019`` .
        :param instance_match_criteria: Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity Reservations in the Fleet inherit this instance matching criteria. Currently, Capacity Reservation Fleets support ``open`` instance matching criteria only. This means that instances that have matching attributes (instance type, platform, and Availability Zone) run in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet to use its reserved capacity.
        :param instance_type_specifications: Information about the instance types for which to reserve the capacity.
        :param no_remove_end_date: Used to add an end date to a Capacity Reservation Fleet that has no end date and time. To add an end date to a Capacity Reservation Fleet, specify ``true`` for this paramater and specify the end date and time (in UTC time format) for the *EndDate* parameter.
        :param remove_end_date: Used to remove an end date from a Capacity Reservation Fleet that is configured to end automatically at a specific date and time. To remove the end date from a Capacity Reservation Fleet, specify ``true`` for this paramater and omit the *EndDate* parameter.
        :param tag_specifications: The tags to assign to the Capacity Reservation Fleet. The tags are automatically assigned to the Capacity Reservations in the Fleet.
        :param tenancy: Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet inherit this tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings: - ``default`` - The Capacity Reservation Fleet is created on hardware that is shared with other AWS accounts . - ``dedicated`` - The Capacity Reservations are created on single-tenant hardware that is dedicated to a single AWS account .
        :param total_target_capacity: The total number of capacity units to be reserved by the Capacity Reservation Fleet. This value, together with the instance type weights that you assign to each instance type used by the Fleet determine the number of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your workload. For more information, see `Total target capacity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity>`_ in the *Amazon EC2 User Guide* .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f6fb1af1f639a354aff6bb978bcac42ab042448e257770c8bed7a46897429542)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnCapacityReservationFleetProps(
            allocation_strategy=allocation_strategy,
            end_date=end_date,
            instance_match_criteria=instance_match_criteria,
            instance_type_specifications=instance_type_specifications,
            no_remove_end_date=no_remove_end_date,
            remove_end_date=remove_end_date,
            tag_specifications=tag_specifications,
            tenancy=tenancy,
            total_target_capacity=total_target_capacity,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35bd03d12bd6116122e660dadd70f4c4735a841ce1b9a702a966cfc70b2123d1)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__12b0be9e7203b7b337996367bda0b1ef3d17273516ce9a44f5f9c1be5c34c081)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCapacityReservationFleetId")
    def attr_capacity_reservation_fleet_id(self) -> builtins.str:
        '''The ID of the Capacity Reservation Fleet.

        :cloudformationAttribute: CapacityReservationFleetId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCapacityReservationFleetId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="allocationStrategy")
    def allocation_strategy(self) -> typing.Optional[builtins.str]:
        '''The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to use.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "allocationStrategy"))

    @allocation_strategy.setter
    def allocation_strategy(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ff7425d803e147608940b2d4d46fa62c8f6d398dfda2f81c4d9daae76caa17bc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allocationStrategy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="endDate")
    def end_date(self) -> typing.Optional[builtins.str]:
        '''The date and time at which the Capacity Reservation Fleet expires.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "endDate"))

    @end_date.setter
    def end_date(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9be4232e54cc46f4b116770555b9dd460cb1f802b7496077e8e31ba3afd7cbff)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endDate", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="instanceMatchCriteria")
    def instance_match_criteria(self) -> typing.Optional[builtins.str]:
        '''Indicates the type of instance launches that the Capacity Reservation Fleet accepts.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceMatchCriteria"))

    @instance_match_criteria.setter
    def instance_match_criteria(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__026984f5fc35753acd6fb74696c8208736dc28838b389118f458535c5762a672)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceMatchCriteria", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="instanceTypeSpecifications")
    def instance_type_specifications(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCapacityReservationFleet.InstanceTypeSpecificationProperty"]]]]:
        '''Information about the instance types for which to reserve the capacity.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCapacityReservationFleet.InstanceTypeSpecificationProperty"]]]], jsii.get(self, "instanceTypeSpecifications"))

    @instance_type_specifications.setter
    def instance_type_specifications(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCapacityReservationFleet.InstanceTypeSpecificationProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__45208597646253fe188d8f1f2e5d2fa3ecdd988899be6ea30229a2132738df76)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceTypeSpecifications", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="noRemoveEndDate")
    def no_remove_end_date(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Used to add an end date to a Capacity Reservation Fleet that has no end date and time.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "noRemoveEndDate"))

    @no_remove_end_date.setter
    def no_remove_end_date(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a98456c361af7e60a4e64d845a4bbe28b249e03540ffe9d59f96245e0f292752)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "noRemoveEndDate", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="removeEndDate")
    def remove_end_date(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Used to remove an end date from a Capacity Reservation Fleet that is configured to end automatically at a specific date and time.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "removeEndDate"))

    @remove_end_date.setter
    def remove_end_date(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6cfebbe0400b084370b4e45e01a83c61bb47ae01d37c14a2ac6591c4708c975a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "removeEndDate", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCapacityReservationFleet.TagSpecificationProperty"]]]]:
        '''The tags to assign to the Capacity Reservation Fleet.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCapacityReservationFleet.TagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnCapacityReservationFleet.TagSpecificationProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9be24701036c5abd754bc67058ec4d8c733250163e5a63577648325991d133e9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagSpecifications", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tenancy")
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''Indicates the tenancy of the Capacity Reservation Fleet.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "tenancy"))

    @tenancy.setter
    def tenancy(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7dbdd25c7a6e3e88f44f955dc9784e01a01af6c00df012edb23a78014e0caae4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tenancy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="totalTargetCapacity")
    def total_target_capacity(self) -> typing.Optional[jsii.Number]:
        '''The total number of capacity units to be reserved by the Capacity Reservation Fleet.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "totalTargetCapacity"))

    @total_target_capacity.setter
    def total_target_capacity(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b98cc8f5cd84b2ab7c2779d1b5d483afac7074f27afc44e507d761ee2987489d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "totalTargetCapacity", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnCapacityReservationFleet.InstanceTypeSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "availability_zone": "availabilityZone",
            "availability_zone_id": "availabilityZoneId",
            "ebs_optimized": "ebsOptimized",
            "instance_platform": "instancePlatform",
            "instance_type": "instanceType",
            "priority": "priority",
            "weight": "weight",
        },
    )
    class InstanceTypeSpecificationProperty:
        def __init__(
            self,
            *,
            availability_zone: typing.Optional[builtins.str] = None,
            availability_zone_id: typing.Optional[builtins.str] = None,
            ebs_optimized: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            instance_platform: typing.Optional[builtins.str] = None,
            instance_type: typing.Optional[builtins.str] = None,
            priority: typing.Optional[jsii.Number] = None,
            weight: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Specifies information about an instance type to use in a Capacity Reservation Fleet.

            ``InstanceTypeSpecification`` is a property of the `AWS::EC2::CapacityReservationFleet <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html>`_ resource.

            :param availability_zone: The Availability Zone in which the Capacity Reservation Fleet reserves the capacity. A Capacity Reservation Fleet can't span Availability Zones. All instance type specifications that you specify for the Fleet must use the same Availability Zone.
            :param availability_zone_id: The ID of the Availability Zone in which the Capacity Reservation Fleet reserves the capacity. A Capacity Reservation Fleet can't span Availability Zones. All instance type specifications that you specify for the Fleet must use the same Availability Zone.
            :param ebs_optimized: Indicates whether the Capacity Reservation Fleet supports EBS-optimized instances types. This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using EBS-optimized instance types.
            :param instance_platform: The type of operating system for which the Capacity Reservation Fleet reserves capacity.
            :param instance_type: The instance type for which the Capacity Reservation Fleet reserves capacity.
            :param priority: The priority to assign to the instance type. This value is used to determine which of the instance types specified for the Fleet should be prioritized for use. A lower value indicates a high priority. For more information, see `Instance type priority <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#instance-priority>`_ in the *Amazon EC2 User Guide* .
            :param weight: The number of capacity units provided by the specified instance type. This value, together with the total target capacity that you specify for the Fleet determine the number of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your workload. For more information, see `Total target capacity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity>`_ in the Amazon EC2 User Guide. Valid Range: Minimum value of ``0.001`` . Maximum value of ``99.999`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                instance_type_specification_property = ec2.CfnCapacityReservationFleet.InstanceTypeSpecificationProperty(
                    availability_zone="availabilityZone",
                    availability_zone_id="availabilityZoneId",
                    ebs_optimized=False,
                    instance_platform="instancePlatform",
                    instance_type="instanceType",
                    priority=123,
                    weight=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1f03a957255637b5857219e572d51eadfa10b8ad587bd37cf340d9e7c2231d21)
                check_type(argname="argument availability_zone", value=availability_zone, expected_type=type_hints["availability_zone"])
                check_type(argname="argument availability_zone_id", value=availability_zone_id, expected_type=type_hints["availability_zone_id"])
                check_type(argname="argument ebs_optimized", value=ebs_optimized, expected_type=type_hints["ebs_optimized"])
                check_type(argname="argument instance_platform", value=instance_platform, expected_type=type_hints["instance_platform"])
                check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
                check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
                check_type(argname="argument weight", value=weight, expected_type=type_hints["weight"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if availability_zone_id is not None:
                self._values["availability_zone_id"] = availability_zone_id
            if ebs_optimized is not None:
                self._values["ebs_optimized"] = ebs_optimized
            if instance_platform is not None:
                self._values["instance_platform"] = instance_platform
            if instance_type is not None:
                self._values["instance_type"] = instance_type
            if priority is not None:
                self._values["priority"] = priority
            if weight is not None:
                self._values["weight"] = weight

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''The Availability Zone in which the Capacity Reservation Fleet reserves the capacity.

            A Capacity Reservation Fleet can't span Availability Zones. All instance type specifications that you specify for the Fleet must use the same Availability Zone.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-availabilityzone
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def availability_zone_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the Availability Zone in which the Capacity Reservation Fleet reserves the capacity.

            A Capacity Reservation Fleet can't span Availability Zones. All instance type specifications that you specify for the Fleet must use the same Availability Zone.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-availabilityzoneid
            '''
            result = self._values.get("availability_zone_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ebs_optimized(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates whether the Capacity Reservation Fleet supports EBS-optimized instances types.

            This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using EBS-optimized instance types.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-ebsoptimized
            '''
            result = self._values.get("ebs_optimized")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def instance_platform(self) -> typing.Optional[builtins.str]:
            '''The type of operating system for which the Capacity Reservation Fleet reserves capacity.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-instanceplatform
            '''
            result = self._values.get("instance_platform")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_type(self) -> typing.Optional[builtins.str]:
            '''The instance type for which the Capacity Reservation Fleet reserves capacity.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-instancetype
            '''
            result = self._values.get("instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''The priority to assign to the instance type.

            This value is used to determine which of the instance types specified for the Fleet should be prioritized for use. A lower value indicates a high priority. For more information, see `Instance type priority <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#instance-priority>`_ in the *Amazon EC2 User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def weight(self) -> typing.Optional[jsii.Number]:
            '''The number of capacity units provided by the specified instance type.

            This value, together with the total target capacity that you specify for the Fleet determine the number of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your workload. For more information, see `Total target capacity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity>`_ in the Amazon EC2 User Guide.

            Valid Range: Minimum value of ``0.001`` . Maximum value of ``99.999`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-instancetypespecification.html#cfn-ec2-capacityreservationfleet-instancetypespecification-weight
            '''
            result = self._values.get("weight")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceTypeSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnCapacityReservationFleet.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: typing.Optional[builtins.str] = None,
            tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The tags to apply to a resource when the resource is being created.

            When you specify a tag, you must specify the resource type to tag, otherwise the request will fail.
            .. epigraph::

               The ``Valid Values`` lists all the resource types that can be tagged. However, the action you're using might not support tagging all of these resource types. If you try to tag a resource type that is unsupported for the action you're using, you'll get an error.

            :param resource_type: The type of resource to tag on creation. Specify ``capacity-reservation-fleet`` . To tag a resource after it has been created, see `CreateTags <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html>`_ .
            :param tags: The tags to apply to the resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-tagspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                tag_specification_property = ec2.CfnCapacityReservationFleet.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2494957f49f85b50a915390ad0dfbf5c119ba87d5b7f2a04a0d67766b4901d34)
                check_type(argname="argument resource_type", value=resource_type, expected_type=type_hints["resource_type"])
                check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if resource_type is not None:
                self._values["resource_type"] = resource_type
            if tags is not None:
                self._values["tags"] = tags

        @builtins.property
        def resource_type(self) -> typing.Optional[builtins.str]:
            '''The type of resource to tag on creation. Specify ``capacity-reservation-fleet`` .

            To tag a resource after it has been created, see `CreateTags <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-tagspecification.html#cfn-ec2-capacityreservationfleet-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
            '''The tags to apply to the resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservationfleet-tagspecification.html#cfn-ec2-capacityreservationfleet-tagspecification-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnCapacityReservationFleetProps",
    jsii_struct_bases=[],
    name_mapping={
        "allocation_strategy": "allocationStrategy",
        "end_date": "endDate",
        "instance_match_criteria": "instanceMatchCriteria",
        "instance_type_specifications": "instanceTypeSpecifications",
        "no_remove_end_date": "noRemoveEndDate",
        "remove_end_date": "removeEndDate",
        "tag_specifications": "tagSpecifications",
        "tenancy": "tenancy",
        "total_target_capacity": "totalTargetCapacity",
    },
)
class CfnCapacityReservationFleetProps:
    def __init__(
        self,
        *,
        allocation_strategy: typing.Optional[builtins.str] = None,
        end_date: typing.Optional[builtins.str] = None,
        instance_match_criteria: typing.Optional[builtins.str] = None,
        instance_type_specifications: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCapacityReservationFleet.InstanceTypeSpecificationProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        no_remove_end_date: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        remove_end_date: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        tag_specifications: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnCapacityReservationFleet.TagSpecificationProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        tenancy: typing.Optional[builtins.str] = None,
        total_target_capacity: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnCapacityReservationFleet``.

        :param allocation_strategy: The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to use. Currently, only the ``prioritized`` allocation strategy is supported. For more information, see `Allocation strategy <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy>`_ in the *Amazon EC2 User Guide* . Valid values: ``prioritized``
        :param end_date: The date and time at which the Capacity Reservation Fleet expires. When the Capacity Reservation Fleet expires, its state changes to ``expired`` and all of the Capacity Reservations in the Fleet expire. The Capacity Reservation Fleet expires within an hour after the specified time. For example, if you specify ``5/31/2019`` , ``13:30:55`` , the Capacity Reservation Fleet is guaranteed to expire between ``13:30:55`` and ``14:30:55`` on ``5/31/2019`` .
        :param instance_match_criteria: Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity Reservations in the Fleet inherit this instance matching criteria. Currently, Capacity Reservation Fleets support ``open`` instance matching criteria only. This means that instances that have matching attributes (instance type, platform, and Availability Zone) run in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet to use its reserved capacity.
        :param instance_type_specifications: Information about the instance types for which to reserve the capacity.
        :param no_remove_end_date: Used to add an end date to a Capacity Reservation Fleet that has no end date and time. To add an end date to a Capacity Reservation Fleet, specify ``true`` for this paramater and specify the end date and time (in UTC time format) for the *EndDate* parameter.
        :param remove_end_date: Used to remove an end date from a Capacity Reservation Fleet that is configured to end automatically at a specific date and time. To remove the end date from a Capacity Reservation Fleet, specify ``true`` for this paramater and omit the *EndDate* parameter.
        :param tag_specifications: The tags to assign to the Capacity Reservation Fleet. The tags are automatically assigned to the Capacity Reservations in the Fleet.
        :param tenancy: Indicates the tenancy of the Capacity Reservation Fleet. All Capacity Reservations in the Fleet inherit this tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings: - ``default`` - The Capacity Reservation Fleet is created on hardware that is shared with other AWS accounts . - ``dedicated`` - The Capacity Reservations are created on single-tenant hardware that is dedicated to a single AWS account .
        :param total_target_capacity: The total number of capacity units to be reserved by the Capacity Reservation Fleet. This value, together with the instance type weights that you assign to each instance type used by the Fleet determine the number of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your workload. For more information, see `Total target capacity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity>`_ in the *Amazon EC2 User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_capacity_reservation_fleet_props = ec2.CfnCapacityReservationFleetProps(
                allocation_strategy="allocationStrategy",
                end_date="endDate",
                instance_match_criteria="instanceMatchCriteria",
                instance_type_specifications=[ec2.CfnCapacityReservationFleet.InstanceTypeSpecificationProperty(
                    availability_zone="availabilityZone",
                    availability_zone_id="availabilityZoneId",
                    ebs_optimized=False,
                    instance_platform="instancePlatform",
                    instance_type="instanceType",
                    priority=123,
                    weight=123
                )],
                no_remove_end_date=False,
                remove_end_date=False,
                tag_specifications=[ec2.CfnCapacityReservationFleet.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )],
                tenancy="tenancy",
                total_target_capacity=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__43bd0a52cfe7402bf156151a3e760ed1bd133935bd29982a3798f854cbee5064)
            check_type(argname="argument allocation_strategy", value=allocation_strategy, expected_type=type_hints["allocation_strategy"])
            check_type(argname="argument end_date", value=end_date, expected_type=type_hints["end_date"])
            check_type(argname="argument instance_match_criteria", value=instance_match_criteria, expected_type=type_hints["instance_match_criteria"])
            check_type(argname="argument instance_type_specifications", value=instance_type_specifications, expected_type=type_hints["instance_type_specifications"])
            check_type(argname="argument no_remove_end_date", value=no_remove_end_date, expected_type=type_hints["no_remove_end_date"])
            check_type(argname="argument remove_end_date", value=remove_end_date, expected_type=type_hints["remove_end_date"])
            check_type(argname="argument tag_specifications", value=tag_specifications, expected_type=type_hints["tag_specifications"])
            check_type(argname="argument tenancy", value=tenancy, expected_type=type_hints["tenancy"])
            check_type(argname="argument total_target_capacity", value=total_target_capacity, expected_type=type_hints["total_target_capacity"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allocation_strategy is not None:
            self._values["allocation_strategy"] = allocation_strategy
        if end_date is not None:
            self._values["end_date"] = end_date
        if instance_match_criteria is not None:
            self._values["instance_match_criteria"] = instance_match_criteria
        if instance_type_specifications is not None:
            self._values["instance_type_specifications"] = instance_type_specifications
        if no_remove_end_date is not None:
            self._values["no_remove_end_date"] = no_remove_end_date
        if remove_end_date is not None:
            self._values["remove_end_date"] = remove_end_date
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if tenancy is not None:
            self._values["tenancy"] = tenancy
        if total_target_capacity is not None:
            self._values["total_target_capacity"] = total_target_capacity

    @builtins.property
    def allocation_strategy(self) -> typing.Optional[builtins.str]:
        '''The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to use.

        Currently, only the ``prioritized`` allocation strategy is supported. For more information, see `Allocation strategy <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy>`_ in the *Amazon EC2 User Guide* .

        Valid values: ``prioritized``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-allocationstrategy
        '''
        result = self._values.get("allocation_strategy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def end_date(self) -> typing.Optional[builtins.str]:
        '''The date and time at which the Capacity Reservation Fleet expires.

        When the Capacity Reservation Fleet expires, its state changes to ``expired`` and all of the Capacity Reservations in the Fleet expire.

        The Capacity Reservation Fleet expires within an hour after the specified time. For example, if you specify ``5/31/2019`` , ``13:30:55`` , the Capacity Reservation Fleet is guaranteed to expire between ``13:30:55`` and ``14:30:55`` on ``5/31/2019`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-enddate
        '''
        result = self._values.get("end_date")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_match_criteria(self) -> typing.Optional[builtins.str]:
        '''Indicates the type of instance launches that the Capacity Reservation Fleet accepts.

        All Capacity Reservations in the Fleet inherit this instance matching criteria.

        Currently, Capacity Reservation Fleets support ``open`` instance matching criteria only. This means that instances that have matching attributes (instance type, platform, and Availability Zone) run in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet to use its reserved capacity.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-instancematchcriteria
        '''
        result = self._values.get("instance_match_criteria")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_type_specifications(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnCapacityReservationFleet.InstanceTypeSpecificationProperty]]]]:
        '''Information about the instance types for which to reserve the capacity.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-instancetypespecifications
        '''
        result = self._values.get("instance_type_specifications")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnCapacityReservationFleet.InstanceTypeSpecificationProperty]]]], result)

    @builtins.property
    def no_remove_end_date(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Used to add an end date to a Capacity Reservation Fleet that has no end date and time.

        To add an end date to a Capacity Reservation Fleet, specify ``true`` for this paramater and specify the end date and time (in UTC time format) for the *EndDate* parameter.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-noremoveenddate
        '''
        result = self._values.get("no_remove_end_date")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def remove_end_date(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Used to remove an end date from a Capacity Reservation Fleet that is configured to end automatically at a specific date and time.

        To remove the end date from a Capacity Reservation Fleet, specify ``true`` for this paramater and omit the *EndDate* parameter.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-removeenddate
        '''
        result = self._values.get("remove_end_date")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnCapacityReservationFleet.TagSpecificationProperty]]]]:
        '''The tags to assign to the Capacity Reservation Fleet.

        The tags are automatically assigned to the Capacity Reservations in the Fleet.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnCapacityReservationFleet.TagSpecificationProperty]]]], result)

    @builtins.property
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''Indicates the tenancy of the Capacity Reservation Fleet.

        All Capacity Reservations in the Fleet inherit this tenancy. The Capacity Reservation Fleet can have one of the following tenancy settings:

        - ``default`` - The Capacity Reservation Fleet is created on hardware that is shared with other AWS accounts .
        - ``dedicated`` - The Capacity Reservations are created on single-tenant hardware that is dedicated to a single AWS account .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-tenancy
        '''
        result = self._values.get("tenancy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def total_target_capacity(self) -> typing.Optional[jsii.Number]:
        '''The total number of capacity units to be reserved by the Capacity Reservation Fleet.

        This value, together with the instance type weights that you assign to each instance type used by the Fleet determine the number of instances for which the Fleet reserves capacity. Both values are based on units that make sense for your workload. For more information, see `Total target capacity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity>`_ in the *Amazon EC2 User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservationfleet.html#cfn-ec2-capacityreservationfleet-totaltargetcapacity
        '''
        result = self._values.get("total_target_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCapacityReservationFleetProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnCapacityReservationProps",
    jsii_struct_bases=[],
    name_mapping={
        "instance_count": "instanceCount",
        "instance_platform": "instancePlatform",
        "instance_type": "instanceType",
        "availability_zone": "availabilityZone",
        "availability_zone_id": "availabilityZoneId",
        "ebs_optimized": "ebsOptimized",
        "end_date": "endDate",
        "end_date_type": "endDateType",
        "ephemeral_storage": "ephemeralStorage",
        "instance_match_criteria": "instanceMatchCriteria",
        "out_post_arn": "outPostArn",
        "placement_group_arn": "placementGroupArn",
        "tag_specifications": "tagSpecifications",
        "tenancy": "tenancy",
        "unused_reservation_billing_owner_id": "unusedReservationBillingOwnerId",
    },
)
class CfnCapacityReservationProps:
    def __init__(
        self,
        *,
        instance_count: jsii.Number,
        instance_platform: builtins.str,
        instance_type: builtins.str,
        availability_zone: typing.Optional[builtins.str] = None,
        availability_zone_id: typing.Optional[builtins.str] = None,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        end_date: typing.Optional[builtins.str] = None,
        end_date_type: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        instance_match_criteria: typing.Optional[builtins.str] = None,
        out_post_arn: typing.Optional[builtins.str] = None,
        placement_group_arn: typing.Optional[builtins.str] = None,
        tag_specifications: typing.Optional[typing.Sequence[typing.Union[CfnCapacityReservation.TagSpecificationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tenancy: typing.Optional[builtins.str] = None,
        unused_reservation_billing_owner_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnCapacityReservation``.

        :param instance_count: The number of instances for which to reserve capacity. .. epigraph:: You can request future-dated Capacity Reservations for an instance count with a minimum of 100 vCPUs. For example, if you request a future-dated Capacity Reservation for ``m5.xlarge`` instances, you must request at least 25 instances ( *25 * m5.xlarge = 100 vCPUs* ). Valid range: 1 - 1000
        :param instance_platform: The type of operating system for which to reserve capacity.
        :param instance_type: The instance type for which to reserve capacity. .. epigraph:: You can request future-dated Capacity Reservations for instance types in the C, M, R, I, and T instance families only. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* .
        :param availability_zone: The Availability Zone in which to create the Capacity Reservation.
        :param availability_zone_id: The Availability Zone ID of the Capacity Reservation.
        :param ebs_optimized: Indicates whether the Capacity Reservation supports EBS-optimized instances. This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using an EBS- optimized instance.
        :param end_date: The date and time at which the Capacity Reservation expires. When a Capacity Reservation expires, the reserved capacity is released and you can no longer launch instances into it. The Capacity Reservation's state changes to ``expired`` when it reaches its end date and time. You must provide an ``EndDate`` value if ``EndDateType`` is ``limited`` . Omit ``EndDate`` if ``EndDateType`` is ``unlimited`` . If the ``EndDateType`` is ``limited`` , the Capacity Reservation is cancelled within an hour from the specified time. For example, if you specify 5/31/2019, 13:30:55, the Capacity Reservation is guaranteed to end between 13:30:55 and 14:30:55 on 5/31/2019. If you are requesting a future-dated Capacity Reservation, you can't specify an end date and time that is within the commitment duration.
        :param end_date_type: Indicates the way in which the Capacity Reservation ends. A Capacity Reservation can have one of the following end types: - ``unlimited`` - The Capacity Reservation remains active until you explicitly cancel it. Do not provide an ``EndDate`` if the ``EndDateType`` is ``unlimited`` . - ``limited`` - The Capacity Reservation expires automatically at a specified date and time. You must provide an ``EndDate`` value if the ``EndDateType`` value is ``limited`` .
        :param ephemeral_storage: *Deprecated.*.
        :param instance_match_criteria: Indicates the type of instance launches that the Capacity Reservation accepts. The options include:. - ``open`` - The Capacity Reservation automatically matches all instances that have matching attributes (instance type, platform, and Availability Zone). Instances that have matching attributes run in the Capacity Reservation automatically without specifying any additional parameters. - ``targeted`` - The Capacity Reservation only accepts instances that have matching attributes (instance type, platform, and Availability Zone), and explicitly target the Capacity Reservation. This ensures that only permitted instances can use the reserved capacity. .. epigraph:: If you are requesting a future-dated Capacity Reservation, you must specify ``targeted`` . Default: ``open``
        :param out_post_arn: .. epigraph:: Not supported for future-dated Capacity Reservations. The Amazon Resource Name (ARN) of the Outpost on which to create the Capacity Reservation.
        :param placement_group_arn: .. epigraph:: Not supported for future-dated Capacity Reservations. The Amazon Resource Name (ARN) of the cluster placement group in which to create the Capacity Reservation. For more information, see `Capacity Reservations for cluster placement groups <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/cr-cpg.html>`_ in the *Amazon EC2 User Guide* .
        :param tag_specifications: The tags to apply to the Capacity Reservation during launch.
        :param tenancy: Indicates the tenancy of the Capacity Reservation. A Capacity Reservation can have one of the following tenancy settings:. - ``default`` - The Capacity Reservation is created on hardware that is shared with other AWS accounts . - ``dedicated`` - The Capacity Reservation is created on single-tenant hardware that is dedicated to a single AWS account .
        :param unused_reservation_billing_owner_id: The ID of the AWS account to which to assign billing of the unused capacity of the Capacity Reservation. A request will be sent to the specified account. That account must accept the request for the billing to be assigned to their account. For more information, see `Billing assignment for shared Amazon EC2 Capacity Reservations <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/assign-billing.html>`_ . You can assign billing only for shared Capacity Reservations. To share a Capacity Reservation, you must add it to a resource share. For more information, see `AWS::RAM::ResourceShare <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ram-resourceshare.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_capacity_reservation_props = ec2.CfnCapacityReservationProps(
                instance_count=123,
                instance_platform="instancePlatform",
                instance_type="instanceType",
            
                # the properties below are optional
                availability_zone="availabilityZone",
                availability_zone_id="availabilityZoneId",
                ebs_optimized=False,
                end_date="endDate",
                end_date_type="endDateType",
                ephemeral_storage=False,
                instance_match_criteria="instanceMatchCriteria",
                out_post_arn="outPostArn",
                placement_group_arn="placementGroupArn",
                tag_specifications=[ec2.CfnCapacityReservation.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )],
                tenancy="tenancy",
                unused_reservation_billing_owner_id="unusedReservationBillingOwnerId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a65d4e8bb2e678a9a6387fd809c3b5428c783211224ece5155ec92d1eda301d)
            check_type(argname="argument instance_count", value=instance_count, expected_type=type_hints["instance_count"])
            check_type(argname="argument instance_platform", value=instance_platform, expected_type=type_hints["instance_platform"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument availability_zone", value=availability_zone, expected_type=type_hints["availability_zone"])
            check_type(argname="argument availability_zone_id", value=availability_zone_id, expected_type=type_hints["availability_zone_id"])
            check_type(argname="argument ebs_optimized", value=ebs_optimized, expected_type=type_hints["ebs_optimized"])
            check_type(argname="argument end_date", value=end_date, expected_type=type_hints["end_date"])
            check_type(argname="argument end_date_type", value=end_date_type, expected_type=type_hints["end_date_type"])
            check_type(argname="argument ephemeral_storage", value=ephemeral_storage, expected_type=type_hints["ephemeral_storage"])
            check_type(argname="argument instance_match_criteria", value=instance_match_criteria, expected_type=type_hints["instance_match_criteria"])
            check_type(argname="argument out_post_arn", value=out_post_arn, expected_type=type_hints["out_post_arn"])
            check_type(argname="argument placement_group_arn", value=placement_group_arn, expected_type=type_hints["placement_group_arn"])
            check_type(argname="argument tag_specifications", value=tag_specifications, expected_type=type_hints["tag_specifications"])
            check_type(argname="argument tenancy", value=tenancy, expected_type=type_hints["tenancy"])
            check_type(argname="argument unused_reservation_billing_owner_id", value=unused_reservation_billing_owner_id, expected_type=type_hints["unused_reservation_billing_owner_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "instance_count": instance_count,
            "instance_platform": instance_platform,
            "instance_type": instance_type,
        }
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if availability_zone_id is not None:
            self._values["availability_zone_id"] = availability_zone_id
        if ebs_optimized is not None:
            self._values["ebs_optimized"] = ebs_optimized
        if end_date is not None:
            self._values["end_date"] = end_date
        if end_date_type is not None:
            self._values["end_date_type"] = end_date_type
        if ephemeral_storage is not None:
            self._values["ephemeral_storage"] = ephemeral_storage
        if instance_match_criteria is not None:
            self._values["instance_match_criteria"] = instance_match_criteria
        if out_post_arn is not None:
            self._values["out_post_arn"] = out_post_arn
        if placement_group_arn is not None:
            self._values["placement_group_arn"] = placement_group_arn
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if tenancy is not None:
            self._values["tenancy"] = tenancy
        if unused_reservation_billing_owner_id is not None:
            self._values["unused_reservation_billing_owner_id"] = unused_reservation_billing_owner_id

    @builtins.property
    def instance_count(self) -> jsii.Number:
        '''The number of instances for which to reserve capacity.

        .. epigraph::

           You can request future-dated Capacity Reservations for an instance count with a minimum of 100 vCPUs. For example, if you request a future-dated Capacity Reservation for ``m5.xlarge`` instances, you must request at least 25 instances ( *25 * m5.xlarge = 100 vCPUs* ).

        Valid range: 1 - 1000

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancecount
        '''
        result = self._values.get("instance_count")
        assert result is not None, "Required property 'instance_count' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def instance_platform(self) -> builtins.str:
        '''The type of operating system for which to reserve capacity.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instanceplatform
        '''
        result = self._values.get("instance_platform")
        assert result is not None, "Required property 'instance_platform' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_type(self) -> builtins.str:
        '''The instance type for which to reserve capacity.

        .. epigraph::

           You can request future-dated Capacity Reservations for instance types in the C, M, R, I, and T instance families only.

        For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancetype
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''The Availability Zone in which to create the Capacity Reservation.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-availabilityzone
        '''
        result = self._values.get("availability_zone")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def availability_zone_id(self) -> typing.Optional[builtins.str]:
        '''The Availability Zone ID of the Capacity Reservation.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-availabilityzoneid
        '''
        result = self._values.get("availability_zone_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether the Capacity Reservation supports EBS-optimized instances.

        This optimization provides dedicated throughput to Amazon EBS and an optimized configuration stack to provide optimal I/O performance. This optimization isn't available with all instance types. Additional usage charges apply when using an EBS- optimized instance.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-ebsoptimized
        '''
        result = self._values.get("ebs_optimized")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def end_date(self) -> typing.Optional[builtins.str]:
        '''The date and time at which the Capacity Reservation expires.

        When a Capacity Reservation expires, the reserved capacity is released and you can no longer launch instances into it. The Capacity Reservation's state changes to ``expired`` when it reaches its end date and time.

        You must provide an ``EndDate`` value if ``EndDateType`` is ``limited`` . Omit ``EndDate`` if ``EndDateType`` is ``unlimited`` .

        If the ``EndDateType`` is ``limited`` , the Capacity Reservation is cancelled within an hour from the specified time. For example, if you specify 5/31/2019, 13:30:55, the Capacity Reservation is guaranteed to end between 13:30:55 and 14:30:55 on 5/31/2019.

        If you are requesting a future-dated Capacity Reservation, you can't specify an end date and time that is within the commitment duration.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-enddate
        '''
        result = self._values.get("end_date")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def end_date_type(self) -> typing.Optional[builtins.str]:
        '''Indicates the way in which the Capacity Reservation ends.

        A Capacity Reservation can have one of the following end types:

        - ``unlimited`` - The Capacity Reservation remains active until you explicitly cancel it. Do not provide an ``EndDate`` if the ``EndDateType`` is ``unlimited`` .
        - ``limited`` - The Capacity Reservation expires automatically at a specified date and time. You must provide an ``EndDate`` value if the ``EndDateType`` value is ``limited`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-enddatetype
        '''
        result = self._values.get("end_date_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''*Deprecated.*.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-ephemeralstorage
        '''
        result = self._values.get("ephemeral_storage")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def instance_match_criteria(self) -> typing.Optional[builtins.str]:
        '''Indicates the type of instance launches that the Capacity Reservation accepts. The options include:.

        - ``open`` - The Capacity Reservation automatically matches all instances that have matching attributes (instance type, platform, and Availability Zone). Instances that have matching attributes run in the Capacity Reservation automatically without specifying any additional parameters.
        - ``targeted`` - The Capacity Reservation only accepts instances that have matching attributes (instance type, platform, and Availability Zone), and explicitly target the Capacity Reservation. This ensures that only permitted instances can use the reserved capacity.

        .. epigraph::

           If you are requesting a future-dated Capacity Reservation, you must specify ``targeted`` .

        Default: ``open``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancematchcriteria
        '''
        result = self._values.get("instance_match_criteria")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def out_post_arn(self) -> typing.Optional[builtins.str]:
        '''.. epigraph::

   Not supported for future-dated Capacity Reservations.

        The Amazon Resource Name (ARN) of the Outpost on which to create the Capacity Reservation.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-outpostarn
        '''
        result = self._values.get("out_post_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def placement_group_arn(self) -> typing.Optional[builtins.str]:
        '''.. epigraph::

   Not supported for future-dated Capacity Reservations.

        The Amazon Resource Name (ARN) of the cluster placement group in which to create the Capacity Reservation. For more information, see `Capacity Reservations for cluster placement groups <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/cr-cpg.html>`_ in the *Amazon EC2 User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-placementgrouparn
        '''
        result = self._values.get("placement_group_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.List[CfnCapacityReservation.TagSpecificationProperty]]:
        '''The tags to apply to the Capacity Reservation during launch.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.List[CfnCapacityReservation.TagSpecificationProperty]], result)

    @builtins.property
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''Indicates the tenancy of the Capacity Reservation. A Capacity Reservation can have one of the following tenancy settings:.

        - ``default`` - The Capacity Reservation is created on hardware that is shared with other AWS accounts .
        - ``dedicated`` - The Capacity Reservation is created on single-tenant hardware that is dedicated to a single AWS account .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-tenancy
        '''
        result = self._values.get("tenancy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def unused_reservation_billing_owner_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the AWS account to which to assign billing of the unused capacity of the Capacity Reservation.

        A request will be sent to the specified account. That account must accept the request for the billing to be assigned to their account. For more information, see `Billing assignment for shared Amazon EC2 Capacity Reservations <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/assign-billing.html>`_ .

        You can assign billing only for shared Capacity Reservations. To share a Capacity Reservation, you must add it to a resource share. For more information, see `AWS::RAM::ResourceShare <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ram-resourceshare.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-unusedreservationbillingownerid
        '''
        result = self._values.get("unused_reservation_billing_owner_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCapacityReservationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnCarrierGateway(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnCarrierGateway",
):
    '''Creates a carrier gateway.

    For more information about carrier gateways, see `Carrier gateways <https://docs.aws.amazon.com/wavelength/latest/developerguide/how-wavelengths-work.html#wavelength-carrier-gateway>`_ in the *AWS Wavelength Developer Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html
    :cloudformationResource: AWS::EC2::CarrierGateway
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_carrier_gateway = ec2.CfnCarrierGateway(self, "MyCfnCarrierGateway",
            vpc_id="vpcId",
        
            # the properties below are optional
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param vpc_id: The ID of the VPC associated with the carrier gateway.
        :param tags: The tags assigned to the carrier gateway.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a96681cf286606ecadca18bda0ea89ce2bcb5ccb239c0ee8192648a3c637a8b7)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnCarrierGatewayProps(vpc_id=vpc_id, tags=tags)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d13b490de5a71d5c115db0ab3676eac3623d858382e7bda0f72126bd9502de90)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__90ea968688bd5c8f89a44e36535e084f93c6f71eba72008a7c8fae86793c0259)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCarrierGatewayId")
    def attr_carrier_gateway_id(self) -> builtins.str:
        '''The ID of the carrier gateway.

        :cloudformationAttribute: CarrierGatewayId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCarrierGatewayId"))

    @builtins.property
    @jsii.member(jsii_name="attrOwnerId")
    def attr_owner_id(self) -> builtins.str:
        '''The AWS account ID of the owner of the carrier gateway.

        :cloudformationAttribute: OwnerId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrOwnerId"))

    @builtins.property
    @jsii.member(jsii_name="attrState")
    def attr_state(self) -> builtins.str:
        '''The state of the carrier gateway.

        :cloudformationAttribute: State
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrState"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> builtins.str:
        '''The ID of the VPC associated with the carrier gateway.'''
        return typing.cast(builtins.str, jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ebb550f90f423a1cfb12d6962e0d7086e672a69ebc839b263de830334bc029ed)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The tags assigned to the carrier gateway.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0dfc5813551a9e82e5f55d72a506cf1c303b862a6feb2c37161876bb3709f183)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnCarrierGatewayProps",
    jsii_struct_bases=[],
    name_mapping={"vpc_id": "vpcId", "tags": "tags"},
)
class CfnCarrierGatewayProps:
    def __init__(
        self,
        *,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnCarrierGateway``.

        :param vpc_id: The ID of the VPC associated with the carrier gateway.
        :param tags: The tags assigned to the carrier gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_carrier_gateway_props = ec2.CfnCarrierGatewayProps(
                vpc_id="vpcId",
            
                # the properties below are optional
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3bbc37c06b02209086fba1b3fa469d662917ffbfa06787d6aa230add064812b5)
            check_type(argname="argument vpc_id", value=vpc_id, expected_type=type_hints["vpc_id"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "vpc_id": vpc_id,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def vpc_id(self) -> builtins.str:
        '''The ID of the VPC associated with the carrier gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html#cfn-ec2-carriergateway-vpcid
        '''
        result = self._values.get("vpc_id")
        assert result is not None, "Required property 'vpc_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The tags assigned to the carrier gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html#cfn-ec2-carriergateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCarrierGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnClientVpnAuthorizationRule(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnAuthorizationRule",
):
    '''Specifies an ingress authorization rule to add to a Client VPN endpoint.

    Ingress authorization rules act as firewall rules that grant access to networks. You must configure ingress authorization rules to enable clients to access resources in AWS or on-premises networks.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html
    :cloudformationResource: AWS::EC2::ClientVpnAuthorizationRule
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_client_vpn_authorization_rule = ec2.CfnClientVpnAuthorizationRule(self, "MyCfnClientVpnAuthorizationRule",
            client_vpn_endpoint_id="clientVpnEndpointId",
            target_network_cidr="targetNetworkCidr",
        
            # the properties below are optional
            access_group_id="accessGroupId",
            authorize_all_groups=False,
            description="description"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        client_vpn_endpoint_id: builtins.str,
        target_network_cidr: builtins.str,
        access_group_id: typing.Optional[builtins.str] = None,
        authorize_all_groups: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint.
        :param target_network_cidr: The IPv4 address range, in CIDR notation, of the network for which access is being authorized.
        :param access_group_id: The ID of the group to grant access to, for example, the Active Directory group or identity provider (IdP) group. Required if ``AuthorizeAllGroups`` is ``false`` or not specified.
        :param authorize_all_groups: Indicates whether to grant access to all clients. Specify ``true`` to grant all clients who successfully establish a VPN connection access to the network. Must be set to ``true`` if ``AccessGroupId`` is not specified.
        :param description: A brief description of the authorization rule.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd74bb129f8a2ee17c8ff18f5e8a8159b69611f04a07a0ba8d4ee6559e6cdcf1)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnClientVpnAuthorizationRuleProps(
            client_vpn_endpoint_id=client_vpn_endpoint_id,
            target_network_cidr=target_network_cidr,
            access_group_id=access_group_id,
            authorize_all_groups=authorize_all_groups,
            description=description,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d12a489bd88f39a7168ebfe201bff6cf3289493f7b5afa2b2922bc47aa6c2c68)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3c2c11aee50519461eefce28d36cb3da24e82c98ec6ae92b06453da946e1d378)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="clientVpnEndpointId")
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint.'''
        return typing.cast(builtins.str, jsii.get(self, "clientVpnEndpointId"))

    @client_vpn_endpoint_id.setter
    def client_vpn_endpoint_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__50e6d04c1e7523cd5b1e3a8aa7e8300bd0269bc5af9b74b075b07f8a571ee275)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientVpnEndpointId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="targetNetworkCidr")
    def target_network_cidr(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, of the network for which access is being authorized.'''
        return typing.cast(builtins.str, jsii.get(self, "targetNetworkCidr"))

    @target_network_cidr.setter
    def target_network_cidr(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__56d17686cc15e94936303baff8d4eedc73db3f1a2eeb09ce0187cc9c3009b3fa)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetNetworkCidr", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="accessGroupId")
    def access_group_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the group to grant access to, for example, the Active Directory group or identity provider (IdP) group.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "accessGroupId"))

    @access_group_id.setter
    def access_group_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c6e18292fe4463a8537a7530cceaa1f4005c5deba7137c98197919623660f02c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "accessGroupId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="authorizeAllGroups")
    def authorize_all_groups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether to grant access to all clients.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "authorizeAllGroups"))

    @authorize_all_groups.setter
    def authorize_all_groups(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__23a4fad72febaec5f7897b0f2d2dc6091bb1118a92f232e90857a191a9149282)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizeAllGroups", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the authorization rule.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4a0b70f87d13c22c58248897e96db8448f527f13ea896b4d14ed47955f79b9e3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnAuthorizationRuleProps",
    jsii_struct_bases=[],
    name_mapping={
        "client_vpn_endpoint_id": "clientVpnEndpointId",
        "target_network_cidr": "targetNetworkCidr",
        "access_group_id": "accessGroupId",
        "authorize_all_groups": "authorizeAllGroups",
        "description": "description",
    },
)
class CfnClientVpnAuthorizationRuleProps:
    def __init__(
        self,
        *,
        client_vpn_endpoint_id: builtins.str,
        target_network_cidr: builtins.str,
        access_group_id: typing.Optional[builtins.str] = None,
        authorize_all_groups: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnClientVpnAuthorizationRule``.

        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint.
        :param target_network_cidr: The IPv4 address range, in CIDR notation, of the network for which access is being authorized.
        :param access_group_id: The ID of the group to grant access to, for example, the Active Directory group or identity provider (IdP) group. Required if ``AuthorizeAllGroups`` is ``false`` or not specified.
        :param authorize_all_groups: Indicates whether to grant access to all clients. Specify ``true`` to grant all clients who successfully establish a VPN connection access to the network. Must be set to ``true`` if ``AccessGroupId`` is not specified.
        :param description: A brief description of the authorization rule.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_client_vpn_authorization_rule_props = ec2.CfnClientVpnAuthorizationRuleProps(
                client_vpn_endpoint_id="clientVpnEndpointId",
                target_network_cidr="targetNetworkCidr",
            
                # the properties below are optional
                access_group_id="accessGroupId",
                authorize_all_groups=False,
                description="description"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c1cce0b54049d26d96ca01a40dc238215d295ed1ab60e488a987cb2bafe1a791)
            check_type(argname="argument client_vpn_endpoint_id", value=client_vpn_endpoint_id, expected_type=type_hints["client_vpn_endpoint_id"])
            check_type(argname="argument target_network_cidr", value=target_network_cidr, expected_type=type_hints["target_network_cidr"])
            check_type(argname="argument access_group_id", value=access_group_id, expected_type=type_hints["access_group_id"])
            check_type(argname="argument authorize_all_groups", value=authorize_all_groups, expected_type=type_hints["authorize_all_groups"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "client_vpn_endpoint_id": client_vpn_endpoint_id,
            "target_network_cidr": target_network_cidr,
        }
        if access_group_id is not None:
            self._values["access_group_id"] = access_group_id
        if authorize_all_groups is not None:
            self._values["authorize_all_groups"] = authorize_all_groups
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-clientvpnendpointid
        '''
        result = self._values.get("client_vpn_endpoint_id")
        assert result is not None, "Required property 'client_vpn_endpoint_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_network_cidr(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, of the network for which access is being authorized.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-targetnetworkcidr
        '''
        result = self._values.get("target_network_cidr")
        assert result is not None, "Required property 'target_network_cidr' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_group_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the group to grant access to, for example, the Active Directory group or identity provider (IdP) group.

        Required if ``AuthorizeAllGroups`` is ``false`` or not specified.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-accessgroupid
        '''
        result = self._values.get("access_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorize_all_groups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether to grant access to all clients.

        Specify ``true`` to grant all clients who successfully establish a VPN connection access to the network. Must be set to ``true`` if ``AccessGroupId`` is not specified.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-authorizeallgroups
        '''
        result = self._values.get("authorize_all_groups")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the authorization rule.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnAuthorizationRuleProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnClientVpnEndpoint(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnEndpoint",
):
    '''Specifies a Client VPN endpoint.

    A Client VPN endpoint is the resource you create and configure to enable and manage client VPN sessions. It is the destination endpoint at which all client VPN sessions are terminated.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html
    :cloudformationResource: AWS::EC2::ClientVpnEndpoint
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_client_vpn_endpoint = ec2.CfnClientVpnEndpoint(self, "MyCfnClientVpnEndpoint",
            authentication_options=[ec2.CfnClientVpnEndpoint.ClientAuthenticationRequestProperty(
                type="type",
        
                # the properties below are optional
                active_directory=ec2.CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty(
                    directory_id="directoryId"
                ),
                federated_authentication=ec2.CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty(
                    saml_provider_arn="samlProviderArn",
        
                    # the properties below are optional
                    self_service_saml_provider_arn="selfServiceSamlProviderArn"
                ),
                mutual_authentication=ec2.CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty(
                    client_root_certificate_chain_arn="clientRootCertificateChainArn"
                )
            )],
            client_cidr_block="clientCidrBlock",
            connection_log_options=ec2.CfnClientVpnEndpoint.ConnectionLogOptionsProperty(
                enabled=False,
        
                # the properties below are optional
                cloudwatch_log_group="cloudwatchLogGroup",
                cloudwatch_log_stream="cloudwatchLogStream"
            ),
            server_certificate_arn="serverCertificateArn",
        
            # the properties below are optional
            client_connect_options=ec2.CfnClientVpnEndpoint.ClientConnectOptionsProperty(
                enabled=False,
        
                # the properties below are optional
                lambda_function_arn="lambdaFunctionArn"
            ),
            client_login_banner_options=ec2.CfnClientVpnEndpoint.ClientLoginBannerOptionsProperty(
                enabled=False,
        
                # the properties below are optional
                banner_text="bannerText"
            ),
            description="description",
            disconnect_on_session_timeout=False,
            dns_servers=["dnsServers"],
            security_group_ids=["securityGroupIds"],
            self_service_portal="selfServicePortal",
            session_timeout_hours=123,
            split_tunnel=False,
            tag_specifications=[ec2.CfnClientVpnEndpoint.TagSpecificationProperty(
                resource_type="resourceType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )],
            transport_protocol="transportProtocol",
            vpc_id="vpcId",
            vpn_port=123
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        authentication_options: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnClientVpnEndpoint.ClientAuthenticationRequestProperty", typing.Dict[builtins.str, typing.Any]]]]],
        client_cidr_block: builtins.str,
        connection_log_options: typing.Union[_IResolvable_da3f097b, typing.Union["CfnClientVpnEndpoint.ConnectionLogOptionsProperty", typing.Dict[builtins.str, typing.Any]]],
        server_certificate_arn: builtins.str,
        client_connect_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnClientVpnEndpoint.ClientConnectOptionsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        client_login_banner_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnClientVpnEndpoint.ClientLoginBannerOptionsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        disconnect_on_session_timeout: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        self_service_portal: typing.Optional[builtins.str] = None,
        session_timeout_hours: typing.Optional[jsii.Number] = None,
        split_tunnel: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        tag_specifications: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnClientVpnEndpoint.TagSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        transport_protocol: typing.Optional[builtins.str] = None,
        vpc_id: typing.Optional[builtins.str] = None,
        vpn_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param authentication_options: Information about the authentication method to be used to authenticate clients.
        :param client_cidr_block: The IPv4 address range, in CIDR notation, from which to assign client IP addresses. The address range cannot overlap with the local CIDR of the VPC in which the associated subnet is located, or the routes that you add manually. The address range cannot be changed after the Client VPN endpoint has been created. Client CIDR range must have a size of at least /22 and must not be greater than /12.
        :param connection_log_options: Information about the client connection logging options. If you enable client connection logging, data about client connections is sent to a Cloudwatch Logs log stream. The following information is logged: - Client connection requests - Client connection results (successful and unsuccessful) - Reasons for unsuccessful client connection requests - Client connection termination time
        :param server_certificate_arn: The ARN of the server certificate. For more information, see the `AWS Certificate Manager User Guide <https://docs.aws.amazon.com/acm/latest/userguide/>`_ .
        :param client_connect_options: The options for managing connection authorization for new client connections.
        :param client_login_banner_options: Options for enabling a customizable text banner that will be displayed on AWS provided clients when a VPN session is established.
        :param description: A brief description of the Client VPN endpoint.
        :param disconnect_on_session_timeout: Indicates whether the client VPN session is disconnected after the maximum ``sessionTimeoutHours`` is reached. If ``true`` , users are prompted to reconnect client VPN. If ``false`` , client VPN attempts to reconnect automatically. The default value is ``false`` .
        :param dns_servers: Information about the DNS servers to be used for DNS resolution. A Client VPN endpoint can have up to two DNS servers. If no DNS server is specified, the DNS address configured on the device is used for the DNS server.
        :param security_group_ids: The IDs of one or more security groups to apply to the target network. You must also specify the ID of the VPC that contains the security groups.
        :param self_service_portal: Specify whether to enable the self-service portal for the Client VPN endpoint. Default Value: ``enabled``
        :param session_timeout_hours: The maximum VPN session duration time in hours. Valid values: ``8 | 10 | 12 | 24`` Default value: ``24``
        :param split_tunnel: Indicates whether split-tunnel is enabled on the AWS Client VPN endpoint. By default, split-tunnel on a VPN endpoint is disabled. For information about split-tunnel VPN endpoints, see `Split-tunnel AWS Client VPN endpoint <https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html>`_ in the *AWS Client VPN Administrator Guide* .
        :param tag_specifications: The tags to apply to the Client VPN endpoint during creation.
        :param transport_protocol: The transport protocol to be used by the VPN session. Default value: ``udp``
        :param vpc_id: The ID of the VPC to associate with the Client VPN endpoint. If no security group IDs are specified in the request, the default security group for the VPC is applied.
        :param vpn_port: The port number to assign to the Client VPN endpoint for TCP and UDP traffic. Valid Values: ``443`` | ``1194`` Default Value: ``443``
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1880bffa9253aaedfa6af6175da6262a96ea34dadf82dc678a1eef91cb7b92f0)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnClientVpnEndpointProps(
            authentication_options=authentication_options,
            client_cidr_block=client_cidr_block,
            connection_log_options=connection_log_options,
            server_certificate_arn=server_certificate_arn,
            client_connect_options=client_connect_options,
            client_login_banner_options=client_login_banner_options,
            description=description,
            disconnect_on_session_timeout=disconnect_on_session_timeout,
            dns_servers=dns_servers,
            security_group_ids=security_group_ids,
            self_service_portal=self_service_portal,
            session_timeout_hours=session_timeout_hours,
            split_tunnel=split_tunnel,
            tag_specifications=tag_specifications,
            transport_protocol=transport_protocol,
            vpc_id=vpc_id,
            vpn_port=vpn_port,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6c918e848a5790bf428aae5434bdf054890599ff5e33a0c7aec91388a7d88962)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4840c2bcc0e38eb5b8ab1bf4318bcc826fafc75f0b1a0d4b4af2671fd548551b)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="authenticationOptions")
    def authentication_options(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ClientAuthenticationRequestProperty"]]]:
        '''Information about the authentication method to be used to authenticate clients.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ClientAuthenticationRequestProperty"]]], jsii.get(self, "authenticationOptions"))

    @authentication_options.setter
    def authentication_options(
        self,
        value: typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ClientAuthenticationRequestProperty"]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8fd721536b7af569ca4dd0cf5d6d3e5a4af03f14f0a211dca122f2c1ac5297c8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authenticationOptions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="clientCidrBlock")
    def client_cidr_block(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, from which to assign client IP addresses.'''
        return typing.cast(builtins.str, jsii.get(self, "clientCidrBlock"))

    @client_cidr_block.setter
    def client_cidr_block(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c82ca35d7edd1869a3d83867786684f955e85c4ac53303d02ebbdd382476bc11)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientCidrBlock", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="connectionLogOptions")
    def connection_log_options(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ConnectionLogOptionsProperty"]:
        '''Information about the client connection logging options.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ConnectionLogOptionsProperty"], jsii.get(self, "connectionLogOptions"))

    @connection_log_options.setter
    def connection_log_options(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ConnectionLogOptionsProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__18d2e85936305e18045946d27a2644a466c3e1398c55ace117c923d19e4e239f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "connectionLogOptions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="serverCertificateArn")
    def server_certificate_arn(self) -> builtins.str:
        '''The ARN of the server certificate.'''
        return typing.cast(builtins.str, jsii.get(self, "serverCertificateArn"))

    @server_certificate_arn.setter
    def server_certificate_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2bd127fecac421e3bc066a629699ad50c84a77153f1ec4f5626641e6e71005ba)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serverCertificateArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="clientConnectOptions")
    def client_connect_options(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ClientConnectOptionsProperty"]]:
        '''The options for managing connection authorization for new client connections.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ClientConnectOptionsProperty"]], jsii.get(self, "clientConnectOptions"))

    @client_connect_options.setter
    def client_connect_options(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ClientConnectOptionsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0ff43378f198f2e58416acaccf8cc15978b8057d3e692a315efe9a325e263a4f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientConnectOptions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="clientLoginBannerOptions")
    def client_login_banner_options(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ClientLoginBannerOptionsProperty"]]:
        '''Options for enabling a customizable text banner that will be displayed on AWS provided clients when a VPN session is established.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ClientLoginBannerOptionsProperty"]], jsii.get(self, "clientLoginBannerOptions"))

    @client_login_banner_options.setter
    def client_login_banner_options(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.ClientLoginBannerOptionsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__497c1bea9e9caa987aeafc2d4334f07cd2d9664a3f3cd38bebd5611705a992a0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientLoginBannerOptions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the Client VPN endpoint.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bb1d5fb0102c1b6de24ad13acfc13837ec5422291e23048395fe03350aea2188)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="disconnectOnSessionTimeout")
    def disconnect_on_session_timeout(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether the client VPN session is disconnected after the maximum ``sessionTimeoutHours`` is reached.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "disconnectOnSessionTimeout"))

    @disconnect_on_session_timeout.setter
    def disconnect_on_session_timeout(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7a04e648e3ab2786626daca4bea9d7e5d9f8fd502183ea90643aafd9a31a5f2c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "disconnectOnSessionTimeout", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="dnsServers")
    def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Information about the DNS servers to be used for DNS resolution.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "dnsServers"))

    @dns_servers.setter
    def dns_servers(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1ebcac79eb30395bbcc38271559390a74d75a7ae45dfe368c2cd3b996b81b462)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dnsServers", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="securityGroupIds")
    def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IDs of one or more security groups to apply to the target network.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "securityGroupIds"))

    @security_group_ids.setter
    def security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__579de23f23e1dd217e8431defd11b303d19733349907433779f8a626aae9cde9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "securityGroupIds", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="selfServicePortal")
    def self_service_portal(self) -> typing.Optional[builtins.str]:
        '''Specify whether to enable the self-service portal for the Client VPN endpoint.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "selfServicePortal"))

    @self_service_portal.setter
    def self_service_portal(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8906bf30210d5bc2dc8408899f5ca0117189261f028b77c92f7388e86156315c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "selfServicePortal", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="sessionTimeoutHours")
    def session_timeout_hours(self) -> typing.Optional[jsii.Number]:
        '''The maximum VPN session duration time in hours.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "sessionTimeoutHours"))

    @session_timeout_hours.setter
    def session_timeout_hours(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__794dd5b3de3a1c6a9d4c9dab081dbf25f0c2c05274d74b3e4fc01f8b17001d25)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sessionTimeoutHours", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="splitTunnel")
    def split_tunnel(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether split-tunnel is enabled on the AWS Client VPN endpoint.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "splitTunnel"))

    @split_tunnel.setter
    def split_tunnel(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0d96887217ae5515add2cb3a59126e1d0cd39a226e9816afa37568bc70cb2f61)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "splitTunnel", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.TagSpecificationProperty"]]]]:
        '''The tags to apply to the Client VPN endpoint during creation.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.TagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.TagSpecificationProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cc170770bc9adc8648a63b630961b275859df1f9019415667b679b2cf764d8eb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagSpecifications", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="transportProtocol")
    def transport_protocol(self) -> typing.Optional[builtins.str]:
        '''The transport protocol to be used by the VPN session.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "transportProtocol"))

    @transport_protocol.setter
    def transport_protocol(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7606d3932bdd042643cd05946ce9c33ae761b0b43aa601cec4d7639cdf467b12)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "transportProtocol", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the VPC to associate with the Client VPN endpoint.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__10f3125d05aba06539b785430760a1829e6fb47f55d9a88648888bf1727da9f5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="vpnPort")
    def vpn_port(self) -> typing.Optional[jsii.Number]:
        '''The port number to assign to the Client VPN endpoint for TCP and UDP traffic.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "vpnPort"))

    @vpn_port.setter
    def vpn_port(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__662eb5a92b0bd04e8bb4b264778b748b7e43ac948946096053c568d8b911e105)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpnPort", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "client_root_certificate_chain_arn": "clientRootCertificateChainArn",
        },
    )
    class CertificateAuthenticationRequestProperty:
        def __init__(self, *, client_root_certificate_chain_arn: builtins.str) -> None:
            '''Information about the client certificate to be used for authentication.

            :param client_root_certificate_chain_arn: The ARN of the client certificate. The certificate must be signed by a certificate authority (CA) and it must be provisioned in AWS Certificate Manager (ACM).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-certificateauthenticationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                certificate_authentication_request_property = ec2.CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty(
                    client_root_certificate_chain_arn="clientRootCertificateChainArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cc37b569930b5db9e1e89a2936bbf73c15c41b1fc566455e26a96a67b711362a)
                check_type(argname="argument client_root_certificate_chain_arn", value=client_root_certificate_chain_arn, expected_type=type_hints["client_root_certificate_chain_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "client_root_certificate_chain_arn": client_root_certificate_chain_arn,
            }

        @builtins.property
        def client_root_certificate_chain_arn(self) -> builtins.str:
            '''The ARN of the client certificate.

            The certificate must be signed by a certificate authority (CA) and it must be provisioned in AWS Certificate Manager (ACM).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-certificateauthenticationrequest.html#cfn-ec2-clientvpnendpoint-certificateauthenticationrequest-clientrootcertificatechainarn
            '''
            result = self._values.get("client_root_certificate_chain_arn")
            assert result is not None, "Required property 'client_root_certificate_chain_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CertificateAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnEndpoint.ClientAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "type": "type",
            "active_directory": "activeDirectory",
            "federated_authentication": "federatedAuthentication",
            "mutual_authentication": "mutualAuthentication",
        },
    )
    class ClientAuthenticationRequestProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            active_directory: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            federated_authentication: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            mutual_authentication: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Describes the authentication method to be used by a Client VPN endpoint.

            For more information, see `Authentication <https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/authentication-authrization.html#client-authentication>`_ in the *AWS Client VPN Administrator Guide* .

            :param type: The type of client authentication to be used.
            :param active_directory: Information about the Active Directory to be used, if applicable. You must provide this information if *Type* is ``directory-service-authentication`` .
            :param federated_authentication: Information about the IAM SAML identity provider, if applicable.
            :param mutual_authentication: Information about the authentication certificates to be used, if applicable. You must provide this information if *Type* is ``certificate-authentication`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                client_authentication_request_property = ec2.CfnClientVpnEndpoint.ClientAuthenticationRequestProperty(
                    type="type",
                
                    # the properties below are optional
                    active_directory=ec2.CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty(
                        directory_id="directoryId"
                    ),
                    federated_authentication=ec2.CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty(
                        saml_provider_arn="samlProviderArn",
                
                        # the properties below are optional
                        self_service_saml_provider_arn="selfServiceSamlProviderArn"
                    ),
                    mutual_authentication=ec2.CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty(
                        client_root_certificate_chain_arn="clientRootCertificateChainArn"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__93d4b5bf25c558dd4e4b4bbec8493db5e9a168f9b8327c99f3b48709b29f0305)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument active_directory", value=active_directory, expected_type=type_hints["active_directory"])
                check_type(argname="argument federated_authentication", value=federated_authentication, expected_type=type_hints["federated_authentication"])
                check_type(argname="argument mutual_authentication", value=mutual_authentication, expected_type=type_hints["mutual_authentication"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if active_directory is not None:
                self._values["active_directory"] = active_directory
            if federated_authentication is not None:
                self._values["federated_authentication"] = federated_authentication
            if mutual_authentication is not None:
                self._values["mutual_authentication"] = mutual_authentication

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of client authentication to be used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def active_directory(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty"]]:
            '''Information about the Active Directory to be used, if applicable.

            You must provide this information if *Type* is ``directory-service-authentication`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-activedirectory
            '''
            result = self._values.get("active_directory")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty"]], result)

        @builtins.property
        def federated_authentication(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty"]]:
            '''Information about the IAM SAML identity provider, if applicable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-federatedauthentication
            '''
            result = self._values.get("federated_authentication")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty"]], result)

        @builtins.property
        def mutual_authentication(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty"]]:
            '''Information about the authentication certificates to be used, if applicable.

            You must provide this information if *Type* is ``certificate-authentication`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-mutualauthentication
            '''
            result = self._values.get("mutual_authentication")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnEndpoint.ClientConnectOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled": "enabled",
            "lambda_function_arn": "lambdaFunctionArn",
        },
    )
    class ClientConnectOptionsProperty:
        def __init__(
            self,
            *,
            enabled: typing.Union[builtins.bool, _IResolvable_da3f097b],
            lambda_function_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Indicates whether client connect options are enabled.

            The default is ``false`` (not enabled).

            :param enabled: Indicates whether client connect options are enabled. The default is ``false`` (not enabled).
            :param lambda_function_arn: The Amazon Resource Name (ARN) of the AWS Lambda function used for connection authorization.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientconnectoptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                client_connect_options_property = ec2.CfnClientVpnEndpoint.ClientConnectOptionsProperty(
                    enabled=False,
                
                    # the properties below are optional
                    lambda_function_arn="lambdaFunctionArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c807bc6c81e334e361f85e32e4442141132b4ee3c4bacdaed8cf79b617caaa57)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument lambda_function_arn", value=lambda_function_arn, expected_type=type_hints["lambda_function_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "enabled": enabled,
            }
            if lambda_function_arn is not None:
                self._values["lambda_function_arn"] = lambda_function_arn

        @builtins.property
        def enabled(self) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Indicates whether client connect options are enabled.

            The default is ``false`` (not enabled).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientconnectoptions.html#cfn-ec2-clientvpnendpoint-clientconnectoptions-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        @builtins.property
        def lambda_function_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the AWS Lambda function used for connection authorization.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientconnectoptions.html#cfn-ec2-clientvpnendpoint-clientconnectoptions-lambdafunctionarn
            '''
            result = self._values.get("lambda_function_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientConnectOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnEndpoint.ClientLoginBannerOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"enabled": "enabled", "banner_text": "bannerText"},
    )
    class ClientLoginBannerOptionsProperty:
        def __init__(
            self,
            *,
            enabled: typing.Union[builtins.bool, _IResolvable_da3f097b],
            banner_text: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Options for enabling a customizable text banner that will be displayed on AWS provided clients when a VPN session is established.

            :param enabled: Enable or disable a customizable text banner that will be displayed on AWS provided clients when a VPN session is established. Valid values: ``true | false`` Default value: ``false``
            :param banner_text: Customizable text that will be displayed in a banner on AWS provided clients when a VPN session is established. UTF-8 encoded characters only. Maximum of 1400 characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientloginbanneroptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                client_login_banner_options_property = ec2.CfnClientVpnEndpoint.ClientLoginBannerOptionsProperty(
                    enabled=False,
                
                    # the properties below are optional
                    banner_text="bannerText"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__907ff3a3bd097ac37a720bbc18cf4e29748beb226ddf13641e48350162da19d3)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument banner_text", value=banner_text, expected_type=type_hints["banner_text"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "enabled": enabled,
            }
            if banner_text is not None:
                self._values["banner_text"] = banner_text

        @builtins.property
        def enabled(self) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Enable or disable a customizable text banner that will be displayed on AWS provided clients when a VPN session is established.

            Valid values: ``true | false``

            Default value: ``false``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientloginbanneroptions.html#cfn-ec2-clientvpnendpoint-clientloginbanneroptions-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        @builtins.property
        def banner_text(self) -> typing.Optional[builtins.str]:
            '''Customizable text that will be displayed in a banner on AWS provided clients when a VPN session is established.

            UTF-8 encoded characters only. Maximum of 1400 characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientloginbanneroptions.html#cfn-ec2-clientvpnendpoint-clientloginbanneroptions-bannertext
            '''
            result = self._values.get("banner_text")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientLoginBannerOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnEndpoint.ConnectionLogOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled": "enabled",
            "cloudwatch_log_group": "cloudwatchLogGroup",
            "cloudwatch_log_stream": "cloudwatchLogStream",
        },
    )
    class ConnectionLogOptionsProperty:
        def __init__(
            self,
            *,
            enabled: typing.Union[builtins.bool, _IResolvable_da3f097b],
            cloudwatch_log_group: typing.Optional[builtins.str] = None,
            cloudwatch_log_stream: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Describes the client connection logging options for the Client VPN endpoint.

            :param enabled: Indicates whether connection logging is enabled.
            :param cloudwatch_log_group: The name of the CloudWatch Logs log group. Required if connection logging is enabled.
            :param cloudwatch_log_stream: The name of the CloudWatch Logs log stream to which the connection data is published.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                connection_log_options_property = ec2.CfnClientVpnEndpoint.ConnectionLogOptionsProperty(
                    enabled=False,
                
                    # the properties below are optional
                    cloudwatch_log_group="cloudwatchLogGroup",
                    cloudwatch_log_stream="cloudwatchLogStream"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2c142e5322bb8bda4ace9616736c0e12d7c7b9b219932b4168a975e68d0d9395)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument cloudwatch_log_group", value=cloudwatch_log_group, expected_type=type_hints["cloudwatch_log_group"])
                check_type(argname="argument cloudwatch_log_stream", value=cloudwatch_log_stream, expected_type=type_hints["cloudwatch_log_stream"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "enabled": enabled,
            }
            if cloudwatch_log_group is not None:
                self._values["cloudwatch_log_group"] = cloudwatch_log_group
            if cloudwatch_log_stream is not None:
                self._values["cloudwatch_log_stream"] = cloudwatch_log_stream

        @builtins.property
        def enabled(self) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Indicates whether connection logging is enabled.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html#cfn-ec2-clientvpnendpoint-connectionlogoptions-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        @builtins.property
        def cloudwatch_log_group(self) -> typing.Optional[builtins.str]:
            '''The name of the CloudWatch Logs log group.

            Required if connection logging is enabled.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html#cfn-ec2-clientvpnendpoint-connectionlogoptions-cloudwatchloggroup
            '''
            result = self._values.get("cloudwatch_log_group")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cloudwatch_log_stream(self) -> typing.Optional[builtins.str]:
            '''The name of the CloudWatch Logs log stream to which the connection data is published.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html#cfn-ec2-clientvpnendpoint-connectionlogoptions-cloudwatchlogstream
            '''
            result = self._values.get("cloudwatch_log_stream")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConnectionLogOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"directory_id": "directoryId"},
    )
    class DirectoryServiceAuthenticationRequestProperty:
        def __init__(self, *, directory_id: builtins.str) -> None:
            '''Describes the Active Directory to be used for client authentication.

            :param directory_id: The ID of the Active Directory to be used for authentication.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-directoryserviceauthenticationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                directory_service_authentication_request_property = ec2.CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty(
                    directory_id="directoryId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c6fc687dd9d11c39776770388c8bde0ccd238cd37eb50b3855604baa114f8fcc)
                check_type(argname="argument directory_id", value=directory_id, expected_type=type_hints["directory_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "directory_id": directory_id,
            }

        @builtins.property
        def directory_id(self) -> builtins.str:
            '''The ID of the Active Directory to be used for authentication.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-directoryserviceauthenticationrequest.html#cfn-ec2-clientvpnendpoint-directoryserviceauthenticationrequest-directoryid
            '''
            result = self._values.get("directory_id")
            assert result is not None, "Required property 'directory_id' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DirectoryServiceAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "saml_provider_arn": "samlProviderArn",
            "self_service_saml_provider_arn": "selfServiceSamlProviderArn",
        },
    )
    class FederatedAuthenticationRequestProperty:
        def __init__(
            self,
            *,
            saml_provider_arn: builtins.str,
            self_service_saml_provider_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The IAM SAML identity provider used for federated authentication.

            :param saml_provider_arn: The Amazon Resource Name (ARN) of the IAM SAML identity provider.
            :param self_service_saml_provider_arn: The Amazon Resource Name (ARN) of the IAM SAML identity provider for the self-service portal.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-federatedauthenticationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                federated_authentication_request_property = ec2.CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty(
                    saml_provider_arn="samlProviderArn",
                
                    # the properties below are optional
                    self_service_saml_provider_arn="selfServiceSamlProviderArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6d301df1b88110c542331637189053fa067891656efcdcce073dea638050e816)
                check_type(argname="argument saml_provider_arn", value=saml_provider_arn, expected_type=type_hints["saml_provider_arn"])
                check_type(argname="argument self_service_saml_provider_arn", value=self_service_saml_provider_arn, expected_type=type_hints["self_service_saml_provider_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "saml_provider_arn": saml_provider_arn,
            }
            if self_service_saml_provider_arn is not None:
                self._values["self_service_saml_provider_arn"] = self_service_saml_provider_arn

        @builtins.property
        def saml_provider_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the IAM SAML identity provider.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-federatedauthenticationrequest.html#cfn-ec2-clientvpnendpoint-federatedauthenticationrequest-samlproviderarn
            '''
            result = self._values.get("saml_provider_arn")
            assert result is not None, "Required property 'saml_provider_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def self_service_saml_provider_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the IAM SAML identity provider for the self-service portal.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-federatedauthenticationrequest.html#cfn-ec2-clientvpnendpoint-federatedauthenticationrequest-selfservicesamlproviderarn
            '''
            result = self._values.get("self_service_saml_provider_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FederatedAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnEndpoint.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: builtins.str,
            tags: typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''Specifies the tags to apply to the Client VPN endpoint.

            :param resource_type: The type of resource to tag. To tag a Client VPN endpoint, ``ResourceType`` must be ``client-vpn-endpoint`` .
            :param tags: The tags to apply to the resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-tagspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                tag_specification_property = ec2.CfnClientVpnEndpoint.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7fadda101a8454212efac8a224e66b4af09c2e630eb4772396408c19bf7444d1)
                check_type(argname="argument resource_type", value=resource_type, expected_type=type_hints["resource_type"])
                check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "resource_type": resource_type,
                "tags": tags,
            }

        @builtins.property
        def resource_type(self) -> builtins.str:
            '''The type of resource to tag.

            To tag a Client VPN endpoint, ``ResourceType`` must be ``client-vpn-endpoint`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-tagspecification.html#cfn-ec2-clientvpnendpoint-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            assert result is not None, "Required property 'resource_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def tags(self) -> typing.List[_CfnTag_f6864754]:
            '''The tags to apply to the resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-tagspecification.html#cfn-ec2-clientvpnendpoint-tagspecification-tags
            '''
            result = self._values.get("tags")
            assert result is not None, "Required property 'tags' is missing"
            return typing.cast(typing.List[_CfnTag_f6864754], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnEndpointProps",
    jsii_struct_bases=[],
    name_mapping={
        "authentication_options": "authenticationOptions",
        "client_cidr_block": "clientCidrBlock",
        "connection_log_options": "connectionLogOptions",
        "server_certificate_arn": "serverCertificateArn",
        "client_connect_options": "clientConnectOptions",
        "client_login_banner_options": "clientLoginBannerOptions",
        "description": "description",
        "disconnect_on_session_timeout": "disconnectOnSessionTimeout",
        "dns_servers": "dnsServers",
        "security_group_ids": "securityGroupIds",
        "self_service_portal": "selfServicePortal",
        "session_timeout_hours": "sessionTimeoutHours",
        "split_tunnel": "splitTunnel",
        "tag_specifications": "tagSpecifications",
        "transport_protocol": "transportProtocol",
        "vpc_id": "vpcId",
        "vpn_port": "vpnPort",
    },
)
class CfnClientVpnEndpointProps:
    def __init__(
        self,
        *,
        authentication_options: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnClientVpnEndpoint.ClientAuthenticationRequestProperty, typing.Dict[builtins.str, typing.Any]]]]],
        client_cidr_block: builtins.str,
        connection_log_options: typing.Union[_IResolvable_da3f097b, typing.Union[CfnClientVpnEndpoint.ConnectionLogOptionsProperty, typing.Dict[builtins.str, typing.Any]]],
        server_certificate_arn: builtins.str,
        client_connect_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnClientVpnEndpoint.ClientConnectOptionsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        client_login_banner_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnClientVpnEndpoint.ClientLoginBannerOptionsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        disconnect_on_session_timeout: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        self_service_portal: typing.Optional[builtins.str] = None,
        session_timeout_hours: typing.Optional[jsii.Number] = None,
        split_tunnel: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        tag_specifications: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnClientVpnEndpoint.TagSpecificationProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        transport_protocol: typing.Optional[builtins.str] = None,
        vpc_id: typing.Optional[builtins.str] = None,
        vpn_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnClientVpnEndpoint``.

        :param authentication_options: Information about the authentication method to be used to authenticate clients.
        :param client_cidr_block: The IPv4 address range, in CIDR notation, from which to assign client IP addresses. The address range cannot overlap with the local CIDR of the VPC in which the associated subnet is located, or the routes that you add manually. The address range cannot be changed after the Client VPN endpoint has been created. Client CIDR range must have a size of at least /22 and must not be greater than /12.
        :param connection_log_options: Information about the client connection logging options. If you enable client connection logging, data about client connections is sent to a Cloudwatch Logs log stream. The following information is logged: - Client connection requests - Client connection results (successful and unsuccessful) - Reasons for unsuccessful client connection requests - Client connection termination time
        :param server_certificate_arn: The ARN of the server certificate. For more information, see the `AWS Certificate Manager User Guide <https://docs.aws.amazon.com/acm/latest/userguide/>`_ .
        :param client_connect_options: The options for managing connection authorization for new client connections.
        :param client_login_banner_options: Options for enabling a customizable text banner that will be displayed on AWS provided clients when a VPN session is established.
        :param description: A brief description of the Client VPN endpoint.
        :param disconnect_on_session_timeout: Indicates whether the client VPN session is disconnected after the maximum ``sessionTimeoutHours`` is reached. If ``true`` , users are prompted to reconnect client VPN. If ``false`` , client VPN attempts to reconnect automatically. The default value is ``false`` .
        :param dns_servers: Information about the DNS servers to be used for DNS resolution. A Client VPN endpoint can have up to two DNS servers. If no DNS server is specified, the DNS address configured on the device is used for the DNS server.
        :param security_group_ids: The IDs of one or more security groups to apply to the target network. You must also specify the ID of the VPC that contains the security groups.
        :param self_service_portal: Specify whether to enable the self-service portal for the Client VPN endpoint. Default Value: ``enabled``
        :param session_timeout_hours: The maximum VPN session duration time in hours. Valid values: ``8 | 10 | 12 | 24`` Default value: ``24``
        :param split_tunnel: Indicates whether split-tunnel is enabled on the AWS Client VPN endpoint. By default, split-tunnel on a VPN endpoint is disabled. For information about split-tunnel VPN endpoints, see `Split-tunnel AWS Client VPN endpoint <https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html>`_ in the *AWS Client VPN Administrator Guide* .
        :param tag_specifications: The tags to apply to the Client VPN endpoint during creation.
        :param transport_protocol: The transport protocol to be used by the VPN session. Default value: ``udp``
        :param vpc_id: The ID of the VPC to associate with the Client VPN endpoint. If no security group IDs are specified in the request, the default security group for the VPC is applied.
        :param vpn_port: The port number to assign to the Client VPN endpoint for TCP and UDP traffic. Valid Values: ``443`` | ``1194`` Default Value: ``443``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_client_vpn_endpoint_props = ec2.CfnClientVpnEndpointProps(
                authentication_options=[ec2.CfnClientVpnEndpoint.ClientAuthenticationRequestProperty(
                    type="type",
            
                    # the properties below are optional
                    active_directory=ec2.CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty(
                        directory_id="directoryId"
                    ),
                    federated_authentication=ec2.CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty(
                        saml_provider_arn="samlProviderArn",
            
                        # the properties below are optional
                        self_service_saml_provider_arn="selfServiceSamlProviderArn"
                    ),
                    mutual_authentication=ec2.CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty(
                        client_root_certificate_chain_arn="clientRootCertificateChainArn"
                    )
                )],
                client_cidr_block="clientCidrBlock",
                connection_log_options=ec2.CfnClientVpnEndpoint.ConnectionLogOptionsProperty(
                    enabled=False,
            
                    # the properties below are optional
                    cloudwatch_log_group="cloudwatchLogGroup",
                    cloudwatch_log_stream="cloudwatchLogStream"
                ),
                server_certificate_arn="serverCertificateArn",
            
                # the properties below are optional
                client_connect_options=ec2.CfnClientVpnEndpoint.ClientConnectOptionsProperty(
                    enabled=False,
            
                    # the properties below are optional
                    lambda_function_arn="lambdaFunctionArn"
                ),
                client_login_banner_options=ec2.CfnClientVpnEndpoint.ClientLoginBannerOptionsProperty(
                    enabled=False,
            
                    # the properties below are optional
                    banner_text="bannerText"
                ),
                description="description",
                disconnect_on_session_timeout=False,
                dns_servers=["dnsServers"],
                security_group_ids=["securityGroupIds"],
                self_service_portal="selfServicePortal",
                session_timeout_hours=123,
                split_tunnel=False,
                tag_specifications=[ec2.CfnClientVpnEndpoint.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )],
                transport_protocol="transportProtocol",
                vpc_id="vpcId",
                vpn_port=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__05994467e800c33b4a01e884b4b20bef2569d710f5dc323617cd814e50e5bdb5)
            check_type(argname="argument authentication_options", value=authentication_options, expected_type=type_hints["authentication_options"])
            check_type(argname="argument client_cidr_block", value=client_cidr_block, expected_type=type_hints["client_cidr_block"])
            check_type(argname="argument connection_log_options", value=connection_log_options, expected_type=type_hints["connection_log_options"])
            check_type(argname="argument server_certificate_arn", value=server_certificate_arn, expected_type=type_hints["server_certificate_arn"])
            check_type(argname="argument client_connect_options", value=client_connect_options, expected_type=type_hints["client_connect_options"])
            check_type(argname="argument client_login_banner_options", value=client_login_banner_options, expected_type=type_hints["client_login_banner_options"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument disconnect_on_session_timeout", value=disconnect_on_session_timeout, expected_type=type_hints["disconnect_on_session_timeout"])
            check_type(argname="argument dns_servers", value=dns_servers, expected_type=type_hints["dns_servers"])
            check_type(argname="argument security_group_ids", value=security_group_ids, expected_type=type_hints["security_group_ids"])
            check_type(argname="argument self_service_portal", value=self_service_portal, expected_type=type_hints["self_service_portal"])
            check_type(argname="argument session_timeout_hours", value=session_timeout_hours, expected_type=type_hints["session_timeout_hours"])
            check_type(argname="argument split_tunnel", value=split_tunnel, expected_type=type_hints["split_tunnel"])
            check_type(argname="argument tag_specifications", value=tag_specifications, expected_type=type_hints["tag_specifications"])
            check_type(argname="argument transport_protocol", value=transport_protocol, expected_type=type_hints["transport_protocol"])
            check_type(argname="argument vpc_id", value=vpc_id, expected_type=type_hints["vpc_id"])
            check_type(argname="argument vpn_port", value=vpn_port, expected_type=type_hints["vpn_port"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "authentication_options": authentication_options,
            "client_cidr_block": client_cidr_block,
            "connection_log_options": connection_log_options,
            "server_certificate_arn": server_certificate_arn,
        }
        if client_connect_options is not None:
            self._values["client_connect_options"] = client_connect_options
        if client_login_banner_options is not None:
            self._values["client_login_banner_options"] = client_login_banner_options
        if description is not None:
            self._values["description"] = description
        if disconnect_on_session_timeout is not None:
            self._values["disconnect_on_session_timeout"] = disconnect_on_session_timeout
        if dns_servers is not None:
            self._values["dns_servers"] = dns_servers
        if security_group_ids is not None:
            self._values["security_group_ids"] = security_group_ids
        if self_service_portal is not None:
            self._values["self_service_portal"] = self_service_portal
        if session_timeout_hours is not None:
            self._values["session_timeout_hours"] = session_timeout_hours
        if split_tunnel is not None:
            self._values["split_tunnel"] = split_tunnel
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if transport_protocol is not None:
            self._values["transport_protocol"] = transport_protocol
        if vpc_id is not None:
            self._values["vpc_id"] = vpc_id
        if vpn_port is not None:
            self._values["vpn_port"] = vpn_port

    @builtins.property
    def authentication_options(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnClientVpnEndpoint.ClientAuthenticationRequestProperty]]]:
        '''Information about the authentication method to be used to authenticate clients.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-authenticationoptions
        '''
        result = self._values.get("authentication_options")
        assert result is not None, "Required property 'authentication_options' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnClientVpnEndpoint.ClientAuthenticationRequestProperty]]], result)

    @builtins.property
    def client_cidr_block(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, from which to assign client IP addresses.

        The address range cannot overlap with the local CIDR of the VPC in which the associated subnet is located, or the routes that you add manually. The address range cannot be changed after the Client VPN endpoint has been created. Client CIDR range must have a size of at least /22 and must not be greater than /12.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-clientcidrblock
        '''
        result = self._values.get("client_cidr_block")
        assert result is not None, "Required property 'client_cidr_block' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def connection_log_options(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnClientVpnEndpoint.ConnectionLogOptionsProperty]:
        '''Information about the client connection logging options.

        If you enable client connection logging, data about client connections is sent to a Cloudwatch Logs log stream. The following information is logged:

        - Client connection requests
        - Client connection results (successful and unsuccessful)
        - Reasons for unsuccessful client connection requests
        - Client connection termination time

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-connectionlogoptions
        '''
        result = self._values.get("connection_log_options")
        assert result is not None, "Required property 'connection_log_options' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnClientVpnEndpoint.ConnectionLogOptionsProperty], result)

    @builtins.property
    def server_certificate_arn(self) -> builtins.str:
        '''The ARN of the server certificate.

        For more information, see the `AWS Certificate Manager User Guide <https://docs.aws.amazon.com/acm/latest/userguide/>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-servercertificatearn
        '''
        result = self._values.get("server_certificate_arn")
        assert result is not None, "Required property 'server_certificate_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def client_connect_options(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnClientVpnEndpoint.ClientConnectOptionsProperty]]:
        '''The options for managing connection authorization for new client connections.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-clientconnectoptions
        '''
        result = self._values.get("client_connect_options")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnClientVpnEndpoint.ClientConnectOptionsProperty]], result)

    @builtins.property
    def client_login_banner_options(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnClientVpnEndpoint.ClientLoginBannerOptionsProperty]]:
        '''Options for enabling a customizable text banner that will be displayed on AWS provided clients when a VPN session is established.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-clientloginbanneroptions
        '''
        result = self._values.get("client_login_banner_options")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnClientVpnEndpoint.ClientLoginBannerOptionsProperty]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the Client VPN endpoint.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def disconnect_on_session_timeout(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether the client VPN session is disconnected after the maximum ``sessionTimeoutHours`` is reached.

        If ``true`` , users are prompted to reconnect client VPN. If ``false`` , client VPN attempts to reconnect automatically. The default value is ``false`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-disconnectonsessiontimeout
        '''
        result = self._values.get("disconnect_on_session_timeout")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Information about the DNS servers to be used for DNS resolution.

        A Client VPN endpoint can have up to two DNS servers. If no DNS server is specified, the DNS address configured on the device is used for the DNS server.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-dnsservers
        '''
        result = self._values.get("dns_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IDs of one or more security groups to apply to the target network.

        You must also specify the ID of the VPC that contains the security groups.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-securitygroupids
        '''
        result = self._values.get("security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def self_service_portal(self) -> typing.Optional[builtins.str]:
        '''Specify whether to enable the self-service portal for the Client VPN endpoint.

        Default Value: ``enabled``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-selfserviceportal
        '''
        result = self._values.get("self_service_portal")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def session_timeout_hours(self) -> typing.Optional[jsii.Number]:
        '''The maximum VPN session duration time in hours.

        Valid values: ``8 | 10 | 12 | 24``

        Default value: ``24``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-sessiontimeouthours
        '''
        result = self._values.get("session_timeout_hours")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def split_tunnel(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether split-tunnel is enabled on the AWS Client VPN endpoint.

        By default, split-tunnel on a VPN endpoint is disabled.

        For information about split-tunnel VPN endpoints, see `Split-tunnel AWS Client VPN endpoint <https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html>`_ in the *AWS Client VPN Administrator Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-splittunnel
        '''
        result = self._values.get("split_tunnel")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnClientVpnEndpoint.TagSpecificationProperty]]]]:
        '''The tags to apply to the Client VPN endpoint during creation.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnClientVpnEndpoint.TagSpecificationProperty]]]], result)

    @builtins.property
    def transport_protocol(self) -> typing.Optional[builtins.str]:
        '''The transport protocol to be used by the VPN session.

        Default value: ``udp``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-transportprotocol
        '''
        result = self._values.get("transport_protocol")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the VPC to associate with the Client VPN endpoint.

        If no security group IDs are specified in the request, the default security group for the VPC is applied.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-vpcid
        '''
        result = self._values.get("vpc_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpn_port(self) -> typing.Optional[jsii.Number]:
        '''The port number to assign to the Client VPN endpoint for TCP and UDP traffic.

        Valid Values: ``443`` | ``1194``

        Default Value: ``443``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-vpnport
        '''
        result = self._values.get("vpn_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnClientVpnRoute(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnRoute",
):
    '''Specifies a network route to add to a Client VPN endpoint.

    Each Client VPN endpoint has a route table that describes the available destination network routes. Each route in the route table specifies the path for traffic to specific resources or networks.

    A target network association must be created before you can specify a route. If you're setting up all the components of a Client VPN endpoint at the same time, you must use the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ to declare a dependency on the ``AWS::EC2::ClientVpnTargetNetworkAssociation`` resource.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html
    :cloudformationResource: AWS::EC2::ClientVpnRoute
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_client_vpn_route = ec2.CfnClientVpnRoute(self, "MyCfnClientVpnRoute",
            client_vpn_endpoint_id="clientVpnEndpointId",
            destination_cidr_block="destinationCidrBlock",
            target_vpc_subnet_id="targetVpcSubnetId",
        
            # the properties below are optional
            description="description"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        client_vpn_endpoint_id: builtins.str,
        destination_cidr_block: builtins.str,
        target_vpc_subnet_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint to which to add the route.
        :param destination_cidr_block: The IPv4 address range, in CIDR notation, of the route destination. For example:. - To add a route for Internet access, enter ``0.0.0.0/0`` - To add a route for a peered VPC, enter the peered VPC's IPv4 CIDR range - To add a route for an on-premises network, enter the AWS Site-to-Site VPN connection's IPv4 CIDR range - To add a route for the local network, enter the client CIDR range
        :param target_vpc_subnet_id: The ID of the subnet through which you want to route traffic. The specified subnet must be an existing target network of the Client VPN endpoint. Alternatively, if you're adding a route for the local network, specify ``local`` .
        :param description: A brief description of the route.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3dfb737204851cc06c49c2c8a2089a474e3c7ab8c9522c2b17a1cda4e8d4e6d9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnClientVpnRouteProps(
            client_vpn_endpoint_id=client_vpn_endpoint_id,
            destination_cidr_block=destination_cidr_block,
            target_vpc_subnet_id=target_vpc_subnet_id,
            description=description,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ebdf42d79ea0b48b9768a7f69f6f849b67a9aceb079a45fd50f96be6327e5155)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f3f633fca54521e752c1bc7455f2ac48bbea65e9b27b2cea6b3fcf87f8910a12)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="clientVpnEndpointId")
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint to which to add the route.'''
        return typing.cast(builtins.str, jsii.get(self, "clientVpnEndpointId"))

    @client_vpn_endpoint_id.setter
    def client_vpn_endpoint_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__87c23fa81bc76240c6ce4c869a804c37228a002dc52b52a37dfc6164270825b1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientVpnEndpointId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="destinationCidrBlock")
    def destination_cidr_block(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, of the route destination.

        For example:.
        '''
        return typing.cast(builtins.str, jsii.get(self, "destinationCidrBlock"))

    @destination_cidr_block.setter
    def destination_cidr_block(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e47d9f8cb47cf5a3263b2bb98b9ce1cc75c52af2f813f0966d8d270ea841ec3b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "destinationCidrBlock", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="targetVpcSubnetId")
    def target_vpc_subnet_id(self) -> builtins.str:
        '''The ID of the subnet through which you want to route traffic.'''
        return typing.cast(builtins.str, jsii.get(self, "targetVpcSubnetId"))

    @target_vpc_subnet_id.setter
    def target_vpc_subnet_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__512be83797e2ddde08781d1160b36d983c2af081515532f8e3bb0689d136cfb4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetVpcSubnetId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the route.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b10f6f91046363ff96a4de67abb063cdae0418c01d141e45220d29a8a6c8d3a8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnRouteProps",
    jsii_struct_bases=[],
    name_mapping={
        "client_vpn_endpoint_id": "clientVpnEndpointId",
        "destination_cidr_block": "destinationCidrBlock",
        "target_vpc_subnet_id": "targetVpcSubnetId",
        "description": "description",
    },
)
class CfnClientVpnRouteProps:
    def __init__(
        self,
        *,
        client_vpn_endpoint_id: builtins.str,
        destination_cidr_block: builtins.str,
        target_vpc_subnet_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnClientVpnRoute``.

        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint to which to add the route.
        :param destination_cidr_block: The IPv4 address range, in CIDR notation, of the route destination. For example:. - To add a route for Internet access, enter ``0.0.0.0/0`` - To add a route for a peered VPC, enter the peered VPC's IPv4 CIDR range - To add a route for an on-premises network, enter the AWS Site-to-Site VPN connection's IPv4 CIDR range - To add a route for the local network, enter the client CIDR range
        :param target_vpc_subnet_id: The ID of the subnet through which you want to route traffic. The specified subnet must be an existing target network of the Client VPN endpoint. Alternatively, if you're adding a route for the local network, specify ``local`` .
        :param description: A brief description of the route.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_client_vpn_route_props = ec2.CfnClientVpnRouteProps(
                client_vpn_endpoint_id="clientVpnEndpointId",
                destination_cidr_block="destinationCidrBlock",
                target_vpc_subnet_id="targetVpcSubnetId",
            
                # the properties below are optional
                description="description"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__04e293f5c6f16213a7dc895687cc68d3defb2606f9d9e24c82d8f28fab9a0c49)
            check_type(argname="argument client_vpn_endpoint_id", value=client_vpn_endpoint_id, expected_type=type_hints["client_vpn_endpoint_id"])
            check_type(argname="argument destination_cidr_block", value=destination_cidr_block, expected_type=type_hints["destination_cidr_block"])
            check_type(argname="argument target_vpc_subnet_id", value=target_vpc_subnet_id, expected_type=type_hints["target_vpc_subnet_id"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "client_vpn_endpoint_id": client_vpn_endpoint_id,
            "destination_cidr_block": destination_cidr_block,
            "target_vpc_subnet_id": target_vpc_subnet_id,
        }
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint to which to add the route.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-clientvpnendpointid
        '''
        result = self._values.get("client_vpn_endpoint_id")
        assert result is not None, "Required property 'client_vpn_endpoint_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def destination_cidr_block(self) -> builtins.str:
        '''The IPv4 address range, in CIDR notation, of the route destination. For example:.

        - To add a route for Internet access, enter ``0.0.0.0/0``
        - To add a route for a peered VPC, enter the peered VPC's IPv4 CIDR range
        - To add a route for an on-premises network, enter the AWS Site-to-Site VPN connection's IPv4 CIDR range
        - To add a route for the local network, enter the client CIDR range

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-destinationcidrblock
        '''
        result = self._values.get("destination_cidr_block")
        assert result is not None, "Required property 'destination_cidr_block' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_vpc_subnet_id(self) -> builtins.str:
        '''The ID of the subnet through which you want to route traffic.

        The specified subnet must be an existing target network of the Client VPN endpoint.

        Alternatively, if you're adding a route for the local network, specify ``local`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-targetvpcsubnetid
        '''
        result = self._values.get("target_vpc_subnet_id")
        assert result is not None, "Required property 'target_vpc_subnet_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A brief description of the route.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnClientVpnTargetNetworkAssociation(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnTargetNetworkAssociation",
):
    '''Specifies a target network to associate with a Client VPN endpoint.

    A target network is a subnet in a VPC. You can associate multiple subnets from the same VPC with a Client VPN endpoint. You can associate only one subnet in each Availability Zone. We recommend that you associate at least two subnets to provide Availability Zone redundancy.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html
    :cloudformationResource: AWS::EC2::ClientVpnTargetNetworkAssociation
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_client_vpn_target_network_association = ec2.CfnClientVpnTargetNetworkAssociation(self, "MyCfnClientVpnTargetNetworkAssociation",
            client_vpn_endpoint_id="clientVpnEndpointId",
            subnet_id="subnetId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        client_vpn_endpoint_id: builtins.str,
        subnet_id: builtins.str,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint.
        :param subnet_id: The ID of the subnet to associate with the Client VPN endpoint.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3246cdc0c4c45c3bb7ca145721849485de3058412b28e18e6a2521dee9feba7)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnClientVpnTargetNetworkAssociationProps(
            client_vpn_endpoint_id=client_vpn_endpoint_id, subnet_id=subnet_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__328449dbe5f58a7132b6b6b1ba468f97ad09c68f19c7763fd188cb55dd0a50c3)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6ce31eae78612f60b70ab52b2bb1a8bfb8da40aed559073873834454e4009da2)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="clientVpnEndpointId")
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint.'''
        return typing.cast(builtins.str, jsii.get(self, "clientVpnEndpointId"))

    @client_vpn_endpoint_id.setter
    def client_vpn_endpoint_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__256c01852f9be0828ae3a474428434d9bcf06b1f2f38dd2cf5ff3440f4f4a5c4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientVpnEndpointId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="subnetId")
    def subnet_id(self) -> builtins.str:
        '''The ID of the subnet to associate with the Client VPN endpoint.'''
        return typing.cast(builtins.str, jsii.get(self, "subnetId"))

    @subnet_id.setter
    def subnet_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__74bdd32493a6798a704c13cb4090daafbc17fc71b0f7e4b1dd8a078962344e35)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "subnetId", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnClientVpnTargetNetworkAssociationProps",
    jsii_struct_bases=[],
    name_mapping={
        "client_vpn_endpoint_id": "clientVpnEndpointId",
        "subnet_id": "subnetId",
    },
)
class CfnClientVpnTargetNetworkAssociationProps:
    def __init__(
        self,
        *,
        client_vpn_endpoint_id: builtins.str,
        subnet_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnClientVpnTargetNetworkAssociation``.

        :param client_vpn_endpoint_id: The ID of the Client VPN endpoint.
        :param subnet_id: The ID of the subnet to associate with the Client VPN endpoint.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_client_vpn_target_network_association_props = ec2.CfnClientVpnTargetNetworkAssociationProps(
                client_vpn_endpoint_id="clientVpnEndpointId",
                subnet_id="subnetId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__121d2c21f06ef1b4efdedfd8846bc605f91f3db0cc9c2844409f378474c9b397)
            check_type(argname="argument client_vpn_endpoint_id", value=client_vpn_endpoint_id, expected_type=type_hints["client_vpn_endpoint_id"])
            check_type(argname="argument subnet_id", value=subnet_id, expected_type=type_hints["subnet_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "client_vpn_endpoint_id": client_vpn_endpoint_id,
            "subnet_id": subnet_id,
        }

    @builtins.property
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''The ID of the Client VPN endpoint.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html#cfn-ec2-clientvpntargetnetworkassociation-clientvpnendpointid
        '''
        result = self._values.get("client_vpn_endpoint_id")
        assert result is not None, "Required property 'client_vpn_endpoint_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def subnet_id(self) -> builtins.str:
        '''The ID of the subnet to associate with the Client VPN endpoint.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html#cfn-ec2-clientvpntargetnetworkassociation-subnetid
        '''
        result = self._values.get("subnet_id")
        assert result is not None, "Required property 'subnet_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnTargetNetworkAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnCustomerGateway(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnCustomerGateway",
):
    '''Specifies a customer gateway.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customergateway.html
    :cloudformationResource: AWS::EC2::CustomerGateway
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_customer_gateway = ec2.CfnCustomerGateway(self, "MyCfnCustomerGateway",
            ip_address="ipAddress",
            type="type",
        
            # the properties below are optional
            bgp_asn=123,
            bgp_asn_extended=123,
            certificate_arn="certificateArn",
            device_name="deviceName",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        ip_address: builtins.str,
        type: builtins.str,
        bgp_asn: typing.Optional[jsii.Number] = None,
        bgp_asn_extended: typing.Optional[jsii.Number] = None,
        certificate_arn: typing.Optional[builtins.str] = None,
        device_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param ip_address: IPv4 address for the customer gateway device's outside interface. The address must be static. If ``OutsideIpAddressType`` in your VPN connection options is set to ``PrivateIpv4`` , you can use an RFC6598 or RFC1918 private IPv4 address. If ``OutsideIpAddressType`` is set to ``PublicIpv4`` , you can use a public IPv4 address.
        :param type: The type of VPN connection that this customer gateway supports ( ``ipsec.1`` ).
        :param bgp_asn: For customer gateway devices that support BGP, specify the device's ASN. You must specify either ``BgpAsn`` or ``BgpAsnExtended`` when creating the customer gateway. If the ASN is larger than ``2,147,483,647`` , you must use ``BgpAsnExtended`` . Default: 65000 Valid values: ``1`` to ``2,147,483,647`` Default: - 65000
        :param bgp_asn_extended: For customer gateway devices that support BGP, specify the device's ASN. You must specify either ``BgpAsn`` or ``BgpAsnExtended`` when creating the customer gateway. If the ASN is larger than ``2,147,483,647`` , you must use ``BgpAsnExtended`` . Valid values: ``2,147,483,648`` to ``4,294,967,295``
        :param certificate_arn: The Amazon Resource Name (ARN) for the customer gateway certificate.
        :param device_name: The name of customer gateway device.
        :param tags: One or more tags for the customer gateway.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__16b41182e007e05b84fd0c97afc1e26001e78a56de2eb5b10c9f809de0654b0d)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnCustomerGatewayProps(
            ip_address=ip_address,
            type=type,
            bgp_asn=bgp_asn,
            bgp_asn_extended=bgp_asn_extended,
            certificate_arn=certificate_arn,
            device_name=device_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ce54eac1ec957fc55ffb27a654680402c3468f669a41d23ab734690d410698df)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a44104c4ad329cfdabba2866cf426d821fa85ba6f3aa04801988c746bd8ddad9)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCustomerGatewayId")
    def attr_customer_gateway_id(self) -> builtins.str:
        '''The ID of the customer gateway.

        :cloudformationAttribute: CustomerGatewayId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCustomerGatewayId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="ipAddress")
    def ip_address(self) -> builtins.str:
        '''IPv4 address for the customer gateway device's outside interface.'''
        return typing.cast(builtins.str, jsii.get(self, "ipAddress"))

    @ip_address.setter
    def ip_address(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a312c99e6832f0396cbd7c4d05fbab836db0503cf37ecee5f1f64bd213516c2e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ipAddress", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''The type of VPN connection that this customer gateway supports ( ``ipsec.1`` ).'''
        return typing.cast(builtins.str, jsii.get(self, "type"))

    @type.setter
    def type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ae973d5ca9904c069d03cbf10a1e3fdf7736cc00ca43663eb07598f97c4e5771)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "type", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="bgpAsn")
    def bgp_asn(self) -> typing.Optional[jsii.Number]:
        '''For customer gateway devices that support BGP, specify the device's ASN.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "bgpAsn"))

    @bgp_asn.setter
    def bgp_asn(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__84dfb7d1775bd2bb124f990570c9a2ef23fafd01744cfe248fcb360562f57ca9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "bgpAsn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="bgpAsnExtended")
    def bgp_asn_extended(self) -> typing.Optional[jsii.Number]:
        '''For customer gateway devices that support BGP, specify the device's ASN.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "bgpAsnExtended"))

    @bgp_asn_extended.setter
    def bgp_asn_extended(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f41644d25c48e5c3c87a361ba478bdb4a18bf473fe1582fa35c6311f6d5284d8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "bgpAsnExtended", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="certificateArn")
    def certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) for the customer gateway certificate.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "certificateArn"))

    @certificate_arn.setter
    def certificate_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4a4b900e840c5be3a2b16a5177f91335cf813daeca359e549a639cb05a03ac63)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "certificateArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="deviceName")
    def device_name(self) -> typing.Optional[builtins.str]:
        '''The name of customer gateway device.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "deviceName"))

    @device_name.setter
    def device_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__71d74664cf79e34328f5f6958fdd674c45b2780c9910dd252b7c5e9caba963f0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deviceName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''One or more tags for the customer gateway.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__05a111d13df5583de4721245805f1cb23e3c81e0e12774d2a044cf75bba55873)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnCustomerGatewayProps",
    jsii_struct_bases=[],
    name_mapping={
        "ip_address": "ipAddress",
        "type": "type",
        "bgp_asn": "bgpAsn",
        "bgp_asn_extended": "bgpAsnExtended",
        "certificate_arn": "certificateArn",
        "device_name": "deviceName",
        "tags": "tags",
    },
)
class CfnCustomerGatewayProps:
    def __init__(
        self,
        *,
        ip_address: builtins.str,
        type: builtins.str,
        bgp_asn: typing.Optional[jsii.Number] = None,
        bgp_asn_extended: typing.Optional[jsii.Number] = None,
        certificate_arn: typing.Optional[builtins.str] = None,
        device_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnCustomerGateway``.

        :param ip_address: IPv4 address for the customer gateway device's outside interface. The address must be static. If ``OutsideIpAddressType`` in your VPN connection options is set to ``PrivateIpv4`` , you can use an RFC6598 or RFC1918 private IPv4 address. If ``OutsideIpAddressType`` is set to ``PublicIpv4`` , you can use a public IPv4 address.
        :param type: The type of VPN connection that this customer gateway supports ( ``ipsec.1`` ).
        :param bgp_asn: For customer gateway devices that support BGP, specify the device's ASN. You must specify either ``BgpAsn`` or ``BgpAsnExtended`` when creating the customer gateway. If the ASN is larger than ``2,147,483,647`` , you must use ``BgpAsnExtended`` . Default: 65000 Valid values: ``1`` to ``2,147,483,647`` Default: - 65000
        :param bgp_asn_extended: For customer gateway devices that support BGP, specify the device's ASN. You must specify either ``BgpAsn`` or ``BgpAsnExtended`` when creating the customer gateway. If the ASN is larger than ``2,147,483,647`` , you must use ``BgpAsnExtended`` . Valid values: ``2,147,483,648`` to ``4,294,967,295``
        :param certificate_arn: The Amazon Resource Name (ARN) for the customer gateway certificate.
        :param device_name: The name of customer gateway device.
        :param tags: One or more tags for the customer gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customergateway.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_customer_gateway_props = ec2.CfnCustomerGatewayProps(
                ip_address="ipAddress",
                type="type",
            
                # the properties below are optional
                bgp_asn=123,
                bgp_asn_extended=123,
                certificate_arn="certificateArn",
                device_name="deviceName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b0ef9a2e3e2b6937b21db500a1cd795126e924d9b920931a413ecdb668bfc7ec)
            check_type(argname="argument ip_address", value=ip_address, expected_type=type_hints["ip_address"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument bgp_asn", value=bgp_asn, expected_type=type_hints["bgp_asn"])
            check_type(argname="argument bgp_asn_extended", value=bgp_asn_extended, expected_type=type_hints["bgp_asn_extended"])
            check_type(argname="argument certificate_arn", value=certificate_arn, expected_type=type_hints["certificate_arn"])
            check_type(argname="argument device_name", value=device_name, expected_type=type_hints["device_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "ip_address": ip_address,
            "type": type,
        }
        if bgp_asn is not None:
            self._values["bgp_asn"] = bgp_asn
        if bgp_asn_extended is not None:
            self._values["bgp_asn_extended"] = bgp_asn_extended
        if certificate_arn is not None:
            self._values["certificate_arn"] = certificate_arn
        if device_name is not None:
            self._values["device_name"] = device_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def ip_address(self) -> builtins.str:
        '''IPv4 address for the customer gateway device's outside interface.

        The address must be static. If ``OutsideIpAddressType`` in your VPN connection options is set to ``PrivateIpv4`` , you can use an RFC6598 or RFC1918 private IPv4 address. If ``OutsideIpAddressType`` is set to ``PublicIpv4`` , you can use a public IPv4 address.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customergateway.html#cfn-ec2-customergateway-ipaddress
        '''
        result = self._values.get("ip_address")
        assert result is not None, "Required property 'ip_address' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type(self) -> builtins.str:
        '''The type of VPN connection that this customer gateway supports ( ``ipsec.1`` ).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customergateway.html#cfn-ec2-customergateway-type
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def bgp_asn(self) -> typing.Optional[jsii.Number]:
        '''For customer gateway devices that support BGP, specify the device's ASN.

        You must specify either ``BgpAsn`` or ``BgpAsnExtended`` when creating the customer gateway. If the ASN is larger than ``2,147,483,647`` , you must use ``BgpAsnExtended`` .

        Default: 65000

        Valid values: ``1`` to ``2,147,483,647``

        :default: - 65000

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customergateway.html#cfn-ec2-customergateway-bgpasn
        '''
        result = self._values.get("bgp_asn")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def bgp_asn_extended(self) -> typing.Optional[jsii.Number]:
        '''For customer gateway devices that support BGP, specify the device's ASN.

        You must specify either ``BgpAsn`` or ``BgpAsnExtended`` when creating the customer gateway. If the ASN is larger than ``2,147,483,647`` , you must use ``BgpAsnExtended`` .

        Valid values: ``2,147,483,648`` to ``4,294,967,295``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customergateway.html#cfn-ec2-customergateway-bgpasnextended
        '''
        result = self._values.get("bgp_asn_extended")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) for the customer gateway certificate.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customergateway.html#cfn-ec2-customergateway-certificatearn
        '''
        result = self._values.get("certificate_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def device_name(self) -> typing.Optional[builtins.str]:
        '''The name of customer gateway device.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customergateway.html#cfn-ec2-customergateway-devicename
        '''
        result = self._values.get("device_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''One or more tags for the customer gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customergateway.html#cfn-ec2-customergateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCustomerGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnDHCPOptions(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnDHCPOptions",
):
    '''Specifies a set of DHCP options for your VPC.

    You must specify at least one of the following properties: ``DomainNameServers`` , ``NetbiosNameServers`` , ``NtpServers`` . If you specify ``NetbiosNameServers`` , you must specify ``NetbiosNodeType`` .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html
    :cloudformationResource: AWS::EC2::DHCPOptions
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_dHCPOptions = ec2.CfnDHCPOptions(self, "MyCfnDHCPOptions",
            domain_name="domainName",
            domain_name_servers=["domainNameServers"],
            ipv6_address_preferred_lease_time=123,
            netbios_name_servers=["netbiosNameServers"],
            netbios_node_type=123,
            ntp_servers=["ntpServers"],
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        domain_name: typing.Optional[builtins.str] = None,
        domain_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        ipv6_address_preferred_lease_time: typing.Optional[jsii.Number] = None,
        netbios_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        netbios_node_type: typing.Optional[jsii.Number] = None,
        ntp_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param domain_name: This value is used to complete unqualified DNS hostnames. If you're using AmazonProvidedDNS in ``us-east-1`` , specify ``ec2.internal`` . If you're using AmazonProvidedDNS in another Region, specify *region* . ``compute.internal`` (for example, ``ap-northeast-1.compute.internal`` ). Otherwise, specify a domain name (for example, *MyCompany.com* ).
        :param domain_name_servers: The IPv4 addresses of up to four domain name servers, or ``AmazonProvidedDNS`` . The default is ``AmazonProvidedDNS`` . To have your instance receive a custom DNS hostname as specified in ``DomainName`` , you must set this property to a custom DNS server.
        :param ipv6_address_preferred_lease_time: A value (in seconds, minutes, hours, or years) for how frequently a running instance with an IPv6 assigned to it goes through DHCPv6 lease renewal. Acceptable values are between 140 and 2147483647 seconds (approximately 68 years). If no value is entered, the default lease time is 140 seconds. If you use long-term addressing for EC2 instances, you can increase the lease time and avoid frequent lease renewal requests. Lease renewal typically occurs when half of the lease time has elapsed.
        :param netbios_name_servers: The IPv4 addresses of up to four NetBIOS name servers.
        :param netbios_node_type: The NetBIOS node type (1, 2, 4, or 8). We recommend that you specify 2 (broadcast and multicast are not currently supported).
        :param ntp_servers: The IPv4 addresses of up to four Network Time Protocol (NTP) servers.
        :param tags: Any tags assigned to the DHCP options set.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5f22238da23e1a95528fc55d3e3a28b661b69a5dfe2d1e64620bef1ebd86deca)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDHCPOptionsProps(
            domain_name=domain_name,
            domain_name_servers=domain_name_servers,
            ipv6_address_preferred_lease_time=ipv6_address_preferred_lease_time,
            netbios_name_servers=netbios_name_servers,
            netbios_node_type=netbios_node_type,
            ntp_servers=ntp_servers,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4bd34186ec5701fad199396bd872adb91c15f71647beefe007bb9dbafe17d86b)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c4b6b354957e45d57a7d74c91d58645803e6d3b3dc74c069422742dd52f73de0)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDhcpOptionsId")
    def attr_dhcp_options_id(self) -> builtins.str:
        '''The ID of the DHCP options set.

        :cloudformationAttribute: DhcpOptionsId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDhcpOptionsId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''This value is used to complete unqualified DNS hostnames.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domainName"))

    @domain_name.setter
    def domain_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3ef924b0a20b9413b973ce7cba216c31e794568c9f99a083b994d3e69ebb3c5e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="domainNameServers")
    def domain_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four domain name servers, or ``AmazonProvidedDNS`` .'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "domainNameServers"))

    @domain_name_servers.setter
    def domain_name_servers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d9ee8e87c9581893159e005c5c7c786324d9f9c9a41ba0af4f72aedd3a9903ae)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainNameServers", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="ipv6AddressPreferredLeaseTime")
    def ipv6_address_preferred_lease_time(self) -> typing.Optional[jsii.Number]:
        '''A value (in seconds, minutes, hours, or years) for how frequently a running instance with an IPv6 assigned to it goes through DHCPv6 lease renewal.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "ipv6AddressPreferredLeaseTime"))

    @ipv6_address_preferred_lease_time.setter
    def ipv6_address_preferred_lease_time(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0cbb91e4162eb4b7aab6b1e90fa6fbbbade04b38a95d6c1bb778946ae93b50a3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ipv6AddressPreferredLeaseTime", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="netbiosNameServers")
    def netbios_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four NetBIOS name servers.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "netbiosNameServers"))

    @netbios_name_servers.setter
    def netbios_name_servers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e3b8e4bd8f79a27f01b9c18995f6a45e13c80f3ff9f392b3c558b6cc5a5e5dba)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "netbiosNameServers", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="netbiosNodeType")
    def netbios_node_type(self) -> typing.Optional[jsii.Number]:
        '''The NetBIOS node type (1, 2, 4, or 8).'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "netbiosNodeType"))

    @netbios_node_type.setter
    def netbios_node_type(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__72bd8121cf33b33a52103c326c339f9ce10fbb12df0d71f411557aa642a9c385)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "netbiosNodeType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="ntpServers")
    def ntp_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four Network Time Protocol (NTP) servers.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "ntpServers"))

    @ntp_servers.setter
    def ntp_servers(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__183fd1ef6a4ee905fe372ccc2f27b5ba3a48f856c21c3c678eb1b4db43bd1ab9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ntpServers", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Any tags assigned to the DHCP options set.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f4c459d5f98dbe6bf3fd3f9f7fca2823e7ee9510398ee1206c9aa28765f1d61c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnDHCPOptionsProps",
    jsii_struct_bases=[],
    name_mapping={
        "domain_name": "domainName",
        "domain_name_servers": "domainNameServers",
        "ipv6_address_preferred_lease_time": "ipv6AddressPreferredLeaseTime",
        "netbios_name_servers": "netbiosNameServers",
        "netbios_node_type": "netbiosNodeType",
        "ntp_servers": "ntpServers",
        "tags": "tags",
    },
)
class CfnDHCPOptionsProps:
    def __init__(
        self,
        *,
        domain_name: typing.Optional[builtins.str] = None,
        domain_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        ipv6_address_preferred_lease_time: typing.Optional[jsii.Number] = None,
        netbios_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        netbios_node_type: typing.Optional[jsii.Number] = None,
        ntp_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDHCPOptions``.

        :param domain_name: This value is used to complete unqualified DNS hostnames. If you're using AmazonProvidedDNS in ``us-east-1`` , specify ``ec2.internal`` . If you're using AmazonProvidedDNS in another Region, specify *region* . ``compute.internal`` (for example, ``ap-northeast-1.compute.internal`` ). Otherwise, specify a domain name (for example, *MyCompany.com* ).
        :param domain_name_servers: The IPv4 addresses of up to four domain name servers, or ``AmazonProvidedDNS`` . The default is ``AmazonProvidedDNS`` . To have your instance receive a custom DNS hostname as specified in ``DomainName`` , you must set this property to a custom DNS server.
        :param ipv6_address_preferred_lease_time: A value (in seconds, minutes, hours, or years) for how frequently a running instance with an IPv6 assigned to it goes through DHCPv6 lease renewal. Acceptable values are between 140 and 2147483647 seconds (approximately 68 years). If no value is entered, the default lease time is 140 seconds. If you use long-term addressing for EC2 instances, you can increase the lease time and avoid frequent lease renewal requests. Lease renewal typically occurs when half of the lease time has elapsed.
        :param netbios_name_servers: The IPv4 addresses of up to four NetBIOS name servers.
        :param netbios_node_type: The NetBIOS node type (1, 2, 4, or 8). We recommend that you specify 2 (broadcast and multicast are not currently supported).
        :param ntp_servers: The IPv4 addresses of up to four Network Time Protocol (NTP) servers.
        :param tags: Any tags assigned to the DHCP options set.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_dHCPOptions_props = ec2.CfnDHCPOptionsProps(
                domain_name="domainName",
                domain_name_servers=["domainNameServers"],
                ipv6_address_preferred_lease_time=123,
                netbios_name_servers=["netbiosNameServers"],
                netbios_node_type=123,
                ntp_servers=["ntpServers"],
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__569097ad87e4dddbcfaee4fb50fc7ddb345c4f97ae82fcf897497039e7f81a38)
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument domain_name_servers", value=domain_name_servers, expected_type=type_hints["domain_name_servers"])
            check_type(argname="argument ipv6_address_preferred_lease_time", value=ipv6_address_preferred_lease_time, expected_type=type_hints["ipv6_address_preferred_lease_time"])
            check_type(argname="argument netbios_name_servers", value=netbios_name_servers, expected_type=type_hints["netbios_name_servers"])
            check_type(argname="argument netbios_node_type", value=netbios_node_type, expected_type=type_hints["netbios_node_type"])
            check_type(argname="argument ntp_servers", value=ntp_servers, expected_type=type_hints["ntp_servers"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if domain_name is not None:
            self._values["domain_name"] = domain_name
        if domain_name_servers is not None:
            self._values["domain_name_servers"] = domain_name_servers
        if ipv6_address_preferred_lease_time is not None:
            self._values["ipv6_address_preferred_lease_time"] = ipv6_address_preferred_lease_time
        if netbios_name_servers is not None:
            self._values["netbios_name_servers"] = netbios_name_servers
        if netbios_node_type is not None:
            self._values["netbios_node_type"] = netbios_node_type
        if ntp_servers is not None:
            self._values["ntp_servers"] = ntp_servers
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''This value is used to complete unqualified DNS hostnames.

        If you're using AmazonProvidedDNS in ``us-east-1`` , specify ``ec2.internal`` . If you're using AmazonProvidedDNS in another Region, specify *region* . ``compute.internal`` (for example, ``ap-northeast-1.compute.internal`` ). Otherwise, specify a domain name (for example, *MyCompany.com* ).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-domainname
        '''
        result = self._values.get("domain_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four domain name servers, or ``AmazonProvidedDNS`` .

        The default is ``AmazonProvidedDNS`` . To have your instance receive a custom DNS hostname as specified in ``DomainName`` , you must set this property to a custom DNS server.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-domainnameservers
        '''
        result = self._values.get("domain_name_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def ipv6_address_preferred_lease_time(self) -> typing.Optional[jsii.Number]:
        '''A value (in seconds, minutes, hours, or years) for how frequently a running instance with an IPv6 assigned to it goes through DHCPv6 lease renewal.

        Acceptable values are between 140 and 2147483647 seconds (approximately 68 years). If no value is entered, the default lease time is 140 seconds. If you use long-term addressing for EC2 instances, you can increase the lease time and avoid frequent lease renewal requests. Lease renewal typically occurs when half of the lease time has elapsed.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-ipv6addresspreferredleasetime
        '''
        result = self._values.get("ipv6_address_preferred_lease_time")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def netbios_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four NetBIOS name servers.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-netbiosnameservers
        '''
        result = self._values.get("netbios_name_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def netbios_node_type(self) -> typing.Optional[jsii.Number]:
        '''The NetBIOS node type (1, 2, 4, or 8).

        We recommend that you specify 2 (broadcast and multicast are not currently supported).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-netbiosnodetype
        '''
        result = self._values.get("netbios_node_type")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def ntp_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The IPv4 addresses of up to four Network Time Protocol (NTP) servers.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-ntpservers
        '''
        result = self._values.get("ntp_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Any tags assigned to the DHCP options set.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcpoptions.html#cfn-ec2-dhcpoptions-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDHCPOptionsProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnEC2Fleet(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet",
):
    '''Specifies the configuration information to launch a fleet--or group--of instances.

    An EC2 Fleet can launch multiple instance types across multiple Availability Zones, using the On-Demand Instance, Reserved Instance, and Spot Instance purchasing models together. Using EC2 Fleet, you can define separate On-Demand and Spot capacity targets, specify the instance types that work best for your applications, and specify how Amazon EC2 should distribute your fleet capacity within each purchasing model. For more information, see `Launching an EC2 Fleet <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet.html>`_ in the *Amazon EC2 User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html
    :cloudformationResource: AWS::EC2::EC2Fleet
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_eC2_fleet = ec2.CfnEC2Fleet(self, "MyCfnEC2Fleet",
            launch_template_configs=[ec2.CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty(
                launch_template_specification=ec2.CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty(
                    version="version",
        
                    # the properties below are optional
                    launch_template_id="launchTemplateId",
                    launch_template_name="launchTemplateName"
                ),
                overrides=[ec2.CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty(
                    availability_zone="availabilityZone",
                    instance_requirements=ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty(
                        accelerator_count=ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                            max=123,
                            min=123
                        ),
                        accelerator_manufacturers=["acceleratorManufacturers"],
                        accelerator_names=["acceleratorNames"],
                        accelerator_total_memory_mi_b=ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                            max=123,
                            min=123
                        ),
                        accelerator_types=["acceleratorTypes"],
                        allowed_instance_types=["allowedInstanceTypes"],
                        bare_metal="bareMetal",
                        baseline_ebs_bandwidth_mbps=ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                            max=123,
                            min=123
                        ),
                        baseline_performance_factors=ec2.CfnEC2Fleet.BaselinePerformanceFactorsRequestProperty(
                            cpu=ec2.CfnEC2Fleet.CpuPerformanceFactorRequestProperty(
                                references=[ec2.CfnEC2Fleet.PerformanceFactorReferenceRequestProperty(
                                    instance_family="instanceFamily"
                                )]
                            )
                        ),
                        burstable_performance="burstablePerformance",
                        cpu_manufacturers=["cpuManufacturers"],
                        excluded_instance_types=["excludedInstanceTypes"],
                        instance_generations=["instanceGenerations"],
                        local_storage="localStorage",
                        local_storage_types=["localStorageTypes"],
                        max_spot_price_as_percentage_of_optimal_on_demand_price=123,
                        memory_gi_bPer_vCpu=ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                            max=123,
                            min=123
                        ),
                        memory_mi_b=ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                            max=123,
                            min=123
                        ),
                        network_bandwidth_gbps=ec2.CfnEC2Fleet.NetworkBandwidthGbpsRequestProperty(
                            max=123,
                            min=123
                        ),
                        network_interface_count=ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                            max=123,
                            min=123
                        ),
                        on_demand_max_price_percentage_over_lowest_price=123,
                        require_hibernate_support=False,
                        spot_max_price_percentage_over_lowest_price=123,
                        total_local_storage_gb=ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                            max=123,
                            min=123
                        ),
                        v_cpu_count=ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                            max=123,
                            min=123
                        )
                    ),
                    instance_type="instanceType",
                    max_price="maxPrice",
                    placement=ec2.CfnEC2Fleet.PlacementProperty(
                        affinity="affinity",
                        availability_zone="availabilityZone",
                        group_name="groupName",
                        host_id="hostId",
                        host_resource_group_arn="hostResourceGroupArn",
                        partition_number=123,
                        spread_domain="spreadDomain",
                        tenancy="tenancy"
                    ),
                    priority=123,
                    subnet_id="subnetId",
                    weighted_capacity=123
                )]
            )],
            target_capacity_specification=ec2.CfnEC2Fleet.TargetCapacitySpecificationRequestProperty(
                total_target_capacity=123,
        
                # the properties below are optional
                default_target_capacity_type="defaultTargetCapacityType",
                on_demand_target_capacity=123,
                spot_target_capacity=123,
                target_capacity_unit_type="targetCapacityUnitType"
            ),
        
            # the properties below are optional
            context="context",
            excess_capacity_termination_policy="excessCapacityTerminationPolicy",
            on_demand_options=ec2.CfnEC2Fleet.OnDemandOptionsRequestProperty(
                allocation_strategy="allocationStrategy",
                capacity_reservation_options=ec2.CfnEC2Fleet.CapacityReservationOptionsRequestProperty(
                    usage_strategy="usageStrategy"
                ),
                max_total_price="maxTotalPrice",
                min_target_capacity=123,
                single_availability_zone=False,
                single_instance_type=False
            ),
            replace_unhealthy_instances=False,
            spot_options=ec2.CfnEC2Fleet.SpotOptionsRequestProperty(
                allocation_strategy="allocationStrategy",
                instance_interruption_behavior="instanceInterruptionBehavior",
                instance_pools_to_use_count=123,
                maintenance_strategies=ec2.CfnEC2Fleet.MaintenanceStrategiesProperty(
                    capacity_rebalance=ec2.CfnEC2Fleet.CapacityRebalanceProperty(
                        replacement_strategy="replacementStrategy",
                        termination_delay=123
                    )
                ),
                max_total_price="maxTotalPrice",
                min_target_capacity=123,
                single_availability_zone=False,
                single_instance_type=False
            ),
            tag_specifications=[ec2.CfnEC2Fleet.TagSpecificationProperty(
                resource_type="resourceType",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )],
            terminate_instances_with_expiration=False,
            type="type",
            valid_from="validFrom",
            valid_until="validUntil"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        launch_template_configs: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty", typing.Dict[builtins.str, typing.Any]]]]],
        target_capacity_specification: typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.TargetCapacitySpecificationRequestProperty", typing.Dict[builtins.str, typing.Any]]],
        context: typing.Optional[builtins.str] = None,
        excess_capacity_termination_policy: typing.Optional[builtins.str] = None,
        on_demand_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.OnDemandOptionsRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        replace_unhealthy_instances: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        spot_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.SpotOptionsRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tag_specifications: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.TagSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        terminate_instances_with_expiration: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        type: typing.Optional[builtins.str] = None,
        valid_from: typing.Optional[builtins.str] = None,
        valid_until: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param launch_template_configs: The configuration for the EC2 Fleet.
        :param target_capacity_specification: The number of units to request.
        :param context: Reserved.
        :param excess_capacity_termination_policy: Indicates whether running instances should be terminated if the total target capacity of the EC2 Fleet is decreased below the current size of the EC2 Fleet. Supported only for fleets of type ``maintain`` .
        :param on_demand_options: Describes the configuration of On-Demand Instances in an EC2 Fleet.
        :param replace_unhealthy_instances: Indicates whether EC2 Fleet should replace unhealthy Spot Instances. Supported only for fleets of type ``maintain`` . For more information, see `EC2 Fleet health checks <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/manage-ec2-fleet.html#ec2-fleet-health-checks>`_ in the *Amazon EC2 User Guide* .
        :param spot_options: Describes the configuration of Spot Instances in an EC2 Fleet.
        :param tag_specifications: The key-value pair for tagging the EC2 Fleet request on creation. For more information, see `Tag your resources <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html#tag-resources>`_ . If the fleet type is ``instant`` , specify a resource type of ``fleet`` to tag the fleet or ``instance`` to tag the instances at launch. If the fleet type is ``maintain`` or ``request`` , specify a resource type of ``fleet`` to tag the fleet. You cannot specify a resource type of ``instance`` . To tag instances at launch, specify the tags in a `launch template <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html#create-launch-template>`_ .
        :param terminate_instances_with_expiration: Indicates whether running instances should be terminated when the EC2 Fleet expires.
        :param type: The fleet type. The default value is ``maintain`` . - ``maintain`` - The EC2 Fleet places an asynchronous request for your desired capacity, and continues to maintain your desired Spot capacity by replenishing interrupted Spot Instances. - ``request`` - The EC2 Fleet places an asynchronous one-time request for your desired capacity, but does submit Spot requests in alternative capacity pools if Spot capacity is unavailable, and does not maintain Spot capacity if Spot Instances are interrupted. - ``instant`` - The EC2 Fleet places a synchronous one-time request for your desired capacity, and returns errors for any instances that could not be launched. For more information, see `EC2 Fleet request types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-request-type.html>`_ in the *Amazon EC2 User Guide* .
        :param valid_from: The start date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z). The default is to start fulfilling the request immediately.
        :param valid_until: The end date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z). At this point, no new EC2 Fleet requests are placed or able to fulfill the request. If no value is specified, the request remains until you cancel it.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9b04bc874649f768ee829fef298c1c886ebb6fc5aa454a0c698e17c25fc9b40f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnEC2FleetProps(
            launch_template_configs=launch_template_configs,
            target_capacity_specification=target_capacity_specification,
            context=context,
            excess_capacity_termination_policy=excess_capacity_termination_policy,
            on_demand_options=on_demand_options,
            replace_unhealthy_instances=replace_unhealthy_instances,
            spot_options=spot_options,
            tag_specifications=tag_specifications,
            terminate_instances_with_expiration=terminate_instances_with_expiration,
            type=type,
            valid_from=valid_from,
            valid_until=valid_until,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ba2b8151d12fc4a02b9b697317595ccdcf5010cff290619525b157c76a9376e5)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__232ae85cfe3baeed9d41ac592afd4aeecf1a0032e258e71efca09cf806f2edcd)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrFleetId")
    def attr_fleet_id(self) -> builtins.str:
        '''The ID of the EC2 Fleet.

        :cloudformationAttribute: FleetId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFleetId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="launchTemplateConfigs")
    def launch_template_configs(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty"]]]:
        '''The configuration for the EC2 Fleet.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty"]]], jsii.get(self, "launchTemplateConfigs"))

    @launch_template_configs.setter
    def launch_template_configs(
        self,
        value: typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty"]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__11e66fc2296b61712f860ba32f7e4f6bae1d4caa8477ebd1d0e90496b03d47cb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "launchTemplateConfigs", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="targetCapacitySpecification")
    def target_capacity_specification(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.TargetCapacitySpecificationRequestProperty"]:
        '''The number of units to request.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.TargetCapacitySpecificationRequestProperty"], jsii.get(self, "targetCapacitySpecification"))

    @target_capacity_specification.setter
    def target_capacity_specification(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.TargetCapacitySpecificationRequestProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__308e9c06cb8e557e27da19507947b7edd3a3aadc38cdcddcdcc835402d1fdce7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetCapacitySpecification", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="context")
    def context(self) -> typing.Optional[builtins.str]:
        '''Reserved.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "context"))

    @context.setter
    def context(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__14055459b56a2936b7e5b3328bd76b98c965834cbec37d83a630495b8e9a8de1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "context", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="excessCapacityTerminationPolicy")
    def excess_capacity_termination_policy(self) -> typing.Optional[builtins.str]:
        '''Indicates whether running instances should be terminated if the total target capacity of the EC2 Fleet is decreased below the current size of the EC2 Fleet.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "excessCapacityTerminationPolicy"))

    @excess_capacity_termination_policy.setter
    def excess_capacity_termination_policy(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76724f7a5d1a753ad3d973f4e918523326a8dd12f7c437da5af31418195c7222)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "excessCapacityTerminationPolicy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="onDemandOptions")
    def on_demand_options(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.OnDemandOptionsRequestProperty"]]:
        '''Describes the configuration of On-Demand Instances in an EC2 Fleet.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.OnDemandOptionsRequestProperty"]], jsii.get(self, "onDemandOptions"))

    @on_demand_options.setter
    def on_demand_options(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.OnDemandOptionsRequestProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__387749e6f1f8ee96abed2de0d2312d91522d94f22e7c4625726360add11b93e4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "onDemandOptions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="replaceUnhealthyInstances")
    def replace_unhealthy_instances(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether EC2 Fleet should replace unhealthy Spot Instances.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "replaceUnhealthyInstances"))

    @replace_unhealthy_instances.setter
    def replace_unhealthy_instances(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__123332aa6b3b1e5b8f117ed3c89fbd1dc4fd57833234901707c819d14f66fe50)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "replaceUnhealthyInstances", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="spotOptions")
    def spot_options(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.SpotOptionsRequestProperty"]]:
        '''Describes the configuration of Spot Instances in an EC2 Fleet.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.SpotOptionsRequestProperty"]], jsii.get(self, "spotOptions"))

    @spot_options.setter
    def spot_options(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.SpotOptionsRequestProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f056b97eccf2fbd77e6683adf7ef1aeb7b2afb2329fde0d2d6cfb1b273233bea)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spotOptions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.TagSpecificationProperty"]]]]:
        '''The key-value pair for tagging the EC2 Fleet request on creation.

        For more information, see `Tag your resources <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html#tag-resources>`_ .
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.TagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.TagSpecificationProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bece0741240e27126cb8a337d5fce1c6a45094b66f86472a241f9f7432e40a77)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagSpecifications", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="terminateInstancesWithExpiration")
    def terminate_instances_with_expiration(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether running instances should be terminated when the EC2 Fleet expires.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "terminateInstancesWithExpiration"))

    @terminate_instances_with_expiration.setter
    def terminate_instances_with_expiration(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0bd5baf142838966e793088318ff6c3b289335c1b15d0d9e87f74507d42d74df)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "terminateInstancesWithExpiration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> typing.Optional[builtins.str]:
        '''The fleet type.

        The default value is ``maintain`` .
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "type"))

    @type.setter
    def type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__59a50ca71a49c9d182b71322c445a4ba7d65a775570ca036914db2bb21276e74)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "type", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="validFrom")
    def valid_from(self) -> typing.Optional[builtins.str]:
        '''The start date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z).'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "validFrom"))

    @valid_from.setter
    def valid_from(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0dbca3955ea50bceaf0d75c83c574fa2b09d61b437759776878b29bf18cf1db8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "validFrom", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="validUntil")
    def valid_until(self) -> typing.Optional[builtins.str]:
        '''The end date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z).'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "validUntil"))

    @valid_until.setter
    def valid_until(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d545dba13e879cf198f83427ce9f9f5071d32955fc00fa93ec5feaa00d5a36c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "validUntil", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.AcceleratorCountRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class AcceleratorCountRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) on an instance.

            To exclude accelerator-enabled instance types, set ``Max`` to ``0`` .

            :param max: The maximum number of accelerators. To specify no maximum limit, omit this parameter. To exclude accelerator-enabled instance types, set ``Max`` to ``0`` .
            :param min: The minimum number of accelerators. To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratorcountrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                accelerator_count_request_property = ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bc7b29adcb33628864e03a448dfeb55fe20e548d25b6d9dcc6fabe20e73cfe20)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of accelerators.

            To specify no maximum limit, omit this parameter. To exclude accelerator-enabled instance types, set ``Max`` to ``0`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratorcountrequest.html#cfn-ec2-ec2fleet-acceleratorcountrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of accelerators.

            To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratorcountrequest.html#cfn-ec2-ec2fleet-acceleratorcountrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AcceleratorCountRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class AcceleratorTotalMemoryMiBRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum amount of total accelerator memory, in MiB.

            :param max: The maximum amount of accelerator memory, in MiB. To specify no maximum limit, omit this parameter.
            :param min: The minimum amount of accelerator memory, in MiB. To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratortotalmemorymibrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                accelerator_total_memory_mi_bRequest_property = ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__12d4abfc68e759d9e5357219f5f252e61b8f61a3e63ec616b710599588d562e0)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of accelerator memory, in MiB.

            To specify no maximum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratortotalmemorymibrequest.html#cfn-ec2-ec2fleet-acceleratortotalmemorymibrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of accelerator memory, in MiB.

            To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-acceleratortotalmemorymibrequest.html#cfn-ec2-ec2fleet-acceleratortotalmemorymibrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AcceleratorTotalMemoryMiBRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class BaselineEbsBandwidthMbpsRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps.

            For more information, see `Amazon EBS–optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html>`_ in the *Amazon EC2 User Guide* .

            :param max: The maximum baseline bandwidth, in Mbps. To specify no maximum limit, omit this parameter.
            :param min: The minimum baseline bandwidth, in Mbps. To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-baselineebsbandwidthmbpsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                baseline_ebs_bandwidth_mbps_request_property = ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e0a4de467a89057f40232d0ebf30c0027feeac4eb5957e8b9a24d27f33ef59ce)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum baseline bandwidth, in Mbps.

            To specify no maximum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-baselineebsbandwidthmbpsrequest.html#cfn-ec2-ec2fleet-baselineebsbandwidthmbpsrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum baseline bandwidth, in Mbps.

            To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-baselineebsbandwidthmbpsrequest.html#cfn-ec2-ec2fleet-baselineebsbandwidthmbpsrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BaselineEbsBandwidthMbpsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.BaselinePerformanceFactorsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"cpu": "cpu"},
    )
    class BaselinePerformanceFactorsRequestProperty:
        def __init__(
            self,
            *,
            cpu: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.CpuPerformanceFactorRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The baseline performance to consider, using an instance family as a baseline reference.

            The instance family establishes the lowest acceptable level of performance. Amazon EC2 uses this baseline to guide instance type selection, but there is no guarantee that the selected instance types will always exceed the baseline for every application.

            Currently, this parameter only supports CPU performance as a baseline performance factor. For example, specifying ``c6i`` would use the CPU performance of the ``c6i`` family as the baseline reference.

            :param cpu: The CPU performance to consider, using an instance family as the baseline reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-baselineperformancefactorsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                baseline_performance_factors_request_property = ec2.CfnEC2Fleet.BaselinePerformanceFactorsRequestProperty(
                    cpu=ec2.CfnEC2Fleet.CpuPerformanceFactorRequestProperty(
                        references=[ec2.CfnEC2Fleet.PerformanceFactorReferenceRequestProperty(
                            instance_family="instanceFamily"
                        )]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__af90912c9ac24abaec82396197e6a69e4a9b7287343456143c11dce7650e71c5)
                check_type(argname="argument cpu", value=cpu, expected_type=type_hints["cpu"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cpu is not None:
                self._values["cpu"] = cpu

        @builtins.property
        def cpu(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.CpuPerformanceFactorRequestProperty"]]:
            '''The CPU performance to consider, using an instance family as the baseline reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-baselineperformancefactorsrequest.html#cfn-ec2-ec2fleet-baselineperformancefactorsrequest-cpu
            '''
            result = self._values.get("cpu")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.CpuPerformanceFactorRequestProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BaselinePerformanceFactorsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.CapacityRebalanceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "replacement_strategy": "replacementStrategy",
            "termination_delay": "terminationDelay",
        },
    )
    class CapacityRebalanceProperty:
        def __init__(
            self,
            *,
            replacement_strategy: typing.Optional[builtins.str] = None,
            termination_delay: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The Spot Instance replacement strategy to use when Amazon EC2 emits a rebalance notification signal that your Spot Instance is at an elevated risk of being interrupted.

            For more information, see `Capacity rebalancing <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-capacity-rebalance.html>`_ in the *Amazon EC2 User Guide* .

            :param replacement_strategy: The replacement strategy to use. Only available for fleets of type ``maintain`` . ``launch`` - EC2 Fleet launches a replacement Spot Instance when a rebalance notification is emitted for an existing Spot Instance in the fleet. EC2 Fleet does not terminate the instances that receive a rebalance notification. You can terminate the old instances, or you can leave them running. You are charged for all instances while they are running. ``launch-before-terminate`` - EC2 Fleet launches a replacement Spot Instance when a rebalance notification is emitted for an existing Spot Instance in the fleet, and then, after a delay that you specify (in ``TerminationDelay`` ), terminates the instances that received a rebalance notification.
            :param termination_delay: The amount of time (in seconds) that Amazon EC2 waits before terminating the old Spot Instance after launching a new replacement Spot Instance. Required when ``ReplacementStrategy`` is set to ``launch-before-terminate`` . Not valid when ``ReplacementStrategy`` is set to ``launch`` . Valid values: Minimum value of ``120`` seconds. Maximum value of ``7200`` seconds.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityrebalance.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                capacity_rebalance_property = ec2.CfnEC2Fleet.CapacityRebalanceProperty(
                    replacement_strategy="replacementStrategy",
                    termination_delay=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1c9644ecfacce71230069287c64c15045d2faa220f98ae11e6e91993204231d1)
                check_type(argname="argument replacement_strategy", value=replacement_strategy, expected_type=type_hints["replacement_strategy"])
                check_type(argname="argument termination_delay", value=termination_delay, expected_type=type_hints["termination_delay"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if replacement_strategy is not None:
                self._values["replacement_strategy"] = replacement_strategy
            if termination_delay is not None:
                self._values["termination_delay"] = termination_delay

        @builtins.property
        def replacement_strategy(self) -> typing.Optional[builtins.str]:
            '''The replacement strategy to use. Only available for fleets of type ``maintain`` .

            ``launch`` - EC2 Fleet launches a replacement Spot Instance when a rebalance notification is emitted for an existing Spot Instance in the fleet. EC2 Fleet does not terminate the instances that receive a rebalance notification. You can terminate the old instances, or you can leave them running. You are charged for all instances while they are running.

            ``launch-before-terminate`` - EC2 Fleet launches a replacement Spot Instance when a rebalance notification is emitted for an existing Spot Instance in the fleet, and then, after a delay that you specify (in ``TerminationDelay`` ), terminates the instances that received a rebalance notification.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityrebalance.html#cfn-ec2-ec2fleet-capacityrebalance-replacementstrategy
            '''
            result = self._values.get("replacement_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def termination_delay(self) -> typing.Optional[jsii.Number]:
            '''The amount of time (in seconds) that Amazon EC2 waits before terminating the old Spot Instance after launching a new replacement Spot Instance.

            Required when ``ReplacementStrategy`` is set to ``launch-before-terminate`` .

            Not valid when ``ReplacementStrategy`` is set to ``launch`` .

            Valid values: Minimum value of ``120`` seconds. Maximum value of ``7200`` seconds.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityrebalance.html#cfn-ec2-ec2fleet-capacityrebalance-terminationdelay
            '''
            result = self._values.get("termination_delay")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityRebalanceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.CapacityReservationOptionsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"usage_strategy": "usageStrategy"},
    )
    class CapacityReservationOptionsRequestProperty:
        def __init__(
            self,
            *,
            usage_strategy: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Describes the strategy for using unused Capacity Reservations for fulfilling On-Demand capacity.

            .. epigraph::

               This strategy can only be used if the EC2 Fleet is of type ``instant`` .

            For more information about Capacity Reservations, see `On-Demand Capacity Reservations <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html>`_ in the *Amazon EC2 User Guide* . For examples of using Capacity Reservations in an EC2 Fleet, see `EC2 Fleet example configurations <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-examples.html>`_ in the *Amazon EC2 User Guide* .

            :param usage_strategy: Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity. If you specify ``use-capacity-reservations-first`` , the fleet uses unused Capacity Reservations to fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance pools have unused Capacity Reservations, the On-Demand allocation strategy ( ``lowest-price`` or ``prioritized`` ) is applied. If the number of unused Capacity Reservations is less than the On-Demand target capacity, the remaining On-Demand target capacity is launched according to the On-Demand allocation strategy ( ``lowest-price`` or ``prioritized`` ). If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen On-Demand allocation strategy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityreservationoptionsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                capacity_reservation_options_request_property = ec2.CfnEC2Fleet.CapacityReservationOptionsRequestProperty(
                    usage_strategy="usageStrategy"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3e503851d3968a25a6118b10e016d02c1a13b189fe5c588615ad2269d37a9b9b)
                check_type(argname="argument usage_strategy", value=usage_strategy, expected_type=type_hints["usage_strategy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if usage_strategy is not None:
                self._values["usage_strategy"] = usage_strategy

        @builtins.property
        def usage_strategy(self) -> typing.Optional[builtins.str]:
            '''Indicates whether to use unused Capacity Reservations for fulfilling On-Demand capacity.

            If you specify ``use-capacity-reservations-first`` , the fleet uses unused Capacity Reservations to fulfill On-Demand capacity up to the target On-Demand capacity. If multiple instance pools have unused Capacity Reservations, the On-Demand allocation strategy ( ``lowest-price`` or ``prioritized`` ) is applied. If the number of unused Capacity Reservations is less than the On-Demand target capacity, the remaining On-Demand target capacity is launched according to the On-Demand allocation strategy ( ``lowest-price`` or ``prioritized`` ).

            If you do not specify a value, the fleet fulfils the On-Demand capacity according to the chosen On-Demand allocation strategy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityreservationoptionsrequest.html#cfn-ec2-ec2fleet-capacityreservationoptionsrequest-usagestrategy
            '''
            result = self._values.get("usage_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityReservationOptionsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.CpuPerformanceFactorRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"references": "references"},
    )
    class CpuPerformanceFactorRequestProperty:
        def __init__(
            self,
            *,
            references: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.PerformanceFactorReferenceRequestProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The CPU performance to consider, using an instance family as the baseline reference.

            :param references: Specify an instance family to use as the baseline reference for CPU performance. All instance types that match your specified attributes will be compared against the CPU performance of the referenced instance family, regardless of CPU manufacturer or architecture differences. .. epigraph:: Currently, only one instance family can be specified in the list.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-cpuperformancefactorrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                cpu_performance_factor_request_property = ec2.CfnEC2Fleet.CpuPerformanceFactorRequestProperty(
                    references=[ec2.CfnEC2Fleet.PerformanceFactorReferenceRequestProperty(
                        instance_family="instanceFamily"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2e29954f7c2c9dfcf753ab53031041abbaad5b5371796092fa6a7e5ca77f22a6)
                check_type(argname="argument references", value=references, expected_type=type_hints["references"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if references is not None:
                self._values["references"] = references

        @builtins.property
        def references(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.PerformanceFactorReferenceRequestProperty"]]]]:
            '''Specify an instance family to use as the baseline reference for CPU performance.

            All instance types that match your specified attributes will be compared against the CPU performance of the referenced instance family, regardless of CPU manufacturer or architecture differences.
            .. epigraph::

               Currently, only one instance family can be specified in the list.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-cpuperformancefactorrequest.html#cfn-ec2-ec2fleet-cpuperformancefactorrequest-references
            '''
            result = self._values.get("references")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.PerformanceFactorReferenceRequestProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CpuPerformanceFactorRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "launch_template_specification": "launchTemplateSpecification",
            "overrides": "overrides",
        },
    )
    class FleetLaunchTemplateConfigRequestProperty:
        def __init__(
            self,
            *,
            launch_template_specification: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            overrides: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''Specifies a launch template and overrides for an EC2 Fleet.

            ``FleetLaunchTemplateConfigRequest`` is a property of the `AWS::EC2::EC2Fleet <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html>`_ resource.

            :param launch_template_specification: The launch template to use. You must specify either the launch template ID or launch template name in the request.
            :param overrides: Any parameters that you specify override the same parameters in the launch template. For fleets of type ``request`` and ``maintain`` , a maximum of 300 items is allowed across all launch templates.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                fleet_launch_template_config_request_property = ec2.CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty(
                    launch_template_specification=ec2.CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty(
                        version="version",
                
                        # the properties below are optional
                        launch_template_id="launchTemplateId",
                        launch_template_name="launchTemplateName"
                    ),
                    overrides=[ec2.CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty(
                        availability_zone="availabilityZone",
                        instance_requirements=ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty(
                            accelerator_count=ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_manufacturers=["acceleratorManufacturers"],
                            accelerator_names=["acceleratorNames"],
                            accelerator_total_memory_mi_b=ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_types=["acceleratorTypes"],
                            allowed_instance_types=["allowedInstanceTypes"],
                            bare_metal="bareMetal",
                            baseline_ebs_bandwidth_mbps=ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                                max=123,
                                min=123
                            ),
                            baseline_performance_factors=ec2.CfnEC2Fleet.BaselinePerformanceFactorsRequestProperty(
                                cpu=ec2.CfnEC2Fleet.CpuPerformanceFactorRequestProperty(
                                    references=[ec2.CfnEC2Fleet.PerformanceFactorReferenceRequestProperty(
                                        instance_family="instanceFamily"
                                    )]
                                )
                            ),
                            burstable_performance="burstablePerformance",
                            cpu_manufacturers=["cpuManufacturers"],
                            excluded_instance_types=["excludedInstanceTypes"],
                            instance_generations=["instanceGenerations"],
                            local_storage="localStorage",
                            local_storage_types=["localStorageTypes"],
                            max_spot_price_as_percentage_of_optimal_on_demand_price=123,
                            memory_gi_bPer_vCpu=ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                                max=123,
                                min=123
                            ),
                            memory_mi_b=ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            network_bandwidth_gbps=ec2.CfnEC2Fleet.NetworkBandwidthGbpsRequestProperty(
                                max=123,
                                min=123
                            ),
                            network_interface_count=ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            on_demand_max_price_percentage_over_lowest_price=123,
                            require_hibernate_support=False,
                            spot_max_price_percentage_over_lowest_price=123,
                            total_local_storage_gb=ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                                max=123,
                                min=123
                            ),
                            v_cpu_count=ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                                max=123,
                                min=123
                            )
                        ),
                        instance_type="instanceType",
                        max_price="maxPrice",
                        placement=ec2.CfnEC2Fleet.PlacementProperty(
                            affinity="affinity",
                            availability_zone="availabilityZone",
                            group_name="groupName",
                            host_id="hostId",
                            host_resource_group_arn="hostResourceGroupArn",
                            partition_number=123,
                            spread_domain="spreadDomain",
                            tenancy="tenancy"
                        ),
                        priority=123,
                        subnet_id="subnetId",
                        weighted_capacity=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a16391ac858f174f4e714e8b72b94e7e61e1b50ead8e638dff708e3d739752d5)
                check_type(argname="argument launch_template_specification", value=launch_template_specification, expected_type=type_hints["launch_template_specification"])
                check_type(argname="argument overrides", value=overrides, expected_type=type_hints["overrides"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if launch_template_specification is not None:
                self._values["launch_template_specification"] = launch_template_specification
            if overrides is not None:
                self._values["overrides"] = overrides

        @builtins.property
        def launch_template_specification(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty"]]:
            '''The launch template to use.

            You must specify either the launch template ID or launch template name in the request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateconfigrequest-launchtemplatespecification
            '''
            result = self._values.get("launch_template_specification")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty"]], result)

        @builtins.property
        def overrides(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty"]]]]:
            '''Any parameters that you specify override the same parameters in the launch template.

            For fleets of type ``request`` and ``maintain`` , a maximum of 300 items is allowed across all launch templates.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateconfigrequest-overrides
            '''
            result = self._values.get("overrides")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FleetLaunchTemplateConfigRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "availability_zone": "availabilityZone",
            "instance_requirements": "instanceRequirements",
            "instance_type": "instanceType",
            "max_price": "maxPrice",
            "placement": "placement",
            "priority": "priority",
            "subnet_id": "subnetId",
            "weighted_capacity": "weightedCapacity",
        },
    )
    class FleetLaunchTemplateOverridesRequestProperty:
        def __init__(
            self,
            *,
            availability_zone: typing.Optional[builtins.str] = None,
            instance_requirements: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.InstanceRequirementsRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            instance_type: typing.Optional[builtins.str] = None,
            max_price: typing.Optional[builtins.str] = None,
            placement: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.PlacementProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            priority: typing.Optional[jsii.Number] = None,
            subnet_id: typing.Optional[builtins.str] = None,
            weighted_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Specifies overrides for a launch template for an EC2 Fleet.

            ``FleetLaunchTemplateOverridesRequest`` is a property of the `FleetLaunchTemplateConfigRequest <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html>`_ property type.

            :param availability_zone: The Availability Zone in which to launch the instances.
            :param instance_requirements: The attributes for the instance types. When you specify instance attributes, Amazon EC2 will identify instance types with those attributes. .. epigraph:: If you specify ``InstanceRequirements`` , you can't specify ``InstanceType`` .
            :param instance_type: The instance type. ``mac1.metal`` is not supported as a launch template override. .. epigraph:: If you specify ``InstanceType`` , you can't specify ``InstanceRequirements`` .
            :param max_price: The maximum price per unit hour that you are willing to pay for a Spot Instance. We do not recommend using this parameter because it can lead to increased interruptions. If you do not specify this parameter, you will pay the current Spot price. .. epigraph:: If you specify a maximum price, your instances will be interrupted more frequently than if you do not specify this parameter. If you specify a maximum price, it must be more than USD $0.001. Specifying a value below USD $0.001 will result in an ``InvalidParameterValue`` error message.
            :param placement: The location where the instance launched, if applicable.
            :param priority: The priority for the launch template override. The highest priority is launched first. If the On-Demand ``AllocationStrategy`` is set to ``prioritized`` , EC2 Fleet uses priority to determine which launch template override to use first in fulfilling On-Demand capacity. If the Spot ``AllocationStrategy`` is set to ``capacity-optimized-prioritized`` , EC2 Fleet uses priority on a best-effort basis to determine which launch template override to use in fulfilling Spot capacity, but optimizes for capacity first. Valid values are whole numbers starting at ``0`` . The lower the number, the higher the priority. If no number is set, the launch template override has the lowest priority. You can set the same priority for different launch template overrides.
            :param subnet_id: The IDs of the subnets in which to launch the instances. Separate multiple subnet IDs using commas (for example, ``subnet-1234abcdeexample1, subnet-0987cdef6example2`` ). A request of type ``instant`` can have only one subnet ID.
            :param weighted_capacity: The number of units provided by the specified instance type. These are the same units that you chose to set the target capacity in terms of instances, or a performance characteristic such as vCPUs, memory, or I/O. If the target capacity divided by this value is not a whole number, Amazon EC2 rounds the number of instances to the next whole number. If this value is not specified, the default is 1. .. epigraph:: When specifying weights, the price used in the ``lowest-price`` and ``price-capacity-optimized`` allocation strategies is per *unit* hour (where the instance price is divided by the specified weight). However, if all the specified weights are above the requested ``TargetCapacity`` , resulting in only 1 instance being launched, the price used is per *instance* hour.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                fleet_launch_template_overrides_request_property = ec2.CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty(
                    availability_zone="availabilityZone",
                    instance_requirements=ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty(
                        accelerator_count=ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                            max=123,
                            min=123
                        ),
                        accelerator_manufacturers=["acceleratorManufacturers"],
                        accelerator_names=["acceleratorNames"],
                        accelerator_total_memory_mi_b=ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                            max=123,
                            min=123
                        ),
                        accelerator_types=["acceleratorTypes"],
                        allowed_instance_types=["allowedInstanceTypes"],
                        bare_metal="bareMetal",
                        baseline_ebs_bandwidth_mbps=ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                            max=123,
                            min=123
                        ),
                        baseline_performance_factors=ec2.CfnEC2Fleet.BaselinePerformanceFactorsRequestProperty(
                            cpu=ec2.CfnEC2Fleet.CpuPerformanceFactorRequestProperty(
                                references=[ec2.CfnEC2Fleet.PerformanceFactorReferenceRequestProperty(
                                    instance_family="instanceFamily"
                                )]
                            )
                        ),
                        burstable_performance="burstablePerformance",
                        cpu_manufacturers=["cpuManufacturers"],
                        excluded_instance_types=["excludedInstanceTypes"],
                        instance_generations=["instanceGenerations"],
                        local_storage="localStorage",
                        local_storage_types=["localStorageTypes"],
                        max_spot_price_as_percentage_of_optimal_on_demand_price=123,
                        memory_gi_bPer_vCpu=ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                            max=123,
                            min=123
                        ),
                        memory_mi_b=ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                            max=123,
                            min=123
                        ),
                        network_bandwidth_gbps=ec2.CfnEC2Fleet.NetworkBandwidthGbpsRequestProperty(
                            max=123,
                            min=123
                        ),
                        network_interface_count=ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                            max=123,
                            min=123
                        ),
                        on_demand_max_price_percentage_over_lowest_price=123,
                        require_hibernate_support=False,
                        spot_max_price_percentage_over_lowest_price=123,
                        total_local_storage_gb=ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                            max=123,
                            min=123
                        ),
                        v_cpu_count=ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                            max=123,
                            min=123
                        )
                    ),
                    instance_type="instanceType",
                    max_price="maxPrice",
                    placement=ec2.CfnEC2Fleet.PlacementProperty(
                        affinity="affinity",
                        availability_zone="availabilityZone",
                        group_name="groupName",
                        host_id="hostId",
                        host_resource_group_arn="hostResourceGroupArn",
                        partition_number=123,
                        spread_domain="spreadDomain",
                        tenancy="tenancy"
                    ),
                    priority=123,
                    subnet_id="subnetId",
                    weighted_capacity=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d758750d188cce1b749f19591d20310ad5bc697f0e6f94f495c9619297998763)
                check_type(argname="argument availability_zone", value=availability_zone, expected_type=type_hints["availability_zone"])
                check_type(argname="argument instance_requirements", value=instance_requirements, expected_type=type_hints["instance_requirements"])
                check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
                check_type(argname="argument max_price", value=max_price, expected_type=type_hints["max_price"])
                check_type(argname="argument placement", value=placement, expected_type=type_hints["placement"])
                check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
                check_type(argname="argument subnet_id", value=subnet_id, expected_type=type_hints["subnet_id"])
                check_type(argname="argument weighted_capacity", value=weighted_capacity, expected_type=type_hints["weighted_capacity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if instance_requirements is not None:
                self._values["instance_requirements"] = instance_requirements
            if instance_type is not None:
                self._values["instance_type"] = instance_type
            if max_price is not None:
                self._values["max_price"] = max_price
            if placement is not None:
                self._values["placement"] = placement
            if priority is not None:
                self._values["priority"] = priority
            if subnet_id is not None:
                self._values["subnet_id"] = subnet_id
            if weighted_capacity is not None:
                self._values["weighted_capacity"] = weighted_capacity

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''The Availability Zone in which to launch the instances.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-availabilityzone
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_requirements(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.InstanceRequirementsRequestProperty"]]:
            '''The attributes for the instance types.

            When you specify instance attributes, Amazon EC2 will identify instance types with those attributes.
            .. epigraph::

               If you specify ``InstanceRequirements`` , you can't specify ``InstanceType`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-instancerequirements
            '''
            result = self._values.get("instance_requirements")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.InstanceRequirementsRequestProperty"]], result)

        @builtins.property
        def instance_type(self) -> typing.Optional[builtins.str]:
            '''The instance type.

            ``mac1.metal`` is not supported as a launch template override.
            .. epigraph::

               If you specify ``InstanceType`` , you can't specify ``InstanceRequirements`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-instancetype
            '''
            result = self._values.get("instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def max_price(self) -> typing.Optional[builtins.str]:
            '''The maximum price per unit hour that you are willing to pay for a Spot Instance.

            We do not recommend using this parameter because it can lead to increased interruptions. If you do not specify this parameter, you will pay the current Spot price.
            .. epigraph::

               If you specify a maximum price, your instances will be interrupted more frequently than if you do not specify this parameter.

               If you specify a maximum price, it must be more than USD $0.001. Specifying a value below USD $0.001 will result in an ``InvalidParameterValue`` error message.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-maxprice
            '''
            result = self._values.get("max_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def placement(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.PlacementProperty"]]:
            '''The location where the instance launched, if applicable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-placement
            '''
            result = self._values.get("placement")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.PlacementProperty"]], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''The priority for the launch template override. The highest priority is launched first.

            If the On-Demand ``AllocationStrategy`` is set to ``prioritized`` , EC2 Fleet uses priority to determine which launch template override to use first in fulfilling On-Demand capacity.

            If the Spot ``AllocationStrategy`` is set to ``capacity-optimized-prioritized`` , EC2 Fleet uses priority on a best-effort basis to determine which launch template override to use in fulfilling Spot capacity, but optimizes for capacity first.

            Valid values are whole numbers starting at ``0`` . The lower the number, the higher the priority. If no number is set, the launch template override has the lowest priority. You can set the same priority for different launch template overrides.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def subnet_id(self) -> typing.Optional[builtins.str]:
            '''The IDs of the subnets in which to launch the instances.

            Separate multiple subnet IDs using commas (for example, ``subnet-1234abcdeexample1, subnet-0987cdef6example2`` ). A request of type ``instant`` can have only one subnet ID.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-subnetid
            '''
            result = self._values.get("subnet_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def weighted_capacity(self) -> typing.Optional[jsii.Number]:
            '''The number of units provided by the specified instance type.

            These are the same units that you chose to set the target capacity in terms of instances, or a performance characteristic such as vCPUs, memory, or I/O.

            If the target capacity divided by this value is not a whole number, Amazon EC2 rounds the number of instances to the next whole number. If this value is not specified, the default is 1.
            .. epigraph::

               When specifying weights, the price used in the ``lowest-price`` and ``price-capacity-optimized`` allocation strategies is per *unit* hour (where the instance price is divided by the specified weight). However, if all the specified weights are above the requested ``TargetCapacity`` , resulting in only 1 instance being launched, the price used is per *instance* hour.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-weightedcapacity
            '''
            result = self._values.get("weighted_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FleetLaunchTemplateOverridesRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "version": "version",
            "launch_template_id": "launchTemplateId",
            "launch_template_name": "launchTemplateName",
        },
    )
    class FleetLaunchTemplateSpecificationRequestProperty:
        def __init__(
            self,
            *,
            version: builtins.str,
            launch_template_id: typing.Optional[builtins.str] = None,
            launch_template_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies the launch template to be used by the EC2 Fleet for configuring Amazon EC2 instances.

            You must specify the following:

            - The ID or the name of the launch template, but not both.
            - The version of the launch template.

            ``FleetLaunchTemplateSpecificationRequest`` is a property of the `FleetLaunchTemplateConfigRequest <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html>`_ property type.

            For information about creating a launch template, see `AWS::EC2::LaunchTemplate <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html>`_ and `Create a launch template <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html#create-launch-template>`_ in the *Amazon EC2 User Guide* .

            For examples of launch templates, see `Examples <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html#aws-resource-ec2-launchtemplate--examples>`_ .

            :param version: The launch template version number, ``$Latest`` , or ``$Default`` . You must specify a value, otherwise the request fails. If the value is ``$Latest`` , Amazon EC2 uses the latest version of the launch template. If the value is ``$Default`` , Amazon EC2 uses the default version of the launch template.
            :param launch_template_id: The ID of the launch template. You must specify the ``LaunchTemplateId`` or the ``LaunchTemplateName`` , but not both.
            :param launch_template_name: The name of the launch template. You must specify the ``LaunchTemplateName`` or the ``LaunchTemplateId`` , but not both.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                fleet_launch_template_specification_request_property = ec2.CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty(
                    version="version",
                
                    # the properties below are optional
                    launch_template_id="launchTemplateId",
                    launch_template_name="launchTemplateName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d89df5a41143fa7395201e459ee2a2dbb7ec441738ee1a58eac3f6d95e953971)
                check_type(argname="argument version", value=version, expected_type=type_hints["version"])
                check_type(argname="argument launch_template_id", value=launch_template_id, expected_type=type_hints["launch_template_id"])
                check_type(argname="argument launch_template_name", value=launch_template_name, expected_type=type_hints["launch_template_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "version": version,
            }
            if launch_template_id is not None:
                self._values["launch_template_id"] = launch_template_id
            if launch_template_name is not None:
                self._values["launch_template_name"] = launch_template_name

        @builtins.property
        def version(self) -> builtins.str:
            '''The launch template version number, ``$Latest`` , or ``$Default`` . You must specify a value, otherwise the request fails.

            If the value is ``$Latest`` , Amazon EC2 uses the latest version of the launch template.

            If the value is ``$Default`` , Amazon EC2 uses the default version of the launch template.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest-version
            '''
            result = self._values.get("version")
            assert result is not None, "Required property 'version' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def launch_template_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the launch template.

            You must specify the ``LaunchTemplateId`` or the ``LaunchTemplateName`` , but not both.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest-launchtemplateid
            '''
            result = self._values.get("launch_template_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def launch_template_name(self) -> typing.Optional[builtins.str]:
            '''The name of the launch template.

            You must specify the ``LaunchTemplateName`` or the ``LaunchTemplateId`` , but not both.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest-launchtemplatename
            '''
            result = self._values.get("launch_template_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FleetLaunchTemplateSpecificationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "accelerator_count": "acceleratorCount",
            "accelerator_manufacturers": "acceleratorManufacturers",
            "accelerator_names": "acceleratorNames",
            "accelerator_total_memory_mib": "acceleratorTotalMemoryMiB",
            "accelerator_types": "acceleratorTypes",
            "allowed_instance_types": "allowedInstanceTypes",
            "bare_metal": "bareMetal",
            "baseline_ebs_bandwidth_mbps": "baselineEbsBandwidthMbps",
            "baseline_performance_factors": "baselinePerformanceFactors",
            "burstable_performance": "burstablePerformance",
            "cpu_manufacturers": "cpuManufacturers",
            "excluded_instance_types": "excludedInstanceTypes",
            "instance_generations": "instanceGenerations",
            "local_storage": "localStorage",
            "local_storage_types": "localStorageTypes",
            "max_spot_price_as_percentage_of_optimal_on_demand_price": "maxSpotPriceAsPercentageOfOptimalOnDemandPrice",
            "memory_gib_per_v_cpu": "memoryGiBPerVCpu",
            "memory_mib": "memoryMiB",
            "network_bandwidth_gbps": "networkBandwidthGbps",
            "network_interface_count": "networkInterfaceCount",
            "on_demand_max_price_percentage_over_lowest_price": "onDemandMaxPricePercentageOverLowestPrice",
            "require_hibernate_support": "requireHibernateSupport",
            "spot_max_price_percentage_over_lowest_price": "spotMaxPricePercentageOverLowestPrice",
            "total_local_storage_gb": "totalLocalStorageGb",
            "v_cpu_count": "vCpuCount",
        },
    )
    class InstanceRequirementsRequestProperty:
        def __init__(
            self,
            *,
            accelerator_count: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.AcceleratorCountRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            accelerator_manufacturers: typing.Optional[typing.Sequence[builtins.str]] = None,
            accelerator_names: typing.Optional[typing.Sequence[builtins.str]] = None,
            accelerator_total_memory_mib: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            accelerator_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            allowed_instance_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            bare_metal: typing.Optional[builtins.str] = None,
            baseline_ebs_bandwidth_mbps: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            baseline_performance_factors: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.BaselinePerformanceFactorsRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            burstable_performance: typing.Optional[builtins.str] = None,
            cpu_manufacturers: typing.Optional[typing.Sequence[builtins.str]] = None,
            excluded_instance_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            instance_generations: typing.Optional[typing.Sequence[builtins.str]] = None,
            local_storage: typing.Optional[builtins.str] = None,
            local_storage_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            max_spot_price_as_percentage_of_optimal_on_demand_price: typing.Optional[jsii.Number] = None,
            memory_gib_per_v_cpu: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            memory_mib: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.MemoryMiBRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            network_bandwidth_gbps: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.NetworkBandwidthGbpsRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            network_interface_count: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.NetworkInterfaceCountRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            on_demand_max_price_percentage_over_lowest_price: typing.Optional[jsii.Number] = None,
            require_hibernate_support: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            spot_max_price_percentage_over_lowest_price: typing.Optional[jsii.Number] = None,
            total_local_storage_gb: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.TotalLocalStorageGBRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            v_cpu_count: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.VCpuCountRangeRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The attributes for the instance types.

            When you specify instance attributes, Amazon EC2 will identify instance types with these attributes.

            You must specify ``VCpuCount`` and ``MemoryMiB`` . All other attributes are optional. Any unspecified optional attribute is set to its default.

            When you specify multiple attributes, you get instance types that satisfy all of the specified attributes. If you specify multiple values for an attribute, you get instance types that satisfy any of the specified values.

            To limit the list of instance types from which Amazon EC2 can identify matching instance types, you can use one of the following parameters, but not both in the same request:

            - ``AllowedInstanceTypes`` - The instance types to include in the list. All other instance types are ignored, even if they match your specified attributes.
            - ``ExcludedInstanceTypes`` - The instance types to exclude from the list, even if they match your specified attributes.

            .. epigraph::

               If you specify ``InstanceRequirements`` , you can't specify ``InstanceType`` .

               Attribute-based instance type selection is only supported when using Auto Scaling groups, EC2 Fleet, and Spot Fleet to launch instances. If you plan to use the launch template in the `launch instance wizard <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-instance-wizard.html>`_ , or with the `RunInstances <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html>`_ API or `AWS::EC2::Instance <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html>`_ AWS CloudFormation resource, you can't specify ``InstanceRequirements`` .

            For more information, see `Specify attributes for instance type selection for EC2 Fleet or Spot Fleet <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-attribute-based-instance-type-selection.html>`_ and `Spot placement score <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-placement-score.html>`_ in the *Amazon EC2 User Guide* .

            :param accelerator_count: The minimum and maximum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) on an instance. To exclude accelerator-enabled instance types, set ``Max`` to ``0`` . Default: No minimum or maximum limits
            :param accelerator_manufacturers: Indicates whether instance types must have accelerators by specific manufacturers. - For instance types with AWS devices, specify ``amazon-web-services`` . - For instance types with AMD devices, specify ``amd`` . - For instance types with Habana devices, specify ``habana`` . - For instance types with NVIDIA devices, specify ``nvidia`` . - For instance types with Xilinx devices, specify ``xilinx`` . Default: Any manufacturer
            :param accelerator_names: The accelerators that must be on the instance type. - For instance types with NVIDIA A10G GPUs, specify ``a10g`` . - For instance types with NVIDIA A100 GPUs, specify ``a100`` . - For instance types with NVIDIA H100 GPUs, specify ``h100`` . - For instance types with AWS Inferentia chips, specify ``inferentia`` . - For instance types with NVIDIA GRID K520 GPUs, specify ``k520`` . - For instance types with NVIDIA K80 GPUs, specify ``k80`` . - For instance types with NVIDIA M60 GPUs, specify ``m60`` . - For instance types with AMD Radeon Pro V520 GPUs, specify ``radeon-pro-v520`` . - For instance types with NVIDIA T4 GPUs, specify ``t4`` . - For instance types with NVIDIA T4G GPUs, specify ``t4g`` . - For instance types with Xilinx VU9P FPGAs, specify ``vu9p`` . - For instance types with NVIDIA V100 GPUs, specify ``v100`` . Default: Any accelerator
            :param accelerator_total_memory_mib: The minimum and maximum amount of total accelerator memory, in MiB. Default: No minimum or maximum limits
            :param accelerator_types: The accelerator types that must be on the instance type. - For instance types with FPGA accelerators, specify ``fpga`` . - For instance types with GPU accelerators, specify ``gpu`` . Default: Any accelerator type
            :param allowed_instance_types: The instance types to apply your specified attributes against. All other instance types are ignored, even if they match your specified attributes. You can use strings with one or more wild cards, represented by an asterisk ( ``*`` ), to allow an instance type, size, or generation. The following are examples: ``m5.8xlarge`` , ``c5*.*`` , ``m5a.*`` , ``r*`` , ``*3*`` . For example, if you specify ``c5*`` ,Amazon EC2 will allow the entire C5 instance family, which includes all C5a and C5n instance types. If you specify ``m5a.*`` , Amazon EC2 will allow all the M5a instance types, but not the M5n instance types. .. epigraph:: If you specify ``AllowedInstanceTypes`` , you can't specify ``ExcludedInstanceTypes`` . Default: All instance types
            :param bare_metal: Indicates whether bare metal instance types must be included, excluded, or required. - To include bare metal instance types, specify ``included`` . - To require only bare metal instance types, specify ``required`` . - To exclude bare metal instance types, specify ``excluded`` . Default: ``excluded``
            :param baseline_ebs_bandwidth_mbps: The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps. For more information, see `Amazon EBS–optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html>`_ in the *Amazon EC2 User Guide* . Default: No minimum or maximum limits
            :param baseline_performance_factors: The baseline performance to consider, using an instance family as a baseline reference. The instance family establishes the lowest acceptable level of performance. Amazon EC2 uses this baseline to guide instance type selection, but there is no guarantee that the selected instance types will always exceed the baseline for every application. Currently, this parameter only supports CPU performance as a baseline performance factor. For more information, see `Performance protection <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-attribute-based-instance-type-selection.html#ec2fleet-abis-performance-protection>`_ in the *Amazon EC2 User Guide* .
            :param burstable_performance: Indicates whether burstable performance T instance types are included, excluded, or required. For more information, see `Burstable performance instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html>`_ . - To include burstable performance instance types, specify ``included`` . - To require only burstable performance instance types, specify ``required`` . - To exclude burstable performance instance types, specify ``excluded`` . Default: ``excluded``
            :param cpu_manufacturers: The CPU manufacturers to include. - For instance types with Intel CPUs, specify ``intel`` . - For instance types with AMD CPUs, specify ``amd`` . - For instance types with AWS CPUs, specify ``amazon-web-services`` . - For instance types with Apple CPUs, specify ``apple`` . .. epigraph:: Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template. Default: Any manufacturer
            :param excluded_instance_types: The instance types to exclude. You can use strings with one or more wild cards, represented by an asterisk ( ``*`` ), to exclude an instance family, type, size, or generation. The following are examples: ``m5.8xlarge`` , ``c5*.*`` , ``m5a.*`` , ``r*`` , ``*3*`` . For example, if you specify ``c5*`` ,Amazon EC2 will exclude the entire C5 instance family, which includes all C5a and C5n instance types. If you specify ``m5a.*`` , Amazon EC2 will exclude all the M5a instance types, but not the M5n instance types. .. epigraph:: If you specify ``ExcludedInstanceTypes`` , you can't specify ``AllowedInstanceTypes`` . Default: No excluded instance types
            :param instance_generations: Indicates whether current or previous generation instance types are included. The current generation instance types are recommended for use. Current generation instance types are typically the latest two to three generations in each instance family. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* . For current generation instance types, specify ``current`` . For previous generation instance types, specify ``previous`` . Default: Current and previous generation instance types
            :param local_storage: Indicates whether instance types with instance store volumes are included, excluded, or required. For more information, `Amazon EC2 instance store <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html>`_ in the *Amazon EC2 User Guide* . - To include instance types with instance store volumes, specify ``included`` . - To require only instance types with instance store volumes, specify ``required`` . - To exclude instance types with instance store volumes, specify ``excluded`` . Default: ``included``
            :param local_storage_types: The type of local storage that is required. - For instance types with hard disk drive (HDD) storage, specify ``hdd`` . - For instance types with solid state drive (SSD) storage, specify ``ssd`` . Default: ``hdd`` and ``ssd``
            :param max_spot_price_as_percentage_of_optimal_on_demand_price: [Price protection] The price protection threshold for Spot Instances, as a percentage of an identified On-Demand price. The identified On-Demand price is the price of the lowest priced current generation C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance type matches your attributes, then the identified price is from the lowest priced current generation instance types, and failing that, from the lowest priced previous generation instance types that match your attributes. When Amazon EC2 selects instance types with your attributes, it will exclude instance types whose price exceeds your specified threshold. The parameter accepts an integer, which Amazon EC2 interprets as a percentage. If you set ``TargetCapacityUnitType`` to ``vcpu`` or ``memory-mib`` , the price protection threshold is based on the per vCPU or per memory price instead of the per instance price. .. epigraph:: Only one of ``SpotMaxPricePercentageOverLowestPrice`` or ``MaxSpotPriceAsPercentageOfOptimalOnDemandPrice`` can be specified. If you don't specify either, Amazon EC2 will automatically apply optimal price protection to consistently select from a wide range of instance types. To indicate no price protection threshold for Spot Instances, meaning you want to consider all instance types that match your attributes, include one of these parameters and specify a high value, such as ``999999`` .
            :param memory_gib_per_v_cpu: The minimum and maximum amount of memory per vCPU, in GiB. Default: No minimum or maximum limits
            :param memory_mib: The minimum and maximum amount of memory, in MiB.
            :param network_bandwidth_gbps: The minimum and maximum amount of baseline network bandwidth, in gigabits per second (Gbps). For more information, see `Amazon EC2 instance network bandwidth <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-network-bandwidth.html>`_ in the *Amazon EC2 User Guide* . Default: No minimum or maximum limits
            :param network_interface_count: The minimum and maximum number of network interfaces. Default: No minimum or maximum limits
            :param on_demand_max_price_percentage_over_lowest_price: [Price protection] The price protection threshold for On-Demand Instances, as a percentage higher than an identified On-Demand price. The identified On-Demand price is the price of the lowest priced current generation C, M, or R instance type with your specified attributes. When Amazon EC2 selects instance types with your attributes, it will exclude instance types whose price exceeds your specified threshold. The parameter accepts an integer, which Amazon EC2 interprets as a percentage. To indicate no price protection threshold, specify a high value, such as ``999999`` . This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ and `GetInstanceTypesFromInstanceRequirements <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html>`_ . .. epigraph:: If you set ``TargetCapacityUnitType`` to ``vcpu`` or ``memory-mib`` , the price protection threshold is applied based on the per-vCPU or per-memory price instead of the per-instance price. Default: ``20``
            :param require_hibernate_support: Indicates whether instance types must support hibernation for On-Demand Instances. This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ . Default: ``false``
            :param spot_max_price_percentage_over_lowest_price: [Price protection] The price protection threshold for Spot Instances, as a percentage higher than an identified Spot price. The identified Spot price is the Spot price of the lowest priced current generation C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance type matches your attributes, then the identified Spot price is from the lowest priced current generation instance types, and failing that, from the lowest priced previous generation instance types that match your attributes. When Amazon EC2 selects instance types with your attributes, it will exclude instance types whose Spot price exceeds your specified threshold. The parameter accepts an integer, which Amazon EC2 interprets as a percentage. If you set ``TargetCapacityUnitType`` to ``vcpu`` or ``memory-mib`` , the price protection threshold is applied based on the per-vCPU or per-memory price instead of the per-instance price. This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ and `GetInstanceTypesFromInstanceRequirements <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html>`_ . .. epigraph:: Only one of ``SpotMaxPricePercentageOverLowestPrice`` or ``MaxSpotPriceAsPercentageOfOptimalOnDemandPrice`` can be specified. If you don't specify either, Amazon EC2 will automatically apply optimal price protection to consistently select from a wide range of instance types. To indicate no price protection threshold for Spot Instances, meaning you want to consider all instance types that match your attributes, include one of these parameters and specify a high value, such as ``999999`` . Default: ``100``
            :param total_local_storage_gb: The minimum and maximum amount of total local storage, in GB. Default: No minimum or maximum limits
            :param v_cpu_count: The minimum and maximum number of vCPUs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                instance_requirements_request_property = ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty(
                    accelerator_count=ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                        max=123,
                        min=123
                    ),
                    accelerator_manufacturers=["acceleratorManufacturers"],
                    accelerator_names=["acceleratorNames"],
                    accelerator_total_memory_mi_b=ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                        max=123,
                        min=123
                    ),
                    accelerator_types=["acceleratorTypes"],
                    allowed_instance_types=["allowedInstanceTypes"],
                    bare_metal="bareMetal",
                    baseline_ebs_bandwidth_mbps=ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                        max=123,
                        min=123
                    ),
                    baseline_performance_factors=ec2.CfnEC2Fleet.BaselinePerformanceFactorsRequestProperty(
                        cpu=ec2.CfnEC2Fleet.CpuPerformanceFactorRequestProperty(
                            references=[ec2.CfnEC2Fleet.PerformanceFactorReferenceRequestProperty(
                                instance_family="instanceFamily"
                            )]
                        )
                    ),
                    burstable_performance="burstablePerformance",
                    cpu_manufacturers=["cpuManufacturers"],
                    excluded_instance_types=["excludedInstanceTypes"],
                    instance_generations=["instanceGenerations"],
                    local_storage="localStorage",
                    local_storage_types=["localStorageTypes"],
                    max_spot_price_as_percentage_of_optimal_on_demand_price=123,
                    memory_gi_bPer_vCpu=ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                        max=123,
                        min=123
                    ),
                    memory_mi_b=ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                        max=123,
                        min=123
                    ),
                    network_bandwidth_gbps=ec2.CfnEC2Fleet.NetworkBandwidthGbpsRequestProperty(
                        max=123,
                        min=123
                    ),
                    network_interface_count=ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                        max=123,
                        min=123
                    ),
                    on_demand_max_price_percentage_over_lowest_price=123,
                    require_hibernate_support=False,
                    spot_max_price_percentage_over_lowest_price=123,
                    total_local_storage_gb=ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                        max=123,
                        min=123
                    ),
                    v_cpu_count=ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                        max=123,
                        min=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a5fc35aff5e96972c3cf7c89fef75da6adc5eb146fd5504925579235a4de6e0a)
                check_type(argname="argument accelerator_count", value=accelerator_count, expected_type=type_hints["accelerator_count"])
                check_type(argname="argument accelerator_manufacturers", value=accelerator_manufacturers, expected_type=type_hints["accelerator_manufacturers"])
                check_type(argname="argument accelerator_names", value=accelerator_names, expected_type=type_hints["accelerator_names"])
                check_type(argname="argument accelerator_total_memory_mib", value=accelerator_total_memory_mib, expected_type=type_hints["accelerator_total_memory_mib"])
                check_type(argname="argument accelerator_types", value=accelerator_types, expected_type=type_hints["accelerator_types"])
                check_type(argname="argument allowed_instance_types", value=allowed_instance_types, expected_type=type_hints["allowed_instance_types"])
                check_type(argname="argument bare_metal", value=bare_metal, expected_type=type_hints["bare_metal"])
                check_type(argname="argument baseline_ebs_bandwidth_mbps", value=baseline_ebs_bandwidth_mbps, expected_type=type_hints["baseline_ebs_bandwidth_mbps"])
                check_type(argname="argument baseline_performance_factors", value=baseline_performance_factors, expected_type=type_hints["baseline_performance_factors"])
                check_type(argname="argument burstable_performance", value=burstable_performance, expected_type=type_hints["burstable_performance"])
                check_type(argname="argument cpu_manufacturers", value=cpu_manufacturers, expected_type=type_hints["cpu_manufacturers"])
                check_type(argname="argument excluded_instance_types", value=excluded_instance_types, expected_type=type_hints["excluded_instance_types"])
                check_type(argname="argument instance_generations", value=instance_generations, expected_type=type_hints["instance_generations"])
                check_type(argname="argument local_storage", value=local_storage, expected_type=type_hints["local_storage"])
                check_type(argname="argument local_storage_types", value=local_storage_types, expected_type=type_hints["local_storage_types"])
                check_type(argname="argument max_spot_price_as_percentage_of_optimal_on_demand_price", value=max_spot_price_as_percentage_of_optimal_on_demand_price, expected_type=type_hints["max_spot_price_as_percentage_of_optimal_on_demand_price"])
                check_type(argname="argument memory_gib_per_v_cpu", value=memory_gib_per_v_cpu, expected_type=type_hints["memory_gib_per_v_cpu"])
                check_type(argname="argument memory_mib", value=memory_mib, expected_type=type_hints["memory_mib"])
                check_type(argname="argument network_bandwidth_gbps", value=network_bandwidth_gbps, expected_type=type_hints["network_bandwidth_gbps"])
                check_type(argname="argument network_interface_count", value=network_interface_count, expected_type=type_hints["network_interface_count"])
                check_type(argname="argument on_demand_max_price_percentage_over_lowest_price", value=on_demand_max_price_percentage_over_lowest_price, expected_type=type_hints["on_demand_max_price_percentage_over_lowest_price"])
                check_type(argname="argument require_hibernate_support", value=require_hibernate_support, expected_type=type_hints["require_hibernate_support"])
                check_type(argname="argument spot_max_price_percentage_over_lowest_price", value=spot_max_price_percentage_over_lowest_price, expected_type=type_hints["spot_max_price_percentage_over_lowest_price"])
                check_type(argname="argument total_local_storage_gb", value=total_local_storage_gb, expected_type=type_hints["total_local_storage_gb"])
                check_type(argname="argument v_cpu_count", value=v_cpu_count, expected_type=type_hints["v_cpu_count"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if accelerator_count is not None:
                self._values["accelerator_count"] = accelerator_count
            if accelerator_manufacturers is not None:
                self._values["accelerator_manufacturers"] = accelerator_manufacturers
            if accelerator_names is not None:
                self._values["accelerator_names"] = accelerator_names
            if accelerator_total_memory_mib is not None:
                self._values["accelerator_total_memory_mib"] = accelerator_total_memory_mib
            if accelerator_types is not None:
                self._values["accelerator_types"] = accelerator_types
            if allowed_instance_types is not None:
                self._values["allowed_instance_types"] = allowed_instance_types
            if bare_metal is not None:
                self._values["bare_metal"] = bare_metal
            if baseline_ebs_bandwidth_mbps is not None:
                self._values["baseline_ebs_bandwidth_mbps"] = baseline_ebs_bandwidth_mbps
            if baseline_performance_factors is not None:
                self._values["baseline_performance_factors"] = baseline_performance_factors
            if burstable_performance is not None:
                self._values["burstable_performance"] = burstable_performance
            if cpu_manufacturers is not None:
                self._values["cpu_manufacturers"] = cpu_manufacturers
            if excluded_instance_types is not None:
                self._values["excluded_instance_types"] = excluded_instance_types
            if instance_generations is not None:
                self._values["instance_generations"] = instance_generations
            if local_storage is not None:
                self._values["local_storage"] = local_storage
            if local_storage_types is not None:
                self._values["local_storage_types"] = local_storage_types
            if max_spot_price_as_percentage_of_optimal_on_demand_price is not None:
                self._values["max_spot_price_as_percentage_of_optimal_on_demand_price"] = max_spot_price_as_percentage_of_optimal_on_demand_price
            if memory_gib_per_v_cpu is not None:
                self._values["memory_gib_per_v_cpu"] = memory_gib_per_v_cpu
            if memory_mib is not None:
                self._values["memory_mib"] = memory_mib
            if network_bandwidth_gbps is not None:
                self._values["network_bandwidth_gbps"] = network_bandwidth_gbps
            if network_interface_count is not None:
                self._values["network_interface_count"] = network_interface_count
            if on_demand_max_price_percentage_over_lowest_price is not None:
                self._values["on_demand_max_price_percentage_over_lowest_price"] = on_demand_max_price_percentage_over_lowest_price
            if require_hibernate_support is not None:
                self._values["require_hibernate_support"] = require_hibernate_support
            if spot_max_price_percentage_over_lowest_price is not None:
                self._values["spot_max_price_percentage_over_lowest_price"] = spot_max_price_percentage_over_lowest_price
            if total_local_storage_gb is not None:
                self._values["total_local_storage_gb"] = total_local_storage_gb
            if v_cpu_count is not None:
                self._values["v_cpu_count"] = v_cpu_count

        @builtins.property
        def accelerator_count(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.AcceleratorCountRequestProperty"]]:
            '''The minimum and maximum number of accelerators (GPUs, FPGAs, or AWS Inferentia chips) on an instance.

            To exclude accelerator-enabled instance types, set ``Max`` to ``0`` .

            Default: No minimum or maximum limits

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-acceleratorcount
            '''
            result = self._values.get("accelerator_count")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.AcceleratorCountRequestProperty"]], result)

        @builtins.property
        def accelerator_manufacturers(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''Indicates whether instance types must have accelerators by specific manufacturers.

            - For instance types with AWS devices, specify ``amazon-web-services`` .
            - For instance types with AMD devices, specify ``amd`` .
            - For instance types with Habana devices, specify ``habana`` .
            - For instance types with NVIDIA devices, specify ``nvidia`` .
            - For instance types with Xilinx devices, specify ``xilinx`` .

            Default: Any manufacturer

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-acceleratormanufacturers
            '''
            result = self._values.get("accelerator_manufacturers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def accelerator_names(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The accelerators that must be on the instance type.

            - For instance types with NVIDIA A10G GPUs, specify ``a10g`` .
            - For instance types with NVIDIA A100 GPUs, specify ``a100`` .
            - For instance types with NVIDIA H100 GPUs, specify ``h100`` .
            - For instance types with AWS Inferentia chips, specify ``inferentia`` .
            - For instance types with NVIDIA GRID K520 GPUs, specify ``k520`` .
            - For instance types with NVIDIA K80 GPUs, specify ``k80`` .
            - For instance types with NVIDIA M60 GPUs, specify ``m60`` .
            - For instance types with AMD Radeon Pro V520 GPUs, specify ``radeon-pro-v520`` .
            - For instance types with NVIDIA T4 GPUs, specify ``t4`` .
            - For instance types with NVIDIA T4G GPUs, specify ``t4g`` .
            - For instance types with Xilinx VU9P FPGAs, specify ``vu9p`` .
            - For instance types with NVIDIA V100 GPUs, specify ``v100`` .

            Default: Any accelerator

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-acceleratornames
            '''
            result = self._values.get("accelerator_names")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def accelerator_total_memory_mib(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty"]]:
            '''The minimum and maximum amount of total accelerator memory, in MiB.

            Default: No minimum or maximum limits

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-acceleratortotalmemorymib
            '''
            result = self._values.get("accelerator_total_memory_mib")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty"]], result)

        @builtins.property
        def accelerator_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The accelerator types that must be on the instance type.

            - For instance types with FPGA accelerators, specify ``fpga`` .
            - For instance types with GPU accelerators, specify ``gpu`` .

            Default: Any accelerator type

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-acceleratortypes
            '''
            result = self._values.get("accelerator_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def allowed_instance_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The instance types to apply your specified attributes against.

            All other instance types are ignored, even if they match your specified attributes.

            You can use strings with one or more wild cards, represented by an asterisk ( ``*`` ), to allow an instance type, size, or generation. The following are examples: ``m5.8xlarge`` , ``c5*.*`` , ``m5a.*`` , ``r*`` , ``*3*`` .

            For example, if you specify ``c5*`` ,Amazon EC2 will allow the entire C5 instance family, which includes all C5a and C5n instance types. If you specify ``m5a.*`` , Amazon EC2 will allow all the M5a instance types, but not the M5n instance types.
            .. epigraph::

               If you specify ``AllowedInstanceTypes`` , you can't specify ``ExcludedInstanceTypes`` .

            Default: All instance types

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-allowedinstancetypes
            '''
            result = self._values.get("allowed_instance_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def bare_metal(self) -> typing.Optional[builtins.str]:
            '''Indicates whether bare metal instance types must be included, excluded, or required.

            - To include bare metal instance types, specify ``included`` .
            - To require only bare metal instance types, specify ``required`` .
            - To exclude bare metal instance types, specify ``excluded`` .

            Default: ``excluded``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-baremetal
            '''
            result = self._values.get("bare_metal")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def baseline_ebs_bandwidth_mbps(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty"]]:
            '''The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps.

            For more information, see `Amazon EBS–optimized instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html>`_ in the *Amazon EC2 User Guide* .

            Default: No minimum or maximum limits

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-baselineebsbandwidthmbps
            '''
            result = self._values.get("baseline_ebs_bandwidth_mbps")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty"]], result)

        @builtins.property
        def baseline_performance_factors(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.BaselinePerformanceFactorsRequestProperty"]]:
            '''The baseline performance to consider, using an instance family as a baseline reference.

            The instance family establishes the lowest acceptable level of performance. Amazon EC2 uses this baseline to guide instance type selection, but there is no guarantee that the selected instance types will always exceed the baseline for every application. Currently, this parameter only supports CPU performance as a baseline performance factor. For more information, see `Performance protection <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-attribute-based-instance-type-selection.html#ec2fleet-abis-performance-protection>`_ in the *Amazon EC2 User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-baselineperformancefactors
            '''
            result = self._values.get("baseline_performance_factors")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.BaselinePerformanceFactorsRequestProperty"]], result)

        @builtins.property
        def burstable_performance(self) -> typing.Optional[builtins.str]:
            '''Indicates whether burstable performance T instance types are included, excluded, or required.

            For more information, see `Burstable performance instances <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html>`_ .

            - To include burstable performance instance types, specify ``included`` .
            - To require only burstable performance instance types, specify ``required`` .
            - To exclude burstable performance instance types, specify ``excluded`` .

            Default: ``excluded``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-burstableperformance
            '''
            result = self._values.get("burstable_performance")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cpu_manufacturers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The CPU manufacturers to include.

            - For instance types with Intel CPUs, specify ``intel`` .
            - For instance types with AMD CPUs, specify ``amd`` .
            - For instance types with AWS CPUs, specify ``amazon-web-services`` .
            - For instance types with Apple CPUs, specify ``apple`` .

            .. epigraph::

               Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template.

            Default: Any manufacturer

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-cpumanufacturers
            '''
            result = self._values.get("cpu_manufacturers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def excluded_instance_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The instance types to exclude.

            You can use strings with one or more wild cards, represented by an asterisk ( ``*`` ), to exclude an instance family, type, size, or generation. The following are examples: ``m5.8xlarge`` , ``c5*.*`` , ``m5a.*`` , ``r*`` , ``*3*`` .

            For example, if you specify ``c5*`` ,Amazon EC2 will exclude the entire C5 instance family, which includes all C5a and C5n instance types. If you specify ``m5a.*`` , Amazon EC2 will exclude all the M5a instance types, but not the M5n instance types.
            .. epigraph::

               If you specify ``ExcludedInstanceTypes`` , you can't specify ``AllowedInstanceTypes`` .

            Default: No excluded instance types

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-excludedinstancetypes
            '''
            result = self._values.get("excluded_instance_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def instance_generations(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Indicates whether current or previous generation instance types are included.

            The current generation instance types are recommended for use. Current generation instance types are typically the latest two to three generations in each instance family. For more information, see `Instance types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html>`_ in the *Amazon EC2 User Guide* .

            For current generation instance types, specify ``current`` .

            For previous generation instance types, specify ``previous`` .

            Default: Current and previous generation instance types

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-instancegenerations
            '''
            result = self._values.get("instance_generations")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def local_storage(self) -> typing.Optional[builtins.str]:
            '''Indicates whether instance types with instance store volumes are included, excluded, or required.

            For more information, `Amazon EC2 instance store <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html>`_ in the *Amazon EC2 User Guide* .

            - To include instance types with instance store volumes, specify ``included`` .
            - To require only instance types with instance store volumes, specify ``required`` .
            - To exclude instance types with instance store volumes, specify ``excluded`` .

            Default: ``included``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-localstorage
            '''
            result = self._values.get("local_storage")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def local_storage_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The type of local storage that is required.

            - For instance types with hard disk drive (HDD) storage, specify ``hdd`` .
            - For instance types with solid state drive (SSD) storage, specify ``ssd`` .

            Default: ``hdd`` and ``ssd``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-localstoragetypes
            '''
            result = self._values.get("local_storage_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def max_spot_price_as_percentage_of_optimal_on_demand_price(
            self,
        ) -> typing.Optional[jsii.Number]:
            '''[Price protection] The price protection threshold for Spot Instances, as a percentage of an identified On-Demand price.

            The identified On-Demand price is the price of the lowest priced current generation C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance type matches your attributes, then the identified price is from the lowest priced current generation instance types, and failing that, from the lowest priced previous generation instance types that match your attributes. When Amazon EC2 selects instance types with your attributes, it will exclude instance types whose price exceeds your specified threshold.

            The parameter accepts an integer, which Amazon EC2 interprets as a percentage.

            If you set ``TargetCapacityUnitType`` to ``vcpu`` or ``memory-mib`` , the price protection threshold is based on the per vCPU or per memory price instead of the per instance price.
            .. epigraph::

               Only one of ``SpotMaxPricePercentageOverLowestPrice`` or ``MaxSpotPriceAsPercentageOfOptimalOnDemandPrice`` can be specified. If you don't specify either, Amazon EC2 will automatically apply optimal price protection to consistently select from a wide range of instance types. To indicate no price protection threshold for Spot Instances, meaning you want to consider all instance types that match your attributes, include one of these parameters and specify a high value, such as ``999999`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-maxspotpriceaspercentageofoptimalondemandprice
            '''
            result = self._values.get("max_spot_price_as_percentage_of_optimal_on_demand_price")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def memory_gib_per_v_cpu(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty"]]:
            '''The minimum and maximum amount of memory per vCPU, in GiB.

            Default: No minimum or maximum limits

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-memorygibpervcpu
            '''
            result = self._values.get("memory_gib_per_v_cpu")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty"]], result)

        @builtins.property
        def memory_mib(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.MemoryMiBRequestProperty"]]:
            '''The minimum and maximum amount of memory, in MiB.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-memorymib
            '''
            result = self._values.get("memory_mib")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.MemoryMiBRequestProperty"]], result)

        @builtins.property
        def network_bandwidth_gbps(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.NetworkBandwidthGbpsRequestProperty"]]:
            '''The minimum and maximum amount of baseline network bandwidth, in gigabits per second (Gbps).

            For more information, see `Amazon EC2 instance network bandwidth <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-network-bandwidth.html>`_ in the *Amazon EC2 User Guide* .

            Default: No minimum or maximum limits

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-networkbandwidthgbps
            '''
            result = self._values.get("network_bandwidth_gbps")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.NetworkBandwidthGbpsRequestProperty"]], result)

        @builtins.property
        def network_interface_count(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.NetworkInterfaceCountRequestProperty"]]:
            '''The minimum and maximum number of network interfaces.

            Default: No minimum or maximum limits

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-networkinterfacecount
            '''
            result = self._values.get("network_interface_count")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.NetworkInterfaceCountRequestProperty"]], result)

        @builtins.property
        def on_demand_max_price_percentage_over_lowest_price(
            self,
        ) -> typing.Optional[jsii.Number]:
            '''[Price protection] The price protection threshold for On-Demand Instances, as a percentage higher than an identified On-Demand price.

            The identified On-Demand price is the price of the lowest priced current generation C, M, or R instance type with your specified attributes. When Amazon EC2 selects instance types with your attributes, it will exclude instance types whose price exceeds your specified threshold.

            The parameter accepts an integer, which Amazon EC2 interprets as a percentage.

            To indicate no price protection threshold, specify a high value, such as ``999999`` .

            This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ and `GetInstanceTypesFromInstanceRequirements <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html>`_ .
            .. epigraph::

               If you set ``TargetCapacityUnitType`` to ``vcpu`` or ``memory-mib`` , the price protection threshold is applied based on the per-vCPU or per-memory price instead of the per-instance price.

            Default: ``20``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-ondemandmaxpricepercentageoverlowestprice
            '''
            result = self._values.get("on_demand_max_price_percentage_over_lowest_price")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def require_hibernate_support(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates whether instance types must support hibernation for On-Demand Instances.

            This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ .

            Default: ``false``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-requirehibernatesupport
            '''
            result = self._values.get("require_hibernate_support")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def spot_max_price_percentage_over_lowest_price(
            self,
        ) -> typing.Optional[jsii.Number]:
            '''[Price protection] The price protection threshold for Spot Instances, as a percentage higher than an identified Spot price.

            The identified Spot price is the Spot price of the lowest priced current generation C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance type matches your attributes, then the identified Spot price is from the lowest priced current generation instance types, and failing that, from the lowest priced previous generation instance types that match your attributes. When Amazon EC2 selects instance types with your attributes, it will exclude instance types whose Spot price exceeds your specified threshold.

            The parameter accepts an integer, which Amazon EC2 interprets as a percentage.

            If you set ``TargetCapacityUnitType`` to ``vcpu`` or ``memory-mib`` , the price protection threshold is applied based on the per-vCPU or per-memory price instead of the per-instance price.

            This parameter is not supported for `GetSpotPlacementScores <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html>`_ and `GetInstanceTypesFromInstanceRequirements <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html>`_ .
            .. epigraph::

               Only one of ``SpotMaxPricePercentageOverLowestPrice`` or ``MaxSpotPriceAsPercentageOfOptimalOnDemandPrice`` can be specified. If you don't specify either, Amazon EC2 will automatically apply optimal price protection to consistently select from a wide range of instance types. To indicate no price protection threshold for Spot Instances, meaning you want to consider all instance types that match your attributes, include one of these parameters and specify a high value, such as ``999999`` .

            Default: ``100``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-spotmaxpricepercentageoverlowestprice
            '''
            result = self._values.get("spot_max_price_percentage_over_lowest_price")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def total_local_storage_gb(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.TotalLocalStorageGBRequestProperty"]]:
            '''The minimum and maximum amount of total local storage, in GB.

            Default: No minimum or maximum limits

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-totallocalstoragegb
            '''
            result = self._values.get("total_local_storage_gb")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.TotalLocalStorageGBRequestProperty"]], result)

        @builtins.property
        def v_cpu_count(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.VCpuCountRangeRequestProperty"]]:
            '''The minimum and maximum number of vCPUs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-instancerequirementsrequest.html#cfn-ec2-ec2fleet-instancerequirementsrequest-vcpucount
            '''
            result = self._values.get("v_cpu_count")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.VCpuCountRangeRequestProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceRequirementsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.MaintenanceStrategiesProperty",
        jsii_struct_bases=[],
        name_mapping={"capacity_rebalance": "capacityRebalance"},
    )
    class MaintenanceStrategiesProperty:
        def __init__(
            self,
            *,
            capacity_rebalance: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.CapacityRebalanceProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The strategies for managing your Spot Instances that are at an elevated risk of being interrupted.

            :param capacity_rebalance: The strategy to use when Amazon EC2 emits a signal that your Spot Instance is at an elevated risk of being interrupted.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-maintenancestrategies.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                maintenance_strategies_property = ec2.CfnEC2Fleet.MaintenanceStrategiesProperty(
                    capacity_rebalance=ec2.CfnEC2Fleet.CapacityRebalanceProperty(
                        replacement_strategy="replacementStrategy",
                        termination_delay=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__74a54f8579657c5e4deb0c052567cd4660664c3fcc6d9232bbc646b1ee0c421e)
                check_type(argname="argument capacity_rebalance", value=capacity_rebalance, expected_type=type_hints["capacity_rebalance"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if capacity_rebalance is not None:
                self._values["capacity_rebalance"] = capacity_rebalance

        @builtins.property
        def capacity_rebalance(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.CapacityRebalanceProperty"]]:
            '''The strategy to use when Amazon EC2 emits a signal that your Spot Instance is at an elevated risk of being interrupted.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-maintenancestrategies.html#cfn-ec2-ec2fleet-maintenancestrategies-capacityrebalance
            '''
            result = self._values.get("capacity_rebalance")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.CapacityRebalanceProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MaintenanceStrategiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class MemoryGiBPerVCpuRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum amount of memory per vCPU, in GiB.

            :param max: The maximum amount of memory per vCPU, in GiB. To specify no maximum limit, omit this parameter.
            :param min: The minimum amount of memory per vCPU, in GiB. To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorygibpervcpurequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                memory_gi_bPer_vCpu_request_property = ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b60bc2d12a913968d0d316eb200c86e5d9ecf567ed9b6c6e51cdbd1a0b417422)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of memory per vCPU, in GiB.

            To specify no maximum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorygibpervcpurequest.html#cfn-ec2-ec2fleet-memorygibpervcpurequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of memory per vCPU, in GiB.

            To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorygibpervcpurequest.html#cfn-ec2-ec2fleet-memorygibpervcpurequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MemoryGiBPerVCpuRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.MemoryMiBRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class MemoryMiBRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum amount of memory, in MiB.

            :param max: The maximum amount of memory, in MiB. To specify no maximum limit, omit this parameter.
            :param min: The minimum amount of memory, in MiB. To specify no minimum limit, specify ``0`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorymibrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                memory_mi_bRequest_property = ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__582bf0e6109937779c349eb63366e7e4cdaa15b4c199c2ec484777174cd8b7d4)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of memory, in MiB.

            To specify no maximum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorymibrequest.html#cfn-ec2-ec2fleet-memorymibrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of memory, in MiB.

            To specify no minimum limit, specify ``0`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-memorymibrequest.html#cfn-ec2-ec2fleet-memorymibrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MemoryMiBRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.NetworkBandwidthGbpsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class NetworkBandwidthGbpsRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum amount of network bandwidth, in gigabits per second (Gbps).

            .. epigraph::

               Setting the minimum bandwidth does not guarantee that your instance will achieve the minimum bandwidth. Amazon EC2 will identify instance types that support the specified minimum bandwidth, but the actual bandwidth of your instance might go below the specified minimum at times. For more information, see `Available instance bandwidth <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-network-bandwidth.html#available-instance-bandwidth>`_ in the *Amazon EC2 User Guide* .

            :param max: The maximum amount of network bandwidth, in Gbps. To specify no maximum limit, omit this parameter.
            :param min: The minimum amount of network bandwidth, in Gbps. To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-networkbandwidthgbpsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                network_bandwidth_gbps_request_property = ec2.CfnEC2Fleet.NetworkBandwidthGbpsRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a1631d95b133eba9c84f539d999721b9029fe1ac10dad3c41c3cf7362e6dcf23)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of network bandwidth, in Gbps.

            To specify no maximum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-networkbandwidthgbpsrequest.html#cfn-ec2-ec2fleet-networkbandwidthgbpsrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of network bandwidth, in Gbps.

            To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-networkbandwidthgbpsrequest.html#cfn-ec2-ec2fleet-networkbandwidthgbpsrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkBandwidthGbpsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class NetworkInterfaceCountRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum number of network interfaces.

            :param max: The maximum number of network interfaces. To specify no maximum limit, omit this parameter.
            :param min: The minimum number of network interfaces. To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-networkinterfacecountrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                network_interface_count_request_property = ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c1c61f48589be52f9f062634b2e52fcd4095fcad95dc9fd0bf2a68bdfe2d9829)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of network interfaces.

            To specify no maximum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-networkinterfacecountrequest.html#cfn-ec2-ec2fleet-networkinterfacecountrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of network interfaces.

            To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-networkinterfacecountrequest.html#cfn-ec2-ec2fleet-networkinterfacecountrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkInterfaceCountRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.OnDemandOptionsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "allocation_strategy": "allocationStrategy",
            "capacity_reservation_options": "capacityReservationOptions",
            "max_total_price": "maxTotalPrice",
            "min_target_capacity": "minTargetCapacity",
            "single_availability_zone": "singleAvailabilityZone",
            "single_instance_type": "singleInstanceType",
        },
    )
    class OnDemandOptionsRequestProperty:
        def __init__(
            self,
            *,
            allocation_strategy: typing.Optional[builtins.str] = None,
            capacity_reservation_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.CapacityReservationOptionsRequestProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            max_total_price: typing.Optional[builtins.str] = None,
            min_target_capacity: typing.Optional[jsii.Number] = None,
            single_availability_zone: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            single_instance_type: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''Specifies the allocation strategy of On-Demand Instances in an EC2 Fleet.

            ``OnDemandOptionsRequest`` is a property of the `AWS::EC2::EC2Fleet <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html>`_ resource.

            :param allocation_strategy: The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand capacity. ``lowest-price`` - EC2 Fleet uses price to determine the order, launching the lowest price first. ``prioritized`` - EC2 Fleet uses the priority that you assigned to each launch template override, launching the highest priority first. Default: ``lowest-price``
            :param capacity_reservation_options: The strategy for using unused Capacity Reservations for fulfilling On-Demand capacity. Supported only for fleets of type ``instant`` .
            :param max_total_price: The maximum amount per hour for On-Demand Instances that you're willing to pay. .. epigraph:: If your fleet includes T instances that are configured as ``unlimited`` , and if their average CPU usage exceeds the baseline utilization, you will incur a charge for surplus credits. The ``MaxTotalPrice`` does not account for surplus credits, and, if you use surplus credits, your final cost might be higher than what you specified for ``MaxTotalPrice`` . For more information, see `Surplus credits can incur charges <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances-unlimited-mode-concepts.html#unlimited-mode-surplus-credits>`_ in the *Amazon EC2 User Guide* .
            :param min_target_capacity: The minimum target capacity for On-Demand Instances in the fleet. If this minimum capacity isn't reached, no instances are launched. Constraints: Maximum value of ``1000`` . Supported only for fleets of type ``instant`` . At least one of the following must be specified: ``SingleAvailabilityZone`` | ``SingleInstanceType``
            :param single_availability_zone: Indicates that the fleet launches all On-Demand Instances into a single Availability Zone. Supported only for fleets of type ``instant`` .
            :param single_instance_type: Indicates that the fleet uses a single instance type to launch all On-Demand Instances in the fleet. Supported only for fleets of type ``instant`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                on_demand_options_request_property = ec2.CfnEC2Fleet.OnDemandOptionsRequestProperty(
                    allocation_strategy="allocationStrategy",
                    capacity_reservation_options=ec2.CfnEC2Fleet.CapacityReservationOptionsRequestProperty(
                        usage_strategy="usageStrategy"
                    ),
                    max_total_price="maxTotalPrice",
                    min_target_capacity=123,
                    single_availability_zone=False,
                    single_instance_type=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dce716d2e517f4e8c297cb02cd9e5e88232758cb9732040686bdfb59b38c71c3)
                check_type(argname="argument allocation_strategy", value=allocation_strategy, expected_type=type_hints["allocation_strategy"])
                check_type(argname="argument capacity_reservation_options", value=capacity_reservation_options, expected_type=type_hints["capacity_reservation_options"])
                check_type(argname="argument max_total_price", value=max_total_price, expected_type=type_hints["max_total_price"])
                check_type(argname="argument min_target_capacity", value=min_target_capacity, expected_type=type_hints["min_target_capacity"])
                check_type(argname="argument single_availability_zone", value=single_availability_zone, expected_type=type_hints["single_availability_zone"])
                check_type(argname="argument single_instance_type", value=single_instance_type, expected_type=type_hints["single_instance_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if allocation_strategy is not None:
                self._values["allocation_strategy"] = allocation_strategy
            if capacity_reservation_options is not None:
                self._values["capacity_reservation_options"] = capacity_reservation_options
            if max_total_price is not None:
                self._values["max_total_price"] = max_total_price
            if min_target_capacity is not None:
                self._values["min_target_capacity"] = min_target_capacity
            if single_availability_zone is not None:
                self._values["single_availability_zone"] = single_availability_zone
            if single_instance_type is not None:
                self._values["single_instance_type"] = single_instance_type

        @builtins.property
        def allocation_strategy(self) -> typing.Optional[builtins.str]:
            '''The strategy that determines the order of the launch template overrides to use in fulfilling On-Demand capacity.

            ``lowest-price`` - EC2 Fleet uses price to determine the order, launching the lowest price first.

            ``prioritized`` - EC2 Fleet uses the priority that you assigned to each launch template override, launching the highest priority first.

            Default: ``lowest-price``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-allocationstrategy
            '''
            result = self._values.get("allocation_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def capacity_reservation_options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.CapacityReservationOptionsRequestProperty"]]:
            '''The strategy for using unused Capacity Reservations for fulfilling On-Demand capacity.

            Supported only for fleets of type ``instant`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-capacityreservationoptions
            '''
            result = self._values.get("capacity_reservation_options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.CapacityReservationOptionsRequestProperty"]], result)

        @builtins.property
        def max_total_price(self) -> typing.Optional[builtins.str]:
            '''The maximum amount per hour for On-Demand Instances that you're willing to pay.

            .. epigraph::

               If your fleet includes T instances that are configured as ``unlimited`` , and if their average CPU usage exceeds the baseline utilization, you will incur a charge for surplus credits. The ``MaxTotalPrice`` does not account for surplus credits, and, if you use surplus credits, your final cost might be higher than what you specified for ``MaxTotalPrice`` . For more information, see `Surplus credits can incur charges <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances-unlimited-mode-concepts.html#unlimited-mode-surplus-credits>`_ in the *Amazon EC2 User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-maxtotalprice
            '''
            result = self._values.get("max_total_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def min_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''The minimum target capacity for On-Demand Instances in the fleet.

            If this minimum capacity isn't reached, no instances are launched.

            Constraints: Maximum value of ``1000`` . Supported only for fleets of type ``instant`` .

            At least one of the following must be specified: ``SingleAvailabilityZone`` | ``SingleInstanceType``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-mintargetcapacity
            '''
            result = self._values.get("min_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def single_availability_zone(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates that the fleet launches all On-Demand Instances into a single Availability Zone.

            Supported only for fleets of type ``instant`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-singleavailabilityzone
            '''
            result = self._values.get("single_availability_zone")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def single_instance_type(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates that the fleet uses a single instance type to launch all On-Demand Instances in the fleet.

            Supported only for fleets of type ``instant`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-singleinstancetype
            '''
            result = self._values.get("single_instance_type")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OnDemandOptionsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.PerformanceFactorReferenceRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"instance_family": "instanceFamily"},
    )
    class PerformanceFactorReferenceRequestProperty:
        def __init__(
            self,
            *,
            instance_family: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specify an instance family to use as the baseline reference for CPU performance.

            All instance types that match your specified attributes will be compared against the CPU performance of the referenced instance family, regardless of CPU manufacturer or architecture.
            .. epigraph::

               Currently, only one instance family can be specified in the list.

            :param instance_family: The instance family to use as a baseline reference. .. epigraph:: Ensure that you specify the correct value for the instance family. The instance family is everything before the period ( ``.`` ) in the instance type name. For example, in the instance type ``c6i.large`` , the instance family is ``c6i`` , not ``c6`` . For more information, see `Amazon EC2 instance type naming conventions <https://docs.aws.amazon.com/ec2/latest/instancetypes/instance-type-names.html>`_ in *Amazon EC2 Instance Types* . The following instance families are *not supported* for performance protection: - ``c1`` - ``g3`` | ``g3s`` - ``hpc7g`` - ``m1`` | ``m2`` - ``mac1`` | ``mac2`` | ``mac2-m1ultra`` | ``mac2-m2`` | ``mac2-m2pro`` - ``p3dn`` | ``p4d`` | ``p5`` - ``t1`` - ``u-12tb1`` | ``u-18tb1`` | ``u-24tb1`` | ``u-3tb1`` | ``u-6tb1`` | ``u-9tb1`` | ``u7i-12tb`` | ``u7in-16tb`` | ``u7in-24tb`` | ``u7in-32tb`` If you enable performance protection by specifying a supported instance family, the returned instance types will exclude the above unsupported instance families. If you specify an unsupported instance family as a value for baseline performance, the API returns an empty response response for `GetInstanceTypesFromInstanceRequirements <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html>`_ and an exception for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet.html>`_ , `RequestSpotFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html>`_ , `ModifyFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFleet.html>`_ , and `ModifySpotFleetRequest <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySpotFleetRequest.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-performancefactorreferencerequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                performance_factor_reference_request_property = ec2.CfnEC2Fleet.PerformanceFactorReferenceRequestProperty(
                    instance_family="instanceFamily"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__763de8714eb641fd97ed93223598e53004b5244dd3fb304b8b01f6c1d7375a58)
                check_type(argname="argument instance_family", value=instance_family, expected_type=type_hints["instance_family"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if instance_family is not None:
                self._values["instance_family"] = instance_family

        @builtins.property
        def instance_family(self) -> typing.Optional[builtins.str]:
            '''The instance family to use as a baseline reference.

            .. epigraph::

               Ensure that you specify the correct value for the instance family. The instance family is everything before the period ( ``.`` ) in the instance type name. For example, in the instance type ``c6i.large`` , the instance family is ``c6i`` , not ``c6`` . For more information, see `Amazon EC2 instance type naming conventions <https://docs.aws.amazon.com/ec2/latest/instancetypes/instance-type-names.html>`_ in *Amazon EC2 Instance Types* .

            The following instance families are *not supported* for performance protection:

            - ``c1``
            - ``g3`` | ``g3s``
            - ``hpc7g``
            - ``m1`` | ``m2``
            - ``mac1`` | ``mac2`` | ``mac2-m1ultra`` | ``mac2-m2`` | ``mac2-m2pro``
            - ``p3dn`` | ``p4d`` | ``p5``
            - ``t1``
            - ``u-12tb1`` | ``u-18tb1`` | ``u-24tb1`` | ``u-3tb1`` | ``u-6tb1`` | ``u-9tb1`` | ``u7i-12tb`` | ``u7in-16tb`` | ``u7in-24tb`` | ``u7in-32tb``

            If you enable performance protection by specifying a supported instance family, the returned instance types will exclude the above unsupported instance families.

            If you specify an unsupported instance family as a value for baseline performance, the API returns an empty response response for `GetInstanceTypesFromInstanceRequirements <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html>`_ and an exception for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet.html>`_ , `RequestSpotFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html>`_ , `ModifyFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFleet.html>`_ , and `ModifySpotFleetRequest <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySpotFleetRequest.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-performancefactorreferencerequest.html#cfn-ec2-ec2fleet-performancefactorreferencerequest-instancefamily
            '''
            result = self._values.get("instance_family")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PerformanceFactorReferenceRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.PlacementProperty",
        jsii_struct_bases=[],
        name_mapping={
            "affinity": "affinity",
            "availability_zone": "availabilityZone",
            "group_name": "groupName",
            "host_id": "hostId",
            "host_resource_group_arn": "hostResourceGroupArn",
            "partition_number": "partitionNumber",
            "spread_domain": "spreadDomain",
            "tenancy": "tenancy",
        },
    )
    class PlacementProperty:
        def __init__(
            self,
            *,
            affinity: typing.Optional[builtins.str] = None,
            availability_zone: typing.Optional[builtins.str] = None,
            group_name: typing.Optional[builtins.str] = None,
            host_id: typing.Optional[builtins.str] = None,
            host_resource_group_arn: typing.Optional[builtins.str] = None,
            partition_number: typing.Optional[jsii.Number] = None,
            spread_domain: typing.Optional[builtins.str] = None,
            tenancy: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Describes the placement of an instance.

            :param affinity: The affinity setting for the instance on the Dedicated Host. This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ or `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ .
            :param availability_zone: The Availability Zone of the instance. If not specified, an Availability Zone will be automatically chosen for you based on the load balancing criteria for the Region. This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .
            :param group_name: The name of the placement group that the instance is in. If you specify ``GroupName`` , you can't specify ``GroupId`` .
            :param host_id: The ID of the Dedicated Host on which the instance resides. This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ or `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ .
            :param host_resource_group_arn: The ARN of the host resource group in which to launch the instances. If you specify this parameter, either omit the *Tenancy* parameter or set it to ``host`` . This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .
            :param partition_number: The number of the partition that the instance is in. Valid only if the placement group strategy is set to ``partition`` . This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .
            :param spread_domain: Reserved for future use.
            :param tenancy: The tenancy of the instance. An instance with a tenancy of ``dedicated`` runs on single-tenant hardware. This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ . The ``host`` tenancy is not supported for `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ or for T3 instances that are configured for the ``unlimited`` CPU credit option.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                placement_property = ec2.CfnEC2Fleet.PlacementProperty(
                    affinity="affinity",
                    availability_zone="availabilityZone",
                    group_name="groupName",
                    host_id="hostId",
                    host_resource_group_arn="hostResourceGroupArn",
                    partition_number=123,
                    spread_domain="spreadDomain",
                    tenancy="tenancy"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__28c598518bf40690e18af7409981451456cd4d41d853891f648b55488b7113f5)
                check_type(argname="argument affinity", value=affinity, expected_type=type_hints["affinity"])
                check_type(argname="argument availability_zone", value=availability_zone, expected_type=type_hints["availability_zone"])
                check_type(argname="argument group_name", value=group_name, expected_type=type_hints["group_name"])
                check_type(argname="argument host_id", value=host_id, expected_type=type_hints["host_id"])
                check_type(argname="argument host_resource_group_arn", value=host_resource_group_arn, expected_type=type_hints["host_resource_group_arn"])
                check_type(argname="argument partition_number", value=partition_number, expected_type=type_hints["partition_number"])
                check_type(argname="argument spread_domain", value=spread_domain, expected_type=type_hints["spread_domain"])
                check_type(argname="argument tenancy", value=tenancy, expected_type=type_hints["tenancy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if affinity is not None:
                self._values["affinity"] = affinity
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if group_name is not None:
                self._values["group_name"] = group_name
            if host_id is not None:
                self._values["host_id"] = host_id
            if host_resource_group_arn is not None:
                self._values["host_resource_group_arn"] = host_resource_group_arn
            if partition_number is not None:
                self._values["partition_number"] = partition_number
            if spread_domain is not None:
                self._values["spread_domain"] = spread_domain
            if tenancy is not None:
                self._values["tenancy"] = tenancy

        @builtins.property
        def affinity(self) -> typing.Optional[builtins.str]:
            '''The affinity setting for the instance on the Dedicated Host.

            This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ or `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-affinity
            '''
            result = self._values.get("affinity")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''The Availability Zone of the instance.

            If not specified, an Availability Zone will be automatically chosen for you based on the load balancing criteria for the Region.

            This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-availabilityzone
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def group_name(self) -> typing.Optional[builtins.str]:
            '''The name of the placement group that the instance is in.

            If you specify ``GroupName`` , you can't specify ``GroupId`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-groupname
            '''
            result = self._values.get("group_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the Dedicated Host on which the instance resides.

            This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ or `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-hostid
            '''
            result = self._values.get("host_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_resource_group_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of the host resource group in which to launch the instances.

            If you specify this parameter, either omit the *Tenancy* parameter or set it to ``host`` .

            This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-hostresourcegrouparn
            '''
            result = self._values.get("host_resource_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def partition_number(self) -> typing.Optional[jsii.Number]:
            '''The number of the partition that the instance is in.

            Valid only if the placement group strategy is set to ``partition`` .

            This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-partitionnumber
            '''
            result = self._values.get("partition_number")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def spread_domain(self) -> typing.Optional[builtins.str]:
            '''Reserved for future use.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-spreaddomain
            '''
            result = self._values.get("spread_domain")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tenancy(self) -> typing.Optional[builtins.str]:
            '''The tenancy of the instance. An instance with a tenancy of ``dedicated`` runs on single-tenant hardware.

            This parameter is not supported for `CreateFleet <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet>`_ . The ``host`` tenancy is not supported for `ImportInstance <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html>`_ or for T3 instances that are configured for the ``unlimited`` CPU credit option.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-tenancy
            '''
            result = self._values.get("tenancy")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.SpotOptionsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "allocation_strategy": "allocationStrategy",
            "instance_interruption_behavior": "instanceInterruptionBehavior",
            "instance_pools_to_use_count": "instancePoolsToUseCount",
            "maintenance_strategies": "maintenanceStrategies",
            "max_total_price": "maxTotalPrice",
            "min_target_capacity": "minTargetCapacity",
            "single_availability_zone": "singleAvailabilityZone",
            "single_instance_type": "singleInstanceType",
        },
    )
    class SpotOptionsRequestProperty:
        def __init__(
            self,
            *,
            allocation_strategy: typing.Optional[builtins.str] = None,
            instance_interruption_behavior: typing.Optional[builtins.str] = None,
            instance_pools_to_use_count: typing.Optional[jsii.Number] = None,
            maintenance_strategies: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnEC2Fleet.MaintenanceStrategiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            max_total_price: typing.Optional[builtins.str] = None,
            min_target_capacity: typing.Optional[jsii.Number] = None,
            single_availability_zone: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            single_instance_type: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''Specifies the configuration of Spot Instances for an EC2 Fleet.

            ``SpotOptionsRequest`` is a property of the `AWS::EC2::EC2Fleet <https://docs.aws.amazon.com//AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html>`_ resource.

            :param allocation_strategy: Indicates how to allocate the target Spot Instance capacity across the Spot Instance pools specified by the EC2 Fleet. If the allocation strategy is ``lowestPrice`` , EC2 Fleet launches instances from the Spot Instance pools with the lowest price. This is the default allocation strategy. If the allocation strategy is ``diversified`` , EC2 Fleet launches instances from all the Spot Instance pools that you specify. If the allocation strategy is ``capacityOptimized`` , EC2 Fleet launches instances from Spot Instance pools that are optimally chosen based on the available Spot Instance capacity. *Allowed Values* : ``lowestPrice`` | ``diversified`` | ``capacityOptimized`` | ``capacityOptimizedPrioritized``
            :param instance_interruption_behavior: The behavior when a Spot Instance is interrupted. Default: ``terminate``
            :param instance_pools_to_use_count: The number of Spot pools across which to allocate your target Spot capacity. Supported only when Spot ``AllocationStrategy`` is set to ``lowest-price`` . EC2 Fleet selects the cheapest Spot pools and evenly allocates your target Spot capacity across the number of Spot pools that you specify. Note that EC2 Fleet attempts to draw Spot Instances from the number of pools that you specify on a best effort basis. If a pool runs out of Spot capacity before fulfilling your target capacity, EC2 Fleet will continue to fulfill your request by drawing from the next cheapest pool. To ensure that your target capacity is met, you might receive Spot Instances from more than the number of pools that you specified. Similarly, if most of the pools have no Spot capacity, you might receive your full target capacity from fewer than the number of pools that you specified.
            :param maintenance_strategies: The strategies for managing your Spot Instances that are at an elevated risk of being interrupted.
            :param max_total_price: The maximum amount per hour for Spot Instances that you're willing to pay. We do not recommend using this parameter because it can lead to increased interruptions. If you do not specify this parameter, you will pay the current Spot price. .. epigraph:: If you specify a maximum price, your Spot Instances will be interrupted more frequently than if you do not specify this parameter. > If your fleet includes T instances that are configured as ``unlimited`` , and if their average CPU usage exceeds the baseline utilization, you will incur a charge for surplus credits. The ``MaxTotalPrice`` does not account for surplus credits, and, if you use surplus credits, your final cost might be higher than what you specified for ``MaxTotalPrice`` . For more information, see `Surplus credits can incur charges <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances-unlimited-mode-concepts.html#unlimited-mode-surplus-credits>`_ in the *Amazon EC2 User Guide* .
            :param min_target_capacity: The minimum target capacity for Spot Instances in the fleet. If this minimum capacity isn't reached, no instances are launched. Constraints: Maximum value of ``1000`` . Supported only for fleets of type ``instant`` . At least one of the following must be specified: ``SingleAvailabilityZone`` | ``SingleInstanceType``
            :param single_availability_zone: Indicates that the fleet launches all Spot Instances into a single Availability Zone. Supported only for fleets of type ``instant`` .
            :param single_instance_type: Indicates that the fleet uses a single instance type to launch all Spot Instances in the fleet. Supported only for fleets of type ``instant`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                spot_options_request_property = ec2.CfnEC2Fleet.SpotOptionsRequestProperty(
                    allocation_strategy="allocationStrategy",
                    instance_interruption_behavior="instanceInterruptionBehavior",
                    instance_pools_to_use_count=123,
                    maintenance_strategies=ec2.CfnEC2Fleet.MaintenanceStrategiesProperty(
                        capacity_rebalance=ec2.CfnEC2Fleet.CapacityRebalanceProperty(
                            replacement_strategy="replacementStrategy",
                            termination_delay=123
                        )
                    ),
                    max_total_price="maxTotalPrice",
                    min_target_capacity=123,
                    single_availability_zone=False,
                    single_instance_type=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c9673542a949ab3e4b488aaa9dfb67cceabe990e15b2281ccc01f9698538f168)
                check_type(argname="argument allocation_strategy", value=allocation_strategy, expected_type=type_hints["allocation_strategy"])
                check_type(argname="argument instance_interruption_behavior", value=instance_interruption_behavior, expected_type=type_hints["instance_interruption_behavior"])
                check_type(argname="argument instance_pools_to_use_count", value=instance_pools_to_use_count, expected_type=type_hints["instance_pools_to_use_count"])
                check_type(argname="argument maintenance_strategies", value=maintenance_strategies, expected_type=type_hints["maintenance_strategies"])
                check_type(argname="argument max_total_price", value=max_total_price, expected_type=type_hints["max_total_price"])
                check_type(argname="argument min_target_capacity", value=min_target_capacity, expected_type=type_hints["min_target_capacity"])
                check_type(argname="argument single_availability_zone", value=single_availability_zone, expected_type=type_hints["single_availability_zone"])
                check_type(argname="argument single_instance_type", value=single_instance_type, expected_type=type_hints["single_instance_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if allocation_strategy is not None:
                self._values["allocation_strategy"] = allocation_strategy
            if instance_interruption_behavior is not None:
                self._values["instance_interruption_behavior"] = instance_interruption_behavior
            if instance_pools_to_use_count is not None:
                self._values["instance_pools_to_use_count"] = instance_pools_to_use_count
            if maintenance_strategies is not None:
                self._values["maintenance_strategies"] = maintenance_strategies
            if max_total_price is not None:
                self._values["max_total_price"] = max_total_price
            if min_target_capacity is not None:
                self._values["min_target_capacity"] = min_target_capacity
            if single_availability_zone is not None:
                self._values["single_availability_zone"] = single_availability_zone
            if single_instance_type is not None:
                self._values["single_instance_type"] = single_instance_type

        @builtins.property
        def allocation_strategy(self) -> typing.Optional[builtins.str]:
            '''Indicates how to allocate the target Spot Instance capacity across the Spot Instance pools specified by the EC2 Fleet.

            If the allocation strategy is ``lowestPrice`` , EC2 Fleet launches instances from the Spot Instance pools with the lowest price. This is the default allocation strategy.

            If the allocation strategy is ``diversified`` , EC2 Fleet launches instances from all the Spot Instance pools that you specify.

            If the allocation strategy is ``capacityOptimized`` , EC2 Fleet launches instances from Spot Instance pools that are optimally chosen based on the available Spot Instance capacity.

            *Allowed Values* : ``lowestPrice`` | ``diversified`` | ``capacityOptimized`` | ``capacityOptimizedPrioritized``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-allocationstrategy
            '''
            result = self._values.get("allocation_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_interruption_behavior(self) -> typing.Optional[builtins.str]:
            '''The behavior when a Spot Instance is interrupted.

            Default: ``terminate``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-instanceinterruptionbehavior
            '''
            result = self._values.get("instance_interruption_behavior")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_pools_to_use_count(self) -> typing.Optional[jsii.Number]:
            '''The number of Spot pools across which to allocate your target Spot capacity.

            Supported only when Spot ``AllocationStrategy`` is set to ``lowest-price`` . EC2 Fleet selects the cheapest Spot pools and evenly allocates your target Spot capacity across the number of Spot pools that you specify.

            Note that EC2 Fleet attempts to draw Spot Instances from the number of pools that you specify on a best effort basis. If a pool runs out of Spot capacity before fulfilling your target capacity, EC2 Fleet will continue to fulfill your request by drawing from the next cheapest pool. To ensure that your target capacity is met, you might receive Spot Instances from more than the number of pools that you specified. Similarly, if most of the pools have no Spot capacity, you might receive your full target capacity from fewer than the number of pools that you specified.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-instancepoolstousecount
            '''
            result = self._values.get("instance_pools_to_use_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def maintenance_strategies(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.MaintenanceStrategiesProperty"]]:
            '''The strategies for managing your Spot Instances that are at an elevated risk of being interrupted.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-maintenancestrategies
            '''
            result = self._values.get("maintenance_strategies")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnEC2Fleet.MaintenanceStrategiesProperty"]], result)

        @builtins.property
        def max_total_price(self) -> typing.Optional[builtins.str]:
            '''The maximum amount per hour for Spot Instances that you're willing to pay.

            We do not recommend using this parameter because it can lead to increased interruptions. If you do not specify this parameter, you will pay the current Spot price.
            .. epigraph::

               If you specify a maximum price, your Spot Instances will be interrupted more frequently than if you do not specify this parameter. > If your fleet includes T instances that are configured as ``unlimited`` , and if their average CPU usage exceeds the baseline utilization, you will incur a charge for surplus credits. The ``MaxTotalPrice`` does not account for surplus credits, and, if you use surplus credits, your final cost might be higher than what you specified for ``MaxTotalPrice`` . For more information, see `Surplus credits can incur charges <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances-unlimited-mode-concepts.html#unlimited-mode-surplus-credits>`_ in the *Amazon EC2 User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-maxtotalprice
            '''
            result = self._values.get("max_total_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def min_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''The minimum target capacity for Spot Instances in the fleet.

            If this minimum capacity isn't reached, no instances are launched.

            Constraints: Maximum value of ``1000`` . Supported only for fleets of type ``instant`` .

            At least one of the following must be specified: ``SingleAvailabilityZone`` | ``SingleInstanceType``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-mintargetcapacity
            '''
            result = self._values.get("min_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def single_availability_zone(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates that the fleet launches all Spot Instances into a single Availability Zone.

            Supported only for fleets of type ``instant`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-singleavailabilityzone
            '''
            result = self._values.get("single_availability_zone")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def single_instance_type(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates that the fleet uses a single instance type to launch all Spot Instances in the fleet.

            Supported only for fleets of type ``instant`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-singleinstancetype
            '''
            result = self._values.get("single_instance_type")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SpotOptionsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: typing.Optional[builtins.str] = None,
            tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Specifies the tags to apply to a resource when the resource is being created for an EC2 Fleet.

            ``TagSpecification`` is a property of the `AWS::EC2::EC2Fleet <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html>`_ resource.

            :param resource_type: The type of resource to tag.
            :param tags: The tags to apply to the resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-tagspecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                tag_specification_property = ec2.CfnEC2Fleet.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__02540c2eeb5628d16d697845e0771b23c9a778ebfa08c4658341538bee2de664)
                check_type(argname="argument resource_type", value=resource_type, expected_type=type_hints["resource_type"])
                check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if resource_type is not None:
                self._values["resource_type"] = resource_type
            if tags is not None:
                self._values["tags"] = tags

        @builtins.property
        def resource_type(self) -> typing.Optional[builtins.str]:
            '''The type of resource to tag.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-tagspecification.html#cfn-ec2-ec2fleet-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
            '''The tags to apply to the resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-tagspecification.html#cfn-ec2-ec2fleet-tagspecification-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.TargetCapacitySpecificationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "total_target_capacity": "totalTargetCapacity",
            "default_target_capacity_type": "defaultTargetCapacityType",
            "on_demand_target_capacity": "onDemandTargetCapacity",
            "spot_target_capacity": "spotTargetCapacity",
            "target_capacity_unit_type": "targetCapacityUnitType",
        },
    )
    class TargetCapacitySpecificationRequestProperty:
        def __init__(
            self,
            *,
            total_target_capacity: jsii.Number,
            default_target_capacity_type: typing.Optional[builtins.str] = None,
            on_demand_target_capacity: typing.Optional[jsii.Number] = None,
            spot_target_capacity: typing.Optional[jsii.Number] = None,
            target_capacity_unit_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies the number of units to request for an EC2 Fleet.

            You can choose to set the target capacity in terms of instances or a performance characteristic that is important to your application workload, such as vCPUs, memory, or I/O. If the request type is ``maintain`` , you can specify a target capacity of ``0`` and add capacity later.

            ``TargetCapacitySpecificationRequest`` is a property of the `AWS::EC2::EC2Fleet <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html>`_ resource.

            :param total_target_capacity: The number of units to request, filled using the default target capacity type.
            :param default_target_capacity_type: The default target capacity type.
            :param on_demand_target_capacity: The number of On-Demand units to request.
            :param spot_target_capacity: The number of Spot units to request.
            :param target_capacity_unit_type: The unit for the target capacity. You can specify this parameter only when using attributed-based instance type selection. Default: ``units`` (the number of instances)

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                target_capacity_specification_request_property = ec2.CfnEC2Fleet.TargetCapacitySpecificationRequestProperty(
                    total_target_capacity=123,
                
                    # the properties below are optional
                    default_target_capacity_type="defaultTargetCapacityType",
                    on_demand_target_capacity=123,
                    spot_target_capacity=123,
                    target_capacity_unit_type="targetCapacityUnitType"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ffe51592d5662fcf623b04b7a62a1d91a35fb6ebea29d504ba00fa1b4e924887)
                check_type(argname="argument total_target_capacity", value=total_target_capacity, expected_type=type_hints["total_target_capacity"])
                check_type(argname="argument default_target_capacity_type", value=default_target_capacity_type, expected_type=type_hints["default_target_capacity_type"])
                check_type(argname="argument on_demand_target_capacity", value=on_demand_target_capacity, expected_type=type_hints["on_demand_target_capacity"])
                check_type(argname="argument spot_target_capacity", value=spot_target_capacity, expected_type=type_hints["spot_target_capacity"])
                check_type(argname="argument target_capacity_unit_type", value=target_capacity_unit_type, expected_type=type_hints["target_capacity_unit_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "total_target_capacity": total_target_capacity,
            }
            if default_target_capacity_type is not None:
                self._values["default_target_capacity_type"] = default_target_capacity_type
            if on_demand_target_capacity is not None:
                self._values["on_demand_target_capacity"] = on_demand_target_capacity
            if spot_target_capacity is not None:
                self._values["spot_target_capacity"] = spot_target_capacity
            if target_capacity_unit_type is not None:
                self._values["target_capacity_unit_type"] = target_capacity_unit_type

        @builtins.property
        def total_target_capacity(self) -> jsii.Number:
            '''The number of units to request, filled using the default target capacity type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-totaltargetcapacity
            '''
            result = self._values.get("total_target_capacity")
            assert result is not None, "Required property 'total_target_capacity' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def default_target_capacity_type(self) -> typing.Optional[builtins.str]:
            '''The default target capacity type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-defaulttargetcapacitytype
            '''
            result = self._values.get("default_target_capacity_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def on_demand_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''The number of On-Demand units to request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-ondemandtargetcapacity
            '''
            result = self._values.get("on_demand_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def spot_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''The number of Spot units to request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-spottargetcapacity
            '''
            result = self._values.get("spot_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def target_capacity_unit_type(self) -> typing.Optional[builtins.str]:
            '''The unit for the target capacity. You can specify this parameter only when using attributed-based instance type selection.

            Default: ``units`` (the number of instances)

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-targetcapacityunittype
            '''
            result = self._values.get("target_capacity_unit_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TargetCapacitySpecificationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class TotalLocalStorageGBRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum amount of total local storage, in GB.

            :param max: The maximum amount of total local storage, in GB. To specify no maximum limit, omit this parameter.
            :param min: The minimum amount of total local storage, in GB. To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-totallocalstoragegbrequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                total_local_storage_gBRequest_property = ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__68dd4f11ba96f8c88ae78d9d9a0bba32f2c60cb68f8364c2a828277864408781)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum amount of total local storage, in GB.

            To specify no maximum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-totallocalstoragegbrequest.html#cfn-ec2-ec2fleet-totallocalstoragegbrequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of total local storage, in GB.

            To specify no minimum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-totallocalstoragegbrequest.html#cfn-ec2-ec2fleet-totallocalstoragegbrequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TotalLocalStorageGBRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"max": "max", "min": "min"},
    )
    class VCpuCountRangeRequestProperty:
        def __init__(
            self,
            *,
            max: typing.Optional[jsii.Number] = None,
            min: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The minimum and maximum number of vCPUs.

            :param max: The maximum number of vCPUs. To specify no maximum limit, omit this parameter.
            :param min: The minimum number of vCPUs. To specify no minimum limit, specify ``0`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-vcpucountrangerequest.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                v_cpu_count_range_request_property = ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                    max=123,
                    min=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c97ef3c582df6d6b2c591c98531be161631a53d0b66f66fe471939622e85fdc5)
                check_type(argname="argument max", value=max, expected_type=type_hints["max"])
                check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max is not None:
                self._values["max"] = max
            if min is not None:
                self._values["min"] = min

        @builtins.property
        def max(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of vCPUs.

            To specify no maximum limit, omit this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-vcpucountrangerequest.html#cfn-ec2-ec2fleet-vcpucountrangerequest-max
            '''
            result = self._values.get("max")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of vCPUs.

            To specify no minimum limit, specify ``0`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-vcpucountrangerequest.html#cfn-ec2-ec2fleet-vcpucountrangerequest-min
            '''
            result = self._values.get("min")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VCpuCountRangeRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnEC2FleetProps",
    jsii_struct_bases=[],
    name_mapping={
        "launch_template_configs": "launchTemplateConfigs",
        "target_capacity_specification": "targetCapacitySpecification",
        "context": "context",
        "excess_capacity_termination_policy": "excessCapacityTerminationPolicy",
        "on_demand_options": "onDemandOptions",
        "replace_unhealthy_instances": "replaceUnhealthyInstances",
        "spot_options": "spotOptions",
        "tag_specifications": "tagSpecifications",
        "terminate_instances_with_expiration": "terminateInstancesWithExpiration",
        "type": "type",
        "valid_from": "validFrom",
        "valid_until": "validUntil",
    },
)
class CfnEC2FleetProps:
    def __init__(
        self,
        *,
        launch_template_configs: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty, typing.Dict[builtins.str, typing.Any]]]]],
        target_capacity_specification: typing.Union[_IResolvable_da3f097b, typing.Union[CfnEC2Fleet.TargetCapacitySpecificationRequestProperty, typing.Dict[builtins.str, typing.Any]]],
        context: typing.Optional[builtins.str] = None,
        excess_capacity_termination_policy: typing.Optional[builtins.str] = None,
        on_demand_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEC2Fleet.OnDemandOptionsRequestProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        replace_unhealthy_instances: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        spot_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEC2Fleet.SpotOptionsRequestProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tag_specifications: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnEC2Fleet.TagSpecificationProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        terminate_instances_with_expiration: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        type: typing.Optional[builtins.str] = None,
        valid_from: typing.Optional[builtins.str] = None,
        valid_until: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnEC2Fleet``.

        :param launch_template_configs: The configuration for the EC2 Fleet.
        :param target_capacity_specification: The number of units to request.
        :param context: Reserved.
        :param excess_capacity_termination_policy: Indicates whether running instances should be terminated if the total target capacity of the EC2 Fleet is decreased below the current size of the EC2 Fleet. Supported only for fleets of type ``maintain`` .
        :param on_demand_options: Describes the configuration of On-Demand Instances in an EC2 Fleet.
        :param replace_unhealthy_instances: Indicates whether EC2 Fleet should replace unhealthy Spot Instances. Supported only for fleets of type ``maintain`` . For more information, see `EC2 Fleet health checks <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/manage-ec2-fleet.html#ec2-fleet-health-checks>`_ in the *Amazon EC2 User Guide* .
        :param spot_options: Describes the configuration of Spot Instances in an EC2 Fleet.
        :param tag_specifications: The key-value pair for tagging the EC2 Fleet request on creation. For more information, see `Tag your resources <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html#tag-resources>`_ . If the fleet type is ``instant`` , specify a resource type of ``fleet`` to tag the fleet or ``instance`` to tag the instances at launch. If the fleet type is ``maintain`` or ``request`` , specify a resource type of ``fleet`` to tag the fleet. You cannot specify a resource type of ``instance`` . To tag instances at launch, specify the tags in a `launch template <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html#create-launch-template>`_ .
        :param terminate_instances_with_expiration: Indicates whether running instances should be terminated when the EC2 Fleet expires.
        :param type: The fleet type. The default value is ``maintain`` . - ``maintain`` - The EC2 Fleet places an asynchronous request for your desired capacity, and continues to maintain your desired Spot capacity by replenishing interrupted Spot Instances. - ``request`` - The EC2 Fleet places an asynchronous one-time request for your desired capacity, but does submit Spot requests in alternative capacity pools if Spot capacity is unavailable, and does not maintain Spot capacity if Spot Instances are interrupted. - ``instant`` - The EC2 Fleet places a synchronous one-time request for your desired capacity, and returns errors for any instances that could not be launched. For more information, see `EC2 Fleet request types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-request-type.html>`_ in the *Amazon EC2 User Guide* .
        :param valid_from: The start date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z). The default is to start fulfilling the request immediately.
        :param valid_until: The end date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z). At this point, no new EC2 Fleet requests are placed or able to fulfill the request. If no value is specified, the request remains until you cancel it.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_eC2_fleet_props = ec2.CfnEC2FleetProps(
                launch_template_configs=[ec2.CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty(
                    launch_template_specification=ec2.CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty(
                        version="version",
            
                        # the properties below are optional
                        launch_template_id="launchTemplateId",
                        launch_template_name="launchTemplateName"
                    ),
                    overrides=[ec2.CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty(
                        availability_zone="availabilityZone",
                        instance_requirements=ec2.CfnEC2Fleet.InstanceRequirementsRequestProperty(
                            accelerator_count=ec2.CfnEC2Fleet.AcceleratorCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_manufacturers=["acceleratorManufacturers"],
                            accelerator_names=["acceleratorNames"],
                            accelerator_total_memory_mi_b=ec2.CfnEC2Fleet.AcceleratorTotalMemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            accelerator_types=["acceleratorTypes"],
                            allowed_instance_types=["allowedInstanceTypes"],
                            bare_metal="bareMetal",
                            baseline_ebs_bandwidth_mbps=ec2.CfnEC2Fleet.BaselineEbsBandwidthMbpsRequestProperty(
                                max=123,
                                min=123
                            ),
                            baseline_performance_factors=ec2.CfnEC2Fleet.BaselinePerformanceFactorsRequestProperty(
                                cpu=ec2.CfnEC2Fleet.CpuPerformanceFactorRequestProperty(
                                    references=[ec2.CfnEC2Fleet.PerformanceFactorReferenceRequestProperty(
                                        instance_family="instanceFamily"
                                    )]
                                )
                            ),
                            burstable_performance="burstablePerformance",
                            cpu_manufacturers=["cpuManufacturers"],
                            excluded_instance_types=["excludedInstanceTypes"],
                            instance_generations=["instanceGenerations"],
                            local_storage="localStorage",
                            local_storage_types=["localStorageTypes"],
                            max_spot_price_as_percentage_of_optimal_on_demand_price=123,
                            memory_gi_bPer_vCpu=ec2.CfnEC2Fleet.MemoryGiBPerVCpuRequestProperty(
                                max=123,
                                min=123
                            ),
                            memory_mi_b=ec2.CfnEC2Fleet.MemoryMiBRequestProperty(
                                max=123,
                                min=123
                            ),
                            network_bandwidth_gbps=ec2.CfnEC2Fleet.NetworkBandwidthGbpsRequestProperty(
                                max=123,
                                min=123
                            ),
                            network_interface_count=ec2.CfnEC2Fleet.NetworkInterfaceCountRequestProperty(
                                max=123,
                                min=123
                            ),
                            on_demand_max_price_percentage_over_lowest_price=123,
                            require_hibernate_support=False,
                            spot_max_price_percentage_over_lowest_price=123,
                            total_local_storage_gb=ec2.CfnEC2Fleet.TotalLocalStorageGBRequestProperty(
                                max=123,
                                min=123
                            ),
                            v_cpu_count=ec2.CfnEC2Fleet.VCpuCountRangeRequestProperty(
                                max=123,
                                min=123
                            )
                        ),
                        instance_type="instanceType",
                        max_price="maxPrice",
                        placement=ec2.CfnEC2Fleet.PlacementProperty(
                            affinity="affinity",
                            availability_zone="availabilityZone",
                            group_name="groupName",
                            host_id="hostId",
                            host_resource_group_arn="hostResourceGroupArn",
                            partition_number=123,
                            spread_domain="spreadDomain",
                            tenancy="tenancy"
                        ),
                        priority=123,
                        subnet_id="subnetId",
                        weighted_capacity=123
                    )]
                )],
                target_capacity_specification=ec2.CfnEC2Fleet.TargetCapacitySpecificationRequestProperty(
                    total_target_capacity=123,
            
                    # the properties below are optional
                    default_target_capacity_type="defaultTargetCapacityType",
                    on_demand_target_capacity=123,
                    spot_target_capacity=123,
                    target_capacity_unit_type="targetCapacityUnitType"
                ),
            
                # the properties below are optional
                context="context",
                excess_capacity_termination_policy="excessCapacityTerminationPolicy",
                on_demand_options=ec2.CfnEC2Fleet.OnDemandOptionsRequestProperty(
                    allocation_strategy="allocationStrategy",
                    capacity_reservation_options=ec2.CfnEC2Fleet.CapacityReservationOptionsRequestProperty(
                        usage_strategy="usageStrategy"
                    ),
                    max_total_price="maxTotalPrice",
                    min_target_capacity=123,
                    single_availability_zone=False,
                    single_instance_type=False
                ),
                replace_unhealthy_instances=False,
                spot_options=ec2.CfnEC2Fleet.SpotOptionsRequestProperty(
                    allocation_strategy="allocationStrategy",
                    instance_interruption_behavior="instanceInterruptionBehavior",
                    instance_pools_to_use_count=123,
                    maintenance_strategies=ec2.CfnEC2Fleet.MaintenanceStrategiesProperty(
                        capacity_rebalance=ec2.CfnEC2Fleet.CapacityRebalanceProperty(
                            replacement_strategy="replacementStrategy",
                            termination_delay=123
                        )
                    ),
                    max_total_price="maxTotalPrice",
                    min_target_capacity=123,
                    single_availability_zone=False,
                    single_instance_type=False
                ),
                tag_specifications=[ec2.CfnEC2Fleet.TagSpecificationProperty(
                    resource_type="resourceType",
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )]
                )],
                terminate_instances_with_expiration=False,
                type="type",
                valid_from="validFrom",
                valid_until="validUntil"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7cf6835c4eccb4808ebe4e3b9b155d296d115ff66f1472a19acccd9978a85f7e)
            check_type(argname="argument launch_template_configs", value=launch_template_configs, expected_type=type_hints["launch_template_configs"])
            check_type(argname="argument target_capacity_specification", value=target_capacity_specification, expected_type=type_hints["target_capacity_specification"])
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
            check_type(argname="argument excess_capacity_termination_policy", value=excess_capacity_termination_policy, expected_type=type_hints["excess_capacity_termination_policy"])
            check_type(argname="argument on_demand_options", value=on_demand_options, expected_type=type_hints["on_demand_options"])
            check_type(argname="argument replace_unhealthy_instances", value=replace_unhealthy_instances, expected_type=type_hints["replace_unhealthy_instances"])
            check_type(argname="argument spot_options", value=spot_options, expected_type=type_hints["spot_options"])
            check_type(argname="argument tag_specifications", value=tag_specifications, expected_type=type_hints["tag_specifications"])
            check_type(argname="argument terminate_instances_with_expiration", value=terminate_instances_with_expiration, expected_type=type_hints["terminate_instances_with_expiration"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument valid_from", value=valid_from, expected_type=type_hints["valid_from"])
            check_type(argname="argument valid_until", value=valid_until, expected_type=type_hints["valid_until"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "launch_template_configs": launch_template_configs,
            "target_capacity_specification": target_capacity_specification,
        }
        if context is not None:
            self._values["context"] = context
        if excess_capacity_termination_policy is not None:
            self._values["excess_capacity_termination_policy"] = excess_capacity_termination_policy
        if on_demand_options is not None:
            self._values["on_demand_options"] = on_demand_options
        if replace_unhealthy_instances is not None:
            self._values["replace_unhealthy_instances"] = replace_unhealthy_instances
        if spot_options is not None:
            self._values["spot_options"] = spot_options
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if terminate_instances_with_expiration is not None:
            self._values["terminate_instances_with_expiration"] = terminate_instances_with_expiration
        if type is not None:
            self._values["type"] = type
        if valid_from is not None:
            self._values["valid_from"] = valid_from
        if valid_until is not None:
            self._values["valid_until"] = valid_until

    @builtins.property
    def launch_template_configs(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty]]]:
        '''The configuration for the EC2 Fleet.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-launchtemplateconfigs
        '''
        result = self._values.get("launch_template_configs")
        assert result is not None, "Required property 'launch_template_configs' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty]]], result)

    @builtins.property
    def target_capacity_specification(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnEC2Fleet.TargetCapacitySpecificationRequestProperty]:
        '''The number of units to request.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-targetcapacityspecification
        '''
        result = self._values.get("target_capacity_specification")
        assert result is not None, "Required property 'target_capacity_specification' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnEC2Fleet.TargetCapacitySpecificationRequestProperty], result)

    @builtins.property
    def context(self) -> typing.Optional[builtins.str]:
        '''Reserved.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-context
        '''
        result = self._values.get("context")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def excess_capacity_termination_policy(self) -> typing.Optional[builtins.str]:
        '''Indicates whether running instances should be terminated if the total target capacity of the EC2 Fleet is decreased below the current size of the EC2 Fleet.

        Supported only for fleets of type ``maintain`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-excesscapacityterminationpolicy
        '''
        result = self._values.get("excess_capacity_termination_policy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def on_demand_options(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEC2Fleet.OnDemandOptionsRequestProperty]]:
        '''Describes the configuration of On-Demand Instances in an EC2 Fleet.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-ondemandoptions
        '''
        result = self._values.get("on_demand_options")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEC2Fleet.OnDemandOptionsRequestProperty]], result)

    @builtins.property
    def replace_unhealthy_instances(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether EC2 Fleet should replace unhealthy Spot Instances.

        Supported only for fleets of type ``maintain`` . For more information, see `EC2 Fleet health checks <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/manage-ec2-fleet.html#ec2-fleet-health-checks>`_ in the *Amazon EC2 User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-replaceunhealthyinstances
        '''
        result = self._values.get("replace_unhealthy_instances")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def spot_options(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEC2Fleet.SpotOptionsRequestProperty]]:
        '''Describes the configuration of Spot Instances in an EC2 Fleet.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-spotoptions
        '''
        result = self._values.get("spot_options")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnEC2Fleet.SpotOptionsRequestProperty]], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnEC2Fleet.TagSpecificationProperty]]]]:
        '''The key-value pair for tagging the EC2 Fleet request on creation. For more information, see `Tag your resources <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Using_Tags.html#tag-resources>`_ .

        If the fleet type is ``instant`` , specify a resource type of ``fleet`` to tag the fleet or ``instance`` to tag the instances at launch.

        If the fleet type is ``maintain`` or ``request`` , specify a resource type of ``fleet`` to tag the fleet. You cannot specify a resource type of ``instance`` . To tag instances at launch, specify the tags in a `launch template <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html#create-launch-template>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnEC2Fleet.TagSpecificationProperty]]]], result)

    @builtins.property
    def terminate_instances_with_expiration(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Indicates whether running instances should be terminated when the EC2 Fleet expires.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-terminateinstanceswithexpiration
        '''
        result = self._values.get("terminate_instances_with_expiration")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def type(self) -> typing.Optional[builtins.str]:
        '''The fleet type. The default value is ``maintain`` .

        - ``maintain`` - The EC2 Fleet places an asynchronous request for your desired capacity, and continues to maintain your desired Spot capacity by replenishing interrupted Spot Instances.
        - ``request`` - The EC2 Fleet places an asynchronous one-time request for your desired capacity, but does submit Spot requests in alternative capacity pools if Spot capacity is unavailable, and does not maintain Spot capacity if Spot Instances are interrupted.
        - ``instant`` - The EC2 Fleet places a synchronous one-time request for your desired capacity, and returns errors for any instances that could not be launched.

        For more information, see `EC2 Fleet request types <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-request-type.html>`_ in the *Amazon EC2 User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-type
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def valid_from(self) -> typing.Optional[builtins.str]:
        '''The start date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z).

        The default is to start fulfilling the request immediately.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-validfrom
        '''
        result = self._values.get("valid_from")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def valid_until(self) -> typing.Optional[builtins.str]:
        '''The end date and time of the request, in UTC format (for example, *YYYY* - *MM* - *DD* T *HH* : *MM* : *SS* Z).

        At this point, no new EC2 Fleet requests are placed or able to fulfill the request. If no value is specified, the request remains until you cancel it.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-validuntil
        '''
        result = self._values.get("valid_until")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEC2FleetProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnEIP(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnEIP",
):
    '''Specifies an Elastic IP (EIP) address and can, optionally, associate it with an Amazon EC2 instance.

    You can allocate an Elastic IP address from an address pool owned by AWS or from an address pool created from a public IPv4 address range that you have brought to AWS for use with your AWS resources using bring your own IP addresses (BYOIP). For more information, see `Bring Your Own IP Addresses (BYOIP) <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html>`_ in the *Amazon EC2 User Guide* .

    For more information, see `Elastic IP Addresses <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/elastic-ip-addresses-eip.html>`_ in the *Amazon EC2 User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eip.html
    :cloudformationResource: AWS::EC2::EIP
    :exampleMetadata: infused

    Example::

        # listener: globalaccelerator.Listener
        # eip: ec2.CfnEIP
        
        
        listener.add_endpoint_group("Group",
            endpoints=[
                ga_endpoints.CfnEipEndpoint(eip,
                    weight=128
                )
            ]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        address: typing.Optional[builtins.str] = None,
        domain: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        ipam_pool_id: typing.Optional[builtins.str] = None,
        network_border_group: typing.Optional[builtins.str] = None,
        public_ipv4_pool: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        transfer_address: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param address: An Elastic IP address or a carrier IP address in a Wavelength Zone.
        :param domain: The network ( ``vpc`` ). If you define an Elastic IP address and associate it with a VPC that is defined in the same template, you must declare a dependency on the VPC-gateway attachment by using the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ on this resource.
        :param instance_id: The ID of the instance. .. epigraph:: Updates to the ``InstanceId`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.
        :param ipam_pool_id: The ID of an IPAM pool which has an Amazon-provided or BYOIP public IPv4 CIDR provisioned to it. For more information, see `Allocate sequential Elastic IP addresses from an IPAM pool <https://docs.aws.amazon.com/vpc/latest/ipam/tutorials-eip-pool.html>`_ in the *Amazon VPC IPAM User Guide* .
        :param network_border_group: A unique set of Availability Zones, Local Zones, or Wavelength Zones from which AWS advertises IP addresses. Use this parameter to limit the IP address to this location. IP addresses cannot move between network border groups. Use `DescribeAvailabilityZones <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html>`_ to view the network border groups.
        :param public_ipv4_pool: The ID of an address pool that you own. Use this parameter to let Amazon EC2 select an address from the address pool. .. epigraph:: Updates to the ``PublicIpv4Pool`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.
        :param tags: Any tags assigned to the Elastic IP address. .. epigraph:: Updates to the ``Tags`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.
        :param transfer_address: The Elastic IP address you are accepting for transfer. You can only accept one transferred address. For more information on Elastic IP address transfers, see `Transfer Elastic IP addresses <https://docs.aws.amazon.com/vpc/latest/userguide/vpc-eips.html#transfer-EIPs-intro>`_ in the *Amazon Virtual Private Cloud User Guide* .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bccc7dacde749a3ad3c41d116d1f509ffb5b6881668bfa61c205201b19574cf9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnEIPProps(
            address=address,
            domain=domain,
            instance_id=instance_id,
            ipam_pool_id=ipam_pool_id,
            network_border_group=network_border_group,
            public_ipv4_pool=public_ipv4_pool,
            tags=tags,
            transfer_address=transfer_address,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4ff25a50366f77a6df3a4ede4ef39917fc3d769d83828726514086b62d179131)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__11eb36186f31c4bea4b03f2dbdc66bc726d4347d12b0636ac6aafbdaf0aaddbe)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrAllocationId")
    def attr_allocation_id(self) -> builtins.str:
        '''The ID that AWS assigns to represent the allocation of the address for use with Amazon VPC.

        This is returned only for VPC elastic IP addresses. For example, ``eipalloc-5723d13e`` .

        :cloudformationAttribute: AllocationId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAllocationId"))

    @builtins.property
    @jsii.member(jsii_name="attrPublicIp")
    def attr_public_ip(self) -> builtins.str:
        '''The Elastic IP address.

        :cloudformationAttribute: PublicIp
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPublicIp"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="address")
    def address(self) -> typing.Optional[builtins.str]:
        '''An Elastic IP address or a carrier IP address in a Wavelength Zone.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "address"))

    @address.setter
    def address(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3181fa5c4d75016f17f69f9df76a3499499668e87d593e6609b7dfb341b5b54a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "address", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="domain")
    def domain(self) -> typing.Optional[builtins.str]:
        '''The network ( ``vpc`` ).'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domain"))

    @domain.setter
    def domain(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c49330fba74ade775140567a6ae48812faa62fb21c67fbdd7930ff5cb2189524)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domain", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the instance.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceId"))

    @instance_id.setter
    def instance_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ca5d5ea7abaea702ac65d5f9e8401058e42e908ac87dfdbb8722dcb03f673d98)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="ipamPoolId")
    def ipam_pool_id(self) -> typing.Optional[builtins.str]:
        '''The ID of an IPAM pool which has an Amazon-provided or BYOIP public IPv4 CIDR provisioned to it.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ipamPoolId"))

    @ipam_pool_id.setter
    def ipam_pool_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__01687921711d1f3c67e4646d6c469f48798d6ad165b0b5a020b27f122f7abf9e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ipamPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="networkBorderGroup")
    def network_border_group(self) -> typing.Optional[builtins.str]:
        '''A unique set of Availability Zones, Local Zones, or Wavelength Zones from which AWS advertises IP addresses.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkBorderGroup"))

    @network_border_group.setter
    def network_border_group(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ad1782664068283c33182db79bb89a1f93f451785b0dfd4fedeaf0a308110149)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "networkBorderGroup", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="publicIpv4Pool")
    def public_ipv4_pool(self) -> typing.Optional[builtins.str]:
        '''The ID of an address pool that you own.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "publicIpv4Pool"))

    @public_ipv4_pool.setter
    def public_ipv4_pool(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf9bd4708b5d85806a12d49dcb54159d4422fd6d4fea47a05f5f00ece5dca5a6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "publicIpv4Pool", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Any tags assigned to the Elastic IP address.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3fbabb7ce878301e495e9c01fb2e1183996e3d3ff9174b1398db7c58a5b47c3d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="transferAddress")
    def transfer_address(self) -> typing.Optional[builtins.str]:
        '''The Elastic IP address you are accepting for transfer.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "transferAddress"))

    @transfer_address.setter
    def transfer_address(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__09d9366c0699f2554304df4c739a5801dd39c1ba371b808dcc89894749cd3433)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "transferAddress", value) # pyright: ignore[reportArgumentType]


@jsii.implements(_IInspectable_c2943556)
class CfnEIPAssociation(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnEIPAssociation",
):
    '''Associates an Elastic IP address with an instance or a network interface.

    Before you can use an Elastic IP address, you must allocate it to your account. For more information about working with Elastic IP addresses, see `Elastic IP address concepts and rules <https://docs.aws.amazon.com/vpc/latest/userguide/vpc-eips.html#vpc-eip-overview>`_ .

    You must specify ``AllocationId`` and either ``InstanceId`` , ``NetworkInterfaceId`` , or ``PrivateIpAddress`` .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eipassociation.html
    :cloudformationResource: AWS::EC2::EIPAssociation
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_eIPAssociation = ec2.CfnEIPAssociation(self, "MyCfnEIPAssociation",
            allocation_id="allocationId",
            eip="eip",
            instance_id="instanceId",
            network_interface_id="networkInterfaceId",
            private_ip_address="privateIpAddress"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        allocation_id: typing.Optional[builtins.str] = None,
        eip: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        network_interface_id: typing.Optional[builtins.str] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param allocation_id: The allocation ID. This is required.
        :param eip: 
        :param instance_id: The ID of the instance. The instance must have exactly one attached network interface. You can specify either the instance ID or the network interface ID, but not both.
        :param network_interface_id: The ID of the network interface. If the instance has more than one network interface, you must specify a network interface ID. You can specify either the instance ID or the network interface ID, but not both.
        :param private_ip_address: The primary or secondary private IP address to associate with the Elastic IP address. If no private IP address is specified, the Elastic IP address is associated with the primary private IP address.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__08efa38e3271575d41d7626b99dd3858f0402a0285c2874241829879f80ad5dc)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnEIPAssociationProps(
            allocation_id=allocation_id,
            eip=eip,
            instance_id=instance_id,
            network_interface_id=network_interface_id,
            private_ip_address=private_ip_address,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d38ac0f8a89e7add67dbeb5b57090fe5a6fec2abaa2cfa15937784c2f88b46f9)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__14b55763d95c0a49faaa7084cf56cd092e1302d037f862850b20fff088532bd2)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID of the association.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="allocationId")
    def allocation_id(self) -> typing.Optional[builtins.str]:
        '''The allocation ID.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "allocationId"))

    @allocation_id.setter
    def allocation_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a145c0a59ed4b8e109758ede53ea195429ed116afe50bef5519ef1c93ef2f3b9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allocationId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="eip")
    def eip(self) -> typing.Optional[builtins.str]:
        '''
        :deprecated: this property has been deprecated

        :stability: deprecated
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "eip"))

    @eip.setter
    def eip(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f947c3656e7fb463745cdd982fbd04fbbd2146e52132e6c3c60b88bbf90d3372)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "eip", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the instance.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceId"))

    @instance_id.setter
    def instance_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__15ec06a481e0b7a8566ea8c48543a95bb46df08bdf87da3f3dcba102756d7221)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="networkInterfaceId")
    def network_interface_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the network interface.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkInterfaceId"))

    @network_interface_id.setter
    def network_interface_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6a3b32138b35250b675dba9ddde0e08a0eb7f077bcd27d5c1bfc7676f3dc799c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "networkInterfaceId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="privateIpAddress")
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''The primary or secondary private IP address to associate with the Elastic IP address.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "privateIpAddress"))

    @private_ip_address.setter
    def private_ip_address(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0013614294ef24c00fe534d4321fe1657b4fd366c4ea9eccfb2d19b2861d8440)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "privateIpAddress", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnEIPAssociationProps",
    jsii_struct_bases=[],
    name_mapping={
        "allocation_id": "allocationId",
        "eip": "eip",
        "instance_id": "instanceId",
        "network_interface_id": "networkInterfaceId",
        "private_ip_address": "privateIpAddress",
    },
)
class CfnEIPAssociationProps:
    def __init__(
        self,
        *,
        allocation_id: typing.Optional[builtins.str] = None,
        eip: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        network_interface_id: typing.Optional[builtins.str] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnEIPAssociation``.

        :param allocation_id: The allocation ID. This is required.
        :param eip: 
        :param instance_id: The ID of the instance. The instance must have exactly one attached network interface. You can specify either the instance ID or the network interface ID, but not both.
        :param network_interface_id: The ID of the network interface. If the instance has more than one network interface, you must specify a network interface ID. You can specify either the instance ID or the network interface ID, but not both.
        :param private_ip_address: The primary or secondary private IP address to associate with the Elastic IP address. If no private IP address is specified, the Elastic IP address is associated with the primary private IP address.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eipassociation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_eIPAssociation_props = ec2.CfnEIPAssociationProps(
                allocation_id="allocationId",
                eip="eip",
                instance_id="instanceId",
                network_interface_id="networkInterfaceId",
                private_ip_address="privateIpAddress"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d7e6118721d9f331f2cfafeff3be65471b4d44c1abf5e5083229fa0bb6ad726c)
            check_type(argname="argument allocation_id", value=allocation_id, expected_type=type_hints["allocation_id"])
            check_type(argname="argument eip", value=eip, expected_type=type_hints["eip"])
            check_type(argname="argument instance_id", value=instance_id, expected_type=type_hints["instance_id"])
            check_type(argname="argument network_interface_id", value=network_interface_id, expected_type=type_hints["network_interface_id"])
            check_type(argname="argument private_ip_address", value=private_ip_address, expected_type=type_hints["private_ip_address"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allocation_id is not None:
            self._values["allocation_id"] = allocation_id
        if eip is not None:
            self._values["eip"] = eip
        if instance_id is not None:
            self._values["instance_id"] = instance_id
        if network_interface_id is not None:
            self._values["network_interface_id"] = network_interface_id
        if private_ip_address is not None:
            self._values["private_ip_address"] = private_ip_address

    @builtins.property
    def allocation_id(self) -> typing.Optional[builtins.str]:
        '''The allocation ID.

        This is required.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eipassociation.html#cfn-ec2-eipassociation-allocationid
        '''
        result = self._values.get("allocation_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def eip(self) -> typing.Optional[builtins.str]:
        '''
        :deprecated: this property has been deprecated

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eipassociation.html#cfn-ec2-eipassociation-eip
        :stability: deprecated
        '''
        result = self._values.get("eip")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the instance.

        The instance must have exactly one attached network interface. You can specify either the instance ID or the network interface ID, but not both.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eipassociation.html#cfn-ec2-eipassociation-instanceid
        '''
        result = self._values.get("instance_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def network_interface_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the network interface.

        If the instance has more than one network interface, you must specify a network interface ID.

        You can specify either the instance ID or the network interface ID, but not both.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eipassociation.html#cfn-ec2-eipassociation-networkinterfaceid
        '''
        result = self._values.get("network_interface_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''The primary or secondary private IP address to associate with the Elastic IP address.

        If no private IP address is specified, the Elastic IP address is associated with the primary private IP address.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eipassociation.html#cfn-ec2-eipassociation-privateipaddress
        '''
        result = self._values.get("private_ip_address")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEIPAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnEIPProps",
    jsii_struct_bases=[],
    name_mapping={
        "address": "address",
        "domain": "domain",
        "instance_id": "instanceId",
        "ipam_pool_id": "ipamPoolId",
        "network_border_group": "networkBorderGroup",
        "public_ipv4_pool": "publicIpv4Pool",
        "tags": "tags",
        "transfer_address": "transferAddress",
    },
)
class CfnEIPProps:
    def __init__(
        self,
        *,
        address: typing.Optional[builtins.str] = None,
        domain: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        ipam_pool_id: typing.Optional[builtins.str] = None,
        network_border_group: typing.Optional[builtins.str] = None,
        public_ipv4_pool: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        transfer_address: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnEIP``.

        :param address: An Elastic IP address or a carrier IP address in a Wavelength Zone.
        :param domain: The network ( ``vpc`` ). If you define an Elastic IP address and associate it with a VPC that is defined in the same template, you must declare a dependency on the VPC-gateway attachment by using the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ on this resource.
        :param instance_id: The ID of the instance. .. epigraph:: Updates to the ``InstanceId`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.
        :param ipam_pool_id: The ID of an IPAM pool which has an Amazon-provided or BYOIP public IPv4 CIDR provisioned to it. For more information, see `Allocate sequential Elastic IP addresses from an IPAM pool <https://docs.aws.amazon.com/vpc/latest/ipam/tutorials-eip-pool.html>`_ in the *Amazon VPC IPAM User Guide* .
        :param network_border_group: A unique set of Availability Zones, Local Zones, or Wavelength Zones from which AWS advertises IP addresses. Use this parameter to limit the IP address to this location. IP addresses cannot move between network border groups. Use `DescribeAvailabilityZones <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html>`_ to view the network border groups.
        :param public_ipv4_pool: The ID of an address pool that you own. Use this parameter to let Amazon EC2 select an address from the address pool. .. epigraph:: Updates to the ``PublicIpv4Pool`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.
        :param tags: Any tags assigned to the Elastic IP address. .. epigraph:: Updates to the ``Tags`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.
        :param transfer_address: The Elastic IP address you are accepting for transfer. You can only accept one transferred address. For more information on Elastic IP address transfers, see `Transfer Elastic IP addresses <https://docs.aws.amazon.com/vpc/latest/userguide/vpc-eips.html#transfer-EIPs-intro>`_ in the *Amazon Virtual Private Cloud User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eip.html
        :exampleMetadata: infused

        Example::

            # instance: ec2.Instance
            
            # my_zone: route53.HostedZone
            
            
            elastic_ip = ec2.CfnEIP(self, "EIP",
                domain="vpc",
                instance_id=instance.instance_id
            )
            route53.ARecord(self, "ARecord",
                zone=my_zone,
                target=route53.RecordTarget.from_ip_addresses(elastic_ip.ref)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ff96bcb9544d5ebc6273ac8c518468df0cc1fd7720428f6e4ec85ecddb06fa60)
            check_type(argname="argument address", value=address, expected_type=type_hints["address"])
            check_type(argname="argument domain", value=domain, expected_type=type_hints["domain"])
            check_type(argname="argument instance_id", value=instance_id, expected_type=type_hints["instance_id"])
            check_type(argname="argument ipam_pool_id", value=ipam_pool_id, expected_type=type_hints["ipam_pool_id"])
            check_type(argname="argument network_border_group", value=network_border_group, expected_type=type_hints["network_border_group"])
            check_type(argname="argument public_ipv4_pool", value=public_ipv4_pool, expected_type=type_hints["public_ipv4_pool"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument transfer_address", value=transfer_address, expected_type=type_hints["transfer_address"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if address is not None:
            self._values["address"] = address
        if domain is not None:
            self._values["domain"] = domain
        if instance_id is not None:
            self._values["instance_id"] = instance_id
        if ipam_pool_id is not None:
            self._values["ipam_pool_id"] = ipam_pool_id
        if network_border_group is not None:
            self._values["network_border_group"] = network_border_group
        if public_ipv4_pool is not None:
            self._values["public_ipv4_pool"] = public_ipv4_pool
        if tags is not None:
            self._values["tags"] = tags
        if transfer_address is not None:
            self._values["transfer_address"] = transfer_address

    @builtins.property
    def address(self) -> typing.Optional[builtins.str]:
        '''An Elastic IP address or a carrier IP address in a Wavelength Zone.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eip.html#cfn-ec2-eip-address
        '''
        result = self._values.get("address")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        '''The network ( ``vpc`` ).

        If you define an Elastic IP address and associate it with a VPC that is defined in the same template, you must declare a dependency on the VPC-gateway attachment by using the `DependsOn Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html>`_ on this resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eip.html#cfn-ec2-eip-domain
        '''
        result = self._values.get("domain")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the instance.

        .. epigraph::

           Updates to the ``InstanceId`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eip.html#cfn-ec2-eip-instanceid
        '''
        result = self._values.get("instance_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ipam_pool_id(self) -> typing.Optional[builtins.str]:
        '''The ID of an IPAM pool which has an Amazon-provided or BYOIP public IPv4 CIDR provisioned to it.

        For more information, see `Allocate sequential Elastic IP addresses from an IPAM pool <https://docs.aws.amazon.com/vpc/latest/ipam/tutorials-eip-pool.html>`_ in the *Amazon VPC IPAM User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eip.html#cfn-ec2-eip-ipampoolid
        '''
        result = self._values.get("ipam_pool_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def network_border_group(self) -> typing.Optional[builtins.str]:
        '''A unique set of Availability Zones, Local Zones, or Wavelength Zones from which AWS advertises IP addresses.

        Use this parameter to limit the IP address to this location. IP addresses cannot move between network border groups.

        Use `DescribeAvailabilityZones <https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html>`_ to view the network border groups.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eip.html#cfn-ec2-eip-networkbordergroup
        '''
        result = self._values.get("network_border_group")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def public_ipv4_pool(self) -> typing.Optional[builtins.str]:
        '''The ID of an address pool that you own.

        Use this parameter to let Amazon EC2 select an address from the address pool.
        .. epigraph::

           Updates to the ``PublicIpv4Pool`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eip.html#cfn-ec2-eip-publicipv4pool
        '''
        result = self._values.get("public_ipv4_pool")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Any tags assigned to the Elastic IP address.

        .. epigraph::

           Updates to the ``Tags`` property may require *some interruptions* . Updates on an EIP reassociates the address on its associated resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eip.html#cfn-ec2-eip-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def transfer_address(self) -> typing.Optional[builtins.str]:
        '''The Elastic IP address you are accepting for transfer.

        You can only accept one transferred address. For more information on Elastic IP address transfers, see `Transfer Elastic IP addresses <https://docs.aws.amazon.com/vpc/latest/userguide/vpc-eips.html#transfer-EIPs-intro>`_ in the *Amazon Virtual Private Cloud User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-eip.html#cfn-ec2-eip-transferaddress
        '''
        result = self._values.get("transfer_address")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEIPProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnEgressOnlyInternetGateway(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnEgressOnlyInternetGateway",
):
    '''[IPv6 only] Specifies an egress-only internet gateway for your VPC.

    An egress-only internet gateway is used to enable outbound communication over IPv6 from instances in your VPC to the internet, and prevents hosts outside of your VPC from initiating an IPv6 connection with your instance.

    For more information, see `Egress-only internet gateway <https://docs.aws.amazon.com/vpc/latest/userguide/egress-only-internet-gateway.html>`_ in the *Amazon VPC User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-egressonlyinternetgateway.html
    :cloudformationResource: AWS::EC2::EgressOnlyInternetGateway
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_egress_only_internet_gateway = ec2.CfnEgressOnlyInternetGateway(self, "MyCfnEgressOnlyInternetGateway",
            vpc_id="vpcId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        vpc_id: builtins.str,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param vpc_id: The ID of the VPC for which to create the egress-only internet gateway.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__84a7ddca98bd1c24713f12588ec54b51cdc19c99c2209e07c964172011c4d7ab)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnEgressOnlyInternetGatewayProps(vpc_id=vpc_id)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3776ca79b4c1077cd8641d4bff766f9377360b8cadb064edee5912829e24521b)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3ac2763e5da89e4e1ee6acf841762d6a8bc01d40589085b652b4f5f545c13102)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID of the egress-only internet gateway.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> builtins.str:
        '''The ID of the VPC for which to create the egress-only internet gateway.'''
        return typing.cast(builtins.str, jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dd41a3676da418b0ac30e8c6707af491b5f32416672bf517e640f6132a942063)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vpcId", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnEgressOnlyInternetGatewayProps",
    jsii_struct_bases=[],
    name_mapping={"vpc_id": "vpcId"},
)
class CfnEgressOnlyInternetGatewayProps:
    def __init__(self, *, vpc_id: builtins.str) -> None:
        '''Properties for defining a ``CfnEgressOnlyInternetGateway``.

        :param vpc_id: The ID of the VPC for which to create the egress-only internet gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-egressonlyinternetgateway.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_egress_only_internet_gateway_props = ec2.CfnEgressOnlyInternetGatewayProps(
                vpc_id="vpcId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b693b2d49003d73758f4c0003564a93353b18fc97434556a2e988e47f367fb84)
            check_type(argname="argument vpc_id", value=vpc_id, expected_type=type_hints["vpc_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "vpc_id": vpc_id,
        }

    @builtins.property
    def vpc_id(self) -> builtins.str:
        '''The ID of the VPC for which to create the egress-only internet gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-egressonlyinternetgateway.html#cfn-ec2-egressonlyinternetgateway-vpcid
        '''
        result = self._values.get("vpc_id")
        assert result is not None, "Required property 'vpc_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEgressOnlyInternetGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnEnclaveCertificateIamRoleAssociation(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnEnclaveCertificateIamRoleAssociation",
):
    '''Associates an AWS Identity and Access Management (IAM) role with an AWS Certificate Manager (ACM) certificate.

    This enables the certificate to be used by the ACM for Nitro Enclaves application inside an enclave. For more information, see `AWS Certificate Manager for Nitro Enclaves <https://docs.aws.amazon.com/enclaves/latest/user/nitro-enclave-refapp.html>`_ in the *AWS Nitro Enclaves User Guide* .

    When the IAM role is associated with the ACM certificate, the certificate, certificate chain, and encrypted private key are placed in an Amazon S3 location that only the associated IAM role can access. The private key of the certificate is encrypted with an AWS managed key that has an attached attestation-based key policy.

    To enable the IAM role to access the Amazon S3 object, you must grant it permission to call ``s3:GetObject`` on the Amazon S3 bucket returned by the command. To enable the IAM role to access the KMS key, you must grant it permission to call ``kms:Decrypt`` on the KMS key returned by the command. For more information, see `Grant the role permission to access the certificate and encryption key <https://docs.aws.amazon.com/enclaves/latest/user/nitro-enclave-refapp.html#add-policy>`_ in the *AWS Nitro Enclaves User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html
    :cloudformationResource: AWS::EC2::EnclaveCertificateIamRoleAssociation
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_enclave_certificate_iam_role_association = ec2.CfnEnclaveCertificateIamRoleAssociation(self, "MyCfnEnclaveCertificateIamRoleAssociation",
            certificate_arn="certificateArn",
            role_arn="roleArn"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        certificate_arn: builtins.str,
        role_arn: builtins.str,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param certificate_arn: The ARN of the ACM certificate with which to associate the IAM role.
        :param role_arn: The ARN of the IAM role to associate with the ACM certificate. You can associate up to 16 IAM roles with an ACM certificate.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d7cc97af7e0281c5467b107a4b4d0bb76b59e1c397161a6fa7d7f0a2a918097)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnEnclaveCertificateIamRoleAssociationProps(
            certificate_arn=certificate_arn, role_arn=role_arn
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__75c5c30f47a18964a08b0e104af0e125e2c6d69d00e53e23a8faba6e7efb2248)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__49fde7f75409bd6c3eef1dc479fe288b3ab71c68799ef372073205fdedd03d81)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCertificateS3BucketName")
    def attr_certificate_s3_bucket_name(self) -> builtins.str:
        '''The name of the Amazon S3 bucket to which the certificate was uploaded.

        :cloudformationAttribute: CertificateS3BucketName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCertificateS3BucketName"))

    @builtins.property
    @jsii.member(jsii_name="attrCertificateS3ObjectKey")
    def attr_certificate_s3_object_key(self) -> builtins.str:
        '''The Amazon S3 object key where the certificate, certificate chain, and encrypted private key bundle are stored.

        The object key is formatted as follows: ``role_arn`` / ``certificate_arn`` .

        :cloudformationAttribute: CertificateS3ObjectKey
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCertificateS3ObjectKey"))

    @builtins.property
    @jsii.member(jsii_name="attrEncryptionKmsKeyId")
    def attr_encryption_kms_key_id(self) -> builtins.str:
        '''The ID of the AWS KMS key used to encrypt the private key of the certificate.

        :cloudformationAttribute: EncryptionKmsKeyId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEncryptionKmsKeyId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="certificateArn")
    def certificate_arn(self) -> builtins.str:
        '''The ARN of the ACM certificate with which to associate the IAM role.'''
        return typing.cast(builtins.str, jsii.get(self, "certificateArn"))

    @certificate_arn.setter
    def certificate_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__36a9750ebcb712bef33823ecd48c04b607764f04df094442b80cab92b9d24239)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "certificateArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        '''The ARN of the IAM role to associate with the ACM certificate.'''
        return typing.cast(builtins.str, jsii.get(self, "roleArn"))

    @role_arn.setter
    def role_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bebb7fefbc76f390f31133552e945be922a94a4a480afe3ae0490c0dad9a102d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "roleArn", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnEnclaveCertificateIamRoleAssociationProps",
    jsii_struct_bases=[],
    name_mapping={"certificate_arn": "certificateArn", "role_arn": "roleArn"},
)
class CfnEnclaveCertificateIamRoleAssociationProps:
    def __init__(
        self,
        *,
        certificate_arn: builtins.str,
        role_arn: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnEnclaveCertificateIamRoleAssociation``.

        :param certificate_arn: The ARN of the ACM certificate with which to associate the IAM role.
        :param role_arn: The ARN of the IAM role to associate with the ACM certificate. You can associate up to 16 IAM roles with an ACM certificate.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_enclave_certificate_iam_role_association_props = ec2.CfnEnclaveCertificateIamRoleAssociationProps(
                certificate_arn="certificateArn",
                role_arn="roleArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c1d46b0b7748b6af77bfec597465bb1c4d453df888f2782e783135feee848f1b)
            check_type(argname="argument certificate_arn", value=certificate_arn, expected_type=type_hints["certificate_arn"])
            check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "certificate_arn": certificate_arn,
            "role_arn": role_arn,
        }

    @builtins.property
    def certificate_arn(self) -> builtins.str:
        '''The ARN of the ACM certificate with which to associate the IAM role.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html#cfn-ec2-enclavecertificateiamroleassociation-certificatearn
        '''
        result = self._values.get("certificate_arn")
        assert result is not None, "Required property 'certificate_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def role_arn(self) -> builtins.str:
        '''The ARN of the IAM role to associate with the ACM certificate.

        You can associate up to 16 IAM roles with an ACM certificate.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html#cfn-ec2-enclavecertificateiamroleassociation-rolearn
        '''
        result = self._values.get("role_arn")
        assert result is not None, "Required property 'role_arn' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEnclaveCertificateIamRoleAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnFlowLog(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnFlowLog",
):
    '''Specifies a VPC flow log that captures IP traffic for a specified network interface, subnet, or VPC.

    To view the log data, use Amazon CloudWatch Logs (CloudWatch Logs) to help troubleshoot connection issues. For example, you can use a flow log to investigate why certain traffic isn't reaching an instance, which can help you diagnose overly restrictive security group rules. For more information, see `VPC Flow Logs <https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html>`_ in the *Amazon VPC User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html
    :cloudformationResource: AWS::EC2::FlowLog
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        # destination_options: Any
        
        cfn_flow_log = ec2.CfnFlowLog(self, "MyCfnFlowLog",
            resource_id="resourceId",
            resource_type="resourceType",
        
            # the properties below are optional
            deliver_cross_account_role="deliverCrossAccountRole",
            deliver_logs_permission_arn="deliverLogsPermissionArn",
            destination_options=destination_options,
            log_destination="logDestination",
            log_destination_type="logDestinationType",
            log_format="logFormat",
            log_group_name="logGroupName",
            max_aggregation_interval=123,
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            traffic_type="trafficType"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        resource_id: builtins.str,
        resource_type: builtins.str,
        deliver_cross_account_role: typing.Optional[builtins.str] = None,
        deliver_logs_permission_arn: typing.Optional[builtins.str] = None,
        destination_options: typing.Any = None,
        log_destination: typing.Optional[builtins.str] = None,
        log_destination_type: typing.Optional[builtins.str] = None,
        log_format: typing.Optional[builtins.str] = None,
        log_group_name: typing.Optional[builtins.str] = None,
        max_aggregation_interval: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        traffic_type: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param resource_id: The ID of the resource to monitor. For example, if the resource type is ``VPC`` , specify the ID of the VPC.
        :param resource_type: The type of resource to monitor.
        :param deliver_cross_account_role: The ARN of the IAM role that allows the service to publish flow logs across accounts.
        :param deliver_logs_permission_arn: The ARN of the IAM role that allows Amazon EC2 to publish flow logs to the log destination. This parameter is required if the destination type is ``cloud-watch-logs`` , or if the destination type is ``kinesis-data-firehose`` and the delivery stream and the resources to monitor are in different accounts.
        :param destination_options: The destination options.
        :param log_destination: The destination for the flow log data. The meaning of this parameter depends on the destination type. - If the destination type is ``cloud-watch-logs`` , specify the ARN of a CloudWatch Logs log group. For example: arn:aws:logs: *region* : *account_id* :log-group: *my_group* Alternatively, use the ``LogGroupName`` parameter. - If the destination type is ``s3`` , specify the ARN of an S3 bucket. For example: arn:aws:s3::: *my_bucket* / *my_subfolder* / The subfolder is optional. Note that you can't use ``AWSLogs`` as a subfolder name. - If the destination type is ``kinesis-data-firehose`` , specify the ARN of a Kinesis Data Firehose delivery stream. For example: arn:aws:firehose: *region* : *account_id* :deliverystream: *my_stream*
        :param log_destination_type: The type of destination for the flow log data. Default: ``cloud-watch-logs``
        :param log_format: The fields to include in the flow log record, in the order in which they should appear. If you omit this parameter, the flow log is created using the default format. If you specify this parameter, you must include at least one field. For more information about the available fields, see `Flow log records <https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records>`_ in the *Amazon VPC User Guide* or `Transit Gateway Flow Log records <https://docs.aws.amazon.com/vpc/latest/tgw/tgw-flow-logs.html#flow-log-records>`_ in the *AWS Transit Gateway Guide* . Specify the fields using the ``${field-id}`` format, separated by spaces.
        :param log_group_name: The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs. This parameter is valid only if the destination type is ``cloud-watch-logs`` .
        :param max_aggregation_interval: The maximum interval of time during which a flow of packets is captured and aggregated into a flow log record. The possible values are 60 seconds (1 minute) or 600 seconds (10 minutes). This parameter must be 60 seconds for transit gateway resource types. When a network interface is attached to a `Nitro-based instance <https://docs.aws.amazon.com/ec2/latest/instancetypes/ec2-nitro-instances.html>`_ , the aggregation interval is always 60 seconds or less, regardless of the value that you specify. Default: 600
        :param tags: The tags to apply to the flow logs.
        :param traffic_type: The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic). This parameter is not supported for transit gateway resource types. It is required for the other resource types.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__72358d22276e2eb5cd641ca06942afc99075017aabb8f3000a9b07b53750fcdb)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnFlowLogProps(
            resource_id=resource_id,
            resource_type=resource_type,
            deliver_cross_account_role=deliver_cross_account_role,
            deliver_logs_permission_arn=deliver_logs_permission_arn,
            destination_options=destination_options,
            log_destination=log_destination,
            log_destination_type=log_destination_type,
            log_format=log_format,
            log_group_name=log_group_name,
            max_aggregation_interval=max_aggregation_interval,
            tags=tags,
            traffic_type=traffic_type,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7e198443ee858ff19e4aabc024d3248d8cb921d93f5a62465d022de2d8cfa5e2)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6eff7a114e2714334c6df6d1b2e56918a556871696e25a588bf089c4a4ea3f2a)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID of the flow log.

        For example, ``fl-123456abc123abc1`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="resourceId")
    def resource_id(self) -> builtins.str:
        '''The ID of the resource to monitor.'''
        return typing.cast(builtins.str, jsii.get(self, "resourceId"))

    @resource_id.setter
    def resource_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__84e5a2c6ad5e904ca809a2f153ebda2b81c23f9d634700537999ae612f2d8462)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "resourceId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="resourceType")
    def resource_type(self) -> builtins.str:
        '''The type of resource to monitor.'''
        return typing.cast(builtins.str, jsii.get(self, "resourceType"))

    @resource_type.setter
    def resource_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d21f005f9535582c1f46053bf4e82c2bbabff7cded5a69095501539e5f469ff3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "resourceType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="deliverCrossAccountRole")
    def deliver_cross_account_role(self) -> typing.Optional[builtins.str]:
        '''The ARN of the IAM role that allows the service to publish flow logs across accounts.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "deliverCrossAccountRole"))

    @deliver_cross_account_role.setter
    def deliver_cross_account_role(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__69de10af6e71879a605a8ff40c4836ff6e89ad4220088a0599668acc26625a53)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deliverCrossAccountRole", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="deliverLogsPermissionArn")
    def deliver_logs_permission_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the IAM role that allows Amazon EC2 to publish flow logs to the log destination.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "deliverLogsPermissionArn"))

    @deliver_logs_permission_arn.setter
    def deliver_logs_permission_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b921f8c4adf8ca7781a1e0174a51a4017191049028a6f93859d3f656946c7c3f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deliverLogsPermissionArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="destinationOptions")
    def destination_options(self) -> typing.Any:
        '''The destination options.'''
        return typing.cast(typing.Any, jsii.get(self, "destinationOptions"))

    @destination_options.setter
    def destination_options(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__49796be537e8b2013810fde10b43929fda215b3a4171af478b31fdb27302d616)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "destinationOptions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="logDestination")
    def log_destination(self) -> typing.Optional[builtins.str]:
        '''The destination for the flow log data.

        The meaning of this parameter depends on the destination type.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logDestination"))

    @log_destination.setter
    def log_destination(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__949d939e627fcdb6a73a2f76769f6cdfb5973630d761825729685b7af98b27f7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "logDestination", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="logDestinationType")
    def log_destination_type(self) -> typing.Optional[builtins.str]:
        '''The type of destination for the flow log data.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logDestinationType"))

    @log_destination_type.setter
    def log_destination_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ce5094d9224f3fc4ab9de0af2048c5229af2bab60296a1532059c7fd0f35152c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "logDestinationType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="logFormat")
    def log_format(self) -> typing.Optional[builtins.str]:
        '''The fields to include in the flow log record, in the order in which they should appear.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logFormat"))

    @log_format.setter
    def log_format(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8c84872a8fa41e81525525e433ae100f134862fcfaf118eb631c2d92ee89fd88)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "logFormat", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="logGroupName")
    def log_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logGroupName"))

    @log_group_name.setter
    def log_group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6d4c0e5979a8d23aa64b8481c35dcd22bd972c3b805f0f16e2bcf9249dd13e78)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "logGroupName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="maxAggregationInterval")
    def max_aggregation_interval(self) -> typing.Optional[jsii.Number]:
        '''The maximum interval of time during which a flow of packets is captured and aggregated into a flow log record.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxAggregationInterval"))

    @max_aggregation_interval.setter
    def max_aggregation_interval(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__408957857658c91ed4a24dd2c0001b02c4d953a90495ab29b67a8f47d4c8637d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "maxAggregationInterval", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The tags to apply to the flow logs.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d5a7fc9ee55af57fe376f2a1925ca134f75de57ffce411d2318f45b1d139b4a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="trafficType")
    def traffic_type(self) -> typing.Optional[builtins.str]:
        '''The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic).'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "trafficType"))

    @traffic_type.setter
    def traffic_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__648073b4921e181f4252118a3f0bd9e0910cc4a9f11632928884f0d6894061ff)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "trafficType", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnFlowLog.DestinationOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "file_format": "fileFormat",
            "hive_compatible_partitions": "hiveCompatiblePartitions",
            "per_hour_partition": "perHourPartition",
        },
    )
    class DestinationOptionsProperty:
        def __init__(
            self,
            *,
            file_format: builtins.str,
            hive_compatible_partitions: typing.Union[builtins.bool, _IResolvable_da3f097b],
            per_hour_partition: typing.Union[builtins.bool, _IResolvable_da3f097b],
        ) -> None:
            '''Describes the destination options for a flow log.

            :param file_format: The format for the flow log. The default is ``plain-text`` .
            :param hive_compatible_partitions: Indicates whether to use Hive-compatible prefixes for flow logs stored in Amazon S3. The default is ``false`` .
            :param per_hour_partition: Indicates whether to partition the flow log per hour. This reduces the cost and response time for queries. The default is ``false`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-flowlog-destinationoptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                destination_options_property = ec2.CfnFlowLog.DestinationOptionsProperty(
                    file_format="fileFormat",
                    hive_compatible_partitions=False,
                    per_hour_partition=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__93145601fe9d5b59646cf267ec0583b6a5749f370d14826d2dc5888efb2826a1)
                check_type(argname="argument file_format", value=file_format, expected_type=type_hints["file_format"])
                check_type(argname="argument hive_compatible_partitions", value=hive_compatible_partitions, expected_type=type_hints["hive_compatible_partitions"])
                check_type(argname="argument per_hour_partition", value=per_hour_partition, expected_type=type_hints["per_hour_partition"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "file_format": file_format,
                "hive_compatible_partitions": hive_compatible_partitions,
                "per_hour_partition": per_hour_partition,
            }

        @builtins.property
        def file_format(self) -> builtins.str:
            '''The format for the flow log.

            The default is ``plain-text`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-flowlog-destinationoptions.html#cfn-ec2-flowlog-destinationoptions-fileformat
            '''
            result = self._values.get("file_format")
            assert result is not None, "Required property 'file_format' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def hive_compatible_partitions(
            self,
        ) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Indicates whether to use Hive-compatible prefixes for flow logs stored in Amazon S3.

            The default is ``false`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-flowlog-destinationoptions.html#cfn-ec2-flowlog-destinationoptions-hivecompatiblepartitions
            '''
            result = self._values.get("hive_compatible_partitions")
            assert result is not None, "Required property 'hive_compatible_partitions' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        @builtins.property
        def per_hour_partition(
            self,
        ) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Indicates whether to partition the flow log per hour.

            This reduces the cost and response time for queries. The default is ``false`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-flowlog-destinationoptions.html#cfn-ec2-flowlog-destinationoptions-perhourpartition
            '''
            result = self._values.get("per_hour_partition")
            assert result is not None, "Required property 'per_hour_partition' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DestinationOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnFlowLogProps",
    jsii_struct_bases=[],
    name_mapping={
        "resource_id": "resourceId",
        "resource_type": "resourceType",
        "deliver_cross_account_role": "deliverCrossAccountRole",
        "deliver_logs_permission_arn": "deliverLogsPermissionArn",
        "destination_options": "destinationOptions",
        "log_destination": "logDestination",
        "log_destination_type": "logDestinationType",
        "log_format": "logFormat",
        "log_group_name": "logGroupName",
        "max_aggregation_interval": "maxAggregationInterval",
        "tags": "tags",
        "traffic_type": "trafficType",
    },
)
class CfnFlowLogProps:
    def __init__(
        self,
        *,
        resource_id: builtins.str,
        resource_type: builtins.str,
        deliver_cross_account_role: typing.Optional[builtins.str] = None,
        deliver_logs_permission_arn: typing.Optional[builtins.str] = None,
        destination_options: typing.Any = None,
        log_destination: typing.Optional[builtins.str] = None,
        log_destination_type: typing.Optional[builtins.str] = None,
        log_format: typing.Optional[builtins.str] = None,
        log_group_name: typing.Optional[builtins.str] = None,
        max_aggregation_interval: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        traffic_type: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnFlowLog``.

        :param resource_id: The ID of the resource to monitor. For example, if the resource type is ``VPC`` , specify the ID of the VPC.
        :param resource_type: The type of resource to monitor.
        :param deliver_cross_account_role: The ARN of the IAM role that allows the service to publish flow logs across accounts.
        :param deliver_logs_permission_arn: The ARN of the IAM role that allows Amazon EC2 to publish flow logs to the log destination. This parameter is required if the destination type is ``cloud-watch-logs`` , or if the destination type is ``kinesis-data-firehose`` and the delivery stream and the resources to monitor are in different accounts.
        :param destination_options: The destination options.
        :param log_destination: The destination for the flow log data. The meaning of this parameter depends on the destination type. - If the destination type is ``cloud-watch-logs`` , specify the ARN of a CloudWatch Logs log group. For example: arn:aws:logs: *region* : *account_id* :log-group: *my_group* Alternatively, use the ``LogGroupName`` parameter. - If the destination type is ``s3`` , specify the ARN of an S3 bucket. For example: arn:aws:s3::: *my_bucket* / *my_subfolder* / The subfolder is optional. Note that you can't use ``AWSLogs`` as a subfolder name. - If the destination type is ``kinesis-data-firehose`` , specify the ARN of a Kinesis Data Firehose delivery stream. For example: arn:aws:firehose: *region* : *account_id* :deliverystream: *my_stream*
        :param log_destination_type: The type of destination for the flow log data. Default: ``cloud-watch-logs``
        :param log_format: The fields to include in the flow log record, in the order in which they should appear. If you omit this parameter, the flow log is created using the default format. If you specify this parameter, you must include at least one field. For more information about the available fields, see `Flow log records <https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records>`_ in the *Amazon VPC User Guide* or `Transit Gateway Flow Log records <https://docs.aws.amazon.com/vpc/latest/tgw/tgw-flow-logs.html#flow-log-records>`_ in the *AWS Transit Gateway Guide* . Specify the fields using the ``${field-id}`` format, separated by spaces.
        :param log_group_name: The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs. This parameter is valid only if the destination type is ``cloud-watch-logs`` .
        :param max_aggregation_interval: The maximum interval of time during which a flow of packets is captured and aggregated into a flow log record. The possible values are 60 seconds (1 minute) or 600 seconds (10 minutes). This parameter must be 60 seconds for transit gateway resource types. When a network interface is attached to a `Nitro-based instance <https://docs.aws.amazon.com/ec2/latest/instancetypes/ec2-nitro-instances.html>`_ , the aggregation interval is always 60 seconds or less, regardless of the value that you specify. Default: 600
        :param tags: The tags to apply to the flow logs.
        :param traffic_type: The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic). This parameter is not supported for transit gateway resource types. It is required for the other resource types.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            # destination_options: Any
            
            cfn_flow_log_props = ec2.CfnFlowLogProps(
                resource_id="resourceId",
                resource_type="resourceType",
            
                # the properties below are optional
                deliver_cross_account_role="deliverCrossAccountRole",
                deliver_logs_permission_arn="deliverLogsPermissionArn",
                destination_options=destination_options,
                log_destination="logDestination",
                log_destination_type="logDestinationType",
                log_format="logFormat",
                log_group_name="logGroupName",
                max_aggregation_interval=123,
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                traffic_type="trafficType"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b70808589b9adfc66e939b8c87268607775dc0e479da5f4bed72c3ba8e3afef5)
            check_type(argname="argument resource_id", value=resource_id, expected_type=type_hints["resource_id"])
            check_type(argname="argument resource_type", value=resource_type, expected_type=type_hints["resource_type"])
            check_type(argname="argument deliver_cross_account_role", value=deliver_cross_account_role, expected_type=type_hints["deliver_cross_account_role"])
            check_type(argname="argument deliver_logs_permission_arn", value=deliver_logs_permission_arn, expected_type=type_hints["deliver_logs_permission_arn"])
            check_type(argname="argument destination_options", value=destination_options, expected_type=type_hints["destination_options"])
            check_type(argname="argument log_destination", value=log_destination, expected_type=type_hints["log_destination"])
            check_type(argname="argument log_destination_type", value=log_destination_type, expected_type=type_hints["log_destination_type"])
            check_type(argname="argument log_format", value=log_format, expected_type=type_hints["log_format"])
            check_type(argname="argument log_group_name", value=log_group_name, expected_type=type_hints["log_group_name"])
            check_type(argname="argument max_aggregation_interval", value=max_aggregation_interval, expected_type=type_hints["max_aggregation_interval"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument traffic_type", value=traffic_type, expected_type=type_hints["traffic_type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "resource_id": resource_id,
            "resource_type": resource_type,
        }
        if deliver_cross_account_role is not None:
            self._values["deliver_cross_account_role"] = deliver_cross_account_role
        if deliver_logs_permission_arn is not None:
            self._values["deliver_logs_permission_arn"] = deliver_logs_permission_arn
        if destination_options is not None:
            self._values["destination_options"] = destination_options
        if log_destination is not None:
            self._values["log_destination"] = log_destination
        if log_destination_type is not None:
            self._values["log_destination_type"] = log_destination_type
        if log_format is not None:
            self._values["log_format"] = log_format
        if log_group_name is not None:
            self._values["log_group_name"] = log_group_name
        if max_aggregation_interval is not None:
            self._values["max_aggregation_interval"] = max_aggregation_interval
        if tags is not None:
            self._values["tags"] = tags
        if traffic_type is not None:
            self._values["traffic_type"] = traffic_type

    @builtins.property
    def resource_id(self) -> builtins.str:
        '''The ID of the resource to monitor.

        For example, if the resource type is ``VPC`` , specify the ID of the VPC.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourceid
        '''
        result = self._values.get("resource_id")
        assert result is not None, "Required property 'resource_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def resource_type(self) -> builtins.str:
        '''The type of resource to monitor.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourcetype
        '''
        result = self._values.get("resource_type")
        assert result is not None, "Required property 'resource_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def deliver_cross_account_role(self) -> typing.Optional[builtins.str]:
        '''The ARN of the IAM role that allows the service to publish flow logs across accounts.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-delivercrossaccountrole
        '''
        result = self._values.get("deliver_cross_account_role")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def deliver_logs_permission_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the IAM role that allows Amazon EC2 to publish flow logs to the log destination.

        This parameter is required if the destination type is ``cloud-watch-logs`` , or if the destination type is ``kinesis-data-firehose`` and the delivery stream and the resources to monitor are in different accounts.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-deliverlogspermissionarn
        '''
        result = self._values.get("deliver_logs_permission_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def destination_options(self) -> typing.Any:
        '''The destination options.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-destinationoptions
        '''
        result = self._values.get("destination_options")
        return typing.cast(typing.Any, result)

    @builtins.property
    def log_destination(self) -> typing.Optional[builtins.str]:
        '''The destination for the flow log data. The meaning of this parameter depends on the destination type.

        - If the destination type is ``cloud-watch-logs`` , specify the ARN of a CloudWatch Logs log group. For example:

        arn:aws:logs: *region* : *account_id* :log-group: *my_group*

        Alternatively, use the ``LogGroupName`` parameter.

        - If the destination type is ``s3`` , specify the ARN of an S3 bucket. For example:

        arn:aws:s3::: *my_bucket* / *my_subfolder* /

        The subfolder is optional. Note that you can't use ``AWSLogs`` as a subfolder name.

        - If the destination type is ``kinesis-data-firehose`` , specify the ARN of a Kinesis Data Firehose delivery stream. For example:

        arn:aws:firehose: *region* : *account_id* :deliverystream: *my_stream*

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestination
        '''
        result = self._values.get("log_destination")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_destination_type(self) -> typing.Optional[builtins.str]:
        '''The type of destination for the flow log data.

        Default: ``cloud-watch-logs``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestinationtype
        '''
        result = self._values.get("log_destination_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_format(self) -> typing.Optional[builtins.str]:
        '''The fields to include in the flow log record, in the order in which they should appear.

        If you omit this parameter, the flow log is created using the default format. If you specify this parameter, you must include at least one field. For more information about the available fields, see `Flow log records <https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records>`_ in the *Amazon VPC User Guide* or `Transit Gateway Flow Log records <https://docs.aws.amazon.com/vpc/latest/tgw/tgw-flow-logs.html#flow-log-records>`_ in the *AWS Transit Gateway Guide* .

        Specify the fields using the ``${field-id}`` format, separated by spaces.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logformat
        '''
        result = self._values.get("log_format")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_group_name(self) -> typing.Optional[builtins.str]:
        '''The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs.

        This parameter is valid only if the destination type is ``cloud-watch-logs`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-loggroupname
        '''
        result = self._values.get("log_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_aggregation_interval(self) -> typing.Optional[jsii.Number]:
        '''The maximum interval of time during which a flow of packets is captured and aggregated into a flow log record.

        The possible values are 60 seconds (1 minute) or 600 seconds (10 minutes). This parameter must be 60 seconds for transit gateway resource types.

        When a network interface is attached to a `Nitro-based instance <https://docs.aws.amazon.com/ec2/latest/instancetypes/ec2-nitro-instances.html>`_ , the aggregation interval is always 60 seconds or less, regardless of the value that you specify.

        Default: 600

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-maxaggregationinterval
        '''
        result = self._values.get("max_aggregation_interval")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The tags to apply to the flow logs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def traffic_type(self) -> typing.Optional[builtins.str]:
        '''The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic).

        This parameter is not supported for transit gateway resource types. It is required for the other resource types.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-traffictype
        '''
        result = self._values.get("traffic_type")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnFlowLogProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnGatewayRouteTableAssociation(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnGatewayRouteTableAssociation",
):
    '''Associates a virtual private gateway or internet gateway with a route table.

    The gateway and route table must be in the same VPC. This association causes the incoming traffic to the gateway to be routed according to the routes in the route table.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html
    :cloudformationResource: AWS::EC2::GatewayRouteTableAssociation
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_gateway_route_table_association = ec2.CfnGatewayRouteTableAssociation(self, "MyCfnGatewayRouteTableAssociation",
            gateway_id="gatewayId",
            route_table_id="routeTableId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        gateway_id: builtins.str,
        route_table_id: builtins.str,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param gateway_id: The ID of the gateway.
        :param route_table_id: The ID of the route table.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4b6e9c558b84db03ce7f87c5229b1660a182dbcc83eb7b603939ad024b053290)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnGatewayRouteTableAssociationProps(
            gateway_id=gateway_id, route_table_id=route_table_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__210fd654bd3dbc467b26765ed18c099ae7b5043c293d926cade19fd3802bd80f)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9b18f652516d55efdc0bbaf9e0aed3e5e7b9ed6a627018605ad3a8ac40bcfb08)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrAssociationId")
    def attr_association_id(self) -> builtins.str:
        '''The ID of the route table association.

        :cloudformationAttribute: AssociationId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAssociationId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="gatewayId")
    def gateway_id(self) -> builtins.str:
        '''The ID of the gateway.'''
        return typing.cast(builtins.str, jsii.get(self, "gatewayId"))

    @gateway_id.setter
    def gateway_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b80b9e3979b7f3bc56f10b10d915c83ac792f4ea63ad245c84317b275a65523d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "gatewayId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="routeTableId")
    def route_table_id(self) -> builtins.str:
        '''The ID of the route table.'''
        return typing.cast(builtins.str, jsii.get(self, "routeTableId"))

    @route_table_id.setter
    def route_table_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9de1cff2e9acdaccf461af9bb29e1e72246de3e47124324639ada6d2b19a81f4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routeTableId", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnGatewayRouteTableAssociationProps",
    jsii_struct_bases=[],
    name_mapping={"gateway_id": "gatewayId", "route_table_id": "routeTableId"},
)
class CfnGatewayRouteTableAssociationProps:
    def __init__(
        self,
        *,
        gateway_id: builtins.str,
        route_table_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnGatewayRouteTableAssociation``.

        :param gateway_id: The ID of the gateway.
        :param route_table_id: The ID of the route table.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_gateway_route_table_association_props = ec2.CfnGatewayRouteTableAssociationProps(
                gateway_id="gatewayId",
                route_table_id="routeTableId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__17bdeaef71282588c502b071ec9708033fec42cd59c61e718eb5899520703999)
            check_type(argname="argument gateway_id", value=gateway_id, expected_type=type_hints["gateway_id"])
            check_type(argname="argument route_table_id", value=route_table_id, expected_type=type_hints["route_table_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "gateway_id": gateway_id,
            "route_table_id": route_table_id,
        }

    @builtins.property
    def gateway_id(self) -> builtins.str:
        '''The ID of the gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html#cfn-ec2-gatewayroutetableassociation-gatewayid
        '''
        result = self._values.get("gateway_id")
        assert result is not None, "Required property 'gateway_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def route_table_id(self) -> builtins.str:
        '''The ID of the route table.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html#cfn-ec2-gatewayroutetableassociation-routetableid
        '''
        result = self._values.get("route_table_id")
        assert result is not None, "Required property 'route_table_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGatewayRouteTableAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnHost(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnHost",
):
    '''Allocates a fully dedicated physical server for launching EC2 instances.

    Because the host is fully dedicated for your use, it can help you address compliance requirements and reduce costs by allowing you to use your existing server-bound software licenses. For more information, see `Dedicated Hosts <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-overview.html>`_ in the *Amazon EC2 User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html
    :cloudformationResource: AWS::EC2::Host
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_host = ec2.CfnHost(self, "MyCfnHost",
            availability_zone="availabilityZone",
        
            # the properties below are optional
            asset_id="assetId",
            auto_placement="autoPlacement",
            host_maintenance="hostMaintenance",
            host_recovery="hostRecovery",
            instance_family="instanceFamily",
            instance_type="instanceType",
            outpost_arn="outpostArn"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        availability_zone: builtins.str,
        asset_id: typing.Optional[builtins.str] = None,
        auto_placement: typing.Optional[builtins.str] = None,
        host_maintenance: typing.Optional[builtins.str] = None,
        host_recovery: typing.Optional[builtins.str] = None,
        instance_family: typing.Optional[builtins.str] = None,
        instance_type: typing.Optional[builtins.str] = None,
        outpost_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param availability_zone: The Availability Zone in which to allocate the Dedicated Host.
        :param asset_id: The ID of the Outpost hardware asset on which the Dedicated Host is allocated.
        :param auto_placement: Indicates whether the host accepts any untargeted instance launches that match its instance type configuration, or if it only accepts Host tenancy instance launches that specify its unique host ID. For more information, see `Understanding auto-placement and affinity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/how-dedicated-hosts-work.html#dedicated-hosts-understanding>`_ in the *Amazon EC2 User Guide* . Default: ``off``
        :param host_maintenance: Indicates whether host maintenance is enabled or disabled for the Dedicated Host.
        :param host_recovery: Indicates whether to enable or disable host recovery for the Dedicated Host. Host recovery is disabled by default. For more information, see `Host recovery <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-recovery.html>`_ in the *Amazon EC2 User Guide* . Default: ``off``
        :param instance_family: The instance family supported by the Dedicated Host. For example, ``m5`` .
        :param instance_type: Specifies the instance type to be supported by the Dedicated Hosts. If you specify an instance type, the Dedicated Hosts support instances of the specified instance type only.
        :param outpost_arn: The Amazon Resource Name (ARN) of the AWS Outpost on which the Dedicated Host is allocated.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e3a07acffdb551edbc817b7c424628c812f21356d7f697757a332323f6dcfde8)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnHostProps(
            availability_zone=availability_zone,
            asset_id=asset_id,
            auto_placement=auto_placement,
            host_maintenance=host_maintenance,
            host_recovery=host_recovery,
            instance_family=instance_family,
            instance_type=instance_type,
            outpost_arn=outpost_arn,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__62948130163263864ea13dc24c244d1ba844fcdcbf6e72536422cfe1f035d0ae)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2f6d598529fd0d16119fff477306582a339f6d734027d9e85531c72b90654a41)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrHostId")
    def attr_host_id(self) -> builtins.str:
        '''The ID of the host.

        :cloudformationAttribute: HostId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrHostId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="availabilityZone")
    def availability_zone(self) -> builtins.str:
        '''The Availability Zone in which to allocate the Dedicated Host.'''
        return typing.cast(builtins.str, jsii.get(self, "availabilityZone"))

    @availability_zone.setter
    def availability_zone(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8f2943fd7038584c6a32abc7ac0a93f4b0f4c63881cfa72150bd73ada16642d0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "availabilityZone", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="assetId")
    def asset_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the Outpost hardware asset on which the Dedicated Host is allocated.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "assetId"))

    @asset_id.setter
    def asset_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ef92a5203275a4f9d2ce9835a43f6cd82cac31cb59c1d0ce172cbfc9e50fbefb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "assetId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="autoPlacement")
    def auto_placement(self) -> typing.Optional[builtins.str]:
        '''Indicates whether the host accepts any untargeted instance launches that match its instance type configuration, or if it only accepts Host tenancy instance launches that specify its unique host ID.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "autoPlacement"))

    @auto_placement.setter
    def auto_placement(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__847519b23588450ab8d14c5778b80f32cdf5fa3f1ca8c772c0b129d3a9ad3c32)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoPlacement", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="hostMaintenance")
    def host_maintenance(self) -> typing.Optional[builtins.str]:
        '''Indicates whether host maintenance is enabled or disabled for the Dedicated Host.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "hostMaintenance"))

    @host_maintenance.setter
    def host_maintenance(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5da78a5d765050694e66dbb10d6bc046deaee511071b83699b9f3c2f2816c2b9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "hostMaintenance", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="hostRecovery")
    def host_recovery(self) -> typing.Optional[builtins.str]:
        '''Indicates whether to enable or disable host recovery for the Dedicated Host.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "hostRecovery"))

    @host_recovery.setter
    def host_recovery(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b92e78abd69ef3031b103aed1c9826c3000239f5d58022c4cb1dd256f65a8da9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "hostRecovery", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="instanceFamily")
    def instance_family(self) -> typing.Optional[builtins.str]:
        '''The instance family supported by the Dedicated Host.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceFamily"))

    @instance_family.setter
    def instance_family(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__17d80a761d1af3dea3988a895a0ef398f8f899481397c0e52127d20797de751e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceFamily", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> typing.Optional[builtins.str]:
        '''Specifies the instance type to be supported by the Dedicated Hosts.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceType"))

    @instance_type.setter
    def instance_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2c99a5cb989b1890ba7a9c45d3bf3f171871e622316f2e23f45489e55fae7586)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instanceType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="outpostArn")
    def outpost_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the AWS Outpost on which the Dedicated Host is allocated.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "outpostArn"))

    @outpost_arn.setter
    def outpost_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e93e24d3ea88403b39f7d948c7495b074446c1b676d25cd60fc9a3a558e7f36)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "outpostArn", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnHostProps",
    jsii_struct_bases=[],
    name_mapping={
        "availability_zone": "availabilityZone",
        "asset_id": "assetId",
        "auto_placement": "autoPlacement",
        "host_maintenance": "hostMaintenance",
        "host_recovery": "hostRecovery",
        "instance_family": "instanceFamily",
        "instance_type": "instanceType",
        "outpost_arn": "outpostArn",
    },
)
class CfnHostProps:
    def __init__(
        self,
        *,
        availability_zone: builtins.str,
        asset_id: typing.Optional[builtins.str] = None,
        auto_placement: typing.Optional[builtins.str] = None,
        host_maintenance: typing.Optional[builtins.str] = None,
        host_recovery: typing.Optional[builtins.str] = None,
        instance_family: typing.Optional[builtins.str] = None,
        instance_type: typing.Optional[builtins.str] = None,
        outpost_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnHost``.

        :param availability_zone: The Availability Zone in which to allocate the Dedicated Host.
        :param asset_id: The ID of the Outpost hardware asset on which the Dedicated Host is allocated.
        :param auto_placement: Indicates whether the host accepts any untargeted instance launches that match its instance type configuration, or if it only accepts Host tenancy instance launches that specify its unique host ID. For more information, see `Understanding auto-placement and affinity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/how-dedicated-hosts-work.html#dedicated-hosts-understanding>`_ in the *Amazon EC2 User Guide* . Default: ``off``
        :param host_maintenance: Indicates whether host maintenance is enabled or disabled for the Dedicated Host.
        :param host_recovery: Indicates whether to enable or disable host recovery for the Dedicated Host. Host recovery is disabled by default. For more information, see `Host recovery <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-recovery.html>`_ in the *Amazon EC2 User Guide* . Default: ``off``
        :param instance_family: The instance family supported by the Dedicated Host. For example, ``m5`` .
        :param instance_type: Specifies the instance type to be supported by the Dedicated Hosts. If you specify an instance type, the Dedicated Hosts support instances of the specified instance type only.
        :param outpost_arn: The Amazon Resource Name (ARN) of the AWS Outpost on which the Dedicated Host is allocated.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_host_props = ec2.CfnHostProps(
                availability_zone="availabilityZone",
            
                # the properties below are optional
                asset_id="assetId",
                auto_placement="autoPlacement",
                host_maintenance="hostMaintenance",
                host_recovery="hostRecovery",
                instance_family="instanceFamily",
                instance_type="instanceType",
                outpost_arn="outpostArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6b2753a5bf48a7bda574bdc6bf8ca7f9c31c7e48329df5f793f75cfb822ea308)
            check_type(argname="argument availability_zone", value=availability_zone, expected_type=type_hints["availability_zone"])
            check_type(argname="argument asset_id", value=asset_id, expected_type=type_hints["asset_id"])
            check_type(argname="argument auto_placement", value=auto_placement, expected_type=type_hints["auto_placement"])
            check_type(argname="argument host_maintenance", value=host_maintenance, expected_type=type_hints["host_maintenance"])
            check_type(argname="argument host_recovery", value=host_recovery, expected_type=type_hints["host_recovery"])
            check_type(argname="argument instance_family", value=instance_family, expected_type=type_hints["instance_family"])
            check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
            check_type(argname="argument outpost_arn", value=outpost_arn, expected_type=type_hints["outpost_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "availability_zone": availability_zone,
        }
        if asset_id is not None:
            self._values["asset_id"] = asset_id
        if auto_placement is not None:
            self._values["auto_placement"] = auto_placement
        if host_maintenance is not None:
            self._values["host_maintenance"] = host_maintenance
        if host_recovery is not None:
            self._values["host_recovery"] = host_recovery
        if instance_family is not None:
            self._values["instance_family"] = instance_family
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if outpost_arn is not None:
            self._values["outpost_arn"] = outpost_arn

    @builtins.property
    def availability_zone(self) -> builtins.str:
        '''The Availability Zone in which to allocate the Dedicated Host.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-availabilityzone
        '''
        result = self._values.get("availability_zone")
        assert result is not None, "Required property 'availability_zone' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def asset_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the Outpost hardware asset on which the Dedicated Host is allocated.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-assetid
        '''
        result = self._values.get("asset_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auto_placement(self) -> typing.Optional[builtins.str]:
        '''Indicates whether the host accepts any untargeted instance launches that match its instance type configuration, or if it only accepts Host tenancy instance launches that specify its unique host ID.

        For more information, see `Understanding auto-placement and affinity <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/how-dedicated-hosts-work.html#dedicated-hosts-understanding>`_ in the *Amazon EC2 User Guide* .

        Default: ``off``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-autoplacement
        '''
        result = self._values.get("auto_placement")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def host_maintenance(self) -> typing.Optional[builtins.str]:
        '''Indicates whether host maintenance is enabled or disabled for the Dedicated Host.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-hostmaintenance
        '''
        result = self._values.get("host_maintenance")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def host_recovery(self) -> typing.Optional[builtins.str]:
        '''Indicates whether to enable or disable host recovery for the Dedicated Host.

        Host recovery is disabled by default. For more information, see `Host recovery <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-recovery.html>`_ in the *Amazon EC2 User Guide* .

        Default: ``off``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-hostrecovery
        '''
        result = self._values.get("host_recovery")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_family(self) -> typing.Optional[builtins.str]:
        '''The instance family supported by the Dedicated Host.

        For example, ``m5`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-instancefamily
        '''
        result = self._values.get("instance_family")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_type(self) -> typing.Optional[builtins.str]:
        '''Specifies the instance type to be supported by the Dedicated Hosts.

        If you specify an instance type, the Dedicated Hosts support instances of the specified instance type only.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-instancetype
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outpost_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the AWS Outpost on which the Dedicated Host is allocated.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-outpostarn
        '''
        result = self._values.get("outpost_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnHostProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnIPAM(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnIPAM",
):
    '''IPAM is a VPC feature that you can use to automate your IP address management workflows including assigning, tracking, troubleshooting, and auditing IP addresses across AWS Regions and accounts throughout your AWS Organization.

    For more information, see `What is IPAM? <https://docs.aws.amazon.com//vpc/latest/ipam/what-is-it-ipam.html>`_ in the *Amazon VPC IPAM User Guide* .

    There are AWS Identity and Access Management (IAM) permissions required to fully manage an IPAM in CloudFormation. For more information, see `Example policy <https://docs.aws.amazon.com//vpc/latest/ipam/iam-ipam-policy-examples.html>`_ in the *Amazon VPC IPAM User Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipam.html
    :cloudformationResource: AWS::EC2::IPAM
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_iPAM = ec2.CfnIPAM(self, "MyCfnIPAM",
            default_resource_discovery_organizational_unit_exclusions=[ec2.CfnIPAM.IpamOrganizationalUnitExclusionProperty(
                organizations_entity_path="organizationsEntityPath"
            )],
            description="description",
            enable_private_gua=False,
            operating_regions=[ec2.CfnIPAM.IpamOperatingRegionProperty(
                region_name="regionName"
            )],
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            tier="tier"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        default_resource_discovery_organizational_unit_exclusions: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnIPAM.IpamOrganizationalUnitExclusionProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        description: typing.Optional[builtins.str] = None,
        enable_private_gua: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        operating_regions: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnIPAM.IpamOperatingRegionProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        tier: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param default_resource_discovery_organizational_unit_exclusions: If your IPAM is integrated with AWS Organizations, you can exclude an `organizational unit (OU) <https://docs.aws.amazon.com/organizations/latest/userguide/orgs_getting-started_concepts.html#organizationalunit>`_ from being managed by IPAM. When you exclude an OU, IPAM will not manage the IP addresses in accounts in that OU. For more information, see `Exclude organizational units from IPAM <https://docs.aws.amazon.com/vpc/latest/ipam/exclude-ous.html>`_ in the *Amazon Virtual Private Cloud IP Address Manager User Guide* .
        :param description: The description for the IPAM.
        :param enable_private_gua: Enable this option to use your own GUA ranges as private IPv6 addresses. This option is disabled by default.
        :param operating_regions: The operating Regions for an IPAM. Operating Regions are AWS Regions where the IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the AWS Regions you select as operating Regions. For more information about operating Regions, see `Create an IPAM <https://docs.aws.amazon.com//vpc/latest/ipam/create-ipam.html>`_ in the *Amazon VPC IPAM User Guide* .
        :param tags: The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.
        :param tier: IPAM is offered in a Free Tier and an Advanced Tier. For more information about the features available in each tier and the costs associated with the tiers, see the `VPC IPAM product pricing page <https://docs.aws.amazon.com//vpc/pricing/>`_ .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5dd8d015864426e689ac2f72f1fdd70371d242931964ab4d571ea56017f00045)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnIPAMProps(
            default_resource_discovery_organizational_unit_exclusions=default_resource_discovery_organizational_unit_exclusions,
            description=description,
            enable_private_gua=enable_private_gua,
            operating_regions=operating_regions,
            tags=tags,
            tier=tier,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__52d60de86ca7ba70a9845a22c132a50b0a472c7cce8b00d8e266f850b9c0e6f4)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__901ef9223e06c12f769b4bd36857fd88adae614a05385d54a0dd6f535824d002)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The ARN of the IPAM.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrDefaultResourceDiscoveryAssociationId")
    def attr_default_resource_discovery_association_id(self) -> builtins.str:
        '''The ID of the default resource discovery association.

        :cloudformationAttribute: DefaultResourceDiscoveryAssociationId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDefaultResourceDiscoveryAssociationId"))

    @builtins.property
    @jsii.member(jsii_name="attrDefaultResourceDiscoveryId")
    def attr_default_resource_discovery_id(self) -> builtins.str:
        '''The ID of the default resource discovery.

        :cloudformationAttribute: DefaultResourceDiscoveryId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDefaultResourceDiscoveryId"))

    @builtins.property
    @jsii.member(jsii_name="attrIpamId")
    def attr_ipam_id(self) -> builtins.str:
        '''The ID of the IPAM.

        :cloudformationAttribute: IpamId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamId"))

    @builtins.property
    @jsii.member(jsii_name="attrPrivateDefaultScopeId")
    def attr_private_default_scope_id(self) -> builtins.str:
        '''The ID of the default private scope.

        :cloudformationAttribute: PrivateDefaultScopeId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPrivateDefaultScopeId"))

    @builtins.property
    @jsii.member(jsii_name="attrPublicDefaultScopeId")
    def attr_public_default_scope_id(self) -> builtins.str:
        '''The ID of the default public scope.

        :cloudformationAttribute: PublicDefaultScopeId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPublicDefaultScopeId"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceDiscoveryAssociationCount")
    def attr_resource_discovery_association_count(self) -> jsii.Number:
        '''The number of resource discovery associations.

        :cloudformationAttribute: ResourceDiscoveryAssociationCount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrResourceDiscoveryAssociationCount"))

    @builtins.property
    @jsii.member(jsii_name="attrScopeCount")
    def attr_scope_count(self) -> jsii.Number:
        '''The number of scopes.

        :cloudformationAttribute: ScopeCount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrScopeCount"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="defaultResourceDiscoveryOrganizationalUnitExclusions")
    def default_resource_discovery_organizational_unit_exclusions(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIPAM.IpamOrganizationalUnitExclusionProperty"]]]]:
        '''If your IPAM is integrated with AWS Organizations, you can exclude an `organizational unit (OU) <https://docs.aws.amazon.com/organizations/latest/userguide/orgs_getting-started_concepts.html#organizationalunit>`_ from being managed by IPAM. When you exclude an OU, IPAM will not manage the IP addresses in accounts in that OU. For more information, see `Exclude organizational units from IPAM <https://docs.aws.amazon.com/vpc/latest/ipam/exclude-ous.html>`_ in the *Amazon Virtual Private Cloud IP Address Manager User Guide* .'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIPAM.IpamOrganizationalUnitExclusionProperty"]]]], jsii.get(self, "defaultResourceDiscoveryOrganizationalUnitExclusions"))

    @default_resource_discovery_organizational_unit_exclusions.setter
    def default_resource_discovery_organizational_unit_exclusions(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIPAM.IpamOrganizationalUnitExclusionProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__df7dc542d06b2df113e30582e811e678d8d8aa54b5759d2ebb6ea3563a53a791)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "defaultResourceDiscoveryOrganizationalUnitExclusions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description for the IPAM.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__959c716c17871ad25c09c928583239d9973366c81c48b69bfc39c5c2a70898ce)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="enablePrivateGua")
    def enable_private_gua(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Enable this option to use your own GUA ranges as private IPv6 addresses.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enablePrivateGua"))

    @enable_private_gua.setter
    def enable_private_gua(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1281235c91c77dfae8d509881b5159b846e5e23d4ec106d47ef850482a5d350b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enablePrivateGua", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="operatingRegions")
    def operating_regions(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIPAM.IpamOperatingRegionProperty"]]]]:
        '''The operating Regions for an IPAM.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIPAM.IpamOperatingRegionProperty"]]]], jsii.get(self, "operatingRegions"))

    @operating_regions.setter
    def operating_regions(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIPAM.IpamOperatingRegionProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fecbb17dcfafe289f5a442dc3b73f564f5de07433cc7e26b8e86425af382451f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "operatingRegions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The key/value combination of a tag assigned to the resource.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6e1f584e5fc1ac961e5449d71a22aa5c200ae3632e3bd5a9caba51ea5d38dde7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tier")
    def tier(self) -> typing.Optional[builtins.str]:
        '''IPAM is offered in a Free Tier and an Advanced Tier.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "tier"))

    @tier.setter
    def tier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2c8cecf24bfc36bfa727b2bcc8526f504fce30463cb6dcc5d49d5b9d67749d5e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tier", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnIPAM.IpamOperatingRegionProperty",
        jsii_struct_bases=[],
        name_mapping={"region_name": "regionName"},
    )
    class IpamOperatingRegionProperty:
        def __init__(self, *, region_name: builtins.str) -> None:
            '''The operating Regions for an IPAM.

            Operating Regions are AWS Regions where the IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the AWS Regions you select as operating Regions.

            For more information about operating Regions, see `Create an IPAM <https://docs.aws.amazon.com//vpc/latest/ipam/create-ipam.html>`_ in the *Amazon VPC IPAM User Guide* .

            :param region_name: The name of the operating Region.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipam-ipamoperatingregion.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                ipam_operating_region_property = ec2.CfnIPAM.IpamOperatingRegionProperty(
                    region_name="regionName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__76ae7ea9478fca86299eafd84b20c22db0b753cea0096278a3be708fdccb0c7d)
                check_type(argname="argument region_name", value=region_name, expected_type=type_hints["region_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "region_name": region_name,
            }

        @builtins.property
        def region_name(self) -> builtins.str:
            '''The name of the operating Region.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipam-ipamoperatingregion.html#cfn-ec2-ipam-ipamoperatingregion-regionname
            '''
            result = self._values.get("region_name")
            assert result is not None, "Required property 'region_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IpamOperatingRegionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnIPAM.IpamOrganizationalUnitExclusionProperty",
        jsii_struct_bases=[],
        name_mapping={"organizations_entity_path": "organizationsEntityPath"},
    )
    class IpamOrganizationalUnitExclusionProperty:
        def __init__(self, *, organizations_entity_path: builtins.str) -> None:
            '''If your IPAM is integrated with AWS Organizations, you can exclude an `organizational unit (OU) <https://docs.aws.amazon.com/organizations/latest/userguide/orgs_getting-started_concepts.html#organizationalunit>`_ from being managed by IPAM. When you exclude an OU, IPAM will not manage the IP addresses in accounts in that OU. For more information, see `Exclude organizational units from IPAM <https://docs.aws.amazon.com/vpc/latest/ipam/exclude-ous.html>`_ in the *Amazon Virtual Private Cloud IP Address Manager User Guide* .

            :param organizations_entity_path: An AWS Organizations entity path. For more information on the entity path, see `Understand the AWS Organizations entity path <https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_last-accessed-view-data-orgs.html#access_policies_access-advisor-viewing-orgs-entity-path>`_ in the *AWS Identity and Access Management User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipam-ipamorganizationalunitexclusion.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                ipam_organizational_unit_exclusion_property = ec2.CfnIPAM.IpamOrganizationalUnitExclusionProperty(
                    organizations_entity_path="organizationsEntityPath"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3a1e082fe403508d7f87288dfed02281b9ed00551965db1ff48cbbe30d9d4b25)
                check_type(argname="argument organizations_entity_path", value=organizations_entity_path, expected_type=type_hints["organizations_entity_path"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "organizations_entity_path": organizations_entity_path,
            }

        @builtins.property
        def organizations_entity_path(self) -> builtins.str:
            '''An AWS Organizations entity path.

            For more information on the entity path, see `Understand the AWS Organizations entity path <https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_last-accessed-view-data-orgs.html#access_policies_access-advisor-viewing-orgs-entity-path>`_ in the *AWS Identity and Access Management User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipam-ipamorganizationalunitexclusion.html#cfn-ec2-ipam-ipamorganizationalunitexclusion-organizationsentitypath
            '''
            result = self._values.get("organizations_entity_path")
            assert result is not None, "Required property 'organizations_entity_path' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IpamOrganizationalUnitExclusionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556)
class CfnIPAMAllocation(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnIPAMAllocation",
):
    '''In IPAM, an allocation is a CIDR assignment from an IPAM pool to another IPAM pool or to a resource.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html
    :cloudformationResource: AWS::EC2::IPAMAllocation
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_iPAMAllocation = ec2.CfnIPAMAllocation(self, "MyCfnIPAMAllocation",
            ipam_pool_id="ipamPoolId",
        
            # the properties below are optional
            cidr="cidr",
            description="description",
            netmask_length=123
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        ipam_pool_id: builtins.str,
        cidr: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        netmask_length: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param ipam_pool_id: The ID of the IPAM pool from which you would like to allocate a CIDR.
        :param cidr: The CIDR you would like to allocate from the IPAM pool. Note the following:. - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR. - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored. Possible values: Any available IPv4 or IPv6 CIDR.
        :param description: A description for the allocation.
        :param netmask_length: The netmask length of the CIDR you would like to allocate from the IPAM pool. Note the following:. - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR. - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea7a8d7dc762f36c7c42acf390ff33a5d901404e11e9316d7df8d7c29c9e6af0)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnIPAMAllocationProps(
            ipam_pool_id=ipam_pool_id,
            cidr=cidr,
            description=description,
            netmask_length=netmask_length,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3ba0f7e823a6d4a60a4cb169b078ea9c23b3c54c729c78632914b49c5f2a4751)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ad213757c25805fb10120b0939b5a219ff19cbe5f92a4728e9e1a0444c5c64dd)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrIpamPoolAllocationId")
    def attr_ipam_pool_allocation_id(self) -> builtins.str:
        '''The ID of an allocation.

        :cloudformationAttribute: IpamPoolAllocationId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamPoolAllocationId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="ipamPoolId")
    def ipam_pool_id(self) -> builtins.str:
        '''The ID of the IPAM pool from which you would like to allocate a CIDR.'''
        return typing.cast(builtins.str, jsii.get(self, "ipamPoolId"))

    @ipam_pool_id.setter
    def ipam_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__64e2f4cf4e3421886a498bb9e45ea019547a5f8e087690bbefc7fcea3586b903)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ipamPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="cidr")
    def cidr(self) -> typing.Optional[builtins.str]:
        '''The CIDR you would like to allocate from the IPAM pool.

        Note the following:.
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cidr"))

    @cidr.setter
    def cidr(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b2e8a509e9710e3df3c5cdd45c6611dddda58579bcf820e4d97bf9783fc86238)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cidr", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description for the allocation.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ae562f8439586f83a0987704d36755f7df5f4a7e5c8e2420a6f2a05dd208d415)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="netmaskLength")
    def netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The netmask length of the CIDR you would like to allocate from the IPAM pool.

        Note the following:.
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "netmaskLength"))

    @netmask_length.setter
    def netmask_length(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9d9043b9cb409d41dda6d6f1b1ac81fccb06180f67f05749e5d26225f4844545)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "netmaskLength", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_ec2.CfnIPAMAllocationProps",
    jsii_struct_bases=[],
    name_mapping={
        "ipam_pool_id": "ipamPoolId",
        "cidr": "cidr",
        "description": "description",
        "netmask_length": "netmaskLength",
    },
)
class CfnIPAMAllocationProps:
    def __init__(
        self,
        *,
        ipam_pool_id: builtins.str,
        cidr: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        netmask_length: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnIPAMAllocation``.

        :param ipam_pool_id: The ID of the IPAM pool from which you would like to allocate a CIDR.
        :param cidr: The CIDR you would like to allocate from the IPAM pool. Note the following:. - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR. - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored. Possible values: Any available IPv4 or IPv6 CIDR.
        :param description: A description for the allocation.
        :param netmask_length: The netmask length of the CIDR you would like to allocate from the IPAM pool. Note the following:. - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR. - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_ec2 as ec2
            
            cfn_iPAMAllocation_props = ec2.CfnIPAMAllocationProps(
                ipam_pool_id="ipamPoolId",
            
                # the properties below are optional
                cidr="cidr",
                description="description",
                netmask_length=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f074bee976b182b3e6932081b7e2343fe058af055ef039867eee673737cd6914)
            check_type(argname="argument ipam_pool_id", value=ipam_pool_id, expected_type=type_hints["ipam_pool_id"])
            check_type(argname="argument cidr", value=cidr, expected_type=type_hints["cidr"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument netmask_length", value=netmask_length, expected_type=type_hints["netmask_length"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "ipam_pool_id": ipam_pool_id,
        }
        if cidr is not None:
            self._values["cidr"] = cidr
        if description is not None:
            self._values["description"] = description
        if netmask_length is not None:
            self._values["netmask_length"] = netmask_length

    @builtins.property
    def ipam_pool_id(self) -> builtins.str:
        '''The ID of the IPAM pool from which you would like to allocate a CIDR.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-ipampoolid
        '''
        result = self._values.get("ipam_pool_id")
        assert result is not None, "Required property 'ipam_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cidr(self) -> typing.Optional[builtins.str]:
        '''The CIDR you would like to allocate from the IPAM pool. Note the following:.

        - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR.
        - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored.

        Possible values: Any available IPv4 or IPv6 CIDR.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-cidr
        '''
        result = self._values.get("cidr")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description for the allocation.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The netmask length of the CIDR you would like to allocate from the IPAM pool. Note the following:.

        - If there is no DefaultNetmaskLength allocation rule set on the pool, you must specify either the NetmaskLength or the CIDR.
        - If the DefaultNetmaskLength allocation rule is set on the pool, you can specify either the NetmaskLength or the CIDR and the DefaultNetmaskLength allocation rule will be ignored.

        Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipamallocation.html#cfn-ec2-ipamallocation-netmasklength
        '''
        result = self._values.get("netmask_length")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnIPAMAllocationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnIPAMPool(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnIPAMPool",
):
    '''In IPAM, a pool is a collection of contiguous IP addresses CIDRs.

    Pools enable you to organize your IP addresses according to your routing and security needs. For example, if you have separate routing and security needs for development and production applications, you can create a pool for each.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampool.html
    :cloudformationResource: AWS::EC2::IPAMPool
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_ec2 as ec2
        
        cfn_iPAMPool = ec2.CfnIPAMPool(self, "MyCfnIPAMPool",
            address_family="addressFamily",
            ipam_scope_id="ipamScopeId",
        
            # the properties below are optional
            allocation_default_netmask_length=123,
            allocation_max_netmask_length=123,
            allocation_min_netmask_length=123,
            allocation_resource_tags=[CfnTag(
                key="key",
                value="value"
            )],
            auto_import=False,
            aws_service="awsService",
            description="description",
            locale="locale",
            provisioned_cidrs=[ec2.CfnIPAMPool.ProvisionedCidrProperty(
                cidr="cidr"
            )],
            public_ip_source="publicIpSource",
            publicly_advertisable=False,
            source_ipam_pool_id="sourceIpamPoolId",
            source_resource=ec2.CfnIPAMPool.SourceResourceProperty(
                resource_id="resourceId",
                resource_owner="resourceOwner",
                resource_region="resourceRegion",
                resource_type="resourceType"
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        address_family: builtins.str,
        ipam_scope_id: builtins.str,
        allocation_default_netmask_length: typing.Optional[jsii.Number] = None,
        allocation_max_netmask_length: typing.Optional[jsii.Number] = None,
        allocation_min_netmask_length: typing.Optional[jsii.Number] = None,
        allocation_resource_tags: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        auto_import: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        aws_service: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        locale: typing.Optional[builtins.str] = None,
        provisioned_cidrs: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnIPAMPool.ProvisionedCidrProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        public_ip_source: typing.Optional[builtins.str] = None,
        publicly_advertisable: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        source_ipam_pool_id: typing.Optional[builtins.str] = None,
        source_resource: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnIPAMPool.SourceResourceProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param address_family: The address family of the pool.
        :param ipam_scope_id: The ID of the scope in which you would like to create the IPAM pool.
        :param allocation_default_netmask_length: The default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.
        :param allocation_max_netmask_length: The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant. The maximum netmask length must be greater than the minimum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
        :param allocation_min_netmask_length: The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant. The minimum netmask length must be less than the maximum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
        :param allocation_resource_tags: Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be marked as noncompliant.
        :param auto_import: If selected, IPAM will continuously look for resources within the CIDR range of this pool and automatically import them as allocations into your IPAM. The CIDRs that will be allocated for these resources must not already be allocated to other resources in order for the import to succeed. IPAM will import a CIDR regardless of its compliance with the pool's allocation rules, so a resource might be imported and subsequently marked as noncompliant. If IPAM discovers multiple CIDRs that overlap, IPAM will import the largest CIDR only. If IPAM discovers multiple CIDRs with matching CIDRs, IPAM will randomly import one of them only. A locale must be set on the pool for this feature to work.
        :param aws_service: Limits which service in AWS that the pool can be used in. "ec2", for example, allows users to use space for Elastic IP addresses and VPCs.
        :param description: The description of the IPAM pool.
        :param locale: The locale of the IPAM pool. The locale for the pool should be one of the following: - An AWS Region where you want this IPAM pool to be available for allocations. - The network border group for an AWS Local Zone where you want this IPAM pool to be available for allocations ( `supported Local Zones <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail>`_ ). This option is only available for IPAM IPv4 pools in the public scope. If you choose an AWS Region for locale that has not been configured as an operating Region for the IPAM, you'll get an error.
        :param provisioned_cidrs: Information about the CIDRs provisioned to an IPAM pool.
        :param public_ip_source: The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in the public scope. Default is ``BYOIP`` . For more information, see `Create IPv6 pools <https://docs.aws.amazon.com//vpc/latest/ipam/intro-create-ipv6-pools.html>`_ in the *Amazon VPC IPAM User Guide* . By default, you can add only one Amazon-provided IPv6 CIDR block to a top-level IPv6 pool. For information on increasing the default limit, see `Quotas for your IPAM <https://docs.aws.amazon.com//vpc/latest/ipam/quotas-ipam.html>`_ in the *Amazon VPC IPAM User Guide* .
        :param publicly_advertisable: Determines if a pool is publicly advertisable. This option is not available for pools with AddressFamily set to ``ipv4`` .
        :param source_ipam_pool_id: The ID of the source IPAM pool. You can use this option to create an IPAM pool within an existing source pool.
        :param source_resource: The resource used to provision CIDRs to a resource planning pool.
        :param tags: The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value as the filter value. For example, to find all resources that have a tag with the key ``Owner`` and the value ``TeamA`` , specify ``tag:Owner`` for the filter name and ``TeamA`` for the filter value.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__49ac69dacc9e55c18363e598ee436fae30723dd4b487a4cd51477854696d9739)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnIPAMPoolProps(
            address_family=address_family,
            ipam_scope_id=ipam_scope_id,
            allocation_default_netmask_length=allocation_default_netmask_length,
            allocation_max_netmask_length=allocation_max_netmask_length,
            allocation_min_netmask_length=allocation_min_netmask_length,
            allocation_resource_tags=allocation_resource_tags,
            auto_import=auto_import,
            aws_service=aws_service,
            description=description,
            locale=locale,
            provisioned_cidrs=provisioned_cidrs,
            public_ip_source=public_ip_source,
            publicly_advertisable=publicly_advertisable,
            source_ipam_pool_id=source_ipam_pool_id,
            source_resource=source_resource,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ed624bf3dc307f04a3b45acad98bea6483b1a66eab0f2a9472fa9843d91564b2)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4e7676e43df5cfe9b5768db72cf66c10335a772ff51e6a339deb5179a8979067)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The ARN of the IPAM pool.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrIpamArn")
    def attr_ipam_arn(self) -> builtins.str:
        '''The ARN of the IPAM.

        :cloudformationAttribute: IpamArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamArn"))

    @builtins.property
    @jsii.member(jsii_name="attrIpamPoolId")
    def attr_ipam_pool_id(self) -> builtins.str:
        '''The ID of the IPAM pool.

        :cloudformationAttribute: IpamPoolId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamPoolId"))

    @builtins.property
    @jsii.member(jsii_name="attrIpamScopeArn")
    def attr_ipam_scope_arn(self) -> builtins.str:
        '''The ARN of the scope of the IPAM pool.

        :cloudformationAttribute: IpamScopeArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamScopeArn"))

    @builtins.property
    @jsii.member(jsii_name="attrIpamScopeType")
    def attr_ipam_scope_type(self) -> builtins.str:
        '''The scope of the IPAM.

        :cloudformationAttribute: IpamScopeType
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrIpamScopeType"))

    @builtins.property
    @jsii.member(jsii_name="attrPoolDepth")
    def attr_pool_depth(self) -> jsii.Number:
        '''The depth of pools in your IPAM pool.

        The pool depth quota is 10.

        :cloudformationAttribute: PoolDepth
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrPoolDepth"))

    @builtins.property
    @jsii.member(jsii_name="attrState")
    def attr_state(self) -> builtins.str:
        '''The state of the IPAM pool.

        :cloudformationAttribute: State
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrState"))

    @builtins.property
    @jsii.member(jsii_name="attrStateMessage")
    def attr_state_message(self) -> builtins.str:
        '''A message related to the failed creation of an IPAM pool.

        :cloudformationAttribute: StateMessage
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrStateMessage"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="addressFamily")
    def address_family(self) -> builtins.str:
        '''The address family of the pool.'''
        return typing.cast(builtins.str, jsii.get(self, "addressFamily"))

    @address_family.setter
    def address_family(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a5b241c251a5a66afb68207fac43059c8a2515b97d3b1cc207e2657902e71b1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "addressFamily", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="ipamScopeId")
    def ipam_scope_id(self) -> builtins.str:
        '''The ID of the scope in which you would like to create the IPAM pool.'''
        return typing.cast(builtins.str, jsii.get(self, "ipamScopeId"))

    @ipam_scope_id.setter
    def ipam_scope_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3fbdbec1bdf539bf7935f66283de5d2455b37fd036bd5f6e2f5ad52c42338fb1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ipamScopeId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="allocationDefaultNetmaskLength")
    def allocation_default_netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The default netmask length for allocations added to this pool.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "allocationDefaultNetmaskLength"))

    @allocation_default_netmask_length.setter
    def allocation_default_netmask_length(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bb67e5df586c19b16abaf86994738112eed024ce90016e4c18b7dd264553f702)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allocationDefaultNetmaskLength", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="allocationMaxNetmaskLength")
    def allocation_max_netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "allocationMaxNetmaskLength"))

    @allocation_max_netmask_length.setter
    def allocation_max_netmask_length(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1ff43fa5869f5613ca15882015827b8534dcb8488bc50fe83a2d7974080f5420)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allocationMaxNetmaskLength", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="allocationMinNetmaskLength")
    def allocation_min_netmask_length(self) -> typing.Optional[jsii.Number]:
        '''The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "allocationMinNetmaskLength"))

    @allocation_min_netmask_length.setter
    def allocation_min_netmask_length(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__841766c7de2c1b457f6e90e16e2c5de9d9ad55f57131bf4ebe7e22c651a5871a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allocationMinNetmaskLength", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="allocationResourceTags")
    def allocation_resource_tags(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, _CfnTag_f6864754]]]]:
        '''Tags that are required for resources that use CIDRs from this IPAM pool.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, _CfnTag_f6864754]]]], jsii.get(self, "allocationResourceTags"))

    @allocation_resource_tags.setter
    def allocation_resource_tags(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, _CfnTag_f6864754]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__00c43e28a217efdf34866f8f436e0d615b3f76ef33d75a0fcd54107f4fcd2395)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allocationResourceTags", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="autoImport")
    def auto_import(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''If selected, IPAM will continuously look for resources within the CIDR range of this pool and automatically import them as allocations into your IPAM.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "autoImport"))

    @auto_import.setter
    def auto_import(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3bef013417f7aecffe6df419f044a34ba6c1056e4c6fe41fab3f61c1c98629e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoImport", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="awsService")
    def aws_service(self) -> typing.Optional[builtins.str]:
        '''Limits which service in AWS that the pool can be used in.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "awsService"))

    @aws_service.setter
    def aws_service(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3e466f53a7750a1bd0f9875419a5a1b56600ca1c8fc7f153ecaab870fc5b5876)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "awsService", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the IPAM pool.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bcec51b995a93f9fe674b9eb0c8ec907bb474c040116a93fcfe16794efc35e8b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="locale")
    def locale(self) -> typing.Optional[builtins.str]:
        '''The locale of the IPAM pool.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "locale"))

    @locale.setter
    def locale(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6f634bcf807180517176b75391374db468661710cf7172f3ee46b626259a33de)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "locale", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="provisionedCidrs")
    def provisioned_cidrs(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIPAMPool.ProvisionedCidrProperty"]]]]:
        '''Information about the CIDRs provisioned to an IPAM pool.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIPAMPool.ProvisionedCidrProperty"]]]], jsii.get(self, "provisionedCidrs"))

    @provisioned_cidrs.setter
    def provisioned_cidrs(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIPAMPool.ProvisionedCidrProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f2ec724b5a099431c168859f8bae8690b6942bf2c4428928ba37a452bf8eab6a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "provisionedCidrs", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="publicIpSource")
    def public_ip_source(self) -> typing.Optional[builtins.str]:
        '''The IP address source for pools in the public scope.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "publicIpSource"))

    @public_ip_source.setter
    def public_ip_source(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__67c2537c8cc935adffb5c8b6b1f780d4d25f1c3bcead923fd22423ccb9eff737)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "publicIpSource", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="publiclyAdvertisable")
    def publicly_advertisable(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Determines if a pool is publicly advertisable.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "publiclyAdvertisable"))

    @publicly_advertisable.setter
    def publicly_advertisable(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a2c1d055ef4dd3ab27084865a5b4dc034dc0c587c1cfc2aef7e63cafc581793c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "publiclyAdvertisable", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="sourceIpamPoolId")
    def source_ipam_pool_id(self) -> typing.Optional[builtins.str]:
        '''The ID of the source IPAM pool.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceIpamPoolId"))

    @source_ipam_pool_id.setter
    def source_ipam_pool_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__324d3ea5d3ecf303ca16390cb4f68ab2b30fc3cd3cf7c82ba59c99a16243a68b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceIpamPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="sourceResource")
    def source_resource(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIPAMPool.SourceResourceProperty"]]:
        '''The resource used to provision CIDRs to a resource planning pool.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIPAMPool.SourceResourceProperty"]], jsii.get(self, "sourceResource"))

    @source_resource.setter
    def source_resource(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIPAMPool.SourceResourceProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3f70db41696848da5f2c3703ae0a09b95494dc9b1ba1e9e1147650712c765d52)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "sourceResource", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The key/value combination of a tag assigned to the resource.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f5a27f882c280937fa0881f1d5fdc26d072d6a34ae1cdbaec63936d887ba4555)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnIPAMPool.ProvisionedCidrProperty",
        jsii_struct_bases=[],
        name_mapping={"cidr": "cidr"},
    )
    class ProvisionedCidrProperty:
        def __init__(self, *, cidr: builtins.str) -> None:
            '''The CIDR provisioned to the IPAM pool.

            A CIDR is a representation of an IP address and its associated network mask (or netmask) and refers to a range of IP addresses. An IPv4 CIDR example is ``10.24.34.0/23`` . An IPv6 CIDR example is ``2001:DB8::/32`` .
            .. epigraph::

               This resource type does not allow you to provision a CIDR using the netmask length. To provision a CIDR using netmask length, use `AWS::EC2::IPAMPoolCidr <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ipampoolcidr.html>`_ .

            :param cidr: The CIDR provisioned to the IPAM pool. A CIDR is a representation of an IP address and its associated network mask (or netmask) and refers to a range of IP addresses. An IPv4 CIDR example is ``10.24.34.0/23`` . An IPv6 CIDR example is ``2001:DB8::/32`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipampool-provisionedcidr.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                provisioned_cidr_property = ec2.CfnIPAMPool.ProvisionedCidrProperty(
                    cidr="cidr"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fff1e35b565a960234fe5a7c634cdd72887d66cc483af55f294c6fc9ca10476d)
                check_type(argname="argument cidr", value=cidr, expected_type=type_hints["cidr"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "cidr": cidr,
            }

        @builtins.property
        def cidr(self) -> builtins.str:
            '''The CIDR provisioned to the IPAM pool.

            A CIDR is a representation of an IP address and its associated network mask (or netmask) and refers to a range of IP addresses. An IPv4 CIDR example is ``10.24.34.0/23`` . An IPv6 CIDR example is ``2001:DB8::/32`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipampool-provisionedcidr.html#cfn-ec2-ipampool-provisionedcidr-cidr
            '''
            result = self._values.get("cidr")
            assert result is not None, "Required property 'cidr' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProvisionedCidrProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_ec2.CfnIPAMPool.SourceResourceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "resource_id": "resourceId",
            "resource_owner": "resourceOwner",
            "resource_region": "resourceRegion",
            "resource_type": "resourceType",
        },
    )
    class SourceResourceProperty:
        def __init__(
            self,
            *,
            resource_id: builtins.str,
            resource_owner: builtins.str,
            resource_region: builtins.str,
            resource_type: builtins.str,
        ) -> None:
            '''The resource used to provision CIDRs to a resource planning pool.

            :param resource_id: The source resource ID.
            :param resource_owner: The source resource owner.
            :param resource_region: The source resource Region.
            :param resource_type: The source resource type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipampool-sourceresource.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_ec2 as ec2
                
                source_resource_property = ec2.CfnIPAMPool.SourceResourceProperty(
                    resource_id="resourceId",
                    resource_owner="resourceOwner",
                    resource_region="resourceRegion",
                    resource_type="resourceType"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__499a6dc3aca0b20e81b7aa4c45945c37a69bc00be56b64d258b95bd99e3d3db8)
                check_type(argname="argument resource_id", value=resource_id, expected_type=type_hints["resource_id"])
                check_type(argname="argument resource_owner", value=resource_owner, expected_type=type_hints["resource_owner"])
                check_type(argname="argument resource_region", value=resource_region, expected_type=type_hints["resource_region"])
                check_type(argname="argument resource_type", value=resource_type, expected_type=type_hints["resource_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "resource_id": resource_id,
                "resource_owner": resource_owner,
                "resource_region": resource_region,
                "resource_type": resource_type,
            }

        @builtins.property
        def resource_id(self) -> builtins.str:
            '''The source resource ID.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipampool-sourceresource.html#cfn-ec2-ipampool-sourceresource-resourceid
            '''
            result = self._values.get("resource_id")
            assert result is not None, "Required property 'resource_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def resource_owner(self) -> builtins.str:
            '''The source resource owner.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipampool-sourceresource.html#cfn-ec2-ipampool-sourceresource-resourceowner
            '''
            result = self._values.get("resource_owner")
            assert result is not None, "Required property 'resource_owner' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def resource_region(self) -> builtins.str:
            '''The source resource Region.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipampool-sourceresource.html#cfn-ec2-ipampool-sourceresource-resourceregion
            '''
            result = self._values.get("resource_region")
            assert result is not None, "Required property 'resource_region' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def resource_type(self) -> builtins.str:
            '''The source resource type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ipampool-sourceresource.html#cfn-ec2-ipampool-sourceresource-resourcetype
            '''
            result = self._values.get("resource_type")
            assert result is not None, "Required property 'resource_type' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SourceResourceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556)
class CfnIPAMPoolCidr(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_ec2.CfnIPAMPoolCidr",
):
    '''A CIDR provisioned to an IPAM pool.

    :see: http://docs.aws.amazon.com/AWSCloudFor