r'''
# Amazon Cognito Construct Library

[Amazon Cognito](https://docs.aws.amazon.com/cognito/latest/developerguide/what-is-amazon-cognito.html) provides
authentication, authorization, and user management for your web and mobile apps. Your users can sign in directly with a
user name and password, or through a third party such as Facebook, Amazon, Google or Apple.

The two main components of Amazon Cognito are [user
pools](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html) and [identity
pools](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html). User pools are user directories
that provide sign-up and sign-in options for your app users. Identity pools enable you to grant your users access to
other AWS services. Identity Pool L2 Constructs can be found [here](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cognito-identitypool-alpha-readme.html).

This module is part of the [AWS Cloud Development Kit](https://github.com/aws/aws-cdk) project.

## Table of Contents

* [Amazon Cognito Construct Library](#amazon-cognito-construct-library)

  * [Table of Contents](#table-of-contents)
  * [User Pools](#user-pools)

    * [User pool feature plans](#user-pool-feature-plans)
    * [Sign Up](#sign-up)

      * [Code Verification](#code-verification)
      * [Link Verification](#link-verification)
    * [Sign In](#sign-in)

      * [Choice-based authentication](#choice-based-authentication-passwordless-sign-in--passkey-sign-in)
    * [Attributes](#attributes)
    * [Attribute verification](#attribute-verification)
    * [Security](#security)

      * [Multi-factor Authentication (MFA)](#multi-factor-authentication-mfa)
      * [Account Recovery Settings](#account-recovery-settings)
      * [Advanced Security Mode](#advanced-security-mode)
      * [Threat Protection](#threat-protection)
    * [Emails](#emails)
    * [Device Tracking](#device-tracking)
    * [Lambda Triggers](#lambda-triggers)

      * [Trigger Permissions](#trigger-permissions)
    * [Importing User Pools](#importing-user-pools)
    * [Identity Providers](#identity-providers)
    * [App Clients](#app-clients)
    * [Resource Servers](#resource-servers)
    * [Domains](#domains)
    * [Deletion protection](#deletion-protection)
    * [Analytics Configuration](#analytics-configuration)

      * [When specifying a Pinpoint application from the same account](#when-specifying-a-pinpoint-application-from-the-same-account)
      * [When specifying a Pinpoint application from a different account](#when-specifying-a-pinpoint-application-from-a-different-account)

## User Pools

User pools allow creating and managing your own directory of users that can sign up and sign in. They enable easy
integration with social identity providers such as Facebook, Google, Amazon, Microsoft Active Directory, etc. through
SAML.

Using the CDK, a new user pool can be created as part of the stack using the construct's constructor. You may specify
the `userPoolName` to give your own identifier to the user pool. If not, CloudFormation will generate a name.

```python
cognito.UserPool(self, "myuserpool",
    user_pool_name="myawesomeapp-userpool",
    sign_in_case_sensitive=False
)
```

By default, usernames and email addresses in user pools are case sensitive, which means `user@example.com` and `User@example.com`
are considered different. In most situations it is preferred to have usernames and email addresses be case insensitive so that
capitalization differences are ignored. As shown above, you can make a user pool case insensitive by setting `signInCaseSensitive`
to `false`. The case sensitivity cannot be changed once a user pool is created.

The default set up for the user pool is configured such that only administrators will be allowed
to create users. Features such as Multi-factor authentication (MFAs) and Lambda Triggers are not
configured by default.

Use the `grant()` method to add an IAM policy statement associated with the user pool to an
IAM principal's policy.

```python
user_pool = cognito.UserPool(self, "myuserpool")
role = iam.Role(self, "role",
    assumed_by=iam.ServicePrincipal("foo")
)
user_pool.grant(role, "cognito-idp:AdminCreateUser")
```

### User pool feature plans

Amazon Cognito has feature plans for user pools. Each plan has a set of features and a monthly cost per active user. Each feature plan unlocks access to more features than the one before it.
Learn more about [feature plans here](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-sign-in-feature-plans.html).

* *Lite* - a low-cost feature plan for user pools with lower numbers of monthly active users.
* *Essentials* - all of the latest user pool authentication features.
* *Plus* - includes everything in the Essentials plan and adds advanced security features that protect your users.

The default feature plan is Essentials for newly create user pools.
For the existing user pools, Lite plan is automatically set.

Previously, some user pool features were included in [an advanced security features](#advanced-security-mode) pricing structure. The features that were included in this structure are now under either the Essentials or Plus plan.

### Sign Up

Users can either be signed up by the app's administrators or can sign themselves up. Once a user has signed up, their
account needs to be confirmed. Cognito provides several ways to sign users up and confirm their accounts. Learn more
about [user sign up here](https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html).

To verify the email address of a user in your user pool with Amazon Cognito, you can send the user an email message
with a link that they can select, or you can send them a code that they can enter.

#### Code Verification

When a user signs up, email and SMS messages are used to verify their account and contact methods. The following code
snippet configures a user pool with properties relevant to these verification messages -

```python
cognito.UserPool(self, "myuserpool",
    # ...
    self_sign_up_enabled=True,
    user_verification=cognito.UserVerificationConfig(
        email_subject="Verify your email for our awesome app!",
        email_body="Thanks for signing up to our awesome app! Your verification code is {####}",
        email_style=cognito.VerificationEmailStyle.CODE,
        sms_message="Thanks for signing up to our awesome app! Your verification code is {####}"
    )
)
```

By default, self sign up is disabled. Learn more about [email and SMS verification messages
here](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-customizations.html).

Besides users signing themselves up, an administrator of any user pool can sign users up. The user then receives an
invitation to join the user pool. The following code snippet configures a user pool with properties relevant to the
invitation messages -

```python
cognito.UserPool(self, "myuserpool",
    # ...
    user_invitation=cognito.UserInvitationConfig(
        email_subject="Invite to join our awesome app!",
        email_body="Hello {username}, you have been invited to join our awesome app! Your temporary password is {####}",
        sms_message="Hello {username}, your temporary password for our awesome app is {####}"
    )
)
```

#### Link Verification

Alternatively, users can use link as a verification method. The following code snippet configures a user pool with
properties relevant to these verification messages and link verification method.

```python
cognito.UserPool(self, "myuserpool",
    user_verification=cognito.UserVerificationConfig(
        email_style=cognito.VerificationEmailStyle.LINK,
        email_subject="Invite to join our awesome app!",
        email_body="You have been invited to join our awesome app! {##Verify Your Email##}"
    )
)
```

All email subjects, bodies and SMS messages for both invitation and verification support Cognito's message templating.
Learn more about [message templates
here](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-templates.html).

### Sign In

Users registering or signing in into your application can do so with multiple identifiers. There are 4 options
available:

* `username`: Allow signing in using the one time immutable user name that the user chose at the time of sign up.
* `email`: Allow signing in using the email address that is associated with the account.
* `phone`: Allow signing in using the phone number that is associated with the account.
* `preferredUsername`: Allow signing in with an alternate user name that the user can change at any time. However, this
  is not available if the `username` option is not chosen.

The following code sets up a user pool so that the user can sign in with either their username or their email address -

```python
cognito.UserPool(self, "myuserpool",
    # ...
    # ...
    sign_in_aliases=cognito.SignInAliases(
        username=True,
        email=True
    )
)
```

User pools can either be configured so that user name is primary sign in form, but also allows for the other three to be
used additionally; or it can be configured so that email and/or phone numbers are the only ways a user can register and
sign in. Read more about this
[here](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases-settings).

⚠️ The Cognito service prevents changing the `signInAlias` property for an existing user pool.

To match with 'Option 1' in the above link, with a verified email, `signInAliases` should be set to
`{ username: true, email: true }`. To match with 'Option 2' in the above link with both a verified
email and phone number, this property should be set to `{ email: true, phone: true }`.

Cognito recommends that email and phone number be automatically verified, if they are one of the sign in methods for
the user pool. Read more about that
[here](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases).
The CDK does this by default, when email and/or phone number are specified as part of `signInAliases`. This can be
overridden by specifying the `autoVerify` property.

The following code snippet sets up only email as a sign in alias, but both email and phone number to be auto-verified.

```python
cognito.UserPool(self, "myuserpool",
    # ...
    # ...
    sign_in_aliases=cognito.SignInAliases(username=True, email=True),
    auto_verify=cognito.AutoVerifiedAttrs(email=True, phone=True)
)
```

A user pool can optionally ignore case when evaluating sign-ins. When `signInCaseSensitive` is false, Cognito will not
check the capitalization of the alias when signing in. Default is true.

#### Choice-based authentication: passwordless sign-in / passkey sign-in

User pools can be configured to allow the following authentication methods in choice-based authentication:

* Passwordless sign-in with email message one-time password
* Passwordless sign-in with SMS message one-time password
* Passkey (WebAuthn) sign-in

To use choice-based authentication, [User pool feature plan](#user-pool-feature-plans) should be Essentials or higher.

For details of authentication methods and client implementation, see [Manage authentication methods in AWS SDKs](https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html).

The following code configures a user pool with choice-based authentication enabled:

```python
user_pool = cognito.UserPool(self, "myuserpool",
    sign_in_policy=cognito.SignInPolicy(
        allowed_first_auth_factors=cognito.AllowedFirstAuthFactors(
            password=True,  # password authentication must be enabled
            email_otp=True,  # enables email message one-time password
            sms_otp=True,  # enables SMS message one-time password
            passkey=True
        )
    )
)

# You should also configure the user pool client with USER_AUTH authentication flow allowed
user_pool.add_client("myclient",
    auth_flows=cognito.AuthFlow(user=True)
)
```

⚠️ Enabling SMS message one-time password requires the AWS account be activated to SMS message sending.
Learn more about [SMS message settings for Amazon Cognito user pools](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-sms-settings.html).

When enabling passkey sign-in, you should specify the authentication domain used as the relying party ID.
Learn more about [passkey sign-in of user pools](https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow-methods.html#amazon-cognito-user-pools-authentication-flow-methods-passkey) and [Web Authentication API](https://developer.mozilla.org/en-US/docs/Web/API/Web_Authentication_API).

```python
# Use the hosted Amazon Cognito domain as the relying party ID
cognito.UserPool(self, "myuserpool",
    sign_in_policy=cognito.SignInPolicy(
        allowed_first_auth_factors=cognito.AllowedFirstAuthFactors(password=True, passkey=True)
    ),
    passkey_relying_party_id="myclientname.auth.region-name.amazoncognito.com"
)

# Use the custom domain as the relying party ID
cognito.UserPool(self, "myuserpool",
    sign_in_policy=cognito.SignInPolicy(
        allowed_first_auth_factors=cognito.AllowedFirstAuthFactors(password=True, passkey=True)
    ),
    passkey_relying_party_id="auth.example.com"
)
```

You can configure user verification to be preferred (default) or required. When you set user verification to preferred, users can set up authenticators that don't have the user verification capability, and registration and authentication operations can succeed without user verification. To mandate user verification in passkey registration and authentication, specify `passkeyUserVerification` to `PasskeyUserVerification.REQUIRED`.

```python
cognito.UserPool(self, "myuserpool",
    sign_in_policy=cognito.SignInPolicy(
        allowed_first_auth_factors=cognito.AllowedFirstAuthFactors(password=True, passkey=True)
    ),
    passkey_relying_party_id="auth.example.com",
    passkey_user_verification=cognito.PasskeyUserVerification.REQUIRED
)
```

To disable choice-based authentication explicitly, specify `password` only.

```python
cognito.UserPool(self, "myuserpool",
    sign_in_policy=cognito.SignInPolicy(
        allowed_first_auth_factors=cognito.AllowedFirstAuthFactors(password=True)
    ),
    feature_plan=cognito.FeaturePlan.LITE
)
```

### Attributes

Attributes represent the various properties of each user that's collected and stored in the user pool. Cognito
provides a set of standard attributes that are available for all user pools. Users are allowed to select any of these
standard attributes to be required. Users will not be able to sign up to the user pool without providing the required
attributes. Besides these, additional attributes can be further defined, and are known as custom attributes.

Learn more on [attributes in Cognito's
documentation](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html).

The following code configures a user pool with two standard attributes (name and address) as required and mutable, and adds
four custom attributes.

```python
cognito.UserPool(self, "myuserpool",
    # ...
    standard_attributes=cognito.StandardAttributes(
        fullname=cognito.StandardAttribute(
            required=True,
            mutable=False
        ),
        address=cognito.StandardAttribute(
            required=False,
            mutable=True
        )
    ),
    custom_attributes={
        "myappid": cognito.StringAttribute(min_len=5, max_len=15, mutable=False),
        "callingcode": cognito.NumberAttribute(min=1, max=3, mutable=True),
        "isEmployee": cognito.BooleanAttribute(mutable=True),
        "joinedOn": cognito.DateTimeAttribute()
    }
)
```

As shown in the code snippet, there are data types that are available for custom attributes. The 'String' and 'Number'
data types allow for further constraints on their length and values, respectively.

Custom attributes cannot be marked as required.

All custom attributes share the property `mutable` that specifies whether the value of the attribute can be changed.
The default value is `false`.

User pools come with two 'built-in' attributes - `email_verified` and `phone_number_verified`. These cannot be
configured (required-ness or mutability) as part of user pool creation. However, user pool administrators can modify
them for specific users using the [AdminUpdateUserAttributes API](https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html).

### Attribute verification

When your user updates an email address or phone number attribute, Amazon Cognito marks it unverified until they verify the new value.
You can’t send messages to an unverified email address or phone number.
Your user can’t sign in with an unverified alias attribute.
You can choose how Amazon Cognito handles an updated email address or phone number after the update and before the verification.

Learn more on [configuring email or phone verification in Cognito's documentation.](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html?icmpid=docs_cognito_console_help_panel)

The following code configures a user pool that keeps the original value for the two standard attributes (email and phone_number) until the new values are verified.

```python
cognito.UserPool(self, "myuserpool",
    # ...
    sign_in_aliases=cognito.SignInAliases(username=True),
    auto_verify=cognito.AutoVerifiedAttrs(email=True, phone=True),
    keep_original=cognito.KeepOriginalAttrs(
        email=True,
        phone=True
    )
)
```

### Security

Cognito sends various messages to its users via SMS, for different actions, ranging from account verification to
marketing. In order to send SMS messages, Cognito needs an IAM role that it can assume, with permissions that allow it
to send SMS messages.

By default, the CDK looks at all of the specified properties (and their defaults when not explicitly specified) and
automatically creates an SMS role, when needed. For example, if MFA second factor by SMS is enabled, the CDK will
create a new role. The `smsRole` property can be used to specify the user supplied role that should be used instead.
Additionally, the property `enableSmsRole` can be used to override the CDK's default behaviour to either enable or
suppress automatic role creation.

```python
pool_sms_role = iam.Role(self, "userpoolsmsrole",
    assumed_by=iam.ServicePrincipal("foo")
)

cognito.UserPool(self, "myuserpool",
    # ...
    sms_role=pool_sms_role,
    sms_role_external_id="c87467be-4f34-11ea-b77f-2e728ce88125"
)
```

When the `smsRole` property is specified, the `smsRoleExternalId` may also be specified. The value of
`smsRoleExternalId` will be used as the `sts:ExternalId` when the Cognito service assumes the role. In turn, the role's
assume role policy should be configured to accept this value as the ExternalId. Learn more about [ExternalId
here](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html).

#### Multi-factor Authentication (MFA)

User pools can be configured to enable multi-factor authentication (MFA). It can either be turned off, set to optional
or made required. Setting MFA to optional means that individual users can choose to enable it.
Additionally, the MFA code can be sent either via SMS text message or via a time-based software token.
See the [documentation on MFA](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html) to
learn more.

The following code snippet marks MFA for the user pool as required. This means that all users are required to
configure an MFA token and use it for sign in. It also allows for the users to use both SMS based MFA, as well,
[time-based one time password
(TOTP)](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa-totp.html).

If you want to enable email-based MFA, set `email` property to the Amazon SES email-sending configuration and set `featurePlan` to `FeaturePlan.ESSENTIALS` or `FeaturePlan.PLUS`.
For more information, see [SMS and email message MFA](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa-sms-email-message.html).

```python
cognito.UserPool(self, "myuserpool",
    # ...
    mfa=cognito.Mfa.REQUIRED,
    mfa_second_factor=cognito.MfaSecondFactor(
        sms=True,
        otp=True,
        email=False
    )
)
```

User pools can be configured with policies around a user's password. This includes the password length and the
character sets that they must contain.

Further to this, it can also be configured with the validity of the auto-generated temporary password. A temporary
password is generated by the user pool either when an admin signs up a user or when a password reset is requested.
The validity of this password dictates how long to give the user to use this password before expiring it.

You can also set a policy for password reuse by setting the `passwordHistorySize` property.
You can prevent a user from resetting their password to a new password that matches their current password or any of up to 23 additional previous passwords, for a maximum total of 24.
The `passwordHistorySize` property can not be set when `featurePlan` is `FeaturePlan.LITE`.

The following code snippet configures these properties -

```python
cognito.UserPool(self, "myuserpool",
    # ...
    password_policy=cognito.PasswordPolicy(
        min_length=12,
        require_lowercase=True,
        require_uppercase=True,
        require_digits=True,
        require_symbols=True,
        temp_password_validity=Duration.days(3)
    )
)
```

Note that, `tempPasswordValidity` can be specified only in whole days. Specifying fractional days would throw an error.

#### Account Recovery Settings

User pools can be configured on which method a user should use when recovering the password for their account. This
can either be email and/or SMS. Read more at [Recovering User Accounts](https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html)

```python
cognito.UserPool(self, "UserPool",
    # ...
    account_recovery=cognito.AccountRecovery.EMAIL_ONLY
)
```

The default for account recovery is by phone if available and by email otherwise.
A user will not be allowed to reset their password via phone if they are also using it for MFA.

#### Advanced Security Mode

⚠️ Advanced Security Mode is deprecated in favor of [Threat Protection](#threat-protection).

User pools can be configured to use Advanced security. You can turn the user pool advanced security features on, and customize the actions that are taken in response to different risks. Or you can use audit mode to gather metrics on detected risks without taking action. In audit mode, the advanced security features publish metrics to Amazon CloudWatch. See the [documentation on Advanced security](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-advanced-security.html) to learn more.

```python
cognito.UserPool(self, "myuserpool",
    # ...
    advanced_security_mode=cognito.AdvancedSecurityMode.ENFORCED
)
```

### Threat Protection

This feature is only available if your Feature Plan is set to PLUS.

Threat Protection can be set to configure enforcement levels and automatic responses for users in password-based and custom-challenge authentication flows.
For configuration, there are 2 options for standard authentication and custom authentication.
These are represented with properties `standardThreatProtectionMode` and `customThreatProtectionMode`.
See the [documentation on Threat Protection](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-threat-protection.html)

### Emails

Cognito sends emails to users in the user pool, when particular actions take place, such as welcome emails, invitation
emails, password resets, etc. The address from which these emails are sent can be configured on the user pool.
Read more at [Email settings for User Pools](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html).

By default, user pools are configured to use Cognito's built in email capability, which will send emails
from `no-reply@verificationemail.com`. If you want to use a custom email address you can configure
Cognito to send emails through Amazon SES, which is detailed below.

```python
cognito.UserPool(self, "myuserpool",
    email=cognito.UserPoolEmail.with_cognito("support@myawesomeapp.com")
)
```

For typical production environments, the default email limit is below the required delivery volume.
To enable a higher delivery volume, you can configure the UserPool to send emails through Amazon SES. To do
so, follow the steps in the [Cognito Developer Guide](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html#user-pool-email-developer)
to verify an email address, move the account out of the SES sandbox, and grant Cognito email permissions via an
authorization policy.

Once the SES setup is complete, the UserPool can be configured to use the SES email.

```python
cognito.UserPool(self, "myuserpool",
    email=cognito.UserPoolEmail.with_sES(
        from_email="noreply@myawesomeapp.com",
        from_name="Awesome App",
        reply_to="support@myawesomeapp.com"
    )
)
```

Sending emails through SES requires that SES be configured (as described above) in a valid SES region.
If the UserPool is being created in a different region, `sesRegion` must be used to specify the correct SES region.

```python
cognito.UserPool(self, "myuserpool",
    email=cognito.UserPoolEmail.with_sES(
        ses_region="us-east-1",
        from_email="noreply@myawesomeapp.com",
        from_name="Awesome App",
        reply_to="support@myawesomeapp.com"
    )
)
```

When sending emails from an SES verified domain, `sesVerifiedDomain` can be used to specify the domain.
The email address does not need to be verified when sending emails from a verified domain, because the identity of the email configuration is can be determined from the domain alone.

```python
cognito.UserPool(self, "myuserpool",
    email=cognito.UserPoolEmail.with_sES(
        ses_region="us-east-1",
        from_email="noreply@myawesomeapp.com",
        from_name="Awesome App",
        reply_to="support@myawesomeapp.com",
        ses_verified_domain="myawesomeapp.com"
    )
)
```

If `fromName` does not comply RFC 5322 atom or quoted-string, it will be quoted or mime-encoded.

```python
cognito.UserPool(self, "myuserpool",
    email=cognito.UserPoolEmail.with_sES(
        from_email="noreply@myawesomeapp.com",
        from_name="myname@mycompany.com"
    )
)
```

### Device Tracking

User pools can be configured to track devices that users have logged in to.
Read more at [Device Tracking](https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html)

```python
cognito.UserPool(self, "myuserpool",
    # ...
    device_tracking=cognito.DeviceTracking(
        challenge_required_on_new_device=True,
        device_only_remembered_on_user_prompt=True
    )
)
```

The default is to not track devices.

### Lambda Triggers

User pools can be configured such that AWS Lambda functions can be triggered when certain user operations or actions
occur, such as, sign up, user confirmation, sign in, etc. They can also be used to add custom authentication
challenges, user migrations and custom verification messages. Learn more about triggers at [User Pool Workflows with
Triggers](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html).

Lambda triggers can either be specified as part of the `UserPool` initialization, or it can be added later, via methods
on the construct, as so -

```python
auth_challenge_fn = lambda_.Function(self, "authChallengeFn",
    runtime=lambda_.Runtime.NODEJS_LATEST,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "path/to/asset"))
)

userpool = cognito.UserPool(self, "myuserpool",
    # ...
    lambda_triggers=cognito.UserPoolTriggers(
        create_auth_challenge=auth_challenge_fn
    )
)

userpool.add_trigger(cognito.UserPoolOperation.USER_MIGRATION, lambda_.Function(self, "userMigrationFn",
    runtime=lambda_.Runtime.NODEJS_LATEST,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "path/to/asset"))
))
```

Additionally, only the pre token generation Lambda trigger supports trigger events with lambda version V2.0 or V3.0.
For details, see [Pre Token Generation Lambda Trigger](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html).

```python
# userpool: cognito.UserPool
# pre_token_generation_fn: lambda.Function


userpool.add_trigger(cognito.UserPoolOperation.PRE_TOKEN_GENERATION_CONFIG, pre_token_generation_fn, cognito.LambdaVersion.V2_0)
```

The following table lists the set of triggers available, and their corresponding method to add it to the user pool.
For more information on the function of these triggers and how to configure them, read [User Pool Workflows with
Triggers](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html).

#### Trigger Permissions

The `function.attachToRolePolicy()` API can be used to add additional IAM permissions to the lambda trigger
as necessary.

⚠️ Using the `attachToRolePolicy` API to provide permissions to your user pool will result in a circular dependency. See [aws/aws-cdk#7016](https://github.com/aws/aws-cdk/issues/7016).
Error message when running `cdk synth` or `cdk deploy`:

> Circular dependency between resources: [pool056F3F7E, fnPostAuthFnCognitoA630A2B1, ...]

To work around the circular dependency issue, use the `attachInlinePolicy()` API instead, as shown below.

```python
# post_auth_fn: lambda.Function


userpool = cognito.UserPool(self, "myuserpool",
    lambda_triggers=cognito.UserPoolTriggers(
        post_authentication=post_auth_fn
    )
)

# provide permissions to describe the user pool scoped to the ARN the user pool
post_auth_fn.role.attach_inline_policy(iam.Policy(self, "userpool-policy",
    statements=[iam.PolicyStatement(
        actions=["cognito-idp:DescribeUserPool"],
        resources=[userpool.user_pool_arn]
    )]
))
```

### Importing User Pools

Any user pool that has been created outside of this stack, can be imported into the CDK app. Importing a user pool
allows for it to be used in other parts of the CDK app that reference an `IUserPool`. However, imported user pools have
limited configurability. As a rule of thumb, none of the properties that are part of the
[`AWS::Cognito::UserPool`](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html)
CloudFormation resource can be configured.

User pools can be imported either using their id via the `UserPool.fromUserPoolId()`, or by using their ARN, via the
`UserPool.fromUserPoolArn()` API.

```python
awesome_pool = cognito.UserPool.from_user_pool_id(self, "awesome-user-pool", "us-east-1_oiuR12Abd")

other_awesome_pool = cognito.UserPool.from_user_pool_arn(self, "other-awesome-user-pool", "arn:aws:cognito-idp:eu-west-1:123456789012:userpool/us-east-1_mtRyYQ14D")
```

### Identity Providers

Users that are part of a user pool can sign in either directly through a user pool, or federate through a third-party
identity provider. Once configured, the Cognito backend will take care of integrating with the third-party provider.
Read more about [Adding User Pool Sign-in Through a Third
Party](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-identity-federation.html).

The following third-party identity providers are currently supported in the CDK -

* [Login With Amazon](https://developer.amazon.com/apps-and-games/login-with-amazon)
* [Facebook Login](https://developers.facebook.com/docs/facebook-login/)
* [Google Login](https://developers.google.com/identity/sign-in/web/sign-in)
* [Sign In With Apple](https://developer.apple.com/sign-in-with-apple/get-started/)
* [OpenID Connect](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-oidc-idp.html)
* [SAML](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-saml-idp.html)

The following code configures a user pool to federate with the third party provider, 'Login with Amazon'. The identity
provider needs to be configured with a set of credentials that the Cognito backend can use to federate with the
third-party identity provider.

```python
userpool = cognito.UserPool(self, "Pool")

provider = cognito.UserPoolIdentityProviderAmazon(self, "Amazon",
    client_id="amzn-client-id",
    client_secret="amzn-client-secret",
    user_pool=userpool
)
```

Using Google identity provider is possible to use clientSecretValue with SecretValue from secrets manager.

```python
userpool = cognito.UserPool(self, "Pool")
secret = secretsmanager.Secret.from_secret_attributes(self, "CognitoClientSecret",
    secret_complete_arn="arn:aws:secretsmanager:xxx:xxx:secret:xxx-xxx"
).secret_value

provider = cognito.UserPoolIdentityProviderGoogle(self, "Google",
    client_id="amzn-client-id",
    client_secret_value=secret,
    user_pool=userpool
)
```

Using SAML identity provider is possible to use SAML metadata file content or SAML metadata file url.

```python
userpool = cognito.UserPool(self, "Pool")

# specify the metadata as a file content
cognito.UserPoolIdentityProviderSaml(self, "userpoolIdpFile",
    user_pool=userpool,
    metadata=cognito.UserPoolIdentityProviderSamlMetadata.file("my-file-contents"),
    # Whether to require encrypted SAML assertions from IdP
    encrypted_responses=True,
    # The signing algorithm for the SAML requests
    request_signing_algorithm=cognito.SigningAlgorithm.RSA_SHA256,
    # Enable IdP initiated SAML auth flow
    idp_initiated=True
)

# specify the metadata as a URL
cognito.UserPoolIdentityProviderSaml(self, "userpoolidpUrl",
    user_pool=userpool,
    metadata=cognito.UserPoolIdentityProviderSamlMetadata.url("https://my-metadata-url.com")
)
```

Attribute mapping allows mapping attributes provided by the third-party identity providers to [standard and custom
attributes](#Attributes) of the user pool. Learn more about [Specifying Identity Provider Attribute Mappings for Your
User Pool](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html).

The following code shows how different attributes provided by 'Login With Amazon' can be mapped to standard and custom
user pool attributes.

```python
userpool = cognito.UserPool(self, "Pool")

cognito.UserPoolIdentityProviderAmazon(self, "Amazon",
    client_id="amzn-client-id",
    client_secret="amzn-client-secret",
    user_pool=userpool,
    attribute_mapping=cognito.AttributeMapping(
        email=cognito.ProviderAttribute.AMAZON_EMAIL,
        website=cognito.ProviderAttribute.other("url"),  # use other() when an attribute is not pre-defined in the CDK
        custom={
            # custom user pool attributes go here
            "unique_id": cognito.ProviderAttribute.AMAZON_USER_ID
        }
    )
)
```

### App Clients

An app is an entity within a user pool that has permission to call unauthenticated APIs (APIs that do not have an
authenticated user), such as APIs to register, sign in, and handle forgotten passwords. To call these APIs, you need an
app client ID and an optional client secret. Read [Configuring a User Pool App
Client](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-client-apps.html) to learn more.

The following code creates an app client and retrieves the client id -

```python
pool = cognito.UserPool(self, "pool")
client = pool.add_client("customer-app-client")
client_id = client.user_pool_client_id
```

Existing app clients can be imported into the CDK app using the `UserPoolClient.fromUserPoolClientId()` API. For new
and imported user pools, clients can also be created via the `UserPoolClient` constructor, as so -

```python
imported_pool = cognito.UserPool.from_user_pool_id(self, "imported-pool", "us-east-1_oiuR12Abd")
cognito.UserPoolClient(self, "customer-app-client",
    user_pool=imported_pool
)
```

Clients can be configured with authentication flows. Authentication flows allow users on a client to be authenticated
with a user pool. Cognito user pools provide several different types of authentication, such as, SRP (Secure
Remote Password) authentication, username-and-password authentication, etc. Learn more about this at [UserPool Authentication
Flow](https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow.html).

The following code configures a client to use both SRP and username-and-password authentication -

```python
pool = cognito.UserPool(self, "pool")
pool.add_client("app-client",
    auth_flows=cognito.AuthFlow(
        user_password=True,
        user_srp=True
    )
)
```

Custom authentication protocols can be configured by setting the `custom` property under `authFlow` and defining lambda
functions for the corresponding user pool [triggers](#lambda-triggers). Learn more at [Custom Authentication
Flow](https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow.html#amazon-cognito-user-pools-custom-authentication-flow).

Choice-based authentication can be configured by setting the `user` property under `authFlow`. This enables the
`USER_AUTH` authentication flow. Learn more at [Choice-based authentication](https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice).

In addition to these authentication mechanisms, Cognito user pools also support using OAuth 2.0 framework for
authenticating users. User pool clients can be configured with OAuth 2.0 authorization flows and scopes. Learn more
about the [OAuth 2.0 authorization framework](https://tools.ietf.org/html/rfc6749) and [Cognito user pool's
implementation of
OAuth2.0](https://aws.amazon.com/blogs/mobile/understanding-amazon-cognito-user-pool-oauth-2-0-grants/).

The following code configures an app client with the authorization code grant flow and registers the the app's welcome
page as a callback (or redirect) URL. It also configures the access token scope to 'openid'. All of these concepts can
be found in the [OAuth 2.0 RFC](https://tools.ietf.org/html/rfc6749).

```python
pool = cognito.UserPool(self, "Pool")
pool.add_client("app-client",
    o_auth=cognito.OAuthSettings(
        flows=cognito.OAuthFlows(
            authorization_code_grant=True
        ),
        scopes=[cognito.OAuthScope.OPENID],
        callback_urls=["https://my-app-domain.com/welcome"],
        logout_urls=["https://my-app-domain.com/signin"]
    )
)
```

To set a default redirect URI, use the `defaultRedirectUri` property.
Its value must be present in the `callbackUrls` list.

```python
pool = cognito.UserPool(self, "Pool")
pool.add_client("app-client",
    o_auth=cognito.OAuthSettings(
        flows=cognito.OAuthFlows(
            authorization_code_grant=True
        ),
        scopes=[cognito.OAuthScope.OPENID],
        default_redirect_uri="https://my-app-domain.com/welcome",
        callback_urls=["https://my-app-domain.com/welcome", "https://my-app-domain.com/hello"],
        logout_urls=["https://my-app-domain.com/signin"]
    )
)
```

An app client can be configured to prevent user existence errors. This
instructs the Cognito authentication API to return generic authentication
failure responses instead of an UserNotFoundException. By default, the flag
is not set, which means the CloudFormation default (false) will be used. See the
[documentation](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-managing-errors.html)
for the full details on the behavior of this flag.

```python
pool = cognito.UserPool(self, "Pool")
pool.add_client("app-client",
    prevent_user_existence_errors=True
)
```

All identity providers created in the CDK app are automatically registered into the corresponding user pool. All app
clients created in the CDK have all of the identity providers enabled by default. The 'Cognito' identity provider,
that allows users to register and sign in directly with the Cognito user pool, is also enabled by default.
Alternatively, the list of supported identity providers for a client can be explicitly specified -

```python
pool = cognito.UserPool(self, "Pool")
pool.add_client("app-client",
    # ...
    supported_identity_providers=[cognito.UserPoolClientIdentityProvider.AMAZON, cognito.UserPoolClientIdentityProvider.COGNITO
    ]
)
```

If the identity provider and the app client are created in the same stack, specify the dependency between both constructs to
make sure that the identity provider already exists when the app client will be created. The app client cannot handle the
dependency to the identity provider automatically because the client does not have access to the provider's construct.

```python
pool = cognito.UserPool(self, "Pool")
provider = cognito.UserPoolIdentityProviderAmazon(self, "Amazon",
    user_pool=pool,
    client_id="amzn-client-id",
    client_secret="amzn-client-secret"
)

client = pool.add_client("app-client",
    # ...
    supported_identity_providers=[cognito.UserPoolClientIdentityProvider.AMAZON
    ]
)

client.node.add_dependency(provider)
```

The property `authSessionValidity` is the session token for each API request in the authentication flow.
Valid duration is from 3 to 15 minutes.

```python
pool = cognito.UserPool(self, "Pool")
pool.add_client("app-client",
    # ...
    auth_session_validity=Duration.minutes(15)
)
```

In accordance with the OIDC open standard, Cognito user pool clients provide access tokens, ID tokens and refresh tokens.
More information is available at [Using Tokens with User Pools](https://docs.aws.amazon.com/en_us/cognito/latest/developerguide/amazon-cognito-user-pools-using-tokens-with-identity-providers.html).
The expiration time for these tokens can be configured as shown below.

```python
pool = cognito.UserPool(self, "Pool")
pool.add_client("app-client",
    # ...
    access_token_validity=Duration.minutes(60),
    id_token_validity=Duration.minutes(60),
    refresh_token_validity=Duration.days(30)
)
```

Clients can (and should) be allowed to read and write relevant user attributes only. Usually every client can be allowed to
read the `given_name` attribute but not every client should be allowed to set the `email_verified` attribute.
The same criteria applies for both standard and custom attributes, more info is available at
[Attribute Permissions and Scopes](https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-attribute-permissions-and-scopes).
The default behaviour is to allow read and write permissions on all attributes. The following code shows how this can be
configured for a client.

```python
pool = cognito.UserPool(self, "Pool")

client_write_attributes = (cognito.ClientAttributes()).with_standard_attributes(fullname=True, email=True).with_custom_attributes("favoritePizza", "favoriteBeverage")

client_read_attributes = client_write_attributes.with_standard_attributes(email_verified=True).with_custom_attributes("pointsEarned")

pool.add_client("app-client",
    # ...
    read_attributes=client_read_attributes,
    write_attributes=client_write_attributes
)
```

[Token revocation](https://docs.aws.amazon.com/cognito/latest/developerguide/token-revocation.html)
can be configured to be able to revoke refresh tokens in app clients. By default, token revocation is enabled for new user
pools. The property can be used to enable the token revocation in existing app clients or to change the default behavior.

```python
pool = cognito.UserPool(self, "Pool")
pool.add_client("app-client",
    # ...
    enable_token_revocation=True
)
```

User Pool clients can generate a client ID as well as a client secret, to support more advanced authentication workflows.

To create a client with an autogenerated client secret, pass the `generateSecret: true` prop:

```python
# imported_pool: cognito.UserPool


user_pool_client = cognito.UserPoolClient(self, "UserPoolClient",
    user_pool=imported_pool,
    generate_secret=True
)

# Allows you to pass the generated secret to other pieces of infrastructure
secret = user_pool_client.user_pool_client_secret
```

If you set `enablePropagateAdditionalUserContextData: true`, you can collect and pass
information about your user's session to Amazon Cognito advanced security
when you use the API to sign them up, sign them in, and reset their password.

```python
# imported_pool: cognito.UserPool


user_pool_client = cognito.UserPoolClient(self, "UserPoolClient",
    user_pool=imported_pool,
    generate_secret=True,
    enable_propagate_additional_user_context_data=True
)
```

See [Adding user device and session data to API requests](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-adaptive-authentication.html#user-pool-settings-adaptive-authentication-device-fingerprint) for more information.

### Resource Servers

A resource server is a server for access-protected resources. It handles authenticated requests from an app that has an
access token. See [Defining Resource
Servers](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html)
for more information.

An application may choose to model custom permissions via OAuth. Resource Servers provide this capability via custom scopes
that are attached to an app client. The following example sets up a resource server for the 'users' resource for two different
app clients and configures the clients to use these scopes.

```python
pool = cognito.UserPool(self, "Pool")

read_only_scope = cognito.ResourceServerScope(scope_name="read", scope_description="Read-only access")
full_access_scope = cognito.ResourceServerScope(scope_name="*", scope_description="Full access")

user_server = pool.add_resource_server("ResourceServer",
    identifier="users",
    scopes=[read_only_scope, full_access_scope]
)

read_only_client = pool.add_client("read-only-client",
    # ...
    o_auth=cognito.OAuthSettings(
        # ...
        scopes=[cognito.OAuthScope.resource_server(user_server, read_only_scope)]
    )
)

full_access_client = pool.add_client("full-access-client",
    # ...
    o_auth=cognito.OAuthSettings(
        # ...
        scopes=[cognito.OAuthScope.resource_server(user_server, full_access_scope)]
    )
)
```

### Domains

After setting up an [app client](#app-clients), the address for the user pool's sign-up and sign-in webpages can be
configured using domains. There are two ways to set up a domain - either the Amazon Cognito hosted domain can be chosen
with an available domain prefix, or a custom domain name can be chosen. The custom domain must be one that is already
owned, and whose certificate is registered in AWS Certificate Manager.

The following code sets up a user pool domain in Amazon Cognito hosted domain with the prefix 'my-awesome-app', and
another domain with the custom domain 'user.myapp.com' -

```python
pool = cognito.UserPool(self, "Pool")

pool.add_domain("CognitoDomain",
    cognito_domain=cognito.CognitoDomainOptions(
        domain_prefix="my-awesome-app"
    )
)

certificate_arn = "arn:aws:acm:us-east-1:123456789012:certificate/11-3336f1-44483d-adc7-9cd375c5169d"

domain_cert = certificatemanager.Certificate.from_certificate_arn(self, "domainCert", certificate_arn)
pool.add_domain("CustomDomain",
    custom_domain=cognito.CustomDomainOptions(
        domain_name="user.myapp.com",
        certificate=domain_cert
    )
)
```

Read more about [Using the Amazon Cognito
Domain](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-assign-domain-prefix.html) and [Using Your Own
Domain](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-add-custom-domain.html).

You can use the [managed login](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-managed-login.html) page provided by Amazon Cognito to sign in users. The managed login page has two versions: a classic version and a new version. You can switch between the two versions by using the `managedLoginVersion` property.

```python
pool = cognito.UserPool(self, "Pool")

# Use the new managed login page
pool.add_domain("CognitoDomainWithBlandingDesignManagedLogin",
    cognito_domain=cognito.CognitoDomainOptions(
        domain_prefix="blanding-design-ui"
    ),
    managed_login_version=cognito.ManagedLoginVersion.NEWER_MANAGED_LOGIN
)

# Use the classic hosted UI
pool.add_domain("DomainWithClassicHostedUi",
    cognito_domain=cognito.CognitoDomainOptions(
        domain_prefix="classic-hosted-ui"
    ),
    managed_login_version=cognito.ManagedLoginVersion.CLASSIC_HOSTED_UI
)
```

The `signInUrl()` methods returns the fully qualified URL to the login page for the user pool. This page comes from the
hosted UI configured with Cognito. Learn more at [Hosted UI with the Amazon Cognito
Console](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-integration.html#cognito-user-pools-create-an-app-integration).

```python
userpool = cognito.UserPool(self, "UserPool")
client = userpool.add_client("Client",
    # ...
    o_auth=cognito.OAuthSettings(
        flows=cognito.OAuthFlows(
            implicit_code_grant=True
        ),
        callback_urls=["https://myapp.com/home", "https://myapp.com/users"
        ]
    )
)
domain = userpool.add_domain("Domain")
sign_in_url = domain.sign_in_url(client,
    redirect_uri="https://myapp.com/home"
)
```

Existing domains can be imported into CDK apps using `UserPoolDomain.fromDomainName()` API

```python
my_user_pool_domain = cognito.UserPoolDomain.from_domain_name(self, "my-user-pool-domain", "domain-name")
```

To get the domain name of the CloudFront distribution associated with the user pool domain, use `cloudFrontEndpoint` method.

```python
userpool = cognito.UserPool(self, "UserPool")
domain = userpool.add_domain("Domain",
    cognito_domain=cognito.CognitoDomainOptions(
        domain_prefix="my-awesome-app"
    )
)

CfnOutput(self, "CloudFrontEndpoint",
    value=domain.cloud_front_endpoint
)
```

### Deletion protection

Deletion protection can be enabled on a user pool to prevent accidental deletion:

```python
userpool = cognito.UserPool(self, "UserPool",
    # ...
    deletion_protection=True
)
```

By default deletion protection is disabled.

### `email_verified` Attribute Mapping

If you use a third-party identity provider, you can specify the `email_verified` attribute in attributeMapping.

```python
userpool = cognito.UserPool(self, "Pool")

cognito.UserPoolIdentityProviderGoogle(self, "google",
    user_pool=userpool,
    client_id="google-client-id",
    attribute_mapping=cognito.AttributeMapping(
        email=cognito.ProviderAttribute.GOOGLE_EMAIL,
        email_verified=cognito.ProviderAttribute.GOOGLE_EMAIL_VERIFIED
    )
)
```

### User Pool Group

Support for groups in Amazon Cognito user pools enables you to create and manage groups and add users to groups.
Use groups to create collections of users to manage their permissions or to represent different types of users.

You can assign an AWS Identity and Access Management (IAM) role to a group to define the permissions for members of a group.

For more information, see [Adding groups to a user pool](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-user-groups.html).

```python
# user_pool: cognito.UserPool
# role: iam.Role


cognito.UserPoolGroup(self, "UserPoolGroup",
    user_pool=user_pool,
    group_name="my-group-name",
    precedence=1,
    role=role
)

# You can also add a group by using addGroup method.
user_pool.add_group("AnotherUserPoolGroup",
    group_name="another-group-name"
)
```

### Analytics Configuration

User pool clients can be configured with Amazon Pinpoint analytics to collect user activity metrics. This integration enables you to track user engagement and campaign effectiveness.

📝 Note: Amazon Pinpoint isn't available in all AWS Regions. For a list of available Regions, see [Amazon Cognito and Amazon Pinpoint Region availability](https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-pinpoint-integration.html#cognito-user-pools-find-region-mappings).

The following example shows how to configure analytics for a user pool client:

#### When specifying a Pinpoint application from the same account

If you specify the `application` property, do not specify the `applicationId`, `externalId`, or `roleArn` properties.

```python
import aws_cdk.aws_pinpoint as pinpoint

# user_pool: cognito.UserPool
# pinpoint_app: pinpoint.CfnApp
# pinpoint_role: iam.Role


cognito.UserPoolClient(self, "Client",
    user_pool=user_pool,
    analytics=cognito.AnalyticsConfiguration(
        # Your Pinpoint project
        application=pinpoint_app,

        # Whether to include user data in analytics events
        share_user_data=True
    )
)
```

#### When specifying a Pinpoint application from a different account

If you specify the `applicationId`, `externalId`, or `roleArn` properties, do not specify the `application` property.
(In this case, the `applicationId`, `externalId`, and `roleArn` must all be specified.)

Those three attributes are for the cases when Cognito user pool need to be connected to Pinpoint app in other account.

```python
import aws_cdk.aws_pinpoint as pinpoint

# user_pool: cognito.UserPool
# pinpoint_app: pinpoint.CfnApp
# pinpoint_role: iam.Role


cognito.UserPoolClient(self, "Client",
    user_pool=user_pool,
    analytics=cognito.AnalyticsConfiguration(
        # Your Pinpoint project ID
        application_id=pinpoint_app.ref,

        # External ID for the IAM role
        external_id="sample-external-id",

        # IAM role that Cognito can assume to publish to Pinpoint
        role=pinpoint_role,

        # Whether to include user data in analytics events
        share_user_data=True
    )
)
```
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    CfnResource as _CfnResource_9df397a6,
    CfnTag as _CfnTag_f6864754,
    Duration as _Duration_4839e8c3,
    IInspectable as _IInspectable_c2943556,
    IResolvable as _IResolvable_da3f097b,
    IResource as _IResource_c80c4260,
    ITaggable as _ITaggable_36806126,
    ITaggableV2 as _ITaggableV2_4e6798f8,
    RemovalPolicy as _RemovalPolicy_9f93c814,
    Resource as _Resource_45bc6135,
    SecretValue as _SecretValue_3dd0ddae,
    TagManager as _TagManager_0a598cb3,
    TreeInspector as _TreeInspector_488e0dd5,
)
from ..aws_certificatemanager import ICertificate as _ICertificate_c194c70b
from ..aws_iam import (
    Grant as _Grant_a7ae64f8,
    IGrantable as _IGrantable_71c4f5de,
    IRole as _IRole_235f5d8e,
)
from ..aws_kms import IKey as _IKey_5f11635f
from ..aws_lambda import IFunction as _IFunction_6adb0ab8
from ..aws_pinpoint import CfnApp as _CfnApp_e8bac60b


@jsii.enum(jsii_type="aws-cdk-lib.aws_cognito.AccountRecovery")
class AccountRecovery(enum.Enum):
    '''How will a user be able to recover their account?

    When a user forgets their password, they can have a code sent to their verified email or verified phone to recover their account.
    You can choose the preferred way to send codes below.
    We recommend not allowing phone to be used for both password resets and multi-factor authentication (MFA).

    :see: https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html
    :exampleMetadata: infused

    Example::

        cognito.UserPool(self, "UserPool",
            # ...
            account_recovery=cognito.AccountRecovery.EMAIL_ONLY
        )
    '''

    EMAIL_AND_PHONE_WITHOUT_MFA = "EMAIL_AND_PHONE_WITHOUT_MFA"
    '''Email if available, otherwise phone, but don’t allow a user to reset their password via phone if they are also using it for MFA.'''
    PHONE_WITHOUT_MFA_AND_EMAIL = "PHONE_WITHOUT_MFA_AND_EMAIL"
    '''Phone if available, otherwise email, but don’t allow a user to reset their password via phone if they are also using it for MFA.'''
    EMAIL_ONLY = "EMAIL_ONLY"
    '''Email only.'''
    PHONE_ONLY_WITHOUT_MFA = "PHONE_ONLY_WITHOUT_MFA"
    '''Phone only, but don’t allow a user to reset their password via phone if they are also using it for MFA.'''
    PHONE_AND_EMAIL = "PHONE_AND_EMAIL"
    '''(Not Recommended) Phone if available, otherwise email, and do allow a user to reset their password via phone if they are also using it for MFA.'''
    NONE = "NONE"
    '''None – users will have to contact an administrator to reset their passwords.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_cognito.AdvancedSecurityMode")
class AdvancedSecurityMode(enum.Enum):
    '''(deprecated) The different ways in which a user pool's Advanced Security Mode can be configured.

    :deprecated: Advanced Security Mode is deprecated due to user pool feature plans. Use StandardThreatProtectionMode and CustomThreatProtectionMode to set Thread Protection level.

    :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-userpooladdons.html
    :stability: deprecated
    :exampleMetadata: infused

    Example::

        cognito.UserPool(self, "myuserpool",
            # ...
            advanced_security_mode=cognito.AdvancedSecurityMode.ENFORCED
        )
    '''

    ENFORCED = "ENFORCED"
    '''(deprecated) Enable advanced security mode.

    :stability: deprecated
    '''
    AUDIT = "AUDIT"
    '''(deprecated) gather metrics on detected risks without taking action.

    Metrics are published to Amazon CloudWatch

    :stability: deprecated
    '''
    OFF = "OFF"
    '''(deprecated) Advanced security mode is disabled.

    :stability: deprecated
    '''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.AllowedFirstAuthFactors",
    jsii_struct_bases=[],
    name_mapping={
        "password": "password",
        "email_otp": "emailOtp",
        "passkey": "passkey",
        "sms_otp": "smsOtp",
    },
)
class AllowedFirstAuthFactors:
    def __init__(
        self,
        *,
        password: builtins.bool,
        email_otp: typing.Optional[builtins.bool] = None,
        passkey: typing.Optional[builtins.bool] = None,
        sms_otp: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''The types of authentication that you want to allow for users' first authentication prompt.

        :param password: Whether the password authentication is allowed. This must be true.
        :param email_otp: Whether the email message one-time password is allowed. Default: false
        :param passkey: Whether the Passkey (WebAuthn) is allowed. Default: false
        :param sms_otp: Whether the SMS message one-time password is allowed. Default: false

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice
        :exampleMetadata: infused

        Example::

            cognito.UserPool(self, "myuserpool",
                sign_in_policy=cognito.SignInPolicy(
                    allowed_first_auth_factors=cognito.AllowedFirstAuthFactors(password=True, passkey=True)
                ),
                passkey_relying_party_id="auth.example.com",
                passkey_user_verification=cognito.PasskeyUserVerification.REQUIRED
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a30a69cc954e920b5bb7f1163c7b6bd8507e3477eca92e83467d77025b4258f)
            check_type(argname="argument password", value=password, expected_type=type_hints["password"])
            check_type(argname="argument email_otp", value=email_otp, expected_type=type_hints["email_otp"])
            check_type(argname="argument passkey", value=passkey, expected_type=type_hints["passkey"])
            check_type(argname="argument sms_otp", value=sms_otp, expected_type=type_hints["sms_otp"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "password": password,
        }
        if email_otp is not None:
            self._values["email_otp"] = email_otp
        if passkey is not None:
            self._values["passkey"] = passkey
        if sms_otp is not None:
            self._values["sms_otp"] = sms_otp

    @builtins.property
    def password(self) -> builtins.bool:
        '''Whether the password authentication is allowed.

        This must be true.
        '''
        result = self._values.get("password")
        assert result is not None, "Required property 'password' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def email_otp(self) -> typing.Optional[builtins.bool]:
        '''Whether the email message one-time password is allowed.

        :default: false
        '''
        result = self._values.get("email_otp")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def passkey(self) -> typing.Optional[builtins.bool]:
        '''Whether the Passkey (WebAuthn) is allowed.

        :default: false
        '''
        result = self._values.get("passkey")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def sms_otp(self) -> typing.Optional[builtins.bool]:
        '''Whether the SMS message one-time password is allowed.

        :default: false
        '''
        result = self._values.get("sms_otp")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AllowedFirstAuthFactors(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.AnalyticsConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "application": "application",
        "application_id": "applicationId",
        "external_id": "externalId",
        "role": "role",
        "share_user_data": "shareUserData",
    },
)
class AnalyticsConfiguration:
    def __init__(
        self,
        *,
        application: typing.Optional[_CfnApp_e8bac60b] = None,
        application_id: typing.Optional[builtins.str] = None,
        external_id: typing.Optional[builtins.str] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
        share_user_data: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''The settings for Amazon Pinpoint analytics configuration.

        With an analytics configuration, your application can collect user-activity metrics for user notifications with an Amazon Pinpoint campaign.
        Amazon Pinpoint isn't available in all AWS Regions.
        For a list of available Regions, see Amazon Cognito and Amazon Pinpoint Region availability: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-pinpoint-integration.html#cognito-user-pools-find-region-mappings.

        :param application: The Amazon Pinpoint project that you want to connect to your user pool app client. Amazon Cognito publishes events to the Amazon Pinpoint project. You can also configure your application to pass an endpoint ID in the ``AnalyticsMetadata`` parameter of sign-in operations. The endpoint ID is information about the destination for push notifications. Default: - no configuration, you need to specify either ``application`` or all of ``applicationId``, ``externalId``, and ``role``.
        :param application_id: Your Amazon Pinpoint project ID. Default: - no configuration, you need to specify either this property along with ``externalId`` and ``role`` or ``application``.
        :param external_id: The external ID of the role that Amazon Cognito assumes to send analytics data to Amazon Pinpoint. More info here: https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html Default: - no configuration, you need to specify either this property along with ``applicationId`` and ``role`` or ``application``.
        :param role: The IAM role that has the permissions required for Amazon Cognito to publish events to Amazon Pinpoint analytics. Default: - no configuration, you need to specify either this property along with ``applicationId`` and ``externalId`` or ``application``.
        :param share_user_data: If ``true``, Amazon Cognito includes user data in the events that it publishes to Amazon Pinpoint analytics. Default: - false

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_pinpoint as pinpoint
            
            # user_pool: cognito.UserPool
            # pinpoint_app: pinpoint.CfnApp
            # pinpoint_role: iam.Role
            
            
            cognito.UserPoolClient(self, "Client",
                user_pool=user_pool,
                analytics=cognito.AnalyticsConfiguration(
                    # Your Pinpoint project
                    application=pinpoint_app,
            
                    # Whether to include user data in analytics events
                    share_user_data=True
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f67277ee392b3c256b3bd87e4afcb7bb83df8d226097757f9c92610348c4456b)
            check_type(argname="argument application", value=application, expected_type=type_hints["application"])
            check_type(argname="argument application_id", value=application_id, expected_type=type_hints["application_id"])
            check_type(argname="argument external_id", value=external_id, expected_type=type_hints["external_id"])
            check_type(argname="argument role", value=role, expected_type=type_hints["role"])
            check_type(argname="argument share_user_data", value=share_user_data, expected_type=type_hints["share_user_data"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if application is not None:
            self._values["application"] = application
        if application_id is not None:
            self._values["application_id"] = application_id
        if external_id is not None:
            self._values["external_id"] = external_id
        if role is not None:
            self._values["role"] = role
        if share_user_data is not None:
            self._values["share_user_data"] = share_user_data

    @builtins.property
    def application(self) -> typing.Optional[_CfnApp_e8bac60b]:
        '''The Amazon Pinpoint project that you want to connect to your user pool app client.

        Amazon Cognito publishes events to the Amazon Pinpoint project.
        You can also configure your application to pass an endpoint ID in the ``AnalyticsMetadata`` parameter of sign-in operations.
        The endpoint ID is information about the destination for push notifications.

        :default: - no configuration, you need to specify either ``application`` or all of ``applicationId``, ``externalId``, and ``role``.
        '''
        result = self._values.get("application")
        return typing.cast(typing.Optional[_CfnApp_e8bac60b], result)

    @builtins.property
    def application_id(self) -> typing.Optional[builtins.str]:
        '''Your Amazon Pinpoint project ID.

        :default: - no configuration, you need to specify either this property along with ``externalId`` and ``role`` or ``application``.
        '''
        result = self._values.get("application_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def external_id(self) -> typing.Optional[builtins.str]:
        '''The external ID of the role that Amazon Cognito assumes to send analytics data to Amazon Pinpoint.

        More info here: https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html

        :default: - no configuration, you need to specify either this property along with ``applicationId`` and ``role`` or ``application``.
        '''
        result = self._values.get("external_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The IAM role that has the permissions required for Amazon Cognito to publish events to Amazon Pinpoint analytics.

        :default: - no configuration, you need to specify either this property along with ``applicationId`` and ``externalId`` or ``application``.
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def share_user_data(self) -> typing.Optional[builtins.bool]:
        '''If ``true``, Amazon Cognito includes user data in the events that it publishes to Amazon Pinpoint analytics.

        :default: - false
        '''
        result = self._values.get("share_user_data")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AnalyticsConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.AttributeMapping",
    jsii_struct_bases=[],
    name_mapping={
        "address": "address",
        "birthdate": "birthdate",
        "custom": "custom",
        "email": "email",
        "email_verified": "emailVerified",
        "family_name": "familyName",
        "fullname": "fullname",
        "gender": "gender",
        "given_name": "givenName",
        "last_update_time": "lastUpdateTime",
        "locale": "locale",
        "middle_name": "middleName",
        "nickname": "nickname",
        "phone_number": "phoneNumber",
        "preferred_username": "preferredUsername",
        "profile_page": "profilePage",
        "profile_picture": "profilePicture",
        "timezone": "timezone",
        "website": "website",
    },
)
class AttributeMapping:
    def __init__(
        self,
        *,
        address: typing.Optional["ProviderAttribute"] = None,
        birthdate: typing.Optional["ProviderAttribute"] = None,
        custom: typing.Optional[typing.Mapping[builtins.str, "ProviderAttribute"]] = None,
        email: typing.Optional["ProviderAttribute"] = None,
        email_verified: typing.Optional["ProviderAttribute"] = None,
        family_name: typing.Optional["ProviderAttribute"] = None,
        fullname: typing.Optional["ProviderAttribute"] = None,
        gender: typing.Optional["ProviderAttribute"] = None,
        given_name: typing.Optional["ProviderAttribute"] = None,
        last_update_time: typing.Optional["ProviderAttribute"] = None,
        locale: typing.Optional["ProviderAttribute"] = None,
        middle_name: typing.Optional["ProviderAttribute"] = None,
        nickname: typing.Optional["ProviderAttribute"] = None,
        phone_number: typing.Optional["ProviderAttribute"] = None,
        preferred_username: typing.Optional["ProviderAttribute"] = None,
        profile_page: typing.Optional["ProviderAttribute"] = None,
        profile_picture: typing.Optional["ProviderAttribute"] = None,
        timezone: typing.Optional["ProviderAttribute"] = None,
        website: typing.Optional["ProviderAttribute"] = None,
    ) -> None:
        '''The mapping of user pool attributes to the attributes provided by the identity providers.

        :param address: The user's postal address is a required attribute. Default: - not mapped
        :param birthdate: The user's birthday. Default: - not mapped
        :param custom: Specify custom attribute mapping here and mapping for any standard attributes not supported yet. Default: - no custom attribute mapping
        :param email: The user's e-mail address. Default: - not mapped
        :param email_verified: The user's e-mail address is verification. Default: - not mapped
        :param family_name: The surname or last name of user. Default: - not mapped
        :param fullname: The user's full name in displayable form. Default: - not mapped
        :param gender: The user's gender. Default: - not mapped
        :param given_name: The user's first name or give name. Default: - not mapped
        :param last_update_time: Time, the user's information was last updated. Default: - not mapped
        :param locale: The user's locale. Default: - not mapped
        :param middle_name: The user's middle name. Default: - not mapped
        :param nickname: The user's nickname or casual name. Default: - not mapped
        :param phone_number: The user's telephone number. Default: - not mapped
        :param preferred_username: The user's preferred username. Default: - not mapped
        :param profile_page: The URL to the user's profile page. Default: - not mapped
        :param profile_picture: The URL to the user's profile picture. Default: - not mapped
        :param timezone: The user's time zone. Default: - not mapped
        :param website: The URL to the user's web page or blog. Default: - not mapped

        :exampleMetadata: infused

        Example::

            userpool = cognito.UserPool(self, "Pool")
            
            cognito.UserPoolIdentityProviderAmazon(self, "Amazon",
                client_id="amzn-client-id",
                client_secret="amzn-client-secret",
                user_pool=userpool,
                attribute_mapping=cognito.AttributeMapping(
                    email=cognito.ProviderAttribute.AMAZON_EMAIL,
                    website=cognito.ProviderAttribute.other("url"),  # use other() when an attribute is not pre-defined in the CDK
                    custom={
                        # custom user pool attributes go here
                        "unique_id": cognito.ProviderAttribute.AMAZON_USER_ID
                    }
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1994c9f3057f350dfde37c21bef42d2ad1a87ae2900a0e48fd7c2506ddbeca5d)
            check_type(argname="argument address", value=address, expected_type=type_hints["address"])
            check_type(argname="argument birthdate", value=birthdate, expected_type=type_hints["birthdate"])
            check_type(argname="argument custom", value=custom, expected_type=type_hints["custom"])
            check_type(argname="argument email", value=email, expected_type=type_hints["email"])
            check_type(argname="argument email_verified", value=email_verified, expected_type=type_hints["email_verified"])
            check_type(argname="argument family_name", value=family_name, expected_type=type_hints["family_name"])
            check_type(argname="argument fullname", value=fullname, expected_type=type_hints["fullname"])
            check_type(argname="argument gender", value=gender, expected_type=type_hints["gender"])
            check_type(argname="argument given_name", value=given_name, expected_type=type_hints["given_name"])
            check_type(argname="argument last_update_time", value=last_update_time, expected_type=type_hints["last_update_time"])
            check_type(argname="argument locale", value=locale, expected_type=type_hints["locale"])
            check_type(argname="argument middle_name", value=middle_name, expected_type=type_hints["middle_name"])
            check_type(argname="argument nickname", value=nickname, expected_type=type_hints["nickname"])
            check_type(argname="argument phone_number", value=phone_number, expected_type=type_hints["phone_number"])
            check_type(argname="argument preferred_username", value=preferred_username, expected_type=type_hints["preferred_username"])
            check_type(argname="argument profile_page", value=profile_page, expected_type=type_hints["profile_page"])
            check_type(argname="argument profile_picture", value=profile_picture, expected_type=type_hints["profile_picture"])
            check_type(argname="argument timezone", value=timezone, expected_type=type_hints["timezone"])
            check_type(argname="argument website", value=website, expected_type=type_hints["website"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if address is not None:
            self._values["address"] = address
        if birthdate is not None:
            self._values["birthdate"] = birthdate
        if custom is not None:
            self._values["custom"] = custom
        if email is not None:
            self._values["email"] = email
        if email_verified is not None:
            self._values["email_verified"] = email_verified
        if family_name is not None:
            self._values["family_name"] = family_name
        if fullname is not None:
            self._values["fullname"] = fullname
        if gender is not None:
            self._values["gender"] = gender
        if given_name is not None:
            self._values["given_name"] = given_name
        if last_update_time is not None:
            self._values["last_update_time"] = last_update_time
        if locale is not None:
            self._values["locale"] = locale
        if middle_name is not None:
            self._values["middle_name"] = middle_name
        if nickname is not None:
            self._values["nickname"] = nickname
        if phone_number is not None:
            self._values["phone_number"] = phone_number
        if preferred_username is not None:
            self._values["preferred_username"] = preferred_username
        if profile_page is not None:
            self._values["profile_page"] = profile_page
        if profile_picture is not None:
            self._values["profile_picture"] = profile_picture
        if timezone is not None:
            self._values["timezone"] = timezone
        if website is not None:
            self._values["website"] = website

    @builtins.property
    def address(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's postal address is a required attribute.

        :default: - not mapped
        '''
        result = self._values.get("address")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def birthdate(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's birthday.

        :default: - not mapped
        '''
        result = self._values.get("birthdate")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def custom(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, "ProviderAttribute"]]:
        '''Specify custom attribute mapping here and mapping for any standard attributes not supported yet.

        :default: - no custom attribute mapping
        '''
        result = self._values.get("custom")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "ProviderAttribute"]], result)

    @builtins.property
    def email(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's e-mail address.

        :default: - not mapped
        '''
        result = self._values.get("email")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def email_verified(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's e-mail address is verification.

        :default: - not mapped
        '''
        result = self._values.get("email_verified")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def family_name(self) -> typing.Optional["ProviderAttribute"]:
        '''The surname or last name of user.

        :default: - not mapped
        '''
        result = self._values.get("family_name")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def fullname(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's full name in displayable form.

        :default: - not mapped
        '''
        result = self._values.get("fullname")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def gender(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's gender.

        :default: - not mapped
        '''
        result = self._values.get("gender")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def given_name(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's first name or give name.

        :default: - not mapped
        '''
        result = self._values.get("given_name")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def last_update_time(self) -> typing.Optional["ProviderAttribute"]:
        '''Time, the user's information was last updated.

        :default: - not mapped
        '''
        result = self._values.get("last_update_time")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def locale(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's locale.

        :default: - not mapped
        '''
        result = self._values.get("locale")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def middle_name(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's middle name.

        :default: - not mapped
        '''
        result = self._values.get("middle_name")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def nickname(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's nickname or casual name.

        :default: - not mapped
        '''
        result = self._values.get("nickname")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def phone_number(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's telephone number.

        :default: - not mapped
        '''
        result = self._values.get("phone_number")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def preferred_username(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's preferred username.

        :default: - not mapped
        '''
        result = self._values.get("preferred_username")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def profile_page(self) -> typing.Optional["ProviderAttribute"]:
        '''The URL to the user's profile page.

        :default: - not mapped
        '''
        result = self._values.get("profile_page")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def profile_picture(self) -> typing.Optional["ProviderAttribute"]:
        '''The URL to the user's profile picture.

        :default: - not mapped
        '''
        result = self._values.get("profile_picture")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def timezone(self) -> typing.Optional["ProviderAttribute"]:
        '''The user's time zone.

        :default: - not mapped
        '''
        result = self._values.get("timezone")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    @builtins.property
    def website(self) -> typing.Optional["ProviderAttribute"]:
        '''The URL to the user's web page or blog.

        :default: - not mapped
        '''
        result = self._values.get("website")
        return typing.cast(typing.Optional["ProviderAttribute"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AttributeMapping(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.AuthFlow",
    jsii_struct_bases=[],
    name_mapping={
        "admin_user_password": "adminUserPassword",
        "custom": "custom",
        "user": "user",
        "user_password": "userPassword",
        "user_srp": "userSrp",
    },
)
class AuthFlow:
    def __init__(
        self,
        *,
        admin_user_password: typing.Optional[builtins.bool] = None,
        custom: typing.Optional[builtins.bool] = None,
        user: typing.Optional[builtins.bool] = None,
        user_password: typing.Optional[builtins.bool] = None,
        user_srp: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Types of authentication flow.

        :param admin_user_password: Enable admin based user password authentication flow. Default: false
        :param custom: Enable custom authentication flow. Default: false
        :param user: Enable Choice-based authentication. Default: false
        :param user_password: Enable auth using username & password. Default: false
        :param user_srp: Enable SRP based authentication. Default: false

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow.html
        :exampleMetadata: infused

        Example::

            user_pool = cognito.UserPool(self, "myuserpool",
                sign_in_policy=cognito.SignInPolicy(
                    allowed_first_auth_factors=cognito.AllowedFirstAuthFactors(
                        password=True,  # password authentication must be enabled
                        email_otp=True,  # enables email message one-time password
                        sms_otp=True,  # enables SMS message one-time password
                        passkey=True
                    )
                )
            )
            
            # You should also configure the user pool client with USER_AUTH authentication flow allowed
            user_pool.add_client("myclient",
                auth_flows=cognito.AuthFlow(user=True)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3dd38e6e4617deee919f37d20a9ae635331043b4cf42c8d31fdbb0d3c29baeda)
            check_type(argname="argument admin_user_password", value=admin_user_password, expected_type=type_hints["admin_user_password"])
            check_type(argname="argument custom", value=custom, expected_type=type_hints["custom"])
            check_type(argname="argument user", value=user, expected_type=type_hints["user"])
            check_type(argname="argument user_password", value=user_password, expected_type=type_hints["user_password"])
            check_type(argname="argument user_srp", value=user_srp, expected_type=type_hints["user_srp"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if admin_user_password is not None:
            self._values["admin_user_password"] = admin_user_password
        if custom is not None:
            self._values["custom"] = custom
        if user is not None:
            self._values["user"] = user
        if user_password is not None:
            self._values["user_password"] = user_password
        if user_srp is not None:
            self._values["user_srp"] = user_srp

    @builtins.property
    def admin_user_password(self) -> typing.Optional[builtins.bool]:
        '''Enable admin based user password authentication flow.

        :default: false
        '''
        result = self._values.get("admin_user_password")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def custom(self) -> typing.Optional[builtins.bool]:
        '''Enable custom authentication flow.

        :default: false
        '''
        result = self._values.get("custom")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def user(self) -> typing.Optional[builtins.bool]:
        '''Enable Choice-based authentication.

        :default: false
        '''
        result = self._values.get("user")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def user_password(self) -> typing.Optional[builtins.bool]:
        '''Enable auth using username & password.

        :default: false
        '''
        result = self._values.get("user_password")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def user_srp(self) -> typing.Optional[builtins.bool]:
        '''Enable SRP based authentication.

        :default: false
        '''
        result = self._values.get("user_srp")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuthFlow(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.AutoVerifiedAttrs",
    jsii_struct_bases=[],
    name_mapping={"email": "email", "phone": "phone"},
)
class AutoVerifiedAttrs:
    def __init__(
        self,
        *,
        email: typing.Optional[builtins.bool] = None,
        phone: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Attributes that can be automatically verified for users in a user pool.

        :param email: Whether the email address of the user should be auto verified at sign up. Note: If both ``email`` and ``phone`` is set, Cognito only verifies the phone number. To also verify email, see here - https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html Default: - true, if email is turned on for ``signIn``. false, otherwise.
        :param phone: Whether the phone number of the user should be auto verified at sign up. Default: - true, if phone is turned on for ``signIn``. false, otherwise.

        :exampleMetadata: infused

        Example::

            cognito.UserPool(self, "myuserpool",
                # ...
                # ...
                sign_in_aliases=cognito.SignInAliases(username=True, email=True),
                auto_verify=cognito.AutoVerifiedAttrs(email=True, phone=True)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d7fb7225ae10a05ff7664a35abd96c1ef10430ed4dad8308ba28708c6ad65bf5)
            check_type(argname="argument email", value=email, expected_type=type_hints["email"])
            check_type(argname="argument phone", value=phone, expected_type=type_hints["phone"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if email is not None:
            self._values["email"] = email
        if phone is not None:
            self._values["phone"] = phone

    @builtins.property
    def email(self) -> typing.Optional[builtins.bool]:
        '''Whether the email address of the user should be auto verified at sign up.

        Note: If both ``email`` and ``phone`` is set, Cognito only verifies the phone number. To also verify email, see here -
        https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html

        :default: - true, if email is turned on for ``signIn``. false, otherwise.
        '''
        result = self._values.get("email")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def phone(self) -> typing.Optional[builtins.bool]:
        '''Whether the phone number of the user should be auto verified at sign up.

        :default: - true, if phone is turned on for ``signIn``. false, otherwise.
        '''
        result = self._values.get("phone")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AutoVerifiedAttrs(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.BaseUrlOptions",
    jsii_struct_bases=[],
    name_mapping={"fips": "fips"},
)
class BaseUrlOptions:
    def __init__(self, *, fips: typing.Optional[builtins.bool] = None) -> None:
        '''Options to customize the behaviour of ``baseUrl()``.

        :param fips: Whether to return the FIPS-compliant endpoint. Default: return the standard URL

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            base_url_options = cognito.BaseUrlOptions(
                fips=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__108bae0fda2d22e93d38a433ec7ec439788ef3214362d7282ce6bfca76e72635)
            check_type(argname="argument fips", value=fips, expected_type=type_hints["fips"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if fips is not None:
            self._values["fips"] = fips

    @builtins.property
    def fips(self) -> typing.Optional[builtins.bool]:
        '''Whether to return the FIPS-compliant endpoint.

        :default: return the standard URL
        '''
        result = self._values.get("fips")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseUrlOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnIdentityPool(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPool",
):
    '''The ``AWS::Cognito::IdentityPool`` resource creates an Amazon Cognito identity pool.

    To avoid deleting the resource accidentally from AWS CloudFormation , use `DeletionPolicy Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-deletionpolicy.html>`_ and the `UpdateReplacePolicy Attribute <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-updatereplacepolicy.html>`_ to retain the resource on deletion or replacement.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html
    :cloudformationResource: AWS::Cognito::IdentityPool
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_cognito as cognito
        
        # my_provider: iam.OpenIdConnectProvider
        
        cognito.CfnIdentityPool(self, "IdentityPool",
            open_id_connect_provider_arns=[my_provider.open_id_connect_provider_arn],
            # And the other properties for your identity pool
            allow_unauthenticated_identities=False
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        allow_unauthenticated_identities: typing.Union[builtins.bool, _IResolvable_da3f097b],
        allow_classic_flow: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        cognito_events: typing.Any = None,
        cognito_identity_providers: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnIdentityPool.CognitoIdentityProviderProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        cognito_streams: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnIdentityPool.CognitoStreamsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        developer_provider_name: typing.Optional[builtins.str] = None,
        identity_pool_name: typing.Optional[builtins.str] = None,
        identity_pool_tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        open_id_connect_provider_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        push_sync: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnIdentityPool.PushSyncProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        saml_provider_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        supported_login_providers: typing.Any = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param allow_unauthenticated_identities: Specifies whether the identity pool supports unauthenticated logins.
        :param allow_classic_flow: Enables the Basic (Classic) authentication flow.
        :param cognito_events: The events to configure.
        :param cognito_identity_providers: The Amazon Cognito user pools and their client IDs.
        :param cognito_streams: Configuration options for configuring Amazon Cognito streams.
        :param developer_provider_name: The "domain" Amazon Cognito uses when referencing your users. This name acts as a placeholder that allows your backend and the Amazon Cognito service to communicate about the developer provider. For the ``DeveloperProviderName`` , you can use letters and periods (.), underscores (_), and dashes (-). *Minimum length* : 1 *Maximum length* : 100
        :param identity_pool_name: The name of your Amazon Cognito identity pool. *Minimum length* : 1 *Maximum length* : 128 *Pattern* : ``[\\w\\s+=,.@-]+``
        :param identity_pool_tags: Tags to assign to the identity pool. A tag is a label that you can apply to identity pools to categorize and manage them in different ways, such as by purpose, owner, environment, or other criteria.
        :param open_id_connect_provider_arns: The Amazon Resource Names (ARNs) of the OpenID connect providers.
        :param push_sync: The configuration options to be applied to the identity pool.
        :param saml_provider_arns: The Amazon Resource Names (ARNs) of the Security Assertion Markup Language (SAML) providers.
        :param supported_login_providers: Key-value pairs that map provider names to provider app IDs.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d5156c08d9bb3b0ceca6f4ec485d84f9775d7be40128a6614e1b9cbf098f4c86)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnIdentityPoolProps(
            allow_unauthenticated_identities=allow_unauthenticated_identities,
            allow_classic_flow=allow_classic_flow,
            cognito_events=cognito_events,
            cognito_identity_providers=cognito_identity_providers,
            cognito_streams=cognito_streams,
            developer_provider_name=developer_provider_name,
            identity_pool_name=identity_pool_name,
            identity_pool_tags=identity_pool_tags,
            open_id_connect_provider_arns=open_id_connect_provider_arns,
            push_sync=push_sync,
            saml_provider_arns=saml_provider_arns,
            supported_login_providers=supported_login_providers,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d0046422699f4c95c1d9df0a173b8caf8590dc3fcb3ada38e03f96cba1a359bc)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a3d9795be15a85f1c5807d837aa5f58acf557934826ab8cafaa6e80c3d04eeb1)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrName")
    def attr_name(self) -> builtins.str:
        '''The name of the Amazon Cognito identity pool, returned as a string.

        :cloudformationAttribute: Name
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrName"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="allowUnauthenticatedIdentities")
    def allow_unauthenticated_identities(
        self,
    ) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
        '''Specifies whether the identity pool supports unauthenticated logins.'''
        return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], jsii.get(self, "allowUnauthenticatedIdentities"))

    @allow_unauthenticated_identities.setter
    def allow_unauthenticated_identities(
        self,
        value: typing.Union[builtins.bool, _IResolvable_da3f097b],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__db6ea31d27c717c9763b6408e0f3923cd9b6dc2e5d79b1a66ce864122450159f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allowUnauthenticatedIdentities", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="allowClassicFlow")
    def allow_classic_flow(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Enables the Basic (Classic) authentication flow.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "allowClassicFlow"))

    @allow_classic_flow.setter
    def allow_classic_flow(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c5b513e4bd7ec3f84408bf9a2b747fb0b06357d05747f6825c473e9274958606)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allowClassicFlow", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="cognitoEvents")
    def cognito_events(self) -> typing.Any:
        '''The events to configure.'''
        return typing.cast(typing.Any, jsii.get(self, "cognitoEvents"))

    @cognito_events.setter
    def cognito_events(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__360f21073c65bd16eeda7bcf34ec57609826f7d58ba6dc4fc009286320ef72de)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cognitoEvents", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="cognitoIdentityProviders")
    def cognito_identity_providers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIdentityPool.CognitoIdentityProviderProperty"]]]]:
        '''The Amazon Cognito user pools and their client IDs.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIdentityPool.CognitoIdentityProviderProperty"]]]], jsii.get(self, "cognitoIdentityProviders"))

    @cognito_identity_providers.setter
    def cognito_identity_providers(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIdentityPool.CognitoIdentityProviderProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fbb45d26060f7e1f74033e6a37741666fb843374b7a2b735483dea6f9b824230)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cognitoIdentityProviders", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="cognitoStreams")
    def cognito_streams(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIdentityPool.CognitoStreamsProperty"]]:
        '''Configuration options for configuring Amazon Cognito streams.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIdentityPool.CognitoStreamsProperty"]], jsii.get(self, "cognitoStreams"))

    @cognito_streams.setter
    def cognito_streams(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIdentityPool.CognitoStreamsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__502164af12ab71961d7f4f2e11817973d435bc4577d049f15a4cf62796079879)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cognitoStreams", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="developerProviderName")
    def developer_provider_name(self) -> typing.Optional[builtins.str]:
        '''The "domain" Amazon Cognito uses when referencing your users.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "developerProviderName"))

    @developer_provider_name.setter
    def developer_provider_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f2b5f037f02889c2a585909dfa772a6165c4470d0984cfa622e1374ed7852936)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "developerProviderName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="identityPoolName")
    def identity_pool_name(self) -> typing.Optional[builtins.str]:
        '''The name of your Amazon Cognito identity pool.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "identityPoolName"))

    @identity_pool_name.setter
    def identity_pool_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6010d0ddbf0b577c8e7ea0796913ba4085881df0b32cec744c271c99300165af)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identityPoolName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="identityPoolTags")
    def identity_pool_tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Tags to assign to the identity pool.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "identityPoolTags"))

    @identity_pool_tags.setter
    def identity_pool_tags(
        self,
        value: typing.Optional[typing.List[_CfnTag_f6864754]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__caf91efc577806bf48a176729ecabaf6b30c2d71ecfff869256e7500e4dcb9bc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identityPoolTags", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="openIdConnectProviderArns")
    def open_id_connect_provider_arns(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''The Amazon Resource Names (ARNs) of the OpenID connect providers.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "openIdConnectProviderArns"))

    @open_id_connect_provider_arns.setter
    def open_id_connect_provider_arns(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__61d80fcbd99af6003aec5d0b96e0ec9abdb3d4f6e2f44f67bdd2e746351a3517)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "openIdConnectProviderArns", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="pushSync")
    def push_sync(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIdentityPool.PushSyncProperty"]]:
        '''The configuration options to be applied to the identity pool.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIdentityPool.PushSyncProperty"]], jsii.get(self, "pushSync"))

    @push_sync.setter
    def push_sync(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIdentityPool.PushSyncProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__37451572b0d939f18081b9fd4743a223e16c702244d56835c0e577ed22f85c70)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "pushSync", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="samlProviderArns")
    def saml_provider_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The Amazon Resource Names (ARNs) of the Security Assertion Markup Language (SAML) providers.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "samlProviderArns"))

    @saml_provider_arns.setter
    def saml_provider_arns(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d2aad48eb5d6c1e553867325f44b35ef0b577c7971cf240ed4b646c07303a760)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "samlProviderArns", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="supportedLoginProviders")
    def supported_login_providers(self) -> typing.Any:
        '''Key-value pairs that map provider names to provider app IDs.'''
        return typing.cast(typing.Any, jsii.get(self, "supportedLoginProviders"))

    @supported_login_providers.setter
    def supported_login_providers(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__905e78c8fb6056ab2fc4961e4c0d8d7817993690aff93e2be1384e79b1effe9a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "supportedLoginProviders", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPool.CognitoIdentityProviderProperty",
        jsii_struct_bases=[],
        name_mapping={
            "client_id": "clientId",
            "provider_name": "providerName",
            "server_side_token_check": "serverSideTokenCheck",
        },
    )
    class CognitoIdentityProviderProperty:
        def __init__(
            self,
            *,
            client_id: builtins.str,
            provider_name: builtins.str,
            server_side_token_check: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''``CognitoIdentityProvider`` is a property of the `AWS::Cognito::IdentityPool <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html>`_ resource that represents an Amazon Cognito user pool and its client ID.

            :param client_id: The client ID for the Amazon Cognito user pool.
            :param provider_name: The provider name for an Amazon Cognito user pool. For example: ``cognito-idp.us-east-2.amazonaws.com/us-east-2_123456789`` .
            :param server_side_token_check: TRUE if server-side token validation is enabled for the identity provider’s token. After you set the ``ServerSideTokenCheck`` to TRUE for an identity pool, that identity pool checks with the integrated user pools to make sure the user has not been globally signed out or deleted before the identity pool provides an OIDC token or AWS credentials for the user. If the user is signed out or deleted, the identity pool returns a 400 Not Authorized error.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypool-cognitoidentityprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                cognito_identity_provider_property = cognito.CfnIdentityPool.CognitoIdentityProviderProperty(
                    client_id="clientId",
                    provider_name="providerName",
                
                    # the properties below are optional
                    server_side_token_check=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__98390431af098ce09b1be6f97cb9f1634533afc529f7b7abc1b708017f1c008a)
                check_type(argname="argument client_id", value=client_id, expected_type=type_hints["client_id"])
                check_type(argname="argument provider_name", value=provider_name, expected_type=type_hints["provider_name"])
                check_type(argname="argument server_side_token_check", value=server_side_token_check, expected_type=type_hints["server_side_token_check"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "client_id": client_id,
                "provider_name": provider_name,
            }
            if server_side_token_check is not None:
                self._values["server_side_token_check"] = server_side_token_check

        @builtins.property
        def client_id(self) -> builtins.str:
            '''The client ID for the Amazon Cognito user pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypool-cognitoidentityprovider.html#cfn-cognito-identitypool-cognitoidentityprovider-clientid
            '''
            result = self._values.get("client_id")
            assert result is not None, "Required property 'client_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def provider_name(self) -> builtins.str:
            '''The provider name for an Amazon Cognito user pool.

            For example: ``cognito-idp.us-east-2.amazonaws.com/us-east-2_123456789`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypool-cognitoidentityprovider.html#cfn-cognito-identitypool-cognitoidentityprovider-providername
            '''
            result = self._values.get("provider_name")
            assert result is not None, "Required property 'provider_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def server_side_token_check(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''TRUE if server-side token validation is enabled for the identity provider’s token.

            After you set the ``ServerSideTokenCheck`` to TRUE for an identity pool, that identity pool checks with the integrated user pools to make sure the user has not been globally signed out or deleted before the identity pool provides an OIDC token or AWS credentials for the user.

            If the user is signed out or deleted, the identity pool returns a 400 Not Authorized error.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypool-cognitoidentityprovider.html#cfn-cognito-identitypool-cognitoidentityprovider-serversidetokencheck
            '''
            result = self._values.get("server_side_token_check")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CognitoIdentityProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPool.CognitoStreamsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "role_arn": "roleArn",
            "streaming_status": "streamingStatus",
            "stream_name": "streamName",
        },
    )
    class CognitoStreamsProperty:
        def __init__(
            self,
            *,
            role_arn: typing.Optional[builtins.str] = None,
            streaming_status: typing.Optional[builtins.str] = None,
            stream_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``CognitoStreams`` is a property of the `AWS::Cognito::IdentityPool <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html>`_ resource that defines configuration options for Amazon Cognito streams.

            :param role_arn: The Amazon Resource Name (ARN) of the role Amazon Cognito can assume to publish to the stream. This role must grant access to Amazon Cognito (cognito-sync) to invoke ``PutRecord`` on your Amazon Cognito stream.
            :param streaming_status: Status of the Amazon Cognito streams. Valid values are: ``ENABLED`` or ``DISABLED`` .
            :param stream_name: The name of the Amazon Cognito stream to receive updates. This stream must be in the developer's account and in the same Region as the identity pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypool-cognitostreams.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                cognito_streams_property = cognito.CfnIdentityPool.CognitoStreamsProperty(
                    role_arn="roleArn",
                    streaming_status="streamingStatus",
                    stream_name="streamName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cad9ed028ee87a8ede4635fd244b918bed85d1ccf15c83ee6642fb1dc2245b13)
                check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
                check_type(argname="argument streaming_status", value=streaming_status, expected_type=type_hints["streaming_status"])
                check_type(argname="argument stream_name", value=stream_name, expected_type=type_hints["stream_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if role_arn is not None:
                self._values["role_arn"] = role_arn
            if streaming_status is not None:
                self._values["streaming_status"] = streaming_status
            if stream_name is not None:
                self._values["stream_name"] = stream_name

        @builtins.property
        def role_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the role Amazon Cognito can assume to publish to the stream.

            This role must grant access to Amazon Cognito (cognito-sync) to invoke ``PutRecord`` on your Amazon Cognito stream.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypool-cognitostreams.html#cfn-cognito-identitypool-cognitostreams-rolearn
            '''
            result = self._values.get("role_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def streaming_status(self) -> typing.Optional[builtins.str]:
            '''Status of the Amazon Cognito streams.

            Valid values are: ``ENABLED`` or ``DISABLED`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypool-cognitostreams.html#cfn-cognito-identitypool-cognitostreams-streamingstatus
            '''
            result = self._values.get("streaming_status")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def stream_name(self) -> typing.Optional[builtins.str]:
            '''The name of the Amazon Cognito stream to receive updates.

            This stream must be in the developer's account and in the same Region as the identity pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypool-cognitostreams.html#cfn-cognito-identitypool-cognitostreams-streamname
            '''
            result = self._values.get("stream_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CognitoStreamsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPool.PushSyncProperty",
        jsii_struct_bases=[],
        name_mapping={"application_arns": "applicationArns", "role_arn": "roleArn"},
    )
    class PushSyncProperty:
        def __init__(
            self,
            *,
            application_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
            role_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``PushSync`` is a property of the `AWS::Cognito::IdentityPool <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html>`_ resource that defines the configuration options to be applied to an Amazon Cognito identity pool.

            :param application_arns: The ARNs of the Amazon SNS platform applications that could be used by clients.
            :param role_arn: An IAM role configured to allow Amazon Cognito to call Amazon SNS on behalf of the developer.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypool-pushsync.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                push_sync_property = cognito.CfnIdentityPool.PushSyncProperty(
                    application_arns=["applicationArns"],
                    role_arn="roleArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__539d8a92287340332dc17b7e75525701018eb45d5761572dde506535aa5b92d8)
                check_type(argname="argument application_arns", value=application_arns, expected_type=type_hints["application_arns"])
                check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if application_arns is not None:
                self._values["application_arns"] = application_arns
            if role_arn is not None:
                self._values["role_arn"] = role_arn

        @builtins.property
        def application_arns(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The ARNs of the Amazon SNS platform applications that could be used by clients.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypool-pushsync.html#cfn-cognito-identitypool-pushsync-applicationarns
            '''
            result = self._values.get("application_arns")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def role_arn(self) -> typing.Optional[builtins.str]:
            '''An IAM role configured to allow Amazon Cognito to call Amazon SNS on behalf of the developer.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypool-pushsync.html#cfn-cognito-identitypool-pushsync-rolearn
            '''
            result = self._values.get("role_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PushSyncProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556)
class CfnIdentityPoolPrincipalTag(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPoolPrincipalTag",
):
    '''A list of the identity pool principal tag assignments for attributes for access control.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolprincipaltag.html
    :cloudformationResource: AWS::Cognito::IdentityPoolPrincipalTag
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        # principal_tags: Any
        
        cfn_identity_pool_principal_tag = cognito.CfnIdentityPoolPrincipalTag(self, "MyCfnIdentityPoolPrincipalTag",
            identity_pool_id="identityPoolId",
            identity_provider_name="identityProviderName",
        
            # the properties below are optional
            principal_tags=principal_tags,
            use_defaults=False
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        identity_pool_id: builtins.str,
        identity_provider_name: builtins.str,
        principal_tags: typing.Any = None,
        use_defaults: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param identity_pool_id: The identity pool that you want to associate with this principal tag map.
        :param identity_provider_name: The identity pool identity provider (IdP) that you want to associate with this principal tag map.
        :param principal_tags: A JSON-formatted list of user claims and the principal tags that you want to associate with them. When Amazon Cognito requests credentials, it sets the value of the principal tag to the value of the user's claim.
        :param use_defaults: Use a default set of mappings between claims and tags for this provider, instead of a custom map.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__036c02730249ce227e84b1b6d6f563004dc63ae9f5f1d1b135ebaf7125703598)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnIdentityPoolPrincipalTagProps(
            identity_pool_id=identity_pool_id,
            identity_provider_name=identity_provider_name,
            principal_tags=principal_tags,
            use_defaults=use_defaults,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f3f48484f4649e6d58d093801018075e48fd42dcf857d13ad24a164c0f41e989)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6e7f66a11cbbdc2a9e0dfc148a09d585162367190407d6814791c05805193623)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="identityPoolId")
    def identity_pool_id(self) -> builtins.str:
        '''The identity pool that you want to associate with this principal tag map.'''
        return typing.cast(builtins.str, jsii.get(self, "identityPoolId"))

    @identity_pool_id.setter
    def identity_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__18add40294e453336d39c4b2518445fd0e172f25c23bd78b598609632a714548)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identityPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="identityProviderName")
    def identity_provider_name(self) -> builtins.str:
        '''The identity pool identity provider (IdP) that you want to associate with this principal tag map.'''
        return typing.cast(builtins.str, jsii.get(self, "identityProviderName"))

    @identity_provider_name.setter
    def identity_provider_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__671325bd1b5eb381434a0ced3b0ca5b3646c379113e1014679b7ebacca38cc3e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identityProviderName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="principalTags")
    def principal_tags(self) -> typing.Any:
        '''A JSON-formatted list of user claims and the principal tags that you want to associate with them.'''
        return typing.cast(typing.Any, jsii.get(self, "principalTags"))

    @principal_tags.setter
    def principal_tags(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__941cd9f870d419478e147197cb0134b687f7c233a8bc805887d04a761669aefc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "principalTags", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="useDefaults")
    def use_defaults(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Use a default set of mappings between claims and tags for this provider, instead of a custom map.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "useDefaults"))

    @use_defaults.setter
    def use_defaults(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__615ac6e1fae0725987511e0b22108f8d03202321eadeee202779d4e5248fd19a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "useDefaults", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPoolPrincipalTagProps",
    jsii_struct_bases=[],
    name_mapping={
        "identity_pool_id": "identityPoolId",
        "identity_provider_name": "identityProviderName",
        "principal_tags": "principalTags",
        "use_defaults": "useDefaults",
    },
)
class CfnIdentityPoolPrincipalTagProps:
    def __init__(
        self,
        *,
        identity_pool_id: builtins.str,
        identity_provider_name: builtins.str,
        principal_tags: typing.Any = None,
        use_defaults: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
    ) -> None:
        '''Properties for defining a ``CfnIdentityPoolPrincipalTag``.

        :param identity_pool_id: The identity pool that you want to associate with this principal tag map.
        :param identity_provider_name: The identity pool identity provider (IdP) that you want to associate with this principal tag map.
        :param principal_tags: A JSON-formatted list of user claims and the principal tags that you want to associate with them. When Amazon Cognito requests credentials, it sets the value of the principal tag to the value of the user's claim.
        :param use_defaults: Use a default set of mappings between claims and tags for this provider, instead of a custom map.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolprincipaltag.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            # principal_tags: Any
            
            cfn_identity_pool_principal_tag_props = cognito.CfnIdentityPoolPrincipalTagProps(
                identity_pool_id="identityPoolId",
                identity_provider_name="identityProviderName",
            
                # the properties below are optional
                principal_tags=principal_tags,
                use_defaults=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d001e3233492e51cba8dc8c26ce2203a307bedc54524056815ba7fa51d7fa93)
            check_type(argname="argument identity_pool_id", value=identity_pool_id, expected_type=type_hints["identity_pool_id"])
            check_type(argname="argument identity_provider_name", value=identity_provider_name, expected_type=type_hints["identity_provider_name"])
            check_type(argname="argument principal_tags", value=principal_tags, expected_type=type_hints["principal_tags"])
            check_type(argname="argument use_defaults", value=use_defaults, expected_type=type_hints["use_defaults"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "identity_pool_id": identity_pool_id,
            "identity_provider_name": identity_provider_name,
        }
        if principal_tags is not None:
            self._values["principal_tags"] = principal_tags
        if use_defaults is not None:
            self._values["use_defaults"] = use_defaults

    @builtins.property
    def identity_pool_id(self) -> builtins.str:
        '''The identity pool that you want to associate with this principal tag map.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolprincipaltag.html#cfn-cognito-identitypoolprincipaltag-identitypoolid
        '''
        result = self._values.get("identity_pool_id")
        assert result is not None, "Required property 'identity_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def identity_provider_name(self) -> builtins.str:
        '''The identity pool identity provider (IdP) that you want to associate with this principal tag map.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolprincipaltag.html#cfn-cognito-identitypoolprincipaltag-identityprovidername
        '''
        result = self._values.get("identity_provider_name")
        assert result is not None, "Required property 'identity_provider_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def principal_tags(self) -> typing.Any:
        '''A JSON-formatted list of user claims and the principal tags that you want to associate with them.

        When Amazon Cognito requests credentials, it sets the value of the principal tag to the value of the user's claim.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolprincipaltag.html#cfn-cognito-identitypoolprincipaltag-principaltags
        '''
        result = self._values.get("principal_tags")
        return typing.cast(typing.Any, result)

    @builtins.property
    def use_defaults(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Use a default set of mappings between claims and tags for this provider, instead of a custom map.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolprincipaltag.html#cfn-cognito-identitypoolprincipaltag-usedefaults
        '''
        result = self._values.get("use_defaults")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnIdentityPoolPrincipalTagProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPoolProps",
    jsii_struct_bases=[],
    name_mapping={
        "allow_unauthenticated_identities": "allowUnauthenticatedIdentities",
        "allow_classic_flow": "allowClassicFlow",
        "cognito_events": "cognitoEvents",
        "cognito_identity_providers": "cognitoIdentityProviders",
        "cognito_streams": "cognitoStreams",
        "developer_provider_name": "developerProviderName",
        "identity_pool_name": "identityPoolName",
        "identity_pool_tags": "identityPoolTags",
        "open_id_connect_provider_arns": "openIdConnectProviderArns",
        "push_sync": "pushSync",
        "saml_provider_arns": "samlProviderArns",
        "supported_login_providers": "supportedLoginProviders",
    },
)
class CfnIdentityPoolProps:
    def __init__(
        self,
        *,
        allow_unauthenticated_identities: typing.Union[builtins.bool, _IResolvable_da3f097b],
        allow_classic_flow: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        cognito_events: typing.Any = None,
        cognito_identity_providers: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnIdentityPool.CognitoIdentityProviderProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        cognito_streams: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnIdentityPool.CognitoStreamsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        developer_provider_name: typing.Optional[builtins.str] = None,
        identity_pool_name: typing.Optional[builtins.str] = None,
        identity_pool_tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        open_id_connect_provider_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        push_sync: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnIdentityPool.PushSyncProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        saml_provider_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        supported_login_providers: typing.Any = None,
    ) -> None:
        '''Properties for defining a ``CfnIdentityPool``.

        :param allow_unauthenticated_identities: Specifies whether the identity pool supports unauthenticated logins.
        :param allow_classic_flow: Enables the Basic (Classic) authentication flow.
        :param cognito_events: The events to configure.
        :param cognito_identity_providers: The Amazon Cognito user pools and their client IDs.
        :param cognito_streams: Configuration options for configuring Amazon Cognito streams.
        :param developer_provider_name: The "domain" Amazon Cognito uses when referencing your users. This name acts as a placeholder that allows your backend and the Amazon Cognito service to communicate about the developer provider. For the ``DeveloperProviderName`` , you can use letters and periods (.), underscores (_), and dashes (-). *Minimum length* : 1 *Maximum length* : 100
        :param identity_pool_name: The name of your Amazon Cognito identity pool. *Minimum length* : 1 *Maximum length* : 128 *Pattern* : ``[\\w\\s+=,.@-]+``
        :param identity_pool_tags: Tags to assign to the identity pool. A tag is a label that you can apply to identity pools to categorize and manage them in different ways, such as by purpose, owner, environment, or other criteria.
        :param open_id_connect_provider_arns: The Amazon Resource Names (ARNs) of the OpenID connect providers.
        :param push_sync: The configuration options to be applied to the identity pool.
        :param saml_provider_arns: The Amazon Resource Names (ARNs) of the Security Assertion Markup Language (SAML) providers.
        :param supported_login_providers: Key-value pairs that map provider names to provider app IDs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html
        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_cognito as cognito
            
            # my_provider: iam.OpenIdConnectProvider
            
            cognito.CfnIdentityPool(self, "IdentityPool",
                open_id_connect_provider_arns=[my_provider.open_id_connect_provider_arn],
                # And the other properties for your identity pool
                allow_unauthenticated_identities=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3f190bc45ca19526ebd4b37441370bac2bb0babaadb5489ad9375252a5441596)
            check_type(argname="argument allow_unauthenticated_identities", value=allow_unauthenticated_identities, expected_type=type_hints["allow_unauthenticated_identities"])
            check_type(argname="argument allow_classic_flow", value=allow_classic_flow, expected_type=type_hints["allow_classic_flow"])
            check_type(argname="argument cognito_events", value=cognito_events, expected_type=type_hints["cognito_events"])
            check_type(argname="argument cognito_identity_providers", value=cognito_identity_providers, expected_type=type_hints["cognito_identity_providers"])
            check_type(argname="argument cognito_streams", value=cognito_streams, expected_type=type_hints["cognito_streams"])
            check_type(argname="argument developer_provider_name", value=developer_provider_name, expected_type=type_hints["developer_provider_name"])
            check_type(argname="argument identity_pool_name", value=identity_pool_name, expected_type=type_hints["identity_pool_name"])
            check_type(argname="argument identity_pool_tags", value=identity_pool_tags, expected_type=type_hints["identity_pool_tags"])
            check_type(argname="argument open_id_connect_provider_arns", value=open_id_connect_provider_arns, expected_type=type_hints["open_id_connect_provider_arns"])
            check_type(argname="argument push_sync", value=push_sync, expected_type=type_hints["push_sync"])
            check_type(argname="argument saml_provider_arns", value=saml_provider_arns, expected_type=type_hints["saml_provider_arns"])
            check_type(argname="argument supported_login_providers", value=supported_login_providers, expected_type=type_hints["supported_login_providers"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "allow_unauthenticated_identities": allow_unauthenticated_identities,
        }
        if allow_classic_flow is not None:
            self._values["allow_classic_flow"] = allow_classic_flow
        if cognito_events is not None:
            self._values["cognito_events"] = cognito_events
        if cognito_identity_providers is not None:
            self._values["cognito_identity_providers"] = cognito_identity_providers
        if cognito_streams is not None:
            self._values["cognito_streams"] = cognito_streams
        if developer_provider_name is not None:
            self._values["developer_provider_name"] = developer_provider_name
        if identity_pool_name is not None:
            self._values["identity_pool_name"] = identity_pool_name
        if identity_pool_tags is not None:
            self._values["identity_pool_tags"] = identity_pool_tags
        if open_id_connect_provider_arns is not None:
            self._values["open_id_connect_provider_arns"] = open_id_connect_provider_arns
        if push_sync is not None:
            self._values["push_sync"] = push_sync
        if saml_provider_arns is not None:
            self._values["saml_provider_arns"] = saml_provider_arns
        if supported_login_providers is not None:
            self._values["supported_login_providers"] = supported_login_providers

    @builtins.property
    def allow_unauthenticated_identities(
        self,
    ) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
        '''Specifies whether the identity pool supports unauthenticated logins.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-allowunauthenticatedidentities
        '''
        result = self._values.get("allow_unauthenticated_identities")
        assert result is not None, "Required property 'allow_unauthenticated_identities' is missing"
        return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

    @builtins.property
    def allow_classic_flow(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Enables the Basic (Classic) authentication flow.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-allowclassicflow
        '''
        result = self._values.get("allow_classic_flow")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def cognito_events(self) -> typing.Any:
        '''The events to configure.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-cognitoevents
        '''
        result = self._values.get("cognito_events")
        return typing.cast(typing.Any, result)

    @builtins.property
    def cognito_identity_providers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnIdentityPool.CognitoIdentityProviderProperty]]]]:
        '''The Amazon Cognito user pools and their client IDs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-cognitoidentityproviders
        '''
        result = self._values.get("cognito_identity_providers")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnIdentityPool.CognitoIdentityProviderProperty]]]], result)

    @builtins.property
    def cognito_streams(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnIdentityPool.CognitoStreamsProperty]]:
        '''Configuration options for configuring Amazon Cognito streams.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-cognitostreams
        '''
        result = self._values.get("cognito_streams")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnIdentityPool.CognitoStreamsProperty]], result)

    @builtins.property
    def developer_provider_name(self) -> typing.Optional[builtins.str]:
        '''The "domain" Amazon Cognito uses when referencing your users.

        This name acts as a placeholder that allows your backend and the Amazon Cognito service to communicate about the developer provider. For the ``DeveloperProviderName`` , you can use letters and periods (.), underscores (_), and dashes (-).

        *Minimum length* : 1

        *Maximum length* : 100

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-developerprovidername
        '''
        result = self._values.get("developer_provider_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def identity_pool_name(self) -> typing.Optional[builtins.str]:
        '''The name of your Amazon Cognito identity pool.

        *Minimum length* : 1

        *Maximum length* : 128

        *Pattern* : ``[\\w\\s+=,.@-]+``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-identitypoolname
        '''
        result = self._values.get("identity_pool_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def identity_pool_tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Tags to assign to the identity pool.

        A tag is a label that you can apply to identity pools to categorize and manage them in different ways, such as by purpose, owner, environment, or other criteria.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-identitypooltags
        '''
        result = self._values.get("identity_pool_tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def open_id_connect_provider_arns(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''The Amazon Resource Names (ARNs) of the OpenID connect providers.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-openidconnectproviderarns
        '''
        result = self._values.get("open_id_connect_provider_arns")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def push_sync(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnIdentityPool.PushSyncProperty]]:
        '''The configuration options to be applied to the identity pool.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-pushsync
        '''
        result = self._values.get("push_sync")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnIdentityPool.PushSyncProperty]], result)

    @builtins.property
    def saml_provider_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The Amazon Resource Names (ARNs) of the Security Assertion Markup Language (SAML) providers.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-samlproviderarns
        '''
        result = self._values.get("saml_provider_arns")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def supported_login_providers(self) -> typing.Any:
        '''Key-value pairs that map provider names to provider app IDs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypool.html#cfn-cognito-identitypool-supportedloginproviders
        '''
        result = self._values.get("supported_login_providers")
        return typing.cast(typing.Any, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnIdentityPoolProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnIdentityPoolRoleAttachment(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPoolRoleAttachment",
):
    '''The ``AWS::Cognito::IdentityPoolRoleAttachment`` resource manages the role configuration for an Amazon Cognito identity pool.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolroleattachment.html
    :cloudformationResource: AWS::Cognito::IdentityPoolRoleAttachment
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        # roles: Any
        
        cfn_identity_pool_role_attachment = cognito.CfnIdentityPoolRoleAttachment(self, "MyCfnIdentityPoolRoleAttachment",
            identity_pool_id="identityPoolId",
        
            # the properties below are optional
            role_mappings={
                "role_mappings_key": cognito.CfnIdentityPoolRoleAttachment.RoleMappingProperty(
                    type="type",
        
                    # the properties below are optional
                    ambiguous_role_resolution="ambiguousRoleResolution",
                    identity_provider="identityProvider",
                    rules_configuration=cognito.CfnIdentityPoolRoleAttachment.RulesConfigurationTypeProperty(
                        rules=[cognito.CfnIdentityPoolRoleAttachment.MappingRuleProperty(
                            claim="claim",
                            match_type="matchType",
                            role_arn="roleArn",
                            value="value"
                        )]
                    )
                )
            },
            roles=roles
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        identity_pool_id: builtins.str,
        role_mappings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, typing.Union["CfnIdentityPoolRoleAttachment.RoleMappingProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        roles: typing.Any = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param identity_pool_id: An identity pool ID in the format ``REGION:GUID`` .
        :param role_mappings: How users for a specific identity provider are mapped to roles. This is a string to the ``RoleMapping`` object map. The string identifies the identity provider. For example: ``graph.facebook.com`` or ``cognito-idp.us-east-1.amazonaws.com/us-east-1_abcdefghi:app_client_id`` . If the ``IdentityProvider`` field isn't provided in this object, the string is used as the identity provider name. For more information, see the `RoleMapping property <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rolemapping.html>`_ .
        :param roles: The map of the roles associated with this pool. For a given role, the key is either "authenticated" or "unauthenticated". The value is the role ARN.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e0686a984bb6b75dbf3aca8c723d268e911604e9a4aafa6bc4c2a74b934449b5)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnIdentityPoolRoleAttachmentProps(
            identity_pool_id=identity_pool_id, role_mappings=role_mappings, roles=roles
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__031bc41cc76f616fd91865be2d561b3fb548a456a85a4fa0fa4bb0e5a7d9b565)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0f9cad81f550438d748031d5745c6929d3a81e6a11179973b40195624704b706)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The resource ID.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="identityPoolId")
    def identity_pool_id(self) -> builtins.str:
        '''An identity pool ID in the format ``REGION:GUID`` .'''
        return typing.cast(builtins.str, jsii.get(self, "identityPoolId"))

    @identity_pool_id.setter
    def identity_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c00bf0f77c984f55d1c0f4d659a057ac347b7a1e253f7f07e04e7cb5836f96be)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identityPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="roleMappings")
    def role_mappings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, "CfnIdentityPoolRoleAttachment.RoleMappingProperty"]]]]:
        '''How users for a specific identity provider are mapped to roles.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, "CfnIdentityPoolRoleAttachment.RoleMappingProperty"]]]], jsii.get(self, "roleMappings"))

    @role_mappings.setter
    def role_mappings(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, "CfnIdentityPoolRoleAttachment.RoleMappingProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__260d54d4ee9e4e92869ee257cefc54f19659c465c953f833a73ba65eac311091)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "roleMappings", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="roles")
    def roles(self) -> typing.Any:
        '''The map of the roles associated with this pool.'''
        return typing.cast(typing.Any, jsii.get(self, "roles"))

    @roles.setter
    def roles(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__987142230d77cd554d510e7265f08bb6c8a2ff277afb6532b8166d2315b9510c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "roles", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPoolRoleAttachment.MappingRuleProperty",
        jsii_struct_bases=[],
        name_mapping={
            "claim": "claim",
            "match_type": "matchType",
            "role_arn": "roleArn",
            "value": "value",
        },
    )
    class MappingRuleProperty:
        def __init__(
            self,
            *,
            claim: builtins.str,
            match_type: builtins.str,
            role_arn: builtins.str,
            value: builtins.str,
        ) -> None:
            '''Defines how to map a claim to a role ARN.

            :param claim: The claim name that must be present in the token. For example: "isAdmin" or "paid".
            :param match_type: The match condition that specifies how closely the claim value in the IdP token must match ``Value`` . Valid values are: ``Equals`` , ``Contains`` , ``StartsWith`` , and ``NotEqual`` .
            :param role_arn: The Amazon Resource Name (ARN) of the role.
            :param value: A brief string that the claim must match. For example, "paid" or "yes".

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-mappingrule.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                mapping_rule_property = cognito.CfnIdentityPoolRoleAttachment.MappingRuleProperty(
                    claim="claim",
                    match_type="matchType",
                    role_arn="roleArn",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a2e874c3b25f3cabfefd5a244c8efe26a7d11d7166f90c0e58b82360db3936be)
                check_type(argname="argument claim", value=claim, expected_type=type_hints["claim"])
                check_type(argname="argument match_type", value=match_type, expected_type=type_hints["match_type"])
                check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "claim": claim,
                "match_type": match_type,
                "role_arn": role_arn,
                "value": value,
            }

        @builtins.property
        def claim(self) -> builtins.str:
            '''The claim name that must be present in the token.

            For example: "isAdmin" or "paid".

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-mappingrule.html#cfn-cognito-identitypoolroleattachment-mappingrule-claim
            '''
            result = self._values.get("claim")
            assert result is not None, "Required property 'claim' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def match_type(self) -> builtins.str:
            '''The match condition that specifies how closely the claim value in the IdP token must match ``Value`` .

            Valid values are: ``Equals`` , ``Contains`` , ``StartsWith`` , and ``NotEqual`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-mappingrule.html#cfn-cognito-identitypoolroleattachment-mappingrule-matchtype
            '''
            result = self._values.get("match_type")
            assert result is not None, "Required property 'match_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def role_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the role.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-mappingrule.html#cfn-cognito-identitypoolroleattachment-mappingrule-rolearn
            '''
            result = self._values.get("role_arn")
            assert result is not None, "Required property 'role_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''A brief string that the claim must match.

            For example, "paid" or "yes".

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-mappingrule.html#cfn-cognito-identitypoolroleattachment-mappingrule-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MappingRuleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPoolRoleAttachment.RoleMappingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "type": "type",
            "ambiguous_role_resolution": "ambiguousRoleResolution",
            "identity_provider": "identityProvider",
            "rules_configuration": "rulesConfiguration",
        },
    )
    class RoleMappingProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            ambiguous_role_resolution: typing.Optional[builtins.str] = None,
            identity_provider: typing.Optional[builtins.str] = None,
            rules_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnIdentityPoolRoleAttachment.RulesConfigurationTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''One of a set of ``RoleMappings`` , a property of the `AWS::Cognito::IdentityPoolRoleAttachment <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolroleattachment.html>`_ resource that defines the role-mapping attributes of an Amazon Cognito identity pool.

            :param type: The role mapping type. Token will use ``cognito:roles`` and ``cognito:preferred_role`` claims from the Cognito identity provider token to map groups to roles. Rules will attempt to match claims from the token to map to a role.
            :param ambiguous_role_resolution: If you specify Token or Rules as the ``Type`` , ``AmbiguousRoleResolution`` is required. Specifies the action to be taken if either no rules match the claim value for the ``Rules`` type, or there is no ``cognito:preferred_role`` claim and there are multiple ``cognito:roles`` matches for the ``Token`` type.
            :param identity_provider: Identifier for the identity provider for which the role is mapped. For example: ``graph.facebook.com`` or ``cognito-idp.us-east-1.amazonaws.com/us-east-1_abcdefghi:app_client_id (http://cognito-idp.us-east-1.amazonaws.com/us-east-1_abcdefghi:app_client_id)`` . This is the identity provider that is used by the user for authentication. If the identity provider property isn't provided, the key of the entry in the ``RoleMappings`` map is used as the identity provider.
            :param rules_configuration: The rules to be used for mapping users to roles. If you specify "Rules" as the role-mapping type, RulesConfiguration is required.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rolemapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                role_mapping_property = cognito.CfnIdentityPoolRoleAttachment.RoleMappingProperty(
                    type="type",
                
                    # the properties below are optional
                    ambiguous_role_resolution="ambiguousRoleResolution",
                    identity_provider="identityProvider",
                    rules_configuration=cognito.CfnIdentityPoolRoleAttachment.RulesConfigurationTypeProperty(
                        rules=[cognito.CfnIdentityPoolRoleAttachment.MappingRuleProperty(
                            claim="claim",
                            match_type="matchType",
                            role_arn="roleArn",
                            value="value"
                        )]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5a272c28ab03d1edcc86eddc7b9aa71f25398c059d8be5f7af4ea057dabe7082)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument ambiguous_role_resolution", value=ambiguous_role_resolution, expected_type=type_hints["ambiguous_role_resolution"])
                check_type(argname="argument identity_provider", value=identity_provider, expected_type=type_hints["identity_provider"])
                check_type(argname="argument rules_configuration", value=rules_configuration, expected_type=type_hints["rules_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if ambiguous_role_resolution is not None:
                self._values["ambiguous_role_resolution"] = ambiguous_role_resolution
            if identity_provider is not None:
                self._values["identity_provider"] = identity_provider
            if rules_configuration is not None:
                self._values["rules_configuration"] = rules_configuration

        @builtins.property
        def type(self) -> builtins.str:
            '''The role mapping type.

            Token will use ``cognito:roles`` and ``cognito:preferred_role`` claims from the Cognito identity provider token to map groups to roles. Rules will attempt to match claims from the token to map to a role.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rolemapping.html#cfn-cognito-identitypoolroleattachment-rolemapping-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def ambiguous_role_resolution(self) -> typing.Optional[builtins.str]:
            '''If you specify Token or Rules as the ``Type`` , ``AmbiguousRoleResolution`` is required.

            Specifies the action to be taken if either no rules match the claim value for the ``Rules`` type, or there is no ``cognito:preferred_role`` claim and there are multiple ``cognito:roles`` matches for the ``Token`` type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rolemapping.html#cfn-cognito-identitypoolroleattachment-rolemapping-ambiguousroleresolution
            '''
            result = self._values.get("ambiguous_role_resolution")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def identity_provider(self) -> typing.Optional[builtins.str]:
            '''Identifier for the identity provider for which the role is mapped.

            For example: ``graph.facebook.com`` or ``cognito-idp.us-east-1.amazonaws.com/us-east-1_abcdefghi:app_client_id (http://cognito-idp.us-east-1.amazonaws.com/us-east-1_abcdefghi:app_client_id)`` . This is the identity provider that is used by the user for authentication.

            If the identity provider property isn't provided, the key of the entry in the ``RoleMappings`` map is used as the identity provider.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rolemapping.html#cfn-cognito-identitypoolroleattachment-rolemapping-identityprovider
            '''
            result = self._values.get("identity_provider")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def rules_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIdentityPoolRoleAttachment.RulesConfigurationTypeProperty"]]:
            '''The rules to be used for mapping users to roles.

            If you specify "Rules" as the role-mapping type, RulesConfiguration is required.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rolemapping.html#cfn-cognito-identitypoolroleattachment-rolemapping-rulesconfiguration
            '''
            result = self._values.get("rules_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnIdentityPoolRoleAttachment.RulesConfigurationTypeProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RoleMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPoolRoleAttachment.RulesConfigurationTypeProperty",
        jsii_struct_bases=[],
        name_mapping={"rules": "rules"},
    )
    class RulesConfigurationTypeProperty:
        def __init__(
            self,
            *,
            rules: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnIdentityPoolRoleAttachment.MappingRuleProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''``RulesConfigurationType`` is a subproperty of the `RoleMapping <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rolemapping.html>`_ property that defines the rules to be used for mapping users to roles.

            :param rules: The rules. You can specify up to 25 rules per identity provider.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rulesconfigurationtype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                rules_configuration_type_property = cognito.CfnIdentityPoolRoleAttachment.RulesConfigurationTypeProperty(
                    rules=[cognito.CfnIdentityPoolRoleAttachment.MappingRuleProperty(
                        claim="claim",
                        match_type="matchType",
                        role_arn="roleArn",
                        value="value"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ee34c709dd9513ee04193cc48ac24cd57fca8873fec78fe528978c6e1d719347)
                check_type(argname="argument rules", value=rules, expected_type=type_hints["rules"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "rules": rules,
            }

        @builtins.property
        def rules(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIdentityPoolRoleAttachment.MappingRuleProperty"]]]:
            '''The rules.

            You can specify up to 25 rules per identity provider.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rulesconfigurationtype.html#cfn-cognito-identitypoolroleattachment-rulesconfigurationtype-rules
            '''
            result = self._values.get("rules")
            assert result is not None, "Required property 'rules' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnIdentityPoolRoleAttachment.MappingRuleProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RulesConfigurationTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnIdentityPoolRoleAttachmentProps",
    jsii_struct_bases=[],
    name_mapping={
        "identity_pool_id": "identityPoolId",
        "role_mappings": "roleMappings",
        "roles": "roles",
    },
)
class CfnIdentityPoolRoleAttachmentProps:
    def __init__(
        self,
        *,
        identity_pool_id: builtins.str,
        role_mappings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, typing.Union[CfnIdentityPoolRoleAttachment.RoleMappingProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        roles: typing.Any = None,
    ) -> None:
        '''Properties for defining a ``CfnIdentityPoolRoleAttachment``.

        :param identity_pool_id: An identity pool ID in the format ``REGION:GUID`` .
        :param role_mappings: How users for a specific identity provider are mapped to roles. This is a string to the ``RoleMapping`` object map. The string identifies the identity provider. For example: ``graph.facebook.com`` or ``cognito-idp.us-east-1.amazonaws.com/us-east-1_abcdefghi:app_client_id`` . If the ``IdentityProvider`` field isn't provided in this object, the string is used as the identity provider name. For more information, see the `RoleMapping property <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rolemapping.html>`_ .
        :param roles: The map of the roles associated with this pool. For a given role, the key is either "authenticated" or "unauthenticated". The value is the role ARN.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolroleattachment.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            # roles: Any
            
            cfn_identity_pool_role_attachment_props = cognito.CfnIdentityPoolRoleAttachmentProps(
                identity_pool_id="identityPoolId",
            
                # the properties below are optional
                role_mappings={
                    "role_mappings_key": cognito.CfnIdentityPoolRoleAttachment.RoleMappingProperty(
                        type="type",
            
                        # the properties below are optional
                        ambiguous_role_resolution="ambiguousRoleResolution",
                        identity_provider="identityProvider",
                        rules_configuration=cognito.CfnIdentityPoolRoleAttachment.RulesConfigurationTypeProperty(
                            rules=[cognito.CfnIdentityPoolRoleAttachment.MappingRuleProperty(
                                claim="claim",
                                match_type="matchType",
                                role_arn="roleArn",
                                value="value"
                            )]
                        )
                    )
                },
                roles=roles
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7531b732f0c0a047085fd9c4031282884fde57038e57a1ebd6d247d39ccd050b)
            check_type(argname="argument identity_pool_id", value=identity_pool_id, expected_type=type_hints["identity_pool_id"])
            check_type(argname="argument role_mappings", value=role_mappings, expected_type=type_hints["role_mappings"])
            check_type(argname="argument roles", value=roles, expected_type=type_hints["roles"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "identity_pool_id": identity_pool_id,
        }
        if role_mappings is not None:
            self._values["role_mappings"] = role_mappings
        if roles is not None:
            self._values["roles"] = roles

    @builtins.property
    def identity_pool_id(self) -> builtins.str:
        '''An identity pool ID in the format ``REGION:GUID`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolroleattachment.html#cfn-cognito-identitypoolroleattachment-identitypoolid
        '''
        result = self._values.get("identity_pool_id")
        assert result is not None, "Required property 'identity_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def role_mappings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, CfnIdentityPoolRoleAttachment.RoleMappingProperty]]]]:
        '''How users for a specific identity provider are mapped to roles.

        This is a string to the ``RoleMapping`` object map. The string identifies the identity provider. For example: ``graph.facebook.com`` or ``cognito-idp.us-east-1.amazonaws.com/us-east-1_abcdefghi:app_client_id`` .

        If the ``IdentityProvider`` field isn't provided in this object, the string is used as the identity provider name.

        For more information, see the `RoleMapping property <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-identitypoolroleattachment-rolemapping.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolroleattachment.html#cfn-cognito-identitypoolroleattachment-rolemappings
        '''
        result = self._values.get("role_mappings")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, CfnIdentityPoolRoleAttachment.RoleMappingProperty]]]], result)

    @builtins.property
    def roles(self) -> typing.Any:
        '''The map of the roles associated with this pool.

        For a given role, the key is either "authenticated" or "unauthenticated". The value is the role ARN.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-identitypoolroleattachment.html#cfn-cognito-identitypoolroleattachment-roles
        '''
        result = self._values.get("roles")
        return typing.cast(typing.Any, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnIdentityPoolRoleAttachmentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnLogDeliveryConfiguration(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnLogDeliveryConfiguration",
):
    '''Sets up or modifies the logging configuration of a user pool.

    User pools can export user notification logs and, when threat protection is active, user-activity logs. For more information, see `Exporting user pool logs <https://docs.aws.amazon.com/cognito/latest/developerguide/exporting-quotas-and-usage.html>`_ .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-logdeliveryconfiguration.html
    :cloudformationResource: AWS::Cognito::LogDeliveryConfiguration
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        cfn_log_delivery_configuration = cognito.CfnLogDeliveryConfiguration(self, "MyCfnLogDeliveryConfiguration",
            user_pool_id="userPoolId",
        
            # the properties below are optional
            log_configurations=[cognito.CfnLogDeliveryConfiguration.LogConfigurationProperty(
                cloud_watch_logs_configuration=cognito.CfnLogDeliveryConfiguration.CloudWatchLogsConfigurationProperty(
                    log_group_arn="logGroupArn"
                ),
                event_source="eventSource",
                firehose_configuration=cognito.CfnLogDeliveryConfiguration.FirehoseConfigurationProperty(
                    stream_arn="streamArn"
                ),
                log_level="logLevel",
                s3_configuration=cognito.CfnLogDeliveryConfiguration.S3ConfigurationProperty(
                    bucket_arn="bucketArn"
                )
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        user_pool_id: builtins.str,
        log_configurations: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnLogDeliveryConfiguration.LogConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param user_pool_id: The ID of the user pool where you configured logging.
        :param log_configurations: A logging destination of a user pool. User pools can have multiple logging destinations for message-delivery and user-activity logs.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e29530a80c41b2b01a02705536af0f4120664e64afaf6cf445105a2dd4a7023d)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnLogDeliveryConfigurationProps(
            user_pool_id=user_pool_id, log_configurations=log_configurations
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9c754043d52de2809138bd323abb0b034a1950dd042f21ac2f36d2f80c2f6ab5)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__089ac90c3942f7680901a3873d6f480819579f6ee19698ffe4041808f8ddfb54)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''A user pool ID, for example ``us-east-1_EXAMPLE`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you configured logging.'''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @user_pool_id.setter
    def user_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2182e97aaa93da941d4811edec2c32bd2af08481e5b611636933cf688c72d4a5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="logConfigurations")
    def log_configurations(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnLogDeliveryConfiguration.LogConfigurationProperty"]]]]:
        '''A logging destination of a user pool.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnLogDeliveryConfiguration.LogConfigurationProperty"]]]], jsii.get(self, "logConfigurations"))

    @log_configurations.setter
    def log_configurations(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnLogDeliveryConfiguration.LogConfigurationProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__40a6b8e2f773442c634fa6b71515839097ae56e3de02083447219aae5f9c1a9f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "logConfigurations", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnLogDeliveryConfiguration.CloudWatchLogsConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"log_group_arn": "logGroupArn"},
    )
    class CloudWatchLogsConfigurationProperty:
        def __init__(
            self,
            *,
            log_group_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Configuration for the CloudWatch log group destination of user pool detailed activity logging, or of user activity log export with advanced security features.

            :param log_group_arn: The Amazon Resource Name (arn) of a CloudWatch Logs log group where your user pool sends logs. The log group must not be encrypted with AWS Key Management Service and must be in the same AWS account as your user pool. To send logs to log groups with a resource policy of a size greater than 5120 characters, configure a log group with a path that starts with ``/aws/vendedlogs`` . For more information, see `Enabling logging from certain AWS services <https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/AWS-logs-and-resource-policy.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-cloudwatchlogsconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                cloud_watch_logs_configuration_property = cognito.CfnLogDeliveryConfiguration.CloudWatchLogsConfigurationProperty(
                    log_group_arn="logGroupArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ff46bbb404bb2daaadf6da44499aa85e733a076298ca686e738f64972365712b)
                check_type(argname="argument log_group_arn", value=log_group_arn, expected_type=type_hints["log_group_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if log_group_arn is not None:
                self._values["log_group_arn"] = log_group_arn

        @builtins.property
        def log_group_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (arn) of a CloudWatch Logs log group where your user pool sends logs.

            The log group must not be encrypted with AWS Key Management Service and must be in the same AWS account as your user pool.

            To send logs to log groups with a resource policy of a size greater than 5120 characters, configure a log group with a path that starts with ``/aws/vendedlogs`` . For more information, see `Enabling logging from certain AWS services <https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/AWS-logs-and-resource-policy.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-cloudwatchlogsconfiguration.html#cfn-cognito-logdeliveryconfiguration-cloudwatchlogsconfiguration-loggrouparn
            '''
            result = self._values.get("log_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CloudWatchLogsConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnLogDeliveryConfiguration.FirehoseConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"stream_arn": "streamArn"},
    )
    class FirehoseConfigurationProperty:
        def __init__(self, *, stream_arn: typing.Optional[builtins.str] = None) -> None:
            '''Configuration for the Amazon Data Firehose stream destination of user activity log export with threat protection.

            :param stream_arn: The ARN of an Amazon Data Firehose stream that's the destination for threat protection log export.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-firehoseconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                firehose_configuration_property = cognito.CfnLogDeliveryConfiguration.FirehoseConfigurationProperty(
                    stream_arn="streamArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ae072d70f2bbea287c140340ad4ca6d5b9e1b95bf2308803bfb1cfbd9e23e5e0)
                check_type(argname="argument stream_arn", value=stream_arn, expected_type=type_hints["stream_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if stream_arn is not None:
                self._values["stream_arn"] = stream_arn

        @builtins.property
        def stream_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of an Amazon Data Firehose stream that's the destination for threat protection log export.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-firehoseconfiguration.html#cfn-cognito-logdeliveryconfiguration-firehoseconfiguration-streamarn
            '''
            result = self._values.get("stream_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FirehoseConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnLogDeliveryConfiguration.LogConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cloud_watch_logs_configuration": "cloudWatchLogsConfiguration",
            "event_source": "eventSource",
            "firehose_configuration": "firehoseConfiguration",
            "log_level": "logLevel",
            "s3_configuration": "s3Configuration",
        },
    )
    class LogConfigurationProperty:
        def __init__(
            self,
            *,
            cloud_watch_logs_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnLogDeliveryConfiguration.CloudWatchLogsConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            event_source: typing.Optional[builtins.str] = None,
            firehose_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnLogDeliveryConfiguration.FirehoseConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            log_level: typing.Optional[builtins.str] = None,
            s3_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnLogDeliveryConfiguration.S3ConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration of user event logs to an external AWS service like Amazon Data Firehose, Amazon S3, or Amazon CloudWatch Logs.

            :param cloud_watch_logs_configuration: Configuration for the CloudWatch log group destination of user pool detailed activity logging, or of user activity log export with advanced security features.
            :param event_source: The source of events that your user pool sends for logging. To send error-level logs about user notification activity, set to ``userNotification`` . To send info-level logs about threat-protection user activity in user pools with the Plus feature plan, set to ``userAuthEvents`` .
            :param firehose_configuration: Configuration for the Amazon Data Firehose stream destination of user activity log export with threat protection.
            :param log_level: The ``errorlevel`` selection of logs that a user pool sends for detailed activity logging. To send ``userNotification`` activity with `information about message delivery <https://docs.aws.amazon.com/cognito/latest/developerguide/exporting-quotas-and-usage.html>`_ , choose ``ERROR`` with ``CloudWatchLogsConfiguration`` . To send ``userAuthEvents`` activity with user logs from threat protection with the Plus feature plan, choose ``INFO`` with one of ``CloudWatchLogsConfiguration`` , ``FirehoseConfiguration`` , or ``S3Configuration`` .
            :param s3_configuration: Configuration for the Amazon S3 bucket destination of user activity log export with threat protection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-logconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                log_configuration_property = cognito.CfnLogDeliveryConfiguration.LogConfigurationProperty(
                    cloud_watch_logs_configuration=cognito.CfnLogDeliveryConfiguration.CloudWatchLogsConfigurationProperty(
                        log_group_arn="logGroupArn"
                    ),
                    event_source="eventSource",
                    firehose_configuration=cognito.CfnLogDeliveryConfiguration.FirehoseConfigurationProperty(
                        stream_arn="streamArn"
                    ),
                    log_level="logLevel",
                    s3_configuration=cognito.CfnLogDeliveryConfiguration.S3ConfigurationProperty(
                        bucket_arn="bucketArn"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3fcaff05e87f499492c9c4ac6949463926f4833510675569310df4f8cb619149)
                check_type(argname="argument cloud_watch_logs_configuration", value=cloud_watch_logs_configuration, expected_type=type_hints["cloud_watch_logs_configuration"])
                check_type(argname="argument event_source", value=event_source, expected_type=type_hints["event_source"])
                check_type(argname="argument firehose_configuration", value=firehose_configuration, expected_type=type_hints["firehose_configuration"])
                check_type(argname="argument log_level", value=log_level, expected_type=type_hints["log_level"])
                check_type(argname="argument s3_configuration", value=s3_configuration, expected_type=type_hints["s3_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cloud_watch_logs_configuration is not None:
                self._values["cloud_watch_logs_configuration"] = cloud_watch_logs_configuration
            if event_source is not None:
                self._values["event_source"] = event_source
            if firehose_configuration is not None:
                self._values["firehose_configuration"] = firehose_configuration
            if log_level is not None:
                self._values["log_level"] = log_level
            if s3_configuration is not None:
                self._values["s3_configuration"] = s3_configuration

        @builtins.property
        def cloud_watch_logs_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnLogDeliveryConfiguration.CloudWatchLogsConfigurationProperty"]]:
            '''Configuration for the CloudWatch log group destination of user pool detailed activity logging, or of user activity log export with advanced security features.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-logconfiguration.html#cfn-cognito-logdeliveryconfiguration-logconfiguration-cloudwatchlogsconfiguration
            '''
            result = self._values.get("cloud_watch_logs_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnLogDeliveryConfiguration.CloudWatchLogsConfigurationProperty"]], result)

        @builtins.property
        def event_source(self) -> typing.Optional[builtins.str]:
            '''The source of events that your user pool sends for logging.

            To send error-level logs about user notification activity, set to ``userNotification`` . To send info-level logs about threat-protection user activity in user pools with the Plus feature plan, set to ``userAuthEvents`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-logconfiguration.html#cfn-cognito-logdeliveryconfiguration-logconfiguration-eventsource
            '''
            result = self._values.get("event_source")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def firehose_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnLogDeliveryConfiguration.FirehoseConfigurationProperty"]]:
            '''Configuration for the Amazon Data Firehose stream destination of user activity log export with threat protection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-logconfiguration.html#cfn-cognito-logdeliveryconfiguration-logconfiguration-firehoseconfiguration
            '''
            result = self._values.get("firehose_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnLogDeliveryConfiguration.FirehoseConfigurationProperty"]], result)

        @builtins.property
        def log_level(self) -> typing.Optional[builtins.str]:
            '''The ``errorlevel`` selection of logs that a user pool sends for detailed activity logging.

            To send ``userNotification`` activity with `information about message delivery <https://docs.aws.amazon.com/cognito/latest/developerguide/exporting-quotas-and-usage.html>`_ , choose ``ERROR`` with ``CloudWatchLogsConfiguration`` . To send ``userAuthEvents`` activity with user logs from threat protection with the Plus feature plan, choose ``INFO`` with one of ``CloudWatchLogsConfiguration`` , ``FirehoseConfiguration`` , or ``S3Configuration`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-logconfiguration.html#cfn-cognito-logdeliveryconfiguration-logconfiguration-loglevel
            '''
            result = self._values.get("log_level")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnLogDeliveryConfiguration.S3ConfigurationProperty"]]:
            '''Configuration for the Amazon S3 bucket destination of user activity log export with threat protection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-logconfiguration.html#cfn-cognito-logdeliveryconfiguration-logconfiguration-s3configuration
            '''
            result = self._values.get("s3_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnLogDeliveryConfiguration.S3ConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LogConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnLogDeliveryConfiguration.S3ConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"bucket_arn": "bucketArn"},
    )
    class S3ConfigurationProperty:
        def __init__(self, *, bucket_arn: typing.Optional[builtins.str] = None) -> None:
            '''Configuration for the Amazon S3 bucket destination of user activity log export with threat protection.

            :param bucket_arn: The ARN of an Amazon S3 bucket that's the destination for threat protection log export.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-s3configuration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                s3_configuration_property = cognito.CfnLogDeliveryConfiguration.S3ConfigurationProperty(
                    bucket_arn="bucketArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__35855407409f14c2401e604d3e25c83bb2d207f28090618c8174149befb93967)
                check_type(argname="argument bucket_arn", value=bucket_arn, expected_type=type_hints["bucket_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if bucket_arn is not None:
                self._values["bucket_arn"] = bucket_arn

        @builtins.property
        def bucket_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of an Amazon S3 bucket that's the destination for threat protection log export.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-logdeliveryconfiguration-s3configuration.html#cfn-cognito-logdeliveryconfiguration-s3configuration-bucketarn
            '''
            result = self._values.get("bucket_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "S3ConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnLogDeliveryConfigurationProps",
    jsii_struct_bases=[],
    name_mapping={
        "user_pool_id": "userPoolId",
        "log_configurations": "logConfigurations",
    },
)
class CfnLogDeliveryConfigurationProps:
    def __init__(
        self,
        *,
        user_pool_id: builtins.str,
        log_configurations: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnLogDeliveryConfiguration.LogConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnLogDeliveryConfiguration``.

        :param user_pool_id: The ID of the user pool where you configured logging.
        :param log_configurations: A logging destination of a user pool. User pools can have multiple logging destinations for message-delivery and user-activity logs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-logdeliveryconfiguration.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            cfn_log_delivery_configuration_props = cognito.CfnLogDeliveryConfigurationProps(
                user_pool_id="userPoolId",
            
                # the properties below are optional
                log_configurations=[cognito.CfnLogDeliveryConfiguration.LogConfigurationProperty(
                    cloud_watch_logs_configuration=cognito.CfnLogDeliveryConfiguration.CloudWatchLogsConfigurationProperty(
                        log_group_arn="logGroupArn"
                    ),
                    event_source="eventSource",
                    firehose_configuration=cognito.CfnLogDeliveryConfiguration.FirehoseConfigurationProperty(
                        stream_arn="streamArn"
                    ),
                    log_level="logLevel",
                    s3_configuration=cognito.CfnLogDeliveryConfiguration.S3ConfigurationProperty(
                        bucket_arn="bucketArn"
                    )
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__585789fa8816c3e4ed9b3aa9967435c1474787f750de3db35983f11efef27366)
            check_type(argname="argument user_pool_id", value=user_pool_id, expected_type=type_hints["user_pool_id"])
            check_type(argname="argument log_configurations", value=log_configurations, expected_type=type_hints["log_configurations"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "user_pool_id": user_pool_id,
        }
        if log_configurations is not None:
            self._values["log_configurations"] = log_configurations

    @builtins.property
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you configured logging.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-logdeliveryconfiguration.html#cfn-cognito-logdeliveryconfiguration-userpoolid
        '''
        result = self._values.get("user_pool_id")
        assert result is not None, "Required property 'user_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def log_configurations(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnLogDeliveryConfiguration.LogConfigurationProperty]]]]:
        '''A logging destination of a user pool.

        User pools can have multiple logging destinations for message-delivery and user-activity logs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-logdeliveryconfiguration.html#cfn-cognito-logdeliveryconfiguration-logconfigurations
        '''
        result = self._values.get("log_configurations")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnLogDeliveryConfiguration.LogConfigurationProperty]]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnLogDeliveryConfigurationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnManagedLoginBranding(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnManagedLoginBranding",
):
    '''Creates a new set of branding settings for a user pool style and associates it with an app client.

    This operation is the programmatic option for the creation of a new style in the branding designer.

    Provides values for UI customization in a ``Settings`` JSON object and image files in an ``Assets`` array. To send the JSON object ``Document`` type parameter in ``Settings`` , you might need to update to the most recent version of your AWS SDK.

    This operation has a 2-megabyte request-size limit and include the CSS settings and image assets for your app client. Your branding settings might exceed 2MB in size. Amazon Cognito doesn't require that you pass all parameters in one request and preserves existing style settings that you don't specify. If your request is larger than 2MB, separate it into multiple requests, each with a size smaller than the limit.

    As a best practice, modify the output of `DescribeManagedLoginBrandingByClient <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeManagedLoginBrandingByClient.html>`_ into the request parameters for this operation. To get all settings, set ``ReturnMergedResources`` to ``true`` . For more information, see `API and SDK operations for managed login branding <https://docs.aws.amazon.com/cognito/latest/developerguide/managed-login-brandingdesigner.html#branding-designer-api>`_

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-managedloginbranding.html
    :cloudformationResource: AWS::Cognito::ManagedLoginBranding
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        # settings: Any
        
        cfn_managed_login_branding = cognito.CfnManagedLoginBranding(self, "MyCfnManagedLoginBranding",
            user_pool_id="userPoolId",
        
            # the properties below are optional
            assets=[cognito.CfnManagedLoginBranding.AssetTypeProperty(
                category="category",
                color_mode="colorMode",
                extension="extension",
        
                # the properties below are optional
                bytes="bytes",
                resource_id="resourceId"
            )],
            client_id="clientId",
            return_merged_resources=False,
            settings=settings,
            use_cognito_provided_values=False
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        user_pool_id: builtins.str,
        assets: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnManagedLoginBranding.AssetTypeProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        client_id: typing.Optional[builtins.str] = None,
        return_merged_resources: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        settings: typing.Any = None,
        use_cognito_provided_values: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param user_pool_id: The user pool where the branding style is assigned.
        :param assets: An array of image files that you want to apply to roles like backgrounds, logos, and icons. Each object must also indicate whether it is for dark mode, light mode, or browser-adaptive mode.
        :param client_id: The app client that's assigned to the branding style that you want more information about.
        :param return_merged_resources: When ``true`` , returns values for branding options that are unchanged from Amazon Cognito defaults. When ``false`` or when you omit this parameter, returns only values that you customized in your branding style.
        :param settings: A JSON file, encoded as a ``Document`` type, with the the settings that you want to apply to your style.
        :param use_cognito_provided_values: When true, applies the default branding style options. This option reverts to default style options that are managed by Amazon Cognito. You can modify them later in the branding designer. When you specify ``true`` for this option, you must also omit values for ``Settings`` and ``Assets`` in the request.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__478f8899894ffccc3f20b06ae18c36beb41bf5c5c9aa65a99dbdbf95ce00be03)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnManagedLoginBrandingProps(
            user_pool_id=user_pool_id,
            assets=assets,
            client_id=client_id,
            return_merged_resources=return_merged_resources,
            settings=settings,
            use_cognito_provided_values=use_cognito_provided_values,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a0d347f9b2c0101529861e949ebe0a802ebc429100648b4c870711c733b50faa)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1112e058064e524fbe515ff8791467e6949341c6ddd8deb9c33af3658b16d447)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrManagedLoginBrandingId")
    def attr_managed_login_branding_id(self) -> builtins.str:
        '''The ID of the managed login branding style.

        :cloudformationAttribute: ManagedLoginBrandingId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrManagedLoginBrandingId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The user pool where the branding style is assigned.'''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @user_pool_id.setter
    def user_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__73b2532ea6e2300654d7fcc90b2b1fd38f772128b765556475cff8c1be577731)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="assets")
    def assets(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnManagedLoginBranding.AssetTypeProperty"]]]]:
        '''An array of image files that you want to apply to roles like backgrounds, logos, and icons.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnManagedLoginBranding.AssetTypeProperty"]]]], jsii.get(self, "assets"))

    @assets.setter
    def assets(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnManagedLoginBranding.AssetTypeProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fc790275f28767420e82246bd64663082d888a2c93af667d6c769ece2924f786)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "assets", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="clientId")
    def client_id(self) -> typing.Optional[builtins.str]:
        '''The app client that's assigned to the branding style that you want more information about.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "clientId"))

    @client_id.setter
    def client_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a668420e0b3cbceec0ade65febad3505a8186912fb1310c4ecdfbbcd6bac7dc2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="returnMergedResources")
    def return_merged_resources(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''When ``true`` , returns values for branding options that are unchanged from Amazon Cognito defaults.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "returnMergedResources"))

    @return_merged_resources.setter
    def return_merged_resources(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea8e49ce2efc2678bcbf1fdf919c5bbeac64755b39b20ef47a3f76532c424dfc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "returnMergedResources", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="settings")
    def settings(self) -> typing.Any:
        '''A JSON file, encoded as a ``Document`` type, with the the settings that you want to apply to your style.'''
        return typing.cast(typing.Any, jsii.get(self, "settings"))

    @settings.setter
    def settings(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f22fe695e1f64d8a038409355220b2e920e04882727bafb532a5728f1ffe677c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "settings", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="useCognitoProvidedValues")
    def use_cognito_provided_values(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''When true, applies the default branding style options.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "useCognitoProvidedValues"))

    @use_cognito_provided_values.setter
    def use_cognito_provided_values(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4b61f0689e78fea36c23c402c48085be3f2c198b922507818947333d59445895)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "useCognitoProvidedValues", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnManagedLoginBranding.AssetTypeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "category": "category",
            "color_mode": "colorMode",
            "extension": "extension",
            "bytes": "bytes",
            "resource_id": "resourceId",
        },
    )
    class AssetTypeProperty:
        def __init__(
            self,
            *,
            category: builtins.str,
            color_mode: builtins.str,
            extension: builtins.str,
            bytes: typing.Optional[builtins.str] = None,
            resource_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An image file from a managed login branding style in a user pool.

            :param category: The category that the image corresponds to in your managed login configuration. Managed login has asset categories for different types of logos, backgrounds, and icons.
            :param color_mode: The display-mode target of the asset: light, dark, or browser-adaptive. For example, Amazon Cognito displays a dark-mode image only when the browser or application is in dark mode, but displays a browser-adaptive file in all contexts.
            :param extension: The file type of the image file.
            :param bytes: The image file, in Base64-encoded binary.
            :param resource_id: The ID of the asset.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-managedloginbranding-assettype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                asset_type_property = cognito.CfnManagedLoginBranding.AssetTypeProperty(
                    category="category",
                    color_mode="colorMode",
                    extension="extension",
                
                    # the properties below are optional
                    bytes="bytes",
                    resource_id="resourceId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a8c0b7bdabc4393d484227225be1727f821e164eec56517d614639ac2059509c)
                check_type(argname="argument category", value=category, expected_type=type_hints["category"])
                check_type(argname="argument color_mode", value=color_mode, expected_type=type_hints["color_mode"])
                check_type(argname="argument extension", value=extension, expected_type=type_hints["extension"])
                check_type(argname="argument bytes", value=bytes, expected_type=type_hints["bytes"])
                check_type(argname="argument resource_id", value=resource_id, expected_type=type_hints["resource_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "category": category,
                "color_mode": color_mode,
                "extension": extension,
            }
            if bytes is not None:
                self._values["bytes"] = bytes
            if resource_id is not None:
                self._values["resource_id"] = resource_id

        @builtins.property
        def category(self) -> builtins.str:
            '''The category that the image corresponds to in your managed login configuration.

            Managed login has asset categories for different types of logos, backgrounds, and icons.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-managedloginbranding-assettype.html#cfn-cognito-managedloginbranding-assettype-category
            '''
            result = self._values.get("category")
            assert result is not None, "Required property 'category' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def color_mode(self) -> builtins.str:
            '''The display-mode target of the asset: light, dark, or browser-adaptive.

            For example, Amazon Cognito displays a dark-mode image only when the browser or application is in dark mode, but displays a browser-adaptive file in all contexts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-managedloginbranding-assettype.html#cfn-cognito-managedloginbranding-assettype-colormode
            '''
            result = self._values.get("color_mode")
            assert result is not None, "Required property 'color_mode' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def extension(self) -> builtins.str:
            '''The file type of the image file.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-managedloginbranding-assettype.html#cfn-cognito-managedloginbranding-assettype-extension
            '''
            result = self._values.get("extension")
            assert result is not None, "Required property 'extension' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def bytes(self) -> typing.Optional[builtins.str]:
            '''The image file, in Base64-encoded binary.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-managedloginbranding-assettype.html#cfn-cognito-managedloginbranding-assettype-bytes
            '''
            result = self._values.get("bytes")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def resource_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the asset.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-managedloginbranding-assettype.html#cfn-cognito-managedloginbranding-assettype-resourceid
            '''
            result = self._values.get("resource_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AssetTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnManagedLoginBrandingProps",
    jsii_struct_bases=[],
    name_mapping={
        "user_pool_id": "userPoolId",
        "assets": "assets",
        "client_id": "clientId",
        "return_merged_resources": "returnMergedResources",
        "settings": "settings",
        "use_cognito_provided_values": "useCognitoProvidedValues",
    },
)
class CfnManagedLoginBrandingProps:
    def __init__(
        self,
        *,
        user_pool_id: builtins.str,
        assets: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnManagedLoginBranding.AssetTypeProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        client_id: typing.Optional[builtins.str] = None,
        return_merged_resources: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        settings: typing.Any = None,
        use_cognito_provided_values: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
    ) -> None:
        '''Properties for defining a ``CfnManagedLoginBranding``.

        :param user_pool_id: The user pool where the branding style is assigned.
        :param assets: An array of image files that you want to apply to roles like backgrounds, logos, and icons. Each object must also indicate whether it is for dark mode, light mode, or browser-adaptive mode.
        :param client_id: The app client that's assigned to the branding style that you want more information about.
        :param return_merged_resources: When ``true`` , returns values for branding options that are unchanged from Amazon Cognito defaults. When ``false`` or when you omit this parameter, returns only values that you customized in your branding style.
        :param settings: A JSON file, encoded as a ``Document`` type, with the the settings that you want to apply to your style.
        :param use_cognito_provided_values: When true, applies the default branding style options. This option reverts to default style options that are managed by Amazon Cognito. You can modify them later in the branding designer. When you specify ``true`` for this option, you must also omit values for ``Settings`` and ``Assets`` in the request.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-managedloginbranding.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            # settings: Any
            
            cfn_managed_login_branding_props = cognito.CfnManagedLoginBrandingProps(
                user_pool_id="userPoolId",
            
                # the properties below are optional
                assets=[cognito.CfnManagedLoginBranding.AssetTypeProperty(
                    category="category",
                    color_mode="colorMode",
                    extension="extension",
            
                    # the properties below are optional
                    bytes="bytes",
                    resource_id="resourceId"
                )],
                client_id="clientId",
                return_merged_resources=False,
                settings=settings,
                use_cognito_provided_values=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__60e207e1aa2ab8ae23b36c3e1ae73765c6f328b13bf0c7b205865e93adc260df)
            check_type(argname="argument user_pool_id", value=user_pool_id, expected_type=type_hints["user_pool_id"])
            check_type(argname="argument assets", value=assets, expected_type=type_hints["assets"])
            check_type(argname="argument client_id", value=client_id, expected_type=type_hints["client_id"])
            check_type(argname="argument return_merged_resources", value=return_merged_resources, expected_type=type_hints["return_merged_resources"])
            check_type(argname="argument settings", value=settings, expected_type=type_hints["settings"])
            check_type(argname="argument use_cognito_provided_values", value=use_cognito_provided_values, expected_type=type_hints["use_cognito_provided_values"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "user_pool_id": user_pool_id,
        }
        if assets is not None:
            self._values["assets"] = assets
        if client_id is not None:
            self._values["client_id"] = client_id
        if return_merged_resources is not None:
            self._values["return_merged_resources"] = return_merged_resources
        if settings is not None:
            self._values["settings"] = settings
        if use_cognito_provided_values is not None:
            self._values["use_cognito_provided_values"] = use_cognito_provided_values

    @builtins.property
    def user_pool_id(self) -> builtins.str:
        '''The user pool where the branding style is assigned.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-managedloginbranding.html#cfn-cognito-managedloginbranding-userpoolid
        '''
        result = self._values.get("user_pool_id")
        assert result is not None, "Required property 'user_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def assets(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnManagedLoginBranding.AssetTypeProperty]]]]:
        '''An array of image files that you want to apply to roles like backgrounds, logos, and icons.

        Each object must also indicate whether it is for dark mode, light mode, or browser-adaptive mode.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-managedloginbranding.html#cfn-cognito-managedloginbranding-assets
        '''
        result = self._values.get("assets")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnManagedLoginBranding.AssetTypeProperty]]]], result)

    @builtins.property
    def client_id(self) -> typing.Optional[builtins.str]:
        '''The app client that's assigned to the branding style that you want more information about.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-managedloginbranding.html#cfn-cognito-managedloginbranding-clientid
        '''
        result = self._values.get("client_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def return_merged_resources(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''When ``true`` , returns values for branding options that are unchanged from Amazon Cognito defaults.

        When ``false`` or when you omit this parameter, returns only values that you customized in your branding style.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-managedloginbranding.html#cfn-cognito-managedloginbranding-returnmergedresources
        '''
        result = self._values.get("return_merged_resources")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def settings(self) -> typing.Any:
        '''A JSON file, encoded as a ``Document`` type, with the the settings that you want to apply to your style.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-managedloginbranding.html#cfn-cognito-managedloginbranding-settings
        '''
        result = self._values.get("settings")
        return typing.cast(typing.Any, result)

    @builtins.property
    def use_cognito_provided_values(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''When true, applies the default branding style options.

        This option reverts to default style options that are managed by Amazon Cognito. You can modify them later in the branding designer.

        When you specify ``true`` for this option, you must also omit values for ``Settings`` and ``Assets`` in the request.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-managedloginbranding.html#cfn-cognito-managedloginbranding-usecognitoprovidedvalues
        '''
        result = self._values.get("use_cognito_provided_values")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnManagedLoginBrandingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnUserPool(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool",
):
    '''The ``AWS::Cognito::UserPool`` resource creates an Amazon Cognito user pool.

    For more information on working with Amazon Cognito user pools, see `Amazon Cognito User Pools <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html>`_ and `CreateUserPool <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPool.html>`_ .
    .. epigraph::

       If you don't specify a value for a parameter, Amazon Cognito sets it to a default value.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html
    :cloudformationResource: AWS::Cognito::UserPool
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        # user_pool_tags: Any
        
        cfn_user_pool = cognito.CfnUserPool(self, "MyCfnUserPool",
            account_recovery_setting=cognito.CfnUserPool.AccountRecoverySettingProperty(
                recovery_mechanisms=[cognito.CfnUserPool.RecoveryOptionProperty(
                    name="name",
                    priority=123
                )]
            ),
            admin_create_user_config=cognito.CfnUserPool.AdminCreateUserConfigProperty(
                allow_admin_create_user_only=False,
                invite_message_template=cognito.CfnUserPool.InviteMessageTemplateProperty(
                    email_message="emailMessage",
                    email_subject="emailSubject",
                    sms_message="smsMessage"
                ),
                unused_account_validity_days=123
            ),
            alias_attributes=["aliasAttributes"],
            auto_verified_attributes=["autoVerifiedAttributes"],
            deletion_protection="deletionProtection",
            device_configuration=cognito.CfnUserPool.DeviceConfigurationProperty(
                challenge_required_on_new_device=False,
                device_only_remembered_on_user_prompt=False
            ),
            email_authentication_message="emailAuthenticationMessage",
            email_authentication_subject="emailAuthenticationSubject",
            email_configuration=cognito.CfnUserPool.EmailConfigurationProperty(
                configuration_set="configurationSet",
                email_sending_account="emailSendingAccount",
                from="from",
                reply_to_email_address="replyToEmailAddress",
                source_arn="sourceArn"
            ),
            email_verification_message="emailVerificationMessage",
            email_verification_subject="emailVerificationSubject",
            enabled_mfas=["enabledMfas"],
            lambda_config=cognito.CfnUserPool.LambdaConfigProperty(
                create_auth_challenge="createAuthChallenge",
                custom_email_sender=cognito.CfnUserPool.CustomEmailSenderProperty(
                    lambda_arn="lambdaArn",
                    lambda_version="lambdaVersion"
                ),
                custom_message="customMessage",
                custom_sms_sender=cognito.CfnUserPool.CustomSMSSenderProperty(
                    lambda_arn="lambdaArn",
                    lambda_version="lambdaVersion"
                ),
                define_auth_challenge="defineAuthChallenge",
                kms_key_id="kmsKeyId",
                post_authentication="postAuthentication",
                post_confirmation="postConfirmation",
                pre_authentication="preAuthentication",
                pre_sign_up="preSignUp",
                pre_token_generation="preTokenGeneration",
                pre_token_generation_config=cognito.CfnUserPool.PreTokenGenerationConfigProperty(
                    lambda_arn="lambdaArn",
                    lambda_version="lambdaVersion"
                ),
                user_migration="userMigration",
                verify_auth_challenge_response="verifyAuthChallengeResponse"
            ),
            mfa_configuration="mfaConfiguration",
            policies=cognito.CfnUserPool.PoliciesProperty(
                password_policy=cognito.CfnUserPool.PasswordPolicyProperty(
                    minimum_length=123,
                    password_history_size=123,
                    require_lowercase=False,
                    require_numbers=False,
                    require_symbols=False,
                    require_uppercase=False,
                    temporary_password_validity_days=123
                ),
                sign_in_policy=cognito.CfnUserPool.SignInPolicyProperty(
                    allowed_first_auth_factors=["allowedFirstAuthFactors"]
                )
            ),
            schema=[cognito.CfnUserPool.SchemaAttributeProperty(
                attribute_data_type="attributeDataType",
                developer_only_attribute=False,
                mutable=False,
                name="name",
                number_attribute_constraints=cognito.CfnUserPool.NumberAttributeConstraintsProperty(
                    max_value="maxValue",
                    min_value="minValue"
                ),
                required=False,
                string_attribute_constraints=cognito.CfnUserPool.StringAttributeConstraintsProperty(
                    max_length="maxLength",
                    min_length="minLength"
                )
            )],
            sms_authentication_message="smsAuthenticationMessage",
            sms_configuration=cognito.CfnUserPool.SmsConfigurationProperty(
                external_id="externalId",
                sns_caller_arn="snsCallerArn",
                sns_region="snsRegion"
            ),
            sms_verification_message="smsVerificationMessage",
            user_attribute_update_settings=cognito.CfnUserPool.UserAttributeUpdateSettingsProperty(
                attributes_require_verification_before_update=["attributesRequireVerificationBeforeUpdate"]
            ),
            username_attributes=["usernameAttributes"],
            username_configuration=cognito.CfnUserPool.UsernameConfigurationProperty(
                case_sensitive=False
            ),
            user_pool_add_ons=cognito.CfnUserPool.UserPoolAddOnsProperty(
                advanced_security_additional_flows=cognito.CfnUserPool.AdvancedSecurityAdditionalFlowsProperty(
                    custom_auth_mode="customAuthMode"
                ),
                advanced_security_mode="advancedSecurityMode"
            ),
            user_pool_name="userPoolName",
            user_pool_tags=user_pool_tags,
            user_pool_tier="userPoolTier",
            verification_message_template=cognito.CfnUserPool.VerificationMessageTemplateProperty(
                default_email_option="defaultEmailOption",
                email_message="emailMessage",
                email_message_by_link="emailMessageByLink",
                email_subject="emailSubject",
                email_subject_by_link="emailSubjectByLink",
                sms_message="smsMessage"
            ),
            web_authn_relying_party_id="webAuthnRelyingPartyId",
            web_authn_user_verification="webAuthnUserVerification"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        account_recovery_setting: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.AccountRecoverySettingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        admin_create_user_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.AdminCreateUserConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        alias_attributes: typing.Optional[typing.Sequence[builtins.str]] = None,
        auto_verified_attributes: typing.Optional[typing.Sequence[builtins.str]] = None,
        deletion_protection: typing.Optional[builtins.str] = None,
        device_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.DeviceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        email_authentication_message: typing.Optional[builtins.str] = None,
        email_authentication_subject: typing.Optional[builtins.str] = None,
        email_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.EmailConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        email_verification_message: typing.Optional[builtins.str] = None,
        email_verification_subject: typing.Optional[builtins.str] = None,
        enabled_mfas: typing.Optional[typing.Sequence[builtins.str]] = None,
        lambda_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.LambdaConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        mfa_configuration: typing.Optional[builtins.str] = None,
        policies: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.PoliciesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        schema: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.SchemaAttributeProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        sms_authentication_message: typing.Optional[builtins.str] = None,
        sms_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.SmsConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        sms_verification_message: typing.Optional[builtins.str] = None,
        user_attribute_update_settings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.UserAttributeUpdateSettingsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        username_attributes: typing.Optional[typing.Sequence[builtins.str]] = None,
        username_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.UsernameConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        user_pool_add_ons: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.UserPoolAddOnsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        user_pool_name: typing.Optional[builtins.str] = None,
        user_pool_tags: typing.Any = None,
        user_pool_tier: typing.Optional[builtins.str] = None,
        verification_message_template: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.VerificationMessageTemplateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        web_authn_relying_party_id: typing.Optional[builtins.str] = None,
        web_authn_user_verification: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param account_recovery_setting: The available verified method a user can use to recover their password when they call ``ForgotPassword`` . You can use this setting to define a preferred method when a user has more than one method available. With this setting, SMS doesn't qualify for a valid password recovery mechanism if the user also has SMS multi-factor authentication (MFA) activated. In the absence of this setting, Amazon Cognito uses the legacy behavior to determine the recovery method where SMS is preferred through email.
        :param admin_create_user_config: The settings for administrator creation of users in a user pool. Contains settings for allowing user sign-up, customizing invitation messages to new users, and the amount of time before temporary passwords expire.
        :param alias_attributes: Attributes supported as an alias for this user pool. For more information about alias attributes, see `Customizing sign-in attributes <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases>`_ .
        :param auto_verified_attributes: The attributes that you want your user pool to automatically verify. For more information, see `Verifying contact information at sign-up <https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html#allowing-users-to-sign-up-and-confirm-themselves>`_ .
        :param deletion_protection: When active, ``DeletionProtection`` prevents accidental deletion of your user pool. Before you can delete a user pool that you have protected against deletion, you must deactivate this feature. When you try to delete a protected user pool in a ``DeleteUserPool`` API request, Amazon Cognito returns an ``InvalidParameterException`` error. To delete a protected user pool, send a new ``DeleteUserPool`` request after you deactivate deletion protection in an ``UpdateUserPool`` API request.
        :param device_configuration: The device-remembering configuration for a user pool. Device remembering or device tracking is a "Remember me on this device" option for user pools that perform authentication with the device key of a trusted device in the back end, instead of a user-provided MFA code. For more information about device authentication, see `Working with user devices in your user pool <https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html>`_ . A null value indicates that you have deactivated device remembering in your user pool. .. epigraph:: When you provide a value for any ``DeviceConfiguration`` field, you activate the Amazon Cognito device-remembering feature. For more information, see `Working with devices <https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html>`_ .
        :param email_authentication_message: 
        :param email_authentication_subject: 
        :param email_configuration: The email configuration of your user pool. The email configuration type sets your preferred sending method, AWS Region, and sender for messages from your user pool.
        :param email_verification_message: This parameter is no longer used. See `VerificationMessageTemplateType <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html>`_ .
        :param email_verification_subject: This parameter is no longer used. See `VerificationMessageTemplateType <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html>`_ .
        :param enabled_mfas: Set enabled MFA options on a specified user pool. To disable all MFAs after it has been enabled, set ``MfaConfiguration`` to ``OFF`` and remove EnabledMfas. MFAs can only be all disabled if ``MfaConfiguration`` is ``OFF`` . After you enable ``SMS_MFA`` , you can only disable it by setting ``MfaConfiguration`` to ``OFF`` . Can be one of the following values: - ``SMS_MFA`` - Enables MFA with SMS for the user pool. To select this option, you must also provide values for ``SmsConfiguration`` . - ``SOFTWARE_TOKEN_MFA`` - Enables software token MFA for the user pool. - ``EMAIL_OTP`` - Enables MFA with email for the user pool. To select this option, you must provide values for ``EmailConfiguration`` and within those, set ``EmailSendingAccount`` to ``DEVELOPER`` . Allowed values: ``SMS_MFA`` | ``SOFTWARE_TOKEN_MFA`` | ``EMAIL_OTP``
        :param lambda_config: A collection of user pool Lambda triggers. Amazon Cognito invokes triggers at several possible stages of authentication operations. Triggers can modify the outcome of the operations that invoked them.
        :param mfa_configuration: Displays the state of multi-factor authentication (MFA) as on, off, or optional. When ``ON`` , all users must set up MFA before they can sign in. When ``OPTIONAL`` , your application must make a client-side determination of whether a user wants to register an MFA device. For user pools with adaptive authentication with threat protection, choose ``OPTIONAL`` . When ``MfaConfiguration`` is ``OPTIONAL`` , managed login doesn't automatically prompt users to set up MFA. Amazon Cognito generates MFA prompts in API responses and in managed login for users who have chosen and configured a preferred MFA factor.
        :param policies: A list of user pool policies. Contains the policy that sets password-complexity requirements.
        :param schema: An array of attributes for the new user pool. You can add custom attributes and modify the properties of default attributes. The specifications in this parameter set the required attributes in your user pool. For more information, see `Working with user attributes <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html>`_ .
        :param sms_authentication_message: The contents of the SMS authentication message.
        :param sms_configuration: The settings for your Amazon Cognito user pool to send SMS messages with Amazon Simple Notification Service. To send SMS messages with Amazon SNS in the AWS Region that you want, the Amazon Cognito user pool uses an AWS Identity and Access Management (IAM) role in your AWS account . For more information see `SMS message settings <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-sms-settings.html>`_ .
        :param sms_verification_message: This parameter is no longer used. See `VerificationMessageTemplateType <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html>`_ .
        :param user_attribute_update_settings: The settings for updates to user attributes. These settings include the property ``AttributesRequireVerificationBeforeUpdate`` , a user-pool setting that tells Amazon Cognito how to handle changes to the value of your users' email address and phone number attributes. For more information, see `Verifying updates to email addresses and phone numbers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html#user-pool-settings-verifications-verify-attribute-updates>`_ .
        :param username_attributes: Specifies whether a user can use an email address or phone number as a username when they sign up.
        :param username_configuration: Sets the case sensitivity option for sign-in usernames. When ``CaseSensitive`` is ``false`` (case insensitive), users can sign in with any combination of capital and lowercase letters. For example, ``username`` , ``USERNAME`` , or ``UserName`` , or for email, ``email@example.com`` or ``EMaiL@eXamplE.Com`` . For most use cases, set case sensitivity to ``false`` as a best practice. When usernames and email addresses are case insensitive, Amazon Cognito treats any variation in case as the same user, and prevents a case variation from being assigned to the same attribute for a different user. When ``CaseSensitive`` is ``true`` (case sensitive), Amazon Cognito interprets ``USERNAME`` and ``UserName`` as distinct users. This configuration is immutable after you set it.
        :param user_pool_add_ons: Contains settings for activation of threat protection, including the operating mode and additional authentication types. To log user security information but take no action, set to ``AUDIT`` . To configure automatic security responses to potentially unwanted traffic to your user pool, set to ``ENFORCED`` . For more information, see `Adding advanced security to a user pool <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-advanced-security.html>`_ . To activate this setting, your user pool must be on the `Plus tier <https://docs.aws.amazon.com/cognito/latest/developerguide/feature-plans-features-plus.html>`_ .
        :param user_pool_name: A friendly name for your user pool.
        :param user_pool_tags: The tag keys and values to assign to the user pool. A tag is a label that you can use to categorize and manage user pools in different ways, such as by purpose, owner, environment, or other criteria.
        :param user_pool_tier: The user pool `feature plan <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-sign-in-feature-plans.html>`_ , or tier. This parameter determines the eligibility of the user pool for features like managed login, access-token customization, and threat protection. Defaults to ``ESSENTIALS`` .
        :param verification_message_template: The template for the verification message that your user pool delivers to users who set an email address or phone number attribute. Set the email message type that corresponds to your ``DefaultEmailOption`` selection. For ``CONFIRM_WITH_LINK`` , specify an ``EmailMessageByLink`` and leave ``EmailMessage`` blank. For ``CONFIRM_WITH_CODE`` , specify an ``EmailMessage`` and leave ``EmailMessageByLink`` blank. When you supply both parameters with either choice, Amazon Cognito returns an error.
        :param web_authn_relying_party_id: Sets or displays the authentication domain, typically your user pool domain, that passkey providers must use as a relying party (RP) in their configuration. Under the following conditions, the passkey relying party ID must be the fully-qualified domain name of your custom domain: - The user pool is configured for passkey authentication. - The user pool has a custom domain, whether or not it also has a prefix domain. - Your application performs authentication with managed login or the classic hosted UI.
        :param web_authn_user_verification: When ``required`` , users can only register and sign in users with passkeys that are capable of `user verification <https://docs.aws.amazon.com/https://www.w3.org/TR/webauthn-2/#enum-userVerificationRequirement>`_ . When ``preferred`` , your user pool doesn't require the use of authenticators with user verification but encourages it.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__32d20f28e2758f9a461380e2ed5d06233baf0f45541047ba837f26ebc37ee551)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUserPoolProps(
            account_recovery_setting=account_recovery_setting,
            admin_create_user_config=admin_create_user_config,
            alias_attributes=alias_attributes,
            auto_verified_attributes=auto_verified_attributes,
            deletion_protection=deletion_protection,
            device_configuration=device_configuration,
            email_authentication_message=email_authentication_message,
            email_authentication_subject=email_authentication_subject,
            email_configuration=email_configuration,
            email_verification_message=email_verification_message,
            email_verification_subject=email_verification_subject,
            enabled_mfas=enabled_mfas,
            lambda_config=lambda_config,
            mfa_configuration=mfa_configuration,
            policies=policies,
            schema=schema,
            sms_authentication_message=sms_authentication_message,
            sms_configuration=sms_configuration,
            sms_verification_message=sms_verification_message,
            user_attribute_update_settings=user_attribute_update_settings,
            username_attributes=username_attributes,
            username_configuration=username_configuration,
            user_pool_add_ons=user_pool_add_ons,
            user_pool_name=user_pool_name,
            user_pool_tags=user_pool_tags,
            user_pool_tier=user_pool_tier,
            verification_message_template=verification_message_template,
            web_authn_relying_party_id=web_authn_relying_party_id,
            web_authn_user_verification=web_authn_user_verification,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__190e7831a65154362cd300369d61a522f7add5d60374fc990db7dd8ecb232388)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f0f2e990078e39ddda95cb24d40eda32e024a9c8840bebebbe6ed0ffb0ab9353)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the user pool, such as ``arn:aws:cognito-idp:us-east-1:123412341234:userpool/us-east-1_123412341`` .

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrProviderName")
    def attr_provider_name(self) -> builtins.str:
        '''A friendly name for the IdP.

        :cloudformationAttribute: ProviderName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrProviderName"))

    @builtins.property
    @jsii.member(jsii_name="attrProviderUrl")
    def attr_provider_url(self) -> builtins.str:
        '''The URL of the provider of the Amazon Cognito user pool, specified as a ``String`` .

        :cloudformationAttribute: ProviderURL
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrProviderUrl"))

    @builtins.property
    @jsii.member(jsii_name="attrUserPoolId")
    def attr_user_pool_id(self) -> builtins.str:
        '''The ID of the user pool.

        :cloudformationAttribute: UserPoolId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUserPoolId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="accountRecoverySetting")
    def account_recovery_setting(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.AccountRecoverySettingProperty"]]:
        '''The available verified method a user can use to recover their password when they call ``ForgotPassword`` .'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.AccountRecoverySettingProperty"]], jsii.get(self, "accountRecoverySetting"))

    @account_recovery_setting.setter
    def account_recovery_setting(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.AccountRecoverySettingProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__da99efaebeb66459045576f8f7065b4ef07804424df0623f038bbd64d754aacc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "accountRecoverySetting", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="adminCreateUserConfig")
    def admin_create_user_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.AdminCreateUserConfigProperty"]]:
        '''The settings for administrator creation of users in a user pool.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.AdminCreateUserConfigProperty"]], jsii.get(self, "adminCreateUserConfig"))

    @admin_create_user_config.setter
    def admin_create_user_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.AdminCreateUserConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eaaa881203c259b6afa777936a9ca0096f8c631069a5ebd10e4ad0bd2c5229d4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "adminCreateUserConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="aliasAttributes")
    def alias_attributes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Attributes supported as an alias for this user pool.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "aliasAttributes"))

    @alias_attributes.setter
    def alias_attributes(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cc3f2402f4f88736f7b230549074e49d6aee20281b90dbeeb4a0bb5c031e81d2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "aliasAttributes", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="autoVerifiedAttributes")
    def auto_verified_attributes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The attributes that you want your user pool to automatically verify.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "autoVerifiedAttributes"))

    @auto_verified_attributes.setter
    def auto_verified_attributes(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__67b5b672ef087a1ed7f05991df5c156df67b6c51d144847ad452e5f52c4bcbe5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoVerifiedAttributes", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="deletionProtection")
    def deletion_protection(self) -> typing.Optional[builtins.str]:
        '''When active, ``DeletionProtection`` prevents accidental deletion of your user pool.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "deletionProtection"))

    @deletion_protection.setter
    def deletion_protection(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__455a3b25859643cc12071049235f2feb590ee06f70ec42d2cd83bc166d9e9c9f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deletionProtection", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="deviceConfiguration")
    def device_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.DeviceConfigurationProperty"]]:
        '''The device-remembering configuration for a user pool.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.DeviceConfigurationProperty"]], jsii.get(self, "deviceConfiguration"))

    @device_configuration.setter
    def device_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.DeviceConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7eb821a70b459056b6d26722d06f8b948b315111a936804d8aba0e7ff07d13bf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deviceConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="emailAuthenticationMessage")
    def email_authentication_message(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "emailAuthenticationMessage"))

    @email_authentication_message.setter
    def email_authentication_message(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3cf4765f879f49f79c6984252af6993fe6fdf6838989608b11e192c544fce53c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "emailAuthenticationMessage", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="emailAuthenticationSubject")
    def email_authentication_subject(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "emailAuthenticationSubject"))

    @email_authentication_subject.setter
    def email_authentication_subject(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e1d4523d17f0641e76142be67287be5dc758d191f5eba3fa217d8c5d0170791)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "emailAuthenticationSubject", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="emailConfiguration")
    def email_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.EmailConfigurationProperty"]]:
        '''The email configuration of your user pool.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.EmailConfigurationProperty"]], jsii.get(self, "emailConfiguration"))

    @email_configuration.setter
    def email_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.EmailConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__32d0b53f74dc294b25f20f54bcdaf8477a3dfc8b505387d70f97f7febe6ae209)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "emailConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="emailVerificationMessage")
    def email_verification_message(self) -> typing.Optional[builtins.str]:
        '''This parameter is no longer used.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "emailVerificationMessage"))

    @email_verification_message.setter
    def email_verification_message(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__82f2fd26b3ec2cf294bdc8be0a8bbae7dbacd31abbdbead2a75eac9566efd913)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "emailVerificationMessage", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="emailVerificationSubject")
    def email_verification_subject(self) -> typing.Optional[builtins.str]:
        '''This parameter is no longer used.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "emailVerificationSubject"))

    @email_verification_subject.setter
    def email_verification_subject(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e72167174985dddad82c3fffee5bd2e72f61e1a560cae39101e0f0bfa55a6c74)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "emailVerificationSubject", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="enabledMfas")
    def enabled_mfas(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Set enabled MFA options on a specified user pool.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "enabledMfas"))

    @enabled_mfas.setter
    def enabled_mfas(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__043cb4b4941782c98919a853fc16bce8663c274df01579f9caeb0cc2a0e4c28a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enabledMfas", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="lambdaConfig")
    def lambda_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.LambdaConfigProperty"]]:
        '''A collection of user pool Lambda triggers.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.LambdaConfigProperty"]], jsii.get(self, "lambdaConfig"))

    @lambda_config.setter
    def lambda_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.LambdaConfigProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8c7be7c0b2e4badc8ab3ce5fb0f327b3c26ebcb18bb31bc39a369f1c5fd64877)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "lambdaConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="mfaConfiguration")
    def mfa_configuration(self) -> typing.Optional[builtins.str]:
        '''Displays the state of multi-factor authentication (MFA) as on, off, or optional.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "mfaConfiguration"))

    @mfa_configuration.setter
    def mfa_configuration(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__02262b9b75d7d340810cfbaca693eeab69c3422010742739d64170db818cd1c5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "mfaConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="policies")
    def policies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.PoliciesProperty"]]:
        '''A list of user pool policies.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.PoliciesProperty"]], jsii.get(self, "policies"))

    @policies.setter
    def policies(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.PoliciesProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cc547f06b5c5b29499b9c1debda606db3aa38d2243e5ea837490a194d6ee562d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "policies", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="schema")
    def schema(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPool.SchemaAttributeProperty"]]]]:
        '''An array of attributes for the new user pool.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPool.SchemaAttributeProperty"]]]], jsii.get(self, "schema"))

    @schema.setter
    def schema(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPool.SchemaAttributeProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f0e5b663faa7be35e96a6bb7a25793a3650b322376c4df96db6ab04037432ec8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "schema", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="smsAuthenticationMessage")
    def sms_authentication_message(self) -> typing.Optional[builtins.str]:
        '''The contents of the SMS authentication message.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "smsAuthenticationMessage"))

    @sms_authentication_message.setter
    def sms_authentication_message(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b0033e90845dce5d5ed18e8cdb36333bda83daa85a52bf9b317447a2b26bb923)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "smsAuthenticationMessage", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="smsConfiguration")
    def sms_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.SmsConfigurationProperty"]]:
        '''The settings for your Amazon Cognito user pool to send SMS messages with Amazon Simple Notification Service.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.SmsConfigurationProperty"]], jsii.get(self, "smsConfiguration"))

    @sms_configuration.setter
    def sms_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.SmsConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7770291e207a266c6092133c98f74702280b1b762eadf5a38996872fd156c65d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "smsConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="smsVerificationMessage")
    def sms_verification_message(self) -> typing.Optional[builtins.str]:
        '''This parameter is no longer used.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "smsVerificationMessage"))

    @sms_verification_message.setter
    def sms_verification_message(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__698f1380b9328c4a773190401993fe201c403d6b9d10c9d4d82aac1836766754)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "smsVerificationMessage", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userAttributeUpdateSettings")
    def user_attribute_update_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.UserAttributeUpdateSettingsProperty"]]:
        '''The settings for updates to user attributes.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.UserAttributeUpdateSettingsProperty"]], jsii.get(self, "userAttributeUpdateSettings"))

    @user_attribute_update_settings.setter
    def user_attribute_update_settings(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.UserAttributeUpdateSettingsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__092e452d396de66cdb0e7fad3a126dc88f5db6854ad416e350af1806e6465101)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userAttributeUpdateSettings", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="usernameAttributes")
    def username_attributes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specifies whether a user can use an email address or phone number as a username when they sign up.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "usernameAttributes"))

    @username_attributes.setter
    def username_attributes(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__70fd85fe188c058531323dc3601df251b6fe7e56cbb9e62fb3d7b9f710f2597c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "usernameAttributes", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="usernameConfiguration")
    def username_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.UsernameConfigurationProperty"]]:
        '''Sets the case sensitivity option for sign-in usernames.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.UsernameConfigurationProperty"]], jsii.get(self, "usernameConfiguration"))

    @username_configuration.setter
    def username_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.UsernameConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5021fa9a3421f51a3fb6735566cc12e369ce2d1a11bc37330e0278145be57e21)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "usernameConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userPoolAddOns")
    def user_pool_add_ons(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.UserPoolAddOnsProperty"]]:
        '''Contains settings for activation of threat protection, including the operating mode and additional authentication types.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.UserPoolAddOnsProperty"]], jsii.get(self, "userPoolAddOns"))

    @user_pool_add_ons.setter
    def user_pool_add_ons(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.UserPoolAddOnsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9387d4cd9039881cdf533356e5c6db2db0d1781598e0bee842b7485eca8c8fb2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolAddOns", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userPoolName")
    def user_pool_name(self) -> typing.Optional[builtins.str]:
        '''A friendly name for your user pool.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "userPoolName"))

    @user_pool_name.setter
    def user_pool_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a877e7ca735a6cf85b712d1b09de98f59adcdbef66ffacb8f24d5687caa6a37a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userPoolTagsRaw")
    def user_pool_tags_raw(self) -> typing.Any:
        '''The tag keys and values to assign to the user pool.'''
        return typing.cast(typing.Any, jsii.get(self, "userPoolTagsRaw"))

    @user_pool_tags_raw.setter
    def user_pool_tags_raw(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c0ac7e31445bc4b1c75709a8cde084565899b188ecd5ed75a434afb3fecf7e3f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolTagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userPoolTier")
    def user_pool_tier(self) -> typing.Optional[builtins.str]:
        '''The user pool `feature plan <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-sign-in-feature-plans.html>`_ , or tier. This parameter determines the eligibility of the user pool for features like managed login, access-token customization, and threat protection. Defaults to ``ESSENTIALS`` .'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "userPoolTier"))

    @user_pool_tier.setter
    def user_pool_tier(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7b34a7e631952732eaf3564630f968b4a1066c2249e1bd77fa5894ac20d552db)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolTier", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="verificationMessageTemplate")
    def verification_message_template(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.VerificationMessageTemplateProperty"]]:
        '''The template for the verification message that your user pool delivers to users who set an email address or phone number attribute.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.VerificationMessageTemplateProperty"]], jsii.get(self, "verificationMessageTemplate"))

    @verification_message_template.setter
    def verification_message_template(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.VerificationMessageTemplateProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9163d1ccc0cf294430031f1b8b5289192a6e048b52e8181e9ca8707780aac888)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "verificationMessageTemplate", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="webAuthnRelyingPartyId")
    def web_authn_relying_party_id(self) -> typing.Optional[builtins.str]:
        '''Sets or displays the authentication domain, typically your user pool domain, that passkey providers must use as a relying party (RP) in their configuration.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "webAuthnRelyingPartyId"))

    @web_authn_relying_party_id.setter
    def web_authn_relying_party_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a2852b3b820fa8903c8ee86e4c615c763dbc2f40270d7dddb4851a596a4b629)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "webAuthnRelyingPartyId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="webAuthnUserVerification")
    def web_authn_user_verification(self) -> typing.Optional[builtins.str]:
        '''When ``required`` , users can only register and sign in users with passkeys that are capable of `user verification <https://docs.aws.amazon.com/https://www.w3.org/TR/webauthn-2/#enum-userVerificationRequirement>`_ . When ``preferred`` , your user pool doesn't require the use of authenticators with user verification but encourages it.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "webAuthnUserVerification"))

    @web_authn_user_verification.setter
    def web_authn_user_verification(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__39e1b7a43a4375c7269c036061949915e9a6e4528f8341df4df0a6b046ac6a11)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "webAuthnUserVerification", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.AccountRecoverySettingProperty",
        jsii_struct_bases=[],
        name_mapping={"recovery_mechanisms": "recoveryMechanisms"},
    )
    class AccountRecoverySettingProperty:
        def __init__(
            self,
            *,
            recovery_mechanisms: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.RecoveryOptionProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The available verified method a user can use to recover their password when they call ``ForgotPassword`` .

            You can use this setting to define a preferred method when a user has more than one method available. With this setting, SMS doesn't qualify for a valid password recovery mechanism if the user also has SMS multi-factor authentication (MFA) activated. In the absence of this setting, Amazon Cognito uses the legacy behavior to determine the recovery method where SMS is preferred through email.

            :param recovery_mechanisms: The list of options and priorities for user message delivery in forgot-password operations. Sets or displays user pool preferences for email or SMS message priority, whether users should fall back to a second delivery method, and whether passwords should only be reset by administrators.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-accountrecoverysetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                account_recovery_setting_property = cognito.CfnUserPool.AccountRecoverySettingProperty(
                    recovery_mechanisms=[cognito.CfnUserPool.RecoveryOptionProperty(
                        name="name",
                        priority=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__79c7f36a6b9a834beef59493981fd8b3c56dae29d4d3d36bb9b0a65305ebd4ce)
                check_type(argname="argument recovery_mechanisms", value=recovery_mechanisms, expected_type=type_hints["recovery_mechanisms"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if recovery_mechanisms is not None:
                self._values["recovery_mechanisms"] = recovery_mechanisms

        @builtins.property
        def recovery_mechanisms(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPool.RecoveryOptionProperty"]]]]:
            '''The list of options and priorities for user message delivery in forgot-password operations.

            Sets or displays user pool preferences for email or SMS message priority, whether users should fall back to a second delivery method, and whether passwords should only be reset by administrators.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-accountrecoverysetting.html#cfn-cognito-userpool-accountrecoverysetting-recoverymechanisms
            '''
            result = self._values.get("recovery_mechanisms")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPool.RecoveryOptionProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccountRecoverySettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.AdminCreateUserConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "allow_admin_create_user_only": "allowAdminCreateUserOnly",
            "invite_message_template": "inviteMessageTemplate",
            "unused_account_validity_days": "unusedAccountValidityDays",
        },
    )
    class AdminCreateUserConfigProperty:
        def __init__(
            self,
            *,
            allow_admin_create_user_only: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            invite_message_template: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.InviteMessageTemplateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            unused_account_validity_days: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The settings for administrator creation of users in a user pool.

            Contains settings for allowing user sign-up, customizing invitation messages to new users, and the amount of time before temporary passwords expire.

            :param allow_admin_create_user_only: The setting for allowing self-service sign-up. When ``true`` , only administrators can create new user profiles. When ``false`` , users can register themselves and create a new user profile with the ``SignUp`` operation.
            :param invite_message_template: The template for the welcome message to new users. This template must include the ``{####}`` temporary password placeholder if you are creating users with passwords. If your users don't have passwords, you can omit the placeholder. See also `Customizing User Invitation Messages <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-customizations.html#cognito-user-pool-settings-user-invitation-message-customization>`_ .
            :param unused_account_validity_days: This parameter is no longer in use. The password expiration limit in days for administrator-created users. When this time expires, the user can't sign in with their temporary password. To reset the account after that time limit, you must call ``AdminCreateUser`` again, specifying ``RESEND`` for the ``MessageAction`` parameter. The default value for this parameter is 7.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-admincreateuserconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                admin_create_user_config_property = cognito.CfnUserPool.AdminCreateUserConfigProperty(
                    allow_admin_create_user_only=False,
                    invite_message_template=cognito.CfnUserPool.InviteMessageTemplateProperty(
                        email_message="emailMessage",
                        email_subject="emailSubject",
                        sms_message="smsMessage"
                    ),
                    unused_account_validity_days=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__eccc17237dcc735721e8ed9027a445ee9831366ce5b71c7ee0c61cc0654098be)
                check_type(argname="argument allow_admin_create_user_only", value=allow_admin_create_user_only, expected_type=type_hints["allow_admin_create_user_only"])
                check_type(argname="argument invite_message_template", value=invite_message_template, expected_type=type_hints["invite_message_template"])
                check_type(argname="argument unused_account_validity_days", value=unused_account_validity_days, expected_type=type_hints["unused_account_validity_days"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if allow_admin_create_user_only is not None:
                self._values["allow_admin_create_user_only"] = allow_admin_create_user_only
            if invite_message_template is not None:
                self._values["invite_message_template"] = invite_message_template
            if unused_account_validity_days is not None:
                self._values["unused_account_validity_days"] = unused_account_validity_days

        @builtins.property
        def allow_admin_create_user_only(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''The setting for allowing self-service sign-up.

            When ``true`` , only administrators can create new user profiles. When ``false`` , users can register themselves and create a new user profile with the ``SignUp`` operation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-admincreateuserconfig.html#cfn-cognito-userpool-admincreateuserconfig-allowadmincreateuseronly
            '''
            result = self._values.get("allow_admin_create_user_only")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def invite_message_template(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.InviteMessageTemplateProperty"]]:
            '''The template for the welcome message to new users.

            This template must include the ``{####}`` temporary password placeholder if you are creating users with passwords. If your users don't have passwords, you can omit the placeholder.

            See also `Customizing User Invitation Messages <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-customizations.html#cognito-user-pool-settings-user-invitation-message-customization>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-admincreateuserconfig.html#cfn-cognito-userpool-admincreateuserconfig-invitemessagetemplate
            '''
            result = self._values.get("invite_message_template")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.InviteMessageTemplateProperty"]], result)

        @builtins.property
        def unused_account_validity_days(self) -> typing.Optional[jsii.Number]:
            '''This parameter is no longer in use.

            The password expiration limit in days for administrator-created users. When this time expires, the user can't sign in with their temporary password. To reset the account after that time limit, you must call ``AdminCreateUser`` again, specifying ``RESEND`` for the ``MessageAction`` parameter.

            The default value for this parameter is 7.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-admincreateuserconfig.html#cfn-cognito-userpool-admincreateuserconfig-unusedaccountvaliditydays
            '''
            result = self._values.get("unused_account_validity_days")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AdminCreateUserConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.AdvancedSecurityAdditionalFlowsProperty",
        jsii_struct_bases=[],
        name_mapping={"custom_auth_mode": "customAuthMode"},
    )
    class AdvancedSecurityAdditionalFlowsProperty:
        def __init__(
            self,
            *,
            custom_auth_mode: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Threat protection configuration options for additional authentication types in your user pool, including custom authentication.

            :param custom_auth_mode: The operating mode of threat protection in custom authentication with `Custom authentication challenge Lambda triggers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-challenge.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-advancedsecurityadditionalflows.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                advanced_security_additional_flows_property = cognito.CfnUserPool.AdvancedSecurityAdditionalFlowsProperty(
                    custom_auth_mode="customAuthMode"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__13d285bc1be8956472b23e9b2a73954ac4f3700dcebec3973e7b6cbad4639935)
                check_type(argname="argument custom_auth_mode", value=custom_auth_mode, expected_type=type_hints["custom_auth_mode"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if custom_auth_mode is not None:
                self._values["custom_auth_mode"] = custom_auth_mode

        @builtins.property
        def custom_auth_mode(self) -> typing.Optional[builtins.str]:
            '''The operating mode of threat protection in custom authentication with `Custom authentication challenge Lambda triggers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-challenge.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-advancedsecurityadditionalflows.html#cfn-cognito-userpool-advancedsecurityadditionalflows-customauthmode
            '''
            result = self._values.get("custom_auth_mode")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AdvancedSecurityAdditionalFlowsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.CustomEmailSenderProperty",
        jsii_struct_bases=[],
        name_mapping={"lambda_arn": "lambdaArn", "lambda_version": "lambdaVersion"},
    )
    class CustomEmailSenderProperty:
        def __init__(
            self,
            *,
            lambda_arn: typing.Optional[builtins.str] = None,
            lambda_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The configuration of a custom email sender Lambda trigger.

            This trigger routes all email notifications from a user pool to a Lambda function that delivers the message using custom logic.

            :param lambda_arn: The Amazon Resource Name (ARN) of the function that you want to assign to your Lambda trigger.
            :param lambda_version: The user pool trigger version of the request that Amazon Cognito sends to your Lambda function. Higher-numbered versions add fields that support new features. You must use a ``LambdaVersion`` of ``V1_0`` with a custom sender function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-customemailsender.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                custom_email_sender_property = cognito.CfnUserPool.CustomEmailSenderProperty(
                    lambda_arn="lambdaArn",
                    lambda_version="lambdaVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__820d7b1c7eb3367a1ba085ab349db46e7dfd21d8e9d99e1a78219dc6a2258553)
                check_type(argname="argument lambda_arn", value=lambda_arn, expected_type=type_hints["lambda_arn"])
                check_type(argname="argument lambda_version", value=lambda_version, expected_type=type_hints["lambda_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if lambda_arn is not None:
                self._values["lambda_arn"] = lambda_arn
            if lambda_version is not None:
                self._values["lambda_version"] = lambda_version

        @builtins.property
        def lambda_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the function that you want to assign to your Lambda trigger.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-customemailsender.html#cfn-cognito-userpool-customemailsender-lambdaarn
            '''
            result = self._values.get("lambda_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def lambda_version(self) -> typing.Optional[builtins.str]:
            '''The user pool trigger version of the request that Amazon Cognito sends to your Lambda function.

            Higher-numbered versions add fields that support new features.

            You must use a ``LambdaVersion`` of ``V1_0`` with a custom sender function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-customemailsender.html#cfn-cognito-userpool-customemailsender-lambdaversion
            '''
            result = self._values.get("lambda_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CustomEmailSenderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.CustomSMSSenderProperty",
        jsii_struct_bases=[],
        name_mapping={"lambda_arn": "lambdaArn", "lambda_version": "lambdaVersion"},
    )
    class CustomSMSSenderProperty:
        def __init__(
            self,
            *,
            lambda_arn: typing.Optional[builtins.str] = None,
            lambda_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The configuration of a custom SMS sender Lambda trigger.

            This trigger routes all SMS notifications from a user pool to a Lambda function that delivers the message using custom logic.

            :param lambda_arn: The Amazon Resource Name (ARN) of the function that you want to assign to your Lambda trigger.
            :param lambda_version: The user pool trigger version of the request that Amazon Cognito sends to your Lambda function. Higher-numbered versions add fields that support new features. You must use a ``LambdaVersion`` of ``V1_0`` with a custom sender function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-customsmssender.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                custom_sMSSender_property = cognito.CfnUserPool.CustomSMSSenderProperty(
                    lambda_arn="lambdaArn",
                    lambda_version="lambdaVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a5d7a0658af79be04111d251e361312d42c0f9017a17b247a2d1638ccf82fe0b)
                check_type(argname="argument lambda_arn", value=lambda_arn, expected_type=type_hints["lambda_arn"])
                check_type(argname="argument lambda_version", value=lambda_version, expected_type=type_hints["lambda_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if lambda_arn is not None:
                self._values["lambda_arn"] = lambda_arn
            if lambda_version is not None:
                self._values["lambda_version"] = lambda_version

        @builtins.property
        def lambda_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the function that you want to assign to your Lambda trigger.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-customsmssender.html#cfn-cognito-userpool-customsmssender-lambdaarn
            '''
            result = self._values.get("lambda_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def lambda_version(self) -> typing.Optional[builtins.str]:
            '''The user pool trigger version of the request that Amazon Cognito sends to your Lambda function.

            Higher-numbered versions add fields that support new features.

            You must use a ``LambdaVersion`` of ``V1_0`` with a custom sender function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-customsmssender.html#cfn-cognito-userpool-customsmssender-lambdaversion
            '''
            result = self._values.get("lambda_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CustomSMSSenderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.DeviceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "challenge_required_on_new_device": "challengeRequiredOnNewDevice",
            "device_only_remembered_on_user_prompt": "deviceOnlyRememberedOnUserPrompt",
        },
    )
    class DeviceConfigurationProperty:
        def __init__(
            self,
            *,
            challenge_required_on_new_device: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            device_only_remembered_on_user_prompt: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''The device-remembering configuration for a user pool.

            :param challenge_required_on_new_device: When true, a remembered device can sign in with device authentication instead of SMS and time-based one-time password (TOTP) factors for multi-factor authentication (MFA). .. epigraph:: Whether or not ``ChallengeRequiredOnNewDevice`` is true, users who sign in with devices that have not been confirmed or remembered must still provide a second factor in a user pool that requires MFA.
            :param device_only_remembered_on_user_prompt: When true, Amazon Cognito doesn't automatically remember a user's device when your app sends a ``ConfirmDevice`` API request. In your app, create a prompt for your user to choose whether they want to remember their device. Return the user's choice in an ``UpdateDeviceStatus`` API request. When ``DeviceOnlyRememberedOnUserPrompt`` is ``false`` , Amazon Cognito immediately remembers devices that you register in a ``ConfirmDevice`` API request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-deviceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                device_configuration_property = cognito.CfnUserPool.DeviceConfigurationProperty(
                    challenge_required_on_new_device=False,
                    device_only_remembered_on_user_prompt=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4f9bd553445392044237353e227a2ec9bd347d85cebe0bacf0660ef00ac950f0)
                check_type(argname="argument challenge_required_on_new_device", value=challenge_required_on_new_device, expected_type=type_hints["challenge_required_on_new_device"])
                check_type(argname="argument device_only_remembered_on_user_prompt", value=device_only_remembered_on_user_prompt, expected_type=type_hints["device_only_remembered_on_user_prompt"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if challenge_required_on_new_device is not None:
                self._values["challenge_required_on_new_device"] = challenge_required_on_new_device
            if device_only_remembered_on_user_prompt is not None:
                self._values["device_only_remembered_on_user_prompt"] = device_only_remembered_on_user_prompt

        @builtins.property
        def challenge_required_on_new_device(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''When true, a remembered device can sign in with device authentication instead of SMS and time-based one-time password (TOTP) factors for multi-factor authentication (MFA).

            .. epigraph::

               Whether or not ``ChallengeRequiredOnNewDevice`` is true, users who sign in with devices that have not been confirmed or remembered must still provide a second factor in a user pool that requires MFA.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-deviceconfiguration.html#cfn-cognito-userpool-deviceconfiguration-challengerequiredonnewdevice
            '''
            result = self._values.get("challenge_required_on_new_device")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def device_only_remembered_on_user_prompt(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''When true, Amazon Cognito doesn't automatically remember a user's device when your app sends a ``ConfirmDevice`` API request.

            In your app, create a prompt for your user to choose whether they want to remember their device. Return the user's choice in an ``UpdateDeviceStatus`` API request.

            When ``DeviceOnlyRememberedOnUserPrompt`` is ``false`` , Amazon Cognito immediately remembers devices that you register in a ``ConfirmDevice`` API request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-deviceconfiguration.html#cfn-cognito-userpool-deviceconfiguration-deviceonlyrememberedonuserprompt
            '''
            result = self._values.get("device_only_remembered_on_user_prompt")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeviceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.EmailConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "configuration_set": "configurationSet",
            "email_sending_account": "emailSendingAccount",
            "from_": "from",
            "reply_to_email_address": "replyToEmailAddress",
            "source_arn": "sourceArn",
        },
    )
    class EmailConfigurationProperty:
        def __init__(
            self,
            *,
            configuration_set: typing.Optional[builtins.str] = None,
            email_sending_account: typing.Optional[builtins.str] = None,
            from_: typing.Optional[builtins.str] = None,
            reply_to_email_address: typing.Optional[builtins.str] = None,
            source_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The email configuration of your user pool.

            The email configuration type sets your preferred sending method, AWS Region, and sender for messages from your user pool.

            :param configuration_set: The set of configuration rules that can be applied to emails sent using Amazon Simple Email Service. A configuration set is applied to an email by including a reference to the configuration set in the headers of the email. Once applied, all of the rules in that configuration set are applied to the email. Configuration sets can be used to apply the following types of rules to emails: - **Event publishing** - Amazon Simple Email Service can track the number of send, delivery, open, click, bounce, and complaint events for each email sent. Use event publishing to send information about these events to other AWS services such as and Amazon CloudWatch - **IP pool management** - When leasing dedicated IP addresses with Amazon Simple Email Service, you can create groups of IP addresses, called dedicated IP pools. You can then associate the dedicated IP pools with configuration sets.
            :param email_sending_account: Specifies whether Amazon Cognito uses its built-in functionality to send your users email messages, or uses your Amazon Simple Email Service email configuration. Specify one of the following values: - **COGNITO_DEFAULT** - When Amazon Cognito emails your users, it uses its built-in email functionality. When you use the default option, Amazon Cognito allows only a limited number of emails each day for your user pool. For typical production environments, the default email limit is less than the required delivery volume. To achieve a higher delivery volume, specify DEVELOPER to use your Amazon SES email configuration. To look up the email delivery limit for the default option, see `Limits <https://docs.aws.amazon.com/cognito/latest/developerguide/limits.html>`_ in the *Amazon Cognito Developer Guide* . The default FROM address is ``no-reply@verificationemail.com`` . To customize the FROM address, provide the Amazon Resource Name (ARN) of an Amazon SES verified email address for the ``SourceArn`` parameter. - **DEVELOPER** - When Amazon Cognito emails your users, it uses your Amazon SES configuration. Amazon Cognito calls Amazon SES on your behalf to send email from your verified email address. When you use this option, the email delivery limits are the same limits that apply to your Amazon SES verified email address in your AWS account . If you use this option, provide the ARN of an Amazon SES verified email address for the ``SourceArn`` parameter. Before Amazon Cognito can email your users, it requires additional permissions to call Amazon SES on your behalf. When you update your user pool with this option, Amazon Cognito creates a *service-linked role* , which is a type of role in your AWS account . This role contains the permissions that allow you to access Amazon SES and send email messages from your email address. For more information about the service-linked role that Amazon Cognito creates, see `Using Service-Linked Roles for Amazon Cognito <https://docs.aws.amazon.com/cognito/latest/developerguide/using-service-linked-roles.html>`_ in the *Amazon Cognito Developer Guide* .
            :param from_: Either the sender’s email address or the sender’s name with their email address. For example, ``testuser@example.com`` or ``Test User <testuser@example.com>`` . This address appears before the body of the email.
            :param reply_to_email_address: The destination to which the receiver of the email should reply.
            :param source_arn: The ARN of a verified email address or an address from a verified domain in Amazon SES. You can set a ``SourceArn`` email from a verified domain only with an API request. You can set a verified email address, but not an address in a verified domain, in the Amazon Cognito console. Amazon Cognito uses the email address that you provide in one of the following ways, depending on the value that you specify for the ``EmailSendingAccount`` parameter: - If you specify ``COGNITO_DEFAULT`` , Amazon Cognito uses this address as the custom FROM address when it emails your users using its built-in email account. - If you specify ``DEVELOPER`` , Amazon Cognito emails your users with this address by calling Amazon SES on your behalf. The Region value of the ``SourceArn`` parameter must indicate a supported AWS Region of your user pool. Typically, the Region in the ``SourceArn`` and the user pool Region are the same. For more information, see `Amazon SES email configuration regions <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html#user-pool-email-developer-region-mapping>`_ in the `Amazon Cognito Developer Guide <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-emailconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                email_configuration_property = cognito.CfnUserPool.EmailConfigurationProperty(
                    configuration_set="configurationSet",
                    email_sending_account="emailSendingAccount",
                    from="from",
                    reply_to_email_address="replyToEmailAddress",
                    source_arn="sourceArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__aec7382b1b4972784ecb012d15fbc2441bfecae360d5f26e90d1da3ce7841785)
                check_type(argname="argument configuration_set", value=configuration_set, expected_type=type_hints["configuration_set"])
                check_type(argname="argument email_sending_account", value=email_sending_account, expected_type=type_hints["email_sending_account"])
                check_type(argname="argument from_", value=from_, expected_type=type_hints["from_"])
                check_type(argname="argument reply_to_email_address", value=reply_to_email_address, expected_type=type_hints["reply_to_email_address"])
                check_type(argname="argument source_arn", value=source_arn, expected_type=type_hints["source_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if configuration_set is not None:
                self._values["configuration_set"] = configuration_set
            if email_sending_account is not None:
                self._values["email_sending_account"] = email_sending_account
            if from_ is not None:
                self._values["from_"] = from_
            if reply_to_email_address is not None:
                self._values["reply_to_email_address"] = reply_to_email_address
            if source_arn is not None:
                self._values["source_arn"] = source_arn

        @builtins.property
        def configuration_set(self) -> typing.Optional[builtins.str]:
            '''The set of configuration rules that can be applied to emails sent using Amazon Simple Email Service.

            A configuration set is applied to an email by including a reference to the configuration set in the headers of the email. Once applied, all of the rules in that configuration set are applied to the email. Configuration sets can be used to apply the following types of rules to emails:

            - **Event publishing** - Amazon Simple Email Service can track the number of send, delivery, open, click, bounce, and complaint events for each email sent. Use event publishing to send information about these events to other AWS services such as and Amazon CloudWatch
            - **IP pool management** - When leasing dedicated IP addresses with Amazon Simple Email Service, you can create groups of IP addresses, called dedicated IP pools. You can then associate the dedicated IP pools with configuration sets.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-emailconfiguration.html#cfn-cognito-userpool-emailconfiguration-configurationset
            '''
            result = self._values.get("configuration_set")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def email_sending_account(self) -> typing.Optional[builtins.str]:
            '''Specifies whether Amazon Cognito uses its built-in functionality to send your users email messages, or uses your Amazon Simple Email Service email configuration.

            Specify one of the following values:

            - **COGNITO_DEFAULT** - When Amazon Cognito emails your users, it uses its built-in email functionality. When you use the default option, Amazon Cognito allows only a limited number of emails each day for your user pool. For typical production environments, the default email limit is less than the required delivery volume. To achieve a higher delivery volume, specify DEVELOPER to use your Amazon SES email configuration.

            To look up the email delivery limit for the default option, see `Limits <https://docs.aws.amazon.com/cognito/latest/developerguide/limits.html>`_ in the *Amazon Cognito Developer Guide* .

            The default FROM address is ``no-reply@verificationemail.com`` . To customize the FROM address, provide the Amazon Resource Name (ARN) of an Amazon SES verified email address for the ``SourceArn`` parameter.

            - **DEVELOPER** - When Amazon Cognito emails your users, it uses your Amazon SES configuration. Amazon Cognito calls Amazon SES on your behalf to send email from your verified email address. When you use this option, the email delivery limits are the same limits that apply to your Amazon SES verified email address in your AWS account .

            If you use this option, provide the ARN of an Amazon SES verified email address for the ``SourceArn`` parameter.

            Before Amazon Cognito can email your users, it requires additional permissions to call Amazon SES on your behalf. When you update your user pool with this option, Amazon Cognito creates a *service-linked role* , which is a type of role in your AWS account . This role contains the permissions that allow you to access Amazon SES and send email messages from your email address. For more information about the service-linked role that Amazon Cognito creates, see `Using Service-Linked Roles for Amazon Cognito <https://docs.aws.amazon.com/cognito/latest/developerguide/using-service-linked-roles.html>`_ in the *Amazon Cognito Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-emailconfiguration.html#cfn-cognito-userpool-emailconfiguration-emailsendingaccount
            '''
            result = self._values.get("email_sending_account")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def from_(self) -> typing.Optional[builtins.str]:
            '''Either the sender’s email address or the sender’s name with their email address.

            For example, ``testuser@example.com`` or ``Test User <testuser@example.com>`` . This address appears before the body of the email.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-emailconfiguration.html#cfn-cognito-userpool-emailconfiguration-from
            '''
            result = self._values.get("from_")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def reply_to_email_address(self) -> typing.Optional[builtins.str]:
            '''The destination to which the receiver of the email should reply.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-emailconfiguration.html#cfn-cognito-userpool-emailconfiguration-replytoemailaddress
            '''
            result = self._values.get("reply_to_email_address")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def source_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of a verified email address or an address from a verified domain in Amazon SES.

            You can set a ``SourceArn`` email from a verified domain only with an API request. You can set a verified email address, but not an address in a verified domain, in the Amazon Cognito console. Amazon Cognito uses the email address that you provide in one of the following ways, depending on the value that you specify for the ``EmailSendingAccount`` parameter:

            - If you specify ``COGNITO_DEFAULT`` , Amazon Cognito uses this address as the custom FROM address when it emails your users using its built-in email account.
            - If you specify ``DEVELOPER`` , Amazon Cognito emails your users with this address by calling Amazon SES on your behalf.

            The Region value of the ``SourceArn`` parameter must indicate a supported AWS Region of your user pool. Typically, the Region in the ``SourceArn`` and the user pool Region are the same. For more information, see `Amazon SES email configuration regions <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html#user-pool-email-developer-region-mapping>`_ in the `Amazon Cognito Developer Guide <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-emailconfiguration.html#cfn-cognito-userpool-emailconfiguration-sourcearn
            '''
            result = self._values.get("source_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EmailConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.InviteMessageTemplateProperty",
        jsii_struct_bases=[],
        name_mapping={
            "email_message": "emailMessage",
            "email_subject": "emailSubject",
            "sms_message": "smsMessage",
        },
    )
    class InviteMessageTemplateProperty:
        def __init__(
            self,
            *,
            email_message: typing.Optional[builtins.str] = None,
            email_subject: typing.Optional[builtins.str] = None,
            sms_message: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The template for the welcome message to new users.

            This template must include the ``{####}`` temporary password placeholder if you are creating users with passwords. If your users don't have passwords, you can omit the placeholder.

            See also `Customizing User Invitation Messages <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-customizations.html#cognito-user-pool-settings-user-invitation-message-customization>`_ .

            :param email_message: The message template for email messages. EmailMessage is allowed only if `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is DEVELOPER.
            :param email_subject: The subject line for email messages. EmailSubject is allowed only if `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is DEVELOPER.
            :param sms_message: The message template for SMS messages.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-invitemessagetemplate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                invite_message_template_property = cognito.CfnUserPool.InviteMessageTemplateProperty(
                    email_message="emailMessage",
                    email_subject="emailSubject",
                    sms_message="smsMessage"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e181999cda105732a159cee174edaaedc5039f9a73ad613e48d7196c7034bf81)
                check_type(argname="argument email_message", value=email_message, expected_type=type_hints["email_message"])
                check_type(argname="argument email_subject", value=email_subject, expected_type=type_hints["email_subject"])
                check_type(argname="argument sms_message", value=sms_message, expected_type=type_hints["sms_message"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if email_message is not None:
                self._values["email_message"] = email_message
            if email_subject is not None:
                self._values["email_subject"] = email_subject
            if sms_message is not None:
                self._values["sms_message"] = sms_message

        @builtins.property
        def email_message(self) -> typing.Optional[builtins.str]:
            '''The message template for email messages.

            EmailMessage is allowed only if `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is DEVELOPER.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-invitemessagetemplate.html#cfn-cognito-userpool-invitemessagetemplate-emailmessage
            '''
            result = self._values.get("email_message")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def email_subject(self) -> typing.Optional[builtins.str]:
            '''The subject line for email messages.

            EmailSubject is allowed only if `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is DEVELOPER.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-invitemessagetemplate.html#cfn-cognito-userpool-invitemessagetemplate-emailsubject
            '''
            result = self._values.get("email_subject")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def sms_message(self) -> typing.Optional[builtins.str]:
            '''The message template for SMS messages.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-invitemessagetemplate.html#cfn-cognito-userpool-invitemessagetemplate-smsmessage
            '''
            result = self._values.get("sms_message")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InviteMessageTemplateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.LambdaConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "create_auth_challenge": "createAuthChallenge",
            "custom_email_sender": "customEmailSender",
            "custom_message": "customMessage",
            "custom_sms_sender": "customSmsSender",
            "define_auth_challenge": "defineAuthChallenge",
            "kms_key_id": "kmsKeyId",
            "post_authentication": "postAuthentication",
            "post_confirmation": "postConfirmation",
            "pre_authentication": "preAuthentication",
            "pre_sign_up": "preSignUp",
            "pre_token_generation": "preTokenGeneration",
            "pre_token_generation_config": "preTokenGenerationConfig",
            "user_migration": "userMigration",
            "verify_auth_challenge_response": "verifyAuthChallengeResponse",
        },
    )
    class LambdaConfigProperty:
        def __init__(
            self,
            *,
            create_auth_challenge: typing.Optional[builtins.str] = None,
            custom_email_sender: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.CustomEmailSenderProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            custom_message: typing.Optional[builtins.str] = None,
            custom_sms_sender: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.CustomSMSSenderProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            define_auth_challenge: typing.Optional[builtins.str] = None,
            kms_key_id: typing.Optional[builtins.str] = None,
            post_authentication: typing.Optional[builtins.str] = None,
            post_confirmation: typing.Optional[builtins.str] = None,
            pre_authentication: typing.Optional[builtins.str] = None,
            pre_sign_up: typing.Optional[builtins.str] = None,
            pre_token_generation: typing.Optional[builtins.str] = None,
            pre_token_generation_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.PreTokenGenerationConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            user_migration: typing.Optional[builtins.str] = None,
            verify_auth_challenge_response: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A collection of user pool Lambda triggers.

            Amazon Cognito invokes triggers at several possible stages of user pool operations. Triggers can modify the outcome of the operations that invoked them.

            :param create_auth_challenge: The configuration of a create auth challenge Lambda trigger, one of three triggers in the sequence of the `custom authentication challenge triggers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-challenge.html>`_ .
            :param custom_email_sender: The configuration of a custom email sender Lambda trigger. This trigger routes all email notifications from a user pool to a Lambda function that delivers the message using custom logic.
            :param custom_message: A custom message Lambda trigger. This trigger is an opportunity to customize all SMS and email messages from your user pool. When a custom message trigger is active, your user pool routes all messages to a Lambda function that returns a runtime-customized message subject and body for your user pool to deliver to a user.
            :param custom_sms_sender: The configuration of a custom SMS sender Lambda trigger. This trigger routes all SMS notifications from a user pool to a Lambda function that delivers the message using custom logic.
            :param define_auth_challenge: The configuration of a define auth challenge Lambda trigger, one of three triggers in the sequence of the `custom authentication challenge triggers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-challenge.html>`_ .
            :param kms_key_id: The ARN of an `KMS key <https://docs.aws.amazon.com//kms/latest/developerguide/concepts.html#master_keys>`_ . Amazon Cognito uses the key to encrypt codes and temporary passwords sent to custom sender Lambda triggers.
            :param post_authentication: The configuration of a `post authentication Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html>`_ in a user pool. This trigger can take custom actions after a user signs in.
            :param post_confirmation: The configuration of a `post confirmation Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html>`_ in a user pool. This trigger can take custom actions after a user confirms their user account and their email address or phone number.
            :param pre_authentication: The configuration of a `pre authentication trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html>`_ in a user pool. This trigger can evaluate and modify user sign-in events.
            :param pre_sign_up: The configuration of a `pre sign-up Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html>`_ in a user pool. This trigger evaluates new users and can bypass confirmation, `link a federated user profile <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-identity-federation-consolidate-users.html>`_ , or block sign-up requests.
            :param pre_token_generation: The legacy configuration of a `pre token generation Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html>`_ in a user pool. Set this parameter for legacy purposes. If you also set an ARN in ``PreTokenGenerationConfig`` , its value must be identical to ``PreTokenGeneration`` . For new instances of pre token generation triggers, set the ``LambdaArn`` of ``PreTokenGenerationConfig`` .
            :param pre_token_generation_config: The detailed configuration of a `pre token generation Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html>`_ in a user pool. If you also set an ARN in ``PreTokenGeneration`` , its value must be identical to ``PreTokenGenerationConfig`` .
            :param user_migration: The configuration of a `migrate user Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html>`_ in a user pool. This trigger can create user profiles when users sign in or attempt to reset their password with credentials that don't exist yet.
            :param verify_auth_challenge_response: The configuration of a verify auth challenge Lambda trigger, one of three triggers in the sequence of the `custom authentication challenge triggers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-challenge.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                lambda_config_property = cognito.CfnUserPool.LambdaConfigProperty(
                    create_auth_challenge="createAuthChallenge",
                    custom_email_sender=cognito.CfnUserPool.CustomEmailSenderProperty(
                        lambda_arn="lambdaArn",
                        lambda_version="lambdaVersion"
                    ),
                    custom_message="customMessage",
                    custom_sms_sender=cognito.CfnUserPool.CustomSMSSenderProperty(
                        lambda_arn="lambdaArn",
                        lambda_version="lambdaVersion"
                    ),
                    define_auth_challenge="defineAuthChallenge",
                    kms_key_id="kmsKeyId",
                    post_authentication="postAuthentication",
                    post_confirmation="postConfirmation",
                    pre_authentication="preAuthentication",
                    pre_sign_up="preSignUp",
                    pre_token_generation="preTokenGeneration",
                    pre_token_generation_config=cognito.CfnUserPool.PreTokenGenerationConfigProperty(
                        lambda_arn="lambdaArn",
                        lambda_version="lambdaVersion"
                    ),
                    user_migration="userMigration",
                    verify_auth_challenge_response="verifyAuthChallengeResponse"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7c28f17b2edc8643f21e960e612cea0cf88cbb14e55f37d1bab97e6fe5682843)
                check_type(argname="argument create_auth_challenge", value=create_auth_challenge, expected_type=type_hints["create_auth_challenge"])
                check_type(argname="argument custom_email_sender", value=custom_email_sender, expected_type=type_hints["custom_email_sender"])
                check_type(argname="argument custom_message", value=custom_message, expected_type=type_hints["custom_message"])
                check_type(argname="argument custom_sms_sender", value=custom_sms_sender, expected_type=type_hints["custom_sms_sender"])
                check_type(argname="argument define_auth_challenge", value=define_auth_challenge, expected_type=type_hints["define_auth_challenge"])
                check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
                check_type(argname="argument post_authentication", value=post_authentication, expected_type=type_hints["post_authentication"])
                check_type(argname="argument post_confirmation", value=post_confirmation, expected_type=type_hints["post_confirmation"])
                check_type(argname="argument pre_authentication", value=pre_authentication, expected_type=type_hints["pre_authentication"])
                check_type(argname="argument pre_sign_up", value=pre_sign_up, expected_type=type_hints["pre_sign_up"])
                check_type(argname="argument pre_token_generation", value=pre_token_generation, expected_type=type_hints["pre_token_generation"])
                check_type(argname="argument pre_token_generation_config", value=pre_token_generation_config, expected_type=type_hints["pre_token_generation_config"])
                check_type(argname="argument user_migration", value=user_migration, expected_type=type_hints["user_migration"])
                check_type(argname="argument verify_auth_challenge_response", value=verify_auth_challenge_response, expected_type=type_hints["verify_auth_challenge_response"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if create_auth_challenge is not None:
                self._values["create_auth_challenge"] = create_auth_challenge
            if custom_email_sender is not None:
                self._values["custom_email_sender"] = custom_email_sender
            if custom_message is not None:
                self._values["custom_message"] = custom_message
            if custom_sms_sender is not None:
                self._values["custom_sms_sender"] = custom_sms_sender
            if define_auth_challenge is not None:
                self._values["define_auth_challenge"] = define_auth_challenge
            if kms_key_id is not None:
                self._values["kms_key_id"] = kms_key_id
            if post_authentication is not None:
                self._values["post_authentication"] = post_authentication
            if post_confirmation is not None:
                self._values["post_confirmation"] = post_confirmation
            if pre_authentication is not None:
                self._values["pre_authentication"] = pre_authentication
            if pre_sign_up is not None:
                self._values["pre_sign_up"] = pre_sign_up
            if pre_token_generation is not None:
                self._values["pre_token_generation"] = pre_token_generation
            if pre_token_generation_config is not None:
                self._values["pre_token_generation_config"] = pre_token_generation_config
            if user_migration is not None:
                self._values["user_migration"] = user_migration
            if verify_auth_challenge_response is not None:
                self._values["verify_auth_challenge_response"] = verify_auth_challenge_response

        @builtins.property
        def create_auth_challenge(self) -> typing.Optional[builtins.str]:
            '''The configuration of a create auth challenge Lambda trigger, one of three triggers in the sequence of the `custom authentication challenge triggers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-challenge.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-createauthchallenge
            '''
            result = self._values.get("create_auth_challenge")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def custom_email_sender(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.CustomEmailSenderProperty"]]:
            '''The configuration of a custom email sender Lambda trigger.

            This trigger routes all email notifications from a user pool to a Lambda function that delivers the message using custom logic.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-customemailsender
            '''
            result = self._values.get("custom_email_sender")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.CustomEmailSenderProperty"]], result)

        @builtins.property
        def custom_message(self) -> typing.Optional[builtins.str]:
            '''A custom message Lambda trigger.

            This trigger is an opportunity to customize all SMS and email messages from your user pool. When a custom message trigger is active, your user pool routes all messages to a Lambda function that returns a runtime-customized message subject and body for your user pool to deliver to a user.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-custommessage
            '''
            result = self._values.get("custom_message")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def custom_sms_sender(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.CustomSMSSenderProperty"]]:
            '''The configuration of a custom SMS sender Lambda trigger.

            This trigger routes all SMS notifications from a user pool to a Lambda function that delivers the message using custom logic.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-customsmssender
            '''
            result = self._values.get("custom_sms_sender")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.CustomSMSSenderProperty"]], result)

        @builtins.property
        def define_auth_challenge(self) -> typing.Optional[builtins.str]:
            '''The configuration of a define auth challenge Lambda trigger, one of three triggers in the sequence of the `custom authentication challenge triggers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-challenge.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-defineauthchallenge
            '''
            result = self._values.get("define_auth_challenge")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def kms_key_id(self) -> typing.Optional[builtins.str]:
            '''The ARN of an `KMS key <https://docs.aws.amazon.com//kms/latest/developerguide/concepts.html#master_keys>`_ . Amazon Cognito uses the key to encrypt codes and temporary passwords sent to custom sender Lambda triggers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-kmskeyid
            '''
            result = self._values.get("kms_key_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def post_authentication(self) -> typing.Optional[builtins.str]:
            '''The configuration of a `post authentication Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html>`_ in a user pool. This trigger can take custom actions after a user signs in.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-postauthentication
            '''
            result = self._values.get("post_authentication")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def post_confirmation(self) -> typing.Optional[builtins.str]:
            '''The configuration of a `post confirmation Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html>`_ in a user pool. This trigger can take custom actions after a user confirms their user account and their email address or phone number.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-postconfirmation
            '''
            result = self._values.get("post_confirmation")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def pre_authentication(self) -> typing.Optional[builtins.str]:
            '''The configuration of a `pre authentication trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html>`_ in a user pool. This trigger can evaluate and modify user sign-in events.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-preauthentication
            '''
            result = self._values.get("pre_authentication")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def pre_sign_up(self) -> typing.Optional[builtins.str]:
            '''The configuration of a `pre sign-up Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html>`_ in a user pool. This trigger evaluates new users and can bypass confirmation, `link a federated user profile <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-identity-federation-consolidate-users.html>`_ , or block sign-up requests.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-presignup
            '''
            result = self._values.get("pre_sign_up")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def pre_token_generation(self) -> typing.Optional[builtins.str]:
            '''The legacy configuration of a `pre token generation Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html>`_ in a user pool.

            Set this parameter for legacy purposes. If you also set an ARN in ``PreTokenGenerationConfig`` , its value must be identical to ``PreTokenGeneration`` . For new instances of pre token generation triggers, set the ``LambdaArn`` of ``PreTokenGenerationConfig`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-pretokengeneration
            '''
            result = self._values.get("pre_token_generation")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def pre_token_generation_config(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.PreTokenGenerationConfigProperty"]]:
            '''The detailed configuration of a `pre token generation Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html>`_ in a user pool. If you also set an ARN in ``PreTokenGeneration`` , its value must be identical to ``PreTokenGenerationConfig`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-pretokengenerationconfig
            '''
            result = self._values.get("pre_token_generation_config")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.PreTokenGenerationConfigProperty"]], result)

        @builtins.property
        def user_migration(self) -> typing.Optional[builtins.str]:
            '''The configuration of a `migrate user Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html>`_ in a user pool. This trigger can create user profiles when users sign in or attempt to reset their password with credentials that don't exist yet.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-usermigration
            '''
            result = self._values.get("user_migration")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def verify_auth_challenge_response(self) -> typing.Optional[builtins.str]:
            '''The configuration of a verify auth challenge Lambda trigger, one of three triggers in the sequence of the `custom authentication challenge triggers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-challenge.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-lambdaconfig.html#cfn-cognito-userpool-lambdaconfig-verifyauthchallengeresponse
            '''
            result = self._values.get("verify_auth_challenge_response")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LambdaConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.NumberAttributeConstraintsProperty",
        jsii_struct_bases=[],
        name_mapping={"max_value": "maxValue", "min_value": "minValue"},
    )
    class NumberAttributeConstraintsProperty:
        def __init__(
            self,
            *,
            max_value: typing.Optional[builtins.str] = None,
            min_value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The minimum and maximum values of an attribute that is of the number type, for example ``custom:age`` .

            :param max_value: The maximum length of a number attribute value. Must be a number less than or equal to ``2^1023`` , represented as a string with a length of 131072 characters or fewer.
            :param min_value: The minimum value of an attribute that is of the number data type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-numberattributeconstraints.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                number_attribute_constraints_property = cognito.CfnUserPool.NumberAttributeConstraintsProperty(
                    max_value="maxValue",
                    min_value="minValue"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fcd4fcf3db45f8dea604d1f3bcc70f5f37fb4061fc1c279ad00b155ccb10477f)
                check_type(argname="argument max_value", value=max_value, expected_type=type_hints["max_value"])
                check_type(argname="argument min_value", value=min_value, expected_type=type_hints["min_value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max_value is not None:
                self._values["max_value"] = max_value
            if min_value is not None:
                self._values["min_value"] = min_value

        @builtins.property
        def max_value(self) -> typing.Optional[builtins.str]:
            '''The maximum length of a number attribute value.

            Must be a number less than or equal to ``2^1023`` , represented as a string with a length of 131072 characters or fewer.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-numberattributeconstraints.html#cfn-cognito-userpool-numberattributeconstraints-maxvalue
            '''
            result = self._values.get("max_value")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def min_value(self) -> typing.Optional[builtins.str]:
            '''The minimum value of an attribute that is of the number data type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-numberattributeconstraints.html#cfn-cognito-userpool-numberattributeconstraints-minvalue
            '''
            result = self._values.get("min_value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NumberAttributeConstraintsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.PasswordPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "minimum_length": "minimumLength",
            "password_history_size": "passwordHistorySize",
            "require_lowercase": "requireLowercase",
            "require_numbers": "requireNumbers",
            "require_symbols": "requireSymbols",
            "require_uppercase": "requireUppercase",
            "temporary_password_validity_days": "temporaryPasswordValidityDays",
        },
    )
    class PasswordPolicyProperty:
        def __init__(
            self,
            *,
            minimum_length: typing.Optional[jsii.Number] = None,
            password_history_size: typing.Optional[jsii.Number] = None,
            require_lowercase: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            require_numbers: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            require_symbols: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            require_uppercase: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            temporary_password_validity_days: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The password policy settings for a user pool, including complexity, history, and length requirements.

            :param minimum_length: The minimum length of the password in the policy that you have set. This value can't be less than 6.
            :param password_history_size: The number of previous passwords that you want Amazon Cognito to restrict each user from reusing. Users can't set a password that matches any of ``n`` previous passwords, where ``n`` is the value of ``PasswordHistorySize`` .
            :param require_lowercase: The requirement in a password policy that users must include at least one lowercase letter in their password.
            :param require_numbers: The requirement in a password policy that users must include at least one number in their password.
            :param require_symbols: The requirement in a password policy that users must include at least one symbol in their password.
            :param require_uppercase: The requirement in a password policy that users must include at least one uppercase letter in their password.
            :param temporary_password_validity_days: The number of days a temporary password is valid in the password policy. If the user doesn't sign in during this time, an administrator must reset their password. Defaults to ``7`` . If you submit a value of ``0`` , Amazon Cognito treats it as a null value and sets ``TemporaryPasswordValidityDays`` to its default value. .. epigraph:: When you set ``TemporaryPasswordValidityDays`` for a user pool, you can no longer set a value for the legacy ``UnusedAccountValidityDays`` parameter in that user pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-passwordpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                password_policy_property = cognito.CfnUserPool.PasswordPolicyProperty(
                    minimum_length=123,
                    password_history_size=123,
                    require_lowercase=False,
                    require_numbers=False,
                    require_symbols=False,
                    require_uppercase=False,
                    temporary_password_validity_days=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__388245a445a407251a06f0f49f236a6b0a76ff7177f23a1d5cd9d4ffa819b2c0)
                check_type(argname="argument minimum_length", value=minimum_length, expected_type=type_hints["minimum_length"])
                check_type(argname="argument password_history_size", value=password_history_size, expected_type=type_hints["password_history_size"])
                check_type(argname="argument require_lowercase", value=require_lowercase, expected_type=type_hints["require_lowercase"])
                check_type(argname="argument require_numbers", value=require_numbers, expected_type=type_hints["require_numbers"])
                check_type(argname="argument require_symbols", value=require_symbols, expected_type=type_hints["require_symbols"])
                check_type(argname="argument require_uppercase", value=require_uppercase, expected_type=type_hints["require_uppercase"])
                check_type(argname="argument temporary_password_validity_days", value=temporary_password_validity_days, expected_type=type_hints["temporary_password_validity_days"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if minimum_length is not None:
                self._values["minimum_length"] = minimum_length
            if password_history_size is not None:
                self._values["password_history_size"] = password_history_size
            if require_lowercase is not None:
                self._values["require_lowercase"] = require_lowercase
            if require_numbers is not None:
                self._values["require_numbers"] = require_numbers
            if require_symbols is not None:
                self._values["require_symbols"] = require_symbols
            if require_uppercase is not None:
                self._values["require_uppercase"] = require_uppercase
            if temporary_password_validity_days is not None:
                self._values["temporary_password_validity_days"] = temporary_password_validity_days

        @builtins.property
        def minimum_length(self) -> typing.Optional[jsii.Number]:
            '''The minimum length of the password in the policy that you have set.

            This value can't be less than 6.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-passwordpolicy.html#cfn-cognito-userpool-passwordpolicy-minimumlength
            '''
            result = self._values.get("minimum_length")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def password_history_size(self) -> typing.Optional[jsii.Number]:
            '''The number of previous passwords that you want Amazon Cognito to restrict each user from reusing.

            Users can't set a password that matches any of ``n`` previous passwords, where ``n`` is the value of ``PasswordHistorySize`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-passwordpolicy.html#cfn-cognito-userpool-passwordpolicy-passwordhistorysize
            '''
            result = self._values.get("password_history_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def require_lowercase(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''The requirement in a password policy that users must include at least one lowercase letter in their password.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-passwordpolicy.html#cfn-cognito-userpool-passwordpolicy-requirelowercase
            '''
            result = self._values.get("require_lowercase")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def require_numbers(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''The requirement in a password policy that users must include at least one number in their password.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-passwordpolicy.html#cfn-cognito-userpool-passwordpolicy-requirenumbers
            '''
            result = self._values.get("require_numbers")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def require_symbols(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''The requirement in a password policy that users must include at least one symbol in their password.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-passwordpolicy.html#cfn-cognito-userpool-passwordpolicy-requiresymbols
            '''
            result = self._values.get("require_symbols")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def require_uppercase(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''The requirement in a password policy that users must include at least one uppercase letter in their password.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-passwordpolicy.html#cfn-cognito-userpool-passwordpolicy-requireuppercase
            '''
            result = self._values.get("require_uppercase")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def temporary_password_validity_days(self) -> typing.Optional[jsii.Number]:
            '''The number of days a temporary password is valid in the password policy.

            If the user doesn't sign in during this time, an administrator must reset their password. Defaults to ``7`` . If you submit a value of ``0`` , Amazon Cognito treats it as a null value and sets ``TemporaryPasswordValidityDays`` to its default value.
            .. epigraph::

               When you set ``TemporaryPasswordValidityDays`` for a user pool, you can no longer set a value for the legacy ``UnusedAccountValidityDays`` parameter in that user pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-passwordpolicy.html#cfn-cognito-userpool-passwordpolicy-temporarypasswordvaliditydays
            '''
            result = self._values.get("temporary_password_validity_days")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PasswordPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.PoliciesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "password_policy": "passwordPolicy",
            "sign_in_policy": "signInPolicy",
        },
    )
    class PoliciesProperty:
        def __init__(
            self,
            *,
            password_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.PasswordPolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            sign_in_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.SignInPolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''A list of user pool policies.

            Contains the policy that sets password-complexity requirements.

            :param password_policy: The password policy settings for a user pool, including complexity, history, and length requirements.
            :param sign_in_policy: The policy for allowed types of authentication in a user pool. To activate this setting, your user pool must be in the `Essentials tier <https://docs.aws.amazon.com/cognito/latest/developerguide/feature-plans-features-essentials.html>`_ or higher.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-policies.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                policies_property = cognito.CfnUserPool.PoliciesProperty(
                    password_policy=cognito.CfnUserPool.PasswordPolicyProperty(
                        minimum_length=123,
                        password_history_size=123,
                        require_lowercase=False,
                        require_numbers=False,
                        require_symbols=False,
                        require_uppercase=False,
                        temporary_password_validity_days=123
                    ),
                    sign_in_policy=cognito.CfnUserPool.SignInPolicyProperty(
                        allowed_first_auth_factors=["allowedFirstAuthFactors"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9a9937f0b75c9ab1976e5dbd8fe12631390f6d478c894cb0164171b2f9dc39c5)
                check_type(argname="argument password_policy", value=password_policy, expected_type=type_hints["password_policy"])
                check_type(argname="argument sign_in_policy", value=sign_in_policy, expected_type=type_hints["sign_in_policy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if password_policy is not None:
                self._values["password_policy"] = password_policy
            if sign_in_policy is not None:
                self._values["sign_in_policy"] = sign_in_policy

        @builtins.property
        def password_policy(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.PasswordPolicyProperty"]]:
            '''The password policy settings for a user pool, including complexity, history, and length requirements.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-policies.html#cfn-cognito-userpool-policies-passwordpolicy
            '''
            result = self._values.get("password_policy")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.PasswordPolicyProperty"]], result)

        @builtins.property
        def sign_in_policy(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.SignInPolicyProperty"]]:
            '''The policy for allowed types of authentication in a user pool.

            To activate this setting, your user pool must be in the `Essentials tier <https://docs.aws.amazon.com/cognito/latest/developerguide/feature-plans-features-essentials.html>`_ or higher.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-policies.html#cfn-cognito-userpool-policies-signinpolicy
            '''
            result = self._values.get("sign_in_policy")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.SignInPolicyProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PoliciesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.PreTokenGenerationConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"lambda_arn": "lambdaArn", "lambda_version": "lambdaVersion"},
    )
    class PreTokenGenerationConfigProperty:
        def __init__(
            self,
            *,
            lambda_arn: typing.Optional[builtins.str] = None,
            lambda_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The properties of a pre token generation Lambda trigger.

            :param lambda_arn: The Amazon Resource Name (ARN) of the function that you want to assign to your Lambda trigger. This parameter and the ``PreTokenGeneration`` property of ``LambdaConfig`` have the same value. For new instances of pre token generation triggers, set ``LambdaArn`` .
            :param lambda_version: The user pool trigger version of the request that Amazon Cognito sends to your Lambda function. Higher-numbered versions add fields that support new features.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-pretokengenerationconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                pre_token_generation_config_property = cognito.CfnUserPool.PreTokenGenerationConfigProperty(
                    lambda_arn="lambdaArn",
                    lambda_version="lambdaVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c435cb573a2ad1bbb4941ea6a84c25f5ac58315d4c18f57c87ed051fdbe8f389)
                check_type(argname="argument lambda_arn", value=lambda_arn, expected_type=type_hints["lambda_arn"])
                check_type(argname="argument lambda_version", value=lambda_version, expected_type=type_hints["lambda_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if lambda_arn is not None:
                self._values["lambda_arn"] = lambda_arn
            if lambda_version is not None:
                self._values["lambda_version"] = lambda_version

        @builtins.property
        def lambda_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the function that you want to assign to your Lambda trigger.

            This parameter and the ``PreTokenGeneration`` property of ``LambdaConfig`` have the same value. For new instances of pre token generation triggers, set ``LambdaArn`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-pretokengenerationconfig.html#cfn-cognito-userpool-pretokengenerationconfig-lambdaarn
            '''
            result = self._values.get("lambda_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def lambda_version(self) -> typing.Optional[builtins.str]:
            '''The user pool trigger version of the request that Amazon Cognito sends to your Lambda function.

            Higher-numbered versions add fields that support new features.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-pretokengenerationconfig.html#cfn-cognito-userpool-pretokengenerationconfig-lambdaversion
            '''
            result = self._values.get("lambda_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PreTokenGenerationConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.RecoveryOptionProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "priority": "priority"},
    )
    class RecoveryOptionProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            priority: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''A recovery option for a user.

            The ``AccountRecoverySettingType`` data type is an array of this object. Each ``RecoveryOptionType`` has a priority property that determines whether it is a primary or secondary option.

            For example, if ``verified_email`` has a priority of ``1`` and ``verified_phone_number`` has a priority of ``2`` , your user pool sends account-recovery messages to a verified email address but falls back to an SMS message if the user has a verified phone number. The ``admin_only`` option prevents self-service account recovery.

            :param name: The recovery method that this object sets a recovery option for.
            :param priority: Your priority preference for using the specified attribute in account recovery. The highest priority is ``1`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-recoveryoption.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                recovery_option_property = cognito.CfnUserPool.RecoveryOptionProperty(
                    name="name",
                    priority=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c9742a28c55768ed927b752fb7849a5b2cd75a327e7ede06af406eb1c817b9b5)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if priority is not None:
                self._values["priority"] = priority

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The recovery method that this object sets a recovery option for.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-recoveryoption.html#cfn-cognito-userpool-recoveryoption-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''Your priority preference for using the specified attribute in account recovery.

            The highest priority is ``1`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-recoveryoption.html#cfn-cognito-userpool-recoveryoption-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RecoveryOptionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.SchemaAttributeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "attribute_data_type": "attributeDataType",
            "developer_only_attribute": "developerOnlyAttribute",
            "mutable": "mutable",
            "name": "name",
            "number_attribute_constraints": "numberAttributeConstraints",
            "required": "required",
            "string_attribute_constraints": "stringAttributeConstraints",
        },
    )
    class SchemaAttributeProperty:
        def __init__(
            self,
            *,
            attribute_data_type: typing.Optional[builtins.str] = None,
            developer_only_attribute: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            mutable: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            name: typing.Optional[builtins.str] = None,
            number_attribute_constraints: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.NumberAttributeConstraintsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            required: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            string_attribute_constraints: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.StringAttributeConstraintsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''A list of the user attributes and their properties in your user pool.

            The attribute schema contains standard attributes, custom attributes with a ``custom:`` prefix, and developer attributes with a ``dev:`` prefix. For more information, see `User pool attributes <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html>`_ .

            Developer-only ``dev:`` attributes are a legacy feature of user pools, and are read-only to all app clients. You can create and update developer-only attributes only with IAM-authenticated API operations. Use app client read/write permissions instead.

            :param attribute_data_type: The data format of the values for your attribute. When you choose an ``AttributeDataType`` , Amazon Cognito validates the input against the data type. A custom attribute value in your user's ID token is always a string, for example ``"custom:isMember" : "true"`` or ``"custom:YearsAsMember" : "12"`` .
            :param developer_only_attribute: .. epigraph:: You should use `WriteAttributes <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UserPoolClientType.html#CognitoUserPools-Type-UserPoolClientType-WriteAttributes>`_ in the user pool client to control how attributes can be mutated for new use cases instead of using ``DeveloperOnlyAttribute`` . Specifies whether the attribute type is developer only. This attribute can only be modified by an administrator. Users won't be able to modify this attribute using their access token. For example, ``DeveloperOnlyAttribute`` can be modified using AdminUpdateUserAttributes but can't be updated using UpdateUserAttributes.
            :param mutable: Specifies whether the value of the attribute can be changed. Any user pool attribute whose value you map from an IdP attribute must be mutable, with a parameter value of ``true`` . Amazon Cognito updates mapped attributes when users sign in to your application through an IdP. If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute. For more information, see `Specifying Identity Provider Attribute Mappings for Your User Pool <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html>`_ .
            :param name: The name of your user pool attribute. When you create or update a user pool, adding a schema attribute creates a custom or developer-only attribute. When you add an attribute with a ``Name`` value of ``MyAttribute`` , Amazon Cognito creates the custom attribute ``custom:MyAttribute`` . When ``DeveloperOnlyAttribute`` is ``true`` , Amazon Cognito creates your attribute as ``dev:MyAttribute`` . In an operation that describes a user pool, Amazon Cognito returns this value as ``value`` for standard attributes, ``custom:value`` for custom attributes, and ``dev:value`` for developer-only attributes..
            :param number_attribute_constraints: Specifies the constraints for an attribute of the number type.
            :param required: Specifies whether a user pool attribute is required. If the attribute is required and the user doesn't provide a value, registration or sign-in will fail.
            :param string_attribute_constraints: Specifies the constraints for an attribute of the string type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-schemaattribute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                schema_attribute_property = cognito.CfnUserPool.SchemaAttributeProperty(
                    attribute_data_type="attributeDataType",
                    developer_only_attribute=False,
                    mutable=False,
                    name="name",
                    number_attribute_constraints=cognito.CfnUserPool.NumberAttributeConstraintsProperty(
                        max_value="maxValue",
                        min_value="minValue"
                    ),
                    required=False,
                    string_attribute_constraints=cognito.CfnUserPool.StringAttributeConstraintsProperty(
                        max_length="maxLength",
                        min_length="minLength"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9814951786a68c04c05f6bdb7eb01a34fa749e2fb6491b5414b8e8e27d9a7faf)
                check_type(argname="argument attribute_data_type", value=attribute_data_type, expected_type=type_hints["attribute_data_type"])
                check_type(argname="argument developer_only_attribute", value=developer_only_attribute, expected_type=type_hints["developer_only_attribute"])
                check_type(argname="argument mutable", value=mutable, expected_type=type_hints["mutable"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument number_attribute_constraints", value=number_attribute_constraints, expected_type=type_hints["number_attribute_constraints"])
                check_type(argname="argument required", value=required, expected_type=type_hints["required"])
                check_type(argname="argument string_attribute_constraints", value=string_attribute_constraints, expected_type=type_hints["string_attribute_constraints"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if attribute_data_type is not None:
                self._values["attribute_data_type"] = attribute_data_type
            if developer_only_attribute is not None:
                self._values["developer_only_attribute"] = developer_only_attribute
            if mutable is not None:
                self._values["mutable"] = mutable
            if name is not None:
                self._values["name"] = name
            if number_attribute_constraints is not None:
                self._values["number_attribute_constraints"] = number_attribute_constraints
            if required is not None:
                self._values["required"] = required
            if string_attribute_constraints is not None:
                self._values["string_attribute_constraints"] = string_attribute_constraints

        @builtins.property
        def attribute_data_type(self) -> typing.Optional[builtins.str]:
            '''The data format of the values for your attribute.

            When you choose an ``AttributeDataType`` , Amazon Cognito validates the input against the data type. A custom attribute value in your user's ID token is always a string, for example ``"custom:isMember" : "true"`` or ``"custom:YearsAsMember" : "12"`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-schemaattribute.html#cfn-cognito-userpool-schemaattribute-attributedatatype
            '''
            result = self._values.get("attribute_data_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def developer_only_attribute(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''.. epigraph::

   You should use `WriteAttributes <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UserPoolClientType.html#CognitoUserPools-Type-UserPoolClientType-WriteAttributes>`_ in the user pool client to control how attributes can be mutated for new use cases instead of using ``DeveloperOnlyAttribute`` .

            Specifies whether the attribute type is developer only. This attribute can only be modified by an administrator. Users won't be able to modify this attribute using their access token. For example, ``DeveloperOnlyAttribute`` can be modified using AdminUpdateUserAttributes but can't be updated using UpdateUserAttributes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-schemaattribute.html#cfn-cognito-userpool-schemaattribute-developeronlyattribute
            '''
            result = self._values.get("developer_only_attribute")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def mutable(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether the value of the attribute can be changed.

            Any user pool attribute whose value you map from an IdP attribute must be mutable, with a parameter value of ``true`` . Amazon Cognito updates mapped attributes when users sign in to your application through an IdP. If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute. For more information, see `Specifying Identity Provider Attribute Mappings for Your User Pool <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-schemaattribute.html#cfn-cognito-userpool-schemaattribute-mutable
            '''
            result = self._values.get("mutable")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of your user pool attribute.

            When you create or update a user pool, adding a schema attribute creates a custom or developer-only attribute. When you add an attribute with a ``Name`` value of ``MyAttribute`` , Amazon Cognito creates the custom attribute ``custom:MyAttribute`` . When ``DeveloperOnlyAttribute`` is ``true`` , Amazon Cognito creates your attribute as ``dev:MyAttribute`` . In an operation that describes a user pool, Amazon Cognito returns this value as ``value`` for standard attributes, ``custom:value`` for custom attributes, and ``dev:value`` for developer-only attributes..

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-schemaattribute.html#cfn-cognito-userpool-schemaattribute-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def number_attribute_constraints(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.NumberAttributeConstraintsProperty"]]:
            '''Specifies the constraints for an attribute of the number type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-schemaattribute.html#cfn-cognito-userpool-schemaattribute-numberattributeconstraints
            '''
            result = self._values.get("number_attribute_constraints")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.NumberAttributeConstraintsProperty"]], result)

        @builtins.property
        def required(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether a user pool attribute is required.

            If the attribute is required and the user doesn't provide a value, registration or sign-in will fail.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-schemaattribute.html#cfn-cognito-userpool-schemaattribute-required
            '''
            result = self._values.get("required")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def string_attribute_constraints(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.StringAttributeConstraintsProperty"]]:
            '''Specifies the constraints for an attribute of the string type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-schemaattribute.html#cfn-cognito-userpool-schemaattribute-stringattributeconstraints
            '''
            result = self._values.get("string_attribute_constraints")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.StringAttributeConstraintsProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SchemaAttributeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.SignInPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"allowed_first_auth_factors": "allowedFirstAuthFactors"},
    )
    class SignInPolicyProperty:
        def __init__(
            self,
            *,
            allowed_first_auth_factors: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The policy for allowed types of authentication in a user pool.

            To activate this setting, your user pool must be in the `Essentials tier <https://docs.aws.amazon.com/cognito/latest/developerguide/feature-plans-features-essentials.html>`_ or higher.

            :param allowed_first_auth_factors: The sign-in methods that a user pool supports as the first factor. You can permit users to start authentication with a standard username and password, or with other one-time password and hardware factors. Supports values of ``EMAIL_OTP`` , ``SMS_OTP`` , ``WEB_AUTHN`` and ``PASSWORD`` ,

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-signinpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                sign_in_policy_property = cognito.CfnUserPool.SignInPolicyProperty(
                    allowed_first_auth_factors=["allowedFirstAuthFactors"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__71f41ee8011d666621169ad6aeb915855a76a5e105809ce7914229f99c53dd8d)
                check_type(argname="argument allowed_first_auth_factors", value=allowed_first_auth_factors, expected_type=type_hints["allowed_first_auth_factors"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if allowed_first_auth_factors is not None:
                self._values["allowed_first_auth_factors"] = allowed_first_auth_factors

        @builtins.property
        def allowed_first_auth_factors(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''The sign-in methods that a user pool supports as the first factor.

            You can permit users to start authentication with a standard username and password, or with other one-time password and hardware factors.

            Supports values of ``EMAIL_OTP`` , ``SMS_OTP`` , ``WEB_AUTHN`` and ``PASSWORD`` ,

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-signinpolicy.html#cfn-cognito-userpool-signinpolicy-allowedfirstauthfactors
            '''
            result = self._values.get("allowed_first_auth_factors")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SignInPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.SmsConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "external_id": "externalId",
            "sns_caller_arn": "snsCallerArn",
            "sns_region": "snsRegion",
        },
    )
    class SmsConfigurationProperty:
        def __init__(
            self,
            *,
            external_id: typing.Optional[builtins.str] = None,
            sns_caller_arn: typing.Optional[builtins.str] = None,
            sns_region: typing.Optional[builtins.str] = None,
        ) -> None:
            '''User pool configuration for delivery of SMS messages with Amazon Simple Notification Service.

            To send SMS messages with Amazon SNS in the AWS Region that you want, the Amazon Cognito user pool uses an AWS Identity and Access Management (IAM) role in your AWS account .

            :param external_id: The external ID provides additional security for your IAM role. You can use an ``ExternalId`` with the IAM role that you use with Amazon SNS to send SMS messages for your user pool. If you provide an ``ExternalId`` , your Amazon Cognito user pool includes it in the request to assume your IAM role. You can configure the role trust policy to require that Amazon Cognito, and any principal, provide the ``ExternalID`` . If you use the Amazon Cognito Management Console to create a role for SMS multi-factor authentication (MFA), Amazon Cognito creates a role with the required permissions and a trust policy that demonstrates use of the ``ExternalId`` . For more information about the ``ExternalId`` of a role, see `How to use an external ID when granting access to your AWS resources to a third party <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html>`_ .
            :param sns_caller_arn: The Amazon Resource Name (ARN) of the Amazon SNS caller. This is the ARN of the IAM role in your AWS account that Amazon Cognito will use to send SMS messages. SMS messages are subject to a `spending limit <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html>`_ .
            :param sns_region: The AWS Region to use with Amazon SNS integration. You can choose the same Region as your user pool, or a supported *Legacy Amazon SNS alternate Region* . Amazon Cognito resources in the Asia Pacific (Seoul) AWS Region must use your Amazon SNS configuration in the Asia Pacific (Tokyo) Region. For more information, see `SMS message settings for Amazon Cognito user pools <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-sms-settings.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-smsconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                sms_configuration_property = cognito.CfnUserPool.SmsConfigurationProperty(
                    external_id="externalId",
                    sns_caller_arn="snsCallerArn",
                    sns_region="snsRegion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7bdd79abbed6d1c2a56f92beb7e51f5c19f5fdeac49af18d379dda0e31605f6e)
                check_type(argname="argument external_id", value=external_id, expected_type=type_hints["external_id"])
                check_type(argname="argument sns_caller_arn", value=sns_caller_arn, expected_type=type_hints["sns_caller_arn"])
                check_type(argname="argument sns_region", value=sns_region, expected_type=type_hints["sns_region"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if external_id is not None:
                self._values["external_id"] = external_id
            if sns_caller_arn is not None:
                self._values["sns_caller_arn"] = sns_caller_arn
            if sns_region is not None:
                self._values["sns_region"] = sns_region

        @builtins.property
        def external_id(self) -> typing.Optional[builtins.str]:
            '''The external ID provides additional security for your IAM role.

            You can use an ``ExternalId`` with the IAM role that you use with Amazon SNS to send SMS messages for your user pool. If you provide an ``ExternalId`` , your Amazon Cognito user pool includes it in the request to assume your IAM role. You can configure the role trust policy to require that Amazon Cognito, and any principal, provide the ``ExternalID`` . If you use the Amazon Cognito Management Console to create a role for SMS multi-factor authentication (MFA), Amazon Cognito creates a role with the required permissions and a trust policy that demonstrates use of the ``ExternalId`` .

            For more information about the ``ExternalId`` of a role, see `How to use an external ID when granting access to your AWS resources to a third party <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-smsconfiguration.html#cfn-cognito-userpool-smsconfiguration-externalid
            '''
            result = self._values.get("external_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def sns_caller_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the Amazon SNS caller.

            This is the ARN of the IAM role in your AWS account that Amazon Cognito will use to send SMS messages. SMS messages are subject to a `spending limit <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-smsconfiguration.html#cfn-cognito-userpool-smsconfiguration-snscallerarn
            '''
            result = self._values.get("sns_caller_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def sns_region(self) -> typing.Optional[builtins.str]:
            '''The AWS Region to use with Amazon SNS integration.

            You can choose the same Region as your user pool, or a supported *Legacy Amazon SNS alternate Region* .

            Amazon Cognito resources in the Asia Pacific (Seoul) AWS Region must use your Amazon SNS configuration in the Asia Pacific (Tokyo) Region. For more information, see `SMS message settings for Amazon Cognito user pools <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-sms-settings.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-smsconfiguration.html#cfn-cognito-userpool-smsconfiguration-snsregion
            '''
            result = self._values.get("sns_region")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SmsConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.StringAttributeConstraintsProperty",
        jsii_struct_bases=[],
        name_mapping={"max_length": "maxLength", "min_length": "minLength"},
    )
    class StringAttributeConstraintsProperty:
        def __init__(
            self,
            *,
            max_length: typing.Optional[builtins.str] = None,
            min_length: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The minimum and maximum length values of an attribute that is of the string type, for example ``custom:department`` .

            :param max_length: The maximum length of a string attribute value. Must be a number less than or equal to ``2^1023`` , represented as a string with a length of 131072 characters or fewer.
            :param min_length: The minimum length of a string attribute value.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-stringattributeconstraints.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                string_attribute_constraints_property = cognito.CfnUserPool.StringAttributeConstraintsProperty(
                    max_length="maxLength",
                    min_length="minLength"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f41214770e927d123d991e256c57adae04d14e4535a0aba7a27061e61291dabb)
                check_type(argname="argument max_length", value=max_length, expected_type=type_hints["max_length"])
                check_type(argname="argument min_length", value=min_length, expected_type=type_hints["min_length"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max_length is not None:
                self._values["max_length"] = max_length
            if min_length is not None:
                self._values["min_length"] = min_length

        @builtins.property
        def max_length(self) -> typing.Optional[builtins.str]:
            '''The maximum length of a string attribute value.

            Must be a number less than or equal to ``2^1023`` , represented as a string with a length of 131072 characters or fewer.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-stringattributeconstraints.html#cfn-cognito-userpool-stringattributeconstraints-maxlength
            '''
            result = self._values.get("max_length")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def min_length(self) -> typing.Optional[builtins.str]:
            '''The minimum length of a string attribute value.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-stringattributeconstraints.html#cfn-cognito-userpool-stringattributeconstraints-minlength
            '''
            result = self._values.get("min_length")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StringAttributeConstraintsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.UserAttributeUpdateSettingsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "attributes_require_verification_before_update": "attributesRequireVerificationBeforeUpdate",
        },
    )
    class UserAttributeUpdateSettingsProperty:
        def __init__(
            self,
            *,
            attributes_require_verification_before_update: typing.Sequence[builtins.str],
        ) -> None:
            '''The settings for updates to user attributes.

            These settings include the property ``AttributesRequireVerificationBeforeUpdate`` ,
            a user-pool setting that tells Amazon Cognito how to handle changes to the value of your users' email address and phone number attributes. For
            more information, see `Verifying updates to email addresses and phone numbers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html#user-pool-settings-verifications-verify-attribute-updates>`_ .

            :param attributes_require_verification_before_update: Requires that your user verifies their email address, phone number, or both before Amazon Cognito updates the value of that attribute. When you update a user attribute that has this option activated, Amazon Cognito sends a verification message to the new phone number or email address. Amazon Cognito doesn’t change the value of the attribute until your user responds to the verification message and confirms the new value. When ``AttributesRequireVerificationBeforeUpdate`` is false, your user pool doesn't require that your users verify attribute changes before Amazon Cognito updates them. In a user pool where ``AttributesRequireVerificationBeforeUpdate`` is false, API operations that change attribute values can immediately update a user’s ``email`` or ``phone_number`` attribute.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-userattributeupdatesettings.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                user_attribute_update_settings_property = cognito.CfnUserPool.UserAttributeUpdateSettingsProperty(
                    attributes_require_verification_before_update=["attributesRequireVerificationBeforeUpdate"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9c5843901d71c5f7088f0e7c239f7dbd10ae5d4810ba8341e48b29354a6e36e7)
                check_type(argname="argument attributes_require_verification_before_update", value=attributes_require_verification_before_update, expected_type=type_hints["attributes_require_verification_before_update"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "attributes_require_verification_before_update": attributes_require_verification_before_update,
            }

        @builtins.property
        def attributes_require_verification_before_update(
            self,
        ) -> typing.List[builtins.str]:
            '''Requires that your user verifies their email address, phone number, or both before Amazon Cognito updates the value of that attribute.

            When you update a user attribute that has this option activated, Amazon Cognito sends a verification message to the new phone number or email address. Amazon Cognito doesn’t change the value of the attribute until your user responds to the verification message and confirms the new value.

            When ``AttributesRequireVerificationBeforeUpdate`` is false, your user pool doesn't require that your users verify attribute changes before Amazon Cognito updates them. In a user pool where ``AttributesRequireVerificationBeforeUpdate`` is false, API operations that change attribute values can immediately update a user’s ``email`` or ``phone_number`` attribute.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-userattributeupdatesettings.html#cfn-cognito-userpool-userattributeupdatesettings-attributesrequireverificationbeforeupdate
            '''
            result = self._values.get("attributes_require_verification_before_update")
            assert result is not None, "Required property 'attributes_require_verification_before_update' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "UserAttributeUpdateSettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.UserPoolAddOnsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "advanced_security_additional_flows": "advancedSecurityAdditionalFlows",
            "advanced_security_mode": "advancedSecurityMode",
        },
    )
    class UserPoolAddOnsProperty:
        def __init__(
            self,
            *,
            advanced_security_additional_flows: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPool.AdvancedSecurityAdditionalFlowsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            advanced_security_mode: typing.Optional[builtins.str] = None,
        ) -> None:
            '''User pool add-ons.

            Contains settings for activation of threat protection. To log user security information but take no action, set to ``AUDIT`` . To configure automatic security responses to risky traffic to your user pool, set to ``ENFORCED`` .

            For more information, see `Adding advanced security to a user pool <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-advanced-security.html>`_ . To activate this setting, your user pool must be on the `Plus tier <https://docs.aws.amazon.com/cognito/latest/developerguide/feature-plans-features-plus.html>`_ .

            :param advanced_security_additional_flows: Threat protection configuration options for additional authentication types in your user pool, including custom authentication.
            :param advanced_security_mode: The operating mode of threat protection for standard authentication types in your user pool, including username-password and secure remote password (SRP) authentication.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-userpooladdons.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                user_pool_add_ons_property = cognito.CfnUserPool.UserPoolAddOnsProperty(
                    advanced_security_additional_flows=cognito.CfnUserPool.AdvancedSecurityAdditionalFlowsProperty(
                        custom_auth_mode="customAuthMode"
                    ),
                    advanced_security_mode="advancedSecurityMode"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__134d2b8e9ae3077b4c1f940aca55cabaa1dcdc1e12d0cd06c074768ee425ce71)
                check_type(argname="argument advanced_security_additional_flows", value=advanced_security_additional_flows, expected_type=type_hints["advanced_security_additional_flows"])
                check_type(argname="argument advanced_security_mode", value=advanced_security_mode, expected_type=type_hints["advanced_security_mode"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if advanced_security_additional_flows is not None:
                self._values["advanced_security_additional_flows"] = advanced_security_additional_flows
            if advanced_security_mode is not None:
                self._values["advanced_security_mode"] = advanced_security_mode

        @builtins.property
        def advanced_security_additional_flows(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.AdvancedSecurityAdditionalFlowsProperty"]]:
            '''Threat protection configuration options for additional authentication types in your user pool, including custom authentication.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-userpooladdons.html#cfn-cognito-userpool-userpooladdons-advancedsecurityadditionalflows
            '''
            result = self._values.get("advanced_security_additional_flows")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPool.AdvancedSecurityAdditionalFlowsProperty"]], result)

        @builtins.property
        def advanced_security_mode(self) -> typing.Optional[builtins.str]:
            '''The operating mode of threat protection for standard authentication types in your user pool, including username-password and secure remote password (SRP) authentication.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-userpooladdons.html#cfn-cognito-userpool-userpooladdons-advancedsecuritymode
            '''
            result = self._values.get("advanced_security_mode")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "UserPoolAddOnsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.UsernameConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"case_sensitive": "caseSensitive"},
    )
    class UsernameConfigurationProperty:
        def __init__(
            self,
            *,
            case_sensitive: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''Case sensitivity of the username input for the selected sign-in option.

            When case sensitivity is set to ``False`` (case insensitive), users can sign in with any combination of capital and lowercase letters. For example, ``username`` , ``USERNAME`` , or ``UserName`` , or for email, ``email@example.com`` or ``EMaiL@eXamplE.Com`` . For most use cases, set case sensitivity to ``False`` (case insensitive) as a best practice. When usernames and email addresses are case insensitive, Amazon Cognito treats any variation in case as the same user, and prevents a case variation from being assigned to the same attribute for a different user.

            :param case_sensitive: Specifies whether user name case sensitivity will be applied for all users in the user pool through Amazon Cognito APIs. For most use cases, set case sensitivity to ``False`` (case insensitive) as a best practice. When usernames and email addresses are case insensitive, users can sign in as the same user when they enter a different capitalization of their user name. Valid values include: - **true** - Enables case sensitivity for all username input. When this option is set to ``true`` , users must sign in using the exact capitalization of their given username, such as “UserName”. This is the default value. - **false** - Enables case insensitivity for all username input. For example, when this option is set to ``false`` , users can sign in using ``username`` , ``USERNAME`` , or ``UserName`` . This option also enables both ``preferred_username`` and ``email`` alias to be case insensitive, in addition to the ``username`` attribute.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-usernameconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                username_configuration_property = cognito.CfnUserPool.UsernameConfigurationProperty(
                    case_sensitive=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__db6dcc5455914ce6665f0af9e83e637f5f18b38787df8606b4fa8bba2c5f9744)
                check_type(argname="argument case_sensitive", value=case_sensitive, expected_type=type_hints["case_sensitive"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if case_sensitive is not None:
                self._values["case_sensitive"] = case_sensitive

        @builtins.property
        def case_sensitive(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether user name case sensitivity will be applied for all users in the user pool through Amazon Cognito APIs.

            For most use cases, set case sensitivity to ``False`` (case insensitive) as a best practice. When usernames and email addresses are case insensitive, users can sign in as the same user when they enter a different capitalization of their user name.

            Valid values include:

            - **true** - Enables case sensitivity for all username input. When this option is set to ``true`` , users must sign in using the exact capitalization of their given username, such as “UserName”. This is the default value.
            - **false** - Enables case insensitivity for all username input. For example, when this option is set to ``false`` , users can sign in using ``username`` , ``USERNAME`` , or ``UserName`` . This option also enables both ``preferred_username`` and ``email`` alias to be case insensitive, in addition to the ``username`` attribute.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-usernameconfiguration.html#cfn-cognito-userpool-usernameconfiguration-casesensitive
            '''
            result = self._values.get("case_sensitive")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "UsernameConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPool.VerificationMessageTemplateProperty",
        jsii_struct_bases=[],
        name_mapping={
            "default_email_option": "defaultEmailOption",
            "email_message": "emailMessage",
            "email_message_by_link": "emailMessageByLink",
            "email_subject": "emailSubject",
            "email_subject_by_link": "emailSubjectByLink",
            "sms_message": "smsMessage",
        },
    )
    class VerificationMessageTemplateProperty:
        def __init__(
            self,
            *,
            default_email_option: typing.Optional[builtins.str] = None,
            email_message: typing.Optional[builtins.str] = None,
            email_message_by_link: typing.Optional[builtins.str] = None,
            email_subject: typing.Optional[builtins.str] = None,
            email_subject_by_link: typing.Optional[builtins.str] = None,
            sms_message: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The template for the verification message that your user pool delivers to users who set an email address or phone number attribute.

            :param default_email_option: The configuration of verification emails to contain a clickable link or a verification code. For link, your template body must contain link text in the format ``{##Click here##}`` . "Click here" in the example is a customizable string. For code, your template body must contain a code placeholder in the format ``{####}`` .
            :param email_message: The template for email messages that Amazon Cognito sends to your users. You can set an ``EmailMessage`` template only if the value of `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` . When your `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` , your user pool sends email messages with your own Amazon SES configuration.
            :param email_message_by_link: The email message template for sending a confirmation link to the user. You can set an ``EmailMessageByLink`` template only if the value of `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` . When your `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` , your user pool sends email messages with your own Amazon SES configuration.
            :param email_subject: The subject line for the email message template. You can set an ``EmailSubject`` template only if the value of `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` . When your `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` , your user pool sends email messages with your own Amazon SES configuration.
            :param email_subject_by_link: The subject line for the email message template for sending a confirmation link to the user. You can set an ``EmailSubjectByLink`` template only if the value of `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` . When your `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` , your user pool sends email messages with your own Amazon SES configuration.
            :param sms_message: The template for SMS messages that Amazon Cognito sends to your users.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                verification_message_template_property = cognito.CfnUserPool.VerificationMessageTemplateProperty(
                    default_email_option="defaultEmailOption",
                    email_message="emailMessage",
                    email_message_by_link="emailMessageByLink",
                    email_subject="emailSubject",
                    email_subject_by_link="emailSubjectByLink",
                    sms_message="smsMessage"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__264d92574057582e7b8534e4d14b09988810009a53fd30e1df55ccea2a789c67)
                check_type(argname="argument default_email_option", value=default_email_option, expected_type=type_hints["default_email_option"])
                check_type(argname="argument email_message", value=email_message, expected_type=type_hints["email_message"])
                check_type(argname="argument email_message_by_link", value=email_message_by_link, expected_type=type_hints["email_message_by_link"])
                check_type(argname="argument email_subject", value=email_subject, expected_type=type_hints["email_subject"])
                check_type(argname="argument email_subject_by_link", value=email_subject_by_link, expected_type=type_hints["email_subject_by_link"])
                check_type(argname="argument sms_message", value=sms_message, expected_type=type_hints["sms_message"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if default_email_option is not None:
                self._values["default_email_option"] = default_email_option
            if email_message is not None:
                self._values["email_message"] = email_message
            if email_message_by_link is not None:
                self._values["email_message_by_link"] = email_message_by_link
            if email_subject is not None:
                self._values["email_subject"] = email_subject
            if email_subject_by_link is not None:
                self._values["email_subject_by_link"] = email_subject_by_link
            if sms_message is not None:
                self._values["sms_message"] = sms_message

        @builtins.property
        def default_email_option(self) -> typing.Optional[builtins.str]:
            '''The configuration of verification emails to contain a clickable link or a verification code.

            For link, your template body must contain link text in the format ``{##Click here##}`` . "Click here" in the example is a customizable string. For code, your template body must contain a code placeholder in the format ``{####}`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html#cfn-cognito-userpool-verificationmessagetemplate-defaultemailoption
            '''
            result = self._values.get("default_email_option")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def email_message(self) -> typing.Optional[builtins.str]:
            '''The template for email messages that Amazon Cognito sends to your users.

            You can set an ``EmailMessage`` template only if the value of `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` . When your `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` , your user pool sends email messages with your own Amazon SES configuration.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html#cfn-cognito-userpool-verificationmessagetemplate-emailmessage
            '''
            result = self._values.get("email_message")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def email_message_by_link(self) -> typing.Optional[builtins.str]:
            '''The email message template for sending a confirmation link to the user.

            You can set an ``EmailMessageByLink`` template only if the value of `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` . When your `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` , your user pool sends email messages with your own Amazon SES configuration.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html#cfn-cognito-userpool-verificationmessagetemplate-emailmessagebylink
            '''
            result = self._values.get("email_message_by_link")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def email_subject(self) -> typing.Optional[builtins.str]:
            '''The subject line for the email message template.

            You can set an ``EmailSubject`` template only if the value of `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` . When your `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` , your user pool sends email messages with your own Amazon SES configuration.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html#cfn-cognito-userpool-verificationmessagetemplate-emailsubject
            '''
            result = self._values.get("email_subject")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def email_subject_by_link(self) -> typing.Optional[builtins.str]:
            '''The subject line for the email message template for sending a confirmation link to the user.

            You can set an ``EmailSubjectByLink`` template only if the value of `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` . When your `EmailSendingAccount <https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount>`_ is ``DEVELOPER`` , your user pool sends email messages with your own Amazon SES configuration.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html#cfn-cognito-userpool-verificationmessagetemplate-emailsubjectbylink
            '''
            result = self._values.get("email_subject_by_link")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def sms_message(self) -> typing.Optional[builtins.str]:
            '''The template for SMS messages that Amazon Cognito sends to your users.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html#cfn-cognito-userpool-verificationmessagetemplate-smsmessage
            '''
            result = self._values.get("sms_message")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VerificationMessageTemplateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556)
class CfnUserPoolClient(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolClient",
):
    '''The ``AWS::Cognito::UserPoolClient`` resource specifies an Amazon Cognito user pool client.

    .. epigraph::

       If you don't specify a value for a parameter, Amazon Cognito sets it to a default value.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html
    :cloudformationResource: AWS::Cognito::UserPoolClient
    :exampleMetadata: infused

    Example::

        from aws_cdk import aws_certificatemanager as acm
        
        # vpc: ec2.Vpc
        # certificate: acm.Certificate
        
        
        lb = elbv2.ApplicationLoadBalancer(self, "LB",
            vpc=vpc,
            internet_facing=True
        )
        
        user_pool = cognito.UserPool(self, "UserPool")
        user_pool_client = cognito.UserPoolClient(self, "Client",
            user_pool=user_pool,
        
            # Required minimal configuration for use with an ELB
            generate_secret=True,
            auth_flows=cognito.AuthFlow(
                user_password=True
            ),
            o_auth=cognito.OAuthSettings(
                flows=cognito.OAuthFlows(
                    authorization_code_grant=True
                ),
                scopes=[cognito.OAuthScope.EMAIL],
                callback_urls=[f"https://{lb.loadBalancerDnsName}/oauth2/idpresponse"
                ]
            )
        )
        cfn_client = user_pool_client.node.default_child
        cfn_client.add_property_override("RefreshTokenValidity", 1)
        cfn_client.add_property_override("SupportedIdentityProviders", ["COGNITO"])
        
        user_pool_domain = cognito.UserPoolDomain(self, "Domain",
            user_pool=user_pool,
            cognito_domain=cognito.CognitoDomainOptions(
                domain_prefix="test-cdk-prefix"
            )
        )
        
        lb.add_listener("Listener",
            port=443,
            certificates=[certificate],
            default_action=actions.AuthenticateCognitoAction(
                user_pool=user_pool,
                user_pool_client=user_pool_client,
                user_pool_domain=user_pool_domain,
                next=elbv2.ListenerAction.fixed_response(200,
                    content_type="text/plain",
                    message_body="Authenticated"
                )
            )
        )
        
        CfnOutput(self, "DNS",
            value=lb.load_balancer_dns_name
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        user_pool_id: builtins.str,
        access_token_validity: typing.Optional[jsii.Number] = None,
        allowed_o_auth_flows: typing.Optional[typing.Sequence[builtins.str]] = None,
        allowed_o_auth_flows_user_pool_client: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        allowed_o_auth_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        analytics_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolClient.AnalyticsConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        auth_session_validity: typing.Optional[jsii.Number] = None,
        callback_ur_ls: typing.Optional[typing.Sequence[builtins.str]] = None,
        client_name: typing.Optional[builtins.str] = None,
        default_redirect_uri: typing.Optional[builtins.str] = None,
        enable_propagate_additional_user_context_data: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        enable_token_revocation: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        explicit_auth_flows: typing.Optional[typing.Sequence[builtins.str]] = None,
        generate_secret: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        id_token_validity: typing.Optional[jsii.Number] = None,
        logout_ur_ls: typing.Optional[typing.Sequence[builtins.str]] = None,
        prevent_user_existence_errors: typing.Optional[builtins.str] = None,
        read_attributes: typing.Optional[typing.Sequence[builtins.str]] = None,
        refresh_token_validity: typing.Optional[jsii.Number] = None,
        supported_identity_providers: typing.Optional[typing.Sequence[builtins.str]] = None,
        token_validity_units: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolClient.TokenValidityUnitsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        write_attributes: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param user_pool_id: The ID of the user pool where you want to create an app client.
        :param access_token_validity: The access token time limit. After this limit expires, your user can't use their access token. To specify the time unit for ``AccessTokenValidity`` as ``seconds`` , ``minutes`` , ``hours`` , or ``days`` , set a ``TokenValidityUnits`` value in your API request. For example, when you set ``AccessTokenValidity`` to ``10`` and ``TokenValidityUnits`` to ``hours`` , your user can authorize access with their access token for 10 hours. The default time unit for ``AccessTokenValidity`` in an API request is hours. *Valid range* is displayed below in seconds. If you don't specify otherwise in the configuration of your app client, your access tokens are valid for one hour.
        :param allowed_o_auth_flows: The OAuth grant types that you want your app client to generate for clients in managed login authentication. To create an app client that generates client credentials grants, you must add ``client_credentials`` as the only allowed OAuth flow. - **code** - Use a code grant flow, which provides an authorization code as the response. This code can be exchanged for access tokens with the ``/oauth2/token`` endpoint. - **implicit** - Issue the access token, and the ID token when scopes like ``openid`` and ``profile`` are requested, directly to your user. - **client_credentials** - Issue the access token from the ``/oauth2/token`` endpoint directly to a non-person user, authorized by a combination of the client ID and client secret.
        :param allowed_o_auth_flows_user_pool_client: Set to ``true`` to use OAuth 2.0 authorization server features in your app client. This parameter must have a value of ``true`` before you can configure the following features in your app client. - ``CallBackURLs`` : Callback URLs. - ``LogoutURLs`` : Sign-out redirect URLs. - ``AllowedOAuthScopes`` : OAuth 2.0 scopes. - ``AllowedOAuthFlows`` : Support for authorization code, implicit, and client credentials OAuth 2.0 grants. To use authorization server features, configure one of these features in the Amazon Cognito console or set ``AllowedOAuthFlowsUserPoolClient`` to ``true`` in a ``CreateUserPoolClient`` or ``UpdateUserPoolClient`` API request. If you don't set a value for ``AllowedOAuthFlowsUserPoolClient`` in a request with the AWS CLI or SDKs, it defaults to ``false`` . When ``false`` , only SDK-based API sign-in is permitted.
        :param allowed_o_auth_scopes: The OAuth, OpenID Connect (OIDC), and custom scopes that you want to permit your app client to authorize access with. Scopes govern access control to user pool self-service API operations, user data from the ``userInfo`` endpoint, and third-party APIs. Scope values include ``phone`` , ``email`` , ``openid`` , and ``profile`` . The ``aws.cognito.signin.user.admin`` scope authorizes user self-service operations. Custom scopes with resource servers authorize access to external APIs.
        :param analytics_configuration: The user pool analytics configuration for collecting metrics and sending them to your Amazon Pinpoint campaign. In AWS Regions where Amazon Pinpoint isn't available, user pools might not have access to analytics or might be configurable with campaigns in the US East (N. Virginia) Region. For more information, see `Using Amazon Pinpoint analytics <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-pinpoint-integration.html>`_ .
        :param auth_session_validity: Amazon Cognito creates a session token for each API request in an authentication flow. ``AuthSessionValidity`` is the duration, in minutes, of that session token. Your user pool native user must respond to each authentication challenge before the session expires.
        :param callback_ur_ls: A list of allowed redirect, or callback, URLs for managed login authentication. These URLs are the paths where you want to send your users' browsers after they complete authentication with managed login or a third-party IdP. Typically, callback URLs are the home of an application that uses OAuth or OIDC libraries to process authentication outcomes. A redirect URI must meet the following requirements: - Be an absolute URI. - Be registered with the authorization server. Amazon Cognito doesn't accept authorization requests with ``redirect_uri`` values that aren't in the list of ``CallbackURLs`` that you provide in this parameter. - Not include a fragment component. See `OAuth 2.0 - Redirection Endpoint <https://docs.aws.amazon.com/https://tools.ietf.org/html/rfc6749#section-3.1.2>`_ . Amazon Cognito requires HTTPS over HTTP except for http://localhost for testing purposes only. App callback URLs such as myapp://example are also supported.
        :param client_name: A friendly name for the app client that you want to create.
        :param default_redirect_uri: The default redirect URI. In app clients with one assigned IdP, replaces ``redirect_uri`` in authentication requests. Must be in the ``CallbackURLs`` list.
        :param enable_propagate_additional_user_context_data: When ``true`` , your application can include additional ``UserContextData`` in authentication requests. This data includes the IP address, and contributes to analysis by threat protection features. For more information about propagation of user context data, see `Adding session data to API requests <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-adaptive-authentication.html#user-pool-settings-adaptive-authentication-device-fingerprint>`_ . If you don’t include this parameter, you can't send the source IP address to Amazon Cognito threat protection features. You can only activate ``EnablePropagateAdditionalUserContextData`` in an app client that has a client secret.
        :param enable_token_revocation: Activates or deactivates token revocation. If you don't include this parameter, token revocation is automatically activated for the new user pool client.
        :param explicit_auth_flows: The `authentication flows <https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow-methods.html>`_ that you want your user pool client to support. For each app client in your user pool, you can sign in your users with any combination of one or more flows, including with a user name and Secure Remote Password (SRP), a user name and password, or a custom authentication process that you define with Lambda functions. .. epigraph:: If you don't specify a value for ``ExplicitAuthFlows`` , your app client supports ``ALLOW_REFRESH_TOKEN_AUTH`` , ``ALLOW_USER_SRP_AUTH`` , and ``ALLOW_CUSTOM_AUTH`` . The values for authentication flow options include the following. - ``ALLOW_USER_AUTH`` : Enable selection-based sign-in with ``USER_AUTH`` . This setting covers username-password, secure remote password (SRP), passwordless, and passkey authentication. This authentiation flow can do username-password and SRP authentication without other ``ExplicitAuthFlows`` permitting them. For example users can complete an SRP challenge through ``USER_AUTH`` without the flow ``USER_SRP_AUTH`` being active for the app client. This flow doesn't include ``CUSTOM_AUTH`` . To activate this setting, your user pool must be in the `Essentials tier <https://docs.aws.amazon.com/cognito/latest/developerguide/feature-plans-features-essentials.html>`_ or higher. - ``ALLOW_ADMIN_USER_PASSWORD_AUTH`` : Enable admin based user password authentication flow ``ADMIN_USER_PASSWORD_AUTH`` . This setting replaces the ``ADMIN_NO_SRP_AUTH`` setting. With this authentication flow, your app passes a user name and password to Amazon Cognito in the request, instead of using the Secure Remote Password (SRP) protocol to securely transmit the password. - ``ALLOW_CUSTOM_AUTH`` : Enable Lambda trigger based authentication. - ``ALLOW_USER_PASSWORD_AUTH`` : Enable user password-based authentication. In this flow, Amazon Cognito receives the password in the request instead of using the SRP protocol to verify passwords. - ``ALLOW_USER_SRP_AUTH`` : Enable SRP-based authentication. - ``ALLOW_REFRESH_TOKEN_AUTH`` : Enable authflow to refresh tokens. In some environments, you will see the values ``ADMIN_NO_SRP_AUTH`` , ``CUSTOM_AUTH_FLOW_ONLY`` , or ``USER_PASSWORD_AUTH`` . You can't assign these legacy ``ExplicitAuthFlows`` values to user pool clients at the same time as values that begin with ``ALLOW_`` , like ``ALLOW_USER_SRP_AUTH`` .
        :param generate_secret: When ``true`` , generates a client secret for the app client. Client secrets are used with server-side and machine-to-machine applications. Client secrets are automatically generated; you can't specify a secret value. For more information, see `App client types <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-client-apps.html#user-pool-settings-client-app-client-types>`_ .
        :param id_token_validity: The ID token time limit. After this limit expires, your user can't use their ID token. To specify the time unit for ``IdTokenValidity`` as ``seconds`` , ``minutes`` , ``hours`` , or ``days`` , set a ``TokenValidityUnits`` value in your API request. For example, when you set ``IdTokenValidity`` as ``10`` and ``TokenValidityUnits`` as ``hours`` , your user can authenticate their session with their ID token for 10 hours. The default time unit for ``IdTokenValidity`` in an API request is hours. *Valid range* is displayed below in seconds. If you don't specify otherwise in the configuration of your app client, your ID tokens are valid for one hour.
        :param logout_ur_ls: A list of allowed logout URLs for managed login authentication. When you pass ``logout_uri`` and ``client_id`` parameters to ``/logout`` , Amazon Cognito signs out your user and redirects them to the logout URL. This parameter describes the URLs that you want to be the permitted targets of ``logout_uri`` . A typical use of these URLs is when a user selects "Sign out" and you redirect them to your public homepage. For more information, see `Logout endpoint <https://docs.aws.amazon.com/cognito/latest/developerguide/logout-endpoint.html>`_ .
        :param prevent_user_existence_errors: Errors and responses that you want Amazon Cognito APIs to return during authentication, account confirmation, and password recovery when the user doesn't exist in the user pool. When set to ``ENABLED`` and the user doesn't exist, authentication returns an error indicating either the username or password was incorrect. Account confirmation and password recovery return a response indicating a code was sent to a simulated destination. When set to ``LEGACY`` , those APIs return a ``UserNotFoundException`` exception if the user doesn't exist in the user pool. Valid values include: - ``ENABLED`` - This prevents user existence-related errors. - ``LEGACY`` - This represents the early behavior of Amazon Cognito where user existence related errors aren't prevented. Defaults to ``LEGACY`` when you don't provide a value.
        :param read_attributes: The list of user attributes that you want your app client to have read access to. After your user authenticates in your app, their access token authorizes them to read their own attribute value for any attribute in this list. An example of this kind of activity is when your user selects a link to view their profile information. When you don't specify the ``ReadAttributes`` for your app client, your app can read the values of ``email_verified`` , ``phone_number_verified`` , and the Standard attributes of your user pool. When your user pool app client has read access to these default attributes, ``ReadAttributes`` doesn't return any information. Amazon Cognito only populates ``ReadAttributes`` in the API response if you have specified your own custom set of read attributes.
        :param refresh_token_validity: The refresh token time limit. After this limit expires, your user can't use their refresh token. To specify the time unit for ``RefreshTokenValidity`` as ``seconds`` , ``minutes`` , ``hours`` , or ``days`` , set a ``TokenValidityUnits`` value in your API request. For example, when you set ``RefreshTokenValidity`` as ``10`` and ``TokenValidityUnits`` as ``days`` , your user can refresh their session and retrieve new access and ID tokens for 10 days. The default time unit for ``RefreshTokenValidity`` in an API request is days. You can't set ``RefreshTokenValidity`` to 0. If you do, Amazon Cognito overrides the value with the default value of 30 days. *Valid range* is displayed below in seconds. If you don't specify otherwise in the configuration of your app client, your refresh tokens are valid for 30 days.
        :param supported_identity_providers: A list of provider names for the identity providers (IdPs) that are supported on this client. The following are supported: ``COGNITO`` , ``Facebook`` , ``Google`` , ``SignInWithApple`` , and ``LoginWithAmazon`` . You can also specify the names that you configured for the SAML and OIDC IdPs in your user pool, for example ``MySAMLIdP`` or ``MyOIDCIdP`` . This parameter sets the IdPs that `managed login <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-managed-login.html>`_ will display on the login page for your app client. The removal of ``COGNITO`` from this list doesn't prevent authentication operations for local users with the user pools API in an AWS SDK. The only way to prevent SDK-based authentication is to block access with a `AWS WAF rule <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-waf.html>`_ .
        :param token_validity_units: The units that validity times are represented in. The default unit for refresh tokens is days, and the default for ID and access tokens are hours.
        :param write_attributes: The list of user attributes that you want your app client to have write access to. After your user authenticates in your app, their access token authorizes them to set or modify their own attribute value for any attribute in this list. When you don't specify the ``WriteAttributes`` for your app client, your app can write the values of the Standard attributes of your user pool. When your user pool has write access to these default attributes, ``WriteAttributes`` doesn't return any information. Amazon Cognito only populates ``WriteAttributes`` in the API response if you have specified your own custom set of write attributes. If your app client allows users to sign in through an IdP, this array must include all attributes that you have mapped to IdP attributes. Amazon Cognito updates mapped attributes when users sign in to your application through an IdP. If your app client does not have write access to a mapped attribute, Amazon Cognito throws an error when it tries to update the attribute. For more information, see `Specifying IdP Attribute Mappings for Your user pool <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html>`_ .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__87712ca9ae8faf9f73a6c5d11987fcf280543ea093bcc4253c800c0151725828)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUserPoolClientProps(
            user_pool_id=user_pool_id,
            access_token_validity=access_token_validity,
            allowed_o_auth_flows=allowed_o_auth_flows,
            allowed_o_auth_flows_user_pool_client=allowed_o_auth_flows_user_pool_client,
            allowed_o_auth_scopes=allowed_o_auth_scopes,
            analytics_configuration=analytics_configuration,
            auth_session_validity=auth_session_validity,
            callback_ur_ls=callback_ur_ls,
            client_name=client_name,
            default_redirect_uri=default_redirect_uri,
            enable_propagate_additional_user_context_data=enable_propagate_additional_user_context_data,
            enable_token_revocation=enable_token_revocation,
            explicit_auth_flows=explicit_auth_flows,
            generate_secret=generate_secret,
            id_token_validity=id_token_validity,
            logout_ur_ls=logout_ur_ls,
            prevent_user_existence_errors=prevent_user_existence_errors,
            read_attributes=read_attributes,
            refresh_token_validity=refresh_token_validity,
            supported_identity_providers=supported_identity_providers,
            token_validity_units=token_validity_units,
            write_attributes=write_attributes,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a9fb7182377f570d9a0052b4ec94ac913c9394d689704450911f34bf1b5d913)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__24cd5fc10220261da123ffed55929e7f1d18d35b5e7d1850e552963aef79c436)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrClientId")
    def attr_client_id(self) -> builtins.str:
        '''The ID of the app client, for example ``1example23456789`` .

        :cloudformationAttribute: ClientId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrClientId"))

    @builtins.property
    @jsii.member(jsii_name="attrClientSecret")
    def attr_client_secret(self) -> builtins.str:
        '''
        :cloudformationAttribute: ClientSecret
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrClientSecret"))

    @builtins.property
    @jsii.member(jsii_name="attrName")
    def attr_name(self) -> builtins.str:
        '''
        :cloudformationAttribute: Name
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you want to create an app client.'''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @user_pool_id.setter
    def user_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1b25b3f1d7835d1e1c5594b9c3dffd6b2c3b2773f547e55b57c8e0919e93a204)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="accessTokenValidity")
    def access_token_validity(self) -> typing.Optional[jsii.Number]:
        '''The access token time limit.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "accessTokenValidity"))

    @access_token_validity.setter
    def access_token_validity(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9c22b177f80150859dd3fc0ec6040fd86d6718dde2ccbb4e6e2993f4f0179616)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "accessTokenValidity", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="allowedOAuthFlows")
    def allowed_o_auth_flows(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The OAuth grant types that you want your app client to generate for clients in managed login authentication.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "allowedOAuthFlows"))

    @allowed_o_auth_flows.setter
    def allowed_o_auth_flows(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a4e6becf32885d0d881b59d33ec61072e6d7812d751551a739cadf93d3754c73)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allowedOAuthFlows", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="allowedOAuthFlowsUserPoolClient")
    def allowed_o_auth_flows_user_pool_client(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Set to ``true`` to use OAuth 2.0 authorization server features in your app client.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "allowedOAuthFlowsUserPoolClient"))

    @allowed_o_auth_flows_user_pool_client.setter
    def allowed_o_auth_flows_user_pool_client(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__21873ac965a2fe0f5bdf6747b41efd7cf29a1e20f79888c68d664a70a929167f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allowedOAuthFlowsUserPoolClient", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="allowedOAuthScopes")
    def allowed_o_auth_scopes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The OAuth, OpenID Connect (OIDC), and custom scopes that you want to permit your app client to authorize access with.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "allowedOAuthScopes"))

    @allowed_o_auth_scopes.setter
    def allowed_o_auth_scopes(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__96cd6c659670b16fdec9188b6f46c9069df3c2c6e960eb82c804a6eee58c0676)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "allowedOAuthScopes", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="analyticsConfiguration")
    def analytics_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolClient.AnalyticsConfigurationProperty"]]:
        '''The user pool analytics configuration for collecting metrics and sending them to your Amazon Pinpoint campaign.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolClient.AnalyticsConfigurationProperty"]], jsii.get(self, "analyticsConfiguration"))

    @analytics_configuration.setter
    def analytics_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolClient.AnalyticsConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__20f4094b4b15bf8a99ce9e0a2bb559a164606de26b84fc2bb8855aa07e1b624b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "analyticsConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="authSessionValidity")
    def auth_session_validity(self) -> typing.Optional[jsii.Number]:
        '''Amazon Cognito creates a session token for each API request in an authentication flow.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "authSessionValidity"))

    @auth_session_validity.setter
    def auth_session_validity(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c933dc37f47234e60d586123e1b18e9c28a667f37c5d5427aa85b4409376ea07)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authSessionValidity", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="callbackUrLs")
    def callback_ur_ls(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of allowed redirect, or callback, URLs for managed login authentication.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "callbackUrLs"))

    @callback_ur_ls.setter
    def callback_ur_ls(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d014efcb2549aac6515f36258ccaf4ecec07ac09bca5d035df6559947c4ac2c1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "callbackUrLs", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="clientName")
    def client_name(self) -> typing.Optional[builtins.str]:
        '''A friendly name for the app client that you want to create.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "clientName"))

    @client_name.setter
    def client_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0137f7e9c73233d74a2df7b39c8c4bb8d81b22d045b5741333e0b7f4cad603ba)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="defaultRedirectUri")
    def default_redirect_uri(self) -> typing.Optional[builtins.str]:
        '''The default redirect URI.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "defaultRedirectUri"))

    @default_redirect_uri.setter
    def default_redirect_uri(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dfdc028ae25dd39e3553917a9d4209340b6c920f4ae1efe20b0c2c400f79d71b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "defaultRedirectUri", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="enablePropagateAdditionalUserContextData")
    def enable_propagate_additional_user_context_data(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''When ``true`` , your application can include additional ``UserContextData`` in authentication requests.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enablePropagateAdditionalUserContextData"))

    @enable_propagate_additional_user_context_data.setter
    def enable_propagate_additional_user_context_data(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1c6e219444196075c25c38bf5995f93c8a7cec9a53523c8ac01a848afffefd1c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enablePropagateAdditionalUserContextData", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="enableTokenRevocation")
    def enable_token_revocation(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Activates or deactivates token revocation.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enableTokenRevocation"))

    @enable_token_revocation.setter
    def enable_token_revocation(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fb0c7908c7db28417fc0f3140fa5d8ceb244c98f8aa1729c6e2945ed0e1c43cb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enableTokenRevocation", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="explicitAuthFlows")
    def explicit_auth_flows(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The `authentication flows <https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow-methods.html>`_ that you want your user pool client to support. For each app client in your user pool, you can sign in your users with any combination of one or more flows, including with a user name and Secure Remote Password (SRP), a user name and password, or a custom authentication process that you define with Lambda functions.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "explicitAuthFlows"))

    @explicit_auth_flows.setter
    def explicit_auth_flows(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__40565d8b2082bb82a66f2c2881526bbe7e4d5f0f2f08ba6dc2976e98f59f7c8f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "explicitAuthFlows", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="generateSecret")
    def generate_secret(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''When ``true`` , generates a client secret for the app client.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "generateSecret"))

    @generate_secret.setter
    def generate_secret(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__957788b79d759b4d61639c557602d69d56690104c1f0aaa40f50729534e68c16)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "generateSecret", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="idTokenValidity")
    def id_token_validity(self) -> typing.Optional[jsii.Number]:
        '''The ID token time limit.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "idTokenValidity"))

    @id_token_validity.setter
    def id_token_validity(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fc02869bb5973b6100efb5c37bedca5b74e381117419f1bac379398aaffc2d11)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "idTokenValidity", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="logoutUrLs")
    def logout_ur_ls(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of allowed logout URLs for managed login authentication.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "logoutUrLs"))

    @logout_ur_ls.setter
    def logout_ur_ls(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__794ffdcf30a3962d0a3f7e5f05ddfd74a6fef4d201947dd6201c7a686f541188)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "logoutUrLs", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="preventUserExistenceErrors")
    def prevent_user_existence_errors(self) -> typing.Optional[builtins.str]:
        '''Errors and responses that you want Amazon Cognito APIs to return during authentication, account confirmation, and password recovery when the user doesn't exist in the user pool.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "preventUserExistenceErrors"))

    @prevent_user_existence_errors.setter
    def prevent_user_existence_errors(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__69feb00103083800cef9fe6cb134d68a642b9492532f75a271736355bf5da06d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "preventUserExistenceErrors", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="readAttributes")
    def read_attributes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of user attributes that you want your app client to have read access to.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "readAttributes"))

    @read_attributes.setter
    def read_attributes(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d0450b9a1c4091ecc8dc0197c1b8b862535cb1a705df2e839371c83db2f684ba)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "readAttributes", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="refreshTokenValidity")
    def refresh_token_validity(self) -> typing.Optional[jsii.Number]:
        '''The refresh token time limit.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "refreshTokenValidity"))

    @refresh_token_validity.setter
    def refresh_token_validity(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ffd4a00853573a9063ab6bfb4bd488904d41fa8453a02d33cc8b44f6638125c5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "refreshTokenValidity", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="supportedIdentityProviders")
    def supported_identity_providers(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of provider names for the identity providers (IdPs) that are supported on this client.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "supportedIdentityProviders"))

    @supported_identity_providers.setter
    def supported_identity_providers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__97736930ca8413d8b00c7851c37dde3d55fed9ca3778ecbfeb14cf25389e28c8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "supportedIdentityProviders", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tokenValidityUnits")
    def token_validity_units(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolClient.TokenValidityUnitsProperty"]]:
        '''The units that validity times are represented in.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolClient.TokenValidityUnitsProperty"]], jsii.get(self, "tokenValidityUnits"))

    @token_validity_units.setter
    def token_validity_units(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolClient.TokenValidityUnitsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dd54ee51ef07c33acd2cbd8e02b234784fa5a6a1c7879a29066514822e4147fb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tokenValidityUnits", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="writeAttributes")
    def write_attributes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of user attributes that you want your app client to have write access to.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "writeAttributes"))

    @write_attributes.setter
    def write_attributes(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d9ed7bc7a68b64df6709c00059cae06196e10a1a3cf55f0554b70dc90d3f9945)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "writeAttributes", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolClient.AnalyticsConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "application_arn": "applicationArn",
            "application_id": "applicationId",
            "external_id": "externalId",
            "role_arn": "roleArn",
            "user_data_shared": "userDataShared",
        },
    )
    class AnalyticsConfigurationProperty:
        def __init__(
            self,
            *,
            application_arn: typing.Optional[builtins.str] = None,
            application_id: typing.Optional[builtins.str] = None,
            external_id: typing.Optional[builtins.str] = None,
            role_arn: typing.Optional[builtins.str] = None,
            user_data_shared: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''The settings for Amazon Pinpoint analytics configuration.

            With an analytics configuration, your application can collect user-activity metrics for user notifications with a Amazon Pinpoint campaign.

            Amazon Pinpoint isn't available in all AWS Regions. For a list of available Regions, see `Amazon Cognito and Amazon Pinpoint Region availability <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-pinpoint-integration.html#cognito-user-pools-find-region-mappings>`_ .

            :param application_arn: The Amazon Resource Name (ARN) of an Amazon Pinpoint project that you want to connect to your user pool app client. Amazon Cognito publishes events to the Amazon Pinpoint project that ``ApplicationArn`` declares. You can also configure your application to pass an endpoint ID in the ``AnalyticsMetadata`` parameter of sign-in operations. The endpoint ID is information about the destination for push notifications
            :param application_id: Your Amazon Pinpoint project ID.
            :param external_id: The `external ID <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html>`_ of the role that Amazon Cognito assumes to send analytics data to Amazon Pinpoint.
            :param role_arn: The ARN of an AWS Identity and Access Management role that has the permissions required for Amazon Cognito to publish events to Amazon Pinpoint analytics.
            :param user_data_shared: If ``UserDataShared`` is ``true`` , Amazon Cognito includes user data in the events that it publishes to Amazon Pinpoint analytics.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolclient-analyticsconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                analytics_configuration_property = cognito.CfnUserPoolClient.AnalyticsConfigurationProperty(
                    application_arn="applicationArn",
                    application_id="applicationId",
                    external_id="externalId",
                    role_arn="roleArn",
                    user_data_shared=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6d6536363f7e284c6e3822670ff24aa20b4a6cc176e0e696a1702da00bbe4816)
                check_type(argname="argument application_arn", value=application_arn, expected_type=type_hints["application_arn"])
                check_type(argname="argument application_id", value=application_id, expected_type=type_hints["application_id"])
                check_type(argname="argument external_id", value=external_id, expected_type=type_hints["external_id"])
                check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
                check_type(argname="argument user_data_shared", value=user_data_shared, expected_type=type_hints["user_data_shared"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if application_arn is not None:
                self._values["application_arn"] = application_arn
            if application_id is not None:
                self._values["application_id"] = application_id
            if external_id is not None:
                self._values["external_id"] = external_id
            if role_arn is not None:
                self._values["role_arn"] = role_arn
            if user_data_shared is not None:
                self._values["user_data_shared"] = user_data_shared

        @builtins.property
        def application_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of an Amazon Pinpoint project that you want to connect to your user pool app client.

            Amazon Cognito publishes events to the Amazon Pinpoint project that ``ApplicationArn`` declares. You can also configure your application to pass an endpoint ID in the ``AnalyticsMetadata`` parameter of sign-in operations. The endpoint ID is information about the destination for push notifications

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolclient-analyticsconfiguration.html#cfn-cognito-userpoolclient-analyticsconfiguration-applicationarn
            '''
            result = self._values.get("application_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def application_id(self) -> typing.Optional[builtins.str]:
            '''Your Amazon Pinpoint project ID.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolclient-analyticsconfiguration.html#cfn-cognito-userpoolclient-analyticsconfiguration-applicationid
            '''
            result = self._values.get("application_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def external_id(self) -> typing.Optional[builtins.str]:
            '''The `external ID <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html>`_ of the role that Amazon Cognito assumes to send analytics data to Amazon Pinpoint.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolclient-analyticsconfiguration.html#cfn-cognito-userpoolclient-analyticsconfiguration-externalid
            '''
            result = self._values.get("external_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def role_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of an AWS Identity and Access Management role that has the permissions required for Amazon Cognito to publish events to Amazon Pinpoint analytics.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolclient-analyticsconfiguration.html#cfn-cognito-userpoolclient-analyticsconfiguration-rolearn
            '''
            result = self._values.get("role_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def user_data_shared(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''If ``UserDataShared`` is ``true`` , Amazon Cognito includes user data in the events that it publishes to Amazon Pinpoint analytics.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolclient-analyticsconfiguration.html#cfn-cognito-userpoolclient-analyticsconfiguration-userdatashared
            '''
            result = self._values.get("user_data_shared")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AnalyticsConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolClient.TokenValidityUnitsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "access_token": "accessToken",
            "id_token": "idToken",
            "refresh_token": "refreshToken",
        },
    )
    class TokenValidityUnitsProperty:
        def __init__(
            self,
            *,
            access_token: typing.Optional[builtins.str] = None,
            id_token: typing.Optional[builtins.str] = None,
            refresh_token: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The units that validity times are represented in.

            The default unit for refresh tokens is days, and the default for ID and access tokens are hours.

            :param access_token: A time unit for the value that you set in the ``AccessTokenValidity`` parameter. The default ``AccessTokenValidity`` time unit is ``hours`` . ``AccessTokenValidity`` duration can range from five minutes to one day.
            :param id_token: A time unit for the value that you set in the ``IdTokenValidity`` parameter. The default ``IdTokenValidity`` time unit is ``hours`` . ``IdTokenValidity`` duration can range from five minutes to one day.
            :param refresh_token: A time unit for the value that you set in the ``RefreshTokenValidity`` parameter. The default ``RefreshTokenValidity`` time unit is ``days`` . ``RefreshTokenValidity`` duration can range from 60 minutes to 10 years.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolclient-tokenvalidityunits.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                token_validity_units_property = cognito.CfnUserPoolClient.TokenValidityUnitsProperty(
                    access_token="accessToken",
                    id_token="idToken",
                    refresh_token="refreshToken"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7b0535ba992a4a94b92b965971fc09c8b8c0a72362c42aaf98b5ae4cbef4d7e6)
                check_type(argname="argument access_token", value=access_token, expected_type=type_hints["access_token"])
                check_type(argname="argument id_token", value=id_token, expected_type=type_hints["id_token"])
                check_type(argname="argument refresh_token", value=refresh_token, expected_type=type_hints["refresh_token"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if access_token is not None:
                self._values["access_token"] = access_token
            if id_token is not None:
                self._values["id_token"] = id_token
            if refresh_token is not None:
                self._values["refresh_token"] = refresh_token

        @builtins.property
        def access_token(self) -> typing.Optional[builtins.str]:
            '''A time unit for the value that you set in the ``AccessTokenValidity`` parameter.

            The default ``AccessTokenValidity`` time unit is ``hours`` . ``AccessTokenValidity`` duration can range from five minutes to one day.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolclient-tokenvalidityunits.html#cfn-cognito-userpoolclient-tokenvalidityunits-accesstoken
            '''
            result = self._values.get("access_token")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def id_token(self) -> typing.Optional[builtins.str]:
            '''A time unit for the value that you set in the ``IdTokenValidity`` parameter.

            The default ``IdTokenValidity`` time unit is ``hours`` . ``IdTokenValidity`` duration can range from five minutes to one day.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolclient-tokenvalidityunits.html#cfn-cognito-userpoolclient-tokenvalidityunits-idtoken
            '''
            result = self._values.get("id_token")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def refresh_token(self) -> typing.Optional[builtins.str]:
            '''A time unit for the value that you set in the ``RefreshTokenValidity`` parameter.

            The default ``RefreshTokenValidity`` time unit is ``days`` . ``RefreshTokenValidity`` duration can range from 60 minutes to 10 years.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolclient-tokenvalidityunits.html#cfn-cognito-userpoolclient-tokenvalidityunits-refreshtoken
            '''
            result = self._values.get("refresh_token")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TokenValidityUnitsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolClientProps",
    jsii_struct_bases=[],
    name_mapping={
        "user_pool_id": "userPoolId",
        "access_token_validity": "accessTokenValidity",
        "allowed_o_auth_flows": "allowedOAuthFlows",
        "allowed_o_auth_flows_user_pool_client": "allowedOAuthFlowsUserPoolClient",
        "allowed_o_auth_scopes": "allowedOAuthScopes",
        "analytics_configuration": "analyticsConfiguration",
        "auth_session_validity": "authSessionValidity",
        "callback_ur_ls": "callbackUrLs",
        "client_name": "clientName",
        "default_redirect_uri": "defaultRedirectUri",
        "enable_propagate_additional_user_context_data": "enablePropagateAdditionalUserContextData",
        "enable_token_revocation": "enableTokenRevocation",
        "explicit_auth_flows": "explicitAuthFlows",
        "generate_secret": "generateSecret",
        "id_token_validity": "idTokenValidity",
        "logout_ur_ls": "logoutUrLs",
        "prevent_user_existence_errors": "preventUserExistenceErrors",
        "read_attributes": "readAttributes",
        "refresh_token_validity": "refreshTokenValidity",
        "supported_identity_providers": "supportedIdentityProviders",
        "token_validity_units": "tokenValidityUnits",
        "write_attributes": "writeAttributes",
    },
)
class CfnUserPoolClientProps:
    def __init__(
        self,
        *,
        user_pool_id: builtins.str,
        access_token_validity: typing.Optional[jsii.Number] = None,
        allowed_o_auth_flows: typing.Optional[typing.Sequence[builtins.str]] = None,
        allowed_o_auth_flows_user_pool_client: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        allowed_o_auth_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        analytics_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPoolClient.AnalyticsConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        auth_session_validity: typing.Optional[jsii.Number] = None,
        callback_ur_ls: typing.Optional[typing.Sequence[builtins.str]] = None,
        client_name: typing.Optional[builtins.str] = None,
        default_redirect_uri: typing.Optional[builtins.str] = None,
        enable_propagate_additional_user_context_data: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        enable_token_revocation: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        explicit_auth_flows: typing.Optional[typing.Sequence[builtins.str]] = None,
        generate_secret: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        id_token_validity: typing.Optional[jsii.Number] = None,
        logout_ur_ls: typing.Optional[typing.Sequence[builtins.str]] = None,
        prevent_user_existence_errors: typing.Optional[builtins.str] = None,
        read_attributes: typing.Optional[typing.Sequence[builtins.str]] = None,
        refresh_token_validity: typing.Optional[jsii.Number] = None,
        supported_identity_providers: typing.Optional[typing.Sequence[builtins.str]] = None,
        token_validity_units: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPoolClient.TokenValidityUnitsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        write_attributes: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnUserPoolClient``.

        :param user_pool_id: The ID of the user pool where you want to create an app client.
        :param access_token_validity: The access token time limit. After this limit expires, your user can't use their access token. To specify the time unit for ``AccessTokenValidity`` as ``seconds`` , ``minutes`` , ``hours`` , or ``days`` , set a ``TokenValidityUnits`` value in your API request. For example, when you set ``AccessTokenValidity`` to ``10`` and ``TokenValidityUnits`` to ``hours`` , your user can authorize access with their access token for 10 hours. The default time unit for ``AccessTokenValidity`` in an API request is hours. *Valid range* is displayed below in seconds. If you don't specify otherwise in the configuration of your app client, your access tokens are valid for one hour.
        :param allowed_o_auth_flows: The OAuth grant types that you want your app client to generate for clients in managed login authentication. To create an app client that generates client credentials grants, you must add ``client_credentials`` as the only allowed OAuth flow. - **code** - Use a code grant flow, which provides an authorization code as the response. This code can be exchanged for access tokens with the ``/oauth2/token`` endpoint. - **implicit** - Issue the access token, and the ID token when scopes like ``openid`` and ``profile`` are requested, directly to your user. - **client_credentials** - Issue the access token from the ``/oauth2/token`` endpoint directly to a non-person user, authorized by a combination of the client ID and client secret.
        :param allowed_o_auth_flows_user_pool_client: Set to ``true`` to use OAuth 2.0 authorization server features in your app client. This parameter must have a value of ``true`` before you can configure the following features in your app client. - ``CallBackURLs`` : Callback URLs. - ``LogoutURLs`` : Sign-out redirect URLs. - ``AllowedOAuthScopes`` : OAuth 2.0 scopes. - ``AllowedOAuthFlows`` : Support for authorization code, implicit, and client credentials OAuth 2.0 grants. To use authorization server features, configure one of these features in the Amazon Cognito console or set ``AllowedOAuthFlowsUserPoolClient`` to ``true`` in a ``CreateUserPoolClient`` or ``UpdateUserPoolClient`` API request. If you don't set a value for ``AllowedOAuthFlowsUserPoolClient`` in a request with the AWS CLI or SDKs, it defaults to ``false`` . When ``false`` , only SDK-based API sign-in is permitted.
        :param allowed_o_auth_scopes: The OAuth, OpenID Connect (OIDC), and custom scopes that you want to permit your app client to authorize access with. Scopes govern access control to user pool self-service API operations, user data from the ``userInfo`` endpoint, and third-party APIs. Scope values include ``phone`` , ``email`` , ``openid`` , and ``profile`` . The ``aws.cognito.signin.user.admin`` scope authorizes user self-service operations. Custom scopes with resource servers authorize access to external APIs.
        :param analytics_configuration: The user pool analytics configuration for collecting metrics and sending them to your Amazon Pinpoint campaign. In AWS Regions where Amazon Pinpoint isn't available, user pools might not have access to analytics or might be configurable with campaigns in the US East (N. Virginia) Region. For more information, see `Using Amazon Pinpoint analytics <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-pinpoint-integration.html>`_ .
        :param auth_session_validity: Amazon Cognito creates a session token for each API request in an authentication flow. ``AuthSessionValidity`` is the duration, in minutes, of that session token. Your user pool native user must respond to each authentication challenge before the session expires.
        :param callback_ur_ls: A list of allowed redirect, or callback, URLs for managed login authentication. These URLs are the paths where you want to send your users' browsers after they complete authentication with managed login or a third-party IdP. Typically, callback URLs are the home of an application that uses OAuth or OIDC libraries to process authentication outcomes. A redirect URI must meet the following requirements: - Be an absolute URI. - Be registered with the authorization server. Amazon Cognito doesn't accept authorization requests with ``redirect_uri`` values that aren't in the list of ``CallbackURLs`` that you provide in this parameter. - Not include a fragment component. See `OAuth 2.0 - Redirection Endpoint <https://docs.aws.amazon.com/https://tools.ietf.org/html/rfc6749#section-3.1.2>`_ . Amazon Cognito requires HTTPS over HTTP except for http://localhost for testing purposes only. App callback URLs such as myapp://example are also supported.
        :param client_name: A friendly name for the app client that you want to create.
        :param default_redirect_uri: The default redirect URI. In app clients with one assigned IdP, replaces ``redirect_uri`` in authentication requests. Must be in the ``CallbackURLs`` list.
        :param enable_propagate_additional_user_context_data: When ``true`` , your application can include additional ``UserContextData`` in authentication requests. This data includes the IP address, and contributes to analysis by threat protection features. For more information about propagation of user context data, see `Adding session data to API requests <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-adaptive-authentication.html#user-pool-settings-adaptive-authentication-device-fingerprint>`_ . If you don’t include this parameter, you can't send the source IP address to Amazon Cognito threat protection features. You can only activate ``EnablePropagateAdditionalUserContextData`` in an app client that has a client secret.
        :param enable_token_revocation: Activates or deactivates token revocation. If you don't include this parameter, token revocation is automatically activated for the new user pool client.
        :param explicit_auth_flows: The `authentication flows <https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow-methods.html>`_ that you want your user pool client to support. For each app client in your user pool, you can sign in your users with any combination of one or more flows, including with a user name and Secure Remote Password (SRP), a user name and password, or a custom authentication process that you define with Lambda functions. .. epigraph:: If you don't specify a value for ``ExplicitAuthFlows`` , your app client supports ``ALLOW_REFRESH_TOKEN_AUTH`` , ``ALLOW_USER_SRP_AUTH`` , and ``ALLOW_CUSTOM_AUTH`` . The values for authentication flow options include the following. - ``ALLOW_USER_AUTH`` : Enable selection-based sign-in with ``USER_AUTH`` . This setting covers username-password, secure remote password (SRP), passwordless, and passkey authentication. This authentiation flow can do username-password and SRP authentication without other ``ExplicitAuthFlows`` permitting them. For example users can complete an SRP challenge through ``USER_AUTH`` without the flow ``USER_SRP_AUTH`` being active for the app client. This flow doesn't include ``CUSTOM_AUTH`` . To activate this setting, your user pool must be in the `Essentials tier <https://docs.aws.amazon.com/cognito/latest/developerguide/feature-plans-features-essentials.html>`_ or higher. - ``ALLOW_ADMIN_USER_PASSWORD_AUTH`` : Enable admin based user password authentication flow ``ADMIN_USER_PASSWORD_AUTH`` . This setting replaces the ``ADMIN_NO_SRP_AUTH`` setting. With this authentication flow, your app passes a user name and password to Amazon Cognito in the request, instead of using the Secure Remote Password (SRP) protocol to securely transmit the password. - ``ALLOW_CUSTOM_AUTH`` : Enable Lambda trigger based authentication. - ``ALLOW_USER_PASSWORD_AUTH`` : Enable user password-based authentication. In this flow, Amazon Cognito receives the password in the request instead of using the SRP protocol to verify passwords. - ``ALLOW_USER_SRP_AUTH`` : Enable SRP-based authentication. - ``ALLOW_REFRESH_TOKEN_AUTH`` : Enable authflow to refresh tokens. In some environments, you will see the values ``ADMIN_NO_SRP_AUTH`` , ``CUSTOM_AUTH_FLOW_ONLY`` , or ``USER_PASSWORD_AUTH`` . You can't assign these legacy ``ExplicitAuthFlows`` values to user pool clients at the same time as values that begin with ``ALLOW_`` , like ``ALLOW_USER_SRP_AUTH`` .
        :param generate_secret: When ``true`` , generates a client secret for the app client. Client secrets are used with server-side and machine-to-machine applications. Client secrets are automatically generated; you can't specify a secret value. For more information, see `App client types <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-client-apps.html#user-pool-settings-client-app-client-types>`_ .
        :param id_token_validity: The ID token time limit. After this limit expires, your user can't use their ID token. To specify the time unit for ``IdTokenValidity`` as ``seconds`` , ``minutes`` , ``hours`` , or ``days`` , set a ``TokenValidityUnits`` value in your API request. For example, when you set ``IdTokenValidity`` as ``10`` and ``TokenValidityUnits`` as ``hours`` , your user can authenticate their session with their ID token for 10 hours. The default time unit for ``IdTokenValidity`` in an API request is hours. *Valid range* is displayed below in seconds. If you don't specify otherwise in the configuration of your app client, your ID tokens are valid for one hour.
        :param logout_ur_ls: A list of allowed logout URLs for managed login authentication. When you pass ``logout_uri`` and ``client_id`` parameters to ``/logout`` , Amazon Cognito signs out your user and redirects them to the logout URL. This parameter describes the URLs that you want to be the permitted targets of ``logout_uri`` . A typical use of these URLs is when a user selects "Sign out" and you redirect them to your public homepage. For more information, see `Logout endpoint <https://docs.aws.amazon.com/cognito/latest/developerguide/logout-endpoint.html>`_ .
        :param prevent_user_existence_errors: Errors and responses that you want Amazon Cognito APIs to return during authentication, account confirmation, and password recovery when the user doesn't exist in the user pool. When set to ``ENABLED`` and the user doesn't exist, authentication returns an error indicating either the username or password was incorrect. Account confirmation and password recovery return a response indicating a code was sent to a simulated destination. When set to ``LEGACY`` , those APIs return a ``UserNotFoundException`` exception if the user doesn't exist in the user pool. Valid values include: - ``ENABLED`` - This prevents user existence-related errors. - ``LEGACY`` - This represents the early behavior of Amazon Cognito where user existence related errors aren't prevented. Defaults to ``LEGACY`` when you don't provide a value.
        :param read_attributes: The list of user attributes that you want your app client to have read access to. After your user authenticates in your app, their access token authorizes them to read their own attribute value for any attribute in this list. An example of this kind of activity is when your user selects a link to view their profile information. When you don't specify the ``ReadAttributes`` for your app client, your app can read the values of ``email_verified`` , ``phone_number_verified`` , and the Standard attributes of your user pool. When your user pool app client has read access to these default attributes, ``ReadAttributes`` doesn't return any information. Amazon Cognito only populates ``ReadAttributes`` in the API response if you have specified your own custom set of read attributes.
        :param refresh_token_validity: The refresh token time limit. After this limit expires, your user can't use their refresh token. To specify the time unit for ``RefreshTokenValidity`` as ``seconds`` , ``minutes`` , ``hours`` , or ``days`` , set a ``TokenValidityUnits`` value in your API request. For example, when you set ``RefreshTokenValidity`` as ``10`` and ``TokenValidityUnits`` as ``days`` , your user can refresh their session and retrieve new access and ID tokens for 10 days. The default time unit for ``RefreshTokenValidity`` in an API request is days. You can't set ``RefreshTokenValidity`` to 0. If you do, Amazon Cognito overrides the value with the default value of 30 days. *Valid range* is displayed below in seconds. If you don't specify otherwise in the configuration of your app client, your refresh tokens are valid for 30 days.
        :param supported_identity_providers: A list of provider names for the identity providers (IdPs) that are supported on this client. The following are supported: ``COGNITO`` , ``Facebook`` , ``Google`` , ``SignInWithApple`` , and ``LoginWithAmazon`` . You can also specify the names that you configured for the SAML and OIDC IdPs in your user pool, for example ``MySAMLIdP`` or ``MyOIDCIdP`` . This parameter sets the IdPs that `managed login <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-managed-login.html>`_ will display on the login page for your app client. The removal of ``COGNITO`` from this list doesn't prevent authentication operations for local users with the user pools API in an AWS SDK. The only way to prevent SDK-based authentication is to block access with a `AWS WAF rule <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-waf.html>`_ .
        :param token_validity_units: The units that validity times are represented in. The default unit for refresh tokens is days, and the default for ID and access tokens are hours.
        :param write_attributes: The list of user attributes that you want your app client to have write access to. After your user authenticates in your app, their access token authorizes them to set or modify their own attribute value for any attribute in this list. When you don't specify the ``WriteAttributes`` for your app client, your app can write the values of the Standard attributes of your user pool. When your user pool has write access to these default attributes, ``WriteAttributes`` doesn't return any information. Amazon Cognito only populates ``WriteAttributes`` in the API response if you have specified your own custom set of write attributes. If your app client allows users to sign in through an IdP, this array must include all attributes that you have mapped to IdP attributes. Amazon Cognito updates mapped attributes when users sign in to your application through an IdP. If your app client does not have write access to a mapped attribute, Amazon Cognito throws an error when it tries to update the attribute. For more information, see `Specifying IdP Attribute Mappings for Your user pool <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            cfn_user_pool_client_props = cognito.CfnUserPoolClientProps(
                user_pool_id="userPoolId",
            
                # the properties below are optional
                access_token_validity=123,
                allowed_oAuth_flows=["allowedOAuthFlows"],
                allowed_oAuth_flows_user_pool_client=False,
                allowed_oAuth_scopes=["allowedOAuthScopes"],
                analytics_configuration=cognito.CfnUserPoolClient.AnalyticsConfigurationProperty(
                    application_arn="applicationArn",
                    application_id="applicationId",
                    external_id="externalId",
                    role_arn="roleArn",
                    user_data_shared=False
                ),
                auth_session_validity=123,
                callback_ur_ls=["callbackUrLs"],
                client_name="clientName",
                default_redirect_uri="defaultRedirectUri",
                enable_propagate_additional_user_context_data=False,
                enable_token_revocation=False,
                explicit_auth_flows=["explicitAuthFlows"],
                generate_secret=False,
                id_token_validity=123,
                logout_ur_ls=["logoutUrLs"],
                prevent_user_existence_errors="preventUserExistenceErrors",
                read_attributes=["readAttributes"],
                refresh_token_validity=123,
                supported_identity_providers=["supportedIdentityProviders"],
                token_validity_units=cognito.CfnUserPoolClient.TokenValidityUnitsProperty(
                    access_token="accessToken",
                    id_token="idToken",
                    refresh_token="refreshToken"
                ),
                write_attributes=["writeAttributes"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__073ea5893b9cfc70c0362e57cd0c1b5397e1c6374434fc1d5c261da79307d954)
            check_type(argname="argument user_pool_id", value=user_pool_id, expected_type=type_hints["user_pool_id"])
            check_type(argname="argument access_token_validity", value=access_token_validity, expected_type=type_hints["access_token_validity"])
            check_type(argname="argument allowed_o_auth_flows", value=allowed_o_auth_flows, expected_type=type_hints["allowed_o_auth_flows"])
            check_type(argname="argument allowed_o_auth_flows_user_pool_client", value=allowed_o_auth_flows_user_pool_client, expected_type=type_hints["allowed_o_auth_flows_user_pool_client"])
            check_type(argname="argument allowed_o_auth_scopes", value=allowed_o_auth_scopes, expected_type=type_hints["allowed_o_auth_scopes"])
            check_type(argname="argument analytics_configuration", value=analytics_configuration, expected_type=type_hints["analytics_configuration"])
            check_type(argname="argument auth_session_validity", value=auth_session_validity, expected_type=type_hints["auth_session_validity"])
            check_type(argname="argument callback_ur_ls", value=callback_ur_ls, expected_type=type_hints["callback_ur_ls"])
            check_type(argname="argument client_name", value=client_name, expected_type=type_hints["client_name"])
            check_type(argname="argument default_redirect_uri", value=default_redirect_uri, expected_type=type_hints["default_redirect_uri"])
            check_type(argname="argument enable_propagate_additional_user_context_data", value=enable_propagate_additional_user_context_data, expected_type=type_hints["enable_propagate_additional_user_context_data"])
            check_type(argname="argument enable_token_revocation", value=enable_token_revocation, expected_type=type_hints["enable_token_revocation"])
            check_type(argname="argument explicit_auth_flows", value=explicit_auth_flows, expected_type=type_hints["explicit_auth_flows"])
            check_type(argname="argument generate_secret", value=generate_secret, expected_type=type_hints["generate_secret"])
            check_type(argname="argument id_token_validity", value=id_token_validity, expected_type=type_hints["id_token_validity"])
            check_type(argname="argument logout_ur_ls", value=logout_ur_ls, expected_type=type_hints["logout_ur_ls"])
            check_type(argname="argument prevent_user_existence_errors", value=prevent_user_existence_errors, expected_type=type_hints["prevent_user_existence_errors"])
            check_type(argname="argument read_attributes", value=read_attributes, expected_type=type_hints["read_attributes"])
            check_type(argname="argument refresh_token_validity", value=refresh_token_validity, expected_type=type_hints["refresh_token_validity"])
            check_type(argname="argument supported_identity_providers", value=supported_identity_providers, expected_type=type_hints["supported_identity_providers"])
            check_type(argname="argument token_validity_units", value=token_validity_units, expected_type=type_hints["token_validity_units"])
            check_type(argname="argument write_attributes", value=write_attributes, expected_type=type_hints["write_attributes"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "user_pool_id": user_pool_id,
        }
        if access_token_validity is not None:
            self._values["access_token_validity"] = access_token_validity
        if allowed_o_auth_flows is not None:
            self._values["allowed_o_auth_flows"] = allowed_o_auth_flows
        if allowed_o_auth_flows_user_pool_client is not None:
            self._values["allowed_o_auth_flows_user_pool_client"] = allowed_o_auth_flows_user_pool_client
        if allowed_o_auth_scopes is not None:
            self._values["allowed_o_auth_scopes"] = allowed_o_auth_scopes
        if analytics_configuration is not None:
            self._values["analytics_configuration"] = analytics_configuration
        if auth_session_validity is not None:
            self._values["auth_session_validity"] = auth_session_validity
        if callback_ur_ls is not None:
            self._values["callback_ur_ls"] = callback_ur_ls
        if client_name is not None:
            self._values["client_name"] = client_name
        if default_redirect_uri is not None:
            self._values["default_redirect_uri"] = default_redirect_uri
        if enable_propagate_additional_user_context_data is not None:
            self._values["enable_propagate_additional_user_context_data"] = enable_propagate_additional_user_context_data
        if enable_token_revocation is not None:
            self._values["enable_token_revocation"] = enable_token_revocation
        if explicit_auth_flows is not None:
            self._values["explicit_auth_flows"] = explicit_auth_flows
        if generate_secret is not None:
            self._values["generate_secret"] = generate_secret
        if id_token_validity is not None:
            self._values["id_token_validity"] = id_token_validity
        if logout_ur_ls is not None:
            self._values["logout_ur_ls"] = logout_ur_ls
        if prevent_user_existence_errors is not None:
            self._values["prevent_user_existence_errors"] = prevent_user_existence_errors
        if read_attributes is not None:
            self._values["read_attributes"] = read_attributes
        if refresh_token_validity is not None:
            self._values["refresh_token_validity"] = refresh_token_validity
        if supported_identity_providers is not None:
            self._values["supported_identity_providers"] = supported_identity_providers
        if token_validity_units is not None:
            self._values["token_validity_units"] = token_validity_units
        if write_attributes is not None:
            self._values["write_attributes"] = write_attributes

    @builtins.property
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you want to create an app client.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-userpoolid
        '''
        result = self._values.get("user_pool_id")
        assert result is not None, "Required property 'user_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_token_validity(self) -> typing.Optional[jsii.Number]:
        '''The access token time limit.

        After this limit expires, your user can't use their access token. To specify the time unit for ``AccessTokenValidity`` as ``seconds`` , ``minutes`` , ``hours`` , or ``days`` , set a ``TokenValidityUnits`` value in your API request.

        For example, when you set ``AccessTokenValidity`` to ``10`` and ``TokenValidityUnits`` to ``hours`` , your user can authorize access with
        their access token for 10 hours.

        The default time unit for ``AccessTokenValidity`` in an API request is hours. *Valid range* is displayed below in seconds.

        If you don't specify otherwise in the configuration of your app client, your access
        tokens are valid for one hour.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-accesstokenvalidity
        '''
        result = self._values.get("access_token_validity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def allowed_o_auth_flows(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The OAuth grant types that you want your app client to generate for clients in managed login authentication.

        To create an app client that generates client credentials grants, you must add ``client_credentials`` as the only allowed OAuth flow.

        - **code** - Use a code grant flow, which provides an authorization code as the response. This code can be exchanged for access tokens with the ``/oauth2/token`` endpoint.
        - **implicit** - Issue the access token, and the ID token when scopes like ``openid`` and ``profile`` are requested, directly to your user.
        - **client_credentials** - Issue the access token from the ``/oauth2/token`` endpoint directly to a non-person user, authorized by a combination of the client ID and client secret.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-allowedoauthflows
        '''
        result = self._values.get("allowed_o_auth_flows")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def allowed_o_auth_flows_user_pool_client(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Set to ``true`` to use OAuth 2.0 authorization server features in your app client.

        This parameter must have a value of ``true`` before you can configure the following features in your app client.

        - ``CallBackURLs`` : Callback URLs.
        - ``LogoutURLs`` : Sign-out redirect URLs.
        - ``AllowedOAuthScopes`` : OAuth 2.0 scopes.
        - ``AllowedOAuthFlows`` : Support for authorization code, implicit, and client credentials OAuth 2.0 grants.

        To use authorization server features, configure one of these features in the Amazon Cognito console or set ``AllowedOAuthFlowsUserPoolClient`` to ``true`` in a ``CreateUserPoolClient`` or ``UpdateUserPoolClient`` API request. If you don't set a value for ``AllowedOAuthFlowsUserPoolClient`` in a request with the AWS CLI or SDKs, it defaults to ``false`` . When ``false`` , only SDK-based API sign-in is permitted.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-allowedoauthflowsuserpoolclient
        '''
        result = self._values.get("allowed_o_auth_flows_user_pool_client")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def allowed_o_auth_scopes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The OAuth, OpenID Connect (OIDC), and custom scopes that you want to permit your app client to authorize access with.

        Scopes govern access control to user pool self-service API operations, user data from the ``userInfo`` endpoint, and third-party APIs. Scope values include ``phone`` , ``email`` , ``openid`` , and ``profile`` . The ``aws.cognito.signin.user.admin`` scope authorizes user self-service operations. Custom scopes with resource servers authorize access to external APIs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-allowedoauthscopes
        '''
        result = self._values.get("allowed_o_auth_scopes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def analytics_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolClient.AnalyticsConfigurationProperty]]:
        '''The user pool analytics configuration for collecting metrics and sending them to your Amazon Pinpoint campaign.

        In AWS Regions where Amazon Pinpoint isn't available, user pools might not have access to analytics or might be configurable with campaigns in the US East (N. Virginia) Region. For more information, see `Using Amazon Pinpoint analytics <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-pinpoint-integration.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-analyticsconfiguration
        '''
        result = self._values.get("analytics_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolClient.AnalyticsConfigurationProperty]], result)

    @builtins.property
    def auth_session_validity(self) -> typing.Optional[jsii.Number]:
        '''Amazon Cognito creates a session token for each API request in an authentication flow.

        ``AuthSessionValidity`` is the duration, in minutes, of that session token. Your user pool native user must respond to each authentication challenge before the session expires.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-authsessionvalidity
        '''
        result = self._values.get("auth_session_validity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def callback_ur_ls(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of allowed redirect, or callback, URLs for managed login authentication.

        These URLs are the paths where you want to send your users' browsers after they complete authentication with managed login or a third-party IdP. Typically, callback URLs are the home of an application that uses OAuth or OIDC libraries to process authentication outcomes.

        A redirect URI must meet the following requirements:

        - Be an absolute URI.
        - Be registered with the authorization server. Amazon Cognito doesn't accept authorization requests with ``redirect_uri`` values that aren't in the list of ``CallbackURLs`` that you provide in this parameter.
        - Not include a fragment component.

        See `OAuth 2.0 - Redirection Endpoint <https://docs.aws.amazon.com/https://tools.ietf.org/html/rfc6749#section-3.1.2>`_ .

        Amazon Cognito requires HTTPS over HTTP except for http://localhost for testing purposes only.

        App callback URLs such as myapp://example are also supported.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-callbackurls
        '''
        result = self._values.get("callback_ur_ls")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def client_name(self) -> typing.Optional[builtins.str]:
        '''A friendly name for the app client that you want to create.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-clientname
        '''
        result = self._values.get("client_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def default_redirect_uri(self) -> typing.Optional[builtins.str]:
        '''The default redirect URI.

        In app clients with one assigned IdP, replaces ``redirect_uri`` in authentication requests. Must be in the ``CallbackURLs`` list.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-defaultredirecturi
        '''
        result = self._values.get("default_redirect_uri")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enable_propagate_additional_user_context_data(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''When ``true`` , your application can include additional ``UserContextData`` in authentication requests.

        This data includes the IP address, and contributes to analysis by threat protection features. For more information about propagation of user context data, see `Adding session data to API requests <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-adaptive-authentication.html#user-pool-settings-adaptive-authentication-device-fingerprint>`_ . If you don’t include this parameter, you can't send the source IP address to Amazon Cognito threat protection features. You can only activate ``EnablePropagateAdditionalUserContextData`` in an app client that has a client secret.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-enablepropagateadditionalusercontextdata
        '''
        result = self._values.get("enable_propagate_additional_user_context_data")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def enable_token_revocation(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Activates or deactivates token revocation.

        If you don't include this parameter, token revocation is automatically activated for the new user pool client.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-enabletokenrevocation
        '''
        result = self._values.get("enable_token_revocation")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def explicit_auth_flows(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The `authentication flows <https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow-methods.html>`_ that you want your user pool client to support. For each app client in your user pool, you can sign in your users with any combination of one or more flows, including with a user name and Secure Remote Password (SRP), a user name and password, or a custom authentication process that you define with Lambda functions.

        .. epigraph::

           If you don't specify a value for ``ExplicitAuthFlows`` , your app client supports ``ALLOW_REFRESH_TOKEN_AUTH`` , ``ALLOW_USER_SRP_AUTH`` , and ``ALLOW_CUSTOM_AUTH`` .

        The values for authentication flow options include the following.

        - ``ALLOW_USER_AUTH`` : Enable selection-based sign-in with ``USER_AUTH`` . This setting covers username-password, secure remote password (SRP), passwordless, and passkey authentication. This authentiation flow can do username-password and SRP authentication without other ``ExplicitAuthFlows`` permitting them. For example users can complete an SRP challenge through ``USER_AUTH`` without the flow ``USER_SRP_AUTH`` being active for the app client. This flow doesn't include ``CUSTOM_AUTH`` .

        To activate this setting, your user pool must be in the `Essentials tier <https://docs.aws.amazon.com/cognito/latest/developerguide/feature-plans-features-essentials.html>`_ or higher.

        - ``ALLOW_ADMIN_USER_PASSWORD_AUTH`` : Enable admin based user password authentication flow ``ADMIN_USER_PASSWORD_AUTH`` . This setting replaces the ``ADMIN_NO_SRP_AUTH`` setting. With this authentication flow, your app passes a user name and password to Amazon Cognito in the request, instead of using the Secure Remote Password (SRP) protocol to securely transmit the password.
        - ``ALLOW_CUSTOM_AUTH`` : Enable Lambda trigger based authentication.
        - ``ALLOW_USER_PASSWORD_AUTH`` : Enable user password-based authentication. In this flow, Amazon Cognito receives the password in the request instead of using the SRP protocol to verify passwords.
        - ``ALLOW_USER_SRP_AUTH`` : Enable SRP-based authentication.
        - ``ALLOW_REFRESH_TOKEN_AUTH`` : Enable authflow to refresh tokens.

        In some environments, you will see the values ``ADMIN_NO_SRP_AUTH`` , ``CUSTOM_AUTH_FLOW_ONLY`` , or ``USER_PASSWORD_AUTH`` . You can't assign these legacy ``ExplicitAuthFlows`` values to user pool clients at the same time as values that begin with ``ALLOW_`` ,
        like ``ALLOW_USER_SRP_AUTH`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-explicitauthflows
        '''
        result = self._values.get("explicit_auth_flows")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def generate_secret(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''When ``true`` , generates a client secret for the app client.

        Client secrets are used with server-side and machine-to-machine applications. Client secrets are automatically generated; you can't specify a secret value. For more information, see `App client types <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-client-apps.html#user-pool-settings-client-app-client-types>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-generatesecret
        '''
        result = self._values.get("generate_secret")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def id_token_validity(self) -> typing.Optional[jsii.Number]:
        '''The ID token time limit.

        After this limit expires, your user can't use their ID token. To specify the time unit for ``IdTokenValidity`` as ``seconds`` , ``minutes`` , ``hours`` , or ``days`` , set a ``TokenValidityUnits`` value in your API request.

        For example, when you set ``IdTokenValidity`` as ``10`` and ``TokenValidityUnits`` as ``hours`` , your user can authenticate their session with their ID token for 10 hours.

        The default time unit for ``IdTokenValidity`` in an API request is hours. *Valid range* is displayed below in seconds.

        If you don't specify otherwise in the configuration of your app client, your ID
        tokens are valid for one hour.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-idtokenvalidity
        '''
        result = self._values.get("id_token_validity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def logout_ur_ls(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of allowed logout URLs for managed login authentication.

        When you pass ``logout_uri`` and ``client_id`` parameters to ``/logout`` , Amazon Cognito signs out your user and redirects them to the logout URL. This parameter describes the URLs that you want to be the permitted targets of ``logout_uri`` . A typical use of these URLs is when a user selects "Sign out" and you redirect them to your public homepage. For more information, see `Logout endpoint <https://docs.aws.amazon.com/cognito/latest/developerguide/logout-endpoint.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-logouturls
        '''
        result = self._values.get("logout_ur_ls")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def prevent_user_existence_errors(self) -> typing.Optional[builtins.str]:
        '''Errors and responses that you want Amazon Cognito APIs to return during authentication, account confirmation, and password recovery when the user doesn't exist in the user pool.

        When set to ``ENABLED`` and the user doesn't exist, authentication returns an error indicating either the username or password was incorrect. Account confirmation and password recovery return a response indicating a code was sent to a simulated destination. When set to ``LEGACY`` , those APIs return a ``UserNotFoundException`` exception if the user doesn't exist in the user pool.

        Valid values include:

        - ``ENABLED`` - This prevents user existence-related errors.
        - ``LEGACY`` - This represents the early behavior of Amazon Cognito where user existence related errors aren't prevented.

        Defaults to ``LEGACY`` when you don't provide a value.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-preventuserexistenceerrors
        '''
        result = self._values.get("prevent_user_existence_errors")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def read_attributes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of user attributes that you want your app client to have read access to.

        After your user authenticates in your app, their access token authorizes them to read their own attribute value for any attribute in this list. An example of this kind of activity is when your user selects a link to view their profile information.

        When you don't specify the ``ReadAttributes`` for your app client, your app can read the values of ``email_verified`` , ``phone_number_verified`` , and the Standard attributes of your user pool. When your user pool app client has read access to these default attributes, ``ReadAttributes`` doesn't return any information. Amazon Cognito only populates ``ReadAttributes`` in the API response if you have specified your own custom set of read attributes.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-readattributes
        '''
        result = self._values.get("read_attributes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def refresh_token_validity(self) -> typing.Optional[jsii.Number]:
        '''The refresh token time limit.

        After this limit expires, your user can't use their refresh token. To specify the time unit for ``RefreshTokenValidity`` as ``seconds`` , ``minutes`` , ``hours`` , or ``days`` , set a ``TokenValidityUnits`` value in your API request.

        For example, when you set ``RefreshTokenValidity`` as ``10`` and ``TokenValidityUnits`` as ``days`` , your user can refresh their session
        and retrieve new access and ID tokens for 10 days.

        The default time unit for ``RefreshTokenValidity`` in an API request is days. You can't set ``RefreshTokenValidity`` to 0. If you do, Amazon Cognito overrides the value with the default value of 30 days. *Valid range* is displayed below in seconds.

        If you don't specify otherwise in the configuration of your app client, your refresh
        tokens are valid for 30 days.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-refreshtokenvalidity
        '''
        result = self._values.get("refresh_token_validity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def supported_identity_providers(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of provider names for the identity providers (IdPs) that are supported on this client.

        The following are supported: ``COGNITO`` , ``Facebook`` , ``Google`` , ``SignInWithApple`` , and ``LoginWithAmazon`` . You can also specify the names that you configured for the SAML and OIDC IdPs in your user pool, for example ``MySAMLIdP`` or ``MyOIDCIdP`` .

        This parameter sets the IdPs that `managed login <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-managed-login.html>`_ will display on the login page for your app client. The removal of ``COGNITO`` from this list doesn't prevent authentication operations for local users with the user pools API in an AWS SDK. The only way to prevent SDK-based authentication is to block access with a `AWS WAF rule <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-waf.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-supportedidentityproviders
        '''
        result = self._values.get("supported_identity_providers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def token_validity_units(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolClient.TokenValidityUnitsProperty]]:
        '''The units that validity times are represented in.

        The default unit for refresh tokens is days, and the default for ID and access tokens are hours.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-tokenvalidityunits
        '''
        result = self._values.get("token_validity_units")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolClient.TokenValidityUnitsProperty]], result)

    @builtins.property
    def write_attributes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of user attributes that you want your app client to have write access to.

        After your user authenticates in your app, their access token authorizes them to set or modify their own attribute value for any attribute in this list.

        When you don't specify the ``WriteAttributes`` for your app client, your app can write the values of the Standard attributes of your user pool. When your user pool has write access to these default attributes, ``WriteAttributes`` doesn't return any information. Amazon Cognito only populates ``WriteAttributes`` in the API response if you have specified your own custom set of write attributes.

        If your app client allows users to sign in through an IdP, this array must include all attributes that you have mapped to IdP attributes. Amazon Cognito updates mapped attributes when users sign in to your application through an IdP. If your app client does not have write access to a mapped attribute, Amazon Cognito throws an error when it tries to update the attribute. For more information, see `Specifying IdP Attribute Mappings for Your user pool <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-writeattributes
        '''
        result = self._values.get("write_attributes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUserPoolClientProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnUserPoolDomain(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolDomain",
):
    '''The AWS::Cognito::UserPoolDomain resource creates a new domain for a user pool.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooldomain.html
    :cloudformationResource: AWS::Cognito::UserPoolDomain
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        cfn_user_pool_domain = cognito.CfnUserPoolDomain(self, "MyCfnUserPoolDomain",
            domain="domain",
            user_pool_id="userPoolId",
        
            # the properties below are optional
            custom_domain_config=cognito.CfnUserPoolDomain.CustomDomainConfigTypeProperty(
                certificate_arn="certificateArn"
            ),
            managed_login_version=123
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        domain: builtins.str,
        user_pool_id: builtins.str,
        custom_domain_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolDomain.CustomDomainConfigTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        managed_login_version: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param domain: The name of the domain that you want to update. For custom domains, this is the fully-qualified domain name, for example ``auth.example.com`` . For prefix domains, this is the prefix alone, such as ``myprefix`` .
        :param user_pool_id: The ID of the user pool that is associated with the domain you're updating.
        :param custom_domain_config: The configuration for a custom domain that hosts the sign-up and sign-in pages for your application. Use this object to specify an SSL certificate that is managed by ACM. When you create a custom domain, the passkey RP ID defaults to the custom domain. If you had a prefix domain active, this will cause passkey integration for your prefix domain to stop working due to a mismatch in RP ID. To keep the prefix domain passkey integration working, you can explicitly set RP ID to the prefix domain.
        :param managed_login_version: A version number that indicates the state of managed login for your domain. Version ``1`` is hosted UI (classic). Version ``2`` is the newer managed login with the branding designer. For more information, see `Managed login <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-managed-login.html>`_ .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6e0b36c4d155cfdfa9801e3f221c4fe6c5403bf24a64d17bd90fb5386301d675)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUserPoolDomainProps(
            domain=domain,
            user_pool_id=user_pool_id,
            custom_domain_config=custom_domain_config,
            managed_login_version=managed_login_version,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__847babb3c81b57d55437dd6a17098995090128630b1accb7f8cd0e345f7e39f6)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__20a5c06e049fbdd5f26b1d38640e46140175b60b5ef53e207fd38f528ca24f48)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCloudFrontDistribution")
    def attr_cloud_front_distribution(self) -> builtins.str:
        '''The Amazon CloudFront endpoint that you use as the target of the alias that you set up with your Domain Name Service (DNS) provider.

        :cloudformationAttribute: CloudFrontDistribution
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCloudFrontDistribution"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="domain")
    def domain(self) -> builtins.str:
        '''The name of the domain that you want to update.'''
        return typing.cast(builtins.str, jsii.get(self, "domain"))

    @domain.setter
    def domain(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__60984af4b9f95693c974811dba163dc41c64348226be0621cde58067d75c2a75)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domain", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool that is associated with the domain you're updating.'''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @user_pool_id.setter
    def user_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__01784117c77cfebe2ef420bdb95cfa641714c4080a504ef08ebfbc5cb9e64418)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="customDomainConfig")
    def custom_domain_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolDomain.CustomDomainConfigTypeProperty"]]:
        '''The configuration for a custom domain that hosts the sign-up and sign-in pages for your application.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolDomain.CustomDomainConfigTypeProperty"]], jsii.get(self, "customDomainConfig"))

    @custom_domain_config.setter
    def custom_domain_config(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolDomain.CustomDomainConfigTypeProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__534c4957c36eac9a89217ff1b762b65d25e33f26c5048218fc840dc7f7ee2b0e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "customDomainConfig", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="managedLoginVersion")
    def managed_login_version(self) -> typing.Optional[jsii.Number]:
        '''A version number that indicates the state of managed login for your domain.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "managedLoginVersion"))

    @managed_login_version.setter
    def managed_login_version(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b1d6e8e96816f3572291ff67691b98d76a166cf058320e0e73e58062b8093526)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "managedLoginVersion", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolDomain.CustomDomainConfigTypeProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_arn": "certificateArn"},
    )
    class CustomDomainConfigTypeProperty:
        def __init__(
            self,
            *,
            certificate_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The configuration for a hosted UI custom domain.

            :param certificate_arn: The Amazon Resource Name (ARN) of an AWS Certificate Manager SSL certificate. You use this certificate for the subdomain of your custom domain.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpooldomain-customdomainconfigtype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                custom_domain_config_type_property = cognito.CfnUserPoolDomain.CustomDomainConfigTypeProperty(
                    certificate_arn="certificateArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dde97995e450b3b0c5468a27b415565086c00f64bdc255f297a8471e77b85243)
                check_type(argname="argument certificate_arn", value=certificate_arn, expected_type=type_hints["certificate_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if certificate_arn is not None:
                self._values["certificate_arn"] = certificate_arn

        @builtins.property
        def certificate_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of an AWS Certificate Manager SSL certificate.

            You use this certificate for the subdomain of your custom domain.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpooldomain-customdomainconfigtype.html#cfn-cognito-userpooldomain-customdomainconfigtype-certificatearn
            '''
            result = self._values.get("certificate_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CustomDomainConfigTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolDomainProps",
    jsii_struct_bases=[],
    name_mapping={
        "domain": "domain",
        "user_pool_id": "userPoolId",
        "custom_domain_config": "customDomainConfig",
        "managed_login_version": "managedLoginVersion",
    },
)
class CfnUserPoolDomainProps:
    def __init__(
        self,
        *,
        domain: builtins.str,
        user_pool_id: builtins.str,
        custom_domain_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPoolDomain.CustomDomainConfigTypeProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        managed_login_version: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``CfnUserPoolDomain``.

        :param domain: The name of the domain that you want to update. For custom domains, this is the fully-qualified domain name, for example ``auth.example.com`` . For prefix domains, this is the prefix alone, such as ``myprefix`` .
        :param user_pool_id: The ID of the user pool that is associated with the domain you're updating.
        :param custom_domain_config: The configuration for a custom domain that hosts the sign-up and sign-in pages for your application. Use this object to specify an SSL certificate that is managed by ACM. When you create a custom domain, the passkey RP ID defaults to the custom domain. If you had a prefix domain active, this will cause passkey integration for your prefix domain to stop working due to a mismatch in RP ID. To keep the prefix domain passkey integration working, you can explicitly set RP ID to the prefix domain.
        :param managed_login_version: A version number that indicates the state of managed login for your domain. Version ``1`` is hosted UI (classic). Version ``2`` is the newer managed login with the branding designer. For more information, see `Managed login <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-managed-login.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooldomain.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            cfn_user_pool_domain_props = cognito.CfnUserPoolDomainProps(
                domain="domain",
                user_pool_id="userPoolId",
            
                # the properties below are optional
                custom_domain_config=cognito.CfnUserPoolDomain.CustomDomainConfigTypeProperty(
                    certificate_arn="certificateArn"
                ),
                managed_login_version=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe5ef2b7c4347565bc988b8d9120bbd5feadcfadd061512019de1519e2cc9ef4)
            check_type(argname="argument domain", value=domain, expected_type=type_hints["domain"])
            check_type(argname="argument user_pool_id", value=user_pool_id, expected_type=type_hints["user_pool_id"])
            check_type(argname="argument custom_domain_config", value=custom_domain_config, expected_type=type_hints["custom_domain_config"])
            check_type(argname="argument managed_login_version", value=managed_login_version, expected_type=type_hints["managed_login_version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "domain": domain,
            "user_pool_id": user_pool_id,
        }
        if custom_domain_config is not None:
            self._values["custom_domain_config"] = custom_domain_config
        if managed_login_version is not None:
            self._values["managed_login_version"] = managed_login_version

    @builtins.property
    def domain(self) -> builtins.str:
        '''The name of the domain that you want to update.

        For custom domains, this is the fully-qualified domain name, for example ``auth.example.com`` . For prefix domains, this is the prefix alone, such as ``myprefix`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooldomain.html#cfn-cognito-userpooldomain-domain
        '''
        result = self._values.get("domain")
        assert result is not None, "Required property 'domain' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool that is associated with the domain you're updating.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooldomain.html#cfn-cognito-userpooldomain-userpoolid
        '''
        result = self._values.get("user_pool_id")
        assert result is not None, "Required property 'user_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def custom_domain_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolDomain.CustomDomainConfigTypeProperty]]:
        '''The configuration for a custom domain that hosts the sign-up and sign-in pages for your application.

        Use this object to specify an SSL certificate that is managed by ACM.

        When you create a custom domain, the passkey RP ID defaults to the custom domain. If you had a prefix domain active, this will cause passkey integration for your prefix domain to stop working due to a mismatch in RP ID. To keep the prefix domain passkey integration working, you can explicitly set RP ID to the prefix domain.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooldomain.html#cfn-cognito-userpooldomain-customdomainconfig
        '''
        result = self._values.get("custom_domain_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolDomain.CustomDomainConfigTypeProperty]], result)

    @builtins.property
    def managed_login_version(self) -> typing.Optional[jsii.Number]:
        '''A version number that indicates the state of managed login for your domain.

        Version ``1`` is hosted UI (classic). Version ``2`` is the newer managed login with the branding designer. For more information, see `Managed login <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-managed-login.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooldomain.html#cfn-cognito-userpooldomain-managedloginversion
        '''
        result = self._values.get("managed_login_version")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUserPoolDomainProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnUserPoolGroup(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolGroup",
):
    '''A user pool group.

    Contains details about the group and the way that it contributes to IAM role decisions with identity pools. Identity pools can make decisions about the IAM role to assign based on groups: users get credentials for the role associated with their highest-priority group.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolgroup.html
    :cloudformationResource: AWS::Cognito::UserPoolGroup
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        cfn_user_pool_group = cognito.CfnUserPoolGroup(self, "MyCfnUserPoolGroup",
            user_pool_id="userPoolId",
        
            # the properties below are optional
            description="description",
            group_name="groupName",
            precedence=123,
            role_arn="roleArn"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        user_pool_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
        group_name: typing.Optional[builtins.str] = None,
        precedence: typing.Optional[jsii.Number] = None,
        role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param user_pool_id: The ID of the user pool where you want to create a user group.
        :param description: A description of the group that you're creating.
        :param group_name: A name for the group. This name must be unique in your user pool.
        :param precedence: A non-negative integer value that specifies the precedence of this group relative to the other groups that a user can belong to in the user pool. Zero is the highest precedence value. Groups with lower ``Precedence`` values take precedence over groups with higher or null ``Precedence`` values. If a user belongs to two or more groups, it is the group with the lowest precedence value whose role ARN is given in the user's tokens for the ``cognito:roles`` and ``cognito:preferred_role`` claims. Two groups can have the same ``Precedence`` value. If this happens, neither group takes precedence over the other. If two groups with the same ``Precedence`` have the same role ARN, that role is used in the ``cognito:preferred_role`` claim in tokens for users in each group. If the two groups have different role ARNs, the ``cognito:preferred_role`` claim isn't set in users' tokens. The default ``Precedence`` value is null. The maximum ``Precedence`` value is ``2^31-1`` .
        :param role_arn: The Amazon Resource Name (ARN) for the IAM role that you want to associate with the group. A group role primarily declares a preferred role for the credentials that you get from an identity pool. Amazon Cognito ID tokens have a ``cognito:preferred_role`` claim that presents the highest-precedence group that a user belongs to. Both ID and access tokens also contain a ``cognito:groups`` claim that list all the groups that a user is a member of.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__64c0c3e44e677deced2e13e5ab4d3c7550518140bbe9dfaf681651d95cc8ab55)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUserPoolGroupProps(
            user_pool_id=user_pool_id,
            description=description,
            group_name=group_name,
            precedence=precedence,
            role_arn=role_arn,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__abc28a51aacedb7e9a644933fe12ec8d528ed92ffc139d943ff29a1cb3d0c875)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__37c5e9ee822db2aa6c46f1ec2a13dafe1becc09a4149823491d646c65d5f9d8a)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you want to create a user group.'''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @user_pool_id.setter
    def user_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ca8402c546b7f090c567d019103e9d2674bfb6e4c199608669a4b2acc2afc5f3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the group that you're creating.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7bda9e548a82a5206aec39e356f410d438353851dcd1f974bd59be186f5ff66a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="groupName")
    def group_name(self) -> typing.Optional[builtins.str]:
        '''A name for the group.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "groupName"))

    @group_name.setter
    def group_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__50cee63463b0a00e9516a517c3ecba89edae39efb6890c70ea606f9e3cfca0bc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "groupName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="precedence")
    def precedence(self) -> typing.Optional[jsii.Number]:
        '''A non-negative integer value that specifies the precedence of this group relative to the other groups that a user can belong to in the user pool.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "precedence"))

    @precedence.setter
    def precedence(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a1542aa260237dd220738bd8c2fdd54d12089e9f7ff0f7f70cd927c2f8d90da2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "precedence", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) for the IAM role that you want to associate with the group.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "roleArn"))

    @role_arn.setter
    def role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3a22fc1c6f27de296195da5e404a5a9d43aea635361d630ed9185ab2d04e6ee2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "roleArn", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "user_pool_id": "userPoolId",
        "description": "description",
        "group_name": "groupName",
        "precedence": "precedence",
        "role_arn": "roleArn",
    },
)
class CfnUserPoolGroupProps:
    def __init__(
        self,
        *,
        user_pool_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
        group_name: typing.Optional[builtins.str] = None,
        precedence: typing.Optional[jsii.Number] = None,
        role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnUserPoolGroup``.

        :param user_pool_id: The ID of the user pool where you want to create a user group.
        :param description: A description of the group that you're creating.
        :param group_name: A name for the group. This name must be unique in your user pool.
        :param precedence: A non-negative integer value that specifies the precedence of this group relative to the other groups that a user can belong to in the user pool. Zero is the highest precedence value. Groups with lower ``Precedence`` values take precedence over groups with higher or null ``Precedence`` values. If a user belongs to two or more groups, it is the group with the lowest precedence value whose role ARN is given in the user's tokens for the ``cognito:roles`` and ``cognito:preferred_role`` claims. Two groups can have the same ``Precedence`` value. If this happens, neither group takes precedence over the other. If two groups with the same ``Precedence`` have the same role ARN, that role is used in the ``cognito:preferred_role`` claim in tokens for users in each group. If the two groups have different role ARNs, the ``cognito:preferred_role`` claim isn't set in users' tokens. The default ``Precedence`` value is null. The maximum ``Precedence`` value is ``2^31-1`` .
        :param role_arn: The Amazon Resource Name (ARN) for the IAM role that you want to associate with the group. A group role primarily declares a preferred role for the credentials that you get from an identity pool. Amazon Cognito ID tokens have a ``cognito:preferred_role`` claim that presents the highest-precedence group that a user belongs to. Both ID and access tokens also contain a ``cognito:groups`` claim that list all the groups that a user is a member of.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolgroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            cfn_user_pool_group_props = cognito.CfnUserPoolGroupProps(
                user_pool_id="userPoolId",
            
                # the properties below are optional
                description="description",
                group_name="groupName",
                precedence=123,
                role_arn="roleArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c67d3106556c2dbeca0c87692e053d37b59118869232a8d862cf3afebbfdb774)
            check_type(argname="argument user_pool_id", value=user_pool_id, expected_type=type_hints["user_pool_id"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument group_name", value=group_name, expected_type=type_hints["group_name"])
            check_type(argname="argument precedence", value=precedence, expected_type=type_hints["precedence"])
            check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "user_pool_id": user_pool_id,
        }
        if description is not None:
            self._values["description"] = description
        if group_name is not None:
            self._values["group_name"] = group_name
        if precedence is not None:
            self._values["precedence"] = precedence
        if role_arn is not None:
            self._values["role_arn"] = role_arn

    @builtins.property
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you want to create a user group.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolgroup.html#cfn-cognito-userpoolgroup-userpoolid
        '''
        result = self._values.get("user_pool_id")
        assert result is not None, "Required property 'user_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the group that you're creating.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolgroup.html#cfn-cognito-userpoolgroup-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def group_name(self) -> typing.Optional[builtins.str]:
        '''A name for the group.

        This name must be unique in your user pool.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolgroup.html#cfn-cognito-userpoolgroup-groupname
        '''
        result = self._values.get("group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def precedence(self) -> typing.Optional[jsii.Number]:
        '''A non-negative integer value that specifies the precedence of this group relative to the other groups that a user can belong to in the user pool.

        Zero is the highest precedence value. Groups with lower ``Precedence`` values take precedence over groups with higher or null ``Precedence`` values. If a user belongs to two or more groups, it is the group with the lowest precedence value whose role ARN is given in the user's tokens for the ``cognito:roles`` and ``cognito:preferred_role`` claims.

        Two groups can have the same ``Precedence`` value. If this happens, neither group takes precedence over the other. If two groups with the same ``Precedence`` have the same role ARN, that role is used in the ``cognito:preferred_role`` claim in tokens for users in each group. If the two groups have different role ARNs, the ``cognito:preferred_role`` claim isn't set in users' tokens.

        The default ``Precedence`` value is null. The maximum ``Precedence`` value is ``2^31-1`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolgroup.html#cfn-cognito-userpoolgroup-precedence
        '''
        result = self._values.get("precedence")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) for the IAM role that you want to associate with the group.

        A group role primarily declares a preferred role for the credentials that you get from an identity pool. Amazon Cognito ID tokens have a ``cognito:preferred_role`` claim that presents the highest-precedence group that a user belongs to. Both ID and access tokens also contain a ``cognito:groups`` claim that list all the groups that a user is a member of.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolgroup.html#cfn-cognito-userpoolgroup-rolearn
        '''
        result = self._values.get("role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUserPoolGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnUserPoolIdentityProvider(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolIdentityProvider",
):
    '''The ``AWS::Cognito::UserPoolIdentityProvider`` resource creates an identity provider for a user pool.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolidentityprovider.html
    :cloudformationResource: AWS::Cognito::UserPoolIdentityProvider
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        # attribute_mapping: Any
        # provider_details: Any
        
        cfn_user_pool_identity_provider = cognito.CfnUserPoolIdentityProvider(self, "MyCfnUserPoolIdentityProvider",
            provider_details=provider_details,
            provider_name="providerName",
            provider_type="providerType",
            user_pool_id="userPoolId",
        
            # the properties below are optional
            attribute_mapping=attribute_mapping,
            idp_identifiers=["idpIdentifiers"]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        provider_details: typing.Any,
        provider_name: builtins.str,
        provider_type: builtins.str,
        user_pool_id: builtins.str,
        attribute_mapping: typing.Any = None,
        idp_identifiers: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param provider_details: The scopes, URLs, and identifiers for your external identity provider. The following examples describe the provider detail keys for each IdP type. These values and their schema are subject to change. Social IdP ``authorize_scopes`` values must match the values listed here. - **OpenID Connect (OIDC)** - Amazon Cognito accepts the following elements when it can't discover endpoint URLs from ``oidc_issuer`` : ``attributes_url`` , ``authorize_url`` , ``jwks_uri`` , ``token_url`` . Create or update request: ``"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }`` Describe response: ``"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "attributes_url_add_attributes": "false", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }`` - **SAML** - Create or update request with Metadata URL: ``"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256" }`` Create or update request with Metadata file: ``"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataFile": "[metadata XML]", "RequestSigningAlgorithm": "rsa-sha256" }`` The value of ``MetadataFile`` must be the plaintext metadata document with all quote (") characters escaped by backslashes. Describe response: ``"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "ActiveEncryptionCertificate": "[certificate]", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256", "SLORedirectBindingURI": "https://auth.example.com/slo/saml", "SSORedirectBindingURI": "https://auth.example.com/sso/saml" }`` - **LoginWithAmazon** - Create or update request: ``"ProviderDetails": { "authorize_scopes": "profile postal_code", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret"`` Describe response: ``"ProviderDetails": { "attributes_url": "https://api.amazon.com/user/profile", "attributes_url_add_attributes": "false", "authorize_scopes": "profile postal_code", "authorize_url": "https://www.amazon.com/ap/oa", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "POST", "token_url": "https://api.amazon.com/auth/o2/token" }`` - **Google** - Create or update request: ``"ProviderDetails": { "authorize_scopes": "email profile openid", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret" }`` Describe response: ``"ProviderDetails": { "attributes_url": "https://people.googleapis.com/v1/people/me?personFields=", "attributes_url_add_attributes": "true", "authorize_scopes": "email profile openid", "authorize_url": "https://accounts.google.com/o/oauth2/v2/auth", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret", "oidc_issuer": "https://accounts.google.com", "token_request_method": "POST", "token_url": "https://www.googleapis.com/oauth2/v4/token" }`` - **SignInWithApple** - Create or update request: ``"ProviderDetails": { "authorize_scopes": "email name", "client_id": "com.example.cognito", "private_key": "1EXAMPLE", "key_id": "2EXAMPLE", "team_id": "3EXAMPLE" }`` Describe response: ``"ProviderDetails": { "attributes_url_add_attributes": "false", "authorize_scopes": "email name", "authorize_url": "https://appleid.apple.com/auth/authorize", "client_id": "com.example.cognito", "key_id": "1EXAMPLE", "oidc_issuer": "https://appleid.apple.com", "team_id": "2EXAMPLE", "token_request_method": "POST", "token_url": "https://appleid.apple.com/auth/token" }`` - **Facebook** - Create or update request: ``"ProviderDetails": { "api_version": "v17.0", "authorize_scopes": "public_profile, email", "client_id": "1example23456789", "client_secret": "provider-app-client-secret" }`` Describe response: ``"ProviderDetails": { "api_version": "v17.0", "attributes_url": "https://graph.facebook.com/v17.0/me?fields=", "attributes_url_add_attributes": "true", "authorize_scopes": "public_profile, email", "authorize_url": "https://www.facebook.com/v17.0/dialog/oauth", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "GET", "token_url": "https://graph.facebook.com/v17.0/oauth/access_token" }``
        :param provider_name: The name that you want to assign to the IdP. You can pass the identity provider name in the ``identity_provider`` query parameter of requests to the `Authorize endpoint <https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html>`_ to silently redirect to sign-in with the associated IdP.
        :param provider_type: The type of IdP that you want to add. Amazon Cognito supports OIDC, SAML 2.0, Login With Amazon, Sign In With Apple, Google, and Facebook IdPs.
        :param user_pool_id: The Id of the user pool where you want to create an IdP.
        :param attribute_mapping: A mapping of IdP attributes to standard and custom user pool attributes. Specify a user pool attribute as the key of the key-value pair, and the IdP attribute claim name as the value.
        :param idp_identifiers: An array of IdP identifiers, for example ``"IdPIdentifiers": [ "MyIdP", "MyIdP2" ]`` . Identifiers are friendly names that you can pass in the ``idp_identifier`` query parameter of requests to the `Authorize endpoint <https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html>`_ to silently redirect to sign-in with the associated IdP. Identifiers in a domain format also enable the use of `email-address matching with SAML providers <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-managing-saml-idp-naming.html>`_ .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__759e90505ceb64aa7002be11d4da4a87090102263927799f662a83f606483634)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUserPoolIdentityProviderProps(
            provider_details=provider_details,
            provider_name=provider_name,
            provider_type=provider_type,
            user_pool_id=user_pool_id,
            attribute_mapping=attribute_mapping,
            idp_identifiers=idp_identifiers,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e932e2192b5a3559a5e803d6dd262a6ac5a8fe2ba71f404e3905fb3db10d9287)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ff11acc316d5d73192edfeab5a5d7fb2aa7891c069fce7ccaa8763007889809)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="providerDetails")
    def provider_details(self) -> typing.Any:
        '''The scopes, URLs, and identifiers for your external identity provider.'''
        return typing.cast(typing.Any, jsii.get(self, "providerDetails"))

    @provider_details.setter
    def provider_details(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dd9b80463fd736be9b8b32bf8d2368b0c44578e3b056d45e068ca1e5fdfdb299)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "providerDetails", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="providerName")
    def provider_name(self) -> builtins.str:
        '''The name that you want to assign to the IdP.'''
        return typing.cast(builtins.str, jsii.get(self, "providerName"))

    @provider_name.setter
    def provider_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__03fef1ca3436f487bdb2ac4c72e914ca702f01a40d12470aaa64c77a0f7e15a2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "providerName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="providerType")
    def provider_type(self) -> builtins.str:
        '''The type of IdP that you want to add.'''
        return typing.cast(builtins.str, jsii.get(self, "providerType"))

    @provider_type.setter
    def provider_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1ee8fc97c7ce3a7223eabd6be18090c6fec027c8b81ecf420c175e278dafb535)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "providerType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The Id of the user pool where you want to create an IdP.'''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @user_pool_id.setter
    def user_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8f109ed40a77a3077cbab955821323c4293e2697420abd221b2032bb4abc2edb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="attributeMapping")
    def attribute_mapping(self) -> typing.Any:
        '''A mapping of IdP attributes to standard and custom user pool attributes.'''
        return typing.cast(typing.Any, jsii.get(self, "attributeMapping"))

    @attribute_mapping.setter
    def attribute_mapping(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bb44cf93ad9714e2bf7765a12f95e3a12d5a4b89bee8d2de779a247dbd587039)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "attributeMapping", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="idpIdentifiers")
    def idp_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''An array of IdP identifiers, for example ``"IdPIdentifiers": [ "MyIdP", "MyIdP2" ]`` .'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "idpIdentifiers"))

    @idp_identifiers.setter
    def idp_identifiers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7662247fd2cd01f6776c3a84fedff308a45861e95cabe426cb256482aaeaead5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "idpIdentifiers", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolIdentityProviderProps",
    jsii_struct_bases=[],
    name_mapping={
        "provider_details": "providerDetails",
        "provider_name": "providerName",
        "provider_type": "providerType",
        "user_pool_id": "userPoolId",
        "attribute_mapping": "attributeMapping",
        "idp_identifiers": "idpIdentifiers",
    },
)
class CfnUserPoolIdentityProviderProps:
    def __init__(
        self,
        *,
        provider_details: typing.Any,
        provider_name: builtins.str,
        provider_type: builtins.str,
        user_pool_id: builtins.str,
        attribute_mapping: typing.Any = None,
        idp_identifiers: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnUserPoolIdentityProvider``.

        :param provider_details: The scopes, URLs, and identifiers for your external identity provider. The following examples describe the provider detail keys for each IdP type. These values and their schema are subject to change. Social IdP ``authorize_scopes`` values must match the values listed here. - **OpenID Connect (OIDC)** - Amazon Cognito accepts the following elements when it can't discover endpoint URLs from ``oidc_issuer`` : ``attributes_url`` , ``authorize_url`` , ``jwks_uri`` , ``token_url`` . Create or update request: ``"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }`` Describe response: ``"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "attributes_url_add_attributes": "false", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }`` - **SAML** - Create or update request with Metadata URL: ``"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256" }`` Create or update request with Metadata file: ``"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataFile": "[metadata XML]", "RequestSigningAlgorithm": "rsa-sha256" }`` The value of ``MetadataFile`` must be the plaintext metadata document with all quote (") characters escaped by backslashes. Describe response: ``"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "ActiveEncryptionCertificate": "[certificate]", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256", "SLORedirectBindingURI": "https://auth.example.com/slo/saml", "SSORedirectBindingURI": "https://auth.example.com/sso/saml" }`` - **LoginWithAmazon** - Create or update request: ``"ProviderDetails": { "authorize_scopes": "profile postal_code", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret"`` Describe response: ``"ProviderDetails": { "attributes_url": "https://api.amazon.com/user/profile", "attributes_url_add_attributes": "false", "authorize_scopes": "profile postal_code", "authorize_url": "https://www.amazon.com/ap/oa", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "POST", "token_url": "https://api.amazon.com/auth/o2/token" }`` - **Google** - Create or update request: ``"ProviderDetails": { "authorize_scopes": "email profile openid", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret" }`` Describe response: ``"ProviderDetails": { "attributes_url": "https://people.googleapis.com/v1/people/me?personFields=", "attributes_url_add_attributes": "true", "authorize_scopes": "email profile openid", "authorize_url": "https://accounts.google.com/o/oauth2/v2/auth", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret", "oidc_issuer": "https://accounts.google.com", "token_request_method": "POST", "token_url": "https://www.googleapis.com/oauth2/v4/token" }`` - **SignInWithApple** - Create or update request: ``"ProviderDetails": { "authorize_scopes": "email name", "client_id": "com.example.cognito", "private_key": "1EXAMPLE", "key_id": "2EXAMPLE", "team_id": "3EXAMPLE" }`` Describe response: ``"ProviderDetails": { "attributes_url_add_attributes": "false", "authorize_scopes": "email name", "authorize_url": "https://appleid.apple.com/auth/authorize", "client_id": "com.example.cognito", "key_id": "1EXAMPLE", "oidc_issuer": "https://appleid.apple.com", "team_id": "2EXAMPLE", "token_request_method": "POST", "token_url": "https://appleid.apple.com/auth/token" }`` - **Facebook** - Create or update request: ``"ProviderDetails": { "api_version": "v17.0", "authorize_scopes": "public_profile, email", "client_id": "1example23456789", "client_secret": "provider-app-client-secret" }`` Describe response: ``"ProviderDetails": { "api_version": "v17.0", "attributes_url": "https://graph.facebook.com/v17.0/me?fields=", "attributes_url_add_attributes": "true", "authorize_scopes": "public_profile, email", "authorize_url": "https://www.facebook.com/v17.0/dialog/oauth", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "GET", "token_url": "https://graph.facebook.com/v17.0/oauth/access_token" }``
        :param provider_name: The name that you want to assign to the IdP. You can pass the identity provider name in the ``identity_provider`` query parameter of requests to the `Authorize endpoint <https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html>`_ to silently redirect to sign-in with the associated IdP.
        :param provider_type: The type of IdP that you want to add. Amazon Cognito supports OIDC, SAML 2.0, Login With Amazon, Sign In With Apple, Google, and Facebook IdPs.
        :param user_pool_id: The Id of the user pool where you want to create an IdP.
        :param attribute_mapping: A mapping of IdP attributes to standard and custom user pool attributes. Specify a user pool attribute as the key of the key-value pair, and the IdP attribute claim name as the value.
        :param idp_identifiers: An array of IdP identifiers, for example ``"IdPIdentifiers": [ "MyIdP", "MyIdP2" ]`` . Identifiers are friendly names that you can pass in the ``idp_identifier`` query parameter of requests to the `Authorize endpoint <https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html>`_ to silently redirect to sign-in with the associated IdP. Identifiers in a domain format also enable the use of `email-address matching with SAML providers <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-managing-saml-idp-naming.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolidentityprovider.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            # attribute_mapping: Any
            # provider_details: Any
            
            cfn_user_pool_identity_provider_props = cognito.CfnUserPoolIdentityProviderProps(
                provider_details=provider_details,
                provider_name="providerName",
                provider_type="providerType",
                user_pool_id="userPoolId",
            
                # the properties below are optional
                attribute_mapping=attribute_mapping,
                idp_identifiers=["idpIdentifiers"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__41106943fcdd509be0174e1e1c8a8c320bd77587c77e22cfc1c1b7378dfb42ec)
            check_type(argname="argument provider_details", value=provider_details, expected_type=type_hints["provider_details"])
            check_type(argname="argument provider_name", value=provider_name, expected_type=type_hints["provider_name"])
            check_type(argname="argument provider_type", value=provider_type, expected_type=type_hints["provider_type"])
            check_type(argname="argument user_pool_id", value=user_pool_id, expected_type=type_hints["user_pool_id"])
            check_type(argname="argument attribute_mapping", value=attribute_mapping, expected_type=type_hints["attribute_mapping"])
            check_type(argname="argument idp_identifiers", value=idp_identifiers, expected_type=type_hints["idp_identifiers"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "provider_details": provider_details,
            "provider_name": provider_name,
            "provider_type": provider_type,
            "user_pool_id": user_pool_id,
        }
        if attribute_mapping is not None:
            self._values["attribute_mapping"] = attribute_mapping
        if idp_identifiers is not None:
            self._values["idp_identifiers"] = idp_identifiers

    @builtins.property
    def provider_details(self) -> typing.Any:
        '''The scopes, URLs, and identifiers for your external identity provider.

        The following
        examples describe the provider detail keys for each IdP type. These values and their
        schema are subject to change. Social IdP ``authorize_scopes`` values must match
        the values listed here.

        - **OpenID Connect (OIDC)** - Amazon Cognito accepts the following elements when it can't discover endpoint URLs from ``oidc_issuer`` : ``attributes_url`` , ``authorize_url`` , ``jwks_uri`` , ``token_url`` .

        Create or update request: ``"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }``

        Describe response: ``"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "attributes_url_add_attributes": "false", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }``

        - **SAML** - Create or update request with Metadata URL: ``"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256" }``

        Create or update request with Metadata file: ``"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataFile": "[metadata XML]", "RequestSigningAlgorithm": "rsa-sha256" }``

        The value of ``MetadataFile`` must be the plaintext metadata document with all quote (") characters escaped by backslashes.

        Describe response: ``"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "ActiveEncryptionCertificate": "[certificate]", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256", "SLORedirectBindingURI": "https://auth.example.com/slo/saml", "SSORedirectBindingURI": "https://auth.example.com/sso/saml" }``

        - **LoginWithAmazon** - Create or update request: ``"ProviderDetails": { "authorize_scopes": "profile postal_code", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret"``

        Describe response: ``"ProviderDetails": { "attributes_url": "https://api.amazon.com/user/profile", "attributes_url_add_attributes": "false", "authorize_scopes": "profile postal_code", "authorize_url": "https://www.amazon.com/ap/oa", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "POST", "token_url": "https://api.amazon.com/auth/o2/token" }``

        - **Google** - Create or update request: ``"ProviderDetails": { "authorize_scopes": "email profile openid", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret" }``

        Describe response: ``"ProviderDetails": { "attributes_url": "https://people.googleapis.com/v1/people/me?personFields=", "attributes_url_add_attributes": "true", "authorize_scopes": "email profile openid", "authorize_url": "https://accounts.google.com/o/oauth2/v2/auth", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret", "oidc_issuer": "https://accounts.google.com", "token_request_method": "POST", "token_url": "https://www.googleapis.com/oauth2/v4/token" }``

        - **SignInWithApple** - Create or update request: ``"ProviderDetails": { "authorize_scopes": "email name", "client_id": "com.example.cognito", "private_key": "1EXAMPLE", "key_id": "2EXAMPLE", "team_id": "3EXAMPLE" }``

        Describe response: ``"ProviderDetails": { "attributes_url_add_attributes": "false", "authorize_scopes": "email name", "authorize_url": "https://appleid.apple.com/auth/authorize", "client_id": "com.example.cognito", "key_id": "1EXAMPLE", "oidc_issuer": "https://appleid.apple.com", "team_id": "2EXAMPLE", "token_request_method": "POST", "token_url": "https://appleid.apple.com/auth/token" }``

        - **Facebook** - Create or update request: ``"ProviderDetails": { "api_version": "v17.0", "authorize_scopes": "public_profile, email", "client_id": "1example23456789", "client_secret": "provider-app-client-secret" }``

        Describe response: ``"ProviderDetails": { "api_version": "v17.0", "attributes_url": "https://graph.facebook.com/v17.0/me?fields=", "attributes_url_add_attributes": "true", "authorize_scopes": "public_profile, email", "authorize_url": "https://www.facebook.com/v17.0/dialog/oauth", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "GET", "token_url": "https://graph.facebook.com/v17.0/oauth/access_token" }``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolidentityprovider.html#cfn-cognito-userpoolidentityprovider-providerdetails
        '''
        result = self._values.get("provider_details")
        assert result is not None, "Required property 'provider_details' is missing"
        return typing.cast(typing.Any, result)

    @builtins.property
    def provider_name(self) -> builtins.str:
        '''The name that you want to assign to the IdP.

        You can pass the identity provider name in the ``identity_provider`` query parameter of requests to the `Authorize endpoint <https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html>`_ to silently redirect to sign-in with the associated IdP.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolidentityprovider.html#cfn-cognito-userpoolidentityprovider-providername
        '''
        result = self._values.get("provider_name")
        assert result is not None, "Required property 'provider_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def provider_type(self) -> builtins.str:
        '''The type of IdP that you want to add.

        Amazon Cognito supports OIDC, SAML 2.0, Login With Amazon, Sign In With Apple, Google, and Facebook IdPs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolidentityprovider.html#cfn-cognito-userpoolidentityprovider-providertype
        '''
        result = self._values.get("provider_type")
        assert result is not None, "Required property 'provider_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def user_pool_id(self) -> builtins.str:
        '''The Id of the user pool where you want to create an IdP.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolidentityprovider.html#cfn-cognito-userpoolidentityprovider-userpoolid
        '''
        result = self._values.get("user_pool_id")
        assert result is not None, "Required property 'user_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def attribute_mapping(self) -> typing.Any:
        '''A mapping of IdP attributes to standard and custom user pool attributes.

        Specify a user pool attribute as the key of the key-value pair, and the IdP attribute claim name as the value.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolidentityprovider.html#cfn-cognito-userpoolidentityprovider-attributemapping
        '''
        result = self._values.get("attribute_mapping")
        return typing.cast(typing.Any, result)

    @builtins.property
    def idp_identifiers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''An array of IdP identifiers, for example ``"IdPIdentifiers": [ "MyIdP", "MyIdP2" ]`` .

        Identifiers are friendly names that you can pass in the ``idp_identifier`` query parameter of requests to the `Authorize endpoint <https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html>`_ to silently redirect to sign-in with the associated IdP. Identifiers in a domain format also enable the use of `email-address matching with SAML providers <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-managing-saml-idp-naming.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolidentityprovider.html#cfn-cognito-userpoolidentityprovider-idpidentifiers
        '''
        result = self._values.get("idp_identifiers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUserPoolIdentityProviderProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolProps",
    jsii_struct_bases=[],
    name_mapping={
        "account_recovery_setting": "accountRecoverySetting",
        "admin_create_user_config": "adminCreateUserConfig",
        "alias_attributes": "aliasAttributes",
        "auto_verified_attributes": "autoVerifiedAttributes",
        "deletion_protection": "deletionProtection",
        "device_configuration": "deviceConfiguration",
        "email_authentication_message": "emailAuthenticationMessage",
        "email_authentication_subject": "emailAuthenticationSubject",
        "email_configuration": "emailConfiguration",
        "email_verification_message": "emailVerificationMessage",
        "email_verification_subject": "emailVerificationSubject",
        "enabled_mfas": "enabledMfas",
        "lambda_config": "lambdaConfig",
        "mfa_configuration": "mfaConfiguration",
        "policies": "policies",
        "schema": "schema",
        "sms_authentication_message": "smsAuthenticationMessage",
        "sms_configuration": "smsConfiguration",
        "sms_verification_message": "smsVerificationMessage",
        "user_attribute_update_settings": "userAttributeUpdateSettings",
        "username_attributes": "usernameAttributes",
        "username_configuration": "usernameConfiguration",
        "user_pool_add_ons": "userPoolAddOns",
        "user_pool_name": "userPoolName",
        "user_pool_tags": "userPoolTags",
        "user_pool_tier": "userPoolTier",
        "verification_message_template": "verificationMessageTemplate",
        "web_authn_relying_party_id": "webAuthnRelyingPartyId",
        "web_authn_user_verification": "webAuthnUserVerification",
    },
)
class CfnUserPoolProps:
    def __init__(
        self,
        *,
        account_recovery_setting: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.AccountRecoverySettingProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        admin_create_user_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.AdminCreateUserConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        alias_attributes: typing.Optional[typing.Sequence[builtins.str]] = None,
        auto_verified_attributes: typing.Optional[typing.Sequence[builtins.str]] = None,
        deletion_protection: typing.Optional[builtins.str] = None,
        device_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.DeviceConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        email_authentication_message: typing.Optional[builtins.str] = None,
        email_authentication_subject: typing.Optional[builtins.str] = None,
        email_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.EmailConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        email_verification_message: typing.Optional[builtins.str] = None,
        email_verification_subject: typing.Optional[builtins.str] = None,
        enabled_mfas: typing.Optional[typing.Sequence[builtins.str]] = None,
        lambda_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.LambdaConfigProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        mfa_configuration: typing.Optional[builtins.str] = None,
        policies: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.PoliciesProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        schema: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.SchemaAttributeProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        sms_authentication_message: typing.Optional[builtins.str] = None,
        sms_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.SmsConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        sms_verification_message: typing.Optional[builtins.str] = None,
        user_attribute_update_settings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.UserAttributeUpdateSettingsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        username_attributes: typing.Optional[typing.Sequence[builtins.str]] = None,
        username_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.UsernameConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        user_pool_add_ons: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.UserPoolAddOnsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        user_pool_name: typing.Optional[builtins.str] = None,
        user_pool_tags: typing.Any = None,
        user_pool_tier: typing.Optional[builtins.str] = None,
        verification_message_template: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPool.VerificationMessageTemplateProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        web_authn_relying_party_id: typing.Optional[builtins.str] = None,
        web_authn_user_verification: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnUserPool``.

        :param account_recovery_setting: The available verified method a user can use to recover their password when they call ``ForgotPassword`` . You can use this setting to define a preferred method when a user has more than one method available. With this setting, SMS doesn't qualify for a valid password recovery mechanism if the user also has SMS multi-factor authentication (MFA) activated. In the absence of this setting, Amazon Cognito uses the legacy behavior to determine the recovery method where SMS is preferred through email.
        :param admin_create_user_config: The settings for administrator creation of users in a user pool. Contains settings for allowing user sign-up, customizing invitation messages to new users, and the amount of time before temporary passwords expire.
        :param alias_attributes: Attributes supported as an alias for this user pool. For more information about alias attributes, see `Customizing sign-in attributes <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases>`_ .
        :param auto_verified_attributes: The attributes that you want your user pool to automatically verify. For more information, see `Verifying contact information at sign-up <https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html#allowing-users-to-sign-up-and-confirm-themselves>`_ .
        :param deletion_protection: When active, ``DeletionProtection`` prevents accidental deletion of your user pool. Before you can delete a user pool that you have protected against deletion, you must deactivate this feature. When you try to delete a protected user pool in a ``DeleteUserPool`` API request, Amazon Cognito returns an ``InvalidParameterException`` error. To delete a protected user pool, send a new ``DeleteUserPool`` request after you deactivate deletion protection in an ``UpdateUserPool`` API request.
        :param device_configuration: The device-remembering configuration for a user pool. Device remembering or device tracking is a "Remember me on this device" option for user pools that perform authentication with the device key of a trusted device in the back end, instead of a user-provided MFA code. For more information about device authentication, see `Working with user devices in your user pool <https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html>`_ . A null value indicates that you have deactivated device remembering in your user pool. .. epigraph:: When you provide a value for any ``DeviceConfiguration`` field, you activate the Amazon Cognito device-remembering feature. For more information, see `Working with devices <https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html>`_ .
        :param email_authentication_message: 
        :param email_authentication_subject: 
        :param email_configuration: The email configuration of your user pool. The email configuration type sets your preferred sending method, AWS Region, and sender for messages from your user pool.
        :param email_verification_message: This parameter is no longer used. See `VerificationMessageTemplateType <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html>`_ .
        :param email_verification_subject: This parameter is no longer used. See `VerificationMessageTemplateType <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html>`_ .
        :param enabled_mfas: Set enabled MFA options on a specified user pool. To disable all MFAs after it has been enabled, set ``MfaConfiguration`` to ``OFF`` and remove EnabledMfas. MFAs can only be all disabled if ``MfaConfiguration`` is ``OFF`` . After you enable ``SMS_MFA`` , you can only disable it by setting ``MfaConfiguration`` to ``OFF`` . Can be one of the following values: - ``SMS_MFA`` - Enables MFA with SMS for the user pool. To select this option, you must also provide values for ``SmsConfiguration`` . - ``SOFTWARE_TOKEN_MFA`` - Enables software token MFA for the user pool. - ``EMAIL_OTP`` - Enables MFA with email for the user pool. To select this option, you must provide values for ``EmailConfiguration`` and within those, set ``EmailSendingAccount`` to ``DEVELOPER`` . Allowed values: ``SMS_MFA`` | ``SOFTWARE_TOKEN_MFA`` | ``EMAIL_OTP``
        :param lambda_config: A collection of user pool Lambda triggers. Amazon Cognito invokes triggers at several possible stages of authentication operations. Triggers can modify the outcome of the operations that invoked them.
        :param mfa_configuration: Displays the state of multi-factor authentication (MFA) as on, off, or optional. When ``ON`` , all users must set up MFA before they can sign in. When ``OPTIONAL`` , your application must make a client-side determination of whether a user wants to register an MFA device. For user pools with adaptive authentication with threat protection, choose ``OPTIONAL`` . When ``MfaConfiguration`` is ``OPTIONAL`` , managed login doesn't automatically prompt users to set up MFA. Amazon Cognito generates MFA prompts in API responses and in managed login for users who have chosen and configured a preferred MFA factor.
        :param policies: A list of user pool policies. Contains the policy that sets password-complexity requirements.
        :param schema: An array of attributes for the new user pool. You can add custom attributes and modify the properties of default attributes. The specifications in this parameter set the required attributes in your user pool. For more information, see `Working with user attributes <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html>`_ .
        :param sms_authentication_message: The contents of the SMS authentication message.
        :param sms_configuration: The settings for your Amazon Cognito user pool to send SMS messages with Amazon Simple Notification Service. To send SMS messages with Amazon SNS in the AWS Region that you want, the Amazon Cognito user pool uses an AWS Identity and Access Management (IAM) role in your AWS account . For more information see `SMS message settings <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-sms-settings.html>`_ .
        :param sms_verification_message: This parameter is no longer used. See `VerificationMessageTemplateType <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html>`_ .
        :param user_attribute_update_settings: The settings for updates to user attributes. These settings include the property ``AttributesRequireVerificationBeforeUpdate`` , a user-pool setting that tells Amazon Cognito how to handle changes to the value of your users' email address and phone number attributes. For more information, see `Verifying updates to email addresses and phone numbers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html#user-pool-settings-verifications-verify-attribute-updates>`_ .
        :param username_attributes: Specifies whether a user can use an email address or phone number as a username when they sign up.
        :param username_configuration: Sets the case sensitivity option for sign-in usernames. When ``CaseSensitive`` is ``false`` (case insensitive), users can sign in with any combination of capital and lowercase letters. For example, ``username`` , ``USERNAME`` , or ``UserName`` , or for email, ``email@example.com`` or ``EMaiL@eXamplE.Com`` . For most use cases, set case sensitivity to ``false`` as a best practice. When usernames and email addresses are case insensitive, Amazon Cognito treats any variation in case as the same user, and prevents a case variation from being assigned to the same attribute for a different user. When ``CaseSensitive`` is ``true`` (case sensitive), Amazon Cognito interprets ``USERNAME`` and ``UserName`` as distinct users. This configuration is immutable after you set it.
        :param user_pool_add_ons: Contains settings for activation of threat protection, including the operating mode and additional authentication types. To log user security information but take no action, set to ``AUDIT`` . To configure automatic security responses to potentially unwanted traffic to your user pool, set to ``ENFORCED`` . For more information, see `Adding advanced security to a user pool <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-advanced-security.html>`_ . To activate this setting, your user pool must be on the `Plus tier <https://docs.aws.amazon.com/cognito/latest/developerguide/feature-plans-features-plus.html>`_ .
        :param user_pool_name: A friendly name for your user pool.
        :param user_pool_tags: The tag keys and values to assign to the user pool. A tag is a label that you can use to categorize and manage user pools in different ways, such as by purpose, owner, environment, or other criteria.
        :param user_pool_tier: The user pool `feature plan <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-sign-in-feature-plans.html>`_ , or tier. This parameter determines the eligibility of the user pool for features like managed login, access-token customization, and threat protection. Defaults to ``ESSENTIALS`` .
        :param verification_message_template: The template for the verification message that your user pool delivers to users who set an email address or phone number attribute. Set the email message type that corresponds to your ``DefaultEmailOption`` selection. For ``CONFIRM_WITH_LINK`` , specify an ``EmailMessageByLink`` and leave ``EmailMessage`` blank. For ``CONFIRM_WITH_CODE`` , specify an ``EmailMessage`` and leave ``EmailMessageByLink`` blank. When you supply both parameters with either choice, Amazon Cognito returns an error.
        :param web_authn_relying_party_id: Sets or displays the authentication domain, typically your user pool domain, that passkey providers must use as a relying party (RP) in their configuration. Under the following conditions, the passkey relying party ID must be the fully-qualified domain name of your custom domain: - The user pool is configured for passkey authentication. - The user pool has a custom domain, whether or not it also has a prefix domain. - Your application performs authentication with managed login or the classic hosted UI.
        :param web_authn_user_verification: When ``required`` , users can only register and sign in users with passkeys that are capable of `user verification <https://docs.aws.amazon.com/https://www.w3.org/TR/webauthn-2/#enum-userVerificationRequirement>`_ . When ``preferred`` , your user pool doesn't require the use of authenticators with user verification but encourages it.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            # user_pool_tags: Any
            
            cfn_user_pool_props = cognito.CfnUserPoolProps(
                account_recovery_setting=cognito.CfnUserPool.AccountRecoverySettingProperty(
                    recovery_mechanisms=[cognito.CfnUserPool.RecoveryOptionProperty(
                        name="name",
                        priority=123
                    )]
                ),
                admin_create_user_config=cognito.CfnUserPool.AdminCreateUserConfigProperty(
                    allow_admin_create_user_only=False,
                    invite_message_template=cognito.CfnUserPool.InviteMessageTemplateProperty(
                        email_message="emailMessage",
                        email_subject="emailSubject",
                        sms_message="smsMessage"
                    ),
                    unused_account_validity_days=123
                ),
                alias_attributes=["aliasAttributes"],
                auto_verified_attributes=["autoVerifiedAttributes"],
                deletion_protection="deletionProtection",
                device_configuration=cognito.CfnUserPool.DeviceConfigurationProperty(
                    challenge_required_on_new_device=False,
                    device_only_remembered_on_user_prompt=False
                ),
                email_authentication_message="emailAuthenticationMessage",
                email_authentication_subject="emailAuthenticationSubject",
                email_configuration=cognito.CfnUserPool.EmailConfigurationProperty(
                    configuration_set="configurationSet",
                    email_sending_account="emailSendingAccount",
                    from="from",
                    reply_to_email_address="replyToEmailAddress",
                    source_arn="sourceArn"
                ),
                email_verification_message="emailVerificationMessage",
                email_verification_subject="emailVerificationSubject",
                enabled_mfas=["enabledMfas"],
                lambda_config=cognito.CfnUserPool.LambdaConfigProperty(
                    create_auth_challenge="createAuthChallenge",
                    custom_email_sender=cognito.CfnUserPool.CustomEmailSenderProperty(
                        lambda_arn="lambdaArn",
                        lambda_version="lambdaVersion"
                    ),
                    custom_message="customMessage",
                    custom_sms_sender=cognito.CfnUserPool.CustomSMSSenderProperty(
                        lambda_arn="lambdaArn",
                        lambda_version="lambdaVersion"
                    ),
                    define_auth_challenge="defineAuthChallenge",
                    kms_key_id="kmsKeyId",
                    post_authentication="postAuthentication",
                    post_confirmation="postConfirmation",
                    pre_authentication="preAuthentication",
                    pre_sign_up="preSignUp",
                    pre_token_generation="preTokenGeneration",
                    pre_token_generation_config=cognito.CfnUserPool.PreTokenGenerationConfigProperty(
                        lambda_arn="lambdaArn",
                        lambda_version="lambdaVersion"
                    ),
                    user_migration="userMigration",
                    verify_auth_challenge_response="verifyAuthChallengeResponse"
                ),
                mfa_configuration="mfaConfiguration",
                policies=cognito.CfnUserPool.PoliciesProperty(
                    password_policy=cognito.CfnUserPool.PasswordPolicyProperty(
                        minimum_length=123,
                        password_history_size=123,
                        require_lowercase=False,
                        require_numbers=False,
                        require_symbols=False,
                        require_uppercase=False,
                        temporary_password_validity_days=123
                    ),
                    sign_in_policy=cognito.CfnUserPool.SignInPolicyProperty(
                        allowed_first_auth_factors=["allowedFirstAuthFactors"]
                    )
                ),
                schema=[cognito.CfnUserPool.SchemaAttributeProperty(
                    attribute_data_type="attributeDataType",
                    developer_only_attribute=False,
                    mutable=False,
                    name="name",
                    number_attribute_constraints=cognito.CfnUserPool.NumberAttributeConstraintsProperty(
                        max_value="maxValue",
                        min_value="minValue"
                    ),
                    required=False,
                    string_attribute_constraints=cognito.CfnUserPool.StringAttributeConstraintsProperty(
                        max_length="maxLength",
                        min_length="minLength"
                    )
                )],
                sms_authentication_message="smsAuthenticationMessage",
                sms_configuration=cognito.CfnUserPool.SmsConfigurationProperty(
                    external_id="externalId",
                    sns_caller_arn="snsCallerArn",
                    sns_region="snsRegion"
                ),
                sms_verification_message="smsVerificationMessage",
                user_attribute_update_settings=cognito.CfnUserPool.UserAttributeUpdateSettingsProperty(
                    attributes_require_verification_before_update=["attributesRequireVerificationBeforeUpdate"]
                ),
                username_attributes=["usernameAttributes"],
                username_configuration=cognito.CfnUserPool.UsernameConfigurationProperty(
                    case_sensitive=False
                ),
                user_pool_add_ons=cognito.CfnUserPool.UserPoolAddOnsProperty(
                    advanced_security_additional_flows=cognito.CfnUserPool.AdvancedSecurityAdditionalFlowsProperty(
                        custom_auth_mode="customAuthMode"
                    ),
                    advanced_security_mode="advancedSecurityMode"
                ),
                user_pool_name="userPoolName",
                user_pool_tags=user_pool_tags,
                user_pool_tier="userPoolTier",
                verification_message_template=cognito.CfnUserPool.VerificationMessageTemplateProperty(
                    default_email_option="defaultEmailOption",
                    email_message="emailMessage",
                    email_message_by_link="emailMessageByLink",
                    email_subject="emailSubject",
                    email_subject_by_link="emailSubjectByLink",
                    sms_message="smsMessage"
                ),
                web_authn_relying_party_id="webAuthnRelyingPartyId",
                web_authn_user_verification="webAuthnUserVerification"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__00bbdbd31eb8d7342ce9883d0851b853acf61f6b243c0aa4323c025da52de96d)
            check_type(argname="argument account_recovery_setting", value=account_recovery_setting, expected_type=type_hints["account_recovery_setting"])
            check_type(argname="argument admin_create_user_config", value=admin_create_user_config, expected_type=type_hints["admin_create_user_config"])
            check_type(argname="argument alias_attributes", value=alias_attributes, expected_type=type_hints["alias_attributes"])
            check_type(argname="argument auto_verified_attributes", value=auto_verified_attributes, expected_type=type_hints["auto_verified_attributes"])
            check_type(argname="argument deletion_protection", value=deletion_protection, expected_type=type_hints["deletion_protection"])
            check_type(argname="argument device_configuration", value=device_configuration, expected_type=type_hints["device_configuration"])
            check_type(argname="argument email_authentication_message", value=email_authentication_message, expected_type=type_hints["email_authentication_message"])
            check_type(argname="argument email_authentication_subject", value=email_authentication_subject, expected_type=type_hints["email_authentication_subject"])
            check_type(argname="argument email_configuration", value=email_configuration, expected_type=type_hints["email_configuration"])
            check_type(argname="argument email_verification_message", value=email_verification_message, expected_type=type_hints["email_verification_message"])
            check_type(argname="argument email_verification_subject", value=email_verification_subject, expected_type=type_hints["email_verification_subject"])
            check_type(argname="argument enabled_mfas", value=enabled_mfas, expected_type=type_hints["enabled_mfas"])
            check_type(argname="argument lambda_config", value=lambda_config, expected_type=type_hints["lambda_config"])
            check_type(argname="argument mfa_configuration", value=mfa_configuration, expected_type=type_hints["mfa_configuration"])
            check_type(argname="argument policies", value=policies, expected_type=type_hints["policies"])
            check_type(argname="argument schema", value=schema, expected_type=type_hints["schema"])
            check_type(argname="argument sms_authentication_message", value=sms_authentication_message, expected_type=type_hints["sms_authentication_message"])
            check_type(argname="argument sms_configuration", value=sms_configuration, expected_type=type_hints["sms_configuration"])
            check_type(argname="argument sms_verification_message", value=sms_verification_message, expected_type=type_hints["sms_verification_message"])
            check_type(argname="argument user_attribute_update_settings", value=user_attribute_update_settings, expected_type=type_hints["user_attribute_update_settings"])
            check_type(argname="argument username_attributes", value=username_attributes, expected_type=type_hints["username_attributes"])
            check_type(argname="argument username_configuration", value=username_configuration, expected_type=type_hints["username_configuration"])
            check_type(argname="argument user_pool_add_ons", value=user_pool_add_ons, expected_type=type_hints["user_pool_add_ons"])
            check_type(argname="argument user_pool_name", value=user_pool_name, expected_type=type_hints["user_pool_name"])
            check_type(argname="argument user_pool_tags", value=user_pool_tags, expected_type=type_hints["user_pool_tags"])
            check_type(argname="argument user_pool_tier", value=user_pool_tier, expected_type=type_hints["user_pool_tier"])
            check_type(argname="argument verification_message_template", value=verification_message_template, expected_type=type_hints["verification_message_template"])
            check_type(argname="argument web_authn_relying_party_id", value=web_authn_relying_party_id, expected_type=type_hints["web_authn_relying_party_id"])
            check_type(argname="argument web_authn_user_verification", value=web_authn_user_verification, expected_type=type_hints["web_authn_user_verification"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if account_recovery_setting is not None:
            self._values["account_recovery_setting"] = account_recovery_setting
        if admin_create_user_config is not None:
            self._values["admin_create_user_config"] = admin_create_user_config
        if alias_attributes is not None:
            self._values["alias_attributes"] = alias_attributes
        if auto_verified_attributes is not None:
            self._values["auto_verified_attributes"] = auto_verified_attributes
        if deletion_protection is not None:
            self._values["deletion_protection"] = deletion_protection
        if device_configuration is not None:
            self._values["device_configuration"] = device_configuration
        if email_authentication_message is not None:
            self._values["email_authentication_message"] = email_authentication_message
        if email_authentication_subject is not None:
            self._values["email_authentication_subject"] = email_authentication_subject
        if email_configuration is not None:
            self._values["email_configuration"] = email_configuration
        if email_verification_message is not None:
            self._values["email_verification_message"] = email_verification_message
        if email_verification_subject is not None:
            self._values["email_verification_subject"] = email_verification_subject
        if enabled_mfas is not None:
            self._values["enabled_mfas"] = enabled_mfas
        if lambda_config is not None:
            self._values["lambda_config"] = lambda_config
        if mfa_configuration is not None:
            self._values["mfa_configuration"] = mfa_configuration
        if policies is not None:
            self._values["policies"] = policies
        if schema is not None:
            self._values["schema"] = schema
        if sms_authentication_message is not None:
            self._values["sms_authentication_message"] = sms_authentication_message
        if sms_configuration is not None:
            self._values["sms_configuration"] = sms_configuration
        if sms_verification_message is not None:
            self._values["sms_verification_message"] = sms_verification_message
        if user_attribute_update_settings is not None:
            self._values["user_attribute_update_settings"] = user_attribute_update_settings
        if username_attributes is not None:
            self._values["username_attributes"] = username_attributes
        if username_configuration is not None:
            self._values["username_configuration"] = username_configuration
        if user_pool_add_ons is not None:
            self._values["user_pool_add_ons"] = user_pool_add_ons
        if user_pool_name is not None:
            self._values["user_pool_name"] = user_pool_name
        if user_pool_tags is not None:
            self._values["user_pool_tags"] = user_pool_tags
        if user_pool_tier is not None:
            self._values["user_pool_tier"] = user_pool_tier
        if verification_message_template is not None:
            self._values["verification_message_template"] = verification_message_template
        if web_authn_relying_party_id is not None:
            self._values["web_authn_relying_party_id"] = web_authn_relying_party_id
        if web_authn_user_verification is not None:
            self._values["web_authn_user_verification"] = web_authn_user_verification

    @builtins.property
    def account_recovery_setting(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.AccountRecoverySettingProperty]]:
        '''The available verified method a user can use to recover their password when they call ``ForgotPassword`` .

        You can use this setting to define a preferred method when a user has more than one method available. With this setting, SMS doesn't qualify for a valid password recovery mechanism if the user also has SMS multi-factor authentication (MFA) activated. In the absence of this setting, Amazon Cognito uses the legacy behavior to determine the recovery method where SMS is preferred through email.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-accountrecoverysetting
        '''
        result = self._values.get("account_recovery_setting")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.AccountRecoverySettingProperty]], result)

    @builtins.property
    def admin_create_user_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.AdminCreateUserConfigProperty]]:
        '''The settings for administrator creation of users in a user pool.

        Contains settings for allowing user sign-up, customizing invitation messages to new users, and the amount of time before temporary passwords expire.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-admincreateuserconfig
        '''
        result = self._values.get("admin_create_user_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.AdminCreateUserConfigProperty]], result)

    @builtins.property
    def alias_attributes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Attributes supported as an alias for this user pool.

        For more information about alias attributes, see `Customizing sign-in attributes <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-aliasattributes
        '''
        result = self._values.get("alias_attributes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def auto_verified_attributes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The attributes that you want your user pool to automatically verify.

        For more information, see `Verifying contact information at sign-up <https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html#allowing-users-to-sign-up-and-confirm-themselves>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-autoverifiedattributes
        '''
        result = self._values.get("auto_verified_attributes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def deletion_protection(self) -> typing.Optional[builtins.str]:
        '''When active, ``DeletionProtection`` prevents accidental deletion of your user pool.

        Before you can delete a user pool that you have protected against deletion, you
        must deactivate this feature.

        When you try to delete a protected user pool in a ``DeleteUserPool`` API request, Amazon Cognito returns an ``InvalidParameterException`` error. To delete a protected user pool, send a new ``DeleteUserPool`` request after you deactivate deletion protection in an ``UpdateUserPool`` API request.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-deletionprotection
        '''
        result = self._values.get("deletion_protection")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def device_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.DeviceConfigurationProperty]]:
        '''The device-remembering configuration for a user pool.

        Device remembering or device tracking is a "Remember me on this device" option for user pools that perform authentication with the device key of a trusted device in the back end, instead of a user-provided MFA code. For more information about device authentication, see `Working with user devices in your user pool <https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html>`_ . A null value indicates that you have deactivated device remembering in your user pool.
        .. epigraph::

           When you provide a value for any ``DeviceConfiguration`` field, you activate the Amazon Cognito device-remembering feature. For more information, see `Working with devices <https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-deviceconfiguration
        '''
        result = self._values.get("device_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.DeviceConfigurationProperty]], result)

    @builtins.property
    def email_authentication_message(self) -> typing.Optional[builtins.str]:
        '''
        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-emailauthenticationmessage
        '''
        result = self._values.get("email_authentication_message")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def email_authentication_subject(self) -> typing.Optional[builtins.str]:
        '''
        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-emailauthenticationsubject
        '''
        result = self._values.get("email_authentication_subject")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def email_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.EmailConfigurationProperty]]:
        '''The email configuration of your user pool.

        The email configuration type sets your preferred sending method, AWS Region, and sender for messages from your user pool.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-emailconfiguration
        '''
        result = self._values.get("email_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.EmailConfigurationProperty]], result)

    @builtins.property
    def email_verification_message(self) -> typing.Optional[builtins.str]:
        '''This parameter is no longer used.

        See `VerificationMessageTemplateType <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-emailverificationmessage
        '''
        result = self._values.get("email_verification_message")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def email_verification_subject(self) -> typing.Optional[builtins.str]:
        '''This parameter is no longer used.

        See `VerificationMessageTemplateType <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-emailverificationsubject
        '''
        result = self._values.get("email_verification_subject")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enabled_mfas(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Set enabled MFA options on a specified user pool.

        To disable all MFAs after it has been enabled, set ``MfaConfiguration`` to ``OFF`` and remove EnabledMfas. MFAs can only be all disabled if ``MfaConfiguration`` is ``OFF`` . After you enable ``SMS_MFA`` , you can only disable it by setting ``MfaConfiguration`` to ``OFF`` . Can be one of the following values:

        - ``SMS_MFA`` - Enables MFA with SMS for the user pool. To select this option, you must also provide values for ``SmsConfiguration`` .
        - ``SOFTWARE_TOKEN_MFA`` - Enables software token MFA for the user pool.
        - ``EMAIL_OTP`` - Enables MFA with email for the user pool. To select this option, you must provide values for ``EmailConfiguration`` and within those, set ``EmailSendingAccount`` to ``DEVELOPER`` .

        Allowed values: ``SMS_MFA`` | ``SOFTWARE_TOKEN_MFA`` | ``EMAIL_OTP``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-enabledmfas
        '''
        result = self._values.get("enabled_mfas")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def lambda_config(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.LambdaConfigProperty]]:
        '''A collection of user pool Lambda triggers.

        Amazon Cognito invokes triggers at several possible stages of authentication operations. Triggers can modify the outcome of the operations that invoked them.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-lambdaconfig
        '''
        result = self._values.get("lambda_config")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.LambdaConfigProperty]], result)

    @builtins.property
    def mfa_configuration(self) -> typing.Optional[builtins.str]:
        '''Displays the state of multi-factor authentication (MFA) as on, off, or optional.

        When ``ON`` , all users must set up MFA before they can sign in. When ``OPTIONAL`` , your application must make a client-side determination of whether a user wants to register an MFA device. For user pools with adaptive authentication with threat protection, choose ``OPTIONAL`` .

        When ``MfaConfiguration`` is ``OPTIONAL`` , managed login doesn't automatically prompt users to set up MFA. Amazon Cognito generates MFA prompts in API responses and in managed login for users who have chosen and configured a preferred MFA factor.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-mfaconfiguration
        '''
        result = self._values.get("mfa_configuration")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def policies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.PoliciesProperty]]:
        '''A list of user pool policies.

        Contains the policy that sets password-complexity requirements.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-policies
        '''
        result = self._values.get("policies")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.PoliciesProperty]], result)

    @builtins.property
    def schema(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnUserPool.SchemaAttributeProperty]]]]:
        '''An array of attributes for the new user pool.

        You can add custom attributes and modify the properties of default attributes. The specifications in this parameter set the required attributes in your user pool. For more information, see `Working with user attributes <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-schema
        '''
        result = self._values.get("schema")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnUserPool.SchemaAttributeProperty]]]], result)

    @builtins.property
    def sms_authentication_message(self) -> typing.Optional[builtins.str]:
        '''The contents of the SMS authentication message.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-smsauthenticationmessage
        '''
        result = self._values.get("sms_authentication_message")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def sms_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.SmsConfigurationProperty]]:
        '''The settings for your Amazon Cognito user pool to send SMS messages with Amazon Simple Notification Service.

        To send SMS messages with Amazon SNS in the AWS Region that you want, the Amazon Cognito user pool uses an AWS Identity and Access Management (IAM) role in your AWS account . For more information see `SMS message settings <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-sms-settings.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-smsconfiguration
        '''
        result = self._values.get("sms_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.SmsConfigurationProperty]], result)

    @builtins.property
    def sms_verification_message(self) -> typing.Optional[builtins.str]:
        '''This parameter is no longer used.

        See `VerificationMessageTemplateType <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-verificationmessagetemplate.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-smsverificationmessage
        '''
        result = self._values.get("sms_verification_message")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def user_attribute_update_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.UserAttributeUpdateSettingsProperty]]:
        '''The settings for updates to user attributes.

        These settings include the property ``AttributesRequireVerificationBeforeUpdate`` ,
        a user-pool setting that tells Amazon Cognito how to handle changes to the value of your users' email address and phone number attributes. For
        more information, see `Verifying updates to email addresses and phone numbers <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html#user-pool-settings-verifications-verify-attribute-updates>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-userattributeupdatesettings
        '''
        result = self._values.get("user_attribute_update_settings")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.UserAttributeUpdateSettingsProperty]], result)

    @builtins.property
    def username_attributes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specifies whether a user can use an email address or phone number as a username when they sign up.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-usernameattributes
        '''
        result = self._values.get("username_attributes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def username_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.UsernameConfigurationProperty]]:
        '''Sets the case sensitivity option for sign-in usernames.

        When ``CaseSensitive`` is ``false`` (case insensitive), users can sign in with any combination of capital and lowercase letters. For example, ``username`` , ``USERNAME`` , or ``UserName`` , or for email, ``email@example.com`` or ``EMaiL@eXamplE.Com`` . For most use cases, set case sensitivity to ``false`` as a best practice. When usernames and email addresses are case insensitive, Amazon Cognito treats any variation in case as the same user, and prevents a case variation from being assigned to the same attribute for a different user.

        When ``CaseSensitive`` is ``true`` (case sensitive), Amazon Cognito interprets ``USERNAME`` and ``UserName`` as distinct users.

        This configuration is immutable after you set it.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-usernameconfiguration
        '''
        result = self._values.get("username_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.UsernameConfigurationProperty]], result)

    @builtins.property
    def user_pool_add_ons(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.UserPoolAddOnsProperty]]:
        '''Contains settings for activation of threat protection, including the operating mode and additional authentication types.

        To log user security information but take no action, set to ``AUDIT`` . To configure automatic security responses to potentially unwanted traffic to your user pool, set to ``ENFORCED`` .

        For more information, see `Adding advanced security to a user pool <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-advanced-security.html>`_ . To activate this setting, your user pool must be on the `Plus tier <https://docs.aws.amazon.com/cognito/latest/developerguide/feature-plans-features-plus.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-userpooladdons
        '''
        result = self._values.get("user_pool_add_ons")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.UserPoolAddOnsProperty]], result)

    @builtins.property
    def user_pool_name(self) -> typing.Optional[builtins.str]:
        '''A friendly name for your user pool.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-userpoolname
        '''
        result = self._values.get("user_pool_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def user_pool_tags(self) -> typing.Any:
        '''The tag keys and values to assign to the user pool.

        A tag is a label that you can use to categorize and manage user pools in different ways, such as by purpose, owner, environment, or other criteria.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-userpooltags
        '''
        result = self._values.get("user_pool_tags")
        return typing.cast(typing.Any, result)

    @builtins.property
    def user_pool_tier(self) -> typing.Optional[builtins.str]:
        '''The user pool `feature plan <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-sign-in-feature-plans.html>`_ , or tier. This parameter determines the eligibility of the user pool for features like managed login, access-token customization, and threat protection. Defaults to ``ESSENTIALS`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-userpooltier
        '''
        result = self._values.get("user_pool_tier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def verification_message_template(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.VerificationMessageTemplateProperty]]:
        '''The template for the verification message that your user pool delivers to users who set an email address or phone number attribute.

        Set the email message type that corresponds to your ``DefaultEmailOption`` selection. For ``CONFIRM_WITH_LINK`` , specify an ``EmailMessageByLink`` and leave ``EmailMessage`` blank. For ``CONFIRM_WITH_CODE`` , specify an ``EmailMessage`` and leave ``EmailMessageByLink`` blank. When you supply both parameters with either choice, Amazon Cognito returns an error.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-verificationmessagetemplate
        '''
        result = self._values.get("verification_message_template")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPool.VerificationMessageTemplateProperty]], result)

    @builtins.property
    def web_authn_relying_party_id(self) -> typing.Optional[builtins.str]:
        '''Sets or displays the authentication domain, typically your user pool domain, that passkey providers must use as a relying party (RP) in their configuration.

        Under the following conditions, the passkey relying party ID must be the fully-qualified domain name of your custom domain:

        - The user pool is configured for passkey authentication.
        - The user pool has a custom domain, whether or not it also has a prefix domain.
        - Your application performs authentication with managed login or the classic hosted UI.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-webauthnrelyingpartyid
        '''
        result = self._values.get("web_authn_relying_party_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def web_authn_user_verification(self) -> typing.Optional[builtins.str]:
        '''When ``required`` , users can only register and sign in users with passkeys that are capable of `user verification <https://docs.aws.amazon.com/https://www.w3.org/TR/webauthn-2/#enum-userVerificationRequirement>`_ . When ``preferred`` , your user pool doesn't require the use of authenticators with user verification but encourages it.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpool.html#cfn-cognito-userpool-webauthnuserverification
        '''
        result = self._values.get("web_authn_user_verification")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUserPoolProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnUserPoolResourceServer(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolResourceServer",
):
    '''The ``AWS::Cognito::UserPoolResourceServer`` resource creates a new OAuth2.0 resource server and defines custom scopes in it.

    .. epigraph::

       If you don't specify a value for a parameter, Amazon Cognito sets it to a default value.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolresourceserver.html
    :cloudformationResource: AWS::Cognito::UserPoolResourceServer
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        cfn_user_pool_resource_server = cognito.CfnUserPoolResourceServer(self, "MyCfnUserPoolResourceServer",
            identifier="identifier",
            name="name",
            user_pool_id="userPoolId",
        
            # the properties below are optional
            scopes=[cognito.CfnUserPoolResourceServer.ResourceServerScopeTypeProperty(
                scope_description="scopeDescription",
                scope_name="scopeName"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        identifier: builtins.str,
        name: builtins.str,
        user_pool_id: builtins.str,
        scopes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolResourceServer.ResourceServerScopeTypeProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param identifier: A unique resource server identifier for the resource server. The identifier can be an API friendly name like ``solar-system-data`` . You can also set an API URL like ``https://solar-system-data-api.example.com`` as your identifier. Amazon Cognito represents scopes in the access token in the format ``$resource-server-identifier/$scope`` . Longer scope-identifier strings increase the size of your access tokens.
        :param name: A friendly name for the resource server.
        :param user_pool_id: The ID of the user pool where you want to create a resource server.
        :param scopes: A list of scopes. Each scope is a map with keys ``ScopeName`` and ``ScopeDescription`` .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5ff380728a32f62434c91f2ef979d9f2a64ce347253fb0640fb5f618be47fd30)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUserPoolResourceServerProps(
            identifier=identifier, name=name, user_pool_id=user_pool_id, scopes=scopes
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e2a96db567ceac3c2f12fc90cf5f16fbfc45a32b92f0b1c83858321911b0ed93)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2b95c9d133fc1dc564adc0c1075c48bf5b8fc600c68e782923bd94e58be1bd65)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="identifier")
    def identifier(self) -> builtins.str:
        '''A unique resource server identifier for the resource server.'''
        return typing.cast(builtins.str, jsii.get(self, "identifier"))

    @identifier.setter
    def identifier(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0784cedf1608a8758f1c44b7e3aacb96f3672ac970fc80673ae8a281382775da)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identifier", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''A friendly name for the resource server.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__091a350381e1228e09795024704316d40f0bce3ead6bff97b512b103b8bb1ee6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you want to create a resource server.'''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @user_pool_id.setter
    def user_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__050918e589a6f0ea5348f6c36799b16672c9d8127f2e78e25499ac0ee092b4fa)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="scopes")
    def scopes(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPoolResourceServer.ResourceServerScopeTypeProperty"]]]]:
        '''A list of scopes.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPoolResourceServer.ResourceServerScopeTypeProperty"]]]], jsii.get(self, "scopes"))

    @scopes.setter
    def scopes(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPoolResourceServer.ResourceServerScopeTypeProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__160fcb672d8fdc3b92b02edf7b6742270c60fcaadc1b0091d06e1413b15e3583)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "scopes", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolResourceServer.ResourceServerScopeTypeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "scope_description": "scopeDescription",
            "scope_name": "scopeName",
        },
    )
    class ResourceServerScopeTypeProperty:
        def __init__(
            self,
            *,
            scope_description: builtins.str,
            scope_name: builtins.str,
        ) -> None:
            '''One custom scope associated with a user pool resource server.

            This data type is a member of ``ResourceServerScopeType`` . For more information, see `Scopes, M2M, and API authorization with resource servers <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html>`_ .

            :param scope_description: A friendly description of a custom scope.
            :param scope_name: The name of the scope. Amazon Cognito renders custom scopes in the format ``resourceServerIdentifier/ScopeName`` . For example, if this parameter is ``exampleScope`` in the resource server with the identifier ``exampleResourceServer`` , you request and receive the scope ``exampleResourceServer/exampleScope`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolresourceserver-resourceserverscopetype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                resource_server_scope_type_property = cognito.CfnUserPoolResourceServer.ResourceServerScopeTypeProperty(
                    scope_description="scopeDescription",
                    scope_name="scopeName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__69c27b6d57821c89462fb3d9567a88f47d5ec8ef45574e28d0f8404ad66bb732)
                check_type(argname="argument scope_description", value=scope_description, expected_type=type_hints["scope_description"])
                check_type(argname="argument scope_name", value=scope_name, expected_type=type_hints["scope_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "scope_description": scope_description,
                "scope_name": scope_name,
            }

        @builtins.property
        def scope_description(self) -> builtins.str:
            '''A friendly description of a custom scope.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolresourceserver-resourceserverscopetype.html#cfn-cognito-userpoolresourceserver-resourceserverscopetype-scopedescription
            '''
            result = self._values.get("scope_description")
            assert result is not None, "Required property 'scope_description' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def scope_name(self) -> builtins.str:
            '''The name of the scope.

            Amazon Cognito renders custom scopes in the format ``resourceServerIdentifier/ScopeName`` . For example, if this parameter is ``exampleScope`` in the resource server with the identifier ``exampleResourceServer`` , you request and receive the scope ``exampleResourceServer/exampleScope`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolresourceserver-resourceserverscopetype.html#cfn-cognito-userpoolresourceserver-resourceserverscopetype-scopename
            '''
            result = self._values.get("scope_name")
            assert result is not None, "Required property 'scope_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ResourceServerScopeTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolResourceServerProps",
    jsii_struct_bases=[],
    name_mapping={
        "identifier": "identifier",
        "name": "name",
        "user_pool_id": "userPoolId",
        "scopes": "scopes",
    },
)
class CfnUserPoolResourceServerProps:
    def __init__(
        self,
        *,
        identifier: builtins.str,
        name: builtins.str,
        user_pool_id: builtins.str,
        scopes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPoolResourceServer.ResourceServerScopeTypeProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnUserPoolResourceServer``.

        :param identifier: A unique resource server identifier for the resource server. The identifier can be an API friendly name like ``solar-system-data`` . You can also set an API URL like ``https://solar-system-data-api.example.com`` as your identifier. Amazon Cognito represents scopes in the access token in the format ``$resource-server-identifier/$scope`` . Longer scope-identifier strings increase the size of your access tokens.
        :param name: A friendly name for the resource server.
        :param user_pool_id: The ID of the user pool where you want to create a resource server.
        :param scopes: A list of scopes. Each scope is a map with keys ``ScopeName`` and ``ScopeDescription`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolresourceserver.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            cfn_user_pool_resource_server_props = cognito.CfnUserPoolResourceServerProps(
                identifier="identifier",
                name="name",
                user_pool_id="userPoolId",
            
                # the properties below are optional
                scopes=[cognito.CfnUserPoolResourceServer.ResourceServerScopeTypeProperty(
                    scope_description="scopeDescription",
                    scope_name="scopeName"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e3ec779f0da483496b42d0bd063d70d51362c7b804d031893cdf65153d2094c1)
            check_type(argname="argument identifier", value=identifier, expected_type=type_hints["identifier"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument user_pool_id", value=user_pool_id, expected_type=type_hints["user_pool_id"])
            check_type(argname="argument scopes", value=scopes, expected_type=type_hints["scopes"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "identifier": identifier,
            "name": name,
            "user_pool_id": user_pool_id,
        }
        if scopes is not None:
            self._values["scopes"] = scopes

    @builtins.property
    def identifier(self) -> builtins.str:
        '''A unique resource server identifier for the resource server.

        The identifier can be an API friendly name like ``solar-system-data`` . You can also set an API URL like ``https://solar-system-data-api.example.com`` as your identifier.

        Amazon Cognito represents scopes in the access token in the format ``$resource-server-identifier/$scope`` . Longer scope-identifier strings increase the size of your access tokens.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolresourceserver.html#cfn-cognito-userpoolresourceserver-identifier
        '''
        result = self._values.get("identifier")
        assert result is not None, "Required property 'identifier' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''A friendly name for the resource server.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolresourceserver.html#cfn-cognito-userpoolresourceserver-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you want to create a resource server.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolresourceserver.html#cfn-cognito-userpoolresourceserver-userpoolid
        '''
        result = self._values.get("user_pool_id")
        assert result is not None, "Required property 'user_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def scopes(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnUserPoolResourceServer.ResourceServerScopeTypeProperty]]]]:
        '''A list of scopes.

        Each scope is a map with keys ``ScopeName`` and ``ScopeDescription`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolresourceserver.html#cfn-cognito-userpoolresourceserver-scopes
        '''
        result = self._values.get("scopes")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnUserPoolResourceServer.ResourceServerScopeTypeProperty]]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUserPoolResourceServerProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnUserPoolRiskConfigurationAttachment(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolRiskConfigurationAttachment",
):
    '''The ``AWS::Cognito::UserPoolRiskConfigurationAttachment`` resource sets the risk configuration that is used for Amazon Cognito advanced security features.

    You can specify risk configuration for a single client (with a specific ``clientId`` ) or for all clients (by setting the ``clientId`` to ``ALL`` ). If you specify ``ALL`` , the default configuration is used for every client that has had no risk configuration set previously. If you specify risk configuration for a particular client, it no longer falls back to the ``ALL`` configuration.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolriskconfigurationattachment.html
    :cloudformationResource: AWS::Cognito::UserPoolRiskConfigurationAttachment
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        cfn_user_pool_risk_configuration_attachment = cognito.CfnUserPoolRiskConfigurationAttachment(self, "MyCfnUserPoolRiskConfigurationAttachment",
            client_id="clientId",
            user_pool_id="userPoolId",
        
            # the properties below are optional
            account_takeover_risk_configuration=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverRiskConfigurationTypeProperty(
                actions=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionsTypeProperty(
                    high_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                        event_action="eventAction",
                        notify=False
                    ),
                    low_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                        event_action="eventAction",
                        notify=False
                    ),
                    medium_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                        event_action="eventAction",
                        notify=False
                    )
                ),
        
                # the properties below are optional
                notify_configuration=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyConfigurationTypeProperty(
                    source_arn="sourceArn",
        
                    # the properties below are optional
                    block_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                        subject="subject",
        
                        # the properties below are optional
                        html_body="htmlBody",
                        text_body="textBody"
                    ),
                    from="from",
                    mfa_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                        subject="subject",
        
                        # the properties below are optional
                        html_body="htmlBody",
                        text_body="textBody"
                    ),
                    no_action_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                        subject="subject",
        
                        # the properties below are optional
                        html_body="htmlBody",
                        text_body="textBody"
                    ),
                    reply_to="replyTo"
                )
            ),
            compromised_credentials_risk_configuration=cognito.CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsRiskConfigurationTypeProperty(
                actions=cognito.CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsActionsTypeProperty(
                    event_action="eventAction"
                ),
        
                # the properties below are optional
                event_filter=["eventFilter"]
            ),
            risk_exception_configuration=cognito.CfnUserPoolRiskConfigurationAttachment.RiskExceptionConfigurationTypeProperty(
                blocked_ip_range_list=["blockedIpRangeList"],
                skipped_ip_range_list=["skippedIpRangeList"]
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        client_id: builtins.str,
        user_pool_id: builtins.str,
        account_takeover_risk_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.AccountTakeoverRiskConfigurationTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        compromised_credentials_risk_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsRiskConfigurationTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        risk_exception_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.RiskExceptionConfigurationTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param client_id: The app client where this configuration is applied. When this parameter isn't present, the risk configuration applies to all user pool app clients that don't have client-level settings.
        :param user_pool_id: The ID of the user pool that has the risk configuration applied.
        :param account_takeover_risk_configuration: The settings for automated responses and notification templates for adaptive authentication with threat protection.
        :param compromised_credentials_risk_configuration: Settings for compromised-credentials actions and authentication types with threat protection in full-function ``ENFORCED`` mode.
        :param risk_exception_configuration: Exceptions to the risk evaluation configuration, including always-allow and always-block IP address ranges.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e3245a667ca916eabc555ef843b4a36bfcb47060fa353bd730066a76dcad5a96)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUserPoolRiskConfigurationAttachmentProps(
            client_id=client_id,
            user_pool_id=user_pool_id,
            account_takeover_risk_configuration=account_takeover_risk_configuration,
            compromised_credentials_risk_configuration=compromised_credentials_risk_configuration,
            risk_exception_configuration=risk_exception_configuration,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c679cad8b164d6bad08830c9bfb57b908cf9c5bb391661d8ccf6732b7e486c2a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__707c8b0947348e149e8012f6d888e4b17a607c761ae75b05be2e38be19814abc)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="clientId")
    def client_id(self) -> builtins.str:
        '''The app client where this configuration is applied.'''
        return typing.cast(builtins.str, jsii.get(self, "clientId"))

    @client_id.setter
    def client_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3a5b05e9407391c0557e201e4e58dfac89a06f57c4adc732162c79169e069fc0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool that has the risk configuration applied.'''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @user_pool_id.setter
    def user_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__258e8ccdb5a39874f49d032edd3b21eafc55726e815291b2ba0aeb3d1e5e57f2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="accountTakeoverRiskConfiguration")
    def account_takeover_risk_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.AccountTakeoverRiskConfigurationTypeProperty"]]:
        '''The settings for automated responses and notification templates for adaptive authentication with threat protection.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.AccountTakeoverRiskConfigurationTypeProperty"]], jsii.get(self, "accountTakeoverRiskConfiguration"))

    @account_takeover_risk_configuration.setter
    def account_takeover_risk_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.AccountTakeoverRiskConfigurationTypeProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b17909dd0638cbbc0f1921202089390881098b016d096fd0e7523987a4e30ba9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "accountTakeoverRiskConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="compromisedCredentialsRiskConfiguration")
    def compromised_credentials_risk_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsRiskConfigurationTypeProperty"]]:
        '''Settings for compromised-credentials actions and authentication types with threat protection in full-function ``ENFORCED`` mode.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsRiskConfigurationTypeProperty"]], jsii.get(self, "compromisedCredentialsRiskConfiguration"))

    @compromised_credentials_risk_configuration.setter
    def compromised_credentials_risk_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsRiskConfigurationTypeProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__114c02dca6d5bbc7b9068e92676886981e1fa4247a8584ed917e5391b74f2d2d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "compromisedCredentialsRiskConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="riskExceptionConfiguration")
    def risk_exception_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.RiskExceptionConfigurationTypeProperty"]]:
        '''Exceptions to the risk evaluation configuration, including always-allow and always-block IP address ranges.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.RiskExceptionConfigurationTypeProperty"]], jsii.get(self, "riskExceptionConfiguration"))

    @risk_exception_configuration.setter
    def risk_exception_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.RiskExceptionConfigurationTypeProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2bce41f342258d692515e8eebb6c2745c7105edc270a2bbcfb45f41d2bf1ca46)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "riskExceptionConfiguration", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty",
        jsii_struct_bases=[],
        name_mapping={"event_action": "eventAction", "notify": "notify"},
    )
    class AccountTakeoverActionTypeProperty:
        def __init__(
            self,
            *,
            event_action: builtins.str,
            notify: typing.Union[builtins.bool, _IResolvable_da3f097b],
        ) -> None:
            '''The automated response to a risk level for adaptive authentication in full-function, or ``ENFORCED`` , mode.

            You can assign an action to each risk level that advanced security features evaluates.

            :param event_action: The action to take for the attempted account takeover action for the associated risk level. Valid values are as follows: - ``BLOCK`` : Block the request. - ``MFA_IF_CONFIGURED`` : Present an MFA challenge if possible. MFA is possible if the user pool has active MFA methods that the user can set up. For example, if the user pool only supports SMS message MFA but the user doesn't have a phone number attribute, MFA setup isn't possible. If MFA setup isn't possible, allow the request. - ``MFA_REQUIRED`` : Present an MFA challenge if possible. Block the request if a user hasn't set up MFA. To sign in with required MFA, users must have an email address or phone number attribute, or a registered TOTP factor. - ``NO_ACTION`` : Take no action. Permit sign-in.
            :param notify: Determines whether Amazon Cognito sends a user a notification message when your user pools assesses a user's session at the associated risk level.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-accounttakeoveractiontype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                account_takeover_action_type_property = cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                    event_action="eventAction",
                    notify=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__30abf81ff553ddb63b2198e60e762f3aecf06b8e0f2991afb69ec6002c5483bd)
                check_type(argname="argument event_action", value=event_action, expected_type=type_hints["event_action"])
                check_type(argname="argument notify", value=notify, expected_type=type_hints["notify"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "event_action": event_action,
                "notify": notify,
            }

        @builtins.property
        def event_action(self) -> builtins.str:
            '''The action to take for the attempted account takeover action for the associated risk level.

            Valid values are as follows:

            - ``BLOCK`` : Block the request.
            - ``MFA_IF_CONFIGURED`` : Present an MFA challenge if possible. MFA is possible if the user pool has active MFA methods that the user can set up. For example, if the user pool only supports SMS message MFA but the user doesn't have a phone number attribute, MFA setup isn't possible. If MFA setup isn't possible, allow the request.
            - ``MFA_REQUIRED`` : Present an MFA challenge if possible. Block the request if a user hasn't set up MFA. To sign in with required MFA, users must have an email address or phone number attribute, or a registered TOTP factor.
            - ``NO_ACTION`` : Take no action. Permit sign-in.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-accounttakeoveractiontype.html#cfn-cognito-userpoolriskconfigurationattachment-accounttakeoveractiontype-eventaction
            '''
            result = self._values.get("event_action")
            assert result is not None, "Required property 'event_action' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def notify(self) -> typing.Union[builtins.bool, _IResolvable_da3f097b]:
            '''Determines whether Amazon Cognito sends a user a notification message when your user pools assesses a user's session at the associated risk level.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-accounttakeoveractiontype.html#cfn-cognito-userpoolriskconfigurationattachment-accounttakeoveractiontype-notify
            '''
            result = self._values.get("notify")
            assert result is not None, "Required property 'notify' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_da3f097b], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccountTakeoverActionTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionsTypeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "high_action": "highAction",
            "low_action": "lowAction",
            "medium_action": "mediumAction",
        },
    )
    class AccountTakeoverActionsTypeProperty:
        def __init__(
            self,
            *,
            high_action: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            low_action: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            medium_action: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''A list of account-takeover actions for each level of risk that Amazon Cognito might assess with advanced security features.

            :param high_action: The action that you assign to a high-risk assessment by threat protection.
            :param low_action: The action that you assign to a low-risk assessment by threat protection.
            :param medium_action: The action that you assign to a medium-risk assessment by threat protection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-accounttakeoveractionstype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                account_takeover_actions_type_property = cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionsTypeProperty(
                    high_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                        event_action="eventAction",
                        notify=False
                    ),
                    low_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                        event_action="eventAction",
                        notify=False
                    ),
                    medium_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                        event_action="eventAction",
                        notify=False
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f069babb0e7383f87bfc13cebc53feb5f8120c387522f3256d59b144efff541f)
                check_type(argname="argument high_action", value=high_action, expected_type=type_hints["high_action"])
                check_type(argname="argument low_action", value=low_action, expected_type=type_hints["low_action"])
                check_type(argname="argument medium_action", value=medium_action, expected_type=type_hints["medium_action"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if high_action is not None:
                self._values["high_action"] = high_action
            if low_action is not None:
                self._values["low_action"] = low_action
            if medium_action is not None:
                self._values["medium_action"] = medium_action

        @builtins.property
        def high_action(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty"]]:
            '''The action that you assign to a high-risk assessment by threat protection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-accounttakeoveractionstype.html#cfn-cognito-userpoolriskconfigurationattachment-accounttakeoveractionstype-highaction
            '''
            result = self._values.get("high_action")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty"]], result)

        @builtins.property
        def low_action(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty"]]:
            '''The action that you assign to a low-risk assessment by threat protection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-accounttakeoveractionstype.html#cfn-cognito-userpoolriskconfigurationattachment-accounttakeoveractionstype-lowaction
            '''
            result = self._values.get("low_action")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty"]], result)

        @builtins.property
        def medium_action(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty"]]:
            '''The action that you assign to a medium-risk assessment by threat protection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-accounttakeoveractionstype.html#cfn-cognito-userpoolriskconfigurationattachment-accounttakeoveractionstype-mediumaction
            '''
            result = self._values.get("medium_action")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccountTakeoverActionsTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverRiskConfigurationTypeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "actions": "actions",
            "notify_configuration": "notifyConfiguration",
        },
    )
    class AccountTakeoverRiskConfigurationTypeProperty:
        def __init__(
            self,
            *,
            actions: typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionsTypeProperty", typing.Dict[builtins.str, typing.Any]]],
            notify_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.NotifyConfigurationTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The settings for automated responses and notification templates for adaptive authentication with advanced security features.

            :param actions: A list of account-takeover actions for each level of risk that Amazon Cognito might assess with threat protection.
            :param notify_configuration: The settings for composing and sending an email message when threat protection assesses a risk level with adaptive authentication. When you choose to notify users in ``AccountTakeoverRiskConfiguration`` , Amazon Cognito sends an email message using the method and template that you set with this data type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-accounttakeoverriskconfigurationtype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                account_takeover_risk_configuration_type_property = cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverRiskConfigurationTypeProperty(
                    actions=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionsTypeProperty(
                        high_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                            event_action="eventAction",
                            notify=False
                        ),
                        low_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                            event_action="eventAction",
                            notify=False
                        ),
                        medium_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                            event_action="eventAction",
                            notify=False
                        )
                    ),
                
                    # the properties below are optional
                    notify_configuration=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyConfigurationTypeProperty(
                        source_arn="sourceArn",
                
                        # the properties below are optional
                        block_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                            subject="subject",
                
                            # the properties below are optional
                            html_body="htmlBody",
                            text_body="textBody"
                        ),
                        from="from",
                        mfa_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                            subject="subject",
                
                            # the properties below are optional
                            html_body="htmlBody",
                            text_body="textBody"
                        ),
                        no_action_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                            subject="subject",
                
                            # the properties below are optional
                            html_body="htmlBody",
                            text_body="textBody"
                        ),
                        reply_to="replyTo"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__171e16add5c171c71e588ba4dc5010b7f4eeed61b1de29de578818bd4f093f3a)
                check_type(argname="argument actions", value=actions, expected_type=type_hints["actions"])
                check_type(argname="argument notify_configuration", value=notify_configuration, expected_type=type_hints["notify_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "actions": actions,
            }
            if notify_configuration is not None:
                self._values["notify_configuration"] = notify_configuration

        @builtins.property
        def actions(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionsTypeProperty"]:
            '''A list of account-takeover actions for each level of risk that Amazon Cognito might assess with threat protection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-accounttakeoverriskconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-accounttakeoverriskconfigurationtype-actions
            '''
            result = self._values.get("actions")
            assert result is not None, "Required property 'actions' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionsTypeProperty"], result)

        @builtins.property
        def notify_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.NotifyConfigurationTypeProperty"]]:
            '''The settings for composing and sending an email message when threat protection assesses a risk level with adaptive authentication.

            When you choose to notify users in ``AccountTakeoverRiskConfiguration`` , Amazon Cognito sends an email message using the method and template that you set with this data type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-accounttakeoverriskconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-accounttakeoverriskconfigurationtype-notifyconfiguration
            '''
            result = self._values.get("notify_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.NotifyConfigurationTypeProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccountTakeoverRiskConfigurationTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsActionsTypeProperty",
        jsii_struct_bases=[],
        name_mapping={"event_action": "eventAction"},
    )
    class CompromisedCredentialsActionsTypeProperty:
        def __init__(self, *, event_action: builtins.str) -> None:
            '''Settings for user pool actions when Amazon Cognito detects compromised credentials with advanced security features in full-function ``ENFORCED`` mode.

            :param event_action: The action that Amazon Cognito takes when it detects compromised credentials.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-compromisedcredentialsactionstype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                compromised_credentials_actions_type_property = cognito.CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsActionsTypeProperty(
                    event_action="eventAction"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d50d850b07485d85851db7411e74523ef2b236b4f79dc4882afa873cba24379e)
                check_type(argname="argument event_action", value=event_action, expected_type=type_hints["event_action"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "event_action": event_action,
            }

        @builtins.property
        def event_action(self) -> builtins.str:
            '''The action that Amazon Cognito takes when it detects compromised credentials.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-compromisedcredentialsactionstype.html#cfn-cognito-userpoolriskconfigurationattachment-compromisedcredentialsactionstype-eventaction
            '''
            result = self._values.get("event_action")
            assert result is not None, "Required property 'event_action' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CompromisedCredentialsActionsTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsRiskConfigurationTypeProperty",
        jsii_struct_bases=[],
        name_mapping={"actions": "actions", "event_filter": "eventFilter"},
    )
    class CompromisedCredentialsRiskConfigurationTypeProperty:
        def __init__(
            self,
            *,
            actions: typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsActionsTypeProperty", typing.Dict[builtins.str, typing.Any]]],
            event_filter: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Settings for compromised-credentials actions and authentication-event sources with advanced security features in full-function ``ENFORCED`` mode.

            :param actions: Settings for the actions that you want your user pool to take when Amazon Cognito detects compromised credentials.
            :param event_filter: Settings for the sign-in activity where you want to configure compromised-credentials actions. Defaults to all events.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-compromisedcredentialsriskconfigurationtype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                compromised_credentials_risk_configuration_type_property = cognito.CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsRiskConfigurationTypeProperty(
                    actions=cognito.CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsActionsTypeProperty(
                        event_action="eventAction"
                    ),
                
                    # the properties below are optional
                    event_filter=["eventFilter"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c4a865171ee7a202a2dae96af5367bc1bbaf5644c4eeeb610126d5d16b69ed96)
                check_type(argname="argument actions", value=actions, expected_type=type_hints["actions"])
                check_type(argname="argument event_filter", value=event_filter, expected_type=type_hints["event_filter"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "actions": actions,
            }
            if event_filter is not None:
                self._values["event_filter"] = event_filter

        @builtins.property
        def actions(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsActionsTypeProperty"]:
            '''Settings for the actions that you want your user pool to take when Amazon Cognito detects compromised credentials.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-compromisedcredentialsriskconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-compromisedcredentialsriskconfigurationtype-actions
            '''
            result = self._values.get("actions")
            assert result is not None, "Required property 'actions' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsActionsTypeProperty"], result)

        @builtins.property
        def event_filter(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Settings for the sign-in activity where you want to configure compromised-credentials actions.

            Defaults to all events.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-compromisedcredentialsriskconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-compromisedcredentialsriskconfigurationtype-eventfilter
            '''
            result = self._values.get("event_filter")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CompromisedCredentialsRiskConfigurationTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolRiskConfigurationAttachment.NotifyConfigurationTypeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "source_arn": "sourceArn",
            "block_email": "blockEmail",
            "from_": "from",
            "mfa_email": "mfaEmail",
            "no_action_email": "noActionEmail",
            "reply_to": "replyTo",
        },
    )
    class NotifyConfigurationTypeProperty:
        def __init__(
            self,
            *,
            source_arn: builtins.str,
            block_email: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            from_: typing.Optional[builtins.str] = None,
            mfa_email: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            no_action_email: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            reply_to: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The configuration for Amazon SES email messages that advanced security features sends to a user when your adaptive authentication automated response has a *Notify* action.

            :param source_arn: The Amazon Resource Name (ARN) of the identity that is associated with the sending authorization policy. This identity permits Amazon Cognito to send for the email address specified in the ``From`` parameter.
            :param block_email: The template for the email message that your user pool sends when a detected risk event is blocked.
            :param from_: The email address that sends the email message. The address must be either individually verified with Amazon Simple Email Service, or from a domain that has been verified with Amazon SES.
            :param mfa_email: The template for the email message that your user pool sends when MFA is challenged in response to a detected risk.
            :param no_action_email: The template for the email message that your user pool sends when no action is taken in response to a detected risk.
            :param reply_to: The reply-to email address of an email template.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                notify_configuration_type_property = cognito.CfnUserPoolRiskConfigurationAttachment.NotifyConfigurationTypeProperty(
                    source_arn="sourceArn",
                
                    # the properties below are optional
                    block_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                        subject="subject",
                
                        # the properties below are optional
                        html_body="htmlBody",
                        text_body="textBody"
                    ),
                    from="from",
                    mfa_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                        subject="subject",
                
                        # the properties below are optional
                        html_body="htmlBody",
                        text_body="textBody"
                    ),
                    no_action_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                        subject="subject",
                
                        # the properties below are optional
                        html_body="htmlBody",
                        text_body="textBody"
                    ),
                    reply_to="replyTo"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2bf1c06aab9b28c8e948bfb55b05866a98728f29cf413f42daba637d09193e97)
                check_type(argname="argument source_arn", value=source_arn, expected_type=type_hints["source_arn"])
                check_type(argname="argument block_email", value=block_email, expected_type=type_hints["block_email"])
                check_type(argname="argument from_", value=from_, expected_type=type_hints["from_"])
                check_type(argname="argument mfa_email", value=mfa_email, expected_type=type_hints["mfa_email"])
                check_type(argname="argument no_action_email", value=no_action_email, expected_type=type_hints["no_action_email"])
                check_type(argname="argument reply_to", value=reply_to, expected_type=type_hints["reply_to"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "source_arn": source_arn,
            }
            if block_email is not None:
                self._values["block_email"] = block_email
            if from_ is not None:
                self._values["from_"] = from_
            if mfa_email is not None:
                self._values["mfa_email"] = mfa_email
            if no_action_email is not None:
                self._values["no_action_email"] = no_action_email
            if reply_to is not None:
                self._values["reply_to"] = reply_to

        @builtins.property
        def source_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the identity that is associated with the sending authorization policy.

            This identity permits Amazon Cognito to send for the email address specified in the ``From`` parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype-sourcearn
            '''
            result = self._values.get("source_arn")
            assert result is not None, "Required property 'source_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def block_email(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty"]]:
            '''The template for the email message that your user pool sends when a detected risk event is blocked.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype-blockemail
            '''
            result = self._values.get("block_email")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty"]], result)

        @builtins.property
        def from_(self) -> typing.Optional[builtins.str]:
            '''The email address that sends the email message.

            The address must be either individually verified with Amazon Simple Email Service, or from a domain that has been verified with Amazon SES.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype-from
            '''
            result = self._values.get("from_")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def mfa_email(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty"]]:
            '''The template for the email message that your user pool sends when MFA is challenged in response to a detected risk.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype-mfaemail
            '''
            result = self._values.get("mfa_email")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty"]], result)

        @builtins.property
        def no_action_email(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty"]]:
            '''The template for the email message that your user pool sends when no action is taken in response to a detected risk.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype-noactionemail
            '''
            result = self._values.get("no_action_email")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty"]], result)

        @builtins.property
        def reply_to(self) -> typing.Optional[builtins.str]:
            '''The reply-to email address of an email template.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-notifyconfigurationtype-replyto
            '''
            result = self._values.get("reply_to")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NotifyConfigurationTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "subject": "subject",
            "html_body": "htmlBody",
            "text_body": "textBody",
        },
    )
    class NotifyEmailTypeProperty:
        def __init__(
            self,
            *,
            subject: builtins.str,
            html_body: typing.Optional[builtins.str] = None,
            text_body: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The template for email messages that advanced security features sends to a user when your threat protection automated response has a *Notify* action.

            :param subject: The subject of the threat protection email notification.
            :param html_body: The body of an email notification formatted in HTML. Choose an ``HtmlBody`` or a ``TextBody`` to send an HTML-formatted or plaintext message, respectively.
            :param text_body: The body of an email notification formatted in plaintext. Choose an ``HtmlBody`` or a ``TextBody`` to send an HTML-formatted or plaintext message, respectively.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-notifyemailtype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                notify_email_type_property = cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                    subject="subject",
                
                    # the properties below are optional
                    html_body="htmlBody",
                    text_body="textBody"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6800925bb1fbc1aba866bb6851b8782035e890449b40e2d78a413f3dcb79df79)
                check_type(argname="argument subject", value=subject, expected_type=type_hints["subject"])
                check_type(argname="argument html_body", value=html_body, expected_type=type_hints["html_body"])
                check_type(argname="argument text_body", value=text_body, expected_type=type_hints["text_body"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "subject": subject,
            }
            if html_body is not None:
                self._values["html_body"] = html_body
            if text_body is not None:
                self._values["text_body"] = text_body

        @builtins.property
        def subject(self) -> builtins.str:
            '''The subject of the threat protection email notification.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-notifyemailtype.html#cfn-cognito-userpoolriskconfigurationattachment-notifyemailtype-subject
            '''
            result = self._values.get("subject")
            assert result is not None, "Required property 'subject' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def html_body(self) -> typing.Optional[builtins.str]:
            '''The body of an email notification formatted in HTML.

            Choose an ``HtmlBody`` or a ``TextBody`` to send an HTML-formatted or plaintext message, respectively.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-notifyemailtype.html#cfn-cognito-userpoolriskconfigurationattachment-notifyemailtype-htmlbody
            '''
            result = self._values.get("html_body")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def text_body(self) -> typing.Optional[builtins.str]:
            '''The body of an email notification formatted in plaintext.

            Choose an ``HtmlBody`` or a ``TextBody`` to send an HTML-formatted or plaintext message, respectively.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-notifyemailtype.html#cfn-cognito-userpoolriskconfigurationattachment-notifyemailtype-textbody
            '''
            result = self._values.get("text_body")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NotifyEmailTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolRiskConfigurationAttachment.RiskExceptionConfigurationTypeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "blocked_ip_range_list": "blockedIpRangeList",
            "skipped_ip_range_list": "skippedIpRangeList",
        },
    )
    class RiskExceptionConfigurationTypeProperty:
        def __init__(
            self,
            *,
            blocked_ip_range_list: typing.Optional[typing.Sequence[builtins.str]] = None,
            skipped_ip_range_list: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Exceptions to the risk evaluation configuration, including always-allow and always-block IP address ranges.

            :param blocked_ip_range_list: An always-block IP address list. Overrides the risk decision and always blocks authentication requests. This parameter is displayed and set in CIDR notation.
            :param skipped_ip_range_list: An always-allow IP address list. Risk detection isn't performed on the IP addresses in this range list. This parameter is displayed and set in CIDR notation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-riskexceptionconfigurationtype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                risk_exception_configuration_type_property = cognito.CfnUserPoolRiskConfigurationAttachment.RiskExceptionConfigurationTypeProperty(
                    blocked_ip_range_list=["blockedIpRangeList"],
                    skipped_ip_range_list=["skippedIpRangeList"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__94550db26287873cca0c0acdb5d5eeb1043c346f4abf4c7e67a461cf6155f180)
                check_type(argname="argument blocked_ip_range_list", value=blocked_ip_range_list, expected_type=type_hints["blocked_ip_range_list"])
                check_type(argname="argument skipped_ip_range_list", value=skipped_ip_range_list, expected_type=type_hints["skipped_ip_range_list"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if blocked_ip_range_list is not None:
                self._values["blocked_ip_range_list"] = blocked_ip_range_list
            if skipped_ip_range_list is not None:
                self._values["skipped_ip_range_list"] = skipped_ip_range_list

        @builtins.property
        def blocked_ip_range_list(self) -> typing.Optional[typing.List[builtins.str]]:
            '''An always-block IP address list.

            Overrides the risk decision and always blocks authentication requests. This parameter is displayed and set in CIDR notation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-riskexceptionconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-riskexceptionconfigurationtype-blockediprangelist
            '''
            result = self._values.get("blocked_ip_range_list")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def skipped_ip_range_list(self) -> typing.Optional[typing.List[builtins.str]]:
            '''An always-allow IP address list.

            Risk detection isn't performed on the IP addresses in this range list. This parameter is displayed and set in CIDR notation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpoolriskconfigurationattachment-riskexceptionconfigurationtype.html#cfn-cognito-userpoolriskconfigurationattachment-riskexceptionconfigurationtype-skippediprangelist
            '''
            result = self._values.get("skipped_ip_range_list")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RiskExceptionConfigurationTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolRiskConfigurationAttachmentProps",
    jsii_struct_bases=[],
    name_mapping={
        "client_id": "clientId",
        "user_pool_id": "userPoolId",
        "account_takeover_risk_configuration": "accountTakeoverRiskConfiguration",
        "compromised_credentials_risk_configuration": "compromisedCredentialsRiskConfiguration",
        "risk_exception_configuration": "riskExceptionConfiguration",
    },
)
class CfnUserPoolRiskConfigurationAttachmentProps:
    def __init__(
        self,
        *,
        client_id: builtins.str,
        user_pool_id: builtins.str,
        account_takeover_risk_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPoolRiskConfigurationAttachment.AccountTakeoverRiskConfigurationTypeProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        compromised_credentials_risk_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsRiskConfigurationTypeProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        risk_exception_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPoolRiskConfigurationAttachment.RiskExceptionConfigurationTypeProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnUserPoolRiskConfigurationAttachment``.

        :param client_id: The app client where this configuration is applied. When this parameter isn't present, the risk configuration applies to all user pool app clients that don't have client-level settings.
        :param user_pool_id: The ID of the user pool that has the risk configuration applied.
        :param account_takeover_risk_configuration: The settings for automated responses and notification templates for adaptive authentication with threat protection.
        :param compromised_credentials_risk_configuration: Settings for compromised-credentials actions and authentication types with threat protection in full-function ``ENFORCED`` mode.
        :param risk_exception_configuration: Exceptions to the risk evaluation configuration, including always-allow and always-block IP address ranges.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolriskconfigurationattachment.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            cfn_user_pool_risk_configuration_attachment_props = cognito.CfnUserPoolRiskConfigurationAttachmentProps(
                client_id="clientId",
                user_pool_id="userPoolId",
            
                # the properties below are optional
                account_takeover_risk_configuration=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverRiskConfigurationTypeProperty(
                    actions=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionsTypeProperty(
                        high_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                            event_action="eventAction",
                            notify=False
                        ),
                        low_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                            event_action="eventAction",
                            notify=False
                        ),
                        medium_action=cognito.CfnUserPoolRiskConfigurationAttachment.AccountTakeoverActionTypeProperty(
                            event_action="eventAction",
                            notify=False
                        )
                    ),
            
                    # the properties below are optional
                    notify_configuration=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyConfigurationTypeProperty(
                        source_arn="sourceArn",
            
                        # the properties below are optional
                        block_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                            subject="subject",
            
                            # the properties below are optional
                            html_body="htmlBody",
                            text_body="textBody"
                        ),
                        from="from",
                        mfa_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                            subject="subject",
            
                            # the properties below are optional
                            html_body="htmlBody",
                            text_body="textBody"
                        ),
                        no_action_email=cognito.CfnUserPoolRiskConfigurationAttachment.NotifyEmailTypeProperty(
                            subject="subject",
            
                            # the properties below are optional
                            html_body="htmlBody",
                            text_body="textBody"
                        ),
                        reply_to="replyTo"
                    )
                ),
                compromised_credentials_risk_configuration=cognito.CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsRiskConfigurationTypeProperty(
                    actions=cognito.CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsActionsTypeProperty(
                        event_action="eventAction"
                    ),
            
                    # the properties below are optional
                    event_filter=["eventFilter"]
                ),
                risk_exception_configuration=cognito.CfnUserPoolRiskConfigurationAttachment.RiskExceptionConfigurationTypeProperty(
                    blocked_ip_range_list=["blockedIpRangeList"],
                    skipped_ip_range_list=["skippedIpRangeList"]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aae4da006a28eede974ab417d5b2bc68196cfdf7df68bb0ddfc79b19b8790f8b)
            check_type(argname="argument client_id", value=client_id, expected_type=type_hints["client_id"])
            check_type(argname="argument user_pool_id", value=user_pool_id, expected_type=type_hints["user_pool_id"])
            check_type(argname="argument account_takeover_risk_configuration", value=account_takeover_risk_configuration, expected_type=type_hints["account_takeover_risk_configuration"])
            check_type(argname="argument compromised_credentials_risk_configuration", value=compromised_credentials_risk_configuration, expected_type=type_hints["compromised_credentials_risk_configuration"])
            check_type(argname="argument risk_exception_configuration", value=risk_exception_configuration, expected_type=type_hints["risk_exception_configuration"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "client_id": client_id,
            "user_pool_id": user_pool_id,
        }
        if account_takeover_risk_configuration is not None:
            self._values["account_takeover_risk_configuration"] = account_takeover_risk_configuration
        if compromised_credentials_risk_configuration is not None:
            self._values["compromised_credentials_risk_configuration"] = compromised_credentials_risk_configuration
        if risk_exception_configuration is not None:
            self._values["risk_exception_configuration"] = risk_exception_configuration

    @builtins.property
    def client_id(self) -> builtins.str:
        '''The app client where this configuration is applied.

        When this parameter isn't present, the risk configuration applies to all user pool app clients that don't have client-level settings.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolriskconfigurationattachment.html#cfn-cognito-userpoolriskconfigurationattachment-clientid
        '''
        result = self._values.get("client_id")
        assert result is not None, "Required property 'client_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool that has the risk configuration applied.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolriskconfigurationattachment.html#cfn-cognito-userpoolriskconfigurationattachment-userpoolid
        '''
        result = self._values.get("user_pool_id")
        assert result is not None, "Required property 'user_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def account_takeover_risk_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolRiskConfigurationAttachment.AccountTakeoverRiskConfigurationTypeProperty]]:
        '''The settings for automated responses and notification templates for adaptive authentication with threat protection.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolriskconfigurationattachment.html#cfn-cognito-userpoolriskconfigurationattachment-accounttakeoverriskconfiguration
        '''
        result = self._values.get("account_takeover_risk_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolRiskConfigurationAttachment.AccountTakeoverRiskConfigurationTypeProperty]], result)

    @builtins.property
    def compromised_credentials_risk_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsRiskConfigurationTypeProperty]]:
        '''Settings for compromised-credentials actions and authentication types with threat protection in full-function ``ENFORCED`` mode.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolriskconfigurationattachment.html#cfn-cognito-userpoolriskconfigurationattachment-compromisedcredentialsriskconfiguration
        '''
        result = self._values.get("compromised_credentials_risk_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolRiskConfigurationAttachment.CompromisedCredentialsRiskConfigurationTypeProperty]], result)

    @builtins.property
    def risk_exception_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolRiskConfigurationAttachment.RiskExceptionConfigurationTypeProperty]]:
        '''Exceptions to the risk evaluation configuration, including always-allow and always-block IP address ranges.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolriskconfigurationattachment.html#cfn-cognito-userpoolriskconfigurationattachment-riskexceptionconfiguration
        '''
        result = self._values.get("risk_exception_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUserPoolRiskConfigurationAttachment.RiskExceptionConfigurationTypeProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUserPoolRiskConfigurationAttachmentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnUserPoolUICustomizationAttachment(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolUICustomizationAttachment",
):
    '''A container for the UI customization information for the hosted UI in a user pool.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluicustomizationattachment.html
    :cloudformationResource: AWS::Cognito::UserPoolUICustomizationAttachment
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        cfn_user_pool_uICustomization_attachment = cognito.CfnUserPoolUICustomizationAttachment(self, "MyCfnUserPoolUICustomizationAttachment",
            client_id="clientId",
            user_pool_id="userPoolId",
        
            # the properties below are optional
            css="css"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        client_id: builtins.str,
        user_pool_id: builtins.str,
        css: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param client_id: The app client ID for your UI customization. When this value isn't present, the customization applies to all user pool app clients that don't have client-level settings..
        :param user_pool_id: The ID of the user pool where you want to apply branding to the classic hosted UI.
        :param css: A plaintext CSS file that contains the custom fields that you want to apply to your user pool or app client. To download a template, go to the Amazon Cognito console. Navigate to your user pool *App clients* tab, select *Login pages* , edit *Hosted UI (classic) style* , and select the link to ``CSS template.css`` .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf3306ea8a9b6f4ebe73eb42059e52138281652a9e2e36e507fd8658eb5da33a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUserPoolUICustomizationAttachmentProps(
            client_id=client_id, user_pool_id=user_pool_id, css=css
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__80b3aec0691b38b614ccbfd3a409e8f6e5a07ef4b029f9f7451ae00fe408ecee)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ff3c67f9277cb7242d2f8cc0f7c9422962350df6fafbc523b7d4b92250f80565)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="clientId")
    def client_id(self) -> builtins.str:
        '''The app client ID for your UI customization.'''
        return typing.cast(builtins.str, jsii.get(self, "clientId"))

    @client_id.setter
    def client_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2fb840d14a15822261415d3d8b15b58403a1bf218216b5d0aba1826076c9cb03)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you want to apply branding to the classic hosted UI.'''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @user_pool_id.setter
    def user_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bc21fab90e665a827c26364efaa1568390b907cce17d5fc3f8621bfe65268d7b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="css")
    def css(self) -> typing.Optional[builtins.str]:
        '''A plaintext CSS file that contains the custom fields that you want to apply to your user pool or app client.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "css"))

    @css.setter
    def css(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__95fdebc4b645acafac0d58bca95a8d84ac0be5fcf3ca3d612ad578dbb3faa891)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "css", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolUICustomizationAttachmentProps",
    jsii_struct_bases=[],
    name_mapping={"client_id": "clientId", "user_pool_id": "userPoolId", "css": "css"},
)
class CfnUserPoolUICustomizationAttachmentProps:
    def __init__(
        self,
        *,
        client_id: builtins.str,
        user_pool_id: builtins.str,
        css: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnUserPoolUICustomizationAttachment``.

        :param client_id: The app client ID for your UI customization. When this value isn't present, the customization applies to all user pool app clients that don't have client-level settings..
        :param user_pool_id: The ID of the user pool where you want to apply branding to the classic hosted UI.
        :param css: A plaintext CSS file that contains the custom fields that you want to apply to your user pool or app client. To download a template, go to the Amazon Cognito console. Navigate to your user pool *App clients* tab, select *Login pages* , edit *Hosted UI (classic) style* , and select the link to ``CSS template.css`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluicustomizationattachment.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            cfn_user_pool_uICustomization_attachment_props = cognito.CfnUserPoolUICustomizationAttachmentProps(
                client_id="clientId",
                user_pool_id="userPoolId",
            
                # the properties below are optional
                css="css"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4e4f062a7f00241b34ab17fd47a3486ca4cbd516aac965d87988b2f16d5ac603)
            check_type(argname="argument client_id", value=client_id, expected_type=type_hints["client_id"])
            check_type(argname="argument user_pool_id", value=user_pool_id, expected_type=type_hints["user_pool_id"])
            check_type(argname="argument css", value=css, expected_type=type_hints["css"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "client_id": client_id,
            "user_pool_id": user_pool_id,
        }
        if css is not None:
            self._values["css"] = css

    @builtins.property
    def client_id(self) -> builtins.str:
        '''The app client ID for your UI customization.

        When this value isn't present, the customization applies to all user pool app clients that don't have client-level settings..

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluicustomizationattachment.html#cfn-cognito-userpooluicustomizationattachment-clientid
        '''
        result = self._values.get("client_id")
        assert result is not None, "Required property 'client_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you want to apply branding to the classic hosted UI.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluicustomizationattachment.html#cfn-cognito-userpooluicustomizationattachment-userpoolid
        '''
        result = self._values.get("user_pool_id")
        assert result is not None, "Required property 'user_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def css(self) -> typing.Optional[builtins.str]:
        '''A plaintext CSS file that contains the custom fields that you want to apply to your user pool or app client.

        To download a template, go to the Amazon Cognito console. Navigate to your user pool *App clients* tab, select *Login pages* , edit *Hosted UI (classic) style* , and select the link to ``CSS template.css`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluicustomizationattachment.html#cfn-cognito-userpooluicustomizationattachment-css
        '''
        result = self._values.get("css")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUserPoolUICustomizationAttachmentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnUserPoolUser(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolUser",
):
    '''The ``AWS::Cognito::UserPoolUser`` resource creates an Amazon Cognito user pool user.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluser.html
    :cloudformationResource: AWS::Cognito::UserPoolUser
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        cfn_user_pool_user = cognito.CfnUserPoolUser(self, "MyCfnUserPoolUser",
            user_pool_id="userPoolId",
        
            # the properties below are optional
            client_metadata={
                "client_metadata_key": "clientMetadata"
            },
            desired_delivery_mediums=["desiredDeliveryMediums"],
            force_alias_creation=False,
            message_action="messageAction",
            user_attributes=[cognito.CfnUserPoolUser.AttributeTypeProperty(
                name="name",
                value="value"
            )],
            username="username",
            validation_data=[cognito.CfnUserPoolUser.AttributeTypeProperty(
                name="name",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        user_pool_id: builtins.str,
        client_metadata: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        desired_delivery_mediums: typing.Optional[typing.Sequence[builtins.str]] = None,
        force_alias_creation: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        message_action: typing.Optional[builtins.str] = None,
        user_attributes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolUser.AttributeTypeProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        username: typing.Optional[builtins.str] = None,
        validation_data: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUserPoolUser.AttributeTypeProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param user_pool_id: The ID of the user pool where you want to create a user.
        :param client_metadata: A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers. You create custom workflows by assigning AWS Lambda functions to user pool triggers. When you use the AdminCreateUser API action, Amazon Cognito invokes the function that is assigned to the *pre sign-up* trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which the function receives as input. This payload contains a ``ClientMetadata`` attribute, which provides the data that you assigned to the ClientMetadata parameter in your AdminCreateUser request. In your function code in AWS Lambda , you can process the ``clientMetadata`` value to enhance your workflow for your specific needs. For more information, see `Using Lambda triggers <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html>`_ in the *Amazon Cognito Developer Guide* . .. epigraph:: When you use the ``ClientMetadata`` parameter, note that Amazon Cognito won't do the following: - Store the ``ClientMetadata`` value. This data is available only to AWS Lambda triggers that are assigned to a user pool to support custom workflows. If your user pool configuration doesn't include triggers, the ``ClientMetadata`` parameter serves no purpose. - Validate the ``ClientMetadata`` value. - Encrypt the ``ClientMetadata`` value. Don't send sensitive information in this parameter.
        :param desired_delivery_mediums: Specify ``EMAIL`` if email will be used to send the welcome message. Specify ``SMS`` if the phone number will be used. The default value is ``SMS`` . You can specify more than one value.
        :param force_alias_creation: This parameter is used only if the ``phone_number_verified`` or ``email_verified`` attribute is set to ``True`` . Otherwise, it is ignored. If this parameter is set to ``True`` and the phone number or email address specified in the ``UserAttributes`` parameter already exists as an alias with a different user, this request migrates the alias from the previous user to the newly-created user. The previous user will no longer be able to log in using that alias. If this parameter is set to ``False`` , the API throws an ``AliasExistsException`` error if the alias already exists. The default value is ``False`` .
        :param message_action: Set to ``RESEND`` to resend the invitation message to a user that already exists, and to reset the temporary-password duration with a new temporary password. Set to ``SUPPRESS`` to suppress sending the message. You can specify only one value.
        :param user_attributes: An array of name-value pairs that contain user attributes and attribute values to be set for the user to be created. You can create a user without specifying any attributes other than ``Username`` . However, any attributes that you specify as required (when creating a user pool or in the *Attributes* tab of the console) either you should supply (in your call to ``AdminCreateUser`` ) or the user should supply (when they sign up in response to your welcome message). For custom attributes, you must prepend the ``custom:`` prefix to the attribute name. To send a message inviting the user to sign up, you must specify the user's email address or phone number. You can do this in your call to AdminCreateUser or in the *Users* tab of the Amazon Cognito console for managing your user pools. You must also provide an email address or phone number when you expect the user to do passwordless sign-in with an email or SMS OTP. These attributes must be provided when passwordless options are the only available, or when you don't submit a ``TemporaryPassword`` . In your call to ``AdminCreateUser`` , you can set the ``email_verified`` attribute to ``True`` , and you can set the ``phone_number_verified`` attribute to ``True`` . - *email* : The email address of the user to whom the message that contains the code and username will be sent. Required if the ``email_verified`` attribute is set to ``True`` , or if ``"EMAIL"`` is specified in the ``DesiredDeliveryMediums`` parameter. - *phone_number* : The phone number of the user to whom the message that contains the code and username will be sent. Required if the ``phone_number_verified`` attribute is set to ``True`` , or if ``"SMS"`` is specified in the ``DesiredDeliveryMediums`` parameter.
        :param username: The value that you want to set as the username sign-in attribute. The following conditions apply to the username parameter. - The username can't be a duplicate of another username in the same user pool. - You can't change the value of a username after you create it. - You can only provide a value if usernames are a valid sign-in attribute for your user pool. If your user pool only supports phone numbers or email addresses as sign-in attributes, Amazon Cognito automatically generates a username value. For more information, see `Customizing sign-in attributes <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases>`_ .
        :param validation_data: Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of key-value pairs are for custom validation of information that you collect from your users but don't need to retain. Your Lambda function can analyze this additional data and act on it. Your function can automatically confirm and verify select users or perform external API operations like logging user attributes and validation data to Amazon CloudWatch Logs. For more information about the pre sign-up Lambda trigger, see `Pre sign-up Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html>`_ .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__392de74de1133635a0d4d21dbd0cb3290007171e021625ff9a125983463dd374)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUserPoolUserProps(
            user_pool_id=user_pool_id,
            client_metadata=client_metadata,
            desired_delivery_mediums=desired_delivery_mediums,
            force_alias_creation=force_alias_creation,
            message_action=message_action,
            user_attributes=user_attributes,
            username=username,
            validation_data=validation_data,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7a682a2322b995c054dfe4130e2fb7a0f0bc4eecc31485fdac372757331071a1)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c1d5d9ebca9f15e71312208c85763ef350bc2128b24951edf30f977cd99b8ca0)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you want to create a user.'''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @user_pool_id.setter
    def user_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__feefc710cb336bda2be62c58dcbdfc764a535ab2e52aa19e44511aca0956258c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="clientMetadata")
    def client_metadata(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers.'''
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], jsii.get(self, "clientMetadata"))

    @client_metadata.setter
    def client_metadata(
        self,
        value: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__674b44537db9d65119536a886fe3b7990cd0df2f1aa1c3c2880711f8b302ae40)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientMetadata", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="desiredDeliveryMediums")
    def desired_delivery_mediums(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify ``EMAIL`` if email will be used to send the welcome message.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "desiredDeliveryMediums"))

    @desired_delivery_mediums.setter
    def desired_delivery_mediums(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__91542a3b74c01d216df5ff651d130ee1327aa0fc3c895cbd605609cf901699ba)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "desiredDeliveryMediums", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="forceAliasCreation")
    def force_alias_creation(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''This parameter is used only if the ``phone_number_verified`` or ``email_verified`` attribute is set to ``True`` .'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "forceAliasCreation"))

    @force_alias_creation.setter
    def force_alias_creation(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7c2c789fa64d9c856de89108f23fbf51d514f0cffa783771cfbacd552aee676c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "forceAliasCreation", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="messageAction")
    def message_action(self) -> typing.Optional[builtins.str]:
        '''Set to ``RESEND`` to resend the invitation message to a user that already exists, and to reset the temporary-password duration with a new temporary password.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "messageAction"))

    @message_action.setter
    def message_action(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9f0192dc3faa77898f91fb1e03cf55e5b740abf20bba3ba939ce4df47ae4227d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "messageAction", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userAttributes")
    def user_attributes(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPoolUser.AttributeTypeProperty"]]]]:
        '''An array of name-value pairs that contain user attributes and attribute values to be set for the user to be created.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPoolUser.AttributeTypeProperty"]]]], jsii.get(self, "userAttributes"))

    @user_attributes.setter
    def user_attributes(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPoolUser.AttributeTypeProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3a84d0d411d439647c27dc7ac7d1d6e760fe129e919aa8a0a8b4194179281b3e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userAttributes", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="username")
    def username(self) -> typing.Optional[builtins.str]:
        '''The value that you want to set as the username sign-in attribute.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "username"))

    @username.setter
    def username(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__942108c1e34b7c1e541ffb7d9f565c31400708ae6fa3757029d02956f7e7086d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "username", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="validationData")
    def validation_data(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPoolUser.AttributeTypeProperty"]]]]:
        '''Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPoolUser.AttributeTypeProperty"]]]], jsii.get(self, "validationData"))

    @validation_data.setter
    def validation_data(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUserPoolUser.AttributeTypeProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__69a6ee7471fbb46c9acd4942399424982966d6d11c89f8748a188ab7f08e54be)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "validationData", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolUser.AttributeTypeProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class AttributeTypeProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The name and value of a user attribute.

            :param name: The name of the attribute.
            :param value: The value of the attribute.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpooluser-attributetype.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_cognito as cognito
                
                attribute_type_property = cognito.CfnUserPoolUser.AttributeTypeProperty(
                    name="name",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__480fd17b16e7156f8c801d35ad2f0806c252a5a32182194061ca2a9566a36851)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the attribute.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpooluser-attributetype.html#cfn-cognito-userpooluser-attributetype-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value of the attribute.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpooluser-attributetype.html#cfn-cognito-userpooluser-attributetype-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AttributeTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolUserProps",
    jsii_struct_bases=[],
    name_mapping={
        "user_pool_id": "userPoolId",
        "client_metadata": "clientMetadata",
        "desired_delivery_mediums": "desiredDeliveryMediums",
        "force_alias_creation": "forceAliasCreation",
        "message_action": "messageAction",
        "user_attributes": "userAttributes",
        "username": "username",
        "validation_data": "validationData",
    },
)
class CfnUserPoolUserProps:
    def __init__(
        self,
        *,
        user_pool_id: builtins.str,
        client_metadata: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        desired_delivery_mediums: typing.Optional[typing.Sequence[builtins.str]] = None,
        force_alias_creation: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        message_action: typing.Optional[builtins.str] = None,
        user_attributes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPoolUser.AttributeTypeProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        username: typing.Optional[builtins.str] = None,
        validation_data: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUserPoolUser.AttributeTypeProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnUserPoolUser``.

        :param user_pool_id: The ID of the user pool where you want to create a user.
        :param client_metadata: A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers. You create custom workflows by assigning AWS Lambda functions to user pool triggers. When you use the AdminCreateUser API action, Amazon Cognito invokes the function that is assigned to the *pre sign-up* trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which the function receives as input. This payload contains a ``ClientMetadata`` attribute, which provides the data that you assigned to the ClientMetadata parameter in your AdminCreateUser request. In your function code in AWS Lambda , you can process the ``clientMetadata`` value to enhance your workflow for your specific needs. For more information, see `Using Lambda triggers <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html>`_ in the *Amazon Cognito Developer Guide* . .. epigraph:: When you use the ``ClientMetadata`` parameter, note that Amazon Cognito won't do the following: - Store the ``ClientMetadata`` value. This data is available only to AWS Lambda triggers that are assigned to a user pool to support custom workflows. If your user pool configuration doesn't include triggers, the ``ClientMetadata`` parameter serves no purpose. - Validate the ``ClientMetadata`` value. - Encrypt the ``ClientMetadata`` value. Don't send sensitive information in this parameter.
        :param desired_delivery_mediums: Specify ``EMAIL`` if email will be used to send the welcome message. Specify ``SMS`` if the phone number will be used. The default value is ``SMS`` . You can specify more than one value.
        :param force_alias_creation: This parameter is used only if the ``phone_number_verified`` or ``email_verified`` attribute is set to ``True`` . Otherwise, it is ignored. If this parameter is set to ``True`` and the phone number or email address specified in the ``UserAttributes`` parameter already exists as an alias with a different user, this request migrates the alias from the previous user to the newly-created user. The previous user will no longer be able to log in using that alias. If this parameter is set to ``False`` , the API throws an ``AliasExistsException`` error if the alias already exists. The default value is ``False`` .
        :param message_action: Set to ``RESEND`` to resend the invitation message to a user that already exists, and to reset the temporary-password duration with a new temporary password. Set to ``SUPPRESS`` to suppress sending the message. You can specify only one value.
        :param user_attributes: An array of name-value pairs that contain user attributes and attribute values to be set for the user to be created. You can create a user without specifying any attributes other than ``Username`` . However, any attributes that you specify as required (when creating a user pool or in the *Attributes* tab of the console) either you should supply (in your call to ``AdminCreateUser`` ) or the user should supply (when they sign up in response to your welcome message). For custom attributes, you must prepend the ``custom:`` prefix to the attribute name. To send a message inviting the user to sign up, you must specify the user's email address or phone number. You can do this in your call to AdminCreateUser or in the *Users* tab of the Amazon Cognito console for managing your user pools. You must also provide an email address or phone number when you expect the user to do passwordless sign-in with an email or SMS OTP. These attributes must be provided when passwordless options are the only available, or when you don't submit a ``TemporaryPassword`` . In your call to ``AdminCreateUser`` , you can set the ``email_verified`` attribute to ``True`` , and you can set the ``phone_number_verified`` attribute to ``True`` . - *email* : The email address of the user to whom the message that contains the code and username will be sent. Required if the ``email_verified`` attribute is set to ``True`` , or if ``"EMAIL"`` is specified in the ``DesiredDeliveryMediums`` parameter. - *phone_number* : The phone number of the user to whom the message that contains the code and username will be sent. Required if the ``phone_number_verified`` attribute is set to ``True`` , or if ``"SMS"`` is specified in the ``DesiredDeliveryMediums`` parameter.
        :param username: The value that you want to set as the username sign-in attribute. The following conditions apply to the username parameter. - The username can't be a duplicate of another username in the same user pool. - You can't change the value of a username after you create it. - You can only provide a value if usernames are a valid sign-in attribute for your user pool. If your user pool only supports phone numbers or email addresses as sign-in attributes, Amazon Cognito automatically generates a username value. For more information, see `Customizing sign-in attributes <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases>`_ .
        :param validation_data: Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of key-value pairs are for custom validation of information that you collect from your users but don't need to retain. Your Lambda function can analyze this additional data and act on it. Your function can automatically confirm and verify select users or perform external API operations like logging user attributes and validation data to Amazon CloudWatch Logs. For more information about the pre sign-up Lambda trigger, see `Pre sign-up Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluser.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            cfn_user_pool_user_props = cognito.CfnUserPoolUserProps(
                user_pool_id="userPoolId",
            
                # the properties below are optional
                client_metadata={
                    "client_metadata_key": "clientMetadata"
                },
                desired_delivery_mediums=["desiredDeliveryMediums"],
                force_alias_creation=False,
                message_action="messageAction",
                user_attributes=[cognito.CfnUserPoolUser.AttributeTypeProperty(
                    name="name",
                    value="value"
                )],
                username="username",
                validation_data=[cognito.CfnUserPoolUser.AttributeTypeProperty(
                    name="name",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__382fb58e358860ff3016c5f0203cf6f5b59ab27ba70ef920ce589784afe54f17)
            check_type(argname="argument user_pool_id", value=user_pool_id, expected_type=type_hints["user_pool_id"])
            check_type(argname="argument client_metadata", value=client_metadata, expected_type=type_hints["client_metadata"])
            check_type(argname="argument desired_delivery_mediums", value=desired_delivery_mediums, expected_type=type_hints["desired_delivery_mediums"])
            check_type(argname="argument force_alias_creation", value=force_alias_creation, expected_type=type_hints["force_alias_creation"])
            check_type(argname="argument message_action", value=message_action, expected_type=type_hints["message_action"])
            check_type(argname="argument user_attributes", value=user_attributes, expected_type=type_hints["user_attributes"])
            check_type(argname="argument username", value=username, expected_type=type_hints["username"])
            check_type(argname="argument validation_data", value=validation_data, expected_type=type_hints["validation_data"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "user_pool_id": user_pool_id,
        }
        if client_metadata is not None:
            self._values["client_metadata"] = client_metadata
        if desired_delivery_mediums is not None:
            self._values["desired_delivery_mediums"] = desired_delivery_mediums
        if force_alias_creation is not None:
            self._values["force_alias_creation"] = force_alias_creation
        if message_action is not None:
            self._values["message_action"] = message_action
        if user_attributes is not None:
            self._values["user_attributes"] = user_attributes
        if username is not None:
            self._values["username"] = username
        if validation_data is not None:
            self._values["validation_data"] = validation_data

    @builtins.property
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool where you want to create a user.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluser.html#cfn-cognito-userpooluser-userpoolid
        '''
        result = self._values.get("user_pool_id")
        assert result is not None, "Required property 'user_pool_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def client_metadata(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers.

        You create custom workflows by assigning AWS Lambda functions to user pool triggers. When you use the AdminCreateUser API action, Amazon Cognito invokes the function that is assigned to the *pre sign-up* trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which the function receives as input. This payload contains a ``ClientMetadata`` attribute, which provides the data that you assigned to the ClientMetadata parameter in your AdminCreateUser request. In your function code in AWS Lambda , you can process the ``clientMetadata`` value to enhance your workflow for your specific needs.

        For more information, see `Using Lambda triggers <https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html>`_ in the *Amazon Cognito Developer Guide* .
        .. epigraph::

           When you use the ``ClientMetadata`` parameter, note that Amazon Cognito won't do the following:

           - Store the ``ClientMetadata`` value. This data is available only to AWS Lambda triggers that are assigned to a user pool to support custom workflows. If your user pool configuration doesn't include triggers, the ``ClientMetadata`` parameter serves no purpose.
           - Validate the ``ClientMetadata`` value.
           - Encrypt the ``ClientMetadata`` value. Don't send sensitive information in this parameter.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluser.html#cfn-cognito-userpooluser-clientmetadata
        '''
        result = self._values.get("client_metadata")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

    @builtins.property
    def desired_delivery_mediums(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Specify ``EMAIL`` if email will be used to send the welcome message.

        Specify ``SMS`` if the phone number will be used. The default value is ``SMS`` . You can specify more than one value.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluser.html#cfn-cognito-userpooluser-desireddeliverymediums
        '''
        result = self._values.get("desired_delivery_mediums")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def force_alias_creation(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''This parameter is used only if the ``phone_number_verified`` or ``email_verified`` attribute is set to ``True`` .

        Otherwise, it is ignored.

        If this parameter is set to ``True`` and the phone number or email address specified in the ``UserAttributes`` parameter already exists as an alias with a different user, this request migrates the alias from the previous user to the newly-created user. The previous user will no longer be able to log in using that alias.

        If this parameter is set to ``False`` , the API throws an ``AliasExistsException`` error if the alias already exists. The default value is ``False`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluser.html#cfn-cognito-userpooluser-forcealiascreation
        '''
        result = self._values.get("force_alias_creation")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def message_action(self) -> typing.Optional[builtins.str]:
        '''Set to ``RESEND`` to resend the invitation message to a user that already exists, and to reset the temporary-password duration with a new temporary password.

        Set to ``SUPPRESS`` to suppress sending the message. You can specify only one value.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluser.html#cfn-cognito-userpooluser-messageaction
        '''
        result = self._values.get("message_action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def user_attributes(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnUserPoolUser.AttributeTypeProperty]]]]:
        '''An array of name-value pairs that contain user attributes and attribute values to be set for the user to be created.

        You can create a user without specifying any attributes other than ``Username`` . However, any attributes that you specify as required (when creating a user pool or in the *Attributes* tab of the console) either you should supply (in your call to ``AdminCreateUser`` ) or the user should supply (when they sign up in response to your welcome message).

        For custom attributes, you must prepend the ``custom:`` prefix to the attribute name.

        To send a message inviting the user to sign up, you must specify the user's email address or phone number. You can do this in your call to AdminCreateUser or in the *Users* tab of the Amazon Cognito console for managing your user pools.

        You must also provide an email address or phone number when you expect the user to do passwordless sign-in with an email or SMS OTP. These attributes must be provided when passwordless options are the only available, or when you don't submit a ``TemporaryPassword`` .

        In your call to ``AdminCreateUser`` , you can set the ``email_verified`` attribute to ``True`` , and you can set the ``phone_number_verified`` attribute to ``True`` .

        - *email* : The email address of the user to whom the message that contains the code and username will be sent. Required if the ``email_verified`` attribute is set to ``True`` , or if ``"EMAIL"`` is specified in the ``DesiredDeliveryMediums`` parameter.
        - *phone_number* : The phone number of the user to whom the message that contains the code and username will be sent. Required if the ``phone_number_verified`` attribute is set to ``True`` , or if ``"SMS"`` is specified in the ``DesiredDeliveryMediums`` parameter.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluser.html#cfn-cognito-userpooluser-userattributes
        '''
        result = self._values.get("user_attributes")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnUserPoolUser.AttributeTypeProperty]]]], result)

    @builtins.property
    def username(self) -> typing.Optional[builtins.str]:
        '''The value that you want to set as the username sign-in attribute.

        The following conditions apply to the username parameter.

        - The username can't be a duplicate of another username in the same user pool.
        - You can't change the value of a username after you create it.
        - You can only provide a value if usernames are a valid sign-in attribute for your user pool. If your user pool only supports phone numbers or email addresses as sign-in attributes, Amazon Cognito automatically generates a username value. For more information, see `Customizing sign-in attributes <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluser.html#cfn-cognito-userpooluser-username
        '''
        result = self._values.get("username")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def validation_data(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnUserPoolUser.AttributeTypeProperty]]]]:
        '''Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger.

        This set of key-value pairs are for custom validation of information that you collect from your users but don't need to retain.

        Your Lambda function can analyze this additional data and act on it. Your function can automatically confirm and verify select users or perform external API operations like logging user attributes and validation data to Amazon CloudWatch Logs.

        For more information about the pre sign-up Lambda trigger, see `Pre sign-up Lambda trigger <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpooluser.html#cfn-cognito-userpooluser-validationdata
        '''
        result = self._values.get("validation_data")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnUserPoolUser.AttributeTypeProperty]]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUserPoolUserProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnUserPoolUserToGroupAttachment(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolUserToGroupAttachment",
):
    '''Adds a user to a group.

    A user who is in a group can present a preferred-role claim to an identity pool, and populates a ``cognito:groups`` claim to their access and identity tokens.
    .. epigraph::

       Amazon Cognito evaluates AWS Identity and Access Management (IAM) policies in requests for this API operation. For this operation, you must use IAM credentials to authorize requests, and you must grant yourself the corresponding IAM permission in a policy.

       **Learn more** - `Signing AWS API Requests <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-signing.html>`_

       - `Using the Amazon Cognito user pools API and user pool endpoints <https://docs.aws.amazon.com/cognito/latest/developerguide/user-pools-API-operations.html>`_

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolusertogroupattachment.html
    :cloudformationResource: AWS::Cognito::UserPoolUserToGroupAttachment
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_cognito as cognito
        
        cfn_user_pool_user_to_group_attachment = cognito.CfnUserPoolUserToGroupAttachment(self, "MyCfnUserPoolUserToGroupAttachment",
            group_name="groupName",
            username="username",
            user_pool_id="userPoolId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        group_name: builtins.str,
        username: builtins.str,
        user_pool_id: builtins.str,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param group_name: The name of the group that you want to add your user to.
        :param username: The user's username.
        :param user_pool_id: The ID of the user pool that contains the group that you want to add the user to.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a7364d75afc292798863e1183adee778bee531b8ea162e2882fe2861f9b8e33c)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUserPoolUserToGroupAttachmentProps(
            group_name=group_name, username=username, user_pool_id=user_pool_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8646ff491f8e398241578077a82a538b7fc6168609db4df0a8980d20dc662a3a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__52c72a8fcf1b906af13f00063c1d25b12e9b2225c5b995b65b452deee4ef4a3d)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="groupName")
    def group_name(self) -> builtins.str:
        '''The name of the group that you want to add your user to.'''
        return typing.cast(builtins.str, jsii.get(self, "groupName"))

    @group_name.setter
    def group_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76f277297f14c067592421ffe39f914d1d4341dd9bc83f4c42c66f73914f3488)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "groupName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="username")
    def username(self) -> builtins.str:
        '''The user's username.'''
        return typing.cast(builtins.str, jsii.get(self, "username"))

    @username.setter
    def username(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a387f868c25297921e89f6892a60def04748b83f31771d79ae06623bc4eb435b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "username", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool that contains the group that you want to add the user to.'''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @user_pool_id.setter
    def user_pool_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__82177f095c7231924dc4eb6d3e0b1affc9151da04e151d35489d6ee497b571cc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "userPoolId", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CfnUserPoolUserToGroupAttachmentProps",
    jsii_struct_bases=[],
    name_mapping={
        "group_name": "groupName",
        "username": "username",
        "user_pool_id": "userPoolId",
    },
)
class CfnUserPoolUserToGroupAttachmentProps:
    def __init__(
        self,
        *,
        group_name: builtins.str,
        username: builtins.str,
        user_pool_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnUserPoolUserToGroupAttachment``.

        :param group_name: The name of the group that you want to add your user to.
        :param username: The user's username.
        :param user_pool_id: The ID of the user pool that contains the group that you want to add the user to.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolusertogroupattachment.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            cfn_user_pool_user_to_group_attachment_props = cognito.CfnUserPoolUserToGroupAttachmentProps(
                group_name="groupName",
                username="username",
                user_pool_id="userPoolId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__795e0d8c9020bae33ab9db3f6ee92c3fab525494304ae5dab03e978b4cc524b3)
            check_type(argname="argument group_name", value=group_name, expected_type=type_hints["group_name"])
            check_type(argname="argument username", value=username, expected_type=type_hints["username"])
            check_type(argname="argument user_pool_id", value=user_pool_id, expected_type=type_hints["user_pool_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "group_name": group_name,
            "username": username,
            "user_pool_id": user_pool_id,
        }

    @builtins.property
    def group_name(self) -> builtins.str:
        '''The name of the group that you want to add your user to.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolusertogroupattachment.html#cfn-cognito-userpoolusertogroupattachment-groupname
        '''
        result = self._values.get("group_name")
        assert result is not None, "Required property 'group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def username(self) -> builtins.str:
        '''The user's username.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolusertogroupattachment.html#cfn-cognito-userpoolusertogroupattachment-username
        '''
        result = self._values.get("username")
        assert result is not None, "Required property 'username' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def user_pool_id(self) -> builtins.str:
        '''The ID of the user pool that contains the group that you want to add the user to.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolusertogroupattachment.html#cfn-cognito-userpoolusertogroupattachment-userpoolid
        '''
        result = self._values.get("user_pool_id")
        assert result is not None, "Required property 'user_pool_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUserPoolUserToGroupAttachmentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ClientAttributes(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.ClientAttributes",
):
    '''A set of attributes, useful to set Read and Write attributes.

    :exampleMetadata: infused

    Example::

        pool = cognito.UserPool(self, "Pool")
        
        client_write_attributes = (cognito.ClientAttributes()).with_standard_attributes(fullname=True, email=True).with_custom_attributes("favoritePizza", "favoriteBeverage")
        
        client_read_attributes = client_write_attributes.with_standard_attributes(email_verified=True).with_custom_attributes("pointsEarned")
        
        pool.add_client("app-client",
            # ...
            read_attributes=client_read_attributes,
            write_attributes=client_write_attributes
        )
    '''

    def __init__(self) -> None:
        '''Creates a ClientAttributes with the specified attributes.

        :default: - a ClientAttributes object without any attributes
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="attributes")
    def attributes(self) -> typing.List[builtins.str]:
        '''The list of attributes represented by this ClientAttributes.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "attributes", []))

    @jsii.member(jsii_name="withCustomAttributes")
    def with_custom_attributes(self, *attributes: builtins.str) -> "ClientAttributes":
        '''Creates a custom ClientAttributes with the specified attributes.

        :param attributes: a list of custom attributes to add to the set.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d8336134a8317e883dbac378420185ca4e7010dbcb09eb3d51235fc3568fbcee)
            check_type(argname="argument attributes", value=attributes, expected_type=typing.Tuple[type_hints["attributes"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("ClientAttributes", jsii.invoke(self, "withCustomAttributes", [*attributes]))

    @jsii.member(jsii_name="withStandardAttributes")
    def with_standard_attributes(
        self,
        *,
        address: typing.Optional[builtins.bool] = None,
        birthdate: typing.Optional[builtins.bool] = None,
        email: typing.Optional[builtins.bool] = None,
        email_verified: typing.Optional[builtins.bool] = None,
        family_name: typing.Optional[builtins.bool] = None,
        fullname: typing.Optional[builtins.bool] = None,
        gender: typing.Optional[builtins.bool] = None,
        given_name: typing.Optional[builtins.bool] = None,
        last_update_time: typing.Optional[builtins.bool] = None,
        locale: typing.Optional[builtins.bool] = None,
        middle_name: typing.Optional[builtins.bool] = None,
        nickname: typing.Optional[builtins.bool] = None,
        phone_number: typing.Optional[builtins.bool] = None,
        phone_number_verified: typing.Optional[builtins.bool] = None,
        preferred_username: typing.Optional[builtins.bool] = None,
        profile_page: typing.Optional[builtins.bool] = None,
        profile_picture: typing.Optional[builtins.bool] = None,
        timezone: typing.Optional[builtins.bool] = None,
        website: typing.Optional[builtins.bool] = None,
    ) -> "ClientAttributes":
        '''Creates a custom ClientAttributes with the specified attributes.

        :param address: The user's postal address. Default: false
        :param birthdate: The user's birthday, represented as an ISO 8601:2004 format. Default: false
        :param email: The user's e-mail address, represented as an RFC 5322 [RFC5322] addr-spec. Default: false
        :param email_verified: Whether the email address has been verified. Default: false
        :param family_name: The surname or last name of the user. Default: false
        :param fullname: The user's full name in displayable form, including all name parts, titles and suffixes. Default: false
        :param gender: The user's gender. Default: false
        :param given_name: The user's first name or give name. Default: false
        :param last_update_time: The time, the user's information was last updated. Default: false
        :param locale: The user's locale, represented as a BCP47 [RFC5646] language tag. Default: false
        :param middle_name: The user's middle name. Default: false
        :param nickname: The user's nickname or casual name. Default: false
        :param phone_number: The user's telephone number. Default: false
        :param phone_number_verified: Whether the phone number has been verified. Default: false
        :param preferred_username: The user's preferred username, different from the immutable user name. Default: false
        :param profile_page: The URL to the user's profile page. Default: false
        :param profile_picture: The URL to the user's profile picture. Default: false
        :param timezone: The user's time zone. Default: false
        :param website: The URL to the user's web page or blog. Default: false
        '''
        attributes = StandardAttributesMask(
            address=address,
            birthdate=birthdate,
            email=email,
            email_verified=email_verified,
            family_name=family_name,
            fullname=fullname,
            gender=gender,
            given_name=given_name,
            last_update_time=last_update_time,
            locale=locale,
            middle_name=middle_name,
            nickname=nickname,
            phone_number=phone_number,
            phone_number_verified=phone_number_verified,
            preferred_username=preferred_username,
            profile_page=profile_page,
            profile_picture=profile_picture,
            timezone=timezone,
            website=website,
        )

        return typing.cast("ClientAttributes", jsii.invoke(self, "withStandardAttributes", [attributes]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CognitoDomainOptions",
    jsii_struct_bases=[],
    name_mapping={"domain_prefix": "domainPrefix"},
)
class CognitoDomainOptions:
    def __init__(self, *, domain_prefix: builtins.str) -> None:
        '''Options while specifying a cognito prefix domain.

        :param domain_prefix: The prefix to the Cognito hosted domain name that will be associated with the user pool.

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-assign-domain-prefix.html
        :exampleMetadata: infused

        Example::

            pool = cognito.UserPool(self, "Pool")
            
            pool.add_domain("CognitoDomain",
                cognito_domain=cognito.CognitoDomainOptions(
                    domain_prefix="my-awesome-app"
                )
            )
            
            certificate_arn = "arn:aws:acm:us-east-1:123456789012:certificate/11-3336f1-44483d-adc7-9cd375c5169d"
            
            domain_cert = certificatemanager.Certificate.from_certificate_arn(self, "domainCert", certificate_arn)
            pool.add_domain("CustomDomain",
                custom_domain=cognito.CustomDomainOptions(
                    domain_name="user.myapp.com",
                    certificate=domain_cert
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ba8eb526d56a50ba180bfdc1cecba07c9bef023027dc66e58aa425ec1363f28f)
            check_type(argname="argument domain_prefix", value=domain_prefix, expected_type=type_hints["domain_prefix"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "domain_prefix": domain_prefix,
        }

    @builtins.property
    def domain_prefix(self) -> builtins.str:
        '''The prefix to the Cognito hosted domain name that will be associated with the user pool.'''
        result = self._values.get("domain_prefix")
        assert result is not None, "Required property 'domain_prefix' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CognitoDomainOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CustomAttributeConfig",
    jsii_struct_bases=[],
    name_mapping={
        "data_type": "dataType",
        "mutable": "mutable",
        "number_constraints": "numberConstraints",
        "string_constraints": "stringConstraints",
    },
)
class CustomAttributeConfig:
    def __init__(
        self,
        *,
        data_type: builtins.str,
        mutable: typing.Optional[builtins.bool] = None,
        number_constraints: typing.Optional[typing.Union["NumberAttributeConstraints", typing.Dict[builtins.str, typing.Any]]] = None,
        string_constraints: typing.Optional[typing.Union["StringAttributeConstraints", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Configuration that will be fed into CloudFormation for any custom attribute type.

        :param data_type: The data type of the custom attribute.
        :param mutable: Specifies whether the value of the attribute can be changed. For any user pool attribute that's mapped to an identity provider attribute, you must set this parameter to true. Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider. If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute. Default: false
        :param number_constraints: The constraints for a custom attribute of the 'Number' data type. Default: - None.
        :param string_constraints: The constraints for a custom attribute of 'String' data type. Default: - None.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            custom_attribute_config = cognito.CustomAttributeConfig(
                data_type="dataType",
            
                # the properties below are optional
                mutable=False,
                number_constraints=cognito.NumberAttributeConstraints(
                    max=123,
                    min=123
                ),
                string_constraints=cognito.StringAttributeConstraints(
                    max_len=123,
                    min_len=123
                )
            )
        '''
        if isinstance(number_constraints, dict):
            number_constraints = NumberAttributeConstraints(**number_constraints)
        if isinstance(string_constraints, dict):
            string_constraints = StringAttributeConstraints(**string_constraints)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c43f23a759d2790f0497ff70752305bfbbab58e89ea40928057b015346f4b0c3)
            check_type(argname="argument data_type", value=data_type, expected_type=type_hints["data_type"])
            check_type(argname="argument mutable", value=mutable, expected_type=type_hints["mutable"])
            check_type(argname="argument number_constraints", value=number_constraints, expected_type=type_hints["number_constraints"])
            check_type(argname="argument string_constraints", value=string_constraints, expected_type=type_hints["string_constraints"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "data_type": data_type,
        }
        if mutable is not None:
            self._values["mutable"] = mutable
        if number_constraints is not None:
            self._values["number_constraints"] = number_constraints
        if string_constraints is not None:
            self._values["string_constraints"] = string_constraints

    @builtins.property
    def data_type(self) -> builtins.str:
        '''The data type of the custom attribute.

        :see: https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SchemaAttributeType.html#CognitoUserPools-Type-SchemaAttributeType-AttributeDataType
        '''
        result = self._values.get("data_type")
        assert result is not None, "Required property 'data_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def mutable(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether the value of the attribute can be changed.

        For any user pool attribute that's mapped to an identity provider attribute, you must set this parameter to true.
        Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider.
        If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute.

        :default: false
        '''
        result = self._values.get("mutable")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def number_constraints(self) -> typing.Optional["NumberAttributeConstraints"]:
        '''The constraints for a custom attribute of the 'Number' data type.

        :default: - None.
        '''
        result = self._values.get("number_constraints")
        return typing.cast(typing.Optional["NumberAttributeConstraints"], result)

    @builtins.property
    def string_constraints(self) -> typing.Optional["StringAttributeConstraints"]:
        '''The constraints for a custom attribute of 'String' data type.

        :default: - None.
        '''
        result = self._values.get("string_constraints")
        return typing.cast(typing.Optional["StringAttributeConstraints"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CustomAttributeConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CustomAttributeProps",
    jsii_struct_bases=[],
    name_mapping={"mutable": "mutable"},
)
class CustomAttributeProps:
    def __init__(self, *, mutable: typing.Optional[builtins.bool] = None) -> None:
        '''Constraints that can be applied to a custom attribute of any type.

        :param mutable: Specifies whether the value of the attribute can be changed. For any user pool attribute that's mapped to an identity provider attribute, you must set this parameter to true. Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider. If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute. Default: false

        :exampleMetadata: infused

        Example::

            cognito.UserPool(self, "myuserpool",
                # ...
                standard_attributes=cognito.StandardAttributes(
                    fullname=cognito.StandardAttribute(
                        required=True,
                        mutable=False
                    ),
                    address=cognito.StandardAttribute(
                        required=False,
                        mutable=True
                    )
                ),
                custom_attributes={
                    "myappid": cognito.StringAttribute(min_len=5, max_len=15, mutable=False),
                    "callingcode": cognito.NumberAttribute(min=1, max=3, mutable=True),
                    "isEmployee": cognito.BooleanAttribute(mutable=True),
                    "joinedOn": cognito.DateTimeAttribute()
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__10625133998d7d4e7fac2876192aae9fef70ad558e321a4d1d6e442f4f99af42)
            check_type(argname="argument mutable", value=mutable, expected_type=type_hints["mutable"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if mutable is not None:
            self._values["mutable"] = mutable

    @builtins.property
    def mutable(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether the value of the attribute can be changed.

        For any user pool attribute that's mapped to an identity provider attribute, you must set this parameter to true.
        Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider.
        If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute.

        :default: false
        '''
        result = self._values.get("mutable")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CustomAttributeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.CustomDomainOptions",
    jsii_struct_bases=[],
    name_mapping={"certificate": "certificate", "domain_name": "domainName"},
)
class CustomDomainOptions:
    def __init__(
        self,
        *,
        certificate: _ICertificate_c194c70b,
        domain_name: builtins.str,
    ) -> None:
        '''Options while specifying custom domain.

        :param certificate: The certificate to associate with this domain.
        :param domain_name: The custom domain name that you would like to associate with this User Pool.

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-add-custom-domain.html
        :exampleMetadata: infused

        Example::

            pool = cognito.UserPool(self, "Pool")
            
            pool.add_domain("CognitoDomain",
                cognito_domain=cognito.CognitoDomainOptions(
                    domain_prefix="my-awesome-app"
                )
            )
            
            certificate_arn = "arn:aws:acm:us-east-1:123456789012:certificate/11-3336f1-44483d-adc7-9cd375c5169d"
            
            domain_cert = certificatemanager.Certificate.from_certificate_arn(self, "domainCert", certificate_arn)
            pool.add_domain("CustomDomain",
                custom_domain=cognito.CustomDomainOptions(
                    domain_name="user.myapp.com",
                    certificate=domain_cert
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3849940c95623161d3f30b88c048744039d927bcb31494b462f19441b4be2cbe)
            check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "certificate": certificate,
            "domain_name": domain_name,
        }

    @builtins.property
    def certificate(self) -> _ICertificate_c194c70b:
        '''The certificate to associate with this domain.'''
        result = self._values.get("certificate")
        assert result is not None, "Required property 'certificate' is missing"
        return typing.cast(_ICertificate_c194c70b, result)

    @builtins.property
    def domain_name(self) -> builtins.str:
        '''The custom domain name that you would like to associate with this User Pool.'''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CustomDomainOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_cognito.CustomThreatProtectionMode")
class CustomThreatProtectionMode(enum.Enum):
    '''The Type of Threat Protection Enabled for Custom Authentication.

    This feature only functions if your FeaturePlan is set to FeaturePlan.PLUS

    :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cognito-userpool-userpooladdons.html
    '''

    FULL_FUNCTION = "FULL_FUNCTION"
    '''Cognito automatically takes preventative actions in response to different levels of risk that you configure for your user pool.'''
    AUDIT_ONLY = "AUDIT_ONLY"
    '''Cognito gathers metrics on detected risks, but doesn't take automatic action.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.DeviceTracking",
    jsii_struct_bases=[],
    name_mapping={
        "challenge_required_on_new_device": "challengeRequiredOnNewDevice",
        "device_only_remembered_on_user_prompt": "deviceOnlyRememberedOnUserPrompt",
    },
)
class DeviceTracking:
    def __init__(
        self,
        *,
        challenge_required_on_new_device: builtins.bool,
        device_only_remembered_on_user_prompt: builtins.bool,
    ) -> None:
        '''Device tracking settings.

        :param challenge_required_on_new_device: Indicates whether a challenge is required on a new device. Only applicable to a new device. Default: false
        :param device_only_remembered_on_user_prompt: If true, a device is only remembered on user prompt. Default: false

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html
        :exampleMetadata: infused

        Example::

            cognito.UserPool(self, "myuserpool",
                # ...
                device_tracking=cognito.DeviceTracking(
                    challenge_required_on_new_device=True,
                    device_only_remembered_on_user_prompt=True
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__22d0ff37f33d5dc8927b832397fbf5dd702d37d7b312d0951c247fa3eae106f5)
            check_type(argname="argument challenge_required_on_new_device", value=challenge_required_on_new_device, expected_type=type_hints["challenge_required_on_new_device"])
            check_type(argname="argument device_only_remembered_on_user_prompt", value=device_only_remembered_on_user_prompt, expected_type=type_hints["device_only_remembered_on_user_prompt"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "challenge_required_on_new_device": challenge_required_on_new_device,
            "device_only_remembered_on_user_prompt": device_only_remembered_on_user_prompt,
        }

    @builtins.property
    def challenge_required_on_new_device(self) -> builtins.bool:
        '''Indicates whether a challenge is required on a new device.

        Only applicable to a new device.

        :default: false

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html
        '''
        result = self._values.get("challenge_required_on_new_device")
        assert result is not None, "Required property 'challenge_required_on_new_device' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def device_only_remembered_on_user_prompt(self) -> builtins.bool:
        '''If true, a device is only remembered on user prompt.

        :default: false

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html
        '''
        result = self._values.get("device_only_remembered_on_user_prompt")
        assert result is not None, "Required property 'device_only_remembered_on_user_prompt' is missing"
        return typing.cast(builtins.bool, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DeviceTracking(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.EmailSettings",
    jsii_struct_bases=[],
    name_mapping={"from_": "from", "reply_to": "replyTo"},
)
class EmailSettings:
    def __init__(
        self,
        *,
        from_: typing.Optional[builtins.str] = None,
        reply_to: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Email settings for the user pool.

        :param from_: The 'from' address on the emails received by the user. Default: noreply@verificationemail.com
        :param reply_to: The 'replyTo' address on the emails received by the user as defined by IETF RFC-5322. When set, most email clients recognize to change 'to' line to this address when a reply is drafted. Default: - Not set.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            email_settings = cognito.EmailSettings(
                from="from",
                reply_to="replyTo"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__701ae53f91916b708c28b316d16f169e30e8e66c32936f4c4b3c96edd3f149ca)
            check_type(argname="argument from_", value=from_, expected_type=type_hints["from_"])
            check_type(argname="argument reply_to", value=reply_to, expected_type=type_hints["reply_to"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if from_ is not None:
            self._values["from_"] = from_
        if reply_to is not None:
            self._values["reply_to"] = reply_to

    @builtins.property
    def from_(self) -> typing.Optional[builtins.str]:
        '''The 'from' address on the emails received by the user.

        :default: noreply@verificationemail.com
        '''
        result = self._values.get("from_")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def reply_to(self) -> typing.Optional[builtins.str]:
        '''The 'replyTo' address on the emails received by the user as defined by IETF RFC-5322.

        When set, most email clients recognize to change 'to' line to this address when a reply is drafted.

        :default: - Not set.
        '''
        result = self._values.get("reply_to")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmailSettings(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_cognito.FeaturePlan")
class FeaturePlan(enum.Enum):
    '''The user pool feature plan, or tier.

    :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-sign-in-feature-plans.html
    :exampleMetadata: infused

    Example::

        cognito.UserPool(self, "myuserpool",
            sign_in_policy=cognito.SignInPolicy(
                allowed_first_auth_factors=cognito.AllowedFirstAuthFactors(password=True)
            ),
            feature_plan=cognito.FeaturePlan.LITE
        )
    '''

    LITE = "LITE"
    '''Lite feature plan.'''
    ESSENTIALS = "ESSENTIALS"
    '''Essentials feature plan.'''
    PLUS = "PLUS"
    '''Plus feature plan.'''


@jsii.interface(jsii_type="aws-cdk-lib.aws_cognito.ICustomAttribute")
class ICustomAttribute(typing_extensions.Protocol):
    '''Represents a custom attribute type.'''

    @jsii.member(jsii_name="bind")
    def bind(self) -> CustomAttributeConfig:
        '''Bind this custom attribute type to the values as expected by CloudFormation.'''
        ...


class _ICustomAttributeProxy:
    '''Represents a custom attribute type.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_cognito.ICustomAttribute"

    @jsii.member(jsii_name="bind")
    def bind(self) -> CustomAttributeConfig:
        '''Bind this custom attribute type to the values as expected by CloudFormation.'''
        return typing.cast(CustomAttributeConfig, jsii.invoke(self, "bind", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ICustomAttribute).__jsii_proxy_class__ = lambda : _ICustomAttributeProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_cognito.IUserPool")
class IUserPool(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents a Cognito UserPool.'''

    @builtins.property
    @jsii.member(jsii_name="identityProviders")
    def identity_providers(self) -> typing.List["IUserPoolIdentityProvider"]:
        '''Get all identity providers registered with this user pool.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="userPoolArn")
    def user_pool_arn(self) -> builtins.str:
        '''The ARN of this user pool resource.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The physical ID of this user pool resource.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="userPoolProviderName")
    def user_pool_provider_name(self) -> builtins.str:
        '''The provider name of this user pool resource.

        :attribute: true
        '''
        ...

    @jsii.member(jsii_name="addClient")
    def add_client(
        self,
        id: builtins.str,
        *,
        access_token_validity: typing.Optional[_Duration_4839e8c3] = None,
        analytics: typing.Optional[typing.Union[AnalyticsConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
        auth_flows: typing.Optional[typing.Union[AuthFlow, typing.Dict[builtins.str, typing.Any]]] = None,
        auth_session_validity: typing.Optional[_Duration_4839e8c3] = None,
        disable_o_auth: typing.Optional[builtins.bool] = None,
        enable_propagate_additional_user_context_data: typing.Optional[builtins.bool] = None,
        enable_token_revocation: typing.Optional[builtins.bool] = None,
        generate_secret: typing.Optional[builtins.bool] = None,
        id_token_validity: typing.Optional[_Duration_4839e8c3] = None,
        o_auth: typing.Optional[typing.Union["OAuthSettings", typing.Dict[builtins.str, typing.Any]]] = None,
        prevent_user_existence_errors: typing.Optional[builtins.bool] = None,
        read_attributes: typing.Optional[ClientAttributes] = None,
        refresh_token_validity: typing.Optional[_Duration_4839e8c3] = None,
        supported_identity_providers: typing.Optional[typing.Sequence["UserPoolClientIdentityProvider"]] = None,
        user_pool_client_name: typing.Optional[builtins.str] = None,
        write_attributes: typing.Optional[ClientAttributes] = None,
    ) -> "UserPoolClient":
        '''Add a new app client to this user pool.

        :param id: -
        :param access_token_validity: Validity of the access token. Values between 5 minutes and 1 day are valid. The duration can not be longer than the refresh token validity. Default: Duration.minutes(60)
        :param analytics: The analytics configuration for this client. Default: - no analytics configuration
        :param auth_flows: The set of OAuth authentication flows to enable on the client. Default: - If you don't specify a value, your user client supports ALLOW_REFRESH_TOKEN_AUTH, ALLOW_USER_SRP_AUTH, and ALLOW_CUSTOM_AUTH.
        :param auth_session_validity: Cognito creates a session token for each API request in an authentication flow. AuthSessionValidity is the duration, in minutes, of that session token. see defaults in ``AuthSessionValidity``. Valid duration is from 3 to 15 minutes. Default: - Duration.minutes(3)
        :param disable_o_auth: Turns off all OAuth interactions for this client. Default: false
        :param enable_propagate_additional_user_context_data: Enable the propagation of additional user context data. You can only activate enablePropagateAdditionalUserContextData in an app client that has a client secret. Default: false for new user pool clients
        :param enable_token_revocation: Enable token revocation for this client. Default: true for new user pool clients
        :param generate_secret: Whether to generate a client secret. Default: false
        :param id_token_validity: Validity of the ID token. Values between 5 minutes and 1 day are valid. The duration can not be longer than the refresh token validity. Default: Duration.minutes(60)
        :param o_auth: OAuth settings for this client to interact with the app. An error is thrown when this is specified and ``disableOAuth`` is set. Default: - see defaults in ``OAuthSettings``. meaningless if ``disableOAuth`` is set.
        :param prevent_user_existence_errors: Whether Cognito returns a UserNotFoundException exception when the user does not exist in the user pool (false), or whether it returns another type of error that doesn't reveal the user's absence. Default: false
        :param read_attributes: The set of attributes this client will be able to read. Default: - all standard and custom attributes
        :param refresh_token_validity: Validity of the refresh token. Values between 60 minutes and 10 years are valid. Default: Duration.days(30)
        :param supported_identity_providers: The list of identity providers that users should be able to use to sign in using this client. Default: - supports all identity providers that are registered with the user pool. If the user pool and/or identity providers are imported, either specify this option explicitly or ensure that the identity providers are registered with the user pool using the ``UserPool.registerIdentityProvider()`` API.
        :param user_pool_client_name: Name of the application client. Default: - cloudformation generated name
        :param write_attributes: The set of attributes this client will be able to write. Default: - all standard and custom attributes

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-client-apps.html
        '''
        ...

    @jsii.member(jsii_name="addDomain")
    def add_domain(
        self,
        id: builtins.str,
        *,
        cognito_domain: typing.Optional[typing.Union[CognitoDomainOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_domain: typing.Optional[typing.Union[CustomDomainOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        managed_login_version: typing.Optional["ManagedLoginVersion"] = None,
    ) -> "UserPoolDomain":
        '''Associate a domain to this user pool.

        :param id: -
        :param cognito_domain: Associate a cognito prefix domain with your user pool Either ``customDomain`` or ``cognitoDomain`` must be specified. Default: - not set if ``customDomain`` is specified, otherwise, throws an error.
        :param custom_domain: Associate a custom domain with your user pool Either ``customDomain`` or ``cognitoDomain`` must be specified. Default: - not set if ``cognitoDomain`` is specified, otherwise, throws an error.
        :param managed_login_version: A version that indicates the state of managed login. This choice applies to all app clients that host services at the domain. Default: undefined - Cognito default setting is ManagedLoginVersion.CLASSIC_HOSTED_UI

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-assign-domain.html
        '''
        ...

    @jsii.member(jsii_name="addGroup")
    def add_group(
        self,
        id: builtins.str,
        *,
        description: typing.Optional[builtins.str] = None,
        group_name: typing.Optional[builtins.str] = None,
        precedence: typing.Optional[jsii.Number] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
    ) -> "UserPoolGroup":
        '''Add a new group to this user pool.

        :param id: -
        :param description: A string containing the description of the group. Default: - no description
        :param group_name: The name of the group. Must be unique. Default: - auto generate a name
        :param precedence: A non-negative integer value that specifies the precedence of this group relative to the other groups that a user can belong to in the user pool. Zero is the highest precedence value. Groups with lower Precedence values take precedence over groups with higher or null Precedence values. If a user belongs to two or more groups, it is the group with the lowest precedence value whose role ARN is given in the user's tokens for the cognito:roles and cognito:preferred_role claims. Two groups can have the same Precedence value. If this happens, neither group takes precedence over the other. If two groups with the same Precedence have the same role ARN, that role is used in the cognito:preferred_role claim in tokens for users in each group. If the two groups have different role ARNs, the cognito:preferred_role claim isn't set in users' tokens. Default: - null
        :param role: The role for the group. Default: - no description

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-user-groups.html
        '''
        ...

    @jsii.member(jsii_name="addResourceServer")
    def add_resource_server(
        self,
        id: builtins.str,
        *,
        identifier: builtins.str,
        scopes: typing.Optional[typing.Sequence["ResourceServerScope"]] = None,
        user_pool_resource_server_name: typing.Optional[builtins.str] = None,
    ) -> "UserPoolResourceServer":
        '''Add a new resource server to this user pool.

        :param id: -
        :param identifier: A unique resource server identifier for the resource server.
        :param scopes: Oauth scopes. Default: - No scopes will be added
        :param user_pool_resource_server_name: A friendly name for the resource server. Default: - same as ``identifier``

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-resource-servers.html
        '''
        ...

    @jsii.member(jsii_name="grant")
    def grant(
        self,
        grantee: _IGrantable_71c4f5de,
        *actions: builtins.str,
    ) -> _Grant_a7ae64f8:
        '''Adds an IAM policy statement associated with this user pool to an IAM principal's policy.

        :param grantee: -
        :param actions: -
        '''
        ...

    @jsii.member(jsii_name="registerIdentityProvider")
    def register_identity_provider(self, provider: "IUserPoolIdentityProvider") -> None:
        '''Register an identity provider with this user pool.

        :param provider: -
        '''
        ...


class _IUserPoolProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents a Cognito UserPool.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_cognito.IUserPool"

    @builtins.property
    @jsii.member(jsii_name="identityProviders")
    def identity_providers(self) -> typing.List["IUserPoolIdentityProvider"]:
        '''Get all identity providers registered with this user pool.'''
        return typing.cast(typing.List["IUserPoolIdentityProvider"], jsii.get(self, "identityProviders"))

    @builtins.property
    @jsii.member(jsii_name="userPoolArn")
    def user_pool_arn(self) -> builtins.str:
        '''The ARN of this user pool resource.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "userPoolArn"))

    @builtins.property
    @jsii.member(jsii_name="userPoolId")
    def user_pool_id(self) -> builtins.str:
        '''The physical ID of this user pool resource.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "userPoolId"))

    @builtins.property
    @jsii.member(jsii_name="userPoolProviderName")
    def user_pool_provider_name(self) -> builtins.str:
        '''The provider name of this user pool resource.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "userPoolProviderName"))

    @jsii.member(jsii_name="addClient")
    def add_client(
        self,
        id: builtins.str,
        *,
        access_token_validity: typing.Optional[_Duration_4839e8c3] = None,
        analytics: typing.Optional[typing.Union[AnalyticsConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
        auth_flows: typing.Optional[typing.Union[AuthFlow, typing.Dict[builtins.str, typing.Any]]] = None,
        auth_session_validity: typing.Optional[_Duration_4839e8c3] = None,
        disable_o_auth: typing.Optional[builtins.bool] = None,
        enable_propagate_additional_user_context_data: typing.Optional[builtins.bool] = None,
        enable_token_revocation: typing.Optional[builtins.bool] = None,
        generate_secret: typing.Optional[builtins.bool] = None,
        id_token_validity: typing.Optional[_Duration_4839e8c3] = None,
        o_auth: typing.Optional[typing.Union["OAuthSettings", typing.Dict[builtins.str, typing.Any]]] = None,
        prevent_user_existence_errors: typing.Optional[builtins.bool] = None,
        read_attributes: typing.Optional[ClientAttributes] = None,
        refresh_token_validity: typing.Optional[_Duration_4839e8c3] = None,
        supported_identity_providers: typing.Optional[typing.Sequence["UserPoolClientIdentityProvider"]] = None,
        user_pool_client_name: typing.Optional[builtins.str] = None,
        write_attributes: typing.Optional[ClientAttributes] = None,
    ) -> "UserPoolClient":
        '''Add a new app client to this user pool.

        :param id: -
        :param access_token_validity: Validity of the access token. Values between 5 minutes and 1 day are valid. The duration can not be longer than the refresh token validity. Default: Duration.minutes(60)
        :param analytics: The analytics configuration for this client. Default: - no analytics configuration
        :param auth_flows: The set of OAuth authentication flows to enable on the client. Default: - If you don't specify a value, your user client supports ALLOW_REFRESH_TOKEN_AUTH, ALLOW_USER_SRP_AUTH, and ALLOW_CUSTOM_AUTH.
        :param auth_session_validity: Cognito creates a session token for each API request in an authentication flow. AuthSessionValidity is the duration, in minutes, of that session token. see defaults in ``AuthSessionValidity``. Valid duration is from 3 to 15 minutes. Default: - Duration.minutes(3)
        :param disable_o_auth: Turns off all OAuth interactions for this client. Default: false
        :param enable_propagate_additional_user_context_data: Enable the propagation of additional user context data. You can only activate enablePropagateAdditionalUserContextData in an app client that has a client secret. Default: false for new user pool clients
        :param enable_token_revocation: Enable token revocation for this client. Default: true for new user pool clients
        :param generate_secret: Whether to generate a client secret. Default: false
        :param id_token_validity: Validity of the ID token. Values between 5 minutes and 1 day are valid. The duration can not be longer than the refresh token validity. Default: Duration.minutes(60)
        :param o_auth: OAuth settings for this client to interact with the app. An error is thrown when this is specified and ``disableOAuth`` is set. Default: - see defaults in ``OAuthSettings``. meaningless if ``disableOAuth`` is set.
        :param prevent_user_existence_errors: Whether Cognito returns a UserNotFoundException exception when the user does not exist in the user pool (false), or whether it returns another type of error that doesn't reveal the user's absence. Default: false
        :param read_attributes: The set of attributes this client will be able to read. Default: - all standard and custom attributes
        :param refresh_token_validity: Validity of the refresh token. Values between 60 minutes and 10 years are valid. Default: Duration.days(30)
        :param supported_identity_providers: The list of identity providers that users should be able to use to sign in using this client. Default: - supports all identity providers that are registered with the user pool. If the user pool and/or identity providers are imported, either specify this option explicitly or ensure that the identity providers are registered with the user pool using the ``UserPool.registerIdentityProvider()`` API.
        :param user_pool_client_name: Name of the application client. Default: - cloudformation generated name
        :param write_attributes: The set of attributes this client will be able to write. Default: - all standard and custom attributes

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-client-apps.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6eaa0ebaf797c6ac4bac11bd73d9ad61c50892a9450e0ff58809034348d71166)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = UserPoolClientOptions(
            access_token_validity=access_token_validity,
            analytics=analytics,
            auth_flows=auth_flows,
            auth_session_validity=auth_session_validity,
            disable_o_auth=disable_o_auth,
            enable_propagate_additional_user_context_data=enable_propagate_additional_user_context_data,
            enable_token_revocation=enable_token_revocation,
            generate_secret=generate_secret,
            id_token_validity=id_token_validity,
            o_auth=o_auth,
            prevent_user_existence_errors=prevent_user_existence_errors,
            read_attributes=read_attributes,
            refresh_token_validity=refresh_token_validity,
            supported_identity_providers=supported_identity_providers,
            user_pool_client_name=user_pool_client_name,
            write_attributes=write_attributes,
        )

        return typing.cast("UserPoolClient", jsii.invoke(self, "addClient", [id, options]))

    @jsii.member(jsii_name="addDomain")
    def add_domain(
        self,
        id: builtins.str,
        *,
        cognito_domain: typing.Optional[typing.Union[CognitoDomainOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        custom_domain: typing.Optional[typing.Union[CustomDomainOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        managed_login_version: typing.Optional["ManagedLoginVersion"] = None,
    ) -> "UserPoolDomain":
        '''Associate a domain to this user pool.

        :param id: -
        :param cognito_domain: Associate a cognito prefix domain with your user pool Either ``customDomain`` or ``cognitoDomain`` must be specified. Default: - not set if ``customDomain`` is specified, otherwise, throws an error.
        :param custom_domain: Associate a custom domain with your user pool Either ``customDomain`` or ``cognitoDomain`` must be specified. Default: - not set if ``cognitoDomain`` is specified, otherwise, throws an error.
        :param managed_login_version: A version that indicates the state of managed login. This choice applies to all app clients that host services at the domain. Default: undefined - Cognito default setting is ManagedLoginVersion.CLASSIC_HOSTED_UI

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-assign-domain.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__792921e0d9eecd6253eadd31c7fba82fdce9c0ba38f25dcba7dcd063e7b1a458)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = UserPoolDomainOptions(
            cognito_domain=cognito_domain,
            custom_domain=custom_domain,
            managed_login_version=managed_login_version,
        )

        return typing.cast("UserPoolDomain", jsii.invoke(self, "addDomain", [id, options]))

    @jsii.member(jsii_name="addGroup")
    def add_group(
        self,
        id: builtins.str,
        *,
        description: typing.Optional[builtins.str] = None,
        group_name: typing.Optional[builtins.str] = None,
        precedence: typing.Optional[jsii.Number] = None,
        role: typing.Optional[_IRole_235f5d8e] = None,
    ) -> "UserPoolGroup":
        '''Add a new group to this user pool.

        :param id: -
        :param description: A string containing the description of the group. Default: - no description
        :param group_name: The name of the group. Must be unique. Default: - auto generate a name
        :param precedence: A non-negative integer value that specifies the precedence of this group relative to the other groups that a user can belong to in the user pool. Zero is the highest precedence value. Groups with lower Precedence values take precedence over groups with higher or null Precedence values. If a user belongs to two or more groups, it is the group with the lowest precedence value whose role ARN is given in the user's tokens for the cognito:roles and cognito:preferred_role claims. Two groups can have the same Precedence value. If this happens, neither group takes precedence over the other. If two groups with the same Precedence have the same role ARN, that role is used in the cognito:preferred_role claim in tokens for users in each group. If the two groups have different role ARNs, the cognito:preferred_role claim isn't set in users' tokens. Default: - null
        :param role: The role for the group. Default: - no description

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-user-groups.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e70d406698753c50dbab4e4d1f9837fc55e7c713f52b3937d20745b5ab2a221e)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = UserPoolGroupOptions(
            description=description,
            group_name=group_name,
            precedence=precedence,
            role=role,
        )

        return typing.cast("UserPoolGroup", jsii.invoke(self, "addGroup", [id, options]))

    @jsii.member(jsii_name="addResourceServer")
    def add_resource_server(
        self,
        id: builtins.str,
        *,
        identifier: builtins.str,
        scopes: typing.Optional[typing.Sequence["ResourceServerScope"]] = None,
        user_pool_resource_server_name: typing.Optional[builtins.str] = None,
    ) -> "UserPoolResourceServer":
        '''Add a new resource server to this user pool.

        :param id: -
        :param identifier: A unique resource server identifier for the resource server.
        :param scopes: Oauth scopes. Default: - No scopes will be added
        :param user_pool_resource_server_name: A friendly name for the resource server. Default: - same as ``identifier``

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-resource-servers.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6e7f4643c9bff39b5095e7aa370612aed9ce88bfde927b1cbbd7b3a21df157a2)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = UserPoolResourceServerOptions(
            identifier=identifier,
            scopes=scopes,
            user_pool_resource_server_name=user_pool_resource_server_name,
        )

        return typing.cast("UserPoolResourceServer", jsii.invoke(self, "addResourceServer", [id, options]))

    @jsii.member(jsii_name="grant")
    def grant(
        self,
        grantee: _IGrantable_71c4f5de,
        *actions: builtins.str,
    ) -> _Grant_a7ae64f8:
        '''Adds an IAM policy statement associated with this user pool to an IAM principal's policy.

        :param grantee: -
        :param actions: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d3a4c5e8a4da82b394a07fc77ae4614ba300e995b69a268ebd9c41430ee2f184)
            check_type(argname="argument grantee", value=grantee, expected_type=type_hints["grantee"])
            check_type(argname="argument actions", value=actions, expected_type=typing.Tuple[type_hints["actions"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast(_Grant_a7ae64f8, jsii.invoke(self, "grant", [grantee, *actions]))

    @jsii.member(jsii_name="registerIdentityProvider")
    def register_identity_provider(self, provider: "IUserPoolIdentityProvider") -> None:
        '''Register an identity provider with this user pool.

        :param provider: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2136c7c5f1f487c9abafff099214714783c966a26ecba5010a34c74ee3f5cb01)
            check_type(argname="argument provider", value=provider, expected_type=type_hints["provider"])
        return typing.cast(None, jsii.invoke(self, "registerIdentityProvider", [provider]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IUserPool).__jsii_proxy_class__ = lambda : _IUserPoolProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_cognito.IUserPoolClient")
class IUserPoolClient(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents a Cognito user pool client.'''

    @builtins.property
    @jsii.member(jsii_name="userPoolClientId")
    def user_pool_client_id(self) -> builtins.str:
        '''Name of the application client.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="userPoolClientSecret")
    def user_pool_client_secret(self) -> _SecretValue_3dd0ddae:
        '''The generated client secret.

        Only available if the "generateSecret" props is set to true

        :attribute: true
        '''
        ...


class _IUserPoolClientProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents a Cognito user pool client.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_cognito.IUserPoolClient"

    @builtins.property
    @jsii.member(jsii_name="userPoolClientId")
    def user_pool_client_id(self) -> builtins.str:
        '''Name of the application client.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "userPoolClientId"))

    @builtins.property
    @jsii.member(jsii_name="userPoolClientSecret")
    def user_pool_client_secret(self) -> _SecretValue_3dd0ddae:
        '''The generated client secret.

        Only available if the "generateSecret" props is set to true

        :attribute: true
        '''
        return typing.cast(_SecretValue_3dd0ddae, jsii.get(self, "userPoolClientSecret"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IUserPoolClient).__jsii_proxy_class__ = lambda : _IUserPoolClientProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_cognito.IUserPoolDomain")
class IUserPoolDomain(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents a user pool domain.'''

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> builtins.str:
        '''The domain that was specified to be created.

        If ``customDomain`` was selected, this holds the full domain name that was specified.
        If the ``cognitoDomain`` was used, it contains the prefix to the Cognito hosted domain.

        :attribute: true
        '''
        ...


class _IUserPoolDomainProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents a user pool domain.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_cognito.IUserPoolDomain"

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> builtins.str:
        '''The domain that was specified to be created.

        If ``customDomain`` was selected, this holds the full domain name that was specified.
        If the ``cognitoDomain`` was used, it contains the prefix to the Cognito hosted domain.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "domainName"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IUserPoolDomain).__jsii_proxy_class__ = lambda : _IUserPoolDomainProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_cognito.IUserPoolGroup")
class IUserPoolGroup(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents a user pool group.'''

    @builtins.property
    @jsii.member(jsii_name="groupName")
    def group_name(self) -> builtins.str:
        '''The user group name.

        :attribute: true
        '''
        ...


class _IUserPoolGroupProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents a user pool group.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_cognito.IUserPoolGroup"

    @builtins.property
    @jsii.member(jsii_name="groupName")
    def group_name(self) -> builtins.str:
        '''The user group name.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "groupName"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IUserPoolGroup).__jsii_proxy_class__ = lambda : _IUserPoolGroupProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_cognito.IUserPoolIdentityProvider")
class IUserPoolIdentityProvider(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents a UserPoolIdentityProvider.'''

    @builtins.property
    @jsii.member(jsii_name="providerName")
    def provider_name(self) -> builtins.str:
        '''The primary identifier of this identity provider.

        :attribute: true
        '''
        ...


class _IUserPoolIdentityProviderProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents a UserPoolIdentityProvider.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_cognito.IUserPoolIdentityProvider"

    @builtins.property
    @jsii.member(jsii_name="providerName")
    def provider_name(self) -> builtins.str:
        '''The primary identifier of this identity provider.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "providerName"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IUserPoolIdentityProvider).__jsii_proxy_class__ = lambda : _IUserPoolIdentityProviderProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_cognito.IUserPoolResourceServer")
class IUserPoolResourceServer(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents a Cognito user pool resource server.'''

    @builtins.property
    @jsii.member(jsii_name="userPoolResourceServerId")
    def user_pool_resource_server_id(self) -> builtins.str:
        '''Resource server id.

        :attribute: true
        '''
        ...


class _IUserPoolResourceServerProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents a Cognito user pool resource server.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_cognito.IUserPoolResourceServer"

    @builtins.property
    @jsii.member(jsii_name="userPoolResourceServerId")
    def user_pool_resource_server_id(self) -> builtins.str:
        '''Resource server id.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "userPoolResourceServerId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IUserPoolResourceServer).__jsii_proxy_class__ = lambda : _IUserPoolResourceServerProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.KeepOriginalAttrs",
    jsii_struct_bases=[],
    name_mapping={"email": "email", "phone": "phone"},
)
class KeepOriginalAttrs:
    def __init__(
        self,
        *,
        email: typing.Optional[builtins.bool] = None,
        phone: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Attributes that will be kept until the user verifies the changed attribute.

        :param email: Whether the email address of the user should remain the original value until the new email address is verified. Default: - false
        :param phone: Whether the phone number of the user should remain the original value until the new phone number is verified. Default: - false

        :exampleMetadata: infused

        Example::

            cognito.UserPool(self, "myuserpool",
                # ...
                sign_in_aliases=cognito.SignInAliases(username=True),
                auto_verify=cognito.AutoVerifiedAttrs(email=True, phone=True),
                keep_original=cognito.KeepOriginalAttrs(
                    email=True,
                    phone=True
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__582b87d2527ed12a750d5fbc9acd4a298bed24b0a4cc1ae8299811b8002d93c5)
            check_type(argname="argument email", value=email, expected_type=type_hints["email"])
            check_type(argname="argument phone", value=phone, expected_type=type_hints["phone"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if email is not None:
            self._values["email"] = email
        if phone is not None:
            self._values["phone"] = phone

    @builtins.property
    def email(self) -> typing.Optional[builtins.bool]:
        '''Whether the email address of the user should remain the original value until the new email address is verified.

        :default: - false
        '''
        result = self._values.get("email")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def phone(self) -> typing.Optional[builtins.bool]:
        '''Whether the phone number of the user should remain the original value until the new phone number is verified.

        :default: - false
        '''
        result = self._values.get("phone")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "KeepOriginalAttrs(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_cognito.LambdaVersion")
class LambdaVersion(enum.Enum):
    '''The user pool trigger version of the request that Amazon Cognito sends to your Lambda function.

    :exampleMetadata: infused

    Example::

        # userpool: cognito.UserPool
        # pre_token_generation_fn: lambda.Function
        
        
        userpool.add_trigger(cognito.UserPoolOperation.PRE_TOKEN_GENERATION_CONFIG, pre_token_generation_fn, cognito.LambdaVersion.V2_0)
    '''

    V1_0 = "V1_0"
    '''V1_0 trigger.'''
    V2_0 = "V2_0"
    '''V2_0 trigger.

    This is supported only for PRE_TOKEN_GENERATION trigger.
    '''
    V3_0 = "V3_0"
    '''V3_0 trigger.

    This is supported only for PRE_TOKEN_GENERATION trigger.
    '''


@jsii.enum(jsii_type="aws-cdk-lib.aws_cognito.ManagedLoginVersion")
class ManagedLoginVersion(enum.Enum):
    '''The branding version of managed login for the domain.

    :exampleMetadata: infused

    Example::

        pool = cognito.UserPool(self, "Pool")
        
        # Use the new managed login page
        pool.add_domain("CognitoDomainWithBlandingDesignManagedLogin",
            cognito_domain=cognito.CognitoDomainOptions(
                domain_prefix="blanding-design-ui"
            ),
            managed_login_version=cognito.ManagedLoginVersion.NEWER_MANAGED_LOGIN
        )
        
        # Use the classic hosted UI
        pool.add_domain("DomainWithClassicHostedUi",
            cognito_domain=cognito.CognitoDomainOptions(
                domain_prefix="classic-hosted-ui"
            ),
            managed_login_version=cognito.ManagedLoginVersion.CLASSIC_HOSTED_UI
        )
    '''

    CLASSIC_HOSTED_UI = "CLASSIC_HOSTED_UI"
    '''The classic hosted UI.'''
    NEWER_MANAGED_LOGIN = "NEWER_MANAGED_LOGIN"
    '''The newer managed login with the branding designer.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_cognito.Mfa")
class Mfa(enum.Enum):
    '''The different ways in which a user pool's MFA enforcement can be configured.

    :see: https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
    :exampleMetadata: infused

    Example::

        cognito.UserPool(self, "myuserpool",
            # ...
            mfa=cognito.Mfa.REQUIRED,
            mfa_second_factor=cognito.MfaSecondFactor(
                sms=True,
                otp=True,
                email=False
            )
        )
    '''

    OFF = "OFF"
    '''Users are not required to use MFA for sign in, and cannot configure one.'''
    OPTIONAL = "OPTIONAL"
    '''Users are not required to use MFA for sign in, but can configure one if they so choose to.'''
    REQUIRED = "REQUIRED"
    '''Users are required to configure an MFA, and have to use it to sign in.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.MfaSecondFactor",
    jsii_struct_bases=[],
    name_mapping={"otp": "otp", "sms": "sms", "email": "email"},
)
class MfaSecondFactor:
    def __init__(
        self,
        *,
        otp: builtins.bool,
        sms: builtins.bool,
        email: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''The different ways in which a user pool can obtain their MFA token for sign in.

        :param otp: The MFA token is a time-based one time password that is generated by a hardware or software token. Default: false
        :param sms: The MFA token is sent to the user via SMS to their verified phone numbers. Default: true
        :param email: The MFA token is sent to the user via EMAIL. To enable email-based MFA, set ``email`` property to the Amazon SES email-sending configuration and set ``feturePlan`` to ``FeaturePlan.ESSENTIALS`` or ``FeaturePlan.PLUS`` Default: false

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
        :exampleMetadata: infused

        Example::

            cognito.UserPool(self, "myuserpool",
                # ...
                mfa=cognito.Mfa.REQUIRED,
                mfa_second_factor=cognito.MfaSecondFactor(
                    sms=True,
                    otp=True,
                    email=False
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd702328c87129c710daf17c5d6eb94fe91bfd54ab5fe2157a787ca7021ea599)
            check_type(argname="argument otp", value=otp, expected_type=type_hints["otp"])
            check_type(argname="argument sms", value=sms, expected_type=type_hints["sms"])
            check_type(argname="argument email", value=email, expected_type=type_hints["email"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "otp": otp,
            "sms": sms,
        }
        if email is not None:
            self._values["email"] = email

    @builtins.property
    def otp(self) -> builtins.bool:
        '''The MFA token is a time-based one time password that is generated by a hardware or software token.

        :default: false

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa-totp.html
        '''
        result = self._values.get("otp")
        assert result is not None, "Required property 'otp' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def sms(self) -> builtins.bool:
        '''The MFA token is sent to the user via SMS to their verified phone numbers.

        :default: true

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa-sms-email-message.html
        '''
        result = self._values.get("sms")
        assert result is not None, "Required property 'sms' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def email(self) -> typing.Optional[builtins.bool]:
        '''The MFA token is sent to the user via EMAIL.

        To enable email-based MFA, set ``email`` property to the Amazon SES email-sending configuration
        and set ``feturePlan`` to ``FeaturePlan.ESSENTIALS`` or ``FeaturePlan.PLUS``

        :default: false

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa-sms-email-message.html
        '''
        result = self._values.get("email")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "MfaSecondFactor(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(ICustomAttribute)
class NumberAttribute(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.NumberAttribute",
):
    '''The Number custom attribute type.

    :exampleMetadata: infused

    Example::

        cognito.UserPool(self, "myuserpool",
            # ...
            standard_attributes=cognito.StandardAttributes(
                fullname=cognito.StandardAttribute(
                    required=True,
                    mutable=False
                ),
                address=cognito.StandardAttribute(
                    required=False,
                    mutable=True
                )
            ),
            custom_attributes={
                "myappid": cognito.StringAttribute(min_len=5, max_len=15, mutable=False),
                "callingcode": cognito.NumberAttribute(min=1, max=3, mutable=True),
                "isEmployee": cognito.BooleanAttribute(mutable=True),
                "joinedOn": cognito.DateTimeAttribute()
            }
        )
    '''

    def __init__(
        self,
        *,
        max: typing.Optional[jsii.Number] = None,
        min: typing.Optional[jsii.Number] = None,
        mutable: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param max: Maximum value of this attribute. Default: - no maximum value
        :param min: Minimum value of this attribute. Default: - no minimum value
        :param mutable: Specifies whether the value of the attribute can be changed. For any user pool attribute that's mapped to an identity provider attribute, you must set this parameter to true. Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider. If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute. Default: false
        '''
        props = NumberAttributeProps(max=max, min=min, mutable=mutable)

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="bind")
    def bind(self) -> CustomAttributeConfig:
        '''Bind this custom attribute type to the values as expected by CloudFormation.'''
        return typing.cast(CustomAttributeConfig, jsii.invoke(self, "bind", []))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.NumberAttributeConstraints",
    jsii_struct_bases=[],
    name_mapping={"max": "max", "min": "min"},
)
class NumberAttributeConstraints:
    def __init__(
        self,
        *,
        max: typing.Optional[jsii.Number] = None,
        min: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Constraints that can be applied to a custom attribute of number type.

        :param max: Maximum value of this attribute. Default: - no maximum value
        :param min: Minimum value of this attribute. Default: - no minimum value

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            number_attribute_constraints = cognito.NumberAttributeConstraints(
                max=123,
                min=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__68475d8f214b5069a5668d8a206de9ffcb9d1608e01852482d3b7af5d5d4599d)
            check_type(argname="argument max", value=max, expected_type=type_hints["max"])
            check_type(argname="argument min", value=min, expected_type=type_hints["min"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if max is not None:
            self._values["max"] = max
        if min is not None:
            self._values["min"] = min

    @builtins.property
    def max(self) -> typing.Optional[jsii.Number]:
        '''Maximum value of this attribute.

        :default: - no maximum value
        '''
        result = self._values.get("max")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min(self) -> typing.Optional[jsii.Number]:
        '''Minimum value of this attribute.

        :default: - no minimum value
        '''
        result = self._values.get("min")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "NumberAttributeConstraints(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.NumberAttributeProps",
    jsii_struct_bases=[NumberAttributeConstraints, CustomAttributeProps],
    name_mapping={"max": "max", "min": "min", "mutable": "mutable"},
)
class NumberAttributeProps(NumberAttributeConstraints, CustomAttributeProps):
    def __init__(
        self,
        *,
        max: typing.Optional[jsii.Number] = None,
        min: typing.Optional[jsii.Number] = None,
        mutable: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Props for NumberAttr.

        :param max: Maximum value of this attribute. Default: - no maximum value
        :param min: Minimum value of this attribute. Default: - no minimum value
        :param mutable: Specifies whether the value of the attribute can be changed. For any user pool attribute that's mapped to an identity provider attribute, you must set this parameter to true. Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider. If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute. Default: false

        :exampleMetadata: infused

        Example::

            cognito.UserPool(self, "myuserpool",
                # ...
                standard_attributes=cognito.StandardAttributes(
                    fullname=cognito.StandardAttribute(
                        required=True,
                        mutable=False
                    ),
                    address=cognito.StandardAttribute(
                        required=False,
                        mutable=True
                    )
                ),
                custom_attributes={
                    "myappid": cognito.StringAttribute(min_len=5, max_len=15, mutable=False),
                    "callingcode": cognito.NumberAttribute(min=1, max=3, mutable=True),
                    "isEmployee": cognito.BooleanAttribute(mutable=True),
                    "joinedOn": cognito.DateTimeAttribute()
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd7e8741a6775cc3a6eb71a3a56e03f2b42a7a15ca168c037eb31742783f6587)
            check_type(argname="argument max", value=max, expected_type=type_hints["max"])
            check_type(argname="argument min", value=min, expected_type=type_hints["min"])
            check_type(argname="argument mutable", value=mutable, expected_type=type_hints["mutable"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if max is not None:
            self._values["max"] = max
        if min is not None:
            self._values["min"] = min
        if mutable is not None:
            self._values["mutable"] = mutable

    @builtins.property
    def max(self) -> typing.Optional[jsii.Number]:
        '''Maximum value of this attribute.

        :default: - no maximum value
        '''
        result = self._values.get("max")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min(self) -> typing.Optional[jsii.Number]:
        '''Minimum value of this attribute.

        :default: - no minimum value
        '''
        result = self._values.get("min")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def mutable(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether the value of the attribute can be changed.

        For any user pool attribute that's mapped to an identity provider attribute, you must set this parameter to true.
        Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider.
        If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute.

        :default: false
        '''
        result = self._values.get("mutable")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "NumberAttributeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.OAuthFlows",
    jsii_struct_bases=[],
    name_mapping={
        "authorization_code_grant": "authorizationCodeGrant",
        "client_credentials": "clientCredentials",
        "implicit_code_grant": "implicitCodeGrant",
    },
)
class OAuthFlows:
    def __init__(
        self,
        *,
        authorization_code_grant: typing.Optional[builtins.bool] = None,
        client_credentials: typing.Optional[builtins.bool] = None,
        implicit_code_grant: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Types of OAuth grant flows.

        :param authorization_code_grant: Initiate an authorization code grant flow, which provides an authorization code as the response. Default: false
        :param client_credentials: Client should get the access token and ID token from the token endpoint using a combination of client and client_secret. Default: false
        :param implicit_code_grant: The client should get the access token and ID token directly. Default: false

        :see: - the 'Allowed OAuth Flows' section at https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
        :exampleMetadata: infused

        Example::

            userpool = cognito.UserPool(self, "UserPool")
            client = userpool.add_client("Client",
                # ...
                o_auth=cognito.OAuthSettings(
                    flows=cognito.OAuthFlows(
                        implicit_code_grant=True
                    ),
                    callback_urls=["https://myapp.com/home", "https://myapp.com/users"
                    ]
                )
            )
            domain = userpool.add_domain("Domain")
            sign_in_url = domain.sign_in_url(client,
                redirect_uri="https://myapp.com/home"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9aa5ba53667700cdf4d217354a8b54484e7e7fa22880031485734768e7a3bf8b)
            check_type(argname="argument authorization_code_grant", value=authorization_code_grant, expected_type=type_hints["authorization_code_grant"])
            check_type(argname="argument client_credentials", value=client_credentials, expected_type=type_hints["client_credentials"])
            check_type(argname="argument implicit_code_grant", value=implicit_code_grant, expected_type=type_hints["implicit_code_grant"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if authorization_code_grant is not None:
            self._values["authorization_code_grant"] = authorization_code_grant
        if client_credentials is not None:
            self._values["client_credentials"] = client_credentials
        if implicit_code_grant is not None:
            self._values["implicit_code_grant"] = implicit_code_grant

    @builtins.property
    def authorization_code_grant(self) -> typing.Optional[builtins.bool]:
        '''Initiate an authorization code grant flow, which provides an authorization code as the response.

        :default: false
        '''
        result = self._values.get("authorization_code_grant")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def client_credentials(self) -> typing.Optional[builtins.bool]:
        '''Client should get the access token and ID token from the token endpoint using a combination of client and client_secret.

        :default: false
        '''
        result = self._values.get("client_credentials")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def implicit_code_grant(self) -> typing.Optional[builtins.bool]:
        '''The client should get the access token and ID token directly.

        :default: false
        '''
        result = self._values.get("implicit_code_grant")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "OAuthFlows(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class OAuthScope(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.OAuthScope",
):
    '''OAuth scopes that are allowed with this client.

    :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
    :exampleMetadata: infused

    Example::

        pool = cognito.UserPool(self, "Pool")
        
        read_only_scope = cognito.ResourceServerScope(scope_name="read", scope_description="Read-only access")
        full_access_scope = cognito.ResourceServerScope(scope_name="*", scope_description="Full access")
        
        user_server = pool.add_resource_server("ResourceServer",
            identifier="users",
            scopes=[read_only_scope, full_access_scope]
        )
        
        read_only_client = pool.add_client("read-only-client",
            # ...
            o_auth=cognito.OAuthSettings(
                # ...
                scopes=[cognito.OAuthScope.resource_server(user_server, read_only_scope)]
            )
        )
        
        full_access_client = pool.add_client("full-access-client",
            # ...
            o_auth=cognito.OAuthSettings(
                # ...
                scopes=[cognito.OAuthScope.resource_server(user_server, full_access_scope)]
            )
        )
    '''

    @jsii.member(jsii_name="custom")
    @builtins.classmethod
    def custom(cls, name: builtins.str) -> "OAuthScope":
        '''Custom scope is one that you define for your own resource server in the Resource Servers.

        The format is 'resource-server-identifier/scope'.

        :param name: -

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0d22a67a9b2e1a844ab0a61af92d0eabd4b5c51b59a1303e77dab9d15737ec8e)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        return typing.cast("OAuthScope", jsii.sinvoke(cls, "custom", [name]))

    @jsii.member(jsii_name="resourceServer")
    @builtins.classmethod
    def resource_server(
        cls,
        server: IUserPoolResourceServer,
        scope: "ResourceServerScope",
    ) -> "OAuthScope":
        '''Adds a custom scope that's tied to a resource server in your stack.

        :param server: -
        :param scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b161664ae7bfb1838c23dd6d7b35e7c009bedb80545b644df24485522ad2b6ac)
            check_type(argname="argument server", value=server, expected_type=type_hints["server"])
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("OAuthScope", jsii.sinvoke(cls, "resourceServer", [server, scope]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="COGNITO_ADMIN")
    def COGNITO_ADMIN(cls) -> "OAuthScope":
        '''Grants access to Amazon Cognito User Pool API operations that require access tokens, such as UpdateUserAttributes and VerifyUserAttribute.'''
        return typing.cast("OAuthScope", jsii.sget(cls, "COGNITO_ADMIN"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="EMAIL")
    def EMAIL(cls) -> "OAuthScope":
        '''Grants access to the 'email' and 'email_verified' claims.

        Automatically includes access to ``OAuthScope.OPENID``.
        '''
        return typing.cast("OAuthScope", jsii.sget(cls, "EMAIL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="OPENID")
    def OPENID(cls) -> "OAuthScope":
        '''Returns all user attributes in the ID token that are readable by the client.'''
        return typing.cast("OAuthScope", jsii.sget(cls, "OPENID"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PHONE")
    def PHONE(cls) -> "OAuthScope":
        '''Grants access to the 'phone_number' and 'phone_number_verified' claims.

        Automatically includes access to ``OAuthScope.OPENID``.
        '''
        return typing.cast("OAuthScope", jsii.sget(cls, "PHONE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="PROFILE")
    def PROFILE(cls) -> "OAuthScope":
        '''Grants access to all user attributes that are readable by the client Automatically includes access to ``OAuthScope.OPENID``.'''
        return typing.cast("OAuthScope", jsii.sget(cls, "PROFILE"))

    @builtins.property
    @jsii.member(jsii_name="scopeName")
    def scope_name(self) -> builtins.str:
        '''The name of this scope as recognized by CloudFormation.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-allowedoauthscopes
        '''
        return typing.cast(builtins.str, jsii.get(self, "scopeName"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.OAuthSettings",
    jsii_struct_bases=[],
    name_mapping={
        "callback_urls": "callbackUrls",
        "default_redirect_uri": "defaultRedirectUri",
        "flows": "flows",
        "logout_urls": "logoutUrls",
        "scopes": "scopes",
    },
)
class OAuthSettings:
    def __init__(
        self,
        *,
        callback_urls: typing.Optional[typing.Sequence[builtins.str]] = None,
        default_redirect_uri: typing.Optional[builtins.str] = None,
        flows: typing.Optional[typing.Union[OAuthFlows, typing.Dict[builtins.str, typing.Any]]] = None,
        logout_urls: typing.Optional[typing.Sequence[builtins.str]] = None,
        scopes: typing.Optional[typing.Sequence[OAuthScope]] = None,
    ) -> None:
        '''OAuth settings to configure the interaction between the app and this client.

        :param callback_urls: List of allowed redirect URLs for the identity providers. Default: - ['https://example.com'] if either authorizationCodeGrant or implicitCodeGrant flows are enabled, no callback URLs otherwise.
        :param default_redirect_uri: The default redirect URI. Must be in the ``callbackUrls`` list. A redirect URI must: - Be an absolute URI - Be registered with the authorization server. - Not include a fragment component. Default: - no default redirect URI
        :param flows: OAuth flows that are allowed with this client. Default: {authorizationCodeGrant:true,implicitCodeGrant:true}
        :param logout_urls: List of allowed logout URLs for the identity providers. Default: - no logout URLs
        :param scopes: OAuth scopes that are allowed with this client. Default: [OAuthScope.PHONE,OAuthScope.EMAIL,OAuthScope.OPENID,OAuthScope.PROFILE,OAuthScope.COGNITO_ADMIN]

        :exampleMetadata: infused

        Example::

            pool = cognito.UserPool(self, "Pool")
            
            read_only_scope = cognito.ResourceServerScope(scope_name="read", scope_description="Read-only access")
            full_access_scope = cognito.ResourceServerScope(scope_name="*", scope_description="Full access")
            
            user_server = pool.add_resource_server("ResourceServer",
                identifier="users",
                scopes=[read_only_scope, full_access_scope]
            )
            
            read_only_client = pool.add_client("read-only-client",
                # ...
                o_auth=cognito.OAuthSettings(
                    # ...
                    scopes=[cognito.OAuthScope.resource_server(user_server, read_only_scope)]
                )
            )
            
            full_access_client = pool.add_client("full-access-client",
                # ...
                o_auth=cognito.OAuthSettings(
                    # ...
                    scopes=[cognito.OAuthScope.resource_server(user_server, full_access_scope)]
                )
            )
        '''
        if isinstance(flows, dict):
            flows = OAuthFlows(**flows)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9322fee969a93d621a972830c0fca7f51a16c368668af0d6da355939ca39b708)
            check_type(argname="argument callback_urls", value=callback_urls, expected_type=type_hints["callback_urls"])
            check_type(argname="argument default_redirect_uri", value=default_redirect_uri, expected_type=type_hints["default_redirect_uri"])
            check_type(argname="argument flows", value=flows, expected_type=type_hints["flows"])
            check_type(argname="argument logout_urls", value=logout_urls, expected_type=type_hints["logout_urls"])
            check_type(argname="argument scopes", value=scopes, expected_type=type_hints["scopes"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if callback_urls is not None:
            self._values["callback_urls"] = callback_urls
        if default_redirect_uri is not None:
            self._values["default_redirect_uri"] = default_redirect_uri
        if flows is not None:
            self._values["flows"] = flows
        if logout_urls is not None:
            self._values["logout_urls"] = logout_urls
        if scopes is not None:
            self._values["scopes"] = scopes

    @builtins.property
    def callback_urls(self) -> typing.Optional[typing.List[builtins.str]]:
        '''List of allowed redirect URLs for the identity providers.

        :default: - ['https://example.com'] if either authorizationCodeGrant or implicitCodeGrant flows are enabled, no callback URLs otherwise.
        '''
        result = self._values.get("callback_urls")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def default_redirect_uri(self) -> typing.Optional[builtins.str]:
        '''The default redirect URI. Must be in the ``callbackUrls`` list.

        A redirect URI must:

        - Be an absolute URI
        - Be registered with the authorization server.
        - Not include a fragment component.

        :default: - no default redirect URI

        :see:

        https://tools.ietf.org/html/rfc6749#section-3.1.2

        Amazon Cognito requires HTTPS over HTTP except for http://localhost for testing purposes only.

        App callback URLs such as myapp://example are also supported.
        '''
        result = self._values.get("default_redirect_uri")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def flows(self) -> typing.Optional[OAuthFlows]:
        '''OAuth flows that are allowed with this client.

        :default: {authorizationCodeGrant:true,implicitCodeGrant:true}

        :see: - the 'Allowed OAuth Flows' section at https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
        '''
        result = self._values.get("flows")
        return typing.cast(typing.Optional[OAuthFlows], result)

    @builtins.property
    def logout_urls(self) -> typing.Optional[typing.List[builtins.str]]:
        '''List of allowed logout URLs for the identity providers.

        :default: - no logout URLs
        '''
        result = self._values.get("logout_urls")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def scopes(self) -> typing.Optional[typing.List[OAuthScope]]:
        '''OAuth scopes that are allowed with this client.

        :default: [OAuthScope.PHONE,OAuthScope.EMAIL,OAuthScope.OPENID,OAuthScope.PROFILE,OAuthScope.COGNITO_ADMIN]

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
        '''
        result = self._values.get("scopes")
        return typing.cast(typing.Optional[typing.List[OAuthScope]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "OAuthSettings(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_cognito.OidcAttributeRequestMethod")
class OidcAttributeRequestMethod(enum.Enum):
    '''The method to use to request attributes.'''

    GET = "GET"
    '''GET.'''
    POST = "POST"
    '''POST.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.OidcEndpoints",
    jsii_struct_bases=[],
    name_mapping={
        "authorization": "authorization",
        "jwks_uri": "jwksUri",
        "token": "token",
        "user_info": "userInfo",
    },
)
class OidcEndpoints:
    def __init__(
        self,
        *,
        authorization: builtins.str,
        jwks_uri: builtins.str,
        token: builtins.str,
        user_info: builtins.str,
    ) -> None:
        '''OpenID Connect endpoints.

        :param authorization: Authorization endpoint.
        :param jwks_uri: Jwks_uri endpoint.
        :param token: Token endpoint.
        :param user_info: UserInfo endpoint.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_cognito as cognito
            
            oidc_endpoints = cognito.OidcEndpoints(
                authorization="authorization",
                jwks_uri="jwksUri",
                token="token",
                user_info="userInfo"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4661d75bb2c69171e02ca544be91ebb3d950a95f070c8471007dc24e8a0bcb33)
            check_type(argname="argument authorization", value=authorization, expected_type=type_hints["authorization"])
            check_type(argname="argument jwks_uri", value=jwks_uri, expected_type=type_hints["jwks_uri"])
            check_type(argname="argument token", value=token, expected_type=type_hints["token"])
            check_type(argname="argument user_info", value=user_info, expected_type=type_hints["user_info"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "authorization": authorization,
            "jwks_uri": jwks_uri,
            "token": token,
            "user_info": user_info,
        }

    @builtins.property
    def authorization(self) -> builtins.str:
        '''Authorization endpoint.'''
        result = self._values.get("authorization")
        assert result is not None, "Required property 'authorization' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def jwks_uri(self) -> builtins.str:
        '''Jwks_uri endpoint.'''
        result = self._values.get("jwks_uri")
        assert result is not None, "Required property 'jwks_uri' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def token(self) -> builtins.str:
        '''Token endpoint.'''
        result = self._values.get("token")
        assert result is not None, "Required property 'token' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def user_info(self) -> builtins.str:
        '''UserInfo endpoint.'''
        result = self._values.get("user_info")
        assert result is not None, "Required property 'user_info' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "OidcEndpoints(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_cognito.PasskeyUserVerification")
class PasskeyUserVerification(enum.Enum):
    '''The user-pool treatment for MFA with a passkey.

    :see: https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow-methods.html#amazon-cognito-user-pools-authentication-flow-methods-passkey
    :exampleMetadata: infused

    Example::

        cognito.UserPool(self, "myuserpool",
            sign_in_policy=cognito.SignInPolicy(
                allowed_first_auth_factors=cognito.AllowedFirstAuthFactors(password=True, passkey=True)
            ),
            passkey_relying_party_id="auth.example.com",
            passkey_user_verification=cognito.PasskeyUserVerification.REQUIRED
        )
    '''

    PREFERRED = "PREFERRED"
    '''Passkey MFA is preferred.'''
    REQUIRED = "REQUIRED"
    '''Passkey MFA is required.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.PasswordPolicy",
    jsii_struct_bases=[],
    name_mapping={
        "min_length": "minLength",
        "password_history_size": "passwordHistorySize",
        "require_digits": "requireDigits",
        "require_lowercase": "requireLowercase",
        "require_symbols": "requireSymbols",
        "require_uppercase": "requireUppercase",
        "temp_password_validity": "tempPasswordValidity",
    },
)
class PasswordPolicy:
    def __init__(
        self,
        *,
        min_length: typing.Optional[jsii.Number] = None,
        password_history_size: typing.Optional[jsii.Number] = None,
        require_digits: typing.Optional[builtins.bool] = None,
        require_lowercase: typing.Optional[builtins.bool] = None,
        require_symbols: typing.Optional[builtins.bool] = None,
        require_uppercase: typing.Optional[builtins.bool] = None,
        temp_password_validity: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''Password policy for User Pools.

        :param min_length: Minimum length required for a user's password. Default: 8
        :param password_history_size: The number of previous passwords that you want Amazon Cognito to restrict each user from reusing. ``passwordHistorySize`` can not be set when ``featurePlan`` is ``FeaturePlan.LITE``. Default: undefined - Cognito default setting is no restriction
        :param require_digits: Whether the user is required to have digits in their password. Default: true
        :param require_lowercase: Whether the user is required to have lowercase characters in their password. Default: true
        :param require_symbols: Whether the user is required to have symbols in their password. Default: true
        :param require_uppercase: Whether the user is required to have uppercase characters in their password. Default: true
        :param temp_password_validity: The length of time the temporary password generated by an admin is valid. This must be provided as whole days, like Duration.days(3) or Duration.hours(48). Fractional days, such as Duration.hours(20), will generate an error. Default: Duration.days(7)

        :exampleMetadata: infused

        Example::

            cognito.UserPool(self, "myuserpool",
                # ...
                password_policy=cognito.PasswordPolicy(
                    min_length=12,
                    require_lowercase=True,
                    require_uppercase=True,
                    require_digits=True,
                    require_symbols=True,
                    temp_password_validity=Duration.days(3)
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e884ae0f43a6b00e4ef287afca45bf6a9a7abbd94b7979efb6efa2a9cc11012b)
            check_type(argname="argument min_length", value=min_length, expected_type=type_hints["min_length"])
            check_type(argname="argument password_history_size", value=password_history_size, expected_type=type_hints["password_history_size"])
            check_type(argname="argument require_digits", value=require_digits, expected_type=type_hints["require_digits"])
            check_type(argname="argument require_lowercase", value=require_lowercase, expected_type=type_hints["require_lowercase"])
            check_type(argname="argument require_symbols", value=require_symbols, expected_type=type_hints["require_symbols"])
            check_type(argname="argument require_uppercase", value=require_uppercase, expected_type=type_hints["require_uppercase"])
            check_type(argname="argument temp_password_validity", value=temp_password_validity, expected_type=type_hints["temp_password_validity"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if min_length is not None:
            self._values["min_length"] = min_length
        if password_history_size is not None:
            self._values["password_history_size"] = password_history_size
        if require_digits is not None:
            self._values["require_digits"] = require_digits
        if require_lowercase is not None:
            self._values["require_lowercase"] = require_lowercase
        if require_symbols is not None:
            self._values["require_symbols"] = require_symbols
        if require_uppercase is not None:
            self._values["require_uppercase"] = require_uppercase
        if temp_password_validity is not None:
            self._values["temp_password_validity"] = temp_password_validity

    @builtins.property
    def min_length(self) -> typing.Optional[jsii.Number]:
        '''Minimum length required for a user's password.

        :default: 8
        '''
        result = self._values.get("min_length")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def password_history_size(self) -> typing.Optional[jsii.Number]:
        '''The number of previous passwords that you want Amazon Cognito to restrict each user from reusing.

        ``passwordHistorySize`` can not be set when ``featurePlan`` is ``FeaturePlan.LITE``.

        :default: undefined - Cognito default setting is no restriction
        '''
        result = self._values.get("password_history_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def require_digits(self) -> typing.Optional[builtins.bool]:
        '''Whether the user is required to have digits in their password.

        :default: true
        '''
        result = self._values.get("require_digits")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def require_lowercase(self) -> typing.Optional[builtins.bool]:
        '''Whether the user is required to have lowercase characters in their password.

        :default: true
        '''
        result = self._values.get("require_lowercase")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def require_symbols(self) -> typing.Optional[builtins.bool]:
        '''Whether the user is required to have symbols in their password.

        :default: true
        '''
        result = self._values.get("require_symbols")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def require_uppercase(self) -> typing.Optional[builtins.bool]:
        '''Whether the user is required to have uppercase characters in their password.

        :default: true
        '''
        result = self._values.get("require_uppercase")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def temp_password_validity(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The length of time the temporary password generated by an admin is valid.

        This must be provided as whole days, like Duration.days(3) or Duration.hours(48).
        Fractional days, such as Duration.hours(20), will generate an error.

        :default: Duration.days(7)
        '''
        result = self._values.get("temp_password_validity")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PasswordPolicy(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ProviderAttribute(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.ProviderAttribute",
):
    '''An attribute available from a third party identity provider.

    :exampleMetadata: infused

    Example::

        userpool = cognito.UserPool(self, "Pool")
        
        cognito.UserPoolIdentityProviderAmazon(self, "Amazon",
            client_id="amzn-client-id",
            client_secret="amzn-client-secret",
            user_pool=userpool,
            attribute_mapping=cognito.AttributeMapping(
                email=cognito.ProviderAttribute.AMAZON_EMAIL,
                website=cognito.ProviderAttribute.other("url"),  # use other() when an attribute is not pre-defined in the CDK
                custom={
                    # custom user pool attributes go here
                    "unique_id": cognito.ProviderAttribute.AMAZON_USER_ID
                }
            )
        )
    '''

    @jsii.member(jsii_name="other")
    @builtins.classmethod
    def other(cls, attribute_name: builtins.str) -> "ProviderAttribute":
        '''Use this to specify an attribute from the identity provider that is not pre-defined in the CDK.

        :param attribute_name: the attribute value string as recognized by the provider.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__933f1896d041fda75048e9688dc72dd0e429afd1f6731de6fbfbbfef6d64e7d2)
            check_type(argname="argument attribute_name", value=attribute_name, expected_type=type_hints["attribute_name"])
        return typing.cast("ProviderAttribute", jsii.sinvoke(cls, "other", [attribute_name]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_EMAIL")
    def AMAZON_EMAIL(cls) -> "ProviderAttribute":
        '''The email attribute provided by Amazon.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "AMAZON_EMAIL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_NAME")
    def AMAZON_NAME(cls) -> "ProviderAttribute":
        '''The name attribute provided by Amazon.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "AMAZON_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_POSTAL_CODE")
    def AMAZON_POSTAL_CODE(cls) -> "ProviderAttribute":
        '''The postal code attribute provided by Amazon.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "AMAZON_POSTAL_CODE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="AMAZON_USER_ID")
    def AMAZON_USER_ID(cls) -> "ProviderAttribute":
        '''The user id attribute provided by Amazon.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "AMAZON_USER_ID"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APPLE_EMAIL")
    def APPLE_EMAIL(cls) -> "ProviderAttribute":
        '''The email attribute provided by Apple.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "APPLE_EMAIL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APPLE_EMAIL_VERIFIED")
    def APPLE_EMAIL_VERIFIED(cls) -> "ProviderAttribute":
        '''The email verified attribute provided by Apple.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "APPLE_EMAIL_VERIFIED"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APPLE_FIRST_NAME")
    def APPLE_FIRST_NAME(cls) -> "ProviderAttribute":
        '''The first name attribute provided by Apple.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "APPLE_FIRST_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APPLE_LAST_NAME")
    def APPLE_LAST_NAME(cls) -> "ProviderAttribute":
        '''The last name attribute provided by Apple.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "APPLE_LAST_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="APPLE_NAME")
    def APPLE_NAME(cls) -> "ProviderAttribute":
        '''The name attribute provided by Apple.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "APPLE_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FACEBOOK_BIRTHDAY")
    def FACEBOOK_BIRTHDAY(cls) -> "ProviderAttribute":
        '''The birthday attribute provided by Facebook.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "FACEBOOK_BIRTHDAY"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FACEBOOK_EMAIL")
    def FACEBOOK_EMAIL(cls) -> "ProviderAttribute":
        '''The email attribute provided by Facebook.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "FACEBOOK_EMAIL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FACEBOOK_FIRST_NAME")
    def FACEBOOK_FIRST_NAME(cls) -> "ProviderAttribute":
        '''The first name attribute provided by Facebook.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "FACEBOOK_FIRST_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FACEBOOK_GENDER")
    def FACEBOOK_GENDER(cls) -> "ProviderAttribute":
        '''The gender attribute provided by Facebook.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "FACEBOOK_GENDER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FACEBOOK_ID")
    def FACEBOOK_ID(cls) -> "ProviderAttribute":
        '''The user id attribute provided by Facebook.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "FACEBOOK_ID"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FACEBOOK_LAST_NAME")
    def FACEBOOK_LAST_NAME(cls) -> "ProviderAttribute":
        '''The last name attribute provided by Facebook.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "FACEBOOK_LAST_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FACEBOOK_LOCALE")
    def FACEBOOK_LOCALE(cls) -> "ProviderAttribute":
        '''The locale attribute provided by Facebook.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "FACEBOOK_LOCALE"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FACEBOOK_MIDDLE_NAME")
    def FACEBOOK_MIDDLE_NAME(cls) -> "ProviderAttribute":
        '''The middle name attribute provided by Facebook.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "FACEBOOK_MIDDLE_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="FACEBOOK_NAME")
    def FACEBOOK_NAME(cls) -> "ProviderAttribute":
        '''The name attribute provided by Facebook.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "FACEBOOK_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GOOGLE_BIRTHDAYS")
    def GOOGLE_BIRTHDAYS(cls) -> "ProviderAttribute":
        '''The birthday attribute provided by Google.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "GOOGLE_BIRTHDAYS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GOOGLE_EMAIL")
    def GOOGLE_EMAIL(cls) -> "ProviderAttribute":
        '''The email attribute provided by Google.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "GOOGLE_EMAIL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GOOGLE_EMAIL_VERIFIED")
    def GOOGLE_EMAIL_VERIFIED(cls) -> "ProviderAttribute":
        '''The email verified attribute provided by Google.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "GOOGLE_EMAIL_VERIFIED"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GOOGLE_FAMILY_NAME")
    def GOOGLE_FAMILY_NAME(cls) -> "ProviderAttribute":
        '''The family name attribute provided by Google.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "GOOGLE_FAMILY_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GOOGLE_GENDER")
    def GOOGLE_GENDER(cls) -> "ProviderAttribute":
        '''The gender attribute provided by Google.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "GOOGLE_GENDER"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GOOGLE_GIVEN_NAME")
    def GOOGLE_GIVEN_NAME(cls) -> "ProviderAttribute":
        '''The given name attribute provided by Google.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "GOOGLE_GIVEN_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GOOGLE_NAME")
    def GOOGLE_NAME(cls) -> "ProviderAttribute":
        '''The name attribute provided by Google.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "GOOGLE_NAME"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GOOGLE_NAMES")
    def GOOGLE_NAMES(cls) -> "ProviderAttribute":
        '''The name attribute provided by Google.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "GOOGLE_NAMES"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GOOGLE_PHONE_NUMBERS")
    def GOOGLE_PHONE_NUMBERS(cls) -> "ProviderAttribute":
        '''The phone number attribute provided by Google.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "GOOGLE_PHONE_NUMBERS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="GOOGLE_PICTURE")
    def GOOGLE_PICTURE(cls) -> "ProviderAttribute":
        '''The picture attribute provided by Google.'''
        return typing.cast("ProviderAttribute", jsii.sget(cls, "GOOGLE_PICTURE"))

    @builtins.property
    @jsii.member(jsii_name="attributeName")
    def attribute_name(self) -> builtins.str:
        '''The attribute value string as recognized by the provider.'''
        return typing.cast(builtins.str, jsii.get(self, "attributeName"))


class ResourceServerScope(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_cognito.ResourceServerScope",
):
    '''A scope for ResourceServer.

    :exampleMetadata: infused

    Example::

        pool = cognito.UserPool(self, "Pool")
        
        read_only_scope = cognito.ResourceServerScope(scope_name="read", scope_description="Read-only access")
        full_access_scope = cognito.ResourceServerScope(scope_name="*", scope_description="Full access")
        
        user_server = pool.add_resource_server("ResourceServer",
            identifier="users",
            scopes=[read_only_scope, full_access_scope]
        )
        
        read_only_client = pool.add_client("read-only-client",
            # ...
            o_auth=cognito.OAuthSettings(
                # ...
                scopes=[cognito.OAuthScope.resource_server(user_server, read_only_scope)]
            )
        )
        
        full_access_client = pool.add_client("full-access-client",
            # ...
            o_auth=cognito.OAuthSettings(
                # ...
                scopes=[cognito.OAuthScope.resource_server(user_server, full_access_scope)]
            )
        )
    '''

    def __init__(
        self,
        *,
        scope_description: builtins.str,
        scope_name: builtins.str,
    ) -> None:
        '''
        :param scope_description: A description of the scope.
        :param scope_name: The name of the scope.
        '''
        props = ResourceServerScopeProps(
            scope_description=scope_description, scope_name=scope_name
        )

        jsii.create(self.__class__, self, [props])

    @builtins.property
    @jsii.member(jsii_name="scopeDescription")
    def scope_description(self) -> builtins.str:
        '''A description of the scope.'''
        return typing.cast(builtins.str, jsii.get(self, "scopeDescription"))

    @builtins.property
    @jsii.member(jsii_name="scopeName")
    def scope_name(self) -> builtins.str:
        '''The name of the scope.'''
        return typing.cast(builtins.str, jsii.get(self, "scopeName"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.ResourceServerScopeProps",
    jsii_struct_bases=[],
    name_mapping={"scope_description": "scopeDescription", "scope_name": "scopeName"},
)
class ResourceServerScopeProps:
    def __init__(
        self,
        *,
        scope_description: builtins.str,
        scope_name: builtins.str,
    ) -> None:
        '''Props to initialize ResourceServerScope.

        :param scope_description: A description of the scope.
        :param scope_name: The name of the scope.

        :exampleMetadata: infused

        Example::

            pool = cognito.UserPool(self, "Pool")
            
            read_only_scope = cognito.ResourceServerScope(scope_name="read", scope_description="Read-only access")
            full_access_scope = cognito.ResourceServerScope(scope_name="*", scope_description="Full access")
            
            user_server = pool.add_resource_server("ResourceServer",
                identifier="users",
                scopes=[read_only_scope, full_access_scope]
            )
            
            read_only_client = pool.add_client("read-only-client",
                # ...
                o_auth=cognito.OAuthSettings(
                    # ...
                    scopes=[cognito.OAuthScope.resource_server(user_server, read_only_scope)]
                )
            )
            
            full_access_client = pool.add_client("full-access-client",
                # ...
                o_auth=cognito.OAuthSettings(
                    # ...
                    scopes=[cognito.OAuthScope.resource_server(user_server, full_access_scope)]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2765bfbec7ddc9a6ea049a27d319f643f29be46bd4637a9cc4afd85536ee993c)
            check_type(argname="argument scope_description", value=scope_description, expected_type=type_hints["scope_description"])
            check_type(argname="argument scope_name", value=scope_name, expected_type=type_hints["scope_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "scope_description": scope_description,
            "scope_name": scope_name,
        }

    @builtins.property
    def scope_description(self) -> builtins.str:
        '''A description of the scope.'''
        result = self._values.get("scope_description")
        assert result is not None, "Required property 'scope_description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def scope_name(self) -> builtins.str:
        '''The name of the scope.'''
        result = self._values.get("scope_name")
        assert result is not None, "Required property 'scope_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ResourceServerScopeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.SignInAliases",
    jsii_struct_bases=[],
    name_mapping={
        "email": "email",
        "phone": "phone",
        "preferred_username": "preferredUsername",
        "username": "username",
    },
)
class SignInAliases:
    def __init__(
        self,
        *,
        email: typing.Optional[builtins.bool] = None,
        phone: typing.Optional[builtins.bool] = None,
        preferred_username: typing.Optional[builtins.bool] = None,
        username: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''The different ways in which users of this pool can sign up or sign in.

        :param email: Whether a user is allowed to sign up or sign in with an email address. Default: false
        :param phone: Whether a user is allowed to sign up or sign in with a phone number. Default: false
        :param preferred_username: Whether a user is allowed to sign in with a secondary username, that can be set and modified after sign up. Can only be used in conjunction with ``USERNAME``. Default: false
        :param username: Whether user is allowed to sign up or sign in with a username. Default: true

        :exampleMetadata: infused

        Example::

            cognito.UserPool(self, "myuserpool",
                # ...
                # ...
                sign_in_aliases=cognito.SignInAliases(username=True, email=True),
                auto_verify=cognito.AutoVerifiedAttrs(email=True, phone=True)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1f85eb7769fbc2d73d7ddedb7d58312be06c85b0446415fcf926cc1e5c1343fa)
            check_type(argname="argument email", value=email, expected_type=type_hints["email"])
            check_type(argname="argument phone", value=phone, expected_type=type_hints["phone"])
            check_type(argname="argument preferred_username", value=preferred_username, expected_type=type_hints["preferred_username"])
            check_type(argname="argument username", value=username, expected_type=type_hints["username"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if email is not None:
            self._values["email"] = email
        if phone is not None:
            self._values["phone"] = phone
        if preferred_username is not None:
            self._values["preferred_username"] = preferred_username
        if username is not None:
            self._values["username"] = username

    @builtins.property
    def email(self) -> typing.Optional[builtins.bool]:
        '''Whether a user is allowed to sign up or sign in with an email address.

        :default: false
        '''
        result = self._values.get("email")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def phone(self) -> typing.Optional[builtins.bool]:
        '''Whether a user is allowed to sign up or sign in with a phone number.

        :default: false
        '''
        result = self._values.get("phone")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def preferred_username(self) -> typing.Optional[builtins.bool]:
        '''Whether a user is allowed to sign in with a secondary username, that can be set and modified after sign up.

        Can only be used in conjunction with ``USERNAME``.

        :default: false
        '''
        result = self._values.get("preferred_username")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def username(self) -> typing.Optional[builtins.bool]:
        '''Whether user is allowed to sign up or sign in with a username.

        :default: true
        '''
        result = self._values.get("username")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SignInAliases(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.SignInPolicy",
    jsii_struct_bases=[],
    name_mapping={"allowed_first_auth_factors": "allowedFirstAuthFactors"},
)
class SignInPolicy:
    def __init__(
        self,
        *,
        allowed_first_auth_factors: typing.Optional[typing.Union[AllowedFirstAuthFactors, typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Sign-in policy for User Pools.

        :param allowed_first_auth_factors: The types of authentication that you want to allow for users' first authentication prompt. Default: - Password only

        :exampleMetadata: infused

        Example::

            cognito.UserPool(self, "myuserpool",
                sign_in_policy=cognito.SignInPolicy(
                    allowed_first_auth_factors=cognito.AllowedFirstAuthFactors(password=True, passkey=True)
                ),
                passkey_relying_party_id="auth.example.com",
                passkey_user_verification=cognito.PasskeyUserVerification.REQUIRED
            )
        '''
        if isinstance(allowed_first_auth_factors, dict):
            allowed_first_auth_factors = AllowedFirstAuthFactors(**allowed_first_auth_factors)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5bda8a1a812b13ba6dfe14c09bb234238503bd86905d8f363571b49c270280f4)
            check_type(argname="argument allowed_first_auth_factors", value=allowed_first_auth_factors, expected_type=type_hints["allowed_first_auth_factors"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allowed_first_auth_factors is not None:
            self._values["allowed_first_auth_factors"] = allowed_first_auth_factors

    @builtins.property
    def allowed_first_auth_factors(self) -> typing.Optional[AllowedFirstAuthFactors]:
        '''The types of authentication that you want to allow for users' first authentication prompt.

        :default: - Password only

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice
        '''
        result = self._values.get("allowed_first_auth_factors")
        return typing.cast(typing.Optional[AllowedFirstAuthFactors], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SignInPolicy(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.SignInUrlOptions",
    jsii_struct_bases=[BaseUrlOptions],
    name_mapping={
        "fips": "fips",
        "redirect_uri": "redirectUri",
        "sign_in_path": "signInPath",
    },
)
class SignInUrlOptions(BaseUrlOptions):
    def __init__(
        self,
        *,
        fips: typing.Optional[builtins.bool] = None,
        redirect_uri: builtins.str,
        sign_in_path: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options to customize the behaviour of ``signInUrl()``.

        :param fips: Whether to return the FIPS-compliant endpoint. Default: return the standard URL
        :param redirect_uri: Where to redirect to after sign in.
        :param sign_in_path: The path in the URI where the sign-in page is located. Default: '/login'

        :exampleMetadata: infused

        Example::

            userpool = cognito.UserPool(self, "UserPool")
            client = userpool.add_client("Client",
                # ...
                o_auth=cognito.OAuthSettings(
                    flows=cognito.OAuthFlows(
                        implicit_code_grant=True
                    ),
                    callback_urls=["https://myapp.com/home", "https://myapp.com/users"
                    ]
                )
            )
            domain = userpool.add_domain("Domain")
            sign_in_url = domain.sign_in_url(client,
                redirect_uri="https://myapp.com/home"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__27aae9c398fe91d31540649394c2469df625de6993272c3b3cff19edc49ec8fa)
            check_type(argname="argument fips", value=fips, expected_type=type_hints["fips"])
            check_type(argname="argument redirect_uri", value=redirect_uri, expected_type=type_hints["redirect_uri"])
            check_type(argname="argument sign_in_path", value=sign_in_path, expected_type=type_hints["sign_in_path"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "redirect_uri": redirect_uri,
        }
        if fips is not None:
            self._values["fips"] = fips
        if sign_in_path is not None:
            self._values["sign_in_path"] = sign_in_path

    @builtins.property
    def fips(self) -> typing.Optional[builtins.bool]:
        '''Whether to return the FIPS-compliant endpoint.

        :default: return the standard URL
        '''
        result = self._values.get("fips")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def redirect_uri(self) -> builtins.str:
        '''Where to redirect to after sign in.'''
        result = self._values.get("redirect_uri")
        assert result is not None, "Required property 'redirect_uri' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def sign_in_path(self) -> typing.Optional[builtins.str]:
        '''The path in the URI where the sign-in page is located.

        :default: '/login'
        '''
        result = self._values.get("sign_in_path")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SignInUrlOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_cognito.SigningAlgorithm")
class SigningAlgorithm(enum.Enum):
    '''Signing algorithms for SAML requests.

    :exampleMetadata: infused

    Example::

        userpool = cognito.UserPool(self, "Pool")
        
        # specify the metadata as a file content
        cognito.UserPoolIdentityProviderSaml(self, "userpoolIdpFile",
            user_pool=userpool,
            metadata=cognito.UserPoolIdentityProviderSamlMetadata.file("my-file-contents"),
            # Whether to require encrypted SAML assertions from IdP
            encrypted_responses=True,
            # The signing algorithm for the SAML requests
            request_signing_algorithm=cognito.SigningAlgorithm.RSA_SHA256,
            # Enable IdP initiated SAML auth flow
            idp_initiated=True
        )
        
        # specify the metadata as a URL
        cognito.UserPoolIdentityProviderSaml(self, "userpoolidpUrl",
            user_pool=userpool,
            metadata=cognito.UserPoolIdentityProviderSamlMetadata.url("https://my-metadata-url.com")
        )
    '''

    RSA_SHA256 = "RSA_SHA256"
    '''RSA with SHA-256.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.StandardAttribute",
    jsii_struct_bases=[],
    name_mapping={"mutable": "mutable", "required": "required"},
)
class StandardAttribute:
    def __init__(
        self,
        *,
        mutable: typing.Optional[builtins.bool] = None,
        required: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Standard attribute that can be marked as required or mutable.

        :param mutable: Specifies whether the value of the attribute can be changed. For any user pool attribute that's mapped to an identity provider attribute, this must be set to ``true``. Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider. If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute. Default: true
        :param required: Specifies whether the attribute is required upon user registration. If the attribute is required and the user does not provide a value, registration or sign-in will fail. Default: false

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#cognito-user-pools-standard-attributes
        :exampleMetadata: infused

        Example::

            cognito.UserPool(self, "myuserpool",
                # ...
                standard_attributes=cognito.StandardAttributes(
                    fullname=cognito.StandardAttribute(
                        required=True,
                        mutable=False
                    ),
                    address=cognito.StandardAttribute(
                        required=False,
                        mutable=True
                    )
                ),
                custom_attributes={
                    "myappid": cognito.StringAttribute(min_len=5, max_len=15, mutable=False),
                    "callingcode": cognito.NumberAttribute(min=1, max=3, mutable=True),
                    "isEmployee": cognito.BooleanAttribute(mutable=True),
                    "joinedOn": cognito.DateTimeAttribute()
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cd704ac88cb959f2d4b07ea6284afc3e90d8c8af6d1e6da921a72abd44411a54)
            check_type(argname="argument mutable", value=mutable, expected_type=type_hints["mutable"])
            check_type(argname="argument required", value=required, expected_type=type_hints["required"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if mutable is not None:
            self._values["mutable"] = mutable
        if required is not None:
            self._values["required"] = required

    @builtins.property
    def mutable(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether the value of the attribute can be changed.

        For any user pool attribute that's mapped to an identity provider attribute, this must be set to ``true``.
        Amazon Cognito updates mapped attributes when users sign in to your application through an identity provider.
        If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute.

        :default: true
        '''
        result = self._values.get("mutable")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def required(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether the attribute is required upon user registration.

        If the attribute is required and the user does not provide a value, registration or sign-in will fail.

        :default: false
        '''
        result = self._values.get("required")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "StandardAttribute(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_cognito.StandardAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "address": "address",
        "birthdate": "birthdate",
        "email": "email",
        "family_name": "familyName",
        "fullname": "fullname",
        "gender": "gender",
        "given_name": "givenName",
        "last_update_time": "lastUpdateTime",
        "locale": "locale",
        "middle_name": "middleName",
        "nickname": "nickname",
        "phone_number": "phoneNumber",
        "preferred_username": "preferredUsername",
        "profile_page": "profilePage",
        "profile_picture": "profilePicture",
        "timezone": "timezone",
        "website": "website",
    },
)
class StandardAttributes:
    def __init__(
        self,
        *,
        address: typing.Optional[typing.Union[StandardAttribute, typing.Dict[builtins.str, typing.Any]]] = None,
        birthdate: typing.Optional[typing.Union[StandardAttribute, typing.Dict[builtins.str, typing.Any]]] = None,
        email: typing.Optional[typing.Union[StandardAttribute, typing.Dict[builtins.str, typing.Any]]] = None,
        family_name: typing.Optional[typing.Union[StandardAttribute, typing.Dict[builtins.str, typing.Any]]] = None,
        fullname: typing.Optional[typing.Union[StandardAttribute, typing.Dict[builtins.str, typing.Any]]] = None,
        gender: typing.Optional[typing.Union[StandardAttribute, typing.Dict[builtins.str, typing.Any]]] = None,
        given_name: typing.Optional[typing.Union[StandardAttribute, typing.Dict[builtins.str, typing.Any]]] = None,
        last_update_time: typing.Optional[typing.Union[StandardAttribute, typing.Dict[builtins.str, typing.Any]]] = None,
        locale: typing.Optional[typing.Union[StandardAttribute, typing.Dict[builtins.str, typing.Any]]] = None,
        middle_name: typing.Opti