r'''
# Amazon Bedrock Construct Library

Amazon Bedrock is a fully managed service that offers a choice of foundation models (FMs)
along with a broad set of capabilities for building generative AI applications.

This construct library provides a collection of constructs that can look up existing Bedrock models
for use with other services' CDK constructs, such as AWS Step Functions.

To look up a Bedrock base foundation model:

```python
import aws_cdk.aws_bedrock as bedrock


bedrock.FoundationModel.from_foundation_model_id(self, "Model", bedrock.FoundationModelIdentifier.ANTHROPIC_CLAUDE_V2)
```

To look up a Bedrock provisioned throughput model:

```python
import aws_cdk.aws_bedrock as bedrock


bedrock.ProvisionedModel.from_provisioned_model_arn(self, "Model", "arn:aws:bedrock:us-east-2:123456789012:provisioned-model/abc-123")
```

There are no official hand-written ([L2](https://docs.aws.amazon.com/cdk/latest/guide/constructs.html#constructs_lib)) constructs for provisioning Bedrock resources yet. Here are some suggestions on how to proceed:

* Search [Construct Hub for Bedrock construct libraries](https://constructs.dev/search?q=bedrock)
* Use the automatically generated [L1](https://docs.aws.amazon.com/cdk/latest/guide/constructs.html#constructs_l1_using) constructs, in the same way you would use [the CloudFormation AWS::Bedrock resources](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/AWS_Bedrock.html) directly.
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    CfnResource as _CfnResource_9df397a6,
    CfnTag as _CfnTag_f6864754,
    IInspectable as _IInspectable_c2943556,
    IResolvable as _IResolvable_da3f097b,
    ITaggableV2 as _ITaggableV2_4e6798f8,
    TagManager as _TagManager_0a598cb3,
    TreeInspector as _TreeInspector_488e0dd5,
)


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnAgent(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent",
):
    '''Specifies an agent as a resource in a top-level template. Minimally, you must specify the following properties:.

    - AgentName – Specify a name for the agent.
    - AgentResourceRoleArn – Specify the Amazon Resource Name (ARN) of the service role with permissions to invoke API operations on the agent. For more information, see `Create a service role for Agents for Amazon Bedrock <https://docs.aws.amazon.com/bedrock/latest/userguide/agents-permissions.html>`_ .
    - FoundationModel – Specify the model ID of a foundation model to use when invoking the agent. For more information, see `Supported regions and models for Agents for Amazon Bedrock <https://docs.aws.amazon.com//bedrock/latest/userguide/agents-supported.html>`_ .

    For more information about using agents in Amazon Bedrock , see `Agents for Amazon Bedrock <https://docs.aws.amazon.com/bedrock/latest/userguide/agents.html>`_ .

    See the *Properties* section below for descriptions of both the required and optional properties.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html
    :cloudformationResource: AWS::Bedrock::Agent
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_bedrock as bedrock
        
        # additional_model_request_fields: Any
        
        cfn_agent = bedrock.CfnAgent(self, "MyCfnAgent",
            agent_name="agentName",
        
            # the properties below are optional
            action_groups=[bedrock.CfnAgent.AgentActionGroupProperty(
                action_group_name="actionGroupName",
        
                # the properties below are optional
                action_group_executor=bedrock.CfnAgent.ActionGroupExecutorProperty(
                    custom_control="customControl",
                    lambda_="lambda"
                ),
                action_group_state="actionGroupState",
                api_schema=bedrock.CfnAgent.APISchemaProperty(
                    payload="payload",
                    s3=bedrock.CfnAgent.S3IdentifierProperty(
                        s3_bucket_name="s3BucketName",
                        s3_object_key="s3ObjectKey"
                    )
                ),
                description="description",
                function_schema=bedrock.CfnAgent.FunctionSchemaProperty(
                    functions=[bedrock.CfnAgent.FunctionProperty(
                        name="name",
        
                        # the properties below are optional
                        description="description",
                        parameters={
                            "parameters_key": bedrock.CfnAgent.ParameterDetailProperty(
                                type="type",
        
                                # the properties below are optional
                                description="description",
                                required=False
                            )
                        },
                        require_confirmation="requireConfirmation"
                    )]
                ),
                parent_action_group_signature="parentActionGroupSignature",
                skip_resource_in_use_check_on_delete=False
            )],
            agent_collaboration="agentCollaboration",
            agent_collaborators=[bedrock.CfnAgent.AgentCollaboratorProperty(
                agent_descriptor=bedrock.CfnAgent.AgentDescriptorProperty(
                    alias_arn="aliasArn"
                ),
                collaboration_instruction="collaborationInstruction",
                collaborator_name="collaboratorName",
        
                # the properties below are optional
                relay_conversation_history="relayConversationHistory"
            )],
            agent_resource_role_arn="agentResourceRoleArn",
            auto_prepare=False,
            customer_encryption_key_arn="customerEncryptionKeyArn",
            custom_orchestration=bedrock.CfnAgent.CustomOrchestrationProperty(
                executor=bedrock.CfnAgent.OrchestrationExecutorProperty(
                    lambda_="lambda"
                )
            ),
            description="description",
            foundation_model="foundationModel",
            guardrail_configuration=bedrock.CfnAgent.GuardrailConfigurationProperty(
                guardrail_identifier="guardrailIdentifier",
                guardrail_version="guardrailVersion"
            ),
            idle_session_ttl_in_seconds=123,
            instruction="instruction",
            knowledge_bases=[bedrock.CfnAgent.AgentKnowledgeBaseProperty(
                description="description",
                knowledge_base_id="knowledgeBaseId",
        
                # the properties below are optional
                knowledge_base_state="knowledgeBaseState"
            )],
            memory_configuration=bedrock.CfnAgent.MemoryConfigurationProperty(
                enabled_memory_types=["enabledMemoryTypes"],
                session_summary_configuration=bedrock.CfnAgent.SessionSummaryConfigurationProperty(
                    max_recent_sessions=123
                ),
                storage_days=123
            ),
            orchestration_type="orchestrationType",
            prompt_override_configuration=bedrock.CfnAgent.PromptOverrideConfigurationProperty(
                prompt_configurations=[bedrock.CfnAgent.PromptConfigurationProperty(
                    additional_model_request_fields=additional_model_request_fields,
                    base_prompt_template="basePromptTemplate",
                    foundation_model="foundationModel",
                    inference_configuration=bedrock.CfnAgent.InferenceConfigurationProperty(
                        maximum_length=123,
                        stop_sequences=["stopSequences"],
                        temperature=123,
                        top_k=123,
                        top_p=123
                    ),
                    parser_mode="parserMode",
                    prompt_creation_mode="promptCreationMode",
                    prompt_state="promptState",
                    prompt_type="promptType"
                )],
        
                # the properties below are optional
                override_lambda="overrideLambda"
            ),
            skip_resource_in_use_check_on_delete=False,
            tags={
                "tags_key": "tags"
            },
            test_alias_tags={
                "test_alias_tags_key": "testAliasTags"
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        agent_name: builtins.str,
        action_groups: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.AgentActionGroupProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        agent_collaboration: typing.Optional[builtins.str] = None,
        agent_collaborators: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.AgentCollaboratorProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        agent_resource_role_arn: typing.Optional[builtins.str] = None,
        auto_prepare: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        customer_encryption_key_arn: typing.Optional[builtins.str] = None,
        custom_orchestration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.CustomOrchestrationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        foundation_model: typing.Optional[builtins.str] = None,
        guardrail_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.GuardrailConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        idle_session_ttl_in_seconds: typing.Optional[jsii.Number] = None,
        instruction: typing.Optional[builtins.str] = None,
        knowledge_bases: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.AgentKnowledgeBaseProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        memory_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.MemoryConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        orchestration_type: typing.Optional[builtins.str] = None,
        prompt_override_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.PromptOverrideConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        skip_resource_in_use_check_on_delete: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        test_alias_tags: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param agent_name: The name of the agent.
        :param action_groups: The action groups that belong to an agent.
        :param agent_collaboration: The agent's collaboration settings.
        :param agent_collaborators: List of Agent Collaborators.
        :param agent_resource_role_arn: The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.
        :param auto_prepare: Specifies whether to automatically update the ``DRAFT`` version of the agent after making changes to the agent. The ``DRAFT`` version can be continually iterated upon during internal development. By default, this value is ``false`` . Default: - false
        :param customer_encryption_key_arn: The Amazon Resource Name (ARN) of the AWS KMS key that encrypts the agent.
        :param custom_orchestration: Contains custom orchestration configurations for the agent.
        :param description: The description of the agent.
        :param foundation_model: The foundation model used for orchestration by the agent.
        :param guardrail_configuration: Details about the guardrail associated with the agent.
        :param idle_session_ttl_in_seconds: The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the agent. A user interaction remains active for the amount of time specified. If no conversation occurs during this time, the session expires and Amazon Bedrock deletes any data provided before the timeout.
        :param instruction: Instructions that tell the agent what it should do and how it should interact with users.
        :param knowledge_bases: The knowledge bases associated with the agent.
        :param memory_configuration: Contains memory configuration for the agent.
        :param orchestration_type: Specifies the orchestration strategy for the agent.
        :param prompt_override_configuration: Contains configurations to override prompt templates in different parts of an agent sequence. For more information, see `Advanced prompts <https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html>`_ .
        :param skip_resource_in_use_check_on_delete: Specifies whether to delete the resource even if it's in use. By default, this value is ``false`` . Default: - false
        :param tags: Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:. - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_ - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_
        :param test_alias_tags: Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:. - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_ - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__facaad57ffe16da42f099d2b7997f3e6fd3b9eba46fd226d8fb5afe286371e74)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAgentProps(
            agent_name=agent_name,
            action_groups=action_groups,
            agent_collaboration=agent_collaboration,
            agent_collaborators=agent_collaborators,
            agent_resource_role_arn=agent_resource_role_arn,
            auto_prepare=auto_prepare,
            customer_encryption_key_arn=customer_encryption_key_arn,
            custom_orchestration=custom_orchestration,
            description=description,
            foundation_model=foundation_model,
            guardrail_configuration=guardrail_configuration,
            idle_session_ttl_in_seconds=idle_session_ttl_in_seconds,
            instruction=instruction,
            knowledge_bases=knowledge_bases,
            memory_configuration=memory_configuration,
            orchestration_type=orchestration_type,
            prompt_override_configuration=prompt_override_configuration,
            skip_resource_in_use_check_on_delete=skip_resource_in_use_check_on_delete,
            tags=tags,
            test_alias_tags=test_alias_tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c671dddc0216853bf62cdd51e6c2889b8dfe0d7819a455df2ad71c5b8d67daba)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0633cd876f44b7c72e1346d6b26f361a0d1afbb01604add4ba48210303ccc35c)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrAgentArn")
    def attr_agent_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the agent.

        :cloudformationAttribute: AgentArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAgentArn"))

    @builtins.property
    @jsii.member(jsii_name="attrAgentId")
    def attr_agent_id(self) -> builtins.str:
        '''The unique identifier of the agent.

        :cloudformationAttribute: AgentId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAgentId"))

    @builtins.property
    @jsii.member(jsii_name="attrAgentStatus")
    def attr_agent_status(self) -> builtins.str:
        '''The status of the agent and whether it is ready for use. The following statuses are possible:.

        - CREATING – The agent is being created.
        - PREPARING – The agent is being prepared.
        - PREPARED – The agent is prepared and ready to be invoked.
        - NOT_PREPARED – The agent has been created but not yet prepared.
        - FAILED – The agent API operation failed.
        - UPDATING – The agent is being updated.
        - DELETING – The agent is being deleted.

        :cloudformationAttribute: AgentStatus
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAgentStatus"))

    @builtins.property
    @jsii.member(jsii_name="attrAgentVersion")
    def attr_agent_version(self) -> builtins.str:
        '''The version of the agent.

        :cloudformationAttribute: AgentVersion
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAgentVersion"))

    @builtins.property
    @jsii.member(jsii_name="attrCreatedAt")
    def attr_created_at(self) -> builtins.str:
        '''The time at which the agent was created.

        :cloudformationAttribute: CreatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreatedAt"))

    @builtins.property
    @jsii.member(jsii_name="attrFailureReasons")
    def attr_failure_reasons(self) -> typing.List[builtins.str]:
        '''Contains reasons that the agent-related API that you invoked failed.

        :cloudformationAttribute: FailureReasons
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "attrFailureReasons"))

    @builtins.property
    @jsii.member(jsii_name="attrPreparedAt")
    def attr_prepared_at(self) -> builtins.str:
        '''The time at which the agent was last prepared.

        :cloudformationAttribute: PreparedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPreparedAt"))

    @builtins.property
    @jsii.member(jsii_name="attrRecommendedActions")
    def attr_recommended_actions(self) -> typing.List[builtins.str]:
        '''Contains recommended actions to take for the agent-related API that you invoked to succeed.

        :cloudformationAttribute: RecommendedActions
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "attrRecommendedActions"))

    @builtins.property
    @jsii.member(jsii_name="attrUpdatedAt")
    def attr_updated_at(self) -> builtins.str:
        '''The time at which the agent was last updated.

        :cloudformationAttribute: UpdatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUpdatedAt"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="agentName")
    def agent_name(self) -> builtins.str:
        '''The name of the agent.'''
        return typing.cast(builtins.str, jsii.get(self, "agentName"))

    @agent_name.setter
    def agent_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__89c19306b2dfaafb18e03ff6cb134d14ba6e0563e4a2c79c53886ba8207714cc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "agentName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="actionGroups")
    def action_groups(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.AgentActionGroupProperty"]]]]:
        '''The action groups that belong to an agent.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.AgentActionGroupProperty"]]]], jsii.get(self, "actionGroups"))

    @action_groups.setter
    def action_groups(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.AgentActionGroupProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f670733e4a574d57292a648f4a8ad68ef64492d8f6667729aee160997a190e16)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "actionGroups", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="agentCollaboration")
    def agent_collaboration(self) -> typing.Optional[builtins.str]:
        '''The agent's collaboration settings.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "agentCollaboration"))

    @agent_collaboration.setter
    def agent_collaboration(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f72731f646fedd7539af4983fabc50f2c79adcf5dcc355889ab876a9fd11ec84)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "agentCollaboration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="agentCollaborators")
    def agent_collaborators(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.AgentCollaboratorProperty"]]]]:
        '''List of Agent Collaborators.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.AgentCollaboratorProperty"]]]], jsii.get(self, "agentCollaborators"))

    @agent_collaborators.setter
    def agent_collaborators(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.AgentCollaboratorProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__beaf0ac8c454f4f172dbcf09f365797fb7a54df60711cb70b40118abac7f58e6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "agentCollaborators", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="agentResourceRoleArn")
    def agent_resource_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "agentResourceRoleArn"))

    @agent_resource_role_arn.setter
    def agent_resource_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__36b021046f759c8bbfbfb02bceb7d337926c83d3e866d3dcd420bb7819e254ca)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "agentResourceRoleArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="autoPrepare")
    def auto_prepare(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to automatically update the ``DRAFT`` version of the agent after making changes to the agent.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "autoPrepare"))

    @auto_prepare.setter
    def auto_prepare(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8dc98d7cbd7435359b362ef11a9b384386d54f878814d4c596d3dfa290281e67)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoPrepare", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="customerEncryptionKeyArn")
    def customer_encryption_key_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the AWS KMS key that encrypts the agent.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "customerEncryptionKeyArn"))

    @customer_encryption_key_arn.setter
    def customer_encryption_key_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ea763219d3fed10a3841f0a137399ddf8f575abf9b1691273415e66f4f0763c2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "customerEncryptionKeyArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="customOrchestration")
    def custom_orchestration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.CustomOrchestrationProperty"]]:
        '''Contains custom orchestration configurations for the agent.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.CustomOrchestrationProperty"]], jsii.get(self, "customOrchestration"))

    @custom_orchestration.setter
    def custom_orchestration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.CustomOrchestrationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7bb692aac81267c6a096be6d5432d60a3b62015a610ff438f037460627f13be6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "customOrchestration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the agent.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0bd6874957339badfe5e02e7733947f3ba0b1fdc8c597003bfe10fc7ff96c537)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="foundationModel")
    def foundation_model(self) -> typing.Optional[builtins.str]:
        '''The foundation model used for orchestration by the agent.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "foundationModel"))

    @foundation_model.setter
    def foundation_model(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__75110aa6cf6cc2a463a78a36e9a6327c59875b1ef94d53da3047b8ec62c79fd4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "foundationModel", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="guardrailConfiguration")
    def guardrail_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.GuardrailConfigurationProperty"]]:
        '''Details about the guardrail associated with the agent.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.GuardrailConfigurationProperty"]], jsii.get(self, "guardrailConfiguration"))

    @guardrail_configuration.setter
    def guardrail_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.GuardrailConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a943d3334097d01594a34df767861ec6709d99cd525e60d2d4b88bcabd8de9f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "guardrailConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="idleSessionTtlInSeconds")
    def idle_session_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the agent.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "idleSessionTtlInSeconds"))

    @idle_session_ttl_in_seconds.setter
    def idle_session_ttl_in_seconds(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e29f42af6a8bf7aee6adb6b35e7809da69175082446368854bd402491896a71f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "idleSessionTtlInSeconds", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="instruction")
    def instruction(self) -> typing.Optional[builtins.str]:
        '''Instructions that tell the agent what it should do and how it should interact with users.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instruction"))

    @instruction.setter
    def instruction(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__51c531a159f9cc0e23a07b5bc12cf53006e4207dd460e3ce580c57fa76109e86)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "instruction", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="knowledgeBases")
    def knowledge_bases(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.AgentKnowledgeBaseProperty"]]]]:
        '''The knowledge bases associated with the agent.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.AgentKnowledgeBaseProperty"]]]], jsii.get(self, "knowledgeBases"))

    @knowledge_bases.setter
    def knowledge_bases(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.AgentKnowledgeBaseProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8842d93181ada69f14f22e8ff6855e528b858dd0baf705a1ae228b3d96ef38d6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "knowledgeBases", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="memoryConfiguration")
    def memory_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.MemoryConfigurationProperty"]]:
        '''Contains memory configuration for the agent.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.MemoryConfigurationProperty"]], jsii.get(self, "memoryConfiguration"))

    @memory_configuration.setter
    def memory_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.MemoryConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9dcb39dfb48769eecfe69ce4ba13939ca98bab37df34c9552a9c60b22b526b0e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "memoryConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="orchestrationType")
    def orchestration_type(self) -> typing.Optional[builtins.str]:
        '''Specifies the orchestration strategy for the agent.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "orchestrationType"))

    @orchestration_type.setter
    def orchestration_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b83053d356c226028b8355aa1c0c7019f211c5e848e3b71a72e98d30697ffe6a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "orchestrationType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="promptOverrideConfiguration")
    def prompt_override_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.PromptOverrideConfigurationProperty"]]:
        '''Contains configurations to override prompt templates in different parts of an agent sequence.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.PromptOverrideConfigurationProperty"]], jsii.get(self, "promptOverrideConfiguration"))

    @prompt_override_configuration.setter
    def prompt_override_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.PromptOverrideConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0d115c78c31e0cc4b4828b953c3afd44f4b9271c4e3aaad46777846216d6e9b8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "promptOverrideConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="skipResourceInUseCheckOnDelete")
    def skip_resource_in_use_check_on_delete(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to delete the resource even if it's in use.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "skipResourceInUseCheckOnDelete"))

    @skip_resource_in_use_check_on_delete.setter
    def skip_resource_in_use_check_on_delete(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b8f4401bf1c7d6b30870233c265d99a973b05e80ad9e53ab7945a82208a3bf49)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "skipResourceInUseCheckOnDelete", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Metadata that you can assign to a resource as key-value pairs.

        For more information, see the following resources:.
        '''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "tags"))

    @tags.setter
    def tags(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f9e79489cae3525b40539f62fddb8a1f8b194bc0b1d166146c12f85a52d978be)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="testAliasTags")
    def test_alias_tags(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''Metadata that you can assign to a resource as key-value pairs.

        For more information, see the following resources:.
        '''
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], jsii.get(self, "testAliasTags"))

    @test_alias_tags.setter
    def test_alias_tags(
        self,
        value: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0adf24775a3a1362c3b4eeb79adc26cdf461e3c52ade9a1522d271295bf0d775)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "testAliasTags", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.APISchemaProperty",
        jsii_struct_bases=[],
        name_mapping={"payload": "payload", "s3": "s3"},
    )
    class APISchemaProperty:
        def __init__(
            self,
            *,
            payload: typing.Optional[builtins.str] = None,
            s3: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.S3IdentifierProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains details about the OpenAPI schema for the action group.

            For more information, see `Action group OpenAPI schemas <https://docs.aws.amazon.com//bedrock/latest/userguide/agents-api-schema.html>`_ . You can either include the schema directly in the payload field or you can upload it to an S3 bucket and specify the S3 bucket location in the s3 field.

            :param payload: The JSON or YAML-formatted payload defining the OpenAPI schema for the action group.
            :param s3: Contains details about the S3 object containing the OpenAPI schema for the action group.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-apischema.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                a_pISchema_property = bedrock.CfnAgent.APISchemaProperty(
                    payload="payload",
                    s3=bedrock.CfnAgent.S3IdentifierProperty(
                        s3_bucket_name="s3BucketName",
                        s3_object_key="s3ObjectKey"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__537a89983eef4dfdf436d52865dbd27f462f778f0de6c8ec6162918153faf331)
                check_type(argname="argument payload", value=payload, expected_type=type_hints["payload"])
                check_type(argname="argument s3", value=s3, expected_type=type_hints["s3"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if payload is not None:
                self._values["payload"] = payload
            if s3 is not None:
                self._values["s3"] = s3

        @builtins.property
        def payload(self) -> typing.Optional[builtins.str]:
            '''The JSON or YAML-formatted payload defining the OpenAPI schema for the action group.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-apischema.html#cfn-bedrock-agent-apischema-payload
            '''
            result = self._values.get("payload")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.S3IdentifierProperty"]]:
            '''Contains details about the S3 object containing the OpenAPI schema for the action group.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-apischema.html#cfn-bedrock-agent-apischema-s3
            '''
            result = self._values.get("s3")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.S3IdentifierProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "APISchemaProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.ActionGroupExecutorProperty",
        jsii_struct_bases=[],
        name_mapping={"custom_control": "customControl", "lambda_": "lambda"},
    )
    class ActionGroupExecutorProperty:
        def __init__(
            self,
            *,
            custom_control: typing.Optional[builtins.str] = None,
            lambda_: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains details about the Lambda function containing the business logic that is carried out upon invoking the action or the custom control method for handling the information elicited from the user.

            :param custom_control: To return the action group invocation results directly in the ``InvokeInlineAgent`` response, specify ``RETURN_CONTROL`` .
            :param lambda_: The Amazon Resource Name (ARN) of the Lambda function containing the business logic that is carried out upon invoking the action.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-actiongroupexecutor.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                action_group_executor_property = bedrock.CfnAgent.ActionGroupExecutorProperty(
                    custom_control="customControl",
                    lambda_="lambda"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__27e0c43d5a1e3dda92d710404aea51beb5f30a149322d9ec3e8d354b889735eb)
                check_type(argname="argument custom_control", value=custom_control, expected_type=type_hints["custom_control"])
                check_type(argname="argument lambda_", value=lambda_, expected_type=type_hints["lambda_"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if custom_control is not None:
                self._values["custom_control"] = custom_control
            if lambda_ is not None:
                self._values["lambda_"] = lambda_

        @builtins.property
        def custom_control(self) -> typing.Optional[builtins.str]:
            '''To return the action group invocation results directly in the ``InvokeInlineAgent`` response, specify ``RETURN_CONTROL`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-actiongroupexecutor.html#cfn-bedrock-agent-actiongroupexecutor-customcontrol
            '''
            result = self._values.get("custom_control")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def lambda_(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the Lambda function containing the business logic that is carried out upon invoking the action.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-actiongroupexecutor.html#cfn-bedrock-agent-actiongroupexecutor-lambda
            '''
            result = self._values.get("lambda_")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ActionGroupExecutorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.AgentActionGroupProperty",
        jsii_struct_bases=[],
        name_mapping={
            "action_group_name": "actionGroupName",
            "action_group_executor": "actionGroupExecutor",
            "action_group_state": "actionGroupState",
            "api_schema": "apiSchema",
            "description": "description",
            "function_schema": "functionSchema",
            "parent_action_group_signature": "parentActionGroupSignature",
            "skip_resource_in_use_check_on_delete": "skipResourceInUseCheckOnDelete",
        },
    )
    class AgentActionGroupProperty:
        def __init__(
            self,
            *,
            action_group_name: builtins.str,
            action_group_executor: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.ActionGroupExecutorProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            action_group_state: typing.Optional[builtins.str] = None,
            api_schema: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.APISchemaProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            description: typing.Optional[builtins.str] = None,
            function_schema: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.FunctionSchemaProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            parent_action_group_signature: typing.Optional[builtins.str] = None,
            skip_resource_in_use_check_on_delete: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''Contains details of the inline agent's action group.

            :param action_group_name: The name of the action group.
            :param action_group_executor: The Amazon Resource Name (ARN) of the Lambda function containing the business logic that is carried out upon invoking the action or the custom control method for handling the information elicited from the user.
            :param action_group_state: Specifies whether the action group is available for the agent to invoke or not when sending an `InvokeAgent <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html>`_ request.
            :param api_schema: Contains either details about the S3 object containing the OpenAPI schema for the action group or the JSON or YAML-formatted payload defining the schema. For more information, see `Action group OpenAPI schemas <https://docs.aws.amazon.com//bedrock/latest/userguide/agents-api-schema.html>`_ .
            :param description: A description of the action group.
            :param function_schema: Contains details about the function schema for the action group or the JSON or YAML-formatted payload defining the schema.
            :param parent_action_group_signature: If this field is set as ``AMAZON.UserInput`` , the agent can request the user for additional information when trying to complete a task. The ``description`` , ``apiSchema`` , and ``actionGroupExecutor`` fields must be blank for this action group. During orchestration, if the agent determines that it needs to invoke an API in an action group, but doesn't have enough information to complete the API request, it will invoke this action group instead and return an `Observation <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Observation.html>`_ reprompting the user for more information.
            :param skip_resource_in_use_check_on_delete: Specifies whether to delete the resource even if it's in use. By default, this value is ``false`` . Default: - false

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentactiongroup.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                agent_action_group_property = bedrock.CfnAgent.AgentActionGroupProperty(
                    action_group_name="actionGroupName",
                
                    # the properties below are optional
                    action_group_executor=bedrock.CfnAgent.ActionGroupExecutorProperty(
                        custom_control="customControl",
                        lambda_="lambda"
                    ),
                    action_group_state="actionGroupState",
                    api_schema=bedrock.CfnAgent.APISchemaProperty(
                        payload="payload",
                        s3=bedrock.CfnAgent.S3IdentifierProperty(
                            s3_bucket_name="s3BucketName",
                            s3_object_key="s3ObjectKey"
                        )
                    ),
                    description="description",
                    function_schema=bedrock.CfnAgent.FunctionSchemaProperty(
                        functions=[bedrock.CfnAgent.FunctionProperty(
                            name="name",
                
                            # the properties below are optional
                            description="description",
                            parameters={
                                "parameters_key": bedrock.CfnAgent.ParameterDetailProperty(
                                    type="type",
                
                                    # the properties below are optional
                                    description="description",
                                    required=False
                                )
                            },
                            require_confirmation="requireConfirmation"
                        )]
                    ),
                    parent_action_group_signature="parentActionGroupSignature",
                    skip_resource_in_use_check_on_delete=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__adb9c2568194cbbe9c081f2a8339eaee9966d82b8c5c5ffbc732275ef4a1b76a)
                check_type(argname="argument action_group_name", value=action_group_name, expected_type=type_hints["action_group_name"])
                check_type(argname="argument action_group_executor", value=action_group_executor, expected_type=type_hints["action_group_executor"])
                check_type(argname="argument action_group_state", value=action_group_state, expected_type=type_hints["action_group_state"])
                check_type(argname="argument api_schema", value=api_schema, expected_type=type_hints["api_schema"])
                check_type(argname="argument description", value=description, expected_type=type_hints["description"])
                check_type(argname="argument function_schema", value=function_schema, expected_type=type_hints["function_schema"])
                check_type(argname="argument parent_action_group_signature", value=parent_action_group_signature, expected_type=type_hints["parent_action_group_signature"])
                check_type(argname="argument skip_resource_in_use_check_on_delete", value=skip_resource_in_use_check_on_delete, expected_type=type_hints["skip_resource_in_use_check_on_delete"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action_group_name": action_group_name,
            }
            if action_group_executor is not None:
                self._values["action_group_executor"] = action_group_executor
            if action_group_state is not None:
                self._values["action_group_state"] = action_group_state
            if api_schema is not None:
                self._values["api_schema"] = api_schema
            if description is not None:
                self._values["description"] = description
            if function_schema is not None:
                self._values["function_schema"] = function_schema
            if parent_action_group_signature is not None:
                self._values["parent_action_group_signature"] = parent_action_group_signature
            if skip_resource_in_use_check_on_delete is not None:
                self._values["skip_resource_in_use_check_on_delete"] = skip_resource_in_use_check_on_delete

        @builtins.property
        def action_group_name(self) -> builtins.str:
            '''The name of the action group.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentactiongroup.html#cfn-bedrock-agent-agentactiongroup-actiongroupname
            '''
            result = self._values.get("action_group_name")
            assert result is not None, "Required property 'action_group_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def action_group_executor(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.ActionGroupExecutorProperty"]]:
            '''The Amazon Resource Name (ARN) of the Lambda function containing the business logic that is carried out upon invoking the action or the custom control method for handling the information elicited from the user.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentactiongroup.html#cfn-bedrock-agent-agentactiongroup-actiongroupexecutor
            '''
            result = self._values.get("action_group_executor")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.ActionGroupExecutorProperty"]], result)

        @builtins.property
        def action_group_state(self) -> typing.Optional[builtins.str]:
            '''Specifies whether the action group is available for the agent to invoke or not when sending an `InvokeAgent <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html>`_ request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentactiongroup.html#cfn-bedrock-agent-agentactiongroup-actiongroupstate
            '''
            result = self._values.get("action_group_state")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def api_schema(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.APISchemaProperty"]]:
            '''Contains either details about the S3 object containing the OpenAPI schema for the action group or the JSON or YAML-formatted payload defining the schema.

            For more information, see `Action group OpenAPI schemas <https://docs.aws.amazon.com//bedrock/latest/userguide/agents-api-schema.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentactiongroup.html#cfn-bedrock-agent-agentactiongroup-apischema
            '''
            result = self._values.get("api_schema")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.APISchemaProperty"]], result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''A description of the action group.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentactiongroup.html#cfn-bedrock-agent-agentactiongroup-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def function_schema(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.FunctionSchemaProperty"]]:
            '''Contains details about the function schema for the action group or the JSON or YAML-formatted payload defining the schema.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentactiongroup.html#cfn-bedrock-agent-agentactiongroup-functionschema
            '''
            result = self._values.get("function_schema")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.FunctionSchemaProperty"]], result)

        @builtins.property
        def parent_action_group_signature(self) -> typing.Optional[builtins.str]:
            '''If this field is set as ``AMAZON.UserInput`` , the agent can request the user for additional information when trying to complete a task. The ``description`` , ``apiSchema`` , and ``actionGroupExecutor`` fields must be blank for this action group.

            During orchestration, if the agent determines that it needs to invoke an API in an action group, but doesn't have enough information to complete the API request, it will invoke this action group instead and return an `Observation <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Observation.html>`_ reprompting the user for more information.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentactiongroup.html#cfn-bedrock-agent-agentactiongroup-parentactiongroupsignature
            '''
            result = self._values.get("parent_action_group_signature")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def skip_resource_in_use_check_on_delete(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether to delete the resource even if it's in use.

            By default, this value is ``false`` .

            :default: - false

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentactiongroup.html#cfn-bedrock-agent-agentactiongroup-skipresourceinusecheckondelete
            '''
            result = self._values.get("skip_resource_in_use_check_on_delete")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AgentActionGroupProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.AgentCollaboratorProperty",
        jsii_struct_bases=[],
        name_mapping={
            "agent_descriptor": "agentDescriptor",
            "collaboration_instruction": "collaborationInstruction",
            "collaborator_name": "collaboratorName",
            "relay_conversation_history": "relayConversationHistory",
        },
    )
    class AgentCollaboratorProperty:
        def __init__(
            self,
            *,
            agent_descriptor: typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.AgentDescriptorProperty", typing.Dict[builtins.str, typing.Any]]],
            collaboration_instruction: builtins.str,
            collaborator_name: builtins.str,
            relay_conversation_history: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An agent collaborator.

            :param agent_descriptor: The collaborator's agent descriptor.
            :param collaboration_instruction: The collaborator's instructions.
            :param collaborator_name: The collaborator's collaborator name.
            :param relay_conversation_history: The collaborator's relay conversation history.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentcollaborator.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                agent_collaborator_property = bedrock.CfnAgent.AgentCollaboratorProperty(
                    agent_descriptor=bedrock.CfnAgent.AgentDescriptorProperty(
                        alias_arn="aliasArn"
                    ),
                    collaboration_instruction="collaborationInstruction",
                    collaborator_name="collaboratorName",
                
                    # the properties below are optional
                    relay_conversation_history="relayConversationHistory"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5b97ab11870f9ef63341839508203eb703dc88dada2d1e54dee2d5aea5d7c15e)
                check_type(argname="argument agent_descriptor", value=agent_descriptor, expected_type=type_hints["agent_descriptor"])
                check_type(argname="argument collaboration_instruction", value=collaboration_instruction, expected_type=type_hints["collaboration_instruction"])
                check_type(argname="argument collaborator_name", value=collaborator_name, expected_type=type_hints["collaborator_name"])
                check_type(argname="argument relay_conversation_history", value=relay_conversation_history, expected_type=type_hints["relay_conversation_history"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "agent_descriptor": agent_descriptor,
                "collaboration_instruction": collaboration_instruction,
                "collaborator_name": collaborator_name,
            }
            if relay_conversation_history is not None:
                self._values["relay_conversation_history"] = relay_conversation_history

        @builtins.property
        def agent_descriptor(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnAgent.AgentDescriptorProperty"]:
            '''The collaborator's agent descriptor.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentcollaborator.html#cfn-bedrock-agent-agentcollaborator-agentdescriptor
            '''
            result = self._values.get("agent_descriptor")
            assert result is not None, "Required property 'agent_descriptor' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnAgent.AgentDescriptorProperty"], result)

        @builtins.property
        def collaboration_instruction(self) -> builtins.str:
            '''The collaborator's instructions.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentcollaborator.html#cfn-bedrock-agent-agentcollaborator-collaborationinstruction
            '''
            result = self._values.get("collaboration_instruction")
            assert result is not None, "Required property 'collaboration_instruction' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def collaborator_name(self) -> builtins.str:
            '''The collaborator's collaborator name.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentcollaborator.html#cfn-bedrock-agent-agentcollaborator-collaboratorname
            '''
            result = self._values.get("collaborator_name")
            assert result is not None, "Required property 'collaborator_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def relay_conversation_history(self) -> typing.Optional[builtins.str]:
            '''The collaborator's relay conversation history.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentcollaborator.html#cfn-bedrock-agent-agentcollaborator-relayconversationhistory
            '''
            result = self._values.get("relay_conversation_history")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AgentCollaboratorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.AgentDescriptorProperty",
        jsii_struct_bases=[],
        name_mapping={"alias_arn": "aliasArn"},
    )
    class AgentDescriptorProperty:
        def __init__(self, *, alias_arn: typing.Optional[builtins.str] = None) -> None:
            '''An agent descriptor.

            :param alias_arn: The agent's alias ARN.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentdescriptor.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                agent_descriptor_property = bedrock.CfnAgent.AgentDescriptorProperty(
                    alias_arn="aliasArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__de3d75bfeea411f0afb63f871615ea4feea519d794973992b49e0847c868c558)
                check_type(argname="argument alias_arn", value=alias_arn, expected_type=type_hints["alias_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if alias_arn is not None:
                self._values["alias_arn"] = alias_arn

        @builtins.property
        def alias_arn(self) -> typing.Optional[builtins.str]:
            '''The agent's alias ARN.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentdescriptor.html#cfn-bedrock-agent-agentdescriptor-aliasarn
            '''
            result = self._values.get("alias_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AgentDescriptorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.AgentKnowledgeBaseProperty",
        jsii_struct_bases=[],
        name_mapping={
            "description": "description",
            "knowledge_base_id": "knowledgeBaseId",
            "knowledge_base_state": "knowledgeBaseState",
        },
    )
    class AgentKnowledgeBaseProperty:
        def __init__(
            self,
            *,
            description: builtins.str,
            knowledge_base_id: builtins.str,
            knowledge_base_state: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains details about a knowledge base that is associated with an agent.

            :param description: The description of the association between the agent and the knowledge base.
            :param knowledge_base_id: The unique identifier of the association between the agent and the knowledge base.
            :param knowledge_base_state: Specifies whether to use the knowledge base or not when sending an `InvokeAgent <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html>`_ request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentknowledgebase.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                agent_knowledge_base_property = bedrock.CfnAgent.AgentKnowledgeBaseProperty(
                    description="description",
                    knowledge_base_id="knowledgeBaseId",
                
                    # the properties below are optional
                    knowledge_base_state="knowledgeBaseState"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__45f582a007ade3e95453c6c5cd09673a7089ab2673014cc97bc0c1eee1d85391)
                check_type(argname="argument description", value=description, expected_type=type_hints["description"])
                check_type(argname="argument knowledge_base_id", value=knowledge_base_id, expected_type=type_hints["knowledge_base_id"])
                check_type(argname="argument knowledge_base_state", value=knowledge_base_state, expected_type=type_hints["knowledge_base_state"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "description": description,
                "knowledge_base_id": knowledge_base_id,
            }
            if knowledge_base_state is not None:
                self._values["knowledge_base_state"] = knowledge_base_state

        @builtins.property
        def description(self) -> builtins.str:
            '''The description of the association between the agent and the knowledge base.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentknowledgebase.html#cfn-bedrock-agent-agentknowledgebase-description
            '''
            result = self._values.get("description")
            assert result is not None, "Required property 'description' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def knowledge_base_id(self) -> builtins.str:
            '''The unique identifier of the association between the agent and the knowledge base.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentknowledgebase.html#cfn-bedrock-agent-agentknowledgebase-knowledgebaseid
            '''
            result = self._values.get("knowledge_base_id")
            assert result is not None, "Required property 'knowledge_base_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def knowledge_base_state(self) -> typing.Optional[builtins.str]:
            '''Specifies whether to use the knowledge base or not when sending an `InvokeAgent <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html>`_ request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-agentknowledgebase.html#cfn-bedrock-agent-agentknowledgebase-knowledgebasestate
            '''
            result = self._values.get("knowledge_base_state")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AgentKnowledgeBaseProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.CustomOrchestrationProperty",
        jsii_struct_bases=[],
        name_mapping={"executor": "executor"},
    )
    class CustomOrchestrationProperty:
        def __init__(
            self,
            *,
            executor: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.OrchestrationExecutorProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Details of custom orchestration.

            :param executor: The structure of the executor invoking the actions in custom orchestration.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-customorchestration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                custom_orchestration_property = bedrock.CfnAgent.CustomOrchestrationProperty(
                    executor=bedrock.CfnAgent.OrchestrationExecutorProperty(
                        lambda_="lambda"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e4ae7734d910a065f88411f3aa364ce4517ea8bee41915539d6dffda73fd47dc)
                check_type(argname="argument executor", value=executor, expected_type=type_hints["executor"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if executor is not None:
                self._values["executor"] = executor

        @builtins.property
        def executor(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.OrchestrationExecutorProperty"]]:
            '''The structure of the executor invoking the actions in custom orchestration.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-customorchestration.html#cfn-bedrock-agent-customorchestration-executor
            '''
            result = self._values.get("executor")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.OrchestrationExecutorProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CustomOrchestrationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.FunctionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "description": "description",
            "parameters": "parameters",
            "require_confirmation": "requireConfirmation",
        },
    )
    class FunctionProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            description: typing.Optional[builtins.str] = None,
            parameters: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.ParameterDetailProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            require_confirmation: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Defines parameters that the agent needs to invoke from the user to complete the function.

            Corresponds to an action in an action group.

            This data type is used in the following API operations:

            - `CreateAgentActionGroup request <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateAgentActionGroup.html#API_agent_CreateAgentActionGroup_RequestSyntax>`_
            - `CreateAgentActionGroup response <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateAgentActionGroup.html#API_agent_CreateAgentActionGroup_ResponseSyntax>`_
            - `UpdateAgentActionGroup request <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateAgentActionGroup.html#API_agent_UpdateAgentActionGroup_RequestSyntax>`_
            - `UpdateAgentActionGroup response <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateAgentActionGroup.html#API_agent_UpdateAgentActionGroup_ResponseSyntax>`_
            - `GetAgentActionGroup response <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetAgentActionGroup.html#API_agent_GetAgentActionGroup_ResponseSyntax>`_

            :param name: A name for the function.
            :param description: A description of the function and its purpose.
            :param parameters: The parameters that the agent elicits from the user to fulfill the function.
            :param require_confirmation: Contains information if user confirmation is required to invoke the function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-function.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                function_property = bedrock.CfnAgent.FunctionProperty(
                    name="name",
                
                    # the properties below are optional
                    description="description",
                    parameters={
                        "parameters_key": bedrock.CfnAgent.ParameterDetailProperty(
                            type="type",
                
                            # the properties below are optional
                            description="description",
                            required=False
                        )
                    },
                    require_confirmation="requireConfirmation"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__19341a6c1d4ad6e6ceea8110fa1f82a06ec1a28df35e4ccc6cdb16f028b12747)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument description", value=description, expected_type=type_hints["description"])
                check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
                check_type(argname="argument require_confirmation", value=require_confirmation, expected_type=type_hints["require_confirmation"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if description is not None:
                self._values["description"] = description
            if parameters is not None:
                self._values["parameters"] = parameters
            if require_confirmation is not None:
                self._values["require_confirmation"] = require_confirmation

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-function.html#cfn-bedrock-agent-function-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''A description of the function and its purpose.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-function.html#cfn-bedrock-agent-function-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, "CfnAgent.ParameterDetailProperty"]]]]:
            '''The parameters that the agent elicits from the user to fulfill the function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-function.html#cfn-bedrock-agent-function-parameters
            '''
            result = self._values.get("parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, "CfnAgent.ParameterDetailProperty"]]]], result)

        @builtins.property
        def require_confirmation(self) -> typing.Optional[builtins.str]:
            '''Contains information if user confirmation is required to invoke the function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-function.html#cfn-bedrock-agent-function-requireconfirmation
            '''
            result = self._values.get("require_confirmation")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FunctionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.FunctionSchemaProperty",
        jsii_struct_bases=[],
        name_mapping={"functions": "functions"},
    )
    class FunctionSchemaProperty:
        def __init__(
            self,
            *,
            functions: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.FunctionProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''Contains details about the function schema for the action group or the JSON or YAML-formatted payload defining the schema.

            :param functions: A list of functions that each define an action in the action group.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-functionschema.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                function_schema_property = bedrock.CfnAgent.FunctionSchemaProperty(
                    functions=[bedrock.CfnAgent.FunctionProperty(
                        name="name",
                
                        # the properties below are optional
                        description="description",
                        parameters={
                            "parameters_key": bedrock.CfnAgent.ParameterDetailProperty(
                                type="type",
                
                                # the properties below are optional
                                description="description",
                                required=False
                            )
                        },
                        require_confirmation="requireConfirmation"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__90c954c5127bf68b4aecb34d9f9ade65e40a4db12d624a80f89415c2de263971)
                check_type(argname="argument functions", value=functions, expected_type=type_hints["functions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "functions": functions,
            }

        @builtins.property
        def functions(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.FunctionProperty"]]]:
            '''A list of functions that each define an action in the action group.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-functionschema.html#cfn-bedrock-agent-functionschema-functions
            '''
            result = self._values.get("functions")
            assert result is not None, "Required property 'functions' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.FunctionProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FunctionSchemaProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.GuardrailConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "guardrail_identifier": "guardrailIdentifier",
            "guardrail_version": "guardrailVersion",
        },
    )
    class GuardrailConfigurationProperty:
        def __init__(
            self,
            *,
            guardrail_identifier: typing.Optional[builtins.str] = None,
            guardrail_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Configuration information for a guardrail that you use with the `Converse <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_Converse.html>`_ operation.

            :param guardrail_identifier: The identifier for the guardrail.
            :param guardrail_version: The version of the guardrail.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-guardrailconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                guardrail_configuration_property = bedrock.CfnAgent.GuardrailConfigurationProperty(
                    guardrail_identifier="guardrailIdentifier",
                    guardrail_version="guardrailVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__efdf581de679cb7af4ff41463b76591d2e1035223ca400da555c5c92eed58106)
                check_type(argname="argument guardrail_identifier", value=guardrail_identifier, expected_type=type_hints["guardrail_identifier"])
                check_type(argname="argument guardrail_version", value=guardrail_version, expected_type=type_hints["guardrail_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if guardrail_identifier is not None:
                self._values["guardrail_identifier"] = guardrail_identifier
            if guardrail_version is not None:
                self._values["guardrail_version"] = guardrail_version

        @builtins.property
        def guardrail_identifier(self) -> typing.Optional[builtins.str]:
            '''The identifier for the guardrail.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-guardrailconfiguration.html#cfn-bedrock-agent-guardrailconfiguration-guardrailidentifier
            '''
            result = self._values.get("guardrail_identifier")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def guardrail_version(self) -> typing.Optional[builtins.str]:
            '''The version of the guardrail.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-guardrailconfiguration.html#cfn-bedrock-agent-guardrailconfiguration-guardrailversion
            '''
            result = self._values.get("guardrail_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GuardrailConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.InferenceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "maximum_length": "maximumLength",
            "stop_sequences": "stopSequences",
            "temperature": "temperature",
            "top_k": "topK",
            "top_p": "topP",
        },
    )
    class InferenceConfigurationProperty:
        def __init__(
            self,
            *,
            maximum_length: typing.Optional[jsii.Number] = None,
            stop_sequences: typing.Optional[typing.Sequence[builtins.str]] = None,
            temperature: typing.Optional[jsii.Number] = None,
            top_k: typing.Optional[jsii.Number] = None,
            top_p: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Base inference parameters to pass to a model in a call to `Converse <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_Converse.html>`_ or `ConverseStream <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_ConverseStream.html>`_ . For more information, see `Inference parameters for foundation models <https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html>`_ .

            If you need to pass additional parameters that the model supports, use the ``additionalModelRequestFields`` request field in the call to ``Converse`` or ``ConverseStream`` . For more information, see `Model parameters <https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html>`_ .

            :param maximum_length: The maximum number of tokens allowed in the generated response.
            :param stop_sequences: A list of stop sequences. A stop sequence is a sequence of characters that causes the model to stop generating the response.
            :param temperature: The likelihood of the model selecting higher-probability options while generating a response. A lower value makes the model more likely to choose higher-probability options, while a higher value makes the model more likely to choose lower-probability options. The default value is the default value for the model that you are using. For more information, see `Inference parameters for foundation models <https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html>`_ .
            :param top_k: While generating a response, the model determines the probability of the following token at each point of generation. The value that you set for ``topK`` is the number of most-likely candidates from which the model chooses the next token in the sequence. For example, if you set ``topK`` to 50, the model selects the next token from among the top 50 most likely choices.
            :param top_p: The percentage of most-likely candidates that the model considers for the next token. For example, if you choose a value of 0.8 for ``topP`` , the model selects from the top 80% of the probability distribution of tokens that could be next in the sequence. The default value is the default value for the model that you are using. For more information, see `Inference parameters for foundation models <https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-inferenceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                inference_configuration_property = bedrock.CfnAgent.InferenceConfigurationProperty(
                    maximum_length=123,
                    stop_sequences=["stopSequences"],
                    temperature=123,
                    top_k=123,
                    top_p=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0b8b0e306d1a46c4fdcedd77b9e6caf3ca221fe8133fe3ce29c1e5e779cc3189)
                check_type(argname="argument maximum_length", value=maximum_length, expected_type=type_hints["maximum_length"])
                check_type(argname="argument stop_sequences", value=stop_sequences, expected_type=type_hints["stop_sequences"])
                check_type(argname="argument temperature", value=temperature, expected_type=type_hints["temperature"])
                check_type(argname="argument top_k", value=top_k, expected_type=type_hints["top_k"])
                check_type(argname="argument top_p", value=top_p, expected_type=type_hints["top_p"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if maximum_length is not None:
                self._values["maximum_length"] = maximum_length
            if stop_sequences is not None:
                self._values["stop_sequences"] = stop_sequences
            if temperature is not None:
                self._values["temperature"] = temperature
            if top_k is not None:
                self._values["top_k"] = top_k
            if top_p is not None:
                self._values["top_p"] = top_p

        @builtins.property
        def maximum_length(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of tokens allowed in the generated response.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-inferenceconfiguration.html#cfn-bedrock-agent-inferenceconfiguration-maximumlength
            '''
            result = self._values.get("maximum_length")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def stop_sequences(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of stop sequences.

            A stop sequence is a sequence of characters that causes the model to stop generating the response.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-inferenceconfiguration.html#cfn-bedrock-agent-inferenceconfiguration-stopsequences
            '''
            result = self._values.get("stop_sequences")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def temperature(self) -> typing.Optional[jsii.Number]:
            '''The likelihood of the model selecting higher-probability options while generating a response.

            A lower value makes the model more likely to choose higher-probability options, while a higher value makes the model more likely to choose lower-probability options.

            The default value is the default value for the model that you are using. For more information, see `Inference parameters for foundation models <https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-inferenceconfiguration.html#cfn-bedrock-agent-inferenceconfiguration-temperature
            '''
            result = self._values.get("temperature")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def top_k(self) -> typing.Optional[jsii.Number]:
            '''While generating a response, the model determines the probability of the following token at each point of generation.

            The value that you set for ``topK`` is the number of most-likely candidates from which the model chooses the next token in the sequence. For example, if you set ``topK`` to 50, the model selects the next token from among the top 50 most likely choices.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-inferenceconfiguration.html#cfn-bedrock-agent-inferenceconfiguration-topk
            '''
            result = self._values.get("top_k")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def top_p(self) -> typing.Optional[jsii.Number]:
            '''The percentage of most-likely candidates that the model considers for the next token.

            For example, if you choose a value of 0.8 for ``topP`` , the model selects from the top 80% of the probability distribution of tokens that could be next in the sequence.

            The default value is the default value for the model that you are using. For more information, see `Inference parameters for foundation models <https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-inferenceconfiguration.html#cfn-bedrock-agent-inferenceconfiguration-topp
            '''
            result = self._values.get("top_p")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InferenceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.MemoryConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled_memory_types": "enabledMemoryTypes",
            "session_summary_configuration": "sessionSummaryConfiguration",
            "storage_days": "storageDays",
        },
    )
    class MemoryConfigurationProperty:
        def __init__(
            self,
            *,
            enabled_memory_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            session_summary_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.SessionSummaryConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            storage_days: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Details of the memory configuration.

            :param enabled_memory_types: The type of memory that is stored.
            :param session_summary_configuration: Contains the configuration for SESSION_SUMMARY memory type enabled for the agent.
            :param storage_days: The number of days the agent is configured to retain the conversational context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-memoryconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                memory_configuration_property = bedrock.CfnAgent.MemoryConfigurationProperty(
                    enabled_memory_types=["enabledMemoryTypes"],
                    session_summary_configuration=bedrock.CfnAgent.SessionSummaryConfigurationProperty(
                        max_recent_sessions=123
                    ),
                    storage_days=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7d96229de78435013ea720d3fc5416b1a58de73df8a7fd00fc563b8fe5255a40)
                check_type(argname="argument enabled_memory_types", value=enabled_memory_types, expected_type=type_hints["enabled_memory_types"])
                check_type(argname="argument session_summary_configuration", value=session_summary_configuration, expected_type=type_hints["session_summary_configuration"])
                check_type(argname="argument storage_days", value=storage_days, expected_type=type_hints["storage_days"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if enabled_memory_types is not None:
                self._values["enabled_memory_types"] = enabled_memory_types
            if session_summary_configuration is not None:
                self._values["session_summary_configuration"] = session_summary_configuration
            if storage_days is not None:
                self._values["storage_days"] = storage_days

        @builtins.property
        def enabled_memory_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The type of memory that is stored.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-memoryconfiguration.html#cfn-bedrock-agent-memoryconfiguration-enabledmemorytypes
            '''
            result = self._values.get("enabled_memory_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def session_summary_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.SessionSummaryConfigurationProperty"]]:
            '''Contains the configuration for SESSION_SUMMARY memory type enabled for the agent.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-memoryconfiguration.html#cfn-bedrock-agent-memoryconfiguration-sessionsummaryconfiguration
            '''
            result = self._values.get("session_summary_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.SessionSummaryConfigurationProperty"]], result)

        @builtins.property
        def storage_days(self) -> typing.Optional[jsii.Number]:
            '''The number of days the agent is configured to retain the conversational context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-memoryconfiguration.html#cfn-bedrock-agent-memoryconfiguration-storagedays
            '''
            result = self._values.get("storage_days")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MemoryConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.OrchestrationExecutorProperty",
        jsii_struct_bases=[],
        name_mapping={"lambda_": "lambda"},
    )
    class OrchestrationExecutorProperty:
        def __init__(self, *, lambda_: builtins.str) -> None:
            '''Contains details about the Lambda function containing the orchestration logic carried out upon invoking the custom orchestration.

            :param lambda_: The Amazon Resource Name (ARN) of the Lambda function containing the business logic that is carried out upon invoking the action.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-orchestrationexecutor.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                orchestration_executor_property = bedrock.CfnAgent.OrchestrationExecutorProperty(
                    lambda_="lambda"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9bcfed5f99644abac9695b68367aa4ad30c588f8cc060125d511ce6cd7b85deb)
                check_type(argname="argument lambda_", value=lambda_, expected_type=type_hints["lambda_"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "lambda_": lambda_,
            }

        @builtins.property
        def lambda_(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the Lambda function containing the business logic that is carried out upon invoking the action.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-orchestrationexecutor.html#cfn-bedrock-agent-orchestrationexecutor-lambda
            '''
            result = self._values.get("lambda_")
            assert result is not None, "Required property 'lambda_' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OrchestrationExecutorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.ParameterDetailProperty",
        jsii_struct_bases=[],
        name_mapping={
            "type": "type",
            "description": "description",
            "required": "required",
        },
    )
    class ParameterDetailProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            description: typing.Optional[builtins.str] = None,
            required: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''Contains details about a parameter in a function for an action group.

            :param type: The data type of the parameter.
            :param description: A description of the parameter. Helps the foundation model determine how to elicit the parameters from the user.
            :param required: Whether the parameter is required for the agent to complete the function for action group invocation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-parameterdetail.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                parameter_detail_property = bedrock.CfnAgent.ParameterDetailProperty(
                    type="type",
                
                    # the properties below are optional
                    description="description",
                    required=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ed93c304b05f6a676428d620999acc5f22bf1cc1920ab4160039feccf941e790)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument description", value=description, expected_type=type_hints["description"])
                check_type(argname="argument required", value=required, expected_type=type_hints["required"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if description is not None:
                self._values["description"] = description
            if required is not None:
                self._values["required"] = required

        @builtins.property
        def type(self) -> builtins.str:
            '''The data type of the parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-parameterdetail.html#cfn-bedrock-agent-parameterdetail-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''A description of the parameter.

            Helps the foundation model determine how to elicit the parameters from the user.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-parameterdetail.html#cfn-bedrock-agent-parameterdetail-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def required(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Whether the parameter is required for the agent to complete the function for action group invocation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-parameterdetail.html#cfn-bedrock-agent-parameterdetail-required
            '''
            result = self._values.get("required")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ParameterDetailProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.PromptConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "additional_model_request_fields": "additionalModelRequestFields",
            "base_prompt_template": "basePromptTemplate",
            "foundation_model": "foundationModel",
            "inference_configuration": "inferenceConfiguration",
            "parser_mode": "parserMode",
            "prompt_creation_mode": "promptCreationMode",
            "prompt_state": "promptState",
            "prompt_type": "promptType",
        },
    )
    class PromptConfigurationProperty:
        def __init__(
            self,
            *,
            additional_model_request_fields: typing.Any = None,
            base_prompt_template: typing.Optional[builtins.str] = None,
            foundation_model: typing.Optional[builtins.str] = None,
            inference_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.InferenceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            parser_mode: typing.Optional[builtins.str] = None,
            prompt_creation_mode: typing.Optional[builtins.str] = None,
            prompt_state: typing.Optional[builtins.str] = None,
            prompt_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains configurations to override a prompt template in one part of an agent sequence.

            For more information, see `Advanced prompts <https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html>`_ .

            :param additional_model_request_fields: If the Converse or ConverseStream operations support the model, ``additionalModelRequestFields`` contains additional inference parameters, beyond the base set of inference parameters in the ``inferenceConfiguration`` field. For more information, see `Inference request parameters and response fields for foundation models <https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html>`_ .
            :param base_prompt_template: Defines the prompt template with which to replace the default prompt template. You can use placeholder variables in the base prompt template to customize the prompt. For more information, see `Prompt template placeholder variables <https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html>`_ . For more information, see `Configure the prompt templates <https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts-configure.html>`_ .
            :param foundation_model: The agent's foundation model.
            :param inference_configuration: Contains inference parameters to use when the agent invokes a foundation model in the part of the agent sequence defined by the ``promptType`` . For more information, see `Inference parameters for foundation models <https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html>`_ .
            :param parser_mode: Specifies whether to override the default parser Lambda function when parsing the raw foundation model output in the part of the agent sequence defined by the ``promptType`` . If you set the field as ``OVERRIDDEN`` , the ``overrideLambda`` field in the `PromptOverrideConfiguration <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptOverrideConfiguration.html>`_ must be specified with the ARN of a Lambda function.
            :param prompt_creation_mode: Specifies whether to override the default prompt template for this ``promptType`` . Set this value to ``OVERRIDDEN`` to use the prompt that you provide in the ``basePromptTemplate`` . If you leave it as ``DEFAULT`` , the agent uses a default prompt template.
            :param prompt_state: Specifies whether to allow the inline agent to carry out the step specified in the ``promptType`` . If you set this value to ``DISABLED`` , the agent skips that step. The default state for each ``promptType`` is as follows. - ``PRE_PROCESSING`` – ``ENABLED`` - ``ORCHESTRATION`` – ``ENABLED`` - ``KNOWLEDGE_BASE_RESPONSE_GENERATION`` – ``ENABLED`` - ``POST_PROCESSING`` – ``DISABLED``
            :param prompt_type: The step in the agent sequence that this prompt configuration applies to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                # additional_model_request_fields: Any
                
                prompt_configuration_property = bedrock.CfnAgent.PromptConfigurationProperty(
                    additional_model_request_fields=additional_model_request_fields,
                    base_prompt_template="basePromptTemplate",
                    foundation_model="foundationModel",
                    inference_configuration=bedrock.CfnAgent.InferenceConfigurationProperty(
                        maximum_length=123,
                        stop_sequences=["stopSequences"],
                        temperature=123,
                        top_k=123,
                        top_p=123
                    ),
                    parser_mode="parserMode",
                    prompt_creation_mode="promptCreationMode",
                    prompt_state="promptState",
                    prompt_type="promptType"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__072678823f2b5bb708860fe2f54d37c6a24153ba05ec6c798f1cbb1222e7cd7b)
                check_type(argname="argument additional_model_request_fields", value=additional_model_request_fields, expected_type=type_hints["additional_model_request_fields"])
                check_type(argname="argument base_prompt_template", value=base_prompt_template, expected_type=type_hints["base_prompt_template"])
                check_type(argname="argument foundation_model", value=foundation_model, expected_type=type_hints["foundation_model"])
                check_type(argname="argument inference_configuration", value=inference_configuration, expected_type=type_hints["inference_configuration"])
                check_type(argname="argument parser_mode", value=parser_mode, expected_type=type_hints["parser_mode"])
                check_type(argname="argument prompt_creation_mode", value=prompt_creation_mode, expected_type=type_hints["prompt_creation_mode"])
                check_type(argname="argument prompt_state", value=prompt_state, expected_type=type_hints["prompt_state"])
                check_type(argname="argument prompt_type", value=prompt_type, expected_type=type_hints["prompt_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if additional_model_request_fields is not None:
                self._values["additional_model_request_fields"] = additional_model_request_fields
            if base_prompt_template is not None:
                self._values["base_prompt_template"] = base_prompt_template
            if foundation_model is not None:
                self._values["foundation_model"] = foundation_model
            if inference_configuration is not None:
                self._values["inference_configuration"] = inference_configuration
            if parser_mode is not None:
                self._values["parser_mode"] = parser_mode
            if prompt_creation_mode is not None:
                self._values["prompt_creation_mode"] = prompt_creation_mode
            if prompt_state is not None:
                self._values["prompt_state"] = prompt_state
            if prompt_type is not None:
                self._values["prompt_type"] = prompt_type

        @builtins.property
        def additional_model_request_fields(self) -> typing.Any:
            '''If the Converse or ConverseStream operations support the model, ``additionalModelRequestFields`` contains additional inference parameters, beyond the base set of inference parameters in the ``inferenceConfiguration`` field.

            For more information, see `Inference request parameters and response fields for foundation models <https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptconfiguration.html#cfn-bedrock-agent-promptconfiguration-additionalmodelrequestfields
            '''
            result = self._values.get("additional_model_request_fields")
            return typing.cast(typing.Any, result)

        @builtins.property
        def base_prompt_template(self) -> typing.Optional[builtins.str]:
            '''Defines the prompt template with which to replace the default prompt template.

            You can use placeholder variables in the base prompt template to customize the prompt. For more information, see `Prompt template placeholder variables <https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html>`_ . For more information, see `Configure the prompt templates <https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts-configure.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptconfiguration.html#cfn-bedrock-agent-promptconfiguration-baseprompttemplate
            '''
            result = self._values.get("base_prompt_template")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def foundation_model(self) -> typing.Optional[builtins.str]:
            '''The agent's foundation model.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptconfiguration.html#cfn-bedrock-agent-promptconfiguration-foundationmodel
            '''
            result = self._values.get("foundation_model")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def inference_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.InferenceConfigurationProperty"]]:
            '''Contains inference parameters to use when the agent invokes a foundation model in the part of the agent sequence defined by the ``promptType`` .

            For more information, see `Inference parameters for foundation models <https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptconfiguration.html#cfn-bedrock-agent-promptconfiguration-inferenceconfiguration
            '''
            result = self._values.get("inference_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnAgent.InferenceConfigurationProperty"]], result)

        @builtins.property
        def parser_mode(self) -> typing.Optional[builtins.str]:
            '''Specifies whether to override the default parser Lambda function when parsing the raw foundation model output in the part of the agent sequence defined by the ``promptType`` .

            If you set the field as ``OVERRIDDEN`` , the ``overrideLambda`` field in the `PromptOverrideConfiguration <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptOverrideConfiguration.html>`_ must be specified with the ARN of a Lambda function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptconfiguration.html#cfn-bedrock-agent-promptconfiguration-parsermode
            '''
            result = self._values.get("parser_mode")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prompt_creation_mode(self) -> typing.Optional[builtins.str]:
            '''Specifies whether to override the default prompt template for this ``promptType`` .

            Set this value to ``OVERRIDDEN`` to use the prompt that you provide in the ``basePromptTemplate`` . If you leave it as ``DEFAULT`` , the agent uses a default prompt template.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptconfiguration.html#cfn-bedrock-agent-promptconfiguration-promptcreationmode
            '''
            result = self._values.get("prompt_creation_mode")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prompt_state(self) -> typing.Optional[builtins.str]:
            '''Specifies whether to allow the inline agent to carry out the step specified in the ``promptType`` .

            If you set this value to ``DISABLED`` , the agent skips that step. The default state for each ``promptType`` is as follows.

            - ``PRE_PROCESSING`` – ``ENABLED``
            - ``ORCHESTRATION`` – ``ENABLED``
            - ``KNOWLEDGE_BASE_RESPONSE_GENERATION`` – ``ENABLED``
            - ``POST_PROCESSING`` – ``DISABLED``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptconfiguration.html#cfn-bedrock-agent-promptconfiguration-promptstate
            '''
            result = self._values.get("prompt_state")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prompt_type(self) -> typing.Optional[builtins.str]:
            '''The step in the agent sequence that this prompt configuration applies to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptconfiguration.html#cfn-bedrock-agent-promptconfiguration-prompttype
            '''
            result = self._values.get("prompt_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.PromptOverrideConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "prompt_configurations": "promptConfigurations",
            "override_lambda": "overrideLambda",
        },
    )
    class PromptOverrideConfigurationProperty:
        def __init__(
            self,
            *,
            prompt_configurations: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgent.PromptConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]]],
            override_lambda: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains configurations to override prompts in different parts of an agent sequence.

            For more information, see `Advanced prompts <https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html>`_ .

            :param prompt_configurations: Contains configurations to override a prompt template in one part of an agent sequence. For more information, see `Advanced prompts <https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html>`_ .
            :param override_lambda: The ARN of the Lambda function to use when parsing the raw foundation model output in parts of the agent sequence. If you specify this field, at least one of the ``promptConfigurations`` must contain a ``parserMode`` value that is set to ``OVERRIDDEN`` . For more information, see `Parser Lambda function in Amazon Bedrock Agents <https://docs.aws.amazon.com/bedrock/latest/userguide/lambda-parser.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptoverrideconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                # additional_model_request_fields: Any
                
                prompt_override_configuration_property = bedrock.CfnAgent.PromptOverrideConfigurationProperty(
                    prompt_configurations=[bedrock.CfnAgent.PromptConfigurationProperty(
                        additional_model_request_fields=additional_model_request_fields,
                        base_prompt_template="basePromptTemplate",
                        foundation_model="foundationModel",
                        inference_configuration=bedrock.CfnAgent.InferenceConfigurationProperty(
                            maximum_length=123,
                            stop_sequences=["stopSequences"],
                            temperature=123,
                            top_k=123,
                            top_p=123
                        ),
                        parser_mode="parserMode",
                        prompt_creation_mode="promptCreationMode",
                        prompt_state="promptState",
                        prompt_type="promptType"
                    )],
                
                    # the properties below are optional
                    override_lambda="overrideLambda"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__79a38248b49cbea1e103efe955cc12c1bd5e500b242bd9490c1c81253830d77d)
                check_type(argname="argument prompt_configurations", value=prompt_configurations, expected_type=type_hints["prompt_configurations"])
                check_type(argname="argument override_lambda", value=override_lambda, expected_type=type_hints["override_lambda"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "prompt_configurations": prompt_configurations,
            }
            if override_lambda is not None:
                self._values["override_lambda"] = override_lambda

        @builtins.property
        def prompt_configurations(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.PromptConfigurationProperty"]]]:
            '''Contains configurations to override a prompt template in one part of an agent sequence.

            For more information, see `Advanced prompts <https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptoverrideconfiguration.html#cfn-bedrock-agent-promptoverrideconfiguration-promptconfigurations
            '''
            result = self._values.get("prompt_configurations")
            assert result is not None, "Required property 'prompt_configurations' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgent.PromptConfigurationProperty"]]], result)

        @builtins.property
        def override_lambda(self) -> typing.Optional[builtins.str]:
            '''The ARN of the Lambda function to use when parsing the raw foundation model output in parts of the agent sequence.

            If you specify this field, at least one of the ``promptConfigurations`` must contain a ``parserMode`` value that is set to ``OVERRIDDEN`` . For more information, see `Parser Lambda function in Amazon Bedrock Agents <https://docs.aws.amazon.com/bedrock/latest/userguide/lambda-parser.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-promptoverrideconfiguration.html#cfn-bedrock-agent-promptoverrideconfiguration-overridelambda
            '''
            result = self._values.get("override_lambda")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptOverrideConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.S3IdentifierProperty",
        jsii_struct_bases=[],
        name_mapping={
            "s3_bucket_name": "s3BucketName",
            "s3_object_key": "s3ObjectKey",
        },
    )
    class S3IdentifierProperty:
        def __init__(
            self,
            *,
            s3_bucket_name: typing.Optional[builtins.str] = None,
            s3_object_key: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The identifier information for an Amazon S3 bucket.

            :param s3_bucket_name: The name of the S3 bucket.
            :param s3_object_key: The S3 object key for the S3 resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-s3identifier.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                s3_identifier_property = bedrock.CfnAgent.S3IdentifierProperty(
                    s3_bucket_name="s3BucketName",
                    s3_object_key="s3ObjectKey"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1e9a91ab9025dd36217ed95088c50220259ce5115620391445737700490cba60)
                check_type(argname="argument s3_bucket_name", value=s3_bucket_name, expected_type=type_hints["s3_bucket_name"])
                check_type(argname="argument s3_object_key", value=s3_object_key, expected_type=type_hints["s3_object_key"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if s3_bucket_name is not None:
                self._values["s3_bucket_name"] = s3_bucket_name
            if s3_object_key is not None:
                self._values["s3_object_key"] = s3_object_key

        @builtins.property
        def s3_bucket_name(self) -> typing.Optional[builtins.str]:
            '''The name of the S3 bucket.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-s3identifier.html#cfn-bedrock-agent-s3identifier-s3bucketname
            '''
            result = self._values.get("s3_bucket_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_object_key(self) -> typing.Optional[builtins.str]:
            '''The S3 object key for the S3 resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-s3identifier.html#cfn-bedrock-agent-s3identifier-s3objectkey
            '''
            result = self._values.get("s3_object_key")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "S3IdentifierProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgent.SessionSummaryConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"max_recent_sessions": "maxRecentSessions"},
    )
    class SessionSummaryConfigurationProperty:
        def __init__(
            self,
            *,
            max_recent_sessions: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Configuration for SESSION_SUMMARY memory type enabled for the agent.

            :param max_recent_sessions: Maximum number of recent session summaries to include in the agent's prompt context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-sessionsummaryconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                session_summary_configuration_property = bedrock.CfnAgent.SessionSummaryConfigurationProperty(
                    max_recent_sessions=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__72ecccaf4321d6b2cc729d5095c1fa5192a7db52f5293b6536bc7d335964a1d8)
                check_type(argname="argument max_recent_sessions", value=max_recent_sessions, expected_type=type_hints["max_recent_sessions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max_recent_sessions is not None:
                self._values["max_recent_sessions"] = max_recent_sessions

        @builtins.property
        def max_recent_sessions(self) -> typing.Optional[jsii.Number]:
            '''Maximum number of recent session summaries to include in the agent's prompt context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agent-sessionsummaryconfiguration.html#cfn-bedrock-agent-sessionsummaryconfiguration-maxrecentsessions
            '''
            result = self._values.get("max_recent_sessions")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SessionSummaryConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnAgentAlias(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_bedrock.CfnAgentAlias",
):
    '''Specifies an agent alias as a resource in a top-level template. Minimally, you must specify the following properties:.

    - AgentAliasName – Specify a name for the alias.

    For more information about creating aliases for an agent in Amazon Bedrock , see `Deploy an Amazon Bedrock agent <https://docs.aws.amazon.com/bedrock/latest/userguide/agents-deploy.html>`_ .

    See the *Properties* section below for descriptions of both the required and optional properties.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agentalias.html
    :cloudformationResource: AWS::Bedrock::AgentAlias
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_bedrock as bedrock
        
        cfn_agent_alias = bedrock.CfnAgentAlias(self, "MyCfnAgentAlias",
            agent_alias_name="agentAliasName",
            agent_id="agentId",
        
            # the properties below are optional
            description="description",
            routing_configuration=[bedrock.CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty(
                agent_version="agentVersion"
            )],
            tags={
                "tags_key": "tags"
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        agent_alias_name: builtins.str,
        agent_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
        routing_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param agent_alias_name: The name of the alias of the agent.
        :param agent_id: The unique identifier of the agent.
        :param description: The description of the alias of the agent.
        :param routing_configuration: Contains details about the routing configuration of the alias.
        :param tags: Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:. - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_ - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a8230a990c5fac91dc09e3de4211aa6f82fce95537f199a7987ca92f4722ee0)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAgentAliasProps(
            agent_alias_name=agent_alias_name,
            agent_id=agent_id,
            description=description,
            routing_configuration=routing_configuration,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__881be8885c059b078e3110beb1aed396db2ce3f0505bec32be8cedeba399356d)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a7100eb19771c4ffaea3a7323fe3cb5fcb6702a3fa7fdaf66bc8a4db1eae875c)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrAgentAliasArn")
    def attr_agent_alias_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the alias of the agent.

        :cloudformationAttribute: AgentAliasArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAgentAliasArn"))

    @builtins.property
    @jsii.member(jsii_name="attrAgentAliasHistoryEvents")
    def attr_agent_alias_history_events(self) -> _IResolvable_da3f097b:
        '''Contains details about the history of the alias.

        :cloudformationAttribute: AgentAliasHistoryEvents
        '''
        return typing.cast(_IResolvable_da3f097b, jsii.get(self, "attrAgentAliasHistoryEvents"))

    @builtins.property
    @jsii.member(jsii_name="attrAgentAliasId")
    def attr_agent_alias_id(self) -> builtins.str:
        '''The unique identifier of the alias of the agent.

        :cloudformationAttribute: AgentAliasId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAgentAliasId"))

    @builtins.property
    @jsii.member(jsii_name="attrAgentAliasStatus")
    def attr_agent_alias_status(self) -> builtins.str:
        '''The status of the alias of the agent and whether it is ready for use.

        The following statuses are possible:

        - CREATING – The agent alias is being created.
        - PREPARED – The agent alias is finished being created or updated and is ready to be invoked.
        - FAILED – The agent alias API operation failed.
        - UPDATING – The agent alias is being updated.
        - DELETING – The agent alias is being deleted.
        - DISSOCIATED - The agent alias has no version associated with it.

        :cloudformationAttribute: AgentAliasStatus
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAgentAliasStatus"))

    @builtins.property
    @jsii.member(jsii_name="attrCreatedAt")
    def attr_created_at(self) -> builtins.str:
        '''The time at which the alias of the agent was created.

        :cloudformationAttribute: CreatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreatedAt"))

    @builtins.property
    @jsii.member(jsii_name="attrUpdatedAt")
    def attr_updated_at(self) -> builtins.str:
        '''The time at which the alias was last updated.

        :cloudformationAttribute: UpdatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUpdatedAt"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="agentAliasName")
    def agent_alias_name(self) -> builtins.str:
        '''The name of the alias of the agent.'''
        return typing.cast(builtins.str, jsii.get(self, "agentAliasName"))

    @agent_alias_name.setter
    def agent_alias_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4929f9d1fe1c750108b588135fd2420ee93abad44713bd60a4f90bf0bab64e78)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "agentAliasName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="agentId")
    def agent_id(self) -> builtins.str:
        '''The unique identifier of the agent.'''
        return typing.cast(builtins.str, jsii.get(self, "agentId"))

    @agent_id.setter
    def agent_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b10c310fa33a8cb7b52c456850e88b42ea1b312acb49b915241cc6f52ef15136)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "agentId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the alias of the agent.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__57ab6386ae765e3eaf37d16d7a5314cdca1d8b60ea095a4fa303d13c84722e78)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="routingConfiguration")
    def routing_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty"]]]]:
        '''Contains details about the routing configuration of the alias.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty"]]]], jsii.get(self, "routingConfiguration"))

    @routing_configuration.setter
    def routing_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ca75abf30c6951a069bde3673588677ff723ad86067bf5365bb584482bed50a8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routingConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Metadata that you can assign to a resource as key-value pairs.

        For more information, see the following resources:.
        '''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "tags"))

    @tags.setter
    def tags(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cf90503b3b2050ab4f0f6ba0bf057a037193369214d4f027b567d3062bda19b8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgentAlias.AgentAliasHistoryEventProperty",
        jsii_struct_bases=[],
        name_mapping={
            "end_date": "endDate",
            "routing_configuration": "routingConfiguration",
            "start_date": "startDate",
        },
    )
    class AgentAliasHistoryEventProperty:
        def __init__(
            self,
            *,
            end_date: typing.Optional[builtins.str] = None,
            routing_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            start_date: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains details about the history of the alias.

            :param end_date: The date that the alias stopped being associated to the version in the ``routingConfiguration`` object.
            :param routing_configuration: Contains details about the version of the agent with which the alias is associated.
            :param start_date: The date that the alias began being associated to the version in the ``routingConfiguration`` object.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agentalias-agentaliashistoryevent.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                agent_alias_history_event_property = bedrock.CfnAgentAlias.AgentAliasHistoryEventProperty(
                    end_date="endDate",
                    routing_configuration=[bedrock.CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty(
                        agent_version="agentVersion"
                    )],
                    start_date="startDate"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4220f7b3bb12c49a75057af394368ca2d4de63c53d832e73767eb01f3bccb2ce)
                check_type(argname="argument end_date", value=end_date, expected_type=type_hints["end_date"])
                check_type(argname="argument routing_configuration", value=routing_configuration, expected_type=type_hints["routing_configuration"])
                check_type(argname="argument start_date", value=start_date, expected_type=type_hints["start_date"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if end_date is not None:
                self._values["end_date"] = end_date
            if routing_configuration is not None:
                self._values["routing_configuration"] = routing_configuration
            if start_date is not None:
                self._values["start_date"] = start_date

        @builtins.property
        def end_date(self) -> typing.Optional[builtins.str]:
            '''The date that the alias stopped being associated to the version in the ``routingConfiguration`` object.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agentalias-agentaliashistoryevent.html#cfn-bedrock-agentalias-agentaliashistoryevent-enddate
            '''
            result = self._values.get("end_date")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def routing_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty"]]]]:
            '''Contains details about the version of the agent with which the alias is associated.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agentalias-agentaliashistoryevent.html#cfn-bedrock-agentalias-agentaliashistoryevent-routingconfiguration
            '''
            result = self._values.get("routing_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty"]]]], result)

        @builtins.property
        def start_date(self) -> typing.Optional[builtins.str]:
            '''The date that the alias began being associated to the version in the ``routingConfiguration`` object.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agentalias-agentaliashistoryevent.html#cfn-bedrock-agentalias-agentaliashistoryevent-startdate
            '''
            result = self._values.get("start_date")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AgentAliasHistoryEventProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty",
        jsii_struct_bases=[],
        name_mapping={"agent_version": "agentVersion"},
    )
    class AgentAliasRoutingConfigurationListItemProperty:
        def __init__(self, *, agent_version: builtins.str) -> None:
            '''Contains details about the routing configuration of the alias.

            :param agent_version: The version of the agent with which the alias is associated.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agentalias-agentaliasroutingconfigurationlistitem.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                agent_alias_routing_configuration_list_item_property = bedrock.CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty(
                    agent_version="agentVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__846d3bcc8614fd45d175f992a20505b220910c107392df0930556bb43637fc79)
                check_type(argname="argument agent_version", value=agent_version, expected_type=type_hints["agent_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "agent_version": agent_version,
            }

        @builtins.property
        def agent_version(self) -> builtins.str:
            '''The version of the agent with which the alias is associated.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-agentalias-agentaliasroutingconfigurationlistitem.html#cfn-bedrock-agentalias-agentaliasroutingconfigurationlistitem-agentversion
            '''
            result = self._values.get("agent_version")
            assert result is not None, "Required property 'agent_version' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AgentAliasRoutingConfigurationListItemProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_bedrock.CfnAgentAliasProps",
    jsii_struct_bases=[],
    name_mapping={
        "agent_alias_name": "agentAliasName",
        "agent_id": "agentId",
        "description": "description",
        "routing_configuration": "routingConfiguration",
        "tags": "tags",
    },
)
class CfnAgentAliasProps:
    def __init__(
        self,
        *,
        agent_alias_name: builtins.str,
        agent_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
        routing_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnAgentAlias``.

        :param agent_alias_name: The name of the alias of the agent.
        :param agent_id: The unique identifier of the agent.
        :param description: The description of the alias of the agent.
        :param routing_configuration: Contains details about the routing configuration of the alias.
        :param tags: Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:. - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_ - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agentalias.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_bedrock as bedrock
            
            cfn_agent_alias_props = bedrock.CfnAgentAliasProps(
                agent_alias_name="agentAliasName",
                agent_id="agentId",
            
                # the properties below are optional
                description="description",
                routing_configuration=[bedrock.CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty(
                    agent_version="agentVersion"
                )],
                tags={
                    "tags_key": "tags"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c3144a1c66c11b4a2b15be859f361c848648aefb3df04b8fce2befe94f215c68)
            check_type(argname="argument agent_alias_name", value=agent_alias_name, expected_type=type_hints["agent_alias_name"])
            check_type(argname="argument agent_id", value=agent_id, expected_type=type_hints["agent_id"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument routing_configuration", value=routing_configuration, expected_type=type_hints["routing_configuration"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "agent_alias_name": agent_alias_name,
            "agent_id": agent_id,
        }
        if description is not None:
            self._values["description"] = description
        if routing_configuration is not None:
            self._values["routing_configuration"] = routing_configuration
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def agent_alias_name(self) -> builtins.str:
        '''The name of the alias of the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agentalias.html#cfn-bedrock-agentalias-agentaliasname
        '''
        result = self._values.get("agent_alias_name")
        assert result is not None, "Required property 'agent_alias_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def agent_id(self) -> builtins.str:
        '''The unique identifier of the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agentalias.html#cfn-bedrock-agentalias-agentid
        '''
        result = self._values.get("agent_id")
        assert result is not None, "Required property 'agent_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the alias of the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agentalias.html#cfn-bedrock-agentalias-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def routing_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty]]]]:
        '''Contains details about the routing configuration of the alias.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agentalias.html#cfn-bedrock-agentalias-routingconfiguration
        '''
        result = self._values.get("routing_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAgentAlias.AgentAliasRoutingConfigurationListItemProperty]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:.

        - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_
        - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agentalias.html#cfn-bedrock-agentalias-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAgentAliasProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_bedrock.CfnAgentProps",
    jsii_struct_bases=[],
    name_mapping={
        "agent_name": "agentName",
        "action_groups": "actionGroups",
        "agent_collaboration": "agentCollaboration",
        "agent_collaborators": "agentCollaborators",
        "agent_resource_role_arn": "agentResourceRoleArn",
        "auto_prepare": "autoPrepare",
        "customer_encryption_key_arn": "customerEncryptionKeyArn",
        "custom_orchestration": "customOrchestration",
        "description": "description",
        "foundation_model": "foundationModel",
        "guardrail_configuration": "guardrailConfiguration",
        "idle_session_ttl_in_seconds": "idleSessionTtlInSeconds",
        "instruction": "instruction",
        "knowledge_bases": "knowledgeBases",
        "memory_configuration": "memoryConfiguration",
        "orchestration_type": "orchestrationType",
        "prompt_override_configuration": "promptOverrideConfiguration",
        "skip_resource_in_use_check_on_delete": "skipResourceInUseCheckOnDelete",
        "tags": "tags",
        "test_alias_tags": "testAliasTags",
    },
)
class CfnAgentProps:
    def __init__(
        self,
        *,
        agent_name: builtins.str,
        action_groups: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAgent.AgentActionGroupProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        agent_collaboration: typing.Optional[builtins.str] = None,
        agent_collaborators: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAgent.AgentCollaboratorProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        agent_resource_role_arn: typing.Optional[builtins.str] = None,
        auto_prepare: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        customer_encryption_key_arn: typing.Optional[builtins.str] = None,
        custom_orchestration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAgent.CustomOrchestrationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        foundation_model: typing.Optional[builtins.str] = None,
        guardrail_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAgent.GuardrailConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        idle_session_ttl_in_seconds: typing.Optional[jsii.Number] = None,
        instruction: typing.Optional[builtins.str] = None,
        knowledge_bases: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAgent.AgentKnowledgeBaseProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        memory_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAgent.MemoryConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        orchestration_type: typing.Optional[builtins.str] = None,
        prompt_override_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnAgent.PromptOverrideConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        skip_resource_in_use_check_on_delete: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        test_alias_tags: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
    ) -> None:
        '''Properties for defining a ``CfnAgent``.

        :param agent_name: The name of the agent.
        :param action_groups: The action groups that belong to an agent.
        :param agent_collaboration: The agent's collaboration settings.
        :param agent_collaborators: List of Agent Collaborators.
        :param agent_resource_role_arn: The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.
        :param auto_prepare: Specifies whether to automatically update the ``DRAFT`` version of the agent after making changes to the agent. The ``DRAFT`` version can be continually iterated upon during internal development. By default, this value is ``false`` . Default: - false
        :param customer_encryption_key_arn: The Amazon Resource Name (ARN) of the AWS KMS key that encrypts the agent.
        :param custom_orchestration: Contains custom orchestration configurations for the agent.
        :param description: The description of the agent.
        :param foundation_model: The foundation model used for orchestration by the agent.
        :param guardrail_configuration: Details about the guardrail associated with the agent.
        :param idle_session_ttl_in_seconds: The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the agent. A user interaction remains active for the amount of time specified. If no conversation occurs during this time, the session expires and Amazon Bedrock deletes any data provided before the timeout.
        :param instruction: Instructions that tell the agent what it should do and how it should interact with users.
        :param knowledge_bases: The knowledge bases associated with the agent.
        :param memory_configuration: Contains memory configuration for the agent.
        :param orchestration_type: Specifies the orchestration strategy for the agent.
        :param prompt_override_configuration: Contains configurations to override prompt templates in different parts of an agent sequence. For more information, see `Advanced prompts <https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html>`_ .
        :param skip_resource_in_use_check_on_delete: Specifies whether to delete the resource even if it's in use. By default, this value is ``false`` . Default: - false
        :param tags: Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:. - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_ - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_
        :param test_alias_tags: Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:. - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_ - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_bedrock as bedrock
            
            # additional_model_request_fields: Any
            
            cfn_agent_props = bedrock.CfnAgentProps(
                agent_name="agentName",
            
                # the properties below are optional
                action_groups=[bedrock.CfnAgent.AgentActionGroupProperty(
                    action_group_name="actionGroupName",
            
                    # the properties below are optional
                    action_group_executor=bedrock.CfnAgent.ActionGroupExecutorProperty(
                        custom_control="customControl",
                        lambda_="lambda"
                    ),
                    action_group_state="actionGroupState",
                    api_schema=bedrock.CfnAgent.APISchemaProperty(
                        payload="payload",
                        s3=bedrock.CfnAgent.S3IdentifierProperty(
                            s3_bucket_name="s3BucketName",
                            s3_object_key="s3ObjectKey"
                        )
                    ),
                    description="description",
                    function_schema=bedrock.CfnAgent.FunctionSchemaProperty(
                        functions=[bedrock.CfnAgent.FunctionProperty(
                            name="name",
            
                            # the properties below are optional
                            description="description",
                            parameters={
                                "parameters_key": bedrock.CfnAgent.ParameterDetailProperty(
                                    type="type",
            
                                    # the properties below are optional
                                    description="description",
                                    required=False
                                )
                            },
                            require_confirmation="requireConfirmation"
                        )]
                    ),
                    parent_action_group_signature="parentActionGroupSignature",
                    skip_resource_in_use_check_on_delete=False
                )],
                agent_collaboration="agentCollaboration",
                agent_collaborators=[bedrock.CfnAgent.AgentCollaboratorProperty(
                    agent_descriptor=bedrock.CfnAgent.AgentDescriptorProperty(
                        alias_arn="aliasArn"
                    ),
                    collaboration_instruction="collaborationInstruction",
                    collaborator_name="collaboratorName",
            
                    # the properties below are optional
                    relay_conversation_history="relayConversationHistory"
                )],
                agent_resource_role_arn="agentResourceRoleArn",
                auto_prepare=False,
                customer_encryption_key_arn="customerEncryptionKeyArn",
                custom_orchestration=bedrock.CfnAgent.CustomOrchestrationProperty(
                    executor=bedrock.CfnAgent.OrchestrationExecutorProperty(
                        lambda_="lambda"
                    )
                ),
                description="description",
                foundation_model="foundationModel",
                guardrail_configuration=bedrock.CfnAgent.GuardrailConfigurationProperty(
                    guardrail_identifier="guardrailIdentifier",
                    guardrail_version="guardrailVersion"
                ),
                idle_session_ttl_in_seconds=123,
                instruction="instruction",
                knowledge_bases=[bedrock.CfnAgent.AgentKnowledgeBaseProperty(
                    description="description",
                    knowledge_base_id="knowledgeBaseId",
            
                    # the properties below are optional
                    knowledge_base_state="knowledgeBaseState"
                )],
                memory_configuration=bedrock.CfnAgent.MemoryConfigurationProperty(
                    enabled_memory_types=["enabledMemoryTypes"],
                    session_summary_configuration=bedrock.CfnAgent.SessionSummaryConfigurationProperty(
                        max_recent_sessions=123
                    ),
                    storage_days=123
                ),
                orchestration_type="orchestrationType",
                prompt_override_configuration=bedrock.CfnAgent.PromptOverrideConfigurationProperty(
                    prompt_configurations=[bedrock.CfnAgent.PromptConfigurationProperty(
                        additional_model_request_fields=additional_model_request_fields,
                        base_prompt_template="basePromptTemplate",
                        foundation_model="foundationModel",
                        inference_configuration=bedrock.CfnAgent.InferenceConfigurationProperty(
                            maximum_length=123,
                            stop_sequences=["stopSequences"],
                            temperature=123,
                            top_k=123,
                            top_p=123
                        ),
                        parser_mode="parserMode",
                        prompt_creation_mode="promptCreationMode",
                        prompt_state="promptState",
                        prompt_type="promptType"
                    )],
            
                    # the properties below are optional
                    override_lambda="overrideLambda"
                ),
                skip_resource_in_use_check_on_delete=False,
                tags={
                    "tags_key": "tags"
                },
                test_alias_tags={
                    "test_alias_tags_key": "testAliasTags"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b4f714080f6d4f9b0a3fe85a8425a8ba69698695e35d6fbd9d710ca5d99ba6e8)
            check_type(argname="argument agent_name", value=agent_name, expected_type=type_hints["agent_name"])
            check_type(argname="argument action_groups", value=action_groups, expected_type=type_hints["action_groups"])
            check_type(argname="argument agent_collaboration", value=agent_collaboration, expected_type=type_hints["agent_collaboration"])
            check_type(argname="argument agent_collaborators", value=agent_collaborators, expected_type=type_hints["agent_collaborators"])
            check_type(argname="argument agent_resource_role_arn", value=agent_resource_role_arn, expected_type=type_hints["agent_resource_role_arn"])
            check_type(argname="argument auto_prepare", value=auto_prepare, expected_type=type_hints["auto_prepare"])
            check_type(argname="argument customer_encryption_key_arn", value=customer_encryption_key_arn, expected_type=type_hints["customer_encryption_key_arn"])
            check_type(argname="argument custom_orchestration", value=custom_orchestration, expected_type=type_hints["custom_orchestration"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument foundation_model", value=foundation_model, expected_type=type_hints["foundation_model"])
            check_type(argname="argument guardrail_configuration", value=guardrail_configuration, expected_type=type_hints["guardrail_configuration"])
            check_type(argname="argument idle_session_ttl_in_seconds", value=idle_session_ttl_in_seconds, expected_type=type_hints["idle_session_ttl_in_seconds"])
            check_type(argname="argument instruction", value=instruction, expected_type=type_hints["instruction"])
            check_type(argname="argument knowledge_bases", value=knowledge_bases, expected_type=type_hints["knowledge_bases"])
            check_type(argname="argument memory_configuration", value=memory_configuration, expected_type=type_hints["memory_configuration"])
            check_type(argname="argument orchestration_type", value=orchestration_type, expected_type=type_hints["orchestration_type"])
            check_type(argname="argument prompt_override_configuration", value=prompt_override_configuration, expected_type=type_hints["prompt_override_configuration"])
            check_type(argname="argument skip_resource_in_use_check_on_delete", value=skip_resource_in_use_check_on_delete, expected_type=type_hints["skip_resource_in_use_check_on_delete"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument test_alias_tags", value=test_alias_tags, expected_type=type_hints["test_alias_tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "agent_name": agent_name,
        }
        if action_groups is not None:
            self._values["action_groups"] = action_groups
        if agent_collaboration is not None:
            self._values["agent_collaboration"] = agent_collaboration
        if agent_collaborators is not None:
            self._values["agent_collaborators"] = agent_collaborators
        if agent_resource_role_arn is not None:
            self._values["agent_resource_role_arn"] = agent_resource_role_arn
        if auto_prepare is not None:
            self._values["auto_prepare"] = auto_prepare
        if customer_encryption_key_arn is not None:
            self._values["customer_encryption_key_arn"] = customer_encryption_key_arn
        if custom_orchestration is not None:
            self._values["custom_orchestration"] = custom_orchestration
        if description is not None:
            self._values["description"] = description
        if foundation_model is not None:
            self._values["foundation_model"] = foundation_model
        if guardrail_configuration is not None:
            self._values["guardrail_configuration"] = guardrail_configuration
        if idle_session_ttl_in_seconds is not None:
            self._values["idle_session_ttl_in_seconds"] = idle_session_ttl_in_seconds
        if instruction is not None:
            self._values["instruction"] = instruction
        if knowledge_bases is not None:
            self._values["knowledge_bases"] = knowledge_bases
        if memory_configuration is not None:
            self._values["memory_configuration"] = memory_configuration
        if orchestration_type is not None:
            self._values["orchestration_type"] = orchestration_type
        if prompt_override_configuration is not None:
            self._values["prompt_override_configuration"] = prompt_override_configuration
        if skip_resource_in_use_check_on_delete is not None:
            self._values["skip_resource_in_use_check_on_delete"] = skip_resource_in_use_check_on_delete
        if tags is not None:
            self._values["tags"] = tags
        if test_alias_tags is not None:
            self._values["test_alias_tags"] = test_alias_tags

    @builtins.property
    def agent_name(self) -> builtins.str:
        '''The name of the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-agentname
        '''
        result = self._values.get("agent_name")
        assert result is not None, "Required property 'agent_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def action_groups(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAgent.AgentActionGroupProperty]]]]:
        '''The action groups that belong to an agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-actiongroups
        '''
        result = self._values.get("action_groups")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAgent.AgentActionGroupProperty]]]], result)

    @builtins.property
    def agent_collaboration(self) -> typing.Optional[builtins.str]:
        '''The agent's collaboration settings.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-agentcollaboration
        '''
        result = self._values.get("agent_collaboration")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def agent_collaborators(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAgent.AgentCollaboratorProperty]]]]:
        '''List of Agent Collaborators.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-agentcollaborators
        '''
        result = self._values.get("agent_collaborators")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAgent.AgentCollaboratorProperty]]]], result)

    @builtins.property
    def agent_resource_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-agentresourcerolearn
        '''
        result = self._values.get("agent_resource_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auto_prepare(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to automatically update the ``DRAFT`` version of the agent after making changes to the agent.

        The ``DRAFT`` version can be continually iterated upon during internal development. By default, this value is ``false`` .

        :default: - false

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-autoprepare
        '''
        result = self._values.get("auto_prepare")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def customer_encryption_key_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the AWS KMS key that encrypts the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-customerencryptionkeyarn
        '''
        result = self._values.get("customer_encryption_key_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def custom_orchestration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAgent.CustomOrchestrationProperty]]:
        '''Contains custom orchestration configurations for the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-customorchestration
        '''
        result = self._values.get("custom_orchestration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAgent.CustomOrchestrationProperty]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def foundation_model(self) -> typing.Optional[builtins.str]:
        '''The foundation model used for orchestration by the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-foundationmodel
        '''
        result = self._values.get("foundation_model")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def guardrail_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAgent.GuardrailConfigurationProperty]]:
        '''Details about the guardrail associated with the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-guardrailconfiguration
        '''
        result = self._values.get("guardrail_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAgent.GuardrailConfigurationProperty]], result)

    @builtins.property
    def idle_session_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the agent.

        A user interaction remains active for the amount of time specified. If no conversation occurs during this time, the session expires and Amazon Bedrock deletes any data provided before the timeout.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-idlesessionttlinseconds
        '''
        result = self._values.get("idle_session_ttl_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def instruction(self) -> typing.Optional[builtins.str]:
        '''Instructions that tell the agent what it should do and how it should interact with users.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-instruction
        '''
        result = self._values.get("instruction")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def knowledge_bases(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAgent.AgentKnowledgeBaseProperty]]]]:
        '''The knowledge bases associated with the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-knowledgebases
        '''
        result = self._values.get("knowledge_bases")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnAgent.AgentKnowledgeBaseProperty]]]], result)

    @builtins.property
    def memory_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAgent.MemoryConfigurationProperty]]:
        '''Contains memory configuration for the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-memoryconfiguration
        '''
        result = self._values.get("memory_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAgent.MemoryConfigurationProperty]], result)

    @builtins.property
    def orchestration_type(self) -> typing.Optional[builtins.str]:
        '''Specifies the orchestration strategy for the agent.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-orchestrationtype
        '''
        result = self._values.get("orchestration_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def prompt_override_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAgent.PromptOverrideConfigurationProperty]]:
        '''Contains configurations to override prompt templates in different parts of an agent sequence.

        For more information, see `Advanced prompts <https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-promptoverrideconfiguration
        '''
        result = self._values.get("prompt_override_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnAgent.PromptOverrideConfigurationProperty]], result)

    @builtins.property
    def skip_resource_in_use_check_on_delete(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to delete the resource even if it's in use.

        By default, this value is ``false`` .

        :default: - false

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-skipresourceinusecheckondelete
        '''
        result = self._values.get("skip_resource_in_use_check_on_delete")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:.

        - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_
        - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def test_alias_tags(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:.

        - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_
        - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-agent.html#cfn-bedrock-agent-testaliastags
        '''
        result = self._values.get("test_alias_tags")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAgentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnApplicationInferenceProfile(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_bedrock.CfnApplicationInferenceProfile",
):
    '''Specifies an inference profile as a resource in a top-level template.

    Use the ``ModelSource`` field to specify the inference profile to copy into the resource. For more information about using inference profiles in Amazon Bedrock , see `Improve resilience with cross-region inference <https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html>`_ .

    See the *Properties* section below for descriptions of both the required and optional properties.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-applicationinferenceprofile.html
    :cloudformationResource: AWS::Bedrock::ApplicationInferenceProfile
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_bedrock as bedrock
        
        cfn_application_inference_profile = bedrock.CfnApplicationInferenceProfile(self, "MyCfnApplicationInferenceProfile",
            inference_profile_name="inferenceProfileName",
        
            # the properties below are optional
            description="description",
            model_source=bedrock.CfnApplicationInferenceProfile.InferenceProfileModelSourceProperty(
                copy_from="copyFrom"
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        inference_profile_name: builtins.str,
        description: typing.Optional[builtins.str] = None,
        model_source: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnApplicationInferenceProfile.InferenceProfileModelSourceProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param inference_profile_name: The name of the inference profile.
        :param description: The description of the inference profile.
        :param model_source: Contains configurations for the inference profile to copy as the resource.
        :param tags: A list of tags associated with the inference profile.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a2222de49518232a3824d971182922daf44d6255370c3805992fca295b2904ad)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnApplicationInferenceProfileProps(
            inference_profile_name=inference_profile_name,
            description=description,
            model_source=model_source,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d2e339334daf20f2a812f78dadefad7c0714dbaaafdfe61e138ba0cfb3f07e2)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d0b7fff48c27eff9823a04431a3cb684b46b8434a66cf5f1941f73cf8eba5762)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCreatedAt")
    def attr_created_at(self) -> builtins.str:
        '''The time at which the inference profile was created.

        :cloudformationAttribute: CreatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreatedAt"))

    @builtins.property
    @jsii.member(jsii_name="attrInferenceProfileArn")
    def attr_inference_profile_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the inference profile.

        :cloudformationAttribute: InferenceProfileArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrInferenceProfileArn"))

    @builtins.property
    @jsii.member(jsii_name="attrInferenceProfileId")
    def attr_inference_profile_id(self) -> builtins.str:
        '''The unique identifier of the inference profile.

        :cloudformationAttribute: InferenceProfileId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrInferenceProfileId"))

    @builtins.property
    @jsii.member(jsii_name="attrInferenceProfileIdentifier")
    def attr_inference_profile_identifier(self) -> builtins.str:
        '''The ID or Amazon Resource Name (ARN) of the inference profile.

        :cloudformationAttribute: InferenceProfileIdentifier
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrInferenceProfileIdentifier"))

    @builtins.property
    @jsii.member(jsii_name="attrModels")
    def attr_models(self) -> _IResolvable_da3f097b:
        '''A list of information about each model in the inference profile.

        :cloudformationAttribute: Models
        '''
        return typing.cast(_IResolvable_da3f097b, jsii.get(self, "attrModels"))

    @builtins.property
    @jsii.member(jsii_name="attrStatus")
    def attr_status(self) -> builtins.str:
        '''The status of the inference profile.

        ``ACTIVE`` means that the inference profile is ready to be used.

        :cloudformationAttribute: Status
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrStatus"))

    @builtins.property
    @jsii.member(jsii_name="attrType")
    def attr_type(self) -> builtins.str:
        '''The type of the inference profile. The following types are possible:.

        - ``SYSTEM_DEFINED`` – The inference profile is defined by Amazon Bedrock. You can route inference requests across regions with these inference profiles.
        - ``APPLICATION`` – The inference profile was created by a user. This type of inference profile can track metrics and costs when invoking the model in it. The inference profile may route requests to one or multiple regions.

        :cloudformationAttribute: Type
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrType"))

    @builtins.property
    @jsii.member(jsii_name="attrUpdatedAt")
    def attr_updated_at(self) -> builtins.str:
        '''The time at which the inference profile was last updated.

        :cloudformationAttribute: UpdatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUpdatedAt"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="inferenceProfileName")
    def inference_profile_name(self) -> builtins.str:
        '''The name of the inference profile.'''
        return typing.cast(builtins.str, jsii.get(self, "inferenceProfileName"))

    @inference_profile_name.setter
    def inference_profile_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6aae972c3170b196c6e4fa5bf8d0dd4b8b0b49a1c0f56c506e744e0442f85c7c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "inferenceProfileName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the inference profile.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2c6572f5531b0a792b64f808b10a21a42b92c9f7be81c2527cca65319dc19869)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="modelSource")
    def model_source(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnApplicationInferenceProfile.InferenceProfileModelSourceProperty"]]:
        '''Contains configurations for the inference profile to copy as the resource.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnApplicationInferenceProfile.InferenceProfileModelSourceProperty"]], jsii.get(self, "modelSource"))

    @model_source.setter
    def model_source(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnApplicationInferenceProfile.InferenceProfileModelSourceProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2237f6f7b3816378634fc620ae2448fa846fa47396f09bc7c0b5e8a0902d65ae)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "modelSource", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''A list of tags associated with the inference profile.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tags"))

    @tags.setter
    def tags(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e13f50624dfcb2d9c61a18114e7992a8def4ddd2f6f340c352ac98a5668d7eea)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnApplicationInferenceProfile.InferenceProfileModelProperty",
        jsii_struct_bases=[],
        name_mapping={"model_arn": "modelArn"},
    )
    class InferenceProfileModelProperty:
        def __init__(self, *, model_arn: typing.Optional[builtins.str] = None) -> None:
            '''Contains information about a model.

            :param model_arn: The Amazon Resource Name (ARN) of the model.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-applicationinferenceprofile-inferenceprofilemodel.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                inference_profile_model_property = bedrock.CfnApplicationInferenceProfile.InferenceProfileModelProperty(
                    model_arn="modelArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ef9de36422391b892f90691f22e956a086a03a2e131a35b0986b5b63d905255f)
                check_type(argname="argument model_arn", value=model_arn, expected_type=type_hints["model_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if model_arn is not None:
                self._values["model_arn"] = model_arn

        @builtins.property
        def model_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the model.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-applicationinferenceprofile-inferenceprofilemodel.html#cfn-bedrock-applicationinferenceprofile-inferenceprofilemodel-modelarn
            '''
            result = self._values.get("model_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InferenceProfileModelProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnApplicationInferenceProfile.InferenceProfileModelSourceProperty",
        jsii_struct_bases=[],
        name_mapping={"copy_from": "copyFrom"},
    )
    class InferenceProfileModelSourceProperty:
        def __init__(self, *, copy_from: builtins.str) -> None:
            '''Contains information about the model or system-defined inference profile that is the source for an inference profile..

            :param copy_from: The ARN of the model or system-defined inference profile that is the source for the inference profile.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-applicationinferenceprofile-inferenceprofilemodelsource.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                inference_profile_model_source_property = bedrock.CfnApplicationInferenceProfile.InferenceProfileModelSourceProperty(
                    copy_from="copyFrom"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4170b8c8243e2096b7c99037eb0c8874b28fe6d08432e8c6fd045defb2d99d81)
                check_type(argname="argument copy_from", value=copy_from, expected_type=type_hints["copy_from"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "copy_from": copy_from,
            }

        @builtins.property
        def copy_from(self) -> builtins.str:
            '''The ARN of the model or system-defined inference profile that is the source for the inference profile.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-applicationinferenceprofile-inferenceprofilemodelsource.html#cfn-bedrock-applicationinferenceprofile-inferenceprofilemodelsource-copyfrom
            '''
            result = self._values.get("copy_from")
            assert result is not None, "Required property 'copy_from' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InferenceProfileModelSourceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_bedrock.CfnApplicationInferenceProfileProps",
    jsii_struct_bases=[],
    name_mapping={
        "inference_profile_name": "inferenceProfileName",
        "description": "description",
        "model_source": "modelSource",
        "tags": "tags",
    },
)
class CfnApplicationInferenceProfileProps:
    def __init__(
        self,
        *,
        inference_profile_name: builtins.str,
        description: typing.Optional[builtins.str] = None,
        model_source: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnApplicationInferenceProfile.InferenceProfileModelSourceProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnApplicationInferenceProfile``.

        :param inference_profile_name: The name of the inference profile.
        :param description: The description of the inference profile.
        :param model_source: Contains configurations for the inference profile to copy as the resource.
        :param tags: A list of tags associated with the inference profile.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-applicationinferenceprofile.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_bedrock as bedrock
            
            cfn_application_inference_profile_props = bedrock.CfnApplicationInferenceProfileProps(
                inference_profile_name="inferenceProfileName",
            
                # the properties below are optional
                description="description",
                model_source=bedrock.CfnApplicationInferenceProfile.InferenceProfileModelSourceProperty(
                    copy_from="copyFrom"
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__396027668dce29a45dd42acb9bb8358fa9135a15e45d6925e8a78acced3bd78d)
            check_type(argname="argument inference_profile_name", value=inference_profile_name, expected_type=type_hints["inference_profile_name"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument model_source", value=model_source, expected_type=type_hints["model_source"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "inference_profile_name": inference_profile_name,
        }
        if description is not None:
            self._values["description"] = description
        if model_source is not None:
            self._values["model_source"] = model_source
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def inference_profile_name(self) -> builtins.str:
        '''The name of the inference profile.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-applicationinferenceprofile.html#cfn-bedrock-applicationinferenceprofile-inferenceprofilename
        '''
        result = self._values.get("inference_profile_name")
        assert result is not None, "Required property 'inference_profile_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the inference profile.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-applicationinferenceprofile.html#cfn-bedrock-applicationinferenceprofile-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def model_source(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnApplicationInferenceProfile.InferenceProfileModelSourceProperty]]:
        '''Contains configurations for the inference profile to copy as the resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-applicationinferenceprofile.html#cfn-bedrock-applicationinferenceprofile-modelsource
        '''
        result = self._values.get("model_source")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnApplicationInferenceProfile.InferenceProfileModelSourceProperty]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''A list of tags associated with the inference profile.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-applicationinferenceprofile.html#cfn-bedrock-applicationinferenceprofile-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnApplicationInferenceProfileProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnBlueprint(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_bedrock.CfnBlueprint",
):
    '''Details about a data automation blueprint.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-blueprint.html
    :cloudformationResource: AWS::Bedrock::Blueprint
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_bedrock as bedrock
        
        # schema: Any
        
        cfn_blueprint = bedrock.CfnBlueprint(self, "MyCfnBlueprint",
            blueprint_name="blueprintName",
            schema=schema,
            type="type",
        
            # the properties below are optional
            kms_encryption_context={
                "kms_encryption_context_key": "kmsEncryptionContext"
            },
            kms_key_id="kmsKeyId",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        blueprint_name: builtins.str,
        schema: typing.Any,
        type: builtins.str,
        kms_encryption_context: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param blueprint_name: The blueprint's name.
        :param schema: The blueprint's schema.
        :param type: The blueprint's type.
        :param kms_encryption_context: Name-value pairs to include as an encryption context.
        :param kms_key_id: The AWS KMS key to use for encryption.
        :param tags: List of Tags.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f30fc18d24fb84f6af985d23602089a0bfe10014de2e42af131b5519fc31c537)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnBlueprintProps(
            blueprint_name=blueprint_name,
            schema=schema,
            type=type,
            kms_encryption_context=kms_encryption_context,
            kms_key_id=kms_key_id,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cf5677a55c9a6605bcaccfa49e6f04c371f839de14216f93f87064deb0c6dc78)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__66d1fc3a88ea934c805135fc981cbbb6cfb1503b8c5d668c96a9c3fc5ffd284c)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrBlueprintArn")
    def attr_blueprint_arn(self) -> builtins.str:
        '''The blueprint's ARN.

        :cloudformationAttribute: BlueprintArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrBlueprintArn"))

    @builtins.property
    @jsii.member(jsii_name="attrBlueprintStage")
    def attr_blueprint_stage(self) -> builtins.str:
        '''The blueprint's stage.

        :cloudformationAttribute: BlueprintStage
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrBlueprintStage"))

    @builtins.property
    @jsii.member(jsii_name="attrCreationTime")
    def attr_creation_time(self) -> builtins.str:
        '''When the blueprint was created.

        :cloudformationAttribute: CreationTime
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreationTime"))

    @builtins.property
    @jsii.member(jsii_name="attrLastModifiedTime")
    def attr_last_modified_time(self) -> builtins.str:
        '''When the blueprint was last updated.

        :cloudformationAttribute: LastModifiedTime
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLastModifiedTime"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="blueprintName")
    def blueprint_name(self) -> builtins.str:
        '''The blueprint's name.'''
        return typing.cast(builtins.str, jsii.get(self, "blueprintName"))

    @blueprint_name.setter
    def blueprint_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5d8d16209c1f5c024594eca807940727054ad1e26101113a8f1d50e6e15a5694)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "blueprintName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="schema")
    def schema(self) -> typing.Any:
        '''The blueprint's schema.'''
        return typing.cast(typing.Any, jsii.get(self, "schema"))

    @schema.setter
    def schema(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__81e5d05d97dfb4480ecd4b06050ecd6d2148e3990b1dd1d0dfe9964f2963cb88)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "schema", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''The blueprint's type.'''
        return typing.cast(builtins.str, jsii.get(self, "type"))

    @type.setter
    def type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ef42e216fb09846368bece5ce6a3e9545529525ae50a99d9652a067c46c68ab)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "type", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="kmsEncryptionContext")
    def kms_encryption_context(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''Name-value pairs to include as an encryption context.'''
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], jsii.get(self, "kmsEncryptionContext"))

    @kms_encryption_context.setter
    def kms_encryption_context(
        self,
        value: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9dd7c2213aa68308e06ad8faf37f220972d3cc98e87609fbab559134d8803b82)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsEncryptionContext", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="kmsKeyId")
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key to use for encryption.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kmsKeyId"))

    @kms_key_id.setter
    def kms_key_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ebfd046b9ca03d7ae283da6715642f11589872c91f5d95bbd3cddd393ce3534e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsKeyId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''List of Tags.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tags"))

    @tags.setter
    def tags(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__886a403dedf81403e658ca68ff545b86f110fb792ac74080f77b3e5f75ff1205)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_bedrock.CfnBlueprintProps",
    jsii_struct_bases=[],
    name_mapping={
        "blueprint_name": "blueprintName",
        "schema": "schema",
        "type": "type",
        "kms_encryption_context": "kmsEncryptionContext",
        "kms_key_id": "kmsKeyId",
        "tags": "tags",
    },
)
class CfnBlueprintProps:
    def __init__(
        self,
        *,
        blueprint_name: builtins.str,
        schema: typing.Any,
        type: builtins.str,
        kms_encryption_context: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnBlueprint``.

        :param blueprint_name: The blueprint's name.
        :param schema: The blueprint's schema.
        :param type: The blueprint's type.
        :param kms_encryption_context: Name-value pairs to include as an encryption context.
        :param kms_key_id: The AWS KMS key to use for encryption.
        :param tags: List of Tags.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-blueprint.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_bedrock as bedrock
            
            # schema: Any
            
            cfn_blueprint_props = bedrock.CfnBlueprintProps(
                blueprint_name="blueprintName",
                schema=schema,
                type="type",
            
                # the properties below are optional
                kms_encryption_context={
                    "kms_encryption_context_key": "kmsEncryptionContext"
                },
                kms_key_id="kmsKeyId",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf95f6258abad47e6c2fe49a96a1875d9121287a505f3fdb3fd8135ecf5396de)
            check_type(argname="argument blueprint_name", value=blueprint_name, expected_type=type_hints["blueprint_name"])
            check_type(argname="argument schema", value=schema, expected_type=type_hints["schema"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument kms_encryption_context", value=kms_encryption_context, expected_type=type_hints["kms_encryption_context"])
            check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "blueprint_name": blueprint_name,
            "schema": schema,
            "type": type,
        }
        if kms_encryption_context is not None:
            self._values["kms_encryption_context"] = kms_encryption_context
        if kms_key_id is not None:
            self._values["kms_key_id"] = kms_key_id
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def blueprint_name(self) -> builtins.str:
        '''The blueprint's name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-blueprint.html#cfn-bedrock-blueprint-blueprintname
        '''
        result = self._values.get("blueprint_name")
        assert result is not None, "Required property 'blueprint_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def schema(self) -> typing.Any:
        '''The blueprint's schema.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-blueprint.html#cfn-bedrock-blueprint-schema
        '''
        result = self._values.get("schema")
        assert result is not None, "Required property 'schema' is missing"
        return typing.cast(typing.Any, result)

    @builtins.property
    def type(self) -> builtins.str:
        '''The blueprint's type.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-blueprint.html#cfn-bedrock-blueprint-type
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def kms_encryption_context(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''Name-value pairs to include as an encryption context.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-blueprint.html#cfn-bedrock-blueprint-kmsencryptioncontext
        '''
        result = self._values.get("kms_encryption_context")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

    @builtins.property
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key to use for encryption.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-blueprint.html#cfn-bedrock-blueprint-kmskeyid
        '''
        result = self._values.get("kms_key_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''List of Tags.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-blueprint.html#cfn-bedrock-blueprint-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnBlueprintProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnDataAutomationProject(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject",
):
    '''A data automation project.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-dataautomationproject.html
    :cloudformationResource: AWS::Bedrock::DataAutomationProject
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_bedrock as bedrock
        
        cfn_data_automation_project = bedrock.CfnDataAutomationProject(self, "MyCfnDataAutomationProject",
            project_name="projectName",
        
            # the properties below are optional
            custom_output_configuration=bedrock.CfnDataAutomationProject.CustomOutputConfigurationProperty(
                blueprints=[bedrock.CfnDataAutomationProject.BlueprintItemProperty(
                    blueprint_arn="blueprintArn",
        
                    # the properties below are optional
                    blueprint_stage="blueprintStage",
                    blueprint_version="blueprintVersion"
                )]
            ),
            kms_encryption_context={
                "kms_encryption_context_key": "kmsEncryptionContext"
            },
            kms_key_id="kmsKeyId",
            override_configuration=bedrock.CfnDataAutomationProject.OverrideConfigurationProperty(
                document=bedrock.CfnDataAutomationProject.DocumentOverrideConfigurationProperty(
                    splitter=bedrock.CfnDataAutomationProject.SplitterConfigurationProperty(
                        state="state"
                    )
                )
            ),
            project_description="projectDescription",
            standard_output_configuration=bedrock.CfnDataAutomationProject.StandardOutputConfigurationProperty(
                audio=bedrock.CfnDataAutomationProject.AudioStandardOutputConfigurationProperty(
                    extraction=bedrock.CfnDataAutomationProject.AudioStandardExtractionProperty(
                        category=bedrock.CfnDataAutomationProject.AudioExtractionCategoryProperty(
                            state="state",
        
                            # the properties below are optional
                            types=["types"]
                        )
                    ),
                    generative_field=bedrock.CfnDataAutomationProject.AudioStandardGenerativeFieldProperty(
                        state="state",
        
                        # the properties below are optional
                        types=["types"]
                    )
                ),
                document=bedrock.CfnDataAutomationProject.DocumentStandardOutputConfigurationProperty(
                    extraction=bedrock.CfnDataAutomationProject.DocumentStandardExtractionProperty(
                        bounding_box=bedrock.CfnDataAutomationProject.DocumentBoundingBoxProperty(
                            state="state"
                        ),
                        granularity=bedrock.CfnDataAutomationProject.DocumentExtractionGranularityProperty(
                            types=["types"]
                        )
                    ),
                    generative_field=bedrock.CfnDataAutomationProject.DocumentStandardGenerativeFieldProperty(
                        state="state"
                    ),
                    output_format=bedrock.CfnDataAutomationProject.DocumentOutputFormatProperty(
                        additional_file_format=bedrock.CfnDataAutomationProject.DocumentOutputAdditionalFileFormatProperty(
                            state="state"
                        ),
                        text_format=bedrock.CfnDataAutomationProject.DocumentOutputTextFormatProperty(
                            types=["types"]
                        )
                    )
                ),
                image=bedrock.CfnDataAutomationProject.ImageStandardOutputConfigurationProperty(
                    extraction=bedrock.CfnDataAutomationProject.ImageStandardExtractionProperty(
                        bounding_box=bedrock.CfnDataAutomationProject.ImageBoundingBoxProperty(
                            state="state"
                        ),
                        category=bedrock.CfnDataAutomationProject.ImageExtractionCategoryProperty(
                            state="state",
        
                            # the properties below are optional
                            types=["types"]
                        )
                    ),
                    generative_field=bedrock.CfnDataAutomationProject.ImageStandardGenerativeFieldProperty(
                        state="state",
        
                        # the properties below are optional
                        types=["types"]
                    )
                ),
                video=bedrock.CfnDataAutomationProject.VideoStandardOutputConfigurationProperty(
                    extraction=bedrock.CfnDataAutomationProject.VideoStandardExtractionProperty(
                        bounding_box=bedrock.CfnDataAutomationProject.VideoBoundingBoxProperty(
                            state="state"
                        ),
                        category=bedrock.CfnDataAutomationProject.VideoExtractionCategoryProperty(
                            state="state",
        
                            # the properties below are optional
                            types=["types"]
                        )
                    ),
                    generative_field=bedrock.CfnDataAutomationProject.VideoStandardGenerativeFieldProperty(
                        state="state",
        
                        # the properties below are optional
                        types=["types"]
                    )
                )
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        project_name: builtins.str,
        custom_output_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.CustomOutputConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        kms_encryption_context: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        override_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.OverrideConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        project_description: typing.Optional[builtins.str] = None,
        standard_output_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.StandardOutputConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param project_name: The project's name.
        :param custom_output_configuration: Blueprints to apply to objects processed by the project.
        :param kms_encryption_context: The AWS KMS encryption context to use for encryption.
        :param kms_key_id: The AWS KMS key to use for encryption.
        :param override_configuration: Additional settings for the project.
        :param project_description: The project's description.
        :param standard_output_configuration: The project's standard output configuration.
        :param tags: List of Tags.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__48d1f11084a27a96045646e74a00baea90ff4f5c949a72220c866f3e0d3e0249)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDataAutomationProjectProps(
            project_name=project_name,
            custom_output_configuration=custom_output_configuration,
            kms_encryption_context=kms_encryption_context,
            kms_key_id=kms_key_id,
            override_configuration=override_configuration,
            project_description=project_description,
            standard_output_configuration=standard_output_configuration,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__14abc329a8487afda0b6eda4590d182ad23fe7b88fb638b1068f7a6d6613d40c)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2905ece72b10d449178eb00a21a15f73ece8ef477abd2fae83c6ae92641ec933)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCreationTime")
    def attr_creation_time(self) -> builtins.str:
        '''When the project was created.

        :cloudformationAttribute: CreationTime
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreationTime"))

    @builtins.property
    @jsii.member(jsii_name="attrLastModifiedTime")
    def attr_last_modified_time(self) -> builtins.str:
        '''When the project was last updated.

        :cloudformationAttribute: LastModifiedTime
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLastModifiedTime"))

    @builtins.property
    @jsii.member(jsii_name="attrProjectArn")
    def attr_project_arn(self) -> builtins.str:
        '''The project's ARN.

        :cloudformationAttribute: ProjectArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrProjectArn"))

    @builtins.property
    @jsii.member(jsii_name="attrProjectStage")
    def attr_project_stage(self) -> builtins.str:
        '''The project's stage.

        :cloudformationAttribute: ProjectStage
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrProjectStage"))

    @builtins.property
    @jsii.member(jsii_name="attrStatus")
    def attr_status(self) -> builtins.str:
        '''The project's status.

        :cloudformationAttribute: Status
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrStatus"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="projectName")
    def project_name(self) -> builtins.str:
        '''The project's name.'''
        return typing.cast(builtins.str, jsii.get(self, "projectName"))

    @project_name.setter
    def project_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__44ebcab2ea1db454660f0efca5cbb1689c2e2e383bab8ac13160ded31828fddd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "projectName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="customOutputConfiguration")
    def custom_output_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.CustomOutputConfigurationProperty"]]:
        '''Blueprints to apply to objects processed by the project.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.CustomOutputConfigurationProperty"]], jsii.get(self, "customOutputConfiguration"))

    @custom_output_configuration.setter
    def custom_output_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.CustomOutputConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b1ab9fb9a4352d579b68cf610175f94c8aa52cfdbbf49f7714af576697d17844)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "customOutputConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="kmsEncryptionContext")
    def kms_encryption_context(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''The AWS KMS encryption context to use for encryption.'''
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], jsii.get(self, "kmsEncryptionContext"))

    @kms_encryption_context.setter
    def kms_encryption_context(
        self,
        value: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6cd36375cf66df3f91af57cca5986cf0e1e2e1d92450af01f6598a0f70704eda)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsEncryptionContext", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="kmsKeyId")
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key to use for encryption.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kmsKeyId"))

    @kms_key_id.setter
    def kms_key_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0b87c4142ac53f03ce6c5b48f72b8974482a859683fea89c79268bb107879366)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "kmsKeyId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="overrideConfiguration")
    def override_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.OverrideConfigurationProperty"]]:
        '''Additional settings for the project.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.OverrideConfigurationProperty"]], jsii.get(self, "overrideConfiguration"))

    @override_configuration.setter
    def override_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.OverrideConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__63214c23c1024f1e68bafcf83c39b09dcd7139f0c84e281682c8622de9336e74)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "overrideConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="projectDescription")
    def project_description(self) -> typing.Optional[builtins.str]:
        '''The project's description.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "projectDescription"))

    @project_description.setter
    def project_description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__66994affd3d7455a04b67bd84c11b22239237bd8483bb056b944d0895b576b11)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "projectDescription", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="standardOutputConfiguration")
    def standard_output_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.StandardOutputConfigurationProperty"]]:
        '''The project's standard output configuration.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.StandardOutputConfigurationProperty"]], jsii.get(self, "standardOutputConfiguration"))

    @standard_output_configuration.setter
    def standard_output_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.StandardOutputConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__58c4ae46198661b93f5957013f1db0f24086f81fb07595882186ae75392677a8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "standardOutputConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''List of Tags.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tags"))

    @tags.setter
    def tags(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c8adc93c3ad061e5a2be17a8fdad5d0761d30eeec1c7a815fcdef93e3acd3cd1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.AudioExtractionCategoryProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state", "types": "types"},
    )
    class AudioExtractionCategoryProperty:
        def __init__(
            self,
            *,
            state: builtins.str,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Settings for generating data from audio.

            :param state: Whether generating categorical data from audio is enabled.
            :param types: The types of data to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-audioextractioncategory.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                audio_extraction_category_property = bedrock.CfnDataAutomationProject.AudioExtractionCategoryProperty(
                    state="state",
                
                    # the properties below are optional
                    types=["types"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__17b4067ba061b31b33f6982f8b1ce49b15211698a89239a94ac83de6e9a86fd0)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "state": state,
            }
            if types is not None:
                self._values["types"] = types

        @builtins.property
        def state(self) -> builtins.str:
            '''Whether generating categorical data from audio is enabled.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-audioextractioncategory.html#cfn-bedrock-dataautomationproject-audioextractioncategory-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The types of data to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-audioextractioncategory.html#cfn-bedrock-dataautomationproject-audioextractioncategory-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AudioExtractionCategoryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.AudioStandardExtractionProperty",
        jsii_struct_bases=[],
        name_mapping={"category": "category"},
    )
    class AudioStandardExtractionProperty:
        def __init__(
            self,
            *,
            category: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.AudioExtractionCategoryProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''Settings for generating data from audio.

            :param category: Settings for generating data from audio.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-audiostandardextraction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                audio_standard_extraction_property = bedrock.CfnDataAutomationProject.AudioStandardExtractionProperty(
                    category=bedrock.CfnDataAutomationProject.AudioExtractionCategoryProperty(
                        state="state",
                
                        # the properties below are optional
                        types=["types"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0cd729d92f87cc8b9ea825fd9aba3f8ad84df311b23d20555e9348067db7ca2d)
                check_type(argname="argument category", value=category, expected_type=type_hints["category"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "category": category,
            }

        @builtins.property
        def category(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.AudioExtractionCategoryProperty"]:
            '''Settings for generating data from audio.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-audiostandardextraction.html#cfn-bedrock-dataautomationproject-audiostandardextraction-category
            '''
            result = self._values.get("category")
            assert result is not None, "Required property 'category' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.AudioExtractionCategoryProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AudioStandardExtractionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.AudioStandardGenerativeFieldProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state", "types": "types"},
    )
    class AudioStandardGenerativeFieldProperty:
        def __init__(
            self,
            *,
            state: builtins.str,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Settings for generating descriptions of audio.

            :param state: Whether generating descriptions is enabled for audio.
            :param types: The types of description to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-audiostandardgenerativefield.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                audio_standard_generative_field_property = bedrock.CfnDataAutomationProject.AudioStandardGenerativeFieldProperty(
                    state="state",
                
                    # the properties below are optional
                    types=["types"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5b5608d313a88cd314aa14811d46d2dfd47794b0a4f3088ed5c8eacb33bd29dd)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "state": state,
            }
            if types is not None:
                self._values["types"] = types

        @builtins.property
        def state(self) -> builtins.str:
            '''Whether generating descriptions is enabled for audio.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-audiostandardgenerativefield.html#cfn-bedrock-dataautomationproject-audiostandardgenerativefield-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The types of description to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-audiostandardgenerativefield.html#cfn-bedrock-dataautomationproject-audiostandardgenerativefield-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AudioStandardGenerativeFieldProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.AudioStandardOutputConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "extraction": "extraction",
            "generative_field": "generativeField",
        },
    )
    class AudioStandardOutputConfigurationProperty:
        def __init__(
            self,
            *,
            extraction: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.AudioStandardExtractionProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            generative_field: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.AudioStandardGenerativeFieldProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Output settings for processing audio.

            :param extraction: Settings for populating data fields that describe the audio.
            :param generative_field: Whether to generate descriptions of the data.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-audiostandardoutputconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                audio_standard_output_configuration_property = bedrock.CfnDataAutomationProject.AudioStandardOutputConfigurationProperty(
                    extraction=bedrock.CfnDataAutomationProject.AudioStandardExtractionProperty(
                        category=bedrock.CfnDataAutomationProject.AudioExtractionCategoryProperty(
                            state="state",
                
                            # the properties below are optional
                            types=["types"]
                        )
                    ),
                    generative_field=bedrock.CfnDataAutomationProject.AudioStandardGenerativeFieldProperty(
                        state="state",
                
                        # the properties below are optional
                        types=["types"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ace4af193273986ad67be780f429ddd01a7d6c0ef84a0a9f5f1333bd5e16c786)
                check_type(argname="argument extraction", value=extraction, expected_type=type_hints["extraction"])
                check_type(argname="argument generative_field", value=generative_field, expected_type=type_hints["generative_field"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if extraction is not None:
                self._values["extraction"] = extraction
            if generative_field is not None:
                self._values["generative_field"] = generative_field

        @builtins.property
        def extraction(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.AudioStandardExtractionProperty"]]:
            '''Settings for populating data fields that describe the audio.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-audiostandardoutputconfiguration.html#cfn-bedrock-dataautomationproject-audiostandardoutputconfiguration-extraction
            '''
            result = self._values.get("extraction")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.AudioStandardExtractionProperty"]], result)

        @builtins.property
        def generative_field(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.AudioStandardGenerativeFieldProperty"]]:
            '''Whether to generate descriptions of the data.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-audiostandardoutputconfiguration.html#cfn-bedrock-dataautomationproject-audiostandardoutputconfiguration-generativefield
            '''
            result = self._values.get("generative_field")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.AudioStandardGenerativeFieldProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AudioStandardOutputConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.BlueprintItemProperty",
        jsii_struct_bases=[],
        name_mapping={
            "blueprint_arn": "blueprintArn",
            "blueprint_stage": "blueprintStage",
            "blueprint_version": "blueprintVersion",
        },
    )
    class BlueprintItemProperty:
        def __init__(
            self,
            *,
            blueprint_arn: builtins.str,
            blueprint_stage: typing.Optional[builtins.str] = None,
            blueprint_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An abbreviated summary of a blueprint.

            :param blueprint_arn: The blueprint's ARN.
            :param blueprint_stage: The blueprint's stage.
            :param blueprint_version: The blueprint's version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-blueprintitem.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                blueprint_item_property = bedrock.CfnDataAutomationProject.BlueprintItemProperty(
                    blueprint_arn="blueprintArn",
                
                    # the properties below are optional
                    blueprint_stage="blueprintStage",
                    blueprint_version="blueprintVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b0aacb3579a06fb3e50ee6ecb37244fb2c3691447d7c0d5ce515e59f8a7f9923)
                check_type(argname="argument blueprint_arn", value=blueprint_arn, expected_type=type_hints["blueprint_arn"])
                check_type(argname="argument blueprint_stage", value=blueprint_stage, expected_type=type_hints["blueprint_stage"])
                check_type(argname="argument blueprint_version", value=blueprint_version, expected_type=type_hints["blueprint_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "blueprint_arn": blueprint_arn,
            }
            if blueprint_stage is not None:
                self._values["blueprint_stage"] = blueprint_stage
            if blueprint_version is not None:
                self._values["blueprint_version"] = blueprint_version

        @builtins.property
        def blueprint_arn(self) -> builtins.str:
            '''The blueprint's ARN.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-blueprintitem.html#cfn-bedrock-dataautomationproject-blueprintitem-blueprintarn
            '''
            result = self._values.get("blueprint_arn")
            assert result is not None, "Required property 'blueprint_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def blueprint_stage(self) -> typing.Optional[builtins.str]:
            '''The blueprint's stage.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-blueprintitem.html#cfn-bedrock-dataautomationproject-blueprintitem-blueprintstage
            '''
            result = self._values.get("blueprint_stage")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def blueprint_version(self) -> typing.Optional[builtins.str]:
            '''The blueprint's version.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-blueprintitem.html#cfn-bedrock-dataautomationproject-blueprintitem-blueprintversion
            '''
            result = self._values.get("blueprint_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BlueprintItemProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.CustomOutputConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"blueprints": "blueprints"},
    )
    class CustomOutputConfigurationProperty:
        def __init__(
            self,
            *,
            blueprints: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.BlueprintItemProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''Blueprints to apply to objects processed by the project.

            :param blueprints: A list of blueprints.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-customoutputconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                custom_output_configuration_property = bedrock.CfnDataAutomationProject.CustomOutputConfigurationProperty(
                    blueprints=[bedrock.CfnDataAutomationProject.BlueprintItemProperty(
                        blueprint_arn="blueprintArn",
                
                        # the properties below are optional
                        blueprint_stage="blueprintStage",
                        blueprint_version="blueprintVersion"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f33831635897ccc78d6e5d7bff1bc5df55cf8effc38e61a7795979a6201be7c4)
                check_type(argname="argument blueprints", value=blueprints, expected_type=type_hints["blueprints"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if blueprints is not None:
                self._values["blueprints"] = blueprints

        @builtins.property
        def blueprints(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.BlueprintItemProperty"]]]]:
            '''A list of blueprints.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-customoutputconfiguration.html#cfn-bedrock-dataautomationproject-customoutputconfiguration-blueprints
            '''
            result = self._values.get("blueprints")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.BlueprintItemProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CustomOutputConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.DocumentBoundingBoxProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state"},
    )
    class DocumentBoundingBoxProperty:
        def __init__(self, *, state: builtins.str) -> None:
            '''Bounding box settings for documents.

            :param state: Whether bounding boxes are enabled for documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentboundingbox.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                document_bounding_box_property = bedrock.CfnDataAutomationProject.DocumentBoundingBoxProperty(
                    state="state"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__638fc367c73405dc66937db6f237d5dca43475e4fc0277b21f7e119dbc4dcf1a)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "state": state,
            }

        @builtins.property
        def state(self) -> builtins.str:
            '''Whether bounding boxes are enabled for documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentboundingbox.html#cfn-bedrock-dataautomationproject-documentboundingbox-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DocumentBoundingBoxProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.DocumentExtractionGranularityProperty",
        jsii_struct_bases=[],
        name_mapping={"types": "types"},
    )
    class DocumentExtractionGranularityProperty:
        def __init__(
            self,
            *,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Granularity settings for documents.

            :param types: Granularity settings for documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentextractiongranularity.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                document_extraction_granularity_property = bedrock.CfnDataAutomationProject.DocumentExtractionGranularityProperty(
                    types=["types"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__658d0f08a19c50c55c6ab5bc4ba13b16deec781eac1399563598ee6eac503804)
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if types is not None:
                self._values["types"] = types

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Granularity settings for documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentextractiongranularity.html#cfn-bedrock-dataautomationproject-documentextractiongranularity-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DocumentExtractionGranularityProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.DocumentOutputAdditionalFileFormatProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state"},
    )
    class DocumentOutputAdditionalFileFormatProperty:
        def __init__(self, *, state: builtins.str) -> None:
            '''Output settings for additional file formats.

            :param state: Whether additional file formats are enabled for a project.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentoutputadditionalfileformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                document_output_additional_file_format_property = bedrock.CfnDataAutomationProject.DocumentOutputAdditionalFileFormatProperty(
                    state="state"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e27524306bfcf9e61defe359fa4f416c60feb05383b43015d3a516349c61fa3e)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "state": state,
            }

        @builtins.property
        def state(self) -> builtins.str:
            '''Whether additional file formats are enabled for a project.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentoutputadditionalfileformat.html#cfn-bedrock-dataautomationproject-documentoutputadditionalfileformat-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DocumentOutputAdditionalFileFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.DocumentOutputFormatProperty",
        jsii_struct_bases=[],
        name_mapping={
            "additional_file_format": "additionalFileFormat",
            "text_format": "textFormat",
        },
    )
    class DocumentOutputFormatProperty:
        def __init__(
            self,
            *,
            additional_file_format: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.DocumentOutputAdditionalFileFormatProperty", typing.Dict[builtins.str, typing.Any]]],
            text_format: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.DocumentOutputTextFormatProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''A document output format.

            :param additional_file_format: Output settings for additional file formats.
            :param text_format: An output text format.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentoutputformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                document_output_format_property = bedrock.CfnDataAutomationProject.DocumentOutputFormatProperty(
                    additional_file_format=bedrock.CfnDataAutomationProject.DocumentOutputAdditionalFileFormatProperty(
                        state="state"
                    ),
                    text_format=bedrock.CfnDataAutomationProject.DocumentOutputTextFormatProperty(
                        types=["types"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a2480f72d6ac1509529180b67b05a66327016a06e1e68e603bd07214a855bab7)
                check_type(argname="argument additional_file_format", value=additional_file_format, expected_type=type_hints["additional_file_format"])
                check_type(argname="argument text_format", value=text_format, expected_type=type_hints["text_format"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "additional_file_format": additional_file_format,
                "text_format": text_format,
            }

        @builtins.property
        def additional_file_format(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentOutputAdditionalFileFormatProperty"]:
            '''Output settings for additional file formats.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentoutputformat.html#cfn-bedrock-dataautomationproject-documentoutputformat-additionalfileformat
            '''
            result = self._values.get("additional_file_format")
            assert result is not None, "Required property 'additional_file_format' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentOutputAdditionalFileFormatProperty"], result)

        @builtins.property
        def text_format(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentOutputTextFormatProperty"]:
            '''An output text format.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentoutputformat.html#cfn-bedrock-dataautomationproject-documentoutputformat-textformat
            '''
            result = self._values.get("text_format")
            assert result is not None, "Required property 'text_format' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentOutputTextFormatProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DocumentOutputFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.DocumentOutputTextFormatProperty",
        jsii_struct_bases=[],
        name_mapping={"types": "types"},
    )
    class DocumentOutputTextFormatProperty:
        def __init__(
            self,
            *,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An output text format.

            :param types: The types of output text to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentoutputtextformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                document_output_text_format_property = bedrock.CfnDataAutomationProject.DocumentOutputTextFormatProperty(
                    types=["types"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b22792c9c0de4d07e8431f8ac3df8e64d6ae45cf4035db3f0657895096a71971)
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if types is not None:
                self._values["types"] = types

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The types of output text to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentoutputtextformat.html#cfn-bedrock-dataautomationproject-documentoutputtextformat-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DocumentOutputTextFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.DocumentOverrideConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"splitter": "splitter"},
    )
    class DocumentOverrideConfigurationProperty:
        def __init__(
            self,
            *,
            splitter: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.SplitterConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Additional settings for a project.

            :param splitter: Whether document splitter is enabled for a project.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentoverrideconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                document_override_configuration_property = bedrock.CfnDataAutomationProject.DocumentOverrideConfigurationProperty(
                    splitter=bedrock.CfnDataAutomationProject.SplitterConfigurationProperty(
                        state="state"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ce28bbcdad808c3d79746aae815f4706b7c664e45cfa21e90b91fef9b498a905)
                check_type(argname="argument splitter", value=splitter, expected_type=type_hints["splitter"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if splitter is not None:
                self._values["splitter"] = splitter

        @builtins.property
        def splitter(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.SplitterConfigurationProperty"]]:
            '''Whether document splitter is enabled for a project.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentoverrideconfiguration.html#cfn-bedrock-dataautomationproject-documentoverrideconfiguration-splitter
            '''
            result = self._values.get("splitter")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.SplitterConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DocumentOverrideConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.DocumentStandardExtractionProperty",
        jsii_struct_bases=[],
        name_mapping={"bounding_box": "boundingBox", "granularity": "granularity"},
    )
    class DocumentStandardExtractionProperty:
        def __init__(
            self,
            *,
            bounding_box: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.DocumentBoundingBoxProperty", typing.Dict[builtins.str, typing.Any]]],
            granularity: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.DocumentExtractionGranularityProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''Settings for generating data from documents.

            :param bounding_box: Whether to generate bounding boxes.
            :param granularity: Which granularities to generate data for.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentstandardextraction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                document_standard_extraction_property = bedrock.CfnDataAutomationProject.DocumentStandardExtractionProperty(
                    bounding_box=bedrock.CfnDataAutomationProject.DocumentBoundingBoxProperty(
                        state="state"
                    ),
                    granularity=bedrock.CfnDataAutomationProject.DocumentExtractionGranularityProperty(
                        types=["types"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6f5006234ba079760460d300fa64838c2631bdf02f7c1d15d07297f857521529)
                check_type(argname="argument bounding_box", value=bounding_box, expected_type=type_hints["bounding_box"])
                check_type(argname="argument granularity", value=granularity, expected_type=type_hints["granularity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "bounding_box": bounding_box,
                "granularity": granularity,
            }

        @builtins.property
        def bounding_box(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentBoundingBoxProperty"]:
            '''Whether to generate bounding boxes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentstandardextraction.html#cfn-bedrock-dataautomationproject-documentstandardextraction-boundingbox
            '''
            result = self._values.get("bounding_box")
            assert result is not None, "Required property 'bounding_box' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentBoundingBoxProperty"], result)

        @builtins.property
        def granularity(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentExtractionGranularityProperty"]:
            '''Which granularities to generate data for.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentstandardextraction.html#cfn-bedrock-dataautomationproject-documentstandardextraction-granularity
            '''
            result = self._values.get("granularity")
            assert result is not None, "Required property 'granularity' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentExtractionGranularityProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DocumentStandardExtractionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.DocumentStandardGenerativeFieldProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state"},
    )
    class DocumentStandardGenerativeFieldProperty:
        def __init__(self, *, state: builtins.str) -> None:
            '''Settings for generating descriptions of documents.

            :param state: Whether generating descriptions is enabled for documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentstandardgenerativefield.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                document_standard_generative_field_property = bedrock.CfnDataAutomationProject.DocumentStandardGenerativeFieldProperty(
                    state="state"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c2650d57759cf3c33400c0331a68258360514ed7c239cbdce0adcdb6faec41c0)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "state": state,
            }

        @builtins.property
        def state(self) -> builtins.str:
            '''Whether generating descriptions is enabled for documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentstandardgenerativefield.html#cfn-bedrock-dataautomationproject-documentstandardgenerativefield-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DocumentStandardGenerativeFieldProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.DocumentStandardOutputConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "extraction": "extraction",
            "generative_field": "generativeField",
            "output_format": "outputFormat",
        },
    )
    class DocumentStandardOutputConfigurationProperty:
        def __init__(
            self,
            *,
            extraction: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.DocumentStandardExtractionProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            generative_field: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.DocumentStandardGenerativeFieldProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            output_format: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.DocumentOutputFormatProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Output settings for processing documents.

            :param extraction: Settings for populating data fields that describe the document.
            :param generative_field: Whether to generate descriptions.
            :param output_format: The output format to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentstandardoutputconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                document_standard_output_configuration_property = bedrock.CfnDataAutomationProject.DocumentStandardOutputConfigurationProperty(
                    extraction=bedrock.CfnDataAutomationProject.DocumentStandardExtractionProperty(
                        bounding_box=bedrock.CfnDataAutomationProject.DocumentBoundingBoxProperty(
                            state="state"
                        ),
                        granularity=bedrock.CfnDataAutomationProject.DocumentExtractionGranularityProperty(
                            types=["types"]
                        )
                    ),
                    generative_field=bedrock.CfnDataAutomationProject.DocumentStandardGenerativeFieldProperty(
                        state="state"
                    ),
                    output_format=bedrock.CfnDataAutomationProject.DocumentOutputFormatProperty(
                        additional_file_format=bedrock.CfnDataAutomationProject.DocumentOutputAdditionalFileFormatProperty(
                            state="state"
                        ),
                        text_format=bedrock.CfnDataAutomationProject.DocumentOutputTextFormatProperty(
                            types=["types"]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f66b2c92645c1c5e9dfb2a198f3d0e18ad200e7e9977205814b91414d012651d)
                check_type(argname="argument extraction", value=extraction, expected_type=type_hints["extraction"])
                check_type(argname="argument generative_field", value=generative_field, expected_type=type_hints["generative_field"])
                check_type(argname="argument output_format", value=output_format, expected_type=type_hints["output_format"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if extraction is not None:
                self._values["extraction"] = extraction
            if generative_field is not None:
                self._values["generative_field"] = generative_field
            if output_format is not None:
                self._values["output_format"] = output_format

        @builtins.property
        def extraction(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentStandardExtractionProperty"]]:
            '''Settings for populating data fields that describe the document.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentstandardoutputconfiguration.html#cfn-bedrock-dataautomationproject-documentstandardoutputconfiguration-extraction
            '''
            result = self._values.get("extraction")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentStandardExtractionProperty"]], result)

        @builtins.property
        def generative_field(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentStandardGenerativeFieldProperty"]]:
            '''Whether to generate descriptions.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentstandardoutputconfiguration.html#cfn-bedrock-dataautomationproject-documentstandardoutputconfiguration-generativefield
            '''
            result = self._values.get("generative_field")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentStandardGenerativeFieldProperty"]], result)

        @builtins.property
        def output_format(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentOutputFormatProperty"]]:
            '''The output format to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-documentstandardoutputconfiguration.html#cfn-bedrock-dataautomationproject-documentstandardoutputconfiguration-outputformat
            '''
            result = self._values.get("output_format")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentOutputFormatProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DocumentStandardOutputConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.ImageBoundingBoxProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state"},
    )
    class ImageBoundingBoxProperty:
        def __init__(self, *, state: builtins.str) -> None:
            '''Bounding box settings for a project.

            :param state: Bounding box settings for a project.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imageboundingbox.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                image_bounding_box_property = bedrock.CfnDataAutomationProject.ImageBoundingBoxProperty(
                    state="state"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b50dac81b8c352bb40845509427e58ef92450cb8bc175e419faeb9a96e01bc35)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "state": state,
            }

        @builtins.property
        def state(self) -> builtins.str:
            '''Bounding box settings for a project.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imageboundingbox.html#cfn-bedrock-dataautomationproject-imageboundingbox-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ImageBoundingBoxProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.ImageExtractionCategoryProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state", "types": "types"},
    )
    class ImageExtractionCategoryProperty:
        def __init__(
            self,
            *,
            state: builtins.str,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Settings for generating categorical data from images.

            :param state: Whether generating categorical data from images is enabled.
            :param types: The types of data to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imageextractioncategory.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                image_extraction_category_property = bedrock.CfnDataAutomationProject.ImageExtractionCategoryProperty(
                    state="state",
                
                    # the properties below are optional
                    types=["types"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b889c5cec633874cf38a0cca25d340045efc344323f8c1f47b451c9845a1f265)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "state": state,
            }
            if types is not None:
                self._values["types"] = types

        @builtins.property
        def state(self) -> builtins.str:
            '''Whether generating categorical data from images is enabled.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imageextractioncategory.html#cfn-bedrock-dataautomationproject-imageextractioncategory-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The types of data to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imageextractioncategory.html#cfn-bedrock-dataautomationproject-imageextractioncategory-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ImageExtractionCategoryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.ImageStandardExtractionProperty",
        jsii_struct_bases=[],
        name_mapping={"bounding_box": "boundingBox", "category": "category"},
    )
    class ImageStandardExtractionProperty:
        def __init__(
            self,
            *,
            bounding_box: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.ImageBoundingBoxProperty", typing.Dict[builtins.str, typing.Any]]],
            category: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.ImageExtractionCategoryProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''Settings for generating data from images.

            :param bounding_box: Settings for generating bounding boxes.
            :param category: Settings for generating categorical data.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imagestandardextraction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                image_standard_extraction_property = bedrock.CfnDataAutomationProject.ImageStandardExtractionProperty(
                    bounding_box=bedrock.CfnDataAutomationProject.ImageBoundingBoxProperty(
                        state="state"
                    ),
                    category=bedrock.CfnDataAutomationProject.ImageExtractionCategoryProperty(
                        state="state",
                
                        # the properties below are optional
                        types=["types"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8509438c1738f2f2a62b74a8b21ad5bb1255f5b381218f083f1a15d23550dae1)
                check_type(argname="argument bounding_box", value=bounding_box, expected_type=type_hints["bounding_box"])
                check_type(argname="argument category", value=category, expected_type=type_hints["category"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "bounding_box": bounding_box,
                "category": category,
            }

        @builtins.property
        def bounding_box(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.ImageBoundingBoxProperty"]:
            '''Settings for generating bounding boxes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imagestandardextraction.html#cfn-bedrock-dataautomationproject-imagestandardextraction-boundingbox
            '''
            result = self._values.get("bounding_box")
            assert result is not None, "Required property 'bounding_box' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.ImageBoundingBoxProperty"], result)

        @builtins.property
        def category(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.ImageExtractionCategoryProperty"]:
            '''Settings for generating categorical data.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imagestandardextraction.html#cfn-bedrock-dataautomationproject-imagestandardextraction-category
            '''
            result = self._values.get("category")
            assert result is not None, "Required property 'category' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.ImageExtractionCategoryProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ImageStandardExtractionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.ImageStandardGenerativeFieldProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state", "types": "types"},
    )
    class ImageStandardGenerativeFieldProperty:
        def __init__(
            self,
            *,
            state: builtins.str,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Settings for generating descriptions of images.

            :param state: Whether generating descriptions is enabled for images.
            :param types: Settings for generating descriptions of images.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imagestandardgenerativefield.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                image_standard_generative_field_property = bedrock.CfnDataAutomationProject.ImageStandardGenerativeFieldProperty(
                    state="state",
                
                    # the properties below are optional
                    types=["types"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6d2e38c35b1a963bb9a80d669ba6289d462a47292cc52326d08b7c4ce22d5c60)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "state": state,
            }
            if types is not None:
                self._values["types"] = types

        @builtins.property
        def state(self) -> builtins.str:
            '''Whether generating descriptions is enabled for images.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imagestandardgenerativefield.html#cfn-bedrock-dataautomationproject-imagestandardgenerativefield-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Settings for generating descriptions of images.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imagestandardgenerativefield.html#cfn-bedrock-dataautomationproject-imagestandardgenerativefield-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ImageStandardGenerativeFieldProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.ImageStandardOutputConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "extraction": "extraction",
            "generative_field": "generativeField",
        },
    )
    class ImageStandardOutputConfigurationProperty:
        def __init__(
            self,
            *,
            extraction: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.ImageStandardExtractionProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            generative_field: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.ImageStandardGenerativeFieldProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Output settings for processing images.

            :param extraction: Settings for populating data fields that describe the image.
            :param generative_field: Whether to generate descriptions of the data.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imagestandardoutputconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                image_standard_output_configuration_property = bedrock.CfnDataAutomationProject.ImageStandardOutputConfigurationProperty(
                    extraction=bedrock.CfnDataAutomationProject.ImageStandardExtractionProperty(
                        bounding_box=bedrock.CfnDataAutomationProject.ImageBoundingBoxProperty(
                            state="state"
                        ),
                        category=bedrock.CfnDataAutomationProject.ImageExtractionCategoryProperty(
                            state="state",
                
                            # the properties below are optional
                            types=["types"]
                        )
                    ),
                    generative_field=bedrock.CfnDataAutomationProject.ImageStandardGenerativeFieldProperty(
                        state="state",
                
                        # the properties below are optional
                        types=["types"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cd2e094ff92095348e69264bdb8606dc0cb4bbaca0f0e5bb44b408709bbe44db)
                check_type(argname="argument extraction", value=extraction, expected_type=type_hints["extraction"])
                check_type(argname="argument generative_field", value=generative_field, expected_type=type_hints["generative_field"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if extraction is not None:
                self._values["extraction"] = extraction
            if generative_field is not None:
                self._values["generative_field"] = generative_field

        @builtins.property
        def extraction(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.ImageStandardExtractionProperty"]]:
            '''Settings for populating data fields that describe the image.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imagestandardoutputconfiguration.html#cfn-bedrock-dataautomationproject-imagestandardoutputconfiguration-extraction
            '''
            result = self._values.get("extraction")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.ImageStandardExtractionProperty"]], result)

        @builtins.property
        def generative_field(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.ImageStandardGenerativeFieldProperty"]]:
            '''Whether to generate descriptions of the data.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-imagestandardoutputconfiguration.html#cfn-bedrock-dataautomationproject-imagestandardoutputconfiguration-generativefield
            '''
            result = self._values.get("generative_field")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.ImageStandardGenerativeFieldProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ImageStandardOutputConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.OverrideConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"document": "document"},
    )
    class OverrideConfigurationProperty:
        def __init__(
            self,
            *,
            document: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.DocumentOverrideConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Additional settings for a project.

            :param document: Additional settings for a project.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-overrideconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                override_configuration_property = bedrock.CfnDataAutomationProject.OverrideConfigurationProperty(
                    document=bedrock.CfnDataAutomationProject.DocumentOverrideConfigurationProperty(
                        splitter=bedrock.CfnDataAutomationProject.SplitterConfigurationProperty(
                            state="state"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__400a92e228907b33f71ad10276a9ad472a21d517da3941d1621dda605806343e)
                check_type(argname="argument document", value=document, expected_type=type_hints["document"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if document is not None:
                self._values["document"] = document

        @builtins.property
        def document(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentOverrideConfigurationProperty"]]:
            '''Additional settings for a project.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-overrideconfiguration.html#cfn-bedrock-dataautomationproject-overrideconfiguration-document
            '''
            result = self._values.get("document")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentOverrideConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OverrideConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.SplitterConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state"},
    )
    class SplitterConfigurationProperty:
        def __init__(self, *, state: typing.Optional[builtins.str] = None) -> None:
            '''Document splitter settings.

            If a document is too large to be processed in one pass, the document splitter splits it into smaller documents.

            :param state: Whether document splitter is enabled for a project.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-splitterconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                splitter_configuration_property = bedrock.CfnDataAutomationProject.SplitterConfigurationProperty(
                    state="state"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c7cc9940422898532f5a5c63c45d42dbf9d52852abdd7380d339b4a0b5631abd)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if state is not None:
                self._values["state"] = state

        @builtins.property
        def state(self) -> typing.Optional[builtins.str]:
            '''Whether document splitter is enabled for a project.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-splitterconfiguration.html#cfn-bedrock-dataautomationproject-splitterconfiguration-state
            '''
            result = self._values.get("state")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SplitterConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.StandardOutputConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "audio": "audio",
            "document": "document",
            "image": "image",
            "video": "video",
        },
    )
    class StandardOutputConfigurationProperty:
        def __init__(
            self,
            *,
            audio: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.AudioStandardOutputConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            document: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.DocumentStandardOutputConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            image: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.ImageStandardOutputConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            video: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.VideoStandardOutputConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The project's standard output configuration.

            :param audio: Settings for processing audio.
            :param document: Settings for processing documents.
            :param image: Settings for processing images.
            :param video: Settings for processing video.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-standardoutputconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                standard_output_configuration_property = bedrock.CfnDataAutomationProject.StandardOutputConfigurationProperty(
                    audio=bedrock.CfnDataAutomationProject.AudioStandardOutputConfigurationProperty(
                        extraction=bedrock.CfnDataAutomationProject.AudioStandardExtractionProperty(
                            category=bedrock.CfnDataAutomationProject.AudioExtractionCategoryProperty(
                                state="state",
                
                                # the properties below are optional
                                types=["types"]
                            )
                        ),
                        generative_field=bedrock.CfnDataAutomationProject.AudioStandardGenerativeFieldProperty(
                            state="state",
                
                            # the properties below are optional
                            types=["types"]
                        )
                    ),
                    document=bedrock.CfnDataAutomationProject.DocumentStandardOutputConfigurationProperty(
                        extraction=bedrock.CfnDataAutomationProject.DocumentStandardExtractionProperty(
                            bounding_box=bedrock.CfnDataAutomationProject.DocumentBoundingBoxProperty(
                                state="state"
                            ),
                            granularity=bedrock.CfnDataAutomationProject.DocumentExtractionGranularityProperty(
                                types=["types"]
                            )
                        ),
                        generative_field=bedrock.CfnDataAutomationProject.DocumentStandardGenerativeFieldProperty(
                            state="state"
                        ),
                        output_format=bedrock.CfnDataAutomationProject.DocumentOutputFormatProperty(
                            additional_file_format=bedrock.CfnDataAutomationProject.DocumentOutputAdditionalFileFormatProperty(
                                state="state"
                            ),
                            text_format=bedrock.CfnDataAutomationProject.DocumentOutputTextFormatProperty(
                                types=["types"]
                            )
                        )
                    ),
                    image=bedrock.CfnDataAutomationProject.ImageStandardOutputConfigurationProperty(
                        extraction=bedrock.CfnDataAutomationProject.ImageStandardExtractionProperty(
                            bounding_box=bedrock.CfnDataAutomationProject.ImageBoundingBoxProperty(
                                state="state"
                            ),
                            category=bedrock.CfnDataAutomationProject.ImageExtractionCategoryProperty(
                                state="state",
                
                                # the properties below are optional
                                types=["types"]
                            )
                        ),
                        generative_field=bedrock.CfnDataAutomationProject.ImageStandardGenerativeFieldProperty(
                            state="state",
                
                            # the properties below are optional
                            types=["types"]
                        )
                    ),
                    video=bedrock.CfnDataAutomationProject.VideoStandardOutputConfigurationProperty(
                        extraction=bedrock.CfnDataAutomationProject.VideoStandardExtractionProperty(
                            bounding_box=bedrock.CfnDataAutomationProject.VideoBoundingBoxProperty(
                                state="state"
                            ),
                            category=bedrock.CfnDataAutomationProject.VideoExtractionCategoryProperty(
                                state="state",
                
                                # the properties below are optional
                                types=["types"]
                            )
                        ),
                        generative_field=bedrock.CfnDataAutomationProject.VideoStandardGenerativeFieldProperty(
                            state="state",
                
                            # the properties below are optional
                            types=["types"]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__73065f098ecb2ad872a4a32fc68952239056e92bbe4974bed8c813d7421e9aab)
                check_type(argname="argument audio", value=audio, expected_type=type_hints["audio"])
                check_type(argname="argument document", value=document, expected_type=type_hints["document"])
                check_type(argname="argument image", value=image, expected_type=type_hints["image"])
                check_type(argname="argument video", value=video, expected_type=type_hints["video"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if audio is not None:
                self._values["audio"] = audio
            if document is not None:
                self._values["document"] = document
            if image is not None:
                self._values["image"] = image
            if video is not None:
                self._values["video"] = video

        @builtins.property
        def audio(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.AudioStandardOutputConfigurationProperty"]]:
            '''Settings for processing audio.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-standardoutputconfiguration.html#cfn-bedrock-dataautomationproject-standardoutputconfiguration-audio
            '''
            result = self._values.get("audio")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.AudioStandardOutputConfigurationProperty"]], result)

        @builtins.property
        def document(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentStandardOutputConfigurationProperty"]]:
            '''Settings for processing documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-standardoutputconfiguration.html#cfn-bedrock-dataautomationproject-standardoutputconfiguration-document
            '''
            result = self._values.get("document")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.DocumentStandardOutputConfigurationProperty"]], result)

        @builtins.property
        def image(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.ImageStandardOutputConfigurationProperty"]]:
            '''Settings for processing images.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-standardoutputconfiguration.html#cfn-bedrock-dataautomationproject-standardoutputconfiguration-image
            '''
            result = self._values.get("image")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.ImageStandardOutputConfigurationProperty"]], result)

        @builtins.property
        def video(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.VideoStandardOutputConfigurationProperty"]]:
            '''Settings for processing video.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-standardoutputconfiguration.html#cfn-bedrock-dataautomationproject-standardoutputconfiguration-video
            '''
            result = self._values.get("video")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.VideoStandardOutputConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StandardOutputConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.VideoBoundingBoxProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state"},
    )
    class VideoBoundingBoxProperty:
        def __init__(self, *, state: builtins.str) -> None:
            '''Bounding box settings for video.

            :param state: Whether bounding boxes are enabled for video.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videoboundingbox.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                video_bounding_box_property = bedrock.CfnDataAutomationProject.VideoBoundingBoxProperty(
                    state="state"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6edaef4d6ad0693410640d9f6735dd17fecb6b8faa3f28bb8aa998b97dbd0b10)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "state": state,
            }

        @builtins.property
        def state(self) -> builtins.str:
            '''Whether bounding boxes are enabled for video.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videoboundingbox.html#cfn-bedrock-dataautomationproject-videoboundingbox-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VideoBoundingBoxProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.VideoExtractionCategoryProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state", "types": "types"},
    )
    class VideoExtractionCategoryProperty:
        def __init__(
            self,
            *,
            state: builtins.str,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Settings for generating categorical data from video.

            :param state: Whether generating categorical data from video is enabled.
            :param types: The types of data to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videoextractioncategory.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                video_extraction_category_property = bedrock.CfnDataAutomationProject.VideoExtractionCategoryProperty(
                    state="state",
                
                    # the properties below are optional
                    types=["types"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e8ab6d9031c06e119e307ce6230035cf42009ab028973d67c752347589a5389b)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "state": state,
            }
            if types is not None:
                self._values["types"] = types

        @builtins.property
        def state(self) -> builtins.str:
            '''Whether generating categorical data from video is enabled.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videoextractioncategory.html#cfn-bedrock-dataautomationproject-videoextractioncategory-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The types of data to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videoextractioncategory.html#cfn-bedrock-dataautomationproject-videoextractioncategory-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VideoExtractionCategoryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.VideoStandardExtractionProperty",
        jsii_struct_bases=[],
        name_mapping={"bounding_box": "boundingBox", "category": "category"},
    )
    class VideoStandardExtractionProperty:
        def __init__(
            self,
            *,
            bounding_box: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.VideoBoundingBoxProperty", typing.Dict[builtins.str, typing.Any]]],
            category: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.VideoExtractionCategoryProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''Settings for generating data from video.

            :param bounding_box: Settings for generating bounding boxes.
            :param category: Settings for generating categorical data.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videostandardextraction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                video_standard_extraction_property = bedrock.CfnDataAutomationProject.VideoStandardExtractionProperty(
                    bounding_box=bedrock.CfnDataAutomationProject.VideoBoundingBoxProperty(
                        state="state"
                    ),
                    category=bedrock.CfnDataAutomationProject.VideoExtractionCategoryProperty(
                        state="state",
                
                        # the properties below are optional
                        types=["types"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__170ed1a2312358b299f6226d2957f47b82b01af6cc40af84f66e89a6c1490049)
                check_type(argname="argument bounding_box", value=bounding_box, expected_type=type_hints["bounding_box"])
                check_type(argname="argument category", value=category, expected_type=type_hints["category"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "bounding_box": bounding_box,
                "category": category,
            }

        @builtins.property
        def bounding_box(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.VideoBoundingBoxProperty"]:
            '''Settings for generating bounding boxes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videostandardextraction.html#cfn-bedrock-dataautomationproject-videostandardextraction-boundingbox
            '''
            result = self._values.get("bounding_box")
            assert result is not None, "Required property 'bounding_box' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.VideoBoundingBoxProperty"], result)

        @builtins.property
        def category(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.VideoExtractionCategoryProperty"]:
            '''Settings for generating categorical data.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videostandardextraction.html#cfn-bedrock-dataautomationproject-videostandardextraction-category
            '''
            result = self._values.get("category")
            assert result is not None, "Required property 'category' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.VideoExtractionCategoryProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VideoStandardExtractionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.VideoStandardGenerativeFieldProperty",
        jsii_struct_bases=[],
        name_mapping={"state": "state", "types": "types"},
    )
    class VideoStandardGenerativeFieldProperty:
        def __init__(
            self,
            *,
            state: builtins.str,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''Settings for generating descriptions of video.

            :param state: Whether generating descriptions is enabled for video.
            :param types: The types of description to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videostandardgenerativefield.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                video_standard_generative_field_property = bedrock.CfnDataAutomationProject.VideoStandardGenerativeFieldProperty(
                    state="state",
                
                    # the properties below are optional
                    types=["types"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fae019d8f31423525c52c8e907f92525da76adc8e4b99705436675579d0954c9)
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "state": state,
            }
            if types is not None:
                self._values["types"] = types

        @builtins.property
        def state(self) -> builtins.str:
            '''Whether generating descriptions is enabled for video.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videostandardgenerativefield.html#cfn-bedrock-dataautomationproject-videostandardgenerativefield-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The types of description to generate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videostandardgenerativefield.html#cfn-bedrock-dataautomationproject-videostandardgenerativefield-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VideoStandardGenerativeFieldProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProject.VideoStandardOutputConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "extraction": "extraction",
            "generative_field": "generativeField",
        },
    )
    class VideoStandardOutputConfigurationProperty:
        def __init__(
            self,
            *,
            extraction: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.VideoStandardExtractionProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            generative_field: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataAutomationProject.VideoStandardGenerativeFieldProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Output settings for processing video.

            :param extraction: Settings for populating data fields that describe the video.
            :param generative_field: Whether to generate descriptions of the video.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videostandardoutputconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                video_standard_output_configuration_property = bedrock.CfnDataAutomationProject.VideoStandardOutputConfigurationProperty(
                    extraction=bedrock.CfnDataAutomationProject.VideoStandardExtractionProperty(
                        bounding_box=bedrock.CfnDataAutomationProject.VideoBoundingBoxProperty(
                            state="state"
                        ),
                        category=bedrock.CfnDataAutomationProject.VideoExtractionCategoryProperty(
                            state="state",
                
                            # the properties below are optional
                            types=["types"]
                        )
                    ),
                    generative_field=bedrock.CfnDataAutomationProject.VideoStandardGenerativeFieldProperty(
                        state="state",
                
                        # the properties below are optional
                        types=["types"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d98976479f1b341fc120b37d7069960429719e93bd9850fb5b89a69a2d4f4ff1)
                check_type(argname="argument extraction", value=extraction, expected_type=type_hints["extraction"])
                check_type(argname="argument generative_field", value=generative_field, expected_type=type_hints["generative_field"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if extraction is not None:
                self._values["extraction"] = extraction
            if generative_field is not None:
                self._values["generative_field"] = generative_field

        @builtins.property
        def extraction(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.VideoStandardExtractionProperty"]]:
            '''Settings for populating data fields that describe the video.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videostandardoutputconfiguration.html#cfn-bedrock-dataautomationproject-videostandardoutputconfiguration-extraction
            '''
            result = self._values.get("extraction")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.VideoStandardExtractionProperty"]], result)

        @builtins.property
        def generative_field(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.VideoStandardGenerativeFieldProperty"]]:
            '''Whether to generate descriptions of the video.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-dataautomationproject-videostandardoutputconfiguration.html#cfn-bedrock-dataautomationproject-videostandardoutputconfiguration-generativefield
            '''
            result = self._values.get("generative_field")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataAutomationProject.VideoStandardGenerativeFieldProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VideoStandardOutputConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_bedrock.CfnDataAutomationProjectProps",
    jsii_struct_bases=[],
    name_mapping={
        "project_name": "projectName",
        "custom_output_configuration": "customOutputConfiguration",
        "kms_encryption_context": "kmsEncryptionContext",
        "kms_key_id": "kmsKeyId",
        "override_configuration": "overrideConfiguration",
        "project_description": "projectDescription",
        "standard_output_configuration": "standardOutputConfiguration",
        "tags": "tags",
    },
)
class CfnDataAutomationProjectProps:
    def __init__(
        self,
        *,
        project_name: builtins.str,
        custom_output_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDataAutomationProject.CustomOutputConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        kms_encryption_context: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        kms_key_id: typing.Optional[builtins.str] = None,
        override_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDataAutomationProject.OverrideConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        project_description: typing.Optional[builtins.str] = None,
        standard_output_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDataAutomationProject.StandardOutputConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDataAutomationProject``.

        :param project_name: The project's name.
        :param custom_output_configuration: Blueprints to apply to objects processed by the project.
        :param kms_encryption_context: The AWS KMS encryption context to use for encryption.
        :param kms_key_id: The AWS KMS key to use for encryption.
        :param override_configuration: Additional settings for the project.
        :param project_description: The project's description.
        :param standard_output_configuration: The project's standard output configuration.
        :param tags: List of Tags.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-dataautomationproject.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_bedrock as bedrock
            
            cfn_data_automation_project_props = bedrock.CfnDataAutomationProjectProps(
                project_name="projectName",
            
                # the properties below are optional
                custom_output_configuration=bedrock.CfnDataAutomationProject.CustomOutputConfigurationProperty(
                    blueprints=[bedrock.CfnDataAutomationProject.BlueprintItemProperty(
                        blueprint_arn="blueprintArn",
            
                        # the properties below are optional
                        blueprint_stage="blueprintStage",
                        blueprint_version="blueprintVersion"
                    )]
                ),
                kms_encryption_context={
                    "kms_encryption_context_key": "kmsEncryptionContext"
                },
                kms_key_id="kmsKeyId",
                override_configuration=bedrock.CfnDataAutomationProject.OverrideConfigurationProperty(
                    document=bedrock.CfnDataAutomationProject.DocumentOverrideConfigurationProperty(
                        splitter=bedrock.CfnDataAutomationProject.SplitterConfigurationProperty(
                            state="state"
                        )
                    )
                ),
                project_description="projectDescription",
                standard_output_configuration=bedrock.CfnDataAutomationProject.StandardOutputConfigurationProperty(
                    audio=bedrock.CfnDataAutomationProject.AudioStandardOutputConfigurationProperty(
                        extraction=bedrock.CfnDataAutomationProject.AudioStandardExtractionProperty(
                            category=bedrock.CfnDataAutomationProject.AudioExtractionCategoryProperty(
                                state="state",
            
                                # the properties below are optional
                                types=["types"]
                            )
                        ),
                        generative_field=bedrock.CfnDataAutomationProject.AudioStandardGenerativeFieldProperty(
                            state="state",
            
                            # the properties below are optional
                            types=["types"]
                        )
                    ),
                    document=bedrock.CfnDataAutomationProject.DocumentStandardOutputConfigurationProperty(
                        extraction=bedrock.CfnDataAutomationProject.DocumentStandardExtractionProperty(
                            bounding_box=bedrock.CfnDataAutomationProject.DocumentBoundingBoxProperty(
                                state="state"
                            ),
                            granularity=bedrock.CfnDataAutomationProject.DocumentExtractionGranularityProperty(
                                types=["types"]
                            )
                        ),
                        generative_field=bedrock.CfnDataAutomationProject.DocumentStandardGenerativeFieldProperty(
                            state="state"
                        ),
                        output_format=bedrock.CfnDataAutomationProject.DocumentOutputFormatProperty(
                            additional_file_format=bedrock.CfnDataAutomationProject.DocumentOutputAdditionalFileFormatProperty(
                                state="state"
                            ),
                            text_format=bedrock.CfnDataAutomationProject.DocumentOutputTextFormatProperty(
                                types=["types"]
                            )
                        )
                    ),
                    image=bedrock.CfnDataAutomationProject.ImageStandardOutputConfigurationProperty(
                        extraction=bedrock.CfnDataAutomationProject.ImageStandardExtractionProperty(
                            bounding_box=bedrock.CfnDataAutomationProject.ImageBoundingBoxProperty(
                                state="state"
                            ),
                            category=bedrock.CfnDataAutomationProject.ImageExtractionCategoryProperty(
                                state="state",
            
                                # the properties below are optional
                                types=["types"]
                            )
                        ),
                        generative_field=bedrock.CfnDataAutomationProject.ImageStandardGenerativeFieldProperty(
                            state="state",
            
                            # the properties below are optional
                            types=["types"]
                        )
                    ),
                    video=bedrock.CfnDataAutomationProject.VideoStandardOutputConfigurationProperty(
                        extraction=bedrock.CfnDataAutomationProject.VideoStandardExtractionProperty(
                            bounding_box=bedrock.CfnDataAutomationProject.VideoBoundingBoxProperty(
                                state="state"
                            ),
                            category=bedrock.CfnDataAutomationProject.VideoExtractionCategoryProperty(
                                state="state",
            
                                # the properties below are optional
                                types=["types"]
                            )
                        ),
                        generative_field=bedrock.CfnDataAutomationProject.VideoStandardGenerativeFieldProperty(
                            state="state",
            
                            # the properties below are optional
                            types=["types"]
                        )
                    )
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__acae67d63fc78195ace949c20ad8c3209d7c1972a999616c83dd5f909bb13ca5)
            check_type(argname="argument project_name", value=project_name, expected_type=type_hints["project_name"])
            check_type(argname="argument custom_output_configuration", value=custom_output_configuration, expected_type=type_hints["custom_output_configuration"])
            check_type(argname="argument kms_encryption_context", value=kms_encryption_context, expected_type=type_hints["kms_encryption_context"])
            check_type(argname="argument kms_key_id", value=kms_key_id, expected_type=type_hints["kms_key_id"])
            check_type(argname="argument override_configuration", value=override_configuration, expected_type=type_hints["override_configuration"])
            check_type(argname="argument project_description", value=project_description, expected_type=type_hints["project_description"])
            check_type(argname="argument standard_output_configuration", value=standard_output_configuration, expected_type=type_hints["standard_output_configuration"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "project_name": project_name,
        }
        if custom_output_configuration is not None:
            self._values["custom_output_configuration"] = custom_output_configuration
        if kms_encryption_context is not None:
            self._values["kms_encryption_context"] = kms_encryption_context
        if kms_key_id is not None:
            self._values["kms_key_id"] = kms_key_id
        if override_configuration is not None:
            self._values["override_configuration"] = override_configuration
        if project_description is not None:
            self._values["project_description"] = project_description
        if standard_output_configuration is not None:
            self._values["standard_output_configuration"] = standard_output_configuration
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def project_name(self) -> builtins.str:
        '''The project's name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-dataautomationproject.html#cfn-bedrock-dataautomationproject-projectname
        '''
        result = self._values.get("project_name")
        assert result is not None, "Required property 'project_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def custom_output_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDataAutomationProject.CustomOutputConfigurationProperty]]:
        '''Blueprints to apply to objects processed by the project.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-dataautomationproject.html#cfn-bedrock-dataautomationproject-customoutputconfiguration
        '''
        result = self._values.get("custom_output_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDataAutomationProject.CustomOutputConfigurationProperty]], result)

    @builtins.property
    def kms_encryption_context(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''The AWS KMS encryption context to use for encryption.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-dataautomationproject.html#cfn-bedrock-dataautomationproject-kmsencryptioncontext
        '''
        result = self._values.get("kms_encryption_context")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

    @builtins.property
    def kms_key_id(self) -> typing.Optional[builtins.str]:
        '''The AWS KMS key to use for encryption.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-dataautomationproject.html#cfn-bedrock-dataautomationproject-kmskeyid
        '''
        result = self._values.get("kms_key_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def override_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDataAutomationProject.OverrideConfigurationProperty]]:
        '''Additional settings for the project.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-dataautomationproject.html#cfn-bedrock-dataautomationproject-overrideconfiguration
        '''
        result = self._values.get("override_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDataAutomationProject.OverrideConfigurationProperty]], result)

    @builtins.property
    def project_description(self) -> typing.Optional[builtins.str]:
        '''The project's description.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-dataautomationproject.html#cfn-bedrock-dataautomationproject-projectdescription
        '''
        result = self._values.get("project_description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def standard_output_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDataAutomationProject.StandardOutputConfigurationProperty]]:
        '''The project's standard output configuration.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-dataautomationproject.html#cfn-bedrock-dataautomationproject-standardoutputconfiguration
        '''
        result = self._values.get("standard_output_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDataAutomationProject.StandardOutputConfigurationProperty]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''List of Tags.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-dataautomationproject.html#cfn-bedrock-dataautomationproject-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDataAutomationProjectProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnDataSource(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource",
):
    '''.. epigraph::

   Properties with ``__Update requires: Replacement__`` can result in the creation of a new data source and deletion of the old one.

    This can happen if you also change the Name of the data source.

    Specifies a data source as a resource in a top-level template. Minimally, you must specify the following properties:

    - Name – Specify a name for the data source.
    - KnowledgeBaseId – Specify the ID of the knowledge base for the data source to belong to.
    - DataSourceConfiguration – Specify information about the Amazon S3 bucket containing the data source. The following sub-properties are required:
    - Type – Specify the value ``S3`` .

    For more information about setting up data sources in Amazon Bedrock , see `Set up a data source for your knowledge base <https://docs.aws.amazon.com/bedrock/latest/userguide/knowledge-base-ds.html>`_ .

    See the *Properties* section below for descriptions of both the required and optional properties.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-datasource.html
    :cloudformationResource: AWS::Bedrock::DataSource
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_bedrock as bedrock
        
        cfn_data_source = bedrock.CfnDataSource(self, "MyCfnDataSource",
            data_source_configuration=bedrock.CfnDataSource.DataSourceConfigurationProperty(
                type="type",
        
                # the properties below are optional
                confluence_configuration=bedrock.CfnDataSource.ConfluenceDataSourceConfigurationProperty(
                    source_configuration=bedrock.CfnDataSource.ConfluenceSourceConfigurationProperty(
                        auth_type="authType",
                        credentials_secret_arn="credentialsSecretArn",
                        host_type="hostType",
                        host_url="hostUrl"
                    ),
        
                    # the properties below are optional
                    crawler_configuration=bedrock.CfnDataSource.ConfluenceCrawlerConfigurationProperty(
                        filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                            type="type",
        
                            # the properties below are optional
                            pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                    object_type="objectType",
        
                                    # the properties below are optional
                                    exclusion_filters=["exclusionFilters"],
                                    inclusion_filters=["inclusionFilters"]
                                )]
                            )
                        )
                    )
                ),
                s3_configuration=bedrock.CfnDataSource.S3DataSourceConfigurationProperty(
                    bucket_arn="bucketArn",
        
                    # the properties below are optional
                    bucket_owner_account_id="bucketOwnerAccountId",
                    inclusion_prefixes=["inclusionPrefixes"]
                ),
                salesforce_configuration=bedrock.CfnDataSource.SalesforceDataSourceConfigurationProperty(
                    source_configuration=bedrock.CfnDataSource.SalesforceSourceConfigurationProperty(
                        auth_type="authType",
                        credentials_secret_arn="credentialsSecretArn",
                        host_url="hostUrl"
                    ),
        
                    # the properties below are optional
                    crawler_configuration=bedrock.CfnDataSource.SalesforceCrawlerConfigurationProperty(
                        filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                            type="type",
        
                            # the properties below are optional
                            pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                    object_type="objectType",
        
                                    # the properties below are optional
                                    exclusion_filters=["exclusionFilters"],
                                    inclusion_filters=["inclusionFilters"]
                                )]
                            )
                        )
                    )
                ),
                share_point_configuration=bedrock.CfnDataSource.SharePointDataSourceConfigurationProperty(
                    source_configuration=bedrock.CfnDataSource.SharePointSourceConfigurationProperty(
                        auth_type="authType",
                        credentials_secret_arn="credentialsSecretArn",
                        domain="domain",
                        host_type="hostType",
                        site_urls=["siteUrls"],
        
                        # the properties below are optional
                        tenant_id="tenantId"
                    ),
        
                    # the properties below are optional
                    crawler_configuration=bedrock.CfnDataSource.SharePointCrawlerConfigurationProperty(
                        filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                            type="type",
        
                            # the properties below are optional
                            pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                    object_type="objectType",
        
                                    # the properties below are optional
                                    exclusion_filters=["exclusionFilters"],
                                    inclusion_filters=["inclusionFilters"]
                                )]
                            )
                        )
                    )
                ),
                web_configuration=bedrock.CfnDataSource.WebDataSourceConfigurationProperty(
                    source_configuration=bedrock.CfnDataSource.WebSourceConfigurationProperty(
                        url_configuration=bedrock.CfnDataSource.UrlConfigurationProperty(
                            seed_urls=[bedrock.CfnDataSource.SeedUrlProperty(
                                url="url"
                            )]
                        )
                    ),
        
                    # the properties below are optional
                    crawler_configuration=bedrock.CfnDataSource.WebCrawlerConfigurationProperty(
                        crawler_limits=bedrock.CfnDataSource.WebCrawlerLimitsProperty(
                            max_pages=123,
                            rate_limit=123
                        ),
                        exclusion_filters=["exclusionFilters"],
                        inclusion_filters=["inclusionFilters"],
                        scope="scope",
                        user_agent="userAgent",
                        user_agent_header="userAgentHeader"
                    )
                )
            ),
            knowledge_base_id="knowledgeBaseId",
            name="name",
        
            # the properties below are optional
            data_deletion_policy="dataDeletionPolicy",
            description="description",
            server_side_encryption_configuration=bedrock.CfnDataSource.ServerSideEncryptionConfigurationProperty(
                kms_key_arn="kmsKeyArn"
            ),
            vector_ingestion_configuration=bedrock.CfnDataSource.VectorIngestionConfigurationProperty(
                chunking_configuration=bedrock.CfnDataSource.ChunkingConfigurationProperty(
                    chunking_strategy="chunkingStrategy",
        
                    # the properties below are optional
                    fixed_size_chunking_configuration=bedrock.CfnDataSource.FixedSizeChunkingConfigurationProperty(
                        max_tokens=123,
                        overlap_percentage=123
                    ),
                    hierarchical_chunking_configuration=bedrock.CfnDataSource.HierarchicalChunkingConfigurationProperty(
                        level_configurations=[bedrock.CfnDataSource.HierarchicalChunkingLevelConfigurationProperty(
                            max_tokens=123
                        )],
                        overlap_tokens=123
                    ),
                    semantic_chunking_configuration=bedrock.CfnDataSource.SemanticChunkingConfigurationProperty(
                        breakpoint_percentile_threshold=123,
                        buffer_size=123,
                        max_tokens=123
                    )
                ),
                context_enrichment_configuration=bedrock.CfnDataSource.ContextEnrichmentConfigurationProperty(
                    type="type",
        
                    # the properties below are optional
                    bedrock_foundation_model_configuration=bedrock.CfnDataSource.BedrockFoundationModelContextEnrichmentConfigurationProperty(
                        enrichment_strategy_configuration=bedrock.CfnDataSource.EnrichmentStrategyConfigurationProperty(
                            method="method"
                        ),
                        model_arn="modelArn"
                    )
                ),
                custom_transformation_configuration=bedrock.CfnDataSource.CustomTransformationConfigurationProperty(
                    intermediate_storage=bedrock.CfnDataSource.IntermediateStorageProperty(
                        s3_location=bedrock.CfnDataSource.S3LocationProperty(
                            uri="uri"
                        )
                    ),
                    transformations=[bedrock.CfnDataSource.TransformationProperty(
                        step_to_apply="stepToApply",
                        transformation_function=bedrock.CfnDataSource.TransformationFunctionProperty(
                            transformation_lambda_configuration=bedrock.CfnDataSource.TransformationLambdaConfigurationProperty(
                                lambda_arn="lambdaArn"
                            )
                        )
                    )]
                ),
                parsing_configuration=bedrock.CfnDataSource.ParsingConfigurationProperty(
                    parsing_strategy="parsingStrategy",
        
                    # the properties below are optional
                    bedrock_data_automation_configuration=bedrock.CfnDataSource.BedrockDataAutomationConfigurationProperty(
                        parsing_modality="parsingModality"
                    ),
                    bedrock_foundation_model_configuration=bedrock.CfnDataSource.BedrockFoundationModelConfigurationProperty(
                        model_arn="modelArn",
        
                        # the properties below are optional
                        parsing_modality="parsingModality",
                        parsing_prompt=bedrock.CfnDataSource.ParsingPromptProperty(
                            parsing_prompt_text="parsingPromptText"
                        )
                    )
                )
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        data_source_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.DataSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
        knowledge_base_id: builtins.str,
        name: builtins.str,
        data_deletion_policy: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        server_side_encryption_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.ServerSideEncryptionConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        vector_ingestion_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.VectorIngestionConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param data_source_configuration: The connection configuration for the data source.
        :param knowledge_base_id: The unique identifier of the knowledge base to which the data source belongs.
        :param name: The name of the data source.
        :param data_deletion_policy: The data deletion policy for the data source.
        :param description: The description of the data source.
        :param server_side_encryption_configuration: Contains details about the configuration of the server-side encryption.
        :param vector_ingestion_configuration: Contains details about how to ingest the documents in the data source.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8b6230da788965416b3262c41917eaab2193b3cfccf7a03f313a8c19186a6b75)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDataSourceProps(
            data_source_configuration=data_source_configuration,
            knowledge_base_id=knowledge_base_id,
            name=name,
            data_deletion_policy=data_deletion_policy,
            description=description,
            server_side_encryption_configuration=server_side_encryption_configuration,
            vector_ingestion_configuration=vector_ingestion_configuration,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e68a33df1c2e42bd5580cdec505e709cbe6a0663b8142f6fb3ba8e70911ceb85)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b23978dd788bb65ad2e2571764c987a4729f9d5c9d17c903f53a10cd4e3ad3e7)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCreatedAt")
    def attr_created_at(self) -> builtins.str:
        '''The time at which the data source was created.

        :cloudformationAttribute: CreatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreatedAt"))

    @builtins.property
    @jsii.member(jsii_name="attrDataSourceConfigurationWebConfigurationCrawlerConfigurationUserAgentHeader")
    def attr_data_source_configuration_web_configuration_crawler_configuration_user_agent_header(
        self,
    ) -> builtins.str:
        '''A string used for identifying the crawler or bot when it accesses a web server.

        The user agent header value consists of the ``bedrockbot`` , UUID, and a user agent suffix for your crawler (if one is provided). By default, it is set to ``bedrockbot_UUID`` . You can optionally append a custom suffix to ``bedrockbot_UUID`` to allowlist a specific user agent permitted to access your source URLs.

        :cloudformationAttribute: DataSourceConfiguration.WebConfiguration.CrawlerConfiguration.UserAgentHeader
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDataSourceConfigurationWebConfigurationCrawlerConfigurationUserAgentHeader"))

    @builtins.property
    @jsii.member(jsii_name="attrDataSourceId")
    def attr_data_source_id(self) -> builtins.str:
        '''The unique identifier of the data source.

        :cloudformationAttribute: DataSourceId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDataSourceId"))

    @builtins.property
    @jsii.member(jsii_name="attrDataSourceStatus")
    def attr_data_source_status(self) -> builtins.str:
        '''The status of the data source. The following statuses are possible:.

        - Available – The data source has been created and is ready for ingestion into the knowledge base.
        - Deleting – The data source is being deleted.

        :cloudformationAttribute: DataSourceStatus
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDataSourceStatus"))

    @builtins.property
    @jsii.member(jsii_name="attrFailureReasons")
    def attr_failure_reasons(self) -> typing.List[builtins.str]:
        '''The detailed reasons on the failure to delete a data source.

        :cloudformationAttribute: FailureReasons
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "attrFailureReasons"))

    @builtins.property
    @jsii.member(jsii_name="attrUpdatedAt")
    def attr_updated_at(self) -> builtins.str:
        '''The time at which the data source was last updated.

        :cloudformationAttribute: UpdatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUpdatedAt"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="dataSourceConfiguration")
    def data_source_configuration(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnDataSource.DataSourceConfigurationProperty"]:
        '''The connection configuration for the data source.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataSource.DataSourceConfigurationProperty"], jsii.get(self, "dataSourceConfiguration"))

    @data_source_configuration.setter
    def data_source_configuration(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnDataSource.DataSourceConfigurationProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a387108ccda2b96813f79478d4429abb297f766ce080c3bf9b77d03af3db534c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dataSourceConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="knowledgeBaseId")
    def knowledge_base_id(self) -> builtins.str:
        '''The unique identifier of the knowledge base to which the data source belongs.'''
        return typing.cast(builtins.str, jsii.get(self, "knowledgeBaseId"))

    @knowledge_base_id.setter
    def knowledge_base_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__62d585d364d6684899cbce3ae80a62858e2f256e166c889705e2ecb2040e935d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "knowledgeBaseId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the data source.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__99613771dd63beb77d047c2f17b9362e99ec38dd6fb89ec0c78f690f7847fc0b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="dataDeletionPolicy")
    def data_deletion_policy(self) -> typing.Optional[builtins.str]:
        '''The data deletion policy for the data source.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "dataDeletionPolicy"))

    @data_deletion_policy.setter
    def data_deletion_policy(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__13070c256fb6cad53f284f3bdd82c32491d75ab72033b19aa76e3daf539a2b8f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "dataDeletionPolicy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the data source.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7fc31b7620f239d59ff8a40cb711119c09cf659e115e22ba5d86a3d2a36aa17b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="serverSideEncryptionConfiguration")
    def server_side_encryption_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ServerSideEncryptionConfigurationProperty"]]:
        '''Contains details about the configuration of the server-side encryption.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ServerSideEncryptionConfigurationProperty"]], jsii.get(self, "serverSideEncryptionConfiguration"))

    @server_side_encryption_configuration.setter
    def server_side_encryption_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ServerSideEncryptionConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1fe19aafcbb9637979bddaf7a230433b26f791bb09744dabcf0e95496322565d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serverSideEncryptionConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="vectorIngestionConfiguration")
    def vector_ingestion_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.VectorIngestionConfigurationProperty"]]:
        '''Contains details about how to ingest the documents in the data source.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.VectorIngestionConfigurationProperty"]], jsii.get(self, "vectorIngestionConfiguration"))

    @vector_ingestion_configuration.setter
    def vector_ingestion_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.VectorIngestionConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__033990e14b671fe96b02a12658cd4beadf8bae975b4e0c94984d3e41913522c8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "vectorIngestionConfiguration", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.BedrockDataAutomationConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"parsing_modality": "parsingModality"},
    )
    class BedrockDataAutomationConfigurationProperty:
        def __init__(
            self,
            *,
            parsing_modality: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains configurations for using Amazon Bedrock Data Automation as the parser for ingesting your data sources.

            :param parsing_modality: Specifies whether to enable parsing of multimodal data, including both text and/or images.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-bedrockdataautomationconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                bedrock_data_automation_configuration_property = bedrock.CfnDataSource.BedrockDataAutomationConfigurationProperty(
                    parsing_modality="parsingModality"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d1c9cf855ebecb64cb9a92684072efa631cc07e2bded88aa0e8cfa6319f97834)
                check_type(argname="argument parsing_modality", value=parsing_modality, expected_type=type_hints["parsing_modality"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if parsing_modality is not None:
                self._values["parsing_modality"] = parsing_modality

        @builtins.property
        def parsing_modality(self) -> typing.Optional[builtins.str]:
            '''Specifies whether to enable parsing of multimodal data, including both text and/or images.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-bedrockdataautomationconfiguration.html#cfn-bedrock-datasource-bedrockdataautomationconfiguration-parsingmodality
            '''
            result = self._values.get("parsing_modality")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BedrockDataAutomationConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.BedrockFoundationModelConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "model_arn": "modelArn",
            "parsing_modality": "parsingModality",
            "parsing_prompt": "parsingPrompt",
        },
    )
    class BedrockFoundationModelConfigurationProperty:
        def __init__(
            self,
            *,
            model_arn: builtins.str,
            parsing_modality: typing.Optional[builtins.str] = None,
            parsing_prompt: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.ParsingPromptProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Settings for a foundation model used to parse documents for a data source.

            :param model_arn: The ARN of the foundation model to use for parsing.
            :param parsing_modality: Specifies whether to enable parsing of multimodal data, including both text and/or images.
            :param parsing_prompt: Instructions for interpreting the contents of a document.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-bedrockfoundationmodelconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                bedrock_foundation_model_configuration_property = bedrock.CfnDataSource.BedrockFoundationModelConfigurationProperty(
                    model_arn="modelArn",
                
                    # the properties below are optional
                    parsing_modality="parsingModality",
                    parsing_prompt=bedrock.CfnDataSource.ParsingPromptProperty(
                        parsing_prompt_text="parsingPromptText"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1988b824c4171f598d30cf03abdb56bc617fd654dd3daf3e641d11499c0e04ac)
                check_type(argname="argument model_arn", value=model_arn, expected_type=type_hints["model_arn"])
                check_type(argname="argument parsing_modality", value=parsing_modality, expected_type=type_hints["parsing_modality"])
                check_type(argname="argument parsing_prompt", value=parsing_prompt, expected_type=type_hints["parsing_prompt"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "model_arn": model_arn,
            }
            if parsing_modality is not None:
                self._values["parsing_modality"] = parsing_modality
            if parsing_prompt is not None:
                self._values["parsing_prompt"] = parsing_prompt

        @builtins.property
        def model_arn(self) -> builtins.str:
            '''The ARN of the foundation model to use for parsing.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-bedrockfoundationmodelconfiguration.html#cfn-bedrock-datasource-bedrockfoundationmodelconfiguration-modelarn
            '''
            result = self._values.get("model_arn")
            assert result is not None, "Required property 'model_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def parsing_modality(self) -> typing.Optional[builtins.str]:
            '''Specifies whether to enable parsing of multimodal data, including both text and/or images.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-bedrockfoundationmodelconfiguration.html#cfn-bedrock-datasource-bedrockfoundationmodelconfiguration-parsingmodality
            '''
            result = self._values.get("parsing_modality")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def parsing_prompt(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ParsingPromptProperty"]]:
            '''Instructions for interpreting the contents of a document.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-bedrockfoundationmodelconfiguration.html#cfn-bedrock-datasource-bedrockfoundationmodelconfiguration-parsingprompt
            '''
            result = self._values.get("parsing_prompt")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ParsingPromptProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BedrockFoundationModelConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.BedrockFoundationModelContextEnrichmentConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enrichment_strategy_configuration": "enrichmentStrategyConfiguration",
            "model_arn": "modelArn",
        },
    )
    class BedrockFoundationModelContextEnrichmentConfigurationProperty:
        def __init__(
            self,
            *,
            enrichment_strategy_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.EnrichmentStrategyConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
            model_arn: builtins.str,
        ) -> None:
            '''Context enrichment configuration is used to provide additional context to the RAG application using Amazon Bedrock foundation models.

            :param enrichment_strategy_configuration: The enrichment stategy used to provide additional context. For example, Neptune GraphRAG uses Amazon Bedrock foundation models to perform chunk entity extraction.
            :param model_arn: The Amazon Resource Name (ARN) of the model used to create vector embeddings for the knowledge base.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-bedrockfoundationmodelcontextenrichmentconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                bedrock_foundation_model_context_enrichment_configuration_property = bedrock.CfnDataSource.BedrockFoundationModelContextEnrichmentConfigurationProperty(
                    enrichment_strategy_configuration=bedrock.CfnDataSource.EnrichmentStrategyConfigurationProperty(
                        method="method"
                    ),
                    model_arn="modelArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5a89b4cf2c8acc5a8649678b03650df048cec78aecf83c8401430c76e7d47bc5)
                check_type(argname="argument enrichment_strategy_configuration", value=enrichment_strategy_configuration, expected_type=type_hints["enrichment_strategy_configuration"])
                check_type(argname="argument model_arn", value=model_arn, expected_type=type_hints["model_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "enrichment_strategy_configuration": enrichment_strategy_configuration,
                "model_arn": model_arn,
            }

        @builtins.property
        def enrichment_strategy_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataSource.EnrichmentStrategyConfigurationProperty"]:
            '''The enrichment stategy used to provide additional context.

            For example, Neptune GraphRAG uses Amazon Bedrock foundation models to perform chunk entity extraction.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-bedrockfoundationmodelcontextenrichmentconfiguration.html#cfn-bedrock-datasource-bedrockfoundationmodelcontextenrichmentconfiguration-enrichmentstrategyconfiguration
            '''
            result = self._values.get("enrichment_strategy_configuration")
            assert result is not None, "Required property 'enrichment_strategy_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataSource.EnrichmentStrategyConfigurationProperty"], result)

        @builtins.property
        def model_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the model used to create vector embeddings for the knowledge base.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-bedrockfoundationmodelcontextenrichmentconfiguration.html#cfn-bedrock-datasource-bedrockfoundationmodelcontextenrichmentconfiguration-modelarn
            '''
            result = self._values.get("model_arn")
            assert result is not None, "Required property 'model_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BedrockFoundationModelContextEnrichmentConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.ChunkingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "chunking_strategy": "chunkingStrategy",
            "fixed_size_chunking_configuration": "fixedSizeChunkingConfiguration",
            "hierarchical_chunking_configuration": "hierarchicalChunkingConfiguration",
            "semantic_chunking_configuration": "semanticChunkingConfiguration",
        },
    )
    class ChunkingConfigurationProperty:
        def __init__(
            self,
            *,
            chunking_strategy: builtins.str,
            fixed_size_chunking_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.FixedSizeChunkingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            hierarchical_chunking_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.HierarchicalChunkingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            semantic_chunking_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.SemanticChunkingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Details about how to chunk the documents in the data source.

            A *chunk* refers to an excerpt from a data source that is returned when the knowledge base that it belongs to is queried.

            :param chunking_strategy: Knowledge base can split your source data into chunks. A *chunk* refers to an excerpt from a data source that is returned when the knowledge base that it belongs to is queried. You have the following options for chunking your data. If you opt for ``NONE`` , then you may want to pre-process your files by splitting them up such that each file corresponds to a chunk. - ``FIXED_SIZE`` – Amazon Bedrock splits your source data into chunks of the approximate size that you set in the ``fixedSizeChunkingConfiguration`` . - ``HIERARCHICAL`` – Split documents into layers of chunks where the first layer contains large chunks, and the second layer contains smaller chunks derived from the first layer. - ``SEMANTIC`` – Split documents into chunks based on groups of similar content derived with natural language processing. - ``NONE`` – Amazon Bedrock treats each file as one chunk. If you choose this option, you may want to pre-process your documents by splitting them into separate files.
            :param fixed_size_chunking_configuration: Configurations for when you choose fixed-size chunking. If you set the ``chunkingStrategy`` as ``NONE`` , exclude this field.
            :param hierarchical_chunking_configuration: Settings for hierarchical document chunking for a data source. Hierarchical chunking splits documents into layers of chunks where the first layer contains large chunks, and the second layer contains smaller chunks derived from the first layer.
            :param semantic_chunking_configuration: Settings for semantic document chunking for a data source. Semantic chunking splits a document into into smaller documents based on groups of similar content derived from the text with natural language processing.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-chunkingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                chunking_configuration_property = bedrock.CfnDataSource.ChunkingConfigurationProperty(
                    chunking_strategy="chunkingStrategy",
                
                    # the properties below are optional
                    fixed_size_chunking_configuration=bedrock.CfnDataSource.FixedSizeChunkingConfigurationProperty(
                        max_tokens=123,
                        overlap_percentage=123
                    ),
                    hierarchical_chunking_configuration=bedrock.CfnDataSource.HierarchicalChunkingConfigurationProperty(
                        level_configurations=[bedrock.CfnDataSource.HierarchicalChunkingLevelConfigurationProperty(
                            max_tokens=123
                        )],
                        overlap_tokens=123
                    ),
                    semantic_chunking_configuration=bedrock.CfnDataSource.SemanticChunkingConfigurationProperty(
                        breakpoint_percentile_threshold=123,
                        buffer_size=123,
                        max_tokens=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8beec747a0f57f641fd4bd3b03f73b678c8e286785d948f80998682f0c0a40e5)
                check_type(argname="argument chunking_strategy", value=chunking_strategy, expected_type=type_hints["chunking_strategy"])
                check_type(argname="argument fixed_size_chunking_configuration", value=fixed_size_chunking_configuration, expected_type=type_hints["fixed_size_chunking_configuration"])
                check_type(argname="argument hierarchical_chunking_configuration", value=hierarchical_chunking_configuration, expected_type=type_hints["hierarchical_chunking_configuration"])
                check_type(argname="argument semantic_chunking_configuration", value=semantic_chunking_configuration, expected_type=type_hints["semantic_chunking_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "chunking_strategy": chunking_strategy,
            }
            if fixed_size_chunking_configuration is not None:
                self._values["fixed_size_chunking_configuration"] = fixed_size_chunking_configuration
            if hierarchical_chunking_configuration is not None:
                self._values["hierarchical_chunking_configuration"] = hierarchical_chunking_configuration
            if semantic_chunking_configuration is not None:
                self._values["semantic_chunking_configuration"] = semantic_chunking_configuration

        @builtins.property
        def chunking_strategy(self) -> builtins.str:
            '''Knowledge base can split your source data into chunks.

            A *chunk* refers to an excerpt from a data source that is returned when the knowledge base that it belongs to is queried. You have the following options for chunking your data. If you opt for ``NONE`` , then you may want to pre-process your files by splitting them up such that each file corresponds to a chunk.

            - ``FIXED_SIZE`` – Amazon Bedrock splits your source data into chunks of the approximate size that you set in the ``fixedSizeChunkingConfiguration`` .
            - ``HIERARCHICAL`` – Split documents into layers of chunks where the first layer contains large chunks, and the second layer contains smaller chunks derived from the first layer.
            - ``SEMANTIC`` – Split documents into chunks based on groups of similar content derived with natural language processing.
            - ``NONE`` – Amazon Bedrock treats each file as one chunk. If you choose this option, you may want to pre-process your documents by splitting them into separate files.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-chunkingconfiguration.html#cfn-bedrock-datasource-chunkingconfiguration-chunkingstrategy
            '''
            result = self._values.get("chunking_strategy")
            assert result is not None, "Required property 'chunking_strategy' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def fixed_size_chunking_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.FixedSizeChunkingConfigurationProperty"]]:
            '''Configurations for when you choose fixed-size chunking.

            If you set the ``chunkingStrategy`` as ``NONE`` , exclude this field.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-chunkingconfiguration.html#cfn-bedrock-datasource-chunkingconfiguration-fixedsizechunkingconfiguration
            '''
            result = self._values.get("fixed_size_chunking_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.FixedSizeChunkingConfigurationProperty"]], result)

        @builtins.property
        def hierarchical_chunking_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.HierarchicalChunkingConfigurationProperty"]]:
            '''Settings for hierarchical document chunking for a data source.

            Hierarchical chunking splits documents into layers of chunks where the first layer contains large chunks, and the second layer contains smaller chunks derived from the first layer.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-chunkingconfiguration.html#cfn-bedrock-datasource-chunkingconfiguration-hierarchicalchunkingconfiguration
            '''
            result = self._values.get("hierarchical_chunking_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.HierarchicalChunkingConfigurationProperty"]], result)

        @builtins.property
        def semantic_chunking_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SemanticChunkingConfigurationProperty"]]:
            '''Settings for semantic document chunking for a data source.

            Semantic chunking splits a document into into smaller documents based on groups of similar content derived from the text with natural language processing.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-chunkingconfiguration.html#cfn-bedrock-datasource-chunkingconfiguration-semanticchunkingconfiguration
            '''
            result = self._values.get("semantic_chunking_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SemanticChunkingConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ChunkingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.ConfluenceCrawlerConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"filter_configuration": "filterConfiguration"},
    )
    class ConfluenceCrawlerConfigurationProperty:
        def __init__(
            self,
            *,
            filter_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.CrawlFilterConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration of the Confluence content.

            For example, configuring specific types of Confluence content.

            :param filter_configuration: The configuration of filtering the Confluence content. For example, configuring regular expression patterns to include or exclude certain content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-confluencecrawlerconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                confluence_crawler_configuration_property = bedrock.CfnDataSource.ConfluenceCrawlerConfigurationProperty(
                    filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                        type="type",
                
                        # the properties below are optional
                        pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                            filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                object_type="objectType",
                
                                # the properties below are optional
                                exclusion_filters=["exclusionFilters"],
                                inclusion_filters=["inclusionFilters"]
                            )]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bd96cc70a533ca8725fe3ea6bf1237979d5f80d5f174652557eff27fb9b78e45)
                check_type(argname="argument filter_configuration", value=filter_configuration, expected_type=type_hints["filter_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if filter_configuration is not None:
                self._values["filter_configuration"] = filter_configuration

        @builtins.property
        def filter_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.CrawlFilterConfigurationProperty"]]:
            '''The configuration of filtering the Confluence content.

            For example, configuring regular expression patterns to include or exclude certain content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-confluencecrawlerconfiguration.html#cfn-bedrock-datasource-confluencecrawlerconfiguration-filterconfiguration
            '''
            result = self._values.get("filter_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.CrawlFilterConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConfluenceCrawlerConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.ConfluenceDataSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "source_configuration": "sourceConfiguration",
            "crawler_configuration": "crawlerConfiguration",
        },
    )
    class ConfluenceDataSourceConfigurationProperty:
        def __init__(
            self,
            *,
            source_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.ConfluenceSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
            crawler_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.ConfluenceCrawlerConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration information to connect to Confluence as your data source.

            :param source_configuration: The endpoint information to connect to your Confluence data source.
            :param crawler_configuration: The configuration of the Confluence content. For example, configuring specific types of Confluence content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-confluencedatasourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                confluence_data_source_configuration_property = bedrock.CfnDataSource.ConfluenceDataSourceConfigurationProperty(
                    source_configuration=bedrock.CfnDataSource.ConfluenceSourceConfigurationProperty(
                        auth_type="authType",
                        credentials_secret_arn="credentialsSecretArn",
                        host_type="hostType",
                        host_url="hostUrl"
                    ),
                
                    # the properties below are optional
                    crawler_configuration=bedrock.CfnDataSource.ConfluenceCrawlerConfigurationProperty(
                        filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                            type="type",
                
                            # the properties below are optional
                            pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                    object_type="objectType",
                
                                    # the properties below are optional
                                    exclusion_filters=["exclusionFilters"],
                                    inclusion_filters=["inclusionFilters"]
                                )]
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6012c6084fea535f9557ea06362a962b55ea34a970b8275cb8328381138ea3b2)
                check_type(argname="argument source_configuration", value=source_configuration, expected_type=type_hints["source_configuration"])
                check_type(argname="argument crawler_configuration", value=crawler_configuration, expected_type=type_hints["crawler_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "source_configuration": source_configuration,
            }
            if crawler_configuration is not None:
                self._values["crawler_configuration"] = crawler_configuration

        @builtins.property
        def source_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataSource.ConfluenceSourceConfigurationProperty"]:
            '''The endpoint information to connect to your Confluence data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-confluencedatasourceconfiguration.html#cfn-bedrock-datasource-confluencedatasourceconfiguration-sourceconfiguration
            '''
            result = self._values.get("source_configuration")
            assert result is not None, "Required property 'source_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataSource.ConfluenceSourceConfigurationProperty"], result)

        @builtins.property
        def crawler_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ConfluenceCrawlerConfigurationProperty"]]:
            '''The configuration of the Confluence content.

            For example, configuring specific types of Confluence content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-confluencedatasourceconfiguration.html#cfn-bedrock-datasource-confluencedatasourceconfiguration-crawlerconfiguration
            '''
            result = self._values.get("crawler_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ConfluenceCrawlerConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConfluenceDataSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.ConfluenceSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auth_type": "authType",
            "credentials_secret_arn": "credentialsSecretArn",
            "host_type": "hostType",
            "host_url": "hostUrl",
        },
    )
    class ConfluenceSourceConfigurationProperty:
        def __init__(
            self,
            *,
            auth_type: builtins.str,
            credentials_secret_arn: builtins.str,
            host_type: builtins.str,
            host_url: builtins.str,
        ) -> None:
            '''The endpoint information to connect to your Confluence data source.

            :param auth_type: The supported authentication type to authenticate and connect to your Confluence instance.
            :param credentials_secret_arn: The Amazon Resource Name of an AWS Secrets Manager secret that stores your authentication credentials for your Confluence instance URL. For more information on the key-value pairs that must be included in your secret, depending on your authentication type, see `Confluence connection configuration <https://docs.aws.amazon.com/bedrock/latest/userguide/confluence-data-source-connector.html#configuration-confluence-connector>`_ .
            :param host_type: The supported host type, whether online/cloud or server/on-premises.
            :param host_url: The Confluence host URL or instance URL.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-confluencesourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                confluence_source_configuration_property = bedrock.CfnDataSource.ConfluenceSourceConfigurationProperty(
                    auth_type="authType",
                    credentials_secret_arn="credentialsSecretArn",
                    host_type="hostType",
                    host_url="hostUrl"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f6b1b77774988ec57c830fc9a0a2a7c4c9f4497cdb86ab6c5fa2ebbb15d01b54)
                check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
                check_type(argname="argument credentials_secret_arn", value=credentials_secret_arn, expected_type=type_hints["credentials_secret_arn"])
                check_type(argname="argument host_type", value=host_type, expected_type=type_hints["host_type"])
                check_type(argname="argument host_url", value=host_url, expected_type=type_hints["host_url"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "auth_type": auth_type,
                "credentials_secret_arn": credentials_secret_arn,
                "host_type": host_type,
                "host_url": host_url,
            }

        @builtins.property
        def auth_type(self) -> builtins.str:
            '''The supported authentication type to authenticate and connect to your Confluence instance.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-confluencesourceconfiguration.html#cfn-bedrock-datasource-confluencesourceconfiguration-authtype
            '''
            result = self._values.get("auth_type")
            assert result is not None, "Required property 'auth_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def credentials_secret_arn(self) -> builtins.str:
            '''The Amazon Resource Name of an AWS Secrets Manager secret that stores your authentication credentials for your Confluence instance URL.

            For more information on the key-value pairs that must be included in your secret, depending on your authentication type, see `Confluence connection configuration <https://docs.aws.amazon.com/bedrock/latest/userguide/confluence-data-source-connector.html#configuration-confluence-connector>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-confluencesourceconfiguration.html#cfn-bedrock-datasource-confluencesourceconfiguration-credentialssecretarn
            '''
            result = self._values.get("credentials_secret_arn")
            assert result is not None, "Required property 'credentials_secret_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def host_type(self) -> builtins.str:
            '''The supported host type, whether online/cloud or server/on-premises.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-confluencesourceconfiguration.html#cfn-bedrock-datasource-confluencesourceconfiguration-hosttype
            '''
            result = self._values.get("host_type")
            assert result is not None, "Required property 'host_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def host_url(self) -> builtins.str:
            '''The Confluence host URL or instance URL.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-confluencesourceconfiguration.html#cfn-bedrock-datasource-confluencesourceconfiguration-hosturl
            '''
            result = self._values.get("host_url")
            assert result is not None, "Required property 'host_url' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConfluenceSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.ContextEnrichmentConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "type": "type",
            "bedrock_foundation_model_configuration": "bedrockFoundationModelConfiguration",
        },
    )
    class ContextEnrichmentConfigurationProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            bedrock_foundation_model_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.BedrockFoundationModelContextEnrichmentConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Context enrichment configuration is used to provide additional context to the RAG application.

            :param type: The method used for context enrichment. It must be Amazon Bedrock foundation models.
            :param bedrock_foundation_model_configuration: The configuration of the Amazon Bedrock foundation model used for context enrichment.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-contextenrichmentconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                context_enrichment_configuration_property = bedrock.CfnDataSource.ContextEnrichmentConfigurationProperty(
                    type="type",
                
                    # the properties below are optional
                    bedrock_foundation_model_configuration=bedrock.CfnDataSource.BedrockFoundationModelContextEnrichmentConfigurationProperty(
                        enrichment_strategy_configuration=bedrock.CfnDataSource.EnrichmentStrategyConfigurationProperty(
                            method="method"
                        ),
                        model_arn="modelArn"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4a8bedb8c03e6599f8bee498faf320fa0afe2ebb758107fb96251adfe3f9031f)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument bedrock_foundation_model_configuration", value=bedrock_foundation_model_configuration, expected_type=type_hints["bedrock_foundation_model_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if bedrock_foundation_model_configuration is not None:
                self._values["bedrock_foundation_model_configuration"] = bedrock_foundation_model_configuration

        @builtins.property
        def type(self) -> builtins.str:
            '''The method used for context enrichment.

            It must be Amazon Bedrock foundation models.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-contextenrichmentconfiguration.html#cfn-bedrock-datasource-contextenrichmentconfiguration-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def bedrock_foundation_model_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.BedrockFoundationModelContextEnrichmentConfigurationProperty"]]:
            '''The configuration of the Amazon Bedrock foundation model used for context enrichment.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-contextenrichmentconfiguration.html#cfn-bedrock-datasource-contextenrichmentconfiguration-bedrockfoundationmodelconfiguration
            '''
            result = self._values.get("bedrock_foundation_model_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.BedrockFoundationModelContextEnrichmentConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContextEnrichmentConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.CrawlFilterConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "pattern_object_filter": "patternObjectFilter"},
    )
    class CrawlFilterConfigurationProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            pattern_object_filter: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.PatternObjectFilterConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration of filtering the data source content.

            For example, configuring regular expression patterns to include or exclude certain content.

            :param type: The type of filtering that you want to apply to certain objects or content of the data source. For example, the ``PATTERN`` type is regular expression patterns you can apply to filter your content.
            :param pattern_object_filter: The configuration of filtering certain objects or content types of the data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-crawlfilterconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                crawl_filter_configuration_property = bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                    type="type",
                
                    # the properties below are optional
                    pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                        filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                            object_type="objectType",
                
                            # the properties below are optional
                            exclusion_filters=["exclusionFilters"],
                            inclusion_filters=["inclusionFilters"]
                        )]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__df771f43b4edc46788782c796b78e1b33220d226fe90547899ccc3c75ce15ac8)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument pattern_object_filter", value=pattern_object_filter, expected_type=type_hints["pattern_object_filter"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if pattern_object_filter is not None:
                self._values["pattern_object_filter"] = pattern_object_filter

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of filtering that you want to apply to certain objects or content of the data source.

            For example, the ``PATTERN`` type is regular expression patterns you can apply to filter your content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-crawlfilterconfiguration.html#cfn-bedrock-datasource-crawlfilterconfiguration-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def pattern_object_filter(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.PatternObjectFilterConfigurationProperty"]]:
            '''The configuration of filtering certain objects or content types of the data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-crawlfilterconfiguration.html#cfn-bedrock-datasource-crawlfilterconfiguration-patternobjectfilter
            '''
            result = self._values.get("pattern_object_filter")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.PatternObjectFilterConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CrawlFilterConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.CustomTransformationConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "intermediate_storage": "intermediateStorage",
            "transformations": "transformations",
        },
    )
    class CustomTransformationConfigurationProperty:
        def __init__(
            self,
            *,
            intermediate_storage: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.IntermediateStorageProperty", typing.Dict[builtins.str, typing.Any]]],
            transformations: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.TransformationProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''Settings for customizing steps in the data source content ingestion pipeline.

            You can configure the data source to process documents with a Lambda function after they are parsed and converted into chunks. When you add a post-chunking transformation, the service stores chunked documents in an S3 bucket and invokes a Lambda function to process them.

            To process chunked documents with a Lambda function, define an S3 bucket path for input and output objects, and a transformation that specifies the Lambda function to invoke. You can use the Lambda function to customize how chunks are split, and the metadata for each chunk.

            :param intermediate_storage: An S3 bucket path for input and output objects.
            :param transformations: A Lambda function that processes documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-customtransformationconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                custom_transformation_configuration_property = bedrock.CfnDataSource.CustomTransformationConfigurationProperty(
                    intermediate_storage=bedrock.CfnDataSource.IntermediateStorageProperty(
                        s3_location=bedrock.CfnDataSource.S3LocationProperty(
                            uri="uri"
                        )
                    ),
                    transformations=[bedrock.CfnDataSource.TransformationProperty(
                        step_to_apply="stepToApply",
                        transformation_function=bedrock.CfnDataSource.TransformationFunctionProperty(
                            transformation_lambda_configuration=bedrock.CfnDataSource.TransformationLambdaConfigurationProperty(
                                lambda_arn="lambdaArn"
                            )
                        )
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__888c8bb26072cb69c76ae9aeabf049ae5233c6a3f34f832513f1399aea8240f7)
                check_type(argname="argument intermediate_storage", value=intermediate_storage, expected_type=type_hints["intermediate_storage"])
                check_type(argname="argument transformations", value=transformations, expected_type=type_hints["transformations"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "intermediate_storage": intermediate_storage,
                "transformations": transformations,
            }

        @builtins.property
        def intermediate_storage(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataSource.IntermediateStorageProperty"]:
            '''An S3 bucket path for input and output objects.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-customtransformationconfiguration.html#cfn-bedrock-datasource-customtransformationconfiguration-intermediatestorage
            '''
            result = self._values.get("intermediate_storage")
            assert result is not None, "Required property 'intermediate_storage' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataSource.IntermediateStorageProperty"], result)

        @builtins.property
        def transformations(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDataSource.TransformationProperty"]]]:
            '''A Lambda function that processes documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-customtransformationconfiguration.html#cfn-bedrock-datasource-customtransformationconfiguration-transformations
            '''
            result = self._values.get("transformations")
            assert result is not None, "Required property 'transformations' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDataSource.TransformationProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CustomTransformationConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.DataSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "type": "type",
            "confluence_configuration": "confluenceConfiguration",
            "s3_configuration": "s3Configuration",
            "salesforce_configuration": "salesforceConfiguration",
            "share_point_configuration": "sharePointConfiguration",
            "web_configuration": "webConfiguration",
        },
    )
    class DataSourceConfigurationProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            confluence_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.ConfluenceDataSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            s3_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.S3DataSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            salesforce_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.SalesforceDataSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            share_point_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.SharePointDataSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            web_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.WebDataSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The connection configuration for the data source.

            :param type: The type of data source.
            :param confluence_configuration: The configuration information to connect to Confluence as your data source. .. epigraph:: Confluence data source connector is in preview release and is subject to change.
            :param s3_configuration: The configuration information to connect to Amazon S3 as your data source.
            :param salesforce_configuration: The configuration information to connect to Salesforce as your data source. .. epigraph:: Salesforce data source connector is in preview release and is subject to change.
            :param share_point_configuration: The configuration information to connect to SharePoint as your data source. .. epigraph:: SharePoint data source connector is in preview release and is subject to change.
            :param web_configuration: The configuration of web URLs to crawl for your data source. You should be authorized to crawl the URLs. .. epigraph:: Crawling web URLs as your data source is in preview release and is subject to change.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-datasourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                data_source_configuration_property = bedrock.CfnDataSource.DataSourceConfigurationProperty(
                    type="type",
                
                    # the properties below are optional
                    confluence_configuration=bedrock.CfnDataSource.ConfluenceDataSourceConfigurationProperty(
                        source_configuration=bedrock.CfnDataSource.ConfluenceSourceConfigurationProperty(
                            auth_type="authType",
                            credentials_secret_arn="credentialsSecretArn",
                            host_type="hostType",
                            host_url="hostUrl"
                        ),
                
                        # the properties below are optional
                        crawler_configuration=bedrock.CfnDataSource.ConfluenceCrawlerConfigurationProperty(
                            filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                                type="type",
                
                                # the properties below are optional
                                pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                    filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                        object_type="objectType",
                
                                        # the properties below are optional
                                        exclusion_filters=["exclusionFilters"],
                                        inclusion_filters=["inclusionFilters"]
                                    )]
                                )
                            )
                        )
                    ),
                    s3_configuration=bedrock.CfnDataSource.S3DataSourceConfigurationProperty(
                        bucket_arn="bucketArn",
                
                        # the properties below are optional
                        bucket_owner_account_id="bucketOwnerAccountId",
                        inclusion_prefixes=["inclusionPrefixes"]
                    ),
                    salesforce_configuration=bedrock.CfnDataSource.SalesforceDataSourceConfigurationProperty(
                        source_configuration=bedrock.CfnDataSource.SalesforceSourceConfigurationProperty(
                            auth_type="authType",
                            credentials_secret_arn="credentialsSecretArn",
                            host_url="hostUrl"
                        ),
                
                        # the properties below are optional
                        crawler_configuration=bedrock.CfnDataSource.SalesforceCrawlerConfigurationProperty(
                            filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                                type="type",
                
                                # the properties below are optional
                                pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                    filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                        object_type="objectType",
                
                                        # the properties below are optional
                                        exclusion_filters=["exclusionFilters"],
                                        inclusion_filters=["inclusionFilters"]
                                    )]
                                )
                            )
                        )
                    ),
                    share_point_configuration=bedrock.CfnDataSource.SharePointDataSourceConfigurationProperty(
                        source_configuration=bedrock.CfnDataSource.SharePointSourceConfigurationProperty(
                            auth_type="authType",
                            credentials_secret_arn="credentialsSecretArn",
                            domain="domain",
                            host_type="hostType",
                            site_urls=["siteUrls"],
                
                            # the properties below are optional
                            tenant_id="tenantId"
                        ),
                
                        # the properties below are optional
                        crawler_configuration=bedrock.CfnDataSource.SharePointCrawlerConfigurationProperty(
                            filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                                type="type",
                
                                # the properties below are optional
                                pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                    filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                        object_type="objectType",
                
                                        # the properties below are optional
                                        exclusion_filters=["exclusionFilters"],
                                        inclusion_filters=["inclusionFilters"]
                                    )]
                                )
                            )
                        )
                    ),
                    web_configuration=bedrock.CfnDataSource.WebDataSourceConfigurationProperty(
                        source_configuration=bedrock.CfnDataSource.WebSourceConfigurationProperty(
                            url_configuration=bedrock.CfnDataSource.UrlConfigurationProperty(
                                seed_urls=[bedrock.CfnDataSource.SeedUrlProperty(
                                    url="url"
                                )]
                            )
                        ),
                
                        # the properties below are optional
                        crawler_configuration=bedrock.CfnDataSource.WebCrawlerConfigurationProperty(
                            crawler_limits=bedrock.CfnDataSource.WebCrawlerLimitsProperty(
                                max_pages=123,
                                rate_limit=123
                            ),
                            exclusion_filters=["exclusionFilters"],
                            inclusion_filters=["inclusionFilters"],
                            scope="scope",
                            user_agent="userAgent",
                            user_agent_header="userAgentHeader"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a2e1b0c807d6904904c91cc13a2f47ee5db24090758446e26a864b57b3d36117)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument confluence_configuration", value=confluence_configuration, expected_type=type_hints["confluence_configuration"])
                check_type(argname="argument s3_configuration", value=s3_configuration, expected_type=type_hints["s3_configuration"])
                check_type(argname="argument salesforce_configuration", value=salesforce_configuration, expected_type=type_hints["salesforce_configuration"])
                check_type(argname="argument share_point_configuration", value=share_point_configuration, expected_type=type_hints["share_point_configuration"])
                check_type(argname="argument web_configuration", value=web_configuration, expected_type=type_hints["web_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if confluence_configuration is not None:
                self._values["confluence_configuration"] = confluence_configuration
            if s3_configuration is not None:
                self._values["s3_configuration"] = s3_configuration
            if salesforce_configuration is not None:
                self._values["salesforce_configuration"] = salesforce_configuration
            if share_point_configuration is not None:
                self._values["share_point_configuration"] = share_point_configuration
            if web_configuration is not None:
                self._values["web_configuration"] = web_configuration

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-datasourceconfiguration.html#cfn-bedrock-datasource-datasourceconfiguration-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def confluence_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ConfluenceDataSourceConfigurationProperty"]]:
            '''The configuration information to connect to Confluence as your data source.

            .. epigraph::

               Confluence data source connector is in preview release and is subject to change.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-datasourceconfiguration.html#cfn-bedrock-datasource-datasourceconfiguration-confluenceconfiguration
            '''
            result = self._values.get("confluence_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ConfluenceDataSourceConfigurationProperty"]], result)

        @builtins.property
        def s3_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.S3DataSourceConfigurationProperty"]]:
            '''The configuration information to connect to Amazon S3 as your data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-datasourceconfiguration.html#cfn-bedrock-datasource-datasourceconfiguration-s3configuration
            '''
            result = self._values.get("s3_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.S3DataSourceConfigurationProperty"]], result)

        @builtins.property
        def salesforce_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SalesforceDataSourceConfigurationProperty"]]:
            '''The configuration information to connect to Salesforce as your data source.

            .. epigraph::

               Salesforce data source connector is in preview release and is subject to change.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-datasourceconfiguration.html#cfn-bedrock-datasource-datasourceconfiguration-salesforceconfiguration
            '''
            result = self._values.get("salesforce_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SalesforceDataSourceConfigurationProperty"]], result)

        @builtins.property
        def share_point_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SharePointDataSourceConfigurationProperty"]]:
            '''The configuration information to connect to SharePoint as your data source.

            .. epigraph::

               SharePoint data source connector is in preview release and is subject to change.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-datasourceconfiguration.html#cfn-bedrock-datasource-datasourceconfiguration-sharepointconfiguration
            '''
            result = self._values.get("share_point_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SharePointDataSourceConfigurationProperty"]], result)

        @builtins.property
        def web_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.WebDataSourceConfigurationProperty"]]:
            '''The configuration of web URLs to crawl for your data source. You should be authorized to crawl the URLs.

            .. epigraph::

               Crawling web URLs as your data source is in preview release and is subject to change.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-datasourceconfiguration.html#cfn-bedrock-datasource-datasourceconfiguration-webconfiguration
            '''
            result = self._values.get("web_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.WebDataSourceConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DataSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.EnrichmentStrategyConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"method": "method"},
    )
    class EnrichmentStrategyConfigurationProperty:
        def __init__(self, *, method: builtins.str) -> None:
            '''The strategy used for performing context enrichment.

            :param method: The method used for the context enrichment strategy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-enrichmentstrategyconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                enrichment_strategy_configuration_property = bedrock.CfnDataSource.EnrichmentStrategyConfigurationProperty(
                    method="method"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__644d76f9050da2eaff32542043cca01ff61629f76ef2e0191725886223a5f368)
                check_type(argname="argument method", value=method, expected_type=type_hints["method"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "method": method,
            }

        @builtins.property
        def method(self) -> builtins.str:
            '''The method used for the context enrichment strategy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-enrichmentstrategyconfiguration.html#cfn-bedrock-datasource-enrichmentstrategyconfiguration-method
            '''
            result = self._values.get("method")
            assert result is not None, "Required property 'method' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EnrichmentStrategyConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.FixedSizeChunkingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_tokens": "maxTokens",
            "overlap_percentage": "overlapPercentage",
        },
    )
    class FixedSizeChunkingConfigurationProperty:
        def __init__(
            self,
            *,
            max_tokens: jsii.Number,
            overlap_percentage: jsii.Number,
        ) -> None:
            '''Configurations for when you choose fixed-size chunking.

            If you set the ``chunkingStrategy`` as ``NONE`` , exclude this field.

            :param max_tokens: The maximum number of tokens to include in a chunk.
            :param overlap_percentage: The percentage of overlap between adjacent chunks of a data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-fixedsizechunkingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                fixed_size_chunking_configuration_property = bedrock.CfnDataSource.FixedSizeChunkingConfigurationProperty(
                    max_tokens=123,
                    overlap_percentage=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7b9e8054cbf79d2292e738f1061947914954b89440b15bf572c87c9695aaf7bc)
                check_type(argname="argument max_tokens", value=max_tokens, expected_type=type_hints["max_tokens"])
                check_type(argname="argument overlap_percentage", value=overlap_percentage, expected_type=type_hints["overlap_percentage"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_tokens": max_tokens,
                "overlap_percentage": overlap_percentage,
            }

        @builtins.property
        def max_tokens(self) -> jsii.Number:
            '''The maximum number of tokens to include in a chunk.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-fixedsizechunkingconfiguration.html#cfn-bedrock-datasource-fixedsizechunkingconfiguration-maxtokens
            '''
            result = self._values.get("max_tokens")
            assert result is not None, "Required property 'max_tokens' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def overlap_percentage(self) -> jsii.Number:
            '''The percentage of overlap between adjacent chunks of a data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-fixedsizechunkingconfiguration.html#cfn-bedrock-datasource-fixedsizechunkingconfiguration-overlappercentage
            '''
            result = self._values.get("overlap_percentage")
            assert result is not None, "Required property 'overlap_percentage' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FixedSizeChunkingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.HierarchicalChunkingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "level_configurations": "levelConfigurations",
            "overlap_tokens": "overlapTokens",
        },
    )
    class HierarchicalChunkingConfigurationProperty:
        def __init__(
            self,
            *,
            level_configurations: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.HierarchicalChunkingLevelConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]]],
            overlap_tokens: jsii.Number,
        ) -> None:
            '''Settings for hierarchical document chunking for a data source.

            Hierarchical chunking splits documents into layers of chunks where the first layer contains large chunks, and the second layer contains smaller chunks derived from the first layer.

            You configure the number of tokens to overlap, or repeat across adjacent chunks. For example, if you set overlap tokens to 60, the last 60 tokens in the first chunk are also included at the beginning of the second chunk. For each layer, you must also configure the maximum number of tokens in a chunk.

            :param level_configurations: Token settings for each layer.
            :param overlap_tokens: The number of tokens to repeat across chunks in the same layer.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-hierarchicalchunkingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                hierarchical_chunking_configuration_property = bedrock.CfnDataSource.HierarchicalChunkingConfigurationProperty(
                    level_configurations=[bedrock.CfnDataSource.HierarchicalChunkingLevelConfigurationProperty(
                        max_tokens=123
                    )],
                    overlap_tokens=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5d601b03a460137760d534703cc747a11f33f25a281361a35ba9b1dd3a52c4d6)
                check_type(argname="argument level_configurations", value=level_configurations, expected_type=type_hints["level_configurations"])
                check_type(argname="argument overlap_tokens", value=overlap_tokens, expected_type=type_hints["overlap_tokens"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "level_configurations": level_configurations,
                "overlap_tokens": overlap_tokens,
            }

        @builtins.property
        def level_configurations(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDataSource.HierarchicalChunkingLevelConfigurationProperty"]]]:
            '''Token settings for each layer.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-hierarchicalchunkingconfiguration.html#cfn-bedrock-datasource-hierarchicalchunkingconfiguration-levelconfigurations
            '''
            result = self._values.get("level_configurations")
            assert result is not None, "Required property 'level_configurations' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDataSource.HierarchicalChunkingLevelConfigurationProperty"]]], result)

        @builtins.property
        def overlap_tokens(self) -> jsii.Number:
            '''The number of tokens to repeat across chunks in the same layer.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-hierarchicalchunkingconfiguration.html#cfn-bedrock-datasource-hierarchicalchunkingconfiguration-overlaptokens
            '''
            result = self._values.get("overlap_tokens")
            assert result is not None, "Required property 'overlap_tokens' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HierarchicalChunkingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.HierarchicalChunkingLevelConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"max_tokens": "maxTokens"},
    )
    class HierarchicalChunkingLevelConfigurationProperty:
        def __init__(self, *, max_tokens: jsii.Number) -> None:
            '''Token settings for a layer in a hierarchical chunking configuration.

            :param max_tokens: The maximum number of tokens that a chunk can contain in this layer.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-hierarchicalchunkinglevelconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                hierarchical_chunking_level_configuration_property = bedrock.CfnDataSource.HierarchicalChunkingLevelConfigurationProperty(
                    max_tokens=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a3a29631db955251086144f75baf14ef472ef7bbe42761f323c0edfce3e4754b)
                check_type(argname="argument max_tokens", value=max_tokens, expected_type=type_hints["max_tokens"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_tokens": max_tokens,
            }

        @builtins.property
        def max_tokens(self) -> jsii.Number:
            '''The maximum number of tokens that a chunk can contain in this layer.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-hierarchicalchunkinglevelconfiguration.html#cfn-bedrock-datasource-hierarchicalchunkinglevelconfiguration-maxtokens
            '''
            result = self._values.get("max_tokens")
            assert result is not None, "Required property 'max_tokens' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HierarchicalChunkingLevelConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.IntermediateStorageProperty",
        jsii_struct_bases=[],
        name_mapping={"s3_location": "s3Location"},
    )
    class IntermediateStorageProperty:
        def __init__(
            self,
            *,
            s3_location: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.S3LocationProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''A location for storing content from data sources temporarily as it is processed by custom components in the ingestion pipeline.

            :param s3_location: An S3 bucket path.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-intermediatestorage.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                intermediate_storage_property = bedrock.CfnDataSource.IntermediateStorageProperty(
                    s3_location=bedrock.CfnDataSource.S3LocationProperty(
                        uri="uri"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5bdd915748b8cf592aa9f03cf95eac889f69f61263bafe87a0ba9aa14454585f)
                check_type(argname="argument s3_location", value=s3_location, expected_type=type_hints["s3_location"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "s3_location": s3_location,
            }

        @builtins.property
        def s3_location(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataSource.S3LocationProperty"]:
            '''An S3 bucket path.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-intermediatestorage.html#cfn-bedrock-datasource-intermediatestorage-s3location
            '''
            result = self._values.get("s3_location")
            assert result is not None, "Required property 's3_location' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataSource.S3LocationProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IntermediateStorageProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.ParsingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "parsing_strategy": "parsingStrategy",
            "bedrock_data_automation_configuration": "bedrockDataAutomationConfiguration",
            "bedrock_foundation_model_configuration": "bedrockFoundationModelConfiguration",
        },
    )
    class ParsingConfigurationProperty:
        def __init__(
            self,
            *,
            parsing_strategy: builtins.str,
            bedrock_data_automation_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.BedrockDataAutomationConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            bedrock_foundation_model_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.BedrockFoundationModelConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Settings for parsing document contents.

            If you exclude this field, the default parser converts the contents of each document into text before splitting it into chunks. Specify the parsing strategy to use in the ``parsingStrategy`` field and include the relevant configuration, or omit it to use the Amazon Bedrock default parser. For more information, see `Parsing options for your data source <https://docs.aws.amazon.com/bedrock/latest/userguide/kb-advanced-parsing.html>`_ .
            .. epigraph::

               If you specify ``BEDROCK_DATA_AUTOMATION`` or ``BEDROCK_FOUNDATION_MODEL`` and it fails to parse a file, the Amazon Bedrock default parser will be used instead.

            :param parsing_strategy: The parsing strategy for the data source.
            :param bedrock_data_automation_configuration: If you specify ``BEDROCK_DATA_AUTOMATION`` as the parsing strategy for ingesting your data source, use this object to modify configurations for using the Amazon Bedrock Data Automation parser.
            :param bedrock_foundation_model_configuration: If you specify ``BEDROCK_FOUNDATION_MODEL`` as the parsing strategy for ingesting your data source, use this object to modify configurations for using a foundation model to parse documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-parsingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                parsing_configuration_property = bedrock.CfnDataSource.ParsingConfigurationProperty(
                    parsing_strategy="parsingStrategy",
                
                    # the properties below are optional
                    bedrock_data_automation_configuration=bedrock.CfnDataSource.BedrockDataAutomationConfigurationProperty(
                        parsing_modality="parsingModality"
                    ),
                    bedrock_foundation_model_configuration=bedrock.CfnDataSource.BedrockFoundationModelConfigurationProperty(
                        model_arn="modelArn",
                
                        # the properties below are optional
                        parsing_modality="parsingModality",
                        parsing_prompt=bedrock.CfnDataSource.ParsingPromptProperty(
                            parsing_prompt_text="parsingPromptText"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__26a14ade018e836fe8686ec2de5e50a05435a92d8d540814f0759c1bb3893ef0)
                check_type(argname="argument parsing_strategy", value=parsing_strategy, expected_type=type_hints["parsing_strategy"])
                check_type(argname="argument bedrock_data_automation_configuration", value=bedrock_data_automation_configuration, expected_type=type_hints["bedrock_data_automation_configuration"])
                check_type(argname="argument bedrock_foundation_model_configuration", value=bedrock_foundation_model_configuration, expected_type=type_hints["bedrock_foundation_model_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "parsing_strategy": parsing_strategy,
            }
            if bedrock_data_automation_configuration is not None:
                self._values["bedrock_data_automation_configuration"] = bedrock_data_automation_configuration
            if bedrock_foundation_model_configuration is not None:
                self._values["bedrock_foundation_model_configuration"] = bedrock_foundation_model_configuration

        @builtins.property
        def parsing_strategy(self) -> builtins.str:
            '''The parsing strategy for the data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-parsingconfiguration.html#cfn-bedrock-datasource-parsingconfiguration-parsingstrategy
            '''
            result = self._values.get("parsing_strategy")
            assert result is not None, "Required property 'parsing_strategy' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def bedrock_data_automation_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.BedrockDataAutomationConfigurationProperty"]]:
            '''If you specify ``BEDROCK_DATA_AUTOMATION`` as the parsing strategy for ingesting your data source, use this object to modify configurations for using the Amazon Bedrock Data Automation parser.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-parsingconfiguration.html#cfn-bedrock-datasource-parsingconfiguration-bedrockdataautomationconfiguration
            '''
            result = self._values.get("bedrock_data_automation_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.BedrockDataAutomationConfigurationProperty"]], result)

        @builtins.property
        def bedrock_foundation_model_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.BedrockFoundationModelConfigurationProperty"]]:
            '''If you specify ``BEDROCK_FOUNDATION_MODEL`` as the parsing strategy for ingesting your data source, use this object to modify configurations for using a foundation model to parse documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-parsingconfiguration.html#cfn-bedrock-datasource-parsingconfiguration-bedrockfoundationmodelconfiguration
            '''
            result = self._values.get("bedrock_foundation_model_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.BedrockFoundationModelConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ParsingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.ParsingPromptProperty",
        jsii_struct_bases=[],
        name_mapping={"parsing_prompt_text": "parsingPromptText"},
    )
    class ParsingPromptProperty:
        def __init__(self, *, parsing_prompt_text: builtins.str) -> None:
            '''Instructions for interpreting the contents of a document.

            :param parsing_prompt_text: Instructions for interpreting the contents of a document.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-parsingprompt.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                parsing_prompt_property = bedrock.CfnDataSource.ParsingPromptProperty(
                    parsing_prompt_text="parsingPromptText"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__57c952f819fbd492c40906223d7646e286faeaf1492ccfe1a050c776f220f8b7)
                check_type(argname="argument parsing_prompt_text", value=parsing_prompt_text, expected_type=type_hints["parsing_prompt_text"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "parsing_prompt_text": parsing_prompt_text,
            }

        @builtins.property
        def parsing_prompt_text(self) -> builtins.str:
            '''Instructions for interpreting the contents of a document.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-parsingprompt.html#cfn-bedrock-datasource-parsingprompt-parsingprompttext
            '''
            result = self._values.get("parsing_prompt_text")
            assert result is not None, "Required property 'parsing_prompt_text' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ParsingPromptProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"filters": "filters"},
    )
    class PatternObjectFilterConfigurationProperty:
        def __init__(
            self,
            *,
            filters: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.PatternObjectFilterProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''The configuration of filtering certain objects or content types of the data source.

            :param filters: The configuration of specific filters applied to your data source content. You can filter out or include certain content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-patternobjectfilterconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                pattern_object_filter_configuration_property = bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                    filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                        object_type="objectType",
                
                        # the properties below are optional
                        exclusion_filters=["exclusionFilters"],
                        inclusion_filters=["inclusionFilters"]
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3ced151210973a4135e752b7dc574bfd684559819a7a07720ae710370867eba6)
                check_type(argname="argument filters", value=filters, expected_type=type_hints["filters"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "filters": filters,
            }

        @builtins.property
        def filters(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDataSource.PatternObjectFilterProperty"]]]:
            '''The configuration of specific filters applied to your data source content.

            You can filter out or include certain content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-patternobjectfilterconfiguration.html#cfn-bedrock-datasource-patternobjectfilterconfiguration-filters
            '''
            result = self._values.get("filters")
            assert result is not None, "Required property 'filters' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDataSource.PatternObjectFilterProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PatternObjectFilterConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.PatternObjectFilterProperty",
        jsii_struct_bases=[],
        name_mapping={
            "object_type": "objectType",
            "exclusion_filters": "exclusionFilters",
            "inclusion_filters": "inclusionFilters",
        },
    )
    class PatternObjectFilterProperty:
        def __init__(
            self,
            *,
            object_type: builtins.str,
            exclusion_filters: typing.Optional[typing.Sequence[builtins.str]] = None,
            inclusion_filters: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The specific filters applied to your data source content.

            You can filter out or include certain content.

            :param object_type: The supported object type or content type of the data source.
            :param exclusion_filters: A list of one or more exclusion regular expression patterns to exclude certain object types that adhere to the pattern. If you specify an inclusion and exclusion filter/pattern and both match a document, the exclusion filter takes precedence and the document isn’t crawled.
            :param inclusion_filters: A list of one or more inclusion regular expression patterns to include certain object types that adhere to the pattern. If you specify an inclusion and exclusion filter/pattern and both match a document, the exclusion filter takes precedence and the document isn’t crawled.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-patternobjectfilter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                pattern_object_filter_property = bedrock.CfnDataSource.PatternObjectFilterProperty(
                    object_type="objectType",
                
                    # the properties below are optional
                    exclusion_filters=["exclusionFilters"],
                    inclusion_filters=["inclusionFilters"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7dd7a9d6c2e0d5d05d02a68d2dea721b600e4ddae8bf5dcf24f149b9e9319bd5)
                check_type(argname="argument object_type", value=object_type, expected_type=type_hints["object_type"])
                check_type(argname="argument exclusion_filters", value=exclusion_filters, expected_type=type_hints["exclusion_filters"])
                check_type(argname="argument inclusion_filters", value=inclusion_filters, expected_type=type_hints["inclusion_filters"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "object_type": object_type,
            }
            if exclusion_filters is not None:
                self._values["exclusion_filters"] = exclusion_filters
            if inclusion_filters is not None:
                self._values["inclusion_filters"] = inclusion_filters

        @builtins.property
        def object_type(self) -> builtins.str:
            '''The supported object type or content type of the data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-patternobjectfilter.html#cfn-bedrock-datasource-patternobjectfilter-objecttype
            '''
            result = self._values.get("object_type")
            assert result is not None, "Required property 'object_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def exclusion_filters(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of one or more exclusion regular expression patterns to exclude certain object types that adhere to the pattern.

            If you specify an inclusion and exclusion filter/pattern and both match a document, the exclusion filter takes precedence and the document isn’t crawled.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-patternobjectfilter.html#cfn-bedrock-datasource-patternobjectfilter-exclusionfilters
            '''
            result = self._values.get("exclusion_filters")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def inclusion_filters(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of one or more inclusion regular expression patterns to include certain object types that adhere to the pattern.

            If you specify an inclusion and exclusion filter/pattern and both match a document, the exclusion filter takes precedence and the document isn’t crawled.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-patternobjectfilter.html#cfn-bedrock-datasource-patternobjectfilter-inclusionfilters
            '''
            result = self._values.get("inclusion_filters")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PatternObjectFilterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.S3DataSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "bucket_arn": "bucketArn",
            "bucket_owner_account_id": "bucketOwnerAccountId",
            "inclusion_prefixes": "inclusionPrefixes",
        },
    )
    class S3DataSourceConfigurationProperty:
        def __init__(
            self,
            *,
            bucket_arn: builtins.str,
            bucket_owner_account_id: typing.Optional[builtins.str] = None,
            inclusion_prefixes: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The configuration information to connect to Amazon S3 as your data source.

            :param bucket_arn: The Amazon Resource Name (ARN) of the S3 bucket that contains your data.
            :param bucket_owner_account_id: The account ID for the owner of the S3 bucket.
            :param inclusion_prefixes: A list of S3 prefixes to include certain files or content. For more information, see `Organizing objects using prefixes <https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-s3datasourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                s3_data_source_configuration_property = bedrock.CfnDataSource.S3DataSourceConfigurationProperty(
                    bucket_arn="bucketArn",
                
                    # the properties below are optional
                    bucket_owner_account_id="bucketOwnerAccountId",
                    inclusion_prefixes=["inclusionPrefixes"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6b4929296343a5ea55bce5d28d11b34fa10885df35596a08102658f8bcbc8c5b)
                check_type(argname="argument bucket_arn", value=bucket_arn, expected_type=type_hints["bucket_arn"])
                check_type(argname="argument bucket_owner_account_id", value=bucket_owner_account_id, expected_type=type_hints["bucket_owner_account_id"])
                check_type(argname="argument inclusion_prefixes", value=inclusion_prefixes, expected_type=type_hints["inclusion_prefixes"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "bucket_arn": bucket_arn,
            }
            if bucket_owner_account_id is not None:
                self._values["bucket_owner_account_id"] = bucket_owner_account_id
            if inclusion_prefixes is not None:
                self._values["inclusion_prefixes"] = inclusion_prefixes

        @builtins.property
        def bucket_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the S3 bucket that contains your data.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-s3datasourceconfiguration.html#cfn-bedrock-datasource-s3datasourceconfiguration-bucketarn
            '''
            result = self._values.get("bucket_arn")
            assert result is not None, "Required property 'bucket_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def bucket_owner_account_id(self) -> typing.Optional[builtins.str]:
            '''The account ID for the owner of the S3 bucket.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-s3datasourceconfiguration.html#cfn-bedrock-datasource-s3datasourceconfiguration-bucketowneraccountid
            '''
            result = self._values.get("bucket_owner_account_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def inclusion_prefixes(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of S3 prefixes to include certain files or content.

            For more information, see `Organizing objects using prefixes <https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-s3datasourceconfiguration.html#cfn-bedrock-datasource-s3datasourceconfiguration-inclusionprefixes
            '''
            result = self._values.get("inclusion_prefixes")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "S3DataSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.S3LocationProperty",
        jsii_struct_bases=[],
        name_mapping={"uri": "uri"},
    )
    class S3LocationProperty:
        def __init__(self, *, uri: builtins.str) -> None:
            '''A storage location in an S3 bucket.

            :param uri: An object URI starting with ``s3://`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-s3location.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                s3_location_property = bedrock.CfnDataSource.S3LocationProperty(
                    uri="uri"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d638c12879ec4e653848ff096c94119612fc29f6181050c0a7980f3714ced173)
                check_type(argname="argument uri", value=uri, expected_type=type_hints["uri"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "uri": uri,
            }

        @builtins.property
        def uri(self) -> builtins.str:
            '''An object URI starting with ``s3://`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-s3location.html#cfn-bedrock-datasource-s3location-uri
            '''
            result = self._values.get("uri")
            assert result is not None, "Required property 'uri' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "S3LocationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.SalesforceCrawlerConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"filter_configuration": "filterConfiguration"},
    )
    class SalesforceCrawlerConfigurationProperty:
        def __init__(
            self,
            *,
            filter_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.CrawlFilterConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration of the Salesforce content.

            For example, configuring specific types of Salesforce content.

            :param filter_configuration: The configuration of filtering the Salesforce content. For example, configuring regular expression patterns to include or exclude certain content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-salesforcecrawlerconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                salesforce_crawler_configuration_property = bedrock.CfnDataSource.SalesforceCrawlerConfigurationProperty(
                    filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                        type="type",
                
                        # the properties below are optional
                        pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                            filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                object_type="objectType",
                
                                # the properties below are optional
                                exclusion_filters=["exclusionFilters"],
                                inclusion_filters=["inclusionFilters"]
                            )]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dcf2c72ca7f7035009e0a5f42f7687d9e34922f1d08c1037a18768fdbbcd714b)
                check_type(argname="argument filter_configuration", value=filter_configuration, expected_type=type_hints["filter_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if filter_configuration is not None:
                self._values["filter_configuration"] = filter_configuration

        @builtins.property
        def filter_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.CrawlFilterConfigurationProperty"]]:
            '''The configuration of filtering the Salesforce content.

            For example, configuring regular expression patterns to include or exclude certain content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-salesforcecrawlerconfiguration.html#cfn-bedrock-datasource-salesforcecrawlerconfiguration-filterconfiguration
            '''
            result = self._values.get("filter_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.CrawlFilterConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SalesforceCrawlerConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.SalesforceDataSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "source_configuration": "sourceConfiguration",
            "crawler_configuration": "crawlerConfiguration",
        },
    )
    class SalesforceDataSourceConfigurationProperty:
        def __init__(
            self,
            *,
            source_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.SalesforceSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
            crawler_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.SalesforceCrawlerConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration information to connect to Salesforce as your data source.

            :param source_configuration: The endpoint information to connect to your Salesforce data source.
            :param crawler_configuration: The configuration of the Salesforce content. For example, configuring specific types of Salesforce content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-salesforcedatasourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                salesforce_data_source_configuration_property = bedrock.CfnDataSource.SalesforceDataSourceConfigurationProperty(
                    source_configuration=bedrock.CfnDataSource.SalesforceSourceConfigurationProperty(
                        auth_type="authType",
                        credentials_secret_arn="credentialsSecretArn",
                        host_url="hostUrl"
                    ),
                
                    # the properties below are optional
                    crawler_configuration=bedrock.CfnDataSource.SalesforceCrawlerConfigurationProperty(
                        filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                            type="type",
                
                            # the properties below are optional
                            pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                    object_type="objectType",
                
                                    # the properties below are optional
                                    exclusion_filters=["exclusionFilters"],
                                    inclusion_filters=["inclusionFilters"]
                                )]
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__19e91cc5df4de808fbd82c02038242b17f720e32ff1b1c457240b738983c4e56)
                check_type(argname="argument source_configuration", value=source_configuration, expected_type=type_hints["source_configuration"])
                check_type(argname="argument crawler_configuration", value=crawler_configuration, expected_type=type_hints["crawler_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "source_configuration": source_configuration,
            }
            if crawler_configuration is not None:
                self._values["crawler_configuration"] = crawler_configuration

        @builtins.property
        def source_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataSource.SalesforceSourceConfigurationProperty"]:
            '''The endpoint information to connect to your Salesforce data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-salesforcedatasourceconfiguration.html#cfn-bedrock-datasource-salesforcedatasourceconfiguration-sourceconfiguration
            '''
            result = self._values.get("source_configuration")
            assert result is not None, "Required property 'source_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataSource.SalesforceSourceConfigurationProperty"], result)

        @builtins.property
        def crawler_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SalesforceCrawlerConfigurationProperty"]]:
            '''The configuration of the Salesforce content.

            For example, configuring specific types of Salesforce content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-salesforcedatasourceconfiguration.html#cfn-bedrock-datasource-salesforcedatasourceconfiguration-crawlerconfiguration
            '''
            result = self._values.get("crawler_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SalesforceCrawlerConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SalesforceDataSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.SalesforceSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auth_type": "authType",
            "credentials_secret_arn": "credentialsSecretArn",
            "host_url": "hostUrl",
        },
    )
    class SalesforceSourceConfigurationProperty:
        def __init__(
            self,
            *,
            auth_type: builtins.str,
            credentials_secret_arn: builtins.str,
            host_url: builtins.str,
        ) -> None:
            '''The endpoint information to connect to your Salesforce data source.

            :param auth_type: The supported authentication type to authenticate and connect to your Salesforce instance.
            :param credentials_secret_arn: The Amazon Resource Name of an AWS Secrets Manager secret that stores your authentication credentials for your Salesforce instance URL. For more information on the key-value pairs that must be included in your secret, depending on your authentication type, see `Salesforce connection configuration <https://docs.aws.amazon.com/bedrock/latest/userguide/salesforce-data-source-connector.html#configuration-salesforce-connector>`_ .
            :param host_url: The Salesforce host URL or instance URL.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-salesforcesourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                salesforce_source_configuration_property = bedrock.CfnDataSource.SalesforceSourceConfigurationProperty(
                    auth_type="authType",
                    credentials_secret_arn="credentialsSecretArn",
                    host_url="hostUrl"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__10d168ba87dbfa2820a4a25447b46f7077984592e3b4d5a50f59c6f822487578)
                check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
                check_type(argname="argument credentials_secret_arn", value=credentials_secret_arn, expected_type=type_hints["credentials_secret_arn"])
                check_type(argname="argument host_url", value=host_url, expected_type=type_hints["host_url"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "auth_type": auth_type,
                "credentials_secret_arn": credentials_secret_arn,
                "host_url": host_url,
            }

        @builtins.property
        def auth_type(self) -> builtins.str:
            '''The supported authentication type to authenticate and connect to your Salesforce instance.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-salesforcesourceconfiguration.html#cfn-bedrock-datasource-salesforcesourceconfiguration-authtype
            '''
            result = self._values.get("auth_type")
            assert result is not None, "Required property 'auth_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def credentials_secret_arn(self) -> builtins.str:
            '''The Amazon Resource Name of an AWS Secrets Manager secret that stores your authentication credentials for your Salesforce instance URL.

            For more information on the key-value pairs that must be included in your secret, depending on your authentication type, see `Salesforce connection configuration <https://docs.aws.amazon.com/bedrock/latest/userguide/salesforce-data-source-connector.html#configuration-salesforce-connector>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-salesforcesourceconfiguration.html#cfn-bedrock-datasource-salesforcesourceconfiguration-credentialssecretarn
            '''
            result = self._values.get("credentials_secret_arn")
            assert result is not None, "Required property 'credentials_secret_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def host_url(self) -> builtins.str:
            '''The Salesforce host URL or instance URL.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-salesforcesourceconfiguration.html#cfn-bedrock-datasource-salesforcesourceconfiguration-hosturl
            '''
            result = self._values.get("host_url")
            assert result is not None, "Required property 'host_url' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SalesforceSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.SeedUrlProperty",
        jsii_struct_bases=[],
        name_mapping={"url": "url"},
    )
    class SeedUrlProperty:
        def __init__(self, *, url: builtins.str) -> None:
            '''The seed or starting point URL.

            You should be authorized to crawl the URL.

            :param url: A seed or starting point URL.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-seedurl.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                seed_url_property = bedrock.CfnDataSource.SeedUrlProperty(
                    url="url"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cf8c0f6926d3f02906dd411ea53c5df121a2ab8fd6cb1cc29270d45f47907263)
                check_type(argname="argument url", value=url, expected_type=type_hints["url"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "url": url,
            }

        @builtins.property
        def url(self) -> builtins.str:
            '''A seed or starting point URL.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-seedurl.html#cfn-bedrock-datasource-seedurl-url
            '''
            result = self._values.get("url")
            assert result is not None, "Required property 'url' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SeedUrlProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.SemanticChunkingConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "breakpoint_percentile_threshold": "breakpointPercentileThreshold",
            "buffer_size": "bufferSize",
            "max_tokens": "maxTokens",
        },
    )
    class SemanticChunkingConfigurationProperty:
        def __init__(
            self,
            *,
            breakpoint_percentile_threshold: jsii.Number,
            buffer_size: jsii.Number,
            max_tokens: jsii.Number,
        ) -> None:
            '''Settings for semantic document chunking for a data source.

            Semantic chunking splits a document into into smaller documents based on groups of similar content derived from the text with natural language processing.

            With semantic chunking, each sentence is compared to the next to determine how similar they are. You specify a threshold in the form of a percentile, where adjacent sentences that are less similar than that percentage of sentence pairs are divided into separate chunks. For example, if you set the threshold to 90, then the 10 percent of sentence pairs that are least similar are split. So if you have 101 sentences, 100 sentence pairs are compared, and the 10 with the least similarity are split, creating 11 chunks. These chunks are further split if they exceed the max token size.

            You must also specify a buffer size, which determines whether sentences are compared in isolation, or within a moving context window that includes the previous and following sentence. For example, if you set the buffer size to ``1`` , the embedding for sentence 10 is derived from sentences 9, 10, and 11 combined.

            :param breakpoint_percentile_threshold: The dissimilarity threshold for splitting chunks.
            :param buffer_size: The buffer size.
            :param max_tokens: The maximum number of tokens that a chunk can contain.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-semanticchunkingconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                semantic_chunking_configuration_property = bedrock.CfnDataSource.SemanticChunkingConfigurationProperty(
                    breakpoint_percentile_threshold=123,
                    buffer_size=123,
                    max_tokens=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5fbe19246811c798768255282a758b9afc14da2c6d7c0419553a9ac98be7f99e)
                check_type(argname="argument breakpoint_percentile_threshold", value=breakpoint_percentile_threshold, expected_type=type_hints["breakpoint_percentile_threshold"])
                check_type(argname="argument buffer_size", value=buffer_size, expected_type=type_hints["buffer_size"])
                check_type(argname="argument max_tokens", value=max_tokens, expected_type=type_hints["max_tokens"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "breakpoint_percentile_threshold": breakpoint_percentile_threshold,
                "buffer_size": buffer_size,
                "max_tokens": max_tokens,
            }

        @builtins.property
        def breakpoint_percentile_threshold(self) -> jsii.Number:
            '''The dissimilarity threshold for splitting chunks.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-semanticchunkingconfiguration.html#cfn-bedrock-datasource-semanticchunkingconfiguration-breakpointpercentilethreshold
            '''
            result = self._values.get("breakpoint_percentile_threshold")
            assert result is not None, "Required property 'breakpoint_percentile_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def buffer_size(self) -> jsii.Number:
            '''The buffer size.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-semanticchunkingconfiguration.html#cfn-bedrock-datasource-semanticchunkingconfiguration-buffersize
            '''
            result = self._values.get("buffer_size")
            assert result is not None, "Required property 'buffer_size' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def max_tokens(self) -> jsii.Number:
            '''The maximum number of tokens that a chunk can contain.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-semanticchunkingconfiguration.html#cfn-bedrock-datasource-semanticchunkingconfiguration-maxtokens
            '''
            result = self._values.get("max_tokens")
            assert result is not None, "Required property 'max_tokens' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SemanticChunkingConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.ServerSideEncryptionConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"kms_key_arn": "kmsKeyArn"},
    )
    class ServerSideEncryptionConfigurationProperty:
        def __init__(
            self,
            *,
            kms_key_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains the configuration for server-side encryption.

            :param kms_key_arn: The Amazon Resource Name (ARN) of the AWS KMS key used to encrypt the resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-serversideencryptionconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                server_side_encryption_configuration_property = bedrock.CfnDataSource.ServerSideEncryptionConfigurationProperty(
                    kms_key_arn="kmsKeyArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e7bf662e3d628b0671b547dd26face3c322a93684b4b5578ec4cda1099b30e6a)
                check_type(argname="argument kms_key_arn", value=kms_key_arn, expected_type=type_hints["kms_key_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if kms_key_arn is not None:
                self._values["kms_key_arn"] = kms_key_arn

        @builtins.property
        def kms_key_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the AWS KMS key used to encrypt the resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-serversideencryptionconfiguration.html#cfn-bedrock-datasource-serversideencryptionconfiguration-kmskeyarn
            '''
            result = self._values.get("kms_key_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServerSideEncryptionConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.SharePointCrawlerConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"filter_configuration": "filterConfiguration"},
    )
    class SharePointCrawlerConfigurationProperty:
        def __init__(
            self,
            *,
            filter_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.CrawlFilterConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration of the SharePoint content.

            For example, configuring specific types of SharePoint content.

            :param filter_configuration: The configuration of filtering the SharePoint content. For example, configuring regular expression patterns to include or exclude certain content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointcrawlerconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                share_point_crawler_configuration_property = bedrock.CfnDataSource.SharePointCrawlerConfigurationProperty(
                    filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                        type="type",
                
                        # the properties below are optional
                        pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                            filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                object_type="objectType",
                
                                # the properties below are optional
                                exclusion_filters=["exclusionFilters"],
                                inclusion_filters=["inclusionFilters"]
                            )]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__738da86cc405fddc64a59295a3d62b697bd1f304c3a1a23dd8387bf4807e5bfc)
                check_type(argname="argument filter_configuration", value=filter_configuration, expected_type=type_hints["filter_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if filter_configuration is not None:
                self._values["filter_configuration"] = filter_configuration

        @builtins.property
        def filter_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.CrawlFilterConfigurationProperty"]]:
            '''The configuration of filtering the SharePoint content.

            For example, configuring regular expression patterns to include or exclude certain content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointcrawlerconfiguration.html#cfn-bedrock-datasource-sharepointcrawlerconfiguration-filterconfiguration
            '''
            result = self._values.get("filter_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.CrawlFilterConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SharePointCrawlerConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.SharePointDataSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "source_configuration": "sourceConfiguration",
            "crawler_configuration": "crawlerConfiguration",
        },
    )
    class SharePointDataSourceConfigurationProperty:
        def __init__(
            self,
            *,
            source_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.SharePointSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
            crawler_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.SharePointCrawlerConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration information to connect to SharePoint as your data source.

            :param source_configuration: The endpoint information to connect to your SharePoint data source.
            :param crawler_configuration: The configuration of the SharePoint content. For example, configuring specific types of SharePoint content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointdatasourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                share_point_data_source_configuration_property = bedrock.CfnDataSource.SharePointDataSourceConfigurationProperty(
                    source_configuration=bedrock.CfnDataSource.SharePointSourceConfigurationProperty(
                        auth_type="authType",
                        credentials_secret_arn="credentialsSecretArn",
                        domain="domain",
                        host_type="hostType",
                        site_urls=["siteUrls"],
                
                        # the properties below are optional
                        tenant_id="tenantId"
                    ),
                
                    # the properties below are optional
                    crawler_configuration=bedrock.CfnDataSource.SharePointCrawlerConfigurationProperty(
                        filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                            type="type",
                
                            # the properties below are optional
                            pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                    object_type="objectType",
                
                                    # the properties below are optional
                                    exclusion_filters=["exclusionFilters"],
                                    inclusion_filters=["inclusionFilters"]
                                )]
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0cd4cda882e18a91758d0c55ba62239b7f2a738f332f000e150340d38474c549)
                check_type(argname="argument source_configuration", value=source_configuration, expected_type=type_hints["source_configuration"])
                check_type(argname="argument crawler_configuration", value=crawler_configuration, expected_type=type_hints["crawler_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "source_configuration": source_configuration,
            }
            if crawler_configuration is not None:
                self._values["crawler_configuration"] = crawler_configuration

        @builtins.property
        def source_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataSource.SharePointSourceConfigurationProperty"]:
            '''The endpoint information to connect to your SharePoint data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointdatasourceconfiguration.html#cfn-bedrock-datasource-sharepointdatasourceconfiguration-sourceconfiguration
            '''
            result = self._values.get("source_configuration")
            assert result is not None, "Required property 'source_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataSource.SharePointSourceConfigurationProperty"], result)

        @builtins.property
        def crawler_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SharePointCrawlerConfigurationProperty"]]:
            '''The configuration of the SharePoint content.

            For example, configuring specific types of SharePoint content.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointdatasourceconfiguration.html#cfn-bedrock-datasource-sharepointdatasourceconfiguration-crawlerconfiguration
            '''
            result = self._values.get("crawler_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SharePointCrawlerConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SharePointDataSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.SharePointSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auth_type": "authType",
            "credentials_secret_arn": "credentialsSecretArn",
            "domain": "domain",
            "host_type": "hostType",
            "site_urls": "siteUrls",
            "tenant_id": "tenantId",
        },
    )
    class SharePointSourceConfigurationProperty:
        def __init__(
            self,
            *,
            auth_type: builtins.str,
            credentials_secret_arn: builtins.str,
            domain: builtins.str,
            host_type: builtins.str,
            site_urls: typing.Sequence[builtins.str],
            tenant_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The endpoint information to connect to your SharePoint data source.

            :param auth_type: The supported authentication type to authenticate and connect to your SharePoint site/sites.
            :param credentials_secret_arn: The Amazon Resource Name of an AWS Secrets Manager secret that stores your authentication credentials for your SharePoint site/sites. For more information on the key-value pairs that must be included in your secret, depending on your authentication type, see `SharePoint connection configuration <https://docs.aws.amazon.com/bedrock/latest/userguide/sharepoint-data-source-connector.html#configuration-sharepoint-connector>`_ .
            :param domain: The domain of your SharePoint instance or site URL/URLs.
            :param host_type: The supported host type, whether online/cloud or server/on-premises.
            :param site_urls: A list of one or more SharePoint site URLs.
            :param tenant_id: The identifier of your Microsoft 365 tenant.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointsourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                share_point_source_configuration_property = bedrock.CfnDataSource.SharePointSourceConfigurationProperty(
                    auth_type="authType",
                    credentials_secret_arn="credentialsSecretArn",
                    domain="domain",
                    host_type="hostType",
                    site_urls=["siteUrls"],
                
                    # the properties below are optional
                    tenant_id="tenantId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__48994fdcb11091f4caa0fb5753c7af914ae9816931a2bef7e3f3324c63e6e07c)
                check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
                check_type(argname="argument credentials_secret_arn", value=credentials_secret_arn, expected_type=type_hints["credentials_secret_arn"])
                check_type(argname="argument domain", value=domain, expected_type=type_hints["domain"])
                check_type(argname="argument host_type", value=host_type, expected_type=type_hints["host_type"])
                check_type(argname="argument site_urls", value=site_urls, expected_type=type_hints["site_urls"])
                check_type(argname="argument tenant_id", value=tenant_id, expected_type=type_hints["tenant_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "auth_type": auth_type,
                "credentials_secret_arn": credentials_secret_arn,
                "domain": domain,
                "host_type": host_type,
                "site_urls": site_urls,
            }
            if tenant_id is not None:
                self._values["tenant_id"] = tenant_id

        @builtins.property
        def auth_type(self) -> builtins.str:
            '''The supported authentication type to authenticate and connect to your SharePoint site/sites.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointsourceconfiguration.html#cfn-bedrock-datasource-sharepointsourceconfiguration-authtype
            '''
            result = self._values.get("auth_type")
            assert result is not None, "Required property 'auth_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def credentials_secret_arn(self) -> builtins.str:
            '''The Amazon Resource Name of an AWS Secrets Manager secret that stores your authentication credentials for your SharePoint site/sites.

            For more information on the key-value pairs that must be included in your secret, depending on your authentication type, see `SharePoint connection configuration <https://docs.aws.amazon.com/bedrock/latest/userguide/sharepoint-data-source-connector.html#configuration-sharepoint-connector>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointsourceconfiguration.html#cfn-bedrock-datasource-sharepointsourceconfiguration-credentialssecretarn
            '''
            result = self._values.get("credentials_secret_arn")
            assert result is not None, "Required property 'credentials_secret_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def domain(self) -> builtins.str:
            '''The domain of your SharePoint instance or site URL/URLs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointsourceconfiguration.html#cfn-bedrock-datasource-sharepointsourceconfiguration-domain
            '''
            result = self._values.get("domain")
            assert result is not None, "Required property 'domain' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def host_type(self) -> builtins.str:
            '''The supported host type, whether online/cloud or server/on-premises.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointsourceconfiguration.html#cfn-bedrock-datasource-sharepointsourceconfiguration-hosttype
            '''
            result = self._values.get("host_type")
            assert result is not None, "Required property 'host_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def site_urls(self) -> typing.List[builtins.str]:
            '''A list of one or more SharePoint site URLs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointsourceconfiguration.html#cfn-bedrock-datasource-sharepointsourceconfiguration-siteurls
            '''
            result = self._values.get("site_urls")
            assert result is not None, "Required property 'site_urls' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def tenant_id(self) -> typing.Optional[builtins.str]:
            '''The identifier of your Microsoft 365 tenant.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-sharepointsourceconfiguration.html#cfn-bedrock-datasource-sharepointsourceconfiguration-tenantid
            '''
            result = self._values.get("tenant_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SharePointSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.TransformationFunctionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "transformation_lambda_configuration": "transformationLambdaConfiguration",
        },
    )
    class TransformationFunctionProperty:
        def __init__(
            self,
            *,
            transformation_lambda_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.TransformationLambdaConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''A Lambda function that processes documents.

            :param transformation_lambda_configuration: The Lambda function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-transformationfunction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                transformation_function_property = bedrock.CfnDataSource.TransformationFunctionProperty(
                    transformation_lambda_configuration=bedrock.CfnDataSource.TransformationLambdaConfigurationProperty(
                        lambda_arn="lambdaArn"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b12219aac867927e5b39fb3f23bae62881fb2b3683eda8f7c5b4f84aa1b85b35)
                check_type(argname="argument transformation_lambda_configuration", value=transformation_lambda_configuration, expected_type=type_hints["transformation_lambda_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "transformation_lambda_configuration": transformation_lambda_configuration,
            }

        @builtins.property
        def transformation_lambda_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataSource.TransformationLambdaConfigurationProperty"]:
            '''The Lambda function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-transformationfunction.html#cfn-bedrock-datasource-transformationfunction-transformationlambdaconfiguration
            '''
            result = self._values.get("transformation_lambda_configuration")
            assert result is not None, "Required property 'transformation_lambda_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataSource.TransformationLambdaConfigurationProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TransformationFunctionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.TransformationLambdaConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"lambda_arn": "lambdaArn"},
    )
    class TransformationLambdaConfigurationProperty:
        def __init__(self, *, lambda_arn: builtins.str) -> None:
            '''A Lambda function that processes documents.

            :param lambda_arn: The function's ARN identifier.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-transformationlambdaconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                transformation_lambda_configuration_property = bedrock.CfnDataSource.TransformationLambdaConfigurationProperty(
                    lambda_arn="lambdaArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__be41a2fbc228bb0e08139d976da72eebeefefc51f1d4a9d0be138450c213a7b5)
                check_type(argname="argument lambda_arn", value=lambda_arn, expected_type=type_hints["lambda_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "lambda_arn": lambda_arn,
            }

        @builtins.property
        def lambda_arn(self) -> builtins.str:
            '''The function's ARN identifier.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-transformationlambdaconfiguration.html#cfn-bedrock-datasource-transformationlambdaconfiguration-lambdaarn
            '''
            result = self._values.get("lambda_arn")
            assert result is not None, "Required property 'lambda_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TransformationLambdaConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.TransformationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "step_to_apply": "stepToApply",
            "transformation_function": "transformationFunction",
        },
    )
    class TransformationProperty:
        def __init__(
            self,
            *,
            step_to_apply: builtins.str,
            transformation_function: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.TransformationFunctionProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''A custom processing step for documents moving through a data source ingestion pipeline.

            To process documents after they have been converted into chunks, set the step to apply to ``POST_CHUNKING`` .

            :param step_to_apply: When the service applies the transformation.
            :param transformation_function: A Lambda function that processes documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-transformation.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                transformation_property = bedrock.CfnDataSource.TransformationProperty(
                    step_to_apply="stepToApply",
                    transformation_function=bedrock.CfnDataSource.TransformationFunctionProperty(
                        transformation_lambda_configuration=bedrock.CfnDataSource.TransformationLambdaConfigurationProperty(
                            lambda_arn="lambdaArn"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b6f1d625210f565b8ded52de7c03c9e5f0e0c6b790365d2f0f995f9dd39c0ae0)
                check_type(argname="argument step_to_apply", value=step_to_apply, expected_type=type_hints["step_to_apply"])
                check_type(argname="argument transformation_function", value=transformation_function, expected_type=type_hints["transformation_function"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "step_to_apply": step_to_apply,
                "transformation_function": transformation_function,
            }

        @builtins.property
        def step_to_apply(self) -> builtins.str:
            '''When the service applies the transformation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-transformation.html#cfn-bedrock-datasource-transformation-steptoapply
            '''
            result = self._values.get("step_to_apply")
            assert result is not None, "Required property 'step_to_apply' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def transformation_function(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataSource.TransformationFunctionProperty"]:
            '''A Lambda function that processes documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-transformation.html#cfn-bedrock-datasource-transformation-transformationfunction
            '''
            result = self._values.get("transformation_function")
            assert result is not None, "Required property 'transformation_function' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataSource.TransformationFunctionProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TransformationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.UrlConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"seed_urls": "seedUrls"},
    )
    class UrlConfigurationProperty:
        def __init__(
            self,
            *,
            seed_urls: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.SeedUrlProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''The configuration of web URLs that you want to crawl.

            You should be authorized to crawl the URLs.

            :param seed_urls: One or more seed or starting point URLs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-urlconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                url_configuration_property = bedrock.CfnDataSource.UrlConfigurationProperty(
                    seed_urls=[bedrock.CfnDataSource.SeedUrlProperty(
                        url="url"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__33e4e6226e3e29b9d96cc7d5d58f6d2d4062908e865a2e74a0c3a6b363de5062)
                check_type(argname="argument seed_urls", value=seed_urls, expected_type=type_hints["seed_urls"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "seed_urls": seed_urls,
            }

        @builtins.property
        def seed_urls(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SeedUrlProperty"]]]:
            '''One or more seed or starting point URLs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-urlconfiguration.html#cfn-bedrock-datasource-urlconfiguration-seedurls
            '''
            result = self._values.get("seed_urls")
            assert result is not None, "Required property 'seed_urls' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDataSource.SeedUrlProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "UrlConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.VectorIngestionConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "chunking_configuration": "chunkingConfiguration",
            "context_enrichment_configuration": "contextEnrichmentConfiguration",
            "custom_transformation_configuration": "customTransformationConfiguration",
            "parsing_configuration": "parsingConfiguration",
        },
    )
    class VectorIngestionConfigurationProperty:
        def __init__(
            self,
            *,
            chunking_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.ChunkingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            context_enrichment_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.ContextEnrichmentConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            custom_transformation_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.CustomTransformationConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            parsing_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.ParsingConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains details about how to ingest the documents in a data source.

            :param chunking_configuration: Details about how to chunk the documents in the data source. A *chunk* refers to an excerpt from a data source that is returned when the knowledge base that it belongs to is queried.
            :param context_enrichment_configuration: The context enrichment configuration used for ingestion of the data into the vector store.
            :param custom_transformation_configuration: A custom document transformer for parsed data source documents.
            :param parsing_configuration: Configurations for a parser to use for parsing documents in your data source. If you exclude this field, the default parser will be used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-vectoringestionconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                vector_ingestion_configuration_property = bedrock.CfnDataSource.VectorIngestionConfigurationProperty(
                    chunking_configuration=bedrock.CfnDataSource.ChunkingConfigurationProperty(
                        chunking_strategy="chunkingStrategy",
                
                        # the properties below are optional
                        fixed_size_chunking_configuration=bedrock.CfnDataSource.FixedSizeChunkingConfigurationProperty(
                            max_tokens=123,
                            overlap_percentage=123
                        ),
                        hierarchical_chunking_configuration=bedrock.CfnDataSource.HierarchicalChunkingConfigurationProperty(
                            level_configurations=[bedrock.CfnDataSource.HierarchicalChunkingLevelConfigurationProperty(
                                max_tokens=123
                            )],
                            overlap_tokens=123
                        ),
                        semantic_chunking_configuration=bedrock.CfnDataSource.SemanticChunkingConfigurationProperty(
                            breakpoint_percentile_threshold=123,
                            buffer_size=123,
                            max_tokens=123
                        )
                    ),
                    context_enrichment_configuration=bedrock.CfnDataSource.ContextEnrichmentConfigurationProperty(
                        type="type",
                
                        # the properties below are optional
                        bedrock_foundation_model_configuration=bedrock.CfnDataSource.BedrockFoundationModelContextEnrichmentConfigurationProperty(
                            enrichment_strategy_configuration=bedrock.CfnDataSource.EnrichmentStrategyConfigurationProperty(
                                method="method"
                            ),
                            model_arn="modelArn"
                        )
                    ),
                    custom_transformation_configuration=bedrock.CfnDataSource.CustomTransformationConfigurationProperty(
                        intermediate_storage=bedrock.CfnDataSource.IntermediateStorageProperty(
                            s3_location=bedrock.CfnDataSource.S3LocationProperty(
                                uri="uri"
                            )
                        ),
                        transformations=[bedrock.CfnDataSource.TransformationProperty(
                            step_to_apply="stepToApply",
                            transformation_function=bedrock.CfnDataSource.TransformationFunctionProperty(
                                transformation_lambda_configuration=bedrock.CfnDataSource.TransformationLambdaConfigurationProperty(
                                    lambda_arn="lambdaArn"
                                )
                            )
                        )]
                    ),
                    parsing_configuration=bedrock.CfnDataSource.ParsingConfigurationProperty(
                        parsing_strategy="parsingStrategy",
                
                        # the properties below are optional
                        bedrock_data_automation_configuration=bedrock.CfnDataSource.BedrockDataAutomationConfigurationProperty(
                            parsing_modality="parsingModality"
                        ),
                        bedrock_foundation_model_configuration=bedrock.CfnDataSource.BedrockFoundationModelConfigurationProperty(
                            model_arn="modelArn",
                
                            # the properties below are optional
                            parsing_modality="parsingModality",
                            parsing_prompt=bedrock.CfnDataSource.ParsingPromptProperty(
                                parsing_prompt_text="parsingPromptText"
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__37255db3e7cacfdeadc2d28bfe7938fd13ab4a2cd72190024a531eddf1fd9ec0)
                check_type(argname="argument chunking_configuration", value=chunking_configuration, expected_type=type_hints["chunking_configuration"])
                check_type(argname="argument context_enrichment_configuration", value=context_enrichment_configuration, expected_type=type_hints["context_enrichment_configuration"])
                check_type(argname="argument custom_transformation_configuration", value=custom_transformation_configuration, expected_type=type_hints["custom_transformation_configuration"])
                check_type(argname="argument parsing_configuration", value=parsing_configuration, expected_type=type_hints["parsing_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if chunking_configuration is not None:
                self._values["chunking_configuration"] = chunking_configuration
            if context_enrichment_configuration is not None:
                self._values["context_enrichment_configuration"] = context_enrichment_configuration
            if custom_transformation_configuration is not None:
                self._values["custom_transformation_configuration"] = custom_transformation_configuration
            if parsing_configuration is not None:
                self._values["parsing_configuration"] = parsing_configuration

        @builtins.property
        def chunking_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ChunkingConfigurationProperty"]]:
            '''Details about how to chunk the documents in the data source.

            A *chunk* refers to an excerpt from a data source that is returned when the knowledge base that it belongs to is queried.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-vectoringestionconfiguration.html#cfn-bedrock-datasource-vectoringestionconfiguration-chunkingconfiguration
            '''
            result = self._values.get("chunking_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ChunkingConfigurationProperty"]], result)

        @builtins.property
        def context_enrichment_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ContextEnrichmentConfigurationProperty"]]:
            '''The context enrichment configuration used for ingestion of the data into the vector store.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-vectoringestionconfiguration.html#cfn-bedrock-datasource-vectoringestionconfiguration-contextenrichmentconfiguration
            '''
            result = self._values.get("context_enrichment_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ContextEnrichmentConfigurationProperty"]], result)

        @builtins.property
        def custom_transformation_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.CustomTransformationConfigurationProperty"]]:
            '''A custom document transformer for parsed data source documents.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-vectoringestionconfiguration.html#cfn-bedrock-datasource-vectoringestionconfiguration-customtransformationconfiguration
            '''
            result = self._values.get("custom_transformation_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.CustomTransformationConfigurationProperty"]], result)

        @builtins.property
        def parsing_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ParsingConfigurationProperty"]]:
            '''Configurations for a parser to use for parsing documents in your data source.

            If you exclude this field, the default parser will be used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-vectoringestionconfiguration.html#cfn-bedrock-datasource-vectoringestionconfiguration-parsingconfiguration
            '''
            result = self._values.get("parsing_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.ParsingConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VectorIngestionConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.WebCrawlerConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "crawler_limits": "crawlerLimits",
            "exclusion_filters": "exclusionFilters",
            "inclusion_filters": "inclusionFilters",
            "scope": "scope",
            "user_agent": "userAgent",
            "user_agent_header": "userAgentHeader",
        },
    )
    class WebCrawlerConfigurationProperty:
        def __init__(
            self,
            *,
            crawler_limits: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.WebCrawlerLimitsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            exclusion_filters: typing.Optional[typing.Sequence[builtins.str]] = None,
            inclusion_filters: typing.Optional[typing.Sequence[builtins.str]] = None,
            scope: typing.Optional[builtins.str] = None,
            user_agent: typing.Optional[builtins.str] = None,
            user_agent_header: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The configuration of web URLs that you want to crawl.

            You should be authorized to crawl the URLs.

            :param crawler_limits: The configuration of crawl limits for the web URLs.
            :param exclusion_filters: A list of one or more exclusion regular expression patterns to exclude certain URLs. If you specify an inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and the web content of the URL isn’t crawled.
            :param inclusion_filters: A list of one or more inclusion regular expression patterns to include certain URLs. If you specify an inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and the web content of the URL isn’t crawled.
            :param scope: The scope of what is crawled for your URLs. You can choose to crawl only web pages that belong to the same host or primary domain. For example, only web pages that contain the seed URL "https://docs.aws.amazon.com/bedrock/latest/userguide/" and no other domains. You can choose to include sub domains in addition to the host or primary domain. For example, web pages that contain "aws.amazon.com" can also include sub domain "docs.aws.amazon.com".
            :param user_agent: Returns the user agent suffix for your web crawler.
            :param user_agent_header: A string used for identifying the crawler or bot when it accesses a web server. The user agent header value consists of the ``bedrockbot`` , UUID, and a user agent suffix for your crawler (if one is provided). By default, it is set to ``bedrockbot_UUID`` . You can optionally append a custom suffix to ``bedrockbot_UUID`` to allowlist a specific user agent permitted to access your source URLs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webcrawlerconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                web_crawler_configuration_property = bedrock.CfnDataSource.WebCrawlerConfigurationProperty(
                    crawler_limits=bedrock.CfnDataSource.WebCrawlerLimitsProperty(
                        max_pages=123,
                        rate_limit=123
                    ),
                    exclusion_filters=["exclusionFilters"],
                    inclusion_filters=["inclusionFilters"],
                    scope="scope",
                    user_agent="userAgent",
                    user_agent_header="userAgentHeader"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__19f2bf049e2cc7c767a94b404cf6594346506f3a78f18601159a383fa27572d2)
                check_type(argname="argument crawler_limits", value=crawler_limits, expected_type=type_hints["crawler_limits"])
                check_type(argname="argument exclusion_filters", value=exclusion_filters, expected_type=type_hints["exclusion_filters"])
                check_type(argname="argument inclusion_filters", value=inclusion_filters, expected_type=type_hints["inclusion_filters"])
                check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
                check_type(argname="argument user_agent", value=user_agent, expected_type=type_hints["user_agent"])
                check_type(argname="argument user_agent_header", value=user_agent_header, expected_type=type_hints["user_agent_header"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if crawler_limits is not None:
                self._values["crawler_limits"] = crawler_limits
            if exclusion_filters is not None:
                self._values["exclusion_filters"] = exclusion_filters
            if inclusion_filters is not None:
                self._values["inclusion_filters"] = inclusion_filters
            if scope is not None:
                self._values["scope"] = scope
            if user_agent is not None:
                self._values["user_agent"] = user_agent
            if user_agent_header is not None:
                self._values["user_agent_header"] = user_agent_header

        @builtins.property
        def crawler_limits(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.WebCrawlerLimitsProperty"]]:
            '''The configuration of crawl limits for the web URLs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webcrawlerconfiguration.html#cfn-bedrock-datasource-webcrawlerconfiguration-crawlerlimits
            '''
            result = self._values.get("crawler_limits")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.WebCrawlerLimitsProperty"]], result)

        @builtins.property
        def exclusion_filters(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of one or more exclusion regular expression patterns to exclude certain URLs.

            If you specify an inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and the web content of the URL isn’t crawled.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webcrawlerconfiguration.html#cfn-bedrock-datasource-webcrawlerconfiguration-exclusionfilters
            '''
            result = self._values.get("exclusion_filters")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def inclusion_filters(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of one or more inclusion regular expression patterns to include certain URLs.

            If you specify an inclusion and exclusion filter/pattern and both match a URL, the exclusion filter takes precedence and the web content of the URL isn’t crawled.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webcrawlerconfiguration.html#cfn-bedrock-datasource-webcrawlerconfiguration-inclusionfilters
            '''
            result = self._values.get("inclusion_filters")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def scope(self) -> typing.Optional[builtins.str]:
            '''The scope of what is crawled for your URLs.

            You can choose to crawl only web pages that belong to the same host or primary domain. For example, only web pages that contain the seed URL "https://docs.aws.amazon.com/bedrock/latest/userguide/" and no other domains. You can choose to include sub domains in addition to the host or primary domain. For example, web pages that contain "aws.amazon.com" can also include sub domain "docs.aws.amazon.com".

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webcrawlerconfiguration.html#cfn-bedrock-datasource-webcrawlerconfiguration-scope
            '''
            result = self._values.get("scope")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def user_agent(self) -> typing.Optional[builtins.str]:
            '''Returns the user agent suffix for your web crawler.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webcrawlerconfiguration.html#cfn-bedrock-datasource-webcrawlerconfiguration-useragent
            '''
            result = self._values.get("user_agent")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def user_agent_header(self) -> typing.Optional[builtins.str]:
            '''A string used for identifying the crawler or bot when it accesses a web server.

            The user agent header value consists of the ``bedrockbot`` , UUID, and a user agent suffix for your crawler (if one is provided). By default, it is set to ``bedrockbot_UUID`` . You can optionally append a custom suffix to ``bedrockbot_UUID`` to allowlist a specific user agent permitted to access your source URLs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webcrawlerconfiguration.html#cfn-bedrock-datasource-webcrawlerconfiguration-useragentheader
            '''
            result = self._values.get("user_agent_header")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "WebCrawlerConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.WebCrawlerLimitsProperty",
        jsii_struct_bases=[],
        name_mapping={"max_pages": "maxPages", "rate_limit": "rateLimit"},
    )
    class WebCrawlerLimitsProperty:
        def __init__(
            self,
            *,
            max_pages: typing.Optional[jsii.Number] = None,
            rate_limit: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The rate limits for the URLs that you want to crawl.

            You should be authorized to crawl the URLs.

            :param max_pages: The max number of web pages crawled from your source URLs, up to 25,000 pages. If the web pages exceed this limit, the data source sync will fail and no web pages will be ingested.
            :param rate_limit: The max rate at which pages are crawled, up to 300 per minute per host.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webcrawlerlimits.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                web_crawler_limits_property = bedrock.CfnDataSource.WebCrawlerLimitsProperty(
                    max_pages=123,
                    rate_limit=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e3c7708618964f317da86f69cc51ebbcf99d7ff170204876d29a3e67fdbefcf9)
                check_type(argname="argument max_pages", value=max_pages, expected_type=type_hints["max_pages"])
                check_type(argname="argument rate_limit", value=rate_limit, expected_type=type_hints["rate_limit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max_pages is not None:
                self._values["max_pages"] = max_pages
            if rate_limit is not None:
                self._values["rate_limit"] = rate_limit

        @builtins.property
        def max_pages(self) -> typing.Optional[jsii.Number]:
            '''The max number of web pages crawled from your source URLs, up to 25,000 pages.

            If the web pages exceed this limit, the data source sync will fail and no web pages will be ingested.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webcrawlerlimits.html#cfn-bedrock-datasource-webcrawlerlimits-maxpages
            '''
            result = self._values.get("max_pages")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def rate_limit(self) -> typing.Optional[jsii.Number]:
            '''The max rate at which pages are crawled, up to 300 per minute per host.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webcrawlerlimits.html#cfn-bedrock-datasource-webcrawlerlimits-ratelimit
            '''
            result = self._values.get("rate_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "WebCrawlerLimitsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.WebDataSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "source_configuration": "sourceConfiguration",
            "crawler_configuration": "crawlerConfiguration",
        },
    )
    class WebDataSourceConfigurationProperty:
        def __init__(
            self,
            *,
            source_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.WebSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
            crawler_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.WebCrawlerConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration details for the web data source.

            :param source_configuration: The source configuration details for the web data source.
            :param crawler_configuration: The Web Crawler configuration details for the web data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webdatasourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                web_data_source_configuration_property = bedrock.CfnDataSource.WebDataSourceConfigurationProperty(
                    source_configuration=bedrock.CfnDataSource.WebSourceConfigurationProperty(
                        url_configuration=bedrock.CfnDataSource.UrlConfigurationProperty(
                            seed_urls=[bedrock.CfnDataSource.SeedUrlProperty(
                                url="url"
                            )]
                        )
                    ),
                
                    # the properties below are optional
                    crawler_configuration=bedrock.CfnDataSource.WebCrawlerConfigurationProperty(
                        crawler_limits=bedrock.CfnDataSource.WebCrawlerLimitsProperty(
                            max_pages=123,
                            rate_limit=123
                        ),
                        exclusion_filters=["exclusionFilters"],
                        inclusion_filters=["inclusionFilters"],
                        scope="scope",
                        user_agent="userAgent",
                        user_agent_header="userAgentHeader"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b1f60d9766ca17f5c37478ae610f90aa4019525520dc8ade1276a6b78c4f8330)
                check_type(argname="argument source_configuration", value=source_configuration, expected_type=type_hints["source_configuration"])
                check_type(argname="argument crawler_configuration", value=crawler_configuration, expected_type=type_hints["crawler_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "source_configuration": source_configuration,
            }
            if crawler_configuration is not None:
                self._values["crawler_configuration"] = crawler_configuration

        @builtins.property
        def source_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataSource.WebSourceConfigurationProperty"]:
            '''The source configuration details for the web data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webdatasourceconfiguration.html#cfn-bedrock-datasource-webdatasourceconfiguration-sourceconfiguration
            '''
            result = self._values.get("source_configuration")
            assert result is not None, "Required property 'source_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataSource.WebSourceConfigurationProperty"], result)

        @builtins.property
        def crawler_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.WebCrawlerConfigurationProperty"]]:
            '''The Web Crawler configuration details for the web data source.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-webdatasourceconfiguration.html#cfn-bedrock-datasource-webdatasourceconfiguration-crawlerconfiguration
            '''
            result = self._values.get("crawler_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDataSource.WebCrawlerConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "WebDataSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSource.WebSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"url_configuration": "urlConfiguration"},
    )
    class WebSourceConfigurationProperty:
        def __init__(
            self,
            *,
            url_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDataSource.UrlConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''The configuration of the URL/URLs for the web content that you want to crawl.

            You should be authorized to crawl the URLs.

            :param url_configuration: The configuration of the URL/URLs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-websourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                web_source_configuration_property = bedrock.CfnDataSource.WebSourceConfigurationProperty(
                    url_configuration=bedrock.CfnDataSource.UrlConfigurationProperty(
                        seed_urls=[bedrock.CfnDataSource.SeedUrlProperty(
                            url="url"
                        )]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__eed5469efe26ecca30b65f76f177d0a63ddfd391d6f79ab740dabde0e6ae1c45)
                check_type(argname="argument url_configuration", value=url_configuration, expected_type=type_hints["url_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "url_configuration": url_configuration,
            }

        @builtins.property
        def url_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnDataSource.UrlConfigurationProperty"]:
            '''The configuration of the URL/URLs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-datasource-websourceconfiguration.html#cfn-bedrock-datasource-websourceconfiguration-urlconfiguration
            '''
            result = self._values.get("url_configuration")
            assert result is not None, "Required property 'url_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDataSource.UrlConfigurationProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "WebSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_bedrock.CfnDataSourceProps",
    jsii_struct_bases=[],
    name_mapping={
        "data_source_configuration": "dataSourceConfiguration",
        "knowledge_base_id": "knowledgeBaseId",
        "name": "name",
        "data_deletion_policy": "dataDeletionPolicy",
        "description": "description",
        "server_side_encryption_configuration": "serverSideEncryptionConfiguration",
        "vector_ingestion_configuration": "vectorIngestionConfiguration",
    },
)
class CfnDataSourceProps:
    def __init__(
        self,
        *,
        data_source_configuration: typing.Union[_IResolvable_da3f097b, typing.Union[CfnDataSource.DataSourceConfigurationProperty, typing.Dict[builtins.str, typing.Any]]],
        knowledge_base_id: builtins.str,
        name: builtins.str,
        data_deletion_policy: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        server_side_encryption_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDataSource.ServerSideEncryptionConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        vector_ingestion_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDataSource.VectorIngestionConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDataSource``.

        :param data_source_configuration: The connection configuration for the data source.
        :param knowledge_base_id: The unique identifier of the knowledge base to which the data source belongs.
        :param name: The name of the data source.
        :param data_deletion_policy: The data deletion policy for the data source.
        :param description: The description of the data source.
        :param server_side_encryption_configuration: Contains details about the configuration of the server-side encryption.
        :param vector_ingestion_configuration: Contains details about how to ingest the documents in the data source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-datasource.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_bedrock as bedrock
            
            cfn_data_source_props = bedrock.CfnDataSourceProps(
                data_source_configuration=bedrock.CfnDataSource.DataSourceConfigurationProperty(
                    type="type",
            
                    # the properties below are optional
                    confluence_configuration=bedrock.CfnDataSource.ConfluenceDataSourceConfigurationProperty(
                        source_configuration=bedrock.CfnDataSource.ConfluenceSourceConfigurationProperty(
                            auth_type="authType",
                            credentials_secret_arn="credentialsSecretArn",
                            host_type="hostType",
                            host_url="hostUrl"
                        ),
            
                        # the properties below are optional
                        crawler_configuration=bedrock.CfnDataSource.ConfluenceCrawlerConfigurationProperty(
                            filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                                type="type",
            
                                # the properties below are optional
                                pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                    filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                        object_type="objectType",
            
                                        # the properties below are optional
                                        exclusion_filters=["exclusionFilters"],
                                        inclusion_filters=["inclusionFilters"]
                                    )]
                                )
                            )
                        )
                    ),
                    s3_configuration=bedrock.CfnDataSource.S3DataSourceConfigurationProperty(
                        bucket_arn="bucketArn",
            
                        # the properties below are optional
                        bucket_owner_account_id="bucketOwnerAccountId",
                        inclusion_prefixes=["inclusionPrefixes"]
                    ),
                    salesforce_configuration=bedrock.CfnDataSource.SalesforceDataSourceConfigurationProperty(
                        source_configuration=bedrock.CfnDataSource.SalesforceSourceConfigurationProperty(
                            auth_type="authType",
                            credentials_secret_arn="credentialsSecretArn",
                            host_url="hostUrl"
                        ),
            
                        # the properties below are optional
                        crawler_configuration=bedrock.CfnDataSource.SalesforceCrawlerConfigurationProperty(
                            filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                                type="type",
            
                                # the properties below are optional
                                pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                    filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                        object_type="objectType",
            
                                        # the properties below are optional
                                        exclusion_filters=["exclusionFilters"],
                                        inclusion_filters=["inclusionFilters"]
                                    )]
                                )
                            )
                        )
                    ),
                    share_point_configuration=bedrock.CfnDataSource.SharePointDataSourceConfigurationProperty(
                        source_configuration=bedrock.CfnDataSource.SharePointSourceConfigurationProperty(
                            auth_type="authType",
                            credentials_secret_arn="credentialsSecretArn",
                            domain="domain",
                            host_type="hostType",
                            site_urls=["siteUrls"],
            
                            # the properties below are optional
                            tenant_id="tenantId"
                        ),
            
                        # the properties below are optional
                        crawler_configuration=bedrock.CfnDataSource.SharePointCrawlerConfigurationProperty(
                            filter_configuration=bedrock.CfnDataSource.CrawlFilterConfigurationProperty(
                                type="type",
            
                                # the properties below are optional
                                pattern_object_filter=bedrock.CfnDataSource.PatternObjectFilterConfigurationProperty(
                                    filters=[bedrock.CfnDataSource.PatternObjectFilterProperty(
                                        object_type="objectType",
            
                                        # the properties below are optional
                                        exclusion_filters=["exclusionFilters"],
                                        inclusion_filters=["inclusionFilters"]
                                    )]
                                )
                            )
                        )
                    ),
                    web_configuration=bedrock.CfnDataSource.WebDataSourceConfigurationProperty(
                        source_configuration=bedrock.CfnDataSource.WebSourceConfigurationProperty(
                            url_configuration=bedrock.CfnDataSource.UrlConfigurationProperty(
                                seed_urls=[bedrock.CfnDataSource.SeedUrlProperty(
                                    url="url"
                                )]
                            )
                        ),
            
                        # the properties below are optional
                        crawler_configuration=bedrock.CfnDataSource.WebCrawlerConfigurationProperty(
                            crawler_limits=bedrock.CfnDataSource.WebCrawlerLimitsProperty(
                                max_pages=123,
                                rate_limit=123
                            ),
                            exclusion_filters=["exclusionFilters"],
                            inclusion_filters=["inclusionFilters"],
                            scope="scope",
                            user_agent="userAgent",
                            user_agent_header="userAgentHeader"
                        )
                    )
                ),
                knowledge_base_id="knowledgeBaseId",
                name="name",
            
                # the properties below are optional
                data_deletion_policy="dataDeletionPolicy",
                description="description",
                server_side_encryption_configuration=bedrock.CfnDataSource.ServerSideEncryptionConfigurationProperty(
                    kms_key_arn="kmsKeyArn"
                ),
                vector_ingestion_configuration=bedrock.CfnDataSource.VectorIngestionConfigurationProperty(
                    chunking_configuration=bedrock.CfnDataSource.ChunkingConfigurationProperty(
                        chunking_strategy="chunkingStrategy",
            
                        # the properties below are optional
                        fixed_size_chunking_configuration=bedrock.CfnDataSource.FixedSizeChunkingConfigurationProperty(
                            max_tokens=123,
                            overlap_percentage=123
                        ),
                        hierarchical_chunking_configuration=bedrock.CfnDataSource.HierarchicalChunkingConfigurationProperty(
                            level_configurations=[bedrock.CfnDataSource.HierarchicalChunkingLevelConfigurationProperty(
                                max_tokens=123
                            )],
                            overlap_tokens=123
                        ),
                        semantic_chunking_configuration=bedrock.CfnDataSource.SemanticChunkingConfigurationProperty(
                            breakpoint_percentile_threshold=123,
                            buffer_size=123,
                            max_tokens=123
                        )
                    ),
                    context_enrichment_configuration=bedrock.CfnDataSource.ContextEnrichmentConfigurationProperty(
                        type="type",
            
                        # the properties below are optional
                        bedrock_foundation_model_configuration=bedrock.CfnDataSource.BedrockFoundationModelContextEnrichmentConfigurationProperty(
                            enrichment_strategy_configuration=bedrock.CfnDataSource.EnrichmentStrategyConfigurationProperty(
                                method="method"
                            ),
                            model_arn="modelArn"
                        )
                    ),
                    custom_transformation_configuration=bedrock.CfnDataSource.CustomTransformationConfigurationProperty(
                        intermediate_storage=bedrock.CfnDataSource.IntermediateStorageProperty(
                            s3_location=bedrock.CfnDataSource.S3LocationProperty(
                                uri="uri"
                            )
                        ),
                        transformations=[bedrock.CfnDataSource.TransformationProperty(
                            step_to_apply="stepToApply",
                            transformation_function=bedrock.CfnDataSource.TransformationFunctionProperty(
                                transformation_lambda_configuration=bedrock.CfnDataSource.TransformationLambdaConfigurationProperty(
                                    lambda_arn="lambdaArn"
                                )
                            )
                        )]
                    ),
                    parsing_configuration=bedrock.CfnDataSource.ParsingConfigurationProperty(
                        parsing_strategy="parsingStrategy",
            
                        # the properties below are optional
                        bedrock_data_automation_configuration=bedrock.CfnDataSource.BedrockDataAutomationConfigurationProperty(
                            parsing_modality="parsingModality"
                        ),
                        bedrock_foundation_model_configuration=bedrock.CfnDataSource.BedrockFoundationModelConfigurationProperty(
                            model_arn="modelArn",
            
                            # the properties below are optional
                            parsing_modality="parsingModality",
                            parsing_prompt=bedrock.CfnDataSource.ParsingPromptProperty(
                                parsing_prompt_text="parsingPromptText"
                            )
                        )
                    )
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4beca3e3b31c91619a3fa9da2bf185ffd738124b7965f1c90a191b43c7b62664)
            check_type(argname="argument data_source_configuration", value=data_source_configuration, expected_type=type_hints["data_source_configuration"])
            check_type(argname="argument knowledge_base_id", value=knowledge_base_id, expected_type=type_hints["knowledge_base_id"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument data_deletion_policy", value=data_deletion_policy, expected_type=type_hints["data_deletion_policy"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument server_side_encryption_configuration", value=server_side_encryption_configuration, expected_type=type_hints["server_side_encryption_configuration"])
            check_type(argname="argument vector_ingestion_configuration", value=vector_ingestion_configuration, expected_type=type_hints["vector_ingestion_configuration"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "data_source_configuration": data_source_configuration,
            "knowledge_base_id": knowledge_base_id,
            "name": name,
        }
        if data_deletion_policy is not None:
            self._values["data_deletion_policy"] = data_deletion_policy
        if description is not None:
            self._values["description"] = description
        if server_side_encryption_configuration is not None:
            self._values["server_side_encryption_configuration"] = server_side_encryption_configuration
        if vector_ingestion_configuration is not None:
            self._values["vector_ingestion_configuration"] = vector_ingestion_configuration

    @builtins.property
    def data_source_configuration(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnDataSource.DataSourceConfigurationProperty]:
        '''The connection configuration for the data source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-datasource.html#cfn-bedrock-datasource-datasourceconfiguration
        '''
        result = self._values.get("data_source_configuration")
        assert result is not None, "Required property 'data_source_configuration' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnDataSource.DataSourceConfigurationProperty], result)

    @builtins.property
    def knowledge_base_id(self) -> builtins.str:
        '''The unique identifier of the knowledge base to which the data source belongs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-datasource.html#cfn-bedrock-datasource-knowledgebaseid
        '''
        result = self._values.get("knowledge_base_id")
        assert result is not None, "Required property 'knowledge_base_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of the data source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-datasource.html#cfn-bedrock-datasource-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def data_deletion_policy(self) -> typing.Optional[builtins.str]:
        '''The data deletion policy for the data source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-datasource.html#cfn-bedrock-datasource-datadeletionpolicy
        '''
        result = self._values.get("data_deletion_policy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the data source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-datasource.html#cfn-bedrock-datasource-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def server_side_encryption_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDataSource.ServerSideEncryptionConfigurationProperty]]:
        '''Contains details about the configuration of the server-side encryption.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-datasource.html#cfn-bedrock-datasource-serversideencryptionconfiguration
        '''
        result = self._values.get("server_side_encryption_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDataSource.ServerSideEncryptionConfigurationProperty]], result)

    @builtins.property
    def vector_ingestion_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDataSource.VectorIngestionConfigurationProperty]]:
        '''Contains details about how to ingest the documents in the data source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-datasource.html#cfn-bedrock-datasource-vectoringestionconfiguration
        '''
        result = self._values.get("vector_ingestion_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDataSource.VectorIngestionConfigurationProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDataSourceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnFlow(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow",
):
    '''Creates a prompt flow that you can use to send an input through various steps to yield an output.

    You define a flow by configuring nodes, each of which corresponds to a step of the flow, and creating connections between the nodes to create paths to different outputs. You can define the flow in one of the following ways:

    - Define a `FlowDefinition <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdefinition.html>`_ in the ``Definition`` property.
    - Provide the definition in the ``DefinitionString`` property as a JSON-formatted string matching the `FlowDefinition <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdefinition.html>`_ property.
    - Provide an Amazon S3 location in the ``DefinitionS3Location`` property that matches the `FlowDefinition <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdefinition.html>`_ .

    If you use the ``DefinitionString`` or ``DefinitionS3Location`` property, you can use the ``DefinitionSubstitutions`` property to define key-value pairs to replace at runtime.

    For more information, see `How it works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-how-it-works.html>`_ and `Create a prompt flow in Amazon Bedrock <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-create.html>`_ in the Amazon Bedrock User Guide.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html
    :cloudformationResource: AWS::Bedrock::Flow
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_bedrock as bedrock
        
        # collector: Any
        # input: Any
        # iterator: Any
        # output: Any
        
        cfn_flow = bedrock.CfnFlow(self, "MyCfnFlow",
            execution_role_arn="executionRoleArn",
            name="name",
        
            # the properties below are optional
            customer_encryption_key_arn="customerEncryptionKeyArn",
            definition=bedrock.CfnFlow.FlowDefinitionProperty(
                connections=[bedrock.CfnFlow.FlowConnectionProperty(
                    name="name",
                    source="source",
                    target="target",
                    type="type",
        
                    # the properties below are optional
                    configuration=bedrock.CfnFlow.FlowConnectionConfigurationProperty(
                        conditional=bedrock.CfnFlow.FlowConditionalConnectionConfigurationProperty(
                            condition="condition"
                        ),
                        data=bedrock.CfnFlow.FlowDataConnectionConfigurationProperty(
                            source_output="sourceOutput",
                            target_input="targetInput"
                        )
                    )
                )],
                nodes=[bedrock.CfnFlow.FlowNodeProperty(
                    name="name",
                    type="type",
        
                    # the properties below are optional
                    configuration=bedrock.CfnFlow.FlowNodeConfigurationProperty(
                        agent=bedrock.CfnFlow.AgentFlowNodeConfigurationProperty(
                            agent_alias_arn="agentAliasArn"
                        ),
                        collector=collector,
                        condition=bedrock.CfnFlow.ConditionFlowNodeConfigurationProperty(
                            conditions=[bedrock.CfnFlow.FlowConditionProperty(
                                name="name",
        
                                # the properties below are optional
                                expression="expression"
                            )]
                        ),
                        input=input,
                        iterator=iterator,
                        knowledge_base=bedrock.CfnFlow.KnowledgeBaseFlowNodeConfigurationProperty(
                            knowledge_base_id="knowledgeBaseId",
        
                            # the properties below are optional
                            guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                                guardrail_identifier="guardrailIdentifier",
                                guardrail_version="guardrailVersion"
                            ),
                            model_id="modelId"
                        ),
                        lambda_function=bedrock.CfnFlow.LambdaFunctionFlowNodeConfigurationProperty(
                            lambda_arn="lambdaArn"
                        ),
                        lex=bedrock.CfnFlow.LexFlowNodeConfigurationProperty(
                            bot_alias_arn="botAliasArn",
                            locale_id="localeId"
                        ),
                        output=output,
                        prompt=bedrock.CfnFlow.PromptFlowNodeConfigurationProperty(
                            source_configuration=bedrock.CfnFlow.PromptFlowNodeSourceConfigurationProperty(
                                inline=bedrock.CfnFlow.PromptFlowNodeInlineConfigurationProperty(
                                    model_id="modelId",
                                    template_configuration=bedrock.CfnFlow.PromptTemplateConfigurationProperty(
                                        text=bedrock.CfnFlow.TextPromptTemplateConfigurationProperty(
                                            text="text",
        
                                            # the properties below are optional
                                            input_variables=[bedrock.CfnFlow.PromptInputVariableProperty(
                                                name="name"
                                            )]
                                        )
                                    ),
                                    template_type="templateType",
        
                                    # the properties below are optional
                                    inference_configuration=bedrock.CfnFlow.PromptInferenceConfigurationProperty(
                                        text=bedrock.CfnFlow.PromptModelInferenceConfigurationProperty(
                                            max_tokens=123,
                                            stop_sequences=["stopSequences"],
                                            temperature=123,
                                            top_p=123
                                        )
                                    )
                                ),
                                resource=bedrock.CfnFlow.PromptFlowNodeResourceConfigurationProperty(
                                    prompt_arn="promptArn"
                                )
                            ),
        
                            # the properties below are optional
                            guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                                guardrail_identifier="guardrailIdentifier",
                                guardrail_version="guardrailVersion"
                            )
                        ),
                        retrieval=bedrock.CfnFlow.RetrievalFlowNodeConfigurationProperty(
                            service_configuration=bedrock.CfnFlow.RetrievalFlowNodeServiceConfigurationProperty(
                                s3=bedrock.CfnFlow.RetrievalFlowNodeS3ConfigurationProperty(
                                    bucket_name="bucketName"
                                )
                            )
                        ),
                        storage=bedrock.CfnFlow.StorageFlowNodeConfigurationProperty(
                            service_configuration=bedrock.CfnFlow.StorageFlowNodeServiceConfigurationProperty(
                                s3=bedrock.CfnFlow.StorageFlowNodeS3ConfigurationProperty(
                                    bucket_name="bucketName"
                                )
                            )
                        )
                    ),
                    inputs=[bedrock.CfnFlow.FlowNodeInputProperty(
                        expression="expression",
                        name="name",
                        type="type"
                    )],
                    outputs=[bedrock.CfnFlow.FlowNodeOutputProperty(
                        name="name",
                        type="type"
                    )]
                )]
            ),
            definition_s3_location=bedrock.CfnFlow.S3LocationProperty(
                bucket="bucket",
                key="key",
        
                # the properties below are optional
                version="version"
            ),
            definition_string="definitionString",
            definition_substitutions={
                "definition_substitutions_key": "definitionSubstitutions"
            },
            description="description",
            tags={
                "tags_key": "tags"
            },
            test_alias_tags={
                "test_alias_tags_key": "testAliasTags"
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        execution_role_arn: builtins.str,
        name: builtins.str,
        customer_encryption_key_arn: typing.Optional[builtins.str] = None,
        definition: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.FlowDefinitionProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        definition_s3_location: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.S3LocationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        definition_string: typing.Optional[builtins.str] = None,
        definition_substitutions: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool, _IResolvable_da3f097b]]]] = None,
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        test_alias_tags: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param execution_role_arn: The Amazon Resource Name (ARN) of the service role with permissions to create a flow. For more information, see `Create a service row for flows <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-permissions.html>`_ in the Amazon Bedrock User Guide.
        :param name: The name of the flow.
        :param customer_encryption_key_arn: The Amazon Resource Name (ARN) of the KMS key that the flow is encrypted with.
        :param definition: The definition of the nodes and connections between the nodes in the flow.
        :param definition_s3_location: The Amazon S3 location of the flow definition.
        :param definition_string: The definition of the flow as a JSON-formatted string. The string must match the format in `FlowDefinition <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdefinition.html>`_ .
        :param definition_substitutions: A map that specifies the mappings for placeholder variables in the prompt flow definition. This enables the customer to inject values obtained at runtime. Variables can be template parameter names, resource logical IDs, resource attributes, or a variable in a key-value map. Only supported with the ``DefinitionString`` and ``DefinitionS3Location`` fields. Substitutions must follow the syntax: ``${key_name}`` or ``${variable_1,variable_2,...}`` .
        :param description: A description of the flow.
        :param tags: Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:. - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_ - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_
        :param test_alias_tags: A map of tag keys and values.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__de6f163cc9336f6925ec17d7466e0c9a53cd276d201d3e7dcf092e2b153409e1)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnFlowProps(
            execution_role_arn=execution_role_arn,
            name=name,
            customer_encryption_key_arn=customer_encryption_key_arn,
            definition=definition,
            definition_s3_location=definition_s3_location,
            definition_string=definition_string,
            definition_substitutions=definition_substitutions,
            description=description,
            tags=tags,
            test_alias_tags=test_alias_tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__afbbfbf59fe4fd85bdd0a9b41f5eeb9b87a60af4b54c39839fc10611790dfef5)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__203d2c4faff72c8c73da02ebdbd5c702cce12257ef579761ca737823c7bddf53)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the flow.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrCreatedAt")
    def attr_created_at(self) -> builtins.str:
        '''The time at which the flow was created.

        :cloudformationAttribute: CreatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreatedAt"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The unique identifier of the flow.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrStatus")
    def attr_status(self) -> builtins.str:
        '''The status of the flow. The following statuses are possible:.

        - NotPrepared – The flow has been created or updated, but hasn't been prepared. If you just created the flow, you can't test it. If you updated the flow, the ``DRAFT`` version won't contain the latest changes for testing. Send a `PrepareFlow <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PrepareFlow.html>`_ request to package the latest changes into the ``DRAFT`` version.
        - Preparing – The flow is being prepared so that the ``DRAFT`` version contains the latest changes for testing.
        - Prepared – The flow is prepared and the ``DRAFT`` version contains the latest changes for testing.
        - Failed – The last API operation that you invoked on the flow failed. Send a `GetFlow <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetFlow.html>`_ request and check the error message in the ``validations`` field.

        :cloudformationAttribute: Status
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrStatus"))

    @builtins.property
    @jsii.member(jsii_name="attrUpdatedAt")
    def attr_updated_at(self) -> builtins.str:
        '''The time at which the flow was last updated.

        :cloudformationAttribute: UpdatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUpdatedAt"))

    @builtins.property
    @jsii.member(jsii_name="attrValidations")
    def attr_validations(self) -> _IResolvable_da3f097b:
        '''List of flow validations.

        :cloudformationAttribute: Validations
        '''
        return typing.cast(_IResolvable_da3f097b, jsii.get(self, "attrValidations"))

    @builtins.property
    @jsii.member(jsii_name="attrVersion")
    def attr_version(self) -> builtins.str:
        '''The latest version of the flow.

        :cloudformationAttribute: Version
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVersion"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="executionRoleArn")
    def execution_role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the service role with permissions to create a flow.'''
        return typing.cast(builtins.str, jsii.get(self, "executionRoleArn"))

    @execution_role_arn.setter
    def execution_role_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ced6deb90295eabf196b6ada4e8d987db7eeb74378cacdafc998cd5624e0b0fd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "executionRoleArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the flow.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ee04f94b9ba27ecbb092c04e1407a9f49851186ac60849eb9a2b502870e6d1b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="customerEncryptionKeyArn")
    def customer_encryption_key_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the KMS key that the flow is encrypted with.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "customerEncryptionKeyArn"))

    @customer_encryption_key_arn.setter
    def customer_encryption_key_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aa845c7cbe71441f89a12cd66601294a3af83142648889f26de31b6b333331cd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "customerEncryptionKeyArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="definition")
    def definition(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowDefinitionProperty"]]:
        '''The definition of the nodes and connections between the nodes in the flow.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowDefinitionProperty"]], jsii.get(self, "definition"))

    @definition.setter
    def definition(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowDefinitionProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c2c15de597c18bdf1cfd80b3544262ec255a2eb266848edd42a70afb0018fcec)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "definition", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="definitionS3Location")
    def definition_s3_location(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.S3LocationProperty"]]:
        '''The Amazon S3 location of the flow definition.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.S3LocationProperty"]], jsii.get(self, "definitionS3Location"))

    @definition_s3_location.setter
    def definition_s3_location(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.S3LocationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__562b099489964ba014f8624b070d1781f2feaeebe40d9388399fc7ae66dff9c1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "definitionS3Location", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="definitionString")
    def definition_string(self) -> typing.Optional[builtins.str]:
        '''The definition of the flow as a JSON-formatted string.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "definitionString"))

    @definition_string.setter
    def definition_string(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__58ac2e2c1626b12db3fe6e46ef64157358fd21ea0518a77674fd75522b27b8d0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "definitionString", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="definitionSubstitutions")
    def definition_substitutions(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool, _IResolvable_da3f097b]]]]:
        '''A map that specifies the mappings for placeholder variables in the prompt flow definition.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool, _IResolvable_da3f097b]]]], jsii.get(self, "definitionSubstitutions"))

    @definition_substitutions.setter
    def definition_substitutions(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool, _IResolvable_da3f097b]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__665f20040e8cdbcc235b9965731830ae44c7f9d50bfdd4c03665e9aba78547c8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "definitionSubstitutions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the flow.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b2df04d9d27600a49c8dc9726bf078edbf14dccccbdf53efbf100dea82058ee6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Metadata that you can assign to a resource as key-value pairs.

        For more information, see the following resources:.
        '''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "tags"))

    @tags.setter
    def tags(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__00fc718d7b494a828bea538bc7c10f721679354bb6555d8e15e32b2169ffe891)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="testAliasTags")
    def test_alias_tags(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''A map of tag keys and values.'''
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], jsii.get(self, "testAliasTags"))

    @test_alias_tags.setter
    def test_alias_tags(
        self,
        value: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c8f573e426abfd3b90c4d9ff9fecb4f008835544cca9591251c138795f9dac1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "testAliasTags", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.AgentFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"agent_alias_arn": "agentAliasArn"},
    )
    class AgentFlowNodeConfigurationProperty:
        def __init__(self, *, agent_alias_arn: builtins.str) -> None:
            '''Defines an agent node in your flow.

            You specify the agent to invoke at this point in the flow. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param agent_alias_arn: The Amazon Resource Name (ARN) of the alias of the agent to invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-agentflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                agent_flow_node_configuration_property = bedrock.CfnFlow.AgentFlowNodeConfigurationProperty(
                    agent_alias_arn="agentAliasArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a835b75c491363488063d3b7d26eef8f2e4a2e2f689eb3676748164e4491c5fe)
                check_type(argname="argument agent_alias_arn", value=agent_alias_arn, expected_type=type_hints["agent_alias_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "agent_alias_arn": agent_alias_arn,
            }

        @builtins.property
        def agent_alias_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the alias of the agent to invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-agentflownodeconfiguration.html#cfn-bedrock-flow-agentflownodeconfiguration-agentaliasarn
            '''
            result = self._values.get("agent_alias_arn")
            assert result is not None, "Required property 'agent_alias_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AgentFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.ConditionFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"conditions": "conditions"},
    )
    class ConditionFlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            conditions: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.FlowConditionProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''Defines a condition node in your flow.

            You can specify conditions that determine which node comes next in the flow. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param conditions: An array of conditions. Each member contains the name of a condition and an expression that defines the condition.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-conditionflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                condition_flow_node_configuration_property = bedrock.CfnFlow.ConditionFlowNodeConfigurationProperty(
                    conditions=[bedrock.CfnFlow.FlowConditionProperty(
                        name="name",
                
                        # the properties below are optional
                        expression="expression"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dda3f0b71fa4dedd72606bcb4dc1b235e5e8f33ebe66758c36774f374b1f09b1)
                check_type(argname="argument conditions", value=conditions, expected_type=type_hints["conditions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "conditions": conditions,
            }

        @builtins.property
        def conditions(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowConditionProperty"]]]:
            '''An array of conditions.

            Each member contains the name of a condition and an expression that defines the condition.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-conditionflownodeconfiguration.html#cfn-bedrock-flow-conditionflownodeconfiguration-conditions
            '''
            result = self._values.get("conditions")
            assert result is not None, "Required property 'conditions' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowConditionProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConditionFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.FlowConditionProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "expression": "expression"},
    )
    class FlowConditionProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            expression: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Defines a condition in the condition node.

            :param name: A name for the condition that you can reference.
            :param expression: Defines the condition. You must refer to at least one of the inputs in the condition. For more information, expand the Condition node section in `Node types in prompt flows <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-how-it-works.html#flows-nodes>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowcondition.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_condition_property = bedrock.CfnFlow.FlowConditionProperty(
                    name="name",
                
                    # the properties below are optional
                    expression="expression"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d3a9f020a1b372979f9d1ec6b911d2295cf1eb524904677904d64a3f6a35055a)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument expression", value=expression, expected_type=type_hints["expression"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if expression is not None:
                self._values["expression"] = expression

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the condition that you can reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowcondition.html#cfn-bedrock-flow-flowcondition-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def expression(self) -> typing.Optional[builtins.str]:
            '''Defines the condition.

            You must refer to at least one of the inputs in the condition. For more information, expand the Condition node section in `Node types in prompt flows <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-how-it-works.html#flows-nodes>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowcondition.html#cfn-bedrock-flow-flowcondition-expression
            '''
            result = self._values.get("expression")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowConditionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.FlowConditionalConnectionConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"condition": "condition"},
    )
    class FlowConditionalConnectionConfigurationProperty:
        def __init__(self, *, condition: builtins.str) -> None:
            '''The configuration of a connection between a condition node and another node.

            :param condition: The condition that triggers this connection. For more information about how to write conditions, see the *Condition* node type in the `Node types <https://docs.aws.amazon.com/bedrock/latest/userguide/node-types.html>`_ topic in the Amazon Bedrock User Guide.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowconditionalconnectionconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_conditional_connection_configuration_property = bedrock.CfnFlow.FlowConditionalConnectionConfigurationProperty(
                    condition="condition"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e883ba05605d1eb123d58573aad4b50ac6d54755cda11847100fd31a2196536e)
                check_type(argname="argument condition", value=condition, expected_type=type_hints["condition"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "condition": condition,
            }

        @builtins.property
        def condition(self) -> builtins.str:
            '''The condition that triggers this connection.

            For more information about how to write conditions, see the *Condition* node type in the `Node types <https://docs.aws.amazon.com/bedrock/latest/userguide/node-types.html>`_ topic in the Amazon Bedrock User Guide.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowconditionalconnectionconfiguration.html#cfn-bedrock-flow-flowconditionalconnectionconfiguration-condition
            '''
            result = self._values.get("condition")
            assert result is not None, "Required property 'condition' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowConditionalConnectionConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.FlowConnectionConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"conditional": "conditional", "data": "data"},
    )
    class FlowConnectionConfigurationProperty:
        def __init__(
            self,
            *,
            conditional: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.FlowConditionalConnectionConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            data: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.FlowDataConnectionConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration of the connection.

            :param conditional: The configuration of a connection originating from a Condition node.
            :param data: The configuration of a connection originating from a node that isn't a Condition node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowconnectionconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_connection_configuration_property = bedrock.CfnFlow.FlowConnectionConfigurationProperty(
                    conditional=bedrock.CfnFlow.FlowConditionalConnectionConfigurationProperty(
                        condition="condition"
                    ),
                    data=bedrock.CfnFlow.FlowDataConnectionConfigurationProperty(
                        source_output="sourceOutput",
                        target_input="targetInput"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__99bc335850e6ead734360fa1f71216709094ac534a41863415ad734580560f48)
                check_type(argname="argument conditional", value=conditional, expected_type=type_hints["conditional"])
                check_type(argname="argument data", value=data, expected_type=type_hints["data"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if conditional is not None:
                self._values["conditional"] = conditional
            if data is not None:
                self._values["data"] = data

        @builtins.property
        def conditional(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowConditionalConnectionConfigurationProperty"]]:
            '''The configuration of a connection originating from a Condition node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowconnectionconfiguration.html#cfn-bedrock-flow-flowconnectionconfiguration-conditional
            '''
            result = self._values.get("conditional")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowConditionalConnectionConfigurationProperty"]], result)

        @builtins.property
        def data(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowDataConnectionConfigurationProperty"]]:
            '''The configuration of a connection originating from a node that isn't a Condition node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowconnectionconfiguration.html#cfn-bedrock-flow-flowconnectionconfiguration-data
            '''
            result = self._values.get("data")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowDataConnectionConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowConnectionConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.FlowConnectionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "source": "source",
            "target": "target",
            "type": "type",
            "configuration": "configuration",
        },
    )
    class FlowConnectionProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            source: builtins.str,
            target: builtins.str,
            type: builtins.str,
            configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.FlowConnectionConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains information about a connection between two nodes in the flow.

            :param name: A name for the connection that you can reference.
            :param source: The node that the connection starts at.
            :param target: The node that the connection ends at.
            :param type: Whether the source node that the connection begins from is a condition node ( ``Conditional`` ) or not ( ``Data`` ).
            :param configuration: The configuration of the connection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowconnection.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_connection_property = bedrock.CfnFlow.FlowConnectionProperty(
                    name="name",
                    source="source",
                    target="target",
                    type="type",
                
                    # the properties below are optional
                    configuration=bedrock.CfnFlow.FlowConnectionConfigurationProperty(
                        conditional=bedrock.CfnFlow.FlowConditionalConnectionConfigurationProperty(
                            condition="condition"
                        ),
                        data=bedrock.CfnFlow.FlowDataConnectionConfigurationProperty(
                            source_output="sourceOutput",
                            target_input="targetInput"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3816533ca4ee1255e0ada08e57423924ab1bea6b0b9f5f4481210079d97a6915)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument source", value=source, expected_type=type_hints["source"])
                check_type(argname="argument target", value=target, expected_type=type_hints["target"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument configuration", value=configuration, expected_type=type_hints["configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "source": source,
                "target": target,
                "type": type,
            }
            if configuration is not None:
                self._values["configuration"] = configuration

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the connection that you can reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowconnection.html#cfn-bedrock-flow-flowconnection-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def source(self) -> builtins.str:
            '''The node that the connection starts at.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowconnection.html#cfn-bedrock-flow-flowconnection-source
            '''
            result = self._values.get("source")
            assert result is not None, "Required property 'source' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def target(self) -> builtins.str:
            '''The node that the connection ends at.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowconnection.html#cfn-bedrock-flow-flowconnection-target
            '''
            result = self._values.get("target")
            assert result is not None, "Required property 'target' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def type(self) -> builtins.str:
            '''Whether the source node that the connection begins from is a condition node ( ``Conditional`` ) or not ( ``Data`` ).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowconnection.html#cfn-bedrock-flow-flowconnection-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowConnectionConfigurationProperty"]]:
            '''The configuration of the connection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowconnection.html#cfn-bedrock-flow-flowconnection-configuration
            '''
            result = self._values.get("configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowConnectionConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowConnectionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.FlowDataConnectionConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"source_output": "sourceOutput", "target_input": "targetInput"},
    )
    class FlowDataConnectionConfigurationProperty:
        def __init__(
            self,
            *,
            source_output: builtins.str,
            target_input: builtins.str,
        ) -> None:
            '''The configuration of a connection originating from a node that isn't a Condition node.

            :param source_output: The name of the output in the source node that the connection begins from.
            :param target_input: The name of the input in the target node that the connection ends at.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdataconnectionconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_data_connection_configuration_property = bedrock.CfnFlow.FlowDataConnectionConfigurationProperty(
                    source_output="sourceOutput",
                    target_input="targetInput"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c7ac426b27ff407377503790d301652931b2d756776b87cdb8b366ed6857aa0a)
                check_type(argname="argument source_output", value=source_output, expected_type=type_hints["source_output"])
                check_type(argname="argument target_input", value=target_input, expected_type=type_hints["target_input"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "source_output": source_output,
                "target_input": target_input,
            }

        @builtins.property
        def source_output(self) -> builtins.str:
            '''The name of the output in the source node that the connection begins from.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdataconnectionconfiguration.html#cfn-bedrock-flow-flowdataconnectionconfiguration-sourceoutput
            '''
            result = self._values.get("source_output")
            assert result is not None, "Required property 'source_output' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def target_input(self) -> builtins.str:
            '''The name of the input in the target node that the connection ends at.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdataconnectionconfiguration.html#cfn-bedrock-flow-flowdataconnectionconfiguration-targetinput
            '''
            result = self._values.get("target_input")
            assert result is not None, "Required property 'target_input' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowDataConnectionConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.FlowDefinitionProperty",
        jsii_struct_bases=[],
        name_mapping={"connections": "connections", "nodes": "nodes"},
    )
    class FlowDefinitionProperty:
        def __init__(
            self,
            *,
            connections: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.FlowConnectionProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            nodes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.FlowNodeProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The definition of the nodes and connections between nodes in the flow.

            :param connections: An array of connection definitions in the flow.
            :param nodes: An array of node definitions in the flow.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdefinition.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                # collector: Any
                # input: Any
                # iterator: Any
                # output: Any
                
                flow_definition_property = bedrock.CfnFlow.FlowDefinitionProperty(
                    connections=[bedrock.CfnFlow.FlowConnectionProperty(
                        name="name",
                        source="source",
                        target="target",
                        type="type",
                
                        # the properties below are optional
                        configuration=bedrock.CfnFlow.FlowConnectionConfigurationProperty(
                            conditional=bedrock.CfnFlow.FlowConditionalConnectionConfigurationProperty(
                                condition="condition"
                            ),
                            data=bedrock.CfnFlow.FlowDataConnectionConfigurationProperty(
                                source_output="sourceOutput",
                                target_input="targetInput"
                            )
                        )
                    )],
                    nodes=[bedrock.CfnFlow.FlowNodeProperty(
                        name="name",
                        type="type",
                
                        # the properties below are optional
                        configuration=bedrock.CfnFlow.FlowNodeConfigurationProperty(
                            agent=bedrock.CfnFlow.AgentFlowNodeConfigurationProperty(
                                agent_alias_arn="agentAliasArn"
                            ),
                            collector=collector,
                            condition=bedrock.CfnFlow.ConditionFlowNodeConfigurationProperty(
                                conditions=[bedrock.CfnFlow.FlowConditionProperty(
                                    name="name",
                
                                    # the properties below are optional
                                    expression="expression"
                                )]
                            ),
                            input=input,
                            iterator=iterator,
                            knowledge_base=bedrock.CfnFlow.KnowledgeBaseFlowNodeConfigurationProperty(
                                knowledge_base_id="knowledgeBaseId",
                
                                # the properties below are optional
                                guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                                    guardrail_identifier="guardrailIdentifier",
                                    guardrail_version="guardrailVersion"
                                ),
                                model_id="modelId"
                            ),
                            lambda_function=bedrock.CfnFlow.LambdaFunctionFlowNodeConfigurationProperty(
                                lambda_arn="lambdaArn"
                            ),
                            lex=bedrock.CfnFlow.LexFlowNodeConfigurationProperty(
                                bot_alias_arn="botAliasArn",
                                locale_id="localeId"
                            ),
                            output=output,
                            prompt=bedrock.CfnFlow.PromptFlowNodeConfigurationProperty(
                                source_configuration=bedrock.CfnFlow.PromptFlowNodeSourceConfigurationProperty(
                                    inline=bedrock.CfnFlow.PromptFlowNodeInlineConfigurationProperty(
                                        model_id="modelId",
                                        template_configuration=bedrock.CfnFlow.PromptTemplateConfigurationProperty(
                                            text=bedrock.CfnFlow.TextPromptTemplateConfigurationProperty(
                                                text="text",
                
                                                # the properties below are optional
                                                input_variables=[bedrock.CfnFlow.PromptInputVariableProperty(
                                                    name="name"
                                                )]
                                            )
                                        ),
                                        template_type="templateType",
                
                                        # the properties below are optional
                                        inference_configuration=bedrock.CfnFlow.PromptInferenceConfigurationProperty(
                                            text=bedrock.CfnFlow.PromptModelInferenceConfigurationProperty(
                                                max_tokens=123,
                                                stop_sequences=["stopSequences"],
                                                temperature=123,
                                                top_p=123
                                            )
                                        )
                                    ),
                                    resource=bedrock.CfnFlow.PromptFlowNodeResourceConfigurationProperty(
                                        prompt_arn="promptArn"
                                    )
                                ),
                
                                # the properties below are optional
                                guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                                    guardrail_identifier="guardrailIdentifier",
                                    guardrail_version="guardrailVersion"
                                )
                            ),
                            retrieval=bedrock.CfnFlow.RetrievalFlowNodeConfigurationProperty(
                                service_configuration=bedrock.CfnFlow.RetrievalFlowNodeServiceConfigurationProperty(
                                    s3=bedrock.CfnFlow.RetrievalFlowNodeS3ConfigurationProperty(
                                        bucket_name="bucketName"
                                    )
                                )
                            ),
                            storage=bedrock.CfnFlow.StorageFlowNodeConfigurationProperty(
                                service_configuration=bedrock.CfnFlow.StorageFlowNodeServiceConfigurationProperty(
                                    s3=bedrock.CfnFlow.StorageFlowNodeS3ConfigurationProperty(
                                        bucket_name="bucketName"
                                    )
                                )
                            )
                        ),
                        inputs=[bedrock.CfnFlow.FlowNodeInputProperty(
                            expression="expression",
                            name="name",
                            type="type"
                        )],
                        outputs=[bedrock.CfnFlow.FlowNodeOutputProperty(
                            name="name",
                            type="type"
                        )]
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9871e2259e9da04030894d1ca021265a5d802265579dabeef193778443c1602e)
                check_type(argname="argument connections", value=connections, expected_type=type_hints["connections"])
                check_type(argname="argument nodes", value=nodes, expected_type=type_hints["nodes"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if connections is not None:
                self._values["connections"] = connections
            if nodes is not None:
                self._values["nodes"] = nodes

        @builtins.property
        def connections(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowConnectionProperty"]]]]:
            '''An array of connection definitions in the flow.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdefinition.html#cfn-bedrock-flow-flowdefinition-connections
            '''
            result = self._values.get("connections")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowConnectionProperty"]]]], result)

        @builtins.property
        def nodes(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowNodeProperty"]]]]:
            '''An array of node definitions in the flow.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdefinition.html#cfn-bedrock-flow-flowdefinition-nodes
            '''
            result = self._values.get("nodes")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowNodeProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowDefinitionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.FlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "agent": "agent",
            "collector": "collector",
            "condition": "condition",
            "input": "input",
            "iterator": "iterator",
            "knowledge_base": "knowledgeBase",
            "lambda_function": "lambdaFunction",
            "lex": "lex",
            "output": "output",
            "prompt": "prompt",
            "retrieval": "retrieval",
            "storage": "storage",
        },
    )
    class FlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            agent: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.AgentFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            collector: typing.Any = None,
            condition: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.ConditionFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            input: typing.Any = None,
            iterator: typing.Any = None,
            knowledge_base: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.KnowledgeBaseFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            lambda_function: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.LambdaFunctionFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            lex: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.LexFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            output: typing.Any = None,
            prompt: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.PromptFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            retrieval: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.RetrievalFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            storage: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.StorageFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains configurations for a node in your flow.

            For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param agent: Contains configurations for an agent node in your flow. Invokes an alias of an agent and returns the response.
            :param collector: Contains configurations for a collector node in your flow. Collects an iteration of inputs and consolidates them into an array of outputs.
            :param condition: Contains configurations for a Condition node in your flow. Defines conditions that lead to different branches of the flow.
            :param input: Contains configurations for an input flow node in your flow. The first node in the flow. ``inputs`` can't be specified for this node.
            :param iterator: Contains configurations for an iterator node in your flow. Takes an input that is an array and iteratively sends each item of the array as an output to the following node. The size of the array is also returned in the output. The output flow node at the end of the flow iteration will return a response for each member of the array. To return only one response, you can include a collector node downstream from the iterator node.
            :param knowledge_base: Contains configurations for a knowledge base node in your flow. Queries a knowledge base and returns the retrieved results or generated response.
            :param lambda_function: Contains configurations for a Lambda function node in your flow. Invokes an AWS Lambda function.
            :param lex: Contains configurations for a Lex node in your flow. Invokes an Amazon Lex bot to identify the intent of the input and return the intent as the output.
            :param output: Contains configurations for an output flow node in your flow. The last node in the flow. ``outputs`` can't be specified for this node.
            :param prompt: Contains configurations for a prompt node in your flow. Runs a prompt and generates the model response as the output. You can use a prompt from Prompt management or you can configure one in this node.
            :param retrieval: Contains configurations for a Retrieval node in your flow. Retrieves data from an Amazon S3 location and returns it as the output.
            :param storage: Contains configurations for a Storage node in your flow. Stores an input in an Amazon S3 location.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                # collector: Any
                # input: Any
                # iterator: Any
                # output: Any
                
                flow_node_configuration_property = bedrock.CfnFlow.FlowNodeConfigurationProperty(
                    agent=bedrock.CfnFlow.AgentFlowNodeConfigurationProperty(
                        agent_alias_arn="agentAliasArn"
                    ),
                    collector=collector,
                    condition=bedrock.CfnFlow.ConditionFlowNodeConfigurationProperty(
                        conditions=[bedrock.CfnFlow.FlowConditionProperty(
                            name="name",
                
                            # the properties below are optional
                            expression="expression"
                        )]
                    ),
                    input=input,
                    iterator=iterator,
                    knowledge_base=bedrock.CfnFlow.KnowledgeBaseFlowNodeConfigurationProperty(
                        knowledge_base_id="knowledgeBaseId",
                
                        # the properties below are optional
                        guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                            guardrail_identifier="guardrailIdentifier",
                            guardrail_version="guardrailVersion"
                        ),
                        model_id="modelId"
                    ),
                    lambda_function=bedrock.CfnFlow.LambdaFunctionFlowNodeConfigurationProperty(
                        lambda_arn="lambdaArn"
                    ),
                    lex=bedrock.CfnFlow.LexFlowNodeConfigurationProperty(
                        bot_alias_arn="botAliasArn",
                        locale_id="localeId"
                    ),
                    output=output,
                    prompt=bedrock.CfnFlow.PromptFlowNodeConfigurationProperty(
                        source_configuration=bedrock.CfnFlow.PromptFlowNodeSourceConfigurationProperty(
                            inline=bedrock.CfnFlow.PromptFlowNodeInlineConfigurationProperty(
                                model_id="modelId",
                                template_configuration=bedrock.CfnFlow.PromptTemplateConfigurationProperty(
                                    text=bedrock.CfnFlow.TextPromptTemplateConfigurationProperty(
                                        text="text",
                
                                        # the properties below are optional
                                        input_variables=[bedrock.CfnFlow.PromptInputVariableProperty(
                                            name="name"
                                        )]
                                    )
                                ),
                                template_type="templateType",
                
                                # the properties below are optional
                                inference_configuration=bedrock.CfnFlow.PromptInferenceConfigurationProperty(
                                    text=bedrock.CfnFlow.PromptModelInferenceConfigurationProperty(
                                        max_tokens=123,
                                        stop_sequences=["stopSequences"],
                                        temperature=123,
                                        top_p=123
                                    )
                                )
                            ),
                            resource=bedrock.CfnFlow.PromptFlowNodeResourceConfigurationProperty(
                                prompt_arn="promptArn"
                            )
                        ),
                
                        # the properties below are optional
                        guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                            guardrail_identifier="guardrailIdentifier",
                            guardrail_version="guardrailVersion"
                        )
                    ),
                    retrieval=bedrock.CfnFlow.RetrievalFlowNodeConfigurationProperty(
                        service_configuration=bedrock.CfnFlow.RetrievalFlowNodeServiceConfigurationProperty(
                            s3=bedrock.CfnFlow.RetrievalFlowNodeS3ConfigurationProperty(
                                bucket_name="bucketName"
                            )
                        )
                    ),
                    storage=bedrock.CfnFlow.StorageFlowNodeConfigurationProperty(
                        service_configuration=bedrock.CfnFlow.StorageFlowNodeServiceConfigurationProperty(
                            s3=bedrock.CfnFlow.StorageFlowNodeS3ConfigurationProperty(
                                bucket_name="bucketName"
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__425c904db23b9629fe6a830be94d214d112c1a6206b762478fd9e04b551b5552)
                check_type(argname="argument agent", value=agent, expected_type=type_hints["agent"])
                check_type(argname="argument collector", value=collector, expected_type=type_hints["collector"])
                check_type(argname="argument condition", value=condition, expected_type=type_hints["condition"])
                check_type(argname="argument input", value=input, expected_type=type_hints["input"])
                check_type(argname="argument iterator", value=iterator, expected_type=type_hints["iterator"])
                check_type(argname="argument knowledge_base", value=knowledge_base, expected_type=type_hints["knowledge_base"])
                check_type(argname="argument lambda_function", value=lambda_function, expected_type=type_hints["lambda_function"])
                check_type(argname="argument lex", value=lex, expected_type=type_hints["lex"])
                check_type(argname="argument output", value=output, expected_type=type_hints["output"])
                check_type(argname="argument prompt", value=prompt, expected_type=type_hints["prompt"])
                check_type(argname="argument retrieval", value=retrieval, expected_type=type_hints["retrieval"])
                check_type(argname="argument storage", value=storage, expected_type=type_hints["storage"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if agent is not None:
                self._values["agent"] = agent
            if collector is not None:
                self._values["collector"] = collector
            if condition is not None:
                self._values["condition"] = condition
            if input is not None:
                self._values["input"] = input
            if iterator is not None:
                self._values["iterator"] = iterator
            if knowledge_base is not None:
                self._values["knowledge_base"] = knowledge_base
            if lambda_function is not None:
                self._values["lambda_function"] = lambda_function
            if lex is not None:
                self._values["lex"] = lex
            if output is not None:
                self._values["output"] = output
            if prompt is not None:
                self._values["prompt"] = prompt
            if retrieval is not None:
                self._values["retrieval"] = retrieval
            if storage is not None:
                self._values["storage"] = storage

        @builtins.property
        def agent(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.AgentFlowNodeConfigurationProperty"]]:
            '''Contains configurations for an agent node in your flow.

            Invokes an alias of an agent and returns the response.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-agent
            '''
            result = self._values.get("agent")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.AgentFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def collector(self) -> typing.Any:
            '''Contains configurations for a collector node in your flow.

            Collects an iteration of inputs and consolidates them into an array of outputs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-collector
            '''
            result = self._values.get("collector")
            return typing.cast(typing.Any, result)

        @builtins.property
        def condition(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.ConditionFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a Condition node in your flow.

            Defines conditions that lead to different branches of the flow.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-condition
            '''
            result = self._values.get("condition")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.ConditionFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def input(self) -> typing.Any:
            '''Contains configurations for an input flow node in your flow.

            The first node in the flow. ``inputs`` can't be specified for this node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-input
            '''
            result = self._values.get("input")
            return typing.cast(typing.Any, result)

        @builtins.property
        def iterator(self) -> typing.Any:
            '''Contains configurations for an iterator node in your flow.

            Takes an input that is an array and iteratively sends each item of the array as an output to the following node. The size of the array is also returned in the output.

            The output flow node at the end of the flow iteration will return a response for each member of the array. To return only one response, you can include a collector node downstream from the iterator node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-iterator
            '''
            result = self._values.get("iterator")
            return typing.cast(typing.Any, result)

        @builtins.property
        def knowledge_base(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.KnowledgeBaseFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a knowledge base node in your flow.

            Queries a knowledge base and returns the retrieved results or generated response.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-knowledgebase
            '''
            result = self._values.get("knowledge_base")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.KnowledgeBaseFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def lambda_function(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.LambdaFunctionFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a Lambda function node in your flow.

            Invokes an AWS Lambda function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-lambdafunction
            '''
            result = self._values.get("lambda_function")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.LambdaFunctionFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def lex(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.LexFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a Lex node in your flow.

            Invokes an Amazon Lex bot to identify the intent of the input and return the intent as the output.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-lex
            '''
            result = self._values.get("lex")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.LexFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def output(self) -> typing.Any:
            '''Contains configurations for an output flow node in your flow.

            The last node in the flow. ``outputs`` can't be specified for this node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-output
            '''
            result = self._values.get("output")
            return typing.cast(typing.Any, result)

        @builtins.property
        def prompt(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a prompt node in your flow.

            Runs a prompt and generates the model response as the output. You can use a prompt from Prompt management or you can configure one in this node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-prompt
            '''
            result = self._values.get("prompt")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def retrieval(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.RetrievalFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a Retrieval node in your flow.

            Retrieves data from an Amazon S3 location and returns it as the output.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-retrieval
            '''
            result = self._values.get("retrieval")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.RetrievalFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def storage(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.StorageFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a Storage node in your flow.

            Stores an input in an Amazon S3 location.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeconfiguration.html#cfn-bedrock-flow-flownodeconfiguration-storage
            '''
            result = self._values.get("storage")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.StorageFlowNodeConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.FlowNodeInputProperty",
        jsii_struct_bases=[],
        name_mapping={"expression": "expression", "name": "name", "type": "type"},
    )
    class FlowNodeInputProperty:
        def __init__(
            self,
            *,
            expression: builtins.str,
            name: builtins.str,
            type: builtins.str,
        ) -> None:
            '''Contains configurations for an input to a node.

            :param expression: An expression that formats the input for the node. For an explanation of how to create expressions, see `Expressions in Prompt flows in Amazon Bedrock <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-expressions.html>`_ .
            :param name: A name for the input that you can reference.
            :param type: The data type of the input. If the input doesn't match this type at runtime, a validation error will be thrown.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeinput.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_node_input_property = bedrock.CfnFlow.FlowNodeInputProperty(
                    expression="expression",
                    name="name",
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bf2833359a5be50f82721be153d18c7eabc0e144d9d6ae37d71e8ac9149e58b4)
                check_type(argname="argument expression", value=expression, expected_type=type_hints["expression"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "expression": expression,
                "name": name,
                "type": type,
            }

        @builtins.property
        def expression(self) -> builtins.str:
            '''An expression that formats the input for the node.

            For an explanation of how to create expressions, see `Expressions in Prompt flows in Amazon Bedrock <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-expressions.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeinput.html#cfn-bedrock-flow-flownodeinput-expression
            '''
            result = self._values.get("expression")
            assert result is not None, "Required property 'expression' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the input that you can reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeinput.html#cfn-bedrock-flow-flownodeinput-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def type(self) -> builtins.str:
            '''The data type of the input.

            If the input doesn't match this type at runtime, a validation error will be thrown.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeinput.html#cfn-bedrock-flow-flownodeinput-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowNodeInputProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.FlowNodeOutputProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "type": "type"},
    )
    class FlowNodeOutputProperty:
        def __init__(self, *, name: builtins.str, type: builtins.str) -> None:
            '''Contains configurations for an output from a node.

            :param name: A name for the output that you can reference.
            :param type: The data type of the output. If the output doesn't match this type at runtime, a validation error will be thrown.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeoutput.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_node_output_property = bedrock.CfnFlow.FlowNodeOutputProperty(
                    name="name",
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6a42d5fb1bc23654d5c7ab05f6e3297ae51d24a2ee5a30e7c6d4d16333bf248d)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "type": type,
            }

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the output that you can reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeoutput.html#cfn-bedrock-flow-flownodeoutput-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def type(self) -> builtins.str:
            '''The data type of the output.

            If the output doesn't match this type at runtime, a validation error will be thrown.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownodeoutput.html#cfn-bedrock-flow-flownodeoutput-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowNodeOutputProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.FlowNodeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "type": "type",
            "configuration": "configuration",
            "inputs": "inputs",
            "outputs": "outputs",
        },
    )
    class FlowNodeProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            type: builtins.str,
            configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.FlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            inputs: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.FlowNodeInputProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            outputs: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.FlowNodeOutputProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''Contains configurations about a node in the flow.

            :param name: A name for the node.
            :param type: The type of node. This value must match the name of the key that you provide in the configuration you provide in the ``FlowNodeConfiguration`` field.
            :param configuration: Contains configurations for the node.
            :param inputs: An array of objects, each of which contains information about an input into the node.
            :param outputs: A list of objects, each of which contains information about an output from the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownode.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                # collector: Any
                # input: Any
                # iterator: Any
                # output: Any
                
                flow_node_property = bedrock.CfnFlow.FlowNodeProperty(
                    name="name",
                    type="type",
                
                    # the properties below are optional
                    configuration=bedrock.CfnFlow.FlowNodeConfigurationProperty(
                        agent=bedrock.CfnFlow.AgentFlowNodeConfigurationProperty(
                            agent_alias_arn="agentAliasArn"
                        ),
                        collector=collector,
                        condition=bedrock.CfnFlow.ConditionFlowNodeConfigurationProperty(
                            conditions=[bedrock.CfnFlow.FlowConditionProperty(
                                name="name",
                
                                # the properties below are optional
                                expression="expression"
                            )]
                        ),
                        input=input,
                        iterator=iterator,
                        knowledge_base=bedrock.CfnFlow.KnowledgeBaseFlowNodeConfigurationProperty(
                            knowledge_base_id="knowledgeBaseId",
                
                            # the properties below are optional
                            guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                                guardrail_identifier="guardrailIdentifier",
                                guardrail_version="guardrailVersion"
                            ),
                            model_id="modelId"
                        ),
                        lambda_function=bedrock.CfnFlow.LambdaFunctionFlowNodeConfigurationProperty(
                            lambda_arn="lambdaArn"
                        ),
                        lex=bedrock.CfnFlow.LexFlowNodeConfigurationProperty(
                            bot_alias_arn="botAliasArn",
                            locale_id="localeId"
                        ),
                        output=output,
                        prompt=bedrock.CfnFlow.PromptFlowNodeConfigurationProperty(
                            source_configuration=bedrock.CfnFlow.PromptFlowNodeSourceConfigurationProperty(
                                inline=bedrock.CfnFlow.PromptFlowNodeInlineConfigurationProperty(
                                    model_id="modelId",
                                    template_configuration=bedrock.CfnFlow.PromptTemplateConfigurationProperty(
                                        text=bedrock.CfnFlow.TextPromptTemplateConfigurationProperty(
                                            text="text",
                
                                            # the properties below are optional
                                            input_variables=[bedrock.CfnFlow.PromptInputVariableProperty(
                                                name="name"
                                            )]
                                        )
                                    ),
                                    template_type="templateType",
                
                                    # the properties below are optional
                                    inference_configuration=bedrock.CfnFlow.PromptInferenceConfigurationProperty(
                                        text=bedrock.CfnFlow.PromptModelInferenceConfigurationProperty(
                                            max_tokens=123,
                                            stop_sequences=["stopSequences"],
                                            temperature=123,
                                            top_p=123
                                        )
                                    )
                                ),
                                resource=bedrock.CfnFlow.PromptFlowNodeResourceConfigurationProperty(
                                    prompt_arn="promptArn"
                                )
                            ),
                
                            # the properties below are optional
                            guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                                guardrail_identifier="guardrailIdentifier",
                                guardrail_version="guardrailVersion"
                            )
                        ),
                        retrieval=bedrock.CfnFlow.RetrievalFlowNodeConfigurationProperty(
                            service_configuration=bedrock.CfnFlow.RetrievalFlowNodeServiceConfigurationProperty(
                                s3=bedrock.CfnFlow.RetrievalFlowNodeS3ConfigurationProperty(
                                    bucket_name="bucketName"
                                )
                            )
                        ),
                        storage=bedrock.CfnFlow.StorageFlowNodeConfigurationProperty(
                            service_configuration=bedrock.CfnFlow.StorageFlowNodeServiceConfigurationProperty(
                                s3=bedrock.CfnFlow.StorageFlowNodeS3ConfigurationProperty(
                                    bucket_name="bucketName"
                                )
                            )
                        )
                    ),
                    inputs=[bedrock.CfnFlow.FlowNodeInputProperty(
                        expression="expression",
                        name="name",
                        type="type"
                    )],
                    outputs=[bedrock.CfnFlow.FlowNodeOutputProperty(
                        name="name",
                        type="type"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3918fc30aa5dc9c688e4ea9c5bf2fb45419e4becfd6e3eccdcd68e52cd947956)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument configuration", value=configuration, expected_type=type_hints["configuration"])
                check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
                check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "type": type,
            }
            if configuration is not None:
                self._values["configuration"] = configuration
            if inputs is not None:
                self._values["inputs"] = inputs
            if outputs is not None:
                self._values["outputs"] = outputs

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownode.html#cfn-bedrock-flow-flownode-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of node.

            This value must match the name of the key that you provide in the configuration you provide in the ``FlowNodeConfiguration`` field.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownode.html#cfn-bedrock-flow-flownode-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowNodeConfigurationProperty"]]:
            '''Contains configurations for the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownode.html#cfn-bedrock-flow-flownode-configuration
            '''
            result = self._values.get("configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowNodeConfigurationProperty"]], result)

        @builtins.property
        def inputs(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowNodeInputProperty"]]]]:
            '''An array of objects, each of which contains information about an input into the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownode.html#cfn-bedrock-flow-flownode-inputs
            '''
            result = self._values.get("inputs")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowNodeInputProperty"]]]], result)

        @builtins.property
        def outputs(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowNodeOutputProperty"]]]]:
            '''A list of objects, each of which contains information about an output from the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flownode.html#cfn-bedrock-flow-flownode-outputs
            '''
            result = self._values.get("outputs")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.FlowNodeOutputProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowNodeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.FlowValidationProperty",
        jsii_struct_bases=[],
        name_mapping={"message": "message"},
    )
    class FlowValidationProperty:
        def __init__(self, *, message: builtins.str) -> None:
            '''Contains information about validation of the flow.

            This data type is used in the following API operations:

            - `GetFlow response <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetFlow.html#API_agent_GetFlow_ResponseSyntax>`_
            - `GetFlowVersion response <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetFlowVersion.html#API_agent_GetFlowVersion_ResponseSyntax>`_

            :param message: A message describing the validation error.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowvalidation.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_validation_property = bedrock.CfnFlow.FlowValidationProperty(
                    message="message"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3eab89f770734deb893367e044b113c86911377a918b039f1ba9310a34002e30)
                check_type(argname="argument message", value=message, expected_type=type_hints["message"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "message": message,
            }

        @builtins.property
        def message(self) -> builtins.str:
            '''A message describing the validation error.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowvalidation.html#cfn-bedrock-flow-flowvalidation-message
            '''
            result = self._values.get("message")
            assert result is not None, "Required property 'message' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowValidationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.GuardrailConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "guardrail_identifier": "guardrailIdentifier",
            "guardrail_version": "guardrailVersion",
        },
    )
    class GuardrailConfigurationProperty:
        def __init__(
            self,
            *,
            guardrail_identifier: typing.Optional[builtins.str] = None,
            guardrail_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Configuration information for a guardrail that you use with the `Converse <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_Converse.html>`_ operation.

            :param guardrail_identifier: The identifier for the guardrail.
            :param guardrail_version: The version of the guardrail.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-guardrailconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                guardrail_configuration_property = bedrock.CfnFlow.GuardrailConfigurationProperty(
                    guardrail_identifier="guardrailIdentifier",
                    guardrail_version="guardrailVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7a58b1756431d0127fbde4a9b98f71b8bd861fde50f58d8b27f6621481f6898a)
                check_type(argname="argument guardrail_identifier", value=guardrail_identifier, expected_type=type_hints["guardrail_identifier"])
                check_type(argname="argument guardrail_version", value=guardrail_version, expected_type=type_hints["guardrail_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if guardrail_identifier is not None:
                self._values["guardrail_identifier"] = guardrail_identifier
            if guardrail_version is not None:
                self._values["guardrail_version"] = guardrail_version

        @builtins.property
        def guardrail_identifier(self) -> typing.Optional[builtins.str]:
            '''The identifier for the guardrail.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-guardrailconfiguration.html#cfn-bedrock-flow-guardrailconfiguration-guardrailidentifier
            '''
            result = self._values.get("guardrail_identifier")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def guardrail_version(self) -> typing.Optional[builtins.str]:
            '''The version of the guardrail.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-guardrailconfiguration.html#cfn-bedrock-flow-guardrailconfiguration-guardrailversion
            '''
            result = self._values.get("guardrail_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GuardrailConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.KnowledgeBaseFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "knowledge_base_id": "knowledgeBaseId",
            "guardrail_configuration": "guardrailConfiguration",
            "model_id": "modelId",
        },
    )
    class KnowledgeBaseFlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            knowledge_base_id: builtins.str,
            guardrail_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.GuardrailConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            model_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains configurations for a knowledge base node in a flow.

            This node takes a query as the input and returns, as the output, the retrieved responses directly (as an array) or a response generated based on the retrieved responses. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param knowledge_base_id: The unique identifier of the knowledge base to query.
            :param guardrail_configuration: Contains configurations for a guardrail to apply during query and response generation for the knowledge base in this configuration.
            :param model_id: The unique identifier of the model or `inference profile <https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html>`_ to use to generate a response from the query results. Omit this field if you want to return the retrieved results as an array.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-knowledgebaseflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                knowledge_base_flow_node_configuration_property = bedrock.CfnFlow.KnowledgeBaseFlowNodeConfigurationProperty(
                    knowledge_base_id="knowledgeBaseId",
                
                    # the properties below are optional
                    guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                        guardrail_identifier="guardrailIdentifier",
                        guardrail_version="guardrailVersion"
                    ),
                    model_id="modelId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__004ed148a98b01e41f3efd077553224d0e5ce97fd3842fc0c6be6c28accdc6cc)
                check_type(argname="argument knowledge_base_id", value=knowledge_base_id, expected_type=type_hints["knowledge_base_id"])
                check_type(argname="argument guardrail_configuration", value=guardrail_configuration, expected_type=type_hints["guardrail_configuration"])
                check_type(argname="argument model_id", value=model_id, expected_type=type_hints["model_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "knowledge_base_id": knowledge_base_id,
            }
            if guardrail_configuration is not None:
                self._values["guardrail_configuration"] = guardrail_configuration
            if model_id is not None:
                self._values["model_id"] = model_id

        @builtins.property
        def knowledge_base_id(self) -> builtins.str:
            '''The unique identifier of the knowledge base to query.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-knowledgebaseflownodeconfiguration.html#cfn-bedrock-flow-knowledgebaseflownodeconfiguration-knowledgebaseid
            '''
            result = self._values.get("knowledge_base_id")
            assert result is not None, "Required property 'knowledge_base_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def guardrail_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.GuardrailConfigurationProperty"]]:
            '''Contains configurations for a guardrail to apply during query and response generation for the knowledge base in this configuration.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-knowledgebaseflownodeconfiguration.html#cfn-bedrock-flow-knowledgebaseflownodeconfiguration-guardrailconfiguration
            '''
            result = self._values.get("guardrail_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.GuardrailConfigurationProperty"]], result)

        @builtins.property
        def model_id(self) -> typing.Optional[builtins.str]:
            '''The unique identifier of the model or `inference profile <https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html>`_ to use to generate a response from the query results. Omit this field if you want to return the retrieved results as an array.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-knowledgebaseflownodeconfiguration.html#cfn-bedrock-flow-knowledgebaseflownodeconfiguration-modelid
            '''
            result = self._values.get("model_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KnowledgeBaseFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.LambdaFunctionFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"lambda_arn": "lambdaArn"},
    )
    class LambdaFunctionFlowNodeConfigurationProperty:
        def __init__(self, *, lambda_arn: builtins.str) -> None:
            '''Contains configurations for a Lambda function node in the flow.

            You specify the Lambda function to invoke and the inputs into the function. The output is the response that is defined in the Lambda function. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param lambda_arn: The Amazon Resource Name (ARN) of the Lambda function to invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-lambdafunctionflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                lambda_function_flow_node_configuration_property = bedrock.CfnFlow.LambdaFunctionFlowNodeConfigurationProperty(
                    lambda_arn="lambdaArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ed2c72c33545f76b09a7350b616fb67997e91bc9181a09b82cd7f28d854f2a5d)
                check_type(argname="argument lambda_arn", value=lambda_arn, expected_type=type_hints["lambda_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "lambda_arn": lambda_arn,
            }

        @builtins.property
        def lambda_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the Lambda function to invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-lambdafunctionflownodeconfiguration.html#cfn-bedrock-flow-lambdafunctionflownodeconfiguration-lambdaarn
            '''
            result = self._values.get("lambda_arn")
            assert result is not None, "Required property 'lambda_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LambdaFunctionFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.LexFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"bot_alias_arn": "botAliasArn", "locale_id": "localeId"},
    )
    class LexFlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            bot_alias_arn: builtins.str,
            locale_id: builtins.str,
        ) -> None:
            '''Contains configurations for a Lex node in the flow.

            You specify a Amazon Lex bot to invoke. This node takes an utterance as the input and returns as the output the intent identified by the Amazon Lex bot. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param bot_alias_arn: The Amazon Resource Name (ARN) of the Amazon Lex bot alias to invoke.
            :param locale_id: The Region to invoke the Amazon Lex bot in.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-lexflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                lex_flow_node_configuration_property = bedrock.CfnFlow.LexFlowNodeConfigurationProperty(
                    bot_alias_arn="botAliasArn",
                    locale_id="localeId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__38478491ab5facb49d25af7399145bf3032f8b76fc619392d4da744bd32b892c)
                check_type(argname="argument bot_alias_arn", value=bot_alias_arn, expected_type=type_hints["bot_alias_arn"])
                check_type(argname="argument locale_id", value=locale_id, expected_type=type_hints["locale_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "bot_alias_arn": bot_alias_arn,
                "locale_id": locale_id,
            }

        @builtins.property
        def bot_alias_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the Amazon Lex bot alias to invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-lexflownodeconfiguration.html#cfn-bedrock-flow-lexflownodeconfiguration-botaliasarn
            '''
            result = self._values.get("bot_alias_arn")
            assert result is not None, "Required property 'bot_alias_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def locale_id(self) -> builtins.str:
            '''The Region to invoke the Amazon Lex bot in.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-lexflownodeconfiguration.html#cfn-bedrock-flow-lexflownodeconfiguration-localeid
            '''
            result = self._values.get("locale_id")
            assert result is not None, "Required property 'locale_id' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LexFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.PromptFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "source_configuration": "sourceConfiguration",
            "guardrail_configuration": "guardrailConfiguration",
        },
    )
    class PromptFlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            source_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.PromptFlowNodeSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
            guardrail_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.GuardrailConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains configurations for a prompt node in the flow.

            You can use a prompt from Prompt management or you can define one in this node. If the prompt contains variables, the inputs into this node will fill in the variables. The output from this node is the response generated by the model. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param source_configuration: Specifies whether the prompt is from Prompt management or defined inline.
            :param guardrail_configuration: Contains configurations for a guardrail to apply to the prompt in this node and the response generated from it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_flow_node_configuration_property = bedrock.CfnFlow.PromptFlowNodeConfigurationProperty(
                    source_configuration=bedrock.CfnFlow.PromptFlowNodeSourceConfigurationProperty(
                        inline=bedrock.CfnFlow.PromptFlowNodeInlineConfigurationProperty(
                            model_id="modelId",
                            template_configuration=bedrock.CfnFlow.PromptTemplateConfigurationProperty(
                                text=bedrock.CfnFlow.TextPromptTemplateConfigurationProperty(
                                    text="text",
                
                                    # the properties below are optional
                                    input_variables=[bedrock.CfnFlow.PromptInputVariableProperty(
                                        name="name"
                                    )]
                                )
                            ),
                            template_type="templateType",
                
                            # the properties below are optional
                            inference_configuration=bedrock.CfnFlow.PromptInferenceConfigurationProperty(
                                text=bedrock.CfnFlow.PromptModelInferenceConfigurationProperty(
                                    max_tokens=123,
                                    stop_sequences=["stopSequences"],
                                    temperature=123,
                                    top_p=123
                                )
                            )
                        ),
                        resource=bedrock.CfnFlow.PromptFlowNodeResourceConfigurationProperty(
                            prompt_arn="promptArn"
                        )
                    ),
                
                    # the properties below are optional
                    guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                        guardrail_identifier="guardrailIdentifier",
                        guardrail_version="guardrailVersion"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b9e2aaf75cba4595f32ad6457fae055981f8fb7d76d181537ab0d3241fa7866d)
                check_type(argname="argument source_configuration", value=source_configuration, expected_type=type_hints["source_configuration"])
                check_type(argname="argument guardrail_configuration", value=guardrail_configuration, expected_type=type_hints["guardrail_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "source_configuration": source_configuration,
            }
            if guardrail_configuration is not None:
                self._values["guardrail_configuration"] = guardrail_configuration

        @builtins.property
        def source_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptFlowNodeSourceConfigurationProperty"]:
            '''Specifies whether the prompt is from Prompt management or defined inline.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownodeconfiguration.html#cfn-bedrock-flow-promptflownodeconfiguration-sourceconfiguration
            '''
            result = self._values.get("source_configuration")
            assert result is not None, "Required property 'source_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptFlowNodeSourceConfigurationProperty"], result)

        @builtins.property
        def guardrail_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.GuardrailConfigurationProperty"]]:
            '''Contains configurations for a guardrail to apply to the prompt in this node and the response generated from it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownodeconfiguration.html#cfn-bedrock-flow-promptflownodeconfiguration-guardrailconfiguration
            '''
            result = self._values.get("guardrail_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.GuardrailConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.PromptFlowNodeInlineConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "model_id": "modelId",
            "template_configuration": "templateConfiguration",
            "template_type": "templateType",
            "inference_configuration": "inferenceConfiguration",
        },
    )
    class PromptFlowNodeInlineConfigurationProperty:
        def __init__(
            self,
            *,
            model_id: builtins.str,
            template_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.PromptTemplateConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
            template_type: builtins.str,
            inference_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.PromptInferenceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains configurations for a prompt defined inline in the node.

            :param model_id: The unique identifier of the model or `inference profile <https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html>`_ to run inference with.
            :param template_configuration: Contains a prompt and variables in the prompt that can be replaced with values at runtime.
            :param template_type: The type of prompt template.
            :param inference_configuration: Contains inference configurations for the prompt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownodeinlineconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_flow_node_inline_configuration_property = bedrock.CfnFlow.PromptFlowNodeInlineConfigurationProperty(
                    model_id="modelId",
                    template_configuration=bedrock.CfnFlow.PromptTemplateConfigurationProperty(
                        text=bedrock.CfnFlow.TextPromptTemplateConfigurationProperty(
                            text="text",
                
                            # the properties below are optional
                            input_variables=[bedrock.CfnFlow.PromptInputVariableProperty(
                                name="name"
                            )]
                        )
                    ),
                    template_type="templateType",
                
                    # the properties below are optional
                    inference_configuration=bedrock.CfnFlow.PromptInferenceConfigurationProperty(
                        text=bedrock.CfnFlow.PromptModelInferenceConfigurationProperty(
                            max_tokens=123,
                            stop_sequences=["stopSequences"],
                            temperature=123,
                            top_p=123
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cbd396b4a7dbcb87ab190e61d2fd875106d30c5baccadb4bcd3931ab29fe395d)
                check_type(argname="argument model_id", value=model_id, expected_type=type_hints["model_id"])
                check_type(argname="argument template_configuration", value=template_configuration, expected_type=type_hints["template_configuration"])
                check_type(argname="argument template_type", value=template_type, expected_type=type_hints["template_type"])
                check_type(argname="argument inference_configuration", value=inference_configuration, expected_type=type_hints["inference_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "model_id": model_id,
                "template_configuration": template_configuration,
                "template_type": template_type,
            }
            if inference_configuration is not None:
                self._values["inference_configuration"] = inference_configuration

        @builtins.property
        def model_id(self) -> builtins.str:
            '''The unique identifier of the model or `inference profile <https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html>`_ to run inference with.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownodeinlineconfiguration.html#cfn-bedrock-flow-promptflownodeinlineconfiguration-modelid
            '''
            result = self._values.get("model_id")
            assert result is not None, "Required property 'model_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def template_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptTemplateConfigurationProperty"]:
            '''Contains a prompt and variables in the prompt that can be replaced with values at runtime.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownodeinlineconfiguration.html#cfn-bedrock-flow-promptflownodeinlineconfiguration-templateconfiguration
            '''
            result = self._values.get("template_configuration")
            assert result is not None, "Required property 'template_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptTemplateConfigurationProperty"], result)

        @builtins.property
        def template_type(self) -> builtins.str:
            '''The type of prompt template.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownodeinlineconfiguration.html#cfn-bedrock-flow-promptflownodeinlineconfiguration-templatetype
            '''
            result = self._values.get("template_type")
            assert result is not None, "Required property 'template_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def inference_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptInferenceConfigurationProperty"]]:
            '''Contains inference configurations for the prompt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownodeinlineconfiguration.html#cfn-bedrock-flow-promptflownodeinlineconfiguration-inferenceconfiguration
            '''
            result = self._values.get("inference_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptInferenceConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptFlowNodeInlineConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.PromptFlowNodeResourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"prompt_arn": "promptArn"},
    )
    class PromptFlowNodeResourceConfigurationProperty:
        def __init__(self, *, prompt_arn: builtins.str) -> None:
            '''Contains configurations for a prompt from Prompt management to use in a node.

            :param prompt_arn: The Amazon Resource Name (ARN) of the prompt from Prompt management.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownoderesourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_flow_node_resource_configuration_property = bedrock.CfnFlow.PromptFlowNodeResourceConfigurationProperty(
                    prompt_arn="promptArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__54146e948ca7adfd068de7abe2312fe9312441f945acbc188b7dd2855d0a5470)
                check_type(argname="argument prompt_arn", value=prompt_arn, expected_type=type_hints["prompt_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "prompt_arn": prompt_arn,
            }

        @builtins.property
        def prompt_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the prompt from Prompt management.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownoderesourceconfiguration.html#cfn-bedrock-flow-promptflownoderesourceconfiguration-promptarn
            '''
            result = self._values.get("prompt_arn")
            assert result is not None, "Required property 'prompt_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptFlowNodeResourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.PromptFlowNodeSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"inline": "inline", "resource": "resource"},
    )
    class PromptFlowNodeSourceConfigurationProperty:
        def __init__(
            self,
            *,
            inline: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.PromptFlowNodeInlineConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            resource: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.PromptFlowNodeResourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains configurations for a prompt and whether it is from Prompt management or defined inline.

            :param inline: Contains configurations for a prompt that is defined inline.
            :param resource: Contains configurations for a prompt from Prompt management.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownodesourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_flow_node_source_configuration_property = bedrock.CfnFlow.PromptFlowNodeSourceConfigurationProperty(
                    inline=bedrock.CfnFlow.PromptFlowNodeInlineConfigurationProperty(
                        model_id="modelId",
                        template_configuration=bedrock.CfnFlow.PromptTemplateConfigurationProperty(
                            text=bedrock.CfnFlow.TextPromptTemplateConfigurationProperty(
                                text="text",
                
                                # the properties below are optional
                                input_variables=[bedrock.CfnFlow.PromptInputVariableProperty(
                                    name="name"
                                )]
                            )
                        ),
                        template_type="templateType",
                
                        # the properties below are optional
                        inference_configuration=bedrock.CfnFlow.PromptInferenceConfigurationProperty(
                            text=bedrock.CfnFlow.PromptModelInferenceConfigurationProperty(
                                max_tokens=123,
                                stop_sequences=["stopSequences"],
                                temperature=123,
                                top_p=123
                            )
                        )
                    ),
                    resource=bedrock.CfnFlow.PromptFlowNodeResourceConfigurationProperty(
                        prompt_arn="promptArn"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9cb538a155ab7131377f6778a9b366ad2df378a4c09b6c89aa50ea0f682819c6)
                check_type(argname="argument inline", value=inline, expected_type=type_hints["inline"])
                check_type(argname="argument resource", value=resource, expected_type=type_hints["resource"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if inline is not None:
                self._values["inline"] = inline
            if resource is not None:
                self._values["resource"] = resource

        @builtins.property
        def inline(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptFlowNodeInlineConfigurationProperty"]]:
            '''Contains configurations for a prompt that is defined inline.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownodesourceconfiguration.html#cfn-bedrock-flow-promptflownodesourceconfiguration-inline
            '''
            result = self._values.get("inline")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptFlowNodeInlineConfigurationProperty"]], result)

        @builtins.property
        def resource(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptFlowNodeResourceConfigurationProperty"]]:
            '''Contains configurations for a prompt from Prompt management.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptflownodesourceconfiguration.html#cfn-bedrock-flow-promptflownodesourceconfiguration-resource
            '''
            result = self._values.get("resource")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptFlowNodeResourceConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptFlowNodeSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.PromptInferenceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"text": "text"},
    )
    class PromptInferenceConfigurationProperty:
        def __init__(
            self,
            *,
            text: typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.PromptModelInferenceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''Contains inference configurations for the prompt.

            :param text: Contains inference configurations for a text prompt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptinferenceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_inference_configuration_property = bedrock.CfnFlow.PromptInferenceConfigurationProperty(
                    text=bedrock.CfnFlow.PromptModelInferenceConfigurationProperty(
                        max_tokens=123,
                        stop_sequences=["stopSequences"],
                        temperature=123,
                        top_p=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__324145c7c75654c12471d7222cc7ce7a0307899eb880d494aff9a791c82d6c7e)
                check_type(argname="argument text", value=text, expected_type=type_hints["text"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "text": text,
            }

        @builtins.property
        def text(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptModelInferenceConfigurationProperty"]:
            '''Contains inference configurations for a text prompt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptinferenceconfiguration.html#cfn-bedrock-flow-promptinferenceconfiguration-text
            '''
            result = self._values.get("text")
            assert result is not None, "Required property 'text' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptModelInferenceConfigurationProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptInferenceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.PromptInputVariableProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name"},
    )
    class PromptInputVariableProperty:
        def __init__(self, *, name: typing.Optional[builtins.str] = None) -> None:
            '''Contains information about a variable in the prompt.

            :param name: The name of the variable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptinputvariable.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_input_variable_property = bedrock.CfnFlow.PromptInputVariableProperty(
                    name="name"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__611a434465223767b53e3e034a0be3f6957e85644e42e58f99d4ebf6bf2e6e75)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the variable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptinputvariable.html#cfn-bedrock-flow-promptinputvariable-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptInputVariableProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.PromptModelInferenceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_tokens": "maxTokens",
            "stop_sequences": "stopSequences",
            "temperature": "temperature",
            "top_p": "topP",
        },
    )
    class PromptModelInferenceConfigurationProperty:
        def __init__(
            self,
            *,
            max_tokens: typing.Optional[jsii.Number] = None,
            stop_sequences: typing.Optional[typing.Sequence[builtins.str]] = None,
            temperature: typing.Optional[jsii.Number] = None,
            top_p: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Contains inference configurations related to model inference for a prompt.

            For more information, see `Inference parameters <https://docs.aws.amazon.com/bedrock/latest/userguide/inference-parameters.html>`_ .

            :param max_tokens: The maximum number of tokens to return in the response.
            :param stop_sequences: A list of strings that define sequences after which the model will stop generating.
            :param temperature: Controls the randomness of the response. Choose a lower value for more predictable outputs and a higher value for more surprising outputs.
            :param top_p: The percentage of most-likely candidates that the model considers for the next token.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptmodelinferenceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_model_inference_configuration_property = bedrock.CfnFlow.PromptModelInferenceConfigurationProperty(
                    max_tokens=123,
                    stop_sequences=["stopSequences"],
                    temperature=123,
                    top_p=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4b72bde5473c072da139578869709761ca3585ad0bf7391b769225231e7ef45b)
                check_type(argname="argument max_tokens", value=max_tokens, expected_type=type_hints["max_tokens"])
                check_type(argname="argument stop_sequences", value=stop_sequences, expected_type=type_hints["stop_sequences"])
                check_type(argname="argument temperature", value=temperature, expected_type=type_hints["temperature"])
                check_type(argname="argument top_p", value=top_p, expected_type=type_hints["top_p"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if max_tokens is not None:
                self._values["max_tokens"] = max_tokens
            if stop_sequences is not None:
                self._values["stop_sequences"] = stop_sequences
            if temperature is not None:
                self._values["temperature"] = temperature
            if top_p is not None:
                self._values["top_p"] = top_p

        @builtins.property
        def max_tokens(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of tokens to return in the response.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptmodelinferenceconfiguration.html#cfn-bedrock-flow-promptmodelinferenceconfiguration-maxtokens
            '''
            result = self._values.get("max_tokens")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def stop_sequences(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of strings that define sequences after which the model will stop generating.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptmodelinferenceconfiguration.html#cfn-bedrock-flow-promptmodelinferenceconfiguration-stopsequences
            '''
            result = self._values.get("stop_sequences")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def temperature(self) -> typing.Optional[jsii.Number]:
            '''Controls the randomness of the response.

            Choose a lower value for more predictable outputs and a higher value for more surprising outputs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptmodelinferenceconfiguration.html#cfn-bedrock-flow-promptmodelinferenceconfiguration-temperature
            '''
            result = self._values.get("temperature")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def top_p(self) -> typing.Optional[jsii.Number]:
            '''The percentage of most-likely candidates that the model considers for the next token.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-promptmodelinferenceconfiguration.html#cfn-bedrock-flow-promptmodelinferenceconfiguration-topp
            '''
            result = self._values.get("top_p")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptModelInferenceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.PromptTemplateConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"text": "text"},
    )
    class PromptTemplateConfigurationProperty:
        def __init__(
            self,
            *,
            text: typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.TextPromptTemplateConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''Contains the message for a prompt.

            For more information, see `Construct and store reusable prompts with Prompt management in Amazon Bedrock <https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-management.html>`_ .

            :param text: Contains configurations for the text in a message for a prompt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-prompttemplateconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_template_configuration_property = bedrock.CfnFlow.PromptTemplateConfigurationProperty(
                    text=bedrock.CfnFlow.TextPromptTemplateConfigurationProperty(
                        text="text",
                
                        # the properties below are optional
                        input_variables=[bedrock.CfnFlow.PromptInputVariableProperty(
                            name="name"
                        )]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5bf1f973d583ea4ef1d4bc5b39d3ee1b4725beaa6d0ec7ffb982b431528220bc)
                check_type(argname="argument text", value=text, expected_type=type_hints["text"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "text": text,
            }

        @builtins.property
        def text(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnFlow.TextPromptTemplateConfigurationProperty"]:
            '''Contains configurations for the text in a message for a prompt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-prompttemplateconfiguration.html#cfn-bedrock-flow-prompttemplateconfiguration-text
            '''
            result = self._values.get("text")
            assert result is not None, "Required property 'text' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnFlow.TextPromptTemplateConfigurationProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptTemplateConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.RetrievalFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"service_configuration": "serviceConfiguration"},
    )
    class RetrievalFlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            service_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.RetrievalFlowNodeServiceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''Contains configurations for a Retrieval node in a flow.

            This node retrieves data from the Amazon S3 location that you specify and returns it as the output.

            :param service_configuration: Contains configurations for the service to use for retrieving data to return as the output from the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-retrievalflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                retrieval_flow_node_configuration_property = bedrock.CfnFlow.RetrievalFlowNodeConfigurationProperty(
                    service_configuration=bedrock.CfnFlow.RetrievalFlowNodeServiceConfigurationProperty(
                        s3=bedrock.CfnFlow.RetrievalFlowNodeS3ConfigurationProperty(
                            bucket_name="bucketName"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__982db19983467a9d8b437235c39ed6a0fbf0e08458563d3e8f78d9a7de277861)
                check_type(argname="argument service_configuration", value=service_configuration, expected_type=type_hints["service_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "service_configuration": service_configuration,
            }

        @builtins.property
        def service_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnFlow.RetrievalFlowNodeServiceConfigurationProperty"]:
            '''Contains configurations for the service to use for retrieving data to return as the output from the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-retrievalflownodeconfiguration.html#cfn-bedrock-flow-retrievalflownodeconfiguration-serviceconfiguration
            '''
            result = self._values.get("service_configuration")
            assert result is not None, "Required property 'service_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnFlow.RetrievalFlowNodeServiceConfigurationProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RetrievalFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.RetrievalFlowNodeS3ConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"bucket_name": "bucketName"},
    )
    class RetrievalFlowNodeS3ConfigurationProperty:
        def __init__(self, *, bucket_name: builtins.str) -> None:
            '''Contains configurations for the Amazon S3 location from which to retrieve data to return as the output from the node.

            :param bucket_name: The name of the Amazon S3 bucket from which to retrieve data.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-retrievalflownodes3configuration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                retrieval_flow_node_s3_configuration_property = bedrock.CfnFlow.RetrievalFlowNodeS3ConfigurationProperty(
                    bucket_name="bucketName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cc1fc39971930d3b8f21c48c16cc46f7b6dcac42391342dc2c0b08f3ab2dd6c9)
                check_type(argname="argument bucket_name", value=bucket_name, expected_type=type_hints["bucket_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "bucket_name": bucket_name,
            }

        @builtins.property
        def bucket_name(self) -> builtins.str:
            '''The name of the Amazon S3 bucket from which to retrieve data.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-retrievalflownodes3configuration.html#cfn-bedrock-flow-retrievalflownodes3configuration-bucketname
            '''
            result = self._values.get("bucket_name")
            assert result is not None, "Required property 'bucket_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RetrievalFlowNodeS3ConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.RetrievalFlowNodeServiceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"s3": "s3"},
    )
    class RetrievalFlowNodeServiceConfigurationProperty:
        def __init__(
            self,
            *,
            s3: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.RetrievalFlowNodeS3ConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains configurations for the service to use for retrieving data to return as the output from the node.

            :param s3: Contains configurations for the Amazon S3 location from which to retrieve data to return as the output from the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-retrievalflownodeserviceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                retrieval_flow_node_service_configuration_property = bedrock.CfnFlow.RetrievalFlowNodeServiceConfigurationProperty(
                    s3=bedrock.CfnFlow.RetrievalFlowNodeS3ConfigurationProperty(
                        bucket_name="bucketName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__71787bd8165076f8853d3c5e8ccf1f76a3ecc1dff4ad49e09330cc7c5fd42328)
                check_type(argname="argument s3", value=s3, expected_type=type_hints["s3"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if s3 is not None:
                self._values["s3"] = s3

        @builtins.property
        def s3(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.RetrievalFlowNodeS3ConfigurationProperty"]]:
            '''Contains configurations for the Amazon S3 location from which to retrieve data to return as the output from the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-retrievalflownodeserviceconfiguration.html#cfn-bedrock-flow-retrievalflownodeserviceconfiguration-s3
            '''
            result = self._values.get("s3")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.RetrievalFlowNodeS3ConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RetrievalFlowNodeServiceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.S3LocationProperty",
        jsii_struct_bases=[],
        name_mapping={"bucket": "bucket", "key": "key", "version": "version"},
    )
    class S3LocationProperty:
        def __init__(
            self,
            *,
            bucket: builtins.str,
            key: builtins.str,
            version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The S3 location of the flow definition.

            :param bucket: The S3 bucket containing the flow definition.
            :param key: The object key for the S3 location containing the definition.
            :param version: The Amazon S3 location from which to retrieve data for an S3 retrieve node or to which to store data for an S3 storage node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-s3location.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                s3_location_property = bedrock.CfnFlow.S3LocationProperty(
                    bucket="bucket",
                    key="key",
                
                    # the properties below are optional
                    version="version"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__05457863dfccc1ba82410713caf4f4f5196707f3818d55f3234a1ee92a836f3a)
                check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "bucket": bucket,
                "key": key,
            }
            if version is not None:
                self._values["version"] = version

        @builtins.property
        def bucket(self) -> builtins.str:
            '''The S3 bucket containing the flow definition.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-s3location.html#cfn-bedrock-flow-s3location-bucket
            '''
            result = self._values.get("bucket")
            assert result is not None, "Required property 'bucket' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def key(self) -> builtins.str:
            '''The object key for the S3 location containing the definition.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-s3location.html#cfn-bedrock-flow-s3location-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def version(self) -> typing.Optional[builtins.str]:
            '''The Amazon S3 location from which to retrieve data for an S3 retrieve node or to which to store data for an S3 storage node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-s3location.html#cfn-bedrock-flow-s3location-version
            '''
            result = self._values.get("version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "S3LocationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.StorageFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"service_configuration": "serviceConfiguration"},
    )
    class StorageFlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            service_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.StorageFlowNodeServiceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''Contains configurations for a Storage node in a flow.

            This node stores the input in an Amazon S3 location that you specify.

            :param service_configuration: Contains configurations for the service to use for storing the input into the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-storageflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                storage_flow_node_configuration_property = bedrock.CfnFlow.StorageFlowNodeConfigurationProperty(
                    service_configuration=bedrock.CfnFlow.StorageFlowNodeServiceConfigurationProperty(
                        s3=bedrock.CfnFlow.StorageFlowNodeS3ConfigurationProperty(
                            bucket_name="bucketName"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2947611a82b29854c7efc58b7c733174ecf09a5853484c0d5e8f1ce11289b86a)
                check_type(argname="argument service_configuration", value=service_configuration, expected_type=type_hints["service_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "service_configuration": service_configuration,
            }

        @builtins.property
        def service_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnFlow.StorageFlowNodeServiceConfigurationProperty"]:
            '''Contains configurations for the service to use for storing the input into the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-storageflownodeconfiguration.html#cfn-bedrock-flow-storageflownodeconfiguration-serviceconfiguration
            '''
            result = self._values.get("service_configuration")
            assert result is not None, "Required property 'service_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnFlow.StorageFlowNodeServiceConfigurationProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StorageFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.StorageFlowNodeS3ConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"bucket_name": "bucketName"},
    )
    class StorageFlowNodeS3ConfigurationProperty:
        def __init__(self, *, bucket_name: builtins.str) -> None:
            '''Contains configurations for the Amazon S3 location in which to store the input into the node.

            :param bucket_name: The name of the Amazon S3 bucket in which to store the input into the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-storageflownodes3configuration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                storage_flow_node_s3_configuration_property = bedrock.CfnFlow.StorageFlowNodeS3ConfigurationProperty(
                    bucket_name="bucketName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4a2fc36b12bce2e01af3c61406e1b388bed1307c7154a5986c5562cc872007f5)
                check_type(argname="argument bucket_name", value=bucket_name, expected_type=type_hints["bucket_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "bucket_name": bucket_name,
            }

        @builtins.property
        def bucket_name(self) -> builtins.str:
            '''The name of the Amazon S3 bucket in which to store the input into the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-storageflownodes3configuration.html#cfn-bedrock-flow-storageflownodes3configuration-bucketname
            '''
            result = self._values.get("bucket_name")
            assert result is not None, "Required property 'bucket_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StorageFlowNodeS3ConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.StorageFlowNodeServiceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"s3": "s3"},
    )
    class StorageFlowNodeServiceConfigurationProperty:
        def __init__(
            self,
            *,
            s3: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.StorageFlowNodeS3ConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains configurations for the service to use for storing the input into the node.

            :param s3: Contains configurations for the Amazon S3 location in which to store the input into the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-storageflownodeserviceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                storage_flow_node_service_configuration_property = bedrock.CfnFlow.StorageFlowNodeServiceConfigurationProperty(
                    s3=bedrock.CfnFlow.StorageFlowNodeS3ConfigurationProperty(
                        bucket_name="bucketName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__de8a16140d8aef7030c7580cb53c7b271917f885403ef4e509d2d3eedefafab7)
                check_type(argname="argument s3", value=s3, expected_type=type_hints["s3"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if s3 is not None:
                self._values["s3"] = s3

        @builtins.property
        def s3(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.StorageFlowNodeS3ConfigurationProperty"]]:
            '''Contains configurations for the Amazon S3 location in which to store the input into the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-storageflownodeserviceconfiguration.html#cfn-bedrock-flow-storageflownodeserviceconfiguration-s3
            '''
            result = self._values.get("s3")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlow.StorageFlowNodeS3ConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StorageFlowNodeServiceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlow.TextPromptTemplateConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"text": "text", "input_variables": "inputVariables"},
    )
    class TextPromptTemplateConfigurationProperty:
        def __init__(
            self,
            *,
            text: builtins.str,
            input_variables: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlow.PromptInputVariableProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''Contains configurations for a text prompt template.

            To include a variable, enclose a word in double curly braces as in ``{{variable}}`` .

            :param text: The message for the prompt.
            :param input_variables: An array of the variables in the prompt template.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-textprompttemplateconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                text_prompt_template_configuration_property = bedrock.CfnFlow.TextPromptTemplateConfigurationProperty(
                    text="text",
                
                    # the properties below are optional
                    input_variables=[bedrock.CfnFlow.PromptInputVariableProperty(
                        name="name"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a1164d8b76fe518886244583bdf3b7b3822830d9f1ee7bc6beec54ce34c0f455)
                check_type(argname="argument text", value=text, expected_type=type_hints["text"])
                check_type(argname="argument input_variables", value=input_variables, expected_type=type_hints["input_variables"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "text": text,
            }
            if input_variables is not None:
                self._values["input_variables"] = input_variables

        @builtins.property
        def text(self) -> builtins.str:
            '''The message for the prompt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-textprompttemplateconfiguration.html#cfn-bedrock-flow-textprompttemplateconfiguration-text
            '''
            result = self._values.get("text")
            assert result is not None, "Required property 'text' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def input_variables(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptInputVariableProperty"]]]]:
            '''An array of the variables in the prompt template.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-textprompttemplateconfiguration.html#cfn-bedrock-flow-textprompttemplateconfiguration-inputvariables
            '''
            result = self._values.get("input_variables")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlow.PromptInputVariableProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TextPromptTemplateConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnFlowAlias(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowAlias",
):
    '''Creates an alias of a flow for deployment.

    For more information, see `Deploy a flow in Amazon Bedrock <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-deploy.html>`_ in the Amazon Bedrock User Guide.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flowalias.html
    :cloudformationResource: AWS::Bedrock::FlowAlias
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_bedrock as bedrock
        
        cfn_flow_alias = bedrock.CfnFlowAlias(self, "MyCfnFlowAlias",
            flow_arn="flowArn",
            name="name",
            routing_configuration=[bedrock.CfnFlowAlias.FlowAliasRoutingConfigurationListItemProperty(
                flow_version="flowVersion"
            )],
        
            # the properties below are optional
            description="description",
            tags={
                "tags_key": "tags"
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        flow_arn: builtins.str,
        name: builtins.str,
        routing_configuration: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowAlias.FlowAliasRoutingConfigurationListItemProperty", typing.Dict[builtins.str, typing.Any]]]]],
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param flow_arn: The Amazon Resource Name (ARN) of the alias.
        :param name: The name of the alias.
        :param routing_configuration: A list of configurations about the versions that the alias maps to. Currently, you can only specify one.
        :param description: A description of the alias.
        :param tags: Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:. - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_ - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8b7067d1caa5036e605a7c1234a4af2231017746b6c462a8b7014db38d2687a5)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnFlowAliasProps(
            flow_arn=flow_arn,
            name=name,
            routing_configuration=routing_configuration,
            description=description,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ddb56a0475d3b8ca63ecfe41b08fa1876a4c97916456646b63f185411dabdda9)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d8001dd4968cbada403f427db8aee45a29ea9e5f14d027c328e37c0416952063)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the alias.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrCreatedAt")
    def attr_created_at(self) -> builtins.str:
        '''The time at which the alias was created.

        :cloudformationAttribute: CreatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreatedAt"))

    @builtins.property
    @jsii.member(jsii_name="attrFlowId")
    def attr_flow_id(self) -> builtins.str:
        '''The unique identifier of the flow.

        :cloudformationAttribute: FlowId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFlowId"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The unique identifier of the alias of the flow.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrUpdatedAt")
    def attr_updated_at(self) -> builtins.str:
        '''The time at which the alias was last updated.

        :cloudformationAttribute: UpdatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUpdatedAt"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="flowArn")
    def flow_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the alias.'''
        return typing.cast(builtins.str, jsii.get(self, "flowArn"))

    @flow_arn.setter
    def flow_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2864acfd6282b8fcfb02ad890bc2e2dbc5dc68753bc1c8fa61e9f1d22698a5e4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "flowArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the alias.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4536ddcb8d056cdb19ccc22267dff11ec2ae20d66e372a842cbf08dcc153098d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="routingConfiguration")
    def routing_configuration(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowAlias.FlowAliasRoutingConfigurationListItemProperty"]]]:
        '''A list of configurations about the versions that the alias maps to.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowAlias.FlowAliasRoutingConfigurationListItemProperty"]]], jsii.get(self, "routingConfiguration"))

    @routing_configuration.setter
    def routing_configuration(
        self,
        value: typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowAlias.FlowAliasRoutingConfigurationListItemProperty"]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__25332829430f5e80beb4046dfc1518084a8e57996d1c7754f19340851fe7e209)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routingConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the alias.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9669636b705c78a8d145d7a1d469c787fd21070582cc73345805622ba2f9ecc0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Metadata that you can assign to a resource as key-value pairs.

        For more information, see the following resources:.
        '''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "tags"))

    @tags.setter
    def tags(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3c3491b1fef915af541cbc09b268c90d18ce0d02ff0917da6521a009344a4af3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowAlias.FlowAliasRoutingConfigurationListItemProperty",
        jsii_struct_bases=[],
        name_mapping={"flow_version": "flowVersion"},
    )
    class FlowAliasRoutingConfigurationListItemProperty:
        def __init__(
            self,
            *,
            flow_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains information about a version that the alias maps to.

            :param flow_version: The version that the alias maps to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowalias-flowaliasroutingconfigurationlistitem.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_alias_routing_configuration_list_item_property = bedrock.CfnFlowAlias.FlowAliasRoutingConfigurationListItemProperty(
                    flow_version="flowVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d9bd09906b076a991f07230b7803c936ed72845057c5587a1a1b04acf98e834b)
                check_type(argname="argument flow_version", value=flow_version, expected_type=type_hints["flow_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if flow_version is not None:
                self._values["flow_version"] = flow_version

        @builtins.property
        def flow_version(self) -> typing.Optional[builtins.str]:
            '''The version that the alias maps to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowalias-flowaliasroutingconfigurationlistitem.html#cfn-bedrock-flowalias-flowaliasroutingconfigurationlistitem-flowversion
            '''
            result = self._values.get("flow_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowAliasRoutingConfigurationListItemProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowAliasProps",
    jsii_struct_bases=[],
    name_mapping={
        "flow_arn": "flowArn",
        "name": "name",
        "routing_configuration": "routingConfiguration",
        "description": "description",
        "tags": "tags",
    },
)
class CfnFlowAliasProps:
    def __init__(
        self,
        *,
        flow_arn: builtins.str,
        name: builtins.str,
        routing_configuration: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFlowAlias.FlowAliasRoutingConfigurationListItemProperty, typing.Dict[builtins.str, typing.Any]]]]],
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnFlowAlias``.

        :param flow_arn: The Amazon Resource Name (ARN) of the alias.
        :param name: The name of the alias.
        :param routing_configuration: A list of configurations about the versions that the alias maps to. Currently, you can only specify one.
        :param description: A description of the alias.
        :param tags: Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:. - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_ - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flowalias.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_bedrock as bedrock
            
            cfn_flow_alias_props = bedrock.CfnFlowAliasProps(
                flow_arn="flowArn",
                name="name",
                routing_configuration=[bedrock.CfnFlowAlias.FlowAliasRoutingConfigurationListItemProperty(
                    flow_version="flowVersion"
                )],
            
                # the properties below are optional
                description="description",
                tags={
                    "tags_key": "tags"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dc38c23ad67fcb375dfaff403a32bd6897c077928003bee3845f452f8e08fb66)
            check_type(argname="argument flow_arn", value=flow_arn, expected_type=type_hints["flow_arn"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument routing_configuration", value=routing_configuration, expected_type=type_hints["routing_configuration"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "flow_arn": flow_arn,
            "name": name,
            "routing_configuration": routing_configuration,
        }
        if description is not None:
            self._values["description"] = description
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def flow_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the alias.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flowalias.html#cfn-bedrock-flowalias-flowarn
        '''
        result = self._values.get("flow_arn")
        assert result is not None, "Required property 'flow_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of the alias.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flowalias.html#cfn-bedrock-flowalias-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def routing_configuration(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnFlowAlias.FlowAliasRoutingConfigurationListItemProperty]]]:
        '''A list of configurations about the versions that the alias maps to.

        Currently, you can only specify one.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flowalias.html#cfn-bedrock-flowalias-routingconfiguration
        '''
        result = self._values.get("routing_configuration")
        assert result is not None, "Required property 'routing_configuration' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnFlowAlias.FlowAliasRoutingConfigurationListItemProperty]]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the alias.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flowalias.html#cfn-bedrock-flowalias-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:.

        - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_
        - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flowalias.html#cfn-bedrock-flowalias-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnFlowAliasProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowProps",
    jsii_struct_bases=[],
    name_mapping={
        "execution_role_arn": "executionRoleArn",
        "name": "name",
        "customer_encryption_key_arn": "customerEncryptionKeyArn",
        "definition": "definition",
        "definition_s3_location": "definitionS3Location",
        "definition_string": "definitionString",
        "definition_substitutions": "definitionSubstitutions",
        "description": "description",
        "tags": "tags",
        "test_alias_tags": "testAliasTags",
    },
)
class CfnFlowProps:
    def __init__(
        self,
        *,
        execution_role_arn: builtins.str,
        name: builtins.str,
        customer_encryption_key_arn: typing.Optional[builtins.str] = None,
        definition: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFlow.FlowDefinitionProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        definition_s3_location: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnFlow.S3LocationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        definition_string: typing.Optional[builtins.str] = None,
        definition_substitutions: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool, _IResolvable_da3f097b]]]] = None,
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        test_alias_tags: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
    ) -> None:
        '''Properties for defining a ``CfnFlow``.

        :param execution_role_arn: The Amazon Resource Name (ARN) of the service role with permissions to create a flow. For more information, see `Create a service row for flows <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-permissions.html>`_ in the Amazon Bedrock User Guide.
        :param name: The name of the flow.
        :param customer_encryption_key_arn: The Amazon Resource Name (ARN) of the KMS key that the flow is encrypted with.
        :param definition: The definition of the nodes and connections between the nodes in the flow.
        :param definition_s3_location: The Amazon S3 location of the flow definition.
        :param definition_string: The definition of the flow as a JSON-formatted string. The string must match the format in `FlowDefinition <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdefinition.html>`_ .
        :param definition_substitutions: A map that specifies the mappings for placeholder variables in the prompt flow definition. This enables the customer to inject values obtained at runtime. Variables can be template parameter names, resource logical IDs, resource attributes, or a variable in a key-value map. Only supported with the ``DefinitionString`` and ``DefinitionS3Location`` fields. Substitutions must follow the syntax: ``${key_name}`` or ``${variable_1,variable_2,...}`` .
        :param description: A description of the flow.
        :param tags: Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:. - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_ - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_
        :param test_alias_tags: A map of tag keys and values.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_bedrock as bedrock
            
            # collector: Any
            # input: Any
            # iterator: Any
            # output: Any
            
            cfn_flow_props = bedrock.CfnFlowProps(
                execution_role_arn="executionRoleArn",
                name="name",
            
                # the properties below are optional
                customer_encryption_key_arn="customerEncryptionKeyArn",
                definition=bedrock.CfnFlow.FlowDefinitionProperty(
                    connections=[bedrock.CfnFlow.FlowConnectionProperty(
                        name="name",
                        source="source",
                        target="target",
                        type="type",
            
                        # the properties below are optional
                        configuration=bedrock.CfnFlow.FlowConnectionConfigurationProperty(
                            conditional=bedrock.CfnFlow.FlowConditionalConnectionConfigurationProperty(
                                condition="condition"
                            ),
                            data=bedrock.CfnFlow.FlowDataConnectionConfigurationProperty(
                                source_output="sourceOutput",
                                target_input="targetInput"
                            )
                        )
                    )],
                    nodes=[bedrock.CfnFlow.FlowNodeProperty(
                        name="name",
                        type="type",
            
                        # the properties below are optional
                        configuration=bedrock.CfnFlow.FlowNodeConfigurationProperty(
                            agent=bedrock.CfnFlow.AgentFlowNodeConfigurationProperty(
                                agent_alias_arn="agentAliasArn"
                            ),
                            collector=collector,
                            condition=bedrock.CfnFlow.ConditionFlowNodeConfigurationProperty(
                                conditions=[bedrock.CfnFlow.FlowConditionProperty(
                                    name="name",
            
                                    # the properties below are optional
                                    expression="expression"
                                )]
                            ),
                            input=input,
                            iterator=iterator,
                            knowledge_base=bedrock.CfnFlow.KnowledgeBaseFlowNodeConfigurationProperty(
                                knowledge_base_id="knowledgeBaseId",
            
                                # the properties below are optional
                                guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                                    guardrail_identifier="guardrailIdentifier",
                                    guardrail_version="guardrailVersion"
                                ),
                                model_id="modelId"
                            ),
                            lambda_function=bedrock.CfnFlow.LambdaFunctionFlowNodeConfigurationProperty(
                                lambda_arn="lambdaArn"
                            ),
                            lex=bedrock.CfnFlow.LexFlowNodeConfigurationProperty(
                                bot_alias_arn="botAliasArn",
                                locale_id="localeId"
                            ),
                            output=output,
                            prompt=bedrock.CfnFlow.PromptFlowNodeConfigurationProperty(
                                source_configuration=bedrock.CfnFlow.PromptFlowNodeSourceConfigurationProperty(
                                    inline=bedrock.CfnFlow.PromptFlowNodeInlineConfigurationProperty(
                                        model_id="modelId",
                                        template_configuration=bedrock.CfnFlow.PromptTemplateConfigurationProperty(
                                            text=bedrock.CfnFlow.TextPromptTemplateConfigurationProperty(
                                                text="text",
            
                                                # the properties below are optional
                                                input_variables=[bedrock.CfnFlow.PromptInputVariableProperty(
                                                    name="name"
                                                )]
                                            )
                                        ),
                                        template_type="templateType",
            
                                        # the properties below are optional
                                        inference_configuration=bedrock.CfnFlow.PromptInferenceConfigurationProperty(
                                            text=bedrock.CfnFlow.PromptModelInferenceConfigurationProperty(
                                                max_tokens=123,
                                                stop_sequences=["stopSequences"],
                                                temperature=123,
                                                top_p=123
                                            )
                                        )
                                    ),
                                    resource=bedrock.CfnFlow.PromptFlowNodeResourceConfigurationProperty(
                                        prompt_arn="promptArn"
                                    )
                                ),
            
                                # the properties below are optional
                                guardrail_configuration=bedrock.CfnFlow.GuardrailConfigurationProperty(
                                    guardrail_identifier="guardrailIdentifier",
                                    guardrail_version="guardrailVersion"
                                )
                            ),
                            retrieval=bedrock.CfnFlow.RetrievalFlowNodeConfigurationProperty(
                                service_configuration=bedrock.CfnFlow.RetrievalFlowNodeServiceConfigurationProperty(
                                    s3=bedrock.CfnFlow.RetrievalFlowNodeS3ConfigurationProperty(
                                        bucket_name="bucketName"
                                    )
                                )
                            ),
                            storage=bedrock.CfnFlow.StorageFlowNodeConfigurationProperty(
                                service_configuration=bedrock.CfnFlow.StorageFlowNodeServiceConfigurationProperty(
                                    s3=bedrock.CfnFlow.StorageFlowNodeS3ConfigurationProperty(
                                        bucket_name="bucketName"
                                    )
                                )
                            )
                        ),
                        inputs=[bedrock.CfnFlow.FlowNodeInputProperty(
                            expression="expression",
                            name="name",
                            type="type"
                        )],
                        outputs=[bedrock.CfnFlow.FlowNodeOutputProperty(
                            name="name",
                            type="type"
                        )]
                    )]
                ),
                definition_s3_location=bedrock.CfnFlow.S3LocationProperty(
                    bucket="bucket",
                    key="key",
            
                    # the properties below are optional
                    version="version"
                ),
                definition_string="definitionString",
                definition_substitutions={
                    "definition_substitutions_key": "definitionSubstitutions"
                },
                description="description",
                tags={
                    "tags_key": "tags"
                },
                test_alias_tags={
                    "test_alias_tags_key": "testAliasTags"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bad8ab7f592d2682a6b9c1a504f97fac9b6e0a7bbbfcf4968c42f3a954576115)
            check_type(argname="argument execution_role_arn", value=execution_role_arn, expected_type=type_hints["execution_role_arn"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument customer_encryption_key_arn", value=customer_encryption_key_arn, expected_type=type_hints["customer_encryption_key_arn"])
            check_type(argname="argument definition", value=definition, expected_type=type_hints["definition"])
            check_type(argname="argument definition_s3_location", value=definition_s3_location, expected_type=type_hints["definition_s3_location"])
            check_type(argname="argument definition_string", value=definition_string, expected_type=type_hints["definition_string"])
            check_type(argname="argument definition_substitutions", value=definition_substitutions, expected_type=type_hints["definition_substitutions"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument test_alias_tags", value=test_alias_tags, expected_type=type_hints["test_alias_tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "execution_role_arn": execution_role_arn,
            "name": name,
        }
        if customer_encryption_key_arn is not None:
            self._values["customer_encryption_key_arn"] = customer_encryption_key_arn
        if definition is not None:
            self._values["definition"] = definition
        if definition_s3_location is not None:
            self._values["definition_s3_location"] = definition_s3_location
        if definition_string is not None:
            self._values["definition_string"] = definition_string
        if definition_substitutions is not None:
            self._values["definition_substitutions"] = definition_substitutions
        if description is not None:
            self._values["description"] = description
        if tags is not None:
            self._values["tags"] = tags
        if test_alias_tags is not None:
            self._values["test_alias_tags"] = test_alias_tags

    @builtins.property
    def execution_role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the service role with permissions to create a flow.

        For more information, see `Create a service row for flows <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-permissions.html>`_ in the Amazon Bedrock User Guide.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html#cfn-bedrock-flow-executionrolearn
        '''
        result = self._values.get("execution_role_arn")
        assert result is not None, "Required property 'execution_role_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of the flow.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html#cfn-bedrock-flow-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def customer_encryption_key_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the KMS key that the flow is encrypted with.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html#cfn-bedrock-flow-customerencryptionkeyarn
        '''
        result = self._values.get("customer_encryption_key_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def definition(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFlow.FlowDefinitionProperty]]:
        '''The definition of the nodes and connections between the nodes in the flow.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html#cfn-bedrock-flow-definition
        '''
        result = self._values.get("definition")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFlow.FlowDefinitionProperty]], result)

    @builtins.property
    def definition_s3_location(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFlow.S3LocationProperty]]:
        '''The Amazon S3 location of the flow definition.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html#cfn-bedrock-flow-definitions3location
        '''
        result = self._values.get("definition_s3_location")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnFlow.S3LocationProperty]], result)

    @builtins.property
    def definition_string(self) -> typing.Optional[builtins.str]:
        '''The definition of the flow as a JSON-formatted string.

        The string must match the format in `FlowDefinition <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flow-flowdefinition.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html#cfn-bedrock-flow-definitionstring
        '''
        result = self._values.get("definition_string")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def definition_substitutions(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool, _IResolvable_da3f097b]]]]:
        '''A map that specifies the mappings for placeholder variables in the prompt flow definition.

        This enables the customer to inject values obtained at runtime. Variables can be template parameter names, resource logical IDs, resource attributes, or a variable in a key-value map. Only supported with the ``DefinitionString`` and ``DefinitionS3Location`` fields.

        Substitutions must follow the syntax: ``${key_name}`` or ``${variable_1,variable_2,...}`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html#cfn-bedrock-flow-definitionsubstitutions
        '''
        result = self._values.get("definition_substitutions")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.str, jsii.Number, builtins.bool, _IResolvable_da3f097b]]]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the flow.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html#cfn-bedrock-flow-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Metadata that you can assign to a resource as key-value pairs. For more information, see the following resources:.

        - `Tag naming limits and requirements <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-conventions>`_
        - `Tagging best practices <https://docs.aws.amazon.com/tag-editor/latest/userguide/tagging.html#tag-best-practices>`_

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html#cfn-bedrock-flow-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def test_alias_tags(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''A map of tag keys and values.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flow.html#cfn-bedrock-flow-testaliastags
        '''
        result = self._values.get("test_alias_tags")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnFlowProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnFlowVersion(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion",
):
    '''Creates a version of the flow that you can deploy.

    For more information, see `Deploy a flow in Amazon Bedrock <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-deploy.html>`_ in the Amazon Bedrock User Guide.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-bedrock-flowversion.html
    :cloudformationResource: AWS::Bedrock::FlowVersion
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_bedrock as bedrock
        
        cfn_flow_version = bedrock.CfnFlowVersion(self, "MyCfnFlowVersion",
            flow_arn="flowArn",
        
            # the properties below are optional
            description="description"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        flow_arn: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param flow_arn: The Amazon Resource Name (ARN) of the flow that the version belongs to.
        :param description: The description of the flow version.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e996d6ff14b49f6abcc0d0964d3cd945bc468ee27dceb388effde1a557cebdb9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnFlowVersionProps(flow_arn=flow_arn, description=description)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf55831ebd6a3528e3e154ed216e112eb5cd8de05a2542001ff8f2a2393908a0)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ee278fde485b2ef3651f6ea91971dbc1a44b6c3cfa00f8ea962a940a8cd4cb3f)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCreatedAt")
    def attr_created_at(self) -> builtins.str:
        '''The time at the version was created.

        :cloudformationAttribute: CreatedAt
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreatedAt"))

    @builtins.property
    @jsii.member(jsii_name="attrCustomerEncryptionKeyArn")
    def attr_customer_encryption_key_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the KMS key that the flow version is encrypted with.

        :cloudformationAttribute: CustomerEncryptionKeyArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCustomerEncryptionKeyArn"))

    @builtins.property
    @jsii.member(jsii_name="attrDefinition")
    def attr_definition(self) -> _IResolvable_da3f097b:
        '''Flow definition.

        :cloudformationAttribute: Definition
        '''
        return typing.cast(_IResolvable_da3f097b, jsii.get(self, "attrDefinition"))

    @builtins.property
    @jsii.member(jsii_name="attrExecutionRoleArn")
    def attr_execution_role_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the service role with permissions to create a flow.

        For more information, see `Create a service row for flows <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-permissions.html>`_ in the Amazon Bedrock User Guide.

        :cloudformationAttribute: ExecutionRoleArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrExecutionRoleArn"))

    @builtins.property
    @jsii.member(jsii_name="attrFlowId")
    def attr_flow_id(self) -> builtins.str:
        '''The unique identifier of the flow.

        :cloudformationAttribute: FlowId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFlowId"))

    @builtins.property
    @jsii.member(jsii_name="attrName")
    def attr_name(self) -> builtins.str:
        '''The name of the flow.

        :cloudformationAttribute: Name
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrName"))

    @builtins.property
    @jsii.member(jsii_name="attrStatus")
    def attr_status(self) -> builtins.str:
        '''The status of the flow.

        :cloudformationAttribute: Status
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrStatus"))

    @builtins.property
    @jsii.member(jsii_name="attrVersion")
    def attr_version(self) -> builtins.str:
        '''The version of the flow.

        :cloudformationAttribute: Version
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVersion"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="flowArn")
    def flow_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the flow that the version belongs to.'''
        return typing.cast(builtins.str, jsii.get(self, "flowArn"))

    @flow_arn.setter
    def flow_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6683eaab0093c03edd8f27520bee97a77a0510e45f6636980bfec6a332e24977)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "flowArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the flow version.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7f982c06560684269ac07c3dc57223c48225800eaacbc87f08a51a55eafb0d69)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.AgentFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"agent_alias_arn": "agentAliasArn"},
    )
    class AgentFlowNodeConfigurationProperty:
        def __init__(self, *, agent_alias_arn: builtins.str) -> None:
            '''Defines an agent node in your flow.

            You specify the agent to invoke at this point in the flow. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param agent_alias_arn: The Amazon Resource Name (ARN) of the alias of the agent to invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-agentflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                agent_flow_node_configuration_property = bedrock.CfnFlowVersion.AgentFlowNodeConfigurationProperty(
                    agent_alias_arn="agentAliasArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d2f9b914c5d40c946bff1dbcc6c016aacfab4e4b5c8f2a45cb392c7a1adb7083)
                check_type(argname="argument agent_alias_arn", value=agent_alias_arn, expected_type=type_hints["agent_alias_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "agent_alias_arn": agent_alias_arn,
            }

        @builtins.property
        def agent_alias_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the alias of the agent to invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-agentflownodeconfiguration.html#cfn-bedrock-flowversion-agentflownodeconfiguration-agentaliasarn
            '''
            result = self._values.get("agent_alias_arn")
            assert result is not None, "Required property 'agent_alias_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AgentFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.ConditionFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"conditions": "conditions"},
    )
    class ConditionFlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            conditions: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.FlowConditionProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''Defines a condition node in your flow.

            You can specify conditions that determine which node comes next in the flow. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param conditions: An array of conditions. Each member contains the name of a condition and an expression that defines the condition.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-conditionflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                condition_flow_node_configuration_property = bedrock.CfnFlowVersion.ConditionFlowNodeConfigurationProperty(
                    conditions=[bedrock.CfnFlowVersion.FlowConditionProperty(
                        name="name",
                
                        # the properties below are optional
                        expression="expression"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__58998504af7a39c970b789ec28fb7e3afe8c42041c2bad9381d2fc98e905dc94)
                check_type(argname="argument conditions", value=conditions, expected_type=type_hints["conditions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "conditions": conditions,
            }

        @builtins.property
        def conditions(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowConditionProperty"]]]:
            '''An array of conditions.

            Each member contains the name of a condition and an expression that defines the condition.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-conditionflownodeconfiguration.html#cfn-bedrock-flowversion-conditionflownodeconfiguration-conditions
            '''
            result = self._values.get("conditions")
            assert result is not None, "Required property 'conditions' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowConditionProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConditionFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.FlowConditionProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "expression": "expression"},
    )
    class FlowConditionProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            expression: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Defines a condition in the condition node.

            :param name: A name for the condition that you can reference.
            :param expression: Defines the condition. You must refer to at least one of the inputs in the condition. For more information, expand the Condition node section in `Node types in prompt flows <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-how-it-works.html#flows-nodes>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowcondition.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_condition_property = bedrock.CfnFlowVersion.FlowConditionProperty(
                    name="name",
                
                    # the properties below are optional
                    expression="expression"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cadc2588fff4a1e90fc2faf2b9d50c41f5b49d11d87f84d8448ccf279aee1969)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument expression", value=expression, expected_type=type_hints["expression"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if expression is not None:
                self._values["expression"] = expression

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the condition that you can reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowcondition.html#cfn-bedrock-flowversion-flowcondition-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def expression(self) -> typing.Optional[builtins.str]:
            '''Defines the condition.

            You must refer to at least one of the inputs in the condition. For more information, expand the Condition node section in `Node types in prompt flows <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-how-it-works.html#flows-nodes>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowcondition.html#cfn-bedrock-flowversion-flowcondition-expression
            '''
            result = self._values.get("expression")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowConditionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.FlowConditionalConnectionConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"condition": "condition"},
    )
    class FlowConditionalConnectionConfigurationProperty:
        def __init__(self, *, condition: builtins.str) -> None:
            '''The configuration of a connection between a condition node and another node.

            :param condition: The condition that triggers this connection. For more information about how to write conditions, see the *Condition* node type in the `Node types <https://docs.aws.amazon.com/bedrock/latest/userguide/node-types.html>`_ topic in the Amazon Bedrock User Guide.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowconditionalconnectionconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_conditional_connection_configuration_property = bedrock.CfnFlowVersion.FlowConditionalConnectionConfigurationProperty(
                    condition="condition"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__39d819859e9d34a2bb668b74f7572cd97777fba5b0a0fb86a0a387b276db1c2c)
                check_type(argname="argument condition", value=condition, expected_type=type_hints["condition"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "condition": condition,
            }

        @builtins.property
        def condition(self) -> builtins.str:
            '''The condition that triggers this connection.

            For more information about how to write conditions, see the *Condition* node type in the `Node types <https://docs.aws.amazon.com/bedrock/latest/userguide/node-types.html>`_ topic in the Amazon Bedrock User Guide.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowconditionalconnectionconfiguration.html#cfn-bedrock-flowversion-flowconditionalconnectionconfiguration-condition
            '''
            result = self._values.get("condition")
            assert result is not None, "Required property 'condition' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowConditionalConnectionConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.FlowConnectionConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"conditional": "conditional", "data": "data"},
    )
    class FlowConnectionConfigurationProperty:
        def __init__(
            self,
            *,
            conditional: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.FlowConditionalConnectionConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            data: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.FlowDataConnectionConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The configuration of the connection.

            :param conditional: The configuration of a connection originating from a Condition node.
            :param data: The configuration of a connection originating from a node that isn't a Condition node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowconnectionconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_connection_configuration_property = bedrock.CfnFlowVersion.FlowConnectionConfigurationProperty(
                    conditional=bedrock.CfnFlowVersion.FlowConditionalConnectionConfigurationProperty(
                        condition="condition"
                    ),
                    data=bedrock.CfnFlowVersion.FlowDataConnectionConfigurationProperty(
                        source_output="sourceOutput",
                        target_input="targetInput"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d302d6f02d96a87319cbeb6aa637aa260168ac6969e17b1be8d60b38a6558699)
                check_type(argname="argument conditional", value=conditional, expected_type=type_hints["conditional"])
                check_type(argname="argument data", value=data, expected_type=type_hints["data"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if conditional is not None:
                self._values["conditional"] = conditional
            if data is not None:
                self._values["data"] = data

        @builtins.property
        def conditional(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowConditionalConnectionConfigurationProperty"]]:
            '''The configuration of a connection originating from a Condition node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowconnectionconfiguration.html#cfn-bedrock-flowversion-flowconnectionconfiguration-conditional
            '''
            result = self._values.get("conditional")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowConditionalConnectionConfigurationProperty"]], result)

        @builtins.property
        def data(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowDataConnectionConfigurationProperty"]]:
            '''The configuration of a connection originating from a node that isn't a Condition node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowconnectionconfiguration.html#cfn-bedrock-flowversion-flowconnectionconfiguration-data
            '''
            result = self._values.get("data")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowDataConnectionConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowConnectionConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.FlowConnectionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "source": "source",
            "target": "target",
            "type": "type",
            "configuration": "configuration",
        },
    )
    class FlowConnectionProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            source: builtins.str,
            target: builtins.str,
            type: builtins.str,
            configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.FlowConnectionConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains information about a connection between two nodes in the flow.

            :param name: A name for the connection that you can reference.
            :param source: The node that the connection starts at.
            :param target: The node that the connection ends at.
            :param type: Whether the source node that the connection begins from is a condition node ( ``Conditional`` ) or not ( ``Data`` ).
            :param configuration: The configuration of the connection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowconnection.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_connection_property = bedrock.CfnFlowVersion.FlowConnectionProperty(
                    name="name",
                    source="source",
                    target="target",
                    type="type",
                
                    # the properties below are optional
                    configuration=bedrock.CfnFlowVersion.FlowConnectionConfigurationProperty(
                        conditional=bedrock.CfnFlowVersion.FlowConditionalConnectionConfigurationProperty(
                            condition="condition"
                        ),
                        data=bedrock.CfnFlowVersion.FlowDataConnectionConfigurationProperty(
                            source_output="sourceOutput",
                            target_input="targetInput"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__90deb666e6b636a82dd13d2a0fb5a5bcfebb83a79b14a1b949a1ffd86fc32558)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument source", value=source, expected_type=type_hints["source"])
                check_type(argname="argument target", value=target, expected_type=type_hints["target"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument configuration", value=configuration, expected_type=type_hints["configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "source": source,
                "target": target,
                "type": type,
            }
            if configuration is not None:
                self._values["configuration"] = configuration

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the connection that you can reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowconnection.html#cfn-bedrock-flowversion-flowconnection-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def source(self) -> builtins.str:
            '''The node that the connection starts at.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowconnection.html#cfn-bedrock-flowversion-flowconnection-source
            '''
            result = self._values.get("source")
            assert result is not None, "Required property 'source' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def target(self) -> builtins.str:
            '''The node that the connection ends at.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowconnection.html#cfn-bedrock-flowversion-flowconnection-target
            '''
            result = self._values.get("target")
            assert result is not None, "Required property 'target' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def type(self) -> builtins.str:
            '''Whether the source node that the connection begins from is a condition node ( ``Conditional`` ) or not ( ``Data`` ).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowconnection.html#cfn-bedrock-flowversion-flowconnection-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowConnectionConfigurationProperty"]]:
            '''The configuration of the connection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowconnection.html#cfn-bedrock-flowversion-flowconnection-configuration
            '''
            result = self._values.get("configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowConnectionConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowConnectionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.FlowDataConnectionConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"source_output": "sourceOutput", "target_input": "targetInput"},
    )
    class FlowDataConnectionConfigurationProperty:
        def __init__(
            self,
            *,
            source_output: builtins.str,
            target_input: builtins.str,
        ) -> None:
            '''The configuration of a connection originating from a node that isn't a Condition node.

            :param source_output: The name of the output in the source node that the connection begins from.
            :param target_input: The name of the input in the target node that the connection ends at.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowdataconnectionconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_data_connection_configuration_property = bedrock.CfnFlowVersion.FlowDataConnectionConfigurationProperty(
                    source_output="sourceOutput",
                    target_input="targetInput"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0a02aa2b464b1c9d91f7752f044076ac0d98ab7a57b199e56f5371917602f4cd)
                check_type(argname="argument source_output", value=source_output, expected_type=type_hints["source_output"])
                check_type(argname="argument target_input", value=target_input, expected_type=type_hints["target_input"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "source_output": source_output,
                "target_input": target_input,
            }

        @builtins.property
        def source_output(self) -> builtins.str:
            '''The name of the output in the source node that the connection begins from.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowdataconnectionconfiguration.html#cfn-bedrock-flowversion-flowdataconnectionconfiguration-sourceoutput
            '''
            result = self._values.get("source_output")
            assert result is not None, "Required property 'source_output' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def target_input(self) -> builtins.str:
            '''The name of the input in the target node that the connection ends at.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowdataconnectionconfiguration.html#cfn-bedrock-flowversion-flowdataconnectionconfiguration-targetinput
            '''
            result = self._values.get("target_input")
            assert result is not None, "Required property 'target_input' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowDataConnectionConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.FlowDefinitionProperty",
        jsii_struct_bases=[],
        name_mapping={"connections": "connections", "nodes": "nodes"},
    )
    class FlowDefinitionProperty:
        def __init__(
            self,
            *,
            connections: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.FlowConnectionProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            nodes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.FlowNodeProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The definition of the nodes and connections between nodes in the flow.

            :param connections: An array of connection definitions in the flow.
            :param nodes: An array of node definitions in the flow.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowdefinition.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                # collector: Any
                # input: Any
                # iterator: Any
                # output: Any
                
                flow_definition_property = bedrock.CfnFlowVersion.FlowDefinitionProperty(
                    connections=[bedrock.CfnFlowVersion.FlowConnectionProperty(
                        name="name",
                        source="source",
                        target="target",
                        type="type",
                
                        # the properties below are optional
                        configuration=bedrock.CfnFlowVersion.FlowConnectionConfigurationProperty(
                            conditional=bedrock.CfnFlowVersion.FlowConditionalConnectionConfigurationProperty(
                                condition="condition"
                            ),
                            data=bedrock.CfnFlowVersion.FlowDataConnectionConfigurationProperty(
                                source_output="sourceOutput",
                                target_input="targetInput"
                            )
                        )
                    )],
                    nodes=[bedrock.CfnFlowVersion.FlowNodeProperty(
                        name="name",
                        type="type",
                
                        # the properties below are optional
                        configuration=bedrock.CfnFlowVersion.FlowNodeConfigurationProperty(
                            agent=bedrock.CfnFlowVersion.AgentFlowNodeConfigurationProperty(
                                agent_alias_arn="agentAliasArn"
                            ),
                            collector=collector,
                            condition=bedrock.CfnFlowVersion.ConditionFlowNodeConfigurationProperty(
                                conditions=[bedrock.CfnFlowVersion.FlowConditionProperty(
                                    name="name",
                
                                    # the properties below are optional
                                    expression="expression"
                                )]
                            ),
                            input=input,
                            iterator=iterator,
                            knowledge_base=bedrock.CfnFlowVersion.KnowledgeBaseFlowNodeConfigurationProperty(
                                knowledge_base_id="knowledgeBaseId",
                
                                # the properties below are optional
                                guardrail_configuration=bedrock.CfnFlowVersion.GuardrailConfigurationProperty(
                                    guardrail_identifier="guardrailIdentifier",
                                    guardrail_version="guardrailVersion"
                                ),
                                model_id="modelId"
                            ),
                            lambda_function=bedrock.CfnFlowVersion.LambdaFunctionFlowNodeConfigurationProperty(
                                lambda_arn="lambdaArn"
                            ),
                            lex=bedrock.CfnFlowVersion.LexFlowNodeConfigurationProperty(
                                bot_alias_arn="botAliasArn",
                                locale_id="localeId"
                            ),
                            output=output,
                            prompt=bedrock.CfnFlowVersion.PromptFlowNodeConfigurationProperty(
                                source_configuration=bedrock.CfnFlowVersion.PromptFlowNodeSourceConfigurationProperty(
                                    inline=bedrock.CfnFlowVersion.PromptFlowNodeInlineConfigurationProperty(
                                        model_id="modelId",
                                        template_configuration=bedrock.CfnFlowVersion.PromptTemplateConfigurationProperty(
                                            text=bedrock.CfnFlowVersion.TextPromptTemplateConfigurationProperty(
                                                text="text",
                
                                                # the properties below are optional
                                                input_variables=[bedrock.CfnFlowVersion.PromptInputVariableProperty(
                                                    name="name"
                                                )]
                                            )
                                        ),
                                        template_type="templateType",
                
                                        # the properties below are optional
                                        inference_configuration=bedrock.CfnFlowVersion.PromptInferenceConfigurationProperty(
                                            text=bedrock.CfnFlowVersion.PromptModelInferenceConfigurationProperty(
                                                max_tokens=123,
                                                stop_sequences=["stopSequences"],
                                                temperature=123,
                                                top_p=123
                                            )
                                        )
                                    ),
                                    resource=bedrock.CfnFlowVersion.PromptFlowNodeResourceConfigurationProperty(
                                        prompt_arn="promptArn"
                                    )
                                ),
                
                                # the properties below are optional
                                guardrail_configuration=bedrock.CfnFlowVersion.GuardrailConfigurationProperty(
                                    guardrail_identifier="guardrailIdentifier",
                                    guardrail_version="guardrailVersion"
                                )
                            ),
                            retrieval=bedrock.CfnFlowVersion.RetrievalFlowNodeConfigurationProperty(
                                service_configuration=bedrock.CfnFlowVersion.RetrievalFlowNodeServiceConfigurationProperty(
                                    s3=bedrock.CfnFlowVersion.RetrievalFlowNodeS3ConfigurationProperty(
                                        bucket_name="bucketName"
                                    )
                                )
                            ),
                            storage=bedrock.CfnFlowVersion.StorageFlowNodeConfigurationProperty(
                                service_configuration=bedrock.CfnFlowVersion.StorageFlowNodeServiceConfigurationProperty(
                                    s3=bedrock.CfnFlowVersion.StorageFlowNodeS3ConfigurationProperty(
                                        bucket_name="bucketName"
                                    )
                                )
                            )
                        ),
                        inputs=[bedrock.CfnFlowVersion.FlowNodeInputProperty(
                            expression="expression",
                            name="name",
                            type="type"
                        )],
                        outputs=[bedrock.CfnFlowVersion.FlowNodeOutputProperty(
                            name="name",
                            type="type"
                        )]
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8475acb2c15a224f24d256dd7b562970391360b68614b931f44c1e9814651009)
                check_type(argname="argument connections", value=connections, expected_type=type_hints["connections"])
                check_type(argname="argument nodes", value=nodes, expected_type=type_hints["nodes"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if connections is not None:
                self._values["connections"] = connections
            if nodes is not None:
                self._values["nodes"] = nodes

        @builtins.property
        def connections(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowConnectionProperty"]]]]:
            '''An array of connection definitions in the flow.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowdefinition.html#cfn-bedrock-flowversion-flowdefinition-connections
            '''
            result = self._values.get("connections")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowConnectionProperty"]]]], result)

        @builtins.property
        def nodes(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowNodeProperty"]]]]:
            '''An array of node definitions in the flow.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flowdefinition.html#cfn-bedrock-flowversion-flowdefinition-nodes
            '''
            result = self._values.get("nodes")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowNodeProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowDefinitionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.FlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "agent": "agent",
            "collector": "collector",
            "condition": "condition",
            "input": "input",
            "iterator": "iterator",
            "knowledge_base": "knowledgeBase",
            "lambda_function": "lambdaFunction",
            "lex": "lex",
            "output": "output",
            "prompt": "prompt",
            "retrieval": "retrieval",
            "storage": "storage",
        },
    )
    class FlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            agent: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.AgentFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            collector: typing.Any = None,
            condition: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.ConditionFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            input: typing.Any = None,
            iterator: typing.Any = None,
            knowledge_base: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.KnowledgeBaseFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            lambda_function: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.LambdaFunctionFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            lex: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.LexFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            output: typing.Any = None,
            prompt: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.PromptFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            retrieval: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.RetrievalFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            storage: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.StorageFlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains configurations for a node in your flow.

            For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param agent: Contains configurations for an agent node in your flow. Invokes an alias of an agent and returns the response.
            :param collector: Contains configurations for a collector node in your flow. Collects an iteration of inputs and consolidates them into an array of outputs.
            :param condition: Contains configurations for a Condition node in your flow. Defines conditions that lead to different branches of the flow.
            :param input: Contains configurations for an input flow node in your flow. The first node in the flow. ``inputs`` can't be specified for this node.
            :param iterator: Contains configurations for an iterator node in your flow. Takes an input that is an array and iteratively sends each item of the array as an output to the following node. The size of the array is also returned in the output. The output flow node at the end of the flow iteration will return a response for each member of the array. To return only one response, you can include a collector node downstream from the iterator node.
            :param knowledge_base: Contains configurations for a knowledge base node in your flow. Queries a knowledge base and returns the retrieved results or generated response.
            :param lambda_function: Contains configurations for a Lambda function node in your flow. Invokes an AWS Lambda function.
            :param lex: Contains configurations for a Lex node in your flow. Invokes an Amazon Lex bot to identify the intent of the input and return the intent as the output.
            :param output: Contains configurations for an output flow node in your flow. The last node in the flow. ``outputs`` can't be specified for this node.
            :param prompt: Contains configurations for a prompt node in your flow. Runs a prompt and generates the model response as the output. You can use a prompt from Prompt management or you can configure one in this node.
            :param retrieval: Contains configurations for a Retrieval node in your flow. Retrieves data from an Amazon S3 location and returns it as the output.
            :param storage: Contains configurations for a Storage node in your flow. Stores an input in an Amazon S3 location.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                # collector: Any
                # input: Any
                # iterator: Any
                # output: Any
                
                flow_node_configuration_property = bedrock.CfnFlowVersion.FlowNodeConfigurationProperty(
                    agent=bedrock.CfnFlowVersion.AgentFlowNodeConfigurationProperty(
                        agent_alias_arn="agentAliasArn"
                    ),
                    collector=collector,
                    condition=bedrock.CfnFlowVersion.ConditionFlowNodeConfigurationProperty(
                        conditions=[bedrock.CfnFlowVersion.FlowConditionProperty(
                            name="name",
                
                            # the properties below are optional
                            expression="expression"
                        )]
                    ),
                    input=input,
                    iterator=iterator,
                    knowledge_base=bedrock.CfnFlowVersion.KnowledgeBaseFlowNodeConfigurationProperty(
                        knowledge_base_id="knowledgeBaseId",
                
                        # the properties below are optional
                        guardrail_configuration=bedrock.CfnFlowVersion.GuardrailConfigurationProperty(
                            guardrail_identifier="guardrailIdentifier",
                            guardrail_version="guardrailVersion"
                        ),
                        model_id="modelId"
                    ),
                    lambda_function=bedrock.CfnFlowVersion.LambdaFunctionFlowNodeConfigurationProperty(
                        lambda_arn="lambdaArn"
                    ),
                    lex=bedrock.CfnFlowVersion.LexFlowNodeConfigurationProperty(
                        bot_alias_arn="botAliasArn",
                        locale_id="localeId"
                    ),
                    output=output,
                    prompt=bedrock.CfnFlowVersion.PromptFlowNodeConfigurationProperty(
                        source_configuration=bedrock.CfnFlowVersion.PromptFlowNodeSourceConfigurationProperty(
                            inline=bedrock.CfnFlowVersion.PromptFlowNodeInlineConfigurationProperty(
                                model_id="modelId",
                                template_configuration=bedrock.CfnFlowVersion.PromptTemplateConfigurationProperty(
                                    text=bedrock.CfnFlowVersion.TextPromptTemplateConfigurationProperty(
                                        text="text",
                
                                        # the properties below are optional
                                        input_variables=[bedrock.CfnFlowVersion.PromptInputVariableProperty(
                                            name="name"
                                        )]
                                    )
                                ),
                                template_type="templateType",
                
                                # the properties below are optional
                                inference_configuration=bedrock.CfnFlowVersion.PromptInferenceConfigurationProperty(
                                    text=bedrock.CfnFlowVersion.PromptModelInferenceConfigurationProperty(
                                        max_tokens=123,
                                        stop_sequences=["stopSequences"],
                                        temperature=123,
                                        top_p=123
                                    )
                                )
                            ),
                            resource=bedrock.CfnFlowVersion.PromptFlowNodeResourceConfigurationProperty(
                                prompt_arn="promptArn"
                            )
                        ),
                
                        # the properties below are optional
                        guardrail_configuration=bedrock.CfnFlowVersion.GuardrailConfigurationProperty(
                            guardrail_identifier="guardrailIdentifier",
                            guardrail_version="guardrailVersion"
                        )
                    ),
                    retrieval=bedrock.CfnFlowVersion.RetrievalFlowNodeConfigurationProperty(
                        service_configuration=bedrock.CfnFlowVersion.RetrievalFlowNodeServiceConfigurationProperty(
                            s3=bedrock.CfnFlowVersion.RetrievalFlowNodeS3ConfigurationProperty(
                                bucket_name="bucketName"
                            )
                        )
                    ),
                    storage=bedrock.CfnFlowVersion.StorageFlowNodeConfigurationProperty(
                        service_configuration=bedrock.CfnFlowVersion.StorageFlowNodeServiceConfigurationProperty(
                            s3=bedrock.CfnFlowVersion.StorageFlowNodeS3ConfigurationProperty(
                                bucket_name="bucketName"
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f9cd0cd05b1dd568a6031e23179baa28ec1045e1a40b1eda1f4240278fd28f94)
                check_type(argname="argument agent", value=agent, expected_type=type_hints["agent"])
                check_type(argname="argument collector", value=collector, expected_type=type_hints["collector"])
                check_type(argname="argument condition", value=condition, expected_type=type_hints["condition"])
                check_type(argname="argument input", value=input, expected_type=type_hints["input"])
                check_type(argname="argument iterator", value=iterator, expected_type=type_hints["iterator"])
                check_type(argname="argument knowledge_base", value=knowledge_base, expected_type=type_hints["knowledge_base"])
                check_type(argname="argument lambda_function", value=lambda_function, expected_type=type_hints["lambda_function"])
                check_type(argname="argument lex", value=lex, expected_type=type_hints["lex"])
                check_type(argname="argument output", value=output, expected_type=type_hints["output"])
                check_type(argname="argument prompt", value=prompt, expected_type=type_hints["prompt"])
                check_type(argname="argument retrieval", value=retrieval, expected_type=type_hints["retrieval"])
                check_type(argname="argument storage", value=storage, expected_type=type_hints["storage"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if agent is not None:
                self._values["agent"] = agent
            if collector is not None:
                self._values["collector"] = collector
            if condition is not None:
                self._values["condition"] = condition
            if input is not None:
                self._values["input"] = input
            if iterator is not None:
                self._values["iterator"] = iterator
            if knowledge_base is not None:
                self._values["knowledge_base"] = knowledge_base
            if lambda_function is not None:
                self._values["lambda_function"] = lambda_function
            if lex is not None:
                self._values["lex"] = lex
            if output is not None:
                self._values["output"] = output
            if prompt is not None:
                self._values["prompt"] = prompt
            if retrieval is not None:
                self._values["retrieval"] = retrieval
            if storage is not None:
                self._values["storage"] = storage

        @builtins.property
        def agent(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.AgentFlowNodeConfigurationProperty"]]:
            '''Contains configurations for an agent node in your flow.

            Invokes an alias of an agent and returns the response.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-agent
            '''
            result = self._values.get("agent")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.AgentFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def collector(self) -> typing.Any:
            '''Contains configurations for a collector node in your flow.

            Collects an iteration of inputs and consolidates them into an array of outputs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-collector
            '''
            result = self._values.get("collector")
            return typing.cast(typing.Any, result)

        @builtins.property
        def condition(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.ConditionFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a Condition node in your flow.

            Defines conditions that lead to different branches of the flow.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-condition
            '''
            result = self._values.get("condition")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.ConditionFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def input(self) -> typing.Any:
            '''Contains configurations for an input flow node in your flow.

            The first node in the flow. ``inputs`` can't be specified for this node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-input
            '''
            result = self._values.get("input")
            return typing.cast(typing.Any, result)

        @builtins.property
        def iterator(self) -> typing.Any:
            '''Contains configurations for an iterator node in your flow.

            Takes an input that is an array and iteratively sends each item of the array as an output to the following node. The size of the array is also returned in the output.

            The output flow node at the end of the flow iteration will return a response for each member of the array. To return only one response, you can include a collector node downstream from the iterator node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-iterator
            '''
            result = self._values.get("iterator")
            return typing.cast(typing.Any, result)

        @builtins.property
        def knowledge_base(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.KnowledgeBaseFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a knowledge base node in your flow.

            Queries a knowledge base and returns the retrieved results or generated response.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-knowledgebase
            '''
            result = self._values.get("knowledge_base")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.KnowledgeBaseFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def lambda_function(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.LambdaFunctionFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a Lambda function node in your flow.

            Invokes an AWS Lambda function.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-lambdafunction
            '''
            result = self._values.get("lambda_function")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.LambdaFunctionFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def lex(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.LexFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a Lex node in your flow.

            Invokes an Amazon Lex bot to identify the intent of the input and return the intent as the output.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-lex
            '''
            result = self._values.get("lex")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.LexFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def output(self) -> typing.Any:
            '''Contains configurations for an output flow node in your flow.

            The last node in the flow. ``outputs`` can't be specified for this node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-output
            '''
            result = self._values.get("output")
            return typing.cast(typing.Any, result)

        @builtins.property
        def prompt(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a prompt node in your flow.

            Runs a prompt and generates the model response as the output. You can use a prompt from Prompt management or you can configure one in this node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-prompt
            '''
            result = self._values.get("prompt")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def retrieval(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.RetrievalFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a Retrieval node in your flow.

            Retrieves data from an Amazon S3 location and returns it as the output.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-retrieval
            '''
            result = self._values.get("retrieval")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.RetrievalFlowNodeConfigurationProperty"]], result)

        @builtins.property
        def storage(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.StorageFlowNodeConfigurationProperty"]]:
            '''Contains configurations for a Storage node in your flow.

            Stores an input in an Amazon S3 location.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeconfiguration.html#cfn-bedrock-flowversion-flownodeconfiguration-storage
            '''
            result = self._values.get("storage")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.StorageFlowNodeConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.FlowNodeInputProperty",
        jsii_struct_bases=[],
        name_mapping={"expression": "expression", "name": "name", "type": "type"},
    )
    class FlowNodeInputProperty:
        def __init__(
            self,
            *,
            expression: builtins.str,
            name: builtins.str,
            type: builtins.str,
        ) -> None:
            '''Contains configurations for an input to a node.

            :param expression: An expression that formats the input for the node. For an explanation of how to create expressions, see `Expressions in Prompt flows in Amazon Bedrock <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-expressions.html>`_ .
            :param name: A name for the input that you can reference.
            :param type: The data type of the input. If the input doesn't match this type at runtime, a validation error will be thrown.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeinput.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_node_input_property = bedrock.CfnFlowVersion.FlowNodeInputProperty(
                    expression="expression",
                    name="name",
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bcae87e7d9ed27624a9d2ce7849f806ab7a70bdeba3d31dac17070ca4c7fca80)
                check_type(argname="argument expression", value=expression, expected_type=type_hints["expression"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "expression": expression,
                "name": name,
                "type": type,
            }

        @builtins.property
        def expression(self) -> builtins.str:
            '''An expression that formats the input for the node.

            For an explanation of how to create expressions, see `Expressions in Prompt flows in Amazon Bedrock <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-expressions.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeinput.html#cfn-bedrock-flowversion-flownodeinput-expression
            '''
            result = self._values.get("expression")
            assert result is not None, "Required property 'expression' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the input that you can reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeinput.html#cfn-bedrock-flowversion-flownodeinput-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def type(self) -> builtins.str:
            '''The data type of the input.

            If the input doesn't match this type at runtime, a validation error will be thrown.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeinput.html#cfn-bedrock-flowversion-flownodeinput-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowNodeInputProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.FlowNodeOutputProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "type": "type"},
    )
    class FlowNodeOutputProperty:
        def __init__(self, *, name: builtins.str, type: builtins.str) -> None:
            '''Contains configurations for an output from a node.

            :param name: A name for the output that you can reference.
            :param type: The data type of the output. If the output doesn't match this type at runtime, a validation error will be thrown.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeoutput.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                flow_node_output_property = bedrock.CfnFlowVersion.FlowNodeOutputProperty(
                    name="name",
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2177b4d29b8a3e9a0db17938038173c60a52042601fda8765681b486840455e9)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "type": type,
            }

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the output that you can reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeoutput.html#cfn-bedrock-flowversion-flownodeoutput-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def type(self) -> builtins.str:
            '''The data type of the output.

            If the output doesn't match this type at runtime, a validation error will be thrown.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownodeoutput.html#cfn-bedrock-flowversion-flownodeoutput-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowNodeOutputProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.FlowNodeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "type": "type",
            "configuration": "configuration",
            "inputs": "inputs",
            "outputs": "outputs",
        },
    )
    class FlowNodeProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            type: builtins.str,
            configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.FlowNodeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            inputs: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.FlowNodeInputProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            outputs: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.FlowNodeOutputProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''Contains configurations about a node in the flow.

            :param name: A name for the node.
            :param type: The type of node. This value must match the name of the key that you provide in the configuration you provide in the ``FlowNodeConfiguration`` field.
            :param configuration: Contains configurations for the node.
            :param inputs: An array of objects, each of which contains information about an input into the node.
            :param outputs: A list of objects, each of which contains information about an output from the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownode.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                # collector: Any
                # input: Any
                # iterator: Any
                # output: Any
                
                flow_node_property = bedrock.CfnFlowVersion.FlowNodeProperty(
                    name="name",
                    type="type",
                
                    # the properties below are optional
                    configuration=bedrock.CfnFlowVersion.FlowNodeConfigurationProperty(
                        agent=bedrock.CfnFlowVersion.AgentFlowNodeConfigurationProperty(
                            agent_alias_arn="agentAliasArn"
                        ),
                        collector=collector,
                        condition=bedrock.CfnFlowVersion.ConditionFlowNodeConfigurationProperty(
                            conditions=[bedrock.CfnFlowVersion.FlowConditionProperty(
                                name="name",
                
                                # the properties below are optional
                                expression="expression"
                            )]
                        ),
                        input=input,
                        iterator=iterator,
                        knowledge_base=bedrock.CfnFlowVersion.KnowledgeBaseFlowNodeConfigurationProperty(
                            knowledge_base_id="knowledgeBaseId",
                
                            # the properties below are optional
                            guardrail_configuration=bedrock.CfnFlowVersion.GuardrailConfigurationProperty(
                                guardrail_identifier="guardrailIdentifier",
                                guardrail_version="guardrailVersion"
                            ),
                            model_id="modelId"
                        ),
                        lambda_function=bedrock.CfnFlowVersion.LambdaFunctionFlowNodeConfigurationProperty(
                            lambda_arn="lambdaArn"
                        ),
                        lex=bedrock.CfnFlowVersion.LexFlowNodeConfigurationProperty(
                            bot_alias_arn="botAliasArn",
                            locale_id="localeId"
                        ),
                        output=output,
                        prompt=bedrock.CfnFlowVersion.PromptFlowNodeConfigurationProperty(
                            source_configuration=bedrock.CfnFlowVersion.PromptFlowNodeSourceConfigurationProperty(
                                inline=bedrock.CfnFlowVersion.PromptFlowNodeInlineConfigurationProperty(
                                    model_id="modelId",
                                    template_configuration=bedrock.CfnFlowVersion.PromptTemplateConfigurationProperty(
                                        text=bedrock.CfnFlowVersion.TextPromptTemplateConfigurationProperty(
                                            text="text",
                
                                            # the properties below are optional
                                            input_variables=[bedrock.CfnFlowVersion.PromptInputVariableProperty(
                                                name="name"
                                            )]
                                        )
                                    ),
                                    template_type="templateType",
                
                                    # the properties below are optional
                                    inference_configuration=bedrock.CfnFlowVersion.PromptInferenceConfigurationProperty(
                                        text=bedrock.CfnFlowVersion.PromptModelInferenceConfigurationProperty(
                                            max_tokens=123,
                                            stop_sequences=["stopSequences"],
                                            temperature=123,
                                            top_p=123
                                        )
                                    )
                                ),
                                resource=bedrock.CfnFlowVersion.PromptFlowNodeResourceConfigurationProperty(
                                    prompt_arn="promptArn"
                                )
                            ),
                
                            # the properties below are optional
                            guardrail_configuration=bedrock.CfnFlowVersion.GuardrailConfigurationProperty(
                                guardrail_identifier="guardrailIdentifier",
                                guardrail_version="guardrailVersion"
                            )
                        ),
                        retrieval=bedrock.CfnFlowVersion.RetrievalFlowNodeConfigurationProperty(
                            service_configuration=bedrock.CfnFlowVersion.RetrievalFlowNodeServiceConfigurationProperty(
                                s3=bedrock.CfnFlowVersion.RetrievalFlowNodeS3ConfigurationProperty(
                                    bucket_name="bucketName"
                                )
                            )
                        ),
                        storage=bedrock.CfnFlowVersion.StorageFlowNodeConfigurationProperty(
                            service_configuration=bedrock.CfnFlowVersion.StorageFlowNodeServiceConfigurationProperty(
                                s3=bedrock.CfnFlowVersion.StorageFlowNodeS3ConfigurationProperty(
                                    bucket_name="bucketName"
                                )
                            )
                        )
                    ),
                    inputs=[bedrock.CfnFlowVersion.FlowNodeInputProperty(
                        expression="expression",
                        name="name",
                        type="type"
                    )],
                    outputs=[bedrock.CfnFlowVersion.FlowNodeOutputProperty(
                        name="name",
                        type="type"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9b55ad1cfa9b68dc6dff2440039243220846238d1a04de053105a499f620c973)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument configuration", value=configuration, expected_type=type_hints["configuration"])
                check_type(argname="argument inputs", value=inputs, expected_type=type_hints["inputs"])
                check_type(argname="argument outputs", value=outputs, expected_type=type_hints["outputs"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "type": type,
            }
            if configuration is not None:
                self._values["configuration"] = configuration
            if inputs is not None:
                self._values["inputs"] = inputs
            if outputs is not None:
                self._values["outputs"] = outputs

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownode.html#cfn-bedrock-flowversion-flownode-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of node.

            This value must match the name of the key that you provide in the configuration you provide in the ``FlowNodeConfiguration`` field.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownode.html#cfn-bedrock-flowversion-flownode-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowNodeConfigurationProperty"]]:
            '''Contains configurations for the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownode.html#cfn-bedrock-flowversion-flownode-configuration
            '''
            result = self._values.get("configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowNodeConfigurationProperty"]], result)

        @builtins.property
        def inputs(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowNodeInputProperty"]]]]:
            '''An array of objects, each of which contains information about an input into the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownode.html#cfn-bedrock-flowversion-flownode-inputs
            '''
            result = self._values.get("inputs")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowNodeInputProperty"]]]], result)

        @builtins.property
        def outputs(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowNodeOutputProperty"]]]]:
            '''A list of objects, each of which contains information about an output from the node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-flownode.html#cfn-bedrock-flowversion-flownode-outputs
            '''
            result = self._values.get("outputs")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.FlowNodeOutputProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FlowNodeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.GuardrailConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "guardrail_identifier": "guardrailIdentifier",
            "guardrail_version": "guardrailVersion",
        },
    )
    class GuardrailConfigurationProperty:
        def __init__(
            self,
            *,
            guardrail_identifier: typing.Optional[builtins.str] = None,
            guardrail_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Configuration information for a guardrail that you use with the `Converse <https://docs.aws.amazon.com/bedrock/latest/APIReference/API_runtime_Converse.html>`_ operation.

            :param guardrail_identifier: The identifier for the guardrail.
            :param guardrail_version: The version of the guardrail.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-guardrailconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                guardrail_configuration_property = bedrock.CfnFlowVersion.GuardrailConfigurationProperty(
                    guardrail_identifier="guardrailIdentifier",
                    guardrail_version="guardrailVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e730caca77d30abef7bb7cb2c01702883c895d80378652c3f9f4bd88643dc186)
                check_type(argname="argument guardrail_identifier", value=guardrail_identifier, expected_type=type_hints["guardrail_identifier"])
                check_type(argname="argument guardrail_version", value=guardrail_version, expected_type=type_hints["guardrail_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if guardrail_identifier is not None:
                self._values["guardrail_identifier"] = guardrail_identifier
            if guardrail_version is not None:
                self._values["guardrail_version"] = guardrail_version

        @builtins.property
        def guardrail_identifier(self) -> typing.Optional[builtins.str]:
            '''The identifier for the guardrail.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-guardrailconfiguration.html#cfn-bedrock-flowversion-guardrailconfiguration-guardrailidentifier
            '''
            result = self._values.get("guardrail_identifier")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def guardrail_version(self) -> typing.Optional[builtins.str]:
            '''The version of the guardrail.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-guardrailconfiguration.html#cfn-bedrock-flowversion-guardrailconfiguration-guardrailversion
            '''
            result = self._values.get("guardrail_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GuardrailConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.KnowledgeBaseFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "knowledge_base_id": "knowledgeBaseId",
            "guardrail_configuration": "guardrailConfiguration",
            "model_id": "modelId",
        },
    )
    class KnowledgeBaseFlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            knowledge_base_id: builtins.str,
            guardrail_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.GuardrailConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            model_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains configurations for a knowledge base node in a flow.

            This node takes a query as the input and returns, as the output, the retrieved responses directly (as an array) or a response generated based on the retrieved responses. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param knowledge_base_id: The unique identifier of the knowledge base to query.
            :param guardrail_configuration: Contains configurations for a guardrail to apply during query and response generation for the knowledge base in this configuration.
            :param model_id: The unique identifier of the model or `inference profile <https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html>`_ to use to generate a response from the query results. Omit this field if you want to return the retrieved results as an array.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-knowledgebaseflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                knowledge_base_flow_node_configuration_property = bedrock.CfnFlowVersion.KnowledgeBaseFlowNodeConfigurationProperty(
                    knowledge_base_id="knowledgeBaseId",
                
                    # the properties below are optional
                    guardrail_configuration=bedrock.CfnFlowVersion.GuardrailConfigurationProperty(
                        guardrail_identifier="guardrailIdentifier",
                        guardrail_version="guardrailVersion"
                    ),
                    model_id="modelId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__17b9fe30408d688aaf08f87e21950af81c63d804d7021f8f66de0b47e6aa51be)
                check_type(argname="argument knowledge_base_id", value=knowledge_base_id, expected_type=type_hints["knowledge_base_id"])
                check_type(argname="argument guardrail_configuration", value=guardrail_configuration, expected_type=type_hints["guardrail_configuration"])
                check_type(argname="argument model_id", value=model_id, expected_type=type_hints["model_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "knowledge_base_id": knowledge_base_id,
            }
            if guardrail_configuration is not None:
                self._values["guardrail_configuration"] = guardrail_configuration
            if model_id is not None:
                self._values["model_id"] = model_id

        @builtins.property
        def knowledge_base_id(self) -> builtins.str:
            '''The unique identifier of the knowledge base to query.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-knowledgebaseflownodeconfiguration.html#cfn-bedrock-flowversion-knowledgebaseflownodeconfiguration-knowledgebaseid
            '''
            result = self._values.get("knowledge_base_id")
            assert result is not None, "Required property 'knowledge_base_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def guardrail_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.GuardrailConfigurationProperty"]]:
            '''Contains configurations for a guardrail to apply during query and response generation for the knowledge base in this configuration.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-knowledgebaseflownodeconfiguration.html#cfn-bedrock-flowversion-knowledgebaseflownodeconfiguration-guardrailconfiguration
            '''
            result = self._values.get("guardrail_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.GuardrailConfigurationProperty"]], result)

        @builtins.property
        def model_id(self) -> typing.Optional[builtins.str]:
            '''The unique identifier of the model or `inference profile <https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html>`_ to use to generate a response from the query results. Omit this field if you want to return the retrieved results as an array.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-knowledgebaseflownodeconfiguration.html#cfn-bedrock-flowversion-knowledgebaseflownodeconfiguration-modelid
            '''
            result = self._values.get("model_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KnowledgeBaseFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.LambdaFunctionFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"lambda_arn": "lambdaArn"},
    )
    class LambdaFunctionFlowNodeConfigurationProperty:
        def __init__(self, *, lambda_arn: builtins.str) -> None:
            '''Contains configurations for a Lambda function node in the flow.

            You specify the Lambda function to invoke and the inputs into the function. The output is the response that is defined in the Lambda function. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param lambda_arn: The Amazon Resource Name (ARN) of the Lambda function to invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-lambdafunctionflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                lambda_function_flow_node_configuration_property = bedrock.CfnFlowVersion.LambdaFunctionFlowNodeConfigurationProperty(
                    lambda_arn="lambdaArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__67ef853cf1903ab6b459d44a62d988a959ee29d637fb6df7b7de8c8c44c3da7d)
                check_type(argname="argument lambda_arn", value=lambda_arn, expected_type=type_hints["lambda_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "lambda_arn": lambda_arn,
            }

        @builtins.property
        def lambda_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the Lambda function to invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-lambdafunctionflownodeconfiguration.html#cfn-bedrock-flowversion-lambdafunctionflownodeconfiguration-lambdaarn
            '''
            result = self._values.get("lambda_arn")
            assert result is not None, "Required property 'lambda_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LambdaFunctionFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.LexFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"bot_alias_arn": "botAliasArn", "locale_id": "localeId"},
    )
    class LexFlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            bot_alias_arn: builtins.str,
            locale_id: builtins.str,
        ) -> None:
            '''Contains configurations for a Lex node in the flow.

            You specify a Amazon Lex bot to invoke. This node takes an utterance as the input and returns as the output the intent identified by the Amazon Lex bot. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param bot_alias_arn: The Amazon Resource Name (ARN) of the Amazon Lex bot alias to invoke.
            :param locale_id: The Region to invoke the Amazon Lex bot in.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-lexflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                lex_flow_node_configuration_property = bedrock.CfnFlowVersion.LexFlowNodeConfigurationProperty(
                    bot_alias_arn="botAliasArn",
                    locale_id="localeId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5ed3b9d7d58ce0e3983c2d3d2353ab30d7ce6785a86992996e15622f29e39a0d)
                check_type(argname="argument bot_alias_arn", value=bot_alias_arn, expected_type=type_hints["bot_alias_arn"])
                check_type(argname="argument locale_id", value=locale_id, expected_type=type_hints["locale_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "bot_alias_arn": bot_alias_arn,
                "locale_id": locale_id,
            }

        @builtins.property
        def bot_alias_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the Amazon Lex bot alias to invoke.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-lexflownodeconfiguration.html#cfn-bedrock-flowversion-lexflownodeconfiguration-botaliasarn
            '''
            result = self._values.get("bot_alias_arn")
            assert result is not None, "Required property 'bot_alias_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def locale_id(self) -> builtins.str:
            '''The Region to invoke the Amazon Lex bot in.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-lexflownodeconfiguration.html#cfn-bedrock-flowversion-lexflownodeconfiguration-localeid
            '''
            result = self._values.get("locale_id")
            assert result is not None, "Required property 'locale_id' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LexFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.PromptFlowNodeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "source_configuration": "sourceConfiguration",
            "guardrail_configuration": "guardrailConfiguration",
        },
    )
    class PromptFlowNodeConfigurationProperty:
        def __init__(
            self,
            *,
            source_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.PromptFlowNodeSourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
            guardrail_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.GuardrailConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains configurations for a prompt node in the flow.

            You can use a prompt from Prompt management or you can define one in this node. If the prompt contains variables, the inputs into this node will fill in the variables. The output from this node is the response generated by the model. For more information, see `Node types in Amazon Bedrock works <https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html>`_ in the Amazon Bedrock User Guide.

            :param source_configuration: Specifies whether the prompt is from Prompt management or defined inline.
            :param guardrail_configuration: Contains configurations for a guardrail to apply to the prompt in this node and the response generated from it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownodeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_flow_node_configuration_property = bedrock.CfnFlowVersion.PromptFlowNodeConfigurationProperty(
                    source_configuration=bedrock.CfnFlowVersion.PromptFlowNodeSourceConfigurationProperty(
                        inline=bedrock.CfnFlowVersion.PromptFlowNodeInlineConfigurationProperty(
                            model_id="modelId",
                            template_configuration=bedrock.CfnFlowVersion.PromptTemplateConfigurationProperty(
                                text=bedrock.CfnFlowVersion.TextPromptTemplateConfigurationProperty(
                                    text="text",
                
                                    # the properties below are optional
                                    input_variables=[bedrock.CfnFlowVersion.PromptInputVariableProperty(
                                        name="name"
                                    )]
                                )
                            ),
                            template_type="templateType",
                
                            # the properties below are optional
                            inference_configuration=bedrock.CfnFlowVersion.PromptInferenceConfigurationProperty(
                                text=bedrock.CfnFlowVersion.PromptModelInferenceConfigurationProperty(
                                    max_tokens=123,
                                    stop_sequences=["stopSequences"],
                                    temperature=123,
                                    top_p=123
                                )
                            )
                        ),
                        resource=bedrock.CfnFlowVersion.PromptFlowNodeResourceConfigurationProperty(
                            prompt_arn="promptArn"
                        )
                    ),
                
                    # the properties below are optional
                    guardrail_configuration=bedrock.CfnFlowVersion.GuardrailConfigurationProperty(
                        guardrail_identifier="guardrailIdentifier",
                        guardrail_version="guardrailVersion"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__62c4630f7833caba0eae680e960eadd4fb8eef6ef080839f7a47f079f386a1be)
                check_type(argname="argument source_configuration", value=source_configuration, expected_type=type_hints["source_configuration"])
                check_type(argname="argument guardrail_configuration", value=guardrail_configuration, expected_type=type_hints["guardrail_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "source_configuration": source_configuration,
            }
            if guardrail_configuration is not None:
                self._values["guardrail_configuration"] = guardrail_configuration

        @builtins.property
        def source_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptFlowNodeSourceConfigurationProperty"]:
            '''Specifies whether the prompt is from Prompt management or defined inline.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownodeconfiguration.html#cfn-bedrock-flowversion-promptflownodeconfiguration-sourceconfiguration
            '''
            result = self._values.get("source_configuration")
            assert result is not None, "Required property 'source_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptFlowNodeSourceConfigurationProperty"], result)

        @builtins.property
        def guardrail_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.GuardrailConfigurationProperty"]]:
            '''Contains configurations for a guardrail to apply to the prompt in this node and the response generated from it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownodeconfiguration.html#cfn-bedrock-flowversion-promptflownodeconfiguration-guardrailconfiguration
            '''
            result = self._values.get("guardrail_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.GuardrailConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptFlowNodeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.PromptFlowNodeInlineConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "model_id": "modelId",
            "template_configuration": "templateConfiguration",
            "template_type": "templateType",
            "inference_configuration": "inferenceConfiguration",
        },
    )
    class PromptFlowNodeInlineConfigurationProperty:
        def __init__(
            self,
            *,
            model_id: builtins.str,
            template_configuration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.PromptTemplateConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
            template_type: builtins.str,
            inference_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.PromptInferenceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains configurations for a prompt defined inline in the node.

            :param model_id: The unique identifier of the model or `inference profile <https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html>`_ to run inference with.
            :param template_configuration: Contains a prompt and variables in the prompt that can be replaced with values at runtime.
            :param template_type: The type of prompt template.
            :param inference_configuration: Contains inference configurations for the prompt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownodeinlineconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_flow_node_inline_configuration_property = bedrock.CfnFlowVersion.PromptFlowNodeInlineConfigurationProperty(
                    model_id="modelId",
                    template_configuration=bedrock.CfnFlowVersion.PromptTemplateConfigurationProperty(
                        text=bedrock.CfnFlowVersion.TextPromptTemplateConfigurationProperty(
                            text="text",
                
                            # the properties below are optional
                            input_variables=[bedrock.CfnFlowVersion.PromptInputVariableProperty(
                                name="name"
                            )]
                        )
                    ),
                    template_type="templateType",
                
                    # the properties below are optional
                    inference_configuration=bedrock.CfnFlowVersion.PromptInferenceConfigurationProperty(
                        text=bedrock.CfnFlowVersion.PromptModelInferenceConfigurationProperty(
                            max_tokens=123,
                            stop_sequences=["stopSequences"],
                            temperature=123,
                            top_p=123
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c1c976b436e2845b9e426b622a34aa7c4ec613fa36f03e1790d10a1444497213)
                check_type(argname="argument model_id", value=model_id, expected_type=type_hints["model_id"])
                check_type(argname="argument template_configuration", value=template_configuration, expected_type=type_hints["template_configuration"])
                check_type(argname="argument template_type", value=template_type, expected_type=type_hints["template_type"])
                check_type(argname="argument inference_configuration", value=inference_configuration, expected_type=type_hints["inference_configuration"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "model_id": model_id,
                "template_configuration": template_configuration,
                "template_type": template_type,
            }
            if inference_configuration is not None:
                self._values["inference_configuration"] = inference_configuration

        @builtins.property
        def model_id(self) -> builtins.str:
            '''The unique identifier of the model or `inference profile <https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html>`_ to run inference with.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownodeinlineconfiguration.html#cfn-bedrock-flowversion-promptflownodeinlineconfiguration-modelid
            '''
            result = self._values.get("model_id")
            assert result is not None, "Required property 'model_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def template_configuration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptTemplateConfigurationProperty"]:
            '''Contains a prompt and variables in the prompt that can be replaced with values at runtime.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownodeinlineconfiguration.html#cfn-bedrock-flowversion-promptflownodeinlineconfiguration-templateconfiguration
            '''
            result = self._values.get("template_configuration")
            assert result is not None, "Required property 'template_configuration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptTemplateConfigurationProperty"], result)

        @builtins.property
        def template_type(self) -> builtins.str:
            '''The type of prompt template.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownodeinlineconfiguration.html#cfn-bedrock-flowversion-promptflownodeinlineconfiguration-templatetype
            '''
            result = self._values.get("template_type")
            assert result is not None, "Required property 'template_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def inference_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptInferenceConfigurationProperty"]]:
            '''Contains inference configurations for the prompt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownodeinlineconfiguration.html#cfn-bedrock-flowversion-promptflownodeinlineconfiguration-inferenceconfiguration
            '''
            result = self._values.get("inference_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptInferenceConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptFlowNodeInlineConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.PromptFlowNodeResourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"prompt_arn": "promptArn"},
    )
    class PromptFlowNodeResourceConfigurationProperty:
        def __init__(self, *, prompt_arn: builtins.str) -> None:
            '''Contains configurations for a prompt from Prompt management to use in a node.

            :param prompt_arn: The Amazon Resource Name (ARN) of the prompt from Prompt management.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownoderesourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_flow_node_resource_configuration_property = bedrock.CfnFlowVersion.PromptFlowNodeResourceConfigurationProperty(
                    prompt_arn="promptArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__886d294a7946757b6a5c696151dc813d33137abbd813b72c05f82c7b68aa6d9b)
                check_type(argname="argument prompt_arn", value=prompt_arn, expected_type=type_hints["prompt_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "prompt_arn": prompt_arn,
            }

        @builtins.property
        def prompt_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the prompt from Prompt management.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownoderesourceconfiguration.html#cfn-bedrock-flowversion-promptflownoderesourceconfiguration-promptarn
            '''
            result = self._values.get("prompt_arn")
            assert result is not None, "Required property 'prompt_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptFlowNodeResourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.PromptFlowNodeSourceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"inline": "inline", "resource": "resource"},
    )
    class PromptFlowNodeSourceConfigurationProperty:
        def __init__(
            self,
            *,
            inline: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.PromptFlowNodeInlineConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            resource: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.PromptFlowNodeResourceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains configurations for a prompt and whether it is from Prompt management or defined inline.

            :param inline: Contains configurations for a prompt that is defined inline.
            :param resource: Contains configurations for a prompt from Prompt management.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownodesourceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_flow_node_source_configuration_property = bedrock.CfnFlowVersion.PromptFlowNodeSourceConfigurationProperty(
                    inline=bedrock.CfnFlowVersion.PromptFlowNodeInlineConfigurationProperty(
                        model_id="modelId",
                        template_configuration=bedrock.CfnFlowVersion.PromptTemplateConfigurationProperty(
                            text=bedrock.CfnFlowVersion.TextPromptTemplateConfigurationProperty(
                                text="text",
                
                                # the properties below are optional
                                input_variables=[bedrock.CfnFlowVersion.PromptInputVariableProperty(
                                    name="name"
                                )]
                            )
                        ),
                        template_type="templateType",
                
                        # the properties below are optional
                        inference_configuration=bedrock.CfnFlowVersion.PromptInferenceConfigurationProperty(
                            text=bedrock.CfnFlowVersion.PromptModelInferenceConfigurationProperty(
                                max_tokens=123,
                                stop_sequences=["stopSequences"],
                                temperature=123,
                                top_p=123
                            )
                        )
                    ),
                    resource=bedrock.CfnFlowVersion.PromptFlowNodeResourceConfigurationProperty(
                        prompt_arn="promptArn"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f552bc73a0bc30c12af1ed647fca52599d616402287d73bb996edc6eed218c4e)
                check_type(argname="argument inline", value=inline, expected_type=type_hints["inline"])
                check_type(argname="argument resource", value=resource, expected_type=type_hints["resource"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if inline is not None:
                self._values["inline"] = inline
            if resource is not None:
                self._values["resource"] = resource

        @builtins.property
        def inline(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptFlowNodeInlineConfigurationProperty"]]:
            '''Contains configurations for a prompt that is defined inline.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownodesourceconfiguration.html#cfn-bedrock-flowversion-promptflownodesourceconfiguration-inline
            '''
            result = self._values.get("inline")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptFlowNodeInlineConfigurationProperty"]], result)

        @builtins.property
        def resource(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptFlowNodeResourceConfigurationProperty"]]:
            '''Contains configurations for a prompt from Prompt management.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptflownodesourceconfiguration.html#cfn-bedrock-flowversion-promptflownodesourceconfiguration-resource
            '''
            result = self._values.get("resource")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptFlowNodeResourceConfigurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptFlowNodeSourceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.PromptInferenceConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"text": "text"},
    )
    class PromptInferenceConfigurationProperty:
        def __init__(
            self,
            *,
            text: typing.Union[_IResolvable_da3f097b, typing.Union["CfnFlowVersion.PromptModelInferenceConfigurationProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''Contains inference configurations for the prompt.

            :param text: Contains inference configurations for a text prompt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptinferenceconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_bedrock as bedrock
                
                prompt_inference_configuration_property = bedrock.CfnFlowVersion.PromptInferenceConfigurationProperty(
                    text=bedrock.CfnFlowVersion.PromptModelInferenceConfigurationProperty(
                        max_tokens=123,
                        stop_sequences=["stopSequences"],
                        temperature=123,
                        top_p=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6b2b87769f13d1e1f10208f7d10cdde89f0494815a6edb978a3d2e7cfae33aec)
                check_type(argname="argument text", value=text, expected_type=type_hints["text"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "text": text,
            }

        @builtins.property
        def text(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptModelInferenceConfigurationProperty"]:
            '''Contains inference configurations for a text prompt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptinferenceconfiguration.html#cfn-bedrock-flowversion-promptinferenceconfiguration-text
            '''
            result = self._values.get("text")
            assert result is not None, "Required property 'text' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnFlowVersion.PromptModelInferenceConfigurationProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PromptInferenceConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_bedrock.CfnFlowVersion.PromptInputVariableProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name"},
    )
    class PromptInputVariableProperty:
        def __init__(self, *, name: typing.Optional[builtins.str] = None) -> None:
            '''Contains information about a variable in the prompt.

            :param name: The name of the variable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-bedrock-flowversion-promptinputvariable.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_