r'''
# AWS Batch Construct Library

This module is part of the [AWS Cloud Development Kit](https://github.com/aws/aws-cdk) project.

AWS Batch is a batch processing tool for efficiently running hundreds of thousands computing jobs in AWS.
Batch can dynamically provision [Amazon EC2](https://aws.amazon.com/ec2/) Instances to meet the resource requirements of submitted jobs
and simplifies the planning, scheduling, and executions of your batch workloads. Batch achieves this through four different resources:

* ComputeEnvironments: Contain the resources used to execute Jobs
* JobDefinitions: Define a type of Job that can be submitted
* JobQueues: Route waiting Jobs to ComputeEnvironments
* SchedulingPolicies: Applied to Queues to control how and when Jobs exit the JobQueue and enter the ComputeEnvironment

`ComputeEnvironment`s can be managed or unmanaged. Batch will automatically provision EC2 Instances in a managed `ComputeEnvironment` and will
not provision any Instances in an unmanaged `ComputeEnvironment`. Managed `ComputeEnvironment`s can use ECS, Fargate, or EKS resources to spin up
EC2 Instances in (ensure your EKS Cluster has [been configured](https://docs.aws.amazon.com/batch/latest/userguide/getting-started-eks.html)
to support a Batch ComputeEnvironment before linking it). You can use Launch Templates and Placement Groups to configure exactly how these resources
will be provisioned.

`JobDefinition`s can use either ECS resources or EKS resources. ECS `JobDefinition`s can use multiple containers to execute distributed workloads.
EKS `JobDefinition`s can only execute a single container. Submitted Jobs use `JobDefinition`s as templates.

`JobQueue`s must link at least one `ComputeEnvironment`. Jobs exit the Queue in FIFO order unless a `SchedulingPolicy` is specified.

`SchedulingPolicy`s tell the Scheduler how to choose which Jobs should be executed next by the ComputeEnvironment.

## Use Cases & Examples

### Cost Optimization

#### Spot Instances

Spot instances are significantly discounted EC2 instances that can be reclaimed at any time by AWS.
Workloads that are fault-tolerant or stateless can take advantage of spot pricing.
To use spot spot instances, set `spot` to `true` on a managed Ec2 or Fargate Compute Environment:

```python
vpc = ec2.Vpc(self, "VPC")
batch.FargateComputeEnvironment(self, "myFargateComputeEnv",
    vpc=vpc,
    spot=True
)
```

Batch allows you to specify the percentage of the on-demand instance that the current spot price
must be to provision the instance using the `spotBidPercentage`.
This defaults to 100%, which is the recommended value.
This value cannot be specified for `FargateComputeEnvironment`s
and only applies to `ManagedEc2EcsComputeEnvironment`s.
The following code configures a Compute Environment to only use spot instances that
are at most 20% the price of the on-demand instance price:

```python
vpc = ec2.Vpc(self, "VPC")
batch.ManagedEc2EcsComputeEnvironment(self, "myEc2ComputeEnv",
    vpc=vpc,
    spot=True,
    spot_bid_percentage=20
)
```

For stateful or otherwise non-interruption-tolerant workflows, omit `spot` or set it to `false` to only provision on-demand instances.

#### Choosing Your Instance Types

Batch allows you to choose the instance types or classes that will run your workload.
This example configures your `ComputeEnvironment` to use only the `M5AD.large` instance:

```python
vpc = ec2.Vpc(self, "VPC")

batch.ManagedEc2EcsComputeEnvironment(self, "myEc2ComputeEnv",
    vpc=vpc,
    instance_types=[ec2.InstanceType.of(ec2.InstanceClass.M5AD, ec2.InstanceSize.LARGE)]
)
```

Batch allows you to specify only the instance class and to let it choose the size, which you can do like this:

```python
# compute_env: batch.IManagedEc2EcsComputeEnvironment
vpc = ec2.Vpc(self, "VPC")
compute_env.add_instance_class(ec2.InstanceClass.M5AD)
# Or, specify it on the constructor:
batch.ManagedEc2EcsComputeEnvironment(self, "myEc2ComputeEnv",
    vpc=vpc,
    instance_classes=[ec2.InstanceClass.R4]
)
```

Unless you explicitly specify `useOptimalInstanceClasses: false`, this compute environment will use `'optimal'` instances,
which tells Batch to pick an instance from the C4, M4, and R4 instance families.
*Note*: Batch does not allow specifying instance types or classes with different architectures.
For example, `InstanceClass.A1` cannot be specified alongside `'optimal'`,
because `A1` uses ARM and `'optimal'` uses x86_64.
You can specify both `'optimal'` alongside several different instance types in the same compute environment:

```python
# vpc: ec2.IVpc


compute_env = batch.ManagedEc2EcsComputeEnvironment(self, "myEc2ComputeEnv",
    instance_types=[ec2.InstanceType.of(ec2.InstanceClass.M5AD, ec2.InstanceSize.LARGE)],
    use_optimal_instance_classes=True,  # default
    vpc=vpc
)
# Note: this is equivalent to specifying
compute_env.add_instance_type(ec2.InstanceType.of(ec2.InstanceClass.M5AD, ec2.InstanceSize.LARGE))
compute_env.add_instance_class(ec2.InstanceClass.C4)
compute_env.add_instance_class(ec2.InstanceClass.M4)
compute_env.add_instance_class(ec2.InstanceClass.R4)
```

#### Configure AMIs

You can configure Amazon Machine Images (AMIs). This example configures your `ComputeEnvironment` to use Amazon Linux 2023.

```python
# vpc: ec2.IVpc


batch.ManagedEc2EcsComputeEnvironment(self, "myEc2ComputeEnv",
    vpc=vpc,
    images=[batch.EcsMachineImage(
        image_type=batch.EcsMachineImageType.ECS_AL2023
    )
    ]
)
```

#### Allocation Strategies

| Allocation Strategy           | Optimized for              | Downsides                     |
| -----------------------       | -------------              | ----------------------------- |
| BEST_FIT                      | Cost                       | May limit throughput          |
| BEST_FIT_PROGRESSIVE          | Throughput                 | May increase cost             |
| SPOT_CAPACITY_OPTIMIZED       | Least interruption         | Only useful on Spot instances |
| SPOT_PRICE_CAPACITY_OPTIMIZED | Least interruption + Price | Only useful on Spot instances |

Batch provides different Allocation Strategies to help it choose which instances to provision.
If your workflow tolerates interruptions, you should enable `spot` on your `ComputeEnvironment`
and use `SPOT_PRICE_CAPACITY_OPTIMIZED` (this is the default if `spot` is enabled).
This will tell Batch to choose the instance types from the ones you’ve specified that have
the most spot capacity available to minimize the chance of interruption and have the lowest price.
To get the most benefit from your spot instances,
you should allow Batch to choose from as many different instance types as possible.
If you only care about minimal interruptions and not want Batch to optimize for cost, use
`SPOT_CAPACITY_OPTIMIZED`. `SPOT_PRICE_CAPACITY_OPTIMIZED` is recommended over `SPOT_CAPACITY_OPTIMIZED`
for most use cases.

If your workflow does not tolerate interruptions and you want to minimize your costs at the expense
of potentially longer waiting times, use `AllocationStrategy.BEST_FIT`.
This will choose the lowest-cost instance type that fits all the jobs in the queue.
If instances of that type are not available,
the queue will not choose a new type; instead, it will wait for the instance to become available.
This can stall your `Queue`, with your compute environment only using part of its max capacity
(or none at all) until the `BEST_FIT` instance becomes available.

If you are running a workflow that does not tolerate interruptions and you want to maximize throughput,
you can use `AllocationStrategy.BEST_FIT_PROGRESSIVE`.
This is the default Allocation Strategy if `spot` is `false` or unspecified.
This strategy will examine the Jobs in the queue and choose whichever instance type meets the requirements
of the jobs in the queue and with the lowest cost per vCPU, just as `BEST_FIT`.
However, if not all of the capacity can be filled with this instance type,
it will choose a new next-best instance type to run any jobs that couldn’t fit into the `BEST_FIT` capacity.
To make the most use of this allocation strategy,
it is recommended to use as many instance classes as is feasible for your workload.
This example shows a `ComputeEnvironment` that uses `BEST_FIT_PROGRESSIVE`
with `'optimal'` and `InstanceClass.M5` instance types:

```python
# vpc: ec2.IVpc


compute_env = batch.ManagedEc2EcsComputeEnvironment(self, "myEc2ComputeEnv",
    vpc=vpc,
    instance_classes=[ec2.InstanceClass.M5]
)
```

This example shows a `ComputeEnvironment` that uses `BEST_FIT` with `'optimal'` instances:

```python
# vpc: ec2.IVpc


compute_env = batch.ManagedEc2EcsComputeEnvironment(self, "myEc2ComputeEnv",
    vpc=vpc,
    allocation_strategy=batch.AllocationStrategy.BEST_FIT
)
```

*Note*: `allocationStrategy` cannot be specified on Fargate Compute Environments.

### Controlling vCPU allocation

You can specify the maximum and minimum vCPUs a managed `ComputeEnvironment` can have at any given time.
Batch will *always* maintain `minvCpus` worth of instances in your ComputeEnvironment, even if it is not executing any jobs,
and even if it is disabled. Batch will scale the instances up to `maxvCpus` worth of instances as
jobs exit the JobQueue and enter the ComputeEnvironment. If you use `AllocationStrategy.BEST_FIT_PROGRESSIVE`,
`AllocationStrategy.SPOT_PRICE_CAPACITY_OPTIMIZED`, or `AllocationStrategy.SPOT_CAPACITY_OPTIMIZED`,
batch may exceed `maxvCpus`; it will never exceed `maxvCpus` by more than a single instance type. This example configures a
`minvCpus` of 10 and a `maxvCpus` of 100:

```python
# vpc: ec2.IVpc


batch.ManagedEc2EcsComputeEnvironment(self, "myEc2ComputeEnv",
    vpc=vpc,
    instance_classes=[ec2.InstanceClass.R4],
    minv_cpus=10,
    maxv_cpus=100
)
```

### Tagging Instances

You can tag any instances launched by your managed EC2 ComputeEnvironments by using the CDK `Tags` API:

```python
# vpc: ec2.IVpc


tag_cE = batch.ManagedEc2EcsComputeEnvironment(self, "CEThatMakesTaggedInstnaces",
    vpc=vpc
)

Tags.of(tag_cE).add("super", "salamander")
```

Unmanaged `ComputeEnvironment`s do not support `maxvCpus` or `minvCpus` because you must provision and manage the instances yourself;
that is, Batch will not scale them up and down as needed.

### Sharing a ComputeEnvironment between multiple JobQueues

Multiple `JobQueue`s can share the same `ComputeEnvironment`.
If multiple Queues are attempting to submit Jobs to the same `ComputeEnvironment`,
Batch will pick the Job from the Queue with the highest priority.
This example creates two `JobQueue`s that share a `ComputeEnvironment`:

```python
# vpc: ec2.IVpc

shared_compute_env = batch.FargateComputeEnvironment(self, "spotEnv",
    vpc=vpc,
    spot=True
)
low_priority_queue = batch.JobQueue(self, "JobQueue",
    priority=1
)
high_priority_queue = batch.JobQueue(self, "JobQueue",
    priority=10
)
low_priority_queue.add_compute_environment(shared_compute_env, 1)
high_priority_queue.add_compute_environment(shared_compute_env, 1)
```

### React to jobs stuck in RUNNABLE state

You can react to jobs stuck in RUNNABLE state by setting a `jobStateTimeLimitActions` in `JobQueue`.
Specifies actions that AWS Batch will take after the job has remained at the head of the queue in the
specified state for longer than the specified time.

```python
batch.JobQueue(self, "JobQueue",
    job_state_time_limit_actions=[batch.JobStateTimeLimitAction(
        action=batch.JobStateTimeLimitActionsAction.CANCEL,
        max_time=cdk.Duration.minutes(10),
        reason=batch.JobStateTimeLimitActionsReason.INSUFFICIENT_INSTANCE_CAPACITY,
        state=batch.JobStateTimeLimitActionsState.RUNNABLE
    )
    ]
)
```

### Fairshare Scheduling

Batch `JobQueue`s execute Jobs submitted to them in FIFO order unless you specify a `SchedulingPolicy`.
FIFO queuing can cause short-running jobs to be starved while long-running jobs fill the compute environment.
To solve this, Jobs can be associated with a share.

Shares consist of a `shareIdentifier` and a `weightFactor`, which is inversely correlated with the vCPU allocated to that share identifier.
When submitting a Job, you can specify its `shareIdentifier` to associate that particular job with that share.
Let's see how the scheduler uses this information to schedule jobs.

For example, if there are two shares defined as follows:

| Share Identifier | Weight Factor |
| ---------------- | ------------- |
| A                | 1             |
| B                | 1             |

The weight factors share the following relationship:

```math
A_{vCpus} / A_{Weight} = B_{vCpus} / B_{Weight}
```

where `BvCpus` is the number of vCPUs allocated to jobs with share identifier `'B'`, and `B_weight` is the weight factor of `B`.

The total number of vCpus allocated to a share is equal to the amount of jobs in that share times the number of vCpus necessary for every job.
Let's say that each A job needs 32 VCpus (`A_requirement` = 32) and each B job needs 64 vCpus (`B_requirement` = 64):

```math
A_{vCpus} = A_{Jobs} * A_{Requirement}
```

```math
B_{vCpus} = B_{Jobs} * B_{Requirement}
```

We have:

```math
A_{vCpus} / A_{Weight} = B_{vCpus} / B_{Weight}
```

```math
A_{Jobs} * A_{Requirement} / A_{Weight} = B_{Jobs} * B_{Requirement} / B_{Weight}
```

```math
A_{Jobs} * 32 / 1 = B_{Jobs} * 64 / 1
```

```math
A_{Jobs} * 32 = B_{Jobs} * 64
```

```math
A_{Jobs} = B_{Jobs} * 2
```

Thus the scheduler will schedule two `'A'` jobs for each `'B'` job.

You can control the weight factors to change these ratios, but note that
weight factors are inversely correlated with the vCpus allocated to the corresponding share.

This example would be configured like this:

```python
fairshare_policy = batch.FairshareSchedulingPolicy(self, "myFairsharePolicy")

fairshare_policy.add_share(
    share_identifier="A",
    weight_factor=1
)
fairshare_policy.add_share(
    share_identifier="B",
    weight_factor=1
)
batch.JobQueue(self, "JobQueue",
    scheduling_policy=fairshare_policy
)
```

*Note*: The scheduler will only consider the current usage of the compute environment unless you specify `shareDecay`.
For example, a `shareDecay` of 5 minutes in the above example means that at any given point in time, twice as many `'A'` jobs
will be scheduled for each `'B'` job, but only for the past 5 minutes. If `'B'` jobs run longer than 5 minutes, then
the scheduler is allowed to put more than two `'A'` jobs for each `'B'` job, because the usage of those long-running
`'B'` jobs will no longer be considered after 5 minutes. `shareDecay` linearly decreases the usage of
long running jobs for calculation purposes. For example if share decay is 60 seconds,
then jobs that run for 30 seconds have their usage considered to be only 50% of what it actually is,
but after a whole minute the scheduler pretends they don't exist for fairness calculations.

The following code specifies a `shareDecay` of 5 minutes:

```python
fairshare_policy = batch.FairshareSchedulingPolicy(self, "myFairsharePolicy",
    share_decay=cdk.Duration.minutes(5)
)
```

If you have high priority jobs that should always be executed as soon as they arrive,
you can define a `computeReservation` to specify the percentage of the
maximum vCPU capacity that should be reserved for shares that are *not in the queue*.
The actual reserved percentage is defined by Batch as:

```math
 (\frac{computeReservation}{100}) ^ {ActiveFairShares}
```

where `ActiveFairShares` is the number of shares for which there exists
at least one job in the queue with a unique share identifier.

This is best illustrated with an example.
Suppose there are three shares with share identifiers `A`, `B` and `C` respectively
and we specify the `computeReservation` to be 75%. The queue is currently empty,
and no other shares exist.

There are no active fair shares, since the queue is empty.
Thus (75/100)^0 = 1 = 100% of the maximum vCpus are reserved for all shares.

A job with identifier `A` enters the queue.

The number of active fair shares is now 1, hence
(75/100)^1 = .75 = 75% of the maximum vCpus are reserved for all shares that do not have the identifier `A`;
for this example, this is `B` and `C`,
(but if jobs are submitted with a share identifier not covered by this fairshare policy, those would be considered just as `B` and `C` are).

Now a `B` job enters the queue. The number of active fair shares is now 2,
so (75/100)^2 = .5625 = 56.25% of the maximum vCpus are reserved for all shares that do not have the identifier `A` or `B`.

Now a second `A` job enters the queue. The number of active fair shares is still 2,
so the percentage reserved is still 56.25%

Now a `C` job enters the queue. The number of active fair shares is now 3,
so (75/100)^3 = .421875 = 42.1875% of the maximum vCpus are reserved for all shares that do not have the identifier `A`, `B`, or `C`.

If there are no other shares that your jobs can specify, this means that 42.1875% of your capacity will never be used!

Now, `A`, `B`, and `C` can only consume 100% - 42.1875% = 57.8125% of the maximum vCpus.
Note that the this percentage is **not** split between `A`, `B`, and `C`.
Instead, the scheduler will use their `weightFactor`s to decide which jobs to schedule;
the only difference is that instead of competing for 100% of the max capacity, jobs compete for 57.8125% of the max capacity.

This example specifies a `computeReservation` of 75% that will behave as explained in the example above:

```python
batch.FairshareSchedulingPolicy(self, "myFairsharePolicy",
    compute_reservation=75,
    shares=[batch.Share(weight_factor=1, share_identifier="A"), batch.Share(weight_factor=0.5, share_identifier="B"), batch.Share(weight_factor=2, share_identifier="C")
    ]
)
```

You can specify a `priority` on your `JobDefinition`s to tell the scheduler to prioritize certain jobs that share the same share identifier.

### Configuring Job Retry Policies

Certain workflows may result in Jobs failing due to intermittent issues.
Jobs can specify retry policies to respond to different failures with different actions.
There are three different ways information about the way a Job exited can be conveyed;

* `exitCode`: the exit code returned from the process executed by the container. Will only match non-zero exit codes.
* `reason`: any middleware errors, like your Docker registry being down.
* `statusReason`: infrastructure errors, most commonly your spot instance being reclaimed.

For most use cases, only one of these will be associated with a particular action at a time.
To specify common `exitCode`s, `reason`s, or `statusReason`s, use the corresponding value from
the `Reason` class. This example shows some common failure reasons:

```python
job_defn = batch.EcsJobDefinition(self, "JobDefn",
    container=batch.EcsEc2ContainerDefinition(self, "containerDefn",
        image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
        memory=cdk.Size.mebibytes(2048),
        cpu=256
    ),
    retry_attempts=5,
    retry_strategies=[
        batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.CANNOT_PULL_CONTAINER)
    ]
)
job_defn.add_retry_strategy(
    batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.SPOT_INSTANCE_RECLAIMED))
job_defn.add_retry_strategy(
    batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.CANNOT_PULL_CONTAINER))
job_defn.add_retry_strategy(
    batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.custom(
        on_exit_code="40*",
        on_reason="some reason"
    )))
```

When specifying a custom reason,
you can specify a glob string to match each of these and react to different failures accordingly.
Up to five different retry strategies can be configured for each Job,
and each strategy can match against some or all of `exitCode`, `reason`, and `statusReason`.
You can optionally configure the number of times a job will be retried,
but you cannot configure different retry counts for different strategies; they all share the same count.
If multiple conditions are specified in a given retry strategy,
they must all match for the action to be taken; the conditions are ANDed together, not ORed.

### Running single-container ECS workflows

Batch can run jobs on ECS or EKS. ECS jobs can be defined as single container or multinode.
This example creates a `JobDefinition` that runs a single container with ECS:

```python
# my_file_system: efs.IFileSystem
# my_job_role: iam.Role

my_file_system.grant_read(my_job_role)

job_defn = batch.EcsJobDefinition(self, "JobDefn",
    container=batch.EcsEc2ContainerDefinition(self, "containerDefn",
        image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
        memory=cdk.Size.mebibytes(2048),
        cpu=256,
        volumes=[batch.EcsVolume.efs(
            name="myVolume",
            file_system=my_file_system,
            container_path="/Volumes/myVolume",
            use_job_role=True
        )],
        job_role=my_job_role
    )
)
```

For workflows that need persistent storage, batch supports mounting `Volume`s to the container.
You can both provision the volume and mount it to the container in a single operation:

```python
# my_file_system: efs.IFileSystem
# job_defn: batch.EcsJobDefinition


job_defn.container.add_volume(batch.EcsVolume.efs(
    name="myVolume",
    file_system=my_file_system,
    container_path="/Volumes/myVolume"
))
```

### Running an ECS workflow with Fargate container

```python
job_defn = batch.EcsJobDefinition(self, "JobDefn",
    container=batch.EcsFargateContainerDefinition(self, "myFargateContainer",
        image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
        memory=cdk.Size.mebibytes(2048),
        cpu=256,
        ephemeral_storage_size=cdk.Size.gibibytes(100),
        fargate_cpu_architecture=ecs.CpuArchitecture.ARM64,
        fargate_operating_system_family=ecs.OperatingSystemFamily.LINUX
    )
)
```

### Secrets

You can expose SecretsManager Secret ARNs or SSM Parameters to your container as environment variables.
The following example defines the `MY_SECRET_ENV_VAR` environment variable that contains the
ARN of the Secret defined by `mySecret`:

```python
# my_secret: secretsmanager.ISecret


job_defn = batch.EcsJobDefinition(self, "JobDefn",
    container=batch.EcsEc2ContainerDefinition(self, "containerDefn",
        image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
        memory=cdk.Size.mebibytes(2048),
        cpu=256,
        secrets={
            "MY_SECRET_ENV_VAR": batch.Secret.from_secrets_manager(my_secret)
        }
    )
)
```

### Running Kubernetes Workflows

Batch also supports running workflows on EKS. The following example creates a `JobDefinition` that runs on EKS:

```python
job_defn = batch.EksJobDefinition(self, "eksf2",
    container=batch.EksContainerDefinition(self, "container",
        image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
        volumes=[batch.EksVolume.empty_dir(
            name="myEmptyDirVolume",
            mount_path="/mount/path",
            medium=batch.EmptyDirMediumType.MEMORY,
            readonly=True,
            size_limit=cdk.Size.mebibytes(2048)
        )]
    )
)
```

You can mount `Volume`s to these containers in a single operation:

```python
# job_defn: batch.EksJobDefinition

job_defn.container.add_volume(batch.EksVolume.empty_dir(
    name="emptyDir",
    mount_path="/Volumes/emptyDir"
))
job_defn.container.add_volume(batch.EksVolume.host_path(
    name="hostPath",
    host_path="/sys",
    mount_path="/Volumes/hostPath"
))
job_defn.container.add_volume(batch.EksVolume.secret(
    name="secret",
    optional=True,
    mount_path="/Volumes/secret",
    secret_name="mySecret"
))
```

### Running Distributed Workflows

Some workflows benefit from parallellization and are most powerful when run in a distributed environment,
such as certain numerical calculations or simulations. Batch offers `MultiNodeJobDefinition`s,
which allow a single job to run on multiple instances in parallel, for this purpose.
Message Passing Interface (MPI) is often used with these workflows.
You must configure your containers to use MPI properly,
but Batch allows different nodes running different containers to communicate easily with one another.
You must configure your containers to use certain environment variables that Batch will provide them,
which lets them know which one is the main node, among other information.
For an in-depth example on using MPI to perform numerical computations on Batch,
see this [blog post](https://aws.amazon.com/blogs/compute/building-a-tightly-coupled-molecular-dynamics-workflow-with-multi-node-parallel-jobs-in-aws-batch/)
In particular, the environment variable that tells the containers which one is the main node can be configured on your `MultiNodeJobDefinition` as follows:

```python
multi_node_job = batch.MultiNodeJobDefinition(self, "JobDefinition",
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.R4, ec2.InstanceSize.LARGE),  # optional, omit to let Batch choose the type for you
    containers=[batch.MultiNodeContainer(
        container=batch.EcsEc2ContainerDefinition(self, "mainMPIContainer",
            image=ecs.ContainerImage.from_registry("yourregsitry.com/yourMPIImage:latest"),
            cpu=256,
            memory=cdk.Size.mebibytes(2048)
        ),
        start_node=0,
        end_node=5
    )]
)
# convenience method
multi_node_job.add_container(
    start_node=6,
    end_node=10,
    container=batch.EcsEc2ContainerDefinition(self, "multiContainer",
        image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
        cpu=256,
        memory=cdk.Size.mebibytes(2048)
    )
)
```

If you need to set the control node to an index other than 0, specify it in directly:

```python
multi_node_job = batch.MultiNodeJobDefinition(self, "JobDefinition",
    main_node=5,
    instance_type=ec2.InstanceType.of(ec2.InstanceClass.R4, ec2.InstanceSize.LARGE)
)
```

### Pass Parameters to a Job

Batch allows you define parameters in your `JobDefinition`, which can be referenced in the container command. For example:

```python
batch.EcsJobDefinition(self, "JobDefn",
    parameters={"echo_param": "foobar"},
    container=batch.EcsEc2ContainerDefinition(self, "containerDefn",
        image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
        memory=cdk.Size.mebibytes(2048),
        cpu=256,
        command=["echo", "Ref::echoParam"
        ]
    )
)
```

### Understanding Progressive Allocation Strategies

AWS Batch uses an [allocation strategy](https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html) to determine what compute resource will efficiently handle incoming job requests. By default, **BEST_FIT** will pick an available compute instance based on vCPU requirements. If none exist, the job will wait until resources become available. However, with this strategy, you may have jobs waiting in the queue unnecessarily despite having more powerful instances available. Below is an example of how that situation might look like:

```plaintext
Compute Environment:

1. m5.xlarge => 4 vCPU
2. m5.2xlarge => 8 vCPU
```

```plaintext
Job Queue:
---------
| A | B |
---------

Job Requirements:
A => 4 vCPU - ALLOCATED TO m5.xlarge
B => 2 vCPU - WAITING
```

In this situation, Batch will allocate **Job A** to compute resource #1 because it is the most cost efficient resource that matches the vCPU requirement. However, with this `BEST_FIT` strategy, **Job B** will not be allocated to our other available compute resource even though it is strong enough to handle it. Instead, it will wait until the first job is finished processing or wait a similar `m5.xlarge` resource to be provisioned.

The alternative would be to use the `BEST_FIT_PROGRESSIVE` strategy in order for the remaining job to be handled in larger containers regardless of vCPU requirement and costs.

### Permissions

You can grant any Principal the `batch:submitJob` permission on both a job definition and a job queue like this:

```python
# vpc: ec2.IVpc


ecs_job = batch.EcsJobDefinition(self, "JobDefn",
    container=batch.EcsEc2ContainerDefinition(self, "containerDefn",
        image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
        memory=cdk.Size.mebibytes(2048),
        cpu=256
    )
)

queue = batch.JobQueue(self, "JobQueue",
    compute_environments=[batch.OrderedComputeEnvironment(
        compute_environment=batch.ManagedEc2EcsComputeEnvironment(self, "managedEc2CE",
            vpc=vpc
        ),
        order=1
    )],
    priority=10
)

user = iam.User(self, "MyUser")
ecs_job.grant_submit_job(user, queue)
```
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    CfnResource as _CfnResource_9df397a6,
    Duration as _Duration_4839e8c3,
    IInspectable as _IInspectable_c2943556,
    IResolvable as _IResolvable_da3f097b,
    IResource as _IResource_c80c4260,
    ITaggable as _ITaggable_36806126,
    ITaggableV2 as _ITaggableV2_4e6798f8,
    Resource as _Resource_45bc6135,
    Size as _Size_7b441c34,
    TagManager as _TagManager_0a598cb3,
    TreeInspector as _TreeInspector_488e0dd5,
)
from ..aws_ec2 import (
    Connections as _Connections_0f31fce8,
    IConnectable as _IConnectable_10015a05,
    ILaunchTemplate as _ILaunchTemplate_f32c0fd7,
    IMachineImage as _IMachineImage_0e8bd50b,
    IPlacementGroup as _IPlacementGroup_583ddf23,
    ISecurityGroup as _ISecurityGroup_acf8a799,
    IVpc as _IVpc_f30d5663,
    InstanceClass as _InstanceClass_85a592e7,
    InstanceType as _InstanceType_f64915b9,
    SubnetSelection as _SubnetSelection_e57d76df,
)
from ..aws_ecs import (
    ContainerImage as _ContainerImage_94af1b43,
    CpuArchitecture as _CpuArchitecture_d27dee43,
    FargatePlatformVersion as _FargatePlatformVersion_55d8be5c,
    LogDriver as _LogDriver_393a21bb,
    LogDriverConfig as _LogDriverConfig_8fb9cc8a,
    OperatingSystemFamily as _OperatingSystemFamily_b23afb56,
)
from ..aws_efs import IFileSystem as _IFileSystem_b2d3a7cb
from ..aws_eks import ICluster as _ICluster_6b2b80df
from ..aws_iam import (
    Grant as _Grant_a7ae64f8,
    IGrantable as _IGrantable_71c4f5de,
    IRole as _IRole_235f5d8e,
)
from ..aws_secretsmanager import ISecret as _ISecret_6e020e6a
from ..aws_ssm import IParameter as _IParameter_509a0f80


@jsii.enum(jsii_type="aws-cdk-lib.aws_batch.Action")
class Action(enum.Enum):
    '''The Action to take when all specified conditions in a RetryStrategy are met.

    :exampleMetadata: infused

    Example::

        job_defn = batch.EcsJobDefinition(self, "JobDefn",
            container=batch.EcsEc2ContainerDefinition(self, "containerDefn",
                image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
                memory=cdk.Size.mebibytes(2048),
                cpu=256
            ),
            retry_attempts=5,
            retry_strategies=[
                batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.CANNOT_PULL_CONTAINER)
            ]
        )
        job_defn.add_retry_strategy(
            batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.SPOT_INSTANCE_RECLAIMED))
        job_defn.add_retry_strategy(
            batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.CANNOT_PULL_CONTAINER))
        job_defn.add_retry_strategy(
            batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.custom(
                on_exit_code="40*",
                on_reason="some reason"
            )))
    '''

    EXIT = "EXIT"
    '''The job will not retry.'''
    RETRY = "RETRY"
    '''The job will retry.

    It can be retried up to the number of times specified in ``retryAttempts``.
    '''


@jsii.enum(jsii_type="aws-cdk-lib.aws_batch.AllocationStrategy")
class AllocationStrategy(enum.Enum):
    '''Determines how this compute environment chooses instances to spawn.

    :see: https://aws.amazon.com/blogs/compute/optimizing-for-cost-availability-and-throughput-by-selecting-your-aws-batch-allocation-strategy/
    :exampleMetadata: infused

    Example::

        # vpc: ec2.IVpc
        
        
        compute_env = batch.ManagedEc2EcsComputeEnvironment(self, "myEc2ComputeEnv",
            vpc=vpc,
            allocation_strategy=batch.AllocationStrategy.BEST_FIT
        )
    '''

    BEST_FIT = "BEST_FIT"
    '''Batch chooses the lowest-cost instance type that fits all the jobs in the queue.

    If instances of that type are not available, the queue will not choose a new type;
    instead, it will wait for the instance to become available.
    This can stall your ``Queue``, with your compute environment only using part of its max capacity
    (or none at all) until the ``BEST_FIT`` instance becomes available.
    This allocation strategy keeps costs lower but can limit scaling.
    ``BEST_FIT`` isn't supported when updating compute environments
    '''
    BEST_FIT_PROGRESSIVE = "BEST_FIT_PROGRESSIVE"
    '''This is the default Allocation Strategy if ``spot`` is ``false`` or unspecified.

    This strategy will examine the Jobs in the queue and choose whichever instance type meets the requirements
    of the jobs in the queue and with the lowest cost per vCPU, just as ``BEST_FIT``.
    However, if not all of the capacity can be filled with this instance type,
    it will choose a new next-best instance type to run any jobs that couldn’t fit into the ``BEST_FIT`` capacity.
    To make the most use of this allocation strategy,
    it is recommended to use as many instance classes as is feasible for your workload.
    '''
    SPOT_CAPACITY_OPTIMIZED = "SPOT_CAPACITY_OPTIMIZED"
    '''If your workflow tolerates interruptions, you should enable ``spot`` on your ``ComputeEnvironment`` and use ``SPOT_CAPACITY_OPTIMIZED`` (this is the default if ``spot`` is enabled).

    This will tell Batch to choose the instance types from the ones you’ve specified that have
    the most spot capacity available to minimize the chance of interruption.
    To get the most benefit from your spot instances,
    you should allow Batch to choose from as many different instance types as possible.
    '''
    SPOT_PRICE_CAPACITY_OPTIMIZED = "SPOT_PRICE_CAPACITY_OPTIMIZED"
    '''The price and capacity optimized allocation strategy looks at both price and capacity to select the Spot Instance pools that are the least likely to be interrupted and have the lowest possible price.

    The Batch team recommends this over ``SPOT_CAPACITY_OPTIMIZED`` in most instances.
    '''


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnComputeEnvironment(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_batch.CfnComputeEnvironment",
):
    '''The ``AWS::Batch::ComputeEnvironment`` resource defines your AWS Batch compute environment.

    You can define ``MANAGED`` or ``UNMANAGED`` compute environments. ``MANAGED`` compute environments can use Amazon EC2 or AWS Fargate resources. ``UNMANAGED`` compute environments can only use EC2 resources. For more information, see `Compute Environments <https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html>`_ in the ** .

    In a managed compute environment, AWS Batch manages the capacity and instance types of the compute resources within the environment. This is based on the compute resource specification that you define or the `launch template <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html>`_ that you specify when you create the compute environment. You can choose either to use EC2 On-Demand Instances and EC2 Spot Instances, or to use Fargate and Fargate Spot capacity in your managed compute environment. You can optionally set a maximum price so that Spot Instances only launch when the Spot Instance price is below a specified percentage of the On-Demand price.
    .. epigraph::

       Multi-node parallel jobs are not supported on Spot Instances.

    In an unmanaged compute environment, you can manage your own EC2 compute resources and have a lot of flexibility with how you configure your compute resources. For example, you can use custom AMI. However, you need to verify that your AMI meets the Amazon ECS container instance AMI specification. For more information, see `container instance AMIs <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container_instance_AMIs.html>`_ in the *Amazon Elastic Container Service Developer Guide* . After you have created your unmanaged compute environment, you can use the `DescribeComputeEnvironments <https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeComputeEnvironments.html>`_ operation to find the Amazon ECS cluster that is associated with it. Then, manually launch your container instances into that Amazon ECS cluster. For more information, see `Launching an Amazon ECS container instance <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_container_instance.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
    .. epigraph::

       To create a compute environment that uses EKS resources, the caller must have permissions to call ``eks:DescribeCluster`` . > AWS Batch doesn't upgrade the AMIs in a compute environment after it's created except under specific conditions. For example, it doesn't automatically update the AMIs when a newer version of the Amazon ECS optimized AMI is available. Therefore, you're responsible for the management of the guest operating system (including updates and security patches) and any additional application software or utilities that you install on the compute resources. There are two ways to use a new AMI for your AWS Batch jobs. The original method is to complete these steps:

       - Create a new compute environment with the new AMI.
       - Add the compute environment to an existing job queue.
       - Remove the earlier compute environment from your job queue.
       - Delete the earlier compute environment.

       In April 2022, AWS Batch added enhanced support for updating compute environments. For example, the ``UpdateComputeEnvironent`` API lets you use the ``ReplaceComputeEnvironment`` property to dynamically update compute environment parameters such as the launch template or instance type without replacement. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .

       To use the enhanced updating of compute environments to update AMIs, follow these rules:

       - Either do not set the `ServiceRole <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-servicerole>`_ property or set it to the *AWSServiceRoleForBatch* service-linked role.
       - Set the `AllocationStrategy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-allocationstrategy>`_ property to ``BEST_FIT_PROGRESSIVE`` or ``SPOT_CAPACITY_OPTIMIZED`` .
       - Set the `ReplaceComputeEnvironment <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-replacecomputeenvironment>`_ property to ``false`` .

       .. epigraph::

          Set the ``ReplaceComputeEnvironment`` property to ``true`` if the compute environment uses the ``BEST_FIT`` allocation strategy. > If the ``ReplaceComputeEnvironment`` property is set to ``false`` , you might receive an error message when you update the CFN template for a compute environment. This issue occurs if the updated ``desiredvcpus`` value is less than the current ``desiredvcpus`` value. As a workaround, delete the ``desiredvcpus`` value from the updated template or use the ``minvcpus`` property to manage the number of vCPUs. For information, see `Error message when you update the ``DesiredvCpus`` setting <https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#error-desired-vcpus-update>`_ .

       - Set the `UpdateToLatestImageVersion <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-updatetolatestimageversion>`_ property to ``true`` . This property is used when you update a compute environment. The `UpdateToLatestImageVersion <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-updatetolatestimageversion>`_ property is ignored when you create a compute environment.
       - Either do not specify an image ID in `ImageId <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-imageid>`_ or `ImageIdOverride <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-ec2configurationobject.html#cfn-batch-computeenvironment-ec2configurationobject-imageidoverride>`_ properties, or in the launch template identified by the `Launch Template <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-launchtemplate>`_ property. In that case AWS Batch will select the latest Amazon ECS optimized AMI supported by AWS Batch at the time the infrastructure update is initiated. Alternatively you can specify the AMI ID in the ``ImageId`` or ``ImageIdOverride`` properties, or the launch template identified by the ``LaunchTemplate`` properties. Changing any of these properties will trigger an infrastructure update.

       If these rules are followed, any update that triggers an infrastructure update will cause the AMI ID to be re-selected. If the `Version <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecification.html#cfn-batch-computeenvironment-launchtemplatespecification-version>`_ property of the `LaunchTemplateSpecification <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecification.html>`_ is set to ``$Latest`` or ``$Default`` , the latest or default version of the launch template will be evaluated up at the time of the infrastructure update, even if the ``LaunchTemplateSpecification`` was not updated.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html
    :cloudformationResource: AWS::Batch::ComputeEnvironment
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_batch as batch
        
        cfn_compute_environment = batch.CfnComputeEnvironment(self, "MyCfnComputeEnvironment",
            type="type",
        
            # the properties below are optional
            compute_environment_name="computeEnvironmentName",
            compute_resources=batch.CfnComputeEnvironment.ComputeResourcesProperty(
                maxv_cpus=123,
                subnets=["subnets"],
                type="type",
        
                # the properties below are optional
                allocation_strategy="allocationStrategy",
                bid_percentage=123,
                desiredv_cpus=123,
                ec2_configuration=[batch.CfnComputeEnvironment.Ec2ConfigurationObjectProperty(
                    image_type="imageType",
        
                    # the properties below are optional
                    image_id_override="imageIdOverride",
                    image_kubernetes_version="imageKubernetesVersion"
                )],
                ec2_key_pair="ec2KeyPair",
                image_id="imageId",
                instance_role="instanceRole",
                instance_types=["instanceTypes"],
                launch_template=batch.CfnComputeEnvironment.LaunchTemplateSpecificationProperty(
                    launch_template_id="launchTemplateId",
                    launch_template_name="launchTemplateName",
                    overrides=[batch.CfnComputeEnvironment.LaunchTemplateSpecificationOverrideProperty(
                        launch_template_id="launchTemplateId",
                        launch_template_name="launchTemplateName",
                        target_instance_types=["targetInstanceTypes"],
                        version="version"
                    )],
                    version="version"
                ),
                minv_cpus=123,
                placement_group="placementGroup",
                security_group_ids=["securityGroupIds"],
                spot_iam_fleet_role="spotIamFleetRole",
                tags={
                    "tags_key": "tags"
                },
                update_to_latest_image_version=False
            ),
            context="context",
            eks_configuration=batch.CfnComputeEnvironment.EksConfigurationProperty(
                eks_cluster_arn="eksClusterArn",
                kubernetes_namespace="kubernetesNamespace"
            ),
            replace_compute_environment=False,
            service_role="serviceRole",
            state="state",
            tags={
                "tags_key": "tags"
            },
            unmanagedv_cpus=123,
            update_policy=batch.CfnComputeEnvironment.UpdatePolicyProperty(
                job_execution_timeout_minutes=123,
                terminate_jobs_on_update=False
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        type: builtins.str,
        compute_environment_name: typing.Optional[builtins.str] = None,
        compute_resources: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnComputeEnvironment.ComputeResourcesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        context: typing.Optional[builtins.str] = None,
        eks_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnComputeEnvironment.EksConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        replace_compute_environment: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        service_role: typing.Optional[builtins.str] = None,
        state: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        unmanagedv_cpus: typing.Optional[jsii.Number] = None,
        update_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnComputeEnvironment.UpdatePolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param type: The type of the compute environment: ``MANAGED`` or ``UNMANAGED`` . For more information, see `Compute Environments <https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html>`_ in the *AWS Batch User Guide* .
        :param compute_environment_name: The name for your compute environment. It can be up to 128 characters long. It can contain uppercase and lowercase letters, numbers, hyphens (-), and underscores (_).
        :param compute_resources: The ComputeResources property type specifies details of the compute resources managed by the compute environment. This parameter is required for managed compute environments. For more information, see `Compute Environments <https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html>`_ in the ** .
        :param context: Reserved.
        :param eks_configuration: The details for the Amazon EKS cluster that supports the compute environment.
        :param replace_compute_environment: Specifies whether the compute environment is replaced if an update is made that requires replacing the instances in the compute environment. The default value is ``true`` . To enable more properties to be updated, set this property to ``false`` . When changing the value of this property to ``false`` , do not change any other properties at the same time. If other properties are changed at the same time, and the change needs to be rolled back but it can't, it's possible for the stack to go into the ``UPDATE_ROLLBACK_FAILED`` state. You can't update a stack that is in the ``UPDATE_ROLLBACK_FAILED`` state. However, if you can continue to roll it back, you can return the stack to its original settings and then try to update it again. For more information, see `Continue rolling back an update <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-continueupdaterollback.html>`_ in the *AWS CloudFormation User Guide* . ``ReplaceComputeEnvironment`` is not applicable for Fargate compute environments. Fargate compute environments are always updated without interruption. The properties that can't be changed without replacing the compute environment are in the ```ComputeResources`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html>`_ property type: ```AllocationStrategy`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-allocationstrategy>`_ , ```BidPercentage`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-bidpercentage>`_ , ```Ec2Configuration`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-ec2configuration>`_ , ```Ec2KeyPair`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-ec2keypair>`_ , ```Ec2KeyPair`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-ec2keypair>`_ , ```ImageId`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-imageid>`_ , ```InstanceRole`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-instancerole>`_ , ```InstanceTypes`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-instancetypes>`_ , ```LaunchTemplate`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-launchtemplate>`_ , ```MaxvCpus`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-maxvcpus>`_ , ```MinvCpus`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-minvcpus>`_ , ```PlacementGroup`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-placementgroup>`_ , ```SecurityGroupIds`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-securitygroupids>`_ , ```Subnets`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-subnets>`_ , `Tags <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-tags>`_ , ```Type`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-type>`_ , and ```UpdateToLatestImageVersion`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-updatetolatestimageversion>`_ . Default: - true
        :param service_role: The full Amazon Resource Name (ARN) of the IAM role that allows AWS Batch to make calls to other AWS services on your behalf. For more information, see `AWS Batch service IAM role <https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html>`_ in the *AWS Batch User Guide* . .. epigraph:: If your account already created the AWS Batch service-linked role, that role is used by default for your compute environment unless you specify a different role here. If the AWS Batch service-linked role doesn't exist in your account, and no role is specified here, the service attempts to create the AWS Batch service-linked role in your account. If your specified role has a path other than ``/`` , then you must specify either the full role ARN (recommended) or prefix the role name with the path. For example, if a role with the name ``bar`` has a path of ``/foo/`` , specify ``/foo/bar`` as the role name. For more information, see `Friendly names and paths <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names>`_ in the *IAM User Guide* . .. epigraph:: Depending on how you created your AWS Batch service role, its ARN might contain the ``service-role`` path prefix. When you only specify the name of the service role, AWS Batch assumes that your ARN doesn't use the ``service-role`` path prefix. Because of this, we recommend that you specify the full ARN of your service role when you create compute environments.
        :param state: The state of the compute environment. If the state is ``ENABLED`` , then the compute environment accepts jobs from a queue and can scale out automatically based on queues. If the state is ``ENABLED`` , then the AWS Batch scheduler can attempt to place jobs from an associated job queue on the compute resources within the environment. If the compute environment is managed, then it can scale its instances out or in automatically, based on the job queue demand. If the state is ``DISABLED`` , then the AWS Batch scheduler doesn't attempt to place jobs within the environment. Jobs in a ``STARTING`` or ``RUNNING`` state continue to progress normally. Managed compute environments in the ``DISABLED`` state don't scale out. .. epigraph:: Compute environments in a ``DISABLED`` state may continue to incur billing charges. To prevent additional charges, turn off and then delete the compute environment. For more information, see `State <https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state>`_ in the *AWS Batch User Guide* . When an instance is idle, the instance scales down to the ``minvCpus`` value. However, the instance size doesn't change. For example, consider a ``c5.8xlarge`` instance with a ``minvCpus`` value of ``4`` and a ``desiredvCpus`` value of ``36`` . This instance doesn't scale down to a ``c5.large`` instance.
        :param tags: The tags applied to the compute environment.
        :param unmanagedv_cpus: The maximum number of vCPUs for an unmanaged compute environment. This parameter is only used for fair-share scheduling to reserve vCPU capacity for new share identifiers. If this parameter isn't provided for a fair-share job queue, no vCPU capacity is reserved. .. epigraph:: This parameter is only supported when the ``type`` parameter is set to ``UNMANAGED`` .
        :param update_policy: Specifies the infrastructure update policy for the compute environment. For more information about infrastructure updates, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3892141757a3fffc40366f3b0a3472c965c97710a96f3bb61a3618dd43af76e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnComputeEnvironmentProps(
            type=type,
            compute_environment_name=compute_environment_name,
            compute_resources=compute_resources,
            context=context,
            eks_configuration=eks_configuration,
            replace_compute_environment=replace_compute_environment,
            service_role=service_role,
            state=state,
            tags=tags,
            unmanagedv_cpus=unmanagedv_cpus,
            update_policy=update_policy,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__181dea136575948e2334f9ffce96122ded2c4edc304f6835d519b8a40e411e4e)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d72cf2605a93f743b750e273ccea4c1be6e5210af9e44b94c6722180f03040d)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrComputeEnvironmentArn")
    def attr_compute_environment_arn(self) -> builtins.str:
        '''Returns the compute environment ARN, such as ``batch: *us-east-1* : *111122223333* :compute-environment/ *ComputeEnvironmentName*`` .

        :cloudformationAttribute: ComputeEnvironmentArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrComputeEnvironmentArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''The type of the compute environment: ``MANAGED`` or ``UNMANAGED`` .'''
        return typing.cast(builtins.str, jsii.get(self, "type"))

    @type.setter
    def type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__af58b33143bf1d8d98a3c10df145b0898751e739fa8cabe1763160d50bc3d730)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "type", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="computeEnvironmentName")
    def compute_environment_name(self) -> typing.Optional[builtins.str]:
        '''The name for your compute environment.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "computeEnvironmentName"))

    @compute_environment_name.setter
    def compute_environment_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__703e5e791be63f407384e398bda6c05261d86a114fb52955ac953213dc086558)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "computeEnvironmentName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="computeResources")
    def compute_resources(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.ComputeResourcesProperty"]]:
        '''The ComputeResources property type specifies details of the compute resources managed by the compute environment.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.ComputeResourcesProperty"]], jsii.get(self, "computeResources"))

    @compute_resources.setter
    def compute_resources(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.ComputeResourcesProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aa0f8cbb024617b335ccf7af8884568ad91340203512578a8d682adfdfce5cd9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "computeResources", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="context")
    def context(self) -> typing.Optional[builtins.str]:
        '''Reserved.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "context"))

    @context.setter
    def context(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__da32f27a124e80b7835c8c0b9d4dd96db23a3745e9e3b0b4850f40ffa88e26fc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "context", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="eksConfiguration")
    def eks_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.EksConfigurationProperty"]]:
        '''The details for the Amazon EKS cluster that supports the compute environment.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.EksConfigurationProperty"]], jsii.get(self, "eksConfiguration"))

    @eks_configuration.setter
    def eks_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.EksConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__896771848c9977e1ce30b550c3e02c1b1e7e047a645b3eb92c3d6d3bcf2ef1cf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "eksConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="replaceComputeEnvironment")
    def replace_compute_environment(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the compute environment is replaced if an update is made that requires replacing the instances in the compute environment.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "replaceComputeEnvironment"))

    @replace_compute_environment.setter
    def replace_compute_environment(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4f7a9dd96ed6e9e51373fc67bf7c1cfa4f89dea1d9ec4216da0ee3a471c21879)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "replaceComputeEnvironment", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="serviceRole")
    def service_role(self) -> typing.Optional[builtins.str]:
        '''The full Amazon Resource Name (ARN) of the IAM role that allows AWS Batch to make calls to other AWS services on your behalf.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "serviceRole"))

    @service_role.setter
    def service_role(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__682bb80f66d8c5016b985338436d49dc6155bd7a11dc1567d1a6478eddb9fc94)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "serviceRole", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="state")
    def state(self) -> typing.Optional[builtins.str]:
        '''The state of the compute environment.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "state"))

    @state.setter
    def state(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30039c901207ae322b43ee9582a8efa812de586638692e722a62baa18097a10d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "state", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags applied to the compute environment.'''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fd461eb02586f268eaf0fc0a27581016dd596744fdebffc9a80af4b63a998bd3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="unmanagedvCpus")
    def unmanagedv_cpus(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of vCPUs for an unmanaged compute environment.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "unmanagedvCpus"))

    @unmanagedv_cpus.setter
    def unmanagedv_cpus(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e874461559bd593606ec74eed9b2200a045433519c0ec8ec8d62a53f967da17a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "unmanagedvCpus", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="updatePolicy")
    def update_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.UpdatePolicyProperty"]]:
        '''Specifies the infrastructure update policy for the compute environment.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.UpdatePolicyProperty"]], jsii.get(self, "updatePolicy"))

    @update_policy.setter
    def update_policy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.UpdatePolicyProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__39e9319d6ed789d3e7f527a145e7be0671311d4703680199b29ca8aef5635e60)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "updatePolicy", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnComputeEnvironment.ComputeResourcesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "maxv_cpus": "maxvCpus",
            "subnets": "subnets",
            "type": "type",
            "allocation_strategy": "allocationStrategy",
            "bid_percentage": "bidPercentage",
            "desiredv_cpus": "desiredvCpus",
            "ec2_configuration": "ec2Configuration",
            "ec2_key_pair": "ec2KeyPair",
            "image_id": "imageId",
            "instance_role": "instanceRole",
            "instance_types": "instanceTypes",
            "launch_template": "launchTemplate",
            "minv_cpus": "minvCpus",
            "placement_group": "placementGroup",
            "security_group_ids": "securityGroupIds",
            "spot_iam_fleet_role": "spotIamFleetRole",
            "tags": "tags",
            "update_to_latest_image_version": "updateToLatestImageVersion",
        },
    )
    class ComputeResourcesProperty:
        def __init__(
            self,
            *,
            maxv_cpus: jsii.Number,
            subnets: typing.Sequence[builtins.str],
            type: builtins.str,
            allocation_strategy: typing.Optional[builtins.str] = None,
            bid_percentage: typing.Optional[jsii.Number] = None,
            desiredv_cpus: typing.Optional[jsii.Number] = None,
            ec2_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnComputeEnvironment.Ec2ConfigurationObjectProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            ec2_key_pair: typing.Optional[builtins.str] = None,
            image_id: typing.Optional[builtins.str] = None,
            instance_role: typing.Optional[builtins.str] = None,
            instance_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            launch_template: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnComputeEnvironment.LaunchTemplateSpecificationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            minv_cpus: typing.Optional[jsii.Number] = None,
            placement_group: typing.Optional[builtins.str] = None,
            security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
            spot_iam_fleet_role: typing.Optional[builtins.str] = None,
            tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
            update_to_latest_image_version: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''Details about the compute resources managed by the compute environment.

            This parameter is required for managed compute environments. For more information, see `Compute Environments <https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html>`_ in the *AWS Batch User Guide* .

            :param maxv_cpus: The maximum number of Amazon EC2 vCPUs that an environment can reach. .. epigraph:: With ``BEST_FIT_PROGRESSIVE`` , ``SPOT_CAPACITY_OPTIMIZED`` and ``SPOT_PRICE_CAPACITY_OPTIMIZED`` (recommended) strategies using On-Demand or Spot Instances, and the ``BEST_FIT`` strategy using Spot Instances, AWS Batch might need to exceed ``maxvCpus`` to meet your capacity requirements. In this event, AWS Batch never exceeds ``maxvCpus`` by more than a single instance.
            :param subnets: The VPC subnets where the compute resources are launched. Fargate compute resources can contain up to 16 subnets. For Fargate compute resources, providing an empty list will be handled as if this parameter wasn't specified and no change is made. For Amazon EC2 compute resources, providing an empty list removes the VPC subnets from the compute resource. For more information, see `VPCs and subnets <https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html>`_ in the *Amazon VPC User Guide* . When updating a compute environment, changing the VPC subnets requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . .. epigraph:: AWS Batch on Amazon EC2 and AWS Batch on Amazon EKS support Local Zones. For more information, see `Local Zones <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones.html#concepts-local-zones>`_ in the *Amazon EC2 User Guide for Linux Instances* , `Amazon EKS and AWS Local Zones <https://docs.aws.amazon.com/eks/latest/userguide/local-zones.html>`_ in the *Amazon EKS User Guide* and `Amazon ECS clusters in Local Zones, Wavelength Zones, and AWS Outposts <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-regions-zones.html#clusters-local-zones>`_ in the *Amazon ECS Developer Guide* . AWS Batch on Fargate doesn't currently support Local Zones.
            :param type: The type of compute environment: ``EC2`` , ``SPOT`` , ``FARGATE`` , or ``FARGATE_SPOT`` . For more information, see `Compute environments <https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html>`_ in the *AWS Batch User Guide* . If you choose ``SPOT`` , you must also specify an Amazon EC2 Spot Fleet role with the ``spotIamFleetRole`` parameter. For more information, see `Amazon EC2 spot fleet role <https://docs.aws.amazon.com/batch/latest/userguide/spot_fleet_IAM_role.html>`_ in the *AWS Batch User Guide* . When updating compute environment, changing the type of a compute environment requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . When updating the type of a compute environment, changing between ``EC2`` and ``SPOT`` or between ``FARGATE`` and ``FARGATE_SPOT`` will initiate an infrastructure update, but if you switch between ``EC2`` and ``FARGATE`` , AWS CloudFormation will create a new compute environment.
            :param allocation_strategy: The allocation strategy to use for the compute resource if not enough instances of the best fitting instance type can be allocated. This might be because of availability of the instance type in the Region or `Amazon EC2 service limits <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html>`_ . For more information, see `Allocation strategies <https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html>`_ in the *AWS Batch User Guide* . When updating a compute environment, changing the allocation strategy requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . ``BEST_FIT`` is not supported when updating a compute environment. .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources, and shouldn't be specified. - **BEST_FIT (default)** - AWS Batch selects an instance type that best fits the needs of the jobs with a preference for the lowest-cost instance type. If additional instances of the selected instance type aren't available, AWS Batch waits for the additional instances to be available. If there aren't enough instances available, or if the user is reaching `Amazon EC2 service limits <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html>`_ then additional jobs aren't run until the currently running jobs have completed. This allocation strategy keeps costs lower but can limit scaling. If you are using Spot Fleets with ``BEST_FIT`` then the Spot Fleet IAM role must be specified. - **BEST_FIT_PROGRESSIVE** - AWS Batch will select additional instance types that are large enough to meet the requirements of the jobs in the queue, with a preference for instance types with a lower cost per unit vCPU. If additional instances of the previously selected instance types aren't available, AWS Batch will select new instance types. - **SPOT_CAPACITY_OPTIMIZED** - AWS Batch will select one or more instance types that are large enough to meet the requirements of the jobs in the queue, with a preference for instance types that are less likely to be interrupted. This allocation strategy is only available for Spot Instance compute resources. - **SPOT_PRICE_CAPACITY_OPTIMIZED** - The price and capacity optimized allocation strategy looks at both price and capacity to select the Spot Instance pools that are the least likely to be interrupted and have the lowest possible price. This allocation strategy is only available for Spot Instance compute resources. .. epigraph:: We recommend that you use ``SPOT_PRICE_CAPACITY_OPTIMIZED`` rather than ``SPOT_CAPACITY_OPTIMIZED`` in most instances. With ``BEST_FIT_PROGRESSIVE`` , ``SPOT_CAPACITY_OPTIMIZED`` , and ``SPOT_PRICE_CAPACITY_OPTIMIZED`` allocation strategies using On-Demand or Spot Instances, and the ``BEST_FIT`` strategy using Spot Instances, AWS Batch might need to go above ``maxvCpus`` to meet your capacity requirements. In this event, AWS Batch never exceeds ``maxvCpus`` by more than a single instance.
            :param bid_percentage: The maximum percentage that a Spot Instance price can be when compared with the On-Demand price for that instance type before instances are launched. For example, if your maximum percentage is 20%, the Spot price must be less than 20% of the current On-Demand price for that Amazon EC2 instance. You always pay the lowest (market) price and never more than your maximum percentage. For most use cases, we recommend leaving this field empty. When updating a compute environment, changing the bid percentage requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.
            :param desiredv_cpus: The desired number of vCPUS in the compute environment. AWS Batch modifies this value between the minimum and maximum values based on job queue demand. .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it. > AWS Batch doesn't support changing the desired number of vCPUs of an existing compute environment. Don't specify this parameter for compute environments using Amazon EKS clusters. > When you update the ``desiredvCpus`` setting, the value must be between the ``minvCpus`` and ``maxvCpus`` values. Additionally, the updated ``desiredvCpus`` value must be greater than or equal to the current ``desiredvCpus`` value. For more information, see `Troubleshooting AWS Batch <https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#error-desired-vcpus-update>`_ in the *AWS Batch User Guide* .
            :param ec2_configuration: Provides information used to select Amazon Machine Images (AMIs) for Amazon EC2 instances in the compute environment. If ``Ec2Configuration`` isn't specified, the default is ``ECS_AL2`` . When updating a compute environment, changing this setting requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . To remove the Amazon EC2 configuration and any custom AMI ID specified in ``imageIdOverride`` , set this value to an empty string. One or two values can be provided. .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.
            :param ec2_key_pair: The Amazon EC2 key pair that's used for instances launched in the compute environment. You can use this key pair to log in to your instances with SSH. To remove the Amazon EC2 key pair, set this value to an empty string. When updating a compute environment, changing the Amazon EC2 key pair requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.
            :param image_id: The Amazon Machine Image (AMI) ID used for instances launched in the compute environment. This parameter is overridden by the ``imageIdOverride`` member of the ``Ec2Configuration`` structure. To remove the custom AMI ID and use the default AMI ID, set this value to an empty string. When updating a compute environment, changing the AMI ID requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it. > The AMI that you choose for a compute environment must match the architecture of the instance types that you intend to use for that compute environment. For example, if your compute environment uses A1 instance types, the compute resource AMI that you choose must support ARM instances. Amazon ECS vends both x86 and ARM versions of the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see `Amazon ECS-optimized Amazon Linux 2 AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param instance_role: The Amazon ECS instance profile applied to Amazon EC2 instances in a compute environment. Required for Amazon EC2 instances. You can specify the short name or full Amazon Resource Name (ARN) of an instance profile. For example, ``*ecsInstanceRole*`` or ``arn:aws:iam:: *<aws_account_id>* :instance-profile/ *ecsInstanceRole*`` . For more information, see `Amazon ECS instance role <https://docs.aws.amazon.com/batch/latest/userguide/instance_IAM_role.html>`_ in the *AWS Batch User Guide* . When updating a compute environment, changing this setting requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.
            :param instance_types: The instances types that can be launched. You can specify instance families to launch any instance type within those families (for example, ``c5`` or ``p3`` ), or you can specify specific sizes within a family (such as ``c5.8xlarge`` ). You can also choose ``optimal`` to select instance types (from the C4, M4, and R4 instance families) that match the demand of your job queues. When updating a compute environment, changing this setting requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it. > When you create a compute environment, the instance types that you select for the compute environment must share the same architecture. For example, you can't mix x86 and ARM instances in the same compute environment. > Currently, ``optimal`` uses instance types from the C4, M4, and R4 instance families. In Regions that don't have instance types from those instance families, instance types from the C5, M5, and R5 instance families are used.
            :param launch_template: The launch template to use for your compute resources. Any other compute resource parameters that you specify in a `CreateComputeEnvironment <https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateComputeEnvironment.html>`_ API operation override the same parameters in the launch template. You must specify either the launch template ID or launch template name in the request, but not both. For more information, see `Launch Template Support <https://docs.aws.amazon.com/batch/latest/userguide/launch-templates.html>`_ in the ** . Removing the launch template from a compute environment will not remove the AMI specified in the launch template. In order to update the AMI specified in a launch template, the ``updateToLatestImageVersion`` parameter must be set to ``true`` . When updating a compute environment, changing the launch template requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the ** . .. epigraph:: This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
            :param minv_cpus: The minimum number of vCPUs that an environment should maintain (even if the compute environment is ``DISABLED`` ). .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.
            :param placement_group: The Amazon EC2 placement group to associate with your compute resources. If you intend to submit multi-node parallel jobs to your compute environment, you should consider creating a cluster placement group and associate it with your compute resources. This keeps your multi-node parallel job on a logical grouping of instances within a single Availability Zone with high network flow potential. For more information, see `Placement groups <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html>`_ in the *Amazon EC2 User Guide for Linux Instances* . When updating a compute environment, changing the placement group requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.
            :param security_group_ids: The Amazon EC2 security groups that are associated with instances launched in the compute environment. This parameter is required for Fargate compute resources, where it can contain up to 5 security groups. For Fargate compute resources, providing an empty list is handled as if this parameter wasn't specified and no change is made. For Amazon EC2 compute resources, providing an empty list removes the security groups from the compute resource. When updating a compute environment, changing the Amazon EC2 security groups requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .
            :param spot_iam_fleet_role: The Amazon Resource Name (ARN) of the Amazon EC2 Spot Fleet IAM role applied to a ``SPOT`` compute environment. This role is required if the allocation strategy set to ``BEST_FIT`` or if the allocation strategy isn't specified. For more information, see `Amazon EC2 spot fleet role <https://docs.aws.amazon.com/batch/latest/userguide/spot_fleet_IAM_role.html>`_ in the *AWS Batch User Guide* . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it. > To tag your Spot Instances on creation, the Spot Fleet IAM role specified here must use the newer *AmazonEC2SpotFleetTaggingRole* managed policy. The previously recommended *AmazonEC2SpotFleetRole* managed policy doesn't have the required permissions to tag Spot Instances. For more information, see `Spot instances not tagged on creation <https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#spot-instance-no-tag>`_ in the *AWS Batch User Guide* .
            :param tags: Key-value pair tags to be applied to Amazon EC2 resources that are launched in the compute environment. For AWS Batch , these take the form of ``"String1": "String2"`` , where ``String1`` is the tag key and ``String2`` is the tag value-for example, ``{ "Name": "Batch Instance - C4OnDemand" }`` . This is helpful for recognizing your Batch instances in the Amazon EC2 console. These tags aren't seen when using the AWS Batch ``ListTagsForResource`` API operation. When updating a compute environment, changing this setting requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.
            :param update_to_latest_image_version: Specifies whether the AMI ID is updated to the latest one that's supported by AWS Batch when the compute environment has an infrastructure update. The default value is ``false`` . .. epigraph:: An AMI ID can either be specified in the ``imageId`` or ``imageIdOverride`` parameters or be determined by the launch template that's specified in the ``launchTemplate`` parameter. If an AMI ID is specified any of these ways, this parameter is ignored. For more information about to update AMI IDs during an infrastructure update, see `Updating the AMI ID <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html#updating-compute-environments-ami>`_ in the *AWS Batch User Guide* . When updating a compute environment, changing this setting requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . Default: - false

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                compute_resources_property = batch.CfnComputeEnvironment.ComputeResourcesProperty(
                    maxv_cpus=123,
                    subnets=["subnets"],
                    type="type",
                
                    # the properties below are optional
                    allocation_strategy="allocationStrategy",
                    bid_percentage=123,
                    desiredv_cpus=123,
                    ec2_configuration=[batch.CfnComputeEnvironment.Ec2ConfigurationObjectProperty(
                        image_type="imageType",
                
                        # the properties below are optional
                        image_id_override="imageIdOverride",
                        image_kubernetes_version="imageKubernetesVersion"
                    )],
                    ec2_key_pair="ec2KeyPair",
                    image_id="imageId",
                    instance_role="instanceRole",
                    instance_types=["instanceTypes"],
                    launch_template=batch.CfnComputeEnvironment.LaunchTemplateSpecificationProperty(
                        launch_template_id="launchTemplateId",
                        launch_template_name="launchTemplateName",
                        overrides=[batch.CfnComputeEnvironment.LaunchTemplateSpecificationOverrideProperty(
                            launch_template_id="launchTemplateId",
                            launch_template_name="launchTemplateName",
                            target_instance_types=["targetInstanceTypes"],
                            version="version"
                        )],
                        version="version"
                    ),
                    minv_cpus=123,
                    placement_group="placementGroup",
                    security_group_ids=["securityGroupIds"],
                    spot_iam_fleet_role="spotIamFleetRole",
                    tags={
                        "tags_key": "tags"
                    },
                    update_to_latest_image_version=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0e554c6eb00e2d197fa806c35d70007a7590c1c363259e3e48971f0671e0e85f)
                check_type(argname="argument maxv_cpus", value=maxv_cpus, expected_type=type_hints["maxv_cpus"])
                check_type(argname="argument subnets", value=subnets, expected_type=type_hints["subnets"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument allocation_strategy", value=allocation_strategy, expected_type=type_hints["allocation_strategy"])
                check_type(argname="argument bid_percentage", value=bid_percentage, expected_type=type_hints["bid_percentage"])
                check_type(argname="argument desiredv_cpus", value=desiredv_cpus, expected_type=type_hints["desiredv_cpus"])
                check_type(argname="argument ec2_configuration", value=ec2_configuration, expected_type=type_hints["ec2_configuration"])
                check_type(argname="argument ec2_key_pair", value=ec2_key_pair, expected_type=type_hints["ec2_key_pair"])
                check_type(argname="argument image_id", value=image_id, expected_type=type_hints["image_id"])
                check_type(argname="argument instance_role", value=instance_role, expected_type=type_hints["instance_role"])
                check_type(argname="argument instance_types", value=instance_types, expected_type=type_hints["instance_types"])
                check_type(argname="argument launch_template", value=launch_template, expected_type=type_hints["launch_template"])
                check_type(argname="argument minv_cpus", value=minv_cpus, expected_type=type_hints["minv_cpus"])
                check_type(argname="argument placement_group", value=placement_group, expected_type=type_hints["placement_group"])
                check_type(argname="argument security_group_ids", value=security_group_ids, expected_type=type_hints["security_group_ids"])
                check_type(argname="argument spot_iam_fleet_role", value=spot_iam_fleet_role, expected_type=type_hints["spot_iam_fleet_role"])
                check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
                check_type(argname="argument update_to_latest_image_version", value=update_to_latest_image_version, expected_type=type_hints["update_to_latest_image_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "maxv_cpus": maxv_cpus,
                "subnets": subnets,
                "type": type,
            }
            if allocation_strategy is not None:
                self._values["allocation_strategy"] = allocation_strategy
            if bid_percentage is not None:
                self._values["bid_percentage"] = bid_percentage
            if desiredv_cpus is not None:
                self._values["desiredv_cpus"] = desiredv_cpus
            if ec2_configuration is not None:
                self._values["ec2_configuration"] = ec2_configuration
            if ec2_key_pair is not None:
                self._values["ec2_key_pair"] = ec2_key_pair
            if image_id is not None:
                self._values["image_id"] = image_id
            if instance_role is not None:
                self._values["instance_role"] = instance_role
            if instance_types is not None:
                self._values["instance_types"] = instance_types
            if launch_template is not None:
                self._values["launch_template"] = launch_template
            if minv_cpus is not None:
                self._values["minv_cpus"] = minv_cpus
            if placement_group is not None:
                self._values["placement_group"] = placement_group
            if security_group_ids is not None:
                self._values["security_group_ids"] = security_group_ids
            if spot_iam_fleet_role is not None:
                self._values["spot_iam_fleet_role"] = spot_iam_fleet_role
            if tags is not None:
                self._values["tags"] = tags
            if update_to_latest_image_version is not None:
                self._values["update_to_latest_image_version"] = update_to_latest_image_version

        @builtins.property
        def maxv_cpus(self) -> jsii.Number:
            '''The maximum number of Amazon EC2 vCPUs that an environment can reach.

            .. epigraph::

               With ``BEST_FIT_PROGRESSIVE`` , ``SPOT_CAPACITY_OPTIMIZED`` and ``SPOT_PRICE_CAPACITY_OPTIMIZED`` (recommended) strategies using On-Demand or Spot Instances, and the ``BEST_FIT`` strategy using Spot Instances, AWS Batch might need to exceed ``maxvCpus`` to meet your capacity requirements. In this event, AWS Batch never exceeds ``maxvCpus`` by more than a single instance.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-maxvcpus
            '''
            result = self._values.get("maxv_cpus")
            assert result is not None, "Required property 'maxv_cpus' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def subnets(self) -> typing.List[builtins.str]:
            '''The VPC subnets where the compute resources are launched.

            Fargate compute resources can contain up to 16 subnets. For Fargate compute resources, providing an empty list will be handled as if this parameter wasn't specified and no change is made. For Amazon EC2 compute resources, providing an empty list removes the VPC subnets from the compute resource. For more information, see `VPCs and subnets <https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html>`_ in the *Amazon VPC User Guide* .

            When updating a compute environment, changing the VPC subnets requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .
            .. epigraph::

               AWS Batch on Amazon EC2 and AWS Batch on Amazon EKS support Local Zones. For more information, see `Local Zones <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones.html#concepts-local-zones>`_ in the *Amazon EC2 User Guide for Linux Instances* , `Amazon EKS and AWS Local Zones <https://docs.aws.amazon.com/eks/latest/userguide/local-zones.html>`_ in the *Amazon EKS User Guide* and `Amazon ECS clusters in Local Zones, Wavelength Zones, and AWS Outposts <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-regions-zones.html#clusters-local-zones>`_ in the *Amazon ECS Developer Guide* .

               AWS Batch on Fargate doesn't currently support Local Zones.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-subnets
            '''
            result = self._values.get("subnets")
            assert result is not None, "Required property 'subnets' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of compute environment: ``EC2`` , ``SPOT`` , ``FARGATE`` , or ``FARGATE_SPOT`` .

            For more information, see `Compute environments <https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html>`_ in the *AWS Batch User Guide* .

            If you choose ``SPOT`` , you must also specify an Amazon EC2 Spot Fleet role with the ``spotIamFleetRole`` parameter. For more information, see `Amazon EC2 spot fleet role <https://docs.aws.amazon.com/batch/latest/userguide/spot_fleet_IAM_role.html>`_ in the *AWS Batch User Guide* .

            When updating compute environment, changing the type of a compute environment requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .

            When updating the type of a compute environment, changing between ``EC2`` and ``SPOT`` or between ``FARGATE`` and ``FARGATE_SPOT`` will initiate an infrastructure update, but if you switch between ``EC2`` and ``FARGATE`` , AWS CloudFormation will create a new compute environment.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def allocation_strategy(self) -> typing.Optional[builtins.str]:
            '''The allocation strategy to use for the compute resource if not enough instances of the best fitting instance type can be allocated.

            This might be because of availability of the instance type in the Region or `Amazon EC2 service limits <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html>`_ . For more information, see `Allocation strategies <https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html>`_ in the *AWS Batch User Guide* .

            When updating a compute environment, changing the allocation strategy requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . ``BEST_FIT`` is not supported when updating a compute environment.
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources, and shouldn't be specified.

            - **BEST_FIT (default)** - AWS Batch selects an instance type that best fits the needs of the jobs with a preference for the lowest-cost instance type. If additional instances of the selected instance type aren't available, AWS Batch waits for the additional instances to be available. If there aren't enough instances available, or if the user is reaching `Amazon EC2 service limits <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html>`_ then additional jobs aren't run until the currently running jobs have completed. This allocation strategy keeps costs lower but can limit scaling. If you are using Spot Fleets with ``BEST_FIT`` then the Spot Fleet IAM role must be specified.
            - **BEST_FIT_PROGRESSIVE** - AWS Batch will select additional instance types that are large enough to meet the requirements of the jobs in the queue, with a preference for instance types with a lower cost per unit vCPU. If additional instances of the previously selected instance types aren't available, AWS Batch will select new instance types.
            - **SPOT_CAPACITY_OPTIMIZED** - AWS Batch will select one or more instance types that are large enough to meet the requirements of the jobs in the queue, with a preference for instance types that are less likely to be interrupted. This allocation strategy is only available for Spot Instance compute resources.
            - **SPOT_PRICE_CAPACITY_OPTIMIZED** - The price and capacity optimized allocation strategy looks at both price and capacity to select the Spot Instance pools that are the least likely to be interrupted and have the lowest possible price. This allocation strategy is only available for Spot Instance compute resources.

            .. epigraph::

               We recommend that you use ``SPOT_PRICE_CAPACITY_OPTIMIZED`` rather than ``SPOT_CAPACITY_OPTIMIZED`` in most instances.

            With ``BEST_FIT_PROGRESSIVE`` , ``SPOT_CAPACITY_OPTIMIZED`` , and ``SPOT_PRICE_CAPACITY_OPTIMIZED`` allocation strategies using On-Demand or Spot Instances, and the ``BEST_FIT`` strategy using Spot Instances, AWS Batch might need to go above ``maxvCpus`` to meet your capacity requirements. In this event, AWS Batch never exceeds ``maxvCpus`` by more than a single instance.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-allocationstrategy
            '''
            result = self._values.get("allocation_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def bid_percentage(self) -> typing.Optional[jsii.Number]:
            '''The maximum percentage that a Spot Instance price can be when compared with the On-Demand price for that instance type before instances are launched.

            For example, if your maximum percentage is 20%, the Spot price must be less than 20% of the current On-Demand price for that Amazon EC2 instance. You always pay the lowest (market) price and never more than your maximum percentage. For most use cases, we recommend leaving this field empty.

            When updating a compute environment, changing the bid percentage requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-bidpercentage
            '''
            result = self._values.get("bid_percentage")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def desiredv_cpus(self) -> typing.Optional[jsii.Number]:
            '''The desired number of vCPUS in the compute environment.

            AWS Batch modifies this value between the minimum and maximum values based on job queue demand.
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it. > AWS Batch doesn't support changing the desired number of vCPUs of an existing compute environment. Don't specify this parameter for compute environments using Amazon EKS clusters. > When you update the ``desiredvCpus`` setting, the value must be between the ``minvCpus`` and ``maxvCpus`` values.

               Additionally, the updated ``desiredvCpus`` value must be greater than or equal to the current ``desiredvCpus`` value. For more information, see `Troubleshooting AWS Batch <https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#error-desired-vcpus-update>`_ in the *AWS Batch User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-desiredvcpus
            '''
            result = self._values.get("desiredv_cpus")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def ec2_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.Ec2ConfigurationObjectProperty"]]]]:
            '''Provides information used to select Amazon Machine Images (AMIs) for Amazon EC2 instances in the compute environment.

            If ``Ec2Configuration`` isn't specified, the default is ``ECS_AL2`` .

            When updating a compute environment, changing this setting requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . To remove the Amazon EC2 configuration and any custom AMI ID specified in ``imageIdOverride`` , set this value to an empty string.

            One or two values can be provided.
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-ec2configuration
            '''
            result = self._values.get("ec2_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.Ec2ConfigurationObjectProperty"]]]], result)

        @builtins.property
        def ec2_key_pair(self) -> typing.Optional[builtins.str]:
            '''The Amazon EC2 key pair that's used for instances launched in the compute environment.

            You can use this key pair to log in to your instances with SSH. To remove the Amazon EC2 key pair, set this value to an empty string.

            When updating a compute environment, changing the Amazon EC2 key pair requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-ec2keypair
            '''
            result = self._values.get("ec2_key_pair")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def image_id(self) -> typing.Optional[builtins.str]:
            '''The Amazon Machine Image (AMI) ID used for instances launched in the compute environment.

            This parameter is overridden by the ``imageIdOverride`` member of the ``Ec2Configuration`` structure. To remove the custom AMI ID and use the default AMI ID, set this value to an empty string.

            When updating a compute environment, changing the AMI ID requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it. > The AMI that you choose for a compute environment must match the architecture of the instance types that you intend to use for that compute environment. For example, if your compute environment uses A1 instance types, the compute resource AMI that you choose must support ARM instances. Amazon ECS vends both x86 and ARM versions of the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see `Amazon ECS-optimized Amazon Linux 2 AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-imageid
            '''
            result = self._values.get("image_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_role(self) -> typing.Optional[builtins.str]:
            '''The Amazon ECS instance profile applied to Amazon EC2 instances in a compute environment.

            Required for Amazon EC2 instances. You can specify the short name or full Amazon Resource Name (ARN) of an instance profile. For example, ``*ecsInstanceRole*`` or ``arn:aws:iam:: *<aws_account_id>* :instance-profile/ *ecsInstanceRole*`` . For more information, see `Amazon ECS instance role <https://docs.aws.amazon.com/batch/latest/userguide/instance_IAM_role.html>`_ in the *AWS Batch User Guide* .

            When updating a compute environment, changing this setting requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-instancerole
            '''
            result = self._values.get("instance_role")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The instances types that can be launched.

            You can specify instance families to launch any instance type within those families (for example, ``c5`` or ``p3`` ), or you can specify specific sizes within a family (such as ``c5.8xlarge`` ). You can also choose ``optimal`` to select instance types (from the C4, M4, and R4 instance families) that match the demand of your job queues.

            When updating a compute environment, changing this setting requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it. > When you create a compute environment, the instance types that you select for the compute environment must share the same architecture. For example, you can't mix x86 and ARM instances in the same compute environment. > Currently, ``optimal`` uses instance types from the C4, M4, and R4 instance families. In Regions that don't have instance types from those instance families, instance types from the C5, M5, and R5 instance families are used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-instancetypes
            '''
            result = self._values.get("instance_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def launch_template(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.LaunchTemplateSpecificationProperty"]]:
            '''The launch template to use for your compute resources.

            Any other compute resource parameters that you specify in a `CreateComputeEnvironment <https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateComputeEnvironment.html>`_ API operation override the same parameters in the launch template. You must specify either the launch template ID or launch template name in the request, but not both. For more information, see `Launch Template Support <https://docs.aws.amazon.com/batch/latest/userguide/launch-templates.html>`_ in the ** . Removing the launch template from a compute environment will not remove the AMI specified in the launch template. In order to update the AMI specified in a launch template, the ``updateToLatestImageVersion`` parameter must be set to ``true`` .

            When updating a compute environment, changing the launch template requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the ** .
            .. epigraph::

               This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-launchtemplate
            '''
            result = self._values.get("launch_template")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.LaunchTemplateSpecificationProperty"]], result)

        @builtins.property
        def minv_cpus(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of vCPUs that an environment should maintain (even if the compute environment is ``DISABLED`` ).

            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-minvcpus
            '''
            result = self._values.get("minv_cpus")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def placement_group(self) -> typing.Optional[builtins.str]:
            '''The Amazon EC2 placement group to associate with your compute resources.

            If you intend to submit multi-node parallel jobs to your compute environment, you should consider creating a cluster placement group and associate it with your compute resources. This keeps your multi-node parallel job on a logical grouping of instances within a single Availability Zone with high network flow potential. For more information, see `Placement groups <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html>`_ in the *Amazon EC2 User Guide for Linux Instances* .

            When updating a compute environment, changing the placement group requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-placementgroup
            '''
            result = self._values.get("placement_group")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The Amazon EC2 security groups that are associated with instances launched in the compute environment.

            This parameter is required for Fargate compute resources, where it can contain up to 5 security groups. For Fargate compute resources, providing an empty list is handled as if this parameter wasn't specified and no change is made. For Amazon EC2 compute resources, providing an empty list removes the security groups from the compute resource.

            When updating a compute environment, changing the Amazon EC2 security groups requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-securitygroupids
            '''
            result = self._values.get("security_group_ids")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def spot_iam_fleet_role(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the Amazon EC2 Spot Fleet IAM role applied to a ``SPOT`` compute environment.

            This role is required if the allocation strategy set to ``BEST_FIT`` or if the allocation strategy isn't specified. For more information, see `Amazon EC2 spot fleet role <https://docs.aws.amazon.com/batch/latest/userguide/spot_fleet_IAM_role.html>`_ in the *AWS Batch User Guide* .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it. > To tag your Spot Instances on creation, the Spot Fleet IAM role specified here must use the newer *AmazonEC2SpotFleetTaggingRole* managed policy. The previously recommended *AmazonEC2SpotFleetRole* managed policy doesn't have the required permissions to tag Spot Instances. For more information, see `Spot instances not tagged on creation <https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#spot-instance-no-tag>`_ in the *AWS Batch User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-spotiamfleetrole
            '''
            result = self._values.get("spot_iam_fleet_role")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
            '''Key-value pair tags to be applied to Amazon EC2 resources that are launched in the compute environment.

            For AWS Batch , these take the form of ``"String1": "String2"`` , where ``String1`` is the tag key and ``String2`` is the tag value-for example, ``{ "Name": "Batch Instance - C4OnDemand" }`` . This is helpful for recognizing your Batch instances in the Amazon EC2 console. These tags aren't seen when using the AWS Batch ``ListTagsForResource`` API operation.

            When updating a compute environment, changing this setting requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't specify it.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

        @builtins.property
        def update_to_latest_image_version(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether the AMI ID is updated to the latest one that's supported by AWS Batch when the compute environment has an infrastructure update.

            The default value is ``false`` .
            .. epigraph::

               An AMI ID can either be specified in the ``imageId`` or ``imageIdOverride`` parameters or be determined by the launch template that's specified in the ``launchTemplate`` parameter. If an AMI ID is specified any of these ways, this parameter is ignored. For more information about to update AMI IDs during an infrastructure update, see `Updating the AMI ID <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html#updating-compute-environments-ami>`_ in the *AWS Batch User Guide* .

            When updating a compute environment, changing this setting requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .

            :default: - false

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-updatetolatestimageversion
            '''
            result = self._values.get("update_to_latest_image_version")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ComputeResourcesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnComputeEnvironment.Ec2ConfigurationObjectProperty",
        jsii_struct_bases=[],
        name_mapping={
            "image_type": "imageType",
            "image_id_override": "imageIdOverride",
            "image_kubernetes_version": "imageKubernetesVersion",
        },
    )
    class Ec2ConfigurationObjectProperty:
        def __init__(
            self,
            *,
            image_type: builtins.str,
            image_id_override: typing.Optional[builtins.str] = None,
            image_kubernetes_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Provides information used to select Amazon Machine Images (AMIs) for instances in the compute environment.

            If ``Ec2Configuration`` isn't specified, the default is ``ECS_AL2`` ( `Amazon Linux 2 <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami>`_ ).
            .. epigraph::

               This object isn't applicable to jobs that are running on Fargate resources.

            :param image_type: The image type to match with the instance type to select an AMI. The supported values are different for ``ECS`` and ``EKS`` resources. - **ECS** - If the ``imageIdOverride`` parameter isn't specified, then a recent `Amazon ECS-optimized Amazon Linux 2 AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami>`_ ( ``ECS_AL2`` ) is used. If a new image type is specified in an update, but neither an ``imageId`` nor a ``imageIdOverride`` parameter is specified, then the latest Amazon ECS optimized AMI for that image type that's supported by AWS Batch is used. - **ECS_AL2** - `Amazon Linux 2 <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami>`_ : Default for all non-GPU instance families. - **ECS_AL2_NVIDIA** - `Amazon Linux 2 (GPU) <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#gpuami>`_ : Default for all GPU instance families (for example ``P4`` and ``G4`` ) and can be used for all non AWS Graviton-based instance types. - **ECS_AL2023** - `Amazon Linux 2023 <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ : AWS Batch supports Amazon Linux 2023. .. epigraph:: Amazon Linux 2023 does not support ``A1`` instances. - **ECS_AL1** - `Amazon Linux <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#alami>`_ . Amazon Linux has reached the end-of-life of standard support. For more information, see `Amazon Linux AMI <https://docs.aws.amazon.com/amazon-linux-ami/>`_ . - **EKS** - If the ``imageIdOverride`` parameter isn't specified, then a recent `Amazon EKS-optimized Amazon Linux AMI <https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html>`_ ( ``EKS_AL2`` ) is used. If a new image type is specified in an update, but neither an ``imageId`` nor a ``imageIdOverride`` parameter is specified, then the latest Amazon EKS optimized AMI for that image type that AWS Batch supports is used. - **EKS_AL2** - `Amazon Linux 2 <https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html>`_ : Default for all non-GPU instance families. - **EKS_AL2_NVIDIA** - `Amazon Linux 2 (accelerated) <https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html>`_ : Default for all GPU instance families (for example, ``P4`` and ``G4`` ) and can be used for all non AWS Graviton-based instance types.
            :param image_id_override: The AMI ID used for instances launched in the compute environment that match the image type. This setting overrides the ``imageId`` set in the ``computeResource`` object. .. epigraph:: The AMI that you choose for a compute environment must match the architecture of the instance types that you intend to use for that compute environment. For example, if your compute environment uses A1 instance types, the compute resource AMI that you choose must support ARM instances. Amazon ECS vends both x86 and ARM versions of the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see `Amazon ECS-optimized Amazon Linux 2 AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param image_kubernetes_version: The Kubernetes version for the compute environment. If you don't specify a value, the latest version that AWS Batch supports is used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-ec2configurationobject.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                ec2_configuration_object_property = batch.CfnComputeEnvironment.Ec2ConfigurationObjectProperty(
                    image_type="imageType",
                
                    # the properties below are optional
                    image_id_override="imageIdOverride",
                    image_kubernetes_version="imageKubernetesVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ce3b98c061a5de7793f276c675aa6c48c626a13d8debc8bf1a12f74113040fcb)
                check_type(argname="argument image_type", value=image_type, expected_type=type_hints["image_type"])
                check_type(argname="argument image_id_override", value=image_id_override, expected_type=type_hints["image_id_override"])
                check_type(argname="argument image_kubernetes_version", value=image_kubernetes_version, expected_type=type_hints["image_kubernetes_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "image_type": image_type,
            }
            if image_id_override is not None:
                self._values["image_id_override"] = image_id_override
            if image_kubernetes_version is not None:
                self._values["image_kubernetes_version"] = image_kubernetes_version

        @builtins.property
        def image_type(self) -> builtins.str:
            '''The image type to match with the instance type to select an AMI.

            The supported values are different for ``ECS`` and ``EKS`` resources.

            - **ECS** - If the ``imageIdOverride`` parameter isn't specified, then a recent `Amazon ECS-optimized Amazon Linux 2 AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami>`_ ( ``ECS_AL2`` ) is used. If a new image type is specified in an update, but neither an ``imageId`` nor a ``imageIdOverride`` parameter is specified, then the latest Amazon ECS optimized AMI for that image type that's supported by AWS Batch is used.
            - **ECS_AL2** - `Amazon Linux 2 <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami>`_ : Default for all non-GPU instance families.
            - **ECS_AL2_NVIDIA** - `Amazon Linux 2 (GPU) <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#gpuami>`_ : Default for all GPU instance families (for example ``P4`` and ``G4`` ) and can be used for all non AWS Graviton-based instance types.
            - **ECS_AL2023** - `Amazon Linux 2023 <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ : AWS Batch supports Amazon Linux 2023.

            .. epigraph::

               Amazon Linux 2023 does not support ``A1`` instances.

            - **ECS_AL1** - `Amazon Linux <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#alami>`_ . Amazon Linux has reached the end-of-life of standard support. For more information, see `Amazon Linux AMI <https://docs.aws.amazon.com/amazon-linux-ami/>`_ .
            - **EKS** - If the ``imageIdOverride`` parameter isn't specified, then a recent `Amazon EKS-optimized Amazon Linux AMI <https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html>`_ ( ``EKS_AL2`` ) is used. If a new image type is specified in an update, but neither an ``imageId`` nor a ``imageIdOverride`` parameter is specified, then the latest Amazon EKS optimized AMI for that image type that AWS Batch supports is used.
            - **EKS_AL2** - `Amazon Linux 2 <https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html>`_ : Default for all non-GPU instance families.
            - **EKS_AL2_NVIDIA** - `Amazon Linux 2 (accelerated) <https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html>`_ : Default for all GPU instance families (for example, ``P4`` and ``G4`` ) and can be used for all non AWS Graviton-based instance types.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-ec2configurationobject.html#cfn-batch-computeenvironment-ec2configurationobject-imagetype
            '''
            result = self._values.get("image_type")
            assert result is not None, "Required property 'image_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def image_id_override(self) -> typing.Optional[builtins.str]:
            '''The AMI ID used for instances launched in the compute environment that match the image type.

            This setting overrides the ``imageId`` set in the ``computeResource`` object.
            .. epigraph::

               The AMI that you choose for a compute environment must match the architecture of the instance types that you intend to use for that compute environment. For example, if your compute environment uses A1 instance types, the compute resource AMI that you choose must support ARM instances. Amazon ECS vends both x86 and ARM versions of the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see `Amazon ECS-optimized Amazon Linux 2 AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-ec2configurationobject.html#cfn-batch-computeenvironment-ec2configurationobject-imageidoverride
            '''
            result = self._values.get("image_id_override")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def image_kubernetes_version(self) -> typing.Optional[builtins.str]:
            '''The Kubernetes version for the compute environment.

            If you don't specify a value, the latest version that AWS Batch supports is used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-ec2configurationobject.html#cfn-batch-computeenvironment-ec2configurationobject-imagekubernetesversion
            '''
            result = self._values.get("image_kubernetes_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "Ec2ConfigurationObjectProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnComputeEnvironment.EksConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "eks_cluster_arn": "eksClusterArn",
            "kubernetes_namespace": "kubernetesNamespace",
        },
    )
    class EksConfigurationProperty:
        def __init__(
            self,
            *,
            eks_cluster_arn: builtins.str,
            kubernetes_namespace: builtins.str,
        ) -> None:
            '''Configuration for the Amazon EKS cluster that supports the AWS Batch compute environment.

            The cluster must exist before the compute environment can be created.

            :param eks_cluster_arn: The Amazon Resource Name (ARN) of the Amazon EKS cluster. An example is ``arn: *aws* :eks: *us-east-1* : *123456789012* :cluster/ *ClusterForBatch*`` .
            :param kubernetes_namespace: The namespace of the Amazon EKS cluster. AWS Batch manages pods in this namespace. The value can't left empty or null. It must be fewer than 64 characters long, can't be set to ``default`` , can't start with " ``kube-`` ," and must match this regular expression: ``^[a-z0-9]([-a-z0-9]*[a-z0-9])?$`` . For more information, see `Namespaces <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/>`_ in the Kubernetes documentation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-eksconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                eks_configuration_property = batch.CfnComputeEnvironment.EksConfigurationProperty(
                    eks_cluster_arn="eksClusterArn",
                    kubernetes_namespace="kubernetesNamespace"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5d01293eced89171b858aec2adc4c3362e7bc583fff04c8572350b492d38c641)
                check_type(argname="argument eks_cluster_arn", value=eks_cluster_arn, expected_type=type_hints["eks_cluster_arn"])
                check_type(argname="argument kubernetes_namespace", value=kubernetes_namespace, expected_type=type_hints["kubernetes_namespace"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "eks_cluster_arn": eks_cluster_arn,
                "kubernetes_namespace": kubernetes_namespace,
            }

        @builtins.property
        def eks_cluster_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the Amazon EKS cluster.

            An example is ``arn: *aws* :eks: *us-east-1* : *123456789012* :cluster/ *ClusterForBatch*`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-eksconfiguration.html#cfn-batch-computeenvironment-eksconfiguration-eksclusterarn
            '''
            result = self._values.get("eks_cluster_arn")
            assert result is not None, "Required property 'eks_cluster_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def kubernetes_namespace(self) -> builtins.str:
            '''The namespace of the Amazon EKS cluster.

            AWS Batch manages pods in this namespace. The value can't left empty or null. It must be fewer than 64 characters long, can't be set to ``default`` , can't start with " ``kube-`` ," and must match this regular expression: ``^[a-z0-9]([-a-z0-9]*[a-z0-9])?$`` . For more information, see `Namespaces <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/>`_ in the Kubernetes documentation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-eksconfiguration.html#cfn-batch-computeenvironment-eksconfiguration-kubernetesnamespace
            '''
            result = self._values.get("kubernetes_namespace")
            assert result is not None, "Required property 'kubernetes_namespace' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EksConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnComputeEnvironment.LaunchTemplateSpecificationOverrideProperty",
        jsii_struct_bases=[],
        name_mapping={
            "launch_template_id": "launchTemplateId",
            "launch_template_name": "launchTemplateName",
            "target_instance_types": "targetInstanceTypes",
            "version": "version",
        },
    )
    class LaunchTemplateSpecificationOverrideProperty:
        def __init__(
            self,
            *,
            launch_template_id: typing.Optional[builtins.str] = None,
            launch_template_name: typing.Optional[builtins.str] = None,
            target_instance_types: typing.Optional[typing.Sequence[builtins.str]] = None,
            version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents a launch template to use in place of the default launch template.

            You must specify either the launch template ID or launch template name in the request, but not both.

            If security groups are specified using both the ``securityGroupIds`` parameter of ``CreateComputeEnvironment`` and the launch template, the values in the ``securityGroupIds`` parameter of ``CreateComputeEnvironment`` will be used.

            You can define up to ten (10) overrides for each compute environment.
            .. epigraph::

               This object isn't applicable to jobs that are running on Fargate resources. > To unset all override templates for a compute environment, you can pass an empty array to the `UpdateComputeEnvironment.overrides <https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html>`_ parameter, or not include the ``overrides`` parameter when submitting the ``UpdateComputeEnvironment`` API operation.

            :param launch_template_id: The ID of the launch template. *Note:* If you specify the ``launchTemplateId`` you can't specify the ``launchTemplateName`` as well.
            :param launch_template_name: The name of the launch template. *Note:* If you specify the ``launchTemplateName`` you can't specify the ``launchTemplateId`` as well.
            :param target_instance_types: The instance type or family that this override launch template should be applied to. This parameter is required when defining a launch template override. Information included in this parameter must meet the following requirements: - Must be a valid Amazon EC2 instance type or family. - ``optimal`` isn't allowed. - ``targetInstanceTypes`` can target only instance types and families that are included within the ```ComputeResource.instanceTypes`` <https://docs.aws.amazon.com/batch/latest/APIReference/API_ComputeResource.html#Batch-Type-ComputeResource-instanceTypes>`_ set. ``targetInstanceTypes`` doesn't need to include all of the instances from the ``instanceType`` set, but at least a subset. For example, if ``ComputeResource.instanceTypes`` includes ``[m5, g5]`` , ``targetInstanceTypes`` can include ``[m5.2xlarge]`` and ``[m5.large]`` but not ``[c5.large]`` . - ``targetInstanceTypes`` included within the same launch template override or across launch template overrides can't overlap for the same compute environment. For example, you can't define one launch template override to target an instance family and another define an instance type within this same family.
            :param version: The version number of the launch template, ``$Default`` , or ``$Latest`` . If the value is ``$Default`` , the default version of the launch template is used. If the value is ``$Latest`` , the latest version of the launch template is used. .. epigraph:: If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed when the compute environment is updated. It's only changed if the ``updateToLatestImageVersion`` parameter for the compute environment is set to ``true`` . During an infrastructure update, if either ``$Default`` or ``$Latest`` is specified, AWS Batch re-evaluates the launch template version, and it might use a different version of the launch template. This is the case even if the launch template isn't specified in the update. When updating a compute environment, changing the launch template requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . Default: ``$Default`` Latest: ``$Latest``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecificationoverride.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                launch_template_specification_override_property = batch.CfnComputeEnvironment.LaunchTemplateSpecificationOverrideProperty(
                    launch_template_id="launchTemplateId",
                    launch_template_name="launchTemplateName",
                    target_instance_types=["targetInstanceTypes"],
                    version="version"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4b5bc20e5d24e0d359d6430d6ef9cdce99590872d896d3e0d65372a69394ad77)
                check_type(argname="argument launch_template_id", value=launch_template_id, expected_type=type_hints["launch_template_id"])
                check_type(argname="argument launch_template_name", value=launch_template_name, expected_type=type_hints["launch_template_name"])
                check_type(argname="argument target_instance_types", value=target_instance_types, expected_type=type_hints["target_instance_types"])
                check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if launch_template_id is not None:
                self._values["launch_template_id"] = launch_template_id
            if launch_template_name is not None:
                self._values["launch_template_name"] = launch_template_name
            if target_instance_types is not None:
                self._values["target_instance_types"] = target_instance_types
            if version is not None:
                self._values["version"] = version

        @builtins.property
        def launch_template_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the launch template.

            *Note:* If you specify the ``launchTemplateId`` you can't specify the ``launchTemplateName`` as well.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecificationoverride.html#cfn-batch-computeenvironment-launchtemplatespecificationoverride-launchtemplateid
            '''
            result = self._values.get("launch_template_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def launch_template_name(self) -> typing.Optional[builtins.str]:
            '''The name of the launch template.

            *Note:* If you specify the ``launchTemplateName`` you can't specify the ``launchTemplateId`` as well.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecificationoverride.html#cfn-batch-computeenvironment-launchtemplatespecificationoverride-launchtemplatename
            '''
            result = self._values.get("launch_template_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def target_instance_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The instance type or family that this override launch template should be applied to.

            This parameter is required when defining a launch template override.

            Information included in this parameter must meet the following requirements:

            - Must be a valid Amazon EC2 instance type or family.
            - ``optimal`` isn't allowed.
            - ``targetInstanceTypes`` can target only instance types and families that are included within the ```ComputeResource.instanceTypes`` <https://docs.aws.amazon.com/batch/latest/APIReference/API_ComputeResource.html#Batch-Type-ComputeResource-instanceTypes>`_ set. ``targetInstanceTypes`` doesn't need to include all of the instances from the ``instanceType`` set, but at least a subset. For example, if ``ComputeResource.instanceTypes`` includes ``[m5, g5]`` , ``targetInstanceTypes`` can include ``[m5.2xlarge]`` and ``[m5.large]`` but not ``[c5.large]`` .
            - ``targetInstanceTypes`` included within the same launch template override or across launch template overrides can't overlap for the same compute environment. For example, you can't define one launch template override to target an instance family and another define an instance type within this same family.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecificationoverride.html#cfn-batch-computeenvironment-launchtemplatespecificationoverride-targetinstancetypes
            '''
            result = self._values.get("target_instance_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def version(self) -> typing.Optional[builtins.str]:
            '''The version number of the launch template, ``$Default`` , or ``$Latest`` .

            If the value is ``$Default`` , the default version of the launch template is used. If the value is ``$Latest`` , the latest version of the launch template is used.
            .. epigraph::

               If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed when the compute environment is updated. It's only changed if the ``updateToLatestImageVersion`` parameter for the compute environment is set to ``true`` . During an infrastructure update, if either ``$Default`` or ``$Latest`` is specified, AWS Batch re-evaluates the launch template version, and it might use a different version of the launch template. This is the case even if the launch template isn't specified in the update. When updating a compute environment, changing the launch template requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .

            Default: ``$Default``

            Latest: ``$Latest``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecificationoverride.html#cfn-batch-computeenvironment-launchtemplatespecificationoverride-version
            '''
            result = self._values.get("version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateSpecificationOverrideProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnComputeEnvironment.LaunchTemplateSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "launch_template_id": "launchTemplateId",
            "launch_template_name": "launchTemplateName",
            "overrides": "overrides",
            "version": "version",
        },
    )
    class LaunchTemplateSpecificationProperty:
        def __init__(
            self,
            *,
            launch_template_id: typing.Optional[builtins.str] = None,
            launch_template_name: typing.Optional[builtins.str] = None,
            overrides: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnComputeEnvironment.LaunchTemplateSpecificationOverrideProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents a launch template that's associated with a compute resource.

            You must specify either the launch template ID or launch template name in the request, but not both.

            If security groups are specified using both the ``securityGroupIds`` parameter of ``CreateComputeEnvironment`` and the launch template, the values in the ``securityGroupIds`` parameter of ``CreateComputeEnvironment`` will be used.
            .. epigraph::

               This object isn't applicable to jobs that are running on Fargate resources.

            :param launch_template_id: The ID of the launch template.
            :param launch_template_name: The name of the launch template.
            :param overrides: A launch template to use in place of the default launch template. You must specify either the launch template ID or launch template name in the request, but not both. You can specify up to ten (10) launch template overrides that are associated to unique instance types or families for each compute environment. .. epigraph:: To unset all override templates for a compute environment, you can pass an empty array to the `UpdateComputeEnvironment.overrides <https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html>`_ parameter, or not include the ``overrides`` parameter when submitting the ``UpdateComputeEnvironment`` API operation.
            :param version: The version number of the launch template, ``$Default`` , or ``$Latest`` . If the value is ``$Default`` , the default version of the launch template is used. If the value is ``$Latest`` , the latest version of the launch template is used. .. epigraph:: If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed when the compute environment is updated. It's only changed if the ``updateToLatestImageVersion`` parameter for the compute environment is set to ``true`` . During an infrastructure update, if either ``$Default`` or ``$Latest`` is specified, AWS Batch re-evaluates the launch template version, and it might use a different version of the launch template. This is the case even if the launch template isn't specified in the update. When updating a compute environment, changing the launch template requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* . Default: ``$Default`` Latest: ``$Latest``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecification.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                launch_template_specification_property = batch.CfnComputeEnvironment.LaunchTemplateSpecificationProperty(
                    launch_template_id="launchTemplateId",
                    launch_template_name="launchTemplateName",
                    overrides=[batch.CfnComputeEnvironment.LaunchTemplateSpecificationOverrideProperty(
                        launch_template_id="launchTemplateId",
                        launch_template_name="launchTemplateName",
                        target_instance_types=["targetInstanceTypes"],
                        version="version"
                    )],
                    version="version"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e2a48b24f0297afff0df36a352af2f9856ad8fac713d1ecbc2161d0b968651da)
                check_type(argname="argument launch_template_id", value=launch_template_id, expected_type=type_hints["launch_template_id"])
                check_type(argname="argument launch_template_name", value=launch_template_name, expected_type=type_hints["launch_template_name"])
                check_type(argname="argument overrides", value=overrides, expected_type=type_hints["overrides"])
                check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if launch_template_id is not None:
                self._values["launch_template_id"] = launch_template_id
            if launch_template_name is not None:
                self._values["launch_template_name"] = launch_template_name
            if overrides is not None:
                self._values["overrides"] = overrides
            if version is not None:
                self._values["version"] = version

        @builtins.property
        def launch_template_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the launch template.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecification.html#cfn-batch-computeenvironment-launchtemplatespecification-launchtemplateid
            '''
            result = self._values.get("launch_template_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def launch_template_name(self) -> typing.Optional[builtins.str]:
            '''The name of the launch template.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecification.html#cfn-batch-computeenvironment-launchtemplatespecification-launchtemplatename
            '''
            result = self._values.get("launch_template_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def overrides(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.LaunchTemplateSpecificationOverrideProperty"]]]]:
            '''A launch template to use in place of the default launch template.

            You must specify either the launch template ID or launch template name in the request, but not both.

            You can specify up to ten (10) launch template overrides that are associated to unique instance types or families for each compute environment.
            .. epigraph::

               To unset all override templates for a compute environment, you can pass an empty array to the `UpdateComputeEnvironment.overrides <https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html>`_ parameter, or not include the ``overrides`` parameter when submitting the ``UpdateComputeEnvironment`` API operation.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecification.html#cfn-batch-computeenvironment-launchtemplatespecification-overrides
            '''
            result = self._values.get("overrides")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnComputeEnvironment.LaunchTemplateSpecificationOverrideProperty"]]]], result)

        @builtins.property
        def version(self) -> typing.Optional[builtins.str]:
            '''The version number of the launch template, ``$Default`` , or ``$Latest`` .

            If the value is ``$Default`` , the default version of the launch template is used. If the value is ``$Latest`` , the latest version of the launch template is used.
            .. epigraph::

               If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed when the compute environment is updated. It's only changed if the ``updateToLatestImageVersion`` parameter for the compute environment is set to ``true`` . During an infrastructure update, if either ``$Default`` or ``$Latest`` is specified, AWS Batch re-evaluates the launch template version, and it might use a different version of the launch template. This is the case even if the launch template isn't specified in the update. When updating a compute environment, changing the launch template requires an infrastructure update of the compute environment. For more information, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .

            Default: ``$Default``

            Latest: ``$Latest``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-launchtemplatespecification.html#cfn-batch-computeenvironment-launchtemplatespecification-version
            '''
            result = self._values.get("version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnComputeEnvironment.UpdatePolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "job_execution_timeout_minutes": "jobExecutionTimeoutMinutes",
            "terminate_jobs_on_update": "terminateJobsOnUpdate",
        },
    )
    class UpdatePolicyProperty:
        def __init__(
            self,
            *,
            job_execution_timeout_minutes: typing.Optional[jsii.Number] = None,
            terminate_jobs_on_update: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''Specifies the infrastructure update policy for the Amazon EC2 compute environment.

            For more information about infrastructure updates, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .

            :param job_execution_timeout_minutes: Specifies the job timeout (in minutes) when the compute environment infrastructure is updated. The default value is 30. Default: - 30
            :param terminate_jobs_on_update: Specifies whether jobs are automatically terminated when the computer environment infrastructure is updated. The default value is ``false`` . Default: - false

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-updatepolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                update_policy_property = batch.CfnComputeEnvironment.UpdatePolicyProperty(
                    job_execution_timeout_minutes=123,
                    terminate_jobs_on_update=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2f01f930e5e22d80ed490169089dfdbf2f416fdeb6b3ae57212518ae0b70b5e3)
                check_type(argname="argument job_execution_timeout_minutes", value=job_execution_timeout_minutes, expected_type=type_hints["job_execution_timeout_minutes"])
                check_type(argname="argument terminate_jobs_on_update", value=terminate_jobs_on_update, expected_type=type_hints["terminate_jobs_on_update"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if job_execution_timeout_minutes is not None:
                self._values["job_execution_timeout_minutes"] = job_execution_timeout_minutes
            if terminate_jobs_on_update is not None:
                self._values["terminate_jobs_on_update"] = terminate_jobs_on_update

        @builtins.property
        def job_execution_timeout_minutes(self) -> typing.Optional[jsii.Number]:
            '''Specifies the job timeout (in minutes) when the compute environment infrastructure is updated.

            The default value is 30.

            :default: - 30

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-updatepolicy.html#cfn-batch-computeenvironment-updatepolicy-jobexecutiontimeoutminutes
            '''
            result = self._values.get("job_execution_timeout_minutes")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def terminate_jobs_on_update(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether jobs are automatically terminated when the computer environment infrastructure is updated.

            The default value is ``false`` .

            :default: - false

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-updatepolicy.html#cfn-batch-computeenvironment-updatepolicy-terminatejobsonupdate
            '''
            result = self._values.get("terminate_jobs_on_update")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "UpdatePolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.CfnComputeEnvironmentProps",
    jsii_struct_bases=[],
    name_mapping={
        "type": "type",
        "compute_environment_name": "computeEnvironmentName",
        "compute_resources": "computeResources",
        "context": "context",
        "eks_configuration": "eksConfiguration",
        "replace_compute_environment": "replaceComputeEnvironment",
        "service_role": "serviceRole",
        "state": "state",
        "tags": "tags",
        "unmanagedv_cpus": "unmanagedvCpus",
        "update_policy": "updatePolicy",
    },
)
class CfnComputeEnvironmentProps:
    def __init__(
        self,
        *,
        type: builtins.str,
        compute_environment_name: typing.Optional[builtins.str] = None,
        compute_resources: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnComputeEnvironment.ComputeResourcesProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        context: typing.Optional[builtins.str] = None,
        eks_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnComputeEnvironment.EksConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        replace_compute_environment: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        service_role: typing.Optional[builtins.str] = None,
        state: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        unmanagedv_cpus: typing.Optional[jsii.Number] = None,
        update_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnComputeEnvironment.UpdatePolicyProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnComputeEnvironment``.

        :param type: The type of the compute environment: ``MANAGED`` or ``UNMANAGED`` . For more information, see `Compute Environments <https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html>`_ in the *AWS Batch User Guide* .
        :param compute_environment_name: The name for your compute environment. It can be up to 128 characters long. It can contain uppercase and lowercase letters, numbers, hyphens (-), and underscores (_).
        :param compute_resources: The ComputeResources property type specifies details of the compute resources managed by the compute environment. This parameter is required for managed compute environments. For more information, see `Compute Environments <https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html>`_ in the ** .
        :param context: Reserved.
        :param eks_configuration: The details for the Amazon EKS cluster that supports the compute environment.
        :param replace_compute_environment: Specifies whether the compute environment is replaced if an update is made that requires replacing the instances in the compute environment. The default value is ``true`` . To enable more properties to be updated, set this property to ``false`` . When changing the value of this property to ``false`` , do not change any other properties at the same time. If other properties are changed at the same time, and the change needs to be rolled back but it can't, it's possible for the stack to go into the ``UPDATE_ROLLBACK_FAILED`` state. You can't update a stack that is in the ``UPDATE_ROLLBACK_FAILED`` state. However, if you can continue to roll it back, you can return the stack to its original settings and then try to update it again. For more information, see `Continue rolling back an update <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-continueupdaterollback.html>`_ in the *AWS CloudFormation User Guide* . ``ReplaceComputeEnvironment`` is not applicable for Fargate compute environments. Fargate compute environments are always updated without interruption. The properties that can't be changed without replacing the compute environment are in the ```ComputeResources`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html>`_ property type: ```AllocationStrategy`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-allocationstrategy>`_ , ```BidPercentage`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-bidpercentage>`_ , ```Ec2Configuration`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-ec2configuration>`_ , ```Ec2KeyPair`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-ec2keypair>`_ , ```Ec2KeyPair`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-ec2keypair>`_ , ```ImageId`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-imageid>`_ , ```InstanceRole`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-instancerole>`_ , ```InstanceTypes`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-instancetypes>`_ , ```LaunchTemplate`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-launchtemplate>`_ , ```MaxvCpus`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-maxvcpus>`_ , ```MinvCpus`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-minvcpus>`_ , ```PlacementGroup`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-placementgroup>`_ , ```SecurityGroupIds`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-securitygroupids>`_ , ```Subnets`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-subnets>`_ , `Tags <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-tags>`_ , ```Type`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-type>`_ , and ```UpdateToLatestImageVersion`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-updatetolatestimageversion>`_ . Default: - true
        :param service_role: The full Amazon Resource Name (ARN) of the IAM role that allows AWS Batch to make calls to other AWS services on your behalf. For more information, see `AWS Batch service IAM role <https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html>`_ in the *AWS Batch User Guide* . .. epigraph:: If your account already created the AWS Batch service-linked role, that role is used by default for your compute environment unless you specify a different role here. If the AWS Batch service-linked role doesn't exist in your account, and no role is specified here, the service attempts to create the AWS Batch service-linked role in your account. If your specified role has a path other than ``/`` , then you must specify either the full role ARN (recommended) or prefix the role name with the path. For example, if a role with the name ``bar`` has a path of ``/foo/`` , specify ``/foo/bar`` as the role name. For more information, see `Friendly names and paths <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names>`_ in the *IAM User Guide* . .. epigraph:: Depending on how you created your AWS Batch service role, its ARN might contain the ``service-role`` path prefix. When you only specify the name of the service role, AWS Batch assumes that your ARN doesn't use the ``service-role`` path prefix. Because of this, we recommend that you specify the full ARN of your service role when you create compute environments.
        :param state: The state of the compute environment. If the state is ``ENABLED`` , then the compute environment accepts jobs from a queue and can scale out automatically based on queues. If the state is ``ENABLED`` , then the AWS Batch scheduler can attempt to place jobs from an associated job queue on the compute resources within the environment. If the compute environment is managed, then it can scale its instances out or in automatically, based on the job queue demand. If the state is ``DISABLED`` , then the AWS Batch scheduler doesn't attempt to place jobs within the environment. Jobs in a ``STARTING`` or ``RUNNING`` state continue to progress normally. Managed compute environments in the ``DISABLED`` state don't scale out. .. epigraph:: Compute environments in a ``DISABLED`` state may continue to incur billing charges. To prevent additional charges, turn off and then delete the compute environment. For more information, see `State <https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state>`_ in the *AWS Batch User Guide* . When an instance is idle, the instance scales down to the ``minvCpus`` value. However, the instance size doesn't change. For example, consider a ``c5.8xlarge`` instance with a ``minvCpus`` value of ``4`` and a ``desiredvCpus`` value of ``36`` . This instance doesn't scale down to a ``c5.large`` instance.
        :param tags: The tags applied to the compute environment.
        :param unmanagedv_cpus: The maximum number of vCPUs for an unmanaged compute environment. This parameter is only used for fair-share scheduling to reserve vCPU capacity for new share identifiers. If this parameter isn't provided for a fair-share job queue, no vCPU capacity is reserved. .. epigraph:: This parameter is only supported when the ``type`` parameter is set to ``UNMANAGED`` .
        :param update_policy: Specifies the infrastructure update policy for the compute environment. For more information about infrastructure updates, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_batch as batch
            
            cfn_compute_environment_props = batch.CfnComputeEnvironmentProps(
                type="type",
            
                # the properties below are optional
                compute_environment_name="computeEnvironmentName",
                compute_resources=batch.CfnComputeEnvironment.ComputeResourcesProperty(
                    maxv_cpus=123,
                    subnets=["subnets"],
                    type="type",
            
                    # the properties below are optional
                    allocation_strategy="allocationStrategy",
                    bid_percentage=123,
                    desiredv_cpus=123,
                    ec2_configuration=[batch.CfnComputeEnvironment.Ec2ConfigurationObjectProperty(
                        image_type="imageType",
            
                        # the properties below are optional
                        image_id_override="imageIdOverride",
                        image_kubernetes_version="imageKubernetesVersion"
                    )],
                    ec2_key_pair="ec2KeyPair",
                    image_id="imageId",
                    instance_role="instanceRole",
                    instance_types=["instanceTypes"],
                    launch_template=batch.CfnComputeEnvironment.LaunchTemplateSpecificationProperty(
                        launch_template_id="launchTemplateId",
                        launch_template_name="launchTemplateName",
                        overrides=[batch.CfnComputeEnvironment.LaunchTemplateSpecificationOverrideProperty(
                            launch_template_id="launchTemplateId",
                            launch_template_name="launchTemplateName",
                            target_instance_types=["targetInstanceTypes"],
                            version="version"
                        )],
                        version="version"
                    ),
                    minv_cpus=123,
                    placement_group="placementGroup",
                    security_group_ids=["securityGroupIds"],
                    spot_iam_fleet_role="spotIamFleetRole",
                    tags={
                        "tags_key": "tags"
                    },
                    update_to_latest_image_version=False
                ),
                context="context",
                eks_configuration=batch.CfnComputeEnvironment.EksConfigurationProperty(
                    eks_cluster_arn="eksClusterArn",
                    kubernetes_namespace="kubernetesNamespace"
                ),
                replace_compute_environment=False,
                service_role="serviceRole",
                state="state",
                tags={
                    "tags_key": "tags"
                },
                unmanagedv_cpus=123,
                update_policy=batch.CfnComputeEnvironment.UpdatePolicyProperty(
                    job_execution_timeout_minutes=123,
                    terminate_jobs_on_update=False
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a81ed1de03c840384ae49a0a4dbeb244507d2327c304fc093af8d720954eb257)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument compute_environment_name", value=compute_environment_name, expected_type=type_hints["compute_environment_name"])
            check_type(argname="argument compute_resources", value=compute_resources, expected_type=type_hints["compute_resources"])
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
            check_type(argname="argument eks_configuration", value=eks_configuration, expected_type=type_hints["eks_configuration"])
            check_type(argname="argument replace_compute_environment", value=replace_compute_environment, expected_type=type_hints["replace_compute_environment"])
            check_type(argname="argument service_role", value=service_role, expected_type=type_hints["service_role"])
            check_type(argname="argument state", value=state, expected_type=type_hints["state"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument unmanagedv_cpus", value=unmanagedv_cpus, expected_type=type_hints["unmanagedv_cpus"])
            check_type(argname="argument update_policy", value=update_policy, expected_type=type_hints["update_policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }
        if compute_environment_name is not None:
            self._values["compute_environment_name"] = compute_environment_name
        if compute_resources is not None:
            self._values["compute_resources"] = compute_resources
        if context is not None:
            self._values["context"] = context
        if eks_configuration is not None:
            self._values["eks_configuration"] = eks_configuration
        if replace_compute_environment is not None:
            self._values["replace_compute_environment"] = replace_compute_environment
        if service_role is not None:
            self._values["service_role"] = service_role
        if state is not None:
            self._values["state"] = state
        if tags is not None:
            self._values["tags"] = tags
        if unmanagedv_cpus is not None:
            self._values["unmanagedv_cpus"] = unmanagedv_cpus
        if update_policy is not None:
            self._values["update_policy"] = update_policy

    @builtins.property
    def type(self) -> builtins.str:
        '''The type of the compute environment: ``MANAGED`` or ``UNMANAGED`` .

        For more information, see `Compute Environments <https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html>`_ in the *AWS Batch User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-type
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def compute_environment_name(self) -> typing.Optional[builtins.str]:
        '''The name for your compute environment.

        It can be up to 128 characters long. It can contain uppercase and lowercase letters, numbers, hyphens (-), and underscores (_).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-computeenvironmentname
        '''
        result = self._values.get("compute_environment_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def compute_resources(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnComputeEnvironment.ComputeResourcesProperty]]:
        '''The ComputeResources property type specifies details of the compute resources managed by the compute environment.

        This parameter is required for managed compute environments. For more information, see `Compute Environments <https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html>`_ in the ** .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-computeresources
        '''
        result = self._values.get("compute_resources")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnComputeEnvironment.ComputeResourcesProperty]], result)

    @builtins.property
    def context(self) -> typing.Optional[builtins.str]:
        '''Reserved.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-context
        '''
        result = self._values.get("context")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def eks_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnComputeEnvironment.EksConfigurationProperty]]:
        '''The details for the Amazon EKS cluster that supports the compute environment.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-eksconfiguration
        '''
        result = self._values.get("eks_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnComputeEnvironment.EksConfigurationProperty]], result)

    @builtins.property
    def replace_compute_environment(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the compute environment is replaced if an update is made that requires replacing the instances in the compute environment.

        The default value is ``true`` . To enable more properties to be updated, set this property to ``false`` . When changing the value of this property to ``false`` , do not change any other properties at the same time. If other properties are changed at the same time, and the change needs to be rolled back but it can't, it's possible for the stack to go into the ``UPDATE_ROLLBACK_FAILED`` state. You can't update a stack that is in the ``UPDATE_ROLLBACK_FAILED`` state. However, if you can continue to roll it back, you can return the stack to its original settings and then try to update it again. For more information, see `Continue rolling back an update <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-continueupdaterollback.html>`_ in the *AWS CloudFormation User Guide* .

        ``ReplaceComputeEnvironment`` is not applicable for Fargate compute environments. Fargate compute environments are always updated without interruption.

        The properties that can't be changed without replacing the compute environment are in the ```ComputeResources`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html>`_ property type: ```AllocationStrategy`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-allocationstrategy>`_ , ```BidPercentage`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-bidpercentage>`_ , ```Ec2Configuration`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-ec2configuration>`_ , ```Ec2KeyPair`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-ec2keypair>`_ , ```Ec2KeyPair`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-ec2keypair>`_ , ```ImageId`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-imageid>`_ , ```InstanceRole`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-instancerole>`_ , ```InstanceTypes`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-instancetypes>`_ , ```LaunchTemplate`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-launchtemplate>`_ , ```MaxvCpus`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-maxvcpus>`_ , ```MinvCpus`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-minvcpus>`_ , ```PlacementGroup`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-placementgroup>`_ , ```SecurityGroupIds`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-securitygroupids>`_ , ```Subnets`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-subnets>`_ , `Tags <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-tags>`_ , ```Type`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-type>`_ , and ```UpdateToLatestImageVersion`` <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-computeenvironment-computeresources.html#cfn-batch-computeenvironment-computeresources-updatetolatestimageversion>`_ .

        :default: - true

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-replacecomputeenvironment
        '''
        result = self._values.get("replace_compute_environment")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def service_role(self) -> typing.Optional[builtins.str]:
        '''The full Amazon Resource Name (ARN) of the IAM role that allows AWS Batch to make calls to other AWS services on your behalf.

        For more information, see `AWS Batch service IAM role <https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html>`_ in the *AWS Batch User Guide* .
        .. epigraph::

           If your account already created the AWS Batch service-linked role, that role is used by default for your compute environment unless you specify a different role here. If the AWS Batch service-linked role doesn't exist in your account, and no role is specified here, the service attempts to create the AWS Batch service-linked role in your account.

        If your specified role has a path other than ``/`` , then you must specify either the full role ARN (recommended) or prefix the role name with the path. For example, if a role with the name ``bar`` has a path of ``/foo/`` , specify ``/foo/bar`` as the role name. For more information, see `Friendly names and paths <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names>`_ in the *IAM User Guide* .
        .. epigraph::

           Depending on how you created your AWS Batch service role, its ARN might contain the ``service-role`` path prefix. When you only specify the name of the service role, AWS Batch assumes that your ARN doesn't use the ``service-role`` path prefix. Because of this, we recommend that you specify the full ARN of your service role when you create compute environments.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-servicerole
        '''
        result = self._values.get("service_role")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def state(self) -> typing.Optional[builtins.str]:
        '''The state of the compute environment.

        If the state is ``ENABLED`` , then the compute environment accepts jobs from a queue and can scale out automatically based on queues.

        If the state is ``ENABLED`` , then the AWS Batch scheduler can attempt to place jobs from an associated job queue on the compute resources within the environment. If the compute environment is managed, then it can scale its instances out or in automatically, based on the job queue demand.

        If the state is ``DISABLED`` , then the AWS Batch scheduler doesn't attempt to place jobs within the environment. Jobs in a ``STARTING`` or ``RUNNING`` state continue to progress normally. Managed compute environments in the ``DISABLED`` state don't scale out.
        .. epigraph::

           Compute environments in a ``DISABLED`` state may continue to incur billing charges. To prevent additional charges, turn off and then delete the compute environment. For more information, see `State <https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state>`_ in the *AWS Batch User Guide* .

        When an instance is idle, the instance scales down to the ``minvCpus`` value. However, the instance size doesn't change. For example, consider a ``c5.8xlarge`` instance with a ``minvCpus`` value of ``4`` and a ``desiredvCpus`` value of ``36`` . This instance doesn't scale down to a ``c5.large`` instance.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-state
        '''
        result = self._values.get("state")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags applied to the compute environment.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def unmanagedv_cpus(self) -> typing.Optional[jsii.Number]:
        '''The maximum number of vCPUs for an unmanaged compute environment.

        This parameter is only used for fair-share scheduling to reserve vCPU capacity for new share identifiers. If this parameter isn't provided for a fair-share job queue, no vCPU capacity is reserved.
        .. epigraph::

           This parameter is only supported when the ``type`` parameter is set to ``UNMANAGED`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-unmanagedvcpus
        '''
        result = self._values.get("unmanagedv_cpus")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def update_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnComputeEnvironment.UpdatePolicyProperty]]:
        '''Specifies the infrastructure update policy for the compute environment.

        For more information about infrastructure updates, see `Updating compute environments <https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html>`_ in the *AWS Batch User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-computeenvironment.html#cfn-batch-computeenvironment-updatepolicy
        '''
        result = self._values.get("update_policy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnComputeEnvironment.UpdatePolicyProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnComputeEnvironmentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnConsumableResource(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_batch.CfnConsumableResource",
):
    '''The ``AWS::Batch::ConsumableResource`` resource specifies the parameters for an AWS Batch consumable resource.

    For more information, see `Resource-aware scheduling <https://docs.aws.amazon.com/batch/latest/userguide/resource-aware-scheduling.html>`_ in the ** .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-consumableresource.html
    :cloudformationResource: AWS::Batch::ConsumableResource
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_batch as batch
        
        cfn_consumable_resource = batch.CfnConsumableResource(self, "MyCfnConsumableResource",
            resource_type="resourceType",
            total_quantity=123,
        
            # the properties below are optional
            consumable_resource_name="consumableResourceName",
            tags={
                "tags_key": "tags"
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        resource_type: builtins.str,
        total_quantity: jsii.Number,
        consumable_resource_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param resource_type: Indicates whether the resource is available to be re-used after a job completes. Can be one of:. - ``REPLENISHABLE`` - ``NON_REPLENISHABLE``
        :param total_quantity: The total amount of the consumable resource that is available.
        :param consumable_resource_name: The name of the consumable resource.
        :param tags: The tags that you apply to the consumable resource to help you categorize and organize your resources. Each tag consists of a key and an optional value. For more information, see `Tagging your AWS Batch resources <https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html>`_ .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5c99b0cde97c1da7fcdbc520729f9e431c6aa0879ca2c3f310f91017cfee5007)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnConsumableResourceProps(
            resource_type=resource_type,
            total_quantity=total_quantity,
            consumable_resource_name=consumable_resource_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b41c1f0a4fca10f0b198693331d558f57596f6e4a572a4cf64008e1ca42e3534)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5511de8b41a1f295801ff1c0ea7a9ee2822d462f774ec168cdb7e0a49196a1ed)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrAvailableQuantity")
    def attr_available_quantity(self) -> jsii.Number:
        '''The amount of the consumable resource that is currently available to use.

        :cloudformationAttribute: AvailableQuantity
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrAvailableQuantity"))

    @builtins.property
    @jsii.member(jsii_name="attrConsumableResourceArn")
    def attr_consumable_resource_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the consumable resource.

        :cloudformationAttribute: ConsumableResourceArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrConsumableResourceArn"))

    @builtins.property
    @jsii.member(jsii_name="attrCreatedAt")
    def attr_created_at(self) -> jsii.Number:
        '''The Unix timestamp (in milliseconds) for when the consumable resource was created.

        :cloudformationAttribute: CreatedAt
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrCreatedAt"))

    @builtins.property
    @jsii.member(jsii_name="attrInUseQuantity")
    def attr_in_use_quantity(self) -> jsii.Number:
        '''The amount of the consumable resource that is currently in use.

        :cloudformationAttribute: InUseQuantity
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrInUseQuantity"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="resourceType")
    def resource_type(self) -> builtins.str:
        '''Indicates whether the resource is available to be re-used after a job completes.

        Can be one of:.
        '''
        return typing.cast(builtins.str, jsii.get(self, "resourceType"))

    @resource_type.setter
    def resource_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9e365d37f3f984f47f3e9155d90709d48dab3b5c99de23c31edf3e0524a3302d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "resourceType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="totalQuantity")
    def total_quantity(self) -> jsii.Number:
        '''The total amount of the consumable resource that is available.'''
        return typing.cast(jsii.Number, jsii.get(self, "totalQuantity"))

    @total_quantity.setter
    def total_quantity(self, value: jsii.Number) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__267ecfd211d87fefa6a82e181ae34839f62811b27349a340c7dd7af7b1d643ac)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "totalQuantity", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="consumableResourceName")
    def consumable_resource_name(self) -> typing.Optional[builtins.str]:
        '''The name of the consumable resource.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "consumableResourceName"))

    @consumable_resource_name.setter
    def consumable_resource_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6b8701734e05768bb24c412344b9c0ac7a78b14234dfd357ae9125b48b5ce5e2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "consumableResourceName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags that you apply to the consumable resource to help you categorize and organize your resources.'''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "tags"))

    @tags.setter
    def tags(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6aeb610201199cf9ed022e6ce90ed818369482ef25877943acf468ba6509693e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.CfnConsumableResourceProps",
    jsii_struct_bases=[],
    name_mapping={
        "resource_type": "resourceType",
        "total_quantity": "totalQuantity",
        "consumable_resource_name": "consumableResourceName",
        "tags": "tags",
    },
)
class CfnConsumableResourceProps:
    def __init__(
        self,
        *,
        resource_type: builtins.str,
        total_quantity: jsii.Number,
        consumable_resource_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnConsumableResource``.

        :param resource_type: Indicates whether the resource is available to be re-used after a job completes. Can be one of:. - ``REPLENISHABLE`` - ``NON_REPLENISHABLE``
        :param total_quantity: The total amount of the consumable resource that is available.
        :param consumable_resource_name: The name of the consumable resource.
        :param tags: The tags that you apply to the consumable resource to help you categorize and organize your resources. Each tag consists of a key and an optional value. For more information, see `Tagging your AWS Batch resources <https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-consumableresource.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_batch as batch
            
            cfn_consumable_resource_props = batch.CfnConsumableResourceProps(
                resource_type="resourceType",
                total_quantity=123,
            
                # the properties below are optional
                consumable_resource_name="consumableResourceName",
                tags={
                    "tags_key": "tags"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__273574f20d47e157a9e993b6d91e9496f22d80d24c48e2d755bfe2493e0fe9cc)
            check_type(argname="argument resource_type", value=resource_type, expected_type=type_hints["resource_type"])
            check_type(argname="argument total_quantity", value=total_quantity, expected_type=type_hints["total_quantity"])
            check_type(argname="argument consumable_resource_name", value=consumable_resource_name, expected_type=type_hints["consumable_resource_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "resource_type": resource_type,
            "total_quantity": total_quantity,
        }
        if consumable_resource_name is not None:
            self._values["consumable_resource_name"] = consumable_resource_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def resource_type(self) -> builtins.str:
        '''Indicates whether the resource is available to be re-used after a job completes. Can be one of:.

        - ``REPLENISHABLE``
        - ``NON_REPLENISHABLE``

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-consumableresource.html#cfn-batch-consumableresource-resourcetype
        '''
        result = self._values.get("resource_type")
        assert result is not None, "Required property 'resource_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def total_quantity(self) -> jsii.Number:
        '''The total amount of the consumable resource that is available.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-consumableresource.html#cfn-batch-consumableresource-totalquantity
        '''
        result = self._values.get("total_quantity")
        assert result is not None, "Required property 'total_quantity' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def consumable_resource_name(self) -> typing.Optional[builtins.str]:
        '''The name of the consumable resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-consumableresource.html#cfn-batch-consumableresource-consumableresourcename
        '''
        result = self._values.get("consumable_resource_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags that you apply to the consumable resource to help you categorize and organize your resources.

        Each tag consists of a key and an optional value. For more information, see `Tagging your AWS Batch resources <https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-consumableresource.html#cfn-batch-consumableresource-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnConsumableResourceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnJobDefinition(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition",
):
    '''The ``AWS::Batch::JobDefinition`` resource specifies the parameters for an AWS Batch job definition.

    For more information, see `Job Definitions <https://docs.aws.amazon.com/batch/latest/userguide/job_definitions.html>`_ in the ** .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html
    :cloudformationResource: AWS::Batch::JobDefinition
    :exampleMetadata: fixture=_generated

    Example::

        
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        type: builtins.str,
        consumable_resource_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.ConsumableResourcePropertiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        container_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.ContainerPropertiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ecs_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EcsPropertiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        eks_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EksPropertiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        job_definition_name: typing.Optional[builtins.str] = None,
        node_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.NodePropertiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        parameters: typing.Any = None,
        platform_capabilities: typing.Optional[typing.Sequence[builtins.str]] = None,
        propagate_tags: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        retry_strategy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.RetryStrategyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        scheduling_priority: typing.Optional[jsii.Number] = None,
        tags: typing.Any = None,
        timeout: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.TimeoutProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param type: The type of job definition. For more information about multi-node parallel jobs, see `Creating a multi-node parallel job definition <https://docs.aws.amazon.com/batch/latest/userguide/multi-node-job-def.html>`_ in the *AWS Batch User Guide* . - If the value is ``container`` , then one of the following is required: ``containerProperties`` , ``ecsProperties`` , or ``eksProperties`` . - If the value is ``multinode`` , then ``nodeProperties`` is required. .. epigraph:: If the job is run on Fargate resources, then ``multinode`` isn't supported.
        :param consumable_resource_properties: Contains a list of consumable resources required by the job.
        :param container_properties: An object with properties specific to Amazon ECS-based jobs. When ``containerProperties`` is used in the job definition, it can't be used in addition to ``eksProperties`` , ``ecsProperties`` , or ``nodeProperties`` .
        :param ecs_properties: An object that contains the properties for the Amazon ECS resources of a job.When ``ecsProperties`` is used in the job definition, it can't be used in addition to ``containerProperties`` , ``eksProperties`` , or ``nodeProperties`` .
        :param eks_properties: An object with properties that are specific to Amazon EKS-based jobs. When ``eksProperties`` is used in the job definition, it can't be used in addition to ``containerProperties`` , ``ecsProperties`` , or ``nodeProperties`` .
        :param job_definition_name: The name of the job definition.
        :param node_properties: An object with properties that are specific to multi-node parallel jobs. When ``nodeProperties`` is used in the job definition, it can't be used in addition to ``containerProperties`` , ``ecsProperties`` , or ``eksProperties`` . .. epigraph:: If the job runs on Fargate resources, don't specify ``nodeProperties`` . Use ``containerProperties`` instead.
        :param parameters: Default parameters or parameter substitution placeholders that are set in the job definition. Parameters are specified as a key-value pair mapping. Parameters in a ``SubmitJob`` request override any corresponding parameter defaults from the job definition. For more information about specifying parameters, see `Job definition parameters <https://docs.aws.amazon.com/batch/latest/userguide/job_definition_parameters.html>`_ in the *AWS Batch User Guide* .
        :param platform_capabilities: The platform capabilities required by the job definition. If no value is specified, it defaults to ``EC2`` . Jobs run on Fargate resources specify ``FARGATE`` .
        :param propagate_tags: Specifies whether to propagate the tags from the job or job definition to the corresponding Amazon ECS task. If no value is specified, the tags aren't propagated. Tags can only be propagated to the tasks when the tasks are created. For tags with the same name, job tags are given priority over job definitions tags. If the total number of combined tags from the job and job definition is over 50, the job is moved to the ``FAILED`` state.
        :param retry_strategy: The retry strategy to use for failed jobs that are submitted with this job definition.
        :param scheduling_priority: The scheduling priority of the job definition. This only affects jobs in job queues with a fair-share policy. Jobs with a higher scheduling priority are scheduled before jobs with a lower scheduling priority.
        :param tags: The tags that are applied to the job definition.
        :param timeout: The timeout time for jobs that are submitted with this job definition. After the amount of time you specify passes, AWS Batch terminates your jobs if they aren't finished.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__37959c68856ab28ea1a57515db976bc2215806d52c75f6166834df6ae651417f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnJobDefinitionProps(
            type=type,
            consumable_resource_properties=consumable_resource_properties,
            container_properties=container_properties,
            ecs_properties=ecs_properties,
            eks_properties=eks_properties,
            job_definition_name=job_definition_name,
            node_properties=node_properties,
            parameters=parameters,
            platform_capabilities=platform_capabilities,
            propagate_tags=propagate_tags,
            retry_strategy=retry_strategy,
            scheduling_priority=scheduling_priority,
            tags=tags,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__84b6af858f7642dbc10029a34c2ed90c61a840b3c9d8a639fbe0690b4aa593d9)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__710ae17567654ca123f279efb5ee033134bde1c49217d5abca4db2e98435f076)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrJobDefinitionArn")
    def attr_job_definition_arn(self) -> builtins.str:
        '''The job definition ARN, such as ``batch: *us-east-1* : *111122223333* :job-definition/ *test-gpu* : *2*`` .

        :cloudformationAttribute: JobDefinitionArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrJobDefinitionArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''The type of job definition.'''
        return typing.cast(builtins.str, jsii.get(self, "type"))

    @type.setter
    def type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c96b92c09c064b55548b6669735edc8ffa8187ffae08746200ea2113f36cdcc6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "type", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="consumableResourceProperties")
    def consumable_resource_properties(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ConsumableResourcePropertiesProperty"]]:
        '''Contains a list of consumable resources required by the job.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ConsumableResourcePropertiesProperty"]], jsii.get(self, "consumableResourceProperties"))

    @consumable_resource_properties.setter
    def consumable_resource_properties(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ConsumableResourcePropertiesProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c368ee191818b7f44968cb3ae44c5bb147fa195ae7df416283fff61f6efc1dd9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "consumableResourceProperties", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="containerProperties")
    def container_properties(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ContainerPropertiesProperty"]]:
        '''An object with properties specific to Amazon ECS-based jobs.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ContainerPropertiesProperty"]], jsii.get(self, "containerProperties"))

    @container_properties.setter
    def container_properties(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ContainerPropertiesProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f69a394e899a3c712c6499f6fcffa98702eb3ab7de5e4489e6e69fb85ae55491)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "containerProperties", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="ecsProperties")
    def ecs_properties(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EcsPropertiesProperty"]]:
        '''An object that contains the properties for the Amazon ECS resources of a job.When ``ecsProperties`` is used in the job definition, it can't be used in addition to ``containerProperties`` , ``eksProperties`` , or ``nodeProperties`` .'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EcsPropertiesProperty"]], jsii.get(self, "ecsProperties"))

    @ecs_properties.setter
    def ecs_properties(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EcsPropertiesProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0f93a9bec3d893845bf7cf59752d55dcec2df30819130514b48897371c2ec224)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ecsProperties", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="eksProperties")
    def eks_properties(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksPropertiesProperty"]]:
        '''An object with properties that are specific to Amazon EKS-based jobs.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksPropertiesProperty"]], jsii.get(self, "eksProperties"))

    @eks_properties.setter
    def eks_properties(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksPropertiesProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ca04be0ba694eb2f262cad48402ba7bf94953530d02c650e2679b1f91498c346)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "eksProperties", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="jobDefinitionName")
    def job_definition_name(self) -> typing.Optional[builtins.str]:
        '''The name of the job definition.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "jobDefinitionName"))

    @job_definition_name.setter
    def job_definition_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__38f628b65bca30d37b13950af18cab324471321abe3a2e97c363552c0adbe924)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "jobDefinitionName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="nodeProperties")
    def node_properties(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.NodePropertiesProperty"]]:
        '''An object with properties that are specific to multi-node parallel jobs.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.NodePropertiesProperty"]], jsii.get(self, "nodeProperties"))

    @node_properties.setter
    def node_properties(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.NodePropertiesProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e9d18dd3dda5e5761ec3a8f3c7af96598c179ac9f1bca0af0078b6aee12c880b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "nodeProperties", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Any:
        '''Default parameters or parameter substitution placeholders that are set in the job definition.'''
        return typing.cast(typing.Any, jsii.get(self, "parameters"))

    @parameters.setter
    def parameters(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0dfa67937c5c5c585d3dedec8653cb0633c802b99afaa23545e8fcc71bf4ef88)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "parameters", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="platformCapabilities")
    def platform_capabilities(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The platform capabilities required by the job definition.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "platformCapabilities"))

    @platform_capabilities.setter
    def platform_capabilities(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__df387fd03e7b9707eac2330bd1c2776f56a71ead17f60e9cc8606c21591b2cca)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "platformCapabilities", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="propagateTags")
    def propagate_tags(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to propagate the tags from the job or job definition to the corresponding Amazon ECS task.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "propagateTags"))

    @propagate_tags.setter
    def propagate_tags(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__269e7fe24ddc93928c4ed8726528871fc789de7275942860c547acd761f99025)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "propagateTags", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="retryStrategy")
    def retry_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.RetryStrategyProperty"]]:
        '''The retry strategy to use for failed jobs that are submitted with this job definition.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.RetryStrategyProperty"]], jsii.get(self, "retryStrategy"))

    @retry_strategy.setter
    def retry_strategy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.RetryStrategyProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2ac615dba37cd2b5a6e9d6f8cdd4b6beef51ff826d65fe0b5cdc31685de03def)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "retryStrategy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="schedulingPriority")
    def scheduling_priority(self) -> typing.Optional[jsii.Number]:
        '''The scheduling priority of the job definition.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "schedulingPriority"))

    @scheduling_priority.setter
    def scheduling_priority(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dba2bfccdff2acf2f90d4e7a4965d8dcb1ae572e2c9cbd92260bf928e42d7e01)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "schedulingPriority", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Any:
        '''The tags that are applied to the job definition.'''
        return typing.cast(typing.Any, jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e523eed1bbd28b55b217e2df4a08d397d7fea1eab9961fca2a86f1f43fb0631f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="timeout")
    def timeout(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.TimeoutProperty"]]:
        '''The timeout time for jobs that are submitted with this job definition.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.TimeoutProperty"]], jsii.get(self, "timeout"))

    @timeout.setter
    def timeout(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.TimeoutProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fa7b04ecd9d9f8e9c609d704cda9a2176a378049913a70b741c06702f6586c79)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "timeout", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.AuthorizationConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"access_point_id": "accessPointId", "iam": "iam"},
    )
    class AuthorizationConfigProperty:
        def __init__(
            self,
            *,
            access_point_id: typing.Optional[builtins.str] = None,
            iam: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param access_point_id: 
            :param iam: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-authorizationconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                authorization_config_property = batch.CfnJobDefinition.AuthorizationConfigProperty(
                    access_point_id="accessPointId",
                    iam="iam"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a944ff69de809bc564eb18b81b5fe5aaf841b525a61c70b907da2a96bd8d9b10)
                check_type(argname="argument access_point_id", value=access_point_id, expected_type=type_hints["access_point_id"])
                check_type(argname="argument iam", value=iam, expected_type=type_hints["iam"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if access_point_id is not None:
                self._values["access_point_id"] = access_point_id
            if iam is not None:
                self._values["iam"] = iam

        @builtins.property
        def access_point_id(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-authorizationconfig.html#cfn-batch-jobdefinition-authorizationconfig-accesspointid
            '''
            result = self._values.get("access_point_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def iam(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-authorizationconfig.html#cfn-batch-jobdefinition-authorizationconfig-iam
            '''
            result = self._values.get("iam")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AuthorizationConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.ConsumableResourcePropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={"consumable_resource_list": "consumableResourceList"},
    )
    class ConsumableResourcePropertiesProperty:
        def __init__(
            self,
            *,
            consumable_resource_list: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.ConsumableResourceRequirementProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''Contains a list of consumable resources required by a job.

            :param consumable_resource_list: The list of consumable resources required by a job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-consumableresourceproperties.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                consumable_resource_properties_property = batch.CfnJobDefinition.ConsumableResourcePropertiesProperty(
                    consumable_resource_list=[batch.CfnJobDefinition.ConsumableResourceRequirementProperty(
                        consumable_resource="consumableResource",
                        quantity=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3b629a35b0e74fbc3eff29818b9c44bbd57ebdeb0ab63ab735a548e6880a79ea)
                check_type(argname="argument consumable_resource_list", value=consumable_resource_list, expected_type=type_hints["consumable_resource_list"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "consumable_resource_list": consumable_resource_list,
            }

        @builtins.property
        def consumable_resource_list(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ConsumableResourceRequirementProperty"]]]:
            '''The list of consumable resources required by a job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-consumableresourceproperties.html#cfn-batch-jobdefinition-consumableresourceproperties-consumableresourcelist
            '''
            result = self._values.get("consumable_resource_list")
            assert result is not None, "Required property 'consumable_resource_list' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ConsumableResourceRequirementProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConsumableResourcePropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.ConsumableResourceRequirementProperty",
        jsii_struct_bases=[],
        name_mapping={
            "consumable_resource": "consumableResource",
            "quantity": "quantity",
        },
    )
    class ConsumableResourceRequirementProperty:
        def __init__(
            self,
            *,
            consumable_resource: builtins.str,
            quantity: jsii.Number,
        ) -> None:
            '''Information about a consumable resource required to run a job.

            :param consumable_resource: The name or ARN of the consumable resource.
            :param quantity: The quantity of the consumable resource that is needed.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-consumableresourcerequirement.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                consumable_resource_requirement_property = batch.CfnJobDefinition.ConsumableResourceRequirementProperty(
                    consumable_resource="consumableResource",
                    quantity=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__14cb9e21c298e41a083a856c9790ba0510d4434a296b7c70c86b029921622e58)
                check_type(argname="argument consumable_resource", value=consumable_resource, expected_type=type_hints["consumable_resource"])
                check_type(argname="argument quantity", value=quantity, expected_type=type_hints["quantity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "consumable_resource": consumable_resource,
                "quantity": quantity,
            }

        @builtins.property
        def consumable_resource(self) -> builtins.str:
            '''The name or ARN of the consumable resource.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-consumableresourcerequirement.html#cfn-batch-jobdefinition-consumableresourcerequirement-consumableresource
            '''
            result = self._values.get("consumable_resource")
            assert result is not None, "Required property 'consumable_resource' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def quantity(self) -> jsii.Number:
            '''The quantity of the consumable resource that is needed.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-consumableresourcerequirement.html#cfn-batch-jobdefinition-consumableresourcerequirement-quantity
            '''
            result = self._values.get("quantity")
            assert result is not None, "Required property 'quantity' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConsumableResourceRequirementProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.ContainerPropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "image": "image",
            "command": "command",
            "environment": "environment",
            "ephemeral_storage": "ephemeralStorage",
            "execution_role_arn": "executionRoleArn",
            "fargate_platform_configuration": "fargatePlatformConfiguration",
            "instance_type": "instanceType",
            "job_role_arn": "jobRoleArn",
            "linux_parameters": "linuxParameters",
            "log_configuration": "logConfiguration",
            "memory": "memory",
            "mount_points": "mountPoints",
            "network_configuration": "networkConfiguration",
            "privileged": "privileged",
            "readonly_root_filesystem": "readonlyRootFilesystem",
            "repository_credentials": "repositoryCredentials",
            "resource_requirements": "resourceRequirements",
            "runtime_platform": "runtimePlatform",
            "secrets": "secrets",
            "ulimits": "ulimits",
            "user": "user",
            "vcpus": "vcpus",
            "volumes": "volumes",
        },
    )
    class ContainerPropertiesProperty:
        def __init__(
            self,
            *,
            image: builtins.str,
            command: typing.Optional[typing.Sequence[builtins.str]] = None,
            environment: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EnvironmentProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            ephemeral_storage: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EphemeralStorageProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            execution_role_arn: typing.Optional[builtins.str] = None,
            fargate_platform_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.FargatePlatformConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            instance_type: typing.Optional[builtins.str] = None,
            job_role_arn: typing.Optional[builtins.str] = None,
            linux_parameters: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.LinuxParametersProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            log_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.LogConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            memory: typing.Optional[jsii.Number] = None,
            mount_points: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.MountPointsProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            network_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.NetworkConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            privileged: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            readonly_root_filesystem: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            repository_credentials: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.RepositoryCredentialsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            resource_requirements: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.ResourceRequirementProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            runtime_platform: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.RuntimePlatformProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            secrets: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.SecretProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            ulimits: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.UlimitProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            user: typing.Optional[builtins.str] = None,
            vcpus: typing.Optional[jsii.Number] = None,
            volumes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.VolumesProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''Container properties are used for Amazon ECS based job definitions.

            These properties to describe the container that's launched as part of a job.

            :param image: Required. The image used to start a container. This string is passed directly to the Docker daemon. Images in the Docker Hub registry are available by default. Other repositories are specified with ``*repository-url* / *image* : *tag*`` . It can be 255 characters long. It can contain uppercase and lowercase letters, numbers, hyphens (-), underscores (_), colons (:), periods (.), forward slashes (/), and number signs (#). This parameter maps to ``Image`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``IMAGE`` parameter of `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . .. epigraph:: Docker image architecture must match the processor architecture of the compute resources that they're scheduled on. For example, ARM-based Docker images can only run on ARM-based compute resources. - Images in Amazon ECR Public repositories use the full ``registry/repository[:tag]`` or ``registry/repository[@digest]`` naming conventions. For example, ``public.ecr.aws/ *registry_alias* / *my-web-app* : *latest*`` . - Images in Amazon ECR repositories use the full registry and repository URI (for example, ``123456789012.dkr.ecr.<region-name>.amazonaws.com/<repository-name>`` ). - Images in official repositories on Docker Hub use a single name (for example, ``ubuntu`` or ``mongo`` ). - Images in other repositories on Docker Hub are qualified with an organization name (for example, ``amazon/amazon-ecs-agent`` ). - Images in other online repositories are qualified further by a domain name (for example, ``quay.io/assemblyline/ubuntu`` ).
            :param command: The command that's passed to the container. This parameter maps to ``Cmd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``COMMAND`` parameter to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . For more information, see `https://docs.docker.com/engine/reference/builder/#cmd <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#cmd>`_ .
            :param environment: The environment variables to pass to a container. This parameter maps to ``Env`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--env`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . .. epigraph:: We don't recommend using plaintext environment variables for sensitive information, such as credential data. > Environment variables cannot start with " ``AWS_BATCH`` ". This naming convention is reserved for variables that AWS Batch sets.
            :param ephemeral_storage: The amount of ephemeral storage to allocate for the task. This parameter is used to expand the total amount of ephemeral storage available, beyond the default amount, for tasks hosted on AWS Fargate .
            :param execution_role_arn: The Amazon Resource Name (ARN) of the execution role that AWS Batch can assume. For jobs that run on Fargate resources, you must provide an execution role. For more information, see `AWS Batch execution IAM role <https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html>`_ in the *AWS Batch User Guide* .
            :param fargate_platform_configuration: The platform configuration for jobs that are running on Fargate resources. Jobs that are running on Amazon EC2 resources must not specify this parameter.
            :param instance_type: 
            :param job_role_arn: The Amazon Resource Name (ARN) of the IAM role that the container can assume for AWS permissions. For more information, see `IAM roles for tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param linux_parameters: Linux-specific modifications that are applied to the container, such as details for device mappings.
            :param log_configuration: The log configuration specification for the container. This parameter maps to ``LogConfig`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--log-driver`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . By default, containers use the same logging driver that the Docker daemon uses. However the container might use a different logging driver than the Docker daemon by specifying a log driver with this parameter in the container definition. To use a different logging driver for a container, the log system must be configured properly on the container instance (or on a different log server for remote logging options). For more information on the options for different supported log drivers, see `Configure logging drivers <https://docs.aws.amazon.com/https://docs.docker.com/engine/admin/logging/overview/>`_ in the Docker documentation. .. epigraph:: AWS Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in the `LogConfiguration <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties-logconfiguration.html>`_ data type). This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version | grep "Server API version"`` .. epigraph:: The Amazon ECS container agent running on a container instance must register the logging drivers available on that instance with the ``ECS_AVAILABLE_LOGGING_DRIVERS`` environment variable before containers placed on that instance can use these log configuration options. For more information, see `Amazon ECS container agent configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param memory: This parameter is deprecated, use ``resourceRequirements`` to specify the memory requirements for the job definition. It's not supported for jobs running on Fargate resources. For jobs that run on Amazon EC2 resources, it specifies the memory hard limit (in MiB) for a container. If your container attempts to exceed the specified number, it's terminated. You must specify at least 4 MiB of memory for a job using this parameter. The memory hard limit can be specified in several places. It must be specified for each node at least once.
            :param mount_points: The mount points for data volumes in your container. This parameter maps to ``Volumes`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--volume`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .
            :param network_configuration: The network configuration for jobs that are running on Fargate resources. Jobs that are running on Amazon EC2 resources must not specify this parameter.
            :param privileged: When this parameter is true, the container is given elevated permissions on the host container instance (similar to the ``root`` user). This parameter maps to ``Privileged`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--privileged`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . The default value is false. .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources and shouldn't be provided, or specified as false.
            :param readonly_root_filesystem: When this parameter is true, the container is given read-only access to its root file system. This parameter maps to ``ReadonlyRootfs`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--read-only`` option to ``docker run`` .
            :param repository_credentials: The private repository authentication credentials to use.
            :param resource_requirements: The type and amount of resources to assign to a container. The supported resources include ``GPU`` , ``MEMORY`` , and ``VCPU`` .
            :param runtime_platform: An object that represents the compute environment architecture for AWS Batch jobs on Fargate.
            :param secrets: The secrets for the container. For more information, see `Specifying sensitive data <https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html>`_ in the *AWS Batch User Guide* .
            :param ulimits: A list of ``ulimits`` to set in the container. This parameter maps to ``Ulimits`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--ulimit`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources and shouldn't be provided.
            :param user: The user name to use inside the container. This parameter maps to ``User`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--user`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .
            :param vcpus: This parameter is deprecated, use ``resourceRequirements`` to specify the vCPU requirements for the job definition. It's not supported for jobs running on Fargate resources. For jobs running on Amazon EC2 resources, it specifies the number of vCPUs reserved for the job. Each vCPU is equivalent to 1,024 CPU shares. This parameter maps to ``CpuShares`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--cpu-shares`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . The number of vCPUs must be specified but can be specified in several places. You must specify it at least once for each node.
            :param volumes: A list of data volumes used in a job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # options: Any
                
                container_properties_property = batch.CfnJobDefinition.ContainerPropertiesProperty(
                    image="image",
                
                    # the properties below are optional
                    command=["command"],
                    environment=[batch.CfnJobDefinition.EnvironmentProperty(
                        name="name",
                        value="value"
                    )],
                    ephemeral_storage=batch.CfnJobDefinition.EphemeralStorageProperty(
                        size_in_gi_b=123
                    ),
                    execution_role_arn="executionRoleArn",
                    fargate_platform_configuration=batch.CfnJobDefinition.FargatePlatformConfigurationProperty(
                        platform_version="platformVersion"
                    ),
                    instance_type="instanceType",
                    job_role_arn="jobRoleArn",
                    linux_parameters=batch.CfnJobDefinition.LinuxParametersProperty(
                        devices=[batch.CfnJobDefinition.DeviceProperty(
                            container_path="containerPath",
                            host_path="hostPath",
                            permissions=["permissions"]
                        )],
                        init_process_enabled=False,
                        max_swap=123,
                        shared_memory_size=123,
                        swappiness=123,
                        tmpfs=[batch.CfnJobDefinition.TmpfsProperty(
                            container_path="containerPath",
                            size=123,
                
                            # the properties below are optional
                            mount_options=["mountOptions"]
                        )]
                    ),
                    log_configuration=batch.CfnJobDefinition.LogConfigurationProperty(
                        log_driver="logDriver",
                
                        # the properties below are optional
                        options=options,
                        secret_options=[batch.CfnJobDefinition.SecretProperty(
                            name="name",
                            value_from="valueFrom"
                        )]
                    ),
                    memory=123,
                    mount_points=[batch.CfnJobDefinition.MountPointsProperty(
                        container_path="containerPath",
                        read_only=False,
                        source_volume="sourceVolume"
                    )],
                    network_configuration=batch.CfnJobDefinition.NetworkConfigurationProperty(
                        assign_public_ip="assignPublicIp"
                    ),
                    privileged=False,
                    readonly_root_filesystem=False,
                    repository_credentials=batch.CfnJobDefinition.RepositoryCredentialsProperty(
                        credentials_parameter="credentialsParameter"
                    ),
                    resource_requirements=[batch.CfnJobDefinition.ResourceRequirementProperty(
                        type="type",
                        value="value"
                    )],
                    runtime_platform=batch.CfnJobDefinition.RuntimePlatformProperty(
                        cpu_architecture="cpuArchitecture",
                        operating_system_family="operatingSystemFamily"
                    ),
                    secrets=[batch.CfnJobDefinition.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )],
                    ulimits=[batch.CfnJobDefinition.UlimitProperty(
                        hard_limit=123,
                        name="name",
                        soft_limit=123
                    )],
                    user="user",
                    vcpus=123,
                    volumes=[batch.CfnJobDefinition.VolumesProperty(
                        efs_volume_configuration=batch.CfnJobDefinition.EfsVolumeConfigurationProperty(
                            file_system_id="fileSystemId",
                
                            # the properties below are optional
                            authorization_config=batch.CfnJobDefinition.AuthorizationConfigProperty(
                                access_point_id="accessPointId",
                                iam="iam"
                            ),
                            root_directory="rootDirectory",
                            transit_encryption="transitEncryption",
                            transit_encryption_port=123
                        ),
                        host=batch.CfnJobDefinition.VolumesHostProperty(
                            source_path="sourcePath"
                        ),
                        name="name"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__09b3c38642739790560033cae597f009c106d353ddc5faf85f6a7bf4891ac4d1)
                check_type(argname="argument image", value=image, expected_type=type_hints["image"])
                check_type(argname="argument command", value=command, expected_type=type_hints["command"])
                check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
                check_type(argname="argument ephemeral_storage", value=ephemeral_storage, expected_type=type_hints["ephemeral_storage"])
                check_type(argname="argument execution_role_arn", value=execution_role_arn, expected_type=type_hints["execution_role_arn"])
                check_type(argname="argument fargate_platform_configuration", value=fargate_platform_configuration, expected_type=type_hints["fargate_platform_configuration"])
                check_type(argname="argument instance_type", value=instance_type, expected_type=type_hints["instance_type"])
                check_type(argname="argument job_role_arn", value=job_role_arn, expected_type=type_hints["job_role_arn"])
                check_type(argname="argument linux_parameters", value=linux_parameters, expected_type=type_hints["linux_parameters"])
                check_type(argname="argument log_configuration", value=log_configuration, expected_type=type_hints["log_configuration"])
                check_type(argname="argument memory", value=memory, expected_type=type_hints["memory"])
                check_type(argname="argument mount_points", value=mount_points, expected_type=type_hints["mount_points"])
                check_type(argname="argument network_configuration", value=network_configuration, expected_type=type_hints["network_configuration"])
                check_type(argname="argument privileged", value=privileged, expected_type=type_hints["privileged"])
                check_type(argname="argument readonly_root_filesystem", value=readonly_root_filesystem, expected_type=type_hints["readonly_root_filesystem"])
                check_type(argname="argument repository_credentials", value=repository_credentials, expected_type=type_hints["repository_credentials"])
                check_type(argname="argument resource_requirements", value=resource_requirements, expected_type=type_hints["resource_requirements"])
                check_type(argname="argument runtime_platform", value=runtime_platform, expected_type=type_hints["runtime_platform"])
                check_type(argname="argument secrets", value=secrets, expected_type=type_hints["secrets"])
                check_type(argname="argument ulimits", value=ulimits, expected_type=type_hints["ulimits"])
                check_type(argname="argument user", value=user, expected_type=type_hints["user"])
                check_type(argname="argument vcpus", value=vcpus, expected_type=type_hints["vcpus"])
                check_type(argname="argument volumes", value=volumes, expected_type=type_hints["volumes"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "image": image,
            }
            if command is not None:
                self._values["command"] = command
            if environment is not None:
                self._values["environment"] = environment
            if ephemeral_storage is not None:
                self._values["ephemeral_storage"] = ephemeral_storage
            if execution_role_arn is not None:
                self._values["execution_role_arn"] = execution_role_arn
            if fargate_platform_configuration is not None:
                self._values["fargate_platform_configuration"] = fargate_platform_configuration
            if instance_type is not None:
                self._values["instance_type"] = instance_type
            if job_role_arn is not None:
                self._values["job_role_arn"] = job_role_arn
            if linux_parameters is not None:
                self._values["linux_parameters"] = linux_parameters
            if log_configuration is not None:
                self._values["log_configuration"] = log_configuration
            if memory is not None:
                self._values["memory"] = memory
            if mount_points is not None:
                self._values["mount_points"] = mount_points
            if network_configuration is not None:
                self._values["network_configuration"] = network_configuration
            if privileged is not None:
                self._values["privileged"] = privileged
            if readonly_root_filesystem is not None:
                self._values["readonly_root_filesystem"] = readonly_root_filesystem
            if repository_credentials is not None:
                self._values["repository_credentials"] = repository_credentials
            if resource_requirements is not None:
                self._values["resource_requirements"] = resource_requirements
            if runtime_platform is not None:
                self._values["runtime_platform"] = runtime_platform
            if secrets is not None:
                self._values["secrets"] = secrets
            if ulimits is not None:
                self._values["ulimits"] = ulimits
            if user is not None:
                self._values["user"] = user
            if vcpus is not None:
                self._values["vcpus"] = vcpus
            if volumes is not None:
                self._values["volumes"] = volumes

        @builtins.property
        def image(self) -> builtins.str:
            '''Required.

            The image used to start a container. This string is passed directly to the Docker daemon. Images in the Docker Hub registry are available by default. Other repositories are specified with ``*repository-url* / *image* : *tag*`` . It can be 255 characters long. It can contain uppercase and lowercase letters, numbers, hyphens (-), underscores (_), colons (:), periods (.), forward slashes (/), and number signs (#). This parameter maps to ``Image`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``IMAGE`` parameter of `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .
            .. epigraph::

               Docker image architecture must match the processor architecture of the compute resources that they're scheduled on. For example, ARM-based Docker images can only run on ARM-based compute resources.

            - Images in Amazon ECR Public repositories use the full ``registry/repository[:tag]`` or ``registry/repository[@digest]`` naming conventions. For example, ``public.ecr.aws/ *registry_alias* / *my-web-app* : *latest*`` .
            - Images in Amazon ECR repositories use the full registry and repository URI (for example, ``123456789012.dkr.ecr.<region-name>.amazonaws.com/<repository-name>`` ).
            - Images in official repositories on Docker Hub use a single name (for example, ``ubuntu`` or ``mongo`` ).
            - Images in other repositories on Docker Hub are qualified with an organization name (for example, ``amazon/amazon-ecs-agent`` ).
            - Images in other online repositories are qualified further by a domain name (for example, ``quay.io/assemblyline/ubuntu`` ).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-image
            '''
            result = self._values.get("image")
            assert result is not None, "Required property 'image' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def command(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The command that's passed to the container.

            This parameter maps to ``Cmd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``COMMAND`` parameter to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . For more information, see `https://docs.docker.com/engine/reference/builder/#cmd <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#cmd>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-command
            '''
            result = self._values.get("command")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def environment(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EnvironmentProperty"]]]]:
            '''The environment variables to pass to a container.

            This parameter maps to ``Env`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--env`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .
            .. epigraph::

               We don't recommend using plaintext environment variables for sensitive information, such as credential data. > Environment variables cannot start with " ``AWS_BATCH`` ". This naming convention is reserved for variables that AWS Batch sets.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-environment
            '''
            result = self._values.get("environment")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EnvironmentProperty"]]]], result)

        @builtins.property
        def ephemeral_storage(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EphemeralStorageProperty"]]:
            '''The amount of ephemeral storage to allocate for the task.

            This parameter is used to expand the total amount of ephemeral storage available, beyond the default amount, for tasks hosted on AWS Fargate .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-ephemeralstorage
            '''
            result = self._values.get("ephemeral_storage")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EphemeralStorageProperty"]], result)

        @builtins.property
        def execution_role_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the execution role that AWS Batch can assume.

            For jobs that run on Fargate resources, you must provide an execution role. For more information, see `AWS Batch execution IAM role <https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html>`_ in the *AWS Batch User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-executionrolearn
            '''
            result = self._values.get("execution_role_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def fargate_platform_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.FargatePlatformConfigurationProperty"]]:
            '''The platform configuration for jobs that are running on Fargate resources.

            Jobs that are running on Amazon EC2 resources must not specify this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-fargateplatformconfiguration
            '''
            result = self._values.get("fargate_platform_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.FargatePlatformConfigurationProperty"]], result)

        @builtins.property
        def instance_type(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-instancetype
            '''
            result = self._values.get("instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def job_role_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the IAM role that the container can assume for AWS permissions.

            For more information, see `IAM roles for tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-jobrolearn
            '''
            result = self._values.get("job_role_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def linux_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.LinuxParametersProperty"]]:
            '''Linux-specific modifications that are applied to the container, such as details for device mappings.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-linuxparameters
            '''
            result = self._values.get("linux_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.LinuxParametersProperty"]], result)

        @builtins.property
        def log_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.LogConfigurationProperty"]]:
            '''The log configuration specification for the container.

            This parameter maps to ``LogConfig`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--log-driver`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . By default, containers use the same logging driver that the Docker daemon uses. However the container might use a different logging driver than the Docker daemon by specifying a log driver with this parameter in the container definition. To use a different logging driver for a container, the log system must be configured properly on the container instance (or on a different log server for remote logging options). For more information on the options for different supported log drivers, see `Configure logging drivers <https://docs.aws.amazon.com/https://docs.docker.com/engine/admin/logging/overview/>`_ in the Docker documentation.
            .. epigraph::

               AWS Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in the `LogConfiguration <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties-logconfiguration.html>`_ data type).

            This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version | grep "Server API version"``
            .. epigraph::

               The Amazon ECS container agent running on a container instance must register the logging drivers available on that instance with the ``ECS_AVAILABLE_LOGGING_DRIVERS`` environment variable before containers placed on that instance can use these log configuration options. For more information, see `Amazon ECS container agent configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-logconfiguration
            '''
            result = self._values.get("log_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.LogConfigurationProperty"]], result)

        @builtins.property
        def memory(self) -> typing.Optional[jsii.Number]:
            '''This parameter is deprecated, use ``resourceRequirements`` to specify the memory requirements for the job definition.

            It's not supported for jobs running on Fargate resources. For jobs that run on Amazon EC2 resources, it specifies the memory hard limit (in MiB) for a container. If your container attempts to exceed the specified number, it's terminated. You must specify at least 4 MiB of memory for a job using this parameter. The memory hard limit can be specified in several places. It must be specified for each node at least once.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-memory
            '''
            result = self._values.get("memory")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def mount_points(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.MountPointsProperty"]]]]:
            '''The mount points for data volumes in your container.

            This parameter maps to ``Volumes`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--volume`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-mountpoints
            '''
            result = self._values.get("mount_points")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.MountPointsProperty"]]]], result)

        @builtins.property
        def network_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.NetworkConfigurationProperty"]]:
            '''The network configuration for jobs that are running on Fargate resources.

            Jobs that are running on Amazon EC2 resources must not specify this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-networkconfiguration
            '''
            result = self._values.get("network_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.NetworkConfigurationProperty"]], result)

        @builtins.property
        def privileged(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''When this parameter is true, the container is given elevated permissions on the host container instance (similar to the ``root`` user).

            This parameter maps to ``Privileged`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--privileged`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . The default value is false.
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources and shouldn't be provided, or specified as false.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-privileged
            '''
            result = self._values.get("privileged")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def readonly_root_filesystem(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''When this parameter is true, the container is given read-only access to its root file system.

            This parameter maps to ``ReadonlyRootfs`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--read-only`` option to ``docker run`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-readonlyrootfilesystem
            '''
            result = self._values.get("readonly_root_filesystem")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def repository_credentials(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.RepositoryCredentialsProperty"]]:
            '''The private repository authentication credentials to use.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-repositorycredentials
            '''
            result = self._values.get("repository_credentials")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.RepositoryCredentialsProperty"]], result)

        @builtins.property
        def resource_requirements(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ResourceRequirementProperty"]]]]:
            '''The type and amount of resources to assign to a container.

            The supported resources include ``GPU`` , ``MEMORY`` , and ``VCPU`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-resourcerequirements
            '''
            result = self._values.get("resource_requirements")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ResourceRequirementProperty"]]]], result)

        @builtins.property
        def runtime_platform(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.RuntimePlatformProperty"]]:
            '''An object that represents the compute environment architecture for AWS Batch jobs on Fargate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-runtimeplatform
            '''
            result = self._values.get("runtime_platform")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.RuntimePlatformProperty"]], result)

        @builtins.property
        def secrets(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.SecretProperty"]]]]:
            '''The secrets for the container.

            For more information, see `Specifying sensitive data <https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html>`_ in the *AWS Batch User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-secrets
            '''
            result = self._values.get("secrets")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.SecretProperty"]]]], result)

        @builtins.property
        def ulimits(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.UlimitProperty"]]]]:
            '''A list of ``ulimits`` to set in the container.

            This parameter maps to ``Ulimits`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--ulimit`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources and shouldn't be provided.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-ulimits
            '''
            result = self._values.get("ulimits")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.UlimitProperty"]]]], result)

        @builtins.property
        def user(self) -> typing.Optional[builtins.str]:
            '''The user name to use inside the container.

            This parameter maps to ``User`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--user`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-user
            '''
            result = self._values.get("user")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def vcpus(self) -> typing.Optional[jsii.Number]:
            '''This parameter is deprecated, use ``resourceRequirements`` to specify the vCPU requirements for the job definition.

            It's not supported for jobs running on Fargate resources. For jobs running on Amazon EC2 resources, it specifies the number of vCPUs reserved for the job.

            Each vCPU is equivalent to 1,024 CPU shares. This parameter maps to ``CpuShares`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--cpu-shares`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . The number of vCPUs must be specified but can be specified in several places. You must specify it at least once for each node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-vcpus
            '''
            result = self._values.get("vcpus")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def volumes(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.VolumesProperty"]]]]:
            '''A list of data volumes used in a job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-containerproperties.html#cfn-batch-jobdefinition-containerproperties-volumes
            '''
            result = self._values.get("volumes")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.VolumesProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContainerPropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.DeviceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_path": "containerPath",
            "host_path": "hostPath",
            "permissions": "permissions",
        },
    )
    class DeviceProperty:
        def __init__(
            self,
            *,
            container_path: typing.Optional[builtins.str] = None,
            host_path: typing.Optional[builtins.str] = None,
            permissions: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents a container instance host device.

            .. epigraph::

               This object isn't applicable to jobs that are running on Fargate resources and shouldn't be provided.

            :param container_path: The path inside the container that's used to expose the host device. By default, the ``hostPath`` value is used.
            :param host_path: The path for the device on the host container instance.
            :param permissions: The explicit permissions to provide to the container for the device. By default, the container has permissions for ``read`` , ``write`` , and ``mknod`` for the device.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-device.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                device_property = batch.CfnJobDefinition.DeviceProperty(
                    container_path="containerPath",
                    host_path="hostPath",
                    permissions=["permissions"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__adbf78805f7f3d50af2c331ead7e1f53e092ea7a76782151236bad4f3bfc77b0)
                check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
                check_type(argname="argument host_path", value=host_path, expected_type=type_hints["host_path"])
                check_type(argname="argument permissions", value=permissions, expected_type=type_hints["permissions"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if container_path is not None:
                self._values["container_path"] = container_path
            if host_path is not None:
                self._values["host_path"] = host_path
            if permissions is not None:
                self._values["permissions"] = permissions

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''The path inside the container that's used to expose the host device.

            By default, the ``hostPath`` value is used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-device.html#cfn-batch-jobdefinition-device-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_path(self) -> typing.Optional[builtins.str]:
            '''The path for the device on the host container instance.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-device.html#cfn-batch-jobdefinition-device-hostpath
            '''
            result = self._values.get("host_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def permissions(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The explicit permissions to provide to the container for the device.

            By default, the container has permissions for ``read`` , ``write`` , and ``mknod`` for the device.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-device.html#cfn-batch-jobdefinition-device-permissions
            '''
            result = self._values.get("permissions")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeviceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EcsPropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={"task_properties": "taskProperties"},
    )
    class EcsPropertiesProperty:
        def __init__(
            self,
            *,
            task_properties: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EcsTaskPropertiesProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''An object that contains the properties for the Amazon ECS resources of a job.

            :param task_properties: An object that contains the properties for the Amazon ECS task definition of a job. .. epigraph:: This object is currently limited to one task element. However, the task element can run up to 10 containers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecsproperties.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # options: Any
                
                ecs_properties_property = batch.CfnJobDefinition.EcsPropertiesProperty(
                    task_properties=[batch.CfnJobDefinition.EcsTaskPropertiesProperty(
                        containers=[batch.CfnJobDefinition.TaskContainerPropertiesProperty(
                            image="image",
                
                            # the properties below are optional
                            command=["command"],
                            depends_on=[batch.CfnJobDefinition.TaskContainerDependencyProperty(
                                condition="condition",
                                container_name="containerName"
                            )],
                            environment=[batch.CfnJobDefinition.EnvironmentProperty(
                                name="name",
                                value="value"
                            )],
                            essential=False,
                            linux_parameters=batch.CfnJobDefinition.LinuxParametersProperty(
                                devices=[batch.CfnJobDefinition.DeviceProperty(
                                    container_path="containerPath",
                                    host_path="hostPath",
                                    permissions=["permissions"]
                                )],
                                init_process_enabled=False,
                                max_swap=123,
                                shared_memory_size=123,
                                swappiness=123,
                                tmpfs=[batch.CfnJobDefinition.TmpfsProperty(
                                    container_path="containerPath",
                                    size=123,
                
                                    # the properties below are optional
                                    mount_options=["mountOptions"]
                                )]
                            ),
                            log_configuration=batch.CfnJobDefinition.LogConfigurationProperty(
                                log_driver="logDriver",
                
                                # the properties below are optional
                                options=options,
                                secret_options=[batch.CfnJobDefinition.SecretProperty(
                                    name="name",
                                    value_from="valueFrom"
                                )]
                            ),
                            mount_points=[batch.CfnJobDefinition.MountPointProperty(
                                container_path="containerPath",
                                read_only=False,
                                source_volume="sourceVolume"
                            )],
                            name="name",
                            privileged=False,
                            readonly_root_filesystem=False,
                            repository_credentials=batch.CfnJobDefinition.RepositoryCredentialsProperty(
                                credentials_parameter="credentialsParameter"
                            ),
                            resource_requirements=[batch.CfnJobDefinition.ResourceRequirementProperty(
                                type="type",
                                value="value"
                            )],
                            secrets=[batch.CfnJobDefinition.SecretProperty(
                                name="name",
                                value_from="valueFrom"
                            )],
                            ulimits=[batch.CfnJobDefinition.UlimitProperty(
                                hard_limit=123,
                                name="name",
                                soft_limit=123
                            )],
                            user="user"
                        )],
                        ephemeral_storage=batch.CfnJobDefinition.EphemeralStorageProperty(
                            size_in_gi_b=123
                        ),
                        execution_role_arn="executionRoleArn",
                        ipc_mode="ipcMode",
                        network_configuration=batch.CfnJobDefinition.NetworkConfigurationProperty(
                            assign_public_ip="assignPublicIp"
                        ),
                        pid_mode="pidMode",
                        platform_version="platformVersion",
                        runtime_platform=batch.CfnJobDefinition.RuntimePlatformProperty(
                            cpu_architecture="cpuArchitecture",
                            operating_system_family="operatingSystemFamily"
                        ),
                        task_role_arn="taskRoleArn",
                        volumes=[batch.CfnJobDefinition.VolumesProperty(
                            efs_volume_configuration=batch.CfnJobDefinition.EfsVolumeConfigurationProperty(
                                file_system_id="fileSystemId",
                
                                # the properties below are optional
                                authorization_config=batch.CfnJobDefinition.AuthorizationConfigProperty(
                                    access_point_id="accessPointId",
                                    iam="iam"
                                ),
                                root_directory="rootDirectory",
                                transit_encryption="transitEncryption",
                                transit_encryption_port=123
                            ),
                            host=batch.CfnJobDefinition.VolumesHostProperty(
                                source_path="sourcePath"
                            ),
                            name="name"
                        )]
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6153a950309796a66c44ef3a234e02ed7ff91444107709427ae345e4f0e64828)
                check_type(argname="argument task_properties", value=task_properties, expected_type=type_hints["task_properties"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "task_properties": task_properties,
            }

        @builtins.property
        def task_properties(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EcsTaskPropertiesProperty"]]]:
            '''An object that contains the properties for the Amazon ECS task definition of a job.

            .. epigraph::

               This object is currently limited to one task element. However, the task element can run up to 10 containers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecsproperties.html#cfn-batch-jobdefinition-ecsproperties-taskproperties
            '''
            result = self._values.get("task_properties")
            assert result is not None, "Required property 'task_properties' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EcsTaskPropertiesProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EcsPropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EcsTaskPropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "containers": "containers",
            "ephemeral_storage": "ephemeralStorage",
            "execution_role_arn": "executionRoleArn",
            "ipc_mode": "ipcMode",
            "network_configuration": "networkConfiguration",
            "pid_mode": "pidMode",
            "platform_version": "platformVersion",
            "runtime_platform": "runtimePlatform",
            "task_role_arn": "taskRoleArn",
            "volumes": "volumes",
        },
    )
    class EcsTaskPropertiesProperty:
        def __init__(
            self,
            *,
            containers: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.TaskContainerPropertiesProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            ephemeral_storage: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EphemeralStorageProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            execution_role_arn: typing.Optional[builtins.str] = None,
            ipc_mode: typing.Optional[builtins.str] = None,
            network_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.NetworkConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            pid_mode: typing.Optional[builtins.str] = None,
            platform_version: typing.Optional[builtins.str] = None,
            runtime_platform: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.RuntimePlatformProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            task_role_arn: typing.Optional[builtins.str] = None,
            volumes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.VolumesProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The properties for a task definition that describes the container and volume definitions of an Amazon ECS task.

            You can specify which Docker images to use, the required resources, and other configurations related to launching the task definition through an Amazon ECS service or task.

            :param containers: This object is a list of containers.
            :param ephemeral_storage: The amount of ephemeral storage to allocate for the task. This parameter is used to expand the total amount of ephemeral storage available, beyond the default amount, for tasks hosted on AWS Fargate .
            :param execution_role_arn: The Amazon Resource Name (ARN) of the execution role that AWS Batch can assume. For jobs that run on Fargate resources, you must provide an execution role. For more information, see `AWS Batch execution IAM role <https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html>`_ in the *AWS Batch User Guide* .
            :param ipc_mode: The IPC resource namespace to use for the containers in the task. The valid values are ``host`` , ``task`` , or ``none`` . If ``host`` is specified, all containers within the tasks that specified the ``host`` IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified ``task`` share the same IPC resources. If ``none`` is specified, the IPC resources within the containers of a task are private, and are not shared with other containers in a task or on the container instance. If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance. For more information, see `IPC settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#ipc-settings---ipc>`_ in the Docker run reference.
            :param network_configuration: The network configuration for jobs that are running on Fargate resources. Jobs that are running on Amazon EC2 resources must not specify this parameter.
            :param pid_mode: The process namespace to use for the containers in the task. The valid values are ``host`` or ``task`` . For example, monitoring sidecars might need ``pidMode`` to access information about other containers running in the same task. If ``host`` is specified, all containers within the tasks that specified the ``host`` PID mode on the same container instance share the process namespace with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same process namespace. If no value is specified, the default is a private namespace for each container. For more information, see `PID settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#pid-settings---pid>`_ in the Docker run reference.
            :param platform_version: The Fargate platform version where the jobs are running. A platform version is specified only for jobs that are running on Fargate resources. If one isn't specified, the ``LATEST`` platform version is used by default. This uses a recent, approved version of the Fargate platform for compute resources. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param runtime_platform: An object that represents the compute environment architecture for AWS Batch jobs on Fargate.
            :param task_role_arn: The Amazon Resource Name (ARN) that's associated with the Amazon ECS task. .. epigraph:: This is object is comparable to `ContainerProperties:jobRoleArn <https://docs.aws.amazon.com/batch/latest/APIReference/API_ContainerProperties.html>`_ .
            :param volumes: A list of volumes that are associated with the job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecstaskproperties.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # options: Any
                
                ecs_task_properties_property = batch.CfnJobDefinition.EcsTaskPropertiesProperty(
                    containers=[batch.CfnJobDefinition.TaskContainerPropertiesProperty(
                        image="image",
                
                        # the properties below are optional
                        command=["command"],
                        depends_on=[batch.CfnJobDefinition.TaskContainerDependencyProperty(
                            condition="condition",
                            container_name="containerName"
                        )],
                        environment=[batch.CfnJobDefinition.EnvironmentProperty(
                            name="name",
                            value="value"
                        )],
                        essential=False,
                        linux_parameters=batch.CfnJobDefinition.LinuxParametersProperty(
                            devices=[batch.CfnJobDefinition.DeviceProperty(
                                container_path="containerPath",
                                host_path="hostPath",
                                permissions=["permissions"]
                            )],
                            init_process_enabled=False,
                            max_swap=123,
                            shared_memory_size=123,
                            swappiness=123,
                            tmpfs=[batch.CfnJobDefinition.TmpfsProperty(
                                container_path="containerPath",
                                size=123,
                
                                # the properties below are optional
                                mount_options=["mountOptions"]
                            )]
                        ),
                        log_configuration=batch.CfnJobDefinition.LogConfigurationProperty(
                            log_driver="logDriver",
                
                            # the properties below are optional
                            options=options,
                            secret_options=[batch.CfnJobDefinition.SecretProperty(
                                name="name",
                                value_from="valueFrom"
                            )]
                        ),
                        mount_points=[batch.CfnJobDefinition.MountPointProperty(
                            container_path="containerPath",
                            read_only=False,
                            source_volume="sourceVolume"
                        )],
                        name="name",
                        privileged=False,
                        readonly_root_filesystem=False,
                        repository_credentials=batch.CfnJobDefinition.RepositoryCredentialsProperty(
                            credentials_parameter="credentialsParameter"
                        ),
                        resource_requirements=[batch.CfnJobDefinition.ResourceRequirementProperty(
                            type="type",
                            value="value"
                        )],
                        secrets=[batch.CfnJobDefinition.SecretProperty(
                            name="name",
                            value_from="valueFrom"
                        )],
                        ulimits=[batch.CfnJobDefinition.UlimitProperty(
                            hard_limit=123,
                            name="name",
                            soft_limit=123
                        )],
                        user="user"
                    )],
                    ephemeral_storage=batch.CfnJobDefinition.EphemeralStorageProperty(
                        size_in_gi_b=123
                    ),
                    execution_role_arn="executionRoleArn",
                    ipc_mode="ipcMode",
                    network_configuration=batch.CfnJobDefinition.NetworkConfigurationProperty(
                        assign_public_ip="assignPublicIp"
                    ),
                    pid_mode="pidMode",
                    platform_version="platformVersion",
                    runtime_platform=batch.CfnJobDefinition.RuntimePlatformProperty(
                        cpu_architecture="cpuArchitecture",
                        operating_system_family="operatingSystemFamily"
                    ),
                    task_role_arn="taskRoleArn",
                    volumes=[batch.CfnJobDefinition.VolumesProperty(
                        efs_volume_configuration=batch.CfnJobDefinition.EfsVolumeConfigurationProperty(
                            file_system_id="fileSystemId",
                
                            # the properties below are optional
                            authorization_config=batch.CfnJobDefinition.AuthorizationConfigProperty(
                                access_point_id="accessPointId",
                                iam="iam"
                            ),
                            root_directory="rootDirectory",
                            transit_encryption="transitEncryption",
                            transit_encryption_port=123
                        ),
                        host=batch.CfnJobDefinition.VolumesHostProperty(
                            source_path="sourcePath"
                        ),
                        name="name"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8e8c39be2b840a158045732ea20d964bf03d380304bbcd66936cb862e6cc2a8e)
                check_type(argname="argument containers", value=containers, expected_type=type_hints["containers"])
                check_type(argname="argument ephemeral_storage", value=ephemeral_storage, expected_type=type_hints["ephemeral_storage"])
                check_type(argname="argument execution_role_arn", value=execution_role_arn, expected_type=type_hints["execution_role_arn"])
                check_type(argname="argument ipc_mode", value=ipc_mode, expected_type=type_hints["ipc_mode"])
                check_type(argname="argument network_configuration", value=network_configuration, expected_type=type_hints["network_configuration"])
                check_type(argname="argument pid_mode", value=pid_mode, expected_type=type_hints["pid_mode"])
                check_type(argname="argument platform_version", value=platform_version, expected_type=type_hints["platform_version"])
                check_type(argname="argument runtime_platform", value=runtime_platform, expected_type=type_hints["runtime_platform"])
                check_type(argname="argument task_role_arn", value=task_role_arn, expected_type=type_hints["task_role_arn"])
                check_type(argname="argument volumes", value=volumes, expected_type=type_hints["volumes"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if containers is not None:
                self._values["containers"] = containers
            if ephemeral_storage is not None:
                self._values["ephemeral_storage"] = ephemeral_storage
            if execution_role_arn is not None:
                self._values["execution_role_arn"] = execution_role_arn
            if ipc_mode is not None:
                self._values["ipc_mode"] = ipc_mode
            if network_configuration is not None:
                self._values["network_configuration"] = network_configuration
            if pid_mode is not None:
                self._values["pid_mode"] = pid_mode
            if platform_version is not None:
                self._values["platform_version"] = platform_version
            if runtime_platform is not None:
                self._values["runtime_platform"] = runtime_platform
            if task_role_arn is not None:
                self._values["task_role_arn"] = task_role_arn
            if volumes is not None:
                self._values["volumes"] = volumes

        @builtins.property
        def containers(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.TaskContainerPropertiesProperty"]]]]:
            '''This object is a list of containers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecstaskproperties.html#cfn-batch-jobdefinition-ecstaskproperties-containers
            '''
            result = self._values.get("containers")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.TaskContainerPropertiesProperty"]]]], result)

        @builtins.property
        def ephemeral_storage(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EphemeralStorageProperty"]]:
            '''The amount of ephemeral storage to allocate for the task.

            This parameter is used to expand the total amount of ephemeral storage available, beyond the default amount, for tasks hosted on AWS Fargate .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecstaskproperties.html#cfn-batch-jobdefinition-ecstaskproperties-ephemeralstorage
            '''
            result = self._values.get("ephemeral_storage")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EphemeralStorageProperty"]], result)

        @builtins.property
        def execution_role_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the execution role that AWS Batch can assume.

            For jobs that run on Fargate resources, you must provide an execution role. For more information, see `AWS Batch execution IAM role <https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html>`_ in the *AWS Batch User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecstaskproperties.html#cfn-batch-jobdefinition-ecstaskproperties-executionrolearn
            '''
            result = self._values.get("execution_role_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ipc_mode(self) -> typing.Optional[builtins.str]:
            '''The IPC resource namespace to use for the containers in the task.

            The valid values are ``host`` , ``task`` , or ``none`` .

            If ``host`` is specified, all containers within the tasks that specified the ``host`` IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance.

            If ``task`` is specified, all containers within the specified ``task`` share the same IPC resources.

            If ``none`` is specified, the IPC resources within the containers of a task are private, and are not shared with other containers in a task or on the container instance.

            If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance. For more information, see `IPC settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#ipc-settings---ipc>`_ in the Docker run reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecstaskproperties.html#cfn-batch-jobdefinition-ecstaskproperties-ipcmode
            '''
            result = self._values.get("ipc_mode")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def network_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.NetworkConfigurationProperty"]]:
            '''The network configuration for jobs that are running on Fargate resources.

            Jobs that are running on Amazon EC2 resources must not specify this parameter.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecstaskproperties.html#cfn-batch-jobdefinition-ecstaskproperties-networkconfiguration
            '''
            result = self._values.get("network_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.NetworkConfigurationProperty"]], result)

        @builtins.property
        def pid_mode(self) -> typing.Optional[builtins.str]:
            '''The process namespace to use for the containers in the task.

            The valid values are ``host`` or ``task`` . For example, monitoring sidecars might need ``pidMode`` to access information about other containers running in the same task.

            If ``host`` is specified, all containers within the tasks that specified the ``host`` PID mode on the same container instance share the process namespace with the host Amazon EC2 instance.

            If ``task`` is specified, all containers within the specified task share the same process namespace.

            If no value is specified, the default is a private namespace for each container. For more information, see `PID settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#pid-settings---pid>`_ in the Docker run reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecstaskproperties.html#cfn-batch-jobdefinition-ecstaskproperties-pidmode
            '''
            result = self._values.get("pid_mode")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def platform_version(self) -> typing.Optional[builtins.str]:
            '''The Fargate platform version where the jobs are running.

            A platform version is specified only for jobs that are running on Fargate resources. If one isn't specified, the ``LATEST`` platform version is used by default. This uses a recent, approved version of the Fargate platform for compute resources. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecstaskproperties.html#cfn-batch-jobdefinition-ecstaskproperties-platformversion
            '''
            result = self._values.get("platform_version")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def runtime_platform(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.RuntimePlatformProperty"]]:
            '''An object that represents the compute environment architecture for AWS Batch jobs on Fargate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecstaskproperties.html#cfn-batch-jobdefinition-ecstaskproperties-runtimeplatform
            '''
            result = self._values.get("runtime_platform")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.RuntimePlatformProperty"]], result)

        @builtins.property
        def task_role_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) that's associated with the Amazon ECS task.

            .. epigraph::

               This is object is comparable to `ContainerProperties:jobRoleArn <https://docs.aws.amazon.com/batch/latest/APIReference/API_ContainerProperties.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecstaskproperties.html#cfn-batch-jobdefinition-ecstaskproperties-taskrolearn
            '''
            result = self._values.get("task_role_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def volumes(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.VolumesProperty"]]]]:
            '''A list of volumes that are associated with the job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ecstaskproperties.html#cfn-batch-jobdefinition-ecstaskproperties-volumes
            '''
            result = self._values.get("volumes")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.VolumesProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EcsTaskPropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EfsVolumeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "file_system_id": "fileSystemId",
            "authorization_config": "authorizationConfig",
            "root_directory": "rootDirectory",
            "transit_encryption": "transitEncryption",
            "transit_encryption_port": "transitEncryptionPort",
        },
    )
    class EfsVolumeConfigurationProperty:
        def __init__(
            self,
            *,
            file_system_id: builtins.str,
            authorization_config: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.AuthorizationConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            root_directory: typing.Optional[builtins.str] = None,
            transit_encryption: typing.Optional[builtins.str] = None,
            transit_encryption_port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param file_system_id: 
            :param authorization_config: 
            :param root_directory: 
            :param transit_encryption: 
            :param transit_encryption_port: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-efsvolumeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                efs_volume_configuration_property = batch.CfnJobDefinition.EfsVolumeConfigurationProperty(
                    file_system_id="fileSystemId",
                
                    # the properties below are optional
                    authorization_config=batch.CfnJobDefinition.AuthorizationConfigProperty(
                        access_point_id="accessPointId",
                        iam="iam"
                    ),
                    root_directory="rootDirectory",
                    transit_encryption="transitEncryption",
                    transit_encryption_port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f673b085b46133477cd372799511104e89e28bf34ac0f6255020b20ef8b7e3e0)
                check_type(argname="argument file_system_id", value=file_system_id, expected_type=type_hints["file_system_id"])
                check_type(argname="argument authorization_config", value=authorization_config, expected_type=type_hints["authorization_config"])
                check_type(argname="argument root_directory", value=root_directory, expected_type=type_hints["root_directory"])
                check_type(argname="argument transit_encryption", value=transit_encryption, expected_type=type_hints["transit_encryption"])
                check_type(argname="argument transit_encryption_port", value=transit_encryption_port, expected_type=type_hints["transit_encryption_port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "file_system_id": file_system_id,
            }
            if authorization_config is not None:
                self._values["authorization_config"] = authorization_config
            if root_directory is not None:
                self._values["root_directory"] = root_directory
            if transit_encryption is not None:
                self._values["transit_encryption"] = transit_encryption
            if transit_encryption_port is not None:
                self._values["transit_encryption_port"] = transit_encryption_port

        @builtins.property
        def file_system_id(self) -> builtins.str:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-efsvolumeconfiguration.html#cfn-batch-jobdefinition-efsvolumeconfiguration-filesystemid
            '''
            result = self._values.get("file_system_id")
            assert result is not None, "Required property 'file_system_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def authorization_config(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.AuthorizationConfigProperty"]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-efsvolumeconfiguration.html#cfn-batch-jobdefinition-efsvolumeconfiguration-authorizationconfig
            '''
            result = self._values.get("authorization_config")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.AuthorizationConfigProperty"]], result)

        @builtins.property
        def root_directory(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-efsvolumeconfiguration.html#cfn-batch-jobdefinition-efsvolumeconfiguration-rootdirectory
            '''
            result = self._values.get("root_directory")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def transit_encryption(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-efsvolumeconfiguration.html#cfn-batch-jobdefinition-efsvolumeconfiguration-transitencryption
            '''
            result = self._values.get("transit_encryption")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def transit_encryption_port(self) -> typing.Optional[jsii.Number]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-efsvolumeconfiguration.html#cfn-batch-jobdefinition-efsvolumeconfiguration-transitencryptionport
            '''
            result = self._values.get("transit_encryption_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EfsVolumeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EksContainerEnvironmentVariableProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class EksContainerEnvironmentVariableProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An environment variable.

            :param name: The name of the environment variable.
            :param value: The value of the environment variable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainerenvironmentvariable.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                eks_container_environment_variable_property = batch.CfnJobDefinition.EksContainerEnvironmentVariableProperty(
                    name="name",
                
                    # the properties below are optional
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__daaaeed078e13c29ac34da5a79bab4c04e8767edaf6d6a60877918a94797c218)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the environment variable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainerenvironmentvariable.html#cfn-batch-jobdefinition-ekscontainerenvironmentvariable-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value of the environment variable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainerenvironmentvariable.html#cfn-batch-jobdefinition-ekscontainerenvironmentvariable-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EksContainerEnvironmentVariableProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EksContainerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "image": "image",
            "args": "args",
            "command": "command",
            "env": "env",
            "image_pull_policy": "imagePullPolicy",
            "name": "name",
            "resources": "resources",
            "security_context": "securityContext",
            "volume_mounts": "volumeMounts",
        },
    )
    class EksContainerProperty:
        def __init__(
            self,
            *,
            image: builtins.str,
            args: typing.Optional[typing.Sequence[builtins.str]] = None,
            command: typing.Optional[typing.Sequence[builtins.str]] = None,
            env: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EksContainerEnvironmentVariableProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            image_pull_policy: typing.Optional[builtins.str] = None,
            name: typing.Optional[builtins.str] = None,
            resources: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.ResourcesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            security_context: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.SecurityContextProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            volume_mounts: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EksContainerVolumeMountProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''EKS container properties are used in job definitions for Amazon EKS based job definitions to describe the properties for a container node in the pod that's launched as part of a job.

            This can't be specified for Amazon ECS based job definitions.

            :param image: The Docker image used to start the container.
            :param args: An array of arguments to the entrypoint. If this isn't specified, the ``CMD`` of the container image is used. This corresponds to the ``args`` member in the `Entrypoint <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint>`_ portion of the `Pod <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/>`_ in Kubernetes. Environment variable references are expanded using the container's environment. If the referenced environment variable doesn't exist, the reference in the command isn't changed. For example, if the reference is to " ``$(NAME1)`` " and the ``NAME1`` environment variable doesn't exist, the command string will remain " ``$(NAME1)`` ." ``$$`` is replaced with ``$`` , and the resulting string isn't expanded. For example, ``$$(VAR_NAME)`` is passed as ``$(VAR_NAME)`` whether or not the ``VAR_NAME`` environment variable exists. For more information, see `Dockerfile reference: CMD <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#cmd>`_ and `Define a command and arguments for a pod <https://docs.aws.amazon.com/https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/>`_ in the *Kubernetes documentation* .
            :param command: The entrypoint for the container. This isn't run within a shell. If this isn't specified, the ``ENTRYPOINT`` of the container image is used. Environment variable references are expanded using the container's environment. If the referenced environment variable doesn't exist, the reference in the command isn't changed. For example, if the reference is to " ``$(NAME1)`` " and the ``NAME1`` environment variable doesn't exist, the command string will remain " ``$(NAME1)`` ." ``$$`` is replaced with ``$`` and the resulting string isn't expanded. For example, ``$$(VAR_NAME)`` will be passed as ``$(VAR_NAME)`` whether or not the ``VAR_NAME`` environment variable exists. The entrypoint can't be updated. For more information, see `ENTRYPOINT <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#entrypoint>`_ in the *Dockerfile reference* and `Define a command and arguments for a container <https://docs.aws.amazon.com/https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/>`_ and `Entrypoint <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint>`_ in the *Kubernetes documentation* .
            :param env: The environment variables to pass to a container. .. epigraph:: Environment variables cannot start with " ``AWS_BATCH`` ". This naming convention is reserved for variables that AWS Batch sets.
            :param image_pull_policy: The image pull policy for the container. Supported values are ``Always`` , ``IfNotPresent`` , and ``Never`` . This parameter defaults to ``IfNotPresent`` . However, if the ``:latest`` tag is specified, it defaults to ``Always`` . For more information, see `Updating images <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/containers/images/#updating-images>`_ in the *Kubernetes documentation* .
            :param name: The name of the container. If the name isn't specified, the default name " ``Default`` " is used. Each container in a pod must have a unique name.
            :param resources: The type and amount of resources to assign to a container. The supported resources include ``memory`` , ``cpu`` , and ``nvidia.com/gpu`` . For more information, see `Resource management for pods and containers <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/>`_ in the *Kubernetes documentation* .
            :param security_context: The security context for a job. For more information, see `Configure a security context for a pod or container <https://docs.aws.amazon.com/https://kubernetes.io/docs/tasks/configure-pod-container/security-context/>`_ in the *Kubernetes documentation* .
            :param volume_mounts: The volume mounts for the container. AWS Batch supports ``emptyDir`` , ``hostPath`` , and ``secret`` volume types. For more information about volumes and volume mounts in Kubernetes, see `Volumes <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainer.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # limits: Any
                # requests: Any
                
                eks_container_property = batch.CfnJobDefinition.EksContainerProperty(
                    image="image",
                
                    # the properties below are optional
                    args=["args"],
                    command=["command"],
                    env=[batch.CfnJobDefinition.EksContainerEnvironmentVariableProperty(
                        name="name",
                
                        # the properties below are optional
                        value="value"
                    )],
                    image_pull_policy="imagePullPolicy",
                    name="name",
                    resources=batch.CfnJobDefinition.ResourcesProperty(
                        limits=limits,
                        requests=requests
                    ),
                    security_context=batch.CfnJobDefinition.SecurityContextProperty(
                        allow_privilege_escalation=False,
                        privileged=False,
                        read_only_root_filesystem=False,
                        run_as_group=123,
                        run_as_non_root=False,
                        run_as_user=123
                    ),
                    volume_mounts=[batch.CfnJobDefinition.EksContainerVolumeMountProperty(
                        mount_path="mountPath",
                        name="name",
                        read_only=False,
                        sub_path="subPath"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a9ece3a6d2417f8d913bdfc7bdf646d1de352c65903e333939133c3e81b1f4f3)
                check_type(argname="argument image", value=image, expected_type=type_hints["image"])
                check_type(argname="argument args", value=args, expected_type=type_hints["args"])
                check_type(argname="argument command", value=command, expected_type=type_hints["command"])
                check_type(argname="argument env", value=env, expected_type=type_hints["env"])
                check_type(argname="argument image_pull_policy", value=image_pull_policy, expected_type=type_hints["image_pull_policy"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument resources", value=resources, expected_type=type_hints["resources"])
                check_type(argname="argument security_context", value=security_context, expected_type=type_hints["security_context"])
                check_type(argname="argument volume_mounts", value=volume_mounts, expected_type=type_hints["volume_mounts"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "image": image,
            }
            if args is not None:
                self._values["args"] = args
            if command is not None:
                self._values["command"] = command
            if env is not None:
                self._values["env"] = env
            if image_pull_policy is not None:
                self._values["image_pull_policy"] = image_pull_policy
            if name is not None:
                self._values["name"] = name
            if resources is not None:
                self._values["resources"] = resources
            if security_context is not None:
                self._values["security_context"] = security_context
            if volume_mounts is not None:
                self._values["volume_mounts"] = volume_mounts

        @builtins.property
        def image(self) -> builtins.str:
            '''The Docker image used to start the container.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainer.html#cfn-batch-jobdefinition-ekscontainer-image
            '''
            result = self._values.get("image")
            assert result is not None, "Required property 'image' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def args(self) -> typing.Optional[typing.List[builtins.str]]:
            '''An array of arguments to the entrypoint.

            If this isn't specified, the ``CMD`` of the container image is used. This corresponds to the ``args`` member in the `Entrypoint <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint>`_ portion of the `Pod <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/>`_ in Kubernetes. Environment variable references are expanded using the container's environment.

            If the referenced environment variable doesn't exist, the reference in the command isn't changed. For example, if the reference is to " ``$(NAME1)`` " and the ``NAME1`` environment variable doesn't exist, the command string will remain " ``$(NAME1)`` ." ``$$`` is replaced with ``$`` , and the resulting string isn't expanded. For example, ``$$(VAR_NAME)`` is passed as ``$(VAR_NAME)`` whether or not the ``VAR_NAME`` environment variable exists. For more information, see `Dockerfile reference: CMD <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#cmd>`_ and `Define a command and arguments for a pod <https://docs.aws.amazon.com/https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainer.html#cfn-batch-jobdefinition-ekscontainer-args
            '''
            result = self._values.get("args")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def command(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The entrypoint for the container.

            This isn't run within a shell. If this isn't specified, the ``ENTRYPOINT`` of the container image is used. Environment variable references are expanded using the container's environment.

            If the referenced environment variable doesn't exist, the reference in the command isn't changed. For example, if the reference is to " ``$(NAME1)`` " and the ``NAME1`` environment variable doesn't exist, the command string will remain " ``$(NAME1)`` ." ``$$`` is replaced with ``$`` and the resulting string isn't expanded. For example, ``$$(VAR_NAME)`` will be passed as ``$(VAR_NAME)`` whether or not the ``VAR_NAME`` environment variable exists. The entrypoint can't be updated. For more information, see `ENTRYPOINT <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#entrypoint>`_ in the *Dockerfile reference* and `Define a command and arguments for a container <https://docs.aws.amazon.com/https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/>`_ and `Entrypoint <https://docs.aws.amazon.com/https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainer.html#cfn-batch-jobdefinition-ekscontainer-command
            '''
            result = self._values.get("command")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def env(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksContainerEnvironmentVariableProperty"]]]]:
            '''The environment variables to pass to a container.

            .. epigraph::

               Environment variables cannot start with " ``AWS_BATCH`` ". This naming convention is reserved for variables that AWS Batch sets.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainer.html#cfn-batch-jobdefinition-ekscontainer-env
            '''
            result = self._values.get("env")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksContainerEnvironmentVariableProperty"]]]], result)

        @builtins.property
        def image_pull_policy(self) -> typing.Optional[builtins.str]:
            '''The image pull policy for the container.

            Supported values are ``Always`` , ``IfNotPresent`` , and ``Never`` . This parameter defaults to ``IfNotPresent`` . However, if the ``:latest`` tag is specified, it defaults to ``Always`` . For more information, see `Updating images <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/containers/images/#updating-images>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainer.html#cfn-batch-jobdefinition-ekscontainer-imagepullpolicy
            '''
            result = self._values.get("image_pull_policy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the container.

            If the name isn't specified, the default name " ``Default`` " is used. Each container in a pod must have a unique name.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainer.html#cfn-batch-jobdefinition-ekscontainer-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def resources(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ResourcesProperty"]]:
            '''The type and amount of resources to assign to a container.

            The supported resources include ``memory`` , ``cpu`` , and ``nvidia.com/gpu`` . For more information, see `Resource management for pods and containers <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainer.html#cfn-batch-jobdefinition-ekscontainer-resources
            '''
            result = self._values.get("resources")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ResourcesProperty"]], result)

        @builtins.property
        def security_context(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.SecurityContextProperty"]]:
            '''The security context for a job.

            For more information, see `Configure a security context for a pod or container <https://docs.aws.amazon.com/https://kubernetes.io/docs/tasks/configure-pod-container/security-context/>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainer.html#cfn-batch-jobdefinition-ekscontainer-securitycontext
            '''
            result = self._values.get("security_context")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.SecurityContextProperty"]], result)

        @builtins.property
        def volume_mounts(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksContainerVolumeMountProperty"]]]]:
            '''The volume mounts for the container.

            AWS Batch supports ``emptyDir`` , ``hostPath`` , and ``secret`` volume types. For more information about volumes and volume mounts in Kubernetes, see `Volumes <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainer.html#cfn-batch-jobdefinition-ekscontainer-volumemounts
            '''
            result = self._values.get("volume_mounts")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksContainerVolumeMountProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EksContainerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EksContainerVolumeMountProperty",
        jsii_struct_bases=[],
        name_mapping={
            "mount_path": "mountPath",
            "name": "name",
            "read_only": "readOnly",
            "sub_path": "subPath",
        },
    )
    class EksContainerVolumeMountProperty:
        def __init__(
            self,
            *,
            mount_path: typing.Optional[builtins.str] = None,
            name: typing.Optional[builtins.str] = None,
            read_only: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            sub_path: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The volume mounts for a container for an Amazon EKS job.

            For more information about volumes and volume mounts in Kubernetes, see `Volumes <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/>`_ in the *Kubernetes documentation* .

            :param mount_path: The path on the container where the volume is mounted.
            :param name: The name the volume mount. This must match the name of one of the volumes in the pod.
            :param read_only: If this value is ``true`` , the container has read-only access to the volume. Otherwise, the container can write to the volume. The default value is ``false`` .
            :param sub_path: A sub-path inside the referenced volume instead of its root.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainervolumemount.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                eks_container_volume_mount_property = batch.CfnJobDefinition.EksContainerVolumeMountProperty(
                    mount_path="mountPath",
                    name="name",
                    read_only=False,
                    sub_path="subPath"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__855f1c6efd210a25d73922fc636cbbe8c4e08f7e49a5b9582f0ac1759eab5147)
                check_type(argname="argument mount_path", value=mount_path, expected_type=type_hints["mount_path"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument read_only", value=read_only, expected_type=type_hints["read_only"])
                check_type(argname="argument sub_path", value=sub_path, expected_type=type_hints["sub_path"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if mount_path is not None:
                self._values["mount_path"] = mount_path
            if name is not None:
                self._values["name"] = name
            if read_only is not None:
                self._values["read_only"] = read_only
            if sub_path is not None:
                self._values["sub_path"] = sub_path

        @builtins.property
        def mount_path(self) -> typing.Optional[builtins.str]:
            '''The path on the container where the volume is mounted.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainervolumemount.html#cfn-batch-jobdefinition-ekscontainervolumemount-mountpath
            '''
            result = self._values.get("mount_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name the volume mount.

            This must match the name of one of the volumes in the pod.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainervolumemount.html#cfn-batch-jobdefinition-ekscontainervolumemount-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def read_only(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''If this value is ``true`` , the container has read-only access to the volume.

            Otherwise, the container can write to the volume. The default value is ``false`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainervolumemount.html#cfn-batch-jobdefinition-ekscontainervolumemount-readonly
            '''
            result = self._values.get("read_only")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def sub_path(self) -> typing.Optional[builtins.str]:
            '''A sub-path inside the referenced volume instead of its root.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekscontainervolumemount.html#cfn-batch-jobdefinition-ekscontainervolumemount-subpath
            '''
            result = self._values.get("sub_path")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EksContainerVolumeMountProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EksPersistentVolumeClaimProperty",
        jsii_struct_bases=[],
        name_mapping={"claim_name": "claimName", "read_only": "readOnly"},
    )
    class EksPersistentVolumeClaimProperty:
        def __init__(
            self,
            *,
            claim_name: builtins.str,
            read_only: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''A ``persistentVolumeClaim`` volume is used to mount a `PersistentVolume <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/persistent-volumes/>`_ into a Pod. PersistentVolumeClaims are a way for users to "claim" durable storage without knowing the details of the particular cloud environment. See the information about `PersistentVolumes <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/persistent-volumes/>`_ in the *Kubernetes documentation* .

            :param claim_name: The name of the ``persistentVolumeClaim`` bounded to a ``persistentVolume`` . For more information, see `Persistent Volume Claims <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/persistent-volumes/#persistentvolumeclaims>`_ in the *Kubernetes documentation* .
            :param read_only: An optional boolean value indicating if the mount is read only. Default is false. For more information, see `Read Only Mounts <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/#read-only-mounts>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekspersistentvolumeclaim.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                eks_persistent_volume_claim_property = batch.CfnJobDefinition.EksPersistentVolumeClaimProperty(
                    claim_name="claimName",
                
                    # the properties below are optional
                    read_only=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__df2f02db37a79a03e0e657d7178a4c5a549ddd6252265363593910bdfac129d2)
                check_type(argname="argument claim_name", value=claim_name, expected_type=type_hints["claim_name"])
                check_type(argname="argument read_only", value=read_only, expected_type=type_hints["read_only"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "claim_name": claim_name,
            }
            if read_only is not None:
                self._values["read_only"] = read_only

        @builtins.property
        def claim_name(self) -> builtins.str:
            '''The name of the ``persistentVolumeClaim`` bounded to a ``persistentVolume`` .

            For more information, see `Persistent Volume Claims <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/persistent-volumes/#persistentvolumeclaims>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekspersistentvolumeclaim.html#cfn-batch-jobdefinition-ekspersistentvolumeclaim-claimname
            '''
            result = self._values.get("claim_name")
            assert result is not None, "Required property 'claim_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def read_only(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''An optional boolean value indicating if the mount is read only.

            Default is false. For more information, see `Read Only Mounts <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/#read-only-mounts>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekspersistentvolumeclaim.html#cfn-batch-jobdefinition-ekspersistentvolumeclaim-readonly
            '''
            result = self._values.get("read_only")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EksPersistentVolumeClaimProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EksPropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={"pod_properties": "podProperties"},
    )
    class EksPropertiesProperty:
        def __init__(
            self,
            *,
            pod_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.PodPropertiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that contains the properties for the Kubernetes resources of a job.

            :param pod_properties: The properties for the Kubernetes pod resources of a job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-eksproperties.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # labels: Any
                # limits: Any
                # requests: Any
                
                eks_properties_property = batch.CfnJobDefinition.EksPropertiesProperty(
                    pod_properties=batch.CfnJobDefinition.PodPropertiesProperty(
                        containers=[batch.CfnJobDefinition.EksContainerProperty(
                            image="image",
                
                            # the properties below are optional
                            args=["args"],
                            command=["command"],
                            env=[batch.CfnJobDefinition.EksContainerEnvironmentVariableProperty(
                                name="name",
                
                                # the properties below are optional
                                value="value"
                            )],
                            image_pull_policy="imagePullPolicy",
                            name="name",
                            resources=batch.CfnJobDefinition.ResourcesProperty(
                                limits=limits,
                                requests=requests
                            ),
                            security_context=batch.CfnJobDefinition.SecurityContextProperty(
                                allow_privilege_escalation=False,
                                privileged=False,
                                read_only_root_filesystem=False,
                                run_as_group=123,
                                run_as_non_root=False,
                                run_as_user=123
                            ),
                            volume_mounts=[batch.CfnJobDefinition.EksContainerVolumeMountProperty(
                                mount_path="mountPath",
                                name="name",
                                read_only=False,
                                sub_path="subPath"
                            )]
                        )],
                        dns_policy="dnsPolicy",
                        host_network=False,
                        image_pull_secrets=[batch.CfnJobDefinition.ImagePullSecretProperty(
                            name="name"
                        )],
                        init_containers=[batch.CfnJobDefinition.EksContainerProperty(
                            image="image",
                
                            # the properties below are optional
                            args=["args"],
                            command=["command"],
                            env=[batch.CfnJobDefinition.EksContainerEnvironmentVariableProperty(
                                name="name",
                
                                # the properties below are optional
                                value="value"
                            )],
                            image_pull_policy="imagePullPolicy",
                            name="name",
                            resources=batch.CfnJobDefinition.ResourcesProperty(
                                limits=limits,
                                requests=requests
                            ),
                            security_context=batch.CfnJobDefinition.SecurityContextProperty(
                                allow_privilege_escalation=False,
                                privileged=False,
                                read_only_root_filesystem=False,
                                run_as_group=123,
                                run_as_non_root=False,
                                run_as_user=123
                            ),
                            volume_mounts=[batch.CfnJobDefinition.EksContainerVolumeMountProperty(
                                mount_path="mountPath",
                                name="name",
                                read_only=False,
                                sub_path="subPath"
                            )]
                        )],
                        metadata=batch.CfnJobDefinition.MetadataProperty(
                            labels=labels
                        ),
                        service_account_name="serviceAccountName",
                        share_process_namespace=False,
                        volumes=[batch.CfnJobDefinition.EksVolumeProperty(
                            name="name",
                
                            # the properties below are optional
                            empty_dir=batch.CfnJobDefinition.EmptyDirProperty(
                                medium="medium",
                                size_limit="sizeLimit"
                            ),
                            host_path=batch.CfnJobDefinition.HostPathProperty(
                                path="path"
                            ),
                            persistent_volume_claim=batch.CfnJobDefinition.EksPersistentVolumeClaimProperty(
                                claim_name="claimName",
                
                                # the properties below are optional
                                read_only=False
                            ),
                            secret=batch.CfnJobDefinition.EksSecretProperty(
                                secret_name="secretName",
                
                                # the properties below are optional
                                optional=False
                            )
                        )]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ab4acc4a05a00897faf964911a9c5d642ebccbae43237575ad3b0b27ecb3ee74)
                check_type(argname="argument pod_properties", value=pod_properties, expected_type=type_hints["pod_properties"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if pod_properties is not None:
                self._values["pod_properties"] = pod_properties

        @builtins.property
        def pod_properties(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.PodPropertiesProperty"]]:
            '''The properties for the Kubernetes pod resources of a job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-eksproperties.html#cfn-batch-jobdefinition-eksproperties-podproperties
            '''
            result = self._values.get("pod_properties")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.PodPropertiesProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EksPropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EksSecretProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName", "optional": "optional"},
    )
    class EksSecretProperty:
        def __init__(
            self,
            *,
            secret_name: builtins.str,
            optional: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''Specifies the configuration of a Kubernetes ``secret`` volume.

            For more information, see `secret <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/#secret>`_ in the *Kubernetes documentation* .

            :param secret_name: The name of the secret. The name must be allowed as a DNS subdomain name. For more information, see `DNS subdomain names <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-subdomain-names>`_ in the *Kubernetes documentation* .
            :param optional: Specifies whether the secret or the secret's keys must be defined.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekssecret.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                eks_secret_property = batch.CfnJobDefinition.EksSecretProperty(
                    secret_name="secretName",
                
                    # the properties below are optional
                    optional=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__285ba8b233801a3c8377d3b178bafde5c49a98f09f3e37c53c663ac43e11a41c)
                check_type(argname="argument secret_name", value=secret_name, expected_type=type_hints["secret_name"])
                check_type(argname="argument optional", value=optional, expected_type=type_hints["optional"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "secret_name": secret_name,
            }
            if optional is not None:
                self._values["optional"] = optional

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''The name of the secret.

            The name must be allowed as a DNS subdomain name. For more information, see `DNS subdomain names <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-subdomain-names>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekssecret.html#cfn-batch-jobdefinition-ekssecret-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def optional(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether the secret or the secret's keys must be defined.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ekssecret.html#cfn-batch-jobdefinition-ekssecret-optional
            '''
            result = self._values.get("optional")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EksSecretProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EksVolumeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "empty_dir": "emptyDir",
            "host_path": "hostPath",
            "persistent_volume_claim": "persistentVolumeClaim",
            "secret": "secret",
        },
    )
    class EksVolumeProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            empty_dir: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EmptyDirProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            host_path: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.HostPathProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            persistent_volume_claim: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EksPersistentVolumeClaimProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            secret: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EksSecretProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Specifies an Amazon EKS volume for a job definition.

            :param name: The name of the volume. The name must be allowed as a DNS subdomain name. For more information, see `DNS subdomain names <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-subdomain-names>`_ in the *Kubernetes documentation* .
            :param empty_dir: Specifies the configuration of a Kubernetes ``emptyDir`` volume. For more information, see `emptyDir <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/#emptydir>`_ in the *Kubernetes documentation* .
            :param host_path: Specifies the configuration of a Kubernetes ``hostPath`` volume. For more information, see `hostPath <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/#hostpath>`_ in the *Kubernetes documentation* .
            :param persistent_volume_claim: Specifies the configuration of a Kubernetes ``persistentVolumeClaim`` bounded to a ``persistentVolume`` . For more information, see `Persistent Volume Claims <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/persistent-volumes/#persistentvolumeclaims>`_ in the *Kubernetes documentation* .
            :param secret: Specifies the configuration of a Kubernetes ``secret`` volume. For more information, see `secret <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/#secret>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-eksvolume.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                eks_volume_property = batch.CfnJobDefinition.EksVolumeProperty(
                    name="name",
                
                    # the properties below are optional
                    empty_dir=batch.CfnJobDefinition.EmptyDirProperty(
                        medium="medium",
                        size_limit="sizeLimit"
                    ),
                    host_path=batch.CfnJobDefinition.HostPathProperty(
                        path="path"
                    ),
                    persistent_volume_claim=batch.CfnJobDefinition.EksPersistentVolumeClaimProperty(
                        claim_name="claimName",
                
                        # the properties below are optional
                        read_only=False
                    ),
                    secret=batch.CfnJobDefinition.EksSecretProperty(
                        secret_name="secretName",
                
                        # the properties below are optional
                        optional=False
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__71bf085e500a34c552f178e2df09efdf138a2c99fb19667f0b4cf9e398ca47eb)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument empty_dir", value=empty_dir, expected_type=type_hints["empty_dir"])
                check_type(argname="argument host_path", value=host_path, expected_type=type_hints["host_path"])
                check_type(argname="argument persistent_volume_claim", value=persistent_volume_claim, expected_type=type_hints["persistent_volume_claim"])
                check_type(argname="argument secret", value=secret, expected_type=type_hints["secret"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if empty_dir is not None:
                self._values["empty_dir"] = empty_dir
            if host_path is not None:
                self._values["host_path"] = host_path
            if persistent_volume_claim is not None:
                self._values["persistent_volume_claim"] = persistent_volume_claim
            if secret is not None:
                self._values["secret"] = secret

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the volume.

            The name must be allowed as a DNS subdomain name. For more information, see `DNS subdomain names <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-subdomain-names>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-eksvolume.html#cfn-batch-jobdefinition-eksvolume-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def empty_dir(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EmptyDirProperty"]]:
            '''Specifies the configuration of a Kubernetes ``emptyDir`` volume.

            For more information, see `emptyDir <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/#emptydir>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-eksvolume.html#cfn-batch-jobdefinition-eksvolume-emptydir
            '''
            result = self._values.get("empty_dir")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EmptyDirProperty"]], result)

        @builtins.property
        def host_path(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.HostPathProperty"]]:
            '''Specifies the configuration of a Kubernetes ``hostPath`` volume.

            For more information, see `hostPath <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/#hostpath>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-eksvolume.html#cfn-batch-jobdefinition-eksvolume-hostpath
            '''
            result = self._values.get("host_path")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.HostPathProperty"]], result)

        @builtins.property
        def persistent_volume_claim(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksPersistentVolumeClaimProperty"]]:
            '''Specifies the configuration of a Kubernetes ``persistentVolumeClaim`` bounded to a ``persistentVolume`` .

            For more information, see `Persistent Volume Claims <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/persistent-volumes/#persistentvolumeclaims>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-eksvolume.html#cfn-batch-jobdefinition-eksvolume-persistentvolumeclaim
            '''
            result = self._values.get("persistent_volume_claim")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksPersistentVolumeClaimProperty"]], result)

        @builtins.property
        def secret(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksSecretProperty"]]:
            '''Specifies the configuration of a Kubernetes ``secret`` volume.

            For more information, see `secret <https://docs.aws.amazon.com/https://kubernetes.io/docs/concepts/storage/volumes/#secret>`_ in the *Kubernetes documentation* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-eksvolume.html#cfn-batch-jobdefinition-eksvolume-secret
            '''
            result = self._values.get("secret")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksSecretProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EksVolumeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EmptyDirProperty",
        jsii_struct_bases=[],
        name_mapping={"medium": "medium", "size_limit": "sizeLimit"},
    )
    class EmptyDirProperty:
        def __init__(
            self,
            *,
            medium: typing.Optional[builtins.str] = None,
            size_limit: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param medium: 
            :param size_limit: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-emptydir.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                empty_dir_property = batch.CfnJobDefinition.EmptyDirProperty(
                    medium="medium",
                    size_limit="sizeLimit"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bdef2f5e1ccd021a3913efc6e9f67cc3f3972961e97a491b8a739c8033fa20a3)
                check_type(argname="argument medium", value=medium, expected_type=type_hints["medium"])
                check_type(argname="argument size_limit", value=size_limit, expected_type=type_hints["size_limit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if medium is not None:
                self._values["medium"] = medium
            if size_limit is not None:
                self._values["size_limit"] = size_limit

        @builtins.property
        def medium(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-emptydir.html#cfn-batch-jobdefinition-emptydir-medium
            '''
            result = self._values.get("medium")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def size_limit(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-emptydir.html#cfn-batch-jobdefinition-emptydir-sizelimit
            '''
            result = self._values.get("size_limit")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EmptyDirProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EnvironmentProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class EnvironmentProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The Environment property type specifies environment variables to use in a job definition.

            :param name: The name of the environment variable.
            :param value: The value of the environment variable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-environment.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                environment_property = batch.CfnJobDefinition.EnvironmentProperty(
                    name="name",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1410eef4fad9b82401792bfb3c1c9342d946ef2f57199de602a1d63db8df8b41)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the environment variable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-environment.html#cfn-batch-jobdefinition-environment-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value of the environment variable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-environment.html#cfn-batch-jobdefinition-environment-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EnvironmentProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EphemeralStorageProperty",
        jsii_struct_bases=[],
        name_mapping={"size_in_gib": "sizeInGiB"},
    )
    class EphemeralStorageProperty:
        def __init__(self, *, size_in_gib: jsii.Number) -> None:
            '''The amount of ephemeral storage to allocate for the task.

            This parameter is used to expand the total amount of ephemeral storage available, beyond the default amount, for tasks hosted on AWS Fargate .

            :param size_in_gib: The total amount, in GiB, of ephemeral storage to set for the task. The minimum supported value is ``21`` GiB and the maximum supported value is ``200`` GiB.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ephemeralstorage.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                ephemeral_storage_property = batch.CfnJobDefinition.EphemeralStorageProperty(
                    size_in_gi_b=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c8e501c23ca082963ca4e3f697be654dd2053f8a6f062fcd801a10d61dd916cf)
                check_type(argname="argument size_in_gib", value=size_in_gib, expected_type=type_hints["size_in_gib"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "size_in_gib": size_in_gib,
            }

        @builtins.property
        def size_in_gib(self) -> jsii.Number:
            '''The total amount, in GiB, of ephemeral storage to set for the task.

            The minimum supported value is ``21`` GiB and the maximum supported value is ``200`` GiB.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ephemeralstorage.html#cfn-batch-jobdefinition-ephemeralstorage-sizeingib
            '''
            result = self._values.get("size_in_gib")
            assert result is not None, "Required property 'size_in_gib' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EphemeralStorageProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.EvaluateOnExitProperty",
        jsii_struct_bases=[],
        name_mapping={
            "action": "action",
            "on_exit_code": "onExitCode",
            "on_reason": "onReason",
            "on_status_reason": "onStatusReason",
        },
    )
    class EvaluateOnExitProperty:
        def __init__(
            self,
            *,
            action: builtins.str,
            on_exit_code: typing.Optional[builtins.str] = None,
            on_reason: typing.Optional[builtins.str] = None,
            on_status_reason: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Specifies an array of up to 5 conditions to be met, and an action to take ( ``RETRY`` or ``EXIT`` ) if all conditions are met.

            If none of the ``EvaluateOnExit`` conditions in a ``RetryStrategy`` match, then the job is retried.

            :param action: Specifies the action to take if all of the specified conditions ( ``onStatusReason`` , ``onReason`` , and ``onExitCode`` ) are met. The values aren't case sensitive.
            :param on_exit_code: Contains a glob pattern to match against the decimal representation of the ``ExitCode`` returned for a job. The pattern can be up to 512 characters long. It can contain only numbers, and can end with an asterisk (*) so that only the start of the string needs to be an exact match. The string can contain up to 512 characters.
            :param on_reason: Contains a glob pattern to match against the ``Reason`` returned for a job. The pattern can contain up to 512 characters. It can contain letters, numbers, periods (.), colons (:), and white space (including spaces and tabs). It can optionally end with an asterisk (*) so that only the start of the string needs to be an exact match.
            :param on_status_reason: Contains a glob pattern to match against the ``StatusReason`` returned for a job. The pattern can contain up to 512 characters. It can contain letters, numbers, periods (.), colons (:), and white spaces (including spaces or tabs). It can optionally end with an asterisk (*) so that only the start of the string needs to be an exact match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-evaluateonexit.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                evaluate_on_exit_property = batch.CfnJobDefinition.EvaluateOnExitProperty(
                    action="action",
                
                    # the properties below are optional
                    on_exit_code="onExitCode",
                    on_reason="onReason",
                    on_status_reason="onStatusReason"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__662f88de3d4a2c44e67c0a3c3ee43319e3b3ee88c557f7669e584f52665e667e)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument on_exit_code", value=on_exit_code, expected_type=type_hints["on_exit_code"])
                check_type(argname="argument on_reason", value=on_reason, expected_type=type_hints["on_reason"])
                check_type(argname="argument on_status_reason", value=on_status_reason, expected_type=type_hints["on_status_reason"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
            }
            if on_exit_code is not None:
                self._values["on_exit_code"] = on_exit_code
            if on_reason is not None:
                self._values["on_reason"] = on_reason
            if on_status_reason is not None:
                self._values["on_status_reason"] = on_status_reason

        @builtins.property
        def action(self) -> builtins.str:
            '''Specifies the action to take if all of the specified conditions ( ``onStatusReason`` , ``onReason`` , and ``onExitCode`` ) are met.

            The values aren't case sensitive.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-evaluateonexit.html#cfn-batch-jobdefinition-evaluateonexit-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def on_exit_code(self) -> typing.Optional[builtins.str]:
            '''Contains a glob pattern to match against the decimal representation of the ``ExitCode`` returned for a job.

            The pattern can be up to 512 characters long. It can contain only numbers, and can end with an asterisk (*) so that only the start of the string needs to be an exact match.

            The string can contain up to 512 characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-evaluateonexit.html#cfn-batch-jobdefinition-evaluateonexit-onexitcode
            '''
            result = self._values.get("on_exit_code")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def on_reason(self) -> typing.Optional[builtins.str]:
            '''Contains a glob pattern to match against the ``Reason`` returned for a job.

            The pattern can contain up to 512 characters. It can contain letters, numbers, periods (.), colons (:), and white space (including spaces and tabs). It can optionally end with an asterisk (*) so that only the start of the string needs to be an exact match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-evaluateonexit.html#cfn-batch-jobdefinition-evaluateonexit-onreason
            '''
            result = self._values.get("on_reason")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def on_status_reason(self) -> typing.Optional[builtins.str]:
            '''Contains a glob pattern to match against the ``StatusReason`` returned for a job.

            The pattern can contain up to 512 characters. It can contain letters, numbers, periods (.), colons (:), and white spaces (including spaces or tabs). It can optionally end with an asterisk (*) so that only the start of the string needs to be an exact match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-evaluateonexit.html#cfn-batch-jobdefinition-evaluateonexit-onstatusreason
            '''
            result = self._values.get("on_status_reason")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EvaluateOnExitProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.FargatePlatformConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"platform_version": "platformVersion"},
    )
    class FargatePlatformConfigurationProperty:
        def __init__(
            self,
            *,
            platform_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The platform configuration for jobs that are running on Fargate resources.

            Jobs that run on Amazon EC2 resources must not specify this parameter.

            :param platform_version: The AWS Fargate platform version where the jobs are running. A platform version is specified only for jobs that are running on Fargate resources. If one isn't specified, the ``LATEST`` platform version is used by default. This uses a recent, approved version of the AWS Fargate platform for compute resources. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-fargateplatformconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                fargate_platform_configuration_property = batch.CfnJobDefinition.FargatePlatformConfigurationProperty(
                    platform_version="platformVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2b088f865053d4cfdc2700de70d732a768d630a892a20da6fda19303df04e9b4)
                check_type(argname="argument platform_version", value=platform_version, expected_type=type_hints["platform_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if platform_version is not None:
                self._values["platform_version"] = platform_version

        @builtins.property
        def platform_version(self) -> typing.Optional[builtins.str]:
            '''The AWS Fargate platform version where the jobs are running.

            A platform version is specified only for jobs that are running on Fargate resources. If one isn't specified, the ``LATEST`` platform version is used by default. This uses a recent, approved version of the AWS Fargate platform for compute resources. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-fargateplatformconfiguration.html#cfn-batch-jobdefinition-fargateplatformconfiguration-platformversion
            '''
            result = self._values.get("platform_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FargatePlatformConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.HostPathProperty",
        jsii_struct_bases=[],
        name_mapping={"path": "path"},
    )
    class HostPathProperty:
        def __init__(self, *, path: typing.Optional[builtins.str] = None) -> None:
            '''
            :param path: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-hostpath.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                host_path_property = batch.CfnJobDefinition.HostPathProperty(
                    path="path"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2bf55e035d36507c507c0ff620eda0a3096496a2cd5226656fc5e4af46e1b9ac)
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if path is not None:
                self._values["path"] = path

        @builtins.property
        def path(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-hostpath.html#cfn-batch-jobdefinition-hostpath-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HostPathProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.ImagePullSecretProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name"},
    )
    class ImagePullSecretProperty:
        def __init__(self, *, name: typing.Optional[builtins.str] = None) -> None:
            '''References a Kubernetes secret resource.

            This name of the secret must start and end with an alphanumeric character, is required to be lowercase, can include periods (.) and hyphens (-), and can't contain more than 253 characters.

            :param name: Provides a unique identifier for the ``ImagePullSecret`` . This object is required when ``EksPodProperties$imagePullSecrets`` is used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-imagepullsecret.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                image_pull_secret_property = batch.CfnJobDefinition.ImagePullSecretProperty(
                    name="name"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__510e7e45fa5176303c62970a1bb0161e95eb0fcd2438671b35f8c5a3c506b95a)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''Provides a unique identifier for the ``ImagePullSecret`` .

            This object is required when ``EksPodProperties$imagePullSecrets`` is used.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-imagepullsecret.html#cfn-batch-jobdefinition-imagepullsecret-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ImagePullSecretProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.JobTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"attempt_duration_seconds": "attemptDurationSeconds"},
    )
    class JobTimeoutProperty:
        def __init__(
            self,
            *,
            attempt_duration_seconds: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a job timeout configuration.

            :param attempt_duration_seconds: The job timeout time (in seconds) that's measured from the job attempt's ``startedAt`` timestamp. After this time passes, AWS Batch terminates your jobs if they aren't finished. The minimum value for the timeout is 60 seconds. For array jobs, the timeout applies to the child jobs, not to the parent array job. For multi-node parallel (MNP) jobs, the timeout applies to the whole job, not to the individual nodes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-jobtimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                job_timeout_property = batch.CfnJobDefinition.JobTimeoutProperty(
                    attempt_duration_seconds=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a3f2b0ca2e9dc87bf15f08f72a000b1f1878e671abdfdb32aae1a6f479b101bf)
                check_type(argname="argument attempt_duration_seconds", value=attempt_duration_seconds, expected_type=type_hints["attempt_duration_seconds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if attempt_duration_seconds is not None:
                self._values["attempt_duration_seconds"] = attempt_duration_seconds

        @builtins.property
        def attempt_duration_seconds(self) -> typing.Optional[jsii.Number]:
            '''The job timeout time (in seconds) that's measured from the job attempt's ``startedAt`` timestamp.

            After this time passes, AWS Batch terminates your jobs if they aren't finished. The minimum value for the timeout is 60 seconds.

            For array jobs, the timeout applies to the child jobs, not to the parent array job.

            For multi-node parallel (MNP) jobs, the timeout applies to the whole job, not to the individual nodes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-jobtimeout.html#cfn-batch-jobdefinition-jobtimeout-attemptdurationseconds
            '''
            result = self._values.get("attempt_duration_seconds")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "JobTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.LinuxParametersProperty",
        jsii_struct_bases=[],
        name_mapping={
            "devices": "devices",
            "init_process_enabled": "initProcessEnabled",
            "max_swap": "maxSwap",
            "shared_memory_size": "sharedMemorySize",
            "swappiness": "swappiness",
            "tmpfs": "tmpfs",
        },
    )
    class LinuxParametersProperty:
        def __init__(
            self,
            *,
            devices: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.DeviceProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            init_process_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            max_swap: typing.Optional[jsii.Number] = None,
            shared_memory_size: typing.Optional[jsii.Number] = None,
            swappiness: typing.Optional[jsii.Number] = None,
            tmpfs: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.TmpfsProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''Linux-specific modifications that are applied to the container, such as details for device mappings.

            :param devices: Any of the host devices to expose to the container. This parameter maps to ``Devices`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--device`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't provide it for these jobs.
            :param init_process_enabled: If true, run an ``init`` process inside the container that forwards signals and reaps processes. This parameter maps to the ``--init`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . This parameter requires version 1.25 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version | grep "Server API version"``
            :param max_swap: The total amount of swap memory (in MiB) a container can use. This parameter is translated to the ``--memory-swap`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ where the value is the sum of the container memory plus the ``maxSwap`` value. For more information, see ```--memory-swap`` details <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/resource_constraints/#--memory-swap-details>`_ in the Docker documentation. If a ``maxSwap`` value of ``0`` is specified, the container doesn't use swap. Accepted values are ``0`` or any positive integer. If the ``maxSwap`` parameter is omitted, the container doesn't use the swap configuration for the container instance on which it runs. A ``maxSwap`` value must be set for the ``swappiness`` parameter to be used. .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't provide it for these jobs.
            :param shared_memory_size: The value for the size (in MiB) of the ``/dev/shm`` volume. This parameter maps to the ``--shm-size`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't provide it for these jobs.
            :param swappiness: You can use this parameter to tune a container's memory swappiness behavior. A ``swappiness`` value of ``0`` causes swapping to not occur unless absolutely necessary. A ``swappiness`` value of ``100`` causes pages to be swapped aggressively. Valid values are whole numbers between ``0`` and ``100`` . If the ``swappiness`` parameter isn't specified, a default value of ``60`` is used. If a value isn't specified for ``maxSwap`` , then this parameter is ignored. If ``maxSwap`` is set to 0, the container doesn't use swap. This parameter maps to the ``--memory-swappiness`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . Consider the following when you use a per-container swap configuration. - Swap space must be enabled and allocated on the container instance for the containers to use. .. epigraph:: By default, the Amazon ECS optimized AMIs don't have swap enabled. You must enable swap on the instance to use this feature. For more information, see `Instance store swap volumes <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-store-swap-volumes.html>`_ in the *Amazon EC2 User Guide for Linux Instances* or `How do I allocate memory to work as swap space in an Amazon EC2 instance by using a swap file? <https://docs.aws.amazon.com/premiumsupport/knowledge-center/ec2-memory-swap-file/>`_ - The swap space parameters are only supported for job definitions using EC2 resources. - If the ``maxSwap`` and ``swappiness`` parameters are omitted from a job definition, each container has a default ``swappiness`` value of 60. Moreover, the total swap usage is limited to two times the memory reservation of the container. .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't provide it for these jobs.
            :param tmpfs: The container path, mount options, and size (in MiB) of the ``tmpfs`` mount. This parameter maps to the ``--tmpfs`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. Don't provide this parameter for this resource type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-linuxparameters.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                linux_parameters_property = batch.CfnJobDefinition.LinuxParametersProperty(
                    devices=[batch.CfnJobDefinition.DeviceProperty(
                        container_path="containerPath",
                        host_path="hostPath",
                        permissions=["permissions"]
                    )],
                    init_process_enabled=False,
                    max_swap=123,
                    shared_memory_size=123,
                    swappiness=123,
                    tmpfs=[batch.CfnJobDefinition.TmpfsProperty(
                        container_path="containerPath",
                        size=123,
                
                        # the properties below are optional
                        mount_options=["mountOptions"]
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__46c01503a14b135de04e03e8a183177fbaa4f728ed5853b4de848d62c1f248ae)
                check_type(argname="argument devices", value=devices, expected_type=type_hints["devices"])
                check_type(argname="argument init_process_enabled", value=init_process_enabled, expected_type=type_hints["init_process_enabled"])
                check_type(argname="argument max_swap", value=max_swap, expected_type=type_hints["max_swap"])
                check_type(argname="argument shared_memory_size", value=shared_memory_size, expected_type=type_hints["shared_memory_size"])
                check_type(argname="argument swappiness", value=swappiness, expected_type=type_hints["swappiness"])
                check_type(argname="argument tmpfs", value=tmpfs, expected_type=type_hints["tmpfs"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if devices is not None:
                self._values["devices"] = devices
            if init_process_enabled is not None:
                self._values["init_process_enabled"] = init_process_enabled
            if max_swap is not None:
                self._values["max_swap"] = max_swap
            if shared_memory_size is not None:
                self._values["shared_memory_size"] = shared_memory_size
            if swappiness is not None:
                self._values["swappiness"] = swappiness
            if tmpfs is not None:
                self._values["tmpfs"] = tmpfs

        @builtins.property
        def devices(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.DeviceProperty"]]]]:
            '''Any of the host devices to expose to the container.

            This parameter maps to ``Devices`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--device`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't provide it for these jobs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-linuxparameters.html#cfn-batch-jobdefinition-linuxparameters-devices
            '''
            result = self._values.get("devices")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.DeviceProperty"]]]], result)

        @builtins.property
        def init_process_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''If true, run an ``init`` process inside the container that forwards signals and reaps processes.

            This parameter maps to the ``--init`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . This parameter requires version 1.25 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version | grep "Server API version"``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-linuxparameters.html#cfn-batch-jobdefinition-linuxparameters-initprocessenabled
            '''
            result = self._values.get("init_process_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def max_swap(self) -> typing.Optional[jsii.Number]:
            '''The total amount of swap memory (in MiB) a container can use.

            This parameter is translated to the ``--memory-swap`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ where the value is the sum of the container memory plus the ``maxSwap`` value. For more information, see ```--memory-swap`` details <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/resource_constraints/#--memory-swap-details>`_ in the Docker documentation.

            If a ``maxSwap`` value of ``0`` is specified, the container doesn't use swap. Accepted values are ``0`` or any positive integer. If the ``maxSwap`` parameter is omitted, the container doesn't use the swap configuration for the container instance on which it runs. A ``maxSwap`` value must be set for the ``swappiness`` parameter to be used.
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't provide it for these jobs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-linuxparameters.html#cfn-batch-jobdefinition-linuxparameters-maxswap
            '''
            result = self._values.get("max_swap")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def shared_memory_size(self) -> typing.Optional[jsii.Number]:
            '''The value for the size (in MiB) of the ``/dev/shm`` volume.

            This parameter maps to the ``--shm-size`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't provide it for these jobs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-linuxparameters.html#cfn-batch-jobdefinition-linuxparameters-sharedmemorysize
            '''
            result = self._values.get("shared_memory_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def swappiness(self) -> typing.Optional[jsii.Number]:
            '''You can use this parameter to tune a container's memory swappiness behavior.

            A ``swappiness`` value of ``0`` causes swapping to not occur unless absolutely necessary. A ``swappiness`` value of ``100`` causes pages to be swapped aggressively. Valid values are whole numbers between ``0`` and ``100`` . If the ``swappiness`` parameter isn't specified, a default value of ``60`` is used. If a value isn't specified for ``maxSwap`` , then this parameter is ignored. If ``maxSwap`` is set to 0, the container doesn't use swap. This parameter maps to the ``--memory-swappiness`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .

            Consider the following when you use a per-container swap configuration.

            - Swap space must be enabled and allocated on the container instance for the containers to use.

            .. epigraph::

               By default, the Amazon ECS optimized AMIs don't have swap enabled. You must enable swap on the instance to use this feature. For more information, see `Instance store swap volumes <https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-store-swap-volumes.html>`_ in the *Amazon EC2 User Guide for Linux Instances* or `How do I allocate memory to work as swap space in an Amazon EC2 instance by using a swap file? <https://docs.aws.amazon.com/premiumsupport/knowledge-center/ec2-memory-swap-file/>`_

            - The swap space parameters are only supported for job definitions using EC2 resources.
            - If the ``maxSwap`` and ``swappiness`` parameters are omitted from a job definition, each container has a default ``swappiness`` value of 60. Moreover, the total swap usage is limited to two times the memory reservation of the container.

            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't provide it for these jobs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-linuxparameters.html#cfn-batch-jobdefinition-linuxparameters-swappiness
            '''
            result = self._values.get("swappiness")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def tmpfs(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.TmpfsProperty"]]]]:
            '''The container path, mount options, and size (in MiB) of the ``tmpfs`` mount.

            This parameter maps to the ``--tmpfs`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .
            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources. Don't provide this parameter for this resource type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-linuxparameters.html#cfn-batch-jobdefinition-linuxparameters-tmpfs
            '''
            result = self._values.get("tmpfs")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.TmpfsProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LinuxParametersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.LogConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "log_driver": "logDriver",
            "options": "options",
            "secret_options": "secretOptions",
        },
    )
    class LogConfigurationProperty:
        def __init__(
            self,
            *,
            log_driver: builtins.str,
            options: typing.Any = None,
            secret_options: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.SecretProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''Log configuration options to send to a custom log driver for the container.

            :param log_driver: The log driver to use for the container. The valid values that are listed for this parameter are log drivers that the Amazon ECS container agent can communicate with by default. The supported log drivers are ``awslogs`` , ``fluentd`` , ``gelf`` , ``json-file`` , ``journald`` , ``logentries`` , ``syslog`` , and ``splunk`` . .. epigraph:: Jobs that are running on Fargate resources are restricted to the ``awslogs`` and ``splunk`` log drivers. - **awslogs** - Specifies the Amazon CloudWatch Logs logging driver. For more information, see `Using the awslogs log driver <https://docs.aws.amazon.com/batch/latest/userguide/using_awslogs.html>`_ in the *AWS Batch User Guide* and `Amazon CloudWatch Logs logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/awslogs/>`_ in the Docker documentation. - **fluentd** - Specifies the Fluentd logging driver. For more information including usage and options, see `Fluentd logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/fluentd/>`_ in the *Docker documentation* . - **gelf** - Specifies the Graylog Extended Format (GELF) logging driver. For more information including usage and options, see `Graylog Extended Format logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/gelf/>`_ in the *Docker documentation* . - **journald** - Specifies the journald logging driver. For more information including usage and options, see `Journald logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/journald/>`_ in the *Docker documentation* . - **json-file** - Specifies the JSON file logging driver. For more information including usage and options, see `JSON File logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/json-file/>`_ in the *Docker documentation* . - **splunk** - Specifies the Splunk logging driver. For more information including usage and options, see `Splunk logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/splunk/>`_ in the *Docker documentation* . - **syslog** - Specifies the syslog logging driver. For more information including usage and options, see `Syslog logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/syslog/>`_ in the *Docker documentation* . .. epigraph:: If you have a custom driver that's not listed earlier that you want to work with the Amazon ECS container agent, you can fork the Amazon ECS container agent project that's `available on GitHub <https://docs.aws.amazon.com/https://github.com/aws/amazon-ecs-agent>`_ and customize it to work with that driver. We encourage you to submit pull requests for changes that you want to have included. However, Amazon Web Services doesn't currently support running modified copies of this software. This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version | grep "Server API version"``
            :param options: The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version | grep "Server API version"``
            :param secret_options: The secrets to pass to the log configuration. For more information, see `Specifying sensitive data <https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html>`_ in the *AWS Batch User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-logconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # options: Any
                
                log_configuration_property = batch.CfnJobDefinition.LogConfigurationProperty(
                    log_driver="logDriver",
                
                    # the properties below are optional
                    options=options,
                    secret_options=[batch.CfnJobDefinition.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__76e3055acf404e0b579b27c323b043bfb9f33e586341d48d4c5a17760f100e72)
                check_type(argname="argument log_driver", value=log_driver, expected_type=type_hints["log_driver"])
                check_type(argname="argument options", value=options, expected_type=type_hints["options"])
                check_type(argname="argument secret_options", value=secret_options, expected_type=type_hints["secret_options"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "log_driver": log_driver,
            }
            if options is not None:
                self._values["options"] = options
            if secret_options is not None:
                self._values["secret_options"] = secret_options

        @builtins.property
        def log_driver(self) -> builtins.str:
            '''The log driver to use for the container.

            The valid values that are listed for this parameter are log drivers that the Amazon ECS container agent can communicate with by default.

            The supported log drivers are ``awslogs`` , ``fluentd`` , ``gelf`` , ``json-file`` , ``journald`` , ``logentries`` , ``syslog`` , and ``splunk`` .
            .. epigraph::

               Jobs that are running on Fargate resources are restricted to the ``awslogs`` and ``splunk`` log drivers.

            - **awslogs** - Specifies the Amazon CloudWatch Logs logging driver. For more information, see `Using the awslogs log driver <https://docs.aws.amazon.com/batch/latest/userguide/using_awslogs.html>`_ in the *AWS Batch User Guide* and `Amazon CloudWatch Logs logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/awslogs/>`_ in the Docker documentation.
            - **fluentd** - Specifies the Fluentd logging driver. For more information including usage and options, see `Fluentd logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/fluentd/>`_ in the *Docker documentation* .
            - **gelf** - Specifies the Graylog Extended Format (GELF) logging driver. For more information including usage and options, see `Graylog Extended Format logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/gelf/>`_ in the *Docker documentation* .
            - **journald** - Specifies the journald logging driver. For more information including usage and options, see `Journald logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/journald/>`_ in the *Docker documentation* .
            - **json-file** - Specifies the JSON file logging driver. For more information including usage and options, see `JSON File logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/json-file/>`_ in the *Docker documentation* .
            - **splunk** - Specifies the Splunk logging driver. For more information including usage and options, see `Splunk logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/splunk/>`_ in the *Docker documentation* .
            - **syslog** - Specifies the syslog logging driver. For more information including usage and options, see `Syslog logging driver <https://docs.aws.amazon.com/https://docs.docker.com/config/containers/logging/syslog/>`_ in the *Docker documentation* .

            .. epigraph::

               If you have a custom driver that's not listed earlier that you want to work with the Amazon ECS container agent, you can fork the Amazon ECS container agent project that's `available on GitHub <https://docs.aws.amazon.com/https://github.com/aws/amazon-ecs-agent>`_ and customize it to work with that driver. We encourage you to submit pull requests for changes that you want to have included. However, Amazon Web Services doesn't currently support running modified copies of this software.

            This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version | grep "Server API version"``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-logconfiguration.html#cfn-batch-jobdefinition-logconfiguration-logdriver
            '''
            result = self._values.get("log_driver")
            assert result is not None, "Required property 'log_driver' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def options(self) -> typing.Any:
            '''The configuration options to send to the log driver.

            This parameter requires version 1.19 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version | grep "Server API version"``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-logconfiguration.html#cfn-batch-jobdefinition-logconfiguration-options
            '''
            result = self._values.get("options")
            return typing.cast(typing.Any, result)

        @builtins.property
        def secret_options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.SecretProperty"]]]]:
            '''The secrets to pass to the log configuration.

            For more information, see `Specifying sensitive data <https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html>`_ in the *AWS Batch User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-logconfiguration.html#cfn-batch-jobdefinition-logconfiguration-secretoptions
            '''
            result = self._values.get("secret_options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.SecretProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LogConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.MetadataProperty",
        jsii_struct_bases=[],
        name_mapping={"labels": "labels"},
    )
    class MetadataProperty:
        def __init__(self, *, labels: typing.Any = None) -> None:
            '''
            :param labels: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-metadata.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # labels: Any
                
                metadata_property = batch.CfnJobDefinition.MetadataProperty(
                    labels=labels
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2a991ab55d6c85b732418b9a39402702cdf7043f946aaf69c13b36a4ccdd84cc)
                check_type(argname="argument labels", value=labels, expected_type=type_hints["labels"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if labels is not None:
                self._values["labels"] = labels

        @builtins.property
        def labels(self) -> typing.Any:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-metadata.html#cfn-batch-jobdefinition-metadata-labels
            '''
            result = self._values.get("labels")
            return typing.cast(typing.Any, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MetadataProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.MountPointProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_path": "containerPath",
            "read_only": "readOnly",
            "source_volume": "sourceVolume",
        },
    )
    class MountPointProperty:
        def __init__(
            self,
            *,
            container_path: typing.Optional[builtins.str] = None,
            read_only: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            source_volume: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Details for a Docker volume mount point that's used in a job's container properties.

            This parameter maps to ``Volumes`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.43/#tag/Container/operation/ContainerCreate>`_ section of the *Docker Remote API* and the ``--volume`` option to docker run.

            :param container_path: The path on the container where the host volume is mounted.
            :param read_only: If this value is ``true`` , the container has read-only access to the volume. Otherwise, the container can write to the volume. The default value is ``false`` .
            :param source_volume: The name of the volume to mount.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-mountpoint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                mount_point_property = batch.CfnJobDefinition.MountPointProperty(
                    container_path="containerPath",
                    read_only=False,
                    source_volume="sourceVolume"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0890bfa00c320f24aeaf31204369e159ba55507af9af6503cd040b7b5051df11)
                check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
                check_type(argname="argument read_only", value=read_only, expected_type=type_hints["read_only"])
                check_type(argname="argument source_volume", value=source_volume, expected_type=type_hints["source_volume"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if container_path is not None:
                self._values["container_path"] = container_path
            if read_only is not None:
                self._values["read_only"] = read_only
            if source_volume is not None:
                self._values["source_volume"] = source_volume

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''The path on the container where the host volume is mounted.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-mountpoint.html#cfn-batch-jobdefinition-mountpoint-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def read_only(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''If this value is ``true`` , the container has read-only access to the volume.

            Otherwise, the container can write to the volume. The default value is ``false`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-mountpoint.html#cfn-batch-jobdefinition-mountpoint-readonly
            '''
            result = self._values.get("read_only")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def source_volume(self) -> typing.Optional[builtins.str]:
            '''The name of the volume to mount.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-mountpoint.html#cfn-batch-jobdefinition-mountpoint-sourcevolume
            '''
            result = self._values.get("source_volume")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MountPointProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.MountPointsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_path": "containerPath",
            "read_only": "readOnly",
            "source_volume": "sourceVolume",
        },
    )
    class MountPointsProperty:
        def __init__(
            self,
            *,
            container_path: typing.Optional[builtins.str] = None,
            read_only: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            source_volume: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param container_path: 
            :param read_only: 
            :param source_volume: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-mountpoints.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                mount_points_property = batch.CfnJobDefinition.MountPointsProperty(
                    container_path="containerPath",
                    read_only=False,
                    source_volume="sourceVolume"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__47aa253c2fbb1ab8f90425d412e691e807d1454f21b10bb59eee6c2b63cf29b5)
                check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
                check_type(argname="argument read_only", value=read_only, expected_type=type_hints["read_only"])
                check_type(argname="argument source_volume", value=source_volume, expected_type=type_hints["source_volume"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if container_path is not None:
                self._values["container_path"] = container_path
            if read_only is not None:
                self._values["read_only"] = read_only
            if source_volume is not None:
                self._values["source_volume"] = source_volume

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-mountpoints.html#cfn-batch-jobdefinition-mountpoints-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def read_only(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-mountpoints.html#cfn-batch-jobdefinition-mountpoints-readonly
            '''
            result = self._values.get("read_only")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def source_volume(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-mountpoints.html#cfn-batch-jobdefinition-mountpoints-sourcevolume
            '''
            result = self._values.get("source_volume")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MountPointsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.MultiNodeEcsPropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={"task_properties": "taskProperties"},
    )
    class MultiNodeEcsPropertiesProperty:
        def __init__(
            self,
            *,
            task_properties: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.MultiNodeEcsTaskPropertiesProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''An object that contains the properties for the Amazon ECS resources of a job.

            :param task_properties: An object that contains the properties for the Amazon ECS task definition of a job. .. epigraph:: This object is currently limited to one task element. However, the task element can run up to 10 containers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-multinodeecsproperties.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # options: Any
                
                multi_node_ecs_properties_property = batch.CfnJobDefinition.MultiNodeEcsPropertiesProperty(
                    task_properties=[batch.CfnJobDefinition.MultiNodeEcsTaskPropertiesProperty(
                        containers=[batch.CfnJobDefinition.TaskContainerPropertiesProperty(
                            image="image",
                
                            # the properties below are optional
                            command=["command"],
                            depends_on=[batch.CfnJobDefinition.TaskContainerDependencyProperty(
                                condition="condition",
                                container_name="containerName"
                            )],
                            environment=[batch.CfnJobDefinition.EnvironmentProperty(
                                name="name",
                                value="value"
                            )],
                            essential=False,
                            linux_parameters=batch.CfnJobDefinition.LinuxParametersProperty(
                                devices=[batch.CfnJobDefinition.DeviceProperty(
                                    container_path="containerPath",
                                    host_path="hostPath",
                                    permissions=["permissions"]
                                )],
                                init_process_enabled=False,
                                max_swap=123,
                                shared_memory_size=123,
                                swappiness=123,
                                tmpfs=[batch.CfnJobDefinition.TmpfsProperty(
                                    container_path="containerPath",
                                    size=123,
                
                                    # the properties below are optional
                                    mount_options=["mountOptions"]
                                )]
                            ),
                            log_configuration=batch.CfnJobDefinition.LogConfigurationProperty(
                                log_driver="logDriver",
                
                                # the properties below are optional
                                options=options,
                                secret_options=[batch.CfnJobDefinition.SecretProperty(
                                    name="name",
                                    value_from="valueFrom"
                                )]
                            ),
                            mount_points=[batch.CfnJobDefinition.MountPointProperty(
                                container_path="containerPath",
                                read_only=False,
                                source_volume="sourceVolume"
                            )],
                            name="name",
                            privileged=False,
                            readonly_root_filesystem=False,
                            repository_credentials=batch.CfnJobDefinition.RepositoryCredentialsProperty(
                                credentials_parameter="credentialsParameter"
                            ),
                            resource_requirements=[batch.CfnJobDefinition.ResourceRequirementProperty(
                                type="type",
                                value="value"
                            )],
                            secrets=[batch.CfnJobDefinition.SecretProperty(
                                name="name",
                                value_from="valueFrom"
                            )],
                            ulimits=[batch.CfnJobDefinition.UlimitProperty(
                                hard_limit=123,
                                name="name",
                                soft_limit=123
                            )],
                            user="user"
                        )],
                        execution_role_arn="executionRoleArn",
                        ipc_mode="ipcMode",
                        pid_mode="pidMode",
                        task_role_arn="taskRoleArn",
                        volumes=[batch.CfnJobDefinition.VolumesProperty(
                            efs_volume_configuration=batch.CfnJobDefinition.EfsVolumeConfigurationProperty(
                                file_system_id="fileSystemId",
                
                                # the properties below are optional
                                authorization_config=batch.CfnJobDefinition.AuthorizationConfigProperty(
                                    access_point_id="accessPointId",
                                    iam="iam"
                                ),
                                root_directory="rootDirectory",
                                transit_encryption="transitEncryption",
                                transit_encryption_port=123
                            ),
                            host=batch.CfnJobDefinition.VolumesHostProperty(
                                source_path="sourcePath"
                            ),
                            name="name"
                        )]
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9a2309f896452e04abc96277b3c04ead84cbae39e501242083e44da7eb775567)
                check_type(argname="argument task_properties", value=task_properties, expected_type=type_hints["task_properties"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "task_properties": task_properties,
            }

        @builtins.property
        def task_properties(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.MultiNodeEcsTaskPropertiesProperty"]]]:
            '''An object that contains the properties for the Amazon ECS task definition of a job.

            .. epigraph::

               This object is currently limited to one task element. However, the task element can run up to 10 containers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-multinodeecsproperties.html#cfn-batch-jobdefinition-multinodeecsproperties-taskproperties
            '''
            result = self._values.get("task_properties")
            assert result is not None, "Required property 'task_properties' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.MultiNodeEcsTaskPropertiesProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MultiNodeEcsPropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.MultiNodeEcsTaskPropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "containers": "containers",
            "execution_role_arn": "executionRoleArn",
            "ipc_mode": "ipcMode",
            "pid_mode": "pidMode",
            "task_role_arn": "taskRoleArn",
            "volumes": "volumes",
        },
    )
    class MultiNodeEcsTaskPropertiesProperty:
        def __init__(
            self,
            *,
            containers: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.TaskContainerPropertiesProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            execution_role_arn: typing.Optional[builtins.str] = None,
            ipc_mode: typing.Optional[builtins.str] = None,
            pid_mode: typing.Optional[builtins.str] = None,
            task_role_arn: typing.Optional[builtins.str] = None,
            volumes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.VolumesProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The properties for a task definition that describes the container and volume definitions of an Amazon ECS task.

            You can specify which Docker images to use, the required resources, and other configurations related to launching the task definition through an Amazon ECS service or task.

            :param containers: This object is a list of containers.
            :param execution_role_arn: The Amazon Resource Name (ARN) of the execution role that AWS Batch can assume. For jobs that run on Fargate resources, you must provide an execution role. For more information, see `AWS Batch execution IAM role <https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html>`_ in the *AWS Batch User Guide* .
            :param ipc_mode: The IPC resource namespace to use for the containers in the task. The valid values are ``host`` , ``task`` , or ``none`` . If ``host`` is specified, all containers within the tasks that specified the ``host`` IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified ``task`` share the same IPC resources. If ``none`` is specified, the IPC resources within the containers of a task are private, and are not shared with other containers in a task or on the container instance. If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance. For more information, see `IPC settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#ipc-settings---ipc>`_ in the Docker run reference.
            :param pid_mode: The process namespace to use for the containers in the task. The valid values are ``host`` or ``task`` . For example, monitoring sidecars might need ``pidMode`` to access information about other containers running in the same task. If ``host`` is specified, all containers within the tasks that specified the ``host`` PID mode on the same container instance share the process namespace with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same process namespace. If no value is specified, the default is a private namespace for each container. For more information, see `PID settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#pid-settings---pid>`_ in the Docker run reference.
            :param task_role_arn: The Amazon Resource Name (ARN) that's associated with the Amazon ECS task. .. epigraph:: This is object is comparable to `ContainerProperties:jobRoleArn <https://docs.aws.amazon.com/batch/latest/APIReference/API_ContainerProperties.html>`_ .
            :param volumes: A list of volumes that are associated with the job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-multinodeecstaskproperties.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # options: Any
                
                multi_node_ecs_task_properties_property = batch.CfnJobDefinition.MultiNodeEcsTaskPropertiesProperty(
                    containers=[batch.CfnJobDefinition.TaskContainerPropertiesProperty(
                        image="image",
                
                        # the properties below are optional
                        command=["command"],
                        depends_on=[batch.CfnJobDefinition.TaskContainerDependencyProperty(
                            condition="condition",
                            container_name="containerName"
                        )],
                        environment=[batch.CfnJobDefinition.EnvironmentProperty(
                            name="name",
                            value="value"
                        )],
                        essential=False,
                        linux_parameters=batch.CfnJobDefinition.LinuxParametersProperty(
                            devices=[batch.CfnJobDefinition.DeviceProperty(
                                container_path="containerPath",
                                host_path="hostPath",
                                permissions=["permissions"]
                            )],
                            init_process_enabled=False,
                            max_swap=123,
                            shared_memory_size=123,
                            swappiness=123,
                            tmpfs=[batch.CfnJobDefinition.TmpfsProperty(
                                container_path="containerPath",
                                size=123,
                
                                # the properties below are optional
                                mount_options=["mountOptions"]
                            )]
                        ),
                        log_configuration=batch.CfnJobDefinition.LogConfigurationProperty(
                            log_driver="logDriver",
                
                            # the properties below are optional
                            options=options,
                            secret_options=[batch.CfnJobDefinition.SecretProperty(
                                name="name",
                                value_from="valueFrom"
                            )]
                        ),
                        mount_points=[batch.CfnJobDefinition.MountPointProperty(
                            container_path="containerPath",
                            read_only=False,
                            source_volume="sourceVolume"
                        )],
                        name="name",
                        privileged=False,
                        readonly_root_filesystem=False,
                        repository_credentials=batch.CfnJobDefinition.RepositoryCredentialsProperty(
                            credentials_parameter="credentialsParameter"
                        ),
                        resource_requirements=[batch.CfnJobDefinition.ResourceRequirementProperty(
                            type="type",
                            value="value"
                        )],
                        secrets=[batch.CfnJobDefinition.SecretProperty(
                            name="name",
                            value_from="valueFrom"
                        )],
                        ulimits=[batch.CfnJobDefinition.UlimitProperty(
                            hard_limit=123,
                            name="name",
                            soft_limit=123
                        )],
                        user="user"
                    )],
                    execution_role_arn="executionRoleArn",
                    ipc_mode="ipcMode",
                    pid_mode="pidMode",
                    task_role_arn="taskRoleArn",
                    volumes=[batch.CfnJobDefinition.VolumesProperty(
                        efs_volume_configuration=batch.CfnJobDefinition.EfsVolumeConfigurationProperty(
                            file_system_id="fileSystemId",
                
                            # the properties below are optional
                            authorization_config=batch.CfnJobDefinition.AuthorizationConfigProperty(
                                access_point_id="accessPointId",
                                iam="iam"
                            ),
                            root_directory="rootDirectory",
                            transit_encryption="transitEncryption",
                            transit_encryption_port=123
                        ),
                        host=batch.CfnJobDefinition.VolumesHostProperty(
                            source_path="sourcePath"
                        ),
                        name="name"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cb3ca5619c68fc8ff93d966eb183ea114ed6764e99467d7b283af54300b4b5ca)
                check_type(argname="argument containers", value=containers, expected_type=type_hints["containers"])
                check_type(argname="argument execution_role_arn", value=execution_role_arn, expected_type=type_hints["execution_role_arn"])
                check_type(argname="argument ipc_mode", value=ipc_mode, expected_type=type_hints["ipc_mode"])
                check_type(argname="argument pid_mode", value=pid_mode, expected_type=type_hints["pid_mode"])
                check_type(argname="argument task_role_arn", value=task_role_arn, expected_type=type_hints["task_role_arn"])
                check_type(argname="argument volumes", value=volumes, expected_type=type_hints["volumes"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if containers is not None:
                self._values["containers"] = containers
            if execution_role_arn is not None:
                self._values["execution_role_arn"] = execution_role_arn
            if ipc_mode is not None:
                self._values["ipc_mode"] = ipc_mode
            if pid_mode is not None:
                self._values["pid_mode"] = pid_mode
            if task_role_arn is not None:
                self._values["task_role_arn"] = task_role_arn
            if volumes is not None:
                self._values["volumes"] = volumes

        @builtins.property
        def containers(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.TaskContainerPropertiesProperty"]]]]:
            '''This object is a list of containers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-multinodeecstaskproperties.html#cfn-batch-jobdefinition-multinodeecstaskproperties-containers
            '''
            result = self._values.get("containers")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.TaskContainerPropertiesProperty"]]]], result)

        @builtins.property
        def execution_role_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the execution role that AWS Batch can assume.

            For jobs that run on Fargate resources, you must provide an execution role. For more information, see `AWS Batch execution IAM role <https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html>`_ in the *AWS Batch User Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-multinodeecstaskproperties.html#cfn-batch-jobdefinition-multinodeecstaskproperties-executionrolearn
            '''
            result = self._values.get("execution_role_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ipc_mode(self) -> typing.Optional[builtins.str]:
            '''The IPC resource namespace to use for the containers in the task.

            The valid values are ``host`` , ``task`` , or ``none`` .

            If ``host`` is specified, all containers within the tasks that specified the ``host`` IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance.

            If ``task`` is specified, all containers within the specified ``task`` share the same IPC resources.

            If ``none`` is specified, the IPC resources within the containers of a task are private, and are not shared with other containers in a task or on the container instance.

            If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance. For more information, see `IPC settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#ipc-settings---ipc>`_ in the Docker run reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-multinodeecstaskproperties.html#cfn-batch-jobdefinition-multinodeecstaskproperties-ipcmode
            '''
            result = self._values.get("ipc_mode")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def pid_mode(self) -> typing.Optional[builtins.str]:
            '''The process namespace to use for the containers in the task.

            The valid values are ``host`` or ``task`` . For example, monitoring sidecars might need ``pidMode`` to access information about other containers running in the same task.

            If ``host`` is specified, all containers within the tasks that specified the ``host`` PID mode on the same container instance share the process namespace with the host Amazon EC2 instance.

            If ``task`` is specified, all containers within the specified task share the same process namespace.

            If no value is specified, the default is a private namespace for each container. For more information, see `PID settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#pid-settings---pid>`_ in the Docker run reference.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-multinodeecstaskproperties.html#cfn-batch-jobdefinition-multinodeecstaskproperties-pidmode
            '''
            result = self._values.get("pid_mode")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def task_role_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) that's associated with the Amazon ECS task.

            .. epigraph::

               This is object is comparable to `ContainerProperties:jobRoleArn <https://docs.aws.amazon.com/batch/latest/APIReference/API_ContainerProperties.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-multinodeecstaskproperties.html#cfn-batch-jobdefinition-multinodeecstaskproperties-taskrolearn
            '''
            result = self._values.get("task_role_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def volumes(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.VolumesProperty"]]]]:
            '''A list of volumes that are associated with the job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-multinodeecstaskproperties.html#cfn-batch-jobdefinition-multinodeecstaskproperties-volumes
            '''
            result = self._values.get("volumes")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.VolumesProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MultiNodeEcsTaskPropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.NetworkConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"assign_public_ip": "assignPublicIp"},
    )
    class NetworkConfigurationProperty:
        def __init__(
            self,
            *,
            assign_public_ip: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The network configuration for jobs that are running on Fargate resources.

            Jobs that are running on Amazon EC2 resources must not specify this parameter.

            :param assign_public_ip: Indicates whether the job has a public IP address. For a job that's running on Fargate resources in a private subnet to send outbound traffic to the internet (for example, to pull container images), the private subnet requires a NAT gateway be attached to route requests to the internet. For more information, see `Amazon ECS task networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* . The default value is " ``DISABLED`` ".

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-networkconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                network_configuration_property = batch.CfnJobDefinition.NetworkConfigurationProperty(
                    assign_public_ip="assignPublicIp"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__681e5c00b804be1ae6d3076ea32a17483eee30db6c96e73768c520d652064300)
                check_type(argname="argument assign_public_ip", value=assign_public_ip, expected_type=type_hints["assign_public_ip"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if assign_public_ip is not None:
                self._values["assign_public_ip"] = assign_public_ip

        @builtins.property
        def assign_public_ip(self) -> typing.Optional[builtins.str]:
            '''Indicates whether the job has a public IP address.

            For a job that's running on Fargate resources in a private subnet to send outbound traffic to the internet (for example, to pull container images), the private subnet requires a NAT gateway be attached to route requests to the internet. For more information, see `Amazon ECS task networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* . The default value is " ``DISABLED`` ".

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-networkconfiguration.html#cfn-batch-jobdefinition-networkconfiguration-assignpublicip
            '''
            result = self._values.get("assign_public_ip")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.NodePropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "main_node": "mainNode",
            "node_range_properties": "nodeRangeProperties",
            "num_nodes": "numNodes",
        },
    )
    class NodePropertiesProperty:
        def __init__(
            self,
            *,
            main_node: jsii.Number,
            node_range_properties: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.NodeRangePropertyProperty", typing.Dict[builtins.str, typing.Any]]]]],
            num_nodes: jsii.Number,
        ) -> None:
            '''An object that represents the node properties of a multi-node parallel job.

            .. epigraph::

               Node properties can't be specified for Amazon EKS based job definitions.

            :param main_node: Specifies the node index for the main node of a multi-node parallel job. This node index value must be fewer than the number of nodes.
            :param node_range_properties: A list of node ranges and their properties that are associated with a multi-node parallel job.
            :param num_nodes: The number of nodes that are associated with a multi-node parallel job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-nodeproperties.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # labels: Any
                # limits: Any
                # options: Any
                # requests: Any
                
                node_properties_property = batch.CfnJobDefinition.NodePropertiesProperty(
                    main_node=123,
                    node_range_properties=[batch.CfnJobDefinition.NodeRangePropertyProperty(
                        target_nodes="targetNodes",
                
                        # the properties below are optional
                        consumable_resource_properties=batch.CfnJobDefinition.ConsumableResourcePropertiesProperty(
                            consumable_resource_list=[batch.CfnJobDefinition.ConsumableResourceRequirementProperty(
                                consumable_resource="consumableResource",
                                quantity=123
                            )]
                        ),
                        container=batch.CfnJobDefinition.ContainerPropertiesProperty(
                            image="image",
                
                            # the properties below are optional
                            command=["command"],
                            environment=[batch.CfnJobDefinition.EnvironmentProperty(
                                name="name",
                                value="value"
                            )],
                            ephemeral_storage=batch.CfnJobDefinition.EphemeralStorageProperty(
                                size_in_gi_b=123
                            ),
                            execution_role_arn="executionRoleArn",
                            fargate_platform_configuration=batch.CfnJobDefinition.FargatePlatformConfigurationProperty(
                                platform_version="platformVersion"
                            ),
                            instance_type="instanceType",
                            job_role_arn="jobRoleArn",
                            linux_parameters=batch.CfnJobDefinition.LinuxParametersProperty(
                                devices=[batch.CfnJobDefinition.DeviceProperty(
                                    container_path="containerPath",
                                    host_path="hostPath",
                                    permissions=["permissions"]
                                )],
                                init_process_enabled=False,
                                max_swap=123,
                                shared_memory_size=123,
                                swappiness=123,
                                tmpfs=[batch.CfnJobDefinition.TmpfsProperty(
                                    container_path="containerPath",
                                    size=123,
                
                                    # the properties below are optional
                                    mount_options=["mountOptions"]
                                )]
                            ),
                            log_configuration=batch.CfnJobDefinition.LogConfigurationProperty(
                                log_driver="logDriver",
                
                                # the properties below are optional
                                options=options,
                                secret_options=[batch.CfnJobDefinition.SecretProperty(
                                    name="name",
                                    value_from="valueFrom"
                                )]
                            ),
                            memory=123,
                            mount_points=[batch.CfnJobDefinition.MountPointsProperty(
                                container_path="containerPath",
                                read_only=False,
                                source_volume="sourceVolume"
                            )],
                            network_configuration=batch.CfnJobDefinition.NetworkConfigurationProperty(
                                assign_public_ip="assignPublicIp"
                            ),
                            privileged=False,
                            readonly_root_filesystem=False,
                            repository_credentials=batch.CfnJobDefinition.RepositoryCredentialsProperty(
                                credentials_parameter="credentialsParameter"
                            ),
                            resource_requirements=[batch.CfnJobDefinition.ResourceRequirementProperty(
                                type="type",
                                value="value"
                            )],
                            runtime_platform=batch.CfnJobDefinition.RuntimePlatformProperty(
                                cpu_architecture="cpuArchitecture",
                                operating_system_family="operatingSystemFamily"
                            ),
                            secrets=[batch.CfnJobDefinition.SecretProperty(
                                name="name",
                                value_from="valueFrom"
                            )],
                            ulimits=[batch.CfnJobDefinition.UlimitProperty(
                                hard_limit=123,
                                name="name",
                                soft_limit=123
                            )],
                            user="user",
                            vcpus=123,
                            volumes=[batch.CfnJobDefinition.VolumesProperty(
                                efs_volume_configuration=batch.CfnJobDefinition.EfsVolumeConfigurationProperty(
                                    file_system_id="fileSystemId",
                
                                    # the properties below are optional
                                    authorization_config=batch.CfnJobDefinition.AuthorizationConfigProperty(
                                        access_point_id="accessPointId",
                                        iam="iam"
                                    ),
                                    root_directory="rootDirectory",
                                    transit_encryption="transitEncryption",
                                    transit_encryption_port=123
                                ),
                                host=batch.CfnJobDefinition.VolumesHostProperty(
                                    source_path="sourcePath"
                                ),
                                name="name"
                            )]
                        ),
                        ecs_properties=batch.CfnJobDefinition.MultiNodeEcsPropertiesProperty(
                            task_properties=[batch.CfnJobDefinition.MultiNodeEcsTaskPropertiesProperty(
                                containers=[batch.CfnJobDefinition.TaskContainerPropertiesProperty(
                                    image="image",
                
                                    # the properties below are optional
                                    command=["command"],
                                    depends_on=[batch.CfnJobDefinition.TaskContainerDependencyProperty(
                                        condition="condition",
                                        container_name="containerName"
                                    )],
                                    environment=[batch.CfnJobDefinition.EnvironmentProperty(
                                        name="name",
                                        value="value"
                                    )],
                                    essential=False,
                                    linux_parameters=batch.CfnJobDefinition.LinuxParametersProperty(
                                        devices=[batch.CfnJobDefinition.DeviceProperty(
                                            container_path="containerPath",
                                            host_path="hostPath",
                                            permissions=["permissions"]
                                        )],
                                        init_process_enabled=False,
                                        max_swap=123,
                                        shared_memory_size=123,
                                        swappiness=123,
                                        tmpfs=[batch.CfnJobDefinition.TmpfsProperty(
                                            container_path="containerPath",
                                            size=123,
                
                                            # the properties below are optional
                                            mount_options=["mountOptions"]
                                        )]
                                    ),
                                    log_configuration=batch.CfnJobDefinition.LogConfigurationProperty(
                                        log_driver="logDriver",
                
                                        # the properties below are optional
                                        options=options,
                                        secret_options=[batch.CfnJobDefinition.SecretProperty(
                                            name="name",
                                            value_from="valueFrom"
                                        )]
                                    ),
                                    mount_points=[batch.CfnJobDefinition.MountPointProperty(
                                        container_path="containerPath",
                                        read_only=False,
                                        source_volume="sourceVolume"
                                    )],
                                    name="name",
                                    privileged=False,
                                    readonly_root_filesystem=False,
                                    repository_credentials=batch.CfnJobDefinition.RepositoryCredentialsProperty(
                                        credentials_parameter="credentialsParameter"
                                    ),
                                    resource_requirements=[batch.CfnJobDefinition.ResourceRequirementProperty(
                                        type="type",
                                        value="value"
                                    )],
                                    secrets=[batch.CfnJobDefinition.SecretProperty(
                                        name="name",
                                        value_from="valueFrom"
                                    )],
                                    ulimits=[batch.CfnJobDefinition.UlimitProperty(
                                        hard_limit=123,
                                        name="name",
                                        soft_limit=123
                                    )],
                                    user="user"
                                )],
                                execution_role_arn="executionRoleArn",
                                ipc_mode="ipcMode",
                                pid_mode="pidMode",
                                task_role_arn="taskRoleArn",
                                volumes=[batch.CfnJobDefinition.VolumesProperty(
                                    efs_volume_configuration=batch.CfnJobDefinition.EfsVolumeConfigurationProperty(
                                        file_system_id="fileSystemId",
                
                                        # the properties below are optional
                                        authorization_config=batch.CfnJobDefinition.AuthorizationConfigProperty(
                                            access_point_id="accessPointId",
                                            iam="iam"
                                        ),
                                        root_directory="rootDirectory",
                                        transit_encryption="transitEncryption",
                                        transit_encryption_port=123
                                    ),
                                    host=batch.CfnJobDefinition.VolumesHostProperty(
                                        source_path="sourcePath"
                                    ),
                                    name="name"
                                )]
                            )]
                        ),
                        eks_properties=batch.CfnJobDefinition.EksPropertiesProperty(
                            pod_properties=batch.CfnJobDefinition.PodPropertiesProperty(
                                containers=[batch.CfnJobDefinition.EksContainerProperty(
                                    image="image",
                
                                    # the properties below are optional
                                    args=["args"],
                                    command=["command"],
                                    env=[batch.CfnJobDefinition.EksContainerEnvironmentVariableProperty(
                                        name="name",
                
                                        # the properties below are optional
                                        value="value"
                                    )],
                                    image_pull_policy="imagePullPolicy",
                                    name="name",
                                    resources=batch.CfnJobDefinition.ResourcesProperty(
                                        limits=limits,
                                        requests=requests
                                    ),
                                    security_context=batch.CfnJobDefinition.SecurityContextProperty(
                                        allow_privilege_escalation=False,
                                        privileged=False,
                                        read_only_root_filesystem=False,
                                        run_as_group=123,
                                        run_as_non_root=False,
                                        run_as_user=123
                                    ),
                                    volume_mounts=[batch.CfnJobDefinition.EksContainerVolumeMountProperty(
                                        mount_path="mountPath",
                                        name="name",
                                        read_only=False,
                                        sub_path="subPath"
                                    )]
                                )],
                                dns_policy="dnsPolicy",
                                host_network=False,
                                image_pull_secrets=[batch.CfnJobDefinition.ImagePullSecretProperty(
                                    name="name"
                                )],
                                init_containers=[batch.CfnJobDefinition.EksContainerProperty(
                                    image="image",
                
                                    # the properties below are optional
                                    args=["args"],
                                    command=["command"],
                                    env=[batch.CfnJobDefinition.EksContainerEnvironmentVariableProperty(
                                        name="name",
                
                                        # the properties below are optional
                                        value="value"
                                    )],
                                    image_pull_policy="imagePullPolicy",
                                    name="name",
                                    resources=batch.CfnJobDefinition.ResourcesProperty(
                                        limits=limits,
                                        requests=requests
                                    ),
                                    security_context=batch.CfnJobDefinition.SecurityContextProperty(
                                        allow_privilege_escalation=False,
                                        privileged=False,
                                        read_only_root_filesystem=False,
                                        run_as_group=123,
                                        run_as_non_root=False,
                                        run_as_user=123
                                    ),
                                    volume_mounts=[batch.CfnJobDefinition.EksContainerVolumeMountProperty(
                                        mount_path="mountPath",
                                        name="name",
                                        read_only=False,
                                        sub_path="subPath"
                                    )]
                                )],
                                metadata=batch.CfnJobDefinition.MetadataProperty(
                                    labels=labels
                                ),
                                service_account_name="serviceAccountName",
                                share_process_namespace=False,
                                volumes=[batch.CfnJobDefinition.EksVolumeProperty(
                                    name="name",
                
                                    # the properties below are optional
                                    empty_dir=batch.CfnJobDefinition.EmptyDirProperty(
                                        medium="medium",
                                        size_limit="sizeLimit"
                                    ),
                                    host_path=batch.CfnJobDefinition.HostPathProperty(
                                        path="path"
                                    ),
                                    persistent_volume_claim=batch.CfnJobDefinition.EksPersistentVolumeClaimProperty(
                                        claim_name="claimName",
                
                                        # the properties below are optional
                                        read_only=False
                                    ),
                                    secret=batch.CfnJobDefinition.EksSecretProperty(
                                        secret_name="secretName",
                
                                        # the properties below are optional
                                        optional=False
                                    )
                                )]
                            )
                        ),
                        instance_types=["instanceTypes"]
                    )],
                    num_nodes=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b765fef4e53f81f8c1a37eea0ea015533bd4621e0f00e5e86923931469f2fbd1)
                check_type(argname="argument main_node", value=main_node, expected_type=type_hints["main_node"])
                check_type(argname="argument node_range_properties", value=node_range_properties, expected_type=type_hints["node_range_properties"])
                check_type(argname="argument num_nodes", value=num_nodes, expected_type=type_hints["num_nodes"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "main_node": main_node,
                "node_range_properties": node_range_properties,
                "num_nodes": num_nodes,
            }

        @builtins.property
        def main_node(self) -> jsii.Number:
            '''Specifies the node index for the main node of a multi-node parallel job.

            This node index value must be fewer than the number of nodes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-nodeproperties.html#cfn-batch-jobdefinition-nodeproperties-mainnode
            '''
            result = self._values.get("main_node")
            assert result is not None, "Required property 'main_node' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def node_range_properties(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.NodeRangePropertyProperty"]]]:
            '''A list of node ranges and their properties that are associated with a multi-node parallel job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-nodeproperties.html#cfn-batch-jobdefinition-nodeproperties-noderangeproperties
            '''
            result = self._values.get("node_range_properties")
            assert result is not None, "Required property 'node_range_properties' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.NodeRangePropertyProperty"]]], result)

        @builtins.property
        def num_nodes(self) -> jsii.Number:
            '''The number of nodes that are associated with a multi-node parallel job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-nodeproperties.html#cfn-batch-jobdefinition-nodeproperties-numnodes
            '''
            result = self._values.get("num_nodes")
            assert result is not None, "Required property 'num_nodes' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NodePropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.NodeRangePropertyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "target_nodes": "targetNodes",
            "consumable_resource_properties": "consumableResourceProperties",
            "container": "container",
            "ecs_properties": "ecsProperties",
            "eks_properties": "eksProperties",
            "instance_types": "instanceTypes",
        },
    )
    class NodeRangePropertyProperty:
        def __init__(
            self,
            *,
            target_nodes: builtins.str,
            consumable_resource_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.ConsumableResourcePropertiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            container: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.ContainerPropertiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            ecs_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.MultiNodeEcsPropertiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            eks_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EksPropertiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            instance_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''This is an object that represents the properties of the node range for a multi-node parallel job.

            :param target_nodes: The range of nodes, using node index values. A range of ``0:3`` indicates nodes with index values of ``0`` through ``3`` . If the starting range value is omitted ( ``:n`` ), then ``0`` is used to start the range. If the ending range value is omitted ( ``n:`` ), then the highest possible node index is used to end the range. Your accumulative node ranges must account for all nodes ( ``0:n`` ). You can nest node ranges (for example, ``0:10`` and ``4:5`` ). In this case, the ``4:5`` range properties override the ``0:10`` properties.
            :param consumable_resource_properties: Contains a list of consumable resources required by a job.
            :param container: The container details for the node range.
            :param ecs_properties: This is an object that represents the properties of the node range for a multi-node parallel job.
            :param eks_properties: This is an object that represents the properties of the node range for a multi-node parallel job.
            :param instance_types: The instance types of the underlying host infrastructure of a multi-node parallel job. .. epigraph:: This parameter isn't applicable to jobs that are running on Fargate resources. In addition, this list object is currently limited to one element.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-noderangeproperty.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # labels: Any
                # limits: Any
                # options: Any
                # requests: Any
                
                node_range_property_property = batch.CfnJobDefinition.NodeRangePropertyProperty(
                    target_nodes="targetNodes",
                
                    # the properties below are optional
                    consumable_resource_properties=batch.CfnJobDefinition.ConsumableResourcePropertiesProperty(
                        consumable_resource_list=[batch.CfnJobDefinition.ConsumableResourceRequirementProperty(
                            consumable_resource="consumableResource",
                            quantity=123
                        )]
                    ),
                    container=batch.CfnJobDefinition.ContainerPropertiesProperty(
                        image="image",
                
                        # the properties below are optional
                        command=["command"],
                        environment=[batch.CfnJobDefinition.EnvironmentProperty(
                            name="name",
                            value="value"
                        )],
                        ephemeral_storage=batch.CfnJobDefinition.EphemeralStorageProperty(
                            size_in_gi_b=123
                        ),
                        execution_role_arn="executionRoleArn",
                        fargate_platform_configuration=batch.CfnJobDefinition.FargatePlatformConfigurationProperty(
                            platform_version="platformVersion"
                        ),
                        instance_type="instanceType",
                        job_role_arn="jobRoleArn",
                        linux_parameters=batch.CfnJobDefinition.LinuxParametersProperty(
                            devices=[batch.CfnJobDefinition.DeviceProperty(
                                container_path="containerPath",
                                host_path="hostPath",
                                permissions=["permissions"]
                            )],
                            init_process_enabled=False,
                            max_swap=123,
                            shared_memory_size=123,
                            swappiness=123,
                            tmpfs=[batch.CfnJobDefinition.TmpfsProperty(
                                container_path="containerPath",
                                size=123,
                
                                # the properties below are optional
                                mount_options=["mountOptions"]
                            )]
                        ),
                        log_configuration=batch.CfnJobDefinition.LogConfigurationProperty(
                            log_driver="logDriver",
                
                            # the properties below are optional
                            options=options,
                            secret_options=[batch.CfnJobDefinition.SecretProperty(
                                name="name",
                                value_from="valueFrom"
                            )]
                        ),
                        memory=123,
                        mount_points=[batch.CfnJobDefinition.MountPointsProperty(
                            container_path="containerPath",
                            read_only=False,
                            source_volume="sourceVolume"
                        )],
                        network_configuration=batch.CfnJobDefinition.NetworkConfigurationProperty(
                            assign_public_ip="assignPublicIp"
                        ),
                        privileged=False,
                        readonly_root_filesystem=False,
                        repository_credentials=batch.CfnJobDefinition.RepositoryCredentialsProperty(
                            credentials_parameter="credentialsParameter"
                        ),
                        resource_requirements=[batch.CfnJobDefinition.ResourceRequirementProperty(
                            type="type",
                            value="value"
                        )],
                        runtime_platform=batch.CfnJobDefinition.RuntimePlatformProperty(
                            cpu_architecture="cpuArchitecture",
                            operating_system_family="operatingSystemFamily"
                        ),
                        secrets=[batch.CfnJobDefinition.SecretProperty(
                            name="name",
                            value_from="valueFrom"
                        )],
                        ulimits=[batch.CfnJobDefinition.UlimitProperty(
                            hard_limit=123,
                            name="name",
                            soft_limit=123
                        )],
                        user="user",
                        vcpus=123,
                        volumes=[batch.CfnJobDefinition.VolumesProperty(
                            efs_volume_configuration=batch.CfnJobDefinition.EfsVolumeConfigurationProperty(
                                file_system_id="fileSystemId",
                
                                # the properties below are optional
                                authorization_config=batch.CfnJobDefinition.AuthorizationConfigProperty(
                                    access_point_id="accessPointId",
                                    iam="iam"
                                ),
                                root_directory="rootDirectory",
                                transit_encryption="transitEncryption",
                                transit_encryption_port=123
                            ),
                            host=batch.CfnJobDefinition.VolumesHostProperty(
                                source_path="sourcePath"
                            ),
                            name="name"
                        )]
                    ),
                    ecs_properties=batch.CfnJobDefinition.MultiNodeEcsPropertiesProperty(
                        task_properties=[batch.CfnJobDefinition.MultiNodeEcsTaskPropertiesProperty(
                            containers=[batch.CfnJobDefinition.TaskContainerPropertiesProperty(
                                image="image",
                
                                # the properties below are optional
                                command=["command"],
                                depends_on=[batch.CfnJobDefinition.TaskContainerDependencyProperty(
                                    condition="condition",
                                    container_name="containerName"
                                )],
                                environment=[batch.CfnJobDefinition.EnvironmentProperty(
                                    name="name",
                                    value="value"
                                )],
                                essential=False,
                                linux_parameters=batch.CfnJobDefinition.LinuxParametersProperty(
                                    devices=[batch.CfnJobDefinition.DeviceProperty(
                                        container_path="containerPath",
                                        host_path="hostPath",
                                        permissions=["permissions"]
                                    )],
                                    init_process_enabled=False,
                                    max_swap=123,
                                    shared_memory_size=123,
                                    swappiness=123,
                                    tmpfs=[batch.CfnJobDefinition.TmpfsProperty(
                                        container_path="containerPath",
                                        size=123,
                
                                        # the properties below are optional
                                        mount_options=["mountOptions"]
                                    )]
                                ),
                                log_configuration=batch.CfnJobDefinition.LogConfigurationProperty(
                                    log_driver="logDriver",
                
                                    # the properties below are optional
                                    options=options,
                                    secret_options=[batch.CfnJobDefinition.SecretProperty(
                                        name="name",
                                        value_from="valueFrom"
                                    )]
                                ),
                                mount_points=[batch.CfnJobDefinition.MountPointProperty(
                                    container_path="containerPath",
                                    read_only=False,
                                    source_volume="sourceVolume"
                                )],
                                name="name",
                                privileged=False,
                                readonly_root_filesystem=False,
                                repository_credentials=batch.CfnJobDefinition.RepositoryCredentialsProperty(
                                    credentials_parameter="credentialsParameter"
                                ),
                                resource_requirements=[batch.CfnJobDefinition.ResourceRequirementProperty(
                                    type="type",
                                    value="value"
                                )],
                                secrets=[batch.CfnJobDefinition.SecretProperty(
                                    name="name",
                                    value_from="valueFrom"
                                )],
                                ulimits=[batch.CfnJobDefinition.UlimitProperty(
                                    hard_limit=123,
                                    name="name",
                                    soft_limit=123
                                )],
                                user="user"
                            )],
                            execution_role_arn="executionRoleArn",
                            ipc_mode="ipcMode",
                            pid_mode="pidMode",
                            task_role_arn="taskRoleArn",
                            volumes=[batch.CfnJobDefinition.VolumesProperty(
                                efs_volume_configuration=batch.CfnJobDefinition.EfsVolumeConfigurationProperty(
                                    file_system_id="fileSystemId",
                
                                    # the properties below are optional
                                    authorization_config=batch.CfnJobDefinition.AuthorizationConfigProperty(
                                        access_point_id="accessPointId",
                                        iam="iam"
                                    ),
                                    root_directory="rootDirectory",
                                    transit_encryption="transitEncryption",
                                    transit_encryption_port=123
                                ),
                                host=batch.CfnJobDefinition.VolumesHostProperty(
                                    source_path="sourcePath"
                                ),
                                name="name"
                            )]
                        )]
                    ),
                    eks_properties=batch.CfnJobDefinition.EksPropertiesProperty(
                        pod_properties=batch.CfnJobDefinition.PodPropertiesProperty(
                            containers=[batch.CfnJobDefinition.EksContainerProperty(
                                image="image",
                
                                # the properties below are optional
                                args=["args"],
                                command=["command"],
                                env=[batch.CfnJobDefinition.EksContainerEnvironmentVariableProperty(
                                    name="name",
                
                                    # the properties below are optional
                                    value="value"
                                )],
                                image_pull_policy="imagePullPolicy",
                                name="name",
                                resources=batch.CfnJobDefinition.ResourcesProperty(
                                    limits=limits,
                                    requests=requests
                                ),
                                security_context=batch.CfnJobDefinition.SecurityContextProperty(
                                    allow_privilege_escalation=False,
                                    privileged=False,
                                    read_only_root_filesystem=False,
                                    run_as_group=123,
                                    run_as_non_root=False,
                                    run_as_user=123
                                ),
                                volume_mounts=[batch.CfnJobDefinition.EksContainerVolumeMountProperty(
                                    mount_path="mountPath",
                                    name="name",
                                    read_only=False,
                                    sub_path="subPath"
                                )]
                            )],
                            dns_policy="dnsPolicy",
                            host_network=False,
                            image_pull_secrets=[batch.CfnJobDefinition.ImagePullSecretProperty(
                                name="name"
                            )],
                            init_containers=[batch.CfnJobDefinition.EksContainerProperty(
                                image="image",
                
                                # the properties below are optional
                                args=["args"],
                                command=["command"],
                                env=[batch.CfnJobDefinition.EksContainerEnvironmentVariableProperty(
                                    name="name",
                
                                    # the properties below are optional
                                    value="value"
                                )],
                                image_pull_policy="imagePullPolicy",
                                name="name",
                                resources=batch.CfnJobDefinition.ResourcesProperty(
                                    limits=limits,
                                    requests=requests
                                ),
                                security_context=batch.CfnJobDefinition.SecurityContextProperty(
                                    allow_privilege_escalation=False,
                                    privileged=False,
                                    read_only_root_filesystem=False,
                                    run_as_group=123,
                                    run_as_non_root=False,
                                    run_as_user=123
                                ),
                                volume_mounts=[batch.CfnJobDefinition.EksContainerVolumeMountProperty(
                                    mount_path="mountPath",
                                    name="name",
                                    read_only=False,
                                    sub_path="subPath"
                                )]
                            )],
                            metadata=batch.CfnJobDefinition.MetadataProperty(
                                labels=labels
                            ),
                            service_account_name="serviceAccountName",
                            share_process_namespace=False,
                            volumes=[batch.CfnJobDefinition.EksVolumeProperty(
                                name="name",
                
                                # the properties below are optional
                                empty_dir=batch.CfnJobDefinition.EmptyDirProperty(
                                    medium="medium",
                                    size_limit="sizeLimit"
                                ),
                                host_path=batch.CfnJobDefinition.HostPathProperty(
                                    path="path"
                                ),
                                persistent_volume_claim=batch.CfnJobDefinition.EksPersistentVolumeClaimProperty(
                                    claim_name="claimName",
                
                                    # the properties below are optional
                                    read_only=False
                                ),
                                secret=batch.CfnJobDefinition.EksSecretProperty(
                                    secret_name="secretName",
                
                                    # the properties below are optional
                                    optional=False
                                )
                            )]
                        )
                    ),
                    instance_types=["instanceTypes"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__005c21025a81827c3f5ed456b171eb7ffdf652583c7da0ff6ff67186436afeee)
                check_type(argname="argument target_nodes", value=target_nodes, expected_type=type_hints["target_nodes"])
                check_type(argname="argument consumable_resource_properties", value=consumable_resource_properties, expected_type=type_hints["consumable_resource_properties"])
                check_type(argname="argument container", value=container, expected_type=type_hints["container"])
                check_type(argname="argument ecs_properties", value=ecs_properties, expected_type=type_hints["ecs_properties"])
                check_type(argname="argument eks_properties", value=eks_properties, expected_type=type_hints["eks_properties"])
                check_type(argname="argument instance_types", value=instance_types, expected_type=type_hints["instance_types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "target_nodes": target_nodes,
            }
            if consumable_resource_properties is not None:
                self._values["consumable_resource_properties"] = consumable_resource_properties
            if container is not None:
                self._values["container"] = container
            if ecs_properties is not None:
                self._values["ecs_properties"] = ecs_properties
            if eks_properties is not None:
                self._values["eks_properties"] = eks_properties
            if instance_types is not None:
                self._values["instance_types"] = instance_types

        @builtins.property
        def target_nodes(self) -> builtins.str:
            '''The range of nodes, using node index values.

            A range of ``0:3`` indicates nodes with index values of ``0`` through ``3`` . If the starting range value is omitted ( ``:n`` ), then ``0`` is used to start the range. If the ending range value is omitted ( ``n:`` ), then the highest possible node index is used to end the range. Your accumulative node ranges must account for all nodes ( ``0:n`` ). You can nest node ranges (for example, ``0:10`` and ``4:5`` ). In this case, the ``4:5`` range properties override the ``0:10`` properties.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-noderangeproperty.html#cfn-batch-jobdefinition-noderangeproperty-targetnodes
            '''
            result = self._values.get("target_nodes")
            assert result is not None, "Required property 'target_nodes' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def consumable_resource_properties(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ConsumableResourcePropertiesProperty"]]:
            '''Contains a list of consumable resources required by a job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-noderangeproperty.html#cfn-batch-jobdefinition-noderangeproperty-consumableresourceproperties
            '''
            result = self._values.get("consumable_resource_properties")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ConsumableResourcePropertiesProperty"]], result)

        @builtins.property
        def container(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ContainerPropertiesProperty"]]:
            '''The container details for the node range.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-noderangeproperty.html#cfn-batch-jobdefinition-noderangeproperty-container
            '''
            result = self._values.get("container")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ContainerPropertiesProperty"]], result)

        @builtins.property
        def ecs_properties(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.MultiNodeEcsPropertiesProperty"]]:
            '''This is an object that represents the properties of the node range for a multi-node parallel job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-noderangeproperty.html#cfn-batch-jobdefinition-noderangeproperty-ecsproperties
            '''
            result = self._values.get("ecs_properties")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.MultiNodeEcsPropertiesProperty"]], result)

        @builtins.property
        def eks_properties(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksPropertiesProperty"]]:
            '''This is an object that represents the properties of the node range for a multi-node parallel job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-noderangeproperty.html#cfn-batch-jobdefinition-noderangeproperty-eksproperties
            '''
            result = self._values.get("eks_properties")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksPropertiesProperty"]], result)

        @builtins.property
        def instance_types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The instance types of the underlying host infrastructure of a multi-node parallel job.

            .. epigraph::

               This parameter isn't applicable to jobs that are running on Fargate resources.

               In addition, this list object is currently limited to one element.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-noderangeproperty.html#cfn-batch-jobdefinition-noderangeproperty-instancetypes
            '''
            result = self._values.get("instance_types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NodeRangePropertyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.PodPropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "containers": "containers",
            "dns_policy": "dnsPolicy",
            "host_network": "hostNetwork",
            "image_pull_secrets": "imagePullSecrets",
            "init_containers": "initContainers",
            "metadata": "metadata",
            "service_account_name": "serviceAccountName",
            "share_process_namespace": "shareProcessNamespace",
            "volumes": "volumes",
        },
    )
    class PodPropertiesProperty:
        def __init__(
            self,
            *,
            containers: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EksContainerProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            dns_policy: typing.Optional[builtins.str] = None,
            host_network: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            image_pull_secrets: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.ImagePullSecretProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            init_containers: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EksContainerProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            metadata: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.MetadataProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            service_account_name: typing.Optional[builtins.str] = None,
            share_process_namespace: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            volumes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EksVolumeProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''
            :param containers: 
            :param dns_policy: 
            :param host_network: 
            :param image_pull_secrets: 
            :param init_containers: 
            :param metadata: 
            :param service_account_name: 
            :param share_process_namespace: 
            :param volumes: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-podproperties.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # labels: Any
                # limits: Any
                # requests: Any
                
                pod_properties_property = batch.CfnJobDefinition.PodPropertiesProperty(
                    containers=[batch.CfnJobDefinition.EksContainerProperty(
                        image="image",
                
                        # the properties below are optional
                        args=["args"],
                        command=["command"],
                        env=[batch.CfnJobDefinition.EksContainerEnvironmentVariableProperty(
                            name="name",
                
                            # the properties below are optional
                            value="value"
                        )],
                        image_pull_policy="imagePullPolicy",
                        name="name",
                        resources=batch.CfnJobDefinition.ResourcesProperty(
                            limits=limits,
                            requests=requests
                        ),
                        security_context=batch.CfnJobDefinition.SecurityContextProperty(
                            allow_privilege_escalation=False,
                            privileged=False,
                            read_only_root_filesystem=False,
                            run_as_group=123,
                            run_as_non_root=False,
                            run_as_user=123
                        ),
                        volume_mounts=[batch.CfnJobDefinition.EksContainerVolumeMountProperty(
                            mount_path="mountPath",
                            name="name",
                            read_only=False,
                            sub_path="subPath"
                        )]
                    )],
                    dns_policy="dnsPolicy",
                    host_network=False,
                    image_pull_secrets=[batch.CfnJobDefinition.ImagePullSecretProperty(
                        name="name"
                    )],
                    init_containers=[batch.CfnJobDefinition.EksContainerProperty(
                        image="image",
                
                        # the properties below are optional
                        args=["args"],
                        command=["command"],
                        env=[batch.CfnJobDefinition.EksContainerEnvironmentVariableProperty(
                            name="name",
                
                            # the properties below are optional
                            value="value"
                        )],
                        image_pull_policy="imagePullPolicy",
                        name="name",
                        resources=batch.CfnJobDefinition.ResourcesProperty(
                            limits=limits,
                            requests=requests
                        ),
                        security_context=batch.CfnJobDefinition.SecurityContextProperty(
                            allow_privilege_escalation=False,
                            privileged=False,
                            read_only_root_filesystem=False,
                            run_as_group=123,
                            run_as_non_root=False,
                            run_as_user=123
                        ),
                        volume_mounts=[batch.CfnJobDefinition.EksContainerVolumeMountProperty(
                            mount_path="mountPath",
                            name="name",
                            read_only=False,
                            sub_path="subPath"
                        )]
                    )],
                    metadata=batch.CfnJobDefinition.MetadataProperty(
                        labels=labels
                    ),
                    service_account_name="serviceAccountName",
                    share_process_namespace=False,
                    volumes=[batch.CfnJobDefinition.EksVolumeProperty(
                        name="name",
                
                        # the properties below are optional
                        empty_dir=batch.CfnJobDefinition.EmptyDirProperty(
                            medium="medium",
                            size_limit="sizeLimit"
                        ),
                        host_path=batch.CfnJobDefinition.HostPathProperty(
                            path="path"
                        ),
                        persistent_volume_claim=batch.CfnJobDefinition.EksPersistentVolumeClaimProperty(
                            claim_name="claimName",
                
                            # the properties below are optional
                            read_only=False
                        ),
                        secret=batch.CfnJobDefinition.EksSecretProperty(
                            secret_name="secretName",
                
                            # the properties below are optional
                            optional=False
                        )
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__75127ae5a1697c34be5f24dcb69fa5c36a3498e1b2c284babc814c444bc47b1c)
                check_type(argname="argument containers", value=containers, expected_type=type_hints["containers"])
                check_type(argname="argument dns_policy", value=dns_policy, expected_type=type_hints["dns_policy"])
                check_type(argname="argument host_network", value=host_network, expected_type=type_hints["host_network"])
                check_type(argname="argument image_pull_secrets", value=image_pull_secrets, expected_type=type_hints["image_pull_secrets"])
                check_type(argname="argument init_containers", value=init_containers, expected_type=type_hints["init_containers"])
                check_type(argname="argument metadata", value=metadata, expected_type=type_hints["metadata"])
                check_type(argname="argument service_account_name", value=service_account_name, expected_type=type_hints["service_account_name"])
                check_type(argname="argument share_process_namespace", value=share_process_namespace, expected_type=type_hints["share_process_namespace"])
                check_type(argname="argument volumes", value=volumes, expected_type=type_hints["volumes"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if containers is not None:
                self._values["containers"] = containers
            if dns_policy is not None:
                self._values["dns_policy"] = dns_policy
            if host_network is not None:
                self._values["host_network"] = host_network
            if image_pull_secrets is not None:
                self._values["image_pull_secrets"] = image_pull_secrets
            if init_containers is not None:
                self._values["init_containers"] = init_containers
            if metadata is not None:
                self._values["metadata"] = metadata
            if service_account_name is not None:
                self._values["service_account_name"] = service_account_name
            if share_process_namespace is not None:
                self._values["share_process_namespace"] = share_process_namespace
            if volumes is not None:
                self._values["volumes"] = volumes

        @builtins.property
        def containers(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksContainerProperty"]]]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-podproperties.html#cfn-batch-jobdefinition-podproperties-containers
            '''
            result = self._values.get("containers")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksContainerProperty"]]]], result)

        @builtins.property
        def dns_policy(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-podproperties.html#cfn-batch-jobdefinition-podproperties-dnspolicy
            '''
            result = self._values.get("dns_policy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_network(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-podproperties.html#cfn-batch-jobdefinition-podproperties-hostnetwork
            '''
            result = self._values.get("host_network")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def image_pull_secrets(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ImagePullSecretProperty"]]]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-podproperties.html#cfn-batch-jobdefinition-podproperties-imagepullsecrets
            '''
            result = self._values.get("image_pull_secrets")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ImagePullSecretProperty"]]]], result)

        @builtins.property
        def init_containers(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksContainerProperty"]]]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-podproperties.html#cfn-batch-jobdefinition-podproperties-initcontainers
            '''
            result = self._values.get("init_containers")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksContainerProperty"]]]], result)

        @builtins.property
        def metadata(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.MetadataProperty"]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-podproperties.html#cfn-batch-jobdefinition-podproperties-metadata
            '''
            result = self._values.get("metadata")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.MetadataProperty"]], result)

        @builtins.property
        def service_account_name(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-podproperties.html#cfn-batch-jobdefinition-podproperties-serviceaccountname
            '''
            result = self._values.get("service_account_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def share_process_namespace(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-podproperties.html#cfn-batch-jobdefinition-podproperties-shareprocessnamespace
            '''
            result = self._values.get("share_process_namespace")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def volumes(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksVolumeProperty"]]]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-podproperties.html#cfn-batch-jobdefinition-podproperties-volumes
            '''
            result = self._values.get("volumes")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EksVolumeProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PodPropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.RepositoryCredentialsProperty",
        jsii_struct_bases=[],
        name_mapping={"credentials_parameter": "credentialsParameter"},
    )
    class RepositoryCredentialsProperty:
        def __init__(self, *, credentials_parameter: builtins.str) -> None:
            '''The repository credentials for private registry authentication.

            :param credentials_parameter: The Amazon Resource Name (ARN) of the secret containing the private repository credentials.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-repositorycredentials.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                repository_credentials_property = batch.CfnJobDefinition.RepositoryCredentialsProperty(
                    credentials_parameter="credentialsParameter"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__efadf8b71047617519478ed3105db0ddee37f23729febb81c4d534eed7eba9dd)
                check_type(argname="argument credentials_parameter", value=credentials_parameter, expected_type=type_hints["credentials_parameter"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "credentials_parameter": credentials_parameter,
            }

        @builtins.property
        def credentials_parameter(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the secret containing the private repository credentials.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-repositorycredentials.html#cfn-batch-jobdefinition-repositorycredentials-credentialsparameter
            '''
            result = self._values.get("credentials_parameter")
            assert result is not None, "Required property 'credentials_parameter' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RepositoryCredentialsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.ResourceRequirementProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "value": "value"},
    )
    class ResourceRequirementProperty:
        def __init__(
            self,
            *,
            type: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The type and amount of a resource to assign to a container.

            The supported resources include ``GPU`` , ``MEMORY`` , and ``VCPU`` .

            :param type: The type of resource to assign to a container. The supported resources include ``GPU`` , ``MEMORY`` , and ``VCPU`` .
            :param value: The quantity of the specified resource to reserve for the container. The values vary based on the ``type`` specified. - **type="GPU"** - The number of physical GPUs to reserve for the container. Make sure that the number of GPUs reserved for all containers in a job doesn't exceed the number of available GPUs on the compute resource that the job is launched on. .. epigraph:: GPUs aren't available for jobs that are running on Fargate resources. - **type="MEMORY"** - The memory hard limit (in MiB) present to the container. This parameter is supported for jobs that are running on Amazon EC2 resources. If your container attempts to exceed the memory specified, the container is terminated. This parameter maps to ``Memory`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--memory`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . You must specify at least 4 MiB of memory for a job. This is required but can be specified in several places for multi-node parallel (MNP) jobs. It must be specified for each node at least once. This parameter maps to ``Memory`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--memory`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . .. epigraph:: If you're trying to maximize your resource utilization by providing your jobs as much memory as possible for a particular instance type, see `Memory management <https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html>`_ in the *AWS Batch User Guide* . For jobs that are running on Fargate resources, then ``value`` is the hard limit (in MiB), and must match one of the supported values and the ``VCPU`` values must be one of the values supported for that memory value. - **value = 512** - ``VCPU`` = 0.25 - **value = 1024** - ``VCPU`` = 0.25 or 0.5 - **value = 2048** - ``VCPU`` = 0.25, 0.5, or 1 - **value = 3072** - ``VCPU`` = 0.5, or 1 - **value = 4096** - ``VCPU`` = 0.5, 1, or 2 - **value = 5120, 6144, or 7168** - ``VCPU`` = 1 or 2 - **value = 8192** - ``VCPU`` = 1, 2, or 4 - **value = 9216, 10240, 11264, 12288, 13312, 14336, or 15360** - ``VCPU`` = 2 or 4 - **value = 16384** - ``VCPU`` = 2, 4, or 8 - **value = 17408, 18432, 19456, 21504, 22528, 23552, 25600, 26624, 27648, 29696, or 30720** - ``VCPU`` = 4 - **value = 20480, 24576, or 28672** - ``VCPU`` = 4 or 8 - **value = 36864, 45056, 53248, or 61440** - ``VCPU`` = 8 - **value = 32768, 40960, 49152, or 57344** - ``VCPU`` = 8 or 16 - **value = 65536, 73728, 81920, 90112, 98304, 106496, 114688, or 122880** - ``VCPU`` = 16 - **type="VCPU"** - The number of vCPUs reserved for the container. This parameter maps to ``CpuShares`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--cpu-shares`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . Each vCPU is equivalent to 1,024 CPU shares. For Amazon EC2 resources, you must specify at least one vCPU. This is required but can be specified in several places; it must be specified for each node at least once. The default for the Fargate On-Demand vCPU resource count quota is 6 vCPUs. For more information about Fargate quotas, see `AWS Fargate quotas <https://docs.aws.amazon.com/general/latest/gr/ecs-service.html#service-quotas-fargate>`_ in the *AWS General Reference* . For jobs that are running on Fargate resources, then ``value`` must match one of the supported values and the ``MEMORY`` values must be one of the values supported for that ``VCPU`` value. The supported values are 0.25, 0.5, 1, 2, 4, 8, and 16 - **value = 0.25** - ``MEMORY`` = 512, 1024, or 2048 - **value = 0.5** - ``MEMORY`` = 1024, 2048, 3072, or 4096 - **value = 1** - ``MEMORY`` = 2048, 3072, 4096, 5120, 6144, 7168, or 8192 - **value = 2** - ``MEMORY`` = 4096, 5120, 6144, 7168, 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, or 16384 - **value = 4** - ``MEMORY`` = 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, 16384, 17408, 18432, 19456, 20480, 21504, 22528, 23552, 24576, 25600, 26624, 27648, 28672, 29696, or 30720 - **value = 8** - ``MEMORY`` = 16384, 20480, 24576, 28672, 32768, 36864, 40960, 45056, 49152, 53248, 57344, or 61440 - **value = 16** - ``MEMORY`` = 32768, 40960, 49152, 57344, 65536, 73728, 81920, 90112, 98304, 106496, 114688, or 122880

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-resourcerequirement.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                resource_requirement_property = batch.CfnJobDefinition.ResourceRequirementProperty(
                    type="type",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7384d4b3b97f8aebbaa3dfe7ec74991276f6b53d6e1885662f675369f1d0166c)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if type is not None:
                self._values["type"] = type
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The type of resource to assign to a container.

            The supported resources include ``GPU`` , ``MEMORY`` , and ``VCPU`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-resourcerequirement.html#cfn-batch-jobdefinition-resourcerequirement-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The quantity of the specified resource to reserve for the container. The values vary based on the ``type`` specified.

            - **type="GPU"** - The number of physical GPUs to reserve for the container. Make sure that the number of GPUs reserved for all containers in a job doesn't exceed the number of available GPUs on the compute resource that the job is launched on.

            .. epigraph::

               GPUs aren't available for jobs that are running on Fargate resources.

            - **type="MEMORY"** - The memory hard limit (in MiB) present to the container. This parameter is supported for jobs that are running on Amazon EC2 resources. If your container attempts to exceed the memory specified, the container is terminated. This parameter maps to ``Memory`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--memory`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . You must specify at least 4 MiB of memory for a job. This is required but can be specified in several places for multi-node parallel (MNP) jobs. It must be specified for each node at least once. This parameter maps to ``Memory`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--memory`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .

            .. epigraph::

               If you're trying to maximize your resource utilization by providing your jobs as much memory as possible for a particular instance type, see `Memory management <https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html>`_ in the *AWS Batch User Guide* .

            For jobs that are running on Fargate resources, then ``value`` is the hard limit (in MiB), and must match one of the supported values and the ``VCPU`` values must be one of the values supported for that memory value.

            - **value = 512** - ``VCPU`` = 0.25
            - **value = 1024** - ``VCPU`` = 0.25 or 0.5
            - **value = 2048** - ``VCPU`` = 0.25, 0.5, or 1
            - **value = 3072** - ``VCPU`` = 0.5, or 1
            - **value = 4096** - ``VCPU`` = 0.5, 1, or 2
            - **value = 5120, 6144, or 7168** - ``VCPU`` = 1 or 2
            - **value = 8192** - ``VCPU`` = 1, 2, or 4
            - **value = 9216, 10240, 11264, 12288, 13312, 14336, or 15360** - ``VCPU`` = 2 or 4
            - **value = 16384** - ``VCPU`` = 2, 4, or 8
            - **value = 17408, 18432, 19456, 21504, 22528, 23552, 25600, 26624, 27648, 29696, or 30720** - ``VCPU`` = 4
            - **value = 20480, 24576, or 28672** - ``VCPU`` = 4 or 8
            - **value = 36864, 45056, 53248, or 61440** - ``VCPU`` = 8
            - **value = 32768, 40960, 49152, or 57344** - ``VCPU`` = 8 or 16
            - **value = 65536, 73728, 81920, 90112, 98304, 106496, 114688, or 122880** - ``VCPU`` = 16
            - **type="VCPU"** - The number of vCPUs reserved for the container. This parameter maps to ``CpuShares`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--cpu-shares`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . Each vCPU is equivalent to 1,024 CPU shares. For Amazon EC2 resources, you must specify at least one vCPU. This is required but can be specified in several places; it must be specified for each node at least once.

            The default for the Fargate On-Demand vCPU resource count quota is 6 vCPUs. For more information about Fargate quotas, see `AWS Fargate quotas <https://docs.aws.amazon.com/general/latest/gr/ecs-service.html#service-quotas-fargate>`_ in the *AWS General Reference* .

            For jobs that are running on Fargate resources, then ``value`` must match one of the supported values and the ``MEMORY`` values must be one of the values supported for that ``VCPU`` value. The supported values are 0.25, 0.5, 1, 2, 4, 8, and 16

            - **value = 0.25** - ``MEMORY`` = 512, 1024, or 2048
            - **value = 0.5** - ``MEMORY`` = 1024, 2048, 3072, or 4096
            - **value = 1** - ``MEMORY`` = 2048, 3072, 4096, 5120, 6144, 7168, or 8192
            - **value = 2** - ``MEMORY`` = 4096, 5120, 6144, 7168, 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, or 16384
            - **value = 4** - ``MEMORY`` = 8192, 9216, 10240, 11264, 12288, 13312, 14336, 15360, 16384, 17408, 18432, 19456, 20480, 21504, 22528, 23552, 24576, 25600, 26624, 27648, 28672, 29696, or 30720
            - **value = 8** - ``MEMORY`` = 16384, 20480, 24576, 28672, 32768, 36864, 40960, 45056, 49152, 53248, 57344, or 61440
            - **value = 16** - ``MEMORY`` = 32768, 40960, 49152, 57344, 65536, 73728, 81920, 90112, 98304, 106496, 114688, or 122880

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-resourcerequirement.html#cfn-batch-jobdefinition-resourcerequirement-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ResourceRequirementProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.ResourcesProperty",
        jsii_struct_bases=[],
        name_mapping={"limits": "limits", "requests": "requests"},
    )
    class ResourcesProperty:
        def __init__(
            self,
            *,
            limits: typing.Any = None,
            requests: typing.Any = None,
        ) -> None:
            '''
            :param limits: 
            :param requests: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-resources.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # limits: Any
                # requests: Any
                
                resources_property = batch.CfnJobDefinition.ResourcesProperty(
                    limits=limits,
                    requests=requests
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__634ef7ff057388be2736fb45bfab10d038183951137f893e49e433ed83008f6a)
                check_type(argname="argument limits", value=limits, expected_type=type_hints["limits"])
                check_type(argname="argument requests", value=requests, expected_type=type_hints["requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if limits is not None:
                self._values["limits"] = limits
            if requests is not None:
                self._values["requests"] = requests

        @builtins.property
        def limits(self) -> typing.Any:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-resources.html#cfn-batch-jobdefinition-resources-limits
            '''
            result = self._values.get("limits")
            return typing.cast(typing.Any, result)

        @builtins.property
        def requests(self) -> typing.Any:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-resources.html#cfn-batch-jobdefinition-resources-requests
            '''
            result = self._values.get("requests")
            return typing.cast(typing.Any, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ResourcesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.RetryStrategyProperty",
        jsii_struct_bases=[],
        name_mapping={"attempts": "attempts", "evaluate_on_exit": "evaluateOnExit"},
    )
    class RetryStrategyProperty:
        def __init__(
            self,
            *,
            attempts: typing.Optional[jsii.Number] = None,
            evaluate_on_exit: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EvaluateOnExitProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The retry strategy that's associated with a job.

            For more information, see `Automated job retries <https://docs.aws.amazon.com/batch/latest/userguide/job_retries.html>`_ in the *AWS Batch User Guide* .

            :param attempts: The number of times to move a job to the ``RUNNABLE`` status. You can specify between 1 and 10 attempts. If the value of ``attempts`` is greater than one, the job is retried on failure the same number of attempts as the value.
            :param evaluate_on_exit: Array of up to 5 objects that specify the conditions where jobs are retried or failed. If this parameter is specified, then the ``attempts`` parameter must also be specified. If none of the listed conditions match, then the job is retried.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-retrystrategy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                retry_strategy_property = batch.CfnJobDefinition.RetryStrategyProperty(
                    attempts=123,
                    evaluate_on_exit=[batch.CfnJobDefinition.EvaluateOnExitProperty(
                        action="action",
                
                        # the properties below are optional
                        on_exit_code="onExitCode",
                        on_reason="onReason",
                        on_status_reason="onStatusReason"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__800b22bca4d215e51bdb19588f0ed454bc972b986dd5a03f640fe8681acd52be)
                check_type(argname="argument attempts", value=attempts, expected_type=type_hints["attempts"])
                check_type(argname="argument evaluate_on_exit", value=evaluate_on_exit, expected_type=type_hints["evaluate_on_exit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if attempts is not None:
                self._values["attempts"] = attempts
            if evaluate_on_exit is not None:
                self._values["evaluate_on_exit"] = evaluate_on_exit

        @builtins.property
        def attempts(self) -> typing.Optional[jsii.Number]:
            '''The number of times to move a job to the ``RUNNABLE`` status.

            You can specify between 1 and 10 attempts. If the value of ``attempts`` is greater than one, the job is retried on failure the same number of attempts as the value.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-retrystrategy.html#cfn-batch-jobdefinition-retrystrategy-attempts
            '''
            result = self._values.get("attempts")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def evaluate_on_exit(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EvaluateOnExitProperty"]]]]:
            '''Array of up to 5 objects that specify the conditions where jobs are retried or failed.

            If this parameter is specified, then the ``attempts`` parameter must also be specified. If none of the listed conditions match, then the job is retried.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-retrystrategy.html#cfn-batch-jobdefinition-retrystrategy-evaluateonexit
            '''
            result = self._values.get("evaluate_on_exit")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EvaluateOnExitProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RetryStrategyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.RuntimePlatformProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cpu_architecture": "cpuArchitecture",
            "operating_system_family": "operatingSystemFamily",
        },
    )
    class RuntimePlatformProperty:
        def __init__(
            self,
            *,
            cpu_architecture: typing.Optional[builtins.str] = None,
            operating_system_family: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the compute environment architecture for AWS Batch jobs on Fargate.

            :param cpu_architecture: The vCPU architecture. The default value is ``X86_64`` . Valid values are ``X86_64`` and ``ARM64`` . .. epigraph:: This parameter must be set to ``X86_64`` for Windows containers. > Fargate Spot is not supported for ``ARM64`` and Windows-based containers on Fargate. A job queue will be blocked if a Fargate ``ARM64`` or Windows job is submitted to a job queue with only Fargate Spot compute environments. However, you can attach both ``FARGATE`` and ``FARGATE_SPOT`` compute environments to the same job queue.
            :param operating_system_family: The operating system for the compute environment. Valid values are: ``LINUX`` (default), ``WINDOWS_SERVER_2019_CORE`` , ``WINDOWS_SERVER_2019_FULL`` , ``WINDOWS_SERVER_2022_CORE`` , and ``WINDOWS_SERVER_2022_FULL`` . .. epigraph:: The following parameters can’t be set for Windows containers: ``linuxParameters`` , ``privileged`` , ``user`` , ``ulimits`` , ``readonlyRootFilesystem`` , and ``efsVolumeConfiguration`` . > The AWS Batch Scheduler checks the compute environments that are attached to the job queue before registering a task definition with Fargate. In this scenario, the job queue is where the job is submitted. If the job requires a Windows container and the first compute environment is ``LINUX`` , the compute environment is skipped and the next compute environment is checked until a Windows-based compute environment is found. > Fargate Spot is not supported for ``ARM64`` and Windows-based containers on Fargate. A job queue will be blocked if a Fargate ``ARM64`` or Windows job is submitted to a job queue with only Fargate Spot compute environments. However, you can attach both ``FARGATE`` and ``FARGATE_SPOT`` compute environments to the same job queue.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-runtimeplatform.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                runtime_platform_property = batch.CfnJobDefinition.RuntimePlatformProperty(
                    cpu_architecture="cpuArchitecture",
                    operating_system_family="operatingSystemFamily"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fbdffb76b27dbbdbbc0d2ebc2229966bd04e50d9dbe061567a0dba94004d9e3e)
                check_type(argname="argument cpu_architecture", value=cpu_architecture, expected_type=type_hints["cpu_architecture"])
                check_type(argname="argument operating_system_family", value=operating_system_family, expected_type=type_hints["operating_system_family"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cpu_architecture is not None:
                self._values["cpu_architecture"] = cpu_architecture
            if operating_system_family is not None:
                self._values["operating_system_family"] = operating_system_family

        @builtins.property
        def cpu_architecture(self) -> typing.Optional[builtins.str]:
            '''The vCPU architecture. The default value is ``X86_64`` . Valid values are ``X86_64`` and ``ARM64`` .

            .. epigraph::

               This parameter must be set to ``X86_64`` for Windows containers. > Fargate Spot is not supported for ``ARM64`` and Windows-based containers on Fargate. A job queue will be blocked if a Fargate ``ARM64`` or Windows job is submitted to a job queue with only Fargate Spot compute environments. However, you can attach both ``FARGATE`` and ``FARGATE_SPOT`` compute environments to the same job queue.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-runtimeplatform.html#cfn-batch-jobdefinition-runtimeplatform-cpuarchitecture
            '''
            result = self._values.get("cpu_architecture")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def operating_system_family(self) -> typing.Optional[builtins.str]:
            '''The operating system for the compute environment.

            Valid values are: ``LINUX`` (default), ``WINDOWS_SERVER_2019_CORE`` , ``WINDOWS_SERVER_2019_FULL`` , ``WINDOWS_SERVER_2022_CORE`` , and ``WINDOWS_SERVER_2022_FULL`` .
            .. epigraph::

               The following parameters can’t be set for Windows containers: ``linuxParameters`` , ``privileged`` , ``user`` , ``ulimits`` , ``readonlyRootFilesystem`` , and ``efsVolumeConfiguration`` . > The AWS Batch Scheduler checks the compute environments that are attached to the job queue before registering a task definition with Fargate. In this scenario, the job queue is where the job is submitted. If the job requires a Windows container and the first compute environment is ``LINUX`` , the compute environment is skipped and the next compute environment is checked until a Windows-based compute environment is found. > Fargate Spot is not supported for ``ARM64`` and Windows-based containers on Fargate. A job queue will be blocked if a Fargate ``ARM64`` or Windows job is submitted to a job queue with only Fargate Spot compute environments. However, you can attach both ``FARGATE`` and ``FARGATE_SPOT`` compute environments to the same job queue.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-runtimeplatform.html#cfn-batch-jobdefinition-runtimeplatform-operatingsystemfamily
            '''
            result = self._values.get("operating_system_family")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RuntimePlatformProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.SecretProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value_from": "valueFrom"},
    )
    class SecretProperty:
        def __init__(self, *, name: builtins.str, value_from: builtins.str) -> None:
            '''An object that represents the secret to expose to your container.

            Secrets can be exposed to a container in the following ways:

            - To inject sensitive data into your containers as environment variables, use the ``secrets`` container definition parameter.
            - To reference sensitive information in the log configuration of a container, use the ``secretOptions`` container definition parameter.

            For more information, see `Specifying sensitive data <https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html>`_ in the *AWS Batch User Guide* .

            :param name: The name of the secret.
            :param value_from: The secret to expose to the container. The supported values are either the full Amazon Resource Name (ARN) of the AWS Secrets Manager secret or the full ARN of the parameter in the AWS Systems Manager Parameter Store. .. epigraph:: If the AWS Systems Manager Parameter Store parameter exists in the same Region as the job you're launching, then you can use either the full Amazon Resource Name (ARN) or name of the parameter. If the parameter exists in a different Region, then the full ARN must be specified.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-secret.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                secret_property = batch.CfnJobDefinition.SecretProperty(
                    name="name",
                    value_from="valueFrom"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__627c25c89399f4c648455cf23b1833f2f5961be5393bea72498980b6dd1b8fac)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument value_from", value=value_from, expected_type=type_hints["value_from"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "value_from": value_from,
            }

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the secret.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-secret.html#cfn-batch-jobdefinition-secret-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value_from(self) -> builtins.str:
            '''The secret to expose to the container.

            The supported values are either the full Amazon Resource Name (ARN) of the AWS Secrets Manager secret or the full ARN of the parameter in the AWS Systems Manager Parameter Store.
            .. epigraph::

               If the AWS Systems Manager Parameter Store parameter exists in the same Region as the job you're launching, then you can use either the full Amazon Resource Name (ARN) or name of the parameter. If the parameter exists in a different Region, then the full ARN must be specified.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-secret.html#cfn-batch-jobdefinition-secret-valuefrom
            '''
            result = self._values.get("value_from")
            assert result is not None, "Required property 'value_from' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SecretProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.SecurityContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "allow_privilege_escalation": "allowPrivilegeEscalation",
            "privileged": "privileged",
            "read_only_root_filesystem": "readOnlyRootFilesystem",
            "run_as_group": "runAsGroup",
            "run_as_non_root": "runAsNonRoot",
            "run_as_user": "runAsUser",
        },
    )
    class SecurityContextProperty:
        def __init__(
            self,
            *,
            allow_privilege_escalation: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            privileged: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            read_only_root_filesystem: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            run_as_group: typing.Optional[jsii.Number] = None,
            run_as_non_root: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            run_as_user: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param allow_privilege_escalation: 
            :param privileged: 
            :param read_only_root_filesystem: 
            :param run_as_group: 
            :param run_as_non_root: 
            :param run_as_user: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-securitycontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                security_context_property = batch.CfnJobDefinition.SecurityContextProperty(
                    allow_privilege_escalation=False,
                    privileged=False,
                    read_only_root_filesystem=False,
                    run_as_group=123,
                    run_as_non_root=False,
                    run_as_user=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ecf4bfd8ece80e1bc3ee0536353fc145f3c5b08f62a604a53b62e7d2526f9d0e)
                check_type(argname="argument allow_privilege_escalation", value=allow_privilege_escalation, expected_type=type_hints["allow_privilege_escalation"])
                check_type(argname="argument privileged", value=privileged, expected_type=type_hints["privileged"])
                check_type(argname="argument read_only_root_filesystem", value=read_only_root_filesystem, expected_type=type_hints["read_only_root_filesystem"])
                check_type(argname="argument run_as_group", value=run_as_group, expected_type=type_hints["run_as_group"])
                check_type(argname="argument run_as_non_root", value=run_as_non_root, expected_type=type_hints["run_as_non_root"])
                check_type(argname="argument run_as_user", value=run_as_user, expected_type=type_hints["run_as_user"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if allow_privilege_escalation is not None:
                self._values["allow_privilege_escalation"] = allow_privilege_escalation
            if privileged is not None:
                self._values["privileged"] = privileged
            if read_only_root_filesystem is not None:
                self._values["read_only_root_filesystem"] = read_only_root_filesystem
            if run_as_group is not None:
                self._values["run_as_group"] = run_as_group
            if run_as_non_root is not None:
                self._values["run_as_non_root"] = run_as_non_root
            if run_as_user is not None:
                self._values["run_as_user"] = run_as_user

        @builtins.property
        def allow_privilege_escalation(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-securitycontext.html#cfn-batch-jobdefinition-securitycontext-allowprivilegeescalation
            '''
            result = self._values.get("allow_privilege_escalation")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def privileged(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-securitycontext.html#cfn-batch-jobdefinition-securitycontext-privileged
            '''
            result = self._values.get("privileged")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def read_only_root_filesystem(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-securitycontext.html#cfn-batch-jobdefinition-securitycontext-readonlyrootfilesystem
            '''
            result = self._values.get("read_only_root_filesystem")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def run_as_group(self) -> typing.Optional[jsii.Number]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-securitycontext.html#cfn-batch-jobdefinition-securitycontext-runasgroup
            '''
            result = self._values.get("run_as_group")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def run_as_non_root(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-securitycontext.html#cfn-batch-jobdefinition-securitycontext-runasnonroot
            '''
            result = self._values.get("run_as_non_root")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def run_as_user(self) -> typing.Optional[jsii.Number]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-securitycontext.html#cfn-batch-jobdefinition-securitycontext-runasuser
            '''
            result = self._values.get("run_as_user")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SecurityContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.TaskContainerDependencyProperty",
        jsii_struct_bases=[],
        name_mapping={"condition": "condition", "container_name": "containerName"},
    )
    class TaskContainerDependencyProperty:
        def __init__(
            self,
            *,
            condition: builtins.str,
            container_name: builtins.str,
        ) -> None:
            '''A list of containers that this task depends on.

            :param condition: The dependency condition of the container. The following are the available conditions and their behavior:. - ``START`` - This condition emulates the behavior of links and volumes today. It validates that a dependent container is started before permitting other containers to start. - ``COMPLETE`` - This condition validates that a dependent container runs to completion (exits) before permitting other containers to start. This can be useful for nonessential containers that run a script and then exit. This condition can't be set on an essential container. - ``SUCCESS`` - This condition is the same as ``COMPLETE`` , but it also requires that the container exits with a zero status. This condition can't be set on an essential container.
            :param container_name: A unique identifier for the container.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerdependency.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                task_container_dependency_property = batch.CfnJobDefinition.TaskContainerDependencyProperty(
                    condition="condition",
                    container_name="containerName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2f29cae294a8ef5e1db972a16fcfa25d5a4a34ac0b2cb14531b35c42dbf2d4e1)
                check_type(argname="argument condition", value=condition, expected_type=type_hints["condition"])
                check_type(argname="argument container_name", value=container_name, expected_type=type_hints["container_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "condition": condition,
                "container_name": container_name,
            }

        @builtins.property
        def condition(self) -> builtins.str:
            '''The dependency condition of the container. The following are the available conditions and their behavior:.

            - ``START`` - This condition emulates the behavior of links and volumes today. It validates that a dependent container is started before permitting other containers to start.
            - ``COMPLETE`` - This condition validates that a dependent container runs to completion (exits) before permitting other containers to start. This can be useful for nonessential containers that run a script and then exit. This condition can't be set on an essential container.
            - ``SUCCESS`` - This condition is the same as ``COMPLETE`` , but it also requires that the container exits with a zero status. This condition can't be set on an essential container.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerdependency.html#cfn-batch-jobdefinition-taskcontainerdependency-condition
            '''
            result = self._values.get("condition")
            assert result is not None, "Required property 'condition' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def container_name(self) -> builtins.str:
            '''A unique identifier for the container.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerdependency.html#cfn-batch-jobdefinition-taskcontainerdependency-containername
            '''
            result = self._values.get("container_name")
            assert result is not None, "Required property 'container_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TaskContainerDependencyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.TaskContainerPropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "image": "image",
            "command": "command",
            "depends_on": "dependsOn",
            "environment": "environment",
            "essential": "essential",
            "linux_parameters": "linuxParameters",
            "log_configuration": "logConfiguration",
            "mount_points": "mountPoints",
            "name": "name",
            "privileged": "privileged",
            "readonly_root_filesystem": "readonlyRootFilesystem",
            "repository_credentials": "repositoryCredentials",
            "resource_requirements": "resourceRequirements",
            "secrets": "secrets",
            "ulimits": "ulimits",
            "user": "user",
        },
    )
    class TaskContainerPropertiesProperty:
        def __init__(
            self,
            *,
            image: builtins.str,
            command: typing.Optional[typing.Sequence[builtins.str]] = None,
            depends_on: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.TaskContainerDependencyProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            environment: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EnvironmentProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            essential: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            linux_parameters: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.LinuxParametersProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            log_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.LogConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            mount_points: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.MountPointProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            name: typing.Optional[builtins.str] = None,
            privileged: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            readonly_root_filesystem: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            repository_credentials: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.RepositoryCredentialsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            resource_requirements: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.ResourceRequirementProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            secrets: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.SecretProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            ulimits: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.UlimitProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            user: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Container properties are used for Amazon ECS-based job definitions.

            These properties to describe the container that's launched as part of a job.

            :param image: The image used to start a container. This string is passed directly to the Docker daemon. By default, images in the Docker Hub registry are available. Other repositories are specified with either ``repository-url/image:tag`` or ``repository-url/image@digest`` . Up to 255 letters (uppercase and lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs are allowed. This parameter maps to ``Image`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``IMAGE`` parameter of the `*docker run* <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            :param command: The command that's passed to the container. This parameter maps to ``Cmd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``COMMAND`` parameter to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . For more information, see `Dockerfile reference: CMD <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#cmd>`_ .
            :param depends_on: A list of containers that this container depends on.
            :param environment: The environment variables to pass to a container. This parameter maps to Env in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--env`` parameter to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . .. epigraph:: We don't recommend using plaintext environment variables for sensitive information, such as credential data. > Environment variables cannot start with ``AWS_BATCH`` . This naming convention is reserved for variables that AWS Batch sets.
            :param essential: If the essential parameter of a container is marked as ``true`` , and that container fails or stops for any reason, all other containers that are part of the task are stopped. If the ``essential`` parameter of a container is marked as false, its failure doesn't affect the rest of the containers in a task. If this parameter is omitted, a container is assumed to be essential. All jobs must have at least one essential container. If you have an application that's composed of multiple containers, group containers that are used for a common purpose into components, and separate the different components into multiple task definitions. For more information, see `Application Architecture <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param linux_parameters: Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information, see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_ .
            :param log_configuration: The log configuration specification for the container. This parameter maps to ``LogConfig`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--log-driver`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . By default, containers use the same logging driver that the Docker daemon uses. However the container can use a different logging driver than the Docker daemon by specifying a log driver with this parameter in the container definition. To use a different logging driver for a container, the log system must be configured properly on the container instance (or on a different log server for remote logging options). For more information about the options for different supported log drivers, see `Configure logging drivers <https://docs.aws.amazon.com/https://docs.docker.com/engine/admin/logging/overview/>`_ in the *Docker documentation* . .. epigraph:: Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the ``LogConfiguration`` data type). Additional log drivers may be available in future releases of the Amazon ECS container agent. This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: sudo docker version ``--format '{{.Server.APIVersion}}'`` .. epigraph:: The Amazon ECS container agent running on a container instance must register the logging drivers available on that instance with the ``ECS_AVAILABLE_LOGGING_DRIVERS`` environment variable before containers placed on that instance can use these log configuration options. For more information, see `Amazon ECS container agent configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param mount_points: The mount points for data volumes in your container. This parameter maps to ``Volumes`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the `--volume <https://docs.aws.amazon.com/>`_ option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . Windows containers can mount whole directories on the same drive as ``$env:ProgramData`` . Windows containers can't mount directories on a different drive, and mount point can't be across drives.
            :param name: The name of a container. The name can be used as a unique identifier to target your ``dependsOn`` and ``Overrides`` objects.
            :param privileged: When this parameter is ``true`` , the container is given elevated privileges on the host container instance (similar to the ``root`` user). This parameter maps to ``Privileged`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--privileged`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers or tasks run on Fargate.
            :param readonly_root_filesystem: When this parameter is true, the container is given read-only access to its root file system. This parameter maps to ``ReadonlyRootfs`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--read-only`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers.
            :param repository_credentials: The private repository authentication credentials to use.
            :param resource_requirements: The type and amount of a resource to assign to a container. The only supported resource is a GPU.
            :param secrets: The secrets to pass to the container. For more information, see `Specifying Sensitive Data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the Amazon Elastic Container Service Developer Guide.
            :param ulimits: A list of ``ulimits`` to set in the container. If a ``ulimit`` value is specified in a task definition, it overrides the default values set by Docker. This parameter maps to ``Ulimits`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--ulimit`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with the exception of the nofile resource limit parameter which Fargate overrides. The ``nofile`` resource limit sets a restriction on the number of open files that a container can use. The default ``nofile`` soft limit is ``1024`` and the default hard limit is ``65535`` . This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: sudo docker version ``--format '{{.Server.APIVersion}}'`` .. epigraph:: This parameter is not supported for Windows containers.
            :param user: The user to use inside the container. This parameter maps to User in the Create a container section of the Docker Remote API and the --user option to docker run. .. epigraph:: When running tasks using the ``host`` network mode, don't run containers using the ``root user (UID 0)`` . We recommend using a non-root user for better security. You can specify the ``user`` using the following formats. If specifying a UID or GID, you must specify it as a positive integer. - ``user`` - ``user:group`` - ``uid`` - ``uid:gid`` - ``user:gi`` - ``uid:group`` .. epigraph:: This parameter is not supported for Windows containers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                # options: Any
                
                task_container_properties_property = batch.CfnJobDefinition.TaskContainerPropertiesProperty(
                    image="image",
                
                    # the properties below are optional
                    command=["command"],
                    depends_on=[batch.CfnJobDefinition.TaskContainerDependencyProperty(
                        condition="condition",
                        container_name="containerName"
                    )],
                    environment=[batch.CfnJobDefinition.EnvironmentProperty(
                        name="name",
                        value="value"
                    )],
                    essential=False,
                    linux_parameters=batch.CfnJobDefinition.LinuxParametersProperty(
                        devices=[batch.CfnJobDefinition.DeviceProperty(
                            container_path="containerPath",
                            host_path="hostPath",
                            permissions=["permissions"]
                        )],
                        init_process_enabled=False,
                        max_swap=123,
                        shared_memory_size=123,
                        swappiness=123,
                        tmpfs=[batch.CfnJobDefinition.TmpfsProperty(
                            container_path="containerPath",
                            size=123,
                
                            # the properties below are optional
                            mount_options=["mountOptions"]
                        )]
                    ),
                    log_configuration=batch.CfnJobDefinition.LogConfigurationProperty(
                        log_driver="logDriver",
                
                        # the properties below are optional
                        options=options,
                        secret_options=[batch.CfnJobDefinition.SecretProperty(
                            name="name",
                            value_from="valueFrom"
                        )]
                    ),
                    mount_points=[batch.CfnJobDefinition.MountPointProperty(
                        container_path="containerPath",
                        read_only=False,
                        source_volume="sourceVolume"
                    )],
                    name="name",
                    privileged=False,
                    readonly_root_filesystem=False,
                    repository_credentials=batch.CfnJobDefinition.RepositoryCredentialsProperty(
                        credentials_parameter="credentialsParameter"
                    ),
                    resource_requirements=[batch.CfnJobDefinition.ResourceRequirementProperty(
                        type="type",
                        value="value"
                    )],
                    secrets=[batch.CfnJobDefinition.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )],
                    ulimits=[batch.CfnJobDefinition.UlimitProperty(
                        hard_limit=123,
                        name="name",
                        soft_limit=123
                    )],
                    user="user"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4946f5f042ef067bff74424b1b71c92ab13e52a7f47a4271e028110521b8a1ae)
                check_type(argname="argument image", value=image, expected_type=type_hints["image"])
                check_type(argname="argument command", value=command, expected_type=type_hints["command"])
                check_type(argname="argument depends_on", value=depends_on, expected_type=type_hints["depends_on"])
                check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
                check_type(argname="argument essential", value=essential, expected_type=type_hints["essential"])
                check_type(argname="argument linux_parameters", value=linux_parameters, expected_type=type_hints["linux_parameters"])
                check_type(argname="argument log_configuration", value=log_configuration, expected_type=type_hints["log_configuration"])
                check_type(argname="argument mount_points", value=mount_points, expected_type=type_hints["mount_points"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument privileged", value=privileged, expected_type=type_hints["privileged"])
                check_type(argname="argument readonly_root_filesystem", value=readonly_root_filesystem, expected_type=type_hints["readonly_root_filesystem"])
                check_type(argname="argument repository_credentials", value=repository_credentials, expected_type=type_hints["repository_credentials"])
                check_type(argname="argument resource_requirements", value=resource_requirements, expected_type=type_hints["resource_requirements"])
                check_type(argname="argument secrets", value=secrets, expected_type=type_hints["secrets"])
                check_type(argname="argument ulimits", value=ulimits, expected_type=type_hints["ulimits"])
                check_type(argname="argument user", value=user, expected_type=type_hints["user"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "image": image,
            }
            if command is not None:
                self._values["command"] = command
            if depends_on is not None:
                self._values["depends_on"] = depends_on
            if environment is not None:
                self._values["environment"] = environment
            if essential is not None:
                self._values["essential"] = essential
            if linux_parameters is not None:
                self._values["linux_parameters"] = linux_parameters
            if log_configuration is not None:
                self._values["log_configuration"] = log_configuration
            if mount_points is not None:
                self._values["mount_points"] = mount_points
            if name is not None:
                self._values["name"] = name
            if privileged is not None:
                self._values["privileged"] = privileged
            if readonly_root_filesystem is not None:
                self._values["readonly_root_filesystem"] = readonly_root_filesystem
            if repository_credentials is not None:
                self._values["repository_credentials"] = repository_credentials
            if resource_requirements is not None:
                self._values["resource_requirements"] = resource_requirements
            if secrets is not None:
                self._values["secrets"] = secrets
            if ulimits is not None:
                self._values["ulimits"] = ulimits
            if user is not None:
                self._values["user"] = user

        @builtins.property
        def image(self) -> builtins.str:
            '''The image used to start a container.

            This string is passed directly to the Docker daemon. By default, images in the Docker Hub registry are available. Other repositories are specified with either ``repository-url/image:tag`` or ``repository-url/image@digest`` . Up to 255 letters (uppercase and lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs are allowed. This parameter maps to ``Image`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``IMAGE`` parameter of the `*docker run* <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-image
            '''
            result = self._values.get("image")
            assert result is not None, "Required property 'image' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def command(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The command that's passed to the container.

            This parameter maps to ``Cmd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``COMMAND`` parameter to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . For more information, see `Dockerfile reference: CMD <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#cmd>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-command
            '''
            result = self._values.get("command")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def depends_on(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.TaskContainerDependencyProperty"]]]]:
            '''A list of containers that this container depends on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-dependson
            '''
            result = self._values.get("depends_on")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.TaskContainerDependencyProperty"]]]], result)

        @builtins.property
        def environment(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EnvironmentProperty"]]]]:
            '''The environment variables to pass to a container.

            This parameter maps to Env in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/#create-a-container>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.23/>`_ and the ``--env`` parameter to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ .
            .. epigraph::

               We don't recommend using plaintext environment variables for sensitive information, such as credential data. > Environment variables cannot start with ``AWS_BATCH`` . This naming convention is reserved for variables that AWS Batch sets.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-environment
            '''
            result = self._values.get("environment")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EnvironmentProperty"]]]], result)

        @builtins.property
        def essential(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''If the essential parameter of a container is marked as ``true`` , and that container fails or stops for any reason, all other containers that are part of the task are stopped.

            If the ``essential`` parameter of a container is marked as false, its failure doesn't affect the rest of the containers in a task. If this parameter is omitted, a container is assumed to be essential.

            All jobs must have at least one essential container. If you have an application that's composed of multiple containers, group containers that are used for a common purpose into components, and separate the different components into multiple task definitions. For more information, see `Application Architecture <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-essential
            '''
            result = self._values.get("essential")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def linux_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.LinuxParametersProperty"]]:
            '''Linux-specific modifications that are applied to the container, such as Linux kernel capabilities.

            For more information, see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-linuxparameters
            '''
            result = self._values.get("linux_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.LinuxParametersProperty"]], result)

        @builtins.property
        def log_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.LogConfigurationProperty"]]:
            '''The log configuration specification for the container.

            This parameter maps to ``LogConfig`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--log-driver`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            By default, containers use the same logging driver that the Docker daemon uses. However the container can use a different logging driver than the Docker daemon by specifying a log driver with this parameter in the container definition. To use a different logging driver for a container, the log system must be configured properly on the container instance (or on a different log server for remote logging options). For more information about the options for different supported log drivers, see `Configure logging drivers <https://docs.aws.amazon.com/https://docs.docker.com/engine/admin/logging/overview/>`_ in the *Docker documentation* .
            .. epigraph::

               Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the ``LogConfiguration`` data type). Additional log drivers may be available in future releases of the Amazon ECS container agent.

            This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: sudo docker version ``--format '{{.Server.APIVersion}}'``
            .. epigraph::

               The Amazon ECS container agent running on a container instance must register the logging drivers available on that instance with the ``ECS_AVAILABLE_LOGGING_DRIVERS`` environment variable before containers placed on that instance can use these log configuration options. For more information, see `Amazon ECS container agent configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-logconfiguration
            '''
            result = self._values.get("log_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.LogConfigurationProperty"]], result)

        @builtins.property
        def mount_points(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.MountPointProperty"]]]]:
            '''The mount points for data volumes in your container.

            This parameter maps to ``Volumes`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the `--volume <https://docs.aws.amazon.com/>`_ option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            Windows containers can mount whole directories on the same drive as ``$env:ProgramData`` . Windows containers can't mount directories on a different drive, and mount point can't be across drives.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-mountpoints
            '''
            result = self._values.get("mount_points")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.MountPointProperty"]]]], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of a container.

            The name can be used as a unique identifier to target your ``dependsOn`` and ``Overrides`` objects.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def privileged(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''When this parameter is ``true`` , the container is given elevated privileges on the host container instance (similar to the ``root`` user).

            This parameter maps to ``Privileged`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--privileged`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers or tasks run on Fargate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-privileged
            '''
            result = self._values.get("privileged")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def readonly_root_filesystem(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''When this parameter is true, the container is given read-only access to its root file system.

            This parameter maps to ``ReadonlyRootfs`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--read-only`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-readonlyrootfilesystem
            '''
            result = self._values.get("readonly_root_filesystem")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def repository_credentials(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.RepositoryCredentialsProperty"]]:
            '''The private repository authentication credentials to use.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-repositorycredentials
            '''
            result = self._values.get("repository_credentials")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.RepositoryCredentialsProperty"]], result)

        @builtins.property
        def resource_requirements(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ResourceRequirementProperty"]]]]:
            '''The type and amount of a resource to assign to a container.

            The only supported resource is a GPU.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-resourcerequirements
            '''
            result = self._values.get("resource_requirements")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.ResourceRequirementProperty"]]]], result)

        @builtins.property
        def secrets(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.SecretProperty"]]]]:
            '''The secrets to pass to the container.

            For more information, see `Specifying Sensitive Data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the Amazon Elastic Container Service Developer Guide.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-secrets
            '''
            result = self._values.get("secrets")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.SecretProperty"]]]], result)

        @builtins.property
        def ulimits(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.UlimitProperty"]]]]:
            '''A list of ``ulimits`` to set in the container.

            If a ``ulimit`` value is specified in a task definition, it overrides the default values set by Docker. This parameter maps to ``Ulimits`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--ulimit`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with the exception of the nofile resource limit parameter which Fargate overrides. The ``nofile`` resource limit sets a restriction on the number of open files that a container can use. The default ``nofile`` soft limit is ``1024`` and the default hard limit is ``65535`` .

            This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: sudo docker version ``--format '{{.Server.APIVersion}}'``
            .. epigraph::

               This parameter is not supported for Windows containers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-ulimits
            '''
            result = self._values.get("ulimits")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.UlimitProperty"]]]], result)

        @builtins.property
        def user(self) -> typing.Optional[builtins.str]:
            '''The user to use inside the container.

            This parameter maps to User in the Create a container section of the Docker Remote API and the --user option to docker run.
            .. epigraph::

               When running tasks using the ``host`` network mode, don't run containers using the ``root user (UID 0)`` . We recommend using a non-root user for better security.

            You can specify the ``user`` using the following formats. If specifying a UID or GID, you must specify it as a positive integer.

            - ``user``
            - ``user:group``
            - ``uid``
            - ``uid:gid``
            - ``user:gi``
            - ``uid:group``

            .. epigraph::

               This parameter is not supported for Windows containers.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-taskcontainerproperties.html#cfn-batch-jobdefinition-taskcontainerproperties-user
            '''
            result = self._values.get("user")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TaskContainerPropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.TimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"attempt_duration_seconds": "attemptDurationSeconds"},
    )
    class TimeoutProperty:
        def __init__(
            self,
            *,
            attempt_duration_seconds: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param attempt_duration_seconds: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-timeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                timeout_property = batch.CfnJobDefinition.TimeoutProperty(
                    attempt_duration_seconds=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5b035a85f593ff8b58d3fbc57a8614a7bc0d733c70f8504eab630022b6186187)
                check_type(argname="argument attempt_duration_seconds", value=attempt_duration_seconds, expected_type=type_hints["attempt_duration_seconds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if attempt_duration_seconds is not None:
                self._values["attempt_duration_seconds"] = attempt_duration_seconds

        @builtins.property
        def attempt_duration_seconds(self) -> typing.Optional[jsii.Number]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-timeout.html#cfn-batch-jobdefinition-timeout-attemptdurationseconds
            '''
            result = self._values.get("attempt_duration_seconds")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.TmpfsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_path": "containerPath",
            "size": "size",
            "mount_options": "mountOptions",
        },
    )
    class TmpfsProperty:
        def __init__(
            self,
            *,
            container_path: builtins.str,
            size: jsii.Number,
            mount_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The container path, mount options, and size of the ``tmpfs`` mount.

            .. epigraph::

               This object isn't applicable to jobs that are running on Fargate resources.

            :param container_path: The absolute file path in the container where the ``tmpfs`` volume is mounted.
            :param size: The size (in MiB) of the ``tmpfs`` volume.
            :param mount_options: The list of ``tmpfs`` volume mount options. Valid values: " ``defaults`` " | " ``ro`` " | " ``rw`` " | " ``suid`` " | " ``nosuid`` " | " ``dev`` " | " ``nodev`` " | " ``exec`` " | " ``noexec`` " | " ``sync`` " | " ``async`` " | " ``dirsync`` " | " ``remount`` " | " ``mand`` " | " ``nomand`` " | " ``atime`` " | " ``noatime`` " | " ``diratime`` " | " ``nodiratime`` " | " ``bind`` " | " ``rbind" | "unbindable" | "runbindable" | "private" | "rprivate" | "shared" | "rshared" | "slave" | "rslave" | "relatime`` " | " ``norelatime`` " | " ``strictatime`` " | " ``nostrictatime`` " | " ``mode`` " | " ``uid`` " | " ``gid`` " | " ``nr_inodes`` " | " ``nr_blocks`` " | " ``mpol`` "

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-tmpfs.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                tmpfs_property = batch.CfnJobDefinition.TmpfsProperty(
                    container_path="containerPath",
                    size=123,
                
                    # the properties below are optional
                    mount_options=["mountOptions"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7c16bfb0295986db06cd649d9ca6994f1d86d2d884185ad1eda92e6da68041e0)
                check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
                check_type(argname="argument size", value=size, expected_type=type_hints["size"])
                check_type(argname="argument mount_options", value=mount_options, expected_type=type_hints["mount_options"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "container_path": container_path,
                "size": size,
            }
            if mount_options is not None:
                self._values["mount_options"] = mount_options

        @builtins.property
        def container_path(self) -> builtins.str:
            '''The absolute file path in the container where the ``tmpfs`` volume is mounted.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-tmpfs.html#cfn-batch-jobdefinition-tmpfs-containerpath
            '''
            result = self._values.get("container_path")
            assert result is not None, "Required property 'container_path' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def size(self) -> jsii.Number:
            '''The size (in MiB) of the ``tmpfs`` volume.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-tmpfs.html#cfn-batch-jobdefinition-tmpfs-size
            '''
            result = self._values.get("size")
            assert result is not None, "Required property 'size' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def mount_options(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The list of ``tmpfs`` volume mount options.

            Valid values: " ``defaults`` " | " ``ro`` " | " ``rw`` " | " ``suid`` " | " ``nosuid`` " | " ``dev`` " | " ``nodev`` " | " ``exec`` " | " ``noexec`` " | " ``sync`` " | " ``async`` " | " ``dirsync`` " | " ``remount`` " | " ``mand`` " | " ``nomand`` " | " ``atime`` " | " ``noatime`` " | " ``diratime`` " | " ``nodiratime`` " | " ``bind`` " | " ``rbind" | "unbindable" | "runbindable" | "private" | "rprivate" | "shared" | "rshared" | "slave" | "rslave" | "relatime`` " | " ``norelatime`` " | " ``strictatime`` " | " ``nostrictatime`` " | " ``mode`` " | " ``uid`` " | " ``gid`` " | " ``nr_inodes`` " | " ``nr_blocks`` " | " ``mpol`` "

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-tmpfs.html#cfn-batch-jobdefinition-tmpfs-mountoptions
            '''
            result = self._values.get("mount_options")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TmpfsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.UlimitProperty",
        jsii_struct_bases=[],
        name_mapping={
            "hard_limit": "hardLimit",
            "name": "name",
            "soft_limit": "softLimit",
        },
    )
    class UlimitProperty:
        def __init__(
            self,
            *,
            hard_limit: jsii.Number,
            name: builtins.str,
            soft_limit: jsii.Number,
        ) -> None:
            '''The ``ulimit`` settings to pass to the container. For more information, see `Ulimit <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html>`_ .

            .. epigraph::

               This object isn't applicable to jobs that are running on Fargate resources.

            :param hard_limit: The hard limit for the ``ulimit`` type.
            :param name: The ``type`` of the ``ulimit`` . Valid values are: ``core`` | ``cpu`` | ``data`` | ``fsize`` | ``locks`` | ``memlock`` | ``msgqueue`` | ``nice`` | ``nofile`` | ``nproc`` | ``rss`` | ``rtprio`` | ``rttime`` | ``sigpending`` | ``stack`` .
            :param soft_limit: The soft limit for the ``ulimit`` type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ulimit.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                ulimit_property = batch.CfnJobDefinition.UlimitProperty(
                    hard_limit=123,
                    name="name",
                    soft_limit=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9422053e6f2d5b223ad6edca9c1d424075ea7f34d3f55f01ed34ea8f4b51e8d4)
                check_type(argname="argument hard_limit", value=hard_limit, expected_type=type_hints["hard_limit"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument soft_limit", value=soft_limit, expected_type=type_hints["soft_limit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "hard_limit": hard_limit,
                "name": name,
                "soft_limit": soft_limit,
            }

        @builtins.property
        def hard_limit(self) -> jsii.Number:
            '''The hard limit for the ``ulimit`` type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ulimit.html#cfn-batch-jobdefinition-ulimit-hardlimit
            '''
            result = self._values.get("hard_limit")
            assert result is not None, "Required property 'hard_limit' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def name(self) -> builtins.str:
            '''The ``type`` of the ``ulimit`` .

            Valid values are: ``core`` | ``cpu`` | ``data`` | ``fsize`` | ``locks`` | ``memlock`` | ``msgqueue`` | ``nice`` | ``nofile`` | ``nproc`` | ``rss`` | ``rtprio`` | ``rttime`` | ``sigpending`` | ``stack`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ulimit.html#cfn-batch-jobdefinition-ulimit-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def soft_limit(self) -> jsii.Number:
            '''The soft limit for the ``ulimit`` type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-ulimit.html#cfn-batch-jobdefinition-ulimit-softlimit
            '''
            result = self._values.get("soft_limit")
            assert result is not None, "Required property 'soft_limit' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "UlimitProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.VolumesHostProperty",
        jsii_struct_bases=[],
        name_mapping={"source_path": "sourcePath"},
    )
    class VolumesHostProperty:
        def __init__(
            self,
            *,
            source_path: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param source_path: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-volumeshost.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                volumes_host_property = batch.CfnJobDefinition.VolumesHostProperty(
                    source_path="sourcePath"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1af85cd0eadb4e77501a8751a1c3fc18ec19311f49db5d0f921588460010456a)
                check_type(argname="argument source_path", value=source_path, expected_type=type_hints["source_path"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if source_path is not None:
                self._values["source_path"] = source_path

        @builtins.property
        def source_path(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-volumeshost.html#cfn-batch-jobdefinition-volumeshost-sourcepath
            '''
            result = self._values.get("source_path")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumesHostProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinition.VolumesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "efs_volume_configuration": "efsVolumeConfiguration",
            "host": "host",
            "name": "name",
        },
    )
    class VolumesProperty:
        def __init__(
            self,
            *,
            efs_volume_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.EfsVolumeConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            host: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobDefinition.VolumesHostProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param efs_volume_configuration: 
            :param host: 
            :param name: 

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-volumes.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                volumes_property = batch.CfnJobDefinition.VolumesProperty(
                    efs_volume_configuration=batch.CfnJobDefinition.EfsVolumeConfigurationProperty(
                        file_system_id="fileSystemId",
                
                        # the properties below are optional
                        authorization_config=batch.CfnJobDefinition.AuthorizationConfigProperty(
                            access_point_id="accessPointId",
                            iam="iam"
                        ),
                        root_directory="rootDirectory",
                        transit_encryption="transitEncryption",
                        transit_encryption_port=123
                    ),
                    host=batch.CfnJobDefinition.VolumesHostProperty(
                        source_path="sourcePath"
                    ),
                    name="name"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a4a9d90db8477ca586c3c89fb2ce5a548e9a3bb699ae625a80a8a105b02a362e)
                check_type(argname="argument efs_volume_configuration", value=efs_volume_configuration, expected_type=type_hints["efs_volume_configuration"])
                check_type(argname="argument host", value=host, expected_type=type_hints["host"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if efs_volume_configuration is not None:
                self._values["efs_volume_configuration"] = efs_volume_configuration
            if host is not None:
                self._values["host"] = host
            if name is not None:
                self._values["name"] = name

        @builtins.property
        def efs_volume_configuration(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EfsVolumeConfigurationProperty"]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-volumes.html#cfn-batch-jobdefinition-volumes-efsvolumeconfiguration
            '''
            result = self._values.get("efs_volume_configuration")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.EfsVolumeConfigurationProperty"]], result)

        @builtins.property
        def host(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.VolumesHostProperty"]]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-volumes.html#cfn-batch-jobdefinition-volumes-host
            '''
            result = self._values.get("host")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnJobDefinition.VolumesHostProperty"]], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''
            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobdefinition-volumes.html#cfn-batch-jobdefinition-volumes-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.CfnJobDefinitionProps",
    jsii_struct_bases=[],
    name_mapping={
        "type": "type",
        "consumable_resource_properties": "consumableResourceProperties",
        "container_properties": "containerProperties",
        "ecs_properties": "ecsProperties",
        "eks_properties": "eksProperties",
        "job_definition_name": "jobDefinitionName",
        "node_properties": "nodeProperties",
        "parameters": "parameters",
        "platform_capabilities": "platformCapabilities",
        "propagate_tags": "propagateTags",
        "retry_strategy": "retryStrategy",
        "scheduling_priority": "schedulingPriority",
        "tags": "tags",
        "timeout": "timeout",
    },
)
class CfnJobDefinitionProps:
    def __init__(
        self,
        *,
        type: builtins.str,
        consumable_resource_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnJobDefinition.ConsumableResourcePropertiesProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        container_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnJobDefinition.ContainerPropertiesProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        ecs_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnJobDefinition.EcsPropertiesProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        eks_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnJobDefinition.EksPropertiesProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        job_definition_name: typing.Optional[builtins.str] = None,
        node_properties: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnJobDefinition.NodePropertiesProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        parameters: typing.Any = None,
        platform_capabilities: typing.Optional[typing.Sequence[builtins.str]] = None,
        propagate_tags: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        retry_strategy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnJobDefinition.RetryStrategyProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        scheduling_priority: typing.Optional[jsii.Number] = None,
        tags: typing.Any = None,
        timeout: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnJobDefinition.TimeoutProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnJobDefinition``.

        :param type: The type of job definition. For more information about multi-node parallel jobs, see `Creating a multi-node parallel job definition <https://docs.aws.amazon.com/batch/latest/userguide/multi-node-job-def.html>`_ in the *AWS Batch User Guide* . - If the value is ``container`` , then one of the following is required: ``containerProperties`` , ``ecsProperties`` , or ``eksProperties`` . - If the value is ``multinode`` , then ``nodeProperties`` is required. .. epigraph:: If the job is run on Fargate resources, then ``multinode`` isn't supported.
        :param consumable_resource_properties: Contains a list of consumable resources required by the job.
        :param container_properties: An object with properties specific to Amazon ECS-based jobs. When ``containerProperties`` is used in the job definition, it can't be used in addition to ``eksProperties`` , ``ecsProperties`` , or ``nodeProperties`` .
        :param ecs_properties: An object that contains the properties for the Amazon ECS resources of a job.When ``ecsProperties`` is used in the job definition, it can't be used in addition to ``containerProperties`` , ``eksProperties`` , or ``nodeProperties`` .
        :param eks_properties: An object with properties that are specific to Amazon EKS-based jobs. When ``eksProperties`` is used in the job definition, it can't be used in addition to ``containerProperties`` , ``ecsProperties`` , or ``nodeProperties`` .
        :param job_definition_name: The name of the job definition.
        :param node_properties: An object with properties that are specific to multi-node parallel jobs. When ``nodeProperties`` is used in the job definition, it can't be used in addition to ``containerProperties`` , ``ecsProperties`` , or ``eksProperties`` . .. epigraph:: If the job runs on Fargate resources, don't specify ``nodeProperties`` . Use ``containerProperties`` instead.
        :param parameters: Default parameters or parameter substitution placeholders that are set in the job definition. Parameters are specified as a key-value pair mapping. Parameters in a ``SubmitJob`` request override any corresponding parameter defaults from the job definition. For more information about specifying parameters, see `Job definition parameters <https://docs.aws.amazon.com/batch/latest/userguide/job_definition_parameters.html>`_ in the *AWS Batch User Guide* .
        :param platform_capabilities: The platform capabilities required by the job definition. If no value is specified, it defaults to ``EC2`` . Jobs run on Fargate resources specify ``FARGATE`` .
        :param propagate_tags: Specifies whether to propagate the tags from the job or job definition to the corresponding Amazon ECS task. If no value is specified, the tags aren't propagated. Tags can only be propagated to the tasks when the tasks are created. For tags with the same name, job tags are given priority over job definitions tags. If the total number of combined tags from the job and job definition is over 50, the job is moved to the ``FAILED`` state.
        :param retry_strategy: The retry strategy to use for failed jobs that are submitted with this job definition.
        :param scheduling_priority: The scheduling priority of the job definition. This only affects jobs in job queues with a fair-share policy. Jobs with a higher scheduling priority are scheduled before jobs with a lower scheduling priority.
        :param tags: The tags that are applied to the job definition.
        :param timeout: The timeout time for jobs that are submitted with this job definition. After the amount of time you specify passes, AWS Batch terminates your jobs if they aren't finished.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html
        :exampleMetadata: fixture=_generated

        Example::

            
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ed150d027b29486332c3fd2205a51f3c6f64c25946114fb86a6eba38434e87c9)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument consumable_resource_properties", value=consumable_resource_properties, expected_type=type_hints["consumable_resource_properties"])
            check_type(argname="argument container_properties", value=container_properties, expected_type=type_hints["container_properties"])
            check_type(argname="argument ecs_properties", value=ecs_properties, expected_type=type_hints["ecs_properties"])
            check_type(argname="argument eks_properties", value=eks_properties, expected_type=type_hints["eks_properties"])
            check_type(argname="argument job_definition_name", value=job_definition_name, expected_type=type_hints["job_definition_name"])
            check_type(argname="argument node_properties", value=node_properties, expected_type=type_hints["node_properties"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument platform_capabilities", value=platform_capabilities, expected_type=type_hints["platform_capabilities"])
            check_type(argname="argument propagate_tags", value=propagate_tags, expected_type=type_hints["propagate_tags"])
            check_type(argname="argument retry_strategy", value=retry_strategy, expected_type=type_hints["retry_strategy"])
            check_type(argname="argument scheduling_priority", value=scheduling_priority, expected_type=type_hints["scheduling_priority"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }
        if consumable_resource_properties is not None:
            self._values["consumable_resource_properties"] = consumable_resource_properties
        if container_properties is not None:
            self._values["container_properties"] = container_properties
        if ecs_properties is not None:
            self._values["ecs_properties"] = ecs_properties
        if eks_properties is not None:
            self._values["eks_properties"] = eks_properties
        if job_definition_name is not None:
            self._values["job_definition_name"] = job_definition_name
        if node_properties is not None:
            self._values["node_properties"] = node_properties
        if parameters is not None:
            self._values["parameters"] = parameters
        if platform_capabilities is not None:
            self._values["platform_capabilities"] = platform_capabilities
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if retry_strategy is not None:
            self._values["retry_strategy"] = retry_strategy
        if scheduling_priority is not None:
            self._values["scheduling_priority"] = scheduling_priority
        if tags is not None:
            self._values["tags"] = tags
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def type(self) -> builtins.str:
        '''The type of job definition.

        For more information about multi-node parallel jobs, see `Creating a multi-node parallel job definition <https://docs.aws.amazon.com/batch/latest/userguide/multi-node-job-def.html>`_ in the *AWS Batch User Guide* .

        - If the value is ``container`` , then one of the following is required: ``containerProperties`` , ``ecsProperties`` , or ``eksProperties`` .
        - If the value is ``multinode`` , then ``nodeProperties`` is required.

        .. epigraph::

           If the job is run on Fargate resources, then ``multinode`` isn't supported.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-type
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def consumable_resource_properties(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.ConsumableResourcePropertiesProperty]]:
        '''Contains a list of consumable resources required by the job.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-consumableresourceproperties
        '''
        result = self._values.get("consumable_resource_properties")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.ConsumableResourcePropertiesProperty]], result)

    @builtins.property
    def container_properties(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.ContainerPropertiesProperty]]:
        '''An object with properties specific to Amazon ECS-based jobs.

        When ``containerProperties`` is used in the job definition, it can't be used in addition to ``eksProperties`` , ``ecsProperties`` , or ``nodeProperties`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-containerproperties
        '''
        result = self._values.get("container_properties")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.ContainerPropertiesProperty]], result)

    @builtins.property
    def ecs_properties(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.EcsPropertiesProperty]]:
        '''An object that contains the properties for the Amazon ECS resources of a job.When ``ecsProperties`` is used in the job definition, it can't be used in addition to ``containerProperties`` , ``eksProperties`` , or ``nodeProperties`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-ecsproperties
        '''
        result = self._values.get("ecs_properties")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.EcsPropertiesProperty]], result)

    @builtins.property
    def eks_properties(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.EksPropertiesProperty]]:
        '''An object with properties that are specific to Amazon EKS-based jobs.

        When ``eksProperties`` is used in the job definition, it can't be used in addition to ``containerProperties`` , ``ecsProperties`` , or ``nodeProperties`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-eksproperties
        '''
        result = self._values.get("eks_properties")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.EksPropertiesProperty]], result)

    @builtins.property
    def job_definition_name(self) -> typing.Optional[builtins.str]:
        '''The name of the job definition.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-jobdefinitionname
        '''
        result = self._values.get("job_definition_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def node_properties(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.NodePropertiesProperty]]:
        '''An object with properties that are specific to multi-node parallel jobs.

        When ``nodeProperties`` is used in the job definition, it can't be used in addition to ``containerProperties`` , ``ecsProperties`` , or ``eksProperties`` .
        .. epigraph::

           If the job runs on Fargate resources, don't specify ``nodeProperties`` . Use ``containerProperties`` instead.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-nodeproperties
        '''
        result = self._values.get("node_properties")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.NodePropertiesProperty]], result)

    @builtins.property
    def parameters(self) -> typing.Any:
        '''Default parameters or parameter substitution placeholders that are set in the job definition.

        Parameters are specified as a key-value pair mapping. Parameters in a ``SubmitJob`` request override any corresponding parameter defaults from the job definition. For more information about specifying parameters, see `Job definition parameters <https://docs.aws.amazon.com/batch/latest/userguide/job_definition_parameters.html>`_ in the *AWS Batch User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Any, result)

    @builtins.property
    def platform_capabilities(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The platform capabilities required by the job definition.

        If no value is specified, it defaults to ``EC2`` . Jobs run on Fargate resources specify ``FARGATE`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-platformcapabilities
        '''
        result = self._values.get("platform_capabilities")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def propagate_tags(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether to propagate the tags from the job or job definition to the corresponding Amazon ECS task.

        If no value is specified, the tags aren't propagated. Tags can only be propagated to the tasks when the tasks are created. For tags with the same name, job tags are given priority over job definitions tags. If the total number of combined tags from the job and job definition is over 50, the job is moved to the ``FAILED`` state.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-propagatetags
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def retry_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.RetryStrategyProperty]]:
        '''The retry strategy to use for failed jobs that are submitted with this job definition.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-retrystrategy
        '''
        result = self._values.get("retry_strategy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.RetryStrategyProperty]], result)

    @builtins.property
    def scheduling_priority(self) -> typing.Optional[jsii.Number]:
        '''The scheduling priority of the job definition.

        This only affects jobs in job queues with a fair-share policy. Jobs with a higher scheduling priority are scheduled before jobs with a lower scheduling priority.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-schedulingpriority
        '''
        result = self._values.get("scheduling_priority")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tags(self) -> typing.Any:
        '''The tags that are applied to the job definition.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Any, result)

    @builtins.property
    def timeout(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.TimeoutProperty]]:
        '''The timeout time for jobs that are submitted with this job definition.

        After the amount of time you specify passes, AWS Batch terminates your jobs if they aren't finished.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobdefinition.html#cfn-batch-jobdefinition-timeout
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnJobDefinition.TimeoutProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnJobDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnJobQueue(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_batch.CfnJobQueue",
):
    '''The ``AWS::Batch::JobQueue`` resource specifies the parameters for an AWS Batch job queue definition.

    For more information, see `Job Queues <https://docs.aws.amazon.com/batch/latest/userguide/job_queues.html>`_ in the ** .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobqueue.html
    :cloudformationResource: AWS::Batch::JobQueue
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_batch as batch
        
        cfn_job_queue = batch.CfnJobQueue(self, "MyCfnJobQueue",
            compute_environment_order=[batch.CfnJobQueue.ComputeEnvironmentOrderProperty(
                compute_environment="computeEnvironment",
                order=123
            )],
            priority=123,
        
            # the properties below are optional
            job_queue_name="jobQueueName",
            job_state_time_limit_actions=[batch.CfnJobQueue.JobStateTimeLimitActionProperty(
                action="action",
                max_time_seconds=123,
                reason="reason",
                state="state"
            )],
            scheduling_policy_arn="schedulingPolicyArn",
            state="state",
            tags={
                "tags_key": "tags"
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        compute_environment_order: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobQueue.ComputeEnvironmentOrderProperty", typing.Dict[builtins.str, typing.Any]]]]],
        priority: jsii.Number,
        job_queue_name: typing.Optional[builtins.str] = None,
        job_state_time_limit_actions: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnJobQueue.JobStateTimeLimitActionProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        scheduling_policy_arn: typing.Optional[builtins.str] = None,
        state: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param compute_environment_order: The set of compute environments mapped to a job queue and their order relative to each other. The job scheduler uses this parameter to determine which compute environment runs a specific job. Compute environments must be in the ``VALID`` state before you can associate them with a job queue. You can associate up to three compute environments with a job queue. All of the compute environments must be either EC2 ( ``EC2`` or ``SPOT`` ) or Fargate ( ``FARGATE`` or ``FARGATE_SPOT`` ); EC2 and Fargate compute environments can't be mixed. .. epigraph:: All compute environments that are associated with a job queue must share the same architecture. AWS Batch doesn't support mixing compute environment architecture types in a single job queue.
        :param priority: The priority of the job queue. Job queues with a higher priority (or a higher integer value for the ``priority`` parameter) are evaluated first when associated with the same compute environment. Priority is determined in descending order. For example, a job queue with a priority value of ``10`` is given scheduling preference over a job queue with a priority value of ``1`` . All of the compute environments must be either EC2 ( ``EC2`` or ``SPOT`` ) or Fargate ( ``FARGATE`` or ``FARGATE_SPOT`` ); EC2 and Fargate compute environments can't be mixed.
        :param job_queue_name: The name of the job queue. It can be up to 128 letters long. It can contain uppercase and lowercase letters, numbers, hyphens (-), and underscores (_).
        :param job_state_time_limit_actions: The set of actions that AWS Batch perform on jobs that remain at the head of the job queue in the specified state longer than specified times. AWS Batch will perform each action after ``maxTimeSeconds`` has passed.
        :param scheduling_policy_arn: The Amazon Resource Name (ARN) of the scheduling policy. The format is ``aws: *Partition* :batch: *Region* : *Account* :scheduling-policy/ *Name*`` . For example, ``aws:aws:batch:us-west-2:123456789012:scheduling-policy/MySchedulingPolicy`` .
        :param state: The state of the job queue. If the job queue state is ``ENABLED`` , it is able to accept jobs. If the job queue state is ``DISABLED`` , new jobs can't be added to the queue, but jobs already in the queue can finish.
        :param tags: The tags that are applied to the job queue. For more information, see `Tagging your AWS Batch resources <https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html>`_ in *AWS Batch User Guide* .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6480ab252bc515b2cdb6dc0c833877438fd473fd39eadf6b8a064969a5fc051d)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnJobQueueProps(
            compute_environment_order=compute_environment_order,
            priority=priority,
            job_queue_name=job_queue_name,
            job_state_time_limit_actions=job_state_time_limit_actions,
            scheduling_policy_arn=scheduling_policy_arn,
            state=state,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0501823e805eabb76b2d16d277e0e42dcf4e0873c3adb0898fab52f723baa571)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ce1fe922d9589f26998c2113a8a19530d98bfc01fe2462a8639a3593512e2368)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrJobQueueArn")
    def attr_job_queue_arn(self) -> builtins.str:
        '''Returns the job queue ARN, such as ``batch: *us-east-1* : *111122223333* :job-queue/ *JobQueueName*`` .

        :cloudformationAttribute: JobQueueArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrJobQueueArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="computeEnvironmentOrder")
    def compute_environment_order(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobQueue.ComputeEnvironmentOrderProperty"]]]:
        '''The set of compute environments mapped to a job queue and their order relative to each other.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobQueue.ComputeEnvironmentOrderProperty"]]], jsii.get(self, "computeEnvironmentOrder"))

    @compute_environment_order.setter
    def compute_environment_order(
        self,
        value: typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobQueue.ComputeEnvironmentOrderProperty"]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aa1c2f3f42165309672c3fdf8c286ff9dc5e99756ec1a5b3ffca9e719d4c7036)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "computeEnvironmentOrder", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="priority")
    def priority(self) -> jsii.Number:
        '''The priority of the job queue.'''
        return typing.cast(jsii.Number, jsii.get(self, "priority"))

    @priority.setter
    def priority(self, value: jsii.Number) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__620331f74c116d31a6d43627e1a45e7041bc3c110194265f30dc193863c8ed94)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "priority", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="jobQueueName")
    def job_queue_name(self) -> typing.Optional[builtins.str]:
        '''The name of the job queue.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "jobQueueName"))

    @job_queue_name.setter
    def job_queue_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3b88e27e62b46f1049263a61bb14f62674a8121083de64c2db32d585651e1c30)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "jobQueueName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="jobStateTimeLimitActions")
    def job_state_time_limit_actions(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobQueue.JobStateTimeLimitActionProperty"]]]]:
        '''The set of actions that AWS Batch perform on jobs that remain at the head of the job queue in the specified state longer than specified times.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobQueue.JobStateTimeLimitActionProperty"]]]], jsii.get(self, "jobStateTimeLimitActions"))

    @job_state_time_limit_actions.setter
    def job_state_time_limit_actions(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnJobQueue.JobStateTimeLimitActionProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6e248d0029fc3043fabb974edf0f06b8f9f721a8f13ce794fd6c39b5ae3bb91e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "jobStateTimeLimitActions", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="schedulingPolicyArn")
    def scheduling_policy_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the scheduling policy.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "schedulingPolicyArn"))

    @scheduling_policy_arn.setter
    def scheduling_policy_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__48a6952b4a8d4426554895a52085bf8c88701735f3f94a3426e81e56ab8a5761)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "schedulingPolicyArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="state")
    def state(self) -> typing.Optional[builtins.str]:
        '''The state of the job queue.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "state"))

    @state.setter
    def state(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e97106b05367d04f8f618f0b8edf4f7fdc4b88436890062ac6e344ede4e180b0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "state", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags that are applied to the job queue.'''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e658491e12e1806302783cf972183ab89bf0d8d1d3abe7164fad544d18b91367)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobQueue.ComputeEnvironmentOrderProperty",
        jsii_struct_bases=[],
        name_mapping={"compute_environment": "computeEnvironment", "order": "order"},
    )
    class ComputeEnvironmentOrderProperty:
        def __init__(
            self,
            *,
            compute_environment: builtins.str,
            order: jsii.Number,
        ) -> None:
            '''The order that compute environments are tried in for job placement within a queue.

            Compute environments are tried in ascending order. For example, if two compute environments are associated with a job queue, the compute environment with a lower order integer value is tried for job placement first. Compute environments must be in the ``VALID`` state before you can associate them with a job queue. All of the compute environments must be either EC2 ( ``EC2`` or ``SPOT`` ) or Fargate ( ``FARGATE`` or ``FARGATE_SPOT`` ); Amazon EC2 and Fargate compute environments can't be mixed.
            .. epigraph::

               All compute environments that are associated with a job queue must share the same architecture. AWS Batch doesn't support mixing compute environment architecture types in a single job queue.

            :param compute_environment: The Amazon Resource Name (ARN) of the compute environment.
            :param order: The order of the compute environment. Compute environments are tried in ascending order. For example, if two compute environments are associated with a job queue, the compute environment with a lower ``order`` integer value is tried for job placement first.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobqueue-computeenvironmentorder.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                compute_environment_order_property = batch.CfnJobQueue.ComputeEnvironmentOrderProperty(
                    compute_environment="computeEnvironment",
                    order=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__939623842116ecc366f7e36591fb2bdd35bd47aacc55c0a794ff81ec2a026c43)
                check_type(argname="argument compute_environment", value=compute_environment, expected_type=type_hints["compute_environment"])
                check_type(argname="argument order", value=order, expected_type=type_hints["order"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "compute_environment": compute_environment,
                "order": order,
            }

        @builtins.property
        def compute_environment(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the compute environment.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobqueue-computeenvironmentorder.html#cfn-batch-jobqueue-computeenvironmentorder-computeenvironment
            '''
            result = self._values.get("compute_environment")
            assert result is not None, "Required property 'compute_environment' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def order(self) -> jsii.Number:
            '''The order of the compute environment.

            Compute environments are tried in ascending order. For example, if two compute environments are associated with a job queue, the compute environment with a lower ``order`` integer value is tried for job placement first.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobqueue-computeenvironmentorder.html#cfn-batch-jobqueue-computeenvironmentorder-order
            '''
            result = self._values.get("order")
            assert result is not None, "Required property 'order' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ComputeEnvironmentOrderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnJobQueue.JobStateTimeLimitActionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "action": "action",
            "max_time_seconds": "maxTimeSeconds",
            "reason": "reason",
            "state": "state",
        },
    )
    class JobStateTimeLimitActionProperty:
        def __init__(
            self,
            *,
            action: builtins.str,
            max_time_seconds: jsii.Number,
            reason: builtins.str,
            state: builtins.str,
        ) -> None:
            '''Specifies an action that AWS Batch will take after the job has remained at the head of the queue in the specified state for longer than the specified time.

            :param action: The action to take when a job is at the head of the job queue in the specified state for the specified period of time. The only supported value is ``CANCEL`` , which will cancel the job.
            :param max_time_seconds: The approximate amount of time, in seconds, that must pass with the job in the specified state before the action is taken. The minimum value is 600 (10 minutes) and the maximum value is 86,400 (24 hours).
            :param reason: The reason to log for the action being taken.
            :param state: The state of the job needed to trigger the action. The only supported value is ``RUNNABLE`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobqueue-jobstatetimelimitaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                job_state_time_limit_action_property = batch.CfnJobQueue.JobStateTimeLimitActionProperty(
                    action="action",
                    max_time_seconds=123,
                    reason="reason",
                    state="state"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__78ff35092216cfb2ffd2106f7da574c86c17494e3404a6f2ccdc020cc2de0437)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument max_time_seconds", value=max_time_seconds, expected_type=type_hints["max_time_seconds"])
                check_type(argname="argument reason", value=reason, expected_type=type_hints["reason"])
                check_type(argname="argument state", value=state, expected_type=type_hints["state"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
                "max_time_seconds": max_time_seconds,
                "reason": reason,
                "state": state,
            }

        @builtins.property
        def action(self) -> builtins.str:
            '''The action to take when a job is at the head of the job queue in the specified state for the specified period of time.

            The only supported value is ``CANCEL`` , which will cancel the job.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobqueue-jobstatetimelimitaction.html#cfn-batch-jobqueue-jobstatetimelimitaction-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def max_time_seconds(self) -> jsii.Number:
            '''The approximate amount of time, in seconds, that must pass with the job in the specified state before the action is taken.

            The minimum value is 600 (10 minutes) and the maximum value is 86,400 (24 hours).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobqueue-jobstatetimelimitaction.html#cfn-batch-jobqueue-jobstatetimelimitaction-maxtimeseconds
            '''
            result = self._values.get("max_time_seconds")
            assert result is not None, "Required property 'max_time_seconds' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def reason(self) -> builtins.str:
            '''The reason to log for the action being taken.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobqueue-jobstatetimelimitaction.html#cfn-batch-jobqueue-jobstatetimelimitaction-reason
            '''
            result = self._values.get("reason")
            assert result is not None, "Required property 'reason' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def state(self) -> builtins.str:
            '''The state of the job needed to trigger the action.

            The only supported value is ``RUNNABLE`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-jobqueue-jobstatetimelimitaction.html#cfn-batch-jobqueue-jobstatetimelimitaction-state
            '''
            result = self._values.get("state")
            assert result is not None, "Required property 'state' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "JobStateTimeLimitActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.CfnJobQueueProps",
    jsii_struct_bases=[],
    name_mapping={
        "compute_environment_order": "computeEnvironmentOrder",
        "priority": "priority",
        "job_queue_name": "jobQueueName",
        "job_state_time_limit_actions": "jobStateTimeLimitActions",
        "scheduling_policy_arn": "schedulingPolicyArn",
        "state": "state",
        "tags": "tags",
    },
)
class CfnJobQueueProps:
    def __init__(
        self,
        *,
        compute_environment_order: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnJobQueue.ComputeEnvironmentOrderProperty, typing.Dict[builtins.str, typing.Any]]]]],
        priority: jsii.Number,
        job_queue_name: typing.Optional[builtins.str] = None,
        job_state_time_limit_actions: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnJobQueue.JobStateTimeLimitActionProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        scheduling_policy_arn: typing.Optional[builtins.str] = None,
        state: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnJobQueue``.

        :param compute_environment_order: The set of compute environments mapped to a job queue and their order relative to each other. The job scheduler uses this parameter to determine which compute environment runs a specific job. Compute environments must be in the ``VALID`` state before you can associate them with a job queue. You can associate up to three compute environments with a job queue. All of the compute environments must be either EC2 ( ``EC2`` or ``SPOT`` ) or Fargate ( ``FARGATE`` or ``FARGATE_SPOT`` ); EC2 and Fargate compute environments can't be mixed. .. epigraph:: All compute environments that are associated with a job queue must share the same architecture. AWS Batch doesn't support mixing compute environment architecture types in a single job queue.
        :param priority: The priority of the job queue. Job queues with a higher priority (or a higher integer value for the ``priority`` parameter) are evaluated first when associated with the same compute environment. Priority is determined in descending order. For example, a job queue with a priority value of ``10`` is given scheduling preference over a job queue with a priority value of ``1`` . All of the compute environments must be either EC2 ( ``EC2`` or ``SPOT`` ) or Fargate ( ``FARGATE`` or ``FARGATE_SPOT`` ); EC2 and Fargate compute environments can't be mixed.
        :param job_queue_name: The name of the job queue. It can be up to 128 letters long. It can contain uppercase and lowercase letters, numbers, hyphens (-), and underscores (_).
        :param job_state_time_limit_actions: The set of actions that AWS Batch perform on jobs that remain at the head of the job queue in the specified state longer than specified times. AWS Batch will perform each action after ``maxTimeSeconds`` has passed.
        :param scheduling_policy_arn: The Amazon Resource Name (ARN) of the scheduling policy. The format is ``aws: *Partition* :batch: *Region* : *Account* :scheduling-policy/ *Name*`` . For example, ``aws:aws:batch:us-west-2:123456789012:scheduling-policy/MySchedulingPolicy`` .
        :param state: The state of the job queue. If the job queue state is ``ENABLED`` , it is able to accept jobs. If the job queue state is ``DISABLED`` , new jobs can't be added to the queue, but jobs already in the queue can finish.
        :param tags: The tags that are applied to the job queue. For more information, see `Tagging your AWS Batch resources <https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html>`_ in *AWS Batch User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobqueue.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_batch as batch
            
            cfn_job_queue_props = batch.CfnJobQueueProps(
                compute_environment_order=[batch.CfnJobQueue.ComputeEnvironmentOrderProperty(
                    compute_environment="computeEnvironment",
                    order=123
                )],
                priority=123,
            
                # the properties below are optional
                job_queue_name="jobQueueName",
                job_state_time_limit_actions=[batch.CfnJobQueue.JobStateTimeLimitActionProperty(
                    action="action",
                    max_time_seconds=123,
                    reason="reason",
                    state="state"
                )],
                scheduling_policy_arn="schedulingPolicyArn",
                state="state",
                tags={
                    "tags_key": "tags"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__009d6d384b1b723169e64875095e05fe852ae3931adf5ba2004d22475a76caa9)
            check_type(argname="argument compute_environment_order", value=compute_environment_order, expected_type=type_hints["compute_environment_order"])
            check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
            check_type(argname="argument job_queue_name", value=job_queue_name, expected_type=type_hints["job_queue_name"])
            check_type(argname="argument job_state_time_limit_actions", value=job_state_time_limit_actions, expected_type=type_hints["job_state_time_limit_actions"])
            check_type(argname="argument scheduling_policy_arn", value=scheduling_policy_arn, expected_type=type_hints["scheduling_policy_arn"])
            check_type(argname="argument state", value=state, expected_type=type_hints["state"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "compute_environment_order": compute_environment_order,
            "priority": priority,
        }
        if job_queue_name is not None:
            self._values["job_queue_name"] = job_queue_name
        if job_state_time_limit_actions is not None:
            self._values["job_state_time_limit_actions"] = job_state_time_limit_actions
        if scheduling_policy_arn is not None:
            self._values["scheduling_policy_arn"] = scheduling_policy_arn
        if state is not None:
            self._values["state"] = state
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def compute_environment_order(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnJobQueue.ComputeEnvironmentOrderProperty]]]:
        '''The set of compute environments mapped to a job queue and their order relative to each other.

        The job scheduler uses this parameter to determine which compute environment runs a specific job. Compute environments must be in the ``VALID`` state before you can associate them with a job queue. You can associate up to three compute environments with a job queue. All of the compute environments must be either EC2 ( ``EC2`` or ``SPOT`` ) or Fargate ( ``FARGATE`` or ``FARGATE_SPOT`` ); EC2 and Fargate compute environments can't be mixed.
        .. epigraph::

           All compute environments that are associated with a job queue must share the same architecture. AWS Batch doesn't support mixing compute environment architecture types in a single job queue.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobqueue.html#cfn-batch-jobqueue-computeenvironmentorder
        '''
        result = self._values.get("compute_environment_order")
        assert result is not None, "Required property 'compute_environment_order' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnJobQueue.ComputeEnvironmentOrderProperty]]], result)

    @builtins.property
    def priority(self) -> jsii.Number:
        '''The priority of the job queue.

        Job queues with a higher priority (or a higher integer value for the ``priority`` parameter) are evaluated first when associated with the same compute environment. Priority is determined in descending order. For example, a job queue with a priority value of ``10`` is given scheduling preference over a job queue with a priority value of ``1`` . All of the compute environments must be either EC2 ( ``EC2`` or ``SPOT`` ) or Fargate ( ``FARGATE`` or ``FARGATE_SPOT`` ); EC2 and Fargate compute environments can't be mixed.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobqueue.html#cfn-batch-jobqueue-priority
        '''
        result = self._values.get("priority")
        assert result is not None, "Required property 'priority' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def job_queue_name(self) -> typing.Optional[builtins.str]:
        '''The name of the job queue.

        It can be up to 128 letters long. It can contain uppercase and lowercase letters, numbers, hyphens (-), and underscores (_).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobqueue.html#cfn-batch-jobqueue-jobqueuename
        '''
        result = self._values.get("job_queue_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def job_state_time_limit_actions(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnJobQueue.JobStateTimeLimitActionProperty]]]]:
        '''The set of actions that AWS Batch perform on jobs that remain at the head of the job queue in the specified state longer than specified times.

        AWS Batch will perform each action after ``maxTimeSeconds`` has passed.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobqueue.html#cfn-batch-jobqueue-jobstatetimelimitactions
        '''
        result = self._values.get("job_state_time_limit_actions")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnJobQueue.JobStateTimeLimitActionProperty]]]], result)

    @builtins.property
    def scheduling_policy_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the scheduling policy.

        The format is ``aws: *Partition* :batch: *Region* : *Account* :scheduling-policy/ *Name*`` . For example, ``aws:aws:batch:us-west-2:123456789012:scheduling-policy/MySchedulingPolicy`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobqueue.html#cfn-batch-jobqueue-schedulingpolicyarn
        '''
        result = self._values.get("scheduling_policy_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def state(self) -> typing.Optional[builtins.str]:
        '''The state of the job queue.

        If the job queue state is ``ENABLED`` , it is able to accept jobs. If the job queue state is ``DISABLED`` , new jobs can't be added to the queue, but jobs already in the queue can finish.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobqueue.html#cfn-batch-jobqueue-state
        '''
        result = self._values.get("state")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags that are applied to the job queue.

        For more information, see `Tagging your AWS Batch resources <https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html>`_ in *AWS Batch User Guide* .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-jobqueue.html#cfn-batch-jobqueue-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnJobQueueProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnSchedulingPolicy(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_batch.CfnSchedulingPolicy",
):
    '''The ``AWS::Batch::SchedulingPolicy`` resource specifies the parameters for an AWS Batch scheduling policy.

    For more information, see `Scheduling Policies <https://docs.aws.amazon.com/batch/latest/userguide/scheduling_policies.html>`_ in the ** .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-schedulingpolicy.html
    :cloudformationResource: AWS::Batch::SchedulingPolicy
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_batch as batch
        
        cfn_scheduling_policy = batch.CfnSchedulingPolicy(self, "MyCfnSchedulingPolicy",
            fairshare_policy=batch.CfnSchedulingPolicy.FairsharePolicyProperty(
                compute_reservation=123,
                share_decay_seconds=123,
                share_distribution=[batch.CfnSchedulingPolicy.ShareAttributesProperty(
                    share_identifier="shareIdentifier",
                    weight_factor=123
                )]
            ),
            name="name",
            tags={
                "tags_key": "tags"
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        fairshare_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnSchedulingPolicy.FairsharePolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param fairshare_policy: The fair-share scheduling policy details.
        :param name: The name of the fair-share scheduling policy. It can be up to 128 letters long. It can contain uppercase and lowercase letters, numbers, hyphens (-), and underscores (_).
        :param tags: The tags that you apply to the scheduling policy to help you categorize and organize your resources. Each tag consists of a key and an optional value. For more information, see `Tagging AWS Resources <https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html>`_ in *AWS General Reference* . These tags can be updated or removed using the `TagResource <https://docs.aws.amazon.com/batch/latest/APIReference/API_TagResource.html>`_ and `UntagResource <https://docs.aws.amazon.com/batch/latest/APIReference/API_UntagResource.html>`_ API operations.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__31a21a4a6536b3efab619619ed719e6b0662b5c83fe7a1dc489b27f0b3ea56db)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnSchedulingPolicyProps(
            fairshare_policy=fairshare_policy, name=name, tags=tags
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__df36e53e7605cc7837477592dff920fd2c974ea55bea2f8469f2e732ec9524ca)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aefdc709a6b9ffad7303fb907694b285dd6ed2a9795b2c93a5ab983d1fc0ab2e)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''Returns the scheduling policy ARN, such as ``batch: *us-east-1* : *111122223333* :scheduling-policy/ *HighPriority*`` .

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="fairsharePolicy")
    def fairshare_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnSchedulingPolicy.FairsharePolicyProperty"]]:
        '''The fair-share scheduling policy details.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnSchedulingPolicy.FairsharePolicyProperty"]], jsii.get(self, "fairsharePolicy"))

    @fairshare_policy.setter
    def fairshare_policy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnSchedulingPolicy.FairsharePolicyProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cfc1fdc7b365c326e80f94bc6ce2c1ba05c54c3d0dc4caaecec7e9eb7ec315b3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "fairsharePolicy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of the fair-share scheduling policy.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__411bf4b677626f26ae0d797bd4706cc6d8b5dbd45471422cf971b438d0f43ee0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags that you apply to the scheduling policy to help you categorize and organize your resources.'''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(
        self,
        value: typing.Optional[typing.Mapping[builtins.str, builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__15f828d01fc7085728330a84e961df84cb35664d5a94e9bbf3bcfe93f7b3c7d5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnSchedulingPolicy.FairsharePolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "compute_reservation": "computeReservation",
            "share_decay_seconds": "shareDecaySeconds",
            "share_distribution": "shareDistribution",
        },
    )
    class FairsharePolicyProperty:
        def __init__(
            self,
            *,
            compute_reservation: typing.Optional[jsii.Number] = None,
            share_decay_seconds: typing.Optional[jsii.Number] = None,
            share_distribution: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnSchedulingPolicy.ShareAttributesProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''The fair-share scheduling policy details.

            :param compute_reservation: A value used to reserve some of the available maximum vCPU for share identifiers that aren't already used. The reserved ratio is ``( *computeReservation* /100)^ *ActiveFairShares*`` where ``*ActiveFairShares*`` is the number of active share identifiers. For example, a ``computeReservation`` value of 50 indicates that AWS Batch reserves 50% of the maximum available vCPU if there's only one share identifier. It reserves 25% if there are two share identifiers. It reserves 12.5% if there are three share identifiers. A ``computeReservation`` value of 25 indicates that AWS Batch should reserve 25% of the maximum available vCPU if there's only one share identifier, 6.25% if there are two fair share identifiers, and 1.56% if there are three share identifiers. The minimum value is 0 and the maximum value is 99.
            :param share_decay_seconds: The amount of time (in seconds) to use to calculate a fair-share percentage for each share identifier in use. A value of zero (0) indicates the default minimum time window (600 seconds). The maximum supported value is 604800 (1 week). The decay allows for more recently run jobs to have more weight than jobs that ran earlier. Consider adjusting this number if you have jobs that (on average) run longer than ten minutes, or a large difference in job count or job run times between share identifiers, and the allocation of resources doesn’t meet your needs.
            :param share_distribution: An array of ``SharedIdentifier`` objects that contain the weights for the share identifiers for the fair-share policy. Share identifiers that aren't included have a default weight of ``1.0`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-schedulingpolicy-fairsharepolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                fairshare_policy_property = batch.CfnSchedulingPolicy.FairsharePolicyProperty(
                    compute_reservation=123,
                    share_decay_seconds=123,
                    share_distribution=[batch.CfnSchedulingPolicy.ShareAttributesProperty(
                        share_identifier="shareIdentifier",
                        weight_factor=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d584604e0200cce08bb1571b600a67b89e1eafdfce088d1cbe150d7b12935adc)
                check_type(argname="argument compute_reservation", value=compute_reservation, expected_type=type_hints["compute_reservation"])
                check_type(argname="argument share_decay_seconds", value=share_decay_seconds, expected_type=type_hints["share_decay_seconds"])
                check_type(argname="argument share_distribution", value=share_distribution, expected_type=type_hints["share_distribution"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if compute_reservation is not None:
                self._values["compute_reservation"] = compute_reservation
            if share_decay_seconds is not None:
                self._values["share_decay_seconds"] = share_decay_seconds
            if share_distribution is not None:
                self._values["share_distribution"] = share_distribution

        @builtins.property
        def compute_reservation(self) -> typing.Optional[jsii.Number]:
            '''A value used to reserve some of the available maximum vCPU for share identifiers that aren't already used.

            The reserved ratio is ``( *computeReservation* /100)^ *ActiveFairShares*`` where ``*ActiveFairShares*`` is the number of active share identifiers.

            For example, a ``computeReservation`` value of 50 indicates that AWS Batch reserves 50% of the maximum available vCPU if there's only one share identifier. It reserves 25% if there are two share identifiers. It reserves 12.5% if there are three share identifiers. A ``computeReservation`` value of 25 indicates that AWS Batch should reserve 25% of the maximum available vCPU if there's only one share identifier, 6.25% if there are two fair share identifiers, and 1.56% if there are three share identifiers.

            The minimum value is 0 and the maximum value is 99.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-schedulingpolicy-fairsharepolicy.html#cfn-batch-schedulingpolicy-fairsharepolicy-computereservation
            '''
            result = self._values.get("compute_reservation")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def share_decay_seconds(self) -> typing.Optional[jsii.Number]:
            '''The amount of time (in seconds) to use to calculate a fair-share percentage for each share identifier in use.

            A value of zero (0) indicates the default minimum time window (600 seconds). The maximum supported value is 604800 (1 week).

            The decay allows for more recently run jobs to have more weight than jobs that ran earlier. Consider adjusting this number if you have jobs that (on average) run longer than ten minutes, or a large difference in job count or job run times between share identifiers, and the allocation of resources doesn’t meet your needs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-schedulingpolicy-fairsharepolicy.html#cfn-batch-schedulingpolicy-fairsharepolicy-sharedecayseconds
            '''
            result = self._values.get("share_decay_seconds")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def share_distribution(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnSchedulingPolicy.ShareAttributesProperty"]]]]:
            '''An array of ``SharedIdentifier`` objects that contain the weights for the share identifiers for the fair-share policy.

            Share identifiers that aren't included have a default weight of ``1.0`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-schedulingpolicy-fairsharepolicy.html#cfn-batch-schedulingpolicy-fairsharepolicy-sharedistribution
            '''
            result = self._values.get("share_distribution")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnSchedulingPolicy.ShareAttributesProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FairsharePolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_batch.CfnSchedulingPolicy.ShareAttributesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "share_identifier": "shareIdentifier",
            "weight_factor": "weightFactor",
        },
    )
    class ShareAttributesProperty:
        def __init__(
            self,
            *,
            share_identifier: typing.Optional[builtins.str] = None,
            weight_factor: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''Specifies the weights for the share identifiers for the fair-share policy.

            Share identifiers that aren't included have a default weight of ``1.0`` .

            :param share_identifier: A share identifier or share identifier prefix. If the string ends with an asterisk (*), this entry specifies the weight factor to use for share identifiers that start with that prefix. The list of share identifiers in a fair-share policy can't overlap. For example, you can't have one that specifies a ``shareIdentifier`` of ``UserA*`` and another that specifies a ``shareIdentifier`` of ``UserA-1`` . There can be no more than 500 share identifiers active in a job queue. The string is limited to 255 alphanumeric characters, and can be followed by an asterisk (*).
            :param weight_factor: The weight factor for the share identifier. The default value is 1.0. A lower value has a higher priority for compute resources. For example, jobs that use a share identifier with a weight factor of 0.125 (1/8) get 8 times the compute resources of jobs that use a share identifier with a weight factor of 1. The smallest supported value is 0.0001, and the largest supported value is 999.9999.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-schedulingpolicy-shareattributes.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_batch as batch
                
                share_attributes_property = batch.CfnSchedulingPolicy.ShareAttributesProperty(
                    share_identifier="shareIdentifier",
                    weight_factor=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a48599c650ce1ded6e5309cc09389c78237f12aa2328bbfe0d3335052f05da70)
                check_type(argname="argument share_identifier", value=share_identifier, expected_type=type_hints["share_identifier"])
                check_type(argname="argument weight_factor", value=weight_factor, expected_type=type_hints["weight_factor"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if share_identifier is not None:
                self._values["share_identifier"] = share_identifier
            if weight_factor is not None:
                self._values["weight_factor"] = weight_factor

        @builtins.property
        def share_identifier(self) -> typing.Optional[builtins.str]:
            '''A share identifier or share identifier prefix.

            If the string ends with an asterisk (*), this entry specifies the weight factor to use for share identifiers that start with that prefix. The list of share identifiers in a fair-share policy can't overlap. For example, you can't have one that specifies a ``shareIdentifier`` of ``UserA*`` and another that specifies a ``shareIdentifier`` of ``UserA-1`` .

            There can be no more than 500 share identifiers active in a job queue.

            The string is limited to 255 alphanumeric characters, and can be followed by an asterisk (*).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-schedulingpolicy-shareattributes.html#cfn-batch-schedulingpolicy-shareattributes-shareidentifier
            '''
            result = self._values.get("share_identifier")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def weight_factor(self) -> typing.Optional[jsii.Number]:
            '''The weight factor for the share identifier.

            The default value is 1.0. A lower value has a higher priority for compute resources. For example, jobs that use a share identifier with a weight factor of 0.125 (1/8) get 8 times the compute resources of jobs that use a share identifier with a weight factor of 1.

            The smallest supported value is 0.0001, and the largest supported value is 999.9999.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-batch-schedulingpolicy-shareattributes.html#cfn-batch-schedulingpolicy-shareattributes-weightfactor
            '''
            result = self._values.get("weight_factor")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ShareAttributesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.CfnSchedulingPolicyProps",
    jsii_struct_bases=[],
    name_mapping={
        "fairshare_policy": "fairsharePolicy",
        "name": "name",
        "tags": "tags",
    },
)
class CfnSchedulingPolicyProps:
    def __init__(
        self,
        *,
        fairshare_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnSchedulingPolicy.FairsharePolicyProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnSchedulingPolicy``.

        :param fairshare_policy: The fair-share scheduling policy details.
        :param name: The name of the fair-share scheduling policy. It can be up to 128 letters long. It can contain uppercase and lowercase letters, numbers, hyphens (-), and underscores (_).
        :param tags: The tags that you apply to the scheduling policy to help you categorize and organize your resources. Each tag consists of a key and an optional value. For more information, see `Tagging AWS Resources <https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html>`_ in *AWS General Reference* . These tags can be updated or removed using the `TagResource <https://docs.aws.amazon.com/batch/latest/APIReference/API_TagResource.html>`_ and `UntagResource <https://docs.aws.amazon.com/batch/latest/APIReference/API_UntagResource.html>`_ API operations.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-schedulingpolicy.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_batch as batch
            
            cfn_scheduling_policy_props = batch.CfnSchedulingPolicyProps(
                fairshare_policy=batch.CfnSchedulingPolicy.FairsharePolicyProperty(
                    compute_reservation=123,
                    share_decay_seconds=123,
                    share_distribution=[batch.CfnSchedulingPolicy.ShareAttributesProperty(
                        share_identifier="shareIdentifier",
                        weight_factor=123
                    )]
                ),
                name="name",
                tags={
                    "tags_key": "tags"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__baa72ee437297b58169f0020ba3178c321d8f72981fcd34857611be31a96093d)
            check_type(argname="argument fairshare_policy", value=fairshare_policy, expected_type=type_hints["fairshare_policy"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if fairshare_policy is not None:
            self._values["fairshare_policy"] = fairshare_policy
        if name is not None:
            self._values["name"] = name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def fairshare_policy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnSchedulingPolicy.FairsharePolicyProperty]]:
        '''The fair-share scheduling policy details.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-schedulingpolicy.html#cfn-batch-schedulingpolicy-fairsharepolicy
        '''
        result = self._values.get("fairshare_policy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnSchedulingPolicy.FairsharePolicyProperty]], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of the fair-share scheduling policy.

        It can be up to 128 letters long. It can contain uppercase and lowercase letters, numbers, hyphens (-), and underscores (_).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-schedulingpolicy.html#cfn-batch-schedulingpolicy-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The tags that you apply to the scheduling policy to help you categorize and organize your resources.

        Each tag consists of a key and an optional value. For more information, see `Tagging AWS Resources <https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html>`_ in *AWS General Reference* .

        These tags can be updated or removed using the `TagResource <https://docs.aws.amazon.com/batch/latest/APIReference/API_TagResource.html>`_ and `UntagResource <https://docs.aws.amazon.com/batch/latest/APIReference/API_UntagResource.html>`_ API operations.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-batch-schedulingpolicy.html#cfn-batch-schedulingpolicy-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnSchedulingPolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.ComputeEnvironmentProps",
    jsii_struct_bases=[],
    name_mapping={
        "compute_environment_name": "computeEnvironmentName",
        "enabled": "enabled",
        "service_role": "serviceRole",
    },
)
class ComputeEnvironmentProps:
    def __init__(
        self,
        *,
        compute_environment_name: typing.Optional[builtins.str] = None,
        enabled: typing.Optional[builtins.bool] = None,
        service_role: typing.Optional[_IRole_235f5d8e] = None,
    ) -> None:
        '''Props common to all ComputeEnvironments.

        :param compute_environment_name: The name of the ComputeEnvironment. Default: - generated by CloudFormation
        :param enabled: Whether or not this ComputeEnvironment can accept jobs from a Queue. Enabled ComputeEnvironments can accept jobs from a Queue and can scale instances up or down. Disabled ComputeEnvironments cannot accept jobs from a Queue or scale instances up or down. If you change a ComputeEnvironment from enabled to disabled while it is executing jobs, Jobs in the ``STARTED`` or ``RUNNING`` states will not be interrupted. As jobs complete, the ComputeEnvironment will scale instances down to ``minvCpus``. To ensure you aren't billed for unused capacity, set ``minvCpus`` to ``0``. Default: true
        :param service_role: The role Batch uses to perform actions on your behalf in your account, such as provision instances to run your jobs. Default: - a serviceRole will be created for managed CEs, none for unmanaged CEs

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_batch as batch
            from aws_cdk import aws_iam as iam
            
            # role: iam.Role
            
            compute_environment_props = batch.ComputeEnvironmentProps(
                compute_environment_name="computeEnvironmentName",
                enabled=False,
                service_role=role
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ed356ed6d550d61b3cd35694bb99d63fcbc2d06693243747eaf82004f05f9958)
            check_type(argname="argument compute_environment_name", value=compute_environment_name, expected_type=type_hints["compute_environment_name"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument service_role", value=service_role, expected_type=type_hints["service_role"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if compute_environment_name is not None:
            self._values["compute_environment_name"] = compute_environment_name
        if enabled is not None:
            self._values["enabled"] = enabled
        if service_role is not None:
            self._values["service_role"] = service_role

    @builtins.property
    def compute_environment_name(self) -> typing.Optional[builtins.str]:
        '''The name of the ComputeEnvironment.

        :default: - generated by CloudFormation
        '''
        result = self._values.get("compute_environment_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enabled(self) -> typing.Optional[builtins.bool]:
        '''Whether or not this ComputeEnvironment can accept jobs from a Queue.

        Enabled ComputeEnvironments can accept jobs from a Queue and
        can scale instances up or down.
        Disabled ComputeEnvironments cannot accept jobs from a Queue or
        scale instances up or down.

        If you change a ComputeEnvironment from enabled to disabled while it is executing jobs,
        Jobs in the ``STARTED`` or ``RUNNING`` states will not
        be interrupted. As jobs complete, the ComputeEnvironment will scale instances down to ``minvCpus``.

        To ensure you aren't billed for unused capacity, set ``minvCpus`` to ``0``.

        :default: true
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def service_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role Batch uses to perform actions on your behalf in your account, such as provision instances to run your jobs.

        :default: - a serviceRole will be created for managed CEs, none for unmanaged CEs
        '''
        result = self._values.get("service_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ComputeEnvironmentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.CustomReason",
    jsii_struct_bases=[],
    name_mapping={
        "on_exit_code": "onExitCode",
        "on_reason": "onReason",
        "on_status_reason": "onStatusReason",
    },
)
class CustomReason:
    def __init__(
        self,
        *,
        on_exit_code: typing.Optional[builtins.str] = None,
        on_reason: typing.Optional[builtins.str] = None,
        on_status_reason: typing.Optional[builtins.str] = None,
    ) -> None:
        '''The corresponding Action will only be taken if *all* of the conditions specified here are met.

        :param on_exit_code: A glob string that will match on the job exit code. For example, ``'40*'`` will match 400, 404, 40123456789012 Default: - will not match on the exit code
        :param on_reason: A glob string that will match on the reason returned by the exiting job For example, ``'CannotPullContainerError*'`` indicates that container needed to start the job could not be pulled. Default: - will not match on the reason
        :param on_status_reason: A glob string that will match on the statusReason returned by the exiting job. For example, ``'Host EC2*'`` indicates that the spot instance has been reclaimed. Default: - will not match on the status reason

        :exampleMetadata: infused

        Example::

            job_defn = batch.EcsJobDefinition(self, "JobDefn",
                container=batch.EcsEc2ContainerDefinition(self, "containerDefn",
                    image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
                    memory=cdk.Size.mebibytes(2048),
                    cpu=256
                ),
                retry_attempts=5,
                retry_strategies=[
                    batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.CANNOT_PULL_CONTAINER)
                ]
            )
            job_defn.add_retry_strategy(
                batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.SPOT_INSTANCE_RECLAIMED))
            job_defn.add_retry_strategy(
                batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.CANNOT_PULL_CONTAINER))
            job_defn.add_retry_strategy(
                batch.RetryStrategy.of(batch.Action.EXIT, batch.Reason.custom(
                    on_exit_code="40*",
                    on_reason="some reason"
                )))
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2f60f15c812c4d574b16bed9f540b41a78c483d379764bc9d3c92705baae5bd7)
            check_type(argname="argument on_exit_code", value=on_exit_code, expected_type=type_hints["on_exit_code"])
            check_type(argname="argument on_reason", value=on_reason, expected_type=type_hints["on_reason"])
            check_type(argname="argument on_status_reason", value=on_status_reason, expected_type=type_hints["on_status_reason"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if on_exit_code is not None:
            self._values["on_exit_code"] = on_exit_code
        if on_reason is not None:
            self._values["on_reason"] = on_reason
        if on_status_reason is not None:
            self._values["on_status_reason"] = on_status_reason

    @builtins.property
    def on_exit_code(self) -> typing.Optional[builtins.str]:
        '''A glob string that will match on the job exit code.

        For example, ``'40*'`` will match 400, 404, 40123456789012

        :default: - will not match on the exit code
        '''
        result = self._values.get("on_exit_code")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def on_reason(self) -> typing.Optional[builtins.str]:
        '''A glob string that will match on the reason returned by the exiting job For example, ``'CannotPullContainerError*'`` indicates that container needed to start the job could not be pulled.

        :default: - will not match on the reason
        '''
        result = self._values.get("on_reason")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def on_status_reason(self) -> typing.Optional[builtins.str]:
        '''A glob string that will match on the statusReason returned by the exiting job.

        For example, ``'Host EC2*'`` indicates that the spot instance has been reclaimed.

        :default: - will not match on the status reason
        '''
        result = self._values.get("on_status_reason")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CustomReason(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.Device",
    jsii_struct_bases=[],
    name_mapping={
        "host_path": "hostPath",
        "container_path": "containerPath",
        "permissions": "permissions",
    },
)
class Device:
    def __init__(
        self,
        *,
        host_path: builtins.str,
        container_path: typing.Optional[builtins.str] = None,
        permissions: typing.Optional[typing.Sequence["DevicePermission"]] = None,
    ) -> None:
        '''A container instance host device.

        :param host_path: The path for the device on the host container instance.
        :param container_path: The path inside the container at which to expose the host device. Default: Same path as the host
        :param permissions: The explicit permissions to provide to the container for the device. By default, the container has permissions for read, write, and mknod for the device. Default: Readonly

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_batch as batch
            
            device = batch.Device(
                host_path="hostPath",
            
                # the properties below are optional
                container_path="containerPath",
                permissions=[batch.DevicePermission.READ]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c476471f9d4e83bba26962fa9e3e8f6b5e149f8e1ea0141b1c768b2e4100d31d)
            check_type(argname="argument host_path", value=host_path, expected_type=type_hints["host_path"])
            check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
            check_type(argname="argument permissions", value=permissions, expected_type=type_hints["permissions"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "host_path": host_path,
        }
        if container_path is not None:
            self._values["container_path"] = container_path
        if permissions is not None:
            self._values["permissions"] = permissions

    @builtins.property
    def host_path(self) -> builtins.str:
        '''The path for the device on the host container instance.'''
        result = self._values.get("host_path")
        assert result is not None, "Required property 'host_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def container_path(self) -> typing.Optional[builtins.str]:
        '''The path inside the container at which to expose the host device.

        :default: Same path as the host
        '''
        result = self._values.get("container_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def permissions(self) -> typing.Optional[typing.List["DevicePermission"]]:
        '''The explicit permissions to provide to the container for the device.

        By default, the container has permissions for read, write, and mknod for the device.

        :default: Readonly
        '''
        result = self._values.get("permissions")
        return typing.cast(typing.Optional[typing.List["DevicePermission"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Device(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_batch.DevicePermission")
class DevicePermission(enum.Enum):
    '''Permissions for device access.'''

    READ = "READ"
    '''Read.'''
    WRITE = "WRITE"
    '''Write.'''
    MKNOD = "MKNOD"
    '''Make a node.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_batch.DnsPolicy")
class DnsPolicy(enum.Enum):
    '''The DNS Policy for the pod used by the Job Definition.

    :see: https://kubernetes.io/docs/concepts/services-networking/dns-pod-service/#pod-s-dns-policy
    '''

    DEFAULT = "DEFAULT"
    '''The Pod inherits the name resolution configuration from the node that the Pods run on.'''
    CLUSTER_FIRST = "CLUSTER_FIRST"
    '''Any DNS query that does not match the configured cluster domain suffix, such as ``"www.kubernetes.io"``, is forwarded to an upstream nameserver by the DNS server. Cluster administrators may have extra stub-domain and upstream DNS servers configured.'''
    CLUSTER_FIRST_WITH_HOST_NET = "CLUSTER_FIRST_WITH_HOST_NET"
    '''For Pods running with ``hostNetwork``, you should explicitly set its DNS policy to ``CLUSTER_FIRST_WITH_HOST_NET``.

    Otherwise, Pods running with ``hostNetwork`` and ``CLUSTER_FIRST`` will fallback to the behavior of the ``DEFAULT`` policy.
    '''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.EcsContainerDefinitionProps",
    jsii_struct_bases=[],
    name_mapping={
        "cpu": "cpu",
        "image": "image",
        "memory": "memory",
        "command": "command",
        "environment": "environment",
        "execution_role": "executionRole",
        "job_role": "jobRole",
        "linux_parameters": "linuxParameters",
        "logging": "logging",
        "readonly_root_filesystem": "readonlyRootFilesystem",
        "secrets": "secrets",
        "user": "user",
        "volumes": "volumes",
    },
)
class EcsContainerDefinitionProps:
    def __init__(
        self,
        *,
        cpu: jsii.Number,
        image: _ContainerImage_94af1b43,
        memory: _Size_7b441c34,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        execution_role: typing.Optional[_IRole_235f5d8e] = None,
        job_role: typing.Optional[_IRole_235f5d8e] = None,
        linux_parameters: typing.Optional["LinuxParameters"] = None,
        logging: typing.Optional[_LogDriver_393a21bb] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        secrets: typing.Optional[typing.Mapping[builtins.str, "Secret"]] = None,
        user: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Sequence["EcsVolume"]] = None,
    ) -> None:
        '''Props to configure an EcsContainerDefinition.

        :param cpu: The number of vCPUs reserved for the container. Each vCPU is equivalent to 1,024 CPU shares. For containers running on EC2 resources, you must specify at least one vCPU.
        :param image: The image that this container will run.
        :param memory: The memory hard limit present to the container. If your container attempts to exceed the memory specified, the container is terminated. You must specify at least 4 MiB of memory for a job.
        :param command: The command that's passed to the container. Default: - no command
        :param environment: The environment variables to pass to a container. Cannot start with ``AWS_BATCH``. We don't recommend using plaintext environment variables for sensitive information, such as credential data. Default: - no environment variables
        :param execution_role: The role used by Amazon ECS container and AWS Fargate agents to make AWS API calls on your behalf. Default: - a Role will be created
        :param job_role: The role that the container can assume. Default: - no job role
        :param linux_parameters: Linux-specific modifications that are applied to the container, such as details for device mappings. Default: none
        :param logging: The loging configuration for this Job. Default: - the log configuration of the Docker daemon
        :param readonly_root_filesystem: Gives the container readonly access to its root filesystem. Default: false
        :param secrets: A map from environment variable names to the secrets for the container. Allows your job definitions to reference the secret by the environment variable name defined in this property. Default: - no secrets
        :param user: The user name to use inside the container. Default: - no user
        :param volumes: The volumes to mount to this container. Automatically added to the job definition. Default: - no volumes

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_batch as batch
            from aws_cdk import aws_ecs as ecs
            from aws_cdk import aws_iam as iam
            
            # container_image: ecs.ContainerImage
            # ecs_volume: batch.EcsVolume
            # linux_parameters: batch.LinuxParameters
            # log_driver: ecs.LogDriver
            # role: iam.Role
            # secret: batch.Secret
            # size: cdk.Size
            
            ecs_container_definition_props = batch.EcsContainerDefinitionProps(
                cpu=123,
                image=container_image,
                memory=size,
            
                # the properties below are optional
                command=["command"],
                environment={
                    "environment_key": "environment"
                },
                execution_role=role,
                job_role=role,
                linux_parameters=linux_parameters,
                logging=log_driver,
                readonly_root_filesystem=False,
                secrets={
                    "secrets_key": secret
                },
                user="user",
                volumes=[ecs_volume]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__be0d7a8daca28c8e38e4e8dc8b20bc2ac1814af7b5292ed57768ad6bbe6f0aa9)
            check_type(argname="argument cpu", value=cpu, expected_type=type_hints["cpu"])
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
            check_type(argname="argument memory", value=memory, expected_type=type_hints["memory"])
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument execution_role", value=execution_role, expected_type=type_hints["execution_role"])
            check_type(argname="argument job_role", value=job_role, expected_type=type_hints["job_role"])
            check_type(argname="argument linux_parameters", value=linux_parameters, expected_type=type_hints["linux_parameters"])
            check_type(argname="argument logging", value=logging, expected_type=type_hints["logging"])
            check_type(argname="argument readonly_root_filesystem", value=readonly_root_filesystem, expected_type=type_hints["readonly_root_filesystem"])
            check_type(argname="argument secrets", value=secrets, expected_type=type_hints["secrets"])
            check_type(argname="argument user", value=user, expected_type=type_hints["user"])
            check_type(argname="argument volumes", value=volumes, expected_type=type_hints["volumes"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cpu": cpu,
            "image": image,
            "memory": memory,
        }
        if command is not None:
            self._values["command"] = command
        if environment is not None:
            self._values["environment"] = environment
        if execution_role is not None:
            self._values["execution_role"] = execution_role
        if job_role is not None:
            self._values["job_role"] = job_role
        if linux_parameters is not None:
            self._values["linux_parameters"] = linux_parameters
        if logging is not None:
            self._values["logging"] = logging
        if readonly_root_filesystem is not None:
            self._values["readonly_root_filesystem"] = readonly_root_filesystem
        if secrets is not None:
            self._values["secrets"] = secrets
        if user is not None:
            self._values["user"] = user
        if volumes is not None:
            self._values["volumes"] = volumes

    @builtins.property
    def cpu(self) -> jsii.Number:
        '''The number of vCPUs reserved for the container.

        Each vCPU is equivalent to 1,024 CPU shares.
        For containers running on EC2 resources, you must specify at least one vCPU.
        '''
        result = self._values.get("cpu")
        assert result is not None, "Required property 'cpu' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def image(self) -> _ContainerImage_94af1b43:
        '''The image that this container will run.'''
        result = self._values.get("image")
        assert result is not None, "Required property 'image' is missing"
        return typing.cast(_ContainerImage_94af1b43, result)

    @builtins.property
    def memory(self) -> _Size_7b441c34:
        '''The memory hard limit present to the container.

        If your container attempts to exceed the memory specified, the container is terminated.
        You must specify at least 4 MiB of memory for a job.
        '''
        result = self._values.get("memory")
        assert result is not None, "Required property 'memory' is missing"
        return typing.cast(_Size_7b441c34, result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The command that's passed to the container.

        :default: - no command

        :see: https://docs.docker.com/engine/reference/builder/#cmd
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to pass to a container.

        Cannot start with ``AWS_BATCH``.
        We don't recommend using plaintext environment variables for sensitive information, such as credential data.

        :default: - no environment variables
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def execution_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role used by Amazon ECS container and AWS Fargate agents to make AWS API calls on your behalf.

        :default: - a Role will be created

        :see: https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html
        '''
        result = self._values.get("execution_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def job_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role that the container can assume.

        :default: - no job role

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html
        '''
        result = self._values.get("job_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def linux_parameters(self) -> typing.Optional["LinuxParameters"]:
        '''Linux-specific modifications that are applied to the container, such as details for device mappings.

        :default: none
        '''
        result = self._values.get("linux_parameters")
        return typing.cast(typing.Optional["LinuxParameters"], result)

    @builtins.property
    def logging(self) -> typing.Optional[_LogDriver_393a21bb]:
        '''The loging configuration for this Job.

        :default: - the log configuration of the Docker daemon
        '''
        result = self._values.get("logging")
        return typing.cast(typing.Optional[_LogDriver_393a21bb], result)

    @builtins.property
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''Gives the container readonly access to its root filesystem.

        :default: false
        '''
        result = self._values.get("readonly_root_filesystem")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def secrets(self) -> typing.Optional[typing.Mapping[builtins.str, "Secret"]]:
        '''A map from environment variable names to the secrets for the container.

        Allows your job definitions
        to reference the secret by the environment variable name defined in this property.

        :default: - no secrets

        :see: https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html
        '''
        result = self._values.get("secrets")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "Secret"]], result)

    @builtins.property
    def user(self) -> typing.Optional[builtins.str]:
        '''The user name to use inside the container.

        :default: - no user
        '''
        result = self._values.get("user")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def volumes(self) -> typing.Optional[typing.List["EcsVolume"]]:
        '''The volumes to mount to this container.

        Automatically added to the job definition.

        :default: - no volumes
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.List["EcsVolume"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsContainerDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.EcsEc2ContainerDefinitionProps",
    jsii_struct_bases=[EcsContainerDefinitionProps],
    name_mapping={
        "cpu": "cpu",
        "image": "image",
        "memory": "memory",
        "command": "command",
        "environment": "environment",
        "execution_role": "executionRole",
        "job_role": "jobRole",
        "linux_parameters": "linuxParameters",
        "logging": "logging",
        "readonly_root_filesystem": "readonlyRootFilesystem",
        "secrets": "secrets",
        "user": "user",
        "volumes": "volumes",
        "gpu": "gpu",
        "privileged": "privileged",
        "ulimits": "ulimits",
    },
)
class EcsEc2ContainerDefinitionProps(EcsContainerDefinitionProps):
    def __init__(
        self,
        *,
        cpu: jsii.Number,
        image: _ContainerImage_94af1b43,
        memory: _Size_7b441c34,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        execution_role: typing.Optional[_IRole_235f5d8e] = None,
        job_role: typing.Optional[_IRole_235f5d8e] = None,
        linux_parameters: typing.Optional["LinuxParameters"] = None,
        logging: typing.Optional[_LogDriver_393a21bb] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        secrets: typing.Optional[typing.Mapping[builtins.str, "Secret"]] = None,
        user: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Sequence["EcsVolume"]] = None,
        gpu: typing.Optional[jsii.Number] = None,
        privileged: typing.Optional[builtins.bool] = None,
        ulimits: typing.Optional[typing.Sequence[typing.Union["Ulimit", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Props to configure an EcsEc2ContainerDefinition.

        :param cpu: The number of vCPUs reserved for the container. Each vCPU is equivalent to 1,024 CPU shares. For containers running on EC2 resources, you must specify at least one vCPU.
        :param image: The image that this container will run.
        :param memory: The memory hard limit present to the container. If your container attempts to exceed the memory specified, the container is terminated. You must specify at least 4 MiB of memory for a job.
        :param command: The command that's passed to the container. Default: - no command
        :param environment: The environment variables to pass to a container. Cannot start with ``AWS_BATCH``. We don't recommend using plaintext environment variables for sensitive information, such as credential data. Default: - no environment variables
        :param execution_role: The role used by Amazon ECS container and AWS Fargate agents to make AWS API calls on your behalf. Default: - a Role will be created
        :param job_role: The role that the container can assume. Default: - no job role
        :param linux_parameters: Linux-specific modifications that are applied to the container, such as details for device mappings. Default: none
        :param logging: The loging configuration for this Job. Default: - the log configuration of the Docker daemon
        :param readonly_root_filesystem: Gives the container readonly access to its root filesystem. Default: false
        :param secrets: A map from environment variable names to the secrets for the container. Allows your job definitions to reference the secret by the environment variable name defined in this property. Default: - no secrets
        :param user: The user name to use inside the container. Default: - no user
        :param volumes: The volumes to mount to this container. Automatically added to the job definition. Default: - no volumes
        :param gpu: The number of physical GPUs to reserve for the container. Make sure that the number of GPUs reserved for all containers in a job doesn't exceed the number of available GPUs on the compute resource that the job is launched on. Default: - no gpus
        :param privileged: When this parameter is true, the container is given elevated permissions on the host container instance (similar to the root user). Default: false
        :param ulimits: Limits to set for the user this docker container will run as. Default: - no ulimits

        :exampleMetadata: infused

        Example::

            # vpc: ec2.IVpc
            
            
            ecs_job = batch.EcsJobDefinition(self, "JobDefn",
                container=batch.EcsEc2ContainerDefinition(self, "containerDefn",
                    image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
                    memory=cdk.Size.mebibytes(2048),
                    cpu=256
                )
            )
            
            queue = batch.JobQueue(self, "JobQueue",
                compute_environments=[batch.OrderedComputeEnvironment(
                    compute_environment=batch.ManagedEc2EcsComputeEnvironment(self, "managedEc2CE",
                        vpc=vpc
                    ),
                    order=1
                )],
                priority=10
            )
            
            user = iam.User(self, "MyUser")
            ecs_job.grant_submit_job(user, queue)
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__397c7ba6003633e0c17fd25493d817d40262ed9fcf2688d3147876a3327e8249)
            check_type(argname="argument cpu", value=cpu, expected_type=type_hints["cpu"])
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
            check_type(argname="argument memory", value=memory, expected_type=type_hints["memory"])
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument execution_role", value=execution_role, expected_type=type_hints["execution_role"])
            check_type(argname="argument job_role", value=job_role, expected_type=type_hints["job_role"])
            check_type(argname="argument linux_parameters", value=linux_parameters, expected_type=type_hints["linux_parameters"])
            check_type(argname="argument logging", value=logging, expected_type=type_hints["logging"])
            check_type(argname="argument readonly_root_filesystem", value=readonly_root_filesystem, expected_type=type_hints["readonly_root_filesystem"])
            check_type(argname="argument secrets", value=secrets, expected_type=type_hints["secrets"])
            check_type(argname="argument user", value=user, expected_type=type_hints["user"])
            check_type(argname="argument volumes", value=volumes, expected_type=type_hints["volumes"])
            check_type(argname="argument gpu", value=gpu, expected_type=type_hints["gpu"])
            check_type(argname="argument privileged", value=privileged, expected_type=type_hints["privileged"])
            check_type(argname="argument ulimits", value=ulimits, expected_type=type_hints["ulimits"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cpu": cpu,
            "image": image,
            "memory": memory,
        }
        if command is not None:
            self._values["command"] = command
        if environment is not None:
            self._values["environment"] = environment
        if execution_role is not None:
            self._values["execution_role"] = execution_role
        if job_role is not None:
            self._values["job_role"] = job_role
        if linux_parameters is not None:
            self._values["linux_parameters"] = linux_parameters
        if logging is not None:
            self._values["logging"] = logging
        if readonly_root_filesystem is not None:
            self._values["readonly_root_filesystem"] = readonly_root_filesystem
        if secrets is not None:
            self._values["secrets"] = secrets
        if user is not None:
            self._values["user"] = user
        if volumes is not None:
            self._values["volumes"] = volumes
        if gpu is not None:
            self._values["gpu"] = gpu
        if privileged is not None:
            self._values["privileged"] = privileged
        if ulimits is not None:
            self._values["ulimits"] = ulimits

    @builtins.property
    def cpu(self) -> jsii.Number:
        '''The number of vCPUs reserved for the container.

        Each vCPU is equivalent to 1,024 CPU shares.
        For containers running on EC2 resources, you must specify at least one vCPU.
        '''
        result = self._values.get("cpu")
        assert result is not None, "Required property 'cpu' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def image(self) -> _ContainerImage_94af1b43:
        '''The image that this container will run.'''
        result = self._values.get("image")
        assert result is not None, "Required property 'image' is missing"
        return typing.cast(_ContainerImage_94af1b43, result)

    @builtins.property
    def memory(self) -> _Size_7b441c34:
        '''The memory hard limit present to the container.

        If your container attempts to exceed the memory specified, the container is terminated.
        You must specify at least 4 MiB of memory for a job.
        '''
        result = self._values.get("memory")
        assert result is not None, "Required property 'memory' is missing"
        return typing.cast(_Size_7b441c34, result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The command that's passed to the container.

        :default: - no command

        :see: https://docs.docker.com/engine/reference/builder/#cmd
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to pass to a container.

        Cannot start with ``AWS_BATCH``.
        We don't recommend using plaintext environment variables for sensitive information, such as credential data.

        :default: - no environment variables
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def execution_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role used by Amazon ECS container and AWS Fargate agents to make AWS API calls on your behalf.

        :default: - a Role will be created

        :see: https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html
        '''
        result = self._values.get("execution_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def job_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role that the container can assume.

        :default: - no job role

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html
        '''
        result = self._values.get("job_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def linux_parameters(self) -> typing.Optional["LinuxParameters"]:
        '''Linux-specific modifications that are applied to the container, such as details for device mappings.

        :default: none
        '''
        result = self._values.get("linux_parameters")
        return typing.cast(typing.Optional["LinuxParameters"], result)

    @builtins.property
    def logging(self) -> typing.Optional[_LogDriver_393a21bb]:
        '''The loging configuration for this Job.

        :default: - the log configuration of the Docker daemon
        '''
        result = self._values.get("logging")
        return typing.cast(typing.Optional[_LogDriver_393a21bb], result)

    @builtins.property
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''Gives the container readonly access to its root filesystem.

        :default: false
        '''
        result = self._values.get("readonly_root_filesystem")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def secrets(self) -> typing.Optional[typing.Mapping[builtins.str, "Secret"]]:
        '''A map from environment variable names to the secrets for the container.

        Allows your job definitions
        to reference the secret by the environment variable name defined in this property.

        :default: - no secrets

        :see: https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html
        '''
        result = self._values.get("secrets")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "Secret"]], result)

    @builtins.property
    def user(self) -> typing.Optional[builtins.str]:
        '''The user name to use inside the container.

        :default: - no user
        '''
        result = self._values.get("user")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def volumes(self) -> typing.Optional[typing.List["EcsVolume"]]:
        '''The volumes to mount to this container.

        Automatically added to the job definition.

        :default: - no volumes
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.List["EcsVolume"]], result)

    @builtins.property
    def gpu(self) -> typing.Optional[jsii.Number]:
        '''The number of physical GPUs to reserve for the container.

        Make sure that the number of GPUs reserved for all containers in a job doesn't exceed
        the number of available GPUs on the compute resource that the job is launched on.

        :default: - no gpus
        '''
        result = self._values.get("gpu")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def privileged(self) -> typing.Optional[builtins.bool]:
        '''When this parameter is true, the container is given elevated permissions on the host container instance (similar to the root user).

        :default: false
        '''
        result = self._values.get("privileged")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def ulimits(self) -> typing.Optional[typing.List["Ulimit"]]:
        '''Limits to set for the user this docker container will run as.

        :default: - no ulimits
        '''
        result = self._values.get("ulimits")
        return typing.cast(typing.Optional[typing.List["Ulimit"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsEc2ContainerDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.EcsFargateContainerDefinitionProps",
    jsii_struct_bases=[EcsContainerDefinitionProps],
    name_mapping={
        "cpu": "cpu",
        "image": "image",
        "memory": "memory",
        "command": "command",
        "environment": "environment",
        "execution_role": "executionRole",
        "job_role": "jobRole",
        "linux_parameters": "linuxParameters",
        "logging": "logging",
        "readonly_root_filesystem": "readonlyRootFilesystem",
        "secrets": "secrets",
        "user": "user",
        "volumes": "volumes",
        "assign_public_ip": "assignPublicIp",
        "ephemeral_storage_size": "ephemeralStorageSize",
        "fargate_cpu_architecture": "fargateCpuArchitecture",
        "fargate_operating_system_family": "fargateOperatingSystemFamily",
        "fargate_platform_version": "fargatePlatformVersion",
    },
)
class EcsFargateContainerDefinitionProps(EcsContainerDefinitionProps):
    def __init__(
        self,
        *,
        cpu: jsii.Number,
        image: _ContainerImage_94af1b43,
        memory: _Size_7b441c34,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        execution_role: typing.Optional[_IRole_235f5d8e] = None,
        job_role: typing.Optional[_IRole_235f5d8e] = None,
        linux_parameters: typing.Optional["LinuxParameters"] = None,
        logging: typing.Optional[_LogDriver_393a21bb] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        secrets: typing.Optional[typing.Mapping[builtins.str, "Secret"]] = None,
        user: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Sequence["EcsVolume"]] = None,
        assign_public_ip: typing.Optional[builtins.bool] = None,
        ephemeral_storage_size: typing.Optional[_Size_7b441c34] = None,
        fargate_cpu_architecture: typing.Optional[_CpuArchitecture_d27dee43] = None,
        fargate_operating_system_family: typing.Optional[_OperatingSystemFamily_b23afb56] = None,
        fargate_platform_version: typing.Optional[_FargatePlatformVersion_55d8be5c] = None,
    ) -> None:
        '''Props to configure an EcsFargateContainerDefinition.

        :param cpu: The number of vCPUs reserved for the container. Each vCPU is equivalent to 1,024 CPU shares. For containers running on EC2 resources, you must specify at least one vCPU.
        :param image: The image that this container will run.
        :param memory: The memory hard limit present to the container. If your container attempts to exceed the memory specified, the container is terminated. You must specify at least 4 MiB of memory for a job.
        :param command: The command that's passed to the container. Default: - no command
        :param environment: The environment variables to pass to a container. Cannot start with ``AWS_BATCH``. We don't recommend using plaintext environment variables for sensitive information, such as credential data. Default: - no environment variables
        :param execution_role: The role used by Amazon ECS container and AWS Fargate agents to make AWS API calls on your behalf. Default: - a Role will be created
        :param job_role: The role that the container can assume. Default: - no job role
        :param linux_parameters: Linux-specific modifications that are applied to the container, such as details for device mappings. Default: none
        :param logging: The loging configuration for this Job. Default: - the log configuration of the Docker daemon
        :param readonly_root_filesystem: Gives the container readonly access to its root filesystem. Default: false
        :param secrets: A map from environment variable names to the secrets for the container. Allows your job definitions to reference the secret by the environment variable name defined in this property. Default: - no secrets
        :param user: The user name to use inside the container. Default: - no user
        :param volumes: The volumes to mount to this container. Automatically added to the job definition. Default: - no volumes
        :param assign_public_ip: Indicates whether the job has a public IP address. For a job that's running on Fargate resources in a private subnet to send outbound traffic to the internet (for example, to pull container images), the private subnet requires a NAT gateway be attached to route requests to the internet. Default: false
        :param ephemeral_storage_size: The size for ephemeral storage. Default: - 20 GiB
        :param fargate_cpu_architecture: The vCPU architecture of Fargate Runtime. Default: - X86_64
        :param fargate_operating_system_family: The operating system for the compute environment. Default: - LINUX
        :param fargate_platform_version: Which version of Fargate to use when running this container. Default: LATEST

        :exampleMetadata: infused

        Example::

            job_defn = batch.EcsJobDefinition(self, "JobDefn",
                container=batch.EcsFargateContainerDefinition(self, "myFargateContainer",
                    image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
                    memory=cdk.Size.mebibytes(2048),
                    cpu=256,
                    ephemeral_storage_size=cdk.Size.gibibytes(100),
                    fargate_cpu_architecture=ecs.CpuArchitecture.ARM64,
                    fargate_operating_system_family=ecs.OperatingSystemFamily.LINUX
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e5c078c2a3d8816b92a00e4027344466064444955323dc2ff127e9c012fac716)
            check_type(argname="argument cpu", value=cpu, expected_type=type_hints["cpu"])
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
            check_type(argname="argument memory", value=memory, expected_type=type_hints["memory"])
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
            check_type(argname="argument environment", value=environment, expected_type=type_hints["environment"])
            check_type(argname="argument execution_role", value=execution_role, expected_type=type_hints["execution_role"])
            check_type(argname="argument job_role", value=job_role, expected_type=type_hints["job_role"])
            check_type(argname="argument linux_parameters", value=linux_parameters, expected_type=type_hints["linux_parameters"])
            check_type(argname="argument logging", value=logging, expected_type=type_hints["logging"])
            check_type(argname="argument readonly_root_filesystem", value=readonly_root_filesystem, expected_type=type_hints["readonly_root_filesystem"])
            check_type(argname="argument secrets", value=secrets, expected_type=type_hints["secrets"])
            check_type(argname="argument user", value=user, expected_type=type_hints["user"])
            check_type(argname="argument volumes", value=volumes, expected_type=type_hints["volumes"])
            check_type(argname="argument assign_public_ip", value=assign_public_ip, expected_type=type_hints["assign_public_ip"])
            check_type(argname="argument ephemeral_storage_size", value=ephemeral_storage_size, expected_type=type_hints["ephemeral_storage_size"])
            check_type(argname="argument fargate_cpu_architecture", value=fargate_cpu_architecture, expected_type=type_hints["fargate_cpu_architecture"])
            check_type(argname="argument fargate_operating_system_family", value=fargate_operating_system_family, expected_type=type_hints["fargate_operating_system_family"])
            check_type(argname="argument fargate_platform_version", value=fargate_platform_version, expected_type=type_hints["fargate_platform_version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cpu": cpu,
            "image": image,
            "memory": memory,
        }
        if command is not None:
            self._values["command"] = command
        if environment is not None:
            self._values["environment"] = environment
        if execution_role is not None:
            self._values["execution_role"] = execution_role
        if job_role is not None:
            self._values["job_role"] = job_role
        if linux_parameters is not None:
            self._values["linux_parameters"] = linux_parameters
        if logging is not None:
            self._values["logging"] = logging
        if readonly_root_filesystem is not None:
            self._values["readonly_root_filesystem"] = readonly_root_filesystem
        if secrets is not None:
            self._values["secrets"] = secrets
        if user is not None:
            self._values["user"] = user
        if volumes is not None:
            self._values["volumes"] = volumes
        if assign_public_ip is not None:
            self._values["assign_public_ip"] = assign_public_ip
        if ephemeral_storage_size is not None:
            self._values["ephemeral_storage_size"] = ephemeral_storage_size
        if fargate_cpu_architecture is not None:
            self._values["fargate_cpu_architecture"] = fargate_cpu_architecture
        if fargate_operating_system_family is not None:
            self._values["fargate_operating_system_family"] = fargate_operating_system_family
        if fargate_platform_version is not None:
            self._values["fargate_platform_version"] = fargate_platform_version

    @builtins.property
    def cpu(self) -> jsii.Number:
        '''The number of vCPUs reserved for the container.

        Each vCPU is equivalent to 1,024 CPU shares.
        For containers running on EC2 resources, you must specify at least one vCPU.
        '''
        result = self._values.get("cpu")
        assert result is not None, "Required property 'cpu' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def image(self) -> _ContainerImage_94af1b43:
        '''The image that this container will run.'''
        result = self._values.get("image")
        assert result is not None, "Required property 'image' is missing"
        return typing.cast(_ContainerImage_94af1b43, result)

    @builtins.property
    def memory(self) -> _Size_7b441c34:
        '''The memory hard limit present to the container.

        If your container attempts to exceed the memory specified, the container is terminated.
        You must specify at least 4 MiB of memory for a job.
        '''
        result = self._values.get("memory")
        assert result is not None, "Required property 'memory' is missing"
        return typing.cast(_Size_7b441c34, result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The command that's passed to the container.

        :default: - no command

        :see: https://docs.docker.com/engine/reference/builder/#cmd
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to pass to a container.

        Cannot start with ``AWS_BATCH``.
        We don't recommend using plaintext environment variables for sensitive information, such as credential data.

        :default: - no environment variables
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def execution_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role used by Amazon ECS container and AWS Fargate agents to make AWS API calls on your behalf.

        :default: - a Role will be created

        :see: https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html
        '''
        result = self._values.get("execution_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def job_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role that the container can assume.

        :default: - no job role

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html
        '''
        result = self._values.get("job_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def linux_parameters(self) -> typing.Optional["LinuxParameters"]:
        '''Linux-specific modifications that are applied to the container, such as details for device mappings.

        :default: none
        '''
        result = self._values.get("linux_parameters")
        return typing.cast(typing.Optional["LinuxParameters"], result)

    @builtins.property
    def logging(self) -> typing.Optional[_LogDriver_393a21bb]:
        '''The loging configuration for this Job.

        :default: - the log configuration of the Docker daemon
        '''
        result = self._values.get("logging")
        return typing.cast(typing.Optional[_LogDriver_393a21bb], result)

    @builtins.property
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''Gives the container readonly access to its root filesystem.

        :default: false
        '''
        result = self._values.get("readonly_root_filesystem")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def secrets(self) -> typing.Optional[typing.Mapping[builtins.str, "Secret"]]:
        '''A map from environment variable names to the secrets for the container.

        Allows your job definitions
        to reference the secret by the environment variable name defined in this property.

        :default: - no secrets

        :see: https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html
        '''
        result = self._values.get("secrets")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "Secret"]], result)

    @builtins.property
    def user(self) -> typing.Optional[builtins.str]:
        '''The user name to use inside the container.

        :default: - no user
        '''
        result = self._values.get("user")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def volumes(self) -> typing.Optional[typing.List["EcsVolume"]]:
        '''The volumes to mount to this container.

        Automatically added to the job definition.

        :default: - no volumes
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.List["EcsVolume"]], result)

    @builtins.property
    def assign_public_ip(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether the job has a public IP address.

        For a job that's running on Fargate resources in a private subnet to send outbound traffic to the internet
        (for example, to pull container images), the private subnet requires a NAT gateway be attached to route requests to the internet.

        :default: false

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html
        '''
        result = self._values.get("assign_public_ip")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def ephemeral_storage_size(self) -> typing.Optional[_Size_7b441c34]:
        '''The size for ephemeral storage.

        :default: - 20 GiB
        '''
        result = self._values.get("ephemeral_storage_size")
        return typing.cast(typing.Optional[_Size_7b441c34], result)

    @builtins.property
    def fargate_cpu_architecture(self) -> typing.Optional[_CpuArchitecture_d27dee43]:
        '''The vCPU architecture of Fargate Runtime.

        :default: - X86_64
        '''
        result = self._values.get("fargate_cpu_architecture")
        return typing.cast(typing.Optional[_CpuArchitecture_d27dee43], result)

    @builtins.property
    def fargate_operating_system_family(
        self,
    ) -> typing.Optional[_OperatingSystemFamily_b23afb56]:
        '''The operating system for the compute environment.

        :default: - LINUX
        '''
        result = self._values.get("fargate_operating_system_family")
        return typing.cast(typing.Optional[_OperatingSystemFamily_b23afb56], result)

    @builtins.property
    def fargate_platform_version(
        self,
    ) -> typing.Optional[_FargatePlatformVersion_55d8be5c]:
        '''Which version of Fargate to use when running this container.

        :default: LATEST
        '''
        result = self._values.get("fargate_platform_version")
        return typing.cast(typing.Optional[_FargatePlatformVersion_55d8be5c], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsFargateContainerDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.EcsMachineImage",
    jsii_struct_bases=[],
    name_mapping={"image": "image", "image_type": "imageType"},
)
class EcsMachineImage:
    def __init__(
        self,
        *,
        image: typing.Optional[_IMachineImage_0e8bd50b] = None,
        image_type: typing.Optional["EcsMachineImageType"] = None,
    ) -> None:
        '''A Batch MachineImage that is compatible with ECS.

        :param image: The machine image to use. Default: - chosen by batch
        :param image_type: Tells Batch which instance type to launch this image on. Default: - 'ECS_AL2' for non-gpu instances, 'ECS_AL2_NVIDIA' for gpu instances

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_batch as batch
            from aws_cdk import aws_ec2 as ec2
            
            # machine_image: ec2.IMachineImage
            
            ecs_machine_image = batch.EcsMachineImage(
                image=machine_image,
                image_type=batch.EcsMachineImageType.ECS_AL2
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__26e759f5b0df2446c2d8b0402ec1e6d5a15a978083317bb3031bb21d02d31179)
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
            check_type(argname="argument image_type", value=image_type, expected_type=type_hints["image_type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if image is not None:
            self._values["image"] = image
        if image_type is not None:
            self._values["image_type"] = image_type

    @builtins.property
    def image(self) -> typing.Optional[_IMachineImage_0e8bd50b]:
        '''The machine image to use.

        :default: - chosen by batch
        '''
        result = self._values.get("image")
        return typing.cast(typing.Optional[_IMachineImage_0e8bd50b], result)

    @builtins.property
    def image_type(self) -> typing.Optional["EcsMachineImageType"]:
        '''Tells Batch which instance type to launch this image on.

        :default: - 'ECS_AL2' for non-gpu instances, 'ECS_AL2_NVIDIA' for gpu instances
        '''
        result = self._values.get("image_type")
        return typing.cast(typing.Optional["EcsMachineImageType"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsMachineImage(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_batch.EcsMachineImageType")
class EcsMachineImageType(enum.Enum):
    '''Maps the image to instance types.

    :exampleMetadata: infused

    Example::

        # vpc: ec2.IVpc
        
        
        batch.ManagedEc2EcsComputeEnvironment(self, "myEc2ComputeEnv",
            vpc=vpc,
            images=[batch.EcsMachineImage(
                image_type=batch.EcsMachineImageType.ECS_AL2023
            )
            ]
        )
    '''

    ECS_AL2 = "ECS_AL2"
    '''Tells Batch that this machine image runs on non-GPU AL2 instances.'''
    ECS_AL2023 = "ECS_AL2023"
    '''Tells Batch that this machine image runs on non-GPU AL2023 instances.

    Amazon Linux 2023 does not support A1 instances.
    '''
    ECS_AL2_NVIDIA = "ECS_AL2_NVIDIA"
    '''Tells Batch that this machine image runs on GPU instances.'''


class EcsVolume(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_batch.EcsVolume",
):
    '''Represents a Volume that can be mounted to a container that uses ECS.

    :exampleMetadata: infused

    Example::

        # my_file_system: efs.IFileSystem
        # my_job_role: iam.Role
        
        my_file_system.grant_read(my_job_role)
        
        job_defn = batch.EcsJobDefinition(self, "JobDefn",
            container=batch.EcsEc2ContainerDefinition(self, "containerDefn",
                image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
                memory=cdk.Size.mebibytes(2048),
                cpu=256,
                volumes=[batch.EcsVolume.efs(
                    name="myVolume",
                    file_system=my_file_system,
                    container_path="/Volumes/myVolume",
                    use_job_role=True
                )],
                job_role=my_job_role
            )
        )
    '''

    def __init__(
        self,
        *,
        container_path: builtins.str,
        name: builtins.str,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param container_path: the path on the container where this volume is mounted.
        :param name: the name of this volume.
        :param readonly: if set, the container will have readonly access to the volume. Default: false
        '''
        options = EcsVolumeOptions(
            container_path=container_path, name=name, readonly=readonly
        )

        jsii.create(self.__class__, self, [options])

    @jsii.member(jsii_name="efs")
    @builtins.classmethod
    def efs(
        cls,
        *,
        file_system: _IFileSystem_b2d3a7cb,
        access_point_id: typing.Optional[builtins.str] = None,
        enable_transit_encryption: typing.Optional[builtins.bool] = None,
        root_directory: typing.Optional[builtins.str] = None,
        transit_encryption_port: typing.Optional[jsii.Number] = None,
        use_job_role: typing.Optional[builtins.bool] = None,
        container_path: builtins.str,
        name: builtins.str,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> "EfsVolume":
        '''Creates a Volume that uses an AWS Elastic File System (EFS);

        this volume can grow and shrink as needed

        :param file_system: The EFS File System that supports this volume.
        :param access_point_id: The Amazon EFS access point ID to use. If an access point is specified, ``rootDirectory`` must either be omitted or set to ``/`` which enforces the path set on the EFS access point. If an access point is used, ``enableTransitEncryption`` must be ``true``. Default: - no accessPointId
        :param enable_transit_encryption: Enables encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon EFS server. Default: false
        :param root_directory: The directory within the Amazon EFS file system to mount as the root directory inside the host. If this parameter is omitted, the root of the Amazon EFS volume is used instead. Specifying ``/`` has the same effect as omitting this parameter. The maximum length is 4,096 characters. Default: - root of the EFS File System
        :param transit_encryption_port: The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server. The value must be between 0 and 65,535. Default: - chosen by the EFS Mount Helper
        :param use_job_role: Whether or not to use the AWS Batch job IAM role defined in a job definition when mounting the Amazon EFS file system. If specified, ``enableTransitEncryption`` must be ``true``. Default: false
        :param container_path: the path on the container where this volume is mounted.
        :param name: the name of this volume.
        :param readonly: if set, the container will have readonly access to the volume. Default: false

        :see: https://docs.aws.amazon.com/batch/latest/userguide/efs-volumes.html
        '''
        options = EfsVolumeOptions(
            file_system=file_system,
            access_point_id=access_point_id,
            enable_transit_encryption=enable_transit_encryption,
            root_directory=root_directory,
            transit_encryption_port=transit_encryption_port,
            use_job_role=use_job_role,
            container_path=container_path,
            name=name,
            readonly=readonly,
        )

        return typing.cast("EfsVolume", jsii.sinvoke(cls, "efs", [options]))

    @jsii.member(jsii_name="host")
    @builtins.classmethod
    def host(
        cls,
        *,
        host_path: typing.Optional[builtins.str] = None,
        container_path: builtins.str,
        name: builtins.str,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> "HostVolume":
        '''Creates a Host volume.

        This volume will persist on the host at the specified ``hostPath``.
        If the ``hostPath`` is not specified, Docker will choose the host path. In this case,
        the data may not persist after the containers that use it stop running.

        :param host_path: The path on the host machine this container will have access to. Default: - Docker will choose the host path. The data may not persist after the containers that use it stop running.
        :param container_path: the path on the container where this volume is mounted.
        :param name: the name of this volume.
        :param readonly: if set, the container will have readonly access to the volume. Default: false
        '''
        options = HostVolumeOptions(
            host_path=host_path,
            container_path=container_path,
            name=name,
            readonly=readonly,
        )

        return typing.cast("HostVolume", jsii.sinvoke(cls, "host", [options]))

    @builtins.property
    @jsii.member(jsii_name="containerPath")
    def container_path(self) -> builtins.str:
        '''The path on the container that this volume will be mounted to.'''
        return typing.cast(builtins.str, jsii.get(self, "containerPath"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of this volume.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @builtins.property
    @jsii.member(jsii_name="readonly")
    def readonly(self) -> typing.Optional[builtins.bool]:
        '''Whether or not the container has readonly access to this volume.

        :default: false
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "readonly"))


class _EcsVolumeProxy(EcsVolume):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, EcsVolume).__jsii_proxy_class__ = lambda : _EcsVolumeProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.EcsVolumeOptions",
    jsii_struct_bases=[],
    name_mapping={
        "container_path": "containerPath",
        "name": "name",
        "readonly": "readonly",
    },
)
class EcsVolumeOptions:
    def __init__(
        self,
        *,
        container_path: builtins.str,
        name: builtins.str,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options to configure an EcsVolume.

        :param container_path: the path on the container where this volume is mounted.
        :param name: the name of this volume.
        :param readonly: if set, the container will have readonly access to the volume. Default: false

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_batch as batch
            
            ecs_volume_options = batch.EcsVolumeOptions(
                container_path="containerPath",
                name="name",
            
                # the properties below are optional
                readonly=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__daa21acea8bfb80b6ec7eb6c5af88db3c9e1cf8f21192c9c35289bd5fff6d626)
            check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument readonly", value=readonly, expected_type=type_hints["readonly"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "container_path": container_path,
            "name": name,
        }
        if readonly is not None:
            self._values["readonly"] = readonly

    @builtins.property
    def container_path(self) -> builtins.str:
        '''the path on the container where this volume is mounted.'''
        result = self._values.get("container_path")
        assert result is not None, "Required property 'container_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''the name of this volume.'''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def readonly(self) -> typing.Optional[builtins.bool]:
        '''if set, the container will have readonly access to the volume.

        :default: false
        '''
        result = self._values.get("readonly")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsVolumeOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EfsVolume(
    EcsVolume,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_batch.EfsVolume",
):
    '''A Volume that uses an AWS Elastic File System (EFS);

    this volume can grow and shrink as needed

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_batch as batch
        from aws_cdk import aws_efs as efs
        
        # file_system: efs.FileSystem
        
        efs_volume = batch.EfsVolume(
            container_path="containerPath",
            file_system=file_system,
            name="name",
        
            # the properties below are optional
            access_point_id="accessPointId",
            enable_transit_encryption=False,
            readonly=False,
            root_directory="rootDirectory",
            transit_encryption_port=123,
            use_job_role=False
        )
    '''

    def __init__(
        self,
        *,
        file_system: _IFileSystem_b2d3a7cb,
        access_point_id: typing.Optional[builtins.str] = None,
        enable_transit_encryption: typing.Optional[builtins.bool] = None,
        root_directory: typing.Optional[builtins.str] = None,
        transit_encryption_port: typing.Optional[jsii.Number] = None,
        use_job_role: typing.Optional[builtins.bool] = None,
        container_path: builtins.str,
        name: builtins.str,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param file_system: The EFS File System that supports this volume.
        :param access_point_id: The Amazon EFS access point ID to use. If an access point is specified, ``rootDirectory`` must either be omitted or set to ``/`` which enforces the path set on the EFS access point. If an access point is used, ``enableTransitEncryption`` must be ``true``. Default: - no accessPointId
        :param enable_transit_encryption: Enables encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon EFS server. Default: false
        :param root_directory: The directory within the Amazon EFS file system to mount as the root directory inside the host. If this parameter is omitted, the root of the Amazon EFS volume is used instead. Specifying ``/`` has the same effect as omitting this parameter. The maximum length is 4,096 characters. Default: - root of the EFS File System
        :param transit_encryption_port: The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server. The value must be between 0 and 65,535. Default: - chosen by the EFS Mount Helper
        :param use_job_role: Whether or not to use the AWS Batch job IAM role defined in a job definition when mounting the Amazon EFS file system. If specified, ``enableTransitEncryption`` must be ``true``. Default: false
        :param container_path: the path on the container where this volume is mounted.
        :param name: the name of this volume.
        :param readonly: if set, the container will have readonly access to the volume. Default: false
        '''
        options = EfsVolumeOptions(
            file_system=file_system,
            access_point_id=access_point_id,
            enable_transit_encryption=enable_transit_encryption,
            root_directory=root_directory,
            transit_encryption_port=transit_encryption_port,
            use_job_role=use_job_role,
            container_path=container_path,
            name=name,
            readonly=readonly,
        )

        jsii.create(self.__class__, self, [options])

    @jsii.member(jsii_name="isEfsVolume")
    @builtins.classmethod
    def is_efs_volume(cls, x: typing.Any) -> builtins.bool:
        '''Returns true if x is an EfsVolume, false otherwise.

        :param x: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5ddbf3eef4a2a65c90ed1f4bc083e010fd20ebfd5db29c2e459958d429750cde)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isEfsVolume", [x]))

    @builtins.property
    @jsii.member(jsii_name="fileSystem")
    def file_system(self) -> _IFileSystem_b2d3a7cb:
        '''The EFS File System that supports this volume.'''
        return typing.cast(_IFileSystem_b2d3a7cb, jsii.get(self, "fileSystem"))

    @builtins.property
    @jsii.member(jsii_name="accessPointId")
    def access_point_id(self) -> typing.Optional[builtins.str]:
        '''The Amazon EFS access point ID to use.

        If an access point is specified, ``rootDirectory`` must either be omitted or set to ``/``
        which enforces the path set on the EFS access point.
        If an access point is used, ``enableTransitEncryption`` must be ``true``.

        :default: - no accessPointId

        :see: https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "accessPointId"))

    @builtins.property
    @jsii.member(jsii_name="enableTransitEncryption")
    def enable_transit_encryption(self) -> typing.Optional[builtins.bool]:
        '''Enables encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon EFS server.

        :default: false

        :see: https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "enableTransitEncryption"))

    @builtins.property
    @jsii.member(jsii_name="rootDirectory")
    def root_directory(self) -> typing.Optional[builtins.str]:
        '''The directory within the Amazon EFS file system to mount as the root directory inside the host.

        If this parameter is omitted, the root of the Amazon EFS volume is used instead.
        Specifying ``/`` has the same effect as omitting this parameter.
        The maximum length is 4,096 characters.

        :default: - root of the EFS File System
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "rootDirectory"))

    @builtins.property
    @jsii.member(jsii_name="transitEncryptionPort")
    def transit_encryption_port(self) -> typing.Optional[jsii.Number]:
        '''The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server.

        The value must be between 0 and 65,535.

        :default: - chosen by the EFS Mount Helper

        :see: https://docs.aws.amazon.com/efs/latest/ug/efs-mount-helper.html
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "transitEncryptionPort"))

    @builtins.property
    @jsii.member(jsii_name="useJobRole")
    def use_job_role(self) -> typing.Optional[builtins.bool]:
        '''Whether or not to use the AWS Batch job IAM role defined in a job definition when mounting the Amazon EFS file system.

        If specified, ``enableTransitEncryption`` must be ``true``.

        :default: false

        :see: https://docs.aws.amazon.com/batch/latest/userguide/efs-volumes.html#efs-volume-accesspoints
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "useJobRole"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.EfsVolumeOptions",
    jsii_struct_bases=[EcsVolumeOptions],
    name_mapping={
        "container_path": "containerPath",
        "name": "name",
        "readonly": "readonly",
        "file_system": "fileSystem",
        "access_point_id": "accessPointId",
        "enable_transit_encryption": "enableTransitEncryption",
        "root_directory": "rootDirectory",
        "transit_encryption_port": "transitEncryptionPort",
        "use_job_role": "useJobRole",
    },
)
class EfsVolumeOptions(EcsVolumeOptions):
    def __init__(
        self,
        *,
        container_path: builtins.str,
        name: builtins.str,
        readonly: typing.Optional[builtins.bool] = None,
        file_system: _IFileSystem_b2d3a7cb,
        access_point_id: typing.Optional[builtins.str] = None,
        enable_transit_encryption: typing.Optional[builtins.bool] = None,
        root_directory: typing.Optional[builtins.str] = None,
        transit_encryption_port: typing.Optional[jsii.Number] = None,
        use_job_role: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for configuring an EfsVolume.

        :param container_path: the path on the container where this volume is mounted.
        :param name: the name of this volume.
        :param readonly: if set, the container will have readonly access to the volume. Default: false
        :param file_system: The EFS File System that supports this volume.
        :param access_point_id: The Amazon EFS access point ID to use. If an access point is specified, ``rootDirectory`` must either be omitted or set to ``/`` which enforces the path set on the EFS access point. If an access point is used, ``enableTransitEncryption`` must be ``true``. Default: - no accessPointId
        :param enable_transit_encryption: Enables encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon EFS server. Default: false
        :param root_directory: The directory within the Amazon EFS file system to mount as the root directory inside the host. If this parameter is omitted, the root of the Amazon EFS volume is used instead. Specifying ``/`` has the same effect as omitting this parameter. The maximum length is 4,096 characters. Default: - root of the EFS File System
        :param transit_encryption_port: The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server. The value must be between 0 and 65,535. Default: - chosen by the EFS Mount Helper
        :param use_job_role: Whether or not to use the AWS Batch job IAM role defined in a job definition when mounting the Amazon EFS file system. If specified, ``enableTransitEncryption`` must be ``true``. Default: false

        :exampleMetadata: infused

        Example::

            # my_file_system: efs.IFileSystem
            # my_job_role: iam.Role
            
            my_file_system.grant_read(my_job_role)
            
            job_defn = batch.EcsJobDefinition(self, "JobDefn",
                container=batch.EcsEc2ContainerDefinition(self, "containerDefn",
                    image=ecs.ContainerImage.from_registry("public.ecr.aws/amazonlinux/amazonlinux:latest"),
                    memory=cdk.Size.mebibytes(2048),
                    cpu=256,
                    volumes=[batch.EcsVolume.efs(
                        name="myVolume",
                        file_system=my_file_system,
                        container_path="/Volumes/myVolume",
                        use_job_role=True
                    )],
                    job_role=my_job_role
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b405b7b8103439b5c13855083d1280f8820180f747c1ac9a750a3c719853666e)
            check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument readonly", value=readonly, expected_type=type_hints["readonly"])
            check_type(argname="argument file_system", value=file_system, expected_type=type_hints["file_system"])
            check_type(argname="argument access_point_id", value=access_point_id, expected_type=type_hints["access_point_id"])
            check_type(argname="argument enable_transit_encryption", value=enable_transit_encryption, expected_type=type_hints["enable_transit_encryption"])
            check_type(argname="argument root_directory", value=root_directory, expected_type=type_hints["root_directory"])
            check_type(argname="argument transit_encryption_port", value=transit_encryption_port, expected_type=type_hints["transit_encryption_port"])
            check_type(argname="argument use_job_role", value=use_job_role, expected_type=type_hints["use_job_role"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "container_path": container_path,
            "name": name,
            "file_system": file_system,
        }
        if readonly is not None:
            self._values["readonly"] = readonly
        if access_point_id is not None:
            self._values["access_point_id"] = access_point_id
        if enable_transit_encryption is not None:
            self._values["enable_transit_encryption"] = enable_transit_encryption
        if root_directory is not None:
            self._values["root_directory"] = root_directory
        if transit_encryption_port is not None:
            self._values["transit_encryption_port"] = transit_encryption_port
        if use_job_role is not None:
            self._values["use_job_role"] = use_job_role

    @builtins.property
    def container_path(self) -> builtins.str:
        '''the path on the container where this volume is mounted.'''
        result = self._values.get("container_path")
        assert result is not None, "Required property 'container_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''the name of this volume.'''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def readonly(self) -> typing.Optional[builtins.bool]:
        '''if set, the container will have readonly access to the volume.

        :default: false
        '''
        result = self._values.get("readonly")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def file_system(self) -> _IFileSystem_b2d3a7cb:
        '''The EFS File System that supports this volume.'''
        result = self._values.get("file_system")
        assert result is not None, "Required property 'file_system' is missing"
        return typing.cast(_IFileSystem_b2d3a7cb, result)

    @builtins.property
    def access_point_id(self) -> typing.Optional[builtins.str]:
        '''The Amazon EFS access point ID to use.

        If an access point is specified, ``rootDirectory`` must either be omitted or set to ``/``
        which enforces the path set on the EFS access point.
        If an access point is used, ``enableTransitEncryption`` must be ``true``.

        :default: - no accessPointId

        :see: https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html
        '''
        result = self._values.get("access_point_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enable_transit_encryption(self) -> typing.Optional[builtins.bool]:
        '''Enables encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon EFS server.

        :default: false

        :see: https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html
        '''
        result = self._values.get("enable_transit_encryption")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def root_directory(self) -> typing.Optional[builtins.str]:
        '''The directory within the Amazon EFS file system to mount as the root directory inside the host.

        If this parameter is omitted, the root of the Amazon EFS volume is used instead.
        Specifying ``/`` has the same effect as omitting this parameter.
        The maximum length is 4,096 characters.

        :default: - root of the EFS File System
        '''
        result = self._values.get("root_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def transit_encryption_port(self) -> typing.Optional[jsii.Number]:
        '''The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server.

        The value must be between 0 and 65,535.

        :default: - chosen by the EFS Mount Helper

        :see: https://docs.aws.amazon.com/efs/latest/ug/efs-mount-helper.html
        '''
        result = self._values.get("transit_encryption_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def use_job_role(self) -> typing.Optional[builtins.bool]:
        '''Whether or not to use the AWS Batch job IAM role defined in a job definition when mounting the Amazon EFS file system.

        If specified, ``enableTransitEncryption`` must be ``true``.

        :default: false

        :see: https://docs.aws.amazon.com/batch/latest/userguide/efs-volumes.html#efs-volume-accesspoints
        '''
        result = self._values.get("use_job_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EfsVolumeOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.EksContainerDefinitionProps",
    jsii_struct_bases=[],
    name_mapping={
        "image": "image",
        "args": "args",
        "command": "command",
        "cpu_limit": "cpuLimit",
        "cpu_reservation": "cpuReservation",
        "env": "env",
        "gpu_limit": "gpuLimit",
        "gpu_reservation": "gpuReservation",
        "image_pull_policy": "imagePullPolicy",
        "memory_limit": "memoryLimit",
        "memory_reservation": "memoryReservation",
        "name": "name",
        "privileged": "privileged",
        "readonly_root_filesystem": "readonlyRootFilesystem",
        "run_as_group": "runAsGroup",
        "run_as_root": "runAsRoot",
        "run_as_user": "runAsUser",
        "volumes": "volumes",
    },
)
class EksContainerDefinitionProps:
    def __init__(
        self,
        *,
        image: _ContainerImage_94af1b43,
        args: typing.Optional[typing.Sequence[builtins.str]] = None,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        cpu_limit: typing.Optional[jsii.Number] = None,
        cpu_reservation: typing.Optional[jsii.Number] = None,
        env: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_limit: typing.Optional[jsii.Number] = None,
        gpu_reservation: typing.Optional[jsii.Number] = None,
        image_pull_policy: typing.Optional["ImagePullPolicy"] = None,
        memory_limit: typing.Optional[_Size_7b441c34] = None,
        memory_reservation: typing.Optional[_Size_7b441c34] = None,
        name: typing.Optional[builtins.str] = None,
        privileged: typing.Optional[builtins.bool] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        run_as_group: typing.Optional[jsii.Number] = None,
        run_as_root: typing.Optional[builtins.bool] = None,
        run_as_user: typing.Optional[jsii.Number] = None,
        volumes: typing.Optional[typing.Sequence["EksVolume"]] = None,
    ) -> None:
        '''Props to configure an EksContainerDefinition.

        :param image: The image that this container will run.
        :param args: An array of arguments to the entrypoint. If this isn't specified, the CMD of the container image is used. This corresponds to the args member in the Entrypoint portion of the Pod in Kubernetes. Environment variable references are expanded using the container's environment. If the referenced environment variable doesn't exist, the reference in the command isn't changed. For example, if the reference is to "$(NAME1)" and the NAME1 environment variable doesn't exist, the command string will remain "$(NAME1)." $$ is replaced with $, and the resulting string isn't expanded. or example, $$(VAR_NAME) is passed as $(VAR_NAME) whether or not the VAR_NAME environment variable exists. Default: - no args
        :param command: The entrypoint for the container. This isn't run within a shell. If this isn't specified, the ``ENTRYPOINT`` of the container image is used. Environment variable references are expanded using the container's environment. If the referenced environment variable doesn't exist, the reference in the command isn't changed. For example, if the reference is to ``"$(NAME1)"`` and the ``NAME1`` environment variable doesn't exist, the command string will remain ``"$(NAME1)."`` ``$$`` is replaced with ``$`` and the resulting string isn't expanded. For example, ``$$(VAR_NAME)`` will be passed as ``$(VAR_NAME)`` whether or not the ``VAR_NAME`` environment variable exists. The entrypoint can't be updated. Default: - no command
        :param cpu_limit: The hard limit of CPUs to present to this container. Must be an even multiple of 0.25. If your container attempts to exceed this limit, it will be terminated. At least one of ``cpuReservation`` and ``cpuLimit`` is required. If both are specified, then ``cpuLimit`` must be at least as large as ``cpuReservation``. Default: - No CPU limit
        :param cpu_reservation: The soft limit of CPUs to reserve for the container Must be an even multiple of 0.25. The container will given at least this many CPUs, but may consume more. At least one of ``cpuReservation`` and ``cpuLimit`` is required. If both are specified, then ``cpuLimit`` must be at least as large as ``cpuReservation``. Default: - No CPUs reserved
        :param env: The environment variables to pass to this container. *Note*: Environment variables cannot start with "AWS_BATCH". This naming convention is reserved for variables that AWS Batch sets. Default: - no environment variables
        :param gpu_limit: The hard limit of GPUs to present to this container. If your container attempts to exceed this limit, it will be terminated. If both ``gpuReservation`` and ``gpuLimit`` are specified, then ``gpuLimit`` must be equal to ``gpuReservation``. Default: - No GPU limit
        :param gpu_reservation: The soft limit of CPUs to reserve for the container Must be an even multiple of 0.25. The container will given at least this many CPUs, but may consume more. If both ``gpuReservation`` and ``gpuLimit`` are specified, then ``gpuLimit`` must be equal to ``gpuReservation``. Default: - No GPUs reserved
        :param image_pull_policy: The image pull policy for this container. Default: - ``ALWAYS`` if the ``:latest`` tag is specified, ``IF_NOT_PRESENT`` otherwise
        :param memory_limit: The amount (in MiB) of memory to present to the container. If your container attempts to exceed the allocated memory, it will be terminated. Must be larger that 4 MiB At least one of ``memoryLimit`` and ``memoryReservation`` is required *Note*: To maximize your resource utilization, provide your jobs with as much memory as possible for the specific instance type that you are using. Default: - No memory limit
        :param memory_reservation: The soft limit (in MiB) of memory to reserve for the container. Your container will be given at least this much memory, but may consume more. Must be larger that 4 MiB When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the memory parameter (if applicable), or all of the available memory on the container instance, whichever comes first. At least one of ``memoryLimit`` and ``memoryReservation`` is required. If both are specified, then ``memoryLimit`` must be equal to ``memoryReservation`` *Note*: To maximize your resource utilization, provide your jobs with as much memory as possible for the specific instance type that you are using. Default: - No memory reserved
        :param name: The name of this container. Default: : ``'Default'``
        :param privileged: If specified, gives this container elevated permissions on the host container instance. The level of permissions are similar to the root user permissions. This parameter maps to ``privileged`` policy in the Privileged pod security policies in the Kubernetes documentation. *Note*: this is only compatible with Kubernetes < v1.25 Default: false
        :param readonly_root_filesystem: If specified, gives this container readonly access to its root file system. This parameter maps to ``ReadOnlyRootFilesystem`` policy in the Volumes and file systems pod security policies in the Kubernetes documentation. *Note*: this is only compatible with Kubernetes < v1.25 Default: false
        :param run_as_group: If specified, the container is run as the specified group ID (``gid``). If this parameter isn't specified, the default is the group that's specified in the image metadata. This parameter maps to ``RunAsGroup`` and ``MustRunAs`` policy in the Users and groups pod security policies in the Kubernetes documentation. *Note*: this is only compatible with Kubernetes < v1.25 Default: none
        :param run_as_root: If specified, the container is run as a user with a ``uid`` other than 0. Otherwise, no such rule is enforced. This parameter maps to ``RunAsUser`` and ``MustRunAsNonRoot`` policy in the Users and groups pod security policies in the Kubernetes documentation. *Note*: this is only compatible with Kubernetes < v1.25 Default: - the container is *not* required to run as a non-root user
        :param run_as_user: If specified, this container is run as the specified user ID (``uid``). This parameter maps to ``RunAsUser`` and ``MustRunAs`` policy in the Users and groups pod security policies in the Kubernetes documentation. *Note*: this is only compatible with Kubernetes < v1.25 Default: - the user that is specified in the image metadata.
        :param volumes: The Volumes to mount to this container. Automatically added to the Pod. Default: - no volumes

        :exampleMetadata: infused

        Example::

            job_defn = batch.EksJobDefinition(self, "eksf2",
                container=batch.EksContainerDefinition(self, "container",
                    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
                    volumes=[batch.EksVolume.empty_dir(
                        name="myEmptyDirVolume",
                        mount_path="/mount/path",
                        medium=batch.EmptyDirMediumType.MEMORY,
                        readonly=True,
                        size_limit=cdk.Size.mebibytes(2048)
                    )]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c6387ebc75a9ee6942109bc9b96c42ac4e40f0bc5c981cd1316f4c35c113d2bb)
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
            check_type(argname="argument args", value=args, expected_type=type_hints["args"])
            check_type(argname="argument command", value=command, expected_type=type_hints["command"])
            check_type(argname="argument cpu_limit", value=cpu_limit, expected_type=type_hints["cpu_limit"])
            check_type(argname="argument cpu_reservation", value=cpu_reservation, expected_type=type_hints["cpu_reservation"])
            check_type(argname="argument env", value=env, expected_type=type_hints["env"])
            check_type(argname="argument gpu_limit", value=gpu_limit, expected_type=type_hints["gpu_limit"])
            check_type(argname="argument gpu_reservation", value=gpu_reservation, expected_type=type_hints["gpu_reservation"])
            check_type(argname="argument image_pull_policy", value=image_pull_policy, expected_type=type_hints["image_pull_policy"])
            check_type(argname="argument memory_limit", value=memory_limit, expected_type=type_hints["memory_limit"])
            check_type(argname="argument memory_reservation", value=memory_reservation, expected_type=type_hints["memory_reservation"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument privileged", value=privileged, expected_type=type_hints["privileged"])
            check_type(argname="argument readonly_root_filesystem", value=readonly_root_filesystem, expected_type=type_hints["readonly_root_filesystem"])
            check_type(argname="argument run_as_group", value=run_as_group, expected_type=type_hints["run_as_group"])
            check_type(argname="argument run_as_root", value=run_as_root, expected_type=type_hints["run_as_root"])
            check_type(argname="argument run_as_user", value=run_as_user, expected_type=type_hints["run_as_user"])
            check_type(argname="argument volumes", value=volumes, expected_type=type_hints["volumes"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "image": image,
        }
        if args is not None:
            self._values["args"] = args
        if command is not None:
            self._values["command"] = command
        if cpu_limit is not None:
            self._values["cpu_limit"] = cpu_limit
        if cpu_reservation is not None:
            self._values["cpu_reservation"] = cpu_reservation
        if env is not None:
            self._values["env"] = env
        if gpu_limit is not None:
            self._values["gpu_limit"] = gpu_limit
        if gpu_reservation is not None:
            self._values["gpu_reservation"] = gpu_reservation
        if image_pull_policy is not None:
            self._values["image_pull_policy"] = image_pull_policy
        if memory_limit is not None:
            self._values["memory_limit"] = memory_limit
        if memory_reservation is not None:
            self._values["memory_reservation"] = memory_reservation
        if name is not None:
            self._values["name"] = name
        if privileged is not None:
            self._values["privileged"] = privileged
        if readonly_root_filesystem is not None:
            self._values["readonly_root_filesystem"] = readonly_root_filesystem
        if run_as_group is not None:
            self._values["run_as_group"] = run_as_group
        if run_as_root is not None:
            self._values["run_as_root"] = run_as_root
        if run_as_user is not None:
            self._values["run_as_user"] = run_as_user
        if volumes is not None:
            self._values["volumes"] = volumes

    @builtins.property
    def image(self) -> _ContainerImage_94af1b43:
        '''The image that this container will run.'''
        result = self._values.get("image")
        assert result is not None, "Required property 'image' is missing"
        return typing.cast(_ContainerImage_94af1b43, result)

    @builtins.property
    def args(self) -> typing.Optional[typing.List[builtins.str]]:
        '''An array of arguments to the entrypoint.

        If this isn't specified, the CMD of the container image is used.
        This corresponds to the args member in the Entrypoint portion of the Pod in Kubernetes.
        Environment variable references are expanded using the container's environment.
        If the referenced environment variable doesn't exist, the reference in the command isn't changed.
        For example, if the reference is to "$(NAME1)" and the NAME1 environment variable doesn't exist,
        the command string will remain "$(NAME1)." $$ is replaced with $, and the resulting string isn't expanded.
        or example, $$(VAR_NAME) is passed as $(VAR_NAME) whether or not the VAR_NAME environment variable exists.

        :default: - no args

        :see: https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/
        '''
        result = self._values.get("args")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The entrypoint for the container.

        This isn't run within a shell.
        If this isn't specified, the ``ENTRYPOINT`` of the container image is used.
        Environment variable references are expanded using the container's environment.
        If the referenced environment variable doesn't exist, the reference in the command isn't changed.
        For example, if the reference is to ``"$(NAME1)"`` and the ``NAME1`` environment variable doesn't exist,
        the command string will remain ``"$(NAME1)."`` ``$$`` is replaced with ``$`` and the resulting string isn't expanded.
        For example, ``$$(VAR_NAME)`` will be passed as ``$(VAR_NAME)`` whether or not the ``VAR_NAME`` environment variable exists.

        The entrypoint can't be updated.

        :default: - no command

        :see: https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def cpu_limit(self) -> typing.Optional[jsii.Number]:
        '''The hard limit of CPUs to present to this container. Must be an even multiple of 0.25.

        If your container attempts to exceed this limit, it will be terminated.

        At least one of ``cpuReservation`` and ``cpuLimit`` is required.
        If both are specified, then ``cpuLimit`` must be at least as large as ``cpuReservation``.

        :default: - No CPU limit

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        result = self._values.get("cpu_limit")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def cpu_reservation(self) -> typing.Optional[jsii.Number]:
        '''The soft limit of CPUs to reserve for the container Must be an even multiple of 0.25.

        The container will given at least this many CPUs, but may consume more.

        At least one of ``cpuReservation`` and ``cpuLimit`` is required.
        If both are specified, then ``cpuLimit`` must be at least as large as ``cpuReservation``.

        :default: - No CPUs reserved

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        result = self._values.get("cpu_reservation")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def env(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to pass to this container.

        *Note*: Environment variables cannot start with "AWS_BATCH".
        This naming convention is reserved for variables that AWS Batch sets.

        :default: - no environment variables
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def gpu_limit(self) -> typing.Optional[jsii.Number]:
        '''The hard limit of GPUs to present to this container.

        If your container attempts to exceed this limit, it will be terminated.

        If both ``gpuReservation`` and ``gpuLimit`` are specified, then ``gpuLimit`` must be equal to ``gpuReservation``.

        :default: - No GPU limit

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        result = self._values.get("gpu_limit")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def gpu_reservation(self) -> typing.Optional[jsii.Number]:
        '''The soft limit of CPUs to reserve for the container Must be an even multiple of 0.25.

        The container will given at least this many CPUs, but may consume more.

        If both ``gpuReservation`` and ``gpuLimit`` are specified, then ``gpuLimit`` must be equal to ``gpuReservation``.

        :default: - No GPUs reserved

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        result = self._values.get("gpu_reservation")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def image_pull_policy(self) -> typing.Optional["ImagePullPolicy"]:
        '''The image pull policy for this container.

        :default: - ``ALWAYS`` if the ``:latest`` tag is specified, ``IF_NOT_PRESENT`` otherwise

        :see: https://kubernetes.io/docs/concepts/containers/images/#updating-images
        '''
        result = self._values.get("image_pull_policy")
        return typing.cast(typing.Optional["ImagePullPolicy"], result)

    @builtins.property
    def memory_limit(self) -> typing.Optional[_Size_7b441c34]:
        '''The amount (in MiB) of memory to present to the container.

        If your container attempts to exceed the allocated memory, it will be terminated.

        Must be larger that 4 MiB

        At least one of ``memoryLimit`` and ``memoryReservation`` is required

        *Note*: To maximize your resource utilization, provide your jobs with as much memory as possible
        for the specific instance type that you are using.

        :default: - No memory limit

        :see: https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html
        '''
        result = self._values.get("memory_limit")
        return typing.cast(typing.Optional[_Size_7b441c34], result)

    @builtins.property
    def memory_reservation(self) -> typing.Optional[_Size_7b441c34]:
        '''The soft limit (in MiB) of memory to reserve for the container.

        Your container will be given at least this much memory, but may consume more.

        Must be larger that 4 MiB

        When system memory is under heavy contention, Docker attempts to keep the
        container memory to this soft limit. However, your container can consume more
        memory when it needs to, up to either the hard limit specified with the memory
        parameter (if applicable), or all of the available memory on the container
        instance, whichever comes first.

        At least one of ``memoryLimit`` and ``memoryReservation`` is required.
        If both are specified, then ``memoryLimit`` must be equal to ``memoryReservation``

        *Note*: To maximize your resource utilization, provide your jobs with as much memory as possible
        for the specific instance type that you are using.

        :default: - No memory reserved

        :see: https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html
        '''
        result = self._values.get("memory_reservation")
        return typing.cast(typing.Optional[_Size_7b441c34], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of this container.

        :default: : ``'Default'``
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def privileged(self) -> typing.Optional[builtins.bool]:
        '''If specified, gives this container elevated permissions on the host container instance.

        The level of permissions are similar to the root user permissions.

        This parameter maps to ``privileged`` policy in the Privileged pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: false

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#volumes-and-file-systems
        '''
        result = self._values.get("privileged")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''If specified, gives this container readonly access to its root file system.

        This parameter maps to ``ReadOnlyRootFilesystem`` policy in the Volumes and file systems pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: false

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#volumes-and-file-systems
        '''
        result = self._values.get("readonly_root_filesystem")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def run_as_group(self) -> typing.Optional[jsii.Number]:
        '''If specified, the container is run as the specified group ID (``gid``).

        If this parameter isn't specified, the default is the group that's specified in the image metadata.
        This parameter maps to ``RunAsGroup`` and ``MustRunAs`` policy in the Users and groups pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: none

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#users-and-groups
        '''
        result = self._values.get("run_as_group")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def run_as_root(self) -> typing.Optional[builtins.bool]:
        '''If specified, the container is run as a user with a ``uid`` other than 0.

        Otherwise, no such rule is enforced.
        This parameter maps to ``RunAsUser`` and ``MustRunAsNonRoot`` policy in the Users and groups pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: - the container is *not* required to run as a non-root user

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#users-and-groups
        '''
        result = self._values.get("run_as_root")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def run_as_user(self) -> typing.Optional[jsii.Number]:
        '''If specified, this container is run as the specified user ID (``uid``).

        This parameter maps to ``RunAsUser`` and ``MustRunAs`` policy in the Users and groups pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: - the user that is specified in the image metadata.

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#users-and-groups
        '''
        result = self._values.get("run_as_user")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def volumes(self) -> typing.Optional[typing.List["EksVolume"]]:
        '''The Volumes to mount to this container.

        Automatically added to the Pod.

        :default: - no volumes

        :see: https://kubernetes.io/docs/concepts/storage/volumes/
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.List["EksVolume"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EksContainerDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.EksMachineImage",
    jsii_struct_bases=[],
    name_mapping={"image": "image", "image_type": "imageType"},
)
class EksMachineImage:
    def __init__(
        self,
        *,
        image: typing.Optional[_IMachineImage_0e8bd50b] = None,
        image_type: typing.Optional["EksMachineImageType"] = None,
    ) -> None:
        '''A Batch MachineImage that is compatible with EKS.

        :param image: The machine image to use. Default: - chosen by batch
        :param image_type: Tells Batch which instance type to launch this image on. Default: - 'EKS_AL2' for non-gpu instances, 'EKS_AL2_NVIDIA' for gpu instances

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_batch as batch
            from aws_cdk import aws_ec2 as ec2
            
            # machine_image: ec2.IMachineImage
            
            eks_machine_image = batch.EksMachineImage(
                image=machine_image,
                image_type=batch.EksMachineImageType.EKS_AL2
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ec523e4b48a322a3fa721814ea039d8ce7d432901223423524d1ef8ca838705c)
            check_type(argname="argument image", value=image, expected_type=type_hints["image"])
            check_type(argname="argument image_type", value=image_type, expected_type=type_hints["image_type"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if image is not None:
            self._values["image"] = image
        if image_type is not None:
            self._values["image_type"] = image_type

    @builtins.property
    def image(self) -> typing.Optional[_IMachineImage_0e8bd50b]:
        '''The machine image to use.

        :default: - chosen by batch
        '''
        result = self._values.get("image")
        return typing.cast(typing.Optional[_IMachineImage_0e8bd50b], result)

    @builtins.property
    def image_type(self) -> typing.Optional["EksMachineImageType"]:
        '''Tells Batch which instance type to launch this image on.

        :default: - 'EKS_AL2' for non-gpu instances, 'EKS_AL2_NVIDIA' for gpu instances
        '''
        result = self._values.get("image_type")
        return typing.cast(typing.Optional["EksMachineImageType"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EksMachineImage(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_batch.EksMachineImageType")
class EksMachineImageType(enum.Enum):
    '''Maps the image to instance types.'''

    EKS_AL2 = "EKS_AL2"
    '''Tells Batch that this machine image runs on non-GPU instances.'''
    EKS_AL2_NVIDIA = "EKS_AL2_NVIDIA"
    '''Tells Batch that this machine image runs on GPU instances.'''


class EksVolume(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_batch.EksVolume",
):
    '''A Volume that can be mounted to a container supported by EKS.

    :exampleMetadata: infused

    Example::

        job_defn = batch.EksJobDefinition(self, "eksf2",
            container=batch.EksContainerDefinition(self, "container",
                image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
                volumes=[batch.EksVolume.empty_dir(
                    name="myEmptyDirVolume",
                    mount_path="/mount/path",
                    medium=batch.EmptyDirMediumType.MEMORY,
                    readonly=True,
                    size_limit=cdk.Size.mebibytes(2048)
                )]
            )
        )
    '''

    def __init__(
        self,
        *,
        name: builtins.str,
        mount_path: typing.Optional[builtins.str] = None,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param name: The name of this volume. The name must be a valid DNS subdomain name.
        :param mount_path: The path on the container where the volume is mounted. Default: - the volume is not mounted
        :param readonly: If specified, the container has readonly access to the volume. Otherwise, the container has read/write access. Default: false
        '''
        options = EksVolumeOptions(name=name, mount_path=mount_path, readonly=readonly)

        jsii.create(self.__class__, self, [options])

    @jsii.member(jsii_name="emptyDir")
    @builtins.classmethod
    def empty_dir(
        cls,
        *,
        medium: typing.Optional["EmptyDirMediumType"] = None,
        size_limit: typing.Optional[_Size_7b441c34] = None,
        name: builtins.str,
        mount_path: typing.Optional[builtins.str] = None,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> "EmptyDirVolume":
        '''Creates a Kubernetes EmptyDir volume.

        :param medium: The storage type to use for this Volume. Default: ``EmptyDirMediumType.DISK``
        :param size_limit: The maximum size for this Volume. Default: - no size limit
        :param name: The name of this volume. The name must be a valid DNS subdomain name.
        :param mount_path: The path on the container where the volume is mounted. Default: - the volume is not mounted
        :param readonly: If specified, the container has readonly access to the volume. Otherwise, the container has read/write access. Default: false

        :see: https://kubernetes.io/docs/concepts/storage/volumes/#emptydir
        '''
        options = EmptyDirVolumeOptions(
            medium=medium,
            size_limit=size_limit,
            name=name,
            mount_path=mount_path,
            readonly=readonly,
        )

        return typing.cast("EmptyDirVolume", jsii.sinvoke(cls, "emptyDir", [options]))

    @jsii.member(jsii_name="hostPath")
    @builtins.classmethod
    def host_path(
        cls,
        *,
        host_path: builtins.str,
        name: builtins.str,
        mount_path: typing.Optional[builtins.str] = None,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> "HostPathVolume":
        '''Creates a Kubernetes HostPath volume.

        :param host_path: The path of the file or directory on the host to mount into containers on the pod. *Note*: HothPath Volumes present many security risks, and should be avoided when possible.
        :param name: The name of this volume. The name must be a valid DNS subdomain name.
        :param mount_path: The path on the container where the volume is mounted. Default: - the volume is not mounted
        :param readonly: If specified, the container has readonly access to the volume. Otherwise, the container has read/write access. Default: false

        :see: https://kubernetes.io/docs/concepts/storage/volumes/#hostpath
        '''
        options = HostPathVolumeOptions(
            host_path=host_path, name=name, mount_path=mount_path, readonly=readonly
        )

        return typing.cast("HostPathVolume", jsii.sinvoke(cls, "hostPath", [options]))

    @jsii.member(jsii_name="secret")
    @builtins.classmethod
    def secret(
        cls,
        *,
        secret_name: builtins.str,
        optional: typing.Optional[builtins.bool] = None,
        name: builtins.str,
        mount_path: typing.Optional[builtins.str] = None,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> "SecretPathVolume":
        '''Creates a Kubernetes Secret volume.

        :param secret_name: The name of the secret. Must be a valid DNS subdomain name.
        :param optional: Specifies whether the secret or the secret's keys must be defined. Default: true
        :param name: The name of this volume. The name must be a valid DNS subdomain name.
        :param mount_path: The path on the container where the volume is mounted. Default: - the volume is not mounted
        :param readonly: If specified, the container has readonly access to the volume. Otherwise, the container has read/write access. Default: false

        :see: https://kubernetes.io/docs/concepts/storage/volumes/#secret
        '''
        options = SecretPathVolumeOptions(
            secret_name=secret_name,
            optional=optional,
            name=name,
            mount_path=mount_path,
            readonly=readonly,
        )

        return typing.cast("SecretPathVolume", jsii.sinvoke(cls, "secret", [options]))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of this volume.

        The name must be a valid DNS subdomain name.

        :see: https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-subdomain-names
        '''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @builtins.property
    @jsii.member(jsii_name="containerPath")
    def container_path(self) -> typing.Optional[builtins.str]:
        '''The path on the container where the container is mounted.

        :default: - the container is not mounted
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "containerPath"))

    @builtins.property
    @jsii.member(jsii_name="readonly")
    def readonly(self) -> typing.Optional[builtins.bool]:
        '''If specified, the container has readonly access to the volume.

        Otherwise, the container has read/write access.

        :default: false
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "readonly"))


class _EksVolumeProxy(EksVolume):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, EksVolume).__jsii_proxy_class__ = lambda : _EksVolumeProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.EksVolumeOptions",
    jsii_struct_bases=[],
    name_mapping={"name": "name", "mount_path": "mountPath", "readonly": "readonly"},
)
class EksVolumeOptions:
    def __init__(
        self,
        *,
        name: builtins.str,
        mount_path: typing.Optional[builtins.str] = None,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options to configure an EksVolume.

        :param name: The name of this volume. The name must be a valid DNS subdomain name.
        :param mount_path: The path on the container where the volume is mounted. Default: - the volume is not mounted
        :param readonly: If specified, the container has readonly access to the volume. Otherwise, the container has read/write access. Default: false

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_batch as batch
            
            eks_volume_options = batch.EksVolumeOptions(
                name="name",
            
                # the properties below are optional
                mount_path="mountPath",
                readonly=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3cb2adea45f174da85f242adad4890349a6ca9158d5afde8788eba599592201b)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument mount_path", value=mount_path, expected_type=type_hints["mount_path"])
            check_type(argname="argument readonly", value=readonly, expected_type=type_hints["readonly"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "name": name,
        }
        if mount_path is not None:
            self._values["mount_path"] = mount_path
        if readonly is not None:
            self._values["readonly"] = readonly

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of this volume.

        The name must be a valid DNS subdomain name.

        :see: https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-subdomain-names
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def mount_path(self) -> typing.Optional[builtins.str]:
        '''The path on the container where the volume is mounted.

        :default: - the volume is not mounted
        '''
        result = self._values.get("mount_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def readonly(self) -> typing.Optional[builtins.bool]:
        '''If specified, the container has readonly access to the volume.

        Otherwise, the container has read/write access.

        :default: false
        '''
        result = self._values.get("readonly")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EksVolumeOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_batch.EmptyDirMediumType")
class EmptyDirMediumType(enum.Enum):
    '''What medium the volume will live in.

    :exampleMetadata: infused

    Example::

        job_defn = batch.EksJobDefinition(self, "eksf2",
            container=batch.EksContainerDefinition(self, "container",
                image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
                volumes=[batch.EksVolume.empty_dir(
                    name="myEmptyDirVolume",
                    mount_path="/mount/path",
                    medium=batch.EmptyDirMediumType.MEMORY,
                    readonly=True,
                    size_limit=cdk.Size.mebibytes(2048)
                )]
            )
        )
    '''

    DISK = "DISK"
    '''Use the disk storage of the node.

    Items written here will survive node reboots.
    '''
    MEMORY = "MEMORY"
    '''Use the ``tmpfs`` volume that is backed by RAM of the node.

    Items written here will *not* survive node reboots.
    '''


class EmptyDirVolume(
    EksVolume,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_batch.EmptyDirVolume",
):
    '''A Kubernetes EmptyDir volume.

    :see: https://kubernetes.io/docs/concepts/storage/volumes/#emptydir
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk as cdk
        from aws_cdk import aws_batch as batch
        
        # size: cdk.Size
        
        empty_dir_volume = batch.EmptyDirVolume(
            name="name",
        
            # the properties below are optional
            medium=batch.EmptyDirMediumType.DISK,
            mount_path="mountPath",
            readonly=False,
            size_limit=size
        )
    '''

    def __init__(
        self,
        *,
        medium: typing.Optional[EmptyDirMediumType] = None,
        size_limit: typing.Optional[_Size_7b441c34] = None,
        name: builtins.str,
        mount_path: typing.Optional[builtins.str] = None,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param medium: The storage type to use for this Volume. Default: ``EmptyDirMediumType.DISK``
        :param size_limit: The maximum size for this Volume. Default: - no size limit
        :param name: The name of this volume. The name must be a valid DNS subdomain name.
        :param mount_path: The path on the container where the volume is mounted. Default: - the volume is not mounted
        :param readonly: If specified, the container has readonly access to the volume. Otherwise, the container has read/write access. Default: false
        '''
        options = EmptyDirVolumeOptions(
            medium=medium,
            size_limit=size_limit,
            name=name,
            mount_path=mount_path,
            readonly=readonly,
        )

        jsii.create(self.__class__, self, [options])

    @jsii.member(jsii_name="isEmptyDirVolume")
    @builtins.classmethod
    def is_empty_dir_volume(cls, x: typing.Any) -> builtins.bool:
        '''Returns ``true`` if ``x`` is an EmptyDirVolume, ``false`` otherwise.

        :param x: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3413bc91f1fd18031058b1013577bfa4fb299926ab8db18c31c238e34e60f4b3)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isEmptyDirVolume", [x]))

    @builtins.property
    @jsii.member(jsii_name="medium")
    def medium(self) -> typing.Optional[EmptyDirMediumType]:
        '''The storage type to use for this Volume.

        :default: ``EmptyDirMediumType.DISK``
        '''
        return typing.cast(typing.Optional[EmptyDirMediumType], jsii.get(self, "medium"))

    @builtins.property
    @jsii.member(jsii_name="sizeLimit")
    def size_limit(self) -> typing.Optional[_Size_7b441c34]:
        '''The maximum size for this Volume.

        :default: - no size limit
        '''
        return typing.cast(typing.Optional[_Size_7b441c34], jsii.get(self, "sizeLimit"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.EmptyDirVolumeOptions",
    jsii_struct_bases=[EksVolumeOptions],
    name_mapping={
        "name": "name",
        "mount_path": "mountPath",
        "readonly": "readonly",
        "medium": "medium",
        "size_limit": "sizeLimit",
    },
)
class EmptyDirVolumeOptions(EksVolumeOptions):
    def __init__(
        self,
        *,
        name: builtins.str,
        mount_path: typing.Optional[builtins.str] = None,
        readonly: typing.Optional[builtins.bool] = None,
        medium: typing.Optional[EmptyDirMediumType] = None,
        size_limit: typing.Optional[_Size_7b441c34] = None,
    ) -> None:
        '''Options for a Kubernetes EmptyDir volume.

        :param name: The name of this volume. The name must be a valid DNS subdomain name.
        :param mount_path: The path on the container where the volume is mounted. Default: - the volume is not mounted
        :param readonly: If specified, the container has readonly access to the volume. Otherwise, the container has read/write access. Default: false
        :param medium: The storage type to use for this Volume. Default: ``EmptyDirMediumType.DISK``
        :param size_limit: The maximum size for this Volume. Default: - no size limit

        :see: https://kubernetes.io/docs/concepts/storage/volumes/#emptydir
        :exampleMetadata: infused

        Example::

            job_defn = batch.EksJobDefinition(self, "eksf2",
                container=batch.EksContainerDefinition(self, "container",
                    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
                    volumes=[batch.EksVolume.empty_dir(
                        name="myEmptyDirVolume",
                        mount_path="/mount/path",
                        medium=batch.EmptyDirMediumType.MEMORY,
                        readonly=True,
                        size_limit=cdk.Size.mebibytes(2048)
                    )]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__872c8b495eedd5cae454d7366dd90dde638e63c75563854cb53bd11f1c1bf48f)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument mount_path", value=mount_path, expected_type=type_hints["mount_path"])
            check_type(argname="argument readonly", value=readonly, expected_type=type_hints["readonly"])
            check_type(argname="argument medium", value=medium, expected_type=type_hints["medium"])
            check_type(argname="argument size_limit", value=size_limit, expected_type=type_hints["size_limit"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "name": name,
        }
        if mount_path is not None:
            self._values["mount_path"] = mount_path
        if readonly is not None:
            self._values["readonly"] = readonly
        if medium is not None:
            self._values["medium"] = medium
        if size_limit is not None:
            self._values["size_limit"] = size_limit

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of this volume.

        The name must be a valid DNS subdomain name.

        :see: https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-subdomain-names
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def mount_path(self) -> typing.Optional[builtins.str]:
        '''The path on the container where the volume is mounted.

        :default: - the volume is not mounted
        '''
        result = self._values.get("mount_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def readonly(self) -> typing.Optional[builtins.bool]:
        '''If specified, the container has readonly access to the volume.

        Otherwise, the container has read/write access.

        :default: false
        '''
        result = self._values.get("readonly")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def medium(self) -> typing.Optional[EmptyDirMediumType]:
        '''The storage type to use for this Volume.

        :default: ``EmptyDirMediumType.DISK``
        '''
        result = self._values.get("medium")
        return typing.cast(typing.Optional[EmptyDirMediumType], result)

    @builtins.property
    def size_limit(self) -> typing.Optional[_Size_7b441c34]:
        '''The maximum size for this Volume.

        :default: - no size limit
        '''
        result = self._values.get("size_limit")
        return typing.cast(typing.Optional[_Size_7b441c34], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmptyDirVolumeOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.FairshareSchedulingPolicyProps",
    jsii_struct_bases=[],
    name_mapping={
        "compute_reservation": "computeReservation",
        "scheduling_policy_name": "schedulingPolicyName",
        "share_decay": "shareDecay",
        "shares": "shares",
    },
)
class FairshareSchedulingPolicyProps:
    def __init__(
        self,
        *,
        compute_reservation: typing.Optional[jsii.Number] = None,
        scheduling_policy_name: typing.Optional[builtins.str] = None,
        share_decay: typing.Optional[_Duration_4839e8c3] = None,
        shares: typing.Optional[typing.Sequence[typing.Union["Share", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Fairshare SchedulingPolicy configuration.

        :param compute_reservation: Used to calculate the percentage of the maximum available vCPU to reserve for share identifiers not present in the Queue. The percentage reserved is defined by the Scheduler as: ``(computeReservation/100)^ActiveFairShares`` where ``ActiveFairShares`` is the number of active fair share identifiers. For example, a computeReservation value of 50 indicates that AWS Batch reserves 50% of the maximum available vCPU if there's only one fair share identifier. It reserves 25% if there are two fair share identifiers. It reserves 12.5% if there are three fair share identifiers. A computeReservation value of 25 indicates that AWS Batch should reserve 25% of the maximum available vCPU if there's only one fair share identifier, 6.25% if there are two fair share identifiers, and 1.56% if there are three fair share identifiers. Default: - no vCPU is reserved
        :param scheduling_policy_name: The name of this SchedulingPolicy. Default: - generated by CloudFormation
        :param share_decay: The amount of time to use to measure the usage of each job. The usage is used to calculate a fair share percentage for each fair share identifier currently in the Queue. A value of zero (0) indicates that only current usage is measured. The decay is linear and gives preference to newer jobs. The maximum supported value is 604800 seconds (1 week). Default: - 0: only the current job usage is considered
        :param shares: The shares that this Scheduling Policy applies to. *Note*: It is possible to submit Jobs to the queue with Share Identifiers that are not recognized by the Scheduling Policy. Default: - no shares

        :exampleMetadata: infused

        Example::

            fairshare_policy = batch.FairshareSchedulingPolicy(self, "myFairsharePolicy",
                share_decay=cdk.Duration.minutes(5)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4211a1f83ffa2ed240d718dae4e2605a1c3636cc581ba42eed653a159ee29449)
            check_type(argname="argument compute_reservation", value=compute_reservation, expected_type=type_hints["compute_reservation"])
            check_type(argname="argument scheduling_policy_name", value=scheduling_policy_name, expected_type=type_hints["scheduling_policy_name"])
            check_type(argname="argument share_decay", value=share_decay, expected_type=type_hints["share_decay"])
            check_type(argname="argument shares", value=shares, expected_type=type_hints["shares"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if compute_reservation is not None:
            self._values["compute_reservation"] = compute_reservation
        if scheduling_policy_name is not None:
            self._values["scheduling_policy_name"] = scheduling_policy_name
        if share_decay is not None:
            self._values["share_decay"] = share_decay
        if shares is not None:
            self._values["shares"] = shares

    @builtins.property
    def compute_reservation(self) -> typing.Optional[jsii.Number]:
        '''Used to calculate the percentage of the maximum available vCPU to reserve for share identifiers not present in the Queue.

        The percentage reserved is defined by the Scheduler as:
        ``(computeReservation/100)^ActiveFairShares`` where ``ActiveFairShares`` is the number of active fair share identifiers.

        For example, a computeReservation value of 50 indicates that AWS Batch reserves 50% of the
        maximum available vCPU if there's only one fair share identifier.
        It reserves 25% if there are two fair share identifiers.
        It reserves 12.5% if there are three fair share identifiers.

        A computeReservation value of 25 indicates that AWS Batch should reserve 25% of the
        maximum available vCPU if there's only one fair share identifier,
        6.25% if there are two fair share identifiers,
        and 1.56% if there are three fair share identifiers.

        :default: - no vCPU is reserved
        '''
        result = self._values.get("compute_reservation")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def scheduling_policy_name(self) -> typing.Optional[builtins.str]:
        '''The name of this SchedulingPolicy.

        :default: - generated by CloudFormation
        '''
        result = self._values.get("scheduling_policy_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def share_decay(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The amount of time to use to measure the usage of each job.

        The usage is used to calculate a fair share percentage for each fair share identifier currently in the Queue.
        A value of zero (0) indicates that only current usage is measured.
        The decay is linear and gives preference to newer jobs.

        The maximum supported value is 604800 seconds (1 week).

        :default: - 0: only the current job usage is considered
        '''
        result = self._values.get("share_decay")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def shares(self) -> typing.Optional[typing.List["Share"]]:
        '''The shares that this Scheduling Policy applies to.

        *Note*: It is possible to submit Jobs to the queue with Share Identifiers that
        are not recognized by the Scheduling Policy.

        :default: - no shares
        '''
        result = self._values.get("shares")
        return typing.cast(typing.Optional[typing.List["Share"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FairshareSchedulingPolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class HostPathVolume(
    EksVolume,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_batch.HostPathVolume",
):
    '''A Kubernetes HostPath volume.

    :see: https://kubernetes.io/docs/concepts/storage/volumes/#hostpath
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_batch as batch
        
        host_path_volume = batch.HostPathVolume(
            host_path="hostPath",
            name="name",
        
            # the properties below are optional
            mount_path="mountPath",
            readonly=False
        )
    '''

    def __init__(
        self,
        *,
        host_path: builtins.str,
        name: builtins.str,
        mount_path: typing.Optional[builtins.str] = None,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param host_path: The path of the file or directory on the host to mount into containers on the pod. *Note*: HothPath Volumes present many security risks, and should be avoided when possible.
        :param name: The name of this volume. The name must be a valid DNS subdomain name.
        :param mount_path: The path on the container where the volume is mounted. Default: - the volume is not mounted
        :param readonly: If specified, the container has readonly access to the volume. Otherwise, the container has read/write access. Default: false
        '''
        options = HostPathVolumeOptions(
            host_path=host_path, name=name, mount_path=mount_path, readonly=readonly
        )

        jsii.create(self.__class__, self, [options])

    @jsii.member(jsii_name="isHostPathVolume")
    @builtins.classmethod
    def is_host_path_volume(cls, x: typing.Any) -> builtins.bool:
        '''returns ``true`` if ``x`` is a HostPathVolume, ``false`` otherwise.

        :param x: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cf74b09c1c1d17283458884fbaa3779bf0b38a3bc6b7e27cb05604faa6f0f445)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isHostPathVolume", [x]))

    @builtins.property
    @jsii.member(jsii_name="path")
    def path(self) -> builtins.str:
        '''The path of the file or directory on the host to mount into containers on the pod.

        *Note*: HothPath Volumes present many security risks, and should be avoided when possible.

        :see: https://kubernetes.io/docs/concepts/storage/volumes/#hostpath
        '''
        return typing.cast(builtins.str, jsii.get(self, "path"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.HostPathVolumeOptions",
    jsii_struct_bases=[EksVolumeOptions],
    name_mapping={
        "name": "name",
        "mount_path": "mountPath",
        "readonly": "readonly",
        "host_path": "hostPath",
    },
)
class HostPathVolumeOptions(EksVolumeOptions):
    def __init__(
        self,
        *,
        name: builtins.str,
        mount_path: typing.Optional[builtins.str] = None,
        readonly: typing.Optional[builtins.bool] = None,
        host_path: builtins.str,
    ) -> None:
        '''Options for a kubernetes HostPath volume.

        :param name: The name of this volume. The name must be a valid DNS subdomain name.
        :param mount_path: The path on the container where the volume is mounted. Default: - the volume is not mounted
        :param readonly: If specified, the container has readonly access to the volume. Otherwise, the container has read/write access. Default: false
        :param host_path: The path of the file or directory on the host to mount into containers on the pod. *Note*: HothPath Volumes present many security risks, and should be avoided when possible.

        :see: https://kubernetes.io/docs/concepts/storage/volumes/#hostpath
        :exampleMetadata: infused

        Example::

            # job_defn: batch.EksJobDefinition
            
            job_defn.container.add_volume(batch.EksVolume.empty_dir(
                name="emptyDir",
                mount_path="/Volumes/emptyDir"
            ))
            job_defn.container.add_volume(batch.EksVolume.host_path(
                name="hostPath",
                host_path="/sys",
                mount_path="/Volumes/hostPath"
            ))
            job_defn.container.add_volume(batch.EksVolume.secret(
                name="secret",
                optional=True,
                mount_path="/Volumes/secret",
                secret_name="mySecret"
            ))
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ef4c28e485c76742ef72c06920fca99bb9826e42360c568977314bd251d1410a)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument mount_path", value=mount_path, expected_type=type_hints["mount_path"])
            check_type(argname="argument readonly", value=readonly, expected_type=type_hints["readonly"])
            check_type(argname="argument host_path", value=host_path, expected_type=type_hints["host_path"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "name": name,
            "host_path": host_path,
        }
        if mount_path is not None:
            self._values["mount_path"] = mount_path
        if readonly is not None:
            self._values["readonly"] = readonly

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of this volume.

        The name must be a valid DNS subdomain name.

        :see: https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-subdomain-names
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def mount_path(self) -> typing.Optional[builtins.str]:
        '''The path on the container where the volume is mounted.

        :default: - the volume is not mounted
        '''
        result = self._values.get("mount_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def readonly(self) -> typing.Optional[builtins.bool]:
        '''If specified, the container has readonly access to the volume.

        Otherwise, the container has read/write access.

        :default: false
        '''
        result = self._values.get("readonly")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def host_path(self) -> builtins.str:
        '''The path of the file or directory on the host to mount into containers on the pod.

        *Note*: HothPath Volumes present many security risks, and should be avoided when possible.

        :see: https://kubernetes.io/docs/concepts/storage/volumes/#hostpath
        '''
        result = self._values.get("host_path")
        assert result is not None, "Required property 'host_path' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HostPathVolumeOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class HostVolume(
    EcsVolume,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_batch.HostVolume",
):
    '''Creates a Host volume.

    This volume will persist on the host at the specified ``hostPath``.
    If the ``hostPath`` is not specified, Docker will choose the host path. In this case,
    the data may not persist after the containers that use it stop running.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_batch as batch
        
        host_volume = batch.HostVolume(
            container_path="containerPath",
            name="name",
        
            # the properties below are optional
            host_path="hostPath",
            readonly=False
        )
    '''

    def __init__(
        self,
        *,
        host_path: typing.Optional[builtins.str] = None,
        container_path: builtins.str,
        name: builtins.str,
        readonly: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param host_path: The path on the host machine this container will have access to. Default: - Docker will choose the host path. The data may not persist after the containers that use it stop running.
        :param container_path: the path on the container where this volume is mounted.
        :param name: the name of this volume.
        :param readonly: if set, the container will have readonly access to the volume. Default: false
        '''
        options = HostVolumeOptions(
            host_path=host_path,
            container_path=container_path,
            name=name,
            readonly=readonly,
        )

        jsii.create(self.__class__, self, [options])

    @jsii.member(jsii_name="isHostVolume")
    @builtins.classmethod
    def is_host_volume(cls, x: typing.Any) -> builtins.bool:
        '''returns ``true`` if ``x`` is a ``HostVolume``, ``false`` otherwise.

        :param x: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__34e9d69e0b38033a541890fd7499d597146918d6fe4b501771f5ccd67bf3778a)
            check_type(argname="argument x", value=x, expected_type=type_hints["x"])
        return typing.cast(builtins.bool, jsii.sinvoke(cls, "isHostVolume", [x]))

    @builtins.property
    @jsii.member(jsii_name="hostPath")
    def host_path(self) -> typing.Optional[builtins.str]:
        '''The path on the host machine this container will have access to.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "hostPath"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_batch.HostVolumeOptions",
    jsii_struct_bases=[EcsVolumeOptions],
    name_mapping={
        "container_path": "containerPath",
        "name": "name",
        "readonly": "readonly",
        "host_path": "hostPath",
    },
)
class HostVolumeOptions(EcsVolumeOptions):
    def __init__(
        self,
        *,
        container_path: builtins.str,
        name: builtins.str,
        readonly: typing.Optional[builtins.bool] = None,
        host_path: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options for configuring an ECS HostVolume.

        :param container_path: the path on the container where this volume is mounted.
        :param name: the name of this volume.
        :param readonly: if set, the container will have readonly access to the volume. Default: false
        :param host_path: The path on the host machine this container will have access to. Default: - Docker will choose the host path. The data may not persist after the containers that use it stop running.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_batch as batch
            
            host_volume_options = batch.HostVolumeOptions(
                container_path="containerPath",
                name="name",
            
                # the properties below are optional
                host_path="hostPath",
                readonly=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a9e0d5da0b1f4721f283358b90cd5b9e831ed97e2b0894e15680cd8d2b07d85f)
            check_type(argname="argument container_path", value=container_path, expected_type=type_hints["container_path"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument readonly", value=readonly, expected_type=type_hints["readonly"])
            check_type(argname="argument host_path", value=host_path, expected_type=type_hints["host_path"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "container_path": container_path,
            "name": name,
        }
        if readonly is not None:
            self._values["readonly"] = readonly
        if host_path is not None:
            self._values["host_path"] = host_path

    @builtins.property
    def container_path(self) -> builtins.str:
        '''the path on the container where this volume is mounted.'''
        result = self._values.get("container_path")
        assert result is not None, "Required property 'container_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''the name of this volume.'''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def readonly(self) -> typing.Optional[builtins.bool]:
        '''if set, the container will have readonly access to the volume.

        :default: false
        '''
        result = self._values.get("readonly")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def host_path(self) -> typing.Optional[builtins.str]:
        '''The path on the host machine this container will have access to.

        :default:

        - Docker will choose the host path.
        The data may not persist after the containers that use it stop running.
        '''
        result = self._values.get("host_path")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HostVolumeOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="aws-cdk-lib.aws_batch.IComputeEnvironment")
class IComputeEnvironment(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents a ComputeEnvironment.'''

    @builtins.property
    @jsii.member(jsii_name="computeEnvironmentArn")
    def compute_environment_arn(self) -> builtins.str:
        '''The ARN of this compute environment.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="computeEnvironmentName")
    def compute_environment_name(self) -> builtins.str:
        '''The name of the ComputeEnvironment.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="enabled")
    def enabled(self) -> builtins.bool:
        '''Whether or not this ComputeEnvironment can accept jobs from a Queue.

        Enabled ComputeEnvironments can accept jobs from a Queue and
        can scale instances up or down.
        Disabled ComputeEnvironments cannot accept jobs from a Queue or
        scale instances up or down.

        If you change a ComputeEnvironment from enabled to disabled while it is executing jobs,
        Jobs in the ``STARTED`` or ``RUNNING`` states will not
        be interrupted. As jobs complete, the ComputeEnvironment will scale instances down to ``minvCpus``.

        To ensure you aren't billed for unused capacity, set ``minvCpus`` to ``0``.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="serviceRole")
    def service_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role Batch uses to perform actions on your behalf in your account, such as provision instances to run your jobs.

        :default: - a serviceRole will be created for managed CEs, none for unmanaged CEs
        '''
        ...


class _IComputeEnvironmentProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents a ComputeEnvironment.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_batch.IComputeEnvironment"

    @builtins.property
    @jsii.member(jsii_name="computeEnvironmentArn")
    def compute_environment_arn(self) -> builtins.str:
        '''The ARN of this compute environment.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "computeEnvironmentArn"))

    @builtins.property
    @jsii.member(jsii_name="computeEnvironmentName")
    def compute_environment_name(self) -> builtins.str:
        '''The name of the ComputeEnvironment.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "computeEnvironmentName"))

    @builtins.property
    @jsii.member(jsii_name="enabled")
    def enabled(self) -> builtins.bool:
        '''Whether or not this ComputeEnvironment can accept jobs from a Queue.

        Enabled ComputeEnvironments can accept jobs from a Queue and
        can scale instances up or down.
        Disabled ComputeEnvironments cannot accept jobs from a Queue or
        scale instances up or down.

        If you change a ComputeEnvironment from enabled to disabled while it is executing jobs,
        Jobs in the ``STARTED`` or ``RUNNING`` states will not
        be interrupted. As jobs complete, the ComputeEnvironment will scale instances down to ``minvCpus``.

        To ensure you aren't billed for unused capacity, set ``minvCpus`` to ``0``.
        '''
        return typing.cast(builtins.bool, jsii.get(self, "enabled"))

    @builtins.property
    @jsii.member(jsii_name="serviceRole")
    def service_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role Batch uses to perform actions on your behalf in your account, such as provision instances to run your jobs.

        :default: - a serviceRole will be created for managed CEs, none for unmanaged CEs
        '''
        return typing.cast(typing.Optional[_IRole_235f5d8e], jsii.get(self, "serviceRole"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IComputeEnvironment).__jsii_proxy_class__ = lambda : _IComputeEnvironmentProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_batch.IEcsContainerDefinition")
class IEcsContainerDefinition(
    _constructs_77d1e7e8.IConstruct,
    typing_extensions.Protocol,
):
    '''A container that can be run with ECS orchestration.'''

    @builtins.property
    @jsii.member(jsii_name="cpu")
    def cpu(self) -> jsii.Number:
        '''The number of vCPUs reserved for the container.

        Each vCPU is equivalent to 1,024 CPU shares.
        For containers running on EC2 resources, you must specify at least one vCPU.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="executionRole")
    def execution_role(self) -> _IRole_235f5d8e:
        '''The role used by Amazon ECS container and AWS Fargate agents to make AWS API calls on your behalf.

        :see: https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="image")
    def image(self) -> _ContainerImage_94af1b43:
        '''The image that this container will run.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="memory")
    def memory(self) -> _Size_7b441c34:
        '''The memory hard limit present to the container.

        If your container attempts to exceed the memory specified, the container is terminated.
        You must specify at least 4 MiB of memory for a job.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="volumes")
    def volumes(self) -> typing.List[EcsVolume]:
        '''The volumes to mount to this container.

        Automatically added to the job definition.

        :default: - no volumes
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="command")
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The command that's passed to the container.

        :see: https://docs.docker.com/engine/reference/builder/#cmd
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="environment")
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to pass to a container.

        Cannot start with ``AWS_BATCH``.
        We don't recommend using plaintext environment variables for sensitive information, such as credential data.

        :default: - no environment variables
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="jobRole")
    def job_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role that the container can assume.

        :default: - no jobRole

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="linuxParameters")
    def linux_parameters(self) -> typing.Optional["LinuxParameters"]:
        '''Linux-specific modifications that are applied to the container, such as details for device mappings.

        :default: none
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="logDriverConfig")
    def log_driver_config(self) -> typing.Optional[_LogDriverConfig_8fb9cc8a]:
        '''The configuration of the log driver.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="readonlyRootFilesystem")
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''Gives the container readonly access to its root filesystem.

        :default: false
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="secrets")
    def secrets(self) -> typing.Optional[typing.Mapping[builtins.str, "Secret"]]:
        '''A map from environment variable names to the secrets for the container.

        Allows your job definitions
        to reference the secret by the environment variable name defined in this property.

        :default: - no secrets

        :see: https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="user")
    def user(self) -> typing.Optional[builtins.str]:
        '''The user name to use inside the container.

        :default: - no user
        '''
        ...

    @jsii.member(jsii_name="addVolume")
    def add_volume(self, volume: EcsVolume) -> None:
        '''Add a Volume to this container.

        :param volume: -
        '''
        ...


class _IEcsContainerDefinitionProxy(
    jsii.proxy_for(_constructs_77d1e7e8.IConstruct), # type: ignore[misc]
):
    '''A container that can be run with ECS orchestration.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_batch.IEcsContainerDefinition"

    @builtins.property
    @jsii.member(jsii_name="cpu")
    def cpu(self) -> jsii.Number:
        '''The number of vCPUs reserved for the container.

        Each vCPU is equivalent to 1,024 CPU shares.
        For containers running on EC2 resources, you must specify at least one vCPU.
        '''
        return typing.cast(jsii.Number, jsii.get(self, "cpu"))

    @builtins.property
    @jsii.member(jsii_name="executionRole")
    def execution_role(self) -> _IRole_235f5d8e:
        '''The role used by Amazon ECS container and AWS Fargate agents to make AWS API calls on your behalf.

        :see: https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html
        '''
        return typing.cast(_IRole_235f5d8e, jsii.get(self, "executionRole"))

    @builtins.property
    @jsii.member(jsii_name="image")
    def image(self) -> _ContainerImage_94af1b43:
        '''The image that this container will run.'''
        return typing.cast(_ContainerImage_94af1b43, jsii.get(self, "image"))

    @builtins.property
    @jsii.member(jsii_name="memory")
    def memory(self) -> _Size_7b441c34:
        '''The memory hard limit present to the container.

        If your container attempts to exceed the memory specified, the container is terminated.
        You must specify at least 4 MiB of memory for a job.
        '''
        return typing.cast(_Size_7b441c34, jsii.get(self, "memory"))

    @builtins.property
    @jsii.member(jsii_name="volumes")
    def volumes(self) -> typing.List[EcsVolume]:
        '''The volumes to mount to this container.

        Automatically added to the job definition.

        :default: - no volumes
        '''
        return typing.cast(typing.List[EcsVolume], jsii.get(self, "volumes"))

    @builtins.property
    @jsii.member(jsii_name="command")
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The command that's passed to the container.

        :see: https://docs.docker.com/engine/reference/builder/#cmd
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "command"))

    @builtins.property
    @jsii.member(jsii_name="environment")
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to pass to a container.

        Cannot start with ``AWS_BATCH``.
        We don't recommend using plaintext environment variables for sensitive information, such as credential data.

        :default: - no environment variables
        '''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "environment"))

    @builtins.property
    @jsii.member(jsii_name="jobRole")
    def job_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''The role that the container can assume.

        :default: - no jobRole

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html
        '''
        return typing.cast(typing.Optional[_IRole_235f5d8e], jsii.get(self, "jobRole"))

    @builtins.property
    @jsii.member(jsii_name="linuxParameters")
    def linux_parameters(self) -> typing.Optional["LinuxParameters"]:
        '''Linux-specific modifications that are applied to the container, such as details for device mappings.

        :default: none
        '''
        return typing.cast(typing.Optional["LinuxParameters"], jsii.get(self, "linuxParameters"))

    @builtins.property
    @jsii.member(jsii_name="logDriverConfig")
    def log_driver_config(self) -> typing.Optional[_LogDriverConfig_8fb9cc8a]:
        '''The configuration of the log driver.'''
        return typing.cast(typing.Optional[_LogDriverConfig_8fb9cc8a], jsii.get(self, "logDriverConfig"))

    @builtins.property
    @jsii.member(jsii_name="readonlyRootFilesystem")
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''Gives the container readonly access to its root filesystem.

        :default: false
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "readonlyRootFilesystem"))

    @builtins.property
    @jsii.member(jsii_name="secrets")
    def secrets(self) -> typing.Optional[typing.Mapping[builtins.str, "Secret"]]:
        '''A map from environment variable names to the secrets for the container.

        Allows your job definitions
        to reference the secret by the environment variable name defined in this property.

        :default: - no secrets

        :see: https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html
        '''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "Secret"]], jsii.get(self, "secrets"))

    @builtins.property
    @jsii.member(jsii_name="user")
    def user(self) -> typing.Optional[builtins.str]:
        '''The user name to use inside the container.

        :default: - no user
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "user"))

    @jsii.member(jsii_name="addVolume")
    def add_volume(self, volume: EcsVolume) -> None:
        '''Add a Volume to this container.

        :param volume: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__621d4cf899373d7a7fd13c2bcc01e6fbd900ffc652f471458f2bb9b074fb5ede)
            check_type(argname="argument volume", value=volume, expected_type=type_hints["volume"])
        return typing.cast(None, jsii.invoke(self, "addVolume", [volume]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEcsContainerDefinition).__jsii_proxy_class__ = lambda : _IEcsContainerDefinitionProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_batch.IEcsEc2ContainerDefinition")
class IEcsEc2ContainerDefinition(IEcsContainerDefinition, typing_extensions.Protocol):
    '''A container orchestrated by ECS that uses EC2 resources.'''

    @builtins.property
    @jsii.member(jsii_name="ulimits")
    def ulimits(self) -> typing.List["Ulimit"]:
        '''Limits to set for the user this docker container will run as.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="gpu")
    def gpu(self) -> typing.Optional[jsii.Number]:
        '''The number of physical GPUs to reserve for the container.

        Make sure that the number of GPUs reserved for all containers in a job doesn't exceed
        the number of available GPUs on the compute resource that the job is launched on.

        :default: - no gpus
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="privileged")
    def privileged(self) -> typing.Optional[builtins.bool]:
        '''When this parameter is true, the container is given elevated permissions on the host container instance (similar to the root user).

        :default: false
        '''
        ...

    @jsii.member(jsii_name="addUlimit")
    def add_ulimit(
        self,
        *,
        hard_limit: jsii.Number,
        name: "UlimitName",
        soft_limit: jsii.Number,
    ) -> None:
        '''Add a ulimit to this container.

        :param hard_limit: The hard limit for this resource. The container will be terminated if it exceeds this limit.
        :param name: The resource to limit.
        :param soft_limit: The reservation for this resource. The container will not be terminated if it exceeds this limit.
        '''
        ...


class _IEcsEc2ContainerDefinitionProxy(
    jsii.proxy_for(IEcsContainerDefinition), # type: ignore[misc]
):
    '''A container orchestrated by ECS that uses EC2 resources.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_batch.IEcsEc2ContainerDefinition"

    @builtins.property
    @jsii.member(jsii_name="ulimits")
    def ulimits(self) -> typing.List["Ulimit"]:
        '''Limits to set for the user this docker container will run as.'''
        return typing.cast(typing.List["Ulimit"], jsii.get(self, "ulimits"))

    @builtins.property
    @jsii.member(jsii_name="gpu")
    def gpu(self) -> typing.Optional[jsii.Number]:
        '''The number of physical GPUs to reserve for the container.

        Make sure that the number of GPUs reserved for all containers in a job doesn't exceed
        the number of available GPUs on the compute resource that the job is launched on.

        :default: - no gpus
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "gpu"))

    @builtins.property
    @jsii.member(jsii_name="privileged")
    def privileged(self) -> typing.Optional[builtins.bool]:
        '''When this parameter is true, the container is given elevated permissions on the host container instance (similar to the root user).

        :default: false
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "privileged"))

    @jsii.member(jsii_name="addUlimit")
    def add_ulimit(
        self,
        *,
        hard_limit: jsii.Number,
        name: "UlimitName",
        soft_limit: jsii.Number,
    ) -> None:
        '''Add a ulimit to this container.

        :param hard_limit: The hard limit for this resource. The container will be terminated if it exceeds this limit.
        :param name: The resource to limit.
        :param soft_limit: The reservation for this resource. The container will not be terminated if it exceeds this limit.
        '''
        ulimit = Ulimit(hard_limit=hard_limit, name=name, soft_limit=soft_limit)

        return typing.cast(None, jsii.invoke(self, "addUlimit", [ulimit]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEcsEc2ContainerDefinition).__jsii_proxy_class__ = lambda : _IEcsEc2ContainerDefinitionProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_batch.IEcsFargateContainerDefinition")
class IEcsFargateContainerDefinition(
    IEcsContainerDefinition,
    typing_extensions.Protocol,
):
    '''A container orchestrated by ECS that uses Fargate resources and is orchestrated by ECS.'''

    @builtins.property
    @jsii.member(jsii_name="assignPublicIp")
    def assign_public_ip(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether the job has a public IP address.

        For a job that's running on Fargate resources in a private subnet to send outbound traffic to the internet
        (for example, to pull container images), the private subnet requires a NAT gateway be attached to route requests to the internet.

        :default: false

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="ephemeralStorageSize")
    def ephemeral_storage_size(self) -> typing.Optional[_Size_7b441c34]:
        '''The size for ephemeral storage.

        :default: - 20 GiB
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="fargateCpuArchitecture")
    def fargate_cpu_architecture(self) -> typing.Optional[_CpuArchitecture_d27dee43]:
        '''The vCPU architecture of Fargate Runtime.

        :default: - X86_64
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="fargateOperatingSystemFamily")
    def fargate_operating_system_family(
        self,
    ) -> typing.Optional[_OperatingSystemFamily_b23afb56]:
        '''The operating system for the compute environment.

        :default: - LINUX
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="fargatePlatformVersion")
    def fargate_platform_version(
        self,
    ) -> typing.Optional[_FargatePlatformVersion_55d8be5c]:
        '''Which version of Fargate to use when running this container.

        :default: LATEST
        '''
        ...


class _IEcsFargateContainerDefinitionProxy(
    jsii.proxy_for(IEcsContainerDefinition), # type: ignore[misc]
):
    '''A container orchestrated by ECS that uses Fargate resources and is orchestrated by ECS.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_batch.IEcsFargateContainerDefinition"

    @builtins.property
    @jsii.member(jsii_name="assignPublicIp")
    def assign_public_ip(self) -> typing.Optional[builtins.bool]:
        '''Indicates whether the job has a public IP address.

        For a job that's running on Fargate resources in a private subnet to send outbound traffic to the internet
        (for example, to pull container images), the private subnet requires a NAT gateway be attached to route requests to the internet.

        :default: false

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "assignPublicIp"))

    @builtins.property
    @jsii.member(jsii_name="ephemeralStorageSize")
    def ephemeral_storage_size(self) -> typing.Optional[_Size_7b441c34]:
        '''The size for ephemeral storage.

        :default: - 20 GiB
        '''
        return typing.cast(typing.Optional[_Size_7b441c34], jsii.get(self, "ephemeralStorageSize"))

    @builtins.property
    @jsii.member(jsii_name="fargateCpuArchitecture")
    def fargate_cpu_architecture(self) -> typing.Optional[_CpuArchitecture_d27dee43]:
        '''The vCPU architecture of Fargate Runtime.

        :default: - X86_64
        '''
        return typing.cast(typing.Optional[_CpuArchitecture_d27dee43], jsii.get(self, "fargateCpuArchitecture"))

    @builtins.property
    @jsii.member(jsii_name="fargateOperatingSystemFamily")
    def fargate_operating_system_family(
        self,
    ) -> typing.Optional[_OperatingSystemFamily_b23afb56]:
        '''The operating system for the compute environment.

        :default: - LINUX
        '''
        return typing.cast(typing.Optional[_OperatingSystemFamily_b23afb56], jsii.get(self, "fargateOperatingSystemFamily"))

    @builtins.property
    @jsii.member(jsii_name="fargatePlatformVersion")
    def fargate_platform_version(
        self,
    ) -> typing.Optional[_FargatePlatformVersion_55d8be5c]:
        '''Which version of Fargate to use when running this container.

        :default: LATEST
        '''
        return typing.cast(typing.Optional[_FargatePlatformVersion_55d8be5c], jsii.get(self, "fargatePlatformVersion"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEcsFargateContainerDefinition).__jsii_proxy_class__ = lambda : _IEcsFargateContainerDefinitionProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_batch.IEksContainerDefinition")
class IEksContainerDefinition(
    _constructs_77d1e7e8.IConstruct,
    typing_extensions.Protocol,
):
    '''A container that can be run with EKS orchestration on EC2 resources.'''

    @builtins.property
    @jsii.member(jsii_name="image")
    def image(self) -> _ContainerImage_94af1b43:
        '''The image that this container will run.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="volumes")
    def volumes(self) -> typing.List[EksVolume]:
        '''The Volumes to mount to this container.

        Automatically added to the Pod.

        :see: https://kubernetes.io/docs/concepts/storage/volumes/
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="args")
    def args(self) -> typing.Optional[typing.List[builtins.str]]:
        '''An array of arguments to the entrypoint.

        If this isn't specified, the CMD of the container image is used.
        This corresponds to the args member in the Entrypoint portion of the Pod in Kubernetes.
        Environment variable references are expanded using the container's environment.
        If the referenced environment variable doesn't exist, the reference in the command isn't changed.
        For example, if the reference is to "$(NAME1)" and the NAME1 environment variable doesn't exist,
        the command string will remain "$(NAME1)." $$ is replaced with $, and the resulting string isn't expanded.
        or example, $$(VAR_NAME) is passed as $(VAR_NAME) whether or not the VAR_NAME environment variable exists.

        :see: https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="command")
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The entrypoint for the container.

        This isn't run within a shell.
        If this isn't specified, the ``ENTRYPOINT`` of the container image is used.
        Environment variable references are expanded using the container's environment.
        If the referenced environment variable doesn't exist, the reference in the command isn't changed.
        For example, if the reference is to ``"$(NAME1)"`` and the ``NAME1`` environment variable doesn't exist,
        the command string will remain ``"$(NAME1)."`` ``$$`` is replaced with ``$`` and the resulting string isn't expanded.
        For example, ``$$(VAR_NAME)`` will be passed as ``$(VAR_NAME)`` whether or not the ``VAR_NAME`` environment variable exists.

        The entrypoint can't be updated.

        :see: https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="cpuLimit")
    def cpu_limit(self) -> typing.Optional[jsii.Number]:
        '''The hard limit of CPUs to present to this container. Must be an even multiple of 0.25.

        If your container attempts to exceed this limit, it will be terminated.

        At least one of ``cpuReservation`` and ``cpuLimit`` is required.
        If both are specified, then ``cpuLimit`` must be at least as large as ``cpuReservation``.

        :default: - No CPU limit

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="cpuReservation")
    def cpu_reservation(self) -> typing.Optional[jsii.Number]:
        '''The soft limit of CPUs to reserve for the container Must be an even multiple of 0.25.

        The container will given at least this many CPUs, but may consume more.

        At least one of ``cpuReservation`` and ``cpuLimit`` is required.
        If both are specified, then ``cpuLimit`` must be at least as large as ``cpuReservation``.

        :default: - No CPUs reserved

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="env")
    def env(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to pass to this container.

        *Note*: Environment variables cannot start with "AWS_BATCH".
        This naming convention is reserved for variables that AWS Batch sets.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="gpuLimit")
    def gpu_limit(self) -> typing.Optional[jsii.Number]:
        '''The hard limit of GPUs to present to this container.

        If your container attempts to exceed this limit, it will be terminated.

        If both ``gpuReservation`` and ``gpuLimit`` are specified, then ``gpuLimit`` must be equal to ``gpuReservation``.

        :default: - No GPU limit

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="gpuReservation")
    def gpu_reservation(self) -> typing.Optional[jsii.Number]:
        '''The soft limit of CPUs to reserve for the container Must be an even multiple of 0.25.

        The container will given at least this many CPUs, but may consume more.

        If both ``gpuReservation`` and ``gpuLimit`` are specified, then ``gpuLimit`` must be equal to ``gpuReservation``.

        :default: - No GPUs reserved

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="imagePullPolicy")
    def image_pull_policy(self) -> typing.Optional["ImagePullPolicy"]:
        '''The image pull policy for this container.

        :default: - ``ALWAYS`` if the ``:latest`` tag is specified, ``IF_NOT_PRESENT`` otherwise

        :see: https://kubernetes.io/docs/concepts/containers/images/#updating-images
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="memoryLimit")
    def memory_limit(self) -> typing.Optional[_Size_7b441c34]:
        '''The amount (in MiB) of memory to present to the container.

        If your container attempts to exceed the allocated memory, it will be terminated.

        Must be larger that 4 MiB

        At least one of ``memoryLimit`` and ``memoryReservation`` is required

        *Note*: To maximize your resource utilization, provide your jobs with as much memory as possible
        for the specific instance type that you are using.

        :default: - No memory limit

        :see: https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="memoryReservation")
    def memory_reservation(self) -> typing.Optional[_Size_7b441c34]:
        '''The soft limit (in MiB) of memory to reserve for the container.

        Your container will be given at least this much memory, but may consume more.

        Must be larger that 4 MiB

        When system memory is under heavy contention, Docker attempts to keep the
        container memory to this soft limit. However, your container can consume more
        memory when it needs to, up to either the hard limit specified with the memory
        parameter (if applicable), or all of the available memory on the container
        instance, whichever comes first.

        At least one of ``memoryLimit`` and ``memoryReservation`` is required.
        If both are specified, then ``memoryLimit`` must be equal to ``memoryReservation``

        *Note*: To maximize your resource utilization, provide your jobs with as much memory as possible
        for the specific instance type that you are using.

        :default: - No memory reserved

        :see: https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of this container.

        :default: : ``'Default'``
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="privileged")
    def privileged(self) -> typing.Optional[builtins.bool]:
        '''If specified, gives this container elevated permissions on the host container instance.

        The level of permissions are similar to the root user permissions.

        This parameter maps to ``privileged`` policy in the Privileged pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: false

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#volumes-and-file-systems
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="readonlyRootFilesystem")
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''If specified, gives this container readonly access to its root file system.

        This parameter maps to ``ReadOnlyRootFilesystem`` policy in the Volumes and file systems pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: false

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#volumes-and-file-systems
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="runAsGroup")
    def run_as_group(self) -> typing.Optional[jsii.Number]:
        '''If specified, the container is run as the specified group ID (``gid``).

        If this parameter isn't specified, the default is the group that's specified in the image metadata.
        This parameter maps to ``RunAsGroup`` and ``MustRunAs`` policy in the Users and groups pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: none

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#users-and-groups
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="runAsRoot")
    def run_as_root(self) -> typing.Optional[builtins.bool]:
        '''If specified, the container is run as a user with a ``uid`` other than 0.

        Otherwise, no such rule is enforced.
        This parameter maps to ``RunAsUser`` and ``MustRunAsNonRoot`` policy in the Users and groups pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: - the container is *not* required to run as a non-root user

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#users-and-groups
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="runAsUser")
    def run_as_user(self) -> typing.Optional[jsii.Number]:
        '''If specified, this container is run as the specified user ID (``uid``).

        This parameter maps to ``RunAsUser`` and ``MustRunAs`` policy in the Users and groups pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: - the user that is specified in the image metadata.

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#users-and-groups
        '''
        ...

    @jsii.member(jsii_name="addVolume")
    def add_volume(self, volume: EksVolume) -> None:
        '''Mount a Volume to this container.

        Automatically added to the Pod.

        :param volume: -
        '''
        ...


class _IEksContainerDefinitionProxy(
    jsii.proxy_for(_constructs_77d1e7e8.IConstruct), # type: ignore[misc]
):
    '''A container that can be run with EKS orchestration on EC2 resources.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_batch.IEksContainerDefinition"

    @builtins.property
    @jsii.member(jsii_name="image")
    def image(self) -> _ContainerImage_94af1b43:
        '''The image that this container will run.'''
        return typing.cast(_ContainerImage_94af1b43, jsii.get(self, "image"))

    @builtins.property
    @jsii.member(jsii_name="volumes")
    def volumes(self) -> typing.List[EksVolume]:
        '''The Volumes to mount to this container.

        Automatically added to the Pod.

        :see: https://kubernetes.io/docs/concepts/storage/volumes/
        '''
        return typing.cast(typing.List[EksVolume], jsii.get(self, "volumes"))

    @builtins.property
    @jsii.member(jsii_name="args")
    def args(self) -> typing.Optional[typing.List[builtins.str]]:
        '''An array of arguments to the entrypoint.

        If this isn't specified, the CMD of the container image is used.
        This corresponds to the args member in the Entrypoint portion of the Pod in Kubernetes.
        Environment variable references are expanded using the container's environment.
        If the referenced environment variable doesn't exist, the reference in the command isn't changed.
        For example, if the reference is to "$(NAME1)" and the NAME1 environment variable doesn't exist,
        the command string will remain "$(NAME1)." $$ is replaced with $, and the resulting string isn't expanded.
        or example, $$(VAR_NAME) is passed as $(VAR_NAME) whether or not the VAR_NAME environment variable exists.

        :see: https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "args"))

    @builtins.property
    @jsii.member(jsii_name="command")
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The entrypoint for the container.

        This isn't run within a shell.
        If this isn't specified, the ``ENTRYPOINT`` of the container image is used.
        Environment variable references are expanded using the container's environment.
        If the referenced environment variable doesn't exist, the reference in the command isn't changed.
        For example, if the reference is to ``"$(NAME1)"`` and the ``NAME1`` environment variable doesn't exist,
        the command string will remain ``"$(NAME1)."`` ``$$`` is replaced with ``$`` and the resulting string isn't expanded.
        For example, ``$$(VAR_NAME)`` will be passed as ``$(VAR_NAME)`` whether or not the ``VAR_NAME`` environment variable exists.

        The entrypoint can't be updated.

        :see: https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "command"))

    @builtins.property
    @jsii.member(jsii_name="cpuLimit")
    def cpu_limit(self) -> typing.Optional[jsii.Number]:
        '''The hard limit of CPUs to present to this container. Must be an even multiple of 0.25.

        If your container attempts to exceed this limit, it will be terminated.

        At least one of ``cpuReservation`` and ``cpuLimit`` is required.
        If both are specified, then ``cpuLimit`` must be at least as large as ``cpuReservation``.

        :default: - No CPU limit

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "cpuLimit"))

    @builtins.property
    @jsii.member(jsii_name="cpuReservation")
    def cpu_reservation(self) -> typing.Optional[jsii.Number]:
        '''The soft limit of CPUs to reserve for the container Must be an even multiple of 0.25.

        The container will given at least this many CPUs, but may consume more.

        At least one of ``cpuReservation`` and ``cpuLimit`` is required.
        If both are specified, then ``cpuLimit`` must be at least as large as ``cpuReservation``.

        :default: - No CPUs reserved

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "cpuReservation"))

    @builtins.property
    @jsii.member(jsii_name="env")
    def env(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The environment variables to pass to this container.

        *Note*: Environment variables cannot start with "AWS_BATCH".
        This naming convention is reserved for variables that AWS Batch sets.
        '''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], jsii.get(self, "env"))

    @builtins.property
    @jsii.member(jsii_name="gpuLimit")
    def gpu_limit(self) -> typing.Optional[jsii.Number]:
        '''The hard limit of GPUs to present to this container.

        If your container attempts to exceed this limit, it will be terminated.

        If both ``gpuReservation`` and ``gpuLimit`` are specified, then ``gpuLimit`` must be equal to ``gpuReservation``.

        :default: - No GPU limit

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "gpuLimit"))

    @builtins.property
    @jsii.member(jsii_name="gpuReservation")
    def gpu_reservation(self) -> typing.Optional[jsii.Number]:
        '''The soft limit of CPUs to reserve for the container Must be an even multiple of 0.25.

        The container will given at least this many CPUs, but may consume more.

        If both ``gpuReservation`` and ``gpuLimit`` are specified, then ``gpuLimit`` must be equal to ``gpuReservation``.

        :default: - No GPUs reserved

        :see: https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "gpuReservation"))

    @builtins.property
    @jsii.member(jsii_name="imagePullPolicy")
    def image_pull_policy(self) -> typing.Optional["ImagePullPolicy"]:
        '''The image pull policy for this container.

        :default: - ``ALWAYS`` if the ``:latest`` tag is specified, ``IF_NOT_PRESENT`` otherwise

        :see: https://kubernetes.io/docs/concepts/containers/images/#updating-images
        '''
        return typing.cast(typing.Optional["ImagePullPolicy"], jsii.get(self, "imagePullPolicy"))

    @builtins.property
    @jsii.member(jsii_name="memoryLimit")
    def memory_limit(self) -> typing.Optional[_Size_7b441c34]:
        '''The amount (in MiB) of memory to present to the container.

        If your container attempts to exceed the allocated memory, it will be terminated.

        Must be larger that 4 MiB

        At least one of ``memoryLimit`` and ``memoryReservation`` is required

        *Note*: To maximize your resource utilization, provide your jobs with as much memory as possible
        for the specific instance type that you are using.

        :default: - No memory limit

        :see: https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html
        '''
        return typing.cast(typing.Optional[_Size_7b441c34], jsii.get(self, "memoryLimit"))

    @builtins.property
    @jsii.member(jsii_name="memoryReservation")
    def memory_reservation(self) -> typing.Optional[_Size_7b441c34]:
        '''The soft limit (in MiB) of memory to reserve for the container.

        Your container will be given at least this much memory, but may consume more.

        Must be larger that 4 MiB

        When system memory is under heavy contention, Docker attempts to keep the
        container memory to this soft limit. However, your container can consume more
        memory when it needs to, up to either the hard limit specified with the memory
        parameter (if applicable), or all of the available memory on the container
        instance, whichever comes first.

        At least one of ``memoryLimit`` and ``memoryReservation`` is required.
        If both are specified, then ``memoryLimit`` must be equal to ``memoryReservation``

        *Note*: To maximize your resource utilization, provide your jobs with as much memory as possible
        for the specific instance type that you are using.

        :default: - No memory reserved

        :see: https://docs.aws.amazon.com/batch/latest/userguide/memory-management.html
        '''
        return typing.cast(typing.Optional[_Size_7b441c34], jsii.get(self, "memoryReservation"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of this container.

        :default: : ``'Default'``
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @builtins.property
    @jsii.member(jsii_name="privileged")
    def privileged(self) -> typing.Optional[builtins.bool]:
        '''If specified, gives this container elevated permissions on the host container instance.

        The level of permissions are similar to the root user permissions.

        This parameter maps to ``privileged`` policy in the Privileged pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: false

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#volumes-and-file-systems
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "privileged"))

    @builtins.property
    @jsii.member(jsii_name="readonlyRootFilesystem")
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''If specified, gives this container readonly access to its root file system.

        This parameter maps to ``ReadOnlyRootFilesystem`` policy in the Volumes and file systems pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: false

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#volumes-and-file-systems
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "readonlyRootFilesystem"))

    @builtins.property
    @jsii.member(jsii_name="runAsGroup")
    def run_as_group(self) -> typing.Optional[jsii.Number]:
        '''If specified, the container is run as the specified group ID (``gid``).

        If this parameter isn't specified, the default is the group that's specified in the image metadata.
        This parameter maps to ``RunAsGroup`` and ``MustRunAs`` policy in the Users and groups pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: none

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#users-and-groups
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "runAsGroup"))

    @builtins.property
    @jsii.member(jsii_name="runAsRoot")
    def run_as_root(self) -> typing.Optional[builtins.bool]:
        '''If specified, the container is run as a user with a ``uid`` other than 0.

        Otherwise, no such rule is enforced.
        This parameter maps to ``RunAsUser`` and ``MustRunAsNonRoot`` policy in the Users and groups pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: - the container is *not* required to run as a non-root user

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#users-and-groups
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "runAsRoot"))

    @builtins.property
    @jsii.member(jsii_name="runAsUser")
    def run_as_user(self) -> typing.Optional[jsii.Number]:
        '''If specified, this container is run as the specified user ID (``uid``).

        This parameter maps to ``RunAsUser`` and ``MustRunAs`` policy in the Users and groups pod security policies in the Kubernetes documentation.

        *Note*: this is only compatible with Kubernetes < v1.25

        :default: - the user that is specified in the image metadata.

        :see: https://kubernetes.io/docs/concepts/security/pod-security-policy/#users-and-groups
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "runAsUser"))

    @jsii.member(jsii_name="addVolume")
    def add_volume(self, volume: EksVolume) -> None:
        '''Mount a Volume to this container.

        Automatically added to the Pod.

        :param volume: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c97587025f2e2da78c21320f554231e20e2b3d601c53519249b32918c7290227)
            check_type(argname="argument volume", value=volume, expected_type=type_hints["volume"])
        return typing.cast(None, jsii.invoke(self, "addVolume", [volume]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEksContainerDefinition).__jsii_proxy_class__ = lambda : _IEksContainerDefinitionProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_batch.IJobDefinition")
class IJobDefinition(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents a JobDefinition.'''

    @builtins.property
    @jsii.member(jsii_name="jobDefinitionArn")
    def job_definition_arn(self) -> builtins.str:
        '''The ARN of this job definition.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="jobDefinitionName")
    def job_definition_name(self) -> builtins.str:
        '''The name of this job definition.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="retryStrategies")
    def retry_strategies(self) -> typing.List["RetryStrategy"]:
        '''Defines the retry behavior for this job.

        :default: - no ``RetryStrategy``
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The default parameters passed to the container These parameters can be referenced in the ``command`` that you give to the container.

        :default: none

        :see: https://docs.aws.amazon.com/batch/latest/userguide/job_definition_parameters.html#parameters
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="retryAttempts")
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The number of times to retry a job.

        The job is retried on failure the same number of attempts as the value.

        :default: 1
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="schedulingPriority")
    def scheduling_priority(self) -> typing.Optional[jsii.Number]:
        '''The priority of this Job.

        Only used in Fairshare Scheduling
        to decide which job to run first when there are multiple jobs
        with the same share identifier.

        :default: none
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="timeout")
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The timeout time for jobs that are submitted with this job definition.

        After the amount of time you specify passes,
        Batch terminates your jobs if they aren't finished.

        :default: - no timeout
        '''
        ...

    @jsii.member(jsii_name="addRetryStrategy")
    def add_retry_strategy(self, strategy: "RetryStrategy") -> None:
        '''Add a RetryStrategy to this JobDefinition.

        :param strategy: -
        '''
        ...


class _IJobDefinitionProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents a JobDefinition.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_batch.IJobDefinition"

    @builtins.property
    @jsii.member(jsii_name="jobDefinitionArn")
    def job_definition_arn(self) -> builtins.str:
        '''The ARN of this job definition.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "jobDefinitionArn"))

    @builtins.property
    @jsii.member(jsii_name="jobDefinitionName")
    def job_definition_name(self) -> builtins.str:
        '''The name of this job definition.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "jobDefinitionName"))

    @builtins.property
    @jsii.member(jsii_name="retryStrategies")
    def retry_strategies(self) -> typing.List["RetryStrategy"]:
        '''Defines the retry behavior for this job.

        :default: - no ``RetryStrategy``
        '''
        return typing.cast(typing.List["RetryStrategy"], jsii.get(self, "retryStrategies"))

    @builtins.property
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''The default parameters passed to the container These parameters can be referenced in the ``command`` that you give to the container.

        :default: none

        :see: https://docs.aws.amazon.com/batch/latest/userguide/job_definition_parameters.html#parameters
        '''
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], jsii.get(self, "parameters"))

    @builtins.property
    @jsii.member(jsii_name="retryAttempts")
    def retry_attempts(self) -> typing.Optional[jsii.Number]:
        '''The number of times to retry a job.

        The job is retried on failure the same number of attempts as the value.

        :default: 1
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "retryAttempts"))

    @builtins.property
    @jsii.member(jsii_name="schedulingPriority")
    def scheduling_priority(self) -> typing.Optional[jsii.Number]:
        '''The priority of this Job.

        Only used in Fairshare Scheduling
        to decide which job to run first when there are multiple jobs
        with the same share identifier.

        :default: none
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "schedulingPriority"))

    @builtins.property
    @jsii.member(jsii_name="timeout")
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The timeout time for jobs that are submitted with this job definition.

        After the amount of time you specify passes,
        Batch terminates your jobs if they aren't finished.

        :default: - no timeout
        '''
        return typing.cast(typing.Optional[_Duration_4839e8c3], jsii.get(self, "timeout"))

    @jsii.member(jsii_name="addRetryStrategy")
    def add_retry_strategy(self, strategy: "RetryStrategy") -> None:
        '''Add a RetryStrategy to this JobDefinition.

        :param strategy: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f5760c945c49bffb46907844e0f7cabef0079a55024105a0d8989b45977505f4)
            check_type(argname="argument strategy", value=strategy, expected_type=type_hints["strategy"])
        return typing.cast(None, jsii.invoke(self, "addRetryStrategy", [strategy]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IJobDefinition).__jsii_proxy_class__ = lambda : _IJobDefinitionProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_batch.IJobQueue")
class IJobQueue(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents a JobQueue.'''

    @builtins.property
    @jsii.member(jsii_name="computeEnvironments")
    def compute_environments(self) -> typing.List["OrderedComputeEnvironment"]:
        '''The set of compute environments mapped to a job queue and their order relative to each other.

        The job scheduler uses this parameter to determine which compute environment runs a specific job.
        Compute environments must be in the VALID state before you can associate them with a job queue.
        You can associate up to three compute environments with a job queue.
        All of the compute environments must be either EC2 (EC2 or SPOT) or Fargate (FARGATE or FARGATE_SPOT);
        EC2 and Fargate compute environments can't be mixed.

        *Note*: All compute environments that are associated with a job queue must share the same architecture.
        AWS Batch doesn't support mixing compute environment architecture types in a single job queue.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="jobQueueArn")
    def job_queue_arn(self) -> builtins.str:
        '''The ARN of this job queue.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="jobQueueName")
    def job_queue_name(self) -> builtins.str:
        '''The name of the job queue.

        It can be up to 128 letters long.
        It can contain uppercase and lowercase letters, numbers, hyphens (-), and underscores (_)

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="priority")
    def priority(self) -> jsii.Number:
        '''The priority of the job queue.

        Job queues with a higher priority are evaluated first when associated with the same compute environment.
        Priority is determined in descending order.
        For example, a job queue with a priority value of 10 is given scheduling preference over a job queue with a priority value of 1.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="enabled")
    def enabled(self) -> typing.Optional[builtins.bool]:
        '''If the job queue is enabled, it is able to accept jobs.

        Otherwise, new jobs can't be added to the queue, but jobs already in the queue can finish.

        :default: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="schedulingPolicy")
    def scheduling_policy(self) -> typing.Optional["ISchedulingPolicy"]:
        '''The SchedulingPolicy for this JobQueue.

        Instructs the Scheduler how to schedule different jobs.

        :default: - no scheduling policy
        '''
        ...

    @jsii.member(jsii_name="addComputeEnvironment")
    def add_compute_environment(
        self,
        compute_environment: IComputeEnvironment,
        order: jsii.Number,
    ) -> None:
        '''Add a ``ComputeEnvironment`` to this Queue.

        The Queue will prefer lower-order ``ComputeEnvironment``s.

        :param compute_environment: -
        :param order: -
        '''
        ...


class _IJobQueueProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents a JobQueue.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_batch.IJobQueue"

    @builtins.property
    @jsii.member(jsii_name="computeEnvironments")
    def compute_environments(self) -> typing.List["OrderedComputeEnvironment"]:
        '''The set of compute environments mapped to a job queue and their order relative to each other.

        The job scheduler uses this parameter to determine which compute environment runs a specific job.
        Compute environments must be in the VALID state before you can associate them with a job queue.
        You can associate up to three compute environments with a job queue.
        All of the compute environments must be either EC2 (EC2 or SPOT) or Fargate (FARGATE or FARGATE_SPOT);
        EC2 and Fargate compute environments can't be mixed.

        *Note*: All compute environments that are associated with a job queue must share the same architecture.
        AWS Batch doesn't support mixing compute environment architecture types in a single job queue.
        '''
        return typing.cast(typing.List["OrderedComputeEnvironment"], jsii.get(self, "computeEnvironments"))

    @builtins.property
    @jsii.member(jsii_name="jobQueueArn")
    def job_queue_arn(self) -> builtins.str:
        '''The ARN of this job queue.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "jobQueueArn"))

    @builtins.property
    @jsii.member(jsii_name="jobQueueName")
    def job_queue_name(self) -> builtins.str:
        '''The name of the job queue.

        It can be up to 128 letters long.
        It can contain uppercase and lowercase letters, numbers, hyphens (-), and underscores (_)

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "jobQueueName"))

    @builtins.property
    @jsii.member(jsii_name="priority")
    def priority(self) -> jsii.Number:
        '''The priority of the job queue.

        Job queues with a higher priority are evaluated first when associated with the same compute environment.
        Priority is determined in descending order.
        For example, a job queue with a priority value of 10 is given scheduling preference over a job queue with a priority value of 1.
        '''
        return typing.cast(jsii.Number, jsii.get(self, "priority"))

    @builtins.property
    @jsii.member(jsii_name="enabled")
    def enabled(self) -> typing.Optional[builtins.bool]:
        '''If the job queue is enabled, it is able to accept jobs.

        Otherwise, new jobs can't be added to the queue, but jobs already in the queue can finish.

        :default: true
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "enabled"))

    @builtins.property
    @jsii.member(jsii_name="schedulingPolicy")
    def scheduling_policy(self) -> typing.Optional["ISchedulingPolicy"]:
        '''The SchedulingPolicy for this JobQueue.

        Instructs the Scheduler how to schedule different jobs.

        :default: - no scheduling policy
        '''
        return typing.cast(typing.Optional["ISchedulingPolicy"], jsii.get(self, "schedulingPolicy"))

    @jsii.member(jsii_name="addComputeEnvironment")
    def add_compute_environment(
        self,
        compute_environment: IComputeEnvironment,
        order: jsii.Number,
    ) -> None:
        '''Add a ``ComputeEnvironment`` to this Queue.

        The Queue will prefer lower-order ``ComputeEnvironment``s.

        :param compute_environment: -
        :param order: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30f04cfa57530bae3195ca823d89191697e4817baf332ff9c874f63c1aa17378)
            check_type(argname="argument compute_environment", value=compute_environment, expected_type=type_hints["compute_environment"])
            check_type(argname="argument order", value=order, expected_type=type_hints["order"])
        return typing.cast(None, jsii.invoke(self, "addComputeEnvironment", [compute_environment, order]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IJobQueue).__jsii_proxy_class__ = lambda : _IJobQueueProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_batch.IManagedComputeEnvironment")
class IManagedComputeEnvironment(
    IComputeEnvironment,
    _IConnectable_10015a05,
    _ITaggable_36806126,
    typing_extensions.Protocol,
):
    '''Represents a Managed ComputeEnvironment.

    Batch will provision EC2 Instances to
    meet the requirements of the jobs executing in this ComputeEnvironment.
    '''

    @builtins.property
    @jsii.member(jsii_name="maxvCpus")
    def maxv_cpus(self) -> jsii.Number:
        '''The maximum vCpus this ``ManagedComputeEnvironment`` can scale up to.

        *Note*: if this Compute Environment uses EC2 resources (not Fargate) with either ``AllocationStrategy.BEST_FIT_PROGRESSIVE`` or
        ``AllocationStrategy.SPOT_CAPACITY_OPTIMIZED``, or ``AllocationStrategy.BEST_FIT`` with Spot instances,
        The scheduler may exceed this number by at most one of the instances specified in ``instanceTypes``
        or ``instanceClasses``.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="securityGroups")
    def security_groups(self) -> typing.List[_ISecurityGroup_acf8a799]:
        '''The security groups this Compute Environment will launch instances in.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="replaceComputeEnvironment")
    def replace_compute_environment(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether this Compute Environment is replaced if an update is made that requires replacing its instances.

        To enable more properties to be updated,
        set this property to ``false``. When changing the value of this property to false,
        do not change any other properties at the same time.
        If other properties are changed at the same time,
        and the change needs to be rolled back but it can't,
        it's possible for the stack to go into the UPDATE_ROLLBACK_FAILED state.
        You can't update a stack that is in the UPDATE_ROLLBACK_FAILED state.
        However, if you can continue to roll it back,
        you can return the stack to its original settings and then try to update it again.

        The properties which require a replacement of the Compute Environment are:

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-updating-stacks-continueupdaterollback.html
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="spot")
    def spot(self) -> typing.Optional[builtins.bool]:
        '''Whether or not to use spot instances.

        Spot instances are less expensive EC2 instances that can be
        reclaimed by EC2 at any time; your job will be given two minutes
        of notice before reclamation.

        :default: false
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="terminateOnUpdate")
    def terminate_on_update(self) -> typing.Optional[builtins.bool]:
        '''Whether or not any running jobs will be immediately terminated when an infrastructure update occurs.

        If this is enabled, any terminated jobs may be retried, depending on the job's
        retry policy.

        :default: false

        :see: https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="updateTimeout")
    def update_timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''Only meaningful if ``terminateOnUpdate`` is ``false``.

        If so,
        when an infrastructure update is triggered, any running jobs
        will be allowed to run until ``updateTimeout`` has expired.

        :default: 30 minutes

        :see: https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="updateToLatestImageVersion")
    def update_to_latest_image_version(self) -> typing.Optional[builtins.bool]:
        '''Whether or not the AMI is updated to the latest one supported by Batch when an infrastructure update occurs.

        If you specify a specific AMI, this property will be ignored.

        Note: the CDK will never set this value by default, ``false`` will set by CFN.
        This is to avoid a deployment failure that occurs when this value is set.

        :default: false

        :see: https://github.com/aws/aws-cdk/issues/27054
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="vpcSubnets")
    def vpc_subnets(self) -> typing.Optional[_SubnetSelection_e57d76df]:
        '''The VPC Subnets this Compute Environment will launch instances in.'''
        ...


class _IManagedComputeEnvironmentProxy(
    jsii.proxy_for(IComputeEnvironment), # type: ignore[misc]
    jsii.proxy_for(_IConnectable_10015a05), # type: ignore[misc]
    jsii.proxy_for(_ITaggable_36806126), # type: ignore[misc]
):
    '''Represents a Managed ComputeEnvironment.

    Batch will provision EC2 Instances to
    meet the requirements of the jobs executing in this ComputeEnvironment.
    '''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_batch.IManagedComputeEnvironment"

    @builtins.property
    @jsii.member(jsii_name="maxvCpus")
    def maxv_cpus(self) -> jsii.Number:
        '''The maximum vCpus this ``ManagedComputeEnvironment`` can scale up to.

        *Note*: if this Compute Environment uses EC2 resources (not Fargate) with either ``AllocationStrategy.BEST_FIT_PROGRESSIVE`` or
        ``AllocationStrategy.SPOT_CAPACITY_OPTIMIZED``, or ``AllocationStrategy.BEST_FIT`` with Spot instances,
        The scheduler may exceed this number by at most one of the instances specified in ``instanceTypes``
        or ``instanceClasses``.
        '''
        return typing.cast(jsii.Number, jsii.get(self, "maxvCpus"))

    @builtins.property
    @jsii.member(jsii_name="securityGroups")
    def security_groups(self) -> typing.List[_ISecurityGroup_acf8a799]:
        '''The security groups this Compute Environment will launch instances in.'''
        return typing.cast(typing.List[_ISecurityGroup_acf8a799], jsii.get(self, "securityGroups"))

    @builtins.property
    @jsii.member(jsii_name="replaceComputeEnvironment")
    def replace_compute_environment(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether this Compute Environment is replaced if an update is made that requires replacing its instances.

        To enable more properties to be updated,
        set this property to ``false``. When changing the value of this property to false,
        do not change any other properties at the same time.
        If other properties are changed at the same time,
        and the change needs to be rolled back but it can't,
        it's possible for the stack to go into the UPDATE_ROLLBACK_FAILED state.
        You can't update a stack that is in the UPDATE_ROLLBACK_FAILED state.
        However, if you can continue to roll it back,
        you can return the stack to its original settings and then try to update it again.

        The properties which require a replacement of the Co