r'''
# AWS App Mesh Construct Library

AWS App Mesh is a service mesh based on the [Envoy](https://www.envoyproxy.io/) proxy that makes it easy to monitor and control microservices. App Mesh standardizes how your microservices communicate, giving you end-to-end visibility and helping to ensure high-availability for your applications.

App Mesh gives you consistent visibility and network traffic controls for every microservice in an application.

App Mesh supports microservice applications that use service discovery naming for their components. To use App Mesh, you must have an existing application running on AWS Fargate, Amazon ECS, Amazon EKS, Kubernetes on AWS, or Amazon EC2.

For further information on **AWS App Mesh**, visit the [AWS App Mesh Documentation](https://docs.aws.amazon.com/app-mesh/index.html).

## Create the App and Stack

```python
app = cdk.App()
stack = cdk.Stack(app, "stack")
```

## Creating the Mesh

A service mesh is a logical boundary for network traffic between the services that reside within it.

After you create your service mesh, you can create virtual services, virtual nodes, virtual routers, and routes to distribute traffic between the applications in your mesh.

The following example creates the `AppMesh` service mesh with the default egress filter of `DROP_ALL`. See [the AWS CloudFormation `EgressFilter` resource](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-egressfilter.html) for more info on egress filters.

```python
mesh = appmesh.Mesh(self, "AppMesh",
    mesh_name="myAwsMesh"
)
```

The mesh can instead be created with the `ALLOW_ALL` egress filter by providing the `egressFilter` property.

```python
mesh = appmesh.Mesh(self, "AppMesh",
    mesh_name="myAwsMesh",
    egress_filter=appmesh.MeshFilterType.ALLOW_ALL
)
```

A mesh with an IP preference can be created by providing the property `serviceDiscovery` that specifes an `ipPreference`.

```python
mesh = appmesh.Mesh(self, "AppMesh",
    mesh_name="myAwsMesh",
    service_discovery=appmesh.MeshServiceDiscovery(
        ip_preference=appmesh.IpPreference.IPV4_ONLY
    )
)
```

## Adding VirtualRouters

A *mesh* uses  *virtual routers* as logical units to route requests to *virtual nodes*.

Virtual routers handle traffic for one or more virtual services within your mesh.
After you create a virtual router, you can create and associate routes to your virtual router that direct incoming requests to different virtual nodes.

```python
# mesh: appmesh.Mesh

router = mesh.add_virtual_router("router",
    listeners=[appmesh.VirtualRouterListener.http(8080)]
)
```

Note that creating the router using the `addVirtualRouter()` method places it in the same stack as the mesh
(which might be different from the current stack).
The router can also be created using the `VirtualRouter` constructor (passing in the mesh) instead of calling the `addVirtualRouter()` method.
This is particularly useful when splitting your resources between many stacks: for example, defining the mesh itself as part of an infrastructure stack, but defining the other resources, such as routers, in the application stack:

```python
# infra_stack: cdk.Stack
# app_stack: cdk.Stack


mesh = appmesh.Mesh(infra_stack, "AppMesh",
    mesh_name="myAwsMesh",
    egress_filter=appmesh.MeshFilterType.ALLOW_ALL
)

# the VirtualRouter will belong to 'appStack',
# even though the Mesh belongs to 'infraStack'
router = appmesh.VirtualRouter(app_stack, "router",
    mesh=mesh,  # notice that mesh is a required property when creating a router with the 'new' statement
    listeners=[appmesh.VirtualRouterListener.http(8081)]
)
```

The same is true for other `add*()` methods in the App Mesh construct library.

The `VirtualRouterListener` class lets you define protocol-specific listeners.
The `http()`, `http2()`, `grpc()` and `tcp()` methods create listeners for the named protocols.
They accept a single parameter that defines the port to on which requests will be matched.
The port parameter defaults to 8080 if omitted.

## Adding a VirtualService

A *virtual service* is an abstraction of a real service that is provided by a virtual node directly, or indirectly by means of a virtual router. Dependent services call your virtual service by its `virtualServiceName`, and those requests are routed to the virtual node or virtual router specified as the provider for the virtual service.

We recommend that you use the service discovery name of the real service that you're targeting (such as `my-service.default.svc.cluster.local`).

When creating a virtual service:

* If you want the virtual service to spread traffic across multiple virtual nodes, specify a virtual router.
* If you want the virtual service to reach a virtual node directly, without a virtual router, specify a virtual node.

Adding a virtual router as the provider:

```python
# router: appmesh.VirtualRouter


appmesh.VirtualService(self, "virtual-service",
    virtual_service_name="my-service.default.svc.cluster.local",  # optional
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_router(router)
)
```

Adding a virtual node as the provider:

```python
# node: appmesh.VirtualNode


appmesh.VirtualService(self, "virtual-service",
    virtual_service_name="my-service.default.svc.cluster.local",  # optional
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_node(node)
)
```

## Adding a VirtualNode

A *virtual node* acts as a logical pointer to a particular task group, such as an Amazon ECS service or a Kubernetes deployment.

When you create a virtual node, accept inbound traffic by specifying a *listener*. Outbound traffic that your virtual node expects to send should be specified as a *back end*.

The response metadata for your new virtual node contains the Amazon Resource Name (ARN) that is associated with the virtual node. Set this value (either the full ARN or the truncated resource name) as the `APPMESH_VIRTUAL_NODE_NAME` environment variable for your task group's Envoy proxy container in your task definition or pod spec. For example, the value could be `mesh/default/virtualNode/simpleapp`. This is then mapped to the `node.id` and `node.cluster` Envoy parameters.

> **Note**
> If you require your Envoy stats or tracing to use a different name, you can override the `node.cluster` value that is set by `APPMESH_VIRTUAL_NODE_NAME` with the `APPMESH_VIRTUAL_NODE_CLUSTER` environment variable.

```python
# mesh: appmesh.Mesh
vpc = ec2.Vpc(self, "vpc")
namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
    vpc=vpc,
    name="domain.local"
)
service = namespace.create_service("Svc")
node = mesh.add_virtual_node("virtual-node",
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        port=8081,
        health_check=appmesh.HealthCheck.http(
            healthy_threshold=3,
            interval=Duration.seconds(5),  # minimum
            path="/health-check-path",
            timeout=Duration.seconds(2),  # minimum
            unhealthy_threshold=2
        )
    )],
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
)
```

Create a `VirtualNode` with the constructor and add tags.

```python
# mesh: appmesh.Mesh
# service: cloudmap.Service


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        port=8080,
        health_check=appmesh.HealthCheck.http(
            healthy_threshold=3,
            interval=Duration.seconds(5),
            path="/ping",
            timeout=Duration.seconds(2),
            unhealthy_threshold=2
        ),
        timeout=appmesh.HttpTimeout(
            idle=Duration.seconds(5)
        )
    )],
    backend_defaults=appmesh.BackendDefaults(
        tls_client_policy=appmesh.TlsClientPolicy(
            validation=appmesh.TlsValidation(
                trust=appmesh.TlsValidationTrust.file("/keys/local_cert_chain.pem")
            )
        )
    ),
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
)

cdk.Tags.of(node).add("Environment", "Dev")
```

Create a `VirtualNode` with the customized access logging format.

```python
# mesh: appmesh.Mesh
# service: cloudmap.Service

node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        port=8080,
        health_check=appmesh.HealthCheck.http(
            healthy_threshold=3,
            interval=cdk.Duration.seconds(5),
            path="/ping",
            timeout=cdk.Duration.seconds(2),
            unhealthy_threshold=2
        ),
        timeout=appmesh.HttpTimeout(
            idle=cdk.Duration.seconds(5)
        )
    )],
    backend_defaults=appmesh.BackendDefaults(
        tls_client_policy=appmesh.TlsClientPolicy(
            validation=appmesh.TlsValidation(
                trust=appmesh.TlsValidationTrust.file("/keys/local_cert_chain.pem")
            )
        )
    ),
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout",
        appmesh.LoggingFormat.from_json({"test_key1": "testValue1", "test_key2": "testValue2"}))
)
```

By using a key-value pair indexed signature, you can specify json key pairs to customize the log entry pattern. You can also use text format as below. You can only specify one of these 2 formats.

```text
  accessLog: appmesh.AccessLog.fromFilePath('/dev/stdout', appmesh.LoggingFormat.fromText('test_pattern')),
```

For what values and operators you can use for these two formats, please visit the latest envoy documentation. (https://www.envoyproxy.io/docs/envoy/latest/configuration/observability/access_log/usage)
Create a `VirtualNode` with the constructor and add backend virtual service.

```python
# mesh: appmesh.Mesh
# router: appmesh.VirtualRouter
# service: cloudmap.Service


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        port=8080,
        health_check=appmesh.HealthCheck.http(
            healthy_threshold=3,
            interval=Duration.seconds(5),
            path="/ping",
            timeout=Duration.seconds(2),
            unhealthy_threshold=2
        ),
        timeout=appmesh.HttpTimeout(
            idle=Duration.seconds(5)
        )
    )],
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
)

virtual_service = appmesh.VirtualService(self, "service-1",
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_router(router),
    virtual_service_name="service1.domain.local"
)

node.add_backend(appmesh.Backend.virtual_service(virtual_service))
```

The `listeners` property can be left blank and added later with the `node.addListener()` method. The `serviceDiscovery` property must be specified when specifying a listener.

The `backends` property can be added with `node.addBackend()`. In the example, we define a virtual service and add it to the virtual node to allow egress traffic to other nodes.

The `backendDefaults` property is added to the node while creating the virtual node. These are the virtual node's default settings for all backends.

The `VirtualNode.addBackend()` method is especially useful if you want to create a circular traffic flow by having a Virtual Service as a backend whose provider is that same Virtual Node:

```python
# mesh: appmesh.Mesh


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node")
)

virtual_service = appmesh.VirtualService(self, "service-1",
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_node(node),
    virtual_service_name="service1.domain.local"
)

node.add_backend(appmesh.Backend.virtual_service(virtual_service))
```

### Adding TLS to a listener

The `tls` property specifies TLS configuration when creating a listener for a virtual node or a virtual gateway.
Provide the TLS certificate to the proxy in one of the following ways:

* A certificate from AWS Certificate Manager (ACM).
* A customer-provided certificate (specify a `certificateChain` path file and a `privateKey` file path).
* A certificate provided by a Secrets Discovery Service (SDS) endpoint over local Unix Domain Socket (specify its `secretName`).

```python
# A Virtual Node with listener TLS from an ACM provided certificate
# cert: certificatemanager.Certificate
# mesh: appmesh.Mesh


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node"),
    listeners=[appmesh.VirtualNodeListener.grpc(
        port=80,
        tls=appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.acm(cert)
        )
    )]
)

# A Virtual Gateway with listener TLS from a customer provided file certificate
gateway = appmesh.VirtualGateway(self, "gateway",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.grpc(
        port=8080,
        tls=appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey")
        )
    )],
    virtual_gateway_name="gateway"
)

# A Virtual Gateway with listener TLS from a SDS provided certificate
gateway2 = appmesh.VirtualGateway(self, "gateway2",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.http2(
        port=8080,
        tls=appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.sds("secrete_certificate")
        )
    )],
    virtual_gateway_name="gateway2"
)
```

### Adding mutual TLS authentication

Mutual TLS authentication is an optional component of TLS that offers two-way peer authentication.
To enable mutual TLS authentication, add the `mutualTlsCertificate` property to TLS client policy and/or the `mutualTlsValidation` property to your TLS listener.

`tls.mutualTlsValidation` and `tlsClientPolicy.mutualTlsCertificate` can be sourced from either:

* A customer-provided certificate (specify a `certificateChain` path file and a `privateKey` file path).
* A certificate provided by a Secrets Discovery Service (SDS) endpoint over local Unix Domain Socket (specify its `secretName`).

> **Note**
> Currently, a certificate from AWS Certificate Manager (ACM) cannot be used for mutual TLS authentication.

```python
# mesh: appmesh.Mesh


node1 = appmesh.VirtualNode(self, "node1",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node"),
    listeners=[appmesh.VirtualNodeListener.grpc(
        port=80,
        tls=appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey"),
            # Validate a file client certificates to enable mutual TLS authentication when a client provides a certificate.
            mutual_tls_validation=appmesh.MutualTlsValidation(
                trust=appmesh.TlsValidationTrust.file("path-to-certificate")
            )
        )
    )]
)

certificate_authority_arn = "arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/12345678-1234-1234-1234-123456789012"
node2 = appmesh.VirtualNode(self, "node2",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node2"),
    backend_defaults=appmesh.BackendDefaults(
        tls_client_policy=appmesh.TlsClientPolicy(
            ports=[8080, 8081],
            validation=appmesh.TlsValidation(
                subject_alternative_names=appmesh.SubjectAlternativeNames.matching_exactly("mesh-endpoint.apps.local"),
                trust=appmesh.TlsValidationTrust.acm([
                    acmpca.CertificateAuthority.from_certificate_authority_arn(self, "certificate", certificate_authority_arn)
                ])
            ),
            # Provide a SDS client certificate when a server requests it and enable mutual TLS authentication.
            mutual_tls_certificate=appmesh.TlsCertificate.sds("secret_certificate")
        )
    )
)
```

### Adding outlier detection to a Virtual Node listener

The `outlierDetection` property adds outlier detection to a Virtual Node listener. The properties
`baseEjectionDuration`, `interval`, `maxEjectionPercent`, and `maxServerErrors` are required.

```python
# mesh: appmesh.Mesh
# Cloud Map service discovery is currently required for host ejection by outlier detection
vpc = ec2.Vpc(self, "vpc")
namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
    vpc=vpc,
    name="domain.local"
)
service = namespace.create_service("Svc")
node = mesh.add_virtual_node("virtual-node",
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        outlier_detection=appmesh.OutlierDetection(
            base_ejection_duration=Duration.seconds(10),
            interval=Duration.seconds(30),
            max_ejection_percent=50,
            max_server_errors=5
        )
    )]
)
```

### Adding a connection pool to a listener

The `connectionPool` property can be added to a Virtual Node listener or Virtual Gateway listener to add a request connection pool. Each listener protocol type has its own connection pool properties.

```python
# A Virtual Node with a gRPC listener with a connection pool set
# mesh: appmesh.Mesh

node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    # DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
    # LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
    # whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
    # By default, the response type is assumed to be LOAD_BALANCER
    service_discovery=appmesh.ServiceDiscovery.dns("node", appmesh.DnsResponseType.ENDPOINTS),
    listeners=[appmesh.VirtualNodeListener.http(
        port=80,
        connection_pool=appmesh.HttpConnectionPool(
            max_connections=100,
            max_pending_requests=10
        )
    )]
)

# A Virtual Gateway with a gRPC listener with a connection pool set
gateway = appmesh.VirtualGateway(self, "gateway",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.grpc(
        port=8080,
        connection_pool=appmesh.GrpcConnectionPool(
            max_requests=10
        )
    )],
    virtual_gateway_name="gateway"
)
```

### Adding an IP Preference to a Virtual Node

An `ipPreference` can be specified as part of a Virtual Node's service discovery. An IP preference defines how clients for this Virtual Node will interact with it.

There a four different IP preferences available to use which each specify what IP versions this Virtual Node will use and prefer.

* `IPv4_ONLY` - Only use IPv4. For CloudMap service discovery, only IPv4 addresses returned from CloudMap will be used. For DNS service discovery, Envoy's DNS resolver will only resolve DNS queries for IPv4.
* `IPv4_PREFERRED` - Prefer IPv4 and fall back to IPv6. For CloudMap service discovery, an IPv4 address will be used if returned from CloudMap. Otherwise, an IPv6 address will be used if available. For DNS service discovery, Envoy's DNS resolver will first attempt to resolve DNS queries using IPv4 and fall back to IPv6.
* `IPv6_ONLY` - Only use IPv6. For CloudMap service discovery, only IPv6 addresses returned from CloudMap will be used. For DNS service discovery, Envoy's DNS resolver will only resolve DNS queries for IPv6.
* `IPv6_PREFERRED` - Prefer IPv6 and fall back to IPv4. For CloudMap service discovery, an IPv6 address will be used if returned from CloudMap. Otherwise, an IPv4 address will be used if available. For DNS service discovery, Envoy's DNS resolver will first attempt to resolve DNS queries using IPv6 and fall back to IPv4.

```python
mesh = appmesh.Mesh(self, "mesh",
    mesh_name="mesh-with-preference"
)

# Virtual Node with DNS service discovery and an IP preference
dns_node = appmesh.VirtualNode(self, "dns-node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("test", appmesh.DnsResponseType.LOAD_BALANCER, appmesh.IpPreference.IPV4_ONLY)
)

# Virtual Node with CloudMap service discovery and an IP preference
vpc = ec2.Vpc(self, "vpc")
namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
    vpc=vpc,
    name="domain.local"
)
service = namespace.create_service("Svc")

instance_attribute = {}
instance_attribute.test_key = "testValue"

cloudmap_node = appmesh.VirtualNode(self, "cloudmap-node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service, instance_attribute, appmesh.IpPreference.IPV4_ONLY)
)
```

## Adding a Route

A *route* matches requests with an associated virtual router and distributes traffic to its associated virtual nodes.
The route distributes matching requests to one or more target virtual nodes with relative weighting.

The `RouteSpec` class lets you define protocol-specific route specifications.
The `tcp()`, `http()`, `http2()`, and `grpc()` methods create a specification for the named protocols.

For HTTP-based routes, the match field can match on path (prefix, exact, or regex), HTTP method, scheme,
HTTP headers, and query parameters. By default, HTTP-based routes match all requests.

For gRPC-based routes, the match field can  match on service name, method name, and metadata.
When specifying the method name, the service name must also be specified.

For example, here's how to add an HTTP route that matches based on a prefix of the URL path:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http",
    route_spec=appmesh.RouteSpec.http(
        weighted_targets=[appmesh.WeightedTarget(
            virtual_node=node
        )
        ],
        match=appmesh.HttpRouteMatch(
            # Path that is passed to this method must start with '/'.
            path=appmesh.HttpRoutePathMatch.starts_with("/path-to-app")
        )
    )
)
```

Add an HTTP2 route that matches based on exact path, method, scheme, headers, and query parameters:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http2",
    route_spec=appmesh.RouteSpec.http2(
        weighted_targets=[appmesh.WeightedTarget(
            virtual_node=node
        )
        ],
        match=appmesh.HttpRouteMatch(
            path=appmesh.HttpRoutePathMatch.exactly("/exact"),
            method=appmesh.HttpRouteMethod.POST,
            protocol=appmesh.HttpRouteProtocol.HTTPS,
            headers=[
                # All specified headers must match for the route to match.
                appmesh.HeaderMatch.value_is("Content-Type", "application/json"),
                appmesh.HeaderMatch.value_is_not("Content-Type", "application/json")
            ],
            query_parameters=[
                # All specified query parameters must match for the route to match.
                appmesh.QueryParameterMatch.value_is("query-field", "value")
            ]
        )
    )
)
```

Add a single route with two targets and split traffic 50/50:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http",
    route_spec=appmesh.RouteSpec.http(
        weighted_targets=[appmesh.WeightedTarget(
            virtual_node=node,
            weight=50
        ), appmesh.WeightedTarget(
            virtual_node=node,
            weight=50
        )
        ],
        match=appmesh.HttpRouteMatch(
            path=appmesh.HttpRoutePathMatch.starts_with("/path-to-app")
        )
    )
)
```

Add an http2 route with retries:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http2-retry",
    route_spec=appmesh.RouteSpec.http2(
        weighted_targets=[appmesh.WeightedTarget(virtual_node=node)],
        retry_policy=appmesh.HttpRetryPolicy(
            # Retry if the connection failed
            tcp_retry_events=[appmesh.TcpRetryEvent.CONNECTION_ERROR],
            # Retry if HTTP responds with a gateway error (502, 503, 504)
            http_retry_events=[appmesh.HttpRetryEvent.GATEWAY_ERROR],
            # Retry five times
            retry_attempts=5,
            # Use a 1 second timeout per retry
            retry_timeout=Duration.seconds(1)
        )
    )
)
```

Add a gRPC route with retries:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-grpc-retry",
    route_spec=appmesh.RouteSpec.grpc(
        weighted_targets=[appmesh.WeightedTarget(virtual_node=node)],
        match=appmesh.GrpcRouteMatch(service_name="servicename"),
        retry_policy=appmesh.GrpcRetryPolicy(
            tcp_retry_events=[appmesh.TcpRetryEvent.CONNECTION_ERROR],
            http_retry_events=[appmesh.HttpRetryEvent.GATEWAY_ERROR],
            # Retry if gRPC responds that the request was cancelled, a resource
            # was exhausted, or if the service is unavailable
            grpc_retry_events=[appmesh.GrpcRetryEvent.CANCELLED, appmesh.GrpcRetryEvent.RESOURCE_EXHAUSTED, appmesh.GrpcRetryEvent.UNAVAILABLE
            ],
            retry_attempts=5,
            retry_timeout=Duration.seconds(1)
        )
    )
)
```

Add an gRPC route that matches based on method name and metadata:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-grpc-retry",
    route_spec=appmesh.RouteSpec.grpc(
        weighted_targets=[appmesh.WeightedTarget(virtual_node=node)],
        match=appmesh.GrpcRouteMatch(
            # When method name is specified, service name must be also specified.
            method_name="methodname",
            service_name="servicename",
            metadata=[
                # All specified metadata must match for the route to match.
                appmesh.HeaderMatch.value_starts_with("Content-Type", "application/"),
                appmesh.HeaderMatch.value_does_not_start_with("Content-Type", "text/")
            ]
        )
    )
)
```

Add a gRPC route that matches based on port:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-grpc-port",
    route_spec=appmesh.RouteSpec.grpc(
        weighted_targets=[appmesh.WeightedTarget(
            virtual_node=node
        )
        ],
        match=appmesh.GrpcRouteMatch(
            port=1234
        )
    )
)
```

Add a gRPC route with timeout:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http",
    route_spec=appmesh.RouteSpec.grpc(
        weighted_targets=[appmesh.WeightedTarget(
            virtual_node=node
        )
        ],
        match=appmesh.GrpcRouteMatch(
            service_name="my-service.default.svc.cluster.local"
        ),
        timeout=appmesh.GrpcTimeout(
            idle=Duration.seconds(2),
            per_request=Duration.seconds(1)
        )
    )
)
```

## Adding a Virtual Gateway

A *virtual gateway* allows resources outside your mesh to communicate with resources inside your mesh.
The virtual gateway represents an Envoy proxy running in an Amazon ECS task, in a Kubernetes service, or on an Amazon EC2 instance.
Unlike a virtual node, which represents Envoy running with an application, a virtual gateway represents Envoy deployed by itself.

A virtual gateway is similar to a virtual node in that it has a listener that accepts traffic for a particular port and protocol (HTTP, HTTP2, gRPC).
Traffic received by the virtual gateway is directed to other services in your mesh
using rules defined in gateway routes which can be added to your virtual gateway.

Create a virtual gateway with the constructor:

```python
# mesh: appmesh.Mesh

certificate_authority_arn = "arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/12345678-1234-1234-1234-123456789012"

gateway = appmesh.VirtualGateway(self, "gateway",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.http(
        port=443,
        health_check=appmesh.HealthCheck.http(
            interval=Duration.seconds(10)
        )
    )],
    backend_defaults=appmesh.BackendDefaults(
        tls_client_policy=appmesh.TlsClientPolicy(
            ports=[8080, 8081],
            validation=appmesh.TlsValidation(
                trust=appmesh.TlsValidationTrust.acm([
                    acmpca.CertificateAuthority.from_certificate_authority_arn(self, "certificate", certificate_authority_arn)
                ])
            )
        )
    ),
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout"),
    virtual_gateway_name="virtualGateway"
)
```

Add a virtual gateway directly to the mesh:

```python
# mesh: appmesh.Mesh


gateway = mesh.add_virtual_gateway("gateway",
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout"),
    virtual_gateway_name="virtualGateway",
    listeners=[appmesh.VirtualGatewayListener.http(
        port=443,
        health_check=appmesh.HealthCheck.http(
            interval=Duration.seconds(10)
        )
    )]
)
```

The `listeners` field defaults to an HTTP Listener on port 8080 if omitted.
A gateway route can be added using the `gateway.addGatewayRoute()` method.

The `backendDefaults` property, provided when creating the virtual gateway, specifies the virtual gateway's default settings for all backends.

## Adding a Gateway Route

A *gateway route* is attached to a virtual gateway and routes matching traffic to an existing virtual service.

For HTTP-based gateway routes, the `match` field can be used to match on
path (prefix, exact, or regex), HTTP method, host name, HTTP headers, and query parameters.
By default, HTTP-based gateway routes match all requests.

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-http",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=appmesh.HttpGatewayRouteMatch(
            path=appmesh.HttpGatewayRoutePathMatch.regex("regex")
        )
    )
)
```

For gRPC-based gateway routes, the `match` field can be used to match on service name, host name, port and metadata.

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-grpc",
    route_spec=appmesh.GatewayRouteSpec.grpc(
        route_target=virtual_service,
        match=appmesh.GrpcGatewayRouteMatch(
            hostname=appmesh.GatewayRouteHostnameMatch.ends_with(".example.com")
        )
    )
)
```

For HTTP based gateway routes, App Mesh automatically rewrites the matched prefix path in Gateway Route to “/”.
This automatic rewrite configuration can be overwritten in following ways:

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-http",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=appmesh.HttpGatewayRouteMatch(
            # This disables the default rewrite to '/', and retains original path.
            path=appmesh.HttpGatewayRoutePathMatch.starts_with("/path-to-app/", "")
        )
    )
)

gateway.add_gateway_route("gateway-route-http-1",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=appmesh.HttpGatewayRouteMatch(
            # If the request full path is '/path-to-app/xxxxx', this rewrites the path to '/rewrittenUri/xxxxx'.
            # Please note both `prefixPathMatch` and `rewriteTo` must start and end with the `/` character.
            path=appmesh.HttpGatewayRoutePathMatch.starts_with("/path-to-app/", "/rewrittenUri/")
        )
    )
)
```

If matching other path (exact or regex), only specific rewrite path can be specified.
Unlike `startsWith()` method above, no default rewrite is performed.

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-http-2",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=appmesh.HttpGatewayRouteMatch(
            # This rewrites the path from '/test' to '/rewrittenPath'.
            path=appmesh.HttpGatewayRoutePathMatch.exactly("/test", "/rewrittenPath")
        )
    )
)
```

For HTTP/gRPC based routes, App Mesh automatically rewrites
the original request received at the Virtual Gateway to the destination Virtual Service name.
This default host name rewrite can be configured by specifying the rewrite rule as one of the `match` property:

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-grpc",
    route_spec=appmesh.GatewayRouteSpec.grpc(
        route_target=virtual_service,
        match=appmesh.GrpcGatewayRouteMatch(
            hostname=appmesh.GatewayRouteHostnameMatch.exactly("example.com"),
            # This disables the default rewrite to virtual service name and retain original request.
            rewrite_request_hostname=False
        )
    )
)
```

## Importing Resources

Each App Mesh resource class comes with two static methods, `from<Resource>Arn` and `from<Resource>Attributes` (where `<Resource>` is replaced with the resource name, such as `VirtualNode`) for importing a reference to an existing App Mesh resource.
These imported resources can be used with other resources in your mesh as if they were defined directly in your CDK application.

```python
arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh/virtualNode/testNode"
appmesh.VirtualNode.from_virtual_node_arn(self, "importedVirtualNode", arn)
```

```python
virtual_node_name = "my-virtual-node"
appmesh.VirtualNode.from_virtual_node_attributes(self, "imported-virtual-node",
    mesh=appmesh.Mesh.from_mesh_name(self, "Mesh", "testMesh"),
    virtual_node_name=virtual_node_name
)
```

To import a mesh, again there are two static methods, `fromMeshArn` and `fromMeshName`.

```python
arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh"
appmesh.Mesh.from_mesh_arn(self, "imported-mesh", arn)
```

```python
appmesh.Mesh.from_mesh_name(self, "imported-mesh", "abc")
```

## IAM Grants

`VirtualNode` and `VirtualGateway` provide `grantStreamAggregatedResources` methods that grant identities that are running
Envoy access to stream generated config from App Mesh.

```python
# mesh: appmesh.Mesh

gateway = appmesh.VirtualGateway(self, "testGateway", mesh=mesh)
envoy_user = iam.User(self, "envoyUser")

#
# This will grant `grantStreamAggregatedResources` ONLY for this gateway.
#
gateway.grant_stream_aggregated_resources(envoy_user)
```

## Adding Resources to shared meshes

A shared mesh allows resources created by different accounts to communicate with each other in the same mesh:

```python
# This is the ARN for the mesh from different AWS IAM account ID.
# Ensure mesh is properly shared with your account. For more details, see: https://github.com/aws/aws-cdk/issues/15404
arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh"
shared_mesh = appmesh.Mesh.from_mesh_arn(self, "imported-mesh", arn)

# This VirtualNode resource can communicate with the resources in the mesh from different AWS IAM account ID.
appmesh.VirtualNode(self, "test-node",
    mesh=shared_mesh
)
```
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    CfnResource as _CfnResource_9df397a6,
    CfnTag as _CfnTag_f6864754,
    Duration as _Duration_4839e8c3,
    IInspectable as _IInspectable_c2943556,
    IResolvable as _IResolvable_da3f097b,
    IResource as _IResource_c80c4260,
    ITaggable as _ITaggable_36806126,
    Resource as _Resource_45bc6135,
    TagManager as _TagManager_0a598cb3,
    TreeInspector as _TreeInspector_488e0dd5,
)
from ..aws_acmpca import ICertificateAuthority as _ICertificateAuthority_26727cab
from ..aws_certificatemanager import ICertificate as _ICertificate_c194c70b
from ..aws_iam import Grant as _Grant_a7ae64f8, IGrantable as _IGrantable_71c4f5de
from ..aws_servicediscovery import IService as _IService_46860ae1


class AccessLog(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_appmesh.AccessLog",
):
    '''Configuration for Envoy Access logs for mesh endpoints.

    :exampleMetadata: infused

    Example::

        # mesh: appmesh.Mesh
        vpc = ec2.Vpc(self, "vpc")
        namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
            vpc=vpc,
            name="domain.local"
        )
        service = namespace.create_service("Svc")
        node = mesh.add_virtual_node("virtual-node",
            service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
            listeners=[appmesh.VirtualNodeListener.http(
                port=8081,
                health_check=appmesh.HealthCheck.http(
                    healthy_threshold=3,
                    interval=Duration.seconds(5),  # minimum
                    path="/health-check-path",
                    timeout=Duration.seconds(2),  # minimum
                    unhealthy_threshold=2
                )
            )],
            access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromFilePath")
    @builtins.classmethod
    def from_file_path(
        cls,
        file_path: builtins.str,
        logging_format: typing.Optional["LoggingFormat"] = None,
    ) -> "AccessLog":
        '''Path to a file to write access logs to.

        :param file_path: -
        :param logging_format: -

        :default: - no file based access logging
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__323d756842301c1e5a6b616e4869057675cd84476bc323393fa15e69535746e5)
            check_type(argname="argument file_path", value=file_path, expected_type=type_hints["file_path"])
            check_type(argname="argument logging_format", value=logging_format, expected_type=type_hints["logging_format"])
        return typing.cast("AccessLog", jsii.sinvoke(cls, "fromFilePath", [file_path, logging_format]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, scope: _constructs_77d1e7e8.Construct) -> "AccessLogConfig":
        '''Called when the AccessLog type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -
        '''
        ...


class _AccessLogProxy(AccessLog):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _constructs_77d1e7e8.Construct) -> "AccessLogConfig":
        '''Called when the AccessLog type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ecd62e6325a2016e6d2b667733a760354124c703f2717aee1e122ef0bf49941e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("AccessLogConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AccessLog).__jsii_proxy_class__ = lambda : _AccessLogProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.AccessLogConfig",
    jsii_struct_bases=[],
    name_mapping={
        "virtual_gateway_access_log": "virtualGatewayAccessLog",
        "virtual_node_access_log": "virtualNodeAccessLog",
    },
)
class AccessLogConfig:
    def __init__(
        self,
        *,
        virtual_gateway_access_log: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayAccessLogProperty", typing.Dict[builtins.str, typing.Any]]] = None,
        virtual_node_access_log: typing.Optional[typing.Union["CfnVirtualNode.AccessLogProperty", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''All Properties for Envoy Access logs for mesh endpoints.

        :param virtual_gateway_access_log: VirtualGateway CFN configuration for Access Logging. Default: - no access logging
        :param virtual_node_access_log: VirtualNode CFN configuration for Access Logging. Default: - no access logging

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            access_log_config = appmesh.AccessLogConfig(
                virtual_gateway_access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                        path="path",
            
                        # the properties below are optional
                        format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                            json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                key="key",
                                value="value"
                            )],
                            text="text"
                        )
                    )
                ),
                virtual_node_access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                    file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                        path="path",
            
                        # the properties below are optional
                        format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                            json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                key="key",
                                value="value"
                            )],
                            text="text"
                        )
                    )
                )
            )
        '''
        if isinstance(virtual_gateway_access_log, dict):
            virtual_gateway_access_log = CfnVirtualGateway.VirtualGatewayAccessLogProperty(**virtual_gateway_access_log)
        if isinstance(virtual_node_access_log, dict):
            virtual_node_access_log = CfnVirtualNode.AccessLogProperty(**virtual_node_access_log)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__24a2aa4dadea1ade090f0bf78eb212cd581c4e6d1890fa1a7ba517c4962e5148)
            check_type(argname="argument virtual_gateway_access_log", value=virtual_gateway_access_log, expected_type=type_hints["virtual_gateway_access_log"])
            check_type(argname="argument virtual_node_access_log", value=virtual_node_access_log, expected_type=type_hints["virtual_node_access_log"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if virtual_gateway_access_log is not None:
            self._values["virtual_gateway_access_log"] = virtual_gateway_access_log
        if virtual_node_access_log is not None:
            self._values["virtual_node_access_log"] = virtual_node_access_log

    @builtins.property
    def virtual_gateway_access_log(
        self,
    ) -> typing.Optional["CfnVirtualGateway.VirtualGatewayAccessLogProperty"]:
        '''VirtualGateway CFN configuration for Access Logging.

        :default: - no access logging
        '''
        result = self._values.get("virtual_gateway_access_log")
        return typing.cast(typing.Optional["CfnVirtualGateway.VirtualGatewayAccessLogProperty"], result)

    @builtins.property
    def virtual_node_access_log(
        self,
    ) -> typing.Optional["CfnVirtualNode.AccessLogProperty"]:
        '''VirtualNode CFN configuration for Access Logging.

        :default: - no access logging
        '''
        result = self._values.get("virtual_node_access_log")
        return typing.cast(typing.Optional["CfnVirtualNode.AccessLogProperty"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AccessLogConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Backend(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_appmesh.Backend",
):
    '''Contains static factory methods to create backends.

    :exampleMetadata: infused

    Example::

        # mesh: appmesh.Mesh
        # router: appmesh.VirtualRouter
        # service: cloudmap.Service
        
        
        node = appmesh.VirtualNode(self, "node",
            mesh=mesh,
            service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
            listeners=[appmesh.VirtualNodeListener.http(
                port=8080,
                health_check=appmesh.HealthCheck.http(
                    healthy_threshold=3,
                    interval=Duration.seconds(5),
                    path="/ping",
                    timeout=Duration.seconds(2),
                    unhealthy_threshold=2
                ),
                timeout=appmesh.HttpTimeout(
                    idle=Duration.seconds(5)
                )
            )],
            access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
        )
        
        virtual_service = appmesh.VirtualService(self, "service-1",
            virtual_service_provider=appmesh.VirtualServiceProvider.virtual_router(router),
            virtual_service_name="service1.domain.local"
        )
        
        node.add_backend(appmesh.Backend.virtual_service(virtual_service))
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="virtualService")
    @builtins.classmethod
    def virtual_service(
        cls,
        virtual_service: "IVirtualService",
        *,
        tls_client_policy: typing.Optional[typing.Union["TlsClientPolicy", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "Backend":
        '''Construct a Virtual Service backend.

        :param virtual_service: -
        :param tls_client_policy: TLS properties for Client policy for the backend. Default: - none
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f6e2987126642f25998eedac77fd7d54bd4af8bbe3373042b19ec980f83caf17)
            check_type(argname="argument virtual_service", value=virtual_service, expected_type=type_hints["virtual_service"])
        props = VirtualServiceBackendOptions(tls_client_policy=tls_client_policy)

        return typing.cast("Backend", jsii.sinvoke(cls, "virtualService", [virtual_service, props]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, _scope: _constructs_77d1e7e8.Construct) -> "BackendConfig":
        '''Return backend config.

        :param _scope: -
        '''
        ...


class _BackendProxy(Backend):
    @jsii.member(jsii_name="bind")
    def bind(self, _scope: _constructs_77d1e7e8.Construct) -> "BackendConfig":
        '''Return backend config.

        :param _scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8cc9a9aee4118102c458d301ad03d602e5b39f705d9a39d5fe1c1b7c0ecc036e)
            check_type(argname="argument _scope", value=_scope, expected_type=type_hints["_scope"])
        return typing.cast("BackendConfig", jsii.invoke(self, "bind", [_scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Backend).__jsii_proxy_class__ = lambda : _BackendProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.BackendConfig",
    jsii_struct_bases=[],
    name_mapping={"virtual_service_backend": "virtualServiceBackend"},
)
class BackendConfig:
    def __init__(
        self,
        *,
        virtual_service_backend: typing.Union["CfnVirtualNode.BackendProperty", typing.Dict[builtins.str, typing.Any]],
    ) -> None:
        '''Properties for a backend.

        :param virtual_service_backend: Config for a Virtual Service backend.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            backend_config = appmesh.BackendConfig(
                virtual_service_backend=appmesh.CfnVirtualNode.BackendProperty(
                    virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                        virtual_service_name="virtualServiceName",
            
                        # the properties below are optional
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
            
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
            
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    )
                )
            )
        '''
        if isinstance(virtual_service_backend, dict):
            virtual_service_backend = CfnVirtualNode.BackendProperty(**virtual_service_backend)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0532902f112b6ddb88288da0cda4ae2518e4206bf27153e1fa7362fa01c3a1ee)
            check_type(argname="argument virtual_service_backend", value=virtual_service_backend, expected_type=type_hints["virtual_service_backend"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "virtual_service_backend": virtual_service_backend,
        }

    @builtins.property
    def virtual_service_backend(self) -> "CfnVirtualNode.BackendProperty":
        '''Config for a Virtual Service backend.'''
        result = self._values.get("virtual_service_backend")
        assert result is not None, "Required property 'virtual_service_backend' is missing"
        return typing.cast("CfnVirtualNode.BackendProperty", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BackendConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.BackendDefaults",
    jsii_struct_bases=[],
    name_mapping={"tls_client_policy": "tlsClientPolicy"},
)
class BackendDefaults:
    def __init__(
        self,
        *,
        tls_client_policy: typing.Optional[typing.Union["TlsClientPolicy", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Represents the properties needed to define backend defaults.

        :param tls_client_policy: TLS properties for Client policy for backend defaults. Default: - none

        :exampleMetadata: infused

        Example::

            # mesh: appmesh.Mesh
            # service: cloudmap.Service
            
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
                listeners=[appmesh.VirtualNodeListener.http(
                    port=8080,
                    health_check=appmesh.HealthCheck.http(
                        healthy_threshold=3,
                        interval=Duration.seconds(5),
                        path="/ping",
                        timeout=Duration.seconds(2),
                        unhealthy_threshold=2
                    ),
                    timeout=appmesh.HttpTimeout(
                        idle=Duration.seconds(5)
                    )
                )],
                backend_defaults=appmesh.BackendDefaults(
                    tls_client_policy=appmesh.TlsClientPolicy(
                        validation=appmesh.TlsValidation(
                            trust=appmesh.TlsValidationTrust.file("/keys/local_cert_chain.pem")
                        )
                    )
                ),
                access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
            )
            
            cdk.Tags.of(node).add("Environment", "Dev")
        '''
        if isinstance(tls_client_policy, dict):
            tls_client_policy = TlsClientPolicy(**tls_client_policy)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__beddee5c3b7d43a79d9f985a1e3df164109f27465077be02cca5511f286cc179)
            check_type(argname="argument tls_client_policy", value=tls_client_policy, expected_type=type_hints["tls_client_policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if tls_client_policy is not None:
            self._values["tls_client_policy"] = tls_client_policy

    @builtins.property
    def tls_client_policy(self) -> typing.Optional["TlsClientPolicy"]:
        '''TLS properties for Client policy for backend defaults.

        :default: - none
        '''
        result = self._values.get("tls_client_policy")
        return typing.cast(typing.Optional["TlsClientPolicy"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BackendDefaults(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnGatewayRoute(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute",
):
    '''Creates a gateway route.

    A gateway route is attached to a virtual gateway and routes traffic to an existing virtual service. If a route matches a request, it can distribute traffic to a target virtual service.

    For more information about gateway routes, see `Gateway routes <https://docs.aws.amazon.com/app-mesh/latest/userguide/gateway-routes.html>`_ .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html
    :cloudformationResource: AWS::AppMesh::GatewayRoute
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_appmesh as appmesh
        
        cfn_gateway_route = appmesh.CfnGatewayRoute(self, "MyCfnGatewayRoute",
            mesh_name="meshName",
            spec=appmesh.CfnGatewayRoute.GatewayRouteSpecProperty(
                grpc_route=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
        
                            # the properties below are optional
                            port=123
                        ),
        
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        port=123,
                        service_name="serviceName"
                    )
                ),
                http2_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
        
                            # the properties below are optional
                            port=123
                        ),
        
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                http_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
        
                            # the properties below are optional
                            port=123
                        ),
        
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                priority=123
            ),
            virtual_gateway_name="virtualGatewayName",
        
            # the properties below are optional
            gateway_route_name="gatewayRouteName",
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GatewayRouteSpecProperty", typing.Dict[builtins.str, typing.Any]]],
        virtual_gateway_name: builtins.str,
        gateway_route_name: typing.Optional[builtins.str] = None,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param mesh_name: The name of the service mesh that the resource resides in.
        :param spec: The specifications of the gateway route.
        :param virtual_gateway_name: The virtual gateway that the gateway route is associated with.
        :param gateway_route_name: The name of the gateway route.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the gateway route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a9ed5e29900670674d1bcefbf0be36d5889c7807e98a89b2e73166538834d49a)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnGatewayRouteProps(
            mesh_name=mesh_name,
            spec=spec,
            virtual_gateway_name=virtual_gateway_name,
            gateway_route_name=gateway_route_name,
            mesh_owner=mesh_owner,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9fee24be2cb2504c9939582c0b16e21c8060816debe0c53ef1bab9397e0bf107)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2309e485df1cfa27e66342203e6cff93122b57b596e29248e5e0d4d4508c395f)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the gateway route.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrGatewayRouteName")
    def attr_gateway_route_name(self) -> builtins.str:
        '''The name of the gateway route.

        :cloudformationAttribute: GatewayRouteName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrGatewayRouteName"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the gateway route resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the gateway route.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualGatewayName")
    def attr_virtual_gateway_name(self) -> builtins.str:
        '''The name of the virtual gateway that the gateway route is associated with.

        :cloudformationAttribute: VirtualGatewayName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualGatewayName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the resource resides in.'''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__128ccc0049923e37dec482e976fa4dc66f246158c3ce39b446939f8254ce4e2f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteSpecProperty"]:
        '''The specifications of the gateway route.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteSpecProperty"], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteSpecProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ecd94785d11414db5b7dcf386baf559cf717a94fb135c1cfaa9a0c840828d82d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="virtualGatewayName")
    def virtual_gateway_name(self) -> builtins.str:
        '''The virtual gateway that the gateway route is associated with.'''
        return typing.cast(builtins.str, jsii.get(self, "virtualGatewayName"))

    @virtual_gateway_name.setter
    def virtual_gateway_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__20041147c2b0866bf142cb5d696bac03015e8f20ec4522350f1c1e77680f8013)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualGatewayName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="gatewayRouteName")
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''The name of the gateway route.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "gatewayRouteName"))

    @gateway_route_name.setter
    def gateway_route_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a7ea5ce436a0198baa56f9858cf4302af4af2c76e6cd0130ab31280e0341ce9b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "gatewayRouteName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cdd41f6caff6cb297cd7466e7e2afc842327a7389744cc27fcc2b1868ed865ab)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the gateway route to assist with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__440ced421bc9c4c53f7541195b1044c884ee26ef796da1dc28d65cfb6bc8953f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact", "suffix": "suffix"},
    )
    class GatewayRouteHostnameMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the gateway route host name to match.

            :param exact: The exact host name to match on.
            :param suffix: The specified ending characters of the host name to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                gateway_route_hostname_match_property = appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                    exact="exact",
                    suffix="suffix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4f860f4218e6dfece4cd3c55b8f66bf114241d9f918d21e26a6f7b03035f1188)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact host name to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamematch.html#cfn-appmesh-gatewayroute-gatewayroutehostnamematch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The specified ending characters of the host name to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamematch.html#cfn-appmesh-gatewayroute-gatewayroutehostnamematch-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteHostnameMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"default_target_hostname": "defaultTargetHostname"},
    )
    class GatewayRouteHostnameRewriteProperty:
        def __init__(
            self,
            *,
            default_target_hostname: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the gateway route host name to rewrite.

            :param default_target_hostname: The default target host name to write to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamerewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                gateway_route_hostname_rewrite_property = appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                    default_target_hostname="defaultTargetHostname"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a9b36503b4d06e5f4e41025387d2b36ed3153e9b11f692ea9ccceec7cc89f615)
                check_type(argname="argument default_target_hostname", value=default_target_hostname, expected_type=type_hints["default_target_hostname"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if default_target_hostname is not None:
                self._values["default_target_hostname"] = default_target_hostname

        @builtins.property
        def default_target_hostname(self) -> typing.Optional[builtins.str]:
            '''The default target host name to write to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamerewrite.html#cfn-appmesh-gatewayroute-gatewayroutehostnamerewrite-defaulttargethostname
            '''
            result = self._values.get("default_target_hostname")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteHostnameRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class GatewayRouteMetadataMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the method header to be matched.

            :param exact: The exact method header to be matched on.
            :param prefix: The specified beginning characters of the method header to be matched on.
            :param range: An object that represents the range of values to match on.
            :param regex: The regex used to match the method header.
            :param suffix: The specified ending characters of the method header to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                gateway_route_metadata_match_property = appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c1db1b20efb51e71c0ca4e4ea39f433438a91251f4d93a11b278ab389ce57d62)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument range", value=range, expected_type=type_hints["range"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
                check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact method header to be matched on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The specified beginning characters of the method header to be matched on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteRangeMatchProperty"]]:
            '''An object that represents the range of values to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteRangeMatchProperty"]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The regex used to match the method header.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The specified ending characters of the method header to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteMetadataMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"end": "end", "start": "start"},
    )
    class GatewayRouteRangeMatchProperty:
        def __init__(self, *, end: jsii.Number, start: jsii.Number) -> None:
            '''An object that represents the range of values to match on.

            The first character of the range is included in the range, though the last character is not. For example, if the range specified were 1-100, only values 1-99 would be matched.

            :param end: The end of the range.
            :param start: The start of the range.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayrouterangematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                gateway_route_range_match_property = appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                    end=123,
                    start=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f32f41b52becb7de46fa6f866d81bf10950603fbc5bc952f8aaff31c4081c154)
                check_type(argname="argument end", value=end, expected_type=type_hints["end"])
                check_type(argname="argument start", value=start, expected_type=type_hints["start"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "end": end,
                "start": start,
            }

        @builtins.property
        def end(self) -> jsii.Number:
            '''The end of the range.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayrouterangematch.html#cfn-appmesh-gatewayroute-gatewayrouterangematch-end
            '''
            result = self._values.get("end")
            assert result is not None, "Required property 'end' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def start(self) -> jsii.Number:
            '''The start of the range.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayrouterangematch.html#cfn-appmesh-gatewayroute-gatewayrouterangematch-start
            '''
            result = self._values.get("start")
            assert result is not None, "Required property 'start' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteRangeMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GatewayRouteSpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "grpc_route": "grpcRoute",
            "http2_route": "http2Route",
            "http_route": "httpRoute",
            "priority": "priority",
        },
    )
    class GatewayRouteSpecProperty:
        def __init__(
            self,
            *,
            grpc_route: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GrpcGatewayRouteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            http2_route: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            http_route: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            priority: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a gateway route specification.

            Specify one gateway route type.

            :param grpc_route: An object that represents the specification of a gRPC gateway route.
            :param http2_route: An object that represents the specification of an HTTP/2 gateway route.
            :param http_route: An object that represents the specification of an HTTP gateway route.
            :param priority: The ordering of the gateway routes spec.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                gateway_route_spec_property = appmesh.CfnGatewayRoute.GatewayRouteSpecProperty(
                    grpc_route=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
                
                                # the properties below are optional
                                port=123
                            ),
                
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            port=123,
                            service_name="serviceName"
                        )
                    ),
                    http2_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
                
                                # the properties below are optional
                                port=123
                            ),
                
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    http_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
                
                                # the properties below are optional
                                port=123
                            ),
                
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    priority=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c6724cbce8e85e22983b78703727e7b12e7b75a0a121d1ef3a8dd2f3201b8e5b)
                check_type(argname="argument grpc_route", value=grpc_route, expected_type=type_hints["grpc_route"])
                check_type(argname="argument http2_route", value=http2_route, expected_type=type_hints["http2_route"])
                check_type(argname="argument http_route", value=http_route, expected_type=type_hints["http_route"])
                check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if grpc_route is not None:
                self._values["grpc_route"] = grpc_route
            if http2_route is not None:
                self._values["http2_route"] = http2_route
            if http_route is not None:
                self._values["http_route"] = http_route
            if priority is not None:
                self._values["priority"] = priority

        @builtins.property
        def grpc_route(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GrpcGatewayRouteProperty"]]:
            '''An object that represents the specification of a gRPC gateway route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-grpcroute
            '''
            result = self._values.get("grpc_route")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GrpcGatewayRouteProperty"]], result)

        @builtins.property
        def http2_route(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteProperty"]]:
            '''An object that represents the specification of an HTTP/2 gateway route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-http2route
            '''
            result = self._values.get("http2_route")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteProperty"]], result)

        @builtins.property
        def http_route(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteProperty"]]:
            '''An object that represents the specification of an HTTP gateway route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-httproute
            '''
            result = self._values.get("http_route")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteProperty"]], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''The ordering of the gateway routes spec.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GatewayRouteTargetProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_service": "virtualService", "port": "port"},
    )
    class GatewayRouteTargetProperty:
        def __init__(
            self,
            *,
            virtual_service: typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GatewayRouteVirtualServiceProperty", typing.Dict[builtins.str, typing.Any]]],
            port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a gateway route target.

            :param virtual_service: An object that represents a virtual service gateway route target.
            :param port: The port number of the gateway route target.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutetarget.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                gateway_route_target_property = appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                    virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                        virtual_service_name="virtualServiceName"
                    ),
                
                    # the properties below are optional
                    port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__80e01a98f9e4165ab4b3348d33baf5e002bb8048c80933e0f8e89fd3655452e4)
                check_type(argname="argument virtual_service", value=virtual_service, expected_type=type_hints["virtual_service"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_service": virtual_service,
            }
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def virtual_service(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteVirtualServiceProperty"]:
            '''An object that represents a virtual service gateway route target.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutetarget.html#cfn-appmesh-gatewayroute-gatewayroutetarget-virtualservice
            '''
            result = self._values.get("virtual_service")
            assert result is not None, "Required property 'virtual_service' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteVirtualServiceProperty"], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port number of the gateway route target.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutetarget.html#cfn-appmesh-gatewayroute-gatewayroutetarget-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteTargetProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_service_name": "virtualServiceName"},
    )
    class GatewayRouteVirtualServiceProperty:
        def __init__(self, *, virtual_service_name: builtins.str) -> None:
            '''An object that represents the virtual service that traffic is routed to.

            :param virtual_service_name: The name of the virtual service that traffic is routed to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutevirtualservice.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                gateway_route_virtual_service_property = appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                    virtual_service_name="virtualServiceName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__48a8d0858c756d0afd934809065604876f2105a4c0c80ac92ba81dd761bb663c)
                check_type(argname="argument virtual_service_name", value=virtual_service_name, expected_type=type_hints["virtual_service_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_service_name": virtual_service_name,
            }

        @builtins.property
        def virtual_service_name(self) -> builtins.str:
            '''The name of the virtual service that traffic is routed to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutevirtualservice.html#cfn-appmesh-gatewayroute-gatewayroutevirtualservice-virtualservicename
            '''
            result = self._values.get("virtual_service_name")
            assert result is not None, "Required property 'virtual_service_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteVirtualServiceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"target": "target", "rewrite": "rewrite"},
    )
    class GrpcGatewayRouteActionProperty:
        def __init__(
            self,
            *,
            target: typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", typing.Dict[builtins.str, typing.Any]]],
            rewrite: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GrpcGatewayRouteRewriteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param target: An object that represents the target that traffic is routed to when a request matches the gateway route.
            :param rewrite: The gateway route action to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_gateway_route_action_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                    target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                        virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                            virtual_service_name="virtualServiceName"
                        ),
                
                        # the properties below are optional
                        port=123
                    ),
                
                    # the properties below are optional
                    rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                            default_target_hostname="defaultTargetHostname"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1b7e3532f9a99842e7e93df1c52fd82858eba7fcdff7fd22b3bf5bb5eead52b1)
                check_type(argname="argument target", value=target, expected_type=type_hints["target"])
                check_type(argname="argument rewrite", value=rewrite, expected_type=type_hints["rewrite"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "target": target,
            }
            if rewrite is not None:
                self._values["rewrite"] = rewrite

        @builtins.property
        def target(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteTargetProperty"]:
            '''An object that represents the target that traffic is routed to when a request matches the gateway route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouteaction.html#cfn-appmesh-gatewayroute-grpcgatewayrouteaction-target
            '''
            result = self._values.get("target")
            assert result is not None, "Required property 'target' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteTargetProperty"], result)

        @builtins.property
        def rewrite(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GrpcGatewayRouteRewriteProperty"]]:
            '''The gateway route action to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouteaction.html#cfn-appmesh-gatewayroute-grpcgatewayrouteaction-rewrite
            '''
            result = self._values.get("rewrite")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GrpcGatewayRouteRewriteProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "hostname": "hostname",
            "metadata": "metadata",
            "port": "port",
            "service_name": "serviceName",
        },
    )
    class GrpcGatewayRouteMatchProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            metadata: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GrpcGatewayRouteMetadataProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            port: typing.Optional[jsii.Number] = None,
            service_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the criteria for determining a request match.

            :param hostname: The gateway route host name to be matched on.
            :param metadata: The gateway route metadata to be matched on.
            :param port: The gateway route port to be matched on.
            :param service_name: The fully qualified domain name for the service to match from the request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_gateway_route_match_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                        exact="exact",
                        suffix="suffix"
                    ),
                    metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    port=123,
                    service_name="serviceName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c237c50dee0b454cad755a56707658a900411a844da2dbdcb2ff025151bffc72)
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
                check_type(argname="argument metadata", value=metadata, expected_type=type_hints["metadata"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument service_name", value=service_name, expected_type=type_hints["service_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname
            if metadata is not None:
                self._values["metadata"] = metadata
            if port is not None:
                self._values["port"] = port
            if service_name is not None:
                self._values["service_name"] = service_name

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteHostnameMatchProperty"]]:
            '''The gateway route host name to be matched on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html#cfn-appmesh-gatewayroute-grpcgatewayroutematch-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteHostnameMatchProperty"]], result)

        @builtins.property
        def metadata(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GrpcGatewayRouteMetadataProperty"]]]]:
            '''The gateway route metadata to be matched on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html#cfn-appmesh-gatewayroute-grpcgatewayroutematch-metadata
            '''
            result = self._values.get("metadata")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GrpcGatewayRouteMetadataProperty"]]]], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The gateway route port to be matched on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html#cfn-appmesh-gatewayroute-grpcgatewayroutematch-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def service_name(self) -> typing.Optional[builtins.str]:
            '''The fully qualified domain name for the service to match from the request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html#cfn-appmesh-gatewayroute-grpcgatewayroutematch-servicename
            '''
            result = self._values.get("service_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class GrpcGatewayRouteMetadataProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            match: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GatewayRouteMetadataMatchProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object representing the metadata of the gateway route.

            :param name: A name for the gateway route metadata.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: The criteria for determining a metadata match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_gateway_route_metadata_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5a88ca18b5f6353327a2e4ad5bfde4666b9f324ca9fb26fdbdd06cb92d9176e4)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument invert", value=invert, expected_type=type_hints["invert"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the gateway route metadata.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html#cfn-appmesh-gatewayroute-grpcgatewayroutemetadata-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html#cfn-appmesh-gatewayroute-grpcgatewayroutemetadata-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteMetadataMatchProperty"]]:
            '''The criteria for determining a metadata match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html#cfn-appmesh-gatewayroute-grpcgatewayroutemetadata-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteMetadataMatchProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteMetadataProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty",
        jsii_struct_bases=[],
        name_mapping={"action": "action", "match": "match"},
    )
    class GrpcGatewayRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GrpcGatewayRouteActionProperty", typing.Dict[builtins.str, typing.Any]]],
            match: typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GrpcGatewayRouteMatchProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''An object that represents a gRPC gateway route.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_gateway_route_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
                
                            # the properties below are optional
                            port=123
                        ),
                
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        port=123,
                        service_name="serviceName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a96f24bc8b6f71fc27e23abcd532311b29b763dd4510266b186ffabdcd4a3b33)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
                "match": match,
            }

        @builtins.property
        def action(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GrpcGatewayRouteActionProperty"]:
            '''An object that represents the action to take if a match is determined.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroute.html#cfn-appmesh-gatewayroute-grpcgatewayroute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GrpcGatewayRouteActionProperty"], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GrpcGatewayRouteMatchProperty"]:
            '''An object that represents the criteria for determining a request match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroute.html#cfn-appmesh-gatewayroute-grpcgatewayroute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GrpcGatewayRouteMatchProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"hostname": "hostname"},
    )
    class GrpcGatewayRouteRewriteProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the gateway route to rewrite.

            :param hostname: The host name of the gateway route to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouterewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_gateway_route_rewrite_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                        default_target_hostname="defaultTargetHostname"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8e6de0058d089b051fd49b059f2d560b6b2143c18269f61b3dd9c385e3a64baa)
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteHostnameRewriteProperty"]]:
            '''The host name of the gateway route to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouterewrite.html#cfn-appmesh-gatewayroute-grpcgatewayrouterewrite-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteHostnameRewriteProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"target": "target", "rewrite": "rewrite"},
    )
    class HttpGatewayRouteActionProperty:
        def __init__(
            self,
            *,
            target: typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", typing.Dict[builtins.str, typing.Any]]],
            rewrite: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.HttpGatewayRouteRewriteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param target: An object that represents the target that traffic is routed to when a request matches the gateway route.
            :param rewrite: The gateway route action to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_gateway_route_action_property = appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                    target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                        virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                            virtual_service_name="virtualServiceName"
                        ),
                
                        # the properties below are optional
                        port=123
                    ),
                
                    # the properties below are optional
                    rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                            default_target_hostname="defaultTargetHostname"
                        ),
                        path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                            exact="exact"
                        ),
                        prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                            default_prefix="defaultPrefix",
                            value="value"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cc74d6b2644effbdc36eeddbacd225f909e01cb555c231aaae9367f7e835625e)
                check_type(argname="argument target", value=target, expected_type=type_hints["target"])
                check_type(argname="argument rewrite", value=rewrite, expected_type=type_hints["rewrite"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "target": target,
            }
            if rewrite is not None:
                self._values["rewrite"] = rewrite

        @builtins.property
        def target(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteTargetProperty"]:
            '''An object that represents the target that traffic is routed to when a request matches the gateway route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteaction.html#cfn-appmesh-gatewayroute-httpgatewayrouteaction-target
            '''
            result = self._values.get("target")
            assert result is not None, "Required property 'target' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteTargetProperty"], result)

        @builtins.property
        def rewrite(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteRewriteProperty"]]:
            '''The gateway route action to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteaction.html#cfn-appmesh-gatewayroute-httpgatewayrouteaction-rewrite
            '''
            result = self._values.get("rewrite")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteRewriteProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class HttpGatewayRouteHeaderMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the method and value to match with the header value sent in a request.

            Specify one match method.

            :param exact: The value sent by the client must match the specified value exactly.
            :param prefix: The value sent by the client must begin with the specified characters.
            :param range: An object that represents the range of values to match on.
            :param regex: The value sent by the client must include the specified characters.
            :param suffix: The value sent by the client must end with the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_gateway_route_header_match_property = appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4773619061d62c99216f636ebf89a96ad15c19ede9312c60cf37aa6250f0e569)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument range", value=range, expected_type=type_hints["range"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
                check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must match the specified value exactly.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must begin with the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteRangeMatchProperty"]]:
            '''An object that represents the range of values to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteRangeMatchProperty"]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must include the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must end with the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteHeaderMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class HttpGatewayRouteHeaderProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            match: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the HTTP header in the gateway route.

            :param name: A name for the HTTP header in the gateway route that will be matched on.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: An object that represents the method and value to match with the header value sent in a request. Specify one match method.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_gateway_route_header_property = appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__53c7f93a0226cea5973f18301aefb1213e498dc9b4e8987aa44dbf7c4484bb1b)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument invert", value=invert, expected_type=type_hints["invert"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the HTTP header in the gateway route that will be matched on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html#cfn-appmesh-gatewayroute-httpgatewayrouteheader-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html#cfn-appmesh-gatewayroute-httpgatewayrouteheader-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty"]]:
            '''An object that represents the method and value to match with the header value sent in a request.

            Specify one match method.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html#cfn-appmesh-gatewayroute-httpgatewayrouteheader-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteHeaderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "headers": "headers",
            "hostname": "hostname",
            "method": "method",
            "path": "path",
            "port": "port",
            "prefix": "prefix",
            "query_parameters": "queryParameters",
        },
    )
    class HttpGatewayRouteMatchProperty:
        def __init__(
            self,
            *,
            headers: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            hostname: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            method: typing.Optional[builtins.str] = None,
            path: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.HttpPathMatchProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            port: typing.Optional[jsii.Number] = None,
            prefix: typing.Optional[builtins.str] = None,
            query_parameters: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.QueryParameterProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''An object that represents the criteria for determining a request match.

            :param headers: The client request headers to match on.
            :param hostname: The host name to match on.
            :param method: The method to match on.
            :param path: The path to match on.
            :param port: The port number to match on.
            :param prefix: Specifies the path to match requests with. This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .
            :param query_parameters: The query parameter to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_gateway_route_match_property = appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                    headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                        exact="exact",
                        suffix="suffix"
                    ),
                    method="method",
                    path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                        exact="exact",
                        regex="regex"
                    ),
                    port=123,
                    prefix="prefix",
                    query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                        name="name",
                
                        # the properties below are optional
                        match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                            exact="exact"
                        )
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cfde5af7b213b2714e97e05e66e4e376a86fac28a02cc4ff2dc8fb26f17a9636)
                check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
                check_type(argname="argument method", value=method, expected_type=type_hints["method"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if headers is not None:
                self._values["headers"] = headers
            if hostname is not None:
                self._values["hostname"] = hostname
            if method is not None:
                self._values["method"] = method
            if path is not None:
                self._values["path"] = path
            if port is not None:
                self._values["port"] = port
            if prefix is not None:
                self._values["prefix"] = prefix
            if query_parameters is not None:
                self._values["query_parameters"] = query_parameters

        @builtins.property
        def headers(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteHeaderProperty"]]]]:
            '''The client request headers to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-headers
            '''
            result = self._values.get("headers")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteHeaderProperty"]]]], result)

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteHostnameMatchProperty"]]:
            '''The host name to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteHostnameMatchProperty"]], result)

        @builtins.property
        def method(self) -> typing.Optional[builtins.str]:
            '''The method to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-method
            '''
            result = self._values.get("method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def path(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpPathMatchProperty"]]:
            '''The path to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpPathMatchProperty"]], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port number to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''Specifies the path to match requests with.

            This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def query_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.QueryParameterProperty"]]]]:
            '''The query parameter to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-queryparameters
            '''
            result = self._values.get("query_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.QueryParameterProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class HttpGatewayRoutePathRewriteProperty:
        def __init__(self, *, exact: typing.Optional[builtins.str] = None) -> None:
            '''An object that represents the path to rewrite.

            :param exact: The exact path to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutepathrewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_gateway_route_path_rewrite_property = appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                    exact="exact"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d38feba6e6105b4b17ababd5cad05eaeaa670773505cbe37ba5823f52007adc8)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact path to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutepathrewrite.html#cfn-appmesh-gatewayroute-httpgatewayroutepathrewrite-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRoutePathRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"default_prefix": "defaultPrefix", "value": "value"},
    )
    class HttpGatewayRoutePrefixRewriteProperty:
        def __init__(
            self,
            *,
            default_prefix: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the beginning characters of the route to rewrite.

            :param default_prefix: The default prefix used to replace the incoming route prefix when rewritten.
            :param value: The value used to replace the incoming route prefix when rewritten.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteprefixrewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_gateway_route_prefix_rewrite_property = appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                    default_prefix="defaultPrefix",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__44e9ba98baf2f0ed29ae87a697da0819d8983f7f19f7db72ac68703d7619d37e)
                check_type(argname="argument default_prefix", value=default_prefix, expected_type=type_hints["default_prefix"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if default_prefix is not None:
                self._values["default_prefix"] = default_prefix
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def default_prefix(self) -> typing.Optional[builtins.str]:
            '''The default prefix used to replace the incoming route prefix when rewritten.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteprefixrewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouteprefixrewrite-defaultprefix
            '''
            result = self._values.get("default_prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value used to replace the incoming route prefix when rewritten.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteprefixrewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouteprefixrewrite-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRoutePrefixRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteProperty",
        jsii_struct_bases=[],
        name_mapping={"action": "action", "match": "match"},
    )
    class HttpGatewayRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.HttpGatewayRouteActionProperty", typing.Dict[builtins.str, typing.Any]]],
            match: typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.HttpGatewayRouteMatchProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''An object that represents an HTTP gateway route.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_gateway_route_property = appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
                
                            # the properties below are optional
                            port=123
                        ),
                
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
                
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f81906ba842c1a65a7a9ca6d9773b88c5da5d8c4c02641f423f45f091e81f611)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
                "match": match,
            }

        @builtins.property
        def action(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteActionProperty"]:
            '''An object that represents the action to take if a match is determined.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroute.html#cfn-appmesh-gatewayroute-httpgatewayroute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteActionProperty"], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteMatchProperty"]:
            '''An object that represents the criteria for determining a request match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroute.html#cfn-appmesh-gatewayroute-httpgatewayroute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRouteMatchProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"hostname": "hostname", "path": "path", "prefix": "prefix"},
    )
    class HttpGatewayRouteRewriteProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            path: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            prefix: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object representing the gateway route to rewrite.

            :param hostname: The host name to rewrite.
            :param path: The path to rewrite.
            :param prefix: The specified beginning characters to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_gateway_route_rewrite_property = appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                        default_target_hostname="defaultTargetHostname"
                    ),
                    path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                        exact="exact"
                    ),
                    prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                        default_prefix="defaultPrefix",
                        value="value"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__764a0d1b08c7145a787e3c9ea6859a84bea7cf0d701afd99864b8d183d7ac955)
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname
            if path is not None:
                self._values["path"] = path
            if prefix is not None:
                self._values["prefix"] = prefix

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteHostnameRewriteProperty"]]:
            '''The host name to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouterewrite-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.GatewayRouteHostnameRewriteProperty"]], result)

        @builtins.property
        def path(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty"]]:
            '''The path to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouterewrite-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty"]], result)

        @builtins.property
        def prefix(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty"]]:
            '''The specified beginning characters to rewrite.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouterewrite-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.HttpPathMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact", "regex": "regex"},
    )
    class HttpPathMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            regex: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the path to match in the request.

            :param exact: The exact path to match on.
            :param regex: The regex used to match the path.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httppathmatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_path_match_property = appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                    exact="exact",
                    regex="regex"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8ded5ffb02fda38f0f75dd2648ca820ddbecb26719f2b6d60d9f568e4fc91d6a)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if regex is not None:
                self._values["regex"] = regex

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact path to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httppathmatch.html#cfn-appmesh-gatewayroute-httppathmatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The regex used to match the path.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httppathmatch.html#cfn-appmesh-gatewayroute-httppathmatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpPathMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class HttpQueryParameterMatchProperty:
        def __init__(self, *, exact: typing.Optional[builtins.str] = None) -> None:
            '''An object representing the query parameter to match.

            :param exact: The exact query parameter to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpqueryparametermatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_query_parameter_match_property = appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                    exact="exact"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__00bfa3d7eeab6ebe6987d0189c3ab0337284e7f3b64cfc164ff846231b512849)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact query parameter to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpqueryparametermatch.html#cfn-appmesh-gatewayroute-httpqueryparametermatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpQueryParameterMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRoute.QueryParameterProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "match": "match"},
    )
    class QueryParameterProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            match: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnGatewayRoute.HttpQueryParameterMatchProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the query parameter in the request.

            :param name: A name for the query parameter that will be matched on.
            :param match: The query parameter to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-queryparameter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                query_parameter_property = appmesh.CfnGatewayRoute.QueryParameterProperty(
                    name="name",
                
                    # the properties below are optional
                    match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                        exact="exact"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b539c51cde2ec66e437addbf528f3a517142c1d42ccaca16b3bd3eb23ed55f84)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the query parameter that will be matched on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-queryparameter.html#cfn-appmesh-gatewayroute-queryparameter-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpQueryParameterMatchProperty"]]:
            '''The query parameter to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-queryparameter.html#cfn-appmesh-gatewayroute-queryparameter-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnGatewayRoute.HttpQueryParameterMatchProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "QueryParameterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.CfnGatewayRouteProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "virtual_gateway_name": "virtualGatewayName",
        "gateway_route_name": "gatewayRouteName",
        "mesh_owner": "meshOwner",
        "tags": "tags",
    },
)
class CfnGatewayRouteProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union[CfnGatewayRoute.GatewayRouteSpecProperty, typing.Dict[builtins.str, typing.Any]]],
        virtual_gateway_name: builtins.str,
        gateway_route_name: typing.Optional[builtins.str] = None,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnGatewayRoute``.

        :param mesh_name: The name of the service mesh that the resource resides in.
        :param spec: The specifications of the gateway route.
        :param virtual_gateway_name: The virtual gateway that the gateway route is associated with.
        :param gateway_route_name: The name of the gateway route.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the gateway route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            cfn_gateway_route_props = appmesh.CfnGatewayRouteProps(
                mesh_name="meshName",
                spec=appmesh.CfnGatewayRoute.GatewayRouteSpecProperty(
                    grpc_route=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
            
                                # the properties below are optional
                                port=123
                            ),
            
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            port=123,
                            service_name="serviceName"
                        )
                    ),
                    http2_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
            
                                # the properties below are optional
                                port=123
                            ),
            
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    http_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                ),
            
                                # the properties below are optional
                                port=123
                            ),
            
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    priority=123
                ),
                virtual_gateway_name="virtualGatewayName",
            
                # the properties below are optional
                gateway_route_name="gatewayRouteName",
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3004eda0fe62c1c7f46ed5e88b9edc434dfe5fad51ed5ddb69a1c3d64f21babd)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument virtual_gateway_name", value=virtual_gateway_name, expected_type=type_hints["virtual_gateway_name"])
            check_type(argname="argument gateway_route_name", value=gateway_route_name, expected_type=type_hints["gateway_route_name"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
            "virtual_gateway_name": virtual_gateway_name,
        }
        if gateway_route_name is not None:
            self._values["gateway_route_name"] = gateway_route_name
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the resource resides in.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnGatewayRoute.GatewayRouteSpecProperty]:
        '''The specifications of the gateway route.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnGatewayRoute.GatewayRouteSpecProperty], result)

    @builtins.property
    def virtual_gateway_name(self) -> builtins.str:
        '''The virtual gateway that the gateway route is associated with.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-virtualgatewayname
        '''
        result = self._values.get("virtual_gateway_name")
        assert result is not None, "Required property 'virtual_gateway_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''The name of the gateway route.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-gatewayroutename
        '''
        result = self._values.get("gateway_route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the gateway route to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGatewayRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnMesh(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_appmesh.CfnMesh",
):
    '''Creates a service mesh.

    A service mesh is a logical boundary for network traffic between services that are represented by resources within the mesh. After you create your service mesh, you can create virtual services, virtual nodes, virtual routers, and routes to distribute traffic between the applications in your mesh.

    For more information about service meshes, see `Service meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html>`_ .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html
    :cloudformationResource: AWS::AppMesh::Mesh
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_appmesh as appmesh
        
        cfn_mesh = appmesh.CfnMesh(self, "MyCfnMesh",
            mesh_name="meshName",
            spec=appmesh.CfnMesh.MeshSpecProperty(
                egress_filter=appmesh.CfnMesh.EgressFilterProperty(
                    type="type"
                ),
                service_discovery=appmesh.CfnMesh.MeshServiceDiscoveryProperty(
                    ip_preference="ipPreference"
                )
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        mesh_name: typing.Optional[builtins.str] = None,
        spec: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnMesh.MeshSpecProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param mesh_name: The name to use for the service mesh.
        :param spec: The service mesh specification to apply.
        :param tags: Optional metadata that you can apply to the service mesh to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__65ffc4a086f699fbb67221c4a4ee330233d339976a46f138426ce0d2f31822d6)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnMeshProps(mesh_name=mesh_name, spec=spec, tags=tags)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__54146a9a2b5648aa17f1fed8a5f9e0701e495a4b6b222fb91ca44446267bfe86)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe688687dfeaeb91ae3d607667de89f03e62f06e81612062b79add0b7acedebf)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the mesh.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the mesh.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the service mesh.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e2e94aec5d09e58dc0f35b5dfdb0d18b35ac7f02e219dfceb5c4f4f7557b25b5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnMesh.MeshSpecProperty"]]:
        '''The service mesh specification to apply.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnMesh.MeshSpecProperty"]], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnMesh.MeshSpecProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__afc49cca6925a830455139b546c836096f2990d4491a963a6d558b8b5c8fa965)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the service mesh to assist with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__16f54ab69e4f9713eaf7ebcb9b1d182cb3f7826f0ef5fff8a2208d6a21829862)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnMesh.EgressFilterProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type"},
    )
    class EgressFilterProperty:
        def __init__(self, *, type: builtins.str) -> None:
            '''An object that represents the egress filter rules for a service mesh.

            :param type: The egress filter type. By default, the type is ``DROP_ALL`` , which allows egress only from virtual nodes to other defined resources in the service mesh (and any traffic to ``*.amazonaws.com`` for AWS API calls). You can set the egress filter type to ``ALLOW_ALL`` to allow egress to any endpoint inside or outside of the service mesh. .. epigraph:: If you specify any backends on a virtual node when using ``ALLOW_ALL`` , you must specifiy all egress for that virtual node as backends. Otherwise, ``ALLOW_ALL`` will no longer work for that virtual node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-egressfilter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                egress_filter_property = appmesh.CfnMesh.EgressFilterProperty(
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a4cbf02d118dd144c663e002b9872dd141130498abb4bff20bfbff872fc5900e)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }

        @builtins.property
        def type(self) -> builtins.str:
            '''The egress filter type.

            By default, the type is ``DROP_ALL`` , which allows egress only from virtual nodes to other defined resources in the service mesh (and any traffic to ``*.amazonaws.com`` for AWS API calls). You can set the egress filter type to ``ALLOW_ALL`` to allow egress to any endpoint inside or outside of the service mesh.
            .. epigraph::

               If you specify any backends on a virtual node when using ``ALLOW_ALL`` , you must specifiy all egress for that virtual node as backends. Otherwise, ``ALLOW_ALL`` will no longer work for that virtual node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-egressfilter.html#cfn-appmesh-mesh-egressfilter-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EgressFilterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnMesh.MeshServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={"ip_preference": "ipPreference"},
    )
    class MeshServiceDiscoveryProperty:
        def __init__(
            self,
            *,
            ip_preference: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the service discovery information for a service mesh.

            :param ip_preference: The IP version to use to control traffic within the mesh.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshservicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                mesh_service_discovery_property = appmesh.CfnMesh.MeshServiceDiscoveryProperty(
                    ip_preference="ipPreference"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d871a14d095c42450acf52b5808292615e8e66a19e6210bc98ac6ab265d4a5f0)
                check_type(argname="argument ip_preference", value=ip_preference, expected_type=type_hints["ip_preference"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if ip_preference is not None:
                self._values["ip_preference"] = ip_preference

        @builtins.property
        def ip_preference(self) -> typing.Optional[builtins.str]:
            '''The IP version to use to control traffic within the mesh.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshservicediscovery.html#cfn-appmesh-mesh-meshservicediscovery-ippreference
            '''
            result = self._values.get("ip_preference")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MeshServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnMesh.MeshSpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "egress_filter": "egressFilter",
            "service_discovery": "serviceDiscovery",
        },
    )
    class MeshSpecProperty:
        def __init__(
            self,
            *,
            egress_filter: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnMesh.EgressFilterProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            service_discovery: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnMesh.MeshServiceDiscoveryProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the specification of a service mesh.

            :param egress_filter: The egress filter rules for the service mesh.
            :param service_discovery: An object that represents the service discovery information for a service mesh.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshspec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                mesh_spec_property = appmesh.CfnMesh.MeshSpecProperty(
                    egress_filter=appmesh.CfnMesh.EgressFilterProperty(
                        type="type"
                    ),
                    service_discovery=appmesh.CfnMesh.MeshServiceDiscoveryProperty(
                        ip_preference="ipPreference"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6114130f1ea71520d87a81a58c84b5844c4aecc9229bcf5763d24f419425c2ed)
                check_type(argname="argument egress_filter", value=egress_filter, expected_type=type_hints["egress_filter"])
                check_type(argname="argument service_discovery", value=service_discovery, expected_type=type_hints["service_discovery"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if egress_filter is not None:
                self._values["egress_filter"] = egress_filter
            if service_discovery is not None:
                self._values["service_discovery"] = service_discovery

        @builtins.property
        def egress_filter(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnMesh.EgressFilterProperty"]]:
            '''The egress filter rules for the service mesh.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshspec.html#cfn-appmesh-mesh-meshspec-egressfilter
            '''
            result = self._values.get("egress_filter")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnMesh.EgressFilterProperty"]], result)

        @builtins.property
        def service_discovery(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnMesh.MeshServiceDiscoveryProperty"]]:
            '''An object that represents the service discovery information for a service mesh.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshspec.html#cfn-appmesh-mesh-meshspec-servicediscovery
            '''
            result = self._values.get("service_discovery")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnMesh.MeshServiceDiscoveryProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MeshSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.CfnMeshProps",
    jsii_struct_bases=[],
    name_mapping={"mesh_name": "meshName", "spec": "spec", "tags": "tags"},
)
class CfnMeshProps:
    def __init__(
        self,
        *,
        mesh_name: typing.Optional[builtins.str] = None,
        spec: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnMesh.MeshSpecProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnMesh``.

        :param mesh_name: The name to use for the service mesh.
        :param spec: The service mesh specification to apply.
        :param tags: Optional metadata that you can apply to the service mesh to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            cfn_mesh_props = appmesh.CfnMeshProps(
                mesh_name="meshName",
                spec=appmesh.CfnMesh.MeshSpecProperty(
                    egress_filter=appmesh.CfnMesh.EgressFilterProperty(
                        type="type"
                    ),
                    service_discovery=appmesh.CfnMesh.MeshServiceDiscoveryProperty(
                        ip_preference="ipPreference"
                    )
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__745301926d9e317e3e465d5dda5b06277fa15bd229178a3f93fa31ed9c28a39b)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if mesh_name is not None:
            self._values["mesh_name"] = mesh_name
        if spec is not None:
            self._values["spec"] = spec
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the service mesh.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-meshname
        '''
        result = self._values.get("mesh_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnMesh.MeshSpecProperty]]:
        '''The service mesh specification to apply.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-spec
        '''
        result = self._values.get("spec")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnMesh.MeshSpecProperty]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the service mesh to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnMeshProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnRoute(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute",
):
    '''Creates a route that is associated with a virtual router.

    You can route several different protocols and define a retry policy for a route. Traffic can be routed to one or more virtual nodes.

    For more information about routes, see `Routes <https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html>`_ .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html
    :cloudformationResource: AWS::AppMesh::Route
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_appmesh as appmesh
        
        cfn_route = appmesh.CfnRoute(self, "MyCfnRoute",
            mesh_name="meshName",
            spec=appmesh.CfnRoute.RouteSpecProperty(
                grpc_route=appmesh.CfnRoute.GrpcRouteProperty(
                    action=appmesh.CfnRoute.GrpcRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
        
                            # the properties below are optional
                            port=123
                        )]
                    ),
                    match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                        metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method_name="methodName",
                        port=123,
                        service_name="serviceName"
                    ),
        
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
        
                        # the properties below are optional
                        grpc_retry_events=["grpcRetryEvents"],
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                ),
                http2_route=appmesh.CfnRoute.HttpRouteProperty(
                    action=appmesh.CfnRoute.HttpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
        
                            # the properties below are optional
                            port=123
                        )]
                    ),
                    match=appmesh.CfnRoute.HttpRouteMatchProperty(
                        headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method="method",
                        path=appmesh.CfnRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )],
                        scheme="scheme"
                    ),
        
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
        
                        # the properties below are optional
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                ),
                http_route=appmesh.CfnRoute.HttpRouteProperty(
                    action=appmesh.CfnRoute.HttpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
        
                            # the properties below are optional
                            port=123
                        )]
                    ),
                    match=appmesh.CfnRoute.HttpRouteMatchProperty(
                        headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method="method",
                        path=appmesh.CfnRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )],
                        scheme="scheme"
                    ),
        
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
        
                        # the properties below are optional
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                ),
                priority=123,
                tcp_route=appmesh.CfnRoute.TcpRouteProperty(
                    action=appmesh.CfnRoute.TcpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
        
                            # the properties below are optional
                            port=123
                        )]
                    ),
        
                    # the properties below are optional
                    match=appmesh.CfnRoute.TcpRouteMatchProperty(
                        port=123
                    ),
                    timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            ),
            virtual_router_name="virtualRouterName",
        
            # the properties below are optional
            mesh_owner="meshOwner",
            route_name="routeName",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.RouteSpecProperty", typing.Dict[builtins.str, typing.Any]]],
        virtual_router_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        route_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param mesh_name: The name of the service mesh to create the route in.
        :param spec: The route specification to apply.
        :param virtual_router_name: The name of the virtual router in which to create the route. If the virtual router is in a shared mesh, then you must be the owner of the virtual router resource.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param route_name: The name to use for the route.
        :param tags: Optional metadata that you can apply to the route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9b7b10a22d83fc5b88ec91baa0139a581ca4a7b54848846580d22cd32e89d28e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnRouteProps(
            mesh_name=mesh_name,
            spec=spec,
            virtual_router_name=virtual_router_name,
            mesh_owner=mesh_owner,
            route_name=route_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5ac0b6bbc950e0d63339f26a908015a90e06e044e8cbfc8147b708cfc204d9ce)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cbc41aa0ea3a6e312042187c4db739d67673170ea60217187957e7fd51afe324)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the route.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the route resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrRouteName")
    def attr_route_name(self) -> builtins.str:
        '''The name of the route.

        :cloudformationAttribute: RouteName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRouteName"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the route.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualRouterName")
    def attr_virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router that the route is associated with.

        :cloudformationAttribute: VirtualRouterName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualRouterName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the route in.'''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5855464877759a6920bea7c176396ecba8ffc278723ba4fa2005b7808c7130d8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(self) -> typing.Union[_IResolvable_da3f097b, "CfnRoute.RouteSpecProperty"]:
        '''The route specification to apply.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnRoute.RouteSpecProperty"], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnRoute.RouteSpecProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1c183b00026b4ff39c5e60f6b1b9fa79d1adc1d9535899638934407176e54bf5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="virtualRouterName")
    def virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router in which to create the route.'''
        return typing.cast(builtins.str, jsii.get(self, "virtualRouterName"))

    @virtual_router_name.setter
    def virtual_router_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f82979ea57c6287fdc6d38fe4eb6308090e9e83c816d2486e05e0ef598204abf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualRouterName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b93461d87b528b1fe450ae161b4254aa1d1bec0dba1627c3dd7b2d0b72eea29)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="routeName")
    def route_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the route.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "routeName"))

    @route_name.setter
    def route_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9045f5e7be138379ef0f3f538d281ed7bf6061034337f8b45c0c605ee9e2feb3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "routeName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the route to assist with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a5df50b837c215e10cf3c9902ca654ddda9fb16c71399945b58961d5ad138ee8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.DurationProperty",
        jsii_struct_bases=[],
        name_mapping={"unit": "unit", "value": "value"},
    )
    class DurationProperty:
        def __init__(self, *, unit: builtins.str, value: jsii.Number) -> None:
            '''An object that represents a duration of time.

            :param unit: A unit of time.
            :param value: A number of time units.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-duration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                duration_property = appmesh.CfnRoute.DurationProperty(
                    unit="unit",
                    value=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8979f54f4d803fcfdd0f11d034327b8f9221fe639b3f075746b5e58e381e3e1e)
                check_type(argname="argument unit", value=unit, expected_type=type_hints["unit"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "unit": unit,
                "value": value,
            }

        @builtins.property
        def unit(self) -> builtins.str:
            '''A unit of time.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-duration.html#cfn-appmesh-route-duration-unit
            '''
            result = self._values.get("unit")
            assert result is not None, "Required property 'unit' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> jsii.Number:
            '''A number of time units.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-duration.html#cfn-appmesh-route-duration-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.GrpcRetryPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_retries": "maxRetries",
            "per_retry_timeout": "perRetryTimeout",
            "grpc_retry_events": "grpcRetryEvents",
            "http_retry_events": "httpRetryEvents",
            "tcp_retry_events": "tcpRetryEvents",
        },
    )
    class GrpcRetryPolicyProperty:
        def __init__(
            self,
            *,
            max_retries: jsii.Number,
            per_retry_timeout: typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]]],
            grpc_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
            http_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
            tcp_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents a retry policy.

            Specify at least one value for at least one of the types of ``RetryEvents`` , a value for ``maxRetries`` , and a value for ``perRetryTimeout`` . Both ``server-error`` and ``gateway-error`` under ``httpRetryEvents`` include the Envoy ``reset`` policy. For more information on the ``reset`` policy, see the `Envoy documentation <https://docs.aws.amazon.com/https://www.envoyproxy.io/docs/envoy/latest/configuration/http/http_filters/router_filter#x-envoy-retry-on>`_ .

            :param max_retries: The maximum number of retry attempts.
            :param per_retry_timeout: The timeout for each retry attempt.
            :param grpc_retry_events: Specify at least one of the valid values.
            :param http_retry_events: Specify at least one of the following values. - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511 - *gateway-error* – HTTP status codes 502, 503, and 504 - *client-error* – HTTP status code 409 - *stream-error* – Retry on refused stream
            :param tcp_retry_events: Specify a valid value. The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_retry_policy_property = appmesh.CfnRoute.GrpcRetryPolicyProperty(
                    max_retries=123,
                    per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                
                    # the properties below are optional
                    grpc_retry_events=["grpcRetryEvents"],
                    http_retry_events=["httpRetryEvents"],
                    tcp_retry_events=["tcpRetryEvents"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4b339357002fc1c229a6e43ed3a792959d631f545c4528974f3e5fc03031ae6a)
                check_type(argname="argument max_retries", value=max_retries, expected_type=type_hints["max_retries"])
                check_type(argname="argument per_retry_timeout", value=per_retry_timeout, expected_type=type_hints["per_retry_timeout"])
                check_type(argname="argument grpc_retry_events", value=grpc_retry_events, expected_type=type_hints["grpc_retry_events"])
                check_type(argname="argument http_retry_events", value=http_retry_events, expected_type=type_hints["http_retry_events"])
                check_type(argname="argument tcp_retry_events", value=tcp_retry_events, expected_type=type_hints["tcp_retry_events"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_retries": max_retries,
                "per_retry_timeout": per_retry_timeout,
            }
            if grpc_retry_events is not None:
                self._values["grpc_retry_events"] = grpc_retry_events
            if http_retry_events is not None:
                self._values["http_retry_events"] = http_retry_events
            if tcp_retry_events is not None:
                self._values["tcp_retry_events"] = tcp_retry_events

        @builtins.property
        def max_retries(self) -> jsii.Number:
            '''The maximum number of retry attempts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-maxretries
            '''
            result = self._values.get("max_retries")
            assert result is not None, "Required property 'max_retries' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def per_retry_timeout(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]:
            '''The timeout for each retry attempt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-perretrytimeout
            '''
            result = self._values.get("per_retry_timeout")
            assert result is not None, "Required property 'per_retry_timeout' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"], result)

        @builtins.property
        def grpc_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify at least one of the valid values.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-grpcretryevents
            '''
            result = self._values.get("grpc_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def http_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify at least one of the following values.

            - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511
            - *gateway-error* – HTTP status codes 502, 503, and 504
            - *client-error* – HTTP status code 409
            - *stream-error* – Retry on refused stream

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-httpretryevents
            '''
            result = self._values.get("http_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def tcp_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify a valid value.

            The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-tcpretryevents
            '''
            result = self._values.get("tcp_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRetryPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.GrpcRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"weighted_targets": "weightedTargets"},
    )
    class GrpcRouteActionProperty:
        def __init__(
            self,
            *,
            weighted_targets: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.WeightedTargetProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param weighted_targets: An object that represents the targets that traffic is routed to when a request matches the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcrouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_route_action_property = appmesh.CfnRoute.GrpcRouteActionProperty(
                    weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                        virtual_node="virtualNode",
                        weight=123,
                
                        # the properties below are optional
                        port=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e8673b195e241e78cd8f173b28b4a6a601b262af57b0e915da1290085976b392)
                check_type(argname="argument weighted_targets", value=weighted_targets, expected_type=type_hints["weighted_targets"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "weighted_targets": weighted_targets,
            }

        @builtins.property
        def weighted_targets(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.WeightedTargetProperty"]]]:
            '''An object that represents the targets that traffic is routed to when a request matches the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcrouteaction.html#cfn-appmesh-route-grpcrouteaction-weightedtargets
            '''
            result = self._values.get("weighted_targets")
            assert result is not None, "Required property 'weighted_targets' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.WeightedTargetProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.GrpcRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "metadata": "metadata",
            "method_name": "methodName",
            "port": "port",
            "service_name": "serviceName",
        },
    )
    class GrpcRouteMatchProperty:
        def __init__(
            self,
            *,
            metadata: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.GrpcRouteMetadataProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            method_name: typing.Optional[builtins.str] = None,
            port: typing.Optional[jsii.Number] = None,
            service_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the criteria for determining a request match.

            :param metadata: An object that represents the data to match from the request.
            :param method_name: The method name to match from the request. If you specify a name, you must also specify a ``serviceName`` .
            :param port: The port number to match on.
            :param service_name: The fully qualified domain name for the service to match from the request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_route_match_property = appmesh.CfnRoute.GrpcRouteMatchProperty(
                    metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnRoute.MatchRangeProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    method_name="methodName",
                    port=123,
                    service_name="serviceName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3da32b185227e7d471d4c1b4b700755959d4971c82a5093e1760941963486f69)
                check_type(argname="argument metadata", value=metadata, expected_type=type_hints["metadata"])
                check_type(argname="argument method_name", value=method_name, expected_type=type_hints["method_name"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument service_name", value=service_name, expected_type=type_hints["service_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if metadata is not None:
                self._values["metadata"] = metadata
            if method_name is not None:
                self._values["method_name"] = method_name
            if port is not None:
                self._values["port"] = port
            if service_name is not None:
                self._values["service_name"] = service_name

        @builtins.property
        def metadata(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRouteMetadataProperty"]]]]:
            '''An object that represents the data to match from the request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html#cfn-appmesh-route-grpcroutematch-metadata
            '''
            result = self._values.get("metadata")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRouteMetadataProperty"]]]], result)

        @builtins.property
        def method_name(self) -> typing.Optional[builtins.str]:
            '''The method name to match from the request.

            If you specify a name, you must also specify a ``serviceName`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html#cfn-appmesh-route-grpcroutematch-methodname
            '''
            result = self._values.get("method_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port number to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html#cfn-appmesh-route-grpcroutematch-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def service_name(self) -> typing.Optional[builtins.str]:
            '''The fully qualified domain name for the service to match from the request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html#cfn-appmesh-route-grpcroutematch-servicename
            '''
            result = self._values.get("service_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class GrpcRouteMetadataMatchMethodProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.MatchRangeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the match method.

            Specify one of the match values.

            :param exact: The value sent by the client must match the specified value exactly.
            :param prefix: The value sent by the client must begin with the specified characters.
            :param range: An object that represents the range of values to match on.
            :param regex: The value sent by the client must include the specified characters.
            :param suffix: The value sent by the client must end with the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_route_metadata_match_method_property = appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnRoute.MatchRangeProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__018b85df7988b9e4205a94cfda54dda4a3e222424c7869b8ddcf88f14317dcf2)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument range", value=range, expected_type=type_hints["range"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
                check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must match the specified value exactly.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must begin with the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.MatchRangeProperty"]]:
            '''An object that represents the range of values to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.MatchRangeProperty"]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must include the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must end with the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteMetadataMatchMethodProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.GrpcRouteMetadataProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class GrpcRouteMetadataProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            match: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.GrpcRouteMetadataMatchMethodProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the match metadata for the route.

            :param name: The name of the route.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: An object that represents the data to match from the request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_route_metadata_property = appmesh.CfnRoute.GrpcRouteMetadataProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnRoute.MatchRangeProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__226918cf4678c98fe225c53ca475a3742dc227eccc1f5c640aeb7c583a23e433)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument invert", value=invert, expected_type=type_hints["invert"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html#cfn-appmesh-route-grpcroutemetadata-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html#cfn-appmesh-route-grpcroutemetadata-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRouteMetadataMatchMethodProperty"]]:
            '''An object that represents the data to match from the request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html#cfn-appmesh-route-grpcroutemetadata-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRouteMetadataMatchMethodProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteMetadataProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.GrpcRouteProperty",
        jsii_struct_bases=[],
        name_mapping={
            "action": "action",
            "match": "match",
            "retry_policy": "retryPolicy",
            "timeout": "timeout",
        },
    )
    class GrpcRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.GrpcRouteActionProperty", typing.Dict[builtins.str, typing.Any]]],
            match: typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.GrpcRouteMatchProperty", typing.Dict[builtins.str, typing.Any]]],
            retry_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.GrpcRetryPolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            timeout: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.GrpcTimeoutProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a gRPC route type.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.
            :param retry_policy: An object that represents a retry policy.
            :param timeout: An object that represents types of timeouts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_route_property = appmesh.CfnRoute.GrpcRouteProperty(
                    action=appmesh.CfnRoute.GrpcRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
                
                            # the properties below are optional
                            port=123
                        )]
                    ),
                    match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                        metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method_name="methodName",
                        port=123,
                        service_name="serviceName"
                    ),
                
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                
                        # the properties below are optional
                        grpc_retry_events=["grpcRetryEvents"],
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c91cb3d53a155c4a2fcfb999b4dd462badd26a2065399443c6d8cf0f7d3685fd)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
                check_type(argname="argument retry_policy", value=retry_policy, expected_type=type_hints["retry_policy"])
                check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
                "match": match,
            }
            if retry_policy is not None:
                self._values["retry_policy"] = retry_policy
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def action(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRouteActionProperty"]:
            '''An object that represents the action to take if a match is determined.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRouteActionProperty"], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRouteMatchProperty"]:
            '''An object that represents the criteria for determining a request match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRouteMatchProperty"], result)

        @builtins.property
        def retry_policy(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRetryPolicyProperty"]]:
            '''An object that represents a retry policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-retrypolicy
            '''
            result = self._values.get("retry_policy")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRetryPolicyProperty"]], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcTimeoutProperty"]]:
            '''An object that represents types of timeouts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcTimeoutProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.GrpcTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class GrpcTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            per_request: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpctimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_timeout_property = appmesh.CfnRoute.GrpcTimeoutProperty(
                    idle=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7f42495c61627fde5944d4b172d6e1df892aa4be61f9598385556e7d338d66fc)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
                check_type(argname="argument per_request", value=per_request, expected_type=type_hints["per_request"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpctimeout.html#cfn-appmesh-route-grpctimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpctimeout.html#cfn-appmesh-route-grpctimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.HeaderMatchMethodProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class HeaderMatchMethodProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.MatchRangeProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the method and value to match with the header value sent in a request.

            Specify one match method.

            :param exact: The value sent by the client must match the specified value exactly.
            :param prefix: The value sent by the client must begin with the specified characters.
            :param range: An object that represents the range of values to match on.
            :param regex: The value sent by the client must include the specified characters.
            :param suffix: The value sent by the client must end with the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                header_match_method_property = appmesh.CfnRoute.HeaderMatchMethodProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnRoute.MatchRangeProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5ca675b5ef4970bfd0b937bb2a1da18625681d35f277ff967f9ddf855e7b2e3e)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument range", value=range, expected_type=type_hints["range"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
                check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must match the specified value exactly.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must begin with the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.MatchRangeProperty"]]:
            '''An object that represents the range of values to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.MatchRangeProperty"]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must include the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must end with the specified characters.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HeaderMatchMethodProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.HttpPathMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact", "regex": "regex"},
    )
    class HttpPathMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            regex: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the path to match in the request.

            :param exact: The exact path to match on.
            :param regex: The regex used to match the path.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httppathmatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_path_match_property = appmesh.CfnRoute.HttpPathMatchProperty(
                    exact="exact",
                    regex="regex"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__00e7d2e371078ecbca1ca834d84ac75226b0d2819f02216ec87a911b8c0d6664)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
                check_type(argname="argument regex", value=regex, expected_type=type_hints["regex"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if regex is not None:
                self._values["regex"] = regex

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact path to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httppathmatch.html#cfn-appmesh-route-httppathmatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The regex used to match the path.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httppathmatch.html#cfn-appmesh-route-httppathmatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpPathMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.HttpQueryParameterMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class HttpQueryParameterMatchProperty:
        def __init__(self, *, exact: typing.Optional[builtins.str] = None) -> None:
            '''An object representing the query parameter to match.

            :param exact: The exact query parameter to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpqueryparametermatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_query_parameter_match_property = appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                    exact="exact"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a71e175d0fa34203e6bb38fa01373bd8b2d886382b7f22b0259a58489aad11df)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact query parameter to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpqueryparametermatch.html#cfn-appmesh-route-httpqueryparametermatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpQueryParameterMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.HttpRetryPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_retries": "maxRetries",
            "per_retry_timeout": "perRetryTimeout",
            "http_retry_events": "httpRetryEvents",
            "tcp_retry_events": "tcpRetryEvents",
        },
    )
    class HttpRetryPolicyProperty:
        def __init__(
            self,
            *,
            max_retries: jsii.Number,
            per_retry_timeout: typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]]],
            http_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
            tcp_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents a retry policy.

            Specify at least one value for at least one of the types of ``RetryEvents`` , a value for ``maxRetries`` , and a value for ``perRetryTimeout`` . Both ``server-error`` and ``gateway-error`` under ``httpRetryEvents`` include the Envoy ``reset`` policy. For more information on the ``reset`` policy, see the `Envoy documentation <https://docs.aws.amazon.com/https://www.envoyproxy.io/docs/envoy/latest/configuration/http/http_filters/router_filter#x-envoy-retry-on>`_ .

            :param max_retries: The maximum number of retry attempts.
            :param per_retry_timeout: The timeout for each retry attempt.
            :param http_retry_events: Specify at least one of the following values. - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511 - *gateway-error* – HTTP status codes 502, 503, and 504 - *client-error* – HTTP status code 409 - *stream-error* – Retry on refused stream
            :param tcp_retry_events: Specify a valid value. The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_retry_policy_property = appmesh.CfnRoute.HttpRetryPolicyProperty(
                    max_retries=123,
                    per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                
                    # the properties below are optional
                    http_retry_events=["httpRetryEvents"],
                    tcp_retry_events=["tcpRetryEvents"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__949ce5d92b6a94727d13ca399e63e3b1539a7c5e5dcfc4483054116e29c1b36e)
                check_type(argname="argument max_retries", value=max_retries, expected_type=type_hints["max_retries"])
                check_type(argname="argument per_retry_timeout", value=per_retry_timeout, expected_type=type_hints["per_retry_timeout"])
                check_type(argname="argument http_retry_events", value=http_retry_events, expected_type=type_hints["http_retry_events"])
                check_type(argname="argument tcp_retry_events", value=tcp_retry_events, expected_type=type_hints["tcp_retry_events"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_retries": max_retries,
                "per_retry_timeout": per_retry_timeout,
            }
            if http_retry_events is not None:
                self._values["http_retry_events"] = http_retry_events
            if tcp_retry_events is not None:
                self._values["tcp_retry_events"] = tcp_retry_events

        @builtins.property
        def max_retries(self) -> jsii.Number:
            '''The maximum number of retry attempts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-maxretries
            '''
            result = self._values.get("max_retries")
            assert result is not None, "Required property 'max_retries' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def per_retry_timeout(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]:
            '''The timeout for each retry attempt.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-perretrytimeout
            '''
            result = self._values.get("per_retry_timeout")
            assert result is not None, "Required property 'per_retry_timeout' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"], result)

        @builtins.property
        def http_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify at least one of the following values.

            - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511
            - *gateway-error* – HTTP status codes 502, 503, and 504
            - *client-error* – HTTP status code 409
            - *stream-error* – Retry on refused stream

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-httpretryevents
            '''
            result = self._values.get("http_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def tcp_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify a valid value.

            The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-tcpretryevents
            '''
            result = self._values.get("tcp_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRetryPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.HttpRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"weighted_targets": "weightedTargets"},
    )
    class HttpRouteActionProperty:
        def __init__(
            self,
            *,
            weighted_targets: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.WeightedTargetProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param weighted_targets: An object that represents the targets that traffic is routed to when a request matches the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_route_action_property = appmesh.CfnRoute.HttpRouteActionProperty(
                    weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                        virtual_node="virtualNode",
                        weight=123,
                
                        # the properties below are optional
                        port=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f581cdb3fa6d8b9593efb14aa3c8da0374c054be5b94173520eb099feccd7776)
                check_type(argname="argument weighted_targets", value=weighted_targets, expected_type=type_hints["weighted_targets"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "weighted_targets": weighted_targets,
            }

        @builtins.property
        def weighted_targets(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.WeightedTargetProperty"]]]:
            '''An object that represents the targets that traffic is routed to when a request matches the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteaction.html#cfn-appmesh-route-httprouteaction-weightedtargets
            '''
            result = self._values.get("weighted_targets")
            assert result is not None, "Required property 'weighted_targets' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.WeightedTargetProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.HttpRouteHeaderProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class HttpRouteHeaderProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            match: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.HeaderMatchMethodProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the HTTP header in the request.

            :param name: A name for the HTTP header in the client request that will be matched on.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: The ``HeaderMatchMethod`` object.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_route_header_property = appmesh.CfnRoute.HttpRouteHeaderProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnRoute.MatchRangeProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1cb6df6734034f53eac84e743218f4ecc0a717561d50ec9ec2efeb9f669b58f4)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument invert", value=invert, expected_type=type_hints["invert"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the HTTP header in the client request that will be matched on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html#cfn-appmesh-route-httprouteheader-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html#cfn-appmesh-route-httprouteheader-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HeaderMatchMethodProperty"]]:
            '''The ``HeaderMatchMethod`` object.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html#cfn-appmesh-route-httprouteheader-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HeaderMatchMethodProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteHeaderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.HttpRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "headers": "headers",
            "method": "method",
            "path": "path",
            "port": "port",
            "prefix": "prefix",
            "query_parameters": "queryParameters",
            "scheme": "scheme",
        },
    )
    class HttpRouteMatchProperty:
        def __init__(
            self,
            *,
            headers: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.HttpRouteHeaderProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            method: typing.Optional[builtins.str] = None,
            path: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.HttpPathMatchProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            port: typing.Optional[jsii.Number] = None,
            prefix: typing.Optional[builtins.str] = None,
            query_parameters: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.QueryParameterProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            scheme: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the requirements for a route to match HTTP requests for a virtual router.

            :param headers: The client request headers to match on.
            :param method: The client request method to match on. Specify only one.
            :param path: The client request path to match on.
            :param port: The port number to match on.
            :param prefix: Specifies the path to match requests with. This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .
            :param query_parameters: The client request query parameters to match on.
            :param scheme: The client request scheme to match on. Specify only one. Applicable only for HTTP2 routes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_route_match_property = appmesh.CfnRoute.HttpRouteMatchProperty(
                    headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnRoute.MatchRangeProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    method="method",
                    path=appmesh.CfnRoute.HttpPathMatchProperty(
                        exact="exact",
                        regex="regex"
                    ),
                    port=123,
                    prefix="prefix",
                    query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                        name="name",
                
                        # the properties below are optional
                        match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                            exact="exact"
                        )
                    )],
                    scheme="scheme"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__81a9794f0d4f32ade6be8df9d349753f807c3a58ceeb8dfe0b45b551547ef54c)
                check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
                check_type(argname="argument method", value=method, expected_type=type_hints["method"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
                check_type(argname="argument query_parameters", value=query_parameters, expected_type=type_hints["query_parameters"])
                check_type(argname="argument scheme", value=scheme, expected_type=type_hints["scheme"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if headers is not None:
                self._values["headers"] = headers
            if method is not None:
                self._values["method"] = method
            if path is not None:
                self._values["path"] = path
            if port is not None:
                self._values["port"] = port
            if prefix is not None:
                self._values["prefix"] = prefix
            if query_parameters is not None:
                self._values["query_parameters"] = query_parameters
            if scheme is not None:
                self._values["scheme"] = scheme

        @builtins.property
        def headers(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRouteHeaderProperty"]]]]:
            '''The client request headers to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-headers
            '''
            result = self._values.get("headers")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRouteHeaderProperty"]]]], result)

        @builtins.property
        def method(self) -> typing.Optional[builtins.str]:
            '''The client request method to match on.

            Specify only one.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-method
            '''
            result = self._values.get("method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def path(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpPathMatchProperty"]]:
            '''The client request path to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpPathMatchProperty"]], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port number to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''Specifies the path to match requests with.

            This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def query_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.QueryParameterProperty"]]]]:
            '''The client request query parameters to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-queryparameters
            '''
            result = self._values.get("query_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.QueryParameterProperty"]]]], result)

        @builtins.property
        def scheme(self) -> typing.Optional[builtins.str]:
            '''The client request scheme to match on.

            Specify only one. Applicable only for HTTP2 routes.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-scheme
            '''
            result = self._values.get("scheme")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.HttpRouteProperty",
        jsii_struct_bases=[],
        name_mapping={
            "action": "action",
            "match": "match",
            "retry_policy": "retryPolicy",
            "timeout": "timeout",
        },
    )
    class HttpRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.HttpRouteActionProperty", typing.Dict[builtins.str, typing.Any]]],
            match: typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.HttpRouteMatchProperty", typing.Dict[builtins.str, typing.Any]]],
            retry_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.HttpRetryPolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            timeout: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.HttpTimeoutProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents an HTTP or HTTP/2 route type.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.
            :param retry_policy: An object that represents a retry policy.
            :param timeout: An object that represents types of timeouts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_route_property = appmesh.CfnRoute.HttpRouteProperty(
                    action=appmesh.CfnRoute.HttpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
                
                            # the properties below are optional
                            port=123
                        )]
                    ),
                    match=appmesh.CfnRoute.HttpRouteMatchProperty(
                        headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method="method",
                        path=appmesh.CfnRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                            name="name",
                
                            # the properties below are optional
                            match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )],
                        scheme="scheme"
                    ),
                
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                
                        # the properties below are optional
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__128da9620dde4cc1c606338fe00deb55f9a6b9795e6957ebf5be8d96e9c183d5)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
                check_type(argname="argument retry_policy", value=retry_policy, expected_type=type_hints["retry_policy"])
                check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
                "match": match,
            }
            if retry_policy is not None:
                self._values["retry_policy"] = retry_policy
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def action(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRouteActionProperty"]:
            '''An object that represents the action to take if a match is determined.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRouteActionProperty"], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRouteMatchProperty"]:
            '''An object that represents the criteria for determining a request match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRouteMatchProperty"], result)

        @builtins.property
        def retry_policy(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRetryPolicyProperty"]]:
            '''An object that represents a retry policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-retrypolicy
            '''
            result = self._values.get("retry_policy")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRetryPolicyProperty"]], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpTimeoutProperty"]]:
            '''An object that represents types of timeouts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpTimeoutProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.HttpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class HttpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            per_request: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_timeout_property = appmesh.CfnRoute.HttpTimeoutProperty(
                    idle=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8e9f42ac7c453b6b05dec769a7861cca303a897e65133bd2a745630ca2b094b8)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
                check_type(argname="argument per_request", value=per_request, expected_type=type_hints["per_request"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httptimeout.html#cfn-appmesh-route-httptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httptimeout.html#cfn-appmesh-route-httptimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.MatchRangeProperty",
        jsii_struct_bases=[],
        name_mapping={"end": "end", "start": "start"},
    )
    class MatchRangeProperty:
        def __init__(self, *, end: jsii.Number, start: jsii.Number) -> None:
            '''An object that represents the range of values to match on.

            The first character of the range is included in the range, though the last character is not. For example, if the range specified were 1-100, only values 1-99 would be matched.

            :param end: The end of the range.
            :param start: The start of the range.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-matchrange.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                match_range_property = appmesh.CfnRoute.MatchRangeProperty(
                    end=123,
                    start=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ea4e7cddbd5eedd56fe72912d95a0affeea0cdd0049df4531ca29ef359cde4d2)
                check_type(argname="argument end", value=end, expected_type=type_hints["end"])
                check_type(argname="argument start", value=start, expected_type=type_hints["start"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "end": end,
                "start": start,
            }

        @builtins.property
        def end(self) -> jsii.Number:
            '''The end of the range.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-matchrange.html#cfn-appmesh-route-matchrange-end
            '''
            result = self._values.get("end")
            assert result is not None, "Required property 'end' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def start(self) -> jsii.Number:
            '''The start of the range.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-matchrange.html#cfn-appmesh-route-matchrange-start
            '''
            result = self._values.get("start")
            assert result is not None, "Required property 'start' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MatchRangeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.QueryParameterProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "match": "match"},
    )
    class QueryParameterProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            match: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.HttpQueryParameterMatchProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the query parameter in the request.

            :param name: A name for the query parameter that will be matched on.
            :param match: The query parameter to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-queryparameter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                query_parameter_property = appmesh.CfnRoute.QueryParameterProperty(
                    name="name",
                
                    # the properties below are optional
                    match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                        exact="exact"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0f7d848a94aaa29598fc3bf035f5a023fc7711eb5782a53767c3d6aa687ec076)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the query parameter that will be matched on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-queryparameter.html#cfn-appmesh-route-queryparameter-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpQueryParameterMatchProperty"]]:
            '''The query parameter to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-queryparameter.html#cfn-appmesh-route-queryparameter-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpQueryParameterMatchProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "QueryParameterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.RouteSpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "grpc_route": "grpcRoute",
            "http2_route": "http2Route",
            "http_route": "httpRoute",
            "priority": "priority",
            "tcp_route": "tcpRoute",
        },
    )
    class RouteSpecProperty:
        def __init__(
            self,
            *,
            grpc_route: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.GrpcRouteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            http2_route: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.HttpRouteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            http_route: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.HttpRouteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            priority: typing.Optional[jsii.Number] = None,
            tcp_route: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.TcpRouteProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a route specification.

            Specify one route type.

            :param grpc_route: An object that represents the specification of a gRPC route.
            :param http2_route: An object that represents the specification of an HTTP/2 route.
            :param http_route: An object that represents the specification of an HTTP route.
            :param priority: The priority for the route. Routes are matched based on the specified value, where 0 is the highest priority.
            :param tcp_route: An object that represents the specification of a TCP route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                route_spec_property = appmesh.CfnRoute.RouteSpecProperty(
                    grpc_route=appmesh.CfnRoute.GrpcRouteProperty(
                        action=appmesh.CfnRoute.GrpcRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
                
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                            metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method_name="methodName",
                            port=123,
                            service_name="serviceName"
                        ),
                
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                
                            # the properties below are optional
                            grpc_retry_events=["grpcRetryEvents"],
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http2_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
                
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
                
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
                
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
                
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    priority=123,
                    tcp_route=appmesh.CfnRoute.TcpRouteProperty(
                        action=appmesh.CfnRoute.TcpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
                
                                # the properties below are optional
                                port=123
                            )]
                        ),
                
                        # the properties below are optional
                        match=appmesh.CfnRoute.TcpRouteMatchProperty(
                            port=123
                        ),
                        timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c3ffd97e428d6d5aa5654bb0d515bab872a65e75a144eb8ff542ee66cd959ce3)
                check_type(argname="argument grpc_route", value=grpc_route, expected_type=type_hints["grpc_route"])
                check_type(argname="argument http2_route", value=http2_route, expected_type=type_hints["http2_route"])
                check_type(argname="argument http_route", value=http_route, expected_type=type_hints["http_route"])
                check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
                check_type(argname="argument tcp_route", value=tcp_route, expected_type=type_hints["tcp_route"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if grpc_route is not None:
                self._values["grpc_route"] = grpc_route
            if http2_route is not None:
                self._values["http2_route"] = http2_route
            if http_route is not None:
                self._values["http_route"] = http_route
            if priority is not None:
                self._values["priority"] = priority
            if tcp_route is not None:
                self._values["tcp_route"] = tcp_route

        @builtins.property
        def grpc_route(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRouteProperty"]]:
            '''An object that represents the specification of a gRPC route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-grpcroute
            '''
            result = self._values.get("grpc_route")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.GrpcRouteProperty"]], result)

        @builtins.property
        def http2_route(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRouteProperty"]]:
            '''An object that represents the specification of an HTTP/2 route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-http2route
            '''
            result = self._values.get("http2_route")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRouteProperty"]], result)

        @builtins.property
        def http_route(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRouteProperty"]]:
            '''An object that represents the specification of an HTTP route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-httproute
            '''
            result = self._values.get("http_route")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.HttpRouteProperty"]], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''The priority for the route.

            Routes are matched based on the specified value, where 0 is the highest priority.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def tcp_route(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.TcpRouteProperty"]]:
            '''An object that represents the specification of a TCP route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-tcproute
            '''
            result = self._values.get("tcp_route")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.TcpRouteProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RouteSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.TcpRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"weighted_targets": "weightedTargets"},
    )
    class TcpRouteActionProperty:
        def __init__(
            self,
            *,
            weighted_targets: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.WeightedTargetProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param weighted_targets: An object that represents the targets that traffic is routed to when a request matches the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcprouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                tcp_route_action_property = appmesh.CfnRoute.TcpRouteActionProperty(
                    weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                        virtual_node="virtualNode",
                        weight=123,
                
                        # the properties below are optional
                        port=123
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__74eee418b469f953415361888261c91898382e2c66f9ead8fb54e207678d74b6)
                check_type(argname="argument weighted_targets", value=weighted_targets, expected_type=type_hints["weighted_targets"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "weighted_targets": weighted_targets,
            }

        @builtins.property
        def weighted_targets(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.WeightedTargetProperty"]]]:
            '''An object that represents the targets that traffic is routed to when a request matches the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcprouteaction.html#cfn-appmesh-route-tcprouteaction-weightedtargets
            '''
            result = self._values.get("weighted_targets")
            assert result is not None, "Required property 'weighted_targets' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnRoute.WeightedTargetProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.TcpRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port"},
    )
    class TcpRouteMatchProperty:
        def __init__(self, *, port: typing.Optional[jsii.Number] = None) -> None:
            '''An object representing the TCP route to match.

            :param port: The port number to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                tcp_route_match_property = appmesh.CfnRoute.TcpRouteMatchProperty(
                    port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__52cdabe50fd214c166f5fe0542a9ebe4b5b95b338aa3535b06c1620a81826c5f)
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port number to match on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproutematch.html#cfn-appmesh-route-tcproutematch-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.TcpRouteProperty",
        jsii_struct_bases=[],
        name_mapping={"action": "action", "match": "match", "timeout": "timeout"},
    )
    class TcpRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.TcpRouteActionProperty", typing.Dict[builtins.str, typing.Any]]],
            match: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.TcpRouteMatchProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            timeout: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.TcpTimeoutProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a TCP route type.

            :param action: The action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.
            :param timeout: An object that represents types of timeouts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                tcp_route_property = appmesh.CfnRoute.TcpRouteProperty(
                    action=appmesh.CfnRoute.TcpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123,
                
                            # the properties below are optional
                            port=123
                        )]
                    ),
                
                    # the properties below are optional
                    match=appmesh.CfnRoute.TcpRouteMatchProperty(
                        port=123
                    ),
                    timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__449518ae382d90da2f1ec45c9efdd0e5165665fdc231ffef0e9b3916ea282ca2)
                check_type(argname="argument action", value=action, expected_type=type_hints["action"])
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
                check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "action": action,
            }
            if match is not None:
                self._values["match"] = match
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def action(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnRoute.TcpRouteActionProperty"]:
            '''The action to take if a match is determined.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproute.html#cfn-appmesh-route-tcproute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnRoute.TcpRouteActionProperty"], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.TcpRouteMatchProperty"]]:
            '''An object that represents the criteria for determining a request match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproute.html#cfn-appmesh-route-tcproute-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.TcpRouteMatchProperty"]], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.TcpTimeoutProperty"]]:
            '''An object that represents types of timeouts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproute.html#cfn-appmesh-route-tcproute-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.TcpTimeoutProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.TcpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle"},
    )
    class TcpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRoute.DurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                tcp_timeout_property = appmesh.CfnRoute.TcpTimeoutProperty(
                    idle=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__319340571e5253faf0b9fa87d075ee236f7d44df36ea03fa402a4dab24b82167)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcptimeout.html#cfn-appmesh-route-tcptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRoute.DurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnRoute.WeightedTargetProperty",
        jsii_struct_bases=[],
        name_mapping={
            "virtual_node": "virtualNode",
            "weight": "weight",
            "port": "port",
        },
    )
    class WeightedTargetProperty:
        def __init__(
            self,
            *,
            virtual_node: builtins.str,
            weight: jsii.Number,
            port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a target and its relative weight.

            Traffic is distributed across targets according to their relative weight. For example, a weighted target with a relative weight of 50 receives five times as much traffic as one with a relative weight of 10. The total weight for all targets combined must be less than or equal to 100.

            :param virtual_node: The virtual node to associate with the weighted target.
            :param weight: The relative weight of the weighted target.
            :param port: The targeted port of the weighted object.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-weightedtarget.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                weighted_target_property = appmesh.CfnRoute.WeightedTargetProperty(
                    virtual_node="virtualNode",
                    weight=123,
                
                    # the properties below are optional
                    port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2ce23e921e09cb74cc6b5fb3ee11acfde11d40ff4b5a95d6335a1df59441740c)
                check_type(argname="argument virtual_node", value=virtual_node, expected_type=type_hints["virtual_node"])
                check_type(argname="argument weight", value=weight, expected_type=type_hints["weight"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_node": virtual_node,
                "weight": weight,
            }
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def virtual_node(self) -> builtins.str:
            '''The virtual node to associate with the weighted target.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-weightedtarget.html#cfn-appmesh-route-weightedtarget-virtualnode
            '''
            result = self._values.get("virtual_node")
            assert result is not None, "Required property 'virtual_node' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def weight(self) -> jsii.Number:
            '''The relative weight of the weighted target.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-weightedtarget.html#cfn-appmesh-route-weightedtarget-weight
            '''
            result = self._values.get("weight")
            assert result is not None, "Required property 'weight' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The targeted port of the weighted object.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-weightedtarget.html#cfn-appmesh-route-weightedtarget-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "WeightedTargetProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.CfnRouteProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "virtual_router_name": "virtualRouterName",
        "mesh_owner": "meshOwner",
        "route_name": "routeName",
        "tags": "tags",
    },
)
class CfnRouteProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union[CfnRoute.RouteSpecProperty, typing.Dict[builtins.str, typing.Any]]],
        virtual_router_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        route_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnRoute``.

        :param mesh_name: The name of the service mesh to create the route in.
        :param spec: The route specification to apply.
        :param virtual_router_name: The name of the virtual router in which to create the route. If the virtual router is in a shared mesh, then you must be the owner of the virtual router resource.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param route_name: The name to use for the route.
        :param tags: Optional metadata that you can apply to the route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            cfn_route_props = appmesh.CfnRouteProps(
                mesh_name="meshName",
                spec=appmesh.CfnRoute.RouteSpecProperty(
                    grpc_route=appmesh.CfnRoute.GrpcRouteProperty(
                        action=appmesh.CfnRoute.GrpcRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
            
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                            metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method_name="methodName",
                            port=123,
                            service_name="serviceName"
                        ),
            
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
            
                            # the properties below are optional
                            grpc_retry_events=["grpcRetryEvents"],
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http2_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
            
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
            
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
            
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
            
                                # the properties below are optional
                                port=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            port=123,
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
            
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
            
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    priority=123,
                    tcp_route=appmesh.CfnRoute.TcpRouteProperty(
                        action=appmesh.CfnRoute.TcpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123,
            
                                # the properties below are optional
                                port=123
                            )]
                        ),
            
                        # the properties below are optional
                        match=appmesh.CfnRoute.TcpRouteMatchProperty(
                            port=123
                        ),
                        timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    )
                ),
                virtual_router_name="virtualRouterName",
            
                # the properties below are optional
                mesh_owner="meshOwner",
                route_name="routeName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5a87706b1575e58a83ddeb3051fe8fca416d12d530846c465c0ae34665af32a9)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument virtual_router_name", value=virtual_router_name, expected_type=type_hints["virtual_router_name"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument route_name", value=route_name, expected_type=type_hints["route_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
            "virtual_router_name": virtual_router_name,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if route_name is not None:
            self._values["route_name"] = route_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the route in.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(self) -> typing.Union[_IResolvable_da3f097b, CfnRoute.RouteSpecProperty]:
        '''The route specification to apply.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnRoute.RouteSpecProperty], result)

    @builtins.property
    def virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router in which to create the route.

        If the virtual router is in a shared mesh, then you must be the owner of the virtual router resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-virtualroutername
        '''
        result = self._values.get("virtual_router_name")
        assert result is not None, "Required property 'virtual_router_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def route_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the route.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-routename
        '''
        result = self._values.get("route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the route to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnVirtualGateway(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway",
):
    '''Creates a virtual gateway.

    A virtual gateway allows resources outside your mesh to communicate to resources that are inside your mesh. The virtual gateway represents an Envoy proxy running in an Amazon ECS task, in a Kubernetes service, or on an Amazon EC2 instance. Unlike a virtual node, which represents an Envoy running with an application, a virtual gateway represents Envoy deployed by itself.

    For more information about virtual gateways, see `Virtual gateways <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html>`_ .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html
    :cloudformationResource: AWS::AppMesh::VirtualGateway
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_appmesh as appmesh
        
        cfn_virtual_gateway = appmesh.CfnVirtualGateway(self, "MyCfnVirtualGateway",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty(
                listeners=[appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                    port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
        
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                            max_connections=123,
        
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                            max_requests=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
        
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                        certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
        
                        # the properties below are optional
                        validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
        
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )],
        
                # the properties below are optional
                backend_defaults=appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
        
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
        
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                ),
                logging=appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                    access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                            path="path",
        
                            # the properties below are optional
                            format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                                json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                    key="key",
                                    value="value"
                                )],
                                text="text"
                            )
                        )
                    )
                )
            ),
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            virtual_gateway_name="virtualGatewayName"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewaySpecProperty", typing.Dict[builtins.str, typing.Any]]],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_gateway_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param mesh_name: The name of the service mesh that the virtual gateway resides in.
        :param spec: The specifications of the virtual gateway.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual gateway to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_gateway_name: The name of the virtual gateway.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0543a7feb4076ec54ae351faecebaceae2aded9f595e1ab9e06d95da471e11b3)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVirtualGatewayProps(
            mesh_name=mesh_name,
            spec=spec,
            mesh_owner=mesh_owner,
            tags=tags,
            virtual_gateway_name=virtual_gateway_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4bb323619892fb3b6adeb751a552df4ef5465462cfddb364e9452c4dfff138df)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f0556f5aa0216f163083b7d56d24d3fe96f15e7680daa02e7c9940c0b38dffe8)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual gateway.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual gateway resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual gateway.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualGatewayName")
    def attr_virtual_gateway_name(self) -> builtins.str:
        '''The name of the virtual gateway.

        :cloudformationAttribute: VirtualGatewayName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualGatewayName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual gateway resides in.'''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__589c5e56d7e99e2aa8c5b24de9903773596dad0233dbd7f0f54e06c09077a4e9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewaySpecProperty"]:
        '''The specifications of the virtual gateway.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewaySpecProperty"], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewaySpecProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__510b3d42a651bf6471450f27f03fe5c67f00152d5a0e35e3627c4fcfb8921d36)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__538618f322715e08d311a2ae4c14a1da686bd208809926581e3cabdaa6956a7c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the virtual gateway to assist with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3f36904ee813091921f0a5d19544f2321ffe82ff49dbbd2a8b6c304dbcb00701)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="virtualGatewayName")
    def virtual_gateway_name(self) -> typing.Optional[builtins.str]:
        '''The name of the virtual gateway.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualGatewayName"))

    @virtual_gateway_name.setter
    def virtual_gateway_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b65d4de99ea85b8c949aa3c904b0edeb9a81a976a8f49fea584910eec951d2f3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualGatewayName", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.JsonFormatRefProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class JsonFormatRefProperty:
        def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
            '''An object that represents the key value pairs for the JSON.

            :param key: The specified key for the JSON.
            :param value: The specified value for the JSON.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-jsonformatref.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                json_format_ref_property = appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bbee08461b4933475112f17cfb94bb85e46d46173683693dbda78b0954ba9de4)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''The specified key for the JSON.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-jsonformatref.html#cfn-appmesh-virtualgateway-jsonformatref-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The specified value for the JSON.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-jsonformatref.html#cfn-appmesh-virtualgateway-jsonformatref-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "JsonFormatRefProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.LoggingFormatProperty",
        jsii_struct_bases=[],
        name_mapping={"json": "json", "text": "text"},
    )
    class LoggingFormatProperty:
        def __init__(
            self,
            *,
            json: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.JsonFormatRefProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            text: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the format for the logs.

            :param json: The logging format for JSON.
            :param text: The logging format for text.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-loggingformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                logging_format_property = appmesh.CfnVirtualGateway.LoggingFormatProperty(
                    json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                        key="key",
                        value="value"
                    )],
                    text="text"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7e9ee5c899cde8f40f05770f8a9ab9a633500e613a8786ee3d9883ccbec6c222)
                check_type(argname="argument json", value=json, expected_type=type_hints["json"])
                check_type(argname="argument text", value=text, expected_type=type_hints["text"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if json is not None:
                self._values["json"] = json
            if text is not None:
                self._values["text"] = text

        @builtins.property
        def json(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.JsonFormatRefProperty"]]]]:
            '''The logging format for JSON.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-loggingformat.html#cfn-appmesh-virtualgateway-loggingformat-json
            '''
            result = self._values.get("json")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.JsonFormatRefProperty"]]]], result)

        @builtins.property
        def text(self) -> typing.Optional[builtins.str]:
            '''The logging format for text.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-loggingformat.html#cfn-appmesh-virtualgateway-loggingformat-text
            '''
            result = self._values.get("text")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class SubjectAlternativeNameMatchersProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents the methods by which a subject alternative name on a peer Transport Layer Security (TLS) certificate can be matched.

            :param exact: The values sent must match the specified values exactly.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenamematchers.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                subject_alternative_name_matchers_property = appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                    exact=["exact"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b590955a6397d2a3150cb96fbe4e95970270cff86d5bfc94ddd23fe1e152fedf)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The values sent must match the specified values exactly.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenamematchers.html#cfn-appmesh-virtualgateway-subjectalternativenamematchers-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNameMatchersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty",
        jsii_struct_bases=[],
        name_mapping={"match": "match"},
    )
    class SubjectAlternativeNamesProperty:
        def __init__(
            self,
            *,
            match: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.SubjectAlternativeNameMatchersProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''An object that represents the subject alternative names secured by the certificate.

            :param match: An object that represents the criteria for determining a SANs match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenames.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                subject_alternative_names_property = appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                        exact=["exact"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__969fbf5b7283c835265a1b45e1955312979ac44335c64192e539715bdbbe4804)
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "match": match,
            }

        @builtins.property
        def match(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.SubjectAlternativeNameMatchersProperty"]:
            '''An object that represents the criteria for determining a SANs match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenames.html#cfn-appmesh-virtualgateway-subjectalternativenames-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.SubjectAlternativeNameMatchersProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNamesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file"},
    )
    class VirtualGatewayAccessLogProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayFileAccessLogProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The access log configuration for a virtual gateway.

            :param file: The file object to send virtual gateway access logs to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayaccesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_access_log_property = appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                        path="path",
                
                        # the properties below are optional
                        format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                            json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                key="key",
                                value="value"
                            )],
                            text="text"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8bb9e8c320b763f85ef609c58745e72421b4be38179bb690cf3c57d361b76459)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayFileAccessLogProperty"]]:
            '''The file object to send virtual gateway access logs to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayaccesslog.html#cfn-appmesh-virtualgateway-virtualgatewayaccesslog-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayFileAccessLogProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayAccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty",
        jsii_struct_bases=[],
        name_mapping={"client_policy": "clientPolicy"},
    )
    class VirtualGatewayBackendDefaultsProperty:
        def __init__(
            self,
            *,
            client_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the default properties for a backend.

            :param client_policy: A reference to an object that represents a client policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaybackenddefaults.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_backend_defaults_property = appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
                
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0512c851821b473daa4f22a6048599c45777ba45aae432a9014333957a5c531d)
                check_type(argname="argument client_policy", value=client_policy, expected_type=type_hints["client_policy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if client_policy is not None:
                self._values["client_policy"] = client_policy

        @builtins.property
        def client_policy(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayClientPolicyProperty"]]:
            '''A reference to an object that represents a client policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaybackenddefaults.html#cfn-appmesh-virtualgateway-virtualgatewaybackenddefaults-clientpolicy
            '''
            result = self._values.get("client_policy")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayClientPolicyProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayBackendDefaultsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"tls": "tls"},
    )
    class VirtualGatewayClientPolicyProperty:
        def __init__(
            self,
            *,
            tls: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a client policy.

            :param tls: A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_client_policy_property = appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                    tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                        validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                    certificate_authority_arns=["certificateAuthorityArns"]
                                ),
                                file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        ),
                
                        # the properties below are optional
                        certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        enforce=False,
                        ports=[123]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e0e87a8838c1991d0174637b983896aa747ebf760a3227910fdfee9e454c7ea7)
                check_type(argname="argument tls", value=tls, expected_type=type_hints["tls"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty"]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicy-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayClientPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "validation": "validation",
            "certificate": "certificate",
            "enforce": "enforce",
            "ports": "ports",
        },
    )
    class VirtualGatewayClientPolicyTlsProperty:
        def __init__(
            self,
            *,
            validation: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty", typing.Dict[builtins.str, typing.Any]]],
            certificate: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            enforce: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            ports: typing.Optional[typing.Union[typing.Sequence[jsii.Number], _IResolvable_da3f097b]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) client policy.

            :param validation: A reference to an object that represents a Transport Layer Security (TLS) validation context.
            :param certificate: A reference to an object that represents a virtual gateway's client's Transport Layer Security (TLS) certificate.
            :param enforce: Whether the policy is enforced. The default is ``True`` , if a value isn't specified.
            :param ports: One or more ports that the policy is enforced for.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_client_policy_tls_property = appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                    validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                        trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                            acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                certificate_authority_arns=["certificateAuthorityArns"]
                            ),
                            file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    ),
                
                    # the properties below are optional
                    certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    enforce=False,
                    ports=[123]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__14272eeb7cb9da842115526007e77a31b8c8c262c8f9523a946de777ae45543a)
                check_type(argname="argument validation", value=validation, expected_type=type_hints["validation"])
                check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
                check_type(argname="argument enforce", value=enforce, expected_type=type_hints["enforce"])
                check_type(argname="argument ports", value=ports, expected_type=type_hints["ports"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "validation": validation,
            }
            if certificate is not None:
                self._values["certificate"] = certificate
            if enforce is not None:
                self._values["enforce"] = enforce
            if ports is not None:
                self._values["ports"] = ports

        @builtins.property
        def validation(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty"]:
            '''A reference to an object that represents a Transport Layer Security (TLS) validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-validation
            '''
            result = self._values.get("validation")
            assert result is not None, "Required property 'validation' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty"], result)

        @builtins.property
        def certificate(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty"]]:
            '''A reference to an object that represents a virtual gateway's client's Transport Layer Security (TLS) certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-certificate
            '''
            result = self._values.get("certificate")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty"]], result)

        @builtins.property
        def enforce(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Whether the policy is enforced.

            The default is ``True`` , if a value isn't specified.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-enforce
            '''
            result = self._values.get("enforce")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def ports(
            self,
        ) -> typing.Optional[typing.Union[typing.List[jsii.Number], _IResolvable_da3f097b]]:
            '''One or more ports that the policy is enforced for.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-ports
            '''
            result = self._values.get("ports")
            return typing.cast(typing.Optional[typing.Union[typing.List[jsii.Number], _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayClientPolicyTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class VirtualGatewayClientTlsCertificateProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            sds: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the virtual gateway's client's Transport Layer Security (TLS) certificate.

            :param file: An object that represents a local file certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .
            :param sds: A reference to an object that represents a virtual gateway's client's Secret Discovery Service certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclienttlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_client_tls_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ec3f919972e81a1499b9945df9437d3e1f0c698ab1b4fd97d686eb5bc148da2b)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty"]]:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclienttlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewayclienttlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty"]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty"]]:
            '''A reference to an object that represents a virtual gateway's client's Secret Discovery Service certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclienttlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewayclienttlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayClientTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"grpc": "grpc", "http": "http", "http2": "http2"},
    )
    class VirtualGatewayConnectionPoolProperty:
        def __init__(
            self,
            *,
            grpc: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            http: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            http2: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the type of virtual gateway connection pool.

            Only one protocol is used at a time and should be the same protocol as the one chosen under port mapping.

            If not present the default value for ``maxPendingRequests`` is ``2147483647`` .

            :param grpc: An object that represents a type of connection pool.
            :param http: An object that represents a type of connection pool.
            :param http2: An object that represents a type of connection pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                    grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                        max_requests=123
                    ),
                    http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                        max_connections=123,
                
                        # the properties below are optional
                        max_pending_requests=123
                    ),
                    http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                        max_requests=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c0c060e1533bc841d9746e3069453809404c9055229403ca35232846b2c81305)
                check_type(argname="argument grpc", value=grpc, expected_type=type_hints["grpc"])
                check_type(argname="argument http", value=http, expected_type=type_hints["http"])
                check_type(argname="argument http2", value=http2, expected_type=type_hints["http2"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if grpc is not None:
                self._values["grpc"] = grpc
            if http is not None:
                self._values["http"] = http
            if http2 is not None:
                self._values["http2"] = http2

        @builtins.property
        def grpc(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty"]]:
            '''An object that represents a type of connection pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayconnectionpool-grpc
            '''
            result = self._values.get("grpc")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty"]], result)

        @builtins.property
        def http(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty"]]:
            '''An object that represents a type of connection pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayconnectionpool-http
            '''
            result = self._values.get("http")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty"]], result)

        @builtins.property
        def http2(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty"]]:
            '''An object that represents a type of connection pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayconnectionpool-http2
            '''
            result = self._values.get("http2")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"path": "path", "format": "format"},
    )
    class VirtualGatewayFileAccessLogProperty:
        def __init__(
            self,
            *,
            path: builtins.str,
            format: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.LoggingFormatProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents an access log file.

            :param path: The file path to write access logs to. You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk.
            :param format: The specified format for the virtual gateway access logs. It can be either ``json_format`` or ``text_format`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayfileaccesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_file_access_log_property = appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                    path="path",
                
                    # the properties below are optional
                    format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                        json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                            key="key",
                            value="value"
                        )],
                        text="text"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0b1fedd19569dc28681b061000353c0e69ba8083ddc4ce7bbe7ef678c9ffd545)
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument format", value=format, expected_type=type_hints["format"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "path": path,
            }
            if format is not None:
                self._values["format"] = format

        @builtins.property
        def path(self) -> builtins.str:
            '''The file path to write access logs to.

            You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayfileaccesslog.html#cfn-appmesh-virtualgateway-virtualgatewayfileaccesslog-path
            '''
            result = self._values.get("path")
            assert result is not None, "Required property 'path' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def format(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.LoggingFormatProperty"]]:
            '''The specified format for the virtual gateway access logs.

            It can be either ``json_format`` or ``text_format`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayfileaccesslog.html#cfn-appmesh-virtualgateway-virtualgatewayfileaccesslog-format
            '''
            result = self._values.get("format")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.LoggingFormatProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayFileAccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualGatewayGrpcConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaygrpcconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_grpc_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                    max_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__85040f4c285f23f5791804c7b383f26cc4913a9a6c258d9ea962ef803bab3c3f)
                check_type(argname="argument max_requests", value=max_requests, expected_type=type_hints["max_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaygrpcconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewaygrpcconnectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayGrpcConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "healthy_threshold": "healthyThreshold",
            "interval_millis": "intervalMillis",
            "protocol": "protocol",
            "timeout_millis": "timeoutMillis",
            "unhealthy_threshold": "unhealthyThreshold",
            "path": "path",
            "port": "port",
        },
    )
    class VirtualGatewayHealthCheckPolicyProperty:
        def __init__(
            self,
            *,
            healthy_threshold: jsii.Number,
            interval_millis: jsii.Number,
            protocol: builtins.str,
            timeout_millis: jsii.Number,
            unhealthy_threshold: jsii.Number,
            path: typing.Optional[builtins.str] = None,
            port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents the health check policy for a virtual gateway's listener.

            :param healthy_threshold: The number of consecutive successful health checks that must occur before declaring the listener healthy.
            :param interval_millis: The time period in milliseconds between each health check execution.
            :param protocol: The protocol for the health check request. If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .
            :param timeout_millis: The amount of time to wait when receiving a response from the health check, in milliseconds.
            :param unhealthy_threshold: The number of consecutive failed health checks that must occur before declaring a virtual gateway unhealthy.
            :param path: The destination path for the health check request. This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.
            :param port: The destination port for the health check request. This port must match the port defined in the ``PortMapping`` for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_health_check_policy_property = appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                    healthy_threshold=123,
                    interval_millis=123,
                    protocol="protocol",
                    timeout_millis=123,
                    unhealthy_threshold=123,
                
                    # the properties below are optional
                    path="path",
                    port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4b38edaf02e0a0ad61bf74a7dcc451f1253be30b34008b1b9c3fb22a9bd4cc2e)
                check_type(argname="argument healthy_threshold", value=healthy_threshold, expected_type=type_hints["healthy_threshold"])
                check_type(argname="argument interval_millis", value=interval_millis, expected_type=type_hints["interval_millis"])
                check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
                check_type(argname="argument timeout_millis", value=timeout_millis, expected_type=type_hints["timeout_millis"])
                check_type(argname="argument unhealthy_threshold", value=unhealthy_threshold, expected_type=type_hints["unhealthy_threshold"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "healthy_threshold": healthy_threshold,
                "interval_millis": interval_millis,
                "protocol": protocol,
                "timeout_millis": timeout_millis,
                "unhealthy_threshold": unhealthy_threshold,
            }
            if path is not None:
                self._values["path"] = path
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def healthy_threshold(self) -> jsii.Number:
            '''The number of consecutive successful health checks that must occur before declaring the listener healthy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-healthythreshold
            '''
            result = self._values.get("healthy_threshold")
            assert result is not None, "Required property 'healthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def interval_millis(self) -> jsii.Number:
            '''The time period in milliseconds between each health check execution.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-intervalmillis
            '''
            result = self._values.get("interval_millis")
            assert result is not None, "Required property 'interval_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol for the health check request.

            If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def timeout_millis(self) -> jsii.Number:
            '''The amount of time to wait when receiving a response from the health check, in milliseconds.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-timeoutmillis
            '''
            result = self._values.get("timeout_millis")
            assert result is not None, "Required property 'timeout_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def unhealthy_threshold(self) -> jsii.Number:
            '''The number of consecutive failed health checks that must occur before declaring a virtual gateway unhealthy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-unhealthythreshold
            '''
            result = self._values.get("unhealthy_threshold")
            assert result is not None, "Required property 'unhealthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def path(self) -> typing.Optional[builtins.str]:
            '''The destination path for the health check request.

            This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The destination port for the health check request.

            This port must match the port defined in the ``PortMapping`` for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayHealthCheckPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualGatewayHttp2ConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttp2connectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_http2_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                    max_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__eeee4f1ba0bef1f20ff651a43f9f42ed4b1110571a18c9818c07ab71980961e6)
                check_type(argname="argument max_requests", value=max_requests, expected_type=type_hints["max_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttp2connectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayhttp2connectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayHttp2ConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_connections": "maxConnections",
            "max_pending_requests": "maxPendingRequests",
        },
    )
    class VirtualGatewayHttpConnectionPoolProperty:
        def __init__(
            self,
            *,
            max_connections: jsii.Number,
            max_pending_requests: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a type of connection pool.

            :param max_connections: Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.
            :param max_pending_requests: Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttpconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_http_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                    max_connections=123,
                
                    # the properties below are optional
                    max_pending_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e187bc9c8a32e237ff6a62ac9ca2d2c5facc934362103a7067bb49c1e2a57f03)
                check_type(argname="argument max_connections", value=max_connections, expected_type=type_hints["max_connections"])
                check_type(argname="argument max_pending_requests", value=max_pending_requests, expected_type=type_hints["max_pending_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_connections": max_connections,
            }
            if max_pending_requests is not None:
                self._values["max_pending_requests"] = max_pending_requests

        @builtins.property
        def max_connections(self) -> jsii.Number:
            '''Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttpconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayhttpconnectionpool-maxconnections
            '''
            result = self._values.get("max_connections")
            assert result is not None, "Required property 'max_connections' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def max_pending_requests(self) -> typing.Optional[jsii.Number]:
            '''Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttpconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayhttpconnectionpool-maxpendingrequests
            '''
            result = self._values.get("max_pending_requests")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayHttpConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "port_mapping": "portMapping",
            "connection_pool": "connectionPool",
            "health_check": "healthCheck",
            "tls": "tls",
        },
    )
    class VirtualGatewayListenerProperty:
        def __init__(
            self,
            *,
            port_mapping: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayPortMappingProperty", typing.Dict[builtins.str, typing.Any]]],
            connection_pool: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            health_check: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            tls: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a listener for a virtual gateway.

            :param port_mapping: The port mapping information for the listener.
            :param connection_pool: The connection pool information for the listener.
            :param health_check: The health check information for the listener.
            :param tls: A reference to an object that represents the Transport Layer Security (TLS) properties for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                    port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
                
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                            max_connections=123,
                
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                            max_requests=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
                
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                        certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
                
                        # the properties below are optional
                        validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fddcf127bed3e28c56ce7aa7724e9f55c3d9540bf06edf70f200a7631f1cc1c0)
                check_type(argname="argument port_mapping", value=port_mapping, expected_type=type_hints["port_mapping"])
                check_type(argname="argument connection_pool", value=connection_pool, expected_type=type_hints["connection_pool"])
                check_type(argname="argument health_check", value=health_check, expected_type=type_hints["health_check"])
                check_type(argname="argument tls", value=tls, expected_type=type_hints["tls"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port_mapping": port_mapping,
            }
            if connection_pool is not None:
                self._values["connection_pool"] = connection_pool
            if health_check is not None:
                self._values["health_check"] = health_check
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def port_mapping(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayPortMappingProperty"]:
            '''The port mapping information for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-portmapping
            '''
            result = self._values.get("port_mapping")
            assert result is not None, "Required property 'port_mapping' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayPortMappingProperty"], result)

        @builtins.property
        def connection_pool(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayConnectionPoolProperty"]]:
            '''The connection pool information for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-connectionpool
            '''
            result = self._values.get("connection_pool")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayConnectionPoolProperty"]], result)

        @builtins.property
        def health_check(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty"]]:
            '''The health check information for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-healthcheck
            '''
            result = self._values.get("health_check")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty"]], result)

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsProperty"]]:
            '''A reference to an object that represents the Transport Layer Security (TLS) properties for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_arn": "certificateArn"},
    )
    class VirtualGatewayListenerTlsAcmCertificateProperty:
        def __init__(self, *, certificate_arn: builtins.str) -> None:
            '''An object that represents an AWS Certificate Manager certificate.

            :param certificate_arn: The Amazon Resource Name (ARN) for the certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsacmcertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_acm_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                    certificate_arn="certificateArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__df3734e46ab88fe2826d6285fc4a024ae4ad7d052dca93fca5e5cd497bea2035)
                check_type(argname="argument certificate_arn", value=certificate_arn, expected_type=type_hints["certificate_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_arn": certificate_arn,
            }

        @builtins.property
        def certificate_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) for the certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsacmcertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsacmcertificate-certificatearn
            '''
            result = self._values.get("certificate_arn")
            assert result is not None, "Required property 'certificate_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsAcmCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class VirtualGatewayListenerTlsCertificateProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            file: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            sds: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) certificate.

            :param acm: A reference to an object that represents an AWS Certificate Manager certificate.
            :param file: A reference to an object that represents a local file certificate.
            :param sds: A reference to an object that represents a virtual gateway's listener's Secret Discovery Service certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                    acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                        certificate_arn="certificateArn"
                    ),
                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3f7161d9e7b246ad1f9915aca1ac49c64bccf1c5e9dd59261a91d8aaf2807c7e)
                check_type(argname="argument acm", value=acm, expected_type=type_hints["acm"])
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty"]]:
            '''A reference to an object that represents an AWS Certificate Manager certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlscertificate-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty"]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty"]]:
            '''A reference to an object that represents a local file certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty"]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty"]]:
            '''A reference to an object that represents a virtual gateway's listener's Secret Discovery Service certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate_chain": "certificateChain",
            "private_key": "privateKey",
        },
    )
    class VirtualGatewayListenerTlsFileCertificateProperty:
        def __init__(
            self,
            *,
            certificate_chain: builtins.str,
            private_key: builtins.str,
        ) -> None:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :param certificate_chain: The certificate chain for the certificate.
            :param private_key: The private key for a certificate stored on the file system of the mesh endpoint that the proxy is running on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_file_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                    certificate_chain="certificateChain",
                    private_key="privateKey"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6e02bb2be1850860f16500c5e0e3284bce381aead2caf5f448482b99ac5e093f)
                check_type(argname="argument certificate_chain", value=certificate_chain, expected_type=type_hints["certificate_chain"])
                check_type(argname="argument private_key", value=private_key, expected_type=type_hints["private_key"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_chain": certificate_chain,
                "private_key": private_key,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate chain for the certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def private_key(self) -> builtins.str:
            '''The private key for a certificate stored on the file system of the mesh endpoint that the proxy is running on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate-privatekey
            '''
            result = self._values.get("private_key")
            assert result is not None, "Required property 'private_key' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsFileCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate": "certificate",
            "mode": "mode",
            "validation": "validation",
        },
    )
    class VirtualGatewayListenerTlsProperty:
        def __init__(
            self,
            *,
            certificate: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty", typing.Dict[builtins.str, typing.Any]]],
            mode: builtins.str,
            validation: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the Transport Layer Security (TLS) properties for a listener.

            :param certificate: An object that represents a Transport Layer Security (TLS) certificate.
            :param mode: Specify one of the following modes. - ** STRICT – Listener only accepts connections with TLS enabled. - ** PERMISSIVE – Listener accepts connections with or without TLS enabled. - ** DISABLED – Listener only accepts connections without TLS.
            :param validation: A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                    certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                        acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                            certificate_arn="certificateArn"
                        ),
                        file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    mode="mode",
                
                    # the properties below are optional
                    validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                        trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__accc5dac4de2f0974b9b2768cea2a5ff0efe0c71c24f8703687f8418adaddedc)
                check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
                check_type(argname="argument mode", value=mode, expected_type=type_hints["mode"])
                check_type(argname="argument validation", value=validation, expected_type=type_hints["validation"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate": certificate,
                "mode": mode,
            }
            if validation is not None:
                self._values["validation"] = validation

        @builtins.property
        def certificate(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty"]:
            '''An object that represents a Transport Layer Security (TLS) certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertls-certificate
            '''
            result = self._values.get("certificate")
            assert result is not None, "Required property 'certificate' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty"], result)

        @builtins.property
        def mode(self) -> builtins.str:
            '''Specify one of the following modes.

            - ** STRICT – Listener only accepts connections with TLS enabled.
            - ** PERMISSIVE – Listener accepts connections with or without TLS enabled.
            - ** DISABLED – Listener only accepts connections without TLS.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertls-mode
            '''
            result = self._values.get("mode")
            assert result is not None, "Required property 'mode' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def validation(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty"]]:
            '''A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertls-validation
            '''
            result = self._values.get("validation")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class VirtualGatewayListenerTlsSdsCertificateProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents the virtual gateway's listener's Secret Discovery Service certificate.The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlssdscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_sds_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                    secret_name="secretName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bd11e37f1734969a913bc4aa0fb74cbea803fafd4ead9f7fe03490fdd58140f3)
                check_type(argname="argument secret_name", value=secret_name, expected_type=type_hints["secret_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlssdscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlssdscertificate-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsSdsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class VirtualGatewayListenerTlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty", typing.Dict[builtins.str, typing.Any]]],
            subject_alternative_names: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a virtual gateway listener's Transport Layer Security (TLS) validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_validation_context_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                    trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e19bca33b59303f234243242bf376c77631f3ed6d11effab9059d100ecbe5cba)
                check_type(argname="argument trust", value=trust, expected_type=type_hints["trust"])
                check_type(argname="argument subject_alternative_names", value=subject_alternative_names, expected_type=type_hints["subject_alternative_names"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty"]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty"], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.SubjectAlternativeNamesProperty"]]:
            '''A reference to an object that represents the SANs for a virtual gateway listener's Transport Layer Security (TLS) validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.SubjectAlternativeNamesProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class VirtualGatewayListenerTlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            sds: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context trust.

            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_validation_context_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bf3395990475e09e1b1db88cef7dabe446dd7c175c9cdfe3115c1236916d8ee4)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty"]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty"]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty"]]:
            '''A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty",
        jsii_struct_bases=[],
        name_mapping={"access_log": "accessLog"},
    )
    class VirtualGatewayLoggingProperty:
        def __init__(
            self,
            *,
            access_log: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayAccessLogProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents logging information.

            :param access_log: The access log configuration.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylogging.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_logging_property = appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                    access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                            path="path",
                
                            # the properties below are optional
                            format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                                json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                    key="key",
                                    value="value"
                                )],
                                text="text"
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__cb63e01c23e232568a029300a7f5a324287ea1aa4d0bb5b2f57dd07abb6a2577)
                check_type(argname="argument access_log", value=access_log, expected_type=type_hints["access_log"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if access_log is not None:
                self._values["access_log"] = access_log

        @builtins.property
        def access_log(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayAccessLogProperty"]]:
            '''The access log configuration.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylogging.html#cfn-appmesh-virtualgateway-virtualgatewaylogging-accesslog
            '''
            result = self._values.get("access_log")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayAccessLogProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayLoggingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port", "protocol": "protocol"},
    )
    class VirtualGatewayPortMappingProperty:
        def __init__(self, *, port: jsii.Number, protocol: builtins.str) -> None:
            '''An object that represents a port mapping.

            :param port: The port used for the port mapping. Specify one protocol.
            :param protocol: The protocol used for the port mapping.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayportmapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_port_mapping_property = appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                    port=123,
                    protocol="protocol"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3a3242313fecdae5f0d9308deb3c26300c6ddf31836996ece3b974bcc1225e0e)
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port": port,
                "protocol": protocol,
            }

        @builtins.property
        def port(self) -> jsii.Number:
            '''The port used for the port mapping.

            Specify one protocol.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayportmapping.html#cfn-appmesh-virtualgateway-virtualgatewayportmapping-port
            '''
            result = self._values.get("port")
            assert result is not None, "Required property 'port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol used for the port mapping.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayportmapping.html#cfn-appmesh-virtualgateway-virtualgatewayportmapping-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayPortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "listeners": "listeners",
            "backend_defaults": "backendDefaults",
            "logging": "logging",
        },
    )
    class VirtualGatewaySpecProperty:
        def __init__(
            self,
            *,
            listeners: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayListenerProperty", typing.Dict[builtins.str, typing.Any]]]]],
            backend_defaults: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            logging: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayLoggingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the specification of a service mesh resource.

            :param listeners: The listeners that the mesh endpoint is expected to receive inbound traffic from. You can specify one listener.
            :param backend_defaults: A reference to an object that represents the defaults for backends.
            :param logging: An object that represents logging information.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_spec_property = appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty(
                    listeners=[appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                        port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
                
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                                max_connections=123,
                
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                                max_requests=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
                
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
                
                            # the properties below are optional
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
                
                    # the properties below are optional
                    backend_defaults=appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                            tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
                
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    logging=appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                        access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                                path="path",
                
                                # the properties below are optional
                                format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                                    json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                        key="key",
                                        value="value"
                                    )],
                                    text="text"
                                )
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f77b26dc8e193ec16a114628fe34bc9564cf9486acb4db503e9628dab046400d)
                check_type(argname="argument listeners", value=listeners, expected_type=type_hints["listeners"])
                check_type(argname="argument backend_defaults", value=backend_defaults, expected_type=type_hints["backend_defaults"])
                check_type(argname="argument logging", value=logging, expected_type=type_hints["logging"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "listeners": listeners,
            }
            if backend_defaults is not None:
                self._values["backend_defaults"] = backend_defaults
            if logging is not None:
                self._values["logging"] = logging

        @builtins.property
        def listeners(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerProperty"]]]:
            '''The listeners that the mesh endpoint is expected to receive inbound traffic from.

            You can specify one listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html#cfn-appmesh-virtualgateway-virtualgatewayspec-listeners
            '''
            result = self._values.get("listeners")
            assert result is not None, "Required property 'listeners' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayListenerProperty"]]], result)

        @builtins.property
        def backend_defaults(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty"]]:
            '''A reference to an object that represents the defaults for backends.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html#cfn-appmesh-virtualgateway-virtualgatewayspec-backenddefaults
            '''
            result = self._values.get("backend_defaults")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty"]], result)

        @builtins.property
        def logging(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayLoggingProperty"]]:
            '''An object that represents logging information.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html#cfn-appmesh-virtualgateway-virtualgatewayspec-logging
            '''
            result = self._values.get("logging")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayLoggingProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewaySpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_authority_arns": "certificateAuthorityArns"},
    )
    class VirtualGatewayTlsValidationContextAcmTrustProperty:
        def __init__(
            self,
            *,
            certificate_authority_arns: typing.Sequence[builtins.str],
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :param certificate_authority_arns: One or more ACM Amazon Resource Name (ARN)s.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextacmtrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_acm_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                    certificate_authority_arns=["certificateAuthorityArns"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bf923498c4054f83e0445bd78fc4c7a17cba72963b0d09f289ed5563b7f0219e)
                check_type(argname="argument certificate_authority_arns", value=certificate_authority_arns, expected_type=type_hints["certificate_authority_arns"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_authority_arns": certificate_authority_arns,
            }

        @builtins.property
        def certificate_authority_arns(self) -> typing.List[builtins.str]:
            '''One or more ACM Amazon Resource Name (ARN)s.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextacmtrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextacmtrust-certificateauthorityarns
            '''
            result = self._values.get("certificate_authority_arns")
            assert result is not None, "Required property 'certificate_authority_arns' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextAcmTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_chain": "certificateChain"},
    )
    class VirtualGatewayTlsValidationContextFileTrustProperty:
        def __init__(self, *, certificate_chain: builtins.str) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :param certificate_chain: The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextfiletrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_file_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                    certificate_chain="certificateChain"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7a722057f4896d18c62a6c78e7212f906f5eb5b880782e96bc2869ce1c10c2c8)
                check_type(argname="argument certificate_chain", value=certificate_chain, expected_type=type_hints["certificate_chain"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_chain": certificate_chain,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextfiletrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextfiletrust-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextFileTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class VirtualGatewayTlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty", typing.Dict[builtins.str, typing.Any]]],
            subject_alternative_names: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                    trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                        acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                            certificate_authority_arns=["certificateAuthorityArns"]
                        ),
                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4f13a52cb3d1e81dd8e40c0222b987a53529b140533f1da5c0dca22e46305e5b)
                check_type(argname="argument trust", value=trust, expected_type=type_hints["trust"])
                check_type(argname="argument subject_alternative_names", value=subject_alternative_names, expected_type=type_hints["subject_alternative_names"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty"]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty"], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.SubjectAlternativeNamesProperty"]]:
            '''A reference to an object that represents the SANs for a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.SubjectAlternativeNamesProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class VirtualGatewayTlsValidationContextSdsTrustProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents a virtual gateway's listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret for a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextsdstrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_sds_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                    secret_name="secretName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bb5597175be0fafdeb1f7c72cc5eb54a4953da4d6edb14c453b86d3ea3bc9f87)
                check_type(argname="argument secret_name", value=secret_name, expected_type=type_hints["secret_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret for a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextsdstrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextsdstrust-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextSdsTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class VirtualGatewayTlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            file: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            sds: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust.

            :param acm: A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.
            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                    acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                        certificate_authority_arns=["certificateAuthorityArns"]
                    ),
                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7f8a92829bf5962ad362b92ef20da2a501a380da3d3a6f56139c67dc618750bd)
                check_type(argname="argument acm", value=acm, expected_type=type_hints["acm"])
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty"]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty"]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty"]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty"]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty"]]:
            '''A reference to an object that represents a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualGatewayProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "mesh_owner": "meshOwner",
        "tags": "tags",
        "virtual_gateway_name": "virtualGatewayName",
    },
)
class CfnVirtualGatewayProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union[CfnVirtualGateway.VirtualGatewaySpecProperty, typing.Dict[builtins.str, typing.Any]]],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_gateway_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualGateway``.

        :param mesh_name: The name of the service mesh that the virtual gateway resides in.
        :param spec: The specifications of the virtual gateway.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual gateway to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_gateway_name: The name of the virtual gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            cfn_virtual_gateway_props = appmesh.CfnVirtualGatewayProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty(
                    listeners=[appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                        port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
            
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                                max_connections=123,
            
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                                max_requests=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
            
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
            
                            # the properties below are optional
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
            
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
            
                    # the properties below are optional
                    backend_defaults=appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                            tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
            
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
            
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    logging=appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                        access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                                path="path",
            
                                # the properties below are optional
                                format=appmesh.CfnVirtualGateway.LoggingFormatProperty(
                                    json=[appmesh.CfnVirtualGateway.JsonFormatRefProperty(
                                        key="key",
                                        value="value"
                                    )],
                                    text="text"
                                )
                            )
                        )
                    )
                ),
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                virtual_gateway_name="virtualGatewayName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e364556621eaf783ac065ab89dcdf9c90380b8d9c377982196b684136c903e68)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument virtual_gateway_name", value=virtual_gateway_name, expected_type=type_hints["virtual_gateway_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags
        if virtual_gateway_name is not None:
            self._values["virtual_gateway_name"] = virtual_gateway_name

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual gateway resides in.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnVirtualGateway.VirtualGatewaySpecProperty]:
        '''The specifications of the virtual gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnVirtualGateway.VirtualGatewaySpecProperty], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the virtual gateway to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def virtual_gateway_name(self) -> typing.Optional[builtins.str]:
        '''The name of the virtual gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-virtualgatewayname
        '''
        result = self._values.get("virtual_gateway_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnVirtualNode(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode",
):
    '''Creates a virtual node within a service mesh.

    A virtual node acts as a logical pointer to a particular task group, such as an Amazon ECS service or a Kubernetes deployment. When you create a virtual node, you can specify the service discovery information for your task group, and whether the proxy running in a task group will communicate with other proxies using Transport Layer Security (TLS).

    You define a ``listener`` for any inbound traffic that your virtual node expects. Any virtual service that your virtual node expects to communicate to is specified as a ``backend`` .

    The response metadata for your new virtual node contains the ``arn`` that is associated with the virtual node. Set this value to the full ARN; for example, ``arn:aws:appmesh:us-west-2:123456789012:myMesh/default/virtualNode/myApp`` ) as the ``APPMESH_RESOURCE_ARN`` environment variable for your task group's Envoy proxy container in your task definition or pod spec. This is then mapped to the ``node.id`` and ``node.cluster`` Envoy parameters.
    .. epigraph::

       By default, App Mesh uses the name of the resource you specified in ``APPMESH_RESOURCE_ARN`` when Envoy is referring to itself in metrics and traces. You can override this behavior by setting the ``APPMESH_RESOURCE_CLUSTER`` environment variable with your own name.

    For more information about virtual nodes, see `Virtual nodes <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html>`_ . You must be using ``1.15.0`` or later of the Envoy image when setting these variables. For more information aboutApp Mesh Envoy variables, see `Envoy image <https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html>`_ in the AWS App Mesh User Guide.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html
    :cloudformationResource: AWS::AppMesh::VirtualNode
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_appmesh as appmesh
        
        cfn_virtual_node = appmesh.CfnVirtualNode(self, "MyCfnVirtualNode",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualNode.VirtualNodeSpecProperty(
                backend_defaults=appmesh.CfnVirtualNode.BackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                        tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
        
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
        
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                ),
                backends=[appmesh.CfnVirtualNode.BackendProperty(
                    virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                        virtual_service_name="virtualServiceName",
        
                        # the properties below are optional
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
        
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
        
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    )
                )],
                listeners=[appmesh.CfnVirtualNode.ListenerProperty(
                    port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
        
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                            max_connections=123,
        
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                            max_requests=123
                        ),
                        tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                            max_connections=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
        
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                        base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        interval=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        max_ejection_percent=123,
                        max_server_errors=123
                    ),
                    timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                        grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                        certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
        
                        # the properties below are optional
                        validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
        
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )],
                logging=appmesh.CfnVirtualNode.LoggingProperty(
                    access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                        file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                            path="path",
        
                            # the properties below are optional
                            format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                                json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                    key="key",
                                    value="value"
                                )],
                                text="text"
                            )
                        )
                    )
                ),
                service_discovery=appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                    aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                        namespace_name="namespaceName",
                        service_name="serviceName",
        
                        # the properties below are optional
                        attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                            key="key",
                            value="value"
                        )],
                        ip_preference="ipPreference"
                    ),
                    dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                        hostname="hostname",
        
                        # the properties below are optional
                        ip_preference="ipPreference",
                        response_type="responseType"
                    )
                )
            ),
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            virtual_node_name="virtualNodeName"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.VirtualNodeSpecProperty", typing.Dict[builtins.str, typing.Any]]],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_node_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param mesh_name: The name of the service mesh to create the virtual node in.
        :param spec: The virtual node specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual node to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_node_name: The name to use for the virtual node.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f2cbb507210b6ffe5823dff6a0501f7d3d5555d3134a25419e94d913c9a088b7)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVirtualNodeProps(
            mesh_name=mesh_name,
            spec=spec,
            mesh_owner=mesh_owner,
            tags=tags,
            virtual_node_name=virtual_node_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b53c6e82be19364b4c8777b10c4e042a093e871ad343d52cc18716877ed492f1)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f455f7f304ea99533c488190dfca3183b5bc53fc4e969888bce4f9839ffa80a0)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual node.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual node resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual node.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualNodeName")
    def attr_virtual_node_name(self) -> builtins.str:
        '''The name of the virtual node.

        :cloudformationAttribute: VirtualNodeName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualNodeName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual node in.'''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4612e75fd582e25acb6aa15c400f597d444d3fbbd5e026c821a83752d19aa40e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeSpecProperty"]:
        '''The virtual node specification to apply.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeSpecProperty"], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeSpecProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__69101b3af439f3968821270efdfaa2b33a95944f88e8ef8ffbce3a7e03334fb6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a102121cc018a2f54ca107d92f2a8e95627cf806d28d0b90d9485345d5ba401a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the virtual node to assist with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__10d8ee0411eb92c3301d69fbdb5b3d0fa2159134bb2831cfbf085d8334c0f3c3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="virtualNodeName")
    def virtual_node_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual node.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualNodeName"))

    @virtual_node_name.setter
    def virtual_node_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__862ebc3aef73ce8dba12bf14d6438454692ff50b207afffecb576feaadc463b9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualNodeName", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.AccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file"},
    )
    class AccessLogProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.FileAccessLogProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the access logging information for a virtual node.

            :param file: The file object to send virtual node access logs to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-accesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                access_log_property = appmesh.CfnVirtualNode.AccessLogProperty(
                    file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                        path="path",
                
                        # the properties below are optional
                        format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                            json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                key="key",
                                value="value"
                            )],
                            text="text"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__48ece5afcf317908f2227bb8909c1b550407fc5a880f033ba0b46bc82fda6d07)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.FileAccessLogProperty"]]:
            '''The file object to send virtual node access logs to.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-accesslog.html#cfn-appmesh-virtualnode-accesslog-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.FileAccessLogProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class AwsCloudMapInstanceAttributeProperty:
        def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
            '''An object that represents the AWS Cloud Map attribute information for your virtual node.

            .. epigraph::

               AWS Cloud Map is not available in the eu-south-1 Region.

            :param key: The name of an AWS Cloud Map service instance attribute key. Any AWS Cloud Map service instance that contains the specified key and value is returned.
            :param value: The value of an AWS Cloud Map service instance attribute key. Any AWS Cloud Map service instance that contains the specified key and value is returned.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapinstanceattribute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                aws_cloud_map_instance_attribute_property = appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__073b64b0a4d057fe469cd698a12a2ac0efb7a2a91c0e97fde9bb8c5140358d14)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''The name of an AWS Cloud Map service instance attribute key.

            Any AWS Cloud Map service instance that contains the specified key and value is returned.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapinstanceattribute.html#cfn-appmesh-virtualnode-awscloudmapinstanceattribute-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The value of an AWS Cloud Map service instance attribute key.

            Any AWS Cloud Map service instance that contains the specified key and value is returned.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapinstanceattribute.html#cfn-appmesh-virtualnode-awscloudmapinstanceattribute-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AwsCloudMapInstanceAttributeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={
            "namespace_name": "namespaceName",
            "service_name": "serviceName",
            "attributes": "attributes",
            "ip_preference": "ipPreference",
        },
    )
    class AwsCloudMapServiceDiscoveryProperty:
        def __init__(
            self,
            *,
            namespace_name: builtins.str,
            service_name: builtins.str,
            attributes: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.AwsCloudMapInstanceAttributeProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            ip_preference: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the AWS Cloud Map service discovery information for your virtual node.

            .. epigraph::

               AWS Cloud Map is not available in the eu-south-1 Region.

            :param namespace_name: The HTTP name of the AWS Cloud Map namespace to use.
            :param service_name: The name of the AWS Cloud Map service to use.
            :param attributes: A string map that contains attributes with values that you can use to filter instances by any custom attribute that you specified when you registered the instance. Only instances that match all of the specified key/value pairs will be returned.
            :param ip_preference: The preferred IP version that this virtual node uses. Setting the IP preference on the virtual node only overrides the IP preference set for the mesh on this specific node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                aws_cloud_map_service_discovery_property = appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                    namespace_name="namespaceName",
                    service_name="serviceName",
                
                    # the properties below are optional
                    attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                        key="key",
                        value="value"
                    )],
                    ip_preference="ipPreference"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__aef4de7e5bdb97a1d973260fac6f38774da182174be69c1313d36563fb3ce3e9)
                check_type(argname="argument namespace_name", value=namespace_name, expected_type=type_hints["namespace_name"])
                check_type(argname="argument service_name", value=service_name, expected_type=type_hints["service_name"])
                check_type(argname="argument attributes", value=attributes, expected_type=type_hints["attributes"])
                check_type(argname="argument ip_preference", value=ip_preference, expected_type=type_hints["ip_preference"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "namespace_name": namespace_name,
                "service_name": service_name,
            }
            if attributes is not None:
                self._values["attributes"] = attributes
            if ip_preference is not None:
                self._values["ip_preference"] = ip_preference

        @builtins.property
        def namespace_name(self) -> builtins.str:
            '''The HTTP name of the AWS Cloud Map namespace to use.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html#cfn-appmesh-virtualnode-awscloudmapservicediscovery-namespacename
            '''
            result = self._values.get("namespace_name")
            assert result is not None, "Required property 'namespace_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def service_name(self) -> builtins.str:
            '''The name of the AWS Cloud Map service to use.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html#cfn-appmesh-virtualnode-awscloudmapservicediscovery-servicename
            '''
            result = self._values.get("service_name")
            assert result is not None, "Required property 'service_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def attributes(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.AwsCloudMapInstanceAttributeProperty"]]]]:
            '''A string map that contains attributes with values that you can use to filter instances by any custom attribute that you specified when you registered the instance.

            Only instances that match all of the specified key/value pairs will be returned.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html#cfn-appmesh-virtualnode-awscloudmapservicediscovery-attributes
            '''
            result = self._values.get("attributes")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.AwsCloudMapInstanceAttributeProperty"]]]], result)

        @builtins.property
        def ip_preference(self) -> typing.Optional[builtins.str]:
            '''The preferred IP version that this virtual node uses.

            Setting the IP preference on the virtual node only overrides the IP preference set for the mesh on this specific node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html#cfn-appmesh-virtualnode-awscloudmapservicediscovery-ippreference
            '''
            result = self._values.get("ip_preference")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AwsCloudMapServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.BackendDefaultsProperty",
        jsii_struct_bases=[],
        name_mapping={"client_policy": "clientPolicy"},
    )
    class BackendDefaultsProperty:
        def __init__(
            self,
            *,
            client_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ClientPolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the default properties for a backend.

            :param client_policy: A reference to an object that represents a client policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backenddefaults.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                backend_defaults_property = appmesh.CfnVirtualNode.BackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                        tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
                
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f18b663f8b68f397fa7f68a924af4ab6e10aad1cdfb05df4eff98888056ccf05)
                check_type(argname="argument client_policy", value=client_policy, expected_type=type_hints["client_policy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if client_policy is not None:
                self._values["client_policy"] = client_policy

        @builtins.property
        def client_policy(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ClientPolicyProperty"]]:
            '''A reference to an object that represents a client policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backenddefaults.html#cfn-appmesh-virtualnode-backenddefaults-clientpolicy
            '''
            result = self._values.get("client_policy")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ClientPolicyProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BackendDefaultsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.BackendProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_service": "virtualService"},
    )
    class BackendProperty:
        def __init__(
            self,
            *,
            virtual_service: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.VirtualServiceBackendProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the backends that a virtual node is expected to send outbound traffic to.

            :param virtual_service: Specifies a virtual service to use as a backend.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backend.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                backend_property = appmesh.CfnVirtualNode.BackendProperty(
                    virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                        virtual_service_name="virtualServiceName",
                
                        # the properties below are optional
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
                
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e8078ecabbccda4f7ccf7f81a6e874c4064894ab4b4f3474cac7f07ba8e379b4)
                check_type(argname="argument virtual_service", value=virtual_service, expected_type=type_hints["virtual_service"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if virtual_service is not None:
                self._values["virtual_service"] = virtual_service

        @builtins.property
        def virtual_service(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualServiceBackendProperty"]]:
            '''Specifies a virtual service to use as a backend.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backend.html#cfn-appmesh-virtualnode-backend-virtualservice
            '''
            result = self._values.get("virtual_service")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualServiceBackendProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BackendProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ClientPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"tls": "tls"},
    )
    class ClientPolicyProperty:
        def __init__(
            self,
            *,
            tls: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ClientPolicyTlsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a client policy.

            :param tls: A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                client_policy_property = appmesh.CfnVirtualNode.ClientPolicyProperty(
                    tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                        validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                            trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                    certificate_authority_arns=["certificateAuthorityArns"]
                                ),
                                file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        ),
                
                        # the properties below are optional
                        certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                            file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        enforce=False,
                        ports=[123]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__64dc1607ab368cc2595804d5ce65ca14f58b07e2176d177672030f42c07a6644)
                check_type(argname="argument tls", value=tls, expected_type=type_hints["tls"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ClientPolicyTlsProperty"]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicy.html#cfn-appmesh-virtualnode-clientpolicy-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ClientPolicyTlsProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ClientPolicyTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "validation": "validation",
            "certificate": "certificate",
            "enforce": "enforce",
            "ports": "ports",
        },
    )
    class ClientPolicyTlsProperty:
        def __init__(
            self,
            *,
            validation: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.TlsValidationContextProperty", typing.Dict[builtins.str, typing.Any]]],
            certificate: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ClientTlsCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            enforce: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            ports: typing.Optional[typing.Union[typing.Sequence[jsii.Number], _IResolvable_da3f097b]] = None,
        ) -> None:
            '''A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :param validation: A reference to an object that represents a TLS validation context.
            :param certificate: A reference to an object that represents a client's TLS certificate.
            :param enforce: Whether the policy is enforced. The default is ``True`` , if a value isn't specified.
            :param ports: One or more ports that the policy is enforced for.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                client_policy_tls_property = appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                    validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                        trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                            acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                certificate_authority_arns=["certificateAuthorityArns"]
                            ),
                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    ),
                
                    # the properties below are optional
                    certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    enforce=False,
                    ports=[123]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a8ef96a5ee4db487464a166902d82b19fdd3c7a628774ca34dd912d836d30774)
                check_type(argname="argument validation", value=validation, expected_type=type_hints["validation"])
                check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
                check_type(argname="argument enforce", value=enforce, expected_type=type_hints["enforce"])
                check_type(argname="argument ports", value=ports, expected_type=type_hints["ports"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "validation": validation,
            }
            if certificate is not None:
                self._values["certificate"] = certificate
            if enforce is not None:
                self._values["enforce"] = enforce
            if ports is not None:
                self._values["ports"] = ports

        @builtins.property
        def validation(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextProperty"]:
            '''A reference to an object that represents a TLS validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-validation
            '''
            result = self._values.get("validation")
            assert result is not None, "Required property 'validation' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextProperty"], result)

        @builtins.property
        def certificate(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ClientTlsCertificateProperty"]]:
            '''A reference to an object that represents a client's TLS certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-certificate
            '''
            result = self._values.get("certificate")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ClientTlsCertificateProperty"]], result)

        @builtins.property
        def enforce(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Whether the policy is enforced.

            The default is ``True`` , if a value isn't specified.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-enforce
            '''
            result = self._values.get("enforce")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def ports(
            self,
        ) -> typing.Optional[typing.Union[typing.List[jsii.Number], _IResolvable_da3f097b]]:
            '''One or more ports that the policy is enforced for.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-ports
            '''
            result = self._values.get("ports")
            return typing.cast(typing.Optional[typing.Union[typing.List[jsii.Number], _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientPolicyTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ClientTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class ClientTlsCertificateProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            sds: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the client's certificate.

            :param file: An object that represents a local file certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .
            :param sds: A reference to an object that represents a client's TLS Secret Discovery Service certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clienttlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                client_tls_certificate_property = appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__70c5ac274c3ef05077548f2a4090504add7dea214c75f1a3137450bf04736ff9)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsFileCertificateProperty"]]:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clienttlscertificate.html#cfn-appmesh-virtualnode-clienttlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsFileCertificateProperty"]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsSdsCertificateProperty"]]:
            '''A reference to an object that represents a client's TLS Secret Discovery Service certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clienttlscertificate.html#cfn-appmesh-virtualnode-clienttlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsSdsCertificateProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={
            "hostname": "hostname",
            "ip_preference": "ipPreference",
            "response_type": "responseType",
        },
    )
    class DnsServiceDiscoveryProperty:
        def __init__(
            self,
            *,
            hostname: builtins.str,
            ip_preference: typing.Optional[builtins.str] = None,
            response_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the DNS service discovery information for your virtual node.

            :param hostname: Specifies the DNS service discovery hostname for the virtual node.
            :param ip_preference: The preferred IP version that this virtual node uses. Setting the IP preference on the virtual node only overrides the IP preference set for the mesh on this specific node.
            :param response_type: Specifies the DNS response type for the virtual node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-dnsservicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                dns_service_discovery_property = appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                    hostname="hostname",
                
                    # the properties below are optional
                    ip_preference="ipPreference",
                    response_type="responseType"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2eff157008589e4321bef2edf17f6f80cde07906feaee21fa4965543ea11bd36)
                check_type(argname="argument hostname", value=hostname, expected_type=type_hints["hostname"])
                check_type(argname="argument ip_preference", value=ip_preference, expected_type=type_hints["ip_preference"])
                check_type(argname="argument response_type", value=response_type, expected_type=type_hints["response_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "hostname": hostname,
            }
            if ip_preference is not None:
                self._values["ip_preference"] = ip_preference
            if response_type is not None:
                self._values["response_type"] = response_type

        @builtins.property
        def hostname(self) -> builtins.str:
            '''Specifies the DNS service discovery hostname for the virtual node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-dnsservicediscovery.html#cfn-appmesh-virtualnode-dnsservicediscovery-hostname
            '''
            result = self._values.get("hostname")
            assert result is not None, "Required property 'hostname' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def ip_preference(self) -> typing.Optional[builtins.str]:
            '''The preferred IP version that this virtual node uses.

            Setting the IP preference on the virtual node only overrides the IP preference set for the mesh on this specific node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-dnsservicediscovery.html#cfn-appmesh-virtualnode-dnsservicediscovery-ippreference
            '''
            result = self._values.get("ip_preference")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def response_type(self) -> typing.Optional[builtins.str]:
            '''Specifies the DNS response type for the virtual node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-dnsservicediscovery.html#cfn-appmesh-virtualnode-dnsservicediscovery-responsetype
            '''
            result = self._values.get("response_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DnsServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.DurationProperty",
        jsii_struct_bases=[],
        name_mapping={"unit": "unit", "value": "value"},
    )
    class DurationProperty:
        def __init__(self, *, unit: builtins.str, value: jsii.Number) -> None:
            '''An object that represents a duration of time.

            :param unit: A unit of time.
            :param value: A number of time units.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-duration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                duration_property = appmesh.CfnVirtualNode.DurationProperty(
                    unit="unit",
                    value=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f13e53cf7e38ad0a2325396c846e465182fc9d253d7b24a638f37b9f06659a90)
                check_type(argname="argument unit", value=unit, expected_type=type_hints["unit"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "unit": unit,
                "value": value,
            }

        @builtins.property
        def unit(self) -> builtins.str:
            '''A unit of time.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-duration.html#cfn-appmesh-virtualnode-duration-unit
            '''
            result = self._values.get("unit")
            assert result is not None, "Required property 'unit' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> jsii.Number:
            '''A number of time units.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-duration.html#cfn-appmesh-virtualnode-duration-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.FileAccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"path": "path", "format": "format"},
    )
    class FileAccessLogProperty:
        def __init__(
            self,
            *,
            path: builtins.str,
            format: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.LoggingFormatProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents an access log file.

            :param path: The file path to write access logs to. You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk. .. epigraph:: The Envoy process must have write permissions to the path that you specify here. Otherwise, Envoy fails to bootstrap properly.
            :param format: The specified format for the logs. The format is either ``json_format`` or ``text_format`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-fileaccesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                file_access_log_property = appmesh.CfnVirtualNode.FileAccessLogProperty(
                    path="path",
                
                    # the properties below are optional
                    format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                        json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                            key="key",
                            value="value"
                        )],
                        text="text"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ab1ec160ebf835e454290e857bd79520b18470bf533de28f80fb1112b0d82cc0)
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument format", value=format, expected_type=type_hints["format"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "path": path,
            }
            if format is not None:
                self._values["format"] = format

        @builtins.property
        def path(self) -> builtins.str:
            '''The file path to write access logs to.

            You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk.
            .. epigraph::

               The Envoy process must have write permissions to the path that you specify here. Otherwise, Envoy fails to bootstrap properly.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-fileaccesslog.html#cfn-appmesh-virtualnode-fileaccesslog-path
            '''
            result = self._values.get("path")
            assert result is not None, "Required property 'path' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def format(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.LoggingFormatProperty"]]:
            '''The specified format for the logs.

            The format is either ``json_format`` or ``text_format`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-fileaccesslog.html#cfn-appmesh-virtualnode-fileaccesslog-format
            '''
            result = self._values.get("format")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.LoggingFormatProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FileAccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.GrpcTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class GrpcTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            per_request: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-grpctimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                grpc_timeout_property = appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                    idle=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__54f61adb59c8093769ace4c8bc781d833c21f21cec94064325412f6b9cc663f6)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
                check_type(argname="argument per_request", value=per_request, expected_type=type_hints["per_request"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-grpctimeout.html#cfn-appmesh-virtualnode-grpctimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-grpctimeout.html#cfn-appmesh-virtualnode-grpctimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.HealthCheckProperty",
        jsii_struct_bases=[],
        name_mapping={
            "healthy_threshold": "healthyThreshold",
            "interval_millis": "intervalMillis",
            "protocol": "protocol",
            "timeout_millis": "timeoutMillis",
            "unhealthy_threshold": "unhealthyThreshold",
            "path": "path",
            "port": "port",
        },
    )
    class HealthCheckProperty:
        def __init__(
            self,
            *,
            healthy_threshold: jsii.Number,
            interval_millis: jsii.Number,
            protocol: builtins.str,
            timeout_millis: jsii.Number,
            unhealthy_threshold: jsii.Number,
            path: typing.Optional[builtins.str] = None,
            port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents the health check policy for a virtual node's listener.

            :param healthy_threshold: The number of consecutive successful health checks that must occur before declaring listener healthy.
            :param interval_millis: The time period in milliseconds between each health check execution.
            :param protocol: The protocol for the health check request. If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .
            :param timeout_millis: The amount of time to wait when receiving a response from the health check, in milliseconds.
            :param unhealthy_threshold: The number of consecutive failed health checks that must occur before declaring a virtual node unhealthy.
            :param path: The destination path for the health check request. This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.
            :param port: The destination port for the health check request. This port must match the port defined in the ``PortMapping`` for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                health_check_property = appmesh.CfnVirtualNode.HealthCheckProperty(
                    healthy_threshold=123,
                    interval_millis=123,
                    protocol="protocol",
                    timeout_millis=123,
                    unhealthy_threshold=123,
                
                    # the properties below are optional
                    path="path",
                    port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6d09af4f2ae65baa63537eb347f1c23ac93509cb5f43d2901aa683f5be6a08c9)
                check_type(argname="argument healthy_threshold", value=healthy_threshold, expected_type=type_hints["healthy_threshold"])
                check_type(argname="argument interval_millis", value=interval_millis, expected_type=type_hints["interval_millis"])
                check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
                check_type(argname="argument timeout_millis", value=timeout_millis, expected_type=type_hints["timeout_millis"])
                check_type(argname="argument unhealthy_threshold", value=unhealthy_threshold, expected_type=type_hints["unhealthy_threshold"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "healthy_threshold": healthy_threshold,
                "interval_millis": interval_millis,
                "protocol": protocol,
                "timeout_millis": timeout_millis,
                "unhealthy_threshold": unhealthy_threshold,
            }
            if path is not None:
                self._values["path"] = path
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def healthy_threshold(self) -> jsii.Number:
            '''The number of consecutive successful health checks that must occur before declaring listener healthy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-healthythreshold
            '''
            result = self._values.get("healthy_threshold")
            assert result is not None, "Required property 'healthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def interval_millis(self) -> jsii.Number:
            '''The time period in milliseconds between each health check execution.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-intervalmillis
            '''
            result = self._values.get("interval_millis")
            assert result is not None, "Required property 'interval_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol for the health check request.

            If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def timeout_millis(self) -> jsii.Number:
            '''The amount of time to wait when receiving a response from the health check, in milliseconds.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-timeoutmillis
            '''
            result = self._values.get("timeout_millis")
            assert result is not None, "Required property 'timeout_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def unhealthy_threshold(self) -> jsii.Number:
            '''The number of consecutive failed health checks that must occur before declaring a virtual node unhealthy.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-unhealthythreshold
            '''
            result = self._values.get("unhealthy_threshold")
            assert result is not None, "Required property 'unhealthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def path(self) -> typing.Optional[builtins.str]:
            '''The destination path for the health check request.

            This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The destination port for the health check request.

            This port must match the port defined in the ``PortMapping`` for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HealthCheckProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.HttpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class HttpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            per_request: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-httptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                http_timeout_property = appmesh.CfnVirtualNode.HttpTimeoutProperty(
                    idle=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9632054602b8b6e4e49fa8b2c346d6280e0d222c28e770c03b0dede6a759ceb6)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
                check_type(argname="argument per_request", value=per_request, expected_type=type_hints["per_request"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-httptimeout.html#cfn-appmesh-virtualnode-httptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-httptimeout.html#cfn-appmesh-virtualnode-httptimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.JsonFormatRefProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class JsonFormatRefProperty:
        def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
            '''An object that represents the key value pairs for the JSON.

            :param key: The specified key for the JSON.
            :param value: The specified value for the JSON.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-jsonformatref.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                json_format_ref_property = appmesh.CfnVirtualNode.JsonFormatRefProperty(
                    key="key",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0b8d0712b00c5797442db3ad7cb32fc0b2be969d7cf2c84126c14b54b5881663)
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''The specified key for the JSON.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-jsonformatref.html#cfn-appmesh-virtualnode-jsonformatref-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The specified value for the JSON.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-jsonformatref.html#cfn-appmesh-virtualnode-jsonformatref-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "JsonFormatRefProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ListenerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "port_mapping": "portMapping",
            "connection_pool": "connectionPool",
            "health_check": "healthCheck",
            "outlier_detection": "outlierDetection",
            "timeout": "timeout",
            "tls": "tls",
        },
    )
    class ListenerProperty:
        def __init__(
            self,
            *,
            port_mapping: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.PortMappingProperty", typing.Dict[builtins.str, typing.Any]]],
            connection_pool: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.VirtualNodeConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            health_check: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.HealthCheckProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            outlier_detection: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.OutlierDetectionProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            timeout: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ListenerTimeoutProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            tls: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ListenerTlsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a listener for a virtual node.

            :param port_mapping: The port mapping information for the listener.
            :param connection_pool: The connection pool information for the listener.
            :param health_check: The health check information for the listener.
            :param outlier_detection: The outlier detection information for the listener.
            :param timeout: An object that represents timeouts for different protocols.
            :param tls: A reference to an object that represents the Transport Layer Security (TLS) properties for a listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                listener_property = appmesh.CfnVirtualNode.ListenerProperty(
                    port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
                
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                            max_connections=123,
                
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                            max_requests=123
                        ),
                        tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                            max_connections=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
                
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                        base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        interval=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        max_ejection_percent=123,
                        max_server_errors=123
                    ),
                    timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                        grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                        certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
                
                        # the properties below are optional
                        validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__071b2d987df2011e0b52d7395f1806ffd2327c7064fc57b135cb9e19b1065289)
                check_type(argname="argument port_mapping", value=port_mapping, expected_type=type_hints["port_mapping"])
                check_type(argname="argument connection_pool", value=connection_pool, expected_type=type_hints["connection_pool"])
                check_type(argname="argument health_check", value=health_check, expected_type=type_hints["health_check"])
                check_type(argname="argument outlier_detection", value=outlier_detection, expected_type=type_hints["outlier_detection"])
                check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
                check_type(argname="argument tls", value=tls, expected_type=type_hints["tls"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port_mapping": port_mapping,
            }
            if connection_pool is not None:
                self._values["connection_pool"] = connection_pool
            if health_check is not None:
                self._values["health_check"] = health_check
            if outlier_detection is not None:
                self._values["outlier_detection"] = outlier_detection
            if timeout is not None:
                self._values["timeout"] = timeout
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def port_mapping(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.PortMappingProperty"]:
            '''The port mapping information for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-portmapping
            '''
            result = self._values.get("port_mapping")
            assert result is not None, "Required property 'port_mapping' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.PortMappingProperty"], result)

        @builtins.property
        def connection_pool(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeConnectionPoolProperty"]]:
            '''The connection pool information for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-connectionpool
            '''
            result = self._values.get("connection_pool")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeConnectionPoolProperty"]], result)

        @builtins.property
        def health_check(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.HealthCheckProperty"]]:
            '''The health check information for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-healthcheck
            '''
            result = self._values.get("health_check")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.HealthCheckProperty"]], result)

        @builtins.property
        def outlier_detection(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.OutlierDetectionProperty"]]:
            '''The outlier detection information for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-outlierdetection
            '''
            result = self._values.get("outlier_detection")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.OutlierDetectionProperty"]], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTimeoutProperty"]]:
            '''An object that represents timeouts for different protocols.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTimeoutProperty"]], result)

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsProperty"]]:
            '''A reference to an object that represents the Transport Layer Security (TLS) properties for a listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ListenerTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"grpc": "grpc", "http": "http", "http2": "http2", "tcp": "tcp"},
    )
    class ListenerTimeoutProperty:
        def __init__(
            self,
            *,
            grpc: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.GrpcTimeoutProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            http: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.HttpTimeoutProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            http2: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.HttpTimeoutProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            tcp: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.TcpTimeoutProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents timeouts for different protocols.

            :param grpc: An object that represents types of timeouts.
            :param http: An object that represents types of timeouts.
            :param http2: An object that represents types of timeouts.
            :param tcp: An object that represents types of timeouts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                listener_timeout_property = appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                    grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    ),
                    http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    ),
                    http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    ),
                    tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6983190095928b6ff6c039006bf0aaf59198f7ccc20a1a5847648de2a0db0b06)
                check_type(argname="argument grpc", value=grpc, expected_type=type_hints["grpc"])
                check_type(argname="argument http", value=http, expected_type=type_hints["http"])
                check_type(argname="argument http2", value=http2, expected_type=type_hints["http2"])
                check_type(argname="argument tcp", value=tcp, expected_type=type_hints["tcp"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if grpc is not None:
                self._values["grpc"] = grpc
            if http is not None:
                self._values["http"] = http
            if http2 is not None:
                self._values["http2"] = http2
            if tcp is not None:
                self._values["tcp"] = tcp

        @builtins.property
        def grpc(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.GrpcTimeoutProperty"]]:
            '''An object that represents types of timeouts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-grpc
            '''
            result = self._values.get("grpc")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.GrpcTimeoutProperty"]], result)

        @builtins.property
        def http(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.HttpTimeoutProperty"]]:
            '''An object that represents types of timeouts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-http
            '''
            result = self._values.get("http")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.HttpTimeoutProperty"]], result)

        @builtins.property
        def http2(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.HttpTimeoutProperty"]]:
            '''An object that represents types of timeouts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-http2
            '''
            result = self._values.get("http2")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.HttpTimeoutProperty"]], result)

        @builtins.property
        def tcp(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TcpTimeoutProperty"]]:
            '''An object that represents types of timeouts.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-tcp
            '''
            result = self._values.get("tcp")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TcpTimeoutProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_arn": "certificateArn"},
    )
    class ListenerTlsAcmCertificateProperty:
        def __init__(self, *, certificate_arn: builtins.str) -> None:
            '''An object that represents an AWS Certificate Manager certificate.

            :param certificate_arn: The Amazon Resource Name (ARN) for the certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsacmcertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                listener_tls_acm_certificate_property = appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                    certificate_arn="certificateArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ff16db68cd166bb39d70efa410a7357bac9079801f063c8923282064e0639bde)
                check_type(argname="argument certificate_arn", value=certificate_arn, expected_type=type_hints["certificate_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_arn": certificate_arn,
            }

        @builtins.property
        def certificate_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) for the certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsacmcertificate.html#cfn-appmesh-virtualnode-listenertlsacmcertificate-certificatearn
            '''
            result = self._values.get("certificate_arn")
            assert result is not None, "Required property 'certificate_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsAcmCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ListenerTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class ListenerTlsCertificateProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ListenerTlsAcmCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            file: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            sds: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) certificate.

            :param acm: A reference to an object that represents an AWS Certificate Manager certificate.
            :param file: A reference to an object that represents a local file certificate.
            :param sds: A reference to an object that represents a listener's Secret Discovery Service certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                listener_tls_certificate_property = appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                    acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                        certificate_arn="certificateArn"
                    ),
                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__468b35cfbe2973c8f84728f65322ef3f6b9a5f349c11c690b89b33a9d7a7b04f)
                check_type(argname="argument acm", value=acm, expected_type=type_hints["acm"])
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsAcmCertificateProperty"]]:
            '''A reference to an object that represents an AWS Certificate Manager certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html#cfn-appmesh-virtualnode-listenertlscertificate-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsAcmCertificateProperty"]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsFileCertificateProperty"]]:
            '''A reference to an object that represents a local file certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html#cfn-appmesh-virtualnode-listenertlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsFileCertificateProperty"]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsSdsCertificateProperty"]]:
            '''A reference to an object that represents a listener's Secret Discovery Service certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html#cfn-appmesh-virtualnode-listenertlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsSdsCertificateProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate_chain": "certificateChain",
            "private_key": "privateKey",
        },
    )
    class ListenerTlsFileCertificateProperty:
        def __init__(
            self,
            *,
            certificate_chain: builtins.str,
            private_key: builtins.str,
        ) -> None:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :param certificate_chain: The certificate chain for the certificate.
            :param private_key: The private key for a certificate stored on the file system of the virtual node that the proxy is running on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsfilecertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                listener_tls_file_certificate_property = appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                    certificate_chain="certificateChain",
                    private_key="privateKey"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a14d8463583263a77e1a5caa4d040280c992cf15480d4512df515d1d6eed80b6)
                check_type(argname="argument certificate_chain", value=certificate_chain, expected_type=type_hints["certificate_chain"])
                check_type(argname="argument private_key", value=private_key, expected_type=type_hints["private_key"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_chain": certificate_chain,
                "private_key": private_key,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate chain for the certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsfilecertificate.html#cfn-appmesh-virtualnode-listenertlsfilecertificate-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def private_key(self) -> builtins.str:
            '''The private key for a certificate stored on the file system of the virtual node that the proxy is running on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsfilecertificate.html#cfn-appmesh-virtualnode-listenertlsfilecertificate-privatekey
            '''
            result = self._values.get("private_key")
            assert result is not None, "Required property 'private_key' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsFileCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ListenerTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate": "certificate",
            "mode": "mode",
            "validation": "validation",
        },
    )
    class ListenerTlsProperty:
        def __init__(
            self,
            *,
            certificate: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ListenerTlsCertificateProperty", typing.Dict[builtins.str, typing.Any]]],
            mode: builtins.str,
            validation: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ListenerTlsValidationContextProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the Transport Layer Security (TLS) properties for a listener.

            :param certificate: A reference to an object that represents a listener's Transport Layer Security (TLS) certificate.
            :param mode: Specify one of the following modes. - ** STRICT – Listener only accepts connections with TLS enabled. - ** PERMISSIVE – Listener accepts connections with or without TLS enabled. - ** DISABLED – Listener only accepts connections without TLS.
            :param validation: A reference to an object that represents a listener's Transport Layer Security (TLS) validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                listener_tls_property = appmesh.CfnVirtualNode.ListenerTlsProperty(
                    certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                        acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                            certificate_arn="certificateArn"
                        ),
                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    mode="mode",
                
                    # the properties below are optional
                    validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                        trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__380dd2cc62ce2506a4ce22683034e088a9339ff3431a5322c7ee5ea26c72c904)
                check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
                check_type(argname="argument mode", value=mode, expected_type=type_hints["mode"])
                check_type(argname="argument validation", value=validation, expected_type=type_hints["validation"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate": certificate,
                "mode": mode,
            }
            if validation is not None:
                self._values["validation"] = validation

        @builtins.property
        def certificate(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsCertificateProperty"]:
            '''A reference to an object that represents a listener's Transport Layer Security (TLS) certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html#cfn-appmesh-virtualnode-listenertls-certificate
            '''
            result = self._values.get("certificate")
            assert result is not None, "Required property 'certificate' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsCertificateProperty"], result)

        @builtins.property
        def mode(self) -> builtins.str:
            '''Specify one of the following modes.

            - ** STRICT – Listener only accepts connections with TLS enabled.
            - ** PERMISSIVE – Listener accepts connections with or without TLS enabled.
            - ** DISABLED – Listener only accepts connections without TLS.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html#cfn-appmesh-virtualnode-listenertls-mode
            '''
            result = self._values.get("mode")
            assert result is not None, "Required property 'mode' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def validation(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsValidationContextProperty"]]:
            '''A reference to an object that represents a listener's Transport Layer Security (TLS) validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html#cfn-appmesh-virtualnode-listenertls-validation
            '''
            result = self._values.get("validation")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsValidationContextProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class ListenerTlsSdsCertificateProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents the listener's Secret Discovery Service certificate.

            The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlssdscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                listener_tls_sds_certificate_property = appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                    secret_name="secretName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__34dac4919f1c7ef3f1d931783388c00eeac8f2daabbb081f95db754d028054ed)
                check_type(argname="argument secret_name", value=secret_name, expected_type=type_hints["secret_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlssdscertificate.html#cfn-appmesh-virtualnode-listenertlssdscertificate-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsSdsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class ListenerTlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ListenerTlsValidationContextTrustProperty", typing.Dict[builtins.str, typing.Any]]],
            subject_alternative_names: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) validation context.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a listener's Transport Layer Security (TLS) validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                listener_tls_validation_context_property = appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                    trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c1c89aca59e2777e3d3469b95aeac311c3170672f1d4ae8d6705e89744438617)
                check_type(argname="argument trust", value=trust, expected_type=type_hints["trust"])
                check_type(argname="argument subject_alternative_names", value=subject_alternative_names, expected_type=type_hints["subject_alternative_names"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsValidationContextTrustProperty"]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontext.html#cfn-appmesh-virtualnode-listenertlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerTlsValidationContextTrustProperty"], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.SubjectAlternativeNamesProperty"]]:
            '''A reference to an object that represents the SANs for a listener's Transport Layer Security (TLS) validation context.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontext.html#cfn-appmesh-virtualnode-listenertlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.SubjectAlternativeNamesProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class ListenerTlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            sds: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) validation context trust.

            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                listener_tls_validation_context_trust_property = appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__242366612bb471e48fd37c5dadedaa724efd6f2cb8b0d8b04869957ddebc5c60)
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextFileTrustProperty"]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-listenertlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextFileTrustProperty"]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextSdsTrustProperty"]]:
            '''A reference to an object that represents a listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-listenertlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextSdsTrustProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.LoggingFormatProperty",
        jsii_struct_bases=[],
        name_mapping={"json": "json", "text": "text"},
    )
    class LoggingFormatProperty:
        def __init__(
            self,
            *,
            json: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.JsonFormatRefProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            text: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the format for the logs.

            :param json: The logging format for JSON.
            :param text: The logging format for text.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-loggingformat.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                logging_format_property = appmesh.CfnVirtualNode.LoggingFormatProperty(
                    json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                        key="key",
                        value="value"
                    )],
                    text="text"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1ca55abbdbabae6f15495794cc9b2449117fca5a790d10fae40d30a3160d1259)
                check_type(argname="argument json", value=json, expected_type=type_hints["json"])
                check_type(argname="argument text", value=text, expected_type=type_hints["text"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if json is not None:
                self._values["json"] = json
            if text is not None:
                self._values["text"] = text

        @builtins.property
        def json(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.JsonFormatRefProperty"]]]]:
            '''The logging format for JSON.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-loggingformat.html#cfn-appmesh-virtualnode-loggingformat-json
            '''
            result = self._values.get("json")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.JsonFormatRefProperty"]]]], result)

        @builtins.property
        def text(self) -> typing.Optional[builtins.str]:
            '''The logging format for text.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-loggingformat.html#cfn-appmesh-virtualnode-loggingformat-text
            '''
            result = self._values.get("text")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingFormatProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.LoggingProperty",
        jsii_struct_bases=[],
        name_mapping={"access_log": "accessLog"},
    )
    class LoggingProperty:
        def __init__(
            self,
            *,
            access_log: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.AccessLogProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the logging information for a virtual node.

            :param access_log: The access log configuration for a virtual node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-logging.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                logging_property = appmesh.CfnVirtualNode.LoggingProperty(
                    access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                        file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                            path="path",
                
                            # the properties below are optional
                            format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                                json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                    key="key",
                                    value="value"
                                )],
                                text="text"
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6bdfa6626b8ee56b3ce536da70f136a04046554c7cdbe9a840ce66646c83f6cd)
                check_type(argname="argument access_log", value=access_log, expected_type=type_hints["access_log"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if access_log is not None:
                self._values["access_log"] = access_log

        @builtins.property
        def access_log(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.AccessLogProperty"]]:
            '''The access log configuration for a virtual node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-logging.html#cfn-appmesh-virtualnode-logging-accesslog
            '''
            result = self._values.get("access_log")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.AccessLogProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.OutlierDetectionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "base_ejection_duration": "baseEjectionDuration",
            "interval": "interval",
            "max_ejection_percent": "maxEjectionPercent",
            "max_server_errors": "maxServerErrors",
        },
    )
    class OutlierDetectionProperty:
        def __init__(
            self,
            *,
            base_ejection_duration: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]]],
            interval: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]]],
            max_ejection_percent: jsii.Number,
            max_server_errors: jsii.Number,
        ) -> None:
            '''An object that represents the outlier detection for a virtual node's listener.

            :param base_ejection_duration: The base amount of time for which a host is ejected.
            :param interval: The time interval between ejection sweep analysis.
            :param max_ejection_percent: Maximum percentage of hosts in load balancing pool for upstream service that can be ejected. Will eject at least one host regardless of the value.
            :param max_server_errors: Number of consecutive ``5xx`` errors required for ejection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                outlier_detection_property = appmesh.CfnVirtualNode.OutlierDetectionProperty(
                    base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    interval=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    max_ejection_percent=123,
                    max_server_errors=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__84d2cd4b997162ecde0b41ec3f2b3a837b47403adb58e1dda22fd2af74dcfbb2)
                check_type(argname="argument base_ejection_duration", value=base_ejection_duration, expected_type=type_hints["base_ejection_duration"])
                check_type(argname="argument interval", value=interval, expected_type=type_hints["interval"])
                check_type(argname="argument max_ejection_percent", value=max_ejection_percent, expected_type=type_hints["max_ejection_percent"])
                check_type(argname="argument max_server_errors", value=max_server_errors, expected_type=type_hints["max_server_errors"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "base_ejection_duration": base_ejection_duration,
                "interval": interval,
                "max_ejection_percent": max_ejection_percent,
                "max_server_errors": max_server_errors,
            }

        @builtins.property
        def base_ejection_duration(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]:
            '''The base amount of time for which a host is ejected.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-baseejectionduration
            '''
            result = self._values.get("base_ejection_duration")
            assert result is not None, "Required property 'base_ejection_duration' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"], result)

        @builtins.property
        def interval(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]:
            '''The time interval between ejection sweep analysis.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-interval
            '''
            result = self._values.get("interval")
            assert result is not None, "Required property 'interval' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"], result)

        @builtins.property
        def max_ejection_percent(self) -> jsii.Number:
            '''Maximum percentage of hosts in load balancing pool for upstream service that can be ejected.

            Will eject at least one host regardless of the value.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-maxejectionpercent
            '''
            result = self._values.get("max_ejection_percent")
            assert result is not None, "Required property 'max_ejection_percent' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def max_server_errors(self) -> jsii.Number:
            '''Number of consecutive ``5xx`` errors required for ejection.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-maxservererrors
            '''
            result = self._values.get("max_server_errors")
            assert result is not None, "Required property 'max_server_errors' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OutlierDetectionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.PortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port", "protocol": "protocol"},
    )
    class PortMappingProperty:
        def __init__(self, *, port: jsii.Number, protocol: builtins.str) -> None:
            '''An object representing a virtual node or virtual router listener port mapping.

            :param port: The port used for the port mapping.
            :param protocol: The protocol used for the port mapping. Specify ``http`` , ``http2`` , ``grpc`` , or ``tcp`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-portmapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                port_mapping_property = appmesh.CfnVirtualNode.PortMappingProperty(
                    port=123,
                    protocol="protocol"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0bb1d3f2eba0689a97d37b0a5b7748e6ee3ab1c5f2d992e68775e22f7573551c)
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port": port,
                "protocol": protocol,
            }

        @builtins.property
        def port(self) -> jsii.Number:
            '''The port used for the port mapping.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-portmapping.html#cfn-appmesh-virtualnode-portmapping-port
            '''
            result = self._values.get("port")
            assert result is not None, "Required property 'port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol used for the port mapping.

            Specify ``http`` , ``http2`` , ``grpc`` , or ``tcp`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-portmapping.html#cfn-appmesh-virtualnode-portmapping-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.ServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={"aws_cloud_map": "awsCloudMap", "dns": "dns"},
    )
    class ServiceDiscoveryProperty:
        def __init__(
            self,
            *,
            aws_cloud_map: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            dns: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.DnsServiceDiscoveryProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the service discovery information for a virtual node.

            :param aws_cloud_map: Specifies any AWS Cloud Map information for the virtual node.
            :param dns: Specifies the DNS information for the virtual node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-servicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                service_discovery_property = appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                    aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                        namespace_name="namespaceName",
                        service_name="serviceName",
                
                        # the properties below are optional
                        attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                            key="key",
                            value="value"
                        )],
                        ip_preference="ipPreference"
                    ),
                    dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                        hostname="hostname",
                
                        # the properties below are optional
                        ip_preference="ipPreference",
                        response_type="responseType"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__57b5ef8ae2913653b69527c10b33c80be36a0042b53aff2f9cb83834d595e2cd)
                check_type(argname="argument aws_cloud_map", value=aws_cloud_map, expected_type=type_hints["aws_cloud_map"])
                check_type(argname="argument dns", value=dns, expected_type=type_hints["dns"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if aws_cloud_map is not None:
                self._values["aws_cloud_map"] = aws_cloud_map
            if dns is not None:
                self._values["dns"] = dns

        @builtins.property
        def aws_cloud_map(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty"]]:
            '''Specifies any AWS Cloud Map information for the virtual node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-servicediscovery.html#cfn-appmesh-virtualnode-servicediscovery-awscloudmap
            '''
            result = self._values.get("aws_cloud_map")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty"]], result)

        @builtins.property
        def dns(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DnsServiceDiscoveryProperty"]]:
            '''Specifies the DNS information for the virtual node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-servicediscovery.html#cfn-appmesh-virtualnode-servicediscovery-dns
            '''
            result = self._values.get("dns")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DnsServiceDiscoveryProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class SubjectAlternativeNameMatchersProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents the methods by which a subject alternative name on a peer Transport Layer Security (TLS) certificate can be matched.

            :param exact: The values sent must match the specified values exactly.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenamematchers.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                subject_alternative_name_matchers_property = appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                    exact=["exact"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__58800912f9dbf66b3093fcf028b78060701160f5bae2d76fb09ce970ca22276b)
                check_type(argname="argument exact", value=exact, expected_type=type_hints["exact"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The values sent must match the specified values exactly.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenamematchers.html#cfn-appmesh-virtualnode-subjectalternativenamematchers-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNameMatchersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty",
        jsii_struct_bases=[],
        name_mapping={"match": "match"},
    )
    class SubjectAlternativeNamesProperty:
        def __init__(
            self,
            *,
            match: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.SubjectAlternativeNameMatchersProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''An object that represents the subject alternative names secured by the certificate.

            :param match: An object that represents the criteria for determining a SANs match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenames.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                subject_alternative_names_property = appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                        exact=["exact"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2d2453f09f838d296ee69f36481c81bb9ad4c1261af956b8fad0ea1aad471516)
                check_type(argname="argument match", value=match, expected_type=type_hints["match"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "match": match,
            }

        @builtins.property
        def match(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.SubjectAlternativeNameMatchersProperty"]:
            '''An object that represents the criteria for determining a SANs match.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenames.html#cfn-appmesh-virtualnode-subjectalternativenames-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.SubjectAlternativeNameMatchersProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNamesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.TcpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle"},
    )
    class TcpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.DurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tcptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                tcp_timeout_property = appmesh.CfnVirtualNode.TcpTimeoutProperty(
                    idle=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e2160f442f984ba01c95df32f2f46d0b3feb7749456efa2a773e6ddce066dda7)
                check_type(argname="argument idle", value=idle, expected_type=type_hints["idle"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tcptimeout.html#cfn-appmesh-virtualnode-tcptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.DurationProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_authority_arns": "certificateAuthorityArns"},
    )
    class TlsValidationContextAcmTrustProperty:
        def __init__(
            self,
            *,
            certificate_authority_arns: typing.Sequence[builtins.str],
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :param certificate_authority_arns: One or more ACM Amazon Resource Name (ARN)s.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextacmtrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                tls_validation_context_acm_trust_property = appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                    certificate_authority_arns=["certificateAuthorityArns"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__08903d4f6c66d4e1d815649dcd8ffd4b6c5ede5c847f97fb73bab33cf7602bd0)
                check_type(argname="argument certificate_authority_arns", value=certificate_authority_arns, expected_type=type_hints["certificate_authority_arns"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_authority_arns": certificate_authority_arns,
            }

        @builtins.property
        def certificate_authority_arns(self) -> typing.List[builtins.str]:
            '''One or more ACM Amazon Resource Name (ARN)s.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextacmtrust.html#cfn-appmesh-virtualnode-tlsvalidationcontextacmtrust-certificateauthorityarns
            '''
            result = self._values.get("certificate_authority_arns")
            assert result is not None, "Required property 'certificate_authority_arns' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextAcmTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_chain": "certificateChain"},
    )
    class TlsValidationContextFileTrustProperty:
        def __init__(self, *, certificate_chain: builtins.str) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :param certificate_chain: The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextfiletrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                tls_validation_context_file_trust_property = appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                    certificate_chain="certificateChain"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0457ec61db111292dcfb08b864fe1c76353edbc8786721664cc0d354f3786485)
                check_type(argname="argument certificate_chain", value=certificate_chain, expected_type=type_hints["certificate_chain"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "certificate_chain": certificate_chain,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextfiletrust.html#cfn-appmesh-virtualnode-tlsvalidationcontextfiletrust-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextFileTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.TlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class TlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.TlsValidationContextTrustProperty", typing.Dict[builtins.str, typing.Any]]],
            subject_alternative_names: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents how the proxy will validate its peer during Transport Layer Security (TLS) negotiation.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a Transport Layer Security (TLS) validation context. If you don't specify SANs on the *terminating* mesh endpoint, the Envoy proxy for that node doesn't verify the SAN on a peer client certificate. If you don't specify SANs on the *originating* mesh endpoint, the SAN on the certificate provided by the terminating endpoint must match the mesh endpoint service discovery configuration. Since SPIRE vended certificates have a SPIFFE ID as a name, you must set the SAN since the name doesn't match the service discovery name.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                tls_validation_context_property = appmesh.CfnVirtualNode.TlsValidationContextProperty(
                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                            certificate_authority_arns=["certificateAuthorityArns"]
                        ),
                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__754c458bfe5d9056dd4bc660cb95db5eddf0e09e39f5781a138f61cfdc6fa9c9)
                check_type(argname="argument trust", value=trust, expected_type=type_hints["trust"])
                check_type(argname="argument subject_alternative_names", value=subject_alternative_names, expected_type=type_hints["subject_alternative_names"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextTrustProperty"]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontext.html#cfn-appmesh-virtualnode-tlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextTrustProperty"], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.SubjectAlternativeNamesProperty"]]:
            '''A reference to an object that represents the SANs for a Transport Layer Security (TLS) validation context.

            If you don't specify SANs on the *terminating* mesh endpoint, the Envoy proxy for that node doesn't verify the SAN on a peer client certificate. If you don't specify SANs on the *originating* mesh endpoint, the SAN on the certificate provided by the terminating endpoint must match the mesh endpoint service discovery configuration. Since SPIRE vended certificates have a SPIFFE ID as a name, you must set the SAN since the name doesn't match the service discovery name.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontext.html#cfn-appmesh-virtualnode-tlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.SubjectAlternativeNamesProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class TlsValidationContextSdsTrustProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret for a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextsdstrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                tls_validation_context_sds_trust_property = appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                    secret_name="secretName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3000b25c1ef9dda0d46dfea901d98be20476e3ed2d4caf3afca155ab1e7fd4a9)
                check_type(argname="argument secret_name", value=secret_name, expected_type=type_hints["secret_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret for a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextsdstrust.html#cfn-appmesh-virtualnode-tlsvalidationcontextsdstrust-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextSdsTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.TlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class TlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.TlsValidationContextAcmTrustProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            file: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            sds: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust.

            :param acm: A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.
            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                tls_validation_context_trust_property = appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                        certificate_authority_arns=["certificateAuthorityArns"]
                    ),
                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8e1f52fd69012429b125810f4aac5909cf4816647958d22d76180f16690bf62f)
                check_type(argname="argument acm", value=acm, expected_type=type_hints["acm"])
                check_type(argname="argument file", value=file, expected_type=type_hints["file"])
                check_type(argname="argument sds", value=sds, expected_type=type_hints["sds"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextAcmTrustProperty"]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-tlsvalidationcontexttrust-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextAcmTrustProperty"]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextFileTrustProperty"]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-tlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextFileTrustProperty"]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextSdsTrustProperty"]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-tlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.TlsValidationContextSdsTrustProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"grpc": "grpc", "http": "http", "http2": "http2", "tcp": "tcp"},
    )
    class VirtualNodeConnectionPoolProperty:
        def __init__(
            self,
            *,
            grpc: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            http: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            http2: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            tcp: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the type of virtual node connection pool.

            Only one protocol is used at a time and should be the same protocol as the one chosen under port mapping.

            If not present the default value for ``maxPendingRequests`` is ``2147483647`` .

            :param grpc: An object that represents a type of connection pool.
            :param http: An object that represents a type of connection pool.
            :param http2: An object that represents a type of connection pool.
            :param tcp: An object that represents a type of connection pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_node_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                    grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                        max_requests=123
                    ),
                    http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                        max_connections=123,
                
                        # the properties below are optional
                        max_pending_requests=123
                    ),
                    http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                        max_requests=123
                    ),
                    tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                        max_connections=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6727c1833d1fb0991bf95ccc3b990295ba962f57ad96ae1029eceb544da5a98b)
                check_type(argname="argument grpc", value=grpc, expected_type=type_hints["grpc"])
                check_type(argname="argument http", value=http, expected_type=type_hints["http"])
                check_type(argname="argument http2", value=http2, expected_type=type_hints["http2"])
                check_type(argname="argument tcp", value=tcp, expected_type=type_hints["tcp"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if grpc is not None:
                self._values["grpc"] = grpc
            if http is not None:
                self._values["http"] = http
            if http2 is not None:
                self._values["http2"] = http2
            if tcp is not None:
                self._values["tcp"] = tcp

        @builtins.property
        def grpc(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty"]]:
            '''An object that represents a type of connection pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-grpc
            '''
            result = self._values.get("grpc")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty"]], result)

        @builtins.property
        def http(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty"]]:
            '''An object that represents a type of connection pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-http
            '''
            result = self._values.get("http")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty"]], result)

        @builtins.property
        def http2(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty"]]:
            '''An object that represents a type of connection pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-http2
            '''
            result = self._values.get("http2")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty"]], result)

        @builtins.property
        def tcp(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty"]]:
            '''An object that represents a type of connection pool.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-tcp
            '''
            result = self._values.get("tcp")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualNodeGrpcConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodegrpcconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_node_grpc_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                    max_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1f6ba2cf391fc834b3ca10dd45e222ce5c3d854ae2fabcd687cb6ecff484e48a)
                check_type(argname="argument max_requests", value=max_requests, expected_type=type_hints["max_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodegrpcconnectionpool.html#cfn-appmesh-virtualnode-virtualnodegrpcconnectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeGrpcConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualNodeHttp2ConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttp2connectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_node_http2_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                    max_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3d0296785abcd271f0ac868ad39ffeb87556d23163332df7120c6ff35c41af13)
                check_type(argname="argument max_requests", value=max_requests, expected_type=type_hints["max_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttp2connectionpool.html#cfn-appmesh-virtualnode-virtualnodehttp2connectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeHttp2ConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_connections": "maxConnections",
            "max_pending_requests": "maxPendingRequests",
        },
    )
    class VirtualNodeHttpConnectionPoolProperty:
        def __init__(
            self,
            *,
            max_connections: jsii.Number,
            max_pending_requests: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a type of connection pool.

            :param max_connections: Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.
            :param max_pending_requests: Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttpconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_node_http_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                    max_connections=123,
                
                    # the properties below are optional
                    max_pending_requests=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5145936c514a72b5e4089d2e628730c8867e2721d3821f320ff71b0aa846aba7)
                check_type(argname="argument max_connections", value=max_connections, expected_type=type_hints["max_connections"])
                check_type(argname="argument max_pending_requests", value=max_pending_requests, expected_type=type_hints["max_pending_requests"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_connections": max_connections,
            }
            if max_pending_requests is not None:
                self._values["max_pending_requests"] = max_pending_requests

        @builtins.property
        def max_connections(self) -> jsii.Number:
            '''Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttpconnectionpool.html#cfn-appmesh-virtualnode-virtualnodehttpconnectionpool-maxconnections
            '''
            result = self._values.get("max_connections")
            assert result is not None, "Required property 'max_connections' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def max_pending_requests(self) -> typing.Optional[jsii.Number]:
            '''Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttpconnectionpool.html#cfn-appmesh-virtualnode-virtualnodehttpconnectionpool-maxpendingrequests
            '''
            result = self._values.get("max_pending_requests")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeHttpConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.VirtualNodeSpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "backend_defaults": "backendDefaults",
            "backends": "backends",
            "listeners": "listeners",
            "logging": "logging",
            "service_discovery": "serviceDiscovery",
        },
    )
    class VirtualNodeSpecProperty:
        def __init__(
            self,
            *,
            backend_defaults: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.BackendDefaultsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            backends: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.BackendProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            listeners: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ListenerProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            logging: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.LoggingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            service_discovery: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ServiceDiscoveryProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the specification of a virtual node.

            :param backend_defaults: A reference to an object that represents the defaults for backends.
            :param backends: The backends that the virtual node is expected to send outbound traffic to. .. epigraph:: App Mesh doesn't validate the existence of those virtual services specified in backends. This is to prevent a cyclic dependency between virtual nodes and virtual services creation. Make sure the virtual service name is correct. The virtual service can be created afterwards if it doesn't already exist.
            :param listeners: The listener that the virtual node is expected to receive inbound traffic from. You can specify one listener.
            :param logging: The inbound and outbound access logging information for the virtual node.
            :param service_discovery: The service discovery information for the virtual node. If your virtual node does not expect ingress traffic, you can omit this parameter. If you specify a ``listener`` , then you must specify service discovery information.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_node_spec_property = appmesh.CfnVirtualNode.VirtualNodeSpecProperty(
                    backend_defaults=appmesh.CfnVirtualNode.BackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
                
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    backends=[appmesh.CfnVirtualNode.BackendProperty(
                        virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                            virtual_service_name="virtualServiceName",
                
                            # the properties below are optional
                            client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                                tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                    validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                        trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                            acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                                certificate_authority_arns=["certificateAuthorityArns"]
                                            ),
                                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                                certificate_chain="certificateChain"
                                            ),
                                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                                secret_name="secretName"
                                            )
                                        ),
                
                                        # the properties below are optional
                                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                                exact=["exact"]
                                            )
                                        )
                                    ),
                
                                    # the properties below are optional
                                    certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                            certificate_chain="certificateChain",
                                            private_key="privateKey"
                                        ),
                                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                                    enforce=False,
                                    ports=[123]
                                )
                            )
                        )
                    )],
                    listeners=[appmesh.CfnVirtualNode.ListenerProperty(
                        port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
                
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                                max_connections=123,
                
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                                max_requests=123
                            ),
                            tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                                max_connections=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
                
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                            base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            interval=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            max_ejection_percent=123,
                            max_server_errors=123
                        ),
                        timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                            grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            )
                        ),
                        tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                            certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
                
                            # the properties below are optional
                            validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
                    logging=appmesh.CfnVirtualNode.LoggingProperty(
                        access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                            file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                                path="path",
                
                                # the properties below are optional
                                format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                                    json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                        key="key",
                                        value="value"
                                    )],
                                    text="text"
                                )
                            )
                        )
                    ),
                    service_discovery=appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                        aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                            namespace_name="namespaceName",
                            service_name="serviceName",
                
                            # the properties below are optional
                            attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                                key="key",
                                value="value"
                            )],
                            ip_preference="ipPreference"
                        ),
                        dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                            hostname="hostname",
                
                            # the properties below are optional
                            ip_preference="ipPreference",
                            response_type="responseType"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9397e9cb5992783fbc0d91bab4ed082c1d99667d6346ed97f95d7b2967703991)
                check_type(argname="argument backend_defaults", value=backend_defaults, expected_type=type_hints["backend_defaults"])
                check_type(argname="argument backends", value=backends, expected_type=type_hints["backends"])
                check_type(argname="argument listeners", value=listeners, expected_type=type_hints["listeners"])
                check_type(argname="argument logging", value=logging, expected_type=type_hints["logging"])
                check_type(argname="argument service_discovery", value=service_discovery, expected_type=type_hints["service_discovery"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if backend_defaults is not None:
                self._values["backend_defaults"] = backend_defaults
            if backends is not None:
                self._values["backends"] = backends
            if listeners is not None:
                self._values["listeners"] = listeners
            if logging is not None:
                self._values["logging"] = logging
            if service_discovery is not None:
                self._values["service_discovery"] = service_discovery

        @builtins.property
        def backend_defaults(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.BackendDefaultsProperty"]]:
            '''A reference to an object that represents the defaults for backends.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-backenddefaults
            '''
            result = self._values.get("backend_defaults")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.BackendDefaultsProperty"]], result)

        @builtins.property
        def backends(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.BackendProperty"]]]]:
            '''The backends that the virtual node is expected to send outbound traffic to.

            .. epigraph::

               App Mesh doesn't validate the existence of those virtual services specified in backends. This is to prevent a cyclic dependency between virtual nodes and virtual services creation. Make sure the virtual service name is correct. The virtual service can be created afterwards if it doesn't already exist.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-backends
            '''
            result = self._values.get("backends")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.BackendProperty"]]]], result)

        @builtins.property
        def listeners(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerProperty"]]]]:
            '''The listener that the virtual node is expected to receive inbound traffic from.

            You can specify one listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-listeners
            '''
            result = self._values.get("listeners")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ListenerProperty"]]]], result)

        @builtins.property
        def logging(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.LoggingProperty"]]:
            '''The inbound and outbound access logging information for the virtual node.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-logging
            '''
            result = self._values.get("logging")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.LoggingProperty"]], result)

        @builtins.property
        def service_discovery(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ServiceDiscoveryProperty"]]:
            '''The service discovery information for the virtual node.

            If your virtual node does not expect ingress traffic, you can omit this parameter. If you specify a ``listener`` , then you must specify service discovery information.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-servicediscovery
            '''
            result = self._values.get("service_discovery")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ServiceDiscoveryProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_connections": "maxConnections"},
    )
    class VirtualNodeTcpConnectionPoolProperty:
        def __init__(self, *, max_connections: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_connections: Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodetcpconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_node_tcp_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                    max_connections=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c48b66ff6eba277a687b57fd7606aa7259a93f693ccac817360f6e6057ef1616)
                check_type(argname="argument max_connections", value=max_connections, expected_type=type_hints["max_connections"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "max_connections": max_connections,
            }

        @builtins.property
        def max_connections(self) -> jsii.Number:
            '''Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodetcpconnectionpool.html#cfn-appmesh-virtualnode-virtualnodetcpconnectionpool-maxconnections
            '''
            result = self._values.get("max_connections")
            assert result is not None, "Required property 'max_connections' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeTcpConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNode.VirtualServiceBackendProperty",
        jsii_struct_bases=[],
        name_mapping={
            "virtual_service_name": "virtualServiceName",
            "client_policy": "clientPolicy",
        },
    )
    class VirtualServiceBackendProperty:
        def __init__(
            self,
            *,
            virtual_service_name: builtins.str,
            client_policy: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualNode.ClientPolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents a virtual service backend for a virtual node.

            :param virtual_service_name: The name of the virtual service that is acting as a virtual node backend. .. epigraph:: App Mesh doesn't validate the existence of those virtual services specified in backends. This is to prevent a cyclic dependency between virtual nodes and virtual services creation. Make sure the virtual service name is correct. The virtual service can be created afterwards if it doesn't already exist.
            :param client_policy: A reference to an object that represents the client policy for a backend.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualservicebackend.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_service_backend_property = appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                    virtual_service_name="virtualServiceName",
                
                    # the properties below are optional
                    client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                        tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
                
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e6f165c7862245b014cc788f8dc1bb08feb1b3508808b67c59fa7cd86c651177)
                check_type(argname="argument virtual_service_name", value=virtual_service_name, expected_type=type_hints["virtual_service_name"])
                check_type(argname="argument client_policy", value=client_policy, expected_type=type_hints["client_policy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_service_name": virtual_service_name,
            }
            if client_policy is not None:
                self._values["client_policy"] = client_policy

        @builtins.property
        def virtual_service_name(self) -> builtins.str:
            '''The name of the virtual service that is acting as a virtual node backend.

            .. epigraph::

               App Mesh doesn't validate the existence of those virtual services specified in backends. This is to prevent a cyclic dependency between virtual nodes and virtual services creation. Make sure the virtual service name is correct. The virtual service can be created afterwards if it doesn't already exist.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualservicebackend.html#cfn-appmesh-virtualnode-virtualservicebackend-virtualservicename
            '''
            result = self._values.get("virtual_service_name")
            assert result is not None, "Required property 'virtual_service_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def client_policy(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ClientPolicyProperty"]]:
            '''A reference to an object that represents the client policy for a backend.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualservicebackend.html#cfn-appmesh-virtualnode-virtualservicebackend-clientpolicy
            '''
            result = self._values.get("client_policy")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualNode.ClientPolicyProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualServiceBackendProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualNodeProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "mesh_owner": "meshOwner",
        "tags": "tags",
        "virtual_node_name": "virtualNodeName",
    },
)
class CfnVirtualNodeProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union[CfnVirtualNode.VirtualNodeSpecProperty, typing.Dict[builtins.str, typing.Any]]],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_node_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualNode``.

        :param mesh_name: The name of the service mesh to create the virtual node in.
        :param spec: The virtual node specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual node to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_node_name: The name to use for the virtual node.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            cfn_virtual_node_props = appmesh.CfnVirtualNodeProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualNode.VirtualNodeSpecProperty(
                    backend_defaults=appmesh.CfnVirtualNode.BackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
            
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
            
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    backends=[appmesh.CfnVirtualNode.BackendProperty(
                        virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                            virtual_service_name="virtualServiceName",
            
                            # the properties below are optional
                            client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                                tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                    validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                        trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                            acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                                certificate_authority_arns=["certificateAuthorityArns"]
                                            ),
                                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                                certificate_chain="certificateChain"
                                            ),
                                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                                secret_name="secretName"
                                            )
                                        ),
            
                                        # the properties below are optional
                                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                                exact=["exact"]
                                            )
                                        )
                                    ),
            
                                    # the properties below are optional
                                    certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                            certificate_chain="certificateChain",
                                            private_key="privateKey"
                                        ),
                                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                                    enforce=False,
                                    ports=[123]
                                )
                            )
                        )
                    )],
                    listeners=[appmesh.CfnVirtualNode.ListenerProperty(
                        port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
            
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                                max_connections=123,
            
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                                max_requests=123
                            ),
                            tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                                max_connections=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
            
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                            base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            interval=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            max_ejection_percent=123,
                            max_server_errors=123
                        ),
                        timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                            grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            )
                        ),
                        tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                            certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
            
                            # the properties below are optional
                            validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
            
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
                    logging=appmesh.CfnVirtualNode.LoggingProperty(
                        access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                            file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                                path="path",
            
                                # the properties below are optional
                                format=appmesh.CfnVirtualNode.LoggingFormatProperty(
                                    json=[appmesh.CfnVirtualNode.JsonFormatRefProperty(
                                        key="key",
                                        value="value"
                                    )],
                                    text="text"
                                )
                            )
                        )
                    ),
                    service_discovery=appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                        aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                            namespace_name="namespaceName",
                            service_name="serviceName",
            
                            # the properties below are optional
                            attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                                key="key",
                                value="value"
                            )],
                            ip_preference="ipPreference"
                        ),
                        dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                            hostname="hostname",
            
                            # the properties below are optional
                            ip_preference="ipPreference",
                            response_type="responseType"
                        )
                    )
                ),
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                virtual_node_name="virtualNodeName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__22c25bf7b2103fd97cba80a70016c31566abb88b44d7b63d2d93c87e9ea4dfa8)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument virtual_node_name", value=virtual_node_name, expected_type=type_hints["virtual_node_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags
        if virtual_node_name is not None:
            self._values["virtual_node_name"] = virtual_node_name

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual node in.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnVirtualNode.VirtualNodeSpecProperty]:
        '''The virtual node specification to apply.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnVirtualNode.VirtualNodeSpecProperty], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the virtual node to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def virtual_node_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual node.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-virtualnodename
        '''
        result = self._values.get("virtual_node_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualNodeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnVirtualRouter(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualRouter",
):
    '''Creates a virtual router within a service mesh.

    Specify a ``listener`` for any inbound traffic that your virtual router receives. Create a virtual router for each protocol and port that you need to route. Virtual routers handle traffic for one or more virtual services within your mesh. After you create your virtual router, create and associate routes for your virtual router that direct incoming requests to different virtual nodes.

    For more information about virtual routers, see `Virtual routers <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_routers.html>`_ .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html
    :cloudformationResource: AWS::AppMesh::VirtualRouter
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_appmesh as appmesh
        
        cfn_virtual_router = appmesh.CfnVirtualRouter(self, "MyCfnVirtualRouter",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualRouter.VirtualRouterSpecProperty(
                listeners=[appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                    port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    )
                )]
            ),
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            virtual_router_name="virtualRouterName"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualRouter.VirtualRouterSpecProperty", typing.Dict[builtins.str, typing.Any]]],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_router_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param mesh_name: The name of the service mesh to create the virtual router in.
        :param spec: The virtual router specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual router to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_router_name: The name to use for the virtual router.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d7db94af589ae6f78bef7744184dec5795b19ce8c237e3e907dfb1fc6094dd1)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVirtualRouterProps(
            mesh_name=mesh_name,
            spec=spec,
            mesh_owner=mesh_owner,
            tags=tags,
            virtual_router_name=virtual_router_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__db2e6a23a74d8cd9076e69f315b35149ab410ea0d964d92d01c5713cba263249)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b2d0e6879685e87b58e7271493bed103f6df7ad49bf21481b103d5a8c64336e)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual router.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual router resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual router.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualRouterName")
    def attr_virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router.

        :cloudformationAttribute: VirtualRouterName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualRouterName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual router in.'''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d7d5c50b61fdf9642d93ea3fc67dd84319b1b26cbb1a0eded4c7d6ce364f2aab)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualRouter.VirtualRouterSpecProperty"]:
        '''The virtual router specification to apply.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualRouter.VirtualRouterSpecProperty"], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnVirtualRouter.VirtualRouterSpecProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b54baf62ad2c86c92d6b933af45c34eab5feed616d8eb1f26d654a471d9d5227)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8e7f64d3a8c8072c2fbf4f5ef0996e14ab9dbe6d76386790cbadacde86f23a9e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the virtual router to assist with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0085d8ee8bb71ad1569243bbeaf036f25378944d65bf336d669c73071521c32e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="virtualRouterName")
    def virtual_router_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual router.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualRouterName"))

    @virtual_router_name.setter
    def virtual_router_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e65ddf8d6aacffab6aaa085d90a6d47d9d46a0b0af98d4c6b011212b413a51dd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualRouterName", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualRouter.PortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port", "protocol": "protocol"},
    )
    class PortMappingProperty:
        def __init__(self, *, port: jsii.Number, protocol: builtins.str) -> None:
            '''An object representing a virtual router listener port mapping.

            :param port: The port used for the port mapping.
            :param protocol: The protocol used for the port mapping. Specify one protocol.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-portmapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                port_mapping_property = appmesh.CfnVirtualRouter.PortMappingProperty(
                    port=123,
                    protocol="protocol"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1cc93b9ca8a34dcc79e0fa489152729f5a1f67f840b7d1e8e1357c8f937751c3)
                check_type(argname="argument port", value=port, expected_type=type_hints["port"])
                check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port": port,
                "protocol": protocol,
            }

        @builtins.property
        def port(self) -> jsii.Number:
            '''The port used for the port mapping.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-portmapping.html#cfn-appmesh-virtualrouter-portmapping-port
            '''
            result = self._values.get("port")
            assert result is not None, "Required property 'port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol used for the port mapping.

            Specify one protocol.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-portmapping.html#cfn-appmesh-virtualrouter-portmapping-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualRouter.VirtualRouterListenerProperty",
        jsii_struct_bases=[],
        name_mapping={"port_mapping": "portMapping"},
    )
    class VirtualRouterListenerProperty:
        def __init__(
            self,
            *,
            port_mapping: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualRouter.PortMappingProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''An object that represents a virtual router listener.

            :param port_mapping: The port mapping information for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterlistener.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_router_listener_property = appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                    port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__012cff065789d3600fab25dcb8c754e7d52b726da0caa562f2bc960bef6c046c)
                check_type(argname="argument port_mapping", value=port_mapping, expected_type=type_hints["port_mapping"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "port_mapping": port_mapping,
            }

        @builtins.property
        def port_mapping(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualRouter.PortMappingProperty"]:
            '''The port mapping information for the listener.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterlistener.html#cfn-appmesh-virtualrouter-virtualrouterlistener-portmapping
            '''
            result = self._values.get("port_mapping")
            assert result is not None, "Required property 'port_mapping' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualRouter.PortMappingProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualRouterListenerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualRouter.VirtualRouterSpecProperty",
        jsii_struct_bases=[],
        name_mapping={"listeners": "listeners"},
    )
    class VirtualRouterSpecProperty:
        def __init__(
            self,
            *,
            listeners: typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualRouter.VirtualRouterListenerProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''An object that represents the specification of a virtual router.

            :param listeners: The listeners that the virtual router is expected to receive inbound traffic from.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterspec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_router_spec_property = appmesh.CfnVirtualRouter.VirtualRouterSpecProperty(
                    listeners=[appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                        port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        )
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__24ba668fed4208e4d7de0ee6ca2caa628bcfdc42d471d2494b80b8cb3a3ce56a)
                check_type(argname="argument listeners", value=listeners, expected_type=type_hints["listeners"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "listeners": listeners,
            }

        @builtins.property
        def listeners(
            self,
        ) -> typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualRouter.VirtualRouterListenerProperty"]]]:
            '''The listeners that the virtual router is expected to receive inbound traffic from.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterspec.html#cfn-appmesh-virtualrouter-virtualrouterspec-listeners
            '''
            result = self._values.get("listeners")
            assert result is not None, "Required property 'listeners' is missing"
            return typing.cast(typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnVirtualRouter.VirtualRouterListenerProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualRouterSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualRouterProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "mesh_owner": "meshOwner",
        "tags": "tags",
        "virtual_router_name": "virtualRouterName",
    },
)
class CfnVirtualRouterProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union[CfnVirtualRouter.VirtualRouterSpecProperty, typing.Dict[builtins.str, typing.Any]]],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        virtual_router_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualRouter``.

        :param mesh_name: The name of the service mesh to create the virtual router in.
        :param spec: The virtual router specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual router to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_router_name: The name to use for the virtual router.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            cfn_virtual_router_props = appmesh.CfnVirtualRouterProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualRouter.VirtualRouterSpecProperty(
                    listeners=[appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                        port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        )
                    )]
                ),
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                virtual_router_name="virtualRouterName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__704081fdaccdc3e4953595977df982841f6b9a865e1e515988aa767c68106abb)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument virtual_router_name", value=virtual_router_name, expected_type=type_hints["virtual_router_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags
        if virtual_router_name is not None:
            self._values["virtual_router_name"] = virtual_router_name

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual router in.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnVirtualRouter.VirtualRouterSpecProperty]:
        '''The virtual router specification to apply.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnVirtualRouter.VirtualRouterSpecProperty], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the virtual router to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def virtual_router_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual router.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-virtualroutername
        '''
        result = self._values.get("virtual_router_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualRouterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnVirtualService(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualService",
):
    '''Creates a virtual service within a service mesh.

    A virtual service is an abstraction of a real service that is provided by a virtual node directly or indirectly by means of a virtual router. Dependent services call your virtual service by its ``virtualServiceName`` , and those requests are routed to the virtual node or virtual router that is specified as the provider for the virtual service.

    For more information about virtual services, see `Virtual services <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html>`_ .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html
    :cloudformationResource: AWS::AppMesh::VirtualService
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_appmesh as appmesh
        
        cfn_virtual_service = appmesh.CfnVirtualService(self, "MyCfnVirtualService",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualService.VirtualServiceSpecProperty(
                provider=appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                    virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                        virtual_node_name="virtualNodeName"
                    ),
                    virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                        virtual_router_name="virtualRouterName"
                    )
                )
            ),
            virtual_service_name="virtualServiceName",
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualService.VirtualServiceSpecProperty", typing.Dict[builtins.str, typing.Any]]],
        virtual_service_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param mesh_name: The name of the service mesh to create the virtual service in.
        :param spec: The virtual service specification to apply.
        :param virtual_service_name: The name to use for the virtual service.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual service to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c766646981f50345ec5c38962151dc5b449c184f8cac2300d54d92713cc4768f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVirtualServiceProps(
            mesh_name=mesh_name,
            spec=spec,
            virtual_service_name=virtual_service_name,
            mesh_owner=mesh_owner,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__28bf2d78f53b648866e63d78cf5729c93859a7e47664047ad5b8dbc2cdd1f250)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__71c5bfdf5f29cd906fd3f18f87743f53d5532f56de98977c2eeda8c062507de2)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual service.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual service resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual service.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property
    @jsii.member(jsii_name="attrVirtualServiceName")
    def attr_virtual_service_name(self) -> builtins.str:
        '''The name of the virtual service.

        :cloudformationAttribute: VirtualServiceName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualServiceName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual service in.'''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3521ae86d106d64958842ee0a49a33e374bc725b273faa3e2687f9f8437cf14e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnVirtualService.VirtualServiceSpecProperty"]:
        '''The virtual service specification to apply.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnVirtualService.VirtualServiceSpecProperty"], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnVirtualService.VirtualServiceSpecProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1590c9a74aaae27ee831b796176b2620ec5f8b1a2df0e6d39a41ddc0cd2c9b58)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "spec", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="virtualServiceName")
    def virtual_service_name(self) -> builtins.str:
        '''The name to use for the virtual service.'''
        return typing.cast(builtins.str, jsii.get(self, "virtualServiceName"))

    @virtual_service_name.setter
    def virtual_service_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4abff200579535e2e48b9a6f48511b764e0aef9bfc73e55e773f4a9c11a68dfa)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "virtualServiceName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1dacbe43f5da44626f5405bd41421bd8826879ce59389bd80f6b34ecfe3170f7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "meshOwner", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the virtual service to assist with categorization and organization.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6c59e67a81903720d425b53745d00a109956c7d771772d047188cbb76a4532ae)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_node_name": "virtualNodeName"},
    )
    class VirtualNodeServiceProviderProperty:
        def __init__(self, *, virtual_node_name: builtins.str) -> None:
            '''An object that represents a virtual node service provider.

            :param virtual_node_name: The name of the virtual node that is acting as a service provider.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualnodeserviceprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_node_service_provider_property = appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                    virtual_node_name="virtualNodeName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b3c6bc3f6548fe931119a50a18fa98d519a6d66dd67ac9d9c50325494cae19ab)
                check_type(argname="argument virtual_node_name", value=virtual_node_name, expected_type=type_hints["virtual_node_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_node_name": virtual_node_name,
            }

        @builtins.property
        def virtual_node_name(self) -> builtins.str:
            '''The name of the virtual node that is acting as a service provider.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualnodeserviceprovider.html#cfn-appmesh-virtualservice-virtualnodeserviceprovider-virtualnodename
            '''
            result = self._values.get("virtual_node_name")
            assert result is not None, "Required property 'virtual_node_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeServiceProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_router_name": "virtualRouterName"},
    )
    class VirtualRouterServiceProviderProperty:
        def __init__(self, *, virtual_router_name: builtins.str) -> None:
            '''An object that represents a virtual node service provider.

            :param virtual_router_name: The name of the virtual router that is acting as a service provider.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualrouterserviceprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_router_service_provider_property = appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                    virtual_router_name="virtualRouterName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4f4d9c857266dd0cb653b28dd5e5d14af1be9f3884c696a054c886f331c69076)
                check_type(argname="argument virtual_router_name", value=virtual_router_name, expected_type=type_hints["virtual_router_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "virtual_router_name": virtual_router_name,
            }

        @builtins.property
        def virtual_router_name(self) -> builtins.str:
            '''The name of the virtual router that is acting as a service provider.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualrouterserviceprovider.html#cfn-appmesh-virtualservice-virtualrouterserviceprovider-virtualroutername
            '''
            result = self._values.get("virtual_router_name")
            assert result is not None, "Required property 'virtual_router_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualRouterServiceProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualService.VirtualServiceProviderProperty",
        jsii_struct_bases=[],
        name_mapping={
            "virtual_node": "virtualNode",
            "virtual_router": "virtualRouter",
        },
    )
    class VirtualServiceProviderProperty:
        def __init__(
            self,
            *,
            virtual_node: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualService.VirtualNodeServiceProviderProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            virtual_router: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualService.VirtualRouterServiceProviderProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the provider for a virtual service.

            :param virtual_node: The virtual node associated with a virtual service.
            :param virtual_router: The virtual router associated with a virtual service.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualserviceprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_service_provider_property = appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                    virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                        virtual_node_name="virtualNodeName"
                    ),
                    virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                        virtual_router_name="virtualRouterName"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8311d7e392f9591883bc4785c3a6f934e9f6f866db7342b17b50785527ca432b)
                check_type(argname="argument virtual_node", value=virtual_node, expected_type=type_hints["virtual_node"])
                check_type(argname="argument virtual_router", value=virtual_router, expected_type=type_hints["virtual_router"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if virtual_node is not None:
                self._values["virtual_node"] = virtual_node
            if virtual_router is not None:
                self._values["virtual_router"] = virtual_router

        @builtins.property
        def virtual_node(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualService.VirtualNodeServiceProviderProperty"]]:
            '''The virtual node associated with a virtual service.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualserviceprovider.html#cfn-appmesh-virtualservice-virtualserviceprovider-virtualnode
            '''
            result = self._values.get("virtual_node")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualService.VirtualNodeServiceProviderProperty"]], result)

        @builtins.property
        def virtual_router(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualService.VirtualRouterServiceProviderProperty"]]:
            '''The virtual router associated with a virtual service.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualserviceprovider.html#cfn-appmesh-virtualservice-virtualserviceprovider-virtualrouter
            '''
            result = self._values.get("virtual_router")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualService.VirtualRouterServiceProviderProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualServiceProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualService.VirtualServiceSpecProperty",
        jsii_struct_bases=[],
        name_mapping={"provider": "provider"},
    )
    class VirtualServiceSpecProperty:
        def __init__(
            self,
            *,
            provider: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnVirtualService.VirtualServiceProviderProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''An object that represents the specification of a virtual service.

            :param provider: The App Mesh object that is acting as the provider for a virtual service. You can specify a single virtual node or virtual router.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualservicespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_appmesh as appmesh
                
                virtual_service_spec_property = appmesh.CfnVirtualService.VirtualServiceSpecProperty(
                    provider=appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                        virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                            virtual_node_name="virtualNodeName"
                        ),
                        virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                            virtual_router_name="virtualRouterName"
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__07c966735ab76a84be5c679d1ca5d09e14a333b67c00abdda30542c6b14da7fe)
                check_type(argname="argument provider", value=provider, expected_type=type_hints["provider"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if provider is not None:
                self._values["provider"] = provider

        @builtins.property
        def provider(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualService.VirtualServiceProviderProperty"]]:
            '''The App Mesh object that is acting as the provider for a virtual service.

            You can specify a single virtual node or virtual router.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualservicespec.html#cfn-appmesh-virtualservice-virtualservicespec-provider
            '''
            result = self._values.get("provider")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnVirtualService.VirtualServiceProviderProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualServiceSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.CfnVirtualServiceProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "virtual_service_name": "virtualServiceName",
        "mesh_owner": "meshOwner",
        "tags": "tags",
    },
)
class CfnVirtualServiceProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[_IResolvable_da3f097b, typing.Union[CfnVirtualService.VirtualServiceSpecProperty, typing.Dict[builtins.str, typing.Any]]],
        virtual_service_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualService``.

        :param mesh_name: The name of the service mesh to create the virtual service in.
        :param spec: The virtual service specification to apply.
        :param virtual_service_name: The name to use for the virtual service.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual service to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            cfn_virtual_service_props = appmesh.CfnVirtualServiceProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualService.VirtualServiceSpecProperty(
                    provider=appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                        virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                            virtual_node_name="virtualNodeName"
                        ),
                        virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                            virtual_router_name="virtualRouterName"
                        )
                    )
                ),
                virtual_service_name="virtualServiceName",
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__871e9e5e313cc45184a80189cb5fdd819f587781d1883088e81e907cf0a300d0)
            check_type(argname="argument mesh_name", value=mesh_name, expected_type=type_hints["mesh_name"])
            check_type(argname="argument spec", value=spec, expected_type=type_hints["spec"])
            check_type(argname="argument virtual_service_name", value=virtual_service_name, expected_type=type_hints["virtual_service_name"])
            check_type(argname="argument mesh_owner", value=mesh_owner, expected_type=type_hints["mesh_owner"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
            "virtual_service_name": virtual_service_name,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual service in.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnVirtualService.VirtualServiceSpecProperty]:
        '''The virtual service specification to apply.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnVirtualService.VirtualServiceSpecProperty], result)

    @builtins.property
    def virtual_service_name(self) -> builtins.str:
        '''The name to use for the virtual service.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-virtualservicename
        '''
        result = self._values.get("virtual_service_name")
        assert result is not None, "Required property 'virtual_service_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''Optional metadata that you can apply to the virtual service to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.CommonGatewayRouteSpecOptions",
    jsii_struct_bases=[],
    name_mapping={"priority": "priority"},
)
class CommonGatewayRouteSpecOptions:
    def __init__(self, *, priority: typing.Optional[jsii.Number] = None) -> None:
        '''Base options for all gateway route specs.

        :param priority: The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            common_gateway_route_spec_options = appmesh.CommonGatewayRouteSpecOptions(
                priority=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__947d512cf7cf51e2102824e848ef7ced770a200c8107e330d2d829c35cec9141)
            check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if priority is not None:
            self._values["priority"] = priority

    @builtins.property
    def priority(self) -> typing.Optional[jsii.Number]:
        '''The priority for the gateway route.

        When a Virtual Gateway has multiple gateway routes, gateway route match
        is performed in the order of specified value, where 0 is the highest priority,
        and first matched gateway route is selected.

        :default: - no particular priority
        '''
        result = self._values.get("priority")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonGatewayRouteSpecOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_appmesh.DnsResponseType")
class DnsResponseType(enum.Enum):
    '''Enum of DNS service discovery response type.

    :exampleMetadata: infused

    Example::

        # A Virtual Node with a gRPC listener with a connection pool set
        # mesh: appmesh.Mesh
        
        node = appmesh.VirtualNode(self, "node",
            mesh=mesh,
            # DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
            # LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
            # whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
            # By default, the response type is assumed to be LOAD_BALANCER
            service_discovery=appmesh.ServiceDiscovery.dns("node", appmesh.DnsResponseType.ENDPOINTS),
            listeners=[appmesh.VirtualNodeListener.http(
                port=80,
                connection_pool=appmesh.HttpConnectionPool(
                    max_connections=100,
                    max_pending_requests=10
                )
            )]
        )
        
        # A Virtual Gateway with a gRPC listener with a connection pool set
        gateway = appmesh.VirtualGateway(self, "gateway",
            mesh=mesh,
            listeners=[appmesh.VirtualGatewayListener.grpc(
                port=8080,
                connection_pool=appmesh.GrpcConnectionPool(
                    max_requests=10
                )
            )],
            virtual_gateway_name="gateway"
        )
    '''

    LOAD_BALANCER = "LOAD_BALANCER"
    '''DNS resolver returns a loadbalanced set of endpoints and the traffic would be sent to the given endpoints.

    It would not drain existing connections to other endpoints that are not part of this list.
    '''
    ENDPOINTS = "ENDPOINTS"
    '''DNS resolver is returning all the endpoints.

    This also means that if an endpoint is missing, it would drain the current connections to the missing endpoint.
    '''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.GatewayRouteAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "gateway_route_name": "gatewayRouteName",
        "virtual_gateway": "virtualGateway",
    },
)
class GatewayRouteAttributes:
    def __init__(
        self,
        *,
        gateway_route_name: builtins.str,
        virtual_gateway: "IVirtualGateway",
    ) -> None:
        '''Interface with properties necessary to import a reusable GatewayRoute.

        :param gateway_route_name: The name of the GatewayRoute.
        :param virtual_gateway: The VirtualGateway this GatewayRoute is associated with.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            # virtual_gateway: appmesh.VirtualGateway
            
            gateway_route_attributes = appmesh.GatewayRouteAttributes(
                gateway_route_name="gatewayRouteName",
                virtual_gateway=virtual_gateway
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a3d2bf80e4e33a238cf2b070f15ebb24c8fa8457a0e4a0ea94239ab9d5769c73)
            check_type(argname="argument gateway_route_name", value=gateway_route_name, expected_type=type_hints["gateway_route_name"])
            check_type(argname="argument virtual_gateway", value=virtual_gateway, expected_type=type_hints["virtual_gateway"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "gateway_route_name": gateway_route_name,
            "virtual_gateway": virtual_gateway,
        }

    @builtins.property
    def gateway_route_name(self) -> builtins.str:
        '''The name of the GatewayRoute.'''
        result = self._values.get("gateway_route_name")
        assert result is not None, "Required property 'gateway_route_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def virtual_gateway(self) -> "IVirtualGateway":
        '''The VirtualGateway this GatewayRoute is associated with.'''
        result = self._values.get("virtual_gateway")
        assert result is not None, "Required property 'virtual_gateway' is missing"
        return typing.cast("IVirtualGateway", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.GatewayRouteBaseProps",
    jsii_struct_bases=[],
    name_mapping={"route_spec": "routeSpec", "gateway_route_name": "gatewayRouteName"},
)
class GatewayRouteBaseProps:
    def __init__(
        self,
        *,
        route_spec: "GatewayRouteSpec",
        gateway_route_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Basic configuration properties for a GatewayRoute.

        :param route_spec: What protocol the route uses.
        :param gateway_route_name: The name of the GatewayRoute. Default: - an automatically generated name

        :exampleMetadata: infused

        Example::

            # gateway: appmesh.VirtualGateway
            # virtual_service: appmesh.VirtualService
            
            
            gateway.add_gateway_route("gateway-route-grpc",
                route_spec=appmesh.GatewayRouteSpec.grpc(
                    route_target=virtual_service,
                    match=appmesh.GrpcGatewayRouteMatch(
                        hostname=appmesh.GatewayRouteHostnameMatch.exactly("example.com"),
                        # This disables the default rewrite to virtual service name and retain original request.
                        rewrite_request_hostname=False
                    )
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__38142e535d40895e1fa0e43473545bac8c208b0b09fbdebe02592dc9291ac166)
            check_type(argname="argument route_spec", value=route_spec, expected_type=type_hints["route_spec"])
            check_type(argname="argument gateway_route_name", value=gateway_route_name, expected_type=type_hints["gateway_route_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "route_spec": route_spec,
        }
        if gateway_route_name is not None:
            self._values["gateway_route_name"] = gateway_route_name

    @builtins.property
    def route_spec(self) -> "GatewayRouteSpec":
        '''What protocol the route uses.'''
        result = self._values.get("route_spec")
        assert result is not None, "Required property 'route_spec' is missing"
        return typing.cast("GatewayRouteSpec", result)

    @builtins.property
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''The name of the GatewayRoute.

        :default: - an automatically generated name
        '''
        result = self._values.get("gateway_route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteBaseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class GatewayRouteHostnameMatch(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_appmesh.GatewayRouteHostnameMatch",
):
    '''Used to generate host name matching methods.

    :exampleMetadata: infused

    Example::

        # gateway: appmesh.VirtualGateway
        # virtual_service: appmesh.VirtualService
        
        
        gateway.add_gateway_route("gateway-route-grpc",
            route_spec=appmesh.GatewayRouteSpec.grpc(
                route_target=virtual_service,
                match=appmesh.GrpcGatewayRouteMatch(
                    hostname=appmesh.GatewayRouteHostnameMatch.ends_with(".example.com")
                )
            )
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="endsWith")
    @builtins.classmethod
    def ends_with(cls, suffix: builtins.str) -> "GatewayRouteHostnameMatch":
        '''The value of the host name with the given name must end with the specified characters.

        :param suffix: The specified ending characters of the host name to match on.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__58033ee6eb3713a86f291c1185af6629bb03acd1b89013f426406f7c11886e0a)
            check_type(argname="argument suffix", value=suffix, expected_type=type_hints["suffix"])
        return typing.cast("GatewayRouteHostnameMatch", jsii.sinvoke(cls, "endsWith", [suffix]))

    @jsii.member(jsii_name="exactly")
    @builtins.classmethod
    def exactly(cls, name: builtins.str) -> "GatewayRouteHostnameMatch":
        '''The value of the host name must match the specified value exactly.

        :param name: The exact host name to match on.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__54784ce6448e1b758240adb6e5b48f9896a4ee871abca10fac1238f2510c6ba3)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
        return typing.cast("GatewayRouteHostnameMatch", jsii.sinvoke(cls, "exactly", [name]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(
        self,
        scope: _constructs_77d1e7e8.Construct,
    ) -> "GatewayRouteHostnameMatchConfig":
        '''Returns the gateway route host name match configuration.

        :param scope: -
        '''
        ...


class _GatewayRouteHostnameMatchProxy(GatewayRouteHostnameMatch):
    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _constructs_77d1e7e8.Construct,
    ) -> "GatewayRouteHostnameMatchConfig":
        '''Returns the gateway route host name match configuration.

        :param scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a7c407c602422706383d87420ac9d1b9e3767d589b4dfdbd09be1dee3c390b6f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("GatewayRouteHostnameMatchConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, GatewayRouteHostnameMatch).__jsii_proxy_class__ = lambda : _GatewayRouteHostnameMatchProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.GatewayRouteHostnameMatchConfig",
    jsii_struct_bases=[],
    name_mapping={"hostname_match": "hostnameMatch"},
)
class GatewayRouteHostnameMatchConfig:
    def __init__(
        self,
        *,
        hostname_match: typing.Union[CfnGatewayRoute.GatewayRouteHostnameMatchProperty, typing.Dict[builtins.str, typing.Any]],
    ) -> None:
        '''Configuration for gateway route host name match.

        :param hostname_match: GatewayRoute CFN configuration for host name match.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            gateway_route_hostname_match_config = appmesh.GatewayRouteHostnameMatchConfig(
                hostname_match=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                    exact="exact",
                    suffix="suffix"
                )
            )
        '''
        if isinstance(hostname_match, dict):
            hostname_match = CfnGatewayRoute.GatewayRouteHostnameMatchProperty(**hostname_match)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ff8c0e799c240c313cb859b5378773c373cf05ea6f035aadf530026de3f58be5)
            check_type(argname="argument hostname_match", value=hostname_match, expected_type=type_hints["hostname_match"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "hostname_match": hostname_match,
        }

    @builtins.property
    def hostname_match(self) -> CfnGatewayRoute.GatewayRouteHostnameMatchProperty:
        '''GatewayRoute CFN configuration for host name match.'''
        result = self._values.get("hostname_match")
        assert result is not None, "Required property 'hostname_match' is missing"
        return typing.cast(CfnGatewayRoute.GatewayRouteHostnameMatchProperty, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteHostnameMatchConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.GatewayRouteProps",
    jsii_struct_bases=[GatewayRouteBaseProps],
    name_mapping={
        "route_spec": "routeSpec",
        "gateway_route_name": "gatewayRouteName",
        "virtual_gateway": "virtualGateway",
    },
)
class GatewayRouteProps(GatewayRouteBaseProps):
    def __init__(
        self,
        *,
        route_spec: "GatewayRouteSpec",
        gateway_route_name: typing.Optional[builtins.str] = None,
        virtual_gateway: "IVirtualGateway",
    ) -> None:
        '''Properties to define a new GatewayRoute.

        :param route_spec: What protocol the route uses.
        :param gateway_route_name: The name of the GatewayRoute. Default: - an automatically generated name
        :param virtual_gateway: The VirtualGateway this GatewayRoute is associated with.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            # gateway_route_spec: appmesh.GatewayRouteSpec
            # virtual_gateway: appmesh.VirtualGateway
            
            gateway_route_props = appmesh.GatewayRouteProps(
                route_spec=gateway_route_spec,
                virtual_gateway=virtual_gateway,
            
                # the properties below are optional
                gateway_route_name="gatewayRouteName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__35d71528ccb204076b0008b3bf3322a659d50920a35d5bb19d86c32f34da818a)
            check_type(argname="argument route_spec", value=route_spec, expected_type=type_hints["route_spec"])
            check_type(argname="argument gateway_route_name", value=gateway_route_name, expected_type=type_hints["gateway_route_name"])
            check_type(argname="argument virtual_gateway", value=virtual_gateway, expected_type=type_hints["virtual_gateway"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "route_spec": route_spec,
            "virtual_gateway": virtual_gateway,
        }
        if gateway_route_name is not None:
            self._values["gateway_route_name"] = gateway_route_name

    @builtins.property
    def route_spec(self) -> "GatewayRouteSpec":
        '''What protocol the route uses.'''
        result = self._values.get("route_spec")
        assert result is not None, "Required property 'route_spec' is missing"
        return typing.cast("GatewayRouteSpec", result)

    @builtins.property
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''The name of the GatewayRoute.

        :default: - an automatically generated name
        '''
        result = self._values.get("gateway_route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def virtual_gateway(self) -> "IVirtualGateway":
        '''The VirtualGateway this GatewayRoute is associated with.'''
        result = self._values.get("virtual_gateway")
        assert result is not None, "Required property 'virtual_gateway' is missing"
        return typing.cast("IVirtualGateway", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class GatewayRouteSpec(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_appmesh.GatewayRouteSpec",
):
    '''Used to generate specs with different protocols for a GatewayRoute.

    :exampleMetadata: infused

    Example::

        # gateway: appmesh.VirtualGateway
        # virtual_service: appmesh.VirtualService
        
        
        gateway.add_gateway_route("gateway-route-grpc",
            route_spec=appmesh.GatewayRouteSpec.grpc(
                route_target=virtual_service,
                match=appmesh.GrpcGatewayRouteMatch(
                    hostname=appmesh.GatewayRouteHostnameMatch.exactly("example.com"),
                    # This disables the default rewrite to virtual service name and retain original request.
                    rewrite_request_hostname=False
                )
            )
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="grpc")
    @builtins.classmethod
    def grpc(
        cls,
        *,
        match: typing.Union["GrpcGatewayRouteMatch", typing.Dict[builtins.str, typing.Any]],
        route_target: "IVirtualService",
        priority: typing.Optional[jsii.Number] = None,
    ) -> "GatewayRouteSpec":
        '''Creates an gRPC Based GatewayRoute.

        :param match: The criterion for determining a request match for this GatewayRoute.
        :param route_target: The VirtualService this GatewayRoute directs traffic to.
        :param priority: The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority
        '''
        options = GrpcGatewayRouteSpecOptions(
            match=match, route_target=route_target, priority=priority
        )

        return typing.cast("GatewayRouteSpec", jsii.sinvoke(cls, "grpc", [options]))

    @jsii.member(jsii_name="http")
    @builtins.classmethod
    def http(
        cls,
        *,
        route_target: "IVirtualService",
        match: typing.Optional[typing.Union["HttpGatewayRouteMatch", typing.Dict[builtins.str, typing.Any]]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> "GatewayRouteSpec":
        '''Creates an HTTP Based GatewayRoute.

        :param route_target: The VirtualService this GatewayRoute directs traffic to.
        :param match: The criterion for determining a request match for this GatewayRoute. When path match is defined, this may optionally determine the path rewrite configuration. Default: - matches any path and automatically rewrites the path to '/'
        :param priority: The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority
        '''
        options = HttpGatewayRouteSpecOptions(
            route_target=route_target, match=match, priority=priority
        )

        return typing.cast("GatewayRouteSpec", jsii.sinvoke(cls, "http", [options]))

    @jsii.member(jsii_name="http2")
    @builtins.classmethod
    def http2(
        cls,
        *,
        route_target: "IVirtualService",
        match: typing.Optional[typing.Union["HttpGatewayRouteMatch", typing.Dict[builtins.str, typing.Any]]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> "GatewayRouteSpec":
        '''Creates an HTTP2 Based GatewayRoute.

        :param route_target: The VirtualService this GatewayRoute directs traffic to.
        :param match: The criterion for determining a request match for this GatewayRoute. When path match is defined, this may optionally determine the path rewrite configuration. Default: - matches any path and automatically rewrites the path to '/'
        :param priority: The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority
        '''
        options = HttpGatewayRouteSpecOptions(
            route_target=route_target, match=match, priority=priority
        )

        return typing.cast("GatewayRouteSpec", jsii.sinvoke(cls, "http2", [options]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, scope: _constructs_77d1e7e8.Construct) -> "GatewayRouteSpecConfig":
        '''Called when the GatewayRouteSpec type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -
        '''
        ...


class _GatewayRouteSpecProxy(GatewayRouteSpec):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _constructs_77d1e7e8.Construct) -> "GatewayRouteSpecConfig":
        '''Called when the GatewayRouteSpec type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ba78d66614f5626690c24cb64899827fa795a9938347a3e18ce56ee4ccda0611)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("GatewayRouteSpecConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, GatewayRouteSpec).__jsii_proxy_class__ = lambda : _GatewayRouteSpecProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.GatewayRouteSpecConfig",
    jsii_struct_bases=[],
    name_mapping={
        "grpc_spec_config": "grpcSpecConfig",
        "http2_spec_config": "http2SpecConfig",
        "http_spec_config": "httpSpecConfig",
        "priority": "priority",
    },
)
class GatewayRouteSpecConfig:
    def __init__(
        self,
        *,
        grpc_spec_config: typing.Optional[typing.Union[CfnGatewayRoute.GrpcGatewayRouteProperty, typing.Dict[builtins.str, typing.Any]]] = None,
        http2_spec_config: typing.Optional[typing.Union[CfnGatewayRoute.HttpGatewayRouteProperty, typing.Dict[builtins.str, typing.Any]]] = None,
        http_spec_config: typing.Optional[typing.Union[CfnGatewayRoute.HttpGatewayRouteProperty, typing.Dict[builtins.str, typing.Any]]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''All Properties for GatewayRoute Specs.

        :param grpc_spec_config: The spec for a grpc gateway route. Default: - no grpc spec
        :param http2_spec_config: The spec for an http2 gateway route. Default: - no http2 spec
        :param http_spec_config: The spec for an http gateway route. Default: - no http spec
        :param priority: The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_appmesh as appmesh
            
            gateway_route_spec_config = appmesh.GatewayRouteSpecConfig(
                grpc_spec_config=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
            
                            # the properties below are optional
                            port=123
                        ),
            
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                            name="name",
            
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        port=123,
                        service_name="serviceName"
                    )
                ),
                http2_spec_config=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
            
                            # the properties below are optional
                            port=123
                        ),
            
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
            
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
            
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                http_spec_config=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            ),
            
                            # the properties below are optional
                            port=123
                        ),
            
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
            
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        port=123,
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
            
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                priority=123
            )
        '''
        if isinstance(grpc_spec_config, dict):
            grpc_spec_config = CfnGatewayRoute.GrpcGatewayRouteProperty(**grpc_spec_config)
        if isinstance(http2_spec_config, dict):
            http2_spec_config = CfnGatewayRoute.HttpGatewayRouteProperty(**http2_spec_config)
        if isinstance(http_spec_config, dict):
            http_spec_config = CfnGatewayRoute.HttpGatewayRouteProperty(**http_spec_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7532101811a0485d097df6308e3744459738f9099b44fbe63b153a7aaf2d87c7)
            check_type(argname="argument grpc_spec_config", value=grpc_spec_config, expected_type=type_hints["grpc_spec_config"])
            check_type(argname="argument http2_spec_config", value=http2_spec_config, expected_type=type_hints["http2_spec_config"])
            check_type(argname="argument http_spec_config", value=http_spec_config, expected_type=type_hints["http_spec_config"])
            check_type(argname="argument priority", value=priority, expected_type=type_hints["priority"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if grpc_spec_config is not None:
            self._values["grpc_spec_config"] = grpc_spec_config
        if http2_spec_config is not None:
            self._values["http2_spec_config"] = http2_spec_config
        if http_spec_config is not None:
            self._values["http_spec_config"] = http_spec_config
        if priority is not None:
            self._values["priority"] = priority

    @builtins.property
    def grpc_spec_config(
        self,
    ) -> typing.Optional[CfnGatewayRoute.GrpcGatewayRouteProperty]:
        '''The spec for a grpc gateway route.

        :default: - no grpc spec
        '''
        result = self._values.get("grpc_spec_config")
        return typing.cast(typing.Optional[CfnGatewayRoute.GrpcGatewayRouteProperty], result)

    @builtins.property
    def http2_spec_config(
        self,
    ) -> typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty]:
        '''The spec for an http2 gateway route.

        :default: - no http2 spec
        '''
        result = self._values.get("http2_spec_config")
        return typing.cast(typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty], result)

    @builtins.property
    def http_spec_config(
        self,
    ) -> typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty]:
        '''The spec for an http gateway route.

        :default: - no http spec
        '''
        result = self._values.get("http_spec_config")
        return typing.cast(typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty], result)

    @builtins.property
    def priority(self) -> typing.Optional[jsii.Number]:
        '''The priority for the gateway route.

        When a Virtual Gateway has multiple gateway routes, gateway route match
        is performed in the order of specified value, where 0 is the highest priority,
        and first matched gateway route is selected.

        :default: - no particular priority
        '''
        result = self._values.get("priority")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteSpecConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.GrpcConnectionPool",
    jsii_struct_bases=[],
    name_mapping={"max_requests": "maxRequests"},
)
class GrpcConnectionPool:
    def __init__(self, *, max_requests: jsii.Number) -> None:
        '''Connection pool properties for gRPC listeners.

        :param max_requests: The maximum requests in the pool. Default: - none

        :exampleMetadata: infused

        Example::

            # A Virtual Node with a gRPC listener with a connection pool set
            # mesh: appmesh.Mesh
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                # DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
                # LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
                # whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
                # By default, the response type is assumed to be LOAD_BALANCER
                service_discovery=appmesh.ServiceDiscovery.dns("node", appmesh.DnsResponseType.ENDPOINTS),
                listeners=[appmesh.VirtualNodeListener.http(
                    port=80,
                    connection_pool=appmesh.HttpConnectionPool(
                        max_connections=100,
                        max_pending_requests=10
                    )
                )]
            )
            
            # A Virtual Gateway with a gRPC listener with a connection pool set
            gateway = appmesh.VirtualGateway(self, "gateway",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.grpc(
                    port=8080,
                    connection_pool=appmesh.GrpcConnectionPool(
                        max_requests=10
                    )
                )],
                virtual_gateway_name="gateway"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d72a343aada3c3e1233cff954a67e838bd9b44bf9d3527900347351116ac0dc6)
            check_type(argname="argument max_requests", value=max_requests, expected_type=type_hints["max_requests"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "max_requests": max_requests,
        }

    @builtins.property
    def max_requests(self) -> jsii.Number:
        '''The maximum requests in the pool.

        :default: - none
        '''
        result = self._values.get("max_requests")
        assert result is not None, "Required property 'max_requests' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GrpcConnectionPool(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_appmesh.GrpcGatewayListenerOptions",
    jsii_struct_bases=[],
    name_mapping={
        "connection_pool": "connectionPool",
        "health_check": "healthCheck",
        "port": "port",
        "tls": "tls",
    },
)
class GrpcGatewayListenerOptions:
    def __init__(
        self,
        *,
        connection_pool: typing.Optional[typing.Union[GrpcConnectionPool, typing.Dict[builtins.str, typing.Any]]] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        port: typing.Optional[jsii.Number] = None,
        tls: typing.Optional[typing.Union["ListenerTlsOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Represents the properties needed to define GRPC Listeners for a VirtualGateway.

        :param connection_pool: Connection pool for http listeners. Default: - None
        :param health_check: The health check information for the listener. Default: - no healthcheck
        :param port: Port to listen for connections on. Default: - 8080
        :param tls: Represents the configuration for enabling TLS on a listener. Default: - none

        :exampleMetadata: infused

        Example::

            # A Virtual Node with listener TLS from an ACM provided certificate
            # cert: certificatemanager.Certificate
            # mesh: appmesh.Mesh
            
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                service_discovery=appmesh.ServiceDiscovery.dns("node"),
                listeners=[appmesh.VirtualNodeListener.grpc(
                    port=80,
                    tls=appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.acm(cert)
                    )
                )]
            )
            
            # A Virtual Gateway with listener TLS from a customer provided file certificate
            gateway = appmesh.VirtualGateway(self, "gateway",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.grpc(
                    port=8080,
                    tls=appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey")
                    )
                )],
                virtual_gateway_name="gateway"
            )
            
            # A Virtual Gateway with listener TLS from a SDS provided certificate
            gateway2 = appmesh.VirtualGateway(self, "gateway2",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.http2(
                    port=8080,
                    tls=appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.sds("secrete_certificate")
                    )
                )],
                virtual_gateway_name="gateway2"
            )
        '''
        if isinstance(connection_pool, dict):
            connection_pool = GrpcConnectionPool(**connection_pool)
        if isinstance(tls, dict):
            tls = ListenerTlsOptions(**tls)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__163f88de35c4d289fef4e45e23be7d3511a58c9fc2f983c205f256befa1bce0c)
            check_type(argname="argument connection_pool", value=connection_pool, expected_type=type_hints["connection_pool"])
            check_type(argname="argument health_check", value=health_check, expected_type=type_hints["health_check"])
            check_type(argname="argument port", value=port, expected_type=type_hints["port"])
            check_type(argname="argument tls", value=tls, expected_type=type_hints["tls"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if connection_pool is not None:
            self._values["connection_pool"] = connection_pool
        if health_check is not None:
            self._values["health_check"] = health_check
        if port is not None:
            self._values["port"] = port
        if tls is not None:
            self._values["tls"] = tls

    @builtins.property
    def connection_pool(self) -> typing.Optional[GrpcConnectionPool]:
        '''Connection pool for http listeners.

        :default: - None
        '''
        result = self._values.get("connection_pool")
        return typing.cast(typing.Optional[GrpcConnectionPool], result)

    @builtins.property
    def health_check(self) -> typing.Optional["HealthCheck"]:
        '''The health check information for the listener.

        :default: - no healthcheck
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional["HealthCheck"], result)

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        '''Port to listen for connections on.

        :default: - 8080
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tls(self) -> typing.Optional["ListenerTlsOptions"]:
        '''Represents the configuration for enabling TLS on a listener.

        :default: - none
        '''
       