r'''
# Amazon API Gateway Construct Library

Amazon API Gateway is a fully managed service that makes it easy for developers
to publish, maintain, monitor, and secure APIs at any scale. Create an API to
access data, business logic, or functionality from your back-end services, such
as applications running on Amazon Elastic Compute Cloud (Amazon EC2), code
running on AWS Lambda, or any web application.

## Table of Contents

* [Amazon API Gateway Construct Library](#amazon-api-gateway-construct-library)

  * [Table of Contents](#table-of-contents)
  * [Defining APIs](#defining-apis)

    * [Breaking up Methods and Resources across Stacks](#breaking-up-methods-and-resources-across-stacks)
  * [AWS Lambda-backed APIs](#aws-lambda-backed-apis)
  * [AWS StepFunctions backed APIs](#aws-stepfunctions-backed-apis)
  * [Integration Targets](#integration-targets)
  * [Usage Plan & API Keys](#usage-plan--api-keys)

    * [Adding an API Key to an imported RestApi](#adding-an-api-key-to-an-imported-restapi)
    * [⚠️ Multiple API Keys](#%EF%B8%8F-multiple-api-keys)
    * [Rate Limited API Key](#rate-limited-api-key)
  * [Working with models](#working-with-models)
  * [Default Integration and Method Options](#default-integration-and-method-options)
  * [Proxy Routes](#proxy-routes)
  * [Authorizers](#authorizers)

    * [IAM-based authorizer](#iam-based-authorizer)
    * [Lambda-based token authorizer](#lambda-based-token-authorizer)
    * [Lambda-based request authorizer](#lambda-based-request-authorizer)
    * [Cognito User Pools authorizer](#cognito-user-pools-authorizer)
  * [Mutual TLS (mTLS)](#mutual-tls-mtls)
  * [Deployments](#deployments)

    * [Deep dive: Invalidation of deployments](#deep-dive-invalidation-of-deployments)
  * [Custom Domains](#custom-domains)

    * [Custom Domains with multi-level api mapping](#custom-domains-with-multi-level-api-mapping)
  * [Access Logging](#access-logging)
  * [Cross Origin Resource Sharing (CORS)](#cross-origin-resource-sharing-cors)
  * [Endpoint Configuration](#endpoint-configuration)
  * [Private Integrations](#private-integrations)
  * [Gateway response](#gateway-response)
  * [OpenAPI Definition](#openapi-definition)

    * [Endpoint configuration](#endpoint-configuration-1)
  * [Metrics](#metrics)
  * [APIGateway v2](#apigateway-v2)

## Defining APIs

APIs are defined as a hierarchy of resources and methods. `addResource` and
`addMethod` can be used to build this hierarchy. The root resource is
`api.root`.

For example, the following code defines an API that includes the following HTTP
endpoints: `ANY /`, `GET /books`, `POST /books`, `GET /books/{book_id}`, `DELETE /books/{book_id}`.

```python
api = apigateway.RestApi(self, "books-api")

api.root.add_method("ANY")

books = api.root.add_resource("books")
books.add_method("GET")
books.add_method("POST")

book = books.add_resource("{book_id}")
book.add_method("GET")
book.add_method("DELETE")
```

To give an IAM User or Role permission to invoke a method, use `grantExecute`:

```python
# api: apigateway.RestApi
# user: iam.User


method = api.root.add_resource("books").add_method("GET")
method.grant_execute(user)
```

### Breaking up Methods and Resources across Stacks

It is fairly common for REST APIs with a large number of Resources and Methods to hit the [CloudFormation
limit](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cloudformation-limits.html) of 500 resources per
stack.

To help with this, Resources and Methods for the same REST API can be re-organized across multiple stacks. A common
way to do this is to have a stack per Resource or groups of Resources, but this is not the only possible way.
The following example uses sets up two Resources '/pets' and '/books' in separate stacks using nested stacks:

```python
from aws_cdk.aws_apigateway import IntegrationResponse, MethodResponse, IntegrationResponse, MethodResponse
from constructs import Construct
from aws_cdk import App, CfnOutput, NestedStack, NestedStackProps, Stack
from aws_cdk.aws_apigateway import Deployment, Method, MockIntegration, PassthroughBehavior, RestApi, Stage

#
# This file showcases how to split up a RestApi's Resources and Methods across nested stacks.
#
# The root stack 'RootStack' first defines a RestApi.
# Two nested stacks BooksStack and PetsStack, create corresponding Resources '/books' and '/pets'.
# They are then deployed to a 'prod' Stage via a third nested stack - DeployStack.
#
# To verify this worked, go to the APIGateway
#

class RootStack(Stack):
    def __init__(self, scope):
        super().__init__(scope, "integ-restapi-import-RootStack")

        rest_api = RestApi(self, "RestApi",
            cloud_watch_role=True,
            deploy=False
        )
        rest_api.root.add_method("ANY")

        pets_stack = PetsStack(self,
            rest_api_id=rest_api.rest_api_id,
            root_resource_id=rest_api.rest_api_root_resource_id
        )
        books_stack = BooksStack(self,
            rest_api_id=rest_api.rest_api_id,
            root_resource_id=rest_api.rest_api_root_resource_id
        )
        DeployStack(self,
            rest_api_id=rest_api.rest_api_id,
            methods=pets_stack.methods.concat(books_stack.methods)
        )

        CfnOutput(self, "PetsURL",
            value=f"https://{restApi.restApiId}.execute-api.{this.region}.amazonaws.com/prod/pets"
        )

        CfnOutput(self, "BooksURL",
            value=f"https://{restApi.restApiId}.execute-api.{this.region}.amazonaws.com/prod/books"
        )

class PetsStack(NestedStack):

    def __init__(self, scope, *, restApiId, rootResourceId, parameters=None, timeout=None, notificationArns=None, removalPolicy=None, description=None):
        super().__init__(scope, "integ-restapi-import-PetsStack", restApiId=restApiId, rootResourceId=rootResourceId, parameters=parameters, timeout=timeout, notificationArns=notificationArns, removalPolicy=removalPolicy, description=description)

        api = RestApi.from_rest_api_attributes(self, "RestApi",
            rest_api_id=rest_api_id,
            root_resource_id=root_resource_id
        )

        method = api.root.add_resource("pets").add_method("GET", MockIntegration(
            integration_responses=[IntegrationResponse(
                status_code="200"
            )],
            passthrough_behavior=PassthroughBehavior.NEVER,
            request_templates={
                "application/json": "{ \"statusCode\": 200 }"
            }
        ),
            method_responses=[MethodResponse(status_code="200")]
        )

        self.methods.push(method)

class BooksStack(NestedStack):

    def __init__(self, scope, *, restApiId, rootResourceId, parameters=None, timeout=None, notificationArns=None, removalPolicy=None, description=None):
        super().__init__(scope, "integ-restapi-import-BooksStack", restApiId=restApiId, rootResourceId=rootResourceId, parameters=parameters, timeout=timeout, notificationArns=notificationArns, removalPolicy=removalPolicy, description=description)

        api = RestApi.from_rest_api_attributes(self, "RestApi",
            rest_api_id=rest_api_id,
            root_resource_id=root_resource_id
        )

        method = api.root.add_resource("books").add_method("GET", MockIntegration(
            integration_responses=[IntegrationResponse(
                status_code="200"
            )],
            passthrough_behavior=PassthroughBehavior.NEVER,
            request_templates={
                "application/json": "{ \"statusCode\": 200 }"
            }
        ),
            method_responses=[MethodResponse(status_code="200")]
        )

        self.methods.push(method)

class DeployStack(NestedStack):
    def __init__(self, scope, *, restApiId, methods=None, parameters=None, timeout=None, notificationArns=None, removalPolicy=None, description=None):
        super().__init__(scope, "integ-restapi-import-DeployStack", restApiId=restApiId, methods=methods, parameters=parameters, timeout=timeout, notificationArns=notificationArns, removalPolicy=removalPolicy, description=description)

        deployment = Deployment(self, "Deployment",
            api=RestApi.from_rest_api_id(self, "RestApi", rest_api_id)
        )
        if methods:
            for method in methods:
                deployment.node.add_dependency(method)
        Stage(self, "Stage", deployment=deployment)

RootStack(App())
```

> **Warning:** In the code above, an API Gateway deployment is created during the initial CDK deployment.
> However, if there are changes to the resources in subsequent CDK deployments, a new API Gateway deployment is not
> automatically created. As a result, the latest state of the resources is not reflected. To ensure the latest state
> of the resources is reflected, a manual deployment of the API Gateway is required after the CDK deployment. See [Controlled triggering of deployments](#controlled-triggering-of-deployments) for more info.

## AWS Lambda-backed APIs

A very common practice is to use Amazon API Gateway with AWS Lambda as the
backend integration. The `LambdaRestApi` construct makes it easy:

The following code defines a REST API that routes all requests to the
specified AWS Lambda function:

```python
# backend: lambda.Function

apigateway.LambdaRestApi(self, "myapi",
    handler=backend
)
```

You can also supply `proxy: false`, in which case you will have to explicitly
define the API model:

```python
# backend: lambda.Function

api = apigateway.LambdaRestApi(self, "myapi",
    handler=backend,
    proxy=False
)

items = api.root.add_resource("items")
items.add_method("GET") # GET /items
items.add_method("POST") # POST /items

item = items.add_resource("{item}")
item.add_method("GET") # GET /items/{item}

# the default integration for methods is "handler", but one can
# customize this behavior per method or even a sub path.
item.add_method("DELETE", apigateway.HttpIntegration("http://amazon.com"))
```

Additionally, `integrationOptions` can be supplied to explicitly define
options of the Lambda integration:

```python
# backend: lambda.Function


api = apigateway.LambdaRestApi(self, "myapi",
    handler=backend,
    integration_options=apigateway.LambdaIntegrationOptions(
        allow_test_invoke=False,
        timeout=Duration.seconds(1)
    )
)
```

## AWS StepFunctions backed APIs

You can use Amazon API Gateway with AWS Step Functions as the backend integration, specifically Synchronous Express Workflows.

The `StepFunctionsRestApi` only supports integration with Synchronous Express state machine. The `StepFunctionsRestApi` construct makes this easy by setting up input, output and error mapping.

The construct sets up an API endpoint and maps the `ANY` HTTP method and any calls to the API endpoint starts an express workflow execution for the underlying state machine.

Invoking the endpoint with any HTTP method (`GET`, `POST`, `PUT`, `DELETE`, ...) in the example below will send the request to the state machine as a new execution. On success, an HTTP code `200` is returned with the execution output as the Response Body.

If the execution fails, an HTTP `500` response is returned with the `error` and `cause` from the execution output as the Response Body. If the request is invalid (ex. bad execution input) HTTP code `400` is returned.

To disable default response models generation use the `useDefaultMethodResponses` property:

```python
# machine: stepfunctions.IStateMachine


apigateway.StepFunctionsRestApi(self, "StepFunctionsRestApi",
    state_machine=machine,
    use_default_method_responses=False
)
```

The response from the invocation contains only the `output` field from the
[StartSyncExecution](https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartSyncExecution.html#API_StartSyncExecution_ResponseSyntax) API.
In case of failures, the fields `error` and `cause` are returned as part of the response.
Other metadata such as billing details, AWS account ID and resource ARNs are not returned in the API response.

By default, a `prod` stage is provisioned.

In order to reduce the payload size sent to AWS Step Functions, `headers` are not forwarded to the Step Functions execution input. It is possible to choose whether `headers`,  `requestContext`, `path`, `querystring`, and `authorizer` are included or not. By default, `headers` are excluded in all requests.

More details about AWS Step Functions payload limit can be found at https://docs.aws.amazon.com/step-functions/latest/dg/limits-overview.html#service-limits-task-executions.

The following code defines a REST API that routes all requests to the specified AWS StepFunctions state machine:

```python
state_machine_definition = stepfunctions.Pass(self, "PassState")

state_machine = stepfunctions.StateMachine(self, "StateMachine",
    definition=state_machine_definition,
    state_machine_type=stepfunctions.StateMachineType.EXPRESS
)

apigateway.StepFunctionsRestApi(self, "StepFunctionsRestApi",
    deploy=True,
    state_machine=state_machine
)
```

When the REST API endpoint configuration above is invoked using POST, as follows -

```bash
curl -X POST -d '{ "customerId": 1 }' https://example.com/
```

AWS Step Functions will receive the request body in its input as follows:

```json
{
  "body": {
    "customerId": 1
  },
  "path": "/",
  "querystring": {}
}
```

When the endpoint is invoked at path '/users/5' using the HTTP GET method as below:

```bash
curl -X GET https://example.com/users/5?foo=bar
```

AWS Step Functions will receive the following execution input:

```json
{
  "body": {},
  "path": {
     "users": "5"
  },
  "querystring": {
    "foo": "bar"
  }
}
```

Additional information around the request such as the request context, authorizer context, and headers can be included as part of the input
forwarded to the state machine. The following example enables headers to be included in the input but not query string.

```python
apigateway.StepFunctionsRestApi(self, "StepFunctionsRestApi",
    state_machine=machine,
    headers=True,
    path=False,
    querystring=False,
    authorizer=False,
    request_context=apigateway.RequestContext(
        caller=True,
        user=True
    )
)
```

In such a case, when the endpoint is invoked as below:

```bash
curl -X GET https://example.com/
```

AWS Step Functions will receive the following execution input:

```json
{
  "headers": {
    "Accept": "...",
    "CloudFront-Forwarded-Proto": "...",
  },
  "requestContext": {
     "accountId": "...",
     "apiKey": "...",
  },
  "body": {}
}
```

## Integration Targets

Methods are associated with backend integrations, which are invoked when this
method is called. API Gateway supports the following integrations:

* `MockIntegration` - can be used to test APIs. This is the default
  integration if one is not specified.
* `AwsIntegration` - can be used to invoke arbitrary AWS service APIs.
* `HttpIntegration` - can be used to invoke HTTP endpoints.
* `LambdaIntegration` - can be used to invoke an AWS Lambda function.
* `SagemakerIntegration` - can be used to invoke Sagemaker Endpoints.

The following example shows how to integrate the `GET /book/{book_id}` method to
an AWS Lambda function:

```python
# get_book_handler: lambda.Function
# book: apigateway.Resource


get_book_integration = apigateway.LambdaIntegration(get_book_handler)
book.add_method("GET", get_book_integration)
```

Integration options can be optionally be specified:

```python
# get_book_handler: lambda.Function
# get_book_integration: apigateway.LambdaIntegration


get_book_integration = apigateway.LambdaIntegration(get_book_handler,
    content_handling=apigateway.ContentHandling.CONVERT_TO_TEXT,  # convert to base64
    credentials_passthrough=True
)
```

Method options can optionally be specified when adding methods:

```python
# book: apigateway.Resource
# get_book_integration: apigateway.LambdaIntegration


book.add_method("GET", get_book_integration,
    authorization_type=apigateway.AuthorizationType.IAM,
    api_key_required=True
)
```

It is possible to also integrate with AWS services in a different region. The following code integrates with Amazon SQS in the
`eu-west-1` region.

```python
get_message_integration = apigateway.AwsIntegration(
    service="sqs",
    path="queueName",
    region="eu-west-1"
)
```

## Usage Plan & API Keys

A usage plan specifies who can access one or more deployed API stages and methods, and the rate at which they can be
accessed. The plan uses API keys to identify API clients and meters access to the associated API stages for each key.
Usage plans also allow configuring throttling limits and quota limits that are enforced on individual client API keys.

The following example shows how to create and associate a usage plan and an API key:

```python
# integration: apigateway.LambdaIntegration


api = apigateway.RestApi(self, "hello-api")

v1 = api.root.add_resource("v1")
echo = v1.add_resource("echo")
echo_method = echo.add_method("GET", integration, api_key_required=True)

plan = api.add_usage_plan("UsagePlan",
    name="Easy",
    throttle=apigateway.ThrottleSettings(
        rate_limit=10,
        burst_limit=2
    )
)

key = api.add_api_key("ApiKey")
plan.add_api_key(key)
```

To associate a plan to a given RestAPI stage:

```python
# plan: apigateway.UsagePlan
# api: apigateway.RestApi
# echo_method: apigateway.Method


plan.add_api_stage(
    stage=api.deployment_stage,
    throttle=[apigateway.ThrottlingPerMethod(
        method=echo_method,
        throttle=apigateway.ThrottleSettings(
            rate_limit=10,
            burst_limit=2
        )
    )
    ]
)
```

Existing usage plans can be imported into a CDK app using its id.

```python
imported_usage_plan = apigateway.UsagePlan.from_usage_plan_id(self, "imported-usage-plan", "<usage-plan-key-id>")
```

The name and value of the API Key can be specified at creation; if not
provided, a name and value will be automatically generated by API Gateway.

```python
# api: apigateway.RestApi

key = api.add_api_key("ApiKey",
    api_key_name="myApiKey1",
    value="MyApiKeyThatIsAtLeast20Characters"
)
```

Existing API keys can also be imported into a CDK app using its id.

```python
imported_key = apigateway.ApiKey.from_api_key_id(self, "imported-key", "<api-key-id>")
```

The "grant" methods can be used to give prepackaged sets of permissions to other resources. The
following code provides read permission to an API key.

```python
# imported_key: apigateway.ApiKey
# lambda_fn: lambda.Function

imported_key.grant_read(lambda_fn)
```

### Adding an API Key to an imported RestApi

API Keys are added to ApiGateway Stages, not to the API itself. When you import a RestApi
it does not have any information on the Stages that may be associated with it. Since adding an API
Key requires a stage, you should instead add the Api Key to the imported Stage.

```python
# rest_api: apigateway.IRestApi

imported_stage = apigateway.Stage.from_stage_attributes(self, "imported-stage",
    stage_name="myStageName",
    rest_api=rest_api
)

imported_stage.add_api_key("MyApiKey")
```

### ⚠️ Multiple API Keys

It is possible to specify multiple API keys for a given Usage Plan, by calling `usagePlan.addApiKey()`.

When using multiple API keys, a past bug of the CDK prevents API key associations to a Usage Plan to be deleted.
If the CDK app had the [feature flag](https://docs.aws.amazon.com/cdk/latest/guide/featureflags.html) - `@aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId` - enabled when the API
keys were created, then the app will not be affected by this bug.

If this is not the case, you will need to ensure that the CloudFormation [logical ids](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html) of the API keys that are not
being deleted remain unchanged.
Make note of the logical ids of these API keys before removing any, and set it as part of the `addApiKey()` method:

```python
# usageplan: apigateway.UsagePlan
# api_key: apigateway.ApiKey


usageplan.add_api_key(api_key,
    override_logical_id="..."
)
```

### Rate Limited API Key

In scenarios where you need to create a single api key and configure rate limiting for it, you can use `RateLimitedApiKey`.
This construct lets you specify rate limiting properties which should be applied only to the api key being created.
The API key created has the specified rate limits, such as quota and throttles, applied.

The following example shows how to use a rate limited api key :

```python
# api: apigateway.RestApi


key = apigateway.RateLimitedApiKey(self, "rate-limited-api-key",
    customer_id="hello-customer",
    api_stages=[apigateway.UsagePlanPerApiStage(stage=api.deployment_stage)],
    quota=apigateway.QuotaSettings(
        limit=10000,
        period=apigateway.Period.MONTH
    )
)
```

## Working with models

When you work with Lambda integrations that are not Proxy integrations, you
have to define your models and mappings for the request, response, and integration.

```python
hello = lambda_.Function(self, "hello",
    runtime=lambda_.Runtime.NODEJS_LATEST,
    handler="hello.handler",
    code=lambda_.Code.from_asset("lambda")
)

api = apigateway.RestApi(self, "hello-api")
resource = api.root.add_resource("v1")
```

You can define more parameters on the integration to tune the behavior of API Gateway

```python
# hello: lambda.Function


integration = apigateway.LambdaIntegration(hello,
    proxy=False,
    request_parameters={
        # You can define mapping parameters from your method to your integration
        # - Destination parameters (the key) are the integration parameters (used in mappings)
        # - Source parameters (the value) are the source request parameters or expressions
        # @see: https://docs.aws.amazon.com/apigateway/latest/developerguide/request-response-data-mappings.html
        "integration.request.querystring.who": "method.request.querystring.who"
    },
    allow_test_invoke=True,
    request_templates={
        # You can define a mapping that will build a payload for your integration, based
        #  on the integration parameters that you have specified
        # Check: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
        "application/json": JSON.stringify({"action": "sayHello", "poll_id": "$util.escapeJavaScript($input.params('who'))"})
    },
    # This parameter defines the behavior of the engine is no suitable response template is found
    passthrough_behavior=apigateway.PassthroughBehavior.NEVER,
    integration_responses=[apigateway.IntegrationResponse(
        # Successful response from the Lambda function, no filter defined
        #  - the selectionPattern filter only tests the error message
        # We will set the response status code to 200
        status_code="200",
        response_templates={
            # This template takes the "message" result from the Lambda function, and embeds it in a JSON response
            # Check https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
            "application/json": JSON.stringify({"state": "ok", "greeting": "$util.escapeJavaScript($input.body)"})
        },
        response_parameters={
            # We can map response parameters
            # - Destination parameters (the key) are the response parameters (used in mappings)
            # - Source parameters (the value) are the integration response parameters or expressions
            "method.response.header.Content-Type": "'application/json'",
            "method.response.header.Access-Control-Allow-Origin": "'*'",
            "method.response.header.Access-Control-Allow-Credentials": "'true'"
        }
    ), apigateway.IntegrationResponse(
        # For errors, we check if the error message is not empty, get the error data
        selection_pattern="(\n|.)+",
        # We will set the response status code to 200
        status_code="400",
        response_templates={
            "application/json": JSON.stringify({"state": "error", "message": "$util.escapeJavaScript($input.path('$.errorMessage'))"})
        },
        response_parameters={
            "method.response.header.Content-Type": "'application/json'",
            "method.response.header.Access-Control-Allow-Origin": "'*'",
            "method.response.header.Access-Control-Allow-Credentials": "'true'"
        }
    )
    ]
)
```

You can define models for your responses (and requests)

```python
# api: apigateway.RestApi


# We define the JSON Schema for the transformed valid response
response_model = api.add_model("ResponseModel",
    content_type="application/json",
    model_name="ResponseModel",
    schema=apigateway.JsonSchema(
        schema=apigateway.JsonSchemaVersion.DRAFT4,
        title="pollResponse",
        type=apigateway.JsonSchemaType.OBJECT,
        properties={
            "state": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING),
            "greeting": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING)
        }
    )
)

# We define the JSON Schema for the transformed error response
error_response_model = api.add_model("ErrorResponseModel",
    content_type="application/json",
    model_name="ErrorResponseModel",
    schema=apigateway.JsonSchema(
        schema=apigateway.JsonSchemaVersion.DRAFT4,
        title="errorResponse",
        type=apigateway.JsonSchemaType.OBJECT,
        properties={
            "state": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING),
            "message": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING)
        }
    )
)
```

And reference all on your method definition.

```python
# integration: apigateway.LambdaIntegration
# resource: apigateway.Resource
# response_model: apigateway.Model
# error_response_model: apigateway.Model


resource.add_method("GET", integration,
    # We can mark the parameters as required
    request_parameters={
        "method.request.querystring.who": True
    },
    # we can set request validator options like below
    request_validator_options=apigateway.RequestValidatorOptions(
        request_validator_name="test-validator",
        validate_request_body=True,
        validate_request_parameters=False
    ),
    method_responses=[apigateway.MethodResponse(
        # Successful response from the integration
        status_code="200",
        # Define what parameters are allowed or not
        response_parameters={
            "method.response.header.Content-Type": True,
            "method.response.header.Access-Control-Allow-Origin": True,
            "method.response.header.Access-Control-Allow-Credentials": True
        },
        # Validate the schema on the response
        response_models={
            "application/json": response_model
        }
    ), apigateway.MethodResponse(
        # Same thing for the error responses
        status_code="400",
        response_parameters={
            "method.response.header.Content-Type": True,
            "method.response.header.Access-Control-Allow-Origin": True,
            "method.response.header.Access-Control-Allow-Credentials": True
        },
        response_models={
            "application/json": error_response_model
        }
    )
    ]
)
```

Specifying `requestValidatorOptions` automatically creates the RequestValidator construct with the given options.
However, if you have your RequestValidator already initialized or imported, use the `requestValidator` option instead.

If you want to use `requestValidatorOptions` in multiple `addMethod()` calls
then you need to set the `@aws-cdk/aws-apigateway:requestValidatorUniqueId`
feature flag. When this feature flag is set, each `RequestValidator` will have a unique generated id.

> **Note** if you enable this feature flag when you have already used
> `addMethod()` with `requestValidatorOptions` the Logical Id of the resource
> will change causing the resource to be replaced.

```python
# integration: apigateway.LambdaIntegration
# resource: apigateway.Resource
# response_model: apigateway.Model
# error_response_model: apigateway.Model


resource.node.set_context("@aws-cdk/aws-apigateway:requestValidatorUniqueId", True)

resource.add_method("GET", integration,
    # we can set request validator options like below
    request_validator_options=apigateway.RequestValidatorOptions(
        request_validator_name="test-validator",
        validate_request_body=True,
        validate_request_parameters=False
    )
)

resource.add_method("POST", integration,
    # we can set request validator options like below
    request_validator_options=apigateway.RequestValidatorOptions(
        request_validator_name="test-validator2",
        validate_request_body=True,
        validate_request_parameters=False
    )
)
```

## Default Integration and Method Options

The `defaultIntegration` and `defaultMethodOptions` properties can be used to
configure a default integration at any resource level. These options will be
used when defining method under this resource (recursively) with undefined
integration or options.

> If not defined, the default integration is `MockIntegration`. See reference
> documentation for default method options.

The following example defines the `booksBackend` integration as a default
integration. This means that all API methods that do not explicitly define an
integration will be routed to this AWS Lambda function.

```python
# books_backend: apigateway.LambdaIntegration

api = apigateway.RestApi(self, "books",
    default_integration=books_backend
)

books = api.root.add_resource("books")
books.add_method("GET") # integrated with `booksBackend`
books.add_method("POST") # integrated with `booksBackend`

book = books.add_resource("{book_id}")
book.add_method("GET")
```

A Method can be configured with authorization scopes. Authorization scopes are
used in conjunction with an [authorizer that uses Amazon Cognito user
pools](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html#apigateway-enable-cognito-user-pool).
Read more about authorization scopes
[here](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizationscopes).

Authorization scopes for a Method can be configured using the `authorizationScopes` property as shown below -

```python
# books: apigateway.Resource


books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
    authorization_type=apigateway.AuthorizationType.COGNITO,
    authorization_scopes=["Scope1", "Scope2"]
)
```

## Proxy Routes

The `addProxy` method can be used to install a greedy `{proxy+}` resource
on a path. By default, this also installs an `"ANY"` method:

```python
# resource: apigateway.Resource
# handler: lambda.Function

proxy = resource.add_proxy(
    default_integration=apigateway.LambdaIntegration(handler),

    # "false" will require explicitly adding methods on the `proxy` resource
    any_method=True
)
```

## Authorizers

API Gateway [supports several different authorization types](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-to-api.html)
that can be used for controlling access to your REST APIs.

### IAM-based authorizer

The following CDK code provides 'execute-api' permission to an IAM user, via IAM policies, for the 'GET' method on the `books` resource:

```python
# books: apigateway.Resource
# iam_user: iam.User


get_books = books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
    authorization_type=apigateway.AuthorizationType.IAM
)

iam_user.attach_inline_policy(iam.Policy(self, "AllowBooks",
    statements=[
        iam.PolicyStatement(
            actions=["execute-api:Invoke"],
            effect=iam.Effect.ALLOW,
            resources=[get_books.method_arn]
        )
    ]
))
```

### Lambda-based token authorizer

API Gateway also allows [lambda functions to be used as authorizers](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html).

This module provides support for token-based Lambda authorizers. When a client makes a request to an API's methods configured with such
an authorizer, API Gateway calls the Lambda authorizer, which takes the caller's identity as input and returns an IAM policy as output.
A token-based Lambda authorizer (also called a token authorizer) receives the caller's identity in a bearer token, such as
a JSON Web Token (JWT) or an OAuth token.

API Gateway interacts with the authorizer Lambda function handler by passing input and expecting the output in a specific format.
The event object that the handler is called with contains the `authorizationToken` and the `methodArn` from the request to the
API Gateway endpoint. The handler is expected to return the `principalId` (i.e. the client identifier) and a `policyDocument` stating
what the client is authorizer to perform.
See [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html) for a detailed specification on
inputs and outputs of the Lambda handler.

The following code attaches a token-based Lambda authorizer to the 'GET' Method of the Book resource:

```python
# auth_fn: lambda.Function
# books: apigateway.Resource


auth = apigateway.TokenAuthorizer(self, "booksAuthorizer",
    handler=auth_fn
)

books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
    authorizer=auth
)
```

A full working example is shown below.

```python
class MyStack(Stack):
    def __init__(self, scope, id):
        super().__init__(scope, id)

        authorizer_fn = lambda_.Function(self, "MyAuthorizerFunction",
            runtime=lambda_.Runtime.NODEJS_LATEST,
            handler="index.handler",
            code=lambda_.AssetCode.from_asset(path.join(__dirname, "integ.token-authorizer.handler"))
        )

        authorizer = TokenAuthorizer(self, "MyAuthorizer",
            handler=authorizer_fn
        )

        restapi = RestApi(self, "MyRestApi",
            cloud_watch_role=True,
            default_method_options=MethodOptions(
                authorizer=authorizer
            ),
            default_cors_preflight_options=CorsOptions(
                allow_origins=Cors.ALL_ORIGINS
            )
        )

        restapi.root.add_method("ANY", MockIntegration(
            integration_responses=[IntegrationResponse(status_code="200")
            ],
            passthrough_behavior=PassthroughBehavior.NEVER,
            request_templates={
                "application/json": "{ \"statusCode\": 200 }"
            }
        ),
            method_responses=[MethodResponse(status_code="200")
            ]
        )
```

By default, the `TokenAuthorizer` looks for the authorization token in the request header with the key 'Authorization'. This can,
however, be modified by changing the `identitySource` property.

Authorizers can also be passed via the `defaultMethodOptions` property within the `RestApi` construct or the `Method` construct. Unless
explicitly overridden, the specified defaults will be applied across all `Method`s across the `RestApi` or across all `Resource`s,
depending on where the defaults were specified.

### Lambda-based request authorizer

This module provides support for request-based Lambda authorizers. When a client makes a request to an API's methods configured with such
an authorizer, API Gateway calls the Lambda authorizer, which takes specified parts of the request, known as identity sources,
as input and returns an IAM policy as output. A request-based Lambda authorizer (also called a request authorizer) receives
the identity sources in a series of values pulled from the request, from the headers, stage variables, query strings, and the context.

API Gateway interacts with the authorizer Lambda function handler by passing input and expecting the output in a specific format.
The event object that the handler is called with contains the body of the request and the `methodArn` from the request to the
API Gateway endpoint. The handler is expected to return the `principalId` (i.e. the client identifier) and a `policyDocument` stating
what the client is authorizer to perform.
See [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html) for a detailed specification on
inputs and outputs of the Lambda handler.

The following code attaches a request-based Lambda authorizer to the 'GET' Method of the Book resource:

```python
# auth_fn: lambda.Function
# books: apigateway.Resource


auth = apigateway.RequestAuthorizer(self, "booksAuthorizer",
    handler=auth_fn,
    identity_sources=[apigateway.IdentitySource.header("Authorization")]
)

books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
    authorizer=auth
)
```

A full working example is shown below.

```python
from aws_cdk.aws_apigateway import IntegrationResponse, MethodResponse, IntegrationResponse, MethodResponse
import path as path
import aws_cdk.aws_lambda as lambda_
from aws_cdk import App, Stack
from aws_cdk.aws_apigateway import MockIntegration, PassthroughBehavior, RestApi, RequestAuthorizer, IdentitySource

# Against the RestApi endpoint from the stack output, run
# `curl -s -o /dev/null -w "%{http_code}" <url>` should return 401
# `curl -s -o /dev/null -w "%{http_code}" -H 'Authorization: deny' <url>?allow=yes` should return 403
# `curl -s -o /dev/null -w "%{http_code}" -H 'Authorization: allow' <url>?allow=yes` should return 200

app = App()
stack = Stack(app, "RequestAuthorizerInteg")

authorizer_fn = lambda_.Function(stack, "MyAuthorizerFunction",
    runtime=lambda_.Runtime.NODEJS_LATEST,
    handler="index.handler",
    code=lambda_.AssetCode.from_asset(path.join(__dirname, "integ.request-authorizer.handler"))
)

restapi = RestApi(stack, "MyRestApi", cloud_watch_role=True)

authorizer = RequestAuthorizer(stack, "MyAuthorizer",
    handler=authorizer_fn,
    identity_sources=[IdentitySource.header("Authorization"), IdentitySource.query_string("allow")]
)

second_authorizer = RequestAuthorizer(stack, "MySecondAuthorizer",
    handler=authorizer_fn,
    identity_sources=[IdentitySource.header("Authorization"), IdentitySource.query_string("allow")]
)

restapi.root.add_method("ANY", MockIntegration(
    integration_responses=[IntegrationResponse(status_code="200")
    ],
    passthrough_behavior=PassthroughBehavior.NEVER,
    request_templates={
        "application/json": "{ \"statusCode\": 200 }"
    }
),
    method_responses=[MethodResponse(status_code="200")
    ],
    authorizer=authorizer
)

restapi.root.resource_for_path("auth").add_method("ANY", MockIntegration(
    integration_responses=[IntegrationResponse(status_code="200")
    ],
    passthrough_behavior=PassthroughBehavior.NEVER,
    request_templates={
        "application/json": "{ \"statusCode\": 200 }"
    }
),
    method_responses=[MethodResponse(status_code="200")
    ],
    authorizer=second_authorizer
)
```

By default, the `RequestAuthorizer` does not pass any kind of information from the request. This can,
however, be modified by changing the `identitySource` property, and is required when specifying a value for caching.

Authorizers can also be passed via the `defaultMethodOptions` property within the `RestApi` construct or the `Method` construct. Unless
explicitly overridden, the specified defaults will be applied across all `Method`s across the `RestApi` or across all `Resource`s,
depending on where the defaults were specified.

### Cognito User Pools authorizer

API Gateway also allows [Amazon Cognito user pools as authorizer](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html)

The following snippet configures a Cognito user pool as an authorizer:

```python
# books: apigateway.Resource
user_pool = cognito.UserPool(self, "UserPool")

auth = apigateway.CognitoUserPoolsAuthorizer(self, "booksAuthorizer",
    cognito_user_pools=[user_pool]
)
books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
    authorizer=auth,
    authorization_type=apigateway.AuthorizationType.COGNITO
)
```

## Mutual TLS (mTLS)

Mutual TLS can be configured to limit access to your API based by using client certificates instead of (or as an extension of) using authorization headers.

```python
# acm: Any


apigateway.DomainName(self, "domain-name",
    domain_name="example.com",
    certificate=acm.Certificate.from_certificate_arn(self, "cert", "arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d"),
    mtls=apigateway.MTLSConfig(
        bucket=s3.Bucket(self, "bucket"),
        key="truststore.pem",
        version="version"
    )
)
```

Instructions for configuring your trust store can be found [here](https://aws.amazon.com/blogs/compute/introducing-mutual-tls-authentication-for-amazon-api-gateway/).

## Deployments

By default, the `RestApi` construct will automatically create an API Gateway
[Deployment](https://docs.aws.amazon.com/apigateway/api-reference/resource/deployment/) and a "prod" [Stage](https://docs.aws.amazon.com/apigateway/api-reference/resource/stage/) which represent the API configuration you
defined in your CDK app. This means that when you deploy your app, your API will
have open access from the internet via the stage URL.

The URL of your API can be obtained from the attribute `restApi.url`, and is
also exported as an `Output` from your stack, so it's printed when you `cdk deploy` your app:

```console
$ cdk deploy
...
books.booksapiEndpointE230E8D5 = https://6lyktd4lpk.execute-api.us-east-1.amazonaws.com/prod/
```

To disable this behavior, you can set `{ deploy: false }` when creating your
API. This means that the API will not be deployed and a stage will not be
created for it. You will need to manually define a `apigateway.Deployment` and
`apigateway.Stage` resources.

Use the `deployOptions` property to customize the deployment options of your
API.

The following example will configure API Gateway to emit logs and data traces to
AWS CloudWatch for all API calls:

> Note: whether or not this is enabled or disabled by default is controlled by the
> `@aws-cdk/aws-apigateway:disableCloudWatchRole` feature flag. When this feature flag
> is set to `false` the default behavior will set `cloudWatchRole=true`

This is controlled via the `@aws-cdk/aws-apigateway:disableCloudWatchRole` feature flag and
is disabled by default. When enabled (or `@aws-cdk/aws-apigateway:disableCloudWatchRole=false`),
an IAM role will be created and associated with API Gateway to allow it to write logs and metrics to AWS CloudWatch.

```python
api = apigateway.RestApi(self, "books",
    cloud_watch_role=True,
    deploy_options=apigateway.StageOptions(
        logging_level=apigateway.MethodLoggingLevel.INFO,
        data_trace_enabled=True
    )
)
```

> Note: there can only be a single apigateway.CfnAccount per AWS environment
> so if you create multiple `RestApi`s with `cloudWatchRole=true` each new `RestApi`
> will overwrite the `CfnAccount`. It is recommended to set `cloudWatchRole=false`
> (the default behavior if `@aws-cdk/aws-apigateway:disableCloudWatchRole` is enabled)
> and only create a single CloudWatch role and account per environment.

You can specify the CloudWatch Role and Account sub-resources removal policy with the
`cloudWatchRoleRemovalPolicy` property, which defaults to `RemovalPolicy.RETAIN`.
This option requires `cloudWatchRole` to be enabled.

```python
import aws_cdk as cdk


api = apigateway.RestApi(self, "books",
    cloud_watch_role=True,
    cloud_watch_role_removal_policy=cdk.RemovalPolicy.DESTROY
)
```

### Deploying to an existing stage

#### Using RestApi

If you want to use an existing stage to deploy your `RestApi`, first set `{ deploy: false }` so the construct doesn't automatically create new `Deployment` and `Stage` resources.  Then you can manually define a `apigateway.Deployment` resource and specify the stage name for your existing stage using the `stageName` property.

Note that as long as the deployment's logical ID doesn't change, it will represent the snapshot in time when the resource was created. To ensure your deployment reflects changes to the `RestApi` model, see [Controlled triggering of deployments](#controlled-triggering-of-deployments).

```python
rest_api = apigateway.RestApi(self, "my-rest-api",
    deploy=False
)

# Use `stageName` to deploy to an existing stage
deployment = apigateway.Deployment(self, "my-deployment",
    api=rest_api,
    stage_name="dev",
    retain_deployments=True
)
```

#### Using SpecRestApi

If you want to use an existing stage to deploy your `SpecRestApi`, first set `{ deploy: false }` so the construct doesn't automatically create new `Deployment` and `Stage` resources. Then you can manually define a `apigateway.Deployment` resource and specify the stage name for your existing stage using the `stageName` property.

To automatically create a new deployment that reflects the latest API changes, you can use the `addToLogicalId()` method and pass in your OpenAPI definition.

```python
my_api_definition = apigateway.ApiDefinition.from_asset("path-to-file.json")
spec_rest_api = apigateway.SpecRestApi(self, "my-specrest-api",
    deploy=False,
    api_definition=my_api_definition
)

# Use `stageName` to deploy to an existing stage
deployment = apigateway.Deployment(self, "my-deployment",
    api=spec_rest_api,
    stage_name="dev",
    retain_deployments=True
)

# Trigger a new deployment on OpenAPI definition updates
deployment.add_to_logical_id(my_api_definition)
```

> Note: If the `stageName` property is set but a stage with the corresponding name does not exist, a new stage resource will be created with the provided stage name.

> Note: If you update the `stageName` property, you should be triggering a new deployment (i.e. with an updated logical ID and API changes). Otherwise, an error will occur during deployment.

### Controlled triggering of deployments

By default, the `RestApi` construct deploys changes immediately. If you want to
control when deployments happen, set `{ deploy: false }` and create a `Deployment` construct yourself. Add a revision counter to the construct ID, and update it in your source code whenever you want to trigger a new deployment:

```python
rest_api = apigateway.RestApi(self, "my-api",
    deploy=False
)

deployment_revision = 5 # Bump this counter to trigger a new deployment
apigateway.Deployment(self, f"Deployment{deploymentRevision}",
    api=rest_api
)
```

### Deep dive: Invalidation of deployments

API Gateway deployments are an immutable snapshot of the API. This means that we
want to automatically create a new deployment resource every time the API model
defined in our CDK app changes.

In order to achieve that, the AWS CloudFormation logical ID of the
`AWS::ApiGateway::Deployment` resource is dynamically calculated by hashing the
API configuration (resources, methods). This means that when the configuration
changes (i.e. a resource or method are added, configuration is changed), a new
logical ID will be assigned to the deployment resource. This will cause
CloudFormation to create a new deployment resource.

By default, old deployments are *deleted*. You can set `retainDeployments: true`
to allow users revert the stage to an old deployment manually.

In order to also create a new deployment when changes are made to any authorizer attached to the API,
the `@aws-cdk/aws-apigateway:authorizerChangeDeploymentLogicalId` [feature flag](https://docs.aws.amazon.com/cdk/v2/guide/featureflags.html) can be enabled. This can be set
in the `cdk.json` file.

```json
{
  "context": {
    "@aws-cdk/aws-apigateway:authorizerChangeDeploymentLogicalId": true
  }
}
```

## Custom Domains

To associate an API with a custom domain, use the `domainName` configuration when
you define your API:

```python
# acm_certificate_for_example_com: Any


api = apigateway.RestApi(self, "MyDomain",
    domain_name=apigateway.DomainNameOptions(
        domain_name="example.com",
        certificate=acm_certificate_for_example_com
    )
)
```

This will define a `DomainName` resource for you, along with a `BasePathMapping`
from the root of the domain to the deployment stage of the API. This is a common
set up.

To route domain traffic to an API Gateway API, use Amazon Route 53 to create an
alias record. An alias record is a Route 53 extension to DNS. It's similar to a
CNAME record, but you can create an alias record both for the root domain, such
as `example.com`, and for subdomains, such as `www.example.com`. (You can create
CNAME records only for subdomains.)

```python
import aws_cdk.aws_route53 as route53
import aws_cdk.aws_route53_targets as targets

# api: apigateway.RestApi
# hosted_zone_for_example_com: Any


route53.ARecord(self, "CustomDomainAliasRecord",
    zone=hosted_zone_for_example_com,
    target=route53.RecordTarget.from_alias(targets.ApiGateway(api))
)
```

You can also define a `DomainName` resource directly in order to customize the default behavior:

```python
# acm_certificate_for_example_com: Any


apigateway.DomainName(self, "custom-domain",
    domain_name="example.com",
    certificate=acm_certificate_for_example_com,
    endpoint_type=apigateway.EndpointType.EDGE,  # default is REGIONAL
    security_policy=apigateway.SecurityPolicy.TLS_1_2
)
```

Once you have a domain, you can map base paths of the domain to APIs.
The following example will map the URL [https://example.com/go-to-api1](https://example.com/go-to-api1)
to the `api1` API and [https://example.com/boom](https://example.com/boom) to the `api2` API.

```python
# domain: apigateway.DomainName
# api1: apigateway.RestApi
# api2: apigateway.RestApi


domain.add_base_path_mapping(api1, base_path="go-to-api1")
domain.add_base_path_mapping(api2, base_path="boom")
```

By default, the base path URL will map to the `deploymentStage` of the `RestApi`.
You can specify a different API `Stage` to which the base path URL will map to.

```python
# domain: apigateway.DomainName
# restapi: apigateway.RestApi


beta_deploy = apigateway.Deployment(self, "beta-deployment",
    api=restapi
)
beta_stage = apigateway.Stage(self, "beta-stage",
    deployment=beta_deploy
)
domain.add_base_path_mapping(restapi, base_path="api/beta", stage=beta_stage)
```

It is possible to create a base path mapping without associating it with a
stage by using the `attachToStage` property. When set to `false`, the stage must be
included in the URL when invoking the API. For example,
[https://example.com/myapi/prod](https://example.com/myapi/prod) will invoke the stage named `prod` from the
`myapi` base path mapping.

```python
# domain: apigateway.DomainName
# api: apigateway.RestApi


domain.add_base_path_mapping(api, base_path="myapi", attach_to_stage=False)
```

If you don't specify `basePath`, all URLs under this domain will be mapped
to the API, and you won't be able to map another API to the same domain:

```python
# domain: apigateway.DomainName
# api: apigateway.RestApi

domain.add_base_path_mapping(api)
```

This can also be achieved through the `mapping` configuration when defining the
domain as demonstrated above.

Base path mappings can also be created with the `BasePathMapping` resource.

```python
# api: apigateway.RestApi


domain_name = apigateway.DomainName.from_domain_name_attributes(self, "DomainName",
    domain_name="domainName",
    domain_name_alias_hosted_zone_id="domainNameAliasHostedZoneId",
    domain_name_alias_target="domainNameAliasTarget"
)

apigateway.BasePathMapping(self, "BasePathMapping",
    domain_name=domain_name,
    rest_api=api
)
```

If you wish to setup this domain with an Amazon Route53 alias, use the `targets.ApiGatewayDomain`:

```python
# hosted_zone_for_example_com: Any
# domain_name: apigateway.DomainName

import aws_cdk.aws_route53 as route53
import aws_cdk.aws_route53_targets as targets


route53.ARecord(self, "CustomDomainAliasRecord",
    zone=hosted_zone_for_example_com,
    target=route53.RecordTarget.from_alias(targets.ApiGatewayDomain(domain_name))
)
```

### Custom Domains with multi-level api mapping

Additional requirements for creating multi-level path mappings for RestApis:

(both are defaults)

* Must use `SecurityPolicy.TLS_1_2`
* DomainNames must be `EndpointType.REGIONAL`

```python
# acm_certificate_for_example_com: Any
# rest_api: apigateway.RestApi


apigateway.DomainName(self, "custom-domain",
    domain_name="example.com",
    certificate=acm_certificate_for_example_com,
    mapping=rest_api,
    base_path="orders/v1/api"
)
```

To then add additional mappings to a domain you can use the `addApiMapping` method.

```python
# acm_certificate_for_example_com: Any
# rest_api: apigateway.RestApi
# second_rest_api: apigateway.RestApi


domain = apigateway.DomainName(self, "custom-domain",
    domain_name="example.com",
    certificate=acm_certificate_for_example_com,
    mapping=rest_api
)

domain.add_api_mapping(second_rest_api.deployment_stage,
    base_path="orders/v2/api"
)
```

## Access Logging

Access logging creates logs every time an API method is accessed. Access logs can have information on
who has accessed the API, how the caller accessed the API and what responses were generated.
Access logs are configured on a Stage of the RestApi.
Access logs can be expressed in a format of your choosing, and can contain any access details, with a
minimum that it must include either 'requestId' or 'extendedRequestId'. The list of  variables that
can be expressed in the access log can be found
[here](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference).
Read more at [Setting Up CloudWatch API Logging in API
Gateway](https://docs.aws.amazon.com/apigateway/latest/developerguide/set-up-logging.html)

```python
# production stage
prod_log_group = logs.LogGroup(self, "PrdLogs")
api = apigateway.RestApi(self, "books",
    deploy_options=apigateway.StageOptions(
        access_log_destination=apigateway.LogGroupLogDestination(prod_log_group),
        access_log_format=apigateway.AccessLogFormat.json_with_standard_fields()
    )
)
deployment = apigateway.Deployment(self, "Deployment", api=api)

# development stage
dev_log_group = logs.LogGroup(self, "DevLogs")
apigateway.Stage(self, "dev",
    deployment=deployment,
    access_log_destination=apigateway.LogGroupLogDestination(dev_log_group),
    access_log_format=apigateway.AccessLogFormat.json_with_standard_fields(
        caller=False,
        http_method=True,
        ip=True,
        protocol=True,
        request_time=True,
        resource_path=True,
        response_length=True,
        status=True,
        user=True
    )
)
```

The following code will generate the access log in the [CLF format](https://en.wikipedia.org/wiki/Common_Log_Format).

```python
log_group = logs.LogGroup(self, "ApiGatewayAccessLogs")
api = apigateway.RestApi(self, "books",
    deploy_options=apigateway.StageOptions(
        access_log_destination=apigateway.LogGroupLogDestination(log_group),
        access_log_format=apigateway.AccessLogFormat.clf()
    )
)
```

You can also configure your own access log format by using the `AccessLogFormat.custom()` API.
`AccessLogField` provides commonly used fields. The following code configures access log to contain.

```python
log_group = logs.LogGroup(self, "ApiGatewayAccessLogs")
apigateway.RestApi(self, "books",
    deploy_options=apigateway.StageOptions(
        access_log_destination=apigateway.LogGroupLogDestination(log_group),
        access_log_format=apigateway.AccessLogFormat.custom(f"""{apigateway.AccessLogField.contextRequestId()} {apigateway.AccessLogField.contextErrorMessage()} {apigateway.AccessLogField.contextErrorMessageString()}
                  {apigateway.AccessLogField.contextAuthorizerError()} {apigateway.AccessLogField.contextAuthorizerIntegrationStatus()}""")
    )
)
```

You can use the `methodOptions` property to configure
[default method throttling](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html#apigateway-api-level-throttling-in-usage-plan)
for a stage. The following snippet configures the a stage that accepts
100 requests per minute, allowing burst up to 200 requests per minute.

```python
api = apigateway.RestApi(self, "books")
deployment = apigateway.Deployment(self, "my-deployment", api=api)
stage = apigateway.Stage(self, "my-stage",
    deployment=deployment,
    method_options={
        "/*/*": apigateway.MethodDeploymentOptions( # This special path applies to all resource paths and all HTTP methods
            throttling_rate_limit=100,
            throttling_burst_limit=200)
    }
)
```

Configuring `methodOptions` on the `deployOptions` of `RestApi` will set the
throttling behaviors on the default stage that is automatically created.

```python
api = apigateway.RestApi(self, "books",
    deploy_options=apigateway.StageOptions(
        method_options={
            "/*/*": apigateway.MethodDeploymentOptions( # This special path applies to all resource paths and all HTTP methods
                throttling_rate_limit=100,
                throttling_burst_limit=1000)
        }
    )
)
```

To write access log files to a Firehose delivery stream destination use the `FirehoseLogDestination` class:

```python
destination_bucket = s3.Bucket(self, "Bucket")
delivery_stream_role = iam.Role(self, "Role",
    assumed_by=iam.ServicePrincipal("firehose.amazonaws.com")
)

stream = firehose.CfnDeliveryStream(self, "MyStream",
    delivery_stream_name="amazon-apigateway-delivery-stream",
    s3_destination_configuration=firehose.CfnDeliveryStream.S3DestinationConfigurationProperty(
        bucket_arn=destination_bucket.bucket_arn,
        role_arn=delivery_stream_role.role_arn
    )
)

api = apigateway.RestApi(self, "books",
    deploy_options=apigateway.StageOptions(
        access_log_destination=apigateway.FirehoseLogDestination(stream),
        access_log_format=apigateway.AccessLogFormat.json_with_standard_fields()
    )
)
```

**Note:** The delivery stream name must start with `amazon-apigateway-`.

> Visit [Logging API calls to Amazon Data Firehose](https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-logging-to-kinesis.html) for more details.

## Cross Origin Resource Sharing (CORS)

[Cross-Origin Resource Sharing (CORS)](https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS) is a mechanism
that uses additional HTTP headers to tell browsers to give a web application
running at one origin, access to selected resources from a different origin. A
web application executes a cross-origin HTTP request when it requests a resource
that has a different origin (domain, protocol, or port) from its own.

You can add the CORS [preflight](https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS#Preflighted_requests) OPTIONS
HTTP method to any API resource via the `defaultCorsPreflightOptions` option or by calling the `addCorsPreflight` on a specific resource.

The following example will enable CORS for all methods and all origins on all resources of the API:

```python
apigateway.RestApi(self, "api",
    default_cors_preflight_options=apigateway.CorsOptions(
        allow_origins=apigateway.Cors.ALL_ORIGINS,
        allow_methods=apigateway.Cors.ALL_METHODS
    )
)
```

The following example will add an OPTIONS method to the `myResource` API resource, which
only allows GET and PUT HTTP requests from the origin [https://amazon.com.](https://amazon.com.)

```python
# my_resource: apigateway.Resource


my_resource.add_cors_preflight(
    allow_origins=["https://amazon.com"],
    allow_methods=["GET", "PUT"]
)
```

See the
[`CorsOptions`](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-apigateway.CorsOptions.html)
API reference for a detailed list of supported configuration options.

You can specify defaults this at the resource level, in which case they will be applied to the entire resource sub-tree:

```python
# resource: apigateway.Resource


subtree = resource.add_resource("subtree",
    default_cors_preflight_options=apigateway.CorsOptions(
        allow_origins=["https://amazon.com"]
    )
)
```

This means that all resources under `subtree` (inclusive) will have a preflight
OPTIONS added to them.

See [#906](https://github.com/aws/aws-cdk/issues/906) for a list of CORS
features which are not yet supported.

## Endpoint Configuration

API gateway allows you to specify an
[API Endpoint Type](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-endpoint-types.html).
To define an endpoint type for the API gateway, use `endpointConfiguration` property:

```python
api = apigateway.RestApi(self, "api",
    endpoint_configuration=apigateway.EndpointConfiguration(
        types=[apigateway.EndpointType.EDGE]
    )
)
```

You can also create an association between your Rest API and a VPC endpoint. By doing so,
API Gateway will generate a new
Route53 Alias DNS record which you can use to invoke your private APIs. More info can be found
[here](https://docs.aws.amazon.com/apigateway/latest/developerguide/associate-private-api-with-vpc-endpoint.html).

Here is an example:

```python
# some_endpoint: ec2.IVpcEndpoint


api = apigateway.RestApi(self, "api",
    endpoint_configuration=apigateway.EndpointConfiguration(
        types=[apigateway.EndpointType.PRIVATE],
        vpc_endpoints=[some_endpoint]
    )
)
```

By performing this association, we can invoke the API gateway using the following format:

```plaintext
https://{rest-api-id}-{vpce-id}.execute-api.{region}.amazonaws.com/{stage}
```

To restrict access to the API Gateway to only the VPC endpoint, you can use the `grantInvokeFromVpcEndpointsOnly` method to [add resource policies](https://docs.aws.amazon.com/apigateway/latest/developerguide/private-api-tutorial.html#private-api-tutorial-attach-resource-policy) to the API Gateway:

```python
# api_gw_vpc_endpoint: ec2.IVpcEndpoint


api = apigateway.RestApi(self, "PrivateApi",
    endpoint_configuration=apigateway.EndpointConfiguration(
        types=[apigateway.EndpointType.PRIVATE],
        vpc_endpoints=[api_gw_vpc_endpoint]
    )
)
api.grant_invoke_from_vpc_endpoints_only([api_gw_vpc_endpoint])
```

## Private Integrations

A private integration makes it simple to expose HTTP/HTTPS resources behind an
Amazon VPC for access by clients outside of the VPC. The private integration uses
an API Gateway resource of `VpcLink` to encapsulate connections between API
Gateway and targeted VPC resources.
The `VpcLink` is then attached to the `Integration` of a specific API Gateway
Method. The following code sets up a private integration with a network load
balancer -

```python
import aws_cdk.aws_elasticloadbalancingv2 as elbv2


vpc = ec2.Vpc(self, "VPC")
nlb = elbv2.NetworkLoadBalancer(self, "NLB",
    vpc=vpc
)
link = apigateway.VpcLink(self, "link",
    targets=[nlb]
)

integration = apigateway.Integration(
    type=apigateway.IntegrationType.HTTP_PROXY,
    integration_http_method="ANY",
    options=apigateway.IntegrationOptions(
        connection_type=apigateway.ConnectionType.VPC_LINK,
        vpc_link=link
    )
)
```

The uri for the private integration, in the case of a VpcLink, will be set to the DNS name of
the VPC Link's NLB. If the VPC Link has multiple NLBs or the VPC Link is imported or the DNS
name cannot be determined for any other reason, the user is expected to specify the `uri`
property.

Any existing `VpcLink` resource can be imported into the CDK app via the `VpcLink.fromVpcLinkId()`.

```python
awesome_link = apigateway.VpcLink.from_vpc_link_id(self, "awesome-vpc-link", "us-east-1_oiuR12Abd")
```

## Gateway response

If the Rest API fails to process an incoming request, it returns to the client an error response without forwarding the
request to the integration backend. API Gateway has a set of standard response messages that are sent to the client for
each type of error. These error responses can be configured on the Rest API. The list of Gateway responses that can be
configured can be found [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html).
Learn more about [Gateway
Responses](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-gatewayResponse-definition.html).

The following code configures a Gateway Response when the response is 'access denied':

```python
api = apigateway.RestApi(self, "books-api")
api.add_gateway_response("test-response",
    type=apigateway.ResponseType.ACCESS_DENIED,
    status_code="500",
    response_headers={
        # Note that values must be enclosed within a pair of single quotes
        "Access-Control-Allow-Origin": "'test.com'",
        "test-key": "'test-value'"
    },
    templates={
        "application/json": "{ \"message\": $context.error.messageString, \"statusCode\": \"488\", \"type\": \"$context.error.responseType\" }"
    }
)
```

## OpenAPI Definition

CDK supports creating a REST API by importing an OpenAPI definition file. It currently supports OpenAPI v2.0 and OpenAPI
v3.0 definition files. Read more about [Configuring a REST API using
OpenAPI](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-import-api.html).

The following code creates a REST API using an external OpenAPI definition JSON file -

```python
# integration: apigateway.Integration


api = apigateway.SpecRestApi(self, "books-api",
    api_definition=apigateway.ApiDefinition.from_asset("path-to-file.json")
)

books_resource = api.root.add_resource("books")
books_resource.add_method("GET", integration)
```

It is possible to use the `addResource()` API to define additional API Gateway Resources.

**Note:** Deployment will fail if a Resource of the same name is already defined in the Open API specification.

**Note:** Any default properties configured, such as `defaultIntegration`, `defaultMethodOptions`, etc. will only be
applied to Resources and Methods defined in the CDK, and not the ones defined in the spec. Use the [API Gateway
extensions to OpenAPI](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions.html)
to configure these.

There are a number of limitations in using OpenAPI definitions in API Gateway. Read the [Amazon API Gateway important
notes for REST APIs](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-known-issues.html#api-gateway-known-issues-rest-apis)
for more details.

**Note:** When starting off with an OpenAPI definition using `SpecRestApi`, it is not possible to configure some
properties that can be configured directly in the OpenAPI specification file. This is to prevent people duplication
of these properties and potential confusion.

### Endpoint configuration

By default, `SpecRestApi` will create an edge optimized endpoint.

This can be modified as shown below:

```python
# api_definition: apigateway.ApiDefinition


api = apigateway.SpecRestApi(self, "ExampleRestApi",
    api_definition=api_definition,
    endpoint_types=[apigateway.EndpointType.PRIVATE]
)
```

**Note:** For private endpoints you will still need to provide the
[`x-amazon-apigateway-policy`](https://docs.aws.amazon.com/apigateway/latest/developerguide/openapi-extensions-policy.html) and
[`x-amazon-apigateway-endpoint-configuration`](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-endpoint-configuration.html)
in your openApi file.

## Metrics

The API Gateway service sends metrics around the performance of Rest APIs to Amazon CloudWatch.
These metrics can be referred to using the metric APIs available on the `RestApi`, `Stage` and `Method` constructs.
Note that detailed metrics must be enabled for a stage to use the `Method` metrics.
Read more about [API Gateway metrics](https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-metrics-and-dimensions.html), including enabling detailed metrics.
The APIs with the `metric` prefix can be used to get reference to specific metrics for this API. For example:

```python
api = apigateway.RestApi(self, "my-api")
stage = api.deployment_stage
method = api.root.add_method("GET")

client_error_api_metric = api.metric_client_error()
server_error_stage_metric = stage.metric_server_error()
latency_method_metric = method.metric_latency(stage)
```

## APIGateway v2

APIGateway v2 APIs are now moved to its own package named `aws-apigatewayv2`. Previously, these APIs were marked
deprecated but retained for backwards compatibility. The deprecated usage of APIGateway v2 APIs within this module
`aws-apigateway` has now been removed from the codebase.

The reason for the removal of these deprecated Constructs is that CloudFormation team is releasing AWS resources
like `AWS::APIGateway::DomainNameV2` and this would cause compatibility issue with the deprecated `CfnDomainNameV2`
resource defined in `apigatewayv2.ts` file during the L1 generation.

Move to using `aws-apigatewayv2` to get the latest APIs and updates.

---


This module is part of the [AWS Cloud Development Kit](https://github.com/aws/aws-cdk) project.
'''
from pkgutil import extend_path
__path__ = extend_path(__path__, __name__)

import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

import typeguard
from importlib.metadata import version as _metadata_package_version
TYPEGUARD_MAJOR_VERSION = int(_metadata_package_version('typeguard').split('.')[0])

def check_type(argname: str, value: object, expected_type: typing.Any) -> typing.Any:
    if TYPEGUARD_MAJOR_VERSION <= 2:
        return typeguard.check_type(argname=argname, value=value, expected_type=expected_type) # type:ignore
    else:
        if isinstance(value, jsii._reference_map.InterfaceDynamicProxy): # pyright: ignore [reportAttributeAccessIssue]
           pass
        else:
            if TYPEGUARD_MAJOR_VERSION == 3:
                typeguard.config.collection_check_strategy = typeguard.CollectionCheckStrategy.ALL_ITEMS # type:ignore
                typeguard.check_type(value=value, expected_type=expected_type) # type:ignore
            else:
                typeguard.check_type(value=value, expected_type=expected_type, collection_check_strategy=typeguard.CollectionCheckStrategy.ALL_ITEMS) # type:ignore

from .._jsii import *

import constructs as _constructs_77d1e7e8
from .. import (
    AssetHashType as _AssetHashType_05b67f2d,
    BundlingOptions as _BundlingOptions_588cc936,
    CfnResource as _CfnResource_9df397a6,
    CfnTag as _CfnTag_f6864754,
    Duration as _Duration_4839e8c3,
    IInspectable as _IInspectable_c2943556,
    IResolvable as _IResolvable_da3f097b,
    IResource as _IResource_c80c4260,
    ITaggable as _ITaggable_36806126,
    ITaggableV2 as _ITaggableV2_4e6798f8,
    IgnoreMode as _IgnoreMode_655a98e8,
    RemovalPolicy as _RemovalPolicy_9f93c814,
    Resource as _Resource_45bc6135,
    ResourceProps as _ResourceProps_15a65b4e,
    Size as _Size_7b441c34,
    SymlinkFollowMode as _SymlinkFollowMode_047ec1f6,
    TagManager as _TagManager_0a598cb3,
    TreeInspector as _TreeInspector_488e0dd5,
)
from ..aws_certificatemanager import ICertificate as _ICertificate_c194c70b
from ..aws_cloudwatch import (
    Metric as _Metric_e396a4dc,
    MetricOptions as _MetricOptions_1788b62f,
    Unit as _Unit_61bc6f70,
)
from ..aws_cognito import IUserPool as _IUserPool_1f1029e2
from ..aws_ec2 import IVpcEndpoint as _IVpcEndpoint_d8ea9bc3
from ..aws_elasticloadbalancingv2 import (
    INetworkLoadBalancer as _INetworkLoadBalancer_96e17101
)
from ..aws_iam import (
    AddToResourcePolicyResult as _AddToResourcePolicyResult_1d0a53ad,
    Grant as _Grant_a7ae64f8,
    IGrantable as _IGrantable_71c4f5de,
    IResourceWithPolicy as _IResourceWithPolicy_720d64fc,
    IRole as _IRole_235f5d8e,
    PolicyDocument as _PolicyDocument_3ac34393,
    PolicyStatement as _PolicyStatement_0fe33853,
)
from ..aws_kinesisfirehose import CfnDeliveryStream as _CfnDeliveryStream_8f3b1735
from ..aws_kms import IKey as _IKey_5f11635f
from ..aws_lambda import IFunction as _IFunction_6adb0ab8
from ..aws_logs import ILogGroup as _ILogGroup_3c4fa718
from ..aws_s3 import IBucket as _IBucket_42e086fd
from ..aws_s3_assets import AssetOptions as _AssetOptions_2aa69621
from ..aws_sagemaker import IEndpoint as _IEndpoint_58fe201a
from ..aws_stepfunctions import IStateMachine as _IStateMachine_73e8d2b0


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.AccessLogDestinationConfig",
    jsii_struct_bases=[],
    name_mapping={"destination_arn": "destinationArn"},
)
class AccessLogDestinationConfig:
    def __init__(self, *, destination_arn: builtins.str) -> None:
        '''Options when binding a log destination to a RestApi Stage.

        :param destination_arn: The Amazon Resource Name (ARN) of the destination resource.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            access_log_destination_config = apigateway.AccessLogDestinationConfig(
                destination_arn="destinationArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0d196c919178e027cbd446f3f274237c2b604071ed32de8cd155441134abe5c9)
            check_type(argname="argument destination_arn", value=destination_arn, expected_type=type_hints["destination_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "destination_arn": destination_arn,
        }

    @builtins.property
    def destination_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the destination resource.'''
        result = self._values.get("destination_arn")
        assert result is not None, "Required property 'destination_arn' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AccessLogDestinationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AccessLogField(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.AccessLogField",
):
    '''$context variables that can be used to customize access log pattern.

    :exampleMetadata: infused

    Example::

        apigateway.AccessLogFormat.custom(JSON.stringify({
            "request_id": apigateway.AccessLogField.context_request_id(),
            "source_ip": apigateway.AccessLogField.context_identity_source_ip(),
            "method": apigateway.AccessLogField.context_http_method(),
            "user_context": {
                "sub": apigateway.AccessLogField.context_authorizer_claims("sub"),
                "email": apigateway.AccessLogField.context_authorizer_claims("email")
            }
        }))
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="contextAccountId")
    @builtins.classmethod
    def context_account_id(cls) -> builtins.str:
        '''(deprecated) The API callers AWS account ID.

        :deprecated: Use ``contextCallerAccountId`` or ``contextOwnerAccountId`` instead

        :stability: deprecated
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAccountId", []))

    @jsii.member(jsii_name="contextApiId")
    @builtins.classmethod
    def context_api_id(cls) -> builtins.str:
        '''The identifier API Gateway assigns to your API.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextApiId", []))

    @jsii.member(jsii_name="contextAuthenticateError")
    @builtins.classmethod
    def context_authenticate_error(cls) -> builtins.str:
        '''The error message returned from an authentication attempt.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthenticateError", []))

    @jsii.member(jsii_name="contextAuthenticateLatency")
    @builtins.classmethod
    def context_authenticate_latency(cls) -> builtins.str:
        '''The authentication latency in ms.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthenticateLatency", []))

    @jsii.member(jsii_name="contextAuthenticateStatus")
    @builtins.classmethod
    def context_authenticate_status(cls) -> builtins.str:
        '''The status code returned from an authentication attempt.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthenticateStatus", []))

    @jsii.member(jsii_name="contextAuthorizeError")
    @builtins.classmethod
    def context_authorize_error(cls) -> builtins.str:
        '''The authorization error message.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizeError", []))

    @jsii.member(jsii_name="contextAuthorizeLatency")
    @builtins.classmethod
    def context_authorize_latency(cls) -> builtins.str:
        '''The authorization latency in ms.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizeLatency", []))

    @jsii.member(jsii_name="contextAuthorizer")
    @builtins.classmethod
    def context_authorizer(cls, property: builtins.str) -> builtins.str:
        '''The stringified value of the specified key-value pair of the ``context`` map returned from an API Gateway Lambda authorizer function.

        :param property: key of the context map.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c1e8763ae27b10d54a089726c9303794b0a5d506463cebf3f46db85f52647d47)
            check_type(argname="argument property", value=property, expected_type=type_hints["property"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizer", [property]))

    @jsii.member(jsii_name="contextAuthorizerClaims")
    @builtins.classmethod
    def context_authorizer_claims(cls, property: builtins.str) -> builtins.str:
        '''A property of the claims returned from the Amazon Cognito user pool after the method caller is successfully authenticated.

        :param property: A property key of the claims.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e347a5c48a1952ce93e3db2d199ae20e5581a8a60327d85281108d4a8195639b)
            check_type(argname="argument property", value=property, expected_type=type_hints["property"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizerClaims", [property]))

    @jsii.member(jsii_name="contextAuthorizerError")
    @builtins.classmethod
    def context_authorizer_error(cls) -> builtins.str:
        '''The error message returned from an authorizer.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizerError", []))

    @jsii.member(jsii_name="contextAuthorizerIntegrationLatency")
    @builtins.classmethod
    def context_authorizer_integration_latency(cls) -> builtins.str:
        '''The authorizer latency in ms.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizerIntegrationLatency", []))

    @jsii.member(jsii_name="contextAuthorizerIntegrationStatus")
    @builtins.classmethod
    def context_authorizer_integration_status(cls) -> builtins.str:
        '''The status code returned from a Lambda authorizer.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizerIntegrationStatus", []))

    @jsii.member(jsii_name="contextAuthorizerLatency")
    @builtins.classmethod
    def context_authorizer_latency(cls) -> builtins.str:
        '''The authorizer latency in ms.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizerLatency", []))

    @jsii.member(jsii_name="contextAuthorizerPrincipalId")
    @builtins.classmethod
    def context_authorizer_principal_id(cls) -> builtins.str:
        '''The principal user identification associated with the token sent by the client and returned from an API Gateway Lambda authorizer (formerly known as a custom authorizer).

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizerPrincipalId", []))

    @jsii.member(jsii_name="contextAuthorizerRequestId")
    @builtins.classmethod
    def context_authorizer_request_id(cls) -> builtins.str:
        '''The AWS endpoint's request ID.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizerRequestId", []))

    @jsii.member(jsii_name="contextAuthorizerStatus")
    @builtins.classmethod
    def context_authorizer_status(cls) -> builtins.str:
        '''The status code returned from an authorizer.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizerStatus", []))

    @jsii.member(jsii_name="contextAuthorizeStatus")
    @builtins.classmethod
    def context_authorize_status(cls) -> builtins.str:
        '''The status code returned from an authorization attempt.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAuthorizeStatus", []))

    @jsii.member(jsii_name="contextAwsEndpointRequestId")
    @builtins.classmethod
    def context_aws_endpoint_request_id(cls) -> builtins.str:
        '''The AWS endpoint's request ID.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextAwsEndpointRequestId", []))

    @jsii.member(jsii_name="contextCallerAccountId")
    @builtins.classmethod
    def context_caller_account_id(cls) -> builtins.str:
        '''The API callers AWS account ID.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextCallerAccountId", []))

    @jsii.member(jsii_name="contextCustomDomainBasePathMatched")
    @builtins.classmethod
    def context_custom_domain_base_path_matched(cls) -> builtins.str:
        '''The path for an API mapping that an incoming request matched.

        Applicable when a client uses a custom domain name to access an API. For example if a client sends a request to
        https://api.example.com/v1/orders/1234, and the request matches the API mapping with the path v1/orders, the value is v1/orders.

        :see: https://docs.aws.amazon.com/en_jp/apigateway/latest/developerguide/rest-api-mappings.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextCustomDomainBasePathMatched", []))

    @jsii.member(jsii_name="contextDomainName")
    @builtins.classmethod
    def context_domain_name(cls) -> builtins.str:
        '''The full domain name used to invoke the API.

        This should be the same as the incoming ``Host`` header.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextDomainName", []))

    @jsii.member(jsii_name="contextDomainPrefix")
    @builtins.classmethod
    def context_domain_prefix(cls) -> builtins.str:
        '''The first label of the ``$context.domainName``. This is often used as a caller/customer identifier.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextDomainPrefix", []))

    @jsii.member(jsii_name="contextErrorMessage")
    @builtins.classmethod
    def context_error_message(cls) -> builtins.str:
        '''A string containing an API Gateway error message.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextErrorMessage", []))

    @jsii.member(jsii_name="contextErrorMessageString")
    @builtins.classmethod
    def context_error_message_string(cls) -> builtins.str:
        '''The quoted value of $context.error.message, namely "$context.error.message".'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextErrorMessageString", []))

    @jsii.member(jsii_name="contextErrorResponseType")
    @builtins.classmethod
    def context_error_response_type(cls) -> builtins.str:
        '''A type of GatewayResponse.

        This variable can only be used for simple variable substitution in a GatewayResponse body-mapping template,
        which is not processed by the Velocity Template Language engine, and in access logging.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextErrorResponseType", []))

    @jsii.member(jsii_name="contextErrorValidationErrorString")
    @builtins.classmethod
    def context_error_validation_error_string(cls) -> builtins.str:
        '''A string containing a detailed validation error message.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextErrorValidationErrorString", []))

    @jsii.member(jsii_name="contextExtendedRequestId")
    @builtins.classmethod
    def context_extended_request_id(cls) -> builtins.str:
        '''The extended ID that API Gateway assigns to the API request, which contains more useful information for debugging/troubleshooting.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextExtendedRequestId", []))

    @jsii.member(jsii_name="contextHttpMethod")
    @builtins.classmethod
    def context_http_method(cls) -> builtins.str:
        '''The HTTP method used.

        Valid values include: ``DELETE``, ``GET``, ``HEAD``, ``OPTIONS``, ``PATCH``, ``POST``, and ``PUT``.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextHttpMethod", []))

    @jsii.member(jsii_name="contextIdentityAccountId")
    @builtins.classmethod
    def context_identity_account_id(cls) -> builtins.str:
        '''The AWS account ID associated with the request.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityAccountId", []))

    @jsii.member(jsii_name="contextIdentityApiKey")
    @builtins.classmethod
    def context_identity_api_key(cls) -> builtins.str:
        '''For API methods that require an API key, this variable is the API key associated with the method request.

        For methods that don't require an API key, this variable is

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityApiKey", []))

    @jsii.member(jsii_name="contextIdentityApiKeyId")
    @builtins.classmethod
    def context_identity_api_key_id(cls) -> builtins.str:
        '''The API key ID associated with an API request that requires an API key.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityApiKeyId", []))

    @jsii.member(jsii_name="contextIdentityCaller")
    @builtins.classmethod
    def context_identity_caller(cls) -> builtins.str:
        '''The principal identifier of the caller making the request.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityCaller", []))

    @jsii.member(jsii_name="contextIdentityClientCertIssunerDN")
    @builtins.classmethod
    def context_identity_client_cert_issuner_dn(cls) -> builtins.str:
        '''The distinguished name of the issuer of the certificate that a client presents.

        Present when a client accesses an API by using a custom domain name that has mutual TLS enabled.
        Present only in access logs if mutual TLS authentication fails.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityClientCertIssunerDN", []))

    @jsii.member(jsii_name="contextIdentityClientCertPem")
    @builtins.classmethod
    def context_identity_client_cert_pem(cls) -> builtins.str:
        '''The PEM-encoded client certificate that the client presented during mutual TLS authentication.

        Present when a client accesses an API by using a custom domain name that has mutual TLS enabled.
        Present only in access logs if mutual TLS authentication fails.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityClientCertPem", []))

    @jsii.member(jsii_name="contextIdentityClientCertSerialNumber")
    @builtins.classmethod
    def context_identity_client_cert_serial_number(cls) -> builtins.str:
        '''The serial number of the certificate.

        Present when a client accesses an API by using a custom domain name that has mutual TLS enabled.
        Present only in access logs if mutual TLS authentication fails.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityClientCertSerialNumber", []))

    @jsii.member(jsii_name="contextIdentityClientCertSubjectDN")
    @builtins.classmethod
    def context_identity_client_cert_subject_dn(cls) -> builtins.str:
        '''The distinguished name of the subject of the certificate that a client presents.

        Present when a client accesses an API by using a custom domain name that has mutual TLS enabled.
        Present only in access logs if mutual TLS authentication fails.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityClientCertSubjectDN", []))

    @jsii.member(jsii_name="contextIdentityClientCertValidityNotAfter")
    @builtins.classmethod
    def context_identity_client_cert_validity_not_after(cls) -> builtins.str:
        '''The date after which the certificate is invalid.

        Present when a client accesses an API by using a custom domain name that has mutual TLS enabled.
        Present only in access logs if mutual TLS authentication fails.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityClientCertValidityNotAfter", []))

    @jsii.member(jsii_name="contextIdentityClientCertValidityNotBefore")
    @builtins.classmethod
    def context_identity_client_cert_validity_not_before(cls) -> builtins.str:
        '''The date before which the certificate is invalid.

        Present when a client accesses an API by using a custom domain name that has mutual TLS enabled.
        Present only in access logs if mutual TLS authentication fails.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityClientCertValidityNotBefore", []))

    @jsii.member(jsii_name="contextIdentityCognitoAuthenticationProvider")
    @builtins.classmethod
    def context_identity_cognito_authentication_provider(cls) -> builtins.str:
        '''The Amazon Cognito authentication provider used by the caller making the request.

        Available only if the request was signed with Amazon Cognito credentials.

        :see: https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityCognitoAuthenticationProvider", []))

    @jsii.member(jsii_name="contextIdentityCognitoAuthenticationType")
    @builtins.classmethod
    def context_identity_cognito_authentication_type(cls) -> builtins.str:
        '''The Amazon Cognito authentication type of the caller making the request.

        Available only if the request was signed with Amazon Cognito credentials.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityCognitoAuthenticationType", []))

    @jsii.member(jsii_name="contextIdentityCognitoIdentityId")
    @builtins.classmethod
    def context_identity_cognito_identity_id(cls) -> builtins.str:
        '''The Amazon Cognito identity ID of the caller making the request.

        Available only if the request was signed with Amazon Cognito credentials.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityCognitoIdentityId", []))

    @jsii.member(jsii_name="contextIdentityCognitoIdentityPoolId")
    @builtins.classmethod
    def context_identity_cognito_identity_pool_id(cls) -> builtins.str:
        '''The Amazon Cognito identity pool ID of the caller making the request.

        Available only if the request was signed with Amazon Cognito credentials.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityCognitoIdentityPoolId", []))

    @jsii.member(jsii_name="contextIdentityPrincipalOrgId")
    @builtins.classmethod
    def context_identity_principal_org_id(cls) -> builtins.str:
        '''The AWS organization ID.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityPrincipalOrgId", []))

    @jsii.member(jsii_name="contextIdentitySourceIp")
    @builtins.classmethod
    def context_identity_source_ip(cls) -> builtins.str:
        '''The source IP address of the TCP connection making the request to API Gateway.

        Warning: You should not trust this value if there is any chance that the ``X-Forwarded-For`` header could be forged.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentitySourceIp", []))

    @jsii.member(jsii_name="contextIdentityUser")
    @builtins.classmethod
    def context_identity_user(cls) -> builtins.str:
        '''The principal identifier of the user making the request.

        Used in Lambda authorizers.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-output.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityUser", []))

    @jsii.member(jsii_name="contextIdentityUserAgent")
    @builtins.classmethod
    def context_identity_user_agent(cls) -> builtins.str:
        '''The User-Agent header of the API caller.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityUserAgent", []))

    @jsii.member(jsii_name="contextIdentityUserArn")
    @builtins.classmethod
    def context_identity_user_arn(cls) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the effective user identified after authentication.

        :see: https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIdentityUserArn", []))

    @jsii.member(jsii_name="contextIntegrationErrorMessage")
    @builtins.classmethod
    def context_integration_error_message(cls) -> builtins.str:
        '''A string that contains an integration error message.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIntegrationErrorMessage", []))

    @jsii.member(jsii_name="contextIntegrationLatency")
    @builtins.classmethod
    def context_integration_latency(cls) -> builtins.str:
        '''The integration latency in ms.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIntegrationLatency", []))

    @jsii.member(jsii_name="contextIntegrationStatus")
    @builtins.classmethod
    def context_integration_status(cls) -> builtins.str:
        '''For Lambda proxy integration, this parameter represents the status code returned from AWS Lambda, not from the backend Lambda function.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextIntegrationStatus", []))

    @jsii.member(jsii_name="contextOwnerAccountId")
    @builtins.classmethod
    def context_owner_account_id(cls) -> builtins.str:
        '''The API owner's AWS account ID.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextOwnerAccountId", []))

    @jsii.member(jsii_name="contextPath")
    @builtins.classmethod
    def context_path(cls) -> builtins.str:
        '''The request path.

        For example, for a non-proxy request URL of https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child,
        this value is /{stage}/root/child.
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextPath", []))

    @jsii.member(jsii_name="contextProtocol")
    @builtins.classmethod
    def context_protocol(cls) -> builtins.str:
        '''The request protocol, for example, HTTP/1.1.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextProtocol", []))

    @jsii.member(jsii_name="contextRequestId")
    @builtins.classmethod
    def context_request_id(cls) -> builtins.str:
        '''The ID that API Gateway assigns to the API request.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestId", []))

    @jsii.member(jsii_name="contextRequestOverrideHeader")
    @builtins.classmethod
    def context_request_override_header(cls, header_name: builtins.str) -> builtins.str:
        '''The request header override.

        If this parameter is defined, it contains the headers to be used instead of the HTTP Headers that are defined in the Integration Request pane.

        :param header_name: -

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8cdb90316374681bdb16f69b2f5eb811508c0c986715bf50c52b0476887a9e06)
            check_type(argname="argument header_name", value=header_name, expected_type=type_hints["header_name"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestOverrideHeader", [header_name]))

    @jsii.member(jsii_name="contextRequestOverridePath")
    @builtins.classmethod
    def context_request_override_path(cls, path_name: builtins.str) -> builtins.str:
        '''The request path override.

        If this parameter is defined,
        it contains the request path to be used instead of the URL Path Parameters that are defined in the Integration Request pane.

        :param path_name: -

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9f7bb822ebe916c42946057ab4781e0e967114267befceaddf4ca829b4dbef47)
            check_type(argname="argument path_name", value=path_name, expected_type=type_hints["path_name"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestOverridePath", [path_name]))

    @jsii.member(jsii_name="contextRequestOverrideQuerystring")
    @builtins.classmethod
    def context_request_override_querystring(
        cls,
        querystring_name: builtins.str,
    ) -> builtins.str:
        '''The request query string override.

        If this parameter is defined, it contains the request query strings to be used instead
        of the URL Query String Parameters that are defined in the Integration Request pane.

        :param querystring_name: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3427a275436427788320487d6cc2c3c016fcc887365ec57fa6d67bc3e47ac12a)
            check_type(argname="argument querystring_name", value=querystring_name, expected_type=type_hints["querystring_name"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestOverrideQuerystring", [querystring_name]))

    @jsii.member(jsii_name="contextRequestTime")
    @builtins.classmethod
    def context_request_time(cls) -> builtins.str:
        '''The CLF-formatted request time (dd/MMM/yyyy:HH:mm:ss +-hhmm).'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestTime", []))

    @jsii.member(jsii_name="contextRequestTimeEpoch")
    @builtins.classmethod
    def context_request_time_epoch(cls) -> builtins.str:
        '''The Epoch-formatted request time.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextRequestTimeEpoch", []))

    @jsii.member(jsii_name="contextResourceId")
    @builtins.classmethod
    def context_resource_id(cls) -> builtins.str:
        '''The identifier that API Gateway assigns to your resource.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResourceId", []))

    @jsii.member(jsii_name="contextResourcePath")
    @builtins.classmethod
    def context_resource_path(cls) -> builtins.str:
        '''The path to your resource.

        For example, for the non-proxy request URI of ``https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child``,
        The $context.resourcePath value is ``/root/child``.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-create-api-step-by-step.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResourcePath", []))

    @jsii.member(jsii_name="contextResponseLatency")
    @builtins.classmethod
    def context_response_latency(cls) -> builtins.str:
        '''The response latency in ms.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResponseLatency", []))

    @jsii.member(jsii_name="contextResponseLength")
    @builtins.classmethod
    def context_response_length(cls) -> builtins.str:
        '''The response payload length.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResponseLength", []))

    @jsii.member(jsii_name="contextResponseOverrideHeader")
    @builtins.classmethod
    def context_response_override_header(
        cls,
        header_name: builtins.str,
    ) -> builtins.str:
        '''The response header override.

        If this parameter is defined, it contains the header to be returned instead of the Response header
        that is defined as the Default mapping in the Integration Response pane.

        :param header_name: -

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d532554505e61f81e1b9ec850cbf52d3ee1550f112073eae6daecd88369e97de)
            check_type(argname="argument header_name", value=header_name, expected_type=type_hints["header_name"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResponseOverrideHeader", [header_name]))

    @jsii.member(jsii_name="contextResponseOverrideStatus")
    @builtins.classmethod
    def context_response_override_status(cls) -> builtins.str:
        '''The response status code override.

        If this parameter is defined, it contains the status code to be returned instead of the Method response status
        that is defined as the Default mapping in the Integration Response pane.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextResponseOverrideStatus", []))

    @jsii.member(jsii_name="contextStage")
    @builtins.classmethod
    def context_stage(cls) -> builtins.str:
        '''The deployment stage of the API request (for example, ``Beta`` or ``Prod``).'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextStage", []))

    @jsii.member(jsii_name="contextStatus")
    @builtins.classmethod
    def context_status(cls) -> builtins.str:
        '''The method response status.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextStatus", []))

    @jsii.member(jsii_name="contextWafError")
    @builtins.classmethod
    def context_waf_error(cls) -> builtins.str:
        '''The error message returned from AWS WAF.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextWafError", []))

    @jsii.member(jsii_name="contextWafLatency")
    @builtins.classmethod
    def context_waf_latency(cls) -> builtins.str:
        '''The AWS WAF latency in ms.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextWafLatency", []))

    @jsii.member(jsii_name="contextWafResponseCode")
    @builtins.classmethod
    def context_waf_response_code(cls) -> builtins.str:
        '''The response received from AWS WAF: ``WAF_ALLOW`` or ``WAF_BLOCK``.

        Will not be set if the stage is not associated with a web ACL.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextWafResponseCode", []))

    @jsii.member(jsii_name="contextWafStatus")
    @builtins.classmethod
    def context_waf_status(cls) -> builtins.str:
        '''The status code returned from AWS WAF.'''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextWafStatus", []))

    @jsii.member(jsii_name="contextWebaclArn")
    @builtins.classmethod
    def context_webacl_arn(cls) -> builtins.str:
        '''The complete ARN of the web ACL that is used to decide whether to allow or block the request.

        Will not be set if the stage is not associated with a web ACL.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextWebaclArn", []))

    @jsii.member(jsii_name="contextXrayTraceId")
    @builtins.classmethod
    def context_xray_trace_id(cls) -> builtins.str:
        '''The trace ID for the X-Ray trace.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-enabling-xray.html
        '''
        return typing.cast(builtins.str, jsii.sinvoke(cls, "contextXrayTraceId", []))


class AccessLogFormat(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.AccessLogFormat",
):
    '''factory methods for access log format.

    :exampleMetadata: infused

    Example::

        destination_bucket = s3.Bucket(self, "Bucket")
        delivery_stream_role = iam.Role(self, "Role",
            assumed_by=iam.ServicePrincipal("firehose.amazonaws.com")
        )
        
        stream = firehose.CfnDeliveryStream(self, "MyStream",
            delivery_stream_name="amazon-apigateway-delivery-stream",
            s3_destination_configuration=firehose.CfnDeliveryStream.S3DestinationConfigurationProperty(
                bucket_arn=destination_bucket.bucket_arn,
                role_arn=delivery_stream_role.role_arn
            )
        )
        
        api = apigateway.RestApi(self, "books",
            deploy_options=apigateway.StageOptions(
                access_log_destination=apigateway.FirehoseLogDestination(stream),
                access_log_format=apigateway.AccessLogFormat.json_with_standard_fields()
            )
        )
    '''

    @jsii.member(jsii_name="clf")
    @builtins.classmethod
    def clf(cls) -> "AccessLogFormat":
        '''Generate Common Log Format.'''
        return typing.cast("AccessLogFormat", jsii.sinvoke(cls, "clf", []))

    @jsii.member(jsii_name="custom")
    @builtins.classmethod
    def custom(cls, format: builtins.str) -> "AccessLogFormat":
        '''Custom log format.

        You can create any log format string. You can easily get the $ context variable by using the methods of AccessLogField.

        :param format: -

        Example::

            apigateway.AccessLogFormat.custom(JSON.stringify({
                "request_id": apigateway.AccessLogField.context_request_id(),
                "source_ip": apigateway.AccessLogField.context_identity_source_ip(),
                "method": apigateway.AccessLogField.context_http_method(),
                "user_context": {
                    "sub": apigateway.AccessLogField.context_authorizer_claims("sub"),
                    "email": apigateway.AccessLogField.context_authorizer_claims("email")
                }
            }))
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4692c15f70ce18849ab172ed2332cc4added75850adcf2e0ed8a0ebce52c27db)
            check_type(argname="argument format", value=format, expected_type=type_hints["format"])
        return typing.cast("AccessLogFormat", jsii.sinvoke(cls, "custom", [format]))

    @jsii.member(jsii_name="jsonWithStandardFields")
    @builtins.classmethod
    def json_with_standard_fields(
        cls,
        *,
        caller: builtins.bool,
        http_method: builtins.bool,
        ip: builtins.bool,
        protocol: builtins.bool,
        request_time: builtins.bool,
        resource_path: builtins.bool,
        response_length: builtins.bool,
        status: builtins.bool,
        user: builtins.bool,
    ) -> "AccessLogFormat":
        '''Access log will be produced in the JSON format with a set of fields most useful in the access log.

        All fields are turned on by default with the
        option to turn off specific fields.

        :param caller: If this flag is enabled, the principal identifier of the caller will be output to the log.
        :param http_method: If this flag is enabled, the http method will be output to the log.
        :param ip: If this flag is enabled, the source IP of request will be output to the log.
        :param protocol: If this flag is enabled, the request protocol will be output to the log.
        :param request_time: If this flag is enabled, the CLF-formatted request time((dd/MMM/yyyy:HH:mm:ss +-hhmm) will be output to the log.
        :param resource_path: If this flag is enabled, the path to your resource will be output to the log.
        :param response_length: If this flag is enabled, the response payload length will be output to the log.
        :param status: If this flag is enabled, the method response status will be output to the log.
        :param user: If this flag is enabled, the principal identifier of the user will be output to the log.
        '''
        fields = JsonWithStandardFieldProps(
            caller=caller,
            http_method=http_method,
            ip=ip,
            protocol=protocol,
            request_time=request_time,
            resource_path=resource_path,
            response_length=response_length,
            status=status,
            user=user,
        )

        return typing.cast("AccessLogFormat", jsii.sinvoke(cls, "jsonWithStandardFields", [fields]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Output a format string to be used with CloudFormation.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.AddApiKeyOptions",
    jsii_struct_bases=[],
    name_mapping={"override_logical_id": "overrideLogicalId"},
)
class AddApiKeyOptions:
    def __init__(
        self,
        *,
        override_logical_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options to the UsagePlan.addApiKey() method.

        :param override_logical_id: Override the CloudFormation logical id of the AWS::ApiGateway::UsagePlanKey resource. Default: - autogenerated by the CDK

        :exampleMetadata: infused

        Example::

            # usageplan: apigateway.UsagePlan
            # api_key: apigateway.ApiKey
            
            
            usageplan.add_api_key(api_key,
                override_logical_id="..."
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b1ed618edf117f4709e95c8b83683418e202f9dc66763b23811591d58c9eba46)
            check_type(argname="argument override_logical_id", value=override_logical_id, expected_type=type_hints["override_logical_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if override_logical_id is not None:
            self._values["override_logical_id"] = override_logical_id

    @builtins.property
    def override_logical_id(self) -> typing.Optional[builtins.str]:
        '''Override the CloudFormation logical id of the AWS::ApiGateway::UsagePlanKey resource.

        :default: - autogenerated by the CDK
        '''
        result = self._values.get("override_logical_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddApiKeyOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ApiDefinition(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="aws-cdk-lib.aws_apigateway.ApiDefinition",
):
    '''Represents an OpenAPI definition asset.

    :exampleMetadata: infused

    Example::

        my_api_definition = apigateway.ApiDefinition.from_asset("path-to-file.json")
        spec_rest_api = apigateway.SpecRestApi(self, "my-specrest-api",
            deploy=False,
            api_definition=my_api_definition
        )
        
        # Use `stageName` to deploy to an existing stage
        deployment = apigateway.Deployment(self, "my-deployment",
            api=spec_rest_api,
            stage_name="dev",
            retain_deployments=True
        )
        
        # Trigger a new deployment on OpenAPI definition updates
        deployment.add_to_logical_id(my_api_definition)
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromAsset")
    @builtins.classmethod
    def from_asset(
        cls,
        file: builtins.str,
        *,
        deploy_time: typing.Optional[builtins.bool] = None,
        display_name: typing.Optional[builtins.str] = None,
        readers: typing.Optional[typing.Sequence[_IGrantable_71c4f5de]] = None,
        source_kms_key: typing.Optional[_IKey_5f11635f] = None,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[_AssetHashType_05b67f2d] = None,
        bundling: typing.Optional[typing.Union[_BundlingOptions_588cc936, typing.Dict[builtins.str, typing.Any]]] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_047ec1f6] = None,
        ignore_mode: typing.Optional[_IgnoreMode_655a98e8] = None,
    ) -> "AssetApiDefinition":
        '''Loads the API specification from a local disk asset.

        :param file: -
        :param deploy_time: Whether or not the asset needs to exist beyond deployment time; i.e. are copied over to a different location and not needed afterwards. Setting this property to true has an impact on the lifecycle of the asset, because we will assume that it is safe to delete after the CloudFormation deployment succeeds. For example, Lambda Function assets are copied over to Lambda during deployment. Therefore, it is not necessary to store the asset in S3, so we consider those deployTime assets. Default: false
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. If the same asset is added multiple times, the display name of the first occurrence is used. The default is the construct path of the Asset construct, with respect to the enclosing stack. If the asset is produced by a construct helper function (such as ``lambda.Code.fromAsset()``), this will look like ``MyFunction/Code``. We use the stack-relative construct path so that in the common case where you have multiple stacks with the same asset, we won't show something like ``/MyBetaStack/MyFunction/Code`` when you are actually deploying to production. Default: - Stack-relative construct path
        :param readers: A list of principals that should be able to read this asset from S3. You can use ``asset.grantRead(principal)`` to grant read permissions later. Default: - No principals that can read file asset.
        :param source_kms_key: The ARN of the KMS key used to encrypt the handler code. Default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.
        :param asset_hash: Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: Bundle the asset by executing a command in a Docker container or a custom bundling provider. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__931fac6af706dd0dae384cc682b2ecb066666ff6165060417e461a3920d8895e)
            check_type(argname="argument file", value=file, expected_type=type_hints["file"])
        options = _AssetOptions_2aa69621(
            deploy_time=deploy_time,
            display_name=display_name,
            readers=readers,
            source_kms_key=source_kms_key,
            asset_hash=asset_hash,
            asset_hash_type=asset_hash_type,
            bundling=bundling,
            exclude=exclude,
            follow_symlinks=follow_symlinks,
            ignore_mode=ignore_mode,
        )

        return typing.cast("AssetApiDefinition", jsii.sinvoke(cls, "fromAsset", [file, options]))

    @jsii.member(jsii_name="fromBucket")
    @builtins.classmethod
    def from_bucket(
        cls,
        bucket: _IBucket_42e086fd,
        key: builtins.str,
        object_version: typing.Optional[builtins.str] = None,
    ) -> "S3ApiDefinition":
        '''Creates an API definition from a specification file in an S3 bucket.

        :param bucket: -
        :param key: -
        :param object_version: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__826b23f1c71b762530c9e7ee78bbb37951d8682e9643f508cdc68c374225ab85)
            check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument object_version", value=object_version, expected_type=type_hints["object_version"])
        return typing.cast("S3ApiDefinition", jsii.sinvoke(cls, "fromBucket", [bucket, key, object_version]))

    @jsii.member(jsii_name="fromInline")
    @builtins.classmethod
    def from_inline(cls, definition: typing.Any) -> "InlineApiDefinition":
        '''Create an API definition from an inline object.

        The inline object must follow the
        schema of OpenAPI 2.0 or OpenAPI 3.0

        :param definition: -

        Example::

            apigateway.ApiDefinition.from_inline({
                "openapi": "3.0.2",
                "paths": {
                    "/pets": {
                        "get": {
                            "responses": {
                                "200": {
                                    "content": {
                                        "application/json": {
                                            "schema": {
                                                "$ref": "#/components/schemas/Empty"
                                            }
                                        }
                                    }
                                }
                            },
                            "x-amazon-apigateway-integration": {
                                "responses": {
                                    "default": {
                                        "status_code": "200"
                                    }
                                },
                                "request_templates": {
                                    "application/json": "{\"statusCode\": 200}"
                                },
                                "passthrough_behavior": "when_no_match",
                                "type": "mock"
                            }
                        }
                    }
                },
                "components": {
                    "schemas": {
                        "Empty": {
                            "title": "Empty Schema",
                            "type": "object"
                        }
                    }
                }
            })
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3fdc39319ab31dc01370dce3eebe8d625768dc937b6e3ca507db636cd42bbee7)
            check_type(argname="argument definition", value=definition, expected_type=type_hints["definition"])
        return typing.cast("InlineApiDefinition", jsii.sinvoke(cls, "fromInline", [definition]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(self, scope: _constructs_77d1e7e8.Construct) -> "ApiDefinitionConfig":
        '''Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.

        :param scope: The binding scope. Don't be smart about trying to down-cast or assume it's initialized. You may just use it as a construct scope.
        '''
        ...

    @jsii.member(jsii_name="bindAfterCreate")
    def bind_after_create(
        self,
        _scope: _constructs_77d1e7e8.Construct,
        _rest_api: "IRestApi",
    ) -> None:
        '''Called after the CFN RestApi resource has been created to allow the Api Definition to bind to it.

        Specifically it's required to allow assets to add
        metadata for tooling like SAM CLI to be able to find their origins.

        :param _scope: -
        :param _rest_api: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3ee382fc2c94d4bece70868d74861f5ede3af81a49e0b03d1b1d7f7808f09fd4)
            check_type(argname="argument _scope", value=_scope, expected_type=type_hints["_scope"])
            check_type(argname="argument _rest_api", value=_rest_api, expected_type=type_hints["_rest_api"])
        return typing.cast(None, jsii.invoke(self, "bindAfterCreate", [_scope, _rest_api]))


class _ApiDefinitionProxy(ApiDefinition):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _constructs_77d1e7e8.Construct) -> "ApiDefinitionConfig":
        '''Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.

        :param scope: The binding scope. Don't be smart about trying to down-cast or assume it's initialized. You may just use it as a construct scope.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bd058ce3ea8954d549b4bc1dbf5a375c26257d3f2ac2da56517c516b14f827c8)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast("ApiDefinitionConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, ApiDefinition).__jsii_proxy_class__ = lambda : _ApiDefinitionProxy


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.ApiDefinitionConfig",
    jsii_struct_bases=[],
    name_mapping={
        "inline_definition": "inlineDefinition",
        "s3_location": "s3Location",
    },
)
class ApiDefinitionConfig:
    def __init__(
        self,
        *,
        inline_definition: typing.Any = None,
        s3_location: typing.Optional[typing.Union["ApiDefinitionS3Location", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Post-Binding Configuration for a CDK construct.

        :param inline_definition: Inline specification (mutually exclusive with ``s3Location``). Default: - API definition is not defined inline
        :param s3_location: The location of the specification in S3 (mutually exclusive with ``inlineDefinition``). Default: - API definition is not an S3 location

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            # inline_definition: Any
            
            api_definition_config = apigateway.ApiDefinitionConfig(
                inline_definition=inline_definition,
                s3_location=apigateway.ApiDefinitionS3Location(
                    bucket="bucket",
                    key="key",
            
                    # the properties below are optional
                    version="version"
                )
            )
        '''
        if isinstance(s3_location, dict):
            s3_location = ApiDefinitionS3Location(**s3_location)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__17b953e45039464bf52cb365fc968dd3aaf53d86a56beecdd9af0a4a44284bf8)
            check_type(argname="argument inline_definition", value=inline_definition, expected_type=type_hints["inline_definition"])
            check_type(argname="argument s3_location", value=s3_location, expected_type=type_hints["s3_location"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if inline_definition is not None:
            self._values["inline_definition"] = inline_definition
        if s3_location is not None:
            self._values["s3_location"] = s3_location

    @builtins.property
    def inline_definition(self) -> typing.Any:
        '''Inline specification (mutually exclusive with ``s3Location``).

        :default: - API definition is not defined inline
        '''
        result = self._values.get("inline_definition")
        return typing.cast(typing.Any, result)

    @builtins.property
    def s3_location(self) -> typing.Optional["ApiDefinitionS3Location"]:
        '''The location of the specification in S3 (mutually exclusive with ``inlineDefinition``).

        :default: - API definition is not an S3 location
        '''
        result = self._values.get("s3_location")
        return typing.cast(typing.Optional["ApiDefinitionS3Location"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApiDefinitionConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.ApiDefinitionS3Location",
    jsii_struct_bases=[],
    name_mapping={"bucket": "bucket", "key": "key", "version": "version"},
)
class ApiDefinitionS3Location:
    def __init__(
        self,
        *,
        bucket: builtins.str,
        key: builtins.str,
        version: typing.Optional[builtins.str] = None,
    ) -> None:
        '''S3 location of the API definition file.

        :param bucket: The S3 bucket.
        :param key: The S3 key.
        :param version: An optional version. Default: - latest version

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            api_definition_s3_location = apigateway.ApiDefinitionS3Location(
                bucket="bucket",
                key="key",
            
                # the properties below are optional
                version="version"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7192edd2401ec44496a976d34ff2eed58269d2776ace8b7d7116d0382f81d0a0)
            check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
            check_type(argname="argument key", value=key, expected_type=type_hints["key"])
            check_type(argname="argument version", value=version, expected_type=type_hints["version"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "bucket": bucket,
            "key": key,
        }
        if version is not None:
            self._values["version"] = version

    @builtins.property
    def bucket(self) -> builtins.str:
        '''The S3 bucket.'''
        result = self._values.get("bucket")
        assert result is not None, "Required property 'bucket' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def key(self) -> builtins.str:
        '''The S3 key.'''
        result = self._values.get("key")
        assert result is not None, "Required property 'key' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def version(self) -> typing.Optional[builtins.str]:
        '''An optional version.

        :default: - latest version
        '''
        result = self._values.get("version")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApiDefinitionS3Location(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_apigateway.ApiKeySourceType")
class ApiKeySourceType(enum.Enum):
    HEADER = "HEADER"
    '''To read the API key from the ``X-API-Key`` header of a request.'''
    AUTHORIZER = "AUTHORIZER"
    '''To read the API key from the ``UsageIdentifierKey`` from a custom authorizer.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.ApiMappingOptions",
    jsii_struct_bases=[],
    name_mapping={"base_path": "basePath"},
)
class ApiMappingOptions:
    def __init__(self, *, base_path: typing.Optional[builtins.str] = None) -> None:
        '''Options for creating an api mapping.

        :param base_path: The api path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string. If this is undefined, a mapping will be added for the empty path. Any request that does not match a mapping will get sent to the API that has been mapped to the empty path. Default: - map requests from the domain root (e.g. ``example.com``).

        :exampleMetadata: infused

        Example::

            # acm_certificate_for_example_com: Any
            # rest_api: apigateway.RestApi
            # second_rest_api: apigateway.RestApi
            
            
            domain = apigateway.DomainName(self, "custom-domain",
                domain_name="example.com",
                certificate=acm_certificate_for_example_com,
                mapping=rest_api
            )
            
            domain.add_api_mapping(second_rest_api.deployment_stage,
                base_path="orders/v2/api"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__76bbf955e57234cf30598a48f4465eed2280ab222c776b1e6404dac905f82764)
            check_type(argname="argument base_path", value=base_path, expected_type=type_hints["base_path"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if base_path is not None:
            self._values["base_path"] = base_path

    @builtins.property
    def base_path(self) -> typing.Optional[builtins.str]:
        '''The api path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string.

        If this is undefined, a mapping will be added for the empty path. Any request
        that does not match a mapping will get sent to the API that has been mapped
        to the empty path.

        :default: - map requests from the domain root (e.g. ``example.com``).
        '''
        result = self._values.get("base_path")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApiMappingOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AssetApiDefinition(
    ApiDefinition,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.AssetApiDefinition",
):
    '''OpenAPI specification from a local file.

    :exampleMetadata: infused

    Example::

        my_api_definition = apigateway.ApiDefinition.from_asset("path-to-file.json")
        spec_rest_api = apigateway.SpecRestApi(self, "my-specrest-api",
            deploy=False,
            api_definition=my_api_definition
        )
        
        # Use `stageName` to deploy to an existing stage
        deployment = apigateway.Deployment(self, "my-deployment",
            api=spec_rest_api,
            stage_name="dev",
            retain_deployments=True
        )
        
        # Trigger a new deployment on OpenAPI definition updates
        deployment.add_to_logical_id(my_api_definition)
    '''

    def __init__(
        self,
        path: builtins.str,
        *,
        deploy_time: typing.Optional[builtins.bool] = None,
        display_name: typing.Optional[builtins.str] = None,
        readers: typing.Optional[typing.Sequence[_IGrantable_71c4f5de]] = None,
        source_kms_key: typing.Optional[_IKey_5f11635f] = None,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[_AssetHashType_05b67f2d] = None,
        bundling: typing.Optional[typing.Union[_BundlingOptions_588cc936, typing.Dict[builtins.str, typing.Any]]] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_047ec1f6] = None,
        ignore_mode: typing.Optional[_IgnoreMode_655a98e8] = None,
    ) -> None:
        '''
        :param path: -
        :param deploy_time: Whether or not the asset needs to exist beyond deployment time; i.e. are copied over to a different location and not needed afterwards. Setting this property to true has an impact on the lifecycle of the asset, because we will assume that it is safe to delete after the CloudFormation deployment succeeds. For example, Lambda Function assets are copied over to Lambda during deployment. Therefore, it is not necessary to store the asset in S3, so we consider those deployTime assets. Default: false
        :param display_name: A display name for this asset. If supplied, the display name will be used in locations where the asset identifier is printed, like in the CLI progress information. If the same asset is added multiple times, the display name of the first occurrence is used. The default is the construct path of the Asset construct, with respect to the enclosing stack. If the asset is produced by a construct helper function (such as ``lambda.Code.fromAsset()``), this will look like ``MyFunction/Code``. We use the stack-relative construct path so that in the common case where you have multiple stacks with the same asset, we won't show something like ``/MyBetaStack/MyFunction/Code`` when you are actually deploying to production. Default: - Stack-relative construct path
        :param readers: A list of principals that should be able to read this asset from S3. You can use ``asset.grantRead(principal)`` to grant read permissions later. Default: - No principals that can read file asset.
        :param source_kms_key: The ARN of the KMS key used to encrypt the handler code. Default: - the default server-side encryption with Amazon S3 managed keys(SSE-S3) key will be used.
        :param asset_hash: Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: Bundle the asset by executing a command in a Docker container or a custom bundling provider. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise
        :param exclude: File paths matching the patterns will be excluded. See ``ignoreMode`` to set the matching behavior. Has no effect on Assets bundled using the ``bundling`` property. Default: - nothing is excluded
        :param follow_symlinks: A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param ignore_mode: The ignore behavior to use for ``exclude`` patterns. Default: IgnoreMode.GLOB
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8d060d6bc8fc5b369cfe71dc05efc964aa6b083b674baa7dc17f1762a8d68421)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        options = _AssetOptions_2aa69621(
            deploy_time=deploy_time,
            display_name=display_name,
            readers=readers,
            source_kms_key=source_kms_key,
            asset_hash=asset_hash,
            asset_hash_type=asset_hash_type,
            bundling=bundling,
            exclude=exclude,
            follow_symlinks=follow_symlinks,
            ignore_mode=ignore_mode,
        )

        jsii.create(self.__class__, self, [path, options])

    @jsii.member(jsii_name="bind")
    def bind(self, scope: _constructs_77d1e7e8.Construct) -> ApiDefinitionConfig:
        '''Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.

        :param scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e81dd80c8a5aadd62f5dcc884b89ddc1076b787b54c75f996c03c1f1859e9b79)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast(ApiDefinitionConfig, jsii.invoke(self, "bind", [scope]))

    @jsii.member(jsii_name="bindAfterCreate")
    def bind_after_create(
        self,
        scope: _constructs_77d1e7e8.Construct,
        rest_api: "IRestApi",
    ) -> None:
        '''Called after the CFN RestApi resource has been created to allow the Api Definition to bind to it.

        Specifically it's required to allow assets to add
        metadata for tooling like SAM CLI to be able to find their origins.

        :param scope: -
        :param rest_api: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b1885a209c59071b4bad06f2e2988a4c685fbb214e4cfd8ee60a749d822b13fc)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument rest_api", value=rest_api, expected_type=type_hints["rest_api"])
        return typing.cast(None, jsii.invoke(self, "bindAfterCreate", [scope, rest_api]))


@jsii.enum(jsii_type="aws-cdk-lib.aws_apigateway.AuthorizationType")
class AuthorizationType(enum.Enum):
    '''
    :exampleMetadata: infused

    Example::

        # books: apigateway.Resource
        user_pool = cognito.UserPool(self, "UserPool")
        
        auth = apigateway.CognitoUserPoolsAuthorizer(self, "booksAuthorizer",
            cognito_user_pools=[user_pool]
        )
        books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
            authorizer=auth,
            authorization_type=apigateway.AuthorizationType.COGNITO
        )
    '''

    NONE = "NONE"
    '''Open access.'''
    IAM = "IAM"
    '''Use AWS IAM permissions.'''
    CUSTOM = "CUSTOM"
    '''Use a custom authorizer.'''
    COGNITO = "COGNITO"
    '''Use an AWS Cognito user pool.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.AwsIntegrationProps",
    jsii_struct_bases=[],
    name_mapping={
        "service": "service",
        "action": "action",
        "action_parameters": "actionParameters",
        "integration_http_method": "integrationHttpMethod",
        "options": "options",
        "path": "path",
        "proxy": "proxy",
        "region": "region",
        "subdomain": "subdomain",
    },
)
class AwsIntegrationProps:
    def __init__(
        self,
        *,
        service: builtins.str,
        action: typing.Optional[builtins.str] = None,
        action_parameters: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        integration_http_method: typing.Optional[builtins.str] = None,
        options: typing.Optional[typing.Union["IntegrationOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        path: typing.Optional[builtins.str] = None,
        proxy: typing.Optional[builtins.bool] = None,
        region: typing.Optional[builtins.str] = None,
        subdomain: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param service: The name of the integrated AWS service (e.g. ``s3``).
        :param action: The AWS action to perform in the integration. Use ``actionParams`` to specify key-value params for the action. Mutually exclusive with ``path``.
        :param action_parameters: Parameters for the action. ``action`` must be set, and ``path`` must be undefined. The action params will be URL encoded.
        :param integration_http_method: The integration's HTTP method type. Default: POST
        :param options: Integration options, such as content handling, request/response mapping, etc.
        :param path: The path to use for path-base APIs. For example, for S3 GET, you can set path to ``bucket/key``. For lambda, you can set path to ``2015-03-31/functions/${function-arn}/invocations`` Mutually exclusive with the ``action`` options.
        :param proxy: Use AWS_PROXY integration. Default: false
        :param region: The region of the integrated AWS service. Default: - same region as the stack
        :param subdomain: A designated subdomain supported by certain AWS service for fast host-name lookup.

        :exampleMetadata: infused

        Example::

            get_message_integration = apigateway.AwsIntegration(
                service="sqs",
                path="queueName",
                region="eu-west-1"
            )
        '''
        if isinstance(options, dict):
            options = IntegrationOptions(**options)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b4f3bcb03a9e9138a26dd7d83cccc47475dedb681bf622b729282c23f0a3ff24)
            check_type(argname="argument service", value=service, expected_type=type_hints["service"])
            check_type(argname="argument action", value=action, expected_type=type_hints["action"])
            check_type(argname="argument action_parameters", value=action_parameters, expected_type=type_hints["action_parameters"])
            check_type(argname="argument integration_http_method", value=integration_http_method, expected_type=type_hints["integration_http_method"])
            check_type(argname="argument options", value=options, expected_type=type_hints["options"])
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
            check_type(argname="argument proxy", value=proxy, expected_type=type_hints["proxy"])
            check_type(argname="argument region", value=region, expected_type=type_hints["region"])
            check_type(argname="argument subdomain", value=subdomain, expected_type=type_hints["subdomain"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "service": service,
        }
        if action is not None:
            self._values["action"] = action
        if action_parameters is not None:
            self._values["action_parameters"] = action_parameters
        if integration_http_method is not None:
            self._values["integration_http_method"] = integration_http_method
        if options is not None:
            self._values["options"] = options
        if path is not None:
            self._values["path"] = path
        if proxy is not None:
            self._values["proxy"] = proxy
        if region is not None:
            self._values["region"] = region
        if subdomain is not None:
            self._values["subdomain"] = subdomain

    @builtins.property
    def service(self) -> builtins.str:
        '''The name of the integrated AWS service (e.g. ``s3``).'''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def action(self) -> typing.Optional[builtins.str]:
        '''The AWS action to perform in the integration.

        Use ``actionParams`` to specify key-value params for the action.

        Mutually exclusive with ``path``.
        '''
        result = self._values.get("action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def action_parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Parameters for the action.

        ``action`` must be set, and ``path`` must be undefined.
        The action params will be URL encoded.
        '''
        result = self._values.get("action_parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def integration_http_method(self) -> typing.Optional[builtins.str]:
        '''The integration's HTTP method type.

        :default: POST
        '''
        result = self._values.get("integration_http_method")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def options(self) -> typing.Optional["IntegrationOptions"]:
        '''Integration options, such as content handling, request/response mapping, etc.'''
        result = self._values.get("options")
        return typing.cast(typing.Optional["IntegrationOptions"], result)

    @builtins.property
    def path(self) -> typing.Optional[builtins.str]:
        '''The path to use for path-base APIs.

        For example, for S3 GET, you can set path to ``bucket/key``.
        For lambda, you can set path to ``2015-03-31/functions/${function-arn}/invocations``

        Mutually exclusive with the ``action`` options.
        '''
        result = self._values.get("path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def proxy(self) -> typing.Optional[builtins.bool]:
        '''Use AWS_PROXY integration.

        :default: false
        '''
        result = self._values.get("proxy")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def region(self) -> typing.Optional[builtins.str]:
        '''The region of the integrated AWS service.

        :default: - same region as the stack
        '''
        result = self._values.get("region")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def subdomain(self) -> typing.Optional[builtins.str]:
        '''A designated subdomain supported by certain AWS service for fast host-name lookup.'''
        result = self._values.get("subdomain")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsIntegrationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class BasePathMapping(
    _Resource_45bc6135,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.BasePathMapping",
):
    '''This resource creates a base path that clients who call your API must use in the invocation URL.

    Unless you're importing a domain with ``DomainName.fromDomainNameAttributes()``,
    you can use ``DomainName.addBasePathMapping()`` to define mappings.

    :exampleMetadata: infused

    Example::

        # api: apigateway.RestApi
        
        
        domain_name = apigateway.DomainName.from_domain_name_attributes(self, "DomainName",
            domain_name="domainName",
            domain_name_alias_hosted_zone_id="domainNameAliasHostedZoneId",
            domain_name_alias_target="domainNameAliasTarget"
        )
        
        apigateway.BasePathMapping(self, "BasePathMapping",
            domain_name=domain_name,
            rest_api=api
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        domain_name: "IDomainName",
        rest_api: "IRestApi",
        attach_to_stage: typing.Optional[builtins.bool] = None,
        base_path: typing.Optional[builtins.str] = None,
        stage: typing.Optional["Stage"] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param domain_name: The DomainName to associate with this base path mapping.
        :param rest_api: The RestApi resource to target.
        :param attach_to_stage: Whether to attach the base path mapping to a stage. Use this property to create a base path mapping without attaching it to the Rest API default stage. This property is ignored if ``stage`` is provided. Default: - true
        :param base_path: The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string. Default: - map requests from the domain root (e.g. ``example.com``). If this is undefined, no additional mappings will be allowed on this domain name.
        :param stage: The Deployment stage of API [disable-awslint:ref-via-interface]. Default: - map to deploymentStage of restApi otherwise stage needs to pass in URL
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ef549043f328dc8eafa81dc926877350a07f0bbf5f070dbbc6d4486d7b65b6e8)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = BasePathMappingProps(
            domain_name=domain_name,
            rest_api=rest_api,
            attach_to_stage=attach_to_stage,
            base_path=base_path,
            stage=stage,
        )

        jsii.create(self.__class__, self, [scope, id, props])


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.BasePathMappingOptions",
    jsii_struct_bases=[],
    name_mapping={
        "attach_to_stage": "attachToStage",
        "base_path": "basePath",
        "stage": "stage",
    },
)
class BasePathMappingOptions:
    def __init__(
        self,
        *,
        attach_to_stage: typing.Optional[builtins.bool] = None,
        base_path: typing.Optional[builtins.str] = None,
        stage: typing.Optional["Stage"] = None,
    ) -> None:
        '''
        :param attach_to_stage: Whether to attach the base path mapping to a stage. Use this property to create a base path mapping without attaching it to the Rest API default stage. This property is ignored if ``stage`` is provided. Default: - true
        :param base_path: The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string. Default: - map requests from the domain root (e.g. ``example.com``). If this is undefined, no additional mappings will be allowed on this domain name.
        :param stage: The Deployment stage of API [disable-awslint:ref-via-interface]. Default: - map to deploymentStage of restApi otherwise stage needs to pass in URL

        :exampleMetadata: infused

        Example::

            # domain: apigateway.DomainName
            # api1: apigateway.RestApi
            # api2: apigateway.RestApi
            
            
            domain.add_base_path_mapping(api1, base_path="go-to-api1")
            domain.add_base_path_mapping(api2, base_path="boom")
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fee7aa7a933ab856b9447872081423504f900cf3f149437b4ec7078c925b551b)
            check_type(argname="argument attach_to_stage", value=attach_to_stage, expected_type=type_hints["attach_to_stage"])
            check_type(argname="argument base_path", value=base_path, expected_type=type_hints["base_path"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if attach_to_stage is not None:
            self._values["attach_to_stage"] = attach_to_stage
        if base_path is not None:
            self._values["base_path"] = base_path
        if stage is not None:
            self._values["stage"] = stage

    @builtins.property
    def attach_to_stage(self) -> typing.Optional[builtins.bool]:
        '''Whether to attach the base path mapping to a stage.

        Use this property to create a base path mapping without attaching it to the Rest API default stage.
        This property is ignored if ``stage`` is provided.

        :default: - true
        '''
        result = self._values.get("attach_to_stage")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def base_path(self) -> typing.Optional[builtins.str]:
        '''The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string.

        :default:

        - map requests from the domain root (e.g. ``example.com``). If this
        is undefined, no additional mappings will be allowed on this domain name.
        '''
        result = self._values.get("base_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage(self) -> typing.Optional["Stage"]:
        '''The Deployment stage of API [disable-awslint:ref-via-interface].

        :default: - map to deploymentStage of restApi otherwise stage needs to pass in URL
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional["Stage"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BasePathMappingOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.BasePathMappingProps",
    jsii_struct_bases=[BasePathMappingOptions],
    name_mapping={
        "attach_to_stage": "attachToStage",
        "base_path": "basePath",
        "stage": "stage",
        "domain_name": "domainName",
        "rest_api": "restApi",
    },
)
class BasePathMappingProps(BasePathMappingOptions):
    def __init__(
        self,
        *,
        attach_to_stage: typing.Optional[builtins.bool] = None,
        base_path: typing.Optional[builtins.str] = None,
        stage: typing.Optional["Stage"] = None,
        domain_name: "IDomainName",
        rest_api: "IRestApi",
    ) -> None:
        '''
        :param attach_to_stage: Whether to attach the base path mapping to a stage. Use this property to create a base path mapping without attaching it to the Rest API default stage. This property is ignored if ``stage`` is provided. Default: - true
        :param base_path: The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string. Default: - map requests from the domain root (e.g. ``example.com``). If this is undefined, no additional mappings will be allowed on this domain name.
        :param stage: The Deployment stage of API [disable-awslint:ref-via-interface]. Default: - map to deploymentStage of restApi otherwise stage needs to pass in URL
        :param domain_name: The DomainName to associate with this base path mapping.
        :param rest_api: The RestApi resource to target.

        :exampleMetadata: infused

        Example::

            # api: apigateway.RestApi
            
            
            domain_name = apigateway.DomainName.from_domain_name_attributes(self, "DomainName",
                domain_name="domainName",
                domain_name_alias_hosted_zone_id="domainNameAliasHostedZoneId",
                domain_name_alias_target="domainNameAliasTarget"
            )
            
            apigateway.BasePathMapping(self, "BasePathMapping",
                domain_name=domain_name,
                rest_api=api
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9e348cc160af15e382c6a3c6d66cd04543644f3672b2679f19c801ae8a07b256)
            check_type(argname="argument attach_to_stage", value=attach_to_stage, expected_type=type_hints["attach_to_stage"])
            check_type(argname="argument base_path", value=base_path, expected_type=type_hints["base_path"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument rest_api", value=rest_api, expected_type=type_hints["rest_api"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "domain_name": domain_name,
            "rest_api": rest_api,
        }
        if attach_to_stage is not None:
            self._values["attach_to_stage"] = attach_to_stage
        if base_path is not None:
            self._values["base_path"] = base_path
        if stage is not None:
            self._values["stage"] = stage

    @builtins.property
    def attach_to_stage(self) -> typing.Optional[builtins.bool]:
        '''Whether to attach the base path mapping to a stage.

        Use this property to create a base path mapping without attaching it to the Rest API default stage.
        This property is ignored if ``stage`` is provided.

        :default: - true
        '''
        result = self._values.get("attach_to_stage")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def base_path(self) -> typing.Optional[builtins.str]:
        '''The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string.

        :default:

        - map requests from the domain root (e.g. ``example.com``). If this
        is undefined, no additional mappings will be allowed on this domain name.
        '''
        result = self._values.get("base_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage(self) -> typing.Optional["Stage"]:
        '''The Deployment stage of API [disable-awslint:ref-via-interface].

        :default: - map to deploymentStage of restApi otherwise stage needs to pass in URL
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional["Stage"], result)

    @builtins.property
    def domain_name(self) -> "IDomainName":
        '''The DomainName to associate with this base path mapping.'''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast("IDomainName", result)

    @builtins.property
    def rest_api(self) -> "IRestApi":
        '''The RestApi resource to target.'''
        result = self._values.get("rest_api")
        assert result is not None, "Required property 'rest_api' is missing"
        return typing.cast("IRestApi", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BasePathMappingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnAccount(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnAccount",
):
    '''The ``AWS::ApiGateway::Account`` resource specifies the IAM role that Amazon API Gateway uses to write API logs to Amazon CloudWatch Logs.

    To avoid overwriting other roles, you should only have one ``AWS::ApiGateway::Account`` resource per region per account.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-account.html
    :cloudformationResource: AWS::ApiGateway::Account
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_account = apigateway.CfnAccount(self, "MyCfnAccount",
            cloud_watch_role_arn="cloudWatchRoleArn"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        cloud_watch_role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param cloud_watch_role_arn: The ARN of an Amazon CloudWatch role for the current Account.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__38bd2193bbc36613b2cff1c53295e808ec15988652a9f30d8104a272c9dd79d9)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAccountProps(cloud_watch_role_arn=cloud_watch_role_arn)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__17d0380f6f2ef94fc59e0cd705a654d4fe9eb705fdb714c2daab4b6b3ebe4cff)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f8bcef013f8c5e25c1289a9e8ee70090085bad08d37f8cf7db17dc8a298137d2)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID for the account.

        For example: ``abc123`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="cloudWatchRoleArn")
    def cloud_watch_role_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of an Amazon CloudWatch role for the current Account.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cloudWatchRoleArn"))

    @cloud_watch_role_arn.setter
    def cloud_watch_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__508e5fd6405b784daeb72f2080dc15b070457ce1a45f97c707562d294cb3bd07)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cloudWatchRoleArn", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnAccountProps",
    jsii_struct_bases=[],
    name_mapping={"cloud_watch_role_arn": "cloudWatchRoleArn"},
)
class CfnAccountProps:
    def __init__(
        self,
        *,
        cloud_watch_role_arn: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnAccount``.

        :param cloud_watch_role_arn: The ARN of an Amazon CloudWatch role for the current Account.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-account.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_account_props = apigateway.CfnAccountProps(
                cloud_watch_role_arn="cloudWatchRoleArn"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cb07dd1e7339c3c46046b4af92bf506a1638d186c8ee249b12e5014bd9c1a4d3)
            check_type(argname="argument cloud_watch_role_arn", value=cloud_watch_role_arn, expected_type=type_hints["cloud_watch_role_arn"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cloud_watch_role_arn is not None:
            self._values["cloud_watch_role_arn"] = cloud_watch_role_arn

    @builtins.property
    def cloud_watch_role_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of an Amazon CloudWatch role for the current Account.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-account.html#cfn-apigateway-account-cloudwatchrolearn
        '''
        result = self._values.get("cloud_watch_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAccountProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnApiKey(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnApiKey",
):
    '''The ``AWS::ApiGateway::ApiKey`` resource creates a unique key that you can distribute to clients who are executing API Gateway ``Method`` resources that require an API key.

    To specify which API key clients must use, map the API key with the ``RestApi`` and ``Stage`` resources that include the methods that require a key.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html
    :cloudformationResource: AWS::ApiGateway::ApiKey
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_api_key = apigateway.CfnApiKey(self, "MyCfnApiKey",
            customer_id="customerId",
            description="description",
            enabled=False,
            generate_distinct_id=False,
            name="name",
            stage_keys=[apigateway.CfnApiKey.StageKeyProperty(
                rest_api_id="restApiId",
                stage_name="stageName"
            )],
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            value="value"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        customer_id: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        generate_distinct_id: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        name: typing.Optional[builtins.str] = None,
        stage_keys: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnApiKey.StageKeyProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        value: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param customer_id: An AWS Marketplace customer identifier, when integrating with the AWS SaaS Marketplace.
        :param description: The description of the ApiKey.
        :param enabled: Specifies whether the ApiKey can be used by callers. Default: - false
        :param generate_distinct_id: Specifies whether ( ``true`` ) or not ( ``false`` ) the key identifier is distinct from the created API key value. This parameter is deprecated and should not be used.
        :param name: A name for the API key. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the API key name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . .. epigraph:: If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param stage_keys: DEPRECATED FOR USAGE PLANS - Specifies stages associated with the API key.
        :param tags: The key-value map of strings. The valid character set is [a-zA-Z+-=._:/]. The tag key can be up to 128 characters and must not start with ``aws:`` . The tag value can be up to 256 characters.
        :param value: Specifies a value of the API key.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7b22ef3a96b2535227fcee63bac2bd1636a3ecaf8ae447d48e70ec92689693d2)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnApiKeyProps(
            customer_id=customer_id,
            description=description,
            enabled=enabled,
            generate_distinct_id=generate_distinct_id,
            name=name,
            stage_keys=stage_keys,
            tags=tags,
            value=value,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3d6e2d573fd59cd7bda27f8af3a06d556bf1f848f8a8571e3bb89f6ae03687b8)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9d447711c93422d5ce557200e57597c279242f4fdb27fe4bc9a87e3fc85f378c)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrApiKeyId")
    def attr_api_key_id(self) -> builtins.str:
        '''The ID for the API key.

        For example: ``abc123`` .

        :cloudformationAttribute: APIKeyId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrApiKeyId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="customerId")
    def customer_id(self) -> typing.Optional[builtins.str]:
        '''An AWS Marketplace customer identifier, when integrating with the AWS SaaS Marketplace.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "customerId"))

    @customer_id.setter
    def customer_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__90a9f552f8e56d3808a44164a2ecfe14be5caab46b1daca23be075cde54d6681)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "customerId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the ApiKey.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a8daff39a782f7c2cfea28dfb80b237f38a3cf29a00cb51b95696d8b18068467)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="enabled")
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the ApiKey can be used by callers.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "enabled"))

    @enabled.setter
    def enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__55577c77ce38fb70e6dc28c7fe5a30c2189533e40f2941cf7f725d0744353a8b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "enabled", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="generateDistinctId")
    def generate_distinct_id(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether ( ``true`` ) or not ( ``false`` ) the key identifier is distinct from the created API key value.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "generateDistinctId"))

    @generate_distinct_id.setter
    def generate_distinct_id(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5e376c81196abe5988b927a412341a6e2d2a2afcbc15de891ab4ab515d7d8575)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "generateDistinctId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''A name for the API key.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5a5d6223e14c62c01c13b14c4c60aca957cf7675f5e1cb1a49dfcf0ce326bdea)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="stageKeys")
    def stage_keys(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnApiKey.StageKeyProperty"]]]]:
        '''DEPRECATED FOR USAGE PLANS - Specifies stages associated with the API key.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnApiKey.StageKeyProperty"]]]], jsii.get(self, "stageKeys"))

    @stage_keys.setter
    def stage_keys(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnApiKey.StageKeyProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f2912acb683bf8ab1749c9ae328a9eb41061f378f662bb1e5c7a0f9511b2f9a6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stageKeys", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The key-value map of strings.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__278929d2497b6992ae8c54f09fa48eee91bfd399ce109802d2af49204daa464d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="value")
    def value(self) -> typing.Optional[builtins.str]:
        '''Specifies a value of the API key.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "value"))

    @value.setter
    def value(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a5ba9bf573202fa63e2f9db125447a71a2606d30799e25ac0e7b350876fd53b4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "value", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnApiKey.StageKeyProperty",
        jsii_struct_bases=[],
        name_mapping={"rest_api_id": "restApiId", "stage_name": "stageName"},
    )
    class StageKeyProperty:
        def __init__(
            self,
            *,
            rest_api_id: typing.Optional[builtins.str] = None,
            stage_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``StageKey`` is a property of the `AWS::ApiGateway::ApiKey <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html>`_ resource that specifies the stage to associate with the API key. This association allows only clients with the key to make requests to methods in that stage.

            :param rest_api_id: The string identifier of the associated RestApi.
            :param stage_name: The stage name associated with the stage key.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-apikey-stagekey.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                stage_key_property = apigateway.CfnApiKey.StageKeyProperty(
                    rest_api_id="restApiId",
                    stage_name="stageName"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5059296d4b54bc12aa45e4a9a14f3c13251789b66016f4c98f7ce019963298d4)
                check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
                check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if rest_api_id is not None:
                self._values["rest_api_id"] = rest_api_id
            if stage_name is not None:
                self._values["stage_name"] = stage_name

        @builtins.property
        def rest_api_id(self) -> typing.Optional[builtins.str]:
            '''The string identifier of the associated RestApi.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-apikey-stagekey.html#cfn-apigateway-apikey-stagekey-restapiid
            '''
            result = self._values.get("rest_api_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def stage_name(self) -> typing.Optional[builtins.str]:
            '''The stage name associated with the stage key.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-apikey-stagekey.html#cfn-apigateway-apikey-stagekey-stagename
            '''
            result = self._values.get("stage_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StageKeyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnApiKeyProps",
    jsii_struct_bases=[],
    name_mapping={
        "customer_id": "customerId",
        "description": "description",
        "enabled": "enabled",
        "generate_distinct_id": "generateDistinctId",
        "name": "name",
        "stage_keys": "stageKeys",
        "tags": "tags",
        "value": "value",
    },
)
class CfnApiKeyProps:
    def __init__(
        self,
        *,
        customer_id: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        generate_distinct_id: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        name: typing.Optional[builtins.str] = None,
        stage_keys: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnApiKey.StageKeyProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        value: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnApiKey``.

        :param customer_id: An AWS Marketplace customer identifier, when integrating with the AWS SaaS Marketplace.
        :param description: The description of the ApiKey.
        :param enabled: Specifies whether the ApiKey can be used by callers. Default: - false
        :param generate_distinct_id: Specifies whether ( ``true`` ) or not ( ``false`` ) the key identifier is distinct from the created API key value. This parameter is deprecated and should not be used.
        :param name: A name for the API key. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the API key name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . .. epigraph:: If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param stage_keys: DEPRECATED FOR USAGE PLANS - Specifies stages associated with the API key.
        :param tags: The key-value map of strings. The valid character set is [a-zA-Z+-=._:/]. The tag key can be up to 128 characters and must not start with ``aws:`` . The tag value can be up to 256 characters.
        :param value: Specifies a value of the API key.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_api_key_props = apigateway.CfnApiKeyProps(
                customer_id="customerId",
                description="description",
                enabled=False,
                generate_distinct_id=False,
                name="name",
                stage_keys=[apigateway.CfnApiKey.StageKeyProperty(
                    rest_api_id="restApiId",
                    stage_name="stageName"
                )],
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                value="value"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__91ca5df9290cd06f3ff1b7f8dab587c3d3d10280f0a4e21d73baba990769ac97)
            check_type(argname="argument customer_id", value=customer_id, expected_type=type_hints["customer_id"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument generate_distinct_id", value=generate_distinct_id, expected_type=type_hints["generate_distinct_id"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument stage_keys", value=stage_keys, expected_type=type_hints["stage_keys"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if customer_id is not None:
            self._values["customer_id"] = customer_id
        if description is not None:
            self._values["description"] = description
        if enabled is not None:
            self._values["enabled"] = enabled
        if generate_distinct_id is not None:
            self._values["generate_distinct_id"] = generate_distinct_id
        if name is not None:
            self._values["name"] = name
        if stage_keys is not None:
            self._values["stage_keys"] = stage_keys
        if tags is not None:
            self._values["tags"] = tags
        if value is not None:
            self._values["value"] = value

    @builtins.property
    def customer_id(self) -> typing.Optional[builtins.str]:
        '''An AWS Marketplace customer identifier, when integrating with the AWS SaaS Marketplace.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-customerid
        '''
        result = self._values.get("customer_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the ApiKey.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether the ApiKey can be used by callers.

        :default: - false

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-enabled
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def generate_distinct_id(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether ( ``true`` ) or not ( ``false`` ) the key identifier is distinct from the created API key value.

        This parameter is deprecated and should not be used.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-generatedistinctid
        '''
        result = self._values.get("generate_distinct_id")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''A name for the API key.

        If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the API key name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ .
        .. epigraph::

           If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage_keys(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnApiKey.StageKeyProperty]]]]:
        '''DEPRECATED FOR USAGE PLANS - Specifies stages associated with the API key.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-stagekeys
        '''
        result = self._values.get("stage_keys")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnApiKey.StageKeyProperty]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The key-value map of strings.

        The valid character set is [a-zA-Z+-=._:/]. The tag key can be up to 128 characters and must not start with ``aws:`` . The tag value can be up to 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def value(self) -> typing.Optional[builtins.str]:
        '''Specifies a value of the API key.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-apikey.html#cfn-apigateway-apikey-value
        '''
        result = self._values.get("value")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnApiKeyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnAuthorizer(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnAuthorizer",
):
    '''The ``AWS::ApiGateway::Authorizer`` resource creates an authorization layer that API Gateway activates for methods that have authorization enabled.

    API Gateway activates the authorizer when a client calls those methods.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html
    :cloudformationResource: AWS::ApiGateway::Authorizer
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_authorizer = apigateway.CfnAuthorizer(self, "MyCfnAuthorizer",
            name="name",
            rest_api_id="restApiId",
            type="type",
        
            # the properties below are optional
            authorizer_credentials="authorizerCredentials",
            authorizer_result_ttl_in_seconds=123,
            authorizer_uri="authorizerUri",
            auth_type="authType",
            identity_source="identitySource",
            identity_validation_expression="identityValidationExpression",
            provider_arns=["providerArns"]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        name: builtins.str,
        rest_api_id: builtins.str,
        type: builtins.str,
        authorizer_credentials: typing.Optional[builtins.str] = None,
        authorizer_result_ttl_in_seconds: typing.Optional[jsii.Number] = None,
        authorizer_uri: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[builtins.str] = None,
        identity_source: typing.Optional[builtins.str] = None,
        identity_validation_expression: typing.Optional[builtins.str] = None,
        provider_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param name: The name of the authorizer.
        :param rest_api_id: The string identifier of the associated RestApi.
        :param type: The authorizer type. Valid values are ``TOKEN`` for a Lambda function using a single authorization token submitted in a custom header, ``REQUEST`` for a Lambda function using incoming request parameters, and ``COGNITO_USER_POOLS`` for using an Amazon Cognito user pool.
        :param authorizer_credentials: Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify an IAM role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based permissions on the Lambda function, specify null.
        :param authorizer_result_ttl_in_seconds: The TTL in seconds of cached authorizer results. If it equals 0, authorization caching is disabled. If it is greater than 0, API Gateway will cache authorizer responses. If this field is not set, the default value is 300. The maximum value is 3600, or 1 hour.
        :param authorizer_uri: Specifies the authorizer's Uniform Resource Identifier (URI). For ``TOKEN`` or ``REQUEST`` authorizers, this must be a well-formed Lambda function URI, for example, ``arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda:us-west-2:{account_id}:function:{lambda_function_name}/invocations`` . In general, the URI has this form ``arn:aws:apigateway:{region}:lambda:path/{service_api}`` , where ``{region}`` is the same as the region hosting the Lambda function, ``path`` indicates that the remaining substring in the URI should be treated as the path to the resource, including the initial ``/`` . For Lambda functions, this is usually of the form ``/2015-03-31/functions/[FunctionARN]/invocations`` .
        :param auth_type: Optional customer-defined field, used in OpenAPI imports and exports without functional impact.
        :param identity_source: The identity source for which authorization is requested. For a ``TOKEN`` or ``COGNITO_USER_POOLS`` authorizer, this is required and specifies the request header mapping expression for the custom header holding the authorization token submitted by the client. For example, if the token header name is ``Auth`` , the header mapping expression is ``method.request.header.Auth`` . For the ``REQUEST`` authorizer, this is required when authorization caching is enabled. The value is a comma-separated string of one or more mapping expressions of the specified request parameters. For example, if an ``Auth`` header, a ``Name`` query string parameter are defined as identity sources, this value is ``method.request.header.Auth, method.request.querystring.Name`` . These parameters will be used to derive the authorization caching key and to perform runtime validation of the ``REQUEST`` authorizer by verifying all of the identity-related request parameters are present, not null and non-empty. Only when this is true does the authorizer invoke the authorizer Lambda function, otherwise, it returns a 401 Unauthorized response without calling the Lambda function. The valid value is a string of comma-separated mapping expressions of the specified request parameters. When the authorization caching is not enabled, this property is optional.
        :param identity_validation_expression: A validation expression for the incoming identity token. For ``TOKEN`` authorizers, this value is a regular expression. For ``COGNITO_USER_POOLS`` authorizers, API Gateway will match the ``aud`` field of the incoming token from the client against the specified regular expression. It will invoke the authorizer's Lambda function when there is a match. Otherwise, it will return a 401 Unauthorized response without calling the Lambda function. The validation expression does not apply to the ``REQUEST`` authorizer.
        :param provider_arns: A list of the Amazon Cognito user pool ARNs for the ``COGNITO_USER_POOLS`` authorizer. Each element is of this format: ``arn:aws:cognito-idp:{region}:{account_id}:userpool/{user_pool_id}`` . For a ``TOKEN`` or ``REQUEST`` authorizer, this is not defined.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b2a6427d61aed56b5bd8abeaeb6e64934982800222c2d3f84ac525023e5c8c3)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnAuthorizerProps(
            name=name,
            rest_api_id=rest_api_id,
            type=type,
            authorizer_credentials=authorizer_credentials,
            authorizer_result_ttl_in_seconds=authorizer_result_ttl_in_seconds,
            authorizer_uri=authorizer_uri,
            auth_type=auth_type,
            identity_source=identity_source,
            identity_validation_expression=identity_validation_expression,
            provider_arns=provider_arns,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cc4a2b9ed8a82f7b8f61af192ed94d5c519637c1c4a6450b858364f86243cdb0)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__53e42fe56acb586d53dcdbdaad7b7683bc96cda75cbb24c26b3c04f6fdd90ab4)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrAuthorizerId")
    def attr_authorizer_id(self) -> builtins.str:
        '''The ID for the authorizer.

        For example: ``abc123`` .

        :cloudformationAttribute: AuthorizerId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAuthorizerId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name of the authorizer.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0ac5bd8743d71dcd039747b51726b2e14050c0735955a7a26b844c161c3a74e4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.'''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__97ce5bb3b8db2fb9d5f8a536101f266dd6e7b79fdb8dbc34c19edbc23cf4d92e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''The authorizer type.'''
        return typing.cast(builtins.str, jsii.get(self, "type"))

    @type.setter
    def type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1aa4a272e4f5cd708bec4db69f4e3b2178dfd3b9f6d5d06f2f95c17815253134)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "type", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="authorizerCredentials")
    def authorizer_credentials(self) -> typing.Optional[builtins.str]:
        '''Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizerCredentials"))

    @authorizer_credentials.setter
    def authorizer_credentials(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8c032cf00d3c33602d5c37baa75de21ec003bf5a6defca2385e421389fc9e932)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerCredentials", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="authorizerResultTtlInSeconds")
    def authorizer_result_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The TTL in seconds of cached authorizer results.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "authorizerResultTtlInSeconds"))

    @authorizer_result_ttl_in_seconds.setter
    def authorizer_result_ttl_in_seconds(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f73b9fff1c9759d3aabe42dea2015ab2e6deac69b065083f1eca55d5fea7ce7a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerResultTtlInSeconds", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="authorizerUri")
    def authorizer_uri(self) -> typing.Optional[builtins.str]:
        '''Specifies the authorizer's Uniform Resource Identifier (URI).'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizerUri"))

    @authorizer_uri.setter
    def authorizer_uri(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8fed74eb319d73929ed995c115ef87d197a9a75e404a0153b65c18f503d5d911)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerUri", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="authType")
    def auth_type(self) -> typing.Optional[builtins.str]:
        '''Optional customer-defined field, used in OpenAPI imports and exports without functional impact.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authType"))

    @auth_type.setter
    def auth_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__37cfc68f3505fd28c894f798088d067a97c42e3f984e409335f42995ece91af9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="identitySource")
    def identity_source(self) -> typing.Optional[builtins.str]:
        '''The identity source for which authorization is requested.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "identitySource"))

    @identity_source.setter
    def identity_source(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__81a19f23e6a62959adf135e84e3fc8ee1087db10c843ff2074517e0325257164)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identitySource", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="identityValidationExpression")
    def identity_validation_expression(self) -> typing.Optional[builtins.str]:
        '''A validation expression for the incoming identity token.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "identityValidationExpression"))

    @identity_validation_expression.setter
    def identity_validation_expression(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9d23a7350b66d64a3a315669820f075d6bbb29c7e65b23d4005f0fdcd817d2db)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "identityValidationExpression", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="providerArns")
    def provider_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the Amazon Cognito user pool ARNs for the ``COGNITO_USER_POOLS`` authorizer.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "providerArns"))

    @provider_arns.setter
    def provider_arns(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6f4d0fe625a57ee895a4018dba27e510cacbfc59d989a0fa3bd0732eb5fc688e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "providerArns", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnAuthorizerProps",
    jsii_struct_bases=[],
    name_mapping={
        "name": "name",
        "rest_api_id": "restApiId",
        "type": "type",
        "authorizer_credentials": "authorizerCredentials",
        "authorizer_result_ttl_in_seconds": "authorizerResultTtlInSeconds",
        "authorizer_uri": "authorizerUri",
        "auth_type": "authType",
        "identity_source": "identitySource",
        "identity_validation_expression": "identityValidationExpression",
        "provider_arns": "providerArns",
    },
)
class CfnAuthorizerProps:
    def __init__(
        self,
        *,
        name: builtins.str,
        rest_api_id: builtins.str,
        type: builtins.str,
        authorizer_credentials: typing.Optional[builtins.str] = None,
        authorizer_result_ttl_in_seconds: typing.Optional[jsii.Number] = None,
        authorizer_uri: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[builtins.str] = None,
        identity_source: typing.Optional[builtins.str] = None,
        identity_validation_expression: typing.Optional[builtins.str] = None,
        provider_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Properties for defining a ``CfnAuthorizer``.

        :param name: The name of the authorizer.
        :param rest_api_id: The string identifier of the associated RestApi.
        :param type: The authorizer type. Valid values are ``TOKEN`` for a Lambda function using a single authorization token submitted in a custom header, ``REQUEST`` for a Lambda function using incoming request parameters, and ``COGNITO_USER_POOLS`` for using an Amazon Cognito user pool.
        :param authorizer_credentials: Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify an IAM role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based permissions on the Lambda function, specify null.
        :param authorizer_result_ttl_in_seconds: The TTL in seconds of cached authorizer results. If it equals 0, authorization caching is disabled. If it is greater than 0, API Gateway will cache authorizer responses. If this field is not set, the default value is 300. The maximum value is 3600, or 1 hour.
        :param authorizer_uri: Specifies the authorizer's Uniform Resource Identifier (URI). For ``TOKEN`` or ``REQUEST`` authorizers, this must be a well-formed Lambda function URI, for example, ``arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda:us-west-2:{account_id}:function:{lambda_function_name}/invocations`` . In general, the URI has this form ``arn:aws:apigateway:{region}:lambda:path/{service_api}`` , where ``{region}`` is the same as the region hosting the Lambda function, ``path`` indicates that the remaining substring in the URI should be treated as the path to the resource, including the initial ``/`` . For Lambda functions, this is usually of the form ``/2015-03-31/functions/[FunctionARN]/invocations`` .
        :param auth_type: Optional customer-defined field, used in OpenAPI imports and exports without functional impact.
        :param identity_source: The identity source for which authorization is requested. For a ``TOKEN`` or ``COGNITO_USER_POOLS`` authorizer, this is required and specifies the request header mapping expression for the custom header holding the authorization token submitted by the client. For example, if the token header name is ``Auth`` , the header mapping expression is ``method.request.header.Auth`` . For the ``REQUEST`` authorizer, this is required when authorization caching is enabled. The value is a comma-separated string of one or more mapping expressions of the specified request parameters. For example, if an ``Auth`` header, a ``Name`` query string parameter are defined as identity sources, this value is ``method.request.header.Auth, method.request.querystring.Name`` . These parameters will be used to derive the authorization caching key and to perform runtime validation of the ``REQUEST`` authorizer by verifying all of the identity-related request parameters are present, not null and non-empty. Only when this is true does the authorizer invoke the authorizer Lambda function, otherwise, it returns a 401 Unauthorized response without calling the Lambda function. The valid value is a string of comma-separated mapping expressions of the specified request parameters. When the authorization caching is not enabled, this property is optional.
        :param identity_validation_expression: A validation expression for the incoming identity token. For ``TOKEN`` authorizers, this value is a regular expression. For ``COGNITO_USER_POOLS`` authorizers, API Gateway will match the ``aud`` field of the incoming token from the client against the specified regular expression. It will invoke the authorizer's Lambda function when there is a match. Otherwise, it will return a 401 Unauthorized response without calling the Lambda function. The validation expression does not apply to the ``REQUEST`` authorizer.
        :param provider_arns: A list of the Amazon Cognito user pool ARNs for the ``COGNITO_USER_POOLS`` authorizer. Each element is of this format: ``arn:aws:cognito-idp:{region}:{account_id}:userpool/{user_pool_id}`` . For a ``TOKEN`` or ``REQUEST`` authorizer, this is not defined.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_authorizer_props = apigateway.CfnAuthorizerProps(
                name="name",
                rest_api_id="restApiId",
                type="type",
            
                # the properties below are optional
                authorizer_credentials="authorizerCredentials",
                authorizer_result_ttl_in_seconds=123,
                authorizer_uri="authorizerUri",
                auth_type="authType",
                identity_source="identitySource",
                identity_validation_expression="identityValidationExpression",
                provider_arns=["providerArns"]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2555e8dbc176397cb55533139d53495b8083ff110393d71c6183eeaaa36cd980)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument authorizer_credentials", value=authorizer_credentials, expected_type=type_hints["authorizer_credentials"])
            check_type(argname="argument authorizer_result_ttl_in_seconds", value=authorizer_result_ttl_in_seconds, expected_type=type_hints["authorizer_result_ttl_in_seconds"])
            check_type(argname="argument authorizer_uri", value=authorizer_uri, expected_type=type_hints["authorizer_uri"])
            check_type(argname="argument auth_type", value=auth_type, expected_type=type_hints["auth_type"])
            check_type(argname="argument identity_source", value=identity_source, expected_type=type_hints["identity_source"])
            check_type(argname="argument identity_validation_expression", value=identity_validation_expression, expected_type=type_hints["identity_validation_expression"])
            check_type(argname="argument provider_arns", value=provider_arns, expected_type=type_hints["provider_arns"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "name": name,
            "rest_api_id": rest_api_id,
            "type": type,
        }
        if authorizer_credentials is not None:
            self._values["authorizer_credentials"] = authorizer_credentials
        if authorizer_result_ttl_in_seconds is not None:
            self._values["authorizer_result_ttl_in_seconds"] = authorizer_result_ttl_in_seconds
        if authorizer_uri is not None:
            self._values["authorizer_uri"] = authorizer_uri
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if identity_source is not None:
            self._values["identity_source"] = identity_source
        if identity_validation_expression is not None:
            self._values["identity_validation_expression"] = identity_validation_expression
        if provider_arns is not None:
            self._values["provider_arns"] = provider_arns

    @builtins.property
    def name(self) -> builtins.str:
        '''The name of the authorizer.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type(self) -> builtins.str:
        '''The authorizer type.

        Valid values are ``TOKEN`` for a Lambda function using a single authorization token submitted in a custom header, ``REQUEST`` for a Lambda function using incoming request parameters, and ``COGNITO_USER_POOLS`` for using an Amazon Cognito user pool.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-type
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def authorizer_credentials(self) -> typing.Optional[builtins.str]:
        '''Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer.

        To specify an IAM role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based permissions on the Lambda function, specify null.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authorizercredentials
        '''
        result = self._values.get("authorizer_credentials")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorizer_result_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
        '''The TTL in seconds of cached authorizer results.

        If it equals 0, authorization caching is disabled. If it is greater than 0, API Gateway will cache authorizer responses. If this field is not set, the default value is 300. The maximum value is 3600, or 1 hour.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authorizerresultttlinseconds
        '''
        result = self._values.get("authorizer_result_ttl_in_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def authorizer_uri(self) -> typing.Optional[builtins.str]:
        '''Specifies the authorizer's Uniform Resource Identifier (URI).

        For ``TOKEN`` or ``REQUEST`` authorizers, this must be a well-formed Lambda function URI, for example, ``arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda:us-west-2:{account_id}:function:{lambda_function_name}/invocations`` . In general, the URI has this form ``arn:aws:apigateway:{region}:lambda:path/{service_api}`` , where ``{region}`` is the same as the region hosting the Lambda function, ``path`` indicates that the remaining substring in the URI should be treated as the path to the resource, including the initial ``/`` . For Lambda functions, this is usually of the form ``/2015-03-31/functions/[FunctionARN]/invocations`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authorizeruri
        '''
        result = self._values.get("authorizer_uri")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[builtins.str]:
        '''Optional customer-defined field, used in OpenAPI imports and exports without functional impact.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-authtype
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def identity_source(self) -> typing.Optional[builtins.str]:
        '''The identity source for which authorization is requested.

        For a ``TOKEN`` or ``COGNITO_USER_POOLS`` authorizer, this is required and specifies the request header mapping expression for the custom header holding the authorization token submitted by the client. For example, if the token header name is ``Auth`` , the header mapping expression is ``method.request.header.Auth`` . For the ``REQUEST`` authorizer, this is required when authorization caching is enabled. The value is a comma-separated string of one or more mapping expressions of the specified request parameters. For example, if an ``Auth`` header, a ``Name`` query string parameter are defined as identity sources, this value is ``method.request.header.Auth, method.request.querystring.Name`` . These parameters will be used to derive the authorization caching key and to perform runtime validation of the ``REQUEST`` authorizer by verifying all of the identity-related request parameters are present, not null and non-empty. Only when this is true does the authorizer invoke the authorizer Lambda function, otherwise, it returns a 401 Unauthorized response without calling the Lambda function. The valid value is a string of comma-separated mapping expressions of the specified request parameters. When the authorization caching is not enabled, this property is optional.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-identitysource
        '''
        result = self._values.get("identity_source")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def identity_validation_expression(self) -> typing.Optional[builtins.str]:
        '''A validation expression for the incoming identity token.

        For ``TOKEN`` authorizers, this value is a regular expression. For ``COGNITO_USER_POOLS`` authorizers, API Gateway will match the ``aud`` field of the incoming token from the client against the specified regular expression. It will invoke the authorizer's Lambda function when there is a match. Otherwise, it will return a 401 Unauthorized response without calling the Lambda function. The validation expression does not apply to the ``REQUEST`` authorizer.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-identityvalidationexpression
        '''
        result = self._values.get("identity_validation_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def provider_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of the Amazon Cognito user pool ARNs for the ``COGNITO_USER_POOLS`` authorizer.

        Each element is of this format: ``arn:aws:cognito-idp:{region}:{account_id}:userpool/{user_pool_id}`` . For a ``TOKEN`` or ``REQUEST`` authorizer, this is not defined.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-authorizer.html#cfn-apigateway-authorizer-providerarns
        '''
        result = self._values.get("provider_arns")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnAuthorizerProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnBasePathMapping(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnBasePathMapping",
):
    '''The ``AWS::ApiGateway::BasePathMapping`` resource creates a base path that clients who call your API must use in the invocation URL.

    Supported only for public custom domain names.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html
    :cloudformationResource: AWS::ApiGateway::BasePathMapping
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_base_path_mapping = apigateway.CfnBasePathMapping(self, "MyCfnBasePathMapping",
            domain_name="domainName",
        
            # the properties below are optional
            base_path="basePath",
            id="id",
            rest_api_id="restApiId",
            stage="stage"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id_: builtins.str,
        *,
        domain_name: builtins.str,
        base_path: typing.Optional[builtins.str] = None,
        id: typing.Optional[builtins.str] = None,
        rest_api_id: typing.Optional[builtins.str] = None,
        stage: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id_: Construct identifier for this resource (unique in its scope).
        :param domain_name: The domain name of the BasePathMapping resource to be described.
        :param base_path: The base path name that callers of the API must provide as part of the URL after the domain name.
        :param id: 
        :param rest_api_id: The string identifier of the associated RestApi.
        :param stage: The name of the associated stage.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30b246430425ce76f3aeef510a71365b6008738bd69977697f2b06291ac51829)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id_", value=id_, expected_type=type_hints["id_"])
        props = CfnBasePathMappingProps(
            domain_name=domain_name,
            base_path=base_path,
            id=id,
            rest_api_id=rest_api_id,
            stage=stage,
        )

        jsii.create(self.__class__, self, [scope, id_, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ed65ed5cb0c4b5ff92177004ec7bd16a83f69be6ef715c444227d2f5083a00ab)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f51401a6d5ecf5d584616a3cfd652f6583e2d13bbc7ee2a8a8dc6e52ad82525e)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> builtins.str:
        '''The domain name of the BasePathMapping resource to be described.'''
        return typing.cast(builtins.str, jsii.get(self, "domainName"))

    @domain_name.setter
    def domain_name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e7e97ee59df27b7c4014bfc651314658c5a426f6640574408fad9786518d8ef8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="basePath")
    def base_path(self) -> typing.Optional[builtins.str]:
        '''The base path name that callers of the API must provide as part of the URL after the domain name.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "basePath"))

    @base_path.setter
    def base_path(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1961a11d250bb544ee74af2ae54dd502460648c5ff3149865a1c7289e6760c4d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "basePath", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="id")
    def id(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "id"))

    @id.setter
    def id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2bf48d4737e3fa7aaa416e06b0362a3cf2cba8350d4b0d2168d2731544ac2633)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "id", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> typing.Optional[builtins.str]:
        '''The string identifier of the associated RestApi.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5ca0fe7c8b43196039340c38e6ad0262edfeb9cc67ad28b55c07c00c2f4e1ed1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="stage")
    def stage(self) -> typing.Optional[builtins.str]:
        '''The name of the associated stage.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stage"))

    @stage.setter
    def stage(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__31952d98177b4dbebd3c6595c11d9157f2d9915ba5267559110e864828aa0ac7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stage", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnBasePathMappingProps",
    jsii_struct_bases=[],
    name_mapping={
        "domain_name": "domainName",
        "base_path": "basePath",
        "id": "id",
        "rest_api_id": "restApiId",
        "stage": "stage",
    },
)
class CfnBasePathMappingProps:
    def __init__(
        self,
        *,
        domain_name: builtins.str,
        base_path: typing.Optional[builtins.str] = None,
        id: typing.Optional[builtins.str] = None,
        rest_api_id: typing.Optional[builtins.str] = None,
        stage: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnBasePathMapping``.

        :param domain_name: The domain name of the BasePathMapping resource to be described.
        :param base_path: The base path name that callers of the API must provide as part of the URL after the domain name.
        :param id: 
        :param rest_api_id: The string identifier of the associated RestApi.
        :param stage: The name of the associated stage.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_base_path_mapping_props = apigateway.CfnBasePathMappingProps(
                domain_name="domainName",
            
                # the properties below are optional
                base_path="basePath",
                id="id",
                rest_api_id="restApiId",
                stage="stage"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7d1bf02dcb15bc3e234f159c18646cccfa570e5c0f1d814a1362edb58e57ca24)
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument base_path", value=base_path, expected_type=type_hints["base_path"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "domain_name": domain_name,
        }
        if base_path is not None:
            self._values["base_path"] = base_path
        if id is not None:
            self._values["id"] = id
        if rest_api_id is not None:
            self._values["rest_api_id"] = rest_api_id
        if stage is not None:
            self._values["stage"] = stage

    @builtins.property
    def domain_name(self) -> builtins.str:
        '''The domain name of the BasePathMapping resource to be described.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-domainname
        '''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def base_path(self) -> typing.Optional[builtins.str]:
        '''The base path name that callers of the API must provide as part of the URL after the domain name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-basepath
        '''
        result = self._values.get("base_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def id(self) -> typing.Optional[builtins.str]:
        '''
        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-id
        '''
        result = self._values.get("id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def rest_api_id(self) -> typing.Optional[builtins.str]:
        '''The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-restapiid
        '''
        result = self._values.get("rest_api_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage(self) -> typing.Optional[builtins.str]:
        '''The name of the associated stage.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmapping.html#cfn-apigateway-basepathmapping-stage
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnBasePathMappingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnBasePathMappingV2(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnBasePathMappingV2",
):
    '''The ``AWS::ApiGateway::BasePathMappingV2`` resource creates a base path that clients who call your API must use in the invocation URL.

    Supported only for private custom domain names.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmappingv2.html
    :cloudformationResource: AWS::ApiGateway::BasePathMappingV2
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_base_path_mapping_v2 = apigateway.CfnBasePathMappingV2(self, "MyCfnBasePathMappingV2",
            domain_name_arn="domainNameArn",
            rest_api_id="restApiId",
        
            # the properties below are optional
            base_path="basePath",
            stage="stage"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        domain_name_arn: builtins.str,
        rest_api_id: builtins.str,
        base_path: typing.Optional[builtins.str] = None,
        stage: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param domain_name_arn: The ARN of the domain name for the BasePathMappingV2 resource to be described.
        :param rest_api_id: The private API's identifier. This identifier is unique across all of your APIs in API Gateway.
        :param base_path: The base path name that callers of the private API must provide as part of the URL after the domain name.
        :param stage: Represents a unique identifier for a version of a deployed private RestApi that is callable by users. The Stage must depend on the ``RestApi`` 's stage. To create a dependency, add a DependsOn attribute to the BasePathMappingV2 resource.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a583f31dead36a23816022dfcb476b0ff18ec097d668a0cf42b19791b749acae)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnBasePathMappingV2Props(
            domain_name_arn=domain_name_arn,
            rest_api_id=rest_api_id,
            base_path=base_path,
            stage=stage,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6eb0641d050becd18a8a9f284493b10283d6aced001cf16285ba0926b9a2ea21)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ec5be7f34d7d1faa1a122900429e9be29d8ba4ea05ed4525cd221ea65131405b)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrBasePathMappingArn")
    def attr_base_path_mapping_arn(self) -> builtins.str:
        '''Amazon Resource Name (ARN) of the resource.

        :cloudformationAttribute: BasePathMappingArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrBasePathMappingArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="domainNameArn")
    def domain_name_arn(self) -> builtins.str:
        '''The ARN of the domain name for the BasePathMappingV2 resource to be described.'''
        return typing.cast(builtins.str, jsii.get(self, "domainNameArn"))

    @domain_name_arn.setter
    def domain_name_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__23fd4874b172b953d177d1eced6c053907a3207a7d931ea7cebfd59673893321)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainNameArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The private API's identifier.'''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9e920388746332df91cb086b28eb2938d57106b8f772017211c3b2f52cd707b6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="basePath")
    def base_path(self) -> typing.Optional[builtins.str]:
        '''The base path name that callers of the private API must provide as part of the URL after the domain name.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "basePath"))

    @base_path.setter
    def base_path(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8b3750f1121b177684129c6fc603e6fc5f80e5a7cfbd3ef4c6b8695fcc42c1d6)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "basePath", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="stage")
    def stage(self) -> typing.Optional[builtins.str]:
        '''Represents a unique identifier for a version of a deployed private RestApi that is callable by users.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stage"))

    @stage.setter
    def stage(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__06763dc8a118854a928d389386d0a3d513848ff9eaaef5bd4ca70a6d09d586e0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stage", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnBasePathMappingV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "domain_name_arn": "domainNameArn",
        "rest_api_id": "restApiId",
        "base_path": "basePath",
        "stage": "stage",
    },
)
class CfnBasePathMappingV2Props:
    def __init__(
        self,
        *,
        domain_name_arn: builtins.str,
        rest_api_id: builtins.str,
        base_path: typing.Optional[builtins.str] = None,
        stage: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnBasePathMappingV2``.

        :param domain_name_arn: The ARN of the domain name for the BasePathMappingV2 resource to be described.
        :param rest_api_id: The private API's identifier. This identifier is unique across all of your APIs in API Gateway.
        :param base_path: The base path name that callers of the private API must provide as part of the URL after the domain name.
        :param stage: Represents a unique identifier for a version of a deployed private RestApi that is callable by users. The Stage must depend on the ``RestApi`` 's stage. To create a dependency, add a DependsOn attribute to the BasePathMappingV2 resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmappingv2.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_base_path_mapping_v2_props = apigateway.CfnBasePathMappingV2Props(
                domain_name_arn="domainNameArn",
                rest_api_id="restApiId",
            
                # the properties below are optional
                base_path="basePath",
                stage="stage"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5f193676fc187703a2da6f9b8342bed4af93ab14e1c8f01d93fb4249b4e41ba3)
            check_type(argname="argument domain_name_arn", value=domain_name_arn, expected_type=type_hints["domain_name_arn"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument base_path", value=base_path, expected_type=type_hints["base_path"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "domain_name_arn": domain_name_arn,
            "rest_api_id": rest_api_id,
        }
        if base_path is not None:
            self._values["base_path"] = base_path
        if stage is not None:
            self._values["stage"] = stage

    @builtins.property
    def domain_name_arn(self) -> builtins.str:
        '''The ARN of the domain name for the BasePathMappingV2 resource to be described.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmappingv2.html#cfn-apigateway-basepathmappingv2-domainnamearn
        '''
        result = self._values.get("domain_name_arn")
        assert result is not None, "Required property 'domain_name_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The private API's identifier.

        This identifier is unique across all of your APIs in API Gateway.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmappingv2.html#cfn-apigateway-basepathmappingv2-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def base_path(self) -> typing.Optional[builtins.str]:
        '''The base path name that callers of the private API must provide as part of the URL after the domain name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmappingv2.html#cfn-apigateway-basepathmappingv2-basepath
        '''
        result = self._values.get("base_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage(self) -> typing.Optional[builtins.str]:
        '''Represents a unique identifier for a version of a deployed private RestApi that is callable by users.

        The Stage must depend on the ``RestApi`` 's stage. To create a dependency, add a DependsOn attribute to the BasePathMappingV2 resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-basepathmappingv2.html#cfn-apigateway-basepathmappingv2-stage
        '''
        result = self._values.get("stage")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnBasePathMappingV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnClientCertificate(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnClientCertificate",
):
    '''The ``AWS::ApiGateway::ClientCertificate`` resource creates a client certificate that API Gateway uses to configure client-side SSL authentication for sending requests to the integration endpoint.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-clientcertificate.html
    :cloudformationResource: AWS::ApiGateway::ClientCertificate
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_client_certificate = apigateway.CfnClientCertificate(self, "MyCfnClientCertificate",
            description="description",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param description: The description of the client certificate.
        :param tags: The collection of tags. Each tag element is associated with a given resource.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b04e1c6e6561781583a05d34684fa7f16c7a0f9593b5bdadf17a39aa7adc53ac)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnClientCertificateProps(description=description, tags=tags)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c56b005c9ce07b196b1f4cb1c80e16999eacd5dbdfac93d2f4f64138b0808a7e)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__172560a34b874dc752ef52b17e7d67bf47b5d2f01b1ae02a77724009790fe3d3)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrClientCertificateId")
    def attr_client_certificate_id(self) -> builtins.str:
        '''The ID for the client certificate.

        For example: ``abc123`` .

        :cloudformationAttribute: ClientCertificateId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrClientCertificateId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the client certificate.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3d010e5f355d8f647cf92aa097a07fc016795c0b4b5415e4c6caaa06c7c01192)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__51a7fa7b78e089f078490333228634a14677f57b2a113b895fc04982489e409b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnClientCertificateProps",
    jsii_struct_bases=[],
    name_mapping={"description": "description", "tags": "tags"},
)
class CfnClientCertificateProps:
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnClientCertificate``.

        :param description: The description of the client certificate.
        :param tags: The collection of tags. Each tag element is associated with a given resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-clientcertificate.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_client_certificate_props = apigateway.CfnClientCertificateProps(
                description="description",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__53659ccccb731cc16da540b3bfeadecae2c59870e0e011d7efca126ba0cb38a9)
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the client certificate.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-clientcertificate.html#cfn-apigateway-clientcertificate-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.

        Each tag element is associated with a given resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-clientcertificate.html#cfn-apigateway-clientcertificate-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientCertificateProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnDeployment(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDeployment",
):
    '''The ``AWS::ApiGateway::Deployment`` resource deploys an API Gateway ``RestApi`` resource to a stage so that clients can call the API over the internet.

    The stage acts as an environment.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
    :cloudformationResource: AWS::ApiGateway::Deployment
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_deployment = apigateway.CfnDeployment(self, "MyCfnDeployment",
            rest_api_id="restApiId",
        
            # the properties below are optional
            deployment_canary_settings=apigateway.CfnDeployment.DeploymentCanarySettingsProperty(
                percent_traffic=123,
                stage_variable_overrides={
                    "stage_variable_overrides_key": "stageVariableOverrides"
                },
                use_stage_cache=False
            ),
            description="description",
            stage_description=apigateway.CfnDeployment.StageDescriptionProperty(
                access_log_setting=apigateway.CfnDeployment.AccessLogSettingProperty(
                    destination_arn="destinationArn",
                    format="format"
                ),
                cache_cluster_enabled=False,
                cache_cluster_size="cacheClusterSize",
                cache_data_encrypted=False,
                cache_ttl_in_seconds=123,
                caching_enabled=False,
                canary_setting=apigateway.CfnDeployment.CanarySettingProperty(
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                ),
                client_certificate_id="clientCertificateId",
                data_trace_enabled=False,
                description="description",
                documentation_version="documentationVersion",
                logging_level="loggingLevel",
                method_settings=[apigateway.CfnDeployment.MethodSettingProperty(
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    data_trace_enabled=False,
                    http_method="httpMethod",
                    logging_level="loggingLevel",
                    metrics_enabled=False,
                    resource_path="resourcePath",
                    throttling_burst_limit=123,
                    throttling_rate_limit=123
                )],
                metrics_enabled=False,
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                throttling_burst_limit=123,
                throttling_rate_limit=123,
                tracing_enabled=False,
                variables={
                    "variables_key": "variables"
                }
            ),
            stage_name="stageName"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        rest_api_id: builtins.str,
        deployment_canary_settings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDeployment.DeploymentCanarySettingsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        stage_description: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDeployment.StageDescriptionProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param rest_api_id: The string identifier of the associated RestApi.
        :param deployment_canary_settings: The input configuration for a canary deployment.
        :param description: The description for the Deployment resource to create.
        :param stage_description: The description of the Stage resource for the Deployment resource to create. To specify a stage description, you must also provide a stage name.
        :param stage_name: The name of the Stage resource for the Deployment resource to create.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b0d6a79724121e56203915fc9f9807d25843c59847cf40c400a4686fab2934d1)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDeploymentProps(
            rest_api_id=rest_api_id,
            deployment_canary_settings=deployment_canary_settings,
            description=description,
            stage_description=stage_description,
            stage_name=stage_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d129e4d1c603fe7c93378a40cdff7de4da36aff1757a2d969a9ccb4c3c4211dd)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__95b4ce7c647727f023e2c27c49cc5e76cc99abcc82f6d753d3036a66b9db9fe9)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDeploymentId")
    def attr_deployment_id(self) -> builtins.str:
        '''The ID for the deployment.

        For example: ``abc123`` .

        :cloudformationAttribute: DeploymentId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDeploymentId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.'''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fd1bfac0862cb501faa5a4bda139dbba65e6786284c3504d72a2b828fd96e728)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="deploymentCanarySettings")
    def deployment_canary_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDeployment.DeploymentCanarySettingsProperty"]]:
        '''The input configuration for a canary deployment.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDeployment.DeploymentCanarySettingsProperty"]], jsii.get(self, "deploymentCanarySettings"))

    @deployment_canary_settings.setter
    def deployment_canary_settings(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDeployment.DeploymentCanarySettingsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__05dc0fbe57a1a262a9a6339c53b1f2ffdf9620fc527d11d71935e87b39d713cf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deploymentCanarySettings", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description for the Deployment resource to create.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__338ed9ecdf7a0fa86b2d2c4a475b027d6ec6fbc9c91070926de42f9c4e1484b4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="stageDescription")
    def stage_description(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDeployment.StageDescriptionProperty"]]:
        '''The description of the Stage resource for the Deployment resource to create.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDeployment.StageDescriptionProperty"]], jsii.get(self, "stageDescription"))

    @stage_description.setter
    def stage_description(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDeployment.StageDescriptionProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6024f7e0d42bb4e5be9c3f4d821e4187c174774ed464c46036ecb95e43bf0fe7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stageDescription", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Stage resource for the Deployment resource to create.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))

    @stage_name.setter
    def stage_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dc44433d7274089bdd03fea84500e9f0beb1eeaff9bfe635e04e176488a4b977)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stageName", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnDeployment.AccessLogSettingProperty",
        jsii_struct_bases=[],
        name_mapping={"destination_arn": "destinationArn", "format": "format"},
    )
    class AccessLogSettingProperty:
        def __init__(
            self,
            *,
            destination_arn: typing.Optional[builtins.str] = None,
            format: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``AccessLogSetting`` property type specifies settings for logging access in this stage.

            ``AccessLogSetting`` is a property of the `StageDescription <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html>`_ property type.

            :param destination_arn: The Amazon Resource Name (ARN) of the CloudWatch Logs log group or Kinesis Data Firehose delivery stream to receive access logs. If you specify a Kinesis Data Firehose delivery stream, the stream name must begin with ``amazon-apigateway-`` .
            :param format: A single line format of the access logs of data, as specified by selected $context variables. The format must include at least ``$context.requestId`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-accesslogsetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                access_log_setting_property = apigateway.CfnDeployment.AccessLogSettingProperty(
                    destination_arn="destinationArn",
                    format="format"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dc164dc3c9aeb5becbf827ecbcb26f0bb6d4c4a52cd6194874d3d3f300ae7a15)
                check_type(argname="argument destination_arn", value=destination_arn, expected_type=type_hints["destination_arn"])
                check_type(argname="argument format", value=format, expected_type=type_hints["format"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if destination_arn is not None:
                self._values["destination_arn"] = destination_arn
            if format is not None:
                self._values["format"] = format

        @builtins.property
        def destination_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the CloudWatch Logs log group or Kinesis Data Firehose delivery stream to receive access logs.

            If you specify a Kinesis Data Firehose delivery stream, the stream name must begin with ``amazon-apigateway-`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-accesslogsetting.html#cfn-apigateway-deployment-accesslogsetting-destinationarn
            '''
            result = self._values.get("destination_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def format(self) -> typing.Optional[builtins.str]:
            '''A single line format of the access logs of data, as specified by selected $context variables.

            The format must include at least ``$context.requestId`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-accesslogsetting.html#cfn-apigateway-deployment-accesslogsetting-format
            '''
            result = self._values.get("format")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessLogSettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnDeployment.CanarySettingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "percent_traffic": "percentTraffic",
            "stage_variable_overrides": "stageVariableOverrides",
            "use_stage_cache": "useStageCache",
        },
    )
    class CanarySettingProperty:
        def __init__(
            self,
            *,
            percent_traffic: typing.Optional[jsii.Number] = None,
            stage_variable_overrides: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
            use_stage_cache: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''The ``CanarySetting`` property type specifies settings for the canary deployment in this stage.

            ``CanarySetting`` is a property of the `StageDescription <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html>`_ property type.

            :param percent_traffic: The percent (0-100) of traffic diverted to a canary deployment.
            :param stage_variable_overrides: Stage variables overridden for a canary release deployment, including new stage variables introduced in the canary. These stage variables are represented as a string-to-string map between stage variable names and their values.
            :param use_stage_cache: A Boolean flag to indicate whether the canary deployment uses the stage cache or not.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-canarysetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                canary_setting_property = apigateway.CfnDeployment.CanarySettingProperty(
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__59d5fc1d5b68620d8cd9a10331f1f28a8cb2bdfd48237c45deba8a3667c15eac)
                check_type(argname="argument percent_traffic", value=percent_traffic, expected_type=type_hints["percent_traffic"])
                check_type(argname="argument stage_variable_overrides", value=stage_variable_overrides, expected_type=type_hints["stage_variable_overrides"])
                check_type(argname="argument use_stage_cache", value=use_stage_cache, expected_type=type_hints["use_stage_cache"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if percent_traffic is not None:
                self._values["percent_traffic"] = percent_traffic
            if stage_variable_overrides is not None:
                self._values["stage_variable_overrides"] = stage_variable_overrides
            if use_stage_cache is not None:
                self._values["use_stage_cache"] = use_stage_cache

        @builtins.property
        def percent_traffic(self) -> typing.Optional[jsii.Number]:
            '''The percent (0-100) of traffic diverted to a canary deployment.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-canarysetting.html#cfn-apigateway-deployment-canarysetting-percenttraffic
            '''
            result = self._values.get("percent_traffic")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def stage_variable_overrides(
            self,
        ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
            '''Stage variables overridden for a canary release deployment, including new stage variables introduced in the canary.

            These stage variables are represented as a string-to-string map between stage variable names and their values.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-canarysetting.html#cfn-apigateway-deployment-canarysetting-stagevariableoverrides
            '''
            result = self._values.get("stage_variable_overrides")
            return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

        @builtins.property
        def use_stage_cache(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''A Boolean flag to indicate whether the canary deployment uses the stage cache or not.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-canarysetting.html#cfn-apigateway-deployment-canarysetting-usestagecache
            '''
            result = self._values.get("use_stage_cache")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CanarySettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnDeployment.DeploymentCanarySettingsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "percent_traffic": "percentTraffic",
            "stage_variable_overrides": "stageVariableOverrides",
            "use_stage_cache": "useStageCache",
        },
    )
    class DeploymentCanarySettingsProperty:
        def __init__(
            self,
            *,
            percent_traffic: typing.Optional[jsii.Number] = None,
            stage_variable_overrides: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
            use_stage_cache: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''The ``DeploymentCanarySettings`` property type specifies settings for the canary deployment.

            :param percent_traffic: The percentage (0.0-100.0) of traffic routed to the canary deployment.
            :param stage_variable_overrides: A stage variable overrides used for the canary release deployment. They can override existing stage variables or add new stage variables for the canary release deployment. These stage variables are represented as a string-to-string map between stage variable names and their values.
            :param use_stage_cache: A Boolean flag to indicate whether the canary release deployment uses the stage cache or not.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-deploymentcanarysettings.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                deployment_canary_settings_property = apigateway.CfnDeployment.DeploymentCanarySettingsProperty(
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__e95de67e9fd472bd51513ee53f90119b4b9ec81399817de038948ff55beeb670)
                check_type(argname="argument percent_traffic", value=percent_traffic, expected_type=type_hints["percent_traffic"])
                check_type(argname="argument stage_variable_overrides", value=stage_variable_overrides, expected_type=type_hints["stage_variable_overrides"])
                check_type(argname="argument use_stage_cache", value=use_stage_cache, expected_type=type_hints["use_stage_cache"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if percent_traffic is not None:
                self._values["percent_traffic"] = percent_traffic
            if stage_variable_overrides is not None:
                self._values["stage_variable_overrides"] = stage_variable_overrides
            if use_stage_cache is not None:
                self._values["use_stage_cache"] = use_stage_cache

        @builtins.property
        def percent_traffic(self) -> typing.Optional[jsii.Number]:
            '''The percentage (0.0-100.0) of traffic routed to the canary deployment.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-deploymentcanarysettings.html#cfn-apigateway-deployment-deploymentcanarysettings-percenttraffic
            '''
            result = self._values.get("percent_traffic")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def stage_variable_overrides(
            self,
        ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
            '''A stage variable overrides used for the canary release deployment.

            They can override existing stage variables or add new stage variables for the canary release deployment. These stage variables are represented as a string-to-string map between stage variable names and their values.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-deploymentcanarysettings.html#cfn-apigateway-deployment-deploymentcanarysettings-stagevariableoverrides
            '''
            result = self._values.get("stage_variable_overrides")
            return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

        @builtins.property
        def use_stage_cache(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''A Boolean flag to indicate whether the canary release deployment uses the stage cache or not.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-deploymentcanarysettings.html#cfn-apigateway-deployment-deploymentcanarysettings-usestagecache
            '''
            result = self._values.get("use_stage_cache")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentCanarySettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnDeployment.MethodSettingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cache_data_encrypted": "cacheDataEncrypted",
            "cache_ttl_in_seconds": "cacheTtlInSeconds",
            "caching_enabled": "cachingEnabled",
            "data_trace_enabled": "dataTraceEnabled",
            "http_method": "httpMethod",
            "logging_level": "loggingLevel",
            "metrics_enabled": "metricsEnabled",
            "resource_path": "resourcePath",
            "throttling_burst_limit": "throttlingBurstLimit",
            "throttling_rate_limit": "throttlingRateLimit",
        },
    )
    class MethodSettingProperty:
        def __init__(
            self,
            *,
            cache_data_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            cache_ttl_in_seconds: typing.Optional[jsii.Number] = None,
            caching_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            data_trace_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            http_method: typing.Optional[builtins.str] = None,
            logging_level: typing.Optional[builtins.str] = None,
            metrics_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            resource_path: typing.Optional[builtins.str] = None,
            throttling_burst_limit: typing.Optional[jsii.Number] = None,
            throttling_rate_limit: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``MethodSetting`` property type configures settings for all methods in a stage.

            The ``MethodSettings`` property of the `Amazon API Gateway Deployment StageDescription <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html>`_ property type contains a list of ``MethodSetting`` property types.

            :param cache_data_encrypted: Specifies whether the cached responses are encrypted.
            :param cache_ttl_in_seconds: Specifies the time to live (TTL), in seconds, for cached responses. The higher the TTL, the longer the response will be cached.
            :param caching_enabled: Specifies whether responses should be cached and returned for requests. A cache cluster must be enabled on the stage for responses to be cached.
            :param data_trace_enabled: Specifies whether data trace logging is enabled for this method, which affects the log entries pushed to Amazon CloudWatch Logs. This can be useful to troubleshoot APIs, but can result in logging sensitive data. We recommend that you don't enable this option for production APIs.
            :param http_method: The HTTP method.
            :param logging_level: Specifies the logging level for this method, which affects the log entries pushed to Amazon CloudWatch Logs. Valid values are ``OFF`` , ``ERROR`` , and ``INFO`` . Choose ``ERROR`` to write only error-level entries to CloudWatch Logs, or choose ``INFO`` to include all ``ERROR`` events as well as extra informational events.
            :param metrics_enabled: Specifies whether Amazon CloudWatch metrics are enabled for this method.
            :param resource_path: The resource path for this method. Forward slashes ( ``/`` ) are encoded as ``~1`` and the initial slash must include a forward slash. For example, the path value ``/resource/subresource`` must be encoded as ``/~1resource~1subresource`` . To specify the root path, use only a slash ( ``/`` ).
            :param throttling_burst_limit: Specifies the throttling burst limit.
            :param throttling_rate_limit: Specifies the throttling rate limit.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                method_setting_property = apigateway.CfnDeployment.MethodSettingProperty(
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    data_trace_enabled=False,
                    http_method="httpMethod",
                    logging_level="loggingLevel",
                    metrics_enabled=False,
                    resource_path="resourcePath",
                    throttling_burst_limit=123,
                    throttling_rate_limit=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__44275fec3e478339800df7870d6b7c3ece2c32b67ab5fa97e55c9f5c19050d17)
                check_type(argname="argument cache_data_encrypted", value=cache_data_encrypted, expected_type=type_hints["cache_data_encrypted"])
                check_type(argname="argument cache_ttl_in_seconds", value=cache_ttl_in_seconds, expected_type=type_hints["cache_ttl_in_seconds"])
                check_type(argname="argument caching_enabled", value=caching_enabled, expected_type=type_hints["caching_enabled"])
                check_type(argname="argument data_trace_enabled", value=data_trace_enabled, expected_type=type_hints["data_trace_enabled"])
                check_type(argname="argument http_method", value=http_method, expected_type=type_hints["http_method"])
                check_type(argname="argument logging_level", value=logging_level, expected_type=type_hints["logging_level"])
                check_type(argname="argument metrics_enabled", value=metrics_enabled, expected_type=type_hints["metrics_enabled"])
                check_type(argname="argument resource_path", value=resource_path, expected_type=type_hints["resource_path"])
                check_type(argname="argument throttling_burst_limit", value=throttling_burst_limit, expected_type=type_hints["throttling_burst_limit"])
                check_type(argname="argument throttling_rate_limit", value=throttling_rate_limit, expected_type=type_hints["throttling_rate_limit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cache_data_encrypted is not None:
                self._values["cache_data_encrypted"] = cache_data_encrypted
            if cache_ttl_in_seconds is not None:
                self._values["cache_ttl_in_seconds"] = cache_ttl_in_seconds
            if caching_enabled is not None:
                self._values["caching_enabled"] = caching_enabled
            if data_trace_enabled is not None:
                self._values["data_trace_enabled"] = data_trace_enabled
            if http_method is not None:
                self._values["http_method"] = http_method
            if logging_level is not None:
                self._values["logging_level"] = logging_level
            if metrics_enabled is not None:
                self._values["metrics_enabled"] = metrics_enabled
            if resource_path is not None:
                self._values["resource_path"] = resource_path
            if throttling_burst_limit is not None:
                self._values["throttling_burst_limit"] = throttling_burst_limit
            if throttling_rate_limit is not None:
                self._values["throttling_rate_limit"] = throttling_rate_limit

        @builtins.property
        def cache_data_encrypted(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether the cached responses are encrypted.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-cachedataencrypted
            '''
            result = self._values.get("cache_data_encrypted")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def cache_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
            '''Specifies the time to live (TTL), in seconds, for cached responses.

            The higher the TTL, the longer the response will be cached.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-cachettlinseconds
            '''
            result = self._values.get("cache_ttl_in_seconds")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def caching_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether responses should be cached and returned for requests.

            A cache cluster must be enabled on the stage for responses to be cached.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-cachingenabled
            '''
            result = self._values.get("caching_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def data_trace_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether data trace logging is enabled for this method, which affects the log entries pushed to Amazon CloudWatch Logs.

            This can be useful to troubleshoot APIs, but can result in logging sensitive data. We recommend that you don't enable this option for production APIs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-datatraceenabled
            '''
            result = self._values.get("data_trace_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def http_method(self) -> typing.Optional[builtins.str]:
            '''The HTTP method.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-httpmethod
            '''
            result = self._values.get("http_method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def logging_level(self) -> typing.Optional[builtins.str]:
            '''Specifies the logging level for this method, which affects the log entries pushed to Amazon CloudWatch Logs.

            Valid values are ``OFF`` , ``ERROR`` , and ``INFO`` . Choose ``ERROR`` to write only error-level entries to CloudWatch Logs, or choose ``INFO`` to include all ``ERROR`` events as well as extra informational events.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-logginglevel
            '''
            result = self._values.get("logging_level")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def metrics_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether Amazon CloudWatch metrics are enabled for this method.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-metricsenabled
            '''
            result = self._values.get("metrics_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def resource_path(self) -> typing.Optional[builtins.str]:
            '''The resource path for this method.

            Forward slashes ( ``/`` ) are encoded as ``~1`` and the initial slash must include a forward slash. For example, the path value ``/resource/subresource`` must be encoded as ``/~1resource~1subresource`` . To specify the root path, use only a slash ( ``/`` ).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-resourcepath
            '''
            result = self._values.get("resource_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def throttling_burst_limit(self) -> typing.Optional[jsii.Number]:
            '''Specifies the throttling burst limit.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-throttlingburstlimit
            '''
            result = self._values.get("throttling_burst_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def throttling_rate_limit(self) -> typing.Optional[jsii.Number]:
            '''Specifies the throttling rate limit.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-methodsetting.html#cfn-apigateway-deployment-methodsetting-throttlingratelimit
            '''
            result = self._values.get("throttling_rate_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MethodSettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnDeployment.StageDescriptionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "access_log_setting": "accessLogSetting",
            "cache_cluster_enabled": "cacheClusterEnabled",
            "cache_cluster_size": "cacheClusterSize",
            "cache_data_encrypted": "cacheDataEncrypted",
            "cache_ttl_in_seconds": "cacheTtlInSeconds",
            "caching_enabled": "cachingEnabled",
            "canary_setting": "canarySetting",
            "client_certificate_id": "clientCertificateId",
            "data_trace_enabled": "dataTraceEnabled",
            "description": "description",
            "documentation_version": "documentationVersion",
            "logging_level": "loggingLevel",
            "method_settings": "methodSettings",
            "metrics_enabled": "metricsEnabled",
            "tags": "tags",
            "throttling_burst_limit": "throttlingBurstLimit",
            "throttling_rate_limit": "throttlingRateLimit",
            "tracing_enabled": "tracingEnabled",
            "variables": "variables",
        },
    )
    class StageDescriptionProperty:
        def __init__(
            self,
            *,
            access_log_setting: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDeployment.AccessLogSettingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            cache_cluster_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            cache_cluster_size: typing.Optional[builtins.str] = None,
            cache_data_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            cache_ttl_in_seconds: typing.Optional[jsii.Number] = None,
            caching_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            canary_setting: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDeployment.CanarySettingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            client_certificate_id: typing.Optional[builtins.str] = None,
            data_trace_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            description: typing.Optional[builtins.str] = None,
            documentation_version: typing.Optional[builtins.str] = None,
            logging_level: typing.Optional[builtins.str] = None,
            method_settings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDeployment.MethodSettingProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            metrics_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
            throttling_burst_limit: typing.Optional[jsii.Number] = None,
            throttling_rate_limit: typing.Optional[jsii.Number] = None,
            tracing_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            variables: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        ) -> None:
            '''``StageDescription`` is a property of the `AWS::ApiGateway::Deployment <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html>`_ resource that configures a deployment stage.

            :param access_log_setting: Specifies settings for logging access in this stage.
            :param cache_cluster_enabled: Specifies whether a cache cluster is enabled for the stage. To activate a method-level cache, set ``CachingEnabled`` to ``true`` for a method.
            :param cache_cluster_size: The size of the stage's cache cluster. For more information, see `cacheClusterSize <https://docs.aws.amazon.com/apigateway/latest/api/API_CreateStage.html#apigw-CreateStage-request-cacheClusterSize>`_ in the *API Gateway API Reference* .
            :param cache_data_encrypted: Indicates whether the cached responses are encrypted.
            :param cache_ttl_in_seconds: The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches responses.
            :param caching_enabled: Indicates whether responses are cached and returned for requests. You must enable a cache cluster on the stage to cache responses. For more information, see `Enable API Gateway Caching in a Stage to Enhance API Performance <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html>`_ in the *API Gateway Developer Guide* .
            :param canary_setting: Specifies settings for the canary deployment in this stage.
            :param client_certificate_id: The identifier of the client certificate that API Gateway uses to call your integration endpoints in the stage.
            :param data_trace_enabled: Indicates whether data trace logging is enabled for methods in the stage. API Gateway pushes these logs to Amazon CloudWatch Logs.
            :param description: A description of the purpose of the stage.
            :param documentation_version: The version identifier of the API documentation snapshot.
            :param logging_level: The logging level for this method. For valid values, see the ``loggingLevel`` property of the `MethodSetting <https://docs.aws.amazon.com/apigateway/latest/api/API_MethodSetting.html>`_ resource in the *Amazon API Gateway API Reference* .
            :param method_settings: Configures settings for all of the stage's methods.
            :param metrics_enabled: Indicates whether Amazon CloudWatch metrics are enabled for methods in the stage.
            :param tags: An array of arbitrary tags (key-value pairs) to associate with the stage.
            :param throttling_burst_limit: The target request burst rate limit. This allows more requests through for a period of time than the target rate limit. For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .
            :param throttling_rate_limit: The target request steady-state rate limit. For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .
            :param tracing_enabled: Specifies whether active tracing with X-ray is enabled for this stage. For more information, see `Trace API Gateway API Execution with AWS X-Ray <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-xray.html>`_ in the *API Gateway Developer Guide* .
            :param variables: A map that defines the stage variables. Variable names must consist of alphanumeric characters, and the values must match the following regular expression: ``[A-Za-z0-9-._~:/?#&=,]+`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                stage_description_property = apigateway.CfnDeployment.StageDescriptionProperty(
                    access_log_setting=apigateway.CfnDeployment.AccessLogSettingProperty(
                        destination_arn="destinationArn",
                        format="format"
                    ),
                    cache_cluster_enabled=False,
                    cache_cluster_size="cacheClusterSize",
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    canary_setting=apigateway.CfnDeployment.CanarySettingProperty(
                        percent_traffic=123,
                        stage_variable_overrides={
                            "stage_variable_overrides_key": "stageVariableOverrides"
                        },
                        use_stage_cache=False
                    ),
                    client_certificate_id="clientCertificateId",
                    data_trace_enabled=False,
                    description="description",
                    documentation_version="documentationVersion",
                    logging_level="loggingLevel",
                    method_settings=[apigateway.CfnDeployment.MethodSettingProperty(
                        cache_data_encrypted=False,
                        cache_ttl_in_seconds=123,
                        caching_enabled=False,
                        data_trace_enabled=False,
                        http_method="httpMethod",
                        logging_level="loggingLevel",
                        metrics_enabled=False,
                        resource_path="resourcePath",
                        throttling_burst_limit=123,
                        throttling_rate_limit=123
                    )],
                    metrics_enabled=False,
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )],
                    throttling_burst_limit=123,
                    throttling_rate_limit=123,
                    tracing_enabled=False,
                    variables={
                        "variables_key": "variables"
                    }
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a4b36eae831f3a5e4188fd26be727cd8d35c14538a8f206c1cb04a713d07dff5)
                check_type(argname="argument access_log_setting", value=access_log_setting, expected_type=type_hints["access_log_setting"])
                check_type(argname="argument cache_cluster_enabled", value=cache_cluster_enabled, expected_type=type_hints["cache_cluster_enabled"])
                check_type(argname="argument cache_cluster_size", value=cache_cluster_size, expected_type=type_hints["cache_cluster_size"])
                check_type(argname="argument cache_data_encrypted", value=cache_data_encrypted, expected_type=type_hints["cache_data_encrypted"])
                check_type(argname="argument cache_ttl_in_seconds", value=cache_ttl_in_seconds, expected_type=type_hints["cache_ttl_in_seconds"])
                check_type(argname="argument caching_enabled", value=caching_enabled, expected_type=type_hints["caching_enabled"])
                check_type(argname="argument canary_setting", value=canary_setting, expected_type=type_hints["canary_setting"])
                check_type(argname="argument client_certificate_id", value=client_certificate_id, expected_type=type_hints["client_certificate_id"])
                check_type(argname="argument data_trace_enabled", value=data_trace_enabled, expected_type=type_hints["data_trace_enabled"])
                check_type(argname="argument description", value=description, expected_type=type_hints["description"])
                check_type(argname="argument documentation_version", value=documentation_version, expected_type=type_hints["documentation_version"])
                check_type(argname="argument logging_level", value=logging_level, expected_type=type_hints["logging_level"])
                check_type(argname="argument method_settings", value=method_settings, expected_type=type_hints["method_settings"])
                check_type(argname="argument metrics_enabled", value=metrics_enabled, expected_type=type_hints["metrics_enabled"])
                check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
                check_type(argname="argument throttling_burst_limit", value=throttling_burst_limit, expected_type=type_hints["throttling_burst_limit"])
                check_type(argname="argument throttling_rate_limit", value=throttling_rate_limit, expected_type=type_hints["throttling_rate_limit"])
                check_type(argname="argument tracing_enabled", value=tracing_enabled, expected_type=type_hints["tracing_enabled"])
                check_type(argname="argument variables", value=variables, expected_type=type_hints["variables"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if access_log_setting is not None:
                self._values["access_log_setting"] = access_log_setting
            if cache_cluster_enabled is not None:
                self._values["cache_cluster_enabled"] = cache_cluster_enabled
            if cache_cluster_size is not None:
                self._values["cache_cluster_size"] = cache_cluster_size
            if cache_data_encrypted is not None:
                self._values["cache_data_encrypted"] = cache_data_encrypted
            if cache_ttl_in_seconds is not None:
                self._values["cache_ttl_in_seconds"] = cache_ttl_in_seconds
            if caching_enabled is not None:
                self._values["caching_enabled"] = caching_enabled
            if canary_setting is not None:
                self._values["canary_setting"] = canary_setting
            if client_certificate_id is not None:
                self._values["client_certificate_id"] = client_certificate_id
            if data_trace_enabled is not None:
                self._values["data_trace_enabled"] = data_trace_enabled
            if description is not None:
                self._values["description"] = description
            if documentation_version is not None:
                self._values["documentation_version"] = documentation_version
            if logging_level is not None:
                self._values["logging_level"] = logging_level
            if method_settings is not None:
                self._values["method_settings"] = method_settings
            if metrics_enabled is not None:
                self._values["metrics_enabled"] = metrics_enabled
            if tags is not None:
                self._values["tags"] = tags
            if throttling_burst_limit is not None:
                self._values["throttling_burst_limit"] = throttling_burst_limit
            if throttling_rate_limit is not None:
                self._values["throttling_rate_limit"] = throttling_rate_limit
            if tracing_enabled is not None:
                self._values["tracing_enabled"] = tracing_enabled
            if variables is not None:
                self._values["variables"] = variables

        @builtins.property
        def access_log_setting(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDeployment.AccessLogSettingProperty"]]:
            '''Specifies settings for logging access in this stage.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-accesslogsetting
            '''
            result = self._values.get("access_log_setting")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDeployment.AccessLogSettingProperty"]], result)

        @builtins.property
        def cache_cluster_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether a cache cluster is enabled for the stage.

            To activate a method-level cache, set ``CachingEnabled`` to ``true`` for a method.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-cacheclusterenabled
            '''
            result = self._values.get("cache_cluster_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def cache_cluster_size(self) -> typing.Optional[builtins.str]:
            '''The size of the stage's cache cluster.

            For more information, see `cacheClusterSize <https://docs.aws.amazon.com/apigateway/latest/api/API_CreateStage.html#apigw-CreateStage-request-cacheClusterSize>`_ in the *API Gateway API Reference* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-cacheclustersize
            '''
            result = self._values.get("cache_cluster_size")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cache_data_encrypted(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates whether the cached responses are encrypted.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-cachedataencrypted
            '''
            result = self._values.get("cache_data_encrypted")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def cache_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
            '''The time-to-live (TTL) period, in seconds, that specifies how long API Gateway caches responses.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-cachettlinseconds
            '''
            result = self._values.get("cache_ttl_in_seconds")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def caching_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates whether responses are cached and returned for requests.

            You must enable a cache cluster on the stage to cache responses. For more information, see `Enable API Gateway Caching in a Stage to Enhance API Performance <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html>`_ in the *API Gateway Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-cachingenabled
            '''
            result = self._values.get("caching_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def canary_setting(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDeployment.CanarySettingProperty"]]:
            '''Specifies settings for the canary deployment in this stage.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-canarysetting
            '''
            result = self._values.get("canary_setting")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDeployment.CanarySettingProperty"]], result)

        @builtins.property
        def client_certificate_id(self) -> typing.Optional[builtins.str]:
            '''The identifier of the client certificate that API Gateway uses to call your integration endpoints in the stage.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-clientcertificateid
            '''
            result = self._values.get("client_certificate_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def data_trace_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates whether data trace logging is enabled for methods in the stage.

            API Gateway pushes these logs to Amazon CloudWatch Logs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-datatraceenabled
            '''
            result = self._values.get("data_trace_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''A description of the purpose of the stage.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def documentation_version(self) -> typing.Optional[builtins.str]:
            '''The version identifier of the API documentation snapshot.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-documentationversion
            '''
            result = self._values.get("documentation_version")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def logging_level(self) -> typing.Optional[builtins.str]:
            '''The logging level for this method.

            For valid values, see the ``loggingLevel`` property of the `MethodSetting <https://docs.aws.amazon.com/apigateway/latest/api/API_MethodSetting.html>`_ resource in the *Amazon API Gateway API Reference* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-logginglevel
            '''
            result = self._values.get("logging_level")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def method_settings(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDeployment.MethodSettingProperty"]]]]:
            '''Configures settings for all of the stage's methods.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-methodsettings
            '''
            result = self._values.get("method_settings")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnDeployment.MethodSettingProperty"]]]], result)

        @builtins.property
        def metrics_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Indicates whether Amazon CloudWatch metrics are enabled for methods in the stage.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-metricsenabled
            '''
            result = self._values.get("metrics_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
            '''An array of arbitrary tags (key-value pairs) to associate with the stage.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

        @builtins.property
        def throttling_burst_limit(self) -> typing.Optional[jsii.Number]:
            '''The target request burst rate limit.

            This allows more requests through for a period of time than the target rate limit. For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-throttlingburstlimit
            '''
            result = self._values.get("throttling_burst_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def throttling_rate_limit(self) -> typing.Optional[jsii.Number]:
            '''The target request steady-state rate limit.

            For more information, see `Manage API Request Throttling <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html>`_ in the *API Gateway Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-throttlingratelimit
            '''
            result = self._values.get("throttling_rate_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def tracing_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether active tracing with X-ray is enabled for this stage.

            For more information, see `Trace API Gateway API Execution with AWS X-Ray <https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-xray.html>`_ in the *API Gateway Developer Guide* .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-tracingenabled
            '''
            result = self._values.get("tracing_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def variables(
            self,
        ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
            '''A map that defines the stage variables.

            Variable names must consist of alphanumeric characters, and the values must match the following regular expression: ``[A-Za-z0-9-._~:/?#&=,]+`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-deployment-stagedescription.html#cfn-apigateway-deployment-stagedescription-variables
            '''
            result = self._values.get("variables")
            return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StageDescriptionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDeploymentProps",
    jsii_struct_bases=[],
    name_mapping={
        "rest_api_id": "restApiId",
        "deployment_canary_settings": "deploymentCanarySettings",
        "description": "description",
        "stage_description": "stageDescription",
        "stage_name": "stageName",
    },
)
class CfnDeploymentProps:
    def __init__(
        self,
        *,
        rest_api_id: builtins.str,
        deployment_canary_settings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDeployment.DeploymentCanarySettingsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        description: typing.Optional[builtins.str] = None,
        stage_description: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDeployment.StageDescriptionProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnDeployment``.

        :param rest_api_id: The string identifier of the associated RestApi.
        :param deployment_canary_settings: The input configuration for a canary deployment.
        :param description: The description for the Deployment resource to create.
        :param stage_description: The description of the Stage resource for the Deployment resource to create. To specify a stage description, you must also provide a stage name.
        :param stage_name: The name of the Stage resource for the Deployment resource to create.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_deployment_props = apigateway.CfnDeploymentProps(
                rest_api_id="restApiId",
            
                # the properties below are optional
                deployment_canary_settings=apigateway.CfnDeployment.DeploymentCanarySettingsProperty(
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                ),
                description="description",
                stage_description=apigateway.CfnDeployment.StageDescriptionProperty(
                    access_log_setting=apigateway.CfnDeployment.AccessLogSettingProperty(
                        destination_arn="destinationArn",
                        format="format"
                    ),
                    cache_cluster_enabled=False,
                    cache_cluster_size="cacheClusterSize",
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    canary_setting=apigateway.CfnDeployment.CanarySettingProperty(
                        percent_traffic=123,
                        stage_variable_overrides={
                            "stage_variable_overrides_key": "stageVariableOverrides"
                        },
                        use_stage_cache=False
                    ),
                    client_certificate_id="clientCertificateId",
                    data_trace_enabled=False,
                    description="description",
                    documentation_version="documentationVersion",
                    logging_level="loggingLevel",
                    method_settings=[apigateway.CfnDeployment.MethodSettingProperty(
                        cache_data_encrypted=False,
                        cache_ttl_in_seconds=123,
                        caching_enabled=False,
                        data_trace_enabled=False,
                        http_method="httpMethod",
                        logging_level="loggingLevel",
                        metrics_enabled=False,
                        resource_path="resourcePath",
                        throttling_burst_limit=123,
                        throttling_rate_limit=123
                    )],
                    metrics_enabled=False,
                    tags=[CfnTag(
                        key="key",
                        value="value"
                    )],
                    throttling_burst_limit=123,
                    throttling_rate_limit=123,
                    tracing_enabled=False,
                    variables={
                        "variables_key": "variables"
                    }
                ),
                stage_name="stageName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__06f6115d0d6ce383d655d7c6244956ed477bdc61ad553bd50def28044860463f)
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument deployment_canary_settings", value=deployment_canary_settings, expected_type=type_hints["deployment_canary_settings"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument stage_description", value=stage_description, expected_type=type_hints["stage_description"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "rest_api_id": rest_api_id,
        }
        if deployment_canary_settings is not None:
            self._values["deployment_canary_settings"] = deployment_canary_settings
        if description is not None:
            self._values["description"] = description
        if stage_description is not None:
            self._values["stage_description"] = stage_description
        if stage_name is not None:
            self._values["stage_name"] = stage_name

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def deployment_canary_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDeployment.DeploymentCanarySettingsProperty]]:
        '''The input configuration for a canary deployment.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-deploymentcanarysettings
        '''
        result = self._values.get("deployment_canary_settings")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDeployment.DeploymentCanarySettingsProperty]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description for the Deployment resource to create.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def stage_description(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDeployment.StageDescriptionProperty]]:
        '''The description of the Stage resource for the Deployment resource to create.

        To specify a stage description, you must also provide a stage name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-stagedescription
        '''
        result = self._values.get("stage_description")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDeployment.StageDescriptionProperty]], result)

    @builtins.property
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''The name of the Stage resource for the Deployment resource to create.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html#cfn-apigateway-deployment-stagename
        '''
        result = self._values.get("stage_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDeploymentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnDocumentationPart(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDocumentationPart",
):
    '''The ``AWS::ApiGateway::DocumentationPart`` resource creates a documentation part for an API.

    For more information, see `Representation of API Documentation in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-documenting-api-content-representation.html>`_ in the *API Gateway Developer Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html
    :cloudformationResource: AWS::ApiGateway::DocumentationPart
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_documentation_part = apigateway.CfnDocumentationPart(self, "MyCfnDocumentationPart",
            location=apigateway.CfnDocumentationPart.LocationProperty(
                method="method",
                name="name",
                path="path",
                status_code="statusCode",
                type="type"
            ),
            properties="properties",
            rest_api_id="restApiId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        location: typing.Union[_IResolvable_da3f097b, typing.Union["CfnDocumentationPart.LocationProperty", typing.Dict[builtins.str, typing.Any]]],
        properties: builtins.str,
        rest_api_id: builtins.str,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param location: The location of the targeted API entity of the to-be-created documentation part.
        :param properties: The new documentation content map of the targeted API entity. Enclosed key-value pairs are API-specific, but only OpenAPI-compliant key-value pairs can be exported and, hence, published.
        :param rest_api_id: The string identifier of the associated RestApi.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__38ba032d40661e68b76931f3085a3dab3e166be67b860e94e2f54fd43cfbdacd)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDocumentationPartProps(
            location=location, properties=properties, rest_api_id=rest_api_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bbd1a7ed85d5d62f62997aec6eb3cfe45a2969b7cf47a7e38ad93d91556d5ceb)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8c7ba659d69adcbd8f7bf3fcbc7bbc172ba9b04c160c98ba1a303f3f77c17ac3)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDocumentationPartId")
    def attr_documentation_part_id(self) -> builtins.str:
        '''The ID for the documentation part.

        :cloudformationAttribute: DocumentationPartId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDocumentationPartId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="location")
    def location(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, "CfnDocumentationPart.LocationProperty"]:
        '''The location of the targeted API entity of the to-be-created documentation part.'''
        return typing.cast(typing.Union[_IResolvable_da3f097b, "CfnDocumentationPart.LocationProperty"], jsii.get(self, "location"))

    @location.setter
    def location(
        self,
        value: typing.Union[_IResolvable_da3f097b, "CfnDocumentationPart.LocationProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__14f4e665aec867e4a2272885876c056410b925cd0b123c722ea45b2955af85a1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "location", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="properties")
    def properties(self) -> builtins.str:
        '''The new documentation content map of the targeted API entity.'''
        return typing.cast(builtins.str, jsii.get(self, "properties"))

    @properties.setter
    def properties(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__93f87e006391ddb3c52721d6dd593d3d4aaed837c384b102ce1fa28c2614ab77)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "properties", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.'''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d314682c2a6c1a3e1a2a0715fc4d7841ba5e92b9328b6f46355136fbfc6f3faf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnDocumentationPart.LocationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "method": "method",
            "name": "name",
            "path": "path",
            "status_code": "statusCode",
            "type": "type",
        },
    )
    class LocationProperty:
        def __init__(
            self,
            *,
            method: typing.Optional[builtins.str] = None,
            name: typing.Optional[builtins.str] = None,
            path: typing.Optional[builtins.str] = None,
            status_code: typing.Optional[builtins.str] = None,
            type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``Location`` property specifies the location of the Amazon API Gateway API entity that the documentation applies to.

            ``Location`` is a property of the `AWS::ApiGateway::DocumentationPart <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html>`_ resource.
            .. epigraph::

               For more information about each property, including constraints and valid values, see `DocumentationPart <https://docs.aws.amazon.com/apigateway/latest/api/API_DocumentationPartLocation.html>`_ in the *Amazon API Gateway REST API Reference* .

            :param method: The HTTP verb of a method. It is a valid field for the API entity types of ``METHOD`` , ``PATH_PARAMETER`` , ``QUERY_PARAMETER`` , ``REQUEST_HEADER`` , ``REQUEST_BODY`` , ``RESPONSE`` , ``RESPONSE_HEADER`` , and ``RESPONSE_BODY`` . The default value is ``*`` for any method. When an applicable child entity inherits the content of an entity of the same type with more general specifications of the other ``location`` attributes, the child entity's ``method`` attribute must match that of the parent entity exactly.
            :param name: The name of the targeted API entity. It is a valid and required field for the API entity types of ``AUTHORIZER`` , ``MODEL`` , ``PATH_PARAMETER`` , ``QUERY_PARAMETER`` , ``REQUEST_HEADER`` , ``REQUEST_BODY`` and ``RESPONSE_HEADER`` . It is an invalid field for any other entity type.
            :param path: The URL path of the target. It is a valid field for the API entity types of ``RESOURCE`` , ``METHOD`` , ``PATH_PARAMETER`` , ``QUERY_PARAMETER`` , ``REQUEST_HEADER`` , ``REQUEST_BODY`` , ``RESPONSE`` , ``RESPONSE_HEADER`` , and ``RESPONSE_BODY`` . The default value is ``/`` for the root resource. When an applicable child entity inherits the content of another entity of the same type with more general specifications of the other ``location`` attributes, the child entity's ``path`` attribute must match that of the parent entity as a prefix.
            :param status_code: The HTTP status code of a response. It is a valid field for the API entity types of ``RESPONSE`` , ``RESPONSE_HEADER`` , and ``RESPONSE_BODY`` . The default value is ``*`` for any status code. When an applicable child entity inherits the content of an entity of the same type with more general specifications of the other ``location`` attributes, the child entity's ``statusCode`` attribute must match that of the parent entity exactly.
            :param type: The type of API entity to which the documentation content applies. Valid values are ``API`` , ``AUTHORIZER`` , ``MODEL`` , ``RESOURCE`` , ``METHOD`` , ``PATH_PARAMETER`` , ``QUERY_PARAMETER`` , ``REQUEST_HEADER`` , ``REQUEST_BODY`` , ``RESPONSE`` , ``RESPONSE_HEADER`` , and ``RESPONSE_BODY`` . Content inheritance does not apply to any entity of the ``API`` , ``AUTHORIZER`` , ``METHOD`` , ``MODEL`` , ``REQUEST_BODY`` , or ``RESOURCE`` type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                location_property = apigateway.CfnDocumentationPart.LocationProperty(
                    method="method",
                    name="name",
                    path="path",
                    status_code="statusCode",
                    type="type"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1d2fece6f4654168298ecc69a873fe8a30ad43fd230f47d35f9ae6fcb449fb21)
                check_type(argname="argument method", value=method, expected_type=type_hints["method"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument path", value=path, expected_type=type_hints["path"])
                check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if method is not None:
                self._values["method"] = method
            if name is not None:
                self._values["name"] = name
            if path is not None:
                self._values["path"] = path
            if status_code is not None:
                self._values["status_code"] = status_code
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def method(self) -> typing.Optional[builtins.str]:
            '''The HTTP verb of a method.

            It is a valid field for the API entity types of ``METHOD`` , ``PATH_PARAMETER`` , ``QUERY_PARAMETER`` , ``REQUEST_HEADER`` , ``REQUEST_BODY`` , ``RESPONSE`` , ``RESPONSE_HEADER`` , and ``RESPONSE_BODY`` . The default value is ``*`` for any method. When an applicable child entity inherits the content of an entity of the same type with more general specifications of the other ``location`` attributes, the child entity's ``method`` attribute must match that of the parent entity exactly.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html#cfn-apigateway-documentationpart-location-method
            '''
            result = self._values.get("method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the targeted API entity.

            It is a valid and required field for the API entity types of ``AUTHORIZER`` , ``MODEL`` , ``PATH_PARAMETER`` , ``QUERY_PARAMETER`` , ``REQUEST_HEADER`` , ``REQUEST_BODY`` and ``RESPONSE_HEADER`` . It is an invalid field for any other entity type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html#cfn-apigateway-documentationpart-location-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def path(self) -> typing.Optional[builtins.str]:
            '''The URL path of the target.

            It is a valid field for the API entity types of ``RESOURCE`` , ``METHOD`` , ``PATH_PARAMETER`` , ``QUERY_PARAMETER`` , ``REQUEST_HEADER`` , ``REQUEST_BODY`` , ``RESPONSE`` , ``RESPONSE_HEADER`` , and ``RESPONSE_BODY`` . The default value is ``/`` for the root resource. When an applicable child entity inherits the content of another entity of the same type with more general specifications of the other ``location`` attributes, the child entity's ``path`` attribute must match that of the parent entity as a prefix.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html#cfn-apigateway-documentationpart-location-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def status_code(self) -> typing.Optional[builtins.str]:
            '''The HTTP status code of a response.

            It is a valid field for the API entity types of ``RESPONSE`` , ``RESPONSE_HEADER`` , and ``RESPONSE_BODY`` . The default value is ``*`` for any status code. When an applicable child entity inherits the content of an entity of the same type with more general specifications of the other ``location`` attributes, the child entity's ``statusCode`` attribute must match that of the parent entity exactly.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html#cfn-apigateway-documentationpart-location-statuscode
            '''
            result = self._values.get("status_code")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The type of API entity to which the documentation content applies.

            Valid values are ``API`` , ``AUTHORIZER`` , ``MODEL`` , ``RESOURCE`` , ``METHOD`` , ``PATH_PARAMETER`` , ``QUERY_PARAMETER`` , ``REQUEST_HEADER`` , ``REQUEST_BODY`` , ``RESPONSE`` , ``RESPONSE_HEADER`` , and ``RESPONSE_BODY`` . Content inheritance does not apply to any entity of the ``API`` , ``AUTHORIZER`` , ``METHOD`` , ``MODEL`` , ``REQUEST_BODY`` , or ``RESOURCE`` type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-documentationpart-location.html#cfn-apigateway-documentationpart-location-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LocationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDocumentationPartProps",
    jsii_struct_bases=[],
    name_mapping={
        "location": "location",
        "properties": "properties",
        "rest_api_id": "restApiId",
    },
)
class CfnDocumentationPartProps:
    def __init__(
        self,
        *,
        location: typing.Union[_IResolvable_da3f097b, typing.Union[CfnDocumentationPart.LocationProperty, typing.Dict[builtins.str, typing.Any]]],
        properties: builtins.str,
        rest_api_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnDocumentationPart``.

        :param location: The location of the targeted API entity of the to-be-created documentation part.
        :param properties: The new documentation content map of the targeted API entity. Enclosed key-value pairs are API-specific, but only OpenAPI-compliant key-value pairs can be exported and, hence, published.
        :param rest_api_id: The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_documentation_part_props = apigateway.CfnDocumentationPartProps(
                location=apigateway.CfnDocumentationPart.LocationProperty(
                    method="method",
                    name="name",
                    path="path",
                    status_code="statusCode",
                    type="type"
                ),
                properties="properties",
                rest_api_id="restApiId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1ce97e316ab1d99303726764131f5f4cf26ff37cf46f422efc545bce573d3b53)
            check_type(argname="argument location", value=location, expected_type=type_hints["location"])
            check_type(argname="argument properties", value=properties, expected_type=type_hints["properties"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "location": location,
            "properties": properties,
            "rest_api_id": rest_api_id,
        }

    @builtins.property
    def location(
        self,
    ) -> typing.Union[_IResolvable_da3f097b, CfnDocumentationPart.LocationProperty]:
        '''The location of the targeted API entity of the to-be-created documentation part.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html#cfn-apigateway-documentationpart-location
        '''
        result = self._values.get("location")
        assert result is not None, "Required property 'location' is missing"
        return typing.cast(typing.Union[_IResolvable_da3f097b, CfnDocumentationPart.LocationProperty], result)

    @builtins.property
    def properties(self) -> builtins.str:
        '''The new documentation content map of the targeted API entity.

        Enclosed key-value pairs are API-specific, but only OpenAPI-compliant key-value pairs can be exported and, hence, published.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html#cfn-apigateway-documentationpart-properties
        '''
        result = self._values.get("properties")
        assert result is not None, "Required property 'properties' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationpart.html#cfn-apigateway-documentationpart-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDocumentationPartProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnDocumentationVersion(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDocumentationVersion",
):
    '''The ``AWS::ApiGateway::DocumentationVersion`` resource creates a snapshot of the documentation for an API.

    For more information, see `Representation of API Documentation in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-documenting-api-content-representation.html>`_ in the *API Gateway Developer Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html
    :cloudformationResource: AWS::ApiGateway::DocumentationVersion
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_documentation_version = apigateway.CfnDocumentationVersion(self, "MyCfnDocumentationVersion",
            documentation_version="documentationVersion",
            rest_api_id="restApiId",
        
            # the properties below are optional
            description="description"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        documentation_version: builtins.str,
        rest_api_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param documentation_version: The version identifier of the to-be-updated documentation version.
        :param rest_api_id: The string identifier of the associated RestApi.
        :param description: A description about the new documentation snapshot.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4ce792256f8131ec9fafbd096588944d5d03c9e79025dc031faf45d3776aa4cd)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDocumentationVersionProps(
            documentation_version=documentation_version,
            rest_api_id=rest_api_id,
            description=description,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8972ba9700d872b4b17175092a898a4ab78317a71ca0f51753e766e3c301d06a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__571b85c565bc49e0f0f899bef689ebeebddabfe0ed5321744e49235f0d6456b0)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="documentationVersion")
    def documentation_version(self) -> builtins.str:
        '''The version identifier of the to-be-updated documentation version.'''
        return typing.cast(builtins.str, jsii.get(self, "documentationVersion"))

    @documentation_version.setter
    def documentation_version(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__14527ce69d3494649d2319600be0b62cea7a86088da67ddb0b293d7632ed29a0)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "documentationVersion", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.'''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f76f5a6a1ba4169ac830506b007c4bbb0ce553540e7d6514f143923702ebeb37)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''A description about the new documentation snapshot.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__574316b6d4d513be2315a81cecd2b6fbd1cb48bdded71155573dafc5d3e6c568)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDocumentationVersionProps",
    jsii_struct_bases=[],
    name_mapping={
        "documentation_version": "documentationVersion",
        "rest_api_id": "restApiId",
        "description": "description",
    },
)
class CfnDocumentationVersionProps:
    def __init__(
        self,
        *,
        documentation_version: builtins.str,
        rest_api_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnDocumentationVersion``.

        :param documentation_version: The version identifier of the to-be-updated documentation version.
        :param rest_api_id: The string identifier of the associated RestApi.
        :param description: A description about the new documentation snapshot.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_documentation_version_props = apigateway.CfnDocumentationVersionProps(
                documentation_version="documentationVersion",
                rest_api_id="restApiId",
            
                # the properties below are optional
                description="description"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d814a1ed5f46df046a60d908da8c64063747374742647ce116d53f1dd4684daa)
            check_type(argname="argument documentation_version", value=documentation_version, expected_type=type_hints["documentation_version"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "documentation_version": documentation_version,
            "rest_api_id": rest_api_id,
        }
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def documentation_version(self) -> builtins.str:
        '''The version identifier of the to-be-updated documentation version.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html#cfn-apigateway-documentationversion-documentationversion
        '''
        result = self._values.get("documentation_version")
        assert result is not None, "Required property 'documentation_version' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html#cfn-apigateway-documentationversion-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description about the new documentation snapshot.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-documentationversion.html#cfn-apigateway-documentationversion-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDocumentationVersionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnDomainName(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDomainName",
):
    '''The ``AWS::ApiGateway::DomainName`` resource specifies a public custom domain name for your API in API Gateway.

    You can use a custom domain name to provide a URL that's more intuitive and easier to recall. For more information about using custom domain names, see `Set up Custom Domain Name for an API in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-custom-domains.html>`_ in the *API Gateway Developer Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html
    :cloudformationResource: AWS::ApiGateway::DomainName
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_domain_name = apigateway.CfnDomainName(self, "MyCfnDomainName",
            certificate_arn="certificateArn",
            domain_name="domainName",
            endpoint_configuration=apigateway.CfnDomainName.EndpointConfigurationProperty(
                ip_address_type="ipAddressType",
                types=["types"]
            ),
            mutual_tls_authentication=apigateway.CfnDomainName.MutualTlsAuthenticationProperty(
                truststore_uri="truststoreUri",
                truststore_version="truststoreVersion"
            ),
            ownership_verification_certificate_arn="ownershipVerificationCertificateArn",
            regional_certificate_arn="regionalCertificateArn",
            security_policy="securityPolicy",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        certificate_arn: typing.Optional[builtins.str] = None,
        domain_name: typing.Optional[builtins.str] = None,
        endpoint_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDomainName.EndpointConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        mutual_tls_authentication: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDomainName.MutualTlsAuthenticationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ownership_verification_certificate_arn: typing.Optional[builtins.str] = None,
        regional_certificate_arn: typing.Optional[builtins.str] = None,
        security_policy: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param certificate_arn: The reference to an AWS -managed certificate that will be used by edge-optimized endpoint or private endpoint for this domain name. AWS Certificate Manager is the only supported source.
        :param domain_name: The custom domain name as an API host name, for example, ``my-api.example.com`` .
        :param endpoint_configuration: The endpoint configuration of this DomainName showing the endpoint types and IP address types of the domain name.
        :param mutual_tls_authentication: The mutual TLS authentication configuration for a custom domain name. If specified, API Gateway performs two-way authentication between the client and the server. Clients must present a trusted certificate to access your API.
        :param ownership_verification_certificate_arn: The ARN of the public certificate issued by ACM to validate ownership of your custom domain. Only required when configuring mutual TLS and using an ACM imported or private CA certificate ARN as the RegionalCertificateArn.
        :param regional_certificate_arn: The reference to an AWS -managed certificate that will be used for validating the regional domain name. AWS Certificate Manager is the only supported source.
        :param security_policy: The Transport Layer Security (TLS) version + cipher suite for this DomainName. The valid values are ``TLS_1_0`` and ``TLS_1_2`` .
        :param tags: The collection of tags. Each tag element is associated with a given resource.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__133ab79e698bd61c150ef7c699a84e864403caafbb03bd0bc5874cee67b604f3)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDomainNameProps(
            certificate_arn=certificate_arn,
            domain_name=domain_name,
            endpoint_configuration=endpoint_configuration,
            mutual_tls_authentication=mutual_tls_authentication,
            ownership_verification_certificate_arn=ownership_verification_certificate_arn,
            regional_certificate_arn=regional_certificate_arn,
            security_policy=security_policy,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__37a1235a5256f1b7d85e64cb2b5f3e81e3b6388eed32f66d2b4549fc757b871a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2937d2f529e3d913b825272173daa1bea3bdab64963a6ffe21d9a4725e6d7795)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDistributionDomainName")
    def attr_distribution_domain_name(self) -> builtins.str:
        '''The Amazon CloudFront distribution domain name that's mapped to the custom domain name.

        This is only applicable for endpoints whose type is ``EDGE`` .

        Example: ``d111111abcdef8.cloudfront.net``

        :cloudformationAttribute: DistributionDomainName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDistributionDomainName"))

    @builtins.property
    @jsii.member(jsii_name="attrDistributionHostedZoneId")
    def attr_distribution_hosted_zone_id(self) -> builtins.str:
        '''The region-agnostic Amazon Route 53 Hosted Zone ID of the edge-optimized endpoint.

        The only valid value is ``Z2FDTNDATAQYW2`` for all regions.

        :cloudformationAttribute: DistributionHostedZoneId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDistributionHostedZoneId"))

    @builtins.property
    @jsii.member(jsii_name="attrRegionalDomainName")
    def attr_regional_domain_name(self) -> builtins.str:
        '''The domain name associated with the regional endpoint for this custom domain name.

        You set up this association by adding a DNS record that points the custom domain name to this regional domain name.

        :cloudformationAttribute: RegionalDomainName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRegionalDomainName"))

    @builtins.property
    @jsii.member(jsii_name="attrRegionalHostedZoneId")
    def attr_regional_hosted_zone_id(self) -> builtins.str:
        '''The region-specific Amazon Route 53 Hosted Zone ID of the regional endpoint.

        :cloudformationAttribute: RegionalHostedZoneId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRegionalHostedZoneId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="certificateArn")
    def certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The reference to an AWS -managed certificate that will be used by edge-optimized endpoint or private endpoint for this domain name.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "certificateArn"))

    @certificate_arn.setter
    def certificate_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4ddcd44c07e8517918b3c0629937de8d9249383bd9fe6dca514c5ae57020f60c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "certificateArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''The custom domain name as an API host name, for example, ``my-api.example.com`` .'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domainName"))

    @domain_name.setter
    def domain_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__301350d1701372cecf10b01001cc69af90a5d9626237dfc86d8e6b3930aa621c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="endpointConfiguration")
    def endpoint_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDomainName.EndpointConfigurationProperty"]]:
        '''The endpoint configuration of this DomainName showing the endpoint types and IP address types of the domain name.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDomainName.EndpointConfigurationProperty"]], jsii.get(self, "endpointConfiguration"))

    @endpoint_configuration.setter
    def endpoint_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDomainName.EndpointConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b6543b7bdb9b509d16f1b80ad0629b95916d74d5bfa3d2cd3479381295a54c3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endpointConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="mutualTlsAuthentication")
    def mutual_tls_authentication(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDomainName.MutualTlsAuthenticationProperty"]]:
        '''The mutual TLS authentication configuration for a custom domain name.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDomainName.MutualTlsAuthenticationProperty"]], jsii.get(self, "mutualTlsAuthentication"))

    @mutual_tls_authentication.setter
    def mutual_tls_authentication(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDomainName.MutualTlsAuthenticationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2f97eed060618803071c91479b39994cbcdc98cdd93e5b0b72f4ffc2872d7544)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "mutualTlsAuthentication", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="ownershipVerificationCertificateArn")
    def ownership_verification_certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the public certificate issued by ACM to validate ownership of your custom domain.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ownershipVerificationCertificateArn"))

    @ownership_verification_certificate_arn.setter
    def ownership_verification_certificate_arn(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c30f641a8ac50e928250c6aa2cebbbef31cde48d9dfbb531dacfcaff08a87853)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "ownershipVerificationCertificateArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="regionalCertificateArn")
    def regional_certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The reference to an AWS -managed certificate that will be used for validating the regional domain name.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "regionalCertificateArn"))

    @regional_certificate_arn.setter
    def regional_certificate_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2fa342814b3288d616098cb51248e53b5ed60cc5a1e78e11251fc88cdc16fb4f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "regionalCertificateArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="securityPolicy")
    def security_policy(self) -> typing.Optional[builtins.str]:
        '''The Transport Layer Security (TLS) version + cipher suite for this DomainName.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "securityPolicy"))

    @security_policy.setter
    def security_policy(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8f65e2b19ddf5075e1df27ef655917695bca8ced9d2b2266dbdadebde2c104bf)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "securityPolicy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30658cbd8b896f811f887630d934f0751b5370d775062da4c8686d7caa346a6d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnDomainName.EndpointConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"ip_address_type": "ipAddressType", "types": "types"},
    )
    class EndpointConfigurationProperty:
        def __init__(
            self,
            *,
            ip_address_type: typing.Optional[builtins.str] = None,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The ``EndpointConfiguration`` property type specifies the endpoint types and IP address types of an Amazon API Gateway domain name.

            ``EndpointConfiguration`` is a property of the `AWS::ApiGateway::DomainName <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html>`_ resource.

            :param ip_address_type: The IP address types that can invoke this DomainName. Use ``ipv4`` to allow only IPv4 addresses to invoke this DomainName, or use ``dualstack`` to allow both IPv4 and IPv6 addresses to invoke this DomainName. For the ``PRIVATE`` endpoint type, only ``dualstack`` is supported.
            :param types: A list of endpoint types of an API (RestApi) or its custom domain name (DomainName). For an edge-optimized API and its custom domain name, the endpoint type is ``"EDGE"`` . For a regional API and its custom domain name, the endpoint type is ``REGIONAL`` . For a private API, the endpoint type is ``PRIVATE`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainname-endpointconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                endpoint_configuration_property = apigateway.CfnDomainName.EndpointConfigurationProperty(
                    ip_address_type="ipAddressType",
                    types=["types"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__1d33c3bf6996efa92019e1b28e2d6c1c9ad4dd0fa50f2ad639962efcfa774fb2)
                check_type(argname="argument ip_address_type", value=ip_address_type, expected_type=type_hints["ip_address_type"])
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if ip_address_type is not None:
                self._values["ip_address_type"] = ip_address_type
            if types is not None:
                self._values["types"] = types

        @builtins.property
        def ip_address_type(self) -> typing.Optional[builtins.str]:
            '''The IP address types that can invoke this DomainName.

            Use ``ipv4`` to allow only IPv4 addresses to invoke this DomainName, or use ``dualstack`` to allow both IPv4 and IPv6 addresses to invoke this DomainName. For the ``PRIVATE`` endpoint type, only ``dualstack`` is supported.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainname-endpointconfiguration.html#cfn-apigateway-domainname-endpointconfiguration-ipaddresstype
            '''
            result = self._values.get("ip_address_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of endpoint types of an API (RestApi) or its custom domain name (DomainName).

            For an edge-optimized API and its custom domain name, the endpoint type is ``"EDGE"`` . For a regional API and its custom domain name, the endpoint type is ``REGIONAL`` . For a private API, the endpoint type is ``PRIVATE`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainname-endpointconfiguration.html#cfn-apigateway-domainname-endpointconfiguration-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndpointConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnDomainName.MutualTlsAuthenticationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "truststore_uri": "truststoreUri",
            "truststore_version": "truststoreVersion",
        },
    )
    class MutualTlsAuthenticationProperty:
        def __init__(
            self,
            *,
            truststore_uri: typing.Optional[builtins.str] = None,
            truststore_version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The mutual TLS authentication configuration for a custom domain name.

            If specified, API Gateway performs two-way authentication between the client and the server. Clients must present a trusted certificate to access your API.

            :param truststore_uri: An Amazon S3 URL that specifies the truststore for mutual TLS authentication, for example ``s3://bucket-name/key-name`` . The truststore can contain certificates from public or private certificate authorities. To update the truststore, upload a new version to S3, and then update your custom domain name to use the new version. To update the truststore, you must have permissions to access the S3 object.
            :param truststore_version: The version of the S3 object that contains your truststore. To specify a version, you must have versioning enabled for the S3 bucket.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainname-mutualtlsauthentication.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                mutual_tls_authentication_property = apigateway.CfnDomainName.MutualTlsAuthenticationProperty(
                    truststore_uri="truststoreUri",
                    truststore_version="truststoreVersion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__20d6debf4c14a68aa82e5a1c08dd101632582c4d47645b17413b831ffec460a8)
                check_type(argname="argument truststore_uri", value=truststore_uri, expected_type=type_hints["truststore_uri"])
                check_type(argname="argument truststore_version", value=truststore_version, expected_type=type_hints["truststore_version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if truststore_uri is not None:
                self._values["truststore_uri"] = truststore_uri
            if truststore_version is not None:
                self._values["truststore_version"] = truststore_version

        @builtins.property
        def truststore_uri(self) -> typing.Optional[builtins.str]:
            '''An Amazon S3 URL that specifies the truststore for mutual TLS authentication, for example ``s3://bucket-name/key-name`` .

            The truststore can contain certificates from public or private certificate authorities. To update the truststore, upload a new version to S3, and then update your custom domain name to use the new version. To update the truststore, you must have permissions to access the S3 object.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainname-mutualtlsauthentication.html#cfn-apigateway-domainname-mutualtlsauthentication-truststoreuri
            '''
            result = self._values.get("truststore_uri")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def truststore_version(self) -> typing.Optional[builtins.str]:
            '''The version of the S3 object that contains your truststore.

            To specify a version, you must have versioning enabled for the S3 bucket.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainname-mutualtlsauthentication.html#cfn-apigateway-domainname-mutualtlsauthentication-truststoreversion
            '''
            result = self._values.get("truststore_version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MutualTlsAuthenticationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnDomainNameAccessAssociation(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDomainNameAccessAssociation",
):
    '''The ``AWS::ApiGateway::DomainNameAccessAssociation`` resource creates a domain name access association between an access association source and a private custom domain name.

    Use a domain name access association to invoke a private custom domain name while isolated from the public internet.

    You can only create or delete a DomainNameAccessAssociation using CloudFormation. To reject a domain name access association, use the AWS CLI.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnameaccessassociation.html
    :cloudformationResource: AWS::ApiGateway::DomainNameAccessAssociation
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_domain_name_access_association = apigateway.CfnDomainNameAccessAssociation(self, "MyCfnDomainNameAccessAssociation",
            access_association_source="accessAssociationSource",
            access_association_source_type="accessAssociationSourceType",
            domain_name_arn="domainNameArn",
        
            # the properties below are optional
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        access_association_source: builtins.str,
        access_association_source_type: builtins.str,
        domain_name_arn: builtins.str,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param access_association_source: The identifier of the domain name access association source. For a ``VPCE`` , the value is the VPC endpoint ID.
        :param access_association_source_type: The type of the domain name access association source. Only ``VPCE`` is currently supported.
        :param domain_name_arn: The ARN of the domain name.
        :param tags: The collection of tags. Each tag element is associated with a given resource.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cafcd3b452004fc09063699f5b2263d0f36c2a080b20c7f0d51493646ac87441)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDomainNameAccessAssociationProps(
            access_association_source=access_association_source,
            access_association_source_type=access_association_source_type,
            domain_name_arn=domain_name_arn,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d4584e14c8cc1eda344bf13cbc96b7ff7cf893e84c7aa3abf7ce385dfd0ebb0b)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__635e79e806ac4eaac4502971d453a0581ed2a8620fc4309d2fa5820e1eea4413)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDomainNameAccessAssociationArn")
    def attr_domain_name_access_association_arn(self) -> builtins.str:
        '''The access association ARN.

        For example: ``arn:aws:apigateway:us-west-2:111122223333:/accessassociations/domainname/private.example.com+abcd1234/vpcesource/vpce-abcd1234efg`` .

        :cloudformationAttribute: DomainNameAccessAssociationArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDomainNameAccessAssociationArn"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="accessAssociationSource")
    def access_association_source(self) -> builtins.str:
        '''The identifier of the domain name access association source.'''
        return typing.cast(builtins.str, jsii.get(self, "accessAssociationSource"))

    @access_association_source.setter
    def access_association_source(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__23c76b51f9a9a8e6cecdd5cb498ecd02cfa61809c66806005dd3fc4f5f33f35c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "accessAssociationSource", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="accessAssociationSourceType")
    def access_association_source_type(self) -> builtins.str:
        '''The type of the domain name access association source.'''
        return typing.cast(builtins.str, jsii.get(self, "accessAssociationSourceType"))

    @access_association_source_type.setter
    def access_association_source_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__22ceba82da2b5b77979cdbe562907d1ee3c704bfcb564f766864aaff695f3400)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "accessAssociationSourceType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="domainNameArn")
    def domain_name_arn(self) -> builtins.str:
        '''The ARN of the domain name.'''
        return typing.cast(builtins.str, jsii.get(self, "domainNameArn"))

    @domain_name_arn.setter
    def domain_name_arn(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5a9c3f25d11f724e7f4c78554b51bc3fc13c83e113087fbbf7d37e9868c51bdb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainNameArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tags"))

    @tags.setter
    def tags(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2c58e960117de6ef208f31f47e57fea4a9b3894dc90dde201f0f6f182cab17d8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDomainNameAccessAssociationProps",
    jsii_struct_bases=[],
    name_mapping={
        "access_association_source": "accessAssociationSource",
        "access_association_source_type": "accessAssociationSourceType",
        "domain_name_arn": "domainNameArn",
        "tags": "tags",
    },
)
class CfnDomainNameAccessAssociationProps:
    def __init__(
        self,
        *,
        access_association_source: builtins.str,
        access_association_source_type: builtins.str,
        domain_name_arn: builtins.str,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDomainNameAccessAssociation``.

        :param access_association_source: The identifier of the domain name access association source. For a ``VPCE`` , the value is the VPC endpoint ID.
        :param access_association_source_type: The type of the domain name access association source. Only ``VPCE`` is currently supported.
        :param domain_name_arn: The ARN of the domain name.
        :param tags: The collection of tags. Each tag element is associated with a given resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnameaccessassociation.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_domain_name_access_association_props = apigateway.CfnDomainNameAccessAssociationProps(
                access_association_source="accessAssociationSource",
                access_association_source_type="accessAssociationSourceType",
                domain_name_arn="domainNameArn",
            
                # the properties below are optional
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__519fe3488e579f2ddfc56721bd12d5c2520b9b787080002ce6f4f88bbc90d50a)
            check_type(argname="argument access_association_source", value=access_association_source, expected_type=type_hints["access_association_source"])
            check_type(argname="argument access_association_source_type", value=access_association_source_type, expected_type=type_hints["access_association_source_type"])
            check_type(argname="argument domain_name_arn", value=domain_name_arn, expected_type=type_hints["domain_name_arn"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "access_association_source": access_association_source,
            "access_association_source_type": access_association_source_type,
            "domain_name_arn": domain_name_arn,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def access_association_source(self) -> builtins.str:
        '''The identifier of the domain name access association source.

        For a ``VPCE`` , the value is the VPC endpoint ID.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnameaccessassociation.html#cfn-apigateway-domainnameaccessassociation-accessassociationsource
        '''
        result = self._values.get("access_association_source")
        assert result is not None, "Required property 'access_association_source' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_association_source_type(self) -> builtins.str:
        '''The type of the domain name access association source.

        Only ``VPCE`` is currently supported.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnameaccessassociation.html#cfn-apigateway-domainnameaccessassociation-accessassociationsourcetype
        '''
        result = self._values.get("access_association_source_type")
        assert result is not None, "Required property 'access_association_source_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def domain_name_arn(self) -> builtins.str:
        '''The ARN of the domain name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnameaccessassociation.html#cfn-apigateway-domainnameaccessassociation-domainnamearn
        '''
        result = self._values.get("domain_name_arn")
        assert result is not None, "Required property 'domain_name_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.

        Each tag element is associated with a given resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnameaccessassociation.html#cfn-apigateway-domainnameaccessassociation-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDomainNameAccessAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDomainNameProps",
    jsii_struct_bases=[],
    name_mapping={
        "certificate_arn": "certificateArn",
        "domain_name": "domainName",
        "endpoint_configuration": "endpointConfiguration",
        "mutual_tls_authentication": "mutualTlsAuthentication",
        "ownership_verification_certificate_arn": "ownershipVerificationCertificateArn",
        "regional_certificate_arn": "regionalCertificateArn",
        "security_policy": "securityPolicy",
        "tags": "tags",
    },
)
class CfnDomainNameProps:
    def __init__(
        self,
        *,
        certificate_arn: typing.Optional[builtins.str] = None,
        domain_name: typing.Optional[builtins.str] = None,
        endpoint_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDomainName.EndpointConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        mutual_tls_authentication: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDomainName.MutualTlsAuthenticationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        ownership_verification_certificate_arn: typing.Optional[builtins.str] = None,
        regional_certificate_arn: typing.Optional[builtins.str] = None,
        security_policy: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDomainName``.

        :param certificate_arn: The reference to an AWS -managed certificate that will be used by edge-optimized endpoint or private endpoint for this domain name. AWS Certificate Manager is the only supported source.
        :param domain_name: The custom domain name as an API host name, for example, ``my-api.example.com`` .
        :param endpoint_configuration: The endpoint configuration of this DomainName showing the endpoint types and IP address types of the domain name.
        :param mutual_tls_authentication: The mutual TLS authentication configuration for a custom domain name. If specified, API Gateway performs two-way authentication between the client and the server. Clients must present a trusted certificate to access your API.
        :param ownership_verification_certificate_arn: The ARN of the public certificate issued by ACM to validate ownership of your custom domain. Only required when configuring mutual TLS and using an ACM imported or private CA certificate ARN as the RegionalCertificateArn.
        :param regional_certificate_arn: The reference to an AWS -managed certificate that will be used for validating the regional domain name. AWS Certificate Manager is the only supported source.
        :param security_policy: The Transport Layer Security (TLS) version + cipher suite for this DomainName. The valid values are ``TLS_1_0`` and ``TLS_1_2`` .
        :param tags: The collection of tags. Each tag element is associated with a given resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_domain_name_props = apigateway.CfnDomainNameProps(
                certificate_arn="certificateArn",
                domain_name="domainName",
                endpoint_configuration=apigateway.CfnDomainName.EndpointConfigurationProperty(
                    ip_address_type="ipAddressType",
                    types=["types"]
                ),
                mutual_tls_authentication=apigateway.CfnDomainName.MutualTlsAuthenticationProperty(
                    truststore_uri="truststoreUri",
                    truststore_version="truststoreVersion"
                ),
                ownership_verification_certificate_arn="ownershipVerificationCertificateArn",
                regional_certificate_arn="regionalCertificateArn",
                security_policy="securityPolicy",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__703d29eb1e26e9742eff449d68ee5c6089daeb58e8544900d248c3a3743347d3)
            check_type(argname="argument certificate_arn", value=certificate_arn, expected_type=type_hints["certificate_arn"])
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument endpoint_configuration", value=endpoint_configuration, expected_type=type_hints["endpoint_configuration"])
            check_type(argname="argument mutual_tls_authentication", value=mutual_tls_authentication, expected_type=type_hints["mutual_tls_authentication"])
            check_type(argname="argument ownership_verification_certificate_arn", value=ownership_verification_certificate_arn, expected_type=type_hints["ownership_verification_certificate_arn"])
            check_type(argname="argument regional_certificate_arn", value=regional_certificate_arn, expected_type=type_hints["regional_certificate_arn"])
            check_type(argname="argument security_policy", value=security_policy, expected_type=type_hints["security_policy"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if certificate_arn is not None:
            self._values["certificate_arn"] = certificate_arn
        if domain_name is not None:
            self._values["domain_name"] = domain_name
        if endpoint_configuration is not None:
            self._values["endpoint_configuration"] = endpoint_configuration
        if mutual_tls_authentication is not None:
            self._values["mutual_tls_authentication"] = mutual_tls_authentication
        if ownership_verification_certificate_arn is not None:
            self._values["ownership_verification_certificate_arn"] = ownership_verification_certificate_arn
        if regional_certificate_arn is not None:
            self._values["regional_certificate_arn"] = regional_certificate_arn
        if security_policy is not None:
            self._values["security_policy"] = security_policy
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The reference to an AWS -managed certificate that will be used by edge-optimized endpoint or private endpoint for this domain name.

        AWS Certificate Manager is the only supported source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-certificatearn
        '''
        result = self._values.get("certificate_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''The custom domain name as an API host name, for example, ``my-api.example.com`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-domainname
        '''
        result = self._values.get("domain_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def endpoint_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDomainName.EndpointConfigurationProperty]]:
        '''The endpoint configuration of this DomainName showing the endpoint types and IP address types of the domain name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-endpointconfiguration
        '''
        result = self._values.get("endpoint_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDomainName.EndpointConfigurationProperty]], result)

    @builtins.property
    def mutual_tls_authentication(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDomainName.MutualTlsAuthenticationProperty]]:
        '''The mutual TLS authentication configuration for a custom domain name.

        If specified, API Gateway performs two-way authentication between the client and the server. Clients must present a trusted certificate to access your API.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-mutualtlsauthentication
        '''
        result = self._values.get("mutual_tls_authentication")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDomainName.MutualTlsAuthenticationProperty]], result)

    @builtins.property
    def ownership_verification_certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The ARN of the public certificate issued by ACM to validate ownership of your custom domain.

        Only required when configuring mutual TLS and using an ACM imported or private CA certificate ARN as the RegionalCertificateArn.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-ownershipverificationcertificatearn
        '''
        result = self._values.get("ownership_verification_certificate_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def regional_certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The reference to an AWS -managed certificate that will be used for validating the regional domain name.

        AWS Certificate Manager is the only supported source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-regionalcertificatearn
        '''
        result = self._values.get("regional_certificate_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def security_policy(self) -> typing.Optional[builtins.str]:
        '''The Transport Layer Security (TLS) version + cipher suite for this DomainName.

        The valid values are ``TLS_1_0`` and ``TLS_1_2`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-securitypolicy
        '''
        result = self._values.get("security_policy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.

        Each tag element is associated with a given resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html#cfn-apigateway-domainname-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDomainNameProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggableV2_4e6798f8)
class CfnDomainNameV2(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDomainNameV2",
):
    '''The ``AWS::ApiGateway::DomainNameV2`` resource specifies a custom domain name for your private APIs in API Gateway.

    You can use a private custom domain name to provide a URL for your private API that's more intuitive and easier to recall.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnamev2.html
    :cloudformationResource: AWS::ApiGateway::DomainNameV2
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        # policy: Any
        
        cfn_domain_name_v2 = apigateway.CfnDomainNameV2(self, "MyCfnDomainNameV2",
            certificate_arn="certificateArn",
            domain_name="domainName",
            endpoint_configuration=apigateway.CfnDomainNameV2.EndpointConfigurationProperty(
                types=["types"]
            ),
            policy=policy,
            security_policy="securityPolicy",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        certificate_arn: typing.Optional[builtins.str] = None,
        domain_name: typing.Optional[builtins.str] = None,
        endpoint_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnDomainNameV2.EndpointConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        policy: typing.Any = None,
        security_policy: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param certificate_arn: The reference to an AWS -managed certificate that will be used by the private endpoint for this domain name. AWS Certificate Manager is the only supported source.
        :param domain_name: Represents a custom domain name as a user-friendly host name of an API (RestApi).
        :param endpoint_configuration: The endpoint configuration to indicate the types of endpoints an API (RestApi) or its custom domain name (DomainName) has and the IP address types that can invoke it.
        :param policy: A stringified JSON policy document that applies to the ``execute-api`` service for this DomainName regardless of the caller and Method configuration. You can use ``Fn::ToJsonString`` to enter your ``policy`` . For more information, see `Fn::ToJsonString <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-ToJsonString.html>`_ .
        :param security_policy: The Transport Layer Security (TLS) version + cipher suite for this DomainName. Only ``TLS_1_2`` is supported.
        :param tags: The collection of tags. Each tag element is associated with a given resource.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a4f501d10d57d39a48913872efdcb7d389a076fed9b5aabb21034b8e60a1e0bf)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDomainNameV2Props(
            certificate_arn=certificate_arn,
            domain_name=domain_name,
            endpoint_configuration=endpoint_configuration,
            policy=policy,
            security_policy=security_policy,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cba990fa84005688ae2a90df64bc7ab39d02ac254f4c8a2fb935da000e626e3b)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__84c84802ec05ba723e240d805b21796f43a8f3cf08c5ef8677460e77d0a2182b)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDomainNameArn")
    def attr_domain_name_arn(self) -> builtins.str:
        '''The ARN of the domain name.

        :cloudformationAttribute: DomainNameArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDomainNameArn"))

    @builtins.property
    @jsii.member(jsii_name="attrDomainNameId")
    def attr_domain_name_id(self) -> builtins.str:
        '''The domain name ID.

        :cloudformationAttribute: DomainNameId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDomainNameId"))

    @builtins.property
    @jsii.member(jsii_name="cdkTagManager")
    def cdk_tag_manager(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "cdkTagManager"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="certificateArn")
    def certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The reference to an AWS -managed certificate that will be used by the private endpoint for this domain name.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "certificateArn"))

    @certificate_arn.setter
    def certificate_arn(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__02e36a521e1f2c0c18cff2c52c016e6370b767eb2b459e2c294bb4288285a448)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "certificateArn", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''Represents a custom domain name as a user-friendly host name of an API (RestApi).'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domainName"))

    @domain_name.setter
    def domain_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c6f92a0eb5f08e654ae978fba555493986c05eed5e088475197fb75aa2780b84)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "domainName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="endpointConfiguration")
    def endpoint_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDomainNameV2.EndpointConfigurationProperty"]]:
        '''The endpoint configuration to indicate the types of endpoints an API (RestApi) or its custom domain name (DomainName) has and the IP address types that can invoke it.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDomainNameV2.EndpointConfigurationProperty"]], jsii.get(self, "endpointConfiguration"))

    @endpoint_configuration.setter
    def endpoint_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnDomainNameV2.EndpointConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a1f648b520c9d40bb72a9028880e48540125da8ba68522e25c483400d79142dc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endpointConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="policy")
    def policy(self) -> typing.Any:
        '''A stringified JSON policy document that applies to the ``execute-api`` service for this DomainName regardless of the caller and Method configuration.'''
        return typing.cast(typing.Any, jsii.get(self, "policy"))

    @policy.setter
    def policy(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a0c6989179a94e32b700994895064456ee57af9eb39330638ade9b7801085b38)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "policy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="securityPolicy")
    def security_policy(self) -> typing.Optional[builtins.str]:
        '''The Transport Layer Security (TLS) version + cipher suite for this DomainName.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "securityPolicy"))

    @security_policy.setter
    def security_policy(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a85cede5bfb66595d91f3753ac1832695db8f9eb734e73ecbc1600f19d466199)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "securityPolicy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tags"))

    @tags.setter
    def tags(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__476074ac17a2d9034c89e6475b463dcb2b5ab63eb68064fdc492d43a910b9470)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tags", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnDomainNameV2.EndpointConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"types": "types"},
    )
    class EndpointConfigurationProperty:
        def __init__(
            self,
            *,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The endpoint configuration to indicate the types of endpoints an API (RestApi) or its custom domain name (DomainName) has and the IP address types that can invoke it.

            :param types: A list of endpoint types of an API (RestApi) or its custom domain name (DomainName). For an edge-optimized API and its custom domain name, the endpoint type is ``"EDGE"`` . For a regional API and its custom domain name, the endpoint type is ``REGIONAL`` . For a private API, the endpoint type is ``PRIVATE`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainnamev2-endpointconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                endpoint_configuration_property = apigateway.CfnDomainNameV2.EndpointConfigurationProperty(
                    types=["types"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6b6f3002516ae2cfe5144f016260236185910ca3b84cc4dc018cc660b10716c4)
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if types is not None:
                self._values["types"] = types

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of endpoint types of an API (RestApi) or its custom domain name (DomainName).

            For an edge-optimized API and its custom domain name, the endpoint type is ``"EDGE"`` . For a regional API and its custom domain name, the endpoint type is ``REGIONAL`` . For a private API, the endpoint type is ``PRIVATE`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-domainnamev2-endpointconfiguration.html#cfn-apigateway-domainnamev2-endpointconfiguration-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndpointConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnDomainNameV2Props",
    jsii_struct_bases=[],
    name_mapping={
        "certificate_arn": "certificateArn",
        "domain_name": "domainName",
        "endpoint_configuration": "endpointConfiguration",
        "policy": "policy",
        "security_policy": "securityPolicy",
        "tags": "tags",
    },
)
class CfnDomainNameV2Props:
    def __init__(
        self,
        *,
        certificate_arn: typing.Optional[builtins.str] = None,
        domain_name: typing.Optional[builtins.str] = None,
        endpoint_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnDomainNameV2.EndpointConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        policy: typing.Any = None,
        security_policy: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDomainNameV2``.

        :param certificate_arn: The reference to an AWS -managed certificate that will be used by the private endpoint for this domain name. AWS Certificate Manager is the only supported source.
        :param domain_name: Represents a custom domain name as a user-friendly host name of an API (RestApi).
        :param endpoint_configuration: The endpoint configuration to indicate the types of endpoints an API (RestApi) or its custom domain name (DomainName) has and the IP address types that can invoke it.
        :param policy: A stringified JSON policy document that applies to the ``execute-api`` service for this DomainName regardless of the caller and Method configuration. You can use ``Fn::ToJsonString`` to enter your ``policy`` . For more information, see `Fn::ToJsonString <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-ToJsonString.html>`_ .
        :param security_policy: The Transport Layer Security (TLS) version + cipher suite for this DomainName. Only ``TLS_1_2`` is supported.
        :param tags: The collection of tags. Each tag element is associated with a given resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnamev2.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            # policy: Any
            
            cfn_domain_name_v2_props = apigateway.CfnDomainNameV2Props(
                certificate_arn="certificateArn",
                domain_name="domainName",
                endpoint_configuration=apigateway.CfnDomainNameV2.EndpointConfigurationProperty(
                    types=["types"]
                ),
                policy=policy,
                security_policy="securityPolicy",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dfb97b3f8aa373f21ebe15318ceaaf08a4d4b4b31e9ba1ed8fb04ef598132e15)
            check_type(argname="argument certificate_arn", value=certificate_arn, expected_type=type_hints["certificate_arn"])
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument endpoint_configuration", value=endpoint_configuration, expected_type=type_hints["endpoint_configuration"])
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
            check_type(argname="argument security_policy", value=security_policy, expected_type=type_hints["security_policy"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if certificate_arn is not None:
            self._values["certificate_arn"] = certificate_arn
        if domain_name is not None:
            self._values["domain_name"] = domain_name
        if endpoint_configuration is not None:
            self._values["endpoint_configuration"] = endpoint_configuration
        if policy is not None:
            self._values["policy"] = policy
        if security_policy is not None:
            self._values["security_policy"] = security_policy
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def certificate_arn(self) -> typing.Optional[builtins.str]:
        '''The reference to an AWS -managed certificate that will be used by the private endpoint for this domain name.

        AWS Certificate Manager is the only supported source.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnamev2.html#cfn-apigateway-domainnamev2-certificatearn
        '''
        result = self._values.get("certificate_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''Represents a custom domain name as a user-friendly host name of an API (RestApi).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnamev2.html#cfn-apigateway-domainnamev2-domainname
        '''
        result = self._values.get("domain_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def endpoint_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDomainNameV2.EndpointConfigurationProperty]]:
        '''The endpoint configuration to indicate the types of endpoints an API (RestApi) or its custom domain name (DomainName) has and the IP address types that can invoke it.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnamev2.html#cfn-apigateway-domainnamev2-endpointconfiguration
        '''
        result = self._values.get("endpoint_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnDomainNameV2.EndpointConfigurationProperty]], result)

    @builtins.property
    def policy(self) -> typing.Any:
        '''A stringified JSON policy document that applies to the ``execute-api`` service for this DomainName regardless of the caller and Method configuration.

        You can use ``Fn::ToJsonString`` to enter your ``policy`` . For more information, see `Fn::ToJsonString <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/intrinsic-function-reference-ToJsonString.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnamev2.html#cfn-apigateway-domainnamev2-policy
        '''
        result = self._values.get("policy")
        return typing.cast(typing.Any, result)

    @builtins.property
    def security_policy(self) -> typing.Optional[builtins.str]:
        '''The Transport Layer Security (TLS) version + cipher suite for this DomainName.

        Only ``TLS_1_2`` is supported.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnamev2.html#cfn-apigateway-domainnamev2-securitypolicy
        '''
        result = self._values.get("security_policy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.

        Each tag element is associated with a given resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainnamev2.html#cfn-apigateway-domainnamev2-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDomainNameV2Props(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnGatewayResponse(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnGatewayResponse",
):
    '''The ``AWS::ApiGateway::GatewayResponse`` resource creates a gateway response for your API.

    For more information, see `API Gateway Responses <https://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html#api-gateway-gatewayResponse-definition>`_ in the *API Gateway Developer Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html
    :cloudformationResource: AWS::ApiGateway::GatewayResponse
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_gateway_response = apigateway.CfnGatewayResponse(self, "MyCfnGatewayResponse",
            response_type="responseType",
            rest_api_id="restApiId",
        
            # the properties below are optional
            response_parameters={
                "response_parameters_key": "responseParameters"
            },
            response_templates={
                "response_templates_key": "responseTemplates"
            },
            status_code="statusCode"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        response_type: builtins.str,
        rest_api_id: builtins.str,
        response_parameters: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        response_templates: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        status_code: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param response_type: The response type of the associated GatewayResponse.
        :param rest_api_id: The string identifier of the associated RestApi.
        :param response_parameters: Response parameters (paths, query strings and headers) of the GatewayResponse as a string-to-string map of key-value pairs.
        :param response_templates: Response templates of the GatewayResponse as a string-to-string map of key-value pairs.
        :param status_code: The HTTP status code for this GatewayResponse.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dfb51ee54cd182788cbab3215f86a2155730ef96e24304891e470a0af6430ec5)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnGatewayResponseProps(
            response_type=response_type,
            rest_api_id=rest_api_id,
            response_parameters=response_parameters,
            response_templates=response_templates,
            status_code=status_code,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__00883a5762b27b0f4132a400de1fbf07a96c596284722198c6fda6473b8e1775)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9a0dd27975eee73600e72672dcd3697e3ce3a839b7d5da100cb0247e93402629)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID for the gateway response.

        For example: ``abc123`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="responseType")
    def response_type(self) -> builtins.str:
        '''The response type of the associated GatewayResponse.'''
        return typing.cast(builtins.str, jsii.get(self, "responseType"))

    @response_type.setter
    def response_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__517489499e704efc0e1334edfbda78a8c3f8fb74c11556a6d3156744c73eb136)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "responseType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.'''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__191f17a0820f1571d8b0a7b5dbf5ce4d5b0c02faad775c2ded98e41ead62ce7c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="responseParameters")
    def response_parameters(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''Response parameters (paths, query strings and headers) of the GatewayResponse as a string-to-string map of key-value pairs.'''
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], jsii.get(self, "responseParameters"))

    @response_parameters.setter
    def response_parameters(
        self,
        value: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__14cecdeaa3ee3a718e1e415f74286686a7f74799e862f6644b9196fb61ecd39e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "responseParameters", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="responseTemplates")
    def response_templates(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''Response templates of the GatewayResponse as a string-to-string map of key-value pairs.'''
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], jsii.get(self, "responseTemplates"))

    @response_templates.setter
    def response_templates(
        self,
        value: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__632192144f7b8204333bf1adf1d758ee8653c884b9a77e07436f3f92f3cc02f1)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "responseTemplates", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="statusCode")
    def status_code(self) -> typing.Optional[builtins.str]:
        '''The HTTP status code for this GatewayResponse.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "statusCode"))

    @status_code.setter
    def status_code(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2b7e4863cef0d08069fc0f5b1902c06c852bbb4264a7bf29eb91dddbcf99796d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "statusCode", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnGatewayResponseProps",
    jsii_struct_bases=[],
    name_mapping={
        "response_type": "responseType",
        "rest_api_id": "restApiId",
        "response_parameters": "responseParameters",
        "response_templates": "responseTemplates",
        "status_code": "statusCode",
    },
)
class CfnGatewayResponseProps:
    def __init__(
        self,
        *,
        response_type: builtins.str,
        rest_api_id: builtins.str,
        response_parameters: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        response_templates: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        status_code: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnGatewayResponse``.

        :param response_type: The response type of the associated GatewayResponse.
        :param rest_api_id: The string identifier of the associated RestApi.
        :param response_parameters: Response parameters (paths, query strings and headers) of the GatewayResponse as a string-to-string map of key-value pairs.
        :param response_templates: Response templates of the GatewayResponse as a string-to-string map of key-value pairs.
        :param status_code: The HTTP status code for this GatewayResponse.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_gateway_response_props = apigateway.CfnGatewayResponseProps(
                response_type="responseType",
                rest_api_id="restApiId",
            
                # the properties below are optional
                response_parameters={
                    "response_parameters_key": "responseParameters"
                },
                response_templates={
                    "response_templates_key": "responseTemplates"
                },
                status_code="statusCode"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0b43368d7d49c0d420e2923e567ff8c91d5c1f75d378bebb996f9a5b2c4e09b1)
            check_type(argname="argument response_type", value=response_type, expected_type=type_hints["response_type"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument response_parameters", value=response_parameters, expected_type=type_hints["response_parameters"])
            check_type(argname="argument response_templates", value=response_templates, expected_type=type_hints["response_templates"])
            check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "response_type": response_type,
            "rest_api_id": rest_api_id,
        }
        if response_parameters is not None:
            self._values["response_parameters"] = response_parameters
        if response_templates is not None:
            self._values["response_templates"] = response_templates
        if status_code is not None:
            self._values["status_code"] = status_code

    @builtins.property
    def response_type(self) -> builtins.str:
        '''The response type of the associated GatewayResponse.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-responsetype
        '''
        result = self._values.get("response_type")
        assert result is not None, "Required property 'response_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def response_parameters(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''Response parameters (paths, query strings and headers) of the GatewayResponse as a string-to-string map of key-value pairs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-responseparameters
        '''
        result = self._values.get("response_parameters")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

    @builtins.property
    def response_templates(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''Response templates of the GatewayResponse as a string-to-string map of key-value pairs.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-responsetemplates
        '''
        result = self._values.get("response_templates")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

    @builtins.property
    def status_code(self) -> typing.Optional[builtins.str]:
        '''The HTTP status code for this GatewayResponse.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-gatewayresponse.html#cfn-apigateway-gatewayresponse-statuscode
        '''
        result = self._values.get("status_code")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGatewayResponseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnMethod(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnMethod",
):
    '''The ``AWS::ApiGateway::Method`` resource creates API Gateway methods that define the parameters and body that clients must send in their requests.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html
    :cloudformationResource: AWS::ApiGateway::Method
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_method = apigateway.CfnMethod(self, "MyCfnMethod",
            http_method="httpMethod",
            resource_id="resourceId",
            rest_api_id="restApiId",
        
            # the properties below are optional
            api_key_required=False,
            authorization_scopes=["authorizationScopes"],
            authorization_type="authorizationType",
            authorizer_id="authorizerId",
            integration=apigateway.CfnMethod.IntegrationProperty(
                type="type",
        
                # the properties below are optional
                cache_key_parameters=["cacheKeyParameters"],
                cache_namespace="cacheNamespace",
                connection_id="connectionId",
                connection_type="connectionType",
                content_handling="contentHandling",
                credentials="credentials",
                integration_http_method="integrationHttpMethod",
                integration_responses=[apigateway.CfnMethod.IntegrationResponseProperty(
                    status_code="statusCode",
        
                    # the properties below are optional
                    content_handling="contentHandling",
                    response_parameters={
                        "response_parameters_key": "responseParameters"
                    },
                    response_templates={
                        "response_templates_key": "responseTemplates"
                    },
                    selection_pattern="selectionPattern"
                )],
                passthrough_behavior="passthroughBehavior",
                request_parameters={
                    "request_parameters_key": "requestParameters"
                },
                request_templates={
                    "request_templates_key": "requestTemplates"
                },
                timeout_in_millis=123,
                uri="uri"
            ),
            method_responses=[apigateway.CfnMethod.MethodResponseProperty(
                status_code="statusCode",
        
                # the properties below are optional
                response_models={
                    "response_models_key": "responseModels"
                },
                response_parameters={
                    "response_parameters_key": False
                }
            )],
            operation_name="operationName",
            request_models={
                "request_models_key": "requestModels"
            },
            request_parameters={
                "request_parameters_key": False
            },
            request_validator_id="requestValidatorId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        http_method: builtins.str,
        resource_id: builtins.str,
        rest_api_id: builtins.str,
        api_key_required: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        authorization_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        authorization_type: typing.Optional[builtins.str] = None,
        authorizer_id: typing.Optional[builtins.str] = None,
        integration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnMethod.IntegrationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        method_responses: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnMethod.MethodResponseProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        operation_name: typing.Optional[builtins.str] = None,
        request_models: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        request_parameters: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_da3f097b]]]] = None,
        request_validator_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param http_method: The method's HTTP verb.
        :param resource_id: The Resource identifier for the MethodResponse resource.
        :param rest_api_id: The string identifier of the associated RestApi.
        :param api_key_required: A boolean flag specifying whether a valid ApiKey is required to invoke this method.
        :param authorization_scopes: A list of authorization scopes configured on the method. The scopes are used with a ``COGNITO_USER_POOLS`` authorizer to authorize the method invocation. The authorization works by matching the method scopes against the scopes parsed from the access token in the incoming request. The method invocation is authorized if any method scopes matches a claimed scope in the access token. Otherwise, the invocation is not authorized. When the method scope is configured, the client must provide an access token instead of an identity token for authorization purposes.
        :param authorization_type: The method's authorization type. This parameter is required. For valid values, see `Method <https://docs.aws.amazon.com/apigateway/latest/api/API_Method.html>`_ in the *API Gateway API Reference* . .. epigraph:: If you specify the ``AuthorizerId`` property, specify ``CUSTOM`` or ``COGNITO_USER_POOLS`` for this property.
        :param authorizer_id: The identifier of an authorizer to use on this method. The method's authorization type must be ``CUSTOM`` or ``COGNITO_USER_POOLS`` .
        :param integration: Represents an ``HTTP`` , ``HTTP_PROXY`` , ``AWS`` , ``AWS_PROXY`` , or Mock integration.
        :param method_responses: Gets a method response associated with a given HTTP status code.
        :param operation_name: A human-friendly operation identifier for the method. For example, you can assign the ``operationName`` of ``ListPets`` for the ``GET /pets`` method in the ``PetStore`` example.
        :param request_models: A key-value map specifying data schemas, represented by Model resources, (as the mapped value) of the request payloads of given content types (as the mapping key).
        :param request_parameters: A key-value map defining required or optional method request parameters that can be accepted by API Gateway. A key is a method request parameter name matching the pattern of ``method.request.{location}.{name}`` , where ``location`` is ``querystring`` , ``path`` , or ``header`` and ``name`` is a valid and unique parameter name. The value associated with the key is a Boolean flag indicating whether the parameter is required ( ``true`` ) or optional ( ``false`` ). The method request parameter names defined here are available in Integration to be mapped to integration request parameters or templates.
        :param request_validator_id: The identifier of a RequestValidator for request validation.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__edfd42d04d09c9ea72d15a6173bfaa68d17ac55c9faa39a502d4e20680006350)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnMethodProps(
            http_method=http_method,
            resource_id=resource_id,
            rest_api_id=rest_api_id,
            api_key_required=api_key_required,
            authorization_scopes=authorization_scopes,
            authorization_type=authorization_type,
            authorizer_id=authorizer_id,
            integration=integration,
            method_responses=method_responses,
            operation_name=operation_name,
            request_models=request_models,
            request_parameters=request_parameters,
            request_validator_id=request_validator_id,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f218ab18a16f7118637cc246823f06b2ec05e9de15b8e1bb3e95a82e05ac462c)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5da6b3d3efbce640b28b8ac42a3898befe0e97536186c708a66ad9d019a6ced6)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="httpMethod")
    def http_method(self) -> builtins.str:
        '''The method's HTTP verb.'''
        return typing.cast(builtins.str, jsii.get(self, "httpMethod"))

    @http_method.setter
    def http_method(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0ccc0188b6af57e2d38ba49462e0896f43d6bfa3ce996c0931251e6dde7040e7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "httpMethod", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="resourceId")
    def resource_id(self) -> builtins.str:
        '''The Resource identifier for the MethodResponse resource.'''
        return typing.cast(builtins.str, jsii.get(self, "resourceId"))

    @resource_id.setter
    def resource_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9fc44d4f5a49a6bdbead9d0390fd73c1a98a79b959bfff89d0112bae8d79fcd3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "resourceId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.'''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3d518e2908644bfeb3c189e7562790cea74e865b221a7053b7fdf9424c4a0a0c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="apiKeyRequired")
    def api_key_required(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A boolean flag specifying whether a valid ApiKey is required to invoke this method.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "apiKeyRequired"))

    @api_key_required.setter
    def api_key_required(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e69e781a2c7e540d207896f94419e3f760ed4c29f9ce3139745233fa6b7981c2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiKeyRequired", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="authorizationScopes")
    def authorization_scopes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of authorization scopes configured on the method.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "authorizationScopes"))

    @authorization_scopes.setter
    def authorization_scopes(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__296d6e993d97eff8859d09bf17c367074b201ed0488160306394b41fbd4be75a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizationScopes", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="authorizationType")
    def authorization_type(self) -> typing.Optional[builtins.str]:
        '''The method's authorization type.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizationType"))

    @authorization_type.setter
    def authorization_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d418c390ea74492805832b8071bfb30cf855e1d0fa881f6aafb287e62805969)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizationType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="authorizerId")
    def authorizer_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of an authorizer to use on this method.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizerId"))

    @authorizer_id.setter
    def authorizer_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1c3293cf517356146f021e82dda7af894961e39437a6857f08ef4cefefe58f39)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "authorizerId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="integration")
    def integration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnMethod.IntegrationProperty"]]:
        '''Represents an ``HTTP`` , ``HTTP_PROXY`` , ``AWS`` , ``AWS_PROXY`` , or Mock integration.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnMethod.IntegrationProperty"]], jsii.get(self, "integration"))

    @integration.setter
    def integration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnMethod.IntegrationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9045db34a991c4e4d9f8cd01b69765ec28cd43ce199d57d60ca42a869e0c8b2f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "integration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="methodResponses")
    def method_responses(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnMethod.MethodResponseProperty"]]]]:
        '''Gets a method response associated with a given HTTP status code.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnMethod.MethodResponseProperty"]]]], jsii.get(self, "methodResponses"))

    @method_responses.setter
    def method_responses(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnMethod.MethodResponseProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f4cd1cd1b53b103d49142d6ce4ddb3ed1e1c07e1d12b9f7f674058a0bf66dbfc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "methodResponses", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="operationName")
    def operation_name(self) -> typing.Optional[builtins.str]:
        '''A human-friendly operation identifier for the method.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "operationName"))

    @operation_name.setter
    def operation_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b0bad81389b4aa3617129e1c3ecfcf0fc9465d4c338f9939facff2be4b99fa89)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "operationName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="requestModels")
    def request_models(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''A key-value map specifying data schemas, represented by Model resources, (as the mapped value) of the request payloads of given content types (as the mapping key).'''
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], jsii.get(self, "requestModels"))

    @request_models.setter
    def request_models(
        self,
        value: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9410b580162450075529d443a9b6c428be7f00e0ec91767954a58ac8cd280749)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requestModels", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="requestParameters")
    def request_parameters(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_da3f097b]]]]:
        '''A key-value map defining required or optional method request parameters that can be accepted by API Gateway.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_da3f097b]]]], jsii.get(self, "requestParameters"))

    @request_parameters.setter
    def request_parameters(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_da3f097b]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__872f1b72145d5ceb544a38e2fd82115857a21400a026e9830b7b2459e6e4b0b3)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requestParameters", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="requestValidatorId")
    def request_validator_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of a RequestValidator for request validation.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "requestValidatorId"))

    @request_validator_id.setter
    def request_validator_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6b90156b429c04a252c729c4a8a40f9ba7225826fe83f21d8874bd11326677ba)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "requestValidatorId", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnMethod.IntegrationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "type": "type",
            "cache_key_parameters": "cacheKeyParameters",
            "cache_namespace": "cacheNamespace",
            "connection_id": "connectionId",
            "connection_type": "connectionType",
            "content_handling": "contentHandling",
            "credentials": "credentials",
            "integration_http_method": "integrationHttpMethod",
            "integration_responses": "integrationResponses",
            "passthrough_behavior": "passthroughBehavior",
            "request_parameters": "requestParameters",
            "request_templates": "requestTemplates",
            "timeout_in_millis": "timeoutInMillis",
            "uri": "uri",
        },
    )
    class IntegrationProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            cache_key_parameters: typing.Optional[typing.Sequence[builtins.str]] = None,
            cache_namespace: typing.Optional[builtins.str] = None,
            connection_id: typing.Optional[builtins.str] = None,
            connection_type: typing.Optional[builtins.str] = None,
            content_handling: typing.Optional[builtins.str] = None,
            credentials: typing.Optional[builtins.str] = None,
            integration_http_method: typing.Optional[builtins.str] = None,
            integration_responses: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnMethod.IntegrationResponseProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            passthrough_behavior: typing.Optional[builtins.str] = None,
            request_parameters: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
            request_templates: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
            timeout_in_millis: typing.Optional[jsii.Number] = None,
            uri: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``Integration`` is a property of the `AWS::ApiGateway::Method <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html>`_ resource that specifies information about the target backend that a method calls.

            :param type: Specifies an API method integration type. The valid value is one of the following:. For the HTTP and HTTP proxy integrations, each integration can specify a protocol ( ``http/https`` ), port and path. Standard 80 and 443 ports are supported as well as custom ports above 1024. An HTTP or HTTP proxy integration with a ``connectionType`` of ``VPC_LINK`` is referred to as a private integration and uses a VpcLink to connect API Gateway to a network load balancer of a VPC.
            :param cache_key_parameters: A list of request parameters whose values API Gateway caches. To be valid values for ``cacheKeyParameters`` , these parameters must also be specified for Method ``requestParameters`` .
            :param cache_namespace: Specifies a group of related cached parameters. By default, API Gateway uses the resource ID as the ``cacheNamespace`` . You can specify the same ``cacheNamespace`` across resources to return the same cached data for requests to different resources.
            :param connection_id: The ID of the VpcLink used for the integration when ``connectionType=VPC_LINK`` and undefined, otherwise.
            :param connection_type: The type of the network connection to the integration endpoint. The valid value is ``INTERNET`` for connections through the public routable internet or ``VPC_LINK`` for private connections between API Gateway and a network load balancer in a VPC. The default value is ``INTERNET`` .
            :param content_handling: Specifies how to handle request payload content type conversions. Supported values are ``CONVERT_TO_BINARY`` and ``CONVERT_TO_TEXT`` , with the following behaviors: If this property is not defined, the request payload will be passed through from the method request to integration request without modification, provided that the ``passthroughBehavior`` is configured to support payload pass-through.
            :param credentials: Specifies the credentials required for the integration, if any. For AWS integrations, three options are available. To specify an IAM Role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To require that the caller's identity be passed through from the request, specify the string ``arn:aws:iam::\\*:user/\\*`` . To use resource-based permissions on supported AWS services, specify null.
            :param integration_http_method: Specifies the integration's HTTP method type. For the Type property, if you specify ``MOCK`` , this property is optional. For Lambda integrations, you must set the integration method to ``POST`` . For all other types, you must specify this property.
            :param integration_responses: Specifies the integration's responses.
            :param passthrough_behavior: Specifies how the method request body of an unmapped content type will be passed through the integration request to the back end without transformation. A content type is unmapped if no mapping template is defined in the integration or the content type does not match any of the mapped content types, as specified in ``requestTemplates`` . The valid value is one of the following: ``WHEN_NO_MATCH`` : passes the method request body through the integration request to the back end without transformation when the method request content type does not match any content type associated with the mapping templates defined in the integration request. ``WHEN_NO_TEMPLATES`` : passes the method request body through the integration request to the back end without transformation when no mapping template is defined in the integration request. If a template is defined when this option is selected, the method request of an unmapped content-type will be rejected with an HTTP 415 Unsupported Media Type response. ``NEVER`` : rejects the method request with an HTTP 415 Unsupported Media Type response when either the method request content type does not match any content type associated with the mapping templates defined in the integration request or no mapping template is defined in the integration request.
            :param request_parameters: A key-value map specifying request parameters that are passed from the method request to the back end. The key is an integration request parameter name and the associated value is a method request parameter value or static value that must be enclosed within single quotes and pre-encoded as required by the back end. The method request parameter value must match the pattern of ``method.request.{location}.{name}`` , where ``location`` is ``querystring`` , ``path`` , or ``header`` and ``name`` must be a valid and unique method request parameter name.
            :param request_templates: Represents a map of Velocity templates that are applied on the request payload based on the value of the Content-Type header sent by the client. The content type value is the key in this map, and the template (as a String) is the value.
            :param timeout_in_millis: Custom timeout between 50 and 29,000 milliseconds. The default value is 29,000 milliseconds or 29 seconds.
            :param uri: Specifies Uniform Resource Identifier (URI) of the integration endpoint. For ``HTTP`` or ``HTTP_PROXY`` integrations, the URI must be a fully formed, encoded HTTP(S) URL according to the RFC-3986 specification for standard integrations. If ``connectionType`` is ``VPC_LINK`` specify the Network Load Balancer DNS name. For ``AWS`` or ``AWS_PROXY`` integrations, the URI is of the form ``arn:aws:apigateway:{region}:{subdomain.service|service}:path|action/{service_api}`` . Here, {Region} is the API Gateway region (e.g., us-east-1); {service} is the name of the integrated AWS service (e.g., s3); and {subdomain} is a designated subdomain supported by certain AWS service for fast host-name lookup. action can be used for an AWS service action-based API, using an Action={name}&{p1}={v1}&p2={v2}... query string. The ensuing {service_api} refers to a supported action {name} plus any required input parameters. Alternatively, path can be used for an AWS service path-based API. The ensuing service_api refers to the path to an AWS service resource, including the region of the integrated AWS service, if applicable. For example, for integration with the S3 API of GetObject, the uri can be either ``arn:aws:apigateway:us-west-2:s3:action/GetObject&Bucket={bucket}&Key={key}`` or ``arn:aws:apigateway:us-west-2:s3:path/{bucket}/{key}``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                integration_property = apigateway.CfnMethod.IntegrationProperty(
                    type="type",
                
                    # the properties below are optional
                    cache_key_parameters=["cacheKeyParameters"],
                    cache_namespace="cacheNamespace",
                    connection_id="connectionId",
                    connection_type="connectionType",
                    content_handling="contentHandling",
                    credentials="credentials",
                    integration_http_method="integrationHttpMethod",
                    integration_responses=[apigateway.CfnMethod.IntegrationResponseProperty(
                        status_code="statusCode",
                
                        # the properties below are optional
                        content_handling="contentHandling",
                        response_parameters={
                            "response_parameters_key": "responseParameters"
                        },
                        response_templates={
                            "response_templates_key": "responseTemplates"
                        },
                        selection_pattern="selectionPattern"
                    )],
                    passthrough_behavior="passthroughBehavior",
                    request_parameters={
                        "request_parameters_key": "requestParameters"
                    },
                    request_templates={
                        "request_templates_key": "requestTemplates"
                    },
                    timeout_in_millis=123,
                    uri="uri"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7fc2a1b31e9e3f92210ced50c17b0bf1cfcdb472f8b61fcc31a3c633be7ced4c)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument cache_key_parameters", value=cache_key_parameters, expected_type=type_hints["cache_key_parameters"])
                check_type(argname="argument cache_namespace", value=cache_namespace, expected_type=type_hints["cache_namespace"])
                check_type(argname="argument connection_id", value=connection_id, expected_type=type_hints["connection_id"])
                check_type(argname="argument connection_type", value=connection_type, expected_type=type_hints["connection_type"])
                check_type(argname="argument content_handling", value=content_handling, expected_type=type_hints["content_handling"])
                check_type(argname="argument credentials", value=credentials, expected_type=type_hints["credentials"])
                check_type(argname="argument integration_http_method", value=integration_http_method, expected_type=type_hints["integration_http_method"])
                check_type(argname="argument integration_responses", value=integration_responses, expected_type=type_hints["integration_responses"])
                check_type(argname="argument passthrough_behavior", value=passthrough_behavior, expected_type=type_hints["passthrough_behavior"])
                check_type(argname="argument request_parameters", value=request_parameters, expected_type=type_hints["request_parameters"])
                check_type(argname="argument request_templates", value=request_templates, expected_type=type_hints["request_templates"])
                check_type(argname="argument timeout_in_millis", value=timeout_in_millis, expected_type=type_hints["timeout_in_millis"])
                check_type(argname="argument uri", value=uri, expected_type=type_hints["uri"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if cache_key_parameters is not None:
                self._values["cache_key_parameters"] = cache_key_parameters
            if cache_namespace is not None:
                self._values["cache_namespace"] = cache_namespace
            if connection_id is not None:
                self._values["connection_id"] = connection_id
            if connection_type is not None:
                self._values["connection_type"] = connection_type
            if content_handling is not None:
                self._values["content_handling"] = content_handling
            if credentials is not None:
                self._values["credentials"] = credentials
            if integration_http_method is not None:
                self._values["integration_http_method"] = integration_http_method
            if integration_responses is not None:
                self._values["integration_responses"] = integration_responses
            if passthrough_behavior is not None:
                self._values["passthrough_behavior"] = passthrough_behavior
            if request_parameters is not None:
                self._values["request_parameters"] = request_parameters
            if request_templates is not None:
                self._values["request_templates"] = request_templates
            if timeout_in_millis is not None:
                self._values["timeout_in_millis"] = timeout_in_millis
            if uri is not None:
                self._values["uri"] = uri

        @builtins.property
        def type(self) -> builtins.str:
            '''Specifies an API method integration type. The valid value is one of the following:.

            For the HTTP and HTTP proxy integrations, each integration can specify a protocol ( ``http/https`` ), port and path. Standard 80 and 443 ports are supported as well as custom ports above 1024. An HTTP or HTTP proxy integration with a ``connectionType`` of ``VPC_LINK`` is referred to as a private integration and uses a VpcLink to connect API Gateway to a network load balancer of a VPC.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def cache_key_parameters(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of request parameters whose values API Gateway caches.

            To be valid values for ``cacheKeyParameters`` , these parameters must also be specified for Method ``requestParameters`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-cachekeyparameters
            '''
            result = self._values.get("cache_key_parameters")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def cache_namespace(self) -> typing.Optional[builtins.str]:
            '''Specifies a group of related cached parameters.

            By default, API Gateway uses the resource ID as the ``cacheNamespace`` . You can specify the same ``cacheNamespace`` across resources to return the same cached data for requests to different resources.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-cachenamespace
            '''
            result = self._values.get("cache_namespace")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def connection_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the VpcLink used for the integration when ``connectionType=VPC_LINK`` and undefined, otherwise.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-connectionid
            '''
            result = self._values.get("connection_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def connection_type(self) -> typing.Optional[builtins.str]:
            '''The type of the network connection to the integration endpoint.

            The valid value is ``INTERNET`` for connections through the public routable internet or ``VPC_LINK`` for private connections between API Gateway and a network load balancer in a VPC. The default value is ``INTERNET`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-connectiontype
            '''
            result = self._values.get("connection_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def content_handling(self) -> typing.Optional[builtins.str]:
            '''Specifies how to handle request payload content type conversions.

            Supported values are ``CONVERT_TO_BINARY`` and ``CONVERT_TO_TEXT`` , with the following behaviors:

            If this property is not defined, the request payload will be passed through from the method request to integration request without modification, provided that the ``passthroughBehavior`` is configured to support payload pass-through.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-contenthandling
            '''
            result = self._values.get("content_handling")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def credentials(self) -> typing.Optional[builtins.str]:
            '''Specifies the credentials required for the integration, if any.

            For AWS integrations, three options are available. To specify an IAM Role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To require that the caller's identity be passed through from the request, specify the string ``arn:aws:iam::\\*:user/\\*`` . To use resource-based permissions on supported AWS services, specify null.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-credentials
            '''
            result = self._values.get("credentials")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def integration_http_method(self) -> typing.Optional[builtins.str]:
            '''Specifies the integration's HTTP method type.

            For the Type property, if you specify ``MOCK`` , this property is optional. For Lambda integrations, you must set the integration method to ``POST`` . For all other types, you must specify this property.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-integrationhttpmethod
            '''
            result = self._values.get("integration_http_method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def integration_responses(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnMethod.IntegrationResponseProperty"]]]]:
            '''Specifies the integration's responses.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-integrationresponses
            '''
            result = self._values.get("integration_responses")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnMethod.IntegrationResponseProperty"]]]], result)

        @builtins.property
        def passthrough_behavior(self) -> typing.Optional[builtins.str]:
            '''Specifies how the method request body of an unmapped content type will be passed through the integration request to the back end without transformation.

            A content type is unmapped if no mapping template is defined in the integration or the content type does not match any of the mapped content types, as specified in ``requestTemplates`` . The valid value is one of the following: ``WHEN_NO_MATCH`` : passes the method request body through the integration request to the back end without transformation when the method request content type does not match any content type associated with the mapping templates defined in the integration request. ``WHEN_NO_TEMPLATES`` : passes the method request body through the integration request to the back end without transformation when no mapping template is defined in the integration request. If a template is defined when this option is selected, the method request of an unmapped content-type will be rejected with an HTTP 415 Unsupported Media Type response. ``NEVER`` : rejects the method request with an HTTP 415 Unsupported Media Type response when either the method request content type does not match any content type associated with the mapping templates defined in the integration request or no mapping template is defined in the integration request.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-passthroughbehavior
            '''
            result = self._values.get("passthrough_behavior")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def request_parameters(
            self,
        ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
            '''A key-value map specifying request parameters that are passed from the method request to the back end.

            The key is an integration request parameter name and the associated value is a method request parameter value or static value that must be enclosed within single quotes and pre-encoded as required by the back end. The method request parameter value must match the pattern of ``method.request.{location}.{name}`` , where ``location`` is ``querystring`` , ``path`` , or ``header`` and ``name`` must be a valid and unique method request parameter name.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-requestparameters
            '''
            result = self._values.get("request_parameters")
            return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

        @builtins.property
        def request_templates(
            self,
        ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
            '''Represents a map of Velocity templates that are applied on the request payload based on the value of the Content-Type header sent by the client.

            The content type value is the key in this map, and the template (as a String) is the value.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-requesttemplates
            '''
            result = self._values.get("request_templates")
            return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

        @builtins.property
        def timeout_in_millis(self) -> typing.Optional[jsii.Number]:
            '''Custom timeout between 50 and 29,000 milliseconds.

            The default value is 29,000 milliseconds or 29 seconds.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-timeoutinmillis
            '''
            result = self._values.get("timeout_in_millis")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def uri(self) -> typing.Optional[builtins.str]:
            '''Specifies Uniform Resource Identifier (URI) of the integration endpoint.

            For ``HTTP`` or ``HTTP_PROXY`` integrations, the URI must be a fully formed, encoded HTTP(S) URL according to the RFC-3986 specification for standard integrations. If ``connectionType`` is ``VPC_LINK`` specify the Network Load Balancer DNS name. For ``AWS`` or ``AWS_PROXY`` integrations, the URI is of the form ``arn:aws:apigateway:{region}:{subdomain.service|service}:path|action/{service_api}`` . Here, {Region} is the API Gateway region (e.g., us-east-1); {service} is the name of the integrated AWS service (e.g., s3); and {subdomain} is a designated subdomain supported by certain AWS service for fast host-name lookup. action can be used for an AWS service action-based API, using an Action={name}&{p1}={v1}&p2={v2}... query string. The ensuing {service_api} refers to a supported action {name} plus any required input parameters. Alternatively, path can be used for an AWS service path-based API. The ensuing service_api refers to the path to an AWS service resource, including the region of the integrated AWS service, if applicable. For example, for integration with the S3 API of GetObject, the uri can be either ``arn:aws:apigateway:us-west-2:s3:action/GetObject&Bucket={bucket}&Key={key}`` or ``arn:aws:apigateway:us-west-2:s3:path/{bucket}/{key}``

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integration.html#cfn-apigateway-method-integration-uri
            '''
            result = self._values.get("uri")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IntegrationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnMethod.IntegrationResponseProperty",
        jsii_struct_bases=[],
        name_mapping={
            "status_code": "statusCode",
            "content_handling": "contentHandling",
            "response_parameters": "responseParameters",
            "response_templates": "responseTemplates",
            "selection_pattern": "selectionPattern",
        },
    )
    class IntegrationResponseProperty:
        def __init__(
            self,
            *,
            status_code: builtins.str,
            content_handling: typing.Optional[builtins.str] = None,
            response_parameters: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
            response_templates: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
            selection_pattern: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``IntegrationResponse`` is a property of the `Amazon API Gateway Method Integration <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apitgateway-method-integration.html>`_ property type that specifies the response that API Gateway sends after a method's backend finishes processing a request.

            :param status_code: Specifies the status code that is used to map the integration response to an existing MethodResponse.
            :param content_handling: Specifies how to handle response payload content type conversions. Supported values are ``CONVERT_TO_BINARY`` and ``CONVERT_TO_TEXT`` , with the following behaviors: If this property is not defined, the response payload will be passed through from the integration response to the method response without modification.
            :param response_parameters: A key-value map specifying response parameters that are passed to the method response from the back end. The key is a method response header parameter name and the mapped value is an integration response header value, a static value enclosed within a pair of single quotes, or a JSON expression from the integration response body. The mapping key must match the pattern of ``method.response.header.{name}`` , where ``name`` is a valid and unique header name. The mapped non-static value must match the pattern of ``integration.response.header.{name}`` or ``integration.response.body.{JSON-expression}`` , where ``name`` is a valid and unique response header name and ``JSON-expression`` is a valid JSON expression without the ``$`` prefix.
            :param response_templates: Specifies the templates used to transform the integration response body. Response templates are represented as a key/value map, with a content-type as the key and a template as the value.
            :param selection_pattern: Specifies the regular expression (regex) pattern used to choose an integration response based on the response from the back end. For example, if the success response returns nothing and the error response returns some string, you could use the ``.+`` regex to match error response. However, make sure that the error response does not contain any newline ( ``\\n`` ) character in such cases. If the back end is an AWS Lambda function, the AWS Lambda function error header is matched. For all other HTTP and AWS back ends, the HTTP status code is matched.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integrationresponse.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                integration_response_property = apigateway.CfnMethod.IntegrationResponseProperty(
                    status_code="statusCode",
                
                    # the properties below are optional
                    content_handling="contentHandling",
                    response_parameters={
                        "response_parameters_key": "responseParameters"
                    },
                    response_templates={
                        "response_templates_key": "responseTemplates"
                    },
                    selection_pattern="selectionPattern"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__32f4c37443968f90a24ccb70ad18cc6b3b2dc77146de035d2a301aab38b9b78e)
                check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
                check_type(argname="argument content_handling", value=content_handling, expected_type=type_hints["content_handling"])
                check_type(argname="argument response_parameters", value=response_parameters, expected_type=type_hints["response_parameters"])
                check_type(argname="argument response_templates", value=response_templates, expected_type=type_hints["response_templates"])
                check_type(argname="argument selection_pattern", value=selection_pattern, expected_type=type_hints["selection_pattern"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "status_code": status_code,
            }
            if content_handling is not None:
                self._values["content_handling"] = content_handling
            if response_parameters is not None:
                self._values["response_parameters"] = response_parameters
            if response_templates is not None:
                self._values["response_templates"] = response_templates
            if selection_pattern is not None:
                self._values["selection_pattern"] = selection_pattern

        @builtins.property
        def status_code(self) -> builtins.str:
            '''Specifies the status code that is used to map the integration response to an existing MethodResponse.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integrationresponse.html#cfn-apigateway-method-integrationresponse-statuscode
            '''
            result = self._values.get("status_code")
            assert result is not None, "Required property 'status_code' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def content_handling(self) -> typing.Optional[builtins.str]:
            '''Specifies how to handle response payload content type conversions.

            Supported values are ``CONVERT_TO_BINARY`` and ``CONVERT_TO_TEXT`` , with the following behaviors:

            If this property is not defined, the response payload will be passed through from the integration response to the method response without modification.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integrationresponse.html#cfn-apigateway-method-integrationresponse-contenthandling
            '''
            result = self._values.get("content_handling")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def response_parameters(
            self,
        ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
            '''A key-value map specifying response parameters that are passed to the method response from the back end.

            The key is a method response header parameter name and the mapped value is an integration response header value, a static value enclosed within a pair of single quotes, or a JSON expression from the integration response body. The mapping key must match the pattern of ``method.response.header.{name}`` , where ``name`` is a valid and unique header name. The mapped non-static value must match the pattern of ``integration.response.header.{name}`` or ``integration.response.body.{JSON-expression}`` , where ``name`` is a valid and unique response header name and ``JSON-expression`` is a valid JSON expression without the ``$`` prefix.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integrationresponse.html#cfn-apigateway-method-integrationresponse-responseparameters
            '''
            result = self._values.get("response_parameters")
            return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

        @builtins.property
        def response_templates(
            self,
        ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
            '''Specifies the templates used to transform the integration response body.

            Response templates are represented as a key/value map, with a content-type as the key and a template as the value.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integrationresponse.html#cfn-apigateway-method-integrationresponse-responsetemplates
            '''
            result = self._values.get("response_templates")
            return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

        @builtins.property
        def selection_pattern(self) -> typing.Optional[builtins.str]:
            '''Specifies the regular expression (regex) pattern used to choose an integration response based on the response from the back end.

            For example, if the success response returns nothing and the error response returns some string, you could use the ``.+`` regex to match error response. However, make sure that the error response does not contain any newline ( ``\\n`` ) character in such cases. If the back end is an AWS Lambda function, the AWS Lambda function error header is matched. For all other HTTP and AWS back ends, the HTTP status code is matched.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-integrationresponse.html#cfn-apigateway-method-integrationresponse-selectionpattern
            '''
            result = self._values.get("selection_pattern")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IntegrationResponseProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnMethod.MethodResponseProperty",
        jsii_struct_bases=[],
        name_mapping={
            "status_code": "statusCode",
            "response_models": "responseModels",
            "response_parameters": "responseParameters",
        },
    )
    class MethodResponseProperty:
        def __init__(
            self,
            *,
            status_code: builtins.str,
            response_models: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
            response_parameters: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_da3f097b]]]] = None,
        ) -> None:
            '''Represents a method response of a given HTTP status code returned to the client.

            The method response is passed from the back end through the associated integration response that can be transformed using a mapping template.

            :param status_code: The method response's status code.
            :param response_models: Specifies the Model resources used for the response's content-type. Response models are represented as a key/value map, with a content-type as the key and a Model name as the value.
            :param response_parameters: A key-value map specifying required or optional response parameters that API Gateway can send back to the caller. A key defines a method response header and the value specifies whether the associated method response header is required or not. The expression of the key must match the pattern ``method.response.header.{name}`` , where ``name`` is a valid and unique header name. API Gateway passes certain integration response data to the method response headers specified here according to the mapping you prescribe in the API's IntegrationResponse. The integration response data that can be mapped include an integration response header expressed in ``integration.response.header.{name}`` , a static value enclosed within a pair of single quotes (e.g., ``'application/json'`` ), or a JSON expression from the back-end response payload in the form of ``integration.response.body.{JSON-expression}`` , where ``JSON-expression`` is a valid JSON expression without the ``$`` prefix.)

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-methodresponse.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                method_response_property = apigateway.CfnMethod.MethodResponseProperty(
                    status_code="statusCode",
                
                    # the properties below are optional
                    response_models={
                        "response_models_key": "responseModels"
                    },
                    response_parameters={
                        "response_parameters_key": False
                    }
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ff600e11d4e4158654132cbc6f3bb60aa2dc5f8337cc3f14f4a0f6be1090a477)
                check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
                check_type(argname="argument response_models", value=response_models, expected_type=type_hints["response_models"])
                check_type(argname="argument response_parameters", value=response_parameters, expected_type=type_hints["response_parameters"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "status_code": status_code,
            }
            if response_models is not None:
                self._values["response_models"] = response_models
            if response_parameters is not None:
                self._values["response_parameters"] = response_parameters

        @builtins.property
        def status_code(self) -> builtins.str:
            '''The method response's status code.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-methodresponse.html#cfn-apigateway-method-methodresponse-statuscode
            '''
            result = self._values.get("status_code")
            assert result is not None, "Required property 'status_code' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def response_models(
            self,
        ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
            '''Specifies the Model resources used for the response's content-type.

            Response models are represented as a key/value map, with a content-type as the key and a Model name as the value.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-methodresponse.html#cfn-apigateway-method-methodresponse-responsemodels
            '''
            result = self._values.get("response_models")
            return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

        @builtins.property
        def response_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_da3f097b]]]]:
            '''A key-value map specifying required or optional response parameters that API Gateway can send back to the caller.

            A key defines a method response header and the value specifies whether the associated method response header is required or not. The expression of the key must match the pattern ``method.response.header.{name}`` , where ``name`` is a valid and unique header name. API Gateway passes certain integration response data to the method response headers specified here according to the mapping you prescribe in the API's IntegrationResponse. The integration response data that can be mapped include an integration response header expressed in ``integration.response.header.{name}`` , a static value enclosed within a pair of single quotes (e.g., ``'application/json'`` ), or a JSON expression from the back-end response payload in the form of ``integration.response.body.{JSON-expression}`` , where ``JSON-expression`` is a valid JSON expression without the ``$`` prefix.)

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-method-methodresponse.html#cfn-apigateway-method-methodresponse-responseparameters
            '''
            result = self._values.get("response_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_da3f097b]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MethodResponseProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnMethodProps",
    jsii_struct_bases=[],
    name_mapping={
        "http_method": "httpMethod",
        "resource_id": "resourceId",
        "rest_api_id": "restApiId",
        "api_key_required": "apiKeyRequired",
        "authorization_scopes": "authorizationScopes",
        "authorization_type": "authorizationType",
        "authorizer_id": "authorizerId",
        "integration": "integration",
        "method_responses": "methodResponses",
        "operation_name": "operationName",
        "request_models": "requestModels",
        "request_parameters": "requestParameters",
        "request_validator_id": "requestValidatorId",
    },
)
class CfnMethodProps:
    def __init__(
        self,
        *,
        http_method: builtins.str,
        resource_id: builtins.str,
        rest_api_id: builtins.str,
        api_key_required: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        authorization_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        authorization_type: typing.Optional[builtins.str] = None,
        authorizer_id: typing.Optional[builtins.str] = None,
        integration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnMethod.IntegrationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        method_responses: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnMethod.MethodResponseProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        operation_name: typing.Optional[builtins.str] = None,
        request_models: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        request_parameters: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_da3f097b]]]] = None,
        request_validator_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnMethod``.

        :param http_method: The method's HTTP verb.
        :param resource_id: The Resource identifier for the MethodResponse resource.
        :param rest_api_id: The string identifier of the associated RestApi.
        :param api_key_required: A boolean flag specifying whether a valid ApiKey is required to invoke this method.
        :param authorization_scopes: A list of authorization scopes configured on the method. The scopes are used with a ``COGNITO_USER_POOLS`` authorizer to authorize the method invocation. The authorization works by matching the method scopes against the scopes parsed from the access token in the incoming request. The method invocation is authorized if any method scopes matches a claimed scope in the access token. Otherwise, the invocation is not authorized. When the method scope is configured, the client must provide an access token instead of an identity token for authorization purposes.
        :param authorization_type: The method's authorization type. This parameter is required. For valid values, see `Method <https://docs.aws.amazon.com/apigateway/latest/api/API_Method.html>`_ in the *API Gateway API Reference* . .. epigraph:: If you specify the ``AuthorizerId`` property, specify ``CUSTOM`` or ``COGNITO_USER_POOLS`` for this property.
        :param authorizer_id: The identifier of an authorizer to use on this method. The method's authorization type must be ``CUSTOM`` or ``COGNITO_USER_POOLS`` .
        :param integration: Represents an ``HTTP`` , ``HTTP_PROXY`` , ``AWS`` , ``AWS_PROXY`` , or Mock integration.
        :param method_responses: Gets a method response associated with a given HTTP status code.
        :param operation_name: A human-friendly operation identifier for the method. For example, you can assign the ``operationName`` of ``ListPets`` for the ``GET /pets`` method in the ``PetStore`` example.
        :param request_models: A key-value map specifying data schemas, represented by Model resources, (as the mapped value) of the request payloads of given content types (as the mapping key).
        :param request_parameters: A key-value map defining required or optional method request parameters that can be accepted by API Gateway. A key is a method request parameter name matching the pattern of ``method.request.{location}.{name}`` , where ``location`` is ``querystring`` , ``path`` , or ``header`` and ``name`` is a valid and unique parameter name. The value associated with the key is a Boolean flag indicating whether the parameter is required ( ``true`` ) or optional ( ``false`` ). The method request parameter names defined here are available in Integration to be mapped to integration request parameters or templates.
        :param request_validator_id: The identifier of a RequestValidator for request validation.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_method_props = apigateway.CfnMethodProps(
                http_method="httpMethod",
                resource_id="resourceId",
                rest_api_id="restApiId",
            
                # the properties below are optional
                api_key_required=False,
                authorization_scopes=["authorizationScopes"],
                authorization_type="authorizationType",
                authorizer_id="authorizerId",
                integration=apigateway.CfnMethod.IntegrationProperty(
                    type="type",
            
                    # the properties below are optional
                    cache_key_parameters=["cacheKeyParameters"],
                    cache_namespace="cacheNamespace",
                    connection_id="connectionId",
                    connection_type="connectionType",
                    content_handling="contentHandling",
                    credentials="credentials",
                    integration_http_method="integrationHttpMethod",
                    integration_responses=[apigateway.CfnMethod.IntegrationResponseProperty(
                        status_code="statusCode",
            
                        # the properties below are optional
                        content_handling="contentHandling",
                        response_parameters={
                            "response_parameters_key": "responseParameters"
                        },
                        response_templates={
                            "response_templates_key": "responseTemplates"
                        },
                        selection_pattern="selectionPattern"
                    )],
                    passthrough_behavior="passthroughBehavior",
                    request_parameters={
                        "request_parameters_key": "requestParameters"
                    },
                    request_templates={
                        "request_templates_key": "requestTemplates"
                    },
                    timeout_in_millis=123,
                    uri="uri"
                ),
                method_responses=[apigateway.CfnMethod.MethodResponseProperty(
                    status_code="statusCode",
            
                    # the properties below are optional
                    response_models={
                        "response_models_key": "responseModels"
                    },
                    response_parameters={
                        "response_parameters_key": False
                    }
                )],
                operation_name="operationName",
                request_models={
                    "request_models_key": "requestModels"
                },
                request_parameters={
                    "request_parameters_key": False
                },
                request_validator_id="requestValidatorId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8eafa27300c8abdc46f7b0453098151ef3832bf587f9964be625f94aee2148bd)
            check_type(argname="argument http_method", value=http_method, expected_type=type_hints["http_method"])
            check_type(argname="argument resource_id", value=resource_id, expected_type=type_hints["resource_id"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument api_key_required", value=api_key_required, expected_type=type_hints["api_key_required"])
            check_type(argname="argument authorization_scopes", value=authorization_scopes, expected_type=type_hints["authorization_scopes"])
            check_type(argname="argument authorization_type", value=authorization_type, expected_type=type_hints["authorization_type"])
            check_type(argname="argument authorizer_id", value=authorizer_id, expected_type=type_hints["authorizer_id"])
            check_type(argname="argument integration", value=integration, expected_type=type_hints["integration"])
            check_type(argname="argument method_responses", value=method_responses, expected_type=type_hints["method_responses"])
            check_type(argname="argument operation_name", value=operation_name, expected_type=type_hints["operation_name"])
            check_type(argname="argument request_models", value=request_models, expected_type=type_hints["request_models"])
            check_type(argname="argument request_parameters", value=request_parameters, expected_type=type_hints["request_parameters"])
            check_type(argname="argument request_validator_id", value=request_validator_id, expected_type=type_hints["request_validator_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "http_method": http_method,
            "resource_id": resource_id,
            "rest_api_id": rest_api_id,
        }
        if api_key_required is not None:
            self._values["api_key_required"] = api_key_required
        if authorization_scopes is not None:
            self._values["authorization_scopes"] = authorization_scopes
        if authorization_type is not None:
            self._values["authorization_type"] = authorization_type
        if authorizer_id is not None:
            self._values["authorizer_id"] = authorizer_id
        if integration is not None:
            self._values["integration"] = integration
        if method_responses is not None:
            self._values["method_responses"] = method_responses
        if operation_name is not None:
            self._values["operation_name"] = operation_name
        if request_models is not None:
            self._values["request_models"] = request_models
        if request_parameters is not None:
            self._values["request_parameters"] = request_parameters
        if request_validator_id is not None:
            self._values["request_validator_id"] = request_validator_id

    @builtins.property
    def http_method(self) -> builtins.str:
        '''The method's HTTP verb.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-httpmethod
        '''
        result = self._values.get("http_method")
        assert result is not None, "Required property 'http_method' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def resource_id(self) -> builtins.str:
        '''The Resource identifier for the MethodResponse resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-resourceid
        '''
        result = self._values.get("resource_id")
        assert result is not None, "Required property 'resource_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def api_key_required(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A boolean flag specifying whether a valid ApiKey is required to invoke this method.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-apikeyrequired
        '''
        result = self._values.get("api_key_required")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def authorization_scopes(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of authorization scopes configured on the method.

        The scopes are used with a ``COGNITO_USER_POOLS`` authorizer to authorize the method invocation. The authorization works by matching the method scopes against the scopes parsed from the access token in the incoming request. The method invocation is authorized if any method scopes matches a claimed scope in the access token. Otherwise, the invocation is not authorized. When the method scope is configured, the client must provide an access token instead of an identity token for authorization purposes.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizationscopes
        '''
        result = self._values.get("authorization_scopes")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def authorization_type(self) -> typing.Optional[builtins.str]:
        '''The method's authorization type.

        This parameter is required. For valid values, see `Method <https://docs.aws.amazon.com/apigateway/latest/api/API_Method.html>`_ in the *API Gateway API Reference* .
        .. epigraph::

           If you specify the ``AuthorizerId`` property, specify ``CUSTOM`` or ``COGNITO_USER_POOLS`` for this property.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizationtype
        '''
        result = self._values.get("authorization_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorizer_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of an authorizer to use on this method.

        The method's authorization type must be ``CUSTOM`` or ``COGNITO_USER_POOLS`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-authorizerid
        '''
        result = self._values.get("authorizer_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnMethod.IntegrationProperty]]:
        '''Represents an ``HTTP`` , ``HTTP_PROXY`` , ``AWS`` , ``AWS_PROXY`` , or Mock integration.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-integration
        '''
        result = self._values.get("integration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnMethod.IntegrationProperty]], result)

    @builtins.property
    def method_responses(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnMethod.MethodResponseProperty]]]]:
        '''Gets a method response associated with a given HTTP status code.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-methodresponses
        '''
        result = self._values.get("method_responses")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnMethod.MethodResponseProperty]]]], result)

    @builtins.property
    def operation_name(self) -> typing.Optional[builtins.str]:
        '''A human-friendly operation identifier for the method.

        For example, you can assign the ``operationName`` of ``ListPets`` for the ``GET /pets`` method in the ``PetStore`` example.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-operationname
        '''
        result = self._values.get("operation_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def request_models(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''A key-value map specifying data schemas, represented by Model resources, (as the mapped value) of the request payloads of given content types (as the mapping key).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-requestmodels
        '''
        result = self._values.get("request_models")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

    @builtins.property
    def request_parameters(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_da3f097b]]]]:
        '''A key-value map defining required or optional method request parameters that can be accepted by API Gateway.

        A key is a method request parameter name matching the pattern of ``method.request.{location}.{name}`` , where ``location`` is ``querystring`` , ``path`` , or ``header`` and ``name`` is a valid and unique parameter name. The value associated with the key is a Boolean flag indicating whether the parameter is required ( ``true`` ) or optional ( ``false`` ). The method request parameter names defined here are available in Integration to be mapped to integration request parameters or templates.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-requestparameters
        '''
        result = self._values.get("request_parameters")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[builtins.bool, _IResolvable_da3f097b]]]], result)

    @builtins.property
    def request_validator_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of a RequestValidator for request validation.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-method.html#cfn-apigateway-method-requestvalidatorid
        '''
        result = self._values.get("request_validator_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnMethodProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnModel(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnModel",
):
    '''The ``AWS::ApiGateway::Model`` resource defines the structure of a request or response payload for an API method.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html
    :cloudformationResource: AWS::ApiGateway::Model
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        # schema: Any
        
        cfn_model = apigateway.CfnModel(self, "MyCfnModel",
            rest_api_id="restApiId",
        
            # the properties below are optional
            content_type="contentType",
            description="description",
            name="name",
            schema=schema
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        rest_api_id: builtins.str,
        content_type: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        name: typing.Optional[builtins.str] = None,
        schema: typing.Any = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param rest_api_id: The string identifier of the associated RestApi.
        :param content_type: The content-type for the model.
        :param description: The description of the model.
        :param name: A name for the model. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the model name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . .. epigraph:: If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param schema: The schema for the model. For ``application/json`` models, this should be JSON schema draft 4 model. Do not include "* /" characters in the description of any properties because such "* /" characters may be interpreted as the closing marker for comments in some languages, such as Java or JavaScript, causing the installation of your API's SDK generated by API Gateway to fail.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a6e1faf9e03b333c72e2b28824ff5dd3c469b86e45694da5032821da855a2d10)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnModelProps(
            rest_api_id=rest_api_id,
            content_type=content_type,
            description=description,
            name=name,
            schema=schema,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b09f532e132c65007e99d8034d811496396cdcd764fe1555005902abe5d3822)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__26954373273c2ca7f9896141adaedab3492a98dc4763ca30f8e086a1792fbe19)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.'''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4a3c5b423cee3a8c0b9103f5d4563c6257372b12489d27543585737ab1c56396)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="contentType")
    def content_type(self) -> typing.Optional[builtins.str]:
        '''The content-type for the model.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "contentType"))

    @content_type.setter
    def content_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7e9538583fde25d5a4bfbf9ad18d6b8e965c725934d293324f3ddd0a2b541cff)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "contentType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the model.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e14bcaa67c5515b8d1e714c0cd74ace7e6734c83e3b4142cb061c10507dfe866)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''A name for the model.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7970ef90ab7e30e7216cfcd425ee6821d9f9f51870d49472d5510bba9479cc61)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="schema")
    def schema(self) -> typing.Any:
        '''The schema for the model.'''
        return typing.cast(typing.Any, jsii.get(self, "schema"))

    @schema.setter
    def schema(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__189ff15874544c903e011bd25d8e6463925d43c41be7532a43378df709f954e5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "schema", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnModelProps",
    jsii_struct_bases=[],
    name_mapping={
        "rest_api_id": "restApiId",
        "content_type": "contentType",
        "description": "description",
        "name": "name",
        "schema": "schema",
    },
)
class CfnModelProps:
    def __init__(
        self,
        *,
        rest_api_id: builtins.str,
        content_type: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        name: typing.Optional[builtins.str] = None,
        schema: typing.Any = None,
    ) -> None:
        '''Properties for defining a ``CfnModel``.

        :param rest_api_id: The string identifier of the associated RestApi.
        :param content_type: The content-type for the model.
        :param description: The description of the model.
        :param name: A name for the model. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the model name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ . .. epigraph:: If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.
        :param schema: The schema for the model. For ``application/json`` models, this should be JSON schema draft 4 model. Do not include "* /" characters in the description of any properties because such "* /" characters may be interpreted as the closing marker for comments in some languages, such as Java or JavaScript, causing the installation of your API's SDK generated by API Gateway to fail.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            # schema: Any
            
            cfn_model_props = apigateway.CfnModelProps(
                rest_api_id="restApiId",
            
                # the properties below are optional
                content_type="contentType",
                description="description",
                name="name",
                schema=schema
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8e1e6d6536772d3f72606938d02638ae58fb2e422aa4c7c249db179ca394fcce)
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument content_type", value=content_type, expected_type=type_hints["content_type"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument schema", value=schema, expected_type=type_hints["schema"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "rest_api_id": rest_api_id,
        }
        if content_type is not None:
            self._values["content_type"] = content_type
        if description is not None:
            self._values["description"] = description
        if name is not None:
            self._values["name"] = name
        if schema is not None:
            self._values["schema"] = schema

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def content_type(self) -> typing.Optional[builtins.str]:
        '''The content-type for the model.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-contenttype
        '''
        result = self._values.get("content_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the model.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''A name for the model.

        If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the model name. For more information, see `Name Type <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-name.html>`_ .
        .. epigraph::

           If you specify a name, you cannot perform updates that require replacement of this resource. You can perform updates that require no or some interruption. If you must replace the resource, specify a new name.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def schema(self) -> typing.Any:
        '''The schema for the model.

        For ``application/json`` models, this should be JSON schema draft 4 model. Do not include "* /" characters in the description of any properties because such "* /" characters may be interpreted as the closing marker for comments in some languages, such as Java or JavaScript, causing the installation of your API's SDK generated by API Gateway to fail.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-model.html#cfn-apigateway-model-schema
        '''
        result = self._values.get("schema")
        return typing.cast(typing.Any, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnModelProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnRequestValidator(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnRequestValidator",
):
    '''The ``AWS::ApiGateway::RequestValidator`` resource sets up basic validation rules for incoming requests to your API.

    For more information, see `Enable Basic Request Validation for an API in API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-method-request-validation.html>`_ in the *API Gateway Developer Guide* .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html
    :cloudformationResource: AWS::ApiGateway::RequestValidator
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_request_validator = apigateway.CfnRequestValidator(self, "MyCfnRequestValidator",
            rest_api_id="restApiId",
        
            # the properties below are optional
            name="name",
            validate_request_body=False,
            validate_request_parameters=False
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        rest_api_id: builtins.str,
        name: typing.Optional[builtins.str] = None,
        validate_request_body: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        validate_request_parameters: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param rest_api_id: The string identifier of the associated RestApi.
        :param name: The name of this RequestValidator.
        :param validate_request_body: A Boolean flag to indicate whether to validate a request body according to the configured Model schema.
        :param validate_request_parameters: A Boolean flag to indicate whether to validate request parameters ( ``true`` ) or not ( ``false`` ).
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__31c91a22069ae0ca6e9846fb3a02b1808c13ac942ad7391e1caa2fca78f3377f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnRequestValidatorProps(
            rest_api_id=rest_api_id,
            name=name,
            validate_request_body=validate_request_body,
            validate_request_parameters=validate_request_parameters,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9e0b16fed4a9140858ad6504de654ade71c20e5231445b00800f6b349b8519bb)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3e5161752d3cb7520bd8e4e1c160b84bf0507a44efbc2b8e037086b3cc08157d)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrRequestValidatorId")
    def attr_request_validator_id(self) -> builtins.str:
        '''The ID for the request validator.

        For example: ``abc123`` .

        :cloudformationAttribute: RequestValidatorId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRequestValidatorId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.'''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__07bb96cf571f51a62f52665a351ca4c709ad118bf59cd4aa41af57efb92eec6e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of this RequestValidator.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3310f00b8f9e73048d492671ff4850531b0974156243790df75c71159938d907)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="validateRequestBody")
    def validate_request_body(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A Boolean flag to indicate whether to validate a request body according to the configured Model schema.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "validateRequestBody"))

    @validate_request_body.setter
    def validate_request_body(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b15e7c43afb2ebc0faf4f31ac1e746c2f98b627e8a0e48213f58832ba86c3ec7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "validateRequestBody", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="validateRequestParameters")
    def validate_request_parameters(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A Boolean flag to indicate whether to validate request parameters ( ``true`` ) or not ( ``false`` ).'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "validateRequestParameters"))

    @validate_request_parameters.setter
    def validate_request_parameters(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__373b014f81740f98fc49cfce56f72c1e0b243f647842baa67363b94506b1aed5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "validateRequestParameters", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnRequestValidatorProps",
    jsii_struct_bases=[],
    name_mapping={
        "rest_api_id": "restApiId",
        "name": "name",
        "validate_request_body": "validateRequestBody",
        "validate_request_parameters": "validateRequestParameters",
    },
)
class CfnRequestValidatorProps:
    def __init__(
        self,
        *,
        rest_api_id: builtins.str,
        name: typing.Optional[builtins.str] = None,
        validate_request_body: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        validate_request_parameters: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
    ) -> None:
        '''Properties for defining a ``CfnRequestValidator``.

        :param rest_api_id: The string identifier of the associated RestApi.
        :param name: The name of this RequestValidator.
        :param validate_request_body: A Boolean flag to indicate whether to validate a request body according to the configured Model schema.
        :param validate_request_parameters: A Boolean flag to indicate whether to validate request parameters ( ``true`` ) or not ( ``false`` ).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_request_validator_props = apigateway.CfnRequestValidatorProps(
                rest_api_id="restApiId",
            
                # the properties below are optional
                name="name",
                validate_request_body=False,
                validate_request_parameters=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9f58946e74f4ea0477acd63d1f517a3c0b56977bdbf74e7dbf82969b550cd552)
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument validate_request_body", value=validate_request_body, expected_type=type_hints["validate_request_body"])
            check_type(argname="argument validate_request_parameters", value=validate_request_parameters, expected_type=type_hints["validate_request_parameters"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "rest_api_id": rest_api_id,
        }
        if name is not None:
            self._values["name"] = name
        if validate_request_body is not None:
            self._values["validate_request_body"] = validate_request_body
        if validate_request_parameters is not None:
            self._values["validate_request_parameters"] = validate_request_parameters

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of this RequestValidator.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def validate_request_body(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A Boolean flag to indicate whether to validate a request body according to the configured Model schema.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-validaterequestbody
        '''
        result = self._values.get("validate_request_body")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def validate_request_parameters(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A Boolean flag to indicate whether to validate request parameters ( ``true`` ) or not ( ``false`` ).

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-requestvalidator.html#cfn-apigateway-requestvalidator-validaterequestparameters
        '''
        result = self._values.get("validate_request_parameters")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRequestValidatorProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556)
class CfnResource(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnResource",
):
    '''The ``AWS::ApiGateway::Resource`` resource creates a resource in an API.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html
    :cloudformationResource: AWS::ApiGateway::Resource
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_resource = apigateway.CfnResource(self, "MyCfnResource",
            parent_id="parentId",
            path_part="pathPart",
            rest_api_id="restApiId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        parent_id: builtins.str,
        path_part: builtins.str,
        rest_api_id: builtins.str,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param parent_id: The parent resource's identifier.
        :param path_part: The last path segment for this resource.
        :param rest_api_id: The string identifier of the associated RestApi.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__44ada1c3ba590eab662ae2f9e418e3c4cdf8612323c1094e30e0658e069f2488)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnResourceProps(
            parent_id=parent_id, path_part=path_part, rest_api_id=rest_api_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2ce3f1e7b8470b4e68e54d923d32a366234d75e1aca06d51b1908fa61f4e3efd)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__367957cd2162c308802b5b870466f4be93cf34b45cbbee115fc1f7dae2f01400)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrResourceId")
    def attr_resource_id(self) -> builtins.str:
        '''The ID for the resource.

        For example: ``abc123`` .

        :cloudformationAttribute: ResourceId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="parentId")
    def parent_id(self) -> builtins.str:
        '''The parent resource's identifier.'''
        return typing.cast(builtins.str, jsii.get(self, "parentId"))

    @parent_id.setter
    def parent_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__733461cc0be55889efa94f529eecfe1e39854350082156c58f940ff2836d1d22)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "parentId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="pathPart")
    def path_part(self) -> builtins.str:
        '''The last path segment for this resource.'''
        return typing.cast(builtins.str, jsii.get(self, "pathPart"))

    @path_part.setter
    def path_part(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2f3bcfc274c219391ada7b47c4a3ad6012b4dd8ce25d74c8dc297162032e3209)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "pathPart", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.'''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1728e101c5ee29c1334ae52f2d1757e5e5f11fa9664aa2d54ef0e876d6590a81)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnResourceProps",
    jsii_struct_bases=[],
    name_mapping={
        "parent_id": "parentId",
        "path_part": "pathPart",
        "rest_api_id": "restApiId",
    },
)
class CfnResourceProps:
    def __init__(
        self,
        *,
        parent_id: builtins.str,
        path_part: builtins.str,
        rest_api_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnResource``.

        :param parent_id: The parent resource's identifier.
        :param path_part: The last path segment for this resource.
        :param rest_api_id: The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_resource_props = apigateway.CfnResourceProps(
                parent_id="parentId",
                path_part="pathPart",
                rest_api_id="restApiId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1d5552a9f35619dbfaba4ec8b3153ac3f07d1a18c68b97a7ccd55ed1e90643e6)
            check_type(argname="argument parent_id", value=parent_id, expected_type=type_hints["parent_id"])
            check_type(argname="argument path_part", value=path_part, expected_type=type_hints["path_part"])
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "parent_id": parent_id,
            "path_part": path_part,
            "rest_api_id": rest_api_id,
        }

    @builtins.property
    def parent_id(self) -> builtins.str:
        '''The parent resource's identifier.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html#cfn-apigateway-resource-parentid
        '''
        result = self._values.get("parent_id")
        assert result is not None, "Required property 'parent_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def path_part(self) -> builtins.str:
        '''The last path segment for this resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html#cfn-apigateway-resource-pathpart
        '''
        result = self._values.get("path_part")
        assert result is not None, "Required property 'path_part' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-resource.html#cfn-apigateway-resource-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnResourceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnRestApi(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnRestApi",
):
    '''The ``AWS::ApiGateway::RestApi`` resource creates a REST API.

    For more information, see `restapi:create <https://docs.aws.amazon.com/apigateway/latest/api/API_CreateRestApi.html>`_ in the *Amazon API Gateway REST API Reference* .
    .. epigraph::

       On January 1, 2016, the Swagger Specification was donated to the `OpenAPI initiative <https://docs.aws.amazon.com/https://www.openapis.org/>`_ , becoming the foundation of the OpenAPI Specification.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html
    :cloudformationResource: AWS::ApiGateway::RestApi
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        # body: Any
        # policy: Any
        
        cfn_rest_api = apigateway.CfnRestApi(self, "MyCfnRestApi",
            api_key_source_type="apiKeySourceType",
            binary_media_types=["binaryMediaTypes"],
            body=body,
            body_s3_location=apigateway.CfnRestApi.S3LocationProperty(
                bucket="bucket",
                e_tag="eTag",
                key="key",
                version="version"
            ),
            clone_from="cloneFrom",
            description="description",
            disable_execute_api_endpoint=False,
            endpoint_configuration=apigateway.CfnRestApi.EndpointConfigurationProperty(
                ip_address_type="ipAddressType",
                types=["types"],
                vpc_endpoint_ids=["vpcEndpointIds"]
            ),
            fail_on_warnings=False,
            minimum_compression_size=123,
            mode="mode",
            name="name",
            parameters={
                "parameters_key": "parameters"
            },
            policy=policy,
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        api_key_source_type: typing.Optional[builtins.str] = None,
        binary_media_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        body: typing.Any = None,
        body_s3_location: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRestApi.S3LocationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        clone_from: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        disable_execute_api_endpoint: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        endpoint_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnRestApi.EndpointConfigurationProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        fail_on_warnings: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        minimum_compression_size: typing.Optional[jsii.Number] = None,
        mode: typing.Optional[builtins.str] = None,
        name: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        policy: typing.Any = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param api_key_source_type: The source of the API key for metering requests according to a usage plan. Valid values are: ``HEADER`` to read the API key from the ``X-API-Key`` header of a request. ``AUTHORIZER`` to read the API key from the ``UsageIdentifierKey`` from a custom authorizer.
        :param binary_media_types: The list of binary media types supported by the RestApi. By default, the RestApi supports only UTF-8-encoded text payloads.
        :param body: An OpenAPI specification that defines a set of RESTful APIs in JSON format. For YAML templates, you can also provide the specification in YAML format.
        :param body_s3_location: The Amazon Simple Storage Service (Amazon S3) location that points to an OpenAPI file, which defines a set of RESTful APIs in JSON or YAML format.
        :param clone_from: The ID of the RestApi that you want to clone from.
        :param description: The description of the RestApi.
        :param disable_execute_api_endpoint: Specifies whether clients can invoke your API by using the default ``execute-api`` endpoint. By default, clients can invoke your API with the default ``https://{api_id}.execute-api.{region}.amazonaws.com`` endpoint. To require that clients use a custom domain name to invoke your API, disable the default endpoint
        :param endpoint_configuration: A list of the endpoint types and IP address types of the API. Use this property when creating an API. When importing an existing API, specify the endpoint configuration types using the ``Parameters`` property.
        :param fail_on_warnings: A query parameter to indicate whether to rollback the API update ( ``true`` ) or not ( ``false`` ) when a warning is encountered. The default value is ``false`` .
        :param minimum_compression_size: A nullable integer that is used to enable compression (with non-negative between 0 and 10485760 (10M) bytes, inclusive) or disable compression (with a null value) on an API. When compression is enabled, compression or decompression is not applied on the payload if the payload size is smaller than this value. Setting it to zero allows compression for any payload size.
        :param mode: This property applies only when you use OpenAPI to define your REST API. The ``Mode`` determines how API Gateway handles resource updates. Valid values are ``overwrite`` or ``merge`` . For ``overwrite`` , the new API definition replaces the existing one. The existing API identifier remains unchanged. For ``merge`` , the new API definition is merged with the existing API. If you don't specify this property, a default value is chosen. For REST APIs created before March 29, 2021, the default is ``overwrite`` . For REST APIs created after March 29, 2021, the new API definition takes precedence, but any container types such as endpoint configurations and binary media types are merged with the existing API. Use the default mode to define top-level ``RestApi`` properties in addition to using OpenAPI. Generally, it's preferred to use API Gateway's OpenAPI extensions to model these properties.
        :param name: The name of the RestApi. A name is required if the REST API is not based on an OpenAPI specification.
        :param parameters: Custom header parameters as part of the request. For example, to exclude DocumentationParts from an imported API, set ``ignore=documentation`` as a ``parameters`` value, as in the AWS CLI command of ``aws apigateway import-rest-api --parameters ignore=documentation --body 'file:///path/to/imported-api-body.json'`` .
        :param policy: A policy document that contains the permissions for the ``RestApi`` resource. To set the ARN for the policy, use the ``!Join`` intrinsic function with ``""`` as delimiter and values of ``"execute-api:/"`` and ``"*"`` .
        :param tags: The key-value map of strings. The valid character set is [a-zA-Z+-=._:/]. The tag key can be up to 128 characters and must not start with ``aws:`` . The tag value can be up to 256 characters.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__781aae0a2f2512495cbd89d2be23edec0478e71fd6a59c0527cabb245cb94761)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnRestApiProps(
            api_key_source_type=api_key_source_type,
            binary_media_types=binary_media_types,
            body=body,
            body_s3_location=body_s3_location,
            clone_from=clone_from,
            description=description,
            disable_execute_api_endpoint=disable_execute_api_endpoint,
            endpoint_configuration=endpoint_configuration,
            fail_on_warnings=fail_on_warnings,
            minimum_compression_size=minimum_compression_size,
            mode=mode,
            name=name,
            parameters=parameters,
            policy=policy,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0d311995dffe9db41f0b189d5aaebea2138fbb7863d3fb06df01c202917e8b13)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9846c0a8b4800d5a296f9aea67a49cdaf59a24f4ab25ea67ad14c7a442d92991)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrRestApiId")
    def attr_rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.

        :cloudformationAttribute: RestApiId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRestApiId"))

    @builtins.property
    @jsii.member(jsii_name="attrRootResourceId")
    def attr_root_resource_id(self) -> builtins.str:
        '''The root resource ID for a ``RestApi`` resource, such as ``a0bc123d4e`` .

        :cloudformationAttribute: RootResourceId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRootResourceId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="apiKeySourceType")
    def api_key_source_type(self) -> typing.Optional[builtins.str]:
        '''The source of the API key for metering requests according to a usage plan.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "apiKeySourceType"))

    @api_key_source_type.setter
    def api_key_source_type(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__885692d26cf256ea028d4888289ae060ae3d89b80470c8ff0f82559c7623a155)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiKeySourceType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="binaryMediaTypes")
    def binary_media_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of binary media types supported by the RestApi.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "binaryMediaTypes"))

    @binary_media_types.setter
    def binary_media_types(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e5382629090fca37327ab38dbd9d2807d25d11bdd4aa29ea10df0383a905ff97)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "binaryMediaTypes", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="body")
    def body(self) -> typing.Any:
        '''An OpenAPI specification that defines a set of RESTful APIs in JSON format.'''
        return typing.cast(typing.Any, jsii.get(self, "body"))

    @body.setter
    def body(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__531f017906edab784c1a550cefadd535f6de66c5fb156fb602b44dbb6ddf59c5)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "body", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="bodyS3Location")
    def body_s3_location(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRestApi.S3LocationProperty"]]:
        '''The Amazon Simple Storage Service (Amazon S3) location that points to an OpenAPI file, which defines a set of RESTful APIs in JSON or YAML format.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRestApi.S3LocationProperty"]], jsii.get(self, "bodyS3Location"))

    @body_s3_location.setter
    def body_s3_location(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRestApi.S3LocationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ef1ea351452be8b47bde86a70f6ceb7d6cdf7e0f3cadbc3aa763c8e6a8ebe58c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "bodyS3Location", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="cloneFrom")
    def clone_from(self) -> typing.Optional[builtins.str]:
        '''The ID of the RestApi that you want to clone from.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cloneFrom"))

    @clone_from.setter
    def clone_from(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8f0c822c5e44a830e35483bcfc17fe0d274c44fa77e7cac67357b8cb05dc0371)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cloneFrom", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the RestApi.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__862c58d133c7c7042cd781761e3c59bc0b1a0787a856efc4acf0f9379d626fde)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="disableExecuteApiEndpoint")
    def disable_execute_api_endpoint(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether clients can invoke your API by using the default ``execute-api`` endpoint.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "disableExecuteApiEndpoint"))

    @disable_execute_api_endpoint.setter
    def disable_execute_api_endpoint(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b9945526e80dc90cbfbb86ac5f423295588dfd680221afef09db1c77df2985fb)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "disableExecuteApiEndpoint", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="endpointConfiguration")
    def endpoint_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRestApi.EndpointConfigurationProperty"]]:
        '''A list of the endpoint types and IP address types of the API.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRestApi.EndpointConfigurationProperty"]], jsii.get(self, "endpointConfiguration"))

    @endpoint_configuration.setter
    def endpoint_configuration(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnRestApi.EndpointConfigurationProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d74e18e19938f0a8587b0106d2093ba2b9044df35435213edb73072f46f6877e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endpointConfiguration", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="failOnWarnings")
    def fail_on_warnings(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A query parameter to indicate whether to rollback the API update ( ``true`` ) or not ( ``false`` ) when a warning is encountered.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "failOnWarnings"))

    @fail_on_warnings.setter
    def fail_on_warnings(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__58328de38298447d398e8e44c69092ec4537ecc0e975cba2b8fcfcd76fba050c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "failOnWarnings", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="minimumCompressionSize")
    def minimum_compression_size(self) -> typing.Optional[jsii.Number]:
        '''A nullable integer that is used to enable compression (with non-negative between 0 and 10485760 (10M) bytes, inclusive) or disable compression (with a null value) on an API.'''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "minimumCompressionSize"))

    @minimum_compression_size.setter
    def minimum_compression_size(self, value: typing.Optional[jsii.Number]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__596348b4cfe4fa636b4ea4d3f191b4cd9319edbbd44bc9b7a4a8578b74311888)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "minimumCompressionSize", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="mode")
    def mode(self) -> typing.Optional[builtins.str]:
        '''This property applies only when you use OpenAPI to define your REST API.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "mode"))

    @mode.setter
    def mode(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0236c7b6983a410b44b3cbaa06c4b2d5531201236c757bf05850f5f3270357b4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "mode", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of the RestApi.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3b2f687ba486b97daef3efdc148372d3d51144ffda7dc8bf1ef44bc7707f30a9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="parameters")
    def parameters(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''Custom header parameters as part of the request.'''
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], jsii.get(self, "parameters"))

    @parameters.setter
    def parameters(
        self,
        value: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ccaf03012418784e0b764088769b0e23937e2028288348151f0ccfbfd74daa7b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "parameters", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="policy")
    def policy(self) -> typing.Any:
        '''A policy document that contains the permissions for the ``RestApi`` resource.'''
        return typing.cast(typing.Any, jsii.get(self, "policy"))

    @policy.setter
    def policy(self, value: typing.Any) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b7c411e289712e4a9bfc8e944902e0103f628540a4cf7e48a8c9167888e17d8f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "policy", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The key-value map of strings.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ae71bb5beb4f360ca0016003cbb508243a9ad4b5e494d1a4655780f8350d620)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnRestApi.EndpointConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "ip_address_type": "ipAddressType",
            "types": "types",
            "vpc_endpoint_ids": "vpcEndpointIds",
        },
    )
    class EndpointConfigurationProperty:
        def __init__(
            self,
            *,
            ip_address_type: typing.Optional[builtins.str] = None,
            types: typing.Optional[typing.Sequence[builtins.str]] = None,
            vpc_endpoint_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The ``EndpointConfiguration`` property type specifies the endpoint types and IP address types of a REST API.

            ``EndpointConfiguration`` is a property of the `AWS::ApiGateway::RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource.

            :param ip_address_type: The IP address types that can invoke an API (RestApi). Use ``ipv4`` to allow only IPv4 addresses to invoke an API, or use ``dualstack`` to allow both IPv4 and IPv6 addresses to invoke an API. For the ``PRIVATE`` endpoint type, only ``dualstack`` is supported.
            :param types: A list of endpoint types of an API (RestApi) or its custom domain name (DomainName). For an edge-optimized API and its custom domain name, the endpoint type is ``"EDGE"`` . For a regional API and its custom domain name, the endpoint type is ``REGIONAL`` . For a private API, the endpoint type is ``PRIVATE`` .
            :param vpc_endpoint_ids: A list of VpcEndpointIds of an API (RestApi) against which to create Route53 ALIASes. It is only supported for ``PRIVATE`` endpoint type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-endpointconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                endpoint_configuration_property = apigateway.CfnRestApi.EndpointConfigurationProperty(
                    ip_address_type="ipAddressType",
                    types=["types"],
                    vpc_endpoint_ids=["vpcEndpointIds"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__df9522828c39c6e4b1e829251c3b77d5b94051a139bf4f10dfa3ac8717e878b6)
                check_type(argname="argument ip_address_type", value=ip_address_type, expected_type=type_hints["ip_address_type"])
                check_type(argname="argument types", value=types, expected_type=type_hints["types"])
                check_type(argname="argument vpc_endpoint_ids", value=vpc_endpoint_ids, expected_type=type_hints["vpc_endpoint_ids"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if ip_address_type is not None:
                self._values["ip_address_type"] = ip_address_type
            if types is not None:
                self._values["types"] = types
            if vpc_endpoint_ids is not None:
                self._values["vpc_endpoint_ids"] = vpc_endpoint_ids

        @builtins.property
        def ip_address_type(self) -> typing.Optional[builtins.str]:
            '''The IP address types that can invoke an API (RestApi).

            Use ``ipv4`` to allow only IPv4 addresses to invoke an API, or use ``dualstack`` to allow both IPv4 and IPv6 addresses to invoke an API. For the ``PRIVATE`` endpoint type, only ``dualstack`` is supported.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-endpointconfiguration.html#cfn-apigateway-restapi-endpointconfiguration-ipaddresstype
            '''
            result = self._values.get("ip_address_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def types(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of endpoint types of an API (RestApi) or its custom domain name (DomainName).

            For an edge-optimized API and its custom domain name, the endpoint type is ``"EDGE"`` . For a regional API and its custom domain name, the endpoint type is ``REGIONAL`` . For a private API, the endpoint type is ``PRIVATE`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-endpointconfiguration.html#cfn-apigateway-restapi-endpointconfiguration-types
            '''
            result = self._values.get("types")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def vpc_endpoint_ids(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of VpcEndpointIds of an API (RestApi) against which to create Route53 ALIASes.

            It is only supported for ``PRIVATE`` endpoint type.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-endpointconfiguration.html#cfn-apigateway-restapi-endpointconfiguration-vpcendpointids
            '''
            result = self._values.get("vpc_endpoint_ids")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndpointConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnRestApi.S3LocationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "bucket": "bucket",
            "e_tag": "eTag",
            "key": "key",
            "version": "version",
        },
    )
    class S3LocationProperty:
        def __init__(
            self,
            *,
            bucket: typing.Optional[builtins.str] = None,
            e_tag: typing.Optional[builtins.str] = None,
            key: typing.Optional[builtins.str] = None,
            version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``S3Location`` is a property of the `AWS::ApiGateway::RestApi <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html>`_ resource that specifies the Amazon S3 location of a OpenAPI (formerly Swagger) file that defines a set of RESTful APIs in JSON or YAML.

            .. epigraph::

               On January 1, 2016, the Swagger Specification was donated to the `OpenAPI initiative <https://docs.aws.amazon.com/https://www.openapis.org/>`_ , becoming the foundation of the OpenAPI Specification.

            :param bucket: The name of the S3 bucket where the OpenAPI file is stored.
            :param e_tag: The Amazon S3 ETag (a file checksum) of the OpenAPI file. If you don't specify a value, API Gateway skips ETag validation of your OpenAPI file.
            :param key: The file name of the OpenAPI file (Amazon S3 object name).
            :param version: For versioning-enabled buckets, a specific version of the OpenAPI file.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-s3location.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                s3_location_property = apigateway.CfnRestApi.S3LocationProperty(
                    bucket="bucket",
                    e_tag="eTag",
                    key="key",
                    version="version"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4db951fe61066510a2f930d6ecfa9b80b97fea25c1baa64d092e5411d4745af2)
                check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
                check_type(argname="argument e_tag", value=e_tag, expected_type=type_hints["e_tag"])
                check_type(argname="argument key", value=key, expected_type=type_hints["key"])
                check_type(argname="argument version", value=version, expected_type=type_hints["version"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if bucket is not None:
                self._values["bucket"] = bucket
            if e_tag is not None:
                self._values["e_tag"] = e_tag
            if key is not None:
                self._values["key"] = key
            if version is not None:
                self._values["version"] = version

        @builtins.property
        def bucket(self) -> typing.Optional[builtins.str]:
            '''The name of the S3 bucket where the OpenAPI file is stored.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-s3location.html#cfn-apigateway-restapi-s3location-bucket
            '''
            result = self._values.get("bucket")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def e_tag(self) -> typing.Optional[builtins.str]:
            '''The Amazon S3 ETag (a file checksum) of the OpenAPI file.

            If you don't specify a value, API Gateway skips ETag validation of your OpenAPI file.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-s3location.html#cfn-apigateway-restapi-s3location-etag
            '''
            result = self._values.get("e_tag")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def key(self) -> typing.Optional[builtins.str]:
            '''The file name of the OpenAPI file (Amazon S3 object name).

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-s3location.html#cfn-apigateway-restapi-s3location-key
            '''
            result = self._values.get("key")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def version(self) -> typing.Optional[builtins.str]:
            '''For versioning-enabled buckets, a specific version of the OpenAPI file.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-s3location.html#cfn-apigateway-restapi-s3location-version
            '''
            result = self._values.get("version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "S3LocationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnRestApiProps",
    jsii_struct_bases=[],
    name_mapping={
        "api_key_source_type": "apiKeySourceType",
        "binary_media_types": "binaryMediaTypes",
        "body": "body",
        "body_s3_location": "bodyS3Location",
        "clone_from": "cloneFrom",
        "description": "description",
        "disable_execute_api_endpoint": "disableExecuteApiEndpoint",
        "endpoint_configuration": "endpointConfiguration",
        "fail_on_warnings": "failOnWarnings",
        "minimum_compression_size": "minimumCompressionSize",
        "mode": "mode",
        "name": "name",
        "parameters": "parameters",
        "policy": "policy",
        "tags": "tags",
    },
)
class CfnRestApiProps:
    def __init__(
        self,
        *,
        api_key_source_type: typing.Optional[builtins.str] = None,
        binary_media_types: typing.Optional[typing.Sequence[builtins.str]] = None,
        body: typing.Any = None,
        body_s3_location: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnRestApi.S3LocationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        clone_from: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        disable_execute_api_endpoint: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        endpoint_configuration: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnRestApi.EndpointConfigurationProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        fail_on_warnings: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        minimum_compression_size: typing.Optional[jsii.Number] = None,
        mode: typing.Optional[builtins.str] = None,
        name: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
        policy: typing.Any = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnRestApi``.

        :param api_key_source_type: The source of the API key for metering requests according to a usage plan. Valid values are: ``HEADER`` to read the API key from the ``X-API-Key`` header of a request. ``AUTHORIZER`` to read the API key from the ``UsageIdentifierKey`` from a custom authorizer.
        :param binary_media_types: The list of binary media types supported by the RestApi. By default, the RestApi supports only UTF-8-encoded text payloads.
        :param body: An OpenAPI specification that defines a set of RESTful APIs in JSON format. For YAML templates, you can also provide the specification in YAML format.
        :param body_s3_location: The Amazon Simple Storage Service (Amazon S3) location that points to an OpenAPI file, which defines a set of RESTful APIs in JSON or YAML format.
        :param clone_from: The ID of the RestApi that you want to clone from.
        :param description: The description of the RestApi.
        :param disable_execute_api_endpoint: Specifies whether clients can invoke your API by using the default ``execute-api`` endpoint. By default, clients can invoke your API with the default ``https://{api_id}.execute-api.{region}.amazonaws.com`` endpoint. To require that clients use a custom domain name to invoke your API, disable the default endpoint
        :param endpoint_configuration: A list of the endpoint types and IP address types of the API. Use this property when creating an API. When importing an existing API, specify the endpoint configuration types using the ``Parameters`` property.
        :param fail_on_warnings: A query parameter to indicate whether to rollback the API update ( ``true`` ) or not ( ``false`` ) when a warning is encountered. The default value is ``false`` .
        :param minimum_compression_size: A nullable integer that is used to enable compression (with non-negative between 0 and 10485760 (10M) bytes, inclusive) or disable compression (with a null value) on an API. When compression is enabled, compression or decompression is not applied on the payload if the payload size is smaller than this value. Setting it to zero allows compression for any payload size.
        :param mode: This property applies only when you use OpenAPI to define your REST API. The ``Mode`` determines how API Gateway handles resource updates. Valid values are ``overwrite`` or ``merge`` . For ``overwrite`` , the new API definition replaces the existing one. The existing API identifier remains unchanged. For ``merge`` , the new API definition is merged with the existing API. If you don't specify this property, a default value is chosen. For REST APIs created before March 29, 2021, the default is ``overwrite`` . For REST APIs created after March 29, 2021, the new API definition takes precedence, but any container types such as endpoint configurations and binary media types are merged with the existing API. Use the default mode to define top-level ``RestApi`` properties in addition to using OpenAPI. Generally, it's preferred to use API Gateway's OpenAPI extensions to model these properties.
        :param name: The name of the RestApi. A name is required if the REST API is not based on an OpenAPI specification.
        :param parameters: Custom header parameters as part of the request. For example, to exclude DocumentationParts from an imported API, set ``ignore=documentation`` as a ``parameters`` value, as in the AWS CLI command of ``aws apigateway import-rest-api --parameters ignore=documentation --body 'file:///path/to/imported-api-body.json'`` .
        :param policy: A policy document that contains the permissions for the ``RestApi`` resource. To set the ARN for the policy, use the ``!Join`` intrinsic function with ``""`` as delimiter and values of ``"execute-api:/"`` and ``"*"`` .
        :param tags: The key-value map of strings. The valid character set is [a-zA-Z+-=._:/]. The tag key can be up to 128 characters and must not start with ``aws:`` . The tag value can be up to 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            # body: Any
            # policy: Any
            
            cfn_rest_api_props = apigateway.CfnRestApiProps(
                api_key_source_type="apiKeySourceType",
                binary_media_types=["binaryMediaTypes"],
                body=body,
                body_s3_location=apigateway.CfnRestApi.S3LocationProperty(
                    bucket="bucket",
                    e_tag="eTag",
                    key="key",
                    version="version"
                ),
                clone_from="cloneFrom",
                description="description",
                disable_execute_api_endpoint=False,
                endpoint_configuration=apigateway.CfnRestApi.EndpointConfigurationProperty(
                    ip_address_type="ipAddressType",
                    types=["types"],
                    vpc_endpoint_ids=["vpcEndpointIds"]
                ),
                fail_on_warnings=False,
                minimum_compression_size=123,
                mode="mode",
                name="name",
                parameters={
                    "parameters_key": "parameters"
                },
                policy=policy,
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5e6d4ed4722f0c07021ff207940f8656b0bbeb24224daef1a2a167c70b6cacc3)
            check_type(argname="argument api_key_source_type", value=api_key_source_type, expected_type=type_hints["api_key_source_type"])
            check_type(argname="argument binary_media_types", value=binary_media_types, expected_type=type_hints["binary_media_types"])
            check_type(argname="argument body", value=body, expected_type=type_hints["body"])
            check_type(argname="argument body_s3_location", value=body_s3_location, expected_type=type_hints["body_s3_location"])
            check_type(argname="argument clone_from", value=clone_from, expected_type=type_hints["clone_from"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument disable_execute_api_endpoint", value=disable_execute_api_endpoint, expected_type=type_hints["disable_execute_api_endpoint"])
            check_type(argname="argument endpoint_configuration", value=endpoint_configuration, expected_type=type_hints["endpoint_configuration"])
            check_type(argname="argument fail_on_warnings", value=fail_on_warnings, expected_type=type_hints["fail_on_warnings"])
            check_type(argname="argument minimum_compression_size", value=minimum_compression_size, expected_type=type_hints["minimum_compression_size"])
            check_type(argname="argument mode", value=mode, expected_type=type_hints["mode"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument parameters", value=parameters, expected_type=type_hints["parameters"])
            check_type(argname="argument policy", value=policy, expected_type=type_hints["policy"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if api_key_source_type is not None:
            self._values["api_key_source_type"] = api_key_source_type
        if binary_media_types is not None:
            self._values["binary_media_types"] = binary_media_types
        if body is not None:
            self._values["body"] = body
        if body_s3_location is not None:
            self._values["body_s3_location"] = body_s3_location
        if clone_from is not None:
            self._values["clone_from"] = clone_from
        if description is not None:
            self._values["description"] = description
        if disable_execute_api_endpoint is not None:
            self._values["disable_execute_api_endpoint"] = disable_execute_api_endpoint
        if endpoint_configuration is not None:
            self._values["endpoint_configuration"] = endpoint_configuration
        if fail_on_warnings is not None:
            self._values["fail_on_warnings"] = fail_on_warnings
        if minimum_compression_size is not None:
            self._values["minimum_compression_size"] = minimum_compression_size
        if mode is not None:
            self._values["mode"] = mode
        if name is not None:
            self._values["name"] = name
        if parameters is not None:
            self._values["parameters"] = parameters
        if policy is not None:
            self._values["policy"] = policy
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def api_key_source_type(self) -> typing.Optional[builtins.str]:
        '''The source of the API key for metering requests according to a usage plan.

        Valid values are: ``HEADER`` to read the API key from the ``X-API-Key`` header of a request. ``AUTHORIZER`` to read the API key from the ``UsageIdentifierKey`` from a custom authorizer.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-apikeysourcetype
        '''
        result = self._values.get("api_key_source_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def binary_media_types(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The list of binary media types supported by the RestApi.

        By default, the RestApi supports only UTF-8-encoded text payloads.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-binarymediatypes
        '''
        result = self._values.get("binary_media_types")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def body(self) -> typing.Any:
        '''An OpenAPI specification that defines a set of RESTful APIs in JSON format.

        For YAML templates, you can also provide the specification in YAML format.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-body
        '''
        result = self._values.get("body")
        return typing.cast(typing.Any, result)

    @builtins.property
    def body_s3_location(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnRestApi.S3LocationProperty]]:
        '''The Amazon Simple Storage Service (Amazon S3) location that points to an OpenAPI file, which defines a set of RESTful APIs in JSON or YAML format.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-bodys3location
        '''
        result = self._values.get("body_s3_location")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnRestApi.S3LocationProperty]], result)

    @builtins.property
    def clone_from(self) -> typing.Optional[builtins.str]:
        '''The ID of the RestApi that you want to clone from.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-clonefrom
        '''
        result = self._values.get("clone_from")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def disable_execute_api_endpoint(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether clients can invoke your API by using the default ``execute-api`` endpoint.

        By default, clients can invoke your API with the default ``https://{api_id}.execute-api.{region}.amazonaws.com`` endpoint. To require that clients use a custom domain name to invoke your API, disable the default endpoint

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-disableexecuteapiendpoint
        '''
        result = self._values.get("disable_execute_api_endpoint")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def endpoint_configuration(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnRestApi.EndpointConfigurationProperty]]:
        '''A list of the endpoint types and IP address types of the API.

        Use this property when creating an API. When importing an existing API, specify the endpoint configuration types using the ``Parameters`` property.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-endpointconfiguration
        '''
        result = self._values.get("endpoint_configuration")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnRestApi.EndpointConfigurationProperty]], result)

    @builtins.property
    def fail_on_warnings(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''A query parameter to indicate whether to rollback the API update ( ``true`` ) or not ( ``false`` ) when a warning is encountered.

        The default value is ``false`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-failonwarnings
        '''
        result = self._values.get("fail_on_warnings")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def minimum_compression_size(self) -> typing.Optional[jsii.Number]:
        '''A nullable integer that is used to enable compression (with non-negative between 0 and 10485760 (10M) bytes, inclusive) or disable compression (with a null value) on an API.

        When compression is enabled, compression or decompression is not applied on the payload if the payload size is smaller than this value. Setting it to zero allows compression for any payload size.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-minimumcompressionsize
        '''
        result = self._values.get("minimum_compression_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def mode(self) -> typing.Optional[builtins.str]:
        '''This property applies only when you use OpenAPI to define your REST API.

        The ``Mode`` determines how API Gateway handles resource updates.

        Valid values are ``overwrite`` or ``merge`` .

        For ``overwrite`` , the new API definition replaces the existing one. The existing API identifier remains unchanged.

        For ``merge`` , the new API definition is merged with the existing API.

        If you don't specify this property, a default value is chosen. For REST APIs created before March 29, 2021, the default is ``overwrite`` . For REST APIs created after March 29, 2021, the new API definition takes precedence, but any container types such as endpoint configurations and binary media types are merged with the existing API.

        Use the default mode to define top-level ``RestApi`` properties in addition to using OpenAPI. Generally, it's preferred to use API Gateway's OpenAPI extensions to model these properties.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-mode
        '''
        result = self._values.get("mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of the RestApi.

        A name is required if the REST API is not based on an OpenAPI specification.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''Custom header parameters as part of the request.

        For example, to exclude DocumentationParts from an imported API, set ``ignore=documentation`` as a ``parameters`` value, as in the AWS CLI command of ``aws apigateway import-rest-api --parameters ignore=documentation --body 'file:///path/to/imported-api-body.json'`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-parameters
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

    @builtins.property
    def policy(self) -> typing.Any:
        '''A policy document that contains the permissions for the ``RestApi`` resource.

        To set the ARN for the policy, use the ``!Join`` intrinsic function with ``""`` as delimiter and values of ``"execute-api:/"`` and ``"*"`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-policy
        '''
        result = self._values.get("policy")
        return typing.cast(typing.Any, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The key-value map of strings.

        The valid character set is [a-zA-Z+-=._:/]. The tag key can be up to 128 characters and must not start with ``aws:`` . The tag value can be up to 256 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html#cfn-apigateway-restapi-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRestApiProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnStage(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnStage",
):
    '''The ``AWS::ApiGateway::Stage`` resource creates a stage for a deployment.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html
    :cloudformationResource: AWS::ApiGateway::Stage
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_stage = apigateway.CfnStage(self, "MyCfnStage",
            rest_api_id="restApiId",
        
            # the properties below are optional
            access_log_setting=apigateway.CfnStage.AccessLogSettingProperty(
                destination_arn="destinationArn",
                format="format"
            ),
            cache_cluster_enabled=False,
            cache_cluster_size="cacheClusterSize",
            canary_setting=apigateway.CfnStage.CanarySettingProperty(
                deployment_id="deploymentId",
                percent_traffic=123,
                stage_variable_overrides={
                    "stage_variable_overrides_key": "stageVariableOverrides"
                },
                use_stage_cache=False
            ),
            client_certificate_id="clientCertificateId",
            deployment_id="deploymentId",
            description="description",
            documentation_version="documentationVersion",
            method_settings=[apigateway.CfnStage.MethodSettingProperty(
                cache_data_encrypted=False,
                cache_ttl_in_seconds=123,
                caching_enabled=False,
                data_trace_enabled=False,
                http_method="httpMethod",
                logging_level="loggingLevel",
                metrics_enabled=False,
                resource_path="resourcePath",
                throttling_burst_limit=123,
                throttling_rate_limit=123
            )],
            stage_name="stageName",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            tracing_enabled=False,
            variables={
                "variables_key": "variables"
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        rest_api_id: builtins.str,
        access_log_setting: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnStage.AccessLogSettingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_cluster_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        cache_cluster_size: typing.Optional[builtins.str] = None,
        canary_setting: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnStage.CanarySettingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        client_certificate_id: typing.Optional[builtins.str] = None,
        deployment_id: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        documentation_version: typing.Optional[builtins.str] = None,
        method_settings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnStage.MethodSettingProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        stage_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        tracing_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        variables: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param rest_api_id: The string identifier of the associated RestApi.
        :param access_log_setting: Access log settings, including the access log format and access log destination ARN.
        :param cache_cluster_enabled: Specifies whether a cache cluster is enabled for the stage. To activate a method-level cache, set ``CachingEnabled`` to ``true`` for a method.
        :param cache_cluster_size: The stage's cache capacity in GB. For more information about choosing a cache size, see `Enabling API caching to enhance responsiveness <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html>`_ .
        :param canary_setting: Settings for the canary deployment in this stage.
        :param client_certificate_id: The identifier of a client certificate for an API stage.
        :param deployment_id: The identifier of the Deployment that the stage points to.
        :param description: The stage's description.
        :param documentation_version: The version of the associated API documentation.
        :param method_settings: A map that defines the method settings for a Stage resource. Keys (designated as ``/{method_setting_key`` below) are method paths defined as ``{resource_path}/{http_method}`` for an individual method override, or ``/\\* /\\*`` for overriding all methods in the stage.
        :param stage_name: The name of the stage is the first path segment in the Uniform Resource Identifier (URI) of a call to API Gateway. Stage names can only contain alphanumeric characters, hyphens, and underscores. Maximum length is 128 characters.
        :param tags: The collection of tags. Each tag element is associated with a given resource.
        :param tracing_enabled: Specifies whether active tracing with X-ray is enabled for the Stage.
        :param variables: A map (string-to-string map) that defines the stage variables, where the variable name is the key and the variable value is the value. Variable names are limited to alphanumeric characters. Values must match the following regular expression: ``[A-Za-z0-9-._~:/?#&=,]+`` .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2c8ae05335242844b55899042c4d5f05d3d40ee60a7702fbc9345ea651a23165)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnStageProps(
            rest_api_id=rest_api_id,
            access_log_setting=access_log_setting,
            cache_cluster_enabled=cache_cluster_enabled,
            cache_cluster_size=cache_cluster_size,
            canary_setting=canary_setting,
            client_certificate_id=client_certificate_id,
            deployment_id=deployment_id,
            description=description,
            documentation_version=documentation_version,
            method_settings=method_settings,
            stage_name=stage_name,
            tags=tags,
            tracing_enabled=tracing_enabled,
            variables=variables,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__469891369dafdfc17786a2c499d65a940d33c0956162ee2765c7a1cdf3b44301)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aa039d741e4fec58df47a18424187e57e8546ede5664ac483f2f2e3e4c7de589)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.'''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @rest_api_id.setter
    def rest_api_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9ca229a168b1b0b5ce09bd9b4e98305cde5990bd63de169188c68b738fe0d449)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "restApiId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="accessLogSetting")
    def access_log_setting(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnStage.AccessLogSettingProperty"]]:
        '''Access log settings, including the access log format and access log destination ARN.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnStage.AccessLogSettingProperty"]], jsii.get(self, "accessLogSetting"))

    @access_log_setting.setter
    def access_log_setting(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnStage.AccessLogSettingProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4ec6e534bced79a9cafcc6abbc05784c0cb0288cb0ceddf15685c1b08de26154)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "accessLogSetting", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="cacheClusterEnabled")
    def cache_cluster_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether a cache cluster is enabled for the stage.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "cacheClusterEnabled"))

    @cache_cluster_enabled.setter
    def cache_cluster_enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3ee668c29f68c01d258ef4be4dd02663e6079d12d2e7a426bb59ee67ced9e11a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cacheClusterEnabled", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="cacheClusterSize")
    def cache_cluster_size(self) -> typing.Optional[builtins.str]:
        '''The stage's cache capacity in GB.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cacheClusterSize"))

    @cache_cluster_size.setter
    def cache_cluster_size(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__00e1683921bb5cd99260b5eecf65249ad0c7ec02b74f9d20b2cbc171292c3f7d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cacheClusterSize", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="canarySetting")
    def canary_setting(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnStage.CanarySettingProperty"]]:
        '''Settings for the canary deployment in this stage.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnStage.CanarySettingProperty"]], jsii.get(self, "canarySetting"))

    @canary_setting.setter
    def canary_setting(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnStage.CanarySettingProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9d5d7d1801b50fbfd2d815556acd623df5f38aeb7411e52819b30b70c8f8c10e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "canarySetting", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="clientCertificateId")
    def client_certificate_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of a client certificate for an API stage.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "clientCertificateId"))

    @client_certificate_id.setter
    def client_certificate_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__049bb95685cc5927fe6ddc6e401647bcd58f7e6ed1fb57110891f9f362d1f301)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "clientCertificateId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="deploymentId")
    def deployment_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of the Deployment that the stage points to.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "deploymentId"))

    @deployment_id.setter
    def deployment_id(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fba2567d2b966821e080aba32b1cd088bc600d1029a4676851f4d300fa7cc1fa)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deploymentId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The stage's description.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e1861f79324f8dab5999eb9db261a7d96b7415bdcdadd56b5ef15d15c387bc02)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="documentationVersion")
    def documentation_version(self) -> typing.Optional[builtins.str]:
        '''The version of the associated API documentation.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "documentationVersion"))

    @documentation_version.setter
    def documentation_version(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__debfee4245a2fcafcc1e179b7f7e2fc02357e6b0b2cd14accda3101b89c0d0ae)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "documentationVersion", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="methodSettings")
    def method_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnStage.MethodSettingProperty"]]]]:
        '''A map that defines the method settings for a Stage resource.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnStage.MethodSettingProperty"]]]], jsii.get(self, "methodSettings"))

    @method_settings.setter
    def method_settings(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnStage.MethodSettingProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e6c0cd8393b418858c9216e7b06705c7834d5bc728b465c401f21e5ea5f71be)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "methodSettings", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''The name of the stage is the first path segment in the Uniform Resource Identifier (URI) of a call to API Gateway.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))

    @stage_name.setter
    def stage_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6a7c85c7f5a361a8304fa7bb7fcaed008b51e0e42a47c71561aa783f667e9b76)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "stageName", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__dfc53c1ffa80a62c546222c14af688ddd25a1a97b9dea37ac1ed4738123c0369)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tracingEnabled")
    def tracing_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether active tracing with X-ray is enabled for the Stage.'''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], jsii.get(self, "tracingEnabled"))

    @tracing_enabled.setter
    def tracing_enabled(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__080398f14f82f9de64dfd739f23cedcc4a0465bf221903bff436a81eae9e6934)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tracingEnabled", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="variables")
    def variables(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''A map (string-to-string map) that defines the stage variables, where the variable name is the key and the variable value is the value.'''
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], jsii.get(self, "variables"))

    @variables.setter
    def variables(
        self,
        value: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0ce3ef85d7e159e99ea72997d34d02601f6ebfa6c900e82024469d23fa9822c8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "variables", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnStage.AccessLogSettingProperty",
        jsii_struct_bases=[],
        name_mapping={"destination_arn": "destinationArn", "format": "format"},
    )
    class AccessLogSettingProperty:
        def __init__(
            self,
            *,
            destination_arn: typing.Optional[builtins.str] = None,
            format: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``AccessLogSetting`` property type specifies settings for logging access in this stage.

            ``AccessLogSetting`` is a property of the `AWS::ApiGateway::Stage <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html>`_ resource.

            :param destination_arn: The Amazon Resource Name (ARN) of the CloudWatch Logs log group or Kinesis Data Firehose delivery stream to receive access logs. If you specify a Kinesis Data Firehose delivery stream, the stream name must begin with ``amazon-apigateway-`` . This parameter is required to enable access logging.
            :param format: A single line format of the access logs of data, as specified by selected `$context variables <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference>`_ . The format must include at least ``$context.requestId`` . This parameter is required to enable access logging.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-accesslogsetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                access_log_setting_property = apigateway.CfnStage.AccessLogSettingProperty(
                    destination_arn="destinationArn",
                    format="format"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9a1674b8f540cdd4aa3d8ecc567b04066a13d800cb4fccce0c187f438bdb096a)
                check_type(argname="argument destination_arn", value=destination_arn, expected_type=type_hints["destination_arn"])
                check_type(argname="argument format", value=format, expected_type=type_hints["format"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if destination_arn is not None:
                self._values["destination_arn"] = destination_arn
            if format is not None:
                self._values["format"] = format

        @builtins.property
        def destination_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the CloudWatch Logs log group or Kinesis Data Firehose delivery stream to receive access logs.

            If you specify a Kinesis Data Firehose delivery stream, the stream name must begin with ``amazon-apigateway-`` . This parameter is required to enable access logging.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-accesslogsetting.html#cfn-apigateway-stage-accesslogsetting-destinationarn
            '''
            result = self._values.get("destination_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def format(self) -> typing.Optional[builtins.str]:
            '''A single line format of the access logs of data, as specified by selected `$context variables <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#context-variable-reference>`_ . The format must include at least ``$context.requestId`` . This parameter is required to enable access logging.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-accesslogsetting.html#cfn-apigateway-stage-accesslogsetting-format
            '''
            result = self._values.get("format")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessLogSettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnStage.CanarySettingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "deployment_id": "deploymentId",
            "percent_traffic": "percentTraffic",
            "stage_variable_overrides": "stageVariableOverrides",
            "use_stage_cache": "useStageCache",
        },
    )
    class CanarySettingProperty:
        def __init__(
            self,
            *,
            deployment_id: typing.Optional[builtins.str] = None,
            percent_traffic: typing.Optional[jsii.Number] = None,
            stage_variable_overrides: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
            use_stage_cache: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        ) -> None:
            '''Configuration settings of a canary deployment.

            :param deployment_id: The ID of the canary deployment.
            :param percent_traffic: The percent (0-100) of traffic diverted to a canary deployment.
            :param stage_variable_overrides: Stage variables overridden for a canary release deployment, including new stage variables introduced in the canary. These stage variables are represented as a string-to-string map between stage variable names and their values.
            :param use_stage_cache: A Boolean flag to indicate whether the canary deployment uses the stage cache or not.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-canarysetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                canary_setting_property = apigateway.CfnStage.CanarySettingProperty(
                    deployment_id="deploymentId",
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0a0fb3474ba60447840d49c0eab8e221e5f81e4477cd328ee3967edc1496b414)
                check_type(argname="argument deployment_id", value=deployment_id, expected_type=type_hints["deployment_id"])
                check_type(argname="argument percent_traffic", value=percent_traffic, expected_type=type_hints["percent_traffic"])
                check_type(argname="argument stage_variable_overrides", value=stage_variable_overrides, expected_type=type_hints["stage_variable_overrides"])
                check_type(argname="argument use_stage_cache", value=use_stage_cache, expected_type=type_hints["use_stage_cache"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if deployment_id is not None:
                self._values["deployment_id"] = deployment_id
            if percent_traffic is not None:
                self._values["percent_traffic"] = percent_traffic
            if stage_variable_overrides is not None:
                self._values["stage_variable_overrides"] = stage_variable_overrides
            if use_stage_cache is not None:
                self._values["use_stage_cache"] = use_stage_cache

        @builtins.property
        def deployment_id(self) -> typing.Optional[builtins.str]:
            '''The ID of the canary deployment.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-canarysetting.html#cfn-apigateway-stage-canarysetting-deploymentid
            '''
            result = self._values.get("deployment_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def percent_traffic(self) -> typing.Optional[jsii.Number]:
            '''The percent (0-100) of traffic diverted to a canary deployment.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-canarysetting.html#cfn-apigateway-stage-canarysetting-percenttraffic
            '''
            result = self._values.get("percent_traffic")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def stage_variable_overrides(
            self,
        ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
            '''Stage variables overridden for a canary release deployment, including new stage variables introduced in the canary.

            These stage variables are represented as a string-to-string map between stage variable names and their values.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-canarysetting.html#cfn-apigateway-stage-canarysetting-stagevariableoverrides
            '''
            result = self._values.get("stage_variable_overrides")
            return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

        @builtins.property
        def use_stage_cache(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''A Boolean flag to indicate whether the canary deployment uses the stage cache or not.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-canarysetting.html#cfn-apigateway-stage-canarysetting-usestagecache
            '''
            result = self._values.get("use_stage_cache")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CanarySettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnStage.MethodSettingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cache_data_encrypted": "cacheDataEncrypted",
            "cache_ttl_in_seconds": "cacheTtlInSeconds",
            "caching_enabled": "cachingEnabled",
            "data_trace_enabled": "dataTraceEnabled",
            "http_method": "httpMethod",
            "logging_level": "loggingLevel",
            "metrics_enabled": "metricsEnabled",
            "resource_path": "resourcePath",
            "throttling_burst_limit": "throttlingBurstLimit",
            "throttling_rate_limit": "throttlingRateLimit",
        },
    )
    class MethodSettingProperty:
        def __init__(
            self,
            *,
            cache_data_encrypted: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            cache_ttl_in_seconds: typing.Optional[jsii.Number] = None,
            caching_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            data_trace_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            http_method: typing.Optional[builtins.str] = None,
            logging_level: typing.Optional[builtins.str] = None,
            metrics_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
            resource_path: typing.Optional[builtins.str] = None,
            throttling_burst_limit: typing.Optional[jsii.Number] = None,
            throttling_rate_limit: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``MethodSetting`` property type configures settings for all methods in a stage.

            The ``MethodSettings`` property of the ``AWS::ApiGateway::Stage`` resource contains a list of ``MethodSetting`` property types.

            :param cache_data_encrypted: Specifies whether the cached responses are encrypted.
            :param cache_ttl_in_seconds: Specifies the time to live (TTL), in seconds, for cached responses. The higher the TTL, the longer the response will be cached.
            :param caching_enabled: Specifies whether responses should be cached and returned for requests. A cache cluster must be enabled on the stage for responses to be cached.
            :param data_trace_enabled: Specifies whether data trace logging is enabled for this method, which affects the log entries pushed to Amazon CloudWatch Logs. This can be useful to troubleshoot APIs, but can result in logging sensitive data. We recommend that you don't enable this option for production APIs.
            :param http_method: The HTTP method. To apply settings to multiple resources and methods, specify an asterisk ( ``*`` ) for the ``HttpMethod`` and ``/*`` for the ``ResourcePath`` . This parameter is required when you specify a ``MethodSetting`` .
            :param logging_level: Specifies the logging level for this method, which affects the log entries pushed to Amazon CloudWatch Logs. Valid values are ``OFF`` , ``ERROR`` , and ``INFO`` . Choose ``ERROR`` to write only error-level entries to CloudWatch Logs, or choose ``INFO`` to include all ``ERROR`` events as well as extra informational events.
            :param metrics_enabled: Specifies whether Amazon CloudWatch metrics are enabled for this method.
            :param resource_path: The resource path for this method. Forward slashes ( ``/`` ) are encoded as ``~1`` and the initial slash must include a forward slash. For example, the path value ``/resource/subresource`` must be encoded as ``/~1resource~1subresource`` . To specify the root path, use only a slash ( ``/`` ). To apply settings to multiple resources and methods, specify an asterisk ( ``*`` ) for the ``HttpMethod`` and ``/*`` for the ``ResourcePath`` . This parameter is required when you specify a ``MethodSetting`` .
            :param throttling_burst_limit: Specifies the throttling burst limit.
            :param throttling_rate_limit: Specifies the throttling rate limit.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                method_setting_property = apigateway.CfnStage.MethodSettingProperty(
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    data_trace_enabled=False,
                    http_method="httpMethod",
                    logging_level="loggingLevel",
                    metrics_enabled=False,
                    resource_path="resourcePath",
                    throttling_burst_limit=123,
                    throttling_rate_limit=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__93f3e57e2c3b961aa0f733a06ed42f8a363845df26d6b26e0642d5e49ab8aecc)
                check_type(argname="argument cache_data_encrypted", value=cache_data_encrypted, expected_type=type_hints["cache_data_encrypted"])
                check_type(argname="argument cache_ttl_in_seconds", value=cache_ttl_in_seconds, expected_type=type_hints["cache_ttl_in_seconds"])
                check_type(argname="argument caching_enabled", value=caching_enabled, expected_type=type_hints["caching_enabled"])
                check_type(argname="argument data_trace_enabled", value=data_trace_enabled, expected_type=type_hints["data_trace_enabled"])
                check_type(argname="argument http_method", value=http_method, expected_type=type_hints["http_method"])
                check_type(argname="argument logging_level", value=logging_level, expected_type=type_hints["logging_level"])
                check_type(argname="argument metrics_enabled", value=metrics_enabled, expected_type=type_hints["metrics_enabled"])
                check_type(argname="argument resource_path", value=resource_path, expected_type=type_hints["resource_path"])
                check_type(argname="argument throttling_burst_limit", value=throttling_burst_limit, expected_type=type_hints["throttling_burst_limit"])
                check_type(argname="argument throttling_rate_limit", value=throttling_rate_limit, expected_type=type_hints["throttling_rate_limit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if cache_data_encrypted is not None:
                self._values["cache_data_encrypted"] = cache_data_encrypted
            if cache_ttl_in_seconds is not None:
                self._values["cache_ttl_in_seconds"] = cache_ttl_in_seconds
            if caching_enabled is not None:
                self._values["caching_enabled"] = caching_enabled
            if data_trace_enabled is not None:
                self._values["data_trace_enabled"] = data_trace_enabled
            if http_method is not None:
                self._values["http_method"] = http_method
            if logging_level is not None:
                self._values["logging_level"] = logging_level
            if metrics_enabled is not None:
                self._values["metrics_enabled"] = metrics_enabled
            if resource_path is not None:
                self._values["resource_path"] = resource_path
            if throttling_burst_limit is not None:
                self._values["throttling_burst_limit"] = throttling_burst_limit
            if throttling_rate_limit is not None:
                self._values["throttling_rate_limit"] = throttling_rate_limit

        @builtins.property
        def cache_data_encrypted(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether the cached responses are encrypted.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-cachedataencrypted
            '''
            result = self._values.get("cache_data_encrypted")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def cache_ttl_in_seconds(self) -> typing.Optional[jsii.Number]:
            '''Specifies the time to live (TTL), in seconds, for cached responses.

            The higher the TTL, the longer the response will be cached.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-cachettlinseconds
            '''
            result = self._values.get("cache_ttl_in_seconds")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def caching_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether responses should be cached and returned for requests.

            A cache cluster must be enabled on the stage for responses to be cached.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-cachingenabled
            '''
            result = self._values.get("caching_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def data_trace_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether data trace logging is enabled for this method, which affects the log entries pushed to Amazon CloudWatch Logs.

            This can be useful to troubleshoot APIs, but can result in logging sensitive data. We recommend that you don't enable this option for production APIs.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-datatraceenabled
            '''
            result = self._values.get("data_trace_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def http_method(self) -> typing.Optional[builtins.str]:
            '''The HTTP method.

            To apply settings to multiple resources and methods, specify an asterisk ( ``*`` ) for the ``HttpMethod`` and ``/*`` for the ``ResourcePath`` . This parameter is required when you specify a ``MethodSetting`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-httpmethod
            '''
            result = self._values.get("http_method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def logging_level(self) -> typing.Optional[builtins.str]:
            '''Specifies the logging level for this method, which affects the log entries pushed to Amazon CloudWatch Logs.

            Valid values are ``OFF`` , ``ERROR`` , and ``INFO`` . Choose ``ERROR`` to write only error-level entries to CloudWatch Logs, or choose ``INFO`` to include all ``ERROR`` events as well as extra informational events.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-logginglevel
            '''
            result = self._values.get("logging_level")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def metrics_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
            '''Specifies whether Amazon CloudWatch metrics are enabled for this method.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-metricsenabled
            '''
            result = self._values.get("metrics_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

        @builtins.property
        def resource_path(self) -> typing.Optional[builtins.str]:
            '''The resource path for this method.

            Forward slashes ( ``/`` ) are encoded as ``~1`` and the initial slash must include a forward slash. For example, the path value ``/resource/subresource`` must be encoded as ``/~1resource~1subresource`` . To specify the root path, use only a slash ( ``/`` ). To apply settings to multiple resources and methods, specify an asterisk ( ``*`` ) for the ``HttpMethod`` and ``/*`` for the ``ResourcePath`` . This parameter is required when you specify a ``MethodSetting`` .

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-resourcepath
            '''
            result = self._values.get("resource_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def throttling_burst_limit(self) -> typing.Optional[jsii.Number]:
            '''Specifies the throttling burst limit.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-throttlingburstlimit
            '''
            result = self._values.get("throttling_burst_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def throttling_rate_limit(self) -> typing.Optional[jsii.Number]:
            '''Specifies the throttling rate limit.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-stage-methodsetting.html#cfn-apigateway-stage-methodsetting-throttlingratelimit
            '''
            result = self._values.get("throttling_rate_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MethodSettingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnStageProps",
    jsii_struct_bases=[],
    name_mapping={
        "rest_api_id": "restApiId",
        "access_log_setting": "accessLogSetting",
        "cache_cluster_enabled": "cacheClusterEnabled",
        "cache_cluster_size": "cacheClusterSize",
        "canary_setting": "canarySetting",
        "client_certificate_id": "clientCertificateId",
        "deployment_id": "deploymentId",
        "description": "description",
        "documentation_version": "documentationVersion",
        "method_settings": "methodSettings",
        "stage_name": "stageName",
        "tags": "tags",
        "tracing_enabled": "tracingEnabled",
        "variables": "variables",
    },
)
class CfnStageProps:
    def __init__(
        self,
        *,
        rest_api_id: builtins.str,
        access_log_setting: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnStage.AccessLogSettingProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        cache_cluster_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        cache_cluster_size: typing.Optional[builtins.str] = None,
        canary_setting: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnStage.CanarySettingProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        client_certificate_id: typing.Optional[builtins.str] = None,
        deployment_id: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        documentation_version: typing.Optional[builtins.str] = None,
        method_settings: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnStage.MethodSettingProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        stage_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        tracing_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]] = None,
        variables: typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]] = None,
    ) -> None:
        '''Properties for defining a ``CfnStage``.

        :param rest_api_id: The string identifier of the associated RestApi.
        :param access_log_setting: Access log settings, including the access log format and access log destination ARN.
        :param cache_cluster_enabled: Specifies whether a cache cluster is enabled for the stage. To activate a method-level cache, set ``CachingEnabled`` to ``true`` for a method.
        :param cache_cluster_size: The stage's cache capacity in GB. For more information about choosing a cache size, see `Enabling API caching to enhance responsiveness <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html>`_ .
        :param canary_setting: Settings for the canary deployment in this stage.
        :param client_certificate_id: The identifier of a client certificate for an API stage.
        :param deployment_id: The identifier of the Deployment that the stage points to.
        :param description: The stage's description.
        :param documentation_version: The version of the associated API documentation.
        :param method_settings: A map that defines the method settings for a Stage resource. Keys (designated as ``/{method_setting_key`` below) are method paths defined as ``{resource_path}/{http_method}`` for an individual method override, or ``/\\* /\\*`` for overriding all methods in the stage.
        :param stage_name: The name of the stage is the first path segment in the Uniform Resource Identifier (URI) of a call to API Gateway. Stage names can only contain alphanumeric characters, hyphens, and underscores. Maximum length is 128 characters.
        :param tags: The collection of tags. Each tag element is associated with a given resource.
        :param tracing_enabled: Specifies whether active tracing with X-ray is enabled for the Stage.
        :param variables: A map (string-to-string map) that defines the stage variables, where the variable name is the key and the variable value is the value. Variable names are limited to alphanumeric characters. Values must match the following regular expression: ``[A-Za-z0-9-._~:/?#&=,]+`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_stage_props = apigateway.CfnStageProps(
                rest_api_id="restApiId",
            
                # the properties below are optional
                access_log_setting=apigateway.CfnStage.AccessLogSettingProperty(
                    destination_arn="destinationArn",
                    format="format"
                ),
                cache_cluster_enabled=False,
                cache_cluster_size="cacheClusterSize",
                canary_setting=apigateway.CfnStage.CanarySettingProperty(
                    deployment_id="deploymentId",
                    percent_traffic=123,
                    stage_variable_overrides={
                        "stage_variable_overrides_key": "stageVariableOverrides"
                    },
                    use_stage_cache=False
                ),
                client_certificate_id="clientCertificateId",
                deployment_id="deploymentId",
                description="description",
                documentation_version="documentationVersion",
                method_settings=[apigateway.CfnStage.MethodSettingProperty(
                    cache_data_encrypted=False,
                    cache_ttl_in_seconds=123,
                    caching_enabled=False,
                    data_trace_enabled=False,
                    http_method="httpMethod",
                    logging_level="loggingLevel",
                    metrics_enabled=False,
                    resource_path="resourcePath",
                    throttling_burst_limit=123,
                    throttling_rate_limit=123
                )],
                stage_name="stageName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                tracing_enabled=False,
                variables={
                    "variables_key": "variables"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a1ebc5406497577cb7c3919d6f6bedaf66797e69790cd7880fb0110e4babdf44)
            check_type(argname="argument rest_api_id", value=rest_api_id, expected_type=type_hints["rest_api_id"])
            check_type(argname="argument access_log_setting", value=access_log_setting, expected_type=type_hints["access_log_setting"])
            check_type(argname="argument cache_cluster_enabled", value=cache_cluster_enabled, expected_type=type_hints["cache_cluster_enabled"])
            check_type(argname="argument cache_cluster_size", value=cache_cluster_size, expected_type=type_hints["cache_cluster_size"])
            check_type(argname="argument canary_setting", value=canary_setting, expected_type=type_hints["canary_setting"])
            check_type(argname="argument client_certificate_id", value=client_certificate_id, expected_type=type_hints["client_certificate_id"])
            check_type(argname="argument deployment_id", value=deployment_id, expected_type=type_hints["deployment_id"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument documentation_version", value=documentation_version, expected_type=type_hints["documentation_version"])
            check_type(argname="argument method_settings", value=method_settings, expected_type=type_hints["method_settings"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument tracing_enabled", value=tracing_enabled, expected_type=type_hints["tracing_enabled"])
            check_type(argname="argument variables", value=variables, expected_type=type_hints["variables"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "rest_api_id": rest_api_id,
        }
        if access_log_setting is not None:
            self._values["access_log_setting"] = access_log_setting
        if cache_cluster_enabled is not None:
            self._values["cache_cluster_enabled"] = cache_cluster_enabled
        if cache_cluster_size is not None:
            self._values["cache_cluster_size"] = cache_cluster_size
        if canary_setting is not None:
            self._values["canary_setting"] = canary_setting
        if client_certificate_id is not None:
            self._values["client_certificate_id"] = client_certificate_id
        if deployment_id is not None:
            self._values["deployment_id"] = deployment_id
        if description is not None:
            self._values["description"] = description
        if documentation_version is not None:
            self._values["documentation_version"] = documentation_version
        if method_settings is not None:
            self._values["method_settings"] = method_settings
        if stage_name is not None:
            self._values["stage_name"] = stage_name
        if tags is not None:
            self._values["tags"] = tags
        if tracing_enabled is not None:
            self._values["tracing_enabled"] = tracing_enabled
        if variables is not None:
            self._values["variables"] = variables

    @builtins.property
    def rest_api_id(self) -> builtins.str:
        '''The string identifier of the associated RestApi.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-restapiid
        '''
        result = self._values.get("rest_api_id")
        assert result is not None, "Required property 'rest_api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_log_setting(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnStage.AccessLogSettingProperty]]:
        '''Access log settings, including the access log format and access log destination ARN.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-accesslogsetting
        '''
        result = self._values.get("access_log_setting")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnStage.AccessLogSettingProperty]], result)

    @builtins.property
    def cache_cluster_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether a cache cluster is enabled for the stage.

        To activate a method-level cache, set ``CachingEnabled`` to ``true`` for a method.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-cacheclusterenabled
        '''
        result = self._values.get("cache_cluster_enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def cache_cluster_size(self) -> typing.Optional[builtins.str]:
        '''The stage's cache capacity in GB.

        For more information about choosing a cache size, see `Enabling API caching to enhance responsiveness <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html>`_ .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-cacheclustersize
        '''
        result = self._values.get("cache_cluster_size")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def canary_setting(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnStage.CanarySettingProperty]]:
        '''Settings for the canary deployment in this stage.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-canarysetting
        '''
        result = self._values.get("canary_setting")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnStage.CanarySettingProperty]], result)

    @builtins.property
    def client_certificate_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of a client certificate for an API stage.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-clientcertificateid
        '''
        result = self._values.get("client_certificate_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def deployment_id(self) -> typing.Optional[builtins.str]:
        '''The identifier of the Deployment that the stage points to.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-deploymentid
        '''
        result = self._values.get("deployment_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The stage's description.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def documentation_version(self) -> typing.Optional[builtins.str]:
        '''The version of the associated API documentation.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-documentationversion
        '''
        result = self._values.get("documentation_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def method_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnStage.MethodSettingProperty]]]]:
        '''A map that defines the method settings for a Stage resource.

        Keys (designated as ``/{method_setting_key`` below) are method paths defined as ``{resource_path}/{http_method}`` for an individual method override, or ``/\\* /\\*`` for overriding all methods in the stage.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-methodsettings
        '''
        result = self._values.get("method_settings")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnStage.MethodSettingProperty]]]], result)

    @builtins.property
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''The name of the stage is the first path segment in the Uniform Resource Identifier (URI) of a call to API Gateway.

        Stage names can only contain alphanumeric characters, hyphens, and underscores. Maximum length is 128 characters.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-stagename
        '''
        result = self._values.get("stage_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.

        Each tag element is associated with a given resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def tracing_enabled(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]]:
        '''Specifies whether active tracing with X-ray is enabled for the Stage.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-tracingenabled
        '''
        result = self._values.get("tracing_enabled")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_da3f097b]], result)

    @builtins.property
    def variables(
        self,
    ) -> typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]]:
        '''A map (string-to-string map) that defines the stage variables, where the variable name is the key and the variable value is the value.

        Variable names are limited to alphanumeric characters. Values must match the following regular expression: ``[A-Za-z0-9-._~:/?#&=,]+`` .

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-stage.html#cfn-apigateway-stage-variables
        '''
        result = self._values.get("variables")
        return typing.cast(typing.Optional[typing.Union[typing.Mapping[builtins.str, builtins.str], _IResolvable_da3f097b]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnStageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnUsagePlan(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnUsagePlan",
):
    '''The ``AWS::ApiGateway::UsagePlan`` resource creates a usage plan for deployed APIs.

    A usage plan sets a target for the throttling and quota limits on individual client API keys. For more information, see `Creating and Using API Usage Plans in Amazon API Gateway <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html>`_ in the *API Gateway Developer Guide* .

    In some cases clients can exceed the targets that you set. Don’t rely on usage plans to control costs. Consider using `AWS Budgets <https://docs.aws.amazon.com/cost-management/latest/userguide/budgets-managing-costs.html>`_ to monitor costs and `AWS WAF <https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html>`_ to manage API requests.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html
    :cloudformationResource: AWS::ApiGateway::UsagePlan
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_usage_plan = apigateway.CfnUsagePlan(self, "MyCfnUsagePlan",
            api_stages=[apigateway.CfnUsagePlan.ApiStageProperty(
                api_id="apiId",
                stage="stage",
                throttle={
                    "throttle_key": apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                        burst_limit=123,
                        rate_limit=123
                    )
                }
            )],
            description="description",
            quota=apigateway.CfnUsagePlan.QuotaSettingsProperty(
                limit=123,
                offset=123,
                period="period"
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            throttle=apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                burst_limit=123,
                rate_limit=123
            ),
            usage_plan_name="usagePlanName"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        api_stages: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUsagePlan.ApiStageProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        description: typing.Optional[builtins.str] = None,
        quota: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUsagePlan.QuotaSettingsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        throttle: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union["CfnUsagePlan.ThrottleSettingsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        usage_plan_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param api_stages: The associated API stages of a usage plan.
        :param description: The description of a usage plan.
        :param quota: The target maximum number of permitted requests per a given unit time interval.
        :param tags: The collection of tags. Each tag element is associated with a given resource.
        :param throttle: A map containing method level throttling information for API stage in a usage plan.
        :param usage_plan_name: The name of a usage plan.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8a8c11347e47e0f4cc9518ba40e4026dd908f54137f8dd41f789c4c65ba38e10)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUsagePlanProps(
            api_stages=api_stages,
            description=description,
            quota=quota,
            tags=tags,
            throttle=throttle,
            usage_plan_name=usage_plan_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__305ebc0d1bd2c496b732d04490b6e35befd6f95c0273c4a987a6d5335feaa2f4)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1ffa7e92afb712ed4e087a6d1bc8bf0872a8c6ab739038650cbc4420863fa2d2)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID for the usage plan.

        For example: ``abc123`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="apiStages")
    def api_stages(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUsagePlan.ApiStageProperty"]]]]:
        '''The associated API stages of a usage plan.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUsagePlan.ApiStageProperty"]]]], jsii.get(self, "apiStages"))

    @api_stages.setter
    def api_stages(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, "CfnUsagePlan.ApiStageProperty"]]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eeb3717cd8ad7650ffb0c5ab0567712b5232a5dc6e6703261d556db1205cf97b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "apiStages", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of a usage plan.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__146f9bb6c76b58140d8075c3f92748c21daaf43964f299dcc8701877d7e4aebc)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="quota")
    def quota(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUsagePlan.QuotaSettingsProperty"]]:
        '''The target maximum number of permitted requests per a given unit time interval.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUsagePlan.QuotaSettingsProperty"]], jsii.get(self, "quota"))

    @quota.setter
    def quota(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUsagePlan.QuotaSettingsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1a290093c5b354ddfe1c3c09c8300496bbaf7cc35398f091f46ac078b87e5e0c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "quota", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__19cbff4e5b2762a5c6d1fe84db84662e69843a892de1e65537b240e3cd06aeda)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="throttle")
    def throttle(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUsagePlan.ThrottleSettingsProperty"]]:
        '''A map containing method level throttling information for API stage in a usage plan.'''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUsagePlan.ThrottleSettingsProperty"]], jsii.get(self, "throttle"))

    @throttle.setter
    def throttle(
        self,
        value: typing.Optional[typing.Union[_IResolvable_da3f097b, "CfnUsagePlan.ThrottleSettingsProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3bd5e493b47368f8e04fdd1f81de0527fa9c9c1fd5b9390595a6be1196acab73)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "throttle", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="usagePlanName")
    def usage_plan_name(self) -> typing.Optional[builtins.str]:
        '''The name of a usage plan.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "usagePlanName"))

    @usage_plan_name.setter
    def usage_plan_name(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f58e0b0f2503d3467f040ade3a60720b88df89f4f7dadeed6b309b7ff0b271a9)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "usagePlanName", value) # pyright: ignore[reportArgumentType]

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnUsagePlan.ApiStageProperty",
        jsii_struct_bases=[],
        name_mapping={"api_id": "apiId", "stage": "stage", "throttle": "throttle"},
    )
    class ApiStageProperty:
        def __init__(
            self,
            *,
            api_id: typing.Optional[builtins.str] = None,
            stage: typing.Optional[builtins.str] = None,
            throttle: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, typing.Union["CfnUsagePlan.ThrottleSettingsProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''API stage name of the associated API stage in a usage plan.

            :param api_id: API Id of the associated API stage in a usage plan.
            :param stage: API stage name of the associated API stage in a usage plan.
            :param throttle: Map containing method level throttling information for API stage in a usage plan.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-apistage.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                api_stage_property = apigateway.CfnUsagePlan.ApiStageProperty(
                    api_id="apiId",
                    stage="stage",
                    throttle={
                        "throttle_key": apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                            burst_limit=123,
                            rate_limit=123
                        )
                    }
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d40839f573fa7292fcddc12e4d0dca807342d041b415a75becaca03f70134cb5)
                check_type(argname="argument api_id", value=api_id, expected_type=type_hints["api_id"])
                check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
                check_type(argname="argument throttle", value=throttle, expected_type=type_hints["throttle"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if api_id is not None:
                self._values["api_id"] = api_id
            if stage is not None:
                self._values["stage"] = stage
            if throttle is not None:
                self._values["throttle"] = throttle

        @builtins.property
        def api_id(self) -> typing.Optional[builtins.str]:
            '''API Id of the associated API stage in a usage plan.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-apistage.html#cfn-apigateway-usageplan-apistage-apiid
            '''
            result = self._values.get("api_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def stage(self) -> typing.Optional[builtins.str]:
            '''API stage name of the associated API stage in a usage plan.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-apistage.html#cfn-apigateway-usageplan-apistage-stage
            '''
            result = self._values.get("stage")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def throttle(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, "CfnUsagePlan.ThrottleSettingsProperty"]]]]:
            '''Map containing method level throttling information for API stage in a usage plan.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-apistage.html#cfn-apigateway-usageplan-apistage-throttle
            '''
            result = self._values.get("throttle")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Mapping[builtins.str, typing.Union[_IResolvable_da3f097b, "CfnUsagePlan.ThrottleSettingsProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ApiStageProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnUsagePlan.QuotaSettingsProperty",
        jsii_struct_bases=[],
        name_mapping={"limit": "limit", "offset": "offset", "period": "period"},
    )
    class QuotaSettingsProperty:
        def __init__(
            self,
            *,
            limit: typing.Optional[jsii.Number] = None,
            offset: typing.Optional[jsii.Number] = None,
            period: typing.Optional[builtins.str] = None,
        ) -> None:
            '''``QuotaSettings`` is a property of the `AWS::ApiGateway::UsagePlan <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html>`_ resource that specifies a target for the maximum number of requests users can make to your REST APIs.

            In some cases clients can exceed the targets that you set. Don’t rely on usage plans to control costs. Consider using `AWS Budgets <https://docs.aws.amazon.com/cost-management/latest/userguide/budgets-managing-costs.html>`_ to monitor costs and `AWS WAF <https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html>`_ to manage API requests.

            :param limit: The target maximum number of requests that can be made in a given time period.
            :param offset: The number of requests subtracted from the given limit in the initial time period.
            :param period: The time period in which the limit applies. Valid values are "DAY", "WEEK" or "MONTH".

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-quotasettings.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                quota_settings_property = apigateway.CfnUsagePlan.QuotaSettingsProperty(
                    limit=123,
                    offset=123,
                    period="period"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__97d1013711d6a9dfc69240a675668eccd93c059143115b99ad21fa3dfab1d863)
                check_type(argname="argument limit", value=limit, expected_type=type_hints["limit"])
                check_type(argname="argument offset", value=offset, expected_type=type_hints["offset"])
                check_type(argname="argument period", value=period, expected_type=type_hints["period"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if limit is not None:
                self._values["limit"] = limit
            if offset is not None:
                self._values["offset"] = offset
            if period is not None:
                self._values["period"] = period

        @builtins.property
        def limit(self) -> typing.Optional[jsii.Number]:
            '''The target maximum number of requests that can be made in a given time period.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-quotasettings.html#cfn-apigateway-usageplan-quotasettings-limit
            '''
            result = self._values.get("limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def offset(self) -> typing.Optional[jsii.Number]:
            '''The number of requests subtracted from the given limit in the initial time period.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-quotasettings.html#cfn-apigateway-usageplan-quotasettings-offset
            '''
            result = self._values.get("offset")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def period(self) -> typing.Optional[builtins.str]:
            '''The time period in which the limit applies.

            Valid values are "DAY", "WEEK" or "MONTH".

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-quotasettings.html#cfn-apigateway-usageplan-quotasettings-period
            '''
            result = self._values.get("period")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "QuotaSettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="aws-cdk-lib.aws_apigateway.CfnUsagePlan.ThrottleSettingsProperty",
        jsii_struct_bases=[],
        name_mapping={"burst_limit": "burstLimit", "rate_limit": "rateLimit"},
    )
    class ThrottleSettingsProperty:
        def __init__(
            self,
            *,
            burst_limit: typing.Optional[jsii.Number] = None,
            rate_limit: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''``ThrottleSettings`` is a property of the `AWS::ApiGateway::UsagePlan <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html>`_ resource that specifies the overall request rate (average requests per second) and burst capacity when users call your REST APIs.

            :param burst_limit: The API target request burst rate limit. This allows more requests through for a period of time than the target rate limit.
            :param rate_limit: The API target request rate limit.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-throttlesettings.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from aws_cdk import aws_apigateway as apigateway
                
                throttle_settings_property = apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                    burst_limit=123,
                    rate_limit=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c6e275a99a7dbc2f0e2b377eb94e405192964c2c4a979af42d874e23e374ed0b)
                check_type(argname="argument burst_limit", value=burst_limit, expected_type=type_hints["burst_limit"])
                check_type(argname="argument rate_limit", value=rate_limit, expected_type=type_hints["rate_limit"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if burst_limit is not None:
                self._values["burst_limit"] = burst_limit
            if rate_limit is not None:
                self._values["rate_limit"] = rate_limit

        @builtins.property
        def burst_limit(self) -> typing.Optional[jsii.Number]:
            '''The API target request burst rate limit.

            This allows more requests through for a period of time than the target rate limit.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-throttlesettings.html#cfn-apigateway-usageplan-throttlesettings-burstlimit
            '''
            result = self._values.get("burst_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def rate_limit(self) -> typing.Optional[jsii.Number]:
            '''The API target request rate limit.

            :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-usageplan-throttlesettings.html#cfn-apigateway-usageplan-throttlesettings-ratelimit
            '''
            result = self._values.get("rate_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ThrottleSettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_c2943556)
class CfnUsagePlanKey(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnUsagePlanKey",
):
    '''The ``AWS::ApiGateway::UsagePlanKey`` resource associates an API key with a usage plan.

    This association determines which users the usage plan is applied to.

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html
    :cloudformationResource: AWS::ApiGateway::UsagePlanKey
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_usage_plan_key = apigateway.CfnUsagePlanKey(self, "MyCfnUsagePlanKey",
            key_id="keyId",
            key_type="keyType",
            usage_plan_id="usagePlanId"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        key_id: builtins.str,
        key_type: builtins.str,
        usage_plan_id: builtins.str,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param key_id: The Id of the UsagePlanKey resource.
        :param key_type: The type of a UsagePlanKey resource for a plan customer.
        :param usage_plan_id: The Id of the UsagePlan resource representing the usage plan containing the UsagePlanKey resource representing a plan customer.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d040334e32d6ff777d0e5db04f227d71a84ac05f4cbfac9fa40ee5d9abf13c7e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnUsagePlanKeyProps(
            key_id=key_id, key_type=key_type, usage_plan_id=usage_plan_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__85396c204e4ebb6de3ec05a19f581a08ede9add349f50699e89aaee264b1aec4)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b9f3521654098f224805b4a056b092ea06aa8113865449a2f0352162f4deee10)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID for the usage plan key.

        For example: ``abc123`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="keyId")
    def key_id(self) -> builtins.str:
        '''The Id of the UsagePlanKey resource.'''
        return typing.cast(builtins.str, jsii.get(self, "keyId"))

    @key_id.setter
    def key_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d7340807c3f5327fe151d1093b1f45745221ef30b694defa5c64bd2da102d3cd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "keyId", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="keyType")
    def key_type(self) -> builtins.str:
        '''The type of a UsagePlanKey resource for a plan customer.'''
        return typing.cast(builtins.str, jsii.get(self, "keyType"))

    @key_type.setter
    def key_type(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f2cb078f0d05612a9a803dad439ae3683916ced402c0ba8c8ffd77b0f5942012)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "keyType", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="usagePlanId")
    def usage_plan_id(self) -> builtins.str:
        '''The Id of the UsagePlan resource representing the usage plan containing the UsagePlanKey resource representing a plan customer.'''
        return typing.cast(builtins.str, jsii.get(self, "usagePlanId"))

    @usage_plan_id.setter
    def usage_plan_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7cbbe93db800e3cbd5bf70d2a26188eca9cc95ab585564a20d35a927abf55572)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "usagePlanId", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnUsagePlanKeyProps",
    jsii_struct_bases=[],
    name_mapping={
        "key_id": "keyId",
        "key_type": "keyType",
        "usage_plan_id": "usagePlanId",
    },
)
class CfnUsagePlanKeyProps:
    def __init__(
        self,
        *,
        key_id: builtins.str,
        key_type: builtins.str,
        usage_plan_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnUsagePlanKey``.

        :param key_id: The Id of the UsagePlanKey resource.
        :param key_type: The type of a UsagePlanKey resource for a plan customer.
        :param usage_plan_id: The Id of the UsagePlan resource representing the usage plan containing the UsagePlanKey resource representing a plan customer.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_usage_plan_key_props = apigateway.CfnUsagePlanKeyProps(
                key_id="keyId",
                key_type="keyType",
                usage_plan_id="usagePlanId"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a83566309e98c3819e5e77ec3a51237257fe42030393661e8bbf7d1c6c2450af)
            check_type(argname="argument key_id", value=key_id, expected_type=type_hints["key_id"])
            check_type(argname="argument key_type", value=key_type, expected_type=type_hints["key_type"])
            check_type(argname="argument usage_plan_id", value=usage_plan_id, expected_type=type_hints["usage_plan_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "key_id": key_id,
            "key_type": key_type,
            "usage_plan_id": usage_plan_id,
        }

    @builtins.property
    def key_id(self) -> builtins.str:
        '''The Id of the UsagePlanKey resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html#cfn-apigateway-usageplankey-keyid
        '''
        result = self._values.get("key_id")
        assert result is not None, "Required property 'key_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def key_type(self) -> builtins.str:
        '''The type of a UsagePlanKey resource for a plan customer.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html#cfn-apigateway-usageplankey-keytype
        '''
        result = self._values.get("key_type")
        assert result is not None, "Required property 'key_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def usage_plan_id(self) -> builtins.str:
        '''The Id of the UsagePlan resource representing the usage plan containing the UsagePlanKey resource representing a plan customer.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplankey.html#cfn-apigateway-usageplankey-usageplanid
        '''
        result = self._values.get("usage_plan_id")
        assert result is not None, "Required property 'usage_plan_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUsagePlanKeyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnUsagePlanProps",
    jsii_struct_bases=[],
    name_mapping={
        "api_stages": "apiStages",
        "description": "description",
        "quota": "quota",
        "tags": "tags",
        "throttle": "throttle",
        "usage_plan_name": "usagePlanName",
    },
)
class CfnUsagePlanProps:
    def __init__(
        self,
        *,
        api_stages: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Sequence[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUsagePlan.ApiStageProperty, typing.Dict[builtins.str, typing.Any]]]]]] = None,
        description: typing.Optional[builtins.str] = None,
        quota: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUsagePlan.QuotaSettingsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
        throttle: typing.Optional[typing.Union[_IResolvable_da3f097b, typing.Union[CfnUsagePlan.ThrottleSettingsProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        usage_plan_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnUsagePlan``.

        :param api_stages: The associated API stages of a usage plan.
        :param description: The description of a usage plan.
        :param quota: The target maximum number of permitted requests per a given unit time interval.
        :param tags: The collection of tags. Each tag element is associated with a given resource.
        :param throttle: A map containing method level throttling information for API stage in a usage plan.
        :param usage_plan_name: The name of a usage plan.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_usage_plan_props = apigateway.CfnUsagePlanProps(
                api_stages=[apigateway.CfnUsagePlan.ApiStageProperty(
                    api_id="apiId",
                    stage="stage",
                    throttle={
                        "throttle_key": apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                            burst_limit=123,
                            rate_limit=123
                        )
                    }
                )],
                description="description",
                quota=apigateway.CfnUsagePlan.QuotaSettingsProperty(
                    limit=123,
                    offset=123,
                    period="period"
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                throttle=apigateway.CfnUsagePlan.ThrottleSettingsProperty(
                    burst_limit=123,
                    rate_limit=123
                ),
                usage_plan_name="usagePlanName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a2024605b90c9cfea35336633432eb81e8ff2ab654e8a52121584afb6de9420)
            check_type(argname="argument api_stages", value=api_stages, expected_type=type_hints["api_stages"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument quota", value=quota, expected_type=type_hints["quota"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument throttle", value=throttle, expected_type=type_hints["throttle"])
            check_type(argname="argument usage_plan_name", value=usage_plan_name, expected_type=type_hints["usage_plan_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if api_stages is not None:
            self._values["api_stages"] = api_stages
        if description is not None:
            self._values["description"] = description
        if quota is not None:
            self._values["quota"] = quota
        if tags is not None:
            self._values["tags"] = tags
        if throttle is not None:
            self._values["throttle"] = throttle
        if usage_plan_name is not None:
            self._values["usage_plan_name"] = usage_plan_name

    @builtins.property
    def api_stages(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnUsagePlan.ApiStageProperty]]]]:
        '''The associated API stages of a usage plan.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-apistages
        '''
        result = self._values.get("api_stages")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, typing.List[typing.Union[_IResolvable_da3f097b, CfnUsagePlan.ApiStageProperty]]]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of a usage plan.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def quota(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUsagePlan.QuotaSettingsProperty]]:
        '''The target maximum number of permitted requests per a given unit time interval.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-quota
        '''
        result = self._values.get("quota")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUsagePlan.QuotaSettingsProperty]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''The collection of tags.

        Each tag element is associated with a given resource.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    @builtins.property
    def throttle(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUsagePlan.ThrottleSettingsProperty]]:
        '''A map containing method level throttling information for API stage in a usage plan.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-throttle
        '''
        result = self._values.get("throttle")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_da3f097b, CfnUsagePlan.ThrottleSettingsProperty]], result)

    @builtins.property
    def usage_plan_name(self) -> typing.Optional[builtins.str]:
        '''The name of a usage plan.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-usageplan.html#cfn-apigateway-usageplan-usageplanname
        '''
        result = self._values.get("usage_plan_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnUsagePlanProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_c2943556, _ITaggable_36806126)
class CfnVpcLink(
    _CfnResource_9df397a6,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.CfnVpcLink",
):
    '''The ``AWS::ApiGateway::VpcLink`` resource creates an API Gateway VPC link for a REST API to access resources in an Amazon Virtual Private Cloud (VPC).

    For more information, see `vpclink:create <https://docs.aws.amazon.com/apigateway/latest/api/API_CreateVpcLink.html>`_ in the ``Amazon API Gateway REST API Reference`` .

    :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-vpclink.html
    :cloudformationResource: AWS::ApiGateway::VpcLink
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        cfn_vpc_link = apigateway.CfnVpcLink(self, "MyCfnVpcLink",
            name="name",
            target_arns=["targetArns"],
        
            # the properties below are optional
            description="description",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        name: builtins.str,
        target_arns: typing.Sequence[builtins.str],
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''
        :param scope: Scope in which this resource is defined.
        :param id: Construct identifier for this resource (unique in its scope).
        :param name: The name used to label and identify the VPC link.
        :param target_arns: The ARN of the network load balancer of the VPC targeted by the VPC link. The network load balancer must be owned by the same AWS account of the API owner.
        :param description: The description of the VPC link.
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the VPC link.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4a1f69db69c9ab3edee19de9b16585e4355791d79ee3b330d18242cc4d9fbfaf)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnVpcLinkProps(
            name=name, target_arns=target_arns, description=description, tags=tags
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_488e0dd5) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ab88de0a7914cd8aabbb92e0ebc49325c3e07a7d032f4f5d29eb7cba850c71b6)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e6aab52d457c2b0db5b2d71daaa7360d8c774c5871069264c9b76df1686e97cf)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrVpcLinkId")
    def attr_vpc_link_id(self) -> builtins.str:
        '''The ID for the VPC link.

        For example: ``abc123`` .

        :cloudformationAttribute: VpcLinkId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVpcLinkId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0a598cb3:
        '''Tag Manager which manages the tags for this resource.'''
        return typing.cast(_TagManager_0a598cb3, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The name used to label and identify the VPC link.'''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2cd3b0f2cc5f6ceabb69c9d0b3da4ccfb8bd07499cca83a529dd602f329f80e8)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="targetArns")
    def target_arns(self) -> typing.List[builtins.str]:
        '''The ARN of the network load balancer of the VPC targeted by the VPC link.'''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "targetArns"))

    @target_arns.setter
    def target_arns(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5b1651d2015e3f87ef3456d5acb73fee922ab3f90f0f2652d1276fe93341237a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "targetArns", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the VPC link.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0185ec4ceafba374d710d7b8cdda1b198dbfe7c36d40a77152f389184b14d91e)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "description", value) # pyright: ignore[reportArgumentType]

    @builtins.property
    @jsii.member(jsii_name="tagsRaw")
    def tags_raw(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''An array of arbitrary tags (key-value pairs) to associate with the VPC link.'''
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], jsii.get(self, "tagsRaw"))

    @tags_raw.setter
    def tags_raw(self, value: typing.Optional[typing.List[_CfnTag_f6864754]]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__652ea3fc4dcb1c64632ca0820a4de66a3d5f773676a1a643b6de49530a665c93)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "tagsRaw", value) # pyright: ignore[reportArgumentType]


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CfnVpcLinkProps",
    jsii_struct_bases=[],
    name_mapping={
        "name": "name",
        "target_arns": "targetArns",
        "description": "description",
        "tags": "tags",
    },
)
class CfnVpcLinkProps:
    def __init__(
        self,
        *,
        name: builtins.str,
        target_arns: typing.Sequence[builtins.str],
        description: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[typing.Union[_CfnTag_f6864754, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnVpcLink``.

        :param name: The name used to label and identify the VPC link.
        :param target_arns: The ARN of the network load balancer of the VPC targeted by the VPC link. The network load balancer must be owned by the same AWS account of the API owner.
        :param description: The description of the VPC link.
        :param tags: An array of arbitrary tags (key-value pairs) to associate with the VPC link.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-vpclink.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            cfn_vpc_link_props = apigateway.CfnVpcLinkProps(
                name="name",
                target_arns=["targetArns"],
            
                # the properties below are optional
                description="description",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__82929f6feb2c8f1cf5d1cf907f6bcadc8873089d76fce54891dd75f983d750ad)
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument target_arns", value=target_arns, expected_type=type_hints["target_arns"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "name": name,
            "target_arns": target_arns,
        }
        if description is not None:
            self._values["description"] = description
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def name(self) -> builtins.str:
        '''The name used to label and identify the VPC link.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-vpclink.html#cfn-apigateway-vpclink-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_arns(self) -> typing.List[builtins.str]:
        '''The ARN of the network load balancer of the VPC targeted by the VPC link.

        The network load balancer must be owned by the same AWS account of the API owner.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-vpclink.html#cfn-apigateway-vpclink-targetarns
        '''
        result = self._values.get("target_arns")
        assert result is not None, "Required property 'target_arns' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''The description of the VPC link.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-vpclink.html#cfn-apigateway-vpclink-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_f6864754]]:
        '''An array of arbitrary tags (key-value pairs) to associate with the VPC link.

        :see: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-vpclink.html#cfn-apigateway-vpclink-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_f6864754]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVpcLinkProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CognitoUserPoolsAuthorizerProps",
    jsii_struct_bases=[],
    name_mapping={
        "cognito_user_pools": "cognitoUserPools",
        "authorizer_name": "authorizerName",
        "identity_source": "identitySource",
        "results_cache_ttl": "resultsCacheTtl",
    },
)
class CognitoUserPoolsAuthorizerProps:
    def __init__(
        self,
        *,
        cognito_user_pools: typing.Sequence[_IUserPool_1f1029e2],
        authorizer_name: typing.Optional[builtins.str] = None,
        identity_source: typing.Optional[builtins.str] = None,
        results_cache_ttl: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''Properties for CognitoUserPoolsAuthorizer.

        :param cognito_user_pools: The user pools to associate with this authorizer.
        :param authorizer_name: An optional human friendly name for the authorizer. Note that, this is not the primary identifier of the authorizer. Default: - the unique construct ID
        :param identity_source: The request header mapping expression for the bearer token. This is typically passed as part of the header, in which case this should be ``method.request.header.Authorizer`` where ``Authorizer`` is the header containing the bearer token. Default: ``IdentitySource.header('Authorization')``
        :param results_cache_ttl: How long APIGateway should cache the results. Max 1 hour. Disable caching by setting this to 0. Default: Duration.minutes(5)

        :exampleMetadata: infused

        Example::

            # books: apigateway.Resource
            user_pool = cognito.UserPool(self, "UserPool")
            
            auth = apigateway.CognitoUserPoolsAuthorizer(self, "booksAuthorizer",
                cognito_user_pools=[user_pool]
            )
            books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
                authorizer=auth,
                authorization_type=apigateway.AuthorizationType.COGNITO
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0b2aebe816f57501ecd01d0d61da5fc7418a43f505b391b033cea4e32bd2b28c)
            check_type(argname="argument cognito_user_pools", value=cognito_user_pools, expected_type=type_hints["cognito_user_pools"])
            check_type(argname="argument authorizer_name", value=authorizer_name, expected_type=type_hints["authorizer_name"])
            check_type(argname="argument identity_source", value=identity_source, expected_type=type_hints["identity_source"])
            check_type(argname="argument results_cache_ttl", value=results_cache_ttl, expected_type=type_hints["results_cache_ttl"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cognito_user_pools": cognito_user_pools,
        }
        if authorizer_name is not None:
            self._values["authorizer_name"] = authorizer_name
        if identity_source is not None:
            self._values["identity_source"] = identity_source
        if results_cache_ttl is not None:
            self._values["results_cache_ttl"] = results_cache_ttl

    @builtins.property
    def cognito_user_pools(self) -> typing.List[_IUserPool_1f1029e2]:
        '''The user pools to associate with this authorizer.'''
        result = self._values.get("cognito_user_pools")
        assert result is not None, "Required property 'cognito_user_pools' is missing"
        return typing.cast(typing.List[_IUserPool_1f1029e2], result)

    @builtins.property
    def authorizer_name(self) -> typing.Optional[builtins.str]:
        '''An optional human friendly name for the authorizer.

        Note that, this is not the primary identifier of the authorizer.

        :default: - the unique construct ID
        '''
        result = self._values.get("authorizer_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def identity_source(self) -> typing.Optional[builtins.str]:
        '''The request header mapping expression for the bearer token.

        This is typically passed as part of the header, in which case
        this should be ``method.request.header.Authorizer`` where ``Authorizer`` is the header containing the bearer token.

        :default: ``IdentitySource.header('Authorization')``

        :see: https://docs.aws.amazon.com/apigateway/latest/api/API_CreateAuthorizer.html#apigw-CreateAuthorizer-request-identitySource
        '''
        result = self._values.get("identity_source")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def results_cache_ttl(self) -> typing.Optional[_Duration_4839e8c3]:
        '''How long APIGateway should cache the results.

        Max 1 hour.
        Disable caching by setting this to 0.

        :default: Duration.minutes(5)
        '''
        result = self._values.get("results_cache_ttl")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CognitoUserPoolsAuthorizerProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_apigateway.ConnectionType")
class ConnectionType(enum.Enum):
    '''
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_elasticloadbalancingv2 as elbv2
        
        
        vpc = ec2.Vpc(self, "VPC")
        nlb = elbv2.NetworkLoadBalancer(self, "NLB",
            vpc=vpc
        )
        link = apigateway.VpcLink(self, "link",
            targets=[nlb]
        )
        
        integration = apigateway.Integration(
            type=apigateway.IntegrationType.HTTP_PROXY,
            integration_http_method="ANY",
            options=apigateway.IntegrationOptions(
                connection_type=apigateway.ConnectionType.VPC_LINK,
                vpc_link=link
            )
        )
    '''

    INTERNET = "INTERNET"
    '''For connections through the public routable internet.'''
    VPC_LINK = "VPC_LINK"
    '''For private connections between API Gateway and a network load balancer in a VPC.'''


@jsii.enum(jsii_type="aws-cdk-lib.aws_apigateway.ContentHandling")
class ContentHandling(enum.Enum):
    '''
    :exampleMetadata: infused

    Example::

        # get_book_handler: lambda.Function
        # get_book_integration: apigateway.LambdaIntegration
        
        
        get_book_integration = apigateway.LambdaIntegration(get_book_handler,
            content_handling=apigateway.ContentHandling.CONVERT_TO_TEXT,  # convert to base64
            credentials_passthrough=True
        )
    '''

    CONVERT_TO_BINARY = "CONVERT_TO_BINARY"
    '''Converts a request payload from a base64-encoded string to a binary blob.'''
    CONVERT_TO_TEXT = "CONVERT_TO_TEXT"
    '''Converts a request payload from a binary blob to a base64-encoded string.'''


class Cors(metaclass=jsii.JSIIMeta, jsii_type="aws-cdk-lib.aws_apigateway.Cors"):
    '''
    :exampleMetadata: infused

    Example::

        apigateway.RestApi(self, "api",
            default_cors_preflight_options=apigateway.CorsOptions(
                allow_origins=apigateway.Cors.ALL_ORIGINS,
                allow_methods=apigateway.Cors.ALL_METHODS
            )
        )
    '''

    @jsii.python.classproperty
    @jsii.member(jsii_name="ALL_METHODS")
    def ALL_METHODS(cls) -> typing.List[builtins.str]:
        '''All HTTP methods.'''
        return typing.cast(typing.List[builtins.str], jsii.sget(cls, "ALL_METHODS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ALL_ORIGINS")
    def ALL_ORIGINS(cls) -> typing.List[builtins.str]:
        '''All origins.'''
        return typing.cast(typing.List[builtins.str], jsii.sget(cls, "ALL_ORIGINS"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="DEFAULT_HEADERS")
    def DEFAULT_HEADERS(cls) -> typing.List[builtins.str]:
        '''The set of default headers allowed for CORS and useful for API Gateway.'''
        return typing.cast(typing.List[builtins.str], jsii.sget(cls, "DEFAULT_HEADERS"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.CorsOptions",
    jsii_struct_bases=[],
    name_mapping={
        "allow_origins": "allowOrigins",
        "allow_credentials": "allowCredentials",
        "allow_headers": "allowHeaders",
        "allow_methods": "allowMethods",
        "disable_cache": "disableCache",
        "expose_headers": "exposeHeaders",
        "max_age": "maxAge",
        "status_code": "statusCode",
    },
)
class CorsOptions:
    def __init__(
        self,
        *,
        allow_origins: typing.Sequence[builtins.str],
        allow_credentials: typing.Optional[builtins.bool] = None,
        allow_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        allow_methods: typing.Optional[typing.Sequence[builtins.str]] = None,
        disable_cache: typing.Optional[builtins.bool] = None,
        expose_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        max_age: typing.Optional[_Duration_4839e8c3] = None,
        status_code: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param allow_origins: Specifies the list of origins that are allowed to make requests to this resource. If you wish to allow all origins, specify ``Cors.ALL_ORIGINS`` or ``[ * ]``. Responses will include the ``Access-Control-Allow-Origin`` response header. If specific origins are specified (not ``Cors.ALL_ORIGINS``), the ``Vary: Origin`` response header will also be included.
        :param allow_credentials: The Access-Control-Allow-Credentials response header tells browsers whether to expose the response to frontend JavaScript code when the request's credentials mode (Request.credentials) is "include". When a request's credentials mode (Request.credentials) is "include", browsers will only expose the response to frontend JavaScript code if the Access-Control-Allow-Credentials value is true. Credentials are cookies, authorization headers or TLS client certificates. Default: false
        :param allow_headers: The Access-Control-Allow-Headers response header is used in response to a preflight request which includes the Access-Control-Request-Headers to indicate which HTTP headers can be used during the actual request. Default: Cors.DEFAULT_HEADERS
        :param allow_methods: The Access-Control-Allow-Methods response header specifies the method or methods allowed when accessing the resource in response to a preflight request. If ``ANY`` is specified, it will be expanded to ``Cors.ALL_METHODS``. Default: Cors.ALL_METHODS
        :param disable_cache: Sets Access-Control-Max-Age to -1, which means that caching is disabled. This option cannot be used with ``maxAge``. Default: - cache is enabled
        :param expose_headers: The Access-Control-Expose-Headers response header indicates which headers can be exposed as part of the response by listing their names. If you want clients to be able to access other headers, you have to list them using the Access-Control-Expose-Headers header. Default: - only the 6 CORS-safelisted response headers are exposed: Cache-Control, Content-Language, Content-Type, Expires, Last-Modified, Pragma
        :param max_age: The Access-Control-Max-Age response header indicates how long the results of a preflight request (that is the information contained in the Access-Control-Allow-Methods and Access-Control-Allow-Headers headers) can be cached. To disable caching altogether use ``disableCache: true``. Default: - browser-specific (see reference)
        :param status_code: Specifies the response status code returned from the OPTIONS method. Default: 204

        :exampleMetadata: infused

        Example::

            apigateway.RestApi(self, "api",
                default_cors_preflight_options=apigateway.CorsOptions(
                    allow_origins=apigateway.Cors.ALL_ORIGINS,
                    allow_methods=apigateway.Cors.ALL_METHODS
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0c385502aa61f3eb6453d24cefc0bbdfae67c5b3fa2fd218bd91f185a38442c9)
            check_type(argname="argument allow_origins", value=allow_origins, expected_type=type_hints["allow_origins"])
            check_type(argname="argument allow_credentials", value=allow_credentials, expected_type=type_hints["allow_credentials"])
            check_type(argname="argument allow_headers", value=allow_headers, expected_type=type_hints["allow_headers"])
            check_type(argname="argument allow_methods", value=allow_methods, expected_type=type_hints["allow_methods"])
            check_type(argname="argument disable_cache", value=disable_cache, expected_type=type_hints["disable_cache"])
            check_type(argname="argument expose_headers", value=expose_headers, expected_type=type_hints["expose_headers"])
            check_type(argname="argument max_age", value=max_age, expected_type=type_hints["max_age"])
            check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "allow_origins": allow_origins,
        }
        if allow_credentials is not None:
            self._values["allow_credentials"] = allow_credentials
        if allow_headers is not None:
            self._values["allow_headers"] = allow_headers
        if allow_methods is not None:
            self._values["allow_methods"] = allow_methods
        if disable_cache is not None:
            self._values["disable_cache"] = disable_cache
        if expose_headers is not None:
            self._values["expose_headers"] = expose_headers
        if max_age is not None:
            self._values["max_age"] = max_age
        if status_code is not None:
            self._values["status_code"] = status_code

    @builtins.property
    def allow_origins(self) -> typing.List[builtins.str]:
        '''Specifies the list of origins that are allowed to make requests to this resource.

        If you wish to allow all origins, specify ``Cors.ALL_ORIGINS`` or
        ``[ * ]``.

        Responses will include the ``Access-Control-Allow-Origin`` response header.
        If specific origins are specified (not ``Cors.ALL_ORIGINS``), the ``Vary: Origin``
        response header will also be included.

        :see: https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin
        '''
        result = self._values.get("allow_origins")
        assert result is not None, "Required property 'allow_origins' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def allow_credentials(self) -> typing.Optional[builtins.bool]:
        '''The Access-Control-Allow-Credentials response header tells browsers whether to expose the response to frontend JavaScript code when the request's credentials mode (Request.credentials) is "include".

        When a request's credentials mode (Request.credentials) is "include",
        browsers will only expose the response to frontend JavaScript code if the
        Access-Control-Allow-Credentials value is true.

        Credentials are cookies, authorization headers or TLS client certificates.

        :default: false

        :see: https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Credentials
        '''
        result = self._values.get("allow_credentials")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def allow_headers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The Access-Control-Allow-Headers response header is used in response to a preflight request which includes the Access-Control-Request-Headers to indicate which HTTP headers can be used during the actual request.

        :default: Cors.DEFAULT_HEADERS

        :see: https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Headers
        '''
        result = self._values.get("allow_headers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def allow_methods(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The Access-Control-Allow-Methods response header specifies the method or methods allowed when accessing the resource in response to a preflight request.

        If ``ANY`` is specified, it will be expanded to ``Cors.ALL_METHODS``.

        :default: Cors.ALL_METHODS

        :see: https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Methods
        '''
        result = self._values.get("allow_methods")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def disable_cache(self) -> typing.Optional[builtins.bool]:
        '''Sets Access-Control-Max-Age to -1, which means that caching is disabled.

        This option cannot be used with ``maxAge``.

        :default: - cache is enabled
        '''
        result = self._values.get("disable_cache")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def expose_headers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The Access-Control-Expose-Headers response header indicates which headers can be exposed as part of the response by listing their names.

        If you want clients to be able to access other headers, you have to list
        them using the Access-Control-Expose-Headers header.

        :default:

        - only the 6 CORS-safelisted response headers are exposed:
        Cache-Control, Content-Language, Content-Type, Expires, Last-Modified,
        Pragma

        :see: https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Expose-Headers
        '''
        result = self._values.get("expose_headers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def max_age(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The Access-Control-Max-Age response header indicates how long the results of a preflight request (that is the information contained in the Access-Control-Allow-Methods and Access-Control-Allow-Headers headers) can be cached.

        To disable caching altogether use ``disableCache: true``.

        :default: - browser-specific (see reference)

        :see: https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Max-Age
        '''
        result = self._values.get("max_age")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def status_code(self) -> typing.Optional[jsii.Number]:
        '''Specifies the response status code returned from the OPTIONS method.

        :default: 204
        '''
        result = self._values.get("status_code")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CorsOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Deployment(
    _Resource_45bc6135,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.Deployment",
):
    '''A Deployment of a REST API.

    An immutable representation of a RestApi resource that can be called by users
    using Stages. A deployment must be associated with a Stage for it to be
    callable over the Internet.

    Normally, you don't need to define deployments manually. The RestApi
    construct manages a Deployment resource that represents the latest model. It
    can be accessed through ``restApi.latestDeployment`` (unless ``deploy: false`` is
    set when defining the ``RestApi``).

    If you manually define this resource, you will need to know that since
    deployments are immutable, as long as the resource's logical ID doesn't
    change, the deployment will represent the snapshot in time in which the
    resource was created. This means that if you modify the RestApi model (i.e.
    add methods or resources), these changes will not be reflected unless a new
    deployment resource is created.

    To achieve this behavior, the method ``addToLogicalId(data)`` can be used to
    augment the logical ID generated for the deployment resource such that it
    will include arbitrary data. This is done automatically for the
    ``restApi.latestDeployment`` deployment.

    Furthermore, since a deployment does not reference any of the REST API
    resources and methods, CloudFormation will likely provision it before these
    resources are created, which means that it will represent a "half-baked"
    model. Use the ``node.addDependency(dep)`` method to circumvent that. This is done
    automatically for the ``restApi.latestDeployment`` deployment.

    :exampleMetadata: infused

    Example::

        # production stage
        prod_log_group = logs.LogGroup(self, "PrdLogs")
        api = apigateway.RestApi(self, "books",
            deploy_options=apigateway.StageOptions(
                access_log_destination=apigateway.LogGroupLogDestination(prod_log_group),
                access_log_format=apigateway.AccessLogFormat.json_with_standard_fields()
            )
        )
        deployment = apigateway.Deployment(self, "Deployment", api=api)
        
        # development stage
        dev_log_group = logs.LogGroup(self, "DevLogs")
        apigateway.Stage(self, "dev",
            deployment=deployment,
            access_log_destination=apigateway.LogGroupLogDestination(dev_log_group),
            access_log_format=apigateway.AccessLogFormat.json_with_standard_fields(
                caller=False,
                http_method=True,
                ip=True,
                protocol=True,
                request_time=True,
                resource_path=True,
                response_length=True,
                status=True,
                user=True
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        api: "IRestApi",
        description: typing.Optional[builtins.str] = None,
        retain_deployments: typing.Optional[builtins.bool] = None,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param api: The Rest API to deploy.
        :param description: A description of the purpose of the API Gateway deployment. Default: - No description.
        :param retain_deployments: When an API Gateway model is updated, a new deployment will automatically be created. If this is true, the old API Gateway Deployment resource will not be deleted. This will allow manually reverting back to a previous deployment in case for example Default: false
        :param stage_name: The name of the stage the API Gateway deployment deploys to. Default: - No stage name. If the ``stageName`` property is set but a stage with the corresponding name does not exist, a new stage resource will be created with the provided stage name.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__60052a26fedba9d01bde406c7bb3c749dd3ea02f314d0f61d824f756c590721b)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = DeploymentProps(
            api=api,
            description=description,
            retain_deployments=retain_deployments,
            stage_name=stage_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addToLogicalId")
    def add_to_logical_id(self, data: typing.Any) -> None:
        '''Adds a component to the hash that determines this Deployment resource's logical ID.

        This should be called by constructs of the API Gateway model that want to
        invalidate the deployment when their settings change. The component will
        be resolved during synthesis so tokens are welcome.

        :param data: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__edf50b07113c98cd1bfd10a213da4c9142a184922cac7c0e47b3ed582807f425)
            check_type(argname="argument data", value=data, expected_type=type_hints["data"])
        return typing.cast(None, jsii.invoke(self, "addToLogicalId", [data]))

    @builtins.property
    @jsii.member(jsii_name="api")
    def api(self) -> "IRestApi":
        return typing.cast("IRestApi", jsii.get(self, "api"))

    @builtins.property
    @jsii.member(jsii_name="deploymentId")
    def deployment_id(self) -> builtins.str:
        '''
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "deploymentId"))

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''The stage of the API gateway deployment.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.DeploymentProps",
    jsii_struct_bases=[],
    name_mapping={
        "api": "api",
        "description": "description",
        "retain_deployments": "retainDeployments",
        "stage_name": "stageName",
    },
)
class DeploymentProps:
    def __init__(
        self,
        *,
        api: "IRestApi",
        description: typing.Optional[builtins.str] = None,
        retain_deployments: typing.Optional[builtins.bool] = None,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param api: The Rest API to deploy.
        :param description: A description of the purpose of the API Gateway deployment. Default: - No description.
        :param retain_deployments: When an API Gateway model is updated, a new deployment will automatically be created. If this is true, the old API Gateway Deployment resource will not be deleted. This will allow manually reverting back to a previous deployment in case for example Default: false
        :param stage_name: The name of the stage the API Gateway deployment deploys to. Default: - No stage name. If the ``stageName`` property is set but a stage with the corresponding name does not exist, a new stage resource will be created with the provided stage name.

        :exampleMetadata: infused

        Example::

            # production stage
            prod_log_group = logs.LogGroup(self, "PrdLogs")
            api = apigateway.RestApi(self, "books",
                deploy_options=apigateway.StageOptions(
                    access_log_destination=apigateway.LogGroupLogDestination(prod_log_group),
                    access_log_format=apigateway.AccessLogFormat.json_with_standard_fields()
                )
            )
            deployment = apigateway.Deployment(self, "Deployment", api=api)
            
            # development stage
            dev_log_group = logs.LogGroup(self, "DevLogs")
            apigateway.Stage(self, "dev",
                deployment=deployment,
                access_log_destination=apigateway.LogGroupLogDestination(dev_log_group),
                access_log_format=apigateway.AccessLogFormat.json_with_standard_fields(
                    caller=False,
                    http_method=True,
                    ip=True,
                    protocol=True,
                    request_time=True,
                    resource_path=True,
                    response_length=True,
                    status=True,
                    user=True
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__689289fb9a086819dc2bb6611b99cf777d39e89a4f67f62a4fb0ee0bacfb0110)
            check_type(argname="argument api", value=api, expected_type=type_hints["api"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument retain_deployments", value=retain_deployments, expected_type=type_hints["retain_deployments"])
            check_type(argname="argument stage_name", value=stage_name, expected_type=type_hints["stage_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "api": api,
        }
        if description is not None:
            self._values["description"] = description
        if retain_deployments is not None:
            self._values["retain_deployments"] = retain_deployments
        if stage_name is not None:
            self._values["stage_name"] = stage_name

    @builtins.property
    def api(self) -> "IRestApi":
        '''The Rest API to deploy.'''
        result = self._values.get("api")
        assert result is not None, "Required property 'api' is missing"
        return typing.cast("IRestApi", result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the purpose of the API Gateway deployment.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def retain_deployments(self) -> typing.Optional[builtins.bool]:
        '''When an API Gateway model is updated, a new deployment will automatically be created.

        If this is true, the old API Gateway Deployment resource will not be deleted.
        This will allow manually reverting back to a previous deployment in case for example

        :default: false
        '''
        result = self._values.get("retain_deployments")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''The name of the stage the API Gateway deployment deploys to.

        :default:

        - No stage name. If the ``stageName`` property is set but a stage with the
        corresponding name does not exist, a new stage resource will be created with the
        provided stage name.
        '''
        result = self._values.get("stage_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DeploymentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.DomainNameAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "domain_name": "domainName",
        "domain_name_alias_hosted_zone_id": "domainNameAliasHostedZoneId",
        "domain_name_alias_target": "domainNameAliasTarget",
    },
)
class DomainNameAttributes:
    def __init__(
        self,
        *,
        domain_name: builtins.str,
        domain_name_alias_hosted_zone_id: builtins.str,
        domain_name_alias_target: builtins.str,
    ) -> None:
        '''
        :param domain_name: The domain name (e.g. ``example.com``).
        :param domain_name_alias_hosted_zone_id: The Route53 hosted zone ID to use in order to connect a record set to this domain through an alias.
        :param domain_name_alias_target: The Route53 alias target to use in order to connect a record set to this domain through an alias.

        :exampleMetadata: infused

        Example::

            # api: apigateway.RestApi
            
            
            domain_name = apigateway.DomainName.from_domain_name_attributes(self, "DomainName",
                domain_name="domainName",
                domain_name_alias_hosted_zone_id="domainNameAliasHostedZoneId",
                domain_name_alias_target="domainNameAliasTarget"
            )
            
            apigateway.BasePathMapping(self, "BasePathMapping",
                domain_name=domain_name,
                rest_api=api
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ecfb12248c0bb90b9eb6413f1a897777b3fad240fd301f464e8379fdb25876f7)
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument domain_name_alias_hosted_zone_id", value=domain_name_alias_hosted_zone_id, expected_type=type_hints["domain_name_alias_hosted_zone_id"])
            check_type(argname="argument domain_name_alias_target", value=domain_name_alias_target, expected_type=type_hints["domain_name_alias_target"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "domain_name": domain_name,
            "domain_name_alias_hosted_zone_id": domain_name_alias_hosted_zone_id,
            "domain_name_alias_target": domain_name_alias_target,
        }

    @builtins.property
    def domain_name(self) -> builtins.str:
        '''The domain name (e.g. ``example.com``).'''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def domain_name_alias_hosted_zone_id(self) -> builtins.str:
        '''The Route53 hosted zone ID to use in order to connect a record set to this domain through an alias.'''
        result = self._values.get("domain_name_alias_hosted_zone_id")
        assert result is not None, "Required property 'domain_name_alias_hosted_zone_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def domain_name_alias_target(self) -> builtins.str:
        '''The Route53 alias target to use in order to connect a record set to this domain through an alias.'''
        result = self._values.get("domain_name_alias_target")
        assert result is not None, "Required property 'domain_name_alias_target' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DomainNameAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.DomainNameOptions",
    jsii_struct_bases=[],
    name_mapping={
        "certificate": "certificate",
        "domain_name": "domainName",
        "base_path": "basePath",
        "endpoint_type": "endpointType",
        "mtls": "mtls",
        "security_policy": "securityPolicy",
    },
)
class DomainNameOptions:
    def __init__(
        self,
        *,
        certificate: _ICertificate_c194c70b,
        domain_name: builtins.str,
        base_path: typing.Optional[builtins.str] = None,
        endpoint_type: typing.Optional["EndpointType"] = None,
        mtls: typing.Optional[typing.Union["MTLSConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        security_policy: typing.Optional["SecurityPolicy"] = None,
    ) -> None:
        '''
        :param certificate: The reference to an AWS-managed certificate for use by the edge-optimized endpoint for the domain name. For "EDGE" domain names, the certificate needs to be in the US East (N. Virginia) region.
        :param domain_name: The custom domain name for your API. Uppercase letters are not supported.
        :param base_path: The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string. Default: - map requests from the domain root (e.g. ``example.com``).
        :param endpoint_type: The type of endpoint for this DomainName. Default: REGIONAL
        :param mtls: The mutual TLS authentication configuration for a custom domain name. Default: - mTLS is not configured.
        :param security_policy: The Transport Layer Security (TLS) version + cipher suite for this domain name. Default: SecurityPolicy.TLS_1_2

        :exampleMetadata: infused

        Example::

            # acm_certificate_for_example_com: Any
            
            
            api = apigateway.RestApi(self, "MyDomain",
                domain_name=apigateway.DomainNameOptions(
                    domain_name="example.com",
                    certificate=acm_certificate_for_example_com
                )
            )
        '''
        if isinstance(mtls, dict):
            mtls = MTLSConfig(**mtls)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0bdcaf279d4b106d16c7f0aa0d23a9d2783cc9dcc73ee5530eac661399281730)
            check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument base_path", value=base_path, expected_type=type_hints["base_path"])
            check_type(argname="argument endpoint_type", value=endpoint_type, expected_type=type_hints["endpoint_type"])
            check_type(argname="argument mtls", value=mtls, expected_type=type_hints["mtls"])
            check_type(argname="argument security_policy", value=security_policy, expected_type=type_hints["security_policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "certificate": certificate,
            "domain_name": domain_name,
        }
        if base_path is not None:
            self._values["base_path"] = base_path
        if endpoint_type is not None:
            self._values["endpoint_type"] = endpoint_type
        if mtls is not None:
            self._values["mtls"] = mtls
        if security_policy is not None:
            self._values["security_policy"] = security_policy

    @builtins.property
    def certificate(self) -> _ICertificate_c194c70b:
        '''The reference to an AWS-managed certificate for use by the edge-optimized endpoint for the domain name.

        For "EDGE" domain names, the certificate
        needs to be in the US East (N. Virginia) region.
        '''
        result = self._values.get("certificate")
        assert result is not None, "Required property 'certificate' is missing"
        return typing.cast(_ICertificate_c194c70b, result)

    @builtins.property
    def domain_name(self) -> builtins.str:
        '''The custom domain name for your API.

        Uppercase letters are not supported.
        '''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def base_path(self) -> typing.Optional[builtins.str]:
        '''The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string.

        :default: - map requests from the domain root (e.g. ``example.com``).
        '''
        result = self._values.get("base_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def endpoint_type(self) -> typing.Optional["EndpointType"]:
        '''The type of endpoint for this DomainName.

        :default: REGIONAL
        '''
        result = self._values.get("endpoint_type")
        return typing.cast(typing.Optional["EndpointType"], result)

    @builtins.property
    def mtls(self) -> typing.Optional["MTLSConfig"]:
        '''The mutual TLS authentication configuration for a custom domain name.

        :default: - mTLS is not configured.
        '''
        result = self._values.get("mtls")
        return typing.cast(typing.Optional["MTLSConfig"], result)

    @builtins.property
    def security_policy(self) -> typing.Optional["SecurityPolicy"]:
        '''The Transport Layer Security (TLS) version + cipher suite for this domain name.

        :default: SecurityPolicy.TLS_1_2

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html
        '''
        result = self._values.get("security_policy")
        return typing.cast(typing.Optional["SecurityPolicy"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DomainNameOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.DomainNameProps",
    jsii_struct_bases=[DomainNameOptions],
    name_mapping={
        "certificate": "certificate",
        "domain_name": "domainName",
        "base_path": "basePath",
        "endpoint_type": "endpointType",
        "mtls": "mtls",
        "security_policy": "securityPolicy",
        "mapping": "mapping",
    },
)
class DomainNameProps(DomainNameOptions):
    def __init__(
        self,
        *,
        certificate: _ICertificate_c194c70b,
        domain_name: builtins.str,
        base_path: typing.Optional[builtins.str] = None,
        endpoint_type: typing.Optional["EndpointType"] = None,
        mtls: typing.Optional[typing.Union["MTLSConfig", typing.Dict[builtins.str, typing.Any]]] = None,
        security_policy: typing.Optional["SecurityPolicy"] = None,
        mapping: typing.Optional["IRestApi"] = None,
    ) -> None:
        '''
        :param certificate: The reference to an AWS-managed certificate for use by the edge-optimized endpoint for the domain name. For "EDGE" domain names, the certificate needs to be in the US East (N. Virginia) region.
        :param domain_name: The custom domain name for your API. Uppercase letters are not supported.
        :param base_path: The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string. Default: - map requests from the domain root (e.g. ``example.com``).
        :param endpoint_type: The type of endpoint for this DomainName. Default: REGIONAL
        :param mtls: The mutual TLS authentication configuration for a custom domain name. Default: - mTLS is not configured.
        :param security_policy: The Transport Layer Security (TLS) version + cipher suite for this domain name. Default: SecurityPolicy.TLS_1_2
        :param mapping: If specified, all requests to this domain will be mapped to the production deployment of this API. If you wish to map this domain to multiple APIs with different base paths, use ``addBasePathMapping`` or ``addApiMapping``. Default: - you will have to call ``addBasePathMapping`` to map this domain to API endpoints.

        :exampleMetadata: infused

        Example::

            # acm_certificate_for_example_com: Any
            # rest_api: apigateway.RestApi
            
            
            apigateway.DomainName(self, "custom-domain",
                domain_name="example.com",
                certificate=acm_certificate_for_example_com,
                mapping=rest_api,
                base_path="orders/v1/api"
            )
        '''
        if isinstance(mtls, dict):
            mtls = MTLSConfig(**mtls)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fd56e3d47a950d5babcb79f21442620ade11693af8810f3b0b64fa0292817253)
            check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument base_path", value=base_path, expected_type=type_hints["base_path"])
            check_type(argname="argument endpoint_type", value=endpoint_type, expected_type=type_hints["endpoint_type"])
            check_type(argname="argument mtls", value=mtls, expected_type=type_hints["mtls"])
            check_type(argname="argument security_policy", value=security_policy, expected_type=type_hints["security_policy"])
            check_type(argname="argument mapping", value=mapping, expected_type=type_hints["mapping"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "certificate": certificate,
            "domain_name": domain_name,
        }
        if base_path is not None:
            self._values["base_path"] = base_path
        if endpoint_type is not None:
            self._values["endpoint_type"] = endpoint_type
        if mtls is not None:
            self._values["mtls"] = mtls
        if security_policy is not None:
            self._values["security_policy"] = security_policy
        if mapping is not None:
            self._values["mapping"] = mapping

    @builtins.property
    def certificate(self) -> _ICertificate_c194c70b:
        '''The reference to an AWS-managed certificate for use by the edge-optimized endpoint for the domain name.

        For "EDGE" domain names, the certificate
        needs to be in the US East (N. Virginia) region.
        '''
        result = self._values.get("certificate")
        assert result is not None, "Required property 'certificate' is missing"
        return typing.cast(_ICertificate_c194c70b, result)

    @builtins.property
    def domain_name(self) -> builtins.str:
        '''The custom domain name for your API.

        Uppercase letters are not supported.
        '''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def base_path(self) -> typing.Optional[builtins.str]:
        '''The base path name that callers of the API must provide in the URL after the domain name (e.g. ``example.com/base-path``). If you specify this property, it can't be an empty string.

        :default: - map requests from the domain root (e.g. ``example.com``).
        '''
        result = self._values.get("base_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def endpoint_type(self) -> typing.Optional["EndpointType"]:
        '''The type of endpoint for this DomainName.

        :default: REGIONAL
        '''
        result = self._values.get("endpoint_type")
        return typing.cast(typing.Optional["EndpointType"], result)

    @builtins.property
    def mtls(self) -> typing.Optional["MTLSConfig"]:
        '''The mutual TLS authentication configuration for a custom domain name.

        :default: - mTLS is not configured.
        '''
        result = self._values.get("mtls")
        return typing.cast(typing.Optional["MTLSConfig"], result)

    @builtins.property
    def security_policy(self) -> typing.Optional["SecurityPolicy"]:
        '''The Transport Layer Security (TLS) version + cipher suite for this domain name.

        :default: SecurityPolicy.TLS_1_2

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-domainname.html
        '''
        result = self._values.get("security_policy")
        return typing.cast(typing.Optional["SecurityPolicy"], result)

    @builtins.property
    def mapping(self) -> typing.Optional["IRestApi"]:
        '''If specified, all requests to this domain will be mapped to the production deployment of this API.

        If you wish to map this domain to multiple APIs
        with different base paths, use ``addBasePathMapping`` or ``addApiMapping``.

        :default:

        - you will have to call ``addBasePathMapping`` to map this domain to
        API endpoints.
        '''
        result = self._values.get("mapping")
        return typing.cast(typing.Optional["IRestApi"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DomainNameProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.EndpointConfiguration",
    jsii_struct_bases=[],
    name_mapping={"types": "types", "vpc_endpoints": "vpcEndpoints"},
)
class EndpointConfiguration:
    def __init__(
        self,
        *,
        types: typing.Sequence["EndpointType"],
        vpc_endpoints: typing.Optional[typing.Sequence[_IVpcEndpoint_d8ea9bc3]] = None,
    ) -> None:
        '''The endpoint configuration of a REST API, including VPCs and endpoint types.

        EndpointConfiguration is a property of the AWS::ApiGateway::RestApi resource.

        :param types: A list of endpoint types of an API or its custom domain name. Default: EndpointType.EDGE
        :param vpc_endpoints: A list of VPC Endpoints against which to create Route53 ALIASes. Default: - no ALIASes are created for the endpoint.

        :exampleMetadata: infused

        Example::

            # some_endpoint: ec2.IVpcEndpoint
            
            
            api = apigateway.RestApi(self, "api",
                endpoint_configuration=apigateway.EndpointConfiguration(
                    types=[apigateway.EndpointType.PRIVATE],
                    vpc_endpoints=[some_endpoint]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2f3c0216cf00594a10062c97bb84b39e25b55231c15de867663dd59bbbede54a)
            check_type(argname="argument types", value=types, expected_type=type_hints["types"])
            check_type(argname="argument vpc_endpoints", value=vpc_endpoints, expected_type=type_hints["vpc_endpoints"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "types": types,
        }
        if vpc_endpoints is not None:
            self._values["vpc_endpoints"] = vpc_endpoints

    @builtins.property
    def types(self) -> typing.List["EndpointType"]:
        '''A list of endpoint types of an API or its custom domain name.

        :default: EndpointType.EDGE
        '''
        result = self._values.get("types")
        assert result is not None, "Required property 'types' is missing"
        return typing.cast(typing.List["EndpointType"], result)

    @builtins.property
    def vpc_endpoints(self) -> typing.Optional[typing.List[_IVpcEndpoint_d8ea9bc3]]:
        '''A list of VPC Endpoints against which to create Route53 ALIASes.

        :default: - no ALIASes are created for the endpoint.
        '''
        result = self._values.get("vpc_endpoints")
        return typing.cast(typing.Optional[typing.List[_IVpcEndpoint_d8ea9bc3]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EndpointConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_apigateway.EndpointType")
class EndpointType(enum.Enum):
    '''
    :exampleMetadata: infused

    Example::

        # some_endpoint: ec2.IVpcEndpoint
        
        
        api = apigateway.RestApi(self, "api",
            endpoint_configuration=apigateway.EndpointConfiguration(
                types=[apigateway.EndpointType.PRIVATE],
                vpc_endpoints=[some_endpoint]
            )
        )
    '''

    EDGE = "EDGE"
    '''For an edge-optimized API and its custom domain name.'''
    REGIONAL = "REGIONAL"
    '''For a regional API and its custom domain name.'''
    PRIVATE = "PRIVATE"
    '''For a private API and its custom domain name.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.GatewayResponseOptions",
    jsii_struct_bases=[],
    name_mapping={
        "type": "type",
        "response_headers": "responseHeaders",
        "status_code": "statusCode",
        "templates": "templates",
    },
)
class GatewayResponseOptions:
    def __init__(
        self,
        *,
        type: "ResponseType",
        response_headers: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        status_code: typing.Optional[builtins.str] = None,
        templates: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''Options to add gateway response.

        :param type: Response type to associate with gateway response.
        :param response_headers: Custom headers parameters for response. Default: - no headers
        :param status_code: Http status code for response. Default: - standard http status code for the response type.
        :param templates: Custom templates to get mapped as response. Default: - Response from api will be returned without applying any transformation.

        :exampleMetadata: infused

        Example::

            api = apigateway.RestApi(self, "books-api")
            api.add_gateway_response("test-response",
                type=apigateway.ResponseType.ACCESS_DENIED,
                status_code="500",
                response_headers={
                    # Note that values must be enclosed within a pair of single quotes
                    "Access-Control-Allow-Origin": "'test.com'",
                    "test-key": "'test-value'"
                },
                templates={
                    "application/json": "{ \"message\": $context.error.messageString, \"statusCode\": \"488\", \"type\": \"$context.error.responseType\" }"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__cca66e7fdb8010ce9c3a24eadb556b1a4419dd8f0e1cac4e7eb36db92f597f8c)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument response_headers", value=response_headers, expected_type=type_hints["response_headers"])
            check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
            check_type(argname="argument templates", value=templates, expected_type=type_hints["templates"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }
        if response_headers is not None:
            self._values["response_headers"] = response_headers
        if status_code is not None:
            self._values["status_code"] = status_code
        if templates is not None:
            self._values["templates"] = templates

    @builtins.property
    def type(self) -> "ResponseType":
        '''Response type to associate with gateway response.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("ResponseType", result)

    @builtins.property
    def response_headers(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Custom headers parameters for response.

        :default: - no headers
        '''
        result = self._values.get("response_headers")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def status_code(self) -> typing.Optional[builtins.str]:
        '''Http status code for response.

        :default: - standard http status code for the response type.
        '''
        result = self._values.get("status_code")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def templates(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Custom templates to get mapped as response.

        :default: - Response from api will be returned without applying any transformation.
        '''
        result = self._values.get("templates")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayResponseOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.GatewayResponseProps",
    jsii_struct_bases=[GatewayResponseOptions],
    name_mapping={
        "type": "type",
        "response_headers": "responseHeaders",
        "status_code": "statusCode",
        "templates": "templates",
        "rest_api": "restApi",
    },
)
class GatewayResponseProps(GatewayResponseOptions):
    def __init__(
        self,
        *,
        type: "ResponseType",
        response_headers: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        status_code: typing.Optional[builtins.str] = None,
        templates: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        rest_api: "IRestApi",
    ) -> None:
        '''Properties for a new gateway response.

        :param type: Response type to associate with gateway response.
        :param response_headers: Custom headers parameters for response. Default: - no headers
        :param status_code: Http status code for response. Default: - standard http status code for the response type.
        :param templates: Custom templates to get mapped as response. Default: - Response from api will be returned without applying any transformation.
        :param rest_api: Rest api resource to target.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            # response_type: apigateway.ResponseType
            # rest_api: apigateway.RestApi
            
            gateway_response_props = apigateway.GatewayResponseProps(
                rest_api=rest_api,
                type=response_type,
            
                # the properties below are optional
                response_headers={
                    "response_headers_key": "responseHeaders"
                },
                status_code="statusCode",
                templates={
                    "templates_key": "templates"
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d3681bf1068bec1901fefcd7da341a1a9cf04c68e5a08852e94f3606c772fa12)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument response_headers", value=response_headers, expected_type=type_hints["response_headers"])
            check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
            check_type(argname="argument templates", value=templates, expected_type=type_hints["templates"])
            check_type(argname="argument rest_api", value=rest_api, expected_type=type_hints["rest_api"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
            "rest_api": rest_api,
        }
        if response_headers is not None:
            self._values["response_headers"] = response_headers
        if status_code is not None:
            self._values["status_code"] = status_code
        if templates is not None:
            self._values["templates"] = templates

    @builtins.property
    def type(self) -> "ResponseType":
        '''Response type to associate with gateway response.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("ResponseType", result)

    @builtins.property
    def response_headers(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Custom headers parameters for response.

        :default: - no headers
        '''
        result = self._values.get("response_headers")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def status_code(self) -> typing.Optional[builtins.str]:
        '''Http status code for response.

        :default: - standard http status code for the response type.
        '''
        result = self._values.get("status_code")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def templates(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Custom templates to get mapped as response.

        :default: - Response from api will be returned without applying any transformation.
        '''
        result = self._values.get("templates")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def rest_api(self) -> "IRestApi":
        '''Rest api resource to target.'''
        result = self._values.get("rest_api")
        assert result is not None, "Required property 'rest_api' is missing"
        return typing.cast("IRestApi", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayResponseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.HttpIntegrationProps",
    jsii_struct_bases=[],
    name_mapping={"http_method": "httpMethod", "options": "options", "proxy": "proxy"},
)
class HttpIntegrationProps:
    def __init__(
        self,
        *,
        http_method: typing.Optional[builtins.str] = None,
        options: typing.Optional[typing.Union["IntegrationOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        proxy: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param http_method: HTTP method to use when invoking the backend URL. Default: GET
        :param options: Integration options, such as request/resopnse mapping, content handling, etc. Default: defaults based on ``IntegrationOptions`` defaults
        :param proxy: Determines whether to use proxy integration or custom integration. Default: true

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_apigateway as apigateway
            from aws_cdk import aws_iam as iam
            
            # role: iam.Role
            # vpc_link: apigateway.VpcLink
            
            http_integration_props = apigateway.HttpIntegrationProps(
                http_method="httpMethod",
                options=apigateway.IntegrationOptions(
                    cache_key_parameters=["cacheKeyParameters"],
                    cache_namespace="cacheNamespace",
                    connection_type=apigateway.ConnectionType.INTERNET,
                    content_handling=apigateway.ContentHandling.CONVERT_TO_BINARY,
                    credentials_passthrough=False,
                    credentials_role=role,
                    integration_responses=[apigateway.IntegrationResponse(
                        status_code="statusCode",
            
                        # the properties below are optional
                        content_handling=apigateway.ContentHandling.CONVERT_TO_BINARY,
                        response_parameters={
                            "response_parameters_key": "responseParameters"
                        },
                        response_templates={
                            "response_templates_key": "responseTemplates"
                        },
                        selection_pattern="selectionPattern"
                    )],
                    passthrough_behavior=apigateway.PassthroughBehavior.WHEN_NO_MATCH,
                    request_parameters={
                        "request_parameters_key": "requestParameters"
                    },
                    request_templates={
                        "request_templates_key": "requestTemplates"
                    },
                    timeout=cdk.Duration.minutes(30),
                    vpc_link=vpc_link
                ),
                proxy=False
            )
        '''
        if isinstance(options, dict):
            options = IntegrationOptions(**options)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e3a97849ce8a01a0a99c02ae97daf2651b1124b4c2bac9a50b3dbb318a1fc392)
            check_type(argname="argument http_method", value=http_method, expected_type=type_hints["http_method"])
            check_type(argname="argument options", value=options, expected_type=type_hints["options"])
            check_type(argname="argument proxy", value=proxy, expected_type=type_hints["proxy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if http_method is not None:
            self._values["http_method"] = http_method
        if options is not None:
            self._values["options"] = options
        if proxy is not None:
            self._values["proxy"] = proxy

    @builtins.property
    def http_method(self) -> typing.Optional[builtins.str]:
        '''HTTP method to use when invoking the backend URL.

        :default: GET
        '''
        result = self._values.get("http_method")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def options(self) -> typing.Optional["IntegrationOptions"]:
        '''Integration options, such as request/resopnse mapping, content handling, etc.

        :default: defaults based on ``IntegrationOptions`` defaults
        '''
        result = self._values.get("options")
        return typing.cast(typing.Optional["IntegrationOptions"], result)

    @builtins.property
    def proxy(self) -> typing.Optional[builtins.bool]:
        '''Determines whether to use proxy integration or custom integration.

        :default: true
        '''
        result = self._values.get("proxy")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpIntegrationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IAccessLogDestination")
class IAccessLogDestination(typing_extensions.Protocol):
    '''Access log destination for a RestApi Stage.'''

    @jsii.member(jsii_name="bind")
    def bind(self, stage: "IStage") -> AccessLogDestinationConfig:
        '''Binds this destination to the RestApi Stage.

        :param stage: -
        '''
        ...


class _IAccessLogDestinationProxy:
    '''Access log destination for a RestApi Stage.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IAccessLogDestination"

    @jsii.member(jsii_name="bind")
    def bind(self, stage: "IStage") -> AccessLogDestinationConfig:
        '''Binds this destination to the RestApi Stage.

        :param stage: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5584cb6e6dfde7a0ce69d852df52ee6697c36df3a33e7d95f1f8945c57eb76d0)
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
        return typing.cast(AccessLogDestinationConfig, jsii.invoke(self, "bind", [stage]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAccessLogDestination).__jsii_proxy_class__ = lambda : _IAccessLogDestinationProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IApiKey")
class IApiKey(_IResource_c80c4260, typing_extensions.Protocol):
    '''API keys are alphanumeric string values that you distribute to app developer customers to grant access to your API.'''

    @builtins.property
    @jsii.member(jsii_name="keyArn")
    def key_arn(self) -> builtins.str:
        '''The API key ARN.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="keyId")
    def key_id(self) -> builtins.str:
        '''The API key ID.

        :attribute: true
        '''
        ...


class _IApiKeyProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''API keys are alphanumeric string values that you distribute to app developer customers to grant access to your API.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IApiKey"

    @builtins.property
    @jsii.member(jsii_name="keyArn")
    def key_arn(self) -> builtins.str:
        '''The API key ARN.'''
        return typing.cast(builtins.str, jsii.get(self, "keyArn"))

    @builtins.property
    @jsii.member(jsii_name="keyId")
    def key_id(self) -> builtins.str:
        '''The API key ID.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "keyId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IApiKey).__jsii_proxy_class__ = lambda : _IApiKeyProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IAuthorizer")
class IAuthorizer(typing_extensions.Protocol):
    '''Represents an API Gateway authorizer.'''

    @builtins.property
    @jsii.member(jsii_name="authorizerId")
    def authorizer_id(self) -> builtins.str:
        '''The authorizer ID.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="authorizationType")
    def authorization_type(self) -> typing.Optional[AuthorizationType]:
        '''The authorization type of this authorizer.'''
        ...


class _IAuthorizerProxy:
    '''Represents an API Gateway authorizer.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IAuthorizer"

    @builtins.property
    @jsii.member(jsii_name="authorizerId")
    def authorizer_id(self) -> builtins.str:
        '''The authorizer ID.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "authorizerId"))

    @builtins.property
    @jsii.member(jsii_name="authorizationType")
    def authorization_type(self) -> typing.Optional[AuthorizationType]:
        '''The authorization type of this authorizer.'''
        return typing.cast(typing.Optional[AuthorizationType], jsii.get(self, "authorizationType"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAuthorizer).__jsii_proxy_class__ = lambda : _IAuthorizerProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IDomainName")
class IDomainName(_IResource_c80c4260, typing_extensions.Protocol):
    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> builtins.str:
        '''The domain name (e.g. ``example.com``).

        :attribute: DomainName
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="domainNameAliasDomainName")
    def domain_name_alias_domain_name(self) -> builtins.str:
        '''The Route53 alias target to use in order to connect a record set to this domain through an alias.

        :attribute: DistributionDomainName,RegionalDomainName
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="domainNameAliasHostedZoneId")
    def domain_name_alias_hosted_zone_id(self) -> builtins.str:
        '''The Route53 hosted zone ID to use in order to connect a record set to this domain through an alias.

        :attribute: DistributionHostedZoneId,RegionalHostedZoneId
        '''
        ...


class _IDomainNameProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IDomainName"

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> builtins.str:
        '''The domain name (e.g. ``example.com``).

        :attribute: DomainName
        '''
        return typing.cast(builtins.str, jsii.get(self, "domainName"))

    @builtins.property
    @jsii.member(jsii_name="domainNameAliasDomainName")
    def domain_name_alias_domain_name(self) -> builtins.str:
        '''The Route53 alias target to use in order to connect a record set to this domain through an alias.

        :attribute: DistributionDomainName,RegionalDomainName
        '''
        return typing.cast(builtins.str, jsii.get(self, "domainNameAliasDomainName"))

    @builtins.property
    @jsii.member(jsii_name="domainNameAliasHostedZoneId")
    def domain_name_alias_hosted_zone_id(self) -> builtins.str:
        '''The Route53 hosted zone ID to use in order to connect a record set to this domain through an alias.

        :attribute: DistributionHostedZoneId,RegionalHostedZoneId
        '''
        return typing.cast(builtins.str, jsii.get(self, "domainNameAliasHostedZoneId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IDomainName).__jsii_proxy_class__ = lambda : _IDomainNameProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IGatewayResponse")
class IGatewayResponse(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents gateway response resource.'''

    pass


class _IGatewayResponseProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents gateway response resource.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IGatewayResponse"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IGatewayResponse).__jsii_proxy_class__ = lambda : _IGatewayResponseProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IModel")
class IModel(typing_extensions.Protocol):
    @builtins.property
    @jsii.member(jsii_name="modelId")
    def model_id(self) -> builtins.str:
        '''Returns the model name, such as 'myModel'.

        :attribute: true
        '''
        ...


class _IModelProxy:
    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IModel"

    @builtins.property
    @jsii.member(jsii_name="modelId")
    def model_id(self) -> builtins.str:
        '''Returns the model name, such as 'myModel'.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "modelId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IModel).__jsii_proxy_class__ = lambda : _IModelProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IRequestValidator")
class IRequestValidator(_IResource_c80c4260, typing_extensions.Protocol):
    @builtins.property
    @jsii.member(jsii_name="requestValidatorId")
    def request_validator_id(self) -> builtins.str:
        '''ID of the request validator, such as abc123.

        :attribute: true
        '''
        ...


class _IRequestValidatorProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IRequestValidator"

    @builtins.property
    @jsii.member(jsii_name="requestValidatorId")
    def request_validator_id(self) -> builtins.str:
        '''ID of the request validator, such as abc123.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "requestValidatorId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IRequestValidator).__jsii_proxy_class__ = lambda : _IRequestValidatorProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IResource")
class IResource(_IResource_c80c4260, typing_extensions.Protocol):
    @builtins.property
    @jsii.member(jsii_name="api")
    def api(self) -> "IRestApi":
        '''The rest API that this resource is part of.

        The reason we need the RestApi object itself and not just the ID is because the model
        is being tracked by the top-level RestApi object for the purpose of calculating it's
        hash to determine the ID of the deployment. This allows us to automatically update
        the deployment when the model of the REST API changes.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="path")
    def path(self) -> builtins.str:
        '''The full path of this resource.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="resourceId")
    def resource_id(self) -> builtins.str:
        '''The ID of the resource.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="defaultCorsPreflightOptions")
    def default_cors_preflight_options(self) -> typing.Optional[CorsOptions]:
        '''Default options for CORS preflight OPTIONS method.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="defaultIntegration")
    def default_integration(self) -> typing.Optional["Integration"]:
        '''An integration to use as a default for all methods created within this API unless an integration is specified.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="defaultMethodOptions")
    def default_method_options(self) -> typing.Optional["MethodOptions"]:
        '''Method options to use as a default for all methods created within this API unless custom options are specified.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="parentResource")
    def parent_resource(self) -> typing.Optional["IResource"]:
        '''The parent of this resource or undefined for the root resource.'''
        ...

    @jsii.member(jsii_name="addCorsPreflight")
    def add_cors_preflight(
        self,
        *,
        allow_origins: typing.Sequence[builtins.str],
        allow_credentials: typing.Optional[builtins.bool] = None,
        allow_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        allow_methods: typing.Optional[typing.Sequence[builtins.str]] = None,
        disable_cache: typing.Optional[builtins.bool] = None,
        expose_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        max_age: typing.Optional[_Duration_4839e8c3] = None,
        status_code: typing.Optional[jsii.Number] = None,
    ) -> "Method":
        '''Adds an OPTIONS method to this resource which responds to Cross-Origin Resource Sharing (CORS) preflight requests.

        Cross-Origin Resource Sharing (CORS) is a mechanism that uses additional
        HTTP headers to tell browsers to give a web application running at one
        origin, access to selected resources from a different origin. A web
        application executes a cross-origin HTTP request when it requests a
        resource that has a different origin (domain, protocol, or port) from its
        own.

        :param allow_origins: Specifies the list of origins that are allowed to make requests to this resource. If you wish to allow all origins, specify ``Cors.ALL_ORIGINS`` or ``[ * ]``. Responses will include the ``Access-Control-Allow-Origin`` response header. If specific origins are specified (not ``Cors.ALL_ORIGINS``), the ``Vary: Origin`` response header will also be included.
        :param allow_credentials: The Access-Control-Allow-Credentials response header tells browsers whether to expose the response to frontend JavaScript code when the request's credentials mode (Request.credentials) is "include". When a request's credentials mode (Request.credentials) is "include", browsers will only expose the response to frontend JavaScript code if the Access-Control-Allow-Credentials value is true. Credentials are cookies, authorization headers or TLS client certificates. Default: false
        :param allow_headers: The Access-Control-Allow-Headers response header is used in response to a preflight request which includes the Access-Control-Request-Headers to indicate which HTTP headers can be used during the actual request. Default: Cors.DEFAULT_HEADERS
        :param allow_methods: The Access-Control-Allow-Methods response header specifies the method or methods allowed when accessing the resource in response to a preflight request. If ``ANY`` is specified, it will be expanded to ``Cors.ALL_METHODS``. Default: Cors.ALL_METHODS
        :param disable_cache: Sets Access-Control-Max-Age to -1, which means that caching is disabled. This option cannot be used with ``maxAge``. Default: - cache is enabled
        :param expose_headers: The Access-Control-Expose-Headers response header indicates which headers can be exposed as part of the response by listing their names. If you want clients to be able to access other headers, you have to list them using the Access-Control-Expose-Headers header. Default: - only the 6 CORS-safelisted response headers are exposed: Cache-Control, Content-Language, Content-Type, Expires, Last-Modified, Pragma
        :param max_age: The Access-Control-Max-Age response header indicates how long the results of a preflight request (that is the information contained in the Access-Control-Allow-Methods and Access-Control-Allow-Headers headers) can be cached. To disable caching altogether use ``disableCache: true``. Default: - browser-specific (see reference)
        :param status_code: Specifies the response status code returned from the OPTIONS method. Default: 204

        :return: a ``Method`` object

        :see: https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS
        '''
        ...

    @jsii.member(jsii_name="addMethod")
    def add_method(
        self,
        http_method: builtins.str,
        target: typing.Optional["Integration"] = None,
        *,
        api_key_required: typing.Optional[builtins.bool] = None,
        authorization_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        authorization_type: typing.Optional[AuthorizationType] = None,
        authorizer: typing.Optional[IAuthorizer] = None,
        method_responses: typing.Optional[typing.Sequence[typing.Union["MethodResponse", typing.Dict[builtins.str, typing.Any]]]] = None,
        operation_name: typing.Optional[builtins.str] = None,
        request_models: typing.Optional[typing.Mapping[builtins.str, IModel]] = None,
        request_parameters: typing.Optional[typing.Mapping[builtins.str, builtins.bool]] = None,
        request_validator: typing.Optional[IRequestValidator] = None,
        request_validator_options: typing.Optional[typing.Union["RequestValidatorOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "Method":
        '''Defines a new method for this resource.

        :param http_method: The HTTP method.
        :param target: The target backend integration for this method.
        :param api_key_required: Indicates whether the method requires clients to submit a valid API key. Default: false
        :param authorization_scopes: A list of authorization scopes configured on the method. The scopes are used with a COGNITO_USER_POOLS authorizer to authorize the method invocation. Default: - no authorization scopes
        :param authorization_type: Method authorization. If the value is set of ``Custom``, an ``authorizer`` must also be specified. If you're using one of the authorizers that are available via the ``Authorizer`` class, such as ``Authorizer#token()``, it is recommended that this option not be specified. The authorizer will take care of setting the correct authorization type. However, specifying an authorization type using this property that conflicts with what is expected by the ``Authorizer`` will result in an error. Default: - open access unless ``authorizer`` is specified
        :param authorizer: If ``authorizationType`` is ``Custom``, this specifies the ID of the method authorizer resource. If specified, the value of ``authorizationType`` must be set to ``Custom``
        :param method_responses: The responses that can be sent to the client who calls the method. Default: None This property is not required, but if these are not supplied for a Lambda proxy integration, the Lambda function must return a value of the correct format, for the integration response to be correctly mapped to a response to the client.
        :param operation_name: A friendly operation name for the method. For example, you can assign the OperationName of ListPets for the GET /pets method.
        :param request_models: The models which describe data structure of request payload. When combined with ``requestValidator`` or ``requestValidatorOptions``, the service will validate the API request payload before it reaches the API's Integration (including proxies). Specify ``requestModels`` as key-value pairs, with a content type (e.g. ``'application/json'``) as the key and an API Gateway Model as the value.
        :param request_parameters: The request parameters that API Gateway accepts. Specify request parameters as key-value pairs (string-to-Boolean mapping), with a source as the key and a Boolean as the value. The Boolean specifies whether a parameter is required. A source must match the format method.request.location.name, where the location is querystring, path, or header, and name is a valid, unique parameter name. Default: None
        :param request_validator: The ID of the associated request validator. Only one of ``requestValidator`` or ``requestValidatorOptions`` must be specified. Works together with ``requestModels`` or ``requestParameters`` to validate the request before it reaches integration like Lambda Proxy Integration. Default: - No default validator
        :param request_validator_options: Request validator options to create new validator Only one of ``requestValidator`` or ``requestValidatorOptions`` must be specified. Works together with ``requestModels`` or ``requestParameters`` to validate the request before it reaches integration like Lambda Proxy Integration. Default: - No default validator

        :return: The newly created ``Method`` object.
        '''
        ...

    @jsii.member(jsii_name="addProxy")
    def add_proxy(
        self,
        *,
        any_method: typing.Optional[builtins.bool] = None,
        default_cors_preflight_options: typing.Optional[typing.Union[CorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        default_integration: typing.Optional["Integration"] = None,
        default_method_options: typing.Optional[typing.Union["MethodOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "ProxyResource":
        '''Adds a greedy proxy resource ("{proxy+}") and an ANY method to this route.

        :param any_method: Adds an "ANY" method to this resource. If set to ``false``, you will have to explicitly add methods to this resource after it's created. Default: true
        :param default_cors_preflight_options: Adds a CORS preflight OPTIONS method to this resource and all child resources. You can add CORS at the resource-level using ``addCorsPreflight``. Default: - CORS is disabled
        :param default_integration: An integration to use as a default for all methods created within this API unless an integration is specified. Default: - Inherited from parent.
        :param default_method_options: Method options to use as a default for all methods created within this API unless custom options are specified. Default: - Inherited from parent.
        '''
        ...

    @jsii.member(jsii_name="addResource")
    def add_resource(
        self,
        path_part: builtins.str,
        *,
        default_cors_preflight_options: typing.Optional[typing.Union[CorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        default_integration: typing.Optional["Integration"] = None,
        default_method_options: typing.Optional[typing.Union["MethodOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "Resource":
        '''Defines a new child resource where this resource is the parent.

        :param path_part: The path part for the child resource.
        :param default_cors_preflight_options: Adds a CORS preflight OPTIONS method to this resource and all child resources. You can add CORS at the resource-level using ``addCorsPreflight``. Default: - CORS is disabled
        :param default_integration: An integration to use as a default for all methods created within this API unless an integration is specified. Default: - Inherited from parent.
        :param default_method_options: Method options to use as a default for all methods created within this API unless custom options are specified. Default: - Inherited from parent.

        :return: A Resource object
        '''
        ...

    @jsii.member(jsii_name="getResource")
    def get_resource(self, path_part: builtins.str) -> typing.Optional["IResource"]:
        '''Retrieves a child resource by path part.

        :param path_part: The path part of the child resource.

        :return: the child resource or undefined if not found
        '''
        ...

    @jsii.member(jsii_name="resourceForPath")
    def resource_for_path(self, path: builtins.str) -> "Resource":
        '''Gets or create all resources leading up to the specified path.

        - Path may only start with "/" if this method is called on the root resource.
        - All resources are created using default options.

        :param path: The relative path.

        :return: a new or existing resource.
        '''
        ...


class _IResourceProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IResource"

    @builtins.property
    @jsii.member(jsii_name="api")
    def api(self) -> "IRestApi":
        '''The rest API that this resource is part of.

        The reason we need the RestApi object itself and not just the ID is because the model
        is being tracked by the top-level RestApi object for the purpose of calculating it's
        hash to determine the ID of the deployment. This allows us to automatically update
        the deployment when the model of the REST API changes.
        '''
        return typing.cast("IRestApi", jsii.get(self, "api"))

    @builtins.property
    @jsii.member(jsii_name="path")
    def path(self) -> builtins.str:
        '''The full path of this resource.'''
        return typing.cast(builtins.str, jsii.get(self, "path"))

    @builtins.property
    @jsii.member(jsii_name="resourceId")
    def resource_id(self) -> builtins.str:
        '''The ID of the resource.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "resourceId"))

    @builtins.property
    @jsii.member(jsii_name="defaultCorsPreflightOptions")
    def default_cors_preflight_options(self) -> typing.Optional[CorsOptions]:
        '''Default options for CORS preflight OPTIONS method.'''
        return typing.cast(typing.Optional[CorsOptions], jsii.get(self, "defaultCorsPreflightOptions"))

    @builtins.property
    @jsii.member(jsii_name="defaultIntegration")
    def default_integration(self) -> typing.Optional["Integration"]:
        '''An integration to use as a default for all methods created within this API unless an integration is specified.'''
        return typing.cast(typing.Optional["Integration"], jsii.get(self, "defaultIntegration"))

    @builtins.property
    @jsii.member(jsii_name="defaultMethodOptions")
    def default_method_options(self) -> typing.Optional["MethodOptions"]:
        '''Method options to use as a default for all methods created within this API unless custom options are specified.'''
        return typing.cast(typing.Optional["MethodOptions"], jsii.get(self, "defaultMethodOptions"))

    @builtins.property
    @jsii.member(jsii_name="parentResource")
    def parent_resource(self) -> typing.Optional[IResource]:
        '''The parent of this resource or undefined for the root resource.'''
        return typing.cast(typing.Optional[IResource], jsii.get(self, "parentResource"))

    @jsii.member(jsii_name="addCorsPreflight")
    def add_cors_preflight(
        self,
        *,
        allow_origins: typing.Sequence[builtins.str],
        allow_credentials: typing.Optional[builtins.bool] = None,
        allow_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        allow_methods: typing.Optional[typing.Sequence[builtins.str]] = None,
        disable_cache: typing.Optional[builtins.bool] = None,
        expose_headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        max_age: typing.Optional[_Duration_4839e8c3] = None,
        status_code: typing.Optional[jsii.Number] = None,
    ) -> "Method":
        '''Adds an OPTIONS method to this resource which responds to Cross-Origin Resource Sharing (CORS) preflight requests.

        Cross-Origin Resource Sharing (CORS) is a mechanism that uses additional
        HTTP headers to tell browsers to give a web application running at one
        origin, access to selected resources from a different origin. A web
        application executes a cross-origin HTTP request when it requests a
        resource that has a different origin (domain, protocol, or port) from its
        own.

        :param allow_origins: Specifies the list of origins that are allowed to make requests to this resource. If you wish to allow all origins, specify ``Cors.ALL_ORIGINS`` or ``[ * ]``. Responses will include the ``Access-Control-Allow-Origin`` response header. If specific origins are specified (not ``Cors.ALL_ORIGINS``), the ``Vary: Origin`` response header will also be included.
        :param allow_credentials: The Access-Control-Allow-Credentials response header tells browsers whether to expose the response to frontend JavaScript code when the request's credentials mode (Request.credentials) is "include". When a request's credentials mode (Request.credentials) is "include", browsers will only expose the response to frontend JavaScript code if the Access-Control-Allow-Credentials value is true. Credentials are cookies, authorization headers or TLS client certificates. Default: false
        :param allow_headers: The Access-Control-Allow-Headers response header is used in response to a preflight request which includes the Access-Control-Request-Headers to indicate which HTTP headers can be used during the actual request. Default: Cors.DEFAULT_HEADERS
        :param allow_methods: The Access-Control-Allow-Methods response header specifies the method or methods allowed when accessing the resource in response to a preflight request. If ``ANY`` is specified, it will be expanded to ``Cors.ALL_METHODS``. Default: Cors.ALL_METHODS
        :param disable_cache: Sets Access-Control-Max-Age to -1, which means that caching is disabled. This option cannot be used with ``maxAge``. Default: - cache is enabled
        :param expose_headers: The Access-Control-Expose-Headers response header indicates which headers can be exposed as part of the response by listing their names. If you want clients to be able to access other headers, you have to list them using the Access-Control-Expose-Headers header. Default: - only the 6 CORS-safelisted response headers are exposed: Cache-Control, Content-Language, Content-Type, Expires, Last-Modified, Pragma
        :param max_age: The Access-Control-Max-Age response header indicates how long the results of a preflight request (that is the information contained in the Access-Control-Allow-Methods and Access-Control-Allow-Headers headers) can be cached. To disable caching altogether use ``disableCache: true``. Default: - browser-specific (see reference)
        :param status_code: Specifies the response status code returned from the OPTIONS method. Default: 204

        :return: a ``Method`` object

        :see: https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS
        '''
        options = CorsOptions(
            allow_origins=allow_origins,
            allow_credentials=allow_credentials,
            allow_headers=allow_headers,
            allow_methods=allow_methods,
            disable_cache=disable_cache,
            expose_headers=expose_headers,
            max_age=max_age,
            status_code=status_code,
        )

        return typing.cast("Method", jsii.invoke(self, "addCorsPreflight", [options]))

    @jsii.member(jsii_name="addMethod")
    def add_method(
        self,
        http_method: builtins.str,
        target: typing.Optional["Integration"] = None,
        *,
        api_key_required: typing.Optional[builtins.bool] = None,
        authorization_scopes: typing.Optional[typing.Sequence[builtins.str]] = None,
        authorization_type: typing.Optional[AuthorizationType] = None,
        authorizer: typing.Optional[IAuthorizer] = None,
        method_responses: typing.Optional[typing.Sequence[typing.Union["MethodResponse", typing.Dict[builtins.str, typing.Any]]]] = None,
        operation_name: typing.Optional[builtins.str] = None,
        request_models: typing.Optional[typing.Mapping[builtins.str, IModel]] = None,
        request_parameters: typing.Optional[typing.Mapping[builtins.str, builtins.bool]] = None,
        request_validator: typing.Optional[IRequestValidator] = None,
        request_validator_options: typing.Optional[typing.Union["RequestValidatorOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "Method":
        '''Defines a new method for this resource.

        :param http_method: The HTTP method.
        :param target: The target backend integration for this method.
        :param api_key_required: Indicates whether the method requires clients to submit a valid API key. Default: false
        :param authorization_scopes: A list of authorization scopes configured on the method. The scopes are used with a COGNITO_USER_POOLS authorizer to authorize the method invocation. Default: - no authorization scopes
        :param authorization_type: Method authorization. If the value is set of ``Custom``, an ``authorizer`` must also be specified. If you're using one of the authorizers that are available via the ``Authorizer`` class, such as ``Authorizer#token()``, it is recommended that this option not be specified. The authorizer will take care of setting the correct authorization type. However, specifying an authorization type using this property that conflicts with what is expected by the ``Authorizer`` will result in an error. Default: - open access unless ``authorizer`` is specified
        :param authorizer: If ``authorizationType`` is ``Custom``, this specifies the ID of the method authorizer resource. If specified, the value of ``authorizationType`` must be set to ``Custom``
        :param method_responses: The responses that can be sent to the client who calls the method. Default: None This property is not required, but if these are not supplied for a Lambda proxy integration, the Lambda function must return a value of the correct format, for the integration response to be correctly mapped to a response to the client.
        :param operation_name: A friendly operation name for the method. For example, you can assign the OperationName of ListPets for the GET /pets method.
        :param request_models: The models which describe data structure of request payload. When combined with ``requestValidator`` or ``requestValidatorOptions``, the service will validate the API request payload before it reaches the API's Integration (including proxies). Specify ``requestModels`` as key-value pairs, with a content type (e.g. ``'application/json'``) as the key and an API Gateway Model as the value.
        :param request_parameters: The request parameters that API Gateway accepts. Specify request parameters as key-value pairs (string-to-Boolean mapping), with a source as the key and a Boolean as the value. The Boolean specifies whether a parameter is required. A source must match the format method.request.location.name, where the location is querystring, path, or header, and name is a valid, unique parameter name. Default: None
        :param request_validator: The ID of the associated request validator. Only one of ``requestValidator`` or ``requestValidatorOptions`` must be specified. Works together with ``requestModels`` or ``requestParameters`` to validate the request before it reaches integration like Lambda Proxy Integration. Default: - No default validator
        :param request_validator_options: Request validator options to create new validator Only one of ``requestValidator`` or ``requestValidatorOptions`` must be specified. Works together with ``requestModels`` or ``requestParameters`` to validate the request before it reaches integration like Lambda Proxy Integration. Default: - No default validator

        :return: The newly created ``Method`` object.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6182792530fe5d451f78de1094a258af7e652a3caeea9cd32eec5de25c89a689)
            check_type(argname="argument http_method", value=http_method, expected_type=type_hints["http_method"])
            check_type(argname="argument target", value=target, expected_type=type_hints["target"])
        options = MethodOptions(
            api_key_required=api_key_required,
            authorization_scopes=authorization_scopes,
            authorization_type=authorization_type,
            authorizer=authorizer,
            method_responses=method_responses,
            operation_name=operation_name,
            request_models=request_models,
            request_parameters=request_parameters,
            request_validator=request_validator,
            request_validator_options=request_validator_options,
        )

        return typing.cast("Method", jsii.invoke(self, "addMethod", [http_method, target, options]))

    @jsii.member(jsii_name="addProxy")
    def add_proxy(
        self,
        *,
        any_method: typing.Optional[builtins.bool] = None,
        default_cors_preflight_options: typing.Optional[typing.Union[CorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        default_integration: typing.Optional["Integration"] = None,
        default_method_options: typing.Optional[typing.Union["MethodOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "ProxyResource":
        '''Adds a greedy proxy resource ("{proxy+}") and an ANY method to this route.

        :param any_method: Adds an "ANY" method to this resource. If set to ``false``, you will have to explicitly add methods to this resource after it's created. Default: true
        :param default_cors_preflight_options: Adds a CORS preflight OPTIONS method to this resource and all child resources. You can add CORS at the resource-level using ``addCorsPreflight``. Default: - CORS is disabled
        :param default_integration: An integration to use as a default for all methods created within this API unless an integration is specified. Default: - Inherited from parent.
        :param default_method_options: Method options to use as a default for all methods created within this API unless custom options are specified. Default: - Inherited from parent.
        '''
        options = ProxyResourceOptions(
            any_method=any_method,
            default_cors_preflight_options=default_cors_preflight_options,
            default_integration=default_integration,
            default_method_options=default_method_options,
        )

        return typing.cast("ProxyResource", jsii.invoke(self, "addProxy", [options]))

    @jsii.member(jsii_name="addResource")
    def add_resource(
        self,
        path_part: builtins.str,
        *,
        default_cors_preflight_options: typing.Optional[typing.Union[CorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        default_integration: typing.Optional["Integration"] = None,
        default_method_options: typing.Optional[typing.Union["MethodOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> "Resource":
        '''Defines a new child resource where this resource is the parent.

        :param path_part: The path part for the child resource.
        :param default_cors_preflight_options: Adds a CORS preflight OPTIONS method to this resource and all child resources. You can add CORS at the resource-level using ``addCorsPreflight``. Default: - CORS is disabled
        :param default_integration: An integration to use as a default for all methods created within this API unless an integration is specified. Default: - Inherited from parent.
        :param default_method_options: Method options to use as a default for all methods created within this API unless custom options are specified. Default: - Inherited from parent.

        :return: A Resource object
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8bfdf0ee0d3cce4d37028932e9b642a2c73a0f911109e1152cc967ae73d762df)
            check_type(argname="argument path_part", value=path_part, expected_type=type_hints["path_part"])
        options = ResourceOptions(
            default_cors_preflight_options=default_cors_preflight_options,
            default_integration=default_integration,
            default_method_options=default_method_options,
        )

        return typing.cast("Resource", jsii.invoke(self, "addResource", [path_part, options]))

    @jsii.member(jsii_name="getResource")
    def get_resource(self, path_part: builtins.str) -> typing.Optional[IResource]:
        '''Retrieves a child resource by path part.

        :param path_part: The path part of the child resource.

        :return: the child resource or undefined if not found
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__00e0e2a62a3bcf18c94f25245cbbeaee0c7674687c580c8af8a0f658cd6e1c03)
            check_type(argname="argument path_part", value=path_part, expected_type=type_hints["path_part"])
        return typing.cast(typing.Optional[IResource], jsii.invoke(self, "getResource", [path_part]))

    @jsii.member(jsii_name="resourceForPath")
    def resource_for_path(self, path: builtins.str) -> "Resource":
        '''Gets or create all resources leading up to the specified path.

        - Path may only start with "/" if this method is called on the root resource.
        - All resources are created using default options.

        :param path: The relative path.

        :return: a new or existing resource.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a3dd70bebc1a29ae27e719fad1dfafb235291688aeebd8afda4d33cebddaf85e)
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
        return typing.cast("Resource", jsii.invoke(self, "resourceForPath", [path]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IResource).__jsii_proxy_class__ = lambda : _IResourceProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IRestApi")
class IRestApi(_IResource_c80c4260, typing_extensions.Protocol):
    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The ID of this API Gateway RestApi.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="restApiName")
    def rest_api_name(self) -> builtins.str:
        '''The name of this API Gateway RestApi.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="restApiRootResourceId")
    def rest_api_root_resource_id(self) -> builtins.str:
        '''The resource ID of the root resource.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="root")
    def root(self) -> IResource:
        '''Represents the root resource ("/") of this API. Use it to define the API model:.

        api.root.addMethod('ANY', redirectToHomePage); // "ANY /"
        api.root.addResource('friends').addMethod('GET', getFriendsHandler); // "GET /friends"
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="latestDeployment")
    def latest_deployment(self) -> typing.Optional[Deployment]:
        '''API Gateway deployment that represents the latest changes of the API.

        This resource will be automatically updated every time the REST API model changes.
        ``undefined`` when no deployment is configured.
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="deploymentStage")
    def deployment_stage(self) -> "Stage":
        '''API Gateway stage that points to the latest deployment (if defined).'''
        ...

    @deployment_stage.setter
    def deployment_stage(self, value: "Stage") -> None:
        ...

    @jsii.member(jsii_name="arnForExecuteApi")
    def arn_for_execute_api(
        self,
        method: typing.Optional[builtins.str] = None,
        path: typing.Optional[builtins.str] = None,
        stage: typing.Optional[builtins.str] = None,
    ) -> builtins.str:
        '''Gets the "execute-api" ARN.

        :param method: The method (default ``*``).
        :param path: The resource path. Must start with '/' (default ``*``)
        :param stage: The stage (default ``*``).

        :default:

        "*" returns the execute API ARN for all methods/resources in
        this API.

        :return: The "execute-api" ARN.
        '''
        ...


class _IRestApiProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IRestApi"

    @builtins.property
    @jsii.member(jsii_name="restApiId")
    def rest_api_id(self) -> builtins.str:
        '''The ID of this API Gateway RestApi.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiId"))

    @builtins.property
    @jsii.member(jsii_name="restApiName")
    def rest_api_name(self) -> builtins.str:
        '''The name of this API Gateway RestApi.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiName"))

    @builtins.property
    @jsii.member(jsii_name="restApiRootResourceId")
    def rest_api_root_resource_id(self) -> builtins.str:
        '''The resource ID of the root resource.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "restApiRootResourceId"))

    @builtins.property
    @jsii.member(jsii_name="root")
    def root(self) -> IResource:
        '''Represents the root resource ("/") of this API. Use it to define the API model:.

        api.root.addMethod('ANY', redirectToHomePage); // "ANY /"
        api.root.addResource('friends').addMethod('GET', getFriendsHandler); // "GET /friends"
        '''
        return typing.cast(IResource, jsii.get(self, "root"))

    @builtins.property
    @jsii.member(jsii_name="latestDeployment")
    def latest_deployment(self) -> typing.Optional[Deployment]:
        '''API Gateway deployment that represents the latest changes of the API.

        This resource will be automatically updated every time the REST API model changes.
        ``undefined`` when no deployment is configured.
        '''
        return typing.cast(typing.Optional[Deployment], jsii.get(self, "latestDeployment"))

    @builtins.property
    @jsii.member(jsii_name="deploymentStage")
    def deployment_stage(self) -> "Stage":
        '''API Gateway stage that points to the latest deployment (if defined).'''
        return typing.cast("Stage", jsii.get(self, "deploymentStage"))

    @deployment_stage.setter
    def deployment_stage(self, value: "Stage") -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe0bd06953138c343b6fef3c8a33c43e09cffe6099ff45fef8c8f71fe6ce482d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "deploymentStage", value) # pyright: ignore[reportArgumentType]

    @jsii.member(jsii_name="arnForExecuteApi")
    def arn_for_execute_api(
        self,
        method: typing.Optional[builtins.str] = None,
        path: typing.Optional[builtins.str] = None,
        stage: typing.Optional[builtins.str] = None,
    ) -> builtins.str:
        '''Gets the "execute-api" ARN.

        :param method: The method (default ``*``).
        :param path: The resource path. Must start with '/' (default ``*``)
        :param stage: The stage (default ``*``).

        :default:

        "*" returns the execute API ARN for all methods/resources in
        this API.

        :return: The "execute-api" ARN.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7d9b4612aee7a92b1fd5815b7822a95666fa1d76edb5326fe15ff34ce82bbcff)
            check_type(argname="argument method", value=method, expected_type=type_hints["method"])
            check_type(argname="argument path", value=path, expected_type=type_hints["path"])
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
        return typing.cast(builtins.str, jsii.invoke(self, "arnForExecuteApi", [method, path, stage]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IRestApi).__jsii_proxy_class__ = lambda : _IRestApiProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IStage")
class IStage(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents an APIGateway Stage.'''

    @builtins.property
    @jsii.member(jsii_name="restApi")
    def rest_api(self) -> IRestApi:
        '''RestApi to which this stage is associated.'''
        ...

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def stage_name(self) -> builtins.str:
        '''Name of this stage.

        :attribute: true
        '''
        ...

    @jsii.member(jsii_name="addApiKey")
    def add_api_key(
        self,
        id: builtins.str,
        *,
        api_key_name: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        value: typing.Optional[builtins.str] = None,
        default_cors_preflight_options: typing.Optional[typing.Union[CorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        default_integration: typing.Optional["Integration"] = None,
        default_method_options: typing.Optional[typing.Union["MethodOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> IApiKey:
        '''Add an ApiKey to this Stage.

        :param id: -
        :param api_key_name: A name for the API key. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the API key name. Default: automically generated name
        :param description: A description of the purpose of the API key. Default: none
        :param value: The value of the API key. Must be at least 20 characters long. Default: none
        :param default_cors_preflight_options: Adds a CORS preflight OPTIONS method to this resource and all child resources. You can add CORS at the resource-level using ``addCorsPreflight``. Default: - CORS is disabled
        :param default_integration: An integration to use as a default for all methods created within this API unless an integration is specified. Default: - Inherited from parent.
        :param default_method_options: Method options to use as a default for all methods created within this API unless custom options are specified. Default: - Inherited from parent.
        '''
        ...


class _IStageProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents an APIGateway Stage.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IStage"

    @builtins.property
    @jsii.member(jsii_name="restApi")
    def rest_api(self) -> IRestApi:
        '''RestApi to which this stage is associated.'''
        return typing.cast(IRestApi, jsii.get(self, "restApi"))

    @builtins.property
    @jsii.member(jsii_name="stageName")
    def stage_name(self) -> builtins.str:
        '''Name of this stage.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "stageName"))

    @jsii.member(jsii_name="addApiKey")
    def add_api_key(
        self,
        id: builtins.str,
        *,
        api_key_name: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        value: typing.Optional[builtins.str] = None,
        default_cors_preflight_options: typing.Optional[typing.Union[CorsOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        default_integration: typing.Optional["Integration"] = None,
        default_method_options: typing.Optional[typing.Union["MethodOptions", typing.Dict[builtins.str, typing.Any]]] = None,
    ) -> IApiKey:
        '''Add an ApiKey to this Stage.

        :param id: -
        :param api_key_name: A name for the API key. If you don't specify a name, AWS CloudFormation generates a unique physical ID and uses that ID for the API key name. Default: automically generated name
        :param description: A description of the purpose of the API key. Default: none
        :param value: The value of the API key. Must be at least 20 characters long. Default: none
        :param default_cors_preflight_options: Adds a CORS preflight OPTIONS method to this resource and all child resources. You can add CORS at the resource-level using ``addCorsPreflight``. Default: - CORS is disabled
        :param default_integration: An integration to use as a default for all methods created within this API unless an integration is specified. Default: - Inherited from parent.
        :param default_method_options: Method options to use as a default for all methods created within this API unless custom options are specified. Default: - Inherited from parent.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0db60e65de0b59c265539f0763570e52f29515365793132c5be8ef41b0b66c82)
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        options = ApiKeyOptions(
            api_key_name=api_key_name,
            description=description,
            value=value,
            default_cors_preflight_options=default_cors_preflight_options,
            default_integration=default_integration,
            default_method_options=default_method_options,
        )

        return typing.cast(IApiKey, jsii.invoke(self, "addApiKey", [id, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStage).__jsii_proxy_class__ = lambda : _IStageProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IUsagePlan")
class IUsagePlan(_IResource_c80c4260, typing_extensions.Protocol):
    '''A UsagePlan, either managed by this CDK app, or imported.'''

    @builtins.property
    @jsii.member(jsii_name="usagePlanId")
    def usage_plan_id(self) -> builtins.str:
        '''Id of the usage plan.

        :attribute: true
        '''
        ...

    @jsii.member(jsii_name="addApiKey")
    def add_api_key(
        self,
        api_key: IApiKey,
        *,
        override_logical_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Adds an ApiKey.

        :param api_key: the api key to associate with this usage plan.
        :param override_logical_id: Override the CloudFormation logical id of the AWS::ApiGateway::UsagePlanKey resource. Default: - autogenerated by the CDK
        '''
        ...


class _IUsagePlanProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''A UsagePlan, either managed by this CDK app, or imported.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IUsagePlan"

    @builtins.property
    @jsii.member(jsii_name="usagePlanId")
    def usage_plan_id(self) -> builtins.str:
        '''Id of the usage plan.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "usagePlanId"))

    @jsii.member(jsii_name="addApiKey")
    def add_api_key(
        self,
        api_key: IApiKey,
        *,
        override_logical_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Adds an ApiKey.

        :param api_key: the api key to associate with this usage plan.
        :param override_logical_id: Override the CloudFormation logical id of the AWS::ApiGateway::UsagePlanKey resource. Default: - autogenerated by the CDK
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0581a1cd8ecb86e004a1d843a397a02b17cb9a9aeacc0117b49cae486cf553be)
            check_type(argname="argument api_key", value=api_key, expected_type=type_hints["api_key"])
        options = AddApiKeyOptions(override_logical_id=override_logical_id)

        return typing.cast(None, jsii.invoke(self, "addApiKey", [api_key, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IUsagePlan).__jsii_proxy_class__ = lambda : _IUsagePlanProxy


@jsii.interface(jsii_type="aws-cdk-lib.aws_apigateway.IVpcLink")
class IVpcLink(_IResource_c80c4260, typing_extensions.Protocol):
    '''Represents an API Gateway VpcLink.'''

    @builtins.property
    @jsii.member(jsii_name="vpcLinkId")
    def vpc_link_id(self) -> builtins.str:
        '''Physical ID of the VpcLink resource.

        :attribute: true
        '''
        ...


class _IVpcLinkProxy(
    jsii.proxy_for(_IResource_c80c4260), # type: ignore[misc]
):
    '''Represents an API Gateway VpcLink.'''

    __jsii_type__: typing.ClassVar[str] = "aws-cdk-lib.aws_apigateway.IVpcLink"

    @builtins.property
    @jsii.member(jsii_name="vpcLinkId")
    def vpc_link_id(self) -> builtins.str:
        '''Physical ID of the VpcLink resource.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "vpcLinkId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IVpcLink).__jsii_proxy_class__ = lambda : _IVpcLinkProxy


class IdentitySource(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.IdentitySource",
):
    '''Represents an identity source.

    The source can be specified either as a literal value (e.g: ``Auth``) which
    cannot be blank, or as an unresolved string token.

    :exampleMetadata: infused

    Example::

        # auth_fn: lambda.Function
        # books: apigateway.Resource
        
        
        auth = apigateway.RequestAuthorizer(self, "booksAuthorizer",
            handler=auth_fn,
            identity_sources=[apigateway.IdentitySource.header("Authorization")]
        )
        
        books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
            authorizer=auth
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="context")
    @builtins.classmethod
    def context(cls, context: builtins.str) -> builtins.str:
        '''Provides a properly formatted request context identity source.

        :param context: the name of the context variable the ``IdentitySource`` will represent.

        :return: a request context identity source.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1e4438df471dfbd44dedba16361d517c7a9667cad770e2dcc98a55e44de286a3)
            check_type(argname="argument context", value=context, expected_type=type_hints["context"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "context", [context]))

    @jsii.member(jsii_name="header")
    @builtins.classmethod
    def header(cls, header_name: builtins.str) -> builtins.str:
        '''Provides a properly formatted header identity source.

        :param header_name: the name of the header the ``IdentitySource`` will represent.

        :return: a header identity source.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d48a106fb6b9004e459d1353626d5db23d1d2eb61c3f6a0d880ee8f051311805)
            check_type(argname="argument header_name", value=header_name, expected_type=type_hints["header_name"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "header", [header_name]))

    @jsii.member(jsii_name="queryString")
    @builtins.classmethod
    def query_string(cls, query_string: builtins.str) -> builtins.str:
        '''Provides a properly formatted query string identity source.

        :param query_string: the name of the query string the ``IdentitySource`` will represent.

        :return: a query string identity source.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a179dad0c3d2ab8b2439d3be1e97cca6313b8319d658162bd4e1f86f9404fa2)
            check_type(argname="argument query_string", value=query_string, expected_type=type_hints["query_string"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "queryString", [query_string]))

    @jsii.member(jsii_name="stageVariable")
    @builtins.classmethod
    def stage_variable(cls, stage_variable: builtins.str) -> builtins.str:
        '''Provides a properly formatted API Gateway stage variable identity source.

        :param stage_variable: the name of the stage variable the ``IdentitySource`` will represent.

        :return: an API Gateway stage variable identity source.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c8f03a83b90713b9bf9975ba2d69451f891ad848c9040a23663c9667cea71520)
            check_type(argname="argument stage_variable", value=stage_variable, expected_type=type_hints["stage_variable"])
        return typing.cast(builtins.str, jsii.sinvoke(cls, "stageVariable", [stage_variable]))


class InlineApiDefinition(
    ApiDefinition,
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.InlineApiDefinition",
):
    '''OpenAPI specification from an inline JSON object.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from aws_cdk import aws_apigateway as apigateway
        
        # definition: Any
        
        inline_api_definition = apigateway.InlineApiDefinition(definition)
    '''

    def __init__(self, definition: typing.Any) -> None:
        '''
        :param definition: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a114919e366f4501628833707aa3cfe23a36055bc618f69c6a4d7b84a59fe97c)
            check_type(argname="argument definition", value=definition, expected_type=type_hints["definition"])
        jsii.create(self.__class__, self, [definition])

    @jsii.member(jsii_name="bind")
    def bind(self, _scope: _constructs_77d1e7e8.Construct) -> ApiDefinitionConfig:
        '''Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.

        :param _scope: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a6eac2ed74c360a3baed610dd89c5ccd9b5b2451bca09fe462592d4e1b55400)
            check_type(argname="argument _scope", value=_scope, expected_type=type_hints["_scope"])
        return typing.cast(ApiDefinitionConfig, jsii.invoke(self, "bind", [_scope]))


class Integration(
    metaclass=jsii.JSIIMeta,
    jsii_type="aws-cdk-lib.aws_apigateway.Integration",
):
    '''Base class for backend integrations for an API Gateway method.

    Use one of the concrete classes such as ``MockIntegration``, ``AwsIntegration``, ``LambdaIntegration``
    or implement on your own by specifying the set of props.

    :exampleMetadata: infused

    Example::

        # books: apigateway.Resource
        # iam_user: iam.User
        
        
        get_books = books.add_method("GET", apigateway.HttpIntegration("http://amazon.com"),
            authorization_type=apigateway.AuthorizationType.IAM
        )
        
        iam_user.attach_inline_policy(iam.Policy(self, "AllowBooks",
            statements=[
                iam.PolicyStatement(
                    actions=["execute-api:Invoke"],
                    effect=iam.Effect.ALLOW,
                    resources=[get_books.method_arn]
                )
            ]
        ))
    '''

    def __init__(
        self,
        *,
        type: "IntegrationType",
        integration_http_method: typing.Optional[builtins.str] = None,
        options: typing.Optional[typing.Union["IntegrationOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        uri: typing.Any = None,
    ) -> None:
        '''
        :param type: Specifies an API method integration type.
        :param integration_http_method: The integration's HTTP method type. Required unless you use a MOCK integration.
        :param options: Integration options.
        :param uri: The Uniform Resource Identifier (URI) for the integration. - If you specify HTTP for the ``type`` property, specify the API endpoint URL. - If you specify MOCK for the ``type`` property, don't specify this property. - If you specify AWS for the ``type`` property, specify an AWS service that follows this form: ``arn:partition:apigateway:region:subdomain.service|service:path|action/service_api.`` For example, a Lambda function URI follows this form: arn:partition:apigateway:region:lambda:path/path. The path is usually in the form /2015-03-31/functions/LambdaFunctionARN/invocations.
        '''
        props = IntegrationProps(
            type=type,
            integration_http_method=integration_http_method,
            options=options,
            uri=uri,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="bind")
    def bind(self, method: "Method") -> "IntegrationConfig":
        '''Can be overridden by subclasses to allow the integration to interact with the method being integrated, access the REST API object, method ARNs, etc.

        :param method: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f556228c2c8f47e496c5aaecf064cf11dfc1b04ae0a1b780af6769e5665391ee)
            check_type(argname="argument method", value=method, expected_type=type_hints["method"])
        return typing.cast("IntegrationConfig", jsii.invoke(self, "bind", [method]))


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.IntegrationConfig",
    jsii_struct_bases=[],
    name_mapping={
        "type": "type",
        "deployment_token": "deploymentToken",
        "integration_http_method": "integrationHttpMethod",
        "options": "options",
        "uri": "uri",
    },
)
class IntegrationConfig:
    def __init__(
        self,
        *,
        type: "IntegrationType",
        deployment_token: typing.Optional[builtins.str] = None,
        integration_http_method: typing.Optional[builtins.str] = None,
        options: typing.Optional[typing.Union["IntegrationOptions", typing.Dict[builtins.str, typing.Any]]] = None,
        uri: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Result of binding an Integration to a Method.

        :param type: Specifies an API method integration type.
        :param deployment_token: This value is included in computing the Deployment's fingerprint. When the fingerprint changes, a new deployment is triggered. This property should contain values associated with the Integration that upon changing should trigger a fresh the Deployment needs to be refreshed. Default: undefined deployments are not triggered for any change to this integration.
        :param integration_http_method: The integration's HTTP method type. Required unless you use a MOCK integration. Default: - no integration method specified.
        :param options: Integration options. Default: - no integration options
        :param uri: The Uniform Resource Identifier (URI) for the integration. Default: - no URI. Usually applies to MOCK integration

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_apigateway as apigateway
            from aws_cdk import aws_iam as iam
            
            # role: iam.Role
            # vpc_link: apigateway.VpcLink
            
            integration_config = apigateway.IntegrationConfig(
                type=apigateway.IntegrationType.AWS,
            
                # the properties below are optional
                deployment_token="deploymentToken",
                integration_http_method="integrationHttpMethod",
                options=apigateway.IntegrationOptions(
                    cache_key_parameters=["cacheKeyParameters"],
                    cache_namespace="cacheNamespace",
                    connection_type=apigateway.ConnectionType.INTERNET,
                    content_handling=apigateway.ContentHandling.CONVERT_TO_BINARY,
                    credentials_passthrough=False,
                    credentials_role=role,
                    integration_responses=[apigateway.IntegrationResponse(
                        status_code="statusCode",
            
                        # the properties below are optional
                        content_handling=apigateway.ContentHandling.CONVERT_TO_BINARY,
                        response_parameters={
                            "response_parameters_key": "responseParameters"
                        },
                        response_templates={
                            "response_templates_key": "responseTemplates"
                        },
                        selection_pattern="selectionPattern"
                    )],
                    passthrough_behavior=apigateway.PassthroughBehavior.WHEN_NO_MATCH,
                    request_parameters={
                        "request_parameters_key": "requestParameters"
                    },
                    request_templates={
                        "request_templates_key": "requestTemplates"
                    },
                    timeout=cdk.Duration.minutes(30),
                    vpc_link=vpc_link
                ),
                uri="uri"
            )
        '''
        if isinstance(options, dict):
            options = IntegrationOptions(**options)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f681fc0e188b8714d6559eeec81787ef2d99aad8145d8e311ddfa4f90c8c2692)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument deployment_token", value=deployment_token, expected_type=type_hints["deployment_token"])
            check_type(argname="argument integration_http_method", value=integration_http_method, expected_type=type_hints["integration_http_method"])
            check_type(argname="argument options", value=options, expected_type=type_hints["options"])
            check_type(argname="argument uri", value=uri, expected_type=type_hints["uri"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }
        if deployment_token is not None:
            self._values["deployment_token"] = deployment_token
        if integration_http_method is not None:
            self._values["integration_http_method"] = integration_http_method
        if options is not None:
            self._values["options"] = options
        if uri is not None:
            self._values["uri"] = uri

    @builtins.property
    def type(self) -> "IntegrationType":
        '''Specifies an API method integration type.'''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("IntegrationType", result)

    @builtins.property
    def deployment_token(self) -> typing.Optional[builtins.str]:
        '''This value is included in computing the Deployment's fingerprint.

        When the fingerprint
        changes, a new deployment is triggered.
        This property should contain values associated with the Integration that upon changing
        should trigger a fresh the Deployment needs to be refreshed.

        :default: undefined deployments are not triggered for any change to this integration.
        '''
        result = self._values.get("deployment_token")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_http_method(self) -> typing.Optional[builtins.str]:
        '''The integration's HTTP method type.

        Required unless you use a MOCK integration.

        :default: - no integration method specified.
        '''
        result = self._values.get("integration_http_method")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def options(self) -> typing.Optional["IntegrationOptions"]:
        '''Integration options.

        :default: - no integration options
        '''
        result = self._values.get("options")
        return typing.cast(typing.Optional["IntegrationOptions"], result)

    @builtins.property
    def uri(self) -> typing.Optional[builtins.str]:
        '''The Uniform Resource Identifier (URI) for the integration.

        :default: - no URI. Usually applies to MOCK integration

        :see: https://docs.aws.amazon.com/apigateway/api-reference/resource/integration/#uri
        '''
        result = self._values.get("uri")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "IntegrationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.IntegrationOptions",
    jsii_struct_bases=[],
    name_mapping={
        "cache_key_parameters": "cacheKeyParameters",
        "cache_namespace": "cacheNamespace",
        "connection_type": "connectionType",
        "content_handling": "contentHandling",
        "credentials_passthrough": "credentialsPassthrough",
        "credentials_role": "credentialsRole",
        "integration_responses": "integrationResponses",
        "passthrough_behavior": "passthroughBehavior",
        "request_parameters": "requestParameters",
        "request_templates": "requestTemplates",
        "timeout": "timeout",
        "vpc_link": "vpcLink",
    },
)
class IntegrationOptions:
    def __init__(
        self,
        *,
        cache_key_parameters: typing.Optional[typing.Sequence[builtins.str]] = None,
        cache_namespace: typing.Optional[builtins.str] = None,
        connection_type: typing.Optional[ConnectionType] = None,
        content_handling: typing.Optional[ContentHandling] = None,
        credentials_passthrough: typing.Optional[builtins.bool] = None,
        credentials_role: typing.Optional[_IRole_235f5d8e] = None,
        integration_responses: typing.Optional[typing.Sequence[typing.Union["IntegrationResponse", typing.Dict[builtins.str, typing.Any]]]] = None,
        passthrough_behavior: typing.Optional["PassthroughBehavior"] = None,
        request_parameters: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        request_templates: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        vpc_link: typing.Optional[IVpcLink] = None,
    ) -> None:
        '''
        :param cache_key_parameters: A list of request parameters whose values are to be cached. It determines request parameters that will make it into the cache key.
        :param cache_namespace: An API-specific tag group of related cached parameters.
        :param connection_type: The type of network connection to the integration endpoint. Default: - ConnectionType.VPC_LINK if ``vpcLink`` property is configured; ConnectionType.Internet otherwise.
        :param content_handling: Specifies how to handle request payload content type conversions. Default: none if this property isn't defined, the request payload is passed through from the method request to the integration request without modification, provided that the ``passthroughBehaviors`` property is configured to support payload pass-through.
        :param credentials_passthrough: Requires that the caller's identity be passed through from the request. Default: Caller identity is not passed through
        :param credentials_role: An IAM role that API Gateway assumes. Mutually exclusive with ``credentialsPassThrough``. Default: A role is not assumed
        :param integration_responses: The response that API Gateway provides after a method's backend completes processing a request. API Gateway intercepts the response from the backend so that you can control how API Gateway surfaces backend responses. For example, you can map the backend status codes to codes that you define.
        :param passthrough_behavior: Specifies the pass-through behavior for incoming requests based on the Content-Type header in the request, and the available mapping templates specified as the requestTemplates property on the Integration resource. There are three valid values: WHEN_NO_MATCH, WHEN_NO_TEMPLATES, and NEVER.
        :param request_parameters: The request parameters that API Gateway sends with the backend request. Specify request parameters as key-value pairs (string-to-string mappings), with a destination as the key and a source as the value. Specify the destination by using the following pattern integration.request.location.name, where location is querystring, path, or header, and name is a valid, unique parameter name. The source must be an existing method request parameter or a static value. You must enclose static values in single quotation marks and pre-encode these values based on their destination in the request.
        :param request_templates: A map of Apache Velocity templates that are applied on the request payload. The template that API Gateway uses is based on the value of the Content-Type header that's sent by the client. The content type value is the key, and the template is the value (specified as a string), such as the following snippet:: { "application/json": "{ \\"statusCode\\": 200 }" }
        :param timeout: The maximum amount of time an integration will run before it returns without a response. By default, the value must be between 50 milliseconds and 29 seconds. The upper bound can be increased for regional and private Rest APIs only, via a quota increase request for your acccount. This increase might require a reduction in your account-level throttle quota limit. See {@link https://docs.aws.amazon.com/apigateway/latest/developerguide/limits.html Amazon API Gateway quotas} for more details. Default: Duration.seconds(29)
        :param vpc_link: The VpcLink used for the integration. Required if connectionType is VPC_LINK

        :exampleMetadata: lit=aws-apigateway/test/authorizers/integ.request-authorizer.lit.ts infused

        Example::

            from aws_cdk.aws_apigateway import IntegrationResponse, MethodResponse, IntegrationResponse, MethodResponse
            import path as path
            import aws_cdk.aws_lambda as lambda_
            from aws_cdk import App, Stack
            from aws_cdk.aws_apigateway import MockIntegration, PassthroughBehavior, RestApi, RequestAuthorizer, IdentitySource
            
            # Against the RestApi endpoint from the stack output, run
            # `curl -s -o /dev/null -w "%{http_code}" <url>` should return 401
            # `curl -s -o /dev/null -w "%{http_code}" -H 'Authorization: deny' <url>?allow=yes` should return 403
            # `curl -s -o /dev/null -w "%{http_code}" -H 'Authorization: allow' <url>?allow=yes` should return 200
            
            app = App()
            stack = Stack(app, "RequestAuthorizerInteg")
            
            authorizer_fn = lambda_.Function(stack, "MyAuthorizerFunction",
                runtime=lambda_.Runtime.NODEJS_LATEST,
                handler="index.handler",
                code=lambda_.AssetCode.from_asset(path.join(__dirname, "integ.request-authorizer.handler"))
            )
            
            restapi = RestApi(stack, "MyRestApi", cloud_watch_role=True)
            
            authorizer = RequestAuthorizer(stack, "MyAuthorizer",
                handler=authorizer_fn,
                identity_sources=[IdentitySource.header("Authorization"), IdentitySource.query_string("allow")]
            )
            
            second_authorizer = RequestAuthorizer(stack, "MySecondAuthorizer",
                handler=authorizer_fn,
                identity_sources=[IdentitySource.header("Authorization"), IdentitySource.query_string("allow")]
            )
            
            restapi.root.add_method("ANY", MockIntegration(
                integration_responses=[IntegrationResponse(status_code="200")
                ],
                passthrough_behavior=PassthroughBehavior.NEVER,
                request_templates={
                    "application/json": "{ \"statusCode\": 200 }"
                }
            ),
                method_responses=[MethodResponse(status_code="200")
                ],
                authorizer=authorizer
            )
            
            restapi.root.resource_for_path("auth").add_method("ANY", MockIntegration(
                integration_responses=[IntegrationResponse(status_code="200")
                ],
                passthrough_behavior=PassthroughBehavior.NEVER,
                request_templates={
                    "application/json": "{ \"statusCode\": 200 }"
                }
            ),
                method_responses=[MethodResponse(status_code="200")
                ],
                authorizer=second_authorizer
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2d8fe5f6bdc4536c27b3cea965d20260495a3bdffbb6422a7b5ce729924b6434)
            check_type(argname="argument cache_key_parameters", value=cache_key_parameters, expected_type=type_hints["cache_key_parameters"])
            check_type(argname="argument cache_namespace", value=cache_namespace, expected_type=type_hints["cache_namespace"])
            check_type(argname="argument connection_type", value=connection_type, expected_type=type_hints["connection_type"])
            check_type(argname="argument content_handling", value=content_handling, expected_type=type_hints["content_handling"])
            check_type(argname="argument credentials_passthrough", value=credentials_passthrough, expected_type=type_hints["credentials_passthrough"])
            check_type(argname="argument credentials_role", value=credentials_role, expected_type=type_hints["credentials_role"])
            check_type(argname="argument integration_responses", value=integration_responses, expected_type=type_hints["integration_responses"])
            check_type(argname="argument passthrough_behavior", value=passthrough_behavior, expected_type=type_hints["passthrough_behavior"])
            check_type(argname="argument request_parameters", value=request_parameters, expected_type=type_hints["request_parameters"])
            check_type(argname="argument request_templates", value=request_templates, expected_type=type_hints["request_templates"])
            check_type(argname="argument timeout", value=timeout, expected_type=type_hints["timeout"])
            check_type(argname="argument vpc_link", value=vpc_link, expected_type=type_hints["vpc_link"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cache_key_parameters is not None:
            self._values["cache_key_parameters"] = cache_key_parameters
        if cache_namespace is not None:
            self._values["cache_namespace"] = cache_namespace
        if connection_type is not None:
            self._values["connection_type"] = connection_type
        if content_handling is not None:
            self._values["content_handling"] = content_handling
        if credentials_passthrough is not None:
            self._values["credentials_passthrough"] = credentials_passthrough
        if credentials_role is not None:
            self._values["credentials_role"] = credentials_role
        if integration_responses is not None:
            self._values["integration_responses"] = integration_responses
        if passthrough_behavior is not None:
            self._values["passthrough_behavior"] = passthrough_behavior
        if request_parameters is not None:
            self._values["request_parameters"] = request_parameters
        if request_templates is not None:
            self._values["request_templates"] = request_templates
        if timeout is not None:
            self._values["timeout"] = timeout
        if vpc_link is not None:
            self._values["vpc_link"] = vpc_link

    @builtins.property
    def cache_key_parameters(self) -> typing.Optional[typing.List[builtins.str]]:
        '''A list of request parameters whose values are to be cached.

        It determines
        request parameters that will make it into the cache key.
        '''
        result = self._values.get("cache_key_parameters")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def cache_namespace(self) -> typing.Optional[builtins.str]:
        '''An API-specific tag group of related cached parameters.'''
        result = self._values.get("cache_namespace")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def connection_type(self) -> typing.Optional[ConnectionType]:
        '''The type of network connection to the integration endpoint.

        :default: - ConnectionType.VPC_LINK if ``vpcLink`` property is configured; ConnectionType.Internet otherwise.
        '''
        result = self._values.get("connection_type")
        return typing.cast(typing.Optional[ConnectionType], result)

    @builtins.property
    def content_handling(self) -> typing.Optional[ContentHandling]:
        '''Specifies how to handle request payload content type conversions.

        :default:

        none if this property isn't defined, the request payload is passed
        through from the method request to the integration request without
        modification, provided that the ``passthroughBehaviors`` property is
        configured to support payload pass-through.
        '''
        result = self._values.get("content_handling")
        return typing.cast(typing.Optional[ContentHandling], result)

    @builtins.property
    def credentials_passthrough(self) -> typing.Optional[builtins.bool]:
        '''Requires that the caller's identity be passed through from the request.

        :default: Caller identity is not passed through
        '''
        result = self._values.get("credentials_passthrough")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def credentials_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''An IAM role that API Gateway assumes.

        Mutually exclusive with ``credentialsPassThrough``.

        :default: A role is not assumed
        '''
        result = self._values.get("credentials_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def integration_responses(
        self,
    ) -> typing.Optional[typing.List["IntegrationResponse"]]:
        '''The response that API Gateway provides after a method's backend completes processing a request.

        API Gateway intercepts the response from the
        backend so that you can control how API Gateway surfaces backend
        responses. For example, you can map the backend status codes to codes
        that you define.
        '''
        result = self._values.get("integration_responses")
        return typing.cast(typing.Optional[typing.List["IntegrationResponse"]], result)

    @builtins.property
    def passthrough_behavior(self) -> typing.Optional["PassthroughBehavior"]:
        '''Specifies the pass-through behavior for incoming requests based on the Content-Type header in the request, and the available mapping templates specified as the requestTemplates property on the Integration resource.

        There are three valid values: WHEN_NO_MATCH, WHEN_NO_TEMPLATES, and
        NEVER.
        '''
        result = self._values.get("passthrough_behavior")
        return typing.cast(typing.Optional["PassthroughBehavior"], result)

    @builtins.property
    def request_parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The request parameters that API Gateway sends with the backend request.

        Specify request parameters as key-value pairs (string-to-string
        mappings), with a destination as the key and a source as the value.

        Specify the destination by using the following pattern
        integration.request.location.name, where location is querystring, path,
        or header, and name is a valid, unique parameter name.

        The source must be an existing method request parameter or a static
        value. You must enclose static values in single quotation marks and
        pre-encode these values based on their destination in the request.
        '''
        result = self._values.get("request_parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def request_templates(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''A map of Apache Velocity templates that are applied on the request payload.

        The template that API Gateway uses is based on the value of the
        Content-Type header that's sent by the client. The content type value is
        the key, and the template is the value (specified as a string), such as
        the following snippet::

             { "application/json": "{ \\"statusCode\\": 200 }" }

        :see: http://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
        '''
        result = self._values.get("request_templates")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_4839e8c3]:
        '''The maximum amount of time an integration will run before it returns without a response.

        By default, the value must be between 50 milliseconds and 29 seconds.
        The upper bound can be increased for regional and private Rest APIs only,
        via a quota increase request for your acccount.
        This increase might require a reduction in your account-level throttle quota limit.

        See {@link https://docs.aws.amazon.com/apigateway/latest/developerguide/limits.html Amazon API Gateway quotas} for more details.

        :default: Duration.seconds(29)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    @builtins.property
    def vpc_link(self) -> typing.Optional[IVpcLink]:
        '''The VpcLink used for the integration.

        Required if connectionType is VPC_LINK
        '''
        result = self._values.get("vpc_link")
        return typing.cast(typing.Optional[IVpcLink], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "IntegrationOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.IntegrationProps",
    jsii_struct_bases=[],
    name_mapping={
        "type": "type",
        "integration_http_method": "integrationHttpMethod",
        "options": "options",
        "uri": "uri",
    },
)
class IntegrationProps:
    def __init__(
        self,
        *,
        type: "IntegrationType",
        integration_http_method: typing.Optional[builtins.str] = None,
        options: typing.Optional[typing.Union[IntegrationOptions, typing.Dict[builtins.str, typing.Any]]] = None,
        uri: typing.Any = None,
    ) -> None:
        '''
        :param type: Specifies an API method integration type.
        :param integration_http_method: The integration's HTTP method type. Required unless you use a MOCK integration.
        :param options: Integration options.
        :param uri: The Uniform Resource Identifier (URI) for the integration. - If you specify HTTP for the ``type`` property, specify the API endpoint URL. - If you specify MOCK for the ``type`` property, don't specify this property. - If you specify AWS for the ``type`` property, specify an AWS service that follows this form: ``arn:partition:apigateway:region:subdomain.service|service:path|action/service_api.`` For example, a Lambda function URI follows this form: arn:partition:apigateway:region:lambda:path/path. The path is usually in the form /2015-03-31/functions/LambdaFunctionARN/invocations.

        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_elasticloadbalancingv2 as elbv2
            
            
            vpc = ec2.Vpc(self, "VPC")
            nlb = elbv2.NetworkLoadBalancer(self, "NLB",
                vpc=vpc
            )
            link = apigateway.VpcLink(self, "link",
                targets=[nlb]
            )
            
            integration = apigateway.Integration(
                type=apigateway.IntegrationType.HTTP_PROXY,
                integration_http_method="ANY",
                options=apigateway.IntegrationOptions(
                    connection_type=apigateway.ConnectionType.VPC_LINK,
                    vpc_link=link
                )
            )
        '''
        if isinstance(options, dict):
            options = IntegrationOptions(**options)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d094c7b740fc0168dee3949ca42f2b0e59bf60093b63a3b27e7f08b4a2282b31)
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument integration_http_method", value=integration_http_method, expected_type=type_hints["integration_http_method"])
            check_type(argname="argument options", value=options, expected_type=type_hints["options"])
            check_type(argname="argument uri", value=uri, expected_type=type_hints["uri"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "type": type,
        }
        if integration_http_method is not None:
            self._values["integration_http_method"] = integration_http_method
        if options is not None:
            self._values["options"] = options
        if uri is not None:
            self._values["uri"] = uri

    @builtins.property
    def type(self) -> "IntegrationType":
        '''Specifies an API method integration type.'''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("IntegrationType", result)

    @builtins.property
    def integration_http_method(self) -> typing.Optional[builtins.str]:
        '''The integration's HTTP method type.

        Required unless you use a MOCK integration.
        '''
        result = self._values.get("integration_http_method")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def options(self) -> typing.Optional[IntegrationOptions]:
        '''Integration options.'''
        result = self._values.get("options")
        return typing.cast(typing.Optional[IntegrationOptions], result)

    @builtins.property
    def uri(self) -> typing.Any:
        '''The Uniform Resource Identifier (URI) for the integration.

        - If you specify HTTP for the ``type`` property, specify the API endpoint URL.
        - If you specify MOCK for the ``type`` property, don't specify this property.
        - If you specify AWS for the ``type`` property, specify an AWS service that
          follows this form: ``arn:partition:apigateway:region:subdomain.service|service:path|action/service_api.``
          For example, a Lambda function URI follows this form:
          arn:partition:apigateway:region:lambda:path/path. The path is usually in the
          form /2015-03-31/functions/LambdaFunctionARN/invocations.

        :see: https://docs.aws.amazon.com/apigateway/api-reference/resource/integration/#uri
        '''
        result = self._values.get("uri")
        return typing.cast(typing.Any, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "IntegrationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.IntegrationResponse",
    jsii_struct_bases=[],
    name_mapping={
        "status_code": "statusCode",
        "content_handling": "contentHandling",
        "response_parameters": "responseParameters",
        "response_templates": "responseTemplates",
        "selection_pattern": "selectionPattern",
    },
)
class IntegrationResponse:
    def __init__(
        self,
        *,
        status_code: builtins.str,
        content_handling: typing.Optional[ContentHandling] = None,
        response_parameters: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        response_templates: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        selection_pattern: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param status_code: The status code that API Gateway uses to map the integration response to a MethodResponse status code.
        :param content_handling: Specifies how to handle request payload content type conversions. Default: none the request payload is passed through from the method request to the integration request without modification.
        :param response_parameters: The response parameters from the backend response that API Gateway sends to the method response. Use the destination as the key and the source as the value: - The destination must be an existing response parameter in the MethodResponse property. - The source must be an existing method request parameter or a static value. You must enclose static values in single quotation marks and pre-encode these values based on the destination specified in the request.
        :param response_templates: The templates that are used to transform the integration response body. Specify templates as key-value pairs, with a content type as the key and a template as the value.
        :param selection_pattern: Specifies the regular expression (regex) pattern used to choose an integration response based on the response from the back end. For example, if the success response returns nothing and the error response returns some string, you could use the ``.+`` regex to match error response. However, make sure that the error response does not contain any newline (``\\n``) character in such cases. If the back end is an AWS Lambda function, the AWS Lambda function error header is matched. For all other HTTP and AWS back ends, the HTTP status code is matched.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from aws_cdk import aws_apigateway as apigateway
            
            integration_response = apigateway.IntegrationResponse(
                status_code="statusCode",
            
                # the properties below are optional
                content_handling=apigateway.ContentHandling.CONVERT_TO_BINARY,
                response_parameters={
                    "response_parameters_key": "responseParameters"
                },
                response_templates={
                    "response_templates_key": "responseTemplates"
                },
                selection_pattern="selectionPattern"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f6cb930af96a75dfa56cd51da050efc5f3c79f1759af7579e8e6394502a4f62e)
            check_type(argname="argument status_code", value=status_code, expected_type=type_hints["status_code"])
            check_type(argname="argument content_handling", value=content_handling, expected_type=type_hints["content_handling"])
            check_type(argname="argument response_parameters", value=response_parameters, expected_type=type_hints["response_parameters"])
            check_type(argname="argument response_templates", value=response_templates, expected_type=type_hints["response_templates"])
            check_type(argname="argument selection_pattern", value=selection_pattern, expected_type=type_hints["selection_pattern"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "status_code": status_code,
        }
        if content_handling is not None:
            self._values["content_handling"] = content_handling
        if response_parameters is not None:
            self._values["response_parameters"] = response_parameters
        if response_templates is not None:
            self._values["response_templates"] = response_templates
        if selection_pattern is not None:
            self._values["selection_pattern"] = selection_pattern

    @builtins.property
    def status_code(self) -> builtins.str:
        '''The status code that API Gateway uses to map the integration response to a MethodResponse status code.'''
        result = self._values.get("status_code")
        assert result is not None, "Required property 'status_code' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def content_handling(self) -> typing.Optional[ContentHandling]:
        '''Specifies how to handle request payload content type conversions.

        :default:

        none the request payload is passed through from the method
        request to the integration request without modification.
        '''
        result = self._values.get("content_handling")
        return typing.cast(typing.Optional[ContentHandling], result)

    @builtins.property
    def response_parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The response parameters from the backend response that API Gateway sends to the method response.

        Use the destination as the key and the source as the value:

        - The destination must be an existing response parameter in the
          MethodResponse property.
        - The source must be an existing method request parameter or a static
          value. You must enclose static values in single quotation marks and
          pre-encode these values based on the destination specified in the
          request.

        :see: http://docs.aws.amazon.com/apigateway/latest/developerguide/request-response-data-mappings.html
        '''
        result = self._values.get("response_parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def response_templates(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''The templates that are used to transform the integration response body.

        Specify templates as key-value pairs, with a content type as the key and
        a template as the value.

        :see: http://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html
        '''
        result = self._values.get("response_templates")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def selection_pattern(self) -> typing.Optional[builtins.str]:
        '''Specifies the regular expression (regex) pattern used to choose an integration response based on the response from the back end.

        For example, if the success response returns nothing and the error response returns some string, you
        could use the ``.+`` regex to match error response. However, make sure that the error response does not contain any
        newline (``\\n``) character in such cases. If the back end is an AWS Lambda function, the AWS Lambda function error
        header is matched. For all other HTTP and AWS back ends, the HTTP status code is matched.

        :see: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-integration-settings-integration-response.html
        '''
        result = self._values.get("selection_pattern")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "IntegrationResponse(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_apigateway.IntegrationType")
class IntegrationType(enum.Enum):
    '''
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_elasticloadbalancingv2 as elbv2
        
        
        vpc = ec2.Vpc(self, "VPC")
        nlb = elbv2.NetworkLoadBalancer(self, "NLB",
            vpc=vpc
        )
        link = apigateway.VpcLink(self, "link",
            targets=[nlb]
        )
        
        integration = apigateway.Integration(
            type=apigateway.IntegrationType.HTTP_PROXY,
            integration_http_method="ANY",
            options=apigateway.IntegrationOptions(
                connection_type=apigateway.ConnectionType.VPC_LINK,
                vpc_link=link
            )
        )
    '''

    AWS = "AWS"
    '''For integrating the API method request with an AWS service action, including the Lambda function-invoking action.

    With the Lambda
    function-invoking action, this is referred to as the Lambda custom
    integration. With any other AWS service action, this is known as AWS
    integration.
    '''
    AWS_PROXY = "AWS_PROXY"
    '''For integrating the API method request with the Lambda function-invoking action with the client request passed through as-is.

    This integration is
    also referred to as the Lambda proxy integration
    '''
    HTTP = "HTTP"
    '''For integrating the API method request with an HTTP endpoint, including a private HTTP endpoint within a VPC.

    This integration is also referred to
    as the HTTP custom integration.
    '''
    HTTP_PROXY = "HTTP_PROXY"
    '''For integrating the API method request with an HTTP endpoint, including a private HTTP endpoint within a VPC, with the client request passed through as-is.

    This is also referred to as the HTTP proxy integration
    '''
    MOCK = "MOCK"
    '''For integrating the API method request with API Gateway as a "loop-back" endpoint without invoking any backend.'''


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.JsonSchema",
    jsii_struct_bases=[],
    name_mapping={
        "additional_items": "additionalItems",
        "additional_properties": "additionalProperties",
        "all_of": "allOf",
        "any_of": "anyOf",
        "contains": "contains",
        "default": "default",
        "definitions": "definitions",
        "dependencies": "dependencies",
        "description": "description",
        "enum": "enum",
        "exclusive_maximum": "exclusiveMaximum",
        "exclusive_minimum": "exclusiveMinimum",
        "format": "format",
        "id": "id",
        "items": "items",
        "maximum": "maximum",
        "max_items": "maxItems",
        "max_length": "maxLength",
        "max_properties": "maxProperties",
        "minimum": "minimum",
        "min_items": "minItems",
        "min_length": "minLength",
        "min_properties": "minProperties",
        "multiple_of": "multipleOf",
        "not_": "not",
        "one_of": "oneOf",
        "pattern": "pattern",
        "pattern_properties": "patternProperties",
        "properties": "properties",
        "property_names": "propertyNames",
        "ref": "ref",
        "required": "required",
        "schema": "schema",
        "title": "title",
        "type": "type",
        "unique_items": "uniqueItems",
    },
)
class JsonSchema:
    def __init__(
        self,
        *,
        additional_items: typing.Optional[typing.Sequence[typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]]] = None,
        additional_properties: typing.Optional[typing.Union[builtins.bool, typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]]] = None,
        all_of: typing.Optional[typing.Sequence[typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]]] = None,
        any_of: typing.Optional[typing.Sequence[typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]]] = None,
        contains: typing.Optional[typing.Union[typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]], typing.Sequence[typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]]]] = None,
        default: typing.Any = None,
        definitions: typing.Optional[typing.Mapping[builtins.str, typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]]] = None,
        dependencies: typing.Optional[typing.Mapping[builtins.str, typing.Union[typing.Sequence[builtins.str], typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]]]] = None,
        description: typing.Optional[builtins.str] = None,
        enum: typing.Optional[typing.Sequence[typing.Any]] = None,
        exclusive_maximum: typing.Optional[builtins.bool] = None,
        exclusive_minimum: typing.Optional[builtins.bool] = None,
        format: typing.Optional[builtins.str] = None,
        id: typing.Optional[builtins.str] = None,
        items: typing.Optional[typing.Union[typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]], typing.Sequence[typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]]]] = None,
        maximum: typing.Optional[jsii.Number] = None,
        max_items: typing.Optional[jsii.Number] = None,
        max_length: typing.Optional[jsii.Number] = None,
        max_properties: typing.Optional[jsii.Number] = None,
        minimum: typing.Optional[jsii.Number] = None,
        min_items: typing.Optional[jsii.Number] = None,
        min_length: typing.Optional[jsii.Number] = None,
        min_properties: typing.Optional[jsii.Number] = None,
        multiple_of: typing.Optional[jsii.Number] = None,
        not_: typing.Optional[typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]] = None,
        one_of: typing.Optional[typing.Sequence[typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]]] = None,
        pattern: typing.Optional[builtins.str] = None,
        pattern_properties: typing.Optional[typing.Mapping[builtins.str, typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]]] = None,
        properties: typing.Optional[typing.Mapping[builtins.str, typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]]] = None,
        property_names: typing.Optional[typing.Union["JsonSchema", typing.Dict[builtins.str, typing.Any]]] = None,
        ref: typing.Optional[builtins.str] = None,
        required: typing.Optional[typing.Sequence[builtins.str]] = None,
        schema: typing.Optional["JsonSchemaVersion"] = None,
        title: typing.Optional[builtins.str] = None,
        type: typing.Optional[typing.Union["JsonSchemaType", typing.Sequence["JsonSchemaType"]]] = None,
        unique_items: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Represents a JSON schema definition of the structure of a REST API model.

        Copied from npm module jsonschema.

        :param additional_items: 
        :param additional_properties: 
        :param all_of: 
        :param any_of: 
        :param contains: 
        :param default: The default value if you use an enum. Default: - not set
        :param definitions: 
        :param dependencies: 
        :param description: 
        :param enum: 
        :param exclusive_maximum: 
        :param exclusive_minimum: 
        :param format: 
        :param id: 
        :param items: 
        :param maximum: 
        :param max_items: 
        :param max_length: 
        :param max_properties: 
        :param minimum: 
        :param min_items: 
        :param min_length: 
        :param min_properties: 
        :param multiple_of: 
        :param not_: 
        :param one_of: 
        :param pattern: 
        :param pattern_properties: 
        :param properties: 
        :param property_names: 
        :param ref: 
        :param required: 
        :param schema: 
        :param title: 
        :param type: 
        :param unique_items: 

        :see: https://github.com/tdegrunt/jsonschema
        :exampleMetadata: infused

        Example::

            # api: apigateway.RestApi
            
            
            # We define the JSON Schema for the transformed valid response
            response_model = api.add_model("ResponseModel",
                content_type="application/json",
                model_name="ResponseModel",
                schema=apigateway.JsonSchema(
                    schema=apigateway.JsonSchemaVersion.DRAFT4,
                    title="pollResponse",
                    type=apigateway.JsonSchemaType.OBJECT,
                    properties={
                        "state": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING),
                        "greeting": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING)
                    }
                )
            )
            
            # We define the JSON Schema for the transformed error response
            error_response_model = api.add_model("ErrorResponseModel",
                content_type="application/json",
                model_name="ErrorResponseModel",
                schema=apigateway.JsonSchema(
                    schema=apigateway.JsonSchemaVersion.DRAFT4,
                    title="errorResponse",
                    type=apigateway.JsonSchemaType.OBJECT,
                    properties={
                        "state": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING),
                        "message": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING)
                    }
                )
            )
        '''
        if isinstance(not_, dict):
            not_ = JsonSchema(**not_)
        if isinstance(property_names, dict):
            property_names = JsonSchema(**property_names)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__689c3507e908daefb625f42f5bfc2c215e135591704691d515c49669d5c3dbc3)
            check_type(argname="argument additional_items", value=additional_items, expected_type=type_hints["additional_items"])
            check_type(argname="argument additional_properties", value=additional_properties, expected_type=type_hints["additional_properties"])
            check_type(argname="argument all_of", value=all_of, expected_type=type_hints["all_of"])
            check_type(argname="argument any_of", value=any_of, expected_type=type_hints["any_of"])
            check_type(argname="argument contains", value=contains, expected_type=type_hints["contains"])
            check_type(argname="argument default", value=default, expected_type=type_hints["default"])
            check_type(argname="argument definitions", value=definitions, expected_type=type_hints["definitions"])
            check_type(argname="argument dependencies", value=dependencies, expected_type=type_hints["dependencies"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument enum", value=enum, expected_type=type_hints["enum"])
            check_type(argname="argument exclusive_maximum", value=exclusive_maximum, expected_type=type_hints["exclusive_maximum"])
            check_type(argname="argument exclusive_minimum", value=exclusive_minimum, expected_type=type_hints["exclusive_minimum"])
            check_type(argname="argument format", value=format, expected_type=type_hints["format"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument items", value=items, expected_type=type_hints["items"])
            check_type(argname="argument maximum", value=maximum, expected_type=type_hints["maximum"])
            check_type(argname="argument max_items", value=max_items, expected_type=type_hints["max_items"])
            check_type(argname="argument max_length", value=max_length, expected_type=type_hints["max_length"])
            check_type(argname="argument max_properties", value=max_properties, expected_type=type_hints["max_properties"])
            check_type(argname="argument minimum", value=minimum, expected_type=type_hints["minimum"])
            check_type(argname="argument min_items", value=min_items, expected_type=type_hints["min_items"])
            check_type(argname="argument min_length", value=min_length, expected_type=type_hints["min_length"])
            check_type(argname="argument min_properties", value=min_properties, expected_type=type_hints["min_properties"])
            check_type(argname="argument multiple_of", value=multiple_of, expected_type=type_hints["multiple_of"])
            check_type(argname="argument not_", value=not_, expected_type=type_hints["not_"])
            check_type(argname="argument one_of", value=one_of, expected_type=type_hints["one_of"])
            check_type(argname="argument pattern", value=pattern, expected_type=type_hints["pattern"])
            check_type(argname="argument pattern_properties", value=pattern_properties, expected_type=type_hints["pattern_properties"])
            check_type(argname="argument properties", value=properties, expected_type=type_hints["properties"])
            check_type(argname="argument property_names", value=property_names, expected_type=type_hints["property_names"])
            check_type(argname="argument ref", value=ref, expected_type=type_hints["ref"])
            check_type(argname="argument required", value=required, expected_type=type_hints["required"])
            check_type(argname="argument schema", value=schema, expected_type=type_hints["schema"])
            check_type(argname="argument title", value=title, expected_type=type_hints["title"])
            check_type(argname="argument type", value=type, expected_type=type_hints["type"])
            check_type(argname="argument unique_items", value=unique_items, expected_type=type_hints["unique_items"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if additional_items is not None:
            self._values["additional_items"] = additional_items
        if additional_properties is not None:
            self._values["additional_properties"] = additional_properties
        if all_of is not None:
            self._values["all_of"] = all_of
        if any_of is not None:
            self._values["any_of"] = any_of
        if contains is not None:
            self._values["contains"] = contains
        if default is not None:
            self._values["default"] = default
        if definitions is not None:
            self._values["definitions"] = definitions
        if dependencies is not None:
            self._values["dependencies"] = dependencies
        if description is not None:
            self._values["description"] = description
        if enum is not None:
            self._values["enum"] = enum
        if exclusive_maximum is not None:
            self._values["exclusive_maximum"] = exclusive_maximum
        if exclusive_minimum is not None:
            self._values["exclusive_minimum"] = exclusive_minimum
        if format is not None:
            self._values["format"] = format
        if id is not None:
            self._values["id"] = id
        if items is not None:
            self._values["items"] = items
        if maximum is not None:
            self._values["maximum"] = maximum
        if max_items is not None:
            self._values["max_items"] = max_items
        if max_length is not None:
            self._values["max_length"] = max_length
        if max_properties is not None:
            self._values["max_properties"] = max_properties
        if minimum is not None:
            self._values["minimum"] = minimum
        if min_items is not None:
            self._values["min_items"] = min_items
        if min_length is not None:
            self._values["min_length"] = min_length
        if min_properties is not None:
            self._values["min_properties"] = min_properties
        if multiple_of is not None:
            self._values["multiple_of"] = multiple_of
        if not_ is not None:
            self._values["not_"] = not_
        if one_of is not None:
            self._values["one_of"] = one_of
        if pattern is not None:
            self._values["pattern"] = pattern
        if pattern_properties is not None:
            self._values["pattern_properties"] = pattern_properties
        if properties is not None:
            self._values["properties"] = properties
        if property_names is not None:
            self._values["property_names"] = property_names
        if ref is not None:
            self._values["ref"] = ref
        if required is not None:
            self._values["required"] = required
        if schema is not None:
            self._values["schema"] = schema
        if title is not None:
            self._values["title"] = title
        if type is not None:
            self._values["type"] = type
        if unique_items is not None:
            self._values["unique_items"] = unique_items

    @builtins.property
    def additional_items(self) -> typing.Optional[typing.List["JsonSchema"]]:
        result = self._values.get("additional_items")
        return typing.cast(typing.Optional[typing.List["JsonSchema"]], result)

    @builtins.property
    def additional_properties(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, "JsonSchema"]]:
        result = self._values.get("additional_properties")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, "JsonSchema"]], result)

    @builtins.property
    def all_of(self) -> typing.Optional[typing.List["JsonSchema"]]:
        result = self._values.get("all_of")
        return typing.cast(typing.Optional[typing.List["JsonSchema"]], result)

    @builtins.property
    def any_of(self) -> typing.Optional[typing.List["JsonSchema"]]:
        result = self._values.get("any_of")
        return typing.cast(typing.Optional[typing.List["JsonSchema"]], result)

    @builtins.property
    def contains(
        self,
    ) -> typing.Optional[typing.Union["JsonSchema", typing.List["JsonSchema"]]]:
        result = self._values.get("contains")
        return typing.cast(typing.Optional[typing.Union["JsonSchema", typing.List["JsonSchema"]]], result)

    @builtins.property
    def default(self) -> typing.Any:
        '''The default value if you use an enum.

        :default: - not set
        '''
        result = self._values.get("default")
        return typing.cast(typing.Any, result)

    @builtins.property
    def definitions(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, "JsonSchema"]]:
        result = self._values.get("definitions")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "JsonSchema"]], result)

    @builtins.property
    def dependencies(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Union[typing.List[builtins.str], "JsonSchema"]]]:
        result = self._values.get("dependencies")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Union[typing.List[builtins.str], "JsonSchema"]]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enum(self) -> typing.Optional[typing.List[typing.Any]]:
        result = self._values.get("enum")
        return typing.cast(typing.Optional[typing.List[typing.Any]], result)

    @builtins.property
    def exclusive_maximum(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("exclusive_maximum")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exclusive_minimum(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("exclusive_minimum")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def format(self) -> typing.Optional[builtins.str]:
        result = self._values.get("format")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def id(self) -> typing.Optional[builtins.str]:
        result = self._values.get("id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def items(
        self,
    ) -> typing.Optional[typing.Union["JsonSchema", typing.List["JsonSchema"]]]:
        result = self._values.get("items")
        return typing.cast(typing.Optional[typing.Union["JsonSchema", typing.List["JsonSchema"]]], result)

    @builtins.property
    def maximum(self) -> typing.Optional[jsii.Number]:
        result = self._values.get("maximum")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_items(self) -> typing.Optional[jsii.Number]:
        result = self._values.get("max_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_length(self) -> typing.Optional[jsii.Number]:
        result = self._values.get("max_length")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_properties(self) -> typing.Optional[jsii.Number]:
        result = self._values.get("max_properties")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def minimum(self) -> typing.Optional[jsii.Number]:
        result = self._values.get("minimum")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_items(self) -> typing.Optional[jsii.Number]:
        result = self._values.get("min_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_length(self) -> typing.Optional[jsii.Number]:
        result = self._values.get("min_length")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_properties(self) -> typing.Optional[jsii.Number]:
        result = self._values.get("min_properties")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def multiple_of(self) -> typing.Optional[jsii.Number]:
        result = self._values.get("multiple_of")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def not_(self) -> typing.Optional["JsonSchema"]:
        result = self._values.get("not_")
        return typing.cast(typing.Optional["JsonSchema"], result)

    @builtins.property
    def one_of(self) -> typing.Optional[typing.List["JsonSchema"]]:
        result = self._values.get("one_of")
        return typing.cast(typing.Optional[typing.List["JsonSchema"]], result)

    @builtins.property
    def pattern(self) -> typing.Optional[builtins.str]:
        result = self._values.get("pattern")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def pattern_properties(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, "JsonSchema"]]:
        result = self._values.get("pattern_properties")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "JsonSchema"]], result)

    @builtins.property
    def properties(self) -> typing.Optional[typing.Mapping[builtins.str, "JsonSchema"]]:
        result = self._values.get("properties")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "JsonSchema"]], result)

    @builtins.property
    def property_names(self) -> typing.Optional["JsonSchema"]:
        result = self._values.get("property_names")
        return typing.cast(typing.Optional["JsonSchema"], result)

    @builtins.property
    def ref(self) -> typing.Optional[builtins.str]:
        result = self._values.get("ref")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def required(self) -> typing.Optional[typing.List[builtins.str]]:
        result = self._values.get("required")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def schema(self) -> typing.Optional["JsonSchemaVersion"]:
        result = self._values.get("schema")
        return typing.cast(typing.Optional["JsonSchemaVersion"], result)

    @builtins.property
    def title(self) -> typing.Optional[builtins.str]:
        result = self._values.get("title")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type(
        self,
    ) -> typing.Optional[typing.Union["JsonSchemaType", typing.List["JsonSchemaType"]]]:
        result = self._values.get("type")
        return typing.cast(typing.Optional[typing.Union["JsonSchemaType", typing.List["JsonSchemaType"]]], result)

    @builtins.property
    def unique_items(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("unique_items")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "JsonSchema(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="aws-cdk-lib.aws_apigateway.JsonSchemaType")
class JsonSchemaType(enum.Enum):
    '''
    :exampleMetadata: infused

    Example::

        # api: apigateway.RestApi
        
        
        # We define the JSON Schema for the transformed valid response
        response_model = api.add_model("ResponseModel",
            content_type="application/json",
            model_name="ResponseModel",
            schema=apigateway.JsonSchema(
                schema=apigateway.JsonSchemaVersion.DRAFT4,
                title="pollResponse",
                type=apigateway.JsonSchemaType.OBJECT,
                properties={
                    "state": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING),
                    "greeting": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING)
                }
            )
        )
        
        # We define the JSON Schema for the transformed error response
        error_response_model = api.add_model("ErrorResponseModel",
            content_type="application/json",
            model_name="ErrorResponseModel",
            schema=apigateway.JsonSchema(
                schema=apigateway.JsonSchemaVersion.DRAFT4,
                title="errorResponse",
                type=apigateway.JsonSchemaType.OBJECT,
                properties={
                    "state": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING),
                    "message": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING)
                }
            )
        )
    '''

    NULL = "NULL"
    BOOLEAN = "BOOLEAN"
    OBJECT = "OBJECT"
    ARRAY = "ARRAY"
    NUMBER = "NUMBER"
    INTEGER = "INTEGER"
    STRING = "STRING"


@jsii.enum(jsii_type="aws-cdk-lib.aws_apigateway.JsonSchemaVersion")
class JsonSchemaVersion(enum.Enum):
    '''
    :exampleMetadata: infused

    Example::

        # api: apigateway.RestApi
        
        
        # We define the JSON Schema for the transformed valid response
        response_model = api.add_model("ResponseModel",
            content_type="application/json",
            model_name="ResponseModel",
            schema=apigateway.JsonSchema(
                schema=apigateway.JsonSchemaVersion.DRAFT4,
                title="pollResponse",
                type=apigateway.JsonSchemaType.OBJECT,
                properties={
                    "state": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING),
                    "greeting": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING)
                }
            )
        )
        
        # We define the JSON Schema for the transformed error response
        error_response_model = api.add_model("ErrorResponseModel",
            content_type="application/json",
            model_name="ErrorResponseModel",
            schema=apigateway.JsonSchema(
                schema=apigateway.JsonSchemaVersion.DRAFT4,
                title="errorResponse",
                type=apigateway.JsonSchemaType.OBJECT,
                properties={
                    "state": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING),
                    "message": apigateway.JsonSchema(type=apigateway.JsonSchemaType.STRING)
                }
            )
        )
    '''

    DRAFT4 = "DRAFT4"
    '''In API Gateway models are defined using the JSON schema draft 4.

    :see: https://tools.ietf.org/html/draft-zyp-json-schema-04
    '''
    DRAFT7 = "DRAFT7"


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.JsonWithStandardFieldProps",
    jsii_struct_bases=[],
    name_mapping={
        "caller": "caller",
        "http_method": "httpMethod",
        "ip": "ip",
        "protocol": "protocol",
        "request_time": "requestTime",
        "resource_path": "resourcePath",
        "response_length": "responseLength",
        "status": "status",
        "user": "user",
    },
)
class JsonWithStandardFieldProps:
    def __init__(
        self,
        *,
        caller: builtins.bool,
        http_method: builtins.bool,
        ip: builtins.bool,
        protocol: builtins.bool,
        request_time: builtins.bool,
        resource_path: builtins.bool,
        response_length: builtins.bool,
        status: builtins.bool,
        user: builtins.bool,
    ) -> None:
        '''Properties for controlling items output in JSON standard format.

        :param caller: If this flag is enabled, the principal identifier of the caller will be output to the log.
        :param http_method: If this flag is enabled, the http method will be output to the log.
        :param ip: If this flag is enabled, the source IP of request will be output to the log.
        :param protocol: If this flag is enabled, the request protocol will be output to the log.
        :param request_time: If this flag is enabled, the CLF-formatted request time((dd/MMM/yyyy:HH:mm:ss +-hhmm) will be output to the log.
        :param resource_path: If this flag is enabled, the path to your resource will be output to the log.
        :param response_length: If this flag is enabled, the response payload length will be output to the log.
        :param status: If this flag is enabled, the method response status will be output to the log.
        :param user: If this flag is enabled, the principal identifier of the user will be output to the log.

        :exampleMetadata: infused

        Example::

            # production stage
            prod_log_group = logs.LogGroup(self, "PrdLogs")
            api = apigateway.RestApi(self, "books",
                deploy_options=apigateway.StageOptions(
                    access_log_destination=apigateway.LogGroupLogDestination(prod_log_group),
                    access_log_format=apigateway.AccessLogFormat.json_with_standard_fields()
                )
            )
            deployment = apigateway.Deployment(self, "Deployment", api=api)
            
            # development stage
            dev_log_group = logs.LogGroup(self, "DevLogs")
            apigateway.Stage(self, "dev",
                deployment=deployment,
                access_log_destination=apigateway.LogGroupLogDestination(dev_log_group),
                access_log_format=apigateway.AccessLogFormat.json_with_standard_fields(
                    caller=False,
                    http_method=True,
                    ip=True,
                    protocol=True,
                    request_time=True,
                    resource_path=True,
                    response_length=True,
                    status=True,
                    user=True
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b46e9bc4a7b31e25ca2ff1231c36d2e58c54c67930eb0e367637e3fc6b87940e)
            check_type(argname="argument caller", value=caller, expected_type=type_hints["caller"])
            check_type(argname="argument http_method", value=http_method, expected_type=type_hints["http_method"])
            check_type(argname="argument ip", value=ip, expected_type=type_hints["ip"])
            check_type(argname="argument protocol", value=protocol, expected_type=type_hints["protocol"])
            check_type(argname="argument request_time", value=request_time, expected_type=type_hints["request_time"])
            check_type(argname="argument resource_path", value=resource_path, expected_type=type_hints["resource_path"])
            check_type(argname="argument response_length", value=response_length, expected_type=type_hints["response_length"])
            check_type(argname="argument status", value=status, expected_type=type_hints["status"])
            check_type(argname="argument user", value=user, expected_type=type_hints["user"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "caller": caller,
            "http_method": http_method,
            "ip": ip,
            "protocol": protocol,
            "request_time": request_time,
            "resource_path": resource_path,
            "response_length": response_length,
            "status": status,
            "user": user,
        }

    @builtins.property
    def caller(self) -> builtins.bool:
        '''If this flag is enabled, the principal identifier of the caller will be output to the log.'''
        result = self._values.get("caller")
        assert result is not None, "Required property 'caller' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def http_method(self) -> builtins.bool:
        '''If this flag is enabled, the http method will be output to the log.'''
        result = self._values.get("http_method")
        assert result is not None, "Required property 'http_method' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def ip(self) -> builtins.bool:
        '''If this flag is enabled, the source IP of request will be output to the log.'''
        result = self._values.get("ip")
        assert result is not None, "Required property 'ip' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def protocol(self) -> builtins.bool:
        '''If this flag is enabled, the request protocol will be output to the log.'''
        result = self._values.get("protocol")
        assert result is not None, "Required property 'protocol' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def request_time(self) -> builtins.bool:
        '''If this flag is enabled, the CLF-formatted request time((dd/MMM/yyyy:HH:mm:ss +-hhmm) will be output to the log.'''
        result = self._values.get("request_time")
        assert result is not None, "Required property 'request_time' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def resource_path(self) -> builtins.bool:
        '''If this flag is enabled, the path to your resource will be output to the log.'''
        result = self._values.get("resource_path")
        assert result is not None, "Required property 'resource_path' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def response_length(self) -> builtins.bool:
        '''If this flag is enabled, the response payload length will be output to the log.'''
        result = self._values.get("response_length")
        assert result is not None, "Required property 'response_length' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def status(self) -> builtins.bool:
        '''If this flag is enabled, the method response status will be output to the log.'''
        result = self._values.get("status")
        assert result is not None, "Required property 'status' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def user(self) -> builtins.bool:
        '''If this flag is enabled, the principal identifier of the user will be output to the log.'''
        result = self._values.get("user")
        assert result is not None, "Required property 'user' is missing"
        return typing.cast(builtins.bool, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "JsonWithStandardFieldProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.LambdaAuthorizerProps",
    jsii_struct_bases=[],
    name_mapping={
        "handler": "handler",
        "assume_role": "assumeRole",
        "authorizer_name": "authorizerName",
        "results_cache_ttl": "resultsCacheTtl",
    },
)
class LambdaAuthorizerProps:
    def __init__(
        self,
        *,
        handler: _IFunction_6adb0ab8,
        assume_role: typing.Optional[_IRole_235f5d8e] = None,
        authorizer_name: typing.Optional[builtins.str] = None,
        results_cache_ttl: typing.Optional[_Duration_4839e8c3] = None,
    ) -> None:
        '''Base properties for all lambda authorizers.

        :param handler: The handler for the authorizer lambda function. The handler must follow a very specific protocol on the input it receives and the output it needs to produce. API Gateway has documented the handler's `input specification <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-input.html>`_ and `output specification <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-output.html>`_.
        :param assume_role: An optional IAM role for APIGateway to assume before calling the Lambda-based authorizer. The IAM role must be assumable by 'apigateway.amazonaws.com'. Default: - A resource policy is added to the Lambda function allowing apigateway.amazonaws.com to invoke the function.
        :param authorizer_name: An optional human friendly name for the authorizer. Note that, this is not the primary identifier of the authorizer. Default: - the unique construct ID
        :param results_cache_ttl: How long APIGateway should cache the results. Max 1 hour. Disable caching by setting this to 0. Default: - Duration.minutes(5)

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk as cdk
            from aws_cdk import aws_apigateway as apigateway
            from aws_cdk import aws_iam as iam
            from aws_cdk import aws_lambda as lambda_
            
            # function_: lambda.Function
            # role: iam.Role
            
            lambda_authorizer_props = apigateway.LambdaAuthorizerProps(
                handler=function_,
            
                # the properties below are optional
                assume_role=role,
                authorizer_name="authorizerName",
                results_cache_ttl=cdk.Duration.minutes(30)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c077bb49a0760b852a8d0b71bedc64e7333185a01065fded15cfca2a3057c0f3)
            check_type(argname="argument handler", value=handler, expected_type=type_hints["handler"])
            check_type(argname="argument assume_role", value=assume_role, expected_type=type_hints["assume_role"])
            check_type(argname="argument authorizer_name", value=authorizer_name, expected_type=type_hints["authorizer_name"])
            check_type(argname="argument results_cache_ttl", value=results_cache_ttl, expected_type=type_hints["results_cache_ttl"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "handler": handler,
        }
        if assume_role is not None:
            self._values["assume_role"] = assume_role
        if authorizer_name is not None:
            self._values["authorizer_name"] = authorizer_name
        if results_cache_ttl is not None:
            self._values["results_cache_ttl"] = results_cache_ttl

    @builtins.property
    def handler(self) -> _IFunction_6adb0ab8:
        '''The handler for the authorizer lambda function.

        The handler must follow a very specific protocol on the input it receives
        and the output it needs to produce.  API Gateway has documented the
        handler's `input specification <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-input.html>`_
        and `output specification <https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-output.html>`_.
        '''
        result = self._values.get("handler")
        assert result is not None, "Required property 'handler' is missing"
        return typing.cast(_IFunction_6adb0ab8, result)

    @builtins.property
    def assume_role(self) -> typing.Optional[_IRole_235f5d8e]:
        '''An optional IAM role for APIGateway to assume before calling the Lambda-based authorizer.

        The IAM role must be
        assumable by 'apigateway.amazonaws.com'.

        :default: - A resource policy is added to the Lambda function allowing apigateway.amazonaws.com to invoke the function.
        '''
        result = self._values.get("assume_role")
        return typing.cast(typing.Optional[_IRole_235f5d8e], result)

    @builtins.property
    def authorizer_name(self) -> typing.Optional[builtins.str]:
        '''An optional human friendly name for the authorizer.

        Note that, this is not the primary identifier of the authorizer.

        :default: - the unique construct ID
        '''
        result = self._values.get("authorizer_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def results_cache_ttl(self) -> typing.Optional[_Duration_4839e8c3]:
        '''How long APIGateway should cache the results.

        Max 1 hour.
        Disable caching by setting this to 0.

        :default: - Duration.minutes(5)
        '''
        result = self._values.get("results_cache_ttl")
        return typing.cast(typing.Optional[_Duration_4839e8c3], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LambdaAuthorizerProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="aws-cdk-lib.aws_apigateway.LambdaIntegrationOptions",
    jsii_struct_bases=[IntegrationOptions],
    name_mapping={
        "cache_key_parameters": "cacheKeyParameters",
        "cache_namespace": "cacheNamespace",
        "connection_type": "connectionType",
        "content_handling": "contentHandling",
        "credentials_passthrough": "credentialsPassthrough",
        "credentials_role": "credentialsRole",
        "integration_responses": "integrationResponses",
        "passthrough_behavior": "passthroughBehavior",
        "request_parameters": "requestParameters",
        "request_templates": "requestTemplates",
        "timeout": "timeout",
        "vpc_link": "vpcLink",
        "allow_test_invoke": "allowTestInvoke",
        "proxy": "proxy",
    },
)
class LambdaIntegrationOptions(IntegrationOptions):
    def __init__(
        self,
        *,
        cache_key_parameters: typing.Optional[typing.Sequence[builtins.str]] = None,
        cache_namespace: typing.Optional[builtins.str] = None,
        connection_type: typing.Optional[ConnectionType] = None,
        content_handling: typing.Optional[ContentHandling] = None,
        credentials_passthrough: typing.Optional[builtins.bool] = None,
        credentials_role: typing.Optional[_IRole_235f5d8e] = None,
        integration_responses: typing.Optional[typing.Sequence[typing.Union[IntegrationResponse, typing.Dict[builtins.str, typing.Any]]]] = None,
        passthrough_behavior: typing.Optional["PassthroughBehavior"] = None,
        request_parameters: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        request_templates: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        timeout: typing.Optional[_Duration_4839e8c3] = None,
        vpc_link: typing.Optional[IVpcLink] = None,
        allow_test_invoke: typing.Optional[builtins.bool] = None,
        proxy: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param cache_key_parameters: A list of request parameters whose values are to be cached. It determines request parameters that will make it into the cache key.
        :param cache_namespace: An API-specific tag group of related cached parameters.
        :param connection_type: The type of network connection to the integration endpoint. Default: - ConnectionType.VPC_LINK if ``vpcLink`` property is configured; ConnectionType.Internet otherwise.
        :param content_handling: Specifies how to handle request payload content type conversions. Default: none if this property isn't defined, the request payload is passed through from the method request to the integration request without modification, provided that the ``passthroughBehaviors`` property is configured to support payload pass-through.
        :param credentials_passthrough: Requires that the caller's identity be passed through from the request. Default: Caller identity is not passed through
        :param credentials_role: An IAM role that API Gateway assumes. Mutually exclusive with ``credentialsPassThrough``. Default: A role is not assumed
        :param integration_responses: The response that API Gateway provides after a method's backend completes processing a request. API Gateway intercepts the response from the backend so that you can control how API Gateway surfaces backend responses. For example, you can map the backend status codes to codes that you define.
        :param passthrough_behavior: Specifies the pass-through behavior for incoming requests based on the Content-Type header in the request, and the available mapping templates specified as the requestTemplates property on the Integration resource. There are three valid values: WHEN_NO_MATCH, WHEN_NO_TEMPLATES, and NEVER.
        :param request_parameters: The request parameters that API Gateway sends with the backend request. Specify request parameters as key-value pairs (string-to-string mappings), with a destination as the key and a source as the value. Specify the destination by using the following pattern integration.request.location.name, where location is querystring, path, or header, and name is a valid, unique parameter name. The source must be an existing method request parameter or a static value. You must enclose static values in single quotation marks and pre-encode these values based on their destination in the request.
        :param request_templates: A map of Apache Velocity templates that are applied on the request payload. The template that API Gateway uses is based on the value of the Content-Type header that's sent by the client. The content type value is the key, and the template is the value (specified as a string), such as the following snippet:: { "application/json": "{ \\"statusCode\\": 200 }" }
        :param timeout: The maximum amount of time an integration will run before it returns without a response. By default, the value must be between 50 milliseconds and 29 seconds. The upper bound can be increased for regional and private Rest APIs only, via a quota increase request for your acccount. This increase might require a reduction in your account-level throttle quota limit. See {@link https://docs.aws.amazon.com/apigateway/latest/developerguide/limits.html Amazon API Gateway quotas} for more details. Default: Duration.seconds(29)
        :param vpc_link: The VpcLink used for the integration. Required if connectionType is VPC_LINK
        :param allow_test_invoke: Allow invoking method from AWS Console UI (for testing purposes). This will add another permission to the AWS Lambda resource policy which will allow the ``test-invoke-stage`` stage to invoke this handler. If this is set to ``false``, the function will only be usable from the deployment endpoint. Default: true
        :param proxy: Use proxy integration or normal (request/response mapping) integration. Default: true

        :exampleMetadata: infused

        Example::

            # backend: lambda.Function
            
            
            api = apigateway.LambdaRestApi(self, "myapi",
                handler=backend,
                integration_options=apigateway.LambdaIntegrationOptions(
                    allow_test_invoke=False,
                    timeout=Duration.seconds(1)
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ce30eee1096e0566120e3d64f9e65409fa005cc047656a224902fd86cd15c8b0)
            check_type(argname="argument cache_key_parameters", value=cache_key_parameters, expected_type=type_hints["cache_key_parameters"])
            check_type(argname="argument cache_namespace", value=cache_namespace, expected_type=type_hints["cache_namespace"])
            check_type(argname="argument connection_type", value=connection_type, expected_type=type_hints["connection_type"])
            check_type(argname="argument content_handling", value=content_handling, expected_type=type_hints["content_handling"])
            check_type(argname="argument credentials_passthrough", value=credentials_passthrough, expected_type=type_hints["credentials_passthrough"])
        